/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 *
 *  Copyright (C) 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Rebecca Schulman <rebecka@eazel.com>
 */

#include <config.h>

#include <errno.h>
#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-popt.h>
#include <libgnomevfs/gnome-vfs-init.h>
#include <libgnomevfs/gnome-vfs-types.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <popt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/resource.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/vfs.h>
#include <time.h>
#include <unistd.h>

#include <libmedusa-internal/medusa-conf.h>
#include <libmedusa-internal/medusa-debug.h>
#include <libmedusa-internal/medusa-lock.h>
#include <libmedusa-internal/medusa-master-db.h>
#include <libmedusa/medusa-search-service-private.h>
#include <libmedusa/medusa-index-progress.h>
#include <libmedusa/medusa-index-service.h>
#include <libmedusa/medusa-index-service-private.h>
#include <libmedusa/medusa-log.h>

#ifndef AF_LOCAL
#define AF_LOCAL AF_UNIX
#endif

#ifndef NAME_MAX
#define NAME_MAX 512
#endif

#define INDEX_SERVICE_LISTENER_DEBUG

#ifndef SUN_LEN
/* This system is not POSIX.1g.         */
#define SUN_LEN(ptr) ((size_t) (((struct sockaddr_un *) 0)->sun_path)  \
       + strlen ((ptr)->sun_path))
#endif

#define ACCOUNT_INDEX_NAME "account"

/* Creates a totally new file index, and writes it over the old one */
static void             do_full_indexing                  (const char *root_directory,
                                                           const char *index_name,
                                                           gboolean use_idle_service);

static void             do_index_accounting               (void);

/* Sits on the indexing socket and waits for requests to do either of the
   above indexings */
static void             wait_for_indexing_signal          (const char *index_name,
                                                           gboolean use_idle_service); 

static void             file_indexer_parse_transmission   (const char *transmission,
							   int client_fd,
                                                           const char *index_name,
                                                           gboolean dont_use_idle_service_for_indexing);

static void             register_command_line_options     (struct poptOption *medusa_options);
/* Do clean up, etc. */
static void             exit_indexer                      (void);

/* Sets up index request socket */
static int              initialize_index_server_socket   (struct sockaddr_un *daemon_address,
                                                          const char *index_name); 




extern int errno;
static gboolean create_index_even_if_disk_is_full;

static void
daemon_init (void)
{
        pid_t parent_process;
        struct rlimit resource_limits;
        int i;
        
        parent_process = fork ();
        /* check if fork failed */
        if (parent_process == -1) {
                medusa_log_fatal_error ("Forking the index service daemon failed\n");
                exit (1);
        }
        if (parent_process != 0) {
                exit (0);
        }

        getrlimit (RLIMIT_NOFILE, &resource_limits);
        for (i = 0; i < resource_limits.rlim_cur; i++) {
                close (i);
        }

        setsid ();

        /* Fork a second time to be sure that the process can't be a session leader */
        parent_process = fork ();
        if (parent_process == -1) {
                medusa_log_fatal_error ("Forking the index service daemon failed\n");
                exit (1);
        }
        if (parent_process != 0) {
                exit (0);
        }

        chdir ("/");
        umask (0);
        
        
}

/* Main index loop.  
   Creates three process to wait for signals, 
   do fast reindexing, and do slow reindexing */
int
main (int argc, char *argv[])
{
        poptContext popt_context;
        const char **unparsed_arguments;
        gboolean start_listener, text_index_accounting;
        gboolean dont_use_idle_service;
        gboolean run_in_listener_debug_mode;
        const char *root_directory, *index_name;
        struct poptOption command_line_options[] = {
                { "folder", 'f', POPT_ARG_STRING, &root_directory, 
                  0, N_("Specify the root folder to start indexing at"),
                  N_("PATH") },
                { "named-index", 'n', POPT_ARG_STRING, &index_name,
                  0, N_("Create a named index"),
                  N_("NAME") },
                { "start-listener", '\0', POPT_ARG_NONE, &start_listener,
                  0, N_("Start listener for interprocess indexing requests"),
                  NULL },
                { "do-text-accounting", '\0', POPT_ARG_NONE, &text_index_accounting,
                  0, N_("Count the space taken in the text index by file name and mime type"),
                  NULL },
                { "without-idle-checks", '\0', POPT_ARG_NONE, &dont_use_idle_service,
                  0, N_("Don't sleep when medusa-idled reports user activity"),
                  NULL },
                { "force-indexing", '\0', POPT_ARG_NONE, &create_index_even_if_disk_is_full,
                  0, N_("Create index even if the disk looks too full to support it"),
                  NULL },
                { "debug-listener", '\0', POPT_ARG_NONE, &run_in_listener_debug_mode,
                  0, N_("Don't fork the listener. Run it in debugging mode"),
                  NULL },
                POPT_AUTOHELP
                { NULL, '\0', 0, NULL, 0, NULL, NULL }
        };



        if (getuid () != 0) {
                puts ("The medusa index daemon must be run as root.  Exiting now\n");
                exit (1);
        }
        
        if (g_getenv ("MEDUSA_DEBUG") != NULL) {
		medusa_make_warnings_and_criticals_stop_in_debugger
			(G_LOG_DOMAIN, g_log_domain_glib,
			 "GnomeVFS",
                         "Medusa",
			 NULL);
	}

        /* Set these to defaults */
        root_directory = "/";
        index_name = "";
        start_listener = FALSE;
        text_index_accounting = FALSE;
        dont_use_idle_service = FALSE;
        run_in_listener_debug_mode = FALSE;

        nice (19);
        gnome_vfs_init ();
        
        register_command_line_options (command_line_options);
        popt_context = gnomelib_parse_args (argc, argv, 0);

        /* Get the remaining arguments */
        unparsed_arguments = poptGetArgs (popt_context);
        if (unparsed_arguments != NULL) {
                fprintf (stderr, "Invalid argument: %s\n", unparsed_arguments[0]);
                fprintf (stderr, "Exiting\n");
                exit_indexer ();
        }
        poptFreeContext (popt_context);

        if (run_in_listener_debug_mode &&
            !start_listener) {
                fprintf (stderr, "Listener debug mode will be ignored, since no listener is being run.\n");
        }

        if (start_listener) {
                if (!run_in_listener_debug_mode) {
                        daemon_init ();
                }
                wait_for_indexing_signal (index_name,
                                          dont_use_idle_service);
        }
        if (text_index_accounting) {
                do_index_accounting ();
                exit_indexer ();
        }
        
        do_full_indexing (root_directory, index_name, dont_use_idle_service); 
        exit_indexer ();
        
        return 1;
}



static void
wait_for_indexing_signal (const char *index_name,
                          gboolean dont_use_idle_service_for_indexing)
{
        struct sockaddr_un *daemon_address;
        guint address_length;
        int index_request_socket, client_fd;
        char transmission[MAX_LINE];
        int transmission_length;
  
        /* FIXME  bugzilla.eazel.com 2621:
           There should be some access control system here, so
           that medusa can be set up to only allow reindex requests
           from certain users */

        daemon_address = g_new0 (struct sockaddr_un, 1);
        index_request_socket = initialize_index_server_socket (daemon_address,
                                                               index_name);
        while (1) {
                address_length = SUN_LEN (daemon_address);
    
                client_fd = accept (index_request_socket, 
                                    (struct sockaddr *) daemon_address,
                                    &address_length);
                if (client_fd == -1) {
                        medusa_log_error ("Accepting transmission from client failed\n");
                        continue;
                }
                memset (transmission, 0, MAX_LINE);
                transmission_length = read (client_fd, transmission, MAX_LINE - 1);
                transmission[MAX_LINE - 1] = 0;

                file_indexer_parse_transmission (transmission, client_fd, 
                                                 index_name,
                                                 dont_use_idle_service_for_indexing);
                memset (transmission, 0, MAX_LINE);
                close (client_fd);
        }
}

void
do_index_accounting (void)
{
        char *root_uri;
  
        root_uri = gnome_vfs_get_uri_from_local_path (ROOT_DIRECTORY);

        medusa_master_db_erase_constructed_index (ACCOUNT_INDEX_NAME);

        medusa_master_db_create_index (root_uri,
                                       ACCOUNT_INDEX_NAME,
                                       FALSE,
                                       MEDUSA_DB_LOG_TEXT_INDEX_DATA);
        g_free (root_uri);
        printf ("Finished indexing \n");

        medusa_master_db_erase_constructed_index (ACCOUNT_INDEX_NAME);
        printf ("Removed account files\n");
}  
		     

static void
exit_if_disk_is_too_full ()
{
        struct statfs statfs_buffer;
        char *error_message;
        int statfs_result;
        
        statfs_result = statfs (MEDUSA_PREFIX INDEX_PATH, &statfs_buffer);                
        if (statfs_result == -1) {
                error_message = g_strdup_printf ("Can't determine file system status for %s, so the disk may be too full for an index. Indexing is continuing", MEDUSA_PREFIX INDEX_PATH);
                medusa_log_error (error_message);
                g_free (error_message);
                return;
        }
        g_print ("File system %s has %ld blocks, %ld free\n", 
                 MEDUSA_PREFIX INDEX_PATH, statfs_buffer.f_blocks, statfs_buffer.f_bfree);
        /* We'll need 40 megs or so for a conservative index estimate */
        if ((statfs_buffer.f_blocks / statfs_buffer.f_bfree) > 20 ||
            statfs_buffer.f_bfree * statfs_buffer.f_bsize < (40<<20)) {
                error_message = g_strdup_printf ("Indexing aborted because disk is either more than 95%% full or 40 megs of space are not available at location %s",
                                                 MEDUSA_PREFIX INDEX_PATH);
                medusa_log_error (error_message);
                g_free (error_message);
                exit (1);
        }
        
}

static void
do_full_indexing (const char *root_directory,
                  const char *index_name,
                  gboolean dont_use_idle_service)
{
        char *root_uri;
        MedusaWriteLock *write_lock;
        MedusaLogLevel log_level_from_environment;

        /* Remove old constructed, incomplete indices */
        medusa_master_db_erase_constructed_index (index_name);
        
        if (!create_index_even_if_disk_is_full) {
                exit_if_disk_is_too_full ();
        }
  
        write_lock = medusa_write_lock_get (index_name);
        if (write_lock == NULL) {
                medusa_log_fatal_error ("Failed to acquire lock file for indexing\n");
                exit (1);
        }

        medusa_index_progress_file_clear ();

        root_uri = gnome_vfs_get_uri_from_local_path (root_directory);
        log_level_from_environment = medusa_log_get_current_log_level ();

        printf ("Indexing...\n");
        medusa_master_db_create_index (root_uri,
                                       index_name,
                                       dont_use_idle_service,
                                       log_level_from_environment);
        g_free (root_uri);
        printf ("Finished indexing \n");

        medusa_master_db_move_completed_index_into_place (index_name);
        printf ("Done moving index files to correct locations\n");

        medusa_write_lock_release (write_lock);
        printf ("Released lock.\n");

        printf ("Exiting\n");
        exit_indexer ();
}

static char *
get_index_socket_name (const char *index_name)
{
        if (strlen (index_name) > 0) {
                return g_strdup_printf ("%s-%s", INDEX_SOCKET_PATH, index_name);
        }
        else {
                return g_strdup (INDEX_SOCKET_PATH);
        }
}


static int
initialize_index_server_socket (struct sockaddr_un *daemon_address,
                                const char *index_name)
{
        int index_listening_fd; 
        int bind_return_code;
        int listen_return_code;
        char *socket_file_name;

        index_listening_fd = socket (AF_LOCAL, SOCK_STREAM, 0);
        g_return_val_if_fail (index_listening_fd != -1, -1);
  
        daemon_address->sun_family = AF_LOCAL; 
        /* FIXME bugzilla.eazel.com 2635:
           Hard coding the size of the struct seems ugly,
           but it is not clear what the more positive alternative is */
        socket_file_name = get_index_socket_name (index_name);
        strncpy (daemon_address->sun_path, socket_file_name, 100);
        daemon_address->sun_path[99] = 0;
        g_free (socket_file_name);
        /* If socket currently exists, delete it */
        unlink (daemon_address->sun_path); 
#ifdef INDEX_DEBUG
        g_message  ("Creating socket at %s\n", daemon_address->sun_path);
#endif

        bind_return_code = bind (index_listening_fd, (struct sockaddr *) daemon_address, 
                                 SUN_LEN (daemon_address));
        if (bind_return_code != 0) {
                medusa_log_error ("Calling bind on the index server socket failed\n");
                return -1;
        }       
        listen_return_code = listen (index_listening_fd, 5);
        if (listen_return_code != 0) {
                medusa_log_error ("Calling listen on the index server socket failed\n");
                return -1;
        }
        chmod (daemon_address->sun_path, S_IRWXU | S_IRWXG | S_IRWXO);

        return index_listening_fd;
} 


static void
file_indexer_parse_transmission (const char *transmission,
				 int client_fd,
                                 const char *index_name,
                                 gboolean dont_use_idle_service_for_indexing)
{
        MedusaWriteLock *write_lock;
        pid_t forked_process_id;
  
        if (!strlen (transmission)) {
#ifdef INDEX_SERVICE_LISTENER_DEBUG
                printf ("Ignoring blank index service request\n");
#endif
                return;
        }
  
        if (strstr (transmission, MEDUSA_REINDEX_REQUEST)) {
                /* Grab DB lock */
                write_lock = medusa_write_lock_get (index_name);
                if (write_lock == NULL) {
                        /* We can't index right now -- someone else has the lock */
                        write (client_fd, MEDUSA_REINDEX_REQUEST_DENIED_BUSY,
                               strlen (MEDUSA_REINDEX_REQUEST_DENIED_BUSY));
                } else {
                        write (client_fd, MEDUSA_REINDEX_REQUEST_ACCEPTED, 
                               strlen (MEDUSA_REINDEX_REQUEST_ACCEPTED));
                        /* FIXME: This is not atomic */
                        medusa_write_lock_release (write_lock);
                        forked_process_id = fork ();
                        if (forked_process_id == 0) {
                                do_full_indexing (ROOT_DIRECTORY,
                                                  index_name,
                                                  dont_use_idle_service_for_indexing);

                        }
                }
        }
        else if (strstr (transmission, MEDUSA_INDEX_SERVICE_AVAILABILITY_REQUEST)) {
                write_lock = medusa_write_lock_get (index_name);
                if (write_lock == NULL) {
                        write (client_fd, MEDUSA_INDEX_SERVICE_AVAILABILITY_BUSY,
                               strlen (MEDUSA_INDEX_SERVICE_AVAILABILITY_BUSY));
                }
                else {
                        medusa_write_lock_release (write_lock);
                        write (client_fd, MEDUSA_INDEX_SERVICE_AVAILABILITY_UP,
                               strlen (MEDUSA_INDEX_SERVICE_AVAILABILITY_UP));
                }
        }
}

static void             
register_command_line_options (struct poptOption *medusa_options)
{
        gnomelib_register_popt_table (medusa_options, "Medusa Indexer Options");
}
static void             
exit_indexer (void)
{
        gnome_vfs_shutdown ();
        exit (0);
}



