#include <dirent.h>
#include <dlfcn.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include "applet.h"


typedef struct {
	void *dl_handle;
	char *filename;
} AppletFile;


static GHashTable *applet_files_ht = NULL;


static AppletFile *
applet_file_new(void *dl_handle, char *filename)
{
	AppletFile *af;

	af = g_new(AppletFile, 1);

	af->dl_handle = dl_handle;
	af->filename  = filename;

	return af;
}


static void
applet_file_destroy(gpointer key, gpointer value, gpointer user_data)
{
	AppletFile *af;

	af = value;

	dlclose(af->dl_handle);
	g_free(af->filename);
	g_free(af);
}


static void *
get_dl_func(void *handle, char *name)
{
	void *func;
	char *error;

	func = dlsym(handle, name);
	if ((error = dlerror()) != NULL) {
		fprintf(stderr, "get_dl_func: %s\n", error);
		exit(1);
	}

	return func;
}


static void
query_applet_file(char *filename)
{
	void            *handle;
	AppletQueryFunc  query;
	char            *error;
	char            *id;
	AppletFile      *af;

	handle = dlopen(filename, RTLD_LAZY);
	if (!handle) {
		/* FIXME: this should be a nice message box */
		error = dlerror();
		fprintf(stderr, "applet_query: %s\n", error);
		return;
	}

	query = get_dl_func(handle, "query");

	id = (*query) ();
	if (!id) {
		fprintf(stderr, "applet_query: query() on %s returned a NULL identifier\n", filename);
		return;
	}

	af = applet_file_new(handle, g_strdup(filename));
	
	g_hash_table_insert(applet_files_ht, id, af);
}


static void
applet_files_init(void)
{
	char          *dirname;
	DIR           *dir;
	struct dirent *dir_ent;
	char          *filename;
	struct stat    filestat;
	int            len;
	
	applet_files_ht = g_hash_table_new(g_string_hash, g_string_equal);

	dirname = gnome_unconditional_libdir_file("."); /* Get libdir name */

	dir = opendir(dirname);
	if (dir == NULL)
		return;

	while ((dir_ent = readdir(dir)) != NULL) {
		/* We only want stuff of the form "libpanel_*.so" */
		
		if (strncmp(dir_ent->d_name, "libpanel_", 9) != 0)
			continue;

		len = strlen(dir_ent->d_name); /* Cannot be less than 9, because of above */

		if (strcmp(dir_ent->d_name + (len - 3), ".so") != 0)
			continue;
		
		filename = g_concat_dir_and_file(dirname, dir_ent->d_name);

		if ((stat(filename, &filestat) == 0) &&
		    S_ISREG(filestat.st_mode))
			query_applet_file(filename);

		g_free(filename);
	}

	closedir(dir);

	g_free(dirname);
}


void
applets_init(void)
{
	applet_files_init();
}


void
applets_destroy(void)
{
	g_hash_table_foreach(applet_files_ht, applet_file_destroy, NULL);
	g_hash_table_destroy(applet_files_ht);
}


void
applets_init_applet(char *name, Panel *panel, char *params, int xpos, int ypos)
{
	AppletFile     *af;
	AppletInitFunc  init;

	af = g_hash_table_lookup(applet_files_ht, name);

	if (!af) {
		/* FIXME: this should be a nice message box */
		fprintf(stderr, "applets_init_applet: applet type \"%s\" is not available\n", name);
		return;
	}

	init = get_dl_func(af->dl_handle, "init");
	(*init) (panel_command, panel, params, xpos, ypos);
}
