#include <config.h>
#include <errno.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <glib.h>
#include "gnome-defs.h"
#include "gnome-util.h"

static char *
gnome_dirrelative_file (char *base, char *sub, char *filename)
{
	static char *gnomedir = NULL;
	char *f, *t, *u;

	/* First try the env GNOMEDIR relative path */
	if (!gnomedir)
		gnomedir = getenv ("GNOMEDIR");

	if (gnomedir){
		t = g_concat_dir_and_file (gnomedir, sub);
		u = g_copy_strings (t, "/", filename, NULL);
		g_free (t);
		
		if (g_file_exists (u))
			return u;
	
		g_free (u);
	}
	
	/* Then try the hardcoded path */
	f = g_concat_dir_and_file (base, filename);
	
	if (g_file_exists (f))
		return f;
	
	g_free (f);
	
	/* Finally, attempt to find it in the current directory */
	f = g_concat_dir_and_file (".", filename);
	
	if (g_file_exists (f))
		return f;
	
	g_free (f);
	return NULL;
}

/* DOC: gnome_libdir_file (char *filename)
 * Returns a newly allocated pathname poiting to a file in the gnome libdir
 */
char *
gnome_libdir_file (char *filename)
{
	return (gnome_dirrelative_file (GNOMELIBDIR, "lib", filename));
}

/* DOC: gnome_sharedir_file (char *filename)
 * Returns a newly allocated pathname poiting to a file in the gnome sharedir
 */
char *
gnome_datadir_file (char *filename)
{
	return (gnome_dirrelative_file (GNOMEDATADIR, "share", filename));
}

/* DOC: g_file_exists (char *filename)
 * Returns true if filename exists
 */
int
g_file_exists (char *filename)
{
	return access (filename, R_OK) == 0;
}


/* DOC: g_copy_strings (const char *first,...)
 * returns a new allocated char * with the concatenation of its arguments
 */
char *
g_copy_strings (const char *first, ...)
{
	va_list ap;
	int len;
	char *data, *result;
	
	if (!first)
		return NULL;
	
	len = strlen (first);
	va_start (ap, first);
	
	while ((data = va_arg (ap, char *)) != NULL)
		len += strlen (data);
	
	len++;
	
	result = (char *) g_malloc (len);
	va_end (ap);
	va_start (ap, first);
	strcpy (result, first);
	while ((data = va_arg (ap, char *)) != NULL)
		strcat (result, data);
	va_end (ap);
	
	return result;
}


/* DOC: g_unix_error_string (int error_num)
 * Returns a pointer to a static location with a description of the errno
 */
char *
g_unix_error_string (int error_num)
{
	static char buffer [256];
	char *error_msg;
	
#ifdef HAVE_STRERROR
	error_msg = strerror (error_num);
#else
	extern int sys_nerr;
	extern char *sys_errlist [];
	if ((0 <= error_num) && (error_num < sys_nerr))
		error_msg = sys_errlist[error_num];
	else
		error_msg = "strange errno";
#endif /* HAVE_STRERROR */
	sprintf (buffer, "%s (%d)", error_msg, error_num);
	return buffer;
}


/* DOC: g_concat_dir_and_file (const char *dir, const char *file)
 * returns a new allocated string that is the concatenation of dir and file,
 * takes care of the exact details for concatenating them.
 */
char *
g_concat_dir_and_file (const char *dir, const char *file)
{
	int l = strlen (dir);
	
	if (dir [l - 1] == PATH_SEP)
		return g_copy_strings (dir, file, NULL);
	else
		return g_copy_strings (dir, PATH_SEP_STR, file, NULL);
}
