/* -*- mode: C++; c-basic-offset: 4; indent-tabs-mode: nil; -*- */
// SPDX-License-Identifier: MIT OR LGPL-2.0-or-later
// SPDX-FileCopyrightText: 2008 litl, LLC

#include <config.h>  // for PACKAGE_STRING

#include <locale.h>  // for setlocale, LC_ALL
#include <stdlib.h>  // for exit
#include <string.h>  // for strcmp, strlen

#ifdef HAVE_UNISTD_H
#    include <unistd.h>  // for close
#elif defined (_WIN32)
#    include <io.h>
#endif

#include <gio/gio.h>
#include <glib-object.h>
#include <glib.h>

#include <gjs/gjs.h>

static char **include_path = NULL;
static char **coverage_prefixes = NULL;
static char *coverage_output_path = NULL;
static char *profile_output_path = nullptr;
static char *command = NULL;
static gboolean print_version = false;
static gboolean print_js_version = false;
static gboolean debugging = false;
static bool enable_profiler = false;

static gboolean parse_profile_arg(const char *, const char *, void *, GError **);

// clang-format off
static GOptionEntry entries[] = {
    { "version", 0, 0, G_OPTION_ARG_NONE, &print_version, "Print GJS version and exit" },
    { "jsversion", 0, 0, G_OPTION_ARG_NONE, &print_js_version,
        "Print version of the JS engine and exit" },
    { "command", 'c', 0, G_OPTION_ARG_STRING, &command, "Program passed in as a string", "COMMAND" },
    { "coverage-prefix", 'C', 0, G_OPTION_ARG_STRING_ARRAY, &coverage_prefixes, "Add the prefix PREFIX to the list of files to generate coverage info for", "PREFIX" },
    { "coverage-output", 0, 0, G_OPTION_ARG_STRING, &coverage_output_path, "Write coverage output to a directory DIR. This option is mandatory when using --coverage-prefix", "DIR", },
    { "include-path", 'I', 0, G_OPTION_ARG_STRING_ARRAY, &include_path, "Add the directory DIR to the list of directories to search for js files.", "DIR" },
    { "profile", 0, G_OPTION_FLAG_OPTIONAL_ARG | G_OPTION_FLAG_FILENAME,
        G_OPTION_ARG_CALLBACK, reinterpret_cast<void *>(&parse_profile_arg),
        "Enable the profiler and write output to FILE (default: gjs-$PID.syscap)",
        "FILE" },
    { "debugger", 'd', 0, G_OPTION_ARG_NONE, &debugging, "Start in debug mode" },
    { NULL }
};
// clang-format on

[[nodiscard]] static char** strndupv(int n, char* const* strv) {
    int ix;
    if (n == 0)
        return NULL;
    char **retval = g_new(char *, n + 1);
    for (ix = 0; ix < n; ix++)
        retval[ix] = g_strdup(strv[ix]);
    retval[n] = NULL;
    return retval;
}

[[nodiscard]] static char** strcatv(char** strv1, char** strv2) {
    if (strv1 == NULL && strv2 == NULL)
        return NULL;
    if (strv1 == NULL)
        return g_strdupv(strv2);
    if (strv2 == NULL)
        return g_strdupv(strv1);

    unsigned len1 = g_strv_length(strv1);
    unsigned len2 = g_strv_length(strv2);
    char **retval = g_new(char *, len1 + len2 + 1);
    unsigned ix;

    for (ix = 0; ix < len1; ix++)
        retval[ix] = g_strdup(strv1[ix]);
    for (ix = 0; ix < len2; ix++)
        retval[len1 + ix] = g_strdup(strv2[ix]);
    retval[len1 + len2] = NULL;

    return retval;
}

static gboolean parse_profile_arg(const char* option_name [[maybe_unused]],
                                  const char* value, void*, GError**) {
    enable_profiler = true;
    g_free(profile_output_path);
    profile_output_path = g_strdup(value);
    return true;
}

static void
check_script_args_for_stray_gjs_args(int           argc,
                                     char * const *argv)
{
    GError *error = NULL;
    char **new_coverage_prefixes = NULL;
    char *new_coverage_output_path = NULL;
    char **new_include_paths = NULL;
    // Don't add new entries here. This is only for arguments that were
    // previously accepted after the script name on the command line, for
    // backwards compatibility.
    GOptionEntry script_check_entries[] = {
        { "coverage-prefix", 'C', 0, G_OPTION_ARG_STRING_ARRAY, &new_coverage_prefixes },
        { "coverage-output", 0, 0, G_OPTION_ARG_STRING, &new_coverage_output_path },
        { "include-path", 'I', 0, G_OPTION_ARG_STRING_ARRAY, &new_include_paths },
        { NULL }
    };
    char **argv_copy = g_new(char *, argc + 2);
    int ix;

    argv_copy[0] = g_strdup("dummy"); /* Fake argv[0] for GOptionContext */
    for (ix = 0; ix < argc; ix++)
        argv_copy[ix + 1] = g_strdup(argv[ix]);
    argv_copy[argc + 1] = NULL;

    GOptionContext *script_options = g_option_context_new(NULL);
    g_option_context_set_ignore_unknown_options(script_options, true);
    g_option_context_set_help_enabled(script_options, false);
    g_option_context_add_main_entries(script_options, script_check_entries, NULL);
    if (!g_option_context_parse_strv(script_options, &argv_copy, &error)) {
        g_warning("Scanning script arguments failed: %s", error->message);
        g_error_free(error);
        g_strfreev(argv_copy);
        return;
    }

    if (new_coverage_prefixes != NULL) {
        g_warning("You used the --coverage-prefix option after the script on "
                  "the GJS command line. Support for this will be removed in a "
                  "future version. Place the option before the script or use "
                  "the GJS_COVERAGE_PREFIXES environment variable.");
        char **old_coverage_prefixes = coverage_prefixes;
        coverage_prefixes = strcatv(old_coverage_prefixes, new_coverage_prefixes);
        g_strfreev(old_coverage_prefixes);
    }
    if (new_include_paths != NULL) {
        g_warning("You used the --include-path option after the script on the "
                  "GJS command line. Support for this will be removed in a "
                  "future version. Place the option before the script or use "
                  "the GJS_PATH environment variable.");
        char **old_include_paths = include_path;
        include_path = strcatv(old_include_paths, new_include_paths);
        g_strfreev(old_include_paths);
    }
    if (new_coverage_output_path != NULL) {
        g_warning("You used the --coverage-output option after the script on "
                  "the GJS command line. Support for this will be removed in a "
                  "future version. Place the option before the script or use "
                  "the GJS_COVERAGE_OUTPUT environment variable.");
        g_free(coverage_output_path);
        coverage_output_path = new_coverage_output_path;
    }

    g_option_context_free(script_options);
    g_strfreev(argv_copy);
}

int define_argv_and_eval_script(GjsContext* js_context, int argc,
                                char* const* argv, const char* script,
                                size_t len, const char* filename) {
    GError* error = nullptr;

    /* prepare command line arguments */
    if (!gjs_context_define_string_array(
            js_context, "ARGV", argc, const_cast<const char**>(argv), &error)) {
        g_critical("Failed to define ARGV: %s", error->message);
        g_clear_error(&error);
        return 1;
    }

    /* evaluate the script */
    int code;
    if (!gjs_context_eval(js_context, script, len, filename, &code, &error)) {
        if (!g_error_matches(error, GJS_ERROR, GJS_ERROR_SYSTEM_EXIT))
            g_critical("%s", error->message);
        g_clear_error(&error);
    }
    return code;
}

int
main(int argc, char **argv)
{
    GOptionContext *context;
    GError *error = NULL;
    GjsContext *js_context;
    GjsCoverage *coverage = NULL;
    char *script;
    const char *filename;
    const char *program_name;
    gsize len;
    int gjs_argc = argc, script_argc, ix;
    char **argv_copy = g_strdupv(argv), **argv_copy_addr = argv_copy;
    char **gjs_argv, **gjs_argv_addr;
    char * const *script_argv;
    const char *env_coverage_output_path;
    bool interactive_mode = false;

    setlocale(LC_ALL, "");

    context = g_option_context_new(NULL);

    g_option_context_set_ignore_unknown_options(context, true);
    g_option_context_set_help_enabled(context, false);

    g_option_context_add_main_entries(context, entries, NULL);
    if (!g_option_context_parse_strv(context, &argv_copy, &error))
        g_error("option parsing failed: %s", error->message);

    /* Split options so we pass unknown ones through to the JS script */
    int argc_copy = g_strv_length(argv_copy);
    for (ix = 1; ix < argc; ix++) {
        /* Check if a file was given and split after it */
        if (argc_copy >= 2 && strcmp(argv[ix], argv_copy[1]) == 0) {
            /* Filename given; split after this argument */
            gjs_argc = ix + 1;
            break;
        }

        /* Check if -c or --command was given and split after following arg */
        if (command != NULL &&
            (strcmp(argv[ix], "-c") == 0 || strcmp(argv[ix], "--command") == 0)) {
            gjs_argc = ix + 2;
            break;
        }
    }
    gjs_argv_addr = gjs_argv = strndupv(gjs_argc, argv);
    script_argc = argc - gjs_argc;
    script_argv = argv + gjs_argc;
    g_strfreev(argv_copy_addr);

    /* Parse again, only the GJS options this time */
    include_path = NULL;
    coverage_prefixes = NULL;
    coverage_output_path = NULL;
    command = NULL;
    print_version = false;
    print_js_version = false;
    debugging = false;
    g_option_context_set_ignore_unknown_options(context, false);
    g_option_context_set_help_enabled(context, true);
    if (!g_option_context_parse_strv(context, &gjs_argv, &error)) {
        char* help_text = g_option_context_get_help(context, true, nullptr);
        g_printerr("%s\n\n%s\n", error->message, help_text);
        g_free(help_text);
        g_option_context_free(context);
        exit(1);
    }

    g_option_context_free (context);

    if (print_version) {
        g_print("%s\n", PACKAGE_STRING);
        exit(0);
    }

    if (print_js_version) {
        g_print("%s\n", gjs_get_js_version());
        exit(0);
    }

    gjs_argc = g_strv_length(gjs_argv);
    if (command != NULL) {
        script = command;
        len = strlen(script);
        filename = "<command line>";
        program_name = gjs_argv[0];
    } else if (gjs_argc == 1) {
        script = g_strdup("const Console = imports.console; Console.interact();");
        len = strlen(script);
        filename = "<stdin>";
        program_name = gjs_argv[0];
        interactive_mode = true;
    } else {
        /* All unprocessed options should be in script_argv */
        g_assert(gjs_argc == 2);
        error = NULL;
        if (!g_file_get_contents(gjs_argv[1], &script, &len, &error)) {
            g_printerr("%s\n", error->message);
            exit(1);
        }
        filename = gjs_argv[1];
        program_name = gjs_argv[1];
    }

    /* This should be removed after a suitable time has passed */
    check_script_args_for_stray_gjs_args(script_argc, script_argv);

    /* Check for GJS_TRACE_FD for sysprof profiling */
    const char* env_tracefd = g_getenv("GJS_TRACE_FD");
    int tracefd = -1;
    if (env_tracefd) {
        tracefd = g_ascii_strtoll(env_tracefd, nullptr, 10);
        g_setenv("GJS_TRACE_FD", "", true);
        if (tracefd > 0)
            enable_profiler = true;
    }

    if (interactive_mode && enable_profiler) {
        g_message("Profiler disabled in interactive mode.");
        enable_profiler = false;
        g_unsetenv("GJS_ENABLE_PROFILER");  /* ignore env var in eval() */
        g_unsetenv("GJS_TRACE_FD");         /* ignore env var in eval() */
    }

    const char* env_coverage_prefixes = g_getenv("GJS_COVERAGE_PREFIXES");
    if (env_coverage_prefixes) {
        if (coverage_prefixes)
            g_strfreev(coverage_prefixes);
        coverage_prefixes = g_strsplit(env_coverage_prefixes, ":", -1);
    }
    if (coverage_prefixes)
        gjs_coverage_enable();

    js_context = (GjsContext*) g_object_new(GJS_TYPE_CONTEXT,
                                            "search-path", include_path,
                                            "program-name", program_name,
                                            "profiler-enabled", enable_profiler,
                                            NULL);

    env_coverage_output_path = g_getenv("GJS_COVERAGE_OUTPUT");
    if (env_coverage_output_path != NULL) {
        g_free(coverage_output_path);
        coverage_output_path = g_strdup(env_coverage_output_path);
    }

    if (coverage_prefixes) {
        if (!coverage_output_path)
            g_error("--coverage-output is required when taking coverage statistics");

        GFile *output = g_file_new_for_commandline_arg(coverage_output_path);
        coverage = gjs_coverage_new(coverage_prefixes, js_context, output);
        g_object_unref(output);
    }

    if (enable_profiler && profile_output_path) {
        GjsProfiler *profiler = gjs_context_get_profiler(js_context);
        gjs_profiler_set_filename(profiler, profile_output_path);
    } else if (enable_profiler && tracefd > -1) {
        GjsProfiler* profiler = gjs_context_get_profiler(js_context);
        gjs_profiler_set_fd(profiler, tracefd);
        tracefd = -1;
    }

    if (tracefd != -1) {
        close(tracefd);
        tracefd = -1;
    }

    /* If we're debugging, set up the debugger. It will break on the first
     * frame. */
    if (debugging)
        gjs_context_setup_debugger_console(js_context);

    int code = define_argv_and_eval_script(js_context, script_argc, script_argv,
                                           script, len, filename);

    g_strfreev(gjs_argv_addr);

    /* Probably doesn't make sense to write statistics on failure */
    if (coverage && code == 0)
        gjs_coverage_write_statistics(coverage);

    g_free(coverage_output_path);
    g_free(profile_output_path);
    g_strfreev(coverage_prefixes);
    if (coverage)
        g_object_unref(coverage);
    g_object_unref(js_context);
    g_free(script);

    if (debugging)
        g_print("Program exited with code %d\n", code);
    exit(code);
}
