/* -*- mode: C++; c-basic-offset: 4; indent-tabs-mode: nil; -*- */
/*
 * Copyright (c) 2013       Intel Corporation
 * Copyright (c) 2008-2010  litl, LLC
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#ifndef __GJS_FUNDAMENTAL_H__
#define __GJS_FUNDAMENTAL_H__

#include <stdbool.h>
#include <glib.h>
#include <girepository.h>

#include "gi/wrapperutils.h"
#include "gjs/jsapi-util.h"
#include "gjs/macros.h"

class FundamentalPrototype;
class FundamentalInstance;

/* To conserve memory, we have two different kinds of private data for JS
 * wrappers for fundamental types: FundamentalInstance, and
 * FundamentalPrototype. Both inherit from FundamentalBase for their common
 * functionality. For more information, see the notes in wrapperutils.h.
 */

class FundamentalBase
    : public GIWrapperBase<FundamentalBase, FundamentalPrototype,
                           FundamentalInstance> {
    friend class GIWrapperBase;

 protected:
    explicit FundamentalBase(FundamentalPrototype* proto = nullptr)
        : GIWrapperBase(proto) {}
    ~FundamentalBase(void) {}

    static const GjsDebugTopic debug_topic = GJS_DEBUG_GFUNDAMENTAL;
    static constexpr const char* debug_tag = "fundamental";

    static const struct JSClassOps class_ops;
    static const struct JSClass klass;

    // Helper methods

    GJS_USE const char* to_string_kind(void) const { return "fundamental"; }
};

class FundamentalPrototype
    : public GIWrapperPrototype<FundamentalBase, FundamentalPrototype,
                                FundamentalInstance> {
    friend class GIWrapperPrototype;
    friend class GIWrapperBase;

    GIObjectInfoRefFunction m_ref_function;
    GIObjectInfoUnrefFunction m_unref_function;
    GIObjectInfoGetValueFunction m_get_value_function;
    GIObjectInfoSetValueFunction m_set_value_function;

    JS::Heap<jsid> m_constructor_name;
    GICallableInfo* m_constructor_info;

    explicit FundamentalPrototype(GIObjectInfo* info, GType gtype);
    ~FundamentalPrototype(void);

    GJS_JSAPI_RETURN_CONVENTION bool init(JSContext* cx);

 public:
    GJS_JSAPI_RETURN_CONVENTION
    static FundamentalPrototype* for_gtype(JSContext* cx, GType gtype);

    // Accessors

    GJS_USE
    jsid constructor_name(void) const { return m_constructor_name.get(); }
    GJS_USE
    GICallableInfo* constructor_info(void) const { return m_constructor_info; }

    void* call_ref_function(void* ptr) const { return m_ref_function(ptr); }
    void call_unref_function(void* ptr) const { m_unref_function(ptr); }
    GJS_USE void* call_get_value_function(const GValue* value) const {
        return m_get_value_function(value);
    }

    // Helper methods

 private:
    GJS_JSAPI_RETURN_CONVENTION
    bool get_parent_proto(JSContext* cx, JS::MutableHandleObject proto) const;

    GJS_USE unsigned constructor_nargs(void) const;

    GJS_JSAPI_RETURN_CONVENTION
    bool resolve_interface(JSContext* cx, JS::HandleObject obj, bool* resolved,
                           const char* name);

    // JSClass operations

    GJS_JSAPI_RETURN_CONVENTION
    bool resolve_impl(JSContext* cx, JS::HandleObject obj, JS::HandleId id,
                      const char* prop_name, bool* resolved);
    void trace_impl(JSTracer* trc);
};

class FundamentalInstance
    : public GIWrapperInstance<FundamentalBase, FundamentalPrototype,
                               FundamentalInstance> {
    friend class GIWrapperInstance;
    friend class GIWrapperBase;

    explicit FundamentalInstance(JSContext* cx, JS::HandleObject obj);
    ~FundamentalInstance(void);

    // Helper methods

    GJS_JSAPI_RETURN_CONVENTION
    bool invoke_constructor(JSContext* cx, JS::HandleObject obj,
                            const JS::HandleValueArray& args,
                            GIArgument* rvalue);

    void ref(void) { get_prototype()->call_ref_function(m_ptr); }
    void unref(void) { get_prototype()->call_unref_function(m_ptr); }

    // JS constructor

    GJS_JSAPI_RETURN_CONVENTION
    bool constructor_impl(JSContext* cx, JS::HandleObject obj,
                          const JS::CallArgs& args);

 public:
    GJS_JSAPI_RETURN_CONVENTION
    bool associate_js_instance(JSContext* cx, JSObject* object,
                               void* gfundamental);
};

G_BEGIN_DECLS

GJS_JSAPI_RETURN_CONVENTION
bool gjs_define_fundamental_class(JSContext              *context,
                                  JS::HandleObject        in_object,
                                  GIObjectInfo           *info,
                                  JS::MutableHandleObject constructor,
                                  JS::MutableHandleObject prototype);

GJS_JSAPI_RETURN_CONVENTION
JSObject* gjs_object_from_g_fundamental      (JSContext     *context,
                                              GIObjectInfo  *info,
                                              void          *fobj);

GJS_JSAPI_RETURN_CONVENTION
void     *gjs_g_fundamental_from_object(JSContext       *context,
                                        JS::HandleObject obj);

GJS_JSAPI_RETURN_CONVENTION
JSObject *gjs_fundamental_from_g_value       (JSContext     *context,
                                              const GValue  *value,
                                              GType          gtype);

GJS_USE
bool      gjs_typecheck_fundamental(JSContext       *context,
                                    JS::HandleObject object,
                                    GType            expected_gtype,
                                    bool             throw_error);

void*     gjs_fundamental_ref                (JSContext     *context,
                                              void          *fobj);
void      gjs_fundamental_unref              (JSContext     *context,
                                              void          *fobj);

G_END_DECLS

#endif  /* __GJS_FUNDAMENTAL_H__ */
