#!/bin/bash

set -e

# generates the linker version script src/NetworkManager.ver
# by looking at the symbols needed by the device and settings
# plugins. Note that this depends on how NetworkManager and
# the plugins are build. For example, compiling without
# --with-more-asserts will yield less symbols.
#
# _build re-builds NetworkManager with relevant compile time
# options to yield the most symbols.
_build() {
    git clean -fdx
    ./autogen.sh --enable-ld-gc --enable-ifcfg-rh --enable-ifupdown \
        --enable-ifnet --enable-ibft --enable-teamdctl --enable-wifi \
        --with-modem-manager-1 --with-ofono --with-more-asserts \
        --with-more-logging
    make -j20
}

_sort() {
    LANG=C sort -u
}

call_nm() {
    nm "$1" |
        sed -n 's/^................ \(.\) \(.*\)$/\1 \2/p'
}

get_symbols_nm () {
    call_nm ./src/NetworkManager |
        sed -n 's/^[tTDR] //p' |
        _sort
}

get_symbols_explict() {
    cat <<EOF | _sort
_IO_stdin_used
EOF
}

get_symbols_missing() {
    (for f in ./src/settings/plugins/*/.libs/*.so ./src/devices/*/.libs/*.so; do
        call_nm "$f" |
            sed -n 's/^\([U]\) \(\(nm_\|nmp_\|_nm\|NM\|_NM\).*\)$/\2/p'
    done) |
        _sort |
        grep -Fx -f <(get_symbols_explict) -v |
        grep -Fx -f <(get_symbols_nm)
}

pretty() {
    sed 's/.*/\t\0;/'
}

do_build() {
    do_update
    touch src/main.c
    make
}

do_rebuild() {
    _build
    do_build
}

do_update() {
    do_generate > ./src/NetworkManager.ver
}

do_generate() {
    cd "$(realpath $(dirname "$0"))/.."
    test -f ./src/NetworkManager

    cat <<EOF
# this file is generated by $0
{
global:
$(get_symbols_missing | pretty)

$(get_symbols_explict | pretty)
local:
	*;
};
EOF
}


case "$1" in
    rebuild)
        do_rebuild
        ;;
    build)
        do_build
        ;;
    update)
        do_update
        ;;
    *)
        do_generate
        ;;
esac

