/* $OpenBSD: tlslegacytest.c,v 1.6 2022/06/10 22:00:15 tb Exp $ */
/*
 * Copyright (c) 2015, 2016, 2017, 2020 Joel Sing <jsing@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <openssl/ssl.h>

#include <openssl/err.h>
#include <openssl/dtls1.h>
#include <openssl/ssl3.h>

#include <err.h>
#include <stdio.h>
#include <string.h>

/* openssl.org:443 */
static uint8_t tls12_server_response[] = {
	0x16, 0x03, 0x03, 0x00, 0x3d, 0x02, 0x00, 0x00,
	0x39, 0x03, 0x03, 0x62, 0x0c, 0x8a, 0x7e, 0x29,
	0x60, 0xcb, 0x08, 0xd1, 0xb4, 0x95, 0x68, 0x76,
	0xea, 0x4e, 0x0c, 0x94, 0xf2, 0x42, 0x3d, 0xd1,
	0x7a, 0xc2, 0xfe, 0x6c, 0xb3, 0xe6, 0x12, 0x8a,
	0x33, 0x02, 0x92, 0x00, 0xc0, 0x30, 0x00, 0x00,
	0x11, 0xff, 0x01, 0x00, 0x01, 0x00, 0x00, 0x0b,
	0x00, 0x04, 0x03, 0x00, 0x01, 0x02, 0x00, 0x23,
	0x00, 0x00,
};

/*
 * outlook.office365.com:587 with starttls - this server response includes
 * multiple handshake messages contained in a single TLS record.
 */
static uint8_t tls12_server_response_with_cert[] = {
	0x16, 0x03, 0x03, 0x0f, 0x2b, 0x02, 0x00, 0x00,
	0x4d, 0x03, 0x03, 0x5f, 0x7c, 0x69, 0x42, 0xe1,
	0x19, 0xf0, 0x22, 0xfb, 0x71, 0x9a, 0xf1, 0x63,
	0x34, 0xbb, 0x61, 0x46, 0xea, 0x5f, 0x0b, 0x5e,
	0xb1, 0x4e, 0x37, 0x96, 0x67, 0xff, 0x83, 0xea,
	0x0e, 0x16, 0x85, 0x20, 0x3a, 0x1b, 0x00, 0x00,
	0x17, 0xe9, 0xac, 0xca, 0x19, 0x61, 0xaf, 0x70,
	0x28, 0x3b, 0x18, 0xaa, 0x6c, 0xa0, 0x0f, 0x78,
	0xd0, 0x83, 0xfc, 0x5d, 0x78, 0xf9, 0x6d, 0xdb,
	0x16, 0x21, 0x15, 0xa2, 0xc0, 0x30, 0x00, 0x00,
	0x05, 0xff, 0x01, 0x00, 0x01, 0x00, 0x0b, 0x00,
	0x0d, 0x47, 0x00, 0x0d, 0x44, 0x00, 0x08, 0xaf,
	0x30, 0x82, 0x08, 0xab, 0x30, 0x82, 0x07, 0x93,
	0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x0c, 0x6d,
	0xea, 0x0b, 0xe1, 0x97, 0x27, 0x60, 0xa1, 0x59,
	0xb1, 0x85, 0x60, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b,
	0x05, 0x00, 0x30, 0x66, 0x31, 0x0b, 0x30, 0x09,
	0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x42,
	0x45, 0x31, 0x19, 0x30, 0x17, 0x06, 0x03, 0x55,
	0x04, 0x0a, 0x13, 0x10, 0x47, 0x6c, 0x6f, 0x62,
	0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x6e,
	0x76, 0x2d, 0x73, 0x61, 0x31, 0x3c, 0x30, 0x3a,
	0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x33, 0x47,
	0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69, 0x67,
	0x6e, 0x20, 0x4f, 0x72, 0x67, 0x61, 0x6e, 0x69,
	0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x56,
	0x61, 0x6c, 0x69, 0x64, 0x61, 0x74, 0x69, 0x6f,
	0x6e, 0x20, 0x43, 0x41, 0x20, 0x2d, 0x20, 0x53,
	0x48, 0x41, 0x32, 0x35, 0x36, 0x20, 0x2d, 0x20,
	0x47, 0x33, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x30,
	0x30, 0x38, 0x31, 0x33, 0x32, 0x33, 0x31, 0x38,
	0x34, 0x39, 0x5a, 0x17, 0x0d, 0x32, 0x32, 0x30,
	0x38, 0x31, 0x34, 0x32, 0x33, 0x31, 0x38, 0x34,
	0x39, 0x5a, 0x30, 0x6a, 0x31, 0x0b, 0x30, 0x09,
	0x06, 0x03, 0x55, 0x04, 0x06, 0x13, 0x02, 0x55,
	0x53, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55,
	0x04, 0x08, 0x13, 0x0a, 0x57, 0x61, 0x73, 0x68,
	0x69, 0x6e, 0x67, 0x74, 0x6f, 0x6e, 0x31, 0x10,
	0x30, 0x0e, 0x06, 0x03, 0x55, 0x04, 0x07, 0x13,
	0x07, 0x52, 0x65, 0x64, 0x6d, 0x6f, 0x6e, 0x64,
	0x31, 0x1e, 0x30, 0x1c, 0x06, 0x03, 0x55, 0x04,
	0x0a, 0x13, 0x15, 0x4d, 0x69, 0x63, 0x72, 0x6f,
	0x73, 0x6f, 0x66, 0x74, 0x20, 0x43, 0x6f, 0x72,
	0x70, 0x6f, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e,
	0x31, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x04,
	0x03, 0x13, 0x0b, 0x6f, 0x75, 0x74, 0x6c, 0x6f,
	0x6f, 0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x82,
	0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
	0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05,
	0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82,
	0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xc9,
	0x20, 0x3f, 0x57, 0xb9, 0xf9, 0x71, 0xaa, 0x3c,
	0x6a, 0x0a, 0x5d, 0x3f, 0xc9, 0x8d, 0x99, 0xa5,
	0x50, 0x26, 0x25, 0x4e, 0xdc, 0x69, 0x52, 0xb3,
	0x33, 0x70, 0xe7, 0x72, 0xa2, 0x83, 0x92, 0x54,
	0xd1, 0xd4, 0x86, 0x15, 0xf2, 0xc4, 0x65, 0xf8,
	0xbc, 0xe5, 0xd2, 0x1e, 0x12, 0x25, 0x9e, 0x75,
	0x8e, 0x77, 0xd2, 0x8e, 0x94, 0xca, 0x03, 0x4b,
	0xf4, 0xc8, 0xca, 0xe3, 0xe3, 0x9b, 0x66, 0xa3,
	0xa1, 0x37, 0x74, 0xcc, 0xfe, 0xc4, 0x1e, 0x64,
	0xdc, 0xe3, 0x18, 0xba, 0xc1, 0x7b, 0x39, 0x5b,
	0xb1, 0x47, 0xe9, 0x11, 0x92, 0xef, 0xee, 0xe6,
	0x08, 0xcd, 0x93, 0x7b, 0x09, 0xc7, 0x39, 0xfe,
	0xe5, 0xe2, 0x47, 0x3f, 0x68, 0x78, 0xa4, 0x17,
	0x78, 0x13, 0xcb, 0x12, 0x38, 0x9d, 0x89, 0x2b,
	0x1f, 0x75, 0x9b, 0x87, 0x5d, 0x53, 0xfc, 0xb0,
	0x2a, 0xaf, 0x2d, 0x86, 0x8a, 0x76, 0x3b, 0xce,
	0x5e, 0xae, 0x43, 0x74, 0x68, 0xc3, 0x28, 0xbf,
	0x10, 0x2f, 0xdd, 0xd9, 0x43, 0x4b, 0x2d, 0xa6,
	0xdc, 0x1f, 0x6d, 0x90, 0xd0, 0xce, 0x14, 0x1e,
	0x6c, 0xdc, 0x7b, 0x06, 0xe4, 0x7b, 0xa9, 0x81,
	0x40, 0xed, 0xde, 0x18, 0xb7, 0xdf, 0x53, 0x61,
	0xbc, 0x18, 0x83, 0x11, 0xc7, 0xb4, 0x1b, 0x99,
	0xef, 0x14, 0xe4, 0x63, 0x39, 0xe3, 0x5c, 0x2f,
	0xe7, 0x89, 0x58, 0x5b, 0xda, 0x03, 0x3a, 0x39,
	0x96, 0x8a, 0xca, 0x4f, 0xd8, 0xe3, 0x6c, 0x7f,
	0x6e, 0xd3, 0xe7, 0x30, 0x34, 0x9c, 0xdb, 0x8b,
	0xe8, 0x6a, 0xa6, 0x08, 0x77, 0x1d, 0x63, 0xd6,
	0x57, 0x9d, 0xcd, 0xa7, 0x47, 0x05, 0x39, 0x96,
	0x7b, 0xfd, 0x9a, 0x09, 0x99, 0xef, 0x49, 0xb1,
	0x89, 0x02, 0xbe, 0x4f, 0xb8, 0xef, 0xa0, 0x04,
	0x29, 0x74, 0xfb, 0x9a, 0x7e, 0x9d, 0xa8, 0x10,
	0xfb, 0x7e, 0xb0, 0x6c, 0x60, 0x4f, 0x57, 0x02,
	0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x05, 0x53,
	0x30, 0x82, 0x05, 0x4f, 0x30, 0x0e, 0x06, 0x03,
	0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04,
	0x03, 0x02, 0x05, 0xa0, 0x30, 0x81, 0x9e, 0x06,
	0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01,
	0x01, 0x04, 0x81, 0x91, 0x30, 0x81, 0x8e, 0x30,
	0x4b, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
	0x07, 0x30, 0x02, 0x86, 0x3f, 0x68, 0x74, 0x74,
	0x70, 0x3a, 0x2f, 0x2f, 0x73, 0x65, 0x63, 0x75,
	0x72, 0x65, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61,
	0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f,
	0x6d, 0x2f, 0x63, 0x61, 0x63, 0x65, 0x72, 0x74,
	0x2f, 0x67, 0x73, 0x6f, 0x72, 0x67, 0x61, 0x6e,
	0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x76,
	0x61, 0x6c, 0x73, 0x68, 0x61, 0x32, 0x67, 0x33,
	0x2e, 0x63, 0x72, 0x74, 0x30, 0x3f, 0x06, 0x08,
	0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x01,
	0x86, 0x33, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f,
	0x2f, 0x6f, 0x63, 0x73, 0x70, 0x32, 0x2e, 0x67,
	0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67,
	0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x73,
	0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61,
	0x74, 0x69, 0x6f, 0x6e, 0x76, 0x61, 0x6c, 0x73,
	0x68, 0x61, 0x32, 0x67, 0x33, 0x30, 0x56, 0x06,
	0x03, 0x55, 0x1d, 0x20, 0x04, 0x4f, 0x30, 0x4d,
	0x30, 0x41, 0x06, 0x09, 0x2b, 0x06, 0x01, 0x04,
	0x01, 0xa0, 0x32, 0x01, 0x14, 0x30, 0x34, 0x30,
	0x32, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
	0x07, 0x02, 0x01, 0x16, 0x26, 0x68, 0x74, 0x74,
	0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77,
	0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x73,
	0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
	0x72, 0x65, 0x70, 0x6f, 0x73, 0x69, 0x74, 0x6f,
	0x72, 0x79, 0x2f, 0x30, 0x08, 0x06, 0x06, 0x67,
	0x81, 0x0c, 0x01, 0x02, 0x02, 0x30, 0x09, 0x06,
	0x03, 0x55, 0x1d, 0x13, 0x04, 0x02, 0x30, 0x00,
	0x30, 0x46, 0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04,
	0x3f, 0x30, 0x3d, 0x30, 0x3b, 0xa0, 0x39, 0xa0,
	0x37, 0x86, 0x35, 0x68, 0x74, 0x74, 0x70, 0x3a,
	0x2f, 0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x67, 0x6c,
	0x6f, 0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e,
	0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x67, 0x73, 0x6f,
	0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74,
	0x69, 0x6f, 0x6e, 0x76, 0x61, 0x6c, 0x73, 0x68,
	0x61, 0x32, 0x67, 0x33, 0x2e, 0x63, 0x72, 0x6c,
	0x30, 0x82, 0x02, 0x10, 0x06, 0x03, 0x55, 0x1d,
	0x11, 0x04, 0x82, 0x02, 0x07, 0x30, 0x82, 0x02,
	0x03, 0x82, 0x0b, 0x6f, 0x75, 0x74, 0x6c, 0x6f,
	0x6f, 0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x16,
	0x2a, 0x2e, 0x63, 0x6c, 0x6f, 0x2e, 0x66, 0x6f,
	0x6f, 0x74, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x64,
	0x6e, 0x73, 0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x0d,
	0x2a, 0x2e, 0x68, 0x6f, 0x74, 0x6d, 0x61, 0x69,
	0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x16, 0x2a,
	0x2e, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x61,
	0x6c, 0x2e, 0x6f, 0x75, 0x74, 0x6c, 0x6f, 0x6f,
	0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x0a, 0x2a,
	0x2e, 0x6c, 0x69, 0x76, 0x65, 0x2e, 0x63, 0x6f,
	0x6d, 0x82, 0x16, 0x2a, 0x2e, 0x6e, 0x72, 0x62,
	0x2e, 0x66, 0x6f, 0x6f, 0x74, 0x70, 0x72, 0x69,
	0x6e, 0x74, 0x64, 0x6e, 0x73, 0x2e, 0x63, 0x6f,
	0x6d, 0x82, 0x0c, 0x2a, 0x2e, 0x6f, 0x66, 0x66,
	0x69, 0x63, 0x65, 0x2e, 0x63, 0x6f, 0x6d, 0x82,
	0x0f, 0x2a, 0x2e, 0x6f, 0x66, 0x66, 0x69, 0x63,
	0x65, 0x33, 0x36, 0x35, 0x2e, 0x63, 0x6f, 0x6d,
	0x82, 0x0d, 0x2a, 0x2e, 0x6f, 0x75, 0x74, 0x6c,
	0x6f, 0x6f, 0x6b, 0x2e, 0x63, 0x6f, 0x6d, 0x82,
	0x17, 0x2a, 0x2e, 0x6f, 0x75, 0x74, 0x6c, 0x6f,
	0x6f, 0x6b, 0x2e, 0x6f, 0x66, 0x66, 0x69, 0x63,
	0x65, 0x33, 0x36, 0x35, 0x2e, 0x63, 0x6f, 0x6d,
	0x82, 0x1b, 0x61, 0x74, 0x74, 0x61, 0x63, 0x68,
	0x6d, 0x65, 0x6e, 0x74, 0x2e, 0x6f, 0x75, 0x74,
	0x6c, 0x6f, 0x6f, 0x6b, 0x2e, 0x6c, 0x69, 0x76,
	0x65, 0x2e, 0x6e, 0x65, 0x74, 0x82, 0x1d, 0x61,
	0x74, 0x74, 0x61, 0x63, 0x68, 0x6d, 0x65, 0x6e,
	0x74, 0x2e, 0x6f, 0x75, 0x74, 0x6c, 0x6f, 0x6f,
	0x6b, 0x2e, 0x6f, 0x66, 0x66, 0x69, 0x63, 0x65,
	0x2e, 0x6e, 0x65, 0x74, 0x82, 0x20, 0x61, 0x74,
	0x74, 0x61, 0x63, 0x68, 0x6d, 0x65, 0x6e, 0x74,
	0x2e, 0x6f, 0x75, 0x74, 0x6c, 0x6f, 0x6f, 0x6b,
	0x2e, 0x6f, 0x66, 0x66, 0x69, 0x63, 0x65, 0x70,
	0x70, 0x65, 0x2e, 0x6e, 0x65, 0x74, 0x82, 0x16,
	0x61, 0x74, 0x74, 0x61, 0x63, 0x68, 0x6d, 0x65,
	0x6e, 0x74, 0x73, 0x2e, 0x6f, 0x66, 0x66, 0x69,
	0x63, 0x65, 0x2e, 0x6e, 0x65, 0x74, 0x82, 0x1a,
	0x61, 0x74, 0x74, 0x61, 0x63, 0x68, 0x6d, 0x65,
	0x6e, 0x74, 0x73, 0x2d, 0x73, 0x64, 0x66, 0x2e,
	0x6f, 0x66, 0x66, 0x69, 0x63, 0x65, 0x2e, 0x6e,
	0x65, 0x74, 0x82, 0x1d, 0x63, 0x63, 0x73, 0x2e,
	0x6c, 0x6f, 0x67, 0x69, 0x6e, 0x2e, 0x6d, 0x69,
	0x63, 0x72, 0x6f, 0x73, 0x6f, 0x66, 0x74, 0x6f,
	0x6e, 0x6c, 0x69, 0x6e, 0x65, 0x2e, 0x63, 0x6f,
	0x6d, 0x82, 0x21, 0x63, 0x63, 0x73, 0x2d, 0x73,
	0x64, 0x66, 0x2e, 0x6c, 0x6f, 0x67, 0x69, 0x6e,
	0x2e, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x73, 0x6f,
	0x66, 0x74, 0x6f, 0x6e, 0x6c, 0x69, 0x6e, 0x65,
	0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x0b, 0x68, 0x6f,
	0x74, 0x6d, 0x61, 0x69, 0x6c, 0x2e, 0x63, 0x6f,
	0x6d, 0x82, 0x16, 0x6d, 0x61, 0x69, 0x6c, 0x2e,
	0x73, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x73,
	0x2e, 0x6c, 0x69, 0x76, 0x65, 0x2e, 0x63, 0x6f,
	0x6d, 0x82, 0x0d, 0x6f, 0x66, 0x66, 0x69, 0x63,
	0x65, 0x33, 0x36, 0x35, 0x2e, 0x63, 0x6f, 0x6d,
	0x82, 0x12, 0x6f, 0x75, 0x74, 0x6c, 0x6f, 0x6f,
	0x6b, 0x2e, 0x6f, 0x66, 0x66, 0x69, 0x63, 0x65,
	0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x14, 0x73, 0x75,
	0x62, 0x73, 0x74, 0x72, 0x61, 0x74, 0x65, 0x2e,
	0x6f, 0x66, 0x66, 0x69, 0x63, 0x65, 0x2e, 0x63,
	0x6f, 0x6d, 0x82, 0x18, 0x73, 0x75, 0x62, 0x73,
	0x74, 0x72, 0x61, 0x74, 0x65, 0x2d, 0x73, 0x64,
	0x66, 0x2e, 0x6f, 0x66, 0x66, 0x69, 0x63, 0x65,
	0x2e, 0x63, 0x6f, 0x6d, 0x30, 0x1d, 0x06, 0x03,
	0x55, 0x1d, 0x25, 0x04, 0x16, 0x30, 0x14, 0x06,
	0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03,
	0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
	0x07, 0x03, 0x02, 0x30, 0x1f, 0x06, 0x03, 0x55,
	0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14,
	0x68, 0x86, 0xb8, 0x7d, 0x7a, 0xd9, 0x6d, 0x49,
	0x6b, 0x87, 0x2f, 0x18, 0x8b, 0x15, 0x34, 0x6c,
	0xd7, 0xb4, 0x7a, 0x0e, 0x30, 0x1d, 0x06, 0x03,
	0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x8a,
	0x7c, 0x73, 0x44, 0x70, 0xa8, 0x4d, 0x83, 0x25,
	0x6f, 0xa6, 0x53, 0xda, 0x42, 0x52, 0x96, 0xc9,
	0x15, 0x71, 0x21, 0x30, 0x82, 0x01, 0x7c, 0x06,
	0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0xd6, 0x79,
	0x02, 0x04, 0x02, 0x04, 0x82, 0x01, 0x6c, 0x04,
	0x82, 0x01, 0x68, 0x01, 0x66, 0x00, 0x76, 0x00,
	0x22, 0x45, 0x45, 0x07, 0x59, 0x55, 0x24, 0x56,
	0x96, 0x3f, 0xa1, 0x2f, 0xf1, 0xf7, 0x6d, 0x86,
	0xe0, 0x23, 0x26, 0x63, 0xad, 0xc0, 0x4b, 0x7f,
	0x5d, 0xc6, 0x83, 0x5c, 0x6e, 0xe2, 0x0f, 0x02,
	0x00, 0x00, 0x01, 0x73, 0xea, 0x1e, 0x7d, 0x2f,
	0x00, 0x00, 0x04, 0x03, 0x00, 0x47, 0x30, 0x45,
	0x02, 0x21, 0x00, 0xf4, 0x50, 0x8f, 0xe7, 0x38,
	0xc9, 0x7a, 0xd1, 0xf7, 0xf7, 0x69, 0xc5, 0x05,
	0xea, 0x8e, 0x03, 0x80, 0x2c, 0x87, 0x06, 0x03,
	0xb6, 0x9b, 0xe6, 0xa5, 0x83, 0x2f, 0xb9, 0xaf,
	0x7b, 0xb4, 0xac, 0x02, 0x20, 0x51, 0xa6, 0x8f,
	0xe8, 0xe5, 0x6c, 0xa7, 0xff, 0x16, 0x01, 0x7e,
	0x15, 0x42, 0x11, 0x31, 0xdc, 0xdc, 0xc7, 0x37,
	0x7c, 0x64, 0x2c, 0xac, 0xdd, 0x42, 0xbb, 0x3c,
	0x79, 0x31, 0x74, 0xcc, 0x9d, 0x00, 0x75, 0x00,
	0x29, 0x79, 0xbe, 0xf0, 0x9e, 0x39, 0x39, 0x21,
	0xf0, 0x56, 0x73, 0x9f, 0x63, 0xa5, 0x77, 0xe5,
	0xbe, 0x57, 0x7d, 0x9c, 0x60, 0x0a, 0xf8, 0xf9,
	0x4d, 0x5d, 0x26, 0x5c, 0x25, 0x5d, 0xc7, 0x84,
	0x00, 0x00, 0x01, 0x73, 0xea, 0x1e, 0x7a, 0xa7,
	0x00, 0x00, 0x04, 0x03, 0x00, 0x46, 0x30, 0x44,
	0x02, 0x20, 0x03, 0xf1, 0x19, 0xd7, 0x0f, 0x2f,
	0xc4, 0xa9, 0x84, 0xa0, 0x33, 0xd4, 0x76, 0xa6,
	0xee, 0xf1, 0xae, 0xe0, 0x03, 0xe7, 0xae, 0x98,
	0x43, 0x17, 0xb0, 0x0f, 0xfb, 0x12, 0xbb, 0x13,
	0xda, 0x34, 0x02, 0x20, 0x10, 0xe6, 0xa9, 0x1d,
	0x8b, 0x1c, 0x64, 0xd4, 0xc9, 0xf7, 0xc0, 0x3d,
	0x3c, 0x77, 0x49, 0xb1, 0x08, 0x3d, 0x1d, 0x5e,
	0x34, 0xf9, 0xd9, 0x10, 0x7c, 0x74, 0x6b, 0x18,
	0xc6, 0x5e, 0x6d, 0x07, 0x00, 0x75, 0x00, 0x55,
	0x81, 0xd4, 0xc2, 0x16, 0x90, 0x36, 0x01, 0x4a,
	0xea, 0x0b, 0x9b, 0x57, 0x3c, 0x53, 0xf0, 0xc0,
	0xe4, 0x38, 0x78, 0x70, 0x25, 0x08, 0x17, 0x2f,
	0xa3, 0xaa, 0x1d, 0x07, 0x13, 0xd3, 0x0c, 0x00,
	0x00, 0x01, 0x73, 0xea, 0x1e, 0x7d, 0xae, 0x00,
	0x00, 0x04, 0x03, 0x00, 0x46, 0x30, 0x44, 0x02,
	0x20, 0x26, 0x21, 0x64, 0xdb, 0xa6, 0xe2, 0x3d,
	0x32, 0x7d, 0x9f, 0xa8, 0xae, 0xb7, 0x29, 0xb7,
	0x42, 0x9b, 0x49, 0xaa, 0xf5, 0xa5, 0xc0, 0x12,
	0x01, 0xa1, 0xb6, 0xe7, 0xf2, 0x01, 0xd4, 0x2f,
	0x45, 0x02, 0x20, 0x4e, 0x19, 0xba, 0x47, 0x75,
	0x8b, 0x49, 0xd7, 0x4b, 0xba, 0x04, 0x62, 0xdd,
	0xa2, 0xb7, 0x6b, 0x05, 0xd0, 0x01, 0x1f, 0x7c,
	0x36, 0x17, 0x27, 0x29, 0xb2, 0x17, 0x1c, 0x7f,
	0x10, 0x81, 0x8a, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b,
	0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x26,
	0xf4, 0xa3, 0x77, 0x1d, 0xdc, 0x9e, 0xc4, 0x1a,
	0x88, 0x23, 0x30, 0x8c, 0xe1, 0x14, 0xf9, 0x62,
	0x0e, 0xbf, 0xad, 0x24, 0xc9, 0xab, 0xab, 0xd0,
	0x68, 0x8b, 0xbc, 0xf1, 0xec, 0x1c, 0xd0, 0x96,
	0xad, 0xf9, 0x5f, 0xdd, 0xe0, 0xee, 0xa8, 0xe0,
	0x2c, 0x3a, 0x19, 0xa5, 0x68, 0x0c, 0x6e, 0xfe,
	0xe6, 0x80, 0xce, 0xa3, 0x3b, 0x6c, 0x00, 0x88,
	0x5c, 0xbf, 0x3c, 0xd8, 0x68, 0x08, 0x36, 0xb9,
	0x9e, 0x84, 0x9b, 0x5f, 0x97, 0xfb, 0x77, 0xea,
	0x72, 0xfb, 0x73, 0x47, 0x00, 0xb0, 0xa8, 0x7c,
	0x64, 0x38, 0xf1, 0xcc, 0xc0, 0x29, 0x71, 0x67,
	0x65, 0x76, 0x4c, 0x80, 0x58, 0x97, 0xc8, 0x62,
	0x63, 0x3e, 0xf1, 0x3e, 0xc0, 0x0e, 0x48, 0x5f,
	0x55, 0x21, 0x8f, 0x96, 0x68, 0xbd, 0x41, 0x14,
	0x7a, 0x0b, 0x8c, 0x31, 0x5b, 0x39, 0xac, 0xa3,
	0xa0, 0x99, 0x58, 0x24, 0xfa, 0xd9, 0x19, 0x32,
	0x1c, 0x9f, 0x2d, 0xa9, 0xed, 0xb9, 0x97, 0xa4,
	0x66, 0x30, 0x29, 0xd8, 0x82, 0xa2, 0xf5, 0xfc,
	0x6d, 0x10, 0xf1, 0xac, 0x1d, 0x3f, 0xfb, 0xde,
	0xa1, 0x0e, 0xb6, 0x84, 0x90, 0xd4, 0x55, 0x5c,
	0x21, 0x1b, 0x1f, 0x21, 0x45, 0x92, 0xc5, 0x9a,
	0x47, 0x05, 0x0f, 0xb8, 0x1c, 0x78, 0x6e, 0xb9,
	0x6b, 0xa3, 0xa9, 0x8d, 0xb1, 0x59, 0xff, 0xf4,
	0xe6, 0x71, 0x77, 0x38, 0x12, 0xfe, 0x41, 0x8f,
	0x04, 0x92, 0x08, 0x3f, 0x32, 0x2a, 0x92, 0x5e,
	0x0a, 0x7b, 0x7e, 0x04, 0xee, 0x24, 0x10, 0x39,
	0xf3, 0xac, 0x5e, 0x04, 0x93, 0x91, 0xa2, 0x8f,
	0x90, 0x04, 0x33, 0x5c, 0x5c, 0x94, 0xb3, 0x80,
	0x2b, 0x43, 0xbf, 0xe3, 0x74, 0x64, 0x20, 0xf4,
	0x00, 0xb2, 0x6c, 0x7b, 0xa8, 0x77, 0xfb, 0x74,
	0x35, 0xce, 0xdd, 0xb6, 0x5f, 0x83, 0x18, 0xc4,
	0xe7, 0x31, 0x1a, 0x8d, 0x30, 0x0d, 0xc4, 0x00,
	0x04, 0x8f, 0x30, 0x82, 0x04, 0x8b, 0x30, 0x82,
	0x03, 0x73, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02,
	0x0e, 0x47, 0x07, 0xb1, 0x01, 0x9a, 0x0c, 0x57,
	0xad, 0x39, 0xb3, 0xe1, 0x7d, 0xa9, 0xf9, 0x30,
	0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
	0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x57,
	0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04,
	0x06, 0x13, 0x02, 0x42, 0x45, 0x31, 0x19, 0x30,
	0x17, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x10,
	0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53, 0x69,
	0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73, 0x61,
	0x31, 0x10, 0x30, 0x0e, 0x06, 0x03, 0x55, 0x04,
	0x0b, 0x13, 0x07, 0x52, 0x6f, 0x6f, 0x74, 0x20,
	0x43, 0x41, 0x31, 0x1b, 0x30, 0x19, 0x06, 0x03,
	0x55, 0x04, 0x03, 0x13, 0x12, 0x47, 0x6c, 0x6f,
	0x62, 0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20,
	0x52, 0x6f, 0x6f, 0x74, 0x20, 0x43, 0x41, 0x30,
	0x1e, 0x17, 0x0d, 0x31, 0x35, 0x30, 0x39, 0x30,
	0x34, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a,
	0x17, 0x0d, 0x32, 0x35, 0x30, 0x39, 0x30, 0x34,
	0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30,
	0x66, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x03, 0x55,
	0x04, 0x06, 0x13, 0x02, 0x42, 0x45, 0x31, 0x19,
	0x30, 0x17, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13,
	0x10, 0x47, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x53,
	0x69, 0x67, 0x6e, 0x20, 0x6e, 0x76, 0x2d, 0x73,
	0x61, 0x31, 0x3c, 0x30, 0x3a, 0x06, 0x03, 0x55,
	0x04, 0x03, 0x13, 0x33, 0x47, 0x6c, 0x6f, 0x62,
	0x61, 0x6c, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x4f,
	0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74,
	0x69, 0x6f, 0x6e, 0x20, 0x56, 0x61, 0x6c, 0x69,
	0x64, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x43,
	0x41, 0x20, 0x2d, 0x20, 0x53, 0x48, 0x41, 0x32,
	0x35, 0x36, 0x20, 0x2d, 0x20, 0x47, 0x33, 0x30,
	0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a,
	0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
	0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30,
	0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00,
	0xc9, 0x86, 0xa2, 0x05, 0x3e, 0xec, 0x77, 0x4d,
	0x79, 0x42, 0x81, 0xf2, 0xc5, 0x46, 0xa9, 0xc2,
	0x9b, 0xf9, 0x57, 0xa9, 0x48, 0xdd, 0x3c, 0x3b,
	0xe2, 0x16, 0x47, 0x83, 0x15, 0x0c, 0x36, 0x88,
	0x61, 0xb3, 0xc8, 0xb9, 0xd5, 0x20, 0x97, 0xb6,
	0xfe, 0x07, 0x30, 0x01, 0x9e, 0x01, 0x3a, 0xf9,
	0x50, 0x87, 0xa0, 0x4f, 0x60, 0xcc, 0x90, 0xf6,
	0xdd, 0x1f, 0xa6, 0xc7, 0x55, 0x00, 0x6c, 0x54,
	0x31, 0x5f, 0x02, 0x9a, 0xf7, 0x7f, 0x07, 0x9a,
	0xd2, 0x22, 0x53, 0x05, 0xcd, 0x9f, 0xc7, 0xbb,
	0x7b, 0x59, 0x3b, 0x8a, 0xb2, 0x93, 0x78, 0x0d,
	0x43, 0x02, 0x92, 0x76, 0xa5, 0x29, 0xf8, 0x7c,
	0x9d, 0x5c, 0x3a, 0xa2, 0xf8, 0x52, 0x72, 0x22,
	0x45, 0x91, 0xfd, 0x90, 0x12, 0x28, 0x4d, 0x75,
	0xe4, 0xdd, 0xaa, 0x79, 0x58, 0x68, 0x6f, 0x2a,
	0x7e, 0x7b, 0xef, 0xd1, 0x9e, 0x7f, 0x52, 0xdc,
	0xcb, 0x1c, 0x48, 0xe2, 0x3e, 0x4d, 0x5c, 0x47,
	0x7a, 0xb4, 0xf1, 0xce, 0xff, 0xd9, 0x60, 0x2b,
	0x77, 0xd1, 0x62, 0x22, 0x2d, 0xa9, 0x5a, 0x06,
	0x16, 0xee, 0x37, 0x6a, 0x51, 0xcf, 0x8e, 0xa5,
	0xd1, 0x6e, 0x70, 0x4a, 0xf0, 0xd8, 0x63, 0x60,
	0x6a, 0x72, 0x55, 0xd7, 0xf1, 0x99, 0x38, 0x86,
	0x44, 0x67, 0x18, 0xe0, 0x71, 0x8e, 0xc1, 0x40,
	0x6d, 0x85, 0xda, 0x4b, 0xdd, 0x31, 0x73, 0xbc,
	0x32, 0xcc, 0x6f, 0x8e, 0x7b, 0xb9, 0x8d, 0x4b,
	0x80, 0xda, 0xb9, 0xc7, 0xc6, 0x24, 0x83, 0x5e,
	0x32, 0xfb, 0x87, 0xe9, 0x8b, 0x61, 0x67, 0xa2,
	0x99, 0x76, 0xdb, 0xa5, 0xaa, 0xb4, 0xe8, 0x6c,
	0x41, 0x9f, 0x5f, 0x2a, 0xb3, 0xd5, 0x7d, 0xd7,
	0x92, 0xc8, 0x27, 0x4b, 0xec, 0x1f, 0xda, 0x05,
	0x6d, 0x88, 0x73, 0x8f, 0x06, 0xb2, 0x38, 0x3d,
	0x03, 0xa2, 0xe1, 0x87, 0x86, 0x3c, 0xc6, 0xa1,
	0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x01,
	0x44, 0x30, 0x82, 0x01, 0x40, 0x30, 0x0e, 0x06,
	0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04,
	0x04, 0x03, 0x02, 0x01, 0x06, 0x30, 0x1d, 0x06,
	0x03, 0x55, 0x1d, 0x25, 0x04, 0x16, 0x30, 0x14,
	0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
	0x03, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05,
	0x05, 0x07, 0x03, 0x02, 0x30, 0x12, 0x06, 0x03,
	0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08,
	0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x00,
	0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04,
	0x16, 0x04, 0x14, 0x68, 0x86, 0xb8, 0x7d, 0x7a,
	0xd9, 0x6d, 0x49, 0x6b, 0x87, 0x2f, 0x18, 0x8b,
	0x15, 0x34, 0x6c, 0xd7, 0xb4, 0x7a, 0x0e, 0x30,
	0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,
	0x30, 0x16, 0x80, 0x14, 0x60, 0x7b, 0x66, 0x1a,
	0x45, 0x0d, 0x97, 0xca, 0x89, 0x50, 0x2f, 0x7d,
	0x04, 0xcd, 0x34, 0xa8, 0xff, 0xfc, 0xfd, 0x4b,
	0x30, 0x3d, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05,
	0x05, 0x07, 0x01, 0x01, 0x04, 0x31, 0x30, 0x2f,
	0x30, 0x2d, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05,
	0x05, 0x07, 0x30, 0x01, 0x86, 0x21, 0x68, 0x74,
	0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6f, 0x63, 0x73,
	0x70, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c,
	0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f, 0x6d,
	0x2f, 0x72, 0x6f, 0x6f, 0x74, 0x72, 0x31, 0x30,
	0x33, 0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04, 0x2c,
	0x30, 0x2a, 0x30, 0x28, 0xa0, 0x26, 0xa0, 0x24,
	0x86, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f,
	0x2f, 0x63, 0x72, 0x6c, 0x2e, 0x67, 0x6c, 0x6f,
	0x62, 0x61, 0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e,
	0x63, 0x6f, 0x6d, 0x2f, 0x72, 0x6f, 0x6f, 0x74,
	0x2e, 0x63, 0x72, 0x6c, 0x30, 0x47, 0x06, 0x03,
	0x55, 0x1d, 0x20, 0x04, 0x40, 0x30, 0x3e, 0x30,
	0x3c, 0x06, 0x04, 0x55, 0x1d, 0x20, 0x00, 0x30,
	0x34, 0x30, 0x32, 0x06, 0x08, 0x2b, 0x06, 0x01,
	0x05, 0x05, 0x07, 0x02, 0x01, 0x16, 0x26, 0x68,
	0x74, 0x74, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x77,
	0x77, 0x77, 0x2e, 0x67, 0x6c, 0x6f, 0x62, 0x61,
	0x6c, 0x73, 0x69, 0x67, 0x6e, 0x2e, 0x63, 0x6f,
	0x6d, 0x2f, 0x72, 0x65, 0x70, 0x6f, 0x73, 0x69,
	0x74, 0x6f, 0x72, 0x79, 0x2f, 0x30, 0x0d, 0x06,
	0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01,
	0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01,
	0x00, 0x9a, 0xb9, 0x82, 0x1c, 0xdd, 0x83, 0x83,
	0x8b, 0x92, 0xc0, 0xc4, 0xed, 0x01, 0xad, 0x84,
	0xfc, 0x4e, 0xee, 0x6d, 0x9c, 0x1d, 0x01, 0xfa,
	0x52, 0x14, 0xdb, 0xd8, 0xc2, 0x10, 0x63, 0x9f,
	0x6b, 0x39, 0x9a, 0xc7, 0x1c, 0x3c, 0xa0, 0xaa,
	0xe3, 0x19, 0x3a, 0xfc, 0x64, 0x46, 0x2a, 0xef,
	0x35, 0x26, 0x03, 0xf6, 0x05, 0x67, 0xfa, 0x6e,
	0x74, 0xe1, 0x46, 0xfb, 0x40, 0xd8, 0x6f, 0xae,
	0x2d, 0x39, 0x21, 0x74, 0x86, 0x9f, 0x00, 0x05,
	0x1a, 0x3f, 0x2f, 0x93, 0x5b, 0xd4, 0xa4, 0x45,
	0xbc, 0x3d, 0x0c, 0x29, 0x17, 0x5a, 0xd3, 0xfb,
	0x68, 0xa6, 0x0f, 0xe0, 0x00, 0x68, 0x79, 0xb0,
	0x4c, 0xb1, 0x45, 0x8b, 0xc8, 0x85, 0x8c, 0x67,
	0x0e, 0x8c, 0x7d, 0x54, 0xf8, 0xb0, 0x75, 0xce,
	0x0a, 0xac, 0x1d, 0xd7, 0x6b, 0x44, 0xac, 0xfe,
	0x1b, 0xd4, 0xa6, 0x98, 0x21, 0x09, 0x3e, 0xa2,
	0x4b, 0x33, 0xba, 0xba, 0x4b, 0x12, 0xa8, 0x6b,
	0x57, 0x27, 0x9d, 0xfa, 0x94, 0x80, 0xb4, 0x68,
	0x4c, 0x77, 0x60, 0xff, 0xd7, 0x29, 0x5a, 0x38,
	0x3d, 0xce, 0x2d, 0x4b, 0x08, 0x56, 0x9f, 0x69,
	0xcb, 0x7b, 0xd8, 0xe2, 0x36, 0xf9, 0x37, 0x69,
	0xc5, 0xce, 0x36, 0x97, 0x1c, 0xba, 0x0d, 0x3f,
	0x15, 0xb3, 0x65, 0xa0, 0xec, 0x74, 0x12, 0xbd,
	0xb3, 0xad, 0xe8, 0xde, 0x9e, 0xa1, 0xec, 0xd3,
	0xbf, 0xa9, 0xe0, 0xa5, 0x91, 0x6d, 0x83, 0x59,
	0x12, 0x56, 0x2f, 0x13, 0xa6, 0x7e, 0x79, 0x73,
	0xa1, 0xa3, 0x89, 0xd5, 0xe1, 0xa5, 0x8c, 0xce,
	0x2d, 0xac, 0x8a, 0xcf, 0x62, 0x16, 0x65, 0xcd,
	0xd9, 0xee, 0xa8, 0xb6, 0x40, 0x08, 0xb5, 0x7c,
	0x50, 0xf9, 0x37, 0x82, 0x7a, 0xa4, 0x0b, 0x34,
	0x66, 0xec, 0xe9, 0x97, 0x57, 0x1f, 0x8a, 0x67,
	0x3e, 0x81, 0xbc, 0x3b, 0x35, 0xd3, 0x2a, 0x48,
	0x0c, 0x0c, 0x00, 0x01, 0x69, 0x03, 0x00, 0x18,
	0x61, 0x04, 0xb7, 0xa9, 0xbd, 0x74, 0x71, 0xd5,
	0x68, 0xbf, 0xd8, 0xa6, 0x84, 0x12, 0xaf, 0x8f,
	0xd4, 0x2c, 0xcf, 0xf9, 0x72, 0x2b, 0x8c, 0x6c,
	0x73, 0xa3, 0x13, 0x74, 0xdb, 0x83, 0x3e, 0xa6,
	0xf4, 0x1b, 0xee, 0xa9, 0x34, 0xe5, 0x65, 0xa7,
	0xaf, 0xef, 0xf2, 0xac, 0xfb, 0x87, 0xb4, 0xdb,
	0x8b, 0x05, 0x4f, 0xe8, 0x25, 0x3d, 0x32, 0x65,
	0xda, 0x47, 0xd8, 0xd2, 0x86, 0xad, 0x9b, 0x37,
	0xbc, 0x45, 0xef, 0xb6, 0x91, 0xa2, 0x71, 0x2f,
	0x13, 0x68, 0xfa, 0xa7, 0x20, 0xe4, 0x8a, 0xa8,
	0x9b, 0xbe, 0xf6, 0x7c, 0xc8, 0x16, 0xd4, 0x50,
	0x9d, 0x63, 0xb3, 0xf4, 0x6e, 0xd3, 0x8f, 0x32,
	0x68, 0x66, 0x04, 0x01, 0x01, 0x00, 0xaa, 0xcb,
	0x90, 0xbd, 0x94, 0x10, 0xab, 0xfc, 0x30, 0x1d,
	0x68, 0x1c, 0xb4, 0x21, 0xcf, 0x73, 0xa5, 0x4b,
	0x20, 0x94, 0xde, 0x66, 0x99, 0x54, 0x3f, 0xba,
	0x40, 0x58, 0x50, 0xe3, 0x64, 0x53, 0x90, 0x9e,
	0xf8, 0x67, 0xcc, 0x85, 0x4a, 0xdc, 0xd8, 0xd7,
	0xc8, 0xb5, 0xe0, 0x92, 0x02, 0x6b, 0xa8, 0x76,
	0x67, 0xc5, 0xae, 0x12, 0x56, 0xff, 0xd1, 0xda,
	0xc0, 0x48, 0x17, 0x99, 0xc9, 0xbe, 0x02, 0xc6,
	0x9e, 0x5c, 0xd9, 0x44, 0x3f, 0x06, 0xbd, 0x98,
	0xe3, 0x4d, 0x46, 0x10, 0xe8, 0x20, 0xed, 0x7b,
	0xcd, 0x73, 0xed, 0x03, 0x6a, 0x4c, 0x49, 0xaf,
	0xbe, 0xa3, 0xe0, 0xab, 0x9a, 0xb8, 0xf8, 0x06,
	0x25, 0x31, 0x8d, 0x32, 0x44, 0xfd, 0xd6, 0xb0,
	0xd4, 0x6c, 0x9a, 0x2a, 0x0f, 0xab, 0xe2, 0x13,
	0x10, 0x6d, 0x41, 0x0b, 0x97, 0x74, 0xa0, 0x04,
	0x16, 0x60, 0xf1, 0x8e, 0x74, 0xf3, 0x91, 0x75,
	0x2b, 0x92, 0x2b, 0xc7, 0x5b, 0x6f, 0x1d, 0x70,
	0xe2, 0xc6, 0x9a, 0x7d, 0x66, 0x55, 0x98, 0x01,
	0x71, 0xb8, 0xdd, 0xf4, 0x70, 0xc9, 0x74, 0x56,
	0xcc, 0xa5, 0x2c, 0x51, 0x70, 0x72, 0xc2, 0x44,
	0xb9, 0x59, 0xc3, 0xc3, 0xf8, 0x29, 0x4e, 0x79,
	0x40, 0x9b, 0x30, 0x35, 0x66, 0xb2, 0xd8, 0x7d,
	0xfe, 0x65, 0x6b, 0xf0, 0x17, 0xa3, 0x13, 0xc7,
	0xc7, 0xc6, 0x48, 0xb2, 0xae, 0x4f, 0x26, 0x0b,
	0x8a, 0x40, 0xaa, 0x06, 0x65, 0x8a, 0x95, 0x00,
	0xc4, 0xc9, 0xfd, 0x69, 0x0a, 0xa9, 0x0a, 0x18,
	0xff, 0x95, 0x40, 0xab, 0x84, 0x75, 0xfe, 0x11,
	0xb1, 0x6f, 0xca, 0x5e, 0xf7, 0xe4, 0x1d, 0x8d,
	0x08, 0x1c, 0xd3, 0x95, 0xf4, 0x9b, 0x17, 0x41,
	0xa8, 0x8f, 0x6e, 0xfa, 0x6c, 0x43, 0x60, 0x39,
	0x0a, 0xa2, 0x7e, 0xdf, 0x3e, 0x74, 0xc2, 0xbf,
	0xaf, 0x96, 0x96, 0xbd, 0x21, 0x4b, 0x0d, 0x00,
	0x00, 0x1a, 0x03, 0x01, 0x02, 0x40, 0x00, 0x12,
	0x04, 0x01, 0x05, 0x01, 0x02, 0x01, 0x04, 0x03,
	0x05, 0x03, 0x02, 0x03, 0x02, 0x02, 0x06, 0x01,
	0x06, 0x03, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
};

struct tlslegacy_client_test {
	const unsigned char *desc;
	unsigned char *server_response;
	const size_t server_response_len;
	const SSL_METHOD *(*ssl_method)(void);
	int want_state;
};

static struct tlslegacy_client_test tlslegacy_client_tests[] = {
	{
		.desc = "TLSv1.2 legacy fallback",
		.server_response = tls12_server_response,
		.server_response_len = sizeof(tls12_server_response),
		.ssl_method = TLS_client_method,
		.want_state = SSL3_ST_CR_CERT_A,
	},
	{
		.desc = "TLSv1.2 legacy fallback with server cert",
		.server_response = tls12_server_response_with_cert,
		.server_response_len = sizeof(tls12_server_response_with_cert),
		.ssl_method = TLS_client_method,
		.want_state = SSL3_ST_CR_KEY_EXCH_B,
	},
};

#define N_TLSLEGACY_CLIENT_TESTS \
    (sizeof(tlslegacy_client_tests) / sizeof(*tlslegacy_client_tests))

static int
tlslegacy_client_test(int testno, struct tlslegacy_client_test *tct)
{
	BIO *rbio = NULL, *wbio = NULL;
	SSL_CTX *ssl_ctx = NULL;
	SSL *ssl = NULL;
	int ret = 1;

	fprintf(stderr, "Test %d - %s\n", testno, tct->desc);

	if ((rbio = BIO_new_mem_buf(tct->server_response,
	    tct->server_response_len)) == NULL) {
		fprintf(stderr, "Failed to setup rbio\n");
		goto failure;
	}
	if ((wbio = BIO_new(BIO_s_mem())) == NULL) {
		fprintf(stderr, "Failed to setup wbio\n");
		goto failure;
	}

	if ((ssl_ctx = SSL_CTX_new(tct->ssl_method())) == NULL) {
		fprintf(stderr, "SSL_CTX_new() returned NULL\n");
		goto failure;
	}

	if ((ssl = SSL_new(ssl_ctx)) == NULL) {
		fprintf(stderr, "SSL_new() returned NULL\n");
		goto failure;
	}

	BIO_up_ref(rbio);
	BIO_up_ref(wbio);
	SSL_set_bio(ssl, rbio, wbio);

	if (SSL_connect(ssl) == 1) {
		fprintf(stderr, "SSL_connect() succeeded\n");
		goto failure;
	}

	if (SSL_state(ssl) != tct->want_state) {
		fprintf(stderr, "FAIL: Got SSL state %x, want %x",
		    SSL_state(ssl), tct->want_state);
		goto failure;
	}

	ret = 0;

 failure:
	SSL_CTX_free(ssl_ctx);
	SSL_free(ssl);

	BIO_free(rbio);
	BIO_free(wbio);

	return (ret);
}

int
main(int argc, char **argv)
{
	int failed = 0;
	size_t i;

	for (i = 0; i < N_TLSLEGACY_CLIENT_TESTS; i++)
		failed |= tlslegacy_client_test(i, &tlslegacy_client_tests[i]);

	return (failed);
}
