/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.managers;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Objects;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Base64Url;
import org.keycloak.common.util.Time;
import org.keycloak.cookie.CookieProvider;
import org.keycloak.cookie.CookieType;
import org.keycloak.crypto.SignatureProvider;
import org.keycloak.crypto.SignatureSignerContext;
import org.keycloak.forms.login.LoginFormsProvider;
import org.keycloak.jose.jws.crypto.HashUtils;
import org.keycloak.models.ClientModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.models.UserSessionProvider;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.models.utils.SessionExpiration;
import org.keycloak.protocol.RestartLoginCookie;
import org.keycloak.services.managers.AuthSessionId;
import org.keycloak.services.managers.AuthenticationManager;
import org.keycloak.sessions.AuthenticationSessionModel;
import org.keycloak.sessions.RootAuthenticationSessionModel;
import org.keycloak.sessions.StickySessionEncoderProvider;

public class AuthenticationSessionManager {
    private static final Logger log = Logger.getLogger(AuthenticationSessionManager.class);
    private final KeycloakSession session;

    public AuthenticationSessionManager(KeycloakSession session) {
        this.session = session;
    }

    public RootAuthenticationSessionModel createAuthenticationSession(RealmModel realm, boolean browserCookie) {
        RootAuthenticationSessionModel rootAuthSession = this.session.authenticationSessions().createRootAuthenticationSession(realm);
        if (browserCookie) {
            this.setAuthSessionCookie(rootAuthSession.getId());
            this.setAuthSessionIdHashCookie(rootAuthSession.getId());
        }
        return rootAuthSession;
    }

    public RootAuthenticationSessionModel getCurrentRootAuthenticationSession(RealmModel realm) {
        String oldEncodedId = this.getAuthSessionCookies(realm);
        if (oldEncodedId == null) {
            return null;
        }
        AuthSessionId authSessionId = this.decodeAuthSessionId(oldEncodedId);
        String sessionId = authSessionId.getDecodedId();
        RootAuthenticationSessionModel rootAuthSession = this.session.authenticationSessions().getRootAuthenticationSession(realm, sessionId);
        if (rootAuthSession != null) {
            this.reencodeAuthSessionCookie(oldEncodedId, authSessionId, realm);
            return rootAuthSession;
        }
        return null;
    }

    public AuthenticationSessionModel getCurrentAuthenticationSession(RealmModel realm, ClientModel client, String tabId) {
        String oldEncodedId = this.getAuthSessionCookies(realm);
        if (oldEncodedId == null) {
            return null;
        }
        AuthSessionId authSessionId = this.decodeAuthSessionId(oldEncodedId);
        String sessionId = authSessionId.getDecodedId();
        AuthenticationSessionModel authSession = this.getAuthenticationSessionByIdAndClient(realm, sessionId, client, tabId);
        if (authSession != null) {
            this.reencodeAuthSessionCookie(oldEncodedId, authSessionId, realm);
            return authSession;
        }
        return null;
    }

    public void setAuthSessionCookie(String authSessionId) {
        StickySessionEncoderProvider encoder = (StickySessionEncoderProvider)this.session.getProvider(StickySessionEncoderProvider.class);
        String signedAuthSessionId = this.signAndEncodeToBase64AuthSessionId(authSessionId);
        String encodedWithRoute = encoder.encodeSessionId(signedAuthSessionId);
        ((CookieProvider)this.session.getProvider(CookieProvider.class)).set(CookieType.AUTH_SESSION_ID, encodedWithRoute);
        log.debugf("Set AUTH_SESSION_ID cookie with value %s", (Object)encodedWithRoute);
    }

    public void setAuthSessionIdHashCookie(String authSessionId) {
        String authSessionIdHash = Base64.getEncoder().withoutPadding().encodeToString(HashUtils.hash((String)"SHA-256", (byte[])authSessionId.getBytes(StandardCharsets.UTF_8)));
        ((CookieProvider)this.session.getProvider(CookieProvider.class)).set(CookieType.AUTH_SESSION_ID_HASH, authSessionIdHash);
        log.debugf("Set KC_AUTH_SESSION_HASH cookie with value %s", (Object)authSessionIdHash);
    }

    AuthSessionId decodeAuthSessionId(String encodedAuthSessionId) {
        log.debugf("Found AUTH_SESSION_ID cookie with value %s", (Object)encodedAuthSessionId);
        StickySessionEncoderProvider encoder = (StickySessionEncoderProvider)this.session.getProvider(StickySessionEncoderProvider.class);
        String decodedAuthSessionId = encoder.decodeSessionId(encodedAuthSessionId);
        String reencoded = encoder.encodeSessionId(decodedAuthSessionId);
        if (!KeycloakModelUtils.isValidUUID((String)decodedAuthSessionId)) {
            decodedAuthSessionId = this.decodeBase64AndValidateSignature(decodedAuthSessionId, false);
        }
        return new AuthSessionId(decodedAuthSessionId, reencoded);
    }

    void reencodeAuthSessionCookie(String oldEncodedAuthSessionId, AuthSessionId newAuthSessionId, RealmModel realm) {
        if (!oldEncodedAuthSessionId.equals(newAuthSessionId.getEncodedId())) {
            log.debugf("Route changed. Will update authentication session cookie. Old: '%s', New: '%s'", (Object)oldEncodedAuthSessionId, (Object)newAuthSessionId.getEncodedId());
            this.setAuthSessionCookie(newAuthSessionId.getDecodedId());
        }
    }

    public String decodeBase64AndValidateSignature(String encodedBase64AuthSessionId, boolean validate) {
        try {
            String decodedAuthSessionId = new String(Base64Url.decode((String)encodedBase64AuthSessionId), StandardCharsets.UTF_8);
            if (decodedAuthSessionId.lastIndexOf(".") != -1) {
                String authSessionId = decodedAuthSessionId.substring(0, decodedAuthSessionId.indexOf("."));
                String signature = decodedAuthSessionId.substring(decodedAuthSessionId.indexOf(".") + 1);
                return validate ? this.validateAuthSessionIdSignature(authSessionId, signature) : authSessionId;
            }
        }
        catch (Exception e) {
            log.errorf("Error decoding auth session id with value: %s", (Object)encodedBase64AuthSessionId, (Object)e);
        }
        return null;
    }

    private String validateAuthSessionIdSignature(String authSessionId, String signature) {
        if (signature.equals(this.session.getAttribute(authSessionId))) {
            return authSessionId;
        }
        SignatureProvider signatureProvider = (SignatureProvider)this.session.getProvider(SignatureProvider.class, "HS512");
        SignatureSignerContext signer = signatureProvider.signer();
        try {
            boolean valid = signatureProvider.verifier(signer.getKid()).verify(authSessionId.getBytes(StandardCharsets.UTF_8), Base64Url.decode((String)signature));
            if (!valid) {
                return null;
            }
            this.session.setAttribute(authSessionId, (Object)signature);
            return authSessionId;
        }
        catch (Exception e) {
            log.errorf("Signature validation failed for auth session id: %s", (Object)authSessionId, (Object)e);
            return null;
        }
    }

    private String signAndEncodeToBase64AuthSessionId(String authSessionId) {
        SignatureProvider signatureProvider = (SignatureProvider)this.session.getProvider(SignatureProvider.class, "HS512");
        SignatureSignerContext signer = signatureProvider.signer();
        StringBuilder buffer = new StringBuilder();
        byte[] signature = signer.sign(authSessionId.getBytes(StandardCharsets.UTF_8));
        buffer.append(authSessionId);
        if (signature != null) {
            buffer.append('.');
            buffer.append(Base64Url.encode((byte[])signature));
        }
        return Base64Url.encode((byte[])buffer.toString().getBytes(StandardCharsets.UTF_8));
    }

    String getAuthSessionCookies(RealmModel realm) {
        String oldEncodedId = ((CookieProvider)this.session.getProvider(CookieProvider.class)).get(CookieType.AUTH_SESSION_ID);
        if (oldEncodedId == null || oldEncodedId.isEmpty()) {
            return null;
        }
        StickySessionEncoderProvider routeEncoder = (StickySessionEncoderProvider)this.session.getProvider(StickySessionEncoderProvider.class);
        String decodedAuthSessionId = routeEncoder.decodeSessionId(oldEncodedId);
        if ((decodedAuthSessionId = this.decodeBase64AndValidateSignature(decodedAuthSessionId, true)) == null) {
            return null;
        }
        RootAuthenticationSessionModel rootAuthenticationSession = this.session.authenticationSessions().getRootAuthenticationSession(realm, decodedAuthSessionId);
        return rootAuthenticationSession != null ? oldEncodedId : null;
    }

    public void removeAuthenticationSession(RealmModel realm, AuthenticationSessionModel authSession, boolean expireRestartCookie) {
        RootAuthenticationSessionModel rootAuthSession = authSession.getParentSession();
        log.debugf("Removing root authSession '%s'. Expire restart cookie: %b", (Object)rootAuthSession.getId(), (Object)expireRestartCookie);
        this.session.authenticationSessions().removeRootAuthenticationSession(realm, rootAuthSession);
        if (expireRestartCookie) {
            RestartLoginCookie.expireRestartCookie(this.session);
            ((LoginFormsProvider)this.session.getProvider(LoginFormsProvider.class)).setDetachedAuthSession();
        }
    }

    public boolean removeTabIdInAuthenticationSession(RealmModel realm, AuthenticationSessionModel authSession) {
        RootAuthenticationSessionModel rootAuthSession = authSession.getParentSession();
        rootAuthSession.removeAuthenticationSessionByTabId(authSession.getTabId());
        if (rootAuthSession.getAuthenticationSessions().isEmpty()) {
            this.removeAuthenticationSession(realm, authSession, true);
            return true;
        }
        return false;
    }

    public void updateAuthenticationSessionAfterSuccessfulAuthentication(RealmModel realm, AuthenticationSessionModel authSession) {
        boolean removedRootAuthSession = this.removeTabIdInAuthenticationSession(realm, authSession);
        if (!removedRootAuthSession) {
            if (realm.getSsoSessionIdleTimeout() < SessionExpiration.getAuthSessionLifespan((RealmModel)realm) && realm.getSsoSessionMaxLifespan() < SessionExpiration.getAuthSessionLifespan((RealmModel)realm)) {
                this.removeAuthenticationSession(realm, authSession, true);
            } else {
                RootAuthenticationSessionModel rootAuthSession = authSession.getParentSession();
                int authSessionExpiresIn = realm.getAccessCodeLifespan();
                int authSessionExpirationTime = Time.currentTime() - SessionExpiration.getAuthSessionLifespan((RealmModel)realm) + authSessionExpiresIn;
                rootAuthSession.setTimestamp(authSessionExpirationTime);
                log.tracef("Removed authentication session of root session '%s' with tabId '%s'. But there are remaining tabs in the root session. Root authentication session will expire in %d seconds", (Object)rootAuthSession.getId(), (Object)authSession.getTabId(), (Object)authSessionExpiresIn);
            }
        }
    }

    public UserSessionModel getUserSession(AuthenticationSessionModel authSession) {
        return this.getUserSessionProvider().getUserSession(authSession.getRealm(), authSession.getParentSession().getId());
    }

    public AuthenticationSessionModel getAuthenticationSessionByIdAndClient(RealmModel realm, String authSessionId, ClientModel client, String tabId) {
        RootAuthenticationSessionModel rootAuthSession = this.session.authenticationSessions().getRootAuthenticationSession(realm, authSessionId);
        return rootAuthSession == null ? null : rootAuthSession.getAuthenticationSession(client, tabId);
    }

    public AuthenticationSessionModel getAuthenticationSessionByEncodedIdAndClient(RealmModel realm, String encodedAuthSesionId, ClientModel client, String tabId) {
        String decodedAuthSessionId = this.decodeBase64AndValidateSignature(encodedAuthSesionId, true);
        return this.getAuthenticationSessionByIdAndClient(realm, decodedAuthSessionId, client, tabId);
    }

    public UserSessionModel getUserSessionFromAuthenticationCookie(RealmModel realm) {
        String oldEncodedId = this.getAuthSessionCookies(realm);
        if (oldEncodedId == null) {
            AuthenticationManager.AuthResult authResult = AuthenticationManager.authenticateIdentityCookie(this.session, realm, true);
            if (authResult != null && authResult.getSession() != null) {
                oldEncodedId = authResult.getSession().getId();
            } else {
                return null;
            }
        }
        AuthSessionId authSessionId = this.decodeAuthSessionId(oldEncodedId);
        String sessionId = authSessionId.getDecodedId();
        UserSessionProvider userSessionProvider = this.getUserSessionProvider();
        userSessionProvider.getUserSessionWithPredicate(realm, sessionId, false, Objects::isNull);
        UserSessionModel userSession = userSessionProvider.getUserSession(realm, sessionId);
        if (userSession != null) {
            this.reencodeAuthSessionCookie(oldEncodedId, authSessionId, realm);
            return userSession;
        }
        return null;
    }

    private UserSessionProvider getUserSessionProvider() {
        return this.session.sessions();
    }
}

