#include "mac68k.h"
#include "dixstruct.h"
#include "dix.h"
#include "opaque.h"
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include "colormapst.h"

/*
 * XXX These #define's are copied from cfb/cfbcmap.c and modified when
 * appropriate.  Probably should have its own cfbcmap.c, instead.
 */

/*
 * XC cfb/cfbcmap.c assumes that in Direct/True/StaticColor visuals, a
 * pixel's R, G, and B values are stored from LSB to MSB, but on the
 * Macintosh, this order is reversed; i.e. R, G, & B values are stored
 * from MSB to LSB.  It also assumes that B value takes whatever left
 * in the pixel unused by the other two values.  This isn't the case
 * on the Macintosh, either; B value has the same size as the other
 * two.  So we have to correct visual->{red,green,blue}Mask and
 * visual->offset{Red,Green,Blue} after cfbInitVisuals() has been
 * called and completed.
 *
 * MAC_[RGB]Z() is the bit length of the [RGB] value in a pixel.
 * MAC_[RGB]S() is the bit number of the LSB of the value (0 is LSB).
 * MAC_[RGB]M() is the maximum value's bit pattern in a pixel.
 */
#define MAC_BZ(d) ((d == 32) ? 8 : (d / 3))
#define MAC_BS(d) 0
#define MAC_BM(d) (((1 << MAC_BZ(d)) - 1) << MAC_BS(d))
#define MAC_GZ(d) ((d == 32) ? 8 : (d / 3))
#define MAC_GS(d) MAC_BZ(d)
#define MAC_GM(d) (((1 << MAC_GZ(d)) - 1) << MAC_GS(d))
#define MAC_RZ(d) ((d == 32) ? 8 : (d / 3))
#define MAC_RS(d) (MAC_BZ(d) + MAC_GZ(d))
#define MAC_RM(d) (((1 << MAC_RZ(d)) - 1) << MAC_RS(d))

#define StaticGrayMask	(1 << StaticGray)
#define GrayScaleMask	(1 << GrayScale)
#define StaticColorMask	(1 << StaticColor)
#define PseudoColorMask	(1 << PseudoColor)
#define TrueColorMask	(1 << TrueColor)
#define DirectColorMask (1 << DirectColor)

#define ALL_VISUALS	(StaticGrayMask|\
			 GrayScaleMask|\
			 StaticColorMask|\
			 PseudoColorMask|\
			 TrueColorMask|\
			 DirectColorMask)

#define SMALL_VISUALS	(StaticGrayMask|\
			 GrayScaleMask|\
			 StaticColorMask|\
			 PseudoColorMask)

#define GRAY_VISUALS	(StaticGrayMask|\
			 GrayScaleMask)

#define LARGE_VISUALS	(TrueColorMask|\
			 DirectColorMask)

/* #include "../../cfb/cfb.h" */

/* #include <machine/grfioctl.h> */

#define DEBUG
#undef DEBUG_CLUT

static ColormapPtr InstalledMaps[MAXSCREENS];

#ifdef DEBUG_CLUT
static FILE *logfp = NULL;
#endif

void
mac68kUpdateColormap(pScreen, cmap)
	ScreenPtr pScreen;
	ColormapPtr cmap;
{
	register int i;
	register Entry *pent;
	ColorSpec Map[256];
	VDEntry vde;
	register VisualPtr pVisual = cmap->pVisual;
	int     index = cmap->pScreen->myNum;

#ifdef DEBUG_CLUT
	fprintf(logfp, "*** Screen %d ***\n", index);
#endif

	if ((pVisual->class | DynamicClass) == DirectColor) {
		for (i = 0; i < 256; i++) {
			Map[i].value = i;
			Map[i].color.red = cmap->red[i].co.local.red;
			Map[i].color.green = cmap->green[i].co.local.green;
			Map[i].color.blue = cmap->blue[i].co.local.blue;
#ifdef DEBUG_CLUT
			fprintf(logfp, "W: %d, %x, %x, %x\n", i,
			    Map[i].color.red, Map[i].color.green, Map[i].color.blue);
#endif
		}
	} else {
		/* PseudoColor */
		for (i = 0, pent = cmap->red;
		    i < pVisual->ColormapEntries;
		    i++, pent++) {
			if (pent->fShared) {
				Map[i].value = i;
				Map[i].color.red = pent->co.shco.red->color;
				Map[i].color.green = pent->co.shco.green->color;
				Map[i].color.blue = pent->co.shco.blue->color;
			} else {
				Map[i].value = i;
				Map[i].color.red = pent->co.local.red;
				Map[i].color.green = pent->co.local.green;
				Map[i].color.blue = pent->co.local.blue;
			}
#ifdef DEBUG_CLUT
			fprintf(logfp, "W: %d, %x, %x, %x\n", i,
			    Map[i].color.red, Map[i].color.green, Map[i].color.blue);
#endif
		}
		for (; i < 256; i++) {
			Map[i].value = i;
		}
	}

	vde.csTable = Map;
	vde.csCount = 255;
	vde.csStart = 0;
	vde.length = sizeof Map;
	ioctl(mac_fbs[pScreen->myNum].fd, GRFIOCSETENTRIES, &vde);
}

void
mac68kStoreColors(pmap, ndef, pdefs)
	ColormapPtr pmap;
	int     ndef;
	xColorItem *pdefs;
{
	if (InstalledMaps[pmap->pScreen->myNum] == pmap) {
		mac68kUpdateColormap(pmap->pScreen, pmap);
	}
}

Bool
mac68kCreateColormap(pmap)
	ColormapPtr pmap;
{
	register VisualPtr pVisual;

	pVisual = pmap->pVisual;

	if (pVisual->class == StaticGray) {
		register unsigned i;
		unsigned lim, maxent, shift;

		lim = (1 << pVisual->bitsPerRGBValue) - 1;
		shift = 16 - pVisual->bitsPerRGBValue;
		maxent = pVisual->ColormapEntries - 1;

		/* Mac in gray mode for whatever reason uses 0 for the
		   brightest (white) and all 1s for the darkest
		   (black) - IOW, the greater the pixel value is, the
		   darker the pixel on screen appears. */
		for (i = 0; i <= maxent; i++)
			/* rescale to [0..65535] then rgb bits */
			pmap->red[maxent - i].co.local.green =
				pmap->red[maxent - i].co.local.blue =
				pmap->red[maxent - i].co.local.red =
				((((i * 65535) / maxent) >> shift) * 65535)
				/ lim;
		pmap->pScreen->whitePixel = 0;
		pmap->pScreen->blackPixel = maxent;
		return TRUE;
	}
	return cfbInitializeColormap(pmap);
}

Bool
mac68kColorSave(screen, on)
	ScreenPtr screen;
	int     on;
{
	/* if(on == SCREEN_SAVER_FORCER) Do something with event time */
	return (FALSE);
}

int
mac68kListInstalledColormaps(pScreen, pCmapList)
	ScreenPtr pScreen;
	Colormap *pCmapList;
{
	/* By the time we are processing requests, we can guarantee that there
	 * is always a colormap installed */
	*pCmapList = InstalledMaps[pScreen->myNum]->mid;
	return (1);
}

void
mac68kInstallColormap(cmap)
	ColormapPtr cmap;
{
	int     index = cmap->pScreen->myNum;
	ColormapPtr oldpmap = InstalledMaps[index];

	if (cmap == InstalledMaps[index]) {
		return;
	}

	if (mac_fbs[index].flags & MAC68K_FBFLAG_ROMDRIVER) {
		/* we have the means; really update the hardware colormap */
		mac68kUpdateColormap(cmap->pScreen, cmap);
	}

	if (cmap != oldpmap) {
		/* Uninstall pInstalledMap. No hardware changes required, just
		 * notify all interested parties. */
		if (oldpmap != (ColormapPtr) None)
			WalkTree(cmap->pScreen, TellLostMap,
				 (char *)&oldpmap->mid);
		/* Install pmap */
		InstalledMaps[index] = cmap;
		WalkTree(cmap->pScreen, TellGainedMap, (char *)&cmap->mid);
	}
}

void
mac68kUninstallColormap(pmap)
	ColormapPtr pmap;
{
	int     index = pmap->pScreen->myNum;
	ColormapPtr curpmap = InstalledMaps[index];

	if (pmap == curpmap) {
		if (pmap->mid != pmap->pScreen->defColormap) {
			curpmap = (ColormapPtr)LookupIDByType(
				pmap->pScreen->defColormap, RT_COLORMAP);
			(*pmap->pScreen->InstallColormap) (curpmap);
		}
	}
}



static void
nullStoreColors(pmap, ndef, pdefs)
	ColormapPtr pmap;
	int     ndef;
	xColorItem *pdefs;
{
#ifdef DEBUG
	ErrorF("nullStoreColormap called\n");
#endif /* DEBUG */
}

Bool
mac68kCloseScreen(index, pScreen)
	int     index;
	ScreenPtr pScreen;
{
	register int i, j, k, l;
	register char pat, *p;
	VDPgInfo vdp;
	VDEntry vde;
	ColorSpec cmap[2];

	if (mac_fbs[index].flags & MAC68K_FBFLAG_ROMDRIVER) {
		/* reinitialize the device back to 1-bit mode */
		bzero((void *) &vdp, sizeof vdp);
		vdp.csMode = 0x80;
		ioctl(mac_fbs[index].fd, GRFIOCSETMODE1, &vdp);

		/* reinitialize the CLUT */
		cmap[0].value = 0;
		cmap[0].color.red =
			cmap[0].color.green = cmap[0].color.blue = 0xffff;
		cmap[1].value = 1;
		cmap[1].color.red =
			cmap[1].color.green = cmap[1].color.blue = 0;
		vde.csTable = cmap;
		vde.csStart = 0;
		vde.csCount = 1;
		vde.length = sizeof cmap;
		ioctl(mac_fbs[index].fd, GRFIOCSETENTRIES, &vde);
	} /* else do nothing */
	return cfbCloseScreen(index, pScreen);
}

Bool
mac68kColorInit(index, screen, argc, argv)
	int     index;
	ScreenPtr screen;
	int     argc;
	char  **argv;
{
    struct grfmode *id;
    char    scrstr[150];
    VDPgInfo vdp;
    int fbwidth, flags = mac_fbs[index].flags;
    int nvisuals, i;
    VisualPtr visual;
    Bool (*screenInitFunc)(ScreenPtr, pointer, int, int, int, int, int);
    extern Bool
	cfb4ScreenInit(ScreenPtr, pointer, int, int, int, int, int),
	cfbScreenInit(ScreenPtr, pointer, int, int, int, int, int),
	cfb16ScreenInit(ScreenPtr, pointer, int, int, int, int, int),
	cfb32ScreenInit(ScreenPtr, pointer, int, int, int, int, int);

    id = &mac_fbs[index].idata;
    if (flags & (MAC68K_FBFLAG_ROMDRIVER | MAC68K_FBFLAG_PREFGRAY)) {
	bzero((void *) &vdp, sizeof vdp);
	ioctl(mac_fbs[index].fd, GRFIOCSETGRAY, &vdp);
    }
#ifdef DEBUG_CLUT
    if (!logfp)
	logfp = fopen("cmapacc.log", "w");
#endif
    if (mac_fbs[index].added)
	return (TRUE);

    screen->SaveScreen = mac68kColorSave;
    screen->InstallColormap = mac68kInstallColormap;
    screen->UninstallColormap = mac68kUninstallColormap;
    screen->ListInstalledColormaps =
	mac68kListInstalledColormaps;

    fbwidth = ((int)id->rowbytes << 3) / (int)id->psize;
    switch (id->psize) {
    case 4:
    case 8:
	screen->whitePixel = 0;
	screen->blackPixel = (1 << id->psize) - 1;
	screenInitFunc =
	    (id->psize == 4) ? cfb4ScreenInit : cfbScreenInit;
	if (flags & MAC68K_FBFLAG_ROMDRIVER) {
	    /* this screen supports dynamic class */
	    screen->StoreColors = mac68kStoreColors;
	    cfbSetVisualTypes(id->psize,
			      (flags & MAC68K_FBFLAG_PREFGRAY) ?
			      GRAY_VISUALS : SMALL_VISUALS,
			      8);
	} else {
	    screen->StoreColors = nullStoreColors;
	    cfbSetVisualTypes(id->psize, StaticGrayMask, 8);
	}
	screen->whitePixel = 0;
	screen->blackPixel = (1 << id->psize) - 1;
	break;
    case 16:
	screen->whitePixel = (1 << id->psize) - 1;
	screen->blackPixel = 0;
	screenInitFunc = cfb16ScreenInit;
	screen->StoreColors = nullStoreColors;
	cfbSetVisualTypes(id->psize, TrueColorMask, 8);
	break;
    case 32:
	screen->whitePixel = 0xffffff;
	screen->blackPixel = 0;
	screenInitFunc = cfb32ScreenInit;
	screen->StoreColors = nullStoreColors;
	cfbSetVisualTypes(id->psize, TrueColorMask, 8);
	/* XXX why? it doesn't make any bloody sense! */
	/* fbwidth = id->rowbytes; */
	break;
    default:
	ErrorF("mac68kColorInit: Unsupported pixel depth %d\n",
	       id->psize);
	return (FALSE);
	break;
    }
    if (!(*screenInitFunc)(screen,
			   (pointer)mac_fbs[index].vaddr,	/* BARF */
			   id->width,
			   id->height,
			   /* id->vRes >> 16 */ 75,	/* BARF */
			   /* id->vRes >> 16 */ 75,	/* BARF */
			   fbwidth)) {
	ErrorF("mac68kColorInit: cfb%dScreenInit failed\n", id->psize);
	return (FALSE);
    }

    /* correct pixel format data in struct screen */
    for (visual = screen->visuals, nvisuals = screen->numVisuals, i = 0;
	 i < nvisuals;
	 i++, visual++) {
	switch (visual->class) {
	case DirectColor:
	case TrueColor:
	case StaticColor:
	    visual->redMask =  MAC_RM(id->psize);
	    visual->greenMask =  MAC_GM(id->psize);
	    visual->blueMask =  MAC_BM(id->psize);
	    visual->offsetRed  =  MAC_RS(id->psize);
	    visual->offsetGreen = MAC_GS(id->psize);
	    visual->offsetBlue =  MAC_BS(id->psize);
	    break;
	default:
	    break;
	}
    }

    /* overwrite cfbCloseScreen with our own */
    screen->CloseScreen = mac68kCloseScreen;
#if 1
    screen->CreateColormap = mac68kCreateColormap;
#endif
#ifdef MORE_DEBUG
    printf("Calling ScreenInit to add screen %d...\n", index);
#endif
    ErrorF("Screen %d at %#08x, %d by %d, %d bpp, rowB %d, fbbase %#x.\n",
	   index, mac_fbs[index].vaddr, id->width,
	   id->height, id->psize, id->rowbytes, id->fbbase);
    mac_fbs[index].added = 1;

    if (!mac68k_screeninit(screen)) {
	ErrorF("mac68kColorInit: mac68k_screeninit failed\n");
	return FALSE;
    }
#ifdef DEBUG
    {
	char *visualStr[] = {
	    "StaticGray", "GrayScale", "StaticColor",
	    "PseudoColor", "TrueColor", "DirectColor",
	};
	VisualID vid = screen->rootVisual;
	VisualPtr visual = screen->visuals;

	ErrorF("rootVisual = %d (%s)\n", vid, visualStr[vid]);
	ErrorF("rootDepth = %d\n", screen->rootDepth);
	ErrorF("visuals->class = %d (%s)\n", visual->class,
	       visualStr[visual->class]);
	ErrorF("visuals->bitsPerRGBValue = %d\n",
	       visual->bitsPerRGBValue);
	ErrorF("visuals->ColormapEntries = %d\n",
	       visual->ColormapEntries);
	ErrorF("visuals->nplanes = %d\n", visual->nplanes);
	ErrorF("visuals->redMask = 0x%x\n", visual->redMask);
	ErrorF("visuals->greenMask = 0x%x\n", visual->greenMask);
	ErrorF("visuals->blueMask = 0x%x\n", visual->blueMask);
	ErrorF("visuals->offsetRed = 0x%x\n", visual->offsetRed);
	ErrorF("visuals->offsetGreen = 0x%x\n", visual->offsetGreen);
	ErrorF("visuals->offsetBlue = 0x%x\n", visual->offsetBlue);
    }
#endif
    if (!cfbCreateDefColormap(screen)) {
	ErrorF("mac68kColorInit: cfbCreateDefColormap failed\n");
	return FALSE;
    }
#ifdef DEBUG
    {
	PixmapPtr pixmap = screen->PixmapPerDepth[0];
	short i, depths = screen->numDepths;

	for (i = 0; i < depths; i++, pixmap++) {
	    DrawablePtr d = &pixmap->drawable;

	    if (d == NULL)
		break;

	    ErrorF("Screen->PixmapPerDepth[0][%d]->drawable = {\n",
		   i);
	    ErrorF("\ttype = %d\n", d->type);
	    ErrorF("\tclass = %d\n", d->class);
	    ErrorF("\tdepth = %d\n", d->depth);
	    ErrorF("\tbitsPerPixel = %d\n", d->bitsPerPixel);
	    ErrorF("\tid = %d\n", d->id);
	    ErrorF("\tx = %d\n", d->x);
	    ErrorF("\ty = %d\n", d->y);
	    ErrorF("\twidth = %d\n", d->width);
	    ErrorF("\theight = %d\n", d->height);
	    ErrorF("\tpScreen = 0x%x", d->pScreen);
	    if (d->pScreen == screen) {
		ErrorF(" (this screen)");
	    }
	    ErrorF("\n\tserialNumber = %d\n", d->serialNumber);
	    ErrorF("}\n");
	}
    }
#endif /* DEBUG */
    return TRUE;
}
