/*	$NetBSD: nouveau_nvkm_subdev_instmem_gk20a.c,v 1.4 2018/08/27 07:36:37 riastradh Exp $	*/

/*
 * Copyright (c) 2015, NVIDIA CORPORATION. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * GK20A does not have dedicated video memory, and to accurately represent this
 * fact Nouveau will not create a RAM device for it. Therefore its instmem
 * implementation must be done directly on top of system memory, while
 * preserving coherency for read and write operations.
 *
 * Instmem can be allocated through two means:
 * 1) If an IOMMU unit has been probed, the IOMMU API is used to make memory
 *    pages contiguous to the GPU. This is the preferred way.
 * 2) If no IOMMU unit is probed, the DMA API is used to allocate physically
 *    contiguous memory.
 *
 * In both cases CPU read and writes are performed by creating a write-combined
 * mapping. The GPU L2 cache must thus be flushed/invalidated when required. To
 * be conservative we do this every time we acquire or release an instobj, but
 * ideally L2 management should be handled at a higher level.
 *
 * To improve performance, CPU mappings are not removed upon instobj release.
 * Instead they are placed into a LRU list to be recycled when the mapped space
 * goes beyond a certain threshold. At the moment this limit is 1MB.
 */
#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: nouveau_nvkm_subdev_instmem_gk20a.c,v 1.4 2018/08/27 07:36:37 riastradh Exp $");

#include "priv.h"

#include <core/memory.h>
#include <core/mm.h>
#include <core/tegra.h>
#include <subdev/fb.h>
#include <subdev/ltc.h>

#ifdef __NetBSD__
#  define	__iomem	__nvkm_memory_iomem
#endif

struct gk20a_instobj {
	struct nvkm_memory memory;
	struct nvkm_mem mem;
	struct gk20a_instmem *imem;

	/* CPU mapping */
	u32 *vaddr;
	struct list_head vaddr_node;
};
#define gk20a_instobj(p) container_of((p), struct gk20a_instobj, memory)

#ifdef __NetBSD__

struct gk20a_instobj_netbsd {
	struct gk20a_instobj base;

	bus_dma_segment_t *segs;
	int nsegs;
	bus_dmamap_t map;
};
#define	gk20a_instobj_netbsd(p) \
	container_of(gk20a_instobj(p), struct gk20a_instobj_netbsd, base)

#else

/*
 * Used for objects allocated using the DMA API
 */
struct gk20a_instobj_dma {
	struct gk20a_instobj base;

	u32 *cpuaddr;
	dma_addr_t handle;
	struct nvkm_mm_node r;
};
#define gk20a_instobj_dma(p) \
	container_of(gk20a_instobj(p), struct gk20a_instobj_dma, base)

/*
 * Used for objects flattened using the IOMMU API
 */
struct gk20a_instobj_iommu {
	struct gk20a_instobj base;

	/* will point to the higher half of pages */
	dma_addr_t *dma_addrs;
	/* array of base.mem->size pages (+ dma_addr_ts) */
	struct page *pages[];
};
#define gk20a_instobj_iommu(p) \
	container_of(gk20a_instobj(p), struct gk20a_instobj_iommu, base)

#endif	/* __NetBSD__ */

struct gk20a_instmem {
	struct nvkm_instmem base;

	/* protects vaddr_* and gk20a_instobj::vaddr* */
	spinlock_t lock;

	/* CPU mappings LRU */
	unsigned int vaddr_use;
	unsigned int vaddr_max;
	struct list_head vaddr_lru;

#ifndef __NetBSD__
	/* Only used if IOMMU if present */
	struct mutex *mm_mutex;
	struct nvkm_mm *mm;
	struct iommu_domain *domain;
	unsigned long iommu_pgshift;
	u16 iommu_bit;

	/* Only used by DMA API */
	struct dma_attrs attrs;
#endif

	void __iomem * (*cpu_map)(struct nvkm_memory *);
};
#define gk20a_instmem(p) container_of((p), struct gk20a_instmem, base)

static enum nvkm_memory_target
gk20a_instobj_target(struct nvkm_memory *memory)
{
	return NVKM_MEM_TARGET_HOST;
}

static u64
gk20a_instobj_addr(struct nvkm_memory *memory)
{
	return gk20a_instobj(memory)->mem.offset;
}

static u64
gk20a_instobj_size(struct nvkm_memory *memory)
{
	return (u64)gk20a_instobj(memory)->mem.size << 12;
}

#ifdef __NetBSD__

static void __iomem *
gk20a_instobj_cpu_map_netbsd(struct nvkm_memory *memory)
{
	struct gk20a_instobj_netbsd *node = gk20a_instobj_netbsd(memory);
	struct nvkm_device *device = node->base.imem->base.subdev.device;
	bus_dma_tag_t dmat = device->func->dma_tag(device);
	uint64_t nbytes = nvkm_memory_size(memory);
	void *kva;
	int ret;

	/* XXX errno NetBSD->Linux */
	ret = -bus_dmamem_map(dmat, node->segs, node->nsegs, nbytes, &kva,
	    BUS_DMA_WAITOK);
	if (ret)
		return NULL;

	return kva;
}

static void
gk20a_instobj_cpu_unmap_netbsd(struct nvkm_memory *memory, void *kva)
{
	struct gk20a_instobj_netbsd *node = gk20a_instobj_netbsd(memory);
	struct nvkm_device *device = node->base.imem->base.subdev.device;
	bus_dma_tag_t dmat = device->func->dma_tag(device);

	bus_dmamem_unmap(dmat, kva, nvkm_memory_size(memory));
}

#else

static void __iomem *
gk20a_instobj_cpu_map_dma(struct nvkm_memory *memory)
{
#if defined(CONFIG_ARM) || defined(CONFIG_ARM64)
	struct gk20a_instobj_dma *node = gk20a_instobj_dma(memory);
	struct device *dev = node->base.imem->base.subdev.device->dev;
	int npages = nvkm_memory_size(memory) >> 12;
	struct page *pages[npages];
	int i;

	/* we shouldn't see a gk20a on anything but arm/arm64 anyways */
	/* phys_to_page does not exist on all platforms... */
	pages[0] = pfn_to_page(dma_to_phys(dev, node->handle) >> PAGE_SHIFT);
	for (i = 1; i < npages; i++)
		pages[i] = pages[0] + i;

	return vmap(pages, npages, VM_MAP, pgprot_writecombine(PAGE_KERNEL));
#else
	BUG();
	return NULL;
#endif
}

static void __iomem *
gk20a_instobj_cpu_map_iommu(struct nvkm_memory *memory)
{
	struct gk20a_instobj_iommu *node = gk20a_instobj_iommu(memory);
	int npages = nvkm_memory_size(memory) >> 12;

	return vmap(node->pages, npages, VM_MAP,
		    pgprot_writecombine(PAGE_KERNEL));
}

#endif	/* __NetBSD__ */

/*
 * Must be called while holding gk20a_instmem_lock
 */
static void
gk20a_instmem_vaddr_gc(struct gk20a_instmem *imem, const u64 size)
{
	while (imem->vaddr_use + size > imem->vaddr_max) {
		struct gk20a_instobj *obj;

		/* no candidate that can be unmapped, abort... */
		if (list_empty(&imem->vaddr_lru))
			break;

		obj = list_first_entry(&imem->vaddr_lru, struct gk20a_instobj,
				       vaddr_node);
		list_del(&obj->vaddr_node);
#ifdef __NetBSD__
		gk20a_instobj_cpu_unmap_netbsd(&obj->memory, obj->vaddr);
#else
		vunmap(obj->vaddr);
#endif
		obj->vaddr = NULL;
		imem->vaddr_use -= nvkm_memory_size(&obj->memory);
		nvkm_debug(&imem->base.subdev, "(GC) vaddr used: %x/%x\n",
			   imem->vaddr_use, imem->vaddr_max);

	}
}

static void __iomem *
gk20a_instobj_acquire(struct nvkm_memory *memory)
{
	struct gk20a_instobj *node = gk20a_instobj(memory);
	struct gk20a_instmem *imem = node->imem;
	struct nvkm_ltc *ltc = imem->base.subdev.device->ltc;
	const u64 size = nvkm_memory_size(memory);
	unsigned long flags;

	nvkm_ltc_flush(ltc);

	spin_lock_irqsave(&imem->lock, flags);

	if (node->vaddr) {
		/* remove us from the LRU list since we cannot be unmapped */
		list_del(&node->vaddr_node);

		goto out;
	}

	/* try to free some address space if we reached the limit */
	gk20a_instmem_vaddr_gc(imem, size);

	node->vaddr = imem->cpu_map(memory);

	if (!node->vaddr) {
		nvkm_error(&imem->base.subdev, "cannot map instobj - "
			   "this is not going to end well...\n");
		goto out;
	}

	imem->vaddr_use += size;
	nvkm_debug(&imem->base.subdev, "vaddr used: %x/%x\n",
		   imem->vaddr_use, imem->vaddr_max);

out:
	spin_unlock_irqrestore(&imem->lock, flags);

	return node->vaddr;
}

static void
gk20a_instobj_release(struct nvkm_memory *memory)
{
	struct gk20a_instobj *node = gk20a_instobj(memory);
	struct gk20a_instmem *imem = node->imem;
	struct nvkm_ltc *ltc = imem->base.subdev.device->ltc;
	unsigned long flags;

	spin_lock_irqsave(&imem->lock, flags);

	/* add ourselves to the LRU list so our CPU mapping can be freed */
	list_add_tail(&node->vaddr_node, &imem->vaddr_lru);

	spin_unlock_irqrestore(&imem->lock, flags);

	wmb();
	nvkm_ltc_invalidate(ltc);
}

static u32
gk20a_instobj_rd32(struct nvkm_memory *memory, u64 offset)
{
	struct gk20a_instobj *node = gk20a_instobj(memory);

	return node->vaddr[offset / 4];
}

static void
gk20a_instobj_wr32(struct nvkm_memory *memory, u64 offset, u32 data)
{
	struct gk20a_instobj *node = gk20a_instobj(memory);

	node->vaddr[offset / 4] = data;
}

static void
gk20a_instobj_map(struct nvkm_memory *memory, struct nvkm_vma *vma, u64 offset)
{
	struct gk20a_instobj *node = gk20a_instobj(memory);

	nvkm_vm_map_at(vma, offset, &node->mem);
}

/*
 * Clear the CPU mapping of an instobj if it exists
 */
static void
gk20a_instobj_dtor(struct gk20a_instobj *node)
{
	struct gk20a_instmem *imem = node->imem;
	struct gk20a_instobj *obj;
	unsigned long flags;

	spin_lock_irqsave(&imem->lock, flags);

	if (!node->vaddr)
		goto out;

	list_for_each_entry(obj, &imem->vaddr_lru, vaddr_node) {
		if (obj == node) {
			list_del(&obj->vaddr_node);
			break;
		}
	}
#ifdef __NetBSD__
	gk20a_instobj_cpu_unmap_netbsd(&node->memory, node->vaddr);
#else
	vunmap(node->vaddr);
#endif
	node->vaddr = NULL;
	imem->vaddr_use -= nvkm_memory_size(&node->memory);
	nvkm_debug(&imem->base.subdev, "vaddr used: %x/%x\n",
		   imem->vaddr_use, imem->vaddr_max);

out:
	spin_unlock_irqrestore(&imem->lock, flags);
}

#ifdef __NetBSD__

static void *
gk20a_instobj_dtor_netbsd(struct nvkm_memory *memory)
{
	struct gk20a_instobj_netbsd *node = gk20a_instobj_netbsd(memory);
	struct gk20a_instmem *imem = node->base.imem;
	struct nvkm_device *device = imem->base.subdev.device;
	bus_dma_tag_t dmat = device->func->dma_tag(device);

	gk20a_instobj_dtor(&node->base);

	if (unlikely(!node->map))
		goto out;

	bus_dmamap_unload(dmat, node->map);
	bus_dmamap_destroy(dmat, node->map);
	bus_dmamem_free(dmat, node->segs, nvkm_memory_size(memory));

out:
	return node;
}

#else

static void *
gk20a_instobj_dtor_dma(struct nvkm_memory *memory)
{
	struct gk20a_instobj_dma *node = gk20a_instobj_dma(memory);
	struct gk20a_instmem *imem = node->base.imem;
	struct device *dev = imem->base.subdev.device->dev;

	gk20a_instobj_dtor(&node->base);

	if (unlikely(!node->cpuaddr))
		goto out;

	dma_free_attrs(dev, node->base.mem.size << PAGE_SHIFT, node->cpuaddr,
		       node->handle, &imem->attrs);

out:
	return node;
}

static void *
gk20a_instobj_dtor_iommu(struct nvkm_memory *memory)
{
	struct gk20a_instobj_iommu *node = gk20a_instobj_iommu(memory);
	struct gk20a_instmem *imem = node->base.imem;
	struct device *dev = imem->base.subdev.device->dev;
	struct nvkm_mm_node *r;
	int i;

	gk20a_instobj_dtor(&node->base);

	if (unlikely(list_empty(&node->base.mem.regions)))
		goto out;

	r = list_first_entry(&node->base.mem.regions, struct nvkm_mm_node,
			     rl_entry);

	/* clear IOMMU bit to unmap pages */
	r->offset &= ~BIT(imem->iommu_bit - imem->iommu_pgshift);

	/* Unmap pages from GPU address space and free them */
	for (i = 0; i < node->base.mem.size; i++) {
		iommu_unmap(imem->domain,
			    (r->offset + i) << imem->iommu_pgshift, PAGE_SIZE);
		dma_unmap_page(dev, node->dma_addrs[i], PAGE_SIZE,
			       DMA_BIDIRECTIONAL);
		__free_page(node->pages[i]);
	}

	/* Release area from GPU address space */
	mutex_lock(imem->mm_mutex);
	nvkm_mm_free(imem->mm, &r);
	mutex_unlock(imem->mm_mutex);

out:
	return node;
}

#endif	/* __NetBSD__ */

#ifdef __NetBSD__

static const struct nvkm_memory_func
gk20a_instobj_func_netbsd = {
	.dtor = gk20a_instobj_dtor_netbsd,
	.target = gk20a_instobj_target,
	.addr = gk20a_instobj_addr,
	.size = gk20a_instobj_size,
	.acquire = gk20a_instobj_acquire,
	.release = gk20a_instobj_release,
	.rd32 = gk20a_instobj_rd32,
	.wr32 = gk20a_instobj_wr32,
	.map = gk20a_instobj_map,
};

#else

static const struct nvkm_memory_func
gk20a_instobj_func_dma = {
	.dtor = gk20a_instobj_dtor_dma,
	.target = gk20a_instobj_target,
	.addr = gk20a_instobj_addr,
	.size = gk20a_instobj_size,
	.acquire = gk20a_instobj_acquire,
	.release = gk20a_instobj_release,
	.rd32 = gk20a_instobj_rd32,
	.wr32 = gk20a_instobj_wr32,
	.map = gk20a_instobj_map,
};

static const struct nvkm_memory_func
gk20a_instobj_func_iommu = {
	.dtor = gk20a_instobj_dtor_iommu,
	.target = gk20a_instobj_target,
	.addr = gk20a_instobj_addr,
	.size = gk20a_instobj_size,
	.acquire = gk20a_instobj_acquire,
	.release = gk20a_instobj_release,
	.rd32 = gk20a_instobj_rd32,
	.wr32 = gk20a_instobj_wr32,
	.map = gk20a_instobj_map,
};

#endif

#ifdef __NetBSD__

static int
gk20a_instobj_ctor_netbsd(struct gk20a_instmem *imem, u32 pages, u32 align,
			  struct gk20a_instobj **_node)
{
	struct gk20a_instobj_netbsd *node;
	struct nvkm_subdev *subdev = &imem->base.subdev;
	struct nvkm_device *device = subdev->device;
	bus_dma_tag_t dmat = device->func->dma_tag(device);
	bus_size_t nbytes = (bus_size_t)pages << PAGE_SHIFT;
	int ret;

	if (!(node = kzalloc(sizeof(*node), GFP_KERNEL))) {
		ret = -ENOMEM;
fail0:		return ret;
	}
	*_node = &node->base;

	nvkm_memory_ctor(&gk20a_instobj_func_netbsd, &node->base.memory);

	node->segs = kcalloc(pages, sizeof(node->segs[0]), GFP_KERNEL);
	if (node->segs == NULL) {
		ret = -ENOMEM;
		goto fail0;
	}

	ret = -bus_dmamem_alloc(dmat, nbytes, align, 0, node->segs, pages,
	    &node->nsegs, BUS_DMA_WAITOK);
	if (ret) {
		kfree(node->segs);
		node->segs = NULL;
fail1:		goto fail0;
	}
	KASSERT(0 <= node->nsegs);
	KASSERT(node->nsegs <= pages);

	/*
	 * Must be a contiguous address range in bus space, so maxsegsz
	 * is the full mapping size.
	 */
	ret = -bus_dmamap_create(dmat, nbytes, node->nsegs, nbytes, 0,
	    BUS_DMA_WAITOK, &node->map);
	if (ret) {
fail2:		bus_dmamem_free(dmat, node->segs, node->nsegs);
		goto fail1;
	}

	ret = -bus_dmamap_load_raw(dmat, node->map, node->segs, node->nsegs,
	    nbytes, BUS_DMA_WAITOK);
	if (ret) {
fail3: __unused
		bus_dmamap_destroy(dmat, node->map);
		node->map = NULL;
		goto fail2;
	}

	/* Success!  */
	return 0;
}

#else

static int
gk20a_instobj_ctor_dma(struct gk20a_instmem *imem, u32 npages, u32 align,
		       struct gk20a_instobj **_node)
{
	struct gk20a_instobj_dma *node;
	struct nvkm_subdev *subdev = &imem->base.subdev;
	struct device *dev = subdev->device->dev;

	if (!(node = kzalloc(sizeof(*node), GFP_KERNEL)))
		return -ENOMEM;
	*_node = &node->base;

	nvkm_memory_ctor(&gk20a_instobj_func_dma, &node->base.memory);

	node->cpuaddr = dma_alloc_attrs(dev, npages << PAGE_SHIFT,
					&node->handle, GFP_KERNEL,
					&imem->attrs);
	if (!node->cpuaddr) {
		nvkm_error(subdev, "cannot allocate DMA memory\n");
		return -ENOMEM;
	}

	/* alignment check */
	if (unlikely(node->handle & (align - 1)))
		nvkm_warn(subdev,
			  "memory not aligned as requested: %pad (0x%x)\n",
			  &node->handle, align);

	/* present memory for being mapped using small pages */
	node->r.type = 12;
	node->r.offset = node->handle >> 12;
	node->r.length = (npages << PAGE_SHIFT) >> 12;

	node->base.mem.offset = node->handle;

	INIT_LIST_HEAD(&node->base.mem.regions);
	list_add_tail(&node->r.rl_entry, &node->base.mem.regions);

	return 0;
}

static int
gk20a_instobj_ctor_iommu(struct gk20a_instmem *imem, u32 npages, u32 align,
			 struct gk20a_instobj **_node)
{
	struct gk20a_instobj_iommu *node;
	struct nvkm_subdev *subdev = &imem->base.subdev;
	struct device *dev = subdev->device->dev;
	struct nvkm_mm_node *r;
	int ret;
	int i;

	/*
	 * despite their variable size, instmem allocations are small enough
	 * (< 1 page) to be handled by kzalloc
	 */
	if (!(node = kzalloc(sizeof(*node) + ((sizeof(node->pages[0]) +
			     sizeof(*node->dma_addrs)) * npages), GFP_KERNEL)))
		return -ENOMEM;
	*_node = &node->base;
	node->dma_addrs = (void *)(node->pages + npages);

	nvkm_memory_ctor(&gk20a_instobj_func_iommu, &node->base.memory);

	/* Allocate backing memory */
	for (i = 0; i < npages; i++) {
		struct page *p = alloc_page(GFP_KERNEL);
		dma_addr_t dma_adr;

		if (p == NULL) {
			ret = -ENOMEM;
			goto free_pages;
		}
		node->pages[i] = p;
		dma_adr = dma_map_page(dev, p, 0, PAGE_SIZE, DMA_BIDIRECTIONAL);
		if (dma_mapping_error(dev, dma_adr)) {
			nvkm_error(subdev, "DMA mapping error!\n");
			ret = -ENOMEM;
			goto free_pages;
		}
		node->dma_addrs[i] = dma_adr;
	}

	mutex_lock(imem->mm_mutex);
	/* Reserve area from GPU address space */
	ret = nvkm_mm_head(imem->mm, 0, 1, npages, npages,
			   align >> imem->iommu_pgshift, &r);
	mutex_unlock(imem->mm_mutex);
	if (ret) {
		nvkm_error(subdev, "IOMMU space is full!\n");
		goto free_pages;
	}

	/* Map into GPU address space */
	for (i = 0; i < npages; i++) {
		u32 offset = (r->offset + i) << imem->iommu_pgshift;

		ret = iommu_map(imem->domain, offset, node->dma_addrs[i],
				PAGE_SIZE, IOMMU_READ | IOMMU_WRITE);
		if (ret < 0) {
			nvkm_error(subdev, "IOMMU mapping failure: %d\n", ret);

			while (i-- > 0) {
				offset -= PAGE_SIZE;
				iommu_unmap(imem->domain, offset, PAGE_SIZE);
			}
			goto release_area;
		}
	}

	/* IOMMU bit tells that an address is to be resolved through the IOMMU */
	r->offset |= BIT(imem->iommu_bit - imem->iommu_pgshift);

	node->base.mem.offset = ((u64)r->offset) << imem->iommu_pgshift;

	INIT_LIST_HEAD(&node->base.mem.regions);
	list_add_tail(&r->rl_entry, &node->base.mem.regions);

	return 0;

release_area:
	mutex_lock(imem->mm_mutex);
	nvkm_mm_free(imem->mm, &r);
	mutex_unlock(imem->mm_mutex);

free_pages:
	for (i = 0; i < npages && node->pages[i] != NULL; i++) {
		dma_addr_t dma_addr = node->dma_addrs[i];
		if (dma_addr)
			dma_unmap_page(dev, dma_addr, PAGE_SIZE,
				       DMA_BIDIRECTIONAL);
		__free_page(node->pages[i]);
	}

	return ret;
}

#endif	/* __NetBSD__ */

static int
gk20a_instobj_new(struct nvkm_instmem *base, u32 size, u32 align, bool zero,
		  struct nvkm_memory **pmemory)
{
	struct gk20a_instmem *imem = gk20a_instmem(base);
	struct nvkm_subdev *subdev = &imem->base.subdev;
	struct gk20a_instobj *node = NULL;
	int ret = 0;

#ifdef __NetBSD__
	nvkm_debug(subdev, "%s (%s): size: %x align: %x\n", __func__,
		   "bus_dma", size, align);
#else
	nvkm_debug(subdev, "%s (%s): size: %x align: %x\n", __func__,
		   imem->domain ? "IOMMU" : "DMA", size, align);
#endif

	/* Round size and align to page bounds */
	size = max(roundup(size, PAGE_SIZE), PAGE_SIZE);
	align = max(roundup(align, PAGE_SIZE), PAGE_SIZE);

#ifdef __NetBSD__
	ret = gk20a_instobj_ctor_netbsd(imem, size >> PAGE_SHIFT, align,
	    &node);
#else
	if (imem->domain)
		ret = gk20a_instobj_ctor_iommu(imem, size >> PAGE_SHIFT,
					       align, &node);
	else
		ret = gk20a_instobj_ctor_dma(imem, size >> PAGE_SHIFT,
					     align, &node);
#endif
	*pmemory = node ? &node->memory : NULL;
	if (ret)
		return ret;

	node->imem = imem;

	/* present memory for being mapped using small pages */
	node->mem.size = size >> 12;
	node->mem.memtype = 0;
	node->mem.page_shift = 12;

	nvkm_debug(subdev, "alloc size: 0x%x, align: 0x%x, gaddr: 0x%"PRIx64"\n",
		   size, align, node->mem.offset);

	return 0;
}

static void *
gk20a_instmem_dtor(struct nvkm_instmem *base)
{
	struct gk20a_instmem *imem = gk20a_instmem(base);

	/* perform some sanity checks... */
	if (!list_empty(&imem->vaddr_lru))
		nvkm_warn(&base->subdev, "instobj LRU not empty!\n");

	if (imem->vaddr_use != 0)
		nvkm_warn(&base->subdev, "instobj vmap area not empty! "
			  "0x%x bytes still mapped\n", imem->vaddr_use);

	return imem;
}

static const struct nvkm_instmem_func
gk20a_instmem = {
	.dtor = gk20a_instmem_dtor,
	.memory_new = gk20a_instobj_new,
	.persistent = true,
	.zero = false,
};

int
gk20a_instmem_new(struct nvkm_device *device, int index,
		  struct nvkm_instmem **pimem)
{
#ifndef __NetBSD__
	struct nvkm_device_tegra *tdev = device->func->tegra(device);
#endif
	struct gk20a_instmem *imem;

	if (!(imem = kzalloc(sizeof(*imem), GFP_KERNEL)))
		return -ENOMEM;
	nvkm_instmem_ctor(&gk20a_instmem, device, index, &imem->base);
	spin_lock_init(&imem->lock);
	*pimem = &imem->base;

	/* do not allow more than 1MB of CPU-mapped instmem */
	imem->vaddr_use = 0;
	imem->vaddr_max = 0x100000;
	INIT_LIST_HEAD(&imem->vaddr_lru);

#ifdef __NetBSD__
	imem->cpu_map = gk20a_instobj_cpu_map_netbsd;
	nvkm_info(&imem->base.subdev, "using bus_dma\n");
#else
	if (tdev->iommu.domain) {
		imem->mm_mutex = &tdev->iommu.mutex;
		imem->mm = &tdev->iommu.mm;
		imem->domain = tdev->iommu.domain;
		imem->iommu_pgshift = tdev->iommu.pgshift;
		imem->cpu_map = gk20a_instobj_cpu_map_iommu;
		imem->iommu_bit = tdev->func->iommu_bit;

		nvkm_info(&imem->base.subdev, "using IOMMU\n");
	} else {
		init_dma_attrs(&imem->attrs);
		/* We will access the memory through our own mapping */
		dma_set_attr(DMA_ATTR_NON_CONSISTENT, &imem->attrs);
		dma_set_attr(DMA_ATTR_WEAK_ORDERING, &imem->attrs);
		dma_set_attr(DMA_ATTR_WRITE_COMBINE, &imem->attrs);
		dma_set_attr(DMA_ATTR_NO_KERNEL_MAPPING, &imem->attrs);
		imem->cpu_map = gk20a_instobj_cpu_map_dma;

		nvkm_info(&imem->base.subdev, "using DMA API\n");
	}
#endif

	return 0;
}
