/*
   This file is part of TALER
   Copyright (C) 2022 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_account_kyc_get_status.c
 * @brief Implementation of the account_kyc_get_status function for Postgres
 * @author Iván Ávalos
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_account_kyc_get_status.h"
#include "pg_helper.h"

/**
 * Closure for kyc_status_cb().
 */
struct KycStatusContext
{
  /**
   * Function to call with results.
   */
  TALER_MERCHANTDB_KycCallback kyc_cb;

  /**
   * Closure for @e kyc_cb.
   */
  void *kyc_cb_cls;

  /**
   * Filter, NULL to not filter.
   */
  const struct TALER_MerchantWireHashP *h_wire;

  /**
   * Filter, NULL to not filter.
   */
  const char *exchange_url;

  /**
   * Number of results found.
   */
  unsigned int count;

  /**
   * Set to true on failure(s).
   */
  bool failure;
};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about accounts.
 *
 * @param[in,out] cls of type `struct KycStatusContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
kyc_status_cb (void *cls,
               PGresult *result,
               unsigned int num_results)
{
  struct KycStatusContext *ksc = cls;

  for (unsigned int i = 0; i < num_results; i++)
  {
    struct TALER_MerchantWireHashP h_wire;
    uint64_t kyc_serial;
    char *exchange_url;
    char *payto_uri;
    struct GNUNET_TIME_Timestamp last_check;
    bool kyc_ok;
    uint32_t aml_decision;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_auto_from_type ("h_wire",
                                            &h_wire),
      GNUNET_PQ_result_spec_uint64 ("exchange_kyc_serial",
                                    &kyc_serial),
      GNUNET_PQ_result_spec_string ("payto_uri",
                                    &payto_uri),
      GNUNET_PQ_result_spec_string ("exchange_url",
                                    &exchange_url),
      GNUNET_PQ_result_spec_timestamp ("kyc_timestamp",
                                       &last_check),
      GNUNET_PQ_result_spec_bool ("kyc_ok",
                                  &kyc_ok),
      GNUNET_PQ_result_spec_uint32 ("aml_decision",
                                    &aml_decision),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      ksc->failure = true;
      return;
    }
    if ( (NULL != ksc->exchange_url) &&
         (0 != strcmp (ksc->exchange_url,
                       exchange_url)) )
    {
      GNUNET_PQ_cleanup_result (rs);
      continue;
    }
    if ( (NULL != ksc->h_wire) &&
         (0 != GNUNET_memcmp (ksc->h_wire,
                              &h_wire)) )
    {
      GNUNET_PQ_cleanup_result (rs);
      continue;
    }
    ksc->count++;
    ksc->kyc_cb (ksc->kyc_cb_cls,
                 &h_wire,
                 kyc_serial,
                 payto_uri,
                 exchange_url,
                 last_check,
                 kyc_ok,
                 (enum TALER_AmlDecisionState) aml_decision);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_account_kyc_get_status (void *cls,
                               const char *merchant_id,
                               const struct TALER_MerchantWireHashP *h_wire,
                               const char *exchange_url,
                               TALER_MERCHANTDB_KycCallback kyc_cb,
                               void *kyc_cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct KycStatusContext ksc = {
    .kyc_cb = kyc_cb,
    .kyc_cb_cls = kyc_cb_cls,
    .exchange_url = exchange_url,
    .h_wire = h_wire
  };
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (merchant_id),
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "lookup_kyc_status",
           "SELECT"
           " h_wire"
           ",exchange_kyc_serial"
           ",payto_uri"
           ",exchange_url"
           ",kyc_timestamp"
           ",kyc_ok"
           ",aml_decision"
           " FROM merchant_instances"
           " JOIN merchant_accounts"
           "   USING (merchant_serial)"
           " JOIN merchant_kyc"
           "   USING (account_serial)"
           " WHERE merchant_instances.merchant_id=$1");
  qs = GNUNET_PQ_eval_prepared_multi_select (pg->conn,
                                             "lookup_kyc_status",
                                             params,
                                             &kyc_status_cb,
                                             &ksc);
  if (ksc.failure)
  {
    GNUNET_break (0);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (0 > qs)
    return qs;
  return ksc.count;
}
