--
-- This file is part of TALER
-- Copyright (C) 2024 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_kycauths_in(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'kycauths_in';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(kycauth_in_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',account_pub BYTEA CHECK (LENGTH(account_pub)=32)'
      ',wire_reference INT8 NOT NULL'
      ',credit taler_amount NOT NULL'
      ',wire_source_h_payto BYTEA CHECK (LENGTH(wire_source_h_payto)=32)'
      ',exchange_account_section TEXT NOT NULL'
      ',execution_date INT8 NOT NULL'
      ',PRIMARY KEY(wire_source_h_payto, wire_reference)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wire_source_h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'list of transfers to register a key for KYC authentication, one per incoming wire transfer'
    ,table_name
    ,partition_suffix
  );
  -- FIXME: check that the *full* payto URI is indeed the best choice here,
  -- given that this is mostly used for KYC, we may prefer the normalized
  -- payto URI instead! Not sure, to be checked!
  PERFORM comment_partitioned_column(
     'Identifies the debited bank account and KYC status by the hash over the full payto URI'
    ,'wire_source_h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key to be associated with the account.'
    ,'account_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Amount that was transferred into the account'
    ,'credit'
    ,table_name
    ,partition_suffix
  );
END $$;


CREATE FUNCTION constrain_table_kycauths_in(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'kycauths_in';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_kycauth_in_serial_id_key'
    ' UNIQUE (kycauth_in_serial_id)'
  );
END
$$;

INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('kycauths_in'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('kycauths_in'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
