/* pkl-gen.c - Code generation phase for the poke compiler.  */

/* Copyright (C) 2019, 2020, 2021, 2022 Jose E. Marchesi */

/* This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>

#include "pkl.h"
#include "pkl-gen.h"
#include "pkl-ast.h"
#include "pkl-env.h"
#include "pkl-pass.h"
#include "pkl-asm.h"
#include "pvm.h"
#include "pk-utils.h"

/* The following macros are used in the rules below, to reduce
   verbosity.  */

#define PKL_GEN_PAYLOAD ((pkl_gen_payload) PKL_PASS_PAYLOAD)

#define PKL_GEN_AN_ASM(ASM)                             \
  (PKL_GEN_PAYLOAD->ASM[PKL_GEN_PAYLOAD->cur_##ASM])

#define PKL_GEN_ASM  PKL_GEN_AN_ASM(pasm)
#define PKL_GEN_ASM2 PKL_GEN_AN_ASM(pasm2)

#define PKL_GEN_PUSH_AN_ASM(ASM,new_pasm)                               \
  do                                                                    \
    {                                                                   \
      assert (PKL_GEN_PAYLOAD->cur_##ASM < PKL_GEN_MAX_PASM);           \
      PKL_GEN_PAYLOAD->ASM[++(PKL_GEN_PAYLOAD->cur_##ASM)] = (new_pasm); \
    }                                                                   \
  while (0)

#define PKL_GEN_PUSH_ASM(new_pasm)  PKL_GEN_PUSH_AN_ASM(pasm,new_pasm)
#define PKL_GEN_PUSH_ASM2(new_pasm) PKL_GEN_PUSH_AN_ASM(pasm2,new_pasm)

#define PKL_GEN_POP_AN_ASM(ASM)                  \
  do                                             \
    {                                            \
      assert (PKL_GEN_PAYLOAD->cur_##ASM > 0);   \
      PKL_GEN_PAYLOAD->cur_##ASM -= 1;           \
    }                                            \
  while (0)

#define PKL_GEN_POP_ASM  PKL_GEN_POP_AN_ASM(pasm)
#define PKL_GEN_POP_ASM2 PKL_GEN_POP_AN_ASM(pasm2)

#define PKL_GEN_IN_CTX_P(CTX)                                           \
  (PKL_GEN_PAYLOAD->context[PKL_GEN_PAYLOAD->cur_context] & (CTX))

#define PKL_GEN_PUSH_CONTEXT                                            \
  do                                                                    \
    {                                                                   \
      assert (PKL_GEN_PAYLOAD->cur_context < PKL_GEN_MAX_CTX);          \
      PKL_GEN_PAYLOAD->context[PKL_GEN_PAYLOAD->cur_context + 1]        \
        = 0;                                                            \
      PKL_GEN_PAYLOAD->cur_context++;                                   \
    }                                                                   \
  while (0)

#define PKL_GEN_POP_CONTEXT                             \
  do                                                    \
    {                                                   \
      assert (PKL_GEN_PAYLOAD->cur_context > 0);        \
      PKL_GEN_PAYLOAD->cur_context--;                   \
    }                                                   \
  while (0)

#define PKL_GEN_SET_CONTEXT(CTX)                                        \
  do                                                                    \
    {                                                                   \
      PKL_GEN_PAYLOAD->context[PKL_GEN_PAYLOAD->cur_context] |= (CTX);  \
    }                                                                   \
  while (0)

#define PKL_GEN_CLEAR_CONTEXT(CTX)                                      \
  do                                                                    \
    {                                                                   \
      PKL_GEN_PAYLOAD->context[PKL_GEN_PAYLOAD->cur_context] &= ~(CTX); \
    }                                                                   \
  while (0)

#define PKL_GEN_PUSH_SET_CONTEXT(CTX)           \
  do                                            \
    {                                           \
      PKL_GEN_PUSH_CONTEXT;                     \
      PKL_GEN_SET_CONTEXT (CTX);                \
    }                                           \
  while (0)

/* Code generated by RAS is used in the handlers below.  Configure it
   to use the main assembler in the GEN payload.  Then just include
   the assembled macros in this file.  */
#define RAS_ASM PKL_GEN_ASM
#define RAS_PUSH_ASM PKL_GEN_PUSH_ASM
#define RAS_POP_ASM PKL_GEN_POP_ASM
#define RAS_COMPILER (PKL_GEN_PAYLOAD->compiler)
#define RAS_COMP_ENV (PKL_GEN_PAYLOAD->env)
#include "pkl-gen.pkc"
#include "pkl-gen-builtins.pkc"

/*
 * SRC
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_src)
{
  PKL_GEN_PAYLOAD->in_file_p
    = (PKL_AST_SRC_FILENAME (PKL_PASS_NODE) != NULL);
}
PKL_PHASE_END_HANDLER

/*
 * PROGRAM
 * | PROGRAM_ELEM
 * | ...
 *
 * This function initializes the payload and also generates the
 * standard prologue.
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_program)
{
  PKL_GEN_ASM = pkl_asm_new (PKL_PASS_AST,
                             PKL_GEN_PAYLOAD->compiler,
                             1 /* prologue */);

  PKL_GEN_PAYLOAD->in_file_p
    = (!pkl_compiling_statement_p (PKL_GEN_PAYLOAD->compiler)
       && !pkl_compiling_expression_p (PKL_GEN_PAYLOAD->compiler));
}
PKL_PHASE_END_HANDLER

/*
 * | PROGRAM_ELEM
 * | ...
 * PROGRAM
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_program)
{
  /* Make sure there is always some value returned in the stack, since
     that is expected in the PVM.  */
  if (!pkl_compiling_expression_p (PKL_GEN_PAYLOAD->compiler)
      && !(pkl_compiling_statement_p (PKL_GEN_PAYLOAD->compiler)
           && PKL_AST_PROGRAM_ELEMS (PKL_PASS_NODE)
           && PKL_AST_CODE (PKL_AST_PROGRAM_ELEMS (PKL_PASS_NODE)) == PKL_AST_EXP_STMT))
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);

  PKL_GEN_PAYLOAD->program = pkl_asm_finish (PKL_GEN_ASM,
                                             1 /* prologue */);
}
PKL_PHASE_END_HANDLER

/*
 * DECL
 * | INITIAL
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_decl)
{
  pkl_ast_node decl = PKL_PASS_NODE;
  pkl_ast_node initial = PKL_AST_DECL_INITIAL (decl);

  /* mktysct only gets information from regular struct fields.
     Therefore, we do not need to process declarations of variables,
     types and methods inside struct types.  */
  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    PKL_PASS_BREAK;

  switch (PKL_AST_DECL_KIND (decl))
    {
    case PKL_AST_DECL_KIND_UNIT:
      /* Nothing to do with units at run-time, for now.  */
      PKL_PASS_BREAK;
      break;
    case PKL_AST_DECL_KIND_TYPE:
      switch (PKL_AST_TYPE_CODE (initial))
        {
        case PKL_TYPE_STRUCT:
          {
            pkl_ast_node type_struct = initial;

            /* Compile the struct closures and complete them using the
               current environment.  */

            if (PKL_AST_TYPE_S_WRITER (type_struct) == PVM_NULL)
              {
                pvm_val writer_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_WRITER);
                {
                  if (PKL_AST_TYPE_S_UNION_P (type_struct))
                    RAS_FUNCTION_UNION_WRITER (writer_closure, type_struct);
                  else
                    RAS_FUNCTION_STRUCT_WRITER (writer_closure, type_struct);
                }
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_S_WRITER (type_struct) = writer_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_S_WRITER (type_struct)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);           /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);          /* _ */

            if (PKL_AST_TYPE_S_MAPPER (type_struct) == PVM_NULL)
              {
                pvm_val mapper_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_MAPPER);
                RAS_FUNCTION_STRUCT_MAPPER (mapper_closure, type_struct);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_S_MAPPER (type_struct) = mapper_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_S_MAPPER (type_struct)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);           /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);          /* _ */

            if (PKL_AST_TYPE_S_CONSTRUCTOR (type_struct) == PVM_NULL)
              {
                pvm_val constructor_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_CONSTRUCTOR);
                RAS_FUNCTION_STRUCT_CONSTRUCTOR (constructor_closure,
                                                 type_struct);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_S_CONSTRUCTOR (type_struct) = constructor_closure;
              }
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_S_CONSTRUCTOR (type_struct)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);                /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);               /* _ */

            if (PKL_AST_TYPE_S_COMPARATOR (type_struct) == PVM_NULL)
              {
                pvm_val comparator_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_COMPARATOR);
                if (PKL_AST_TYPE_S_UNION_P (type_struct))
                  RAS_FUNCTION_UNION_COMPARATOR (comparator_closure,
                                                 type_struct);
                else
                  RAS_FUNCTION_STRUCT_COMPARATOR (comparator_closure,
                                                  type_struct);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_S_COMPARATOR (type_struct) = comparator_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_S_COMPARATOR (type_struct)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);               /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);              /* _ */

            if (PKL_AST_TYPE_S_FORMATER (type_struct) == PVM_NULL)
              {
                pvm_val formater_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_FORMATER);
                RAS_FUNCTION_STRUCT_FORMATER (formater_closure,
                                              type_struct);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_S_FORMATER (type_struct) = formater_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_S_FORMATER (type_struct)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);             /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);            /* _ */

            if (PKL_AST_TYPE_S_PRINTER (type_struct) == PVM_NULL)
              {
                pvm_val printer_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_PRINTER);
                RAS_FUNCTION_STRUCT_PRINTER (printer_closure,
                                             type_struct);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_S_PRINTER (type_struct) = printer_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_S_PRINTER (type_struct)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);            /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);           /* _ */

            if (PKL_AST_TYPE_S_ITYPE (type_struct))
              {
                if (PKL_AST_TYPE_S_INTEGRATOR (type_struct) == PVM_NULL)
                  {
                    pvm_val integrator_closure;

                    PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_INTEGRATOR);
                    RAS_FUNCTION_STRUCT_INTEGRATOR (integrator_closure,
                                                    type_struct);
                    PKL_GEN_POP_CONTEXT;
                    PKL_AST_TYPE_S_INTEGRATOR (type_struct) = integrator_closure;
                  }

                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                              PKL_AST_TYPE_S_INTEGRATOR (type_struct)); /* CLS */
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);               /* CLS */
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);              /* _ */

                if (PKL_AST_TYPE_S_DEINTEGRATOR (type_struct) == PVM_NULL)
                  {
                    pvm_val deintegrator_closure;

                    PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_DEINTEGRATOR);
                    RAS_FUNCTION_STRUCT_DEINTEGRATOR (deintegrator_closure,
                                                      type_struct);
                    PKL_GEN_POP_CONTEXT;
                    PKL_AST_TYPE_S_DEINTEGRATOR (type_struct) = deintegrator_closure;
                  }

                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                              PKL_AST_TYPE_S_DEINTEGRATOR (type_struct)); /* CLS */
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);                 /* CLS */
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);                /* _ */
              }

            /* Note that the typifier must be processed last, as it
               uses the closures installed above.  */
            if (PKL_AST_TYPE_S_TYPIFIER (type_struct) == PVM_NULL)
              {
                pvm_val typifier_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_TYPIFIER);
                RAS_FUNCTION_STRUCT_TYPIFIER (typifier_closure,
                                              type_struct);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_S_TYPIFIER (type_struct) = typifier_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_S_TYPIFIER (type_struct)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);             /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);            /* _ */

            PKL_PASS_BREAK;
            break;
          }
        case PKL_TYPE_ARRAY:
          {
            pkl_ast_node array_type = initial;

            /* Compile the arrays closures and complete them using the
               current environment.  */

            if (PKL_AST_TYPE_A_BOUNDER (array_type) == PVM_NULL)
              {
                /* The bounder closures for this array and possibly
                   contained sub-arrays are installed in the
                   pkl_gen_pr_type_array handler.  This should be done
                   before compiling the rest of the closures below, to
                   assure the bounder closures capture the right
                   lexical context!  This makes the calls to
                   in_array_bounder in array mappers/constructors to
                   only happen for anonymous array types.  */
                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_ARRAY_BOUNDER);
                PKL_PASS_SUBPASS (array_type);
                PKL_GEN_POP_CONTEXT;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_A_BOUNDER (array_type)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);           /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);          /* _ */

            if (PKL_AST_TYPE_A_WRITER (array_type) == PVM_NULL)
              {
                pvm_val writer_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_WRITER);
                RAS_FUNCTION_ARRAY_WRITER (writer_closure, array_type);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_A_WRITER (array_type) = writer_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_A_WRITER (array_type)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);          /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);         /* _ */

            if (PKL_AST_TYPE_A_MAPPER (array_type) == PVM_NULL)
              {
                pvm_val mapper_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_MAPPER);
                RAS_FUNCTION_ARRAY_MAPPER (mapper_closure, array_type);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_A_MAPPER (array_type) = mapper_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_A_MAPPER (array_type)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);          /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);         /* _ */

            if (PKL_AST_TYPE_A_CONSTRUCTOR (array_type) == PVM_NULL)
              {
                pvm_val constructor_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_CONSTRUCTOR);
                RAS_FUNCTION_ARRAY_CONSTRUCTOR (constructor_closure,
                                                array_type);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_A_CONSTRUCTOR (array_type) = constructor_closure;
              }
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_A_CONSTRUCTOR (array_type)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);               /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);              /* _ */

            if (pkl_ast_type_integrable_p (array_type))
              {
                if (PKL_AST_TYPE_A_INTEGRATOR (array_type) == PVM_NULL)
                  {
                    pvm_val integrator_closure;

                    PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_INTEGRATOR);
                    RAS_FUNCTION_ARRAY_INTEGRATOR (integrator_closure,
                                                   array_type);
                    PKL_GEN_POP_CONTEXT;
                    PKL_AST_TYPE_A_INTEGRATOR (array_type) = integrator_closure;
                  }
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                              PKL_AST_TYPE_A_INTEGRATOR (array_type)); /* CLS */
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);              /* CLS */
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);             /* _ */
              }

            if (PKL_AST_TYPE_A_FORMATER (array_type) == PVM_NULL)
              {
                pvm_val formater_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_FORMATER);
                RAS_FUNCTION_ARRAY_FORMATER (formater_closure,
                                             array_type);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_A_FORMATER (array_type) = formater_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_A_FORMATER (array_type)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);            /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);           /* _ */

            if (PKL_AST_TYPE_A_PRINTER (array_type) == PVM_NULL)
              {
                pvm_val printer_closure;

                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_PRINTER);
                RAS_FUNCTION_ARRAY_PRINTER (printer_closure,
                                            array_type);
                PKL_GEN_POP_CONTEXT;
                PKL_AST_TYPE_A_PRINTER (array_type) = printer_closure;
              }

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          PKL_AST_TYPE_A_PRINTER (array_type)); /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);           /* CLS */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);          /* _ */
            PKL_PASS_BREAK;
            break;
          }
        default:
          break;
        }
      break;
    case PKL_AST_DECL_KIND_FUNC:
      {
        pvm_program program;
        pvm_val closure;

        if (PKL_AST_FUNC_PROGRAM (initial))
          program = PKL_AST_FUNC_PROGRAM (initial);
        else
          {
            /* INITIAL is a PKL_AST_FUNC, that will compile into a
               program containing the function code.  Push a new
               assembler to the stack of assemblers in the payload and
               use it to process INITIAL.  */
            PKL_GEN_PUSH_ASM (pkl_asm_new (PKL_PASS_AST,
                                           PKL_GEN_PAYLOAD->compiler,
                                           0 /* prologue */));

            PKL_PASS_SUBPASS (initial);

            /* At this point the code for the function specification
               INITIAL has been assembled in the current
               macroassembler.  Finalize the program and put it in a
               PVM closure, along with the current environment.  */

            program = pkl_asm_finish (PKL_GEN_ASM,
                                      0 /* epilogue */);
            PKL_GEN_POP_ASM;
            pvm_program_make_executable (program);

            /* XXX */
            //            pvm_disassemble_program (program);
            PKL_AST_FUNC_PROGRAM (initial) = program;
          }

        closure = pvm_make_cls (program);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, closure);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DUC);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REGVAR);

        PKL_PASS_BREAK;
        break;
      }
    default:
      break;
    }
}
PKL_PHASE_END_HANDLER

/*
 * | INITIAL
 * DECL
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_decl)
{
  pkl_ast_node decl = PKL_PASS_NODE;
  pkl_ast_node initial = PKL_AST_DECL_INITIAL (decl);

  switch (PKL_AST_DECL_KIND (decl))
    {
    case PKL_AST_DECL_KIND_VAR:
      /* The value is in the stack.  Just register the variable.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REGVAR);
      break;
    case PKL_AST_DECL_KIND_TYPE:
      if (PKL_AST_TYPE_CODE (initial) == PKL_TYPE_STRUCT
          || PKL_AST_TYPE_CODE (initial) == PKL_TYPE_ARRAY)
        assert (0);
      break;
    default:
      assert (0);
      break;
    }
}
PKL_PHASE_END_HANDLER

/*
 * VAR
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_var)
{
  pkl_ast_node var = PKL_PASS_NODE;

  if (PKL_PASS_PARENT == NULL
      && PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_LVALUE))
    {
      /* This is a l-value in an assignment.  Generate nothing, as
         this node is only used as a recipient for the lexical address
         of the variable.  */
    }
  else
    {
      pkl_ast_node var_decl = PKL_AST_VAR_DECL (var);
      pkl_ast_node var_type = PKL_AST_TYPE (var);
      pkl_ast_node var_function = PKL_AST_VAR_FUNCTION (var);

      /* If the declaration associated with the variable is in a
         struct _and_ we are in a method body.

         Instead of accessing the variable in the lexical environment,
         we push the implicit struct and sref it with the name of the
         variable.  The implicit struct is the first argument passed
         to the current function.  */
      if (var_function
          && PKL_AST_FUNC_METHOD_P (var_function)
          && (PKL_AST_DECL_STRUCT_FIELD_P (var_decl)
              || (PKL_AST_DECL_KIND (var_decl) == PKL_AST_DECL_KIND_FUNC
                  && PKL_AST_FUNC_METHOD_P (PKL_AST_DECL_INITIAL (var_decl)))))
        {
          pkl_ast_node var_name = PKL_AST_VAR_NAME (var);
          int var_function_back = PKL_AST_VAR_FUNCTION_BACK (var);

          assert (var_name != NULL);

          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHVAR,
                        var_function_back, 0);              /* SCT */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        pvm_make_string (PKL_AST_IDENTIFIER_POINTER (var_name)));
                                                            /* SCT STR */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SREF);        /* SCT STR VAL */

          if (PKL_AST_DECL_KIND (var_decl) == PKL_AST_DECL_KIND_FUNC)
            /* Method call: leave the implicit struct so it is passed
               as the last argument to the method.  */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
          else
            /* Normal field.  */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);
        }
      else
        /* Normal variable.  */
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHVAR,
                      PKL_AST_VAR_BACK (var), PKL_AST_VAR_OVER (var));

      /* If the declaration associated with the variable is in a
         struct and we are not in a method, i.e. we are in a context
         like a constraint expression or an optional field condition,
         then raise E_elem if the value is null.  */
      if (PKL_AST_DECL_STRUCT_FIELD_P (var_decl)
          && !var_function)
        {
          pvm_program_label label
            = pkl_asm_fresh_label (PKL_GEN_ASM);

          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BNN, label);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        pvm_make_exception (PVM_E_ELEM, PVM_E_ELEM_NAME,
                                            PVM_E_ELEM_ESTATUS, NULL, NULL));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RAISE);
          pkl_asm_label (PKL_GEN_ASM, label);
        }

      /* If the value holds a value that could be mapped, then use the
         REMAP instruction.  */
      if (PKL_AST_TYPE_CODE (var_type) == PKL_TYPE_ARRAY
          || PKL_AST_TYPE_CODE (var_type) == PKL_TYPE_STRUCT)
        {
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REMAP);
        }
    }
}
PKL_PHASE_END_HANDLER

/*
 * LAMBDA
 * | FUNCTION
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_lambda)
{
  /* FUNCTION is a PKL_AST_FUNC, that will compile into a program
     containing the function code.  Push a new assembler.  */
  PKL_GEN_PUSH_ASM (pkl_asm_new (PKL_PASS_AST,
                                 PKL_GEN_PAYLOAD->compiler,
                                 0 /* prologue */));
}
PKL_PHASE_END_HANDLER

/*
 * | FUNCTION
 * LAMBDA
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_lambda)
{
  /* At this point the code for FUNCTION has been assembled in the
     current macroassembler.  Finalize the program and put it in a PVM
     closure, along with the current environment.  */

  pvm_program program = pkl_asm_finish (PKL_GEN_ASM,
                                        0 /* epilogue */);
  pvm_val closure;

  PKL_GEN_POP_ASM;
  pvm_program_make_executable (program);
  closure = pvm_make_cls (program);

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, closure);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DUC);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);
}
PKL_PHASE_END_HANDLER

/*
 * NULL_STMT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_null_stmt)
{
  /* Null is nothing, nada.  */
}
PKL_PHASE_END_HANDLER

/*
 * COMP_STMT
 * | (STMT | DECL)
 * | ...
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_comp_stmt)
{
  pkl_ast_node comp_stmt = PKL_PASS_NODE;

  if (PKL_AST_COMP_STMT_BUILTIN (comp_stmt) == PKL_AST_BUILTIN_NONE)
    {
      /* If the compound statement is empty, do not generate
         anything.  */
      if (PKL_AST_COMP_STMT_STMTS (comp_stmt) == NULL)
        PKL_PASS_BREAK;

      /* Push a frame into the environment.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHF,
                    PKL_AST_COMP_STMT_NUMVARS (comp_stmt));
    }
}
PKL_PHASE_END_HANDLER

/*
 * | (STMT | DECL)
 * | ...
 * COMP_STMT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_comp_stmt)
{
  pkl_ast_node comp_stmt = PKL_PASS_NODE;
  int comp_stmt_builtin
    = PKL_AST_COMP_STMT_BUILTIN (comp_stmt);

  if (comp_stmt_builtin != PKL_AST_BUILTIN_NONE)
    {
      switch (comp_stmt_builtin)
        {
        case PKL_AST_BUILTIN_RAND:
          RAS_MACRO_BUILTIN_RAND;
          break;
        case PKL_AST_BUILTIN_GET_ENDIAN:
          RAS_MACRO_BUILTIN_GET_ENDIAN;
          break;
        case PKL_AST_BUILTIN_GET_IOS:
          RAS_MACRO_BUILTIN_GET_IOS;
          break;
        case PKL_AST_BUILTIN_SET_ENDIAN:
          RAS_MACRO_BUILTIN_SET_ENDIAN;
          break;
        case PKL_AST_BUILTIN_SET_IOS:
          RAS_MACRO_BUILTIN_SET_IOS;
          break;
        case PKL_AST_BUILTIN_OPEN:
          RAS_MACRO_BUILTIN_OPEN;
          break;
        case PKL_AST_BUILTIN_CLOSE:
          RAS_MACRO_BUILTIN_CLOSE;
          break;
        case PKL_AST_BUILTIN_IOSIZE:
          RAS_MACRO_BUILTIN_IOSIZE;
          break;
        case PKL_AST_BUILTIN_IOFLAGS:
          RAS_MACRO_BUILTIN_IOFLAGS;
          break;
        case PKL_AST_BUILTIN_IOGETB:
          RAS_MACRO_BUILTIN_IOBIAS;
          break;
        case PKL_AST_BUILTIN_IOSETB:
          RAS_MACRO_BUILTIN_IOSETBIAS;
          break;
        case PKL_AST_BUILTIN_FORGET:
          RAS_MACRO_BUILTIN_FLUSH;
          break;
        case PKL_AST_BUILTIN_GET_TIME:
          RAS_MACRO_BUILTIN_GET_TIME;
          break;
        case PKL_AST_BUILTIN_SLEEP:
          RAS_MACRO_BUILTIN_SLEEP;
          break;
        case PKL_AST_BUILTIN_STRACE:
          RAS_MACRO_BUILTIN_STRACE;
          break;
        case PKL_AST_BUILTIN_GETENV:
          RAS_MACRO_BUILTIN_GETENV;
          break;
        case PKL_AST_BUILTIN_TERM_GET_COLOR:
        case PKL_AST_BUILTIN_TERM_GET_BGCOLOR:
          RAS_MACRO_BUILTIN_GET_COLOR_BGCOLOR;
          break;
        case PKL_AST_BUILTIN_TERM_SET_COLOR:
        case PKL_AST_BUILTIN_TERM_SET_BGCOLOR:
          RAS_MACRO_BUILTIN_SET_COLOR_BGCOLOR;
          break;
        case PKL_AST_BUILTIN_TERM_BEGIN_CLASS:
          RAS_MACRO_BUILTIN_TERM_BEGIN_CLASS;
          break;
        case PKL_AST_BUILTIN_TERM_END_CLASS:
          RAS_MACRO_BUILTIN_TERM_END_CLASS;
          break;
        case PKL_AST_BUILTIN_TERM_BEGIN_HYPERLINK:
          RAS_MACRO_BUILTIN_TERM_BEGIN_HYPERLINK;
          break;
        case PKL_AST_BUILTIN_TERM_END_HYPERLINK:
          RAS_MACRO_BUILTIN_TERM_END_HYPERLINK;
          break;
        case PKL_AST_BUILTIN_VM_OBASE:
          RAS_MACRO_BUILTIN_VM_OBASE;
          break;
        case PKL_AST_BUILTIN_VM_SET_OBASE:
          RAS_MACRO_BUILTIN_VM_SET_OBASE;
          break;
        case PKL_AST_BUILTIN_VM_OPPRINT:
          RAS_MACRO_BUILTIN_VM_OPPRINT;
          break;
        case PKL_AST_BUILTIN_VM_SET_OPPRINT:
          RAS_MACRO_BUILTIN_VM_SET_OPPRINT;
          break;
        case PKL_AST_BUILTIN_VM_OACUTOFF:
          RAS_MACRO_BUILTIN_VM_OACUTOFF;
          break;
        case PKL_AST_BUILTIN_VM_SET_OACUTOFF:
          RAS_MACRO_BUILTIN_VM_SET_OACUTOFF;
          break;
        case PKL_AST_BUILTIN_VM_ODEPTH:
          RAS_MACRO_BUILTIN_VM_ODEPTH;
          break;
        case PKL_AST_BUILTIN_VM_SET_ODEPTH:
          RAS_MACRO_BUILTIN_VM_SET_ODEPTH;
          break;
        case PKL_AST_BUILTIN_VM_OINDENT:
          RAS_MACRO_BUILTIN_VM_OINDENT;
          break;
        case PKL_AST_BUILTIN_VM_SET_OINDENT:
          RAS_MACRO_BUILTIN_VM_SET_OINDENT;
          break;
        case PKL_AST_BUILTIN_VM_OMAPS:
          RAS_MACRO_BUILTIN_VM_OMAPS;
          break;
        case PKL_AST_BUILTIN_VM_SET_OMAPS:
          RAS_MACRO_BUILTIN_VM_SET_OMAPS;
          break;
        case PKL_AST_BUILTIN_VM_OMODE:
          RAS_MACRO_BUILTIN_VM_OMODE;
          break;
        case PKL_AST_BUILTIN_VM_SET_OMODE:
          RAS_MACRO_BUILTIN_VM_SET_OMODE;
          break;
        case PKL_AST_BUILTIN_UNSAFE_STRING_SET:
          RAS_MACRO_BUILTIN_UNSAFE_STRING_SET;
          break;
        default:
          assert (0);
        }
    }
  else
    /* Pop the lexical frame created by the compound statement.  */
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPF, 1);
}
PKL_PHASE_END_HANDLER

/*
 * INCRDECR
 * | EXP
 * | ASS_STMT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_incrdecr)
{
  pkl_ast_node incrdecr = PKL_PASS_NODE;
  pkl_ast_node incrdecr_exp = PKL_AST_INCRDECR_EXP (incrdecr);
  pkl_ast_node incrdecr_ass_stmt = PKL_AST_INCRDECR_ASS_STMT (incrdecr);
  int incrdecr_order = PKL_AST_INCRDECR_ORDER (incrdecr);

  if (incrdecr_order == PKL_AST_ORDER_PRE)
    PKL_PASS_SUBPASS (incrdecr_ass_stmt);

  PKL_PASS_SUBPASS (incrdecr_exp);

  if (incrdecr_order == PKL_AST_ORDER_POST)
    PKL_PASS_SUBPASS (incrdecr_ass_stmt);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * ASS_STMT
 * | EXP
 * | LVALUE
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_ass_stmt)
{
#define LMAP(TYPE)                                                      \
  do                                                                    \
    {                                                                   \
      int lvalue_type_code = PKL_AST_TYPE_CODE ((TYPE));                \
                                                                        \
      switch (lvalue_type_code)                                         \
        {                                                               \
        case PKL_TYPE_ARRAY:                                            \
        case PKL_TYPE_STRUCT:                                           \
          {                                                             \
            pvm_val writer =                                            \
              (lvalue_type_code == PKL_TYPE_ARRAY                       \
               ? PKL_AST_TYPE_A_WRITER ((TYPE))                         \
               : PKL_AST_TYPE_S_WRITER ((TYPE)));                       \
                                                                        \
            /* If the type is anonymous it wont' have a compiled */     \
            /* writer.  */                                              \
            if (!PKL_AST_TYPE_NAME ((TYPE)))                            \
              {                                                         \
                assert (writer == PVM_NULL);                            \
                PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_WRITER);       \
                {                                                       \
                  if (lvalue_type_code == PKL_TYPE_ARRAY)               \
                    RAS_FUNCTION_ARRAY_WRITER (writer, (TYPE));         \
                  else if (PKL_AST_TYPE_S_UNION_P ((TYPE)))             \
                    RAS_FUNCTION_UNION_WRITER (writer, (TYPE));         \
                  else                                                  \
                    RAS_FUNCTION_STRUCT_WRITER (writer, (TYPE));        \
                }                                                       \
                PKL_GEN_POP_CONTEXT;                                    \
                                                                        \
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, writer);      \
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);               \
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);              \
            }                                                           \
                                                                        \
            /* VAL IOS BOFF */                                          \
            RAS_MACRO_COMPLEX_LMAP ((TYPE), writer); /* _ */            \
            break;                                                      \
          }                                                             \
        default:                                                        \
          /* The map at the l-value is of a simple type, i.e. of */     \
          /* types whose values cannot be mapped (integers, offsets, */ \
          /* strings, etc).  The strategy here is simple: we just */    \
          /*  generate a writer for the type.  */                       \
                                                                        \
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT); /* IOS BOFF VAL */  \
                                                                        \
          PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_WRITER);             \
          PKL_PASS_SUBPASS ((TYPE));                                    \
          PKL_GEN_POP_CONTEXT;                                          \
          break;                                                        \
        }                                                               \
    }                                                                   \
  while (0)

  pkl_ast_node ass_stmt = PKL_PASS_NODE;
  pkl_ast_node lvalue = PKL_AST_ASS_STMT_LVALUE (ass_stmt);
  pkl_ast_node lvalue_type = PKL_AST_TYPE (lvalue);
  pkl_ast_node exp = PKL_AST_ASS_STMT_EXP (ass_stmt);
  pvm_program_label done = pkl_asm_fresh_label (PKL_GEN_ASM);

  PKL_PASS_SUBPASS (exp);

  PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_LVALUE);
  PKL_PASS_SUBPASS (lvalue);
  PKL_GEN_POP_CONTEXT;

  /* At this point the r-value, generated from executing EXP, is in
     the stack.  */

  /* If the base array to the indexer, or the referred struct, are
     mapped, and the assigned value is a complex value, then we have
     to reflect the effect of the assignment in the corresponding IO
     space.  */
  if ((PKL_AST_CODE (lvalue) == PKL_AST_INDEXER
       || PKL_AST_CODE (lvalue) == PKL_AST_STRUCT_REF)
      && (PKL_AST_TYPE_CODE (lvalue_type) == PKL_TYPE_ARRAY
          || PKL_AST_TYPE_CODE (lvalue_type) == PKL_TYPE_STRUCT))
    {
      pvm_program_label not_mapped = pkl_asm_fresh_label (PKL_GEN_ASM);

      /* Stack: VAL (SCT|ARR) (ID|IDX) */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP); /* VAL ID SCT */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MM);  /* VAL ID SCT MAPPED_P */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BZI, not_mapped);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* VAL ID SCT */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP); /* VAL SCT ID */

      if (PKL_AST_CODE (lvalue) == PKL_AST_INDEXER)
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AREFO); /* VAL SCT ID BOFF */
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SREFO); /* VAL SCT ID BOFF */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);     /* VAL SCT BOFF */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);    /* VAL BOFF SCT */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MGETIOS); /* VAL BOFF SCT IOS */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);     /* VAL BOFF IOS */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);    /* VAL IOS BOFF */

      /* VAL IOS OFF */
      LMAP (PKL_AST_TYPE (lvalue));

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BA, done);
      pkl_asm_label (PKL_GEN_ASM, not_mapped);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* VAL ID SCT */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP); /* VAL SCT ID */
    }

  /* All right, now assign the resulting rvalue to the lvalue.  */
  switch (PKL_AST_CODE (lvalue))
    {
    case PKL_AST_VAR:
      {
        /* Stack: VAL */

        pkl_ast_node var = lvalue;
        pkl_ast_node var_decl = PKL_AST_VAR_DECL (var);
        pkl_ast_node var_function = PKL_AST_VAR_FUNCTION (var);

        /* If the declaration associated with the variable is in a
           struct _and_ we are in a method body, we update the
           implicit struct argument.  */
        if (var_function
            && PKL_AST_FUNC_METHOD_P (var_function)
            && (PKL_AST_DECL_STRUCT_FIELD_P (var_decl)
                || (PKL_AST_DECL_KIND (var_decl) == PKL_AST_DECL_KIND_FUNC
                    && PKL_AST_FUNC_METHOD_P (PKL_AST_DECL_INITIAL (var_decl)))))
          {
            pkl_ast_node var_name = PKL_AST_VAR_NAME (var);
            int var_function_back = PKL_AST_VAR_FUNCTION_BACK (var);

            assert (var_name != NULL);

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHVAR,
                          var_function_back, 0);              /* VAL SCT */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          pvm_make_string (PKL_AST_IDENTIFIER_POINTER (var_name)));
                                                              /* VAL SCT STR */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT);         /* SCT STR VAL */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SSET);        /* SCT */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_WRITE);
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
          }
        else
          /* Normal variable.  */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPVAR,
                        PKL_AST_VAR_BACK (lvalue), PKL_AST_VAR_OVER (lvalue));
        break;
      }
    case PKL_AST_INDEXER:
      {
        /* Stack: VAL ARRAY INDEX */

        pkl_ast_node array = PKL_AST_INDEXER_ENTITY (lvalue);
        pkl_ast_node array_type = PKL_AST_TYPE (array);
        pkl_ast_node etype = PKL_AST_TYPE_A_ETYPE (array_type);

        /* If the type of the array is ANY[], then check at runtime
           that the type of the value matches the type of the elements
           in the array.  */
        if (PKL_AST_TYPE_CODE (etype) == PKL_TYPE_ANY)
          {
            pvm_program_label label = pkl_asm_fresh_label (PKL_GEN_ASM);

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NROT);  /* INDEX VAL ARRAY */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TYPOF); /* INDEX VAL ARRAY ATYPE */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TYAGETT); /* INDEX VAL ARRAY ATYPE ETYPE */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);   /* INDEX VAL ARRAY ETYPE */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT);   /* INDEX ARRAY ETYPE VAL */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* INDEX ARRAY VAL ETYPE */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ISA);   /* INDEX ARRAY VAL ETYPE BOOL */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);   /* INDEX ARRAY VAL BOOL */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BNZI, label);

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          pvm_make_exception (PVM_E_CONV, PVM_E_CONV_NAME,
                                              PVM_E_CONV_ESTATUS, NULL, NULL));
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RAISE);

            pkl_asm_label (PKL_GEN_ASM, label);
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);  /* INDEX ARRAY VAL */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* INDEX VAL ARRAY */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT);   /* VAL ARRAY INDEX */
          }

        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT);   /* ARRAY INDEX VAL */
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ASET);  /* ARRAY */
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_WRITE); /* ARRAY */
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);  /* The array
                                                       value.  */
        break;
      }
    case PKL_AST_STRUCT_REF:
      {
        /* Stack: VAL SCT ID */

        pkl_ast_node sct = PKL_AST_INDEXER_ENTITY (lvalue);
        pkl_ast_node struct_type = PKL_AST_TYPE (sct);
        pvm_program_label label1 = pkl_asm_fresh_label (PKL_GEN_ASM);
        pvm_program_label label2 = pkl_asm_fresh_label (PKL_GEN_ASM);

        assert (PKL_AST_TYPE_S_CONSTRUCTOR (struct_type) != PVM_NULL);

        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OVER); /* VAL SCT ID SCT */
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MGETS); /* VAL SCT ID SCT STRICT_P */
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);  /* VAL SCT ID STRICT_P */
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BZI, label1);

        /* Strict value: set with integrity.  */
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SSETC, struct_type);

        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BA, label2);
        pkl_asm_label (PKL_GEN_ASM, label1);

        /* Non-strict value: set with no integrity.  */
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SSET, struct_type);

        pkl_asm_label (PKL_GEN_ASM, label2);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_WRITE);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The struct
                                                      value.  */
        break;
      }
    case PKL_AST_MAP:
      /* Stack: VAL IOS OFF */
      /* Convert the offset to a bit-offset.  The offset is */
      /* guaranteed to be ulong<64> with unit bits, as per  */
      /* promo.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OGETM);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP); /* VAL IOS BOFF */
      LMAP (PKL_AST_TYPE (lvalue));
      break;
    default:
      break;
    }

  if (PKL_AST_CODE (lvalue) == PKL_AST_INDEXER
      || PKL_AST_CODE (lvalue) == PKL_AST_STRUCT_REF)
    pkl_asm_label (PKL_GEN_ASM, done);

  PKL_PASS_BREAK;

#undef LMAP
}
PKL_PHASE_END_HANDLER

/*
 * IF_STMT
 * | EXP
 * | THEN_STMT
 * | [ELSE_STMT]
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_if_stmt)
{
  pkl_ast_node if_stmt = PKL_PASS_NODE;
  pkl_ast_node if_exp = PKL_AST_IF_STMT_EXP (if_stmt);
  pkl_ast_node if_then_stmt = PKL_AST_IF_STMT_THEN_STMT (if_stmt);
  pkl_ast_node if_else_stmt = PKL_AST_IF_STMT_ELSE_STMT (if_stmt);

  if (PKL_AST_CODE (if_exp) == PKL_AST_INTEGER)
    {
      uint64_t exp_value = PKL_AST_INTEGER_VALUE (if_exp);

      if (exp_value == 0)
        {
          if (if_else_stmt)
            PKL_PASS_SUBPASS (if_else_stmt);
        }
      else
        {
          PKL_PASS_SUBPASS (if_then_stmt);
        }

      PKL_PASS_BREAK;
    }

  pkl_asm_if (PKL_GEN_ASM, if_exp);
  {
    PKL_PASS_SUBPASS (if_exp);
  }
  pkl_asm_then (PKL_GEN_ASM);
  {
    PKL_PASS_SUBPASS (if_then_stmt);
  }
  pkl_asm_else (PKL_GEN_ASM);
  {
    if (if_else_stmt)
      PKL_PASS_SUBPASS (if_else_stmt);
  }
  pkl_asm_endif (PKL_GEN_ASM);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * BREAK_STMT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_break_stmt)
{
  int nframes = PKL_AST_BREAK_STMT_NFRAMES (PKL_PASS_NODE);

  if (nframes > 0)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPF, nframes);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BA,
                pkl_asm_break_label (PKL_GEN_ASM));
}
PKL_PHASE_END_HANDLER

/*
 * CONTINUE_STMT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_continue_stmt)
{
  int nframes = PKL_AST_CONTINUE_STMT_NFRAMES (PKL_PASS_NODE);

  if (nframes > 0)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPF, nframes);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BA,
                pkl_asm_continue_label (PKL_GEN_ASM));
}
PKL_PHASE_END_HANDLER

/*
 * LOOP_STMT
 * | PARAMS
 * | BODY
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_loop_stmt)
{
  pkl_ast_node loop_stmt = PKL_PASS_NODE;
  int loop_stmt_kind = PKL_AST_LOOP_STMT_KIND (loop_stmt);
  pkl_ast_node body = PKL_AST_LOOP_STMT_BODY (loop_stmt);

  switch (loop_stmt_kind)
    {
    case PKL_AST_LOOP_STMT_KIND_WHILE:
      {
        pkl_ast_node condition
          = PKL_AST_LOOP_STMT_CONDITION (loop_stmt);

        assert (condition && body);

        if (PKL_AST_CODE (condition) == PKL_AST_INTEGER)
          {
            if (PKL_AST_INTEGER_VALUE (condition) == 0)
              ; /* while (0) is optimized away.  */
            else
              {
                pkl_asm_loop (PKL_GEN_ASM);
                PKL_PASS_SUBPASS (body);
                pkl_asm_endloop (PKL_GEN_ASM);
              }
          }
        else
          {
            pkl_asm_while (PKL_GEN_ASM);
            {
              PKL_PASS_SUBPASS (condition);
            }
            pkl_asm_while_loop (PKL_GEN_ASM);
            {
              PKL_PASS_SUBPASS (body);
            }
            pkl_asm_while_endloop (PKL_GEN_ASM);
          }

      break;
      }
    case PKL_AST_LOOP_STMT_KIND_FOR:
      {
        pkl_ast_node head = PKL_AST_LOOP_STMT_HEAD (loop_stmt);

        pkl_asm_for (PKL_GEN_ASM, head);
        {
          for (; head; head = PKL_AST_CHAIN (head))
            PKL_PASS_SUBPASS (head);
        }
        pkl_asm_for_condition (PKL_GEN_ASM);
        {
          pkl_ast_node condition
            = PKL_AST_LOOP_STMT_CONDITION (loop_stmt);

          if (condition)
            PKL_PASS_SUBPASS (condition);
          else
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_int (1, 32));
        }
        pkl_asm_for_loop (PKL_GEN_ASM);
        {
          PKL_PASS_SUBPASS (body);
        }
        pkl_asm_for_tail (PKL_GEN_ASM);
        {
          pkl_ast_node tail = PKL_AST_LOOP_STMT_TAIL (loop_stmt);

          for (; tail; tail = PKL_AST_CHAIN (tail))
            PKL_PASS_SUBPASS (tail);
        }
        pkl_asm_for_endloop (PKL_GEN_ASM);

        break;
      }
    case PKL_AST_LOOP_STMT_KIND_FOR_IN:
      {
        pkl_ast_node condition
          = PKL_AST_LOOP_STMT_CONDITION (loop_stmt);
        pkl_ast_node loop_stmt_iterator
          = PKL_AST_LOOP_STMT_ITERATOR (loop_stmt);
        pkl_ast_node container = NULL;
        pkl_ast_node container_type = NULL;

        assert (loop_stmt_iterator);

        container = PKL_AST_LOOP_STMT_ITERATOR_CONTAINER (loop_stmt_iterator);
        container_type = PKL_AST_TYPE (container);

        pkl_asm_for_in (PKL_GEN_ASM,
                        PKL_AST_TYPE_CODE (container_type),
                        condition);
        {
          PKL_PASS_SUBPASS (container);
        }
        pkl_asm_for_in_where (PKL_GEN_ASM);
        {
          if (condition)
            PKL_PASS_SUBPASS (condition);
        }
        pkl_asm_for_in_loop (PKL_GEN_ASM);
        {
          PKL_PASS_SUBPASS (body);
        }
        pkl_asm_for_in_endloop (PKL_GEN_ASM);

        break;
      }
    default:
      assert (0);
    }

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * RETURN
 * | EXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_return_stmt)
{
  /* Clean the stack before returning.  */
  size_t i;

  for (i = 0; i < PKL_AST_RETURN_STMT_NDROPS (PKL_PASS_NODE); ++i)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
}
PKL_PHASE_END_HANDLER

/*
 * | EXP
 * RETURN
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_return_stmt)
{
  /* Return from the function: pop N frames and generate a return
     instruction.  */

  pkl_ast_node return_stmt = PKL_PASS_NODE;
  pkl_ast_node function = PKL_AST_RETURN_STMT_FUNCTION (return_stmt);
  pkl_ast_node function_type = PKL_AST_TYPE (function);
  int i;

  for (i = 0; i < PKL_AST_RETURN_STMT_NPOPES (PKL_PASS_NODE); ++i)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPE);

  /* In a void function, return PVM_NULL in the stack.  */
  if (PKL_AST_TYPE_CODE (PKL_AST_TYPE_F_RTYPE (function_type))
      == PKL_TYPE_VOID)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RETURN);
}
PKL_PHASE_END_HANDLER

/*
 * | EXP
 * EXP_STMT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_exp_stmt)
{
  /* Drop the expression from the stack, but not if we are compiling a
     single statement.  */
  if (!(pkl_compiling_statement_p (PKL_GEN_PAYLOAD->compiler)
        && PKL_PASS_PARENT
        && PKL_AST_CODE (PKL_PASS_PARENT) == PKL_AST_PROGRAM)
      || PKL_GEN_PAYLOAD->in_file_p)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
}
PKL_PHASE_END_HANDLER

/*
 * FORMAT
 * | ARG
 * | ...
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_format)
{
  pkl_ast_node format = PKL_PASS_NODE;
  pkl_ast_node format_args = PKL_AST_FORMAT_ARGS (format);
  pkl_ast_node arg;
  char *prefix = PKL_AST_FORMAT_PREFIX (format);
  int nstr = 0;
  /* XXX this hard limit should go away.
     See pkl-tran.c:pkl_trans1_ps_format.  */
#define MAX_CLASS_TAGS 32
  int nclasses = 0;
  struct
  {
    char *name;
    int index;
  } classes[MAX_CLASS_TAGS];

  /* Save all the intermediate strings in an array of strings and at the
     end, concatenate all of elements into a single string on the stack.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_string_type ());
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKTYA);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                pvm_make_ulong (0, 64)); /* FIXME use better hint */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKA);

  if (prefix)
    {
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_ulong (0, 64));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_string (prefix));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AINS);
      ++nstr;
    }

  for (arg = format_args; arg; arg = PKL_AST_CHAIN (arg))
    {
      pkl_ast_node exp = PKL_AST_FORMAT_ARG_EXP (arg);
      char *begin_sc = PKL_AST_FORMAT_ARG_BEGIN_SC (arg);
      char *end_sc = PKL_AST_FORMAT_ARG_END_SC (arg);
      char *suffix = PKL_AST_FORMAT_ARG_SUFFIX (arg);
      int base = PKL_AST_FORMAT_ARG_BASE (arg);
      pkl_ast_node exp_type;
      int arg_omode;
      int arg_odepth;

      if (begin_sc)
        {
          assert (nclasses < MAX_CLASS_TAGS);
          classes[nclasses].name = begin_sc;
          classes[nclasses].index = nstr;
          ++nclasses;
        }

      if (end_sc)
        {
          char* cname;
          int cindex, n;
          pvm_val idx;

          assert (nclasses > 0);
          --nclasses;
          cname = classes[nclasses].name;
          cindex = classes[nclasses].index;

          assert (STREQ (cname, end_sc));

          n = nstr - cindex;
          if (n > 0)
            {
              nstr = cindex + 1;
              idx = pvm_make_ulong (cindex, 64);

              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DUP);       /* ARR ARR */
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, idx); /* ARR ARR IDX */
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                            pvm_make_ulong (n, 64));         /* ARR ARR IDX LEN */
              pkl_asm_call (PKL_GEN_ASM, PKL_GEN_PAYLOAD->env,
                            "_pkl_reduce_string_array"); /* ARR STR */
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);     /* STR ARR */

              for (; n > 1; --n)
                {
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, idx);
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AREM);
                }

              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, idx); /* STR ARR IDX */
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT);       /* ARR IDX STR */

              /* Set the string style property.  */
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SEL);
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                            pvm_make_ulong (0, 64));
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP); /* STR 0UL LEN */
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                            pvm_make_string (cname)); /* STR 0UL LEN CLASS */
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SPROPS); /* ARR STR */

              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ASET); /* ARR */
            }
        }

      if (!exp)
        goto fmt_suffix;

      /* Generate code to put the value on the stack.  */
      PKL_PASS_SUBPASS (exp);

      /* Everything except %v.  */
      if (!PKL_AST_FORMAT_ARG_VALUE_P (arg))
        {
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        base ? pvm_make_int (base, 32) : PVM_NULL);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FORMAT, PKL_AST_TYPE (exp));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_ulong (nstr++, 64));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AINS);
          goto fmt_suffix;
        }

      /* Generate code to format the literal value (%v).  */
      exp_type = PKL_AST_TYPE (exp);
      arg_omode = PKL_AST_FORMAT_ARG_FORMAT_MODE (arg);
      arg_odepth = PKL_AST_FORMAT_ARG_FORMAT_DEPTH (arg);

      /* Set the argument's own omode and odepth, saving
         the VM's own.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHOM); /* OMODE */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_int (arg_omode, 32)); /* OMODE NOMODE */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPOM);  /* OMODE */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHOD); /* OMODE ODEPTH */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_int (arg_odepth, 32)); /* OMODE ODEPTH NODEPTH */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPOD);   /* OMODE ODEPTH */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT); /* OMODE ODEPTH EXP */

      /* Format the value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_int (0, 32)); /* OMODE ODEPTH EXP DEPTH */
      PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_FORMATER);
      PKL_PASS_SUBPASS (exp_type); /* OMODE ODEPTH STR */
      PKL_GEN_POP_CONTEXT;

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NROT); /* STR OMODE ODEPTH */

      /* Restore the current omode and odepth in the VM.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPOD); /* ARR STR OMODE */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPOM); /* ARR STR */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_ulong (nstr++, 64)); /* ARR STR IDX */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* ARR IDX STR */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AINS);  /* ARR */

    fmt_suffix:
      if (suffix)
        {
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_ulong (nstr++, 64));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_string (suffix));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AINS);
        }
    }

  if (nstr)
    {
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DUP);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_ulong (0, 64));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_ulong (nstr, 64));
      pkl_asm_call (PKL_GEN_ASM, PKL_GEN_PAYLOAD->env, "_pkl_reduce_string_array");
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
    }

  PKL_PASS_BREAK;

#undef MAX_CLASS_TAGS
}
PKL_PHASE_END_HANDLER

/*
 * PRINT_STMT
 * | ARG
 * | ...
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_print_stmt)
{
  pkl_ast_node print_stmt = PKL_PASS_NODE;
  pkl_ast_node print_stmt_str_exp = PKL_AST_PRINT_STMT_STR_EXP (print_stmt);

  if (print_stmt_str_exp) /* print statement */
    {
      PKL_PASS_SUBPASS (print_stmt_str_exp);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PRINTS);
    }
  else /* printf statement */
    {
      pkl_ast_node print_stmt_fmt = PKL_AST_PRINT_STMT_FORMAT (print_stmt);
      pkl_ast_node print_stmt_args = PKL_AST_FORMAT_ARGS (print_stmt_fmt);
      pkl_ast_node arg;
      char *prefix = PKL_AST_FORMAT_PREFIX (print_stmt_fmt);
      int nexp;

      /* First, compute the arguments and push them to the stack.  */

      for (nexp = 0, arg = print_stmt_args;
           arg;
           arg = PKL_AST_CHAIN (arg))
        {
          pkl_ast_node exp = PKL_AST_FORMAT_ARG_EXP (arg);

          if (exp)
            {
              PKL_PASS_SUBPASS (exp);
              nexp++;
            }
        }

      /* Reverse the arguments in the stack so we can print it in the
         right order.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REV, nexp);

      /* Now print out the stuff.  */
      if (prefix)
        {
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_string (prefix));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PRINTS);
        }

      for (arg = print_stmt_args; arg; arg = PKL_AST_CHAIN (arg))
        {
          /* Handle the argument.  */

          pkl_ast_node exp = PKL_AST_FORMAT_ARG_EXP (arg);
          char *begin_sc = PKL_AST_FORMAT_ARG_BEGIN_SC (arg);
          char *end_sc = PKL_AST_FORMAT_ARG_END_SC (arg);
          char *suffix = PKL_AST_FORMAT_ARG_SUFFIX (arg);
          int base = PKL_AST_FORMAT_ARG_BASE (arg);

          if (begin_sc)
            {
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                            pvm_make_string (begin_sc));
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BEGINSC);
            }

          if (end_sc)
            {
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                            pvm_make_string (end_sc));
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ENDSC);
            }

          if (exp)
            {
              if (PKL_AST_FORMAT_ARG_VALUE_P (arg))
                {
                  /* Generate code to print the value.  */
                  pkl_ast_node exp_type = PKL_AST_TYPE (exp);
                  int arg_omode = PKL_AST_FORMAT_ARG_FORMAT_MODE (arg);
                  int arg_odepth = PKL_AST_FORMAT_ARG_FORMAT_DEPTH (arg);

                  /* Set the argument's own omode and odepth, saving
                     the VM's own.  */
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHOM); /* OMODE */
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                                pvm_make_int (arg_omode, 32)); /* OMODE NOMODE */
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPOM); /* OMODE */

                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHOD); /* OMODE ODEPTH */
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                                pvm_make_int (arg_odepth, 32)); /* OMODE ODEPTH NODEPTH */
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPOD); /* OMODE ODEPTH */

                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT); /* OMODE ODEPTH EXP */

                  /* Print out the value.  */
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                                pvm_make_int (0, 32)); /* OMODE ODEPTH EXP DEPTH */
                  PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_PRINTER);
                  PKL_PASS_SUBPASS (exp_type);
                  PKL_GEN_POP_CONTEXT;

                  /* Restore the current omode and odepth in the VM.  */
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPOD); /* OMODE */
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POPOM); /* _ */
                }
              else
                {
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                                base ? pvm_make_int (base, 32) : PVM_NULL);
                  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PRINT, PKL_AST_TYPE (exp));
                }
            }

          if (suffix)
            {
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_string (suffix));
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PRINTS);
            }
        }
    }

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * | [EXP]
 * RAISE_STMT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_raise_stmt)
{
  pkl_ast_node raise_stmt = PKL_PASS_NODE;

  /* If the `raise' statement was anonymous, then we need to push the
     exception to raise, which by default, is 0.  */
  if (PKL_AST_RAISE_STMT_EXP (raise_stmt) == NULL)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                  pvm_make_exception (PVM_E_GENERIC, PVM_E_GENERIC_NAME,
                                      PVM_E_GENERIC_ESTATUS, NULL, NULL));

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RAISE);
}
PKL_PHASE_END_HANDLER

/*
 * | CODE
 * | EXP
 * TRY_UNTIL_STMT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_try_until_stmt)
{
  pkl_ast_node try_until_stmt = PKL_PASS_NODE;
  pkl_ast_node code = PKL_AST_TRY_UNTIL_STMT_CODE (try_until_stmt);
  pkl_ast_node exp = PKL_AST_TRY_UNTIL_STMT_EXP (try_until_stmt);

  /* Push the exception to catch.  */
  PKL_PASS_SUBPASS (exp);
  pkl_asm_try (PKL_GEN_ASM, NULL);
  {
    pkl_asm_loop (PKL_GEN_ASM);
    PKL_PASS_SUBPASS (code);
    pkl_asm_endloop (PKL_GEN_ASM);
  }
  pkl_asm_catch (PKL_GEN_ASM);
  {
  }
  pkl_asm_endtry (PKL_GEN_ASM);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * | CODE
 * | HANDLER
 * | [ARG]
 * | [EXP]
 * TRY_CATCH_STMT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_try_catch_stmt)
{
  pkl_ast_node try_catch_stmt = PKL_PASS_NODE;
  pkl_ast_node code = PKL_AST_TRY_CATCH_STMT_CODE (try_catch_stmt);
  pkl_ast_node handler = PKL_AST_TRY_CATCH_STMT_HANDLER (try_catch_stmt);
  pkl_ast_node catch_arg = PKL_AST_TRY_CATCH_STMT_ARG (try_catch_stmt);
  pkl_ast_node catch_exp = PKL_AST_TRY_CATCH_STMT_EXP (try_catch_stmt);

  /* Push the exception that will be catched by the sentence.  This is
     EXP if it is defined, or E_generic if it isnt.  */
  if (catch_exp)
    PKL_PASS_SUBPASS (catch_exp);
  else
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                  pvm_make_exception (PVM_E_GENERIC, PVM_E_GENERIC_NAME,
                                      PVM_E_GENERIC_ESTATUS, NULL, NULL));

  pkl_asm_try (PKL_GEN_ASM, catch_arg);
  {
    PKL_PASS_SUBPASS (code);
  }
  pkl_asm_catch (PKL_GEN_ASM);
  {
    PKL_PASS_SUBPASS (handler);
  }
  pkl_asm_endtry (PKL_GEN_ASM);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * | EXP
 * FUNCALL_ARG
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_funcall_arg)
{
  /* No extra action is required here.  */
}
PKL_PHASE_END_HANDLER


/* FUNCALL
 * | [ARG]
 * | ...
 * | FUNCTION
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_funcall)
{
  pkl_ast_node funcall = PKL_PASS_NODE;
  pkl_ast_node function = PKL_AST_FUNCALL_FUNCTION (funcall);
  pkl_ast_node function_type = PKL_AST_TYPE (function);
  int vararg = PKL_AST_TYPE_F_VARARG (function_type);
  int i, aindex = 0, vararg_actual = 0, optionals_specified = 0;
  pkl_ast_node aa;

  /* Push the actuals to the stack. */
  for (aa = PKL_AST_FUNCALL_ARGS (funcall); aa; aa = PKL_AST_CHAIN (aa))
    {
      if (PKL_AST_FUNCALL_ARG_FIRST_VARARG (aa))
        vararg_actual = 1;

      if (vararg_actual)
        aindex++;

      if (!PKL_AST_FUNCALL_ARG_EXP (aa))
        {
          /* This is a non-specified actual for a formal having a
             default value.  */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
          optionals_specified++;
        }
      else
        PKL_PASS_SUBPASS (aa);
    }

  if (vararg)
    {
      /* The actuals are stored in the stack in reverse order.
         Reverse them.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REV, aindex);

      /* Create the array of variable arguments.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_any_type ());
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKTYA);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_ulong (aindex, 64));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKA);

      /* Insert the elements in the array.  */
      for (i = 0; i < aindex; ++i)
        {
                                                     /* ... ELEM ARR */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        pvm_make_ulong (i, 64));     /* ... ELEM ARR IDX */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ROT);  /* ... ARR IDX ELEM */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AINS); /* ... ARR */
        }
    }

  /* Complete non-specified actuals for formals having default values.
     For these, we should push nulls.  But beware the vararg!  */
  {
    int non_specified
      = (PKL_AST_TYPE_F_NARG (function_type)
         - PKL_AST_FUNCALL_NARG (funcall)
         - PKL_AST_TYPE_F_VARARG (function_type)
         - optionals_specified);

    for (i = 0; i < non_specified; ++i)
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
  }

  /* Push the closure for FUNCTION and call the bloody function.  */
  PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_FUNCALL);
  PKL_PASS_SUBPASS (PKL_AST_FUNCALL_FUNCTION (funcall));
  PKL_GEN_POP_CONTEXT;
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);
  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * FUNC
 * | [TYPE]
 * | [FUNC_ARG]
 * | ...
 * | BODY
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_func)
{
  pkl_ast_node function = PKL_PASS_NODE;
  int nargs = PKL_AST_FUNC_NARGS (function);
  int method_p = PKL_AST_FUNC_METHOD_P (PKL_PASS_NODE);

  /* This is a function prologue.  */
  if (PKL_AST_FUNC_NAME (function))
    pkl_asm_note (PKL_GEN_ASM,
                  PKL_AST_FUNC_NAME (function));
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PROLOG);

  if (nargs > 1)
    {
      /* Reverse the arguments.

         Note that in methods the implicit struct argument is passed
         as the last actual.  However, we have to process it as the
         _first_ formal.  We achieve this by not reversing it, saving
         it in the return stack temporarily.  */

      if (method_p)
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TOR);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REV, nargs);

      if (method_p)
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FROMR);
    }

  /* If the function's return type is an array type, make sure it has
     a bounder.  If it hasn't one, then compute it in this
     environment.  */
  {
    pkl_ast_node rtype = PKL_AST_FUNC_RET_TYPE (function);

    if (PKL_AST_TYPE_CODE (rtype) == PKL_TYPE_ARRAY
        && PKL_AST_TYPE_A_BOUNDER (rtype) == PVM_NULL)
      {
        PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_ARRAY_BOUNDER);
        PKL_PASS_SUBPASS (rtype);
        PKL_GEN_POP_CONTEXT;
      }
  }

  /* Push the function environment, for the arguments.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHF,
                method_p ? nargs + 1 : nargs);

  /* If in a method, register the implicit argument.  */
  if (method_p)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REGVAR);
}
PKL_PHASE_END_HANDLER

/*
 * FUNC_ARG
 * | TYPE
 * | INITIAL
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_func_arg)
{
  pkl_asm pasm = PKL_GEN_ASM;
  pkl_ast_node func_arg = PKL_PASS_NODE;
  pkl_ast_node func_arg_initial = PKL_AST_FUNC_ARG_INITIAL (func_arg);
  pkl_ast_node func_arg_type = PKL_AST_FUNC_ARG_TYPE (func_arg);
  pvm_program_label after_conv_label = pkl_asm_fresh_label (PKL_GEN_ASM);

  /* Traverse the argument type in normal context.  */
  PKL_GEN_PUSH_CONTEXT;
  PKL_PASS_SUBPASS (func_arg_type); /* _ */
  PKL_GEN_POP_CONTEXT;

  if (func_arg_initial)
    {
      pvm_program_label label = pkl_asm_fresh_label (PKL_GEN_ASM);

      /* If the value on the stack is `null', that means we need to
         use the default value for the argument.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BNN, label);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* Drop the null */
      PKL_PASS_SUBPASS (func_arg_initial);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BA, after_conv_label);
      pkl_asm_label (PKL_GEN_ASM, label);
    }

  /* If the argument is an array, check/cast to its type, in order to
     perform whatever needed run-time checks.  This is done here and
     not in a cast at funcall time because the argument's type is
     evaluated in the function's lexical environment.  As per promo,
     we know that the value on the stack is an array with the same
     base type, but possibly different bounding.

     Note that if the initial argument is used, then the flow jumps to
     `after_conv_label' and therefore the code below is not executed,
     as promo already performed a cast if needed.  */
  if (PKL_AST_TYPE_CODE (func_arg_type) == PKL_TYPE_ARRAY)
    {
      /* Make sure the cast type has a bounder.  If it doesn't,
         compile and install one.  */
      int bounder_created_p = 0;

      if (PKL_AST_TYPE_A_BOUNDER (func_arg_type) == PVM_NULL)
        {
          bounder_created_p = 1;
          PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_ARRAY_BOUNDER);
          PKL_PASS_SUBPASS (func_arg_type);
          PKL_GEN_POP_CONTEXT;
        }

      pkl_asm_insn (pasm, PKL_INSN_ATOA,
                    NULL /* from_type */, func_arg_type);

      if (bounder_created_p)
        pkl_ast_array_type_remove_bounders (func_arg_type);
    }

  pkl_asm_label (PKL_GEN_ASM, after_conv_label);

  /* Pop the actual argument from the stack and put it in the current
     environment.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REGVAR);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * | [TYPE]
 * | [FUNC_ARG]
 * | ...
 * | BODY
 * FUNC
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_func)
{
  /* Function epilogue.  */

  pkl_ast_node function = PKL_PASS_NODE;
  pkl_ast_node function_type = PKL_AST_TYPE (function);

  /* In a void function, return PVM_NULL in the stack.  Otherwise, it
     is a run-time error to reach this point.  */
  if (PKL_AST_TYPE_CODE (PKL_AST_TYPE_F_RTYPE (function_type))
      == PKL_TYPE_VOID)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
  else
    {
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_exception (PVM_E_NO_RETURN, PVM_E_NO_RETURN_NAME,
                                        PVM_E_NO_RETURN_ESTATUS, NULL, NULL));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RAISE);
    }

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RETURN);
}
PKL_PHASE_END_HANDLER

/*
 * INTEGER
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_integer)
{
  pkl_ast_node integer = PKL_PASS_NODE;
  pkl_ast_node type;
  pvm_val val;
  int size;
  uint64_t value;

  type = PKL_AST_TYPE (integer);
  assert (type != NULL
          && PKL_AST_TYPE_CODE (type) == PKL_TYPE_INTEGRAL);

  size = PKL_AST_TYPE_I_SIZE (type);
  value = PKL_AST_INTEGER_VALUE (integer);

  if ((size - 1) & ~0x1f)
    {
      if (PKL_AST_TYPE_I_SIGNED_P (type))
        val = pvm_make_long (value, size);
      else
        val = pvm_make_ulong (value, size);
    }
  else
    {
      if (PKL_AST_TYPE_I_SIGNED_P (type))
        val = pvm_make_int (value, size);
      else
        val = pvm_make_uint (value, size);
    }

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, val);
}
PKL_PHASE_END_HANDLER

/*
 * IDENTIFIER
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_identifier)
{
  pkl_ast_node identifier = PKL_PASS_NODE;
  pvm_val val
    = pvm_make_string (PKL_AST_IDENTIFIER_POINTER (identifier));

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, val);
}
PKL_PHASE_END_HANDLER

/*
 * STRING
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_string)
{
  pkl_ast_node string = PKL_PASS_NODE;
  pvm_val val
    = pvm_make_string (PKL_AST_STRING_POINTER (string));

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, val);
}
PKL_PHASE_END_HANDLER

/*
 * TYPE_OFFSET
 * | BASE_TYPE
 * | UNIT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_type_offset)
{
  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_WRITER))
    {
      /* Stack: IOS BOFF VAL */
      /* The offset to poke is stored in the TOS.  Replace the offset
         at the TOS with the magnitude of the offset and let the
         BASE_TYPE handler to tackle it.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OGETM); /* IOS BOFF VAL VMAG */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);   /* IOS BOFF VMAG */

      PKL_PASS_SUBPASS (PKL_AST_TYPE_O_BASE_TYPE (PKL_PASS_NODE)); /* _ */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER | PKL_GEN_CTX_IN_CONSTRUCTOR))
    {
      PKL_PASS_SUBPASS (PKL_AST_TYPE_O_BASE_TYPE (PKL_PASS_NODE)); /* VAL */
      PKL_PASS_SUBPASS (PKL_AST_TYPE_O_UNIT (PKL_PASS_NODE));      /* VAL UNIT */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKO);                    /* OFF */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_PRINTER))
    {
                                                /* VAL DEPTH */
      RAS_MACRO_OFFSET_PRINTER (PKL_PASS_NODE); /* _ */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_FORMATER))
    {
                                                /* VAL DEPTH */
      RAS_MACRO_OFFSET_FORMATER (PKL_PASS_NODE); /* _ */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPIFIER))
    {
      RAS_MACRO_OFFSET_TYPIFIER (PKL_PASS_NODE); /* SCT */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    {
    /* Just build an offset type.  */
      PKL_PASS_SUBPASS (PKL_AST_TYPE_O_BASE_TYPE (PKL_PASS_NODE)); /* BASE_TYPE */
      PKL_PASS_SUBPASS (PKL_AST_TYPE_O_UNIT (PKL_PASS_NODE));      /* UNIT */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKTYO);
      PKL_PASS_BREAK;
    }

  /* We are in the normal context.  Process the base type, but not the
     unit as we don't need it.  */
  PKL_PASS_SUBPASS (PKL_AST_TYPE_O_BASE_TYPE (PKL_PASS_NODE));
  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * | TYPE
 * | MAGNITUDE
 * | UNIT
 * OFFSET
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_offset)
{
  pkl_asm pasm = PKL_GEN_ASM;

  pkl_asm_insn (pasm, PKL_INSN_MKO);
}
PKL_PHASE_END_HANDLER

/*
 * | TYPE
 * | EXP
 * ISA
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_isa)
{
  PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_TYPE);
  PKL_PASS_SUBPASS (PKL_AST_ISA_TYPE (PKL_PASS_NODE));
  PKL_GEN_POP_CONTEXT;

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ISA);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);
}
PKL_PHASE_END_HANDLER

/*
 * CAST
 * | TYPE
 * | EXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_cast)
{
  pkl_asm pasm = PKL_GEN_ASM;
  pkl_ast_node node = PKL_PASS_NODE;

  pkl_ast_node exp;
  pkl_ast_node to_type;
  pkl_ast_node from_type;

  exp = PKL_AST_CAST_EXP (node);

  to_type = PKL_AST_CAST_TYPE (node);
  from_type = PKL_AST_TYPE (exp);

  /* Traverse the type and expression in normal context.  */
  PKL_GEN_PUSH_CONTEXT;
  PKL_PASS_SUBPASS (to_type);  /* _ */
  PKL_PASS_SUBPASS (exp);      /* EXP */
  PKL_GEN_POP_CONTEXT;

  /* And finally generate code for the cast operation.  */
  if (PKL_AST_TYPE_CODE (from_type) == PKL_TYPE_ANY)
    {
      pvm_program_label label = pkl_asm_fresh_label (PKL_GEN_ASM);

      PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_TYPE);
      PKL_PASS_SUBPASS (to_type);
      PKL_GEN_POP_CONTEXT;

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ISA);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BNZI, label);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_exception (PVM_E_CONV, PVM_E_CONV_NAME,
                                        PVM_E_CONV_ESTATUS, NULL, NULL));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RAISE);
      pkl_asm_label (PKL_GEN_ASM, label);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
    }
  else if (PKL_AST_TYPE_CODE (from_type) == PKL_TYPE_INTEGRAL
           && PKL_AST_TYPE_CODE (to_type) == PKL_TYPE_INTEGRAL)
    {
      pkl_asm_insn (pasm, PKL_INSN_NTON,
                    from_type, to_type);
      pkl_asm_insn (pasm, PKL_INSN_NIP);
    }
  else if (PKL_AST_TYPE_CODE (from_type) == PKL_TYPE_OFFSET
           && PKL_AST_TYPE_CODE (to_type) == PKL_TYPE_OFFSET)
    {
      pkl_ast_node to_unit = PKL_AST_TYPE_O_UNIT (to_type);

      pkl_asm_insn (pasm, PKL_INSN_PUSH,
                    pvm_make_ulong (PKL_AST_INTEGER_VALUE (to_unit), 64));
      pkl_asm_insn (pasm, PKL_INSN_OTO, from_type, to_type);
    }
  else if (PKL_AST_TYPE_CODE (to_type) == PKL_TYPE_STRING)
    {
      pkl_asm_insn (pasm, PKL_INSN_CTOS);
      pkl_asm_insn (pasm, PKL_INSN_NIP);
    }
  else if (PKL_AST_TYPE_CODE (to_type) == PKL_TYPE_ARRAY
           && PKL_AST_TYPE_CODE (from_type) == PKL_TYPE_ARRAY)
    {
      /* Make sure the cast type has a bounder.  If it doesn't,
         compile and install one.  */
      int bounder_created_p = 0;

      if (PKL_AST_TYPE_A_BOUNDER (to_type) == PVM_NULL)
        {
          bounder_created_p = 1;
          PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_ARRAY_BOUNDER);
          PKL_PASS_SUBPASS (to_type);
          PKL_GEN_POP_CONTEXT;
        }

      pkl_asm_insn (pasm, PKL_INSN_ATOA, from_type, to_type);

      if (bounder_created_p)
        pkl_ast_array_type_remove_bounders (to_type);
    }
  else if (PKL_AST_TYPE_CODE (to_type) == PKL_TYPE_STRUCT
           && PKL_AST_TYPE_CODE (from_type) == PKL_TYPE_STRUCT)
    {
      pvm_val constructor = PKL_AST_TYPE_S_CONSTRUCTOR (to_type);

      /* The constructor should exist, because a struct type specified
         in a cast shall be referred by name.  */
      assert (constructor != PVM_NULL);

      /* Apply the constructor to the expression, which is also a
         struct.  */
      pkl_asm_insn (pasm, PKL_INSN_PUSH, constructor);
      pkl_asm_insn (pasm, PKL_INSN_CALL);
    }
  else if (PKL_AST_TYPE_CODE (to_type) == PKL_TYPE_STRUCT
           && PKL_AST_TYPE_CODE (from_type) == PKL_TYPE_INTEGRAL)
    {
      PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_DEINTEGRATOR);
      PKL_PASS_SUBPASS (to_type);
      PKL_GEN_POP_CONTEXT;
    }
  else if (PKL_AST_TYPE_CODE (to_type) == PKL_TYPE_INTEGRAL
           && PKL_AST_TYPE_CODE (from_type) == PKL_TYPE_STRUCT)
    {
      pkl_ast_node itype = PKL_AST_TYPE_S_ITYPE (from_type);

      /* This is guaranteed as per typify.  */
      assert (itype);

      PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_INTEGRATOR);
      PKL_PASS_SUBPASS (from_type);
      PKL_GEN_POP_CONTEXT;

      pkl_asm_insn (pasm, PKL_INSN_NTON, itype, to_type);
      pkl_asm_insn (pasm, PKL_INSN_NIP);
    }
  else if (PKL_AST_TYPE_CODE (from_type) == PKL_TYPE_ARRAY
           && PKL_AST_TYPE_CODE (to_type) == PKL_TYPE_INTEGRAL)
    {
      PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_INTEGRATOR);
      PKL_PASS_SUBPASS (from_type);
      PKL_GEN_POP_CONTEXT;

                                          /* IVAL(ULONG) WIDTH(UINT) */
      pkl_asm_insn (pasm, PKL_INSN_DROP); /* IVAL(ULONG) */
      pkl_asm_insn (pasm, PKL_INSN_NTON,
                    pkl_ast_make_integral_type (PKL_PASS_AST, 64, 0),
                    to_type);             /* IVAL(ULONG) IVAL */
      pkl_asm_insn (pasm, PKL_INSN_NIP);  /* IVAL */
    }
  else
    assert (0);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * | CONS_TYPE
 * | [CONS_VALUE]
 * CONS
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_cons)
{
  pkl_ast_node cons = PKL_PASS_NODE;
  int cons_kind = PKL_AST_CONS_KIND (cons);
  pkl_ast_node cons_type = PKL_AST_CONS_TYPE (cons);

  switch (cons_kind)
    {
    case PKL_AST_CONS_KIND_ARRAY:
      /* Build an array with default values.  Note how array
         constructors do not use their argument.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
      PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_CONSTRUCTOR);
      PKL_PASS_SUBPASS (cons_type);
      PKL_GEN_POP_CONTEXT;

      /* If an initial value has been provided, set the elements of
         the array to this value.  */
      if (PKL_AST_CONS_VALUE (cons))
        {
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* ARR IVAL */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AFILL); /* ARR IVAL */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);  /* ARR */
        }
      break;
    case PKL_AST_CONS_KIND_STRUCT:
      PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_CONSTRUCTOR);
      PKL_PASS_SUBPASS (cons_type);
      PKL_GEN_POP_CONTEXT;
      break;
    default:
      assert (0);
    }
}
PKL_PHASE_END_HANDLER

/*
 * MAP
 * | [MAP_IOS]
 * | MAP_OFFSET
 * | MAP_TYPE
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_map)
{
  pkl_ast_node map = PKL_PASS_NODE;
  pkl_ast_node map_offset = PKL_AST_MAP_OFFSET (map);
  pkl_ast_node map_ios = PKL_AST_MAP_IOS (map);
  pkl_ast_node map_type = PKL_AST_MAP_TYPE (map);

  /* Traverse the map type in normal context.  */
  PKL_GEN_PUSH_CONTEXT;
  PKL_PASS_SUBPASS (map_type);
  PKL_GEN_POP_CONTEXT;

  if (PKL_PASS_PARENT == NULL
      && PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_LVALUE))
    {
      /* This is an l-value in an assignment.  Generate code for the
         offset, which is expected by the ass_stmt PS handler.  */
      if (map_ios)
        {
          PKL_GEN_PUSH_CONTEXT;
          PKL_PASS_SUBPASS (map_ios);
          PKL_GEN_POP_CONTEXT;
        }
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHIOS);

      PKL_GEN_PUSH_CONTEXT;
      PKL_PASS_SUBPASS (map_offset);
      PKL_GEN_POP_CONTEXT;
    }
  else
    {
      pkl_ast_node map_offset_magnitude = NULL;

      /* Push the strictness to use for the map.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_int (PKL_AST_MAP_STRICT_P (map), 32));

      /* Push the IOS of the map.  */
      if (map_ios)
        PKL_PASS_SUBPASS (map_ios);
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSHIOS);

      /* Push the offset of the map and convert to a bit-offset.  Note
         that the offset is guaranteed to be an ulong<64> with unit
         bits, as per promo.

         But optimize for offsets whose magnitude is an integer node,
         transforming to bit offsets at compile time.  */
      if (PKL_AST_CODE (map_offset) == PKL_AST_OFFSET)
        map_offset_magnitude = PKL_AST_OFFSET_MAGNITUDE (map_offset);

      if (map_offset_magnitude
          && PKL_AST_CODE (map_offset_magnitude) == PKL_AST_INTEGER)
        {
          uint64_t magnitude
            = PKL_AST_INTEGER_VALUE (map_offset_magnitude);

          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        pvm_make_ulong (magnitude, 64));
        }
      else
        {
          PKL_PASS_SUBPASS (map_offset);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OGETM);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
        }

      PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_MAPPER);
      PKL_PASS_SUBPASS (map_type);
      PKL_GEN_POP_CONTEXT;
    }

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * ARRAY_INITIALIZER
 * | ARRAY_INITIALIZER_INDEX
 * | ARRAY_INITIALIZER_EXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_array_initializer)
{
  /* Do nothing.  */
}
PKL_PHASE_END_HANDLER

/*
 * | ARRAY_INITIALIZER_INDEX
 * | ARRAY_INITIALIZER_EXP
 * ARRAY_INITIALIZER
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_array_initializer)
{
  /* Insert this initializer in the array.  */
                                             /* ARR IDX EXP */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AINS); /* ARR */
}
PKL_PHASE_END_HANDLER

/*
 *  ARRAY
 *  | ARRAY_INITIALIZER
 *  | ...
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_array)
{
  pkl_ast_node array = PKL_PASS_NODE;
  pkl_ast_node array_type = PKL_AST_TYPE (array);
  pvm_val array_type_writer = PVM_NULL;

  /* Create a new empty array of the right type, having the right
     number of elements.  */

  PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_TYPE);
  PKL_PASS_SUBPASS (array_type);             /* TYP */
  PKL_GEN_POP_CONTEXT;

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                pvm_make_ulong (PKL_AST_ARRAY_NELEM (array), 64));
                                            /* TYP NELEM */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKA); /* ARR */

  /* Install a writer in the array.  */
  PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_WRITER);
  RAS_FUNCTION_ARRAY_WRITER (array_type_writer, array_type);
  PKL_GEN_POP_CONTEXT;
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, array_type_writer); /* CLS */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);                     /* CLS */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETW);                   /* ARR */
}
PKL_PHASE_END_HANDLER

/*
 * | ARRAY_TYPE
 * | ARRAY_INITIALIZER
 * | ...
 * ARRAY
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_array)
{
  /* Nothing to do here.  */
}
PKL_PHASE_END_HANDLER

/*
 * TRIMMER
 * | ENTITY
 * | FROM
 * | TO
 * | ADDEND
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_trimmer)
{
  pkl_ast_node trimmer = PKL_PASS_NODE;
  pkl_ast_node trimmer_type = PKL_AST_TYPE (trimmer);
  pkl_ast_node trimmer_entity = PKL_AST_TRIMMER_ENTITY (trimmer);
  pkl_ast_node trimmer_from = PKL_AST_TRIMMER_FROM (trimmer);
  pkl_ast_node trimmer_to = PKL_AST_TRIMMER_TO (trimmer);

  PKL_PASS_SUBPASS (trimmer_entity);
  PKL_PASS_SUBPASS (trimmer_from);
  PKL_PASS_SUBPASS (trimmer_to);

  switch (PKL_AST_TYPE_CODE (trimmer_type))
    {
    case PKL_TYPE_STRING:
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SUBSTR);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
      break;
    case PKL_TYPE_ARRAY:
      {
        pkl_ast_node array = PKL_AST_TRIMMER_ENTITY (trimmer);

        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ATRIM,
                      PKL_AST_TYPE (array));
        break;
      }
    default:
      assert (0);
    }

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * INDEXER
 * | INDEXER_ENTITY
 * | INDEXER_INDEX
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_indexer)
{
  pkl_ast_node indexer = PKL_PASS_NODE;
  pkl_ast_node indexer_entity = PKL_AST_INDEXER_ENTITY (indexer);
  pkl_ast_node indexer_index = PKL_AST_INDEXER_INDEX (indexer);

  /* Traverse the entity and indexer in normal context.  */
  PKL_GEN_PUSH_CONTEXT;
  PKL_PASS_SUBPASS (indexer_entity);
  PKL_PASS_SUBPASS (indexer_index);
  PKL_GEN_POP_CONTEXT;

  if (PKL_PASS_PARENT == NULL
      && PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_LVALUE))
    {
      /* This is a l-value in an assignment.  The array and the index
         are pushed to the stack for the ass_stmt PR handler.  Nothing
         else to do here.  Note that analf guarantees that the entity
         in this indexer is an array, not a string.  */
    }
  else
    {
      pkl_ast_node indexer_type = PKL_AST_TYPE (indexer);
      pkl_ast_node container = PKL_AST_INDEXER_ENTITY (indexer);
      pkl_ast_node container_type = PKL_AST_TYPE (container);

      switch (PKL_AST_TYPE_CODE (container_type))
        {
        case PKL_TYPE_ARRAY:
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AREF);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);

          /* To cover cases where the referenced array is not mapped, but
             the value stored in it is a mapped value, we issue a
             REMAP.  */
          switch (PKL_AST_TYPE_CODE (indexer_type))
            {
            case PKL_TYPE_ARRAY:
            case PKL_TYPE_STRUCT:
              /* XXX: this is redundant IO for many (most?) cases.  */
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REMAP);
              break;
            default:
              break;
            }
          break;
        case PKL_TYPE_STRING:
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_STRREF);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);
          break;
        default:
          assert (0);
        }
    }

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * STRUCT
 * | STRUCT_FIELD
 * | ...
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_struct)
{
  /* The offset of the new struct, which should be PVM_NULL, as it is
     not mapped.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
}
PKL_PHASE_END_HANDLER

/*
 *  | STRUCT_FIELD
 *  | ...
 *  STRUCT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_struct)
{
  pkl_ast_node sct = PKL_PASS_NODE;
  pkl_ast_node sct_type = PKL_AST_TYPE (sct);

  /* No methods in struct literals.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_ulong (0, 64));
  /* The number of elements in struct literals corresponds to the
     number of fields, since there are no declarations in them.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                pvm_make_ulong (PKL_AST_STRUCT_NELEM (sct), 64));

  PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_TYPE);
  PKL_PASS_SUBPASS (sct_type); /* TYP */
  PKL_GEN_POP_CONTEXT;

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKSCT);
}
PKL_PHASE_END_HANDLER

/*
 *  STRUCT_FIELD
 *  | [STRUCT_FIELD_NAME]
 *  | STRUCT_FIELD_EXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_struct_field)
{
  pkl_ast_node struct_field = PKL_PASS_NODE;
  pkl_ast_node struct_field_name
    = PKL_AST_STRUCT_FIELD_NAME (struct_field);

  /* Element's offset.  PVM_NULL means use the "natural" offset.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);

  /* If the struct initializer doesn't include a name, generate a null
     value as expected by the mksct instruction.  */
  if (!struct_field_name)
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
}
PKL_PHASE_END_HANDLER

/*
 * | STRUCT
 * | IDENTIFIER
 * STRUCT_REF
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_struct_ref)
{
  if (PKL_PASS_PARENT == NULL
      && PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_LVALUE))
    {
      /* This is a -lvalue in an assignment.  The struct and the
         identifier are pushed to the stack for the ass_stmt PS
         handler.  Nothing else to do here.  */
    }
  else
    {
      pkl_ast_node struct_ref = PKL_PASS_NODE;
      pkl_ast_node struct_ref_type = PKL_AST_TYPE (struct_ref);
      pkl_ast_node struct_ref_struct
        = PKL_AST_STRUCT_REF_STRUCT (struct_ref);
      pkl_ast_node struct_ref_identifier
        = PKL_AST_STRUCT_REF_IDENTIFIER (struct_ref);
      pkl_ast_node struct_ref_struct_type = PKL_AST_TYPE (struct_ref_struct);
      pkl_ast_node elem;
      int is_field_p = 0;

      /* Determine whether the referred struct element is a field or a
         declaration.  */
      for (elem = PKL_AST_TYPE_S_ELEMS (struct_ref_struct_type);
           elem;
           elem = PKL_AST_CHAIN (elem))
        {
          if (PKL_AST_CODE (elem) == PKL_AST_STRUCT_TYPE_FIELD)
            {
              pkl_ast_node field_name
                = PKL_AST_STRUCT_TYPE_FIELD_NAME (elem);

              if (field_name != NULL
                  && strcmp (PKL_AST_IDENTIFIER_POINTER (field_name),
                             PKL_AST_IDENTIFIER_POINTER (struct_ref_identifier)) == 0)
                {
                  is_field_p = 1;
                  break;
                }
            }
        }

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SREF);
      /* If the parent is a funcall and the referred field is a struct
         method, then leave both the struct and the closure.  */
      if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_FUNCALL)
          && !PKL_PASS_PARENT && !is_field_p)
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);

      /* To cover cases where the referenced struct is not mapped, but
         the value stored in it is a mapped value, we issue a
         REMAP.  */
      switch (PKL_AST_TYPE_CODE (struct_ref_type))
        {
        case PKL_TYPE_ARRAY:
        case PKL_TYPE_STRUCT:
          /* XXX: this is redundant IO for many (most?) cases.  */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_REMAP);
          break;
        default:
          break;
        }
    }
}
PKL_PHASE_END_HANDLER

/*
 * TYPE_VOID
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_type_void)
{
  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKTYV);
}
PKL_PHASE_END_HANDLER

/*
 * TYPE_ANY
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_type_any)
{
  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPIFIER))
    ; /* Do nothing here.  */
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_CONSTRUCTOR))
    {
      /* This value is arbitrary... literally `any' value.. :D */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_int (0, 32));
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKTYANY);
}
PKL_PHASE_END_HANDLER

/*
 * TYPE_INTEGRAL
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_type_integral)
{
  pkl_asm pasm = PKL_GEN_ASM;
  pkl_ast_node integral_type = PKL_PASS_NODE;

  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_WRITER))
    {
      /* Stack: IOS BOFF VAL */
      switch (PKL_GEN_PAYLOAD->endian)
        {
        case PKL_AST_ENDIAN_DFL:
          pkl_asm_insn (pasm, PKL_INSN_POKED, integral_type);
          break;
        case PKL_AST_ENDIAN_LSB:
          pkl_asm_insn (pasm, PKL_INSN_POKE, integral_type,
                        IOS_NENC_2, IOS_ENDIAN_LSB);
          break;
        case PKL_AST_ENDIAN_MSB:
          pkl_asm_insn (pasm, PKL_INSN_POKE, integral_type,
                        IOS_NENC_2, IOS_ENDIAN_MSB);
          break;
        default:
          assert (0);
        }
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER))
    {
      /* Stack: STRICT IOS BOFF */
      switch (PKL_GEN_PAYLOAD->endian)
        {
        case PKL_AST_ENDIAN_DFL:
          pkl_asm_insn (pasm, PKL_INSN_PEEKD, integral_type);
          break;
        case PKL_AST_ENDIAN_LSB:
          pkl_asm_insn (pasm, PKL_INSN_PEEK, integral_type,
                        IOS_NENC_2, IOS_ENDIAN_LSB);
          break;
        case PKL_AST_ENDIAN_MSB:
          pkl_asm_insn (pasm, PKL_INSN_PEEK, integral_type,
                        IOS_NENC_2, IOS_ENDIAN_MSB);
          break;
        default:
          assert (0);
        }

      pkl_asm_insn (pasm, PKL_INSN_NIP); /* STRICT is not used.  */
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_CONSTRUCTOR))
    {
      /* Stack: NULL */
      int size = PKL_AST_TYPE_I_SIZE (integral_type);
      pvm_val zero;

      if (PKL_AST_TYPE_I_SIGNED_P (integral_type))
        {
          if (size <= 32)
            zero = pvm_make_int (0, size);
          else
            zero = pvm_make_long (0, size);
        }
      else
        {
          if (size <= 32)
            zero = pvm_make_uint (0, size);
          else
            zero = pvm_make_ulong (0, size);
        }

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The NULL */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, zero);
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_PRINTER))
    {
                                                  /* VAL DEPTH */
      RAS_MACRO_INTEGRAL_PRINTER (PKL_PASS_NODE); /* _ */
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_FORMATER))
    {
                                                  /* VAL DEPTH */
      RAS_MACRO_INTEGRAL_FORMATER (PKL_PASS_NODE); /* _ */
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPIFIER))
    {
      RAS_MACRO_INTEGRAL_TYPIFIER (PKL_PASS_NODE); /* SCT */
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    {
      pkl_asm_insn (pasm, PKL_INSN_PUSH,
                    pvm_make_ulong (PKL_AST_TYPE_I_SIZE (integral_type),
                                    64));

      pkl_asm_insn (pasm, PKL_INSN_PUSH,
                    pvm_make_int (PKL_AST_TYPE_I_SIGNED_P (integral_type),
                                  32));

      pkl_asm_insn (pasm, PKL_INSN_MKTYI);
    }
}
PKL_PHASE_END_HANDLER

/*
 * FUNC_TYPE_ARG
 * | FUNC_TYPE_ARG_TYPE
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_func_type_arg)
{
  /* Nothing to do.  */
}
PKL_PHASE_END_HANDLER

/* TYPE_FUNCTION
 * | FUNC_TYPE_ARG
 * | ...
 * | FUNC_TYPE_RTYPE
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_type_function)
{
  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_WRITER))
    {
      /* Writing a function value is a NOP.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The VAL */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The BOFF */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The IOS */

      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER | PKL_GEN_CTX_IN_CONSTRUCTOR))
    {
      /* We construct the same function value for mappings and
         constructions.  */

      pkl_ast_node function_type = PKL_PASS_NODE;
      pkl_ast_node function_rtype = PKL_AST_TYPE_F_RTYPE (function_type);
      pvm_program program;

      /* Compile the body for the function value.  */
      PKL_GEN_PUSH_ASM (pkl_asm_new (PKL_PASS_AST,
                                     PKL_GEN_PAYLOAD->compiler,
                                     0 /* prologue */));
      {
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PROLOG);
        int i;

        /* Discard arguments.  */
        for (i = 0; i < PKL_AST_TYPE_F_NARG (function_type); ++i)
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);

        /* If the function returns a value, construct it.  */
        if (PKL_AST_TYPE_CODE (function_rtype) == PKL_TYPE_VOID)
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
        else
          {
            /* Constructor argument.  */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);

            PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_CONSTRUCTOR);
            PKL_PASS_SUBPASS (function_rtype);
            PKL_GEN_POP_CONTEXT;
          }

        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RETURN);
      }

      program = pkl_asm_finish (PKL_GEN_ASM, 0 /* epilogue */);
      PKL_GEN_POP_ASM;

      pvm_program_make_executable (program);

      /* Discard constructor/mapper arguments.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
      if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);

      /* Push the constructed closure and install the current lexical
         environment.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_cls (program));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DUC);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      /* If in a mapper, get rid of the unused STRICT.  */
      if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);

      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_PRINTER))
    {
      /* Stack: VAL DEPTH  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* DEPTH is not used.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* VAL is not used.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_string ("#<closure>"));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PRINTS);
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_FORMATER))
    {
      /* Stack: VAL DEPTH  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* DEPTH is not used.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* VAL is not used.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_string ("#<closure>"));
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPIFIER))
    {
      RAS_MACRO_FUNCTION_TYPIFIER (PKL_PASS_NODE); /* SCT */
      PKL_PASS_BREAK;
    }
}
PKL_PHASE_END_HANDLER

/*
 * | FUNC_TYPE_ARG
 * | ...
 * | FUNC_TYPE_RTYPE
 * TYPE_FUNCTION
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_type_function)
{
  pkl_ast_node ftype = PKL_PASS_NODE;

  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    {
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_ulong (PKL_AST_TYPE_F_NARG (ftype), 64));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKTYC);
    }
}
PKL_PHASE_END_HANDLER

/*
 * TYPE_ARRAY
 * | ETYPE
 * | NELEM
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_type_array)
{
  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_WRITER))
    {
      /* Stack: IOS OFF ARR */

      /* Note that we don't use the offset, nor the IOS, since these
         are attributes of the mapped value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_WRITE);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The array.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The offset. */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The ios.  */

      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_ARRAY_BOUNDER))
    {
      pkl_ast_node array_type = PKL_PASS_NODE;
      pkl_ast_node etype = PKL_AST_TYPE_A_ETYPE (array_type);
      pvm_val bounder_closure;

      if (PKL_AST_TYPE_CODE (etype) == PKL_TYPE_ARRAY)
        PKL_PASS_SUBPASS (etype);

      if (PKL_AST_TYPE_A_BOUNDER (array_type) == PVM_NULL)
        {
          RAS_FUNCTION_ARRAY_BOUNDER (bounder_closure, array_type);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, bounder_closure); /* CLS */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);                   /* CLS */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);                  /* _ */
          PKL_AST_TYPE_A_BOUNDER (array_type) = bounder_closure;
        }

      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER))
    {
      pkl_ast_node array_type = PKL_PASS_NODE;
      pkl_ast_node array_type_bound = PKL_AST_TYPE_A_BOUND (array_type);

      pvm_val array_type_mapper = PKL_AST_TYPE_A_MAPPER (array_type);
      pvm_val array_type_writer = PKL_AST_TYPE_A_WRITER (array_type);

      int bounder_created = 0;

      /* Make a copy of the IOS.  We will need to install it in the
         resulting value later.  */
                                                 /* STRICT IOS OFF */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TOR);  /* STRICT IOS [OFF] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OVER);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OVER);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FROMR);/* STRICT IOS STRICT IOS OFF */

      /* Make sure the array type has a bounder.  Note that this
         should be done before compiling mapper, writer, constructor,
         etc functions, in order to make sure the bounder closures are
         compiled in the right environment.  */

      PKL_GEN_PAYLOAD->mapper_depth++;

      if (PKL_GEN_PAYLOAD->mapper_depth == 1
          && PKL_AST_TYPE_A_BOUNDER (array_type) == PVM_NULL)
        {
          /* Note that this only happens at the top-level of an
             anonymous array type, and compiles a bounder for it.
             Named array types have their bounder compiled in
             pkl_gen_pr_decl.  */
          bounder_created = 1;

          assert (!PKL_AST_TYPE_NAME (array_type));
          PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_ARRAY_BOUNDER);
          PKL_PASS_SUBPASS (array_type);
          PKL_GEN_POP_CONTEXT;
        }

      if (array_type_mapper == PVM_NULL)
        RAS_FUNCTION_ARRAY_MAPPER (array_type_mapper, array_type);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, array_type_mapper);
      if (!PKL_AST_TYPE_NAME (array_type))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);     /* ... STRICT IOS OFF CLS */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TOR);       /* ... STRICT IOS OFF [CLS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ATR);       /* ... STRICT IOS OFF CLS [CLS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NROT);      /* ... STRICT CLS IOS OFF [CLS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TOR);       /* ... STRICT CLS IOS [CLS OFF] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TOR);       /* ... STRICT CLS [CLS OFF IOS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);      /* ... CLS STRICT [CLS OFF IOS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FROMR);     /* ... CLS STRICT IOS [CLS OFF] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FROMR);     /* ... CLS STRICT IOS OFF [CLS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FROMR);     /* ... CLS STRICT IOS OFF CLS */

      /* Build the arguments and call the mapper to get a mapped array
         value.  Whether the mapping is bounded, and exactly how, is
         determined from the array type.  */
      if (array_type_bound
          && (PKL_AST_TYPE_CODE (PKL_AST_TYPE (array_type_bound))
              == PKL_TYPE_INTEGRAL))
        {
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        PKL_AST_TYPE_A_BOUNDER (array_type));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);
        }
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
                                                  /* ... CLS STRICT IOS OFF CLS EBOUND */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* ... CLS STRICT IOS OFF EBOUND CLS */

      if (array_type_bound
          && (PKL_AST_TYPE_CODE (PKL_AST_TYPE (array_type_bound))
              == PKL_TYPE_OFFSET))
        {
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        PKL_AST_TYPE_A_BOUNDER (array_type));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);

          /* Convert to bit-offset.  */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OGETM);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
        }
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
                                               /* ... CLS STRICT IOS OFF EBOUND CLS SBOUND */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);
                                            /* ... CLS STRICT IOS OFF EBOUND SBOUND CLS */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);  /* STRICT IOS CLS VAL */

      /* Install the mapper into the value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* STRICT IOS VAL CLS */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETM); /* STRICT IOS VAL */

      /* Install the IOS into the value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);    /* STRICT VAL IOS */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETIOS); /* STRICT VAL */

      /* Install the strictness attribute of the value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* VAL STRICT */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETS); /* VAL */

      if (array_type_writer == PVM_NULL)
        {
          PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_WRITER);
          RAS_FUNCTION_ARRAY_WRITER (array_type_writer, array_type);
          PKL_GEN_POP_CONTEXT;
        }
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, array_type_writer);
      if (!PKL_AST_TYPE_NAME (array_type))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC); /* VAL CLS */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETW);                /* VAL */
      /* Yay!, we are done ;) */

      if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER))
        PKL_GEN_PAYLOAD->mapper_depth--;

      if (bounder_created)
        pkl_ast_array_type_remove_bounders (array_type);

      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_PRINTER))
    {
      /* Stack: ARR DEPTH */

      pkl_ast_node array_type = PKL_PASS_NODE;
      pvm_val printer_closure = PKL_AST_TYPE_A_PRINTER (array_type);

      /* If the array type doesn't have a printer, compile one.  */
      if (printer_closure == PVM_NULL)
        RAS_FUNCTION_ARRAY_PRINTER (printer_closure, array_type);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, printer_closure);
      if (!PKL_AST_TYPE_NAME (array_type))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      /* Invoke the printer.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL); /* _ */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_FORMATER))
    {
      /* Stack: ARR DEPTH */

      pkl_ast_node array_type = PKL_PASS_NODE;
      pvm_val formater_closure = PKL_AST_TYPE_A_FORMATER (array_type);

      /* If the array type doesn't have a formater, compile one.  */
      if (formater_closure == PVM_NULL)
        RAS_FUNCTION_ARRAY_FORMATER (formater_closure, array_type);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, formater_closure);
      if (!PKL_AST_TYPE_NAME (array_type))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      /* Invoke the formater.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL); /* _ */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_INTEGRATOR))
    {
      /* Stack: ARR */

      pkl_ast_node array_type = PKL_PASS_NODE;
      pvm_val integrator_closure = PKL_AST_TYPE_A_INTEGRATOR (array_type);

      /* If the array type doesn't have a integrator, compile one.  */
      if (integrator_closure == PVM_NULL)
        RAS_FUNCTION_ARRAY_INTEGRATOR (integrator_closure, array_type);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, integrator_closure); /* CLS */
      if (!PKL_AST_TYPE_NAME (array_type))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);                    /* CLS */

      /* Invoke the integrator.  IVAL is either ULONG or an offset with
         ULONG as the base type.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL); /* IVAL(ULONG) WIDTH(UINT) */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_CONSTRUCTOR))
    {
      /* Stack: null */
      pkl_ast_node array_type = PKL_PASS_NODE;
      pkl_ast_node array_type_bound = PKL_AST_TYPE_A_BOUND (array_type);
      pvm_val array_type_constructor = PKL_AST_TYPE_A_CONSTRUCTOR (array_type);
      int bounder_created = 0;

      PKL_GEN_PAYLOAD->constructor_depth++;

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The null.  */

      /* Make sure the array type has a bounder.  */
      if (PKL_GEN_PAYLOAD->constructor_depth == 1
          && PKL_AST_TYPE_A_BOUNDER (array_type) == PVM_NULL)
        {
          /* Note that this only happens at the top-level of an
             anonymous array type, and compiles a bounder for it.
             Named array types have their bounder compiled in
             pkl_gen_pr_decl.  */
          bounder_created = 1;
          assert (!PKL_AST_TYPE_NAME (array_type));
          PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_ARRAY_BOUNDER);
          PKL_PASS_SUBPASS (array_type);
          PKL_GEN_POP_CONTEXT;
        }

      /* Push the EBOUND argument for the constructor.  */
      if (array_type_bound
          && (PKL_AST_TYPE_CODE (PKL_AST_TYPE (array_type_bound))
              == PKL_TYPE_INTEGRAL))
        {
          pkl_asm_note (PKL_GEN_ASM, "bounder");
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        PKL_AST_TYPE_A_BOUNDER (array_type));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);
        }
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);

      /* Push the SBOUND argument for the constructor, converted to a
         bit-offset.  */
      if (array_type_bound
          && (PKL_AST_TYPE_CODE (PKL_AST_TYPE (array_type_bound))
              == PKL_TYPE_OFFSET))
        {
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        PKL_AST_TYPE_A_BOUNDER (array_type));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);

          /* Convert to bit-offset.  */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OGETM);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
        }
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);

      /* Make sure the array type has a constructor, and call it.  */
      if (array_type_constructor == PVM_NULL)
        RAS_FUNCTION_ARRAY_CONSTRUCTOR (array_type_constructor, array_type);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, array_type_constructor);
      if (!PKL_AST_TYPE_NAME (array_type))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);          /* ARR */
      PKL_GEN_PAYLOAD->constructor_depth--;

      if (bounder_created)
        pkl_ast_array_type_remove_bounders (array_type);

      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPIFIER))
    {
      RAS_MACRO_ARRAY_TYPIFIER (PKL_PASS_NODE); /* SCT */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    {
      /* Generating a PVM array type.  */

      pkl_ast_node etype = PKL_AST_TYPE_A_ETYPE (PKL_PASS_NODE);

      PKL_PASS_SUBPASS (etype);

      /* XXX at the moment the run-time bound in array types is unused
         so we just push null here.  If it is ever used, this will be
         problematic because due to the additional lexical level
         introduced by array mappers subpassing on bound here will
         result on invalid variable accesses.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKTYA);

      PKL_PASS_BREAK;
    }

  /* In normal context, just subpass on the type of the elements,
     ignoring the number of elements.  */
  PKL_PASS_SUBPASS (PKL_AST_TYPE_A_ETYPE (PKL_PASS_NODE));
  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * TYPE_STRING
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_type_string)
{
  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_WRITER))
    {
      /* Stack: IOS BOFF STR */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_POKES);
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER))
    {
      /* Stack: STRICT IOS BOFF */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEEKS);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP); /* Get rid of STRICT */
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_CONSTRUCTOR))
    {
      /* Stack: NULL */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_string (""));
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_PRINTER))
    {
      /* Stack: VAL DEPTH */
      RAS_MACRO_STRING_PRINTER; /* _ */
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_FORMATER))
    {
      /* Stack: VAL DEPTH */
      RAS_MACRO_STRING_FORMATER; /* _ */
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPIFIER))
    {
      RAS_MACRO_STRING_TYPIFIER (PKL_PASS_NODE); /* SCT */
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKTYS);
}
PKL_PHASE_END_HANDLER

/*
 * TYPE_STRUCT
 * | (STRUCT_TYPE_FIELD|DECL)
 * | ...
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_type_struct)
{
  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_WRITER))
    {
      /* Stack: IOS OFF SCT */

      /* Note that we don't use the offset, nor the IOS, because these
         are attributes of the mapped value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_WRITE);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The struct.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The offset. */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The IOS.  */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER))
    {
      /* Stack: STRICT IOS OFF */
      pkl_ast_node type_struct = PKL_PASS_NODE;

      pvm_val type_struct_mapper = PKL_AST_TYPE_S_MAPPER (type_struct);
      pvm_val type_struct_writer = PKL_AST_TYPE_S_WRITER (type_struct);
      pvm_val type_struct_constructor = PKL_AST_TYPE_S_CONSTRUCTOR (type_struct);

      /* Make a copy of the IOS and STRICT.  We will need to install
         them in the resulting value later.  */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TOR);  /* STRICT IOS [OFF] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OVER);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OVER);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FROMR); /* STRICT IOS STRICT IOS OFF */

      /* Compile a mapper function and complete it using the current
         environment.  */
      if (type_struct_mapper == PVM_NULL)
        RAS_FUNCTION_STRUCT_MAPPER (type_struct_mapper, type_struct);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, type_struct_mapper);
      if (!PKL_AST_TYPE_NAME (type_struct))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC); /* ... STRICT IOS OFF CLS */

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TOR);   /* ... STRICT IOS OFF [CLS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_ATR);   /* ... STRICT IOS OFF CLS [CLS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NROT);  /* ... STRICT CLS IOS OFF [CLS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TOR);   /* ... STRICT CLS IOS [CLS OFF] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TOR);   /* ... STRICT CLS [CLS OFF IOS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* ... CLS STRICT [CLS OFF IOS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FROMR); /* ... CLS STRICT IOS [CLS OFF] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FROMR); /* ... CLS STRICT IOS OFF [CLS] */

      /* Build the arguments and call the mapper to get a struct
         value.  For structs, both EBOUND and SBOUND are always
         null.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
                          /* ... CLS STRICT IOS OFF EBOUND [CLS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
                          /* ... CLS STRICT IOS OFF EBOUND SBOUND [CLS] */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_FROMR);
                          /* ... CLS STRICT IOS OFF EBOUND SBOUND CLS */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);
                          /* STRICT IOS CLS VAL */

      /* Install the mapper into the value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* STRICT IOS VAL CLS */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETM); /* STRICT IOS VAL */

      /* Install the ios into the value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);    /* STRICT VAL IOS */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETIOS); /* STRICT VAL */

      /* Install the strictness property into the value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SWAP);  /* VAL STRICT */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETS); /* VAL */

      /* Compile a constructor function and complete it using the
         current environment.  */
      if (type_struct_constructor == PVM_NULL)
        {
          assert (!PKL_AST_TYPE_NAME (type_struct));
          PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_CONSTRUCTOR);
          RAS_FUNCTION_STRUCT_CONSTRUCTOR (type_struct_constructor, type_struct);
          PKL_GEN_POP_CONTEXT;
          /* We normally do not install closures in anonymous types,
             but this one is needed by ssetc.  */
          PKL_AST_TYPE_S_CONSTRUCTOR (type_struct) = type_struct_constructor;
        }
      if (!PKL_AST_TYPE_NAME (type_struct))
        {
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, type_struct_constructor);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
        }

      /* Compile a writer function and complete it using the current
         environment.  */
      if (type_struct_writer == PVM_NULL)
        {
          assert (!PKL_AST_TYPE_NAME (type_struct));
          PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_WRITER);
          if (PKL_AST_TYPE_S_UNION_P (type_struct))
            RAS_FUNCTION_UNION_WRITER (type_struct_writer, type_struct);
          else
            RAS_FUNCTION_STRUCT_WRITER (type_struct_writer, type_struct);
          PKL_GEN_POP_CONTEXT;
        }

        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, type_struct_writer); /* VAL CLS */
        if (!PKL_AST_TYPE_NAME (type_struct))
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);                  /* VAL CLS */

      /* Install the writer into the value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETW);                /* VAL */

      /* And we are done.  */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_CONSTRUCTOR))
    {
      /* Stack: SCT */
      pkl_ast_node type_struct = PKL_PASS_NODE;
      pvm_val type_struct_constructor = PKL_AST_TYPE_S_CONSTRUCTOR (type_struct);
      pvm_val type_struct_writer = PKL_AST_TYPE_S_WRITER (type_struct);

      /* If the given structure is null, then create an empty AST
         struct of the right type.  */
      {
        pvm_program_label label = pkl_asm_fresh_label (PKL_GEN_ASM);
        pkl_ast_node s = pkl_ast_make_struct (PKL_PASS_AST,
                                              0, NULL);

        PKL_AST_TYPE (s) = ASTREF (type_struct);

        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BNN, label);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* The null */
        PKL_GEN_PUSH_CONTEXT;
        PKL_PASS_SUBPASS (s);
        PKL_GEN_POP_CONTEXT;

        pkl_asm_label (PKL_GEN_ASM, label);
        s = ASTREF(s); pkl_ast_node_free (s);
      }

      if (type_struct_constructor == PVM_NULL)
        {
          RAS_FUNCTION_STRUCT_CONSTRUCTOR (type_struct_constructor, type_struct);
          /* We normally do not install closures in anonymous types,
             but this one is needed by ssetc.  */
          PKL_AST_TYPE_S_CONSTRUCTOR (type_struct) = type_struct_constructor;
        }
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, type_struct_constructor);
      if (!PKL_AST_TYPE_NAME (type_struct))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC); /* SCT CLS */

      /* Call the constructor to get a new struct.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);    /* NSCT */

      /* Compile a writer function and complete it using the current
         environment.  */
      if (type_struct_writer == PVM_NULL)
        {
          assert (!PKL_AST_TYPE_NAME (type_struct));
          PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_WRITER);
          {
            if (PKL_AST_TYPE_S_UNION_P (type_struct))
              RAS_FUNCTION_UNION_WRITER (type_struct_writer, type_struct);
            else
              RAS_FUNCTION_STRUCT_WRITER (type_struct_writer, type_struct);
          }
          PKL_GEN_POP_CONTEXT;
        }

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, type_struct_writer);
      if (!PKL_AST_TYPE_NAME (type_struct))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC); /* NCSCT CLS */

      /* Install the writer into the value.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MSETW); /* NCSCT */

      /* And we are done.  */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_COMPARATOR))
    {
      /* Stack: SCT1 SCT2 */

      pkl_ast_node type_struct = PKL_PASS_NODE;
      pvm_val comparator_closure
        = PKL_AST_TYPE_S_COMPARATOR (type_struct);

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OVER);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OVER);

      if (comparator_closure == PVM_NULL)
        {
          if (PKL_AST_TYPE_S_UNION_P (type_struct))
            RAS_FUNCTION_UNION_COMPARATOR (comparator_closure, type_struct);
          else
            RAS_FUNCTION_STRUCT_COMPARATOR (comparator_closure, type_struct);
        }
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, comparator_closure);
      if (!PKL_AST_TYPE_NAME (type_struct))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      /* Call the comparator.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL); /* SCT1 SCT2 INT */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_PRINTER))
    {
      /* Stack: SCT DEPTH */

      pkl_ast_node struct_type = PKL_PASS_NODE;
      pvm_val printer_closure = PKL_AST_TYPE_S_PRINTER (struct_type);

      /* If the struct type doesn't have a printer, compile one.  */
      if (printer_closure == PVM_NULL)
        {
          RAS_FUNCTION_STRUCT_PRINTER (printer_closure, struct_type);
          PKL_AST_TYPE_S_PRINTER (struct_type) = printer_closure;
        }

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, printer_closure);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      /* Invoke the printer.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL); /* _ */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_FORMATER))
    {
      /* Stack: SCT DEPTH */

      pkl_ast_node struct_type = PKL_PASS_NODE;
      pvm_val formater_closure = PKL_AST_TYPE_S_FORMATER (struct_type);

      /* If the struct type doesn't have a formater, compile one.  */
      if (formater_closure == PVM_NULL)
        {
          RAS_FUNCTION_STRUCT_FORMATER (formater_closure, struct_type);
          PKL_AST_TYPE_S_FORMATER (struct_type) = formater_closure;
        }

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, formater_closure);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      /* Invoke the formater.  */
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL); /* _ */
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_INTEGRATOR))
    {
      pkl_ast_node type_struct = PKL_PASS_NODE;
      pvm_val integrator_closure = PKL_AST_TYPE_S_INTEGRATOR (type_struct);

      if (integrator_closure == PVM_NULL)
        RAS_FUNCTION_STRUCT_INTEGRATOR (integrator_closure, type_struct);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, integrator_closure);
      if (!PKL_AST_TYPE_NAME (type_struct))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_DEINTEGRATOR))
    {
      pkl_ast_node type_struct = PKL_PASS_NODE;
      pvm_val deintegrator_closure = PKL_AST_TYPE_S_DEINTEGRATOR (type_struct);

      if (deintegrator_closure == PVM_NULL)
        RAS_FUNCTION_STRUCT_DEINTEGRATOR (deintegrator_closure, type_struct);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, deintegrator_closure);
      if (!PKL_AST_TYPE_NAME (type_struct))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPIFIER))
    {
      pkl_ast_node type_struct = PKL_PASS_NODE;
      pvm_val typifier_closure = PKL_AST_TYPE_S_TYPIFIER (type_struct);

      if (typifier_closure == PVM_NULL)
        RAS_FUNCTION_STRUCT_TYPIFIER (typifier_closure, type_struct);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, typifier_closure);
      if (!PKL_AST_TYPE_NAME (type_struct))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PEC);

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL);
      PKL_PASS_BREAK;
    }
  else if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    {
      /* Do nothing.  See PS hook.  */
    }
  else
    {
      /* In normal context, process the fields of the struct, but not
         the declarations contained within it.  */

      pkl_ast_node elem;

      for (elem = PKL_AST_TYPE_S_ELEMS (PKL_PASS_NODE);
           elem;
           elem = PKL_AST_CHAIN (elem))
        {
          if (PKL_AST_CODE (elem) == PKL_AST_STRUCT_TYPE_FIELD)
            PKL_PASS_SUBPASS (elem);
        }

      PKL_PASS_BREAK;
    }
}
PKL_PHASE_END_HANDLER

/*
 * | STRUCT_TYPE_FIELD
 * | ...
 * TYPE_STRUCT
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_type_struct)
{
  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    {
      /* We are generating a PVM struct type.  */

      pkl_ast_node struct_type = PKL_PASS_NODE;
      pkl_ast_node type_name = PKL_AST_TYPE_NAME (struct_type);

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_ulong (PKL_AST_TYPE_S_NFIELD (struct_type), 64));
      if (type_name)
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                      pvm_make_string (PKL_AST_IDENTIFIER_POINTER (type_name)));
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);

      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKTYSCT);
    }
}
PKL_PHASE_END_HANDLER

/*
 * STRUCT_TYPE_FIELD
 * | [STRUCT_TYPE_FIELD_NAME]
 * | STRUCT_TYPE_FIELD_TYPE
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_struct_type_field)
{
  assert (!PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_MAPPER));
  assert (!PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_WRITER));
  assert (!PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_CONSTRUCTOR));

  if (PKL_GEN_IN_CTX_P (PKL_GEN_CTX_IN_TYPE))
    {
      /* We are generating a PVM struct type.  */

      /* If the struct type element doesn't include a name, generate a
         null value as expected by the mktysct instruction.  */
      if (!PKL_AST_STRUCT_TYPE_FIELD_NAME (PKL_PASS_NODE))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, PVM_NULL);
      else
        PKL_PASS_SUBPASS (PKL_AST_STRUCT_TYPE_FIELD_NAME (PKL_PASS_NODE));
      PKL_PASS_SUBPASS (PKL_AST_STRUCT_TYPE_FIELD_TYPE (PKL_PASS_NODE));

      PKL_PASS_BREAK;
    }

  /* In normal context, subpass on the field type and ignore the
     name.  */
  PKL_PASS_SUBPASS (PKL_AST_STRUCT_TYPE_FIELD_TYPE (PKL_PASS_NODE));
  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * Expression handlers.
 *
 * | OPERAND1
 * | [OPERAND2]
 * EXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_add)
{
  pkl_asm pasm = PKL_GEN_ASM;
  pkl_ast_node node = PKL_PASS_NODE;
  pkl_ast_node type = PKL_AST_TYPE (node);

  switch (PKL_AST_TYPE_CODE (type))
    {
    case PKL_TYPE_INTEGRAL:
      pkl_asm_insn (pasm, PKL_INSN_ADD, type);
      pkl_asm_insn (pasm, PKL_INSN_NIP2);
      break;
    case PKL_TYPE_STRING:
      pkl_asm_insn (pasm, PKL_INSN_SCONC);
      pkl_asm_insn (pasm, PKL_INSN_NIP2);
      break;
    case PKL_TYPE_ARRAY:
      pkl_asm_insn (pasm, PKL_INSN_ACONC);
      pkl_asm_insn (pasm, PKL_INSN_NIP2);
      break;
    case PKL_TYPE_OFFSET:
      {
        pkl_ast_node base_type = PKL_AST_TYPE_O_BASE_TYPE (type);
        pkl_ast_node unit = PKL_AST_TYPE_O_UNIT (type);

        pkl_asm_insn (pasm, PKL_INSN_ADDO, base_type, unit);
        pkl_asm_insn (pasm, PKL_INSN_NIP2);
      }
      break;
    default:
      assert (0);
      break;
    }
}
PKL_PHASE_END_HANDLER

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_sub)
{
  pkl_asm pasm = PKL_GEN_ASM;
  pkl_ast_node node = PKL_PASS_NODE;
  pkl_ast_node type = PKL_AST_TYPE (node);

  switch (PKL_AST_TYPE_CODE (type))
    {
    case PKL_TYPE_INTEGRAL:
      pkl_asm_insn (pasm, PKL_INSN_SUB, type);
      pkl_asm_insn (pasm, PKL_INSN_NIP2);
      break;
    case PKL_TYPE_OFFSET:
      {
        pkl_ast_node base_type = PKL_AST_TYPE_O_BASE_TYPE (type);
        pkl_ast_node unit = PKL_AST_TYPE_O_UNIT (type);

        pkl_asm_insn (pasm, PKL_INSN_SUBO, base_type, unit);
        pkl_asm_insn (pasm, PKL_INSN_NIP2);
      }
      break;
    default:
      assert (0);
      break;
    }
}
PKL_PHASE_END_HANDLER

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_mul)
{
  pkl_asm pasm = PKL_GEN_ASM;
  pkl_ast_node node = PKL_PASS_NODE;
  pkl_ast_node type = PKL_AST_TYPE (node);

  switch (PKL_AST_TYPE_CODE (type))
    {
    case PKL_TYPE_INTEGRAL:
      pkl_asm_insn (pasm, PKL_INSN_MUL, type);
      pkl_asm_insn (pasm, PKL_INSN_NIP2);
      break;
    case PKL_TYPE_OFFSET:
      {
        pkl_ast_node op1 = PKL_AST_EXP_OPERAND (node, 0);
        pkl_ast_node op2 = PKL_AST_EXP_OPERAND (node, 1);
        pkl_ast_node op1_type = PKL_AST_TYPE (op1);
        pkl_ast_node op2_type = PKL_AST_TYPE (op2);
        pkl_ast_node base_type;

        if (PKL_AST_TYPE_CODE (op1_type) == PKL_TYPE_OFFSET)
          base_type = PKL_AST_TYPE_O_BASE_TYPE (op1_type);
        else
          {
            base_type = PKL_AST_TYPE_O_BASE_TYPE (op2_type);
            pkl_asm_insn (pasm, PKL_INSN_SWAP);
          }

        pkl_asm_insn (pasm, PKL_INSN_MULO, base_type);
        pkl_asm_insn (pasm, PKL_INSN_NIP2);
      }
      break;
    case PKL_TYPE_STRING:
      {
        pkl_ast_node op2 = PKL_AST_EXP_OPERAND (node, 1);
        pkl_ast_node op2_type = PKL_AST_TYPE (op2);

        if (PKL_AST_TYPE_CODE (op2_type) == PKL_TYPE_STRING)
          pkl_asm_insn (pasm, PKL_INSN_SWAP);

        pkl_asm_insn (pasm, PKL_INSN_MULS);
        pkl_asm_insn (pasm, PKL_INSN_NIP2);
      }
      break;
    default:
      assert (0);
      break;
    }
}
PKL_PHASE_END_HANDLER

/*
 * | OP1
 * | OP2
 * DIV
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_div)
{
  pkl_ast_node node = PKL_PASS_NODE;
  pkl_asm pasm = PKL_GEN_ASM;
  pkl_ast_node type = PKL_AST_TYPE (node);
  pkl_ast_node op1 = PKL_AST_EXP_OPERAND (node, 0);
  pkl_ast_node op1_type = PKL_AST_TYPE (op1);
  int div_insn, offset_div_insn;

  if (PKL_AST_EXP_CODE (node) == PKL_AST_OP_DIV)
    {
      div_insn = PKL_INSN_DIV;
      offset_div_insn = PKL_INSN_DIVO;
    }
  else
    {
      div_insn = PKL_INSN_CDIV;
      offset_div_insn = PKL_INSN_CDIVO;
    }

  switch (PKL_AST_TYPE_CODE (type))
    {
    case PKL_TYPE_INTEGRAL:
      {
        if (PKL_AST_TYPE_CODE (op1_type) == PKL_TYPE_OFFSET)
          {
            /* This is O / O -> I */
            pkl_asm_insn (pasm, offset_div_insn,
                          PKL_AST_TYPE_O_BASE_TYPE (op1_type));
            pkl_asm_insn (pasm, PKL_INSN_NIP2);
          }
        else
          {
            /* This is I / I -> I */
            pkl_asm_insn (pasm, div_insn, type);
            pkl_asm_insn (pasm, PKL_INSN_NIP2);
          }
        break;
      }
    case PKL_TYPE_OFFSET:
      {
        /* This is O / I -> O */
        pkl_ast_node op2 = PKL_AST_EXP_OPERAND (node, 1);
        pkl_ast_node op2_type = PKL_AST_TYPE (op2);

        pkl_asm_insn (pasm, PKL_INSN_SWAP); /* OP2 OP1 */
        pkl_asm_insn (pasm, PKL_INSN_OGETM); /* OP2 OP1 OMAG1 */
        pkl_asm_insn (pasm, PKL_INSN_SWAP);
        pkl_asm_insn (pasm, PKL_INSN_OGETU);
        pkl_asm_insn (pasm, PKL_INSN_NIP); /* OP2 OMAG1 UNIT */
        pkl_asm_insn (pasm, PKL_INSN_NROT); /* UNIT OP2 OMAG1 */
        pkl_asm_insn (pasm, PKL_INSN_SWAP); /* UNIT OMAG1 OP2 */
        pkl_asm_insn (pasm, div_insn, op2_type);
        pkl_asm_insn (pasm, PKL_INSN_NIP2); /* UNIT (OMAG1/OP2) */
        pkl_asm_insn (pasm, PKL_INSN_SWAP);
        pkl_asm_insn (pasm, PKL_INSN_MKO);
        break;
      }
    default:
      assert (0);
      break;
    }
}
PKL_PHASE_END_HANDLER

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_mod)
{
  pkl_ast_node node = PKL_PASS_NODE;

  pkl_asm pasm = PKL_GEN_ASM;
  pkl_ast_node type = PKL_AST_TYPE (node);

  switch (PKL_AST_TYPE_CODE (type))
    {
    case PKL_TYPE_INTEGRAL:
      pkl_asm_insn (pasm, PKL_INSN_MOD, type);
      pkl_asm_insn (pasm, PKL_INSN_NIP2);
      break;
    case PKL_TYPE_OFFSET:
      {
        pkl_ast_node base_type = PKL_AST_TYPE_O_BASE_TYPE (type);
        pkl_ast_node unit = PKL_AST_TYPE_O_UNIT (type);

        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MODO, base_type, unit);
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);
        break;
      }
    default:
      assert (0);
      break;
    }
}
PKL_PHASE_END_HANDLER

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_binexp)
{
  pkl_asm pasm = PKL_GEN_ASM;

  pkl_ast_node node = PKL_PASS_NODE;
  pkl_ast_node type = PKL_AST_TYPE (node);

  enum pkl_asm_insn insn;

  if (PKL_AST_EXP_CODE (node) == PKL_AST_OP_POS)
    /* POS in both integers and offsets is basically a nop.  */
    PKL_PASS_DONE;

  switch (PKL_AST_EXP_CODE (node))
    {
    case PKL_AST_OP_BAND: insn = PKL_INSN_BAND; break;
    case PKL_AST_OP_BNOT: insn = PKL_INSN_BNOT; break;
    case PKL_AST_OP_NEG: insn = PKL_INSN_NEG; break;
    case PKL_AST_OP_IOR: insn = PKL_INSN_BOR; break;
    case PKL_AST_OP_XOR: insn = PKL_INSN_BXOR; break;
    case PKL_AST_OP_SL: insn = PKL_INSN_SL; break;
    case PKL_AST_OP_SR: insn = PKL_INSN_SR; break;
    case PKL_AST_OP_POW: insn = PKL_INSN_POW; break;
    default:
      assert (0);
      break;
    }

  switch (PKL_AST_TYPE_CODE (type))
    {
    case PKL_TYPE_OFFSET:
      /* Fallthrough.  */
    case PKL_TYPE_INTEGRAL:
      pkl_asm_insn (pasm, insn, type);
      pkl_asm_insn (pasm, PKL_INSN_NIP);
      if (insn != PKL_INSN_NEG
          && insn != PKL_INSN_BNOT)
        pkl_asm_insn (pasm, PKL_INSN_NIP);
      break;
    default:
      assert (0);
      break;
    }
}
PKL_PHASE_END_HANDLER

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_op_and)
{
  pkl_ast_node op1 = PKL_AST_EXP_OPERAND (PKL_PASS_NODE, 0);
  pkl_ast_node op2 = PKL_AST_EXP_OPERAND (PKL_PASS_NODE, 1);

  pvm_program_label label = pkl_asm_fresh_label (PKL_GEN_ASM);

  PKL_PASS_SUBPASS (op1);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BZI, label);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
  PKL_PASS_SUBPASS (op2);
  pkl_asm_label (PKL_GEN_ASM, label);

  /* Normalize the result to 0 or 1.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NOT);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NOT);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_op_or)
{
  pkl_ast_node op1 = PKL_AST_EXP_OPERAND (PKL_PASS_NODE, 0);
  pkl_ast_node op2 = PKL_AST_EXP_OPERAND (PKL_PASS_NODE, 1);

  pvm_program_label label = pkl_asm_fresh_label (PKL_GEN_ASM);

  PKL_PASS_SUBPASS (op1);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BNZI, label);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
  PKL_PASS_SUBPASS (op2);
  pkl_asm_label (PKL_GEN_ASM, label);

  /* Normalize the result to 0 or 1.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NOT);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NOT);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_not)
{
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NOT);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
}
PKL_PHASE_END_HANDLER

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_op_impl)
{
  pkl_ast_node op1 = PKL_AST_EXP_OPERAND (PKL_PASS_NODE, 0);
  pkl_ast_node op2 = PKL_AST_EXP_OPERAND (PKL_PASS_NODE, 1);

  pvm_program_label label1 = pkl_asm_fresh_label (PKL_GEN_ASM);
  pvm_program_label label2 = pkl_asm_fresh_label (PKL_GEN_ASM);

  PKL_PASS_SUBPASS (op1);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BNZI, label1);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NOT);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BA, label2);

  pkl_asm_label (PKL_GEN_ASM, label1);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
  PKL_PASS_SUBPASS (op2);
  /* Normalize the result to 0 or 1.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NOT);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NOT);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);

  pkl_asm_label (PKL_GEN_ASM, label2);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * EXCOND
 * | EXP|STMT
 * | EXCEPTION_ID
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_op_excond)
{
  pkl_asm pasm = PKL_GEN_ASM;

  pkl_ast_node exp = PKL_PASS_NODE;
  pkl_ast_node op1 = PKL_AST_EXP_OPERAND (exp, 0);
  pkl_ast_node op2 = PKL_AST_EXP_OPERAND (exp, 1);

  pvm_program_label exception_handler = pkl_asm_fresh_label (PKL_GEN_ASM);
  pvm_program_label done = pkl_asm_fresh_label (PKL_GEN_ASM);

  /* Push the provisional result of the operation, which is
     `true'.  */
  pkl_asm_insn (pasm, PKL_INSN_PUSH, pvm_make_int (1, 32));

  /* Install a handler for the exception specified in the second
     operand.  */
  PKL_PASS_SUBPASS (op2);
  pkl_asm_insn (pasm, PKL_INSN_PUSHE, exception_handler);

  /* Execute the expression or statement in `op1'.  If it is an
     expression, discard the result value.  */
  PKL_PASS_SUBPASS (op1);
  if (PKL_AST_IS_EXP (op1))
    pkl_asm_insn (pasm, PKL_INSN_DROP);

  pkl_asm_insn (pasm, PKL_INSN_POPE);
  pkl_asm_insn (pasm, PKL_INSN_BA, done);

  /* The exception handler just drops the raised exception and the
     provisional result `true' and pushes `false' to reflect the
     exception was raised. */
  pkl_asm_label (pasm, exception_handler);
  pkl_asm_insn (pasm, PKL_INSN_DROP); /* The exception.  */
  pkl_asm_insn (pasm, PKL_INSN_DROP); /* The provisional result.  */
  pkl_asm_insn (pasm, PKL_INSN_PUSH, pvm_make_int (0, 32));

  pkl_asm_label (pasm, done);
  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

/*
 * TYPEOF
 * | EXPR|TYPE
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_op_typeof)
{
  pkl_ast_node exp = PKL_PASS_NODE;
  pkl_ast_node type = PKL_AST_TYPE (exp);
  pkl_ast_node op = PKL_AST_EXP_OPERAND (exp, 0);
  pkl_ast_node op_type = (PKL_AST_CODE (op) == PKL_AST_TYPE
                          ? op : PKL_AST_TYPE (op));
  pvm_val type_constructor = PKL_AST_TYPE_S_CONSTRUCTOR (type);
  int pk_type_code = 0;

  /* Create a struct Type on the stack calling its constructor.  We
     know that the constructor exists in a bootstrapped compiler,
     because `Type' is a named struct defined in the compiler
     run-time.  */
  assert (type_constructor != PVM_NULL);

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                pvm_make_ulong (0, 64)); /* OFF */

  /* Set the code in the argument to the constructor.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                pvm_make_ulong (0, 64)); /* EOFF */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_string ("code"));
                                         /* EOFF ENAME */

#define PK_TYPE_CODE(TYPE)                                              \
  ({                                                                    \
    pkl_ast_node initial = pkl_env_lookup_var (pkl_get_env (PKL_PASS_COMPILER), \
                                               (TYPE));                 \
    assert (PKL_AST_CODE (initial) == PKL_AST_INTEGER);                 \
    PKL_AST_INTEGER_VALUE (initial);                                    \
  })

  int pk_type_unknown = PK_TYPE_CODE ("PK_TYPE_UNKNOWN");

  switch (PKL_AST_TYPE_CODE (op_type))
    {
    case PKL_TYPE_INTEGRAL:
      pk_type_code = PK_TYPE_CODE ("PK_TYPE_INTEGRAL");
      break;
    case PKL_TYPE_OFFSET:
      pk_type_code = PK_TYPE_CODE ("PK_TYPE_OFFSET");
      break;
    case PKL_TYPE_STRING:
      pk_type_code = PK_TYPE_CODE ("PK_TYPE_STRING");
      break;
    case PKL_TYPE_ARRAY:
      pk_type_code = PK_TYPE_CODE ("PK_TYPE_ARRAY");
      break;
    case PKL_TYPE_STRUCT:
      pk_type_code = PK_TYPE_CODE ("PK_TYPE_STRUCT");
      break;
    case PKL_TYPE_ANY:
      pk_type_code = PK_TYPE_CODE ("PK_TYPE_ANY");
      break;
    case PKL_TYPE_FUNCTION:
      pk_type_code = PK_TYPE_CODE ("PK_TYPE_FUNCTION");
      break;
    default:
      pk_type_code = pk_type_unknown;
  }
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_int (pk_type_code, 32));
                                         /* EOFF ENAME EVAL */

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                pvm_make_ulong (0, 64)); /* OFF 0UL */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                pvm_make_ulong (1, 64)); /* OFF 0UL 1UL */

  /* Type of the `Type' struct */
  PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_TYPE);
  PKL_PASS_SUBPASS (type);
  PKL_GEN_POP_CONTEXT;                   /* OFF 0UL 0UL TYP */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKSCT);

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, type_constructor);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_CALL); /* Type {} */

  if (pk_type_code != pk_type_unknown)
    {
      /* Subpass in IN_TYPIFIER context to calculate the values of the
         Type struct */
      PKL_GEN_PUSH_SET_CONTEXT (PKL_GEN_CTX_IN_TYPIFIER);
      PKL_PASS_SUBPASS (op_type);
      PKL_GEN_POP_CONTEXT;
    }
  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_rela)
{
  pkl_asm pasm = PKL_GEN_ASM;
  pkl_ast_node exp = PKL_PASS_NODE;
  int exp_code = PKL_AST_EXP_CODE (exp);
  pkl_ast_node op1 = PKL_AST_EXP_OPERAND (exp, 0);
  pkl_ast_node op1_type = PKL_AST_TYPE (op1);

  enum pkl_asm_insn rela_insn;

  switch (exp_code)
    {
    case PKL_AST_OP_EQ: rela_insn = PKL_INSN_EQ; break;
    case PKL_AST_OP_NE: rela_insn = PKL_INSN_NE; break;
    case PKL_AST_OP_LT: rela_insn = PKL_INSN_LT; break;
    case PKL_AST_OP_GT: rela_insn = PKL_INSN_GT; break;
    case PKL_AST_OP_LE: rela_insn = PKL_INSN_LE; break;
    case PKL_AST_OP_GE: rela_insn = PKL_INSN_GE; break;
    default:
      assert (0);
      break;
    }

  switch (PKL_AST_TYPE_CODE (op1_type))
    {
    case PKL_TYPE_ARRAY:
      /* Fallthrough.  */
    case PKL_TYPE_STRUCT:
      /* Fallthrough.  */
    case PKL_TYPE_FUNCTION:
      assert (exp_code == PKL_AST_OP_EQ
              || exp_code == PKL_AST_OP_NE);
      /* Fallthrough.  */
    case PKL_TYPE_INTEGRAL:
    case PKL_TYPE_OFFSET:
    case PKL_TYPE_STRING:
      pkl_asm_insn (pasm, rela_insn, op1_type);
      pkl_asm_insn (pasm, PKL_INSN_NIP2);
      break;
    default:
      assert (0);
      break;
    }
}
PKL_PHASE_END_HANDLER

/*
 * | OPERAND1
 * EXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_attr)
{
  pkl_ast_node exp = PKL_PASS_NODE;
  pkl_ast_node operand = PKL_AST_EXP_OPERAND (exp, 0);
  pkl_ast_node operand_type = PKL_AST_TYPE (operand);
  enum pkl_ast_attr attr = PKL_AST_EXP_ATTR (exp);

  switch (attr)
    {
    case PKL_AST_ATTR_SIZE:
      /* If the value is an ANY, check the type is NOT a function
         value.  */
      if (PKL_AST_TYPE_CODE (operand_type) == PKL_TYPE_ANY)
        {
          pvm_program_label label = pkl_asm_fresh_label (PKL_GEN_ASM);

          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_TYISC);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BZI, label);

          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        pvm_make_exception (PVM_E_CONV, PVM_E_CONV_NAME,
                                            PVM_E_CONV_ESTATUS, NULL, NULL));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RAISE);

          pkl_asm_label (PKL_GEN_ASM, label);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
        }
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SIZ);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                    pvm_make_ulong (1, 64));
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKO);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
      /* XXX up-unit to the highest possible power of 2.  */
      break;
    case PKL_AST_ATTR_MAGNITUDE:
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OGETM);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
      break;
    case PKL_AST_ATTR_UNIT:
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_OGETU);
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
      break;
    case PKL_AST_ATTR_SIGNED:
      pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
      if (PKL_AST_TYPE_I_SIGNED_P (operand_type))
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_int (1, 32));
      else
        pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH, pvm_make_int (0, 32));
      break;
    case PKL_AST_ATTR_LENGTH:
      switch (PKL_AST_TYPE_CODE (operand_type))
        {
        case PKL_TYPE_STRING:
        case PKL_TYPE_ARRAY:
        case PKL_TYPE_STRUCT:
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_SEL);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
          break;
        default:
          /* This should not happen.  */
          assert (0);
        }
      break;
    case PKL_AST_ATTR_ALIGNMENT:
      /* XXX writeme */
      assert (0);
      break;
    case PKL_AST_ATTR_OFFSET:
      /* Fallthrough.  */
    case PKL_AST_ATTR_IOS:
      switch (PKL_AST_TYPE_CODE (operand_type))
        {
        case PKL_TYPE_ANY:
          /* Fallthrough.  */
        case PKL_TYPE_ARRAY:
          /* Fallthrough.  */
        case PKL_TYPE_STRUCT:
          {
            pvm_program_label label = pkl_asm_fresh_label (PKL_GEN_ASM);

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MM); /* VAL MAPPED_P */
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BNZI, label);

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                          pvm_make_exception (PVM_E_MAP, PVM_E_MAP_NAME,
                                              PVM_E_MAP_ESTATUS, NULL, NULL));
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RAISE);
            pkl_asm_label (PKL_GEN_ASM, label);

            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP); /* VAL */
            if (attr == PKL_AST_ATTR_OFFSET)
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MGETO);
            else
              pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MGETIOS);
            pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP); /* (BOFF|IOS) */

            if (attr == PKL_AST_ATTR_OFFSET)
              {
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                              pvm_make_ulong (1, 64));
                pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MKO);
              }
            break;
          }
        default:
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        pvm_make_exception (PVM_E_MAP, PVM_E_MAP_NAME,
                                            PVM_E_MAP_ESTATUS, NULL, NULL));
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_RAISE);
          break;
        }
      break;
    case PKL_AST_ATTR_MAPPED:
      switch (PKL_AST_TYPE_CODE (operand_type))
        {
        case PKL_TYPE_ANY:
        case PKL_TYPE_ARRAY:
        case PKL_TYPE_STRUCT:
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MM);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
          break;
        default:
          /* Other types are never mapped.  */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        pvm_make_int (0, 32));
          break;
        }
      break;
    case PKL_AST_ATTR_STRICT:
      switch (PKL_AST_TYPE_CODE (operand_type))
        {
        case PKL_TYPE_ANY:
        case PKL_TYPE_ARRAY:
        case PKL_TYPE_STRUCT:
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_MGETS);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);
          break;
        default:
          /* Other types are considered strict.  */
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_DROP);
          pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_PUSH,
                        pvm_make_int (1, 32));
          break;
        }
      break;
    default:
      PKL_ICE (PKL_AST_LOC (exp),
               "unhandled attribute expression code #%d in code generator",
               attr);
      PKL_PASS_ERROR;
      break;
    }
}
PKL_PHASE_END_HANDLER

/* | OPERAND1
 * | OPERAND2
 * EXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_bconc)
{
  pkl_ast_node exp = PKL_PASS_NODE;
  pkl_ast_node op1 = PKL_AST_EXP_OPERAND (exp, 0);
  pkl_ast_node op2 = PKL_AST_EXP_OPERAND (exp, 1);

  pkl_ast_node op1_type = PKL_AST_TYPE (op1);
  pkl_ast_node op2_type = PKL_AST_TYPE (op2);
  pkl_ast_node exp_type = PKL_AST_TYPE (exp);

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BCONC,
                op1_type, op2_type, exp_type);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);
}
PKL_PHASE_END_HANDLER

/*
 * | OPERAND1
 * EXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_unmap)
{
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_UNMAP);
}
PKL_PHASE_END_HANDLER

/*
 * | OPERAND1
 * | OPERAND2
 * EXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_ps_op_in)
{
  pkl_ast_node exp = PKL_PASS_NODE;
  //  pkl_ast_node elem = PKL_AST_EXP_OPERAND (exp, 0);
  pkl_ast_node container = PKL_AST_EXP_OPERAND (exp, 1);
  pkl_ast_node container_type = PKL_AST_TYPE (container);
  //  pkl_ast_node elem_type = PKL_AST_TYPE (elem);

  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_AIS, container_type);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP2);
}
PKL_PHASE_END_HANDLER

/* The handler below generates and ICE if a given node isn't handled
   by the code generator.  */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_noimpl)
{
  pkl_ast_node node = PKL_PASS_NODE;

  if (PKL_AST_CODE (node) == PKL_AST_EXP)
    {
      PKL_ICE (PKL_AST_LOC (node),
               "unhandled node #%" PRIu64 " with code %d opcode %d in code generator",
               PKL_AST_UID (node), PKL_AST_CODE (node), PKL_AST_EXP_CODE (node));
    }
  else if (PKL_AST_CODE (node) == PKL_AST_TYPE)
    {
      PKL_ICE (PKL_AST_LOC (node),
               "unhandled node #%" PRIu64 " with code %d typecode %d in code generator",
               PKL_AST_UID (node), PKL_AST_CODE (node), PKL_AST_TYPE_CODE (node));
    }
  else
    PKL_ICE (PKL_AST_LOC (node),
             "unhandled node #%" PRIu64 " with code %d in code generator",
             PKL_AST_UID (node), PKL_AST_CODE (node));

  PKL_PASS_ERROR;
}
PKL_PHASE_END_HANDLER

/*
 * COND_EXP
 * | COND
 * | THENEXP
 * | ELSEEXP
 */

PKL_PHASE_BEGIN_HANDLER (pkl_gen_pr_cond_exp)
{
  pkl_ast_node cond_exp = PKL_PASS_NODE;
  pkl_ast_node cond = PKL_AST_COND_EXP_COND (cond_exp);
  pkl_ast_node thenexp = PKL_AST_COND_EXP_THENEXP (cond_exp);
  pkl_ast_node elseexp = PKL_AST_COND_EXP_ELSEEXP (cond_exp);

  pvm_program_label label1 = pkl_asm_fresh_label (PKL_GEN_ASM);
  pvm_program_label label2 = pkl_asm_fresh_label (PKL_GEN_ASM);

  PKL_PASS_SUBPASS (cond);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BZI, label1);
  PKL_PASS_SUBPASS (thenexp);
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_BA, label2);
  pkl_asm_label (PKL_GEN_ASM, label1);
  PKL_PASS_SUBPASS (elseexp);
  pkl_asm_label (PKL_GEN_ASM, label2);

  /* Get rid fo the condition expression.  */
  pkl_asm_insn (PKL_GEN_ASM, PKL_INSN_NIP);

  PKL_PASS_BREAK;
}
PKL_PHASE_END_HANDLER

struct pkl_phase pkl_phase_gen =
  {
   PKL_PHASE_PS_HANDLER (PKL_AST_SRC, pkl_gen_ps_src),
   PKL_PHASE_PR_HANDLER (PKL_AST_DECL, pkl_gen_pr_decl),
   PKL_PHASE_PS_HANDLER (PKL_AST_DECL, pkl_gen_ps_decl),
   PKL_PHASE_PS_HANDLER (PKL_AST_VAR, pkl_gen_ps_var),
   PKL_PHASE_PR_HANDLER (PKL_AST_LAMBDA, pkl_gen_pr_lambda),
   PKL_PHASE_PS_HANDLER (PKL_AST_LAMBDA, pkl_gen_ps_lambda),
   PKL_PHASE_PR_HANDLER (PKL_AST_COND_EXP, pkl_gen_pr_cond_exp),
   PKL_PHASE_PR_HANDLER (PKL_AST_COMP_STMT, pkl_gen_pr_comp_stmt),
   PKL_PHASE_PS_HANDLER (PKL_AST_COMP_STMT, pkl_gen_ps_comp_stmt),
   PKL_PHASE_PS_HANDLER (PKL_AST_NULL_STMT, pkl_gen_ps_null_stmt),
   PKL_PHASE_PR_HANDLER (PKL_AST_ASS_STMT, pkl_gen_pr_ass_stmt),
   PKL_PHASE_PR_HANDLER (PKL_AST_INCRDECR, pkl_gen_pr_incrdecr),
   PKL_PHASE_PR_HANDLER (PKL_AST_IF_STMT, pkl_gen_pr_if_stmt),
   PKL_PHASE_PS_HANDLER (PKL_AST_BREAK_STMT, pkl_gen_ps_break_stmt),
   PKL_PHASE_PS_HANDLER (PKL_AST_CONTINUE_STMT, pkl_gen_ps_continue_stmt),
   PKL_PHASE_PR_HANDLER (PKL_AST_LOOP_STMT, pkl_gen_pr_loop_stmt),
   PKL_PHASE_PR_HANDLER (PKL_AST_RETURN_STMT, pkl_gen_pr_return_stmt),
   PKL_PHASE_PS_HANDLER (PKL_AST_RETURN_STMT, pkl_gen_ps_return_stmt),
   PKL_PHASE_PS_HANDLER (PKL_AST_EXP_STMT, pkl_gen_ps_exp_stmt),
   PKL_PHASE_PR_HANDLER (PKL_AST_FORMAT, pkl_gen_pr_format),
   PKL_PHASE_PR_HANDLER (PKL_AST_PRINT_STMT, pkl_gen_pr_print_stmt),
   PKL_PHASE_PS_HANDLER (PKL_AST_RAISE_STMT, pkl_gen_ps_raise_stmt),
   PKL_PHASE_PR_HANDLER (PKL_AST_TRY_CATCH_STMT, pkl_gen_pr_try_catch_stmt),
   PKL_PHASE_PR_HANDLER (PKL_AST_TRY_UNTIL_STMT, pkl_gen_pr_try_until_stmt),
   PKL_PHASE_PS_HANDLER (PKL_AST_FUNCALL_ARG, pkl_gen_ps_funcall_arg),
   PKL_PHASE_PR_HANDLER (PKL_AST_FUNCALL, pkl_gen_pr_funcall),
   PKL_PHASE_PR_HANDLER (PKL_AST_FUNC, pkl_gen_pr_func),
   PKL_PHASE_PS_HANDLER (PKL_AST_FUNC, pkl_gen_ps_func),
   PKL_PHASE_PR_HANDLER (PKL_AST_FUNC_ARG, pkl_gen_pr_func_arg),
   PKL_PHASE_PR_HANDLER (PKL_AST_FUNC_TYPE_ARG, pkl_gen_pr_func_type_arg),
   PKL_PHASE_PR_HANDLER (PKL_AST_PROGRAM, pkl_gen_pr_program),
   PKL_PHASE_PS_HANDLER (PKL_AST_PROGRAM, pkl_gen_ps_program),
   PKL_PHASE_PS_HANDLER (PKL_AST_INTEGER, pkl_gen_ps_integer),
   PKL_PHASE_PS_HANDLER (PKL_AST_IDENTIFIER, pkl_gen_ps_identifier),
   PKL_PHASE_PS_HANDLER (PKL_AST_STRING, pkl_gen_ps_string),
   PKL_PHASE_PS_HANDLER (PKL_AST_OFFSET, pkl_gen_ps_offset),
   PKL_PHASE_PR_HANDLER (PKL_AST_CAST, pkl_gen_pr_cast),
   PKL_PHASE_PS_HANDLER (PKL_AST_ISA, pkl_gen_ps_isa),
   PKL_PHASE_PR_HANDLER (PKL_AST_MAP, pkl_gen_pr_map),
   PKL_PHASE_PS_HANDLER (PKL_AST_CONS, pkl_gen_ps_cons),
   PKL_PHASE_PR_HANDLER (PKL_AST_ARRAY, pkl_gen_pr_array),
   PKL_PHASE_PS_HANDLER (PKL_AST_ARRAY, pkl_gen_ps_array),
   PKL_PHASE_PR_HANDLER (PKL_AST_TRIMMER, pkl_gen_pr_trimmer),
   PKL_PHASE_PR_HANDLER (PKL_AST_INDEXER, pkl_gen_pr_indexer),
   PKL_PHASE_PR_HANDLER (PKL_AST_ARRAY_INITIALIZER, pkl_gen_pr_array_initializer),
   PKL_PHASE_PS_HANDLER (PKL_AST_ARRAY_INITIALIZER, pkl_gen_ps_array_initializer),
   PKL_PHASE_PR_HANDLER (PKL_AST_STRUCT, pkl_gen_pr_struct),
   PKL_PHASE_PS_HANDLER (PKL_AST_STRUCT, pkl_gen_ps_struct),
   PKL_PHASE_PR_HANDLER (PKL_AST_STRUCT_FIELD, pkl_gen_pr_struct_field),
   PKL_PHASE_PS_HANDLER (PKL_AST_STRUCT_REF, pkl_gen_ps_struct_ref),
   PKL_PHASE_PR_HANDLER (PKL_AST_STRUCT_TYPE_FIELD, pkl_gen_pr_struct_type_field),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_ADD, pkl_gen_ps_op_add),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_SUB, pkl_gen_ps_op_sub),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_MUL, pkl_gen_ps_op_mul),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_MOD, pkl_gen_ps_op_mod),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_BAND, pkl_gen_ps_op_binexp),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_BNOT, pkl_gen_ps_op_binexp),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_NEG, pkl_gen_ps_op_binexp),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_POS, pkl_gen_ps_op_binexp),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_IOR, pkl_gen_ps_op_binexp),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_XOR, pkl_gen_ps_op_binexp),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_SL, pkl_gen_ps_op_binexp),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_SR, pkl_gen_ps_op_binexp),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_DIV, pkl_gen_ps_op_div),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_CEILDIV, pkl_gen_ps_op_div),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_POW, pkl_gen_ps_op_binexp),
   PKL_PHASE_PR_OP_HANDLER (PKL_AST_OP_AND, pkl_gen_pr_op_and),
   PKL_PHASE_PR_OP_HANDLER (PKL_AST_OP_OR, pkl_gen_pr_op_or),
   PKL_PHASE_PR_OP_HANDLER (PKL_AST_OP_IMPL, pkl_gen_pr_op_impl),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_NOT, pkl_gen_ps_op_not),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_EQ, pkl_gen_ps_op_rela),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_NE, pkl_gen_ps_op_rela),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_LT, pkl_gen_ps_op_rela),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_LE, pkl_gen_ps_op_rela),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_GT, pkl_gen_ps_op_rela),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_GE, pkl_gen_ps_op_rela),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_ATTR, pkl_gen_ps_op_attr),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_BCONC, pkl_gen_ps_op_bconc),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_UNMAP, pkl_gen_ps_op_unmap),
   PKL_PHASE_PS_OP_HANDLER (PKL_AST_OP_IN, pkl_gen_ps_op_in),
   PKL_PHASE_PR_OP_HANDLER (PKL_AST_OP_EXCOND, pkl_gen_pr_op_excond),
   PKL_PHASE_PR_OP_HANDLER (PKL_AST_OP_TYPEOF, pkl_gen_pr_op_typeof),
   PKL_PHASE_PS_TYPE_HANDLER (PKL_TYPE_VOID, pkl_gen_ps_type_void),
   PKL_PHASE_PS_TYPE_HANDLER (PKL_TYPE_ANY, pkl_gen_ps_type_any),
   PKL_PHASE_PS_TYPE_HANDLER (PKL_TYPE_INTEGRAL, pkl_gen_ps_type_integral),
   PKL_PHASE_PR_TYPE_HANDLER (PKL_TYPE_OFFSET, pkl_gen_pr_type_offset),
   PKL_PHASE_PR_TYPE_HANDLER (PKL_TYPE_FUNCTION, pkl_gen_pr_type_function),
   PKL_PHASE_PS_TYPE_HANDLER (PKL_TYPE_FUNCTION, pkl_gen_ps_type_function),
   PKL_PHASE_PR_TYPE_HANDLER (PKL_TYPE_ARRAY, pkl_gen_pr_type_array),
   PKL_PHASE_PS_TYPE_HANDLER (PKL_TYPE_STRING, pkl_gen_ps_type_string),
   PKL_PHASE_PR_TYPE_HANDLER (PKL_TYPE_STRUCT, pkl_gen_pr_type_struct),
   PKL_PHASE_PS_TYPE_HANDLER (PKL_TYPE_STRUCT, pkl_gen_ps_type_struct),
   PKL_PHASE_ELSE_HANDLER (pkl_gen_noimpl),
  };
