/* -*-c-*- -------------- mixgtk_cmd_dispatcher.c :
 * Implementation of the functions declared in mixgtk_cmd_dispatcher.h
 * ------------------------------------------------------------------
 *  Last change: Time-stamp: "01/03/15 16:01:43 jose"
 * ------------------------------------------------------------------
 * Copyright (C) 2001 Free Software Foundation, Inc.
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *  
 */

#define _GNU_SOURCE 1

#include <stdlib.h>
#include <stdio.h>
#include <mixlib/mix_vm_command.h>
#include "mixgtk_cmd_dispatcher.h"
#include "mixgtk_mixvm.h"
#include "mixgtk_mixal.h"


/* a mix vm command dispatcher */
struct mixgtk_dispatch_
{
  mix_vm_cmd_dispatcher_t *dispatcher;
  FILE *out;
  char *out_buffer;
  size_t out_buffer_size;
  GtkWidget *prompt;
  GtkWidget *log;
  GtkWidget *status;
  guint context;
  GCompletion *completions;
};

static struct mixgtk_dispatch_ dis_data_ = {NULL};

/* completions */
static void
init_completions_ (void)
{
  GList *cmds = NULL;
  gint k;
  
  dis_data_.completions = g_completion_new (NULL);
  for (k = 0; k < MIX_CMD_INVALID; ++k)
    cmds = g_list_append (cmds, (gpointer) mix_vm_command_to_string (k));
  g_completion_add_items (dis_data_.completions, cmds);
}

/* global hooks for the command dispatcher */
static void
global_pre_hook_ (mix_vm_cmd_dispatcher_t *dis,
		  mix_vm_command_t cmd, const gchar *arg, gpointer data)
{
  if (cmd < MIX_CMD_INVALID)
    {
      gtk_text_insert (GTK_TEXT (dis_data_.log), NULL, NULL, NULL,
		       "MIX> ", -1);
      gtk_text_insert (GTK_TEXT (dis_data_.log), NULL, NULL, NULL,
		       mix_vm_command_to_string (cmd), -1);
      gtk_text_insert (GTK_TEXT (dis_data_.log), NULL, NULL, NULL,
		       " ", -1);
      gtk_text_insert (GTK_TEXT (dis_data_.log), NULL, NULL, NULL,
		       arg, -1);
      gtk_text_insert (GTK_TEXT (dis_data_.log), NULL, NULL, NULL,
		       "\n", -1);
    }
  else
    {
      gtk_text_insert (GTK_TEXT (dis_data_.log), NULL, NULL, NULL,
		       _("\n*** Invalid command ***\n"), -1);
    }
}

static void
global_post_hook_ (mix_vm_cmd_dispatcher_t *dis,
		   mix_vm_command_t cmd, const gchar *arg, gpointer data)
{
  fflush (dis_data_.out);
  if (cmd < MIX_CMD_INVALID)
    {
      gtk_text_insert (GTK_TEXT (dis_data_.log), NULL, NULL, NULL,
		       dis_data_.out_buffer, dis_data_.out_buffer_size);
    }
  rewind (dis_data_.out);
  mixgtk_mixvm_update_vm_widgets ();
}

static void
load_post_hook_ (mix_vm_cmd_dispatcher_t *dis, const gchar *arg,
		 gpointer data)
{
  static glong id = -1;
 
  if (mix_vm_cmd_dispatcher_get_last_result (dis))
    {
      gchar *file = g_strdup (arg);
      mixgtk_mixal_load_file ();
      mixgtk_mixal_update ();

      if (id != -1)
	gtk_statusbar_remove (GTK_STATUSBAR (dis_data_.status),
			      dis_data_.context, (guint)id);
      id =
	gtk_statusbar_push (GTK_STATUSBAR (dis_data_.status), dis_data_.context,
			    file);
    }
}

static void
run_post_hook_ (mix_vm_cmd_dispatcher_t *dis, const gchar *arg,
		 gpointer data)
{
  mixgtk_mixal_update ();
}

static void
next_post_hook_ (mix_vm_cmd_dispatcher_t *dis, const gchar *arg,
		 gpointer data)
{
  mixgtk_mixal_update ();
}

static void
linebp_post_hook_ (mix_vm_cmd_dispatcher_t *dis, const gchar *arg,
		gpointer data)
{
  if (arg && strlen (arg)) mixgtk_mixal_update_bp_at_line (atoi (arg));
}

static void
addrbp_post_hook_ (mix_vm_cmd_dispatcher_t *dis, const gchar *arg,
		gpointer data)
{
  if (arg && strlen (arg)) mixgtk_mixal_update_bp_at_address (atoi (arg));
}

static void
allbp_post_hook_ (mix_vm_cmd_dispatcher_t *dis, const gchar *arg,
		  gpointer data)
{
  mixgtk_mixal_update_bp_all ();
}


static void
install_hooks_ (void)
{
  mix_vm_cmd_dispatcher_global_pre_hook (dis_data_.dispatcher,
					 global_pre_hook_, NULL);
  mix_vm_cmd_dispatcher_global_post_hook (dis_data_.dispatcher,
					  global_post_hook_, NULL);
  mix_vm_cmd_dispatcher_post_hook (dis_data_.dispatcher,
				   MIX_CMD_LOAD, load_post_hook_,
				   NULL);
  mix_vm_cmd_dispatcher_post_hook (dis_data_.dispatcher,
				   MIX_CMD_RUN, run_post_hook_,
				   NULL);
  mix_vm_cmd_dispatcher_post_hook (dis_data_.dispatcher,
				   MIX_CMD_NEXT, next_post_hook_,
				   NULL);
  mix_vm_cmd_dispatcher_post_hook (dis_data_.dispatcher,
				   MIX_CMD_SBP, linebp_post_hook_,
				   NULL);
  mix_vm_cmd_dispatcher_post_hook (dis_data_.dispatcher,
				   MIX_CMD_CBP, linebp_post_hook_,
				   NULL);
  mix_vm_cmd_dispatcher_post_hook (dis_data_.dispatcher,
				   MIX_CMD_SBPA, addrbp_post_hook_,
				   NULL);
  mix_vm_cmd_dispatcher_post_hook (dis_data_.dispatcher,
				   MIX_CMD_CBPA, addrbp_post_hook_,
				   NULL);
  mix_vm_cmd_dispatcher_post_hook (dis_data_.dispatcher,
				   MIX_CMD_CABP, allbp_post_hook_,
				   NULL);
}


/* initialise the command dispatcher */
gboolean
mixgtk_cmd_dispatcher_init (void)
{
  if (!dis_data_.prompt)
    {
      dis_data_.prompt = mixgtk_widget_factory_get (MIXGTK_WIDGET_PROMPT);
      g_return_val_if_fail (dis_data_.prompt != NULL, FALSE);
      dis_data_.log = mixgtk_widget_factory_get (MIXGTK_WIDGET_LOG);
      g_return_val_if_fail (dis_data_.log != NULL, FALSE);
    }
    
  if (!dis_data_.dispatcher)
    {
      FILE *out = open_memstream (&(dis_data_.out_buffer),
				  &(dis_data_.out_buffer_size));
      g_return_val_if_fail (out != NULL, FALSE);
      dis_data_.out = out;
      dis_data_.dispatcher = mix_vm_cmd_dispatcher_new (out, out);
      mix_vm_cmd_dispatcher_print_time (dis_data_.dispatcher, FALSE);
      install_hooks_ ();
    }
  
  if (!dis_data_.status)
    {
      dis_data_.status = mixgtk_widget_factory_get (MIXGTK_WIDGET_STATUSBAR);
      g_return_val_if_fail (dis_data_.status != NULL, FALSE);
      dis_data_.context = gtk_statusbar_get_context_id (GTK_STATUSBAR
							(dis_data_.status),
							"cmd_dis_context");
    }
  
  if (!dis_data_.completions) init_completions_ ();
  
  return TRUE;
}

/* dispatch an externally provided command */
void
mixgtk_cmd_dispatcher_dispatch (const gchar *command)
{
  GtkWidget *entry = dis_data_.prompt;
  g_return_if_fail (command != NULL);
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), command);
  on_mixvm_cmd_entry_activate (entry, NULL);
}

/* get times */
void
mixgtk_cmd_dispatcher_get_times (gint *uptime, gint *progtime, gint *laptime)
{
  if (uptime != NULL)
    *uptime = mix_vm_cmd_dispatcher_get_uptime (dis_data_.dispatcher);
  if (progtime != NULL)
    *progtime = mix_vm_cmd_dispatcher_get_progtime (dis_data_.dispatcher);
  if (laptime != NULL)
    *laptime = mix_vm_cmd_dispatcher_get_laptime (dis_data_.dispatcher);
}

/* get the underlying vm */
extern mix_vm_t *
mixgtk_cmd_dispatcher_get_vm (void)
{
  return (mix_vm_t *) mix_vm_cmd_dispatcher_get_vm (dis_data_.dispatcher);
}

/* process new command */
void
on_mixvm_cmd_entry_activate (GtkWidget *w, gpointer e)
{
  gchar *text;
  GList *cmds = NULL;
  gchar *prefix = NULL;
  
  text = gtk_entry_get_text (GTK_ENTRY (w));

  if (mix_vm_command_from_string (g_strchomp (text)) != MIX_CMD_INVALID)
    {
      mix_vm_cmd_dispatcher_dispatch_text (dis_data_.dispatcher, text);
      gtk_entry_set_text (GTK_ENTRY (w), "");
      return;
    }
  
  cmds = g_completion_complete (dis_data_.completions, text, &prefix);

  if (prefix != NULL)
    {
      gtk_entry_set_text (GTK_ENTRY (w), prefix);
      g_free (prefix);
      if (g_list_length (cmds) > 1)
	{
	  gint k = 0;
	  gtk_text_insert (GTK_TEXT (dis_data_.log),
			   NULL, NULL, NULL, "Completions:\n", -1);
	  while (cmds)
	    {
	      ++k;
	      gtk_text_insert (GTK_TEXT (dis_data_.log),
			       NULL, NULL, NULL,
			       (gchar *)(cmds->data), -1);
	      gtk_text_insert (GTK_TEXT (dis_data_.log),
			       NULL, NULL, NULL, "     ", -1);
	      if (k%5 == 0)
		gtk_text_insert (GTK_TEXT (dis_data_.log),
				 NULL, NULL, NULL, "\n", -1);
	      
	      cmds = cmds->next;
	    }
	  if (k%5 != 0)
	    gtk_text_insert (GTK_TEXT (dis_data_.log),
			     NULL, NULL, NULL, "\n", -1);
	}
      else
	gtk_entry_append_text (GTK_ENTRY (w), " ");
    }
  else
    {
      mix_vm_cmd_dispatcher_dispatch_text (dis_data_.dispatcher, text);
      gtk_entry_set_text (GTK_ENTRY (w), "");
    }
}
