/*-*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * $Id: Library.java,v 1.19 2002/02/12 19:44:11 metlov Exp $
 *
 * This file is part of the Java Expressions Library (JEL).
 *   For more information about JEL visit :
 *    http://galaxy.fzu.cz/JEL/
 *
 * (c) 1998 -- 2000 by Konstantin Metlov(metlov@fzu.cz);
 *
 * JEL is Distributed under the terms of GNU General Public License.
 *    This code comes with ABSOLUTELY NO WARRANTY.
 *  For license details see COPYING file in this directory.
 */

package gnu.jel;

import gnu.jel.debug.Debug;
import java.lang.reflect.Member;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Constructor;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;

//+-+-+-+TESTS
import gnu.jel.debug.Tester;
//+-+-+-+TESTS

/**
 * A namespace for JEL expressions.
 * <P> There are two types of members in the library, those which are stateless
 * (i.e. their value depends only on their arguments, if there are any) and
 * stateful (also called here dynamic). The result of evaluation of 
 * stateful members may depend on other factors besides their arguments.
 *
 * <P>Examples of possible stateless members are : <TT>Math.sin(double)</TT>,
 * <TT>Math.PI</TT>.
 * <P>Examples of possible stateful members are : <TT>Math.random()</TT>,
 * <TT>System.currentTimeMillis()</TT>.
 *
 * <P>Stateless members of this library are always static members of the 
 * classes, which define them. The inverse is generally not true. However,
 * this library goes as far as assuming that static members are stateless, 
 * if this assumption does not hold for some of Your members it is possible to 
 * mark them as stateful using the <TT>markStateDependent()</TT> method of
 * this class.
 *
 * <P>The most crucial difference between the two kind of members of this
 * library is that evaluation of stateless methods is attempted by JEL at
 * a compile time during the constants folding phase.
 */
public class Library {

  private Hashtable names; // Hashtable(names, Hashtable(signatures,members))
  
  private Hashtable dynIDs; // Hashtable(members, integers)

  private Hashtable stateless; // Hashtable(members, booleans)

  private Hashtable dotClasses; // Hashtable(classes, Hashtable(names, Hashtable(signatures, members)))
  private boolean noDotSecurity=false;

  public DVResolver resolver;   // directly accessed from EC.jj

  public Hashtable cnmap;

  /**
   * Creates a library for JEL.
   * <P> See the three argument constructor for more info.
   * @param staticLib is the array of classes, whose public static 
   *  methods are exported.
   * @param dynamicLib is the array of classes, whose public virutal
   *  methods are exported.
   * @deprecated Please us 5 argument constructor with unused arguments
   *  set to <TT>null</TT>. This constructor is scheduled for removal in
   *  JEL 1.0.
   */
  public Library(Class[] staticLib,Class[] dynamicLib) {
    this(staticLib,dynamicLib,null);
  };

  /**
   * Creates a library for JEL.
   * <P> See the three argument constructor for more info.
   * @param staticLib is the array of classes, whose public static 
   *  methods are exported.
   * @param dynamicLib is the array of classes, whose public virutal
   *  methods are exported.
   * @param dotClasses Controls access to the "dot" operator on classes.
   * @deprecated Please us 5 argument constructor with unused arguments
   *  set to <TT>null</TT>. This constructor is scheduled for removal in
   *  JEL 1.0.
   */
  public Library(Class[] staticLib,Class[] dynamicLib,Class[] dotClasses) {
    this(staticLib,dynamicLib,dotClasses,null);
  };

  /**
   * Creates a library for JEL.
   * <P> See the three argument constructor for more info.
   * @param staticLib is the array of classes, whose public static 
   *  methods are exported.
   * @param dynamicLib is the array of classes, whose public virutal
   *  methods are exported.
   * @param dotClasses Controls access to the "dot" operator on classes.
   * @param resolver Controls resolution of dynamic variables.
   * @deprecated Please us 5 argument constructor with unused arguments
   *  set to <TT>null</TT>. This constructor is scheduled for removal in
   *  JEL 1.0.
   */
  public Library(Class[] staticLib,Class[] dynamicLib,Class[] dotClasses,
                 DVResolver resolver) {
    this(staticLib,dynamicLib,dotClasses,resolver,null);
  };

  /**
   * Creates a library for JEL.
   * <P> The following should be kept in mind when constructing a library:
   * <OL>
   * <LI>This constructor may throw IllegalArgumentException if it does not
   * like something in Your library. The requirements to the method names
   * are somewhat more strict, than in Java because members of several
   * classes can be merged in root namespace.
   * <LI>When calling the 
   * <TT>CompiledExpression.evaluate(Object[] dynalib)</TT> of the
   * expression, using dynamic library methods it is needed to pass as
   *  <TT>dynalib</TT> parameter the array of objects, of the classes 
   * _exactly_ in the same order as they are appearing in the
   * <TT>dynamicLib</TT> parameter of this constructor. If You do not
   * allow to call dynamic methods (there is no sense, then, to use a compiler)
   * it is possible to pass <TT>null</TT> istead of <TT>dynalib</TT>.
   * <LI> Generally speaking, this class should not let You to create wrong
   * libraries. It's methods will throw exceptions, return <TT>false</TT>'s , 
   * ignore Your actions,... ;)
   * </OL>
   * If methods in the library classes conflict with each other, the last
   * conflicting method will be skipped. You will not get any messages unless
   * debugging is ON (see <TT>gnu.jel.debug.Debug.enabled</TT>). This is
   * done to avoid unnecessary error messages in the production code of the
   * compiler.
   * <P>The array (dotClasses), which is the third argument of 
   * this constructor determines how (and whether) to compile the 
   * dot operators encountered in expressions. These operators are
   * of the form <object>.(<method>|<field>), 
   * which means to call method (access field)
   * of an <object>. There can be three types of the behaviour: 
   * <P>1) dot operators are prohibited (<TT>dotClasses==null</TT>),
   * this is behaviour of older version of JEL.
   * <P>2) dot operators are allowed on all classes
   * (<TT>dotClasses==new Class[0], an empty array).
   * Depending on the types of objects returned by the static/dynamic library
   * classes this may pose a security risk.
   * <P>3) dot operators are allowed only on some classes. This is achieved 
   * by listing these classes in the dotClasses array.
   * @param staticLib is the array of classes, whose public static 
   *  methods are exported.
   * @param dynamicLib is the array of classes, whose public virutal
   *  methods are exported.
   * @param dotClasses is the array of classes on which the dot ('.')
   * operation is allowed.
   * @param resolver is the object used to resolve the names.
   * @param cnmap Maps class names into classes for non-primitive type casts.
   */
  public Library(Class[] staticLib,Class[] dynamicLib,Class[] dotClasses,
                 DVResolver resolver,Hashtable cnmap) {
    this.cnmap=cnmap;
    this.resolver=resolver;
    if (dotClasses==null) {
      this.dotClasses=null;
    } else {
      noDotSecurity=(dotClasses.length==0);
      this.dotClasses=new Hashtable();
      // hash the names
      Class[] temp=new Class[1];
      for(int i=0;i<dotClasses.length;i++) 
        rehash(dotClasses[i]);
    };

    names = new Hashtable();
    dynIDs = new Hashtable();
    stateless=new Hashtable();

    if (staticLib!=null)
      rehash(staticLib,names,null,stateless);
    if (dynamicLib!=null)
      rehash(dynamicLib,names,dynIDs,null);
  };

  private void rehash(Class cls) {
    Hashtable tempNames=new Hashtable();
    Class[] temp=new Class[1];
    temp[0]=cls;
    //    rehash(temp,tempNames,null,new Hashtable());
    rehash(temp,tempNames,new Hashtable(),null);
    dotClasses.put(cls,tempNames);
  };

  private static void rehash(Class[] arr, Hashtable hashedNames, 
                             Hashtable dynIDs, Hashtable stateless) {
    for (int i=0; i<arr.length; i++) {
      Integer dynID=new Integer(i);

      Method[] marr=arr[i].getMethods();
      Field[] farr=arr[i].getFields();

      int totalMethods=marr.length;
      int totalMembers=totalMethods+farr.length;
      for(int j=0;j<totalMembers;j++) {
        Member m = j<totalMethods?(Member)marr[j]:(Member)farr[j-totalMethods];
        if ((m.getModifiers() & 0x0008)>0) { // static
          if ((stateless!=null) && rehash(hashedNames,m)) 
            stateless.put(m,Boolean.TRUE);
        } else { // not static
          if ((dynIDs!=null) && rehash(hashedNames,m))
            dynIDs.put(m,dynID);
        };
      };
    };
  };

  private static boolean rehash(Hashtable hashedNames, Member m) {
    String name=m.getName();
    String signature=getSignature(m);

    // for the purpose of matching fields behave like methods with no
    // arguments
    if (isField(m)) signature="()"+signature;
    
    Hashtable signatures=(Hashtable)hashedNames.get(name);
    if (signatures==null) { 
      // No method with this name was added
      Hashtable signatures_new=new Hashtable();
      signatures_new.put(signature,m);
      hashedNames.put(name,signatures_new);
      return true;
    };
    // Name exists in the library, check for possible signature conflict.
    Object conflicting_method=signatures.get(signature);
    if (conflicting_method==null) { // No conflict
      signatures.put(signature,m);
      return true;
    };
//      if (Debug.enabled) {
//        Debug.println("Conflict was detected during the library "+
//                      "initialization."+
//                      " Conflicting "+"\""+name+signature+
//                      "\", conflicting :"+ conflicting_method+" and "+m+" .");
//      };
    // If no debug then the method is ignored.
    return false;
  };

  /**
   * This method marks a static member as having the internal state.
   * <P> If You include <TT>java.lang.Math</TT> into the library it is
   * necessary to mark <TT>java.lang.random()</TT> as having the state. This
   * can be done by calling <TT>markStateDependent("random",null)</TT>
   * <P> Please specify parameters as close as possible, otherwise You can
   * accidentally mark another function.
   * @param name is the function name.
   * @param params are the possible invocation parameters of the function.
   * @exception CompilationException if the method can't be resolved   
   */
  public void markStateDependent(String name, Class[] params) 
       throws CompilationException {
    Object m=getMember(null,name,params);
    Object removed=stateless.remove(m);
    if (Debug.enabled)
      Debug.assert(removed!=null,"State dependent methos \""+m+
		   "\"is made state dependend again.");
  };
  
  /**
   * Used to check if the given method is stateless.
   * @param m is method or field to check.
   * @return true if the method is stateless and can be invoked at
   *    compile time.
   */
  public boolean isStateless(Member o) {
    return stateless.containsKey(o);
  };

  /**
   * Searches the namespace defined by this library object for method or field.
   * <P> The method with the same name, and closest (convertible) parameter
   * types is returned. If there are several methods the most specific one
   * is used. 
   * <P>Ambiguities are detected. Example of detectable ambiguity:<BR>
   * You want to call : <TT>someName(int, int)<TT> <BR>
   * There are two applicable methods : <TT>someName(int, double)<TT>,
   * <TT>someName(double, int)<TT> , requirements to parameter types of each
   *  of those can be satisfied by _widening_ conversions.<BR>
   * Those methods are same specific, there is no most specific method in 
   * terms of Java Language Specification (15.11.2.2). Thus there is ambiguity
   * and null will be returned.
   * <P> Java compiler normally would not allow to define such ambiguous 
   * methods in the same class. However, as this library is assembled from
   * several Java classes such ambiguities can happen, and should be
   * detected.
   * @param container the class to search the method within, if <TT>null</TT>
   *                  the root namespace is searched.
   * @param name is the name of the method to find.
   * @param params are the types of formal parameters in the method invocation.
   * @return the method/field object of the resolved method/field.
   * @exception CompilationException if the method can't be resolved
   */
  public Member getMember(Class container,String name,Class[] params) 
    throws CompilationException {
    Hashtable hashedMembers=names;

    // dot operator security    
    if (container!=null) {
      if (dotClasses==null) // dot operator is prohibited
        throw new CompilationException(11,null);
      else if (! (noDotSecurity || dotClasses.containsKey(container))) {
        // dot is not allowed in this particular class
        Object[] paramsExc={container};
        throw new CompilationException(12,paramsExc);
      };
      if ((hashedMembers=(Hashtable)dotClasses.get(container))==null) {
        rehash(container);
        hashedMembers=(Hashtable)dotClasses.get(container);
      };
    };
    

    Hashtable signatures=(Hashtable) hashedMembers.get(name);

    //      System.out.print("RESOLVING: ");
    //      System.out.println(describe(name,params));

    if (signatures==null) { // name is not found
      Object[] paramsExc={name,container};
      throw new CompilationException(container==null?5:6,paramsExc);
    };
    
    // Choose applicable methods
    Vector applicable_methods=new Vector();
    for(Enumeration e=signatures.elements();e.hasMoreElements();) {
      Member cm = (Member)e.nextElement();
      Class[] cp=getParameterTypes(cm);
      
      boolean applicable=false;
      if (params!=null) { // if no parameters requested
        if (cp.length==params.length) {  // If number of arguments matches
          applicable=true;
          for(int i=0;((i<cp.length) && applicable);i++) {
            applicable=OP.isWidening(params[i],cp[i]);
          };
        };
      } else {
        applicable=(cp.length==0);
      };
      
      if (applicable) applicable_methods.addElement(cm);
    };
    
    if (applicable_methods.size()==0) {
      Object[] paramsExc={name,describe(name,params),container};
      throw new CompilationException(container==null?7:8,paramsExc);
    };
    
    if (applicable_methods.size()==1) 
      return (Member)applicable_methods.firstElement();
    
    // Search for the most specific method
    Enumeration e=applicable_methods.elements();
    Member most_specific=(Member)e.nextElement();
    Class[] most_specific_params=getParameterTypes(most_specific);

    //    System.out.println("--- APPLICABLE METHODS ---");
    //    System.out.println(most_specific.getName()+
    //                       ClassFile.getSignature(most_specific));
    while (e.hasMoreElements()) {
      Member cm= (Member)e.nextElement();
      Class[] cp=getParameterTypes(cm);
      boolean moreSpecific=true;
      boolean lessSpecific=true;

      //      System.out.println(cm.getName()+ClassFile.getSignature(cm));
      
      for(int i=0; i<cp.length; i++) {
        moreSpecific = moreSpecific && 
          OP.isWidening(cp[i],most_specific_params[i]);
        lessSpecific = lessSpecific &&
          OP.isWidening(most_specific_params[i],cp[i]);
      };
      
      if (moreSpecific && (!lessSpecific)) {
        most_specific=cm;
        most_specific_params=cp;
      };
      
      if (! (moreSpecific ^ lessSpecific)) {
        Object[] paramsExc={describe(name,most_specific_params),
                           describe(name,cp),
                           describe(name,params),
                           container};
        throw new CompilationException(container==null?9:10,paramsExc);        
      };
    };
    //    System.out.println("--- END APPLICABLE METHODS ---");
    return most_specific;
  };

  protected static String describe(String name,Class[] params) {
    StringBuffer invp=new StringBuffer();
    invp.append(name); 
    invp.append('(');
    
    if (params!=null)
      for(int k=0;k<params.length;k++) {
        if (k!=0) invp.append(',');
        invp.append(params[k].toString());
      };
    invp.append(')');
    return invp.toString();
  };
  
  /**
   * Returns ID(position in the object array) of the dynamic Method.
   * <P> ID's are used to locate the pointers to the objects, implementing
   * dynamic methods, in the array, argument of evaluate(Object[]) function.
   * @param m method to get an ID of.
   * @return the ID of the method or -1 if the method is static.
   * @exception NullPointerException if method is not a dynamic method of
   *            this library.
   */
  public int getDynamicMethodClassID(Member m) {
    Integer id=(Integer)dynIDs.get(m);
    if (id==null) return -1;
    return id.intValue();
  };

    /**
   * Used to get return type of a class member.
   * <P>The type of a Method is its return type, the type of a Constructor is
   * void.
   * @param  m member whose type is to be determined
   * @return type of the member
   */
  public static Class getType(Member m) {
    if (m instanceof Method) return ((Method)m).getReturnType();
    if (m instanceof Field) return ((Field)m).getType();
    if (m instanceof LocalField) return ((LocalField)m).getType();
    // otherwise it must be java.lang.reflect.Constructor
    if (Debug.enabled)
      Debug.assert(m instanceof java.lang.reflect.Constructor);
    return OP.specialTypes[9]; // java.lang.reflect.Void.TYPE
  };

  /**
   * Used to get types of formal parameters of a member.
   * <P> The reference to the class instance "this" is not counted by
   * this method.
   * @param  m member whose formal parameters are to be obtained
   * @return array of formal parameter types (empty array if none).
   */
  public static Class[] getParameterTypes(Member m) {
    if (m instanceof Method) return ((Method)m).getParameterTypes();
    if (m instanceof LocalMethod) return ((LocalMethod)m).getParameterTypes();
    if (m instanceof Constructor) return ((Constructor)m).getParameterTypes();

    if (Debug.enabled)
      Debug.assert((m instanceof Field)||(m instanceof LocalField));
    
    return new Class[0];
  };

  /**
   * Computes signature of the given member.
   * @param m the member to compute the sugnature of.
   * @return the signature.
   */
  public static String getSignature(Member m) {
    StringBuffer signature=new StringBuffer();
    if (!isField(m)) {
      Class parameters[]=getParameterTypes(m);
      signature.append('(');
      for(int i=0;i<parameters.length;i++) 
        appendSignature(signature,parameters[i]);
      signature.append(')');
    };
    appendSignature(signature,getType(m));
    return signature.toString();
  };

  public static boolean isField(Member m) {
    return (m instanceof Field) || ((m instanceof LocalField)
                                    && !(m instanceof LocalMethod));
  };

  /**
   * Computes the signature of the given class.
   * <P> The signature of the class (Field descriptor) is the string and 
   * it's format is described in the paragraph 4.3.2 of the Java VM 
   * specification (ISBN 0-201-63451-1).
   * <P>The same can be done using <TT>java.lang.Class.getName()</TT> by 
   * converting it's result into the "historical form".
   * <P> This utility method can be used outside of the JEL package
   * it does not involve any JEL specific assumptions and should follow
   * JVM Specification precisely.
   * @param cls is the class to compute the signature of. Can be primitive or
   *            array type.
   * @return the class signature.
   */
  public static String getSignature(Class cls) {
    return appendSignature(new StringBuffer(),cls).toString();
  };

  private static StringBuffer appendSignature(StringBuffer buff, Class cls) {
    if (cls.isPrimitive()) {
      int tid;
      buff.append((tid=OP.typeID(cls))>9?'L':"ZBCSIJFDLV".charAt(tid));
    } else if (cls.isArray()) {
      buff.append('[');
      appendSignature(buff,cls.getComponentType());
    } else { // just a class
      buff.append('L');
      appendHistoricalForm(buff,cls.getName());
      buff.append(';');
    };
    return buff;
  };

  public static String toHistoricalForm(String className) {
    return appendHistoricalForm(new StringBuffer(),className).toString();
  };

  private static StringBuffer appendHistoricalForm(StringBuffer buff,
                                                   String className) {
    int namelen=className.length();
    for(int i=0;i<className.length();i++) {
      char cch=className.charAt(i);
      if (cch=='.') cch='/';
      buff.append(cch);
    };
    return buff;
  };

  
  //+-+-+-+TESTS  
  // ------------------------- TESTSUITE -------------------------------
  
  /**
   * Performs unitary test of the library.
   * @param args ignored.
   */
  public static void main(String[] args) {
    Tester t=new Tester(System.out);
    test(t);
    t.summarize();
  };

  /**
   * Performs unitary test of the library.
   * <p> Used if all package is being tested and not just codegen.
   * @param t Tester to report test results.
   */
  public static void test(Tester t) {
    Library ll=null;
    Class math=null;
    try {
      math=Class.forName("java.lang.Math");
    } catch(ClassNotFoundException e) {
      Debug.println("It is IMPOSSIBLE :)");
    };

    t.startTest("Creating the library of java.lang.Math");
    try {
      Class[] sl=new Class[1];
      sl[0]=math;
      Library l=new Library(sl,null);
      ll=l;
      t.testOK();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Attempt of invocation round(double)");
    try {
      Class[] par=new Class[1];
      par[0]=Double.TYPE;
      Member mf=ll.getMember(null,"round",par);
      if ((mf!=null) && 
          (mf.equals(math.getMethod("round",par))))
	  t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Attempt of invocation round(float)");
    try {
      Class[] par=new Class[1];
      par[0]=Float.TYPE;
      Member mf=ll.getMember(null,"round",par);
      if ((mf!=null) && 
          (mf.equals(math.getMethod("round",par))))
	  t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Attempt of invocation round(int) best is round(float)");
    try {
      Class[] par=new Class[1];
      par[0]=Integer.TYPE;
      Member mf=ll.getMember(null,"round",par);

      Class[] par1=new Class[1];
      par1[0]=Float.TYPE;


      if ((mf!=null) && 
          (mf.equals(math.getMethod("round",par1))))
	  t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Attempt of invocation abs(int) best is abs(int)");
    try {
      Class[] par=new Class[1];
      par[0]=Integer.TYPE;
      Member mf=ll.getMember(null,"abs",par);

      Class[] par1=new Class[1];
      par1[0]=Integer.TYPE;

      if ((mf!=null) && 
          (mf.equals(math.getMethod("abs",par1))))
	  t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Attempt of invocation abs(byte) best is abs(int)");
    try {
      Class[] par=new Class[1];
      par[0]=Byte.TYPE;
      Member mf=ll.getMember(null,"abs",par);
      
      Class[] par1=new Class[1];
      par1[0]=Integer.TYPE;
      
      if ((mf!=null) && 
          (mf.equals(math.getMethod("abs",par1))))
	  t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Attempt of invocation abs(char) best is abs(int)");
    try {
      Class[] par=new Class[1];
      par[0]=Character.TYPE;
      Member mf=ll.getMember(null,"abs",par);
      
      Class[] par1=new Class[1];
      par1[0]=Integer.TYPE;
      
      if ((mf!=null) && 
          (mf.equals(math.getMethod("abs",par1))))
	  t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };
    
    t.startTest("Attempt of invocation min(int,float) best is min(float,float)");
    try {
      Class[] par=new Class[2];
      par[0]=Integer.TYPE;
      par[1]=Float.TYPE;
      Member mf=ll.getMember(null,"min",par);
      
      Class[] par1=new Class[2];
      par1[0]=Float.TYPE;
      par1[1]=Float.TYPE;
      
      if ((mf!=null) && 
          (mf.equals(math.getMethod("min",par1))))
	  t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Attempt of access to the field PI");
    try {
      Class[] par=new Class[0];
      Member f=ll.getMember(null,"PI",par);
      if ((f!=null) && (f.getName().equals("PI")))
	t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Checking assignment of state dependence ");
    try {
      ll.markStateDependent("random",null);
      if (!ll.isStateless(ll.getMember(null,"random",null)))
	t.testOK();
      else
	t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Creating a library with dot operations allowed ");
    Library ldot=null;
    try {
      Class[] stl=new Class[1];
      Class[] dynl=new Class[1];
      Class[] dotl=new Class[2];
      stl[0]=Class.forName("java.lang.Math");
      dynl[0]=Class.forName("java.util.Random");
      dotl[0]=Class.forName("java.util.Hashtable");
      dotl[1]=Class.forName("java.util.Vector");
      
      ldot=new Library(stl,dynl,dotl);
      t.testOK();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Resolving size() on Hashtable");
    try {
      Class[] params=new Class[0];
      Class htable=Class.forName("java.util.Hashtable");
      Member mf=ldot.getMember(htable,"size",params);
      if ((mf!=null) && 
          (mf.equals(htable.getMethod("size",params))))
	  t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Resolving size() on Vector");
    try {
      Class[] params=new Class[0];
      Class vctr=Class.forName("java.util.Vector");
      Member mf=ldot.getMember(vctr,"size",params);
      if ((mf!=null) && 
          (mf.equals(vctr.getMethod("size",params))))
	  t.testOK(); else t.testFail();
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

    t.startTest("Ensuring size() is not leaked to global context.");
    try {
      Class[] params=new Class[0];
      try {
        Member mf=ldot.getMember(null,"size",params);
        t.testFail();
      } catch (CompilationException exc) {
        t.testOK();
      };
    } catch (Throwable e) {
      Debug.reportThrowable(e);
      t.testFail();
    };

  };
  //+-+-+-+TESTS
};
