/*
 * $Id: Optimizer.java,v 1.2 1998/05/23 12:34:50 metlov Exp $
 *
 * This file is part of the Java Expressions Library (JEL).
 *   For more information about JEL visit :
 *    http://galaxy.fzu.cz/JEL/
 *
 * (c) 1998 by Konstantin Metlov(metlov@fzu.cz);
 *
 * JEL is Distributed under the terms of GNU General Public License.
 *    This code comes with ABSOLUTELY NO WARRANTY.
 *  For license details see COPYING file in this directory.
 */

package gnu.jel;

import java.util.Stack;
import gnu.jel.debug.Debug;
import gnu.jel.debug.Tester;
import java.lang.reflect.Method;

/**
 * This class handles storage of the expressions, user level type 
 * checking and optimizations.
 * <P> The only optimization currently supported is the evaluation of the 
 * constant subexpressions. This class goes even further in its attempts to
 * optimize, it evaluates not only arifmetic constant subexpressions, but 
 * also calls to the static methods of the constant arguments. Namely, 
 * expressions such as sin(1) will be evaluated at compile time.
 * <P> Another function of this class is the error handling. If during 
 * evaluation of the constant subexpression a error occurs (Throwable is 
 * thrown) it is caught and compiled into the resulting class image to be 
 * rethrown at execution time. Example :
 * <TT>1/0</TT> will not cause any exceptions to come out of the compiler, but 
 * will be compiled into <TT>{ throw caughtArithmeticError; }</TT> if the 
 * optimization is turned on. When optimization is off the previous
 * expression will be compiled to <TT>{ return 1/0 }</TT>, which will throw
 * exception at run time ( execution of the <TT>evaluate(...)</TT> method
 * of the <TT>gnu.jel.CompiledExpression</TT>). Another thing to note, that 
 * <TT>Throwable</TT>s caught at compile time are not recreated at compile 
 * time  ( they are not compiled into <TT>{throw <B>new</B> 
 * ArithmeticError()}</TT>). Errors being thrown from <TT>evaluate(..)</TT>
 * method are EXACTLY the same object as those caught during the constant 
 * subexpression eveluation. This approach should greatly simplify the error
 * handling and applies not only to arithmetic expressions but also to any 
 * exceptions, thrown by interpretable methods.
 * <P> For more information on how to specify static non interpretable 
 * method (as in the case of <TT>java.Math.random()</TT> see the 
 * documentation for <TT>gnu.jel.Library</TT>
 * @see gnu.jel.Library
 * @author Konstantin L. Metlov (metlov@fzu.cz)
 */
public class Optimizer {
  
  private Library lib;

  private Stack types=new Stack();
  private OPlist code=new OPlist();
  private Stack functions=new Stack();
  private Stack functionsDescriptors=new Stack();  
  private Throwable caughtThrowable=null;
  
  private boolean finished=false;

  /**
   * Constructs the new "empty" optimizer with the library specified.
   * @param lib is the library, used for function names resolution.
   */
  public Optimizer(Library lib) {
    this.lib=lib;
  };


  /**
   * Generates a "load boolean constant" operation.
   * @param c is the constant to load.
   */
  public void load(boolean c) {
    load(Boolean.TYPE,new Boolean(c));
  };

  
  /**
   * Generates a "load byte constant" operation.
   * @param c is the constant to load.
   */
  public void load(byte c) {
    load(Byte.TYPE,new Byte(c));
  };

  /**
   * Generates a "load char constant" operation.
   * @param c is the constant to load.
   */
  public void load(char c) {
    load(Character.TYPE,new Character(c));
  };


  /**
   * Generates a "load short constant" operation.
   * @param c is the constant to load.
   */
  public void load(short c) {
    load(Short.TYPE,new Short(c));
  };


  /**
   * Generates a "load int constant" operation.
   * @param c is the constant to load.
   */
  public void load(int c) {
    load(Integer.TYPE,new Integer(c));
  };


  /**
   * Generates a "load long constant" operation.
   * @param c is the constant to load.
   */
  public void load(long c) {
    load(Long.TYPE,new Long(c));
  };

  /**
   * Generates a "load float constant" operation.
   * @param c is the constant to load.
   */
  public void load(float c) {
    load(Float.TYPE,new Float(c));
  };
  
  /**
   * Generates a "load double constant" operation.
   * @param c is the constant to load.
   */
  public void load(double c) {
    load(Double.TYPE,new Double(c));
  };
  
  
  private void load(Class t, Object o) {
    seeNonFinished();
    OP nop = new OP_load(t,o);
    types.push(t);
    code.addLast(nop);
  };
  
  /**
   * Generates an explicit "convert type" operation.
   * <P> It has the sense to call this function only for narrowing 
   * conversions (see &sect;5.1.3 and &sect;5.1.5 of the Java Language
   * Specification (JLS) ). 
   * Widening ( &sect;5.1.2,  &sect;5.1.4 of JLS) type conversions are 
   * performed by this compiler automatically.
   * @param to is the class to convert to.
   * @param widening if true prohibits doing narrowing conversions.
   * @exception IllegalStateException if requested conversion is not supported.
   */
  public void convert(Class to,boolean widening) throws IllegalStateException {
    seeNonFinished();
    Class ct=(Class)types.peek();
    
    if (!ExpressionImage.canConvert(ct,to))
      throw new IllegalStateException("Can not convert "+ct.toString()+
				      " to "+to.toString()+
				      ".");
    
    if (widening && !ExpressionImage.canConvertByWidening(ct,to))
      throw new IllegalStateException("You must specify narrowing conversion"+
				      " from "+ct.toString()+" to "+
				      to.toString()+" explicitly");
    
    if (ct!=to) { // need convert
      OP lastOP=code.getLast();
      if (lastOP instanceof OP_convert) { // can patch
	((OP_convert)lastOP).setType(to);
      } else 
	code.addLast(new OP_convert(to));
    };
    types.pop();
    types.push(to);
  };

  /**
   * Generates an explicit "convert type" operation.
   * <P> This function is equivalent to <TT>convert(to,false)</TT>.
   * Widening ( &sect;5.1.2,  &sect;5.1.4 of JLS) type conversions are 
   * performed by this compiler automatically.
   * @param to is the class to convert to.
   * @exception IllegalStateException if requested conversion is not supported.
   */
  public void convert(Class to) throws IllegalStateException {
    convert(to,false);
  };
  
  
  /**
   * Generates an unary operation.
   * <P> The only unary operation for now is the negation (invert sign) 
   * operation.
   * @param is the operation code, for now it can be only 
   * <TT>ExpressionImage.UN_NE</TT>.
   * @exception IllegalStateException if operation is not supported
   * for types in stack.
   */
  public void unary(int o) {
    seeNonFinished();
    Class  ct=(Class)types.peek();
    
    if (!ExpressionImage.canGenerateUnary(o,ct)) {
      throw new IllegalStateException("Unary "+ExpressionImage.unaryNames[o]+
				      " is not supported on "+ct.getName()+
				      "'s");
    };
    code.addLast(new OP_unary(o));
  };
  
  /**
   * Denotes the start of the function call.
   * <P> Example of the sequence of method calls to perform (compile)
   * the function invocation is given in the description of
   * gnu.jel.Optimizer.function_call(...) method.
   * @see gnu.jel.Optimizer#function_call
   */
  public void function_start() {
    seeNonFinished();
    OP_call opc=new OP_call();
    functions.push(opc);
    int[] descrs={0};
    functionsDescriptors.push(descrs);
    code.addLast(new OP_start(opc));
  };

  /**
   * Specifies that the parameter for the binary operation is now in stack.
   * <P> Example of the sequence of method calls to perform (compile)
   * the binary operation is given in the description of
   * gnu.jel.Optimizer.binaryOP() method.
   * @see gnu.jel.Optimizer#binaryOP
   */
  public void binaryOP_param() {
    seeNonFinished();
    OP_binary opc=new OP_binary();
    functions.push(opc);
    int[] descrs={0};
    functionsDescriptors.push(descrs);
    code.addLast(new OP_param(opc,(Class)types.peek()));
  };

  /**
   * Specifies that the parameter for the function is now in stack.
   * <P> Example of the sequence of method calls to perform (compile)
   * the function is given in the description of
   * gnu.jel.Optimizer.function_call(...) method.
   * @see gnu.jel.Optimizer#function_call
   */
  public boolean function_param() {
    seeNonFinished();
    OP_function opf=(OP_function) functions.peek();
    int[] descrs=(int[]) functionsDescriptors.peek();
    descrs[0]++;
    
    code.addLast(new OP_param(opf,(Class)types.peek()));
    return true; 
  };

  /**
   * Generates the function call.
   * Example : "how to call the function"
   * <PRE>
   * optimizer.function_start();
   * optimizer.load(2L);
   * optimizer.function_param();
   * optimizer.load(2.0);
   * optimizer.function_param();
   * optimizer.function_call("name_in_the_library");
   * </PRE>
   * The function name is searched in the library. If there are
   * several applicable functions in the library with the same name (not
   * necessary in the same object originally) the most specific one is
   * called ( See &sect; 15.11.2.2 in the Java Language Specification). Types
   * conversion takes place automatically.
   * @param name is the name of the function to call.
   * @exception  IllegalStateException if no function with the given name
   * and parameter types in stack can be found in the library.
   */
  public void function_call(String name) throws IllegalStateException {
    seeNonFinished();
    OP_call opf=(OP_call) functions.peek();
    int[] descrs=(int[])functionsDescriptors.peek();
    
    int nparams=descrs[0];
    Class[] paramTypes=new Class[nparams];
    for(int i=nparams-1;i>=0;i--) paramTypes[i]=(Class)types.pop();

    Method m;
    try {
      m=lib.getMethod(name,paramTypes);
    } catch (NoSuchMethodException e) {
      // push parameters back
      for(int i=0;i<nparams;i++) types.push(paramTypes[i]);
      throw new IllegalStateException(e.getMessage());
    };
    
    OP_call opc=(OP_call)opf;
    code.addLast(opc);
    
    opc.setMethod(code,m,lib.getDynamicMethodClassID(m),lib.isStateless(m));
    
    types.push(m.getReturnType());
    functions.pop(); functionsDescriptors.pop();
  };

  /**
   * Generates a binary operation.
   * <P> The binary operation codes are defined in 
   * <TT>gnu.jel.ExpressionImage</TT> as constants BI_XX .
   * <P> Example : "how to sum up two numbers with this optimizer"
   * <PRE>
   * optimizer.load(2L);
   * optimizer.binaryOP_param();
   * optimizer.load(2.0);
   * optimizer.binaryOP(ExpressionImage.BI_PL);
   * </PRE>
   * Note different types of operands, conversion ( to double) will be
   * performed automatically. Note also binaryOP_param() usage, its use is
   * obligatory to denote that the parameter for subsequent binary operation
   * is now ion stack.
   * @param is the operation code, see above.
   * @exception  IllegalStateException if this binary op is not supported
   * for the types in stack.
   * @see gnu.jel.ExpressionImage
   */
  public void binaryOP(int o) throws IllegalStateException {
    seeNonFinished();
    OP_binary opf=(OP_binary) functions.peek();
    int[] descrs=(int[])functionsDescriptors.peek();
    
    code.addLast(new OP_param(opf,(Class)types.peek()));
    
    Class op2=(Class)types.pop();
    Class op1=(Class)types.pop();
    
    if (!ExpressionImage.canGenerateBinary(o,op1,op2)) {
      types.push(op1);
      types.push(op2);
      throw new IllegalStateException("Types "+op1.toString()+" and "+
				      op2.toString()+" are not suitable "+
				      "for the operation \""+
				      ExpressionImage.binaryNames[o]+"\"");
    };
    
    Class ft=ExpressionImage.getMostGeneralType(op1,op2);
    
    OP_binary opc=(OP_binary)opf;    
    
    code.addLast(opc);
    
    opc.setOperation(code,o,ft);
    
    types.push(ft);
    functions.pop(); functionsDescriptors.pop();
  };
  
  /**
   * Finishes the function.
   * <P> This method should be called exactly once before the attempt
   * to compile is made.
   * <P> If there was nothing in the stack before it's called this method
   * generates the returning of the null pointer.
   * @exception  IllegalStateException if the function is not in the
   * proper state to be finished (some extra items are in stack).
   */
  public void finish() throws IllegalStateException {
    seeNonFinished();
    if ((code.size!=0) && (types.size()!=1))
      throw new IllegalStateException("Stack should contain single item on"+
				      " exit.");
    if ((functions.size() !=0) || (functionsDescriptors.size() !=0)) 
      throw new IllegalStateException("There are function calls in progress.");
    
    finished=true;
  };
  
  /**
   * Optimizes the function.
   * <P> Currently only evaluation of constant subexpressions is performed.
   * @param of is the optimization flags (0 - do nothing; >0 optimize).
   */
  public void optimize(int of) {
    seeFinished();
    if (of==0) return;
    while(optimizeIteration());
  };
  
  protected boolean optimizeIteration() {
    OP cop=code.getFirst();
    boolean didsome=false;
    while ((cop!=null) && (caughtThrowable==null)) {
      OP ncop=cop.next();
      if (cop instanceof OP_function) {
	OP_function copf=(OP_function) cop;
	if (copf.canInterpret()) {
	  try {
	    copf.interpret(code);
	    didsome=true;
	  } catch (Throwable exc) {
	    caughtThrowable=exc;
	  };
	};
      };
      cop=ncop;
    };
    return didsome;
  };

  /**
   * Compiles the expression.
   * @return The instance of the <TT>gnu.jel.CompiledExpression</TT> subclass,
   * with the function, represented by this optimizer compiled in.
   */
  public CompiledExpression compile() {
    seeFinished();
    ExpressionImage image=new ExpressionImage();
    code.compile(image);
    //    if (caughtThrowable==null) 
    image.asm_return();
      //    else {
      //      image.asm_load_object(caughtThrowable);
      //      image.asm_ThrowReturn();
      //    };
    ExpressionImage.dumpImage(image);
    return image.getExpression();
  };

  /**
   * Represents the expression, contained in this optimizer as <TT>String</TT>.
   * @return String representation of the expression.
   */
  public String toString() {
    //    if (caughtThrowable==null) 
    return code.toString();
    //    return code.toString()+" $"+caughtThrowable.getClass().toString();
  };

  private void seeFinished() {
    if (!finished) throw new IllegalStateException("Attempt to instantiate "+
						   "unfinished function.");
  };

  private void seeNonFinished() {
    if (finished) throw new IllegalStateException("Attempt to modify "+
						  "finished function.");
  };

  //-----------------------TESTS----------------------------------
  
  /**
   * Performs unitary test of the interpreter.
   * <P> This function works only if the Debugging is turned "ON".
   * @param args ignored.
   * @see gnu.jel.debug.Debug#enabled
   */
  public static void main(String[] args) {
    if (Debug.enabled) {
      Tester t=new Tester(System.out);
      test(t);
      t.summarize();
    };
  };


  /**
   * Performs unitary test of the interpreter.
   * <P> This function works only if the Debugging is turned "ON".
   *  Used if all package is being tested and not just this class alone.
   * @param t Tester to report test results.
   * @see gnu.jel.debug.Debug#enabled
   */
  public static void test(Tester t) {
    if (Debug.enabled) {
      // Prepare library
      Library clib=null;
      try {
	Class[] statlb={Class.forName("java.lang.Math")};
	clib=new Library(statlb,null);
	clib.markStateDependent("random",null);
      } catch (Throwable e) {Debug.reportThrowable(e);};
      
      
      t.startTest("Enter \"(2.0, 2,*)\" ");
      try {
       	Optimizer op=new Optimizer(clib);
	op.load(2.0);
	op.binaryOP_param();
	op.load(2L);
	op.binaryOP(ExpressionImage.BI_MU);
	op.finish();
	Debug.println("");
	Debug.println(op.toString());
	CompiledExpression ce=op.compile();
	Number res=(Number)ce.evaluate(null);
	Debug.println("And the result is "+res.toString());
	if (res.intValue()==4) t.testOK(); else t.testFail();
      } catch (Throwable e) {
	Debug.reportThrowable(e);
	t.testFail();
      };

      t.startTest("Enter \"5+(2.0*2)*exp(0)\" ");
      Debug.println("");
      try {
       	Optimizer op=new Optimizer(clib);
	op.load(5L);
	op.binaryOP_param();
	op.load(2.0);
	op.binaryOP_param();
	op.load(2L);
	op.binaryOP(ExpressionImage.BI_MU);
	op.binaryOP_param();
	   op.function_start();
	   op.load(0);
	   op.function_param();
           op.function_call("exp");
	op.binaryOP(ExpressionImage.BI_MU);
	op.binaryOP(ExpressionImage.BI_PL);
	op.finish();
	Debug.println(op.toString());
	CompiledExpression ce=op.compile();
	Number res=(Number)ce.evaluate(null);
	Debug.println("And the result is "+res.toString());
	if (res.intValue()==9) t.testOK(); else t.testFail();
      } catch (Throwable e) {
	Debug.reportThrowable(e);
	t.testFail();
      };


    };
  };

  

  //--------------------------------------------------------------
 
};


// Below there are classes for the linked list (LL), representing
// expression. List is. probably, the most effective representation of the
// expression in the polish notation. And, well, I can get Scheme 
// representation of the expression easily :)

// JDK 1.2 will have the LL implementation built in, thus
// classes below should be later migrated to the JDK 1.2 list.

abstract class OP {
  protected OP next=null;
  protected OP prev=null;
  
  abstract void compile(ExpressionImage ei);

  public OP next() { return next;};
};

class OPlist extends OP {
  
  protected int size=0;
 
  int size() {return size;}; 
  
  OP getFirst() {
    if (Debug.enabled)
      Debug.assert(size>0,"First in empty list.");
    return next;
  };

  OP getLast()  {
    if (Debug.enabled)
      Debug.assert(size>0,"Last in empty list.");
    return prev;
  };

  void addFirst(OP o) {
    o.prev=null; o.next=null;
    if (next!=null) { o.next=next; next.prev=o; };
    next=o;
    size++;
  };
  
  void addLast(OP o) {
    o.prev=null; o.next=null;
    if (prev!=null) { o.prev=prev; prev.next=o; };
    prev=o;
    if (next==null) next=o;  // if first
    size++;
  };
  
  void addBefore(OP e, OP ne) {
    if (e==next) {
      addFirst(ne);
      return;
    };
    ne.prev=e.prev;
    ne.next=e;
    e.prev.next=ne;
    e.prev=ne;
    size++;
  };
  
  void remove(OP o) {
    if (o.prev!=null) o.prev.next=o.next; else this.next=o.next;
    if (o.next!=null) o.next.prev=o.prev; else this.prev=o.prev;
    o.next=null; o.prev=null; // help GC
    size--;
  };
  
  // This compiles whole expression
  void compile(ExpressionImage ei) {
    OP curr=next;
    while(curr!=null) {
      curr.compile(ei);
      curr=curr.next;
    };
  };

  public String toString() {
    OP curr=next;
    StringBuffer res=new StringBuffer();
    while (curr!=null) {
      res.append(curr.toString());
      curr=curr.next;
    };
    return res.toString();
  };
  
  
};

class OP_start extends OP {  // Analogous to the list delimiter character
  protected OP_function f=null;
  protected Method m=null;
  protected int id=999;
  
  OP_start(OP_function f) {this.f=f;};
  
  OP_function getFunction() {return f;};
  
  // Called by corresponding OP_function, when method to call is determined.
  // For binary primitive operations this is never called, thus no code
  // will be generated
  void setMethod(Method m,int id) {this.m=m; this.id=id;};
  
  void compile(ExpressionImage ei) {
    if (m !=null) 
      ei.asm_func_start(m,id);
  };

  public String toString() {
    return "[";
  };
  
};

// This class is a placeholder to insert the reference to the object, imple
// memnting the function. As function overloading is allowed the actual
// reference is unknown before ALL formal arguments are parsed.
class OP_param extends OP_start {  // Analogous to opening bracket in Lisp
  OP_function f=null;
  Class typeInStack=null;
  
  // f is the reference to the OP_function, which is the "closing bracket".
  OP_param(OP_function f,Class typeInStack) {
    super(f);this.typeInStack=typeInStack;
  };

  // Called, when the type of parameter is determined to make param conversion.
  void setConvert(OPlist list, Class cls) {
    if (cls!=typeInStack) {
      if (prev instanceof OP_convert) {
	((OP_convert) prev).setType(cls);
      } else {
	list.addBefore(this,new OP_convert(cls)); // Add conversion OP
      };
    };
  };

  void compile(ExpressionImage ei) {
    if (m!=null)
      ei.asm_func_param();
  };

  public String toString() {
    return ",";
  };
 
};


class OP_load extends OP {
  Class type;
  Object what;
  
  static final char[] typecodes = {'Z','B','C','S','I','J','F','D'};

  OP_load(Class type,Object what) {
    this.type=type;
    this.what=what;
  };
  
  void compile(ExpressionImage ei) {
    if ((type==null) || (what==null)) {
      ei.asm_load_object(null);
      return;
    };
    if (type.isPrimitive())
      ei.asm_load_primitive(what);
    else 
      ei.asm_load_object(what);
    return;
  }; 

  public String toString() {

    if (type.isPrimitive()) {
      return what.toString()+typecodes[ExpressionImage.primitiveID(type)];
    };

    return what.toString();
  };

  
};

abstract class OP_function extends OP {
  
  abstract boolean canInterpret();
  
  abstract void interpret(OPlist list) throws Throwable;
  
  void consume(OPlist list,OP_load replace_with, int nop,boolean match_start) {
    int toremove=nop*2+1; // nop[load param ] this
    if (match_start) toremove++; // start ?
    
    OP next_op=next;

    OP cop=this;
    for(int i=0; i<toremove; i++) {
      OP prv=cop.prev; list.remove(cop); cop=prv;
    };
    if (next_op!=null) 
      list.addBefore(next_op, replace_with);
    else
      list.addLast(replace_with);
  };
  
  boolean operandsReady(int nop,boolean match_start) {
    boolean ready=true;
    OP oper=prev;
    for(int i=0;(i<nop) && ready;i++) {
      ready= ((oper instanceof OP_param) && (oper.prev instanceof OP_load));
      oper=oper.prev.prev;
    };
    if (match_start)
      ready = ready && (oper instanceof OP_start);
    return ready;
  };
  
  static boolean isFloat(int clsID) {
    return (clsID>5);
  };

  static Number widen(Object o, int clsID) {
    switch (clsID) {
    case 0: // Z
      if (((Boolean)o).booleanValue()) 
	return new Long(1L); 
      else 
	return new Long(0L);
    case 1: // B
      return (Number)o;
    case 2: // C
      return new Long((long)((Character)o).charValue());
    case 3: // S
    case 4: // I
    case 5: // J
    case 6: // F
    case 7: // D
      return (Number)o;
    default:
      if (Debug.enabled)
	Debug.println("Attempt to widen wrong primitive ("+clsID+").");
      return new Long(0L);
    };
    
  };

  static Object narrow(Number val, int clsID) {
    switch (clsID) {
    case 0: // Z
      if (val.longValue()!=0) return Boolean.TRUE; else return Boolean.FALSE;
    case 1: // B
      return new Byte(val.byteValue());
    case 2: // C
      return new Character((char)val.longValue());
    case 3: // S
      return new Short(val.shortValue());
    case 4: // I
      return new Integer(val.intValue());
    case 5: // J
      return new Long(val.longValue());
    case 6:
      return new Float(val.floatValue());
    case 7:
      return new Double(val.doubleValue());
    default:
      if (Debug.enabled)
	Debug.println("Attempt to narrow wrong primitive ("+clsID+").");
      return null;
    };
  };
};


abstract class OP_unary_primitive extends OP_function {

  boolean canInterpret() {
    return (prev instanceof OP_load);
  };

  void interpret(OPlist list) throws Throwable {
    OP_load val=(OP_load) prev;
    doOperation(val);
    list.remove(this);
  };
  
  // Should modify OP_load to represent correct value
  abstract void doOperation(OP_load val) throws Throwable;
     
};

class OP_convert extends OP_unary_primitive {
  
  Class to;
  
  OP_convert(Class to) {
    this.to=to;
  };
  
  void setType(Class new_to) {to=new_to;};
  
  void doOperation(OP_load c) throws Throwable {
    if (c.type==to) return ;   // it's already the same
    if (((c.type==null) || (c.what==null)) && (!to.isPrimitive()) ) 
      return;  // null has any object type
    
    boolean pc=c.type.isPrimitive();
    boolean pt=to.isPrimitive();
    if (!pc && !pt) {  // object to object
      Debug.assert(to.isAssignableFrom(c.type),"Class types not compatible.");
      return; 
    };
    if (pc && pt) { // Z C B S I L F D
      int pfi=ExpressionImage.primitiveID(c.type);
      int pti=ExpressionImage.primitiveID(to);
      c.what=narrow(widen(c.what,pfi),pti);
      c.type=to;
      return; 
    };
    return; // can't convert between objects and primitives 
  };

  void compile(ExpressionImage ei) {
    ei.asm_convert(to);
  };
  
  public String toString() {
    return "("+to.toString()+")";
  };
  

};

class OP_unary extends OP_unary_primitive {
  int opc;
  
  OP_unary(int opc) {
    this.opc=opc;
  };

  void doOperation(OP_load c) throws Throwable {  
    if (Debug.enabled)
      Debug.assert((c.type!=null) && (c.type.isPrimitive()),
		   "Wrong types for binary OP");
    
    int id=ExpressionImage.primitiveID(c.type);
    
    Number val=widen(c.what,id);
    if (opc==0) {
      if (isFloat(id)) 
	val=new Double(-val.doubleValue());
      else
	val=new Long(-val.longValue());
    } else {
      if (Debug.enabled)
	Debug.println("Wrong unary opcode.");
    };
    c.what=narrow(val,id);
  };
  
  void compile(ExpressionImage ei) {
    ei.asm_unary(opc);
  };

  public String toString() {
    return "-";
  };
  
};

class OP_binary extends OP_function {
  int opc=999;
  OP_binary() {
  };
  
  void setOperation(OPlist list, int opc,Class type) {
    this.opc=opc;
    int cpar=2;
    OP cop=prev;
    while ((cop!=null) && (cpar>=0)) {
      if (cop instanceof OP_param) {
	OP_param copp=(OP_param)cop;
	if (copp.getFunction()==this) {
	  copp.setConvert(list,type);
	};
      };
      cop=cop.prev;
    };
  };

  boolean canInterpret() { return operandsReady(2,false); };
  
  void interpret(OPlist list) throws Throwable {
    OP_load c2=(OP_load) prev.prev;
    OP_load c1=(OP_load) prev.prev.prev.prev;
    
    if (Debug.enabled)
      Debug.assert((c1.type!=null) && (c1.type==c2.type) && 
		   (c2.type.isPrimitive()),"Wrong types for binary OP");
    
    int id=ExpressionImage.primitiveID(c1.type);
    
    // Widen
    Number n1=widen(c1.what,id),n2=widen(c2.what,id);

    // Perform      
    if (isFloat(id)) {
      double d1=n1.doubleValue(),d2=n2.doubleValue();
      switch (opc) {
      case 0: d1=d1+d2; break; //PL
      case 1: d1=d1-d2; break; //MI
      case 2: d1=d1*d2; break; //MU
      case 3: d1=d1/d2; break; //DI
      case 4: d1=d1%d2; break; //RE
      default :
	if (Debug.enabled)
	  Debug.println("Wrong operation on float ("+opc+").");      
      };
      n1=new Double(d1);
    } else { 
      long l1=n1.longValue(),l2=n2.longValue();
      switch (opc) {
      case 0: l1=l1+l2;break; //PL
      case 1: l1=l1-l2;break; //MI
      case 2: l1=l1*l2; break; //MU
      case 3: l1=l1/l2; break; //DI
      case 4: l1=l1%l2; break; //RE
      case 5: l1=l1&l2; break; //AN
      case 6: l1=l1|l2; break; //OR
      case 7: l1=l1^l2; break; //XO
      default :
	if (Debug.enabled)
	  Debug.println("Wrong operation on integer ("+opc+").");      
      };
      n1=new Long(l1);
    };

    // Narrow    
    c1.what=narrow(n1,id);
    
    consume(list,c1,2,false);
  };

  void compile(ExpressionImage ei) {
    ei.asm_binary(opc);
  };

  public String toString() {
    char ch='?';
    switch (opc) {
    case 0: ch='+';break; //PL
    case 1: ch='-';break; //MI
    case 2: ch='*'; break; //MU
    case 3: ch='/'; break; //DI
    case 4: ch='%'; break; //RE
    case 5: ch='&'; break; //AN
    case 6: ch='|'; break; //OR
    case 7: ch='^'; break; //XO
    default :
      if (Debug.enabled)
	Debug.println("Wrong operation on integer ("+opc+").");      
    };
    
    return ""+ch;
  };

  
};



class OP_call extends OP_function {
  
  Method m=null;
  int nargs=999;
  Class[] args=null;
  boolean interpretable=false;

  OP_call() {};

  boolean canInterpret() { 
    return (interpretable && operandsReady(nargs,true)); 
  };
  
  void setMethod(OPlist list,Method m, int id,boolean interpretable) {
    args=m.getParameterTypes();
    nargs=args.length;
    this.interpretable=interpretable;
    this.m=m;
    // Search for parameters and set their types
    int cpar=args.length-1;
    OP cop=prev;
    while ((cop!=null) && (cpar>=0)) {
      if (cop instanceof OP_param) {
	OP_param copp=(OP_param)cop;
	if (copp.getFunction()==this) {
	  copp.setConvert(list,args[cpar--]);
	  copp.setMethod(m,id);
	};
      };
      cop=cop.prev;
    };
    if (Debug.enabled) 
      Debug.assert((cop!=null) && (cpar==-1),
		  "Not all parameters are present at function call");
    
    OP_start ops=null;
    while ((cop!=null) && (ops==null)) {
      if (cop instanceof OP_start) {
	ops=(OP_start)cop;
	if (ops.getFunction()!=this) ops=null;
      };
      cop=cop.prev;
    };
    if (Debug.enabled) 
      Debug.assert(ops!=null,"OP_start was not found.");
    ops.setMethod(m,id);
  };
  
  void interpret(OPlist list) throws Throwable {
    Object[] args=new Object[nargs];
    
    // collect arguments
    OP cop=prev;
    int argc=nargs-1;
    while (argc>=0) {
      args[argc--]=((OP_load)cop.prev).what;
      cop=cop.prev.prev;
    };
    if (Debug.enabled) 
      Debug.assert(cop instanceof OP_start,"Wrongly formed function call.");
    
    Object val=m.invoke(null,args);
    OP_load result=new OP_load(m.getReturnType(),val);
    
    consume(list,result,nargs,true);
    
  };
  
  void compile(ExpressionImage ei) {
    ei.asm_func_call();
  };

  public String toString() {
    return m.getName()+"]";
  };
  
}






