/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright(C) 2000, 2001 Edgar Denny(e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <ctype.h>

#include "text_util.h"
#include "food.h"
#include "plan.h"
#include "plan_win.h"
#include "wrap_mysql.h"
#include "init_database.h"
#include "person.h"

/* add an ingredient to the meal plan. */
void
gnutr_add_ingredient_to_plan( char *food_no, 
                              char *amount, 
                              char *msre_no)
{
	char *date; 
	char *time_of_day;
	char *person_no;
	char *query;
	gboolean success;

	g_return_if_fail(food_no != NULL);
	g_return_if_fail(amount != NULL);
	g_return_if_fail(msre_no != NULL);

	/* get the date from the plan_win. */
	date = gnutr_get_date();
	
	/* get the time of day from the plan_win. */
	time_of_day = gnutr_get_time_of_day();

	/* get the person_no. */
	person_no = gnutr_get_connected_person_no();

	/* add a new food to the plan. Note that the table is a 
	 * temporary one. */
	query = g_strconcat( "INSERT INTO food_plan_temp "
		"VALUES ( '", person_no, "', '", date, "', '", 
		time_of_day, "', ", "'", amount, "', '", msre_no, "', '", 
		food_no, "' )", NULL);
	success = no_ret_val_query_db( query);
	g_assert( success);
	g_free( query);
	g_free( person_no);

	/* update the clist containing the recipe and food plan. */
	gnutr_update_plan_win();
}

/* add a recipe to the meal plan. */
void
gnutr_add_recipe_to_plan( char *recipe_name, 
                          char *recipe_no,
                          char *no_portions)
{
	char *date; 
	char *time_of_day;
	char *person_no;
	char *query;
	gboolean success;

	g_return_if_fail(recipe_name != NULL);
	g_return_if_fail(recipe_no != NULL);
	g_return_if_fail(no_portions != NULL);

	/* get the date from the plan_win. */
	date = gnutr_get_date();
	
	/* get the time of day from the plan_win. */
	time_of_day = gnutr_get_time_of_day();

	/* get the person_no. */
	person_no = gnutr_get_connected_person_no();

	/* add a new recipe to the plan. Note that the table is a
	 * temporary one. */
	query = g_strconcat( "INSERT INTO recipe_plan_temp "
		"VALUES ( '", person_no, "', '", date, "', '", 
		time_of_day, "', ", "'", no_portions, "', '", 
		recipe_no, "' )", NULL);
	success = no_ret_val_query_db( query);
	g_assert( success);
	g_free( query);
	g_free( person_no);

	/* update the clist containing the recipe and food plan. */
	gnutr_update_plan_win();
}

/* save the plan data to file. */
void
gnutr_save_plan()
{
	char *person_no;
	char *query;
	gboolean success;
	GList *list = NULL, *ptr;
	char **elm;

	person_no = gnutr_get_connected_person_no();

	/* delete the old plan from the database. */
	query = g_strconcat( "DELETE FROM food_plan WHERE "
		"person_no = '", person_no, "'", NULL);
	success = no_ret_val_query_db( query);
	g_assert( success);
	g_free( query);

	query = g_strconcat( "DELETE FROM recipe_plan WHERE "
		"person_no = '", person_no, "'", NULL);
	success = no_ret_val_query_db( query);
	g_assert( success);
	g_free( query);

	/* write the temporary table data to the stored table. */
	query = g_strdup( "SELECT * FROM food_plan_temp");
	list = rows_glist_ret_val_query_db( query);
	g_free( query);

	for ( ptr = list; ptr; ptr = ptr->next) {
		elm = (char **)ptr->data;

		query = g_strconcat( "INSERT INTO food_plan "
			"VALUES ( '", elm[0], "', '", elm[1], "', '", 
			elm[2], "', ", "'", elm[3], "', '", elm[4], "', '", 
			elm[5], "' )", NULL);
		success = no_ret_val_query_db( query);
		g_assert( success);
		g_free( query);
	}
	if ( list) gnutr_free_row_list( list, 6);

	query = g_strdup( "SELECT * FROM recipe_plan_temp");
	list = rows_glist_ret_val_query_db( query);
	g_free( query);

	for ( ptr = list; ptr; ptr = ptr->next) {
		elm = (char **)ptr->data;

		query = g_strconcat( "INSERT INTO recipe_plan "
			"VALUES ( '", elm[0], "', '", elm[1], "', '", 
			elm[2], "', ", "'", elm[3], "', '", 
			elm[4], "' )", NULL);
		success = no_ret_val_query_db( query);
		g_assert( success);
		g_free( query);
	}
	if ( list) gnutr_free_row_list( list, 5);
	g_free( person_no);
}

GList *
gnutr_get_foods_for_date( char *date)
{
	char *query;
	GList *ret_list;

	g_return_val_if_fail( date, NULL);

	/* get foods for the date from the temporary table. */
	query = g_strconcat( "SELECT time, amount, msre_no, fd_no "
		"FROM food_plan_temp ",
		"WHERE date = '", date, "'", NULL);
	ret_list = rows_glist_ret_val_query_db( query);
	g_free(query);

	return ret_list;
}

GList *
gnutr_get_recipes_for_date( char *date)
{
	char *query;
	GList *ret_list;

	g_return_val_if_fail( date, NULL);

	/* get recipes for the date from the temporary table. */
	query = g_strconcat( 
		"SELECT time, no_portions, recipe.recipe_no, recipe_name ",
		"FROM recipe_plan_temp, recipe "
		"WHERE date = '", date, "' ",
		"AND recipe_plan_temp.recipe_no = recipe.recipe_no", NULL);
	ret_list = rows_glist_ret_val_query_db( query);
	g_free(query);

	return ret_list;
}

/* delete a recipe or food from the meal plan. */
void
gnutr_delete_from_plan_temp_db( char *date, 
                                char *time_of_day, 
                                int   table_type, 
                                char *hidden_data)
{
	char *query;
	gboolean success;
	enum { FOOD_ROW, RECIPE_ROW };

	g_return_if_fail( date);
	g_return_if_fail( time_of_day);
	g_return_if_fail( table_type == FOOD_ROW || table_type == RECIPE_ROW);
	g_return_if_fail( hidden_data);

	if ( table_type == FOOD_ROW) {
		query = g_strconcat( "DELETE FROM food_plan_temp WHERE ",
			"date = '", date, "' AND ",
			"time = '", time_of_day, "' AND ",
			"fd_no = '", hidden_data, "'", NULL);
		success = no_ret_val_query_db( query);
		g_assert( success);
		g_free( query);
	} else {
		query = g_strconcat( "DELETE FROM recipe_plan_temp WHERE ",
			"date = '", date, "' AND ",
			"time = '", time_of_day, "' AND ",
			"recipe_no = '", hidden_data, "'", NULL);
		success = no_ret_val_query_db( query);
		g_assert( success);
		g_free( query);
	}
}

/* edit the selected food or recipe in the meal plan. */
void
gnutr_edit_plan_temp_db( char *date, 
                         char *time_of_day, 
                         int table_type, 
                         char *new_amount,
                         char *new_msre_desc,
                         char *new_no_portions,
                         char *hidden_data)
{
	char *query;
	char **row;
	gboolean success;
	enum { FOOD_ROW, RECIPE_ROW };
	int i;

	g_return_if_fail( date);
	g_return_if_fail( time_of_day);
	g_return_if_fail( table_type == FOOD_ROW || table_type == RECIPE_ROW);
	g_return_if_fail( hidden_data);

	if ( table_type == FOOD_ROW) {
		char *new_msre_no;
		GHashTable *htbl;

		g_return_if_fail( new_amount);
		g_return_if_fail ( new_msre_desc);

		htbl = get_htbl_msre_desc_msre_no();
		new_msre_no = (char *)g_hash_table_lookup( htbl, 
			(gpointer)new_msre_desc);

		query = g_strconcat( "SELECT * FROM food_plan_temp WHERE ",
			"date = '", date, "' AND ",
			"time = '", time_of_day, "' AND ",
			"fd_no = '", hidden_data, "'", NULL);
		row = row_ret_val_query_db( query);
		g_assert( row);
		g_free( query);

		query = g_strconcat( "DELETE FROM food_plan_temp WHERE ",
			"date = '", date, "' AND ",
			"time = '", time_of_day, "' AND ",
			"fd_no = '", hidden_data, "'", NULL);
		success = no_ret_val_query_db( query);
		g_assert( success);
		g_free( query);

		query = g_strconcat( "INSERT INTO food_plan_temp "
			"VALUES ( '", row[0], "', '", row[1], "', '", 
			row[2], "', ", "'", new_amount, "', '", 
			new_msre_no, "', '", row[5], "' )", NULL);
		success = no_ret_val_query_db( query);
		g_assert( success);
		g_free( query);

		for( i=0; i<6; i++) g_free( row[i]);
		g_free( row);
	} else {
		g_return_if_fail ( new_no_portions);

		query = g_strconcat( "SELECT * FROM recipe_plan_temp WHERE ",
			"date = '", date, "' AND ",
			"time = '", time_of_day, "' AND ",
			"recipe_no = '", hidden_data, "'", NULL);
		row = row_ret_val_query_db( query);
		g_assert( row);
		g_free( query);

		query = g_strconcat( "DELETE FROM recipe_plan_temp WHERE ",
			"date = '", date, "' AND ",
			"time = '", time_of_day, "' AND ",
			"recipe_no = '", hidden_data, "'", NULL);
		success = no_ret_val_query_db( query);
		g_assert( success);
		g_free( query);

		query = g_strconcat( "INSERT INTO recipe_plan_temp "
			"VALUES ( '", row[0], "', '", row[1], "', '", 
			row[2], "', ", "'", new_no_portions, "', '", 
			row[4], "')", NULL);
		success = no_ret_val_query_db( query);
		g_assert( success);
		g_free( query);

		for( i=0; i<5; i++) g_free( row[i]);
		g_free( row);
	}
}
