/*
    Copyright (C) 2004  Dale Mellor

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License,  or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#include "guile-hooks.h"


#include <stdlib.h>
#include <memory.h>
#include <dirent.h>
#include <stdio.h>
#include <pwd.h>
#include <unistd.h>

#include "cube.h"
#include "ui.h"
#include "widget-set.h"


/* The head of a singly linked list of menu items that the guile scripts request
   as they are executed. */

static Menu_Item_List *list_head = NULL;


/* When a script runs,  the first cube movement that is requested by the script
   must flush the move queue after the place of current insertion; further move
   requests must be appended to the end of the move queue. This variable is used
   to track the first request. */

static int moved;




/* This function is called from the menu when the user makes a selection. The
   data is a string which was registered with the menu system and gives the name
   of a scheme procedure to execute. */

static void run_scheme (gpointer data,  guint action,  GtkWidget *widget)
{
    moved = 0;

    char *buffer = (char*) malloc (strlen (data) + 3);

    sprintf (buffer,  "(%s)",  (char*) data);

    scm_eval_string (scm_makfrom0str (buffer));

    free (buffer);

    request_play ();
}




/* Function callable from scheme (as gnubik-register-script) which allows a
   script to specify a menu entry and the name of a procedure to call when that
   menu entry is selected. Note that /Script-fu/ is always appended,  so all
   scripts are forced under the Script-fu main menu item. This function prepends
   a new item to the head of the linked list,  with the data necessary for the
   menu system to function. */

static SCM gnubik_register_script (SCM menu_location,
                                   SCM callback)
{
    const int string_length = scm_c_string_length (menu_location);
    const char menu_path_prefix [] = "/Script-_fu/";
    Menu_Item_List *new_entry;
    char *ml = scm_to_locale_string (menu_location);
    char *const buffer
        = (char*) malloc (strlen (menu_path_prefix) + string_length + 1);


    strcpy (buffer,  menu_path_prefix);
    strcat (buffer,  ml);
    free (ml);

    new_entry = (Menu_Item_List*) malloc (sizeof (Menu_Item_List));

    new_entry->entry.path = buffer;
    new_entry->entry.accelerator = 0;
    new_entry->entry.callback = run_scheme;
    new_entry->entry.callback_action = 1;
    new_entry->entry.item_type = NULL;
    new_entry->callback_data = scm_to_locale_string (callback);
    new_entry->next = list_head;

    list_head = new_entry;

    return SCM_UNSPECIFIED;
}




/* Function callable from scheme as gnubik-cube-state which returns a structure
   reflecting the current state of the cube. */

static SCM gnubik_cube_state ()
{
    return make_scm_cube (the_cube);
}




/* The first time a script makes a move on the cube,  the move_queue must be
   truncated to the current place,  and the place is marked so that the viewer
   can rewind the effects of the script. This script performs the necessary
   preparations. */

static void start_moves_if_first ()
{
    if (! moved)
    {
        moved = 1;
        request_truncate_move_queue ();
        request_mark_move_queue ();
    }
}




/* Function which,  when called from scheme as gnubik-rotate-animated,  causes one
   side of the cube to rotate on-screen. */

static SCM gnubik_rotate_animated (SCM list)
{
    start_moves_if_first ();

    for (; ! SCM_NULLP (list); list = SCM_CDR (list))
    {
        Move_Data move;

        move.axis = scm_to_int (SCM_CAAR (list));
        move.slice = scm_to_int (SCM_CADAR (list));
        move.dir = scm_to_int (SCM_CADDAR (list));

        request_delayed_rotation (&move);
    }

    return SCM_UNSPECIFIED;
}



/* Function allowing a script to apply all its moves in one go to the cube,
   without creating animations on the display. */

static SCM gnubik_run_moves ()
{
    request_fast_forward ();

    return SCM_UNSPECIFIED;
}




/* Function to allow a guile script to display a message to the user. */

static SCM gnubik_error_dialog (SCM message)
{
    char *msg = scm_to_locale_string (message);
    error_dialog (main_application_window,  msg);
    free (msg);

    return SCM_UNSPECIFIED;
}




/* Function to scan the named directory for all files with a .scm extension,  and
   execute the contents of each file. */
static void 
read_script_directory (const char *const dir_name)
{
    static char buffer [1024];

    DIR *const directory = opendir (dir_name);

    if (directory)
    {
        struct dirent *entry;

        for (entry = readdir (directory); entry; entry = readdir (directory))

            if (strcmp (".scm",
                        entry->d_name + strlen (entry->d_name) - 4)
                     == 0)
            {
                snprintf (buffer,  1024,  "%s/%s",  dir_name,  entry->d_name);

                scm_primitive_load (scm_makfrom0str (buffer));
            }
    }
}




/* This function initializes the scheme world for us,  and once the scripts have
   all been run,  it returns the requested menu structure to the caller. Before
   running the scripts,  however,  it first makes sure all the pertinent C
   functions are registered in the guile world. */

Menu_Item_List *
startup_guile_scripts (void)
{
    static const char local_dir [] = "/.gnubik/scripts";
    const char *home_dir;
    char *buffer;


    /* Register C functions that the scheme world can access. */

    scm_c_define_gsubr ("gnubik-register-script",
                        2,  0,  0,
                        gnubik_register_script);

    scm_c_define_gsubr ("gnubik-cube-state",
                        0,  0,  0,
                        gnubik_cube_state);

    scm_c_define_gsubr ("gnubik-rotate-animated",
                        1,  0,  0,
                        gnubik_rotate_animated);

    scm_c_define_gsubr ("gnubik-run-moves",
                        0,  0,  0,
                        gnubik_run_moves);

    scm_c_define_gsubr ("gnubik-error-dialog",
                        1,  0,  0,
                        gnubik_error_dialog);


    /* Run all the initialization files in .../share/gnubik/guile,  and the
       system scripts in .../share/gnubik/scripts. */

    read_script_directory (GUILEDIR);
    read_script_directory (SCRIPTDIR);


    /* Run all the user scripts in $(HOME)/.gnubik/scripts. */

    home_dir = getenv ("HOME");

#if HAVE_GETPWUID
    if (home_dir == NULL)
        home_dir = getpwuid (getuid ())->pw_dir;
#endif

    if (home_dir)
      {
	buffer = (char*) malloc (strlen (home_dir) + strlen (local_dir) + 1);
	strcpy (buffer,  home_dir);
	strcat (buffer,  local_dir);

	read_script_directory (buffer);

	free (buffer);
      }


    /* Hopefully the scripts that we have run will have called
       gnubik_register_script above to populate the list_head,  which we now
       return to the calling application which can in turn make the menu entries
       that run the scripts. */

    return list_head;
}
