/* Mapping tables for CNS 11643, plane 2 handling.
   Copyright (C) 1998 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Ulrich Drepper <drepper@cygnus.com>, 1998.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdint.h>

/* The following table contains no whole and is therefore compact.  Nice
   work compared with the chaos of other character sets (or plane 1).

   The table can be generated using

   egrep '^0x2' .../mappings/eastasia/other/cns11643.txt |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0x22121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint16_t __cns11643l2_to_ucs4_tab[] =
{
  [0x0000] = 0x4e42, [0x0001] = 0x4e5c, [0x0002] = 0x51f5, [0x0003] = 0x531a,
  [0x0004] = 0x5382, [0x0005] = 0x4e07, [0x0006] = 0x4e0c, [0x0007] = 0x4e47,
  [0x0008] = 0x4e8d, [0x0009] = 0x56d7, [0x000a] = 0x5c6e, [0x000b] = 0x5f73,
  [0x000c] = 0x4e0f, [0x000d] = 0x5187, [0x000e] = 0x4e0e, [0x000f] = 0x4e2e,
  [0x0010] = 0x4e93, [0x0011] = 0x4ec2, [0x0012] = 0x4ec9, [0x0013] = 0x4ec8,
  [0x0014] = 0x5198, [0x0015] = 0x52fc, [0x0016] = 0x536c, [0x0017] = 0x53b9,
  [0x0018] = 0x5720, [0x0019] = 0x5903, [0x001a] = 0x592c, [0x001b] = 0x5c10,
  [0x001c] = 0x5dff, [0x001d] = 0x65e1, [0x001e] = 0x6bb3, [0x001f] = 0x6bcc,
  [0x0020] = 0x6c14, [0x0021] = 0x723f, [0x0022] = 0x4e31, [0x0023] = 0x4e3c,
  [0x0024] = 0x4ee8, [0x0025] = 0x4edc, [0x0026] = 0x4ee9, [0x0027] = 0x4ee1,
  [0x0028] = 0x4edd, [0x0029] = 0x4eda, [0x002a] = 0x520c, [0x002b] = 0x5209,
  [0x002c] = 0x531c, [0x002d] = 0x534c, [0x002e] = 0x5722, [0x002f] = 0x5723,
  [0x0030] = 0x5917, [0x0031] = 0x592f, [0x0032] = 0x5b81, [0x0033] = 0x5b84,
  [0x0034] = 0x5c12, [0x0035] = 0x5c3b, [0x0036] = 0x5c74, [0x0037] = 0x5c73,
  [0x0038] = 0x5e04, [0x0039] = 0x5e80, [0x003a] = 0x5e82, [0x003b] = 0x5fc9,
  [0x003c] = 0x6209, [0x003d] = 0x6250, [0x003e] = 0x6c15, [0x003f] = 0x6c36,
  [0x0040] = 0x6c43, [0x0041] = 0x6c3f, [0x0042] = 0x6c3b, [0x0043] = 0x72ae,
  [0x0044] = 0x72b0, [0x0045] = 0x738a, [0x0046] = 0x79b8, [0x0047] = 0x808a,
  [0x0048] = 0x961e, [0x0049] = 0x4f0e, [0x004a] = 0x4f18, [0x004b] = 0x4f2c,
  [0x004c] = 0x4ef5, [0x004d] = 0x4f14, [0x004e] = 0x4ef1, [0x004f] = 0x4f00,
  [0x0050] = 0x4ef7, [0x0051] = 0x4f08, [0x0052] = 0x4f1d, [0x0053] = 0x4f02,
  [0x0054] = 0x4f05, [0x0055] = 0x4f22, [0x0056] = 0x4f13, [0x0057] = 0x4f04,
  [0x0058] = 0x4ef4, [0x0059] = 0x4f12, [0x005a] = 0x51b1, [0x005b] = 0x5213,
  [0x005c] = 0x5210, [0x005d] = 0x52a6, [0x005e] = 0x5322, [0x005f] = 0x531f,
  [0x0060] = 0x534d, [0x0061] = 0x538a, [0x0062] = 0x5407, [0x0063] = 0x56e1,
  [0x0064] = 0x56df, [0x0065] = 0x572e, [0x0066] = 0x572a, [0x0067] = 0x5734,
  [0x0068] = 0x593c, [0x0069] = 0x5980, [0x006a] = 0x597c, [0x006b] = 0x5985,
  [0x006c] = 0x597b, [0x006d] = 0x597e, [0x006e] = 0x5977, [0x006f] = 0x597f,
  [0x0070] = 0x5b56, [0x0071] = 0x5c15, [0x0072] = 0x5c25, [0x0073] = 0x5c7c,
  [0x0074] = 0x5c7a, [0x0075] = 0x5c7b, [0x0076] = 0x5c7e, [0x0077] = 0x5ddf,
  [0x0078] = 0x5e75, [0x0079] = 0x5e84, [0x007a] = 0x5f02, [0x007b] = 0x5f1a,
  [0x007c] = 0x5f74, [0x007d] = 0x5fd5, [0x007e] = 0x5fd4, [0x007f] = 0x5fcf,
  [0x0080] = 0x625c, [0x0081] = 0x625e, [0x0082] = 0x6264, [0x0083] = 0x6261,
  [0x0084] = 0x6266, [0x0085] = 0x6262, [0x0086] = 0x6259, [0x0087] = 0x6260,
  [0x0088] = 0x625a, [0x0089] = 0x6265, [0x008a] = 0x6537, [0x008b] = 0x65ef,
  [0x008c] = 0x65ee, [0x008d] = 0x673e, [0x008e] = 0x6739, [0x008f] = 0x6738,
  [0x0090] = 0x673b, [0x0091] = 0x673a, [0x0092] = 0x673f, [0x0093] = 0x673c,
  [0x0094] = 0x6733, [0x0095] = 0x6c18, [0x0096] = 0x6c46, [0x0097] = 0x6c52,
  [0x0098] = 0x6c5c, [0x0099] = 0x6c4f, [0x009a] = 0x6c4a, [0x009b] = 0x6c54,
  [0x009c] = 0x6c4b, [0x009d] = 0x6c4c, [0x009e] = 0x7071, [0x009f] = 0x725e,
  [0x00a0] = 0x72b4, [0x00a1] = 0x72b5, [0x00a2] = 0x738e, [0x00a3] = 0x752a,
  [0x00a4] = 0x767f, [0x00a5] = 0x7a75, [0x00a6] = 0x7f51, [0x00a7] = 0x8278,
  [0x00a8] = 0x827c, [0x00a9] = 0x8280, [0x00aa] = 0x827d, [0x00ab] = 0x827f,
  [0x00ac] = 0x864d, [0x00ad] = 0x897e, [0x00ae] = 0x9099, [0x00af] = 0x9097,
  [0x00b0] = 0x9098, [0x00b1] = 0x909b, [0x00b2] = 0x9094, [0x00b3] = 0x9622,
  [0x00b4] = 0x9624, [0x00b5] = 0x9620, [0x00b6] = 0x9623, [0x00b7] = 0x4f56,
  [0x00b8] = 0x4f3b, [0x00b9] = 0x4f62, [0x00ba] = 0x4f49, [0x00bb] = 0x4f53,
  [0x00bc] = 0x4f64, [0x00bd] = 0x4f3e, [0x00be] = 0x4f67, [0x00bf] = 0x4f52,
  [0x00c0] = 0x4f5f, [0x00c1] = 0x4f41, [0x00c2] = 0x4f58, [0x00c3] = 0x4f2d,
  [0x00c4] = 0x4f33, [0x00c5] = 0x4f3f, [0x00c6] = 0x4f61, [0x00c7] = 0x518f,
  [0x00c8] = 0x51b9, [0x00c9] = 0x521c, [0x00ca] = 0x521e, [0x00cb] = 0x5221,
  [0x00cc] = 0x52ad, [0x00cd] = 0x52ae, [0x00ce] = 0x5309, [0x00cf] = 0x5363,
  [0x00d0] = 0x5372, [0x00d1] = 0x538e, [0x00d2] = 0x538f, [0x00d3] = 0x5430,
  [0x00d4] = 0x5437, [0x00d5] = 0x542a, [0x00d6] = 0x5454, [0x00d7] = 0x5445,
  [0x00d8] = 0x5419, [0x00d9] = 0x541c, [0x00da] = 0x5425, [0x00db] = 0x5418,
  [0x00dc] = 0x543d, [0x00dd] = 0x544f, [0x00de] = 0x5441, [0x00df] = 0x5428,
  [0x00e0] = 0x5424, [0x00e1] = 0x5447, [0x00e2] = 0x56ee, [0x00e3] = 0x56e7,
  [0x00e4] = 0x56e5, [0x00e5] = 0x5741, [0x00e6] = 0x5745, [0x00e7] = 0x574c,
  [0x00e8] = 0x5749, [0x00e9] = 0x574b, [0x00ea] = 0x5752, [0x00eb] = 0x5906,
  [0x00ec] = 0x5940, [0x00ed] = 0x59a6, [0x00ee] = 0x5998, [0x00ef] = 0x59a0,
  [0x00f0] = 0x5997, [0x00f1] = 0x598e, [0x00f2] = 0x59a2, [0x00f3] = 0x5990,
  [0x00f4] = 0x598f, [0x00f5] = 0x59a7, [0x00f6] = 0x59a1, [0x00f7] = 0x5b8e,
  [0x00f8] = 0x5b92, [0x00f9] = 0x5c28, [0x00fa] = 0x5c2a, [0x00fb] = 0x5c8d,
  [0x00fc] = 0x5c8f, [0x00fd] = 0x5c88, [0x00fe] = 0x5c8b, [0x00ff] = 0x5c89,
  [0x0100] = 0x5c92, [0x0101] = 0x5c8a, [0x0102] = 0x5c86, [0x0103] = 0x5c93,
  [0x0104] = 0x5c95, [0x0105] = 0x5de0, [0x0106] = 0x5e0a, [0x0107] = 0x5e0e,
  [0x0108] = 0x5e8b, [0x0109] = 0x5e89, [0x010a] = 0x5e8c, [0x010b] = 0x5e88,
  [0x010c] = 0x5e8d, [0x010d] = 0x5f05, [0x010e] = 0x5f1d, [0x010f] = 0x5f78,
  [0x0110] = 0x5f76, [0x0111] = 0x5fd2, [0x0112] = 0x5fd1, [0x0113] = 0x5fd0,
  [0x0114] = 0x5fed, [0x0115] = 0x5fe8, [0x0116] = 0x5fee, [0x0117] = 0x5ff3,
  [0x0118] = 0x5fe1, [0x0119] = 0x5fe4, [0x011a] = 0x5fe3, [0x011b] = 0x5ffa,
  [0x011c] = 0x5fef, [0x011d] = 0x5ff7, [0x011e] = 0x5ffb, [0x011f] = 0x6000,
  [0x0120] = 0x5ff4, [0x0121] = 0x623a, [0x0122] = 0x6283, [0x0123] = 0x628c,
  [0x0124] = 0x628e, [0x0125] = 0x628f, [0x0126] = 0x6294, [0x0127] = 0x6287,
  [0x0128] = 0x6271, [0x0129] = 0x627b, [0x012a] = 0x627a, [0x012b] = 0x6270,
  [0x012c] = 0x6281, [0x012d] = 0x6288, [0x012e] = 0x6277, [0x012f] = 0x627d,
  [0x0130] = 0x6272, [0x0131] = 0x6274, [0x0132] = 0x65f0, [0x0133] = 0x65f4,
  [0x0134] = 0x65f3, [0x0135] = 0x65f2, [0x0136] = 0x65f5, [0x0137] = 0x6745,
  [0x0138] = 0x6747, [0x0139] = 0x6759, [0x013a] = 0x6755, [0x013b] = 0x674c,
  [0x013c] = 0x6748, [0x013d] = 0x675d, [0x013e] = 0x674d, [0x013f] = 0x675a,
  [0x0140] = 0x674b, [0x0141] = 0x6bd0, [0x0142] = 0x6c19, [0x0143] = 0x6c1a,
  [0x0144] = 0x6c78, [0x0145] = 0x6c67, [0x0146] = 0x6c6b, [0x0147] = 0x6c84,
  [0x0148] = 0x6c8b, [0x0149] = 0x6c8f, [0x014a] = 0x6c71, [0x014b] = 0x6c6f,
  [0x014c] = 0x6c69, [0x014d] = 0x6c9a, [0x014e] = 0x6c6d, [0x014f] = 0x6c87,
  [0x0150] = 0x6c95, [0x0151] = 0x6c9c, [0x0152] = 0x6c66, [0x0153] = 0x6c73,
  [0x0154] = 0x6c65, [0x0155] = 0x6c7b, [0x0156] = 0x6c8e, [0x0157] = 0x7074,
  [0x0158] = 0x707a, [0x0159] = 0x7263, [0x015a] = 0x72bf, [0x015b] = 0x72bd,
  [0x015c] = 0x72c3, [0x015d] = 0x72c6, [0x015e] = 0x72c1, [0x015f] = 0x72ba,
  [0x0160] = 0x72c5, [0x0161] = 0x7395, [0x0162] = 0x7397, [0x0163] = 0x7393,
  [0x0164] = 0x7394, [0x0165] = 0x7392, [0x0166] = 0x753a, [0x0167] = 0x7539,
  [0x0168] = 0x7594, [0x0169] = 0x7595, [0x016a] = 0x7681, [0x016b] = 0x793d,
  [0x016c] = 0x8034, [0x016d] = 0x8095, [0x016e] = 0x8099, [0x016f] = 0x8090,
  [0x0170] = 0x8092, [0x0171] = 0x809c, [0x0172] = 0x8290, [0x0173] = 0x828f,
  [0x0174] = 0x8285, [0x0175] = 0x828e, [0x0176] = 0x8291, [0x0177] = 0x8293,
  [0x0178] = 0x828a, [0x0179] = 0x8283, [0x017a] = 0x8284, [0x017b] = 0x8c78,
  [0x017c] = 0x8fc9, [0x017d] = 0x8fbf, [0x017e] = 0x909f, [0x017f] = 0x90a1,
  [0x0180] = 0x90a5, [0x0181] = 0x909e, [0x0182] = 0x90a7, [0x0183] = 0x90a0,
  [0x0184] = 0x9630, [0x0185] = 0x9628, [0x0186] = 0x962f, [0x0187] = 0x962d,
  [0x0188] = 0x4e33, [0x0189] = 0x4f98, [0x018a] = 0x4f7c, [0x018b] = 0x4f85,
  [0x018c] = 0x4f7d, [0x018d] = 0x4f80, [0x018e] = 0x4f87, [0x018f] = 0x4f76,
  [0x0190] = 0x4f74, [0x0191] = 0x4f89, [0x0192] = 0x4f84, [0x0193] = 0x4f77,
  [0x0194] = 0x4f4c, [0x0195] = 0x4f97, [0x0196] = 0x4f6a, [0x0197] = 0x4f9a,
  [0x0198] = 0x4f79, [0x0199] = 0x4f81, [0x019a] = 0x4f78, [0x019b] = 0x4f90,
  [0x019c] = 0x4f9c, [0x019d] = 0x4f94, [0x019e] = 0x4f9e, [0x019f] = 0x4f92,
  [0x01a0] = 0x4f82, [0x01a1] = 0x4f95, [0x01a2] = 0x4f6b, [0x01a3] = 0x4f6e,
  [0x01a4] = 0x519e, [0x01a5] = 0x51bc, [0x01a6] = 0x51be, [0x01a7] = 0x5235,
  [0x01a8] = 0x5232, [0x01a9] = 0x5233, [0x01aa] = 0x5246, [0x01ab] = 0x5231,
  [0x01ac] = 0x52bc, [0x01ad] = 0x530a, [0x01ae] = 0x530b, [0x01af] = 0x533c,
  [0x01b0] = 0x5392, [0x01b1] = 0x5394, [0x01b2] = 0x5487, [0x01b3] = 0x547f,
  [0x01b4] = 0x5481, [0x01b5] = 0x5491, [0x01b6] = 0x5482, [0x01b7] = 0x5488,
  [0x01b8] = 0x546b, [0x01b9] = 0x547a, [0x01ba] = 0x547e, [0x01bb] = 0x5465,
  [0x01bc] = 0x546c, [0x01bd] = 0x5474, [0x01be] = 0x5466, [0x01bf] = 0x548d,
  [0x01c0] = 0x546f, [0x01c1] = 0x5461, [0x01c2] = 0x5460, [0x01c3] = 0x5498,
  [0x01c4] = 0x5463, [0x01c5] = 0x5467, [0x01c6] = 0x5464, [0x01c7] = 0x56f7,
  [0x01c8] = 0x56f9, [0x01c9] = 0x576f, [0x01ca] = 0x5772, [0x01cb] = 0x576d,
  [0x01cc] = 0x576b, [0x01cd] = 0x5771, [0x01ce] = 0x5770, [0x01cf] = 0x5776,
  [0x01d0] = 0x5780, [0x01d1] = 0x5775, [0x01d2] = 0x577b, [0x01d3] = 0x5773,
  [0x01d4] = 0x5774, [0x01d5] = 0x5762, [0x01d6] = 0x5768, [0x01d7] = 0x577d,
  [0x01d8] = 0x590c, [0x01d9] = 0x5945, [0x01da] = 0x59b5, [0x01db] = 0x59ba,
  [0x01dc] = 0x59cf, [0x01dd] = 0x59ce, [0x01de] = 0x59b2, [0x01df] = 0x59cc,
  [0x01e0] = 0x59c1, [0x01e1] = 0x59b6, [0x01e2] = 0x59bc, [0x01e3] = 0x59c3,
  [0x01e4] = 0x59d6, [0x01e5] = 0x59b1, [0x01e6] = 0x59bd, [0x01e7] = 0x59c0,
  [0x01e8] = 0x59c8, [0x01e9] = 0x59b4, [0x01ea] = 0x59c7, [0x01eb] = 0x5b62,
  [0x01ec] = 0x5b65, [0x01ed] = 0x5b93, [0x01ee] = 0x5b95, [0x01ef] = 0x5c44,
  [0x01f0] = 0x5c47, [0x01f1] = 0x5cae, [0x01f2] = 0x5ca4, [0x01f3] = 0x5ca0,
  [0x01f4] = 0x5cb5, [0x01f5] = 0x5caf, [0x01f6] = 0x5ca8, [0x01f7] = 0x5cac,
  [0x01f8] = 0x5c9f, [0x01f9] = 0x5ca3, [0x01fa] = 0x5cad, [0x01fb] = 0x5ca2,
  [0x01fc] = 0x5caa, [0x01fd] = 0x5ca7, [0x01fe] = 0x5c9d, [0x01ff] = 0x5ca5,
  [0x0200] = 0x5cb6, [0x0201] = 0x5cb0, [0x0202] = 0x5ca6, [0x0203] = 0x5e17,
  [0x0204] = 0x5e14, [0x0205] = 0x5e19, [0x0206] = 0x5f28, [0x0207] = 0x5f22,
  [0x0208] = 0x5f23, [0x0209] = 0x5f24, [0x020a] = 0x5f54, [0x020b] = 0x5f82,
  [0x020c] = 0x5f7e, [0x020d] = 0x5f7d, [0x020e] = 0x5fde, [0x020f] = 0x5fe5,
  [0x0210] = 0x602d, [0x0211] = 0x6026, [0x0212] = 0x6019, [0x0213] = 0x6032,
  [0x0214] = 0x600b, [0x0215] = 0x6034, [0x0216] = 0x600a, [0x0217] = 0x6017,
  [0x0218] = 0x6033, [0x0219] = 0x601a, [0x021a] = 0x601e, [0x021b] = 0x602c,
  [0x021c] = 0x6022, [0x021d] = 0x600d, [0x021e] = 0x6010, [0x021f] = 0x602e,
  [0x0220] = 0x6013, [0x0221] = 0x6011, [0x0222] = 0x600c, [0x0223] = 0x6009,
  [0x0224] = 0x601c, [0x0225] = 0x6214, [0x0226] = 0x623d, [0x0227] = 0x62ad,
  [0x0228] = 0x62b4, [0x0229] = 0x62d1, [0x022a] = 0x62be, [0x022b] = 0x62aa,
  [0x022c] = 0x62b6, [0x022d] = 0x62ca, [0x022e] = 0x62ae, [0x022f] = 0x62b3,
  [0x0230] = 0x62af, [0x0231] = 0x62bb, [0x0232] = 0x62a9, [0x0233] = 0x62b0,
  [0x0234] = 0x62b8, [0x0235] = 0x653d, [0x0236] = 0x65a8, [0x0237] = 0x65bb,
  [0x0238] = 0x6609, [0x0239] = 0x65fc, [0x023a] = 0x6604, [0x023b] = 0x6612,
  [0x023c] = 0x6608, [0x023d] = 0x65fb, [0x023e] = 0x6603, [0x023f] = 0x660b,
  [0x0240] = 0x660d, [0x0241] = 0x6605, [0x0242] = 0x65fd, [0x0243] = 0x6611,
  [0x0244] = 0x6610, [0x0245] = 0x66f6, [0x0246] = 0x670a, [0x0247] = 0x6785,
  [0x0248] = 0x676c, [0x0249] = 0x678e, [0x024a] = 0x6792, [0x024b] = 0x6776,
  [0x024c] = 0x677b, [0x024d] = 0x6798, [0x024e] = 0x6786, [0x024f] = 0x6784,
  [0x0250] = 0x6774, [0x0251] = 0x678d, [0x0252] = 0x678c, [0x0253] = 0x677a,
  [0x0254] = 0x679f, [0x0255] = 0x6791, [0x0256] = 0x6799, [0x0257] = 0x6783,
  [0x0258] = 0x677d, [0x0259] = 0x6781, [0x025a] = 0x6778, [0x025b] = 0x6779,
  [0x025c] = 0x6794, [0x025d] = 0x6b25, [0x025e] = 0x6b80, [0x025f] = 0x6b7e,
  [0x0260] = 0x6bde, [0x0261] = 0x6c1d, [0x0262] = 0x6c93, [0x0263] = 0x6cec,
  [0x0264] = 0x6ceb, [0x0265] = 0x6cee, [0x0266] = 0x6cd9, [0x0267] = 0x6cb6,
  [0x0268] = 0x6cd4, [0x0269] = 0x6cad, [0x026a] = 0x6ce7, [0x026b] = 0x6cb7,
  [0x026c] = 0x6cd0, [0x026d] = 0x6cc2, [0x026e] = 0x6cba, [0x026f] = 0x6cc3,
  [0x0270] = 0x6cc6, [0x0271] = 0x6ced, [0x0272] = 0x6cf2, [0x0273] = 0x6cd2,
  [0x0274] = 0x6cdd, [0x0275] = 0x6cb4, [0x0276] = 0x6c8a, [0x0277] = 0x6c9d,
  [0x0278] = 0x6c80, [0x0279] = 0x6cde, [0x027a] = 0x6cc0, [0x027b] = 0x6d30,
  [0x027c] = 0x6ccd, [0x027d] = 0x6cc7, [0x027e] = 0x6cb0, [0x027f] = 0x6cf9,
  [0x0280] = 0x6ccf, [0x0281] = 0x6ce9, [0x0282] = 0x6cd1, [0x0283] = 0x7094,
  [0x0284] = 0x7098, [0x0285] = 0x7085, [0x0286] = 0x7093, [0x0287] = 0x7086,
  [0x0288] = 0x7084, [0x0289] = 0x7091, [0x028a] = 0x7096, [0x028b] = 0x7082,
  [0x028c] = 0x709a, [0x028d] = 0x7083, [0x028e] = 0x726a, [0x028f] = 0x72d6,
  [0x0290] = 0x72cb, [0x0291] = 0x72d8, [0x0292] = 0x72c9, [0x0293] = 0x72dc,
  [0x0294] = 0x72d2, [0x0295] = 0x72d4, [0x0296] = 0x72da, [0x0297] = 0x72cc,
  [0x0298] = 0x72d1, [0x0299] = 0x73a4, [0x029a] = 0x73a1, [0x029b] = 0x73ad,
  [0x029c] = 0x73a6, [0x029d] = 0x73a2, [0x029e] = 0x73a0, [0x029f] = 0x73ac,
  [0x02a0] = 0x739d, [0x02a1] = 0x74dd, [0x02a2] = 0x74e8, [0x02a3] = 0x753f,
  [0x02a4] = 0x7540, [0x02a5] = 0x753e, [0x02a6] = 0x758c, [0x02a7] = 0x7598,
  [0x02a8] = 0x76af, [0x02a9] = 0x76f3, [0x02aa] = 0x76f1, [0x02ab] = 0x76f0,
  [0x02ac] = 0x76f5, [0x02ad] = 0x77f8, [0x02ae] = 0x77fc, [0x02af] = 0x77f9,
  [0x02b0] = 0x77fb, [0x02b1] = 0x77fa, [0x02b2] = 0x77f7, [0x02b3] = 0x7942,
  [0x02b4] = 0x793f, [0x02b5] = 0x79c5, [0x02b6] = 0x7a78, [0x02b7] = 0x7a7b,
  [0x02b8] = 0x7afb, [0x02b9] = 0x7c75, [0x02ba] = 0x7cfd, [0x02bb] = 0x8035,
  [0x02bc] = 0x808f, [0x02bd] = 0x80ae, [0x02be] = 0x80a3, [0x02bf] = 0x80b8,
  [0x02c0] = 0x80b5, [0x02c1] = 0x80ad, [0x02c2] = 0x8220, [0x02c3] = 0x82a0,
  [0x02c4] = 0x82c0, [0x02c5] = 0x82ab, [0x02c6] = 0x829a, [0x02c7] = 0x8298,
  [0x02c8] = 0x829b, [0x02c9] = 0x82b5, [0x02ca] = 0x82a7, [0x02cb] = 0x82ae,
  [0x02cc] = 0x82bc, [0x02cd] = 0x829e, [0x02ce] = 0x82ba, [0x02cf] = 0x82b4,
  [0x02d0] = 0x82a8, [0x02d1] = 0x82a1, [0x02d2] = 0x82a9, [0x02d3] = 0x82c2,
  [0x02d4] = 0x82a4, [0x02d5] = 0x82c3, [0x02d6] = 0x82b6, [0x02d7] = 0x82a2,
  [0x02d8] = 0x8670, [0x02d9] = 0x866f, [0x02da] = 0x866d, [0x02db] = 0x866e,
  [0x02dc] = 0x8c56, [0x02dd] = 0x8fd2, [0x02de] = 0x8fcb, [0x02df] = 0x8fd3,
  [0x02e0] = 0x8fcd, [0x02e1] = 0x8fd6, [0x02e2] = 0x8fd5, [0x02e3] = 0x8fd7,
  [0x02e4] = 0x90b2, [0x02e5] = 0x90b4, [0x02e6] = 0x90af, [0x02e7] = 0x90b3,
  [0x02e8] = 0x90b0, [0x02e9] = 0x9639, [0x02ea] = 0x963d, [0x02eb] = 0x963c,
  [0x02ec] = 0x963a, [0x02ed] = 0x9643, [0x02ee] = 0x4fcd, [0x02ef] = 0x4fc5,
  [0x02f0] = 0x4fd3, [0x02f1] = 0x4fb2, [0x02f2] = 0x4fc9, [0x02f3] = 0x4fcb,
  [0x02f4] = 0x4fc1, [0x02f5] = 0x4fd4, [0x02f6] = 0x4fdc, [0x02f7] = 0x4fd9,
  [0x02f8] = 0x4fbb, [0x02f9] = 0x4fb3, [0x02fa] = 0x4fdb, [0x02fb] = 0x4fc7,
  [0x02fc] = 0x4fd6, [0x02fd] = 0x4fba, [0x02fe] = 0x4fc0, [0x02ff] = 0x4fb9,
  [0x0300] = 0x4fec, [0x0301] = 0x5244, [0x0302] = 0x5249, [0x0303] = 0x52c0,
  [0x0304] = 0x52c2, [0x0305] = 0x533d, [0x0306] = 0x537c, [0x0307] = 0x5397,
  [0x0308] = 0x5396, [0x0309] = 0x5399, [0x030a] = 0x5398, [0x030b] = 0x54ba,
  [0x030c] = 0x54a1, [0x030d] = 0x54ad, [0x030e] = 0x54a5, [0x030f] = 0x54cf,
  [0x0310] = 0x54c3, [0x0311] = 0x830d, [0x0312] = 0x54b7, [0x0313] = 0x54ae,
  [0x0314] = 0x54d6, [0x0315] = 0x54b6, [0x0316] = 0x54c5, [0x0317] = 0x54c6,
  [0x0318] = 0x54a0, [0x0319] = 0x5470, [0x031a] = 0x54bc, [0x031b] = 0x54a2,
  [0x031c] = 0x54be, [0x031d] = 0x5472, [0x031e] = 0x54de, [0x031f] = 0x54b0,
  [0x0320] = 0x57b5, [0x0321] = 0x579e, [0x0322] = 0x579f, [0x0323] = 0x57a4,
  [0x0324] = 0x578c, [0x0325] = 0x5797, [0x0326] = 0x579d, [0x0327] = 0x579b,
  [0x0328] = 0x5794, [0x0329] = 0x5798, [0x032a] = 0x578f, [0x032b] = 0x5799,
  [0x032c] = 0x57a5, [0x032d] = 0x579a, [0x032e] = 0x5795, [0x032f] = 0x58f4,
  [0x0330] = 0x590d, [0x0331] = 0x5953, [0x0332] = 0x59e1, [0x0333] = 0x59de,
  [0x0334] = 0x59ee, [0x0335] = 0x5a00, [0x0336] = 0x59f1, [0x0337] = 0x59dd,
  [0x0338] = 0x59fa, [0x0339] = 0x59fd, [0x033a] = 0x59fc, [0x033b] = 0x59f6,
  [0x033c] = 0x59e4, [0x033d] = 0x59f2, [0x033e] = 0x59f7, [0x033f] = 0x59db,
  [0x0340] = 0x59e9, [0x0341] = 0x59f3, [0x0342] = 0x59f5, [0x0343] = 0x59e0,
  [0x0344] = 0x59fe, [0x0345] = 0x59f4, [0x0346] = 0x59ed, [0x0347] = 0x5ba8,
  [0x0348] = 0x5c4c, [0x0349] = 0x5cd0, [0x034a] = 0x5cd8, [0x034b] = 0x5ccc,
  [0x034c] = 0x5cd7, [0x034d] = 0x5ccb, [0x034e] = 0x5cdb, [0x034f] = 0x5cde,
  [0x0350] = 0x5cda, [0x0351] = 0x5cc9, [0x0352] = 0x5cc7, [0x0353] = 0x5cca,
  [0x0354] = 0x5cd6, [0x0355] = 0x5cd3, [0x0356] = 0x5cd4, [0x0357] = 0x5ccf,
  [0x0358] = 0x5cc8, [0x0359] = 0x5cc6, [0x035a] = 0x5cce, [0x035b] = 0x5cdf,
  [0x035c] = 0x5cf8, [0x035d] = 0x5df9, [0x035e] = 0x5e21, [0x035f] = 0x5e22,
  [0x0360] = 0x5e23, [0x0361] = 0x5e20, [0x0362] = 0x5e24, [0x0363] = 0x5eb0,
  [0x0364] = 0x5ea4, [0x0365] = 0x5ea2, [0x0366] = 0x5e9b, [0x0367] = 0x5ea3,
  [0x0368] = 0x5ea5, [0x0369] = 0x5f07, [0x036a] = 0x5f2e, [0x036b] = 0x5f56,
  [0x036c] = 0x5f86, [0x036d] = 0x6037, [0x036e] = 0x6039, [0x036f] = 0x6054,
  [0x0370] = 0x6072, [0x0371] = 0x605e, [0x0372] = 0x6045, [0x0373] = 0x6053,
  [0x0374] = 0x6047, [0x0375] = 0x6049, [0x0376] = 0x605b, [0x0377] = 0x604c,
  [0x0378] = 0x6040, [0x0379] = 0x6042, [0x037a] = 0x605f, [0x037b] = 0x6024,
  [0x037c] = 0x6044, [0x037d] = 0x6058, [0x037e] = 0x6066, [0x037f] = 0x606e,
  [0x0380] = 0x6242, [0x0381] = 0x6243, [0x0382] = 0x62cf, [0x0383] = 0x630d,
  [0x0384] = 0x630b, [0x0385] = 0x62f5, [0x0386] = 0x630e, [0x0387] = 0x6303,
  [0x0388] = 0x62eb, [0x0389] = 0x62f9, [0x038a] = 0x630f, [0x038b] = 0x630c,
  [0x038c] = 0x62f8, [0x038d] = 0x62f6, [0x038e] = 0x6300, [0x038f] = 0x6313,
  [0x0390] = 0x6314, [0x0391] = 0x62fa, [0x0392] = 0x6315, [0x0393] = 0x62fb,
  [0x0394] = 0x62f0, [0x0395] = 0x6541, [0x0396] = 0x6543, [0x0397] = 0x65aa,
  [0x0398] = 0x65bf, [0x0399] = 0x6636, [0x039a] = 0x6621, [0x039b] = 0x6632,
  [0x039c] = 0x6635, [0x039d] = 0x661c, [0x039e] = 0x6626, [0x039f] = 0x6622,
  [0x03a0] = 0x6633, [0x03a1] = 0x662b, [0x03a2] = 0x663a, [0x03a3] = 0x661d,
  [0x03a4] = 0x6634, [0x03a5] = 0x6639, [0x03a6] = 0x662e, [0x03a7] = 0x670f,
  [0x03a8] = 0x6710, [0x03a9] = 0x67c1, [0x03aa] = 0x67f2, [0x03ab] = 0x67c8,
  [0x03ac] = 0x67ba, [0x03ad] = 0x67dc, [0x03ae] = 0x67bb, [0x03af] = 0x67f8,
  [0x03b0] = 0x67d8, [0x03b1] = 0x67c0, [0x03b2] = 0x67b7, [0x03b3] = 0x67c5,
  [0x03b4] = 0x67eb, [0x03b5] = 0x67e4, [0x03b6] = 0x67df, [0x03b7] = 0x67b5,
  [0x03b8] = 0x67cd, [0x03b9] = 0x67b3, [0x03ba] = 0x67f7, [0x03bb] = 0x67f6,
  [0x03bc] = 0x67ee, [0x03bd] = 0x67e3, [0x03be] = 0x67c2, [0x03bf] = 0x67b9,
  [0x03c0] = 0x67ce, [0x03c1] = 0x67e7, [0x03c2] = 0x67f0, [0x03c3] = 0x67b2,
  [0x03c4] = 0x67fc, [0x03c5] = 0x67c6, [0x03c6] = 0x67ed, [0x03c7] = 0x67cc,
  [0x03c8] = 0x67ae, [0x03c9] = 0x67e6, [0x03ca] = 0x67db, [0x03cb] = 0x67fa,
  [0x03cc] = 0x67c9, [0x03cd] = 0x67ca, [0x03ce] = 0x67c3, [0x03cf] = 0x67ea,
  [0x03d0] = 0x67cb, [0x03d1] = 0x6b28, [0x03d2] = 0x6b82, [0x03d3] = 0x6b84,
  [0x03d4] = 0x6bb6, [0x03d5] = 0x6bd6, [0x03d6] = 0x6bd8, [0x03d7] = 0x6be0,
  [0x03d8] = 0x6c20, [0x03d9] = 0x6c21, [0x03da] = 0x6d28, [0x03db] = 0x6d34,
  [0x03dc] = 0x6d2d, [0x03dd] = 0x6d1f, [0x03de] = 0x6d3c, [0x03df] = 0x6d3f,
  [0x03e0] = 0x6d12, [0x03e1] = 0x6d0a, [0x03e2] = 0x6cda, [0x03e3] = 0x6d33,
  [0x03e4] = 0x6d04, [0x03e5] = 0x6d19, [0x03e6] = 0x6d3a, [0x03e7] = 0x6d1a,
  [0x03e8] = 0x6d11, [0x03e9] = 0x6d00, [0x03ea] = 0x6d1d, [0x03eb] = 0x6d42,
  [0x03ec] = 0x6d01, [0x03ed] = 0x6d18, [0x03ee] = 0x6d37, [0x03ef] = 0x6d03,
  [0x03f0] = 0x6d0f, [0x03f1] = 0x6d40, [0x03f2] = 0x6d07, [0x03f3] = 0x6d20,
  [0x03f4] = 0x6d2c, [0x03f5] = 0x6d08, [0x03f6] = 0x6d22, [0x03f7] = 0x6d09,
  [0x03f8] = 0x6d10, [0x03f9] = 0x70b7, [0x03fa] = 0x709f, [0x03fb] = 0x70be,
  [0x03fc] = 0x70b1, [0x03fd] = 0x70b0, [0x03fe] = 0x70a1, [0x03ff] = 0x70b4,
  [0x0400] = 0x70b5, [0x0401] = 0x70a9, [0x0402] = 0x7241, [0x0403] = 0x7249,
  [0x0404] = 0x724a, [0x0405] = 0x726c, [0x0406] = 0x7270, [0x0407] = 0x7273,
  [0x0408] = 0x726e, [0x0409] = 0x72ca, [0x040a] = 0x72e4, [0x040b] = 0x72e8,
  [0x040c] = 0x72eb, [0x040d] = 0x72df, [0x040e] = 0x72ea, [0x040f] = 0x72e6,
  [0x0410] = 0x72e3, [0x0411] = 0x7385, [0x0412] = 0x73cc, [0x0413] = 0x73c2,
  [0x0414] = 0x73c8, [0x0415] = 0x73c5, [0x0416] = 0x73b9, [0x0417] = 0x73b6,
  [0x0418] = 0x73b5, [0x0419] = 0x73b4, [0x041a] = 0x73eb, [0x041b] = 0x73bf,
  [0x041c] = 0x73c7, [0x041d] = 0x73be, [0x041e] = 0x73c3, [0x041f] = 0x73c6,
  [0x0420] = 0x73b8, [0x0421] = 0x73cb, [0x0422] = 0x74ec, [0x0423] = 0x74ee,
  [0x0424] = 0x752e, [0x0425] = 0x7547, [0x0426] = 0x7548, [0x0427] = 0x75a7,
  [0x0428] = 0x75aa, [0x0429] = 0x7679, [0x042a] = 0x76c4, [0x042b] = 0x7708,
  [0x042c] = 0x7703, [0x042d] = 0x7704, [0x042e] = 0x7705, [0x042f] = 0x770a,
  [0x0430] = 0x76f7, [0x0431] = 0x76fb, [0x0432] = 0x76fa, [0x0433] = 0x77e7,
  [0x0434] = 0x77e8, [0x0435] = 0x7806, [0x0436] = 0x7811, [0x0437] = 0x7812,
  [0x0438] = 0x7805, [0x0439] = 0x7810, [0x043a] = 0x780f, [0x043b] = 0x780e,
  [0x043c] = 0x7809, [0x043d] = 0x7803, [0x043e] = 0x7813, [0x043f] = 0x794a,
  [0x0440] = 0x794c, [0x0441] = 0x794b, [0x0442] = 0x7945, [0x0443] = 0x7944,
  [0x0444] = 0x79d5, [0x0445] = 0x79cd, [0x0446] = 0x79cf, [0x0447] = 0x79d6,
  [0x0448] = 0x79ce, [0x0449] = 0x7a80, [0x044a] = 0x7a7e, [0x044b] = 0x7ad1,
  [0x044c] = 0x7b00, [0x044d] = 0x7b01, [0x044e] = 0x7c7a, [0x044f] = 0x7c78,
  [0x0450] = 0x7c79, [0x0451] = 0x7c7f, [0x0452] = 0x7c80, [0x0453] = 0x7c81,
  [0x0454] = 0x7d03, [0x0455] = 0x7d08, [0x0456] = 0x7d01, [0x0457] = 0x7f58,
  [0x0458] = 0x7f91, [0x0459] = 0x7f8d, [0x045a] = 0x7fbe, [0x045b] = 0x8007,
  [0x045c] = 0x800e, [0x045d] = 0x800f, [0x045e] = 0x8014, [0x045f] = 0x8037,
  [0x0460] = 0x80d8, [0x0461] = 0x80c7, [0x0462] = 0x80e0, [0x0463] = 0x80d1,
  [0x0464] = 0x80c8, [0x0465] = 0x80c2, [0x0466] = 0x80d0, [0x0467] = 0x80c5,
  [0x0468] = 0x80e3, [0x0469] = 0x80d9, [0x046a] = 0x80dc, [0x046b] = 0x80ca,
  [0x046c] = 0x80d5, [0x046d] = 0x80c9, [0x046e] = 0x80cf, [0x046f] = 0x80d7,
  [0x0470] = 0x80e6, [0x0471] = 0x80cd, [0x0472] = 0x81ff, [0x0473] = 0x8221,
  [0x0474] = 0x8294, [0x0475] = 0x82d9, [0x0476] = 0x82fe, [0x0477] = 0x82f9,
  [0x0478] = 0x8307, [0x0479] = 0x82e8, [0x047a] = 0x8300, [0x047b] = 0x82d5,
  [0x047c] = 0x833a, [0x047d] = 0x82eb, [0x047e] = 0x82d6, [0x047f] = 0x82f4,
  [0x0480] = 0x82ec, [0x0481] = 0x82e1, [0x0482] = 0x82f2, [0x0483] = 0x82f5,
  [0x0484] = 0x830c, [0x0485] = 0x82fb, [0x0486] = 0x82f6, [0x0487] = 0x82f0,
  [0x0488] = 0x82ea, [0x0489] = 0x82e4, [0x048a] = 0x82e0, [0x048b] = 0x82fa,
  [0x048c] = 0x82f3, [0x048d] = 0x82ed, [0x048e] = 0x8677, [0x048f] = 0x8674,
  [0x0490] = 0x867c, [0x0491] = 0x8673, [0x0492] = 0x8841, [0x0493] = 0x884e,
  [0x0494] = 0x8867, [0x0495] = 0x886a, [0x0496] = 0x8869, [0x0497] = 0x89d3,
  [0x0498] = 0x8a04, [0x0499] = 0x8a07, [0x049a] = 0x8d72, [0x049b] = 0x8fe3,
  [0x049c] = 0x8fe1, [0x049d] = 0x8fee, [0x049e] = 0x8fe0, [0x049f] = 0x90f1,
  [0x04a0] = 0x90bd, [0x04a1] = 0x90bf, [0x04a2] = 0x90d5, [0x04a3] = 0x90c5,
  [0x04a4] = 0x90be, [0x04a5] = 0x90c7, [0x04a6] = 0x90cb, [0x04a7] = 0x90c8,
  [0x04a8] = 0x91d4, [0x04a9] = 0x91d3, [0x04aa] = 0x9654, [0x04ab] = 0x964f,
  [0x04ac] = 0x9651, [0x04ad] = 0x9653, [0x04ae] = 0x964a, [0x04af] = 0x964e,
  [0x04b0] = 0x501e, [0x04b1] = 0x5005, [0x04b2] = 0x5007, [0x04b3] = 0x5013,
  [0x04b4] = 0x5022, [0x04b5] = 0x5030, [0x04b6] = 0x501b, [0x04b7] = 0x4ff5,
  [0x04b8] = 0x4ff4, [0x04b9] = 0x5033, [0x04ba] = 0x5037, [0x04bb] = 0x502c,
  [0x04bc] = 0x4ff6, [0x04bd] = 0x4ff7, [0x04be] = 0x5017, [0x04bf] = 0x501c,
  [0x04c0] = 0x5020, [0x04c1] = 0x5027, [0x04c2] = 0x5035, [0x04c3] = 0x502f,
  [0x04c4] = 0x5031, [0x04c5] = 0x500e, [0x04c6] = 0x515a, [0x04c7] = 0x5194,
  [0x04c8] = 0x5193, [0x04c9] = 0x51ca, [0x04ca] = 0x51c4, [0x04cb] = 0x51c5,
  [0x04cc] = 0x51c8, [0x04cd] = 0x51ce, [0x04ce] = 0x5261, [0x04cf] = 0x525a,
  [0x04d0] = 0x5252, [0x04d1] = 0x525e, [0x04d2] = 0x525f, [0x04d3] = 0x5255,
  [0x04d4] = 0x5262, [0x04d5] = 0x52cd, [0x04d6] = 0x530e, [0x04d7] = 0x539e,
  [0x04d8] = 0x5526, [0x04d9] = 0x54e2, [0x04da] = 0x5517, [0x04db] = 0x5512,
  [0x04dc] = 0x54e7, [0x04dd] = 0x54f3, [0x04de] = 0x54e4, [0x04df] = 0x551a,
  [0x04e0] = 0x54ff, [0x04e1] = 0x5504, [0x04e2] = 0x5508, [0x04e3] = 0x54eb,
  [0x04e4] = 0x5511, [0x04e5] = 0x5505, [0x04e6] = 0x54f1, [0x04e7] = 0x550a,
  [0x04e8] = 0x54fb, [0x04e9] = 0x54f7, [0x04ea] = 0x54f8, [0x04eb] = 0x54e0,
  [0x04ec] = 0x550e, [0x04ed] = 0x5503, [0x04ee] = 0x550b, [0x04ef] = 0x5701,
  [0x04f0] = 0x5702, [0x04f1] = 0x57cc, [0x04f2] = 0x5832, [0x04f3] = 0x57d5,
  [0x04f4] = 0x57d2, [0x04f5] = 0x57ba, [0x04f6] = 0x57c6, [0x04f7] = 0x57bd,
  [0x04f8] = 0x57bc, [0x04f9] = 0x57b8, [0x04fa] = 0x57b6, [0x04fb] = 0x57bf,
  [0x04fc] = 0x57c7, [0x04fd] = 0x57d0, [0x04fe] = 0x57b9, [0x04ff] = 0x57c1,
  [0x0500] = 0x590e, [0x0501] = 0x594a, [0x0502] = 0x5a19, [0x0503] = 0x5a16,
  [0x0504] = 0x5a2d, [0x0505] = 0x5a2e, [0x0506] = 0x5a15, [0x0507] = 0x5a0f,
  [0x0508] = 0x5a17, [0x0509] = 0x5a0a, [0x050a] = 0x5a1e, [0x050b] = 0x5a33,
  [0x050c] = 0x5b6c, [0x050d] = 0x5ba7, [0x050e] = 0x5bad, [0x050f] = 0x5bac,
  [0x0510] = 0x5c03, [0x0511] = 0x5c56, [0x0512] = 0x5c54, [0x0513] = 0x5cec,
  [0x0514] = 0x5cff, [0x0515] = 0x5cee, [0x0516] = 0x5cf1, [0x0517] = 0x5cf7,
  [0x0518] = 0x5d00, [0x0519] = 0x5cf9, [0x051a] = 0x5e29, [0x051b] = 0x5e28,
  [0x051c] = 0x5ea8, [0x051d] = 0x5eae, [0x051e] = 0x5eaa, [0x051f] = 0x5eac,
  [0x0520] = 0x5f33, [0x0521] = 0x5f30, [0x0522] = 0x5f67, [0x0523] = 0x605d,
  [0x0524] = 0x605a, [0x0525] = 0x6067, [0x0526] = 0x6041, [0x0527] = 0x60a2,
  [0x0528] = 0x6088, [0x0529] = 0x6080, [0x052a] = 0x6092, [0x052b] = 0x6081,
  [0x052c] = 0x609d, [0x052d] = 0x6083, [0x052e] = 0x6095, [0x052f] = 0x609b,
  [0x0530] = 0x6097, [0x0531] = 0x6087, [0x0532] = 0x609c, [0x0533] = 0x608e,
  [0x0534] = 0x6219, [0x0535] = 0x6246, [0x0536] = 0x62f2, [0x0537] = 0x6310,
  [0x0538] = 0x6356, [0x0539] = 0x632c, [0x053a] = 0x6344, [0x053b] = 0x6345,
  [0x053c] = 0x6336, [0x053d] = 0x6343, [0x053e] = 0x63e4, [0x053f] = 0x6339,
  [0x0540] = 0x634b, [0x0541] = 0x634a, [0x0542] = 0x633c, [0x0543] = 0x6329,
  [0x0544] = 0x6341, [0x0545] = 0x6334, [0x0546] = 0x6358, [0x0547] = 0x6354,
  [0x0548] = 0x6359, [0x0549] = 0x632d, [0x054a] = 0x6347, [0x054b] = 0x6333,
  [0x054c] = 0x635a, [0x054d] = 0x6351, [0x054e] = 0x6338, [0x054f] = 0x6357,
  [0x0550] = 0x6340, [0x0551] = 0x6348, [0x0552] = 0x654a, [0x0553] = 0x6546,
  [0x0554] = 0x65c6, [0x0555] = 0x65c3, [0x0556] = 0x65c4, [0x0557] = 0x65c2,
  [0x0558] = 0x664a, [0x0559] = 0x665f, [0x055a] = 0x6647, [0x055b] = 0x6651,
  [0x055c] = 0x6712, [0x055d] = 0x6713, [0x055e] = 0x681f, [0x055f] = 0x681a,
  [0x0560] = 0x6849, [0x0561] = 0x6832, [0x0562] = 0x6833, [0x0563] = 0x683b,
  [0x0564] = 0x684b, [0x0565] = 0x684f, [0x0566] = 0x6816, [0x0567] = 0x6831,
  [0x0568] = 0x681c, [0x0569] = 0x6835, [0x056a] = 0x682b, [0x056b] = 0x682d,
  [0x056c] = 0x682f, [0x056d] = 0x684e, [0x056e] = 0x6844, [0x056f] = 0x6834,
  [0x0570] = 0x681d, [0x0571] = 0x6812, [0x0572] = 0x6814, [0x0573] = 0x6826,
  [0x0574] = 0x6828, [0x0575] = 0x682e, [0x0576] = 0x684d, [0x0577] = 0x683a,
  [0x0578] = 0x6825, [0x0579] = 0x6820, [0x057a] = 0x6b2c, [0x057b] = 0x6b2f,
  [0x057c] = 0x6b2d, [0x057d] = 0x6b31, [0x057e] = 0x6b34, [0x057f] = 0x6b6d,
  [0x0580] = 0x8082, [0x0581] = 0x6b88, [0x0582] = 0x6be6, [0x0583] = 0x6be4,
  [0x0584] = 0x6be8, [0x0585] = 0x6be3, [0x0586] = 0x6be2, [0x0587] = 0x6be7,
  [0x0588] = 0x6c25, [0x0589] = 0x6d7a, [0x058a] = 0x6d63, [0x058b] = 0x6d64,
  [0x058c] = 0x6d76, [0x058d] = 0x6d0d, [0x058e] = 0x6d61, [0x058f] = 0x6d92,
  [0x0590] = 0x6d58, [0x0591] = 0x6d62, [0x0592] = 0x6d6d, [0x0593] = 0x6d6f,
  [0x0594] = 0x6d91, [0x0595] = 0x6d8d, [0x0596] = 0x6def, [0x0597] = 0x6d7f,
  [0x0598] = 0x6d86, [0x0599] = 0x6d5e, [0x059a] = 0x6d67, [0x059b] = 0x6d60,
  [0x059c] = 0x6d97, [0x059d] = 0x6d70, [0x059e] = 0x6d7c, [0x059f] = 0x6d5f,
  [0x05a0] = 0x6d82, [0x05a1] = 0x6d98, [0x05a2] = 0x6d2f, [0x05a3] = 0x6d68,
  [0x05a4] = 0x6d8b, [0x05a5] = 0x6d7e, [0x05a6] = 0x6d80, [0x05a7] = 0x6d84,
  [0x05a8] = 0x6d16, [0x05a9] = 0x6d83, [0x05aa] = 0x6d7b, [0x05ab] = 0x6d7d,
  [0x05ac] = 0x6d75, [0x05ad] = 0x6d90, [0x05ae] = 0x70dc, [0x05af] = 0x70d3,
  [0x05b0] = 0x70d1, [0x05b1] = 0x70dd, [0x05b2] = 0x70cb, [0x05b3] = 0x7f39,
  [0x05b4] = 0x70e2, [0x05b5] = 0x70d7, [0x05b6] = 0x70d2, [0x05b7] = 0x70de,
  [0x05b8] = 0x70e0, [0x05b9] = 0x70d4, [0x05ba] = 0x70cd, [0x05bb] = 0x70c5,
  [0x05bc] = 0x70c6, [0x05bd] = 0x70c7, [0x05be] = 0x70da, [0x05bf] = 0x70ce,
  [0x05c0] = 0x70e1, [0x05c1] = 0x7242, [0x05c2] = 0x7278, [0x05c3] = 0x7277,
  [0x05c4] = 0x7276, [0x05c5] = 0x7300, [0x05c6] = 0x72fa, [0x05c7] = 0x72f4,
  [0x05c8] = 0x72fe, [0x05c9] = 0x72f6, [0x05ca] = 0x72f3, [0x05cb] = 0x72fb,
  [0x05cc] = 0x7301, [0x05cd] = 0x73d3, [0x05ce] = 0x73d9, [0x05cf] = 0x73e5,
  [0x05d0] = 0x73d6, [0x05d1] = 0x73bc, [0x05d2] = 0x73e7, [0x05d3] = 0x73e3,
  [0x05d4] = 0x73e9, [0x05d5] = 0x73dc, [0x05d6] = 0x73d2, [0x05d7] = 0x73db,
  [0x05d8] = 0x73d4, [0x05d9] = 0x73dd, [0x05da] = 0x73da, [0x05db] = 0x73d7,
  [0x05dc] = 0x73d8, [0x05dd] = 0x73e8, [0x05de] = 0x74de, [0x05df] = 0x74df,
  [0x05e0] = 0x74f4, [0x05e1] = 0x74f5, [0x05e2] = 0x7521, [0x05e3] = 0x755b,
  [0x05e4] = 0x755f, [0x05e5] = 0x75b0, [0x05e6] = 0x75c1, [0x05e7] = 0x75bb,
  [0x05e8] = 0x75c4, [0x05e9] = 0x75c0, [0x05ea] = 0x75bf, [0x05eb] = 0x75b6,
  [0x05ec] = 0x75ba, [0x05ed] = 0x768a, [0x05ee] = 0x76c9, [0x05ef] = 0x771d,
  [0x05f0] = 0x771b, [0x05f1] = 0x7710, [0x05f2] = 0x7713, [0x05f3] = 0x7712,
  [0x05f4] = 0x7723, [0x05f5] = 0x7711, [0x05f6] = 0x7715, [0x05f7] = 0x7719,
  [0x05f8] = 0x771a, [0x05f9] = 0x7722, [0x05fa] = 0x7727, [0x05fb] = 0x7823,
  [0x05fc] = 0x782c, [0x05fd] = 0x7822, [0x05fe] = 0x7835, [0x05ff] = 0x782f,
  [0x0600] = 0x7828, [0x0601] = 0x782e, [0x0602] = 0x782b, [0x0603] = 0x7821,
  [0x0604] = 0x7829, [0x0605] = 0x7833, [0x0606] = 0x782a, [0x0607] = 0x7831,
  [0x0608] = 0x7954, [0x0609] = 0x795b, [0x060a] = 0x794f, [0x060b] = 0x795c,
  [0x060c] = 0x7953, [0x060d] = 0x7952, [0x060e] = 0x7951, [0x060f] = 0x79eb,
  [0x0610] = 0x79ec, [0x0611] = 0x79e0, [0x0612] = 0x79ee, [0x0613] = 0x79ed,
  [0x0614] = 0x79ea, [0x0615] = 0x79dc, [0x0616] = 0x79de, [0x0617] = 0x79dd,
  [0x0618] = 0x7a86, [0x0619] = 0x7a89, [0x061a] = 0x7a85, [0x061b] = 0x7a8b,
  [0x061c] = 0x7a8c, [0x061d] = 0x7a8a, [0x061e] = 0x7a87, [0x061f] = 0x7ad8,
  [0x0620] = 0x7b10, [0x0621] = 0x7b04, [0x0622] = 0x7b13, [0x0623] = 0x7b05,
  [0x0624] = 0x7b0f, [0x0625] = 0x7b08, [0x0626] = 0x7b0a, [0x0627] = 0x7b0e,
  [0x0628] = 0x7b09, [0x0629] = 0x7b12, [0x062a] = 0x7c84, [0x062b] = 0x7c91,
  [0x062c] = 0x7c8a, [0x062d] = 0x7c8c, [0x062e] = 0x7c88, [0x062f] = 0x7c8d,
  [0x0630] = 0x7c85, [0x0631] = 0x7d1e, [0x0632] = 0x7d1d, [0x0633] = 0x7d11,
  [0x0634] = 0x7d0e, [0x0635] = 0x7d18, [0x0636] = 0x7d16, [0x0637] = 0x7d13,
  [0x0638] = 0x7d1f, [0x0639] = 0x7d12, [0x063a] = 0x7d0f, [0x063b] = 0x7d0c,
  [0x063c] = 0x7f5c, [0x063d] = 0x7f61, [0x063e] = 0x7f5e, [0x063f] = 0x7f60,
  [0x0640] = 0x7f5d, [0x0641] = 0x7f5b, [0x0642] = 0x7f96, [0x0643] = 0x7f92,
  [0x0644] = 0x7fc3, [0x0645] = 0x7fc2, [0x0646] = 0x7fc0, [0x0647] = 0x8016,
  [0x0648] = 0x803e, [0x0649] = 0x8039, [0x064a] = 0x80fa, [0x064b] = 0x80f2,
  [0x064c] = 0x80f9, [0x064d] = 0x80f5, [0x064e] = 0x8101, [0x064f] = 0x80fb,
  [0x0650] = 0x8100, [0x0651] = 0x8201, [0x0652] = 0x822f, [0x0653] = 0x8225,
  [0x0654] = 0x8333, [0x0655] = 0x832d, [0x0656] = 0x8344, [0x0657] = 0x8319,
  [0x0658] = 0x8351, [0x0659] = 0x8325, [0x065a] = 0x8356, [0x065b] = 0x833f,
  [0x065c] = 0x8341, [0x065d] = 0x8326, [0x065e] = 0x831c, [0x065f] = 0x8322,
  [0x0660] = 0x8342, [0x0661] = 0x834e, [0x0662] = 0x831b, [0x0663] = 0x832a,
  [0x0664] = 0x8308, [0x0665] = 0x833c, [0x0666] = 0x834d, [0x0667] = 0x8316,
  [0x0668] = 0x8324, [0x0669] = 0x8320, [0x066a] = 0x8337, [0x066b] = 0x832f,
  [0x066c] = 0x8329, [0x066d] = 0x8347, [0x066e] = 0x8345, [0x066f] = 0x834c,
  [0x0670] = 0x8353, [0x0671] = 0x831e, [0x0672] = 0x832c, [0x0673] = 0x834b,
  [0x0674] = 0x8327, [0x0675] = 0x8348, [0x0676] = 0x8653, [0x0677] = 0x8652,
  [0x0678] = 0x86a2, [0x0679] = 0x86a8, [0x067a] = 0x8696, [0x067b] = 0x868d,
  [0x067c] = 0x8691, [0x067d] = 0x869e, [0x067e] = 0x8687, [0x067f] = 0x8697,
  [0x0680] = 0x8686, [0x0681] = 0x868b, [0x0682] = 0x869a, [0x0683] = 0x8685,
  [0x0684] = 0x86a5, [0x0685] = 0x8699, [0x0686] = 0x86a1, [0x0687] = 0x86a7,
  [0x0688] = 0x8695, [0x0689] = 0x8698, [0x068a] = 0x868e, [0x068b] = 0x869d,
  [0x068c] = 0x8690, [0x068d] = 0x8694, [0x068e] = 0x8843, [0x068f] = 0x8844,
  [0x0690] = 0x886d, [0x0691] = 0x8875, [0x0692] = 0x8876, [0x0693] = 0x8872,
  [0x0694] = 0x8880, [0x0695] = 0x8871, [0x0696] = 0x887f, [0x0697] = 0x886f,
  [0x0698] = 0x8883, [0x0699] = 0x887e, [0x069a] = 0x8874, [0x069b] = 0x887c,
  [0x069c] = 0x8a12, [0x069d] = 0x8c47, [0x069e] = 0x8c57, [0x069f] = 0x8c7b,
  [0x06a0] = 0x8ca4, [0x06a1] = 0x8ca3, [0x06a2] = 0x8d76, [0x06a3] = 0x8d78,
  [0x06a4] = 0x8db5, [0x06a5] = 0x8db7, [0x06a6] = 0x8db6, [0x06a7] = 0x8ed1,
  [0x06a8] = 0x8ed3, [0x06a9] = 0x8ffe, [0x06aa] = 0x8ff5, [0x06ab] = 0x9002,
  [0x06ac] = 0x8fff, [0x06ad] = 0x8ffb, [0x06ae] = 0x9004, [0x06af] = 0x8ffc,
  [0x06b0] = 0x8ff6, [0x06b1] = 0x90d6, [0x06b2] = 0x90e0, [0x06b3] = 0x90d9,
  [0x06b4] = 0x90da, [0x06b5] = 0x90e3, [0x06b6] = 0x90df, [0x06b7] = 0x90e5,
  [0x06b8] = 0x90d8, [0x06b9] = 0x90db, [0x06ba] = 0x90d7, [0x06bb] = 0x90dc,
  [0x06bc] = 0x90e4, [0x06bd] = 0x9150, [0x06be] = 0x914e, [0x06bf] = 0x914f,
  [0x06c0] = 0x91d5, [0x06c1] = 0x91e2, [0x06c2] = 0x91da, [0x06c3] = 0x965c,
  [0x06c4] = 0x965f, [0x06c5] = 0x96bc, [0x06c6] = 0x98e3, [0x06c7] = 0x9adf,
  [0x06c8] = 0x9b2f, [0x06c9] = 0x4e7f, [0x06ca] = 0x5070, [0x06cb] = 0x506a,
  [0x06cc] = 0x5061, [0x06cd] = 0x505e, [0x06ce] = 0x5060, [0x06cf] = 0x5053,
  [0x06d0] = 0x504b, [0x06d1] = 0x505d, [0x06d2] = 0x5072, [0x06d3] = 0x5048,
  [0x06d4] = 0x504d, [0x06d5] = 0x5041, [0x06d6] = 0x505b, [0x06d7] = 0x504a,
  [0x06d8] = 0x5062, [0x06d9] = 0x5015, [0x06da] = 0x5045, [0x06db] = 0x505f,
  [0x06dc] = 0x5069, [0x06dd] = 0x506b, [0x06de] = 0x5063, [0x06df] = 0x5064,
  [0x06e0] = 0x5046, [0x06e1] = 0x5040, [0x06e2] = 0x506e, [0x06e3] = 0x5073,
  [0x06e4] = 0x5057, [0x06e5] = 0x5051, [0x06e6] = 0x51d0, [0x06e7] = 0x526b,
  [0x06e8] = 0x526d, [0x06e9] = 0x526c, [0x06ea] = 0x526e, [0x06eb] = 0x52d6,
  [0x06ec] = 0x52d3, [0x06ed] = 0x532d, [0x06ee] = 0x539c, [0x06ef] = 0x5575,
  [0x06f0] = 0x5576, [0x06f1] = 0x553c, [0x06f2] = 0x554d, [0x06f3] = 0x5550,
  [0x06f4] = 0x5534, [0x06f5] = 0x552a, [0x06f6] = 0x5551, [0x06f7] = 0x5562,
  [0x06f8] = 0x5536, [0x06f9] = 0x5535, [0x06fa] = 0x5530, [0x06fb] = 0x5552,
  [0x06fc] = 0x5545, [0x06fd] = 0x550c, [0x06fe] = 0x5532, [0x06ff] = 0x5565,
  [0x0700] = 0x554e, [0x0701] = 0x5539, [0x0702] = 0x5548, [0x0703] = 0x552d,
  [0x0704] = 0x553b, [0x0705] = 0x5540, [0x0706] = 0x554b, [0x0707] = 0x570a,
  [0x0708] = 0x5707, [0x0709] = 0x57fb, [0x070a] = 0x5814, [0x070b] = 0x57e2,
  [0x070c] = 0x57f6, [0x070d] = 0x57dc, [0x070e] = 0x57f4, [0x070f] = 0x5800,
  [0x0710] = 0x57ed, [0x0711] = 0x57fd, [0x0712] = 0x5808, [0x0713] = 0x57f8,
  [0x0714] = 0x580b, [0x0715] = 0x57f3, [0x0716] = 0x57cf, [0x0717] = 0x5807,
  [0x0718] = 0x57ee, [0x0719] = 0x57e3, [0x071a] = 0x57f2, [0x071b] = 0x57e5,
  [0x071c] = 0x57ec, [0x071d] = 0x57e1, [0x071e] = 0x580e, [0x071f] = 0x57fc,
  [0x0720] = 0x5810, [0x0721] = 0x57e7, [0x0722] = 0x5801, [0x0723] = 0x580c,
  [0x0724] = 0x57f1, [0x0725] = 0x57e9, [0x0726] = 0x57f0, [0x0727] = 0x580d,
  [0x0728] = 0x5804, [0x0729] = 0x595c, [0x072a] = 0x5a60, [0x072b] = 0x5a58,
  [0x072c] = 0x5a55, [0x072d] = 0x5a67, [0x072e] = 0x5a5e, [0x072f] = 0x5a38,
  [0x0730] = 0x5a35, [0x0731] = 0x5a6d, [0x0732] = 0x5a50, [0x0733] = 0x5a5f,
  [0x0734] = 0x5a65, [0x0735] = 0x5a6c, [0x0736] = 0x5a53, [0x0737] = 0x5a64,
  [0x0738] = 0x5a57, [0x0739] = 0x5a43, [0x073a] = 0x5a5d, [0x073b] = 0x5a52,
  [0x073c] = 0x5a44, [0x073d] = 0x5a5b, [0x073e] = 0x5a48, [0x073f] = 0x5a8e,
  [0x0740] = 0x5a3e, [0x0741] = 0x5a4d, [0x0742] = 0x5a39, [0x0743] = 0x5a4c,
  [0x0744] = 0x5a70, [0x0745] = 0x5a69, [0x0746] = 0x5a47, [0x0747] = 0x5a51,
  [0x0748] = 0x5a56, [0x0749] = 0x5a42, [0x074a] = 0x5a5c, [0x074b] = 0x5b72,
  [0x074c] = 0x5b6e, [0x074d] = 0x5bc1, [0x074e] = 0x5bc0, [0x074f] = 0x5c59,
  [0x0750] = 0x5d1e, [0x0751] = 0x5d0b, [0x0752] = 0x5d1d, [0x0753] = 0x5d1a,
  [0x0754] = 0x5d20, [0x0755] = 0x5d0c, [0x0756] = 0x5d28, [0x0757] = 0x5d0d,
  [0x0758] = 0x5d26, [0x0759] = 0x5d25, [0x075a] = 0x5d0f, [0x075b] = 0x5d30,
  [0x075c] = 0x5d12, [0x075d] = 0x5d23, [0x075e] = 0x5d1f, [0x075f] = 0x5d2e,
  [0x0760] = 0x5e3e, [0x0761] = 0x5e34, [0x0762] = 0x5eb1, [0x0763] = 0x5eb4,
  [0x0764] = 0x5eb9, [0x0765] = 0x5eb2, [0x0766] = 0x5eb3, [0x0767] = 0x5f36,
  [0x0768] = 0x5f38, [0x0769] = 0x5f9b, [0x076a] = 0x5f96, [0x076b] = 0x5f9f,
  [0x076c] = 0x608a, [0x076d] = 0x6090, [0x076e] = 0x6086, [0x076f] = 0x60be,
  [0x0770] = 0x60b0, [0x0771] = 0x60ba, [0x0772] = 0x60d3, [0x0773] = 0x60d4,
  [0x0774] = 0x60cf, [0x0775] = 0x60e4, [0x0776] = 0x60d9, [0x0777] = 0x60dd,
  [0x0778] = 0x60c8, [0x0779] = 0x60b1, [0x077a] = 0x60db, [0x077b] = 0x60b7,
  [0x077c] = 0x60ca, [0x077d] = 0x60bf, [0x077e] = 0x60c3, [0x077f] = 0x60cd,
  [0x0780] = 0x60c0, [0x0781] = 0x6332, [0x0782] = 0x6365, [0x0783] = 0x638a,
  [0x0784] = 0x6382, [0x0785] = 0x637d, [0x0786] = 0x63bd, [0x0787] = 0x639e,
  [0x0788] = 0x63ad, [0x0789] = 0x639d, [0x078a] = 0x6397, [0x078b] = 0x63ab,
  [0x078c] = 0x638e, [0x078d] = 0x636f, [0x078e] = 0x6387, [0x078f] = 0x6390,
  [0x0790] = 0x636e, [0x0791] = 0x63af, [0x0792] = 0x6375, [0x0793] = 0x639c,
  [0x0794] = 0x636d, [0x0795] = 0x63ae, [0x0796] = 0x637c, [0x0797] = 0x63a4,
  [0x0798] = 0x633b, [0x0799] = 0x639f, [0x079a] = 0x6378, [0x079b] = 0x6385,
  [0x079c] = 0x6381, [0x079d] = 0x6391, [0x079e] = 0x638d, [0x079f] = 0x6370,
  [0x07a0] = 0x6553, [0x07a1] = 0x65cd, [0x07a2] = 0x6665, [0x07a3] = 0x6661,
  [0x07a4] = 0x665b, [0x07a5] = 0x6659, [0x07a6] = 0x665c, [0x07a7] = 0x6662,
  [0x07a8] = 0x6718, [0x07a9] = 0x6879, [0x07aa] = 0x6887, [0x07ab] = 0x6890,
  [0x07ac] = 0x689c, [0x07ad] = 0x686d, [0x07ae] = 0x686e, [0x07af] = 0x68ae,
  [0x07b0] = 0x68ab, [0x07b1] = 0x6956, [0x07b2] = 0x686f, [0x07b3] = 0x68a3,
  [0x07b4] = 0x68ac, [0x07b5] = 0x68a9, [0x07b6] = 0x6875, [0x07b7] = 0x6874,
  [0x07b8] = 0x68b2, [0x07b9] = 0x688f, [0x07ba] = 0x6877, [0x07bb] = 0x6892,
  [0x07bc] = 0x687c, [0x07bd] = 0x686b, [0x07be] = 0x6872, [0x07bf] = 0x68aa,
  [0x07c0] = 0x6880, [0x07c1] = 0x6871, [0x07c2] = 0x687e, [0x07c3] = 0x689b,
  [0x07c4] = 0x6896, [0x07c5] = 0x688b, [0x07c6] = 0x68a0, [0x07c7] = 0x6889,
  [0x07c8] = 0x68a4, [0x07c9] = 0x6878, [0x07ca] = 0x687b, [0x07cb] = 0x6891,
  [0x07cc] = 0x688c, [0x07cd] = 0x688a, [0x07ce] = 0x687d, [0x07cf] = 0x6b36,
  [0x07d0] = 0x6b33, [0x07d1] = 0x6b37, [0x07d2] = 0x6b38, [0x07d3] = 0x6b91,
  [0x07d4] = 0x6b8f, [0x07d5] = 0x6b8d, [0x07d6] = 0x6b8e, [0x07d7] = 0x6b8c,
  [0x07d8] = 0x6c2a, [0x07d9] = 0x6dc0, [0x07da] = 0x6dab, [0x07db] = 0x6db4,
  [0x07dc] = 0x6db3, [0x07dd] = 0x6e74, [0x07de] = 0x6dac, [0x07df] = 0x6de9,
  [0x07e0] = 0x6de2, [0x07e1] = 0x6db7, [0x07e2] = 0x6df6, [0x07e3] = 0x6dd4,
  [0x07e4] = 0x6e00, [0x07e5] = 0x6dc8, [0x07e6] = 0x6de0, [0x07e7] = 0x6ddf,
  [0x07e8] = 0x6dd6, [0x07e9] = 0x6dbe, [0x07ea] = 0x6de5, [0x07eb] = 0x6ddc,
  [0x07ec] = 0x6ddd, [0x07ed] = 0x6ddb, [0x07ee] = 0x6df4, [0x07ef] = 0x6dca,
  [0x07f0] = 0x6dbd, [0x07f1] = 0x6ded, [0x07f2] = 0x6df0, [0x07f3] = 0x6dba,
  [0x07f4] = 0x6dd5, [0x07f5] = 0x6dc2, [0x07f6] = 0x6dcf, [0x07f7] = 0x6dc9,
  [0x07f8] = 0x6dd0, [0x07f9] = 0x6df2, [0x07fa] = 0x6dd3, [0x07fb] = 0x6dfd,
  [0x07fc] = 0x6dd7, [0x07fd] = 0x6dcd, [0x07fe] = 0x6de3, [0x07ff] = 0x6dbb,
  [0x0800] = 0x70fa, [0x0801] = 0x710d, [0x0802] = 0x70f7, [0x0803] = 0x7117,
  [0x0804] = 0x70f4, [0x0805] = 0x710c, [0x0806] = 0x70f0, [0x0807] = 0x7104,
  [0x0808] = 0x70f3, [0x0809] = 0x7110, [0x080a] = 0x70fc, [0x080b] = 0x70ff,
  [0x080c] = 0x7106, [0x080d] = 0x7113, [0x080e] = 0x7100, [0x080f] = 0x70f8,
  [0x0810] = 0x70f6, [0x0811] = 0x710b, [0x0812] = 0x7102, [0x0813] = 0x710e,
  [0x0814] = 0x727e, [0x0815] = 0x727b, [0x0816] = 0x727c, [0x0817] = 0x727f,
  [0x0818] = 0x731d, [0x0819] = 0x7317, [0x081a] = 0x7307, [0x081b] = 0x7311,
  [0x081c] = 0x7318, [0x081d] = 0x730a, [0x081e] = 0x7308, [0x081f] = 0x72ff,
  [0x0820] = 0x730f, [0x0821] = 0x731e, [0x0822] = 0x7388, [0x0823] = 0x73f6,
  [0x0824] = 0x73f8, [0x0825] = 0x73f5, [0x0826] = 0x7404, [0x0827] = 0x7401,
  [0x0828] = 0x73fd, [0x0829] = 0x7407, [0x082a] = 0x7400, [0x082b] = 0x73fa,
  [0x082c] = 0x73fc, [0x082d] = 0x73ff, [0x082e] = 0x740c, [0x082f] = 0x740b,
  [0x0830] = 0x73f4, [0x0831] = 0x7408, [0x0832] = 0x7564, [0x0833] = 0x7563,
  [0x0834] = 0x75ce, [0x0835] = 0x75d2, [0x0836] = 0x75cf, [0x0837] = 0x75cb,
  [0x0838] = 0x75cc, [0x0839] = 0x75d1, [0x083a] = 0x75d0, [0x083b] = 0x768f,
  [0x083c] = 0x7689, [0x083d] = 0x76d3, [0x083e] = 0x7739, [0x083f] = 0x772f,
  [0x0840] = 0x772d, [0x0841] = 0x7731, [0x0842] = 0x7732, [0x0843] = 0x7734,
  [0x0844] = 0x7733, [0x0845] = 0x773d, [0x0846] = 0x7725, [0x0847] = 0x773b,
  [0x0848] = 0x7735, [0x0849] = 0x7848, [0x084a] = 0x7852, [0x084b] = 0x7849,
  [0x084c] = 0x784d, [0x084d] = 0x784a, [0x084e] = 0x784c, [0x084f] = 0x7826,
  [0x0850] = 0x7845, [0x0851] = 0x7850, [0x0852] = 0x7964, [0x0853] = 0x7967,
  [0x0854] = 0x7969, [0x0855] = 0x796a, [0x0856] = 0x7963, [0x0857] = 0x796b,
  [0x0858] = 0x7961, [0x0859] = 0x79bb, [0x085a] = 0x79fa, [0x085b] = 0x79f8,
  [0x085c] = 0x79f6, [0x085d] = 0x79f7, [0x085e] = 0x7a8f, [0x085f] = 0x7a94,
  [0x0860] = 0x7a90, [0x0861] = 0x7b35, [0x0862] = 0x7b3b, [0x0863] = 0x7b34,
  [0x0864] = 0x7b25, [0x0865] = 0x7b30, [0x0866] = 0x7b22, [0x0867] = 0x7b24,
  [0x0868] = 0x7b33, [0x0869] = 0x7b18, [0x086a] = 0x7b2a, [0x086b] = 0x7b1d,
  [0x086c] = 0x7b31, [0x086d] = 0x7b2b, [0x086e] = 0x7b2d, [0x086f] = 0x7b2f,
  [0x0870] = 0x7b32, [0x0871] = 0x7b38, [0x0872] = 0x7b1a, [0x0873] = 0x7b23,
  [0x0874] = 0x7c94, [0x0875] = 0x7c98, [0x0876] = 0x7c96, [0x0877] = 0x7ca3,
  [0x0878] = 0x7d35, [0x0879] = 0x7d3d, [0x087a] = 0x7d38, [0x087b] = 0x7d36,
  [0x087c] = 0x7d3a, [0x087d] = 0x7d45, [0x087e] = 0x7d2c, [0x087f] = 0x7d29,
  [0x0880] = 0x7d41, [0x0881] = 0x7d47, [0x0882] = 0x7d3e, [0x0883] = 0x7d3f,
  [0x0884] = 0x7d4a, [0x0885] = 0x7d3b, [0x0886] = 0x7d28, [0x0887] = 0x7f63,
  [0x0888] = 0x7f95, [0x0889] = 0x7f9c, [0x088a] = 0x7f9d, [0x088b] = 0x7f9b,
  [0x088c] = 0x7fca, [0x088d] = 0x7fcb, [0x088e] = 0x7fcd, [0x088f] = 0x7fd0,
  [0x0890] = 0x7fd1, [0x0891] = 0x7fc7, [0x0892] = 0x7fcf, [0x0893] = 0x7fc9,
  [0x0894] = 0x801f, [0x0895] = 0x801e, [0x0896] = 0x801b, [0x0897] = 0x8047,
  [0x0898] = 0x8043, [0x0899] = 0x8048, [0x089a] = 0x8118, [0x089b] = 0x8125,
  [0x089c] = 0x8119, [0x089d] = 0x811b, [0x089e] = 0x812d, [0x089f] = 0x811f,
  [0x08a0] = 0x812c, [0x08a1] = 0x811e, [0x08a2] = 0x8121, [0x08a3] = 0x8115,
  [0x08a4] = 0x8127, [0x08a5] = 0x811d, [0x08a6] = 0x8122, [0x08a7] = 0x8211,
  [0x08a8] = 0x8238, [0x08a9] = 0x8233, [0x08aa] = 0x823a, [0x08ab] = 0x8234,
  [0x08ac] = 0x8232, [0x08ad] = 0x8274, [0x08ae] = 0x8390, [0x08af] = 0x83a3,
  [0x08b0] = 0x83a8, [0x08b1] = 0x838d, [0x08b2] = 0x837a, [0x08b3] = 0x8373,
  [0x08b4] = 0x83a4, [0x08b5] = 0x8374, [0x08b6] = 0x838f, [0x08b7] = 0x8381,
  [0x08b8] = 0x8395, [0x08b9] = 0x8399, [0x08ba] = 0x8375, [0x08bb] = 0x8394,
  [0x08bc] = 0x83a9, [0x08bd] = 0x837d, [0x08be] = 0x8383, [0x08bf] = 0x838c,
  [0x08c0] = 0x839d, [0x08c1] = 0x839b, [0x08c2] = 0x83aa, [0x08c3] = 0x838b,
  [0x08c4] = 0x837e, [0x08c5] = 0x83a5, [0x08c6] = 0x83af, [0x08c7] = 0x8388,
  [0x08c8] = 0x8397, [0x08c9] = 0x83b0, [0x08ca] = 0x837f, [0x08cb] = 0x83a6,
  [0x08cc] = 0x8387, [0x08cd] = 0x83ae, [0x08ce] = 0x8376, [0x08cf] = 0x8659,
  [0x08d0] = 0x8656, [0x08d1] = 0x86bf, [0x08d2] = 0x86b7, [0x08d3] = 0x86c2,
  [0x08d4] = 0x86c1, [0x08d5] = 0x86c5, [0x08d6] = 0x86ba, [0x08d7] = 0x86b0,
  [0x08d8] = 0x86c8, [0x08d9] = 0x86b9, [0x08da] = 0x86b3, [0x08db] = 0x86b8,
  [0x08dc] = 0x86cc, [0x08dd] = 0x86b4, [0x08de] = 0x86bb, [0x08df] = 0x86bc,
  [0x08e0] = 0x86c3, [0x08e1] = 0x86bd, [0x08e2] = 0x86be, [0x08e3] = 0x8852,
  [0x08e4] = 0x8889, [0x08e5] = 0x8895, [0x08e6] = 0x88a8, [0x08e7] = 0x88a2,
  [0x08e8] = 0x88aa, [0x08e9] = 0x889a, [0x08ea] = 0x8891, [0x08eb] = 0x88a1,
  [0x08ec] = 0x889f, [0x08ed] = 0x8898, [0x08ee] = 0x88a7, [0x08ef] = 0x8899,
  [0x08f0] = 0x889b, [0x08f1] = 0x8897, [0x08f2] = 0x88a4, [0x08f3] = 0x88ac,
  [0x08f4] = 0x888c, [0x08f5] = 0x8893, [0x08f6] = 0x888e, [0x08f7] = 0x8982,
  [0x08f8] = 0x89d6, [0x08f9] = 0x89d9, [0x08fa] = 0x89d5, [0x08fb] = 0x8a30,
  [0x08fc] = 0x8a27, [0x08fd] = 0x8a2c, [0x08fe] = 0x8a1e, [0x08ff] = 0x8c39,
  [0x0900] = 0x8c3b, [0x0901] = 0x8c5c, [0x0902] = 0x8c5d, [0x0903] = 0x8c7d,
  [0x0904] = 0x8ca5, [0x0905] = 0x8d7d, [0x0906] = 0x8d7b, [0x0907] = 0x8d79,
  [0x0908] = 0x8dbc, [0x0909] = 0x8dc2, [0x090a] = 0x8db9, [0x090b] = 0x8dbf,
  [0x090c] = 0x8dc1, [0x090d] = 0x8ed8, [0x090e] = 0x8ede, [0x090f] = 0x8edd,
  [0x0910] = 0x8edc, [0x0911] = 0x8ed7, [0x0912] = 0x8ee0, [0x0913] = 0x8ee1,
  [0x0914] = 0x9024, [0x0915] = 0x900b, [0x0916] = 0x9011, [0x0917] = 0x901c,
  [0x0918] = 0x900c, [0x0919] = 0x9021, [0x091a] = 0x90ef, [0x091b] = 0x90ea,
  [0x091c] = 0x90f0, [0x091d] = 0x90f4, [0x091e] = 0x90f2, [0x091f] = 0x90f3,
  [0x0920] = 0x90d4, [0x0921] = 0x90eb, [0x0922] = 0x90ec, [0x0923] = 0x90e9,
  [0x0924] = 0x9156, [0x0925] = 0x9158, [0x0926] = 0x915a, [0x0927] = 0x9153,
  [0x0928] = 0x9155, [0x0929] = 0x91ec, [0x092a] = 0x91f4, [0x092b] = 0x91f1,
  [0x092c] = 0x91f3, [0x092d] = 0x91f8, [0x092e] = 0x91e4, [0x092f] = 0x91f9,
  [0x0930] = 0x91ea, [0x0931] = 0x91eb, [0x0932] = 0x91f7, [0x0933] = 0x91e8,
  [0x0934] = 0x91ee, [0x0935] = 0x957a, [0x0936] = 0x9586, [0x0937] = 0x9588,
  [0x0938] = 0x967c, [0x0939] = 0x966d, [0x093a] = 0x966b, [0x093b] = 0x9671,
  [0x093c] = 0x966f, [0x093d] = 0x96bf, [0x093e] = 0x976a, [0x093f] = 0x9804,
  [0x0940] = 0x98e5, [0x0941] = 0x9997, [0x0942] = 0x509b, [0x0943] = 0x5095,
  [0x0944] = 0x5094, [0x0945] = 0x509e, [0x0946] = 0x508b, [0x0947] = 0x50a3,
  [0x0948] = 0x5083, [0x0949] = 0x508c, [0x094a] = 0x508e, [0x094b] = 0x509d,
  [0x094c] = 0x5068, [0x094d] = 0x509c, [0x094e] = 0x5092, [0x094f] = 0x5082,
  [0x0950] = 0x5087, [0x0951] = 0x515f, [0x0952] = 0x51d4, [0x0953] = 0x5312,
  [0x0954] = 0x5311, [0x0955] = 0x53a4, [0x0956] = 0x53a7, [0x0957] = 0x5591,
  [0x0958] = 0x55a8, [0x0959] = 0x55a5, [0x095a] = 0x55ad, [0x095b] = 0x5577,
  [0x095c] = 0x5645, [0x095d] = 0x55a2, [0x095e] = 0x5593, [0x095f] = 0x5588,
  [0x0960] = 0x558f, [0x0961] = 0x55b5, [0x0962] = 0x5581, [0x0963] = 0x55a3,
  [0x0964] = 0x5592, [0x0965] = 0x55a4, [0x0966] = 0x557d, [0x0967] = 0x558c,
  [0x0968] = 0x55a6, [0x0969] = 0x557f, [0x096a] = 0x5595, [0x096b] = 0x55a1,
  [0x096c] = 0x558e, [0x096d] = 0x570c, [0x096e] = 0x5829, [0x096f] = 0x5837,
  [0x0970] = 0x5819, [0x0971] = 0x581e, [0x0972] = 0x5827, [0x0973] = 0x5823,
  [0x0974] = 0x5828, [0x0975] = 0x57f5, [0x0976] = 0x5848, [0x0977] = 0x5825,
  [0x0978] = 0x581c, [0x0979] = 0x581b, [0x097a] = 0x5833, [0x097b] = 0x583f,
  [0x097c] = 0x5836, [0x097d] = 0x582e, [0x097e] = 0x5839, [0x097f] = 0x5838,
  [0x0980] = 0x582d, [0x0981] = 0x582c, [0x0982] = 0x583b, [0x0983] = 0x5961,
  [0x0984] = 0x5aaf, [0x0985] = 0x5a94, [0x0986] = 0x5a9f, [0x0987] = 0x5a7a,
  [0x0988] = 0x5aa2, [0x0989] = 0x5a9e, [0x098a] = 0x5a78, [0x098b] = 0x5aa6,
  [0x098c] = 0x5a7c, [0x098d] = 0x5aa5, [0x098e] = 0x5aac, [0x098f] = 0x5a95,
  [0x0990] = 0x5aae, [0x0991] = 0x5a37, [0x0992] = 0x5a84, [0x0993] = 0x5a8a,
  [0x0994] = 0x5a97, [0x0995] = 0x5a83, [0x0996] = 0x5a8b, [0x0997] = 0x5aa9,
  [0x0998] = 0x5a7b, [0x0999] = 0x5a7d, [0x099a] = 0x5a8c, [0x099b] = 0x5a9c,
  [0x099c] = 0x5a8f, [0x099d] = 0x5a93, [0x099e] = 0x5a9d, [0x099f] = 0x5bea,
  [0x09a0] = 0x5bcd, [0x09a1] = 0x5bcb, [0x09a2] = 0x5bd4, [0x09a3] = 0x5bd1,
  [0x09a4] = 0x5bca, [0x09a5] = 0x5bce, [0x09a6] = 0x5c0c, [0x09a7] = 0x5c30,
  [0x09a8] = 0x5d37, [0x09a9] = 0x5d43, [0x09aa] = 0x5d6b, [0x09ab] = 0x5d41,
  [0x09ac] = 0x5d4b, [0x09ad] = 0x5d3f, [0x09ae] = 0x5d35, [0x09af] = 0x5d51,
  [0x09b0] = 0x5d4e, [0x09b1] = 0x5d55, [0x09b2] = 0x5d33, [0x09b3] = 0x5d3a,
  [0x09b4] = 0x5d52, [0x09b5] = 0x5d3d, [0x09b6] = 0x5d31, [0x09b7] = 0x5d59,
  [0x09b8] = 0x5d42, [0x09b9] = 0x5d39, [0x09ba] = 0x5d49, [0x09bb] = 0x5d38,
  [0x09bc] = 0x5d3c, [0x09bd] = 0x5d32, [0x09be] = 0x5d36, [0x09bf] = 0x5d40,
  [0x09c0] = 0x5d45, [0x09c1] = 0x5e44, [0x09c2] = 0x5e41, [0x09c3] = 0x5f58,
  [0x09c4] = 0x5fa6, [0x09c5] = 0x5fa5, [0x09c6] = 0x5fab, [0x09c7] = 0x60c9,
  [0x09c8] = 0x60b9, [0x09c9] = 0x60cc, [0x09ca] = 0x60e2, [0x09cb] = 0x60ce,
  [0x09cc] = 0x60c4, [0x09cd] = 0x6114, [0x09ce] = 0x60f2, [0x09cf] = 0x610a,
  [0x09d0] = 0x6116, [0x09d1] = 0x6105, [0x09d2] = 0x60f5, [0x09d3] = 0x6113,
  [0x09d4] = 0x60f8, [0x09d5] = 0x60fc, [0x09d6] = 0x60fe, [0x09d7] = 0x60c1,
  [0x09d8] = 0x6103, [0x09d9] = 0x6118, [0x09da] = 0x611d, [0x09db] = 0x6110,
  [0x09dc] = 0x60ff, [0x09dd] = 0x6104, [0x09de] = 0x610b, [0x09df] = 0x624a,
  [0x09e0] = 0x6394, [0x09e1] = 0x63b1, [0x09e2] = 0x63b0, [0x09e3] = 0x63ce,
  [0x09e4] = 0x63e5, [0x09e5] = 0x63e8, [0x09e6] = 0x63ef, [0x09e7] = 0x63c3,
  [0x09e8] = 0x649d, [0x09e9] = 0x63f3, [0x09ea] = 0x63ca, [0x09eb] = 0x63e0,
  [0x09ec] = 0x63f6, [0x09ed] = 0x63d5, [0x09ee] = 0x63f2, [0x09ef] = 0x63f5,
  [0x09f0] = 0x6461, [0x09f1] = 0x63df, [0x09f2] = 0x63be, [0x09f3] = 0x63dd,
  [0x09f4] = 0x63dc, [0x09f5] = 0x63c4, [0x09f6] = 0x63d8, [0x09f7] = 0x63d3,
  [0x09f8] = 0x63c2, [0x09f9] = 0x63c7, [0x09fa] = 0x63cc, [0x09fb] = 0x63cb,
  [0x09fc] = 0x63c8, [0x09fd] = 0x63f0, [0x09fe] = 0x63d7, [0x09ff] = 0x63d9,
  [0x0a00] = 0x6532, [0x0a01] = 0x6567, [0x0a02] = 0x656a, [0x0a03] = 0x6564,
  [0x0a04] = 0x655c, [0x0a05] = 0x6568, [0x0a06] = 0x6565, [0x0a07] = 0x658c,
  [0x0a08] = 0x659d, [0x0a09] = 0x659e, [0x0a0a] = 0x65ae, [0x0a0b] = 0x65d0,
  [0x0a0c] = 0x65d2, [0x0a0d] = 0x667c, [0x0a0e] = 0x666c, [0x0a0f] = 0x667b,
  [0x0a10] = 0x6680, [0x0a11] = 0x6671, [0x0a12] = 0x6679, [0x0a13] = 0x666a,
  [0x0a14] = 0x6672, [0x0a15] = 0x6701, [0x0a16] = 0x690c, [0x0a17] = 0x68d3,
  [0x0a18] = 0x6904, [0x0a19] = 0x68dc, [0x0a1a] = 0x692a, [0x0a1b] = 0x68ec,
  [0x0a1c] = 0x68ea, [0x0a1d] = 0x68f1, [0x0a1e] = 0x690f, [0x0a1f] = 0x68d6,
  [0x0a20] = 0x68f7, [0x0a21] = 0x68eb, [0x0a22] = 0x68e4, [0x0a23] = 0x68f6,
  [0x0a24] = 0x6913, [0x0a25] = 0x6910, [0x0a26] = 0x68f3, [0x0a27] = 0x68e1,
  [0x0a28] = 0x6907, [0x0a29] = 0x68cc, [0x0a2a] = 0x6908, [0x0a2b] = 0x6970,
  [0x0a2c] = 0x68b4, [0x0a2d] = 0x6911, [0x0a2e] = 0x68ef, [0x0a2f] = 0x68c6,
  [0x0a30] = 0x6914, [0x0a31] = 0x68f8, [0x0a32] = 0x68d0, [0x0a33] = 0x68fd,
  [0x0a34] = 0x68fc, [0x0a35] = 0x68e8, [0x0a36] = 0x690b, [0x0a37] = 0x690a,
  [0x0a38] = 0x6917, [0x0a39] = 0x68ce, [0x0a3a] = 0x68c8, [0x0a3b] = 0x68dd,
  [0x0a3c] = 0x68de, [0x0a3d] = 0x68e6, [0x0a3e] = 0x68f4, [0x0a3f] = 0x68d1,
  [0x0a40] = 0x6906, [0x0a41] = 0x68d4, [0x0a42] = 0x68e9, [0x0a43] = 0x6915,
  [0x0a44] = 0x6925, [0x0a45] = 0x68c7, [0x0a46] = 0x6b39, [0x0a47] = 0x6b3b,
  [0x0a48] = 0x6b3f, [0x0a49] = 0x6b3c, [0x0a4a] = 0x6b94, [0x0a4b] = 0x6b97,
  [0x0a4c] = 0x6b99, [0x0a4d] = 0x6b95, [0x0a4e] = 0x6bbd, [0x0a4f] = 0x6bf0,
  [0x0a50] = 0x6bf2, [0x0a51] = 0x6bf3, [0x0a52] = 0x6c30, [0x0a53] = 0x6dfc,
  [0x0a54] = 0x6e46, [0x0a55] = 0x6e47, [0x0a56] = 0x6e1f, [0x0a57] = 0x6e49,
  [0x0a58] = 0x6e88, [0x0a59] = 0x6e3c, [0x0a5a] = 0x6e3d, [0x0a5b] = 0x6e45,
  [0x0a5c] = 0x6e62, [0x0a5d] = 0x6e2b, [0x0a5e] = 0x6e3f, [0x0a5f] = 0x6e41,
  [0x0a60] = 0x6e5d, [0x0a61] = 0x6e73, [0x0a62] = 0x6e1c, [0x0a63] = 0x6e33,
  [0x0a64] = 0x6e4b, [0x0a65] = 0x6e40, [0x0a66] = 0x6e51, [0x0a67] = 0x6e3b,
  [0x0a68] = 0x6e03, [0x0a69] = 0x6e2e, [0x0a6a] = 0x6e5e, [0x0a6b] = 0x6e68,
  [0x0a6c] = 0x6e5c, [0x0a6d] = 0x6e61, [0x0a6e] = 0x6e31, [0x0a6f] = 0x6e28,
  [0x0a70] = 0x6e60, [0x0a71] = 0x6e71, [0x0a72] = 0x6e6b, [0x0a73] = 0x6e39,
  [0x0a74] = 0x6e22, [0x0a75] = 0x6e30, [0x0a76] = 0x6e53, [0x0a77] = 0x6e65,
  [0x0a78] = 0x6e27, [0x0a79] = 0x6e78, [0x0a7a] = 0x6e64, [0x0a7b] = 0x6e77,
  [0x0a7c] = 0x6e55, [0x0a7d] = 0x6e79, [0x0a7e] = 0x6e52, [0x0a7f] = 0x6e66,
  [0x0a80] = 0x6e35, [0x0a81] = 0x6e36, [0x0a82] = 0x6e5a, [0x0a83] = 0x7120,
  [0x0a84] = 0x711e, [0x0a85] = 0x712f, [0x0a86] = 0x70fb, [0x0a87] = 0x712e,
  [0x0a88] = 0x7131, [0x0a89] = 0x7123, [0x0a8a] = 0x7125, [0x0a8b] = 0x7122,
  [0x0a8c] = 0x7132, [0x0a8d] = 0x711f, [0x0a8e] = 0x7128, [0x0a8f] = 0x713a,
  [0x0a90] = 0x711b, [0x0a91] = 0x724b, [0x0a92] = 0x725a, [0x0a93] = 0x7288,
  [0x0a94] = 0x7289, [0x0a95] = 0x7286, [0x0a96] = 0x7285, [0x0a97] = 0x728b,
  [0x0a98] = 0x7312, [0x0a99] = 0x730b, [0x0a9a] = 0x7330, [0x0a9b] = 0x7322,
  [0x0a9c] = 0x7331, [0x0a9d] = 0x7333, [0x0a9e] = 0x7327, [0x0a9f] = 0x7332,
  [0x0aa0] = 0x732d, [0x0aa1] = 0x7326, [0x0aa2] = 0x7323, [0x0aa3] = 0x7335,
  [0x0aa4] = 0x730c, [0x0aa5] = 0x742e, [0x0aa6] = 0x742c, [0x0aa7] = 0x7430,
  [0x0aa8] = 0x742b, [0x0aa9] = 0x7416, [0x0aaa] = 0x741a, [0x0aab] = 0x7421,
  [0x0aac] = 0x742d, [0x0aad] = 0x7431, [0x0aae] = 0x7424, [0x0aaf] = 0x7423,
  [0x0ab0] = 0x741d, [0x0ab1] = 0x7429, [0x0ab2] = 0x7420, [0x0ab3] = 0x7432,
  [0x0ab4] = 0x74fb, [0x0ab5] = 0x752f, [0x0ab6] = 0x756f, [0x0ab7] = 0x756c,
  [0x0ab8] = 0x75e7, [0x0ab9] = 0x75da, [0x0aba] = 0x75e1, [0x0abb] = 0x75e6,
  [0x0abc] = 0x75dd, [0x0abd] = 0x75df, [0x0abe] = 0x75e4, [0x0abf] = 0x75d7,
  [0x0ac0] = 0x7695, [0x0ac1] = 0x7692, [0x0ac2] = 0x76da, [0x0ac3] = 0x7746,
  [0x0ac4] = 0x7747, [0x0ac5] = 0x7744, [0x0ac6] = 0x774d, [0x0ac7] = 0x7745,
  [0x0ac8] = 0x774a, [0x0ac9] = 0x774e, [0x0aca] = 0x774b, [0x0acb] = 0x774c,
  [0x0acc] = 0x77de, [0x0acd] = 0x77ec, [0x0ace] = 0x7860, [0x0acf] = 0x7864,
  [0x0ad0] = 0x7865, [0x0ad1] = 0x785c, [0x0ad2] = 0x786d, [0x0ad3] = 0x7871,
  [0x0ad4] = 0x786a, [0x0ad5] = 0x786e, [0x0ad6] = 0x7870, [0x0ad7] = 0x7869,
  [0x0ad8] = 0x7868, [0x0ad9] = 0x785e, [0x0ada] = 0x7862, [0x0adb] = 0x7974,
  [0x0adc] = 0x7973, [0x0add] = 0x7972, [0x0ade] = 0x7970, [0x0adf] = 0x7a02,
  [0x0ae0] = 0x7a0a, [0x0ae1] = 0x7a03, [0x0ae2] = 0x7a0c, [0x0ae3] = 0x7a04,
  [0x0ae4] = 0x7a99, [0x0ae5] = 0x7ae6, [0x0ae6] = 0x7ae4, [0x0ae7] = 0x7b4a,
  [0x0ae8] = 0x7b47, [0x0ae9] = 0x7b44, [0x0aea] = 0x7b48, [0x0aeb] = 0x7b4c,
  [0x0aec] = 0x7b4e, [0x0aed] = 0x7b40, [0x0aee] = 0x7b58, [0x0aef] = 0x7b45,
  [0x0af0] = 0x7ca2, [0x0af1] = 0x7c9e, [0x0af2] = 0x7ca8, [0x0af3] = 0x7ca1,
  [0x0af4] = 0x7d58, [0x0af5] = 0x7d6f, [0x0af6] = 0x7d63, [0x0af7] = 0x7d53,
  [0x0af8] = 0x7d56, [0x0af9] = 0x7d67, [0x0afa] = 0x7d6a, [0x0afb] = 0x7d4f,
  [0x0afc] = 0x7d6d, [0x0afd] = 0x7d5c, [0x0afe] = 0x7d6b, [0x0aff] = 0x7d52,
  [0x0b00] = 0x7d54, [0x0b01] = 0x7d69, [0x0b02] = 0x7d51, [0x0b03] = 0x7d5f,
  [0x0b04] = 0x7d4e, [0x0b05] = 0x7f3e, [0x0b06] = 0x7f3f, [0x0b07] = 0x7f65,
  [0x0b08] = 0x7f66, [0x0b09] = 0x7fa2, [0x0b0a] = 0x7fa0, [0x0b0b] = 0x7fa1,
  [0x0b0c] = 0x7fd7, [0x0b0d] = 0x8051, [0x0b0e] = 0x804f, [0x0b0f] = 0x8050,
  [0x0b10] = 0x80fe, [0x0b11] = 0x80d4, [0x0b12] = 0x8143, [0x0b13] = 0x814a,
  [0x0b14] = 0x8152, [0x0b15] = 0x814f, [0x0b16] = 0x8147, [0x0b17] = 0x813d,
  [0x0b18] = 0x814d, [0x0b19] = 0x813a, [0x0b1a] = 0x81e6, [0x0b1b] = 0x81ee,
  [0x0b1c] = 0x81f7, [0x0b1d] = 0x81f8, [0x0b1e] = 0x81f9, [0x0b1f] = 0x8204,
  [0x0b20] = 0x823c, [0x0b21] = 0x823d, [0x0b22] = 0x823f, [0x0b23] = 0x8275,
  [0x0b24] = 0x833b, [0x0b25] = 0x83cf, [0x0b26] = 0x83f9, [0x0b27] = 0x8423,
  [0x0b28] = 0x83c0, [0x0b29] = 0x83e8, [0x0b2a] = 0x8412, [0x0b2b] = 0x83e7,
  [0x0b2c] = 0x83e4, [0x0b2d] = 0x83fc, [0x0b2e] = 0x83f6, [0x0b2f] = 0x8410,
  [0x0b30] = 0x83c6, [0x0b31] = 0x83c8, [0x0b32] = 0x83eb, [0x0b33] = 0x83e3,
  [0x0b34] = 0x83bf, [0x0b35] = 0x8401, [0x0b36] = 0x83dd, [0x0b37] = 0x83e5,
  [0x0b38] = 0x83d8, [0x0b39] = 0x83ff, [0x0b3a] = 0x83e1, [0x0b3b] = 0x83cb,
  [0x0b3c] = 0x83ce, [0x0b3d] = 0x83d6, [0x0b3e] = 0x83f5, [0x0b3f] = 0x83c9,
  [0x0b40] = 0x8409, [0x0b41] = 0x840f, [0x0b42] = 0x83de, [0x0b43] = 0x8411,
  [0x0b44] = 0x8406, [0x0b45] = 0x83c2, [0x0b46] = 0x83f3, [0x0b47] = 0x83d5,
  [0x0b48] = 0x83fa, [0x0b49] = 0x83c7, [0x0b4a] = 0x83d1, [0x0b4b] = 0x83ea,
  [0x0b4c] = 0x8413, [0x0b4d] = 0x839a, [0x0b4e] = 0x83c3, [0x0b4f] = 0x83ec,
  [0x0b50] = 0x83ee, [0x0b51] = 0x83c4, [0x0b52] = 0x83fb, [0x0b53] = 0x83d7,
  [0x0b54] = 0x83e2, [0x0b55] = 0x841b, [0x0b56] = 0x83db, [0x0b57] = 0x83fe,
  [0x0b58] = 0x86d8, [0x0b59] = 0x86e2, [0x0b5a] = 0x86e6, [0x0b5b] = 0x86d3,
  [0x0b5c] = 0x86e3, [0x0b5d] = 0x86da, [0x0b5e] = 0x86ea, [0x0b5f] = 0x86dd,
  [0x0b60] = 0x86eb, [0x0b61] = 0x86dc, [0x0b62] = 0x86ec, [0x0b63] = 0x86e9,
  [0x0b64] = 0x86d7, [0x0b65] = 0x86e8, [0x0b66] = 0x86d1, [0x0b67] = 0x8848,
  [0x0b68] = 0x8856, [0x0b69] = 0x8855, [0x0b6a] = 0x88ba, [0x0b6b] = 0x88d7,
  [0x0b6c] = 0x88b9, [0x0b6d] = 0x88b8, [0x0b6e] = 0x88c0, [0x0b6f] = 0x88be,
  [0x0b70] = 0x88b6, [0x0b71] = 0x88bc, [0x0b72] = 0x88b7, [0x0b73] = 0x88bd,
  [0x0b74] = 0x88b2, [0x0b75] = 0x8901, [0x0b76] = 0x88c9, [0x0b77] = 0x8995,
  [0x0b78] = 0x8998, [0x0b79] = 0x8997, [0x0b7a] = 0x89dd, [0x0b7b] = 0x89da,
  [0x0b7c] = 0x89db, [0x0b7d] = 0x8a4e, [0x0b7e] = 0x8a4d, [0x0b7f] = 0x8a39,
  [0x0b80] = 0x8a59, [0x0b81] = 0x8a40, [0x0b82] = 0x8a57, [0x0b83] = 0x8a58,
  [0x0b84] = 0x8a44, [0x0b85] = 0x8a45, [0x0b86] = 0x8a52, [0x0b87] = 0x8a48,
  [0x0b88] = 0x8a51, [0x0b89] = 0x8a4a, [0x0b8a] = 0x8a4c, [0x0b8b] = 0x8a4f,
  [0x0b8c] = 0x8c5f, [0x0b8d] = 0x8c81, [0x0b8e] = 0x8c80, [0x0b8f] = 0x8cba,
  [0x0b90] = 0x8cbe, [0x0b91] = 0x8cb0, [0x0b92] = 0x8cb9, [0x0b93] = 0x8cb5,
  [0x0b94] = 0x8d84, [0x0b95] = 0x8d80, [0x0b96] = 0x8d89, [0x0b97] = 0x8dd8,
  [0x0b98] = 0x8dd3, [0x0b99] = 0x8dcd, [0x0b9a] = 0x8dc7, [0x0b9b] = 0x8dd6,
  [0x0b9c] = 0x8ddc, [0x0b9d] = 0x8dcf, [0x0b9e] = 0x8dd5, [0x0b9f] = 0x8dd9,
  [0x0ba0] = 0x8dc8, [0x0ba1] = 0x8dd7, [0x0ba2] = 0x8dc5, [0x0ba3] = 0x8eef,
  [0x0ba4] = 0x8ef7, [0x0ba5] = 0x8efa, [0x0ba6] = 0x8ef9, [0x0ba7] = 0x8ee6,
  [0x0ba8] = 0x8eee, [0x0ba9] = 0x8ee5, [0x0baa] = 0x8ef5, [0x0bab] = 0x8ee7,
  [0x0bac] = 0x8ee8, [0x0bad] = 0x8ef6, [0x0bae] = 0x8eeb, [0x0baf] = 0x8ef1,
  [0x0bb0] = 0x8eec, [0x0bb1] = 0x8ef4, [0x0bb2] = 0x8ee9, [0x0bb3] = 0x902d,
  [0x0bb4] = 0x9034, [0x0bb5] = 0x902f, [0x0bb6] = 0x9106, [0x0bb7] = 0x912c,
  [0x0bb8] = 0x9104, [0x0bb9] = 0x90ff, [0x0bba] = 0x90fc, [0x0bbb] = 0x9108,
  [0x0bbc] = 0x90f9, [0x0bbd] = 0x90fb, [0x0bbe] = 0x9101, [0x0bbf] = 0x9100,
  [0x0bc0] = 0x9107, [0x0bc1] = 0x9105, [0x0bc2] = 0x9103, [0x0bc3] = 0x9161,
  [0x0bc4] = 0x9164, [0x0bc5] = 0x915f, [0x0bc6] = 0x9162, [0x0bc7] = 0x9160,
  [0x0bc8] = 0x9201, [0x0bc9] = 0x920a, [0x0bca] = 0x9225, [0x0bcb] = 0x9203,
  [0x0bcc] = 0x921a, [0x0bcd] = 0x9226, [0x0bce] = 0x920f, [0x0bcf] = 0x920c,
  [0x0bd0] = 0x9200, [0x0bd1] = 0x9212, [0x0bd2] = 0x91ff, [0x0bd3] = 0x91fd,
  [0x0bd4] = 0x9206, [0x0bd5] = 0x9204, [0x0bd6] = 0x9227, [0x0bd7] = 0x9202,
  [0x0bd8] = 0x921c, [0x0bd9] = 0x9224, [0x0bda] = 0x9219, [0x0bdb] = 0x9217,
  [0x0bdc] = 0x9205, [0x0bdd] = 0x9216, [0x0bde] = 0x957b, [0x0bdf] = 0x958d,
  [0x0be0] = 0x958c, [0x0be1] = 0x9590, [0x0be2] = 0x9687, [0x0be3] = 0x967e,
  [0x0be4] = 0x9688, [0x0be5] = 0x9689, [0x0be6] = 0x9683, [0x0be7] = 0x9680,
  [0x0be8] = 0x96c2, [0x0be9] = 0x96c8, [0x0bea] = 0x96c3, [0x0beb] = 0x96f1,
  [0x0bec] = 0x96f0, [0x0bed] = 0x976c, [0x0bee] = 0x9770, [0x0bef] = 0x976e,
  [0x0bf0] = 0x9807, [0x0bf1] = 0x98a9, [0x0bf2] = 0x98eb, [0x0bf3] = 0x9ce6,
  [0x0bf4] = 0x9ef9, [0x0bf5] = 0x4e83, [0x0bf6] = 0x4e84, [0x0bf7] = 0x4eb6,
  [0x0bf8] = 0x50bd, [0x0bf9] = 0x50bf, [0x0bfa] = 0x50c6, [0x0bfb] = 0x50ae,
  [0x0bfc] = 0x50c4, [0x0bfd] = 0x50ca, [0x0bfe] = 0x50b4, [0x0bff] = 0x50c8,
  [0x0c00] = 0x50c2, [0x0c01] = 0x50b0, [0x0c02] = 0x50c1, [0x0c03] = 0x50ba,
  [0x0c04] = 0x50b1, [0x0c05] = 0x50cb, [0x0c06] = 0x50c9, [0x0c07] = 0x50b6,
  [0x0c08] = 0x50b8, [0x0c09] = 0x51d7, [0x0c0a] = 0x527a, [0x0c0b] = 0x5278,
  [0x0c0c] = 0x527b, [0x0c0d] = 0x527c, [0x0c0e] = 0x55c3, [0x0c0f] = 0x55db,
  [0x0c10] = 0x55cc, [0x0c11] = 0x55d0, [0x0c12] = 0x55cb, [0x0c13] = 0x55ca,
  [0x0c14] = 0x55dd, [0x0c15] = 0x55c0, [0x0c16] = 0x55d4, [0x0c17] = 0x55c4,
  [0x0c18] = 0x55e9, [0x0c19] = 0x55bf, [0x0c1a] = 0x55d2, [0x0c1b] = 0x558d,
  [0x0c1c] = 0x55cf, [0x0c1d] = 0x55d5, [0x0c1e] = 0x55e2, [0x0c1f] = 0x55d6,
  [0x0c20] = 0x55c8, [0x0c21] = 0x55f2, [0x0c22] = 0x55cd, [0x0c23] = 0x55d9,
  [0x0c24] = 0x55c2, [0x0c25] = 0x5714, [0x0c26] = 0x5853, [0x0c27] = 0x5868,
  [0x0c28] = 0x5864, [0x0c29] = 0x584f, [0x0c2a] = 0x584d, [0x0c2b] = 0x5849,
  [0x0c2c] = 0x586f, [0x0c2d] = 0x5855, [0x0c2e] = 0x584e, [0x0c2f] = 0x585d,
  [0x0c30] = 0x5859, [0x0c31] = 0x5865, [0x0c32] = 0x585b, [0x0c33] = 0x583d,
  [0x0c34] = 0x5863, [0x0c35] = 0x5871, [0x0c36] = 0x58fc, [0x0c37] = 0x5ac7,
  [0x0c38] = 0x5ac4, [0x0c39] = 0x5acb, [0x0c3a] = 0x5aba, [0x0c3b] = 0x5ab8,
  [0x0c3c] = 0x5ab1, [0x0c3d] = 0x5ab5, [0x0c3e] = 0x5ab0, [0x0c3f] = 0x5abf,
  [0x0c40] = 0x5ac8, [0x0c41] = 0x5abb, [0x0c42] = 0x5ac6, [0x0c43] = 0x5ab7,
  [0x0c44] = 0x5ac0, [0x0c45] = 0x5aca, [0x0c46] = 0x5ab4, [0x0c47] = 0x5ab6,
  [0x0c48] = 0x5acd, [0x0c49] = 0x5ab9, [0x0c4a] = 0x5a90, [0x0c4b] = 0x5bd6,
  [0x0c4c] = 0x5bd8, [0x0c4d] = 0x5bd9, [0x0c4e] = 0x5c1f, [0x0c4f] = 0x5c33,
  [0x0c50] = 0x5d71, [0x0c51] = 0x5d63, [0x0c52] = 0x5d4a, [0x0c53] = 0x5d65,
  [0x0c54] = 0x5d72, [0x0c55] = 0x5d6c, [0x0c56] = 0x5d5e, [0x0c57] = 0x5d68,
  [0x0c58] = 0x5d67, [0x0c59] = 0x5d62, [0x0c5a] = 0x5df0, [0x0c5b] = 0x5e4f,
  [0x0c5c] = 0x5e4e, [0x0c5d] = 0x5e4a, [0x0c5e] = 0x5e4d, [0x0c5f] = 0x5e4b,
  [0x0c60] = 0x5ec5, [0x0c61] = 0x5ecc, [0x0c62] = 0x5ec6, [0x0c63] = 0x5ecb,
  [0x0c64] = 0x5ec7, [0x0c65] = 0x5f40, [0x0c66] = 0x5faf, [0x0c67] = 0x5fad,
  [0x0c68] = 0x60f7, [0x0c69] = 0x6149, [0x0c6a] = 0x614a, [0x0c6b] = 0x612b,
  [0x0c6c] = 0x6145, [0x0c6d] = 0x6136, [0x0c6e] = 0x6132, [0x0c6f] = 0x612e,
  [0x0c70] = 0x6146, [0x0c71] = 0x612f, [0x0c72] = 0x614f, [0x0c73] = 0x6129,
  [0x0c74] = 0x6140, [0x0c75] = 0x6220, [0x0c76] = 0x9168, [0x0c77] = 0x6223,
  [0x0c78] = 0x6225, [0x0c79] = 0x6224, [0x0c7a] = 0x63c5, [0x0c7b] = 0x63f1,
  [0x0c7c] = 0x63eb, [0x0c7d] = 0x6410, [0x0c7e] = 0x6412, [0x0c7f] = 0x6409,
  [0x0c80] = 0x6420, [0x0c81] = 0x6424, [0x0c82] = 0x6433, [0x0c83] = 0x6443,
  [0x0c84] = 0x641f, [0x0c85] = 0x6415, [0x0c86] = 0x6418, [0x0c87] = 0x6439,
  [0x0c88] = 0x6437, [0x0c89] = 0x6422, [0x0c8a] = 0x6423, [0x0c8b] = 0x640c,
  [0x0c8c] = 0x6426, [0x0c8d] = 0x6430, [0x0c8e] = 0x6428, [0x0c8f] = 0x6441,
  [0x0c90] = 0x6435, [0x0c91] = 0x642f, [0x0c92] = 0x640a, [0x0c93] = 0x641a,
  [0x0c94] = 0x6440, [0x0c95] = 0x6425, [0x0c96] = 0x6427, [0x0c97] = 0x640b,
  [0x0c98] = 0x63e7, [0x0c99] = 0x641b, [0x0c9a] = 0x642e, [0x0c9b] = 0x6421,
  [0x0c9c] = 0x640e, [0x0c9d] = 0x656f, [0x0c9e] = 0x6592, [0x0c9f] = 0x65d3,
  [0x0ca0] = 0x6686, [0x0ca1] = 0x668c, [0x0ca2] = 0x6695, [0x0ca3] = 0x6690,
  [0x0ca4] = 0x668b, [0x0ca5] = 0x668a, [0x0ca6] = 0x6699, [0x0ca7] = 0x6694,
  [0x0ca8] = 0x6678, [0x0ca9] = 0x6720, [0x0caa] = 0x6966, [0x0cab] = 0x695f,
  [0x0cac] = 0x6938, [0x0cad] = 0x694e, [0x0cae] = 0x6962, [0x0caf] = 0x6971,
  [0x0cb0] = 0x693f, [0x0cb1] = 0x6945, [0x0cb2] = 0x696a, [0x0cb3] = 0x6939,
  [0x0cb4] = 0x6942, [0x0cb5] = 0x6957, [0x0cb6] = 0x6959, [0x0cb7] = 0x697a,
  [0x0cb8] = 0x6948, [0x0cb9] = 0x6949, [0x0cba] = 0x6935, [0x0cbb] = 0x696c,
  [0x0cbc] = 0x6933, [0x0cbd] = 0x693d, [0x0cbe] = 0x6965, [0x0cbf] = 0x68f0,
  [0x0cc0] = 0x6978, [0x0cc1] = 0x6934, [0x0cc2] = 0x6969, [0x0cc3] = 0x6940,
  [0x0cc4] = 0x696f, [0x0cc5] = 0x6944, [0x0cc6] = 0x6976, [0x0cc7] = 0x6958,
  [0x0cc8] = 0x6941, [0x0cc9] = 0x6974, [0x0cca] = 0x694c, [0x0ccb] = 0x693b,
  [0x0ccc] = 0x694b, [0x0ccd] = 0x6937, [0x0cce] = 0x695c, [0x0ccf] = 0x694f,
  [0x0cd0] = 0x6951, [0x0cd1] = 0x6932, [0x0cd2] = 0x6952, [0x0cd3] = 0x692f,
  [0x0cd4] = 0x697b, [0x0cd5] = 0x693c, [0x0cd6] = 0x6b46, [0x0cd7] = 0x6b45,
  [0x0cd8] = 0x6b43, [0x0cd9] = 0x6b42, [0x0cda] = 0x6b48, [0x0cdb] = 0x6b41,
  [0x0cdc] = 0x6b9b, [0x0cdd] = 0x6bfb, [0x0cde] = 0x6bfc, [0x0cdf] = 0x6bf9,
  [0x0ce0] = 0x6bf7, [0x0ce1] = 0x6bf8, [0x0ce2] = 0x6e9b, [0x0ce3] = 0x6ed6,
  [0x0ce4] = 0x6ec8, [0x0ce5] = 0x6e8f, [0x0ce6] = 0x6ec0, [0x0ce7] = 0x6e9f,
  [0x0ce8] = 0x6e93, [0x0ce9] = 0x6e94, [0x0cea] = 0x6ea0, [0x0ceb] = 0x6eb1,
  [0x0cec] = 0x6eb9, [0x0ced] = 0x6ec6, [0x0cee] = 0x6ed2, [0x0cef] = 0x6ebd,
  [0x0cf0] = 0x6ec1, [0x0cf1] = 0x6e9e, [0x0cf2] = 0x6ec9, [0x0cf3] = 0x6eb7,
  [0x0cf4] = 0x6eb0, [0x0cf5] = 0x6ecd, [0x0cf6] = 0x6ea6, [0x0cf7] = 0x6ecf,
  [0x0cf8] = 0x6eb2, [0x0cf9] = 0x6ebe, [0x0cfa] = 0x6ec3, [0x0cfb] = 0x6edc,
  [0x0cfc] = 0x6ed8, [0x0cfd] = 0x6e99, [0x0cfe] = 0x6e92, [0x0cff] = 0x6e8e,
  [0x0d00] = 0x6e8d, [0x0d01] = 0x6ea4, [0x0d02] = 0x6ea1, [0x0d03] = 0x6ebf,
  [0x0d04] = 0x6eb3, [0x0d05] = 0x6ed0, [0x0d06] = 0x6eca, [0x0d07] = 0x6e97,
  [0x0d08] = 0x6eae, [0x0d09] = 0x6ea3, [0x0d0a] = 0x7147, [0x0d0b] = 0x7154,
  [0x0d0c] = 0x7152, [0x0d0d] = 0x7163, [0x0d0e] = 0x7160, [0x0d0f] = 0x7141,
  [0x0d10] = 0x715d, [0x0d11] = 0x7162, [0x0d12] = 0x7172, [0x0d13] = 0x7178,
  [0x0d14] = 0x716a, [0x0d15] = 0x7161, [0x0d16] = 0x7142, [0x0d17] = 0x7158,
  [0x0d18] = 0x7143, [0x0d19] = 0x714b, [0x0d1a] = 0x7170, [0x0d1b] = 0x715f,
  [0x0d1c] = 0x7150, [0x0d1d] = 0x7153, [0x0d1e] = 0x7144, [0x0d1f] = 0x714d,
  [0x0d20] = 0x715a, [0x0d21] = 0x724f, [0x0d22] = 0x728d, [0x0d23] = 0x728c,
  [0x0d24] = 0x7291, [0x0d25] = 0x7290, [0x0d26] = 0x728e, [0x0d27] = 0x733c,
  [0x0d28] = 0x7342, [0x0d29] = 0x733b, [0x0d2a] = 0x733a, [0x0d2b] = 0x7340,
  [0x0d2c] = 0x734a, [0x0d2d] = 0x7349, [0x0d2e] = 0x7444, [0x0d2f] = 0x744a,
  [0x0d30] = 0x744b, [0x0d31] = 0x7452, [0x0d32] = 0x7451, [0x0d33] = 0x7457,
  [0x0d34] = 0x7440, [0x0d35] = 0x744f, [0x0d36] = 0x7450, [0x0d37] = 0x744e,
  [0x0d38] = 0x7442, [0x0d39] = 0x7446, [0x0d3a] = 0x744d, [0x0d3b] = 0x7454,
  [0x0d3c] = 0x74e1, [0x0d3d] = 0x74ff, [0x0d3e] = 0x74fe, [0x0d3f] = 0x74fd,
  [0x0d40] = 0x751d, [0x0d41] = 0x7579, [0x0d42] = 0x7577, [0x0d43] = 0x6983,
  [0x0d44] = 0x75ef, [0x0d45] = 0x760f, [0x0d46] = 0x7603, [0x0d47] = 0x75f7,
  [0x0d48] = 0x75fe, [0x0d49] = 0x75fc, [0x0d4a] = 0x75f9, [0x0d4b] = 0x75f8,
  [0x0d4c] = 0x7610, [0x0d4d] = 0x75fb, [0x0d4e] = 0x75f6, [0x0d4f] = 0x75ed,
  [0x0d50] = 0x75f5, [0x0d51] = 0x75fd, [0x0d52] = 0x7699, [0x0d53] = 0x76b5,
  [0x0d54] = 0x76dd, [0x0d55] = 0x7755, [0x0d56] = 0x775f, [0x0d57] = 0x7760,
  [0x0d58] = 0x7752, [0x0d59] = 0x7756, [0x0d5a] = 0x775a, [0x0d5b] = 0x7769,
  [0x0d5c] = 0x7767, [0x0d5d] = 0x7754, [0x0d5e] = 0x7759, [0x0d5f] = 0x776d,
  [0x0d60] = 0x77e0, [0x0d61] = 0x7887, [0x0d62] = 0x789a, [0x0d63] = 0x7894,
  [0x0d64] = 0x788f, [0x0d65] = 0x7884, [0x0d66] = 0x7895, [0x0d67] = 0x7885,
  [0x0d68] = 0x7886, [0x0d69] = 0x78a1, [0x0d6a] = 0x7883, [0x0d6b] = 0x7879,
  [0x0d6c] = 0x7899, [0x0d6d] = 0x7880, [0x0d6e] = 0x7896, [0x0d6f] = 0x787b,
  [0x0d70] = 0x797c, [0x0d71] = 0x7982, [0x0d72] = 0x797d, [0x0d73] = 0x7979,
  [0x0d74] = 0x7a11, [0x0d75] = 0x7a18, [0x0d76] = 0x7a19, [0x0d77] = 0x7a12,
  [0x0d78] = 0x7a17, [0x0d79] = 0x7a15, [0x0d7a] = 0x7a22, [0x0d7b] = 0x7a13,
  [0x0d7c] = 0x7a1b, [0x0d7d] = 0x7a10, [0x0d7e] = 0x7aa3, [0x0d7f] = 0x7aa2,
  [0x0d80] = 0x7a9e, [0x0d81] = 0x7aeb, [0x0d82] = 0x7b66, [0x0d83] = 0x7b64,
  [0x0d84] = 0x7b6d, [0x0d85] = 0x7b74, [0x0d86] = 0x7b69, [0x0d87] = 0x7b72,
  [0x0d88] = 0x7b65, [0x0d89] = 0x7b73, [0x0d8a] = 0x7b71, [0x0d8b] = 0x7b70,
  [0x0d8c] = 0x7b61, [0x0d8d] = 0x7b78, [0x0d8e] = 0x7b76, [0x0d8f] = 0x7b63,
  [0x0d90] = 0x7cb2, [0x0d91] = 0x7cb4, [0x0d92] = 0x7caf, [0x0d93] = 0x7d88,
  [0x0d94] = 0x7d86, [0x0d95] = 0x7d80, [0x0d96] = 0x7d8d, [0x0d97] = 0x7d7f,
  [0x0d98] = 0x7d85, [0x0d99] = 0x7d7a, [0x0d9a] = 0x7d8e, [0x0d9b] = 0x7d7b,
  [0x0d9c] = 0x7d83, [0x0d9d] = 0x7d7c, [0x0d9e] = 0x7d8c, [0x0d9f] = 0x7d94,
  [0x0da0] = 0x7d84, [0x0da1] = 0x7d7d, [0x0da2] = 0x7d92, [0x0da3] = 0x7f6d,
  [0x0da4] = 0x7f6b, [0x0da5] = 0x7f67, [0x0da6] = 0x7f68, [0x0da7] = 0x7f6c,
  [0x0da8] = 0x7fa6, [0x0da9] = 0x7fa5, [0x0daa] = 0x7fa7, [0x0dab] = 0x7fdb,
  [0x0dac] = 0x7fdc, [0x0dad] = 0x8021, [0x0dae] = 0x8164, [0x0daf] = 0x8160,
  [0x0db0] = 0x8177, [0x0db1] = 0x815c, [0x0db2] = 0x8169, [0x0db3] = 0x815b,
  [0x0db4] = 0x8162, [0x0db5] = 0x8172, [0x0db6] = 0x6721, [0x0db7] = 0x815e,
  [0x0db8] = 0x8176, [0x0db9] = 0x8167, [0x0dba] = 0x816f, [0x0dbb] = 0x8144,
  [0x0dbc] = 0x8161, [0x0dbd] = 0x821d, [0x0dbe] = 0x8249, [0x0dbf] = 0x8244,
  [0x0dc0] = 0x8240, [0x0dc1] = 0x8242, [0x0dc2] = 0x8245, [0x0dc3] = 0x84f1,
  [0x0dc4] = 0x843f, [0x0dc5] = 0x8456, [0x0dc6] = 0x8476, [0x0dc7] = 0x8479,
  [0x0dc8] = 0x848f, [0x0dc9] = 0x848d, [0x0dca] = 0x8465, [0x0dcb] = 0x8451,
  [0x0dcc] = 0x8440, [0x0dcd] = 0x8486, [0x0dce] = 0x8467, [0x0dcf] = 0x8430,
  [0x0dd0] = 0x844d, [0x0dd1] = 0x847d, [0x0dd2] = 0x845a, [0x0dd3] = 0x8459,
  [0x0dd4] = 0x8474, [0x0dd5] = 0x8473, [0x0dd6] = 0x845d, [0x0dd7] = 0x8507,
  [0x0dd8] = 0x845e, [0x0dd9] = 0x8437, [0x0dda] = 0x843a, [0x0ddb] = 0x8434,
  [0x0ddc] = 0x847a, [0x0ddd] = 0x8443, [0x0dde] = 0x8478, [0x0ddf] = 0x8432,
  [0x0de0] = 0x8445, [0x0de1] = 0x8429, [0x0de2] = 0x83d9, [0x0de3] = 0x844b,
  [0x0de4] = 0x842f, [0x0de5] = 0x8442, [0x0de6] = 0x842d, [0x0de7] = 0x845f,
  [0x0de8] = 0x8470, [0x0de9] = 0x8439, [0x0dea] = 0x844e, [0x0deb] = 0x844c,
  [0x0dec] = 0x8452, [0x0ded] = 0x846f, [0x0dee] = 0x84c5, [0x0def] = 0x848e,
  [0x0df0] = 0x843b, [0x0df1] = 0x8447, [0x0df2] = 0x8436, [0x0df3] = 0x8433,
  [0x0df4] = 0x8468, [0x0df5] = 0x847e, [0x0df6] = 0x8444, [0x0df7] = 0x842b,
  [0x0df8] = 0x8460, [0x0df9] = 0x8454, [0x0dfa] = 0x846e, [0x0dfb] = 0x8450,
  [0x0dfc] = 0x870b, [0x0dfd] = 0x8704, [0x0dfe] = 0x86f7, [0x0dff] = 0x870c,
  [0x0e00] = 0x86fa, [0x0e01] = 0x86d6, [0x0e02] = 0x86f5, [0x0e03] = 0x874d,
  [0x0e04] = 0x86f8, [0x0e05] = 0x870e, [0x0e06] = 0x8709, [0x0e07] = 0x8701,
  [0x0e08] = 0x86f6, [0x0e09] = 0x870d, [0x0e0a] = 0x8705, [0x0e0b] = 0x88d6,
  [0x0e0c] = 0x88cb, [0x0e0d] = 0x88cd, [0x0e0e] = 0x88ce, [0x0e0f] = 0x88de,
  [0x0e10] = 0x88db, [0x0e11] = 0x88da, [0x0e12] = 0x88cc, [0x0e13] = 0x88d0,
  [0x0e14] = 0x8985, [0x0e15] = 0x899b, [0x0e16] = 0x89df, [0x0e17] = 0x89e5,
  [0x0e18] = 0x89e4, [0x0e19] = 0x89e1, [0x0e1a] = 0x89e0, [0x0e1b] = 0x89e2,
  [0x0e1c] = 0x89dc, [0x0e1d] = 0x89e6, [0x0e1e] = 0x8a76, [0x0e1f] = 0x8a86,
  [0x0e20] = 0x8a7f, [0x0e21] = 0x8a61, [0x0e22] = 0x8a3f, [0x0e23] = 0x8a77,
  [0x0e24] = 0x8a82, [0x0e25] = 0x8a84, [0x0e26] = 0x8a75, [0x0e27] = 0x8a83,
  [0x0e28] = 0x8a81, [0x0e29] = 0x8a74, [0x0e2a] = 0x8a7a, [0x0e2b] = 0x8c3c,
  [0x0e2c] = 0x8c4b, [0x0e2d] = 0x8c4a, [0x0e2e] = 0x8c65, [0x0e2f] = 0x8c64,
  [0x0e30] = 0x8c66, [0x0e31] = 0x8c86, [0x0e32] = 0x8c84, [0x0e33] = 0x8c85,
  [0x0e34] = 0x8ccc, [0x0e35] = 0x8d68, [0x0e36] = 0x8d69, [0x0e37] = 0x8d91,
  [0x0e38] = 0x8d8c, [0x0e39] = 0x8d8e, [0x0e3a] = 0x8d8f, [0x0e3b] = 0x8d8d,
  [0x0e3c] = 0x8d93, [0x0e3d] = 0x8d94, [0x0e3e] = 0x8d90, [0x0e3f] = 0x8d92,
  [0x0e40] = 0x8df0, [0x0e41] = 0x8de0, [0x0e42] = 0x8dec, [0x0e43] = 0x8df1,
  [0x0e44] = 0x8dee, [0x0e45] = 0x8dd0, [0x0e46] = 0x8de9, [0x0e47] = 0x8de3,
  [0x0e48] = 0x8de2, [0x0e49] = 0x8de7, [0x0e4a] = 0x8df2, [0x0e4b] = 0x8deb,
  [0x0e4c] = 0x8df4, [0x0e4d] = 0x8f06, [0x0e4e] = 0x8eff, [0x0e4f] = 0x8f01,
  [0x0e50] = 0x8f00, [0x0e51] = 0x8f05, [0x0e52] = 0x8f07, [0x0e53] = 0x8f08,
  [0x0e54] = 0x8f02, [0x0e55] = 0x8f0b, [0x0e56] = 0x9052, [0x0e57] = 0x903f,
  [0x0e58] = 0x9044, [0x0e59] = 0x9049, [0x0e5a] = 0x903d, [0x0e5b] = 0x9110,
  [0x0e5c] = 0x910d, [0x0e5d] = 0x910f, [0x0e5e] = 0x9111, [0x0e5f] = 0x9116,
  [0x0e60] = 0x9114, [0x0e61] = 0x910b, [0x0e62] = 0x910e, [0x0e63] = 0x916e,
  [0x0e64] = 0x916f, [0x0e65] = 0x9248, [0x0e66] = 0x9252, [0x0e67] = 0x9230,
  [0x0e68] = 0x923a, [0x0e69] = 0x9266, [0x0e6a] = 0x9233, [0x0e6b] = 0x9265,
  [0x0e6c] = 0x925e, [0x0e6d] = 0x9283, [0x0e6e] = 0x922e, [0x0e6f] = 0x924a,
  [0x0e70] = 0x9246, [0x0e71] = 0x926d, [0x0e72] = 0x926c, [0x0e73] = 0x924f,
  [0x0e74] = 0x9260, [0x0e75] = 0x9267, [0x0e76] = 0x926f, [0x0e77] = 0x9236,
  [0x0e78] = 0x9261, [0x0e79] = 0x9270, [0x0e7a] = 0x9231, [0x0e7b] = 0x9254,
  [0x0e7c] = 0x9263, [0x0e7d] = 0x9250, [0x0e7e] = 0x9272, [0x0e7f] = 0x924e,
  [0x0e80] = 0x9253, [0x0e81] = 0x924c, [0x0e82] = 0x9256, [0x0e83] = 0x9232,
  [0x0e84] = 0x959f, [0x0e85] = 0x959c, [0x0e86] = 0x959e, [0x0e87] = 0x959b,
  [0x0e88] = 0x9692, [0x0e89] = 0x9693, [0x0e8a] = 0x9691, [0x0e8b] = 0x9697,
  [0x0e8c] = 0x96ce, [0x0e8d] = 0x96fa, [0x0e8e] = 0x96fd, [0x0e8f] = 0x96f8,
  [0x0e90] = 0x96f5, [0x0e91] = 0x9773, [0x0e92] = 0x9777, [0x0e93] = 0x9778,
  [0x0e94] = 0x9772, [0x0e95] = 0x980f, [0x0e96] = 0x980d, [0x0e97] = 0x980e,
  [0x0e98] = 0x98ac, [0x0e99] = 0x98f6, [0x0e9a] = 0x98f9, [0x0e9b] = 0x99af,
  [0x0e9c] = 0x99b2, [0x0e9d] = 0x99b0, [0x0e9e] = 0x99b5, [0x0e9f] = 0x9aad,
  [0x0ea0] = 0x9aab, [0x0ea1] = 0x9b5b, [0x0ea2] = 0x9cea, [0x0ea3] = 0x9ced,
  [0x0ea4] = 0x9ce7, [0x0ea5] = 0x9e80, [0x0ea6] = 0x9efd, [0x0ea7] = 0x50e6,
  [0x0ea8] = 0x50d4, [0x0ea9] = 0x50d7, [0x0eaa] = 0x50e8, [0x0eab] = 0x50f3,
  [0x0eac] = 0x50db, [0x0ead] = 0x50ea, [0x0eae] = 0x50dd, [0x0eaf] = 0x50e4,
  [0x0eb0] = 0x50d3, [0x0eb1] = 0x50ec, [0x0eb2] = 0x50f0, [0x0eb3] = 0x50ef,
  [0x0eb4] = 0x50e3, [0x0eb5] = 0x50e0, [0x0eb6] = 0x51d8, [0x0eb7] = 0x5280,
  [0x0eb8] = 0x5281, [0x0eb9] = 0x52e9, [0x0eba] = 0x52eb, [0x0ebb] = 0x5330,
  [0x0ebc] = 0x53ac, [0x0ebd] = 0x5627, [0x0ebe] = 0x5615, [0x0ebf] = 0x560c,
  [0x0ec0] = 0x5612, [0x0ec1] = 0x55fc, [0x0ec2] = 0x560f, [0x0ec3] = 0x561c,
  [0x0ec4] = 0x5601, [0x0ec5] = 0x5613, [0x0ec6] = 0x5602, [0x0ec7] = 0x55fa,
  [0x0ec8] = 0x561d, [0x0ec9] = 0x5604, [0x0eca] = 0x55ff, [0x0ecb] = 0x55f9,
  [0x0ecc] = 0x5889, [0x0ecd] = 0x587c, [0x0ece] = 0x5890, [0x0ecf] = 0x5898,
  [0x0ed0] = 0x5886, [0x0ed1] = 0x5881, [0x0ed2] = 0x587f, [0x0ed3] = 0x5874,
  [0x0ed4] = 0x588b, [0x0ed5] = 0x587a, [0x0ed6] = 0x5887, [0x0ed7] = 0x5891,
  [0x0ed8] = 0x588e, [0x0ed9] = 0x5876, [0x0eda] = 0x5882, [0x0edb] = 0x5888,
  [0x0edc] = 0x587b, [0x0edd] = 0x5894, [0x0ede] = 0x588f, [0x0edf] = 0x58fe,
  [0x0ee0] = 0x596b, [0x0ee1] = 0x5adc, [0x0ee2] = 0x5aee, [0x0ee3] = 0x5ae5,
  [0x0ee4] = 0x5ad5, [0x0ee5] = 0x5aea, [0x0ee6] = 0x5ada, [0x0ee7] = 0x5aed,
  [0x0ee8] = 0x5aeb, [0x0ee9] = 0x5af3, [0x0eea] = 0x5ae2, [0x0eeb] = 0x5ae0,
  [0x0eec] = 0x5adb, [0x0eed] = 0x5aec, [0x0eee] = 0x5ade, [0x0eef] = 0x5add,
  [0x0ef0] = 0x5ad9, [0x0ef1] = 0x5ae8, [0x0ef2] = 0x5adf, [0x0ef3] = 0x5b77,
  [0x0ef4] = 0x5be0, [0x0ef5] = 0x5be3, [0x0ef6] = 0x5c63, [0x0ef7] = 0x5d82,
  [0x0ef8] = 0x5d80, [0x0ef9] = 0x5d7d, [0x0efa] = 0x5d86, [0x0efb] = 0x5d7a,
  [0x0efc] = 0x5d81, [0x0efd] = 0x5d77, [0x0efe] = 0x5d8a, [0x0eff] = 0x5d89,
  [0x0f00] = 0x5d88, [0x0f01] = 0x5d7e, [0x0f02] = 0x5d7c, [0x0f03] = 0x5d8d,
  [0x0f04] = 0x5d79, [0x0f05] = 0x5d7f, [0x0f06] = 0x5e58, [0x0f07] = 0x5e59,
  [0x0f08] = 0x5e53, [0x0f09] = 0x5ed8, [0x0f0a] = 0x5ed1, [0x0f0b] = 0x5ed7,
  [0x0f0c] = 0x5ece, [0x0f0d] = 0x5edc, [0x0f0e] = 0x5ed5, [0x0f0f] = 0x5ed9,
  [0x0f10] = 0x5ed2, [0x0f11] = 0x5ed4, [0x0f12] = 0x5f44, [0x0f13] = 0x5f43,
  [0x0f14] = 0x5f6f, [0x0f15] = 0x5fb6, [0x0f16] = 0x612c, [0x0f17] = 0x6128,
  [0x0f18] = 0x6141, [0x0f19] = 0x615e, [0x0f1a] = 0x6171, [0x0f1b] = 0x6173,
  [0x0f1c] = 0x6152, [0x0f1d] = 0x6153, [0x0f1e] = 0x6172, [0x0f1f] = 0x616c,
  [0x0f20] = 0x6180, [0x0f21] = 0x6174, [0x0f22] = 0x6154, [0x0f23] = 0x617a,
  [0x0f24] = 0x615b, [0x0f25] = 0x6165, [0x0f26] = 0x613b, [0x0f27] = 0x616a,
  [0x0f28] = 0x6161, [0x0f29] = 0x6156, [0x0f2a] = 0x6229, [0x0f2b] = 0x6227,
  [0x0f2c] = 0x622b, [0x0f2d] = 0x642b, [0x0f2e] = 0x644d, [0x0f2f] = 0x645b,
  [0x0f30] = 0x645d, [0x0f31] = 0x6474, [0x0f32] = 0x6476, [0x0f33] = 0x6472,
  [0x0f34] = 0x6473, [0x0f35] = 0x647d, [0x0f36] = 0x6475, [0x0f37] = 0x6466,
  [0x0f38] = 0x64a6, [0x0f39] = 0x644e, [0x0f3a] = 0x6482, [0x0f3b] = 0x645e,
  [0x0f3c] = 0x645c, [0x0f3d] = 0x644b, [0x0f3e] = 0x6453, [0x0f3f] = 0x6460,
  [0x0f40] = 0x6450, [0x0f41] = 0x647f, [0x0f42] = 0x643f, [0x0f43] = 0x646c,
  [0x0f44] = 0x646b, [0x0f45] = 0x6459, [0x0f46] = 0x6465, [0x0f47] = 0x6477,
  [0x0f48] = 0x6573, [0x0f49] = 0x65a0, [0x0f4a] = 0x66a1, [0x0f4b] = 0x66a0,
  [0x0f4c] = 0x669f, [0x0f4d] = 0x6705, [0x0f4e] = 0x6704, [0x0f4f] = 0x6722,
  [0x0f50] = 0x69b1, [0x0f51] = 0x69b6, [0x0f52] = 0x69c9, [0x0f53] = 0x69a0,
  [0x0f54] = 0x69ce, [0x0f55] = 0x6996, [0x0f56] = 0x69b0, [0x0f57] = 0x69ac,
  [0x0f58] = 0x69bc, [0x0f59] = 0x6991, [0x0f5a] = 0x6999, [0x0f5b] = 0x698e,
  [0x0f5c] = 0x69a7, [0x0f5d] = 0x698d, [0x0f5e] = 0x69a9, [0x0f5f] = 0x69be,
  [0x0f60] = 0x69af, [0x0f61] = 0x69bf, [0x0f62] = 0x69c4, [0x0f63] = 0x69bd,
  [0x0f64] = 0x69a4, [0x0f65] = 0x69d4, [0x0f66] = 0x69b9, [0x0f67] = 0x69ca,
  [0x0f68] = 0x699a, [0x0f69] = 0x69cf, [0x0f6a] = 0x69b3, [0x0f6b] = 0x6993,
  [0x0f6c] = 0x69aa, [0x0f6d] = 0x69a1, [0x0f6e] = 0x699e, [0x0f6f] = 0x69d9,
  [0x0f70] = 0x6997, [0x0f71] = 0x6990, [0x0f72] = 0x69c2, [0x0f73] = 0x69b5,
  [0x0f74] = 0x69a5, [0x0f75] = 0x69c6, [0x0f76] = 0x6b4a, [0x0f77] = 0x6b4d,
  [0x0f78] = 0x6b4b, [0x0f79] = 0x6b9e, [0x0f7a] = 0x6b9f, [0x0f7b] = 0x6ba0,
  [0x0f7c] = 0x6bc3, [0x0f7d] = 0x6bc4, [0x0f7e] = 0x6bfe, [0x0f7f] = 0x6ece,
  [0x0f80] = 0x6ef5, [0x0f81] = 0x6ef1, [0x0f82] = 0x6f03, [0x0f83] = 0x6f25,
  [0x0f84] = 0x6ef8, [0x0f85] = 0x6f37, [0x0f86] = 0x6efb, [0x0f87] = 0x6f2e,
  [0x0f88] = 0x6f09, [0x0f89] = 0x6f4e, [0x0f8a] = 0x6f19, [0x0f8b] = 0x6f1a,
  [0x0f8c] = 0x6f27, [0x0f8d] = 0x6f18, [0x0f8e] = 0x6f3b, [0x0f8f] = 0x6f12,
  [0x0f90] = 0x6eed, [0x0f91] = 0x6f0a, [0x0f92] = 0x6f36, [0x0f93] = 0x6f73,
  [0x0f94] = 0x6ef9, [0x0f95] = 0x6eee, [0x0f96] = 0x6f2d, [0x0f97] = 0x6f40,
  [0x0f98] = 0x6f30, [0x0f99] = 0x6f3c, [0x0f9a] = 0x6f35, [0x0f9b] = 0x6eeb,
  [0x0f9c] = 0x6f07, [0x0f9d] = 0x6f0e, [0x0f9e] = 0x6f43, [0x0f9f] = 0x6f05,
  [0x0fa0] = 0x6efd, [0x0fa1] = 0x6ef6, [0x0fa2] = 0x6f39, [0x0fa3] = 0x6f1c,
  [0x0fa4] = 0x6efc, [0x0fa5] = 0x6f3a, [0x0fa6] = 0x6f1f, [0x0fa7] = 0x6f0d,
  [0x0fa8] = 0x6f1e, [0x0fa9] = 0x6f08, [0x0faa] = 0x6f21, [0x0fab] = 0x7187,
  [0x0fac] = 0x7190, [0x0fad] = 0x7189, [0x0fae] = 0x7180, [0x0faf] = 0x7185,
  [0x0fb0] = 0x7182, [0x0fb1] = 0x718f, [0x0fb2] = 0x717b, [0x0fb3] = 0x7186,
  [0x0fb4] = 0x7181, [0x0fb5] = 0x7197, [0x0fb6] = 0x7244, [0x0fb7] = 0x7253,
  [0x0fb8] = 0x7297, [0x0fb9] = 0x7295, [0x0fba] = 0x7293, [0x0fbb] = 0x7343,
  [0x0fbc] = 0x734d, [0x0fbd] = 0x7351, [0x0fbe] = 0x734c, [0x0fbf] = 0x7462,
  [0x0fc0] = 0x7473, [0x0fc1] = 0x7471, [0x0fc2] = 0x7475, [0x0fc3] = 0x7472,
  [0x0fc4] = 0x7467, [0x0fc5] = 0x746e, [0x0fc6] = 0x7500, [0x0fc7] = 0x7502,
  [0x0fc8] = 0x7503, [0x0fc9] = 0x757d, [0x0fca] = 0x7590, [0x0fcb] = 0x7616,
  [0x0fcc] = 0x7608, [0x0fcd] = 0x760c, [0x0fce] = 0x7615, [0x0fcf] = 0x7611,
  [0x0fd0] = 0x760a, [0x0fd1] = 0x7614, [0x0fd2] = 0x76b8, [0x0fd3] = 0x7781,
  [0x0fd4] = 0x777c, [0x0fd5] = 0x7785, [0x0fd6] = 0x7782, [0x0fd7] = 0x776e,
  [0x0fd8] = 0x7780, [0x0fd9] = 0x776f, [0x0fda] = 0x777e, [0x0fdb] = 0x7783,
  [0x0fdc] = 0x78b2, [0x0fdd] = 0x78aa, [0x0fde] = 0x78b4, [0x0fdf] = 0x78ad,
  [0x0fe0] = 0x78a8, [0x0fe1] = 0x787e, [0x0fe2] = 0x78ab, [0x0fe3] = 0x789e,
  [0x0fe4] = 0x78a5, [0x0fe5] = 0x78a0, [0x0fe6] = 0x78ac, [0x0fe7] = 0x78a2,
  [0x0fe8] = 0x78a4, [0x0fe9] = 0x7998, [0x0fea] = 0x798a, [0x0feb] = 0x798b,
  [0x0fec] = 0x7996, [0x0fed] = 0x7995, [0x0fee] = 0x7994, [0x0fef] = 0x7993,
  [0x0ff0] = 0x7997, [0x0ff1] = 0x7988, [0x0ff2] = 0x7992, [0x0ff3] = 0x7990,
  [0x0ff4] = 0x7a2b, [0x0ff5] = 0x7a4a, [0x0ff6] = 0x7a30, [0x0ff7] = 0x7a2f,
  [0x0ff8] = 0x7a28, [0x0ff9] = 0x7a26, [0x0ffa] = 0x7aa8, [0x0ffb] = 0x7aab,
  [0x0ffc] = 0x7aac, [0x0ffd] = 0x7aee, [0x0ffe] = 0x7b88, [0x0fff] = 0x7b9c,
  [0x1000] = 0x7b8a, [0x1001] = 0x7b91, [0x1002] = 0x7b90, [0x1003] = 0x7b96,
  [0x1004] = 0x7b8d, [0x1005] = 0x7b8c, [0x1006] = 0x7b9b, [0x1007] = 0x7b8e,
  [0x1008] = 0x7b85, [0x1009] = 0x7b98, [0x100a] = 0x5284, [0x100b] = 0x7b99,
  [0x100c] = 0x7ba4, [0x100d] = 0x7b82, [0x100e] = 0x7cbb, [0x100f] = 0x7cbf,
  [0x1010] = 0x7cbc, [0x1011] = 0x7cba, [0x1012] = 0x7da7, [0x1013] = 0x7db7,
  [0x1014] = 0x7dc2, [0x1015] = 0x7da3, [0x1016] = 0x7daa, [0x1017] = 0x7dc1,
  [0x1018] = 0x7dc0, [0x1019] = 0x7dc5, [0x101a] = 0x7d9d, [0x101b] = 0x7dce,
  [0x101c] = 0x7dc4, [0x101d] = 0x7dc6, [0x101e] = 0x7dcb, [0x101f] = 0x7dcc,
  [0x1020] = 0x7daf, [0x1021] = 0x7db9, [0x1022] = 0x7d96, [0x1023] = 0x7dbc,
  [0x1024] = 0x7d9f, [0x1025] = 0x7da6, [0x1026] = 0x7dae, [0x1027] = 0x7da9,
  [0x1028] = 0x7da1, [0x1029] = 0x7dc9, [0x102a] = 0x7f73, [0x102b] = 0x7fe2,
  [0x102c] = 0x7fe3, [0x102d] = 0x7fe5, [0x102e] = 0x7fde, [0x102f] = 0x8024,
  [0x1030] = 0x805d, [0x1031] = 0x805c, [0x1032] = 0x8189, [0x1033] = 0x8186,
  [0x1034] = 0x8183, [0x1035] = 0x8187, [0x1036] = 0x818d, [0x1037] = 0x818c,
  [0x1038] = 0x818b, [0x1039] = 0x8215, [0x103a] = 0x8497, [0x103b] = 0x84a4,
  [0x103c] = 0x84a1, [0x103d] = 0x849f, [0x103e] = 0x84ba, [0x103f] = 0x84ce,
  [0x1040] = 0x84c2, [0x1041] = 0x84ac, [0x1042] = 0x84ae, [0x1043] = 0x84ab,
  [0x1044] = 0x84b9, [0x1045] = 0x84b4, [0x1046] = 0x84c1, [0x1047] = 0x84cd,
  [0x1048] = 0x84aa, [0x1049] = 0x849a, [0x104a] = 0x84b1, [0x104b] = 0x84d0,
  [0x104c] = 0x849d, [0x104d] = 0x84a7, [0x104e] = 0x84bb, [0x104f] = 0x84a2,
  [0x1050] = 0x8494, [0x1051] = 0x84c7, [0x1052] = 0x84cc, [0x1053] = 0x849b,
  [0x1054] = 0x84a9, [0x1055] = 0x84af, [0x1056] = 0x84a8, [0x1057] = 0x84d6,
  [0x1058] = 0x8498, [0x1059] = 0x84b6, [0x105a] = 0x84cf, [0x105b] = 0x84a0,
  [0x105c] = 0x84d7, [0x105d] = 0x84d4, [0x105e] = 0x84d2, [0x105f] = 0x84db,
  [0x1060] = 0x84b0, [0x1061] = 0x8491, [0x1062] = 0x8661, [0x1063] = 0x8733,
  [0x1064] = 0x8723, [0x1065] = 0x8728, [0x1066] = 0x876b, [0x1067] = 0x8740,
  [0x1068] = 0x872e, [0x1069] = 0x871e, [0x106a] = 0x8721, [0x106b] = 0x8719,
  [0x106c] = 0x871b, [0x106d] = 0x8743, [0x106e] = 0x872c, [0x106f] = 0x8741,
  [0x1070] = 0x873e, [0x1071] = 0x8746, [0x1072] = 0x8720, [0x1073] = 0x8732,
  [0x1074] = 0x872a, [0x1075] = 0x872d, [0x1076] = 0x873c, [0x1077] = 0x8712,
  [0x1078] = 0x873a, [0x1079] = 0x8731, [0x107a] = 0x8735, [0x107b] = 0x8742,
  [0x107c] = 0x8726, [0x107d] = 0x8727, [0x107e] = 0x8738, [0x107f] = 0x8724,
  [0x1080] = 0x871a, [0x1081] = 0x8730, [0x1082] = 0x8711, [0x1083] = 0x88f7,
  [0x1084] = 0x88e7, [0x1085] = 0x88f1, [0x1086] = 0x88f2, [0x1087] = 0x88fa,
  [0x1088] = 0x88fe, [0x1089] = 0x88ee, [0x108a] = 0x88fc, [0x108b] = 0x88f6,
  [0x108c] = 0x88fb, [0x108d] = 0x88f0, [0x108e] = 0x88ec, [0x108f] = 0x88eb,
  [0x1090] = 0x899d, [0x1091] = 0x89a1, [0x1092] = 0x899f, [0x1093] = 0x899e,
  [0x1094] = 0x89e9, [0x1095] = 0x89eb, [0x1096] = 0x89e8, [0x1097] = 0x8aab,
  [0x1098] = 0x8a99, [0x1099] = 0x8a8b, [0x109a] = 0x8a92, [0x109b] = 0x8a8f,
  [0x109c] = 0x8a96, [0x109d] = 0x8c3d, [0x109e] = 0x8c68, [0x109f] = 0x8c69,
  [0x10a0] = 0x8cd5, [0x10a1] = 0x8ccf, [0x10a2] = 0x8cd7, [0x10a3] = 0x8d96,
  [0x10a4] = 0x8e09, [0x10a5] = 0x8e02, [0x10a6] = 0x8dff, [0x10a7] = 0x8e0d,
  [0x10a8] = 0x8dfd, [0x10a9] = 0x8e0a, [0x10aa] = 0x8e03, [0x10ab] = 0x8e07,
  [0x10ac] = 0x8e06, [0x10ad] = 0x8e05, [0x10ae] = 0x8dfe, [0x10af] = 0x8e00,
  [0x10b0] = 0x8e04, [0x10b1] = 0x8f10, [0x10b2] = 0x8f11, [0x10b3] = 0x8f0e,
  [0x10b4] = 0x8f0d, [0x10b5] = 0x9123, [0x10b6] = 0x911c, [0x10b7] = 0x9120,
  [0x10b8] = 0x9122, [0x10b9] = 0x911f, [0x10ba] = 0x911d, [0x10bb] = 0x911a,
  [0x10bc] = 0x9124, [0x10bd] = 0x9121, [0x10be] = 0x911b, [0x10bf] = 0x917a,
  [0x10c0] = 0x9172, [0x10c1] = 0x9179, [0x10c2] = 0x9173, [0x10c3] = 0x92a5,
  [0x10c4] = 0x92a4, [0x10c5] = 0x9276, [0x10c6] = 0x929b, [0x10c7] = 0x927a,
  [0x10c8] = 0x92a0, [0x10c9] = 0x9294, [0x10ca] = 0x92aa, [0x10cb] = 0x928d,
  [0x10cc] = 0x92a6, [0x10cd] = 0x929a, [0x10ce] = 0x92ab, [0x10cf] = 0x9279,
  [0x10d0] = 0x9297, [0x10d1] = 0x927f, [0x10d2] = 0x92a3, [0x10d3] = 0x92ee,
  [0x10d4] = 0x928e, [0x10d5] = 0x9282, [0x10d6] = 0x9295, [0x10d7] = 0x92a2,
  [0x10d8] = 0x927d, [0x10d9] = 0x9288, [0x10da] = 0x92a1, [0x10db] = 0x928a,
  [0x10dc] = 0x9286, [0x10dd] = 0x928c, [0x10de] = 0x9299, [0x10df] = 0x92a7,
  [0x10e0] = 0x927e, [0x10e1] = 0x9287, [0x10e2] = 0x92a9, [0x10e3] = 0x929d,
  [0x10e4] = 0x928b, [0x10e5] = 0x922d, [0x10e6] = 0x969e, [0x10e7] = 0x96a1,
  [0x10e8] = 0x96ff, [0x10e9] = 0x9758, [0x10ea] = 0x977d, [0x10eb] = 0x977a,
  [0x10ec] = 0x977e, [0x10ed] = 0x9783, [0x10ee] = 0x9780, [0x10ef] = 0x9782,
  [0x10f0] = 0x977b, [0x10f1] = 0x9784, [0x10f2] = 0x9781, [0x10f3] = 0x977f,
  [0x10f4] = 0x97ce, [0x10f5] = 0x97cd, [0x10f6] = 0x9816, [0x10f7] = 0x98ad,
  [0x10f8] = 0x98ae, [0x10f9] = 0x9902, [0x10fa] = 0x9900, [0x10fb] = 0x9907,
  [0x10fc] = 0x999d, [0x10fd] = 0x999c, [0x10fe] = 0x99c3, [0x10ff] = 0x99b9,
  [0x1100] = 0x99bb, [0x1101] = 0x99ba, [0x1102] = 0x99c2, [0x1103] = 0x99bd,
  [0x1104] = 0x99c7, [0x1105] = 0x9ab1, [0x1106] = 0x9ae3, [0x1107] = 0x9ae7,
  [0x1108] = 0x9b3e, [0x1109] = 0x9b3f, [0x110a] = 0x9b60, [0x110b] = 0x9b61,
  [0x110c] = 0x9b5f, [0x110d] = 0x9cf1, [0x110e] = 0x9cf2, [0x110f] = 0x9cf5,
  [0x1110] = 0x9ea7, [0x1111] = 0x50ff, [0x1112] = 0x5103, [0x1113] = 0x5130,
  [0x1114] = 0x50f8, [0x1115] = 0x5106, [0x1116] = 0x5107, [0x1117] = 0x50f6,
  [0x1118] = 0x50fe, [0x1119] = 0x510b, [0x111a] = 0x510c, [0x111b] = 0x50fd,
  [0x111c] = 0x510a, [0x111d] = 0x528b, [0x111e] = 0x528c, [0x111f] = 0x52f1,
  [0x1120] = 0x52ef, [0x1121] = 0x5648, [0x1122] = 0x5642, [0x1123] = 0x564c,
  [0x1124] = 0x5635, [0x1125] = 0x5641, [0x1126] = 0x564a, [0x1127] = 0x5649,
  [0x1128] = 0x5646, [0x1129] = 0x5658, [0x112a] = 0x565a, [0x112b] = 0x5640,
  [0x112c] = 0x5633, [0x112d] = 0x563d, [0x112e] = 0x562c, [0x112f] = 0x563e,
  [0x1130] = 0x5638, [0x1131] = 0x562a, [0x1132] = 0x563a, [0x1133] = 0x571a,
  [0x1134] = 0x58ab, [0x1135] = 0x589d, [0x1136] = 0x58b1, [0x1137] = 0x58a0,
  [0x1138] = 0x58a3, [0x1139] = 0x58af, [0x113a] = 0x58ac, [0x113b] = 0x58a5,
  [0x113c] = 0x58a1, [0x113d] = 0x58ff, [0x113e] = 0x5aff, [0x113f] = 0x5af4,
  [0x1140] = 0x5afd, [0x1141] = 0x5af7, [0x1142] = 0x5af6, [0x1143] = 0x5b03,
  [0x1144] = 0x5af8, [0x1145] = 0x5b02, [0x1146] = 0x5af9, [0x1147] = 0x5b01,
  [0x1148] = 0x5b07, [0x1149] = 0x5b05, [0x114a] = 0x5b0f, [0x114b] = 0x5c67,
  [0x114c] = 0x5d99, [0x114d] = 0x5d97, [0x114e] = 0x5d9f, [0x114f] = 0x5d92,
  [0x1150] = 0x5da2, [0x1151] = 0x5d93, [0x1152] = 0x5d95, [0x1153] = 0x5da0,
  [0x1154] = 0x5d9c, [0x1155] = 0x5da1, [0x1156] = 0x5d9a, [0x1157] = 0x5d9e,
  [0x1158] = 0x5e69, [0x1159] = 0x5e5d, [0x115a] = 0x5e60, [0x115b] = 0x5e5c,
  [0x115c] = 0x7df3, [0x115d] = 0x5edb, [0x115e] = 0x5ede, [0x115f] = 0x5ee1,
  [0x1160] = 0x5f49, [0x1161] = 0x5fb2, [0x1162] = 0x618b, [0x1163] = 0x6183,
  [0x1164] = 0x6179, [0x1165] = 0x61b1, [0x1166] = 0x61b0, [0x1167] = 0x61a2,
  [0x1168] = 0x6189, [0x1169] = 0x619b, [0x116a] = 0x6193, [0x116b] = 0x61af,
  [0x116c] = 0x61ad, [0x116d] = 0x619f, [0x116e] = 0x6192, [0x116f] = 0x61aa,
  [0x1170] = 0x61a1, [0x1171] = 0x618d, [0x1172] = 0x6166, [0x1173] = 0x61b3,
  [0x1174] = 0x622d, [0x1175] = 0x646e, [0x1176] = 0x6470, [0x1177] = 0x6496,
  [0x1178] = 0x64a0, [0x1179] = 0x6485, [0x117a] = 0x6497, [0x117b] = 0x649c,
  [0x117c] = 0x648f, [0x117d] = 0x648b, [0x117e] = 0x648a, [0x117f] = 0x648c,
  [0x1180] = 0x64a3, [0x1181] = 0x649f, [0x1182] = 0x6468, [0x1183] = 0x64b1,
  [0x1184] = 0x6498, [0x1185] = 0x6576, [0x1186] = 0x657a, [0x1187] = 0x6579,
  [0x1188] = 0x657b, [0x1189] = 0x65b2, [0x118a] = 0x65b3, [0x118b] = 0x66b5,
  [0x118c] = 0x66b0, [0x118d] = 0x66a9, [0x118e] = 0x66b2, [0x118f] = 0x66b7,
  [0x1190] = 0x66aa, [0x1191] = 0x66af, [0x1192] = 0x6a00, [0x1193] = 0x6a06,
  [0x1194] = 0x6a17, [0x1195] = 0x69e5, [0x1196] = 0x69f8, [0x1197] = 0x6a15,
  [0x1198] = 0x69f1, [0x1199] = 0x69e4, [0x119a] = 0x6a20, [0x119b] = 0x69ff,
  [0x119c] = 0x69ec, [0x119d] = 0x69e2, [0x119e] = 0x6a1b, [0x119f] = 0x6a1d,
  [0x11a0] = 0x69fe, [0x11a1] = 0x6a27, [0x11a2] = 0x69f2, [0x11a3] = 0x69ee,
  [0x11a4] = 0x6a14, [0x11a5] = 0x69f7, [0x11a6] = 0x69e7, [0x11a7] = 0x6a40,
  [0x11a8] = 0x6a08, [0x11a9] = 0x69e6, [0x11aa] = 0x69fb, [0x11ab] = 0x6a0d,
  [0x11ac] = 0x69fc, [0x11ad] = 0x69eb, [0x11ae] = 0x6a09, [0x11af] = 0x6a04,
  [0x11b0] = 0x6a18, [0x11b1] = 0x6a25, [0x11b2] = 0x6a0f, [0x11b3] = 0x69f6,
  [0x11b4] = 0x6a26, [0x11b5] = 0x6a07, [0x11b6] = 0x69f4, [0x11b7] = 0x6a16,
  [0x11b8] = 0x6b51, [0x11b9] = 0x6ba5, [0x11ba] = 0x6ba3, [0x11bb] = 0x6ba2,
  [0x11bc] = 0x6ba6, [0x11bd] = 0x6c01, [0x11be] = 0x6c00, [0x11bf] = 0x6bff,
  [0x11c0] = 0x6c02, [0x11c1] = 0x6f41, [0x11c2] = 0x6f26, [0x11c3] = 0x6f7e,
  [0x11c4] = 0x6f87, [0x11c5] = 0x6fc6, [0x11c6] = 0x6f92, [0x11c7] = 0x6f8d,
  [0x11c8] = 0x6f89, [0x11c9] = 0x6f8c, [0x11ca] = 0x6f62, [0x11cb] = 0x6f4f,
  [0x11cc] = 0x6f85, [0x11cd] = 0x6f5a, [0x11ce] = 0x6f96, [0x11cf] = 0x6f76,
  [0x11d0] = 0x6f6c, [0x11d1] = 0x6f82, [0x11d2] = 0x6f55, [0x11d3] = 0x6f72,
  [0x11d4] = 0x6f52, [0x11d5] = 0x6f50, [0x11d6] = 0x6f57, [0x11d7] = 0x6f94,
  [0x11d8] = 0x6f93, [0x11d9] = 0x6f5d, [0x11da] = 0x6f00, [0x11db] = 0x6f61,
  [0x11dc] = 0x6f6b, [0x11dd] = 0x6f7d, [0x11de] = 0x6f67, [0x11df] = 0x6f90,
  [0x11e0] = 0x6f53, [0x11e1] = 0x6f8b, [0x11e2] = 0x6f69, [0x11e3] = 0x6f7f,
  [0x11e4] = 0x6f95, [0x11e5] = 0x6f63, [0x11e6] = 0x6f77, [0x11e7] = 0x6f6a,
  [0x11e8] = 0x6f7b, [0x11e9] = 0x71b2, [0x11ea] = 0x71af, [0x11eb] = 0x719b,
  [0x11ec] = 0x71b0, [0x11ed] = 0x71a0, [0x11ee] = 0x719a, [0x11ef] = 0x71a9,
  [0x11f0] = 0x71b5, [0x11f1] = 0x719d, [0x11f2] = 0x71a5, [0x11f3] = 0x719e,
  [0x11f4] = 0x71a4, [0x11f5] = 0x71a1, [0x11f6] = 0x71aa, [0x11f7] = 0x719c,
  [0x11f8] = 0x71a7, [0x11f9] = 0x71b3, [0x11fa] = 0x7298, [0x11fb] = 0x729a,
  [0x11fc] = 0x7358, [0x11fd] = 0x7352, [0x11fe] = 0x735e, [0x11ff] = 0x735f,
  [0x1200] = 0x7360, [0x1201] = 0x735d, [0x1202] = 0x735b, [0x1203] = 0x7361,
  [0x1204] = 0x735a, [0x1205] = 0x7359, [0x1206] = 0x7362, [0x1207] = 0x7487,
  [0x1208] = 0x7489, [0x1209] = 0x748a, [0x120a] = 0x7486, [0x120b] = 0x7481,
  [0x120c] = 0x747d, [0x120d] = 0x7485, [0x120e] = 0x7488, [0x120f] = 0x747c,
  [0x1210] = 0x7479, [0x1211] = 0x7508, [0x1212] = 0x7507, [0x1213] = 0x757e,
  [0x1214] = 0x7625, [0x1215] = 0x761e, [0x1216] = 0x7619, [0x1217] = 0x761d,
  [0x1218] = 0x761c, [0x1219] = 0x7623, [0x121a] = 0x761a, [0x121b] = 0x7628,
  [0x121c] = 0x761b, [0x121d] = 0x769c, [0x121e] = 0x769d, [0x121f] = 0x769e,
  [0x1220] = 0x769b, [0x1221] = 0x778d, [0x1222] = 0x778f, [0x1223] = 0x7789,
  [0x1224] = 0x7788, [0x1225] = 0x78cd, [0x1226] = 0x78bb, [0x1227] = 0x78cf,
  [0x1228] = 0x78cc, [0x1229] = 0x78d1, [0x122a] = 0x78ce, [0x122b] = 0x78d4,
  [0x122c] = 0x78c8, [0x122d] = 0x78c3, [0x122e] = 0x78c4, [0x122f] = 0x78c9,
  [0x1230] = 0x799a, [0x1231] = 0x79a1, [0x1232] = 0x79a0, [0x1233] = 0x799c,
  [0x1234] = 0x79a2, [0x1235] = 0x799b, [0x1236] = 0x6b76, [0x1237] = 0x7a39,
  [0x1238] = 0x7ab2, [0x1239] = 0x7ab4, [0x123a] = 0x7ab3, [0x123b] = 0x7bb7,
  [0x123c] = 0x7bcb, [0x123d] = 0x7bbe, [0x123e] = 0x7bac, [0x123f] = 0x7bce,
  [0x1240] = 0x7baf, [0x1241] = 0x7bb9, [0x1242] = 0x7bca, [0x1243] = 0x7bb5,
  [0x1244] = 0x7cc5, [0x1245] = 0x7cc8, [0x1246] = 0x7ccc, [0x1247] = 0x7ccb,
  [0x1248] = 0x7df7, [0x1249] = 0x7ddb, [0x124a] = 0x7dea, [0x124b] = 0x7de7,
  [0x124c] = 0x7dd7, [0x124d] = 0x7de1, [0x124e] = 0x7e03, [0x124f] = 0x7dfa,
  [0x1250] = 0x7de6, [0x1251] = 0x7df6, [0x1252] = 0x7df1, [0x1253] = 0x7df0,
  [0x1254] = 0x7dee, [0x1255] = 0x7ddf, [0x1256] = 0x7f76, [0x1257] = 0x7fac,
  [0x1258] = 0x7fb0, [0x1259] = 0x7fad, [0x125a] = 0x7fed, [0x125b] = 0x7feb,
  [0x125c] = 0x7fea, [0x125d] = 0x7fec, [0x125e] = 0x7fe6, [0x125f] = 0x7fe8,
  [0x1260] = 0x8064, [0x1261] = 0x8067, [0x1262] = 0x81a3, [0x1263] = 0x819f,
  [0x1264] = 0x819e, [0x1265] = 0x8195, [0x1266] = 0x81a2, [0x1267] = 0x8199,
  [0x1268] = 0x8197, [0x1269] = 0x8216, [0x126a] = 0x824f, [0x126b] = 0x8253,
  [0x126c] = 0x8252, [0x126d] = 0x8250, [0x126e] = 0x824e, [0x126f] = 0x8251,
  [0x1270] = 0x8524, [0x1271] = 0x853b, [0x1272] = 0x850f, [0x1273] = 0x8500,
  [0x1274] = 0x8529, [0x1275] = 0x850e, [0x1276] = 0x8509, [0x1277] = 0x850d,
  [0x1278] = 0x851f, [0x1279] = 0x850a, [0x127a] = 0x8527, [0x127b] = 0x851c,
  [0x127c] = 0x84fb, [0x127d] = 0x852b, [0x127e] = 0x84fa, [0x127f] = 0x8508,
  [0x1280] = 0x850c, [0x1281] = 0x84f4, [0x1282] = 0x852a, [0x1283] = 0x84f2,
  [0x1284] = 0x8515, [0x1285] = 0x84f7, [0x1286] = 0x84eb, [0x1287] = 0x84f3,
  [0x1288] = 0x84fc, [0x1289] = 0x8512, [0x128a] = 0x84ea, [0x128b] = 0x84e9,
  [0x128c] = 0x8516, [0x128d] = 0x84fe, [0x128e] = 0x8528, [0x128f] = 0x851d,
  [0x1290] = 0x852e, [0x1291] = 0x8502, [0x1292] = 0x84fd, [0x1293] = 0x851e,
  [0x1294] = 0x84f6, [0x1295] = 0x8531, [0x1296] = 0x8526, [0x1297] = 0x84e7,
  [0x1298] = 0x84e8, [0x1299] = 0x84f0, [0x129a] = 0x84ef, [0x129b] = 0x84f9,
  [0x129c] = 0x8518, [0x129d] = 0x8520, [0x129e] = 0x8530, [0x129f] = 0x850b,
  [0x12a0] = 0x8519, [0x12a1] = 0x852f, [0x12a2] = 0x8662, [0x12a3] = 0x8756,
  [0x12a4] = 0x8763, [0x12a5] = 0x8764, [0x12a6] = 0x8777, [0x12a7] = 0x87e1,
  [0x12a8] = 0x8773, [0x12a9] = 0x8758, [0x12aa] = 0x8754, [0x12ab] = 0x875b,
  [0x12ac] = 0x8752, [0x12ad] = 0x8761, [0x12ae] = 0x875a, [0x12af] = 0x8751,
  [0x12b0] = 0x875e, [0x12b1] = 0x876d, [0x12b2] = 0x876a, [0x12b3] = 0x8750,
  [0x12b4] = 0x874e, [0x12b5] = 0x875f, [0x12b6] = 0x875d, [0x12b7] = 0x876f,
  [0x12b8] = 0x876c, [0x12b9] = 0x877a, [0x12ba] = 0x876e, [0x12bb] = 0x875c,
  [0x12bc] = 0x8765, [0x12bd] = 0x874f, [0x12be] = 0x877b, [0x12bf] = 0x8775,
  [0x12c0] = 0x8762, [0x12c1] = 0x8767, [0x12c2] = 0x8769, [0x12c3] = 0x885a,
  [0x12c4] = 0x8905, [0x12c5] = 0x890c, [0x12c6] = 0x8914, [0x12c7] = 0x890b,
  [0x12c8] = 0x8917, [0x12c9] = 0x8918, [0x12ca] = 0x8919, [0x12cb] = 0x8906,
  [0x12cc] = 0x8916, [0x12cd] = 0x8911, [0x12ce] = 0x890e, [0x12cf] = 0x8909,
  [0x12d0] = 0x89a2, [0x12d1] = 0x89a4, [0x12d2] = 0x89a3, [0x12d3] = 0x89ed,
  [0x12d4] = 0x89f0, [0x12d5] = 0x89ec, [0x12d6] = 0x8acf, [0x12d7] = 0x8ac6,
  [0x12d8] = 0x8ab8, [0x12d9] = 0x8ad3, [0x12da] = 0x8ad1, [0x12db] = 0x8ad4,
  [0x12dc] = 0x8ad5, [0x12dd] = 0x8abb, [0x12de] = 0x8ad7, [0x12df] = 0x8abe,
  [0x12e0] = 0x8ac0, [0x12e1] = 0x8ac5, [0x12e2] = 0x8ad8, [0x12e3] = 0x8ac3,
  [0x12e4] = 0x8aba, [0x12e5] = 0x8abd, [0x12e6] = 0x8ad9, [0x12e7] = 0x8c3e,
  [0x12e8] = 0x8c4d, [0x12e9] = 0x8c8f, [0x12ea] = 0x8ce5, [0x12eb] = 0x8cdf,
  [0x12ec] = 0x8cd9, [0x12ed] = 0x8ce8, [0x12ee] = 0x8cda, [0x12ef] = 0x8cdd,
  [0x12f0] = 0x8ce7, [0x12f1] = 0x8da0, [0x12f2] = 0x8d9c, [0x12f3] = 0x8da1,
  [0x12f4] = 0x8d9b, [0x12f5] = 0x8e20, [0x12f6] = 0x8e23, [0x12f7] = 0x8e25,
  [0x12f8] = 0x8e24, [0x12f9] = 0x8e2e, [0x12fa] = 0x8e15, [0x12fb] = 0x8e1b,
  [0x12fc] = 0x8e16, [0x12fd] = 0x8e11, [0x12fe] = 0x8e19, [0x12ff] = 0x8e26,
  [0x1300] = 0x8e27, [0x1301] = 0x8e14, [0x1302] = 0x8e12, [0x1303] = 0x8e18,
  [0x1304] = 0x8e13, [0x1305] = 0x8e1c, [0x1306] = 0x8e17, [0x1307] = 0x8e1a,
  [0x1308] = 0x8f2c, [0x1309] = 0x8f24, [0x130a] = 0x8f18, [0x130b] = 0x8f1a,
  [0x130c] = 0x8f20, [0x130d] = 0x8f23, [0x130e] = 0x8f16, [0x130f] = 0x8f17,
  [0x1310] = 0x9073, [0x1311] = 0x9070, [0x1312] = 0x906f, [0x1313] = 0x9067,
  [0x1314] = 0x906b, [0x1315] = 0x912f, [0x1316] = 0x912b, [0x1317] = 0x9129,
  [0x1318] = 0x912a, [0x1319] = 0x9132, [0x131a] = 0x9126, [0x131b] = 0x912e,
  [0x131c] = 0x9185, [0x131d] = 0x9186, [0x131e] = 0x918a, [0x131f] = 0x9181,
  [0x1320] = 0x9182, [0x1321] = 0x9184, [0x1322] = 0x9180, [0x1323] = 0x92d0,
  [0x1324] = 0x92c3, [0x1325] = 0x92c4, [0x1326] = 0x92c0, [0x1327] = 0x92d9,
  [0x1328] = 0x92b6, [0x1329] = 0x92cf, [0x132a] = 0x92f1, [0x132b] = 0x92df,
  [0x132c] = 0x92d8, [0x132d] = 0x92e9, [0x132e] = 0x92d7, [0x132f] = 0x92dd,
  [0x1330] = 0x92cc, [0x1331] = 0x92ef, [0x1332] = 0x92c2, [0x1333] = 0x92e8,
  [0x1334] = 0x92ca, [0x1335] = 0x92c8, [0x1336] = 0x92ce, [0x1337] = 0x92e6,
  [0x1338] = 0x92cd, [0x1339] = 0x92d5, [0x133a] = 0x92c9, [0x133b] = 0x92e0,
  [0x133c] = 0x92de, [0x133d] = 0x92e7, [0x133e] = 0x92d1, [0x133f] = 0x92d3,
  [0x1340] = 0x92b5, [0x1341] = 0x92e1, [0x1342] = 0x9325, [0x1343] = 0x92c6,
  [0x1344] = 0x92b4, [0x1345] = 0x957c, [0x1346] = 0x95ac, [0x1347] = 0x95ab,
  [0x1348] = 0x95ae, [0x1349] = 0x95b0, [0x134a] = 0x96a4, [0x134b] = 0x96a2,
  [0x134c] = 0x96d3, [0x134d] = 0x9705, [0x134e] = 0x9708, [0x134f] = 0x9702,
  [0x1350] = 0x975a, [0x1351] = 0x978a, [0x1352] = 0x978e, [0x1353] = 0x9788,
  [0x1354] = 0x97d0, [0x1355] = 0x97cf, [0x1356] = 0x981e, [0x1357] = 0x981d,
  [0x1358] = 0x9826, [0x1359] = 0x9829, [0x135a] = 0x9828, [0x135b] = 0x9820,
  [0x135c] = 0x981b, [0x135d] = 0x9827, [0x135e] = 0x98b2, [0x135f] = 0x9908,
  [0x1360] = 0x98fa, [0x1361] = 0x9911, [0x1362] = 0x9914, [0x1363] = 0x9916,
  [0x1364] = 0x9917, [0x1365] = 0x9915, [0x1366] = 0x99dc, [0x1367] = 0x99cd,
  [0x1368] = 0x99cf, [0x1369] = 0x99d3, [0x136a] = 0x99d4, [0x136b] = 0x99ce,
  [0x136c] = 0x99c9, [0x136d] = 0x99d6, [0x136e] = 0x99d8, [0x136f] = 0x99cb,
  [0x1370] = 0x99d7, [0x1371] = 0x99cc, [0x1372] = 0x9ab3, [0x1373] = 0x9aec,
  [0x1374] = 0x9aeb, [0x1375] = 0x9af3, [0x1376] = 0x9af2, [0x1377] = 0x9af1,
  [0x1378] = 0x9b46, [0x1379] = 0x9b43, [0x137a] = 0x9b67, [0x137b] = 0x9b74,
  [0x137c] = 0x9b71, [0x137d] = 0x9b66, [0x137e] = 0x9b76, [0x137f] = 0x9b75,
  [0x1380] = 0x9b70, [0x1381] = 0x9b68, [0x1382] = 0x9b64, [0x1383] = 0x9b6c,
  [0x1384] = 0x9cfc, [0x1385] = 0x9cfa, [0x1386] = 0x9cfd, [0x1387] = 0x9cff,
  [0x1388] = 0x9cf7, [0x1389] = 0x9d07, [0x138a] = 0x9d00, [0x138b] = 0x9cf9,
  [0x138c] = 0x9cfb, [0x138d] = 0x9d08, [0x138e] = 0x9d05, [0x138f] = 0x9d04,
  [0x1390] = 0x9e83, [0x1391] = 0x9ed3, [0x1392] = 0x9f0f, [0x1393] = 0x9f10,
  [0x1394] = 0x511c, [0x1395] = 0x5113, [0x1396] = 0x5117, [0x1397] = 0x511a,
  [0x1398] = 0x5111, [0x1399] = 0x51de, [0x139a] = 0x5334, [0x139b] = 0x53e1,
  [0x139c] = 0x5670, [0x139d] = 0x5660, [0x139e] = 0x566e, [0x139f] = 0x5673,
  [0x13a0] = 0x5666, [0x13a1] = 0x5663, [0x13a2] = 0x566d, [0x13a3] = 0x5672,
  [0x13a4] = 0x565e, [0x13a5] = 0x5677, [0x13a6] = 0x571c, [0x13a7] = 0x571b,
  [0x13a8] = 0x58c8, [0x13a9] = 0x58bd, [0x13aa] = 0x58c9, [0x13ab] = 0x58bf,
  [0x13ac] = 0x58ba, [0x13ad] = 0x58c2, [0x13ae] = 0x58bc, [0x13af] = 0x58c6,
  [0x13b0] = 0x5b17, [0x13b1] = 0x5b19, [0x13b2] = 0x5b1b, [0x13b3] = 0x5b21,
  [0x13b4] = 0x5b14, [0x13b5] = 0x5b13, [0x13b6] = 0x5b10, [0x13b7] = 0x5b16,
  [0x13b8] = 0x5b28, [0x13b9] = 0x5b1a, [0x13ba] = 0x5b20, [0x13bb] = 0x5b1e,
  [0x13bc] = 0x5bef, [0x13bd] = 0x5dac, [0x13be] = 0x5db1, [0x13bf] = 0x5da9,
  [0x13c0] = 0x5da7, [0x13c1] = 0x5db5, [0x13c2] = 0x5db0, [0x13c3] = 0x5dae,
  [0x13c4] = 0x5daa, [0x13c5] = 0x5da8, [0x13c6] = 0x5db2, [0x13c7] = 0x5dad,
  [0x13c8] = 0x5daf, [0x13c9] = 0x5db4, [0x13ca] = 0x5e67, [0x13cb] = 0x5e68,
  [0x13cc] = 0x5e66, [0x13cd] = 0x5e6f, [0x13ce] = 0x5ee9, [0x13cf] = 0x5ee7,
  [0x13d0] = 0x5ee6, [0x13d1] = 0x5ee8, [0x13d2] = 0x5ee5, [0x13d3] = 0x5f4b,
  [0x13d4] = 0x5fbc, [0x13d5] = 0x5fbb, [0x13d6] = 0x619d, [0x13d7] = 0x61a8,
  [0x13d8] = 0x6196, [0x13d9] = 0x61c5, [0x13da] = 0x61b4, [0x13db] = 0x61c6,
  [0x13dc] = 0x61c1, [0x13dd] = 0x61cc, [0x13de] = 0x61ba, [0x13df] = 0x61bf,
  [0x13e0] = 0x61b8, [0x13e1] = 0x618c, [0x13e2] = 0x64d7, [0x13e3] = 0x64d6,
  [0x13e4] = 0x64d0, [0x13e5] = 0x64cf, [0x13e6] = 0x64c9, [0x13e7] = 0x64bd,
  [0x13e8] = 0x6489, [0x13e9] = 0x64c3, [0x13ea] = 0x64db, [0x13eb] = 0x64f3,
  [0x13ec] = 0x64d9, [0x13ed] = 0x6533, [0x13ee] = 0x657f, [0x13ef] = 0x657c,
  [0x13f0] = 0x65a2, [0x13f1] = 0x66c8, [0x13f2] = 0x66be, [0x13f3] = 0x66c0,
  [0x13f4] = 0x66ca, [0x13f5] = 0x66cb, [0x13f6] = 0x66cf, [0x13f7] = 0x66bd,
  [0x13f8] = 0x66bb, [0x13f9] = 0x66ba, [0x13fa] = 0x66cc, [0x13fb] = 0x6723,
  [0x13fc] = 0x6a34, [0x13fd] = 0x6a66, [0x13fe] = 0x6a49, [0x13ff] = 0x6a67,
  [0x1400] = 0x6a32, [0x1401] = 0x6a68, [0x1402] = 0x6a3e, [0x1403] = 0x6a5d,
  [0x1404] = 0x6a6d, [0x1405] = 0x6a76, [0x1406] = 0x6a5b, [0x1407] = 0x6a51,
  [0x1408] = 0x6a28, [0x1409] = 0x6a5a, [0x140a] = 0x6a3b, [0x140b] = 0x6a3f,
  [0x140c] = 0x6a41, [0x140d] = 0x6a6a, [0x140e] = 0x6a64, [0x140f] = 0x6a50,
  [0x1410] = 0x6a4f, [0x1411] = 0x6a54, [0x1412] = 0x6a6f, [0x1413] = 0x6a69,
  [0x1414] = 0x6a60, [0x1415] = 0x6a3c, [0x1416] = 0x6a5e, [0x1417] = 0x6a56,
  [0x1418] = 0x6a55, [0x1419] = 0x6a4d, [0x141a] = 0x6a4e, [0x141b] = 0x6a46,
  [0x141c] = 0x6b55, [0x141d] = 0x6b54, [0x141e] = 0x6b56, [0x141f] = 0x6ba7,
  [0x1420] = 0x6baa, [0x1421] = 0x6bab, [0x1422] = 0x6bc8, [0x1423] = 0x6bc7,
  [0x1424] = 0x6c04, [0x1425] = 0x6c03, [0x1426] = 0x6c06, [0x1427] = 0x6fad,
  [0x1428] = 0x6fcb, [0x1429] = 0x6fa3, [0x142a] = 0x6fc7, [0x142b] = 0x6fbc,
  [0x142c] = 0x6fce, [0x142d] = 0x6fc8, [0x142e] = 0x6f5e, [0x142f] = 0x6fc4,
  [0x1430] = 0x6fbd, [0x1431] = 0x6f9e, [0x1432] = 0x6fca, [0x1433] = 0x6fa8,
  [0x1434] = 0x7004, [0x1435] = 0x6fa5, [0x1436] = 0x6fae, [0x1437] = 0x6fba,
  [0x1438] = 0x6fac, [0x1439] = 0x6faa, [0x143a] = 0x6fcf, [0x143b] = 0x6fbf,
  [0x143c] = 0x6fb8, [0x143d] = 0x6fa2, [0x143e] = 0x6fc9, [0x143f] = 0x6fab,
  [0x1440] = 0x6fcd, [0x1441] = 0x6faf, [0x1442] = 0x6fb2, [0x1443] = 0x6fb0,
  [0x1444] = 0x71c5, [0x1445] = 0x71c2, [0x1446] = 0x71bf, [0x1447] = 0x71b8,
  [0x1448] = 0x71d6, [0x1449] = 0x71c0, [0x144a] = 0x71c1, [0x144b] = 0x71cb,
  [0x144c] = 0x71d4, [0x144d] = 0x71ca, [0x144e] = 0x71c7, [0x144f] = 0x71cf,
  [0x1450] = 0x71bd, [0x1451] = 0x71d8, [0x1452] = 0x71bc, [0x1453] = 0x71c6,
  [0x1454] = 0x71da, [0x1455] = 0x71db, [0x1456] = 0x729d, [0x1457] = 0x729e,
  [0x1458] = 0x7369, [0x1459] = 0x7366, [0x145a] = 0x7367, [0x145b] = 0x736c,
  [0x145c] = 0x7365, [0x145d] = 0x736b, [0x145e] = 0x736a, [0x145f] = 0x747f,
  [0x1460] = 0x749a, [0x1461] = 0x74a0, [0x1462] = 0x7494, [0x1463] = 0x7492,
  [0x1464] = 0x7495, [0x1465] = 0x74a1, [0x1466] = 0x750b, [0x1467] = 0x7580,
  [0x1468] = 0x762f, [0x1469] = 0x762d, [0x146a] = 0x7631, [0x146b] = 0x763d,
  [0x146c] = 0x7633, [0x146d] = 0x763c, [0x146e] = 0x7635, [0x146f] = 0x7632,
  [0x1470] = 0x7630, [0x1471] = 0x76bb, [0x1472] = 0x76e6, [0x1473] = 0x779a,
  [0x1474] = 0x779d, [0x1475] = 0x77a1, [0x1476] = 0x779c, [0x1477] = 0x779b,
  [0x1478] = 0x77a2, [0x1479] = 0x77a3, [0x147a] = 0x7795, [0x147b] = 0x7799,
  [0x147c] = 0x7797, [0x147d] = 0x78dd, [0x147e] = 0x78e9, [0x147f] = 0x78e5,
  [0x1480] = 0x78ea, [0x1481] = 0x78de, [0x1482] = 0x78e3, [0x1483] = 0x78db,
  [0x1484] = 0x78e1, [0x1485] = 0x78e2, [0x1486] = 0x78ed, [0x1487] = 0x78df,
  [0x1488] = 0x78e0, [0x1489] = 0x79a4, [0x148a] = 0x7a44, [0x148b] = 0x7a48,
  [0x148c] = 0x7a47, [0x148d] = 0x7ab6, [0x148e] = 0x7ab8, [0x148f] = 0x7ab5,
  [0x1490] = 0x7ab1, [0x1491] = 0x7ab7, [0x1492] = 0x7bde, [0x1493] = 0x7be3,
  [0x1494] = 0x7be7, [0x1495] = 0x7bdd, [0x1496] = 0x7bd5, [0x1497] = 0x7be5,
  [0x1498] = 0x7bda, [0x1499] = 0x7be8, [0x149a] = 0x7bf9, [0x149b] = 0x7bd4,
  [0x149c] = 0x7bea, [0x149d] = 0x7be2, [0x149e] = 0x7bdc, [0x149f] = 0x7beb,
  [0x14a0] = 0x7bd8, [0x14a1] = 0x7bdf, [0x14a2] = 0x7cd2, [0x14a3] = 0x7cd4,
  [0x14a4] = 0x7cd7, [0x14a5] = 0x7cd0, [0x14a6] = 0x7cd1, [0x14a7] = 0x7e12,
  [0x14a8] = 0x7e21, [0x14a9] = 0x7e17, [0x14aa] = 0x7e0c, [0x14ab] = 0x7e1f,
  [0x14ac] = 0x7e20, [0x14ad] = 0x7e13, [0x14ae] = 0x7e0e, [0x14af] = 0x7e1c,
  [0x14b0] = 0x7e15, [0x14b1] = 0x7e1a, [0x14b2] = 0x7e22, [0x14b3] = 0x7e0b,
  [0x14b4] = 0x7e0f, [0x14b5] = 0x7e16, [0x14b6] = 0x7e0d, [0x14b7] = 0x7e14,
  [0x14b8] = 0x7e25, [0x14b9] = 0x7e24, [0x14ba] = 0x7f43, [0x14bb] = 0x7f7b,
  [0x14bc] = 0x7f7c, [0x14bd] = 0x7f7a, [0x14be] = 0x7fb1, [0x14bf] = 0x7fef,
  [0x14c0] = 0x802a, [0x14c1] = 0x8029, [0x14c2] = 0x806c, [0x14c3] = 0x81b1,
  [0x14c4] = 0x81a6, [0x14c5] = 0x81ae, [0x14c6] = 0x81b9, [0x14c7] = 0x81b5,
  [0x14c8] = 0x81ab, [0x14c9] = 0x81b0, [0x14ca] = 0x81ac, [0x14cb] = 0x81b4,
  [0x14cc] = 0x81b2, [0x14cd] = 0x81b7, [0x14ce] = 0x81a7, [0x14cf] = 0x81f2,
  [0x14d0] = 0x8255, [0x14d1] = 0x8256, [0x14d2] = 0x8257, [0x14d3] = 0x8556,
  [0x14d4] = 0x8545, [0x14d5] = 0x856b, [0x14d6] = 0x854d, [0x14d7] = 0x8553,
  [0x14d8] = 0x8561, [0x14d9] = 0x8558, [0x14da] = 0x8540, [0x14db] = 0x8546,
  [0x14dc] = 0x8564, [0x14dd] = 0x8541, [0x14de] = 0x8562, [0x14df] = 0x8544,
  [0x14e0] = 0x8551, [0x14e1] = 0x8547, [0x14e2] = 0x8563, [0x14e3] = 0x853e,
  [0x14e4] = 0x855b, [0x14e5] = 0x8571, [0x14e6] = 0x854e, [0x14e7] = 0x856e,
  [0x14e8] = 0x8575, [0x14e9] = 0x8555, [0x14ea] = 0x8567, [0x14eb] = 0x8560,
  [0x14ec] = 0x858c, [0x14ed] = 0x8566, [0x14ee] = 0x855d, [0x14ef] = 0x8554,
  [0x14f0] = 0x8565, [0x14f1] = 0x856c, [0x14f2] = 0x8663, [0x14f3] = 0x8665,
  [0x14f4] = 0x8664, [0x14f5] = 0x87a4, [0x14f6] = 0x879b, [0x14f7] = 0x878f,
  [0x14f8] = 0x8797, [0x14f9] = 0x8793, [0x14fa] = 0x8792, [0x14fb] = 0x8788,
  [0x14fc] = 0x8781, [0x14fd] = 0x8796, [0x14fe] = 0x8798, [0x14ff] = 0x8779,
  [0x1500] = 0x8787, [0x1501] = 0x87a3, [0x1502] = 0x8785, [0x1503] = 0x8790,
  [0x1504] = 0x8791, [0x1505] = 0x879d, [0x1506] = 0x8784, [0x1507] = 0x8794,
  [0x1508] = 0x879c, [0x1509] = 0x879a, [0x150a] = 0x8789, [0x150b] = 0x891e,
  [0x150c] = 0x8926, [0x150d] = 0x8930, [0x150e] = 0x892d, [0x150f] = 0x892e,
  [0x1510] = 0x8927, [0x1511] = 0x8931, [0x1512] = 0x8922, [0x1513] = 0x8929,
  [0x1514] = 0x8923, [0x1515] = 0x892f, [0x1516] = 0x892c, [0x1517] = 0x891f,
  [0x1518] = 0x89f1, [0x1519] = 0x8ae0, [0x151a] = 0x8ae2, [0x151b] = 0x8af2,
  [0x151c] = 0x8af4, [0x151d] = 0x8af5, [0x151e] = 0x8add, [0x151f] = 0x8b14,
  [0x1520] = 0x8ae4, [0x1521] = 0x8adf, [0x1522] = 0x8af0, [0x1523] = 0x8ac8,
  [0x1524] = 0x8ade, [0x1525] = 0x8ae1, [0x1526] = 0x8ae8, [0x1527] = 0x8aff,
  [0x1528] = 0x8aef, [0x1529] = 0x8afb, [0x152a] = 0x8c91, [0x152b] = 0x8c92,
  [0x152c] = 0x8c90, [0x152d] = 0x8cf5, [0x152e] = 0x8cee, [0x152f] = 0x8cf1,
  [0x1530] = 0x8cf0, [0x1531] = 0x8cf3, [0x1532] = 0x8d6c, [0x1533] = 0x8d6e,
  [0x1534] = 0x8da5, [0x1535] = 0x8da7, [0x1536] = 0x8e33, [0x1537] = 0x8e3e,
  [0x1538] = 0x8e38, [0x1539] = 0x8e40, [0x153a] = 0x8e45, [0x153b] = 0x8e36,
  [0x153c] = 0x8e3c, [0x153d] = 0x8e3d, [0x153e] = 0x8e41, [0x153f] = 0x8e30,
  [0x1540] = 0x8e3f, [0x1541] = 0x8ebd, [0x1542] = 0x8f36, [0x1543] = 0x8f2e,
  [0x1544] = 0x8f35, [0x1545] = 0x8f32, [0x1546] = 0x8f39, [0x1547] = 0x8f37,
  [0x1548] = 0x8f34, [0x1549] = 0x9076, [0x154a] = 0x9079, [0x154b] = 0x907b,
  [0x154c] = 0x9086, [0x154d] = 0x90fa, [0x154e] = 0x9133, [0x154f] = 0x9135,
  [0x1550] = 0x9136, [0x1551] = 0x9193, [0x1552] = 0x9190, [0x1553] = 0x9191,
  [0x1554] = 0x918d, [0x1555] = 0x918f, [0x1556] = 0x9327, [0x1557] = 0x931e,
  [0x1558] = 0x9308, [0x1559] = 0x931f, [0x155a] = 0x9306, [0x155b] = 0x930f,
  [0x155c] = 0x937a, [0x155d] = 0x9338, [0x155e] = 0x933c, [0x155f] = 0x931b,
  [0x1560] = 0x9323, [0x1561] = 0x9312, [0x1562] = 0x9301, [0x1563] = 0x9346,
  [0x1564] = 0x932d, [0x1565] = 0x930e, [0x1566] = 0x930d, [0x1567] = 0x92cb,
  [0x1568] = 0x931d, [0x1569] = 0x92fa, [0x156a] = 0x9313, [0x156b] = 0x92f9,
  [0x156c] = 0x92f7, [0x156d] = 0x9334, [0x156e] = 0x9302, [0x156f] = 0x9324,
  [0x1570] = 0x92ff, [0x1571] = 0x9329, [0x1572] = 0x9339, [0x1573] = 0x9335,
  [0x1574] = 0x932a, [0x1575] = 0x9314, [0x1576] = 0x930c, [0x1577] = 0x930b,
  [0x1578] = 0x92fe, [0x1579] = 0x9309, [0x157a] = 0x9300, [0x157b] = 0x92fb,
  [0x157c] = 0x9316, [0x157d] = 0x95bc, [0x157e] = 0x95cd, [0x157f] = 0x95be,
  [0x1580] = 0x95b9, [0x1581] = 0x95ba, [0x1582] = 0x95b6, [0x1583] = 0x95bf,
  [0x1584] = 0x95b5, [0x1585] = 0x95bd, [0x1586] = 0x96a9, [0x1587] = 0x96d4,
  [0x1588] = 0x970b, [0x1589] = 0x9712, [0x158a] = 0x9710, [0x158b] = 0x9799,
  [0x158c] = 0x9797, [0x158d] = 0x9794, [0x158e] = 0x97f0, [0x158f] = 0x97f8,
  [0x1590] = 0x9835, [0x1591] = 0x982f, [0x1592] = 0x9832, [0x1593] = 0x9924,
  [0x1594] = 0x991f, [0x1595] = 0x9927, [0x1596] = 0x9929, [0x1597] = 0x999e,
  [0x1598] = 0x99ee, [0x1599] = 0x99ec, [0x159a] = 0x99e5, [0x159b] = 0x99e4,
  [0x159c] = 0x99f0, [0x159d] = 0x99e3, [0x159e] = 0x99ea, [0x159f] = 0x99e9,
  [0x15a0] = 0x99e7, [0x15a1] = 0x9ab9, [0x15a2] = 0x9abf, [0x15a3] = 0x9ab4,
  [0x15a4] = 0x9abb, [0x15a5] = 0x9af6, [0x15a6] = 0x9afa, [0x15a7] = 0x9af9,
  [0x15a8] = 0x9af7, [0x15a9] = 0x9b33, [0x15aa] = 0x9b80, [0x15ab] = 0x9b85,
  [0x15ac] = 0x9b87, [0x15ad] = 0x9b7c, [0x15ae] = 0x9b7e, [0x15af] = 0x9b7b,
  [0x15b0] = 0x9b82, [0x15b1] = 0x9b93, [0x15b2] = 0x9b92, [0x15b3] = 0x9b90,
  [0x15b4] = 0x9b7a, [0x15b5] = 0x9b95, [0x15b6] = 0x9b7d, [0x15b7] = 0x9b88,
  [0x15b8] = 0x9d25, [0x15b9] = 0x9d17, [0x15ba] = 0x9d20, [0x15bb] = 0x9d1e,
  [0x15bc] = 0x9d14, [0x15bd] = 0x9d29, [0x15be] = 0x9d1d, [0x15bf] = 0x9d18,
  [0x15c0] = 0x9d22, [0x15c1] = 0x9d10, [0x15c2] = 0x9d19, [0x15c3] = 0x9d1f,
  [0x15c4] = 0x9e88, [0x15c5] = 0x9e86, [0x15c6] = 0x9e87, [0x15c7] = 0x9eae,
  [0x15c8] = 0x9ead, [0x15c9] = 0x9ed5, [0x15ca] = 0x9ed6, [0x15cb] = 0x9efa,
  [0x15cc] = 0x9f12, [0x15cd] = 0x9f3d, [0x15ce] = 0x5126, [0x15cf] = 0x5125,
  [0x15d0] = 0x5122, [0x15d1] = 0x5124, [0x15d2] = 0x5120, [0x15d3] = 0x5129,
  [0x15d4] = 0x52f4, [0x15d5] = 0x5693, [0x15d6] = 0x568c, [0x15d7] = 0x568d,
  [0x15d8] = 0x5686, [0x15d9] = 0x5684, [0x15da] = 0x5683, [0x15db] = 0x567e,
  [0x15dc] = 0x5682, [0x15dd] = 0x567f, [0x15de] = 0x5681, [0x15df] = 0x58d6,
  [0x15e0] = 0x58d4, [0x15e1] = 0x58cf, [0x15e2] = 0x58d2, [0x15e3] = 0x5b2d,
  [0x15e4] = 0x5b25, [0x15e5] = 0x5b32, [0x15e6] = 0x5b23, [0x15e7] = 0x5b2c,
  [0x15e8] = 0x5b27, [0x15e9] = 0x5b26, [0x15ea] = 0x5b2f, [0x15eb] = 0x5b2e,
  [0x15ec] = 0x5b7b, [0x15ed] = 0x5bf1, [0x15ee] = 0x5bf2, [0x15ef] = 0x5db7,
  [0x15f0] = 0x5e6c, [0x15f1] = 0x5e6a, [0x15f2] = 0x5fbe, [0x15f3] = 0x61c3,
  [0x15f4] = 0x61b5, [0x15f5] = 0x61bc, [0x15f6] = 0x61e7, [0x15f7] = 0x61e0,
  [0x15f8] = 0x61e5, [0x15f9] = 0x61e4, [0x15fa] = 0x61e8, [0x15fb] = 0x61de,
  [0x15fc] = 0x64ef, [0x15fd] = 0x64e9, [0x15fe] = 0x64e3, [0x15ff] = 0x64eb,
  [0x1600] = 0x64e4, [0x1601] = 0x64e8, [0x1602] = 0x6581, [0x1603] = 0x6580,
  [0x1604] = 0x65b6, [0x1605] = 0x65da, [0x1606] = 0x66d2, [0x1607] = 0x6a8d,
  [0x1608] = 0x6a96, [0x1609] = 0x6a81, [0x160a] = 0x6aa5, [0x160b] = 0x6a89,
  [0x160c] = 0x6a9f, [0x160d] = 0x6a9b, [0x160e] = 0x6aa1, [0x160f] = 0x6a9e,
  [0x1610] = 0x6a87, [0x1611] = 0x6a93, [0x1612] = 0x6a8e, [0x1613] = 0x6a95,
  [0x1614] = 0x6a83, [0x1615] = 0x6aa8, [0x1616] = 0x6aa4, [0x1617] = 0x6a91,
  [0x1618] = 0x6a7f, [0x1619] = 0x6aa6, [0x161a] = 0x6a9a, [0x161b] = 0x6a85,
  [0x161c] = 0x6a8c, [0x161d] = 0x6a92, [0x161e] = 0x6b5b, [0x161f] = 0x6bad,
  [0x1620] = 0x6c09, [0x1621] = 0x6fcc, [0x1622] = 0x6fa9, [0x1623] = 0x6ff4,
  [0x1624] = 0x6fd4, [0x1625] = 0x6fe3, [0x1626] = 0x6fdc, [0x1627] = 0x6fed,
  [0x1628] = 0x6fe7, [0x1629] = 0x6fe6, [0x162a] = 0x6fde, [0x162b] = 0x6ff2,
  [0x162c] = 0x6fdd, [0x162d] = 0x6fe2, [0x162e] = 0x6fe8, [0x162f] = 0x71e1,
  [0x1630] = 0x71f1, [0x1631] = 0x71e8, [0x1632] = 0x71f2, [0x1633] = 0x71e4,
  [0x1634] = 0x71f0, [0x1635] = 0x71e2, [0x1636] = 0x7373, [0x1637] = 0x736e,
  [0x1638] = 0x736f, [0x1639] = 0x7497, [0x163a] = 0x74b2, [0x163b] = 0x74ab,
  [0x163c] = 0x7490, [0x163d] = 0x74aa, [0x163e] = 0x74ad, [0x163f] = 0x74b1,
  [0x1640] = 0x74a5, [0x1641] = 0x74af, [0x1642] = 0x7510, [0x1643] = 0x7511,
  [0x1644] = 0x7512, [0x1645] = 0x750f, [0x1646] = 0x7584, [0x1647] = 0x7643,
  [0x1648] = 0x7648, [0x1649] = 0x7649, [0x164a] = 0x7647, [0x164b] = 0x76a4,
  [0x164c] = 0x76e9, [0x164d] = 0x77b5, [0x164e] = 0x77ab, [0x164f] = 0x77b2,
  [0x1650] = 0x77b7, [0x1651] = 0x77b6, [0x1652] = 0x77b4, [0x1653] = 0x77b1,
  [0x1654] = 0x77a8, [0x1655] = 0x77f0, [0x1656] = 0x78f3, [0x1657] = 0x78fd,
  [0x1658] = 0x7902, [0x1659] = 0x78fb, [0x165a] = 0x78fc, [0x165b] = 0x78ff,
  [0x165c] = 0x78f2, [0x165d] = 0x7905, [0x165e] = 0x78f9, [0x165f] = 0x78fe,
  [0x1660] = 0x7904, [0x1661] = 0x79ab, [0x1662] = 0x79a8, [0x1663] = 0x7a5c,
  [0x1664] = 0x7a5b, [0x1665] = 0x7a56, [0x1666] = 0x7a58, [0x1667] = 0x7a54,
  [0x1668] = 0x7a5a, [0x1669] = 0x7abe, [0x166a] = 0x7ac0, [0x166b] = 0x7ac1,
  [0x166c] = 0x7c05, [0x166d] = 0x7c0f, [0x166e] = 0x7bf2, [0x166f] = 0x7c00,
  [0x1670] = 0x7bff, [0x1671] = 0x7bfb, [0x1672] = 0x7c0e, [0x1673] = 0x7bf4,
  [0x1674] = 0x7c0b, [0x1675] = 0x7bf3, [0x1676] = 0x7c02, [0x1677] = 0x7c09,
  [0x1678] = 0x7c03, [0x1679] = 0x7c01, [0x167a] = 0x7bf8, [0x167b] = 0x7bfd,
  [0x167c] = 0x7c06, [0x167d] = 0x7bf0, [0x167e] = 0x7bf1, [0x167f] = 0x7c10,
  [0x1680] = 0x7c0a, [0x1681] = 0x7ce8, [0x1682] = 0x7e2d, [0x1683] = 0x7e3c,
  [0x1684] = 0x7e42, [0x1685] = 0x7e33, [0x1686] = 0x9848, [0x1687] = 0x7e38,
  [0x1688] = 0x7e2a, [0x1689] = 0x7e49, [0x168a] = 0x7e40, [0x168b] = 0x7e47,
  [0x168c] = 0x7e29, [0x168d] = 0x7e4c, [0x168e] = 0x7e30, [0x168f] = 0x7e3b,
  [0x1690] = 0x7e36, [0x1691] = 0x7e44, [0x1692] = 0x7e3a, [0x1693] = 0x7f45,
  [0x1694] = 0x7f7f, [0x1695] = 0x7f7e, [0x1696] = 0x7f7d, [0x1697] = 0x7ff4,
  [0x1698] = 0x7ff2, [0x1699] = 0x802c, [0x169a] = 0x81bb, [0x169b] = 0x81c4,
  [0x169c] = 0x81cc, [0x169d] = 0x81ca, [0x169e] = 0x81c5, [0x169f] = 0x81c7,
  [0x16a0] = 0x81bc, [0x16a1] = 0x81e9, [0x16a2] = 0x825b, [0x16a3] = 0x825a,
  [0x16a4] = 0x825c, [0x16a5] = 0x8583, [0x16a6] = 0x8580, [0x16a7] = 0x858f,
  [0x16a8] = 0x85a7, [0x16a9] = 0x8595, [0x16aa] = 0x85a0, [0x16ab] = 0x858b,
  [0x16ac] = 0x85a3, [0x16ad] = 0x857b, [0x16ae] = 0x85a4, [0x16af] = 0x859a,
  [0x16b0] = 0x859e, [0x16b1] = 0x8577, [0x16b2] = 0x857c, [0x16b3] = 0x8589,
  [0x16b4] = 0x85a1, [0x16b5] = 0x857a, [0x16b6] = 0x8578, [0x16b7] = 0x8557,
  [0x16b8] = 0x858e, [0x16b9] = 0x8596, [0x16ba] = 0x8586, [0x16bb] = 0x858d,
  [0x16bc] = 0x8599, [0x16bd] = 0x859d, [0x16be] = 0x8581, [0x16bf] = 0x85a2,
  [0x16c0] = 0x8582, [0x16c1] = 0x8588, [0x16c2] = 0x8585, [0x16c3] = 0x8579,
  [0x16c4] = 0x8576, [0x16c5] = 0x8598, [0x16c6] = 0x8590, [0x16c7] = 0x859f,
  [0x16c8] = 0x8668, [0x16c9] = 0x87be, [0x16ca] = 0x87aa, [0x16cb] = 0x87ad,
  [0x16cc] = 0x87c5, [0x16cd] = 0x87b0, [0x16ce] = 0x87ac, [0x16cf] = 0x87b9,
  [0x16d0] = 0x87b5, [0x16d1] = 0x87bc, [0x16d2] = 0x87ae, [0x16d3] = 0x87c9,
  [0x16d4] = 0x87c3, [0x16d5] = 0x87c2, [0x16d6] = 0x87cc, [0x16d7] = 0x87b7,
  [0x16d8] = 0x87af, [0x16d9] = 0x87c4, [0x16da] = 0x87ca, [0x16db] = 0x87b4,
  [0x16dc] = 0x87b6, [0x16dd] = 0x87bf, [0x16de] = 0x87b8, [0x16df] = 0x87bd,
  [0x16e0] = 0x87de, [0x16e1] = 0x87b2, [0x16e2] = 0x8935, [0x16e3] = 0x8933,
  [0x16e4] = 0x893c, [0x16e5] = 0x893e, [0x16e6] = 0x8941, [0x16e7] = 0x8952,
  [0x16e8] = 0x8937, [0x16e9] = 0x8942, [0x16ea] = 0x89ad, [0x16eb] = 0x89af,
  [0x16ec] = 0x89ae, [0x16ed] = 0x89f2, [0x16ee] = 0x89f3, [0x16ef] = 0x8b1e,
  [0x16f0] = 0x8b18, [0x16f1] = 0x8b16, [0x16f2] = 0x8b11, [0x16f3] = 0x8b05,
  [0x16f4] = 0x8b0b, [0x16f5] = 0x8b22, [0x16f6] = 0x8b0f, [0x16f7] = 0x8b12,
  [0x16f8] = 0x8b15, [0x16f9] = 0x8b07, [0x16fa] = 0x8b0d, [0x16fb] = 0x8b08,
  [0x16fc] = 0x8b06, [0x16fd] = 0x8b1c, [0x16fe] = 0x8b13, [0x16ff] = 0x8b1a,
  [0x1700] = 0x8c4f, [0x1701] = 0x8c70, [0x1702] = 0x8c72, [0x1703] = 0x8c71,
  [0x1704] = 0x8c6f, [0x1705] = 0x8c95, [0x1706] = 0x8c94, [0x1707] = 0x8cf9,
  [0x1708] = 0x8d6f, [0x1709] = 0x8e4e, [0x170a] = 0x8e4d, [0x170b] = 0x8e53,
  [0x170c] = 0x8e50, [0x170d] = 0x8e4c, [0x170e] = 0x8e47, [0x170f] = 0x8f43,
  [0x1710] = 0x8f40, [0x1711] = 0x9085, [0x1712] = 0x907e, [0x1713] = 0x9138,
  [0x1714] = 0x919a, [0x1715] = 0x91a2, [0x1716] = 0x919b, [0x1717] = 0x9199,
  [0x1718] = 0x919f, [0x1719] = 0x91a1, [0x171a] = 0x919d, [0x171b] = 0x91a0,
  [0x171c] = 0x93a1, [0x171d] = 0x9383, [0x171e] = 0x93af, [0x171f] = 0x9364,
  [0x1720] = 0x9356, [0x1721] = 0x9347, [0x1722] = 0x937c, [0x1723] = 0x9358,
  [0x1724] = 0x935c, [0x1725] = 0x9376, [0x1726] = 0x9349, [0x1727] = 0x9350,
  [0x1728] = 0x9351, [0x1729] = 0x9360, [0x172a] = 0x936d, [0x172b] = 0x938f,
  [0x172c] = 0x934c, [0x172d] = 0x936a, [0x172e] = 0x9379, [0x172f] = 0x9357,
  [0x1730] = 0x9355, [0x1731] = 0x9352, [0x1732] = 0x934f, [0x1733] = 0x9371,
  [0x1734] = 0x9377, [0x1735] = 0x937b, [0x1736] = 0x9361, [0x1737] = 0x935e,
  [0x1738] = 0x9363, [0x1739] = 0x9367, [0x173a] = 0x934e, [0x173b] = 0x9359,
  [0x173c] = 0x95c7, [0x173d] = 0x95c0, [0x173e] = 0x95c9, [0x173f] = 0x95c3,
  [0x1740] = 0x95c5, [0x1741] = 0x95b7, [0x1742] = 0x96ae, [0x1743] = 0x96b0,
  [0x1744] = 0x96ac, [0x1745] = 0x9720, [0x1746] = 0x971f, [0x1747] = 0x9718,
  [0x1748] = 0x971d, [0x1749] = 0x9719, [0x174a] = 0x979a, [0x174b] = 0x97a1,
  [0x174c] = 0x979c, [0x174d] = 0x979e, [0x174e] = 0x979d, [0x174f] = 0x97d5,
  [0x1750] = 0x97d4, [0x1751] = 0x97f1, [0x1752] = 0x9841, [0x1753] = 0x9844,
  [0x1754] = 0x984a, [0x1755] = 0x9849, [0x1756] = 0x9845, [0x1757] = 0x9843,
  [0x1758] = 0x9925, [0x1759] = 0x992b, [0x175a] = 0x992c, [0x175b] = 0x992a,
  [0x175c] = 0x9933, [0x175d] = 0x9932, [0x175e] = 0x992f, [0x175f] = 0x992d,
  [0x1760] = 0x9931, [0x1761] = 0x9930, [0x1762] = 0x9998, [0x1763] = 0x99a3,
  [0x1764] = 0x99a1, [0x1765] = 0x9a02, [0x1766] = 0x99fa, [0x1767] = 0x99f4,
  [0x1768] = 0x99f7, [0x1769] = 0x99f9, [0x176a] = 0x99f8, [0x176b] = 0x99f6,
  [0x176c] = 0x99fb, [0x176d] = 0x99fd, [0x176e] = 0x99fe, [0x176f] = 0x99fc,
  [0x1770] = 0x9a03, [0x1771] = 0x9abe, [0x1772] = 0x9afe, [0x1773] = 0x9afd,
  [0x1774] = 0x9b01, [0x1775] = 0x9afc, [0x1776] = 0x9b48, [0x1777] = 0x9b9a,
  [0x1778] = 0x9ba8, [0x1779] = 0x9b9e, [0x177a] = 0x9b9b, [0x177b] = 0x9ba6,
  [0x177c] = 0x9ba1, [0x177d] = 0x9ba5, [0x177e] = 0x9ba4, [0x177f] = 0x9b86,
  [0x1780] = 0x9ba2, [0x1781] = 0x9ba0, [0x1782] = 0x9baf, [0x1783] = 0x9d33,
  [0x1784] = 0x9d41, [0x1785] = 0x9d67, [0x1786] = 0x9d36, [0x1787] = 0x9d2e,
  [0x1788] = 0x9d2f, [0x1789] = 0x9d31, [0x178a] = 0x9d38, [0x178b] = 0x9d30,
  [0x178c] = 0x9d45, [0x178d] = 0x9d42, [0x178e] = 0x9d43, [0x178f] = 0x9d3e,
  [0x1790] = 0x9d37, [0x1791] = 0x9d40, [0x1792] = 0x9d3d, [0x1793] = 0x7ff5,
  [0x1794] = 0x9d2d, [0x1795] = 0x9e8a, [0x1796] = 0x9e89, [0x1797] = 0x9e8d,
  [0x1798] = 0x9eb0, [0x1799] = 0x9ec8, [0x179a] = 0x9eda, [0x179b] = 0x9efb,
  [0x179c] = 0x9eff, [0x179d] = 0x9f24, [0x179e] = 0x9f23, [0x179f] = 0x9f22,
  [0x17a0] = 0x9f54, [0x17a1] = 0x9fa0, [0x17a2] = 0x5131, [0x17a3] = 0x512d,
  [0x17a4] = 0x512e, [0x17a5] = 0x5698, [0x17a6] = 0x569c, [0x17a7] = 0x5697,
  [0x17a8] = 0x569a, [0x17a9] = 0x569d, [0x17aa] = 0x5699, [0x17ab] = 0x5970,
  [0x17ac] = 0x5b3c, [0x17ad] = 0x5c69, [0x17ae] = 0x5c6a, [0x17af] = 0x5dc0,
  [0x17b0] = 0x5e6d, [0x17b1] = 0x5e6e, [0x17b2] = 0x61d8, [0x17b3] = 0x61df,
  [0x17b4] = 0x61ed, [0x17b5] = 0x61ee, [0x17b6] = 0x61f1, [0x17b7] = 0x61ea,
  [0x17b8] = 0x61f0, [0x17b9] = 0x61eb, [0x17ba] = 0x61d6, [0x17bb] = 0x61e9,
  [0x17bc] = 0x64ff, [0x17bd] = 0x6504, [0x17be] = 0x64fd, [0x17bf] = 0x64f8,
  [0x17c0] = 0x6501, [0x17c1] = 0x6503, [0x17c2] = 0x64fc, [0x17c3] = 0x6594,
  [0x17c4] = 0x65db, [0x17c5] = 0x66da, [0x17c6] = 0x66db, [0x17c7] = 0x66d8,
  [0x17c8] = 0x6ac5, [0x17c9] = 0x6ab9, [0x17ca] = 0x6abd, [0x17cb] = 0x6ae1,
  [0x17cc] = 0x6ac6, [0x17cd] = 0x6aba, [0x17ce] = 0x6ab6, [0x17cf] = 0x6ab7,
  [0x17d0] = 0x6ac7, [0x17d1] = 0x6ab4, [0x17d2] = 0x6aad, [0x17d3] = 0x6b5e,
  [0x17d4] = 0x6bc9, [0x17d5] = 0x6c0b, [0x17d6] = 0x7007, [0x17d7] = 0x700c,
  [0x17d8] = 0x700d, [0x17d9] = 0x7001, [0x17da] = 0x7005, [0x17db] = 0x7014,
  [0x17dc] = 0x700e, [0x17dd] = 0x6fff, [0x17de] = 0x7000, [0x17df] = 0x6ffb,
  [0x17e0] = 0x7026, [0x17e1] = 0x6ffc, [0x17e2] = 0x6ff7, [0x17e3] = 0x700a,
  [0x17e4] = 0x7201, [0x17e5] = 0x71ff, [0x17e6] = 0x71f9, [0x17e7] = 0x7203,
  [0x17e8] = 0x71fd, [0x17e9] = 0x7376, [0x17ea] = 0x74b8, [0x17eb] = 0x74c0,
  [0x17ec] = 0x74b5, [0x17ed] = 0x74c1, [0x17ee] = 0x74be, [0x17ef] = 0x74b6,
  [0x17f0] = 0x74bb, [0x17f1] = 0x74c2, [0x17f2] = 0x7514, [0x17f3] = 0x7513,
  [0x17f4] = 0x765c, [0x17f5] = 0x7664, [0x17f6] = 0x7659, [0x17f7] = 0x7650,
  [0x17f8] = 0x7653, [0x17f9] = 0x7657, [0x17fa] = 0x765a, [0x17fb] = 0x76a6,
  [0x17fc] = 0x76bd, [0x17fd] = 0x76ec, [0x17fe] = 0x77c2, [0x17ff] = 0x77ba,
  [0x1800] = 0x790c, [0x1801] = 0x7913, [0x1802] = 0x7914, [0x1803] = 0x7909,
  [0x1804] = 0x7910, [0x1805] = 0x7912, [0x1806] = 0x7911, [0x1807] = 0x79ad,
  [0x1808] = 0x79ac, [0x1809] = 0x7a5f, [0x180a] = 0x7c1c, [0x180b] = 0x7c29,
  [0x180c] = 0x7c19, [0x180d] = 0x7c20, [0x180e] = 0x7c1f, [0x180f] = 0x7c2d,
  [0x1810] = 0x7c1d, [0x1811] = 0x7c26, [0x1812] = 0x7c28, [0x1813] = 0x7c22,
  [0x1814] = 0x7c25, [0x1815] = 0x7c30, [0x1816] = 0x7e5c, [0x1817] = 0x7e50,
  [0x1818] = 0x7e56, [0x1819] = 0x7e63, [0x181a] = 0x7e58, [0x181b] = 0x7e62,
  [0x181c] = 0x7e5f, [0x181d] = 0x7e51, [0x181e] = 0x7e60, [0x181f] = 0x7e57,
  [0x1820] = 0x7e53, [0x1821] = 0x7fb5, [0x1822] = 0x7fb3, [0x1823] = 0x7ff7,
  [0x1824] = 0x7ff8, [0x1825] = 0x8075, [0x1826] = 0x81d1, [0x1827] = 0x81d2,
  [0x1828] = 0x81d0, [0x1829] = 0x825f, [0x182a] = 0x825e, [0x182b] = 0x85b4,
  [0x182c] = 0x85c6, [0x182d] = 0x85c0, [0x182e] = 0x85c3, [0x182f] = 0x85c2,
  [0x1830] = 0x85b3, [0x1831] = 0x85b5, [0x1832] = 0x85bd, [0x1833] = 0x85c7,
  [0x1834] = 0x85c4, [0x1835] = 0x85bf, [0x1836] = 0x85cb, [0x1837] = 0x85ce,
  [0x1838] = 0x85c8, [0x1839] = 0x85c5, [0x183a] = 0x85b1, [0x183b] = 0x85b6,
  [0x183c] = 0x85d2, [0x183d] = 0x8624, [0x183e] = 0x85b8, [0x183f] = 0x85b7,
  [0x1840] = 0x85be, [0x1841] = 0x8669, [0x1842] = 0x87e7, [0x1843] = 0x87e6,
  [0x1844] = 0x87e2, [0x1845] = 0x87db, [0x1846] = 0x87eb, [0x1847] = 0x87ea,
  [0x1848] = 0x87e5, [0x1849] = 0x87df, [0x184a] = 0x87f3, [0x184b] = 0x87e4,
  [0x184c] = 0x87d4, [0x184d] = 0x87dc, [0x184e] = 0x87d3, [0x184f] = 0x87ed,
  [0x1850] = 0x87d8, [0x1851] = 0x87e3, [0x1852] = 0x87d7, [0x1853] = 0x87d9,
  [0x1854] = 0x8801, [0x1855] = 0x87f4, [0x1856] = 0x87e8, [0x1857] = 0x87dd,
  [0x1858] = 0x8953, [0x1859] = 0x894b, [0x185a] = 0x894f, [0x185b] = 0x894c,
  [0x185c] = 0x8946, [0x185d] = 0x8950, [0x185e] = 0x8951, [0x185f] = 0x8949,
  [0x1860] = 0x8b2a, [0x1861] = 0x8b27, [0x1862] = 0x8b23, [0x1863] = 0x8b33,
  [0x1864] = 0x8b30, [0x1865] = 0x8b35, [0x1866] = 0x8b47, [0x1867] = 0x8b2f,
  [0x1868] = 0x8b3c, [0x1869] = 0x8b3e, [0x186a] = 0x8b31, [0x186b] = 0x8b25,
  [0x186c] = 0x8b37, [0x186d] = 0x8b26, [0x186e] = 0x8b36, [0x186f] = 0x8b2e,
  [0x1870] = 0x8b24, [0x1871] = 0x8b3b, [0x1872] = 0x8b3d, [0x1873] = 0x8b3a,
  [0x1874] = 0x8c42, [0x1875] = 0x8c75, [0x1876] = 0x8c99, [0x1877] = 0x8c98,
  [0x1878] = 0x8c97, [0x1879] = 0x8cfe, [0x187a] = 0x8d04, [0x187b] = 0x8d02,
  [0x187c] = 0x8d00, [0x187d] = 0x8e5c, [0x187e] = 0x8e62, [0x187f] = 0x8e60,
  [0x1880] = 0x8e57, [0x1881] = 0x8e56, [0x1882] = 0x8e5e, [0x1883] = 0x8e65,
  [0x1884] = 0x8e67, [0x1885] = 0x8e5b, [0x1886] = 0x8e5a, [0x1887] = 0x8e61,
  [0x1888] = 0x8e5d, [0x1889] = 0x8e69, [0x188a] = 0x8e54, [0x188b] = 0x8f46,
  [0x188c] = 0x8f47, [0x188d] = 0x8f48, [0x188e] = 0x8f4b, [0x188f] = 0x9128,
  [0x1890] = 0x913a, [0x1891] = 0x913b, [0x1892] = 0x913e, [0x1893] = 0x91a8,
  [0x1894] = 0x91a5, [0x1895] = 0x91a7, [0x1896] = 0x91af, [0x1897] = 0x91aa,
  [0x1898] = 0x93b5, [0x1899] = 0x938c, [0x189a] = 0x9392, [0x189b] = 0x93b7,
  [0x189c] = 0x939b, [0x189d] = 0x939d, [0x189e] = 0x9389, [0x189f] = 0x93a7,
  [0x18a0] = 0x938e, [0x18a1] = 0x93aa, [0x18a2] = 0x939e, [0x18a3] = 0x93a6,
  [0x18a4] = 0x9395, [0x18a5] = 0x9388, [0x18a6] = 0x9399, [0x18a7] = 0x939f,
  [0x18a8] = 0x9380, [0x18a9] = 0x938d, [0x18aa] = 0x93b1, [0x18ab] = 0x9391,
  [0x18ac] = 0x93b2, [0x18ad] = 0x93a4, [0x18ae] = 0x93a8, [0x18af] = 0x93b4,
  [0x18b0] = 0x93a3, [0x18b1] = 0x95d2, [0x18b2] = 0x95d3, [0x18b3] = 0x95d1,
  [0x18b4] = 0x96b3, [0x18b5] = 0x96d7, [0x18b6] = 0x96da, [0x18b7] = 0x5dc2,
  [0x18b8] = 0x96df, [0x18b9] = 0x96d8, [0x18ba] = 0x96dd, [0x18bb] = 0x9723,
  [0x18bc] = 0x9722, [0x18bd] = 0x9725, [0x18be] = 0x97ac, [0x18bf] = 0x97ae,
  [0x18c0] = 0x97a8, [0x18c1] = 0x97ab, [0x18c2] = 0x97a4, [0x18c3] = 0x97aa,
  [0x18c4] = 0x97a2, [0x18c5] = 0x97a5, [0x18c6] = 0x97d7, [0x18c7] = 0x97d9,
  [0x18c8] = 0x97d6, [0x18c9] = 0x97d8, [0x18ca] = 0x97fa, [0x18cb] = 0x9850,
  [0x18cc] = 0x9851, [0x18cd] = 0x9852, [0x18ce] = 0x98b8, [0x18cf] = 0x9941,
  [0x18d0] = 0x993c, [0x18d1] = 0x993a, [0x18d2] = 0x9a0f, [0x18d3] = 0x9a0b,
  [0x18d4] = 0x9a09, [0x18d5] = 0x9a0d, [0x18d6] = 0x9a04, [0x18d7] = 0x9a11,
  [0x18d8] = 0x9a0a, [0x18d9] = 0x9a05, [0x18da] = 0x9a07, [0x18db] = 0x9a06,
  [0x18dc] = 0x9ac0, [0x18dd] = 0x9adc, [0x18de] = 0x9b08, [0x18df] = 0x9b04,
  [0x18e0] = 0x9b05, [0x18e1] = 0x9b29, [0x18e2] = 0x9b35, [0x18e3] = 0x9b4a,
  [0x18e4] = 0x9b4c, [0x18e5] = 0x9b4b, [0x18e6] = 0x9bc7, [0x18e7] = 0x9bc6,
  [0x18e8] = 0x9bc3, [0x18e9] = 0x9bbf, [0x18ea] = 0x9bc1, [0x18eb] = 0x9bb5,
  [0x18ec] = 0x9bb8, [0x18ed] = 0x9bd3, [0x18ee] = 0x9bb6, [0x18ef] = 0x9bc4,
  [0x18f0] = 0x9bb9, [0x18f1] = 0x9bbd, [0x18f2] = 0x9d5c, [0x18f3] = 0x9d53,
  [0x18f4] = 0x9d4f, [0x18f5] = 0x9d4a, [0x18f6] = 0x9d5b, [0x18f7] = 0x9d4b,
  [0x18f8] = 0x9d59, [0x18f9] = 0x9d56, [0x18fa] = 0x9d4c, [0x18fb] = 0x9d57,
  [0x18fc] = 0x9d52, [0x18fd] = 0x9d54, [0x18fe] = 0x9d5f, [0x18ff] = 0x9d58,
  [0x1900] = 0x9d5a, [0x1901] = 0x9e8e, [0x1902] = 0x9e8c, [0x1903] = 0x9edf,
  [0x1904] = 0x9f01, [0x1905] = 0x9f00, [0x1906] = 0x9f16, [0x1907] = 0x9f25,
  [0x1908] = 0x9f2b, [0x1909] = 0x9f2a, [0x190a] = 0x9f29, [0x190b] = 0x9f28,
  [0x190c] = 0x9f4c, [0x190d] = 0x9f55, [0x190e] = 0x5134, [0x190f] = 0x5135,
  [0x1910] = 0x5296, [0x1911] = 0x52f7, [0x1912] = 0x53b4, [0x1913] = 0x56ab,
  [0x1914] = 0x56ad, [0x1915] = 0x56a6, [0x1916] = 0x56a7, [0x1917] = 0x56aa,
  [0x1918] = 0x56ac, [0x1919] = 0x58da, [0x191a] = 0x58dd, [0x191b] = 0x58db,
  [0x191c] = 0x5912, [0x191d] = 0x5b3d, [0x191e] = 0x5b3e, [0x191f] = 0x5b3f,
  [0x1920] = 0x5dc3, [0x1921] = 0x5e70, [0x1922] = 0x5fbf, [0x1923] = 0x61fb,
  [0x1924] = 0x6507, [0x1925] = 0x6510, [0x1926] = 0x650d, [0x1927] = 0x6509,
  [0x1928] = 0x650c, [0x1929] = 0x650e, [0x192a] = 0x6584, [0x192b] = 0x65de,
  [0x192c] = 0x65dd, [0x192d] = 0x66de, [0x192e] = 0x6ae7, [0x192f] = 0x6ae0,
  [0x1930] = 0x6acc, [0x1931] = 0x6ad1, [0x1932] = 0x6ad9, [0x1933] = 0x6acb,
  [0x1934] = 0x6adf, [0x1935] = 0x6adc, [0x1936] = 0x6ad0, [0x1937] = 0x6aeb,
  [0x1938] = 0x6acf, [0x1939] = 0x6acd, [0x193a] = 0x6ade, [0x193b] = 0x6b60,
  [0x193c] = 0x6bb0, [0x193d] = 0x6c0c, [0x193e] = 0x7019, [0x193f] = 0x7027,
  [0x1940] = 0x7020, [0x1941] = 0x7016, [0x1942] = 0x702b, [0x1943] = 0x7021,
  [0x1944] = 0x7022, [0x1945] = 0x7023, [0x1946] = 0x7029, [0x1947] = 0x7017,
  [0x1948] = 0x7024, [0x1949] = 0x701c, [0x194a] = 0x720c, [0x194b] = 0x720a,
  [0x194c] = 0x7207, [0x194d] = 0x7202, [0x194e] = 0x7205, [0x194f] = 0x72a5,
  [0x1950] = 0x72a6, [0x1951] = 0x72a4, [0x1952] = 0x72a3, [0x1953] = 0x72a1,
  [0x1954] = 0x74cb, [0x1955] = 0x74c5, [0x1956] = 0x74b7, [0x1957] = 0x74c3,
  [0x1958] = 0x7516, [0x1959] = 0x7660, [0x195a] = 0x77c9, [0x195b] = 0x77ca,
  [0x195c] = 0x77c4, [0x195d] = 0x77f1, [0x195e] = 0x791d, [0x195f] = 0x791b,
  [0x1960] = 0x7921, [0x1961] = 0x791c, [0x1962] = 0x7917, [0x1963] = 0x791e,
  [0x1964] = 0x79b0, [0x1965] = 0x7a67, [0x1966] = 0x7a68, [0x1967] = 0x7c33,
  [0x1968] = 0x7c3c, [0x1969] = 0x7c39, [0x196a] = 0x7c2c, [0x196b] = 0x7c3b,
  [0x196c] = 0x7cec, [0x196d] = 0x7cea, [0x196e] = 0x7e76, [0x196f] = 0x7e75,
  [0x1970] = 0x7e78, [0x1971] = 0x7e70, [0x1972] = 0x7e77, [0x1973] = 0x7e6f,
  [0x1974] = 0x7e7a, [0x1975] = 0x7e72, [0x1976] = 0x7e74, [0x1977] = 0x7e68,
  [0x1978] = 0x7f4b, [0x1979] = 0x7f4a, [0x197a] = 0x7f83, [0x197b] = 0x7f86,
  [0x197c] = 0x7fb7, [0x197d] = 0x7ffd, [0x197e] = 0x7ffe, [0x197f] = 0x8078,
  [0x1980] = 0x81d7, [0x1981] = 0x81d5, [0x1982] = 0x820b, [0x1983] = 0x8264,
  [0x1984] = 0x8261, [0x1985] = 0x8263, [0x1986] = 0x85eb, [0x1987] = 0x85f1,
  [0x1988] = 0x85ed, [0x1989] = 0x85d9, [0x198a] = 0x85e1, [0x198b] = 0x85e8,
  [0x198c] = 0x85da, [0x198d] = 0x85d7, [0x198e] = 0x85ec, [0x198f] = 0x85f2,
  [0x1990] = 0x85f8, [0x1991] = 0x85d8, [0x1992] = 0x85df, [0x1993] = 0x85e3,
  [0x1994] = 0x85dc, [0x1995] = 0x85d1, [0x1996] = 0x85f0, [0x1997] = 0x85e6,
  [0x1998] = 0x85ef, [0x1999] = 0x85de, [0x199a] = 0x85e2, [0x199b] = 0x8800,
  [0x199c] = 0x87fa, [0x199d] = 0x8803, [0x199e] = 0x87f6, [0x199f] = 0x87f7,
  [0x19a0] = 0x8809, [0x19a1] = 0x880c, [0x19a2] = 0x880b, [0x19a3] = 0x8806,
  [0x19a4] = 0x87fc, [0x19a5] = 0x8808, [0x19a6] = 0x87ff, [0x19a7] = 0x880a,
  [0x19a8] = 0x8802, [0x19a9] = 0x8962, [0x19aa] = 0x895a, [0x19ab] = 0x895b,
  [0x19ac] = 0x8957, [0x19ad] = 0x8961, [0x19ae] = 0x895c, [0x19af] = 0x8958,
  [0x19b0] = 0x895d, [0x19b1] = 0x8959, [0x19b2] = 0x8988, [0x19b3] = 0x89b7,
  [0x19b4] = 0x89b6, [0x19b5] = 0x89f6, [0x19b6] = 0x8b50, [0x19b7] = 0x8b48,
  [0x19b8] = 0x8b4a, [0x19b9] = 0x8b40, [0x19ba] = 0x8b53, [0x19bb] = 0x8b56,
  [0x19bc] = 0x8b54, [0x19bd] = 0x8b4b, [0x19be] = 0x8b55, [0x19bf] = 0x8b51,
  [0x19c0] = 0x8b42, [0x19c1] = 0x8b52, [0x19c2] = 0x8b57, [0x19c3] = 0x8c43,
  [0x19c4] = 0x8c77, [0x19c5] = 0x8c76, [0x19c6] = 0x8c9a, [0x19c7] = 0x8d06,
  [0x19c8] = 0x8d07, [0x19c9] = 0x8d09, [0x19ca] = 0x8dac, [0x19cb] = 0x8daa,
  [0x19cc] = 0x8dad, [0x19cd] = 0x8dab, [0x19ce] = 0x8e6d, [0x19cf] = 0x8e78,
  [0x19d0] = 0x8e73, [0x19d1] = 0x8e6a, [0x19d2] = 0x8e6f, [0x19d3] = 0x8e7b,
  [0x19d4] = 0x8ec2, [0x19d5] = 0x8f52, [0x19d6] = 0x8f51, [0x19d7] = 0x8f4f,
  [0x19d8] = 0x8f50, [0x19d9] = 0x8f53, [0x19da] = 0x8fb4, [0x19db] = 0x9140,
  [0x19dc] = 0x913f, [0x19dd] = 0x91b0, [0x19de] = 0x91ad, [0x19df] = 0x93de,
  [0x19e0] = 0x93c7, [0x19e1] = 0x93cf, [0x19e2] = 0x93c2, [0x19e3] = 0x93da,
  [0x19e4] = 0x93d0, [0x19e5] = 0x93f9, [0x19e6] = 0x93ec, [0x19e7] = 0x93cc,
  [0x19e8] = 0x93d9, [0x19e9] = 0x93a9, [0x19ea] = 0x93e6, [0x19eb] = 0x93ca,
  [0x19ec] = 0x93d4, [0x19ed] = 0x93ee, [0x19ee] = 0x93e3, [0x19ef] = 0x93d5,
  [0x19f0] = 0x93c4, [0x19f1] = 0x93ce, [0x19f2] = 0x93c0, [0x19f3] = 0x93d2,
  [0x19f4] = 0x93a5, [0x19f5] = 0x93e7, [0x19f6] = 0x957d, [0x19f7] = 0x95da,
  [0x19f8] = 0x95db, [0x19f9] = 0x96e1, [0x19fa] = 0x9729, [0x19fb] = 0x972b,
  [0x19fc] = 0x972c, [0x19fd] = 0x9728, [0x19fe] = 0x9726, [0x19ff] = 0x97b3,
  [0x1a00] = 0x97b7, [0x1a01] = 0x97b6, [0x1a02] = 0x97dd, [0x1a03] = 0x97de,
  [0x1a04] = 0x97df, [0x1a05] = 0x985c, [0x1a06] = 0x9859, [0x1a07] = 0x985d,
  [0x1a08] = 0x9857, [0x1a09] = 0x98bf, [0x1a0a] = 0x98bd, [0x1a0b] = 0x98bb,
  [0x1a0c] = 0x98be, [0x1a0d] = 0x9948, [0x1a0e] = 0x9947, [0x1a0f] = 0x9943,
  [0x1a10] = 0x99a6, [0x1a11] = 0x99a7, [0x1a12] = 0x9a1a, [0x1a13] = 0x9a15,
  [0x1a14] = 0x9a25, [0x1a15] = 0x9a1d, [0x1a16] = 0x9a24, [0x1a17] = 0x9a1b,
  [0x1a18] = 0x9a22, [0x1a19] = 0x9a20, [0x1a1a] = 0x9a27, [0x1a1b] = 0x9a23,
  [0x1a1c] = 0x9a1e, [0x1a1d] = 0x9a1c, [0x1a1e] = 0x9a14, [0x1a1f] = 0x9ac2,
  [0x1a20] = 0x9b0b, [0x1a21] = 0x9b0a, [0x1a22] = 0x9b0e, [0x1a23] = 0x9b0c,
  [0x1a24] = 0x9b37, [0x1a25] = 0x9bea, [0x1a26] = 0x9beb, [0x1a27] = 0x9be0,
  [0x1a28] = 0x9bde, [0x1a29] = 0x9be4, [0x1a2a] = 0x9be6, [0x1a2b] = 0x9be2,
  [0x1a2c] = 0x9bf0, [0x1a2d] = 0x9bd4, [0x1a2e] = 0x9bd7, [0x1a2f] = 0x9bec,
  [0x1a30] = 0x9bdc, [0x1a31] = 0x9bd9, [0x1a32] = 0x9be5, [0x1a33] = 0x9bd5,
  [0x1a34] = 0x9be1, [0x1a35] = 0x9bda, [0x1a36] = 0x9d77, [0x1a37] = 0x9d81,
  [0x1a38] = 0x9d8a, [0x1a39] = 0x9d84, [0x1a3a] = 0x9d88, [0x1a3b] = 0x9d71,
  [0x1a3c] = 0x9d80, [0x1a3d] = 0x9d78, [0x1a3e] = 0x9d86, [0x1a3f] = 0x9d8b,
  [0x1a40] = 0x9d8c, [0x1a41] = 0x9d7d, [0x1a42] = 0x9d6b, [0x1a43] = 0x9d74,
  [0x1a44] = 0x9d75, [0x1a45] = 0x9d70, [0x1a46] = 0x9d69, [0x1a47] = 0x9d85,
  [0x1a48] = 0x9d73, [0x1a49] = 0x9d7b, [0x1a4a] = 0x9d82, [0x1a4b] = 0x9d6f,
  [0x1a4c] = 0x9d79, [0x1a4d] = 0x9d7f, [0x1a4e] = 0x9d87, [0x1a4f] = 0x9d68,
  [0x1a50] = 0x9e94, [0x1a51] = 0x9e91, [0x1a52] = 0x9ec0, [0x1a53] = 0x9efc,
  [0x1a54] = 0x9f2d, [0x1a55] = 0x9f40, [0x1a56] = 0x9f41, [0x1a57] = 0x9f4d,
  [0x1a58] = 0x9f56, [0x1a59] = 0x9f57, [0x1a5a] = 0x9f58, [0x1a5b] = 0x5337,
  [0x1a5c] = 0x56b2, [0x1a5d] = 0x56b5, [0x1a5e] = 0x56b3, [0x1a5f] = 0x58e3,
  [0x1a60] = 0x5b45, [0x1a61] = 0x5dc6, [0x1a62] = 0x5dc7, [0x1a63] = 0x5eee,
  [0x1a64] = 0x5eef, [0x1a65] = 0x5fc0, [0x1a66] = 0x5fc1, [0x1a67] = 0x61f9,
  [0x1a68] = 0x6517, [0x1a69] = 0x6516, [0x1a6a] = 0x6515, [0x1a6b] = 0x6513,
  [0x1a6c] = 0x65df, [0x1a6d] = 0x66e8, [0x1a6e] = 0x66e3, [0x1a6f] = 0x66e4,
  [0x1a70] = 0x6af3, [0x1a71] = 0x6af0, [0x1a72] = 0x6aea, [0x1a73] = 0x6ae8,
  [0x1a74] = 0x6af9, [0x1a75] = 0x6af1, [0x1a76] = 0x6aee, [0x1a77] = 0x6aef,
  [0x1a78] = 0x703c, [0x1a79] = 0x7035, [0x1a7a] = 0x702f, [0x1a7b] = 0x7037,
  [0x1a7c] = 0x7034, [0x1a7d] = 0x7031, [0x1a7e] = 0x7042, [0x1a7f] = 0x7038,
  [0x1a80] = 0x703f, [0x1a81] = 0x703a, [0x1a82] = 0x7039, [0x1a83] = 0x702a,
  [0x1a84] = 0x7040, [0x1a85] = 0x703b, [0x1a86] = 0x7033, [0x1a87] = 0x7041,
  [0x1a88] = 0x7213, [0x1a89] = 0x7214, [0x1a8a] = 0x72a8, [0x1a8b] = 0x737d,
  [0x1a8c] = 0x737c, [0x1a8d] = 0x74ba, [0x1a8e] = 0x76ab, [0x1a8f] = 0x76aa,
  [0x1a90] = 0x76be, [0x1a91] = 0x76ed, [0x1a92] = 0x77cc, [0x1a93] = 0x77ce,
  [0x1a94] = 0x77cf, [0x1a95] = 0x77cd, [0x1a96] = 0x77f2, [0x1a97] = 0x7925,
  [0x1a98] = 0x7923, [0x1a99] = 0x7927, [0x1a9a] = 0x7928, [0x1a9b] = 0x7924,
  [0x1a9c] = 0x7929, [0x1a9d] = 0x79b2, [0x1a9e] = 0x7a6e, [0x1a9f] = 0x7a6c,
  [0x1aa0] = 0x7a6d, [0x1aa1] = 0x7af7, [0x1aa2] = 0x7c49, [0x1aa3] = 0x7c48,
  [0x1aa4] = 0x7c4a, [0x1aa5] = 0x7c47, [0x1aa6] = 0x7c45, [0x1aa7] = 0x7cee,
  [0x1aa8] = 0x7e7b, [0x1aa9] = 0x7e7e, [0x1aaa] = 0x7e81, [0x1aab] = 0x7e80,
  [0x1aac] = 0x7fba, [0x1aad] = 0x7fff, [0x1aae] = 0x8079, [0x1aaf] = 0x81db,
  [0x1ab0] = 0x81d9, [0x1ab1] = 0x8268, [0x1ab2] = 0x8269, [0x1ab3] = 0x8622,
  [0x1ab4] = 0x85ff, [0x1ab5] = 0x8601, [0x1ab6] = 0x85fe, [0x1ab7] = 0x861b,
  [0x1ab8] = 0x8600, [0x1ab9] = 0x85f6, [0x1aba] = 0x8604, [0x1abb] = 0x8609,
  [0x1abc] = 0x8605, [0x1abd] = 0x860c, [0x1abe] = 0x85fd, [0x1abf] = 0x8819,
  [0x1ac0] = 0x8810, [0x1ac1] = 0x8811, [0x1ac2] = 0x8817, [0x1ac3] = 0x8813,
  [0x1ac4] = 0x8816, [0x1ac5] = 0x8963, [0x1ac6] = 0x8966, [0x1ac7] = 0x89b9,
  [0x1ac8] = 0x89f7, [0x1ac9] = 0x8b60, [0x1aca] = 0x8b6a, [0x1acb] = 0x8b5d,
  [0x1acc] = 0x8b68, [0x1acd] = 0x8b63, [0x1ace] = 0x8b65, [0x1acf] = 0x8b67,
  [0x1ad0] = 0x8b6d, [0x1ad1] = 0x8dae, [0x1ad2] = 0x8e86, [0x1ad3] = 0x8e88,
  [0x1ad4] = 0x8e84, [0x1ad5] = 0x8f59, [0x1ad6] = 0x8f56, [0x1ad7] = 0x8f57,
  [0x1ad8] = 0x8f55, [0x1ad9] = 0x8f58, [0x1ada] = 0x8f5a, [0x1adb] = 0x908d,
  [0x1adc] = 0x9143, [0x1add] = 0x9141, [0x1ade] = 0x91b7, [0x1adf] = 0x91b5,
  [0x1ae0] = 0x91b2, [0x1ae1] = 0x91b3, [0x1ae2] = 0x940b, [0x1ae3] = 0x9413,
  [0x1ae4] = 0x93fb, [0x1ae5] = 0x9420, [0x1ae6] = 0x940f, [0x1ae7] = 0x9414,
  [0x1ae8] = 0x93fe, [0x1ae9] = 0x9415, [0x1aea] = 0x9410, [0x1aeb] = 0x9428,
  [0x1aec] = 0x9419, [0x1aed] = 0x940d, [0x1aee] = 0x93f5, [0x1aef] = 0x9400,
  [0x1af0] = 0x93f7, [0x1af1] = 0x9407, [0x1af2] = 0x940e, [0x1af3] = 0x9416,
  [0x1af4] = 0x9412, [0x1af5] = 0x93fa, [0x1af6] = 0x9409, [0x1af7] = 0x93f8,
  [0x1af8] = 0x943c, [0x1af9] = 0x940a, [0x1afa] = 0x93ff, [0x1afb] = 0x93fc,
  [0x1afc] = 0x940c, [0x1afd] = 0x93f6, [0x1afe] = 0x9411, [0x1aff] = 0x9406,
  [0x1b00] = 0x95de, [0x1b01] = 0x95e0, [0x1b02] = 0x95df, [0x1b03] = 0x972e,
  [0x1b04] = 0x972f, [0x1b05] = 0x97b9, [0x1b06] = 0x97bb, [0x1b07] = 0x97fd,
  [0x1b08] = 0x97fe, [0x1b09] = 0x9860, [0x1b0a] = 0x9862, [0x1b0b] = 0x9863,
  [0x1b0c] = 0x985f, [0x1b0d] = 0x98c1, [0x1b0e] = 0x98c2, [0x1b0f] = 0x9950,
  [0x1b10] = 0x994e, [0x1b11] = 0x9959, [0x1b12] = 0x994c, [0x1b13] = 0x994b,
  [0x1b14] = 0x9953, [0x1b15] = 0x9a32, [0x1b16] = 0x9a34, [0x1b17] = 0x9a31,
  [0x1b18] = 0x9a2c, [0x1b19] = 0x9a2a, [0x1b1a] = 0x9a36, [0x1b1b] = 0x9a29,
  [0x1b1c] = 0x9a2e, [0x1b1d] = 0x9a38, [0x1b1e] = 0x9a2d, [0x1b1f] = 0x9ac7,
  [0x1b20] = 0x9aca, [0x1b21] = 0x9ac6, [0x1b22] = 0x9b10, [0x1b23] = 0x9b12,
  [0x1b24] = 0x9b11, [0x1b25] = 0x9c0b, [0x1b26] = 0x9c08, [0x1b27] = 0x9bf7,
  [0x1b28] = 0x9c05, [0x1b29] = 0x9c12, [0x1b2a] = 0x9bf8, [0x1b2b] = 0x9c40,
  [0x1b2c] = 0x9c07, [0x1b2d] = 0x9c0e, [0x1b2e] = 0x9c06, [0x1b2f] = 0x9c17,
  [0x1b30] = 0x9c14, [0x1b31] = 0x9c09, [0x1b32] = 0x9d9f, [0x1b33] = 0x9d99,
  [0x1b34] = 0x9da4, [0x1b35] = 0x9d9d, [0x1b36] = 0x9d92, [0x1b37] = 0x9d98,
  [0x1b38] = 0x9d90, [0x1b39] = 0x9d9b, [0x1b3a] = 0x9da0, [0x1b3b] = 0x9d94,
  [0x1b3c] = 0x9d9c, [0x1b3d] = 0x9daa, [0x1b3e] = 0x9d97, [0x1b3f] = 0x9da1,
  [0x1b40] = 0x9d9a, [0x1b41] = 0x9da2, [0x1b42] = 0x9da8, [0x1b43] = 0x9d9e,
  [0x1b44] = 0x9da3, [0x1b45] = 0x9dbf, [0x1b46] = 0x9da9, [0x1b47] = 0x9d96,
  [0x1b48] = 0x9da6, [0x1b49] = 0x9da7, [0x1b4a] = 0x9e99, [0x1b4b] = 0x9e9b,
  [0x1b4c] = 0x9e9a, [0x1b4d] = 0x9ee5, [0x1b4e] = 0x9ee4, [0x1b4f] = 0x9ee7,
  [0x1b50] = 0x9ee6, [0x1b51] = 0x9f30, [0x1b52] = 0x9f2e, [0x1b53] = 0x9f5b,
  [0x1b54] = 0x9f60, [0x1b55] = 0x9f5e, [0x1b56] = 0x9f5d, [0x1b57] = 0x9f59,
  [0x1b58] = 0x9f91, [0x1b59] = 0x513a, [0x1b5a] = 0x5139, [0x1b5b] = 0x5298,
  [0x1b5c] = 0x5297, [0x1b5d] = 0x56c3, [0x1b5e] = 0x56bd, [0x1b5f] = 0x56be,
  [0x1b60] = 0x5b48, [0x1b61] = 0x5b47, [0x1b62] = 0x5dcb, [0x1b63] = 0x5dcf,
  [0x1b64] = 0x5ef1, [0x1b65] = 0x61fd, [0x1b66] = 0x651b, [0x1b67] = 0x6b02,
  [0x1b68] = 0x6afc, [0x1b69] = 0x6b03, [0x1b6a] = 0x6af8, [0x1b6b] = 0x6b00,
  [0x1b6c] = 0x7043, [0x1b6d] = 0x7044, [0x1b6e] = 0x704a, [0x1b6f] = 0x7048,
  [0x1b70] = 0x7049, [0x1b71] = 0x7045, [0x1b72] = 0x7046, [0x1b73] = 0x721d,
  [0x1b74] = 0x721a, [0x1b75] = 0x7219, [0x1b76] = 0x737e, [0x1b77] = 0x7517,
  [0x1b78] = 0x766a, [0x1b79] = 0x77d0, [0x1b7a] = 0x792d, [0x1b7b] = 0x7931,
  [0x1b7c] = 0x792f, [0x1b7d] = 0x7c54, [0x1b7e] = 0x7c53, [0x1b7f] = 0x7cf2,
  [0x1b80] = 0x7e8a, [0x1b81] = 0x7e87, [0x1b82] = 0x7e88, [0x1b83] = 0x7e8b,
  [0x1b84] = 0x7e86, [0x1b85] = 0x7e8d, [0x1b86] = 0x7f4d, [0x1b87] = 0x7fbb,
  [0x1b88] = 0x8030, [0x1b89] = 0x81dd, [0x1b8a] = 0x8618, [0x1b8b] = 0x862a,
  [0x1b8c] = 0x8626, [0x1b8d] = 0x861f, [0x1b8e] = 0x8623, [0x1b8f] = 0x861c,
  [0x1b90] = 0x8619, [0x1b91] = 0x8627, [0x1b92] = 0x862e, [0x1b93] = 0x8621,
  [0x1b94] = 0x8620, [0x1b95] = 0x8629, [0x1b96] = 0x861e, [0x1b97] = 0x8625,
  [0x1b98] = 0x8829, [0x1b99] = 0x881d, [0x1b9a] = 0x881b, [0x1b9b] = 0x8820,
  [0x1b9c] = 0x8824, [0x1b9d] = 0x881c, [0x1b9e] = 0x882b, [0x1b9f] = 0x884a,
  [0x1ba0] = 0x896d, [0x1ba1] = 0x8969, [0x1ba2] = 0x896e, [0x1ba3] = 0x896b,
  [0x1ba4] = 0x89fa, [0x1ba5] = 0x8b79, [0x1ba6] = 0x8b78, [0x1ba7] = 0x8b45,
  [0x1ba8] = 0x8b7a, [0x1ba9] = 0x8b7b, [0x1baa] = 0x8d10, [0x1bab] = 0x8d14,
  [0x1bac] = 0x8daf, [0x1bad] = 0x8e8e, [0x1bae] = 0x8e8c, [0x1baf] = 0x8f5e,
  [0x1bb0] = 0x8f5b, [0x1bb1] = 0x8f5d, [0x1bb2] = 0x9146, [0x1bb3] = 0x9144,
  [0x1bb4] = 0x9145, [0x1bb5] = 0x91b9, [0x1bb6] = 0x943f, [0x1bb7] = 0x943b,
  [0x1bb8] = 0x9436, [0x1bb9] = 0x9429, [0x1bba] = 0x943d, [0x1bbb] = 0x9430,
  [0x1bbc] = 0x9439, [0x1bbd] = 0x942a, [0x1bbe] = 0x9437, [0x1bbf] = 0x942c,
  [0x1bc0] = 0x9440, [0x1bc1] = 0x9431, [0x1bc2] = 0x95e5, [0x1bc3] = 0x95e4,
  [0x1bc4] = 0x95e3, [0x1bc5] = 0x9735, [0x1bc6] = 0x973a, [0x1bc7] = 0x97bf,
  [0x1bc8] = 0x97e1, [0x1bc9] = 0x9864, [0x1bca] = 0x98c9, [0x1bcb] = 0x98c6,
  [0x1bcc] = 0x98c0, [0x1bcd] = 0x9958, [0x1bce] = 0x9956, [0x1bcf] = 0x9a39,
  [0x1bd0] = 0x9a3d, [0x1bd1] = 0x9a46, [0x1bd2] = 0x9a44, [0x1bd3] = 0x9a42,
  [0x1bd4] = 0x9a41, [0x1bd5] = 0x9a3a, [0x1bd6] = 0x9a3f, [0x1bd7] = 0x9acd,
  [0x1bd8] = 0x9b15, [0x1bd9] = 0x9b17, [0x1bda] = 0x9b18, [0x1bdb] = 0x9b16,
  [0x1bdc] = 0x9b3a, [0x1bdd] = 0x9b52, [0x1bde] = 0x9c2b, [0x1bdf] = 0x9c1d,
  [0x1be0] = 0x9c1c, [0x1be1] = 0x9c2c, [0x1be2] = 0x9c23, [0x1be3] = 0x9c28,
  [0x1be4] = 0x9c29, [0x1be5] = 0x9c24, [0x1be6] = 0x9c21, [0x1be7] = 0x9db7,
  [0x1be8] = 0x9db6, [0x1be9] = 0x9dbc, [0x1bea] = 0x9dc1, [0x1beb] = 0x9dc7,
  [0x1bec] = 0x9dca, [0x1bed] = 0x9dcf, [0x1bee] = 0x9dbe, [0x1bef] = 0x9dc5,
  [0x1bf0] = 0x9dc3, [0x1bf1] = 0x9dbb, [0x1bf2] = 0x9db5, [0x1bf3] = 0x9dce,
  [0x1bf4] = 0x9db9, [0x1bf5] = 0x9dba, [0x1bf6] = 0x9dac, [0x1bf7] = 0x9dc8,
  [0x1bf8] = 0x9db1, [0x1bf9] = 0x9dad, [0x1bfa] = 0x9dcc, [0x1bfb] = 0x9db3,
  [0x1bfc] = 0x9dcd, [0x1bfd] = 0x9db2, [0x1bfe] = 0x9e7a, [0x1bff] = 0x9e9c,
  [0x1c00] = 0x9eeb, [0x1c01] = 0x9eee, [0x1c02] = 0x9eed, [0x1c03] = 0x9f1b,
  [0x1c04] = 0x9f18, [0x1c05] = 0x9f1a, [0x1c06] = 0x9f31, [0x1c07] = 0x9f4e,
  [0x1c08] = 0x9f65, [0x1c09] = 0x9f64, [0x1c0a] = 0x9f92, [0x1c0b] = 0x4eb9,
  [0x1c0c] = 0x56c6, [0x1c0d] = 0x56c5, [0x1c0e] = 0x56cb, [0x1c0f] = 0x5971,
  [0x1c10] = 0x5b4b, [0x1c11] = 0x5b4c, [0x1c12] = 0x5dd5, [0x1c13] = 0x5dd1,
  [0x1c14] = 0x5ef2, [0x1c15] = 0x6521, [0x1c16] = 0x6520, [0x1c17] = 0x6526,
  [0x1c18] = 0x6522, [0x1c19] = 0x6b0b, [0x1c1a] = 0x6b08, [0x1c1b] = 0x6b09,
  [0x1c1c] = 0x6c0d, [0x1c1d] = 0x7055, [0x1c1e] = 0x7056, [0x1c1f] = 0x7057,
  [0x1c20] = 0x7052, [0x1c21] = 0x721e, [0x1c22] = 0x721f, [0x1c23] = 0x72a9,
  [0x1c24] = 0x737f, [0x1c25] = 0x74d8, [0x1c26] = 0x74d5, [0x1c27] = 0x74d9,
  [0x1c28] = 0x74d7, [0x1c29] = 0x766d, [0x1c2a] = 0x76ad, [0x1c2b] = 0x7935,
  [0x1c2c] = 0x79b4, [0x1c2d] = 0x7a70, [0x1c2e] = 0x7a71, [0x1c2f] = 0x7c57,
  [0x1c30] = 0x7c5c, [0x1c31] = 0x7c59, [0x1c32] = 0x7c5b, [0x1c33] = 0x7c5a,
  [0x1c34] = 0x7cf4, [0x1c35] = 0x7cf1, [0x1c36] = 0x7e91, [0x1c37] = 0x7f4f,
  [0x1c38] = 0x7f87, [0x1c39] = 0x81de, [0x1c3a] = 0x826b, [0x1c3b] = 0x8634,
  [0x1c3c] = 0x8635, [0x1c3d] = 0x8633, [0x1c3e] = 0x862c, [0x1c3f] = 0x8632,
  [0x1c40] = 0x8636, [0x1c41] = 0x882c, [0x1c42] = 0x8828, [0x1c43] = 0x8826,
  [0x1c44] = 0x882a, [0x1c45] = 0x8825, [0x1c46] = 0x8971, [0x1c47] = 0x89bf,
  [0x1c48] = 0x89be, [0x1c49] = 0x89fb, [0x1c4a] = 0x8b7e, [0x1c4b] = 0x8b84,
  [0x1c4c] = 0x8b82, [0x1c4d] = 0x8b86, [0x1c4e] = 0x8b85, [0x1c4f] = 0x8b7f,
  [0x1c50] = 0x8d15, [0x1c51] = 0x8e95, [0x1c52] = 0x8e94, [0x1c53] = 0x8e9a,
  [0x1c54] = 0x8e92, [0x1c55] = 0x8e90, [0x1c56] = 0x8e96, [0x1c57] = 0x8e97,
  [0x1c58] = 0x8f60, [0x1c59] = 0x8f62, [0x1c5a] = 0x9147, [0x1c5b] = 0x944c,
  [0x1c5c] = 0x9450, [0x1c5d] = 0x944a, [0x1c5e] = 0x944b, [0x1c5f] = 0x944f,
  [0x1c60] = 0x9447, [0x1c61] = 0x9445, [0x1c62] = 0x9448, [0x1c63] = 0x9449,
  [0x1c64] = 0x9446, [0x1c65] = 0x973f, [0x1c66] = 0x97e3, [0x1c67] = 0x986a,
  [0x1c68] = 0x9869, [0x1c69] = 0x98cb, [0x1c6a] = 0x9954, [0x1c6b] = 0x995b,
  [0x1c6c] = 0x9a4e, [0x1c6d] = 0x9a53, [0x1c6e] = 0x9a54, [0x1c6f] = 0x9a4c,
  [0x1c70] = 0x9a4f, [0x1c71] = 0x9a48, [0x1c72] = 0x9a4a, [0x1c73] = 0x9a49,
  [0x1c74] = 0x9a52, [0x1c75] = 0x9a50, [0x1c76] = 0x9ad0, [0x1c77] = 0x9b19,
  [0x1c78] = 0x9b2b, [0x1c79] = 0x9b3b, [0x1c7a] = 0x9b56, [0x1c7b] = 0x9b55,
  [0x1c7c] = 0x9c46, [0x1c7d] = 0x9c48, [0x1c7e] = 0x9c3f, [0x1c7f] = 0x9c44,
  [0x1c80] = 0x9c39, [0x1c81] = 0x9c33, [0x1c82] = 0x9c41, [0x1c83] = 0x9c3c,
  [0x1c84] = 0x9c37, [0x1c85] = 0x9c34, [0x1c86] = 0x9c32, [0x1c87] = 0x9c3d,
  [0x1c88] = 0x9c36, [0x1c89] = 0x9ddb, [0x1c8a] = 0x9dd2, [0x1c8b] = 0x9dde,
  [0x1c8c] = 0x9dda, [0x1c8d] = 0x9dcb, [0x1c8e] = 0x9dd0, [0x1c8f] = 0x9ddc,
  [0x1c90] = 0x9dd1, [0x1c91] = 0x9ddf, [0x1c92] = 0x9de9, [0x1c93] = 0x9dd9,
  [0x1c94] = 0x9dd8, [0x1c95] = 0x9dd6, [0x1c96] = 0x9df5, [0x1c97] = 0x9dd5,
  [0x1c98] = 0x9ddd, [0x1c99] = 0x9eb6, [0x1c9a] = 0x9ef0, [0x1c9b] = 0x9f35,
  [0x1c9c] = 0x9f33, [0x1c9d] = 0x9f32, [0x1c9e] = 0x9f42, [0x1c9f] = 0x9f6b,
  [0x1ca0] = 0x9f95, [0x1ca1] = 0x9fa2, [0x1ca2] = 0x513d, [0x1ca3] = 0x5299,
  [0x1ca4] = 0x58e8, [0x1ca5] = 0x58e7, [0x1ca6] = 0x5972, [0x1ca7] = 0x5b4d,
  [0x1ca8] = 0x5dd8, [0x1ca9] = 0x882f, [0x1caa] = 0x5f4f, [0x1cab] = 0x6201,
  [0x1cac] = 0x6203, [0x1cad] = 0x6204, [0x1cae] = 0x6529, [0x1caf] = 0x6525,
  [0x1cb0] = 0x6596, [0x1cb1] = 0x66eb, [0x1cb2] = 0x6b11, [0x1cb3] = 0x6b12,
  [0x1cb4] = 0x6b0f, [0x1cb5] = 0x6bca, [0x1cb6] = 0x705b, [0x1cb7] = 0x705a,
  [0x1cb8] = 0x7222, [0x1cb9] = 0x7382, [0x1cba] = 0x7381, [0x1cbb] = 0x7383,
  [0x1cbc] = 0x7670, [0x1cbd] = 0x77d4, [0x1cbe] = 0x7c67, [0x1cbf] = 0x7c66,
  [0x1cc0] = 0x7e95, [0x1cc1] = 0x826c, [0x1cc2] = 0x863a, [0x1cc3] = 0x8640,
  [0x1cc4] = 0x8639, [0x1cc5] = 0x863c, [0x1cc6] = 0x8631, [0x1cc7] = 0x863b,
  [0x1cc8] = 0x863e, [0x1cc9] = 0x8830, [0x1cca] = 0x8832, [0x1ccb] = 0x882e,
  [0x1ccc] = 0x8833, [0x1ccd] = 0x8976, [0x1cce] = 0x8974, [0x1ccf] = 0x8973,
  [0x1cd0] = 0x89fe, [0x1cd1] = 0x8b8c, [0x1cd2] = 0x8b8e, [0x1cd3] = 0x8b8b,
  [0x1cd4] = 0x8b88, [0x1cd5] = 0x8c45, [0x1cd6] = 0x8d19, [0x1cd7] = 0x8e98,
  [0x1cd8] = 0x8f64, [0x1cd9] = 0x8f63, [0x1cda] = 0x91bc, [0x1cdb] = 0x9462,
  [0x1cdc] = 0x9455, [0x1cdd] = 0x945d, [0x1cde] = 0x9457, [0x1cdf] = 0x945e,
  [0x1ce0] = 0x97c4, [0x1ce1] = 0x97c5, [0x1ce2] = 0x9800, [0x1ce3] = 0x9a56,
  [0x1ce4] = 0x9a59, [0x1ce5] = 0x9b1e, [0x1ce6] = 0x9b1f, [0x1ce7] = 0x9b20,
  [0x1ce8] = 0x9c52, [0x1ce9] = 0x9c58, [0x1cea] = 0x9c50, [0x1ceb] = 0x9c4a,
  [0x1cec] = 0x9c4d, [0x1ced] = 0x9c4b, [0x1cee] = 0x9c55, [0x1cef] = 0x9c59,
  [0x1cf0] = 0x9c4c, [0x1cf1] = 0x9c4e, [0x1cf2] = 0x9dfb, [0x1cf3] = 0x9df7,
  [0x1cf4] = 0x9def, [0x1cf5] = 0x9de3, [0x1cf6] = 0x9deb, [0x1cf7] = 0x9df8,
  [0x1cf8] = 0x9de4, [0x1cf9] = 0x9df6, [0x1cfa] = 0x9de1, [0x1cfb] = 0x9dee,
  [0x1cfc] = 0x9de6, [0x1cfd] = 0x9df2, [0x1cfe] = 0x9df0, [0x1cff] = 0x9de2,
  [0x1d00] = 0x9dec, [0x1d01] = 0x9df4, [0x1d02] = 0x9df3, [0x1d03] = 0x9de8,
  [0x1d04] = 0x9ded, [0x1d05] = 0x9ec2, [0x1d06] = 0x9ed0, [0x1d07] = 0x9ef2,
  [0x1d08] = 0x9ef3, [0x1d09] = 0x9f06, [0x1d0a] = 0x9f1c, [0x1d0b] = 0x9f38,
  [0x1d0c] = 0x9f37, [0x1d0d] = 0x9f36, [0x1d0e] = 0x9f43, [0x1d0f] = 0x9f4f,
  [0x1d10] = 0x9f71, [0x1d11] = 0x9f70, [0x1d12] = 0x9f6e, [0x1d13] = 0x9f6f,
  [0x1d14] = 0x56d3, [0x1d15] = 0x56cd, [0x1d16] = 0x5b4e, [0x1d17] = 0x5c6d,
  [0x1d18] = 0x652d, [0x1d19] = 0x66ed, [0x1d1a] = 0x66ee, [0x1d1b] = 0x6b13,
  [0x1d1c] = 0x705f, [0x1d1d] = 0x7061, [0x1d1e] = 0x705d, [0x1d1f] = 0x7060,
  [0x1d20] = 0x7223, [0x1d21] = 0x74db, [0x1d22] = 0x74e5, [0x1d23] = 0x77d5,
  [0x1d24] = 0x7938, [0x1d25] = 0x79b7, [0x1d26] = 0x79b6, [0x1d27] = 0x7c6a,
  [0x1d28] = 0x7e97, [0x1d29] = 0x7f89, [0x1d2a] = 0x826d, [0x1d2b] = 0x8643,
  [0x1d2c] = 0x8838, [0x1d2d] = 0x8837, [0x1d2e] = 0x8835, [0x1d2f] = 0x884b,
  [0x1d30] = 0x8b94, [0x1d31] = 0x8b95, [0x1d32] = 0x8e9e, [0x1d33] = 0x8e9f,
  [0x1d34] = 0x8ea0, [0x1d35] = 0x8e9d, [0x1d36] = 0x91be, [0x1d37] = 0x91bd,
  [0x1d38] = 0x91c2, [0x1d39] = 0x946b, [0x1d3a] = 0x9468, [0x1d3b] = 0x9469,
  [0x1d3c] = 0x96e5, [0x1d3d] = 0x9746, [0x1d3e] = 0x9743, [0x1d3f] = 0x9747,
  [0x1d40] = 0x97c7, [0x1d41] = 0x97e5, [0x1d42] = 0x9a5e, [0x1d43] = 0x9ad5,
  [0x1d44] = 0x9b59, [0x1d45] = 0x9c63, [0x1d46] = 0x9c67, [0x1d47] = 0x9c66,
  [0x1d48] = 0x9c62, [0x1d49] = 0x9c5e, [0x1d4a] = 0x9c60, [0x1d4b] = 0x9e02,
  [0x1d4c] = 0x9dfe, [0x1d4d] = 0x9e07, [0x1d4e] = 0x9e03, [0x1d4f] = 0x9e06,
  [0x1d50] = 0x9e05, [0x1d51] = 0x9e00, [0x1d52] = 0x9e01, [0x1d53] = 0x9e09,
  [0x1d54] = 0x9dff, [0x1d55] = 0x9dfd, [0x1d56] = 0x9e04, [0x1d57] = 0x9ea0,
  [0x1d58] = 0x9f1e, [0x1d59] = 0x9f46, [0x1d5a] = 0x9f74, [0x1d5b] = 0x9f75,
  [0x1d5c] = 0x9f76, [0x1d5d] = 0x56d4, [0x1d5e] = 0x652e, [0x1d5f] = 0x65b8,
  [0x1d60] = 0x6b18, [0x1d61] = 0x6b19, [0x1d62] = 0x6b17, [0x1d63] = 0x6b1a,
  [0x1d64] = 0x7062, [0x1d65] = 0x7226, [0x1d66] = 0x72aa, [0x1d67] = 0x77d8,
  [0x1d68] = 0x77d9, [0x1d69] = 0x7939, [0x1d6a] = 0x7c69, [0x1d6b] = 0x7c6b,
  [0x1d6c] = 0x7cf6, [0x1d6d] = 0x7e9a, [0x1d6e] = 0x7e98, [0x1d6f] = 0x7e9b,
  [0x1d70] = 0x7e99, [0x1d71] = 0x81e0, [0x1d72] = 0x81e1, [0x1d73] = 0x8646,
  [0x1d74] = 0x8647, [0x1d75] = 0x8648, [0x1d76] = 0x8979, [0x1d77] = 0x897a,
  [0x1d78] = 0x897c, [0x1d79] = 0x897b, [0x1d7a] = 0x89ff, [0x1d7b] = 0x8b98,
  [0x1d7c] = 0x8b99, [0x1d7d] = 0x8ea5, [0x1d7e] = 0x8ea4, [0x1d7f] = 0x8ea3,
  [0x1d80] = 0x946e, [0x1d81] = 0x946d, [0x1d82] = 0x946f, [0x1d83] = 0x9471,
  [0x1d84] = 0x9473, [0x1d85] = 0x9749, [0x1d86] = 0x9872, [0x1d87] = 0x995f,
  [0x1d88] = 0x9c68, [0x1d89] = 0x9c6e, [0x1d8a] = 0x9c6d, [0x1d8b] = 0x9e0b,
  [0x1d8c] = 0x9e0d, [0x1d8d] = 0x9e10, [0x1d8e] = 0x9e0f, [0x1d8f] = 0x9e12,
  [0x1d90] = 0x9e11, [0x1d91] = 0x9ea1, [0x1d92] = 0x9ef5, [0x1d93] = 0x9f09,
  [0x1d94] = 0x9f47, [0x1d95] = 0x9f78, [0x1d96] = 0x9f7b, [0x1d97] = 0x9f7a,
  [0x1d98] = 0x9f79, [0x1d99] = 0x571e, [0x1d9a] = 0x7066, [0x1d9b] = 0x7c6f,
  [0x1d9c] = 0x883c, [0x1d9d] = 0x8db2, [0x1d9e] = 0x8ea6, [0x1d9f] = 0x91c3,
  [0x1da0] = 0x9474, [0x1da1] = 0x9478, [0x1da2] = 0x9476, [0x1da3] = 0x9475,
  [0x1da4] = 0x9a60, [0x1da5] = 0x9b2e, [0x1da6] = 0x9c74, [0x1da7] = 0x9c73,
  [0x1da8] = 0x9c71, [0x1da9] = 0x9c75, [0x1daa] = 0x9e14, [0x1dab] = 0x9e13,
  [0x1dac] = 0x9ef6, [0x1dad] = 0x9f0a, [0x1dae] = 0x9fa4, [0x1daf] = 0x7068,
  [0x1db0] = 0x7065, [0x1db1] = 0x7cf7, [0x1db2] = 0x866a, [0x1db3] = 0x883e,
  [0x1db4] = 0x883d, [0x1db5] = 0x883f, [0x1db6] = 0x8b9e, [0x1db7] = 0x8c9c,
  [0x1db8] = 0x8ea9, [0x1db9] = 0x8ec9, [0x1dba] = 0x974b, [0x1dbb] = 0x9873,
  [0x1dbc] = 0x9874, [0x1dbd] = 0x98cc, [0x1dbe] = 0x9961, [0x1dbf] = 0x99ab,
  [0x1dc0] = 0x9a64, [0x1dc1] = 0x9a66, [0x1dc2] = 0x9a67, [0x1dc3] = 0x9b24,
  [0x1dc4] = 0x9e15, [0x1dc5] = 0x9e17, [0x1dc6] = 0x9f48, [0x1dc7] = 0x6207,
  [0x1dc8] = 0x6b1e, [0x1dc9] = 0x7227, [0x1dca] = 0x864c, [0x1dcb] = 0x8ea8,
  [0x1dcc] = 0x9482, [0x1dcd] = 0x9480, [0x1dce] = 0x9481, [0x1dcf] = 0x9a69,
  [0x1dd0] = 0x9a68, [0x1dd1] = 0x9e19, [0x1dd2] = 0x864b, [0x1dd3] = 0x8b9f,
  [0x1dd4] = 0x9483, [0x1dd5] = 0x9c79, [0x1dd6] = 0x9eb7, [0x1dd7] = 0x7675,
  [0x1dd8] = 0x9a6b, [0x1dd9] = 0x9c7a, [0x1dda] = 0x9e1d, [0x1ddb] = 0x7069,
  [0x1ddc] = 0x706a, [0x1ddd] = 0x7229, [0x1dde] = 0x9ea4, [0x1ddf] = 0x9f7e,
  [0x1de0] = 0x9f49, [0x1de1] = 0x9f98,
};

/* The following table contains lots of small holes.  It does not pay off
   to use an extra index table.

   The table can be generated using

   egrep '^0xE' .../mappings/eastasia/other/cns11643.txt |
   awk '{print $1, $2}' | perl tab.pl

   where tab.pl is:

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($cns, $ucs4, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns)-0xE2121;
     printf ("\n ") if ($n % 4 eq 0);
     ++$n;
     printf (" [0x%04x] = 0x%04x,",
             int($c / 256) * 94 + ($c & 0xff), $u);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const uint16_t __cns11643l14_to_ucs4_tab[] =
{
  [0x0000] = 0x4e28, [0x0001] = 0x4e36, [0x0002] = 0x4e3f, [0x0003] = 0x4e85,
  [0x0004] = 0x4e05, [0x0005] = 0x4e04, [0x0006] = 0x5182, [0x0007] = 0x5196,
  [0x0008] = 0x5338, [0x0009] = 0x5369, [0x000a] = 0x53b6, [0x000b] = 0x4e2a,
  [0x000c] = 0x4e87, [0x000d] = 0x4e49, [0x000e] = 0x51e2, [0x000f] = 0x4e46,
  [0x0010] = 0x4e8f, [0x0011] = 0x4ebc, [0x0012] = 0x4ebe, [0x0013] = 0x5166,
  [0x0014] = 0x51e3, [0x0015] = 0x5204, [0x0016] = 0x529c, [0x0018] = 0x5902,
  [0x0019] = 0x590a, [0x001a] = 0x5b80, [0x001b] = 0x5ddb, [0x001c] = 0x5e7a,
  [0x001d] = 0x5e7f, [0x001e] = 0x5ef4, [0x001f] = 0x5f50, [0x0020] = 0x5f51,
  [0x0021] = 0x5f61, [0x0022] = 0x961d, [0x0024] = 0x4e63, [0x0025] = 0x4e62,
  [0x0026] = 0x4ea3, [0x0027] = 0x5185, [0x0028] = 0x4ec5, [0x0029] = 0x4ecf,
  [0x002a] = 0x4ece, [0x002b] = 0x4ecc, [0x002c] = 0x5184, [0x002d] = 0x5186,
  [0x0030] = 0x51e4, [0x0031] = 0x5205, [0x0032] = 0x529e, [0x0033] = 0x529d,
  [0x0034] = 0x52fd, [0x0035] = 0x5300, [0x0036] = 0x533a, [0x0038] = 0x5346,
  [0x0039] = 0x535d, [0x003a] = 0x5386, [0x003b] = 0x53b7, [0x003d] = 0x53cc,
  [0x003f] = 0x53ce, [0x0040] = 0x5721, [0x0042] = 0x5e00, [0x0043] = 0x5f0c,
  [0x0044] = 0x6237, [0x0045] = 0x6238, [0x0046] = 0x6534, [0x0047] = 0x6535,
  [0x0048] = 0x65e0, [0x004a] = 0x738d, [0x004b] = 0x4e97, [0x004c] = 0x4ee0,
  [0x004f] = 0x4ee7, [0x0051] = 0x4ee6, [0x0056] = 0x56d8, [0x0057] = 0x518b,
  [0x0058] = 0x518c, [0x0059] = 0x5199, [0x005a] = 0x51e5, [0x005c] = 0x520b,
  [0x005f] = 0x5304, [0x0060] = 0x5303, [0x0061] = 0x5307, [0x0063] = 0x531e,
  [0x0064] = 0x535f, [0x0065] = 0x536d, [0x0066] = 0x5389, [0x0067] = 0x53ba,
  [0x0068] = 0x53d0, [0x006a] = 0x53f6, [0x006b] = 0x53f7, [0x006c] = 0x53f9,
  [0x006e] = 0x53f4, [0x0071] = 0x5724, [0x0072] = 0x5904, [0x0073] = 0x5918,
  [0x0074] = 0x5932, [0x0075] = 0x5930, [0x0076] = 0x5934, [0x0078] = 0x5975,
  [0x007a] = 0x5b82, [0x007b] = 0x5bf9, [0x007c] = 0x5c14, [0x0084] = 0x5e81,
  [0x0085] = 0x5e83, [0x0086] = 0x5f0d, [0x0087] = 0x5f52, [0x0089] = 0x5fca,
  [0x008a] = 0x5fc7, [0x008b] = 0x6239, [0x008d] = 0x624f, [0x008e] = 0x65e7,
  [0x008f] = 0x672f, [0x0090] = 0x6b7a, [0x0091] = 0x6c39, [0x0094] = 0x6c37,
  [0x0095] = 0x6c44, [0x0096] = 0x6c45, [0x0097] = 0x738c, [0x0098] = 0x7592,
  [0x0099] = 0x7676, [0x009a] = 0x9093, [0x009b] = 0x9092, [0x009e] = 0x4e21,
  [0x009f] = 0x4e20, [0x00a0] = 0x4e22, [0x00a1] = 0x4e68, [0x00a2] = 0x4e89,
  [0x00a3] = 0x4e98, [0x00a4] = 0x4ef9, [0x00a5] = 0x4eef, [0x00a8] = 0x4ef8,
  [0x00a9] = 0x4f06, [0x00aa] = 0x4f03, [0x00ab] = 0x4efc, [0x00ac] = 0x4eee,
  [0x00ad] = 0x4f16, [0x00af] = 0x4f28, [0x00b0] = 0x4f1c, [0x00b1] = 0x4f07,
  [0x00b2] = 0x4f1a, [0x00b3] = 0x4efa, [0x00b4] = 0x4f17, [0x00b5] = 0x514a,
  [0x00b7] = 0x5172, [0x00b9] = 0x51b4, [0x00ba] = 0x51b3, [0x00bb] = 0x51b2,
  [0x00bd] = 0x51e8, [0x00bf] = 0x5214, [0x00c0] = 0x520f, [0x00c1] = 0x5215,
  [0x00c2] = 0x5218, [0x00c3] = 0x52a8, [0x00c5] = 0x534b, [0x00c6] = 0x534f,
  [0x00c8] = 0x5350, [0x00ca] = 0x538b, [0x00cc] = 0x53be, [0x00ce] = 0x53d2,
  [0x00cf] = 0x5416, [0x00d0] = 0x53ff, [0x00d2] = 0x5400, [0x00d4] = 0x5405,
  [0x00d5] = 0x5413, [0x00d6] = 0x5415, [0x00d9] = 0x56e3, [0x00da] = 0x5735,
  [0x00db] = 0x5736, [0x00dc] = 0x5731, [0x00dd] = 0x5732, [0x00de] = 0x58ee,
  [0x00df] = 0x5905, [0x00e0] = 0x4e54, [0x00e2] = 0x5936, [0x00e6] = 0x597a,
  [0x00e8] = 0x5986, [0x00eb] = 0x5b86, [0x00ec] = 0x5f53, [0x00ed] = 0x5c18,
  [0x00ef] = 0x5c3d, [0x00f0] = 0x5c78, [0x00f5] = 0x5c80, [0x00f7] = 0x5e08,
  [0x00fc] = 0x5ef5, [0x00fd] = 0x5f0e, [0x0101] = 0x5fd3, [0x0102] = 0x5fda,
  [0x0104] = 0x5fdb, [0x0106] = 0x620f, [0x0107] = 0x625d, [0x0108] = 0x625f,
  [0x0109] = 0x6267, [0x010a] = 0x6257, [0x010b] = 0x9f50, [0x010d] = 0x65eb,
  [0x010e] = 0x65ea, [0x0110] = 0x6737, [0x0112] = 0x6732, [0x0113] = 0x6736,
  [0x0114] = 0x6b22, [0x0115] = 0x6bce, [0x0117] = 0x6c58, [0x0118] = 0x6c51,
  [0x0119] = 0x6c77, [0x011a] = 0x6c3c, [0x011c] = 0x6c5a, [0x011e] = 0x6c53,
  [0x011f] = 0x706f, [0x0120] = 0x7072, [0x0121] = 0x706e, [0x0124] = 0x7073,
  [0x0125] = 0x72b1, [0x0126] = 0x72b2, [0x0128] = 0x738f, [0x012c] = 0x793c,
  [0x012e] = 0x808d, [0x012f] = 0x808e, [0x0131] = 0x827b, [0x0133] = 0x8d71,
  [0x0134] = 0x8fb9, [0x0135] = 0x9096, [0x0136] = 0x909a, [0x0138] = 0x4e24,
  [0x0139] = 0x4e71, [0x013b] = 0x4e9c, [0x013c] = 0x4f45, [0x013d] = 0x4f4a,
  [0x013e] = 0x4f39, [0x013f] = 0x4f37, [0x0141] = 0x4f32, [0x0142] = 0x4f42,
  [0x0144] = 0x4f44, [0x0145] = 0x4f4b, [0x0147] = 0x4f40, [0x0148] = 0x4f35,
  [0x0149] = 0x4f31, [0x014a] = 0x5151, [0x014c] = 0x5150, [0x014d] = 0x514e,
  [0x0150] = 0x519d, [0x0152] = 0x51b5, [0x0153] = 0x51b8, [0x0154] = 0x51ec,
  [0x0155] = 0x5223, [0x0156] = 0x5227, [0x0157] = 0x5226, [0x0158] = 0x521f,
  [0x0159] = 0x522b, [0x015a] = 0x5220, [0x015b] = 0x52b4, [0x015c] = 0x52b3,
  [0x015e] = 0x5325, [0x015f] = 0x533b, [0x0160] = 0x5374, [0x0166] = 0x544d,
  [0x0169] = 0x543a, [0x016c] = 0x5444, [0x016d] = 0x544c, [0x016e] = 0x5423,
  [0x016f] = 0x541a, [0x0170] = 0x5432, [0x0171] = 0x544b, [0x0172] = 0x5421,
  [0x0174] = 0x5434, [0x0175] = 0x5449, [0x0176] = 0x5450, [0x0177] = 0x5422,
  [0x0178] = 0x543f, [0x0179] = 0x5451, [0x017a] = 0x545a, [0x017b] = 0x542f,
  [0x017d] = 0x56e9, [0x017e] = 0x56f2, [0x017f] = 0x56f3, [0x0180] = 0x56ef,
  [0x0181] = 0x56ed, [0x0182] = 0x56ec, [0x0183] = 0x56e6, [0x0184] = 0x5748,
  [0x0186] = 0x5744, [0x0187] = 0x573f, [0x0188] = 0x573c, [0x0189] = 0x5753,
  [0x018a] = 0x5756, [0x018c] = 0x575f, [0x018d] = 0x5743, [0x018e] = 0x5758,
  [0x018f] = 0x5757, [0x0193] = 0x5746, [0x0195] = 0x573d, [0x0197] = 0x5742,
  [0x0198] = 0x5754, [0x0199] = 0x5755, [0x019a] = 0x58f1, [0x019b] = 0x58f2,
  [0x019c] = 0x58f0, [0x019d] = 0x590b, [0x019e] = 0x9ea6, [0x019f] = 0x56f1,
  [0x01a0] = 0x593d, [0x01a2] = 0x5994, [0x01a3] = 0x598c, [0x01a5] = 0x599c,
  [0x01a8] = 0x599f, [0x01aa] = 0x599b, [0x01ac] = 0x5989, [0x01ad] = 0x599a,
  [0x01af] = 0x6588, [0x01b1] = 0x5b8d, [0x01b3] = 0x5bfe, [0x01b4] = 0x5bff,
  [0x01b5] = 0x5bfd, [0x01b6] = 0x5c2b, [0x01b8] = 0x5c84, [0x01b9] = 0x5c8e,
  [0x01ba] = 0x5c9c, [0x01bd] = 0x5c85, [0x01be] = 0x5df5, [0x01bf] = 0x5e09,
  [0x01c2] = 0x5e0b, [0x01c4] = 0x5e92, [0x01c5] = 0x5e90, [0x01c6] = 0x5f03,
  [0x01c8] = 0x5f1e, [0x01c9] = 0x5f63, [0x01cb] = 0x5fe7, [0x01cc] = 0x5ffe,
  [0x01cd] = 0x5fe6, [0x01ce] = 0x5fdc, [0x01cf] = 0x5fce, [0x01d1] = 0x5ffc,
  [0x01d2] = 0x5fdf, [0x01d3] = 0x5fec, [0x01d4] = 0x5ff6, [0x01d6] = 0x5ff2,
  [0x01d7] = 0x5ff0, [0x01d8] = 0x5ff9, [0x01da] = 0x6213, [0x01dd] = 0x623b,
  [0x01de] = 0x623c, [0x01df] = 0x6282, [0x01e3] = 0x6278, [0x01e4] = 0x628b,
  [0x01e6] = 0x629e, [0x01e7] = 0x62a5, [0x01e8] = 0x629b, [0x01e9] = 0x629c,
  [0x01ea] = 0x6299, [0x01eb] = 0x628d, [0x01ec] = 0x6285, [0x01ed] = 0x629d,
  [0x01ee] = 0x6275, [0x01f2] = 0x65f6, [0x01f6] = 0x66f5, [0x01f7] = 0x675b,
  [0x01f9] = 0x6754, [0x01fa] = 0x6752, [0x01fc] = 0x6758, [0x01fd] = 0x6744,
  [0x01fe] = 0x674a, [0x01ff] = 0x6761, [0x0201] = 0x6c7f, [0x0202] = 0x6c91,
  [0x0203] = 0x6c9e, [0x0205] = 0x6c6e, [0x0206] = 0x6c7c, [0x0207] = 0x6c9f,
  [0x0208] = 0x6c75, [0x020a] = 0x6c56, [0x020b] = 0x6ca2, [0x020c] = 0x6c79,
  [0x020e] = 0x6ca1, [0x0210] = 0x6caa, [0x0211] = 0x6ca0, [0x0213] = 0x7079,
  [0x0214] = 0x7077, [0x0215] = 0x707e, [0x0217] = 0x7075, [0x0218] = 0x707b,
  [0x0219] = 0x7264, [0x021b] = 0x72bb, [0x021c] = 0x72bc, [0x021d] = 0x72c7,
  [0x021e] = 0x72b9, [0x021f] = 0x72be, [0x0220] = 0x72b6, [0x0223] = 0x7398,
  [0x0228] = 0x7593, [0x0229] = 0x7680, [0x022b] = 0x7683, [0x022c] = 0x76c0,
  [0x022d] = 0x76c1, [0x0230] = 0x77f4, [0x0231] = 0x77f5, [0x0233] = 0x7acc,
  [0x0234] = 0x7acd, [0x0235] = 0x7cfa, [0x0236] = 0x809f, [0x0237] = 0x8091,
  [0x0238] = 0x8097, [0x0239] = 0x8094, [0x023b] = 0x8286, [0x023c] = 0x828c,
  [0x023e] = 0x8295, [0x0240] = 0x866c, [0x0242] = 0x8fb5, [0x0243] = 0x8fbe,
  [0x0244] = 0x8fc7, [0x0246] = 0x8fc1, [0x0247] = 0x90a9, [0x0248] = 0x90a4,
  [0x024c] = 0x90a8, [0x024d] = 0x9627, [0x024e] = 0x9626, [0x024f] = 0x962b,
  [0x0250] = 0x9633, [0x0251] = 0x9634, [0x0252] = 0x9629, [0x0253] = 0x4e3d,
  [0x0255] = 0x4e9d, [0x0256] = 0x4f93, [0x0257] = 0x4f8a, [0x025a] = 0x4f6d,
  [0x025b] = 0x4f8e, [0x025c] = 0x4fa0, [0x025d] = 0x4fa2, [0x025e] = 0x4fa1,
  [0x025f] = 0x4f9f, [0x0260] = 0x4fa3, [0x0262] = 0x4f72, [0x0264] = 0x4f8c,
  [0x0265] = 0x5156, [0x0268] = 0x5190, [0x026c] = 0x51ed, [0x026d] = 0x51fe,
  [0x026e] = 0x522f, [0x0270] = 0x523c, [0x0271] = 0x5234, [0x0272] = 0x5239,
  [0x0273] = 0x52b9, [0x0274] = 0x52b5, [0x0275] = 0x52bf, [0x0276] = 0x5355,
  [0x0278] = 0x5376, [0x0279] = 0x537a, [0x027a] = 0x5393, [0x027c] = 0x53c1,
  [0x027d] = 0x53c2, [0x027e] = 0x53d5, [0x027f] = 0x5485, [0x0281] = 0x545f,
  [0x0282] = 0x5493, [0x0283] = 0x5489, [0x0284] = 0x5479, [0x0285] = 0x9efe,
  [0x0286] = 0x548f, [0x0287] = 0x5469, [0x0288] = 0x546d, [0x028a] = 0x5494,
  [0x028b] = 0x546a, [0x028c] = 0x548a, [0x028e] = 0x56fd, [0x028f] = 0x56fb,
  [0x0290] = 0x56f8, [0x0292] = 0x56fc, [0x0293] = 0x56f6, [0x0294] = 0x5765,
  [0x0295] = 0x5781, [0x0296] = 0x5763, [0x0297] = 0x5767, [0x0299] = 0x576e,
  [0x029a] = 0x5778, [0x029b] = 0x577f, [0x029e] = 0x58f3, [0x029f] = 0x594b,
  [0x02a0] = 0x594c, [0x02a4] = 0x59ad, [0x02a6] = 0x59c4, [0x02a8] = 0x59c2,
  [0x02a9] = 0x59b0, [0x02ae] = 0x59bf, [0x02b0] = 0x59c9, [0x02b1] = 0x59b8,
  [0x02b2] = 0x59ac, [0x02b6] = 0x59b7, [0x02b7] = 0x59d7, [0x02b9] = 0x5b60,
  [0x02bb] = 0x5b96, [0x02bc] = 0x5b9e, [0x02bd] = 0x5b94, [0x02be] = 0x5b9f,
  [0x02bf] = 0x5b9d, [0x02c1] = 0x5c00, [0x02c2] = 0x5c19, [0x02c5] = 0x5c49,
  [0x02c6] = 0x5c4a, [0x02c8] = 0x5cbb, [0x02c9] = 0x5cc1, [0x02cd] = 0x5cb9,
  [0x02ce] = 0x5c9e, [0x02cf] = 0x5cb4, [0x02d0] = 0x5cba, [0x02d1] = 0x5df6,
  [0x02d2] = 0x5e13, [0x02d3] = 0x5e12, [0x02d4] = 0x5e77, [0x02d6] = 0x5e98,
  [0x02d8] = 0x5e99, [0x02d9] = 0x5e9d, [0x02da] = 0x5ef8, [0x02dc] = 0x5ef9,
  [0x02de] = 0x5f06, [0x02df] = 0x5f21, [0x02e1] = 0x5f25, [0x02e2] = 0x5f55,
  [0x02e6] = 0x5f84, [0x02e7] = 0x5f83, [0x02e8] = 0x6030, [0x02e9] = 0x6007,
  [0x02eb] = 0x6036, [0x02ef] = 0x5fe9, [0x02f0] = 0x603d, [0x02f1] = 0x6008,
  [0x02f4] = 0x62ba, [0x02f5] = 0x62b2, [0x02f7] = 0x62b7, [0x02f8] = 0x62e4,
  [0x02f9] = 0x62a7, [0x02fd] = 0x62d5, [0x02fe] = 0x62e1, [0x02ff] = 0x62dd,
  [0x0300] = 0x62a6, [0x0301] = 0x62c1, [0x0302] = 0x62c5, [0x0303] = 0x62c0,
  [0x0304] = 0x62df, [0x0305] = 0x62e0, [0x0306] = 0x62de, [0x0308] = 0x6589,
  [0x030a] = 0x65a6, [0x030b] = 0x65ba, [0x030d] = 0x65ff, [0x030f] = 0x6617,
  [0x0310] = 0x6618, [0x0311] = 0x6601, [0x0312] = 0x65fe, [0x0314] = 0x670c,
  [0x0316] = 0x676b, [0x0317] = 0x6796, [0x0318] = 0x6782, [0x0319] = 0x678a,
  [0x031b] = 0x67a3, [0x031d] = 0x67a2, [0x031e] = 0x678f, [0x0320] = 0x67f9,
  [0x0321] = 0x6780, [0x0322] = 0x6b26, [0x0323] = 0x6b27, [0x0324] = 0x6b68,
  [0x0325] = 0x6b69, [0x0327] = 0x6b81, [0x0328] = 0x6bb4, [0x0329] = 0x6bd1,
  [0x032c] = 0x6c1c, [0x0332] = 0x6c97, [0x0333] = 0x6c6c, [0x0334] = 0x6cdf,
  [0x0336] = 0x6cea, [0x0338] = 0x6ce4, [0x0339] = 0x6cd8, [0x033a] = 0x6cb2,
  [0x033b] = 0x6cce, [0x033c] = 0x6cc8, [0x033e] = 0x708b, [0x033f] = 0x7088,
  [0x0340] = 0x7090, [0x0341] = 0x708f, [0x0343] = 0x7087, [0x0344] = 0x7089,
  [0x0345] = 0x708d, [0x0346] = 0x7081, [0x0348] = 0x708c, [0x034b] = 0x7240,
  [0x034e] = 0x7265, [0x034f] = 0x7266, [0x0350] = 0x7268, [0x0353] = 0x72cd,
  [0x0354] = 0x72d3, [0x0355] = 0x72db, [0x0357] = 0x72cf, [0x0358] = 0x73a7,
  [0x0359] = 0x73a3, [0x035a] = 0x739e, [0x035c] = 0x73af, [0x035f] = 0x73aa,
  [0x0360] = 0x739c, [0x0362] = 0x7542, [0x0363] = 0x7544, [0x0364] = 0x753b,
  [0x0365] = 0x7541, [0x0367] = 0x759b, [0x0368] = 0x759e, [0x036a] = 0x79c4,
  [0x036b] = 0x79c3, [0x036c] = 0x79c6, [0x036f] = 0x79c7, [0x0371] = 0x79ca,
  [0x0374] = 0x7acf, [0x0375] = 0x7c76, [0x0376] = 0x7c74, [0x0377] = 0x7cff,
  [0x0378] = 0x7cfc, [0x037b] = 0x7f59, [0x037c] = 0x80a8, [0x037f] = 0x80b0,
  [0x0381] = 0x80b3, [0x0383] = 0x80a4, [0x0384] = 0x80b6, [0x0385] = 0x80a7,
  [0x0386] = 0x80ac, [0x0388] = 0x80a6, [0x0389] = 0x5367, [0x038a] = 0x820e,
  [0x038b] = 0x82c4, [0x038c] = 0x833e, [0x038d] = 0x829c, [0x0393] = 0x82aa,
  [0x0395] = 0x82c9, [0x0398] = 0x82a6, [0x0399] = 0x82b2, [0x039d] = 0x8fcc,
  [0x039e] = 0x8fd9, [0x039f] = 0x8fca, [0x03a0] = 0x8fd8, [0x03a1] = 0x8fcf,
  [0x03a2] = 0x90b7, [0x03a4] = 0x90ad, [0x03a5] = 0x90b9, [0x03a6] = 0x9637,
  [0x03a8] = 0x9641, [0x03a9] = 0x963e, [0x03aa] = 0x96b6, [0x03ab] = 0x9751,
  [0x03ac] = 0x9763, [0x03ad] = 0x4e57, [0x03ae] = 0x4e79, [0x03af] = 0x4eb2,
  [0x03b0] = 0x4eb0, [0x03b1] = 0x4eaf, [0x03b2] = 0x4eb1, [0x03b3] = 0x4fd2,
  [0x03b4] = 0x4fd5, [0x03b6] = 0x4fbe, [0x03b7] = 0x4fb8, [0x03b8] = 0x4fb0,
  [0x03b9] = 0x4fb1, [0x03ba] = 0x4fc8, [0x03bd] = 0x4fc6, [0x03be] = 0x4fcc,
  [0x03bf] = 0x4fe5, [0x03c0] = 0x4fe3, [0x03c1] = 0x4fb4, [0x03c2] = 0x516a,
  [0x03c4] = 0x519f, [0x03c6] = 0x51c1, [0x03c8] = 0x51c2, [0x03c9] = 0x51c3,
  [0x03ca] = 0x5245, [0x03cb] = 0x5248, [0x03ce] = 0x524f, [0x03d1] = 0x52c5,
  [0x03d2] = 0x52ca, [0x03d3] = 0x52c4, [0x03d4] = 0x5327, [0x03d5] = 0x5358,
  [0x03d6] = 0x537d, [0x03d8] = 0x53dd, [0x03d9] = 0x53dc, [0x03da] = 0x53da,
  [0x03db] = 0x53d9, [0x03dc] = 0x54b9, [0x03de] = 0x54d0, [0x03df] = 0x54b4,
  [0x03e0] = 0x54ca, [0x03e2] = 0x54a3, [0x03e3] = 0x54da, [0x03e4] = 0x54a4,
  [0x03e6] = 0x54b2, [0x03e7] = 0x549e, [0x03e8] = 0x549f, [0x03e9] = 0x54b5,
  [0x03ec] = 0x54cd, [0x03ee] = 0x54cc, [0x03f0] = 0x5700, [0x03f1] = 0x57ac,
  [0x03f2] = 0x5791, [0x03f3] = 0x578e, [0x03f4] = 0x578d, [0x03f5] = 0x5792,
  [0x03f6] = 0x57a1, [0x03f7] = 0x5790, [0x03f8] = 0x57a6, [0x03f9] = 0x57a8,
  [0x03fb] = 0x579c, [0x03fc] = 0x5796, [0x03fd] = 0x57a7, [0x0402] = 0x58f5,
  [0x0404] = 0x5909, [0x0405] = 0x5908, [0x0407] = 0x5952, [0x040a] = 0x59df,
  [0x040c] = 0x59eb, [0x040d] = 0x59ef, [0x040e] = 0x59f0, [0x040f] = 0x59d5,
  [0x0410] = 0x5a0d, [0x0411] = 0x5a04, [0x0412] = 0x59f9, [0x0413] = 0x5a02,
  [0x0414] = 0x59f8, [0x0415] = 0x59e2, [0x0416] = 0x59d9, [0x0417] = 0x59e7,
  [0x0418] = 0x5b6a, [0x041b] = 0x5bab, [0x041d] = 0x5c1b, [0x041e] = 0x5c2f,
  [0x0420] = 0x663c, [0x0424] = 0x5cd1, [0x0425] = 0x5cdc, [0x0426] = 0x5ce6,
  [0x0427] = 0x5ce1, [0x0428] = 0x5ccd, [0x042a] = 0x5ce2, [0x042b] = 0x5cdd,
  [0x042c] = 0x5ce5, [0x042d] = 0x5dfb, [0x042e] = 0x5dfa, [0x042f] = 0x5e1e,
  [0x0431] = 0x5ea1, [0x0434] = 0x5efc, [0x0435] = 0x5efb, [0x0436] = 0x5f2f,
  [0x0439] = 0x5f66, [0x043d] = 0x605c, [0x043f] = 0x604e, [0x0440] = 0x6051,
  [0x0443] = 0x6023, [0x0444] = 0x6031, [0x0445] = 0x607c, [0x0446] = 0x6052,
  [0x0448] = 0x6060, [0x0449] = 0x604a, [0x044a] = 0x6061, [0x044c] = 0x6218,
  [0x0454] = 0x631f, [0x0455] = 0x6317, [0x0456] = 0x62ea, [0x0457] = 0x6321,
  [0x0458] = 0x6304, [0x0459] = 0x6305, [0x045b] = 0x6531, [0x045c] = 0x6544,
  [0x045d] = 0x6540, [0x045f] = 0x6542, [0x0460] = 0x65be, [0x0462] = 0x6629,
  [0x0463] = 0x661b, [0x0465] = 0x6623, [0x0466] = 0x662c, [0x0467] = 0x661a,
  [0x0468] = 0x6630, [0x0469] = 0x663b, [0x046a] = 0x661e, [0x046b] = 0x6637,
  [0x046c] = 0x6638, [0x046e] = 0x670e, [0x0471] = 0x67e8, [0x0472] = 0x67d6,
  [0x0474] = 0x67c7, [0x0475] = 0x67bc, [0x0476] = 0x6852, [0x0477] = 0x67bf,
  [0x0478] = 0x67d5, [0x0479] = 0x67fe, [0x047a] = 0x8363, [0x047b] = 0x67fb,
  [0x047d] = 0x67b1, [0x047e] = 0x6801, [0x047f] = 0x6805, [0x0480] = 0x6800,
  [0x0481] = 0x67d7, [0x0483] = 0x6b2a, [0x0484] = 0x6b6b, [0x0489] = 0x6be1,
  [0x048c] = 0x6d23, [0x048d] = 0x6cff, [0x048e] = 0x6d14, [0x048f] = 0x6d05,
  [0x0490] = 0x6d13, [0x0491] = 0x6d06, [0x0492] = 0x6d21, [0x0494] = 0x6d15,
  [0x0495] = 0x6caf, [0x0496] = 0x6cf4, [0x0497] = 0x6d02, [0x0498] = 0x6d45,
  [0x049a] = 0x6d26, [0x049c] = 0x6d44, [0x049e] = 0x6d24, [0x049f] = 0x70a5,
  [0x04a1] = 0x70a3, [0x04a3] = 0x70a2, [0x04a4] = 0x70bb, [0x04a5] = 0x70a0,
  [0x04a6] = 0x70aa, [0x04a9] = 0x70a8, [0x04aa] = 0x70b6, [0x04ab] = 0x70b2,
  [0x04ac] = 0x70a7, [0x04af] = 0x70b9, [0x04b0] = 0x722e, [0x04b2] = 0x723c,
  [0x04b4] = 0x726d, [0x04b7] = 0x72e7, [0x04b8] = 0x72ed, [0x04ba] = 0x72ec,
  [0x04bb] = 0x72e5, [0x04bc] = 0x72e2, [0x04be] = 0x73c4, [0x04bf] = 0x73bd,
  [0x04c0] = 0x73cf, [0x04c1] = 0x73c9, [0x04c2] = 0x73c1, [0x04c3] = 0x73d0,
  [0x04c5] = 0x73ce, [0x04c6] = 0x74ed, [0x04c7] = 0x74eb, [0x04c9] = 0x74ef,
  [0x04ca] = 0x7549, [0x04cb] = 0x7550, [0x04cc] = 0x7546, [0x04cd] = 0x754a,
  [0x04cf] = 0x754d, [0x04d0] = 0x75a6, [0x04d4] = 0x75a8, [0x04d7] = 0x76c7,
  [0x04d8] = 0x76ff, [0x04da] = 0x76fd, [0x04db] = 0x77e6, [0x04dc] = 0x780a,
  [0x04de] = 0x7804, [0x04df] = 0x780b, [0x04e0] = 0x7807, [0x04e2] = 0x7815,
  [0x04e3] = 0x7808, [0x04e5] = 0x79d3, [0x04e6] = 0x79d4, [0x04e7] = 0x79d0,
  [0x04e8] = 0x79d7, [0x04e9] = 0x7a7c, [0x04ec] = 0x7a7d, [0x04ed] = 0x7a83,
  [0x04ee] = 0x7a82, [0x04f0] = 0x7ad4, [0x04f1] = 0x7ad5, [0x04f2] = 0x7ad3,
  [0x04f3] = 0x7ad0, [0x04f4] = 0x7ad2, [0x04f5] = 0x7afe, [0x04f6] = 0x7afc,
  [0x04f7] = 0x7c77, [0x04f8] = 0x7c7c, [0x04f9] = 0x7c7b, [0x0502] = 0x7f8f,
  [0x0503] = 0x80d3, [0x0505] = 0x80cb, [0x0506] = 0x80d2, [0x0508] = 0x8109,
  [0x0509] = 0x80e2, [0x050a] = 0x80df, [0x050b] = 0x80c6, [0x050d] = 0x8224,
  [0x050e] = 0x82f7, [0x050f] = 0x82d8, [0x0510] = 0x82dd, [0x0513] = 0x82f8,
  [0x0514] = 0x82fc, [0x0517] = 0x82e9, [0x0519] = 0x82ee, [0x051b] = 0x82d0,
  [0x051c] = 0x830e, [0x051d] = 0x82e2, [0x051e] = 0x830b, [0x051f] = 0x82fd,
  [0x0520] = 0x5179, [0x0521] = 0x8676, [0x0523] = 0x8678, [0x0526] = 0x8675,
  [0x0527] = 0x867d, [0x0529] = 0x8842, [0x052a] = 0x8866, [0x052c] = 0x898c,
  [0x052d] = 0x8a05, [0x052f] = 0x8a06, [0x0531] = 0x8c9f, [0x0533] = 0x8ff1,
  [0x0534] = 0x8fe7, [0x0535] = 0x8fe9, [0x0536] = 0x8fef, [0x0537] = 0x90c2,
  [0x0538] = 0x90bc, [0x053a] = 0x90c6, [0x053b] = 0x90c0, [0x053e] = 0x90cd,
  [0x053f] = 0x90c9, [0x0541] = 0x90c4, [0x0543] = 0x9581, [0x0545] = 0x9cec,
  [0x0546] = 0x5032, [0x0547] = 0x4ff9, [0x0548] = 0x501d, [0x0549] = 0x4fff,
  [0x054a] = 0x5004, [0x054b] = 0x4ff0, [0x054c] = 0x5003, [0x054e] = 0x5002,
  [0x054f] = 0x4ffc, [0x0550] = 0x4ff2, [0x0551] = 0x5024, [0x0552] = 0x5008,
  [0x0553] = 0x5036, [0x0554] = 0x502e, [0x0556] = 0x5010, [0x0557] = 0x5038,
  [0x0558] = 0x5039, [0x0559] = 0x4ffd, [0x055a] = 0x5056, [0x055b] = 0x4ffb,
  [0x055c] = 0x51a3, [0x055d] = 0x51a6, [0x055e] = 0x51a1, [0x0561] = 0x51c7,
  [0x0562] = 0x51c9, [0x0563] = 0x5260, [0x0564] = 0x5264, [0x0565] = 0x5259,
  [0x0566] = 0x5265, [0x0567] = 0x5267, [0x0568] = 0x5257, [0x0569] = 0x5263,
  [0x056b] = 0x5253, [0x056d] = 0x52cf, [0x056f] = 0x52ce, [0x0570] = 0x52d0,
  [0x0571] = 0x52d1, [0x0572] = 0x52cc, [0x0576] = 0x550d, [0x0577] = 0x54f4,
  [0x0579] = 0x5513, [0x057a] = 0x54ef, [0x057b] = 0x54f5, [0x057c] = 0x54f9,
  [0x057d] = 0x5502, [0x057e] = 0x5500, [0x0581] = 0x5518, [0x0582] = 0x54f0,
  [0x0583] = 0x54f6, [0x0586] = 0x5519, [0x0588] = 0x5705, [0x0589] = 0x57c9,
  [0x058b] = 0x57b7, [0x058c] = 0x57cd, [0x0590] = 0x57be, [0x0591] = 0x57bb,
  [0x0593] = 0x57db, [0x0594] = 0x57c8, [0x0595] = 0x57c4, [0x0596] = 0x57c5,
  [0x0597] = 0x57d1, [0x0598] = 0x57ca, [0x0599] = 0x57c0, [0x059c] = 0x5a21,
  [0x059d] = 0x5a2a, [0x059f] = 0x5a1d, [0x05a1] = 0x5a0b, [0x05a6] = 0x5a22,
  [0x05a9] = 0x5a24, [0x05ab] = 0x5a14, [0x05ac] = 0x5a31, [0x05ae] = 0x5a2f,
  [0x05af] = 0x5a1a, [0x05b0] = 0x5a12, [0x05b3] = 0x5a26, [0x05b6] = 0x5bbc,
  [0x05b7] = 0x5bbb, [0x05b8] = 0x5bb7, [0x05b9] = 0x5c05, [0x05ba] = 0x5c06,
  [0x05bb] = 0x5c52, [0x05bc] = 0x5c53, [0x05bf] = 0x5cfa, [0x05c0] = 0x5ceb,
  [0x05c2] = 0x5cf3, [0x05c3] = 0x5cf5, [0x05c4] = 0x5ce9, [0x05c5] = 0x5cef,
  [0x05c7] = 0x5e2a, [0x05c8] = 0x5e30, [0x05c9] = 0x5e2e, [0x05ca] = 0x5e2c,
  [0x05cb] = 0x5e2f, [0x05cc] = 0x5eaf, [0x05cd] = 0x5ea9, [0x05cf] = 0x5efd,
  [0x05d0] = 0x5f32, [0x05d1] = 0x5f8e, [0x05d2] = 0x5f93, [0x05d3] = 0x5f8f,
  [0x05d4] = 0x604f, [0x05d5] = 0x6099, [0x05d7] = 0x607e, [0x05d9] = 0x6074,
  [0x05da] = 0x604b, [0x05db] = 0x6073, [0x05dc] = 0x6075, [0x05df] = 0x6056,
  [0x05e0] = 0x60a9, [0x05e1] = 0x608b, [0x05e2] = 0x60a6, [0x05e4] = 0x6093,
  [0x05e5] = 0x60ae, [0x05e6] = 0x609e, [0x05e7] = 0x60a7, [0x05e8] = 0x6245,
  [0x05eb] = 0x632e, [0x05ed] = 0x6352, [0x05ee] = 0x6330, [0x05ef] = 0x635b,
  [0x05f1] = 0x6319, [0x05f2] = 0x631b, [0x05f4] = 0x6331, [0x05f5] = 0x635d,
  [0x05f6] = 0x6337, [0x05f7] = 0x6335, [0x05f8] = 0x6353, [0x05fa] = 0x635c,
  [0x05fb] = 0x633f, [0x05fc] = 0x654b, [0x05ff] = 0x658b, [0x0601] = 0x659a,
  [0x0602] = 0x6650, [0x0603] = 0x6646, [0x0604] = 0x664e, [0x0605] = 0x6640,
  [0x0607] = 0x664b, [0x0608] = 0x6648, [0x060a] = 0x6660, [0x060b] = 0x6644,
  [0x060c] = 0x664d, [0x060e] = 0x6837, [0x060f] = 0x6824, [0x0612] = 0x681b,
  [0x0613] = 0x6836, [0x0615] = 0x682c, [0x0616] = 0x6819, [0x0617] = 0x6856,
  [0x0618] = 0x6847, [0x0619] = 0x683e, [0x061a] = 0x681e, [0x061c] = 0x6815,
  [0x061d] = 0x6822, [0x061e] = 0x6827, [0x061f] = 0x6859, [0x0620] = 0x6858,
  [0x0621] = 0x6855, [0x0622] = 0x6830, [0x0623] = 0x6823, [0x0624] = 0x6b2e,
  [0x0625] = 0x6b2b, [0x0626] = 0x6b30, [0x0627] = 0x6b6c, [0x0629] = 0x6b8b,
  [0x062b] = 0x6be9, [0x062c] = 0x6bea, [0x062d] = 0x6be5, [0x062e] = 0x6d6b,
  [0x0631] = 0x6d73, [0x0632] = 0x6d57, [0x0635] = 0x6d5d, [0x0636] = 0x6d56,
  [0x0637] = 0x6d8f, [0x0638] = 0x6d5b, [0x0639] = 0x6d1c, [0x063a] = 0x6d9a,
  [0x063b] = 0x6d9b, [0x063c] = 0x6d99, [0x063e] = 0x6d81, [0x063f] = 0x6d71,
  [0x0642] = 0x6d72, [0x0643] = 0x6d5c, [0x0644] = 0x6d96, [0x0645] = 0x70c4,
  [0x0646] = 0x70db, [0x0647] = 0x70cc, [0x0648] = 0x70d0, [0x0649] = 0x70e3,
  [0x064a] = 0x70df, [0x064c] = 0x70d6, [0x064d] = 0x70ee, [0x064e] = 0x70d5,
  [0x0653] = 0x727a, [0x0655] = 0x72f5, [0x0656] = 0x7302, [0x0659] = 0x73e2,
  [0x065a] = 0x73ec, [0x065b] = 0x73d5, [0x065c] = 0x73f9, [0x065d] = 0x73df,
  [0x065e] = 0x73e6, [0x0663] = 0x73e4, [0x0664] = 0x73e1, [0x0665] = 0x74f3,
  [0x066a] = 0x7556, [0x066b] = 0x7555, [0x066c] = 0x7558, [0x066d] = 0x7557,
  [0x066e] = 0x755e, [0x066f] = 0x75c3, [0x0672] = 0x75b4, [0x0674] = 0x75b1,
  [0x0677] = 0x76cb, [0x0678] = 0x76cc, [0x0679] = 0x772a, [0x067b] = 0x7716,
  [0x067c] = 0x770f, [0x067f] = 0x773f, [0x0680] = 0x772b, [0x0681] = 0x770e,
  [0x0682] = 0x7724, [0x0684] = 0x7721, [0x0685] = 0x7718, [0x0686] = 0x77dd,
  [0x0689] = 0x7824, [0x068a] = 0x7836, [0x068c] = 0x7958, [0x068d] = 0x7959,
  [0x068f] = 0x7962, [0x0690] = 0x79da, [0x0691] = 0x79d9, [0x0693] = 0x79e1,
  [0x0694] = 0x79e5, [0x0695] = 0x79e8, [0x0696] = 0x79db, [0x0698] = 0x79e2,
  [0x0699] = 0x79f0, [0x069e] = 0x7ada, [0x069f] = 0x7add, [0x06a1] = 0x7adb,
  [0x06a2] = 0x7adc, [0x06a5] = 0x7b0d, [0x06a6] = 0x7b0b, [0x06a7] = 0x7b14,
  [0x06a8] = 0x7c8e, [0x06a9] = 0x7c86, [0x06ab] = 0x7c87, [0x06ac] = 0x7c83,
  [0x06ad] = 0x7c8b, [0x06b2] = 0x7d24, [0x06b6] = 0x7d25, [0x06b7] = 0x7f62,
  [0x06b8] = 0x7f93, [0x06b9] = 0x7f99, [0x06ba] = 0x7f97, [0x06bd] = 0x7fc4,
  [0x06be] = 0x7fc6, [0x06bf] = 0x800a, [0x06c2] = 0x8040, [0x06c3] = 0x803c,
  [0x06c4] = 0x803b, [0x06c5] = 0x80f6, [0x06c6] = 0x80ff, [0x06c7] = 0x80ee,
  [0x06c8] = 0x8104, [0x06c9] = 0x8103, [0x06ca] = 0x8107, [0x06cd] = 0x80f7,
  [0x06d0] = 0x822d, [0x06d2] = 0x8227, [0x06d3] = 0x8229, [0x06d4] = 0x831f,
  [0x06d5] = 0x8357, [0x06da] = 0x8321, [0x06dd] = 0x8318, [0x06de] = 0x8358,
  [0x06e4] = 0x8684, [0x06e5] = 0x869f, [0x06e6] = 0x869b, [0x06e7] = 0x8689,
  [0x06e8] = 0x86a6, [0x06e9] = 0x8692, [0x06ea] = 0x868f, [0x06eb] = 0x86a0,
  [0x06ec] = 0x884f, [0x06ed] = 0x8878, [0x06ee] = 0x887a, [0x06ef] = 0x886e,
  [0x06f0] = 0x887b, [0x06f1] = 0x8884, [0x06f2] = 0x8873, [0x06f5] = 0x8a0d,
  [0x06f6] = 0x8a0b, [0x06f7] = 0x8a19, [0x06fe] = 0x8ff9, [0x06ff] = 0x9009,
  [0x0700] = 0x9008, [0x0702] = 0x90de, [0x0703] = 0x9151, [0x0706] = 0x91db,
  [0x0707] = 0x91df, [0x0708] = 0x91de, [0x0709] = 0x91d6, [0x070a] = 0x91e0,
  [0x070b] = 0x9585, [0x070c] = 0x9660, [0x070d] = 0x9659, [0x070f] = 0x9656,
  [0x0712] = 0x96bd, [0x0715] = 0x5042, [0x0716] = 0x5059, [0x0718] = 0x5044,
  [0x0719] = 0x5066, [0x071a] = 0x5052, [0x071b] = 0x5054, [0x071c] = 0x5071,
  [0x071d] = 0x5050, [0x071e] = 0x507b, [0x071f] = 0x507c, [0x0720] = 0x5058,
  [0x0723] = 0x5079, [0x0724] = 0x506c, [0x0725] = 0x5078, [0x0726] = 0x51a8,
  [0x0727] = 0x51d1, [0x0728] = 0x51cf, [0x0729] = 0x5268, [0x072a] = 0x5276,
  [0x072b] = 0x52d4, [0x072d] = 0x53a0, [0x072e] = 0x53c4, [0x0730] = 0x5558,
  [0x0731] = 0x554c, [0x0732] = 0x5568, [0x0734] = 0x5549, [0x0737] = 0x555d,
  [0x0738] = 0x5529, [0x073a] = 0x5554, [0x073b] = 0x5553, [0x073d] = 0x555a,
  [0x073f] = 0x553a, [0x0740] = 0x553f, [0x0741] = 0x552b, [0x0742] = 0x57ea,
  [0x0744] = 0x57ef, [0x0747] = 0x57dd, [0x0748] = 0x57fe, [0x074a] = 0x57de,
  [0x074b] = 0x57e6, [0x074d] = 0x57e8, [0x074e] = 0x57ff, [0x074f] = 0x5803,
  [0x0750] = 0x58f7, [0x0751] = 0x68a6, [0x0752] = 0x591f, [0x0754] = 0x595b,
  [0x0755] = 0x595d, [0x0756] = 0x595e, [0x0759] = 0x5a2b, [0x075b] = 0x5a3b,
  [0x075e] = 0x5a61, [0x075f] = 0x5a3a, [0x0760] = 0x5a6e, [0x0761] = 0x5a4b,
  [0x0762] = 0x5a6b, [0x0765] = 0x5a45, [0x0766] = 0x5a4e, [0x0767] = 0x5a68,
  [0x0768] = 0x5a3d, [0x0769] = 0x5a71, [0x076a] = 0x5a3f, [0x076b] = 0x5a6f,
  [0x076c] = 0x5a75, [0x076e] = 0x5a73, [0x076f] = 0x5a2c, [0x0770] = 0x5a59,
  [0x0771] = 0x5a54, [0x0772] = 0x5a4f, [0x0773] = 0x5a63, [0x0776] = 0x5bc8,
  [0x0778] = 0x5bc3, [0x077a] = 0x5c5b, [0x077b] = 0x5c61, [0x077d] = 0x5d21,
  [0x077e] = 0x5d0a, [0x077f] = 0x5d09, [0x0781] = 0x5d2c, [0x0782] = 0x5d08,
  [0x0785] = 0x5d2a, [0x0786] = 0x5d15, [0x0788] = 0x5d10, [0x0789] = 0x5d13,
  [0x078b] = 0x5d2f, [0x078c] = 0x5d18, [0x078e] = 0x5de3, [0x078f] = 0x5e39,
  [0x0790] = 0x5e35, [0x0791] = 0x5e3a, [0x0792] = 0x5e32, [0x0797] = 0x5ebb,
  [0x0798] = 0x5eba, [0x0799] = 0x5f34, [0x079a] = 0x5f39, [0x079f] = 0x6098,
  [0x07a1] = 0x60d0, [0x07a5] = 0x60d7, [0x07a6] = 0x60aa, [0x07a8] = 0x60a1,
  [0x07a9] = 0x60a4, [0x07ab] = 0x60ee, [0x07ad] = 0x60e7, [0x07b0] = 0x60de,
  [0x07b3] = 0x637e, [0x07b4] = 0x638b, [0x07b7] = 0x6379, [0x07b8] = 0x6386,
  [0x07b9] = 0x6393, [0x07bb] = 0x6373, [0x07bc] = 0x636a, [0x07be] = 0x636c,
  [0x07c0] = 0x637f, [0x07c2] = 0x63b2, [0x07c3] = 0x63ba, [0x07c6] = 0x6366,
  [0x07c7] = 0x6374, [0x07c9] = 0x655a, [0x07cb] = 0x654e, [0x07cc] = 0x654d,
  [0x07cd] = 0x658d, [0x07ce] = 0x658e, [0x07cf] = 0x65ad, [0x07d1] = 0x65c7,
  [0x07d2] = 0x65ca, [0x07d4] = 0x65c9, [0x07d6] = 0x65e3, [0x07d7] = 0x6657,
  [0x07d9] = 0x6663, [0x07da] = 0x6667, [0x07db] = 0x671a, [0x07dc] = 0x6719,
  [0x07dd] = 0x6716, [0x07e0] = 0x689e, [0x07e1] = 0x68b6, [0x07e2] = 0x6898,
  [0x07e3] = 0x6873, [0x07e5] = 0x689a, [0x07e6] = 0x688e, [0x07e7] = 0x68b7,
  [0x07e8] = 0x68db, [0x07e9] = 0x68a5, [0x07ea] = 0x686c, [0x07eb] = 0x68c1,
  [0x07ec] = 0x6884, [0x07ef] = 0x6895, [0x07f0] = 0x687a, [0x07f1] = 0x6899,
  [0x07f3] = 0x68b8, [0x07f4] = 0x68b9, [0x07f5] = 0x6870, [0x07f7] = 0x6b35,
  [0x07f9] = 0x6b90, [0x07fa] = 0x6bbb, [0x07fb] = 0x6bed, [0x07ff] = 0x6dc1,
  [0x0800] = 0x6dc3, [0x0801] = 0x6dce, [0x0804] = 0x6dad, [0x0805] = 0x6e04,
  [0x0807] = 0x6db9, [0x0809] = 0x6de7, [0x080b] = 0x6e08, [0x080c] = 0x6e06,
  [0x080e] = 0x6e0a, [0x080f] = 0x6db0, [0x0811] = 0x6df8, [0x0812] = 0x6e0c,
  [0x0814] = 0x6db1, [0x0816] = 0x6e02, [0x0817] = 0x6e07, [0x0818] = 0x6e09,
  [0x0819] = 0x6e01, [0x081a] = 0x6e17, [0x081b] = 0x6dff, [0x081c] = 0x6e12,
  [0x081f] = 0x7103, [0x0820] = 0x7107, [0x0821] = 0x7101, [0x0822] = 0x70f5,
  [0x0823] = 0x70f1, [0x0824] = 0x7108, [0x0825] = 0x70f2, [0x0826] = 0x710f,
  [0x0828] = 0x70fe, [0x082c] = 0x731a, [0x082d] = 0x7310, [0x082e] = 0x730e,
  [0x082f] = 0x7402, [0x0830] = 0x73f3, [0x0833] = 0x73fb, [0x0837] = 0x751b,
  [0x0838] = 0x7523, [0x0839] = 0x7561, [0x083a] = 0x7568, [0x083c] = 0x7567,
  [0x083d] = 0x75d3, [0x0840] = 0x7690, [0x0843] = 0x76d5, [0x0844] = 0x76d7,
  [0x0845] = 0x76d6, [0x0846] = 0x7730, [0x0848] = 0x7726, [0x084a] = 0x7740,
  [0x084c] = 0x771e, [0x0850] = 0x7847, [0x0852] = 0x784b, [0x0853] = 0x7851,
  [0x0854] = 0x784f, [0x0855] = 0x7842, [0x0856] = 0x7846, [0x0858] = 0x796e,
  [0x0859] = 0x796c, [0x085a] = 0x79f2, [0x085c] = 0x79f1, [0x085d] = 0x79f5,
  [0x085e] = 0x79f3, [0x085f] = 0x79f9, [0x0863] = 0x7a9a, [0x0864] = 0x7a93,
  [0x0865] = 0x7a91, [0x0866] = 0x7ae1, [0x0869] = 0x7b21, [0x086a] = 0x7b1c,
  [0x086b] = 0x7b16, [0x086c] = 0x7b17, [0x086d] = 0x7b36, [0x086e] = 0x7b1f,
  [0x0870] = 0x7c93, [0x0871] = 0x7c99, [0x0872] = 0x7c9a, [0x0873] = 0x7c9c,
  [0x0875] = 0x7d49, [0x0877] = 0x7d34, [0x0878] = 0x7d37, [0x087a] = 0x7d2d,
  [0x087c] = 0x7d4c, [0x087f] = 0x7d48, [0x0882] = 0x7f3b, [0x0887] = 0x8008,
  [0x0888] = 0x801a, [0x088a] = 0x801d, [0x088c] = 0x8049, [0x088d] = 0x8045,
  [0x088e] = 0x8044, [0x088f] = 0x7c9b, [0x0892] = 0x812a, [0x0893] = 0x812e,
  [0x0896] = 0x8131, [0x0898] = 0x811a, [0x0899] = 0x8134, [0x089a] = 0x8117,
  [0x089e] = 0x831d, [0x089f] = 0x8371, [0x08a0] = 0x8384, [0x08a1] = 0x8380,
  [0x08a2] = 0x8372, [0x08a3] = 0x83a1, [0x08a5] = 0x8379, [0x08a6] = 0x8391,
  [0x08a8] = 0x839f, [0x08a9] = 0x83ad, [0x08ac] = 0x8323, [0x08ae] = 0x8385,
  [0x08af] = 0x839c, [0x08b0] = 0x83b7, [0x08b1] = 0x8658, [0x08b2] = 0x865a,
  [0x08b4] = 0x8657, [0x08b5] = 0x86b2, [0x08b7] = 0x86ae, [0x08bb] = 0x8845,
  [0x08bc] = 0x889c, [0x08bd] = 0x8894, [0x08be] = 0x88a3, [0x08bf] = 0x888f,
  [0x08c0] = 0x88a5, [0x08c1] = 0x88a9, [0x08c2] = 0x88a6, [0x08c3] = 0x888a,
  [0x08c4] = 0x88a0, [0x08c5] = 0x8890, [0x08c6] = 0x8992, [0x08c7] = 0x8991,
  [0x08c8] = 0x8994, [0x08ca] = 0x8a26, [0x08cb] = 0x8a32, [0x08cc] = 0x8a28,
  [0x08cf] = 0x8a1c, [0x08d1] = 0x8a2b, [0x08d2] = 0x8a20, [0x08d4] = 0x8a29,
  [0x08d8] = 0x8a21, [0x08d9] = 0x8c3a, [0x08db] = 0x8c5b, [0x08dc] = 0x8c58,
  [0x08dd] = 0x8c7c, [0x08df] = 0x8ca6, [0x08e0] = 0x8cae, [0x08e1] = 0x8cad,
  [0x08e2] = 0x8d65, [0x08e4] = 0x8d7e, [0x08e6] = 0x8d7c, [0x08e7] = 0x8d7f,
  [0x08e8] = 0x8d7a, [0x08e9] = 0x8dbd, [0x08ec] = 0x8dc0, [0x08ed] = 0x8dbb,
  [0x08ee] = 0x8ead, [0x08ef] = 0x8eaf, [0x08f0] = 0x8ed6, [0x08f6] = 0x8ed9,
  [0x08f9] = 0x9012, [0x08fa] = 0x900e, [0x08fb] = 0x9025, [0x08fd] = 0x9013,
  [0x08fe] = 0x90ee, [0x0900] = 0x90ab, [0x0901] = 0x90f7, [0x0903] = 0x9159,
  [0x0904] = 0x9154, [0x0905] = 0x91f2, [0x0906] = 0x91f0, [0x0907] = 0x91e5,
  [0x0908] = 0x91f6, [0x090b] = 0x9587, [0x090d] = 0x965a, [0x0910] = 0x966e,
  [0x0914] = 0x9679, [0x0916] = 0x98e1, [0x0917] = 0x98e6, [0x0919] = 0x9ec4,
  [0x091a] = 0x9ed2, [0x091b] = 0x4e80, [0x091d] = 0x4e81, [0x091e] = 0x508f,
  [0x091f] = 0x5097, [0x0920] = 0x5088, [0x0921] = 0x5089, [0x0924] = 0x5081,
  [0x0925] = 0x5160, [0x0928] = 0x5e42, [0x0929] = 0x51d3, [0x092c] = 0x51d2,
  [0x092d] = 0x51d6, [0x092e] = 0x5273, [0x0930] = 0x5270, [0x0934] = 0x53a8,
  [0x0935] = 0x53a6, [0x0936] = 0x53c5, [0x0937] = 0x5597, [0x0938] = 0x55de,
  [0x093b] = 0x5596, [0x093c] = 0x55b4, [0x093e] = 0x5585, [0x0940] = 0x559b,
  [0x0941] = 0x55a0, [0x0943] = 0x5559, [0x0945] = 0x5586, [0x0948] = 0x55af,
  [0x0949] = 0x557a, [0x094d] = 0x559e, [0x094f] = 0x55a9, [0x0950] = 0x570f,
  [0x0951] = 0x570e, [0x0952] = 0x581a, [0x0954] = 0x581f, [0x0956] = 0x583c,
  [0x0957] = 0x5818, [0x0958] = 0x583e, [0x0959] = 0x5826, [0x095b] = 0x583a,
  [0x095d] = 0x5822, [0x095f] = 0x58fb, [0x0960] = 0x5963, [0x0961] = 0x5964,
  [0x0963] = 0x5aa8, [0x0964] = 0x5aa3, [0x0965] = 0x5a82, [0x0966] = 0x5a88,
  [0x0967] = 0x5aa1, [0x0968] = 0x5a85, [0x0969] = 0x5a98, [0x096b] = 0x5a99,
  [0x096d] = 0x5a89, [0x096e] = 0x5a81, [0x096f] = 0x5a96, [0x0970] = 0x5a80,
  [0x0973] = 0x5a91, [0x0978] = 0x5acf, [0x097f] = 0x5a87, [0x0980] = 0x5aa0,
  [0x0982] = 0x5a79, [0x0984] = 0x5a86, [0x0985] = 0x5aab, [0x0986] = 0x5aaa,
  [0x0987] = 0x5aa4, [0x0988] = 0x5a8d, [0x0989] = 0x5a7e, [0x098b] = 0x5bd5,
  [0x098f] = 0x5c1e, [0x0990] = 0x5c5f, [0x0991] = 0x5c5e, [0x0992] = 0x5d44,
  [0x0993] = 0x5d3e, [0x0995] = 0x5d48, [0x0996] = 0x5d1c, [0x0998] = 0x5d5b,
  [0x0999] = 0x5d4d, [0x099c] = 0x5d57, [0x099e] = 0x5d53, [0x099f] = 0x5d4f,
  [0x09a1] = 0x5d3b, [0x09a2] = 0x5d46, [0x09a5] = 0x5e46, [0x09a6] = 0x5e47,
  [0x09a8] = 0x5e48, [0x09a9] = 0x5ec0, [0x09aa] = 0x5ebd, [0x09ab] = 0x5ebf,
  [0x09ad] = 0x5f11, [0x09af] = 0x5f3e, [0x09b0] = 0x5f3b, [0x09b2] = 0x5f3a,
  [0x09b6] = 0x5fa7, [0x09b8] = 0x60ea, [0x09ba] = 0x6107, [0x09bb] = 0x6122,
  [0x09bc] = 0x610c, [0x09bf] = 0x60b3, [0x09c0] = 0x60d6, [0x09c1] = 0x60d2,
  [0x09c3] = 0x60e3, [0x09c4] = 0x60e5, [0x09c5] = 0x60e9, [0x09c8] = 0x6111,
  [0x09c9] = 0x60fd, [0x09cc] = 0x611e, [0x09cd] = 0x6120, [0x09ce] = 0x6121,
  [0x09cf] = 0x621e, [0x09d1] = 0x63e2, [0x09d2] = 0x63de, [0x09d3] = 0x63e6,
  [0x09d8] = 0x63f8, [0x09da] = 0x63fe, [0x09db] = 0x63c1, [0x09dc] = 0x63bf,
  [0x09dd] = 0x63f7, [0x09de] = 0x63d1, [0x09df] = 0x655f, [0x09e0] = 0x6560,
  [0x09e1] = 0x6561, [0x09e4] = 0x65d1, [0x09e7] = 0x667d, [0x09e8] = 0x666b,
  [0x09e9] = 0x667f, [0x09ec] = 0x6673, [0x09ed] = 0x6681, [0x09ee] = 0x666d,
  [0x09ef] = 0x6669, [0x09f2] = 0x671e, [0x09f3] = 0x68ed, [0x09f8] = 0x6903,
  [0x09fa] = 0x68fe, [0x09fb] = 0x68e5, [0x09fc] = 0x691e, [0x09fd] = 0x6902,
  [0x0a00] = 0x6909, [0x0a01] = 0x68ca, [0x0a02] = 0x6900, [0x0a04] = 0x6901,
  [0x0a05] = 0x6918, [0x0a06] = 0x68e2, [0x0a07] = 0x68cf, [0x0a09] = 0x692e,
  [0x0a0a] = 0x68c5, [0x0a0b] = 0x68ff, [0x0a0d] = 0x691c, [0x0a0e] = 0x68c3,
  [0x0a10] = 0x6b6f, [0x0a12] = 0x6b6e, [0x0a14] = 0x6bbe, [0x0a16] = 0x6bf4,
  [0x0a17] = 0x6c2d, [0x0a19] = 0x6db6, [0x0a1a] = 0x6e75, [0x0a1b] = 0x6e1e,
  [0x0a1d] = 0x6e18, [0x0a1f] = 0x6e48, [0x0a21] = 0x6e4f, [0x0a23] = 0x6e42,
  [0x0a24] = 0x6e6a, [0x0a25] = 0x6e70, [0x0a26] = 0x6dfe, [0x0a29] = 0x6e6d,
  [0x0a2b] = 0x6e7b, [0x0a2c] = 0x6e7e, [0x0a2d] = 0x6e59, [0x0a2f] = 0x6e57,
  [0x0a31] = 0x6e80, [0x0a32] = 0x6e50, [0x0a34] = 0x6e29, [0x0a35] = 0x6e76,
  [0x0a36] = 0x6e2a, [0x0a37] = 0x6e4c, [0x0a38] = 0x712a, [0x0a3a] = 0x7135,
  [0x0a3b] = 0x712c, [0x0a3c] = 0x7137, [0x0a3d] = 0x711d, [0x0a40] = 0x7138,
  [0x0a42] = 0x7134, [0x0a43] = 0x712b, [0x0a44] = 0x7133, [0x0a45] = 0x7127,
  [0x0a46] = 0x7124, [0x0a48] = 0x712d, [0x0a49] = 0x7232, [0x0a4a] = 0x7283,
  [0x0a4b] = 0x7282, [0x0a4c] = 0x7287, [0x0a4d] = 0x7306, [0x0a4e] = 0x7324,
  [0x0a4f] = 0x7338, [0x0a50] = 0x732a, [0x0a51] = 0x732c, [0x0a52] = 0x732b,
  [0x0a54] = 0x732f, [0x0a55] = 0x7328, [0x0a56] = 0x7417, [0x0a59] = 0x7419,
  [0x0a5a] = 0x7438, [0x0a5c] = 0x741f, [0x0a5d] = 0x7414, [0x0a5e] = 0x743c,
  [0x0a5f] = 0x73f7, [0x0a60] = 0x741c, [0x0a61] = 0x7415, [0x0a62] = 0x7418,
  [0x0a63] = 0x7439, [0x0a64] = 0x74f9, [0x0a65] = 0x7524, [0x0a69] = 0x756e,
  [0x0a6a] = 0x756d, [0x0a6b] = 0x7571, [0x0a6c] = 0x758e, [0x0a6e] = 0x75e5,
  [0x0a73] = 0x7694, [0x0a74] = 0x76b3, [0x0a76] = 0x76d9, [0x0a78] = 0x7748,
  [0x0a79] = 0x7749, [0x0a7a] = 0x7743, [0x0a7d] = 0x7742, [0x0a7e] = 0x77df,
  [0x0a80] = 0x7863, [0x0a81] = 0x7876, [0x0a83] = 0x785f, [0x0a84] = 0x7866,
  [0x0a85] = 0x7966, [0x0a86] = 0x7971, [0x0a89] = 0x7976, [0x0a8a] = 0x7984,
  [0x0a8b] = 0x7975, [0x0a8c] = 0x79ff, [0x0a8d] = 0x7a07, [0x0a8f] = 0x7a0e,
  [0x0a90] = 0x7a09, [0x0a97] = 0x7ae7, [0x0a98] = 0x7ae2, [0x0a99] = 0x7b55,
  [0x0a9c] = 0x7b43, [0x0a9d] = 0x7b57, [0x0a9e] = 0x7b6c, [0x0a9f] = 0x7b42,
  [0x0aa0] = 0x7b53, [0x0aa2] = 0x7b41, [0x0aa5] = 0x7ca7, [0x0aa6] = 0x7ca0,
  [0x0aa7] = 0x7ca6, [0x0aa8] = 0x7ca4, [0x0aa9] = 0x7d74, [0x0aab] = 0x7d59,
  [0x0aad] = 0x7d60, [0x0aae] = 0x7d57, [0x0aaf] = 0x7d6c, [0x0ab0] = 0x7d7e,
  [0x0ab1] = 0x7d64, [0x0ab3] = 0x7d5a, [0x0ab4] = 0x7d5d, [0x0ab8] = 0x7d76,
  [0x0ab9] = 0x7d4d, [0x0aba] = 0x7d75, [0x0abc] = 0x7fd3, [0x0abd] = 0x7fd6,
  [0x0ac0] = 0x8060, [0x0ac1] = 0x804e, [0x0ac2] = 0x8145, [0x0ac3] = 0x813b,
  [0x0ac5] = 0x8148, [0x0ac6] = 0x8142, [0x0ac7] = 0x8149, [0x0ac8] = 0x8140,
  [0x0ac9] = 0x8114, [0x0aca] = 0x8141, [0x0acc] = 0x81ef, [0x0acd] = 0x81f6,
  [0x0ace] = 0x8203, [0x0ad0] = 0x83ed, [0x0ad2] = 0x83da, [0x0ad3] = 0x8418,
  [0x0ad4] = 0x83d2, [0x0ad5] = 0x8408, [0x0ad7] = 0x8400, [0x0adb] = 0x8417,
  [0x0adc] = 0x8346, [0x0add] = 0x8414, [0x0ade] = 0x83d3, [0x0adf] = 0x8405,
  [0x0ae0] = 0x841f, [0x0ae1] = 0x8402, [0x0ae2] = 0x8416, [0x0ae3] = 0x83cd,
  [0x0ae4] = 0x83e6, [0x0ae6] = 0x865d, [0x0ae7] = 0x86d5, [0x0ae8] = 0x86e1,
  [0x0aed] = 0x86ee, [0x0aee] = 0x8847, [0x0aef] = 0x8846, [0x0af2] = 0x88bb,
  [0x0af4] = 0x88bf, [0x0af5] = 0x88b4, [0x0af7] = 0x88b5, [0x0af9] = 0x899a,
  [0x0afa] = 0x8a43, [0x0afd] = 0x8a5a, [0x0b01] = 0x8a35, [0x0b02] = 0x8a38,
  [0x0b03] = 0x8a42, [0x0b04] = 0x8a49, [0x0b05] = 0x8a5d, [0x0b06] = 0x8a4b,
  [0x0b07] = 0x8a3d, [0x0b0c] = 0x8c60, [0x0b0d] = 0x8c5e, [0x0b0e] = 0x8c7f,
  [0x0b0f] = 0x8c7e, [0x0b10] = 0x8c83, [0x0b12] = 0x8cb1, [0x0b13] = 0x8d87,
  [0x0b16] = 0x8d88, [0x0b17] = 0x8d83, [0x0b1a] = 0x8d86, [0x0b1b] = 0x8d8b,
  [0x0b1c] = 0x8d82, [0x0b1d] = 0x8dca, [0x0b1e] = 0x8dd2, [0x0b21] = 0x8dd4,
  [0x0b22] = 0x8dc9, [0x0b23] = 0x8eb0, [0x0b27] = 0x8ef2, [0x0b28] = 0x8ee4,
  [0x0b29] = 0x8ef3, [0x0b2a] = 0x8eea, [0x0b2c] = 0x8efd, [0x0b2e] = 0x8f9d,
  [0x0b2f] = 0x902b, [0x0b30] = 0x902a, [0x0b32] = 0x9028, [0x0b33] = 0x9029,
  [0x0b34] = 0x902c, [0x0b37] = 0x903a, [0x0b38] = 0x9030, [0x0b39] = 0x9037,
  [0x0b3a] = 0x903b, [0x0b3c] = 0x910a, [0x0b40] = 0x91fe, [0x0b41] = 0x9220,
  [0x0b43] = 0x920b, [0x0b45] = 0x9218, [0x0b46] = 0x9222, [0x0b48] = 0x921b,
  [0x0b49] = 0x9208, [0x0b4b] = 0x920e, [0x0b4c] = 0x9213, [0x0b4f] = 0x9595,
  [0x0b53] = 0x968c, [0x0b54] = 0x967b, [0x0b55] = 0x967f, [0x0b56] = 0x9681,
  [0x0b58] = 0x9682, [0x0b5e] = 0x96ee, [0x0b5f] = 0x96ed, [0x0b61] = 0x96ec,
  [0x0b62] = 0x975f, [0x0b63] = 0x976f, [0x0b65] = 0x976d, [0x0b6c] = 0x98f0,
  [0x0b70] = 0x9aa9, [0x0b73] = 0x9ae0, [0x0b74] = 0x4eb7, [0x0b77] = 0x50cc,
  [0x0b78] = 0x50bc, [0x0b7a] = 0x50aa, [0x0b7b] = 0x50b9, [0x0b7d] = 0x50ab,
  [0x0b7e] = 0x50c3, [0x0b7f] = 0x50cd, [0x0b80] = 0x517e, [0x0b81] = 0x527e,
  [0x0b82] = 0x5279, [0x0b85] = 0x52e1, [0x0b86] = 0x52e0, [0x0b87] = 0x52e7,
  [0x0b88] = 0x5380, [0x0b89] = 0x53ab, [0x0b8a] = 0x53aa, [0x0b8b] = 0x53a9,
  [0x0b8c] = 0x53e0, [0x0b8d] = 0x55ea, [0x0b8f] = 0x55d7, [0x0b92] = 0x55c1,
  [0x0b93] = 0x5715, [0x0b95] = 0x586c, [0x0b97] = 0x585c, [0x0b98] = 0x5850,
  [0x0b99] = 0x5861, [0x0b9a] = 0x586a, [0x0b9b] = 0x5869, [0x0b9c] = 0x5856,
  [0x0b9d] = 0x5860, [0x0b9e] = 0x5866, [0x0b9f] = 0x585f, [0x0ba0] = 0x5923,
  [0x0ba1] = 0x5966, [0x0ba2] = 0x5968, [0x0ba5] = 0x5ace, [0x0ba7] = 0x5ac5,
  [0x0ba8] = 0x5ac3, [0x0bab] = 0x5ad0, [0x0bb2] = 0x5b74, [0x0bb3] = 0x5b76,
  [0x0bb4] = 0x5bdc, [0x0bb5] = 0x5bd7, [0x0bb6] = 0x5bda, [0x0bb7] = 0x5bdb,
  [0x0bb9] = 0x5c20, [0x0bba] = 0x5d6d, [0x0bbb] = 0x5d66, [0x0bbd] = 0x5d64,
  [0x0bbe] = 0x5d6e, [0x0bc0] = 0x5d60, [0x0bc1] = 0x5f42, [0x0bc2] = 0x5f5a,
  [0x0bc3] = 0x5f6e, [0x0bc6] = 0x6130, [0x0bc7] = 0x613a, [0x0bc8] = 0x612a,
  [0x0bc9] = 0x6143, [0x0bca] = 0x6119, [0x0bcb] = 0x6131, [0x0bcd] = 0x613d,
  [0x0bd1] = 0x6408, [0x0bd2] = 0x6432, [0x0bd3] = 0x6438, [0x0bd5] = 0x6431,
  [0x0bd7] = 0x6419, [0x0bd9] = 0x6411, [0x0bdc] = 0x6429, [0x0bdd] = 0x641d,
  [0x0be1] = 0x643c, [0x0be3] = 0x6446, [0x0be4] = 0x6447, [0x0be7] = 0x643a,
  [0x0be8] = 0x6407, [0x0bea] = 0x656b, [0x0bec] = 0x6570, [0x0bed] = 0x656d,
  [0x0bef] = 0x65e4, [0x0bf0] = 0x6693, [0x0bf5] = 0x668f, [0x0bf8] = 0x6692,
  [0x0bfa] = 0x668e, [0x0bfc] = 0x6946, [0x0c04] = 0x6931, [0x0c07] = 0x693e,
  [0x0c09] = 0x697c, [0x0c0a] = 0x6943, [0x0c0c] = 0x6973, [0x0c0e] = 0x6955,
  [0x0c11] = 0x6985, [0x0c12] = 0x694d, [0x0c13] = 0x6950, [0x0c14] = 0x6947,
  [0x0c15] = 0x6967, [0x0c16] = 0x6936, [0x0c17] = 0x6964, [0x0c18] = 0x6961,
  [0x0c1a] = 0x697d, [0x0c1b] = 0x6b44, [0x0c1c] = 0x6b40, [0x0c1d] = 0x6b71,
  [0x0c1e] = 0x6b73, [0x0c1f] = 0x6b9c, [0x0c23] = 0x6bc1, [0x0c25] = 0x6bfa,
  [0x0c26] = 0x6c31, [0x0c27] = 0x6c32, [0x0c2a] = 0x6eb8, [0x0c2b] = 0x6ea8,
  [0x0c2d] = 0x6e91, [0x0c2e] = 0x6ebb, [0x0c30] = 0x6e9a, [0x0c33] = 0x6ea9,
  [0x0c36] = 0x6eb5, [0x0c37] = 0x6e6c, [0x0c38] = 0x6ee8, [0x0c3a] = 0x6edd,
  [0x0c3b] = 0x6eda, [0x0c3c] = 0x6ee6, [0x0c3d] = 0x6eac, [0x0c41] = 0x6ed9,
  [0x0c42] = 0x6ee3, [0x0c43] = 0x6ee9, [0x0c44] = 0x6edb, [0x0c46] = 0x716f,
  [0x0c49] = 0x7148, [0x0c4b] = 0x714a, [0x0c4c] = 0x716b, [0x0c4e] = 0x714f,
  [0x0c4f] = 0x7157, [0x0c50] = 0x7174, [0x0c54] = 0x7145, [0x0c55] = 0x7151,
  [0x0c56] = 0x716d, [0x0c58] = 0x7251, [0x0c59] = 0x7250, [0x0c5a] = 0x724e,
  [0x0c5c] = 0x7341, [0x0c5e] = 0x732e, [0x0c5f] = 0x7346, [0x0c61] = 0x7427,
  [0x0c63] = 0x7448, [0x0c64] = 0x7453, [0x0c65] = 0x743d, [0x0c67] = 0x745d,
  [0x0c68] = 0x7456, [0x0c6a] = 0x741e, [0x0c6b] = 0x7447, [0x0c6c] = 0x7443,
  [0x0c6d] = 0x7458, [0x0c6e] = 0x7449, [0x0c70] = 0x744c, [0x0c71] = 0x7445,
  [0x0c72] = 0x743e, [0x0c74] = 0x7501, [0x0c75] = 0x751e, [0x0c78] = 0x757a,
  [0x0c79] = 0x75ee, [0x0c7a] = 0x7602, [0x0c7b] = 0x7697, [0x0c7c] = 0x7698,
  [0x0c80] = 0x775d, [0x0c81] = 0x7764, [0x0c82] = 0x7753, [0x0c83] = 0x7758,
  [0x0c84] = 0x7882, [0x0c85] = 0x7890, [0x0c86] = 0x788a, [0x0c88] = 0x787a,
  [0x0c89] = 0x787d, [0x0c8b] = 0x788b, [0x0c8c] = 0x7878, [0x0c8f] = 0x788d,
  [0x0c90] = 0x7888, [0x0c91] = 0x7892, [0x0c92] = 0x7881, [0x0c93] = 0x797e,
  [0x0c94] = 0x7983, [0x0c98] = 0x7980, [0x0c9c] = 0x7a0f, [0x0c9f] = 0x7a1d,
  [0x0ca1] = 0x7aa1, [0x0ca2] = 0x7aa4, [0x0ca4] = 0x7ae9, [0x0ca5] = 0x7aea,
  [0x0ca7] = 0x7b62, [0x0ca8] = 0x7b6b, [0x0caa] = 0x7b5e, [0x0cac] = 0x7b79,
  [0x0caf] = 0x7b6f, [0x0cb0] = 0x7b68, [0x0cb3] = 0x7cae, [0x0cb7] = 0x7cb0,
  [0x0cb9] = 0x7d90, [0x0cbb] = 0x7d8a, [0x0cbd] = 0x7d8b, [0x0cbe] = 0x7d99,
  [0x0cbf] = 0x7d95, [0x0cc1] = 0x7d87, [0x0cc2] = 0x7d78, [0x0cc3] = 0x7d97,
  [0x0cc4] = 0x7d89, [0x0cc5] = 0x7d98, [0x0cc9] = 0x7fa3, [0x0ccd] = 0x7fdd,
  [0x0cce] = 0x8057, [0x0cd0] = 0x8163, [0x0cd1] = 0x816a, [0x0cd2] = 0x816c,
  [0x0cd6] = 0x815d, [0x0cd7] = 0x8175, [0x0cd9] = 0x815f, [0x0cdb] = 0x817d,
  [0x0cdc] = 0x816d, [0x0cdf] = 0x8241, [0x0ce0] = 0x844f, [0x0ce1] = 0x8484,
  [0x0ce3] = 0x847f, [0x0ce5] = 0x8448, [0x0ce6] = 0x842a, [0x0ce7] = 0x847b,
  [0x0ce8] = 0x8472, [0x0ce9] = 0x8464, [0x0cea] = 0x842e, [0x0ceb] = 0x845c,
  [0x0cec] = 0x8453, [0x0cee] = 0x8441, [0x0cef] = 0x84c8, [0x0cf1] = 0x8462,
  [0x0cf2] = 0x8480, [0x0cf3] = 0x843e, [0x0cf4] = 0x8483, [0x0cf5] = 0x8471,
  [0x0cf7] = 0x844a, [0x0cf8] = 0x8455, [0x0cf9] = 0x8458, [0x0cfd] = 0x86fc,
  [0x0cfe] = 0x86fd, [0x0cff] = 0x8715, [0x0d01] = 0x8716, [0x0d02] = 0x86ff,
  [0x0d06] = 0x8858, [0x0d07] = 0x88cf, [0x0d08] = 0x88e0, [0x0d0d] = 0x89e7,
  [0x0d0e] = 0x8a6a, [0x0d0f] = 0x8a80, [0x0d11] = 0x8a6f, [0x0d12] = 0x8a65,
  [0x0d14] = 0x8a78, [0x0d15] = 0x8a7d, [0x0d16] = 0x8a88, [0x0d19] = 0x8a64,
  [0x0d1a] = 0x8a7e, [0x0d1c] = 0x8a67, [0x0d1d] = 0x8c63, [0x0d1e] = 0x8c88,
  [0x0d20] = 0x8ccd, [0x0d22] = 0x8cc9, [0x0d24] = 0x8ded, [0x0d2c] = 0x8eb1,
  [0x0d2f] = 0x8f04, [0x0d30] = 0x8f9e, [0x0d31] = 0x8fa0, [0x0d32] = 0x9043,
  [0x0d33] = 0x9046, [0x0d34] = 0x9048, [0x0d35] = 0x9045, [0x0d36] = 0x9040,
  [0x0d37] = 0x904c, [0x0d3a] = 0x910c, [0x0d3b] = 0x9113, [0x0d3c] = 0x9115,
  [0x0d3e] = 0x916b, [0x0d3f] = 0x9167, [0x0d40] = 0x925d, [0x0d41] = 0x9255,
  [0x0d42] = 0x9235, [0x0d44] = 0x9259, [0x0d45] = 0x922f, [0x0d46] = 0x923c,
  [0x0d47] = 0x928f, [0x0d48] = 0x925c, [0x0d49] = 0x926a, [0x0d4a] = 0x9262,
  [0x0d4b] = 0x925f, [0x0d4c] = 0x926b, [0x0d4d] = 0x926e, [0x0d4e] = 0x923b,
  [0x0d4f] = 0x9244, [0x0d50] = 0x9241, [0x0d51] = 0x959a, [0x0d53] = 0x9599,
  [0x0d57] = 0x968f, [0x0d59] = 0x9696, [0x0d5d] = 0x96f4, [0x0d5e] = 0x96fc,
  [0x0d60] = 0x9755, [0x0d62] = 0x9779, [0x0d66] = 0x97ee, [0x0d67] = 0x97f5,
  [0x0d69] = 0x980b, [0x0d6b] = 0x98f3, [0x0d6e] = 0x98f7, [0x0d6f] = 0x98ff,
  [0x0d70] = 0x98f5, [0x0d72] = 0x98ec, [0x0d73] = 0x98f1, [0x0d76] = 0x999a,
  [0x0d78] = 0x9ae2, [0x0d79] = 0x9b3d, [0x0d7a] = 0x9b5d, [0x0d7b] = 0x9ce8,
  [0x0d7d] = 0x9ceb, [0x0d7e] = 0x9cef, [0x0d7f] = 0x9cee, [0x0d80] = 0x9e81,
  [0x0d81] = 0x9f14, [0x0d82] = 0x50d0, [0x0d83] = 0x50d9, [0x0d84] = 0x50dc,
  [0x0d85] = 0x50d8, [0x0d87] = 0x50e1, [0x0d88] = 0x50eb, [0x0d8b] = 0x50f4,
  [0x0d8c] = 0x50e2, [0x0d8d] = 0x50de, [0x0d91] = 0x51f4, [0x0d95] = 0x52ed,
  [0x0d96] = 0x52ea, [0x0d98] = 0x5332, [0x0d9a] = 0x53ae, [0x0d9b] = 0x53b0,
  [0x0d9d] = 0x55fb, [0x0d9e] = 0x5603, [0x0d9f] = 0x560b, [0x0da1] = 0x5607,
  [0x0da3] = 0x55f8, [0x0da5] = 0x5628, [0x0da6] = 0x561e, [0x0da8] = 0x5618,
  [0x0da9] = 0x5611, [0x0daa] = 0x5651, [0x0dab] = 0x5605, [0x0dac] = 0x5717,
  [0x0dad] = 0x5892, [0x0daf] = 0x588c, [0x0db1] = 0x5878, [0x0db2] = 0x5884,
  [0x0db3] = 0x5873, [0x0db4] = 0x58ad, [0x0db5] = 0x5897, [0x0db6] = 0x5895,
  [0x0db7] = 0x5877, [0x0db8] = 0x5872, [0x0db9] = 0x5896, [0x0dba] = 0x588d,
  [0x0dbb] = 0x5910, [0x0dbd] = 0x596c, [0x0dbf] = 0x5ae7, [0x0dc1] = 0x5ae4,
  [0x0dc4] = 0x5aef, [0x0dc5] = 0x5626, [0x0dc8] = 0x5af0, [0x0dc9] = 0x5d7b,
  [0x0dcb] = 0x5d83, [0x0dce] = 0x5d8b, [0x0dcf] = 0x5d8c, [0x0dd1] = 0x5d78,
  [0x0dd2] = 0x5e52, [0x0dd5] = 0x5ed0, [0x0dd6] = 0x5ecf, [0x0dd8] = 0x5fb3,
  [0x0dd9] = 0x5fb4, [0x0ddd] = 0x617b, [0x0ddf] = 0x616f, [0x0de0] = 0x6181,
  [0x0de1] = 0x613c, [0x0de2] = 0x6142, [0x0de3] = 0x6138, [0x0de4] = 0x6133,
  [0x0de6] = 0x6160, [0x0de7] = 0x6169, [0x0de8] = 0x617d, [0x0de9] = 0x6186,
  [0x0dea] = 0x622c, [0x0deb] = 0x6228, [0x0ded] = 0x644c, [0x0def] = 0x6457,
  [0x0df0] = 0x647c, [0x0df3] = 0x6455, [0x0df4] = 0x6462, [0x0df5] = 0x6471,
  [0x0df6] = 0x646a, [0x0df7] = 0x6456, [0x0df8] = 0x643b, [0x0df9] = 0x6481,
  [0x0dfb] = 0x644f, [0x0dfc] = 0x647e, [0x0dfd] = 0x6464, [0x0e03] = 0x6571,
  [0x0e06] = 0x66a5, [0x0e07] = 0x669a, [0x0e08] = 0x669c, [0x0e0a] = 0x66a6,
  [0x0e0c] = 0x66a4, [0x0e0d] = 0x698f, [0x0e0e] = 0x69c5, [0x0e0f] = 0x69c8,
  [0x0e10] = 0x6992, [0x0e11] = 0x69b2, [0x0e15] = 0x69e3, [0x0e16] = 0x69c0,
  [0x0e17] = 0x69d6, [0x0e18] = 0x69d1, [0x0e19] = 0x699f, [0x0e1a] = 0x69a2,
  [0x0e1b] = 0x69d2, [0x0e1f] = 0x69e1, [0x0e20] = 0x69d5, [0x0e21] = 0x699d,
  [0x0e24] = 0x6998, [0x0e26] = 0x6b74, [0x0e27] = 0x6ba1, [0x0e29] = 0x6ef0,
  [0x0e2a] = 0x6ef3, [0x0e2d] = 0x6f1b, [0x0e2e] = 0x6f0c, [0x0e2f] = 0x6f1d,
  [0x0e30] = 0x6f34, [0x0e31] = 0x6f28, [0x0e32] = 0x6f17, [0x0e34] = 0x6f44,
  [0x0e35] = 0x6f42, [0x0e36] = 0x6f04, [0x0e37] = 0x6f11, [0x0e38] = 0x6efa,
  [0x0e39] = 0x6f4a, [0x0e3a] = 0x7191, [0x0e3b] = 0x718e, [0x0e3d] = 0x718b,
  [0x0e3e] = 0x718d, [0x0e3f] = 0x717f, [0x0e40] = 0x718c, [0x0e41] = 0x717e,
  [0x0e42] = 0x717c, [0x0e43] = 0x7183, [0x0e45] = 0x7188, [0x0e48] = 0x7294,
  [0x0e4a] = 0x7355, [0x0e4b] = 0x7353, [0x0e4c] = 0x734f, [0x0e4d] = 0x7354,
  [0x0e4e] = 0x746c, [0x0e4f] = 0x7465, [0x0e50] = 0x7466, [0x0e51] = 0x7461,
  [0x0e52] = 0x746b, [0x0e53] = 0x7468, [0x0e54] = 0x7476, [0x0e56] = 0x7460,
  [0x0e58] = 0x7474, [0x0e59] = 0x7506, [0x0e5a] = 0x760e, [0x0e5c] = 0x7607,
  [0x0e5f] = 0x76b9, [0x0e61] = 0x76b7, [0x0e62] = 0x76e2, [0x0e64] = 0x7774,
  [0x0e65] = 0x7777, [0x0e66] = 0x7776, [0x0e67] = 0x7775, [0x0e69] = 0x7778,
  [0x0e6a] = 0x7771, [0x0e6c] = 0x777a, [0x0e6d] = 0x715b, [0x0e6e] = 0x777b,
  [0x0e6f] = 0x78a6, [0x0e70] = 0x78ae, [0x0e71] = 0x78b8, [0x0e75] = 0x78b1,
  [0x0e76] = 0x78af, [0x0e78] = 0x7989, [0x0e79] = 0x7987, [0x0e7c] = 0x7a29,
  [0x0e7e] = 0x7a2a, [0x0e80] = 0x7a2d, [0x0e81] = 0x7a2c, [0x0e83] = 0x7a32,
  [0x0e85] = 0x7aec, [0x0e86] = 0x7af0, [0x0e87] = 0x7b81, [0x0e88] = 0x7b9e,
  [0x0e89] = 0x7b83, [0x0e8b] = 0x7b92, [0x0e8d] = 0x7ba3, [0x0e8e] = 0x7b9f,
  [0x0e8f] = 0x7b93, [0x0e91] = 0x7b86, [0x0e92] = 0x7cb8, [0x0e93] = 0x7cb7,
  [0x0e99] = 0x7dc8, [0x0e9a] = 0x7db6, [0x0e9c] = 0x7dd1, [0x0e9e] = 0x7da8,
  [0x0e9f] = 0x7dab, [0x0ea1] = 0x7db3, [0x0ea2] = 0x7dcd, [0x0ea4] = 0x7dcf,
  [0x0ea5] = 0x7da4, [0x0ea8] = 0x7f41, [0x0ea9] = 0x7f6f, [0x0eaa] = 0x7f71,
  [0x0eb1] = 0x8023, [0x0eb2] = 0x805b, [0x0eb4] = 0x8061, [0x0eb5] = 0x805f,
  [0x0eb6] = 0x8181, [0x0eb9] = 0x8184, [0x0eba] = 0x8213, [0x0ebc] = 0x824a,
  [0x0ebd] = 0x824c, [0x0ec1] = 0x84bd, [0x0ec2] = 0x8495, [0x0ec4] = 0x8492,
  [0x0ec5] = 0x84c3, [0x0ec7] = 0x8496, [0x0ec8] = 0x84a5, [0x0ec9] = 0x84b5,
  [0x0eca] = 0x84b3, [0x0ecb] = 0x84a3, [0x0ecc] = 0x84e4, [0x0ecd] = 0x84d8,
  [0x0ece] = 0x84d5, [0x0ed0] = 0x84b7, [0x0ed1] = 0x84ad, [0x0ed2] = 0x84da,
  [0x0ed3] = 0x8493, [0x0ed4] = 0x8736, [0x0ed8] = 0x873d, [0x0ed9] = 0x872b,
  [0x0eda] = 0x8747, [0x0edb] = 0x8739, [0x0edd] = 0x8745, [0x0ede] = 0x871d,
  [0x0ee0] = 0x88ff, [0x0ee1] = 0x88ea, [0x0ee3] = 0x88f5, [0x0ee5] = 0x8900,
  [0x0ee6] = 0x88ed, [0x0ee7] = 0x8903, [0x0ee8] = 0x88e9, [0x0eeb] = 0x89ea,
  [0x0eed] = 0x8a9b, [0x0eee] = 0x8a8e, [0x0eef] = 0x8aa2, [0x0ef1] = 0x8a9c,
  [0x0ef2] = 0x8a94, [0x0ef3] = 0x8a90, [0x0ef4] = 0x8aa9, [0x0ef5] = 0x8aac,
  [0x0ef7] = 0x8a9f, [0x0efa] = 0x8a9d, [0x0efc] = 0x8c67, [0x0eff] = 0x8cd0,
  [0x0f00] = 0x8cd6, [0x0f01] = 0x8cd4, [0x0f02] = 0x8d98, [0x0f03] = 0x8d9a,
  [0x0f04] = 0x8d97, [0x0f08] = 0x8e0b, [0x0f09] = 0x8e08, [0x0f0a] = 0x8e01,
  [0x0f0b] = 0x8eb4, [0x0f0c] = 0x8eb3, [0x0f0e] = 0x8fa1, [0x0f0f] = 0x8fa2,
  [0x0f11] = 0x905a, [0x0f13] = 0x9061, [0x0f14] = 0x905f, [0x0f17] = 0x9125,
  [0x0f18] = 0x917b, [0x0f19] = 0x9176, [0x0f1a] = 0x917c, [0x0f1c] = 0x9289,
  [0x0f1d] = 0x92f6, [0x0f1e] = 0x92b1, [0x0f1f] = 0x92ad, [0x0f20] = 0x9292,
  [0x0f21] = 0x9281, [0x0f22] = 0x9284, [0x0f24] = 0x92ae, [0x0f25] = 0x9290,
  [0x0f26] = 0x929e, [0x0f2a] = 0x95a2, [0x0f2b] = 0x95a7, [0x0f31] = 0x96a0,
  [0x0f32] = 0x969d, [0x0f33] = 0x969f, [0x0f34] = 0x96d0, [0x0f36] = 0x96d1,
  [0x0f39] = 0x9759, [0x0f3b] = 0x9764, [0x0f3f] = 0x9819, [0x0f41] = 0x9814,
  [0x0f42] = 0x9815, [0x0f43] = 0x981a, [0x0f48] = 0x9906, [0x0f4a] = 0x98f8,
  [0x0f4b] = 0x9901, [0x0f4d] = 0x99be, [0x0f4e] = 0x99bc, [0x0f4f] = 0x99b7,
  [0x0f50] = 0x99b6, [0x0f51] = 0x99c0, [0x0f53] = 0x99b8, [0x0f57] = 0x99c4,
  [0x0f59] = 0x99bf, [0x0f5b] = 0x9ada, [0x0f5c] = 0x9ae4, [0x0f5d] = 0x9ae9,
  [0x0f5e] = 0x9ae8, [0x0f5f] = 0x9aea, [0x0f60] = 0x9ae5, [0x0f62] = 0x9b26,
  [0x0f65] = 0x9b40, [0x0f6d] = 0x9ebd, [0x0f72] = 0x510e, [0x0f74] = 0x50f7,
  [0x0f76] = 0x50fc, [0x0f77] = 0x510d, [0x0f78] = 0x5101, [0x0f79] = 0x51da,
  [0x0f7a] = 0x51d9, [0x0f7b] = 0x51db, [0x0f7c] = 0x5286, [0x0f7d] = 0x528e,
  [0x0f7e] = 0x52ee, [0x0f7f] = 0x5333, [0x0f80] = 0x53b1, [0x0f82] = 0x5647,
  [0x0f83] = 0x562d, [0x0f84] = 0x5654, [0x0f86] = 0x564b, [0x0f87] = 0x5652,
  [0x0f88] = 0x5631, [0x0f89] = 0x5644, [0x0f8a] = 0x5656, [0x0f8b] = 0x5650,
  [0x0f8c] = 0x562b, [0x0f8e] = 0x564d, [0x0f8f] = 0x5637, [0x0f90] = 0x564f,
  [0x0f91] = 0x58a2, [0x0f92] = 0x58b7, [0x0f94] = 0x58b2, [0x0f96] = 0x58aa,
  [0x0f97] = 0x58b5, [0x0f98] = 0x58b0, [0x0f9a] = 0x58b4, [0x0f9b] = 0x58a4,
  [0x0f9c] = 0x58a7, [0x0f9e] = 0x5926, [0x0f9f] = 0x5afe, [0x0fa1] = 0x5b04,
  [0x0fa3] = 0x5afc, [0x0fa5] = 0x5b06, [0x0fa6] = 0x5b0a, [0x0fa7] = 0x5afa,
  [0x0fa8] = 0x5b0d, [0x0fa9] = 0x5b00, [0x0faa] = 0x5b0e, [0x0fae] = 0x5d91,
  [0x0fb0] = 0x5d8f, [0x0fb1] = 0x5d90, [0x0fb2] = 0x5d98, [0x0fb3] = 0x5da4,
  [0x0fb4] = 0x5d9b, [0x0fb5] = 0x5da3, [0x0fb6] = 0x5d96, [0x0fb7] = 0x5de4,
  [0x0fb8] = 0x5e5a, [0x0fbb] = 0x5e5e, [0x0fbd] = 0x5fb8, [0x0fbe] = 0x6157,
  [0x0fbf] = 0x615c, [0x0fc0] = 0x61a6, [0x0fc1] = 0x6195, [0x0fc2] = 0x6188,
  [0x0fc4] = 0x61a3, [0x0fc5] = 0x618f, [0x0fc7] = 0x6164, [0x0fc9] = 0x6159,
  [0x0fca] = 0x6178, [0x0fcc] = 0x6185, [0x0fcd] = 0x6187, [0x0fce] = 0x619e,
  [0x0fd1] = 0x6198, [0x0fd2] = 0x619c, [0x0fd5] = 0x622f, [0x0fd6] = 0x6480,
  [0x0fd7] = 0x649b, [0x0fd8] = 0x648e, [0x0fd9] = 0x648d, [0x0fda] = 0x6494,
  [0x0fdb] = 0x64c6, [0x0fdd] = 0x64a8, [0x0fde] = 0x6483, [0x0fe0] = 0x64b9,
  [0x0fe1] = 0x6486, [0x0fe2] = 0x64b4, [0x0fe3] = 0x64af, [0x0fe4] = 0x6491,
  [0x0fe6] = 0x64aa, [0x0fe7] = 0x64a1, [0x0fe8] = 0x64a7, [0x0fe9] = 0x66b6,
  [0x0fea] = 0x66b3, [0x0fec] = 0x66bc, [0x0fed] = 0x66ac, [0x0fef] = 0x66ad,
  [0x0ff0] = 0x6a0e, [0x0ff2] = 0x6a1c, [0x0ff3] = 0x6a1a, [0x0ff6] = 0x6a0b,
  [0x0ff8] = 0x69ef, [0x0ff9] = 0x6a0c, [0x0ffa] = 0x69f0, [0x0ffb] = 0x6a22,
  [0x0ffd] = 0x69d8, [0x0fff] = 0x6a12, [0x1000] = 0x69fa, [0x1002] = 0x6a2a,
  [0x1004] = 0x6a10, [0x1007] = 0x6a29, [0x1008] = 0x69f9, [0x1009] = 0x69ea,
  [0x100a] = 0x6a2c, [0x100b] = 0x6a24, [0x100d] = 0x69e9, [0x100e] = 0x6b52,
  [0x100f] = 0x6b4f, [0x1010] = 0x6b53, [0x1013] = 0x6f10, [0x1014] = 0x6f65,
  [0x1015] = 0x6f75, [0x101a] = 0x6fd0, [0x101c] = 0x6f5c, [0x101d] = 0x6f3d,
  [0x101e] = 0x6f71, [0x1020] = 0x6f91, [0x1021] = 0x6f0b, [0x1022] = 0x6f79,
  [0x1023] = 0x6f81, [0x1024] = 0x6f8f, [0x1026] = 0x6f59, [0x1027] = 0x6f74,
  [0x1029] = 0x71ae, [0x102b] = 0x71a3, [0x102c] = 0x71ad, [0x102f] = 0x71ab,
  [0x1030] = 0x71a6, [0x1031] = 0x71a2, [0x1033] = 0x52f2, [0x1034] = 0x7257,
  [0x1035] = 0x7255, [0x1036] = 0x7299, [0x1037] = 0x734b, [0x1038] = 0x747a,
  [0x103c] = 0x748c, [0x103d] = 0x7484, [0x1040] = 0x7482, [0x1041] = 0x7493,
  [0x1042] = 0x747b, [0x1044] = 0x7509, [0x104b] = 0x778a, [0x104d] = 0x7790,
  [0x104f] = 0x78c6, [0x1050] = 0x78d3, [0x1051] = 0x78c0, [0x1052] = 0x78d2,
  [0x1053] = 0x78c7, [0x1054] = 0x78c2, [0x1056] = 0x799f, [0x1057] = 0x799d,
  [0x1058] = 0x799e, [0x105a] = 0x7a41, [0x105c] = 0x7a38, [0x105d] = 0x7a3a,
  [0x105e] = 0x7a42, [0x1061] = 0x7a3e, [0x1062] = 0x7ab0, [0x1063] = 0x7bae,
  [0x1064] = 0x7bb3, [0x1067] = 0x7bbf, [0x106a] = 0x7bcd, [0x106c] = 0x7bb2,
  [0x1074] = 0x7cc4, [0x1075] = 0x7ccd, [0x1076] = 0x7cc2, [0x1077] = 0x7cc6,
  [0x1078] = 0x7cc3, [0x1079] = 0x7cc9, [0x107a] = 0x7cc7, [0x107c] = 0x7df8,
  [0x107e] = 0x7ded, [0x107f] = 0x7de2, [0x1083] = 0x7ddc, [0x1084] = 0x7e02,
  [0x1085] = 0x7e01, [0x1087] = 0x7dd6, [0x1089] = 0x7de4, [0x108a] = 0x7dfe,
  [0x108c] = 0x7e00, [0x108d] = 0x7dfc, [0x108e] = 0x7dfd, [0x1090] = 0x7df5,
  [0x1091] = 0x7dff, [0x1093] = 0x7deb, [0x1094] = 0x7de5, [0x1095] = 0x7f78,
  [0x1096] = 0x7fae, [0x1097] = 0x7fe7, [0x1099] = 0x8065, [0x109a] = 0x806a,
  [0x109b] = 0x8066, [0x109c] = 0x8068, [0x109d] = 0x806b, [0x109e] = 0x8194,
  [0x109f] = 0x81a1, [0x10a0] = 0x8192, [0x10a1] = 0x8196, [0x10a2] = 0x8193,
  [0x10a5] = 0x8501, [0x10a7] = 0x84f8, [0x10a9] = 0x84f5, [0x10ab] = 0x8504,
  [0x10b0] = 0x851b, [0x10b1] = 0x8503, [0x10b2] = 0x8533, [0x10b3] = 0x8534,
  [0x10b4] = 0x84ed, [0x10b7] = 0x8535, [0x10b9] = 0x8505, [0x10be] = 0x877d,
  [0x10c2] = 0x8771, [0x10c4] = 0x885c, [0x10c5] = 0x88e6, [0x10c6] = 0x890f,
  [0x10c7] = 0x891b, [0x10c9] = 0x89a9, [0x10ca] = 0x89a5, [0x10cb] = 0x89ee,
  [0x10cc] = 0x8ab1, [0x10ce] = 0x8acc, [0x10cf] = 0x8ace, [0x10d1] = 0x8ab7,
  [0x10d3] = 0x8ab5, [0x10d4] = 0x8ae9, [0x10d5] = 0x8ab4, [0x10d7] = 0x8ab3,
  [0x10d8] = 0x8ac1, [0x10d9] = 0x8aaf, [0x10da] = 0x8aca, [0x10db] = 0x8ad0,
  [0x10df] = 0x8c8e, [0x10e2] = 0x8ce9, [0x10e3] = 0x8cdb, [0x10e5] = 0x8ceb,
  [0x10e6] = 0x8da4, [0x10e8] = 0x8da2, [0x10e9] = 0x8d9d, [0x10ee] = 0x8e2a,
  [0x10ef] = 0x8e28, [0x10f2] = 0x8eb8, [0x10f3] = 0x8eb6, [0x10f4] = 0x8eb9,
  [0x10f5] = 0x8eb7, [0x10f6] = 0x8f22, [0x10f7] = 0x8f2b, [0x10f8] = 0x8f27,
  [0x10f9] = 0x8f19, [0x10fa] = 0x8fa4, [0x10fc] = 0x8fb3, [0x10fe] = 0x9071,
  [0x10ff] = 0x906a, [0x1102] = 0x9188, [0x1103] = 0x918c, [0x1104] = 0x92bf,
  [0x1105] = 0x92b8, [0x1106] = 0x92be, [0x1107] = 0x92dc, [0x1108] = 0x92e5,
  [0x110b] = 0x92d4, [0x110c] = 0x92d6, [0x110e] = 0x92da, [0x110f] = 0x92ed,
  [0x1110] = 0x92f3, [0x1111] = 0x92db, [0x1113] = 0x92b9, [0x1114] = 0x92e2,
  [0x1115] = 0x92eb, [0x1116] = 0x95af, [0x1118] = 0x95b2, [0x1119] = 0x95b3,
  [0x111d] = 0x96a3, [0x111e] = 0x96a5, [0x1123] = 0x970a, [0x1125] = 0x9787,
  [0x1126] = 0x9789, [0x1127] = 0x978c, [0x1128] = 0x97ef, [0x1129] = 0x982a,
  [0x112a] = 0x9822, [0x112c] = 0x981f, [0x112e] = 0x9919, [0x1130] = 0x99ca,
  [0x1131] = 0x99da, [0x1135] = 0x99de, [0x1136] = 0x99c8, [0x1137] = 0x99e0,
  [0x1139] = 0x9ab6, [0x113a] = 0x9ab5, [0x113c] = 0x9af4, [0x113e] = 0x9b6b,
  [0x113f] = 0x9b69, [0x1140] = 0x9b72, [0x1141] = 0x9b63, [0x1143] = 0x9d0d,
  [0x1145] = 0x9d01, [0x1146] = 0x9d0c, [0x1148] = 0x9cf8, [0x114b] = 0x9cfe,
  [0x114c] = 0x9d02, [0x114d] = 0x9e84, [0x114f] = 0x9eab, [0x1150] = 0x9eaa,
  [0x1151] = 0x511d, [0x1152] = 0x5116, [0x1154] = 0x512b, [0x1155] = 0x511e,
  [0x1156] = 0x511b, [0x1157] = 0x5290, [0x1158] = 0x5294, [0x1159] = 0x5314,
  [0x115c] = 0x5667, [0x115e] = 0x567b, [0x1160] = 0x565f, [0x1161] = 0x5661,
  [0x1169] = 0x58c3, [0x116a] = 0x58ca, [0x116b] = 0x58bb, [0x116c] = 0x58c0,
  [0x116d] = 0x58c4, [0x116e] = 0x5901, [0x116f] = 0x5b1f, [0x1170] = 0x5b18,
  [0x1171] = 0x5b11, [0x1172] = 0x5b15, [0x1174] = 0x5b12, [0x1175] = 0x5b1c,
  [0x1177] = 0x5b22, [0x1178] = 0x5b79, [0x1179] = 0x5da6, [0x117b] = 0x5db3,
  [0x117c] = 0x5dab, [0x117d] = 0x5eea, [0x117f] = 0x5f5b, [0x1182] = 0x61b7,
  [0x1183] = 0x61ce, [0x1184] = 0x61b9, [0x1185] = 0x61bd, [0x1186] = 0x61cf,
  [0x1187] = 0x61c0, [0x1188] = 0x6199, [0x1189] = 0x6197, [0x118b] = 0x61bb,
  [0x118c] = 0x61d0, [0x118d] = 0x61c4, [0x118e] = 0x6231, [0x1190] = 0x64d3,
  [0x1191] = 0x64c0, [0x1196] = 0x64dc, [0x1197] = 0x64d1, [0x1198] = 0x64c8,
  [0x119a] = 0x64d5, [0x119b] = 0x66c3, [0x119e] = 0x66bf, [0x119f] = 0x66c5,
  [0x11a1] = 0x66cd, [0x11a2] = 0x66c1, [0x11a3] = 0x6706, [0x11a5] = 0x6724,
  [0x11a6] = 0x6a63, [0x11a7] = 0x6a42, [0x11a8] = 0x6a52, [0x11aa] = 0x6a43,
  [0x11ab] = 0x6a33, [0x11ad] = 0x6a6c, [0x11ae] = 0x6a57, [0x11b0] = 0x6a4c,
  [0x11b1] = 0x6a6e, [0x11b7] = 0x6a37, [0x11b9] = 0x6a71, [0x11ba] = 0x6a4a,
  [0x11bb] = 0x6a36, [0x11bd] = 0x6a53, [0x11bf] = 0x6a45, [0x11c0] = 0x6a70,
  [0x11c3] = 0x6a5c, [0x11c4] = 0x6b58, [0x11c5] = 0x6b57, [0x11cc] = 0x6fbb,
  [0x11cf] = 0x6fbe, [0x11d3] = 0x6fb5, [0x11d4] = 0x6fd3, [0x11d5] = 0x6f9f,
  [0x11d7] = 0x6fb7, [0x11d8] = 0x6ff5, [0x11d9] = 0x71b7, [0x11db] = 0x71bb,
  [0x11dd] = 0x71d1, [0x11df] = 0x71ba, [0x11e1] = 0x71b6, [0x11e2] = 0x71cc,
  [0x11e5] = 0x71d3, [0x11e6] = 0x749b, [0x11e9] = 0x7496, [0x11ea] = 0x74a2,
  [0x11eb] = 0x749d, [0x11ec] = 0x750a, [0x11ed] = 0x750e, [0x11ef] = 0x7581,
  [0x11f0] = 0x762c, [0x11f1] = 0x7637, [0x11f2] = 0x7636, [0x11f3] = 0x763b,
  [0x11f5] = 0x76a1, [0x11f8] = 0x7798, [0x11fa] = 0x7796, [0x11fe] = 0x78d6,
  [0x11ff] = 0x78eb, [0x1201] = 0x78dc, [0x1203] = 0x79a5, [0x1204] = 0x79a9,
  [0x1205] = 0x9834, [0x1206] = 0x7a53, [0x1207] = 0x7a45, [0x1209] = 0x7a4f,
  [0x120b] = 0x7abd, [0x120c] = 0x7abb, [0x120d] = 0x7af1, [0x1210] = 0x7bec,
  [0x1211] = 0x7bed, [0x1214] = 0x7cd3, [0x1216] = 0x7ce1, [0x1218] = 0x7e19,
  [0x121c] = 0x7e27, [0x121d] = 0x7e26, [0x1220] = 0x806e, [0x1221] = 0x81af,
  [0x1224] = 0x81ad, [0x1226] = 0x81aa, [0x1227] = 0x8218, [0x122c] = 0x856f,
  [0x122d] = 0x854c, [0x122f] = 0x8542, [0x1231] = 0x855c, [0x1232] = 0x8570,
  [0x1233] = 0x855f, [0x1235] = 0x855a, [0x1236] = 0x854b, [0x1237] = 0x853f,
  [0x1238] = 0x878a, [0x123a] = 0x878b, [0x123b] = 0x87a1, [0x123c] = 0x878e,
  [0x123f] = 0x8799, [0x1240] = 0x885e, [0x1241] = 0x885f, [0x1242] = 0x8924,
  [0x1243] = 0x89a7, [0x1244] = 0x8aea, [0x1245] = 0x8afd, [0x1246] = 0x8af9,
  [0x1247] = 0x8ae3, [0x1248] = 0x8ae5, [0x124b] = 0x8aec, [0x1250] = 0x8cf2,
  [0x1252] = 0x8cef, [0x1254] = 0x8da6, [0x1258] = 0x8e3b, [0x1259] = 0x8e43,
  [0x125b] = 0x8e32, [0x125c] = 0x8f31, [0x125d] = 0x8f30, [0x125f] = 0x8f2d,
  [0x1260] = 0x8f3c, [0x1261] = 0x8fa7, [0x1262] = 0x8fa5, [0x1266] = 0x9137,
  [0x1267] = 0x9195, [0x1268] = 0x918e, [0x126a] = 0x9196, [0x126c] = 0x9345,
  [0x126d] = 0x930a, [0x1270] = 0x92fd, [0x1271] = 0x9317, [0x1272] = 0x931c,
  [0x1273] = 0x9307, [0x1274] = 0x9331, [0x1275] = 0x9332, [0x1276] = 0x932c,
  [0x1277] = 0x9330, [0x1278] = 0x9303, [0x1279] = 0x9305, [0x127b] = 0x95c2,
  [0x127d] = 0x95b8, [0x127f] = 0x95c1, [0x1283] = 0x96ab, [0x1284] = 0x96b7,
  [0x1287] = 0x9715, [0x1288] = 0x9714, [0x128b] = 0x970c, [0x128c] = 0x9717,
  [0x128e] = 0x9793, [0x1290] = 0x97d2, [0x1293] = 0x9836, [0x1294] = 0x9831,
  [0x1295] = 0x9833, [0x1296] = 0x983c, [0x1297] = 0x982e, [0x1298] = 0x983a,
  [0x129a] = 0x983d, [0x129c] = 0x98b5, [0x129d] = 0x9922, [0x129e] = 0x9923,
  [0x129f] = 0x9920, [0x12a0] = 0x991c, [0x12a1] = 0x991d, [0x12a3] = 0x99a0,
  [0x12a5] = 0x99ef, [0x12a6] = 0x99e8, [0x12a7] = 0x99eb, [0x12ab] = 0x99e1,
  [0x12ac] = 0x99e6, [0x12af] = 0x9af8, [0x12b0] = 0x9af5, [0x12b3] = 0x9b83,
  [0x12b4] = 0x9b94, [0x12b5] = 0x9b84, [0x12b7] = 0x9b8b, [0x12b8] = 0x9b8f,
  [0x12ba] = 0x9b8c, [0x12bc] = 0x9b89, [0x12be] = 0x9b8e, [0x12c2] = 0x9d24,
  [0x12c3] = 0x9d0f, [0x12c5] = 0x9d13, [0x12c6] = 0x9d0a, [0x12cb] = 0x9d2a,
  [0x12cc] = 0x9d1a, [0x12ce] = 0x9d27, [0x12cf] = 0x9d16, [0x12d0] = 0x9d21,
  [0x12d2] = 0x9e85, [0x12d3] = 0x9eac, [0x12d4] = 0x9ec6, [0x12d5] = 0x9ec5,
  [0x12d6] = 0x9ed7, [0x12d7] = 0x9f53, [0x12d9] = 0x5128, [0x12da] = 0x5127,
  [0x12db] = 0x51df, [0x12dd] = 0x5335, [0x12de] = 0x53b3, [0x12e0] = 0x568a,
  [0x12e1] = 0x567d, [0x12e2] = 0x5689, [0x12e4] = 0x58cd, [0x12e5] = 0x58d0,
  [0x12e7] = 0x5b2b, [0x12e8] = 0x5b33, [0x12e9] = 0x5b29, [0x12ea] = 0x5b35,
  [0x12eb] = 0x5b31, [0x12ec] = 0x5b37, [0x12ed] = 0x5c36, [0x12ee] = 0x5dbe,
  [0x12f0] = 0x5db9, [0x12f2] = 0x5dbb, [0x12f4] = 0x61e2, [0x12f5] = 0x61db,
  [0x12f6] = 0x61dd, [0x12f7] = 0x61dc, [0x12f8] = 0x61da, [0x12fa] = 0x61d9,
  [0x12fd] = 0x64df, [0x1300] = 0x64e1, [0x1302] = 0x64ee, [0x1304] = 0x65b5,
  [0x1305] = 0x66d4, [0x1306] = 0x66d5, [0x1308] = 0x66d0, [0x1309] = 0x66d1,
  [0x130a] = 0x66ce, [0x130b] = 0x66d7, [0x130e] = 0x6a7d, [0x130f] = 0x6a8a,
  [0x1311] = 0x6aa7, [0x1313] = 0x6a99, [0x1314] = 0x6a82, [0x1315] = 0x6a88,
  [0x1318] = 0x6a86, [0x131a] = 0x6a98, [0x131b] = 0x6a9d, [0x131e] = 0x6a8f,
  [0x1320] = 0x6aaa, [0x1322] = 0x6b5d, [0x1324] = 0x6c0a, [0x1326] = 0x6fd7,
  [0x1327] = 0x6fd6, [0x1328] = 0x6fe5, [0x132c] = 0x6fd9, [0x132d] = 0x6fda,
  [0x132e] = 0x6fea, [0x1330] = 0x6ff6, [0x1333] = 0x71e3, [0x1335] = 0x71e9,
  [0x1337] = 0x71eb, [0x1338] = 0x71ef, [0x1339] = 0x71f3, [0x133a] = 0x71ea,
  [0x1340] = 0x7371, [0x1342] = 0x74ae, [0x1344] = 0x74b3, [0x1346] = 0x74ac,
  [0x1349] = 0x7583, [0x134a] = 0x7645, [0x134b] = 0x764e, [0x134c] = 0x7644,
  [0x134d] = 0x76a3, [0x134e] = 0x76a5, [0x134f] = 0x77a6, [0x1350] = 0x77a4,
  [0x1352] = 0x77a9, [0x1353] = 0x77af, [0x1357] = 0x78f0, [0x1358] = 0x78f8,
  [0x1359] = 0x78f1, [0x135b] = 0x7a49, [0x135f] = 0x7ac2, [0x1360] = 0x7af2,
  [0x1361] = 0x7af3, [0x1362] = 0x7bfa, [0x1364] = 0x7bf6, [0x1365] = 0x7bfc,
  [0x1366] = 0x7c18, [0x1367] = 0x7c08, [0x1368] = 0x7c12, [0x136b] = 0x7cdb,
  [0x136c] = 0x7cda, [0x1370] = 0x7e2c, [0x1371] = 0x7e4d, [0x1374] = 0x7f46,
  [0x1375] = 0x7ff6, [0x1376] = 0x802b, [0x1377] = 0x8074, [0x1378] = 0x81b8,
  [0x1379] = 0x81c8, [0x137d] = 0x8592, [0x137e] = 0x8593, [0x1380] = 0x857f,
  [0x1381] = 0x85ab, [0x1382] = 0x8597, [0x1385] = 0x85ac, [0x1389] = 0x87ce,
  [0x138b] = 0x87cd, [0x138e] = 0x87c1, [0x138f] = 0x87b1, [0x1390] = 0x87c7,
  [0x1392] = 0x8940, [0x1394] = 0x893f, [0x1395] = 0x8939, [0x1397] = 0x8943,
  [0x139b] = 0x89ab, [0x139d] = 0x8b1f, [0x139e] = 0x8b09, [0x139f] = 0x8b0c,
  [0x13a2] = 0x8c40, [0x13a4] = 0x8c96, [0x13a6] = 0x8cf6, [0x13a7] = 0x8cf7,
  [0x13a9] = 0x8e46, [0x13aa] = 0x8e4f, [0x13ae] = 0x8f3d, [0x13af] = 0x8f41,
  [0x13b0] = 0x9366, [0x13b1] = 0x9378, [0x13b2] = 0x935d, [0x13b3] = 0x9369,
  [0x13b4] = 0x9374, [0x13b5] = 0x937d, [0x13b6] = 0x936e, [0x13b7] = 0x9372,
  [0x13b8] = 0x9373, [0x13b9] = 0x9362, [0x13ba] = 0x9348, [0x13bb] = 0x9353,
  [0x13bc] = 0x935f, [0x13bd] = 0x9368, [0x13bf] = 0x937f, [0x13c0] = 0x936b,
  [0x13c2] = 0x95c4, [0x13c4] = 0x96af, [0x13c5] = 0x96ad, [0x13c6] = 0x96b2,
  [0x13c9] = 0x971a, [0x13ca] = 0x971b, [0x13cf] = 0x979b, [0x13d0] = 0x979f,
  [0x13d9] = 0x9840, [0x13db] = 0x9847, [0x13dd] = 0x98b7, [0x13e3] = 0x99a2,
  [0x13e6] = 0x9a00, [0x13e7] = 0x99f3, [0x13ea] = 0x99f5, [0x13ed] = 0x9abd,
  [0x13ee] = 0x9b00, [0x13ef] = 0x9b02, [0x13f1] = 0x9b34, [0x13f2] = 0x9b49,
  [0x13f3] = 0x9b9f, [0x13f5] = 0x9ba3, [0x13f6] = 0x9bcd, [0x13f7] = 0x9b99,
  [0x13f8] = 0x9b9d, [0x13fb] = 0x9d39, [0x13fd] = 0x9d44, [0x1400] = 0x9d35,
  [0x1403] = 0x9eaf, [0x1405] = 0x512f, [0x1408] = 0x9f8e, [0x140a] = 0x569f,
  [0x140b] = 0x569b, [0x140c] = 0x569e, [0x140d] = 0x5696, [0x140e] = 0x5694,
  [0x140f] = 0x56a0, [0x1411] = 0x5b3b, [0x1414] = 0x5b3a, [0x1415] = 0x5dc1,
  [0x1416] = 0x5f4d, [0x1417] = 0x5f5d, [0x1418] = 0x61f3, [0x141d] = 0x64f6,
  [0x141e] = 0x64e5, [0x141f] = 0x64ea, [0x1420] = 0x64e7, [0x1421] = 0x6505,
  [0x1423] = 0x64f9, [0x1427] = 0x6aab, [0x1428] = 0x6aed, [0x1429] = 0x6ab2,
  [0x142a] = 0x6ab0, [0x142b] = 0x6ab5, [0x142c] = 0x6abe, [0x142d] = 0x6ac1,
  [0x142e] = 0x6ac8, [0x1430] = 0x6ac0, [0x1431] = 0x6abc, [0x1432] = 0x6ab1,
  [0x1433] = 0x6ac4, [0x1434] = 0x6abf, [0x1437] = 0x7008, [0x1438] = 0x7003,
  [0x1439] = 0x6ffd, [0x143a] = 0x7010, [0x143b] = 0x7002, [0x143c] = 0x7013,
  [0x143e] = 0x71fa, [0x143f] = 0x7200, [0x1440] = 0x74b9, [0x1441] = 0x74bc,
  [0x1443] = 0x765b, [0x1444] = 0x7651, [0x1445] = 0x764f, [0x1446] = 0x76eb,
  [0x1447] = 0x77b8, [0x1449] = 0x77b9, [0x144a] = 0x77c1, [0x144b] = 0x77c0,
  [0x144c] = 0x77be, [0x144d] = 0x790b, [0x144f] = 0x7907, [0x1450] = 0x790a,
  [0x1451] = 0x7908, [0x1453] = 0x790d, [0x1454] = 0x7906, [0x1455] = 0x7915,
  [0x1456] = 0x79af, [0x145a] = 0x7af5, [0x145d] = 0x7c2e, [0x145f] = 0x7c1b,
  [0x1461] = 0x7c1a, [0x1462] = 0x7c24, [0x1465] = 0x7ce6, [0x1466] = 0x7ce3,
  [0x1469] = 0x7e5d, [0x146a] = 0x7e4f, [0x146b] = 0x7e66, [0x146c] = 0x7e5b,
  [0x146d] = 0x7f47, [0x146e] = 0x7fb4, [0x1472] = 0x7ffa, [0x1473] = 0x802e,
  [0x1476] = 0x81ce, [0x1479] = 0x8219, [0x147c] = 0x85cc, [0x147d] = 0x85b2,
  [0x147f] = 0x85bb, [0x1480] = 0x85c1, [0x1484] = 0x87e9, [0x1485] = 0x87ee,
  [0x1486] = 0x87f0, [0x1487] = 0x87d6, [0x1488] = 0x880e, [0x1489] = 0x87da,
  [0x148a] = 0x8948, [0x148b] = 0x894a, [0x148c] = 0x894e, [0x148d] = 0x894d,
  [0x148e] = 0x89b1, [0x148f] = 0x89b0, [0x1490] = 0x89b3, [0x1492] = 0x8b38,
  [0x1493] = 0x8b32, [0x1495] = 0x8b2d, [0x1497] = 0x8b34, [0x1499] = 0x8b29,
  [0x149a] = 0x8c74, [0x149d] = 0x8d03, [0x14a0] = 0x8da9, [0x14a1] = 0x8e58,
  [0x14a4] = 0x8ebf, [0x14a5] = 0x8ec1, [0x14a6] = 0x8f4a, [0x14a7] = 0x8fac,
  [0x14a9] = 0x9089, [0x14aa] = 0x913d, [0x14ab] = 0x913c, [0x14ac] = 0x91a9,
  [0x14ad] = 0x93a0, [0x14af] = 0x9390, [0x14b1] = 0x9393, [0x14b2] = 0x938b,
  [0x14b3] = 0x93ad, [0x14b4] = 0x93bb, [0x14b5] = 0x93b8, [0x14b8] = 0x939c,
  [0x14b9] = 0x95d8, [0x14ba] = 0x95d7, [0x14be] = 0x975d, [0x14bf] = 0x97a9,
  [0x14c0] = 0x97da, [0x14c5] = 0x9854, [0x14c7] = 0x9855, [0x14c8] = 0x984b,
  [0x14ca] = 0x983f, [0x14cb] = 0x98b9, [0x14d0] = 0x9938, [0x14d1] = 0x9936,
  [0x14d2] = 0x9940, [0x14d4] = 0x993b, [0x14d5] = 0x9939, [0x14d6] = 0x99a4,
  [0x14d9] = 0x9a08, [0x14da] = 0x9a0c, [0x14dc] = 0x9a10, [0x14de] = 0x9b07,
  [0x14e0] = 0x9bd2, [0x14e2] = 0x9bc2, [0x14e3] = 0x9bbb, [0x14e4] = 0x9bcc,
  [0x14e5] = 0x9bcb, [0x14e8] = 0x9d4d, [0x14e9] = 0x9d63, [0x14ea] = 0x9d4e,
  [0x14ec] = 0x9d50, [0x14ed] = 0x9d55, [0x14ef] = 0x9d5e, [0x14f1] = 0x9e90,
  [0x14f2] = 0x9eb2, [0x14f3] = 0x9eb1, [0x14f5] = 0x9eca, [0x14f6] = 0x9f02,
  [0x14f7] = 0x9f27, [0x14f8] = 0x9f26, [0x14fa] = 0x56af, [0x14fb] = 0x58e0,
  [0x14fc] = 0x58dc, [0x14fe] = 0x5b39, [0x1501] = 0x5b7c, [0x1502] = 0x5bf3,
  [0x1505] = 0x5c6b, [0x1506] = 0x5dc4, [0x1507] = 0x650b, [0x1508] = 0x6508,
  [0x1509] = 0x650a, [0x150c] = 0x65dc, [0x150f] = 0x66e1, [0x1510] = 0x66df,
  [0x1511] = 0x6ace, [0x1512] = 0x6ad4, [0x1513] = 0x6ae3, [0x1514] = 0x6ad7,
  [0x1515] = 0x6ae2, [0x151a] = 0x6ad8, [0x151b] = 0x6ad5, [0x151c] = 0x6ad2,
  [0x151f] = 0x701e, [0x1520] = 0x702c, [0x1521] = 0x7025, [0x1522] = 0x6ff3,
  [0x1523] = 0x7204, [0x1524] = 0x7208, [0x1525] = 0x7215, [0x1527] = 0x74c4,
  [0x1528] = 0x74c9, [0x1529] = 0x74c7, [0x152a] = 0x74c8, [0x152b] = 0x76a9,
  [0x152c] = 0x77c6, [0x152d] = 0x77c5, [0x152e] = 0x7918, [0x152f] = 0x791a,
  [0x1530] = 0x7920, [0x1532] = 0x7a66, [0x1533] = 0x7a64, [0x1534] = 0x7a6a,
  [0x153b] = 0x7c35, [0x153c] = 0x7c34, [0x153f] = 0x7e6c, [0x1541] = 0x7e6e,
  [0x1542] = 0x7e71, [0x1544] = 0x81d4, [0x1545] = 0x81d6, [0x1546] = 0x821a,
  [0x1547] = 0x8262, [0x1548] = 0x8265, [0x1549] = 0x8276, [0x154a] = 0x85db,
  [0x154b] = 0x85d6, [0x154d] = 0x85e7, [0x1550] = 0x85f4, [0x1552] = 0x87fd,
  [0x1553] = 0x87d5, [0x1554] = 0x8807, [0x1556] = 0x880f, [0x1557] = 0x87f8,
  [0x155a] = 0x8987, [0x155c] = 0x89b5, [0x155d] = 0x89f5, [0x155f] = 0x8b3f,
  [0x1560] = 0x8b43, [0x1561] = 0x8b4c, [0x1563] = 0x8d0b, [0x1564] = 0x8e6b,
  [0x1565] = 0x8e68, [0x1566] = 0x8e70, [0x1567] = 0x8e75, [0x1568] = 0x8e77,
  [0x156a] = 0x8ec3, [0x156c] = 0x93e9, [0x156d] = 0x93ea, [0x156e] = 0x93cb,
  [0x156f] = 0x93c5, [0x1570] = 0x93c6, [0x1572] = 0x93ed, [0x1573] = 0x93d3,
  [0x1575] = 0x93e5, [0x1578] = 0x93db, [0x1579] = 0x93eb, [0x157a] = 0x93e0,
  [0x157b] = 0x93c1, [0x157e] = 0x95dd, [0x1588] = 0x97b2, [0x1589] = 0x97b4,
  [0x158a] = 0x97b1, [0x158b] = 0x97b5, [0x158c] = 0x97f2, [0x1590] = 0x9856,
  [0x1594] = 0x9944, [0x1596] = 0x9a26, [0x1597] = 0x9a1f, [0x1598] = 0x9a18,
  [0x1599] = 0x9a21, [0x159a] = 0x9a17, [0x159c] = 0x9b09, [0x159f] = 0x9bc5,
  [0x15a0] = 0x9bdf, [0x15a2] = 0x9be3, [0x15a4] = 0x9be9, [0x15a5] = 0x9bee,
  [0x15a8] = 0x9d66, [0x15a9] = 0x9d7a, [0x15ab] = 0x9d6e, [0x15ac] = 0x9d91,
  [0x15ad] = 0x9d83, [0x15ae] = 0x9d76, [0x15af] = 0x9d7e, [0x15b0] = 0x9d6d,
  [0x15b2] = 0x9e95, [0x15b3] = 0x9ee3, [0x15b6] = 0x9f03, [0x15b7] = 0x9f04,
  [0x15b9] = 0x9f17, [0x15bb] = 0x5136, [0x15bd] = 0x5336, [0x15bf] = 0x5b42,
  [0x15c2] = 0x5b44, [0x15c3] = 0x5b46, [0x15c4] = 0x5b7e, [0x15c5] = 0x5dca,
  [0x15c6] = 0x5dc8, [0x15c7] = 0x5dcc, [0x15c8] = 0x5ef0, [0x15ca] = 0x6585,
  [0x15cb] = 0x66e5, [0x15cc] = 0x66e7, [0x15d0] = 0x6af4, [0x15d2] = 0x6ae9,
  [0x15d8] = 0x703d, [0x15da] = 0x7036, [0x15dc] = 0x7216, [0x15de] = 0x7212,
  [0x15df] = 0x720f, [0x15e0] = 0x7217, [0x15e1] = 0x7211, [0x15e2] = 0x720b,
  [0x15e5] = 0x74cd, [0x15e6] = 0x74d0, [0x15e7] = 0x74cc, [0x15e8] = 0x74ce,
  [0x15e9] = 0x74d1, [0x15eb] = 0x7589, [0x15ed] = 0x7a6f, [0x15ee] = 0x7c4b,
  [0x15ef] = 0x7c44, [0x15f5] = 0x7e7f, [0x15f6] = 0x8b71, [0x15f8] = 0x802f,
  [0x15f9] = 0x807a, [0x15fa] = 0x807b, [0x15fb] = 0x807c, [0x15ff] = 0x85fc,
  [0x1600] = 0x8610, [0x1601] = 0x8602, [0x1604] = 0x85ee, [0x1605] = 0x8603,
  [0x1607] = 0x860d, [0x1608] = 0x8613, [0x1609] = 0x8608, [0x160a] = 0x860f,
  [0x160b] = 0x8818, [0x160c] = 0x8812, [0x160f] = 0x8967, [0x1610] = 0x8965,
  [0x1611] = 0x89bb, [0x1612] = 0x8b69, [0x1613] = 0x8b62, [0x1615] = 0x8b6e,
  [0x1617] = 0x8b61, [0x1619] = 0x8b64, [0x161a] = 0x8b4d, [0x161b] = 0x8c51,
  [0x161e] = 0x8e83, [0x161f] = 0x8ec6, [0x1621] = 0x941f, [0x1623] = 0x9404,
  [0x1624] = 0x9417, [0x1625] = 0x9408, [0x1626] = 0x9405, [0x1628] = 0x93f3,
  [0x1629] = 0x941e, [0x162a] = 0x9402, [0x162b] = 0x941a, [0x162c] = 0x941b,
  [0x162d] = 0x9427, [0x162e] = 0x941c, [0x1630] = 0x96b5, [0x1633] = 0x9733,
  [0x1635] = 0x9734, [0x1636] = 0x9731, [0x1637] = 0x97b8, [0x1638] = 0x97ba,
  [0x163a] = 0x97fc, [0x163d] = 0x98c3, [0x163f] = 0x994d, [0x1641] = 0x9a2f,
  [0x1645] = 0x9ac9, [0x1647] = 0x9ac8, [0x1648] = 0x9ac4, [0x1649] = 0x9b2a,
  [0x164a] = 0x9b38, [0x164b] = 0x9b50, [0x164d] = 0x9c0a, [0x164e] = 0x9bfb,
  [0x164f] = 0x9c04, [0x1650] = 0x9bfc, [0x1651] = 0x9bfe, [0x1655] = 0x9c02,
  [0x1656] = 0x9bf6, [0x1657] = 0x9c1b, [0x1658] = 0x9bf9, [0x1659] = 0x9c15,
  [0x165a] = 0x9c10, [0x165b] = 0x9bff, [0x165c] = 0x9c00, [0x165d] = 0x9c0c,
  [0x1660] = 0x9d95, [0x1661] = 0x9da5, [0x1666] = 0x9e98, [0x1667] = 0x9ec1,
  [0x1669] = 0x9f5a, [0x166a] = 0x5164, [0x166b] = 0x56bb, [0x166d] = 0x58e6,
  [0x166e] = 0x5b49, [0x166f] = 0x5bf7, [0x1672] = 0x5dd0, [0x1674] = 0x5fc2,
  [0x1676] = 0x6511, [0x1678] = 0x6aff, [0x1679] = 0x6afe, [0x167a] = 0x6afd,
  [0x167c] = 0x6b01, [0x167f] = 0x704b, [0x1680] = 0x704d, [0x1681] = 0x7047,
  [0x1682] = 0x74d3, [0x1683] = 0x7668, [0x1684] = 0x7667, [0x1687] = 0x77d1,
  [0x1688] = 0x7930, [0x1689] = 0x7932, [0x168a] = 0x792e, [0x168c] = 0x9f9d,
  [0x168d] = 0x7ac9, [0x168e] = 0x7ac8, [0x1690] = 0x7c56, [0x1691] = 0x7c51,
  [0x1695] = 0x7e85, [0x1696] = 0x7e89, [0x1697] = 0x7e8e, [0x1698] = 0x7e84,
  [0x169a] = 0x826a, [0x169b] = 0x862b, [0x169c] = 0x862f, [0x169d] = 0x8628,
  [0x169f] = 0x8616, [0x16a0] = 0x8615, [0x16a1] = 0x861d, [0x16a2] = 0x881a,
  [0x16a6] = 0x89bc, [0x16a7] = 0x8b75, [0x16a8] = 0x8b7c, [0x16aa] = 0x8d11,
  [0x16ab] = 0x8d12, [0x16ac] = 0x8f5c, [0x16ad] = 0x91bb, [0x16af] = 0x93f4,
  [0x16b2] = 0x942d, [0x16b5] = 0x96e4, [0x16b6] = 0x9737, [0x16b7] = 0x9736,
  [0x16b8] = 0x9767, [0x16b9] = 0x97be, [0x16ba] = 0x97bd, [0x16bb] = 0x97e2,
  [0x16bc] = 0x9868, [0x16bd] = 0x9866, [0x16be] = 0x98c8, [0x16bf] = 0x98ca,
  [0x16c0] = 0x98c7, [0x16c1] = 0x98dc, [0x16c3] = 0x994f, [0x16c4] = 0x99a9,
  [0x16c5] = 0x9a3c, [0x16c7] = 0x9a3b, [0x16c8] = 0x9ace, [0x16ca] = 0x9b14,
  [0x16cb] = 0x9b53, [0x16cd] = 0x9c2e, [0x16cf] = 0x9c1f, [0x16d4] = 0x9db0,
  [0x16d5] = 0x9dbd, [0x16d8] = 0x9dae, [0x16d9] = 0x9dc4, [0x16da] = 0x9e7b,
  [0x16dd] = 0x9e9e, [0x16df] = 0x9f05, [0x16e1] = 0x9f69, [0x16e2] = 0x9fa1,
  [0x16e3] = 0x56c7, [0x16e4] = 0x571d, [0x16e5] = 0x5b4a, [0x16e6] = 0x5dd3,
  [0x16e8] = 0x5f72, [0x16e9] = 0x6202, [0x16eb] = 0x6235, [0x16ec] = 0x6527,
  [0x16ed] = 0x651e, [0x16ee] = 0x651f, [0x16f1] = 0x6b07, [0x16f2] = 0x6b06,
  [0x16f5] = 0x7054, [0x16f6] = 0x721c, [0x16f7] = 0x7220, [0x16f8] = 0x7af8,
  [0x16fa] = 0x7c5d, [0x16fb] = 0x7c58, [0x16fd] = 0x7e92, [0x16fe] = 0x7f4e,
  [0x1702] = 0x8827, [0x1704] = 0x8b81, [0x1705] = 0x8b83, [0x1707] = 0x8c44,
  [0x170c] = 0x9442, [0x170d] = 0x944d, [0x170e] = 0x9454, [0x170f] = 0x944e,
  [0x1711] = 0x9443, [0x1714] = 0x973c, [0x1715] = 0x9740, [0x1716] = 0x97c0,
  [0x171b] = 0x995a, [0x171c] = 0x9a51, [0x171e] = 0x9add, [0x1721] = 0x9c38,
  [0x1723] = 0x9c45, [0x1724] = 0x9c3a, [0x1726] = 0x9c35, [0x172a] = 0x9ef1,
  [0x172c] = 0x9f93, [0x172d] = 0x529a, [0x1730] = 0x8641, [0x1731] = 0x5dd7,
  [0x1733] = 0x6528, [0x1737] = 0x7053, [0x1738] = 0x7059, [0x173a] = 0x7221,
  [0x173c] = 0x766f, [0x173d] = 0x7937, [0x173e] = 0x79b5, [0x173f] = 0x7c62,
  [0x1740] = 0x7c5e, [0x1741] = 0x7cf5, [0x1744] = 0x863d, [0x1746] = 0x882d,
  [0x1747] = 0x8989, [0x1748] = 0x8b8d, [0x1749] = 0x8b87, [0x174a] = 0x8b90,
  [0x174b] = 0x8d1a, [0x174c] = 0x8e99, [0x1750] = 0x945f, [0x1753] = 0x9456,
  [0x1754] = 0x9461, [0x1755] = 0x945b, [0x1756] = 0x945a, [0x1757] = 0x945c,
  [0x1758] = 0x9465, [0x175a] = 0x9741, [0x175d] = 0x986e, [0x175e] = 0x986c,
  [0x175f] = 0x986d, [0x1761] = 0x99aa, [0x1762] = 0x9a5c, [0x1763] = 0x9a58,
  [0x1764] = 0x9ade, [0x1766] = 0x9c4f, [0x1767] = 0x9c51, [0x1769] = 0x9c53,
  [0x176d] = 0x9dfc, [0x176e] = 0x9f39, [0x1770] = 0x513e, [0x1772] = 0x56d2,
  [0x1774] = 0x5b4f, [0x1775] = 0x6b14, [0x1777] = 0x7a72, [0x1778] = 0x7a73,
  [0x177c] = 0x8b91, [0x177f] = 0x91bf, [0x1781] = 0x946c, [0x1784] = 0x96e6,
  [0x1785] = 0x9745, [0x1787] = 0x97c8, [0x1788] = 0x97e4, [0x1789] = 0x995d,
  [0x178b] = 0x9b21, [0x178d] = 0x9b2c, [0x178e] = 0x9b57, [0x1791] = 0x9c5d,
  [0x1792] = 0x9c61, [0x1793] = 0x9c65, [0x1794] = 0x9e08, [0x179a] = 0x9f45,
  [0x179d] = 0x6205, [0x179e] = 0x66ef, [0x179f] = 0x6b1b, [0x17a0] = 0x6b1d,
  [0x17a1] = 0x7225, [0x17a2] = 0x7224, [0x17a3] = 0x7c6d, [0x17a5] = 0x8642,
  [0x17a6] = 0x8649, [0x17a8] = 0x8978, [0x17a9] = 0x898a, [0x17aa] = 0x8b97,
  [0x17ac] = 0x8c9b, [0x17ad] = 0x8d1c, [0x17af] = 0x8ea2, [0x17b8] = 0x9c6c,
  [0x17ba] = 0x9c6f, [0x17bc] = 0x9e0e, [0x17be] = 0x9f08, [0x17bf] = 0x9f1d,
  [0x17c0] = 0x9fa3, [0x17c3] = 0x5f60, [0x17c4] = 0x6b1c, [0x17c8] = 0x7cf3,
  [0x17ca] = 0x8b9b, [0x17cb] = 0x8ea7, [0x17cc] = 0x91c4, [0x17ce] = 0x947a,
  [0x17d1] = 0x9a61, [0x17d2] = 0x9a63, [0x17d3] = 0x9ad7, [0x17d4] = 0x9c76,
  [0x17d6] = 0x9fa5, [0x17d8] = 0x7067, [0x17da] = 0x72ab, [0x17db] = 0x864a,
  [0x17dc] = 0x897d, [0x17dd] = 0x8b9d, [0x17de] = 0x8c53, [0x17df] = 0x8f65,
  [0x17e0] = 0x947b, [0x17e2] = 0x98cd, [0x17e3] = 0x98dd, [0x17e5] = 0x9b30,
  [0x17e6] = 0x9e16, [0x17ec] = 0x96e7, [0x17ed] = 0x9e18, [0x17ee] = 0x9ea2,
  [0x17f0] = 0x9f7c, [0x17f2] = 0x7e9e, [0x17f3] = 0x9484, [0x17f5] = 0x9e1c,
  [0x17f7] = 0x7c71, [0x17f8] = 0x97ca, [0x17fc] = 0x9ea3, [0x17fe] = 0x9c7b,
  [0x17ff] = 0x9f97, [0x1802] = 0x9750, [0x1806] = 0x5727, [0x1807] = 0x5c13,
  [0x180e] = 0x5fc8, [0x1814] = 0x6765, [0x1817] = 0x52bd, [0x1819] = 0x5b66,
  [0x181b] = 0x65f9, [0x181c] = 0x6788, [0x181d] = 0x6ce6, [0x181e] = 0x6ccb,
  [0x1820] = 0x4fbd, [0x1821] = 0x5f8d, [0x1823] = 0x6018, [0x1824] = 0x6048,
  [0x1826] = 0x6b29, [0x1827] = 0x70a6, [0x1829] = 0x7706, [0x182d] = 0x5a10,
  [0x182e] = 0x5cfc, [0x182f] = 0x5cfe, [0x1836] = 0x70c9, [0x183e] = 0x9579,
  [0x1840] = 0x96ba, [0x1848] = 0x7b29, [0x1849] = 0x8128, [0x184b] = 0x8a2e,
  [0x184f] = 0x9ad9, [0x1851] = 0x582b, [0x1852] = 0x5845, [0x1854] = 0x63fa,
  [0x1858] = 0x6e86, [0x185e] = 0x5867, [0x1860] = 0x5bdd, [0x1861] = 0x656e,
  [0x1865] = 0x8c87, [0x1867] = 0x50d2, [0x1868] = 0x50df, [0x186d] = 0x69ba,
  [0x186f] = 0x6b9d, [0x1871] = 0x8059, [0x187e] = 0x6f8a, [0x1881] = 0x7bc3,
  [0x1882] = 0x7bc2, [0x1887] = 0x90f6, [0x1889] = 0x9823, [0x188f] = 0x71cd,
  [0x1890] = 0x7499, [0x1896] = 0x9842, [0x189b] = 0x7f84, [0x18a1] = 0x8d0e,
  [0x18a3] = 0x9861, [0x18a6] = 0x8b73, [0x18a8] = 0x9c27, [0x18aa] = 0x9458,
  [0x18ab] = 0x77d6, [0x18ac] = 0x9b2d, [0x18c1] = 0x4f66, [0x18c2] = 0x4f68,
  [0x18c3] = 0x4fe7, [0x18c4] = 0x503f, [0x18c6] = 0x50a6, [0x18c7] = 0x510f,
  [0x18c8] = 0x523e, [0x18c9] = 0x5324, [0x18ca] = 0x5365, [0x18cb] = 0x539b,
  [0x18cc] = 0x517f, [0x18cd] = 0x54cb, [0x18ce] = 0x5573, [0x18cf] = 0x5571,
  [0x18d0] = 0x556b, [0x18d1] = 0x55f4, [0x18d2] = 0x5622, [0x18d3] = 0x5620,
  [0x18d4] = 0x5692, [0x18d5] = 0x56ba, [0x18d6] = 0x5691, [0x18d7] = 0x56b0,
  [0x18d8] = 0x5759, [0x18d9] = 0x578a, [0x18da] = 0x580f, [0x18db] = 0x5812,
  [0x18dc] = 0x5813, [0x18dd] = 0x5847, [0x18de] = 0x589b, [0x18df] = 0x5900,
  [0x18e0] = 0x594d, [0x18e1] = 0x5ad1, [0x18e2] = 0x5ad3, [0x18e3] = 0x5b67,
  [0x18e4] = 0x5c57, [0x18e5] = 0x5c77, [0x18e6] = 0x5cd5, [0x18e7] = 0x5d75,
  [0x18e8] = 0x5d8e, [0x18e9] = 0x5da5, [0x18ea] = 0x5db6, [0x18eb] = 0x5dbf,
  [0x18ec] = 0x5e65, [0x18ed] = 0x5ecd, [0x18ee] = 0x5eed, [0x18ef] = 0x5f94,
  [0x18f0] = 0x5f9a, [0x18f1] = 0x5fba, [0x18f2] = 0x6125, [0x18f3] = 0x6150,
  [0x18f4] = 0x62a3, [0x18f5] = 0x6360, [0x18f6] = 0x6364, [0x18f7] = 0x63b6,
  [0x18f8] = 0x6403, [0x18f9] = 0x64b6, [0x18fa] = 0x651a, [0x18fb] = 0x7a25,
  [0x18fc] = 0x5c21, [0x18fd] = 0x66e2, [0x18fe] = 0x6702, [0x18ff] = 0x67a4,
  [0x1900] = 0x67ac, [0x1901] = 0x6810, [0x1902] = 0x6806, [0x1903] = 0x685e,
  [0x1904] = 0x685a, [0x1905] = 0x692c, [0x1906] = 0x6929, [0x1907] = 0x6a2d,
  [0x1908] = 0x6a77, [0x1909] = 0x6a7a, [0x190a] = 0x6aca, [0x190b] = 0x6ae6,
  [0x190c] = 0x6af5, [0x190d] = 0x6b0d, [0x190e] = 0x6b0e, [0x190f] = 0x6bdc,
  [0x1910] = 0x6bdd, [0x1911] = 0x6bf6, [0x1912] = 0x6c1e, [0x1913] = 0x6c63,
  [0x1914] = 0x6da5, [0x1915] = 0x6e0f, [0x1916] = 0x6e8a, [0x1917] = 0x6e84,
  [0x1918] = 0x6e8b, [0x1919] = 0x6e7c, [0x191a] = 0x6f4c, [0x191b] = 0x6f48,
  [0x191c] = 0x6f49, [0x191d] = 0x6f9d, [0x191e] = 0x6f99, [0x191f] = 0x6ff8,
  [0x1920] = 0x702e, [0x1921] = 0x702d, [0x1922] = 0x705c, [0x1923] = 0x79cc,
  [0x1924] = 0x70bf, [0x1925] = 0x70ea, [0x1926] = 0x70e5, [0x1927] = 0x7111,
  [0x1928] = 0x7112, [0x1929] = 0x713f, [0x192a] = 0x7139, [0x192b] = 0x713b,
  [0x192c] = 0x713d, [0x192d] = 0x7177, [0x192e] = 0x7175, [0x192f] = 0x7176,
  [0x1930] = 0x7171, [0x1931] = 0x7196, [0x1932] = 0x7193, [0x1933] = 0x71b4,
  [0x1934] = 0x71dd, [0x1935] = 0x71de, [0x1936] = 0x720e, [0x1937] = 0x5911,
  [0x1938] = 0x7218, [0x1939] = 0x7347, [0x193a] = 0x7348, [0x193b] = 0x73ef,
  [0x193c] = 0x7412, [0x193d] = 0x743b, [0x193e] = 0x74a4, [0x193f] = 0x748d,
  [0x1940] = 0x74b4, [0x1941] = 0x7673, [0x1942] = 0x7677, [0x1943] = 0x76bc,
  [0x1944] = 0x7819, [0x1945] = 0x781b, [0x1946] = 0x783d, [0x1947] = 0x7853,
  [0x1948] = 0x7854, [0x1949] = 0x7858, [0x194a] = 0x78b7, [0x194b] = 0x78d8,
  [0x194c] = 0x78ee, [0x194d] = 0x7922, [0x194e] = 0x794d, [0x194f] = 0x7986,
  [0x1950] = 0x7999, [0x1951] = 0x79a3, [0x1952] = 0x79bc, [0x1953] = 0x7aa7,
  [0x1954] = 0x7b37, [0x1955] = 0x7b59, [0x1956] = 0x7bd0, [0x1957] = 0x7c2f,
  [0x1958] = 0x7c32, [0x1959] = 0x7c42, [0x195a] = 0x7c4e, [0x195b] = 0x7c68,
  [0x195c] = 0x7ca9, [0x195d] = 0x7ced, [0x195e] = 0x7dd0, [0x195f] = 0x7e07,
  [0x1960] = 0x7dd3, [0x1961] = 0x7e64, [0x1962] = 0x7f40, [0x1964] = 0x8041,
  [0x1965] = 0x8063, [0x1966] = 0x80bb, [0x1967] = 0x6711, [0x1968] = 0x6725,
  [0x1969] = 0x8248, [0x196a] = 0x8310, [0x196b] = 0x8362, [0x196c] = 0x8312,
  [0x196d] = 0x8421, [0x196e] = 0x841e, [0x196f] = 0x84e2, [0x1970] = 0x84de,
  [0x1971] = 0x84e1, [0x1972] = 0x8573, [0x1973] = 0x85d4, [0x1974] = 0x85f5,
  [0x1975] = 0x8637, [0x1976] = 0x8645, [0x1977] = 0x8672, [0x1978] = 0x874a,
  [0x1979] = 0x87a9, [0x197a] = 0x87a5, [0x197b] = 0x87f5, [0x197c] = 0x8834,
  [0x197d] = 0x8850, [0x197e] = 0x8887, [0x197f] = 0x8954, [0x1980] = 0x8984,
  [0x1981] = 0x8b03, [0x1982] = 0x8c52, [0x1983] = 0x8cd8, [0x1984] = 0x8d0c,
  [0x1985] = 0x8d18, [0x1986] = 0x8db0, [0x1987] = 0x8ebc, [0x1988] = 0x8ed5,
  [0x1989] = 0x8faa, [0x198a] = 0x909c, [0x198c] = 0x915c, [0x198d] = 0x922b,
  [0x198e] = 0x9221, [0x198f] = 0x9273, [0x1990] = 0x92f4, [0x1991] = 0x92f5,
  [0x1992] = 0x933f, [0x1993] = 0x9342, [0x1994] = 0x9386, [0x1995] = 0x93be,
  [0x1996] = 0x93bc, [0x1997] = 0x93bd, [0x1998] = 0x93f1, [0x1999] = 0x93f2,
  [0x199a] = 0x93ef, [0x199b] = 0x9422, [0x199c] = 0x9423, [0x199d] = 0x9424,
  [0x199e] = 0x9467, [0x199f] = 0x9466, [0x19a0] = 0x9597, [0x19a1] = 0x95ce,
  [0x19a2] = 0x95e7, [0x19a3] = 0x973b, [0x19a4] = 0x974d, [0x19a5] = 0x98e4,
  [0x19a6] = 0x9942, [0x19a7] = 0x9b1d, [0x19a8] = 0x9b98, [0x19aa] = 0x9d49,
  [0x19ab] = 0x6449, [0x19ac] = 0x5e71, [0x19ad] = 0x5e85, [0x19ae] = 0x61d3,
  [0x19af] = 0x990e, [0x19b0] = 0x8002, [0x19b1] = 0x781e, [0x19b4] = 0x5528,
  [0x19b5] = 0x5572, [0x19b6] = 0x55ba, [0x19b7] = 0x55f0, [0x19b8] = 0x55ee,
  [0x19b9] = 0x56b8, [0x19ba] = 0x56b9, [0x19bb] = 0x56c4, [0x19bc] = 0x8053,
  [0x19bd] = 0x92b0,
};

/* For the non-CJK ideograph glyphs (all in plane 1) we can use the tables
   in cns11643l1.

   CJK Ideographs.  The table can be created using

   egrep '^0x' .../mappings/eastasia/other/cns11643.txt |
   awk '{ print $2, $1 }' | sort | egrep '^0x[4-9]' | perl tab.pl

   where tab.pl is

   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   $n=0;
   while (<>) {
     local($ucs4, $cns, %rest) = split;
     local($u)=hex($ucs4);
     local($c)=hex($cns);
     printf ("\n ") if ($n % 2 == 0);
     ++$n;
     printf (" [0x%04x] = \"\\x%02x\\x%02x\\x%02x\",", $u - 0x4e00,
             $c < 0x10000 ? 0 : int($c/0x10000),
             $c < 0x100 ? $c : int(($c&0xff00)/256), $c < 0x100 ? 0 : $c&255);
   }
   printf ("\n");
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
const char __cns11643_from_ucs4_tab[][3] =
{
  [0x0000] = "\x01\x44\x21", [0x0001] = "\x01\x44\x23",
  [0x0003] = "\x01\x44\x24", [0x0004] = "\x0e\x21\x26",
  [0x0005] = "\x0e\x21\x25", [0x0007] = "\x02\x21\x26",
  [0x0008] = "\x01\x44\x37", [0x0009] = "\x01\x44\x35",
  [0x000a] = "\x01\x44\x38", [0x000b] = "\x01\x44\x36",
  [0x000c] = "\x02\x21\x27", [0x000d] = "\x01\x44\x62",
  [0x000e] = "\x02\x21\x2f", [0x000f] = "\x02\x21\x2d",
  [0x0010] = "\x01\x44\x61", [0x0011] = "\x01\x44\x60",
  [0x0014] = "\x01\x45\x62", [0x0015] = "\x01\x45\x61",
  [0x0016] = "\x01\x45\x60", [0x0018] = "\x01\x45\x63",
  [0x0019] = "\x01\x45\x5f", [0x001e] = "\x01\x47\x22",
  [0x001f] = "\x01\x47\x23", [0x0020] = "\x0e\x22\x62",
  [0x0021] = "\x0e\x22\x61", [0x0022] = "\x0e\x22\x63",
  [0x0024] = "\x0e\x24\x3f", [0x0026] = "\x01\x4b\x64",
  [0x0028] = "\x0e\x21\x21", [0x002a] = "\x0e\x21\x2c",
  [0x002b] = "\x01\x44\x39", [0x002d] = "\x01\x44\x63",
  [0x002e] = "\x02\x21\x30", [0x0030] = "\x01\x44\x64",
  [0x0031] = "\x02\x21\x43", [0x0032] = "\x01\x48\x6b",
  [0x0033] = "\x02\x25\x31", [0x0036] = "\x0e\x21\x22",
  [0x0038] = "\x01\x44\x3a", [0x0039] = "\x01\x44\x65",
  [0x003b] = "\x01\x45\x64", [0x003c] = "\x02\x21\x44",
  [0x003d] = "\x0e\x27\x40", [0x003f] = "\x0e\x21\x23",
  [0x0042] = "\x02\x21\x21", [0x0043] = "\x01\x44\x25",
  [0x0045] = "\x01\x44\x3c", [0x0046] = "\x0e\x21\x30",
  [0x0047] = "\x02\x21\x28", [0x0048] = "\x01\x44\x3d",
  [0x0049] = "\x0e\x21\x2e", [0x004b] = "\x01\x44\x66",
  [0x004d] = "\x01\x45\x65", [0x004e] = "\x01\x45\x67",
  [0x004f] = "\x01\x45\x66", [0x0052] = "\x01\x47\x24",
  [0x0053] = "\x01\x47\x25", [0x0054] = "\x0e\x23\x45",
  [0x0056] = "\x01\x4b\x65", [0x0057] = "\x0e\x2b\x22",
  [0x0058] = "\x01\x53\x7d", [0x0059] = "\x01\x44\x22",
  [0x005c] = "\x02\x21\x22", [0x005d] = "\x01\x44\x26",
  [0x005e] = "\x01\x44\x3f", [0x005f] = "\x01\x44\x3e",
  [0x0062] = "\x0e\x21\x46", [0x0063] = "\x0e\x21\x45",
  [0x0068] = "\x0e\x22\x64", [0x0069] = "\x01\x47\x26",
  [0x0071] = "\x0e\x24\x40", [0x0073] = "\x01\x4b\x66",
  [0x0079] = "\x0e\x2b\x23", [0x007e] = "\x01\x58\x71",
  [0x007f] = "\x02\x33\x4e", [0x0080] = "\x0e\x39\x6c",
  [0x0081] = "\x0e\x39\x6e", [0x0082] = "\x01\x63\x2a",
  [0x0083] = "\x02\x41\x56", [0x0084] = "\x02\x41\x57",
  [0x0085] = "\x0e\x21\x24", [0x0086] = "\x01\x44\x27",
  [0x0087] = "\x0e\x21\x2d", [0x0088] = "\x01\x44\x68",
  [0x0089] = "\x0e\x22\x65", [0x008b] = "\x01\x4b\x67",
  [0x008c] = "\x01\x44\x28", [0x008d] = "\x02\x21\x29",
  [0x008e] = "\x01\x44\x40", [0x008f] = "\x0e\x21\x31",
  [0x0091] = "\x01\x44\x69", [0x0092] = "\x01\x44\x6b",
  [0x0093] = "\x02\x21\x31", [0x0094] = "\x01\x44\x6c",
  [0x0095] = "\x01\x44\x6a", [0x0097] = "\x0e\x21\x6c",
  [0x0098] = "\x0e\x22\x66", [0x0099] = "\x01\x47\x27",
  [0x009b] = "\x01\x4b\x68", [0x009c] = "\x0e\x24\x42",
  [0x009d] = "\x0e\x27\x42", [0x009e] = "\x01\x4b\x69",
  [0x009f] = "\x01\x4f\x67", [0x00a1] = "\x01\x44\x41",
  [0x00a2] = "\x01\x44\x6d", [0x00a3] = "\x0e\x21\x47",
  [0x00a4] = "\x01\x47\x28", [0x00a5] = "\x01\x47\x2a",
  [0x00a6] = "\x01\x47\x29", [0x00a8] = "\x01\x48\x6c",
  [0x00ab] = "\x01\x4b\x6a", [0x00ac] = "\x01\x4b\x6b",
  [0x00ad] = "\x01\x4f\x68", [0x00ae] = "\x01\x4f\x69",
  [0x00af] = "\x0e\x2b\x26", [0x00b0] = "\x0e\x2b\x25",
  [0x00b1] = "\x0e\x2b\x27", [0x00b2] = "\x0e\x2b\x24",
  [0x00b3] = "\x01\x53\x7e", [0x00b6] = "\x02\x41\x58",
  [0x00b7] = "\x0e\x40\x33", [0x00b9] = "\x02\x6d\x44",
  [0x00ba] = "\x01\x44\x29", [0x00bc] = "\x0e\x21\x32",
  [0x00be] = "\x0e\x21\x33", [0x00c0] = "\x01\x44\x6f",
  [0x00c1] = "\x01\x44\x6e", [0x00c2] = "\x02\x21\x32",
  [0x00c3] = "\x01\x44\x70", [0x00c4] = "\x01\x44\x76",
  [0x00c5] = "\x0e\x21\x49", [0x00c6] = "\x01\x44\x71",
  [0x00c7] = "\x01\x44\x72", [0x00c8] = "\x02\x21\x34",
  [0x00c9] = "\x02\x21\x33", [0x00ca] = "\x01\x44\x74",
  [0x00cb] = "\x01\x44\x75", [0x00cc] = "\x0e\x21\x4c",
  [0x00cd] = "\x01\x44\x73", [0x00ce] = "\x0e\x21\x4b",
  [0x00cf] = "\x0e\x21\x4a", [0x00d4] = "\x01\x45\x6a",
  [0x00d5] = "\x01\x45\x6b", [0x00d6] = "\x01\x45\x6c",
  [0x00d7] = "\x01\x45\x6d", [0x00d8] = "\x01\x45\x69",
  [0x00d9] = "\x01\x45\x70", [0x00da] = "\x02\x21\x4a",
  [0x00dc] = "\x02\x21\x46", [0x00dd] = "\x02\x21\x49",
  [0x00de] = "\x01\x45\x71", [0x00df] = "\x01\x46\x23",
  [0x00e0] = "\x0e\x21\x6d", [0x00e1] = "\x02\x21\x48",
  [0x00e3] = "\x01\x45\x6e", [0x00e4] = "\x01\x45\x6f",
  [0x00e5] = "\x01\x45\x68", [0x00e6] = "\x0e\x21\x72",
  [0x00e7] = "\x0e\x21\x70", [0x00e8] = "\x02\x21\x45",
  [0x00e9] = "\x02\x21\x47", [0x00ee] = "\x0e\x22\x6f",
  [0x00ef] = "\x0e\x22\x68", [0x00f0] = "\x01\x47\x37",
  [0x00f1] = "\x02\x21\x6f", [0x00f2] = "\x01\x47\x34",
  [0x00f3] = "\x01\x47\x38", [0x00f4] = "\x02\x21\x79",
  [0x00f5] = "\x02\x21\x6d", [0x00f6] = "\x01\x47\x35",
  [0x00f7] = "\x02\x21\x71", [0x00f8] = "\x0e\x22\x6b",
  [0x00f9] = "\x0e\x22\x67", [0x00fa] = "\x0e\x22\x76",
  [0x00fb] = "\x01\x47\x36", [0x00fc] = "\x0e\x22\x6e",
  [0x00fd] = "\x01\x47\x39", [0x00ff] = "\x01\x47\x2b",
  [0x0100] = "\x02\x21\x70", [0x0101] = "\x01\x47\x3a",
  [0x0102] = "\x02\x21\x74", [0x0103] = "\x0e\x22\x6d",
  [0x0104] = "\x02\x21\x78", [0x0105] = "\x02\x21\x75",
  [0x0106] = "\x0e\x22\x6c", [0x0107] = "\x0e\x22\x74",
  [0x0108] = "\x02\x21\x72", [0x0109] = "\x01\x47\x2c",
  [0x010a] = "\x01\x47\x2e", [0x010b] = "\x01\x47\x3b",
  [0x010d] = "\x01\x47\x30", [0x010e] = "\x02\x21\x6a",
  [0x010f] = "\x01\x47\x33", [0x0110] = "\x01\x47\x31",
  [0x0111] = "\x01\x47\x32", [0x0112] = "\x02\x21\x7a",
  [0x0113] = "\x02\x21\x77", [0x0114] = "\x02\x21\x6e",
  [0x0115] = "\x01\x47\x2f", [0x0116] = "\x0e\x22\x70",
  [0x0117] = "\x0e\x22\x77", [0x0118] = "\x02\x21\x6b",
  [0x0119] = "\x01\x47\x2d", [0x011a] = "\x0e\x22\x75",
  [0x011c] = "\x0e\x22\x73", [0x011d] = "\x02\x21\x73",
  [0x0122] = "\x02\x21\x76", [0x0128] = "\x0e\x22\x72",
  [0x012c] = "\x02\x21\x6c", [0x012d] = "\x02\x23\x28",
  [0x012f] = "\x01\x49\x24", [0x0130] = "\x01\x48\x75",
  [0x0131] = "\x0e\x24\x50", [0x0132] = "\x0e\x24\x48",
  [0x0133] = "\x02\x23\x29", [0x0134] = "\x01\x48\x72",
  [0x0135] = "\x0e\x24\x4f", [0x0136] = "\x01\x49\x26",
  [0x0137] = "\x0e\x24\x46", [0x0138] = "\x01\x48\x7a",
  [0x0139] = "\x0e\x24\x45", [0x013a] = "\x01\x48\x79",
  [0x013b] = "\x02\x22\x7b", [0x013c] = "\x01\x48\x7d",
  [0x013d] = "\x01\x48\x78", [0x013e] = "\x02\x23\x22",
  [0x013f] = "\x02\x23\x2a", [0x0140] = "\x0e\x24\x4e",
  [0x0141] = "\x02\x23\x26", [0x0142] = "\x0e\x24\x49",
  [0x0143] = "\x01\x48\x7b", [0x0144] = "\x0e\x24\x4b",
  [0x0145] = "\x0e\x24\x43", [0x0146] = "\x01\x48\x7e",
  [0x0147] = "\x01\x48\x6f", [0x0148] = "\x01\x49\x29",
  [0x0149] = "\x02\x22\x7d", [0x014a] = "\x0e\x24\x44",
  [0x014b] = "\x0e\x24\x4c", [0x014c] = "\x02\x25\x3d",
  [0x014d] = "\x01\x48\x6d", [0x014e] = "\x01\x49\x25",
  [0x014f] = "\x01\x48\x6e", [0x0150] = "\x01\x48\x76",
  [0x0151] = "\x01\x48\x77", [0x0152] = "\x02\x23\x24",
  [0x0153] = "\x02\x22\x7e", [0x0154] = "\x01\x48\x7c",
  [0x0155] = "\x01\x48\x74", [0x0156] = "\x02\x22\x7a",
  [0x0157] = "\x01\x48\x70", [0x0158] = "\x02\x23\x27",
  [0x0159] = "\x01\x49\x27", [0x015a] = "\x01\x49\x2a",
  [0x015b] = "\x01\x48\x73", [0x015c] = "\x01\x49\x22",
  [0x015d] = "\x01\x49\x28", [0x015e] = "\x01\x48\x71",
  [0x015f] = "\x02\x23\x25", [0x0160] = "\x01\x49\x23",
  [0x0161] = "\x02\x23\x2b", [0x0162] = "\x02\x22\x7c",
  [0x0163] = "\x01\x49\x21", [0x0164] = "\x02\x23\x21",
  [0x0166] = "\x0e\x64\x48", [0x0167] = "\x02\x23\x23",
  [0x0168] = "\x0e\x64\x49", [0x0169] = "\x01\x4b\x79",
  [0x016a] = "\x02\x25\x3f", [0x016b] = "\x02\x25\x4b",
  [0x016c] = "\x01\x4b\x71", [0x016d] = "\x0e\x27\x47",
  [0x016e] = "\x02\x25\x4c", [0x016f] = "\x01\x4b\x6c",
  [0x0170] = "\x01\x4b\x76", [0x0172] = "\x0e\x27\x4f",
  [0x0173] = "\x01\x4b\x6f", [0x0174] = "\x02\x25\x39",
  [0x0175] = "\x01\x4b\x77", [0x0176] = "\x02\x25\x38",
  [0x0177] = "\x02\x25\x3c", [0x0178] = "\x02\x25\x43",
  [0x0179] = "\x02\x25\x41", [0x017a] = "\x01\x4c\x21",
  [0x017b] = "\x01\x4b\x7a", [0x017c] = "\x02\x25\x33",
  [0x017d] = "\x02\x25\x35", [0x017e] = "\x01\x4b\x7c",
  [0x017f] = "\x01\x4b\x70", [0x0180] = "\x02\x25\x36",
  [0x0181] = "\x02\x25\x42", [0x0182] = "\x02\x25\x49",
  [0x0183] = "\x01\x4b\x75", [0x0184] = "\x02\x25\x3b",
  [0x0185] = "\x02\x25\x34", [0x0186] = "\x01\x4b\x74",
  [0x0187] = "\x02\x25\x37", [0x0188] = "\x01\x4b\x78",
  [0x0189] = "\x02\x25\x3a", [0x018a] = "\x0e\x27\x44",
  [0x018b] = "\x01\x4b\x73", [0x018c] = "\x0e\x27\x51",
  [0x018d] = "\x01\x4b\x6e", [0x018e] = "\x0e\x27\x48",
  [0x018f] = "\x01\x4b\x7d", [0x0190] = "\x02\x25\x44",
  [0x0191] = "\x01\x4b\x7e", [0x0192] = "\x02\x25\x48",
  [0x0193] = "\x0e\x27\x43", [0x0194] = "\x02\x25\x46",
  [0x0195] = "\x02\x25\x4a", [0x0196] = "\x01\x4b\x7b",
  [0x0197] = "\x02\x25\x3e", [0x0198] = "\x02\x25\x32",
  [0x019a] = "\x02\x25\x40", [0x019b] = "\x01\x4b\x72",
  [0x019c] = "\x02\x25\x45", [0x019d] = "\x01\x4b\x6d",
  [0x019e] = "\x02\x25\x47", [0x019f] = "\x0e\x27\x4c",
  [0x01a0] = "\x0e\x27\x49", [0x01a1] = "\x0e\x27\x4b",
  [0x01a2] = "\x0e\x27\x4a", [0x01a3] = "\x0e\x27\x4d",
  [0x01ae] = "\x01\x4f\x78", [0x01af] = "\x01\x4f\x6c",
  [0x01b0] = "\x0e\x2b\x2d", [0x01b1] = "\x0e\x2b\x2e",
  [0x01b2] = "\x02\x29\x22", [0x01b3] = "\x02\x29\x2a",
  [0x01b4] = "\x0e\x2b\x36", [0x01b5] = "\x01\x4f\x6b",
  [0x01b6] = "\x01\x4f\x73", [0x01b7] = "\x01\x50\x21",
  [0x01b8] = "\x0e\x2b\x2c", [0x01b9] = "\x02\x29\x30",
  [0x01ba] = "\x02\x29\x2e", [0x01bb] = "\x02\x29\x29",
  [0x01bd] = "\x0e\x62\x63", [0x01be] = "\x0e\x2b\x2b",
  [0x01bf] = "\x01\x4f\x6d", [0x01c0] = "\x02\x29\x2f",
  [0x01c1] = "\x02\x29\x25", [0x01c2] = "\x01\x4f\x7b",
  [0x01c3] = "\x01\x4f\x72", [0x01c4] = "\x01\x4f\x7a",
  [0x01c5] = "\x02\x28\x7e", [0x01c6] = "\x0e\x2b\x32",
  [0x01c7] = "\x02\x29\x2c", [0x01c8] = "\x0e\x2b\x2f",
  [0x01c9] = "\x02\x29\x23", [0x01ca] = "\x01\x4f\x76",
  [0x01cb] = "\x02\x29\x24", [0x01cc] = "\x0e\x2b\x33",
  [0x01cd] = "\x02\x28\x7d", [0x01ce] = "\x01\x4f\x7d",
  [0x01cf] = "\x01\x4f\x70", [0x01d0] = "\x01\x4f\x79",
  [0x01d1] = "\x01\x4f\x6f", [0x01d2] = "\x0e\x2b\x28",
  [0x01d3] = "\x02\x29\x21", [0x01d4] = "\x02\x29\x26",
  [0x01d5] = "\x0e\x2b\x29", [0x01d6] = "\x02\x29\x2d",
  [0x01d7] = "\x01\x4f\x77", [0x01d8] = "\x01\x4f\x74",
  [0x01d9] = "\x02\x29\x28", [0x01da] = "\x01\x4f\x7c",
  [0x01db] = "\x02\x29\x2b", [0x01dc] = "\x02\x29\x27",
  [0x01dd] = "\x01\x4f\x71", [0x01de] = "\x01\x4f\x7e",
  [0x01df] = "\x01\x4f\x75", [0x01e0] = "\x01\x4f\x6e",
  [0x01e1] = "\x01\x4f\x6a", [0x01e3] = "\x0e\x2b\x35",
  [0x01e5] = "\x0e\x2b\x34", [0x01e7] = "\x0e\x64\x4a",
  [0x01ec] = "\x02\x29\x31", [0x01ee] = "\x01\x54\x3a",
  [0x01ef] = "\x01\x54\x24", [0x01f0] = "\x0e\x2f\x48",
  [0x01f1] = "\x01\x54\x34", [0x01f2] = "\x0e\x2f\x4d",
  [0x01f3] = "\x01\x54\x39", [0x01f4] = "\x02\x2d\x71",
  [0x01f5] = "\x02\x2d\x70", [0x01f6] = "\x02\x2d\x75",
  [0x01f7] = "\x02\x2d\x76", [0x01f8] = "\x01\x54\x27",
  [0x01f9] = "\x0e\x2f\x44", [0x01fa] = "\x01\x54\x30",
  [0x01fb] = "\x0e\x2f\x58", [0x01fc] = "\x0e\x2f\x4c",
  [0x01fd] = "\x0e\x2f\x56", [0x01fe] = "\x01\x54\x3d",
  [0x01ff] = "\x0e\x2f\x46", [0x0200] = "\x01\x54\x31",
  [0x0202] = "\x0e\x2f\x4b", [0x0203] = "\x0e\x2f\x49",
  [0x0204] = "\x0e\x2f\x47", [0x0205] = "\x02\x2d\x6a",
  [0x0206] = "\x01\x54\x2a", [0x0207] = "\x02\x2d\x6b",
  [0x0208] = "\x0e\x2f\x4f", [0x0209] = "\x01\x54\x3f",
  [0x020b] = "\x01\x54\x36", [0x020c] = "\x01\x54\x21",
  [0x020d] = "\x01\x54\x22", [0x020e] = "\x02\x2d\x7e",
  [0x020f] = "\x01\x59\x24", [0x0210] = "\x0e\x2f\x53",
  [0x0211] = "\x01\x54\x2f", [0x0212] = "\x01\x54\x2e",
  [0x0213] = "\x02\x2d\x6c", [0x0214] = "\x01\x54\x32",
  [0x0215] = "\x02\x33\x5e", [0x0216] = "\x01\x54\x29",
  [0x0217] = "\x02\x2d\x77", [0x0218] = "\x01\x54\x38",
  [0x0219] = "\x01\x54\x37", [0x021a] = "\x01\x54\x2d",
  [0x021b] = "\x02\x2d\x6f", [0x021c] = "\x02\x2d\x78",
  [0x021d] = "\x0e\x2f\x45", [0x021e] = "\x02\x2d\x69",
  [0x021f] = "\x01\x54\x2c", [0x0220] = "\x02\x2d\x79",
  [0x0221] = "\x01\x54\x35", [0x0222] = "\x02\x2d\x6d",
  [0x0223] = "\x01\x54\x23", [0x0224] = "\x0e\x2f\x4e",
  [0x0225] = "\x01\x54\x26", [0x0226] = "\x01\x54\x25",
  [0x0227] = "\x02\x2d\x7a", [0x0228] = "\x01\x54\x33",
  [0x0229] = "\x01\x54\x28", [0x022a] = "\x01\x54\x3c",
  [0x022b] = "\x01\x54\x3e", [0x022c] = "\x02\x2d\x74",
  [0x022d] = "\x01\x54\x3b", [0x022e] = "\x0e\x2f\x51",
  [0x022f] = "\x02\x2d\x7c", [0x0230] = "\x02\x2d\x6e",
  [0x0231] = "\x02\x2d\x7d", [0x0232] = "\x0e\x2f\x43",
  [0x0233] = "\x02\x2d\x72", [0x0235] = "\x02\x2d\x7b",
  [0x0236] = "\x0e\x2f\x50", [0x0237] = "\x02\x2d\x73",
  [0x0238] = "\x0e\x2f\x54", [0x0239] = "\x0e\x2f\x55",
  [0x023c] = "\x01\x54\x2b", [0x023f] = "\x0e\x64\x4b",
  [0x0240] = "\x02\x33\x66", [0x0241] = "\x02\x33\x5a",
  [0x0242] = "\x0e\x34\x3c", [0x0243] = "\x01\x58\x76",
  [0x0244] = "\x0e\x34\x3f", [0x0245] = "\x02\x33\x5f",
  [0x0246] = "\x02\x33\x65", [0x0247] = "\x01\x58\x75",
  [0x0248] = "\x02\x33\x58", [0x0249] = "\x01\x58\x79",
  [0x024a] = "\x02\x33\x5c", [0x024b] = "\x02\x33\x55",
  [0x024c] = "\x01\x58\x77", [0x024d] = "\x02\x33\x59",
  [0x024e] = "\x01\x58\x7c", [0x024f] = "\x01\x59\x23",
  [0x0250] = "\x0e\x34\x44", [0x0251] = "\x02\x33\x6a",
  [0x0252] = "\x0e\x34\x41", [0x0253] = "\x02\x33\x54",
  [0x0254] = "\x0e\x34\x42", [0x0255] = "\x01\x58\x7d",
  [0x0256] = "\x0e\x2f\x57", [0x0257] = "\x02\x33\x69",
  [0x0258] = "\x0e\x34\x47", [0x0259] = "\x0e\x34\x3d",
  [0x025a] = "\x01\x58\x78", [0x025b] = "\x02\x33\x5b",
  [0x025c] = "\x01\x58\x74", [0x025d] = "\x02\x33\x56",
  [0x025e] = "\x02\x33\x52", [0x025f] = "\x02\x33\x60",
  [0x0260] = "\x02\x33\x53", [0x0261] = "\x02\x33\x51",
  [0x0262] = "\x02\x33\x5d", [0x0263] = "\x02\x33\x63",
  [0x0264] = "\x02\x33\x64", [0x0265] = "\x01\x58\x7a",
  [0x0266] = "\x0e\x34\x40", [0x0268] = "\x02\x3a\x3f",
  [0x0269] = "\x02\x33\x61", [0x026a] = "\x02\x33\x50",
  [0x026b] = "\x02\x33\x62", [0x026c] = "\x0e\x34\x4b",
  [0x026d] = "\x01\x59\x26", [0x026e] = "\x02\x33\x67",
  [0x026f] = "\x01\x59\x25", [0x0270] = "\x02\x33\x4f",
  [0x0271] = "\x0e\x34\x43", [0x0272] = "\x02\x33\x57",
  [0x0273] = "\x02\x33\x68", [0x0274] = "\x01\x59\x21",
  [0x0275] = "\x01\x58\x7e", [0x0276] = "\x01\x58\x7b",
  [0x0277] = "\x01\x59\x22", [0x0278] = "\x0e\x34\x4c",
  [0x0279] = "\x0e\x34\x4a", [0x027a] = "\x01\x58\x72",
  [0x027b] = "\x0e\x34\x45", [0x027c] = "\x0e\x34\x46",
  [0x027d] = "\x01\x58\x73", [0x0280] = "\x01\x5e\x2e",
  [0x0281] = "\x0e\x39\x75", [0x0282] = "\x02\x3a\x42",
  [0x0283] = "\x02\x3a\x3b", [0x0285] = "\x01\x5e\x2b",
  [0x0287] = "\x02\x3a\x43", [0x0288] = "\x0e\x39\x71",
  [0x0289] = "\x0e\x39\x72", [0x028b] = "\x02\x3a\x39",
  [0x028c] = "\x02\x3a\x3c", [0x028d] = "\x01\x5e\x2a",
  [0x028e] = "\x02\x3a\x3d", [0x028f] = "\x0e\x39\x6f",
  [0x0291] = "\x01\x5e\x2d", [0x0292] = "\x02\x3a\x41",
  [0x0294] = "\x02\x3a\x37", [0x0295] = "\x02\x3a\x36",
  [0x0296] = "\x01\x5e\x2f", [0x0297] = "\x0e\x39\x70",
  [0x0298] = "\x01\x5e\x30", [0x0299] = "\x01\x5e\x2c",
  [0x029a] = "\x01\x5e\x31", [0x029b] = "\x02\x3a\x35",
  [0x029c] = "\x02\x3a\x40", [0x029d] = "\x02\x3a\x3e",
  [0x029e] = "\x02\x3a\x38", [0x02a2] = "\x01\x5e\x29",
  [0x02a3] = "\x02\x3a\x3a", [0x02a6] = "\x0e\x64\x4d",
  [0x02aa] = "\x0e\x40\x39", [0x02ab] = "\x0e\x40\x3c",
  [0x02ac] = "\x01\x63\x31", [0x02ad] = "\x01\x63\x2b",
  [0x02ae] = "\x02\x41\x5c", [0x02af] = "\x01\x63\x34",
  [0x02b0] = "\x02\x41\x62", [0x02b1] = "\x02\x41\x65",
  [0x02b2] = "\x01\x63\x2d", [0x02b3] = "\x01\x63\x2e",
  [0x02b4] = "\x02\x41\x5f", [0x02b5] = "\x01\x63\x2c",
  [0x02b6] = "\x02\x41\x68", [0x02b7] = "\x01\x63\x32",
  [0x02b8] = "\x02\x41\x69", [0x02b9] = "\x0e\x40\x3a",
  [0x02ba] = "\x02\x41\x64", [0x02bb] = "\x01\x63\x33",
  [0x02bc] = "\x0e\x40\x37", [0x02bd] = "\x02\x41\x59",
  [0x02be] = "\x01\x63\x30", [0x02bf] = "\x02\x41\x5a",
  [0x02c1] = "\x02\x41\x63", [0x02c2] = "\x02\x41\x61",
  [0x02c3] = "\x0e\x40\x3d", [0x02c4] = "\x02\x41\x5d",
  [0x02c5] = "\x01\x63\x2f", [0x02c6] = "\x02\x41\x5b",
  [0x02c7] = "\x01\x63\x35", [0x02c8] = "\x02\x41\x60",
  [0x02c9] = "\x02\x41\x67", [0x02ca] = "\x02\x41\x5e",
  [0x02cb] = "\x02\x41\x66", [0x02cc] = "\x0e\x40\x36",
  [0x02cd] = "\x0e\x40\x3e", [0x02ce] = "\x01\x67\x7c",
  [0x02cf] = "\x01\x67\x79", [0x02d0] = "\x0e\x45\x6b",
  [0x02d1] = "\x01\x67\x7a", [0x02d2] = "\x0e\x63\x4c",
  [0x02d3] = "\x02\x49\x21", [0x02d4] = "\x02\x48\x77",
  [0x02d5] = "\x01\x67\x78", [0x02d6] = "\x01\x67\x75",
  [0x02d7] = "\x02\x48\x78", [0x02d8] = "\x0e\x45\x6e",
  [0x02d9] = "\x0e\x45\x6c", [0x02da] = "\x01\x67\x77",
  [0x02db] = "\x02\x48\x7b", [0x02dc] = "\x0e\x45\x6d",
  [0x02dd] = "\x02\x48\x7d", [0x02de] = "\x0e\x45\x76",
  [0x02df] = "\x0e\x63\x4d", [0x02e0] = "\x02\x49\x26",
  [0x02e1] = "\x0e\x45\x70", [0x02e2] = "\x0e\x45\x75",
  [0x02e3] = "\x02\x49\x25", [0x02e4] = "\x02\x48\x7e",
  [0x02e5] = "\x01\x67\x74", [0x02e6] = "\x02\x48\x76",
  [0x02e7] = "\x01\x67\x72", [0x02e8] = "\x02\x48\x79",
  [0x02e9] = "\x01\x67\x7d", [0x02ea] = "\x02\x48\x7c",
  [0x02eb] = "\x0e\x45\x71", [0x02ec] = "\x02\x49\x22",
  [0x02ed] = "\x01\x67\x76", [0x02ee] = "\x01\x67\x73",
  [0x02ef] = "\x02\x49\x24", [0x02f0] = "\x02\x49\x23",
  [0x02f1] = "\x01\x67\x7b", [0x02f3] = "\x02\x48\x7a",
  [0x02f4] = "\x0e\x45\x74", [0x02f5] = "\x01\x6c\x23",
  [0x02f6] = "\x02\x4f\x54", [0x02f7] = "\x0e\x4b\x29",
  [0x02f8] = "\x02\x4f\x51", [0x02f9] = "\x01\x6c\x24",
  [0x02fb] = "\x01\x6c\x22", [0x02fc] = "\x0e\x4b\x2b",
  [0x02fd] = "\x02\x4f\x58", [0x02fe] = "\x02\x4f\x55",
  [0x02ff] = "\x02\x4f\x4e", [0x0300] = "\x01\x6c\x21",
  [0x0301] = "\x0e\x4b\x2d", [0x0302] = "\x01\x6c\x25",
  [0x0303] = "\x02\x4f\x4f", [0x0304] = "\x01\x6b\x7e",
  [0x0305] = "\x01\x6c\x28", [0x0306] = "\x02\x4f\x52",
  [0x0307] = "\x02\x4f\x53", [0x0308] = "\x01\x6c\x26",
  [0x0309] = "\x01\x6c\x27", [0x030a] = "\x02\x4f\x59",
  [0x030b] = "\x02\x4f\x56", [0x030c] = "\x02\x4f\x57",
  [0x030d] = "\x0e\x4b\x2c", [0x030e] = "\x0e\x4b\x27",
  [0x030f] = "\x0e\x64\x4e", [0x0310] = "\x01\x70\x33",
  [0x0311] = "\x02\x56\x43", [0x0312] = "\x01\x70\x30",
  [0x0313] = "\x02\x56\x40", [0x0314] = "\x01\x70\x32",
  [0x0315] = "\x01\x70\x34", [0x0316] = "\x0e\x50\x31",
  [0x0317] = "\x02\x56\x41", [0x0318] = "\x01\x70\x31",
  [0x031a] = "\x02\x56\x42", [0x031b] = "\x0e\x50\x35",
  [0x031c] = "\x02\x56\x3f", [0x031d] = "\x0e\x50\x30",
  [0x031e] = "\x0e\x50\x34", [0x031f] = "\x01\x73\x41",
  [0x0320] = "\x02\x5c\x49", [0x0321] = "\x01\x73\x42",
  [0x0322] = "\x02\x5c\x47", [0x0324] = "\x02\x5c\x48",
  [0x0325] = "\x02\x5c\x46", [0x0326] = "\x02\x5c\x45",
  [0x0327] = "\x0e\x54\x41", [0x0328] = "\x0e\x54\x40",
  [0x0329] = "\x02\x5c\x4a", [0x032a] = "\x01\x73\x40",
  [0x032b] = "\x0e\x50\x33", [0x032d] = "\x02\x61\x44",
  [0x032e] = "\x02\x61\x45", [0x032f] = "\x0e\x57\x52",
  [0x0330] = "\x02\x4f\x50", [0x0331] = "\x02\x61\x43",
  [0x0332] = "\x01\x73\x43", [0x0333] = "\x01\x78\x2b",
  [0x0334] = "\x02\x65\x37", [0x0335] = "\x02\x65\x38",
  [0x0336] = "\x0e\x5c\x32", [0x0337] = "\x01\x7a\x62",
  [0x0338] = "\x01\x7a\x63", [0x0339] = "\x02\x6b\x4f",
  [0x033a] = "\x02\x6b\x4e", [0x033b] = "\x01\x7b\x58",
  [0x033c] = "\x01\x7b\x57", [0x033d] = "\x02\x6e\x7d",
  [0x033e] = "\x0e\x60\x6f", [0x033f] = "\x01\x44\x2a",
  [0x0340] = "\x01\x44\x42", [0x0341] = "\x01\x44\x78",
  [0x0343] = "\x01\x44\x77", [0x0344] = "\x01\x45\x73",
  [0x0345] = "\x01\x45\x72", [0x0346] = "\x01\x47\x3e",
  [0x0347] = "\x01\x47\x3d", [0x0348] = "\x01\x47\x3f",
  [0x0349] = "\x01\x47\x3c", [0x034a] = "\x0e\x22\x78",
  [0x034b] = "\x01\x49\x2c", [0x034c] = "\x01\x49\x2b",
  [0x034d] = "\x01\x49\x2d", [0x034e] = "\x0e\x24\x54",
  [0x0350] = "\x0e\x24\x53", [0x0351] = "\x0e\x24\x51",
  [0x0352] = "\x01\x4c\x23", [0x0354] = "\x01\x4c\x22",
  [0x0355] = "\x01\x4c\x24", [0x0356] = "\x0e\x27\x52",
  [0x0357] = "\x01\x50\x22", [0x0359] = "\x01\x22\x79",
  [0x035a] = "\x02\x2e\x21", [0x035b] = "\x01\x22\x7a",
  [0x035c] = "\x01\x59\x27", [0x035d] = "\x01\x22\x7c",
  [0x035e] = "\x01\x22\x7b", [0x035f] = "\x02\x3a\x44",
  [0x0360] = "\x0e\x39\x76", [0x0361] = "\x01\x22\x7d",
  [0x0362] = "\x01\x67\x7e", [0x0363] = "\x01\x22\x7e",
  [0x0364] = "\x0e\x5e\x25", [0x0365] = "\x01\x44\x2b",
  [0x0366] = "\x0e\x21\x34", [0x0367] = "\x01\x44\x79",
  [0x0368] = "\x01\x47\x40", [0x0369] = "\x01\x4c\x25",
  [0x036a] = "\x0e\x2b\x37", [0x036b] = "\x01\x44\x2c",
  [0x036c] = "\x01\x44\x7c", [0x036d] = "\x01\x44\x7a",
  [0x036e] = "\x01\x44\x7b", [0x0371] = "\x01\x47\x41",
  [0x0372] = "\x0e\x22\x7a", [0x0375] = "\x01\x49\x2e",
  [0x0376] = "\x01\x4c\x27", [0x0377] = "\x01\x4c\x26",
  [0x0378] = "\x01\x4c\x28", [0x0379] = "\x0e\x2e\x7b",
  [0x037c] = "\x01\x54\x40", [0x037e] = "\x0e\x40\x3f",
  [0x037f] = "\x0e\x64\x53", [0x0380] = "\x01\x70\x35",
  [0x0382] = "\x0e\x21\x27", [0x0384] = "\x0e\x21\x4d",
  [0x0385] = "\x0e\x21\x48", [0x0386] = "\x0e\x21\x4e",
  [0x0387] = "\x02\x21\x2e", [0x0389] = "\x01\x45\x74",
  [0x038a] = "\x01\x45\x75", [0x038b] = "\x0e\x21\x78",
  [0x038c] = "\x0e\x21\x79", [0x038d] = "\x01\x47\x42",
  [0x038f] = "\x02\x23\x2c", [0x0390] = "\x0e\x27\x55",
  [0x0391] = "\x01\x50\x24", [0x0392] = "\x01\x50\x23",
  [0x0393] = "\x02\x2e\x23", [0x0394] = "\x02\x2e\x22",
  [0x0395] = "\x01\x59\x28", [0x0396] = "\x0e\x21\x28",
  [0x0397] = "\x01\x44\x7d", [0x0398] = "\x02\x21\x35",
  [0x0399] = "\x0e\x21\x7a", [0x039d] = "\x0e\x24\x57",
  [0x039e] = "\x02\x25\x4d", [0x039f] = "\x0e\x2b\x39",
  [0x03a0] = "\x01\x50\x25", [0x03a1] = "\x0e\x2f\x5b",
  [0x03a2] = "\x01\x54\x43", [0x03a3] = "\x0e\x2f\x59",
  [0x03a4] = "\x01\x54\x41", [0x03a5] = "\x01\x54\x42",
  [0x03a6] = "\x0e\x2f\x5a", [0x03a8] = "\x0e\x34\x4d",
  [0x03aa] = "\x01\x70\x36", [0x03ac] = "\x01\x45\x76",
  [0x03b0] = "\x01\x47\x43", [0x03b1] = "\x02\x21\x7b",
  [0x03b2] = "\x0e\x22\x7e", [0x03b3] = "\x0e\x22\x7d",
  [0x03b4] = "\x0e\x22\x7c", [0x03b5] = "\x0e\x24\x59",
  [0x03b6] = "\x01\x49\x2f", [0x03b7] = "\x01\x49\x30",
  [0x03b8] = "\x0e\x24\x5a", [0x03b9] = "\x02\x23\x2d",
  [0x03bc] = "\x02\x25\x4e", [0x03bd] = "\x01\x4c\x29",
  [0x03be] = "\x02\x25\x4f", [0x03c1] = "\x0e\x2b\x3b",
  [0x03c2] = "\x0e\x2b\x3d", [0x03c3] = "\x0e\x2b\x3e",
  [0x03c4] = "\x02\x2e\x25", [0x03c5] = "\x02\x2e\x26",
  [0x03c6] = "\x01\x54\x46", [0x03c7] = "\x0e\x2f\x5e",
  [0x03c8] = "\x02\x2e\x27", [0x03c9] = "\x0e\x2f\x5f",
  [0x03ca] = "\x02\x2e\x24", [0x03cb] = "\x01\x54\x47",
  [0x03cc] = "\x01\x54\x45", [0x03cd] = "\x01\x54\x44",
  [0x03ce] = "\x02\x2e\x28", [0x03cf] = "\x0e\x34\x4f",
  [0x03d0] = "\x02\x33\x6b", [0x03d1] = "\x0e\x34\x4e",
  [0x03d2] = "\x0e\x39\x7d", [0x03d3] = "\x0e\x39\x7a",
  [0x03d4] = "\x02\x3a\x45", [0x03d6] = "\x0e\x39\x7e",
  [0x03d7] = "\x02\x41\x6a", [0x03d8] = "\x02\x49\x27",
  [0x03d9] = "\x0e\x4b\x2f", [0x03da] = "\x0e\x4b\x2e",
  [0x03db] = "\x0e\x4b\x30", [0x03dc] = "\x01\x6c\x29",
  [0x03dd] = "\x01\x70\x37", [0x03de] = "\x02\x56\x44",
  [0x03df] = "\x0e\x54\x42", [0x03e0] = "\x01\x44\x2d",
  [0x03e1] = "\x01\x44\x3b", [0x03e2] = "\x0e\x21\x2f",
  [0x03e3] = "\x0e\x21\x35", [0x03e4] = "\x0e\x21\x51",
  [0x03e5] = "\x0e\x21\x7b", [0x03e8] = "\x0e\x23\x22",
  [0x03ec] = "\x0e\x24\x5b", [0x03ed] = "\x0e\x27\x59",
  [0x03f0] = "\x01\x59\x29", [0x03f1] = "\x01\x5e\x33",
  [0x03f3] = "\x01\x68\x21", [0x03f4] = "\x0e\x45\x7a",
  [0x03f5] = "\x02\x21\x23", [0x03f6] = "\x01\x44\x7e",
  [0x03f8] = "\x01\x45\x79", [0x03f9] = "\x01\x45\x77",
  [0x03fa] = "\x01\x45\x78", [0x03fd] = "\x01\x4c\x2a",
  [0x03fe] = "\x0e\x27\x5a", [0x0400] = "\x01\x44\x2e",
  [0x0401] = "\x01\x44\x2f", [0x0403] = "\x01\x44\x43",
  [0x0404] = "\x0e\x21\x36", [0x0405] = "\x0e\x21\x52",
  [0x0406] = "\x01\x45\x21", [0x0407] = "\x01\x45\x22",
  [0x0408] = "\x01\x45\x23", [0x0409] = "\x02\x21\x4c",
  [0x040a] = "\x01\x45\x7a", [0x040b] = "\x0e\x21\x7d",
  [0x040c] = "\x02\x21\x4b", [0x040e] = "\x01\x47\x47",
  [0x040f] = "\x0e\x23\x25", [0x0410] = "\x02\x21\x7d",
  [0x0411] = "\x01\x47\x45", [0x0412] = "\x01\x47\x46",
  [0x0413] = "\x02\x21\x7c", [0x0414] = "\x0e\x23\x24",
  [0x0415] = "\x0e\x23\x26", [0x0416] = "\x01\x47\x48",
  [0x0417] = "\x01\x47\x44", [0x0418] = "\x0e\x23\x27",
  [0x041c] = "\x02\x23\x2e", [0x041d] = "\x01\x4f\x4f",
  [0x041e] = "\x02\x23\x2f", [0x041f] = "\x0e\x24\x5f",
  [0x0420] = "\x0e\x24\x61", [0x0421] = "\x02\x23\x30",
  [0x0423] = "\x0e\x24\x5c", [0x0424] = "\x01\x49\x32",
  [0x0425] = "\x01\x49\x31", [0x0426] = "\x0e\x24\x5e",
  [0x0427] = "\x0e\x24\x5d", [0x0428] = "\x01\x49\x35",
  [0x0429] = "\x01\x49\x33", [0x042a] = "\x01\x49\x34",
  [0x042b] = "\x0e\x24\x60", [0x042e] = "\x01\x4c\x30",
  [0x042f] = "\x0e\x27\x5b", [0x0430] = "\x01\x4c\x2f",
  [0x0431] = "\x02\x25\x54", [0x0432] = "\x02\x25\x51",
  [0x0433] = "\x02\x25\x52", [0x0434] = "\x0e\x27\x5e",
  [0x0435] = "\x02\x25\x50", [0x0436] = "\x01\x4c\x31",
  [0x0437] = "\x01\x4c\x2d", [0x0438] = "\x01\x4c\x2c",
  [0x0439] = "\x0e\x27\x5f", [0x043a] = "\x01\x4c\x2e",
  [0x043b] = "\x01\x4c\x2b", [0x043c] = "\x0e\x27\x5d",
  [0x043e] = "\x0e\x64\x4f", [0x0441] = "\x01\x4c\x32",
  [0x0443] = "\x01\x50\x27", [0x0444] = "\x02\x29\x32",
  [0x0445] = "\x0e\x2b\x3f", [0x0446] = "\x02\x25\x53",
  [0x0447] = "\x01\x50\x2c", [0x0448] = "\x0e\x2b\x40",
  [0x0449] = "\x02\x29\x33", [0x044a] = "\x01\x50\x28",
  [0x044b] = "\x01\x50\x2b", [0x044c] = "\x01\x50\x2a",
  [0x044d] = "\x01\x50\x29", [0x044e] = "\x01\x50\x26",
  [0x044f] = "\x0e\x2b\x43", [0x0452] = "\x02\x2e\x2b",
  [0x0453] = "\x0e\x2f\x68", [0x0454] = "\x01\x54\x4a",
  [0x0455] = "\x02\x2e\x2e", [0x0456] = "\x01\x54\x48",
  [0x0457] = "\x0e\x2f\x65", [0x0459] = "\x0e\x2f\x62",
  [0x045a] = "\x02\x2e\x2a", [0x045b] = "\x01\x54\x4b",
  [0x045c] = "\x01\x54\x49", [0x045d] = "\x01\x54\x4c",
  [0x045e] = "\x02\x2e\x2c", [0x045f] = "\x02\x2e\x2d",
  [0x0460] = "\x0e\x2f\x60", [0x0461] = "\x02\x2e\x29",
  [0x0462] = "\x02\x2e\x2f", [0x0463] = "\x0e\x2f\x66",
  [0x0464] = "\x0e\x2f\x61", [0x0465] = "\x0e\x2f\x63",
  [0x0467] = "\x0e\x2f\x64", [0x0468] = "\x0e\x34\x50",
  [0x0469] = "\x01\x5e\x37", [0x046a] = "\x01\x59\x2a",
  [0x046b] = "\x02\x33\x6c", [0x046c] = "\x02\x33\x6e",
  [0x046d] = "\x02\x33\x6d", [0x046e] = "\x02\x33\x6f",
  [0x046f] = "\x01\x59\x2b", [0x0470] = "\x0e\x3a\x23",
  [0x0472] = "\x01\x5e\x34", [0x0473] = "\x0e\x3a\x21",
  [0x0474] = "\x01\x5e\x35", [0x0475] = "\x01\x5e\x36",
  [0x0476] = "\x0e\x34\x51", [0x0477] = "\x01\x63\x37",
  [0x0478] = "\x02\x41\x6c", [0x0479] = "\x0e\x40\x41",
  [0x047a] = "\x02\x41\x6b", [0x047b] = "\x02\x41\x6d",
  [0x047c] = "\x02\x41\x6e", [0x047d] = "\x01\x63\x38",
  [0x047e] = "\x0e\x40\x40", [0x047f] = "\x01\x63\x36",
  [0x0480] = "\x02\x49\x28", [0x0481] = "\x02\x49\x29",
  [0x0482] = "\x01\x68\x23", [0x0483] = "\x01\x68\x22",
  [0x0484] = "\x02\x4c\x61", [0x0486] = "\x0e\x4b\x31",
  [0x0487] = "\x01\x6c\x2a", [0x0488] = "\x01\x6c\x2b",
  [0x0489] = "\x01\x6c\x2c", [0x048a] = "\x01\x6c\x2e",
  [0x048b] = "\x02\x4f\x5a", [0x048c] = "\x02\x4f\x5b",
  [0x048d] = "\x01\x6c\x2d", [0x048e] = "\x0e\x4b\x32",
  [0x0490] = "\x0e\x50\x36", [0x0491] = "\x01\x70\x38",
  [0x0493] = "\x01\x70\x39", [0x0494] = "\x0e\x50\x37",
  [0x0496] = "\x02\x65\x39", [0x0497] = "\x02\x6b\x51",
  [0x0498] = "\x02\x6b\x50", [0x0499] = "\x02\x6e\x7e",
  [0x049a] = "\x0e\x60\x2c", [0x049b] = "\x01\x44\x30",
  [0x049c] = "\x0e\x21\x37", [0x049d] = "\x0e\x21\x54",
  [0x049e] = "\x0e\x21\x53", [0x049f] = "\x01\x45\x7c",
  [0x04a0] = "\x01\x45\x7b", [0x04a3] = "\x01\x47\x49",
  [0x04a6] = "\x02\x21\x7e", [0x04a8] = "\x0e\x23\x28",
  [0x04a9] = "\x01\x49\x37", [0x04aa] = "\x01\x49\x38",
  [0x04ab] = "\x01\x49\x36", [0x04ac] = "\x01\x49\x39",
  [0x04ad] = "\x02\x23\x31", [0x04ae] = "\x02\x23\x32",
  [0x04b3] = "\x0e\x24\x63", [0x04b4] = "\x0e\x24\x62",
  [0x04b5] = "\x0e\x27\x61", [0x04b9] = "\x0e\x27\x60",
  [0x04bb] = "\x01\x4c\x34", [0x04bc] = "\x02\x25\x55",
  [0x04bd] = "\x0e\x62\x5a", [0x04be] = "\x01\x4c\x33",
  [0x04bf] = "\x0e\x27\x62", [0x04c0] = "\x02\x29\x34",
  [0x04c1] = "\x01\x50\x30", [0x04c2] = "\x02\x29\x35",
  [0x04c3] = "\x01\x50\x2f", [0x04c4] = "\x0e\x2b\x48",
  [0x04c5] = "\x0e\x2b\x46", [0x04c7] = "\x01\x50\x2d",
  [0x04c9] = "\x01\x50\x2e", [0x04ca] = "\x0e\x2b\x47",
  [0x04cc] = "\x0e\x2f\x6f", [0x04cd] = "\x02\x2e\x30",
  [0x04ce] = "\x0e\x2f\x6c", [0x04cf] = "\x0e\x2f\x6a",
  [0x04d0] = "\x0e\x2f\x6d", [0x04d1] = "\x0e\x2f\x6e",
  [0x04d2] = "\x01\x59\x2c", [0x04d3] = "\x02\x33\x71",
  [0x04d4] = "\x0e\x34\x52", [0x04d5] = "\x01\x59\x2f",
  [0x04d6] = "\x02\x33\x70", [0x04d7] = "\x01\x5b\x2b",
  [0x04d8] = "\x01\x59\x2e", [0x04d9] = "\x01\x59\x2d",
  [0x04db] = "\x01\x5e\x3a", [0x04dd] = "\x01\x5e\x39",
  [0x04de] = "\x01\x5e\x38", [0x04df] = "\x01\x63\x39",
  [0x04e0] = "\x0e\x40\x45", [0x04e1] = "\x0e\x40\x44",
  [0x04e2] = "\x01\x63\x3c", [0x04e3] = "\x01\x63\x3d",
  [0x04e4] = "\x01\x63\x3b", [0x04e6] = "\x01\x63\x3a",
  [0x04e7] = "\x0e\x40\x46", [0x04e9] = "\x02\x49\x2a",
  [0x04ea] = "\x0e\x46\x21", [0x04eb] = "\x02\x49\x2b",
  [0x04ed] = "\x0e\x45\x7e", [0x04ee] = "\x0e\x4b\x33",
  [0x04ef] = "\x02\x4f\x5d", [0x04f0] = "\x01\x6c\x2f",
  [0x04f1] = "\x02\x4f\x5c", [0x04f2] = "\x0e\x4d\x2c",
  [0x04f3] = "\x01\x70\x3a", [0x04f4] = "\x02\x5c\x4b",
  [0x04f5] = "\x01\x73\x44", [0x04f7] = "\x02\x65\x3a",
  [0x04f8] = "\x01\x79\x61", [0x04fa] = "\x01\x44\x44",
  [0x04fb] = "\x01\x45\x24", [0x04fc] = "\x02\x21\x36",
  [0x04fd] = "\x0e\x21\x55", [0x04fe] = "\x01\x45\x25",
  [0x04ff] = "\x01\x45\x26", [0x0500] = "\x0e\x21\x56",
  [0x0503] = "\x0e\x22\x23", [0x0504] = "\x0e\x22\x22",
  [0x0505] = "\x01\x45\x7d", [0x0506] = "\x01\x45\x7e",
  [0x0507] = "\x0e\x22\x24", [0x0508] = "\x01\x47\x4a",
  [0x0509] = "\x02\x23\x33", [0x050a] = "\x02\x25\x56",
  [0x050b] = "\x02\x25\x57", [0x050d] = "\x01\x50\x31",
  [0x050e] = "\x02\x2e\x31", [0x050f] = "\x01\x59\x31",
  [0x0510] = "\x01\x59\x30", [0x0511] = "\x02\x3a\x47",
  [0x0512] = "\x02\x3a\x46", [0x0514] = "\x0e\x50\x38",
  [0x0515] = "\x01\x44\x31", [0x0516] = "\x01\x45\x27",
  [0x0517] = "\x01\x46\x21", [0x0519] = "\x01\x59\x32",
  [0x051a] = "\x02\x21\x24", [0x051c] = "\x02\x21\x4d",
  [0x051d] = "\x01\x46\x22", [0x051e] = "\x0e\x22\x26",
  [0x051f] = "\x02\x22\x22", [0x0520] = "\x01\x47\x4c",
  [0x0521] = "\x01\x47\x4b", [0x0522] = "\x02\x22\x21",
  [0x0523] = "\x01\x49\x3a", [0x0524] = "\x0e\x64\x50",
  [0x0525] = "\x0e\x24\x65", [0x0527] = "\x0e\x2b\x49",
  [0x052a] = "\x01\x54\x4d", [0x052d] = "\x02\x33\x72",
  [0x052f] = "\x01\x63\x3e", [0x0530] = "\x02\x49\x2c",
  [0x0531] = "\x01\x68\x24", [0x0532] = "\x0e\x46\x23",
  [0x0533] = "\x0e\x4b\x34", [0x0534] = "\x02\x56\x45",
  [0x0535] = "\x0e\x54\x44", [0x0536] = "\x0e\x5c\x34",
  [0x0537] = "\x02\x68\x6a", [0x0538] = "\x0e\x21\x29",
  [0x0539] = "\x01\x45\x28", [0x053a] = "\x0e\x21\x57",
  [0x053b] = "\x0e\x24\x66", [0x053c] = "\x02\x25\x58",
  [0x053d] = "\x02\x29\x36", [0x053e] = "\x01\x59\x35",
  [0x053f] = "\x01\x59\x33", [0x0540] = "\x01\x59\x34",
  [0x0541] = "\x01\x44\x32", [0x0543] = "\x01\x44\x45",
  [0x0544] = "\x01\x24\x3f", [0x0545] = "\x01\x45\x2b",
  [0x0546] = "\x0e\x21\x59", [0x0547] = "\x01\x45\x2a",
  [0x0548] = "\x01\x45\x29", [0x0549] = "\x01\x46\x25",
  [0x054a] = "\x01\x46\x24", [0x054b] = "\x0e\x23\x2a",
  [0x054c] = "\x02\x21\x4e", [0x054d] = "\x02\x22\x23",
  [0x054f] = "\x0e\x23\x2b", [0x0550] = "\x0e\x23\x2d",
  [0x0551] = "\x01\x4c\x38", [0x0552] = "\x01\x4c\x35",
  [0x0553] = "\x01\x4c\x37", [0x0554] = "\x01\x4c\x36",
  [0x0555] = "\x0e\x27\x63", [0x0557] = "\x01\x50\x32",
  [0x0558] = "\x0e\x2b\x4a", [0x055a] = "\x01\x5e\x3b",
  [0x055c] = "\x01\x44\x33", [0x055d] = "\x0e\x21\x5a",
  [0x055e] = "\x01\x45\x2c", [0x055f] = "\x0e\x22\x27",
  [0x0560] = "\x01\x46\x27", [0x0561] = "\x01\x46\x26",
  [0x0563] = "\x02\x23\x34", [0x0565] = "\x0e\x64\x51",
  [0x0566] = "\x01\x4c\x39", [0x0567] = "\x0e\x2a\x5c",
  [0x0569] = "\x0e\x21\x2a", [0x056c] = "\x02\x21\x37",
  [0x056d] = "\x0e\x22\x28", [0x056e] = "\x01\x46\x29",
  [0x056f] = "\x01\x46\x28", [0x0570] = "\x01\x47\x4d",
  [0x0571] = "\x01\x47\x4e", [0x0572] = "\x02\x23\x35",
  [0x0573] = "\x01\x49\x3b", [0x0574] = "\x0e\x24\x67",
  [0x0575] = "\x01\x49\x3c", [0x0576] = "\x0e\x27\x65",
  [0x0577] = "\x01\x4c\x3a", [0x0578] = "\x01\x4c\x3b",
  [0x0579] = "\x01\x4c\x3c", [0x057a] = "\x0e\x27\x66",
  [0x057b] = "\x01\x50\x33", [0x057c] = "\x02\x29\x37",
  [0x057d] = "\x0e\x2b\x4b", [0x057f] = "\x01\x54\x4e",
  [0x0580] = "\x0e\x40\x47", [0x0582] = "\x02\x21\x25",
  [0x0584] = "\x01\x45\x2d", [0x0586] = "\x0e\x21\x5b",
  [0x0589] = "\x0e\x22\x29", [0x058a] = "\x02\x22\x24",
  [0x058b] = "\x0e\x23\x2f", [0x058e] = "\x02\x23\x36",
  [0x058f] = "\x02\x23\x37", [0x0592] = "\x02\x25\x59",
  [0x0593] = "\x0e\x27\x67", [0x0594] = "\x02\x25\x5a",
  [0x0596] = "\x02\x29\x39", [0x0597] = "\x02\x29\x38",
  [0x0598] = "\x02\x29\x3b", [0x0599] = "\x02\x29\x3a",
  [0x059a] = "\x01\x50\x34", [0x059b] = "\x0e\x64\x52",
  [0x059c] = "\x02\x33\x73", [0x059d] = "\x01\x54\x50",
  [0x059e] = "\x02\x2e\x32", [0x059f] = "\x01\x54\x4f",
  [0x05a0] = "\x0e\x34\x54", [0x05a4] = "\x02\x3a\x48",
  [0x05a5] = "\x01\x5e\x3c", [0x05a6] = "\x0e\x3a\x28",
  [0x05a7] = "\x02\x3a\x49", [0x05a8] = "\x0e\x3a\x27",
  [0x05a9] = "\x0e\x40\x4a", [0x05aa] = "\x0e\x40\x49",
  [0x05ab] = "\x0e\x40\x48", [0x05ac] = "\x02\x49\x2d",
  [0x05ad] = "\x01\x68\x25", [0x05ae] = "\x0e\x46\x25",
  [0x05b0] = "\x0e\x46\x26", [0x05b1] = "\x0e\x4b\x35",
  [0x05b2] = "\x01\x6c\x30", [0x05b3] = "\x0e\x54\x45",
  [0x05b4] = "\x02\x65\x3b", [0x05b6] = "\x0e\x21\x2b",
  [0x05b7] = "\x0e\x21\x5c", [0x05b9] = "\x02\x21\x38",
  [0x05ba] = "\x0e\x22\x2a", [0x05bb] = "\x01\x46\x2a",
  [0x05be] = "\x0e\x23\x31", [0x05c1] = "\x0e\x27\x69",
  [0x05c2] = "\x0e\x27\x6a", [0x05c3] = "\x01\x59\x36",
  [0x05c4] = "\x0e\x34\x55", [0x05c5] = "\x0e\x3a\x29",
  [0x05c8] = "\x01\x44\x34", [0x05c9] = "\x01\x44\x46",
  [0x05ca] = "\x01\x45\x2f", [0x05cb] = "\x01\x45\x2e",
  [0x05cc] = "\x0e\x21\x5e", [0x05cd] = "\x01\x45\x30",
  [0x05ce] = "\x0e\x21\x60", [0x05d0] = "\x0e\x22\x2b",
  [0x05d2] = "\x0e\x23\x33", [0x05d4] = "\x01\x4c\x3e",
  [0x05d5] = "\x0e\x27\x6b", [0x05d6] = "\x01\x4c\x3d",
  [0x05d7] = "\x01\x4c\x3f", [0x05d9] = "\x0e\x2b\x50",
  [0x05da] = "\x0e\x2b\x4f", [0x05db] = "\x01\x50\x35",
  [0x05dc] = "\x0e\x2b\x4e", [0x05dd] = "\x0e\x2b\x4d",
  [0x05df] = "\x01\x54\x51", [0x05e0] = "\x0e\x40\x4b",
  [0x05e1] = "\x02\x56\x46", [0x05e2] = "\x01\x76\x3b",
  [0x05e3] = "\x01\x44\x47", [0x05e4] = "\x01\x46\x2c",
  [0x05e5] = "\x01\x46\x3b", [0x05e6] = "\x01\x46\x36",
  [0x05e8] = "\x01\x46\x31", [0x05e9] = "\x01\x46\x30",
  [0x05ea] = "\x01\x46\x37", [0x05eb] = "\x01\x46\x35",
  [0x05ec] = "\x01\x46\x2e", [0x05ed] = "\x01\x46\x3c",
  [0x05ee] = "\x01\x46\x2f", [0x05ef] = "\x01\x46\x2b",
  [0x05f0] = "\x01\x46\x3a", [0x05f1] = "\x01\x46\x39",
  [0x05f2] = "\x01\x46\x38", [0x05f3] = "\x01\x46\x2d",
  [0x05f4] = "\x0e\x22\x31", [0x05f5] = "\x01\x46\x34",
  [0x05f6] = "\x0e\x22\x2d", [0x05f7] = "\x0e\x22\x2e",
  [0x05f8] = "\x01\x46\x33", [0x05f9] = "\x0e\x22\x2f",
  [0x05fb] = "\x01\x46\x3d", [0x05fc] = "\x01\x46\x32",
  [0x05ff] = "\x0e\x23\x35", [0x0600] = "\x0e\x23\x37",
  [0x0601] = "\x01\x47\x54", [0x0603] = "\x01\x47\x5a",
  [0x0604] = "\x01\x47\x56", [0x0605] = "\x0e\x23\x39",
  [0x0606] = "\x01\x47\x5c", [0x0607] = "\x02\x22\x25",
  [0x0608] = "\x01\x47\x59", [0x0609] = "\x01\x47\x4f",
  [0x060a] = "\x01\x47\x52", [0x060b] = "\x01\x47\x55",
  [0x060c] = "\x01\x47\x51", [0x060d] = "\x01\x47\x58",
  [0x060e] = "\x01\x47\x5b", [0x060f] = "\x01\x47\x50",
  [0x0610] = "\x01\x47\x53", [0x0611] = "\x01\x47\x57",
  [0x0612] = "\x01\x47\x5d", [0x0613] = "\x0e\x23\x3a",
  [0x0615] = "\x0e\x23\x3b", [0x0616] = "\x0e\x23\x34",
  [0x0618] = "\x02\x23\x40", [0x0619] = "\x02\x23\x3d",
  [0x061a] = "\x0e\x24\x76", [0x061b] = "\x01\x49\x49",
  [0x061c] = "\x02\x23\x3e", [0x061d] = "\x01\x49\x3d",
  [0x061e] = "\x01\x49\x3f", [0x061f] = "\x01\x49\x57",
  [0x0620] = "\x01\x49\x52", [0x0621] = "\x0e\x24\x79",
  [0x0622] = "\x0e\x24\x7e", [0x0623] = "\x0e\x24\x75",
  [0x0624] = "\x02\x23\x45", [0x0625] = "\x02\x23\x3f",
  [0x0626] = "\x01\x49\x41", [0x0627] = "\x01\x49\x43",
  [0x0628] = "\x02\x23\x44", [0x0629] = "\x01\x49\x4a",
  [0x062a] = "\x02\x23\x3a", [0x062b] = "\x01\x49\x56",
  [0x062c] = "\x01\x49\x58", [0x062d] = "\x01\x49\x3e",
  [0x062e] = "\x01\x49\x4f", [0x062f] = "\x0e\x25\x24",
  [0x0630] = "\x02\x23\x38", [0x0631] = "\x01\x49\x55",
  [0x0632] = "\x0e\x24\x77", [0x0633] = "\x01\x49\x46",
  [0x0634] = "\x0e\x24\x7b", [0x0635] = "\x01\x49\x50",
  [0x0636] = "\x01\x49\x51", [0x0637] = "\x02\x23\x39",
  [0x0638] = "\x01\x49\x4e", [0x0639] = "\x01\x49\x4c",
  [0x063a] = "\x0e\x24\x70", [0x063b] = "\x01\x49\x4d",
  [0x063c] = "\x01\x49\x53", [0x063d] = "\x02\x23\x41",
  [0x063e] = "\x01\x49\x40", [0x063f] = "\x0e\x25\x21",
  [0x0640] = "\x01\x49\x54", [0x0641] = "\x02\x23\x43",
  [0x0642] = "\x01\x49\x48", [0x0643] = "\x01\x49\x45",
  [0x0644] = "\x0e\x24\x73", [0x0645] = "\x02\x23\x3c",
  [0x0646] = "\x01\x49\x44", [0x0647] = "\x02\x23\x46",
  [0x0648] = "\x01\x49\x47", [0x0649] = "\x0e\x24\x7c",
  [0x064a] = "\x01\x49\x4b", [0x064b] = "\x0e\x24\x78",
  [0x064c] = "\x0e\x24\x74", [0x064d] = "\x0e\x24\x6d",
  [0x064e] = "\x01\x49\x42", [0x064f] = "\x02\x23\x42",
  [0x0650] = "\x0e\x24\x7d", [0x0651] = "\x0e\x25\x22",
  [0x0654] = "\x02\x23\x3b", [0x065a] = "\x0e\x25\x23",
  [0x065f] = "\x0e\x27\x6e", [0x0660] = "\x02\x25\x6b",
  [0x0661] = "\x02\x25\x6a", [0x0662] = "\x01\x4c\x51",
  [0x0663] = "\x02\x25\x6d", [0x0664] = "\x02\x25\x6f",
  [0x0665] = "\x02\x25\x64", [0x0666] = "\x02\x25\x67",
  [0x0667] = "\x02\x25\x6e", [0x0668] = "\x01\x4c\x52",
  [0x0669] = "\x0e\x27\x74", [0x066a] = "\x0e\x27\x78",
  [0x066b] = "\x02\x25\x61", [0x066c] = "\x02\x25\x65",
  [0x066d] = "\x0e\x27\x75", [0x066f] = "\x02\x25\x69",
  [0x0670] = "\x02\x29\x4a", [0x0671] = "\x01\x4c\x4d",
  [0x0672] = "\x02\x29\x4e", [0x0673] = "\x01\x4c\x40",
  [0x0674] = "\x02\x25\x66", [0x0675] = "\x01\x4c\x41",
  [0x0676] = "\x01\x4c\x4e", [0x0677] = "\x01\x4c\x47",
  [0x0678] = "\x01\x4c\x43", [0x0679] = "\x0e\x27\x71",
  [0x067a] = "\x02\x25\x62", [0x067b] = "\x01\x4c\x46",
  [0x067c] = "\x01\x4c\x4b", [0x067d] = "\x01\x4c\x54",
  [0x067e] = "\x02\x25\x63", [0x067f] = "\x02\x25\x5c",
  [0x0680] = "\x01\x4c\x45", [0x0681] = "\x02\x25\x5d",
  [0x0682] = "\x02\x25\x5f", [0x0684] = "\x01\x4c\x48",
  [0x0685] = "\x0e\x27\x6c", [0x0686] = "\x01\x4c\x4a",
  [0x0687] = "\x02\x25\x5b", [0x0688] = "\x02\x25\x60",
  [0x0689] = "\x0e\x27\x70", [0x068a] = "\x0e\x27\x79",
  [0x068b] = "\x01\x4c\x53", [0x068c] = "\x01\x4c\x4f",
  [0x068d] = "\x02\x25\x68", [0x068e] = "\x01\x4c\x55",
  [0x068f] = "\x0e\x27\x73", [0x0690] = "\x01\x4c\x4c",
  [0x0691] = "\x02\x25\x5e", [0x0692] = "\x01\x4c\x49",
  [0x0693] = "\x0e\x27\x6f", [0x0694] = "\x0e\x27\x77",
  [0x0695] = "\x01\x4c\x44", [0x0696] = "\x01\x4c\x42",
  [0x0698] = "\x02\x25\x6c", [0x069a] = "\x01\x4c\x50",
  [0x069e] = "\x0e\x2b\x5c", [0x069f] = "\x0e\x2b\x5d",
  [0x06a0] = "\x02\x29\x49", [0x06a1] = "\x02\x29\x3d",
  [0x06a2] = "\x02\x29\x4c", [0x06a3] = "\x0e\x2b\x57",
  [0x06a4] = "\x0e\x2b\x59", [0x06a5] = "\x02\x29\x3f",
  [0x06a6] = "\x01\x50\x3c", [0x06a7] = "\x01\x50\x4a",
  [0x06a8] = "\x01\x50\x38", [0x06a9] = "\x01\x50\x49",
  [0x06aa] = "\x01\x50\x41", [0x06ab] = "\x01\x50\x46",
  [0x06ac] = "\x01\x50\x36", [0x06ad] = "\x02\x29\x3e",
  [0x06ae] = "\x02\x29\x44", [0x06af] = "\x01\x50\x45",
  [0x06b0] = "\x02\x29\x50", [0x06b1] = "\x01\x50\x47",
  [0x06b2] = "\x0e\x2b\x5b", [0x06b3] = "\x01\x50\x3d",
  [0x06b4] = "\x0e\x2b\x54", [0x06b5] = "\x0e\x2b\x5e",
  [0x06b6] = "\x02\x29\x46", [0x06b7] = "\x02\x29\x43",
  [0x06b8] = "\x01\x50\x3b", [0x06b9] = "\x0e\x2b\x51",
  [0x06ba] = "\x02\x29\x3c", [0x06bb] = "\x01\x50\x48",
  [0x06bc] = "\x02\x29\x4b", [0x06bd] = "\x01\x50\x40",
  [0x06be] = "\x02\x29\x4d", [0x06bf] = "\x01\x50\x4b",
  [0x06c0] = "\x01\x50\x37", [0x06c1] = "\x01\x50\x42",
  [0x06c2] = "\x01\x50\x3f", [0x06c3] = "\x02\x29\x41",
  [0x06c4] = "\x01\x50\x43", [0x06c5] = "\x02\x29\x47",
  [0x06c6] = "\x02\x29\x48", [0x06c7] = "\x01\x50\x3e",
  [0x06c8] = "\x01\x50\x44", [0x06c9] = "\x01\x50\x3a",
  [0x06ca] = "\x0e\x2b\x55", [0x06cb] = "\x0e\x64\x54",
  [0x06cc] = "\x0e\x2b\x63", [0x06cd] = "\x0e\x2b\x61",
  [0x06ce] = "\x01\x50\x39", [0x06cf] = "\x02\x29\x40",
  [0x06d0] = "\x0e\x2b\x53", [0x06d6] = "\x02\x29\x45",
  [0x06da] = "\x0e\x2b\x58", [0x06de] = "\x02\x29\x4f",
  [0x06e0] = "\x02\x2e\x46", [0x06e1] = "\x01\x54\x5e",
  [0x06e2] = "\x02\x2e\x34", [0x06e4] = "\x02\x2e\x39",
  [0x06e5] = "\x01\x54\x57", [0x06e6] = "\x01\x54\x62",
  [0x06e7] = "\x02\x2e\x37", [0x06e8] = "\x01\x54\x52",
  [0x06e9] = "\x01\x54\x5c", [0x06ea] = "\x01\x54\x61",
  [0x06eb] = "\x02\x2e\x3e", [0x06ed] = "\x01\x54\x5d",
  [0x06ee] = "\x01\x54\x60", [0x06ef] = "\x0e\x2f\x77",
  [0x06f0] = "\x0e\x30\x21", [0x06f1] = "\x02\x2e\x41",
  [0x06f2] = "\x01\x54\x58", [0x06f3] = "\x02\x2e\x38",
  [0x06f4] = "\x0e\x2f\x74", [0x06f5] = "\x0e\x2f\x78",
  [0x06f6] = "\x0e\x30\x22", [0x06f7] = "\x02\x2e\x44",
  [0x06f8] = "\x02\x2e\x45", [0x06f9] = "\x0e\x2f\x79",
  [0x06fa] = "\x01\x54\x5a", [0x06fb] = "\x02\x2e\x43",
  [0x06fc] = "\x01\x54\x56", [0x06fd] = "\x01\x54\x65",
  [0x06ff] = "\x02\x2e\x3b", [0x0700] = "\x0e\x2f\x7b",
  [0x0701] = "\x01\x54\x54", [0x0702] = "\x0e\x2f\x7a",
  [0x0703] = "\x02\x2e\x48", [0x0704] = "\x02\x2e\x3c",
  [0x0705] = "\x02\x2e\x40", [0x0706] = "\x01\x54\x59",
  [0x0707] = "\x01\x54\x64", [0x0708] = "\x02\x2e\x3d",
  [0x0709] = "\x01\x54\x5f", [0x070a] = "\x02\x2e\x42",
  [0x070b] = "\x02\x2e\x49", [0x070c] = "\x02\x34\x24",
  [0x070d] = "\x0e\x2f\x73", [0x070e] = "\x02\x2e\x47",
  [0x070f] = "\x01\x54\x66", [0x0710] = "\x01\x54\x53",
  [0x0711] = "\x02\x2e\x3f", [0x0712] = "\x02\x2e\x36",
  [0x0713] = "\x0e\x2f\x76", [0x0714] = "\x01\x54\x5b",
  [0x0717] = "\x02\x2e\x35", [0x0718] = "\x0e\x2f\x7e",
  [0x0719] = "\x0e\x30\x25", [0x071a] = "\x02\x2e\x3a",
  [0x0726] = "\x02\x2e\x33", [0x0727] = "\x01\x54\x63",
  [0x0728] = "\x0e\x67\x21", [0x0729] = "\x0e\x34\x5f",
  [0x072a] = "\x02\x33\x7a", [0x072b] = "\x0e\x34\x68",
  [0x072c] = "\x01\x59\x49", [0x072d] = "\x02\x34\x2a",
  [0x072e] = "\x01\x59\x47", [0x072f] = "\x01\x59\x44",
  [0x0730] = "\x02\x34\x21", [0x0731] = "\x01\x59\x40",
  [0x0732] = "\x02\x34\x25", [0x0733] = "\x01\x59\x4b",
  [0x0734] = "\x02\x33\x79", [0x0735] = "\x02\x33\x7e",
  [0x0736] = "\x02\x33\x7d", [0x0737] = "\x01\x54\x55",
  [0x0738] = "\x01\x59\x46", [0x0739] = "\x02\x34\x28",
  [0x073a] = "\x0e\x34\x66", [0x073b] = "\x02\x34\x2b",
  [0x073c] = "\x02\x33\x76", [0x073e] = "\x01\x5e\x4e",
  [0x073f] = "\x0e\x34\x67", [0x0740] = "\x02\x34\x2c",
  [0x0741] = "\x01\x59\x4c", [0x0743] = "\x01\x59\x3e",
  [0x0744] = "\x01\x59\x3b", [0x0745] = "\x02\x34\x23",
  [0x0746] = "\x01\x59\x38", [0x0748] = "\x02\x34\x29",
  [0x0749] = "\x0e\x34\x5b", [0x074a] = "\x01\x59\x3f",
  [0x074b] = "\x02\x34\x2d", [0x074c] = "\x0e\x34\x58",
  [0x074d] = "\x02\x33\x77", [0x074e] = "\x02\x34\x27",
  [0x074f] = "\x01\x59\x42", [0x0750] = "\x02\x33\x78",
  [0x0751] = "\x02\x33\x7b", [0x0752] = "\x02\x34\x22",
  [0x0753] = "\x0e\x34\x62", [0x0754] = "\x0e\x34\x61",
  [0x0755] = "\x01\x59\x43", [0x0756] = "\x01\x59\x41",
  [0x0757] = "\x01\x59\x4d", [0x0758] = "\x0e\x34\x57",
  [0x0759] = "\x0e\x3a\x36", [0x075a] = "\x0e\x34\x64",
  [0x075c] = "\x01\x59\x48", [0x075d] = "\x0e\x34\x5e",
  [0x075e] = "\x01\x59\x3c", [0x075f] = "\x01\x5a\x76",
  [0x0761] = "\x01\x59\x3d", [0x0762] = "\x02\x33\x7c",
  [0x0763] = "\x01\x59\x4a", [0x0764] = "\x01\x59\x45",
  [0x0765] = "\x02\x34\x26", [0x0766] = "\x01\x59\x3a",
  [0x0768] = "\x0e\x34\x59", [0x076a] = "\x01\x59\x39",
  [0x076b] = "\x0e\x64\x57", [0x0771] = "\x0e\x64\x56",
  [0x0772] = "\x0e\x67\x22", [0x0773] = "\x0e\x64\x55",
  [0x0775] = "\x02\x33\x74", [0x0776] = "\x02\x33\x75",
  [0x0777] = "\x02\x3a\x4e", [0x077a] = "\x0e\x3a\x3c",
  [0x077b] = "\x01\x5e\x3d", [0x077c] = "\x01\x5e\x40",
  [0x077d] = "\x02\x3a\x59", [0x077e] = "\x01\x5e\x54",
  [0x077f] = "\x02\x3a\x5c", [0x0780] = "\x01\x5e\x3e",
  [0x0781] = "\x02\x3a\x55", [0x0782] = "\x01\x5e\x44",
  [0x0783] = "\x01\x5e\x4a", [0x0784] = "\x01\x61\x43",
  [0x0785] = "\x0e\x3a\x31", [0x0786] = "\x0e\x3a\x38",
  [0x0787] = "\x01\x5e\x48", [0x0788] = "\x02\x3a\x52",
  [0x0789] = "\x01\x5e\x55", [0x078a] = "\x01\x5e\x41",
  [0x078b] = "\x01\x5e\x49", [0x078c] = "\x02\x3a\x5a",
  [0x078d] = "\x02\x41\x7c", [0x078e] = "\x02\x3a\x5f",
  [0x078f] = "\x02\x3a\x53", [0x0791] = "\x02\x3a\x4a",
  [0x0792] = "\x02\x3a\x57", [0x0793] = "\x02\x3a\x51",
  [0x0794] = "\x01\x5e\x47", [0x0795] = "\x02\x3a\x5d",
  [0x0796] = "\x0e\x3a\x2e", [0x0797] = "\x0e\x3a\x2a",
  [0x0798] = "\x01\x5e\x43", [0x0799] = "\x01\x5e\x57",
  [0x079a] = "\x01\x5e\x50", [0x079b] = "\x0e\x3a\x33",
  [0x079c] = "\x01\x5e\x45", [0x079d] = "\x01\x5e\x42",
  [0x079e] = "\x0e\x3a\x40", [0x079f] = "\x01\x5e\x4d",
  [0x07a0] = "\x0e\x3a\x34", [0x07a1] = "\x02\x3a\x5e",
  [0x07a2] = "\x02\x3a\x50", [0x07a3] = "\x02\x3a\x56",
  [0x07a4] = "\x02\x3a\x58", [0x07a5] = "\x02\x3a\x4c",
  [0x07a6] = "\x02\x3a\x5b", [0x07a7] = "\x01\x5e\x3f",
  [0x07a8] = "\x02\x3a\x4b", [0x07a9] = "\x0e\x3a\x42",
  [0x07aa] = "\x01\x5e\x46", [0x07ab] = "\x01\x5e\x56",
  [0x07ac] = "\x01\x5e\x52", [0x07ad] = "\x02\x3a\x4d",
  [0x07ae] = "\x01\x5e\x4c", [0x07af] = "\x0e\x3a\x3b",
  [0x07b1] = "\x01\x5e\x53", [0x07b2] = "\x01\x5e\x4f",
  [0x07b3] = "\x01\x5e\x4b", [0x07b4] = "\x0e\x3a\x2f",
  [0x07b5] = "\x02\x3a\x54", [0x07ba] = "\x0e\x67\x23",
  [0x07bb] = "\x01\x5e\x51", [0x07bf] = "\x02\x41\x7a",
  [0x07c0] = "\x02\x41\x76", [0x07c1] = "\x0e\x40\x51",
  [0x07c2] = "\x02\x42\x27", [0x07c3] = "\x02\x41\x6f",
  [0x07c4] = "\x02\x41\x78", [0x07c5] = "\x01\x63\x4c",
  [0x07c6] = "\x01\x63\x4d", [0x07c7] = "\x01\x63\x45",
  [0x07c8] = "\x02\x42\x23", [0x07c9] = "\x01\x63\x4f",
  [0x07ca] = "\x02\x41\x74", [0x07cb] = "\x02\x41\x73",
  [0x07cc] = "\x02\x41\x71", [0x07cd] = "\x02\x42\x25",
  [0x07ce] = "\x01\x63\x43", [0x07cf] = "\x02\x41\x7d",
  [0x07d0] = "\x02\x41\x72", [0x07d1] = "\x01\x63\x46",
  [0x07d2] = "\x02\x41\x7b", [0x07d3] = "\x01\x63\x41",
  [0x07d4] = "\x02\x41\x77", [0x07d5] = "\x02\x41\x7e",
  [0x07d6] = "\x02\x42\x22", [0x07d7] = "\x0e\x40\x4e",
  [0x07d9] = "\x02\x42\x26", [0x07da] = "\x01\x63\x4a",
  [0x07db] = "\x02\x41\x70", [0x07dc] = "\x01\x63\x44",
  [0x07dd] = "\x02\x41\x75", [0x07de] = "\x0e\x3a\x2b",
  [0x07df] = "\x01\x63\x3f", [0x07e1] = "\x01\x63\x4b",
  [0x07e2] = "\x02\x42\x21", [0x07e3] = "\x01\x63\x47",
  [0x07e4] = "\x01\x63\x48", [0x07e5] = "\x01\x63\x4e",
  [0x07e6] = "\x01\x63\x42", [0x07e7] = "\x01\x23\x21",
  [0x07e8] = "\x01\x63\x40", [0x07e9] = "\x02\x41\x79",
  [0x07ea] = "\x0e\x40\x4c", [0x07ee] = "\x0e\x67\x25",
  [0x07ef] = "\x01\x63\x49", [0x07f0] = "\x0e\x67\x24",
  [0x07f2] = "\x02\x42\x24", [0x07f4] = "\x0e\x64\x58",
  [0x07f6] = "\x01\x68\x35", [0x07f7] = "\x01\x68\x30",
  [0x07f8] = "\x0e\x46\x2e", [0x07f9] = "\x02\x49\x3c",
  [0x07fa] = "\x02\x49\x38", [0x07fb] = "\x0e\x46\x28",
  [0x07fc] = "\x02\x49\x32", [0x07fd] = "\x01\x68\x2a",
  [0x07fe] = "\x01\x68\x26", [0x07ff] = "\x02\x49\x3b",
  [0x0800] = "\x01\x68\x27", [0x0801] = "\x02\x49\x35",
  [0x0802] = "\x02\x49\x37", [0x0803] = "\x0e\x46\x29",
  [0x0804] = "\x02\x49\x3a", [0x0805] = "\x0e\x46\x36",
  [0x0806] = "\x01\x68\x2c", [0x0807] = "\x0e\x46\x2c",
  [0x0808] = "\x01\x68\x33", [0x0809] = "\x01\x68\x2d",
  [0x080b] = "\x0e\x46\x2a", [0x080c] = "\x02\x49\x30",
  [0x080d] = "\x01\x68\x2e", [0x080e] = "\x01\x68\x2f",
  [0x080f] = "\x02\x49\x33", [0x0810] = "\x01\x68\x34",
  [0x0811] = "\x0e\x46\x34", [0x0812] = "\x02\x49\x31",
  [0x0813] = "\x02\x49\x36", [0x0814] = "\x01\x68\x2b",
  [0x0815] = "\x02\x49\x2f", [0x0816] = "\x01\x68\x31",
  [0x0817] = "\x01\x68\x29", [0x0818] = "\x0e\x46\x33",
  [0x081b] = "\x01\x68\x28", [0x081c] = "\x02\x49\x34",
  [0x081d] = "\x02\x49\x39", [0x081e] = "\x0e\x46\x31",
  [0x081f] = "\x01\x68\x32", [0x0820] = "\x0e\x64\x5a",
  [0x0822] = "\x0e\x64\x59", [0x0826] = "\x0e\x46\x50",
  [0x0827] = "\x02\x49\x2e", [0x0828] = "\x0e\x46\x30",
  [0x0829] = "\x01\x6c\x37", [0x082a] = "\x02\x4f\x6e",
  [0x082b] = "\x0e\x4b\x41", [0x082c] = "\x02\x4f\x6b",
  [0x082d] = "\x0e\x4b\x38", [0x082e] = "\x01\x6c\x31",
  [0x082f] = "\x01\x6c\x3d", [0x0830] = "\x01\x6c\x3e",
  [0x0831] = "\x0e\x4b\x3d", [0x0832] = "\x01\x6c\x34",
  [0x0833] = "\x02\x4f\x69", [0x0834] = "\x01\x6c\x36",
  [0x0835] = "\x02\x4f\x61", [0x0836] = "\x01\x6c\x3c",
  [0x0837] = "\x0e\x4b\x44", [0x0838] = "\x02\x4f\x6d",
  [0x0839] = "\x01\x6c\x33", [0x083a] = "\x02\x4f\x6f",
  [0x083b] = "\x01\x6c\x32", [0x083d] = "\x02\x4f\x6a",
  [0x083e] = "\x02\x4f\x6c", [0x083f] = "\x01\x6c\x35",
  [0x0840] = "\x02\x4f\x68", [0x0841] = "\x02\x4f\x62",
  [0x0842] = "\x02\x4f\x5f", [0x0844] = "\x0e\x4b\x3e",
  [0x0845] = "\x02\x3a\x4f", [0x0846] = "\x02\x4f\x65",
  [0x0847] = "\x0e\x4b\x37", [0x0848] = "\x02\x4f\x5e",
  [0x0849] = "\x02\x4f\x64", [0x084a] = "\x02\x4f\x63",
  [0x084b] = "\x0e\x4b\x3b", [0x084c] = "\x02\x4f\x60",
  [0x084d] = "\x0e\x4b\x43", [0x084e] = "\x01\x6c\x39",
  [0x084f] = "\x0e\x4b\x45", [0x0850] = "\x0e\x4b\x40",
  [0x0851] = "\x0e\x46\x35", [0x0852] = "\x0e\x4b\x3c",
  [0x0853] = "\x01\x6c\x38", [0x0854] = "\x0e\x4b\x39",
  [0x0856] = "\x0e\x4b\x3f", [0x0857] = "\x01\x6c\x3a",
  [0x0858] = "\x02\x4f\x66", [0x0859] = "\x01\x70\x3b",
  [0x085a] = "\x02\x4f\x67", [0x085e] = "\x02\x56\x4f",
  [0x085f] = "\x0e\x50\x3f", [0x0860] = "\x02\x56\x48",
  [0x0861] = "\x0e\x50\x40", [0x0862] = "\x01\x70\x47",
  [0x0863] = "\x02\x56\x4c", [0x0864] = "\x01\x70\x3f",
  [0x0865] = "\x01\x70\x43", [0x0866] = "\x02\x56\x4b",
  [0x0867] = "\x0e\x50\x3b", [0x0868] = "\x01\x70\x42",
  [0x0869] = "\x01\x70\x3e", [0x086a] = "\x01\x70\x41",
  [0x086b] = "\x01\x70\x3c", [0x086c] = "\x01\x70\x46",
  [0x086d] = "\x02\x56\x4d", [0x086e] = "\x02\x56\x49",
  [0x086f] = "\x01\x70\x45", [0x0870] = "\x02\x56\x47",
  [0x0871] = "\x01\x70\x44", [0x0872] = "\x02\x56\x4e",
  [0x0873] = "\x02\x56\x4a", [0x0874] = "\x01\x6c\x3b",
  [0x0876] = "\x01\x70\x48", [0x0877] = "\x02\x56\x50",
  [0x0878] = "\x01\x70\x40", [0x0879] = "\x01\x70\x3d",
  [0x087b] = "\x0e\x50\x3d", [0x087d] = "\x0e\x54\x48",
  [0x087e] = "\x02\x5c\x52", [0x087f] = "\x02\x5c\x54",
  [0x0880] = "\x01\x73\x46", [0x0881] = "\x02\x5c\x55",
  [0x0882] = "\x02\x5c\x53", [0x0883] = "\x02\x5c\x51",
  [0x0884] = "\x02\x5c\x50", [0x0885] = "\x01\x73\x48",
  [0x0886] = "\x02\x5c\x4f", [0x0887] = "\x01\x73\x49",
  [0x0889] = "\x0e\x54\x49", [0x088a] = "\x0e\x54\x47",
  [0x088c] = "\x02\x5c\x4d", [0x088d] = "\x02\x5c\x4e",
  [0x088e] = "\x01\x73\x45", [0x088f] = "\x01\x73\x4a",
  [0x0890] = "\x01\x73\x47", [0x0891] = "\x0e\x64\x5d",
  [0x0892] = "\x0e\x64\x5b", [0x0893] = "\x02\x5c\x4c",
  [0x0894] = "\x0e\x57\x5b", [0x0895] = "\x01\x76\x3c",
  [0x0896] = "\x0e\x57\x5a", [0x0897] = "\x02\x61\x48",
  [0x0898] = "\x02\x61\x46", [0x0899] = "\x02\x61\x4b",
  [0x089a] = "\x02\x61\x49", [0x089b] = "\x0e\x57\x58",
  [0x089c] = "\x02\x61\x47", [0x089d] = "\x02\x61\x4a",
  [0x089e] = "\x0e\x57\x59", [0x089f] = "\x0e\x57\x57",
  [0x08a0] = "\x0e\x57\x5c", [0x08a5] = "\x01\x78\x2c",
  [0x08a6] = "\x02\x65\x3e", [0x08a7] = "\x02\x65\x3f",
  [0x08a8] = "\x01\x78\x2d", [0x08aa] = "\x02\x65\x40",
  [0x08ab] = "\x02\x65\x3c", [0x08ac] = "\x02\x65\x41",
  [0x08ad] = "\x02\x65\x3d", [0x08ae] = "\x01\x76\x3d",
  [0x08af] = "\x0e\x5a\x2d", [0x08b0] = "\x0e\x64\x5e",
  [0x08b2] = "\x02\x68\x6b", [0x08b3] = "\x02\x68\x6d",
  [0x08b4] = "\x01\x79\x64", [0x08b5] = "\x02\x68\x6c",
  [0x08b6] = "\x01\x79\x63", [0x08b7] = "\x01\x79\x62",
  [0x08b8] = "\x0e\x67\x26", [0x08b9] = "\x0e\x67\x27",
  [0x08ba] = "\x0e\x64\x5c", [0x08bb] = "\x0e\x5e\x26",
  [0x08bc] = "\x01\x79\x65", [0x08bd] = "\x02\x6b\x53",
  [0x08be] = "\x02\x6b\x54", [0x08c0] = "\x01\x7a\x65",
  [0x08c1] = "\x01\x7a\x64", [0x08c2] = "\x01\x7a\x66",
  [0x08c3] = "\x02\x6b\x52", [0x08c4] = "\x0e\x67\x28",
  [0x08c5] = "\x02\x6d\x46", [0x08c6] = "\x02\x6d\x45",
  [0x08c7] = "\x0e\x5f\x40", [0x08c8] = "\x01\x7b\x59",
  [0x08c9] = "\x01\x7b\x5b", [0x08ca] = "\x01\x7b\x5a",
  [0x08cb] = "\x02\x6d\x47", [0x08cc] = "\x01\x7c\x34",
  [0x08cd] = "\x02\x70\x34", [0x08d1] = "\x01\x7c\x5d",
  [0x08d2] = "\x0e\x60\x71", [0x08d3] = "\x02\x70\x33",
  [0x08d4] = "\x02\x70\x7c", [0x08d7] = "\x02\x21\x2a",
  [0x08d8] = "\x0e\x21\x77", [0x08da] = "\x01\x46\x3f",
  [0x08db] = "\x01\x46\x3e", [0x08dd] = "\x01\x47\x60",
  [0x08de] = "\x01\x47\x5f", [0x08df] = "\x02\x22\x27",
  [0x08e0] = "\x01\x47\x5e", [0x08e1] = "\x02\x22\x26",
  [0x08e3] = "\x0e\x23\x3e", [0x08e4] = "\x01\x49\x5b",
  [0x08e5] = "\x02\x23\x49", [0x08e6] = "\x0e\x25\x2c",
  [0x08e7] = "\x02\x23\x48", [0x08e9] = "\x0e\x25\x26",
  [0x08ea] = "\x01\x49\x59", [0x08eb] = "\x01\x49\x5c",
  [0x08ec] = "\x0e\x25\x2b", [0x08ed] = "\x0e\x25\x2a",
  [0x08ee] = "\x02\x23\x47", [0x08ef] = "\x0e\x25\x29",
  [0x08f0] = "\x01\x49\x5a", [0x08f1] = "\x0e\x25\x48",
  [0x08f2] = "\x0e\x25\x27", [0x08f3] = "\x0e\x25\x28",
  [0x08f6] = "\x0e\x28\x22", [0x08f7] = "\x02\x25\x70",
  [0x08f8] = "\x0e\x27\x7d", [0x08f9] = "\x02\x25\x71",
  [0x08fa] = "\x01\x4c\x56", [0x08fb] = "\x0e\x27\x7c",
  [0x08fc] = "\x0e\x28\x21", [0x08fd] = "\x0e\x27\x7b",
  [0x08ff] = "\x01\x50\x4c", [0x0900] = "\x0e\x2b\x65",
  [0x0901] = "\x02\x2e\x4a", [0x0902] = "\x02\x2e\x4b",
  [0x0903] = "\x01\x54\x67", [0x0904] = "\x01\x54\x68",
  [0x0905] = "\x0e\x30\x27", [0x0907] = "\x02\x34\x2f",
  [0x0908] = "\x01\x59\x4e", [0x0909] = "\x01\x59\x50",
  [0x090a] = "\x02\x34\x2e", [0x090b] = "\x01\x59\x4f",
  [0x090c] = "\x02\x3a\x60", [0x090d] = "\x01\x5e\x58",
  [0x090e] = "\x0e\x3a\x44", [0x090f] = "\x0e\x3a\x43",
  [0x0912] = "\x01\x63\x50", [0x0913] = "\x01\x63\x51",
  [0x0914] = "\x02\x42\x28", [0x0915] = "\x0e\x40\x52",
  [0x0916] = "\x01\x68\x37", [0x0917] = "\x0e\x46\x37",
  [0x0918] = "\x01\x68\x36", [0x091a] = "\x02\x4f\x70",
  [0x091b] = "\x02\x56\x52", [0x091c] = "\x02\x56\x51",
  [0x091d] = "\x0e\x5f\x41", [0x091e] = "\x02\x71\x5a",
  [0x091f] = "\x01\x44\x48", [0x0920] = "\x02\x21\x39",
  [0x0921] = "\x0e\x21\x61", [0x0922] = "\x02\x21\x4f",
  [0x0923] = "\x02\x21\x50", [0x0924] = "\x0e\x22\x34",
  [0x0927] = "\x0e\x62\x49", [0x0928] = "\x01\x47\x63",
  [0x0929] = "\x01\x47\x67", [0x092a] = "\x02\x22\x29",
  [0x092c] = "\x01\x47\x65", [0x092d] = "\x01\x47\x64",
  [0x092e] = "\x02\x22\x28", [0x092f] = "\x01\x47\x66",
  [0x0930] = "\x01\x47\x62", [0x0931] = "\x0e\x23\x41",
  [0x0932] = "\x0e\x23\x42", [0x0933] = "\x01\x47\x61",
  [0x0934] = "\x02\x22\x2a", [0x0935] = "\x0e\x23\x3f",
  [0x0936] = "\x0e\x23\x40", [0x093b] = "\x01\x49\x66",
  [0x093c] = "\x0e\x25\x31", [0x093d] = "\x0e\x25\x3e",
  [0x093e] = "\x01\x49\x63", [0x093f] = "\x0e\x25\x30",
  [0x0940] = "\x01\x49\x5f", [0x0941] = "\x02\x23\x4a",
  [0x0942] = "\x0e\x25\x40", [0x0943] = "\x0e\x25\x36",
  [0x0944] = "\x0e\x25\x2f", [0x0945] = "\x02\x23\x4b",
  [0x0946] = "\x0e\x25\x3c", [0x0947] = "\x01\x49\x61",
  [0x0948] = "\x0e\x25\x2d", [0x0949] = "\x02\x23\x4d",
  [0x094a] = "\x01\x49\x5d", [0x094b] = "\x02\x23\x4e",
  [0x094c] = "\x02\x23\x4c", [0x094d] = "\x01\x49\x60",
  [0x094e] = "\x01\x49\x62", [0x094f] = "\x01\x49\x65",
  [0x0950] = "\x01\x49\x64", [0x0951] = "\x01\x49\x5e",
  [0x0952] = "\x02\x23\x4f", [0x0953] = "\x0e\x25\x32",
  [0x0954] = "\x0e\x25\x41", [0x0955] = "\x0e\x25\x42",
  [0x0956] = "\x0e\x25\x33", [0x0957] = "\x0e\x25\x38",
  [0x0958] = "\x0e\x25\x37", [0x0959] = "\x0e\x64\x5f",
  [0x095f] = "\x0e\x25\x35", [0x0961] = "\x01\x4c\x5b",
  [0x0962] = "\x02\x25\x7e", [0x0963] = "\x0e\x28\x25",
  [0x0964] = "\x01\x4c\x5d", [0x0965] = "\x0e\x28\x23",
  [0x0966] = "\x01\x4c\x5c", [0x0967] = "\x0e\x28\x26",
  [0x0968] = "\x02\x26\x21", [0x0969] = "\x01\x4c\x5a",
  [0x096a] = "\x01\x4c\x59", [0x096b] = "\x02\x25\x75",
  [0x096d] = "\x02\x25\x74", [0x096e] = "\x0e\x28\x28",
  [0x096f] = "\x02\x25\x72", [0x0970] = "\x02\x25\x77",
  [0x0971] = "\x02\x25\x76", [0x0972] = "\x02\x25\x73",
  [0x0973] = "\x02\x25\x7c", [0x0974] = "\x02\x25\x7d",
  [0x0975] = "\x02\x25\x7a", [0x0976] = "\x02\x25\x78",
  [0x0977] = "\x01\x4c\x58", [0x0978] = "\x0e\x28\x29",
  [0x097b] = "\x02\x25\x7b", [0x097c] = "\x01\x4c\x5e",
  [0x097d] = "\x02\x26\x22", [0x097f] = "\x0e\x28\x2a",
  [0x0980] = "\x02\x25\x79", [0x0981] = "\x0e\x28\x24",
  [0x0982] = "\x01\x50\x4d", [0x0983] = "\x01\x4c\x57",
  [0x098a] = "\x0e\x64\x60", [0x098b] = "\x01\x50\x4e",
  [0x098c] = "\x02\x29\x55", [0x098d] = "\x0e\x2b\x69",
  [0x098e] = "\x0e\x2b\x68", [0x098f] = "\x02\x29\x5b",
  [0x0990] = "\x0e\x2b\x6c", [0x0991] = "\x0e\x2b\x67",
  [0x0992] = "\x0e\x2b\x6a", [0x0993] = "\x01\x50\x54",
  [0x0994] = "\x02\x29\x59", [0x0995] = "\x02\x29\x5f",
  [0x0996] = "\x0e\x2b\x71", [0x0997] = "\x02\x29\x56",
  [0x0998] = "\x02\x29\x5a", [0x0999] = "\x02\x29\x5c",
  [0x099a] = "\x02\x29\x5e", [0x099b] = "\x02\x29\x58",
  [0x099c] = "\x0e\x2b\x70", [0x099d] = "\x02\x29\x57",
  [0x099e] = "\x02\x29\x52", [0x099f] = "\x02\x29\x53",
  [0x09a0] = "\x01\x50\x4f", [0x09a1] = "\x0e\x2b\x6b",
  [0x09a2] = "\x01\x50\x51", [0x09a3] = "\x01\x50\x50",
  [0x09a4] = "\x02\x29\x54", [0x09a5] = "\x02\x29\x5d",
  [0x09a6] = "\x0e\x2b\x6d", [0x09a7] = "\x0e\x2b\x72",
  [0x09a8] = "\x0e\x2b\x6e", [0x09ac] = "\x0e\x2b\x66",
  [0x09ae] = "\x01\x50\x53", [0x09b5] = "\x02\x29\x51",
  [0x09b6] = "\x02\x2e\x55", [0x09b7] = "\x0e\x30\x2a",
  [0x09b8] = "\x02\x2e\x54", [0x09b9] = "\x02\x2e\x59",
  [0x09ba] = "\x02\x2e\x50", [0x09bb] = "\x0e\x30\x30",
  [0x09bc] = "\x02\x2e\x53", [0x09bd] = "\x02\x2e\x52",
  [0x09be] = "\x0e\x30\x2f", [0x09bf] = "\x02\x2e\x56",
  [0x09c0] = "\x0e\x30\x38", [0x09c1] = "\x02\x2e\x5a",
  [0x09c2] = "\x01\x54\x69", [0x09c3] = "\x01\x54\x6c",
  [0x09c4] = "\x0e\x30\x34", [0x09c5] = "\x0e\x30\x35",
  [0x09c6] = "\x02\x2e\x51", [0x09c7] = "\x02\x2e\x57",
  [0x09c8] = "\x0e\x30\x33", [0x09c9] = "\x0e\x30\x28",
  [0x09ca] = "\x0e\x30\x37", [0x09cb] = "\x01\x54\x6b",
  [0x09cc] = "\x02\x2e\x4c", [0x09cd] = "\x0e\x30\x2b",
  [0x09ce] = "\x01\x50\x52", [0x09cf] = "\x02\x34\x3d",
  [0x09d0] = "\x02\x2e\x58", [0x09d1] = "\x0e\x30\x36",
  [0x09d2] = "\x02\x2e\x4f", [0x09d4] = "\x01\x54\x6a",
  [0x09d5] = "\x02\x2e\x4e", [0x09db] = "\x0e\x30\x32",
  [0x09dc] = "\x02\x34\x34", [0x09dd] = "\x0e\x34\x6e",
  [0x09de] = "\x0e\x34\x71", [0x09df] = "\x01\x59\x51",
  [0x09e0] = "\x01\x59\x55", [0x09e1] = "\x02\x34\x44",
  [0x09e2] = "\x02\x34\x32", [0x09e3] = "\x02\x34\x40",
  [0x09e4] = "\x01\x59\x56", [0x09e5] = "\x02\x34\x42",
  [0x09e6] = "\x0e\x34\x72", [0x09e7] = "\x02\x34\x48",
  [0x09e8] = "\x0e\x34\x74", [0x09e9] = "\x02\x34\x4c",
  [0x09ea] = "\x0e\x34\x69", [0x09ec] = "\x02\x34\x43",
  [0x09ed] = "\x02\x34\x37", [0x09ee] = "\x02\x34\x3f",
  [0x09ef] = "\x0e\x34\x6b", [0x09f0] = "\x02\x34\x4d",
  [0x09f1] = "\x02\x34\x4b", [0x09f2] = "\x02\x34\x41",
  [0x09f3] = "\x02\x34\x3c", [0x09f4] = "\x02\x34\x35",
  [0x09f5] = "\x02\x3a\x68", [0x09f6] = "\x02\x34\x33",
  [0x09f7] = "\x01\x59\x5a", [0x09f8] = "\x02\x34\x3a",
  [0x09f9] = "\x01\x59\x5b", [0x09fa] = "\x01\x59\x57",
  [0x09fb] = "\x02\x34\x30", [0x09fc] = "\x02\x34\x46",
  [0x09fd] = "\x02\x34\x38", [0x09fe] = "\x0e\x34\x6f",
  [0x09ff] = "\x0e\x34\x75", [0x0a00] = "\x02\x34\x36",
  [0x0a01] = "\x02\x34\x49", [0x0a02] = "\x01\x59\x58",
  [0x0a03] = "\x0e\x34\x76", [0x0a04] = "\x02\x34\x4f",
  [0x0a05] = "\x01\x59\x52", [0x0a06] = "\x01\x59\x54",
  [0x0a07] = "\x02\x34\x3e", [0x0a08] = "\x02\x34\x39",
  [0x0a09] = "\x01\x54\x6d", [0x0a0a] = "\x01\x59\x53",
  [0x0a0b] = "\x02\x34\x3b", [0x0a0c] = "\x02\x34\x4a",
  [0x0a0d] = "\x02\x34\x4e", [0x0a0e] = "\x02\x34\x45",
  [0x0a0f] = "\x0e\x64\x61", [0x0a10] = "\x02\x34\x47",
  [0x0a12] = "\x0e\x64\x62", [0x0a13] = "\x0e\x64\x63",
  [0x0a14] = "\x02\x34\x31", [0x0a18] = "\x0e\x3a\x4a",
  [0x0a19] = "\x02\x3a\x63", [0x0a1a] = "\x0e\x3a\x45",
  [0x0a1b] = "\x02\x3a\x6c", [0x0a1c] = "\x02\x3a\x6b",
  [0x0a1d] = "\x01\x5e\x60", [0x0a1e] = "\x02\x3a\x64",
  [0x0a1f] = "\x0e\x3a\x47", [0x0a20] = "\x01\x5e\x61",
  [0x0a21] = "\x01\x5e\x5f", [0x0a22] = "\x0e\x3a\x50",
  [0x0a23] = "\x02\x3a\x66", [0x0a24] = "\x01\x5e\x5c",
  [0x0a25] = "\x02\x3a\x6a", [0x0a26] = "\x0e\x3a\x4c",
  [0x0a27] = "\x02\x3a\x65", [0x0a28] = "\x02\x3a\x67",
  [0x0a29] = "\x02\x3a\x61", [0x0a2a] = "\x01\x5e\x5a",
  [0x0a2b] = "\x0e\x63\x36", [0x0a2c] = "\x02\x3a\x74",
  [0x0a2d] = "\x02\x3a\x73", [0x0a2e] = "\x02\x3a\x70",
  [0x0a2f] = "\x01\x5e\x59", [0x0a30] = "\x01\x5e\x5d",
  [0x0a31] = "\x01\x5e\x5e", [0x0a32] = "\x02\x2e\x4d",
  [0x0a33] = "\x02\x3a\x6d", [0x0a34] = "\x01\x5e\x5b",
  [0x0a35] = "\x01\x59\x59", [0x0a36] = "\x02\x3a\x6f",
  [0x0a37] = "\x02\x3a\x62", [0x0a38] = "\x02\x3a\x72",
  [0x0a39] = "\x02\x3a\x71", [0x0a3a] = "\x0e\x3a\x4e",
  [0x0a3b] = "\x02\x3a\x75", [0x0a3c] = "\x0e\x3a\x49",
  [0x0a3d] = "\x02\x42\x36", [0x0a3e] = "\x0e\x3a\x4b",
  [0x0a3f] = "\x02\x3a\x6e", [0x0a45] = "\x0e\x63\x37",
  [0x0a47] = "\x0e\x64\x64", [0x0a48] = "\x02\x3a\x69",
  [0x0a49] = "\x02\x42\x2e", [0x0a4a] = "\x01\x63\x5b",
  [0x0a4b] = "\x01\x63\x5e", [0x0a4c] = "\x01\x63\x59",
  [0x0a4d] = "\x02\x42\x2d", [0x0a4e] = "\x02\x42\x31",
  [0x0a4f] = "\x02\x42\x2c", [0x0a50] = "\x0e\x40\x57",
  [0x0a51] = "\x01\x63\x53", [0x0a52] = "\x01\x63\x5d",
  [0x0a53] = "\x02\x42\x29", [0x0a54] = "\x01\x63\x57",
  [0x0a55] = "\x02\x42\x30", [0x0a56] = "\x0e\x40\x5b",
  [0x0a57] = "\x01\x63\x55", [0x0a58] = "\x01\x63\x54",
  [0x0a59] = "\x02\x42\x33", [0x0a5a] = "\x01\x63\x56",
  [0x0a5b] = "\x02\x42\x35", [0x0a5c] = "\x0e\x40\x56",
  [0x0a5d] = "\x02\x42\x32", [0x0a5e] = "\x01\x63\x52",
  [0x0a5f] = "\x0e\x40\x5e", [0x0a60] = "\x0e\x40\x5c",
  [0x0a61] = "\x0e\x40\x58", [0x0a62] = "\x01\x63\x5c",
  [0x0a63] = "\x02\x42\x37", [0x0a64] = "\x02\x42\x2b",
  [0x0a65] = "\x02\x42\x34", [0x0a66] = "\x0e\x40\x5d",
  [0x0a67] = "\x0e\x63\x43", [0x0a68] = "\x02\x42\x2a",
  [0x0a69] = "\x0e\x40\x5a", [0x0a6a] = "\x0e\x40\x59",
  [0x0a6b] = "\x01\x63\x58", [0x0a6c] = "\x0e\x40\x54",
  [0x0a6d] = "\x01\x63\x5a", [0x0a6f] = "\x02\x42\x2f",
  [0x0a71] = "\x02\x42\x38", [0x0a72] = "\x0e\x46\x43",
  [0x0a73] = "\x0e\x46\x3e", [0x0a74] = "\x02\x49\x44",
  [0x0a75] = "\x01\x68\x38", [0x0a76] = "\x02\x49\x4a",
  [0x0a77] = "\x0e\x46\x42", [0x0a78] = "\x0e\x46\x3c",
  [0x0a79] = "\x01\x68\x3d", [0x0a7a] = "\x02\x49\x46",
  [0x0a7b] = "\x02\x49\x4d", [0x0a7c] = "\x02\x49\x3e",
  [0x0a7d] = "\x01\x68\x3f", [0x0a7e] = "\x01\x68\x39",
  [0x0a7f] = "\x02\x49\x43", [0x0a80] = "\x01\x6c\x3f",
  [0x0a81] = "\x02\x49\x42", [0x0a82] = "\x02\x49\x4b",
  [0x0a83] = "\x01\x68\x3a", [0x0a84] = "\x0e\x46\x3d",
  [0x0a85] = "\x01\x68\x3e", [0x0a86] = "\x02\x49\x41",
  [0x0a87] = "\x02\x49\x47", [0x0a88] = "\x02\x49\x4c",
  [0x0a89] = "\x02\x49\x3d", [0x0a8a] = "\x01\x68\x3c",
  [0x0a8b] = "\x02\x49\x45", [0x0a8c] = "\x0e\x46\x3a",
  [0x0a8d] = "\x0e\x46\x45", [0x0a8e] = "\x02\x49\x49",
  [0x0a8f] = "\x02\x49\x4f", [0x0a90] = "\x02\x49\x3f",
  [0x0a91] = "\x02\x49\x48", [0x0a92] = "\x0e\x46\x38",
  [0x0a93] = "\x01\x68\x3b", [0x0a94] = "\x02\x49\x4e",
  [0x0a95] = "\x0e\x46\x41", [0x0a96] = "\x0e\x46\x44",
  [0x0a97] = "\x0e\x46\x40", [0x0a98] = "\x02\x49\x40",
  [0x0a9b] = "\x0e\x64\x65", [0x0a9c] = "\x01\x6c\x43",
  [0x0a9d] = "\x02\x4f\x72", [0x0a9e] = "\x01\x6c\x41",
  [0x0a9f] = "\x01\x6c\x40", [0x0aa0] = "\x02\x4f\x74",
  [0x0aa1] = "\x02\x4f\x79", [0x0aa2] = "\x0e\x4b\x46",
  [0x0aa3] = "\x02\x4f\x75", [0x0aa4] = "\x0e\x4b\x50",
  [0x0aa5] = "\x02\x4f\x78", [0x0aa6] = "\x01\x6c\x46",
  [0x0aa7] = "\x0e\x4b\x51", [0x0aa8] = "\x01\x70\x2e",
  [0x0aa9] = "\x01\x6c\x45", [0x0aaa] = "\x0e\x4b\x4b",
  [0x0aab] = "\x02\x4f\x71", [0x0aac] = "\x02\x4f\x77",
  [0x0aad] = "\x0e\x46\x3f", [0x0aae] = "\x01\x6c\x44",
  [0x0aaf] = "\x02\x4f\x76", [0x0ab0] = "\x0e\x4b\x4d",
  [0x0ab1] = "\x02\x4f\x73", [0x0ab2] = "\x0e\x4b\x49",
  [0x0ab3] = "\x01\x6c\x42", [0x0ab4] = "\x0e\x4b\x4f",
  [0x0ab5] = "\x0e\x4b\x4c", [0x0ab7] = "\x0e\x4b\x47",
  [0x0aba] = "\x02\x56\x57", [0x0abb] = "\x0e\x50\x4a",
  [0x0abc] = "\x02\x56\x59", [0x0abd] = "\x02\x56\x54",
  [0x0abe] = "\x01\x70\x4a", [0x0abf] = "\x02\x56\x56",
  [0x0ac0] = "\x0e\x50\x4b", [0x0ac1] = "\x01\x70\x49",
  [0x0ac2] = "\x02\x56\x58", [0x0ac3] = "\x0e\x50\x48",
  [0x0ac4] = "\x0e\x50\x4c", [0x0ac5] = "\x01\x70\x4c",
  [0x0ac6] = "\x02\x56\x5a", [0x0ac7] = "\x01\x70\x4b",
  [0x0ac8] = "\x02\x56\x53", [0x0ac9] = "\x02\x56\x55",
  [0x0aca] = "\x0e\x50\x49", [0x0acd] = "\x0e\x54\x4b",
  [0x0ace] = "\x01\x73\x4e", [0x0acf] = "\x02\x5c\x58",
  [0x0ad0] = "\x0e\x54\x4c", [0x0ad1] = "\x01\x73\x4d",
  [0x0ad2] = "\x02\x5c\x59", [0x0ad3] = "\x01\x73\x4c",
  [0x0ad4] = "\x02\x5c\x57", [0x0ad5] = "\x01\x73\x4b",
  [0x0ad6] = "\x02\x5c\x56", [0x0ad8] = "\x01\x76\x3f",
  [0x0ad9] = "\x01\x76\x3e", [0x0ada] = "\x02\x65\x42",
  [0x0adb] = "\x02\x65\x44", [0x0adc] = "\x0e\x5a\x2f",
  [0x0add] = "\x02\x65\x43", [0x0ade] = "\x01\x78\x2e",
  [0x0adf] = "\x01\x78\x2f", [0x0ae0] = "\x0e\x5a\x2e",
  [0x0ae2] = "\x01\x78\x30", [0x0ae3] = "\x02\x68\x6e",
  [0x0ae4] = "\x01\x79\x66", [0x0ae6] = "\x0e\x5e\x28",
  [0x0ae7] = "\x02\x6f\x22", [0x0ae8] = "\x02\x6f\x21",
  [0x0ae9] = "\x01\x7c\x5e", [0x0aeb] = "\x01\x44\x49",
  [0x0aec] = "\x01\x45\x31", [0x0aee] = "\x0e\x23\x43",
  [0x0aef] = "\x01\x49\x67", [0x0af0] = "\x0e\x25\x45",
  [0x0af1] = "\x0e\x25\x43", [0x0af2] = "\x0e\x25\x44",
  [0x0af3] = "\x0e\x28\x2d", [0x0af4] = "\x02\x29\x60",
  [0x0af5] = "\x0e\x2b\x77", [0x0af7] = "\x0e\x34\x77",
  [0x0af9] = "\x01\x5e\x62", [0x0afa] = "\x01\x5e\x63",
  [0x0afb] = "\x0e\x3a\x52", [0x0afc] = "\x02\x42\x39",
  [0x0afd] = "\x01\x68\x40", [0x0afe] = "\x02\x49\x50",
  [0x0aff] = "\x02\x4f\x7a", [0x0b00] = "\x0e\x64\x66",
  [0x0b01] = "\x0e\x50\x4d", [0x0b02] = "\x0e\x21\x39",
  [0x0b03] = "\x02\x21\x3a", [0x0b04] = "\x0e\x22\x35",
  [0x0b05] = "\x0e\x23\x44", [0x0b06] = "\x02\x23\x50",
  [0x0b08] = "\x0e\x2b\x7a", [0x0b09] = "\x0e\x2b\x79",
  [0x0b0a] = "\x0e\x21\x3a", [0x0b0b] = "\x0e\x25\x46",
  [0x0b0c] = "\x02\x26\x23", [0x0b0d] = "\x02\x29\x61",
  [0x0b0e] = "\x02\x2e\x5b", [0x0b0f] = "\x01\x54\x6e",
  [0x0b10] = "\x0e\x46\x46", [0x0b11] = "\x0e\x65\x60",
  [0x0b12] = "\x02\x65\x45", [0x0b14] = "\x01\x7a\x67",
  [0x0b15] = "\x01\x44\x4a", [0x0b16] = "\x01\x46\x40",
  [0x0b17] = "\x02\x21\x51", [0x0b18] = "\x0e\x22\x36",
  [0x0b19] = "\x01\x47\x68", [0x0b1a] = "\x01\x47\x69",
  [0x0b1c] = "\x01\x4c\x5f", [0x0b1f] = "\x0e\x34\x79",
  [0x0b20] = "\x01\x59\x5c", [0x0b22] = "\x01\x68\x42",
  [0x0b23] = "\x0e\x40\x5f", [0x0b24] = "\x01\x68\x43",
  [0x0b25] = "\x01\x68\x41", [0x0b26] = "\x0e\x4b\x53",
  [0x0b27] = "\x01\x44\x4b", [0x0b29] = "\x01\x45\x32",
  [0x0b2a] = "\x01\x45\x34", [0x0b2b] = "\x01\x45\x33",
  [0x0b2c] = "\x02\x21\x3b", [0x0b2d] = "\x01\x45\x35",
  [0x0b2e] = "\x01\x46\x41", [0x0b2f] = "\x02\x21\x52",
  [0x0b30] = "\x0e\x22\x38", [0x0b31] = "\x01\x46\x42",
  [0x0b32] = "\x0e\x22\x37", [0x0b34] = "\x0e\x22\x39",
  [0x0b36] = "\x0e\x23\x47", [0x0b37] = "\x01\x47\x6a",
  [0x0b38] = "\x01\x47\x6b", [0x0b3c] = "\x02\x22\x2b",
  [0x0b3d] = "\x0e\x25\x49", [0x0b3e] = "\x01\x49\x68",
  [0x0b40] = "\x02\x23\x51", [0x0b44] = "\x01\x4c\x63",
  [0x0b45] = "\x02\x26\x24", [0x0b47] = "\x01\x4c\x61",
  [0x0b48] = "\x01\x4c\x62", [0x0b49] = "\x01\x4c\x60",
  [0x0b4a] = "\x02\x2e\x5c", [0x0b4b] = "\x0e\x28\x2e",
  [0x0b4c] = "\x0e\x28\x2f", [0x0b4d] = "\x0e\x64\x67",
  [0x0b4e] = "\x01\x50\x58", [0x0b4f] = "\x01\x50\x57",
  [0x0b50] = "\x01\x50\x59", [0x0b51] = "\x01\x50\x56",
  [0x0b52] = "\x0e\x2b\x7c", [0x0b53] = "\x02\x29\x62",
  [0x0b54] = "\x01\x4c\x64", [0x0b55] = "\x01\x50\x55",
  [0x0b57] = "\x01\x54\x6f", [0x0b58] = "\x01\x54\x70",
  [0x0b5a] = "\x01\x54\x71", [0x0b5b] = "\x0e\x34\x7b",
  [0x0b5c] = "\x02\x34\x50", [0x0b5d] = "\x0e\x34\x7c",
  [0x0b5e] = "\x0e\x34\x7d", [0x0b60] = "\x01\x5e\x64",
  [0x0b61] = "\x02\x3a\x76", [0x0b62] = "\x01\x59\x5d",
  [0x0b63] = "\x0e\x3a\x53", [0x0b64] = "\x0e\x3a\x54",
  [0x0b66] = "\x0e\x40\x60", [0x0b67] = "\x01\x63\x5f",
  [0x0b68] = "\x0e\x40\x61", [0x0b69] = "\x01\x68\x45",
  [0x0b6a] = "\x01\x68\x44", [0x0b6b] = "\x02\x49\x51",
  [0x0b6c] = "\x0e\x46\x48", [0x0b6d] = "\x01\x6c\x47",
  [0x0b6e] = "\x01\x70\x4d", [0x0b70] = "\x02\x61\x4c",
  [0x0b71] = "\x02\x6d\x48", [0x0b72] = "\x02\x6f\x23",
  [0x0b73] = "\x01\x44\x4c", [0x0b74] = "\x01\x46\x43",
  [0x0b75] = "\x0e\x22\x3b", [0x0b76] = "\x01\x46\x44",
  [0x0b77] = "\x02\x22\x31", [0x0b78] = "\x01\x47\x6d",
  [0x0b79] = "\x01\x47\x70", [0x0b7a] = "\x0e\x23\x4b",
  [0x0b7b] = "\x02\x22\x2f", [0x0b7c] = "\x02\x22\x2d",
  [0x0b7d] = "\x01\x47\x6f", [0x0b7e] = "\x02\x22\x30",
  [0x0b7f] = "\x02\x22\x32", [0x0b80] = "\x02\x22\x2c",
  [0x0b81] = "\x01\x47\x72", [0x0b82] = "\x01\x47\x71",
  [0x0b83] = "\x01\x47\x6e", [0x0b84] = "\x01\x47\x6c",
  [0x0b85] = "\x02\x22\x2e", [0x0b86] = "\x0e\x23\x4d",
  [0x0b89] = "\x0e\x25\x55", [0x0b8a] = "\x01\x49\x73",
  [0x0b8c] = "\x0e\x25\x4c", [0x0b8d] = "\x01\x49\x70",
  [0x0b8e] = "\x02\x23\x56", [0x0b8f] = "\x02\x23\x59",
  [0x0b90] = "\x02\x23\x58", [0x0b92] = "\x01\x49\x6a",
  [0x0b93] = "\x01\x49\x72", [0x0b94] = "\x0e\x25\x4b",
  [0x0b96] = "\x01\x49\x6f", [0x0b97] = "\x02\x23\x55",
  [0x0b98] = "\x02\x23\x53", [0x0b99] = "\x01\x49\x6e",
  [0x0b9a] = "\x0e\x25\x56", [0x0b9b] = "\x0e\x25\x53",
  [0x0b9c] = "\x0e\x25\x4e", [0x0b9d] = "\x01\x49\x69",
  [0x0b9e] = "\x01\x49\x6c", [0x0b9f] = "\x0e\x25\x51",
  [0x0ba0] = "\x02\x23\x54", [0x0ba1] = "\x02\x23\x5b",
  [0x0ba2] = "\x02\x23\x57", [0x0ba3] = "\x01\x49\x6d",
  [0x0ba4] = "\x01\x49\x71", [0x0ba5] = "\x01\x49\x74",
  [0x0ba6] = "\x02\x23\x52", [0x0ba7] = "\x02\x23\x5a",
  [0x0ba8] = "\x01\x49\x6b", [0x0bac] = "\x0e\x28\x41",
  [0x0bad] = "\x0e\x28\x33", [0x0bae] = "\x01\x4c\x69",
  [0x0baf] = "\x01\x4c\x71", [0x0bb0] = "\x0e\x28\x38",
  [0x0bb1] = "\x02\x26\x30", [0x0bb2] = "\x02\x26\x29",
  [0x0bb3] = "\x01\x4c\x72", [0x0bb4] = "\x02\x26\x34",
  [0x0bb5] = "\x02\x26\x25", [0x0bb6] = "\x02\x26\x2c",
  [0x0bb7] = "\x0e\x28\x45", [0x0bb8] = "\x0e\x28\x40",
  [0x0bb9] = "\x01\x4c\x68", [0x0bba] = "\x02\x26\x26",
  [0x0bbb] = "\x01\x4c\x66", [0x0bbc] = "\x02\x26\x2d",
  [0x0bbd] = "\x02\x26\x31", [0x0bbe] = "\x01\x4c\x65",
  [0x0bbf] = "\x0e\x28\x3d", [0x0bc0] = "\x02\x26\x32",
  [0x0bc1] = "\x02\x26\x2b", [0x0bc2] = "\x0e\x28\x37",
  [0x0bc3] = "\x02\x26\x2e", [0x0bc4] = "\x0e\x28\x35",
  [0x0bc5] = "\x01\x4c\x74", [0x0bc6] = "\x01\x4c\x6b",
  [0x0bc7] = "\x02\x26\x35", [0x0bc8] = "\x02\x26\x33",
  [0x0bc9] = "\x0e\x28\x3f", [0x0bca] = "\x01\x4c\x70",
  [0x0bcb] = "\x01\x4c\x6e", [0x0bcc] = "\x02\x26\x2a",
  [0x0bcd] = "\x01\x4c\x6d", [0x0bce] = "\x02\x26\x28",
  [0x0bcf] = "\x02\x26\x27", [0x0bd0] = "\x01\x4c\x6c",
  [0x0bd1] = "\x01\x4c\x6a", [0x0bd2] = "\x01\x4c\x73",
  [0x0bd3] = "\x01\x4c\x6f", [0x0bd4] = "\x01\x4c\x67",
  [0x0bd5] = "\x0e\x2c\x26", [0x0bd6] = "\x02\x26\x2f",
  [0x0bd7] = "\x0e\x28\x46", [0x0bd8] = "\x01\x50\x5b",
  [0x0bd9] = "\x0e\x2c\x2d", [0x0bda] = "\x01\x50\x62",
  [0x0bdb] = "\x02\x29\x70", [0x0bdc] = "\x01\x50\x5a",
  [0x0bdd] = "\x02\x29\x68", [0x0bde] = "\x02\x29\x64",
  [0x0bdf] = "\x0e\x2c\x21", [0x0be0] = "\x02\x29\x74",
  [0x0be1] = "\x02\x29\x63", [0x0be2] = "\x0e\x2c\x2c",
  [0x0be3] = "\x01\x50\x5d", [0x0be4] = "\x02\x29\x6d",
  [0x0be5] = "\x01\x50\x60", [0x0be6] = "\x01\x50\x63",
  [0x0be7] = "\x0e\x2c\x2e", [0x0be8] = "\x01\x50\x5e",
  [0x0be9] = "\x02\x29\x71", [0x0bea] = "\x01\x50\x61",
  [0x0beb] = "\x0e\x2c\x23", [0x0bec] = "\x01\x54\x78",
  [0x0bed] = "\x02\x29\x77", [0x0bee] = "\x02\x29\x65",
  [0x0bef] = "\x0e\x2c\x24", [0x0bf0] = "\x0e\x2c\x25",
  [0x0bf1] = "\x02\x29\x67", [0x0bf2] = "\x02\x29\x6e",
  [0x0bf3] = "\x02\x29\x72", [0x0bf4] = "\x02\x29\x76",
  [0x0bf5] = "\x02\x29\x73", [0x0bf6] = "\x02\x29\x6c",
  [0x0bf7] = "\x02\x29\x6f", [0x0bf8] = "\x0e\x2c\x2b",
  [0x0bf9] = "\x0e\x2c\x29", [0x0bfa] = "\x02\x29\x69",
  [0x0bfb] = "\x01\x50\x65", [0x0bfc] = "\x02\x29\x6b",
  [0x0bfd] = "\x02\x29\x6a", [0x0bfe] = "\x02\x29\x75",
  [0x0bff] = "\x01\x50\x5c", [0x0c00] = "\x02\x29\x66",
  [0x0c01] = "\x01\x50\x64", [0x0c02] = "\x0e\x2c\x2a",
  [0x0c03] = "\x01\x50\x5f", [0x0c04] = "\x0e\x2c\x28",
  [0x0c09] = "\x01\x54\x7e", [0x0c0a] = "\x02\x2e\x64",
  [0x0c0b] = "\x0e\x30\x40", [0x0c0c] = "\x01\x54\x7d",
  [0x0c0d] = "\x0e\x2c\x27", [0x0c0f] = "\x02\x2e\x62",
  [0x0c10] = "\x0e\x62\x70", [0x0c11] = "\x01\x54\x72",
  [0x0c12] = "\x0e\x30\x4f", [0x0c13] = "\x01\x54\x77",
  [0x0c14] = "\x0e\x30\x4a", [0x0c15] = "\x02\x2e\x61",
  [0x0c16] = "\x02\x2e\x5e", [0x0c17] = "\x02\x2e\x63",
  [0x0c18] = "\x01\x54\x73", [0x0c19] = "\x02\x2e\x5d",
  [0x0c1a] = "\x0e\x30\x4e", [0x0c1b] = "\x01\x54\x76",
  [0x0c1c] = "\x01\x54\x74", [0x0c1d] = "\x0e\x30\x3e",
  [0x0c1e] = "\x02\x2e\x65", [0x0c1f] = "\x01\x54\x75",
  [0x0c20] = "\x01\x54\x79", [0x0c21] = "\x0e\x30\x3b",
  [0x0c22] = "\x0e\x30\x45", [0x0c23] = "\x01\x54\x7a",
  [0x0c24] = "\x0e\x30\x48", [0x0c25] = "\x01\x54\x7c",
  [0x0c26] = "\x0e\x30\x52", [0x0c29] = "\x01\x54\x7b",
  [0x0c2a] = "\x0e\x30\x3c", [0x0c2b] = "\x0e\x35\x22",
  [0x0c2c] = "\x0e\x35\x38", [0x0c2d] = "\x02\x2e\x5f",
  [0x0c2e] = "\x02\x2e\x60", [0x0c2f] = "\x0e\x30\x4d",
  [0x0c31] = "\x0e\x30\x4b", [0x0c33] = "\x02\x2e\x66",
  [0x0c35] = "\x02\x34\x57", [0x0c36] = "\x01\x59\x5e",
  [0x0c37] = "\x02\x3b\x26", [0x0c38] = "\x02\x34\x56",
  [0x0c39] = "\x02\x34\x69", [0x0c3a] = "\x0e\x35\x28",
  [0x0c3b] = "\x0e\x35\x24", [0x0c3c] = "\x01\x59\x64",
  [0x0c3d] = "\x0e\x35\x31", [0x0c3e] = "\x02\x34\x67",
  [0x0c3f] = "\x0e\x35\x33", [0x0c40] = "\x01\x59\x63",
  [0x0c41] = "\x01\x59\x5f", [0x0c42] = "\x02\x34\x70",
  [0x0c43] = "\x02\x34\x60", [0x0c44] = "\x02\x34\x63",
  [0x0c45] = "\x0e\x35\x2e", [0x0c46] = "\x01\x59\x67",
  [0x0c47] = "\x02\x34\x6d", [0x0c48] = "\x02\x34\x65",
  [0x0c49] = "\x01\x59\x60", [0x0c4a] = "\x01\x59\x68",
  [0x0c4b] = "\x0e\x35\x2a", [0x0c4c] = "\x02\x34\x6a",
  [0x0c4d] = "\x02\x34\x68", [0x0c4e] = "\x0e\x35\x2f",
  [0x0c4f] = "\x0e\x35\x3b", [0x0c50] = "\x02\x34\x59",
  [0x0c51] = "\x02\x34\x6e", [0x0c52] = "\x02\x34\x62",
  [0x0c53] = "\x02\x34\x5d", [0x0c54] = "\x0e\x35\x3a",
  [0x0c55] = "\x02\x34\x53", [0x0c56] = "\x02\x34\x6f",
  [0x0c57] = "\x02\x34\x5f", [0x0c58] = "\x02\x34\x52",
  [0x0c59] = "\x0e\x35\x39", [0x0c5a] = "\x01\x59\x66",
  [0x0c5b] = "\x02\x34\x64", [0x0c5c] = "\x02\x34\x71",
  [0x0c5d] = "\x02\x34\x61", [0x0c5e] = "\x02\x34\x55",
  [0x0c5f] = "\x02\x34\x5a", [0x0c60] = "\x02\x34\x51",
  [0x0c61] = "\x0e\x35\x27", [0x0c62] = "\x01\x59\x65",
  [0x0c63] = "\x0e\x35\x3c", [0x0c64] = "\x02\x34\x5e",
  [0x0c65] = "\x02\x34\x5b", [0x0c66] = "\x01\x59\x61",
  [0x0c67] = "\x02\x34\x54", [0x0c68] = "\x0e\x35\x30",
  [0x0c69] = "\x02\x34\x6c", [0x0c6a] = "\x01\x59\x62",
  [0x0c6b] = "\x0e\x35\x2b", [0x0c6c] = "\x02\x34\x5c",
  [0x0c6d] = "\x02\x34\x58", [0x0c6e] = "\x0e\x35\x29",
  [0x0c6f] = "\x0e\x35\x34", [0x0c70] = "\x02\x34\x6b",
  [0x0c71] = "\x0e\x35\x32", [0x0c73] = "\x0e\x35\x37",
  [0x0c75] = "\x0e\x35\x35", [0x0c77] = "\x01\x5e\x65",
  [0x0c78] = "\x02\x3a\x7d", [0x0c79] = "\x0e\x3a\x75",
  [0x0c7a] = "\x02\x3a\x7a", [0x0c7b] = "\x02\x3b\x2d",
  [0x0c7c] = "\x02\x3b\x21", [0x0c7d] = "\x02\x3b\x2e",
  [0x0c7e] = "\x0e\x3a\x7c", [0x0c7f] = "\x01\x5e\x67",
  [0x0c80] = "\x0e\x3a\x63", [0x0c81] = "\x0e\x3a\x61",
  [0x0c82] = "\x0e\x3a\x58", [0x0c83] = "\x02\x3b\x2a",
  [0x0c84] = "\x02\x3b\x27", [0x0c85] = "\x0e\x3a\x5b",
  [0x0c86] = "\x0e\x3a\x77", [0x0c87] = "\x0e\x3a\x72",
  [0x0c88] = "\x0e\x3a\x59", [0x0c89] = "\x0e\x3a\x60",
  [0x0c8a] = "\x02\x3b\x28", [0x0c8b] = "\x02\x3b\x2b",
  [0x0c8c] = "\x02\x3b\x2f", [0x0c8d] = "\x0e\x3a\x7b",
  [0x0c8e] = "\x02\x34\x66", [0x0c8f] = "\x02\x3b\x31",
  [0x0c90] = "\x02\x42\x4d", [0x0c91] = "\x0e\x3a\x66",
  [0x0c92] = "\x01\x5e\x68", [0x0c93] = "\x02\x3b\x32",
  [0x0c94] = "\x02\x3a\x78", [0x0c95] = "\x02\x3b\x24",
  [0x0c96] = "\x0e\x3a\x62", [0x0c97] = "\x02\x3b\x29",
  [0x0c98] = "\x0e\x3a\x5c", [0x0c99] = "\x0e\x3a\x5e",
  [0x0c9a] = "\x01\x5e\x66", [0x0c9b] = "\x01\x5e\x69",
  [0x0c9c] = "\x02\x3b\x30", [0x0c9d] = "\x02\x3b\x33",
  [0x0c9e] = "\x02\x3a\x7c", [0x0c9f] = "\x02\x3a\x79",
  [0x0ca0] = "\x0e\x3a\x73", [0x0ca1] = "\x0e\x3a\x5a",
  [0x0ca2] = "\x02\x3a\x7b", [0x0ca3] = "\x0e\x3a\x57",
  [0x0ca4] = "\x0e\x3a\x7a", [0x0ca5] = "\x02\x3b\x22",
  [0x0ca6] = "\x02\x3a\x7e", [0x0ca7] = "\x01\x5e\x6a",
  [0x0ca8] = "\x0e\x3a\x56", [0x0ca9] = "\x02\x3b\x2c",
  [0x0caa] = "\x0e\x3a\x79", [0x0cab] = "\x0e\x3a\x78",
  [0x0cac] = "\x02\x3b\x23", [0x0cae] = "\x02\x3b\x25",
  [0x0caf] = "\x02\x3a\x77", [0x0cb0] = "\x02\x42\x41",
  [0x0cb1] = "\x02\x42\x3f", [0x0cb2] = "\x01\x63\x68",
  [0x0cb3] = "\x01\x63\x66", [0x0cb4] = "\x02\x42\x49",
  [0x0cb5] = "\x02\x42\x40", [0x0cb6] = "\x02\x42\x4a",
  [0x0cb7] = "\x02\x42\x46", [0x0cb8] = "\x02\x42\x3e",
  [0x0cb9] = "\x02\x42\x4c", [0x0cba] = "\x02\x42\x3d",
  [0x0cbb] = "\x02\x42\x44", [0x0cbc] = "\x01\x63\x65",
  [0x0cbd] = "\x01\x63\x64", [0x0cbe] = "\x01\x63\x63",
  [0x0cbf] = "\x02\x42\x42", [0x0cc0] = "\x02\x42\x47",
  [0x0cc1] = "\x01\x63\x60", [0x0cc2] = "\x01\x63\x67",
  [0x0cc3] = "\x0e\x40\x67", [0x0cc4] = "\x02\x42\x3b",
  [0x0cc5] = "\x0e\x40\x66", [0x0cc6] = "\x02\x42\x45",
  [0x0cc7] = "\x02\x42\x3a", [0x0cc8] = "\x02\x42\x43",
  [0x0cc9] = "\x01\x63\x61", [0x0cca] = "\x02\x42\x48",
  [0x0ccb] = "\x02\x42\x3c", [0x0ccc] = "\x01\x63\x62",
  [0x0ccd] = "\x02\x42\x4b", [0x0cce] = "\x0e\x40\x64",
  [0x0ccf] = "\x0e\x3a\x6b", [0x0cd0] = "\x0e\x40\x6a",
  [0x0cd1] = "\x0e\x64\x68", [0x0cd3] = "\x0e\x64\x69",
  [0x0cd5] = "\x02\x49\x55", [0x0cd6] = "\x01\x68\x4a",
  [0x0cd7] = "\x01\x68\x49", [0x0cd8] = "\x01\x68\x4b",
  [0x0cd9] = "\x02\x49\x61", [0x0cda] = "\x02\x49\x57",
  [0x0cdb] = "\x02\x49\x5d", [0x0cdc] = "\x02\x49\x52",
  [0x0cdd] = "\x02\x49\x60", [0x0cde] = "\x02\x49\x5f",
  [0x0cdf] = "\x02\x49\x63", [0x0ce0] = "\x02\x49\x5c",
  [0x0ce1] = "\x01\x68\x46", [0x0ce2] = "\x02\x49\x5b",
  [0x0ce3] = "\x01\x68\x4c", [0x0ce4] = "\x0e\x46\x4c",
  [0x0ce5] = "\x02\x49\x54", [0x0ce6] = "\x01\x68\x47",
  [0x0ce7] = "\x0e\x46\x4a", [0x0ce8] = "\x02\x49\x62",
  [0x0ce9] = "\x01\x68\x48", [0x0cea] = "\x02\x49\x56",
  [0x0ceb] = "\x02\x49\x59", [0x0cec] = "\x02\x49\x5e",
  [0x0ced] = "\x02\x49\x58", [0x0cee] = "\x02\x49\x53",
  [0x0cef] = "\x0e\x46\x4f", [0x0cf0] = "\x0e\x46\x53",
  [0x0cf3] = "\x02\x49\x5a", [0x0cf4] = "\x02\x4f\x7c",
  [0x0cf5] = "\x01\x6c\x4b", [0x0cf6] = "\x02\x50\x21",
  [0x0cf7] = "\x02\x4f\x7e", [0x0cf8] = "\x02\x50\x23",
  [0x0cf9] = "\x02\x50\x25", [0x0cfa] = "\x0e\x4b\x5c",
  [0x0cfb] = "\x01\x6c\x49", [0x0cfc] = "\x0e\x4b\x58",
  [0x0cfd] = "\x02\x4f\x7d", [0x0cfe] = "\x0e\x4b\x54",
  [0x0cff] = "\x02\x4f\x7b", [0x0d00] = "\x0e\x4b\x5e",
  [0x0d01] = "\x02\x50\x26", [0x0d02] = "\x02\x50\x24",
  [0x0d03] = "\x02\x50\x22", [0x0d04] = "\x0e\x4b\x56",
  [0x0d05] = "\x02\x50\x28", [0x0d06] = "\x0e\x4b\x5a",
  [0x0d07] = "\x02\x50\x27", [0x0d08] = "\x01\x6c\x4d",
  [0x0d09] = "\x01\x6c\x48", [0x0d0a] = "\x0e\x4b\x5b",
  [0x0d0b] = "\x01\x6c\x4a", [0x0d0c] = "\x01\x6c\x4c",
  [0x0d0d] = "\x0e\x4b\x5d", [0x0d0e] = "\x0e\x4b\x5f",
  [0x0d0f] = "\x02\x50\x29", [0x0d10] = "\x02\x56\x61",
  [0x0d11] = "\x0e\x50\x50", [0x0d12] = "\x0e\x50\x53",
  [0x0d13] = "\x02\x56\x60", [0x0d14] = "\x02\x56\x5f",
  [0x0d15] = "\x0e\x50\x51", [0x0d16] = "\x02\x56\x62",
  [0x0d17] = "\x02\x56\x5b", [0x0d18] = "\x0e\x50\x4f",
  [0x0d19] = "\x02\x56\x5c", [0x0d1a] = "\x02\x56\x64",
  [0x0d1b] = "\x02\x56\x5d", [0x0d1c] = "\x0e\x50\x54",
  [0x0d1d] = "\x01\x70\x4e", [0x0d1e] = "\x02\x56\x66",
  [0x0d1f] = "\x0e\x50\x4e", [0x0d20] = "\x02\x56\x65",
  [0x0d21] = "\x02\x56\x5e", [0x0d22] = "\x0e\x50\x56",
  [0x0d23] = "\x02\x5c\x5d", [0x0d24] = "\x01\x73\x51",
  [0x0d25] = "\x02\x5c\x5b", [0x0d26] = "\x02\x5c\x60",
  [0x0d27] = "\x02\x5c\x5f", [0x0d28] = "\x02\x56\x63",
  [0x0d29] = "\x0e\x54\x50", [0x0d2a] = "\x01\x73\x50",
  [0x0d2b] = "\x0e\x54\x4e", [0x0d2c] = "\x02\x5c\x5e",
  [0x0d2d] = "\x02\x5c\x5a", [0x0d2e] = "\x02\x5c\x62",
  [0x0d2f] = "\x02\x5c\x61", [0x0d30] = "\x01\x73\x4f",
  [0x0d31] = "\x0e\x54\x52", [0x0d32] = "\x02\x5c\x5c",
  [0x0d33] = "\x0e\x54\x4f", [0x0d34] = "\x01\x70\x4f",
  [0x0d35] = "\x0e\x54\x51", [0x0d37] = "\x0e\x54\x53",
  [0x0d38] = "\x01\x76\x40", [0x0d39] = "\x0e\x5a\x31",
  [0x0d3a] = "\x0e\x57\x61", [0x0d3b] = "\x0e\x57\x5e",
  [0x0d3c] = "\x02\x61\x4d", [0x0d3d] = "\x02\x65\x46",
  [0x0d3e] = "\x02\x65\x47", [0x0d3f] = "\x02\x65\x48",
  [0x0d40] = "\x01\x79\x67", [0x0d42] = "\x0e\x5c\x36",
  [0x0d43] = "\x01\x79\x68", [0x0d44] = "\x0e\x5c\x39",
  [0x0d45] = "\x02\x68\x6f", [0x0d46] = "\x0e\x5c\x3a",
  [0x0d47] = "\x02\x6b\x56", [0x0d48] = "\x02\x6b\x55",
  [0x0d49] = "\x0e\x5e\x29", [0x0d4a] = "\x0e\x5f\x42",
  [0x0d4b] = "\x02\x6d\x49", [0x0d4c] = "\x02\x6d\x4a",
  [0x0d4d] = "\x02\x6f\x24", [0x0d4e] = "\x02\x70\x35",
  [0x0d4f] = "\x0e\x60\x73", [0x0d50] = "\x01\x44\x4d",
  [0x0d51] = "\x01\x44\x4e", [0x0d53] = "\x01\x44\x4f",
  [0x0d54] = "\x01\x45\x36", [0x0d55] = "\x01\x46\x45",
  [0x0d56] = "\x02\x22\x33", [0x0d57] = "\x01\x47\x73",
  [0x0d58] = "\x01\x47\x74", [0x0d5a] = "\x01\x49\x77",
  [0x0d5b] = "\x01\x49\x78", [0x0d5c] = "\x01\x49\x76",
  [0x0d5d] = "\x01\x49\x75", [0x0d5f] = "\x01\x4c\x75",
  [0x0d60] = "\x0e\x28\x48", [0x0d62] = "\x02\x26\x36",
  [0x0d63] = "\x01\x4c\x77", [0x0d64] = "\x01\x4c\x76",
  [0x0d65] = "\x02\x26\x37", [0x0d66] = "\x0e\x62\x5c",
  [0x0d67] = "\x0e\x64\x6a", [0x0d69] = "\x01\x50\x66",
  [0x0d6a] = "\x0e\x2c\x2f", [0x0d6b] = "\x01\x55\x21",
  [0x0d6c] = "\x02\x2e\x67", [0x0d6e] = "\x02\x34\x73",
  [0x0d70] = "\x01\x59\x69", [0x0d71] = "\x01\x5e\x6c",
  [0x0d72] = "\x02\x34\x72", [0x0d73] = "\x01\x5e\x6b",
  [0x0d74] = "\x0e\x40\x71", [0x0d75] = "\x01\x68\x4d",
  [0x0d76] = "\x0e\x40\x72", [0x0d77] = "\x02\x49\x64",
  [0x0d78] = "\x01\x70\x50", [0x0d79] = "\x0e\x50\x57",
  [0x0d7a] = "\x01\x73\x52", [0x0d7b] = "\x02\x5c\x63",
  [0x0d7c] = "\x0e\x5a\x34", [0x0d7d] = "\x01\x79\x69",
  [0x0d7e] = "\x0e\x5c\x3b", [0x0d7f] = "\x01\x7b\x5c",
  [0x0d80] = "\x0e\x21\x3b", [0x0d81] = "\x02\x21\x53",
  [0x0d82] = "\x0e\x22\x3d", [0x0d83] = "\x01\x46\x46",
  [0x0d84] = "\x02\x21\x54", [0x0d85] = "\x01\x47\x77",
  [0x0d86] = "\x0e\x23\x50", [0x0d87] = "\x01\x47\x75",
  [0x0d88] = "\x01\x47\x76", [0x0d89] = "\x01\x47\x78",
  [0x0d8b] = "\x01\x49\x7a", [0x0d8c] = "\x01\x49\x79",
  [0x0d8d] = "\x0e\x25\x5a", [0x0d8e] = "\x02\x23\x5c",
  [0x0d8f] = "\x01\x49\x7b", [0x0d92] = "\x02\x23\x5d",
  [0x0d93] = "\x02\x26\x38", [0x0d94] = "\x0e\x28\x4c",
  [0x0d95] = "\x02\x26\x39", [0x0d96] = "\x0e\x28\x4a",
  [0x0d97] = "\x01\x4c\x78", [0x0d98] = "\x01\x4c\x7a",
  [0x0d99] = "\x01\x4c\x7c", [0x0d9a] = "\x01\x4c\x79",
  [0x0d9b] = "\x01\x4c\x7d", [0x0d9c] = "\x01\x4c\x7b",
  [0x0d9d] = "\x0e\x28\x4e", [0x0d9e] = "\x0e\x28\x4b",
  [0x0d9f] = "\x0e\x28\x4d", [0x0da2] = "\x01\x50\x6a",
  [0x0da3] = "\x01\x50\x67", [0x0da4] = "\x01\x50\x69",
  [0x0da5] = "\x01\x50\x6b", [0x0da6] = "\x01\x50\x68",
  [0x0da7] = "\x02\x2e\x68", [0x0da8] = "\x02\x29\x78",
  [0x0dab] = "\x0e\x2c\x32", [0x0dac] = "\x02\x2e\x6a",
  [0x0dad] = "\x02\x2e\x69", [0x0dae] = "\x01\x55\x27",
  [0x0db0] = "\x01\x55\x23", [0x0db3] = "\x01\x55\x24",
  [0x0db4] = "\x01\x55\x26", [0x0db5] = "\x01\x55\x28",
  [0x0db6] = "\x01\x55\x25", [0x0db7] = "\x0e\x30\x57",
  [0x0db8] = "\x01\x55\x2a", [0x0db9] = "\x01\x55\x29",
  [0x0dbb] = "\x0e\x30\x56", [0x0dbc] = "\x0e\x30\x55",
  [0x0dbf] = "\x01\x59\x6e", [0x0dc0] = "\x02\x34\x75",
  [0x0dc1] = "\x02\x34\x74", [0x0dc2] = "\x01\x59\x6d",
  [0x0dc3] = "\x0e\x35\x41", [0x0dc4] = "\x01\x59\x6c",
  [0x0dc5] = "\x01\x59\x6b", [0x0dc6] = "\x01\x59\x6f",
  [0x0dc7] = "\x01\x59\x6a", [0x0dc8] = "\x0e\x35\x3f",
  [0x0dca] = "\x02\x3b\x39", [0x0dcb] = "\x02\x3b\x36",
  [0x0dcc] = "\x01\x5e\x6e", [0x0dcd] = "\x02\x3b\x35",
  [0x0dce] = "\x02\x3b\x3a", [0x0dd0] = "\x01\x5e\x70",
  [0x0dd1] = "\x02\x3b\x38", [0x0dd2] = "\x01\x5e\x6d",
  [0x0dd3] = "\x01\x5e\x6f", [0x0dd4] = "\x02\x3b\x37",
  [0x0dd5] = "\x0e\x3a\x7e", [0x0dd6] = "\x02\x42\x4e",
  [0x0dd7] = "\x0e\x40\x74", [0x0dd8] = "\x02\x42\x4f",
  [0x0dd9] = "\x02\x42\x50", [0x0dda] = "\x0e\x40\x75",
  [0x0ddb] = "\x0e\x40\x76", [0x0ddc] = "\x0e\x40\x73",
  [0x0ddd] = "\x0e\x63\x45", [0x0dde] = "\x01\x68\x4e",
  [0x0ddf] = "\x01\x68\x56", [0x0de0] = "\x02\x49\x65",
  [0x0de1] = "\x01\x68\x50", [0x0de2] = "\x01\x68\x54",
  [0x0de3] = "\x02\x49\x66", [0x0de4] = "\x01\x68\x55",
  [0x0de5] = "\x01\x68\x51", [0x0de6] = "\x01\x68\x52",
  [0x0de7] = "\x01\x68\x4f", [0x0de8] = "\x01\x68\x53",
  [0x0de9] = "\x01\x6c\x50", [0x0dea] = "\x02\x3b\x34",
  [0x0deb] = "\x01\x6c\x51", [0x0dec] = "\x01\x6c\x4f",
  [0x0dee] = "\x01\x6c\x4e", [0x0def] = "\x02\x56\x67",
  [0x0df0] = "\x01\x70\x51", [0x0df1] = "\x02\x5c\x64",
  [0x0df2] = "\x02\x5c\x65", [0x0df3] = "\x0e\x5a\x35",
  [0x0df5] = "\x01\x78\x31", [0x0df6] = "\x01\x79\x6a",
  [0x0df7] = "\x0e\x5e\x2a", [0x0df8] = "\x01\x44\x50",
  [0x0df9] = "\x0e\x22\x3e", [0x0dfa] = "\x01\x47\x79",
  [0x0dfd] = "\x0e\x25\x5e", [0x0dfe] = "\x0e\x25\x5c",
  [0x0dff] = "\x0e\x25\x5d", [0x0e00] = "\x0e\x28\x50",
  [0x0e01] = "\x01\x50\x6c", [0x0e03] = "\x02\x2e\x6b",
  [0x0e04] = "\x01\x55\x2b", [0x0e05] = "\x0e\x30\x58",
  [0x0e06] = "\x0e\x30\x59", [0x0e07] = "\x01\x59\x72",
  [0x0e08] = "\x01\x59\x71", [0x0e09] = "\x01\x59\x70",
  [0x0e0a] = "\x01\x5e\x71", [0x0e0b] = "\x01\x5e\x72",
  [0x0e0c] = "\x02\x3b\x3b", [0x0e0d] = "\x01\x68\x57",
  [0x0e0e] = "\x01\x70\x52", [0x0e0f] = "\x01\x44\x51",
  [0x0e10] = "\x02\x21\x3c", [0x0e11] = "\x01\x45\x37",
  [0x0e12] = "\x02\x21\x55", [0x0e13] = "\x0e\x62\x4a",
  [0x0e14] = "\x0e\x22\x3f", [0x0e15] = "\x02\x22\x34",
  [0x0e16] = "\x01\x47\x7a", [0x0e18] = "\x0e\x23\x52",
  [0x0e19] = "\x0e\x28\x51", [0x0e1a] = "\x01\x4c\x7e",
  [0x0e1b] = "\x0e\x2c\x34", [0x0e1e] = "\x0e\x3b\x24",
  [0x0e1f] = "\x02\x42\x51", [0x0e20] = "\x0e\x40\x78",
  [0x0e21] = "\x0e\x65\x25", [0x0e22] = "\x01\x44\x52",
  [0x0e24] = "\x01\x45\x38", [0x0e25] = "\x02\x22\x35",
  [0x0e28] = "\x02\x23\x5e", [0x0e2a] = "\x02\x23\x5f",
  [0x0e2b] = "\x0e\x25\x5f", [0x0e2c] = "\x01\x49\x7c",
  [0x0e2f] = "\x0e\x2c\x35", [0x0e30] = "\x02\x3b\x3c",
  [0x0e31] = "\x01\x5e\x73", [0x0e33] = "\x02\x42\x52",
  [0x0e36] = "\x0e\x54\x54", [0x0e37] = "\x01\x73\x53",
  [0x0e38] = "\x01\x44\x53", [0x0e39] = "\x01\x44\x67",
  [0x0e3a] = "\x01\x45\x39", [0x0e3b] = "\x02\x21\x56",
  [0x0e3c] = "\x01\x46\x47", [0x0e3d] = "\x0e\x23\x54",
  [0x0e3e] = "\x01\x4a\x22", [0x0e3f] = "\x01\x4a\x21",
  [0x0e40] = "\x01\x49\x7d", [0x0e41] = "\x01\x49\x7e",
  [0x0e44] = "\x02\x26\x3a", [0x0e45] = "\x01\x4d\x22",
  [0x0e46] = "\x01\x4d\x23", [0x0e47] = "\x02\x26\x3b",
  [0x0e48] = "\x01\x4d\x21", [0x0e49] = "\x0e\x28\x54",
  [0x0e4a] = "\x0e\x28\x55", [0x0e4b] = "\x01\x50\x70",
  [0x0e4c] = "\x02\x29\x79", [0x0e4d] = "\x01\x50\x6f",
  [0x0e4e] = "\x01\x50\x6d", [0x0e4f] = "\x01\x50\x6e",
  [0x0e50] = "\x01\x55\x2e", [0x0e51] = "\x01\x55\x2c",
  [0x0e52] = "\x0e\x30\x5a", [0x0e53] = "\x0e\x30\x5b",
  [0x0e54] = "\x02\x2e\x6d", [0x0e55] = "\x01\x55\x2d",
  [0x0e56] = "\x02\x2e\x6c", [0x0e57] = "\x0e\x64\x6b",
  [0x0e58] = "\x01\x55\x22", [0x0e59] = "\x02\x34\x76",
  [0x0e5b] = "\x0e\x35\x43", [0x0e5c] = "\x01\x59\x74",
  [0x0e5d] = "\x01\x59\x75", [0x0e5e] = "\x0e\x3b\x26",
  [0x0e5f] = "\x0e\x3b\x25", [0x0e60] = "\x01\x59\x73",
  [0x0e61] = "\x0e\x35\x44", [0x0e62] = "\x01\x68\x58",
  [0x0e63] = "\x02\x49\x67", [0x0e64] = "\x01\x6c\x52",
  [0x0e65] = "\x01\x6c\x53", [0x0e67] = "\x02\x50\x2a",
  [0x0e68] = "\x01\x73\x54", [0x0e69] = "\x02\x61\x4e",
  [0x0e6a] = "\x02\x61\x4f", [0x0e6b] = "\x0e\x5a\x38",
  [0x0e6c] = "\x01\x7a\x68", [0x0e6d] = "\x02\x70\x36",
  [0x0e6e] = "\x02\x21\x2b", [0x0e6f] = "\x01\x45\x3a",
  [0x0e71] = "\x01\x44\x54", [0x0e73] = "\x02\x21\x58",
  [0x0e74] = "\x02\x21\x57", [0x0e77] = "\x0e\x64\x6c",
  [0x0e78] = "\x0e\x23\x55", [0x0e79] = "\x01\x47\x7b",
  [0x0e7a] = "\x02\x22\x37", [0x0e7b] = "\x02\x22\x38",
  [0x0e7c] = "\x02\x22\x36", [0x0e7e] = "\x02\x22\x39",
  [0x0e80] = "\x0e\x23\x5a", [0x0e84] = "\x0e\x25\x61",
  [0x0e85] = "\x0e\x25\x66", [0x0e86] = "\x02\x23\x67",
  [0x0e88] = "\x02\x23\x62", [0x0e89] = "\x02\x23\x64",
  [0x0e8a] = "\x02\x23\x66", [0x0e8b] = "\x02\x23\x63",
  [0x0e8c] = "\x01\x4a\x26", [0x0e8d] = "\x02\x23\x60",
  [0x0e8e] = "\x0e\x25\x62", [0x0e8f] = "\x02\x23\x61",
  [0x0e90] = "\x01\x4a\x23", [0x0e91] = "\x01\x4a\x24",
  [0x0e92] = "\x02\x23\x65", [0x0e93] = "\x02\x23\x68",
  [0x0e94] = "\x01\x4a\x25", [0x0e95] = "\x02\x23\x69",
  [0x0e9c] = "\x0e\x25\x63", [0x0e9d] = "\x02\x26\x49",
  [0x0e9e] = "\x0e\x28\x5d", [0x0e9f] = "\x02\x26\x43",
  [0x0ea0] = "\x02\x26\x3e", [0x0ea1] = "\x01\x4d\x25",
  [0x0ea2] = "\x02\x26\x46", [0x0ea3] = "\x02\x26\x44",
  [0x0ea4] = "\x02\x26\x3d", [0x0ea5] = "\x02\x26\x4a",
  [0x0ea6] = "\x02\x26\x4d", [0x0ea7] = "\x02\x26\x48",
  [0x0ea8] = "\x02\x26\x41", [0x0ea9] = "\x01\x4d\x27",
  [0x0eaa] = "\x02\x26\x47", [0x0eab] = "\x01\x4d\x28",
  [0x0eac] = "\x02\x26\x42", [0x0ead] = "\x02\x26\x45",
  [0x0eae] = "\x02\x26\x3c", [0x0eaf] = "\x02\x26\x40",
  [0x0eb0] = "\x02\x26\x4c", [0x0eb1] = "\x01\x4d\x29",
  [0x0eb3] = "\x01\x4d\x2a", [0x0eb4] = "\x0e\x28\x5e",
  [0x0eb5] = "\x02\x26\x3f", [0x0eb6] = "\x02\x26\x4b",
  [0x0eb7] = "\x01\x4d\x24", [0x0eb8] = "\x01\x4d\x26",
  [0x0eb9] = "\x0e\x28\x5c", [0x0eba] = "\x0e\x28\x5f",
  [0x0ebb] = "\x0e\x28\x57", [0x0ec1] = "\x0e\x28\x58",
  [0x0ec6] = "\x02\x2a\x2c", [0x0ec7] = "\x02\x2a\x25",
  [0x0ec8] = "\x02\x2a\x2b", [0x0ec9] = "\x02\x2a\x24",
  [0x0eca] = "\x02\x2a\x26", [0x0ecb] = "\x02\x29\x7e",
  [0x0ecc] = "\x02\x29\x7c", [0x0ecd] = "\x0e\x2c\x3f",
  [0x0ece] = "\x02\x2a\x2d", [0x0ecf] = "\x02\x2a\x2a",
  [0x0ed0] = "\x02\x29\x7a", [0x0ed1] = "\x0e\x2c\x3b",
  [0x0ed2] = "\x01\x50\x72", [0x0ed3] = "\x02\x2a\x28",
  [0x0ed4] = "\x02\x2a\x29", [0x0ed5] = "\x0e\x64\x6d",
  [0x0ed6] = "\x02\x2a\x27", [0x0ed7] = "\x02\x29\x7d",
  [0x0ed8] = "\x02\x29\x7b", [0x0ed9] = "\x01\x50\x71",
  [0x0eda] = "\x02\x2a\x23", [0x0edb] = "\x02\x2a\x21",
  [0x0edc] = "\x0e\x2c\x3c", [0x0edd] = "\x0e\x2c\x42",
  [0x0ede] = "\x02\x2a\x22", [0x0edf] = "\x02\x2a\x2e",
  [0x0ee1] = "\x0e\x2c\x3e", [0x0ee2] = "\x0e\x2c\x41",
  [0x0ee5] = "\x0e\x2c\x43", [0x0ee6] = "\x0e\x2c\x3d",
  [0x0ee8] = "\x01\x55\x33", [0x0ee9] = "\x0e\x30\x63",
  [0x0eea] = "\x01\x55\x32", [0x0eeb] = "\x0e\x30\x5f",
  [0x0eec] = "\x02\x2e\x6e", [0x0eed] = "\x01\x55\x2f",
  [0x0eee] = "\x02\x2e\x70", [0x0eef] = "\x0e\x30\x64",
  [0x0ef0] = "\x01\x55\x34", [0x0ef1] = "\x02\x2e\x71",
  [0x0ef3] = "\x0e\x30\x61", [0x0ef4] = "\x01\x55\x37",
  [0x0ef5] = "\x0e\x30\x62", [0x0ef6] = "\x01\x55\x35",
  [0x0ef7] = "\x02\x2e\x72", [0x0ef8] = "\x02\x2a\x2f",
  [0x0ef9] = "\x02\x2e\x74", [0x0efa] = "\x0e\x30\x5e",
  [0x0efb] = "\x01\x55\x31", [0x0efc] = "\x0e\x62\x71",
  [0x0efd] = "\x01\x55\x30", [0x0efe] = "\x0e\x62\x72",
  [0x0eff] = "\x02\x2e\x6f", [0x0f00] = "\x02\x2e\x73",
  [0x0f01] = "\x01\x55\x36", [0x0f06] = "\x01\x59\x77",
  [0x0f07] = "\x01\x59\x76", [0x0f08] = "\x0e\x35\x4b",
  [0x0f09] = "\x0e\x35\x48", [0x0f0a] = "\x0e\x35\x47",
  [0x0f0b] = "\x02\x34\x78", [0x0f0c] = "\x02\x34\x7c",
  [0x0f0d] = "\x02\x34\x7e", [0x0f0e] = "\x01\x59\x78",
  [0x0f0f] = "\x02\x35\x23", [0x0f10] = "\x0e\x35\x51",
  [0x0f11] = "\x01\x59\x7c", [0x0f12] = "\x02\x35\x25",
  [0x0f13] = "\x0e\x35\x52", [0x0f14] = "\x01\x59\x7e",
  [0x0f15] = "\x0e\x35\x4f", [0x0f16] = "\x01\x59\x7a",
  [0x0f17] = "\x01\x5a\x24", [0x0f18] = "\x0e\x35\x55",
  [0x0f19] = "\x01\x5a\x21", [0x0f1a] = "\x02\x34\x7a",
  [0x0f1b] = "\x01\x59\x79", [0x0f1c] = "\x0e\x3b\x2b",
  [0x0f1d] = "\x02\x34\x79", [0x0f1e] = "\x02\x34\x77",
  [0x0f1f] = "\x02\x35\x27", [0x0f20] = "\x02\x34\x7b",
  [0x0f21] = "\x0e\x35\x46", [0x0f22] = "\x01\x59\x7b",
  [0x0f23] = "\x02\x35\x26", [0x0f24] = "\x01\x5a\x22",
  [0x0f25] = "\x02\x35\x22", [0x0f26] = "\x02\x35\x21",
  [0x0f27] = "\x01\x5a\x23", [0x0f28] = "\x02\x34\x7d",
  [0x0f29] = "\x01\x59\x7d", [0x0f2a] = "\x0e\x35\x4e",
  [0x0f2c] = "\x0e\x35\x4a", [0x0f2e] = "\x02\x35\x28",
  [0x0f2f] = "\x0e\x35\x54", [0x0f30] = "\x02\x35\x24",
  [0x0f31] = "\x02\x3b\x4b", [0x0f32] = "\x02\x3b\x52",
  [0x0f33] = "\x02\x3b\x47", [0x0f34] = "\x01\x5e\x76",
  [0x0f35] = "\x02\x3b\x43", [0x0f36] = "\x02\x3b\x53",
  [0x0f37] = "\x02\x3b\x3d", [0x0f38] = "\x02\x3b\x50",
  [0x0f39] = "\x02\x3b\x4e", [0x0f3a] = "\x02\x3b\x48",
  [0x0f3b] = "\x0e\x3b\x36", [0x0f3c] = "\x02\x3b\x51",
  [0x0f3d] = "\x02\x3b\x4a", [0x0f3e] = "\x0e\x3b\x28",
  [0x0f3f] = "\x02\x3b\x42", [0x0f40] = "\x02\x3b\x54",
  [0x0f41] = "\x02\x3b\x40", [0x0f42] = "\x02\x3b\x4d",
  [0x0f43] = "\x02\x3b\x3e", [0x0f44] = "\x0e\x3b\x27",
  [0x0f45] = "\x02\x3b\x55", [0x0f46] = "\x0e\x3b\x37",
  [0x0f47] = "\x01\x5e\x77", [0x0f48] = "\x0e\x3b\x2a",
  [0x0f49] = "\x02\x3b\x4f", [0x0f4a] = "\x02\x42\x55",
  [0x0f4b] = "\x02\x3b\x41", [0x0f4c] = "\x01\x5e\x74",
  [0x0f4d] = "\x0e\x3b\x2e", [0x0f4e] = "\x02\x3b\x45",
  [0x0f4f] = "\x0e\x3b\x34", [0x0f50] = "\x01\x5e\x75",
  [0x0f51] = "\x02\x3b\x44", [0x0f52] = "\x02\x3b\x49",
  [0x0f53] = "\x0e\x3b\x33", [0x0f55] = "\x02\x3b\x46",
  [0x0f57] = "\x0e\x3b\x31", [0x0f59] = "\x02\x3b\x4c",
  [0x0f5b] = "\x0e\x3b\x2d", [0x0f5e] = "\x02\x42\x59",
  [0x0f60] = "\x0e\x41\x21", [0x0f62] = "\x02\x42\x5c",
  [0x0f63] = "\x02\x42\x54", [0x0f64] = "\x0e\x40\x7c",
  [0x0f65] = "\x02\x42\x56", [0x0f66] = "\x0e\x40\x7a",
  [0x0f67] = "\x02\x42\x5b", [0x0f68] = "\x02\x42\x5a",
  [0x0f69] = "\x01\x63\x69", [0x0f6b] = "\x02\x3b\x3f",
  [0x0f6c] = "\x02\x42\x58", [0x0f6d] = "\x0e\x40\x79",
  [0x0f6e] = "\x0e\x40\x7d", [0x0f6f] = "\x01\x63\x6a",
  [0x0f71] = "\x02\x42\x53", [0x0f72] = "\x02\x42\x57",
  [0x0f75] = "\x0e\x64\x6e", [0x0f77] = "\x02\x49\x6e",
  [0x0f78] = "\x0e\x46\x5c", [0x0f79] = "\x02\x49\x75",
  [0x0f7a] = "\x02\x49\x6c", [0x0f7b] = "\x0e\x46\x54",
  [0x0f7c] = "\x02\x49\x73", [0x0f7d] = "\x02\x49\x6a",
  [0x0f7e] = "\x02\x49\x72", [0x0f7f] = "\x02\x49\x76",
  [0x0f80] = "\x02\x49\x69", [0x0f81] = "\x02\x49\x6d",
  [0x0f82] = "\x02\x49\x68", [0x0f83] = "\x0e\x46\x56",
  [0x0f84] = "\x01\x68\x59", [0x0f86] = "\x02\x49\x6b",
  [0x0f87] = "\x01\x68\x5a", [0x0f88] = "\x02\x49\x71",
  [0x0f89] = "\x02\x49\x70", [0x0f8a] = "\x02\x49\x6f",
  [0x0f8b] = "\x0e\x46\x59", [0x0f8c] = "\x0e\x46\x5a",
  [0x0f8d] = "\x02\x49\x74", [0x0f8e] = "\x0e\x64\x6f",
  [0x0f8f] = "\x0e\x4b\x65", [0x0f90] = "\x0e\x4b\x66",
  [0x0f91] = "\x0e\x4b\x63", [0x0f92] = "\x02\x50\x2e",
  [0x0f93] = "\x02\x50\x30", [0x0f94] = "\x01\x6c\x55",
  [0x0f95] = "\x02\x50\x31", [0x0f96] = "\x0e\x4b\x6b",
  [0x0f97] = "\x02\x50\x2c", [0x0f98] = "\x0e\x4b\x67",
  [0x0f99] = "\x02\x50\x2b", [0x0f9a] = "\x02\x50\x35",
  [0x0f9b] = "\x0e\x4b\x69", [0x0f9c] = "\x02\x50\x33",
  [0x0f9d] = "\x01\x6c\x54", [0x0f9e] = "\x02\x50\x36",
  [0x0f9f] = "\x02\x50\x2d", [0x0fa0] = "\x02\x50\x32",
  [0x0fa1] = "\x02\x50\x34", [0x0fa2] = "\x02\x50\x2f",
  [0x0fa3] = "\x0e\x4b\x6a", [0x0fa4] = "\x0e\x4b\x68",
  [0x0fa5] = "\x0e\x64\x70", [0x0fa6] = "\x0e\x50\x58",
  [0x0fa7] = "\x02\x56\x6b", [0x0fa8] = "\x02\x56\x70",
  [0x0fa9] = "\x02\x56\x6a", [0x0faa] = "\x02\x56\x6f",
  [0x0fab] = "\x0e\x50\x5b", [0x0fac] = "\x02\x56\x68",
  [0x0fad] = "\x02\x56\x72", [0x0fae] = "\x02\x56\x6e",
  [0x0faf] = "\x02\x56\x73", [0x0fb0] = "\x02\x56\x6d",
  [0x0fb1] = "\x02\x56\x69", [0x0fb2] = "\x02\x56\x71",
  [0x0fb3] = "\x0e\x50\x5a", [0x0fb4] = "\x02\x56\x74",
  [0x0fb5] = "\x02\x56\x6c", [0x0fb6] = "\x0e\x64\x71",
  [0x0fb7] = "\x02\x5c\x66", [0x0fb8] = "\x01\x73\x58",
  [0x0fb9] = "\x0e\x54\x57", [0x0fba] = "\x01\x73\x56",
  [0x0fbb] = "\x0e\x54\x59", [0x0fbc] = "\x01\x73\x55",
  [0x0fbd] = "\x01\x73\x57", [0x0fbe] = "\x0e\x54\x55",
  [0x0fbf] = "\x0e\x64\x72", [0x0fc0] = "\x02\x61\x50",
  [0x0fc1] = "\x0e\x57\x62", [0x0fc2] = "\x02\x64\x3e",
  [0x0fc3] = "\x02\x65\x49", [0x0fc4] = "\x0e\x5a\x39",
  [0x0fc6] = "\x02\x68\x70", [0x0fc7] = "\x02\x68\x71",
  [0x0fc8] = "\x0e\x5c\x3d", [0x0fc9] = "\x01\x79\x6b",
  [0x0fca] = "\x0e\x5c\x3c", [0x0fcb] = "\x02\x6b\x57",
  [0x0fcc] = "\x0e\x5c\x3e", [0x0fcd] = "\x01\x7a\x69",
  [0x0fcf] = "\x02\x6b\x58", [0x0fd0] = "\x0e\x5e\x2d",
  [0x0fd1] = "\x02\x6d\x4c", [0x0fd2] = "\x01\x7b\x5e",
  [0x0fd3] = "\x0e\x5f\x43", [0x0fd4] = "\x01\x7b\x5d",
  [0x0fd5] = "\x02\x6d\x4b", [0x0fd6] = "\x01\x7c\x35",
  [0x0fd7] = "\x0e\x60\x30", [0x0fd8] = "\x02\x6f\x25",
  [0x0fdb] = "\x0e\x21\x3c", [0x0fdd] = "\x01\x44\x55",
  [0x0fde] = "\x01\x47\x7c", [0x0fdf] = "\x02\x22\x3a",
  [0x0fe0] = "\x02\x23\x6a", [0x0fe1] = "\x01\x4b\x56",
  [0x0fe2] = "\x01\x5a\x25", [0x0fe3] = "\x0e\x35\x57",
  [0x0fe4] = "\x0e\x4b\x6c", [0x0fe5] = "\x01\x44\x56",
  [0x0fe6] = "\x01\x46\x4a", [0x0fe7] = "\x01\x46\x49",
  [0x0fe8] = "\x01\x46\x48", [0x0feb] = "\x01\x4a\x27",
  [0x0fee] = "\x01\x55\x38", [0x0ff0] = "\x02\x42\x5d",
  [0x0ff1] = "\x01\x44\x57", [0x0ff2] = "\x01\x44\x58",
  [0x0ff3] = "\x01\x44\x59", [0x0ff4] = "\x01\x45\x3b",
  [0x0ff5] = "\x0e\x25\x67", [0x0ff6] = "\x0e\x28\x60",
  [0x0ff7] = "\x01\x50\x73", [0x0ff9] = "\x02\x2a\x30",
  [0x0ffa] = "\x0e\x2c\x45", [0x0ffb] = "\x0e\x2c\x44",
  [0x0ffd] = "\x01\x5e\x78", [0x0ffe] = "\x01\x44\x5a",
  [0x0fff] = "\x02\x21\x3d", [0x1000] = "\x0e\x21\x63",
  [0x1002] = "\x01\x46\x4b", [0x1003] = "\x01\x46\x4c",
  [0x1004] = "\x02\x21\x59", [0x1006] = "\x01\x47\x7d",
  [0x1008] = "\x0e\x23\x5c", [0x1009] = "\x0e\x25\x68",
  [0x100a] = "\x02\x23\x6b", [0x100b] = "\x0e\x25\x6b",
  [0x100c] = "\x01\x4a\x28", [0x100e] = "\x02\x23\x6c",
  [0x1011] = "\x01\x4d\x30", [0x1012] = "\x0e\x28\x62",
  [0x1013] = "\x0e\x28\x61", [0x1014] = "\x02\x26\x4f",
  [0x1015] = "\x01\x4d\x2e", [0x1016] = "\x01\x4d\x2d",
  [0x1017] = "\x02\x26\x4e", [0x1018] = "\x01\x4d\x2b",
  [0x1019] = "\x02\x26\x50", [0x101a] = "\x01\x4d\x2c",
  [0x101b] = "\x01\x4d\x2f", [0x101d] = "\x01\x50\x74",
  [0x101e] = "\x0e\x2c\x46", [0x101f] = "\x01\x50\x76",
  [0x1020] = "\x02\x2a\x34", [0x1021] = "\x02\x2a\x31",
  [0x1022] = "\x02\x2a\x32", [0x1023] = "\x02\x2a\x33",
  [0x1024] = "\x02\x2a\x35", [0x1025] = "\x01\x50\x75",
  [0x1028] = "\x02\x2e\x76", [0x1029] = "\x02\x2e\x75",
  [0x102a] = "\x0e\x30\x66", [0x102b] = "\x01\x55\x3a",
  [0x102c] = "\x0e\x30\x69", [0x102d] = "\x01\x55\x39",
  [0x102e] = "\x0e\x30\x68", [0x102f] = "\x0e\x30\x6a",
  [0x1030] = "\x0e\x30\x67", [0x1032] = "\x0e\x35\x5b",
  [0x1033] = "\x01\x5a\x28", [0x1034] = "\x02\x35\x2a",
  [0x1035] = "\x0e\x35\x59", [0x1036] = "\x01\x5a\x27",
  [0x1037] = "\x01\x5a\x29", [0x1038] = "\x01\x5a\x26",
  [0x1039] = "\x0e\x35\x58", [0x103a] = "\x0e\x35\x5a",
  [0x103d] = "\x01\x5e\x7a", [0x103e] = "\x02\x35\x29",
  [0x1040] = "\x01\x5e\x7b", [0x1041] = "\x02\x3b\x57",
  [0x1042] = "\x0e\x39\x79", [0x1043] = "\x01\x5e\x7c",
  [0x1044] = "\x02\x3b\x56", [0x1045] = "\x01\x5e\x79",
  [0x1046] = "\x0e\x3b\x3a", [0x1047] = "\x0e\x3b\x3b",
  [0x1048] = "\x0e\x3b\x3d", [0x104a] = "\x02\x42\x60",
  [0x104b] = "\x02\x42\x62", [0x104c] = "\x01\x63\x6b",
  [0x104d] = "\x02\x42\x61", [0x104e] = "\x02\x42\x5f",
  [0x104f] = "\x02\x42\x5e", [0x1052] = "\x0e\x46\x5d",
  [0x1053] = "\x02\x49\x79", [0x1054] = "\x01\x68\x5f",
  [0x1055] = "\x01\x68\x5d", [0x1057] = "\x01\x68\x5e",
  [0x1058] = "\x02\x49\x77", [0x1059] = "\x02\x49\x78",
  [0x105a] = "\x0e\x4b\x6d", [0x105b] = "\x01\x68\x5b",
  [0x105c] = "\x02\x50\x3a", [0x105d] = "\x02\x50\x38",
  [0x105e] = "\x0e\x4b\x70", [0x105f] = "\x01\x6c\x57",
  [0x1060] = "\x02\x50\x39", [0x1061] = "\x01\x6c\x58",
  [0x1062] = "\x01\x6c\x56", [0x1063] = "\x01\x68\x5c",
  [0x1065] = "\x0e\x64\x73", [0x1066] = "\x02\x56\x77",
  [0x1067] = "\x02\x56\x75", [0x1068] = "\x02\x56\x76",
  [0x1069] = "\x02\x50\x37", [0x106a] = "\x02\x5c\x68",
  [0x106b] = "\x01\x73\x59", [0x106c] = "\x02\x5c\x67",
  [0x106d] = "\x02\x61\x51", [0x106e] = "\x02\x61\x52",
  [0x106f] = "\x02\x56\x78", [0x1070] = "\x02\x65\x4a",
  [0x1071] = "\x0e\x66\x77", [0x1072] = "\x01\x44\x5b",
  [0x1073] = "\x01\x46\x4d", [0x1074] = "\x01\x48\x21",
  [0x1075] = "\x02\x22\x3b", [0x1076] = "\x01\x47\x7e",
  [0x1077] = "\x0e\x28\x63", [0x1078] = "\x01\x4d\x31",
  [0x1079] = "\x01\x63\x6c", [0x107a] = "\x0e\x21\x3d",
  [0x107b] = "\x01\x45\x3c", [0x107c] = "\x01\x46\x4e",
  [0x107d] = "\x01\x50\x77", [0x107e] = "\x01\x5e\x7d",
  [0x107f] = "\x0e\x21\x3e", [0x1080] = "\x02\x21\x5a",
  [0x1081] = "\x0e\x22\x47", [0x1082] = "\x02\x21\x5b",
  [0x1083] = "\x0e\x22\x48", [0x1084] = "\x02\x22\x3c",
  [0x1085] = "\x0e\x66\x78", [0x1087] = "\x01\x4a\x2a",
  [0x1088] = "\x02\x23\x70", [0x1089] = "\x02\x23\x6e",
  [0x108a] = "\x01\x4a\x2b", [0x108b] = "\x02\x23\x6d",
  [0x108c] = "\x02\x23\x6f", [0x108d] = "\x02\x23\x71",
  [0x108f] = "\x01\x4a\x29", [0x1090] = "\x0e\x25\x6e",
  [0x1092] = "\x0e\x25\x6d", [0x1095] = "\x01\x4d\x35",
  [0x1096] = "\x01\x4d\x36", [0x1097] = "\x01\x4d\x33",
  [0x1098] = "\x0e\x28\x65", [0x1099] = "\x0e\x28\x67",
  [0x109a] = "\x01\x4d\x32", [0x109b] = "\x02\x2a\x39",
  [0x109c] = "\x01\x4d\x34", [0x109d] = "\x0e\x28\x68",
  [0x10a0] = "\x01\x50\x78", [0x10a1] = "\x0e\x2c\x48",
  [0x10a2] = "\x02\x2a\x38", [0x10a3] = "\x02\x2a\x3a",
  [0x10a4] = "\x02\x2a\x37", [0x10a5] = "\x02\x2a\x3b",
  [0x10a6] = "\x01\x50\x79", [0x10a7] = "\x01\x55\x3d",
  [0x10a8] = "\x02\x2e\x77", [0x10a9] = "\x0e\x30\x6c",
  [0x10aa] = "\x02\x2e\x79", [0x10ab] = "\x01\x55\x3b",
  [0x10ac] = "\x02\x2e\x7a", [0x10ad] = "\x01\x55\x3c",
  [0x10ae] = "\x02\x2e\x78", [0x10af] = "\x0e\x30\x6b",
  [0x10b0] = "\x02\x2a\x36", [0x10b1] = "\x02\x35\x2b",
  [0x10b2] = "\x02\x35\x2e", [0x10b3] = "\x02\x35\x2f",
  [0x10b4] = "\x02\x35\x2c", [0x10b5] = "\x01\x5a\x2d",
  [0x10b6] = "\x01\x5a\x2c", [0x10b7] = "\x01\x5a\x2a",
  [0x10b8] = "\x01\x5a\x2b", [0x10b9] = "\x02\x35\x2d",
  [0x10ba] = "\x0e\x35\x61", [0x10bb] = "\x0e\x35\x60",
  [0x10bd] = "\x0e\x3b\x3f", [0x10be] = "\x01\x5a\x2e",
  [0x10bf] = "\x0e\x3b\x40", [0x10c0] = "\x0e\x3b\x3e",
  [0x10c1] = "\x01\x5f\x21", [0x10c2] = "\x01\x5f\x22",
  [0x10c4] = "\x01\x5f\x23", [0x10c5] = "\x02\x42\x63",
  [0x10c6] = "\x02\x42\x65", [0x10c7] = "\x02\x42\x67",
  [0x10c8] = "\x01\x63\x6e", [0x10c9] = "\x01\x63\x6d",
  [0x10ca] = "\x01\x5e\x7e", [0x10cb] = "\x02\x42\x66",
  [0x10cc] = "\x02\x42\x64", [0x10cd] = "\x0e\x64\x74",
  [0x10ce] = "\x02\x49\x7d", [0x10cf] = "\x0e\x46\x61",
  [0x10d0] = "\x0e\x46\x60", [0x10d1] = "\x02\x49\x7b",
  [0x10d2] = "\x02\x4a\x23", [0x10d3] = "\x01\x68\x60",
  [0x10d4] = "\x02\x4a\x24", [0x10d5] = "\x02\x4a\x21",
  [0x10d6] = "\x01\x68\x61", [0x10d7] = "\x02\x49\x7c",
  [0x10d8] = "\x02\x49\x7a", [0x10d9] = "\x02\x4a\x22",
  [0x10da] = "\x01\x6c\x5a", [0x10db] = "\x02\x50\x3c",
  [0x10dc] = "\x02\x49\x7e", [0x10dd] = "\x01\x6c\x5c",
  [0x10de] = "\x02\x50\x3d", [0x10df] = "\x01\x6c\x5b",
  [0x10e0] = "\x01\x6c\x5e", [0x10e1] = "\x02\x50\x3e",
  [0x10e2] = "\x01\x6c\x59", [0x10e3] = "\x01\x6c\x5d",
  [0x10e5] = "\x02\x56\x7d", [0x10e6] = "\x02\x56\x7b",
  [0x10e7] = "\x02\x56\x7a", [0x10e8] = "\x02\x56\x7c",
  [0x10e9] = "\x02\x56\x79", [0x10ea] = "\x0e\x50\x5c",
  [0x10ec] = "\x01\x78\x33", [0x10ed] = "\x0e\x64\x75",
  [0x10ee] = "\x02\x68\x72", [0x10ef] = "\x02\x68\x73",
  [0x10f0] = "\x0e\x5c\x3f", [0x10f1] = "\x02\x6b\x59",
  [0x10f2] = "\x02\x6d\x4d", [0x10f3] = "\x01\x7d\x22",
  [0x10f4] = "\x0e\x21\x3f", [0x10f5] = "\x0e\x23\x61",
  [0x10f6] = "\x01\x4d\x37", [0x10f7] = "\x01\x4a\x2c",
  [0x10f8] = "\x0e\x28\x69", [0x10f9] = "\x0e\x28\x6b",
  [0x10fa] = "\x01\x50\x7a", [0x10fb] = "\x0e\x2c\x4c",
  [0x10fc] = "\x0e\x2c\x4b", [0x10fd] = "\x0e\x30\x6e",
  [0x10fe] = "\x01\x44\x5c", [0x10ff] = "\x01\x45\x3d",
  [0x1101] = "\x01\x46\x4f", [0x1102] = "\x02\x22\x3d",
  [0x1103] = "\x0e\x25\x6f", [0x1104] = "\x01\x4a\x2d",
  [0x1105] = "\x02\x23\x72", [0x1106] = "\x0e\x28\x6d",
  [0x1107] = "\x02\x2a\x3c", [0x1108] = "\x01\x50\x7b",
  [0x110a] = "\x01\x68\x62", [0x110b] = "\x01\x44\x5d",
  [0x110c] = "\x0e\x21\x64", [0x110d] = "\x0e\x22\x49",
  [0x110e] = "\x0e\x23\x62", [0x110f] = "\x01\x48\x22",
  [0x1111] = "\x0e\x3b\x42", [0x1112] = "\x01\x63\x6f",
  [0x1113] = "\x01\x44\x5e", [0x1114] = "\x01\x45\x3e",
  [0x1115] = "\x01\x45\x3f", [0x1117] = "\x01\x46\x51",
  [0x1118] = "\x01\x46\x50", [0x111a] = "\x02\x22\x3e",
  [0x111b] = "\x01\x48\x23", [0x111d] = "\x02\x23\x73",
  [0x111e] = "\x0e\x25\x71", [0x111f] = "\x01\x4a\x2e",
  [0x1121] = "\x0e\x28\x6e", [0x1122] = "\x02\x26\x52",
  [0x1123] = "\x02\x26\x53", [0x1124] = "\x02\x26\x54",
  [0x1125] = "\x0e\x28\x70", [0x1126] = "\x01\x4d\x38",
  [0x1127] = "\x01\x4d\x39", [0x1128] = "\x02\x26\x51",
  [0x1129] = "\x01\x4d\x3a", [0x112d] = "\x01\x50\x7c",
  [0x112e] = "\x02\x2a\x3d", [0x112f] = "\x0e\x2c\x4d",
  [0x1130] = "\x02\x2e\x7c", [0x1131] = "\x01\x55\x3e",
  [0x1132] = "\x0e\x30\x6f", [0x1133] = "\x02\x2e\x7b",
  [0x1134] = "\x0e\x35\x62", [0x1135] = "\x01\x5a\x2f",
  [0x1136] = "\x02\x35\x30", [0x1137] = "\x01\x5a\x30",
  [0x1138] = "\x02\x35\x31", [0x1139] = "\x0e\x35\x63",
  [0x113a] = "\x0e\x3b\x47", [0x113b] = "\x0e\x3b\x45",
  [0x113c] = "\x01\x5f\x24", [0x113e] = "\x0e\x3b\x44",
  [0x1140] = "\x02\x42\x68", [0x1142] = "\x0e\x41\x22",
  [0x1143] = "\x02\x4a\x26", [0x1144] = "\x02\x4a\x25",
  [0x1146] = "\x01\x68\x63", [0x1148] = "\x01\x6c\x5f",
  [0x1149] = "\x02\x50\x3f", [0x114a] = "\x01\x70\x53",
  [0x114b] = "\x02\x56\x7e", [0x114c] = "\x01\x73\x5a",
  [0x114d] = "\x0e\x57\x63", [0x114e] = "\x01\x7b\x5f",
  [0x114f] = "\x02\x6f\x27", [0x1150] = "\x0e\x21\x40",
  [0x1151] = "\x0e\x21\x41", [0x1152] = "\x0e\x22\x4a",
  [0x1153] = "\x0e\x23\x51", [0x1154] = "\x02\x26\x55",
  [0x1155] = "\x0e\x28\x71", [0x1156] = "\x02\x2a\x3e",
  [0x1157] = "\x01\x5a\x31", [0x1158] = "\x02\x3b\x58",
  [0x1159] = "\x01\x63\x70", [0x115a] = "\x0e\x41\x23",
  [0x115b] = "\x0e\x50\x5e", [0x115d] = "\x0e\x57\x64",
  [0x115e] = "\x01\x76\x41", [0x1160] = "\x0e\x61\x64",
  [0x1161] = "\x0e\x21\x42", [0x1162] = "\x01\x4a\x30",
  [0x1163] = "\x0e\x25\x72", [0x1164] = "\x01\x4a\x2f",
  [0x1165] = "\x01\x50\x7d", [0x1166] = "\x0e\x2c\x50",
  [0x1167] = "\x02\x2e\x7d", [0x1169] = "\x01\x5a\x33",
  [0x116a] = "\x01\x5d\x29", [0x116b] = "\x01\x5a\x34",
  [0x116c] = "\x01\x5a\x32", [0x116d] = "\x01\x5f\x25",
  [0x116e] = "\x0e\x41\x24", [0x116f] = "\x02\x4a\x27",
  [0x1170] = "\x01\x68\x64", [0x1171] = "\x01\x6c\x60",
  [0x1172] = "\x0e\x5f\x45", [0x1173] = "\x02\x21\x2c",
  [0x1174] = "\x02\x22\x3f", [0x1176] = "\x02\x23\x75",
  [0x1177] = "\x01\x4a\x31", [0x1178] = "\x02\x23\x74",
  [0x1179] = "\x01\x4a\x32", [0x117c] = "\x01\x4d\x3e",
  [0x117d] = "\x02\x26\x58", [0x117e] = "\x02\x26\x57",
  [0x117f] = "\x01\x4d\x3d", [0x1180] = "\x01\x4d\x3b",
  [0x1181] = "\x01\x4d\x3c", [0x1182] = "\x02\x26\x56",
  [0x1183] = "\x0e\x28\x76", [0x1184] = "\x0e\x28\x75",
  [0x1185] = "\x01\x51\x21", [0x1186] = "\x02\x2a\x3f",
  [0x1187] = "\x01\x51\x24", [0x1188] = "\x01\x50\x7e",
  [0x1189] = "\x01\x51\x26", [0x118a] = "\x01\x51\x22",
  [0x118b] = "\x01\x51\x23", [0x118c] = "\x01\x51\x25",
  [0x118d] = "\x0e\x62\x64", [0x118e] = "\x0e\x30\x70",
  [0x118f] = "\x0e\x30\x72", [0x1190] = "\x01\x55\x41",
  [0x1191] = "\x01\x55\x40", [0x1192] = "\x01\x55\x3f",
  [0x1193] = "\x0e\x30\x71", [0x1194] = "\x0e\x64\x76",
  [0x1196] = "\x02\x35\x33", [0x1197] = "\x01\x5a\x35",
  [0x1198] = "\x01\x5a\x38", [0x1199] = "\x01\x5a\x36",
  [0x119a] = "\x0e\x64\x77", [0x119b] = "\x02\x35\x32",
  [0x119c] = "\x01\x5a\x3b", [0x119e] = "\x01\x5a\x37",
  [0x119f] = "\x02\x35\x34", [0x11a0] = "\x01\x5a\x3a",
  [0x11a1] = "\x01\x5a\x39", [0x11a5] = "\x02\x3b\x5a",
  [0x11a6] = "\x02\x3b\x59", [0x11a7] = "\x0e\x3b\x4b",
  [0x11a8] = "\x01\x5f\x28", [0x11a9] = "\x01\x5f\x26",
  [0x11aa] = "\x01\x5f\x27", [0x11ab] = "\x02\x3b\x5b",
  [0x11ac] = "\x01\x63\x71", [0x11ad] = "\x02\x42\x6a",
  [0x11ae] = "\x01\x63\x72", [0x11af] = "\x02\x42\x69",
  [0x11b2] = "\x02\x50\x40", [0x11b3] = "\x0e\x46\x63",
  [0x11b4] = "\x0e\x46\x64", [0x11b5] = "\x01\x6c\x62",
  [0x11b6] = "\x02\x4a\x28", [0x11b7] = "\x01\x6c\x61",
  [0x11b8] = "\x0e\x4b\x72", [0x11b9] = "\x01\x68\x65",
  [0x11ba] = "\x0e\x64\x78", [0x11bb] = "\x02\x57\x22",
  [0x11bc] = "\x02\x57\x21", [0x11bd] = "\x01\x73\x5b",
  [0x11be] = "\x02\x5c\x69", [0x11bf] = "\x02\x65\x4b",
  [0x11c0] = "\x02\x68\x74", [0x11c1] = "\x02\x68\x75",
  [0x11c2] = "\x0e\x5e\x2f", [0x11c3] = "\x01\x45\x40",
  [0x11c5] = "\x01\x46\x52", [0x11c7] = "\x0e\x22\x4d",
  [0x11c8] = "\x0e\x62\x51", [0x11c9] = "\x02\x21\x5c",
  [0x11ca] = "\x0e\x22\x4c", [0x11cc] = "\x01\x4a\x34",
  [0x11cd] = "\x01\x4a\x36", [0x11ce] = "\x0e\x25\x78",
  [0x11cf] = "\x02\x22\x42", [0x11d0] = "\x02\x23\x78",
  [0x11d1] = "\x02\x23\x77", [0x11d2] = "\x02\x23\x76",
  [0x11d3] = "\x0e\x23\x66", [0x11d4] = "\x02\x22\x41",
  [0x11d5] = "\x02\x22\x40", [0x11d6] = "\x01\x48\x25",
  [0x11d7] = "\x01\x4a\x35", [0x11d8] = "\x01\x4a\x33",
  [0x11d9] = "\x01\x48\x24", [0x11da] = "\x0e\x23\x67",
  [0x11db] = "\x0e\x23\x69", [0x11dc] = "\x0e\x25\x77",
  [0x11dd] = "\x01\x4d\x3f", [0x11de] = "\x02\x26\x59",
  [0x11df] = "\x0e\x25\x7b", [0x11e0] = "\x01\x4d\x40",
  [0x11e1] = "\x02\x23\x7d", [0x11e3] = "\x02\x24\x21",
  [0x11e4] = "\x02\x23\x7e", [0x11e5] = "\x02\x26\x5a",
  [0x11e6] = "\x0e\x25\x76", [0x11e7] = "\x0e\x25\x74",
  [0x11e8] = "\x02\x23\x7a", [0x11e9] = "\x0e\x28\x7e",
  [0x11ea] = "\x01\x4a\x3a", [0x11eb] = "\x01\x4a\x38",
  [0x11ec] = "\x0e\x25\x7c", [0x11ed] = "\x02\x23\x79",
  [0x11ee] = "\x02\x23\x7b", [0x11ef] = "\x02\x24\x23",
  [0x11f0] = "\x0e\x26\x22", [0x11f1] = "\x01\x4a\x37",
  [0x11f2] = "\x0e\x26\x21", [0x11f3] = "\x02\x23\x7c",
  [0x11f4] = "\x02\x24\x27", [0x11f5] = "\x01\x4d\x42",
  [0x11f6] = "\x0e\x25\x7d", [0x11f7] = "\x02\x24\x24",
  [0x11f8] = "\x01\x4a\x39", [0x11f9] = "\x0e\x26\x23",
  [0x11fa] = "\x02\x24\x22", [0x11fb] = "\x02\x24\x25",
  [0x11fc] = "\x0e\x25\x7a", [0x11fd] = "\x01\x4d\x41",
  [0x11fe] = "\x0e\x25\x75", [0x11ff] = "\x01\x4d\x43",
  [0x1200] = "\x02\x24\x26", [0x1207] = "\x0e\x28\x78",
  [0x1208] = "\x0e\x29\x22", [0x1209] = "\x02\x26\x6e",
  [0x120a] = "\x02\x26\x61", [0x120b] = "\x02\x26\x5f",
  [0x120c] = "\x02\x26\x6d", [0x120d] = "\x02\x26\x68",
  [0x120e] = "\x01\x51\x2b", [0x120f] = "\x01\x4d\x44",
  [0x1210] = "\x02\x26\x69", [0x1211] = "\x02\x26\x6c",
  [0x1212] = "\x01\x51\x27", [0x1213] = "\x02\x26\x6b",
  [0x1214] = "\x01\x4d\x45", [0x1215] = "\x01\x4d\x4a",
  [0x1216] = "\x01\x4d\x48", [0x1217] = "\x02\x26\x62",
  [0x1218] = "\x0e\x62\x66", [0x1219] = "\x02\x26\x5d",
  [0x121a] = "\x02\x26\x64", [0x121b] = "\x01\x4d\x4f",
  [0x121c] = "\x02\x26\x6f", [0x121d] = "\x01\x51\x28",
  [0x121e] = "\x02\x26\x65", [0x1220] = "\x01\x51\x29",
  [0x1221] = "\x01\x4d\x4b", [0x1222] = "\x02\x26\x67",
  [0x1223] = "\x0e\x2c\x5a", [0x1224] = "\x02\x2a\x4e",
  [0x1225] = "\x01\x51\x2a", [0x1226] = "\x02\x26\x5c",
  [0x1227] = "\x01\x4d\x4c", [0x1228] = "\x01\x51\x2c",
  [0x1229] = "\x01\x4d\x4d", [0x122a] = "\x01\x4d\x49",
  [0x122b] = "\x01\x4d\x4e", [0x122c] = "\x02\x26\x66",
  [0x122d] = "\x02\x26\x5b", [0x122e] = "\x02\x26\x6a",
  [0x122f] = "\x01\x4d\x46", [0x1230] = "\x0e\x28\x77",
  [0x1231] = "\x0e\x2c\x5b", [0x1232] = "\x02\x26\x5e",
  [0x1233] = "\x02\x26\x63", [0x1234] = "\x02\x26\x60",
  [0x1235] = "\x01\x4d\x47", [0x1236] = "\x0e\x28\x7a",
  [0x1237] = "\x02\x2a\x40", [0x1239] = "\x02\x2a\x41",
  [0x123d] = "\x0e\x29\x21", [0x1240] = "\x02\x2a\x4b",
  [0x1241] = "\x02\x2f\x23", [0x1242] = "\x02\x2a\x4c",
  [0x1243] = "\x01\x51\x32", [0x1244] = "\x02\x2a\x4f",
  [0x1245] = "\x02\x2a\x45", [0x1246] = "\x01\x51\x31",
  [0x1247] = "\x02\x2a\x47", [0x1248] = "\x0e\x62\x67",
  [0x1249] = "\x02\x2a\x48", [0x124a] = "\x0e\x2c\x60",
  [0x124b] = "\x0e\x30\x79", [0x124c] = "\x02\x2a\x4a",
  [0x124d] = "\x01\x51\x2d", [0x124e] = "\x0e\x2c\x56",
  [0x124f] = "\x0e\x30\x73", [0x1250] = "\x01\x55\x45",
  [0x1251] = "\x0e\x2c\x57", [0x1252] = "\x0e\x2c\x5d",
  [0x1253] = "\x02\x2a\x46", [0x1254] = "\x02\x2a\x42",
  [0x1255] = "\x01\x55\x46", [0x1256] = "\x0e\x30\x7e",
  [0x1258] = "\x02\x2a\x50", [0x1259] = "\x01\x55\x42",
  [0x125a] = "\x02\x2f\x21", [0x125b] = "\x02\x2a\x49",
  [0x125c] = "\x0e\x2c\x54", [0x125d] = "\x02\x2e\x7e",
  [0x125e] = "\x02\x2a\x44", [0x125f] = "\x02\x2a\x4d",
  [0x1260] = "\x0e\x2c\x5f", [0x1261] = "\x0e\x2c\x61",
  [0x1262] = "\x01\x51\x30", [0x1263] = "\x01\x55\x43",
  [0x1264] = "\x01\x51\x36", [0x1265] = "\x01\x55\x44",
  [0x1266] = "\x02\x2a\x51", [0x1267] = "\x02\x2f\x22",
  [0x1268] = "\x01\x51\x2f", [0x1269] = "\x01\x55\x48",
  [0x126a] = "\x01\x51\x35", [0x126b] = "\x01\x51\x34",
  [0x126c] = "\x01\x51\x33", [0x126d] = "\x01\x55\x47",
  [0x126e] = "\x02\x2a\x52", [0x126f] = "\x01\x55\x49",
  [0x1270] = "\x01\x51\x2e", [0x1272] = "\x02\x2a\x43",
  [0x1273] = "\x0e\x30\x7a", [0x1274] = "\x0e\x30\x78",
  [0x1275] = "\x0e\x30\x7b", [0x127c] = "\x0e\x2c\x5c",
  [0x127e] = "\x0e\x30\x76", [0x127f] = "\x01\x5a\x3c",
  [0x1280] = "\x02\x2f\x26", [0x1281] = "\x02\x2f\x28",
  [0x1283] = "\x02\x2f\x2a", [0x1284] = "\x01\x55\x4a",
  [0x1285] = "\x01\x55\x50", [0x1286] = "\x02\x35\x37",
  [0x1287] = "\x02\x2f\x2e", [0x1288] = "\x02\x2f\x25",
  [0x1289] = "\x01\x5a\x3e", [0x128a] = "\x02\x35\x35",
  [0x128b] = "\x0e\x31\x22", [0x128c] = "\x01\x55\x4f",
  [0x128d] = "\x01\x55\x4d", [0x128e] = "\x02\x2f\x30",
  [0x1290] = "\x02\x35\x36", [0x1292] = "\x02\x2f\x27",
  [0x1293] = "\x0e\x31\x25", [0x1294] = "\x01\x55\x4e",
  [0x1295] = "\x02\x2f\x2b", [0x1296] = "\x01\x55\x51",
  [0x1297] = "\x02\x2f\x2d", [0x1298] = "\x0e\x35\x68",
  [0x1299] = "\x0e\x30\x74", [0x129a] = "\x01\x55\x4c",
  [0x129b] = "\x02\x2f\x2c", [0x129c] = "\x02\x2f\x2f",
  [0x129d] = "\x02\x2f\x29", [0x129e] = "\x0e\x31\x27",
  [0x129f] = "\x01\x55\x4b", [0x12a0] = "\x01\x5a\x3f",
  [0x12a1] = "\x0e\x35\x71", [0x12a2] = "\x02\x2f\x24",
  [0x12a3] = "\x01\x5a\x3d", [0x12a4] = "\x0e\x35\x72",
  [0x12a6] = "\x0e\x31\x23", [0x12a7] = "\x0e\x31\x28",
  [0x12a8] = "\x01\x5a\x40", [0x12a9] = "\x0e\x31\x21",
  [0x12aa] = "\x0e\x35\x6f", [0x12ae] = "\x0e\x31\x26",
  [0x12b0] = "\x02\x35\x39", [0x12b1] = "\x02\x35\x42",
  [0x12b2] = "\x01\x5f\x2b", [0x12b3] = "\x0e\x3b\x54",
  [0x12b4] = "\x01\x5a\x42", [0x12b5] = "\x01\x5a\x47",
  [0x12b6] = "\x01\x5f\x2c", [0x12b7] = "\x02\x35\x44",
  [0x12b8] = "\x01\x5a\x4e", [0x12b9] = "\x02\x3b\x5d",
  [0x12ba] = "\x02\x35\x3a", [0x12bb] = "\x01\x5a\x46",
  [0x12bc] = "\x01\x5a\x49", [0x12bd] = "\x01\x5a\x44",
  [0x12be] = "\x02\x35\x38", [0x12bf] = "\x02\x35\x46",
  [0x12c0] = "\x02\x35\x49", [0x12c1] = "\x02\x3b\x6c",
  [0x12c3] = "\x02\x35\x47", [0x12c4] = "\x02\x3b\x61",
  [0x12c5] = "\x01\x5a\x45", [0x12c6] = "\x01\x5a\x4c",
  [0x12c7] = "\x01\x5a\x50", [0x12c8] = "\x02\x35\x41",
  [0x12c9] = "\x02\x3b\x5c", [0x12ca] = "\x02\x35\x45",
  [0x12cb] = "\x01\x5a\x41", [0x12cc] = "\x02\x3b\x5e",
  [0x12cd] = "\x02\x35\x48", [0x12ce] = "\x02\x3b\x60",
  [0x12cf] = "\x02\x35\x3d", [0x12d0] = "\x0e\x35\x6a",
  [0x12d1] = "\x01\x5f\x29", [0x12d2] = "\x0e\x3b\x56",
  [0x12d3] = "\x02\x35\x3b", [0x12d4] = "\x02\x35\x3c",
  [0x12d5] = "\x01\x5a\x4b", [0x12d6] = "\x0e\x3b\x55",
  [0x12d7] = "\x0e\x35\x6e", [0x12d8] = "\x01\x5a\x4a",
  [0x12d9] = "\x02\x35\x3f", [0x12da] = "\x01\x5a\x4f",
  [0x12db] = "\x02\x35\x43", [0x12dc] = "\x01\x5a\x48",
  [0x12dd] = "\x02\x35\x40", [0x12de] = "\x0e\x35\x79",
  [0x12df] = "\x01\x5a\x4d", [0x12e0] = "\x01\x5f\x2d",
  [0x12e1] = "\x01\x5f\x2a", [0x12e2] = "\x02\x3b\x5f",
  [0x12e3] = "\x0e\x3b\x58", [0x12e4] = "\x02\x35\x3e",
  [0x12e5] = "\x0e\x3b\x59", [0x12e6] = "\x01\x5a\x43",
  [0x12e7] = "\x0e\x35\x76", [0x12e9] = "\x0e\x3b\x5a",
  [0x12ea] = "\x0e\x3b\x4d", [0x12ee] = "\x0e\x35\x74",
  [0x12f0] = "\x01\x5f\x32", [0x12f1] = "\x01\x5f\x36",
  [0x12f2] = "\x02\x3b\x63", [0x12f3] = "\x01\x63\x77",
  [0x12f4] = "\x01\x5f\x34", [0x12f5] = "\x02\x3b\x67",
  [0x12f6] = "\x01\x5f\x38", [0x12f7] = "\x02\x42\x6b",
  [0x12f8] = "\x02\x3b\x69", [0x12f9] = "\x01\x63\x79",
  [0x12fa] = "\x01\x5f\x30", [0x12fb] = "\x01\x5f\x33",
  [0x12fc] = "\x02\x3b\x6a", [0x12fd] = "\x0e\x3b\x5e",
  [0x12fe] = "\x02\x3b\x6b", [0x12ff] = "\x02\x3b\x71",
  [0x1300] = "\x01\x5f\x3a", [0x1301] = "\x01\x63\x7a",
  [0x1303] = "\x02\x3b\x6d", [0x1304] = "\x02\x3b\x72",
  [0x1305] = "\x02\x3b\x66", [0x1306] = "\x01\x64\x26",
  [0x1307] = "\x0e\x3b\x4f", [0x1308] = "\x01\x63\x7b",
  [0x1309] = "\x01\x5f\x39", [0x130a] = "\x02\x3b\x64",
  [0x130b] = "\x02\x3b\x73", [0x130c] = "\x0e\x3b\x51",
  [0x130d] = "\x01\x64\x25", [0x130e] = "\x01\x5f\x37",
  [0x130f] = "\x01\x63\x74", [0x1310] = "\x02\x3b\x70",
  [0x1311] = "\x0e\x3b\x5d", [0x1312] = "\x01\x5f\x3b",
  [0x1313] = "\x02\x3b\x68", [0x1314] = "\x02\x3b\x62",
  [0x1315] = "\x01\x5f\x31", [0x1316] = "\x02\x3b\x65",
  [0x1318] = "\x02\x3b\x6e", [0x1319] = "\x0e\x41\x2b",
  [0x131a] = "\x01\x63\x73", [0x131b] = "\x01\x63\x78",
  [0x131c] = "\x01\x5f\x2e", [0x131d] = "\x02\x3b\x6f",
  [0x131e] = "\x0e\x3b\x61", [0x131f] = "\x01\x63\x76",
  [0x1320] = "\x0e\x3b\x62", [0x1321] = "\x0e\x3b\x63",
  [0x1322] = "\x0e\x3b\x50", [0x1323] = "\x01\x5f\x2f",
  [0x1325] = "\x0e\x64\x79", [0x1327] = "\x01\x64\x24",
  [0x1328] = "\x02\x4a\x2a", [0x1329] = "\x02\x42\x76",
  [0x132a] = "\x0e\x41\x29", [0x132b] = "\x02\x42\x6e",
  [0x132c] = "\x02\x4a\x29", [0x132e] = "\x02\x42\x72",
  [0x132f] = "\x02\x42\x74", [0x1330] = "\x0e\x41\x27",
  [0x1331] = "\x0e\x41\x2c", [0x1332] = "\x02\x42\x71",
  [0x1333] = "\x0e\x46\x6f", [0x1334] = "\x01\x64\x23",
  [0x1336] = "\x02\x42\x70", [0x1337] = "\x01\x64\x27",
  [0x1338] = "\x0e\x46\x6e", [0x133a] = "\x0e\x41\x28",
  [0x133b] = "\x02\x4a\x39", [0x133c] = "\x0e\x46\x6c",
  [0x133d] = "\x0e\x41\x2e", [0x133e] = "\x01\x64\x22",
  [0x133f] = "\x01\x68\x67", [0x1340] = "\x02\x42\x77",
  [0x1341] = "\x02\x4a\x2b", [0x1342] = "\x0e\x46\x6d",
  [0x1343] = "\x0e\x41\x2a", [0x1344] = "\x01\x63\x7e",
  [0x1345] = "\x02\x42\x6f", [0x1346] = "\x02\x42\x73",
  [0x1347] = "\x01\x68\x66", [0x1348] = "\x01\x63\x75",
  [0x1349] = "\x02\x42\x6c", [0x134a] = "\x02\x42\x6d",
  [0x134b] = "\x01\x68\x68", [0x134c] = "\x01\x63\x7d",
  [0x134d] = "\x01\x64\x21", [0x134e] = "\x01\x63\x7c",
  [0x134f] = "\x02\x42\x75", [0x1350] = "\x0e\x64\x7a",
  [0x1352] = "\x02\x4a\x2f", [0x1353] = "\x02\x4a\x30",
  [0x1354] = "\x02\x4a\x35", [0x1355] = "\x01\x6c\x67",
  [0x1356] = "\x02\x4a\x3c", [0x1357] = "\x0e\x4b\x73",
  [0x1358] = "\x01\x68\x6e", [0x1359] = "\x0e\x4b\x7e",
  [0x135a] = "\x01\x68\x6d", [0x135b] = "\x02\x4a\x37",
  [0x135c] = "\x0e\x4b\x74", [0x135d] = "\x01\x6c\x66",
  [0x135e] = "\x02\x4a\x2c", [0x135f] = "\x01\x68\x6c",
  [0x1360] = "\x0e\x46\x71", [0x1361] = "\x02\x4a\x3b",
  [0x1362] = "\x01\x68\x6a", [0x1363] = "\x01\x68\x6b",
  [0x1364] = "\x0e\x4b\x7c", [0x1365] = "\x02\x4a\x38",
  [0x1366] = "\x02\x50\x51", [0x1367] = "\x01\x6c\x64",
  [0x1368] = "\x01\x5f\x35", [0x1369] = "\x0e\x46\x72",
  [0x136a] = "\x02\x4a\x3a", [0x136b] = "\x01\x6c\x6b",
  [0x136c] = "\x02\x4a\x32", [0x136e] = "\x01\x6c\x65",
  [0x136f] = "\x0e\x46\x6a", [0x1370] = "\x01\x6c\x6a",
  [0x1371] = "\x02\x4a\x2d", [0x1372] = "\x02\x4a\x31",
  [0x1373] = "\x02\x4a\x2e", [0x1374] = "\x02\x4a\x34",
  [0x1375] = "\x01\x68\x6f", [0x1376] = "\x01\x6c\x63",
  [0x1377] = "\x01\x68\x69", [0x1378] = "\x0e\x4c\x21",
  [0x1379] = "\x02\x50\x43", [0x137a] = "\x02\x4a\x36",
  [0x137b] = "\x0e\x46\x68", [0x137c] = "\x01\x6c\x69",
  [0x137d] = "\x0e\x46\x73", [0x137e] = "\x01\x6c\x6c",
  [0x1380] = "\x02\x4a\x33", [0x1381] = "\x0e\x46\x6b",
  [0x1382] = "\x01\x6c\x68", [0x1383] = "\x02\x50\x42",
  [0x1385] = "\x0e\x4c\x23", [0x1386] = "\x0e\x46\x74",
  [0x1387] = "\x0e\x4c\x24", [0x1388] = "\x0e\x4b\x77",
  [0x1389] = "\x02\x50\x47", [0x138a] = "\x01\x70\x57",
  [0x138b] = "\x02\x50\x41", [0x138c] = "\x02\x57\x2e",
  [0x138d] = "\x02\x50\x50", [0x138e] = "\x01\x6c\x70",
  [0x138f] = "\x0e\x4b\x7a", [0x1390] = "\x01\x6c\x6e",
  [0x1391] = "\x01\x70\x55", [0x1392] = "\x02\x50\x4d",
  [0x1393] = "\x02\x50\x49", [0x1394] = "\x01\x6c\x74",
  [0x1395] = "\x0e\x4b\x76", [0x1396] = "\x02\x57\x25",
  [0x1397] = "\x0e\x50\x68", [0x1398] = "\x0e\x4c\x28",
  [0x1399] = "\x0e\x50\x67", [0x139a] = "\x01\x6c\x72",
  [0x139b] = "\x02\x50\x48", [0x139c] = "\x0e\x4c\x29",
  [0x139d] = "\x02\x57\x23", [0x139e] = "\x0e\x4c\x25",
  [0x139f] = "\x02\x50\x4c", [0x13a1] = "\x02\x50\x4f",
  [0x13a2] = "\x02\x50\x46", [0x13a3] = "\x0e\x4b\x79",
  [0x13a4] = "\x01\x6c\x73", [0x13a6] = "\x0e\x4b\x75",
  [0x13a7] = "\x01\x6c\x6d", [0x13a8] = "\x02\x57\x24",
  [0x13a9] = "\x01\x70\x56", [0x13aa] = "\x02\x50\x4e",
  [0x13ab] = "\x01\x6c\x6f", [0x13ac] = "\x01\x6c\x71",
  [0x13ad] = "\x02\x50\x4b", [0x13ae] = "\x01\x6c\x75",
  [0x13af] = "\x02\x50\x4a", [0x13b0] = "\x02\x50\x45",
  [0x13b1] = "\x02\x50\x44", [0x13b2] = "\x01\x70\x54",
  [0x13b3] = "\x02\x50\x52", [0x13b4] = "\x02\x57\x27",
  [0x13b5] = "\x02\x5c\x6b", [0x13b6] = "\x01\x70\x59",
  [0x13b7] = "\x0e\x50\x61", [0x13b8] = "\x02\x57\x2d",
  [0x13b9] = "\x0e\x50\x63", [0x13ba] = "\x02\x57\x2b",
  [0x13bb] = "\x0e\x50\x6a", [0x13bc] = "\x02\x5c\x6c",
  [0x13bd] = "\x0e\x50\x64", [0x13be] = "\x01\x70\x5a",
  [0x13bf] = "\x02\x57\x2c", [0x13c0] = "\x0e\x50\x66",
  [0x13c1] = "\x02\x57\x29", [0x13c2] = "\x01\x73\x5d",
  [0x13c3] = "\x02\x5c\x6a", [0x13c4] = "\x0e\x50\x6c",
  [0x13c5] = "\x02\x57\x26", [0x13c6] = "\x02\x57\x28",
  [0x13c7] = "\x01\x73\x5e", [0x13c8] = "\x01\x70\x5c",
  [0x13c9] = "\x01\x73\x5c", [0x13ca] = "\x01\x70\x5b",
  [0x13cb] = "\x01\x73\x60", [0x13cc] = "\x02\x57\x2a",
  [0x13cd] = "\x01\x70\x58", [0x13ce] = "\x0e\x50\x62",
  [0x13cf] = "\x0e\x50\x65", [0x13d0] = "\x0e\x50\x6b",
  [0x13d3] = "\x0e\x66\x79", [0x13d6] = "\x02\x61\x5b",
  [0x13d8] = "\x02\x61\x53", [0x13d9] = "\x0e\x54\x61",
  [0x13da] = "\x0e\x54\x5f", [0x13db] = "\x0e\x54\x5c",
  [0x13dc] = "\x0e\x54\x5e", [0x13dd] = "\x0e\x54\x5d",
  [0x13de] = "\x02\x5c\x72", [0x13df] = "\x02\x61\x54",
  [0x13e0] = "\x02\x5c\x6e", [0x13e2] = "\x0e\x54\x5b",
  [0x13e3] = "\x01\x76\x42", [0x13e4] = "\x02\x5c\x70",
  [0x13e5] = "\x02\x5c\x6f", [0x13e6] = "\x01\x73\x5f",
  [0x13e7] = "\x02\x5c\x6d", [0x13e8] = "\x02\x5c\x71",
  [0x13e9] = "\x02\x61\x5c", [0x13ea] = "\x02\x61\x58",
  [0x13eb] = "\x02\x61\x5a", [0x13ed] = "\x02\x61\x55",
  [0x13ee] = "\x02\x61\x56", [0x13f0] = "\x02\x61\x59",
  [0x13f1] = "\x02\x61\x57", [0x13f2] = "\x01\x78\x34",
  [0x13f3] = "\x0e\x57\x65", [0x13f5] = "\x01\x78\x37",
  [0x13f6] = "\x01\x78\x36", [0x13f7] = "\x01\x78\x35",
  [0x13f8] = "\x01\x79\x6c", [0x13f9] = "\x02\x68\x76",
  [0x13fa] = "\x01\x79\x6d", [0x13fb] = "\x02\x65\x4c",
  [0x13fc] = "\x01\x7a\x6a", [0x13fd] = "\x02\x6b\x5a",
  [0x13fe] = "\x01\x7a\x6b", [0x13ff] = "\x01\x7b\x60",
  [0x1400] = "\x01\x7c\x36", [0x1401] = "\x02\x6f\x28",
  [0x1402] = "\x0e\x5f\x46", [0x1403] = "\x02\x6f\x29",
  [0x1404] = "\x02\x6f\x2a", [0x1405] = "\x0e\x61\x3e",
  [0x1407] = "\x02\x72\x2a", [0x1408] = "\x01\x45\x41",
  [0x1409] = "\x02\x21\x5d", [0x140a] = "\x01\x46\x53",
  [0x140c] = "\x01\x48\x27", [0x140d] = "\x01\x48\x28",
  [0x140e] = "\x01\x48\x26", [0x140f] = "\x0e\x23\x6b",
  [0x1410] = "\x01\x48\x29", [0x1411] = "\x01\x4a\x3c",
  [0x1412] = "\x01\x4a\x3b", [0x1413] = "\x0e\x26\x25",
  [0x1414] = "\x02\x26\x70", [0x1415] = "\x01\x4d\x51",
  [0x1416] = "\x01\x4d\x50", [0x1418] = "\x0e\x2c\x63",
  [0x1419] = "\x02\x2f\x31", [0x141a] = "\x01\x5a\x51",
  [0x141b] = "\x01\x5a\x52", [0x141e] = "\x0e\x3b\x64",
  [0x141f] = "\x01\x5f\x3c", [0x1420] = "\x02\x42\x78",
  [0x1421] = "\x01\x64\x28", [0x1422] = "\x01\x64\x29",
  [0x1423] = "\x02\x42\x7a", [0x1424] = "\x02\x42\x7c",
  [0x1425] = "\x02\x42\x7b", [0x1427] = "\x02\x4a\x3e",
  [0x1428] = "\x0e\x46\x76", [0x1429] = "\x02\x4a\x3d",
  [0x142a] = "\x01\x68\x70", [0x142b] = "\x02\x4a\x3f",
  [0x142c] = "\x0e\x46\x75", [0x142d] = "\x02\x50\x53",
  [0x142e] = "\x01\x6c\x76", [0x142f] = "\x0e\x4c\x2c",
  [0x1430] = "\x01\x70\x5d", [0x1431] = "\x0e\x50\x6d",
  [0x1432] = "\x01\x73\x61", [0x1433] = "\x01\x76\x43",
  [0x1434] = "\x01\x73\x62", [0x1435] = "\x0e\x5f\x48",
  [0x1436] = "\x01\x45\x42", [0x1437] = "\x0e\x21\x65",
  [0x1438] = "\x0e\x21\x66", [0x1439] = "\x0e\x22\x4e",
  [0x143a] = "\x02\x24\x28", [0x143b] = "\x0e\x26\x28",
  [0x143c] = "\x0e\x26\x29", [0x143d] = "\x02\x26\x71",
  [0x143e] = "\x01\x4d\x53", [0x143f] = "\x01\x4d\x52",
  [0x1440] = "\x01\x4d\x54", [0x1441] = "\x01\x51\x37",
  [0x1442] = "\x02\x2a\x53", [0x1443] = "\x02\x2a\x54",
  [0x1445] = "\x0e\x31\x29", [0x1446] = "\x02\x2f\x32",
  [0x1447] = "\x01\x55\x52", [0x1448] = "\x01\x5a\x53",
  [0x1449] = "\x01\x5f\x3d", [0x144a] = "\x02\x3b\x74",
  [0x144b] = "\x01\x45\x43", [0x144d] = "\x01\x44\x5f",
  [0x144e] = "\x01\x45\x44", [0x144f] = "\x0e\x22\x50",
  [0x1450] = "\x02\x21\x5e", [0x1451] = "\x01\x46\x57",
  [0x1452] = "\x01\x46\x56", [0x1453] = "\x01\x46\x54",
  [0x1454] = "\x01\x46\x55", [0x1457] = "\x0e\x23\x6f",
  [0x1458] = "\x01\x48\x2c", [0x1459] = "\x02\x22\x49",
  [0x145a] = "\x02\x22\x4b", [0x145b] = "\x01\x48\x2b",
  [0x145c] = "\x02\x22\x43", [0x145d] = "\x0e\x23\x6c",
  [0x145e] = "\x02\x22\x44", [0x145f] = "\x0e\x23\x6d",
  [0x1460] = "\x02\x22\x4a", [0x1461] = "\x02\x22\x46",
  [0x1462] = "\x02\x22\x48", [0x1463] = "\x01\x48\x2a",
  [0x1464] = "\x02\x22\x45", [0x1465] = "\x02\x22\x4c",
  [0x1466] = "\x02\x22\x47", [0x1467] = "\x0e\x23\x6e",
  [0x146d] = "\x01\x4a\x43", [0x146e] = "\x01\x4a\x4c",
  [0x146f] = "\x01\x4a\x4a", [0x1470] = "\x02\x24\x32",
  [0x1471] = "\x02\x24\x2f", [0x1472] = "\x02\x24\x37",
  [0x1473] = "\x01\x4a\x48", [0x1474] = "\x02\x24\x38",
  [0x1475] = "\x0e\x26\x39", [0x1476] = "\x01\x4a\x41",
  [0x1477] = "\x02\x24\x35", [0x1478] = "\x0e\x26\x2e",
  [0x1479] = "\x01\x4a\x47", [0x147a] = "\x02\x24\x31",
  [0x147b] = "\x02\x24\x30", [0x147c] = "\x01\x4a\x45",
  [0x147d] = "\x02\x24\x36", [0x147e] = "\x01\x4a\x46",
  [0x147f] = "\x01\x4d\x55", [0x1480] = "\x01\x4a\x40",
  [0x1481] = "\x02\x24\x33", [0x1482] = "\x0e\x26\x2a",
  [0x1483] = "\x02\x24\x29", [0x1484] = "\x01\x4a\x3d",
  [0x1485] = "\x0e\x26\x37", [0x1486] = "\x01\x4a\x50",
  [0x1487] = "\x02\x24\x2e", [0x1488] = "\x02\x24\x34",
  [0x1489] = "\x01\x4a\x42", [0x148a] = "\x01\x4a\x44",
  [0x148b] = "\x0e\x26\x2f", [0x148c] = "\x02\x24\x2a",
  [0x148d] = "\x0e\x26\x36", [0x148e] = "\x02\x24\x2b",
  [0x148f] = "\x02\x24\x2c", [0x1491] = "\x01\x4a\x4f",
  [0x1492] = "\x01\x4a\x49", [0x1493] = "\x01\x4a\x4e",
  [0x1494] = "\x02\x24\x2d", [0x1495] = "\x01\x4a\x4d",
  [0x1496] = "\x01\x4a\x3f", [0x1497] = "\x01\x4a\x3e",
  [0x1498] = "\x01\x4a\x4b", [0x1499] = "\x0e\x26\x35",
  [0x149b] = "\x0e\x26\x33", [0x149c] = "\x0e\x26\x34",
  [0x149d] = "\x0e\x26\x38", [0x149e] = "\x0e\x26\x31",
  [0x14a3] = "\x0e\x64\x7b", [0x14a5] = "\x0e\x26\x32",
  [0x14a6] = "\x0e\x29\x31", [0x14a7] = "\x0e\x29\x2a",
  [0x14a8] = "\x01\x4d\x63", [0x14a9] = "\x02\x26\x7d",
  [0x14aa] = "\x02\x26\x76", [0x14ab] = "\x01\x4d\x5e",
  [0x14ac] = "\x01\x4d\x71", [0x14ad] = "\x02\x26\x72",
  [0x14ae] = "\x02\x26\x79", [0x14af] = "\x02\x26\x7b",
  [0x14b0] = "\x02\x26\x7e", [0x14b1] = "\x01\x4d\x6c",
  [0x14b2] = "\x0e\x29\x26", [0x14b3] = "\x02\x26\x7a",
  [0x14b4] = "\x02\x26\x73", [0x14b5] = "\x01\x4d\x6a",
  [0x14b6] = "\x02\x26\x77", [0x14b7] = "\x0e\x29\x28",
  [0x14b8] = "\x02\x27\x21", [0x14b9] = "\x01\x4d\x5b",
  [0x14ba] = "\x0e\x29\x25", [0x14bb] = "\x02\x26\x7c",
  [0x14bc] = "\x01\x4d\x65", [0x14bd] = "\x01\x4d\x64",
  [0x14be] = "\x02\x26\x75", [0x14bf] = "\x01\x4d\x59",
  [0x14c0] = "\x0e\x29\x34", [0x14c1] = "\x0e\x29\x32",
  [0x14c2] = "\x01\x4d\x5a", [0x14c4] = "\x01\x4d\x58",
  [0x14c5] = "\x0e\x29\x33", [0x14c6] = "\x01\x4d\x70",
  [0x14c7] = "\x01\x4d\x68", [0x14c8] = "\x01\x4d\x62",
  [0x14c9] = "\x01\x4d\x56", [0x14ca] = "\x02\x26\x78",
  [0x14cb] = "\x01\x4d\x61", [0x14cc] = "\x01\x4d\x57",
  [0x14cd] = "\x01\x4d\x69", [0x14ce] = "\x01\x4d\x72",
  [0x14cf] = "\x02\x2a\x55", [0x14d0] = "\x01\x4d\x66",
  [0x14d1] = "\x02\x26\x74", [0x14d2] = "\x01\x4d\x5c",
  [0x14d3] = "\x01\x4d\x5f", [0x14d4] = "\x01\x4d\x60",
  [0x14d5] = "\x0e\x29\x2e", [0x14d6] = "\x01\x4d\x6e",
  [0x14d7] = "\x01\x4d\x6f", [0x14d8] = "\x01\x4d\x6d",
  [0x14d9] = "\x01\x4d\x67", [0x14da] = "\x01\x4d\x6b",
  [0x14db] = "\x01\x4d\x5d", [0x14dc] = "\x01\x51\x38",
  [0x14dd] = "\x0e\x29\x30", [0x14de] = "\x0e\x29\x37",
  [0x14df] = "\x0e\x29\x35", [0x14e0] = "\x0e\x29\x36",
  [0x14e1] = "\x0e\x29\x2f", [0x14e4] = "\x0e\x29\x29",
  [0x14ea] = "\x0e\x2c\x6d", [0x14eb] = "\x02\x2a\x5b",
  [0x14ec] = "\x01\x51\x44", [0x14ed] = "\x01\x51\x3c",
  [0x14ee] = "\x01\x51\x3e", [0x14ef] = "\x01\x51\x43",
  [0x14f0] = "\x02\x2a\x67", [0x14f1] = "\x01\x51\x41",
  [0x14f2] = "\x02\x2f\x33", [0x14f3] = "\x01\x55\x53",
  [0x14f4] = "\x01\x51\x46", [0x14f5] = "\x02\x2a\x58",
  [0x14f6] = "\x02\x2a\x60", [0x14f7] = "\x01\x51\x42",
  [0x14f8] = "\x02\x2a\x5f", [0x14f9] = "\x02\x2a\x5c",
  [0x14fa] = "\x02\x2a\x64", [0x14fb] = "\x02\x2a\x66",
  [0x14fc] = "\x01\x51\x3b", [0x14fd] = "\x01\x51\x3f",
  [0x14fe] = "\x01\x51\x45", [0x14ff] = "\x01\x55\x55",
  [0x1500] = "\x02\x2a\x61", [0x1501] = "\x01\x51\x3d",
  [0x1502] = "\x01\x51\x48", [0x1503] = "\x02\x2a\x5a",
  [0x1504] = "\x0e\x2c\x6f", [0x1505] = "\x0e\x2c\x70",
  [0x1507] = "\x01\x51\x40", [0x1508] = "\x01\x55\x54",
  [0x1509] = "\x01\x51\x3a", [0x150b] = "\x02\x2a\x57",
  [0x150c] = "\x02\x2a\x5e", [0x150d] = "\x02\x2a\x56",
  [0x150e] = "\x02\x2a\x59", [0x150f] = "\x02\x2a\x5d",
  [0x1510] = "\x02\x2f\x34", [0x1511] = "\x01\x51\x47",
  [0x1513] = "\x02\x2a\x62", [0x1514] = "\x02\x2a\x63",
  [0x1515] = "\x02\x2a\x65", [0x1516] = "\x01\x51\x39",
  [0x1517] = "\x0e\x2c\x6c", [0x1519] = "\x0e\x31\x32",
  [0x151b] = "\x0e\x31\x33", [0x151f] = "\x0e\x2c\x6b",
  [0x1521] = "\x0e\x2c\x6e", [0x1528] = "\x01\x55\x63",
  [0x1529] = "\x02\x2f\x40", [0x152a] = "\x01\x55\x61",
  [0x152b] = "\x01\x55\x62", [0x152c] = "\x02\x2f\x36",
  [0x152d] = "\x02\x2f\x46", [0x152e] = "\x0e\x31\x2c",
  [0x152f] = "\x01\x55\x58", [0x1530] = "\x0e\x31\x2f",
  [0x1531] = "\x0e\x31\x35", [0x1532] = "\x02\x35\x4a",
  [0x1533] = "\x02\x2f\x48", [0x1534] = "\x02\x2f\x42",
  [0x1535] = "\x0e\x31\x38", [0x1536] = "\x02\x2f\x39",
  [0x1537] = "\x0e\x31\x37", [0x1538] = "\x02\x2f\x4b",
  [0x1539] = "\x02\x2f\x3c", [0x153a] = "\x01\x55\x5e",
  [0x153b] = "\x02\x35\x61", [0x153c] = "\x02\x2f\x3f",
  [0x153d] = "\x01\x55\x60", [0x153e] = "\x01\x55\x57",
  [0x153f] = "\x0e\x31\x3c", [0x1540] = "\x02\x2f\x4d",
  [0x1541] = "\x02\x2f\x41", [0x1542] = "\x01\x55\x5a",
  [0x1543] = "\x02\x2f\x3a", [0x1544] = "\x02\x2f\x37",
  [0x1545] = "\x02\x2f\x38", [0x1546] = "\x01\x55\x5b",
  [0x1547] = "\x02\x2f\x47", [0x1548] = "\x02\x2f\x4e",
  [0x1549] = "\x01\x55\x5d", [0x154a] = "\x02\x2f\x3e",
  [0x154b] = "\x02\x2f\x3d", [0x154c] = "\x01\x55\x65",
  [0x154d] = "\x01\x55\x64", [0x154e] = "\x01\x55\x56",
  [0x154f] = "\x01\x55\x5c", [0x1550] = "\x01\x55\x5f",
  [0x1551] = "\x02\x2f\x4a", [0x1552] = "\x0e\x31\x2e",
  [0x1553] = "\x0e\x31\x39", [0x1554] = "\x02\x2f\x44",
  [0x1555] = "\x01\x55\x59", [0x1556] = "\x02\x2f\x35",
  [0x1557] = "\x02\x2f\x4c", [0x1558] = "\x02\x2f\x43",
  [0x1559] = "\x02\x2f\x45", [0x155a] = "\x02\x2f\x49",
  [0x155b] = "\x0e\x31\x30", [0x155c] = "\x0e\x31\x3b",
  [0x155d] = "\x0e\x31\x36", [0x1560] = "\x0e\x64\x7c",
  [0x1564] = "\x0e\x64\x7d", [0x1565] = "\x02\x35\x4b",
  [0x1566] = "\x0e\x36\x31", [0x1567] = "\x01\x5a\x5b",
  [0x1568] = "\x01\x5a\x6f", [0x1569] = "\x01\x5a\x6e",
  [0x156a] = "\x0e\x36\x27", [0x156b] = "\x01\x5a\x63",
  [0x156c] = "\x0e\x36\x29", [0x156d] = "\x02\x35\x5d",
  [0x156e] = "\x02\x35\x59", [0x156f] = "\x02\x35\x56",
  [0x1570] = "\x02\x35\x68", [0x1571] = "\x01\x5a\x5e",
  [0x1572] = "\x01\x5a\x56", [0x1573] = "\x0e\x36\x26",
  [0x1574] = "\x0e\x36\x32", [0x1575] = "\x02\x35\x5b",
  [0x1576] = "\x01\x5f\x4d", [0x1577] = "\x01\x5a\x5a",
  [0x1578] = "\x02\x35\x63", [0x1579] = "\x0e\x36\x22",
  [0x157a] = "\x01\x5a\x70", [0x157b] = "\x01\x5a\x6d",
  [0x157c] = "\x02\x35\x5f", [0x157d] = "\x02\x35\x4e",
  [0x157e] = "\x0e\x35\x7c", [0x157f] = "\x0e\x36\x2b",
  [0x1580] = "\x01\x5a\x6c", [0x1581] = "\x02\x35\x65",
  [0x1582] = "\x02\x35\x4d", [0x1583] = "\x01\x5a\x61",
  [0x1584] = "\x01\x5a\x65", [0x1585] = "\x02\x35\x64",
  [0x1586] = "\x0e\x36\x23", [0x1587] = "\x02\x35\x57",
  [0x1588] = "\x01\x5a\x66", [0x1589] = "\x01\x5a\x60",
  [0x158a] = "\x02\x35\x4c", [0x158b] = "\x0e\x35\x7d",
  [0x158c] = "\x01\x5f\x3f", [0x158d] = "\x02\x35\x67",
  [0x158e] = "\x02\x35\x55", [0x158f] = "\x01\x5a\x6b",
  [0x1590] = "\x02\x35\x58", [0x1591] = "\x02\x35\x66",
  [0x1592] = "\x01\x5a\x6a", [0x1593] = "\x0e\x36\x24",
  [0x1594] = "\x02\x3b\x75", [0x1596] = "\x01\x5a\x57",
  [0x1597] = "\x02\x35\x53", [0x1598] = "\x01\x5a\x5c",
  [0x1599] = "\x01\x5a\x67", [0x159b] = "\x01\x5a\x62",
  [0x159c] = "\x02\x35\x5c", [0x159d] = "\x02\x35\x52",
  [0x159e] = "\x02\x35\x50", [0x159f] = "\x02\x35\x62",
  [0x15a0] = "\x01\x5a\x54", [0x15a1] = "\x01\x5a\x68",
  [0x15a2] = "\x01\x5a\x58", [0x15a3] = "\x01\x5f\x3e",
  [0x15a4] = "\x02\x35\x60", [0x15a5] = "\x01\x5a\x59",
  [0x15a7] = "\x01\x5a\x55", [0x15a8] = "\x01\x5a\x64",
  [0x15a9] = "\x01\x5a\x5f", [0x15aa] = "\x01\x5a\x5d",
  [0x15ab] = "\x02\x35\x54", [0x15ac] = "\x01\x5a\x69",
  [0x15ad] = "\x02\x35\x51", [0x15ae] = "\x02\x35\x5e",
  [0x15af] = "\x02\x35\x5a", [0x15b0] = "\x02\x3b\x77",
  [0x15b1] = "\x02\x3b\x76", [0x15b2] = "\x0e\x36\x2d",
  [0x15b6] = "\x0e\x64\x7e", [0x15ba] = "\x0e\x36\x2e",
  [0x15bd] = "\x02\x35\x4f", [0x15be] = "\x02\x3c\x29",
  [0x15bf] = "\x0e\x3b\x71", [0x15c0] = "\x01\x5f\x41",
  [0x15c1] = "\x0e\x3b\x70", [0x15c2] = "\x02\x3c\x2f",
  [0x15c3] = "\x02\x3b\x7c", [0x15c4] = "\x02\x3c\x2c",
  [0x15c5] = "\x02\x42\x7d", [0x15c6] = "\x01\x5f\x44",
  [0x15c7] = "\x02\x3c\x30", [0x15c8] = "\x02\x3c\x33",
  [0x15c9] = "\x01\x5f\x43", [0x15ca] = "\x02\x3c\x21",
  [0x15cb] = "\x02\x3c\x32", [0x15cc] = "\x02\x3c\x31",
  [0x15cd] = "\x01\x5f\x45", [0x15ce] = "\x02\x3b\x78",
  [0x15cf] = "\x01\x5f\x40", [0x15d0] = "\x01\x5f\x48",
  [0x15d1] = "\x0e\x3b\x73", [0x15d2] = "\x01\x5f\x46",
  [0x15d3] = "\x02\x3c\x2e", [0x15d5] = "\x02\x3c\x24",
  [0x15d6] = "\x01\x5f\x4a", [0x15d7] = "\x02\x3c\x35",
  [0x15d8] = "\x02\x3c\x2d", [0x15d9] = "\x02\x3c\x36",
  [0x15da] = "\x01\x5f\x52", [0x15db] = "\x01\x5f\x50",
  [0x15dc] = "\x02\x3c\x2b", [0x15dd] = "\x02\x3c\x2a",
  [0x15de] = "\x0e\x3b\x67", [0x15df] = "\x02\x3c\x28",
  [0x15e0] = "\x02\x3c\x22", [0x15e1] = "\x01\x5f\x49",
  [0x15e2] = "\x0e\x3b\x66", [0x15e3] = "\x01\x5f\x47",
  [0x15e4] = "\x02\x2f\x3b", [0x15e5] = "\x02\x3b\x79",
  [0x15e6] = "\x0e\x3b\x68", [0x15e7] = "\x02\x43\x3d",
  [0x15e8] = "\x02\x3b\x7a", [0x15e9] = "\x01\x5f\x42",
  [0x15ea] = "\x01\x5f\x4f", [0x15eb] = "\x02\x43\x21",
  [0x15ed] = "\x01\x5f\x4b", [0x15ee] = "\x01\x5f\x4c",
  [0x15ef] = "\x02\x3b\x7b", [0x15f0] = "\x02\x3c\x34",
  [0x15f1] = "\x02\x42\x7e", [0x15f2] = "\x02\x3c\x25",
  [0x15f3] = "\x02\x3b\x7e", [0x15f4] = "\x01\x5f\x4e",
  [0x15f5] = "\x02\x3c\x26", [0x15f6] = "\x02\x3c\x23",
  [0x15f7] = "\x0e\x3b\x72", [0x15f8] = "\x0e\x3b\x6d",
  [0x15f9] = "\x01\x5f\x53", [0x15fa] = "\x0e\x63\x39",
  [0x15fe] = "\x0e\x3b\x6f", [0x1603] = "\x0e\x65\x21",
  [0x1606] = "\x01\x64\x38", [0x1607] = "\x0e\x41\x49",
  [0x1608] = "\x0e\x41\x32", [0x1609] = "\x02\x43\x24",
  [0x160a] = "\x02\x43\x37", [0x160b] = "\x02\x43\x3c",
  [0x160c] = "\x02\x43\x30", [0x160d] = "\x01\x64\x34",
  [0x160e] = "\x02\x43\x41", [0x160f] = "\x01\x64\x31",
  [0x1610] = "\x02\x43\x22", [0x1611] = "\x0e\x41\x3a",
  [0x1612] = "\x02\x43\x23", [0x1613] = "\x01\x64\x2a",
  [0x1614] = "\x01\x64\x33", [0x1615] = "\x02\x43\x2a",
  [0x1616] = "\x01\x64\x36", [0x1617] = "\x01\x64\x37",
  [0x1618] = "\x02\x43\x2b", [0x1619] = "\x0e\x41\x38",
  [0x161a] = "\x02\x43\x38", [0x161b] = "\x02\x43\x3e",
  [0x161c] = "\x01\x64\x32", [0x161d] = "\x0e\x41\x3e",
  [0x161e] = "\x01\x64\x2c", [0x161f] = "\x02\x43\x29",
  [0x1620] = "\x02\x43\x25", [0x1621] = "\x02\x43\x40",
  [0x1622] = "\x02\x43\x2e", [0x1623] = "\x02\x43\x2f",
  [0x1624] = "\x02\x43\x26", [0x1625] = "\x02\x43\x3a",
  [0x1626] = "\x02\x43\x31", [0x1627] = "\x02\x43\x3b",
  [0x1628] = "\x02\x43\x33", [0x1629] = "\x0e\x41\x3d",
  [0x162a] = "\x01\x64\x2d", [0x162b] = "\x02\x4a\x40",
  [0x162c] = "\x01\x64\x30", [0x162d] = "\x01\x64\x2e",
  [0x162e] = "\x02\x43\x3f", [0x162f] = "\x02\x43\x36",
  [0x1630] = "\x02\x43\x32", [0x1631] = "\x0e\x41\x36",
  [0x1632] = "\x0e\x41\x33", [0x1633] = "\x02\x43\x27",
  [0x1634] = "\x01\x68\x7a", [0x1635] = "\x02\x43\x35",
  [0x1636] = "\x01\x64\x35", [0x1637] = "\x02\x43\x2d",
  [0x1638] = "\x0e\x41\x34", [0x1639] = "\x02\x43\x2c",
  [0x163a] = "\x0e\x41\x48", [0x163b] = "\x0e\x47\x25",
  [0x163c] = "\x0e\x41\x42", [0x163d] = "\x01\x64\x2f",
  [0x163e] = "\x01\x64\x2b", [0x163f] = "\x02\x4a\x55",
  [0x1640] = "\x02\x43\x39", [0x1641] = "\x02\x43\x34",
  [0x1643] = "\x02\x43\x28", [0x1646] = "\x0e\x41\x44",
  [0x1647] = "\x0e\x41\x45", [0x1649] = "\x0e\x66\x76",
  [0x164b] = "\x02\x4a\x50", [0x164c] = "\x0e\x46\x78",
  [0x164d] = "\x02\x4a\x41", [0x164e] = "\x02\x4a\x4c",
  [0x164f] = "\x0e\x47\x28", [0x1650] = "\x02\x4a\x53",
  [0x1651] = "\x01\x68\x78", [0x1652] = "\x01\x5f\x51",
  [0x1653] = "\x02\x4a\x51", [0x1654] = "\x01\x68\x73",
  [0x1655] = "\x0e\x46\x7e", [0x1656] = "\x0e\x47\x24",
  [0x1657] = "\x0e\x46\x7a", [0x1658] = "\x01\x68\x72",
  [0x1659] = "\x02\x4a\x58", [0x165b] = "\x02\x4a\x42",
  [0x165c] = "\x02\x4a\x4f", [0x165d] = "\x02\x4a\x43",
  [0x165e] = "\x02\x4a\x4e", [0x165f] = "\x01\x68\x76",
  [0x1660] = "\x02\x4a\x52", [0x1661] = "\x02\x3c\x27",
  [0x1662] = "\x0e\x47\x21", [0x1664] = "\x0e\x47\x2a",
  [0x1665] = "\x02\x4a\x59", [0x1666] = "\x02\x4a\x4a",
  [0x1667] = "\x01\x68\x79", [0x1668] = "\x02\x50\x61",
  [0x1669] = "\x01\x6c\x77", [0x166a] = "\x0e\x47\x23",
  [0x166b] = "\x02\x4a\x57", [0x166c] = "\x02\x4a\x56",
  [0x166d] = "\x01\x68\x7b", [0x166e] = "\x02\x50\x54",
  [0x166f] = "\x01\x6c\x78", [0x1670] = "\x02\x50\x55",
  [0x1671] = "\x0e\x47\x22", [0x1672] = "\x02\x4a\x46",
  [0x1673] = "\x02\x4a\x47", [0x1674] = "\x02\x4a\x44",
  [0x1675] = "\x02\x4a\x49", [0x1676] = "\x02\x4a\x45",
  [0x1677] = "\x02\x4a\x5a", [0x1678] = "\x01\x68\x75",
  [0x1679] = "\x01\x6c\x79", [0x167a] = "\x01\x68\x77",
  [0x167b] = "\x01\x68\x7c", [0x167c] = "\x0e\x46\x7b",
  [0x167d] = "\x02\x4a\x48", [0x167e] = "\x0e\x47\x29",
  [0x167f] = "\x02\x4a\x54", [0x1680] = "\x0e\x4c\x2d",
  [0x1681] = "\x0e\x47\x26", [0x1682] = "\x02\x4a\x4d",
  [0x1683] = "\x0e\x4c\x35", [0x1685] = "\x02\x50\x58",
  [0x1686] = "\x0e\x4c\x38", [0x1687] = "\x01\x68\x71",
  [0x1688] = "\x01\x6c\x7c", [0x1689] = "\x02\x57\x35",
  [0x168a] = "\x02\x50\x5d", [0x168b] = "\x02\x50\x5c",
  [0x168c] = "\x02\x50\x5e", [0x168d] = "\x0e\x4c\x30",
  [0x168e] = "\x0e\x4c\x2f", [0x168f] = "\x02\x50\x5b",
  [0x1690] = "\x01\x6c\x7d", [0x1691] = "\x0e\x4c\x3b",
  [0x1692] = "\x01\x6d\x25", [0x1693] = "\x01\x6d\x22",
  [0x1694] = "\x0e\x4c\x31", [0x1695] = "\x01\x6d\x23",
  [0x1696] = "\x02\x50\x56", [0x1697] = "\x02\x50\x59",
  [0x1698] = "\x02\x50\x63", [0x1699] = "\x01\x6d\x2b",
  [0x169a] = "\x01\x6d\x29", [0x169b] = "\x0e\x4c\x2e",
  [0x169c] = "\x02\x50\x5a", [0x169d] = "\x02\x3b\x7d",
  [0x169e] = "\x01\x6c\x7a", [0x169f] = "\x02\x50\x60",
  [0x16a0] = "\x02\x50\x57", [0x16a1] = "\x0e\x4c\x3e",
  [0x16a2] = "\x01\x6d\x2c", [0x16a3] = "\x02\x50\x5f",
  [0x16a4] = "\x01\x68\x74", [0x16a5] = "\x01\x6d\x21",
  [0x16a6] = "\x02\x4a\x4b", [0x16a7] = "\x0e\x4c\x3f",
  [0x16a8] = "\x0e\x4c\x34", [0x16a9] = "\x01\x6d\x24",
  [0x16aa] = "\x0e\x4c\x3d", [0x16ab] = "\x01\x6d\x28",
  [0x16ac] = "\x01\x6d\x2a", [0x16ad] = "\x01\x6d\x27",
  [0x16ae] = "\x01\x6d\x26", [0x16af] = "\x0e\x4c\x3a",
  [0x16b0] = "\x01\x6c\x7e", [0x16b1] = "\x02\x50\x62",
  [0x16b2] = "\x01\x6c\x7b", [0x16b3] = "\x01\x6d\x2d",
  [0x16b4] = "\x0e\x4c\x39", [0x16b6] = "\x0e\x65\x22",
  [0x16b9] = "\x0e\x4c\x37", [0x16bb] = "\x01\x70\x61",
  [0x16bc] = "\x01\x70\x62", [0x16bd] = "\x02\x57\x34",
  [0x16be] = "\x01\x70\x6b", [0x16bf] = "\x01\x70\x68",
  [0x16c0] = "\x0e\x50\x70", [0x16c1] = "\x01\x70\x5f",
  [0x16c2] = "\x01\x70\x66", [0x16c3] = "\x02\x57\x36",
  [0x16c4] = "\x01\x70\x64", [0x16c5] = "\x01\x70\x5e",
  [0x16c6] = "\x0e\x4c\x32", [0x16c7] = "\x01\x70\x65",
  [0x16c8] = "\x0e\x50\x77", [0x16c9] = "\x02\x57\x33",
  [0x16ca] = "\x01\x73\x64", [0x16cb] = "\x01\x70\x60",
  [0x16cd] = "\x01\x70\x67", [0x16ce] = "\x01\x73\x63",
  [0x16cf] = "\x02\x57\x32", [0x16d0] = "\x02\x57\x31",
  [0x16d1] = "\x0e\x50\x76", [0x16d2] = "\x01\x70\x69",
  [0x16d3] = "\x0e\x50\x6f", [0x16d4] = "\x01\x70\x6a",
  [0x16d5] = "\x0e\x50\x79", [0x16d6] = "\x02\x57\x30",
  [0x16d7] = "\x02\x57\x2f", [0x16d8] = "\x01\x73\x65",
  [0x16d9] = "\x02\x57\x39", [0x16da] = "\x01\x70\x63",
  [0x16db] = "\x02\x57\x37", [0x16dc] = "\x0e\x50\x75",
  [0x16df] = "\x0e\x54\x64", [0x16e0] = "\x01\x73\x66",
  [0x16e1] = "\x0e\x54\x67", [0x16e2] = "\x01\x73\x6b",
  [0x16e3] = "\x02\x5c\x75", [0x16e4] = "\x02\x5c\x77",
  [0x16e5] = "\x0e\x57\x6b", [0x16e6] = "\x01\x73\x68",
  [0x16e7] = "\x0e\x57\x6d", [0x16e8] = "\x02\x5c\x78",
  [0x16e9] = "\x02\x5c\x74", [0x16ea] = "\x0e\x57\x6c",
  [0x16eb] = "\x02\x5c\x76", [0x16ec] = "\x01\x73\x69",
  [0x16ed] = "\x01\x73\x6c", [0x16ee] = "\x0e\x54\x69",
  [0x16ef] = "\x02\x5c\x73", [0x16f0] = "\x01\x73\x67",
  [0x16f1] = "\x01\x73\x6a", [0x16f2] = "\x01\x76\x45",
  [0x16f3] = "\x02\x57\x38", [0x16f4] = "\x01\x76\x44",
  [0x16f6] = "\x0e\x57\x6a", [0x16f7] = "\x01\x76\x4a",
  [0x16f8] = "\x02\x61\x60", [0x16f9] = "\x0e\x57\x70",
  [0x16fa] = "\x01\x76\x48", [0x16fb] = "\x01\x76\x49",
  [0x16fc] = "\x02\x61\x63", [0x16fd] = "\x02\x61\x5f",
  [0x16fe] = "\x01\x76\x46", [0x16ff] = "\x02\x61\x5d",
  [0x1700] = "\x01\x78\x38", [0x1701] = "\x02\x61\x61",
  [0x1703] = "\x02\x61\x62", [0x1704] = "\x02\x61\x5e",
  [0x1705] = "\x0e\x57\x6e", [0x1706] = "\x01\x76\x47",
  [0x1707] = "\x02\x65\x4d", [0x1708] = "\x0e\x5a\x3b",
  [0x1709] = "\x02\x65\x50", [0x170a] = "\x0e\x5a\x3c",
  [0x170b] = "\x0e\x5a\x3a", [0x170c] = "\x02\x65\x51",
  [0x170d] = "\x02\x65\x4f", [0x170e] = "\x02\x65\x52",
  [0x170f] = "\x01\x78\x39", [0x1710] = "\x02\x65\x4e",
  [0x1711] = "\x0e\x5e\x31", [0x1713] = "\x02\x68\x7a",
  [0x1714] = "\x01\x79\x6f", [0x1715] = "\x02\x68\x79",
  [0x1716] = "\x02\x68\x78", [0x1717] = "\x02\x68\x77",
  [0x1718] = "\x01\x79\x6e", [0x1719] = "\x01\x79\x70",
  [0x171a] = "\x0e\x65\x23", [0x171b] = "\x02\x6b\x5b",
  [0x171c] = "\x01\x7a\x6d", [0x171d] = "\x01\x7a\x6c",
  [0x171e] = "\x0e\x5f\x4a", [0x171f] = "\x0e\x5f\x4b",
  [0x1720] = "\x02\x6d\x4f", [0x1721] = "\x02\x6d\x4e",
  [0x1722] = "\x02\x6d\x51", [0x1723] = "\x01\x7c\x37",
  [0x1724] = "\x01\x7b\x61", [0x1725] = "\x02\x6f\x2c",
  [0x1726] = "\x02\x6d\x50", [0x1727] = "\x0e\x5f\x49",
  [0x1728] = "\x0e\x60\x32", [0x1729] = "\x02\x6f\x2b",
  [0x172a] = "\x01\x7c\x39", [0x172b] = "\x01\x7c\x38",
  [0x172c] = "\x01\x7c\x5f", [0x172d] = "\x02\x70\x37",
  [0x172e] = "\x02\x70\x7d", [0x172f] = "\x01\x45\x45",
  [0x1731] = "\x0e\x2c\x72", [0x1732] = "\x02\x3c\x37",
  [0x1733] = "\x02\x57\x3a", [0x1734] = "\x0e\x21\x67",
  [0x1735] = "\x0e\x21\x68", [0x1736] = "\x01\x48\x2d",
  [0x1737] = "\x02\x22\x4d", [0x1738] = "\x01\x4a\x53",
  [0x1739] = "\x01\x4a\x51", [0x173b] = "\x01\x4a\x52",
  [0x173d] = "\x02\x27\x22", [0x173e] = "\x01\x4d\x73",
  [0x173f] = "\x01\x51\x49", [0x1740] = "\x0e\x2c\x74",
  [0x1741] = "\x02\x2a\x68", [0x1742] = "\x0e\x2c\x76",
  [0x1743] = "\x02\x2a\x69", [0x1744] = "\x0e\x2c\x73",
  [0x1745] = "\x01\x51\x4a", [0x1746] = "\x02\x2f\x50",
  [0x1748] = "\x01\x55\x66", [0x1749] = "\x01\x55\x67",
  [0x174a] = "\x02\x2f\x4f", [0x174b] = "\x0e\x31\x3d",
  [0x174d] = "\x0e\x36\x37", [0x174e] = "\x0e\x36\x36",
  [0x174f] = "\x01\x5a\x77", [0x1751] = "\x01\x5a\x73",
  [0x1753] = "\x02\x35\x69", [0x1754] = "\x01\x5a\x7a",
  [0x1755] = "\x01\x5a\x79", [0x1756] = "\x01\x5a\x72",
  [0x1757] = "\x01\x5a\x75", [0x1758] = "\x01\x5a\x78",
  [0x1759] = "\x01\x5a\x74", [0x175a] = "\x0e\x36\x34",
  [0x175c] = "\x02\x3c\x3b", [0x175d] = "\x01\x5a\x71",
  [0x175e] = "\x01\x5f\x54", [0x175f] = "\x0e\x3b\x74",
  [0x1760] = "\x0e\x3b\x75", [0x1761] = "\x0e\x3b\x76",
  [0x1762] = "\x01\x5f\x56", [0x1763] = "\x01\x5f\x57",
  [0x1764] = "\x02\x3c\x3a", [0x1765] = "\x02\x3c\x3d",
  [0x1766] = "\x01\x5f\x55", [0x1767] = "\x02\x3c\x38",
  [0x1768] = "\x02\x3c\x3c", [0x176a] = "\x02\x3c\x39",
  [0x176b] = "\x0e\x41\x4b", [0x176c] = "\x01\x64\x39",
  [0x176d] = "\x0e\x41\x4e", [0x176e] = "\x0e\x63\x46",
  [0x176f] = "\x02\x43\x42", [0x1770] = "\x0e\x41\x4d",
  [0x1771] = "\x0e\x47\x30", [0x1772] = "\x01\x68\x7d",
  [0x1773] = "\x02\x4a\x5b", [0x1774] = "\x01\x70\x6c",
  [0x1775] = "\x01\x6d\x2e", [0x1776] = "\x02\x50\x64",
  [0x1777] = "\x01\x6d\x2f", [0x1778] = "\x01\x6d\x30",
  [0x1779] = "\x02\x50\x66", [0x177a] = "\x02\x50\x65",
  [0x177b] = "\x02\x50\x67", [0x177c] = "\x02\x57\x3c",
  [0x177f] = "\x02\x57\x3b", [0x1780] = "\x02\x5c\x7a",
  [0x1781] = "\x02\x5c\x79", [0x1782] = "\x01\x73\x6d",
  [0x1783] = "\x01\x73\x6e", [0x1784] = "\x02\x65\x53",
  [0x1785] = "\x0e\x5c\x41", [0x1787] = "\x01\x45\x46",
  [0x1788] = "\x0e\x25\x58", [0x1789] = "\x0e\x29\x39",
  [0x178b] = "\x0e\x31\x40", [0x178c] = "\x02\x3c\x3e",
  [0x178d] = "\x0e\x36\x38", [0x178e] = "\x0e\x36\x39",
  [0x1790] = "\x01\x5f\x59", [0x1791] = "\x01\x5f\x58",
  [0x1792] = "\x02\x43\x43", [0x1794] = "\x02\x61\x64",
  [0x1795] = "\x01\x7a\x6e", [0x1796] = "\x02\x6f\x2d",
  [0x1797] = "\x01\x45\x47", [0x1799] = "\x01\x55\x68",
  [0x179a] = "\x0e\x31\x42", [0x179b] = "\x01\x5a\x7c",
  [0x179c] = "\x01\x5a\x7b", [0x179d] = "\x02\x3c\x3f",
  [0x179e] = "\x02\x3c\x40", [0x179f] = "\x01\x64\x3a",
  [0x17a0] = "\x02\x4a\x5c", [0x17a1] = "\x01\x68\x7e",
  [0x17a2] = "\x02\x57\x3d", [0x17a4] = "\x01\x45\x48",
  [0x17a5] = "\x01\x46\x58", [0x17a6] = "\x0e\x29\x3b",
  [0x17a7] = "\x01\x4d\x74", [0x17a8] = "\x02\x27\x23",
  [0x17aa] = "\x02\x2a\x6a", [0x17ab] = "\x01\x51\x4b",
  [0x17ac] = "\x01\x5a\x7d", [0x17ad] = "\x0e\x36\x3a",
  [0x17ae] = "\x02\x3c\x41", [0x17af] = "\x01\x5f\x5a",
  [0x17b0] = "\x01\x64\x3b", [0x17b2] = "\x02\x50\x68",
  [0x17b3] = "\x02\x50\x69", [0x17b5] = "\x0e\x54\x6b",
  [0x17b6] = "\x02\x5c\x7b", [0x17b7] = "\x01\x76\x4b",
  [0x17b8] = "\x02\x70\x7e", [0x17b9] = "\x01\x45\x49",
  [0x17ba] = "\x0e\x29\x3c", [0x17bb] = "\x02\x27\x24",
  [0x17bc] = "\x01\x4d\x75", [0x17bd] = "\x01\x51\x4c",
  [0x17be] = "\x0e\x2c\x77", [0x17bf] = "\x02\x2a\x6b",
  [0x17c1] = "\x01\x55\x69", [0x17c2] = "\x02\x2f\x54",
  [0x17c3] = "\x02\x2f\x52", [0x17c4] = "\x02\x2f\x53",
  [0x17c5] = "\x01\x55\x6a", [0x17c6] = "\x02\x2f\x51",
  [0x17c7] = "\x0e\x36\x3c", [0x17c9] = "\x0e\x36\x3f",
  [0x17ca] = "\x0e\x36\x3d", [0x17cb] = "\x01\x5b\x21",
  [0x17cc] = "\x01\x5b\x22", [0x17cd] = "\x02\x35\x6a",
  [0x17ce] = "\x01\x5b\x23", [0x17cf] = "\x01\x5a\x7e",
  [0x17d0] = "\x02\x3c\x42", [0x17d1] = "\x0e\x3b\x79",
  [0x17d2] = "\x02\x3c\x43", [0x17d3] = "\x02\x43\x44",
  [0x17d6] = "\x01\x69\x22", [0x17d7] = "\x01\x69\x21",
  [0x17da] = "\x02\x5c\x7c", [0x17db] = "\x02\x61\x65",
  [0x17dc] = "\x0e\x5a\x3f", [0x17dd] = "\x02\x65\x55",
  [0x17de] = "\x02\x65\x54", [0x17df] = "\x02\x68\x7b",
  [0x17e0] = "\x0e\x21\x69", [0x17e1] = "\x02\x21\x3e",
  [0x17e2] = "\x01\x51\x4d", [0x17e3] = "\x0e\x36\x41",
  [0x17e4] = "\x0e\x41\x50", [0x17e5] = "\x01\x45\x4a",
  [0x17e6] = "\x01\x46\x59", [0x17e7] = "\x0e\x22\x51",
  [0x17e8] = "\x01\x48\x2f", [0x17e9] = "\x01\x48\x2e",
  [0x17ea] = "\x0e\x23\x73", [0x17eb] = "\x0e\x23\x72",
  [0x17ec] = "\x01\x48\x30", [0x17ed] = "\x01\x48\x31",
  [0x17ee] = "\x02\x22\x4f", [0x17ef] = "\x02\x22\x4e",
  [0x17f0] = "\x02\x24\x39", [0x17f1] = "\x01\x4a\x54",
  [0x17f2] = "\x02\x24\x3c", [0x17f3] = "\x02\x24\x3b",
  [0x17f4] = "\x02\x24\x3a", [0x17f5] = "\x02\x24\x3d",
  [0x17f6] = "\x0e\x26\x3d", [0x17f9] = "\x0e\x62\x5e",
  [0x17fa] = "\x01\x4d\x76", [0x17fb] = "\x02\x27\x2a",
  [0x17fc] = "\x02\x27\x26", [0x17fd] = "\x02\x27\x2f",
  [0x17fe] = "\x0e\x29\x43", [0x17ff] = "\x0e\x29\x3e",
  [0x1800] = "\x01\x4d\x7d", [0x1801] = "\x0e\x29\x42",
  [0x1802] = "\x01\x4d\x7b", [0x1803] = "\x02\x27\x2b",
  [0x1804] = "\x02\x27\x27", [0x1805] = "\x02\x27\x2e",
  [0x1806] = "\x01\x4d\x7a", [0x1807] = "\x01\x4e\x23",
  [0x1808] = "\x02\x27\x29", [0x1809] = "\x02\x27\x25",
  [0x180a] = "\x01\x4e\x22", [0x180b] = "\x02\x27\x2c",
  [0x180c] = "\x01\x4d\x79", [0x180d] = "\x02\x27\x2d",
  [0x180e] = "\x01\x4d\x7c", [0x180f] = "\x01\x4d\x7e",
  [0x1810] = "\x02\x27\x31", [0x1811] = "\x02\x27\x30",
  [0x1812] = "\x02\x27\x28", [0x1813] = "\x01\x4d\x78",
  [0x1814] = "\x01\x4d\x77", [0x1815] = "\x01\x4e\x21",
  [0x1817] = "\x0e\x29\x40", [0x1818] = "\x0e\x29\x41",
  [0x181a] = "\x0e\x2c\x7e", [0x181b] = "\x0e\x2c\x7a",
  [0x181c] = "\x02\x2a\x70", [0x181d] = "\x02\x2a\x76",
  [0x181e] = "\x0e\x2d\x23", [0x181f] = "\x01\x51\x53",
  [0x1820] = "\x01\x51\x50", [0x1821] = "\x02\x2a\x6d",
  [0x1822] = "\x02\x2a\x72", [0x1823] = "\x0e\x2c\x7c",
  [0x1824] = "\x01\x51\x56", [0x1825] = "\x01\x51\x4e",
  [0x1826] = "\x02\x2a\x71", [0x1827] = "\x01\x51\x51",
  [0x1828] = "\x01\x51\x54", [0x1829] = "\x0e\x2c\x79",
  [0x182b] = "\x02\x2a\x74", [0x182c] = "\x0e\x2c\x7d",
  [0x182d] = "\x01\x51\x4f", [0x182e] = "\x02\x2a\x79",
  [0x182f] = "\x01\x51\x52", [0x1830] = "\x0e\x2d\x21",
  [0x1831] = "\x01\x51\x55", [0x1832] = "\x02\x2a\x6e",
  [0x1833] = "\x02\x2a\x73", [0x1834] = "\x02\x2a\x77",
  [0x1835] = "\x02\x2a\x6f", [0x1836] = "\x02\x2a\x6c",
  [0x1837] = "\x0e\x2d\x24", [0x1838] = "\x0e\x2d\x25",
  [0x1839] = "\x02\x2a\x78", [0x183a] = "\x02\x2a\x75",
  [0x183b] = "\x0e\x2d\x22", [0x183c] = "\x0e\x2c\x37",
  [0x1840] = "\x0e\x31\x46", [0x1841] = "\x01\x55\x72",
  [0x1842] = "\x01\x55\x6b", [0x1843] = "\x01\x55\x6e",
  [0x1844] = "\x0e\x31\x4c", [0x1845] = "\x01\x55\x71",
  [0x1846] = "\x0e\x31\x44", [0x1847] = "\x02\x2f\x57",
  [0x1848] = "\x0e\x31\x49", [0x1849] = "\x01\x55\x6c",
  [0x184a] = "\x02\x2f\x55", [0x184b] = "\x0e\x31\x48",
  [0x184c] = "\x01\x55\x70", [0x184d] = "\x0e\x31\x4d",
  [0x184e] = "\x0e\x31\x45", [0x184f] = "\x01\x55\x6d",
  [0x1850] = "\x0e\x31\x43", [0x1851] = "\x02\x2f\x58",
  [0x1852] = "\x01\x55\x6f", [0x1857] = "\x0e\x36\x42",
  [0x1859] = "\x02\x35\x6e", [0x185a] = "\x01\x5b\x25",
  [0x185b] = "\x02\x35\x6d", [0x185c] = "\x02\x35\x6f",
  [0x185d] = "\x01\x5b\x24", [0x185e] = "\x01\x5b\x29",
  [0x185f] = "\x02\x2f\x56", [0x1860] = "\x0e\x31\x4b",
  [0x1861] = "\x02\x35\x6c", [0x1862] = "\x02\x35\x70",
  [0x1863] = "\x0e\x36\x44", [0x1864] = "\x01\x5b\x26",
  [0x1865] = "\x02\x35\x6b", [0x1866] = "\x01\x5b\x28",
  [0x1867] = "\x0e\x36\x45", [0x1868] = "\x01\x5b\x27",
  [0x1869] = "\x0e\x3c\x26", [0x186a] = "\x02\x3c\x4a",
  [0x186b] = "\x0e\x3b\x7d", [0x186c] = "\x02\x3c\x45",
  [0x186d] = "\x0e\x3c\x25", [0x186e] = "\x01\x5f\x5b",
  [0x186f] = "\x01\x5f\x5f", [0x1870] = "\x01\x5f\x5c",
  [0x1871] = "\x02\x3c\x48", [0x1872] = "\x02\x3c\x4b",
  [0x1873] = "\x0e\x3c\x23", [0x1874] = "\x01\x5f\x5d",
  [0x1876] = "\x01\x5f\x5e", [0x1877] = "\x01\x5f\x63",
  [0x1878] = "\x02\x43\x4d", [0x1879] = "\x02\x3c\x49",
  [0x187a] = "\x01\x5f\x61", [0x187b] = "\x02\x3c\x46",
  [0x187c] = "\x02\x3c\x44", [0x187d] = "\x0e\x3b\x7c",
  [0x187e] = "\x01\x5f\x62", [0x187f] = "\x0e\x3b\x7e",
  [0x1880] = "\x02\x3c\x47", [0x1881] = "\x0e\x3c\x24",
  [0x1884] = "\x01\x64\x41", [0x1886] = "\x02\x43\x45",
  [0x1887] = "\x01\x64\x3e", [0x1888] = "\x01\x64\x3f",
  [0x1889] = "\x01\x64\x3d", [0x188a] = "\x02\x43\x4a",
  [0x188b] = "\x02\x43\x49", [0x188c] = "\x02\x43\x46",
  [0x188d] = "\x01\x64\x43", [0x188e] = "\x0e\x41\x5b",
  [0x188f] = "\x0e\x41\x56", [0x1890] = "\x02\x43\x48",
  [0x1891] = "\x01\x5f\x60", [0x1892] = "\x0e\x41\x59",
  [0x1893] = "\x0e\x41\x51", [0x1894] = "\x02\x43\x4c",
  [0x1895] = "\x02\x43\x47", [0x1896] = "\x01\x64\x40",
  [0x1897] = "\x01\x64\x3c", [0x1898] = "\x01\x64\x42",
  [0x1899] = "\x02\x43\x4b", [0x189a] = "\x0e\x47\x34",
  [0x189c] = "\x0e\x47\x35", [0x189d] = "\x01\x69\x25",
  [0x189f] = "\x02\x4a\x5f", [0x18a0] = "\x02\x4a\x5e",
  [0x18a1] = "\x02\x4a\x5d", [0x18a2] = "\x01\x69\x23",
  [0x18a4] = "\x0e\x47\x39", [0x18a5] = "\x0e\x47\x33",
  [0x18a6] = "\x0e\x47\x37", [0x18a8] = "\x01\x69\x24",
  [0x18a9] = "\x02\x50\x6c", [0x18aa] = "\x02\x50\x6f",
  [0x18ab] = "\x01\x6d\x32", [0x18ac] = "\x0e\x4c\x44",
  [0x18ad] = "\x0e\x4c\x46", [0x18ae] = "\x01\x6d\x31",
  [0x18af] = "\x02\x50\x70", [0x18b0] = "\x02\x50\x6b",
  [0x18b1] = "\x01\x6d\x34", [0x18b2] = "\x02\x50\x6d",
  [0x18b3] = "\x0e\x4c\x41", [0x18b4] = "\x01\x6d\x33",
  [0x18b5] = "\x02\x50\x6a", [0x18b6] = "\x0e\x4c\x40",
  [0x18b7] = "\x02\x50\x6e", [0x18b8] = "\x01\x70\x72",
  [0x18b9] = "\x01\x70\x6f", [0x18ba] = "\x02\x57\x46",
  [0x18bb] = "\x02\x57\x45", [0x18bc] = "\x0e\x4c\x43",
  [0x18bd] = "\x02\x57\x44", [0x18be] = "\x02\x57\x3f",
  [0x18bf] = "\x0e\x50\x7d", [0x18c0] = "\x02\x57\x40",
  [0x18c1] = "\x0e\x51\x23", [0x18c3] = "\x0e\x50\x7a",
  [0x18c4] = "\x01\x70\x70", [0x18c5] = "\x0e\x50\x7e",
  [0x18c6] = "\x01\x70\x6d", [0x18c7] = "\x01\x70\x71",
  [0x18c8] = "\x02\x57\x3e", [0x18c9] = "\x01\x70\x6e",
  [0x18ca] = "\x02\x57\x41", [0x18cb] = "\x02\x57\x42",
  [0x18cc] = "\x02\x57\x47", [0x18cd] = "\x0e\x51\x22",
  [0x18ce] = "\x0e\x54\x71", [0x18cf] = "\x02\x57\x43",
  [0x18d0] = "\x0e\x54\x6f", [0x18d1] = "\x0e\x54\x70",
  [0x18d2] = "\x02\x5c\x7d", [0x18d4] = "\x0e\x54\x6c",
  [0x18d5] = "\x0e\x54\x6d", [0x18d6] = "\x01\x73\x70",
  [0x18d7] = "\x0e\x54\x72", [0x18d8] = "\x02\x61\x68",
  [0x18d9] = "\x01\x73\x6f", [0x18da] = "\x02\x61\x66",
  [0x18db] = "\x02\x61\x67", [0x18dc] = "\x01\x76\x4c",
  [0x18dd] = "\x01\x78\x3b", [0x18de] = "\x02\x65\x56",
  [0x18df] = "\x0e\x5a\x43", [0x18e0] = "\x01\x78\x3a",
  [0x18e1] = "\x0e\x5a\x42", [0x18e2] = "\x0e\x65\x26",
  [0x18e3] = "\x02\x68\x7d", [0x18e4] = "\x02\x68\x7e",
  [0x18e5] = "\x0e\x5c\x42", [0x18e6] = "\x01\x79\x71",
  [0x18e7] = "\x0e\x5c\x43", [0x18e8] = "\x02\x68\x7c",
  [0x18e9] = "\x01\x7a\x6f", [0x18eb] = "\x02\x6f\x2e",
  [0x18ec] = "\x01\x7c\x3a", [0x18ed] = "\x02\x70\x38",
  [0x18ee] = "\x02\x70\x39", [0x18ef] = "\x0e\x61\x3f",
  [0x18f0] = "\x01\x45\x4b", [0x18f2] = "\x01\x48\x32",
  [0x18f3] = "\x01\x48\x33", [0x18f4] = "\x01\x4a\x55",
  [0x18f5] = "\x0e\x26\x41", [0x18f6] = "\x02\x27\x32",
  [0x18f7] = "\x01\x51\x57", [0x18f8] = "\x01\x55\x73",
  [0x18f9] = "\x01\x5b\x2a", [0x18fc] = "\x01\x59\x37",
  [0x18fe] = "\x01\x5f\x64", [0x18ff] = "\x01\x5f\x65",
  [0x1900] = "\x01\x5e\x32", [0x1901] = "\x02\x3c\x4c",
  [0x1902] = "\x0e\x65\x27", [0x1903] = "\x01\x64\x44",
  [0x1904] = "\x02\x4a\x61", [0x1905] = "\x02\x4a\x60",
  [0x1906] = "\x0e\x51\x24", [0x1908] = "\x01\x45\x4c",
  [0x1909] = "\x01\x48\x34", [0x190a] = "\x02\x27\x33",
  [0x190b] = "\x01\x4e\x25", [0x190c] = "\x0e\x29\x45",
  [0x190d] = "\x01\x4e\x24", [0x190e] = "\x0e\x2d\x27",
  [0x190f] = "\x02\x2a\x7a", [0x1910] = "\x02\x2a\x7b",
  [0x1911] = "\x0e\x66\x32", [0x1912] = "\x02\x2f\x59",
  [0x1913] = "\x02\x2f\x5a", [0x1914] = "\x01\x55\x74",
  [0x1915] = "\x01\x55\x75", [0x1916] = "\x0e\x36\x48",
  [0x1917] = "\x01\x55\x76", [0x1918] = "\x02\x35\x71",
  [0x1919] = "\x0e\x36\x47", [0x191a] = "\x0e\x36\x46",
  [0x191b] = "\x01\x5b\x2c", [0x191d] = "\x01\x5f\x67",
  [0x191e] = "\x0e\x3c\x29", [0x191f] = "\x01\x5f\x66",
  [0x1920] = "\x02\x43\x4e", [0x1921] = "\x02\x46\x41",
  [0x1922] = "\x02\x4a\x62", [0x1923] = "\x02\x57\x48",
  [0x1924] = "\x0e\x51\x26", [0x1925] = "\x0e\x66\x33",
  [0x1926] = "\x01\x76\x4d", [0x1927] = "\x01\x79\x72",
  [0x1928] = "\x01\x45\x4d", [0x192a] = "\x01\x46\x5c",
  [0x192b] = "\x01\x46\x5d", [0x192c] = "\x01\x46\x5b",
  [0x192d] = "\x01\x46\x5e", [0x192e] = "\x01\x46\x5a",
  [0x192f] = "\x0e\x22\x52", [0x1931] = "\x01\x48\x37",
  [0x1932] = "\x0e\x23\x77", [0x1933] = "\x02\x22\x57",
  [0x1934] = "\x01\x48\x36", [0x1935] = "\x01\x48\x38",
  [0x1936] = "\x0e\x23\x78", [0x1937] = "\x0e\x23\x75",
  [0x1938] = "\x02\x22\x52", [0x1939] = "\x02\x22\x51",
  [0x193a] = "\x02\x22\x54", [0x193b] = "\x02\x22\x53",
  [0x193c] = "\x02\x22\x56", [0x193d] = "\x01\x48\x35",
  [0x193e] = "\x02\x22\x50", [0x193f] = "\x02\x22\x55",
  [0x1944] = "\x0e\x26\x48", [0x1945] = "\x02\x24\x3e",
  [0x1946] = "\x01\x4a\x5f", [0x1947] = "\x02\x24\x3f",
  [0x1948] = "\x02\x24\x43", [0x1949] = "\x01\x4a\x5e",
  [0x194a] = "\x0e\x26\x49", [0x194b] = "\x02\x24\x47",
  [0x194c] = "\x02\x24\x42", [0x194d] = "\x02\x24\x45",
  [0x194e] = "\x01\x4a\x57", [0x194f] = "\x01\x4a\x58",
  [0x1950] = "\x01\x4a\x59", [0x1951] = "\x01\x4a\x5a",
  [0x1952] = "\x0e\x26\x45", [0x1953] = "\x01\x4a\x61",
  [0x1954] = "\x0e\x26\x44", [0x1955] = "\x02\x24\x41",
  [0x1956] = "\x01\x4a\x5c", [0x1957] = "\x01\x4a\x62",
  [0x1958] = "\x0e\x26\x47", [0x1959] = "\x02\x24\x40",
  [0x195a] = "\x02\x24\x46", [0x195b] = "\x0e\x26\x42",
  [0x195c] = "\x01\x4a\x5b", [0x195d] = "\x02\x24\x44",
  [0x195e] = "\x01\x4a\x5d", [0x195f] = "\x01\x4a\x56",
  [0x1960] = "\x01\x4a\x60", [0x1961] = "\x0e\x26\x4a",
  [0x1965] = "\x0e\x62\x57", [0x196a] = "\x01\x4e\x3a",
  [0x196b] = "\x0e\x29\x47", [0x196c] = "\x02\x27\x35",
  [0x196d] = "\x01\x4e\x26", [0x196f] = "\x01\x4e\x30",
  [0x1970] = "\x01\x4e\x31", [0x1971] = "\x01\x4e\x29",
  [0x1972] = "\x01\x4e\x3b", [0x1973] = "\x01\x4e\x2b",
  [0x1974] = "\x02\x27\x3d", [0x1975] = "\x01\x4e\x36",
  [0x1976] = "\x02\x27\x38", [0x1977] = "\x01\x4e\x2c",
  [0x1978] = "\x02\x27\x47", [0x1979] = "\x02\x27\x48",
  [0x197a] = "\x02\x27\x40", [0x197b] = "\x02\x27\x39",
  [0x197c] = "\x01\x4e\x39", [0x197d] = "\x02\x27\x45",
  [0x197e] = "\x01\x4e\x34", [0x197f] = "\x01\x4e\x32",
  [0x1980] = "\x0e\x29\x52", [0x1981] = "\x02\x27\x46",
  [0x1982] = "\x0e\x29\x49", [0x1983] = "\x02\x27\x44",
  [0x1984] = "\x02\x27\x3c", [0x1985] = "\x02\x27\x34",
  [0x1986] = "\x02\x27\x3b", [0x1987] = "\x01\x4e\x2d",
  [0x1988] = "\x0e\x62\x5f", [0x1989] = "\x01\x4e\x33",
  [0x198a] = "\x0e\x29\x4a", [0x198b] = "\x01\x4e\x27",
  [0x198c] = "\x02\x27\x3f", [0x198d] = "\x02\x27\x3e",
  [0x198e] = "\x02\x27\x36", [0x198f] = "\x0e\x29\x4f",
  [0x1990] = "\x01\x4e\x35", [0x1991] = "\x02\x27\x42",
  [0x1992] = "\x02\x27\x37", [0x1993] = "\x01\x4e\x38",
  [0x1994] = "\x02\x27\x49", [0x1995] = "\x01\x4e\x28",
  [0x1996] = "\x0e\x29\x48", [0x1997] = "\x01\x4e\x2f",
  [0x1998] = "\x02\x27\x3a", [0x1999] = "\x02\x27\x43",
  [0x199a] = "\x01\x4e\x37", [0x199c] = "\x01\x4e\x2a",
  [0x199d] = "\x01\x4e\x2e", [0x199f] = "\x02\x27\x41",
  [0x19a2] = "\x0e\x29\x4e", [0x19a3] = "\x0e\x29\x4c",
  [0x19a4] = "\x0e\x65\x28", [0x19ac] = "\x0e\x65\x29",
  [0x19ae] = "\x02\x2b\x3d", [0x19af] = "\x01\x51\x5f",
  [0x19b0] = "\x01\x51\x6c", [0x19b1] = "\x0e\x2d\x36",
  [0x19b2] = "\x02\x2b\x38", [0x19b3] = "\x02\x2b\x2e",
  [0x19b4] = "\x01\x51\x65", [0x19b5] = "\x02\x2b\x2c",
  [0x19b6] = "\x01\x51\x5e", [0x19b7] = "\x02\x2b\x27",
  [0x19b8] = "\x01\x51\x68", [0x19b9] = "\x02\x2b\x34",
  [0x19ba] = "\x02\x2b\x21", [0x19bb] = "\x02\x2b\x23",
  [0x19bc] = "\x0e\x2d\x2e", [0x19bf] = "\x0e\x2d\x30",
  [0x19c0] = "\x02\x2b\x26", [0x19c1] = "\x02\x2a\x7c",
  [0x19c2] = "\x02\x2b\x33", [0x19c3] = "\x02\x2b\x43",
  [0x19c4] = "\x01\x51\x63", [0x19c5] = "\x02\x2b\x28",
  [0x19c6] = "\x02\x2b\x3a", [0x19c7] = "\x0e\x2d\x2d",
  [0x19c8] = "\x02\x2a\x7e", [0x19c9] = "\x02\x2b\x41",
  [0x19ca] = "\x02\x2b\x42", [0x19cb] = "\x02\x2b\x45",
  [0x19cc] = "\x02\x2b\x3c", [0x19cd] = "\x02\x2b\x2d",
  [0x19ce] = "\x02\x2b\x35", [0x19cf] = "\x01\x51\x69",
  [0x19d0] = "\x01\x51\x5c", [0x19d1] = "\x01\x51\x64",
  [0x19d2] = "\x01\x51\x70", [0x19d3] = "\x01\x51\x59",
  [0x19d4] = "\x01\x51\x5b", [0x19d5] = "\x0e\x2d\x31",
  [0x19d6] = "\x0e\x2d\x2b", [0x19d7] = "\x0e\x2d\x3a",
  [0x19d8] = "\x02\x2b\x25", [0x19d9] = "\x01\x51\x6d",
  [0x19da] = "\x01\x51\x66", [0x19db] = "\x02\x2b\x3f",
  [0x19dc] = "\x02\x2b\x22", [0x19dd] = "\x01\x51\x6f",
  [0x19de] = "\x01\x51\x6a", [0x19df] = "\x02\x2b\x2b",
  [0x19e2] = "\x01\x51\x6e", [0x19e3] = "\x02\x2b\x32",
  [0x19e4] = "\x02\x2b\x2a", [0x19e5] = "\x01\x51\x67",
  [0x19e6] = "\x02\x2b\x3e", [0x19e7] = "\x02\x2b\x36",
  [0x19e8] = "\x0e\x2d\x2a", [0x19e9] = "\x01\x51\x61",
  [0x19ea] = "\x02\x2b\x44", [0x19eb] = "\x02\x2b\x29",
  [0x19ec] = "\x01\x51\x5d", [0x19ed] = "\x02\x2b\x3b",
  [0x19ee] = "\x02\x2b\x31", [0x19ef] = "\x01\x51\x62",
  [0x19f0] = "\x02\x2b\x37", [0x19f1] = "\x01\x51\x5a",
  [0x19f2] = "\x02\x2a\x7d", [0x19f3] = "\x01\x51\x6b",
  [0x19f4] = "\x01\x56\x27", [0x19f5] = "\x01\x51\x60",
  [0x19f6] = "\x02\x2b\x30", [0x19f7] = "\x02\x2b\x2f",
  [0x19f8] = "\x02\x2b\x24", [0x19f9] = "\x0e\x29\x51",
  [0x19fa] = "\x02\x2b\x40", [0x19fb] = "\x0e\x2d\x34",
  [0x19fc] = "\x02\x2b\x39", [0x19fe] = "\x0e\x2d\x32",
  [0x19ff] = "\x01\x51\x58", [0x1a00] = "\x0e\x2d\x39",
  [0x1a01] = "\x0e\x2d\x37", [0x1a05] = "\x0e\x2d\x38",
  [0x1a06] = "\x0e\x65\x2b", [0x1a10] = "\x0e\x65\x2a",
  [0x1a12] = "\x02\x2f\x6e", [0x1a13] = "\x01\x56\x2e",
  [0x1a14] = "\x02\x2f\x6f", [0x1a15] = "\x0e\x31\x5d",
  [0x1a16] = "\x02\x2f\x63", [0x1a17] = "\x01\x56\x23",
  [0x1a18] = "\x01\x56\x2f", [0x1a19] = "\x0e\x31\x57",
  [0x1a1a] = "\x02\x2f\x5c", [0x1a1b] = "\x0e\x31\x53",
  [0x1a1c] = "\x02\x2f\x65", [0x1a1d] = "\x02\x2f\x6d",
  [0x1a1e] = "\x0e\x31\x5b", [0x1a1f] = "\x02\x2f\x5b",
  [0x1a20] = "\x02\x2f\x76", [0x1a21] = "\x01\x55\x77",
  [0x1a22] = "\x0e\x31\x5e", [0x1a23] = "\x0e\x31\x64",
  [0x1a24] = "\x0e\x31\x50", [0x1a25] = "\x02\x2f\x75",
  [0x1a26] = "\x02\x2f\x70", [0x1a27] = "\x0e\x31\x5f",
  [0x1a28] = "\x02\x2f\x71", [0x1a29] = "\x01\x56\x21",
  [0x1a2a] = "\x01\x56\x2c", [0x1a2b] = "\x02\x2f\x67",
  [0x1a2c] = "\x0e\x31\x56", [0x1a2d] = "\x02\x2f\x68",
  [0x1a2e] = "\x02\x2f\x72", [0x1a2f] = "\x02\x2f\x69",
  [0x1a30] = "\x0e\x31\x63", [0x1a31] = "\x02\x2f\x64",
  [0x1a32] = "\x02\x2f\x5e", [0x1a33] = "\x02\x2f\x5f",
  [0x1a34] = "\x02\x2f\x6c", [0x1a35] = "\x02\x2f\x66",
  [0x1a36] = "\x0e\x31\x54", [0x1a37] = "\x0e\x31\x4f",
  [0x1a38] = "\x01\x55\x78", [0x1a39] = "\x01\x55\x7c",
  [0x1a3a] = "\x02\x2f\x74", [0x1a3b] = "\x02\x2f\x60",
  [0x1a3c] = "\x01\x56\x2a", [0x1a3d] = "\x01\x56\x26",
  [0x1a3e] = "\x0e\x31\x5a", [0x1a40] = "\x01\x56\x29",
  [0x1a41] = "\x01\x56\x30", [0x1a42] = "\x01\x55\x7d",
  [0x1a43] = "\x01\x56\x2b", [0x1a44] = "\x02\x2f\x6b",
  [0x1a45] = "\x01\x56\x2d", [0x1a46] = "\x01\x55\x7a",
  [0x1a47] = "\x0e\x31\x59", [0x1a48] = "\x01\x55\x79",
  [0x1a49] = "\x02\x2f\x5d", [0x1a4b] = "\x02\x2f\x61",
  [0x1a4c] = "\x01\x56\x24", [0x1a4d] = "\x02\x2f\x73",
  [0x1a4e] = "\x02\x2f\x6a", [0x1a4f] = "\x02\x2f\x62",
  [0x1a50] = "\x01\x56\x28", [0x1a51] = "\x01\x56\x25",
  [0x1a52] = "\x0e\x2d\x2f", [0x1a53] = "\x01\x55\x7b",
  [0x1a54] = "\x01\x55\x7e", [0x1a55] = "\x0e\x31\x62",
  [0x1a56] = "\x0e\x31\x58", [0x1a58] = "\x0e\x31\x61",
  [0x1a59] = "\x0e\x31\x60", [0x1a5a] = "\x0e\x65\x2d",
  [0x1a5e] = "\x0e\x65\x2c", [0x1a6b] = "\x02\x36\x28",
  [0x1a6c] = "\x0e\x36\x55", [0x1a6d] = "\x02\x35\x76",
  [0x1a6e] = "\x02\x35\x77", [0x1a6f] = "\x02\x35\x7b",
  [0x1a70] = "\x0e\x36\x60", [0x1a71] = "\x02\x36\x2c",
  [0x1a72] = "\x02\x36\x29", [0x1a73] = "\x0e\x36\x4e",
  [0x1a74] = "\x02\x36\x22", [0x1a75] = "\x02\x36\x21",
  [0x1a76] = "\x01\x5b\x33", [0x1a77] = "\x02\x36\x25",
  [0x1a78] = "\x02\x36\x34", [0x1a79] = "\x02\x35\x72",
  [0x1a7a] = "\x0e\x36\x5b", [0x1a7b] = "\x02\x36\x35",
  [0x1a7c] = "\x02\x36\x27", [0x1a7d] = "\x02\x36\x39",
  [0x1a7e] = "\x02\x36\x2d", [0x1a7f] = "\x01\x5b\x32",
  [0x1a80] = "\x02\x36\x2b", [0x1a81] = "\x01\x5b\x2d",
  [0x1a82] = "\x01\x5b\x42", [0x1a83] = "\x01\x5b\x38",
  [0x1a84] = "\x0e\x36\x57", [0x1a85] = "\x01\x5b\x3c",
  [0x1a86] = "\x01\x5b\x3b", [0x1a87] = "\x02\x35\x73",
  [0x1a89] = "\x02\x36\x32", [0x1a8a] = "\x02\x36\x38",
  [0x1a8b] = "\x02\x36\x30", [0x1a8c] = "\x02\x36\x37",
  [0x1a8e] = "\x0e\x36\x51", [0x1a8f] = "\x02\x36\x24",
  [0x1a90] = "\x02\x35\x74", [0x1a91] = "\x02\x36\x36",
  [0x1a92] = "\x02\x36\x26", [0x1a93] = "\x01\x5b\x30",
  [0x1a94] = "\x01\x5b\x3d", [0x1a95] = "\x0e\x36\x5a",
  [0x1a96] = "\x02\x36\x2f", [0x1a97] = "\x01\x5b\x36",
  [0x1a98] = "\x0e\x36\x4d", [0x1a99] = "\x0e\x36\x5c",
  [0x1a9a] = "\x0e\x36\x50", [0x1a9b] = "\x02\x36\x2e",
  [0x1a9c] = "\x02\x35\x75", [0x1a9d] = "\x01\x5b\x3e",
  [0x1a9e] = "\x0e\x36\x4b", [0x1a9f] = "\x01\x5b\x40",
  [0x1aa0] = "\x02\x36\x31", [0x1aa1] = "\x01\x5b\x41",
  [0x1aa2] = "\x01\x5b\x2f", [0x1aa3] = "\x02\x35\x7c",
  [0x1aa4] = "\x02\x36\x33", [0x1aa5] = "\x0e\x36\x54",
  [0x1aa6] = "\x0e\x34\x78", [0x1aa7] = "\x01\x5b\x35",
  [0x1aa8] = "\x01\x5b\x3f", [0x1aa9] = "\x02\x35\x7e",
  [0x1aaa] = "\x02\x36\x2a", [0x1aab] = "\x02\x35\x79",
  [0x1aac] = "\x02\x35\x7d", [0x1aad] = "\x01\x5b\x3a",
  [0x1aae] = "\x02\x35\x78", [0x1aaf] = "\x01\x5b\x2e",
  [0x1ab0] = "\x01\x5b\x37", [0x1ab1] = "\x01\x5b\x34",
  [0x1ab2] = "\x02\x36\x23", [0x1ab3] = "\x01\x56\x22",
  [0x1ab4] = "\x02\x3c\x63", [0x1ab5] = "\x01\x5b\x31",
  [0x1ab6] = "\x0e\x36\x4c", [0x1ab7] = "\x0e\x36\x52",
  [0x1ab8] = "\x0e\x36\x5e", [0x1ab9] = "\x0e\x36\x5f",
  [0x1ac1] = "\x0e\x36\x56", [0x1ac3] = "\x0e\x3c\x45",
  [0x1ac4] = "\x01\x5b\x39", [0x1ac5] = "\x0e\x3c\x41",
  [0x1ac6] = "\x02\x3c\x66", [0x1ac7] = "\x02\x3c\x7c",
  [0x1ac8] = "\x02\x3c\x71", [0x1ac9] = "\x01\x5f\x7b",
  [0x1aca] = "\x0e\x3c\x38", [0x1acb] = "\x01\x5f\x76",
  [0x1acc] = "\x02\x3c\x60", [0x1acd] = "\x01\x5f\x77",
  [0x1ace] = "\x02\x3c\x70", [0x1acf] = "\x0e\x3c\x3e",
  [0x1ad0] = "\x02\x3c\x69", [0x1ad1] = "\x02\x3c\x76",
  [0x1ad2] = "\x01\x5f\x73", [0x1ad3] = "\x02\x3c\x4e",
  [0x1ad4] = "\x02\x3c\x78", [0x1ad5] = "\x01\x5f\x69",
  [0x1ad6] = "\x02\x3c\x56", [0x1ad7] = "\x01\x5f\x6c",
  [0x1ad8] = "\x01\x5f\x6b", [0x1ada] = "\x01\x5f\x7c",
  [0x1adb] = "\x0e\x36\x53", [0x1adc] = "\x02\x3c\x50",
  [0x1add] = "\x02\x3c\x72", [0x1ade] = "\x02\x3c\x73",
  [0x1adf] = "\x01\x5f\x6e", [0x1ae0] = "\x01\x5f\x6a",
  [0x1ae1] = "\x02\x3c\x5e", [0x1ae2] = "\x0e\x3c\x3d",
  [0x1ae3] = "\x01\x5f\x75", [0x1ae4] = "\x02\x3c\x59",
  [0x1ae5] = "\x0e\x3c\x32", [0x1ae6] = "\x02\x3c\x74",
  [0x1ae7] = "\x01\x5f\x71", [0x1ae8] = "\x02\x3c\x6c",
  [0x1ae9] = "\x02\x3c\x79", [0x1aea] = "\x02\x3c\x53",
  [0x1aeb] = "\x02\x3c\x58", [0x1aec] = "\x02\x3c\x52",
  [0x1aed] = "\x0e\x3c\x2a", [0x1aee] = "\x01\x5f\x70",
  [0x1aef] = "\x02\x3c\x65", [0x1af0] = "\x02\x43\x64",
  [0x1af1] = "\x02\x3c\x54", [0x1af2] = "\x01\x5f\x74",
  [0x1af3] = "\x02\x3c\x5d", [0x1af4] = "\x02\x3c\x75",
  [0x1af5] = "\x01\x5f\x6f", [0x1af6] = "\x02\x3c\x5a",
  [0x1af7] = "\x02\x3c\x57", [0x1af8] = "\x02\x3c\x68",
  [0x1af9] = "\x01\x5f\x72", [0x1afa] = "\x01\x5f\x68",
  [0x1afb] = "\x01\x5f\x7e", [0x1afc] = "\x02\x3c\x6b",
  [0x1afd] = "\x02\x3c\x6a", [0x1afe] = "\x0e\x3c\x31",
  [0x1aff] = "\x0e\x3c\x42", [0x1b00] = "\x0e\x3c\x39",
  [0x1b01] = "\x0e\x3c\x3b", [0x1b02] = "\x0e\x3c\x34",
  [0x1b03] = "\x0e\x3c\x2f", [0x1b04] = "\x02\x3c\x4f",
  [0x1b05] = "\x01\x5f\x6d", [0x1b06] = "\x02\x3c\x77",
  [0x1b07] = "\x02\x3c\x5f", [0x1b08] = "\x02\x3c\x61",
  [0x1b09] = "\x0e\x3c\x37", [0x1b0a] = "\x02\x3c\x6e",
  [0x1b0b] = "\x02\x3c\x6d", [0x1b0c] = "\x02\x3c\x4d",
  [0x1b0d] = "\x01\x5f\x78", [0x1b0e] = "\x01\x5f\x7a",
  [0x1b0f] = "\x02\x3c\x55", [0x1b10] = "\x02\x3c\x5c",
  [0x1b11] = "\x02\x3c\x64", [0x1b12] = "\x01\x5f\x79",
  [0x1b13] = "\x02\x3c\x5b", [0x1b14] = "\x02\x3c\x67",
  [0x1b15] = "\x02\x3c\x7a", [0x1b17] = "\x02\x3c\x6f",
  [0x1b18] = "\x0e\x3c\x3c", [0x1b1c] = "\x0e\x3c\x44",
  [0x1b1e] = "\x0e\x3c\x33", [0x1b25] = "\x02\x3c\x7b",
  [0x1b29] = "\x0e\x65\x2f", [0x1b2a] = "\x02\x3c\x51",
  [0x1b2c] = "\x0e\x65\x2e", [0x1b2e] = "\x0e\x3c\x40",
  [0x1b2f] = "\x02\x43\x78", [0x1b30] = "\x01\x64\x4c",
  [0x1b31] = "\x0e\x41\x65", [0x1b32] = "\x02\x43\x76",
  [0x1b33] = "\x02\x43\x61", [0x1b34] = "\x02\x43\x66",
  [0x1b35] = "\x02\x43\x5f", [0x1b36] = "\x0e\x41\x77",
  [0x1b37] = "\x02\x43\x72", [0x1b38] = "\x02\x43\x51",
  [0x1b39] = "\x02\x43\x58", [0x1b3b] = "\x02\x43\x70",
  [0x1b3c] = "\x02\x43\x7a", [0x1b3d] = "\x02\x43\x62",
  [0x1b3e] = "\x0e\x41\x68", [0x1b3f] = "\x02\x43\x55",
  [0x1b40] = "\x02\x43\x68", [0x1b41] = "\x02\x43\x6d",
  [0x1b42] = "\x02\x43\x59", [0x1b43] = "\x0e\x41\x6b",
  [0x1b44] = "\x02\x43\x6a", [0x1b45] = "\x02\x43\x56",
  [0x1b46] = "\x0e\x41\x5d", [0x1b47] = "\x0e\x41\x75",
  [0x1b48] = "\x02\x43\x5d", [0x1b49] = "\x02\x43\x5e",
  [0x1b4a] = "\x01\x64\x4e", [0x1b4b] = "\x02\x43\x71",
  [0x1b4c] = "\x02\x43\x6f", [0x1b4d] = "\x0e\x41\x73",
  [0x1b4e] = "\x02\x43\x52", [0x1b4f] = "\x02\x43\x74",
  [0x1b50] = "\x0e\x41\x74", [0x1b51] = "\x02\x43\x75",
  [0x1b52] = "\x02\x43\x77", [0x1b53] = "\x01\x64\x52",
  [0x1b54] = "\x01\x64\x4a", [0x1b55] = "\x0e\x41\x6f",
  [0x1b56] = "\x02\x35\x7a", [0x1b57] = "\x02\x43\x5a",
  [0x1b58] = "\x02\x43\x6c", [0x1b59] = "\x02\x43\x5b",
  [0x1b5a] = "\x01\x64\x47", [0x1b5b] = "\x01\x64\x57",
  [0x1b5c] = "\x02\x43\x73", [0x1b5d] = "\x01\x64\x55",
  [0x1b5e] = "\x01\x64\x51", [0x1b5f] = "\x02\x43\x50",
  [0x1b60] = "\x01\x64\x49", [0x1b61] = "\x0e\x41\x79",
  [0x1b62] = "\x02\x43\x53", [0x1b63] = "\x01\x64\x56",
  [0x1b64] = "\x0e\x41\x78", [0x1b65] = "\x02\x43\x63",
  [0x1b66] = "\x02\x43\x4f", [0x1b67] = "\x0e\x41\x76",
  [0x1b68] = "\x01\x64\x4f", [0x1b69] = "\x02\x43\x67",
  [0x1b6a] = "\x02\x43\x57", [0x1b6b] = "\x01\x64\x50",
  [0x1b6c] = "\x02\x43\x60", [0x1b6d] = "\x01\x64\x46",
  [0x1b6e] = "\x01\x5f\x7d", [0x1b6f] = "\x02\x43\x69",
  [0x1b70] = "\x02\x3c\x62", [0x1b71] = "\x02\x43\x54",
  [0x1b73] = "\x0e\x41\x6d", [0x1b74] = "\x02\x43\x6e",
  [0x1b75] = "\x01\x64\x4b", [0x1b76] = "\x02\x43\x6b",
  [0x1b77] = "\x01\x64\x48", [0x1b78] = "\x02\x43\x65",
  [0x1b79] = "\x01\x64\x53", [0x1b7a] = "\x02\x43\x5c",
  [0x1b7b] = "\x02\x43\x79", [0x1b7c] = "\x0e\x41\x6a",
  [0x1b7d] = "\x0e\x41\x7b", [0x1b82] = "\x01\x64\x4d",
  [0x1b83] = "\x02\x45\x2c", [0x1b85] = "\x0e\x41\x72",
  [0x1b86] = "\x01\x64\x54", [0x1b8d] = "\x02\x4a\x70",
  [0x1b8e] = "\x02\x4a\x6e", [0x1b8f] = "\x0e\x47\x3a",
  [0x1b90] = "\x02\x4b\x26", [0x1b91] = "\x02\x4a\x6c",
  [0x1b92] = "\x0e\x47\x3d", [0x1b93] = "\x02\x4a\x7e",
  [0x1b94] = "\x01\x64\x45", [0x1b95] = "\x01\x69\x28",
  [0x1b96] = "\x02\x4a\x68", [0x1b97] = "\x02\x4b\x25",
  [0x1b98] = "\x0e\x47\x51", [0x1b99] = "\x02\x4a\x6d",
  [0x1b9a] = "\x02\x4a\x7b", [0x1b9b] = "\x01\x69\x2d",
  [0x1b9c] = "\x01\x69\x26", [0x1b9d] = "\x0e\x47\x4e",
  [0x1b9e] = "\x02\x4b\x23", [0x1b9f] = "\x0e\x47\x46",
  [0x1ba0] = "\x02\x4a\x66", [0x1ba1] = "\x02\x4b\x22",
  [0x1ba2] = "\x0e\x47\x47", [0x1ba3] = "\x01\x69\x38",
  [0x1ba4] = "\x02\x4a\x77", [0x1ba5] = "\x02\x4b\x29",
  [0x1ba6] = "\x01\x69\x36", [0x1ba7] = "\x02\x4a\x6f",
  [0x1ba8] = "\x01\x69\x27", [0x1ba9] = "\x02\x4a\x71",
  [0x1baa] = "\x02\x4b\x21", [0x1bab] = "\x01\x69\x30",
  [0x1bac] = "\x02\x4a\x6a", [0x1bad] = "\x01\x69\x34",
  [0x1bae] = "\x01\x69\x2a", [0x1baf] = "\x02\x4a\x73",
  [0x1bb0] = "\x02\x4a\x69", [0x1bb1] = "\x02\x4a\x63",
  [0x1bb2] = "\x0e\x47\x3e", [0x1bb3] = "\x02\x4a\x7d",
  [0x1bb4] = "\x01\x69\x31", [0x1bb5] = "\x02\x4b\x28",
  [0x1bb6] = "\x02\x4a\x64", [0x1bb7] = "\x01\x69\x2e",
  [0x1bb9] = "\x02\x4a\x79", [0x1bba] = "\x0e\x63\x52",
  [0x1bbb] = "\x01\x69\x2f", [0x1bbc] = "\x02\x4a\x6b",
  [0x1bbd] = "\x02\x4a\x76", [0x1bbe] = "\x02\x4a\x72",
  [0x1bbf] = "\x02\x4a\x74", [0x1bc0] = "\x0e\x47\x43",
  [0x1bc1] = "\x01\x69\x29", [0x1bc2] = "\x02\x4b\x27",
  [0x1bc3] = "\x01\x69\x37", [0x1bc4] = "\x02\x4a\x75",
  [0x1bc5] = "\x0e\x47\x3b", [0x1bc6] = "\x02\x4b\x2a",
  [0x1bc8] = "\x0e\x47\x3c", [0x1bc9] = "\x02\x4a\x65",
  [0x1bca] = "\x02\x4a\x7a", [0x1bcb] = "\x01\x69\x2c",
  [0x1bcc] = "\x01\x69\x35", [0x1bcd] = "\x01\x69\x33",
  [0x1bce] = "\x02\x4a\x67", [0x1bcf] = "\x02\x4a\x7c",
  [0x1bd0] = "\x01\x69\x32", [0x1bd1] = "\x0e\x47\x45",
  [0x1bd2] = "\x0e\x47\x48", [0x1bd3] = "\x01\x69\x2b",
  [0x1bd4] = "\x02\x4a\x78", [0x1bd5] = "\x0e\x47\x4d",
  [0x1bd6] = "\x0e\x47\x44", [0x1bd8] = "\x0e\x4c\x54",
  [0x1bd9] = "\x02\x4b\x24", [0x1be1] = "\x0e\x47\x4c",
  [0x1be2] = "\x02\x50\x7c", [0x1be3] = "\x0e\x47\x42",
  [0x1be4] = "\x02\x50\x78", [0x1be5] = "\x02\x50\x74",
  [0x1be6] = "\x02\x51\x2a", [0x1be7] = "\x02\x51\x27",
  [0x1be8] = "\x01\x6d\x37", [0x1be9] = "\x0e\x4c\x64",
  [0x1bea] = "\x0e\x4c\x60", [0x1beb] = "\x02\x51\x2e",
  [0x1bec] = "\x02\x50\x7b", [0x1bed] = "\x01\x6d\x42",
  [0x1bee] = "\x02\x51\x24", [0x1bef] = "\x0e\x4c\x4f",
  [0x1bf0] = "\x0e\x4c\x51", [0x1bf1] = "\x02\x50\x77",
  [0x1bf2] = "\x02\x51\x23", [0x1bf3] = "\x01\x6d\x3f",
  [0x1bf4] = "\x02\x51\x37", [0x1bf6] = "\x02\x51\x34",
  [0x1bf7] = "\x02\x51\x26", [0x1bf8] = "\x02\x50\x75",
  [0x1bf9] = "\x0e\x4c\x5f", [0x1bfa] = "\x0e\x4c\x57",
  [0x1bfb] = "\x02\x51\x2b", [0x1bfc] = "\x02\x51\x2d",
  [0x1bfd] = "\x01\x6d\x3b", [0x1bfe] = "\x02\x51\x21",
  [0x1bff] = "\x02\x50\x7a", [0x1c00] = "\x02\x50\x71",
  [0x1c01] = "\x01\x6d\x38", [0x1c02] = "\x01\x6d\x40",
  [0x1c04] = "\x02\x51\x30", [0x1c05] = "\x01\x6d\x41",
  [0x1c06] = "\x02\x50\x72", [0x1c07] = "\x02\x51\x36",
  [0x1c08] = "\x02\x51\x29", [0x1c09] = "\x02\x51\x2f",
  [0x1c0a] = "\x01\x6d\x3e", [0x1c0b] = "\x0e\x4c\x4d",
  [0x1c0c] = "\x0e\x4c\x50", [0x1c0d] = "\x02\x51\x2c",
  [0x1c0e] = "\x0e\x4c\x47", [0x1c0f] = "\x02\x51\x33",
  [0x1c10] = "\x0e\x4c\x5b", [0x1c11] = "\x01\x6d\x43",
  [0x1c12] = "\x0e\x4c\x56", [0x1c13] = "\x01\x6d\x3d",
  [0x1c14] = "\x02\x51\x25", [0x1c15] = "\x02\x50\x76",
  [0x1c16] = "\x02\x51\x38", [0x1c17] = "\x02\x50\x73",
  [0x1c18] = "\x02\x51\x31", [0x1c19] = "\x01\x6d\x3a",
  [0x1c1a] = "\x0e\x4c\x4a", [0x1c1b] = "\x02\x50\x7d",
  [0x1c1c] = "\x0e\x4c\x49", [0x1c1d] = "\x02\x50\x7e",
  [0x1c1e] = "\x01\x6d\x39", [0x1c1f] = "\x01\x6d\x36",
  [0x1c20] = "\x02\x50\x79", [0x1c21] = "\x01\x6d\x3c",
  [0x1c22] = "\x0e\x4c\x52", [0x1c23] = "\x01\x6d\x35",
  [0x1c24] = "\x0e\x4c\x62", [0x1c25] = "\x02\x51\x32",
  [0x1c26] = "\x02\x51\x35", [0x1c27] = "\x02\x51\x22",
  [0x1c28] = "\x02\x57\x55", [0x1c29] = "\x0e\x4c\x5e",
  [0x1c2a] = "\x0e\x4c\x59", [0x1c2c] = "\x0e\x4c\x61",
  [0x1c2d] = "\x0e\x65\x30", [0x1c32] = "\x02\x57\x4d",
  [0x1c33] = "\x0e\x51\x2c", [0x1c34] = "\x02\x57\x49",
  [0x1c35] = "\x01\x71\x21", [0x1c36] = "\x0e\x51\x3c",
  [0x1c37] = "\x0e\x51\x38", [0x1c38] = "\x01\x70\x74",
  [0x1c39] = "\x01\x70\x79", [0x1c3a] = "\x01\x70\x75",
  [0x1c3b] = "\x02\x57\x57", [0x1c3c] = "\x02\x57\x62",
  [0x1c3d] = "\x01\x70\x73", [0x1c3e] = "\x02\x57\x4f",
  [0x1c3f] = "\x02\x57\x58", [0x1c40] = "\x02\x51\x28",
  [0x1c41] = "\x02\x57\x59", [0x1c42] = "\x0e\x51\x28",
  [0x1c43] = "\x0e\x51\x2b", [0x1c44] = "\x01\x70\x7a",
  [0x1c45] = "\x0e\x51\x40", [0x1c46] = "\x02\x57\x68",
  [0x1c47] = "\x01\x70\x7e", [0x1c48] = "\x01\x71\x23",
  [0x1c49] = "\x02\x57\x4b", [0x1c4a] = "\x0e\x51\x3b",
  [0x1c4b] = "\x01\x70\x7d", [0x1c4c] = "\x0e\x51\x31",
  [0x1c4d] = "\x02\x57\x66", [0x1c4e] = "\x02\x57\x67",
  [0x1c4f] = "\x02\x57\x5d", [0x1c50] = "\x02\x57\x5c",
  [0x1c51] = "\x02\x57\x54", [0x1c52] = "\x0e\x51\x29",
  [0x1c53] = "\x0e\x51\x3e", [0x1c54] = "\x02\x57\x5e",
  [0x1c55] = "\x02\x57\x65", [0x1c56] = "\x02\x57\x64",
  [0x1c57] = "\x0e\x51\x2f", [0x1c58] = "\x01\x70\x78",
  [0x1c59] = "\x01\x70\x76", [0x1c5a] = "\x02\x57\x56",
  [0x1c5b] = "\x02\x57\x53", [0x1c5c] = "\x0e\x51\x44",
  [0x1c5d] = "\x02\x57\x50", [0x1c5e] = "\x02\x57\x63",
  [0x1c5f] = "\x01\x71\x22", [0x1c60] = "\x02\x57\x61",
  [0x1c61] = "\x01\x70\x7c", [0x1c62] = "\x01\x70\x7b",
  [0x1c63] = "\x0e\x51\x27", [0x1c64] = "\x02\x57\x5b",
  [0x1c66] = "\x02\x57\x4a", [0x1c67] = "\x02\x57\x4c",
  [0x1c68] = "\x02\x57\x4e", [0x1c69] = "\x02\x57\x60",
  [0x1c6a] = "\x02\x57\x5a", [0x1c6b] = "\x01\x70\x77",
  [0x1c6c] = "\x0e\x51\x2e", [0x1c6d] = "\x02\x57\x51",
  [0x1c6e] = "\x0e\x51\x32", [0x1c6f] = "\x02\x57\x5f",
  [0x1c70] = "\x0e\x51\x41", [0x1c71] = "\x0e\x51\x3a",
  [0x1c76] = "\x02\x57\x52", [0x1c77] = "\x0e\x65\x31",
  [0x1c7a] = "\x0e\x65\x32", [0x1c7d] = "\x0e\x54\x75",
  [0x1c7e] = "\x01\x73\x78", [0x1c7f] = "\x02\x5d\x31",
  [0x1c80] = "\x01\x73\x71", [0x1c81] = "\x02\x5d\x22",
  [0x1c82] = "\x0e\x54\x7b", [0x1c83] = "\x02\x5d\x2d",
  [0x1c84] = "\x01\x73\x73", [0x1c85] = "\x02\x5d\x34",
  [0x1c86] = "\x0e\x55\x21", [0x1c87] = "\x02\x5d\x29",
  [0x1c88] = "\x0e\x54\x7c", [0x1c89] = "\x02\x5d\x24",
  [0x1c8a] = "\x0e\x54\x76", [0x1c8c] = "\x02\x5d\x35",
  [0x1c8d] = "\x02\x5c\x7e", [0x1c8e] = "\x02\x5d\x2b",
  [0x1c8f] = "\x0e\x55\x27", [0x1c90] = "\x01\x73\x7a",
  [0x1c91] = "\x02\x5d\x30", [0x1c92] = "\x02\x5d\x36",
  [0x1c93] = "\x02\x5d\x2a", [0x1c94] = "\x01\x73\x72",
  [0x1c95] = "\x02\x5d\x2c", [0x1c96] = "\x02\x5d\x21",
  [0x1c97] = "\x01\x73\x79", [0x1c98] = "\x0e\x55\x23",
  [0x1c99] = "\x0e\x54\x7a", [0x1c9a] = "\x02\x5d\x33",
  [0x1c9b] = "\x02\x5d\x26", [0x1c9c] = "\x01\x73\x75",
  [0x1c9d] = "\x0e\x55\x24", [0x1c9e] = "\x02\x5d\x28",
  [0x1c9f] = "\x02\x5d\x25", [0x1ca0] = "\x01\x73\x7b",
  [0x1ca1] = "\x02\x5d\x27", [0x1ca2] = "\x01\x73\x74",
  [0x1ca3] = "\x01\x73\x77", [0x1ca4] = "\x02\x5d\x2f",
  [0x1ca5] = "\x02\x5d\x23", [0x1ca6] = "\x02\x5d\x32",
  [0x1ca7] = "\x0e\x54\x78", [0x1ca8] = "\x02\x5d\x2e",
  [0x1caa] = "\x0e\x55\x29", [0x1cab] = "\x0e\x57\x74",
  [0x1cac] = "\x01\x76\x4f", [0x1cad] = "\x02\x61\x73",
  [0x1cae] = "\x01\x76\x54", [0x1caf] = "\x01\x76\x55",
  [0x1cb0] = "\x0e\x57\x77", [0x1cb1] = "\x0e\x58\x21",
  [0x1cb2] = "\x0e\x57\x76", [0x1cb3] = "\x01\x76\x4e",
  [0x1cb4] = "\x02\x61\x72", [0x1cb5] = "\x0e\x57\x78",
  [0x1cb6] = "\x02\x61\x6f", [0x1cb7] = "\x02\x61\x70",
  [0x1cb8] = "\x01\x76\x52", [0x1cb9] = "\x02\x61\x6a",
  [0x1cba] = "\x02\x61\x6e", [0x1cbb] = "\x01\x76\x51",
  [0x1cbc] = "\x0e\x57\x7e", [0x1cbd] = "\x02\x61\x6b",
  [0x1cbe] = "\x0e\x57\x79", [0x1cbf] = "\x0e\x58\x23",
  [0x1cc0] = "\x0e\x57\x7d", [0x1cc1] = "\x0e\x57\x7a",
  [0x1cc2] = "\x01\x76\x53", [0x1cc3] = "\x01\x76\x50",
  [0x1cc4] = "\x0e\x58\x22", [0x1cc5] = "\x02\x61\x69",
  [0x1cc6] = "\x02\x61\x6d", [0x1cc7] = "\x02\x61\x71",
  [0x1cc8] = "\x0e\x57\x7b", [0x1cca] = "\x0e\x65\x33",
  [0x1ccb] = "\x02\x65\x5c", [0x1ccc] = "\x02\x65\x59",
  [0x1ccd] = "\x02\x65\x62", [0x1cce] = "\x0e\x5a\x44",
  [0x1ccf] = "\x02\x65\x61", [0x1cd0] = "\x02\x65\x5f",
  [0x1cd1] = "\x02\x65\x5a", [0x1cd2] = "\x0e\x5a\x4f",
  [0x1cd3] = "\x01\x78\x3f", [0x1cd4] = "\x0e\x5a\x45",
  [0x1cd5] = "\x0e\x5a\x4e", [0x1cd7] = "\x0e\x5a\x47",
  [0x1cd8] = "\x0e\x5a\x4d", [0x1cd9] = "\x02\x65\x5b",
  [0x1cda] = "\x01\x78\x3e", [0x1cdb] = "\x01\x73\x76",
  [0x1cdc] = "\x02\x65\x5e", [0x1cdd] = "\x01\x78\x3d",
  [0x1cde] = "\x02\x65\x63", [0x1cdf] = "\x02\x65\x5d",
  [0x1ce0] = "\x02\x65\x58", [0x1ce1] = "\x02\x61\x6c",
  [0x1ce2] = "\x0e\x5a\x48", [0x1ce3] = "\x0e\x5a\x46",
  [0x1ce5] = "\x01\x78\x3c", [0x1ce6] = "\x0e\x65\x34",
  [0x1ce7] = "\x02\x65\x57", [0x1ce8] = "\x02\x69\x24",
  [0x1ce9] = "\x0e\x5c\x49", [0x1cea] = "\x02\x69\x23",
  [0x1ceb] = "\x02\x65\x60", [0x1cec] = "\x01\x79\x73",
  [0x1ced] = "\x0e\x57\x75", [0x1cee] = "\x02\x69\x27",
  [0x1cef] = "\x02\x69\x28", [0x1cf0] = "\x02\x69\x22",
  [0x1cf1] = "\x02\x69\x26", [0x1cf3] = "\x02\x69\x21",
  [0x1cf4] = "\x0e\x5c\x47", [0x1cf5] = "\x0e\x65\x35",
  [0x1cf8] = "\x02\x6b\x5f", [0x1cf9] = "\x02\x69\x25",
  [0x1cfa] = "\x01\x7a\x72", [0x1cfb] = "\x01\x7a\x70",
  [0x1cfc] = "\x02\x6b\x5d", [0x1cfd] = "\x0e\x5e\x35",
  [0x1cfe] = "\x0e\x5e\x34", [0x1cff] = "\x0e\x5e\x33",
  [0x1d00] = "\x02\x6b\x60", [0x1d01] = "\x0e\x5e\x37",
  [0x1d02] = "\x02\x6b\x5c", [0x1d03] = "\x02\x6b\x5e",
  [0x1d04] = "\x01\x7a\x71", [0x1d06] = "\x0e\x5f\x4f",
  [0x1d07] = "\x0e\x5f\x4e", [0x1d08] = "\x02\x6d\x53",
  [0x1d09] = "\x02\x6d\x54", [0x1d0a] = "\x01\x7b\x62",
  [0x1d0b] = "\x02\x6d\x52", [0x1d0d] = "\x0e\x65\x36",
  [0x1d0e] = "\x0e\x65\x37", [0x1d0f] = "\x02\x6f\x31",
  [0x1d10] = "\x01\x7c\x3b", [0x1d11] = "\x02\x6f\x2f",
  [0x1d12] = "\x02\x6f\x30", [0x1d13] = "\x02\x70\x3a",
  [0x1d14] = "\x0e\x60\x74", [0x1d16] = "\x01\x7d\x23",
  [0x1d17] = "\x02\x71\x23", [0x1d18] = "\x02\x71\x21",
  [0x1d19] = "\x02\x71\x22", [0x1d1a] = "\x02\x71\x24",
  [0x1d1b] = "\x0e\x61\x40", [0x1d1c] = "\x0e\x61\x65",
  [0x1d1d] = "\x0e\x61\x41", [0x1d1e] = "\x02\x72\x2b",
  [0x1d20] = "\x01\x45\x4e", [0x1d21] = "\x01\x48\x39",
  [0x1d22] = "\x0e\x23\x79", [0x1d23] = "\x01\x4e\x3c",
  [0x1d25] = "\x02\x27\x4a", [0x1d26] = "\x0e\x29\x53",
  [0x1d27] = "\x0e\x29\x54", [0x1d28] = "\x02\x2b\x46",
  [0x1d29] = "\x0e\x62\x69", [0x1d2a] = "\x0e\x2d\x3c",
  [0x1d2b] = "\x0e\x31\x66", [0x1d2c] = "\x02\x2f\x77",
  [0x1d2d] = "\x02\x2f\x79", [0x1d2e] = "\x0e\x31\x65",
  [0x1d2f] = "\x02\x2f\x78", [0x1d30] = "\x0e\x31\x67",
  [0x1d31] = "\x02\x2f\x7a", [0x1d32] = "\x01\x5b\x43",
  [0x1d33] = "\x02\x36\x3b", [0x1d34] = "\x02\x2f\x7b",
  [0x1d35] = "\x0e\x36\x62", [0x1d36] = "\x02\x36\x3a",
  [0x1d37] = "\x02\x36\x3c", [0x1d38] = "\x02\x36\x3d",
  [0x1d39] = "\x02\x3c\x7d", [0x1d3a] = "\x01\x60\x22",
  [0x1d3b] = "\x02\x3c\x7e", [0x1d3c] = "\x02\x3d\x22",
  [0x1d3d] = "\x01\x60\x23", [0x1d3e] = "\x01\x60\x21",
  [0x1d3f] = "\x02\x3d\x21", [0x1d40] = "\x0e\x41\x7d",
  [0x1d41] = "\x02\x44\x22", [0x1d42] = "\x02\x43\x7e",
  [0x1d43] = "\x02\x43\x7d", [0x1d44] = "\x0e\x41\x7c",
  [0x1d45] = "\x02\x43\x7c", [0x1d46] = "\x02\x43\x7b",
  [0x1d47] = "\x01\x64\x58", [0x1d48] = "\x02\x44\x21",
  [0x1d49] = "\x01\x69\x39", [0x1d4a] = "\x02\x4b\x2b",
  [0x1d4b] = "\x02\x4b\x2d", [0x1d4c] = "\x01\x69\x3a",
  [0x1d4d] = "\x02\x4b\x2c", [0x1d4e] = "\x01\x6d\x45",
  [0x1d4f] = "\x0e\x4c\x66", [0x1d50] = "\x01\x6d\x44",
  [0x1d51] = "\x02\x51\x39", [0x1d52] = "\x0e\x4c\x65",
  [0x1d53] = "\x0e\x4c\x67", [0x1d54] = "\x02\x57\x6a",
  [0x1d55] = "\x02\x57\x69", [0x1d56] = "\x02\x57\x6b",
  [0x1d57] = "\x0e\x51\x46", [0x1d58] = "\x0e\x51\x45",
  [0x1d59] = "\x01\x71\x24", [0x1d5b] = "\x02\x5d\x37",
  [0x1d5c] = "\x01\x73\x7c", [0x1d5d] = "\x0e\x55\x2b",
  [0x1d5e] = "\x02\x61\x74", [0x1d5f] = "\x01\x76\x56",
  [0x1d60] = "\x02\x65\x64", [0x1d61] = "\x01\x7b\x63",
  [0x1d62] = "\x01\x45\x4f", [0x1d63] = "\x01\x46\x5f",
  [0x1d64] = "\x01\x48\x3a", [0x1d65] = "\x01\x4a\x63",
  [0x1d66] = "\x01\x4e\x3d", [0x1d67] = "\x01\x4e\x3e",
  [0x1d68] = "\x0e\x29\x55", [0x1d69] = "\x0e\x29\x56",
  [0x1d6a] = "\x01\x51\x71", [0x1d6b] = "\x0e\x2d\x3d",
  [0x1d6c] = "\x0e\x31\x68", [0x1d6d] = "\x02\x2f\x7c",
  [0x1d6e] = "\x0e\x3c\x49", [0x1d6f] = "\x0e\x3c\x47",
  [0x1d71] = "\x0e\x41\x7e", [0x1d72] = "\x01\x64\x59",
  [0x1d73] = "\x0e\x42\x21", [0x1d74] = "\x0e\x47\x53",
  [0x1d76] = "\x02\x52\x59", [0x1d77] = "\x01\x71\x25",
  [0x1d78] = "\x01\x76\x57", [0x1d79] = "\x01\x45\x50",
  [0x1d7a] = "\x0e\x22\x53", [0x1d7b] = "\x01\x48\x3b",
  [0x1d7e] = "\x02\x27\x4c", [0x1d7f] = "\x01\x4e\x3f",
  [0x1d80] = "\x02\x27\x4b", [0x1d81] = "\x0e\x29\x58",
  [0x1d82] = "\x02\x2b\x47", [0x1d83] = "\x01\x51\x72",
  [0x1d84] = "\x02\x2b\x48", [0x1d86] = "\x01\x51\x73",
  [0x1d88] = "\x02\x2f\x7e", [0x1d89] = "\x01\x56\x32",
  [0x1d8a] = "\x01\x56\x31", [0x1d8b] = "\x0e\x31\x6a",
  [0x1d8c] = "\x02\x36\x42", [0x1d8d] = "\x02\x36\x40",
  [0x1d8e] = "\x02\x36\x41", [0x1d8f] = "\x02\x36\x3f",
  [0x1d90] = "\x0e\x36\x64", [0x1d91] = "\x02\x36\x3e",
  [0x1d94] = "\x02\x3d\x23", [0x1d95] = "\x02\x3d\x26",
  [0x1d96] = "\x01\x60\x25", [0x1d97] = "\x02\x3d\x24",
  [0x1d98] = "\x01\x60\x24", [0x1d99] = "\x02\x3d\x25",
  [0x1d9b] = "\x02\x44\x23", [0x1d9c] = "\x0e\x42\x22",
  [0x1d9d] = "\x0e\x63\x54", [0x1d9e] = "\x02\x4b\x2e",
  [0x1d9f] = "\x02\x4b\x2f", [0x1da0] = "\x02\x4b\x30",
  [0x1da1] = "\x0e\x47\x54", [0x1da2] = "\x02\x51\x3c",
  [0x1da3] = "\x02\x51\x3b", [0x1da4] = "\x01\x6d\x46",
  [0x1da5] = "\x02\x51\x3a", [0x1da6] = "\x02\x51\x3d",
  [0x1da7] = "\x02\x57\x6c", [0x1daa] = "\x02\x57\x6d",
  [0x1dab] = "\x02\x57\x6e", [0x1dad] = "\x02\x5d\x38",
  [0x1dae] = "\x01\x73\x7d", [0x1daf] = "\x01\x76\x58",
  [0x1db0] = "\x02\x65\x65", [0x1db2] = "\x01\x7a\x73",
  [0x1db3] = "\x02\x21\x3f", [0x1db4] = "\x0e\x29\x59",
  [0x1db5] = "\x01\x51\x74", [0x1db6] = "\x02\x2b\x49",
  [0x1db7] = "\x01\x56\x33", [0x1dba] = "\x01\x5b\x44",
  [0x1dbb] = "\x0e\x36\x65", [0x1dbc] = "\x01\x60\x26",
  [0x1dbd] = "\x02\x3d\x27", [0x1dbe] = "\x0e\x3c\x4b",
  [0x1dbf] = "\x01\x64\x5b", [0x1dc0] = "\x01\x64\x5a",
  [0x1dc1] = "\x0e\x42\x26", [0x1dc3] = "\x02\x4b\x31",
  [0x1dc4] = "\x02\x4b\x32", [0x1dc5] = "\x01\x6d\x47",
  [0x1dc6] = "\x01\x6d\x48", [0x1dc7] = "\x02\x57\x70",
  [0x1dc8] = "\x02\x57\x6f", [0x1dc9] = "\x02\x61\x75",
  [0x1dca] = "\x02\x6f\x32", [0x1dcb] = "\x01\x45\x51",
  [0x1dcc] = "\x02\x21\x40", [0x1dcd] = "\x01\x46\x60",
  [0x1dce] = "\x0e\x23\x7a", [0x1dcf] = "\x01\x4a\x64",
  [0x1dd0] = "\x02\x24\x48", [0x1dd1] = "\x0e\x29\x5a",
  [0x1dd2] = "\x01\x51\x75", [0x1dd3] = "\x01\x64\x5c",
  [0x1dd4] = "\x01\x45\x52", [0x1dd6] = "\x02\x2b\x4a",
  [0x1dd7] = "\x01\x51\x76", [0x1dd8] = "\x02\x2b\x4b",
  [0x1dda] = "\x01\x73\x7e", [0x1ddb] = "\x01\x45\x53",
  [0x1ddc] = "\x0e\x65\x38", [0x1ddd] = "\x0e\x65\x39",
  [0x1dde] = "\x02\x27\x4d", [0x1de0] = "\x02\x2b\x4c",
  [0x1de1] = "\x0e\x2d\x42", [0x1de2] = "\x02\x30\x25",
  [0x1de3] = "\x02\x30\x24", [0x1de4] = "\x02\x30\x22",
  [0x1de5] = "\x0e\x31\x6e", [0x1de6] = "\x02\x30\x21",
  [0x1de7] = "\x02\x30\x26", [0x1de8] = "\x02\x30\x23",
  [0x1de9] = "\x0e\x31\x6c", [0x1dea] = "\x0e\x31\x6d",
  [0x1deb] = "\x01\x5b\x45", [0x1dec] = "\x01\x5b\x46",
  [0x1ded] = "\x0e\x36\x66", [0x1def] = "\x01\x60\x27",
  [0x1df0] = "\x02\x3d\x28", [0x1df2] = "\x02\x3d\x29",
  [0x1df3] = "\x02\x3d\x2a", [0x1df4] = "\x0e\x3c\x4d",
  [0x1df6] = "\x0e\x65\x3a", [0x1df7] = "\x02\x44\x27",
  [0x1df8] = "\x02\x44\x28", [0x1df9] = "\x02\x44\x26",
  [0x1dfa] = "\x0e\x42\x28", [0x1dfb] = "\x02\x44\x24",
  [0x1dfc] = "\x02\x44\x25", [0x1dfd] = "\x01\x64\x5d",
  [0x1dfe] = "\x02\x4b\x33", [0x1dff] = "\x02\x51\x40",
  [0x1e00] = "\x02\x51\x3f", [0x1e01] = "\x02\x51\x3e",
  [0x1e02] = "\x02\x51\x41", [0x1e03] = "\x02\x57\x72",
  [0x1e04] = "\x02\x57\x71", [0x1e05] = "\x01\x71\x26",
  [0x1e06] = "\x02\x57\x73", [0x1e08] = "\x01\x74\x21",
  [0x1e09] = "\x02\x5d\x39", [0x1e0a] = "\x0e\x55\x2d",
  [0x1e0b] = "\x02\x61\x76", [0x1e0c] = "\x02\x65\x66",
  [0x1e0d] = "\x02\x6d\x55", [0x1e0f] = "\x01\x45\x54",
  [0x1e10] = "\x01\x46\x62", [0x1e11] = "\x01\x46\x61",
  [0x1e13] = "\x01\x4e\x40", [0x1e14] = "\x02\x21\x41",
  [0x1e15] = "\x02\x21\x5f", [0x1e16] = "\x01\x48\x3c",
  [0x1e18] = "\x02\x22\x58", [0x1e19] = "\x02\x24\x49",
  [0x1e1a] = "\x02\x24\x4a", [0x1e1b] = "\x01\x4e\x41",
  [0x1e1c] = "\x0e\x29\x5d", [0x1e1d] = "\x02\x27\x4e",
  [0x1e1e] = "\x0e\x65\x3b", [0x1e1f] = "\x01\x51\x77",
  [0x1e20] = "\x02\x2b\x4d", [0x1e21] = "\x02\x2b\x4e",
  [0x1e23] = "\x01\x56\x34", [0x1e24] = "\x01\x56\x38",
  [0x1e25] = "\x02\x30\x27", [0x1e26] = "\x01\x56\x37",
  [0x1e27] = "\x01\x56\x35", [0x1e28] = "\x01\x56\x36",
  [0x1e2a] = "\x02\x36\x43", [0x1e2b] = "\x01\x5b\x47",
  [0x1e2c] = "\x01\x60\x2a", [0x1e2d] = "\x0e\x3c\x4e",
  [0x1e2e] = "\x01\x60\x28", [0x1e2f] = "\x01\x60\x29",
  [0x1e30] = "\x02\x3d\x2b", [0x1e31] = "\x0e\x42\x29",
  [0x1e32] = "\x0e\x42\x2a", [0x1e33] = "\x01\x69\x3b",
  [0x1e34] = "\x01\x45\x55", [0x1e36] = "\x02\x21\x60",
  [0x1e37] = "\x0e\x22\x57", [0x1e38] = "\x01\x46\x63",
  [0x1e39] = "\x0e\x22\x54", [0x1e3b] = "\x02\x21\x63",
  [0x1e3c] = "\x0e\x24\x21", [0x1e3e] = "\x01\x46\x66",
  [0x1e3f] = "\x02\x21\x62", [0x1e40] = "\x01\x46\x65",
  [0x1e41] = "\x01\x46\x64", [0x1e42] = "\x01\x4a\x65",
  [0x1e43] = "\x02\x21\x61", [0x1e44] = "\x0e\x22\x58",
  [0x1e45] = "\x0e\x22\x59", [0x1e46] = "\x02\x22\x59",
  [0x1e4a] = "\x02\x22\x5d", [0x1e4b] = "\x02\x22\x5f",
  [0x1e4c] = "\x02\x22\x60", [0x1e4d] = "\x01\x48\x46",
  [0x1e4e] = "\x01\x48\x47", [0x1e4f] = "\x02\x22\x5c",
  [0x1e50] = "\x01\x48\x42", [0x1e51] = "\x0e\x23\x7d",
  [0x1e52] = "\x02\x22\x5a", [0x1e53] = "\x0e\x24\x25",
  [0x1e54] = "\x02\x22\x5e", [0x1e55] = "\x01\x48\x43",
  [0x1e56] = "\x0e\x26\x55", [0x1e57] = "\x01\x48\x3e",
  [0x1e58] = "\x0e\x23\x7c", [0x1e59] = "\x01\x48\x3f",
  [0x1e5a] = "\x0e\x24\x23", [0x1e5b] = "\x01\x48\x45",
  [0x1e5c] = "\x02\x22\x5b", [0x1e5d] = "\x01\x48\x3d",
  [0x1e5e] = "\x01\x4a\x66", [0x1e5f] = "\x01\x48\x40",
  [0x1e60] = "\x01\x48\x41", [0x1e61] = "\x01\x48\x44",
  [0x1e63] = "\x0e\x65\x3c", [0x1e65] = "\x02\x24\x5b",
  [0x1e66] = "\x02\x24\x59", [0x1e67] = "\x02\x24\x4c",
  [0x1e68] = "\x01\x4a\x72", [0x1e69] = "\x02\x24\x53",
  [0x1e6a] = "\x01\x4a\x6d", [0x1e6b] = "\x02\x24\x4d",
  [0x1e6c] = "\x0e\x29\x64", [0x1e6d] = "\x02\x24\x55",
  [0x1e6e] = "\x0e\x26\x50", [0x1e6f] = "\x02\x24\x52",
  [0x1e70] = "\x01\x4a\x70", [0x1e71] = "\x02\x24\x51",
  [0x1e72] = "\x01\x4a\x77", [0x1e73] = "\x02\x24\x5a",
  [0x1e74] = "\x01\x4a\x79", [0x1e75] = "\x0e\x26\x53",
  [0x1e76] = "\x01\x4a\x7b", [0x1e77] = "\x0e\x23\x7e",
  [0x1e78] = "\x02\x24\x4b", [0x1e79] = "\x0e\x26\x57",
  [0x1e7a] = "\x01\x4a\x6e", [0x1e7b] = "\x02\x24\x5c",
  [0x1e7c] = "\x0e\x26\x51", [0x1e7d] = "\x01\x4a\x75",
  [0x1e7e] = "\x01\x4a\x78", [0x1e7f] = "\x0e\x26\x4c",
  [0x1e80] = "\x02\x27\x65", [0x1e81] = "\x01\x4a\x68",
  [0x1e82] = "\x01\x4b\x21", [0x1e83] = "\x01\x4a\x76",
  [0x1e84] = "\x02\x24\x4e", [0x1e85] = "\x01\x4a\x6b",
  [0x1e86] = "\x01\x4a\x7a", [0x1e87] = "\x02\x24\x56",
  [0x1e88] = "\x01\x4a\x69", [0x1e89] = "\x01\x4a\x6a",
  [0x1e8a] = "\x02\x27\x63", [0x1e8b] = "\x02\x24\x4f",
  [0x1e8c] = "\x01\x4a\x71", [0x1e8d] = "\x01\x4a\x7c",
  [0x1e8e] = "\x02\x24\x5d", [0x1e8f] = "\x02\x24\x50",
  [0x1e90] = "\x01\x4a\x6f", [0x1e91] = "\x0e\x26\x4d",
  [0x1e92] = "\x01\x4a\x74", [0x1e93] = "\x02\x27\x4f",
  [0x1e94] = "\x01\x4a\x7d", [0x1e95] = "\x02\x24\x57",
  [0x1e96] = "\x01\x4a\x73", [0x1e97] = "\x0e\x29\x63",
  [0x1e98] = "\x01\x4a\x7e", [0x1e99] = "\x01\x4a\x67",
  [0x1e9a] = "\x02\x24\x54", [0x1e9b] = "\x01\x4a\x6c",
  [0x1e9c] = "\x02\x24\x58", [0x1e9d] = "\x02\x27\x64",
  [0x1e9e] = "\x0e\x26\x4e", [0x1e9f] = "\x0e\x26\x52",
  [0x1ea0] = "\x0e\x26\x5c", [0x1ea1] = "\x0e\x26\x59",
  [0x1ea2] = "\x0e\x26\x56", [0x1eaa] = "\x0e\x26\x5b",
  [0x1eab] = "\x01\x4e\x4d", [0x1eac] = "\x01\x4e\x5d",
  [0x1ead] = "\x02\x27\x56", [0x1eae] = "\x01\x4e\x54",
  [0x1eaf] = "\x0e\x2d\x4e", [0x1eb0] = "\x02\x27\x6b",
  [0x1eb1] = "\x01\x4e\x45", [0x1eb2] = "\x0e\x29\x6b",
  [0x1eb3] = "\x01\x4e\x48", [0x1eb4] = "\x02\x27\x62",
  [0x1eb6] = "\x02\x27\x54", [0x1eb7] = "\x02\x27\x58",
  [0x1eb8] = "\x01\x4e\x50", [0x1eb9] = "\x01\x4e\x52",
  [0x1eba] = "\x02\x27\x5b", [0x1ebb] = "\x01\x4e\x59",
  [0x1ebc] = "\x01\x4e\x4b", [0x1ebd] = "\x01\x4e\x49",
  [0x1ebe] = "\x01\x4e\x4a", [0x1ebf] = "\x01\x4e\x58",
  [0x1ec0] = "\x02\x27\x67", [0x1ec1] = "\x01\x4e\x53",
  [0x1ec2] = "\x02\x27\x5a", [0x1ec3] = "\x02\x27\x5c",
  [0x1ec4] = "\x01\x4e\x51", [0x1ec5] = "\x01\x4e\x56",
  [0x1ec6] = "\x02\x27\x5d", [0x1ec7] = "\x02\x27\x6a",
  [0x1ec8] = "\x0e\x29\x6d", [0x1ec9] = "\x01\x51\x78",
  [0x1eca] = "\x01\x4e\x5c", [0x1ecb] = "\x0e\x62\x61",
  [0x1ecc] = "\x01\x4e\x46", [0x1ecd] = "\x02\x27\x69",
  [0x1ece] = "\x0e\x29\x6c", [0x1ecf] = "\x02\x27\x6d",
  [0x1ed0] = "\x02\x27\x59", [0x1ed1] = "\x02\x27\x6f",
  [0x1ed2] = "\x02\x27\x60", [0x1ed3] = "\x01\x4e\x4f",
  [0x1ed4] = "\x02\x27\x55", [0x1ed5] = "\x01\x4e\x4e",
  [0x1ed6] = "\x01\x4e\x60", [0x1ed7] = "\x01\x4e\x55",
  [0x1ed8] = "\x0e\x29\x6a", [0x1ed9] = "\x02\x27\x53",
  [0x1eda] = "\x02\x2b\x57", [0x1edb] = "\x01\x4e\x5b",
  [0x1edc] = "\x01\x4e\x5f", [0x1edd] = "\x02\x27\x61",
  [0x1ede] = "\x02\x27\x66", [0x1edf] = "\x0e\x29\x65",
  [0x1ee0] = "\x01\x4e\x61", [0x1ee1] = "\x01\x4e\x5a",
  [0x1ee2] = "\x01\x4e\x4c", [0x1ee3] = "\x01\x4e\x42",
  [0x1ee4] = "\x0e\x29\x69", [0x1ee5] = "\x01\x4e\x47",
  [0x1ee6] = "\x0e\x62\x60", [0x1ee7] = "\x02\x27\x57",
  [0x1ee8] = "\x01\x4e\x43", [0x1ee9] = "\x02\x27\x6e",
  [0x1eea] = "\x0e\x29\x67", [0x1eeb] = "\x02\x27\x51",
  [0x1eec] = "\x02\x27\x50", [0x1eed] = "\x02\x27\x5e",
  [0x1eee] = "\x02\x27\x52", [0x1eef] = "\x01\x4e\x5e",
  [0x1ef0] = "\x01\x56\x39", [0x1ef1] = "\x01\x4e\x57",
  [0x1ef2] = "\x02\x27\x5f", [0x1ef3] = "\x01\x4e\x44",
  [0x1ef4] = "\x0e\x2d\x4f", [0x1ef5] = "\x01\x52\x29",
  [0x1ef9] = "\x02\x27\x6c", [0x1eff] = "\x0e\x2d\x46",
  [0x1f00] = "\x02\x2b\x5e", [0x1f01] = "\x02\x2b\x61",
  [0x1f02] = "\x0e\x2d\x50", [0x1f03] = "\x02\x2b\x64",
  [0x1f04] = "\x02\x2b\x59", [0x1f05] = "\x0e\x2d\x48",
  [0x1f06] = "\x0e\x2d\x4a", [0x1f07] = "\x02\x2b\x67",
  [0x1f08] = "\x02\x2b\x6a", [0x1f09] = "\x02\x2b\x6c",
  [0x1f0a] = "\x02\x2b\x56", [0x1f0b] = "\x01\x51\x79",
  [0x1f0c] = "\x01\x51\x7e", [0x1f0d] = "\x02\x30\x2c",
  [0x1f0e] = "\x01\x52\x30", [0x1f0f] = "\x02\x2b\x65",
  [0x1f10] = "\x02\x2b\x6d", [0x1f11] = "\x02\x2b\x5d",
  [0x1f12] = "\x02\x2b\x55", [0x1f13] = "\x0e\x2d\x49",
  [0x1f14] = "\x0e\x2d\x47", [0x1f15] = "\x0e\x2d\x4d",
  [0x1f16] = "\x02\x30\x47", [0x1f17] = "\x01\x52\x23",
  [0x1f18] = "\x02\x2b\x62", [0x1f19] = "\x02\x2b\x5a",
  [0x1f1a] = "\x02\x2b\x5c", [0x1f1b] = "\x01\x52\x28",
  [0x1f1c] = "\x0e\x31\x7a", [0x1f1d] = "\x02\x2b\x5f",
  [0x1f1e] = "\x01\x52\x22", [0x1f1f] = "\x02\x2b\x52",
  [0x1f20] = "\x02\x2b\x68", [0x1f21] = "\x0e\x2d\x4b",
  [0x1f22] = "\x02\x2b\x6b", [0x1f23] = "\x0e\x2d\x45",
  [0x1f24] = "\x0e\x2d\x57", [0x1f25] = "\x01\x51\x7d",
  [0x1f26] = "\x0e\x2d\x53", [0x1f27] = "\x01\x52\x2b",
  [0x1f28] = "\x02\x2b\x4f", [0x1f29] = "\x01\x52\x2d",
  [0x1f2a] = "\x01\x51\x7b", [0x1f2b] = "\x01\x52\x31",
  [0x1f2c] = "\x02\x2b\x69", [0x1f2d] = "\x02\x2b\x51",
  [0x1f2e] = "\x01\x52\x2e", [0x1f2f] = "\x02\x30\x41",
  [0x1f30] = "\x02\x27\x68", [0x1f31] = "\x01\x52\x21",
  [0x1f32] = "\x01\x51\x7a", [0x1f33] = "\x02\x2b\x58",
  [0x1f34] = "\x02\x2b\x50", [0x1f35] = "\x01\x52\x2f",
  [0x1f36] = "\x01\x52\x27", [0x1f37] = "\x02\x2b\x63",
  [0x1f38] = "\x01\x52\x2c", [0x1f39] = "\x01\x52\x2a",
  [0x1f3a] = "\x02\x2b\x5b", [0x1f3b] = "\x01\x52\x24",
  [0x1f3c] = "\x02\x2b\x53", [0x1f3d] = "\x01\x52\x25",
  [0x1f3e] = "\x01\x52\x26", [0x1f3f] = "\x02\x2b\x54",
  [0x1f40] = "\x02\x2b\x66", [0x1f41] = "\x01\x51\x7c",
  [0x1f42] = "\x02\x2b\x60", [0x1f44] = "\x0e\x2d\x55",
  [0x1f45] = "\x0e\x2d\x51", [0x1f56] = "\x0e\x31\x77",
  [0x1f57] = "\x0e\x31\x73", [0x1f58] = "\x02\x30\x2f",
  [0x1f59] = "\x01\x56\x41", [0x1f5a] = "\x01\x56\x46",
  [0x1f5b] = "\x0e\x31\x79", [0x1f5c] = "\x0e\x32\x26",
  [0x1f5d] = "\x0e\x31\x76", [0x1f5e] = "\x02\x30\x38",
  [0x1f5f] = "\x02\x30\x3e", [0x1f60] = "\x02\x30\x3a",
  [0x1f61] = "\x02\x30\x2d", [0x1f62] = "\x02\x30\x30",
  [0x1f63] = "\x02\x30\x29", [0x1f64] = "\x02\x30\x2a",
  [0x1f65] = "\x01\x56\x4d", [0x1f66] = "\x01\x56\x3e",
  [0x1f67] = "\x02\x30\x39", [0x1f68] = "\x02\x30\x42",
  [0x1f69] = "\x01\x56\x48", [0x1f6a] = "\x01\x56\x3a",
  [0x1f6b] = "\x0e\x31\x6f", [0x1f6c] = "\x01\x56\x43",
  [0x1f6d] = "\x02\x30\x31", [0x1f6e] = "\x01\x56\x45",
  [0x1f6f] = "\x02\x30\x32", [0x1f70] = "\x02\x30\x3c",
  [0x1f71] = "\x0e\x32\x22", [0x1f72] = "\x0e\x32\x25",
  [0x1f73] = "\x0e\x31\x72", [0x1f74] = "\x01\x56\x47",
  [0x1f75] = "\x02\x30\x4b", [0x1f76] = "\x02\x30\x2b",
  [0x1f77] = "\x01\x56\x40", [0x1f78] = "\x01\x56\x3f",
  [0x1f79] = "\x01\x56\x4b", [0x1f7a] = "\x02\x30\x28",
  [0x1f7b] = "\x02\x30\x49", [0x1f7c] = "\x02\x30\x3d",
  [0x1f7d] = "\x02\x30\x4a", [0x1f7e] = "\x02\x30\x44",
  [0x1f7f] = "\x02\x30\x36", [0x1f80] = "\x02\x30\x45",
  [0x1f81] = "\x0e\x32\x21", [0x1f82] = "\x02\x30\x3f",
  [0x1f83] = "\x02\x30\x48", [0x1f84] = "\x02\x30\x46",
  [0x1f85] = "\x01\x56\x4c", [0x1f86] = "\x02\x30\x37",
  [0x1f87] = "\x01\x56\x3d", [0x1f88] = "\x01\x56\x3c",
  [0x1f89] = "\x01\x56\x44", [0x1f8a] = "\x01\x56\x4a",
  [0x1f8b] = "\x02\x30\x43", [0x1f8c] = "\x01\x56\x49",
  [0x1f8d] = "\x02\x30\x34", [0x1f8e] = "\x01\x5b\x48",
  [0x1f8f] = "\x0e\x31\x78", [0x1f90] = "\x02\x30\x4c",
  [0x1f91] = "\x02\x30\x33", [0x1f92] = "\x02\x30\x2e",
  [0x1f93] = "\x01\x56\x42", [0x1f94] = "\x01\x56\x4e",
  [0x1f95] = "\x01\x56\x3b", [0x1f96] = "\x0e\x32\x27",
  [0x1f97] = "\x02\x30\x3b", [0x1f98] = "\x02\x30\x40",
  [0x1f99] = "\x0e\x31\x7d", [0x1f9a] = "\x0e\x31\x7b",
  [0x1f9b] = "\x0e\x31\x7c", [0x1fa5] = "\x0e\x65\x3d",
  [0x1faa] = "\x01\x5b\x6a", [0x1fab] = "\x02\x36\x45",
  [0x1fac] = "\x02\x36\x49", [0x1fad] = "\x0e\x36\x6f",
  [0x1fae] = "\x01\x5b\x57", [0x1faf] = "\x01\x5b\x55",
  [0x1fb0] = "\x0e\x36\x7a", [0x1fb1] = "\x0e\x37\x21",
  [0x1fb2] = "\x01\x5b\x4c", [0x1fb3] = "\x02\x36\x47",
  [0x1fb4] = "\x02\x36\x46", [0x1fb5] = "\x01\x5b\x60",
  [0x1fb6] = "\x0e\x3c\x50", [0x1fb7] = "\x02\x36\x4c",
  [0x1fb8] = "\x01\x5b\x5a", [0x1fb9] = "\x0e\x36\x72",
  [0x1fba] = "\x02\x36\x5e", [0x1fbb] = "\x02\x36\x6a",
  [0x1fbc] = "\x01\x5b\x49", [0x1fbd] = "\x02\x36\x5b",
  [0x1fbe] = "\x02\x36\x54", [0x1fbf] = "\x01\x5b\x6c",
  [0x1fc0] = "\x02\x36\x44", [0x1fc1] = "\x0e\x36\x6a",
  [0x1fc2] = "\x02\x36\x60", [0x1fc3] = "\x0e\x36\x6b",
  [0x1fc4] = "\x01\x5b\x69", [0x1fc5] = "\x01\x5b\x5d",
  [0x1fc6] = "\x01\x5b\x68", [0x1fc7] = "\x01\x5b\x53",
  [0x1fc8] = "\x02\x36\x50", [0x1fc9] = "\x02\x36\x62",
  [0x1fca] = "\x02\x36\x5a", [0x1fcb] = "\x01\x5b\x54",
  [0x1fcc] = "\x01\x5b\x4e", [0x1fcd] = "\x02\x36\x68",
  [0x1fce] = "\x0e\x36\x6c", [0x1fcf] = "\x02\x36\x61",
  [0x1fd0] = "\x02\x36\x63", [0x1fd1] = "\x01\x5b\x56",
  [0x1fd2] = "\x01\x5b\x5e", [0x1fd3] = "\x02\x36\x65",
  [0x1fd4] = "\x02\x36\x4e", [0x1fd5] = "\x02\x36\x5f",
  [0x1fd6] = "\x02\x36\x53", [0x1fd7] = "\x02\x36\x67",
  [0x1fd8] = "\x01\x5b\x63", [0x1fd9] = "\x01\x5b\x4b",
  [0x1fda] = "\x01\x5b\x61", [0x1fdb] = "\x02\x36\x58",
  [0x1fdc] = "\x02\x36\x56", [0x1fdd] = "\x02\x36\x57",
  [0x1fde] = "\x01\x5b\x58", [0x1fdf] = "\x02\x36\x52",
  [0x1fe0] = "\x02\x36\x51", [0x1fe1] = "\x01\x5b\x4d",
  [0x1fe2] = "\x02\x36\x4b", [0x1fe3] = "\x02\x36\x69",
  [0x1fe4] = "\x01\x5b\x4f", [0x1fe5] = "\x02\x36\x55",
  [0x1fe6] = "\x01\x5b\x6d", [0x1fe7] = "\x0e\x36\x74",
  [0x1fe8] = "\x01\x5b\x67", [0x1fe9] = "\x02\x36\x4a",
  [0x1fea] = "\x01\x5b\x64", [0x1feb] = "\x01\x5b\x62",
  [0x1fec] = "\x01\x5b\x6b", [0x1fed] = "\x02\x36\x5c",
  [0x1fee] = "\x01\x5b\x66", [0x1fef] = "\x02\x30\x35",
  [0x1ff0] = "\x02\x36\x5d", [0x1ff1] = "\x01\x5b\x65",
  [0x1ff2] = "\x02\x36\x64", [0x1ff3] = "\x01\x5b\x4a",
  [0x1ff4] = "\x02\x36\x59", [0x1ff5] = "\x01\x5b\x5c",
  [0x1ff6] = "\x02\x36\x4d", [0x1ff7] = "\x01\x5b\x5b",
  [0x1ff8] = "\x0e\x36\x7c", [0x1ff9] = "\x01\x5b\x59",
  [0x1ffa] = "\x01\x5b\x51", [0x1ffb] = "\x01\x5b\x50",
  [0x1ffc] = "\x02\x3d\x2c", [0x1ffd] = "\x02\x36\x66",
  [0x1ffe] = "\x0e\x3c\x5d", [0x1fff] = "\x0e\x37\x28",
  [0x2000] = "\x02\x36\x4f", [0x2001] = "\x0e\x37\x26",
  [0x2002] = "\x0e\x37\x23", [0x2003] = "\x02\x3d\x41",
  [0x2004] = "\x0e\x36\x70", [0x2005] = "\x01\x5b\x52",
  [0x2006] = "\x0e\x36\x77", [0x2007] = "\x0e\x37\x24",
  [0x2008] = "\x0e\x36\x76", [0x2009] = "\x0e\x37\x25",
  [0x200a] = "\x0e\x36\x79", [0x200c] = "\x0e\x36\x7d",
  [0x200f] = "\x0e\x65\x3e", [0x2012] = "\x0e\x37\x29",
  [0x2017] = "\x0e\x37\x27", [0x2018] = "\x0e\x3c\x54",
  [0x2019] = "\x01\x60\x47", [0x201a] = "\x01\x5b\x5f",
  [0x201b] = "\x01\x60\x35", [0x201c] = "\x02\x3d\x3b",
  [0x201d] = "\x01\x60\x43", [0x201e] = "\x0e\x3c\x52",
  [0x201f] = "\x02\x3d\x2f", [0x2020] = "\x01\x60\x32",
  [0x2021] = "\x01\x60\x2e", [0x2022] = "\x02\x3d\x4d",
  [0x2023] = "\x01\x60\x34", [0x2024] = "\x01\x60\x38",
  [0x2025] = "\x01\x60\x33", [0x2026] = "\x01\x60\x3c",
  [0x2027] = "\x02\x3d\x51", [0x2028] = "\x02\x3d\x48",
  [0x2029] = "\x0e\x3c\x6b", [0x202a] = "\x0e\x3c\x6d",
  [0x202b] = "\x02\x3d\x36", [0x202c] = "\x01\x60\x41",
  [0x202d] = "\x01\x60\x3b", [0x202e] = "\x02\x3d\x42",
  [0x202f] = "\x01\x60\x2b", [0x2030] = "\x02\x3d\x4e",
  [0x2031] = "\x02\x3d\x47", [0x2032] = "\x01\x60\x2f",
  [0x2033] = "\x02\x3d\x3c", [0x2034] = "\x01\x60\x3e",
  [0x2035] = "\x02\x3d\x59", [0x2036] = "\x02\x3d\x5a",
  [0x2038] = "\x01\x60\x2c", [0x2039] = "\x02\x3d\x4c",
  [0x203a] = "\x01\x60\x40", [0x203b] = "\x02\x3d\x40",
  [0x203c] = "\x02\x3d\x32", [0x203d] = "\x02\x3d\x33",
  [0x203e] = "\x01\x60\x44", [0x203f] = "\x02\x3d\x37",
  [0x2040] = "\x02\x3d\x3e", [0x2041] = "\x02\x3d\x38",
  [0x2042] = "\x0e\x3c\x5a", [0x2043] = "\x01\x60\x42",
  [0x2044] = "\x01\x60\x4a", [0x2045] = "\x02\x3d\x34",
  [0x2046] = "\x02\x3d\x2d", [0x2047] = "\x02\x3d\x2e",
  [0x2048] = "\x0e\x3c\x56", [0x2049] = "\x02\x3d\x30",
  [0x204a] = "\x01\x60\x31", [0x204b] = "\x02\x3d\x3d",
  [0x204c] = "\x0e\x3c\x6e", [0x204d] = "\x01\x60\x3f",
  [0x204e] = "\x01\x60\x48", [0x204f] = "\x0e\x3c\x58",
  [0x2050] = "\x0e\x3c\x69", [0x2051] = "\x02\x3d\x3f",
  [0x2052] = "\x02\x3d\x57", [0x2053] = "\x02\x3d\x4f",
  [0x2054] = "\x01\x60\x2d", [0x2055] = "\x02\x3d\x55",
  [0x2056] = "\x01\x60\x39", [0x2057] = "\x0e\x3c\x66",
  [0x2058] = "\x01\x60\x37", [0x2059] = "\x0e\x3c\x64",
  [0x205a] = "\x02\x3d\x5b", [0x205b] = "\x01\x60\x36",
  [0x205c] = "\x02\x3d\x45", [0x205d] = "\x02\x3d\x39",
  [0x205e] = "\x02\x3d\x43", [0x205f] = "\x01\x60\x4d",
  [0x2060] = "\x02\x3d\x49", [0x2061] = "\x02\x3d\x46",
  [0x2062] = "\x02\x3d\x35", [0x2063] = "\x01\x60\x49",
  [0x2064] = "\x02\x3d\x53", [0x2065] = "\x02\x3d\x50",
  [0x2066] = "\x02\x3d\x58", [0x2067] = "\x01\x60\x30",
  [0x2068] = "\x02\x3d\x44", [0x2069] = "\x01\x60\x4c",
  [0x206a] = "\x0e\x3c\x5b", [0x206b] = "\x02\x3d\x4b",
  [0x206c] = "\x0e\x42\x3a", [0x206d] = "\x0e\x3c\x60",
  [0x206e] = "\x01\x60\x3a", [0x206f] = "\x01\x60\x3d",
  [0x2070] = "\x0e\x3c\x5c", [0x2071] = "\x02\x3d\x4a",
  [0x2072] = "\x01\x60\x4b", [0x2073] = "\x02\x3d\x3a",
  [0x2074] = "\x02\x36\x48", [0x2075] = "\x0e\x3c\x51",
  [0x2076] = "\x0e\x3c\x6c", [0x2077] = "\x02\x3d\x54",
  [0x2078] = "\x02\x3d\x52", [0x2079] = "\x02\x3d\x56",
  [0x207b] = "\x0e\x3c\x62", [0x207c] = "\x0e\x65\x42",
  [0x207e] = "\x0e\x3c\x63", [0x2080] = "\x0e\x3c\x68",
  [0x2084] = "\x0e\x65\x40", [0x2086] = "\x0e\x63\x3d",
  [0x2088] = "\x02\x3d\x31", [0x2089] = "\x01\x60\x46",
  [0x208a] = "\x0e\x65\x3f", [0x208b] = "\x0e\x65\x41",
  [0x208d] = "\x02\x44\x47", [0x208e] = "\x02\x44\x46",
  [0x208f] = "\x02\x44\x2c", [0x2090] = "\x01\x64\x63",
  [0x2091] = "\x0e\x42\x30", [0x2092] = "\x02\x44\x45",
  [0x2093] = "\x02\x44\x2f", [0x2094] = "\x02\x44\x30",
  [0x2096] = "\x01\x64\x6d", [0x2097] = "\x02\x44\x4e",
  [0x2098] = "\x01\x64\x68", [0x2099] = "\x02\x44\x44",
  [0x209a] = "\x0e\x42\x33", [0x209b] = "\x02\x44\x29",
  [0x209c] = "\x01\x64\x6e", [0x209d] = "\x01\x64\x64",
  [0x209e] = "\x02\x44\x38", [0x209f] = "\x02\x44\x2e",
  [0x20a0] = "\x02\x44\x31", [0x20a1] = "\x02\x44\x49",
  [0x20a2] = "\x01\x64\x5e", [0x20a3] = "\x02\x44\x50",
  [0x20a4] = "\x02\x44\x48", [0x20a5] = "\x01\x64\x67",
  [0x20a6] = "\x02\x44\x3d", [0x20a7] = "\x01\x64\x72",
  [0x20a8] = "\x0e\x42\x2e", [0x20a9] = "\x0e\x42\x36",
  [0x20aa] = "\x01\x64\x71", [0x20ab] = "\x01\x64\x6b",
  [0x20ac] = "\x0e\x42\x40", [0x20ae] = "\x02\x44\x4f",
  [0x20af] = "\x01\x64\x5f", [0x20b0] = "\x02\x44\x3b",
  [0x20b1] = "\x02\x44\x32", [0x20b2] = "\x02\x44\x3f",
  [0x20b3] = "\x02\x44\x4b", [0x20b4] = "\x01\x64\x73",
  [0x20b5] = "\x0e\x42\x39", [0x20b6] = "\x01\x64\x61",
  [0x20b7] = "\x02\x44\x3a", [0x20b8] = "\x0e\x42\x2d",
  [0x20b9] = "\x02\x44\x33", [0x20ba] = "\x01\x64\x6a",
  [0x20bb] = "\x0e\x42\x31", [0x20bc] = "\x01\x64\x69",
  [0x20bd] = "\x02\x44\x36", [0x20be] = "\x02\x44\x40",
  [0x20bf] = "\x02\x44\x4a", [0x20c0] = "\x02\x44\x2d",
  [0x20c1] = "\x02\x44\x37", [0x20c2] = "\x01\x64\x62",
  [0x20c3] = "\x02\x44\x41", [0x20c4] = "\x01\x64\x6f",
  [0x20c5] = "\x01\x64\x66", [0x20c6] = "\x02\x44\x34",
  [0x20c7] = "\x01\x64\x65", [0x20c8] = "\x02\x44\x2b",
  [0x20c9] = "\x02\x44\x39", [0x20ca] = "\x02\x44\x4d",
  [0x20cb] = "\x01\x60\x45", [0x20cc] = "\x01\x69\x57",
  [0x20cd] = "\x02\x44\x3c", [0x20ce] = "\x02\x4b\x34",
  [0x20cf] = "\x02\x44\x3e", [0x20d0] = "\x02\x44\x4c",
  [0x20d1] = "\x01\x64\x6c", [0x20d2] = "\x02\x44\x35",
  [0x20d3] = "\x01\x64\x60", [0x20d4] = "\x01\x64\x70",
  [0x20d5] = "\x01\x6d\x5a", [0x20d6] = "\x02\x44\x2a",
  [0x20d8] = "\x02\x44\x43", [0x20d9] = "\x0e\x42\x44",
  [0x20da] = "\x0e\x42\x3e", [0x20db] = "\x0e\x42\x47",
  [0x20dc] = "\x02\x44\x42", [0x20dd] = "\x0e\x42\x3d",
  [0x20e3] = "\x0e\x42\x45", [0x20e6] = "\x0e\x42\x3f",
  [0x20e8] = "\x0e\x42\x3b", [0x20e9] = "\x0e\x42\x46",
  [0x20eb] = "\x02\x4b\x50", [0x20ec] = "\x01\x69\x54",
  [0x20ed] = "\x02\x4b\x45", [0x20ee] = "\x02\x4b\x4a",
  [0x20ef] = "\x01\x69\x49", [0x20f0] = "\x0e\x47\x56",
  [0x20f1] = "\x02\x4b\x36", [0x20f2] = "\x01\x69\x56",
  [0x20f3] = "\x0e\x47\x57", [0x20f4] = "\x01\x69\x40",
  [0x20f5] = "\x02\x4b\x35", [0x20f6] = "\x02\x4b\x56",
  [0x20f7] = "\x01\x69\x58", [0x20f8] = "\x02\x4b\x39",
  [0x20f9] = "\x02\x4b\x49", [0x20fa] = "\x0e\x47\x65",
  [0x20fb] = "\x02\x4b\x3b", [0x20fc] = "\x02\x4b\x59",
  [0x20fd] = "\x02\x4b\x55", [0x20fe] = "\x01\x69\x3e",
  [0x20ff] = "\x01\x69\x48", [0x2100] = "\x02\x51\x5b",
  [0x2101] = "\x01\x69\x55", [0x2102] = "\x01\x69\x46",
  [0x2103] = "\x02\x4b\x37", [0x2104] = "\x0e\x47\x63",
  [0x2105] = "\x02\x4b\x54", [0x2106] = "\x01\x69\x4a",
  [0x2107] = "\x02\x4b\x51", [0x2108] = "\x02\x4b\x5e",
  [0x2109] = "\x02\x4b\x3d", [0x210a] = "\x02\x4b\x46",
  [0x210b] = "\x0e\x4c\x78", [0x210c] = "\x0e\x47\x5b",
  [0x210d] = "\x02\x4b\x5c", [0x210e] = "\x02\x4b\x52",
  [0x210f] = "\x01\x69\x45", [0x2110] = "\x0e\x4c\x6a",
  [0x2111] = "\x0e\x47\x64", [0x2112] = "\x02\x4b\x44",
  [0x2113] = "\x01\x69\x3f", [0x2114] = "\x01\x69\x3d",
  [0x2115] = "\x01\x69\x4f", [0x2117] = "\x0e\x47\x5f",
  [0x2118] = "\x02\x4b\x42", [0x2119] = "\x02\x4b\x3f",
  [0x211a] = "\x02\x4b\x40", [0x211b] = "\x0e\x47\x5a",
  [0x211c] = "\x02\x4b\x58", [0x211d] = "\x0e\x47\x5c",
  [0x211e] = "\x02\x4b\x5d", [0x211f] = "\x02\x4b\x5b",
  [0x2120] = "\x01\x69\x43", [0x2121] = "\x02\x4b\x5f",
  [0x2122] = "\x01\x69\x47", [0x2123] = "\x01\x69\x4e",
  [0x2125] = "\x02\x4b\x38", [0x2126] = "\x02\x51\x43",
  [0x2127] = "\x02\x4b\x41", [0x2128] = "\x0e\x47\x5e",
  [0x2129] = "\x01\x69\x41", [0x212a] = "\x01\x69\x53",
  [0x212b] = "\x01\x69\x50", [0x212c] = "\x01\x69\x44",
  [0x212d] = "\x02\x4b\x4b", [0x212e] = "\x02\x4b\x3c",
  [0x212f] = "\x01\x69\x51", [0x2130] = "\x02\x4b\x4d",
  [0x2131] = "\x01\x69\x4b", [0x2132] = "\x01\x69\x4d",
  [0x2133] = "\x01\x69\x3c", [0x2134] = "\x0e\x47\x5d",
  [0x2135] = "\x02\x4b\x4f", [0x2136] = "\x02\x4b\x47",
  [0x2137] = "\x02\x4b\x3a", [0x2138] = "\x01\x69\x4c",
  [0x2139] = "\x02\x4b\x57", [0x213a] = "\x02\x4b\x5a",
  [0x213b] = "\x02\x4b\x43", [0x213c] = "\x02\x4b\x4e",
  [0x213d] = "\x0e\x4c\x74", [0x213e] = "\x01\x69\x42",
  [0x213f] = "\x01\x6d\x49", [0x2140] = "\x02\x4b\x4c",
  [0x2141] = "\x02\x51\x42", [0x2142] = "\x0e\x47\x62",
  [0x2143] = "\x02\x4b\x53", [0x2144] = "\x0e\x47\x61",
  [0x2148] = "\x0e\x65\x44", [0x2149] = "\x0e\x65\x45",
  [0x214a] = "\x0e\x47\x66", [0x214c] = "\x0e\x65\x43",
  [0x214e] = "\x02\x4b\x3e", [0x214f] = "\x02\x51\x4c",
  [0x2150] = "\x02\x51\x56", [0x2151] = "\x01\x6d\x4c",
  [0x2152] = "\x02\x51\x55", [0x2153] = "\x02\x51\x61",
  [0x2154] = "\x01\x6d\x4e", [0x2155] = "\x02\x51\x53",
  [0x2157] = "\x02\x51\x57", [0x2158] = "\x01\x6d\x59",
  [0x2159] = "\x0e\x4c\x7d", [0x215a] = "\x02\x51\x4e",
  [0x215b] = "\x01\x6d\x51", [0x215c] = "\x0e\x4c\x73",
  [0x215d] = "\x02\x51\x5a", [0x215e] = "\x02\x57\x7b",
  [0x215f] = "\x01\x6d\x5d", [0x2160] = "\x01\x6d\x5c",
  [0x2161] = "\x02\x51\x5c", [0x2162] = "\x02\x51\x4b",
  [0x2163] = "\x02\x51\x66", [0x2164] = "\x01\x6d\x57",
  [0x2165] = "\x0e\x4c\x6b", [0x2166] = "\x01\x6d\x4d",
  [0x2167] = "\x02\x51\x5f", [0x2169] = "\x02\x51\x63",
  [0x216a] = "\x02\x51\x68", [0x216b] = "\x02\x51\x5d",
  [0x216c] = "\x02\x51\x51", [0x216d] = "\x01\x6d\x50",
  [0x216e] = "\x01\x6d\x53", [0x216f] = "\x01\x6d\x5b",
  [0x2170] = "\x01\x6d\x56", [0x2171] = "\x0e\x4c\x75",
  [0x2172] = "\x02\x51\x54", [0x2173] = "\x02\x4b\x48",
  [0x2174] = "\x0e\x4c\x7e", [0x2175] = "\x0e\x4c\x6c",
  [0x2176] = "\x02\x51\x50", [0x2177] = "\x02\x51\x67",
  [0x2178] = "\x01\x6d\x52", [0x2179] = "\x0e\x4c\x79",
  [0x217a] = "\x01\x6d\x55", [0x217b] = "\x02\x51\x69",
  [0x217c] = "\x01\x6d\x4a", [0x217d] = "\x02\x51\x5e",
  [0x217e] = "\x02\x51\x44", [0x217f] = "\x02\x51\x64",
  [0x2180] = "\x01\x74\x2a", [0x2181] = "\x0e\x4c\x7a",
  [0x2182] = "\x02\x51\x52", [0x2184] = "\x01\x6d\x4b",
  [0x2185] = "\x02\x51\x4d", [0x2186] = "\x01\x6d\x4f",
  [0x2187] = "\x02\x51\x45", [0x2188] = "\x01\x69\x52",
  [0x2189] = "\x02\x51\x49", [0x218a] = "\x0e\x63\x63",
  [0x218b] = "\x02\x51\x62", [0x218c] = "\x02\x51\x4a",
  [0x218d] = "\x02\x51\x48", [0x218e] = "\x01\x6d\x54",
  [0x218f] = "\x0e\x4c\x7b", [0x2190] = "\x02\x51\x60",
  [0x2191] = "\x0e\x4c\x77", [0x2192] = "\x02\x51\x47",
  [0x2193] = "\x02\x51\x59", [0x2194] = "\x02\x51\x58",
  [0x2195] = "\x02\x51\x65", [0x2196] = "\x02\x51\x4f",
  [0x2197] = "\x01\x6d\x58", [0x2199] = "\x0e\x65\x47",
  [0x219d] = "\x0e\x65\x46", [0x219e] = "\x02\x57\x7e",
  [0x219f] = "\x0e\x51\x56", [0x21a0] = "\x01\x71\x33",
  [0x21a1] = "\x01\x71\x29", [0x21a2] = "\x02\x58\x2c",
  [0x21a3] = "\x02\x57\x76", [0x21a4] = "\x01\x71\x2b",
  [0x21a5] = "\x02\x58\x24", [0x21a6] = "\x01\x71\x32",
  [0x21a7] = "\x01\x71\x2d", [0x21a8] = "\x02\x58\x22",
  [0x21a9] = "\x02\x5d\x3b", [0x21aa] = "\x02\x58\x28",
  [0x21ab] = "\x02\x58\x2e", [0x21ac] = "\x02\x58\x27",
  [0x21ad] = "\x02\x57\x74", [0x21ae] = "\x02\x58\x25",
  [0x21af] = "\x02\x58\x30", [0x21b0] = "\x02\x58\x32",
  [0x21b1] = "\x01\x71\x28", [0x21b2] = "\x02\x58\x31",
  [0x21b3] = "\x01\x71\x2e", [0x21b4] = "\x01\x71\x34",
  [0x21b5] = "\x0e\x51\x54", [0x21b6] = "\x01\x71\x31",
  [0x21b7] = "\x0e\x51\x58", [0x21b8] = "\x02\x58\x2b",
  [0x21b9] = "\x01\x71\x30", [0x21ba] = "\x02\x58\x26",
  [0x21bb] = "\x0e\x51\x4d", [0x21bc] = "\x02\x57\x78",
  [0x21bd] = "\x02\x57\x7d", [0x21be] = "\x0e\x51\x50",
  [0x21bf] = "\x02\x58\x2a", [0x21c0] = "\x01\x71\x2f",
  [0x21c1] = "\x01\x71\x2c", [0x21c2] = "\x01\x71\x27",
  [0x21c3] = "\x01\x71\x2a", [0x21c4] = "\x02\x57\x7c",
  [0x21c6] = "\x02\x51\x46", [0x21c7] = "\x02\x57\x77",
  [0x21c8] = "\x02\x57\x7a", [0x21c9] = "\x02\x58\x2d",
  [0x21ca] = "\x02\x58\x21", [0x21cb] = "\x02\x57\x75",
  [0x21cc] = "\x02\x5d\x3a", [0x21cd] = "\x02\x58\x2f",
  [0x21ce] = "\x02\x57\x79", [0x21cf] = "\x02\x58\x29",
  [0x21d0] = "\x0e\x4c\x71", [0x21d3] = "\x0e\x51\x55",
  [0x21d4] = "\x02\x5d\x3d", [0x21d5] = "\x01\x74\x2e",
  [0x21d6] = "\x0e\x55\x30", [0x21d7] = "\x0e\x55\x2f",
  [0x21d8] = "\x01\x74\x22", [0x21d9] = "\x0e\x55\x35",
  [0x21da] = "\x0e\x55\x36", [0x21db] = "\x01\x74\x26",
  [0x21dc] = "\x02\x5d\x3f", [0x21dd] = "\x02\x5d\x45",
  [0x21de] = "\x02\x5d\x43", [0x21df] = "\x01\x74\x24",
  [0x21e0] = "\x01\x74\x25", [0x21e1] = "\x01\x74\x2c",
  [0x21e2] = "\x02\x5d\x46", [0x21e3] = "\x02\x5d\x3e",
  [0x21e4] = "\x01\x74\x27", [0x21e5] = "\x0e\x55\x31",
  [0x21e6] = "\x02\x5d\x42", [0x21e7] = "\x02\x5d\x41",
  [0x21e8] = "\x02\x5d\x47", [0x21e9] = "\x01\x74\x2d",
  [0x21ea] = "\x0e\x55\x37", [0x21eb] = "\x01\x74\x28",
  [0x21ec] = "\x01\x74\x2b", [0x21ed] = "\x02\x5d\x40",
  [0x21ee] = "\x01\x74\x2f", [0x21ef] = "\x01\x74\x29",
  [0x21f0] = "\x01\x74\x30", [0x21f1] = "\x01\x74\x23",
  [0x21f2] = "\x02\x5d\x44", [0x21f3] = "\x0e\x5a\x55",
  [0x21f4] = "\x02\x5d\x3c", [0x21f5] = "\x0e\x51\x59",
  [0x21f6] = "\x0e\x55\x39", [0x21f7] = "\x02\x62\x25",
  [0x21f8] = "\x0e\x65\x48", [0x21fa] = "\x01\x76\x5d",
  [0x21fb] = "\x02\x62\x22", [0x21fc] = "\x02\x62\x24",
  [0x21fd] = "\x0e\x58\x28", [0x21fe] = "\x01\x76\x5b",
  [0x21ff] = "\x02\x61\x7e", [0x2200] = "\x02\x62\x21",
  [0x2201] = "\x02\x61\x7a", [0x2202] = "\x0e\x58\x2a",
  [0x2203] = "\x0e\x58\x27", [0x2204] = "\x02\x58\x23",
  [0x2205] = "\x02\x61\x7b", [0x2206] = "\x01\x76\x5c",
  [0x2207] = "\x02\x61\x77", [0x2208] = "\x0e\x58\x26",
  [0x2209] = "\x01\x76\x59", [0x220a] = "\x02\x62\x26",
  [0x220b] = "\x01\x76\x5a", [0x220c] = "\x02\x61\x78",
  [0x220d] = "\x02\x61\x79", [0x220e] = "\x02\x61\x7d",
  [0x220f] = "\x01\x76\x5f", [0x2210] = "\x0e\x58\x29",
  [0x2211] = "\x01\x76\x5e", [0x2213] = "\x0e\x58\x2b",
  [0x2214] = "\x02\x61\x7c", [0x2215] = "\x01\x78\x45",
  [0x2216] = "\x02\x65\x6a", [0x2217] = "\x02\x65\x70",
  [0x2218] = "\x01\x78\x46", [0x2219] = "\x02\x65\x67",
  [0x221a] = "\x01\x78\x43", [0x221b] = "\x01\x78\x40",
  [0x221c] = "\x02\x65\x72", [0x221d] = "\x01\x78\x44",
  [0x221e] = "\x0e\x5a\x52", [0x221f] = "\x01\x78\x41",
  [0x2220] = "\x02\x65\x69", [0x2221] = "\x02\x65\x6c",
  [0x2222] = "\x02\x65\x6d", [0x2223] = "\x02\x65\x6e",
  [0x2224] = "\x02\x65\x71", [0x2225] = "\x0e\x5a\x54",
  [0x2226] = "\x02\x62\x23", [0x2227] = "\x02\x65\x68",
  [0x2228] = "\x01\x78\x42", [0x2229] = "\x02\x65\x6f",
  [0x222a] = "\x02\x69\x34", [0x222b] = "\x02\x65\x6b",
  [0x222c] = "\x0e\x5a\x53", [0x222d] = "\x0e\x65\x4a",
  [0x222e] = "\x0e\x65\x49", [0x222f] = "\x02\x69\x2b",
  [0x2230] = "\x01\x79\x75", [0x2231] = "\x02\x69\x2e",
  [0x2232] = "\x01\x79\x76", [0x2233] = "\x02\x69\x37",
  [0x2234] = "\x02\x69\x2d", [0x2235] = "\x02\x69\x2a",
  [0x2236] = "\x0e\x5c\x51", [0x2237] = "\x02\x69\x2c",
  [0x2238] = "\x02\x69\x30", [0x2239] = "\x02\x69\x33",
  [0x223a] = "\x02\x69\x32", [0x223b] = "\x02\x69\x36",
  [0x223c] = "\x02\x69\x29", [0x223d] = "\x0e\x5c\x4f",
  [0x223e] = "\x01\x79\x74", [0x223f] = "\x02\x69\x31",
  [0x2240] = "\x02\x69\x35", [0x2241] = "\x02\x69\x38",
  [0x2242] = "\x02\x69\x2f", [0x2243] = "\x02\x6b\x61",
  [0x2244] = "\x02\x6b\x62", [0x2245] = "\x02\x6b\x66",
  [0x2246] = "\x02\x6b\x67", [0x2247] = "\x0e\x5e\x3c",
  [0x2248] = "\x02\x6b\x64", [0x2249] = "\x02\x6b\x65",
  [0x224a] = "\x02\x6b\x63", [0x224b] = "\x0e\x5e\x3a",
  [0x224c] = "\x01\x7a\x74", [0x224d] = "\x0e\x5e\x3b",
  [0x2251] = "\x01\x7b\x64", [0x2252] = "\x02\x6d\x59",
  [0x2253] = "\x0e\x60\x36", [0x2254] = "\x0e\x5f\x52",
  [0x2255] = "\x02\x6d\x56", [0x2256] = "\x02\x6d\x57",
  [0x2257] = "\x02\x6d\x58", [0x2258] = "\x01\x7b\x65",
  [0x2259] = "\x0e\x60\x37", [0x225a] = "\x02\x6f\x34",
  [0x225b] = "\x02\x6f\x33", [0x225c] = "\x0e\x65\x4b",
  [0x225d] = "\x02\x70\x3d", [0x225e] = "\x01\x7c\x60",
  [0x225f] = "\x02\x70\x3b", [0x2260] = "\x02\x70\x3e",
  [0x2261] = "\x02\x70\x3c", [0x2262] = "\x02\x71\x25",
  [0x2263] = "\x01\x7d\x24", [0x2264] = "\x01\x7d\x32",
  [0x2265] = "\x02\x71\x71", [0x2266] = "\x02\x71\x5b",
  [0x2267] = "\x0e\x61\x79", [0x2268] = "\x02\x71\x70",
  [0x2269] = "\x02\x72\x3e", [0x226a] = "\x02\x72\x3f",
  [0x226b] = "\x01\x45\x56", [0x226e] = "\x0e\x24\x28",
  [0x226f] = "\x0e\x24\x26", [0x2270] = "\x01\x48\x48",
  [0x2271] = "\x02\x22\x61", [0x2272] = "\x0e\x24\x27",
  [0x2273] = "\x0e\x24\x2b", [0x2274] = "\x02\x24\x5e",
  [0x2275] = "\x0e\x26\x62", [0x2276] = "\x01\x4b\x22",
  [0x2277] = "\x0e\x26\x5f", [0x2278] = "\x01\x4b\x25",
  [0x2279] = "\x0e\x26\x5e", [0x227a] = "\x02\x24\x5f",
  [0x227b] = "\x0e\x26\x63", [0x227c] = "\x01\x4b\x23",
  [0x227d] = "\x01\x4b\x24", [0x227e] = "\x0e\x26\x60",
  [0x2281] = "\x0e\x29\x77", [0x2282] = "\x02\x27\x78",
  [0x2283] = "\x02\x27\x7a", [0x2284] = "\x02\x27\x75",
  [0x2285] = "\x02\x27\x72", [0x2286] = "\x02\x27\x74",
  [0x2287] = "\x0e\x29\x74", [0x2288] = "\x0e\x29\x70",
  [0x2289] = "\x0e\x29\x75", [0x228a] = "\x01\x4e\x65",
  [0x228b] = "\x0e\x29\x6f", [0x228c] = "\x0e\x29\x79",
  [0x228d] = "\x0e\x29\x76", [0x228e] = "\x01\x4e\x63",
  [0x228f] = "\x0e\x29\x72", [0x2290] = "\x0e\x29\x71",
  [0x2291] = "\x02\x27\x76", [0x2292] = "\x01\x4e\x64",
  [0x2293] = "\x02\x27\x73", [0x2294] = "\x02\x27\x70",
  [0x2295] = "\x01\x4e\x62", [0x2296] = "\x02\x27\x77",
  [0x2298] = "\x02\x27\x71", [0x2299] = "\x01\x4e\x66",
  [0x229a] = "\x02\x27\x79", [0x229f] = "\x02\x2b\x6f",
  [0x22a0] = "\x0e\x2d\x5e", [0x22a1] = "\x02\x2b\x73",
  [0x22a2] = "\x0e\x2d\x5c", [0x22a3] = "\x0e\x2d\x5a",
  [0x22a4] = "\x01\x52\x3a", [0x22a5] = "\x0e\x2d\x58",
  [0x22a6] = "\x0e\x62\x6a", [0x22a7] = "\x0e\x2d\x65",
  [0x22a8] = "\x0e\x2d\x62", [0x22a9] = "\x02\x2b\x76",
  [0x22aa] = "\x0e\x2d\x5f", [0x22ab] = "\x01\x52\x32",
  [0x22ac] = "\x01\x52\x35", [0x22ad] = "\x01\x52\x37",
  [0x22ae] = "\x01\x52\x39", [0x22af] = "\x01\x52\x36",
  [0x22b0] = "\x02\x2b\x72", [0x22b1] = "\x02\x2b\x71",
  [0x22b2] = "\x0e\x2d\x64", [0x22b3] = "\x01\x52\x34",
  [0x22b4] = "\x02\x2b\x74", [0x22b5] = "\x02\x2b\x75",
  [0x22b6] = "\x0e\x2d\x63", [0x22b7] = "\x02\x2b\x6e",
  [0x22b8] = "\x01\x52\x38", [0x22b9] = "\x0e\x2d\x68",
  [0x22ba] = "\x01\x52\x33", [0x22bb] = "\x0e\x2d\x5d",
  [0x22be] = "\x02\x2b\x70", [0x22bf] = "\x0e\x65\x4d",
  [0x22c4] = "\x0e\x32\x28", [0x22c5] = "\x02\x30\x5a",
  [0x22c6] = "\x02\x30\x5b", [0x22c7] = "\x02\x30\x5c",
  [0x22c8] = "\x01\x56\x53", [0x22c9] = "\x0e\x62\x79",
  [0x22ca] = "\x01\x56\x4f", [0x22cb] = "\x02\x30\x51",
  [0x22cc] = "\x0e\x32\x2a", [0x22cd] = "\x02\x30\x59",
  [0x22ce] = "\x02\x30\x5e", [0x22cf] = "\x01\x56\x54",
  [0x22d0] = "\x0e\x32\x2b", [0x22d1] = "\x02\x30\x4f",
  [0x22d2] = "\x02\x30\x55", [0x22d3] = "\x02\x30\x4e",
  [0x22d4] = "\x02\x30\x58", [0x22d5] = "\x0e\x32\x31",
  [0x22d6] = "\x0e\x32\x2f", [0x22d7] = "\x02\x30\x54",
  [0x22d8] = "\x01\x56\x50", [0x22d9] = "\x01\x56\x52",
  [0x22da] = "\x02\x30\x5d", [0x22db] = "\x0e\x32\x29",
  [0x22dc] = "\x02\x30\x4d", [0x22dd] = "\x02\x30\x50",
  [0x22de] = "\x02\x30\x56", [0x22df] = "\x0e\x32\x2d",
  [0x22e0] = "\x02\x30\x57", [0x22e1] = "\x02\x30\x5f",
  [0x22e2] = "\x02\x30\x53", [0x22e3] = "\x0e\x32\x2c",
  [0x22e4] = "\x01\x56\x51", [0x22e5] = "\x0e\x65\x4f",
  [0x22ea] = "\x0e\x65\x4e", [0x22ee] = "\x0e\x32\x30",
  [0x22ef] = "\x01\x5b\x72", [0x22f0] = "\x02\x36\x71",
  [0x22f1] = "\x0e\x37\x30", [0x22f2] = "\x0e\x37\x32",
  [0x22f3] = "\x02\x36\x73", [0x22f4] = "\x02\x36\x6f",
  [0x22f5] = "\x0e\x37\x2f", [0x22f6] = "\x02\x36\x7b",
  [0x22f7] = "\x02\x36\x6d", [0x22f8] = "\x02\x36\x7a",
  [0x22f9] = "\x01\x5b\x6e", [0x22fa] = "\x02\x36\x6b",
  [0x22fb] = "\x02\x3d\x5f", [0x22fc] = "\x02\x36\x75",
  [0x22fd] = "\x01\x5b\x71", [0x22fe] = "\x0e\x37\x35",
  [0x22ff] = "\x02\x36\x76", [0x2300] = "\x02\x36\x79",
  [0x2301] = "\x0e\x37\x2e", [0x2302] = "\x02\x36\x7d",
  [0x2303] = "\x0e\x37\x2c", [0x2304] = "\x02\x36\x72",
  [0x2306] = "\x02\x36\x77", [0x2307] = "\x0e\x37\x2d",
  [0x2308] = "\x0e\x37\x31", [0x2309] = "\x01\x5b\x6f",
  [0x230a] = "\x01\x5b\x70", [0x230b] = "\x02\x36\x7c",
  [0x230c] = "\x02\x36\x70", [0x230d] = "\x02\x36\x6c",
  [0x230e] = "\x02\x36\x7e", [0x230f] = "\x0e\x37\x33",
  [0x2310] = "\x02\x36\x74", [0x2311] = "\x0e\x65\x50",
  [0x2312] = "\x0e\x65\x51", [0x2313] = "\x02\x36\x78",
  [0x2317] = "\x02\x36\x6e", [0x2319] = "\x01\x60\x4e",
  [0x231a] = "\x01\x60\x4f", [0x231b] = "\x02\x3d\x69",
  [0x231c] = "\x01\x60\x55", [0x231d] = "\x0e\x3c\x74",
  [0x231e] = "\x02\x3d\x5d", [0x231f] = "\x02\x3d\x66",
  [0x2320] = "\x02\x3d\x5c", [0x2321] = "\x01\x60\x52",
  [0x2322] = "\x02\x3d\x64", [0x2323] = "\x02\x3d\x62",
  [0x2324] = "\x0e\x3c\x7d", [0x2325] = "\x02\x3d\x63",
  [0x2326] = "\x01\x60\x50", [0x2327] = "\x0e\x3c\x7c",
  [0x2328] = "\x02\x3d\x67", [0x232a] = "\x0e\x3c\x6f",
  [0x232b] = "\x0e\x3c\x7a", [0x232c] = "\x0e\x3c\x72",
  [0x232d] = "\x0e\x3d\x21", [0x232e] = "\x02\x3d\x60",
  [0x232f] = "\x02\x3d\x5e", [0x2330] = "\x01\x60\x51",
  [0x2331] = "\x02\x3d\x61", [0x2332] = "\x02\x3d\x65",
  [0x2333] = "\x0e\x3c\x7b", [0x2334] = "\x0e\x3c\x79",
  [0x2335] = "\x0e\x3c\x71", [0x2336] = "\x01\x60\x53",
  [0x2337] = "\x0e\x3c\x73", [0x2338] = "\x0e\x3c\x77",
  [0x2339] = "\x0e\x65\x53", [0x233a] = "\x02\x3d\x68",
  [0x233b] = "\x0e\x65\x54", [0x233d] = "\x0e\x65\x55",
  [0x233f] = "\x0e\x65\x52", [0x2341] = "\x02\x44\x56",
  [0x2342] = "\x02\x44\x5d", [0x2343] = "\x02\x44\x5f",
  [0x2344] = "\x02\x44\x65", [0x2345] = "\x0e\x42\x57",
  [0x2346] = "\x01\x65\x22", [0x2347] = "\x02\x44\x51",
  [0x2348] = "\x0e\x42\x4c", [0x2349] = "\x01\x64\x78",
  [0x234a] = "\x0e\x42\x4e", [0x234b] = "\x02\x44\x60",
  [0x234c] = "\x01\x64\x7d", [0x234d] = "\x02\x44\x66",
  [0x234e] = "\x01\x64\x74", [0x234f] = "\x0e\x42\x51",
  [0x2350] = "\x02\x44\x63", [0x2351] = "\x0e\x42\x58",
  [0x2352] = "\x02\x44\x53", [0x2353] = "\x02\x44\x64",
  [0x2354] = "\x02\x44\x52", [0x2356] = "\x01\x65\x24",
  [0x2357] = "\x0e\x42\x52", [0x2358] = "\x02\x44\x5e",
  [0x2359] = "\x01\x64\x75", [0x235a] = "\x02\x44\x67",
  [0x235b] = "\x0e\x48\x3c", [0x235c] = "\x01\x64\x7a",
  [0x235d] = "\x02\x44\x57", [0x235e] = "\x01\x65\x21",
  [0x235f] = "\x02\x44\x62", [0x2360] = "\x02\x44\x55",
  [0x2361] = "\x02\x44\x5c", [0x2362] = "\x02\x44\x58",
  [0x2363] = "\x02\x44\x54", [0x2364] = "\x01\x64\x77",
  [0x2365] = "\x01\x64\x7e", [0x2366] = "\x01\x64\x7c",
  [0x2367] = "\x01\x64\x79", [0x2368] = "\x01\x65\x23",
  [0x2369] = "\x01\x64\x76", [0x236a] = "\x02\x44\x5b",
  [0x236b] = "\x0e\x42\x4f", [0x236c] = "\x01\x64\x7b",
  [0x236d] = "\x0e\x42\x59", [0x236e] = "\x01\x60\x54",
  [0x236f] = "\x0e\x42\x49", [0x2370] = "\x02\x44\x61",
  [0x2371] = "\x0e\x65\x59", [0x2372] = "\x02\x44\x59",
  [0x2374] = "\x0e\x42\x53", [0x2375] = "\x0e\x65\x57",
  [0x2376] = "\x0e\x65\x58", [0x2377] = "\x0e\x65\x56",
  [0x2378] = "\x02\x44\x5a", [0x237b] = "\x02\x4b\x67",
  [0x237c] = "\x0e\x47\x6f", [0x237d] = "\x01\x69\x5b",
  [0x237e] = "\x0e\x47\x6e", [0x237f] = "\x0e\x47\x6c",
  [0x2380] = "\x02\x4b\x63", [0x2381] = "\x02\x4b\x69",
  [0x2382] = "\x02\x4b\x65", [0x2383] = "\x0e\x47\x70",
  [0x2384] = "\x01\x69\x5d", [0x2385] = "\x02\x4b\x64",
  [0x2386] = "\x02\x4b\x68", [0x2387] = "\x02\x4b\x60",
  [0x2388] = "\x0e\x47\x72", [0x2389] = "\x02\x4b\x62",
  [0x238a] = "\x01\x69\x5c", [0x238b] = "\x0e\x47\x6a",
  [0x238c] = "\x0e\x47\x6d", [0x238d] = "\x0e\x47\x6b",
  [0x238e] = "\x0e\x47\x68", [0x238f] = "\x02\x4b\x66",
  [0x2390] = "\x02\x4b\x61", [0x2391] = "\x0e\x47\x67",
  [0x2392] = "\x01\x69\x5e", [0x2393] = "\x0e\x65\x5b",
  [0x2394] = "\x01\x69\x59", [0x2396] = "\x0e\x65\x5a",
  [0x2397] = "\x02\x4b\x6a", [0x2399] = "\x01\x69\x5a",
  [0x239a] = "\x02\x51\x6f", [0x239b] = "\x02\x51\x6c",
  [0x239c] = "\x02\x51\x78", [0x239d] = "\x02\x51\x72",
  [0x239e] = "\x02\x51\x74", [0x239f] = "\x01\x6d\x5e",
  [0x23a0] = "\x02\x51\x6e", [0x23a1] = "\x02\x51\x76",
  [0x23a2] = "\x0e\x4d\x2a", [0x23a3] = "\x0e\x4d\x24",
  [0x23a4] = "\x02\x51\x75", [0x23a5] = "\x02\x51\x73",
  [0x23a6] = "\x0e\x4d\x29", [0x23a7] = "\x02\x51\x79",
  [0x23a8] = "\x01\x6d\x61", [0x23a9] = "\x02\x51\x70",
  [0x23aa] = "\x02\x51\x77", [0x23ab] = "\x0e\x4d\x28",
  [0x23ac] = "\x01\x6d\x5f", [0x23ad] = "\x0e\x4d\x25",
  [0x23ae] = "\x0e\x4d\x22", [0x23af] = "\x02\x51\x6b",
  [0x23b0] = "\x02\x51\x6d", [0x23b1] = "\x01\x6d\x60",
  [0x23b2] = "\x02\x51\x6a", [0x23b3] = "\x02\x51\x7a",
  [0x23b4] = "\x0e\x65\x5c", [0x23b5] = "\x02\x51\x71",
  [0x23b6] = "\x0e\x51\x62", [0x23b7] = "\x0e\x51\x5a",
  [0x23b8] = "\x02\x58\x36", [0x23b9] = "\x01\x71\x3b",
  [0x23ba] = "\x0e\x51\x60", [0x23bb] = "\x0e\x51\x5c",
  [0x23bc] = "\x02\x58\x41", [0x23bd] = "\x02\x58\x3f",
  [0x23be] = "\x01\x71\x35", [0x23bf] = "\x02\x58\x35",
  [0x23c0] = "\x02\x58\x38", [0x23c1] = "\x02\x58\x39",
  [0x23c2] = "\x02\x58\x34", [0x23c3] = "\x01\x71\x3f",
  [0x23c4] = "\x01\x71\x40", [0x23c5] = "\x02\x58\x33",
  [0x23c6] = "\x02\x58\x42", [0x23c7] = "\x02\x58\x3d",
  [0x23c8] = "\x01\x71\x39", [0x23c9] = "\x01\x71\x36",
  [0x23ca] = "\x02\x58\x3c", [0x23cb] = "\x02\x58\x3a",
  [0x23cc] = "\x0e\x51\x63", [0x23cd] = "\x0e\x63\x74",
  [0x23ce] = "\x01\x71\x3c", [0x23cf] = "\x02\x58\x3e",
  [0x23d0] = "\x01\x71\x37", [0x23d1] = "\x0e\x51\x5e",
  [0x23d2] = "\x01\x71\x38", [0x23d3] = "\x0e\x51\x66",
  [0x23d4] = "\x02\x58\x3b", [0x23d5] = "\x01\x71\x3a",
  [0x23d6] = "\x02\x58\x37", [0x23d8] = "\x02\x58\x40",
  [0x23d9] = "\x01\x71\x3d", [0x23da] = "\x02\x58\x43",
  [0x23db] = "\x02\x58\x44", [0x23dc] = "\x01\x71\x3e",
  [0x23dd] = "\x0e\x65\x5d", [0x23de] = "\x0e\x65\x5e",
  [0x23df] = "\x01\x74\x32", [0x23e0] = "\x01\x74\x39",
  [0x23e1] = "\x02\x5d\x48", [0x23e2] = "\x02\x5d\x4e",
  [0x23e3] = "\x0e\x55\x3c", [0x23e4] = "\x02\x5d\x4c",
  [0x23e5] = "\x01\x74\x35", [0x23e6] = "\x01\x74\x34",
  [0x23e7] = "\x01\x74\x31", [0x23e8] = "\x02\x5d\x4a",
  [0x23e9] = "\x0e\x55\x3e", [0x23ea] = "\x0e\x55\x43",
  [0x23eb] = "\x0e\x55\x40", [0x23ec] = "\x01\x74\x37",
  [0x23ed] = "\x01\x74\x36", [0x23ee] = "\x01\x74\x33",
  [0x23ef] = "\x0e\x55\x41", [0x23f0] = "\x02\x5d\x4d",
  [0x23f1] = "\x02\x5d\x49", [0x23f2] = "\x02\x5d\x4b",
  [0x23f3] = "\x0e\x55\x42", [0x23f4] = "\x01\x74\x38",
  [0x23f8] = "\x01\x76\x63", [0x23f9] = "\x02\x62\x29",
  [0x23fa] = "\x0e\x58\x2d", [0x23fb] = "\x01\x76\x60",
  [0x23fc] = "\x01\x76\x61", [0x23fd] = "\x02\x62\x2b",
  [0x23fe] = "\x01\x76\x62", [0x23ff] = "\x02\x62\x28",
  [0x2400] = "\x0e\x58\x2e", [0x2401] = "\x02\x62\x27",
  [0x2402] = "\x02\x65\x76", [0x2403] = "\x02\x62\x2a",
  [0x2404] = "\x0e\x5a\x56", [0x2405] = "\x02\x65\x77",
  [0x2406] = "\x01\x78\x47", [0x2407] = "\x02\x65\x75",
  [0x2408] = "\x0e\x5a\x57", [0x240a] = "\x02\x65\x74",
  [0x240b] = "\x0e\x5c\x59", [0x240c] = "\x02\x65\x73",
  [0x240d] = "\x01\x78\x48", [0x240e] = "\x0e\x65\x5f",
  [0x240f] = "\x0e\x5c\x56", [0x2410] = "\x01\x79\x77",
  [0x2411] = "\x0e\x5c\x58", [0x2412] = "\x0e\x5c\x55",
  [0x2413] = "\x02\x69\x39", [0x2414] = "\x02\x69\x3a",
  [0x2415] = "\x0e\x5a\x58", [0x2416] = "\x0e\x5c\x53",
  [0x2417] = "\x0e\x5c\x57", [0x2418] = "\x0e\x65\x61",
  [0x2419] = "\x02\x6b\x6a", [0x241a] = "\x02\x6b\x69",
  [0x241b] = "\x01\x7a\x75", [0x241c] = "\x0e\x5f\x53",
  [0x241d] = "\x02\x6b\x68", [0x241e] = "\x02\x6d\x5a",
  [0x241f] = "\x02\x6d\x5b", [0x2420] = "\x0e\x5f\x54",
  [0x2421] = "\x0e\x60\x39", [0x2422] = "\x02\x6f\x35",
  [0x2423] = "\x02\x70\x3f", [0x2424] = "\x0e\x61\x43",
  [0x2425] = "\x0e\x61\x42", [0x2426] = "\x02\x71\x26",
  [0x2427] = "\x02\x72\x2c", [0x2428] = "\x01\x7d\x46",
  [0x2429] = "\x02\x72\x40", [0x242a] = "\x01\x45\x57",
  [0x242c] = "\x01\x4e\x67", [0x242d] = "\x01\x4e\x68",
  [0x242e] = "\x0e\x2d\x69", [0x2430] = "\x01\x52\x3b",
  [0x2432] = "\x0e\x3d\x22", [0x2435] = "\x01\x74\x3a",
  [0x2436] = "\x01\x45\x58", [0x2438] = "\x01\x4e\x69",
  [0x2439] = "\x01\x56\x55", [0x243a] = "\x01\x65\x25",
  [0x243b] = "\x01\x45\x59", [0x243c] = "\x0e\x2d\x6b",
  [0x243d] = "\x01\x5b\x73", [0x243e] = "\x01\x69\x5f",
  [0x243f] = "\x02\x21\x42", [0x2440] = "\x0e\x29\x7c",
  [0x2441] = "\x02\x2b\x77", [0x2442] = "\x02\x30\x60",
  [0x2444] = "\x02\x4b\x6b", [0x2446] = "\x01\x74\x3b",
  [0x2447] = "\x01\x45\x5a", [0x2448] = "\x01\x4e\x6a",
  [0x2449] = "\x02\x2b\x78", [0x244a] = "\x02\x2b\x79",
  [0x244b] = "\x02\x3d\x6a", [0x244c] = "\x01\x60\x56",
  [0x244e] = "\x0e\x42\x5d", [0x244f] = "\x02\x44\x68",
  [0x2450] = "\x0e\x42\x5c", [0x2451] = "\x0e\x42\x5b",
  [0x2452] = "\x01\x65\x26", [0x2453] = "\x02\x4b\x6c",
  [0x2455] = "\x0e\x4d\x2e", [0x2456] = "\x01\x6d\x62",
  [0x2457] = "\x0e\x4d\x2d", [0x2458] = "\x01\x78\x49",
  [0x2459] = "\x01\x45\x5b", [0x245a] = "\x02\x3d\x6b",
  [0x245b] = "\x01\x45\x5c", [0x245d] = "\x01\x48\x4a",
  [0x245e] = "\x02\x22\x62", [0x245f] = "\x01\x48\x49",
  [0x2460] = "\x01\x4b\x28", [0x2461] = "\x01\x4b\x27",
  [0x2462] = "\x01\x4b\x26", [0x2463] = "\x02\x24\x60",
  [0x2464] = "\x0e\x26\x64", [0x2465] = "\x0e\x2a\x21",
  [0x2466] = "\x0e\x2a\x22", [0x2467] = "\x01\x4e\x6b",
  [0x2468] = "\x0e\x2a\x23", [0x2469] = "\x01\x4e\x6c",
  [0x246a] = "\x02\x27\x7b", [0x246c] = "\x02\x2b\x7a",
  [0x246d] = "\x0e\x2d\x6d", [0x246e] = "\x02\x2b\x7d",
  [0x246f] = "\x01\x52\x3d", [0x2470] = "\x02\x2b\x7b",
  [0x2472] = "\x01\x52\x3c", [0x2473] = "\x02\x2b\x7c",
  [0x2474] = "\x01\x52\x3e", [0x2476] = "\x02\x30\x63",
  [0x2477] = "\x02\x30\x62", [0x2478] = "\x02\x30\x61",
  [0x2479] = "\x01\x56\x56", [0x247a] = "\x0e\x32\x36",
  [0x247b] = "\x02\x37\x22", [0x247c] = "\x02\x37\x23",
  [0x247d] = "\x01\x5b\x74", [0x247e] = "\x02\x37\x21",
  [0x247f] = "\x02\x37\x24", [0x2480] = "\x01\x60\x58",
  [0x2481] = "\x01\x5b\x75", [0x2482] = "\x0e\x3d\x24",
  [0x2483] = "\x0e\x3d\x23", [0x2484] = "\x01\x60\x57",
  [0x2485] = "\x02\x3d\x6f", [0x2486] = "\x02\x3d\x6e",
  [0x2487] = "\x0e\x3d\x25", [0x2488] = "\x02\x3d\x6c",
  [0x2489] = "\x02\x3d\x6d", [0x248b] = "\x02\x3d\x70",
  [0x248c] = "\x02\x44\x6a", [0x248d] = "\x02\x44\x69",
  [0x248e] = "\x02\x44\x6d", [0x2490] = "\x02\x44\x6c",
  [0x2491] = "\x02\x44\x6b", [0x2492] = "\x01\x69\x60",
  [0x2493] = "\x02\x4b\x6f", [0x2494] = "\x0e\x47\x75",
  [0x2495] = "\x02\x4b\x6e", [0x2496] = "\x01\x69\x61",
  [0x2497] = "\x02\x4b\x6d", [0x2498] = "\x02\x51\x7b",
  [0x2499] = "\x0e\x4d\x2f", [0x249a] = "\x02\x51\x7c",
  [0x249b] = "\x01\x6d\x63", [0x249d] = "\x02\x58\x45",
  [0x249e] = "\x02\x58\x46", [0x24a1] = "\x02\x65\x7c",
  [0x24a2] = "\x01\x78\x4a", [0x24a3] = "\x02\x65\x7b",
  [0x24a4] = "\x02\x65\x7a", [0x24a5] = "\x02\x65\x78",
  [0x24a6] = "\x02\x65\x79", [0x24a7] = "\x01\x7a\x76",
  [0x24a8] = "\x02\x69\x3b", [0x24a9] = "\x02\x6d\x5c",
  [0x24aa] = "\x02\x71\x27", [0x24ab] = "\x0e\x61\x7b",
  [0x24ac] = "\x01\x45\x5d", [0x24ae] = "\x02\x21\x64",
  [0x24af] = "\x01\x46\x67", [0x24b0] = "\x02\x21\x65",
  [0x24b1] = "\x0e\x24\x2c", [0x24b2] = "\x0e\x24\x2d",
  [0x24b4] = "\x02\x22\x63", [0x24b5] = "\x02\x22\x64",
  [0x24b6] = "\x0e\x26\x6b", [0x24b9] = "\x0e\x26\x69",
  [0x24ba] = "\x02\x24\x66", [0x24bb] = "\x0e\x26\x66",
  [0x24bc] = "\x0e\x26\x67", [0x24bd] = "\x02\x24\x62",
  [0x24be] = "\x0e\x26\x6a", [0x24bf] = "\x02\x24\x61",
  [0x24c0] = "\x01\x4e\x6d", [0x24c1] = "\x02\x24\x65",
  [0x24c2] = "\x01\x4b\x2a", [0x24c3] = "\x02\x24\x63",
  [0x24c4] = "\x01\x4b\x29", [0x24c5] = "\x02\x24\x67",
  [0x24c6] = "\x02\x24\x64", [0x24c7] = "\x0e\x26\x68",
  [0x24c9] = "\x02\x28\x21", [0x24ca] = "\x02\x2b\x7e",
  [0x24cb] = "\x02\x27\x7d", [0x24cc] = "\x02\x28\x26",
  [0x24cd] = "\x0e\x2a\x26", [0x24ce] = "\x01\x4e\x6e",
  [0x24cf] = "\x0e\x2a\x2a", [0x24d0] = "\x01\x4e\x71",
  [0x24d1] = "\x02\x28\x27", [0x24d2] = "\x02\x28\x23",
  [0x24d3] = "\x0e\x2a\x27", [0x24d4] = "\x02\x28\x24",
  [0x24d6] = "\x02\x27\x7c", [0x24d7] = "\x01\x4e\x70",
  [0x24d8] = "\x02\x27\x7e", [0x24d9] = "\x01\x4e\x6f",
  [0x24da] = "\x02\x28\x25", [0x24db] = "\x0e\x2a\x28",
  [0x24dc] = "\x02\x28\x22", [0x24df] = "\x02\x2c\x24",
  [0x24e0] = "\x01\x52\x40", [0x24e1] = "\x01\x52\x41",
  [0x24e2] = "\x0e\x2d\x75", [0x24e3] = "\x02\x2c\x27",
  [0x24e4] = "\x02\x2c\x21", [0x24e5] = "\x0e\x2d\x74",
  [0x24e6] = "\x02\x2c\x26", [0x24e7] = "\x0e\x2d\x70",
  [0x24e8] = "\x02\x2c\x22", [0x24e9] = "\x01\x52\x3f",
  [0x24ea] = "\x02\x2c\x25", [0x24eb] = "\x02\x2c\x23",
  [0x24ec] = "\x0e\x2d\x73", [0x24ed] = "\x0e\x2d\x71",
  [0x24f3] = "\x02\x30\x69", [0x24f4] = "\x02\x30\x66",
  [0x24f5] = "\x0e\x32\x38", [0x24f6] = "\x02\x30\x68",
  [0x24f7] = "\x01\x56\x5b", [0x24f8] = "\x01\x56\x5a",
  [0x24f9] = "\x01\x56\x58", [0x24fa] = "\x02\x30\x65",
  [0x24fb] = "\x02\x30\x6a", [0x24fc] = "\x01\x56\x57",
  [0x24fd] = "\x01\x56\x59", [0x24fe] = "\x02\x30\x67",
  [0x24ff] = "\x02\x37\x2c", [0x2500] = "\x02\x30\x64",
  [0x2501] = "\x02\x30\x6b", [0x2502] = "\x0e\x32\x39",
  [0x2506] = "\x0e\x3d\x26", [0x2507] = "\x02\x37\x27",
  [0x2508] = "\x02\x37\x2b", [0x250a] = "\x02\x37\x2a",
  [0x250b] = "\x02\x3d\x72", [0x250c] = "\x02\x3d\x7d",
  [0x250e] = "\x0e\x37\x3b", [0x250f] = "\x02\x37\x2d",
  [0x2510] = "\x0e\x37\x3a", [0x2511] = "\x02\x37\x28",
  [0x2512] = "\x02\x3d\x71", [0x2513] = "\x01\x5b\x79",
  [0x2516] = "\x01\x5b\x78", [0x2517] = "\x02\x37\x26",
  [0x2518] = "\x02\x37\x29", [0x2519] = "\x01\x5b\x7a",
  [0x251a] = "\x0e\x37\x39", [0x251b] = "\x01\x5b\x77",
  [0x251c] = "\x01\x5b\x76", [0x251d] = "\x02\x37\x25",
  [0x251e] = "\x02\x37\x2e", [0x2522] = "\x02\x3d\x74",
  [0x2523] = "\x02\x3d\x7b", [0x2524] = "\x0e\x3d\x27",
  [0x2525] = "\x01\x60\x5a", [0x2526] = "\x02\x3d\x7a",
  [0x2527] = "\x02\x3d\x77", [0x2528] = "\x0e\x3d\x2e",
  [0x2529] = "\x01\x60\x5c", [0x252a] = "\x0e\x3d\x29",
  [0x252b] = "\x0e\x3d\x2b", [0x252c] = "\x0e\x3d\x2a",
  [0x252d] = "\x02\x3d\x79", [0x252e] = "\x0e\x42\x61",
  [0x252f] = "\x0e\x3d\x2d", [0x2530] = "\x02\x3d\x73",
  [0x2531] = "\x02\x3d\x75", [0x2532] = "\x02\x3d\x78",
  [0x2533] = "\x02\x3d\x76", [0x2534] = "\x01\x60\x5b",
  [0x2535] = "\x02\x3d\x7c", [0x2536] = "\x01\x60\x59",
  [0x2537] = "\x01\x65\x27", [0x2538] = "\x0e\x3d\x28",
  [0x253a] = "\x02\x44\x71", [0x253b] = "\x02\x44\x70",
  [0x253c] = "\x02\x44\x6e", [0x253e] = "\x01\x65\x2a",
  [0x253f] = "\x01\x65\x29", [0x2540] = "\x02\x44\x72",
  [0x2541] = "\x0e\x42\x5f", [0x2542] = "\x02\x44\x6f",
  [0x2543] = "\x02\x4b\x70", [0x2544] = "\x01\x69\x62",
  [0x2545] = "\x01\x65\x28", [0x2546] = "\x0e\x42\x62",
  [0x2547] = "\x0e\x65\x62", [0x2548] = "\x0e\x65\x63",
  [0x2549] = "\x02\x44\x74", [0x254a] = "\x02\x44\x73",
  [0x254b] = "\x0e\x4d\x30", [0x254c] = "\x02\x4b\x73",
  [0x254d] = "\x02\x4b\x71", [0x254e] = "\x01\x6d\x64",
  [0x254f] = "\x0e\x47\x79", [0x2550] = "\x01\x69\x63",
  [0x2551] = "\x02\x4b\x72", [0x2552] = "\x02\x51\x7e",
  [0x2553] = "\x0e\x47\x78", [0x2554] = "\x0e\x47\x7a",
  [0x2555] = "\x0e\x47\x77", [0x2557] = "\x01\x6d\x65",
  [0x2558] = "\x02\x51\x7d", [0x2559] = "\x02\x52\x28",
  [0x255a] = "\x02\x52\x27", [0x255b] = "\x02\x52\x25",
  [0x255d] = "\x02\x52\x24", [0x255e] = "\x02\x52\x21",
  [0x255f] = "\x02\x52\x22", [0x2560] = "\x02\x52\x23",
  [0x2561] = "\x02\x52\x26", [0x2562] = "\x02\x52\x29",
  [0x2565] = "\x02\x58\x4b", [0x2566] = "\x02\x58\x48",
  [0x2567] = "\x02\x58\x49", [0x2568] = "\x01\x71\x41",
  [0x2569] = "\x02\x58\x47", [0x256a] = "\x02\x58\x4d",
  [0x256b] = "\x02\x58\x4c", [0x256c] = "\x02\x58\x4a",
  [0x256e] = "\x02\x5d\x50", [0x256f] = "\x02\x5d\x51",
  [0x2570] = "\x01\x74\x3c", [0x2571] = "\x0e\x55\x49",
  [0x2572] = "\x01\x74\x3d", [0x2573] = "\x02\x5d\x4f",
  [0x2575] = "\x01\x76\x65", [0x2576] = "\x02\x62\x2c",
  [0x2577] = "\x01\x76\x64", [0x2578] = "\x01\x78\x4b",
  [0x257a] = "\x01\x78\x4c", [0x257b] = "\x01\x79\x78",
  [0x257c] = "\x02\x69\x3d", [0x257d] = "\x02\x69\x3c",
  [0x257e] = "\x02\x6b\x6b", [0x257f] = "\x02\x6d\x5d",
  [0x2580] = "\x01\x7b\x66", [0x2581] = "\x02\x6f\x37",
  [0x2582] = "\x02\x6f\x36", [0x2583] = "\x02\x6f\x38",
  [0x2584] = "\x01\x46\x68", [0x2585] = "\x02\x2c\x28",
  [0x2586] = "\x01\x56\x5c", [0x2587] = "\x01\x5b\x7b",
  [0x2588] = "\x02\x37\x2f", [0x2589] = "\x01\x46\x69",
  [0x258a] = "\x02\x21\x66", [0x258b] = "\x01\x45\x5e",
  [0x258c] = "\x0e\x22\x5a", [0x258d] = "\x0e\x21\x6b",
  [0x258e] = "\x02\x22\x65", [0x258f] = "\x0e\x24\x2f",
  [0x2592] = "\x02\x24\x6c", [0x2593] = "\x02\x24\x6a",
  [0x2594] = "\x02\x24\x6b", [0x2595] = "\x02\x24\x68",
  [0x2596] = "\x01\x4b\x2b", [0x2597] = "\x02\x24\x69",
  [0x2598] = "\x0e\x26\x6e", [0x259c] = "\x0e\x2a\x33",
  [0x259d] = "\x02\x28\x2f", [0x259e] = "\x0e\x2a\x2d",
  [0x259f] = "\x01\x4e\x74", [0x25a0] = "\x02\x28\x2d",
  [0x25a1] = "\x02\x28\x29", [0x25a2] = "\x02\x28\x2c",
  [0x25a3] = "\x0e\x2a\x2c", [0x25a4] = "\x02\x28\x28",
  [0x25a5] = "\x01\x4e\x76", [0x25a6] = "\x02\x28\x2b",
  [0x25a7] = "\x0e\x2a\x2b", [0x25a8] = "\x01\x4e\x73",
  [0x25a9] = "\x01\x4e\x72", [0x25aa] = "\x0e\x2a\x32",
  [0x25ab] = "\x01\x4e\x75", [0x25ac] = "\x02\x28\x2e",
  [0x25ad] = "\x02\x28\x2a", [0x25af] = "\x0e\x2a\x2f",
  [0x25b2] = "\x01\x52\x45", [0x25b3] = "\x01\x52\x48",
  [0x25b4] = "\x02\x2c\x30", [0x25b5] = "\x02\x2c\x2f",
  [0x25b6] = "\x02\x2c\x2e", [0x25b7] = "\x01\x52\x42",
  [0x25b8] = "\x02\x2c\x37", [0x25b9] = "\x02\x2c\x2d",
  [0x25bb] = "\x01\x52\x44", [0x25bc] = "\x02\x30\x70",
  [0x25bd] = "\x0e\x2d\x78", [0x25be] = "\x02\x2c\x34",
  [0x25bf] = "\x02\x2c\x32", [0x25c0] = "\x01\x52\x47",
  [0x25c1] = "\x0e\x2d\x7b", [0x25c2] = "\x02\x2c\x2a",
  [0x25c3] = "\x02\x2c\x35", [0x25c4] = "\x0e\x2d\x77",
  [0x25c5] = "\x02\x2c\x2c", [0x25c6] = "\x02\x2c\x36",
  [0x25c7] = "\x02\x2c\x33", [0x25c8] = "\x02\x2c\x2b",
  [0x25c9] = "\x0e\x2d\x7a", [0x25ca] = "\x01\x52\x43",
  [0x25cb] = "\x02\x2c\x38", [0x25cc] = "\x02\x2c\x29",
  [0x25cd] = "\x01\x52\x46", [0x25ce] = "\x0e\x2d\x7e",
  [0x25cf] = "\x0e\x2d\x79", [0x25d0] = "\x0e\x2d\x7c",
  [0x25d2] = "\x02\x30\x75", [0x25d3] = "\x02\x30\x6c",
  [0x25d4] = "\x02\x30\x77", [0x25d5] = "\x0e\x32\x3e",
  [0x25d6] = "\x02\x30\x6f", [0x25d7] = "\x02\x30\x7a",
  [0x25d8] = "\x02\x30\x7b", [0x25d9] = "\x02\x30\x6d",
  [0x25da] = "\x02\x30\x79", [0x25db] = "\x02\x30\x76",
  [0x25dc] = "\x02\x30\x74", [0x25dd] = "\x02\x30\x78",
  [0x25de] = "\x01\x56\x62", [0x25df] = "\x0e\x32\x40",
  [0x25e0] = "\x01\x56\x60", [0x25e1] = "\x0e\x32\x47",
  [0x25e2] = "\x0e\x32\x3c", [0x25e3] = "\x02\x30\x72",
  [0x25e4] = "\x0e\x32\x46", [0x25e5] = "\x02\x30\x6e",
  [0x25e6] = "\x0e\x32\x41", [0x25e7] = "\x02\x30\x71",
  [0x25e8] = "\x02\x30\x7c", [0x25e9] = "\x02\x30\x73",
  [0x25ea] = "\x01\x56\x61", [0x25eb] = "\x02\x2c\x31",
  [0x25ec] = "\x0e\x32\x3d", [0x25ed] = "\x01\x56\x5d",
  [0x25ee] = "\x01\x56\x5f", [0x25ef] = "\x0e\x65\x64",
  [0x25f3] = "\x0e\x37\x3d", [0x25f4] = "\x02\x37\x3d",
  [0x25f5] = "\x02\x37\x32", [0x25f6] = "\x02\x37\x30",
  [0x25f7] = "\x0e\x3d\x38", [0x25f8] = "\x02\x37\x31",
  [0x25f9] = "\x0e\x32\x3f", [0x25fa] = "\x02\x37\x38",
  [0x25fb] = "\x0e\x37\x40", [0x25fc] = "\x02\x37\x39",
  [0x25fd] = "\x02\x37\x35", [0x25fe] = "\x01\x5c\x22",
  [0x25ff] = "\x02\x37\x3a", [0x2600] = "\x02\x37\x37",
  [0x2601] = "\x02\x37\x34", [0x2602] = "\x0e\x37\x3c",
  [0x2603] = "\x01\x5b\x7e", [0x2604] = "\x02\x37\x33",
  [0x2605] = "\x01\x5b\x7c", [0x2606] = "\x01\x5c\x21",
  [0x2607] = "\x02\x37\x36", [0x2608] = "\x02\x37\x3e",
  [0x2609] = "\x01\x56\x5e", [0x260a] = "\x01\x5b\x7d",
  [0x260b] = "\x02\x37\x3c", [0x260c] = "\x02\x37\x3b",
  [0x260d] = "\x01\x5c\x23", [0x2612] = "\x0e\x65\x65",
  [0x2614] = "\x0e\x3d\x36", [0x2615] = "\x0e\x3d\x3a",
  [0x2616] = "\x02\x3e\x24", [0x2617] = "\x0e\x3d\x2f",
  [0x2618] = "\x0e\x3d\x3b", [0x2619] = "\x0e\x3d\x32",
  [0x261a] = "\x02\x3e\x25", [0x261b] = "\x01\x60\x66",
  [0x261c] = "\x0e\x3d\x39", [0x261d] = "\x02\x3e\x2b",
  [0x261e] = "\x0e\x42\x6d", [0x261f] = "\x0e\x3d\x35",
  [0x2620] = "\x02\x3e\x2d", [0x2621] = "\x02\x3e\x26",
  [0x2622] = "\x01\x60\x60", [0x2623] = "\x02\x3e\x2a",
  [0x2624] = "\x02\x3e\x29", [0x2625] = "\x01\x60\x61",
  [0x2626] = "\x01\x60\x67", [0x2627] = "\x0e\x42\x64",
  [0x2628] = "\x01\x60\x68", [0x2629] = "\x02\x3e\x2c",
  [0x262a] = "\x01\x60\x5e", [0x262b] = "\x02\x3e\x23",
  [0x262c] = "\x02\x3e\x21", [0x262d] = "\x02\x3e\x27",
  [0x262e] = "\x02\x3d\x7e", [0x262f] = "\x01\x60\x65",
  [0x2630] = "\x02\x3e\x22", [0x2631] = "\x02\x3e\x28",
  [0x2632] = "\x02\x3e\x2e", [0x2633] = "\x01\x60\x5f",
  [0x2634] = "\x01\x60\x64", [0x2635] = "\x01\x60\x62",
  [0x2636] = "\x01\x60\x63", [0x2638] = "\x0e\x3d\x33",
  [0x2639] = "\x0e\x3d\x3c", [0x263a] = "\x01\x60\x5d",
  [0x263b] = "\x0e\x65\x66", [0x263c] = "\x0e\x3d\x37",
  [0x263d] = "\x0e\x42\x68", [0x263e] = "\x0e\x42\x75",
  [0x263f] = "\x01\x65\x31", [0x2640] = "\x02\x44\x7b",
  [0x2641] = "\x01\x65\x30", [0x2642] = "\x02\x45\x21",
  [0x2643] = "\x0e\x42\x6f", [0x2644] = "\x02\x44\x75",
  [0x2645] = "\x0e\x42\x74", [0x2646] = "\x02\x45\x22",
  [0x2647] = "\x0e\x42\x6e", [0x2648] = "\x0e\x42\x66",
  [0x2649] = "\x0e\x42\x71", [0x264a] = "\x02\x44\x76",
  [0x264b] = "\x02\x44\x77", [0x264c] = "\x0e\x42\x73",
  [0x264d] = "\x02\x45\x23", [0x264e] = "\x02\x44\x7e",
  [0x264f] = "\x02\x44\x7c", [0x2650] = "\x02\x44\x7d",
  [0x2651] = "\x02\x44\x79", [0x2652] = "\x02\x44\x78",
  [0x2653] = "\x0e\x42\x67", [0x2654] = "\x02\x45\x24",
  [0x2655] = "\x01\x65\x2d", [0x2656] = "\x0e\x42\x6b",
  [0x2657] = "\x02\x44\x7a", [0x2658] = "\x0e\x42\x70",
  [0x2659] = "\x01\x65\x32", [0x265a] = "\x01\x65\x2c",
  [0x265b] = "\x01\x65\x33", [0x265c] = "\x01\x65\x34",
  [0x265d] = "\x0e\x42\x6a", [0x265e] = "\x01\x65\x2f",
  [0x265f] = "\x01\x65\x2e", [0x2660] = "\x0e\x48\x25",
  [0x2661] = "\x0e\x47\x7e", [0x2662] = "\x02\x4b\x74",
  [0x2663] = "\x01\x69\x65", [0x2664] = "\x01\x69\x64",
  [0x2665] = "\x0e\x47\x7c", [0x2666] = "\x0e\x47\x7d",
  [0x2667] = "\x02\x4b\x79", [0x2668] = "\x0e\x48\x22",
  [0x2669] = "\x01\x6d\x66", [0x266a] = "\x01\x69\x66",
  [0x266b] = "\x0e\x48\x21", [0x266c] = "\x0e\x47\x7b",
  [0x266d] = "\x01\x69\x68", [0x266e] = "\x02\x4b\x7a",
  [0x266f] = "\x01\x65\x2b", [0x2670] = "\x01\x69\x67",
  [0x2671] = "\x02\x4b\x76", [0x2672] = "\x02\x4b\x78",
  [0x2673] = "\x02\x4b\x75", [0x2674] = "\x0e\x48\x27",
  [0x2675] = "\x02\x4b\x77", [0x2676] = "\x0e\x48\x23",
  [0x2679] = "\x02\x52\x33", [0x267a] = "\x0e\x4d\x31",
  [0x267b] = "\x0e\x4d\x3b", [0x267c] = "\x02\x52\x32",
  [0x267d] = "\x02\x52\x2f", [0x267e] = "\x01\x6d\x69",
  [0x267f] = "\x02\x58\x4e", [0x2680] = "\x01\x6d\x6a",
  [0x2681] = "\x02\x52\x2e", [0x2682] = "\x0e\x4d\x39",
  [0x2683] = "\x01\x6d\x68", [0x2684] = "\x0e\x4d\x36",
  [0x2685] = "\x02\x52\x30", [0x2686] = "\x02\x52\x2d",
  [0x2687] = "\x02\x52\x2a", [0x2688] = "\x02\x52\x31",
  [0x2689] = "\x02\x52\x2b", [0x268a] = "\x02\x52\x2c",
  [0x268b] = "\x01\x6d\x67", [0x268c] = "\x0e\x4d\x35",
  [0x268d] = "\x0e\x65\x68", [0x2690] = "\x02\x5d\x55",
  [0x2692] = "\x02\x58\x52", [0x2693] = "\x0e\x4d\x3a",
  [0x2694] = "\x02\x58\x51", [0x2695] = "\x02\x58\x53",
  [0x2696] = "\x0e\x51\x6a", [0x2697] = "\x02\x5d\x52",
  [0x2698] = "\x01\x71\x44", [0x2699] = "\x0e\x63\x75",
  [0x269a] = "\x02\x58\x4f", [0x269b] = "\x0e\x51\x67",
  [0x269c] = "\x01\x71\x42", [0x269d] = "\x0e\x51\x6c",
  [0x269e] = "\x01\x71\x46", [0x269f] = "\x01\x71\x45",
  [0x26a0] = "\x02\x58\x50", [0x26a1] = "\x02\x58\x54",
  [0x26a2] = "\x0e\x51\x6b", [0x26a3] = "\x01\x71\x43",
  [0x26a4] = "\x0e\x65\x67", [0x26a5] = "\x02\x5d\x59",
  [0x26a6] = "\x01\x74\x40", [0x26a7] = "\x01\x76\x66",
  [0x26a8] = "\x01\x74\x41", [0x26a9] = "\x01\x74\x3e",
  [0x26aa] = "\x02\x5d\x56", [0x26ab] = "\x02\x5d\x54",
  [0x26ac] = "\x0e\x55\x4f", [0x26ad] = "\x02\x5d\x57",
  [0x26ae] = "\x0e\x55\x4b", [0x26af] = "\x02\x5d\x5a",
  [0x26b0] = "\x01\x74\x3f", [0x26b1] = "\x02\x5d\x58",
  [0x26b2] = "\x02\x5d\x53", [0x26b3] = "\x0e\x55\x4d",
  [0x26b4] = "\x0e\x65\x69", [0x26b5] = "\x02\x62\x2f",
  [0x26b6] = "\x02\x62\x32", [0x26b7] = "\x02\x66\x21",
  [0x26b8] = "\x02\x62\x2d", [0x26b9] = "\x0e\x58\x2f",
  [0x26ba] = "\x02\x69\x3e", [0x26bb] = "\x02\x62\x33",
  [0x26bc] = "\x0e\x58\x30", [0x26bd] = "\x01\x78\x4d",
  [0x26be] = "\x02\x62\x31", [0x26bf] = "\x01\x76\x67",
  [0x26c0] = "\x02\x62\x2e", [0x26c1] = "\x02\x62\x30",
  [0x26c2] = "\x02\x62\x34", [0x26c3] = "\x02\x66\x22",
  [0x26c4] = "\x0e\x5a\x5a", [0x26c5] = "\x02\x65\x7e",
  [0x26c7] = "\x0e\x5a\x5c", [0x26c8] = "\x0e\x5a\x5d",
  [0x26c9] = "\x0e\x5a\x5b", [0x26ca] = "\x01\x78\x4e",
  [0x26cb] = "\x02\x65\x7d", [0x26cc] = "\x0e\x5c\x5e",
  [0x26cd] = "\x0e\x5c\x5c", [0x26ce] = "\x0e\x5c\x5f",
  [0x26cf] = "\x01\x79\x79", [0x26d0] = "\x0e\x5c\x5d",
  [0x26d1] = "\x0e\x5c\x60", [0x26d3] = "\x0e\x5e\x3d",
  [0x26d4] = "\x01\x7a\x78", [0x26d5] = "\x02\x6d\x5f",
  [0x26d6] = "\x01\x7a\x77", [0x26d7] = "\x02\x6d\x61",
  [0x26d8] = "\x02\x6d\x5e", [0x26d9] = "\x02\x6d\x60",
  [0x26da] = "\x01\x7c\x3c", [0x26db] = "\x02\x70\x40",
  [0x26dc] = "\x01\x46\x6a", [0x26dd] = "\x02\x28\x30",
  [0x26de] = "\x02\x30\x7d", [0x26df] = "\x02\x30\x7e",
  [0x26e0] = "\x01\x5c\x24", [0x26e1] = "\x02\x45\x25",
  [0x26e2] = "\x01\x71\x47", [0x26e3] = "\x01\x78\x4f",
  [0x26e4] = "\x01\x7b\x67", [0x26e5] = "\x02\x70\x41",
  [0x26e6] = "\x01\x46\x6b", [0x26e8] = "\x02\x28\x31",
  [0x26e9] = "\x01\x23\x22", [0x26eb] = "\x0e\x2e\x22",
  [0x26ec] = "\x02\x2c\x39", [0x26ed] = "\x0e\x2e\x21",
  [0x26ee] = "\x02\x2c\x3a", [0x26ef] = "\x0e\x2e\x24",
  [0x26f3] = "\x0e\x32\x48", [0x26f4] = "\x02\x31\x21",
  [0x26f5] = "\x02\x31\x22", [0x26f6] = "\x01\x5c\x25",
  [0x26f7] = "\x01\x5c\x26", [0x26f9] = "\x0e\x3d\x3d",
  [0x26fb] = "\x02\x3e\x2f", [0x26fd] = "\x02\x45\x28",
  [0x26fe] = "\x02\x45\x27", [0x26ff] = "\x02\x45\x26",
  [0x2700] = "\x02\x4b\x7b", [0x2701] = "\x0e\x42\x77",
  [0x2702] = "\x02\x4b\x7c", [0x2703] = "\x02\x4b\x7d",
  [0x2704] = "\x01\x69\x69", [0x2706] = "\x0e\x48\x28",
  [0x2707] = "\x02\x52\x35", [0x2708] = "\x02\x52\x34",
  [0x2709] = "\x0e\x4d\x3d", [0x270a] = "\x0e\x51\x6d",
  [0x270b] = "\x02\x58\x55", [0x270c] = "\x01\x71\x48",
  [0x270d] = "\x01\x71\x49", [0x270e] = "\x0e\x51\x6e",
  [0x270f] = "\x02\x5d\x5e", [0x2710] = "\x02\x5d\x5b",
  [0x2711] = "\x02\x5d\x5c", [0x2712] = "\x02\x5d\x5d",
  [0x2713] = "\x02\x62\x36", [0x2714] = "\x02\x62\x35",
  [0x2715] = "\x01\x76\x68", [0x2716] = "\x02\x66\x23",
  [0x2717] = "\x02\x6b\x6c", [0x2718] = "\x01\x46\x6c",
  [0x271a] = "\x01\x52\x49", [0x271b] = "\x0e\x37\x44",
  [0x271c] = "\x01\x5c\x27", [0x271d] = "\x02\x45\x29",
  [0x271e] = "\x0e\x42\x78", [0x271f] = "\x01\x46\x6d",
  [0x2721] = "\x02\x31\x23", [0x2722] = "\x01\x5c\x28",
  [0x2723] = "\x0e\x37\x45", [0x2724] = "\x0e\x3d\x3e",
  [0x2725] = "\x01\x60\x69", [0x2726] = "\x01\x60\x6a",
  [0x2728] = "\x01\x46\x6e", [0x2729] = "\x01\x46\x6f",
  [0x272a] = "\x02\x22\x66", [0x272b] = "\x01\x4b\x2d",
  [0x272c] = "\x01\x4b\x2c", [0x272d] = "\x01\x52\x4a",
  [0x272e] = "\x02\x2c\x3b", [0x272f] = "\x02\x3e\x30",
  [0x2730] = "\x01\x46\x70", [0x2731] = "\x01\x46\x71",
  [0x2732] = "\x01\x46\x72", [0x2733] = "\x01\x46\x73",
  [0x2737] = "\x01\x4b\x2e", [0x2738] = "\x01\x4b\x2f",
  [0x2739] = "\x02\x24\x6e", [0x273a] = "\x02\x24\x6d",
  [0x273b] = "\x0e\x2a\x37", [0x273d] = "\x01\x4e\x77",
  [0x273e] = "\x02\x28\x34", [0x273f] = "\x02\x28\x32",
  [0x2740] = "\x02\x28\x33", [0x2741] = "\x0e\x2a\x38",
  [0x2742] = "\x0e\x2a\x35", [0x2744] = "\x0e\x2a\x36",
  [0x2746] = "\x0e\x2e\x27", [0x2747] = "\x02\x2c\x3c",
  [0x2748] = "\x02\x2c\x3d", [0x2749] = "\x0e\x2e\x25",
  [0x274a] = "\x0e\x2e\x28", [0x274b] = "\x01\x52\x4e",
  [0x274c] = "\x01\x52\x4c", [0x274d] = "\x0e\x2e\x2a",
  [0x274e] = "\x01\x52\x4d", [0x274f] = "\x01\x52\x4b",
  [0x2750] = "\x0e\x2e\x26", [0x2754] = "\x01\x56\x63",
  [0x2755] = "\x0e\x32\x4e", [0x2756] = "\x0e\x32\x4d",
  [0x2757] = "\x0e\x32\x50", [0x2758] = "\x0e\x32\x4f",
  [0x2759] = "\x01\x56\x67", [0x275a] = "\x01\x56\x66",
  [0x275b] = "\x02\x31\x24", [0x275c] = "\x01\x56\x65",
  [0x275d] = "\x01\x56\x64", [0x275e] = "\x0e\x32\x51",
  [0x275f] = "\x02\x31\x25", [0x2761] = "\x0e\x37\x46",
  [0x2762] = "\x01\x5c\x2b", [0x2763] = "\x02\x37\x40",
  [0x2764] = "\x02\x37\x3f", [0x2765] = "\x01\x5c\x29",
  [0x2766] = "\x01\x5c\x2a", [0x2767] = "\x0e\x37\x49",
  [0x2768] = "\x0e\x37\x47", [0x276a] = "\x01\x60\x6c",
  [0x276b] = "\x01\x60\x6b", [0x276c] = "\x02\x3e\x32",
  [0x276d] = "\x0e\x3d\x43", [0x276e] = "\x0e\x3d\x42",
  [0x276f] = "\x02\x3e\x31", [0x2770] = "\x01\x5c\x2c",
  [0x2771] = "\x0e\x3d\x44", [0x2776] = "\x01\x65\x35",
  [0x2777] = "\x02\x45\x2b", [0x2778] = "\x01\x65\x36",
  [0x2779] = "\x02\x45\x2a", [0x277a] = "\x0e\x42\x7b",
  [0x277d] = "\x02\x4b\x7e", [0x277e] = "\x02\x52\x36",
  [0x277f] = "\x01\x6d\x6b", [0x2780] = "\x02\x58\x56",
  [0x2781] = "\x0e\x51\x70", [0x2783] = "\x0e\x55\x52",
  [0x2784] = "\x02\x5d\x5f", [0x2786] = "\x01\x78\x51",
  [0x2787] = "\x01\x78\x50", [0x2789] = "\x0e\x5c\x62",
  [0x278a] = "\x01\x7b\x68", [0x278b] = "\x01\x46\x74",
  [0x278c] = "\x02\x28\x35", [0x278e] = "\x0e\x3d\x45",
  [0x278f] = "\x01\x5c\x2d", [0x2790] = "\x02\x4c\x21",
  [0x2791] = "\x01\x69\x6a", [0x2792] = "\x0e\x22\x5b",
  [0x2793] = "\x0e\x26\x73", [0x2794] = "\x02\x24\x6f",
  [0x2795] = "\x02\x24\x70", [0x2798] = "\x02\x28\x36",
  [0x2799] = "\x01\x4e\x79", [0x279a] = "\x01\x4e\x7a",
  [0x279b] = "\x0e\x2a\x3a", [0x279d] = "\x01\x4e\x78",
  [0x279e] = "\x0e\x2a\x3b", [0x27a2] = "\x01\x52\x52",
  [0x27a3] = "\x01\x52\x53", [0x27a4] = "\x01\x52\x50",
  [0x27a5] = "\x01\x52\x51", [0x27a6] = "\x0e\x2e\x2b",
  [0x27a7] = "\x02\x2c\x3e", [0x27a8] = "\x0e\x2e\x2f",
  [0x27aa] = "\x02\x2c\x3f", [0x27ab] = "\x01\x52\x4f",
  [0x27b0] = "\x02\x31\x26", [0x27b1] = "\x0e\x32\x57",
  [0x27b2] = "\x01\x56\x6b", [0x27b3] = "\x01\x56\x6c",
  [0x27b4] = "\x0e\x32\x55", [0x27b5] = "\x01\x5c\x30",
  [0x27b6] = "\x02\x31\x2c", [0x27b8] = "\x01\x56\x71",
  [0x27b9] = "\x01\x56\x6f", [0x27ba] = "\x02\x31\x2d",
  [0x27bb] = "\x02\x31\x28", [0x27bc] = "\x01\x56\x6e",
  [0x27bd] = "\x01\x56\x6d", [0x27be] = "\x01\x56\x68",
  [0x27bf] = "\x02\x31\x2b", [0x27c0] = "\x02\x31\x2a",
  [0x27c1] = "\x02\x31\x27", [0x27c2] = "\x01\x56\x70",
  [0x27c3] = "\x0e\x32\x52", [0x27c4] = "\x02\x31\x29",
  [0x27c5] = "\x01\x56\x69", [0x27c7] = "\x01\x56\x6a",
  [0x27ca] = "\x01\x5c\x31", [0x27cb] = "\x02\x37\x44",
  [0x27cc] = "\x02\x37\x45", [0x27cd] = "\x01\x5c\x32",
  [0x27ce] = "\x02\x37\x41", [0x27cf] = "\x02\x37\x43",
  [0x27d0] = "\x02\x37\x47", [0x27d1] = "\x02\x37\x46",
  [0x27d2] = "\x02\x37\x42", [0x27d3] = "\x0e\x37\x4a",
  [0x27d4] = "\x01\x5c\x2e", [0x27d5] = "\x01\x5c\x2f",
  [0x27d7] = "\x02\x3e\x3a", [0x27d8] = "\x01\x60\x71",
  [0x27d9] = "\x01\x60\x70", [0x27da] = "\x02\x3e\x34",
  [0x27db] = "\x01\x60\x6e", [0x27dd] = "\x02\x3e\x37",
  [0x27de] = "\x01\x60\x72", [0x27df] = "\x02\x3e\x38",
  [0x27e0] = "\x01\x60\x73", [0x27e1] = "\x02\x3e\x35",
  [0x27e2] = "\x01\x60\x6d", [0x27e3] = "\x01\x60\x6f",
  [0x27e4] = "\x02\x3e\x39", [0x27e5] = "\x0e\x3d\x47",
  [0x27e6] = "\x02\x3e\x36", [0x27e7] = "\x02\x3e\x33",
  [0x27ed] = "\x02\x45\x38", [0x27ee] = "\x0e\x42\x7c",
  [0x27ef] = "\x02\x45\x2d", [0x27f0] = "\x01\x65\x38",
  [0x27f1] = "\x01\x65\x3b", [0x27f2] = "\x01\x65\x3a",
  [0x27f3] = "\x01\x65\x3f", [0x27f4] = "\x01\x65\x3e",
  [0x27f5] = "\x02\x45\x39", [0x27f6] = "\x02\x45\x37",
  [0x27f7] = "\x02\x45\x30", [0x27f8] = "\x02\x45\x34",
  [0x27f9] = "\x02\x45\x33", [0x27fa] = "\x01\x65\x3c",
  [0x27fb] = "\x02\x45\x36", [0x27fc] = "\x02\x45\x32",
  [0x27fd] = "\x02\x45\x3a", [0x27fe] = "\x02\x45\x31",
  [0x27ff] = "\x01\x65\x3d", [0x2800] = "\x01\x65\x37",
  [0x2801] = "\x01\x65\x39", [0x2802] = "\x0e\x42\x7d",
  [0x2803] = "\x02\x45\x2f", [0x2807] = "\x0e\x48\x2b",
  [0x2808] = "\x02\x4c\x23", [0x2809] = "\x01\x69\x6e",
  [0x280a] = "\x02\x4c\x27", [0x280b] = "\x01\x69\x6d",
  [0x280c] = "\x02\x4c\x24", [0x280d] = "\x01\x69\x6c",
  [0x280e] = "\x0e\x48\x29", [0x280f] = "\x02\x45\x2e",
  [0x2810] = "\x02\x45\x35", [0x2811] = "\x02\x4c\x26",
  [0x2813] = "\x01\x69\x6f", [0x2814] = "\x02\x4c\x28",
  [0x2815] = "\x02\x4c\x25", [0x2816] = "\x02\x4c\x22",
  [0x2819] = "\x02\x52\x39", [0x281a] = "\x02\x52\x3d",
  [0x281b] = "\x02\x52\x3f", [0x281c] = "\x02\x52\x3b",
  [0x281d] = "\x02\x52\x3a", [0x281e] = "\x02\x52\x38",
  [0x281f] = "\x01\x6d\x6e", [0x2820] = "\x01\x6d\x6c",
  [0x2821] = "\x01\x6d\x71", [0x2822] = "\x01\x6d\x72",
  [0x2823] = "\x02\x52\x3c", [0x2824] = "\x01\x6d\x6f",
  [0x2825] = "\x02\x52\x37", [0x2826] = "\x01\x6d\x70",
  [0x2827] = "\x01\x69\x6b", [0x2828] = "\x02\x52\x3e",
  [0x2829] = "\x01\x6d\x6d", [0x282c] = "\x0e\x51\x71",
  [0x282d] = "\x02\x58\x58", [0x282f] = "\x02\x58\x57",
  [0x2830] = "\x02\x58\x5f", [0x2831] = "\x02\x58\x59",
  [0x2832] = "\x02\x58\x5e", [0x2833] = "\x02\x58\x5b",
  [0x2834] = "\x01\x71\x4a", [0x2835] = "\x02\x58\x5d",
  [0x2836] = "\x0e\x51\x73", [0x2837] = "\x0e\x51\x72",
  [0x2838] = "\x01\x71\x4b", [0x283a] = "\x01\x71\x4c",
  [0x283b] = "\x0e\x51\x74", [0x283c] = "\x02\x58\x5c",
  [0x283d] = "\x02\x58\x5a", [0x2842] = "\x01\x74\x43",
  [0x2843] = "\x02\x5d\x60", [0x2844] = "\x0e\x55\x55",
  [0x2845] = "\x0e\x55\x53", [0x2846] = "\x01\x74\x42",
  [0x2847] = "\x02\x5d\x63", [0x2848] = "\x02\x5d\x61",
  [0x2849] = "\x02\x5d\x62", [0x284c] = "\x01\x74\x44",
  [0x284e] = "\x0e\x55\x54", [0x284f] = "\x0e\x58\x34",
  [0x2850] = "\x02\x62\x3a", [0x2851] = "\x0e\x58\x33",
  [0x2852] = "\x01\x76\x6b", [0x2853] = "\x02\x62\x3b",
  [0x2856] = "\x01\x76\x69", [0x2857] = "\x02\x62\x3c",
  [0x2858] = "\x01\x76\x6a", [0x2859] = "\x02\x62\x39",
  [0x285a] = "\x02\x62\x3d", [0x285b] = "\x0e\x58\x32",
  [0x285c] = "\x02\x62\x37", [0x285f] = "\x01\x78\x52",
  [0x2860] = "\x02\x66\x24", [0x2861] = "\x01\x78\x53",
  [0x2862] = "\x01\x79\x7a", [0x2864] = "\x02\x62\x38",
  [0x2865] = "\x01\x79\x7b", [0x2867] = "\x0e\x5e\x3f",
  [0x2868] = "\x0e\x5e\x3e", [0x2869] = "\x01\x7a\x79",
  [0x286a] = "\x02\x6b\x6d", [0x286c] = "\x01\x7b\x6a",
  [0x286d] = "\x02\x6d\x62", [0x286e] = "\x01\x7b\x69",
  [0x286f] = "\x0e\x60\x3b", [0x2870] = "\x02\x6f\x39",
  [0x2871] = "\x01\x7c\x61", [0x2872] = "\x01\x7c\x62",
  [0x2873] = "\x0e\x65\x6a", [0x2875] = "\x02\x72\x3a",
  [0x2876] = "\x0e\x22\x5c", [0x2877] = "\x0e\x65\x6b",
  [0x2878] = "\x01\x52\x54", [0x2879] = "\x02\x2c\x40",
  [0x287b] = "\x01\x60\x74", [0x287c] = "\x01\x60\x75",
  [0x287d] = "\x01\x46\x75", [0x287e] = "\x01\x48\x4b",
  [0x287f] = "\x02\x22\x67", [0x2880] = "\x0e\x26\x74",
  [0x2881] = "\x02\x24\x71", [0x2882] = "\x01\x4b\x30",
  [0x2883] = "\x0e\x26\x76", [0x2884] = "\x01\x4e\x7b",
  [0x2886] = "\x01\x52\x55", [0x2887] = "\x01\x52\x56",
  [0x2888] = "\x01\x52\x57", [0x2889] = "\x02\x37\x49",
  [0x288a] = "\x02\x31\x2e", [0x288b] = "\x01\x56\x72",
  [0x288e] = "\x01\x5c\x33", [0x288f] = "\x02\x37\x48",
  [0x2890] = "\x0e\x37\x4d", [0x2892] = "\x02\x3e\x3c",
  [0x2893] = "\x01\x60\x77", [0x2894] = "\x0e\x3d\x4c",
  [0x2895] = "\x02\x3e\x3b", [0x2896] = "\x01\x60\x76",
  [0x2897] = "\x0e\x42\x7e", [0x2898] = "\x0e\x43\x21",
  [0x2899] = "\x02\x45\x3b", [0x289a] = "\x01\x6d\x73",
  [0x289b] = "\x02\x52\x43", [0x289c] = "\x02\x52\x40",
  [0x289d] = "\x02\x52\x41", [0x289e] = "\x02\x52\x42",
  [0x28a1] = "\x0e\x51\x76", [0x28a3] = "\x0e\x55\x56",
  [0x28a4] = "\x02\x5d\x64", [0x28a5] = "\x0e\x55\x57",
  [0x28a6] = "\x02\x62\x3e", [0x28a9] = "\x0e\x5a\x5e",
  [0x28aa] = "\x02\x69\x40", [0x28ab] = "\x02\x69\x3f",
  [0x28ad] = "\x02\x6d\x63", [0x28ae] = "\x01\x46\x76",
  [0x28af] = "\x02\x28\x37", [0x28b0] = "\x01\x56\x73",
  [0x28b3] = "\x0e\x3d\x4d", [0x28b4] = "\x01\x60\x78",
  [0x28b5] = "\x02\x45\x3c", [0x28b7] = "\x0e\x48\x30",
  [0x28b8] = "\x02\x4c\x29", [0x28b9] = "\x0e\x48\x2e",
  [0x28ba] = "\x01\x6d\x74", [0x28bb] = "\x02\x58\x60",
  [0x28bc] = "\x0e\x65\x6c", [0x28bd] = "\x02\x62\x3f",
  [0x28be] = "\x02\x69\x41", [0x28bf] = "\x01\x46\x77",
  [0x28c0] = "\x0e\x26\x77", [0x28c1] = "\x0e\x26\x78",
  [0x28c2] = "\x01\x4e\x7c", [0x28c3] = "\x01\x52\x5a",
  [0x28c4] = "\x02\x2c\x41", [0x28c5] = "\x01\x52\x5b",
  [0x28c6] = "\x01\x52\x59", [0x28c7] = "\x0e\x2e\x32",
  [0x28c8] = "\x01\x52\x58", [0x28c9] = "\x02\x31\x2f",
  [0x28ca] = "\x01\x56\x74", [0x28cb] = "\x0e\x32\x5a",
  [0x28cc] = "\x0e\x32\x5b", [0x28cd] = "\x01\x56\x75",
  [0x28ce] = "\x01\x56\x76", [0x28d2] = "\x01\x5c\x35",
  [0x28d3] = "\x02\x37\x4a", [0x28d4] = "\x01\x5c\x34",
  [0x28d5] = "\x0e\x37\x50", [0x28d6] = "\x0e\x37\x52",
  [0x28d7] = "\x0e\x37\x51", [0x28d9] = "\x0e\x3d\x4f",
  [0x28da] = "\x02\x3e\x3d", [0x28db] = "\x01\x5c\x36",
  [0x28dc] = "\x01\x60\x79", [0x28dd] = "\x02\x45\x3d",
  [0x28de] = "\x01\x65\x40", [0x28df] = "\x01\x65\x41",
  [0x28e1] = "\x01\x69\x70", [0x28e2] = "\x0e\x48\x31",
  [0x28e3] = "\x01\x69\x71", [0x28e4] = "\x01\x6d\x75",
  [0x28e5] = "\x01\x71\x4e", [0x28e6] = "\x02\x58\x61",
  [0x28e7] = "\x01\x71\x4d", [0x28e9] = "\x02\x5d\x65",
  [0x28ea] = "\x01\x74\x45", [0x28eb] = "\x0e\x58\x35",
  [0x28ec] = "\x02\x62\x40", [0x28ed] = "\x02\x69\x42",
  [0x28ee] = "\x01\x46\x78", [0x28ef] = "\x01\x4b\x31",
  [0x28f0] = "\x02\x28\x3a", [0x28f1] = "\x02\x28\x39",
  [0x28f2] = "\x01\x4e\x7d", [0x28f3] = "\x02\x28\x38",
  [0x28f4] = "\x01\x4e\x7e", [0x28f5] = "\x02\x28\x3b",
  [0x28f7] = "\x02\x2c\x47", [0x28f8] = "\x01\x52\x5e",
  [0x28f9] = "\x01\x52\x5d", [0x28fa] = "\x02\x2c\x49",
  [0x28fb] = "\x02\x2c\x48", [0x28fc] = "\x01\x52\x62",
  [0x28fd] = "\x0e\x2e\x35", [0x28fe] = "\x01\x52\x61",
  [0x28ff] = "\x0e\x2e\x33", [0x2901] = "\x01\x52\x5c",
  [0x2903] = "\x02\x2c\x43", [0x2904] = "\x02\x2c\x44",
  [0x2905] = "\x02\x2c\x45", [0x2906] = "\x0e\x62\x6c",
  [0x2907] = "\x01\x52\x63", [0x2908] = "\x02\x2c\x42",
  [0x2909] = "\x01\x52\x5f", [0x290a] = "\x02\x2c\x46",
  [0x290b] = "\x01\x52\x60", [0x290e] = "\x0e\x32\x64",
  [0x290f] = "\x0e\x32\x5f", [0x2910] = "\x02\x31\x32",
  [0x2911] = "\x02\x31\x36", [0x2912] = "\x02\x31\x34",
  [0x2913] = "\x02\x31\x33", [0x2915] = "\x02\x31\x37",
  [0x2916] = "\x0e\x32\x5e", [0x2918] = "\x0e\x32\x68",
  [0x2919] = "\x02\x31\x38", [0x291a] = "\x02\x31\x39",
  [0x291b] = "\x02\x31\x31", [0x291d] = "\x02\x31\x30",
  [0x291e] = "\x0e\x37\x59", [0x291f] = "\x01\x56\x78",
  [0x2920] = "\x01\x56\x79", [0x2921] = "\x0e\x32\x67",
  [0x2922] = "\x02\x31\x3a", [0x2923] = "\x02\x31\x35",
  [0x2924] = "\x0e\x32\x65", [0x2925] = "\x02\x37\x53",
  [0x2926] = "\x0e\x37\x55", [0x2927] = "\x02\x31\x3b",
  [0x2928] = "\x01\x56\x7a", [0x2929] = "\x01\x56\x77",
  [0x292a] = "\x0e\x32\x5c", [0x292b] = "\x0e\x32\x63",
  [0x292d] = "\x02\x37\x4d", [0x292f] = "\x02\x37\x4c",
  [0x2930] = "\x0e\x37\x53", [0x2931] = "\x02\x37\x4e",
  [0x2932] = "\x02\x37\x4f", [0x2933] = "\x02\x37\x51",
  [0x2934] = "\x02\x37\x50", [0x2935] = "\x02\x37\x55",
  [0x2936] = "\x01\x5c\x3a", [0x2937] = "\x01\x5c\x37",
  [0x2938] = "\x01\x5c\x3b", [0x2939] = "\x02\x37\x4b",
  [0x293a] = "\x01\x5c\x3c", [0x293b] = "\x02\x37\x54",
  [0x293c] = "\x01\x5c\x39", [0x293d] = "\x02\x37\x52",
  [0x293e] = "\x01\x5c\x38", [0x293f] = "\x0e\x32\x62",
  [0x2940] = "\x0e\x37\x57", [0x2942] = "\x0e\x3d\x56",
  [0x2943] = "\x0e\x3d\x53", [0x2944] = "\x02\x3e\x40",
  [0x2945] = "\x02\x3e\x42", [0x2946] = "\x02\x3e\x3e",
  [0x2947] = "\x02\x3e\x3f", [0x2948] = "\x0e\x3d\x51",
  [0x2949] = "\x0e\x3d\x52", [0x294a] = "\x02\x3e\x43",
  [0x294b] = "\x02\x3e\x45", [0x294c] = "\x02\x3e\x46",
  [0x294d] = "\x02\x3e\x41", [0x294e] = "\x02\x3e\x44",
  [0x294f] = "\x01\x60\x7a", [0x2952] = "\x02\x45\x41",
  [0x2953] = "\x0e\x43\x27", [0x2954] = "\x02\x45\x46",
  [0x2955] = "\x02\x45\x3e", [0x2956] = "\x02\x45\x42",
  [0x2958] = "\x0e\x43\x28", [0x2959] = "\x02\x45\x47",
  [0x295a] = "\x02\x45\x43", [0x295b] = "\x01\x65\x42",
  [0x295c] = "\x01\x65\x4a", [0x295d] = "\x0e\x43\x25",
  [0x295e] = "\x01\x65\x45", [0x295f] = "\x02\x45\x3f",
  [0x2960] = "\x02\x45\x40", [0x2961] = "\x01\x69\x75",
  [0x2962] = "\x01\x65\x4d", [0x2963] = "\x01\x65\x46",
  [0x2964] = "\x0e\x43\x26", [0x2965] = "\x01\x65\x4b",
  [0x2966] = "\x01\x65\x44", [0x2967] = "\x02\x45\x45",
  [0x2968] = "\x01\x65\x4c", [0x2969] = "\x02\x45\x44",
  [0x296a] = "\x01\x65\x48", [0x296b] = "\x01\x65\x43",
  [0x296c] = "\x01\x65\x49", [0x296d] = "\x02\x45\x48",
  [0x296e] = "\x02\x4c\x2e", [0x296f] = "\x02\x4c\x30",
  [0x2971] = "\x0e\x48\x39", [0x2974] = "\x0e\x48\x33",
  [0x2975] = "\x0e\x48\x36", [0x2976] = "\x0e\x48\x35",
  [0x2977] = "\x0e\x48\x34", [0x2978] = "\x0e\x48\x38",
  [0x2979] = "\x01\x65\x47", [0x297a] = "\x0e\x48\x3b",
  [0x297b] = "\x0e\x48\x3d", [0x297c] = "\x02\x4c\x2b",
  [0x297d] = "\x01\x69\x73", [0x297e] = "\x02\x4c\x31",
  [0x297f] = "\x01\x69\x74", [0x2980] = "\x02\x4c\x2f",
  [0x2981] = "\x02\x4c\x2a", [0x2982] = "\x02\x4c\x2d",
  [0x2983] = "\x02\x4c\x32", [0x2984] = "\x01\x69\x72",
  [0x2985] = "\x02\x4c\x2c", [0x2987] = "\x01\x6d\x77",
  [0x2988] = "\x02\x52\x47", [0x2989] = "\x02\x52\x46",
  [0x298a] = "\x0e\x4d\x44", [0x298b] = "\x01\x6d\x7a",
  [0x298c] = "\x01\x6d\x78", [0x298d] = "\x02\x52\x44",
  [0x298e] = "\x01\x6d\x76", [0x298f] = "\x02\x52\x45",
  [0x2990] = "\x0e\x4d\x46", [0x2991] = "\x01\x6d\x79",
  [0x2995] = "\x02\x58\x69", [0x2996] = "\x0e\x51\x7b",
  [0x2997] = "\x02\x58\x6b", [0x2998] = "\x0e\x51\x79",
  [0x2999] = "\x02\x58\x6a", [0x299a] = "\x02\x58\x62",
  [0x299b] = "\x02\x58\x66", [0x299c] = "\x02\x58\x65",
  [0x299d] = "\x02\x58\x63", [0x299e] = "\x01\x71\x50",
  [0x299f] = "\x01\x71\x51", [0x29a0] = "\x01\x71\x4f",
  [0x29a1] = "\x02\x58\x64", [0x29a2] = "\x02\x58\x67",
  [0x29a3] = "\x02\x58\x68", [0x29a4] = "\x0e\x55\x59",
  [0x29a5] = "\x01\x71\x52", [0x29a6] = "\x0e\x55\x58",
  [0x29a7] = "\x01\x74\x4a", [0x29a8] = "\x02\x5d\x6d",
  [0x29a9] = "\x0e\x55\x5b", [0x29aa] = "\x01\x74\x47",
  [0x29ab] = "\x02\x5d\x67", [0x29ac] = "\x01\x74\x49",
  [0x29ad] = "\x01\x74\x4b", [0x29af] = "\x0e\x55\x5c",
  [0x29b0] = "\x01\x74\x48", [0x29b1] = "\x02\x5d\x6c",
  [0x29b2] = "\x02\x5d\x68", [0x29b3] = "\x01\x74\x46",
  [0x29b4] = "\x02\x5d\x6b", [0x29b5] = "\x02\x5d\x66",
  [0x29b6] = "\x02\x5d\x6a", [0x29b7] = "\x02\x5d\x69",
  [0x29b8] = "\x0e\x58\x36", [0x29b9] = "\x0e\x58\x38",
  [0x29ba] = "\x02\x62\x42", [0x29bb] = "\x01\x76\x6e",
  [0x29bc] = "\x01\x76\x6f", [0x29bd] = "\x01\x76\x6c",
  [0x29be] = "\x0e\x58\x3b", [0x29bf] = "\x01\x76\x6d",
  [0x29c0] = "\x0e\x58\x3a", [0x29c1] = "\x0e\x58\x39",
  [0x29c2] = "\x02\x62\x41", [0x29c4] = "\x02\x66\x27",
  [0x29c5] = "\x0e\x5a\x60", [0x29c6] = "\x0e\x5a\x5f",
  [0x29c7] = "\x01\x78\x54", [0x29c9] = "\x02\x66\x25",
  [0x29ca] = "\x02\x66\x26", [0x29cc] = "\x02\x69\x43",
  [0x29cd] = "\x02\x69\x46", [0x29ce] = "\x02\x69\x44",
  [0x29cf] = "\x02\x69\x45", [0x29d0] = "\x02\x6b\x6e",
  [0x29d1] = "\x0e\x5e\x42", [0x29d3] = "\x01\x7a\x7a",
  [0x29d4] = "\x02\x6f\x3a", [0x29d5] = "\x02\x70\x42",
  [0x29d6] = "\x0e\x64\x32", [0x29d7] = "\x01\x7c\x63",
  [0x29d8] = "\x02\x71\x28", [0x29d9] = "\x02\x71\x29",
  [0x29da] = "\x01\x7d\x33", [0x29db] = "\x01\x46\x79",
  [0x29dc] = "\x01\x52\x64", [0x29dd] = "\x0e\x32\x69",
  [0x29de] = "\x02\x3e\x47", [0x29df] = "\x0e\x3d\x57",
  [0x29e0] = "\x02\x45\x49", [0x29e2] = "\x01\x46\x7a",
  [0x29e3] = "\x01\x4b\x32", [0x29e5] = "\x01\x4f\x21",
  [0x29e6] = "\x0e\x2e\x36", [0x29e7] = "\x02\x2c\x4a",
  [0x29e8] = "\x02\x2c\x4b", [0x29e9] = "\x01\x56\x7b",
  [0x29ec] = "\x02\x3e\x48", [0x29ed] = "\x01\x60\x7b",
  [0x29ee] = "\x01\x65\x4e", [0x29ef] = "\x01\x74\x4c",
  [0x29f0] = "\x02\x5d\x6e", [0x29f1] = "\x02\x66\x28",
  [0x29f2] = "\x02\x69\x47", [0x29f3] = "\x01\x46\x7b",
  [0x29f4] = "\x0e\x26\x7b", [0x29f5] = "\x0e\x26\x7c",
  [0x29f7] = "\x02\x28\x41", [0x29f8] = "\x02\x28\x3c",
  [0x29f9] = "\x02\x28\x3e", [0x29fa] = "\x02\x28\x40",
  [0x29fb] = "\x02\x28\x3f", [0x29fc] = "\x02\x28\x3d",
  [0x29fd] = "\x01\x4f\x22", [0x2a02] = "\x01\x52\x65",
  [0x2a03] = "\x02\x2c\x54", [0x2a04] = "\x0e\x2e\x39",
  [0x2a05] = "\x02\x2c\x4f", [0x2a06] = "\x02\x2c\x4c",
  [0x2a07] = "\x0e\x2e\x3b", [0x2a08] = "\x0e\x2e\x3e",
  [0x2a09] = "\x02\x2c\x53", [0x2a0a] = "\x0e\x2e\x37",
  [0x2a0b] = "\x0e\x2e\x3a", [0x2a0c] = "\x01\x52\x67",
  [0x2a0d] = "\x01\x52\x68", [0x2a0e] = "\x02\x2c\x52",
  [0x2a0f] = "\x02\x2c\x51", [0x2a10] = "\x02\x2c\x50",
  [0x2a11] = "\x02\x2c\x4d", [0x2a12] = "\x02\x2c\x4e",
  [0x2a13] = "\x02\x2c\x55", [0x2a14] = "\x01\x52\x66",
  [0x2a15] = "\x0e\x2e\x3d", [0x2a19] = "\x0e\x65\x6d",
  [0x2a1b] = "\x0e\x65\x6e", [0x2a1d] = "\x01\x57\x21",
  [0x2a1e] = "\x0e\x66\x7c", [0x2a1f] = "\x01\x57\x27",
  [0x2a20] = "\x01\x57\x26", [0x2a21] = "\x02\x31\x44",
  [0x2a22] = "\x02\x31\x3e", [0x2a23] = "\x02\x31\x3c",
  [0x2a24] = "\x0e\x32\x6c", [0x2a25] = "\x01\x57\x24",
  [0x2a26] = "\x02\x37\x5c", [0x2a27] = "\x01\x56\x7d",
  [0x2a28] = "\x02\x31\x41", [0x2a29] = "\x02\x31\x45",
  [0x2a2a] = "\x02\x31\x47", [0x2a2b] = "\x02\x31\x43",
  [0x2a2c] = "\x02\x31\x3d", [0x2a2d] = "\x01\x57\x25",
  [0x2a2e] = "\x02\x31\x42", [0x2a2f] = "\x02\x31\x40",
  [0x2a30] = "\x01\x56\x7c", [0x2a31] = "\x02\x31\x48",
  [0x2a32] = "\x01\x57\x28", [0x2a33] = "\x02\x31\x46",
  [0x2a34] = "\x01\x57\x22", [0x2a35] = "\x02\x31\x3f",
  [0x2a36] = "\x0e\x32\x6d", [0x2a37] = "\x01\x57\x23",
  [0x2a38] = "\x01\x56\x7e", [0x2a3d] = "\x0e\x65\x6f",
  [0x2a42] = "\x0e\x37\x62", [0x2a43] = "\x01\x5c\x3e",
  [0x2a45] = "\x02\x37\x5d", [0x2a46] = "\x0e\x37\x63",
  [0x2a47] = "\x0e\x37\x5d", [0x2a48] = "\x02\x37\x56",
  [0x2a49] = "\x02\x37\x58", [0x2a4a] = "\x02\x37\x5a",
  [0x2a4b] = "\x0e\x37\x5f", [0x2a4c] = "\x02\x37\x5b",
  [0x2a4d] = "\x02\x37\x59", [0x2a4e] = "\x01\x5c\x3f",
  [0x2a4f] = "\x0e\x37\x61", [0x2a50] = "\x02\x37\x5e",
  [0x2a51] = "\x0e\x37\x60", [0x2a52] = "\x02\x37\x57",
  [0x2a53] = "\x0e\x65\x70", [0x2a54] = "\x0e\x65\x71",
  [0x2a58] = "\x0e\x65\x72", [0x2a5c] = "\x02\x3e\x4c",
  [0x2a5d] = "\x01\x60\x7c", [0x2a5e] = "\x02\x3e\x54",
  [0x2a5f] = "\x0e\x3d\x5c", [0x2a60] = "\x02\x3e\x49",
  [0x2a62] = "\x02\x3e\x55", [0x2a63] = "\x0e\x3d\x59",
  [0x2a64] = "\x02\x3e\x4a", [0x2a65] = "\x02\x3e\x4b",
  [0x2a66] = "\x0e\x3d\x5d", [0x2a68] = "\x02\x3e\x53",
  [0x2a69] = "\x02\x3e\x52", [0x2a6a] = "\x02\x3e\x4f",
  [0x2a6b] = "\x01\x5c\x3d", [0x2a6c] = "\x01\x60\x7d",
  [0x2a6d] = "\x02\x3e\x4d", [0x2a6e] = "\x02\x3e\x50",
  [0x2a6f] = "\x01\x60\x7e", [0x2a70] = "\x02\x3e\x51",
  [0x2a71] = "\x02\x3e\x4e", [0x2a76] = "\x0e\x3d\x5a",
  [0x2a78] = "\x0e\x43\x31", [0x2a79] = "\x02\x45\x54",
  [0x2a7a] = "\x0e\x43\x2d", [0x2a7b] = "\x02\x45\x58",
  [0x2a7c] = "\x01\x65\x55", [0x2a7d] = "\x0e\x43\x2e",
  [0x2a7e] = "\x02\x4c\x38", [0x2a7f] = "\x01\x65\x58",
  [0x2a80] = "\x02\x45\x56", [0x2a81] = "\x0e\x43\x37",
  [0x2a82] = "\x0e\x43\x29", [0x2a83] = "\x02\x45\x53",
  [0x2a84] = "\x02\x45\x4e", [0x2a85] = "\x02\x45\x50",
  [0x2a86] = "\x02\x45\x51", [0x2a87] = "\x02\x45\x4a",
  [0x2a88] = "\x0e\x43\x35", [0x2a89] = "\x01\x65\x54",
  [0x2a8a] = "\x0e\x43\x2b", [0x2a8b] = "\x0e\x43\x30",
  [0x2a8c] = "\x01\x65\x53", [0x2a8d] = "\x0e\x43\x34",
  [0x2a8e] = "\x01\x65\x4f", [0x2a8f] = "\x02\x45\x4d",
  [0x2a90] = "\x0e\x43\x2a", [0x2a91] = "\x01\x65\x56",
  [0x2a92] = "\x0e\x43\x36", [0x2a93] = "\x01\x65\x57",
  [0x2a94] = "\x02\x45\x4c", [0x2a95] = "\x02\x45\x4f",
  [0x2a96] = "\x02\x45\x57", [0x2a97] = "\x01\x65\x51",
  [0x2a98] = "\x01\x65\x52", [0x2a99] = "\x02\x45\x55",
  [0x2a9a] = "\x02\x45\x4b", [0x2a9e] = "\x02\x4c\x3a",
  [0x2a9f] = "\x01\x69\x77", [0x2aa0] = "\x02\x4c\x3c",
  [0x2aa1] = "\x02\x45\x52", [0x2aa2] = "\x02\x4c\x3e",
  [0x2aa3] = "\x01\x69\x7b", [0x2aa4] = "\x02\x4c\x3f",
  [0x2aa5] = "\x02\x4c\x3b", [0x2aa6] = "\x0e\x48\x3e",
  [0x2aa7] = "\x01\x69\x78", [0x2aa8] = "\x02\x4c\x37",
  [0x2aa9] = "\x01\x69\x7a", [0x2aaa] = "\x02\x4c\x34",
  [0x2aab] = "\x02\x4c\x39", [0x2aac] = "\x02\x4c\x3d",
  [0x2aad] = "\x02\x4c\x36", [0x2aae] = "\x0e\x48\x3f",
  [0x2aaf] = "\x0e\x48\x45", [0x2ab0] = "\x01\x65\x50",
  [0x2ab1] = "\x0e\x48\x44", [0x2ab2] = "\x02\x4c\x33",
  [0x2ab3] = "\x01\x69\x79", [0x2ab4] = "\x02\x4c\x35",
  [0x2ab7] = "\x0e\x65\x73", [0x2ab8] = "\x0e\x48\x40",
  [0x2aba] = "\x01\x6d\x7d", [0x2abb] = "\x02\x52\x49",
  [0x2abc] = "\x01\x6e\x23", [0x2abe] = "\x01\x6e\x21",
  [0x2ac0] = "\x0e\x4d\x4a", [0x2ac1] = "\x01\x69\x76",
  [0x2ac2] = "\x0e\x4d\x4d", [0x2ac3] = "\x02\x52\x50",
  [0x2ac4] = "\x02\x52\x51", [0x2ac5] = "\x01\x6d\x7c",
  [0x2ac6] = "\x0e\x4d\x48", [0x2ac7] = "\x0e\x4d\x4c",
  [0x2ac8] = "\x02\x52\x4f", [0x2ac9] = "\x02\x52\x52",
  [0x2aca] = "\x01\x6d\x7e", [0x2acb] = "\x01\x6d\x7b",
  [0x2acc] = "\x02\x52\x4b", [0x2acd] = "\x02\x52\x48",
  [0x2ace] = "\x02\x52\x4d", [0x2acf] = "\x02\x52\x4a",
  [0x2ad0] = "\x01\x6e\x24", [0x2ad1] = "\x02\x52\x4c",
  [0x2ad2] = "\x0e\x4d\x4b", [0x2ad3] = "\x0e\x4d\x49",
  [0x2ad4] = "\x02\x52\x4e", [0x2ad5] = "\x01\x6e\x22",
  [0x2ad6] = "\x0e\x52\x21", [0x2ad8] = "\x0e\x65\x74",
  [0x2ada] = "\x01\x71\x54", [0x2adb] = "\x02\x58\x72",
  [0x2adc] = "\x0e\x52\x24", [0x2add] = "\x02\x58\x6c",
  [0x2ade] = "\x02\x58\x70", [0x2adf] = "\x02\x58\x76",
  [0x2ae0] = "\x02\x58\x77", [0x2ae1] = "\x02\x58\x73",
  [0x2ae2] = "\x02\x58\x74", [0x2ae3] = "\x02\x58\x71",
  [0x2ae5] = "\x02\x58\x6e", [0x2ae7] = "\x01\x71\x56",
  [0x2ae8] = "\x01\x71\x53", [0x2ae9] = "\x02\x58\x6d",
  [0x2aea] = "\x02\x58\x6f", [0x2aeb] = "\x0e\x52\x22",
  [0x2aec] = "\x01\x71\x55", [0x2aed] = "\x02\x58\x75",
  [0x2aee] = "\x0e\x65\x75", [0x2aef] = "\x01\x74\x50",
  [0x2af0] = "\x0e\x55\x60", [0x2af1] = "\x0e\x55\x62",
  [0x2af2] = "\x02\x5d\x75", [0x2af3] = "\x02\x5d\x6f",
  [0x2af4] = "\x01\x74\x4f", [0x2af7] = "\x01\x74\x4d",
  [0x2af8] = "\x0e\x55\x61", [0x2af9] = "\x02\x5d\x77",
  [0x2afa] = "\x01\x74\x4e", [0x2afb] = "\x02\x5d\x72",
  [0x2afc] = "\x02\x5d\x73", [0x2afd] = "\x02\x5d\x70",
  [0x2afe] = "\x02\x5d\x78", [0x2aff] = "\x02\x5d\x74",
  [0x2b01] = "\x01\x74\x51", [0x2b02] = "\x02\x5d\x71",
  [0x2b04] = "\x02\x5d\x79", [0x2b05] = "\x02\x5d\x76",
  [0x2b06] = "\x0e\x58\x43", [0x2b07] = "\x0e\x58\x3e",
  [0x2b08] = "\x0e\x58\x40", [0x2b09] = "\x02\x62\x46",
  [0x2b0a] = "\x0e\x58\x3f", [0x2b0b] = "\x0e\x58\x3c",
  [0x2b0c] = "\x02\x62\x43", [0x2b0d] = "\x0e\x58\x42",
  [0x2b0e] = "\x01\x76\x70", [0x2b10] = "\x02\x62\x47",
  [0x2b11] = "\x02\x62\x49", [0x2b12] = "\x02\x62\x48",
  [0x2b13] = "\x02\x62\x44", [0x2b14] = "\x02\x62\x45",
  [0x2b15] = "\x0e\x58\x44", [0x2b17] = "\x02\x66\x2d",
  [0x2b18] = "\x0e\x5a\x61", [0x2b19] = "\x01\x78\x55",
  [0x2b1a] = "\x0e\x5a\x62", [0x2b1b] = "\x02\x66\x2a",
  [0x2b1c] = "\x02\x66\x2c", [0x2b1d] = "\x02\x66\x29",
  [0x2b1e] = "\x02\x66\x2e", [0x2b20] = "\x0e\x5a\x63",
  [0x2b21] = "\x02\x66\x2b", [0x2b22] = "\x0e\x65\x76",
  [0x2b23] = "\x02\x69\x49", [0x2b24] = "\x02\x69\x4c",
  [0x2b25] = "\x02\x69\x48", [0x2b26] = "\x01\x79\x7c",
  [0x2b27] = "\x02\x69\x4a", [0x2b28] = "\x02\x69\x4b",
  [0x2b29] = "\x02\x69\x4d", [0x2b2a] = "\x01\x79\x7d",
  [0x2b2b] = "\x01\x7a\x21", [0x2b2c] = "\x01\x79\x7e",
  [0x2b2d] = "\x02\x6b\x6f", [0x2b2e] = "\x0e\x5e\x45",
  [0x2b2f] = "\x02\x6b\x71", [0x2b30] = "\x0e\x5e\x43",
  [0x2b31] = "\x02\x6b\x70", [0x2b32] = "\x0e\x5e\x44",
  [0x2b35] = "\x02\x6d\x64", [0x2b37] = "\x0e\x60\x3c",
  [0x2b38] = "\x02\x70\x43", [0x2b39] = "\x02\x71\x2a",
  [0x2b3a] = "\x01\x46\x7c", [0x2b3c] = "\x0e\x24\x33",
  [0x2b3d] = "\x02\x24\x72", [0x2b3e] = "\x01\x4f\x23",
  [0x2b3f] = "\x02\x28\x43", [0x2b40] = "\x01\x4f\x24",
  [0x2b41] = "\x01\x4f\x25", [0x2b42] = "\x02\x28\x42",
  [0x2b44] = "\x02\x2c\x5a", [0x2b45] = "\x02\x2c\x59",
  [0x2b46] = "\x01\x52\x69", [0x2b47] = "\x01\x52\x6c",
  [0x2b48] = "\x01\x52\x6b", [0x2b49] = "\x01\x52\x6a",
  [0x2b4a] = "\x02\x2c\x56", [0x2b4b] = "\x02\x2c\x58",
  [0x2b4c] = "\x02\x2c\x57", [0x2b4d] = "\x0e\x65\x77",
  [0x2b4f] = "\x02\x31\x4b", [0x2b50] = "\x01\x57\x2a",
  [0x2b51] = "\x02\x31\x4f", [0x2b52] = "\x02\x31\x4e",
  [0x2b53] = "\x02\x31\x4d", [0x2b54] = "\x02\x31\x49",
  [0x2b55] = "\x01\x57\x29", [0x2b56] = "\x01\x57\x2d",
  [0x2b57] = "\x01\x57\x30", [0x2b58] = "\x0e\x32\x6f",
  [0x2b59] = "\x0e\x32\x70", [0x2b5a] = "\x01\x57\x31",
  [0x2b5b] = "\x02\x31\x4a", [0x2b5c] = "\x02\x31\x4c",
  [0x2b5d] = "\x01\x57\x2f", [0x2b5e] = "\x01\x57\x2e",
  [0x2b5f] = "\x01\x57\x2c", [0x2b60] = "\x01\x57\x2b",
  [0x2b61] = "\x02\x37\x65", [0x2b62] = "\x0e\x32\x72",
  [0x2b63] = "\x02\x37\x63", [0x2b64] = "\x02\x37\x5f",
  [0x2b65] = "\x01\x5c\x40", [0x2b66] = "\x0e\x3d\x5e",
  [0x2b67] = "\x02\x37\x60", [0x2b68] = "\x01\x5c\x41",
  [0x2b69] = "\x02\x37\x61", [0x2b6a] = "\x02\x37\x62",
  [0x2b6b] = "\x02\x37\x64", [0x2b6c] = "\x0e\x37\x66",
  [0x2b6d] = "\x01\x5c\x42", [0x2b6e] = "\x0e\x37\x65",
  [0x2b70] = "\x02\x3e\x59", [0x2b71] = "\x0e\x3d\x5f",
  [0x2b72] = "\x02\x3e\x58", [0x2b73] = "\x02\x3e\x57",
  [0x2b74] = "\x02\x3e\x56", [0x2b75] = "\x0e\x3d\x64",
  [0x2b76] = "\x0e\x3d\x62", [0x2b79] = "\x02\x45\x5c",
  [0x2b7a] = "\x01\x65\x59", [0x2b7c] = "\x02\x45\x59",
  [0x2b7d] = "\x02\x45\x5b", [0x2b7e] = "\x0e\x43\x38",
  [0x2b7f] = "\x01\x65\x5a", [0x2b80] = "\x0e\x43\x3d",
  [0x2b81] = "\x01\x65\x5b", [0x2b82] = "\x02\x45\x5a",
  [0x2b83] = "\x0e\x43\x39", [0x2b84] = "\x0e\x3d\x63",
  [0x2b86] = "\x0e\x65\x78", [0x2b87] = "\x0e\x48\x48",
  [0x2b88] = "\x02\x4c\x48", [0x2b89] = "\x0e\x48\x47",
  [0x2b8a] = "\x02\x4c\x41", [0x2b8b] = "\x02\x4c\x42",
  [0x2b8d] = "\x01\x69\x7e", [0x2b8e] = "\x01\x69\x7c",
  [0x2b8f] = "\x01\x69\x7d", [0x2b90] = "\x02\x4c\x4a",
  [0x2b92] = "\x02\x4c\x49", [0x2b93] = "\x02\x4c\x46",
  [0x2b94] = "\x02\x4c\x45", [0x2b95] = "\x02\x4c\x44",
  [0x2b96] = "\x02\x4c\x43", [0x2b97] = "\x02\x4c\x47",
  [0x2b98] = "\x02\x4c\x40", [0x2b99] = "\x0e\x65\x79",
  [0x2b9a] = "\x02\x52\x53", [0x2b9b] = "\x02\x52\x58",
  [0x2b9c] = "\x02\x52\x56", [0x2b9d] = "\x0e\x4d\x50",
  [0x2b9e] = "\x0e\x4d\x51", [0x2b9f] = "\x0e\x4d\x4f",
  [0x2ba0] = "\x02\x52\x55", [0x2ba1] = "\x02\x52\x54",
  [0x2ba2] = "\x02\x52\x57", [0x2ba3] = "\x0e\x65\x7a",
  [0x2ba4] = "\x02\x58\x78", [0x2ba5] = "\x0e\x52\x26",
  [0x2ba6] = "\x01\x71\x57", [0x2ba7] = "\x01\x74\x52",
  [0x2ba8] = "\x02\x5d\x7b", [0x2ba9] = "\x0e\x52\x27",
  [0x2baa] = "\x01\x74\x53", [0x2bab] = "\x02\x5d\x7a",
  [0x2bac] = "\x02\x62\x4b", [0x2bad] = "\x02\x62\x4a",
  [0x2bae] = "\x01\x76\x71", [0x2baf] = "\x0e\x58\x45",
  [0x2bb0] = "\x02\x66\x2f", [0x2bb1] = "\x01\x78\x56",
  [0x2bb2] = "\x02\x69\x4e", [0x2bb3] = "\x01\x7b\x6b",
  [0x2bb4] = "\x02\x6d\x65", [0x2bb5] = "\x0e\x60\x3d",
  [0x2bb6] = "\x02\x70\x45", [0x2bb7] = "\x02\x70\x44",
  [0x2bb8] = "\x02\x21\x67", [0x2bb9] = "\x01\x52\x6d",
  [0x2bba] = "\x01\x52\x6e", [0x2bbb] = "\x02\x37\x66",
  [0x2bbc] = "\x0e\x65\x7b", [0x2bbd] = "\x01\x65\x5d",
  [0x2bbe] = "\x01\x46\x7d", [0x2bbf] = "\x01\x4b\x35",
  [0x2bc0] = "\x01\x4b\x34", [0x2bc1] = "\x01\x4b\x33",
  [0x2bc3] = "\x0e\x2a\x3e", [0x2bc4] = "\x0e\x2a\x3d",
  [0x2bc5] = "\x02\x28\x44", [0x2bc6] = "\x0e\x2a\x3f",
  [0x2bc7] = "\x0e\x2a\x42", [0x2bc8] = "\x01\x4f\x27",
  [0x2bc9] = "\x01\x4f\x26", [0x2bca] = "\x0e\x2a\x44",
  [0x2bcb] = "\x01\x52\x71", [0x2bcc] = "\x0e\x65\x4c",
  [0x2bcd] = "\x02\x2c\x5c", [0x2bce] = "\x02\x2c\x5f",
  [0x2bcf] = "\x02\x2c\x5d", [0x2bd0] = "\x0e\x2e\x42",
  [0x2bd1] = "\x01\x52\x6f", [0x2bd2] = "\x01\x52\x70",
  [0x2bd3] = "\x0e\x2e\x40", [0x2bd4] = "\x0e\x2e\x41",
  [0x2bd5] = "\x02\x2c\x5b", [0x2bd6] = "\x02\x2c\x5e",
  [0x2bd7] = "\x0e\x2e\x43", [0x2bd8] = "\x01\x57\x38",
  [0x2bd9] = "\x0e\x32\x74", [0x2bda] = "\x0e\x32\x73",
  [0x2bdb] = "\x0e\x32\x79", [0x2bdc] = "\x02\x31\x56",
  [0x2bdd] = "\x02\x31\x58", [0x2bde] = "\x02\x31\x57",
  [0x2bdf] = "\x01\x57\x35", [0x2be0] = "\x02\x31\x52",
  [0x2be1] = "\x0e\x32\x76", [0x2be2] = "\x0e\x32\x7b",
  [0x2be3] = "\x01\x57\x33", [0x2be4] = "\x01\x57\x32",
  [0x2be5] = "\x0e\x32\x77", [0x2be6] = "\x01\x57\x36",
  [0x2be7] = "\x01\x57\x34", [0x2be8] = "\x0e\x32\x78",
  [0x2be9] = "\x01\x57\x37", [0x2bea] = "\x02\x31\x55",
  [0x2beb] = "\x02\x31\x50", [0x2bec] = "\x02\x31\x51",
  [0x2bed] = "\x02\x31\x54", [0x2bee] = "\x02\x31\x53",
  [0x2bf0] = "\x0e\x32\x7c", [0x2bf1] = "\x0e\x37\x69",
  [0x2bf2] = "\x0e\x37\x67", [0x2bf3] = "\x0e\x37\x6b",
  [0x2bf5] = "\x0e\x37\x6a", [0x2bf6] = "\x02\x37\x69",
  [0x2bf7] = "\x02\x37\x6a", [0x2bf8] = "\x02\x37\x68",
  [0x2bf9] = "\x0e\x37\x6c", [0x2bfa] = "\x02\x37\x67",
  [0x2bfb] = "\x01\x5c\x43", [0x2bff] = "\x0e\x3d\x65",
  [0x2c00] = "\x01\x61\x25", [0x2c02] = "\x02\x3e\x5a",
  [0x2c03] = "\x02\x3e\x5c", [0x2c04] = "\x02\x3e\x5e",
  [0x2c05] = "\x01\x61\x24", [0x2c07] = "\x0e\x3d\x66",
  [0x2c08] = "\x01\x61\x22", [0x2c09] = "\x0e\x3d\x69",
  [0x2c0a] = "\x02\x3e\x5b", [0x2c0b] = "\x01\x61\x23",
  [0x2c0c] = "\x02\x3e\x5d", [0x2c0d] = "\x01\x61\x21",
  [0x2c0e] = "\x0e\x3d\x68", [0x2c0f] = "\x0e\x43\x41",
  [0x2c10] = "\x02\x45\x66", [0x2c11] = "\x02\x45\x5d",
  [0x2c12] = "\x02\x45\x60", [0x2c13] = "\x02\x45\x64",
  [0x2c14] = "\x01\x65\x61", [0x2c15] = "\x02\x45\x62",
  [0x2c17] = "\x02\x45\x61", [0x2c18] = "\x02\x45\x5e",
  [0x2c19] = "\x02\x45\x5f", [0x2c1a] = "\x01\x65\x5f",
  [0x2c1b] = "\x02\x45\x65", [0x2c1c] = "\x01\x65\x5e",
  [0x2c1d] = "\x0e\x43\x44", [0x2c1e] = "\x01\x65\x63",
  [0x2c1f] = "\x01\x65\x62", [0x2c20] = "\x01\x65\x60",
  [0x2c22] = "\x02\x45\x63", [0x2c25] = "\x0e\x65\x24",
  [0x2c26] = "\x02\x4c\x50", [0x2c28] = "\x02\x4c\x4f",
  [0x2c29] = "\x0e\x48\x4b", [0x2c2a] = "\x0e\x48\x4d",
  [0x2c2b] = "\x02\x4c\x4b", [0x2c2c] = "\x0e\x48\x50",
  [0x2c2d] = "\x0e\x48\x4f", [0x2c2e] = "\x01\x6a\x21",
  [0x2c2f] = "\x02\x4c\x4e", [0x2c30] = "\x02\x4c\x4d",
  [0x2c31] = "\x01\x6a\x22", [0x2c32] = "\x0e\x48\x52",
  [0x2c37] = "\x01\x6e\x29", [0x2c38] = "\x0e\x4d\x55",
  [0x2c39] = "\x02\x52\x5a", [0x2c3a] = "\x0e\x4d\x56",
  [0x2c3b] = "\x01\x6e\x2a", [0x2c3c] = "\x01\x6e\x26",
  [0x2c3d] = "\x01\x6e\x28", [0x2c3e] = "\x0e\x4d\x5a",
  [0x2c3f] = "\x01\x6e\x25", [0x2c40] = "\x01\x6e\x27",
  [0x2c41] = "\x0e\x4d\x53", [0x2c42] = "\x0e\x4d\x57",
  [0x2c44] = "\x02\x58\x79", [0x2c45] = "\x0e\x52\x2a",
  [0x2c46] = "\x01\x71\x5a", [0x2c47] = "\x02\x58\x7b",
  [0x2c48] = "\x02\x58\x7a", [0x2c49] = "\x0e\x55\x64",
  [0x2c4a] = "\x02\x4c\x4c", [0x2c4b] = "\x01\x71\x5c",
  [0x2c4c] = "\x01\x71\x5b", [0x2c4d] = "\x01\x71\x58",
  [0x2c4e] = "\x01\x71\x59", [0x2c4f] = "\x0e\x52\x2c",
  [0x2c53] = "\x0e\x52\x29", [0x2c54] = "\x02\x5e\x22",
  [0x2c56] = "\x02\x5d\x7e", [0x2c57] = "\x01\x74\x54",
  [0x2c58] = "\x02\x5e\x21", [0x2c5a] = "\x02\x5e\x23",
  [0x2c5b] = "\x02\x5d\x7d", [0x2c5c] = "\x02\x5d\x7c",
  [0x2c5f] = "\x02\x62\x4c", [0x2c60] = "\x01\x76\x74",
  [0x2c61] = "\x01\x76\x72", [0x2c62] = "\x01\x76\x73",
  [0x2c64] = "\x0e\x5a\x66", [0x2c66] = "\x0e\x5a\x65",
  [0x2c67] = "\x02\x66\x30", [0x2c68] = "\x02\x66\x31",
  [0x2c69] = "\x01\x78\x58", [0x2c6a] = "\x0e\x5a\x67",
  [0x2c6b] = "\x01\x78\x57", [0x2c6c] = "\x02\x69\x50",
  [0x2c6d] = "\x02\x69\x51", [0x2c6e] = "\x02\x69\x4f",
  [0x2c6f] = "\x0e\x5c\x64", [0x2c70] = "\x02\x6d\x66",
  [0x2c71] = "\x02\x6d\x67", [0x2c72] = "\x0e\x60\x76",
  [0x2c73] = "\x0e\x60\x77", [0x2c74] = "\x01\x46\x7e",
  [0x2c75] = "\x02\x22\x68", [0x2c76] = "\x01\x4b\x36",
  [0x2c78] = "\x02\x28\x45", [0x2c79] = "\x01\x4f\x29",
  [0x2c7a] = "\x01\x4f\x28", [0x2c7b] = "\x02\x28\x46",
  [0x2c7c] = "\x0e\x2e\x44", [0x2c7d] = "\x0e\x2e\x47",
  [0x2c7e] = "\x02\x2c\x61", [0x2c7f] = "\x01\x52\x72",
  [0x2c80] = "\x02\x2c\x60", [0x2c81] = "\x01\x52\x73",
  [0x2c82] = "\x0e\x2e\x49", [0x2c83] = "\x0e\x2e\x48",
  [0x2c84] = "\x01\x57\x39", [0x2c85] = "\x02\x31\x5b",
  [0x2c86] = "\x02\x31\x59", [0x2c87] = "\x02\x31\x5f",
  [0x2c88] = "\x01\x57\x3a", [0x2c89] = "\x02\x31\x5a",
  [0x2c8a] = "\x02\x31\x5e", [0x2c8b] = "\x02\x31\x5c",
  [0x2c8c] = "\x02\x31\x5d", [0x2c8f] = "\x02\x37\x6b",
  [0x2c90] = "\x02\x37\x6d", [0x2c91] = "\x0e\x37\x72",
  [0x2c92] = "\x01\x5c\x44", [0x2c93] = "\x0e\x37\x71",
  [0x2c94] = "\x02\x37\x6c", [0x2c95] = "\x01\x5c\x45",
  [0x2c96] = "\x01\x61\x28", [0x2c97] = "\x01\x61\x27",
  [0x2c98] = "\x01\x61\x26", [0x2c99] = "\x02\x3e\x5f",
  [0x2c9a] = "\x0e\x37\x70", [0x2c9e] = "\x02\x45\x69",
  [0x2c9f] = "\x01\x65\x64", [0x2ca0] = "\x01\x65\x65",
  [0x2ca1] = "\x0e\x43\x46", [0x2ca2] = "\x02\x45\x68",
  [0x2ca3] = "\x02\x45\x67", [0x2ca4] = "\x0e\x43\x47",
  [0x2ca7] = "\x0e\x65\x7c", [0x2ca8] = "\x02\x4c\x51",
  [0x2ca9] = "\x01\x6a\x24", [0x2caa] = "\x01\x6a\x23",
  [0x2cab] = "\x02\x4c\x52", [0x2cac] = "\x02\x4c\x53",
  [0x2cae] = "\x01\x6e\x2c", [0x2caf] = "\x01\x6e\x2b",
  [0x2cb0] = "\x0e\x4d\x5b", [0x2cb1] = "\x02\x59\x21",
  [0x2cb2] = "\x02\x52\x5b", [0x2cb3] = "\x02\x52\x5d",
  [0x2cb4] = "\x02\x52\x5c", [0x2cb5] = "\x02\x58\x7e",
  [0x2cb6] = "\x02\x58\x7c", [0x2cb7] = "\x02\x59\x22",
  [0x2cb8] = "\x02\x58\x7d", [0x2cba] = "\x01\x71\x5d",
  [0x2cbb] = "\x0e\x52\x2f", [0x2cbd] = "\x0e\x52\x2e",
  [0x2cbe] = "\x02\x5e\x24", [0x2cbf] = "\x01\x74\x55",
  [0x2cc0] = "\x02\x5e\x25", [0x2cc1] = "\x02\x5e\x26",
  [0x2cc2] = "\x0e\x55\x68", [0x2cc4] = "\x01\x76\x75",
  [0x2cc5] = "\x01\x76\x76", [0x2cc7] = "\x01\x7a\x22",
  [0x2cc8] = "\x0e\x5e\x49", [0x2cc9] = "\x0e\x5e\x48",
  [0x2cca] = "\x01\x7c\x3d", [0x2ccb] = "\x01\x47\x21",
  [0x2ccc] = "\x0e\x26\x7e", [0x2ccd] = "\x0e\x27\x21",
  [0x2ccf] = "\x0e\x2a\x47", [0x2cd0] = "\x0e\x2e\x4e",
  [0x2cd1] = "\x02\x2c\x62", [0x2cd2] = "\x0e\x2e\x4f",
  [0x2cd3] = "\x0e\x2e\x4d", [0x2cd4] = "\x0e\x2e\x4b",
  [0x2cd5] = "\x0e\x2e\x4c", [0x2cd8] = "\x02\x31\x60",
  [0x2cd9] = "\x01\x57\x3b", [0x2cda] = "\x0e\x33\x23",
  [0x2cdb] = "\x0e\x33\x26", [0x2cdc] = "\x0e\x33\x27",
  [0x2cdd] = "\x0e\x33\x24", [0x2cdf] = "\x01\x5d\x7e",
  [0x2ce0] = "\x01\x5d\x7d", [0x2ce1] = "\x0e\x37\x73",
  [0x2ce2] = "\x0e\x3d\x71", [0x2ce3] = "\x01\x61\x2a",
  [0x2ce4] = "\x02\x3e\x61", [0x2ce5] = "\x01\x61\x29",
  [0x2ce6] = "\x02\x3e\x60", [0x2ce7] = "\x0e\x3d\x70",
  [0x2ce9] = "\x0e\x43\x49", [0x2cea] = "\x0e\x43\x4a",
  [0x2ceb] = "\x02\x45\x6a", [0x2cec] = "\x0e\x48\x54",
  [0x2ced] = "\x01\x6a\x25", [0x2cee] = "\x02\x4c\x54",
  [0x2cef] = "\x01\x6a\x26", [0x2cf0] = "\x0e\x48\x55",
  [0x2cf1] = "\x0e\x52\x30", [0x2cf2] = "\x0e\x55\x69",
  [0x2cf3] = "\x0e\x55\x6a", [0x2cf5] = "\x0e\x58\x49",
  [0x2cf6] = "\x01\x7a\x23", [0x2cf7] = "\x02\x69\x52",
  [0x2cf8] = "\x0e\x5f\x55", [0x2cf9] = "\x01\x48\x4c",
  [0x2cfa] = "\x01\x4f\x2a", [0x2cfb] = "\x02\x28\x47",
  [0x2cfc] = "\x0e\x2e\x51", [0x2cfd] = "\x01\x52\x75",
  [0x2cfe] = "\x0e\x2e\x50", [0x2cff] = "\x01\x52\x74",
  [0x2d00] = "\x02\x2c\x63", [0x2d01] = "\x02\x2c\x64",
  [0x2d04] = "\x02\x31\x62", [0x2d05] = "\x02\x31\x64",
  [0x2d06] = "\x01\x57\x3c", [0x2d08] = "\x02\x31\x66",
  [0x2d09] = "\x02\x31\x69", [0x2d0a] = "\x02\x31\x67",
  [0x2d0b] = "\x0e\x33\x2b", [0x2d0d] = "\x0e\x33\x2a",
  [0x2d0e] = "\x02\x31\x68", [0x2d0f] = "\x02\x31\x65",
  [0x2d10] = "\x02\x31\x61", [0x2d11] = "\x01\x57\x3d",
  [0x2d12] = "\x02\x31\x6a", [0x2d13] = "\x02\x31\x63",
  [0x2d14] = "\x0e\x33\x2c", [0x2d16] = "\x0e\x37\x78",
  [0x2d17] = "\x0e\x37\x79", [0x2d18] = "\x02\x37\x76",
  [0x2d19] = "\x01\x5c\x4b", [0x2d1a] = "\x02\x38\x21",
  [0x2d1b] = "\x01\x5c\x48", [0x2d1c] = "\x0e\x37\x77",
  [0x2d1d] = "\x02\x37\x78", [0x2d1e] = "\x01\x5c\x4c",
  [0x2d1f] = "\x0e\x37\x7b", [0x2d20] = "\x01\x5c\x46",
  [0x2d21] = "\x0e\x37\x76", [0x2d22] = "\x02\x37\x73",
  [0x2d23] = "\x02\x38\x22", [0x2d24] = "\x02\x37\x74",
  [0x2d25] = "\x02\x37\x71", [0x2d26] = "\x01\x5c\x4a",
  [0x2d28] = "\x01\x5c\x47", [0x2d29] = "\x0e\x63\x2d",
  [0x2d2a] = "\x02\x37\x77", [0x2d2b] = "\x02\x37\x7a",
  [0x2d2c] = "\x01\x5c\x49", [0x2d2d] = "\x02\x37\x7b",
  [0x2d2e] = "\x01\x5c\x4d", [0x2d2f] = "\x02\x37\x7c",
  [0x2d30] = "\x02\x37\x72", [0x2d31] = "\x02\x37\x79",
  [0x2d32] = "\x02\x37\x7d", [0x2d33] = "\x02\x37\x75",
  [0x2d34] = "\x02\x37\x70", [0x2d35] = "\x02\x37\x6e",
  [0x2d36] = "\x0e\x37\x7a", [0x2d37] = "\x0e\x65\x7d",
  [0x2d38] = "\x02\x37\x7e", [0x2d3b] = "\x02\x37\x6f",
  [0x2d40] = "\x02\x3e\x68", [0x2d41] = "\x0e\x3d\x7b",
  [0x2d42] = "\x0e\x3d\x78", [0x2d43] = "\x0e\x3d\x75",
  [0x2d44] = "\x02\x3e\x64", [0x2d45] = "\x02\x3e\x6a",
  [0x2d46] = "\x01\x61\x2d", [0x2d47] = "\x02\x3e\x63",
  [0x2d48] = "\x02\x3e\x65", [0x2d49] = "\x01\x61\x2b",
  [0x2d4a] = "\x02\x3e\x62", [0x2d4b] = "\x01\x61\x32",
  [0x2d4c] = "\x02\x3e\x66", [0x2d4d] = "\x01\x61\x31",
  [0x2d4e] = "\x02\x3e\x67", [0x2d4f] = "\x01\x61\x33",
  [0x2d50] = "\x01\x61\x2e", [0x2d51] = "\x01\x61\x34",
  [0x2d52] = "\x01\x61\x2f", [0x2d53] = "\x0e\x3d\x79",
  [0x2d54] = "\x01\x61\x30", [0x2d55] = "\x0e\x3d\x72",
  [0x2d56] = "\x01\x61\x2c", [0x2d57] = "\x0e\x3d\x76",
  [0x2d58] = "\x02\x3e\x69", [0x2d59] = "\x0e\x65\x7e",
  [0x2d5e] = "\x0e\x43\x4f", [0x2d60] = "\x01\x65\x68",
  [0x2d61] = "\x02\x45\x75", [0x2d62] = "\x0e\x43\x4c",
  [0x2d63] = "\x02\x45\x78", [0x2d64] = "\x02\x45\x6c",
  [0x2d65] = "\x02\x45\x71", [0x2d66] = "\x02\x45\x6b",
  [0x2d67] = "\x01\x65\x6a", [0x2d68] = "\x0e\x43\x55",
  [0x2d69] = "\x02\x45\x6f", [0x2d6b] = "\x0e\x43\x4d",
  [0x2d6c] = "\x0e\x3d\x77", [0x2d6d] = "\x02\x45\x6d",
  [0x2d6e] = "\x01\x65\x69", [0x2d6f] = "\x0e\x43\x54",
  [0x2d70] = "\x02\x45\x74", [0x2d71] = "\x02\x45\x73",
  [0x2d72] = "\x02\x45\x70", [0x2d73] = "\x02\x45\x72",
  [0x2d74] = "\x02\x45\x6e", [0x2d75] = "\x01\x6a\x2a",
  [0x2d76] = "\x02\x45\x77", [0x2d77] = "\x01\x65\x66",
  [0x2d78] = "\x02\x45\x76", [0x2d79] = "\x0e\x43\x51",
  [0x2d81] = "\x0e\x48\x56", [0x2d82] = "\x02\x4c\x64",
  [0x2d83] = "\x0e\x48\x58", [0x2d84] = "\x01\x6a\x31",
  [0x2d85] = "\x02\x4c\x5f", [0x2d86] = "\x0e\x48\x60",
  [0x2d87] = "\x01\x6a\x30", [0x2d88] = "\x02\x4c\x55",
  [0x2d8a] = "\x02\x4c\x57", [0x2d8b] = "\x01\x6a\x29",
  [0x2d8c] = "\x02\x4c\x5c", [0x2d8d] = "\x02\x4c\x5b",
  [0x2d8e] = "\x02\x4c\x5e", [0x2d8f] = "\x01\x6a\x2e",
  [0x2d90] = "\x02\x4c\x59", [0x2d91] = "\x02\x4c\x58",
  [0x2d92] = "\x0e\x48\x5a", [0x2d93] = "\x0e\x48\x5e",
  [0x2d94] = "\x01\x6a\x2d", [0x2d95] = "\x01\x6a\x28",
  [0x2d96] = "\x02\x4c\x5a", [0x2d97] = "\x01\x6a\x2b",
  [0x2d98] = "\x02\x4c\x60", [0x2d99] = "\x02\x4c\x62",
  [0x2d9b] = "\x02\x4c\x5d", [0x2d9c] = "\x02\x4c\x56",
  [0x2d9d] = "\x01\x6a\x2c", [0x2d9e] = "\x0e\x48\x57",
  [0x2d9f] = "\x0e\x48\x5d", [0x2da0] = "\x01\x6e\x34",
  [0x2da1] = "\x01\x6a\x27", [0x2da3] = "\x0e\x48\x5c",
  [0x2da4] = "\x02\x4c\x63", [0x2dac] = "\x02\x52\x61",
  [0x2dad] = "\x01\x6e\x2d", [0x2dae] = "\x0e\x4d\x5c",
  [0x2daf] = "\x02\x52\x63", [0x2db1] = "\x01\x6e\x2e",
  [0x2db2] = "\x0e\x4d\x65", [0x2db3] = "\x0e\x4d\x5d",
  [0x2db4] = "\x01\x6e\x30", [0x2db5] = "\x02\x52\x66",
  [0x2db7] = "\x02\x52\x5e", [0x2db8] = "\x01\x6a\x2f",
  [0x2db9] = "\x02\x52\x64", [0x2dbe] = "\x02\x52\x60",
  [0x2dbf] = "\x0e\x4d\x60", [0x2dc0] = "\x01\x65\x67",
  [0x2dc1] = "\x01\x6e\x33", [0x2dc2] = "\x0e\x63\x67",
  [0x2dc3] = "\x0e\x63\x66", [0x2dc4] = "\x01\x6e\x2f",
  [0x2dc6] = "\x01\x6e\x31", [0x2dc7] = "\x01\x6e\x32",
  [0x2dc9] = "\x01\x71\x60", [0x2dca] = "\x02\x52\x65",
  [0x2dcb] = "\x02\x52\x5f", [0x2dcc] = "\x01\x6e\x35",
  [0x2dcd] = "\x0e\x4d\x63", [0x2dce] = "\x02\x52\x62",
  [0x2dd0] = "\x0e\x66\x21", [0x2dd4] = "\x02\x59\x2c",
  [0x2dd5] = "\x02\x59\x27", [0x2dd8] = "\x02\x59\x31",
  [0x2dd9] = "\x01\x71\x5e", [0x2dda] = "\x02\x59\x29",
  [0x2ddb] = "\x01\x71\x62", [0x2ddc] = "\x02\x59\x2f",
  [0x2ddd] = "\x02\x59\x26", [0x2dde] = "\x02\x59\x23",
  [0x2ddf] = "\x02\x59\x32", [0x2de0] = "\x01\x74\x5b",
  [0x2de1] = "\x01\x71\x63", [0x2de2] = "\x02\x59\x2e",
  [0x2de3] = "\x02\x59\x24", [0x2de4] = "\x01\x71\x61",
  [0x2de5] = "\x02\x59\x28", [0x2de6] = "\x01\x71\x65",
  [0x2de7] = "\x02\x59\x25", [0x2de8] = "\x02\x59\x2a",
  [0x2de9] = "\x01\x71\x64", [0x2dea] = "\x02\x59\x2d",
  [0x2deb] = "\x02\x59\x30", [0x2dec] = "\x0e\x52\x33",
  [0x2ded] = "\x0e\x52\x34", [0x2df0] = "\x02\x5e\x38",
  [0x2df1] = "\x02\x5e\x39", [0x2df2] = "\x02\x5e\x29",
  [0x2df3] = "\x02\x5e\x30", [0x2df4] = "\x02\x5e\x2e",
  [0x2df6] = "\x0e\x55\x6d", [0x2df7] = "\x01\x74\x59",
  [0x2df8] = "\x02\x5e\x35", [0x2df9] = "\x02\x59\x2b",
  [0x2dfa] = "\x0e\x55\x6b", [0x2dfb] = "\x02\x5e\x2c",
  [0x2dfc] = "\x0e\x55\x6e", [0x2dfd] = "\x02\x5e\x36",
  [0x2dfe] = "\x01\x74\x58", [0x2dff] = "\x02\x5e\x2b",
  [0x2e00] = "\x02\x5e\x2a", [0x2e01] = "\x02\x5e\x34",
  [0x2e02] = "\x02\x5e\x31", [0x2e03] = "\x02\x5e\x33",
  [0x2e05] = "\x02\x5e\x27", [0x2e06] = "\x02\x5e\x37",
  [0x2e07] = "\x01\x74\x56", [0x2e08] = "\x0e\x55\x70",
  [0x2e09] = "\x02\x5e\x32", [0x2e0a] = "\x02\x5e\x3b",
  [0x2e0b] = "\x02\x5e\x2f", [0x2e0c] = "\x01\x74\x5a",
  [0x2e0d] = "\x01\x74\x57", [0x2e0e] = "\x02\x5e\x2d",
  [0x2e0f] = "\x02\x5e\x28", [0x2e10] = "\x02\x5e\x3a",
  [0x2e11] = "\x01\x71\x5f", [0x2e12] = "\x0e\x55\x71",
  [0x2e18] = "\x0e\x55\x6f", [0x2e19] = "\x02\x62\x4f",
  [0x2e1a] = "\x0e\x58\x50", [0x2e1b] = "\x0e\x58\x4e",
  [0x2e1c] = "\x02\x62\x4d", [0x2e1d] = "\x02\x62\x53",
  [0x2e1e] = "\x01\x76\x7a", [0x2e1f] = "\x02\x62\x51",
  [0x2e20] = "\x02\x62\x50", [0x2e21] = "\x01\x76\x7c",
  [0x2e22] = "\x02\x62\x56", [0x2e23] = "\x01\x76\x7b",
  [0x2e24] = "\x0e\x58\x51", [0x2e25] = "\x02\x62\x57",
  [0x2e26] = "\x02\x62\x54", [0x2e27] = "\x01\x76\x78",
  [0x2e28] = "\x02\x62\x55", [0x2e29] = "\x02\x62\x4e",
  [0x2e2a] = "\x01\x76\x79", [0x2e2b] = "\x01\x76\x77",
  [0x2e2c] = "\x02\x66\x35", [0x2e2d] = "\x02\x62\x52",
  [0x2e2e] = "\x0e\x58\x4c", [0x2e2f] = "\x0e\x66\x22",
  [0x2e30] = "\x02\x62\x58", [0x2e32] = "\x0e\x66\x23",
  [0x2e33] = "\x02\x66\x32", [0x2e34] = "\x0e\x5a\x6f",
  [0x2e35] = "\x0e\x5a\x6e", [0x2e37] = "\x01\x78\x5d",
  [0x2e38] = "\x01\x78\x5b", [0x2e39] = "\x02\x66\x34",
  [0x2e3b] = "\x02\x66\x36", [0x2e3c] = "\x02\x66\x33",
  [0x2e3d] = "\x01\x78\x5c", [0x2e3e] = "\x01\x78\x59",
  [0x2e3f] = "\x01\x78\x5a", [0x2e40] = "\x01\x78\x5e",
  [0x2e42] = "\x0e\x66\x24", [0x2e43] = "\x01\x7a\x25",
  [0x2e44] = "\x0e\x5c\x66", [0x2e45] = "\x02\x69\x57",
  [0x2e47] = "\x02\x69\x56", [0x2e48] = "\x02\x69\x54",
  [0x2e49] = "\x02\x69\x53", [0x2e4a] = "\x02\x69\x55",
  [0x2e4b] = "\x0e\x5c\x65", [0x2e4c] = "\x01\x7a\x24",
  [0x2e4d] = "\x01\x7a\x26", [0x2e4e] = "\x0e\x66\x25",
  [0x2e50] = "\x01\x7a\x7b", [0x2e51] = "\x0e\x5e\x4c",
  [0x2e53] = "\x02\x6b\x73", [0x2e54] = "\x02\x6b\x72",
  [0x2e56] = "\x0e\x5e\x4b", [0x2e57] = "\x02\x6d\x68",
  [0x2e58] = "\x0e\x5f\x58", [0x2e59] = "\x02\x6d\x6a",
  [0x2e5a] = "\x02\x6d\x6c", [0x2e5b] = "\x02\x6d\x6b",
  [0x2e5c] = "\x02\x6d\x69", [0x2e5d] = "\x0e\x5f\x57",
  [0x2e5e] = "\x0e\x60\x3f", [0x2e5f] = "\x01\x7b\x6d",
  [0x2e60] = "\x01\x7b\x6c", [0x2e62] = "\x0e\x60\x3e",
  [0x2e63] = "\x01\x7c\x3f", [0x2e64] = "\x01\x7c\x3e",
  [0x2e65] = "\x01\x7c\x40", [0x2e66] = "\x02\x6f\x3c",
  [0x2e67] = "\x02\x6f\x3b", [0x2e68] = "\x0e\x66\x26",
  [0x2e69] = "\x02\x71\x2b", [0x2e6a] = "\x02\x70\x46",
  [0x2e6b] = "\x02\x71\x2c", [0x2e6c] = "\x01\x7d\x25",
  [0x2e6d] = "\x0e\x61\x44", [0x2e6e] = "\x01\x7d\x26",
  [0x2e6f] = "\x02\x71\x5c", [0x2e71] = "\x0e\x62\x3a",
  [0x2e72] = "\x01\x7d\x4b", [0x2e73] = "\x01\x48\x4d",
  [0x2e74] = "\x0e\x2a\x49", [0x2e75] = "\x02\x28\x48",
  [0x2e76] = "\x0e\x2a\x48", [0x2e77] = "\x0e\x2e\x52",
  [0x2e78] = "\x02\x2c\x66", [0x2e79] = "\x02\x2c\x67",
  [0x2e7a] = "\x02\x2c\x65", [0x2e7b] = "\x0e\x2e\x54",
  [0x2e7c] = "\x0e\x2e\x53", [0x2e7d] = "\x01\x52\x76",
  [0x2e7f] = "\x02\x2c\x68", [0x2e80] = "\x02\x2c\x69",
  [0x2e81] = "\x02\x2c\x6a", [0x2e83] = "\x0e\x33\x31",
  [0x2e84] = "\x02\x31\x6b", [0x2e85] = "\x02\x31\x71",
  [0x2e86] = "\x0e\x33\x2e", [0x2e87] = "\x0e\x33\x30",
  [0x2e88] = "\x02\x31\x6f", [0x2e89] = "\x01\x57\x3e",
  [0x2e8a] = "\x02\x31\x6d", [0x2e8b] = "\x0e\x33\x32",
  [0x2e8c] = "\x02\x31\x6e", [0x2e8d] = "\x02\x31\x70",
  [0x2e8e] = "\x0e\x33\x2d", [0x2e91] = "\x02\x31\x6c",
  [0x2e92] = "\x01\x5c\x4e", [0x2e93] = "\x0e\x37\x7d",
  [0x2e94] = "\x02\x38\x23", [0x2e95] = "\x01\x5c\x50",
  [0x2e96] = "\x02\x38\x25", [0x2e97] = "\x01\x5c\x4f",
  [0x2e98] = "\x02\x38\x24", [0x2e99] = "\x0e\x37\x7e",
  [0x2e9a] = "\x0e\x38\x21", [0x2e9b] = "\x0e\x38\x3e",
  [0x2e9c] = "\x0e\x38\x22", [0x2e9e] = "\x02\x3e\x6c",
  [0x2e9f] = "\x01\x61\x35", [0x2ea0] = "\x0e\x3e\x21",
  [0x2ea1] = "\x02\x3e\x6e", [0x2ea2] = "\x02\x3e\x6b",
  [0x2ea3] = "\x02\x38\x26", [0x2ea4] = "\x0e\x3e\x23",
  [0x2ea5] = "\x01\x61\x36", [0x2ea6] = "\x0e\x3e\x22",
  [0x2ea7] = "\x0e\x3d\x7e", [0x2ea8] = "\x02\x3e\x6d",
  [0x2ea9] = "\x0e\x66\x27", [0x2eae] = "\x0e\x43\x58",
  [0x2eaf] = "\x02\x45\x7b", [0x2eb0] = "\x0e\x43\x5c",
  [0x2eb1] = "\x01\x65\x6b", [0x2eb2] = "\x02\x45\x79",
  [0x2eb3] = "\x01\x65\x6c", [0x2eb4] = "\x02\x45\x7a",
  [0x2eb5] = "\x01\x65\x6d", [0x2eb7] = "\x0e\x48\x62",
  [0x2eb8] = "\x0e\x48\x61", [0x2eb9] = "\x01\x6a\x32",
  [0x2eba] = "\x02\x4c\x68", [0x2ebb] = "\x02\x4c\x65",
  [0x2ebc] = "\x02\x4c\x67", [0x2ebd] = "\x01\x6a\x33",
  [0x2ebe] = "\x01\x6a\x34", [0x2ebf] = "\x02\x4c\x66",
  [0x2ec2] = "\x0e\x4d\x6f", [0x2ec3] = "\x0e\x4d\x71",
  [0x2ec4] = "\x0e\x4d\x6d", [0x2ec5] = "\x02\x52\x67",
  [0x2ec6] = "\x0e\x4d\x70", [0x2ec7] = "\x0e\x4d\x73",
  [0x2ec8] = "\x02\x52\x68", [0x2ec9] = "\x0e\x4d\x72",
  [0x2eca] = "\x01\x6e\x36", [0x2ecb] = "\x02\x52\x6a",
  [0x2ecc] = "\x02\x52\x69", [0x2ecd] = "\x0e\x4d\x6e",
  [0x2ece] = "\x01\x23\x23", [0x2ed0] = "\x02\x59\x36",
  [0x2ed1] = "\x02\x59\x37", [0x2ed2] = "\x02\x59\x33",
  [0x2ed3] = "\x0e\x52\x37", [0x2ed4] = "\x02\x59\x34",
  [0x2ed5] = "\x01\x71\x66", [0x2ed6] = "\x01\x71\x67",
  [0x2ed7] = "\x02\x59\x35", [0x2ed9] = "\x01\x74\x61",
  [0x2eda] = "\x0e\x55\x75", [0x2edb] = "\x0e\x55\x74",
  [0x2edc] = "\x01\x74\x5d", [0x2edd] = "\x01\x74\x62",
  [0x2ede] = "\x01\x74\x5e", [0x2edf] = "\x01\x74\x60",
  [0x2ee0] = "\x01\x74\x5c", [0x2ee1] = "\x0e\x52\x39",
  [0x2ee2] = "\x01\x74\x5f", [0x2ee3] = "\x0e\x58\x55",
  [0x2ee6] = "\x0e\x58\x54", [0x2ee7] = "\x01\x76\x7d",
  [0x2ee8] = "\x02\x5e\x3c", [0x2eea] = "\x02\x66\x38",
  [0x2eec] = "\x02\x66\x37", [0x2eed] = "\x0e\x66\x28",
  [0x2eee] = "\x02\x69\x58", [0x2eef] = "\x01\x7a\x27",
  [0x2ef0] = "\x01\x7a\x28", [0x2ef1] = "\x02\x6d\x6e",
  [0x2ef2] = "\x02\x6b\x74", [0x2ef3] = "\x0e\x61\x69",
  [0x2ef4] = "\x02\x6d\x6d", [0x2ef5] = "\x0e\x60\x40",
  [0x2ef6] = "\x02\x71\x2d", [0x2ef7] = "\x02\x71\x72",
  [0x2ef8] = "\x01\x48\x4e", [0x2efa] = "\x0e\x27\x22",
  [0x2efb] = "\x01\x4b\x37", [0x2efc] = "\x0e\x2a\x4b",
  [0x2efd] = "\x02\x28\x49", [0x2efe] = "\x01\x4f\x2b",
  [0x2eff] = "\x0e\x2a\x4a", [0x2f00] = "\x01\x52\x79",
  [0x2f01] = "\x02\x2c\x6d", [0x2f02] = "\x01\x52\x77",
  [0x2f03] = "\x02\x2c\x6b", [0x2f04] = "\x01\x52\x7c",
  [0x2f05] = "\x01\x52\x78", [0x2f06] = "\x01\x52\x7d",
  [0x2f07] = "\x01\x52\x7b", [0x2f08] = "\x02\x2c\x6c",
  [0x2f09] = "\x01\x52\x7a", [0x2f0a] = "\x01\x57\x42",
  [0x2f0b] = "\x01\x57\x41", [0x2f0c] = "\x02\x31\x7c",
  [0x2f0d] = "\x01\x57\x4a", [0x2f0e] = "\x02\x31\x75",
  [0x2f0f] = "\x02\x31\x7b", [0x2f10] = "\x01\x57\x46",
  [0x2f11] = "\x02\x31\x74", [0x2f12] = "\x02\x31\x7a",
  [0x2f13] = "\x02\x31\x78", [0x2f14] = "\x01\x57\x45",
  [0x2f15] = "\x01\x57\x47", [0x2f16] = "\x02\x31\x77",
  [0x2f17] = "\x01\x57\x40", [0x2f18] = "\x02\x31\x76",
  [0x2f19] = "\x01\x57\x4b", [0x2f1a] = "\x01\x57\x48",
  [0x2f1b] = "\x01\x57\x4c", [0x2f1c] = "\x01\x57\x49",
  [0x2f1d] = "\x02\x31\x73", [0x2f1e] = "\x02\x31\x72",
  [0x2f1f] = "\x02\x31\x79", [0x2f20] = "\x01\x57\x43",
  [0x2f21] = "\x01\x57\x3f", [0x2f22] = "\x01\x57\x44",
  [0x2f24] = "\x0e\x33\x37", [0x2f25] = "\x0e\x33\x3b",
  [0x2f28] = "\x02\x38\x35", [0x2f29] = "\x02\x38\x2e",
  [0x2f2b] = "\x01\x61\x3b", [0x2f2c] = "\x02\x38\x2d",
  [0x2f2d] = "\x0e\x38\x29", [0x2f2e] = "\x01\x5c\x54",
  [0x2f2f] = "\x01\x5c\x5b", [0x2f30] = "\x01\x5c\x58",
  [0x2f31] = "\x01\x5c\x5e", [0x2f32] = "\x01\x5c\x5d",
  [0x2f33] = "\x01\x5c\x59", [0x2f34] = "\x0e\x38\x26",
  [0x2f35] = "\x02\x38\x27", [0x2f36] = "\x02\x38\x2a",
  [0x2f37] = "\x0e\x38\x27", [0x2f38] = "\x02\x38\x29",
  [0x2f39] = "\x01\x5c\x55", [0x2f3a] = "\x02\x38\x2b",
  [0x2f3b] = "\x02\x38\x34", [0x2f3c] = "\x01\x5c\x56",
  [0x2f3d] = "\x02\x38\x28", [0x2f3e] = "\x02\x38\x31",
  [0x2f3f] = "\x02\x38\x32", [0x2f40] = "\x01\x5c\x57",
  [0x2f41] = "\x02\x38\x2f", [0x2f42] = "\x01\x5c\x5c",
  [0x2f43] = "\x01\x5c\x52", [0x2f44] = "\x01\x5c\x5a",
  [0x2f45] = "\x02\x38\x2c", [0x2f46] = "\x01\x5c\x51",
  [0x2f47] = "\x02\x38\x30", [0x2f48] = "\x0e\x38\x2e",
  [0x2f49] = "\x0e\x38\x24", [0x2f4a] = "\x02\x38\x33",
  [0x2f4c] = "\x0e\x38\x2b", [0x2f4d] = "\x0e\x3e\x34",
  [0x2f4e] = "\x02\x3f\x21", [0x2f4f] = "\x02\x3e\x76",
  [0x2f50] = "\x01\x61\x38", [0x2f51] = "\x02\x3e\x7d",
  [0x2f52] = "\x02\x3e\x7a", [0x2f53] = "\x02\x3e\x72",
  [0x2f54] = "\x02\x3e\x7b", [0x2f55] = "\x01\x61\x3a",
  [0x2f56] = "\x02\x3e\x73", [0x2f57] = "\x0e\x3e\x29",
  [0x2f58] = "\x02\x3e\x6f", [0x2f59] = "\x0e\x3e\x26",
  [0x2f5a] = "\x0e\x3e\x2e", [0x2f5b] = "\x01\x65\x73",
  [0x2f5c] = "\x02\x3e\x78", [0x2f5d] = "\x0e\x3e\x2f",
  [0x2f5e] = "\x01\x61\x37", [0x2f5f] = "\x02\x3e\x7e",
  [0x2f60] = "\x0e\x3e\x28", [0x2f61] = "\x01\x61\x3e",
  [0x2f62] = "\x01\x61\x40", [0x2f63] = "\x02\x3e\x71",
  [0x2f64] = "\x0e\x3e\x2c", [0x2f66] = "\x01\x61\x3f",
  [0x2f67] = "\x02\x3e\x74", [0x2f68] = "\x01\x61\x39",
  [0x2f69] = "\x02\x3e\x7c", [0x2f6a] = "\x02\x3e\x75",
  [0x2f6b] = "\x02\x3e\x79", [0x2f6c] = "\x0e\x3e\x2a",
  [0x2f6d] = "\x02\x3e\x77", [0x2f6e] = "\x01\x61\x3c",
  [0x2f6f] = "\x02\x3e\x70", [0x2f70] = "\x01\x61\x41",
  [0x2f71] = "\x01\x5c\x53", [0x2f72] = "\x01\x61\x3d",
  [0x2f73] = "\x01\x61\x42", [0x2f74] = "\x0e\x3e\x24",
  [0x2f75] = "\x0e\x3e\x35", [0x2f76] = "\x0e\x3e\x33",
  [0x2f78] = "\x0e\x43\x67", [0x2f79] = "\x01\x65\x6f",
  [0x2f7a] = "\x02\x46\x24", [0x2f7b] = "\x02\x46\x26",
  [0x2f7c] = "\x02\x46\x28", [0x2f7d] = "\x02\x46\x2c",
  [0x2f7e] = "\x0e\x3e\x2b", [0x2f7f] = "\x02\x46\x22",
  [0x2f80] = "\x02\x45\x7e", [0x2f81] = "\x01\x65\x71",
  [0x2f83] = "\x02\x46\x27", [0x2f84] = "\x02\x46\x2b",
  [0x2f85] = "\x02\x46\x23", [0x2f86] = "\x02\x45\x7d",
  [0x2f87] = "\x0e\x43\x66", [0x2f88] = "\x02\x45\x7c",
  [0x2f89] = "\x0e\x43\x69", [0x2f8a] = "\x0e\x43\x60",
  [0x2f8b] = "\x0e\x43\x62", [0x2f8c] = "\x02\x46\x29",
  [0x2f8d] = "\x02\x46\x21", [0x2f8e] = "\x02\x46\x25",
  [0x2f8f] = "\x01\x65\x72", [0x2f90] = "\x0e\x43\x5e",
  [0x2f91] = "\x01\x65\x70", [0x2f92] = "\x02\x46\x2d",
  [0x2f93] = "\x01\x65\x6e", [0x2f94] = "\x02\x46\x2a",
  [0x2f95] = "\x0e\x43\x64", [0x2f96] = "\x02\x4c\x79",
  [0x2f97] = "\x0e\x43\x68", [0x2f98] = "\x0e\x43\x6a",
  [0x2f99] = "\x0e\x43\x63", [0x2f9c] = "\x01\x6a\x37",
  [0x2f9d] = "\x02\x4c\x71", [0x2f9e] = "\x01\x6e\x43",
  [0x2f9f] = "\x02\x4c\x7b", [0x2fa0] = "\x01\x6a\x3a",
  [0x2fa1] = "\x02\x4d\x21", [0x2fa2] = "\x01\x6a\x40",
  [0x2fa3] = "\x02\x4c\x6c", [0x2fa4] = "\x0e\x48\x74",
  [0x2fa6] = "\x02\x4c\x7c", [0x2fa7] = "\x02\x4c\x69",
  [0x2fa8] = "\x0e\x48\x6d", [0x2fa9] = "\x02\x4c\x7e",
  [0x2faa] = "\x02\x4c\x6d", [0x2fab] = "\x0e\x48\x6e",
  [0x2fac] = "\x01\x6a\x47", [0x2fad] = "\x01\x6a\x44",
  [0x2fae] = "\x02\x4c\x7d", [0x2faf] = "\x02\x4c\x77",
  [0x2fb0] = "\x01\x6a\x36", [0x2fb1] = "\x01\x6a\x3e",
  [0x2fb2] = "\x01\x6a\x3d", [0x2fb3] = "\x0e\x48\x70",
  [0x2fb4] = "\x01\x6a\x3c", [0x2fb5] = "\x01\x6a\x42",
  [0x2fb6] = "\x0e\x48\x69", [0x2fb7] = "\x02\x4c\x6a",
  [0x2fb8] = "\x01\x6a\x43", [0x2fb9] = "\x02\x4c\x78",
  [0x2fba] = "\x01\x6a\x3f", [0x2fbb] = "\x01\x6a\x35",
  [0x2fbc] = "\x02\x4c\x7a", [0x2fbd] = "\x01\x6a\x38",
  [0x2fbe] = "\x01\x6a\x39", [0x2fbf] = "\x01\x6a\x41",
  [0x2fc0] = "\x02\x4c\x6f", [0x2fc1] = "\x02\x4c\x6e",
  [0x2fc2] = "\x02\x4c\x6b", [0x2fc4] = "\x02\x4c\x73",
  [0x2fc5] = "\x02\x4c\x70", [0x2fc6] = "\x02\x4c\x74",
  [0x2fc7] = "\x01\x6a\x46", [0x2fc8] = "\x0e\x48\x68",
  [0x2fc9] = "\x02\x4d\x22", [0x2fca] = "\x01\x6a\x3b",
  [0x2fcb] = "\x02\x4c\x75", [0x2fcc] = "\x02\x4c\x76",
  [0x2fcd] = "\x0e\x48\x71", [0x2fce] = "\x02\x4c\x72",
  [0x2fcf] = "\x0e\x48\x73", [0x2fd0] = "\x0e\x66\x29",
  [0x2fd1] = "\x0e\x48\x6b", [0x2fd2] = "\x01\x6a\x45",
  [0x2fd3] = "\x0e\x66\x2b", [0x2fd6] = "\x0e\x4e\x22",
  [0x2fd7] = "\x02\x52\x6f", [0x2fd8] = "\x01\x6e\x3b",
  [0x2fd9] = "\x01\x6e\x44", [0x2fda] = "\x01\x6e\x40",
  [0x2fdb] = "\x02\x52\x6c", [0x2fdc] = "\x0e\x4d\x7c",
  [0x2fdd] = "\x01\x6e\x3d", [0x2fde] = "\x01\x6e\x41",
  [0x2fdf] = "\x02\x52\x78", [0x2fe0] = "\x01\x6e\x37",
  [0x2fe1] = "\x02\x52\x70", [0x2fe2] = "\x0e\x4d\x78",
  [0x2fe3] = "\x01\x6e\x3f", [0x2fe4] = "\x0e\x4e\x24",
  [0x2fe5] = "\x0e\x4e\x2f", [0x2fe6] = "\x02\x52\x73",
  [0x2fe7] = "\x02\x52\x6e", [0x2fe8] = "\x01\x6e\x3e",
  [0x2fe9] = "\x01\x6e\x42", [0x2fea] = "\x02\x52\x6d",
  [0x2feb] = "\x0e\x4e\x2e", [0x2fec] = "\x01\x6e\x3c",
  [0x2fed] = "\x0e\x4d\x77", [0x2fee] = "\x02\x52\x77",
  [0x2fef] = "\x01\x6e\x39", [0x2ff0] = "\x02\x52\x76",
  [0x2ff1] = "\x02\x52\x75", [0x2ff2] = "\x01\x6e\x45",
  [0x2ff3] = "\x02\x50\x3b", [0x2ff4] = "\x01\x6e\x38",
  [0x2ff5] = "\x0e\x4e\x2b", [0x2ff6] = "\x02\x52\x74",
  [0x2ff7] = "\x02\x52\x6b", [0x2ff8] = "\x0e\x4d\x75",
  [0x2ff9] = "\x01\x6e\x46", [0x2ffa] = "\x02\x52\x72",
  [0x2ffb] = "\x01\x6e\x3a", [0x2ffc] = "\x0e\x4e\x28",
  [0x2ffd] = "\x0e\x4e\x29", [0x2ffe] = "\x0e\x4e\x25",
  [0x2fff] = "\x0e\x4e\x2c", [0x3000] = "\x0e\x4e\x27",
  [0x3001] = "\x0e\x4d\x7e", [0x3002] = "\x0e\x4d\x7d",
  [0x3003] = "\x02\x52\x71", [0x3007] = "\x0e\x66\x2a",
  [0x3008] = "\x01\x71\x6a", [0x3009] = "\x01\x71\x6f",
  [0x300a] = "\x01\x71\x68", [0x300b] = "\x02\x59\x44",
  [0x300c] = "\x02\x59\x3b", [0x300d] = "\x02\x59\x47",
  [0x300e] = "\x02\x59\x3f", [0x300f] = "\x02\x59\x45",
  [0x3010] = "\x01\x71\x70", [0x3011] = "\x01\x71\x69",
  [0x3012] = "\x02\x59\x38", [0x3013] = "\x02\x59\x3e",
  [0x3014] = "\x02\x59\x48", [0x3015] = "\x02\x59\x41",
  [0x3016] = "\x02\x59\x46", [0x3017] = "\x02\x59\x3a",
  [0x3019] = "\x0e\x52\x3b", [0x301a] = "\x02\x59\x42",
  [0x301b] = "\x01\x71\x6b", [0x301c] = "\x02\x59\x40",
  [0x301d] = "\x01\x71\x6e", [0x301e] = "\x01\x71\x6d",
  [0x301f] = "\x02\x59\x3c", [0x3020] = "\x02\x59\x3d",
  [0x3021] = "\x02\x59\x39", [0x3022] = "\x02\x59\x43",
  [0x3023] = "\x01\x71\x6c", [0x3024] = "\x02\x59\x4a",
  [0x3025] = "\x02\x59\x49", [0x3026] = "\x0e\x52\x40",
  [0x3027] = "\x0e\x52\x3f", [0x3029] = "\x02\x5e\x47",
  [0x302a] = "\x02\x5e\x43", [0x302b] = "\x01\x74\x69",
  [0x302c] = "\x0e\x55\x79", [0x302d] = "\x02\x5e\x3d",
  [0x302e] = "\x01\x74\x63", [0x302f] = "\x01\x74\x73",
  [0x3030] = "\x02\x5e\x49", [0x3031] = "\x01\x74\x6b",
  [0x3032] = "\x01\x74\x67", [0x3033] = "\x02\x5e\x40",
  [0x3034] = "\x01\x74\x6e", [0x3035] = "\x01\x74\x71",
  [0x3036] = "\x02\x5e\x4b", [0x3037] = "\x01\x74\x66",
  [0x3038] = "\x02\x5e\x42", [0x3039] = "\x01\x74\x6f",
  [0x303a] = "\x02\x5e\x4d", [0x303b] = "\x02\x5e\x4a",
  [0x303c] = "\x02\x5e\x3e", [0x303d] = "\x01\x74\x6a",
  [0x303e] = "\x01\x74\x64", [0x303f] = "\x01\x74\x72",
  [0x3040] = "\x02\x5e\x45", [0x3041] = "\x01\x74\x6d",
  [0x3042] = "\x02\x5e\x3f", [0x3043] = "\x01\x74\x68",
  [0x3044] = "\x02\x5e\x4c", [0x3045] = "\x01\x74\x6c",
  [0x3046] = "\x01\x74\x65", [0x3047] = "\x02\x5e\x46",
  [0x3048] = "\x01\x74\x70", [0x3049] = "\x02\x5e\x44",
  [0x304c] = "\x02\x5e\x48", [0x304d] = "\x0e\x55\x7a",
  [0x304f] = "\x0e\x58\x59", [0x3050] = "\x02\x62\x5a",
  [0x3051] = "\x02\x62\x60", [0x3052] = "\x01\x77\x25",
  [0x3053] = "\x02\x62\x63", [0x3054] = "\x01\x76\x7e",
  [0x3055] = "\x01\x77\x21", [0x3056] = "\x02\x62\x5b",
  [0x3057] = "\x02\x62\x62", [0x3058] = "\x02\x62\x5d",
  [0x3059] = "\x01\x77\x26", [0x305a] = "\x01\x77\x23",
  [0x305b] = "\x0e\x58\x5b", [0x305c] = "\x02\x62\x59",
  [0x305d] = "\x0e\x58\x58", [0x305e] = "\x01\x77\x22",
  [0x305f] = "\x02\x62\x5f", [0x3060] = "\x02\x62\x61",
  [0x3061] = "\x01\x77\x24", [0x3062] = "\x02\x62\x5e",
  [0x3063] = "\x02\x62\x5c", [0x3064] = "\x0e\x66\x2c",
  [0x3066] = "\x0e\x58\x5a", [0x3068] = "\x02\x66\x42",
  [0x3069] = "\x01\x78\x62", [0x306a] = "\x01\x78\x63",
  [0x306b] = "\x01\x78\x5f", [0x306c] = "\x0e\x5a\x72",
  [0x306d] = "\x01\x78\x60", [0x306e] = "\x0e\x5a\x74",
  [0x306f] = "\x02\x66\x3e", [0x3070] = "\x02\x66\x3c",
  [0x3071] = "\x0e\x5a\x75", [0x3072] = "\x02\x66\x40",
  [0x3073] = "\x01\x78\x64", [0x3074] = "\x02\x66\x41",
  [0x3075] = "\x02\x66\x3a", [0x3076] = "\x02\x66\x39",
  [0x3077] = "\x02\x66\x3d", [0x3078] = "\x02\x66\x3b",
  [0x3079] = "\x01\x78\x61", [0x307a] = "\x02\x66\x3f",
  [0x307b] = "\x02\x69\x59", [0x307c] = "\x01\x7a\x2b",
  [0x307d] = "\x01\x7a\x2a", [0x307e] = "\x02\x69\x5a",
  [0x307f] = "\x0e\x5c\x6c", [0x3080] = "\x02\x69\x5c",
  [0x3081] = "\x02\x69\x5b", [0x3082] = "\x01\x7a\x2c",
  [0x3084] = "\x0e\x5e\x53", [0x3085] = "\x0e\x5e\x50",
  [0x3086] = "\x02\x6b\x79", [0x3087] = "\x02\x6b\x76",
  [0x3088] = "\x02\x6b\x77", [0x3089] = "\x0e\x5e\x51",
  [0x308a] = "\x02\x6b\x75", [0x308b] = "\x02\x6b\x78",
  [0x308c] = "\x01\x7a\x7d", [0x308d] = "\x02\x6b\x7a",
  [0x308e] = "\x0e\x5e\x52", [0x308f] = "\x01\x7a\x7c",
  [0x3091] = "\x02\x6d\x6f", [0x3092] = "\x0e\x5f\x5a",
  [0x3093] = "\x01\x7c\x41", [0x3094] = "\x01\x7c\x43",
  [0x3095] = "\x02\x6f\x3d", [0x3096] = "\x01\x7c\x42",
  [0x3097] = "\x02\x70\x47", [0x3098] = "\x02\x71\x2f",
  [0x3099] = "\x02\x71\x31", [0x309a] = "\x02\x71\x2e",
  [0x309b] = "\x02\x71\x30", [0x309c] = "\x01\x7d\x39",
  [0x309e] = "\x0e\x62\x35", [0x3136] = "\x01\x48\x4f",
  [0x3138] = "\x01\x52\x7e", [0x3139] = "\x02\x30\x52",
  [0x313a] = "\x01\x57\x4d", [0x313b] = "\x0e\x38\x31",
  [0x313d] = "\x01\x5c\x5f", [0x313e] = "\x02\x3f\x22",
  [0x313f] = "\x02\x3f\x23", [0x3140] = "\x0e\x66\x2d",
  [0x3141] = "\x0e\x48\x77", [0x3143] = "\x02\x59\x4b",
  [0x3144] = "\x01\x74\x74", [0x3145] = "\x02\x5e\x4e",
  [0x3146] = "\x0e\x55\x7d", [0x3147] = "\x0e\x58\x5c",
  [0x3148] = "\x01\x77\x27", [0x314a] = "\x02\x66\x44",
  [0x314b] = "\x02\x66\x43", [0x314c] = "\x01\x7a\x2d",
  [0x314d] = "\x02\x6b\x7b", [0x314e] = "\x0e\x5f\x5b",
  [0x314f] = "\x02\x6d\x70", [0x3150] = "\x01\x7c\x64",
  [0x3151] = "\x02\x22\x69", [0x3154] = "\x01\x4f\x2c",
  [0x3155] = "\x01\x4b\x38", [0x3158] = "\x02\x2c\x6e",
  [0x3159] = "\x0e\x2a\x4e", [0x315b] = "\x02\x32\x24",
  [0x315c] = "\x02\x31\x7d", [0x315d] = "\x02\x32\x23",
  [0x315e] = "\x02\x32\x21", [0x315f] = "\x01\x57\x4e",
  [0x3160] = "\x02\x32\x22", [0x3161] = "\x02\x31\x7e",
  [0x3162] = "\x0e\x33\x3c", [0x3163] = "\x02\x38\x36",
  [0x3165] = "\x02\x3f\x24", [0x3166] = "\x02\x3f\x25",
  [0x3167] = "\x02\x46\x30", [0x3168] = "\x02\x46\x31",
  [0x3169] = "\x01\x65\x75", [0x316a] = "\x01\x65\x76",
  [0x316b] = "\x02\x46\x2f", [0x316c] = "\x02\x46\x32",
  [0x316d] = "\x02\x46\x2e", [0x316e] = "\x01\x65\x74",
  [0x316f] = "\x0e\x48\x78", [0x3170] = "\x01\x6a\x48",
  [0x3171] = "\x0e\x48\x79", [0x3172] = "\x01\x65\x77",
  [0x3173] = "\x02\x4d\x23", [0x3175] = "\x01\x6e\x47",
  [0x3176] = "\x02\x52\x79", [0x3177] = "\x01\x6e\x48",
  [0x3178] = "\x0e\x4e\x30", [0x3179] = "\x01\x71\x71",
  [0x317a] = "\x02\x59\x4e", [0x317b] = "\x02\x59\x4c",
  [0x317c] = "\x02\x59\x4d", [0x317d] = "\x02\x5e\x51",
  [0x317e] = "\x02\x5e\x50", [0x317f] = "\x02\x5e\x4f",
  [0x3183] = "\x02\x66\x45", [0x3184] = "\x0e\x64\x22",
  [0x3185] = "\x01\x78\x65", [0x3186] = "\x02\x66\x46",
  [0x3187] = "\x02\x6d\x71", [0x3188] = "\x01\x7c\x65",
  [0x3189] = "\x02\x70\x48", [0x318a] = "\x01\x48\x50",
  [0x318b] = "\x01\x4f\x2e", [0x318c] = "\x01\x4f\x2d",
  [0x318d] = "\x02\x2c\x70", [0x318e] = "\x01\x53\x21",
  [0x318f] = "\x0e\x2e\x5d", [0x3191] = "\x02\x2c\x6f",
  [0x3192] = "\x02\x32\x26", [0x3193] = "\x0e\x33\x3d",
  [0x3194] = "\x01\x57\x4f", [0x3195] = "\x02\x38\x37",
  [0x3196] = "\x02\x32\x25", [0x3197] = "\x0e\x33\x3f",
  [0x3199] = "\x0e\x33\x3e", [0x319a] = "\x01\x5c\x61",
  [0x319b] = "\x02\x38\x3a", [0x319c] = "\x02\x38\x38",
  [0x319d] = "\x02\x38\x39", [0x319e] = "\x01\x5c\x60",
  [0x31a0] = "\x02\x3f\x27", [0x31a1] = "\x02\x3f\x28",
  [0x31a2] = "\x02\x3f\x26", [0x31a3] = "\x0e\x43\x6e",
  [0x31a4] = "\x01\x65\x7a", [0x31a5] = "\x02\x46\x34",
  [0x31a6] = "\x02\x46\x33", [0x31a7] = "\x02\x46\x35",
  [0x31a8] = "\x01\x65\x79", [0x31a9] = "\x01\x65\x78",
  [0x31ac] = "\x02\x52\x7a", [0x31ad] = "\x02\x52\x7c",
  [0x31ae] = "\x0e\x4e\x31", [0x31af] = "\x01\x6e\x49",
  [0x31b0] = "\x02\x52\x7b", [0x31b1] = "\x02\x59\x4f",
  [0x31b2] = "\x01\x71\x72", [0x31b3] = "\x02\x62\x65",
  [0x31b4] = "\x0e\x58\x5d", [0x31b5] = "\x02\x62\x64",
  [0x31b6] = "\x01\x78\x66", [0x31b7] = "\x02\x66\x47",
  [0x31b8] = "\x01\x78\x68", [0x31b9] = "\x01\x78\x67",
  [0x31ba] = "\x02\x69\x5d", [0x31bb] = "\x02\x6b\x7c",
  [0x31bc] = "\x01\x7a\x7e", [0x31bd] = "\x01\x48\x51",
  [0x31be] = "\x02\x2c\x71", [0x31bf] = "\x01\x53\x22",
  [0x31c0] = "\x02\x32\x29", [0x31c1] = "\x01\x57\x51",
  [0x31c2] = "\x02\x32\x28", [0x31c3] = "\x02\x32\x27",
  [0x31c4] = "\x0e\x33\x42", [0x31c5] = "\x01\x57\x50",
  [0x31c6] = "\x0e\x33\x43", [0x31c7] = "\x02\x38\x40",
  [0x31c9] = "\x02\x38\x42", [0x31ca] = "\x02\x38\x3b",
  [0x31cb] = "\x02\x38\x3c", [0x31cc] = "\x01\x5c\x62",
  [0x31cd] = "\x02\x38\x3d", [0x31ce] = "\x01\x5c\x63",
  [0x31cf] = "\x02\x38\x41", [0x31d0] = "\x02\x38\x3e",
  [0x31d1] = "\x02\x38\x3f", [0x31d2] = "\x01\x5c\x64",
  [0x31d3] = "\x0e\x3e\x37", [0x31d4] = "\x01\x61\x44",
  [0x31d5] = "\x01\x61\x45", [0x31d6] = "\x0e\x3e\x38",
  [0x31d7] = "\x02\x3f\x29", [0x31db] = "\x02\x46\x36",
  [0x31dc] = "\x02\x46\x37", [0x31dd] = "\x0e\x43\x72",
  [0x31de] = "\x02\x4d\x27", [0x31df] = "\x01\x6a\x4b",
  [0x31e0] = "\x01\x6a\x49", [0x31e1] = "\x01\x6a\x4a",
  [0x31e2] = "\x02\x4d\x24", [0x31e3] = "\x02\x4d\x25",
  [0x31e5] = "\x02\x4d\x26", [0x31e6] = "\x02\x53\x23",
  [0x31e7] = "\x0e\x4e\x32", [0x31e8] = "\x02\x53\x24",
  [0x31e9] = "\x01\x6e\x4a", [0x31ea] = "\x02\x53\x21",
  [0x31eb] = "\x02\x52\x7e", [0x31ec] = "\x02\x53\x22",
  [0x31ed] = "\x02\x52\x7d", [0x31ee] = "\x01\x71\x75",
  [0x31ef] = "\x02\x59\x50", [0x31f0] = "\x01\x71\x73",
  [0x31f1] = "\x01\x71\x74", [0x31f2] = "\x02\x5e\x53",
  [0x31f3] = "\x01\x74\x75", [0x31f4] = "\x02\x5e\x52",
  [0x31f5] = "\x02\x61\x34", [0x31f6] = "\x0e\x55\x7e",
  [0x31f7] = "\x02\x62\x66", [0x31f8] = "\x02\x62\x67",
  [0x31f9] = "\x01\x77\x28", [0x31fa] = "\x0e\x58\x61",
  [0x31fb] = "\x01\x77\x29", [0x31fc] = "\x01\x74\x76",
  [0x31fd] = "\x02\x66\x48", [0x31fe] = "\x02\x66\x49",
  [0x31ff] = "\x02\x69\x5e", [0x3200] = "\x01\x7a\x2e",
  [0x3201] = "\x01\x48\x52", [0x3202] = "\x0e\x66\x7b",
  [0x3203] = "\x01\x48\x53", [0x3204] = "\x01\x57\x53",
  [0x3205] = "\x01\x4f\x2f", [0x3206] = "\x01\x57\x52",
  [0x3207] = "\x02\x2c\x72", [0x3208] = "\x0e\x38\x36",
  [0x320a] = "\x0e\x33\x44", [0x320b] = "\x01\x61\x46",
  [0x320c] = "\x01\x48\x54", [0x320d] = "\x01\x53\x24",
  [0x320e] = "\x02\x2c\x73", [0x320f] = "\x02\x2c\x74",
  [0x3210] = "\x01\x53\x23", [0x3211] = "\x01\x53\x25",
  [0x3212] = "\x01\x48\x55", [0x3214] = "\x02\x2c\x75",
  [0x3215] = "\x01\x57\x55", [0x3216] = "\x02\x32\x2a",
  [0x3217] = "\x01\x57\x57", [0x3218] = "\x01\x57\x54",
  [0x3219] = "\x01\x57\x56", [0x321a] = "\x0e\x38\x37",
  [0x321b] = "\x02\x38\x45", [0x321c] = "\x01\x5c\x65",
  [0x321d] = "\x0e\x38\x39", [0x321e] = "\x02\x38\x44",
  [0x321f] = "\x02\x38\x43", [0x3221] = "\x02\x46\x38",
  [0x3223] = "\x0e\x49\x22", [0x3224] = "\x02\x4d\x28",
  [0x3226] = "\x01\x6e\x4b", [0x3228] = "\x01\x71\x76",
  [0x3229] = "\x02\x59\x52", [0x322a] = "\x02\x59\x51",
  [0x322b] = "\x0e\x56\x21", [0x322c] = "\x02\x5e\x54",
  [0x322e] = "\x0e\x58\x62", [0x322f] = "\x0e\x5c\x6f",
  [0x3230] = "\x02\x6b\x7d", [0x3233] = "\x01\x48\x56",
  [0x3234] = "\x02\x24\x73", [0x3235] = "\x02\x28\x4a",
  [0x3236] = "\x01\x53\x26", [0x3237] = "\x02\x2c\x76",
  [0x3239] = "\x02\x32\x2c", [0x323b] = "\x0e\x33\x49",
  [0x323c] = "\x0e\x33\x48", [0x323d] = "\x01\x57\x58",
  [0x323e] = "\x02\x32\x2b", [0x323f] = "\x01\x57\x59",
  [0x3240] = "\x0e\x33\x47", [0x3241] = "\x0e\x66\x2f",
  [0x3243] = "\x02\x38\x47", [0x3244] = "\x0e\x38\x3d",
  [0x3245] = "\x0e\x38\x3c", [0x3246] = "\x01\x5c\x67",
  [0x3247] = "\x02\x38\x46", [0x3248] = "\x02\x38\x48",
  [0x3249] = "\x0e\x38\x3b", [0x324a] = "\x01\x5c\x66",
  [0x324e] = "\x0e\x3e\x3c", [0x324f] = "\x02\x3f\x2b",
  [0x3250] = "\x02\x3f\x2c", [0x3251] = "\x02\x3f\x2a",
  [0x3252] = "\x01\x61\x47", [0x3253] = "\x0e\x67\x29",
  [0x3256] = "\x01\x65\x7b", [0x3257] = "\x0e\x43\x73",
  [0x3258] = "\x01\x65\x7c", [0x3259] = "\x0e\x63\x56",
  [0x325a] = "\x01\x6a\x4d", [0x325b] = "\x0e\x49\x23",
  [0x325c] = "\x02\x4d\x2a", [0x325d] = "\x02\x4d\x29",
  [0x325e] = "\x01\x6a\x4c", [0x325f] = "\x0e\x49\x26",
  [0x3260] = "\x0e\x3e\x3b", [0x3261] = "\x0e\x49\x25",
  [0x3263] = "\x0e\x66\x30", [0x3264] = "\x02\x53\x25",
  [0x3265] = "\x0e\x4e\x34", [0x3266] = "\x0e\x4e\x36",
  [0x3267] = "\x02\x53\x26", [0x3268] = "\x0e\x4e\x37",
  [0x326a] = "\x0e\x4e\x35", [0x326b] = "\x0e\x4e\x38",
  [0x326c] = "\x02\x59\x53", [0x326e] = "\x0e\x52\x43",
  [0x326f] = "\x01\x74\x7a", [0x3270] = "\x01\x74\x79",
  [0x3271] = "\x01\x74\x77", [0x3272] = "\x01\x74\x78",
  [0x3273] = "\x01\x74\x7b", [0x3274] = "\x0e\x56\x22",
  [0x3275] = "\x02\x62\x68", [0x3276] = "\x01\x77\x2b",
  [0x3277] = "\x01\x77\x2a", [0x3278] = "\x02\x66\x4a",
  [0x3279] = "\x02\x69\x5f", [0x327a] = "\x0e\x5c\x70",
  [0x327b] = "\x0e\x5c\x71", [0x327c] = "\x0e\x5c\x72",
  [0x327d] = "\x01\x7b\x6f", [0x327e] = "\x01\x7b\x6e",
  [0x327f] = "\x01\x48\x57", [0x3282] = "\x02\x2f\x7d",
  [0x3284] = "\x01\x65\x7e", [0x3285] = "\x01\x61\x48",
  [0x3286] = "\x01\x65\x7d", [0x3287] = "\x01\x6a\x4e",
  [0x3289] = "\x01\x48\x58", [0x328a] = "\x02\x21\x68",
  [0x328b] = "\x01\x48\x59", [0x328c] = "\x01\x48\x5a",
  [0x328d] = "\x0e\x24\x35", [0x328e] = "\x0e\x24\x36",
  [0x328f] = "\x02\x28\x4b", [0x3290] = "\x02\x24\x76",
  [0x3291] = "\x0e\x27\x24", [0x3292] = "\x02\x24\x77",
  [0x3293] = "\x01\x4b\x3a", [0x3294] = "\x0e\x27\x26",
  [0x3295] = "\x02\x24\x74", [0x3296] = "\x01\x4b\x39",
  [0x3297] = "\x0e\x27\x25", [0x3298] = "\x01\x4b\x3c",
  [0x3299] = "\x02\x24\x75", [0x329a] = "\x01\x4b\x3e",
  [0x329b] = "\x01\x4b\x3d", [0x329c] = "\x02\x24\x78",
  [0x329d] = "\x01\x4b\x3b", [0x329f] = "\x0e\x27\x23",
  [0x32a1] = "\x01\x4f\x34", [0x32a2] = "\x01\x4f\x32",
  [0x32a3] = "\x02\x28\x4d", [0x32a4] = "\x0e\x2a\x56",
  [0x32a5] = "\x01\x4f\x31", [0x32a6] = "\x0e\x2a\x5b",
  [0x32a7] = "\x0e\x2a\x58", [0x32a8] = "\x0e\x2a\x4f",
  [0x32a9] = "\x01\x4f\x36", [0x32aa] = "\x01\x4f\x38",
  [0x32ab] = "\x01\x4f\x35", [0x32ac] = "\x0e\x2a\x59",
  [0x32ad] = "\x02\x28\x50", [0x32ae] = "\x02\x28\x4c",
  [0x32af] = "\x01\x4f\x39", [0x32b0] = "\x0e\x2a\x52",
  [0x32b1] = "\x01\x4f\x33", [0x32b2] = "\x01\x4b\x3f",
  [0x32b3] = "\x0e\x2a\x54", [0x32b4] = "\x01\x4f\x37",
  [0x32b5] = "\x02\x28\x4f", [0x32b6] = "\x0e\x2a\x57",
  [0x32b8] = "\x02\x28\x4e", [0x32ba] = "\x01\x4f\x30",
  [0x32bb] = "\x0e\x66\x31", [0x32c2] = "\x02\x2c\x7c",
  [0x32c3] = "\x01\x53\x2a", [0x32c4] = "\x01\x53\x2b",
  [0x32c5] = "\x02\x2c\x7e", [0x32c6] = "\x0e\x2e\x66",
  [0x32c7] = "\x02\x2c\x78", [0x32c8] = "\x02\x2c\x7b",
  [0x32c9] = "\x02\x2d\x26", [0x32ca] = "\x02\x2d\x24",
  [0x32cb] = "\x0e\x2e\x60", [0x32cc] = "\x01\x53\x2c",
  [0x32cd] = "\x02\x2d\x2a", [0x32ce] = "\x01\x53\x2f",
  [0x32cf] = "\x02\x2d\x27", [0x32d0] = "\x02\x2c\x7d",
  [0x32d1] = "\x02\x2c\x7a", [0x32d2] = "\x0e\x2e\x61",
  [0x32d3] = "\x0e\x2e\x5e", [0x32d4] = "\x02\x3f\x2e",
  [0x32d5] = "\x02\x2d\x25", [0x32d6] = "\x01\x53\x27",
  [0x32d7] = "\x02\x2d\x28", [0x32d8] = "\x02\x2c\x77",
  [0x32d9] = "\x02\x2d\x22", [0x32da] = "\x01\x53\x29",
  [0x32db] = "\x01\x53\x2e", [0x32dc] = "\x02\x2d\x23",
  [0x32dd] = "\x01\x53\x32", [0x32de] = "\x01\x53\x30",
  [0x32df] = "\x0e\x2e\x65", [0x32e0] = "\x02\x2c\x79",
  [0x32e1] = "\x01\x53\x2d", [0x32e2] = "\x0e\x2e\x64",
  [0x32e3] = "\x02\x2d\x21", [0x32e4] = "\x01\x53\x31",
  [0x32e5] = "\x01\x53\x28", [0x32e6] = "\x02\x2d\x29",
  [0x32ed] = "\x01\x57\x5e", [0x32ee] = "\x0e\x33\x4c",
  [0x32ef] = "\x01\x57\x67", [0x32f0] = "\x01\x57\x5c",
  [0x32f1] = "\x01\x57\x5a", [0x32f2] = "\x02\x32\x2e",
  [0x32f3] = "\x01\x57\x62", [0x32f4] = "\x01\x57\x5f",
  [0x32f5] = "\x02\x32\x30", [0x32f6] = "\x0e\x33\x4a",
  [0x32f7] = "\x0e\x33\x52", [0x32f8] = "\x01\x57\x61",
  [0x32f9] = "\x02\x32\x2f", [0x32fa] = "\x02\x32\x2d",
  [0x32fb] = "\x02\x32\x32", [0x32fc] = "\x01\x57\x66",
  [0x32fd] = "\x01\x57\x64", [0x32fe] = "\x02\x3f\x2d",
  [0x32ff] = "\x0e\x33\x4b", [0x3300] = "\x02\x32\x33",
  [0x3301] = "\x02\x32\x31", [0x3302] = "\x01\x57\x5b",
  [0x3303] = "\x0e\x33\x4e", [0x3304] = "\x0e\x33\x4d",
  [0x3305] = "\x01\x57\x5d", [0x3306] = "\x01\x57\x60",
  [0x3307] = "\x0e\x33\x4f", [0x3308] = "\x01\x57\x63",
  [0x3309] = "\x0e\x2e\x63", [0x330a] = "\x01\x57\x65",
  [0x3314] = "\x0e\x3e\x44", [0x3315] = "\x02\x38\x52",
  [0x3316] = "\x01\x5c\x69", [0x3317] = "\x0e\x38\x49",
  [0x3318] = "\x02\x38\x49", [0x3319] = "\x02\x38\x4b",
  [0x331a] = "\x0e\x38\x47", [0x331b] = "\x02\x38\x4c",
  [0x331d] = "\x02\x38\x54", [0x331e] = "\x02\x38\x50",
  [0x331f] = "\x02\x38\x4e", [0x3321] = "\x02\x38\x51",
  [0x3322] = "\x02\x38\x55", [0x3323] = "\x01\x5c\x6a",
  [0x3324] = "\x01\x5c\x6e", [0x3325] = "\x02\x38\x4a",
  [0x3327] = "\x02\x38\x53", [0x3328] = "\x0e\x63\x2e",
  [0x3329] = "\x01\x5c\x6c", [0x332a] = "\x0e\x38\x41",
  [0x332b] = "\x01\x5c\x6b", [0x332c] = "\x02\x38\x4f",
  [0x332d] = "\x02\x38\x4d", [0x332e] = "\x0e\x38\x42",
  [0x332f] = "\x01\x5c\x68", [0x3330] = "\x01\x5c\x6d",
  [0x3331] = "\x0e\x38\x45", [0x3334] = "\x0e\x38\x48",
  [0x3339] = "\x01\x61\x4e", [0x333a] = "\x02\x3f\x36",
  [0x333b] = "\x0e\x3e\x3e", [0x333d] = "\x02\x3f\x34",
  [0x333e] = "\x01\x61\x50", [0x3340] = "\x0e\x3e\x43",
  [0x3341] = "\x0e\x3e\x45", [0x3342] = "\x0e\x3e\x41",
  [0x3343] = "\x02\x3f\x2f", [0x3344] = "\x02\x46\x46",
  [0x3345] = "\x0e\x3e\x3d", [0x3346] = "\x01\x61\x4f",
  [0x3347] = "\x02\x3f\x33", [0x3348] = "\x0e\x3e\x40",
  [0x3349] = "\x0e\x3e\x42", [0x334a] = "\x02\x3f\x30",
  [0x334b] = "\x01\x61\x4b", [0x334c] = "\x01\x61\x51",
  [0x334d] = "\x02\x3f\x35", [0x334e] = "\x01\x61\x4d",
  [0x334f] = "\x02\x3f\x32", [0x3350] = "\x01\x6a\x4f",
  [0x3351] = "\x01\x61\x4c", [0x3352] = "\x02\x3f\x31",
  [0x3353] = "\x01\x61\x52", [0x3354] = "\x01\x61\x4a",
  [0x3355] = "\x01\x61\x49", [0x335b] = "\x02\x46\x3e",
  [0x335c] = "\x02\x46\x3c", [0x335d] = "\x0e\x43\x7b",
  [0x335e] = "\x02\x46\x42", [0x335f] = "\x0e\x43\x7e",
  [0x3360] = "\x02\x46\x3a", [0x3361] = "\x02\x46\x47",
  [0x3362] = "\x02\x46\x3f", [0x3363] = "\x0e\x43\x75",
  [0x3364] = "\x02\x46\x39", [0x3365] = "\x01\x66\x24",
  [0x3366] = "\x01\x66\x2a", [0x3367] = "\x02\x46\x44",
  [0x3369] = "\x02\x46\x3d", [0x336a] = "\x0e\x43\x76",
  [0x336b] = "\x01\x66\x27", [0x336c] = "\x0e\x43\x77",
  [0x336d] = "\x0e\x44\x23", [0x336e] = "\x01\x66\x25",
  [0x336f] = "\x02\x46\x45", [0x3370] = "\x01\x66\x22",
  [0x3371] = "\x01\x66\x21", [0x3372] = "\x02\x46\x40",
  [0x3373] = "\x01\x66\x26", [0x3374] = "\x01\x61\x53",
  [0x3375] = "\x0e\x43\x7c", [0x3376] = "\x02\x46\x43",
  [0x3377] = "\x02\x46\x3b", [0x3378] = "\x01\x66\x23",
  [0x3379] = "\x01\x66\x28", [0x337a] = "\x01\x66\x29",
  [0x337d] = "\x0e\x44\x22", [0x337f] = "\x01\x6a\x54",
  [0x3380] = "\x01\x6a\x50", [0x3381] = "\x0e\x49\x27",
  [0x3382] = "\x01\x6a\x55", [0x3383] = "\x02\x4d\x2d",
  [0x3384] = "\x0e\x49\x2a", [0x3386] = "\x02\x4d\x2c",
  [0x3387] = "\x02\x4d\x2e", [0x3388] = "\x01\x6a\x52",
  [0x3389] = "\x02\x4d\x2b", [0x338a] = "\x01\x6a\x53",
  [0x338b] = "\x02\x4d\x31", [0x338c] = "\x02\x4d\x30",
  [0x338d] = "\x02\x4d\x2f", [0x338f] = "\x01\x6a\x51",
  [0x3392] = "\x0e\x4e\x3b", [0x3393] = "\x0e\x4e\x3d",
  [0x3394] = "\x0e\x4e\x39", [0x3395] = "\x02\x53\x2a",
  [0x3396] = "\x0e\x4e\x3c", [0x3397] = "\x02\x53\x2d",
  [0x3398] = "\x01\x6e\x51", [0x3399] = "\x02\x53\x2c",
  [0x339a] = "\x01\x6e\x50", [0x339b] = "\x01\x6e\x4c",
  [0x339c] = "\x01\x6e\x4d", [0x339d] = "\x01\x6e\x4e",
  [0x339e] = "\x02\x53\x29", [0x339f] = "\x02\x53\x28",
  [0x33a0] = "\x01\x6e\x4f", [0x33a1] = "\x0e\x4e\x3a",
  [0x33a2] = "\x02\x53\x2b", [0x33a3] = "\x02\x53\x27",
  [0x33a6] = "\x02\x59\x55", [0x33a7] = "\x02\x59\x5f",
  [0x33a8] = "\x01\x71\x79", [0x33a9] = "\x01\x71\x78",
  [0x33aa] = "\x0e\x52\x49", [0x33ab] = "\x02\x59\x59",
  [0x33ac] = "\x02\x59\x5b", [0x33ad] = "\x0e\x52\x47",
  [0x33ae] = "\x02\x59\x56", [0x33af] = "\x0e\x52\x44",
  [0x33b0] = "\x02\x59\x5a", [0x33b1] = "\x02\x59\x54",
  [0x33b2] = "\x02\x59\x5d", [0x33b3] = "\x01\x71\x77",
  [0x33b4] = "\x02\x59\x5c", [0x33b5] = "\x02\x59\x58",
  [0x33b7] = "\x02\x59\x5e", [0x33b8] = "\x0e\x56\x23",
  [0x33b9] = "\x02\x59\x57", [0x33ba] = "\x01\x74\x7e",
  [0x33bb] = "\x02\x5e\x55", [0x33bc] = "\x02\x5e\x5b",
  [0x33bd] = "\x01\x75\x24", [0x33be] = "\x01\x75\x26",
  [0x33bf] = "\x01\x75\x23", [0x33c0] = "\x01\x75\x22",
  [0x33c2] = "\x01\x75\x21", [0x33c3] = "\x01\x74\x7d",
  [0x33c4] = "\x02\x5e\x56", [0x33c5] = "\x02\x5e\x59",
  [0x33c6] = "\x01\x74\x7c", [0x33c7] = "\x02\x5e\x5a",
  [0x33c8] = "\x0e\x56\x24", [0x33c9] = "\x01\x75\x25",
  [0x33ca] = "\x02\x5e\x58", [0x33cc] = "\x02\x5e\x57",
  [0x33cd] = "\x01\x77\x2c", [0x33ce] = "\x0e\x58\x65",
  [0x33cf] = "\x01\x77\x2d", [0x33d0] = "\x02\x62\x6b",
  [0x33d1] = "\x02\x62\x69", [0x33d2] = "\x02\x62\x6a",
  [0x33d4] = "\x0e\x5a\x77", [0x33d5] = "\x02\x66\x4c",
  [0x33d6] = "\x0e\x5a\x78", [0x33d7] = "\x02\x66\x4b",
  [0x33d8] = "\x01\x78\x69", [0x33d9] = "\x02\x69\x61",
  [0x33da] = "\x01\x7a\x2f", [0x33db] = "\x02\x69\x60",
  [0x33dd] = "\x02\x6b\x7e", [0x33de] = "\x02\x6d\x72",
  [0x33df] = "\x01\x7b\x70", [0x33e0] = "\x02\x71\x32",
  [0x33e1] = "\x02\x71\x33", [0x33e2] = "\x01\x7c\x44",
  [0x33e3] = "\x01\x48\x5b", [0x33e5] = "\x01\x4f\x3a",
  [0x33e6] = "\x02\x3f\x37", [0x33e7] = "\x01\x6a\x56",
  [0x33e8] = "\x01\x75\x27", [0x33e9] = "\x02\x5e\x5c",
  [0x33ea] = "\x01\x48\x5c", [0x33ec] = "\x01\x57\x69",
  [0x33ed] = "\x01\x57\x68", [0x33ee] = "\x02\x3f\x38",
  [0x33ef] = "\x0e\x3e\x47", [0x33f2] = "\x02\x59\x60",
  [0x33f3] = "\x01\x48\x5d", [0x33f4] = "\x01\x53\x33",
  [0x33f6] = "\x0e\x3e\x48", [0x33f7] = "\x02\x3f\x39",
  [0x33f8] = "\x02\x3f\x3a", [0x33f9] = "\x02\x3f\x3b",
  [0x33fa] = "\x01\x6a\x57", [0x33fb] = "\x01\x71\x7a",
  [0x33fc] = "\x01\x48\x5e", [0x33fe] = "\x01\x4f\x3b",
  [0x33ff] = "\x02\x2d\x2b", [0x3400] = "\x01\x57\x6a",
  [0x3401] = "\x02\x32\x34", [0x3402] = "\x01\x5c\x6f",
  [0x3403] = "\x0e\x3e\x49", [0x3404] = "\x02\x3f\x3c",
  [0x3405] = "\x01\x66\x2b", [0x3407] = "\x01\x6a\x58",
  [0x3408] = "\x01\x71\x7b", [0x3409] = "\x01\x75\x28",
  [0x340a] = "\x01\x77\x2e", [0x340b] = "\x02\x66\x4d",
  [0x340c] = "\x01\x48\x5f", [0x340d] = "\x01\x4f\x3c",
  [0x340e] = "\x0e\x2a\x5d", [0x3410] = "\x01\x57\x6b",
  [0x3411] = "\x02\x38\x56", [0x3412] = "\x01\x61\x54",
  [0x3413] = "\x0e\x49\x2b", [0x3414] = "\x01\x6a\x59",
  [0x3415] = "\x02\x4d\x32", [0x3416] = "\x02\x53\x2e",
  [0x3418] = "\x0e\x52\x4a", [0x3419] = "\x0e\x58\x68",
  [0x341a] = "\x0e\x5a\x79", [0x341b] = "\x01\x48\x60",
  [0x341c] = "\x01\x61\x55", [0x341d] = "\x02\x46\x48",
  [0x341e] = "\x01\x6a\x5a", [0x341f] = "\x01\x48\x61",
  [0x3420] = "\x02\x28\x51", [0x3421] = "\x02\x2d\x2c",
  [0x3422] = "\x01\x53\x34", [0x3424] = "\x0e\x2e\x68",
  [0x3425] = "\x02\x32\x36", [0x3427] = "\x0e\x33\x57",
  [0x3428] = "\x01\x57\x6e", [0x3429] = "\x0e\x33\x58",
  [0x342a] = "\x01\x57\x6c", [0x342b] = "\x01\x57\x6d",
  [0x342c] = "\x01\x57\x6f", [0x342d] = "\x0e\x33\x55",
  [0x342f] = "\x02\x32\x35", [0x3432] = "\x02\x38\x5b",
  [0x3433] = "\x02\x38\x58", [0x3434] = "\x02\x38\x5a",
  [0x3435] = "\x01\x5c\x70", [0x3436] = "\x01\x5c\x72",
  [0x3437] = "\x01\x5c\x71", [0x3438] = "\x02\x38\x57",
  [0x3439] = "\x01\x5c\x73", [0x343a] = "\x02\x38\x59",
  [0x343c] = "\x02\x3f\x3d", [0x343d] = "\x02\x3f\x3e",
  [0x343f] = "\x02\x3f\x3f", [0x3440] = "\x02\x46\x4b",
  [0x3441] = "\x0e\x44\x26", [0x3442] = "\x02\x46\x4c",
  [0x3444] = "\x02\x46\x4a", [0x3445] = "\x02\x46\x4d",
  [0x3447] = "\x01\x66\x2c", [0x3448] = "\x0e\x66\x34",
  [0x3449] = "\x02\x46\x49", [0x344a] = "\x0e\x49\x2d",
  [0x344b] = "\x01\x6a\x5b", [0x344c] = "\x0e\x49\x2e",
  [0x344e] = "\x02\x53\x33", [0x344f] = "\x02\x53\x2f",
  [0x3450] = "\x02\x53\x32", [0x3451] = "\x02\x53\x34",
  [0x3452] = "\x02\x53\x31", [0x3453] = "\x02\x53\x30",
  [0x3455] = "\x02\x59\x61", [0x3456] = "\x02\x59\x62",
  [0x3457] = "\x02\x59\x63", [0x3458] = "\x01\x71\x7c",
  [0x3459] = "\x01\x71\x7d", [0x345a] = "\x02\x5e\x5e",
  [0x345b] = "\x02\x5e\x5d", [0x345c] = "\x02\x5e\x5f",
  [0x345e] = "\x02\x62\x6d", [0x345f] = "\x02\x62\x6c",
  [0x3461] = "\x02\x66\x4f", [0x3462] = "\x0e\x5a\x7a",
  [0x3463] = "\x02\x66\x50", [0x3464] = "\x02\x66\x4e",
  [0x3465] = "\x0e\x5a\x7b", [0x3466] = "\x01\x7a\x30",
  [0x3468] = "\x02\x69\x62", [0x3469] = "\x02\x69\x63",
  [0x346a] = "\x0e\x5e\x55", [0x346b] = "\x02\x6d\x73",
  [0x346c] = "\x02\x6f\x3e", [0x346d] = "\x02\x70\x49",
  [0x346e] = "\x01\x48\x62", [0x346f] = "\x01\x4b\x40",
  [0x3471] = "\x01\x75\x29", [0x3472] = "\x01\x48\x63",
  [0x3474] = "\x02\x38\x5c", [0x3475] = "\x02\x3f\x40",
  [0x3476] = "\x0e\x5a\x7c", [0x3477] = "\x01\x7c\x6c",
  [0x3478] = "\x02\x22\x6a", [0x347b] = "\x0e\x24\x38",
  [0x347c] = "\x02\x22\x6b", [0x347d] = "\x02\x22\x6d",
  [0x347e] = "\x01\x48\x64", [0x347f] = "\x02\x22\x6e",
  [0x3480] = "\x02\x22\x6c", [0x3483] = "\x02\x25\x22",
  [0x3484] = "\x02\x25\x23", [0x3485] = "\x02\x24\x7b",
  [0x3486] = "\x0e\x27\x28", [0x348a] = "\x02\x25\x21",
  [0x348b] = "\x01\x4b\x42", [0x348c] = "\x0e\x27\x29",
  [0x348d] = "\x01\x4b\x43", [0x348e] = "\x02\x24\x7c",
  [0x348f] = "\x02\x24\x7a", [0x3490] = "\x02\x24\x79",
  [0x3491] = "\x02\x24\x7d", [0x3492] = "\x01\x4b\x41",
  [0x3493] = "\x02\x24\x7e", [0x3494] = "\x02\x2d\x2d",
  [0x3495] = "\x0e\x27\x2b", [0x3498] = "\x02\x28\x56",
  [0x3499] = "\x01\x4f\x3f", [0x349a] = "\x02\x28\x55",
  [0x349b] = "\x02\x28\x57", [0x349c] = "\x0e\x2a\x60",
  [0x349d] = "\x01\x4f\x3e", [0x349e] = "\x02\x28\x5c",
  [0x349f] = "\x01\x4f\x42", [0x34a0] = "\x02\x28\x52",
  [0x34a1] = "\x02\x28\x60", [0x34a2] = "\x02\x28\x66",
  [0x34a3] = "\x01\x4f\x49", [0x34a4] = "\x02\x28\x63",
  [0x34a5] = "\x01\x4f\x46", [0x34a6] = "\x0e\x2a\x6b",
  [0x34a7] = "\x02\x28\x59", [0x34a8] = "\x02\x28\x5f",
  [0x34a9] = "\x02\x28\x61", [0x34aa] = "\x0e\x2a\x66",
  [0x34ab] = "\x02\x28\x54", [0x34ac] = "\x01\x4f\x45",
  [0x34ad] = "\x01\x4f\x40", [0x34ae] = "\x02\x28\x5a",
  [0x34af] = "\x01\x4f\x47", [0x34b0] = "\x01\x4f\x4a",
  [0x34b1] = "\x01\x4f\x44", [0x34b2] = "\x0e\x2a\x6c",
  [0x34b3] = "\x01\x4f\x3d", [0x34b4] = "\x02\x28\x5e",
  [0x34b5] = "\x02\x28\x58", [0x34b6] = "\x02\x28\x65",
  [0x34b7] = "\x01\x4f\x4c", [0x34b8] = "\x01\x4f\x48",
  [0x34b9] = "\x01\x4f\x43", [0x34ba] = "\x02\x28\x5d",
  [0x34bb] = "\x01\x57\x70", [0x34bc] = "\x02\x28\x5b",
  [0x34bd] = "\x01\x4f\x41", [0x34be] = "\x01\x4f\x4b",
  [0x34c0] = "\x02\x28\x53", [0x34c2] = "\x02\x28\x62",
  [0x34c3] = "\x02\x28\x64", [0x34c4] = "\x0e\x2a\x5e",
  [0x34c9] = "\x0e\x2a\x68", [0x34d0] = "\x0e\x2e\x76",
  [0x34d1] = "\x01\x53\x45", [0x34d2] = "\x01\x53\x3f",
  [0x34d3] = "\x01\x53\x47", [0x34d4] = "\x01\x53\x44",
  [0x34d5] = "\x02\x2d\x34", [0x34d6] = "\x02\x2d\x37",
  [0x34d7] = "\x01\x53\x40", [0x34d8] = "\x0e\x2e\x6a",
  [0x34d9] = "\x02\x2d\x2e", [0x34db] = "\x01\x53\x39",
  [0x34dc] = "\x01\x53\x43", [0x34dd] = "\x0e\x2e\x6b",
  [0x34de] = "\x01\x53\x46", [0x34df] = "\x01\x53\x48",
  [0x34e0] = "\x02\x2d\x43", [0x34e1] = "\x02\x2d\x3a",
  [0x34e2] = "\x0e\x2e\x78", [0x34e3] = "\x01\x53\x38",
  [0x34e4] = "\x02\x2d\x42", [0x34e5] = "\x01\x53\x3c",
  [0x34e6] = "\x01\x53\x3a", [0x34e7] = "\x01\x53\x35",
  [0x34e8] = "\x02\x2d\x32", [0x34e9] = "\x0e\x2e\x72",
  [0x34ea] = "\x02\x2d\x41", [0x34eb] = "\x02\x2d\x36",
  [0x34ec] = "\x02\x2d\x39", [0x34ed] = "\x02\x2d\x46",
  [0x34ee] = "\x0e\x2e\x74", [0x34ef] = "\x01\x53\x49",
  [0x34f0] = "\x02\x2d\x40", [0x34f1] = "\x01\x53\x41",
  [0x34f2] = "\x02\x2d\x3b", [0x34f3] = "\x02\x2d\x45",
  [0x34f4] = "\x02\x2d\x38", [0x34f5] = "\x02\x2d\x3c",
  [0x34f6] = "\x02\x2d\x3f", [0x34f7] = "\x0e\x2e\x69",
  [0x34f8] = "\x0e\x2e\x6e", [0x34f9] = "\x02\x2d\x30",
  [0x34fa] = "\x02\x2d\x44", [0x34fb] = "\x02\x2d\x3e",
  [0x34fc] = "\x0e\x2e\x6f", [0x34fd] = "\x0e\x2e\x7a",
  [0x34fe] = "\x02\x2d\x2f", [0x3500] = "\x02\x2d\x33",
  [0x3501] = "\x01\x53\x42", [0x3502] = "\x01\x53\x3d",
  [0x3503] = "\x01\x53\x36", [0x3504] = "\x01\x53\x3b",
  [0x3505] = "\x01\x53\x37", [0x3506] = "\x01\x53\x4a",
  [0x3507] = "\x02\x2d\x31", [0x3508] = "\x02\x32\x47",
  [0x3509] = "\x01\x53\x3e", [0x350b] = "\x0e\x2e\x79",
  [0x350c] = "\x02\x2d\x3d", [0x350d] = "\x02\x29\x42",
  [0x350e] = "\x0e\x2e\x77", [0x3510] = "\x0e\x66\x35",
  [0x3512] = "\x0e\x66\x37", [0x3516] = "\x02\x32\x4a",
  [0x3517] = "\x01\x57\x7e", [0x3518] = "\x0e\x33\x62",
  [0x3519] = "\x02\x32\x3a", [0x351b] = "\x02\x32\x45",
  [0x351c] = "\x02\x32\x41", [0x351d] = "\x0e\x38\x4d",
  [0x351e] = "\x02\x32\x54", [0x351f] = "\x0e\x33\x59",
  [0x3520] = "\x02\x32\x4c", [0x3521] = "\x0e\x33\x5f",
  [0x3522] = "\x02\x32\x42", [0x3523] = "\x0e\x38\x5b",
  [0x3524] = "\x02\x32\x4b", [0x3525] = "\x02\x32\x3c",
  [0x3526] = "\x02\x32\x40", [0x3527] = "\x02\x32\x57",
  [0x3528] = "\x01\x58\x23", [0x3529] = "\x02\x32\x4f",
  [0x352a] = "\x02\x32\x46", [0x352b] = "\x01\x57\x71",
  [0x352c] = "\x02\x32\x55", [0x352d] = "\x02\x32\x38",
  [0x352f] = "\x02\x32\x4e", [0x3531] = "\x01\x58\x22",
  [0x3532] = "\x01\x57\x7b", [0x3533] = "\x02\x32\x37",
  [0x3534] = "\x01\x57\x79", [0x3535] = "\x01\x57\x78",
  [0x3536] = "\x01\x57\x7d", [0x3537] = "\x02\x32\x4d",
  [0x3538] = "\x01\x57\x75", [0x3539] = "\x01\x57\x7c",
  [0x353a] = "\x02\x2d\x35", [0x353b] = "\x02\x3f\x41",
  [0x353c] = "\x02\x32\x48", [0x353e] = "\x0e\x2a\x5f",
  [0x353f] = "\x02\x32\x3e", [0x3540] = "\x01\x58\x21",
  [0x3541] = "\x02\x32\x3f", [0x3542] = "\x02\x32\x43",
  [0x3543] = "\x01\x58\x24", [0x3544] = "\x02\x32\x39",
  [0x3545] = "\x02\x32\x51", [0x3546] = "\x0e\x3e\x57",
  [0x3547] = "\x02\x32\x50", [0x3548] = "\x02\x32\x58",
  [0x3549] = "\x01\x57\x77", [0x354a] = "\x01\x57\x74",
  [0x354b] = "\x02\x32\x56", [0x354c] = "\x02\x32\x52",
  [0x354d] = "\x02\x32\x49", [0x354e] = "\x02\x32\x44",
  [0x354f] = "\x01\x57\x7a", [0x3550] = "\x01\x57\x76",
  [0x3551] = "\x02\x32\x3b", [0x3552] = "\x01\x57\x72",
  [0x3553] = "\x02\x32\x53", [0x3554] = "\x01\x57\x73",
  [0x3556] = "\x02\x32\x3d", [0x3557] = "\x0e\x33\x5a",
  [0x3558] = "\x0e\x33\x63", [0x3562] = "\x0e\x66\x36",
  [0x3563] = "\x0e\x2d\x33", [0x3571] = "\x0e\x38\x4e",
  [0x3572] = "\x0e\x38\x51", [0x3573] = "\x02\x38\x62",
  [0x3574] = "\x02\x38\x64", [0x3575] = "\x02\x38\x69",
  [0x3576] = "\x02\x38\x7d", [0x3577] = "\x01\x5d\x23",
  [0x3578] = "\x01\x5c\x77", [0x3579] = "\x0e\x38\x54",
  [0x357a] = "\x02\x38\x61", [0x357b] = "\x01\x5d\x24",
  [0x357c] = "\x01\x5d\x25", [0x357d] = "\x02\x38\x6c",
  [0x357e] = "\x02\x38\x73", [0x357f] = "\x02\x38\x79",
  [0x3580] = "\x0e\x38\x50", [0x3581] = "\x02\x38\x66",
  [0x3583] = "\x02\x38\x6d", [0x3584] = "\x0e\x38\x4f",
  [0x3585] = "\x0e\x38\x5d", [0x3586] = "\x01\x5d\x26",
  [0x3587] = "\x02\x38\x7b", [0x3588] = "\x02\x38\x76",
  [0x3589] = "\x01\x5d\x21", [0x358a] = "\x01\x5c\x7d",
  [0x358b] = "\x02\x38\x72", [0x358c] = "\x02\x38\x6e",
  [0x358d] = "\x02\x38\x60", [0x358e] = "\x01\x5c\x74",
  [0x358f] = "\x02\x38\x65", [0x3590] = "\x02\x38\x5d",
  [0x3591] = "\x0e\x38\x55", [0x3592] = "\x01\x5c\x7c",
  [0x3593] = "\x01\x5c\x7e", [0x3594] = "\x02\x38\x6a",
  [0x3595] = "\x02\x38\x67", [0x3596] = "\x01\x5c\x79",
  [0x3597] = "\x02\x38\x77", [0x3598] = "\x01\x5c\x76",
  [0x3599] = "\x02\x38\x68", [0x359a] = "\x02\x3f\x6a",
  [0x359b] = "\x02\x38\x70", [0x359c] = "\x0e\x38\x5e",
  [0x359d] = "\x02\x38\x6f", [0x359e] = "\x01\x5c\x75",
  [0x359f] = "\x0e\x38\x57", [0x35a0] = "\x01\x5d\x22",
  [0x35a1] = "\x0e\x38\x52", [0x35a2] = "\x01\x5c\x78",
  [0x35a3] = "\x02\x38\x5e", [0x35a4] = "\x02\x38\x63",
  [0x35a5] = "\x02\x38\x74", [0x35a6] = "\x02\x38\x7a",
  [0x35a7] = "\x01\x5d\x27", [0x35a8] = "\x02\x38\x5f",
  [0x35a9] = "\x02\x38\x6b", [0x35aa] = "\x02\x38\x71",
  [0x35ab] = "\x01\x5c\x7b", [0x35ad] = "\x0e\x38\x58",
  [0x35ae] = "\x02\x38\x7c", [0x35af] = "\x02\x38\x75",
  [0x35b0] = "\x02\x38\x78", [0x35b7] = "\x0e\x38\x5f",
  [0x35bd] = "\x01\x5c\x7a", [0x35bf] = "\x02\x3f\x51",
  [0x35c0] = "\x02\x3f\x45", [0x35c1] = "\x01\x61\x5d",
  [0x35c2] = "\x02\x3f\x62", [0x35c3] = "\x02\x3f\x6b",
  [0x35c4] = "\x02\x3f\x6e", [0x35c5] = "\x01\x61\x5b",
  [0x35c6] = "\x02\x3f\x4d", [0x35c7] = "\x02\x3f\x66",
  [0x35c8] = "\x02\x3f\x4e", [0x35c9] = "\x02\x3f\x5c",
  [0x35ca] = "\x01\x61\x68", [0x35cb] = "\x02\x3f\x58",
  [0x35cc] = "\x01\x61\x65", [0x35cd] = "\x0e\x3e\x5e",
  [0x35ce] = "\x02\x3f\x59", [0x35cf] = "\x02\x3f\x42",
  [0x35d1] = "\x02\x3f\x67", [0x35d2] = "\x0e\x3e\x4f",
  [0x35d3] = "\x0e\x3e\x59", [0x35d4] = "\x01\x61\x6e",
  [0x35d5] = "\x02\x3f\x64", [0x35d6] = "\x02\x3f\x5a",
  [0x35d7] = "\x02\x3f\x70", [0x35d8] = "\x02\x3f\x55",
  [0x35d9] = "\x02\x46\x6d", [0x35da] = "\x0e\x3e\x4d",
  [0x35db] = "\x02\x3f\x73", [0x35dc] = "\x01\x61\x6c",
  [0x35dd] = "\x02\x3f\x53", [0x35de] = "\x02\x3f\x5f",
  [0x35df] = "\x01\x61\x6f", [0x35e0] = "\x01\x61\x5a",
  [0x35e1] = "\x02\x3f\x57", [0x35e2] = "\x02\x3f\x71",
  [0x35e3] = "\x02\x3f\x50", [0x35e4] = "\x02\x3f\x49",
  [0x35e5] = "\x02\x3f\x54", [0x35e6] = "\x0e\x3e\x5f",
  [0x35e7] = "\x02\x3f\x48", [0x35e8] = "\x02\x3f\x46",
  [0x35e9] = "\x01\x61\x56", [0x35ea] = "\x02\x3f\x68",
  [0x35eb] = "\x02\x3f\x4f", [0x35ec] = "\x02\x3f\x6c",
  [0x35ed] = "\x0e\x3e\x4b", [0x35ee] = "\x02\x3f\x6d",
  [0x35ef] = "\x01\x61\x5e", [0x35f0] = "\x01\x61\x63",
  [0x35f1] = "\x01\x61\x5f", [0x35f2] = "\x01\x61\x67",
  [0x35f3] = "\x02\x3f\x63", [0x35f4] = "\x01\x61\x60",
  [0x35f5] = "\x02\x3f\x5b", [0x35f6] = "\x02\x3f\x4b",
  [0x35f8] = "\x01\x61\x58", [0x35f9] = "\x02\x3f\x43",
  [0x35fa] = "\x02\x3f\x65", [0x35fb] = "\x02\x3f\x6f",
  [0x35fc] = "\x02\x3f\x4a", [0x35fd] = "\x01\x61\x66",
  [0x35fe] = "\x02\x3f\x74", [0x35ff] = "\x02\x3f\x56",
  [0x3600] = "\x0e\x3e\x52", [0x3601] = "\x02\x3f\x52",
  [0x3602] = "\x0e\x3e\x5c", [0x3603] = "\x01\x61\x57",
  [0x3604] = "\x01\x61\x6b", [0x3605] = "\x0e\x3e\x5a",
  [0x3606] = "\x02\x3f\x61", [0x3607] = "\x01\x61\x6d",
  [0x3608] = "\x0e\x3e\x50", [0x3609] = "\x02\x3f\x5d",
  [0x360a] = "\x01\x61\x62", [0x360b] = "\x01\x61\x5c",
  [0x360c] = "\x01\x61\x64", [0x360d] = "\x01\x61\x59",
  [0x360e] = "\x01\x61\x6a", [0x360f] = "\x02\x3f\x5e",
  [0x3610] = "\x02\x3f\x4c", [0x3611] = "\x02\x3f\x60",
  [0x3612] = "\x02\x3f\x47", [0x3613] = "\x02\x3f\x69",
  [0x3614] = "\x0e\x3e\x58", [0x3616] = "\x0e\x3e\x5d",
  [0x3617] = "\x0e\x3e\x56", [0x3618] = "\x0e\x3e\x4e",
  [0x361b] = "\x02\x3f\x72", [0x361e] = "\x0e\x66\x39",
  [0x361f] = "\x0e\x3e\x5b", [0x3621] = "\x0e\x66\x38",
  [0x3623] = "\x02\x3f\x44", [0x3629] = "\x02\x46\x6c",
  [0x362a] = "\x0e\x44\x2d", [0x362b] = "\x02\x47\x24",
  [0x362c] = "\x01\x65\x5c", [0x362d] = "\x02\x46\x71",
  [0x362e] = "\x0e\x44\x31", [0x362f] = "\x02\x46\x6f",
  [0x3630] = "\x02\x46\x5a", [0x3631] = "\x01\x66\x30",
  [0x3632] = "\x02\x46\x6a", [0x3633] = "\x02\x46\x7e",
  [0x3634] = "\x02\x46\x66", [0x3635] = "\x01\x66\x38",
  [0x3636] = "\x02\x46\x7d", [0x3637] = "\x02\x46\x64",
  [0x3638] = "\x01\x61\x69", [0x3639] = "\x02\x46\x74",
  [0x363a] = "\x02\x46\x65", [0x363b] = "\x02\x46\x7b",
  [0x363c] = "\x01\x66\x37", [0x363d] = "\x01\x66\x2f",
  [0x363e] = "\x0e\x44\x3a", [0x363f] = "\x02\x46\x4f",
  [0x3640] = "\x02\x46\x57", [0x3641] = "\x0e\x44\x35",
  [0x3642] = "\x02\x46\x70", [0x3643] = "\x02\x46\x68",
  [0x3644] = "\x02\x47\x23", [0x3645] = "\x02\x46\x6b",
  [0x3646] = "\x01\x66\x3d", [0x3647] = "\x02\x46\x7c",
  [0x3648] = "\x0e\x44\x2c", [0x3649] = "\x01\x66\x34",
  [0x364a] = "\x0e\x44\x3e", [0x364b] = "\x02\x46\x6e",
  [0x364c] = "\x02\x46\x76", [0x364d] = "\x02\x46\x5b",
  [0x364e] = "\x02\x46\x75", [0x364f] = "\x0e\x44\x27",
  [0x3650] = "\x02\x47\x28", [0x3651] = "\x02\x46\x56",
  [0x3652] = "\x02\x46\x77", [0x3653] = "\x0e\x44\x33",
  [0x3654] = "\x02\x47\x26", [0x3655] = "\x0e\x44\x3f",
  [0x3656] = "\x02\x46\x50", [0x3657] = "\x01\x61\x61",
  [0x3658] = "\x0e\x44\x40", [0x3659] = "\x02\x46\x5e",
  [0x365a] = "\x02\x46\x5d", [0x365b] = "\x01\x66\x36",
  [0x365c] = "\x0e\x44\x32", [0x365d] = "\x02\x46\x61",
  [0x365e] = "\x02\x46\x63", [0x365f] = "\x02\x46\x72",
  [0x3660] = "\x02\x47\x25", [0x3661] = "\x01\x66\x39",
  [0x3662] = "\x0e\x44\x38", [0x3663] = "\x01\x66\x3a",
  [0x3664] = "\x0e\x44\x30", [0x3665] = "\x02\x46\x55",
  [0x3666] = "\x01\x66\x32", [0x3667] = "\x02\x46\x59",
  [0x3668] = "\x02\x47\x21", [0x3669] = "\x01\x66\x3b",
  [0x366b] = "\x01\x66\x33", [0x366c] = "\x01\x66\x35",
  [0x366d] = "\x01\x66\x3c", [0x366e] = "\x02\x47\x27",
  [0x366f] = "\x02\x46\x78", [0x3670] = "\x02\x46\x73",
  [0x3671] = "\x0e\x44\x3c", [0x3672] = "\x0e\x44\x2f",
  [0x3673] = "\x02\x46\x60", [0x3674] = "\x02\x46\x5f",
  [0x3675] = "\x01\x66\x31", [0x3676] = "\x02\x46\x51",
  [0x3677] = "\x01\x66\x2e", [0x3678] = "\x02\x46\x69",
  [0x3679] = "\x02\x46\x52", [0x367a] = "\x02\x46\x67",
  [0x367b] = "\x0e\x44\x2e", [0x367d] = "\x02\x46\x5c",
  [0x367e] = "\x02\x47\x22", [0x367f] = "\x0e\x44\x2a",
  [0x3680] = "\x0e\x44\x39", [0x3682] = "\x01\x66\x2d",
  [0x3683] = "\x0e\x44\x3b", [0x3684] = "\x0e\x44\x28",
  [0x3686] = "\x02\x46\x58", [0x368d] = "\x02\x46\x54",
  [0x368e] = "\x02\x46\x7a", [0x368f] = "\x02\x46\x53",
  [0x3690] = "\x01\x6a\x68", [0x3691] = "\x02\x4d\x5a",
  [0x3692] = "\x0e\x49\x35", [0x3693] = "\x0e\x49\x44",
  [0x3694] = "\x02\x4d\x49", [0x3695] = "\x0e\x49\x33",
  [0x3696] = "\x0e\x49\x38", [0x3697] = "\x02\x4d\x33",
  [0x3698] = "\x02\x4d\x51", [0x3699] = "\x01\x6a\x60",
  [0x369a] = "\x02\x4d\x42", [0x369b] = "\x02\x4d\x4c",
  [0x369c] = "\x01\x6a\x63", [0x369d] = "\x02\x4d\x45",
  [0x369e] = "\x01\x6a\x61", [0x369f] = "\x02\x4d\x36",
  [0x36a0] = "\x02\x4d\x54", [0x36a1] = "\x02\x4d\x35",
  [0x36a2] = "\x02\x4d\x48", [0x36a3] = "\x0e\x49\x3c",
  [0x36a4] = "\x02\x4d\x34", [0x36a5] = "\x0e\x49\x39",
  [0x36a7] = "\x02\x4d\x46", [0x36a8] = "\x02\x4d\x4f",
  [0x36a9] = "\x02\x4d\x4d", [0x36aa] = "\x02\x4d\x41",
  [0x36ab] = "\x02\x4d\x3c", [0x36ac] = "\x02\x4d\x3a",
  [0x36ad] = "\x0e\x49\x42", [0x36ae] = "\x02\x4d\x3b",
  [0x36af] = "\x02\x4d\x4e", [0x36b0] = "\x02\x4d\x59",
  [0x36b1] = "\x02\x4d\x43", [0x36b2] = "\x01\x6a\x62",
  [0x36b3] = "\x0e\x49\x3b", [0x36b4] = "\x02\x4d\x3e",
  [0x36b5] = "\x0e\x49\x3a", [0x36b6] = "\x02\x4d\x52",
  [0x36b7] = "\x0e\x49\x41", [0x36b8] = "\x01\x6a\x65",
  [0x36b9] = "\x02\x4d\x3d", [0x36ba] = "\x02\x4d\x37",
  [0x36bb] = "\x02\x4d\x47", [0x36bc] = "\x01\x6a\x69",
  [0x36bd] = "\x0e\x49\x32", [0x36bf] = "\x01\x6a\x5d",
  [0x36c0] = "\x01\x6a\x66", [0x36c1] = "\x02\x4d\x3f",
  [0x36c2] = "\x02\x4d\x39", [0x36c3] = "\x0e\x49\x36",
  [0x36c4] = "\x01\x6a\x5f", [0x36c5] = "\x02\x46\x79",
  [0x36c6] = "\x01\x6a\x5e", [0x36c7] = "\x02\x4d\x4a",
  [0x36c8] = "\x0e\x44\x36", [0x36c9] = "\x01\x6a\x5c",
  [0x36ca] = "\x01\x6a\x6b", [0x36cb] = "\x01\x6a\x64",
  [0x36cc] = "\x02\x4d\x4b", [0x36cd] = "\x02\x4d\x40",
  [0x36ce] = "\x02\x4d\x38", [0x36cf] = "\x02\x4d\x53",
  [0x36d0] = "\x02\x4d\x44", [0x36d1] = "\x01\x6a\x6a",
  [0x36d2] = "\x02\x4d\x57", [0x36d3] = "\x01\x6a\x67",
  [0x36d4] = "\x02\x4d\x56", [0x36d5] = "\x0e\x49\x3f",
  [0x36d6] = "\x02\x4d\x50", [0x36d7] = "\x02\x4d\x55",
  [0x36d8] = "\x0e\x49\x3e", [0x36da] = "\x0e\x49\x43",
  [0x36db] = "\x02\x4d\x58", [0x36de] = "\x0e\x66\x3b",
  [0x36e1] = "\x0e\x66\x3c", [0x36e2] = "\x0e\x66\x3a",
  [0x36e4] = "\x0e\x49\x3d", [0x36e7] = "\x02\x53\x5c",
  [0x36e8] = "\x02\x53\x5d", [0x36e9] = "\x02\x53\x50",
  [0x36ea] = "\x02\x53\x4f", [0x36eb] = "\x02\x53\x4b",
  [0x36ec] = "\x01\x6e\x5d", [0x36ed] = "\x0e\x4e\x4f",
  [0x36ee] = "\x01\x6e\x55", [0x36ef] = "\x02\x53\x5f",
  [0x36f0] = "\x02\x53\x5e", [0x36f1] = "\x02\x46\x4e",
  [0x36f2] = "\x02\x53\x48", [0x36f3] = "\x02\x53\x4c",
  [0x36f4] = "\x02\x53\x46", [0x36f5] = "\x0e\x4e\x44",
  [0x36f6] = "\x02\x53\x59", [0x36f7] = "\x02\x53\x4a",
  [0x36f8] = "\x0e\x4e\x42", [0x36f9] = "\x02\x53\x60",
  [0x36fa] = "\x02\x53\x43", [0x36fb] = "\x02\x53\x41",
  [0x36fc] = "\x02\x53\x4d", [0x36fd] = "\x02\x53\x57",
  [0x36fe] = "\x02\x53\x52", [0x36ff] = "\x01\x6e\x5f",
  [0x3700] = "\x02\x53\x38", [0x3701] = "\x0e\x4e\x40",
  [0x3702] = "\x02\x53\x56", [0x3703] = "\x0e\x4e\x4c",
  [0x3704] = "\x0e\x4e\x46", [0x3705] = "\x0e\x4e\x54",
  [0x3706] = "\x01\x6e\x60", [0x3707] = "\x02\x46\x62",
  [0x3708] = "\x02\x53\x44", [0x3709] = "\x02\x53\x3b",
  [0x370a] = "\x02\x53\x3e", [0x370b] = "\x02\x53\x64",
  [0x370c] = "\x02\x53\x45", [0x370d] = "\x02\x53\x3c",
  [0x370e] = "\x02\x53\x3a", [0x370f] = "\x02\x53\x37",
  [0x3711] = "\x01\x6e\x59", [0x3712] = "\x02\x53\x4e",
  [0x3713] = "\x01\x6e\x58", [0x3714] = "\x01\x6e\x5c",
  [0x3715] = "\x02\x53\x49", [0x3716] = "\x02\x53\x51",
  [0x3717] = "\x01\x6e\x52", [0x3718] = "\x02\x53\x61",
  [0x3719] = "\x02\x53\x65", [0x371a] = "\x01\x6e\x54",
  [0x371b] = "\x0e\x4e\x4b", [0x371c] = "\x02\x53\x40",
  [0x371d] = "\x02\x53\x54", [0x371e] = "\x02\x53\x58",
  [0x371f] = "\x02\x53\x3d", [0x3720] = "\x02\x53\x62",
  [0x3721] = "\x01\x6e\x5b", [0x3723] = "\x01\x6e\x5a",
  [0x3724] = "\x02\x53\x35", [0x3725] = "\x01\x6e\x5e",
  [0x3726] = "\x02\x53\x5b", [0x3727] = "\x02\x53\x3f",
  [0x3728] = "\x02\x53\x53", [0x3729] = "\x02\x53\x39",
  [0x372a] = "\x02\x53\x47", [0x372b] = "\x02\x53\x42",
  [0x372c] = "\x01\x6e\x56", [0x372d] = "\x01\x6e\x57",
  [0x372e] = "\x02\x53\x55", [0x372f] = "\x02\x53\x66",
  [0x3730] = "\x02\x53\x63", [0x3731] = "\x02\x53\x5a",
  [0x3733] = "\x0e\x4e\x4d", [0x3734] = "\x0e\x4e\x4e",
  [0x3735] = "\x0e\x4e\x52", [0x373b] = "\x02\x53\x36",
  [0x373d] = "\x01\x6e\x53", [0x373e] = "\x02\x59\x74",
  [0x373f] = "\x0e\x52\x5a", [0x3740] = "\x02\x59\x6b",
  [0x3741] = "\x02\x59\x6e", [0x3742] = "\x0e\x52\x52",
  [0x3743] = "\x01\x72\x25", [0x3744] = "\x02\x59\x70",
  [0x3745] = "\x02\x59\x65", [0x3746] = "\x02\x59\x6c",
  [0x3747] = "\x02\x59\x72", [0x3748] = "\x01\x72\x22",
  [0x3749] = "\x01\x72\x26", [0x374a] = "\x01\x71\x7e",
  [0x374b] = "\x0e\x52\x59", [0x374c] = "\x0e\x52\x50",
  [0x374d] = "\x02\x59\x67", [0x374e] = "\x02\x59\x77",
  [0x3751] = "\x02\x59\x71", [0x3753] = "\x02\x59\x68",
  [0x3754] = "\x02\x5a\x22", [0x3755] = "\x02\x59\x7a",
  [0x3756] = "\x02\x59\x64", [0x3757] = "\x02\x5e\x72",
  [0x3758] = "\x02\x59\x6a", [0x3759] = "\x01\x72\x21",
  [0x375a] = "\x0e\x52\x58", [0x375b] = "\x02\x59\x75",
  [0x375c] = "\x0e\x52\x54", [0x375d] = "\x02\x5a\x21",
  [0x375e] = "\x01\x72\x29", [0x375f] = "\x0e\x52\x56",
  [0x3760] = "\x02\x59\x7c", [0x3761] = "\x02\x59\x69",
  [0x3762] = "\x02\x59\x6f", [0x3763] = "\x02\x59\x73",
  [0x3764] = "\x02\x59\x6d", [0x3765] = "\x02\x5a\x23",
  [0x3766] = "\x02\x59\x7e", [0x3767] = "\x02\x59\x7b",
  [0x3768] = "\x01\x72\x23", [0x3769] = "\x01\x72\x24",
  [0x376a] = "\x01\x72\x28", [0x376b] = "\x02\x59\x66",
  [0x376c] = "\x02\x5a\x24", [0x376d] = "\x01\x72\x27",
  [0x376e] = "\x02\x59\x78", [0x376f] = "\x0e\x52\x4f",
  [0x3770] = "\x0e\x52\x55", [0x3771] = "\x02\x59\x76",
  [0x3773] = "\x0e\x66\x3d", [0x3775] = "\x02\x59\x79",
  [0x3776] = "\x02\x5f\x21", [0x3777] = "\x02\x5e\x6c",
  [0x3778] = "\x02\x5e\x71", [0x3779] = "\x02\x5e\x7e",
  [0x377a] = "\x02\x5e\x70", [0x377b] = "\x02\x5e\x68",
  [0x377c] = "\x02\x5e\x6d", [0x377e] = "\x01\x75\x2c",
  [0x377f] = "\x0e\x56\x2b", [0x3780] = "\x02\x5e\x61",
  [0x3781] = "\x02\x5e\x79", [0x3782] = "\x02\x5e\x7b",
  [0x3783] = "\x02\x5e\x60", [0x3784] = "\x01\x75\x2b",
  [0x3785] = "\x02\x5e\x7d", [0x3786] = "\x02\x5e\x75",
  [0x3787] = "\x01\x75\x32", [0x3788] = "\x02\x5e\x7c",
  [0x3789] = "\x02\x5e\x6e", [0x378a] = "\x01\x75\x34",
  [0x378b] = "\x02\x5e\x66", [0x378c] = "\x02\x59\x7d",
  [0x378d] = "\x02\x5e\x76", [0x378e] = "\x02\x5e\x73",
  [0x378f] = "\x02\x5e\x62", [0x3790] = "\x02\x5f\x23",
  [0x3791] = "\x01\x75\x2e", [0x3792] = "\x0e\x56\x28",
  [0x3793] = "\x0e\x56\x29", [0x3794] = "\x01\x75\x2f",
  [0x3795] = "\x02\x5e\x64", [0x3796] = "\x02\x5e\x74",
  [0x3797] = "\x0e\x56\x2d", [0x3798] = "\x02\x5f\x22",
  [0x3799] = "\x02\x5e\x77", [0x379a] = "\x02\x5e\x6a",
  [0x379b] = "\x01\x75\x31", [0x379c] = "\x01\x75\x2d",
  [0x379d] = "\x02\x5e\x78", [0x379e] = "\x02\x5e\x6b",
  [0x379f] = "\x02\x5f\x24", [0x37a0] = "\x02\x5e\x65",
  [0x37a1] = "\x02\x5e\x6f", [0x37a2] = "\x02\x5e\x7a",
  [0x37a3] = "\x02\x5e\x67", [0x37a4] = "\x02\x5e\x69",
  [0x37a6] = "\x01\x75\x35", [0x37a7] = "\x02\x5e\x63",
  [0x37a8] = "\x01\x75\x33", [0x37a9] = "\x01\x77\x30",
  [0x37aa] = "\x01\x75\x2a", [0x37ab] = "\x0e\x56\x2c",
  [0x37ac] = "\x0e\x56\x30", [0x37af] = "\x01\x75\x30",
  [0x37b0] = "\x01\x77\x34", [0x37b1] = "\x02\x62\x7d",
  [0x37b2] = "\x0e\x58\x6c", [0x37b3] = "\x02\x62\x73",
  [0x37b4] = "\x02\x62\x6e", [0x37b5] = "\x02\x62\x74",
  [0x37b6] = "\x02\x62\x7e", [0x37b7] = "\x02\x63\x24",
  [0x37b8] = "\x02\x63\x23", [0x37b9] = "\x01\x77\x36",
  [0x37ba] = "\x01\x77\x35", [0x37bb] = "\x0e\x58\x6e",
  [0x37bd] = "\x02\x62\x75", [0x37be] = "\x02\x63\x25",
  [0x37bf] = "\x02\x62\x78", [0x37c0] = "\x02\x62\x70",
  [0x37c1] = "\x0e\x58\x6f", [0x37c2] = "\x02\x62\x72",
  [0x37c3] = "\x02\x62\x71", [0x37c4] = "\x02\x62\x77",
  [0x37c5] = "\x02\x62\x7c", [0x37c6] = "\x02\x62\x6f",
  [0x37c7] = "\x02\x62\x76", [0x37c8] = "\x02\x62\x7b",
  [0x37c9] = "\x01\x77\x33", [0x37cb] = "\x02\x62\x79",
  [0x37cc] = "\x0e\x58\x6b", [0x37cd] = "\x01\x77\x31",
  [0x37ce] = "\x02\x62\x7a", [0x37cf] = "\x01\x77\x2f",
  [0x37d0] = "\x01\x77\x32", [0x37d1] = "\x02\x66\x60",
  [0x37d2] = "\x02\x63\x21", [0x37d4] = "\x0e\x66\x3e",
  [0x37d5] = "\x01\x78\x6d", [0x37d6] = "\x0e\x5a\x7e",
  [0x37d7] = "\x02\x66\x58", [0x37d8] = "\x02\x66\x5c",
  [0x37d9] = "\x02\x66\x54", [0x37da] = "\x02\x66\x57",
  [0x37db] = "\x0e\x5a\x7d", [0x37dc] = "\x02\x66\x5f",
  [0x37dd] = "\x01\x78\x6b", [0x37de] = "\x02\x66\x64",
  [0x37df] = "\x02\x66\x5d", [0x37e1] = "\x02\x66\x55",
  [0x37e2] = "\x02\x66\x65", [0x37e3] = "\x02\x66\x5e",
  [0x37e4] = "\x01\x78\x6e", [0x37e5] = "\x01\x78\x6f",
  [0x37e6] = "\x02\x66\x62", [0x37e7] = "\x0e\x5b\x22",
  [0x37e8] = "\x02\x66\x56", [0x37e9] = "\x01\x78\x6a",
  [0x37ea] = "\x01\x78\x6c", [0x37eb] = "\x02\x66\x51",
  [0x37ec] = "\x02\x66\x59", [0x37ed] = "\x02\x66\x53",
  [0x37ee] = "\x0e\x5c\x7b", [0x37ef] = "\x02\x66\x63",
  [0x37f0] = "\x02\x66\x61", [0x37f1] = "\x02\x66\x52",
  [0x37f2] = "\x02\x66\x5a", [0x37f4] = "\x0e\x5b\x25",
  [0x37f5] = "\x0e\x66\x3f", [0x37f6] = "\x02\x69\x6a",
  [0x37f7] = "\x01\x78\x70", [0x37f8] = "\x02\x66\x5b",
  [0x37f9] = "\x01\x7a\x32", [0x37fa] = "\x01\x7a\x34",
  [0x37fb] = "\x01\x7a\x31", [0x37fc] = "\x0e\x5c\x76",
  [0x37fd] = "\x02\x69\x6f", [0x37fe] = "\x02\x69\x67",
  [0x37ff] = "\x02\x69\x65", [0x3800] = "\x02\x69\x69",
  [0x3801] = "\x02\x69\x66", [0x3802] = "\x0e\x5c\x78",
  [0x3803] = "\x0e\x5c\x7c", [0x3804] = "\x02\x69\x6b",
  [0x3805] = "\x02\x69\x6d", [0x3806] = "\x01\x7a\x35",
  [0x3807] = "\x01\x7a\x37", [0x3808] = "\x0e\x5d\x22",
  [0x3809] = "\x02\x69\x6c", [0x380a] = "\x01\x7a\x38",
  [0x380b] = "\x01\x7a\x36", [0x380c] = "\x02\x69\x6e",
  [0x380d] = "\x0e\x5c\x7e", [0x380f] = "\x0e\x5d\x23",
  [0x3810] = "\x0e\x5c\x77", [0x3811] = "\x01\x7a\x33",
  [0x3813] = "\x0e\x5d\x21", [0x3815] = "\x0e\x5e\x5b",
  [0x3816] = "\x0e\x5e\x5a", [0x3817] = "\x01\x7b\x21",
  [0x3818] = "\x02\x6c\x21", [0x3819] = "\x02\x6c\x27",
  [0x381a] = "\x01\x7b\x23", [0x381b] = "\x02\x69\x68",
  [0x381c] = "\x02\x6c\x26", [0x381d] = "\x0e\x5e\x5c",
  [0x381e] = "\x02\x6c\x2d", [0x381f] = "\x02\x6c\x24",
  [0x3820] = "\x02\x6c\x2b", [0x3821] = "\x02\x6c\x2a",
  [0x3822] = "\x02\x69\x64", [0x3823] = "\x02\x6c\x25",
  [0x3824] = "\x02\x63\x22", [0x3825] = "\x02\x6c\x2e",
  [0x3826] = "\x02\x6c\x23", [0x3827] = "\x02\x6c\x28",
  [0x3828] = "\x0e\x5e\x58", [0x3829] = "\x02\x6c\x2c",
  [0x382a] = "\x02\x6c\x22", [0x382b] = "\x0e\x5e\x56",
  [0x382c] = "\x02\x6d\x77", [0x382d] = "\x01\x7b\x22",
  [0x382e] = "\x02\x6c\x29", [0x382f] = "\x0e\x5e\x57",
  [0x3831] = "\x02\x6f\x43", [0x3832] = "\x02\x6d\x78",
  [0x3833] = "\x02\x6d\x76", [0x3834] = "\x02\x6d\x74",
  [0x3835] = "\x02\x6d\x75", [0x3836] = "\x02\x6d\x79",
  [0x3837] = "\x0e\x66\x40", [0x3838] = "\x01\x7c\x45",
  [0x3839] = "\x02\x6f\x41", [0x383a] = "\x02\x6f\x3f",
  [0x383b] = "\x02\x6f\x44", [0x383c] = "\x02\x6f\x42",
  [0x383d] = "\x0e\x60\x43", [0x383e] = "\x02\x6f\x45",
  [0x383f] = "\x01\x7c\x46", [0x3840] = "\x02\x6f\x40",
  [0x3841] = "\x0e\x60\x2f", [0x3842] = "\x0e\x61\x46",
  [0x3843] = "\x02\x70\x4a", [0x3845] = "\x0e\x66\x41",
  [0x3846] = "\x02\x71\x34", [0x3847] = "\x02\x71\x35",
  [0x3848] = "\x02\x71\x36", [0x3849] = "\x0e\x61\x47",
  [0x384a] = "\x0e\x61\x7c", [0x384b] = "\x02\x72\x35",
  [0x384c] = "\x02\x72\x2d", [0x384d] = "\x02\x22\x6f",
  [0x384e] = "\x01\x4f\x4d", [0x3850] = "\x01\x53\x4b",
  [0x3852] = "\x02\x32\x5a", [0x3853] = "\x02\x32\x59",
  [0x3854] = "\x01\x58\x25", [0x3855] = "\x01\x5d\x28",
  [0x3856] = "\x02\x39\x21", [0x3857] = "\x0e\x38\x63",
  [0x3858] = "\x0e\x38\x60", [0x3859] = "\x02\x38\x7e",
  [0x385a] = "\x0e\x38\x61", [0x385b] = "\x01\x61\x70",
  [0x385c] = "\x01\x66\x3f", [0x385d] = "\x0e\x3e\x61",
  [0x385e] = "\x01\x66\x3e", [0x385f] = "\x01\x66\x40",
  [0x3861] = "\x02\x4d\x5b", [0x3862] = "\x02\x53\x67",
  [0x3863] = "\x02\x5a\x25", [0x3864] = "\x02\x5a\x27",
  [0x3865] = "\x02\x5a\x26", [0x3867] = "\x01\x75\x36",
  [0x3868] = "\x02\x5f\x25", [0x3869] = "\x02\x63\x26",
  [0x386a] = "\x02\x71\x73", [0x386b] = "\x01\x48\x65",
  [0x386c] = "\x0e\x27\x2d", [0x386d] = "\x02\x28\x69",
  [0x386e] = "\x02\x28\x6a", [0x386f] = "\x02\x28\x68",
  [0x3870] = "\x02\x28\x67", [0x3871] = "\x01\x4f\x4e",
  [0x3872] = "\x0e\x66\x42", [0x3873] = "\x02\x2d\x4a",
  [0x3874] = "\x02\x2d\x48", [0x3875] = "\x0e\x2f\x23",
  [0x3876] = "\x0e\x2e\x7c", [0x3877] = "\x02\x2d\x47",
  [0x3878] = "\x0e\x2e\x7e", [0x3879] = "\x01\x53\x4c",
  [0x387a] = "\x01\x53\x4e", [0x387b] = "\x01\x53\x4d",
  [0x387c] = "\x02\x2d\x49", [0x387d] = "\x0e\x2f\x24",
  [0x3884] = "\x0e\x33\x69", [0x3885] = "\x02\x32\x66",
  [0x3886] = "\x02\x32\x63", [0x3887] = "\x02\x32\x61",
  [0x3889] = "\x0e\x33\x6c", [0x388a] = "\x01\x58\x26",
  [0x388b] = "\x02\x32\x64", [0x388c] = "\x01\x58\x2b",
  [0x388d] = "\x02\x32\x5e", [0x388e] = "\x02\x32\x6d",
  [0x388f] = "\x0e\x33\x6f", [0x3890] = "\x02\x32\x6f",
  [0x3891] = "\x02\x32\x5f", [0x3892] = "\x0e\x33\x6e",
  [0x3893] = "\x01\x58\x28", [0x3894] = "\x02\x32\x70",
  [0x3895] = "\x02\x32\x6b", [0x3896] = "\x02\x32\x5d",
  [0x3897] = "\x02\x32\x62", [0x3898] = "\x02\x32\x6c",
  [0x3899] = "\x02\x32\x68", [0x389a] = "\x02\x32\x65",
  [0x389b] = "\x0e\x33\x6b", [0x389c] = "\x01\x58\x2d",
  [0x389d] = "\x02\x32\x6e", [0x389e] = "\x02\x32\x60",
  [0x389f] = "\x0e\x33\x6a", [0x38a0] = "\x0e\x33\x70",
  [0x38a1] = "\x02\x32\x69", [0x38a2] = "\x02\x32\x5b",
  [0x38a3] = "\x01\x58\x2c", [0x38a4] = "\x01\x58\x29",
  [0x38a5] = "\x02\x32\x67", [0x38a6] = "\x0e\x33\x6d",
  [0x38a7] = "\x02\x32\x6a", [0x38a8] = "\x02\x32\x5c",
  [0x38a9] = "\x01\x58\x2a", [0x38aa] = "\x01\x58\x27",
  [0x38ae] = "\x0e\x38\x66", [0x38af] = "\x01\x5d\x32",
  [0x38b0] = "\x02\x39\x28", [0x38b1] = "\x01\x5d\x31",
  [0x38b2] = "\x0e\x38\x64", [0x38b3] = "\x02\x39\x2b",
  [0x38b4] = "\x02\x39\x2e", [0x38b5] = "\x01\x5d\x2e",
  [0x38b6] = "\x01\x5d\x2c", [0x38b7] = "\x02\x39\x23",
  [0x38b8] = "\x02\x39\x2c", [0x38b9] = "\x02\x39\x2a",
  [0x38ba] = "\x02\x39\x27", [0x38bb] = "\x02\x39\x2f",
  [0x38bc] = "\x02\x39\x30", [0x38bd] = "\x02\x39\x32",
  [0x38be] = "\x02\x39\x33", [0x38bf] = "\x02\x39\x22",
  [0x38c0] = "\x01\x5d\x2b", [0x38c1] = "\x02\x39\x25",
  [0x38c2] = "\x02\x39\x24", [0x38c3] = "\x02\x39\x31",
  [0x38c4] = "\x01\x5d\x2d", [0x38c5] = "\x02\x39\x26",
  [0x38c6] = "\x01\x5d\x2f", [0x38c7] = "\x01\x5d\x2a",
  [0x38c8] = "\x02\x39\x29", [0x38c9] = "\x01\x5d\x33",
  [0x38cb] = "\x01\x5d\x30", [0x38cc] = "\x02\x39\x2d",
  [0x38d0] = "\x01\x61\x77", [0x38d1] = "\x02\x40\x25",
  [0x38d3] = "\x02\x3f\x78", [0x38d4] = "\x01\x61\x74",
  [0x38d5] = "\x0e\x3e\x62", [0x38d6] = "\x02\x47\x2e",
  [0x38d7] = "\x02\x40\x23", [0x38d8] = "\x02\x3f\x75",
  [0x38d9] = "\x01\x61\x72", [0x38da] = "\x02\x3f\x7a",
  [0x38db] = "\x01\x61\x75", [0x38dc] = "\x02\x3f\x7e",
  [0x38dd] = "\x02\x3f\x7c", [0x38de] = "\x01\x61\x78",
  [0x38df] = "\x01\x61\x71", [0x38e1] = "\x0e\x3e\x63",
  [0x38e2] = "\x02\x3f\x76", [0x38e3] = "\x02\x3f\x79",
  [0x38e4] = "\x01\x61\x76", [0x38e6] = "\x02\x3f\x77",
  [0x38e8] = "\x02\x40\x24", [0x38e9] = "\x02\x40\x22",
  [0x38ea] = "\x02\x3f\x7b", [0x38eb] = "\x02\x3f\x7d",
  [0x38ec] = "\x02\x40\x21", [0x38ed] = "\x01\x61\x73",
  [0x38ee] = "\x0e\x3e\x68", [0x38f5] = "\x02\x47\x2f",
  [0x38f6] = "\x02\x47\x35", [0x38f7] = "\x02\x47\x2b",
  [0x38f8] = "\x02\x47\x31", [0x38f9] = "\x01\x66\x41",
  [0x38fa] = "\x02\x47\x2d", [0x38fb] = "\x01\x66\x47",
  [0x38fc] = "\x0e\x44\x44", [0x38fd] = "\x0e\x44\x45",
  [0x38fe] = "\x01\x66\x46", [0x38ff] = "\x0e\x44\x49",
  [0x3900] = "\x01\x66\x45", [0x3901] = "\x02\x47\x34",
  [0x3902] = "\x01\x66\x48", [0x3903] = "\x01\x66\x49",
  [0x3904] = "\x02\x47\x2a", [0x3905] = "\x02\x47\x37",
  [0x3906] = "\x01\x66\x4a", [0x3907] = "\x01\x66\x44",
  [0x3908] = "\x01\x66\x43", [0x3909] = "\x02\x47\x33",
  [0x390a] = "\x01\x66\x4b", [0x390b] = "\x02\x47\x29",
  [0x390c] = "\x02\x47\x2c", [0x390d] = "\x02\x47\x36",
  [0x390e] = "\x02\x47\x32", [0x3911] = "\x02\x4d\x7b",
  [0x3912] = "\x02\x4d\x70", [0x3913] = "\x01\x66\x42",
  [0x3915] = "\x0e\x44\x46", [0x3916] = "\x0e\x44\x48",
  [0x3918] = "\x01\x6a\x72", [0x3919] = "\x02\x4d\x64",
  [0x391a] = "\x02\x4d\x79", [0x391b] = "\x02\x4d\x65",
  [0x391c] = "\x01\x6a\x6d", [0x391d] = "\x0e\x49\x4f",
  [0x391e] = "\x02\x4d\x62", [0x3920] = "\x02\x4d\x6b",
  [0x3921] = "\x02\x4d\x63", [0x3922] = "\x01\x6a\x6f",
  [0x3923] = "\x02\x4d\x5d", [0x3924] = "\x02\x4d\x78",
  [0x3925] = "\x01\x6a\x70", [0x3926] = "\x02\x4d\x75",
  [0x3927] = "\x02\x4d\x76", [0x3928] = "\x02\x4d\x5e",
  [0x3929] = "\x01\x6a\x75", [0x392a] = "\x02\x4d\x6d",
  [0x392b] = "\x0e\x49\x4a", [0x392c] = "\x02\x4d\x67",
  [0x392d] = "\x02\x4d\x6e", [0x392e] = "\x02\x4d\x61",
  [0x3930] = "\x02\x4d\x7a", [0x3931] = "\x02\x4d\x72",
  [0x3932] = "\x02\x4d\x6c", [0x3933] = "\x02\x4d\x5c",
  [0x3934] = "\x01\x6a\x71", [0x3935] = "\x02\x4d\x73",
  [0x3936] = "\x0e\x49\x45", [0x3937] = "\x01\x6a\x74",
  [0x3938] = "\x02\x4d\x77", [0x3939] = "\x0e\x49\x4c",
  [0x393a] = "\x02\x4d\x71", [0x393b] = "\x01\x6a\x6e",
  [0x393c] = "\x02\x4d\x6f", [0x393d] = "\x0e\x49\x49",
  [0x393e] = "\x02\x4d\x69", [0x393f] = "\x01\x6a\x6c",
  [0x3940] = "\x02\x4d\x60", [0x3941] = "\x02\x4d\x68",
  [0x3942] = "\x02\x4d\x74", [0x3943] = "\x02\x4d\x66",
  [0x3945] = "\x0e\x49\x4e", [0x3946] = "\x02\x4d\x6a",
  [0x3947] = "\x0e\x49\x4b", [0x394a] = "\x0e\x66\x43",
  [0x394c] = "\x01\x6e\x6a", [0x394d] = "\x02\x47\x30",
  [0x394e] = "\x02\x53\x79", [0x394f] = "\x02\x54\x24",
  [0x3950] = "\x02\x53\x78", [0x3951] = "\x02\x53\x74",
  [0x3952] = "\x02\x53\x71", [0x3953] = "\x01\x6e\x6b",
  [0x3954] = "\x02\x53\x6f", [0x3955] = "\x01\x6a\x73",
  [0x3956] = "\x02\x53\x68", [0x3957] = "\x01\x6e\x69",
  [0x3958] = "\x02\x53\x6e", [0x3959] = "\x01\x6e\x68",
  [0x395a] = "\x02\x53\x73", [0x395b] = "\x02\x53\x70",
  [0x395c] = "\x02\x54\x22", [0x395d] = "\x02\x53\x7b",
  [0x395e] = "\x02\x53\x75", [0x395f] = "\x02\x53\x7a",
  [0x3960] = "\x01\x6e\x64", [0x3961] = "\x02\x53\x72",
  [0x3962] = "\x02\x54\x27", [0x3963] = "\x02\x53\x69",
  [0x3964] = "\x02\x53\x6a", [0x3965] = "\x02\x54\x23",
  [0x3966] = "\x01\x6e\x65", [0x3967] = "\x02\x54\x28",
  [0x3968] = "\x01\x6e\x67", [0x3969] = "\x02\x54\x29",
  [0x396a] = "\x02\x53\x77", [0x396b] = "\x02\x4d\x5f",
  [0x396c] = "\x02\x53\x7d", [0x396d] = "\x02\x53\x76",
  [0x396e] = "\x02\x54\x21", [0x396f] = "\x02\x53\x7c",
  [0x3971] = "\x0e\x4e\x5d", [0x3973] = "\x02\x53\x6d",
  [0x3974] = "\x01\x6e\x62", [0x3975] = "\x02\x54\x26",
  [0x3976] = "\x01\x6e\x63", [0x3977] = "\x02\x53\x6b",
  [0x3978] = "\x01\x6e\x66", [0x3979] = "\x02\x5a\x32",
  [0x397a] = "\x02\x53\x7e", [0x397b] = "\x02\x54\x25",
  [0x397d] = "\x0e\x4e\x59", [0x3981] = "\x02\x5a\x2f",
  [0x3982] = "\x01\x6e\x61", [0x3983] = "\x01\x72\x2a",
  [0x3984] = "\x02\x5a\x39", [0x3985] = "\x02\x5a\x35",
  [0x3987] = "\x02\x5a\x33", [0x3988] = "\x02\x5a\x2e",
  [0x3989] = "\x02\x5a\x3d", [0x398a] = "\x0e\x52\x5b",
  [0x398b] = "\x0e\x52\x5d", [0x398d] = "\x01\x72\x2e",
  [0x398e] = "\x0e\x52\x5f", [0x398f] = "\x02\x5a\x2a",
  [0x3990] = "\x02\x5a\x36", [0x3991] = "\x02\x5a\x37",
  [0x3992] = "\x02\x5a\x2d", [0x3993] = "\x02\x5a\x2c",
  [0x3994] = "\x02\x5a\x3a", [0x3996] = "\x02\x5a\x30",
  [0x3997] = "\x02\x5a\x2b", [0x3998] = "\x02\x5a\x31",
  [0x3999] = "\x0e\x52\x62", [0x399a] = "\x02\x5a\x3c",
  [0x399b] = "\x02\x5a\x29", [0x399c] = "\x02\x5a\x3b",
  [0x399d] = "\x02\x5a\x38", [0x399e] = "\x01\x72\x2c",
  [0x399f] = "\x01\x72\x2b", [0x39a1] = "\x0e\x52\x5e",
  [0x39a2] = "\x01\x72\x2d", [0x39a3] = "\x02\x5a\x34",
  [0x39a4] = "\x02\x5a\x28", [0x39a5] = "\x0e\x66\x45",
  [0x39a9] = "\x0e\x66\x44", [0x39aa] = "\x02\x5f\x27",
  [0x39ab] = "\x01\x75\x3c", [0x39ac] = "\x02\x5f\x2b",
  [0x39ad] = "\x02\x5f\x28", [0x39ae] = "\x02\x5f\x2f",
  [0x39af] = "\x02\x5f\x35", [0x39b0] = "\x02\x5f\x2a",
  [0x39b1] = "\x0e\x56\x3a", [0x39b2] = "\x02\x5f\x3e",
  [0x39b3] = "\x01\x75\x39", [0x39b4] = "\x02\x5f\x38",
  [0x39b5] = "\x02\x5f\x2d", [0x39b6] = "\x02\x5f\x39",
  [0x39b7] = "\x02\x5f\x34", [0x39b8] = "\x02\x5f\x3b",
  [0x39b9] = "\x02\x5f\x2c", [0x39ba] = "\x01\x75\x3e",
  [0x39bb] = "\x01\x75\x3d", [0x39bc] = "\x02\x5f\x2e",
  [0x39bd] = "\x02\x5f\x3c", [0x39be] = "\x02\x5f\x26",
  [0x39bf] = "\x02\x5f\x3a", [0x39c0] = "\x01\x75\x37",
  [0x39c1] = "\x0e\x56\x39", [0x39c2] = "\x02\x5f\x32",
  [0x39c3] = "\x02\x5f\x31", [0x39c4] = "\x02\x5f\x36",
  [0x39c5] = "\x02\x5f\x29", [0x39c6] = "\x01\x75\x3b",
  [0x39c7] = "\x0e\x56\x3b", [0x39c8] = "\x01\x75\x3f",
  [0x39c9] = "\x02\x5f\x30", [0x39ca] = "\x02\x5f\x37",
  [0x39cb] = "\x01\x75\x40", [0x39cc] = "\x02\x5f\x33",
  [0x39cd] = "\x0e\x56\x36", [0x39ce] = "\x0e\x56\x34",
  [0x39d1] = "\x01\x75\x38", [0x39d2] = "\x01\x75\x3a",
  [0x39d3] = "\x02\x63\x33", [0x39d4] = "\x02\x63\x31",
  [0x39d5] = "\x0e\x5b\x28", [0x39d6] = "\x0e\x58\x76",
  [0x39d7] = "\x02\x63\x37", [0x39d8] = "\x02\x63\x35",
  [0x39d9] = "\x02\x63\x38", [0x39da] = "\x0e\x58\x78",
  [0x39db] = "\x02\x63\x2a", [0x39dc] = "\x02\x63\x32",
  [0x39dd] = "\x02\x63\x3c", [0x39de] = "\x02\x5f\x3d",
  [0x39df] = "\x02\x63\x2e", [0x39e0] = "\x01\x77\x3a",
  [0x39e1] = "\x02\x53\x6c", [0x39e2] = "\x02\x63\x29",
  [0x39e3] = "\x02\x63\x36", [0x39e4] = "\x02\x63\x30",
  [0x39e5] = "\x02\x63\x2d", [0x39e6] = "\x02\x63\x28",
  [0x39e7] = "\x02\x63\x27", [0x39e8] = "\x02\x63\x3b",
  [0x39e9] = "\x0e\x58\x73", [0x39ea] = "\x02\x63\x2c",
  [0x39eb] = "\x02\x63\x2b", [0x39ec] = "\x01\x77\x38",
  [0x39ed] = "\x02\x63\x34", [0x39ee] = "\x0e\x58\x74",
  [0x39ef] = "\x01\x77\x37", [0x39f0] = "\x0e\x58\x75",
  [0x39f2] = "\x01\x77\x39", [0x39f3] = "\x02\x63\x2f",
  [0x39f4] = "\x02\x63\x3a", [0x39f5] = "\x0e\x66\x46",
  [0x39f6] = "\x02\x66\x69", [0x39f7] = "\x02\x66\x6a",
  [0x39f8] = "\x0e\x5b\x2c", [0x39f9] = "\x01\x78\x74",
  [0x39fa] = "\x02\x66\x67", [0x39fb] = "\x01\x78\x71",
  [0x39fc] = "\x02\x66\x6f", [0x39fd] = "\x0e\x5b\x27",
  [0x39fe] = "\x01\x78\x75", [0x39ff] = "\x02\x66\x71",
  [0x3a00] = "\x02\x66\x66", [0x3a01] = "\x02\x63\x39",
  [0x3a02] = "\x02\x66\x73", [0x3a03] = "\x02\x66\x68",
  [0x3a05] = "\x01\x78\x72", [0x3a06] = "\x02\x66\x6e",
  [0x3a07] = "\x0e\x5b\x29", [0x3a08] = "\x02\x66\x70",
  [0x3a09] = "\x02\x66\x6b", [0x3a0a] = "\x02\x66\x72",
  [0x3a0b] = "\x02\x66\x6d", [0x3a0c] = "\x02\x66\x6c",
  [0x3a0d] = "\x01\x78\x73", [0x3a0e] = "\x0e\x58\x77",
  [0x3a0f] = "\x0e\x5b\x2b", [0x3a10] = "\x02\x69\x71",
  [0x3a11] = "\x02\x69\x72", [0x3a12] = "\x0e\x5d\x25",
  [0x3a13] = "\x02\x69\x74", [0x3a14] = "\x01\x7a\x39",
  [0x3a15] = "\x01\x7a\x3a", [0x3a16] = "\x02\x69\x75",
  [0x3a17] = "\x02\x69\x73", [0x3a18] = "\x0e\x5d\x24",
  [0x3a19] = "\x02\x69\x70", [0x3a1a] = "\x0e\x5e\x5d",
  [0x3a1b] = "\x02\x6c\x31", [0x3a1c] = "\x02\x6c\x34",
  [0x3a1d] = "\x02\x6c\x30", [0x3a1f] = "\x01\x7b\x27",
  [0x3a20] = "\x02\x6c\x32", [0x3a21] = "\x01\x7b\x26",
  [0x3a22] = "\x01\x7b\x25", [0x3a23] = "\x01\x7b\x24",
  [0x3a24] = "\x02\x6c\x33", [0x3a25] = "\x02\x6d\x7e",
  [0x3a26] = "\x02\x6d\x7c", [0x3a27] = "\x0e\x5f\x5f",
  [0x3a28] = "\x02\x6d\x7b", [0x3a29] = "\x02\x6c\x2f",
  [0x3a2a] = "\x02\x6d\x7d", [0x3a2b] = "\x02\x6c\x35",
  [0x3a2c] = "\x02\x6d\x7a", [0x3a2d] = "\x0e\x60\x45",
  [0x3a2e] = "\x02\x6f\x48", [0x3a2f] = "\x02\x6f\x26",
  [0x3a30] = "\x02\x6f\x46", [0x3a31] = "\x01\x7c\x47",
  [0x3a32] = "\x02\x6f\x47", [0x3a33] = "\x02\x6f\x49",
  [0x3a34] = "\x0e\x66\x47", [0x3a35] = "\x02\x70\x4d",
  [0x3a36] = "\x01\x7c\x66", [0x3a37] = "\x02\x70\x4c",
  [0x3a38] = "\x02\x70\x4b", [0x3a39] = "\x01\x7c\x67",
  [0x3a3b] = "\x01\x7d\x27", [0x3a3c] = "\x02\x71\x5d",
  [0x3a3d] = "\x02\x71\x75", [0x3a3e] = "\x02\x71\x74",
  [0x3a3f] = "\x02\x71\x76", [0x3a40] = "\x01\x48\x66",
  [0x3a41] = "\x02\x2d\x4b", [0x3a42] = "\x0e\x2f\x26",
  [0x3a43] = "\x02\x32\x71", [0x3a44] = "\x02\x32\x72",
  [0x3a45] = "\x0e\x38\x6a", [0x3a46] = "\x0e\x3e\x6a",
  [0x3a47] = "\x0e\x3e\x69", [0x3a48] = "\x02\x40\x26",
  [0x3a4a] = "\x02\x6c\x36", [0x3a4b] = "\x02\x70\x4e",
  [0x3a4c] = "\x01\x48\x67", [0x3a4d] = "\x01\x53\x4f",
  [0x3a4e] = "\x02\x2d\x4c", [0x3a4f] = "\x0e\x33\x71",
  [0x3a50] = "\x0e\x66\x48", [0x3a52] = "\x02\x39\x34",
  [0x3a53] = "\x01\x5d\x34", [0x3a55] = "\x02\x40\x28",
  [0x3a56] = "\x02\x40\x27", [0x3a57] = "\x01\x61\x79",
  [0x3a58] = "\x0e\x44\x4d", [0x3a59] = "\x01\x66\x4c",
  [0x3a5a] = "\x02\x54\x2a", [0x3a5b] = "\x01\x6e\x6c",
  [0x3a5c] = "\x0e\x4e\x5f", [0x3a5d] = "\x01\x6e\x6d",
  [0x3a5e] = "\x0e\x52\x63", [0x3a5f] = "\x0e\x52\x64",
  [0x3a61] = "\x01\x72\x2f", [0x3a62] = "\x01\x7c\x68",
  [0x3a63] = "\x01\x48\x68", [0x3a66] = "\x0e\x2f\x27",
  [0x3a67] = "\x02\x2d\x4d", [0x3a68] = "\x01\x4f\x50",
  [0x3a69] = "\x02\x2d\x4f", [0x3a6a] = "\x02\x2d\x4e",
  [0x3a6b] = "\x01\x53\x50", [0x3a6d] = "\x02\x32\x73",
  [0x3a6e] = "\x0e\x33\x74", [0x3a6f] = "\x02\x32\x7a",
  [0x3a70] = "\x01\x58\x2e", [0x3a71] = "\x02\x32\x78",
  [0x3a72] = "\x02\x32\x76", [0x3a73] = "\x0e\x33\x77",
  [0x3a74] = "\x02\x32\x7d", [0x3a75] = "\x02\x32\x74",
  [0x3a76] = "\x02\x32\x75", [0x3a77] = "\x01\x58\x2f",
  [0x3a78] = "\x0e\x33\x72", [0x3a79] = "\x01\x58\x33",
  [0x3a7a] = "\x0e\x33\x73", [0x3a7b] = "\x0e\x33\x75",
  [0x3a7c] = "\x02\x32\x7e", [0x3a7d] = "\x01\x58\x32",
  [0x3a7e] = "\x02\x32\x7c", [0x3a7f] = "\x02\x32\x79",
  [0x3a80] = "\x02\x32\x77", [0x3a81] = "\x01\x58\x30",
  [0x3a82] = "\x01\x58\x31", [0x3a83] = "\x02\x32\x7b",
  [0x3a84] = "\x0e\x33\x76", [0x3a87] = "\x0e\x66\x49",
  [0x3a88] = "\x01\x5d\x36", [0x3a89] = "\x02\x39\x35",
  [0x3a8a] = "\x0e\x38\x72", [0x3a8b] = "\x01\x5d\x3b",
  [0x3a8c] = "\x02\x39\x45", [0x3a8d] = "\x01\x5d\x3a",
  [0x3a8e] = "\x02\x39\x47", [0x3a8f] = "\x0e\x38\x6e",
  [0x3a90] = "\x0e\x38\x74", [0x3a91] = "\x02\x39\x3b",
  [0x3a92] = "\x01\x5d\x38", [0x3a93] = "\x02\x39\x46",
  [0x3a94] = "\x0e\x38\x6c", [0x3a95] = "\x02\x39\x36",
  [0x3a96] = "\x01\x5d\x39", [0x3a97] = "\x02\x39\x42",
  [0x3a98] = "\x02\x39\x3e", [0x3a99] = "\x02\x39\x40",
  [0x3a9a] = "\x02\x39\x3a", [0x3a9b] = "\x02\x39\x41",
  [0x3a9c] = "\x0e\x38\x6b", [0x3a9e] = "\x01\x5d\x35",
  [0x3a9f] = "\x02\x39\x3d", [0x3aa0] = "\x0e\x38\x73",
  [0x3aa1] = "\x02\x39\x3c", [0x3aa2] = "\x02\x39\x38",
  [0x3aa3] = "\x0e\x38\x6d", [0x3aa4] = "\x02\x39\x43",
  [0x3aa5] = "\x0e\x38\x6f", [0x3aa6] = "\x0e\x38\x71",
  [0x3aa7] = "\x02\x39\x3f", [0x3aa8] = "\x02\x39\x37",
  [0x3aa9] = "\x0e\x38\x70", [0x3aaa] = "\x02\x39\x39",
  [0x3aab] = "\x01\x5d\x37", [0x3aac] = "\x02\x39\x44",
  [0x3ab1] = "\x01\x61\x7c", [0x3ab2] = "\x02\x40\x33",
  [0x3ab4] = "\x0e\x3e\x70", [0x3ab5] = "\x0e\x3e\x72",
  [0x3ab6] = "\x02\x40\x2f", [0x3ab7] = "\x02\x40\x31",
  [0x3ab8] = "\x02\x40\x2c", [0x3ab9] = "\x02\x40\x2b",
  [0x3aba] = "\x02\x40\x29", [0x3abb] = "\x0e\x3e\x6d",
  [0x3abc] = "\x02\x40\x30", [0x3abd] = "\x02\x40\x32",
  [0x3abe] = "\x02\x40\x2e", [0x3abf] = "\x0e\x3e\x6f",
  [0x3ac0] = "\x02\x40\x2d", [0x3ac1] = "\x01\x61\x7a",
  [0x3ac2] = "\x01\x61\x7b", [0x3ac9] = "\x02\x40\x35",
  [0x3aca] = "\x01\x66\x54", [0x3acb] = "\x02\x47\x39",
  [0x3acc] = "\x02\x47\x3f", [0x3acd] = "\x02\x47\x3a",
  [0x3ace] = "\x02\x47\x3b", [0x3acf] = "\x0e\x44\x4e",
  [0x3ad0] = "\x02\x47\x40", [0x3ad2] = "\x01\x66\x56",
  [0x3ad4] = "\x01\x66\x4e", [0x3ad5] = "\x01\x66\x55",
  [0x3ad6] = "\x02\x47\x38", [0x3ad7] = "\x02\x40\x2a",
  [0x3ad8] = "\x01\x66\x51", [0x3ad9] = "\x01\x66\x4f",
  [0x3ada] = "\x02\x47\x3e", [0x3adb] = "\x02\x47\x3d",
  [0x3adc] = "\x01\x66\x50", [0x3add] = "\x01\x66\x52",
  [0x3ade] = "\x02\x47\x3c", [0x3adf] = "\x01\x66\x4d",
  [0x3ae0] = "\x0e\x44\x4f", [0x3ae1] = "\x01\x66\x53",
  [0x3ae6] = "\x0e\x4e\x60", [0x3ae7] = "\x02\x4d\x7d",
  [0x3ae8] = "\x01\x6a\x7c", [0x3ae9] = "\x0e\x49\x59",
  [0x3aea] = "\x0e\x49\x52", [0x3aeb] = "\x02\x4e\x2a",
  [0x3aec] = "\x02\x4e\x29", [0x3aed] = "\x0e\x49\x57",
  [0x3aee] = "\x02\x4e\x24", [0x3aef] = "\x01\x6a\x7e",
  [0x3af0] = "\x02\x4e\x28", [0x3af1] = "\x02\x4d\x7e",
  [0x3af2] = "\x02\x4e\x21", [0x3af3] = "\x01\x6a\x76",
  [0x3af4] = "\x01\x6a\x78", [0x3af5] = "\x0e\x49\x54",
  [0x3af6] = "\x02\x4e\x26", [0x3af7] = "\x02\x4d\x7c",
  [0x3af8] = "\x01\x6a\x7a", [0x3af9] = "\x01\x6a\x79",
  [0x3afa] = "\x02\x4e\x22", [0x3afb] = "\x02\x4e\x27",
  [0x3afc] = "\x02\x4e\x25", [0x3afd] = "\x01\x6a\x7b",
  [0x3afe] = "\x02\x4e\x23", [0x3aff] = "\x0e\x49\x51",
  [0x3b00] = "\x0e\x49\x56", [0x3b01] = "\x02\x40\x34",
  [0x3b02] = "\x01\x6a\x77", [0x3b03] = "\x0e\x49\x58",
  [0x3b05] = "\x02\x54\x2b", [0x3b06] = "\x02\x54\x32",
  [0x3b07] = "\x01\x6e\x6f", [0x3b09] = "\x02\x54\x36",
  [0x3b0a] = "\x01\x6e\x73", [0x3b0b] = "\x02\x54\x2e",
  [0x3b0c] = "\x02\x54\x2c", [0x3b0e] = "\x02\x54\x35",
  [0x3b0f] = "\x0e\x4e\x61", [0x3b10] = "\x01\x6e\x6e",
  [0x3b11] = "\x02\x54\x34", [0x3b12] = "\x01\x6e\x70",
  [0x3b13] = "\x01\x6e\x71", [0x3b14] = "\x02\x54\x2d",
  [0x3b15] = "\x01\x6e\x72", [0x3b16] = "\x02\x54\x33",
  [0x3b17] = "\x02\x54\x2f", [0x3b18] = "\x02\x54\x30",
  [0x3b19] = "\x02\x54\x31", [0x3b1a] = "\x01\x6a\x7d",
  [0x3b1b] = "\x0e\x4e\x62", [0x3b1e] = "\x02\x5a\x3e",
  [0x3b1f] = "\x02\x5a\x4a", [0x3b21] = "\x01\x72\x34",
  [0x3b22] = "\x02\x5a\x45", [0x3b23] = "\x02\x5a\x47",
  [0x3b24] = "\x0e\x52\x65", [0x3b25] = "\x01\x72\x32",
  [0x3b26] = "\x02\x5a\x3f", [0x3b27] = "\x02\x5a\x43",
  [0x3b29] = "\x02\x5a\x46", [0x3b2a] = "\x01\x72\x30",
  [0x3b2b] = "\x01\x72\x33", [0x3b2c] = "\x02\x5a\x49",
  [0x3b2d] = "\x02\x5a\x41", [0x3b2e] = "\x02\x5a\x42",
  [0x3b2f] = "\x02\x5a\x48", [0x3b30] = "\x02\x5a\x40",
  [0x3b31] = "\x02\x5a\x44", [0x3b32] = "\x01\x72\x31",
  [0x3b33] = "\x02\x5f\x40", [0x3b35] = "\x02\x5f\x3f",
  [0x3b36] = "\x01\x75\x42", [0x3b37] = "\x02\x5f\x45",
  [0x3b38] = "\x01\x75\x44", [0x3b39] = "\x0e\x56\x40",
  [0x3b3b] = "\x01\x75\x41", [0x3b3c] = "\x02\x5f\x41",
  [0x3b3d] = "\x01\x75\x45", [0x3b3e] = "\x02\x5f\x42",
  [0x3b3f] = "\x0e\x56\x3f", [0x3b40] = "\x0e\x56\x3d",
  [0x3b41] = "\x02\x5f\x43", [0x3b42] = "\x02\x5f\x46",
  [0x3b43] = "\x0e\x56\x42", [0x3b44] = "\x01\x75\x43",
  [0x3b46] = "\x02\x63\x41", [0x3b48] = "\x0e\x58\x79",
  [0x3b49] = "\x02\x63\x44", [0x3b4a] = "\x0e\x58\x7a",
  [0x3b4b] = "\x02\x63\x3e", [0x3b4c] = "\x02\x63\x40",
  [0x3b4d] = "\x0e\x58\x7c", [0x3b4e] = "\x0e\x58\x7b",
  [0x3b4f] = "\x02\x63\x3f", [0x3b50] = "\x02\x63\x42",
  [0x3b51] = "\x02\x63\x43", [0x3b52] = "\x02\x5f\x44",
  [0x3b53] = "\x02\x63\x3d", [0x3b54] = "\x0e\x66\x4a",
  [0x3b56] = "\x01\x78\x78", [0x3b57] = "\x02\x66\x77",
  [0x3b58] = "\x02\x66\x7a", [0x3b59] = "\x02\x66\x7c",
  [0x3b5a] = "\x02\x66\x75", [0x3b5b] = "\x02\x66\x76",
  [0x3b5c] = "\x02\x66\x79", [0x3b5d] = "\x02\x66\x7b",
  [0x3b5e] = "\x01\x78\x79", [0x3b5f] = "\x01\x78\x77",
  [0x3b60] = "\x01\x78\x76", [0x3b61] = "\x02\x66\x78",
  [0x3b62] = "\x02\x66\x74", [0x3b63] = "\x02\x69\x76",
  [0x3b64] = "\x01\x7a\x3b", [0x3b65] = "\x0e\x5d\x29",
  [0x3b66] = "\x02\x69\x77", [0x3b67] = "\x0e\x5d\x28",
  [0x3b69] = "\x02\x6c\x38", [0x3b6a] = "\x01\x7b\x28",
  [0x3b6b] = "\x02\x6c\x3a", [0x3b6c] = "\x01\x7b\x29",
  [0x3b6d] = "\x02\x6c\x37", [0x3b6e] = "\x02\x6c\x39",
  [0x3b6f] = "\x01\x7b\x72", [0x3b71] = "\x02\x6e\x21",
  [0x3b72] = "\x01\x7b\x71", [0x3b73] = "\x02\x6f\x4c",
  [0x3b74] = "\x02\x6f\x4b", [0x3b76] = "\x02\x6f\x4a",
  [0x3b78] = "\x0e\x61\x49", [0x3b79] = "\x02\x71\x37",
  [0x3b7a] = "\x02\x71\x38", [0x3b7b] = "\x02\x71\x3a",
  [0x3b7c] = "\x02\x71\x39", [0x3b7d] = "\x0e\x61\x7d",
  [0x3b7e] = "\x02\x22\x70", [0x3b7f] = "\x01\x48\x69",
  [0x3b81] = "\x01\x53\x51", [0x3b82] = "\x02\x39\x48",
  [0x3b83] = "\x01\x61\x7d", [0x3b84] = "\x0e\x66\x4b",
  [0x3b85] = "\x02\x47\x41", [0x3b86] = "\x01\x77\x3b",
  [0x3b87] = "\x0e\x5b\x2f", [0x3b88] = "\x02\x66\x7d",
  [0x3b89] = "\x0e\x60\x46", [0x3b8a] = "\x0e\x61\x4a",
  [0x3b8b] = "\x01\x4b\x44", [0x3b8c] = "\x0e\x2f\x29",
  [0x3b8f] = "\x01\x5d\x3d", [0x3b91] = "\x0e\x38\x76",
  [0x3b92] = "\x0e\x38\x75", [0x3b93] = "\x01\x5d\x3c",
  [0x3b94] = "\x0e\x38\x77", [0x3b95] = "\x02\x40\x36",
  [0x3b96] = "\x01\x61\x7e", [0x3b97] = "\x02\x40\x38",
  [0x3b98] = "\x02\x40\x37", [0x3b9a] = "\x0e\x3e\x74",
  [0x3b9b] = "\x02\x47\x42", [0x3b9c] = "\x01\x66\x57",
  [0x3b9d] = "\x02\x4e\x2b", [0x3b9e] = "\x02\x4e\x2e",
  [0x3b9f] = "\x02\x4e\x2d", [0x3ba1] = "\x02\x4e\x2c",
  [0x3ba2] = "\x02\x54\x37", [0x3ba3] = "\x02\x54\x39",
  [0x3ba4] = "\x02\x54\x38", [0x3ba5] = "\x0e\x4e\x65",
  [0x3ba6] = "\x01\x72\x36", [0x3ba7] = "\x0e\x52\x66",
  [0x3ba9] = "\x0e\x4e\x64", [0x3baa] = "\x01\x72\x35",
  [0x3bab] = "\x0e\x56\x46", [0x3bac] = "\x01\x75\x46",
  [0x3bad] = "\x02\x5f\x47", [0x3bae] = "\x02\x5f\x49",
  [0x3baf] = "\x02\x5f\x48", [0x3bb0] = "\x0e\x58\x7e",
  [0x3bb1] = "\x0e\x58\x7d", [0x3bb2] = "\x01\x77\x3c",
  [0x3bb3] = "\x0e\x59\x21", [0x3bb5] = "\x0e\x5b\x31",
  [0x3bb6] = "\x02\x67\x21", [0x3bb7] = "\x02\x66\x7e",
  [0x3bb9] = "\x02\x69\x78", [0x3bba] = "\x01\x7a\x3c",
  [0x3bbb] = "\x0e\x5d\x2a", [0x3bbc] = "\x0e\x5e\x61",
  [0x3bbd] = "\x01\x7b\x2a", [0x3bbe] = "\x02\x6e\x23",
  [0x3bbf] = "\x02\x6e\x22", [0x3bc0] = "\x01\x7d\x28",
  [0x3bd2] = "\x01\x4b\x45", [0x3bd3] = "\x02\x2d\x50",
  [0x3bd4] = "\x01\x53\x52", [0x3bd5] = "\x02\x39\x4b",
  [0x3bd6] = "\x02\x39\x49", [0x3bd9] = "\x02\x39\x4a",
  [0x3bda] = "\x02\x40\x3a", [0x3bdb] = "\x02\x40\x3b",
  [0x3bdc] = "\x02\x47\x49", [0x3bdd] = "\x02\x40\x39",
  [0x3bdf] = "\x02\x47\x43", [0x3be0] = "\x02\x47\x47",
  [0x3be1] = "\x02\x47\x46", [0x3be2] = "\x02\x47\x48",
  [0x3be3] = "\x01\x66\x58", [0x3be4] = "\x02\x47\x45",
  [0x3be5] = "\x02\x47\x44", [0x3be6] = "\x02\x47\x4a",
  [0x3be7] = "\x0e\x44\x54", [0x3be8] = "\x02\x4e\x31",
  [0x3be9] = "\x02\x4e\x2f", [0x3bea] = "\x0e\x49\x5c",
  [0x3beb] = "\x02\x4e\x30", [0x3bec] = "\x02\x54\x3c",
  [0x3bed] = "\x02\x54\x3a", [0x3bee] = "\x0e\x4e\x66",
  [0x3bf0] = "\x02\x54\x3b", [0x3bf1] = "\x02\x5a\x4b",
  [0x3bf2] = "\x02\x5f\x4a", [0x3bf3] = "\x02\x5f\x4b",
  [0x3bf4] = "\x01\x77\x3d", [0x3bf5] = "\x0e\x5b\x32",
  [0x3bf6] = "\x02\x67\x22", [0x3bf7] = "\x02\x69\x79",
  [0x3bf8] = "\x01\x7a\x3d", [0x3bfa] = "\x02\x6c\x3b",
  [0x3bfb] = "\x02\x6e\x24", [0x3bfc] = "\x01\x7b\x73",
  [0x3bfe] = "\x02\x6f\x4d", [0x3bff] = "\x02\x71\x3b",
  [0x3c00] = "\x01\x4b\x46", [0x3c02] = "\x01\x53\x54",
  [0x3c03] = "\x01\x53\x55", [0x3c04] = "\x02\x2d\x51",
  [0x3c05] = "\x0e\x2f\x2a", [0x3c06] = "\x0e\x2f\x2c",
  [0x3c07] = "\x02\x2d\x52", [0x3c08] = "\x01\x53\x53",
  [0x3c0a] = "\x01\x58\x39", [0x3c0b] = "\x0e\x33\x7b",
  [0x3c0c] = "\x01\x58\x37", [0x3c0d] = "\x0e\x33\x7a",
  [0x3c0e] = "\x01\x58\x36", [0x3c0f] = "\x01\x58\x3d",
  [0x3c10] = "\x01\x58\x35", [0x3c11] = "\x01\x58\x3e",
  [0x3c12] = "\x02\x33\x21", [0x3c13] = "\x01\x58\x3b",
  [0x3c15] = "\x01\x58\x38", [0x3c16] = "\x01\x58\x3c",
  [0x3c17] = "\x01\x58\x3a", [0x3c18] = "\x01\x58\x34",
  [0x3c19] = "\x0e\x33\x7c", [0x3c1b] = "\x01\x5d\x45",
  [0x3c1c] = "\x0e\x38\x7e", [0x3c1d] = "\x01\x5d\x3f",
  [0x3c1e] = "\x02\x39\x4f", [0x3c1f] = "\x01\x5d\x44",
  [0x3c20] = "\x0e\x39\x23", [0x3c21] = "\x0e\x39\x29",
  [0x3c22] = "\x01\x5d\x46", [0x3c23] = "\x01\x5d\x40",
  [0x3c25] = "\x01\x5d\x41", [0x3c26] = "\x0e\x38\x79",
  [0x3c27] = "\x02\x39\x4d", [0x3c28] = "\x0e\x38\x7b",
  [0x3c29] = "\x0e\x39\x25", [0x3c2a] = "\x01\x5d\x3e",
  [0x3c2b] = "\x0e\x39\x22", [0x3c2c] = "\x02\x39\x4e",
  [0x3c2d] = "\x01\x5d\x43", [0x3c2e] = "\x0e\x63\x30",
  [0x3c30] = "\x02\x39\x4c", [0x3c31] = "\x01\x5d\x42",
  [0x3c32] = "\x0e\x38\x7a", [0x3c34] = "\x01\x62\x2b",
  [0x3c35] = "\x0e\x3e\x7c", [0x3c36] = "\x01\x62\x2d",
  [0x3c38] = "\x0e\x3e\x7d", [0x3c39] = "\x02\x40\x3e",
  [0x3c3a] = "\x01\x62\x2c", [0x3c3b] = "\x01\x62\x21",
  [0x3c3c] = "\x01\x62\x25", [0x3c3d] = "\x0e\x3f\x24",
  [0x3c3e] = "\x01\x66\x6b", [0x3c3f] = "\x02\x47\x4f",
  [0x3c40] = "\x02\x40\x40", [0x3c41] = "\x01\x62\x26",
  [0x3c42] = "\x0e\x3e\x7e", [0x3c43] = "\x0e\x3e\x75",
  [0x3c44] = "\x02\x40\x43", [0x3c45] = "\x02\x40\x44",
  [0x3c46] = "\x01\x62\x2a", [0x3c48] = "\x02\x40\x46",
  [0x3c49] = "\x0e\x3f\x21", [0x3c4a] = "\x02\x40\x48",
  [0x3c4b] = "\x0e\x3f\x23", [0x3c4c] = "\x02\x40\x49",
  [0x3c4d] = "\x02\x40\x3d", [0x3c4e] = "\x02\x40\x3c",
  [0x3c4f] = "\x02\x40\x4a", [0x3c50] = "\x01\x62\x29",
  [0x3c51] = "\x02\x40\x47", [0x3c52] = "\x02\x40\x45",
  [0x3c54] = "\x01\x62\x27", [0x3c55] = "\x01\x62\x23",
  [0x3c56] = "\x01\x62\x2e", [0x3c57] = "\x02\x40\x41",
  [0x3c58] = "\x02\x40\x42", [0x3c59] = "\x02\x40\x3f",
  [0x3c5a] = "\x0e\x3e\x78", [0x3c5b] = "\x01\x62\x28",
  [0x3c5d] = "\x0e\x3f\x22", [0x3c5e] = "\x01\x62\x24",
  [0x3c60] = "\x01\x62\x22", [0x3c61] = "\x02\x47\x4e",
  [0x3c62] = "\x01\x66\x66", [0x3c63] = "\x01\x66\x61",
  [0x3c64] = "\x0e\x44\x60", [0x3c65] = "\x0e\x44\x59",
  [0x3c66] = "\x01\x66\x5c", [0x3c67] = "\x0e\x44\x63",
  [0x3c68] = "\x01\x66\x6c", [0x3c69] = "\x01\x66\x5d",
  [0x3c6a] = "\x0e\x44\x55", [0x3c6b] = "\x01\x66\x59",
  [0x3c6c] = "\x01\x66\x68", [0x3c6d] = "\x01\x66\x65",
  [0x3c6e] = "\x01\x66\x67", [0x3c6f] = "\x0e\x44\x58",
  [0x3c70] = "\x01\x66\x5e", [0x3c71] = "\x01\x66\x63",
  [0x3c72] = "\x01\x66\x5a", [0x3c73] = "\x01\x66\x5b",
  [0x3c74] = "\x02\x47\x56", [0x3c75] = "\x02\x47\x53",
  [0x3c76] = "\x02\x47\x4b", [0x3c77] = "\x02\x47\x50",
  [0x3c78] = "\x0e\x44\x5b", [0x3c79] = "\x01\x66\x69",
  [0x3c7a] = "\x02\x47\x57", [0x3c7b] = "\x01\x66\x6a",
  [0x3c7c] = "\x01\x66\x60", [0x3c7d] = "\x0e\x44\x5c",
  [0x3c7e] = "\x0e\x44\x61", [0x3c7f] = "\x02\x47\x4d",
  [0x3c80] = "\x0e\x44\x56", [0x3c81] = "\x02\x47\x55",
  [0x3c82] = "\x02\x47\x51", [0x3c83] = "\x02\x47\x54",
  [0x3c84] = "\x02\x47\x52", [0x3c85] = "\x01\x66\x64",
  [0x3c86] = "\x02\x47\x4c", [0x3c87] = "\x01\x66\x5f",
  [0x3c88] = "\x0e\x44\x5d", [0x3c8b] = "\x02\x4e\x34",
  [0x3c8c] = "\x01\x6b\x22", [0x3c8d] = "\x01\x6b\x25",
  [0x3c8e] = "\x0e\x49\x5f", [0x3c8f] = "\x02\x4e\x36",
  [0x3c90] = "\x0e\x49\x64", [0x3c91] = "\x01\x6b\x2d",
  [0x3c92] = "\x02\x4e\x35", [0x3c93] = "\x01\x6b\x27",
  [0x3c94] = "\x0e\x49\x63", [0x3c95] = "\x01\x6e\x78",
  [0x3c96] = "\x02\x4e\x37", [0x3c98] = "\x01\x6b\x2c",
  [0x3c99] = "\x02\x4e\x33", [0x3c9a] = "\x01\x6b\x2e",
  [0x3c9b] = "\x0e\x49\x5e", [0x3c9c] = "\x0e\x49\x62",
  [0x3c9d] = "\x0e\x49\x6b", [0x3c9e] = "\x01\x6b\x23",
  [0x3c9f] = "\x0e\x49\x68", [0x3ca0] = "\x01\x66\x62",
  [0x3ca1] = "\x01\x6b\x26", [0x3ca2] = "\x0e\x49\x60",
  [0x3ca3] = "\x01\x6b\x24", [0x3ca4] = "\x01\x6b\x28",
  [0x3ca5] = "\x01\x6b\x2a", [0x3ca6] = "\x01\x6b\x21",
  [0x3ca7] = "\x01\x6b\x2f", [0x3ca8] = "\x01\x6b\x2b",
  [0x3ca9] = "\x0e\x49\x65", [0x3caa] = "\x01\x6b\x29",
  [0x3cab] = "\x02\x4e\x32", [0x3cac] = "\x0e\x49\x66",
  [0x3caf] = "\x0e\x4e\x74", [0x3cb0] = "\x01\x6f\x21",
  [0x3cb1] = "\x0e\x4e\x67", [0x3cb2] = "\x01\x6e\x7b",
  [0x3cb3] = "\x0e\x4e\x72", [0x3cb4] = "\x0e\x4e\x70",
  [0x3cb5] = "\x0e\x4e\x6e", [0x3cb6] = "\x01\x6f\x24",
  [0x3cb7] = "\x0e\x4e\x6c", [0x3cb8] = "\x02\x54\x3f",
  [0x3cb9] = "\x01\x6f\x25", [0x3cba] = "\x02\x54\x4b",
  [0x3cbb] = "\x02\x54\x44", [0x3cbc] = "\x01\x6e\x74",
  [0x3cbd] = "\x02\x54\x4c", [0x3cbe] = "\x02\x54\x46",
  [0x3cbf] = "\x01\x6e\x7e", [0x3cc0] = "\x02\x54\x47",
  [0x3cc1] = "\x0e\x4e\x73", [0x3cc2] = "\x01\x6e\x7d",
  [0x3cc3] = "\x02\x54\x4a", [0x3cc4] = "\x01\x6e\x77",
  [0x3cc5] = "\x02\x54\x48", [0x3cc6] = "\x02\x54\x3e",
  [0x3cc7] = "\x01\x6e\x76", [0x3cc8] = "\x02\x5a\x56",
  [0x3cc9] = "\x01\x6e\x7c", [0x3cca] = "\x0e\x4e\x75",
  [0x3ccb] = "\x01\x6e\x79", [0x3ccc] = "\x0e\x4e\x69",
  [0x3ccd] = "\x01\x6f\x23", [0x3cce] = "\x0e\x4e\x6a",
  [0x3ccf] = "\x02\x54\x3d", [0x3cd0] = "\x0e\x4e\x76",
  [0x3cd1] = "\x02\x54\x41", [0x3cd2] = "\x01\x6e\x75",
  [0x3cd3] = "\x02\x54\x40", [0x3cd4] = "\x02\x54\x42",
  [0x3cd5] = "\x02\x54\x43", [0x3cd6] = "\x01\x6f\x22",
  [0x3cd7] = "\x02\x54\x45", [0x3cd8] = "\x02\x54\x49",
  [0x3cd9] = "\x02\x54\x4d", [0x3cdb] = "\x01\x6f\x26",
  [0x3cdc] = "\x01\x72\x3c", [0x3cdd] = "\x02\x5a\x51",
  [0x3cde] = "\x02\x5a\x57", [0x3cdf] = "\x02\x5a\x54",
  [0x3ce0] = "\x02\x5a\x4c", [0x3ce1] = "\x02\x5a\x58",
  [0x3ce2] = "\x02\x5a\x4d", [0x3ce3] = "\x0e\x52\x6a",
  [0x3ce4] = "\x02\x5a\x53", [0x3ce5] = "\x0e\x52\x6b",
  [0x3ce6] = "\x01\x72\x37", [0x3ce7] = "\x01\x72\x3d",
  [0x3ce8] = "\x02\x5a\x59", [0x3ce9] = "\x0e\x4e\x6f",
  [0x3cea] = "\x0e\x52\x67", [0x3ceb] = "\x01\x72\x39",
  [0x3cec] = "\x0e\x52\x6e", [0x3ced] = "\x01\x72\x43",
  [0x3cee] = "\x01\x72\x3e", [0x3cef] = "\x02\x5a\x5b",
  [0x3cf0] = "\x02\x5a\x55", [0x3cf1] = "\x01\x72\x3a",
  [0x3cf2] = "\x02\x5a\x4e", [0x3cf3] = "\x01\x72\x44",
  [0x3cf4] = "\x02\x5a\x4f", [0x3cf5] = "\x02\x5a\x50",
  [0x3cf6] = "\x01\x72\x45", [0x3cf7] = "\x01\x72\x42",
  [0x3cf8] = "\x01\x6e\x7a", [0x3cf9] = "\x0e\x52\x69",
  [0x3cfa] = "\x01\x72\x38", [0x3cfb] = "\x02\x5a\x5c",
  [0x3cfc] = "\x01\x72\x46", [0x3cfd] = "\x0e\x52\x68",
  [0x3cfe] = "\x01\x72\x3f", [0x3cff] = "\x02\x5a\x5a",
  [0x3d00] = "\x01\x72\x3b", [0x3d01] = "\x01\x72\x40",
  [0x3d02] = "\x01\x72\x41", [0x3d03] = "\x0e\x66\x4c",
  [0x3d04] = "\x01\x75\x4e", [0x3d05] = "\x02\x5f\x50",
  [0x3d06] = "\x02\x5f\x59", [0x3d07] = "\x02\x5f\x56",
  [0x3d08] = "\x02\x5f\x58", [0x3d09] = "\x0e\x56\x49",
  [0x3d0a] = "\x01\x75\x4b", [0x3d0b] = "\x02\x5f\x51",
  [0x3d0c] = "\x0e\x56\x4a", [0x3d0d] = "\x02\x5f\x57",
  [0x3d0e] = "\x01\x75\x47", [0x3d0f] = "\x02\x5f\x53",
  [0x3d10] = "\x01\x75\x4f", [0x3d11] = "\x02\x5f\x4f",
  [0x3d12] = "\x02\x5f\x54", [0x3d13] = "\x02\x5f\x5b",
  [0x3d14] = "\x02\x5a\x52", [0x3d15] = "\x02\x5f\x55",
  [0x3d16] = "\x02\x5f\x4e", [0x3d17] = "\x01\x75\x48",
  [0x3d18] = "\x02\x5f\x4d", [0x3d19] = "\x01\x75\x49",
  [0x3d1a] = "\x02\x5f\x5c", [0x3d1b] = "\x01\x75\x4a",
  [0x3d1c] = "\x02\x5f\x5a", [0x3d1d] = "\x01\x75\x4d",
  [0x3d1e] = "\x02\x5f\x4c", [0x3d1f] = "\x0e\x56\x48",
  [0x3d20] = "\x01\x75\x4c", [0x3d22] = "\x02\x5f\x52",
  [0x3d23] = "\x02\x63\x47", [0x3d24] = "\x02\x63\x55",
  [0x3d25] = "\x02\x63\x50", [0x3d26] = "\x02\x63\x52",
  [0x3d27] = "\x02\x63\x46", [0x3d28] = "\x01\x77\x3e",
  [0x3d29] = "\x0e\x59\x2a", [0x3d2a] = "\x02\x63\x45",
  [0x3d2b] = "\x01\x77\x41", [0x3d2c] = "\x01\x77\x40",
  [0x3d2d] = "\x0e\x59\x26", [0x3d2e] = "\x02\x63\x54",
  [0x3d2f] = "\x02\x63\x4c", [0x3d30] = "\x02\x63\x49",
  [0x3d31] = "\x02\x63\x4f", [0x3d32] = "\x0e\x59\x24",
  [0x3d33] = "\x02\x63\x48", [0x3d34] = "\x0e\x59\x28",
  [0x3d35] = "\x02\x63\x4a", [0x3d36] = "\x02\x63\x53",
  [0x3d37] = "\x02\x63\x51", [0x3d38] = "\x0e\x59\x23",
  [0x3d39] = "\x01\x77\x3f", [0x3d3a] = "\x02\x63\x58",
  [0x3d3b] = "\x02\x63\x56", [0x3d3c] = "\x02\x63\x4d",
  [0x3d3d] = "\x02\x63\x57", [0x3d3e] = "\x02\x63\x4e",
  [0x3d3f] = "\x0e\x5b\x34", [0x3d40] = "\x02\x67\x26",
  [0x3d41] = "\x01\x78\x7a", [0x3d42] = "\x02\x67\x2d",
  [0x3d43] = "\x0e\x5b\x35", [0x3d45] = "\x02\x6c\x3e",
  [0x3d46] = "\x01\x79\x23", [0x3d47] = "\x02\x63\x4b",
  [0x3d48] = "\x02\x67\x24", [0x3d49] = "\x01\x78\x7d",
  [0x3d4a] = "\x02\x67\x25", [0x3d4b] = "\x02\x67\x2a",
  [0x3d4c] = "\x0e\x5b\x36", [0x3d4d] = "\x0e\x5d\x33",
  [0x3d4e] = "\x01\x79\x21", [0x3d4f] = "\x01\x79\x22",
  [0x3d50] = "\x02\x67\x23", [0x3d51] = "\x02\x67\x2c",
  [0x3d52] = "\x02\x67\x2e", [0x3d53] = "\x02\x67\x27",
  [0x3d54] = "\x02\x67\x29", [0x3d55] = "\x02\x67\x2b",
  [0x3d56] = "\x02\x67\x28", [0x3d57] = "\x02\x67\x2f",
  [0x3d58] = "\x01\x78\x7c", [0x3d59] = "\x01\x79\x24",
  [0x3d5a] = "\x01\x78\x7e", [0x3d5c] = "\x01\x78\x7b",
  [0x3d5d] = "\x02\x69\x7c", [0x3d5f] = "\x01\x7a\x42",
  [0x3d60] = "\x02\x69\x7a", [0x3d61] = "\x0e\x5d\x30",
  [0x3d62] = "\x0e\x5d\x2c", [0x3d63] = "\x02\x69\x7e",
  [0x3d64] = "\x0e\x5d\x32", [0x3d65] = "\x02\x6a\x21",
  [0x3d66] = "\x01\x7a\x40", [0x3d67] = "\x02\x6a\x22",
  [0x3d68] = "\x02\x69\x7d", [0x3d69] = "\x0e\x5d\x2b",
  [0x3d6a] = "\x02\x69\x7b", [0x3d6b] = "\x01\x7a\x43",
  [0x3d6c] = "\x01\x7a\x3f", [0x3d6d] = "\x02\x6a\x23",
  [0x3d6e] = "\x0e\x5d\x2e", [0x3d6f] = "\x01\x7a\x41",
  [0x3d70] = "\x01\x7a\x3e", [0x3d71] = "\x0e\x5c\x6d",
  [0x3d73] = "\x0e\x64\x2d", [0x3d74] = "\x01\x7b\x2b",
  [0x3d75] = "\x0e\x5e\x62", [0x3d77] = "\x01\x7b\x2c",
  [0x3d78] = "\x02\x6c\x3d", [0x3d79] = "\x02\x6c\x3c",
  [0x3d7a] = "\x02\x6c\x3f", [0x3d7b] = "\x02\x6c\x40",
  [0x3d7c] = "\x0e\x5e\x63", [0x3d7d] = "\x01\x7b\x2d",
  [0x3d7e] = "\x02\x6e\x25", [0x3d7f] = "\x02\x6e\x2a",
  [0x3d80] = "\x01\x7b\x74", [0x3d81] = "\x0e\x5f\x61",
  [0x3d82] = "\x02\x6e\x27", [0x3d83] = "\x0e\x5f\x62",
  [0x3d84] = "\x02\x6e\x26", [0x3d85] = "\x02\x6e\x29",
  [0x3d86] = "\x02\x6e\x28", [0x3d87] = "\x0e\x60\x48",
  [0x3d88] = "\x02\x6f\x51", [0x3d8a] = "\x01\x7c\x48",
  [0x3d8b] = "\x02\x6f\x50", [0x3d8c] = "\x02\x6f\x4e",
  [0x3d8d] = "\x0e\x60\x47", [0x3d8e] = "\x02\x6f\x4f",
  [0x3d90] = "\x0e\x60\x49", [0x3d91] = "\x0e\x60\x7b",
  [0x3d92] = "\x01\x7c\x6a", [0x3d93] = "\x01\x7c\x69",
  [0x3d94] = "\x02\x70\x4f", [0x3d95] = "\x02\x70\x50",
  [0x3d96] = "\x01\x7c\x6b", [0x3d97] = "\x0e\x61\x4b",
  [0x3d98] = "\x02\x71\x3c", [0x3d99] = "\x02\x71\x3d",
  [0x3d9a] = "\x01\x7d\x34", [0x3d9b] = "\x0e\x61\x6b",
  [0x3d9c] = "\x01\x7d\x3a", [0x3d9d] = "\x0e\x61\x7e",
  [0x3d9e] = "\x02\x71\x77", [0x3d9f] = "\x02\x72\x36",
  [0x3e37] = "\x01\x4b\x47", [0x3e39] = "\x02\x39\x50",
  [0x3e3a] = "\x0e\x39\x2a", [0x3e3b] = "\x02\x39\x51",
  [0x3e3c] = "\x02\x47\x58", [0x3e3d] = "\x02\x4e\x38",
  [0x3e3e] = "\x02\x54\x4e", [0x3e3f] = "\x01\x75\x51",
  [0x3e40] = "\x0e\x56\x4d", [0x3e41] = "\x01\x75\x50",
  [0x3e42] = "\x02\x63\x59", [0x3e43] = "\x02\x67\x30",
  [0x3e44] = "\x0e\x5f\x64", [0x3e45] = "\x02\x6f\x52",
  [0x3e46] = "\x01\x4b\x48", [0x3e47] = "\x02\x33\x22",
  [0x3e48] = "\x01\x58\x3f", [0x3e49] = "\x01\x5d\x47",
  [0x3e4a] = "\x02\x47\x5a", [0x3e4b] = "\x02\x47\x59",
  [0x3e4c] = "\x01\x6f\x27", [0x3e4d] = "\x02\x54\x4f",
  [0x3e4e] = "\x01\x6f\x28", [0x3e4f] = "\x02\x5f\x5d",
  [0x3e50] = "\x01\x77\x42", [0x3e51] = "\x0e\x5d\x34",
  [0x3e52] = "\x0e\x66\x4d", [0x3e53] = "\x0e\x62\x21",
  [0x3e54] = "\x01\x7d\x43", [0x3e55] = "\x01\x4b\x49",
  [0x3e56] = "\x02\x28\x6b", [0x3e57] = "\x02\x33\x23",
  [0x3e58] = "\x0e\x39\x2d", [0x3e5a] = "\x01\x5d\x48",
  [0x3e5b] = "\x0e\x39\x2c", [0x3e5c] = "\x02\x39\x52",
  [0x3e5d] = "\x02\x39\x53", [0x3e5e] = "\x0e\x3f\x2a",
  [0x3e5f] = "\x02\x40\x4b", [0x3e60] = "\x0e\x3f\x29",
  [0x3e61] = "\x01\x62\x2f", [0x3e62] = "\x01\x66\x6d",
  [0x3e63] = "\x0e\x44\x64", [0x3e64] = "\x02\x47\x5c",
  [0x3e65] = "\x02\x47\x5b", [0x3e66] = "\x02\x47\x5d",
  [0x3e67] = "\x0e\x49\x6d", [0x3e68] = "\x02\x4e\x39",
  [0x3e69] = "\x02\x4e\x3a", [0x3e6a] = "\x01\x6b\x30",
  [0x3e6b] = "\x01\x72\x47", [0x3e6c] = "\x01\x6f\x29",
  [0x3e6d] = "\x01\x72\x48", [0x3e6f] = "\x02\x5f\x61",
  [0x3e70] = "\x02\x5f\x5e", [0x3e71] = "\x02\x5f\x60",
  [0x3e72] = "\x02\x5f\x5f", [0x3e73] = "\x01\x75\x52",
  [0x3e74] = "\x0e\x59\x2b", [0x3e75] = "\x02\x63\x5a",
  [0x3e76] = "\x02\x67\x32", [0x3e77] = "\x02\x67\x31",
  [0x3e78] = "\x02\x25\x24", [0x3e79] = "\x01\x58\x41",
  [0x3e7a] = "\x01\x58\x40", [0x3e7b] = "\x02\x33\x24",
  [0x3e7c] = "\x0e\x39\x2e", [0x3e7d] = "\x02\x39\x54",
  [0x3e7e] = "\x0e\x3f\x2c", [0x3e7f] = "\x0e\x3f\x2b",
  [0x3e80] = "\x02\x40\x4d", [0x3e81] = "\x02\x40\x4c",
  [0x3e82] = "\x01\x62\x30", [0x3e83] = "\x0e\x3f\x2d",
  [0x3e84] = "\x02\x47\x5f", [0x3e85] = "\x02\x47\x60",
  [0x3e86] = "\x02\x47\x5e", [0x3e87] = "\x0e\x63\x4a",
  [0x3e88] = "\x0e\x44\x65", [0x3e89] = "\x01\x66\x6f",
  [0x3e8a] = "\x01\x66\x6e", [0x3e8c] = "\x01\x6b\x32",
  [0x3e8d] = "\x01\x6b\x31", [0x3e8e] = "\x0e\x4e\x7a",
  [0x3e8f] = "\x02\x54\x50", [0x3e90] = "\x02\x5a\x5f",
  [0x3e91] = "\x02\x5a\x5d", [0x3e92] = "\x02\x5a\x5e",
  [0x3e93] = "\x01\x72\x49", [0x3e94] = "\x02\x5f\x63",
  [0x3e95] = "\x02\x5f\x62", [0x3e96] = "\x0e\x56\x4f",
  [0x3e97] = "\x02\x63\x5d", [0x3e98] = "\x02\x63\x5c",
  [0x3e99] = "\x02\x63\x5b", [0x3e9a] = "\x02\x67\x33",
  [0x3e9b] = "\x0e\x61\x4d", [0x3e9c] = "\x02\x71\x78",
  [0x3e9d] = "\x01\x4b\x4a", [0x3e9e] = "\x01\x53\x56",
  [0x3e9f] = "\x0e\x2f\x2e", [0x3ea0] = "\x01\x53\x57",
  [0x3ea1] = "\x01\x58\x42", [0x3ea2] = "\x01\x58\x43",
  [0x3ea3] = "\x02\x33\x26", [0x3ea4] = "\x02\x33\x25",
  [0x3ea5] = "\x02\x39\x55", [0x3ea6] = "\x0e\x39\x30",
  [0x3ea7] = "\x01\x5d\x4e", [0x3ea8] = "\x01\x5d\x4c",
  [0x3ea9] = "\x01\x5d\x49", [0x3eaa] = "\x01\x5d\x4d",
  [0x3eab] = "\x01\x5d\x4b", [0x3eac] = "\x01\x5d\x4a",
  [0x3ead] = "\x0e\x39\x32", [0x3eae] = "\x0e\x39\x31",
  [0x3eaf] = "\x01\x62\x31", [0x3eb0] = "\x02\x40\x50",
  [0x3eb1] = "\x0e\x3f\x2f", [0x3eb2] = "\x01\x66\x74",
  [0x3eb3] = "\x01\x62\x33", [0x3eb4] = "\x01\x62\x38",
  [0x3eb5] = "\x02\x40\x52", [0x3eb6] = "\x01\x62\x3a",
  [0x3eb7] = "\x01\x62\x39", [0x3eb8] = "\x01\x62\x3c",
  [0x3eb9] = "\x02\x40\x51", [0x3eba] = "\x02\x40\x4e",
  [0x3ebb] = "\x01\x62\x36", [0x3ebc] = "\x01\x62\x32",
  [0x3ebd] = "\x01\x62\x34", [0x3ebe] = "\x02\x40\x4f",
  [0x3ebf] = "\x01\x62\x3b", [0x3ec0] = "\x01\x62\x37",
  [0x3ec1] = "\x01\x62\x35", [0x3ec2] = "\x01\x66\x76",
  [0x3ec3] = "\x01\x66\x75", [0x3ec4] = "\x01\x66\x73",
  [0x3ec5] = "\x01\x66\x77", [0x3ec7] = "\x01\x66\x71",
  [0x3ec8] = "\x01\x66\x72", [0x3ec9] = "\x0e\x44\x69",
  [0x3eca] = "\x01\x66\x70", [0x3ecc] = "\x02\x47\x61",
  [0x3ecd] = "\x0e\x44\x67", [0x3ecf] = "\x02\x4e\x3c",
  [0x3ed0] = "\x0e\x49\x70", [0x3ed1] = "\x01\x6b\x34",
  [0x3ed2] = "\x01\x6b\x35", [0x3ed3] = "\x01\x6b\x33",
  [0x3ed4] = "\x0e\x49\x72", [0x3ed5] = "\x02\x4e\x3b",
  [0x3ed6] = "\x0e\x49\x71", [0x3ed7] = "\x02\x4e\x3d",
  [0x3ed8] = "\x0e\x66\x4e", [0x3ed9] = "\x02\x54\x53",
  [0x3eda] = "\x02\x54\x55", [0x3edb] = "\x0e\x4e\x7e",
  [0x3edc] = "\x01\x6f\x32", [0x3edd] = "\x02\x54\x56",
  [0x3ede] = "\x01\x6f\x2b", [0x3edf] = "\x02\x54\x52",
  [0x3ee0] = "\x01\x6f\x2a", [0x3ee1] = "\x01\x6f\x34",
  [0x3ee2] = "\x01\x6f\x30", [0x3ee3] = "\x01\x6f\x31",
  [0x3ee4] = "\x01\x6f\x2d", [0x3ee5] = "\x02\x54\x51",
  [0x3ee6] = "\x01\x6f\x2c", [0x3ee7] = "\x02\x54\x57",
  [0x3ee8] = "\x02\x54\x54", [0x3ee9] = "\x0e\x4e\x7d",
  [0x3eea] = "\x01\x6f\x33", [0x3eeb] = "\x0e\x4f\x22",
  [0x3eec] = "\x01\x6f\x2e", [0x3eed] = "\x01\x6f\x2f",
  [0x3eee] = "\x02\x5a\x61", [0x3eef] = "\x0e\x52\x75",
  [0x3ef0] = "\x02\x5a\x63", [0x3ef1] = "\x02\x5a\x62",
  [0x3ef2] = "\x0e\x52\x73", [0x3ef3] = "\x02\x5a\x64",
  [0x3ef4] = "\x01\x72\x4a", [0x3ef5] = "\x02\x5a\x60",
  [0x3ef6] = "\x0e\x56\x51", [0x3ef7] = "\x0e\x56\x52",
  [0x3ef8] = "\x01\x75\x56", [0x3ef9] = "\x02\x5f\x64",
  [0x3efa] = "\x01\x75\x53", [0x3efb] = "\x01\x75\x57",
  [0x3efc] = "\x01\x75\x55", [0x3efd] = "\x01\x75\x54",
  [0x3efe] = "\x02\x63\x5e", [0x3f00] = "\x02\x63\x61",
  [0x3f02] = "\x02\x63\x60", [0x3f03] = "\x0e\x59\x2e",
  [0x3f04] = "\x02\x63\x5f", [0x3f05] = "\x01\x77\x43",
  [0x3f06] = "\x02\x67\x34", [0x3f07] = "\x02\x67\x35",
  [0x3f08] = "\x01\x79\x25", [0x3f09] = "\x02\x67\x36",
  [0x3f0a] = "\x01\x79\x26", [0x3f0b] = "\x0e\x5b\x38",
  [0x3f0c] = "\x0e\x66\x4f", [0x3f0d] = "\x01\x7a\x45",
  [0x3f0e] = "\x0e\x64\x28", [0x3f0f] = "\x01\x7a\x44",
  [0x3f10] = "\x02\x6c\x41", [0x3f11] = "\x0e\x5e\x65",
  [0x3f12] = "\x0e\x5e\x66", [0x3f13] = "\x01\x7b\x2e",
  [0x3f14] = "\x02\x6c\x42", [0x3f15] = "\x02\x6e\x2b",
  [0x3f16] = "\x01\x7b\x75", [0x3f17] = "\x01\x7b\x76",
  [0x3f18] = "\x0e\x66\x50", [0x3f19] = "\x02\x6f\x53",
  [0x3f1a] = "\x0e\x60\x4a", [0x3f1b] = "\x01\x7c\x6d",
  [0x3f1c] = "\x0e\x61\x4e", [0x3f64] = "\x01\x4b\x4b",
  [0x3f65] = "\x0e\x39\x33", [0x3f66] = "\x01\x5d\x50",
  [0x3f67] = "\x01\x5d\x4f", [0x3f68] = "\x02\x47\x62",
  [0x3f69] = "\x02\x47\x63", [0x3f6b] = "\x01\x6b\x36",
  [0x3f6c] = "\x02\x5a\x65", [0x3f6d] = "\x01\x6f\x35",
  [0x3f6e] = "\x02\x5a\x66", [0x3f6f] = "\x02\x5f\x65",
  [0x3f70] = "\x01\x4b\x4c", [0x3f71] = "\x0e\x24\x3a",
  [0x3f72] = "\x02\x2d\x53", [0x3f73] = "\x01\x53\x59",
  [0x3f74] = "\x01\x53\x58", [0x3f76] = "\x02\x33\x27",
  [0x3f77] = "\x01\x58\x44", [0x3f78] = "\x02\x33\x28",
  [0x3f79] = "\x02\x39\x58", [0x3f7a] = "\x0e\x39\x39",
  [0x3f7b] = "\x02\x39\x57", [0x3f7c] = "\x0e\x39\x37",
  [0x3f7d] = "\x02\x39\x56", [0x3f7e] = "\x0e\x39\x35",
  [0x3f7f] = "\x0e\x39\x38", [0x3f80] = "\x02\x40\x54",
  [0x3f81] = "\x01\x62\x3f", [0x3f82] = "\x0e\x3f\x39",
  [0x3f83] = "\x0e\x3f\x34", [0x3f84] = "\x02\x40\x53",
  [0x3f85] = "\x01\x62\x3e", [0x3f86] = "\x0e\x3f\x37",
  [0x3f87] = "\x0e\x3f\x30", [0x3f88] = "\x0e\x3f\x33",
  [0x3f89] = "\x02\x40\x55", [0x3f8a] = "\x01\x62\x3d",
  [0x3f8b] = "\x0e\x3f\x38", [0x3f8c] = "\x02\x47\x65",
  [0x3f8d] = "\x02\x47\x68", [0x3f8e] = "\x02\x47\x66",
  [0x3f8f] = "\x02\x47\x67", [0x3f90] = "\x02\x47\x6b",
  [0x3f91] = "\x02\x47\x64", [0x3f92] = "\x02\x47\x6c",
  [0x3f93] = "\x02\x47\x69", [0x3f94] = "\x02\x47\x6a",
  [0x3f95] = "\x01\x6b\x38", [0x3f96] = "\x02\x4e\x3e",
  [0x3f97] = "\x0e\x49\x75", [0x3f98] = "\x0e\x49\x73",
  [0x3f99] = "\x01\x6b\x37", [0x3f9a] = "\x0e\x49\x74",
  [0x3f9b] = "\x02\x54\x5b", [0x3f9c] = "\x02\x54\x59",
  [0x3f9d] = "\x0e\x4f\x26", [0x3f9f] = "\x01\x6f\x36",
  [0x3fa0] = "\x02\x54\x58", [0x3fa1] = "\x02\x54\x5a",
  [0x3fa2] = "\x0e\x4f\x25", [0x3fa3] = "\x01\x6f\x37",
  [0x3fa4] = "\x0e\x4f\x23", [0x3fa5] = "\x02\x5a\x67",
  [0x3fa6] = "\x0e\x52\x77", [0x3fa7] = "\x02\x5a\x68",
  [0x3fa8] = "\x01\x75\x58", [0x3fa9] = "\x0e\x59\x31",
  [0x3faa] = "\x02\x67\x38", [0x3fab] = "\x02\x67\x3a",
  [0x3fac] = "\x02\x67\x37", [0x3fad] = "\x02\x67\x39",
  [0x3fae] = "\x02\x6a\x24", [0x3faf] = "\x02\x6c\x43",
  [0x3fb0] = "\x0e\x66\x51", [0x3fb2] = "\x02\x71\x5e",
  [0x3fb3] = "\x01\x4b\x4d", [0x3fb4] = "\x01\x53\x5a",
  [0x3fb5] = "\x02\x33\x29", [0x3fb6] = "\x02\x33\x2b",
  [0x3fb7] = "\x02\x33\x2a", [0x3fb9] = "\x02\x39\x5b",
  [0x3fba] = "\x01\x5d\x52", [0x3fbb] = "\x0e\x39\x3e",
  [0x3fbc] = "\x02\x39\x59", [0x3fbd] = "\x0e\x39\x3a",
  [0x3fbe] = "\x01\x5d\x51", [0x3fbf] = "\x02\x39\x5c",
  [0x3fc0] = "\x0e\x39\x3d", [0x3fc1] = "\x02\x39\x5d",
  [0x3fc2] = "\x02\x39\x5a", [0x3fc5] = "\x02\x40\x61",
  [0x3fc6] = "\x01\x62\x47", [0x3fc7] = "\x02\x40\x59",
  [0x3fc8] = "\x02\x40\x5f", [0x3fc9] = "\x0e\x3f\x3f",
  [0x3fca] = "\x0e\x3f\x3a", [0x3fcb] = "\x01\x62\x42",
  [0x3fcc] = "\x01\x62\x45", [0x3fcd] = "\x02\x40\x58",
  [0x3fce] = "\x01\x62\x40", [0x3fcf] = "\x02\x40\x5c",
  [0x3fd0] = "\x02\x47\x72", [0x3fd1] = "\x01\x62\x44",
  [0x3fd2] = "\x0e\x3f\x3b", [0x3fd3] = "\x02\x40\x57",
  [0x3fd4] = "\x0e\x3f\x3e", [0x3fd5] = "\x02\x40\x5d",
  [0x3fd6] = "\x02\x40\x5a", [0x3fd7] = "\x02\x40\x60",
  [0x3fd8] = "\x02\x40\x56", [0x3fd9] = "\x02\x40\x5e",
  [0x3fda] = "\x01\x62\x43", [0x3fdb] = "\x01\x62\x46",
  [0x3fdc] = "\x02\x40\x5b", [0x3fdd] = "\x01\x62\x41",
  [0x3fdf] = "\x01\x66\x79", [0x3fe0] = "\x02\x47\x6e",
  [0x3fe1] = "\x01\x66\x78", [0x3fe2] = "\x02\x47\x75",
  [0x3fe3] = "\x02\x47\x74", [0x3fe4] = "\x01\x67\x21",
  [0x3fe6] = "\x01\x67\x22", [0x3fe7] = "\x02\x47\x76",
  [0x3fe8] = "\x01\x66\x7a", [0x3fe9] = "\x02\x47\x73",
  [0x3fea] = "\x01\x66\x7e", [0x3feb] = "\x02\x47\x78",
  [0x3fec] = "\x02\x47\x6f", [0x3fed] = "\x0e\x44\x6b",
  [0x3fee] = "\x02\x47\x71", [0x3fef] = "\x01\x66\x7b",
  [0x3ff0] = "\x02\x47\x6d", [0x3ff1] = "\x02\x47\x70",
  [0x3ff2] = "\x02\x47\x77", [0x3ff3] = "\x01\x66\x7c",
  [0x3ff4] = "\x02\x47\x79", [0x3ffa] = "\x01\x66\x7d",
  [0x3ffc] = "\x01\x6b\x39", [0x3ffd] = "\x02\x4e\x43",
  [0x3ffe] = "\x02\x4e\x49", [0x3fff] = "\x02\x4e\x41",
  [0x4000] = "\x02\x4e\x4a", [0x4001] = "\x0e\x49\x7b",
  [0x4002] = "\x02\x4e\x40", [0x4003] = "\x02\x4e\x45",
  [0x4004] = "\x02\x4e\x4b", [0x4005] = "\x02\x4e\x48",
  [0x4006] = "\x02\x4e\x47", [0x4007] = "\x02\x4e\x46",
  [0x4008] = "\x0e\x49\x7a", [0x4009] = "\x02\x4e\x3f",
  [0x400a] = "\x02\x4e\x44", [0x400b] = "\x0e\x49\x79",
  [0x400d] = "\x02\x4e\x42", [0x400f] = "\x01\x6f\x3c",
  [0x4010] = "\x01\x6f\x39", [0x4011] = "\x02\x54\x64",
  [0x4012] = "\x02\x54\x69", [0x4013] = "\x02\x54\x6b",
  [0x4014] = "\x02\x54\x68", [0x4015] = "\x02\x54\x61",
  [0x4016] = "\x02\x54\x63", [0x4017] = "\x02\x54\x6d",
  [0x4018] = "\x02\x54\x6a", [0x4019] = "\x02\x54\x65",
  [0x401a] = "\x02\x54\x6e", [0x401b] = "\x02\x54\x62",
  [0x401c] = "\x02\x54\x6c", [0x401d] = "\x01\x6f\x3a",
  [0x401e] = "\x01\x6f\x40", [0x401f] = "\x01\x6f\x3e",
  [0x4020] = "\x02\x54\x5c", [0x4021] = "\x01\x6f\x3f",
  [0x4022] = "\x01\x6f\x3b", [0x4023] = "\x02\x54\x5d",
  [0x4024] = "\x02\x54\x5f", [0x4025] = "\x02\x54\x5e",
  [0x4026] = "\x02\x54\x66", [0x4027] = "\x02\x54\x67",
  [0x4028] = "\x0e\x4f\x2c", [0x4029] = "\x01\x6f\x3d",
  [0x402a] = "\x0e\x4f\x2b", [0x402b] = "\x01\x6f\x38",
  [0x402e] = "\x02\x54\x60", [0x4030] = "\x02\x5a\x72",
  [0x4031] = "\x01\x72\x4c", [0x4032] = "\x0e\x52\x7e",
  [0x4033] = "\x02\x5a\x69", [0x4034] = "\x01\x72\x4d",
  [0x4035] = "\x01\x72\x50", [0x4036] = "\x02\x5a\x6e",
  [0x4038] = "\x02\x5a\x6b", [0x4039] = "\x01\x72\x4f",
  [0x403b] = "\x0e\x52\x7b", [0x403c] = "\x02\x5a\x6f",
  [0x403d] = "\x02\x5a\x70", [0x403e] = "\x02\x5a\x6a",
  [0x403f] = "\x02\x5a\x73", [0x4040] = "\x02\x5a\x6c",
  [0x4041] = "\x02\x5a\x71", [0x4042] = "\x01\x72\x4e",
  [0x4043] = "\x0e\x52\x7c", [0x4044] = "\x01\x72\x4b",
  [0x4045] = "\x02\x5a\x6d", [0x4046] = "\x0e\x56\x54",
  [0x4047] = "\x02\x5f\x6b", [0x4048] = "\x01\x75\x5b",
  [0x4049] = "\x01\x75\x59", [0x404a] = "\x01\x75\x5c",
  [0x404b] = "\x01\x75\x5a", [0x404c] = "\x02\x5f\x6a",
  [0x404d] = "\x02\x5f\x67", [0x404e] = "\x02\x5f\x66",
  [0x404f] = "\x0e\x56\x55", [0x4050] = "\x02\x5f\x69",
  [0x4053] = "\x02\x5f\x68", [0x4054] = "\x02\x63\x6f",
  [0x4055] = "\x01\x77\x49", [0x4056] = "\x02\x63\x66",
  [0x4057] = "\x02\x63\x65", [0x4058] = "\x0e\x59\x32",
  [0x4059] = "\x01\x77\x44", [0x405a] = "\x02\x63\x6b",
  [0x405b] = "\x02\x63\x6a", [0x405c] = "\x02\x63\x62",
  [0x405d] = "\x02\x63\x6d", [0x405e] = "\x02\x63\x67",
  [0x405f] = "\x01\x77\x48", [0x4060] = "\x02\x63\x64",
  [0x4061] = "\x02\x63\x6c", [0x4062] = "\x02\x63\x63",
  [0x4063] = "\x01\x77\x45", [0x4064] = "\x01\x77\x47",
  [0x4065] = "\x02\x63\x68", [0x4066] = "\x01\x77\x46",
  [0x4067] = "\x02\x63\x69", [0x4068] = "\x0e\x5b\x3a",
  [0x4069] = "\x02\x63\x6e", [0x406a] = "\x02\x67\x3e",
  [0x406b] = "\x0e\x5b\x39", [0x406c] = "\x01\x79\x2b",
  [0x406d] = "\x02\x67\x3b", [0x406f] = "\x02\x67\x3f",
  [0x4070] = "\x0e\x5b\x3b", [0x4072] = "\x01\x79\x28",
  [0x4073] = "\x02\x67\x3d", [0x4074] = "\x01\x79\x2d",
  [0x4075] = "\x0e\x5b\x3c", [0x4076] = "\x01\x79\x2a",
  [0x4077] = "\x0e\x5b\x3d", [0x4078] = "\x02\x67\x3c",
  [0x407a] = "\x01\x79\x2c", [0x407b] = "\x02\x67\x40",
  [0x407c] = "\x01\x79\x27", [0x4081] = "\x01\x7a\x47",
  [0x4082] = "\x01\x7a\x49", [0x4083] = "\x0e\x5d\x37",
  [0x4084] = "\x02\x6a\x27", [0x4085] = "\x01\x7a\x48",
  [0x4086] = "\x02\x6a\x25", [0x4087] = "\x01\x79\x29",
  [0x4088] = "\x02\x6a\x26", [0x4089] = "\x01\x7a\x46",
  [0x408a] = "\x01\x7b\x2f", [0x408b] = "\x01\x7b\x31",
  [0x408c] = "\x02\x6c\x45", [0x408d] = "\x01\x7b\x30",
  [0x408e] = "\x02\x6c\x44", [0x4090] = "\x02\x6e\x30",
  [0x4091] = "\x01\x7b\x77", [0x4092] = "\x02\x6e\x2f",
  [0x4093] = "\x01\x7b\x78", [0x4094] = "\x02\x6e\x2d",
  [0x4095] = "\x02\x6e\x2c", [0x4096] = "\x02\x6e\x31",
  [0x4097] = "\x02\x6e\x32", [0x4098] = "\x02\x6f\x54",
  [0x4099] = "\x0e\x60\x4b", [0x409a] = "\x02\x6e\x2e",
  [0x409d] = "\x02\x70\x54", [0x409e] = "\x02\x70\x51",
  [0x409f] = "\x02\x70\x52", [0x40a0] = "\x02\x70\x53",
  [0x40a1] = "\x01\x7d\x29", [0x40a2] = "\x0e\x61\x50",
  [0x40a3] = "\x02\x71\x40", [0x40a4] = "\x02\x71\x3f",
  [0x40a5] = "\x02\x71\x3e", [0x40a6] = "\x02\x71\x5f",
  [0x40a7] = "\x0e\x61\x6c", [0x40a8] = "\x02\x72\x2e",
  [0x40a9] = "\x02\x71\x79", [0x40aa] = "\x01\x7d\x3b",
  [0x40ab] = "\x01\x4b\x4e", [0x40ac] = "\x01\x58\x45",
  [0x40ad] = "\x0e\x39\x3f", [0x40af] = "\x0e\x39\x40",
  [0x40b0] = "\x0e\x3f\x40", [0x40b1] = "\x0e\x44\x73",
  [0x40b2] = "\x01\x67\x23", [0x40b3] = "\x0e\x49\x7d",
  [0x40b4] = "\x0e\x49\x7c", [0x40b6] = "\x0e\x4f\x30",
  [0x40b7] = "\x0e\x4f\x32", [0x40b8] = "\x0e\x4f\x2f",
  [0x40b9] = "\x0e\x4f\x31", [0x40ba] = "\x01\x6f\x41",
  [0x40bc] = "\x0e\x66\x52", [0x40bd] = "\x02\x5a\x74",
  [0x40bf] = "\x0e\x59\x35", [0x40c0] = "\x01\x77\x4a",
  [0x40c1] = "\x0e\x59\x36", [0x40c2] = "\x02\x67\x41",
  [0x40c3] = "\x0e\x5b\x3f", [0x40c6] = "\x0e\x5d\x38",
  [0x40c9] = "\x02\x71\x7a", [0x40ca] = "\x01\x4b\x4f",
  [0x40cb] = "\x01\x4f\x51", [0x40cc] = "\x01\x53\x5c",
  [0x40cd] = "\x01\x53\x5b", [0x40cf] = "\x01\x58\x48",
  [0x40d1] = "\x02\x33\x2c", [0x40d2] = "\x01\x58\x46",
  [0x40d3] = "\x02\x33\x2d", [0x40d4] = "\x01\x58\x47",
  [0x40d5] = "\x0e\x66\x53", [0x40d6] = "\x0e\x39\x41",
  [0x40d7] = "\x02\x39\x62", [0x40d8] = "\x02\x39\x5e",
  [0x40d9] = "\x0e\x39\x47", [0x40db] = "\x01\x5d\x53",
  [0x40dc] = "\x02\x39\x61", [0x40dd] = "\x02\x39\x60",
  [0x40de] = "\x02\x39\x5f", [0x40df] = "\x01\x5d\x54",
  [0x40e0] = "\x02\x39\x63", [0x40e1] = "\x02\x39\x64",
  [0x40e4] = "\x0e\x3f\x45", [0x40e5] = "\x02\x40\x68",
  [0x40e6] = "\x02\x40\x66", [0x40e7] = "\x02\x40\x6a",
  [0x40e8] = "\x02\x40\x6b", [0x40e9] = "\x02\x40\x71",
  [0x40ea] = "\x0e\x3f\x47", [0x40eb] = "\x02\x40\x6d",
  [0x40ec] = "\x02\x40\x6f", [0x40ee] = "\x02\x40\x67",
  [0x40ef] = "\x02\x40\x62", [0x40f1] = "\x02\x40\x6e",
  [0x40f2] = "\x0e\x3f\x44", [0x40f3] = "\x0e\x3f\x46",
  [0x40f4] = "\x02\x40\x70", [0x40f5] = "\x02\x40\x69",
  [0x40f6] = "\x02\x40\x6c", [0x40f7] = "\x02\x40\x63",
  [0x40f8] = "\x01\x62\x49", [0x40f9] = "\x02\x40\x65",
  [0x40fa] = "\x02\x40\x64", [0x40fb] = "\x01\x62\x48",
  [0x40fc] = "\x01\x62\x4a", [0x40fd] = "\x0e\x3f\x49",
  [0x40fe] = "\x01\x67\x26", [0x40ff] = "\x02\x47\x7b",
  [0x4100] = "\x02\x47\x7d", [0x4101] = "\x02\x47\x7c",
  [0x4102] = "\x02\x48\x23", [0x4103] = "\x01\x67\x24",
  [0x4104] = "\x0e\x44\x76", [0x4105] = "\x02\x47\x7e",
  [0x4106] = "\x02\x47\x7a", [0x4107] = "\x02\x48\x21",
  [0x4108] = "\x02\x48\x22", [0x4109] = "\x01\x67\x25",
  [0x410a] = "\x01\x67\x27", [0x410b] = "\x02\x48\x24",
  [0x410d] = "\x02\x4e\x4f", [0x410e] = "\x02\x4e\x4e",
  [0x4110] = "\x02\x4e\x4c", [0x4111] = "\x02\x4e\x4d",
  [0x4112] = "\x01\x6b\x3b", [0x4113] = "\x01\x6b\x3d",
  [0x4114] = "\x01\x6b\x3a", [0x4115] = "\x01\x6b\x3c",
  [0x4116] = "\x02\x54\x75", [0x4117] = "\x02\x54\x76",
  [0x4118] = "\x02\x54\x71", [0x4119] = "\x0e\x4f\x36",
  [0x411a] = "\x02\x54\x72", [0x411b] = "\x01\x6f\x43",
  [0x411c] = "\x01\x6f\x48", [0x411d] = "\x01\x6f\x42",
  [0x411e] = "\x01\x6f\x49", [0x411f] = "\x01\x6f\x44",
  [0x4120] = "\x02\x54\x73", [0x4122] = "\x0e\x4f\x33",
  [0x4123] = "\x02\x54\x74", [0x4124] = "\x02\x54\x70",
  [0x4125] = "\x01\x6f\x4a", [0x4126] = "\x01\x6f\x46",
  [0x4127] = "\x0e\x4f\x35", [0x4129] = "\x01\x6f\x45",
  [0x412a] = "\x01\x6f\x47", [0x412b] = "\x0e\x4f\x34",
  [0x412c] = "\x02\x54\x6f", [0x412d] = "\x0e\x53\x24",
  [0x412e] = "\x02\x5a\x76", [0x412f] = "\x01\x72\x52",
  [0x4130] = "\x0e\x53\x22", [0x4131] = "\x0e\x53\x21",
  [0x4132] = "\x02\x5a\x78", [0x4133] = "\x01\x72\x54",
  [0x4134] = "\x02\x5a\x7b", [0x4135] = "\x02\x5a\x77",
  [0x4136] = "\x02\x5a\x75", [0x4137] = "\x02\x5a\x7a",
  [0x4138] = "\x01\x72\x53", [0x4139] = "\x02\x5a\x79",
  [0x413b] = "\x01\x72\x51", [0x413c] = "\x0e\x53\x25",
  [0x413d] = "\x0e\x56\x59", [0x413e] = "\x01\x75\x5e",
  [0x413f] = "\x01\x75\x61", [0x4140] = "\x02\x5f\x6d",
  [0x4141] = "\x0e\x56\x5a", [0x4142] = "\x01\x75\x5f",
  [0x4143] = "\x02\x5f\x6c", [0x4144] = "\x01\x75\x5d",
  [0x4145] = "\x01\x75\x60", [0x4146] = "\x02\x63\x70",
  [0x4147] = "\x02\x63\x71", [0x4148] = "\x02\x63\x72",
  [0x4149] = "\x01\x77\x4b", [0x414a] = "\x0e\x59\x37",
  [0x414b] = "\x02\x63\x73", [0x414d] = "\x01\x77\x4c",
  [0x414e] = "\x01\x79\x2f", [0x414f] = "\x02\x67\x44",
  [0x4150] = "\x02\x67\x45", [0x4151] = "\x02\x67\x43",
  [0x4152] = "\x02\x67\x42", [0x4153] = "\x02\x67\x46",
  [0x4154] = "\x01\x79\x2e", [0x4155] = "\x02\x6a\x2b",
  [0x4156] = "\x02\x6a\x29", [0x4157] = "\x02\x6a\x2a",
  [0x4158] = "\x02\x6a\x2c", [0x4159] = "\x02\x6a\x28",
  [0x415a] = "\x02\x6a\x2d", [0x415b] = "\x02\x6c\x47",
  [0x415c] = "\x0e\x5e\x67", [0x415d] = "\x02\x6c\x48",
  [0x415e] = "\x02\x6c\x46", [0x415f] = "\x01\x7b\x32",
  [0x4160] = "\x02\x6e\x33", [0x4161] = "\x01\x7b\x79",
  [0x4162] = "\x02\x6e\x34", [0x4163] = "\x02\x6f\x56",
  [0x4164] = "\x02\x6f\x55", [0x4165] = "\x0e\x62\x22",
  [0x419b] = "\x01\x4b\x50", [0x419c] = "\x01\x62\x4b",
  [0x419d] = "\x0e\x3f\x4b", [0x419e] = "\x0e\x44\x77",
  [0x419f] = "\x01\x67\x28", [0x41a0] = "\x0e\x44\x78",
  [0x41a1] = "\x0e\x4a\x21", [0x41a2] = "\x0e\x4a\x22",
  [0x41a3] = "\x01\x6b\x3e", [0x41a4] = "\x0e\x4f\x37",
  [0x41a5] = "\x0e\x53\x27", [0x41a6] = "\x01\x72\x56",
  [0x41a7] = "\x0e\x53\x26", [0x41a8] = "\x01\x72\x55",
  [0x41aa] = "\x0e\x66\x54", [0x41ac] = "\x0e\x59\x38",
  [0x41ad] = "\x01\x79\x30", [0x41ae] = "\x01\x7a\x29",
  [0x41af] = "\x01\x7b\x33", [0x41b0] = "\x01\x4b\x51",
  [0x41b1] = "\x01\x58\x49", [0x41b2] = "\x01\x67\x29",
  [0x41b3] = "\x0e\x4f\x39", [0x41b4] = "\x02\x67\x47",
  [0x41b5] = "\x0e\x27\x2f", [0x41b9] = "\x0e\x24\x3b",
  [0x41be] = "\x0e\x27\x30", [0x41bf] = "\x02\x25\x26",
  [0x41c1] = "\x0e\x27\x33", [0x41c2] = "\x01\x4b\x52",
  [0x41c4] = "\x01\x4b\x55", [0x41c5] = "\x01\x4b\x54",
  [0x41c6] = "\x01\x4b\x53", [0x41c7] = "\x0e\x27\x31",
  [0x41c9] = "\x02\x25\x25", [0x41ca] = "\x0e\x2a\x72",
  [0x41cb] = "\x02\x28\x6d", [0x41cc] = "\x0e\x2a\x70",
  [0x41cd] = "\x02\x28\x6f", [0x41ce] = "\x01\x4f\x52",
  [0x41cf] = "\x0e\x2a\x74", [0x41d1] = "\x01\x4f\x54",
  [0x41d2] = "\x02\x28\x6c", [0x41d3] = "\x02\x28\x6e",
  [0x41d4] = "\x01\x4f\x53", [0x41d5] = "\x02\x28\x71",
  [0x41d6] = "\x02\x28\x70", [0x41d7] = "\x02\x28\x72",
  [0x41d8] = "\x0e\x2a\x73", [0x41d9] = "\x0e\x2a\x71",
  [0x41e0] = "\x02\x2d\x57", [0x41e1] = "\x02\x2d\x55",
  [0x41e2] = "\x01\x53\x5f", [0x41e3] = "\x02\x2d\x54",
  [0x41e4] = "\x01\x53\x64", [0x41e5] = "\x01\x53\x61",
  [0x41e6] = "\x01\x53\x5e", [0x41e7] = "\x0e\x2f\x31",
  [0x41e8] = "\x01\x53\x65", [0x41e9] = "\x0e\x2f\x32",
  [0x41ea] = "\x01\x53\x60", [0x41eb] = "\x01\x53\x63",
  [0x41ed] = "\x01\x53\x62", [0x41ee] = "\x02\x2d\x56",
  [0x41ef] = "\x0e\x2f\x33", [0x41f0] = "\x01\x53\x5d",
  [0x41f1] = "\x0e\x2f\x30", [0x41f4] = "\x01\x58\x4f",
  [0x41f5] = "\x02\x33\x2f", [0x41f6] = "\x02\x33\x35",
  [0x41f7] = "\x01\x58\x4c", [0x41f8] = "\x01\x58\x53",
  [0x41f9] = "\x0e\x34\x25", [0x41fa] = "\x01\x58\x4e",
  [0x41fb] = "\x02\x33\x32", [0x41fc] = "\x02\x33\x34",
  [0x41fd] = "\x01\x58\x51", [0x41fe] = "\x02\x33\x2e",
  [0x41ff] = "\x02\x33\x31", [0x4200] = "\x01\x58\x4d",
  [0x4201] = "\x01\x58\x4a", [0x4202] = "\x02\x33\x30",
  [0x4203] = "\x01\x58\x50", [0x4204] = "\x02\x33\x33",
  [0x4205] = "\x01\x58\x52", [0x4206] = "\x01\x58\x4b",
  [0x4208] = "\x0e\x34\x27", [0x4209] = "\x0e\x34\x26",
  [0x420b] = "\x02\x39\x66", [0x420c] = "\x02\x39\x69",
  [0x420d] = "\x01\x5d\x56", [0x420e] = "\x0e\x39\x4b",
  [0x420f] = "\x01\x5d\x60", [0x4210] = "\x01\x5d\x5c",
  [0x4211] = "\x02\x39\x67", [0x4212] = "\x0e\x39\x4a",
  [0x4213] = "\x0e\x39\x4e", [0x4214] = "\x01\x5d\x64",
  [0x4215] = "\x01\x5d\x5d", [0x4216] = "\x01\x5d\x62",
  [0x4217] = "\x01\x5d\x58", [0x4219] = "\x01\x5d\x55",
  [0x421a] = "\x01\x5d\x57", [0x421b] = "\x01\x5d\x63",
  [0x421c] = "\x02\x39\x68", [0x421d] = "\x01\x5d\x5b",
  [0x421e] = "\x01\x5d\x5e", [0x421f] = "\x01\x5d\x5a",
  [0x4220] = "\x01\x5d\x5f", [0x4221] = "\x02\x39\x6a",
  [0x4222] = "\x01\x5d\x61", [0x4223] = "\x01\x5d\x59",
  [0x4224] = "\x02\x39\x65", [0x4225] = "\x0e\x39\x4c",
  [0x4228] = "\x0e\x3f\x4f", [0x4229] = "\x0e\x3f\x50",
  [0x422a] = "\x0e\x3f\x4d", [0x422b] = "\x0e\x3f\x4c",
  [0x422c] = "\x0e\x3f\x51", [0x422d] = "\x02\x40\x72",
  [0x422e] = "\x01\x62\x4c", [0x422f] = "\x02\x40\x74",
  [0x4230] = "\x0e\x3f\x55", [0x4231] = "\x01\x62\x4e",
  [0x4232] = "\x01\x62\x50", [0x4234] = "\x02\x40\x73",
  [0x4235] = "\x01\x62\x4d", [0x4236] = "\x01\x62\x51",
  [0x4237] = "\x0e\x3f\x56", [0x4238] = "\x01\x62\x4f",
  [0x423a] = "\x0e\x3f\x54", [0x423b] = "\x0e\x3f\x57",
  [0x423c] = "\x01\x67\x2f", [0x423d] = "\x02\x48\x29",
  [0x423e] = "\x01\x67\x37", [0x423f] = "\x02\x48\x26",
  [0x4240] = "\x0e\x44\x7d", [0x4241] = "\x01\x67\x38",
  [0x4242] = "\x01\x67\x2d", [0x4243] = "\x0e\x44\x79",
  [0x4244] = "\x02\x48\x27", [0x4245] = "\x0e\x44\x7c",
  [0x4246] = "\x0e\x44\x7a", [0x4247] = "\x01\x67\x32",
  [0x4248] = "\x0e\x44\x7b", [0x4249] = "\x02\x48\x28",
  [0x424a] = "\x01\x67\x2b", [0x424b] = "\x01\x67\x2a",
  [0x424c] = "\x0e\x44\x7e", [0x424d] = "\x01\x67\x35",
  [0x424e] = "\x01\x67\x34", [0x424f] = "\x01\x67\x33",
  [0x4250] = "\x01\x67\x31", [0x4251] = "\x01\x67\x36",
  [0x4252] = "\x02\x48\x25", [0x4253] = "\x01\x67\x2c",
  [0x4254] = "\x01\x67\x2e", [0x4255] = "\x01\x67\x30",
  [0x4258] = "\x01\x6b\x40", [0x4259] = "\x01\x6b\x43",
  [0x425a] = "\x0e\x4a\x24", [0x425b] = "\x01\x6b\x47",
  [0x425c] = "\x01\x6b\x41", [0x425d] = "\x01\x6b\x46",
  [0x425e] = "\x01\x6b\x44", [0x425f] = "\x0e\x4a\x27",
  [0x4260] = "\x01\x6b\x3f", [0x4261] = "\x0e\x4a\x26",
  [0x4262] = "\x01\x6b\x45", [0x4263] = "\x01\x6b\x42",
  [0x4267] = "\x02\x54\x7a", [0x4268] = "\x01\x6f\x4d",
  [0x4269] = "\x01\x6f\x4b", [0x426a] = "\x0e\x4f\x3c",
  [0x426b] = "\x02\x54\x7b", [0x426d] = "\x01\x6f\x4e",
  [0x426e] = "\x01\x6f\x4c", [0x426f] = "\x02\x54\x79",
  [0x4270] = "\x02\x54\x78", [0x4271] = "\x0e\x4f\x3b",
  [0x4272] = "\x01\x72\x5a", [0x4273] = "\x02\x54\x77",
  [0x4274] = "\x01\x72\x58", [0x4275] = "\x01\x72\x57",
  [0x4276] = "\x02\x5a\x7c", [0x4277] = "\x01\x6f\x4f",
  [0x4278] = "\x01\x72\x59", [0x4279] = "\x02\x5a\x7d",
  [0x427a] = "\x01\x72\x5c", [0x427b] = "\x02\x5a\x7e",
  [0x427c] = "\x01\x72\x5b", [0x427d] = "\x01\x75\x63",
  [0x427e] = "\x02\x5f\x6f", [0x427f] = "\x01\x75\x62",
  [0x4280] = "\x01\x75\x67", [0x4281] = "\x01\x75\x65",
  [0x4282] = "\x01\x75\x66", [0x4283] = "\x01\x77\x4e",
  [0x4284] = "\x01\x75\x64", [0x4285] = "\x02\x5f\x6e",
  [0x4286] = "\x02\x5b\x21", [0x4287] = "\x01\x77\x4d",
  [0x4288] = "\x01\x77\x4f", [0x4289] = "\x0e\x59\x3a",
  [0x428a] = "\x01\x79\x31", [0x428b] = "\x01\x79\x32",
  [0x428d] = "\x02\x6a\x2e", [0x428f] = "\x01\x7c\x4a",
  [0x4290] = "\x01\x7c\x49", [0x4291] = "\x01\x4b\x57",
  [0x4292] = "\x0e\x22\x5e", [0x4293] = "\x0e\x22\x5d",
  [0x4294] = "\x02\x22\x75", [0x4295] = "\x01\x58\x54",
  [0x4296] = "\x0e\x24\x3c", [0x4297] = "\x02\x22\x72",
  [0x4298] = "\x02\x22\x73", [0x4299] = "\x02\x22\x71",
  [0x429a] = "\x0e\x24\x3d", [0x429b] = "\x02\x22\x74",
  [0x429c] = "\x0e\x66\x55", [0x429e] = "\x02\x25\x2a",
  [0x429f] = "\x02\x25\x27", [0x42a0] = "\x02\x25\x2c",
  [0x42a1] = "\x02\x25\x28", [0x42a2] = "\x01\x4b\x58",
  [0x42a3] = "\x01\x4b\x5b", [0x42a4] = "\x0e\x27\x35",
  [0x42a5] = "\x02\x25\x29", [0x42a6] = "\x01\x4b\x5a",
  [0x42a7] = "\x02\x25\x2b", [0x42a8] = "\x0e\x27\x39",
  [0x42a9] = "\x0e\x27\x34", [0x42aa] = "\x01\x4b\x59",
  [0x42ab] = "\x0e\x39\x51", [0x42ad] = "\x0e\x2a\x77",
  [0x42af] = "\x02\x28\x75", [0x42b0] = "\x02\x28\x77",
  [0x42b1] = "\x01\x4f\x57", [0x42b2] = "\x02\x28\x73",
  [0x42b3] = "\x02\x28\x76", [0x42b4] = "\x02\x28\x74",
  [0x42b5] = "\x01\x4f\x55", [0x42b6] = "\x01\x4f\x58",
  [0x42b7] = "\x0e\x2a\x75", [0x42b8] = "\x01\x4f\x56",
  [0x42b9] = "\x0e\x2a\x78", [0x42bc] = "\x0e\x2f\x35",
  [0x42bd] = "\x02\x2d\x59", [0x42be] = "\x02\x2d\x5d",
  [0x42bf] = "\x02\x2d\x5a", [0x42c0] = "\x0e\x2f\x38",
  [0x42c1] = "\x01\x53\x68", [0x42c2] = "\x0e\x2f\x34",
  [0x42c3] = "\x01\x53\x69", [0x42c4] = "\x0e\x2f\x3e",
  [0x42c5] = "\x02\x2d\x5c", [0x42c6] = "\x0e\x2f\x37",
  [0x42c7] = "\x02\x2d\x5e", [0x42c8] = "\x02\x2d\x60",
  [0x42c9] = "\x0e\x2f\x3c", [0x42ca] = "\x01\x53\x66",
  [0x42cb] = "\x02\x2d\x5f", [0x42cd] = "\x0e\x2f\x3b",
  [0x42ce] = "\x01\x53\x67", [0x42d4] = "\x02\x39\x71",
  [0x42d5] = "\x02\x2d\x5b", [0x42d6] = "\x02\x33\x36",
  [0x42d7] = "\x02\x33\x3f", [0x42d8] = "\x02\x33\x3d",
  [0x42d9] = "\x02\x33\x38", [0x42da] = "\x02\x33\x39",
  [0x42db] = "\x02\x33\x3e", [0x42dc] = "\x02\x33\x40",
  [0x42dd] = "\x01\x58\x56", [0x42de] = "\x0e\x34\x29",
  [0x42df] = "\x02\x33\x3b", [0x42e0] = "\x02\x33\x37",
  [0x42e1] = "\x01\x58\x55", [0x42e2] = "\x01\x58\x57",
  [0x42e3] = "\x02\x33\x3a", [0x42e4] = "\x02\x33\x41",
  [0x42e5] = "\x02\x33\x3c", [0x42e8] = "\x01\x5d\x65",
  [0x42e9] = "\x02\x39\x74", [0x42ea] = "\x02\x39\x6c",
  [0x42eb] = "\x02\x39\x72", [0x42ec] = "\x02\x39\x73",
  [0x42ed] = "\x01\x5d\x66", [0x42ee] = "\x0e\x39\x4f",
  [0x42ef] = "\x02\x39\x6b", [0x42f0] = "\x02\x39\x6d",
  [0x42f1] = "\x02\x2d\x58", [0x42f2] = "\x02\x39\x6f",
  [0x42f3] = "\x02\x39\x70", [0x42f4] = "\x02\x39\x6e",
  [0x42f5] = "\x01\x62\x53", [0x42f6] = "\x0e\x63\x6c",
  [0x42f7] = "\x0e\x39\x52", [0x42f9] = "\x02\x40\x7b",
  [0x42fa] = "\x02\x5b\x22", [0x42fb] = "\x02\x40\x7c",
  [0x42fc] = "\x02\x40\x79", [0x42fd] = "\x01\x5d\x67",
  [0x42fe] = "\x01\x62\x55", [0x42ff] = "\x02\x40\x78",
  [0x4300] = "\x02\x40\x7e", [0x4301] = "\x02\x40\x7d",
  [0x4302] = "\x01\x62\x52", [0x4303] = "\x02\x41\x23",
  [0x4304] = "\x02\x40\x77", [0x4305] = "\x02\x41\x22",
  [0x4306] = "\x02\x40\x75", [0x4307] = "\x02\x41\x21",
  [0x4308] = "\x02\x40\x7a", [0x4309] = "\x01\x62\x54",
  [0x430a] = "\x0e\x3f\x59", [0x430b] = "\x02\x48\x30",
  [0x430c] = "\x0e\x45\x23", [0x430d] = "\x02\x48\x2b",
  [0x430e] = "\x02\x48\x31", [0x430f] = "\x02\x48\x2c",
  [0x4310] = "\x02\x48\x2a", [0x4311] = "\x02\x48\x2d",
  [0x4312] = "\x01\x67\x39", [0x4313] = "\x0e\x45\x24",
  [0x4314] = "\x02\x48\x2f", [0x4315] = "\x0e\x45\x25",
  [0x4316] = "\x02\x48\x2e", [0x4317] = "\x01\x67\x3a",
  [0x4318] = "\x01\x6b\x49", [0x4319] = "\x01\x6b\x48",
  [0x431a] = "\x02\x4e\x56", [0x431b] = "\x02\x4e\x59",
  [0x431c] = "\x02\x4e\x51", [0x431d] = "\x02\x4e\x55",
  [0x431e] = "\x01\x6b\x4a", [0x431f] = "\x02\x4e\x54",
  [0x4320] = "\x02\x4e\x52", [0x4321] = "\x02\x4e\x58",
  [0x4322] = "\x02\x4e\x53", [0x4323] = "\x02\x4e\x50",
  [0x4324] = "\x02\x4e\x57", [0x4325] = "\x0e\x4a\x2a",
  [0x4326] = "\x02\x55\x23", [0x4327] = "\x01\x6f\x52",
  [0x4328] = "\x02\x63\x74", [0x4329] = "\x02\x54\x7e",
  [0x432a] = "\x02\x55\x21", [0x432b] = "\x02\x54\x7d",
  [0x432c] = "\x02\x40\x76", [0x432d] = "\x01\x6f\x51",
  [0x432e] = "\x02\x55\x24", [0x432f] = "\x02\x54\x7c",
  [0x4330] = "\x01\x6f\x50", [0x4331] = "\x01\x6f\x53",
  [0x4332] = "\x02\x55\x22", [0x4333] = "\x02\x5b\x23",
  [0x4334] = "\x01\x72\x5d", [0x4335] = "\x02\x5b\x24",
  [0x4336] = "\x02\x5b\x25", [0x4337] = "\x0e\x53\x2b",
  [0x4338] = "\x02\x5f\x70", [0x4339] = "\x01\x75\x68",
  [0x433a] = "\x02\x63\x75", [0x433b] = "\x02\x63\x76",
  [0x433c] = "\x0e\x59\x3c", [0x433d] = "\x0e\x59\x3b",
  [0x433e] = "\x02\x63\x77", [0x433f] = "\x02\x67\x49",
  [0x4340] = "\x02\x67\x48", [0x4341] = "\x02\x6a\x30",
  [0x4343] = "\x02\x6a\x2f", [0x4344] = "\x02\x6c\x4a",
  [0x4345] = "\x02\x6c\x4b", [0x4346] = "\x02\x6c\x49",
  [0x4347] = "\x02\x6e\x35", [0x4348] = "\x01\x7b\x7a",
  [0x4349] = "\x01\x4b\x5c", [0x434a] = "\x01\x53\x6b",
  [0x434b] = "\x01\x53\x6a", [0x434c] = "\x01\x58\x5a",
  [0x434d] = "\x01\x58\x59", [0x434e] = "\x02\x33\x43",
  [0x434f] = "\x02\x33\x44", [0x4350] = "\x02\x33\x42",
  [0x4351] = "\x0e\x34\x2a", [0x4352] = "\x01\x58\x58",
  [0x4353] = "\x02\x39\x78", [0x4354] = "\x0e\x39\x55",
  [0x4355] = "\x02\x39\x79", [0x4356] = "\x02\x39\x75",
  [0x4357] = "\x01\x5d\x68", [0x4358] = "\x02\x39\x76",
  [0x4359] = "\x0e\x39\x54", [0x435a] = "\x02\x39\x77",
  [0x435c] = "\x0e\x66\x57", [0x435f] = "\x02\x41\x26",
  [0x4360] = "\x02\x41\x28", [0x4361] = "\x02\x41\x24",
  [0x4362] = "\x02\x41\x27", [0x4363] = "\x01\x62\x56",
  [0x4364] = "\x02\x41\x25", [0x4365] = "\x01\x62\x57",
  [0x4367] = "\x0e\x45\x28", [0x4368] = "\x02\x42\x79",
  [0x4369] = "\x01\x67\x3d", [0x436a] = "\x01\x67\x3c",
  [0x436b] = "\x0e\x45\x27", [0x436c] = "\x01\x67\x3b",
  [0x436e] = "\x02\x48\x32", [0x436f] = "\x02\x48\x33",
  [0x4372] = "\x02\x4e\x5b", [0x4373] = "\x02\x4e\x5d",
  [0x4374] = "\x01\x6b\x4e", [0x4375] = "\x01\x6b\x4b",
  [0x4376] = "\x0e\x4a\x2c", [0x4377] = "\x01\x6b\x4d",
  [0x4378] = "\x01\x6b\x4c", [0x4379] = "\x02\x4e\x5c",
  [0x437a] = "\x02\x4e\x5a", [0x437b] = "\x0e\x4a\x2b",
  [0x437c] = "\x0e\x4a\x2d", [0x4380] = "\x02\x55\x2b",
  [0x4381] = "\x02\x55\x28", [0x4382] = "\x02\x55\x29",
  [0x4383] = "\x01\x6f\x57", [0x4384] = "\x02\x55\x2a",
  [0x4385] = "\x02\x55\x25", [0x4386] = "\x02\x55\x26",
  [0x4387] = "\x01\x6f\x54", [0x4388] = "\x0e\x4f\x3f",
  [0x4389] = "\x01\x6f\x55", [0x438a] = "\x02\x55\x27",
  [0x438b] = "\x01\x6f\x56", [0x438c] = "\x0e\x4f\x40",
  [0x438d] = "\x02\x5b\x29", [0x438e] = "\x0e\x53\x2d",
  [0x438f] = "\x02\x5b\x2a", [0x4390] = "\x02\x5b\x27",
  [0x4391] = "\x02\x5b\x28", [0x4392] = "\x01\x72\x5e",
  [0x4393] = "\x02\x5b\x26", [0x4395] = "\x0e\x53\x2c",
  [0x4396] = "\x0e\x53\x2f", [0x4399] = "\x02\x5f\x74",
  [0x439a] = "\x02\x5f\x71", [0x439b] = "\x02\x5f\x73",
  [0x439c] = "\x01\x75\x6b", [0x439d] = "\x02\x5f\x77",
  [0x439e] = "\x01\x75\x6a", [0x439f] = "\x02\x5f\x75",
  [0x43a0] = "\x02\x5f\x78", [0x43a1] = "\x02\x5f\x76",
  [0x43a2] = "\x02\x5f\x72", [0x43a3] = "\x01\x75\x69",
  [0x43a5] = "\x02\x63\x79", [0x43a7] = "\x02\x63\x7a",
  [0x43a8] = "\x02\x63\x78", [0x43a9] = "\x0e\x59\x3d",
  [0x43aa] = "\x02\x63\x7c", [0x43ab] = "\x01\x77\x50",
  [0x43ac] = "\x01\x77\x51", [0x43ad] = "\x02\x67\x4b",
  [0x43ae] = "\x01\x79\x34", [0x43af] = "\x02\x63\x7b",
  [0x43b0] = "\x02\x67\x4a", [0x43b1] = "\x01\x79\x33",
  [0x43b2] = "\x02\x6a\x33", [0x43b3] = "\x02\x6a\x34",
  [0x43b4] = "\x01\x7a\x4a", [0x43b5] = "\x02\x6a\x32",
  [0x43b7] = "\x02\x6a\x31", [0x43b9] = "\x02\x6c\x4c",
  [0x43ba] = "\x01\x7b\x34", [0x43bb] = "\x0e\x5e\x68",
  [0x43bc] = "\x02\x6f\x57", [0x43bd] = "\x02\x70\x56",
  [0x43be] = "\x02\x70\x55", [0x43bf] = "\x0e\x60\x7e",
  [0x43c0] = "\x01\x7c\x6e", [0x43c1] = "\x01\x7d\x2a",
  [0x43c2] = "\x02\x70\x57", [0x43c3] = "\x02\x71\x60",
  [0x43c4] = "\x0e\x61\x6d", [0x43c5] = "\x01\x7d\x3c",
  [0x43c6] = "\x01\x4b\x5d", [0x43c7] = "\x01\x4f\x59",
  [0x43c9] = "\x01\x67\x3e", [0x43cb] = "\x01\x7a\x4b",
  [0x43cc] = "\x01\x4b\x5e", [0x43cd] = "\x01\x53\x6c",
  [0x43ce] = "\x01\x5d\x69", [0x43cf] = "\x01\x62\x58",
  [0x43d0] = "\x01\x77\x52", [0x43d1] = "\x01\x4f\x5a",
  [0x43d3] = "\x02\x2d\x62", [0x43d4] = "\x02\x2d\x61",
  [0x43d5] = "\x02\x33\x45", [0x43d6] = "\x0e\x34\x30",
  [0x43d7] = "\x01\x58\x5d", [0x43d8] = "\x01\x58\x5b",
  [0x43d9] = "\x01\x58\x5f", [0x43da] = "\x02\x33\x47",
  [0x43db] = "\x0e\x34\x2d", [0x43dc] = "\x01\x58\x5e",
  [0x43dd] = "\x01\x58\x5c", [0x43de] = "\x0e\x34\x2f",
  [0x43df] = "\x0e\x34\x2e", [0x43e0] = "\x0e\x34\x31",
  [0x43e2] = "\x02\x33\x46", [0x43e3] = "\x01\x5d\x6c",
  [0x43e4] = "\x02\x3a\x21", [0x43e5] = "\x0e\x39\x58",
  [0x43e6] = "\x01\x5d\x6b", [0x43e7] = "\x01\x5d\x6d",
  [0x43e8] = "\x02\x3a\x26", [0x43e9] = "\x01\x5d\x6f",
  [0x43ea] = "\x02\x3a\x23", [0x43eb] = "\x02\x3a\x24",
  [0x43ec] = "\x02\x39\x7a", [0x43ed] = "\x01\x5d\x6e",
  [0x43ee] = "\x02\x3a\x27", [0x43f0] = "\x0e\x39\x57",
  [0x43f1] = "\x02\x39\x7c", [0x43f2] = "\x0e\x39\x56",
  [0x43f3] = "\x02\x39\x7d", [0x43f4] = "\x02\x39\x7b",
  [0x43f5] = "\x01\x5d\x6a", [0x43f6] = "\x0e\x39\x59",
  [0x43f7] = "\x02\x3a\x25", [0x43f8] = "\x02\x39\x7e",
  [0x43f9] = "\x02\x3a\x22", [0x43fd] = "\x02\x41\x34",
  [0x43fe] = "\x0e\x3f\x5d", [0x43ff] = "\x02\x41\x33",
  [0x4400] = "\x02\x41\x31", [0x4401] = "\x02\x41\x29",
  [0x4402] = "\x02\x41\x38", [0x4403] = "\x02\x41\x2c",
  [0x4404] = "\x02\x41\x36", [0x4405] = "\x02\x41\x3d",
  [0x4406] = "\x02\x41\x35", [0x4407] = "\x01\x62\x60",
  [0x4408] = "\x0e\x3f\x66", [0x4409] = "\x01\x62\x5c",
  [0x440a] = "\x02\x41\x2a", [0x440b] = "\x0e\x3f\x60",
  [0x440c] = "\x02\x41\x30", [0x440d] = "\x01\x62\x5e",
  [0x440e] = "\x0e\x3f\x68", [0x440f] = "\x02\x41\x2f",
  [0x4410] = "\x01\x62\x5f", [0x4411] = "\x01\x62\x61",
  [0x4412] = "\x02\x41\x32", [0x4413] = "\x0e\x3f\x69",
  [0x4414] = "\x01\x62\x59", [0x4415] = "\x01\x62\x5a",
  [0x4416] = "\x02\x41\x3e", [0x4417] = "\x02\x41\x3c",
  [0x4418] = "\x0e\x3f\x62", [0x4419] = "\x02\x41\x3b",
  [0x441a] = "\x02\x41\x2d", [0x441b] = "\x0e\x3f\x65",
  [0x441c] = "\x02\x41\x39", [0x441e] = "\x01\x62\x5d",
  [0x4420] = "\x0e\x3f\x5e", [0x4421] = "\x0e\x66\x59",
  [0x4422] = "\x0e\x3f\x63", [0x4423] = "\x01\x62\x5b",
  [0x4424] = "\x02\x41\x3a", [0x4425] = "\x02\x41\x2b",
  [0x4426] = "\x02\x41\x2e", [0x4427] = "\x02\x41\x37",
  [0x442b] = "\x0e\x66\x58", [0x442d] = "\x02\x4f\x22",
  [0x442e] = "\x02\x48\x3d", [0x442f] = "\x0e\x45\x2e",
  [0x4430] = "\x02\x48\x36", [0x4431] = "\x02\x48\x49",
  [0x4432] = "\x02\x48\x52", [0x4433] = "\x02\x48\x39",
  [0x4434] = "\x01\x67\x49", [0x4435] = "\x0e\x45\x2b",
  [0x4436] = "\x02\x48\x46", [0x4437] = "\x01\x67\x3f",
  [0x4438] = "\x01\x67\x41", [0x4439] = "\x01\x67\x4d",
  [0x443a] = "\x02\x48\x37", [0x443b] = "\x0e\x45\x37",
  [0x443c] = "\x0e\x45\x2f", [0x443d] = "\x01\x67\x42",
  [0x443e] = "\x01\x67\x44", [0x443f] = "\x01\x67\x4e",
  [0x4440] = "\x01\x67\x43", [0x4441] = "\x0e\x45\x39",
  [0x4444] = "\x0e\x45\x38", [0x4445] = "\x01\x67\x4c",
  [0x4446] = "\x02\x48\x3f", [0x4448] = "\x02\x48\x34",
  [0x4449] = "\x01\x67\x4a", [0x444a] = "\x02\x48\x3e",
  [0x444b] = "\x01\x67\x46", [0x444c] = "\x02\x48\x50",
  [0x444d] = "\x01\x67\x4b", [0x444e] = "\x02\x48\x4e",
  [0x444f] = "\x02\x48\x42", [0x4450] = "\x02\x48\x4c",
  [0x4451] = "\x01\x67\x48", [0x4452] = "\x02\x48\x35",
  [0x4453] = "\x02\x48\x4f", [0x4454] = "\x02\x48\x4a",
  [0x4455] = "\x0e\x45\x2a", [0x4456] = "\x02\x48\x51",
  [0x4457] = "\x01\x67\x40", [0x4459] = "\x0e\x45\x2d",
  [0x445a] = "\x01\x67\x4f", [0x445b] = "\x01\x67\x45",
  [0x445c] = "\x0e\x45\x31", [0x445d] = "\x0e\x45\x29",
  [0x445e] = "\x02\x48\x3b", [0x445f] = "\x0e\x45\x34",
  [0x4460] = "\x02\x48\x43", [0x4461] = "\x02\x48\x47",
  [0x4462] = "\x0e\x45\x33", [0x4463] = "\x02\x48\x4b",
  [0x4464] = "\x01\x67\x47", [0x4465] = "\x02\x48\x3a",
  [0x4466] = "\x02\x48\x38", [0x4467] = "\x02\x48\x44",
  [0x446a] = "\x0e\x45\x32", [0x446b] = "\x0e\x45\x35",
  [0x446c] = "\x02\x48\x41", [0x446d] = "\x02\x48\x40",
  [0x446e] = "\x0e\x45\x36", [0x446f] = "\x02\x48\x45",
  [0x4470] = "\x02\x48\x48", [0x4472] = "\x02\x48\x4d",
  [0x4473] = "\x0e\x66\x5a", [0x4476] = "\x02\x4e\x60",
  [0x4478] = "\x01\x6b\x4f", [0x4479] = "\x02\x4e\x6a",
  [0x447a] = "\x02\x4e\x62", [0x447b] = "\x01\x6b\x55",
  [0x447c] = "\x01\x6b\x59", [0x447d] = "\x02\x4e\x73",
  [0x447e] = "\x02\x4e\x7b", [0x447f] = "\x02\x4e\x6c",
  [0x4480] = "\x01\x6b\x51", [0x4481] = "\x0e\x4a\x34",
  [0x4482] = "\x02\x4e\x70", [0x4483] = "\x02\x48\x3c",
  [0x4484] = "\x0e\x4a\x35", [0x4485] = "\x01\x6b\x52",
  [0x4486] = "\x02\x4e\x77", [0x4487] = "\x02\x4e\x7c",
  [0x4488] = "\x02\x4e\x74", [0x4489] = "\x0e\x4a\x2f",
  [0x448a] = "\x02\x4e\x76", [0x448b] = "\x02\x4f\x21",
  [0x448c] = "\x02\x4e\x78", [0x448d] = "\x02\x4e\x66",
  [0x448e] = "\x02\x4e\x6f", [0x448f] = "\x0e\x45\x30",
  [0x4490] = "\x0e\x4a\x38", [0x4491] = "\x01\x6b\x5a",
  [0x4492] = "\x0e\x4a\x33", [0x4493] = "\x01\x6b\x56",
  [0x4494] = "\x02\x4e\x64", [0x4495] = "\x02\x4e\x71",
  [0x4496] = "\x01\x6b\x54", [0x4497] = "\x02\x4e\x6b",
  [0x4498] = "\x01\x6b\x53", [0x4499] = "\x02\x4e\x79",
  [0x449a] = "\x02\x4e\x68", [0x449b] = "\x02\x4e\x61",
  [0x449c] = "\x01\x6b\x57", [0x449d] = "\x02\x4e\x7e",
  [0x449e] = "\x0e\x4a\x39", [0x44a0] = "\x02\x4e\x63",
  [0x44a1] = "\x02\x4e\x75", [0x44a2] = "\x02\x4e\x72",
  [0x44a3] = "\x02\x4e\x6d", [0x44a4] = "\x02\x4e\x5f",
  [0x44a5] = "\x02\x4e\x5e", [0x44a6] = "\x02\x4e\x67",
  [0x44a7] = "\x02\x4e\x7a", [0x44a8] = "\x01\x6b\x58",
  [0x44a9] = "\x02\x4e\x7d", [0x44aa] = "\x02\x4e\x65",
  [0x44ab] = "\x02\x4e\x69", [0x44ac] = "\x01\x6b\x50",
  [0x44ad] = "\x0e\x4a\x32", [0x44ae] = "\x0e\x4a\x37",
  [0x44b0] = "\x0e\x67\x2a", [0x44b1] = "\x0e\x4a\x31",
  [0x44b2] = "\x01\x6f\x63", [0x44b3] = "\x01\x6f\x5e",
  [0x44b4] = "\x02\x55\x4d", [0x44b5] = "\x02\x55\x49",
  [0x44b6] = "\x02\x55\x31", [0x44b7] = "\x01\x6f\x5a",
  [0x44b8] = "\x0e\x4f\x42", [0x44b9] = "\x0e\x4f\x50",
  [0x44bb] = "\x01\x6f\x59", [0x44bc] = "\x01\x6f\x5f",
  [0x44be] = "\x0e\x4f\x43", [0x44bf] = "\x0e\x4f\x41",
  [0x44c0] = "\x02\x55\x2f", [0x44c1] = "\x01\x6f\x5d",
  [0x44c2] = "\x02\x55\x3b", [0x44c3] = "\x02\x55\x2d",
  [0x44c4] = "\x02\x55\x2e", [0x44c5] = "\x01\x6f\x58",
  [0x44c6] = "\x02\x55\x4c", [0x44c7] = "\x01\x6f\x61",
  [0x44c8] = "\x02\x55\x3e", [0x44c9] = "\x02\x55\x43",
  [0x44ca] = "\x02\x55\x3d", [0x44cb] = "\x02\x5b\x3c",
  [0x44cc] = "\x02\x55\x39", [0x44cd] = "\x02\x55\x41",
  [0x44ce] = "\x02\x55\x3f", [0x44cf] = "\x02\x55\x32",
  [0x44d0] = "\x02\x55\x2c", [0x44d1] = "\x02\x55\x47",
  [0x44d2] = "\x01\x6f\x60", [0x44d3] = "\x02\x55\x48",
  [0x44d4] = "\x0e\x4f\x48", [0x44d5] = "\x02\x55\x42",
  [0x44d6] = "\x0e\x4f\x49", [0x44d7] = "\x02\x55\x37",
  [0x44d8] = "\x02\x55\x35", [0x44d9] = "\x02\x55\x30",
  [0x44da] = "\x0e\x4f\x4b", [0x44db] = "\x0e\x4f\x4e",
  [0x44dc] = "\x0e\x4f\x44", [0x44dd] = "\x02\x55\x38",
  [0x44de] = "\x02\x55\x45", [0x44df] = "\x02\x55\x34",
  [0x44e0] = "\x02\x55\x44", [0x44e1] = "\x02\x55\x4a",
  [0x44e2] = "\x0e\x4f\x51", [0x44e4] = "\x01\x6f\x5c",
  [0x44e5] = "\x0e\x4f\x45", [0x44e6] = "\x02\x55\x40",
  [0x44e7] = "\x02\x55\x46", [0x44e8] = "\x02\x55\x3c",
  [0x44e9] = "\x02\x55\x36", [0x44ea] = "\x01\x6f\x5b",
  [0x44eb] = "\x0e\x4f\x52", [0x44ed] = "\x0e\x4f\x4c",
  [0x44ee] = "\x02\x4e\x6e", [0x44ef] = "\x02\x55\x3a",
  [0x44f0] = "\x01\x6f\x62", [0x44f1] = "\x02\x55\x33",
  [0x44f3] = "\x0e\x4f\x4d", [0x44f4] = "\x0e\x66\x5b",
  [0x44f5] = "\x0e\x66\x5c", [0x44f6] = "\x0e\x4a\x30",
  [0x44f7] = "\x02\x5b\x41", [0x44f8] = "\x01\x72\x61",
  [0x44f9] = "\x02\x5b\x40", [0x44fa] = "\x02\x5b\x3e",
  [0x44fb] = "\x02\x5b\x50", [0x44fc] = "\x01\x72\x65",
  [0x44fd] = "\x0e\x53\x35", [0x44fe] = "\x02\x5b\x4d",
  [0x44ff] = "\x02\x5b\x45", [0x4500] = "\x02\x5b\x4f",
  [0x4501] = "\x02\x5b\x37", [0x4502] = "\x02\x5b\x43",
  [0x4503] = "\x0e\x53\x3d", [0x4504] = "\x01\x72\x67",
  [0x4505] = "\x0e\x53\x3e", [0x4506] = "\x02\x5b\x2f",
  [0x4507] = "\x0e\x53\x38", [0x4508] = "\x02\x5b\x2d",
  [0x4509] = "\x02\x5b\x4e", [0x450a] = "\x0e\x53\x32",
  [0x450b] = "\x02\x5b\x4c", [0x450c] = "\x02\x5b\x4b",
  [0x450d] = "\x02\x5b\x3b", [0x450e] = "\x02\x5b\x3a",
  [0x450f] = "\x02\x5b\x30", [0x4510] = "\x01\x72\x69",
  [0x4512] = "\x02\x5b\x36", [0x4513] = "\x02\x5b\x3f",
  [0x4514] = "\x02\x5b\x4a", [0x4515] = "\x01\x72\x6c",
  [0x4516] = "\x02\x5b\x51", [0x4517] = "\x0e\x53\x36",
  [0x4518] = "\x01\x75\x73", [0x4519] = "\x01\x72\x6e",
  [0x451a] = "\x01\x72\x68", [0x451b] = "\x02\x5b\x34",
  [0x451c] = "\x0e\x53\x37", [0x451d] = "\x02\x5b\x3d",
  [0x451e] = "\x02\x5b\x2c", [0x451f] = "\x02\x5b\x2e",
  [0x4520] = "\x01\x72\x5f", [0x4521] = "\x01\x72\x6b",
  [0x4522] = "\x01\x72\x64", [0x4523] = "\x02\x5b\x35",
  [0x4524] = "\x02\x5b\x44", [0x4525] = "\x02\x55\x4b",
  [0x4526] = "\x01\x72\x6a", [0x4527] = "\x02\x5b\x2b",
  [0x4528] = "\x01\x75\x6e", [0x4529] = "\x02\x5b\x46",
  [0x452a] = "\x02\x5b\x49", [0x452b] = "\x01\x72\x66",
  [0x452c] = "\x0e\x53\x3b", [0x452d] = "\x02\x5b\x39",
  [0x452e] = "\x01\x72\x6d", [0x452f] = "\x01\x72\x63",
  [0x4530] = "\x0e\x53\x3c", [0x4531] = "\x0e\x53\x39",
  [0x4532] = "\x0e\x53\x3a", [0x4533] = "\x01\x72\x62",
  [0x4534] = "\x02\x5b\x42", [0x4535] = "\x02\x5b\x48",
  [0x4536] = "\x01\x72\x60", [0x4538] = "\x02\x5b\x32",
  [0x4539] = "\x02\x5b\x47", [0x453c] = "\x02\x5b\x33",
  [0x453f] = "\x0e\x66\x5d", [0x4542] = "\x0e\x66\x5e",
  [0x4545] = "\x0e\x53\x31", [0x4546] = "\x02\x5b\x38",
  [0x4547] = "\x02\x5f\x7e", [0x4548] = "\x0e\x56\x65",
  [0x4549] = "\x02\x60\x25", [0x454a] = "\x01\x75\x70",
  [0x454b] = "\x01\x75\x72", [0x454c] = "\x02\x60\x2b",
  [0x454d] = "\x01\x75\x6c", [0x454e] = "\x02\x60\x39",
  [0x454f] = "\x02\x60\x31", [0x4550] = "\x02\x60\x26",
  [0x4551] = "\x02\x60\x27", [0x4552] = "\x02\x60\x30",
  [0x4553] = "\x0e\x56\x66", [0x4554] = "\x01\x75\x79",
  [0x4555] = "\x02\x60\x2f", [0x4556] = "\x02\x5f\x7d",
  [0x4557] = "\x02\x60\x2e", [0x4558] = "\x02\x60\x22",
  [0x4559] = "\x02\x60\x3a", [0x455a] = "\x01\x75\x78",
  [0x455b] = "\x01\x75\x76", [0x455c] = "\x02\x60\x23",
  [0x455d] = "\x0e\x56\x5d", [0x455e] = "\x02\x60\x36",
  [0x455f] = "\x0e\x56\x67", [0x4560] = "\x02\x60\x28",
  [0x4561] = "\x02\x60\x35", [0x4562] = "\x0e\x56\x64",
  [0x4563] = "\x02\x60\x37", [0x4564] = "\x02\x5f\x7c",
  [0x4565] = "\x01\x75\x71", [0x4566] = "\x0e\x56\x5b",
  [0x4567] = "\x02\x60\x38", [0x4568] = "\x0e\x56\x68",
  [0x4569] = "\x0e\x56\x5e", [0x456a] = "\x02\x60\x2c",
  [0x456b] = "\x0e\x56\x6b", [0x456c] = "\x01\x75\x75",
  [0x456d] = "\x02\x60\x29", [0x456e] = "\x0e\x56\x61",
  [0x4570] = "\x01\x75\x77", [0x4571] = "\x02\x60\x32",
  [0x4572] = "\x0e\x56\x62", [0x4573] = "\x0e\x56\x63",
  [0x4574] = "\x0e\x56\x5f", [0x4575] = "\x01\x75\x6f",
  [0x4576] = "\x02\x60\x24", [0x4577] = "\x02\x60\x33",
  [0x4578] = "\x0e\x56\x5c", [0x4579] = "\x02\x60\x2d",
  [0x457a] = "\x02\x5b\x31", [0x457b] = "\x02\x60\x34",
  [0x457c] = "\x02\x60\x21", [0x457d] = "\x0e\x56\x60",
  [0x457e] = "\x01\x75\x74", [0x457f] = "\x0e\x56\x6a",
  [0x4580] = "\x02\x64\x2f", [0x4582] = "\x01\x75\x6d",
  [0x4583] = "\x02\x5f\x7a", [0x4586] = "\x0e\x66\x5f",
  [0x4588] = "\x02\x64\x2c", [0x4589] = "\x02\x64\x25",
  [0x458a] = "\x01\x77\x54", [0x458b] = "\x0e\x59\x43",
  [0x458c] = "\x02\x63\x7e", [0x458d] = "\x02\x64\x30",
  [0x458e] = "\x02\x64\x27", [0x458f] = "\x02\x60\x2a",
  [0x4590] = "\x0e\x59\x40", [0x4591] = "\x02\x64\x32",
  [0x4592] = "\x02\x64\x21", [0x4593] = "\x0e\x59\x42",
  [0x4594] = "\x01\x77\x53", [0x4595] = "\x02\x64\x2b",
  [0x4596] = "\x01\x77\x55", [0x4597] = "\x01\x77\x5d",
  [0x4598] = "\x01\x77\x5b", [0x4599] = "\x02\x64\x2d",
  [0x459a] = "\x01\x77\x5c", [0x459b] = "\x02\x64\x23",
  [0x459c] = "\x0e\x59\x49", [0x459d] = "\x02\x64\x24",
  [0x459e] = "\x02\x64\x29", [0x459f] = "\x02\x64\x2e",
  [0x45a0] = "\x0e\x59\x3e", [0x45a1] = "\x02\x5f\x79",
  [0x45a2] = "\x01\x77\x56", [0x45a3] = "\x02\x64\x37",
  [0x45a4] = "\x02\x64\x34", [0x45a5] = "\x02\x67\x61",
  [0x45a6] = "\x02\x64\x2a", [0x45a7] = "\x02\x64\x26",
  [0x45a8] = "\x02\x64\x35", [0x45a9] = "\x02\x67\x56",
  [0x45aa] = "\x02\x64\x28", [0x45ac] = "\x01\x77\x59",
  [0x45ad] = "\x0e\x59\x44", [0x45ae] = "\x01\x77\x58",
  [0x45af] = "\x02\x5f\x7b", [0x45b0] = "\x01\x77\x5a",
  [0x45b1] = "\x02\x64\x31", [0x45b2] = "\x02\x64\x33",
  [0x45b3] = "\x01\x77\x57", [0x45b4] = "\x02\x64\x36",
  [0x45b5] = "\x02\x63\x7d", [0x45b7] = "\x02\x64\x22",
  [0x45b8] = "\x0e\x59\x46", [0x45bb] = "\x0e\x59\x45",
  [0x45bc] = "\x0e\x66\x61", [0x45bd] = "\x0e\x66\x62",
  [0x45be] = "\x0e\x66\x60", [0x45c0] = "\x02\x67\x5f",
  [0x45c1] = "\x0e\x5b\x50", [0x45c2] = "\x02\x67\x4f",
  [0x45c3] = "\x01\x79\x38", [0x45c4] = "\x02\x67\x5d",
  [0x45c5] = "\x0e\x5b\x44", [0x45c6] = "\x0e\x5b\x45",
  [0x45c7] = "\x02\x67\x4d", [0x45c8] = "\x01\x79\x39",
  [0x45ca] = "\x02\x67\x58", [0x45cb] = "\x0e\x5b\x43",
  [0x45cc] = "\x02\x67\x54", [0x45cd] = "\x01\x79\x3e",
  [0x45ce] = "\x02\x67\x5e", [0x45cf] = "\x02\x67\x4e",
  [0x45d0] = "\x02\x67\x51", [0x45d1] = "\x01\x79\x36",
  [0x45d2] = "\x02\x67\x60", [0x45d3] = "\x0e\x5b\x48",
  [0x45d4] = "\x02\x67\x59", [0x45d5] = "\x02\x67\x5c",
  [0x45d6] = "\x01\x79\x3c", [0x45d7] = "\x01\x79\x41",
  [0x45d8] = "\x01\x79\x3f", [0x45d9] = "\x02\x67\x55",
  [0x45da] = "\x02\x67\x50", [0x45db] = "\x0e\x5b\x4d",
  [0x45dc] = "\x01\x79\x3a", [0x45dd] = "\x01\x79\x3b",
  [0x45de] = "\x02\x67\x4c", [0x45df] = "\x01\x79\x37",
  [0x45e0] = "\x0e\x5b\x4f", [0x45e1] = "\x01\x79\x35",
  [0x45e2] = "\x01\x79\x3d", [0x45e3] = "\x02\x67\x5b",
  [0x45e4] = "\x01\x79\x40", [0x45e5] = "\x0e\x5b\x4a",
  [0x45e6] = "\x02\x67\x57", [0x45e7] = "\x02\x67\x62",
  [0x45e8] = "\x01\x79\x42", [0x45e9] = "\x0e\x5b\x41",
  [0x45ea] = "\x0e\x5b\x42", [0x45eb] = "\x0e\x5b\x4e",
  [0x45ec] = "\x02\x67\x53", [0x45ed] = "\x0e\x5b\x47",
  [0x45ee] = "\x02\x67\x5a", [0x45ef] = "\x0e\x66\x65",
  [0x45f1] = "\x0e\x66\x63", [0x45f2] = "\x0e\x66\x64",
  [0x45f3] = "\x0e\x5d\x41", [0x45f4] = "\x0e\x5e\x6a",
  [0x45f5] = "\x02\x6a\x41", [0x45f6] = "\x02\x6a\x50",
  [0x45f7] = "\x02\x6a\x43", [0x45f8] = "\x02\x6a\x4a",
  [0x45f9] = "\x02\x67\x52", [0x45fa] = "\x02\x6a\x48",
  [0x45fb] = "\x02\x6a\x37", [0x45fc] = "\x02\x6a\x4e",
  [0x45fd] = "\x01\x7a\x4e", [0x45fe] = "\x02\x6a\x3b",
  [0x45ff] = "\x02\x6a\x4d", [0x4600] = "\x02\x6a\x42",
  [0x4602] = "\x0e\x5d\x43", [0x4603] = "\x01\x7a\x4d",
  [0x4604] = "\x0e\x5d\x3c", [0x4605] = "\x0e\x5d\x3f",
  [0x4606] = "\x02\x6a\x52", [0x4607] = "\x02\x6a\x44",
  [0x4608] = "\x0e\x5d\x3e", [0x4609] = "\x02\x6a\x49",
  [0x460a] = "\x02\x6a\x4c", [0x460b] = "\x02\x6a\x35",
  [0x460c] = "\x02\x6a\x4f", [0x460d] = "\x02\x6a\x40",
  [0x460e] = "\x02\x6a\x45", [0x460f] = "\x02\x6a\x39",
  [0x4610] = "\x02\x6a\x3d", [0x4611] = "\x02\x6a\x51",
  [0x4612] = "\x02\x6a\x47", [0x4613] = "\x02\x6a\x36",
  [0x4614] = "\x02\x6a\x3a", [0x4615] = "\x02\x6a\x3c",
  [0x4616] = "\x02\x6a\x46", [0x4617] = "\x0e\x5d\x3d",
  [0x4618] = "\x01\x7a\x4c", [0x4619] = "\x02\x6a\x3f",
  [0x461a] = "\x0e\x5d\x44", [0x461b] = "\x0e\x5d\x45",
  [0x461c] = "\x0e\x5d\x47", [0x461e] = "\x0e\x5d\x42",
  [0x461f] = "\x0e\x5d\x3a", [0x4620] = "\x02\x6a\x38",
  [0x4622] = "\x0e\x66\x66", [0x4623] = "\x0e\x66\x67",
  [0x4624] = "\x0e\x66\x68", [0x4627] = "\x0e\x5d\x46",
  [0x4628] = "\x02\x6a\x3e", [0x4629] = "\x02\x6c\x50",
  [0x462a] = "\x02\x6c\x54", [0x462b] = "\x01\x7b\x3b",
  [0x462c] = "\x02\x6c\x56", [0x462d] = "\x0e\x5e\x6d",
  [0x462e] = "\x01\x7b\x35", [0x4630] = "\x02\x6c\x52",
  [0x4631] = "\x02\x6c\x58", [0x4632] = "\x01\x7b\x3a",
  [0x4633] = "\x01\x7b\x36", [0x4635] = "\x01\x7b\x37",
  [0x4636] = "\x02\x6c\x4f", [0x4637] = "\x02\x6c\x55",
  [0x4638] = "\x01\x7b\x39", [0x4639] = "\x02\x6c\x53",
  [0x463a] = "\x01\x7b\x38", [0x463b] = "\x02\x6c\x4e",
  [0x463c] = "\x02\x6a\x4b", [0x463d] = "\x02\x6c\x51",
  [0x463f] = "\x02\x6c\x4d", [0x4640] = "\x02\x6c\x57",
  [0x4642] = "\x0e\x5f\x69", [0x4643] = "\x0e\x5f\x6e",
  [0x4644] = "\x01\x7b\x7b", [0x4645] = "\x02\x6e\x3c",
  [0x4646] = "\x02\x6e\x3f", [0x4647] = "\x02\x6e\x3b",
  [0x4648] = "\x02\x6e\x3d", [0x4649] = "\x02\x6e\x3e",
  [0x464a] = "\x02\x6e\x38", [0x464b] = "\x02\x6e\x39",
  [0x464c] = "\x02\x6e\x36", [0x464d] = "\x0e\x5f\x6a",
  [0x464e] = "\x0e\x5f\x6c", [0x464f] = "\x02\x6e\x3a",
  [0x4650] = "\x02\x6e\x37", [0x4651] = "\x01\x7b\x7c",
  [0x4652] = "\x01\x7b\x7d", [0x4654] = "\x0e\x5f\x6b",
  [0x4655] = "\x02\x6f\x59", [0x4656] = "\x0e\x60\x52",
  [0x4657] = "\x02\x6f\x5b", [0x4658] = "\x0e\x64\x31",
  [0x465a] = "\x0e\x60\x55", [0x465b] = "\x0e\x60\x54",
  [0x465c] = "\x0e\x60\x56", [0x465d] = "\x02\x6f\x5a",
  [0x465e] = "\x02\x6f\x5c", [0x465f] = "\x0e\x60\x4f",
  [0x4660] = "\x01\x7c\x4c", [0x4661] = "\x0e\x60\x53",
  [0x4662] = "\x02\x6f\x58", [0x4663] = "\x01\x7c\x4b",
  [0x4664] = "\x01\x7c\x4d", [0x4665] = "\x0e\x60\x57",
  [0x4666] = "\x0e\x66\x6a", [0x4667] = "\x0e\x66\x69",
  [0x4668] = "\x02\x70\x59", [0x4669] = "\x02\x70\x5a",
  [0x466a] = "\x01\x7c\x6f", [0x466b] = "\x02\x70\x58",
  [0x466c] = "\x0e\x61\x22", [0x466d] = "\x02\x71\x42",
  [0x466e] = "\x02\x71\x41", [0x466f] = "\x02\x71\x43",
  [0x4670] = "\x01\x7d\x2c", [0x4671] = "\x02\x71\x44",
  [0x4672] = "\x01\x7d\x2b", [0x4673] = "\x02\x71\x45",
  [0x4674] = "\x02\x71\x61", [0x4675] = "\x02\x71\x64",
  [0x4676] = "\x02\x71\x63", [0x4677] = "\x01\x7d\x35",
  [0x4678] = "\x02\x71\x62", [0x467a] = "\x0e\x61\x6f",
  [0x467b] = "\x0e\x62\x23", [0x467c] = "\x01\x7d\x3f",
  [0x467d] = "\x01\x7d\x3d", [0x467e] = "\x01\x7d\x3e",
  [0x467f] = "\x01\x7d\x44", [0x4680] = "\x02\x72\x30",
  [0x4681] = "\x02\x72\x31", [0x4682] = "\x02\x72\x2f",
  [0x4683] = "\x02\x72\x37", [0x4684] = "\x0e\x62\x36",
  [0x4777] = "\x01\x4f\x5b", [0x4779] = "\x0e\x63\x23",
  [0x477a] = "\x02\x3a\x28", [0x477b] = "\x02\x41\x3f",
  [0x477c] = "\x02\x55\x4e", [0x477d] = "\x02\x67\x63",
  [0x4780] = "\x01\x4f\x5c", [0x4781] = "\x0e\x2f\x40",
  [0x4782] = "\x01\x53\x6d", [0x4783] = "\x01\x58\x60",
  [0x4785] = "\x0e\x34\x32", [0x4786] = "\x02\x3a\x29",
  [0x4787] = "\x0e\x39\x5c", [0x4788] = "\x02\x3a\x2a",
  [0x4789] = "\x01\x5d\x70", [0x478b] = "\x01\x62\x64",
  [0x478c] = "\x02\x41\x41", [0x478d] = "\x02\x41\x40",
  [0x478e] = "\x01\x62\x68", [0x478f] = "\x01\x62\x63",
  [0x4790] = "\x02\x41\x42", [0x4791] = "\x01\x62\x65",
  [0x4792] = "\x01\x62\x67", [0x4793] = "\x01\x62\x66",
  [0x4794] = "\x01\x62\x62", [0x4795] = "\x0e\x3f\x6c",
  [0x4797] = "\x0e\x66\x6b", [0x4798] = "\x01\x67\x50",
  [0x4799] = "\x0e\x45\x3c", [0x479a] = "\x0e\x45\x3a",
  [0x479b] = "\x02\x48\x56", [0x479c] = "\x02\x48\x54",
  [0x479e] = "\x02\x48\x55", [0x479f] = "\x02\x48\x53",
  [0x47a1] = "\x01\x6b\x5b", [0x47a2] = "\x0e\x4a\x3d",
  [0x47a3] = "\x01\x6b\x5e", [0x47a4] = "\x01\x6b\x60",
  [0x47a5] = "\x01\x6b\x5f", [0x47a7] = "\x0e\x4a\x3e",
  [0x47a8] = "\x01\x6b\x5c", [0x47a9] = "\x01\x6b\x5d",
  [0x47ab] = "\x02\x55\x50", [0x47ac] = "\x02\x55\x4f",
  [0x47ad] = "\x01\x6f\x64", [0x47ae] = "\x02\x55\x51",
  [0x47af] = "\x0e\x4f\x53", [0x47b0] = "\x02\x55\x52",
  [0x47b1] = "\x01\x6f\x65", [0x47b2] = "\x0e\x4f\x55",
  [0x47b3] = "\x0e\x4f\x56", [0x47b5] = "\x02\x5b\x59",
  [0x47b6] = "\x02\x5b\x57", [0x47b7] = "\x02\x60\x40",
  [0x47b8] = "\x0e\x53\x42", [0x47b9] = "\x02\x5b\x55",
  [0x47ba] = "\x02\x5b\x56", [0x47bb] = "\x01\x72\x6f",
  [0x47bc] = "\x02\x5b\x52", [0x47bd] = "\x02\x5b\x5a",
  [0x47be] = "\x02\x5b\x54", [0x47bf] = "\x02\x5b\x58",
  [0x47c0] = "\x02\x60\x3c", [0x47c1] = "\x0e\x53\x44",
  [0x47c2] = "\x0e\x53\x40", [0x47c3] = "\x02\x60\x3e",
  [0x47c4] = "\x0e\x56\x6d", [0x47c5] = "\x02\x60\x3f",
  [0x47c6] = "\x01\x75\x7e", [0x47c7] = "\x02\x60\x3b",
  [0x47c8] = "\x01\x75\x7d", [0x47c9] = "\x02\x60\x3d",
  [0x47ca] = "\x01\x75\x7a", [0x47cb] = "\x01\x75\x7b",
  [0x47cc] = "\x01\x75\x7c", [0x47cd] = "\x02\x5b\x53",
  [0x47ce] = "\x0e\x66\x6c", [0x47d0] = "\x01\x77\x60",
  [0x47d1] = "\x02\x64\x3a", [0x47d2] = "\x02\x64\x38",
  [0x47d3] = "\x02\x64\x39", [0x47d4] = "\x01\x77\x5e",
  [0x47d5] = "\x01\x77\x61", [0x47d6] = "\x01\x77\x5f",
  [0x47d7] = "\x0e\x59\x4b", [0x47d8] = "\x0e\x59\x4a",
  [0x47da] = "\x02\x67\x64", [0x47db] = "\x02\x67\x65",
  [0x47dc] = "\x01\x79\x43", [0x47dd] = "\x0e\x5b\x53",
  [0x47de] = "\x02\x6a\x53", [0x47df] = "\x02\x6a\x55",
  [0x47e0] = "\x02\x6a\x54", [0x47e1] = "\x01\x7a\x4f",
  [0x47e2] = "\x01\x7b\x3c", [0x47e3] = "\x02\x6c\x5b",
  [0x47e4] = "\x02\x6c\x5a", [0x47e5] = "\x02\x6c\x59",
  [0x47e7] = "\x0e\x66\x6d", [0x481c] = "\x01\x4f\x5d",
  [0x481d] = "\x0e\x21\x43", [0x481e] = "\x02\x21\x69",
  [0x4820] = "\x02\x22\x78", [0x4821] = "\x01\x48\x6a",
  [0x4822] = "\x02\x22\x76", [0x4823] = "\x02\x22\x79",
  [0x4824] = "\x02\x22\x77", [0x4826] = "\x0e\x27\x3b",
  [0x4827] = "\x0e\x27\x3a", [0x4828] = "\x02\x25\x2e",
  [0x4829] = "\x0e\x27\x3f", [0x482a] = "\x01\x4b\x62",
  [0x482b] = "\x0e\x27\x3c", [0x482c] = "\x01\x4b\x63",
  [0x482d] = "\x02\x25\x30", [0x482e] = "\x01\x4b\x60",
  [0x482f] = "\x02\x25\x2f", [0x4830] = "\x02\x25\x2d",
  [0x4831] = "\x01\x4b\x61", [0x4832] = "\x01\x4b\x5f",
  [0x4833] = "\x0e\x27\x3d", [0x4834] = "\x0e\x27\x3e",
  [0x4837] = "\x0e\x2a\x79", [0x4839] = "\x02\x28\x78",
  [0x483a] = "\x02\x28\x7b", [0x483b] = "\x01\x4f\x60",
  [0x483c] = "\x02\x28\x7a", [0x483d] = "\x02\x28\x79",
  [0x483e] = "\x0e\x2a\x7c", [0x483f] = "\x01\x4f\x5f",
  [0x4840] = "\x01\x4f\x5e", [0x4841] = "\x0e\x2a\x7b",
  [0x4842] = "\x01\x4f\x62", [0x4843] = "\x02\x28\x7c",
  [0x4844] = "\x01\x4f\x61", [0x484a] = "\x02\x2d\x67",
  [0x484b] = "\x01\x53\x6f", [0x484c] = "\x01\x53\x70",
  [0x484d] = "\x01\x53\x71", [0x484e] = "\x02\x2d\x68",
  [0x484f] = "\x02\x2d\x64", [0x4850] = "\x01\x53\x6e",
  [0x4851] = "\x02\x2d\x65", [0x4853] = "\x02\x2d\x66",
  [0x4854] = "\x02\x2d\x63", [0x4856] = "\x0e\x34\x36",
  [0x4858] = "\x01\x58\x67", [0x4859] = "\x0e\x34\x34",
  [0x485a] = "\x0e\x39\x5e", [0x485b] = "\x01\x58\x64",
  [0x485c] = "\x02\x33\x48", [0x485d] = "\x01\x58\x65",
  [0x485e] = "\x01\x58\x68", [0x485f] = "\x02\x33\x49",
  [0x4860] = "\x0e\x34\x33", [0x4861] = "\x01\x58\x63",
  [0x4862] = "\x01\x58\x61", [0x4863] = "\x01\x58\x62",
  [0x4864] = "\x01\x58\x66", [0x486a] = "\x01\x5d\x71",
  [0x486b] = "\x02\x3a\x2d", [0x486c] = "\x01\x5d\x79",
  [0x486d] = "\x02\x3a\x2c", [0x486e] = "\x0e\x39\x61",
  [0x486f] = "\x02\x3a\x2f", [0x4870] = "\x01\x5d\x75",
  [0x4871] = "\x02\x3a\x2e", [0x4872] = "\x01\x62\x70",
  [0x4873] = "\x01\x5d\x73", [0x4874] = "\x01\x5d\x76",
  [0x4875] = "\x01\x5d\x72", [0x4876] = "\x01\x5d\x77",
  [0x4877] = "\x01\x5d\x78", [0x4878] = "\x01\x5d\x74",
  [0x4879] = "\x0e\x39\x65", [0x487b] = "\x0e\x3f\x71",
  [0x487c] = "\x02\x3a\x2b", [0x487d] = "\x01\x62\x6c",
  [0x487e] = "\x02\x41\x44", [0x487f] = "\x0e\x3f\x72",
  [0x4880] = "\x02\x41\x48", [0x4881] = "\x0e\x3f\x73",
  [0x4882] = "\x0e\x3f\x75", [0x4883] = "\x02\x41\x47",
  [0x4884] = "\x01\x62\x71", [0x4885] = "\x01\x62\x6d",
  [0x4886] = "\x01\x62\x6e", [0x4887] = "\x02\x41\x43",
  [0x4888] = "\x02\x41\x45", [0x4889] = "\x02\x41\x46",
  [0x488a] = "\x01\x62\x69", [0x488b] = "\x01\x62\x6b",
  [0x488c] = "\x0e\x3f\x70", [0x488d] = "\x01\x62\x6f",
  [0x488e] = "\x01\x62\x6a", [0x488f] = "\x0e\x45\x40",
  [0x4891] = "\x02\x48\x59", [0x4892] = "\x02\x48\x57",
  [0x4893] = "\x02\x48\x58", [0x4894] = "\x01\x67\x52",
  [0x4895] = "\x01\x67\x53", [0x4896] = "\x0e\x45\x42",
  [0x4897] = "\x02\x48\x5a", [0x4898] = "\x01\x67\x51",
  [0x4899] = "\x01\x6b\x61", [0x489b] = "\x01\x6b\x63",
  [0x489c] = "\x01\x6b\x62", [0x489d] = "\x0e\x4a\x45",
  [0x489e] = "\x02\x4f\x23", [0x489f] = "\x0e\x4a\x46",
  [0x48a0] = "\x0e\x4a\x44", [0x48a1] = "\x02\x4f\x24",
  [0x48a2] = "\x02\x55\x54", [0x48a3] = "\x0e\x4f\x5a",
  [0x48a4] = "\x02\x55\x53", [0x48a5] = "\x0e\x4f\x5b",
  [0x48a7] = "\x01\x72\x70", [0x48a8] = "\x01\x72\x71",
  [0x48a9] = "\x02\x5b\x5b", [0x48aa] = "\x01\x72\x72",
  [0x48ab] = "\x0e\x53\x48", [0x48ac] = "\x02\x60\x43",
  [0x48ad] = "\x0e\x56\x70", [0x48ae] = "\x02\x60\x41",
  [0x48af] = "\x0e\x56\x6f", [0x48b0] = "\x02\x60\x42",
  [0x48b1] = "\x01\x76\x21", [0x48b2] = "\x0e\x56\x71",
  [0x48b3] = "\x02\x64\x3b", [0x48b4] = "\x01\x79\x44",
  [0x48b5] = "\x0e\x5d\x49", [0x48b6] = "\x0e\x2a\x7d",
  [0x48b7] = "\x0e\x53\x49", [0x48b8] = "\x01\x76\x22",
  [0x48b9] = "\x01\x4f\x63", [0x48ba] = "\x0e\x63\x25",
  [0x48bb] = "\x01\x58\x69", [0x48bc] = "\x02\x33\x4a",
  [0x48bd] = "\x0e\x34\x39", [0x48bf] = "\x02\x3a\x30",
  [0x48c0] = "\x01\x5d\x7a", [0x48c1] = "\x01\x62\x72",
  [0x48c2] = "\x02\x41\x49", [0x48c3] = "\x02\x41\x4b",
  [0x48c4] = "\x01\x62\x74", [0x48c5] = "\x01\x62\x73",
  [0x48c6] = "\x01\x62\x75", [0x48c7] = "\x01\x62\x76",
  [0x48c8] = "\x02\x41\x4a", [0x48c9] = "\x01\x67\x56",
  [0x48ca] = "\x01\x67\x57", [0x48cb] = "\x01\x67\x55",
  [0x48cc] = "\x01\x6b\x64", [0x48cd] = "\x01\x67\x54",
  [0x48ce] = "\x02\x48\x5b", [0x48d0] = "\x0e\x4a\x47",
  [0x48d1] = "\x0e\x4a\x49", [0x48d2] = "\x01\x6b\x65",
  [0x48d3] = "\x02\x55\x55", [0x48d4] = "\x02\x5b\x5c",
  [0x48d5] = "\x01\x72\x73", [0x48d6] = "\x01\x76\x23",
  [0x48d7] = "\x02\x64\x3c", [0x48d8] = "\x02\x64\x40",
  [0x48d9] = "\x01\x77\x64", [0x48da] = "\x02\x64\x3d",
  [0x48db] = "\x01\x77\x65", [0x48dc] = "\x01\x77\x63",
  [0x48dd] = "\x02\x64\x41", [0x48de] = "\x01\x77\x66",
  [0x48df] = "\x02\x64\x3f", [0x48e1] = "\x02\x67\x66",
  [0x48e2] = "\x01\x77\x62", [0x48e3] = "\x01\x79\x45",
  [0x48e4] = "\x0e\x5e\x70", [0x48e5] = "\x02\x70\x5b",
  [0x48e6] = "\x0e\x61\x25", [0x48e7] = "\x0e\x62\x2f",
  [0x48e8] = "\x01\x4f\x64", [0x48e9] = "\x01\x5d\x7c",
  [0x48ea] = "\x01\x5d\x7b", [0x48ec] = "\x0e\x3f\x7e",
  [0x48ed] = "\x0e\x3f\x7c", [0x48ee] = "\x0e\x3f\x7b",
  [0x48ef] = "\x01\x62\x77", [0x48f0] = "\x02\x41\x4d",
  [0x48f1] = "\x02\x41\x4c", [0x48f2] = "\x01\x62\x78",
  [0x48f4] = "\x0e\x45\x46", [0x48f5] = "\x02\x48\x5f",
  [0x48f6] = "\x01\x67\x5b", [0x48f7] = "\x01\x67\x58",
  [0x48f8] = "\x02\x48\x5e", [0x48f9] = "\x01\x67\x5a",
  [0x48fa] = "\x02\x48\x5c", [0x48fb] = "\x01\x67\x59",
  [0x48fc] = "\x0e\x45\x47", [0x48fd] = "\x02\x48\x5d",
  [0x48ff] = "\x02\x4f\x25", [0x4900] = "\x01\x6b\x66",
  [0x4902] = "\x02\x55\x58", [0x4904] = "\x01\x6f\x66",
  [0x4905] = "\x02\x55\x56", [0x4906] = "\x01\x6f\x67",
  [0x4907] = "\x01\x6f\x68", [0x4908] = "\x02\x55\x57",
  [0x4909] = "\x01\x6f\x69", [0x490a] = "\x0e\x4f\x60",
  [0x490b] = "\x02\x5b\x5d", [0x490c] = "\x0e\x53\x50",
  [0x490d] = "\x01\x72\x77", [0x490e] = "\x01\x72\x74",
  [0x490f] = "\x01\x72\x79", [0x4910] = "\x02\x5b\x5f",
  [0x4911] = "\x01\x72\x75", [0x4912] = "\x02\x5b\x5e",
  [0x4913] = "\x01\x72\x78", [0x4914] = "\x0e\x53\x4d",
  [0x4915] = "\x0e\x53\x4c", [0x4916] = "\x01\x72\x76",
  [0x4917] = "\x0e\x53\x51", [0x4918] = "\x02\x60\x46",
  [0x4919] = "\x02\x60\x48", [0x491a] = "\x0e\x56\x74",
  [0x491b] = "\x0e\x56\x75", [0x491c] = "\x01\x76\x24",
  [0x491d] = "\x02\x60\x47", [0x491e] = "\x01\x76\x25",
  [0x491f] = "\x02\x60\x45", [0x4920] = "\x02\x60\x44",
  [0x4922] = "\x02\x64\x43", [0x4923] = "\x02\x64\x42",
  [0x4924] = "\x01\x77\x67", [0x4925] = "\x02\x64\x44",
  [0x4926] = "\x02\x67\x6b", [0x4927] = "\x01\x79\x47",
  [0x4928] = "\x02\x67\x6a", [0x4929] = "\x02\x67\x67",
  [0x492a] = "\x01\x79\x46", [0x492b] = "\x02\x67\x68",
  [0x492c] = "\x02\x67\x69", [0x492e] = "\x02\x6a\x56",
  [0x492f] = "\x02\x6a\x57", [0x4930] = "\x01\x7a\x50",
  [0x4931] = "\x0e\x5d\x4f", [0x4932] = "\x01\x7b\x3f",
  [0x4933] = "\x0e\x5d\x4c", [0x4934] = "\x0e\x5d\x4e",
  [0x4935] = "\x02\x6c\x5c", [0x4936] = "\x0e\x5e\x72",
  [0x4937] = "\x0e\x5e\x71", [0x4938] = "\x01\x7b\x3d",
  [0x4939] = "\x01\x7b\x3e", [0x493a] = "\x02\x6c\x5d",
  [0x493b] = "\x0e\x66\x6e", [0x493c] = "\x0e\x5f\x71",
  [0x493d] = "\x01\x7b\x7e", [0x493e] = "\x01\x7c\x21",
  [0x493f] = "\x02\x6e\x40", [0x4940] = "\x0e\x5f\x72",
  [0x4941] = "\x0e\x60\x59", [0x4942] = "\x01\x7c\x70",
  [0x4943] = "\x02\x70\x5d", [0x4944] = "\x01\x7c\x72",
  [0x4945] = "\x0e\x61\x26", [0x4946] = "\x02\x70\x5c",
  [0x4947] = "\x02\x70\x5e", [0x4948] = "\x01\x7c\x71",
  [0x4949] = "\x02\x71\x46", [0x494b] = "\x02\x71\x7b",
  [0x494d] = "\x0e\x66\x6f", [0x4950] = "\x0e\x62\x45",
  [0x4951] = "\x0e\x2a\x7e", [0x4952] = "\x01\x4f\x65",
  [0x4955] = "\x0e\x45\x49", [0x4956] = "\x01\x67\x5c",
  [0x4958] = "\x02\x4f\x26", [0x4959] = "\x0e\x4a\x4c",
  [0x495a] = "\x02\x55\x59", [0x495b] = "\x01\x72\x7a",
  [0x495c] = "\x01\x72\x7b", [0x495d] = "\x0e\x59\x4f",
  [0x495e] = "\x01\x4f\x66", [0x495f] = "\x0e\x40\x21",
  [0x4960] = "\x01\x6f\x6a", [0x4961] = "\x01\x79\x48",
  [0x4962] = "\x01\x53\x72", [0x4963] = "\x0e\x2b\x21",
  [0x4964] = "\x0e\x4a\x4e", [0x4966] = "\x01\x72\x7c",
  [0x4967] = "\x0e\x5e\x73", [0x4968] = "\x01\x7c\x4e",
  [0x4969] = "\x01\x53\x73", [0x496a] = "\x02\x3a\x31",
  [0x496c] = "\x02\x41\x4e", [0x496d] = "\x0e\x40\x24",
  [0x496e] = "\x02\x41\x50", [0x496f] = "\x0e\x40\x22",
  [0x4970] = "\x02\x41\x4f", [0x4972] = "\x02\x48\x63",
  [0x4973] = "\x02\x48\x60", [0x4974] = "\x01\x67\x5d",
  [0x4976] = "\x01\x67\x5e", [0x4977] = "\x02\x48\x61",
  [0x4978] = "\x02\x48\x62", [0x4979] = "\x0e\x45\x4b",
  [0x497a] = "\x02\x4f\x28", [0x497b] = "\x02\x4f\x2d",
  [0x497c] = "\x01\x6b\x67", [0x497d] = "\x02\x4f\x27",
  [0x497e] = "\x02\x4f\x29", [0x497f] = "\x02\x4f\x30",
  [0x4980] = "\x02\x4f\x2b", [0x4981] = "\x02\x4f\x2f",
  [0x4982] = "\x02\x4f\x2c", [0x4983] = "\x02\x4f\x2a",
  [0x4984] = "\x02\x4f\x2e", [0x4985] = "\x01\x6b\x68",
  [0x4987] = "\x0e\x4f\x62", [0x4988] = "\x02\x55\x5c",
  [0x4989] = "\x0e\x4f\x63", [0x498a] = "\x02\x55\x5a",
  [0x498b] = "\x01\x6f\x6c", [0x498c] = "\x0e\x4f\x64",
  [0x498d] = "\x01\x6f\x6b", [0x498e] = "\x02\x55\x5b",
  [0x498f] = "\x01\x6f\x6d", [0x4993] = "\x0e\x53\x53",
  [0x4994] = "\x02\x5b\x62", [0x4997] = "\x02\x5b\x61",
  [0x4998] = "\x01\x72\x7d", [0x4999] = "\x02\x5b\x60",
  [0x499a] = "\x02\x60\x49", [0x499b] = "\x0e\x56\x7a",
  [0x499c] = "\x02\x60\x4b", [0x499d] = "\x02\x60\x4d",
  [0x499e] = "\x02\x60\x4c", [0x499f] = "\x0e\x56\x7b",
  [0x49a0] = "\x01\x76\x26", [0x49a1] = "\x02\x60\x4a",
  [0x49a2] = "\x02\x64\x4b", [0x49a3] = "\x01\x77\x68",
  [0x49a4] = "\x02\x64\x49", [0x49a5] = "\x02\x64\x4c",
  [0x49a6] = "\x01\x77\x69", [0x49a8] = "\x02\x64\x47",
  [0x49a9] = "\x0e\x59\x50", [0x49aa] = "\x02\x64\x4a",
  [0x49ab] = "\x02\x64\x48", [0x49ac] = "\x02\x64\x45",
  [0x49ad] = "\x01\x77\x6a", [0x49ae] = "\x02\x64\x46",
  [0x49b1] = "\x0e\x5b\x5f", [0x49b2] = "\x0e\x5b\x5d",
  [0x49b3] = "\x02\x67\x6c", [0x49b4] = "\x0e\x5b\x5e",
  [0x49b5] = "\x0e\x5b\x60", [0x49b6] = "\x02\x67\x6e",
  [0x49b7] = "\x02\x67\x6d", [0x49b8] = "\x0e\x5d\x50",
  [0x49b9] = "\x02\x6a\x58", [0x49ba] = "\x0e\x5d\x51",
  [0x49bb] = "\x02\x6a\x59", [0x49bd] = "\x0e\x5e\x75",
  [0x49be] = "\x0e\x5e\x74", [0x49bf] = "\x02\x6c\x5e",
  [0x49c0] = "\x0e\x5f\x73", [0x49c1] = "\x01\x7c\x23",
  [0x49c3] = "\x01\x7c\x22", [0x49c4] = "\x02\x6f\x5d",
  [0x49c5] = "\x02\x6f\x5e", [0x49c6] = "\x01\x7c\x73",
  [0x49c7] = "\x02\x70\x5f", [0x49c8] = "\x0e\x61\x28",
  [0x49c9] = "\x01\x7d\x36", [0x49ca] = "\x0e\x62\x3b",
  [0x49cb] = "\x01\x53\x74", [0x49cc] = "\x01\x62\x79",
  [0x49cd] = "\x02\x4f\x32", [0x49ce] = "\x02\x4f\x31",
  [0x49cf] = "\x02\x55\x5e", [0x49d0] = "\x02\x55\x5d",
  [0x49d2] = "\x0e\x53\x55", [0x49d3] = "\x01\x76\x27",
  [0x49d4] = "\x02\x60\x4f", [0x49d5] = "\x02\x60\x4e",
  [0x49d6] = "\x02\x64\x4f", [0x49d7] = "\x02\x64\x4d",
  [0x49d8] = "\x02\x64\x50", [0x49d9] = "\x02\x64\x4e",
  [0x49da] = "\x0e\x59\x51", [0x49dc] = "\x01\x79\x49",
  [0x49dd] = "\x02\x67\x6f", [0x49de] = "\x02\x67\x70",
  [0x49df] = "\x02\x67\x71", [0x49e1] = "\x02\x6c\x5f",
  [0x49e2] = "\x0e\x5e\x76", [0x49e3] = "\x02\x6e\x41",
  [0x49e4] = "\x0e\x61\x29", [0x49e5] = "\x02\x70\x60",
  [0x49ed] = "\x01\x53\x75", [0x49ee] = "\x0e\x45\x4f",
  [0x49ef] = "\x0e\x4f\x65", [0x49f0] = "\x02\x5b\x63",
  [0x49f1] = "\x02\x60\x50", [0x49f2] = "\x0e\x5b\x61",
  [0x49f3] = "\x01\x53\x76", [0x49f5] = "\x0e\x45\x50",
  [0x49f6] = "\x01\x6b\x69", [0x49f8] = "\x02\x5b\x64",
  [0x49f9] = "\x01\x77\x6b", [0x49fa] = "\x02\x64\x51",
  [0x49fb] = "\x01\x79\x4a", [0x49fc] = "\x0e\x5d\x53",
  [0x49fd] = "\x02\x6a\x5a", [0x49fe] = "\x02\x6a\x5b",
  [0x49ff] = "\x01\x7b\x40", [0x4a00] = "\x02\x6f\x5f",
  [0x4a01] = "\x01\x53\x77", [0x4a02] = "\x01\x5e\x21",
  [0x4a03] = "\x01\x5e\x22", [0x4a04] = "\x02\x3a\x32",
  [0x4a05] = "\x01\x62\x7a", [0x4a06] = "\x01\x62\x7b",
  [0x4a07] = "\x02\x41\x51", [0x4a08] = "\x01\x62\x7c",
  [0x4a0a] = "\x01\x67\x62", [0x4a0b] = "\x0e\x45\x52",
  [0x4a0c] = "\x01\x67\x64", [0x4a0d] = "\x02\x48\x65",
  [0x4a0e] = "\x02\x48\x66", [0x4a0f] = "\x02\x48\x64",
  [0x4a10] = "\x01\x67\x5f", [0x4a11] = "\x01\x67\x60",
  [0x4a12] = "\x01\x67\x63", [0x4a13] = "\x01\x67\x61",
  [0x4a14] = "\x0e\x4a\x54", [0x4a15] = "\x0e\x4a\x55",
  [0x4a16] = "\x02\x4f\x33", [0x4a17] = "\x01\x6b\x6a",
  [0x4a18] = "\x01\x6b\x6b", [0x4a19] = "\x0e\x4a\x52",
  [0x4a1a] = "\x0e\x4a\x56", [0x4a1b] = "\x02\x55\x65",
  [0x4a1c] = "\x01\x6f\x70", [0x4a1d] = "\x02\x55\x60",
  [0x4a1e] = "\x02\x55\x5f", [0x4a1f] = "\x0e\x4f\x69",
  [0x4a20] = "\x02\x55\x64", [0x4a21] = "\x01\x6f\x6e",
  [0x4a22] = "\x0e\x4f\x67", [0x4a23] = "\x0e\x63\x6e",
  [0x4a24] = "\x01\x73\x26", [0x4a26] = "\x02\x55\x61",
  [0x4a27] = "\x02\x55\x66", [0x4a28] = "\x02\x55\x63",
  [0x4a29] = "\x02\x55\x62", [0x4a2a] = "\x0e\x4f\x66",
  [0x4a2b] = "\x01\x6f\x6f", [0x4a2d] = "\x01\x73\x24",
  [0x4a2e] = "\x0e\x53\x5c", [0x4a2f] = "\x02\x5b\x66",
  [0x4a30] = "\x01\x72\x7e", [0x4a31] = "\x0e\x53\x59",
  [0x4a32] = "\x02\x5b\x67", [0x4a33] = "\x0e\x53\x5a",
  [0x4a34] = "\x0e\x52\x28", [0x4a35] = "\x02\x5b\x65",
  [0x4a36] = "\x0e\x53\x58", [0x4a37] = "\x01\x73\x23",
  [0x4a38] = "\x01\x73\x21", [0x4a39] = "\x01\x73\x25",
  [0x4a3a] = "\x0e\x53\x5d", [0x4a3b] = "\x01\x73\x22",
  [0x4a3c] = "\x0e\x53\x5b", [0x4a3d] = "\x0e\x53\x5f",
  [0x4a3f] = "\x0e\x59\x5b", [0x4a40] = "\x0e\x57\x26",
  [0x4a41] = "\x02\x60\x51", [0x4a42] = "\x0e\x63\x7b",
  [0x4a43] = "\x02\x60\x56", [0x4a44] = "\x02\x60\x52",
  [0x4a45] = "\x02\x60\x55", [0x4a46] = "\x01\x76\x28",
  [0x4a47] = "\x0e\x57\x28", [0x4a48] = "\x02\x5e\x41",
  [0x4a49] = "\x02\x60\x54", [0x4a4a] = "\x02\x60\x53",
  [0x4a4b] = "\x0e\x59\x59", [0x4a4c] = "\x01\x77\x6e",
  [0x4a4d] = "\x01\x77\x6c", [0x4a4e] = "\x01\x77\x6f",
  [0x4a4f] = "\x01\x77\x6d", [0x4a50] = "\x02\x64\x52",
  [0x4a51] = "\x02\x64\x53", [0x4a52] = "\x02\x64\x54",
  [0x4a53] = "\x01\x77\x70", [0x4a54] = "\x0e\x59\x56",
  [0x4a55] = "\x0e\x59\x58", [0x4a56] = "\x0e\x5b\x65",
  [0x4a57] = "\x02\x67\x75", [0x4a58] = "\x01\x79\x4c",
  [0x4a59] = "\x02\x67\x73", [0x4a5b] = "\x01\x79\x4d",
  [0x4a5c] = "\x02\x67\x72", [0x4a5d] = "\x02\x67\x74",
  [0x4a5e] = "\x01\x79\x4b", [0x4a5f] = "\x02\x6a\x5f",
  [0x4a60] = "\x02\x6a\x5c", [0x4a61] = "\x0e\x64\x2a",
  [0x4a62] = "\x02\x6a\x5d", [0x4a63] = "\x02\x6a\x5e",
  [0x4a64] = "\x02\x6c\x60", [0x4a65] = "\x01\x7b\x42",
  [0x4a66] = "\x0e\x5e\x78", [0x4a67] = "\x01\x7b\x41",
  [0x4a68] = "\x0e\x5e\x77", [0x4a69] = "\x02\x6e\x43",
  [0x4a6a] = "\x02\x6e\x42", [0x4a6b] = "\x01\x7c\x24",
  [0x4a6c] = "\x0e\x60\x5d", [0x4a6d] = "\x0e\x60\x5e",
  [0x4a6e] = "\x0e\x60\x5c", [0x4a6f] = "\x01\x7c\x4f",
  [0x4a70] = "\x01\x7c\x74", [0x4a71] = "\x01\x7d\x2d",
  [0x4a72] = "\x02\x71\x47", [0x4a73] = "\x02\x71\x7c",
  [0x4a74] = "\x02\x71\x7d", [0x4aa8] = "\x01\x53\x78",
  [0x4aa9] = "\x02\x41\x52", [0x4aac] = "\x02\x48\x67",
  [0x4aad] = "\x02\x4f\x34", [0x4aae] = "\x02\x4f\x35",
  [0x4aaf] = "\x01\x6b\x6c", [0x4ab1] = "\x01\x6b\x6d",
  [0x4ab2] = "\x02\x55\x67", [0x4ab3] = "\x01\x6f\x71",
  [0x4ab5] = "\x0e\x53\x61", [0x4ab6] = "\x01\x76\x29",
  [0x4ab7] = "\x0e\x57\x2a", [0x4ab8] = "\x02\x64\x55",
  [0x4ab9] = "\x0e\x59\x5c", [0x4aba] = "\x01\x77\x71",
  [0x4abb] = "\x02\x67\x78", [0x4abc] = "\x01\x79\x4e",
  [0x4abd] = "\x02\x67\x77", [0x4abe] = "\x02\x67\x79",
  [0x4abf] = "\x02\x67\x76", [0x4ac0] = "\x02\x6c\x63",
  [0x4ac1] = "\x02\x6a\x60", [0x4ac2] = "\x02\x6a\x61",
  [0x4ac3] = "\x0e\x5d\x56", [0x4ac4] = "\x01\x7a\x51",
  [0x4ac6] = "\x02\x6c\x62", [0x4ac7] = "\x0e\x5e\x7b",
  [0x4ac8] = "\x0e\x5e\x79", [0x4ac9] = "\x02\x6c\x61",
  [0x4aca] = "\x0e\x5e\x7a", [0x4acb] = "\x02\x6e\x44",
  [0x4acc] = "\x02\x71\x7e", [0x4acd] = "\x0e\x62\x25",
  [0x4adb] = "\x01\x53\x79", [0x4adc] = "\x0e\x5e\x7c",
  [0x4add] = "\x0e\x62\x26", [0x4adf] = "\x01\x53\x7a",
  [0x4ae1] = "\x0e\x39\x67", [0x4ae2] = "\x01\x58\x6a",
  [0x4ae3] = "\x02\x33\x4b", [0x4ae4] = "\x0e\x66\x70",
  [0x4ae5] = "\x02\x3a\x33", [0x4ae6] = "\x0e\x39\x68",
  [0x4ae7] = "\x01\x62\x7d", [0x4ae9] = "\x01\x63\x22",
  [0x4aea] = "\x01\x62\x7e", [0x4aeb] = "\x02\x41\x53",
  [0x4aec] = "\x0e\x45\x5b", [0x4aed] = "\x01\x63\x24",
  [0x4aef] = "\x01\x63\x21", [0x4af0] = "\x0e\x40\x2b",
  [0x4af1] = "\x0e\x45\x5c", [0x4af2] = "\x01\x63\x23",
  [0x4af3] = "\x0e\x45\x54", [0x4af4] = "\x01\x67\x66",
  [0x4af5] = "\x0e\x45\x59", [0x4af6] = "\x02\x48\x68",
  [0x4af7] = "\x0e\x45\x57", [0x4af8] = "\x0e\x4a\x5d",
  [0x4af9] = "\x02\x48\x69", [0x4afa] = "\x02\x55\x69",
  [0x4afc] = "\x01\x67\x65", [0x4afd] = "\x01\x67\x67",
  [0x4afe] = "\x01\x67\x68", [0x4aff] = "\x0e\x45\x58",
  [0x4b00] = "\x02\x4f\x37", [0x4b01] = "\x0e\x4a\x5e",
  [0x4b02] = "\x02\x4f\x36", [0x4b03] = "\x01\x6b\x6e",
  [0x4b05] = "\x01\x6b\x6f", [0x4b06] = "\x0e\x4a\x5b",
  [0x4b07] = "\x02\x4f\x38", [0x4b08] = "\x02\x55\x68",
  [0x4b09] = "\x01\x6b\x71", [0x4b0a] = "\x01\x6f\x72",
  [0x4b0c] = "\x01\x6b\x70", [0x4b0e] = "\x0e\x66\x7a",
  [0x4b10] = "\x01\x73\x27", [0x4b11] = "\x02\x55\x6a",
  [0x4b12] = "\x01\x6f\x74", [0x4b13] = "\x01\x6f\x73",
  [0x4b14] = "\x02\x55\x6b", [0x4b15] = "\x02\x55\x6e",
  [0x4b16] = "\x02\x55\x6c", [0x4b17] = "\x02\x55\x6d",
  [0x4b18] = "\x01\x6f\x75", [0x4b19] = "\x0e\x4f\x6b",
  [0x4b1a] = "\x01\x73\x2c", [0x4b1b] = "\x01\x73\x2a",
  [0x4b1c] = "\x0e\x53\x65", [0x4b1d] = "\x0e\x53\x66",
  [0x4b1e] = "\x01\x73\x29", [0x4b1f] = "\x02\x5b\x69",
  [0x4b20] = "\x0e\x53\x64", [0x4b21] = "\x01\x73\x2b",
  [0x4b22] = "\x0e\x53\x62", [0x4b23] = "\x0e\x53\x63",
  [0x4b24] = "\x02\x5b\x68", [0x4b25] = "\x02\x60\x57",
  [0x4b27] = "\x02\x5b\x6a", [0x4b28] = "\x01\x73\x28",
  [0x4b29] = "\x02\x5b\x6b", [0x4b2a] = "\x02\x60\x5a",
  [0x4b2b] = "\x02\x60\x58", [0x4b2c] = "\x02\x60\x59",
  [0x4b2d] = "\x02\x60\x5e", [0x4b2e] = "\x01\x77\x75",
  [0x4b2f] = "\x02\x60\x5d", [0x4b30] = "\x02\x60\x60",
  [0x4b31] = "\x02\x60\x5f", [0x4b32] = "\x02\x60\x5c",
  [0x4b33] = "\x02\x60\x5b", [0x4b35] = "\x01\x76\x2a",
  [0x4b36] = "\x0e\x59\x62", [0x4b38] = "\x0e\x59\x61",
  [0x4b39] = "\x0e\x59\x66", [0x4b3a] = "\x02\x64\x58",
  [0x4b3b] = "\x0e\x59\x65", [0x4b3c] = "\x02\x64\x57",
  [0x4b3d] = "\x01\x77\x74", [0x4b3e] = "\x01\x77\x72",
  [0x4b3f] = "\x01\x77\x73", [0x4b40] = "\x0e\x59\x63",
  [0x4b41] = "\x02\x64\x56", [0x4b42] = "\x0e\x66\x71",
  [0x4b43] = "\x02\x67\x7c", [0x4b44] = "\x0e\x5b\x69",
  [0x4b45] = "\x01\x79\x4f", [0x4b47] = "\x02\x67\x7b",
  [0x4b48] = "\x02\x67\x7a", [0x4b49] = "\x01\x79\x50",
  [0x4b4b] = "\x02\x6a\x66", [0x4b4c] = "\x02\x6a\x65",
  [0x4b4d] = "\x0e\x5d\x58", [0x4b4e] = "\x02\x6a\x63",
  [0x4b4f] = "\x0e\x5e\x7e", [0x4b50] = "\x02\x6a\x62",
  [0x4b51] = "\x01\x7a\x53", [0x4b52] = "\x01\x7a\x52",
  [0x4b53] = "\x02\x6a\x67", [0x4b54] = "\x02\x6e\x45",
  [0x4b55] = "\x01\x7c\x25", [0x4b56] = "\x02\x6c\x65",
  [0x4b57] = "\x01\x7b\x43", [0x4b58] = "\x02\x6c\x64",
  [0x4b59] = "\x02\x6a\x64", [0x4b5a] = "\x0e\x5f\x78",
  [0x4b5b] = "\x02\x6e\x46", [0x4b5c] = "\x01\x7c\x50",
  [0x4b5d] = "\x0e\x61\x2a", [0x4b5e] = "\x01\x7d\x2e",
  [0x4b5f] = "\x02\x71\x48", [0x4b61] = "\x02\x72\x21",
  [0x4b96] = "\x01\x53\x7b", [0x4b97] = "\x02\x3a\x34",
  [0x4b98] = "\x02\x60\x61", [0x4b99] = "\x01\x53\x7c",
  [0x4b9a] = "\x0e\x45\x5f", [0x4b9c] = "\x02\x4f\x3a",
  [0x4b9d] = "\x02\x4f\x39", [0x4b9e] = "\x02\x5b\x6c",
  [0x4ba0] = "\x0e\x53\x68", [0x4ba1] = "\x02\x60\x63",
  [0x4ba2] = "\x0e\x57\x30", [0x4ba3] = "\x02\x60\x62",
  [0x4ba4] = "\x0e\x59\x67", [0x4ba5] = "\x01\x77\x76",
  [0x4ba6] = "\x02\x67\x7d", [0x4ba7] = "\x02\x67\x7e",
  [0x4ba8] = "\x01\x7a\x54", [0x4ba9] = "\x0e\x5f\x21",
  [0x4baa] = "\x0e\x60\x60", [0x4bab] = "\x02\x72\x22",
  [0x4bac] = "\x01\x58\x6b", [0x4bad] = "\x01\x63\x26",
  [0x4bae] = "\x01\x63\x25", [0x4baf] = "\x02\x48\x6a",
  [0x4bb0] = "\x02\x48\x6c", [0x4bb1] = "\x01\x67\x6a",
  [0x4bb2] = "\x02\x48\x6b", [0x4bb3] = "\x01\x67\x69",
  [0x4bb4] = "\x01\x67\x6b", [0x4bb5] = "\x02\x48\x6d",
  [0x4bb6] = "\x0e\x4a\x63", [0x4bb7] = "\x0e\x4a\x62",
  [0x4bb8] = "\x0e\x4a\x66", [0x4bb9] = "\x02\x4f\x3c",
  [0x4bba] = "\x02\x4f\x3e", [0x4bbb] = "\x02\x4f\x3d",
  [0x4bbc] = "\x0e\x4a\x61", [0x4bbd] = "\x02\x4f\x40",
  [0x4bbe] = "\x0e\x4a\x60", [0x4bbf] = "\x0e\x4a\x6c",
  [0x4bc0] = "\x0e\x4a\x64", [0x4bc1] = "\x01\x6b\x72",
  [0x4bc2] = "\x02\x4f\x3f", [0x4bc3] = "\x02\x4f\x3b",
  [0x4bc4] = "\x0e\x4a\x6a", [0x4bc7] = "\x02\x4f\x41",
  [0x4bc8] = "\x0e\x4f\x73", [0x4bc9] = "\x02\x55\x75",
  [0x4bca] = "\x0e\x4f\x6d", [0x4bcb] = "\x02\x55\x78",
  [0x4bcc] = "\x02\x55\x7a", [0x4bcd] = "\x02\x55\x70",
  [0x4bce] = "\x02\x55\x74", [0x4bcf] = "\x02\x55\x71",
  [0x4bd0] = "\x01\x6f\x77", [0x4bd1] = "\x01\x6f\x7a",
  [0x4bd2] = "\x01\x6f\x7c", [0x4bd3] = "\x02\x55\x72",
  [0x4bd4] = "\x02\x55\x73", [0x4bd5] = "\x01\x6f\x7b",
  [0x4bd6] = "\x02\x55\x76", [0x4bd7] = "\x02\x55\x79",
  [0x4bd8] = "\x02\x55\x77", [0x4bd9] = "\x01\x6f\x7d",
  [0x4bda] = "\x0e\x4f\x6e", [0x4bdb] = "\x01\x6f\x79",
  [0x4bdc] = "\x02\x55\x6f", [0x4bdd] = "\x01\x6f\x76",
  [0x4bde] = "\x0e\x4f\x72", [0x4bdf] = "\x01\x6f\x78",
  [0x4be0] = "\x0e\x4f\x74", [0x4be1] = "\x0e\x53\x70",
  [0x4be2] = "\x01\x73\x2e", [0x4be3] = "\x02\x5b\x72",
  [0x4be4] = "\x02\x5b\x70", [0x4be5] = "\x02\x5b\x6f",
  [0x4be6] = "\x0e\x53\x71", [0x4be7] = "\x02\x5b\x75",
  [0x4be8] = "\x0e\x53\x6b", [0x4be9] = "\x02\x5b\x74",
  [0x4bea] = "\x02\x5b\x73", [0x4beb] = "\x0e\x53\x6c",
  [0x4bec] = "\x02\x5b\x6e", [0x4bed] = "\x01\x73\x2d",
  [0x4bee] = "\x02\x5b\x6d", [0x4bef] = "\x0e\x53\x6a",
  [0x4bf0] = "\x02\x5b\x71", [0x4bf1] = "\x01\x73\x2f",
  [0x4bf3] = "\x0e\x57\x34", [0x4bf4] = "\x02\x60\x66",
  [0x4bf5] = "\x0e\x57\x37", [0x4bf6] = "\x02\x60\x6a",
  [0x4bf7] = "\x02\x60\x67", [0x4bf8] = "\x02\x60\x69",
  [0x4bf9] = "\x02\x60\x68", [0x4bfa] = "\x02\x60\x65",
  [0x4bfb] = "\x02\x60\x6b", [0x4bfc] = "\x02\x60\x6e",
  [0x4bfd] = "\x02\x60\x6c", [0x4bfe] = "\x02\x60\x6d",
  [0x4bff] = "\x01\x76\x2c", [0x4c00] = "\x0e\x57\x33",
  [0x4c01] = "\x01\x76\x2b", [0x4c02] = "\x02\x60\x64",
  [0x4c03] = "\x02\x60\x6f", [0x4c04] = "\x02\x64\x5d",
  [0x4c05] = "\x02\x64\x60", [0x4c06] = "\x02\x64\x62",
  [0x4c07] = "\x02\x64\x61", [0x4c08] = "\x0e\x59\x6a",
  [0x4c09] = "\x02\x64\x5b", [0x4c0a] = "\x02\x64\x5f",
  [0x4c0b] = "\x02\x64\x5a", [0x4c0c] = "\x0e\x59\x6b",
  [0x4c0d] = "\x02\x64\x5c", [0x4c0e] = "\x01\x77\x77",
  [0x4c0f] = "\x02\x64\x59", [0x4c10] = "\x0e\x59\x6d",
  [0x4c11] = "\x02\x64\x5e", [0x4c14] = "\x02\x68\x2d",
  [0x4c15] = "\x02\x68\x22", [0x4c16] = "\x01\x79\x51",
  [0x4c17] = "\x0e\x5b\x6f", [0x4c18] = "\x0e\x5b\x6d",
  [0x4c19] = "\x01\x79\x52", [0x4c1a] = "\x02\x68\x21",
  [0x4c1b] = "\x02\x68\x26", [0x4c1c] = "\x02\x68\x2c",
  [0x4c1d] = "\x02\x68\x24", [0x4c1e] = "\x02\x68\x2b",
  [0x4c1f] = "\x0e\x5b\x6c", [0x4c20] = "\x02\x68\x28",
  [0x4c21] = "\x0e\x5b\x6e", [0x4c22] = "\x02\x68\x27",
  [0x4c23] = "\x02\x68\x2a", [0x4c24] = "\x02\x68\x25",
  [0x4c25] = "\x02\x68\x23", [0x4c26] = "\x0e\x5b\x6b",
  [0x4c27] = "\x02\x68\x29", [0x4c29] = "\x02\x6a\x6e",
  [0x4c2a] = "\x02\x6a\x6c", [0x4c2b] = "\x01\x7a\x55",
  [0x4c2c] = "\x02\x6a\x6b", [0x4c2d] = "\x02\x6a\x71",
  [0x4c2e] = "\x02\x6a\x6f", [0x4c2f] = "\x0e\x5d\x5a",
  [0x4c30] = "\x01\x7a\x56", [0x4c31] = "\x02\x6a\x6a",
  [0x4c32] = "\x02\x6a\x68", [0x4c34] = "\x02\x6a\x69",
  [0x4c35] = "\x01\x7a\x58", [0x4c36] = "\x02\x6a\x6d",
  [0x4c37] = "\x01\x7a\x57", [0x4c38] = "\x02\x6a\x70",
  [0x4c39] = "\x02\x6c\x66", [0x4c3a] = "\x02\x6c\x6c",
  [0x4c3b] = "\x0e\x5f\x24", [0x4c3c] = "\x0e\x5f\x22",
  [0x4c3d] = "\x02\x6c\x67", [0x4c3e] = "\x01\x7b\x47",
  [0x4c3f] = "\x02\x6c\x6d", [0x4c40] = "\x01\x7b\x46",
  [0x4c41] = "\x02\x6c\x6b", [0x4c42] = "\x02\x6c\x6a",
  [0x4c43] = "\x01\x7b\x45", [0x4c44] = "\x02\x6c\x69",
  [0x4c45] = "\x01\x7b\x44", [0x4c46] = "\x02\x6c\x68",
  [0x4c48] = "\x02\x6e\x4c", [0x4c49] = "\x02\x6e\x4e",
  [0x4c4a] = "\x02\x6e\x4d", [0x4c4c] = "\x02\x6e\x4a",
  [0x4c4d] = "\x01\x7c\x27", [0x4c4e] = "\x02\x6e\x47",
  [0x4c4f] = "\x02\x6e\x4b", [0x4c50] = "\x02\x6e\x50",
  [0x4c51] = "\x0e\x5f\x79", [0x4c52] = "\x02\x6e\x4f",
  [0x4c53] = "\x02\x6e\x48", [0x4c54] = "\x02\x6e\x49",
  [0x4c55] = "\x01\x7c\x26", [0x4c56] = "\x02\x6f\x60",
  [0x4c57] = "\x01\x7c\x53", [0x4c58] = "\x0e\x60\x62",
  [0x4c59] = "\x02\x6f\x61", [0x4c5a] = "\x01\x7c\x51",
  [0x4c5b] = "\x01\x7c\x52", [0x4c5c] = "\x0e\x60\x61",
  [0x4c5e] = "\x02\x70\x61", [0x4c5f] = "\x01\x7c\x75",
  [0x4c60] = "\x02\x71\x65", [0x4c61] = "\x0e\x61\x72",
  [0x4c62] = "\x01\x7d\x37", [0x4c63] = "\x0e\x61\x73",
  [0x4c64] = "\x02\x72\x23", [0x4c65] = "\x01\x7d\x38",
  [0x4c66] = "\x02\x72\x24", [0x4c67] = "\x02\x72\x25",
  [0x4c68] = "\x02\x72\x33", [0x4c69] = "\x02\x72\x32",
  [0x4c6a] = "\x01\x7d\x47", [0x4c6b] = "\x02\x72\x3b",
  [0x4ca8] = "\x01\x58\x6c", [0x4ca9] = "\x0e\x40\x2f",
  [0x4cab] = "\x02\x48\x6f", [0x4cad] = "\x02\x48\x6e",
  [0x4caf] = "\x01\x6b\x73", [0x4cb0] = "\x01\x6b\x74",
  [0x4cb1] = "\x02\x4f\x42", [0x4cb3] = "\x02\x55\x7b",
  [0x4cb4] = "\x02\x5b\x78", [0x4cb5] = "\x0e\x4f\x77",
  [0x4cb6] = "\x0e\x4f\x76", [0x4cb7] = "\x01\x6f\x7e",
  [0x4cb8] = "\x01\x73\x30", [0x4cb9] = "\x02\x5b\x76",
  [0x4cbb] = "\x02\x5b\x79", [0x4cbc] = "\x01\x73\x31",
  [0x4cbd] = "\x0e\x57\x3a", [0x4cbe] = "\x02\x60\x70",
  [0x4cbf] = "\x02\x5b\x77", [0x4cc0] = "\x02\x64\x63",
  [0x4cc1] = "\x01\x77\x78", [0x4cc2] = "\x02\x68\x2e",
  [0x4cc4] = "\x0e\x5d\x61", [0x4cc6] = "\x02\x6a\x74",
  [0x4cc7] = "\x02\x6a\x72", [0x4cc8] = "\x0e\x5d\x60",
  [0x4cc9] = "\x0e\x5d\x5e", [0x4cca] = "\x02\x6a\x73",
  [0x4ccd] = "\x02\x6c\x6e", [0x4cce] = "\x0e\x5f\x25",
  [0x4ccf] = "\x01\x7b\x48", [0x4cd0] = "\x02\x6e\x51",
  [0x4cd1] = "\x01\x7c\x56", [0x4cd2] = "\x01\x7c\x28",
  [0x4cd3] = "\x01\x7c\x54", [0x4cd4] = "\x01\x7c\x55",
  [0x4cd5] = "\x02\x70\x62", [0x4cd6] = "\x01\x7d\x2f",
  [0x4cd7] = "\x0e\x61\x74", [0x4cd8] = "\x01\x58\x6d",
  [0x4cd9] = "\x0e\x63\x34", [0x4cda] = "\x0e\x4a\x6e",
  [0x4cdc] = "\x02\x64\x64", [0x4cdd] = "\x0e\x5f\x7b",
  [0x4cde] = "\x0e\x60\x63", [0x4cdf] = "\x02\x33\x4c",
  [0x4ce0] = "\x0e\x40\x32", [0x4ce1] = "\x01\x67\x6c",
  [0x4ce2] = "\x0e\x45\x61", [0x4ce3] = "\x02\x4f\x43",
  [0x4ce4] = "\x0e\x4a\x6f", [0x4ce5] = "\x0e\x4a\x73",
  [0x4ce6] = "\x01\x6b\x75", [0x4ce7] = "\x02\x4f\x44",
  [0x4ce8] = "\x0e\x4a\x71", [0x4ce9] = "\x0e\x4a\x70",
  [0x4cea] = "\x0e\x4a\x72", [0x4ceb] = "\x02\x55\x7d",
  [0x4cec] = "\x02\x55\x7c", [0x4ced] = "\x01\x73\x33",
  [0x4cee] = "\x01\x70\x21", [0x4cef] = "\x01\x70\x22",
  [0x4cf1] = "\x02\x56\x22", [0x4cf2] = "\x02\x56\x21",
  [0x4cf3] = "\x02\x55\x7e", [0x4cf4] = "\x0e\x4f\x79",
  [0x4cf5] = "\x0e\x53\x75", [0x4cf6] = "\x02\x5b\x7a",
  [0x4cf7] = "\x02\x5b\x7d", [0x4cf8] = "\x0e\x53\x74",
  [0x4cf9] = "\x02\x5b\x7c", [0x4cfa] = "\x02\x5b\x7b",
  [0x4cfb] = "\x01\x73\x32", [0x4cfc] = "\x02\x60\x74",
  [0x4cfd] = "\x02\x60\x72", [0x4cfe] = "\x02\x60\x71",
  [0x4d00] = "\x0e\x57\x3b", [0x4d01] = "\x02\x60\x73",
  [0x4d02] = "\x0e\x57\x3c", [0x4d03] = "\x01\x77\x79",
  [0x4d04] = "\x02\x64\x66", [0x4d05] = "\x02\x64\x67",
  [0x4d06] = "\x01\x77\x7a", [0x4d07] = "\x0e\x59\x6f",
  [0x4d08] = "\x02\x64\x65", [0x4d09] = "\x0e\x5b\x71",
  [0x4d0a] = "\x02\x68\x30", [0x4d0b] = "\x02\x68\x2f",
  [0x4d0c] = "\x02\x68\x32", [0x4d0d] = "\x01\x79\x53",
  [0x4d0e] = "\x02\x68\x31", [0x4d10] = "\x02\x6a\x75",
  [0x4d11] = "\x02\x6a\x77", [0x4d12] = "\x02\x6a\x76",
  [0x4d14] = "\x0e\x5f\x27", [0x4d15] = "\x02\x6c\x6f",
  [0x4d16] = "\x02\x6c\x72", [0x4d17] = "\x02\x6c\x70",
  [0x4d18] = "\x02\x6c\x71", [0x4d19] = "\x02\x6e\x52",
  [0x4d1a] = "\x01\x7c\x29", [0x4d1d] = "\x0e\x66\x72",
  [0x4d1e] = "\x02\x6f\x62", [0x4d1f] = "\x02\x6f\x63",
  [0x4d20] = "\x02\x6f\x64", [0x4d21] = "\x0e\x61\x2c",
  [0x4d22] = "\x01\x7c\x76", [0x4d23] = "\x01\x7d\x30",
  [0x4d24] = "\x02\x72\x26", [0x4d25] = "\x01\x58\x6e",
  [0x4d26] = "\x0e\x4a\x75", [0x4d27] = "\x01\x70\x23",
  [0x4d28] = "\x01\x73\x34", [0x4d29] = "\x02\x64\x68",
  [0x4d2a] = "\x0e\x5d\x62", [0x4d2b] = "\x02\x6e\x53",
  [0x4d2c] = "\x0e\x61\x2e", [0x4d2d] = "\x0e\x64\x33",
  [0x4d2e] = "\x02\x71\x66", [0x4d2f] = "\x02\x33\x4d",
  [0x4d30] = "\x0e\x62\x28", [0x4d31] = "\x01\x7d\x48",
  [0x4d32] = "\x01\x58\x6f", [0x4d33] = "\x02\x5b\x7e",
  [0x4d34] = "\x0e\x57\x3e", [0x4d35] = "\x02\x64\x69",
  [0x4d37] = "\x02\x68\x33", [0x4d38] = "\x0e\x5d\x63",
  [0x4d3a] = "\x02\x6c\x73", [0x4d3b] = "\x02\x6e\x54",
  [0x4d3c] = "\x01\x58\x70", [0x4d3d] = "\x0e\x45\x62",
  [0x4d3e] = "\x02\x4f\x45", [0x4d3f] = "\x02\x4f\x46",
  [0x4d40] = "\x0e\x4a\x78", [0x4d41] = "\x01\x6b\x76",
  [0x4d42] = "\x01\x6b\x77", [0x4d43] = "\x02\x56\x24",
  [0x4d44] = "\x01\x70\x25", [0x4d45] = "\x01\x70\x24",
  [0x4d46] = "\x02\x56\x23", [0x4d48] = "\x02\x60\x75",
  [0x4d49] = "\x0e\x57\x3f", [0x4d4a] = "\x02\x64\x6a",
  [0x4d4b] = "\x02\x64\x6c", [0x4d4c] = "\x02\x64\x6b",
  [0x4d4d] = "\x01\x77\x7d", [0x4d4e] = "\x01\x77\x7c",
  [0x4d4f] = "\x01\x77\x7b", [0x4d50] = "\x0e\x5d\x64",
  [0x4d51] = "\x01\x7b\x4a", [0x4d52] = "\x02\x6c\x74",
  [0x4d53] = "\x0e\x5f\x28", [0x4d54] = "\x01\x7b\x49",
  [0x4d55] = "\x02\x6e\x56", [0x4d56] = "\x02\x6e\x55",
  [0x4d57] = "\x0e\x61\x2f", [0x4d58] = "\x01\x7c\x77",
  [0x4d59] = "\x02\x70\x63", [0x4d5a] = "\x01\x5e\x23",
  [0x4d5b] = "\x02\x48\x70", [0x4d5d] = "\x0e\x45\x63",
  [0x4d5f] = "\x02\x4f\x49", [0x4d60] = "\x02\x4f\x47",
  [0x4d61] = "\x02\x4f\x48", [0x4d63] = "\x0e\x4f\x7e",
  [0x4d64] = "\x02\x56\x2d", [0x4d66] = "\x02\x56\x28",
  [0x4d67] = "\x02\x56\x25", [0x4d68] = "\x02\x56\x2c",
  [0x4d69] = "\x0e\x4f\x7c", [0x4d6b] = "\x0e\x4f\x7b",
  [0x4d6c] = "\x02\x56\x2e", [0x4d6f] = "\x01\x70\x27",
  [0x4d70] = "\x02\x56\x2b", [0x4d71] = "\x02\x56\x27",
  [0x4d72] = "\x0e\x4f\x7d", [0x4d74] = "\x02\x56\x26",
  [0x4d75] = "\x02\x56\x2a", [0x4d76] = "\x02\x56\x29",
  [0x4d77] = "\x01\x70\x26", [0x4d7a] = "\x02\x5c\x2b",
  [0x4d7b] = "\x02\x5c\x26", [0x4d7c] = "\x02\x5c\x24",
  [0x4d7d] = "\x02\x5c\x2d", [0x4d7e] = "\x02\x5c\x25",
  [0x4d80] = "\x02\x5c\x21", [0x4d82] = "\x02\x5c\x27",
  [0x4d83] = "\x0e\x53\x78", [0x4d84] = "\x0e\x53\x7a",
  [0x4d85] = "\x02\x5c\x22", [0x4d86] = "\x02\x60\x7e",
  [0x4d87] = "\x02\x5c\x23", [0x4d88] = "\x02\x5c\x2e",
  [0x4d89] = "\x0e\x54\x23", [0x4d8b] = "\x0e\x53\x7c",
  [0x4d8c] = "\x0e\x54\x21", [0x4d8e] = "\x0e\x54\x25",
  [0x4d8f] = "\x0e\x53\x7d", [0x4d90] = "\x02\x5c\x2a",
  [0x4d91] = "\x01\x73\x35", [0x4d92] = "\x02\x5c\x29",
  [0x4d93] = "\x02\x5c\x28", [0x4d94] = "\x0e\x53\x79",
  [0x4d95] = "\x02\x5c\x2c", [0x4d98] = "\x0e\x66\x73",
  [0x4d99] = "\x0e\x57\x44", [0x4d9a] = "\x02\x60\x76",
  [0x4d9b] = "\x02\x60\x79", [0x4d9d] = "\x0e\x57\x45",
  [0x4d9e] = "\x02\x60\x78", [0x4d9f] = "\x0e\x57\x40",
  [0x4da0] = "\x02\x61\x22", [0x4da1] = "\x02\x60\x7b",
  [0x4da2] = "\x02\x61\x21", [0x4da3] = "\x0e\x57\x42",
  [0x4da4] = "\x02\x60\x7d", [0x4da5] = "\x02\x60\x7c",
  [0x4da6] = "\x02\x60\x7a", [0x4da8] = "\x02\x60\x77",
  [0x4daa] = "\x01\x76\x2f", [0x4dab] = "\x01\x76\x2e",
  [0x4dad] = "\x01\x76\x30", [0x4dae] = "\x01\x76\x2d",
  [0x4daf] = "\x02\x61\x23", [0x4db5] = "\x02\x64\x72",
  [0x4db6] = "\x02\x64\x75", [0x4db8] = "\x02\x64\x73",
  [0x4db9] = "\x02\x64\x77", [0x4dbb] = "\x0e\x59\x74",
  [0x4dbd] = "\x02\x64\x78", [0x4dbf] = "\x02\x64\x70",
  [0x4dc0] = "\x01\x78\x24", [0x4dc1] = "\x02\x64\x71",
  [0x4dc2] = "\x0e\x59\x73", [0x4dc3] = "\x02\x64\x6f",
  [0x4dc4] = "\x02\x64\x76", [0x4dc5] = "\x0e\x5b\x74",
  [0x4dc6] = "\x02\x64\x6e", [0x4dc7] = "\x02\x64\x6d",
  [0x4dc8] = "\x01\x78\x23", [0x4dc9] = "\x01\x78\x21",
  [0x4dca] = "\x01\x77\x7e", [0x4dcb] = "\x0e\x59\x76",
  [0x4dcc] = "\x0e\x59\x75", [0x4dcd] = "\x0e\x57\x43",
  [0x4dd2] = "\x0e\x59\x71", [0x4dd3] = "\x02\x64\x74",
  [0x4dd4] = "\x02\x68\x3c", [0x4dd5] = "\x02\x68\x42",
  [0x4dd6] = "\x01\x79\x56", [0x4dd7] = "\x02\x68\x3d",
  [0x4dd9] = "\x02\x68\x40", [0x4dda] = "\x02\x68\x44",
  [0x4ddb] = "\x01\x79\x57", [0x4ddc] = "\x02\x68\x3f",
  [0x4dde] = "\x02\x68\x37", [0x4ddf] = "\x0e\x5b\x75",
  [0x4de0] = "\x02\x68\x36", [0x4de1] = "\x02\x68\x43",
  [0x4de2] = "\x02\x68\x3a", [0x4de3] = "\x0e\x5b\x77",
  [0x4de4] = "\x02\x68\x38", [0x4de5] = "\x02\x68\x41",
  [0x4de6] = "\x02\x68\x39", [0x4de7] = "\x01\x79\x55",
  [0x4de8] = "\x01\x79\x54", [0x4de9] = "\x0e\x5b\x79",
  [0x4dea] = "\x02\x68\x34", [0x4deb] = "\x02\x68\x35",
  [0x4dec] = "\x02\x68\x3e", [0x4dee] = "\x0e\x5b\x7a",
  [0x4df0] = "\x02\x68\x3b", [0x4df6] = "\x0e\x5d\x6f",
  [0x4df7] = "\x02\x6a\x7a", [0x4df8] = "\x02\x6a\x7d",
  [0x4df9] = "\x0e\x5d\x71", [0x4dfb] = "\x0e\x5d\x67",
  [0x4dfc] = "\x0e\x5d\x69", [0x4dfd] = "\x01\x78\x22",
  [0x4dfe] = "\x0e\x5d\x6a", [0x4dff] = "\x0e\x5d\x74",
  [0x4e00] = "\x0e\x5d\x75", [0x4e02] = "\x0e\x5d\x6e",
  [0x4e04] = "\x0e\x5d\x68", [0x4e05] = "\x02\x6a\x7b",
  [0x4e06] = "\x02\x6b\x23", [0x4e07] = "\x02\x6b\x21",
  [0x4e08] = "\x02\x6a\x79", [0x4e09] = "\x02\x6b\x26",
  [0x4e0a] = "\x0e\x5d\x66", [0x4e0b] = "\x02\x6a\x78",
  [0x4e0c] = "\x0e\x5d\x76", [0x4e0d] = "\x01\x7a\x5a",
  [0x4e0e] = "\x02\x6b\x22", [0x4e10] = "\x0e\x5d\x73",
  [0x4e12] = "\x02\x6a\x7c", [0x4e13] = "\x01\x7a\x59",
  [0x4e14] = "\x02\x6b\x25", [0x4e15] = "\x0e\x5d\x72",
  [0x4e17] = "\x02\x6b\x24", [0x4e1b] = "\x0e\x5d\x70",
  [0x4e1c] = "\x02\x6c\x77", [0x4e1d] = "\x02\x6c\x76",
  [0x4e1f] = "\x0e\x5f\x2c", [0x4e21] = "\x02\x6c\x7d",
  [0x4e23] = "\x02\x6c\x79", [0x4e24] = "\x02\x6c\x7c",
  [0x4e25] = "\x01\x7b\x4c", [0x4e27] = "\x0e\x64\x2f",
  [0x4e28] = "\x02\x6c\x7a", [0x4e29] = "\x02\x6c\x7b",
  [0x4e2b] = "\x02\x6c\x75", [0x4e2c] = "\x02\x6c\x78",
  [0x4e2d] = "\x01\x7b\x4b", [0x4e2e] = "\x0e\x5f\x2a",
  [0x4e31] = "\x01\x7c\x2b", [0x4e32] = "\x02\x6e\x61",
  [0x4e33] = "\x02\x6e\x5c", [0x4e34] = "\x02\x6e\x60",
  [0x4e35] = "\x0e\x60\x25", [0x4e36] = "\x02\x6e\x63",
  [0x4e37] = "\x02\x6e\x5f", [0x4e38] = "\x0e\x5f\x7e",
  [0x4e39] = "\x02\x6e\x5b", [0x4e3a] = "\x0e\x60\x23",
  [0x4e3b] = "\x01\x7c\x2d", [0x4e3c] = "\x02\x6e\x5e",
  [0x4e3d] = "\x02\x6e\x62", [0x4e3e] = "\x01\x7c\x2c",
  [0x4e3f] = "\x02\x6e\x59", [0x4e40] = "\x02\x6a\x7e",
  [0x4e41] = "\x02\x6e\x5d", [0x4e44] = "\x02\x6e\x5a",
  [0x4e45] = "\x0e\x60\x22", [0x4e46] = "\x02\x6e\x57",
  [0x4e48] = "\x02\x6e\x58", [0x4e49] = "\x01\x7c\x2a",
  [0x4e4a] = "\x02\x6f\x68", [0x4e4b] = "\x02\x6f\x6a",
  [0x4e4c] = "\x02\x6f\x6d", [0x4e4d] = "\x02\x6f\x69",
  [0x4e4e] = "\x02\x6f\x6e", [0x4e4f] = "\x0e\x60\x65",
  [0x4e50] = "\x02\x6f\x67", [0x4e51] = "\x0e\x60\x66",
  [0x4e52] = "\x02\x6f\x65", [0x4e53] = "\x0e\x60\x68",
  [0x4e54] = "\x01\x7c\x57", [0x4e55] = "\x02\x6f\x6b",
  [0x4e56] = "\x01\x7c\x59", [0x4e57] = "\x01\x7c\x58",
  [0x4e58] = "\x02\x6f\x66", [0x4e59] = "\x02\x6f\x6c",
  [0x4e5d] = "\x0e\x61\x32", [0x4e5e] = "\x02\x70\x68",
  [0x4e5f] = "\x01\x7c\x78", [0x4e60] = "\x02\x70\x69",
  [0x4e61] = "\x0e\x61\x33", [0x4e62] = "\x02\x70\x67",
  [0x4e63] = "\x02\x70\x64", [0x4e65] = "\x0e\x61\x34",
  [0x4e66] = "\x02\x70\x66", [0x4e67] = "\x02\x70\x65",
  [0x4e68] = "\x02\x71\x49", [0x4e6c] = "\x0e\x61\x59",
  [0x4e6d] = "\x02\x71\x4b", [0x4e6e] = "\x02\x71\x4a",
  [0x4e6f] = "\x0e\x61\x5b", [0x4e71] = "\x02\x71\x69",
  [0x4e73] = "\x02\x71\x68", [0x4e74] = "\x02\x71\x67",
  [0x4e75] = "\x02\x71\x6a", [0x4e76] = "\x0e\x61\x75",
  [0x4e77] = "\x01\x7d\x40", [0x4e78] = "\x01\x7d\x41",
  [0x4e79] = "\x02\x72\x38", [0x4e7a] = "\x02\x72\x3c",
  [0x4e7b] = "\x0e\x62\x41", [0x4ee5] = "\x01\x5e\x24",
  [0x4ee6] = "\x02\x41\x54", [0x4ee7] = "\x02\x48\x73",
  [0x4ee8] = "\x0e\x45\x64", [0x4ee9] = "\x01\x67\x6d",
  [0x4eea] = "\x02\x48\x71", [0x4eeb] = "\x0e\x45\x66",
  [0x4eec] = "\x0e\x2f\x42", [0x4eed] = "\x02\x48\x72",
  [0x4eee] = "\x0e\x45\x68", [0x4eef] = "\x0e\x45\x67",
  [0x4ef1] = "\x02\x4f\x4a", [0x4ef2] = "\x02\x4f\x4b",
  [0x4ef3] = "\x01\x6b\x7a", [0x4ef4] = "\x01\x6b\x78",
  [0x4ef5] = "\x02\x4f\x4c", [0x4ef6] = "\x01\x6b\x79",
  [0x4ef7] = "\x02\x56\x33", [0x4ef8] = "\x0e\x50\x27",
  [0x4ef9] = "\x02\x56\x36", [0x4efa] = "\x02\x56\x30",
  [0x4efb] = "\x02\x56\x37", [0x4efc] = "\x02\x56\x2f",
  [0x4efd] = "\x02\x56\x31", [0x4efe] = "\x0e\x50\x2a",
  [0x4eff] = "\x02\x56\x32", [0x4f00] = "\x02\x56\x35",
  [0x4f01] = "\x0e\x50\x24", [0x4f02] = "\x0e\x50\x2b",
  [0x4f03] = "\x01\x70\x2a", [0x4f04] = "\x02\x56\x3a",
  [0x4f05] = "\x02\x56\x39", [0x4f06] = "\x01\x70\x28",
  [0x4f07] = "\x02\x56\x34", [0x4f08] = "\x02\x56\x38",
  [0x4f09] = "\x01\x70\x29", [0x4f0a] = "\x0e\x54\x2d",
  [0x4f0c] = "\x0e\x50\x25", [0x4f0d] = "\x0e\x50\x22",
  [0x4f0f] = "\x0e\x54\x2a", [0x4f10] = "\x02\x5c\x38",
  [0x4f12] = "\x01\x73\x3a", [0x4f13] = "\x0e\x54\x2c",
  [0x4f14] = "\x02\x5c\x33", [0x4f15] = "\x01\x73\x36",
  [0x4f16] = "\x0e\x54\x36", [0x4f17] = "\x02\x5c\x30",
  [0x4f18] = "\x02\x5c\x36", [0x4f19] = "\x02\x5c\x39",
  [0x4f1a] = "\x0e\x54\x33", [0x4f1b] = "\x01\x73\x3b",
  [0x4f1d] = "\x02\x5c\x35", [0x4f1e] = "\x02\x5c\x32",
  [0x4f1f] = "\x02\x5c\x3a", [0x4f20] = "\x02\x5c\x31",
  [0x4f21] = "\x0e\x54\x37", [0x4f22] = "\x02\x5c\x37",
  [0x4f23] = "\x01\x73\x37", [0x4f24] = "\x0e\x54\x29",
  [0x4f25] = "\x02\x5c\x2f", [0x4f26] = "\x01\x73\x38",
  [0x4f27] = "\x0e\x54\x35", [0x4f28] = "\x01\x73\x39",
  [0x4f29] = "\x02\x5c\x34", [0x4f2a] = "\x0e\x54\x32",
  [0x4f2d] = "\x02\x61\x35", [0x4f2e] = "\x02\x61\x28",
  [0x4f2f] = "\x02\x61\x29", [0x4f30] = "\x02\x61\x2c",
  [0x4f31] = "\x02\x61\x2a", [0x4f33] = "\x02\x61\x24",
  [0x4f35] = "\x0e\x57\x4d", [0x4f36] = "\x02\x61\x27",
  [0x4f37] = "\x02\x61\x31", [0x4f38] = "\x02\x61\x2b",
  [0x4f39] = "\x0e\x57\x48", [0x4f3b] = "\x01\x76\x31",
  [0x4f3d] = "\x02\x61\x33", [0x4f3e] = "\x02\x61\x30",
  [0x4f3f] = "\x01\x76\x32", [0x4f40] = "\x02\x61\x32",
  [0x4f41] = "\x02\x61\x25", [0x4f42] = "\x02\x61\x2e",
  [0x4f43] = "\x02\x61\x2f", [0x4f44] = "\x0e\x57\x4a",
  [0x4f45] = "\x02\x61\x2d", [0x4f49] = "\x0e\x66\x75",
  [0x4f4a] = "\x02\x64\x7c", [0x4f4b] = "\x02\x64\x7e",
  [0x4f4c] = "\x02\x65\x23", [0x4f4d] = "\x0e\x59\x79",
  [0x4f4e] = "\x0e\x59\x7b", [0x4f4f] = "\x02\x64\x7b",
  [0x4f50] = "\x0e\x59\x7d", [0x4f51] = "\x01\x78\x25",
  [0x4f52] = "\x02\x65\x25", [0x4f53] = "\x02\x64\x7a",
  [0x4f54] = "\x02\x65\x26", [0x4f55] = "\x0e\x59\x7e",
  [0x4f56] = "\x02\x65\x22", [0x4f57] = "\x02\x65\x24",
  [0x4f58] = "\x02\x65\x28", [0x4f59] = "\x02\x65\x21",
  [0x4f5a] = "\x02\x65\x29", [0x4f5b] = "\x02\x64\x7d",
  [0x4f5c] = "\x02\x64\x79", [0x4f5d] = "\x01\x78\x26",
  [0x4f5e] = "\x0e\x5a\x22", [0x4f5f] = "\x02\x65\x27",
  [0x4f60] = "\x01\x78\x27", [0x4f61] = "\x01\x79\x59",
  [0x4f63] = "\x0e\x59\x7a", [0x4f66] = "\x0e\x5b\x7d",
  [0x4f67] = "\x02\x61\x26", [0x4f68] = "\x02\x68\x5e",
  [0x4f69] = "\x02\x68\x55", [0x4f6a] = "\x01\x79\x5b",
  [0x4f6b] = "\x02\x68\x51", [0x4f6c] = "\x01\x79\x5c",
  [0x4f6d] = "\x0e\x5c\x27", [0x4f6e] = "\x0e\x5c\x22",
  [0x4f6f] = "\x02\x68\x5a", [0x4f70] = "\x02\x68\x54",
  [0x4f71] = "\x02\x68\x4a", [0x4f72] = "\x01\x79\x5a",
  [0x4f73] = "\x02\x68\x57", [0x4f74] = "\x02\x68\x52",
  [0x4f75] = "\x02\x68\x53", [0x4f76] = "\x0e\x5c\x25",
  [0x4f77] = "\x02\x68\x45", [0x4f78] = "\x02\x68\x4c",
  [0x4f79] = "\x02\x68\x5b", [0x4f7a] = "\x0e\x5b\x7e",
  [0x4f7b] = "\x02\x68\x58", [0x4f7d] = "\x02\x68\x50",
  [0x4f7e] = "\x0e\x5c\x26", [0x4f7f] = "\x02\x68\x5c",
  [0x4f80] = "\x02\x68\x4b", [0x4f81] = "\x02\x68\x46",
  [0x4f82] = "\x02\x68\x59", [0x4f83] = "\x0e\x5c\x24",
  [0x4f84] = "\x02\x68\x48", [0x4f85] = "\x02\x68\x56",
  [0x4f86] = "\x02\x68\x4d", [0x4f87] = "\x02\x68\x5d",
  [0x4f88] = "\x02\x68\x49", [0x4f89] = "\x01\x79\x58",
  [0x4f8a] = "\x02\x68\x47", [0x4f8b] = "\x02\x68\x4e",
  [0x4f8c] = "\x02\x68\x4f", [0x4f90] = "\x02\x6b\x2d",
  [0x4f91] = "\x0e\x5c\x23", [0x4f92] = "\x02\x6b\x2b",
  [0x4f94] = "\x02\x6b\x30", [0x4f95] = "\x0e\x5d\x79",
  [0x4f96] = "\x02\x6b\x3c", [0x4f97] = "\x02\x6b\x33",
  [0x4f98] = "\x02\x6b\x2c", [0x4f99] = "\x02\x6b\x28",
  [0x4f9a] = "\x02\x6b\x35", [0x4f9b] = "\x02\x6b\x2e",
  [0x4f9c] = "\x02\x6b\x31", [0x4f9d] = "\x02\x6b\x2a",
  [0x4f9e] = "\x02\x6b\x38", [0x4f9f] = "\x02\x6b\x27",
  [0x4fa0] = "\x02\x6b\x2f", [0x4fa1] = "\x02\x6b\x34",
  [0x4fa2] = "\x02\x6b\x36", [0x4fa3] = "\x02\x6b\x39",
  [0x4fa4] = "\x02\x6b\x29", [0x4fa5] = "\x0e\x5d\x7a",
  [0x4fa6] = "\x02\x6b\x3d", [0x4fa7] = "\x02\x6b\x3e",
  [0x4fa8] = "\x02\x6b\x37", [0x4fa9] = "\x02\x6b\x3b",
  [0x4faa] = "\x02\x6b\x32", [0x4fac] = "\x02\x6d\x2f",
  [0x4fad] = "\x02\x6d\x32", [0x4fae] = "\x0e\x5f\x35",
  [0x4faf] = "\x01\x7b\x4d", [0x4fb0] = "\x0e\x5f\x31",
  [0x4fb1] = "\x02\x6d\x31", [0x4fb2] = "\x02\x6d\x36",
  [0x4fb3] = "\x02\x6d\x34", [0x4fb4] = "\x01\x7b\x4e",
  [0x4fb5] = "\x02\x6d\x2b", [0x4fb6] = "\x02\x6d\x21",
  [0x4fb7] = "\x02\x6c\x7e", [0x4fb8] = "\x01\x7b\x50",
  [0x4fb9] = "\x02\x6d\x2d", [0x4fba] = "\x02\x6d\x2e",
  [0x4fbb] = "\x02\x6d\x2a", [0x4fbc] = "\x02\x6d\x22",
  [0x4fbd] = "\x0e\x5f\x32", [0x4fbe] = "\x02\x6d\x27",
  [0x4fbf] = "\x02\x6b\x3a", [0x4fc1] = "\x02\x6d\x23",
  [0x4fc2] = "\x01\x7b\x4f", [0x4fc3] = "\x02\x6d\x29",
  [0x4fc4] = "\x0e\x5f\x36", [0x4fc5] = "\x02\x6d\x28",
  [0x4fc7] = "\x02\x6d\x24", [0x4fc8] = "\x02\x6d\x30",
  [0x4fca] = "\x02\x6d\x25", [0x4fcb] = "\x02\x6e\x68",
  [0x4fcc] = "\x02\x6d\x33", [0x4fcd] = "\x02\x6d\x35",
  [0x4fce] = "\x02\x6d\x2c", [0x4fcf] = "\x02\x6d\x26",
  [0x4fd0] = "\x02\x6e\x69", [0x4fd1] = "\x02\x6e\x6b",
  [0x4fd2] = "\x02\x6e\x65", [0x4fd3] = "\x01\x7c\x2e",
  [0x4fd5] = "\x02\x6e\x72", [0x4fd6] = "\x02\x6e\x70",
  [0x4fd7] = "\x01\x7c\x2f", [0x4fd8] = "\x02\x6e\x6f",
  [0x4fd9] = "\x02\x6e\x6e", [0x4fda] = "\x02\x6e\x67",
  [0x4fdb] = "\x02\x6e\x64", [0x4fdc] = "\x02\x6e\x6a",
  [0x4fdd] = "\x02\x6e\x73", [0x4fde] = "\x02\x6e\x66",
  [0x4fdf] = "\x02\x6e\x6c", [0x4fe1] = "\x02\x6f\x77",
  [0x4fe2] = "\x02\x6f\x7c", [0x4fe3] = "\x02\x6f\x72",
  [0x4fe4] = "\x02\x6f\x75", [0x4fe5] = "\x01\x7c\x5a",
  [0x4fe6] = "\x02\x6f\x79", [0x4fe8] = "\x02\x70\x22",
  [0x4fe9] = "\x02\x6e\x6d", [0x4feb] = "\x02\x6f\x73",
  [0x4fec] = "\x02\x6f\x7d", [0x4fed] = "\x02\x70\x23",
  [0x4fee] = "\x02\x6f\x78", [0x4fef] = "\x02\x6f\x71",
  [0x4ff0] = "\x02\x6f\x7b", [0x4ff2] = "\x02\x6f\x7a",
  [0x4ff3] = "\x02\x70\x21", [0x4ff4] = "\x02\x6f\x7e",
  [0x4ff5] = "\x02\x6e\x71", [0x4ff6] = "\x02\x6f\x76",
  [0x4ff7] = "\x02\x6f\x70", [0x4ff8] = "\x02\x6f\x74",
  [0x4ff9] = "\x01\x7c\x79", [0x4ffa] = "\x01\x7c\x7a",
  [0x4ffb] = "\x02\x6f\x6f", [0x4ffc] = "\x0e\x60\x6c",
  [0x4ffd] = "\x02\x70\x74", [0x4ffe] = "\x02\x70\x6b",
  [0x4fff] = "\x02\x70\x73", [0x5000] = "\x02\x70\x70",
  [0x5001] = "\x02\x70\x71", [0x5002] = "\x02\x70\x6a",
  [0x5003] = "\x02\x70\x6d", [0x5004] = "\x02\x70\x75",
  [0x5005] = "\x02\x70\x6f", [0x5006] = "\x02\x70\x6e",
  [0x5007] = "\x02\x70\x6c", [0x5008] = "\x0e\x61\x35",
  [0x5009] = "\x02\x70\x72", [0x500b] = "\x02\x71\x4c",
  [0x500d] = "\x02\x71\x4d", [0x500e] = "\x0e\x61\x5d",
  [0x500f] = "\x02\x71\x4f", [0x5010] = "\x02\x71\x4e",
  [0x5011] = "\x02\x71\x51", [0x5012] = "\x02\x71\x50",
  [0x5013] = "\x02\x71\x6c", [0x5014] = "\x02\x71\x6b",
  [0x5015] = "\x02\x72\x27", [0x5016] = "\x0e\x62\x29",
  [0x5017] = "\x02\x72\x28", [0x5018] = "\x0e\x62\x30",
  [0x5019] = "\x02\x72\x34", [0x501a] = "\x01\x7d\x45",
  [0x501b] = "\x01\x7d\x49", [0x501c] = "\x0e\x62\x38",
  [0x501d] = "\x02\x72\x3d", [0x501e] = "\x01\x7d\x4a",
  [0x5075] = "\x01\x5e\x25", [0x5079] = "\x01\x7a\x5b",
  [0x507a] = "\x02\x6d\x37", [0x507b] = "\x0e\x5f\x37",
  [0x507c] = "\x01\x7c\x7b", [0x507d] = "\x01\x7c\x7c",
  [0x507f] = "\x01\x5e\x26", [0x5080] = "\x02\x48\x74",
  [0x5081] = "\x0e\x45\x69", [0x5082] = "\x01\x67\x6e",
  [0x5083] = "\x02\x56\x3b", [0x5084] = "\x0e\x50\x2c",
  [0x5085] = "\x0e\x54\x39", [0x5086] = "\x02\x5c\x3c",
  [0x5087] = "\x02\x5c\x3d", [0x5088] = "\x02\x5c\x3b",
  [0x5089] = "\x02\x61\x37", [0x508a] = "\x02\x61\x36",
  [0x508b] = "\x01\x76\x33", [0x508c] = "\x02\x65\x2b",
  [0x508d] = "\x02\x61\x38", [0x508e] = "\x02\x65\x2a",
  [0x5090] = "\x0e\x5a\x24", [0x5091] = "\x02\x68\x60",
  [0x5092] = "\x01\x79\x5d", [0x5093] = "\x01\x79\x5f",
  [0x5094] = "\x02\x68\x5f", [0x5095] = "\x0e\x5c\x29",
  [0x5097] = "\x01\x79\x5e", [0x5098] = "\x0e\x5e\x21",
  [0x5099] = "\x02\x6b\x3f", [0x509a] = "\x02\x6b\x41",
  [0x509b] = "\x02\x6b\x40", [0x509c] = "\x02\x6d\x38",
  [0x509d] = "\x01\x7b\x51", [0x509e] = "\x0e\x5f\x3a",
  [0x509f] = "\x01\x7c\x5b", [0x50a0] = "\x02\x70\x76",
  [0x50a1] = "\x02\x71\x52", [0x50a2] = "\x0e\x62\x31",
  [0x50a3] = "\x0e\x62\x3f", [0x50a4] = "\x02\x72\x41",
  [0x50a5] = "\x01\x5e\x27", [0x50a6] = "\x0e\x25\x47",
  [0x50a7] = "\x02\x4f\x4d", [0x50a9] = "\x01\x70\x2b",
  [0x50aa] = "\x0e\x50\x2f", [0x50ab] = "\x0e\x50\x2e",
  [0x50ac] = "\x0e\x54\x3a", [0x50ad] = "\x02\x5c\x3f",
  [0x50ae] = "\x02\x5c\x3e", [0x50af] = "\x0e\x57\x50",
  [0x50b0] = "\x02\x61\x39", [0x50b1] = "\x0e\x5a\x26",
  [0x50b2] = "\x0e\x5a\x25", [0x50b4] = "\x01\x79\x60",
  [0x50b5] = "\x01\x7a\x5c", [0x50b6] = "\x02\x6e\x74",
  [0x50b7] = "\x02\x72\x39", [0x50bb] = "\x01\x5e\x28",
  [0x50bc] = "\x01\x6b\x7b", [0x50bd] = "\x0e\x4b\x22",
  [0x50be] = "\x01\x70\x2c", [0x50c0] = "\x02\x68\x61",
  [0x50c1] = "\x0e\x5e\x22", [0x50c2] = "\x02\x70\x24",
  [0x50c3] = "\x01\x63\x27", [0x50c4] = "\x0e\x39\x6a",
  [0x50c5] = "\x0e\x54\x3c", [0x50c6] = "\x0e\x54\x3b",
  [0x50c8] = "\x02\x61\x3a", [0x50ca] = "\x0e\x5a\x28",
  [0x50cc] = "\x01\x7d\x31", [0x50cd] = "\x01\x63\x28",
  [0x50ce] = "\x01\x70\x2d", [0x50cf] = "\x01\x76\x34",
  [0x50d0] = "\x02\x70\x25", [0x50d1] = "\x01\x63\x29",
  [0x50d2] = "\x0e\x39\x6b", [0x50d3] = "\x02\x56\x3c",
  [0x50d4] = "\x01\x73\x3d", [0x50d5] = "\x02\x5c\x40",
  [0x50d6] = "\x02\x5c\x41", [0x50d7] = "\x0e\x54\x3d",
  [0x50d8] = "\x01\x73\x3c", [0x50da] = "\x02\x61\x3b",
  [0x50db] = "\x01\x76\x38", [0x50dc] = "\x01\x76\x36",
  [0x50dd] = "\x01\x76\x37", [0x50de] = "\x01\x76\x35",
  [0x50df] = "\x02\x65\x2c", [0x50e0] = "\x01\x78\x28",
  [0x50e3] = "\x0e\x5c\x2a", [0x50e4] = "\x02\x6b\x43",
  [0x50e5] = "\x02\x6b\x42", [0x50e6] = "\x02\x6b\x45",
  [0x50e7] = "\x02\x6b\x44", [0x50e8] = "\x01\x7a\x5d",
  [0x50eb] = "\x02\x6d\x39", [0x50ed] = "\x02\x6d\x3b",
  [0x50ee] = "\x02\x6d\x3a", [0x50ef] = "\x01\x7b\x52",
  [0x50f0] = "\x02\x6e\x75", [0x50f1] = "\x0e\x60\x29",
  [0x50f2] = "\x02\x70\x26", [0x50f3] = "\x02\x70\x27",
  [0x50f4] = "\x01\x7c\x5c", [0x50f5] = "\x02\x71\x53",
  [0x50f6] = "\x02\x71\x6d", [0x50f7] = "\x01\x7d\x42",
  [0x50f9] = "\x02\x41\x55", [0x50fa] = "\x02\x5c\x42",
  [0x50fb] = "\x02\x61\x3c", [0x50fc] = "\x02\x68\x62",
  [0x50fd] = "\x02\x48\x75", [0x50fe] = "\x0e\x27\x72",
  [0x50ff] = "\x02\x61\x3d", [0x5100] = "\x02\x65\x2e",
  [0x5101] = "\x02\x65\x2d", [0x5102] = "\x0e\x5a\x29",
  [0x5103] = "\x0e\x5c\x2d", [0x5104] = "\x0e\x5c\x2e",
  [0x5105] = "\x0e\x5f\x3c", [0x5106] = "\x02\x70\x28",
  [0x5107] = "\x01\x7c\x7d", [0x5108] = "\x0e\x61\x5f",
  [0x5109] = "\x02\x71\x54", [0x510a] = "\x02\x71\x6e",
  [0x510e] = "\x01\x67\x6f", [0x510f] = "\x02\x56\x3d",
  [0x5110] = "\x02\x56\x3e", [0x5112] = "\x02\x5c\x43",
  [0x5113] = "\x01\x67\x70", [0x5114] = "\x0e\x45\x6a",
  [0x5115] = "\x01\x78\x29", [0x5116] = "\x02\x65\x2f",
  [0x5117] = "\x0e\x5c\x30", [0x5118] = "\x02\x6d\x3d",
  [0x5119] = "\x01\x7b\x53", [0x511a] = "\x02\x6d\x3e",
  [0x511b] = "\x02\x6d\x3c", [0x511c] = "\x02\x70\x29",
  [0x511d] = "\x0e\x61\x60", [0x511e] = "\x02\x70\x77",
  [0x5120] = "\x01\x67\x71", [0x5122] = "\x02\x61\x40",
  [0x5123] = "\x02\x61\x3f", [0x5124] = "\x02\x61\x3e",
  [0x5125] = "\x02\x65\x30", [0x5126] = "\x0e\x5a\x2b",
  [0x5127] = "\x0e\x5a\x2a", [0x5128] = "\x02\x65\x34",
  [0x5129] = "\x02\x65\x33", [0x512a] = "\x02\x65\x32",
  [0x512b] = "\x02\x65\x31", [0x512c] = "\x01\x78\x2a",
  [0x512d] = "\x02\x68\x63", [0x512e] = "\x02\x6b\x47",
  [0x512f] = "\x01\x7a\x5e", [0x5130] = "\x02\x6b\x46",
  [0x5131] = "\x02\x6d\x3f", [0x5132] = "\x02\x6e\x78",
  [0x5133] = "\x02\x6e\x77", [0x5134] = "\x01\x7c\x30",
  [0x5135] = "\x02\x6e\x76", [0x5136] = "\x02\x70\x2c",
  [0x5137] = "\x02\x70\x2b", [0x5138] = "\x02\x70\x2a",
  [0x5139] = "\x0e\x60\x6d", [0x513b] = "\x01\x6b\x7c",
  [0x513d] = "\x02\x5c\x44", [0x513e] = "\x01\x76\x39",
  [0x5140] = "\x02\x68\x64", [0x5141] = "\x02\x68\x65",
  [0x5142] = "\x02\x6e\x79", [0x5143] = "\x02\x70\x2d",
  [0x5145] = "\x0e\x61\x3b", [0x5146] = "\x02\x70\x78",
  [0x5147] = "\x02\x71\x55", [0x5148] = "\x02\x72\x29",
  [0x5149] = "\x02\x72\x43", [0x514a] = "\x01\x6b\x7d",
  [0x514b] = "\x01\x76\x3a", [0x514c] = "\x02\x65\x35",
  [0x514d] = "\x02\x68\x66", [0x514e] = "\x02\x6d\x40",
  [0x514f] = "\x02\x70\x2e", [0x5150] = "\x0e\x23\x70",
  [0x5152] = "\x01\x70\x2f", [0x5153] = "\x0e\x54\x3e",
  [0x5154] = "\x02\x61\x41", [0x5155] = "\x02\x65\x36",
  [0x5156] = "\x02\x68\x67", [0x5157] = "\x02\x68\x68",
  [0x5158] = "\x02\x68\x69", [0x5159] = "\x02\x6b\x4c",
  [0x515a] = "\x0e\x5e\x24", [0x515b] = "\x02\x6b\x48",
  [0x515c] = "\x01\x7b\x54", [0x515d] = "\x02\x6b\x4b",
  [0x515e] = "\x02\x6b\x4a", [0x515f] = "\x01\x7a\x5f",
  [0x5160] = "\x02\x6b\x49", [0x5161] = "\x01\x7a\x61",
  [0x5163] = "\x01\x7a\x60", [0x5164] = "\x02\x6d\x42",
  [0x5165] = "\x02\x6d\x41", [0x5166] = "\x01\x7b\x55",
  [0x5167] = "\x01\x7b\x56", [0x5169] = "\x0e\x5f\x3e",
  [0x516a] = "\x01\x7c\x32", [0x516b] = "\x02\x6e\x7a",
  [0x516c] = "\x01\x7c\x31", [0x516e] = "\x02\x70\x31",
  [0x516f] = "\x02\x70\x32", [0x5170] = "\x02\x70\x30",
  [0x5171] = "\x02\x70\x2f", [0x5172] = "\x01\x7d\x21",
  [0x5174] = "\x02\x70\x79", [0x5175] = "\x02\x70\x7a",
  [0x5176] = "\x02\x70\x7b", [0x5177] = "\x01\x7c\x7e",
  [0x5178] = "\x02\x71\x56", [0x5179] = "\x02\x71\x59",
  [0x517a] = "\x02\x71\x58", [0x517b] = "\x02\x71\x57",
  [0x517c] = "\x0e\x62\x33", [0x517e] = "\x02\x72\x42",
  [0x518d] = "\x01\x73\x3e", [0x518e] = "\x0e\x57\x55",
  [0x5190] = "\x01\x78\x32", [0x5191] = "\x02\x6b\x4d",
  [0x5192] = "\x02\x6d\x43", [0x5193] = "\x0e\x60\x2b",
  [0x5194] = "\x01\x7c\x33", [0x5195] = "\x02\x6e\x7b",
  [0x5197] = "\x0e\x62\x42", [0x5198] = "\x02\x72\x44",
  [0x519c] = "\x01\x73\x3f", [0x519d] = "\x0e\x5e\x47",
  [0x51a0] = "\x02\x61\x42", [0x51a1] = "\x0e\x5f\x3f",
  [0x51a2] = "\x02\x6e\x7c", [0x51a3] = "\x0e\x61\x61",
  [0x51a4] = "\x02\x71\x6f", [0x51a5] = "\x0e\x61\x77",
};
