/* GNU gettext - internationalization aids
   Copyright (C) 1995-1998, 2000-2003 Free Software Foundation, Inc.

   This file was written by Peter Miller <millerp@canb.auug.org.au>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free SoftwareFoundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#ifndef _MESSAGE_H
#define _MESSAGE_H

#include "str-list.h"
#include "pos.h"
#include "hash.h"

#include <stdbool.h>

/* According to Sun's Uniforum proposal the default message domain is
   named 'messages'.  */
#define MESSAGE_DOMAIN_DEFAULT "messages"


/* Kinds of format strings.  */
enum format_type
{
  format_c,
  format_python,
  format_lisp,
  format_elisp,
  format_librep,
  format_smalltalk,
  format_java,
  format_awk,
  format_pascal,
  format_ycp,
  format_tcl,
  format_php
};
#define NFORMATS 12	/* Number of format_type enum values.  */
extern const char *const format_language[NFORMATS];
extern const char *const format_language_pretty[NFORMATS];

/* Is current msgid a format string?  */
enum is_format
{
  undecided,
  yes,
  no,
  possible,
  impossible
};

extern bool
       possible_format_p (enum is_format);


/* Is current msgid wrappable?  */
#if 0
enum is_wrap
{
  undecided,
  yes,
  no
};
#else /* HACK - C's enum concept is so stupid */
#define is_wrap is_format
#endif


typedef struct message_ty message_ty;
struct message_ty
{
  /* The msgid string.  */
  const char *msgid;

  /* The msgid's plural, if present.  */
  const char *msgid_plural;

  /* The msgstr strings.  */
  const char *msgstr;
  /* The number of bytes in msgstr, including the terminating NUL.  */
  size_t msgstr_len;

  /* Position in the source PO file.  */
  lex_pos_ty pos;

  /* Plain comments (#) appearing before the message.  */
  string_list_ty *comment;

  /* Extracted comments (#.) appearing before the message.  */
  string_list_ty *comment_dot;

  /* File position comments (#:) appearing before the message, one for
     each unique file position instance, sorted by file name and then
     by line.  */
  size_t filepos_count;
  lex_pos_ty *filepos;

  /* Informations from special comments (e.g. generated by msgmerge).  */
  bool is_fuzzy;
  enum is_format is_format[NFORMATS];

  /* Do we want the string to be wrapped in the emitted PO file?  */
  enum is_wrap do_wrap;

  /* If set the message is obsolete and while writing out it should be
     commented out.  */
  bool obsolete;

  /* Used for checking that messages have been used, in the msgcmp,
     msgmerge, msgcomm and msgcat programs.  */
  int used;

  /* Used for looking up the target message, in the msgcat program.  */
  message_ty *tmp;

  /* Used for combining alternative translations, in the msgcat program.  */
  int alternative_count;
  struct altstr
    {
      const char *msgstr;
      size_t msgstr_len;
      const char *msgstr_end;
      string_list_ty *comment;
      string_list_ty *comment_dot;
      char *id;
    }
    *alternative;
};

extern message_ty *
       message_alloc (const char *msgid, const char *msgid_plural,
		      const char *msgstr, size_t msgstr_len,
		      const lex_pos_ty *pp);
extern void
       message_free (message_ty *mp);
extern void
       message_comment_append (message_ty *mp, const char *comment);
extern void
       message_comment_dot_append (message_ty *mp, const char *comment);
extern void
       message_comment_filepos (message_ty *mp, const char *name, size_t line);
extern message_ty *
       message_copy (message_ty *mp);


typedef struct message_list_ty message_list_ty;
struct message_list_ty
{
  message_ty **item;
  size_t nitems;
  size_t nitems_max;
  bool use_hashtable;
  hash_table htable;	/* Table mapping msgid to 'message_ty *'.  */
};

/* Create a fresh message list.
   If USE_HASHTABLE is true, a hash table will be used to speed up
   message_list_search().  USE_HASHTABLE can only be set to true if it is
   known that the message list will not contain duplicate msgids.  */
extern message_list_ty *
       message_list_alloc (bool use_hashtable);
extern void
       message_list_free (message_list_ty *mlp);
extern void
       message_list_append (message_list_ty *mlp, message_ty *mp);
extern void
       message_list_prepend (message_list_ty *mlp, message_ty *mp);
extern void
       message_list_delete_nth (message_list_ty *mlp, size_t n);
typedef bool message_predicate_ty (const message_ty *mp);
extern void
       message_list_remove_if_not (message_list_ty *mlp,
				   message_predicate_ty *predicate);
extern message_ty *
       message_list_search (message_list_ty *mlp, const char *msgid);
extern message_ty *
       message_list_search_fuzzy (message_list_ty *mlp, const char *msgid);


typedef struct message_list_list_ty message_list_list_ty;
struct message_list_list_ty
{
  message_list_ty **item;
  size_t nitems;
  size_t nitems_max;
};

extern message_list_list_ty *
       message_list_list_alloc (void);
extern void
       message_list_list_free (message_list_list_ty *mllp);
extern void
       message_list_list_append (message_list_list_ty *mllp,
				 message_list_ty *mlp);
extern void
       message_list_list_append_list (message_list_list_ty *mllp,
				      message_list_list_ty *mllp2);
extern message_ty *
       message_list_list_search (message_list_list_ty *mllp,
				 const char *msgid);
extern message_ty *
       message_list_list_search_fuzzy (message_list_list_ty *mllp,
				       const char *msgid);


typedef struct msgdomain_ty msgdomain_ty;
struct msgdomain_ty
{
  const char *domain;
  message_list_ty *messages;
};

extern msgdomain_ty *
       msgdomain_alloc (const char *domain, bool use_hashtable);
extern void
       msgdomain_free (msgdomain_ty *mdp);


typedef struct msgdomain_list_ty msgdomain_list_ty;
struct msgdomain_list_ty
{
  msgdomain_ty **item;
  size_t nitems;
  size_t nitems_max;
  bool use_hashtable;
  const char *encoding;		/* canonicalized encoding or NULL if unknown */
};

extern msgdomain_list_ty *
       msgdomain_list_alloc (bool use_hashtable);
extern void
       msgdomain_list_free (msgdomain_list_ty *mdlp);
extern void
       msgdomain_list_append (msgdomain_list_ty *mdlp, msgdomain_ty *mdp);
extern void
       msgdomain_list_append_list (msgdomain_list_ty *mdlp,
				   msgdomain_list_ty *mdlp2);
extern message_list_ty *
       msgdomain_list_sublist (msgdomain_list_ty *mdlp, const char *domain,
			       bool create);
extern message_ty *
       msgdomain_list_search (msgdomain_list_ty *mdlp, const char *msgid);
extern message_ty *
       msgdomain_list_search_fuzzy (msgdomain_list_ty *mdlp, const char *msgid);


#endif /* message.h */
