/*
 * Electric(tm) VLSI Design System
 *
 * File: netflat.c
 * Network tool: module for fully instantiating a hierarchical network
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "network.h"
#include "efunction.h"
#include "tecschem.h"

#define IGNOREFOURPORT 1				/* comment out to handle 4-port transistors properly */

static INTBIG    net_pseudonode;		/* net number for pseudonetworks */
static NODEINST *net_toplevelinst;		/* actual top-instance associated with this node */
static PCOMP   **net_tempconlist;		/* temp list of connections */
static INTSML   *net_tempconlistindex;	/* temp list of indices */
static INTSML   *net_tempportindices;	/* temp list of normalized indices */
static INTBIG    net_tempcontotal = 0;	/* size of temp list of connections */
static PCOMP    *net_serialend1, *net_serialend2;	/* end component of serial transistor chains */
static INTBIG    net_serialcon1,  net_serialcon2;	/* end index of serial transistor chains */

/* working memory for "net_comparewires()" */
static INTBIG     net_comparelistwsize = 0;
static INTSML    *net_comparelistw;

/* working memory for "net_comparewirelist()" */
static INTBIG     net_comparelistsize = 0;
static INTSML    *net_comparelist;

/* working memory for "net_mergeserial()" */
static PCOMP    **net_seriallist;
static INTBIG     net_seriallistcount;
static INTBIG     net_seriallisttotal = 0;

/* prototypes for local routines */
static INTSML  net_addtoseriallist(PCOMP *pc);
static PCOMP  *net_allocpcomp(void);
static PNET   *net_allocpnet(void);
static PCOMP  *net_buildpseudo(NODEPROTO*, PCOMP*, PNET**, PNET**, INTBIG*, PNET**, INTSML);
static INTSML  net_comparewires(PCOMP *p1, INTBIG ind1, PCOMP *p2, INTBIG ind2,
				INTSML useportnames);
static void    net_crawlforserial(PCOMP *pc);
static INTBIG  net_findotherexporttopology(NODEPROTO *np, char *exportname);
static PCOMP **net_gatherserial(PCOMP *pc, INTBIG *serialcount);
static INTBIG  net_getfunction(NODEINST*);
static float   net_getpartvalue(NODEINST *ni);
static INTSML  net_getpnetandstate(NODEINST *ni, PORTPROTO *pp, NETWORK *forcenet,
				INTBIG index, PNET **netnumber, INTSML *state, PNET **pnetlist);
static INTBIG  net_mergeserial(PCOMP **pcomp, INTBIG *components);
static PNET   *net_newpnet(PNET**);
static void    net_removepcompfromlists(PCOMP *pcomp, PCOMP *removedpc, PCOMP *newpc);
static void    net_setallexporttopology(void);
static void    net_setthisexporttopology(PORTPROTO *pp, INTBIG *index);

/*
 * Routine to free all memory associated with this module.
 */
void net_freeflatmemory(void)
{
	REGISTER PCOMP *p;
	REGISTER PNET *pn;

	while (net_pcompfree != NOPCOMP)
	{
		p = net_pcompfree;
		net_pcompfree = net_pcompfree->nextpcomp;
		efree((char *)p);
	}
	while (net_pnetfree != NOPNET)
	{
		pn = net_pnetfree;
		net_pnetfree = net_pnetfree->nextpnet;
		efree((char *)pn);
	}
	if (net_tempcontotal > 0)
	{
		efree((char *)net_tempconlist);
		efree((char *)net_tempconlistindex);
		efree((char *)net_tempportindices);
	}
	if (net_comparelistsize > 0) efree((char *)net_comparelist);
	if (net_comparelistwsize > 0) efree((char *)net_comparelistw);
}

/*********************** PSEUDO-NETWORK CONSTRUCTION ***********************/

/*
 * The usage of this module is:
 *   #include "network.h"
 *   PCOMP *pcomp;
 *   PNET *pnet;
 *   INTBIG components, nets, powernets, groundnets;
 *   pcomp = net_makepseudo(facet, &components, &nets, &powernets, &groundnets,
 *	    &pnet, hierarchical, mergeparallel, mergeserial, checkfacetoverrides);
 *   .....
 *        do something with the network in "pcomp" and "pnet"
 *   .....
 *   net_freeallpnet(pnet);
 *   net_freeallpcomp(pcomp);
 *
 * "net_makepseudo" builds a pseudonetwork structure that represents the
 * network in facet "facet".  If it returns NOPCOMP, there is an error.
 * A linked list of PCOMP objects is returned, one for every component
 * in the pseudonetwork.  A linked list of PNET objects is also returned,
 * one for every network in the pseudonetwork.  Finally, the number of
 * components, networks, power networks, and ground networks is returned
 * in the reference parameters "components", "nets", "powernets", and
 * "groundnets".
 *
 * A number of switches controls the flattening:
 * If "hierarchical"        is nonzero, the network will be fully instantiated.
 * If "mergeparallel"       is nonzero, parallel components are merged into one.
 * If "mergeserial"         is nonzero, serial transistors are merged into one.
 * If "checkfacetoverrides" is nonzero, individual facets may override "mergeparallel".
 */
PCOMP *net_makepseudo(NODEPROTO *facet, INTBIG *components, INTBIG *nets, INTBIG *powernets,
	INTBIG *groundnets, PNET **pnetlist, INTSML hierarchical, INTSML mergeparallel,
	INTSML mergeserial, INTSML checkfacetoverrides)
{
	REGISTER PCOMP *pcomp, *p, **newconlist, *thepcomp;
	PCOMP *pcomplist;
	REGISTER INTSML *newconlistindex, *newportindices, localmergeparallel,
		localmergeserial;
	REGISTER INTBIG list, i, j, k, theindex, tempconcount, newtotal, counter,
		mergecount;
	PNET *power, *ground;
	REGISTER NETWORK *net, *subnet;
	REGISTER PNET *pn;
	REGISTER VARIABLE *var;
	REGISTER PORTPROTO *pp, *opp, *npp, **newportlist;

	/* see if the current facet overrides the options */
	localmergeparallel = mergeparallel;
	localmergeserial = mergeserial;
	if (checkfacetoverrides != 0)
	{
		var = getvalkey((INTBIG)facet, VNODEPROTO, VINTEGER, net_ncc_optionskey);
		if (var != NOVARIABLE)
		{
			if ((var->addr&NCCNOMERGEPARALLELOVER) != 0)
			{
				if ((var->addr&NCCNOMERGEPARALLEL) == 0) localmergeparallel = 1; else
					localmergeparallel = 0;
			}
			if ((var->addr&NCCMERGESERIALOVER) != 0)
			{
				if ((var->addr&NCCMERGESERIAL) != 0) localmergeserial = 1; else
					localmergeserial = 0;
			}
		}
	}

	/* first create net numbers inside of this facet */
	net_pseudonode = 0;
	*pnetlist = NOPNET;
	power = ground = NOPNET;
	for(net = facet->firstnetwork; net != NONETWORK; net = net->nextnetwork)
		net->temp1 = (INTBIG)NOPNET;
	for(pp = facet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		pp->temp1 = (INTBIG)NOPNET;
	for(pp = facet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (pp->temp1 != (INTBIG)NOPNET) continue;
		if ((pp->userbits&STATEBITS) == PWRPORT)
		{
			pn = net_newpnet(pnetlist);
			if (pn == NOPNET) return(NOPCOMP);
			pn->flags = POWERNET;
			power = pn;
		} else if ((pp->userbits&STATEBITS) == GNDPORT)
		{
			pn = net_newpnet(pnetlist);
			if (pn == NOPNET) return(NOPCOMP);
			pn->flags = GROUNDNET;
			ground = pn;
		} else
		{
			pn = net_newpnet(pnetlist);
			if (pn == NOPNET) return(NOPCOMP);
		}
		if (pn != NOPNET)
		{
			pn->flags |= EXPORTEDNET;
			net = pp->network;
			pn->network = net;
			pn->realportcount = 1;
			pn->realportlist = pp;
			pp->temp1 = (INTBIG)pn;
			net->temp1 = (INTBIG)pn;
			for(opp = pp->nextportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
			{
				if (net != opp->network) continue;
				opp->temp1 = pp->temp1;
				newportlist = (PORTPROTO **)emalloc((pn->realportcount+1) *
					(sizeof (PORTPROTO *)), net_tool->cluster);
				if (newportlist == 0) return(NOPCOMP);
				for(i=0; i<pn->realportcount; i++)
				{
					if (pn->realportcount == 1) npp = (PORTPROTO *)pn->realportlist; else
						npp = ((PORTPROTO **)pn->realportlist)[i];
					newportlist[i] = npp;
				}
				newportlist[pn->realportcount] = opp;
				if (pn->realportcount > 1) efree((char *)pn->realportlist);
				pn->realportlist = newportlist;
				pn->realportcount++;
			}
			if (net->signals > 1)
			{
				for(i=0; i<net->signals; i++)
				{
					subnet = net->networklist[i];
					if (subnet->temp1 != (INTBIG)NOPNET) continue;
					pn = net_newpnet(pnetlist);
					pn->flags |= EXPORTEDNET;
					pn->network = subnet;
					subnet->temp1 = (INTBIG)pn;
				}
			}
		}
	}

	/* establish topology of exports */
	net_setallexporttopology();

	/* create a list of pseudocomponents in this facet */
	*components = 0;
	begintraversehierarchy();
	net_toplevelinst = NONODEINST;
	pcomplist = net_buildpseudo(facet, NOPCOMP, &power, &ground, components,
		pnetlist, hierarchical);
	if (pcomplist == 0) return(NOPCOMP);
	if (*components == 0)
	{
		ttyputmsg(_("There are no components in facet %s"), describenodeproto(facet));
		return(NOPCOMP);
	}

	/* now run through each network, accumulating pointers */
	counter = 0;
	for(pcomp = pcomplist; pcomp != NOPCOMP; pcomp = pcomp->nextpcomp)
	{
		counter++;
		if ((counter%5) == 0)
		{
			if (stopping(STOPREASONNCC))
			{
				net_freeallpcomp(pcomplist);
				net_freeallpnet(*pnetlist);
				*pnetlist = NOPNET;
				return(NOPCOMP);
			}
		}
		for(list=0; list<pcomp->wirecount; list++)
		{
			if (pcomp->count[list] > 0) continue;
			pn = pcomp->netnumbers[list];

			/* accumulate all other nodes that have this connection */
			tempconcount = 0;
			for(p = pcomplist; p != NOPCOMP; p = p->nextpcomp)
			{
				for(i=0; i<p->wirecount; i++)
				{
					if (p->netnumbers[i] != pn) continue;

					/* create a pseudo-net object for this */
					if (tempconcount >= net_tempcontotal)
					{
						newtotal = net_tempcontotal * 2;
						if (newtotal == 0) newtotal = 5;
						newconlist = (PCOMP **)emalloc(newtotal * (sizeof (PCOMP *)), net_tool->cluster);
						if (newconlist == 0) return(NOPCOMP);
						newconlistindex = (INTSML *)emalloc(newtotal * SIZEOFINTSML, net_tool->cluster);
						if (newconlistindex == 0) return(NOPCOMP);
						newportindices = (INTSML *)emalloc(newtotal * SIZEOFINTSML, net_tool->cluster);
						if (newportindices == 0) return(NOPCOMP);
						for(j=0; j<tempconcount; j++)
						{
							newconlist[j] = net_tempconlist[j];
							newconlistindex[j] = net_tempconlistindex[j];
							newportindices[j] = net_tempportindices[j];
						}
						if (net_tempcontotal > 0)
						{
							efree((char *)net_tempconlist);
							efree((char *)net_tempconlistindex);
							efree((char *)net_tempportindices);
						}
						net_tempconlist = newconlist;
						net_tempconlistindex = newconlistindex;
						net_tempportindices = newportindices;
						net_tempcontotal = newtotal;
					}
					net_tempconlist[tempconcount] = p;
					net_tempconlistindex[tempconcount] = (INTSML)i;
					net_tempportindices[tempconcount] = p->portindices[i];
					tempconcount++;
				}
			}
			if (tempconcount > 1)
			{
				for(j=0; j<tempconcount; j++)
				{
					thepcomp = net_tempconlist[j];
					theindex = net_tempconlistindex[j];
					thepcomp->pconnarray[theindex] = (PCOMP **)emalloc((tempconcount-1) * (sizeof (PCOMP *)),
						net_tool->cluster);
					if (thepcomp->pconnarray[theindex] == 0) return(NOPCOMP);

					thepcomp->portindexarray[theindex] = (INTSML *)emalloc((tempconcount-1) * SIZEOFINTSML,
						net_tool->cluster);
					if (thepcomp->portindexarray[theindex] == 0) return(NOPCOMP);
					k = 0;
					for(i=0; i<tempconcount; i++)
					{
						if (net_tempconlist[i] == thepcomp && net_tempconlistindex[i] == theindex)
							continue;
						thepcomp->pconnarray[theindex][k] = net_tempconlist[i];
						thepcomp->portindexarray[theindex][k] = net_tempportindices[i];
						k++;
					}
					thepcomp->count[theindex] = (INTSML)(tempconcount-1);
				}
			}
		}
	}

	/* reduce network by merging parallel components */
	if (localmergeparallel != 0)
	{
		mergecount = net_mergeparallel(&pcomplist, components);
		if (mergecount < 0) return(NOPCOMP);
		if (mergecount != 0)
			ttyputmsg(_("Merged %ld parallel components in facet %s"),
				mergecount, describenodeproto(facet));
	}
	if (localmergeserial != 0)
	{
		mergecount = net_mergeserial(&pcomplist, components);
		if (mergecount != 0)
			ttyputmsg(_("Merged %ld serial transistors in facet %s"),
				mergecount, describenodeproto(facet));
	}

	/* report the total number of nets */
	*nets = net_pseudonode;

	/* count the power and ground nets */
	*powernets = *groundnets = 0;
	for(pn = *pnetlist; pn != NOPNET; pn = pn->nextpnet)
	{
		if ((pn->flags&POWERNET) != 0) (*powernets)++;
		if ((pn->flags&GROUNDNET) != 0) (*groundnets)++;
	}
	return(pcomplist);
}

/*
 * routine to build a linked list of pseudocomponents in facet "facet".  The
 * list is appended to "initiallist" and returned.  The values of "power" and
 * "ground" are the PNETs of such components.  Routine increments the
 * integer at "components" for every component created.  If
 * "compare_hierarchically" is nonzero, net is flattened.  Returns zero on error.
 */
PCOMP *net_buildpseudo(NODEPROTO *facet, PCOMP *initiallist, PNET **power,
	PNET **ground, INTBIG *components, PNET **pnetlist, INTSML compare_hierarchically)
{
	REGISTER PCOMP *pcomp;
	REGISTER PORTPROTO *pp, *opp, *realpp;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER ARCINST *ai;
	REGISTER NETWORK *net, *outsidenet, *subnet;
	REGISTER NODEINST *ni;
	NODEINST **nilist;
	REGISTER INTBIG fun, i, j, k, l, toldshort, toplevel, flattenit, lambda;
	INTBIG width, length;
	REGISTER NODEPROTO *realnp, *anp, *cnp;
	REGISTER PNET *pn;

	if (net_toplevelinst == NONODEINST) toplevel = 1; else
		toplevel = 0;

	/* make simple checks that port characteristics match the name */
	for(pp = facet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (portisnamedpower(pp) != 0)
		{
			/* port has "pwr" in its name, make sure it is marked power */
			if ((pp->userbits&STATEBITS) != PWRPORT)
				ttyputmsg(_("Warning: export %s in facet %s is named as a power port but has type %s"),
					pp->protoname, describenodeproto(facet), describeportbits(pp));
		}
		if ((pp->userbits&STATEBITS) == PWRPORT)
		{
			/* port is marked power, see if it has an appropriate name */
			if (portisnamedpower(pp) == 0)
				ttyputmsg(_("Warning: power export %s in facet %s is not named appropriately"),
					pp->protoname, describenodeproto(facet));
		}
		if (portisnamedground(pp) != 0)
		{
			/* port has "ground" in its name, make sure it is marked ground */
			if ((pp->userbits&STATEBITS) != GNDPORT)
				ttyputmsg(_("Warning: export %s in facet %s is named as a ground port but has type %s"),
					pp->protoname, describenodeproto(facet), describeportbits(pp));
		}
		if ((pp->userbits&STATEBITS) == GNDPORT)
		{
			/* port is marked ground, see if it has an appropriate name */
			if (portisnamedground(pp) == 0)
				ttyputmsg(_("Warning: ground export %s in facet %s is not named appropriately"),
					pp->protoname, describenodeproto(facet));
		}

		/* check busses for consistent component characteristics */
		if (pp->network->signals > 1)
		{
			for(i=0; i<pp->network->signals; i++)
			{
				subnet = pp->network->networklist[i];
				for(opp = facet->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
				{
					if (opp->network != subnet) continue;
					if ((opp->userbits&STATEBITS) != (pp->userbits&STATEBITS))
					{
						ttyputerr(_("Warning: bus export %s is %s but export %s is %s"),
							pp->protoname, describeportbits(pp), opp->protoname,
								describeportbits(opp));
					}
					break;
				}
			}
		}
	}

	/* spread power and ground information from appropriate nodes */
	for(ni = facet->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		/* see if power or ground comes from this node */
		for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
			if ((pe->exportproto->userbits&STATEBITS) == PWRPORT ||
				(pe->exportproto->userbits&STATEBITS) == GNDPORT) break;
		if (ni->proto->primindex == 0) fun = 0; else
			fun = net_getfunction(ni);
		if (fun != NPCONPOWER && fun != NPCONGROUND && pe == NOPORTEXPINST) continue;

		/* they do: get the network */
		if (pe != NOPORTEXPINST)
			pp = pe->proto; else
				pp = ni->proto->firstportproto;
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			if (pi->proto == pp) break;
		if (pi == NOPORTARCINST) continue;

		/* propagate power and ground */
		if (fun == NPCONPOWER ||
			(pe != NOPORTEXPINST && (pe->exportproto->userbits&STATEBITS) == PWRPORT))
		{
			pn = *power;
			if (pn == NOPNET)
			{
				pn = (PNET *)pi->conarcinst->network->temp1;
				if (pn == NOPNET)
				{
					pn = net_newpnet(pnetlist);
					pn->network = NONETWORK;
				}
				pn->flags = POWERNET;
				*power = pn;
			}
		}
		if (fun == NPCONGROUND ||
			(pe != NOPORTEXPINST && (pe->exportproto->userbits&STATEBITS) == GNDPORT))
		{
			pn = *ground;
			if (pn == NOPNET)
			{
				pn = (PNET *)pi->conarcinst->network->temp1;
				if (pn == NOPNET)
				{
					pn = net_newpnet(pnetlist);
					pn->network = NONETWORK;
				}
				pn->flags = GROUNDNET;
				*ground = pn;
			}
		}
		for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			if (pi->proto != pp) continue;
			ai = pi->conarcinst;
			if (pn->network == NONETWORK) pn->network = ai->network;
			if (ai->network->temp1 != (INTBIG)pn)
			{
				if (*power != NOPNET && *ground != NOPNET &&
					((pn == *power && (PNET *)ai->network->temp1 == *ground) ||
					 (pn == *ground && (PNET *)ai->network->temp1 == *power)))
				{
					ttyputerr(_("Warning: power/ground connected incorrectly in facet %s"),
						describenodeproto(facet));
				}
			}
			ai->network->temp1 = (INTBIG)pn;
		}
	}

	/* generate new pseudo-netnumbers for networks not connected to ports */
	for(net = facet->firstnetwork; net != NONETWORK; net = net->nextnetwork)
	{
		if (net->temp1 != (INTBIG)NOPNET) continue;
		pn = net_newpnet(pnetlist);
		if (pn == NOPNET) return(0);
		net->temp1 = (INTBIG)pn;
		pn->network = net;
		if (net->signals > 1)
		{
			for(i=0; i<net->signals; i++)
			{
				subnet = net->networklist[i];
				if (subnet->temp1 != (INTBIG)NOPNET) continue;
				pn = net_newpnet(pnetlist);
				pn->flags |= EXPORTEDNET;
				pn->network = subnet;
				subnet->temp1 = (INTBIG)pn;
			}
		}
	}

	/* search every component in the facet */
	toldshort = 0;
	for(ni = facet->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		anp = ni->proto;
		if (toplevel != 0) net_toplevelinst = ni;

		/* ignore recursive references (showing icon in contents) */
		if (anp->cell == facet->cell) continue;

		/* if flattening the circuit, explore contents of facet instances */
		flattenit = 0;
		if (anp->primindex == 0 && compare_hierarchically != 0) flattenit = 1;

		if (flattenit != 0)		
		{
			/* if there is an alternate contents facet, use it */
			realnp = contentsview(anp);
			if (realnp == NONODEPROTO) realnp = anp;

			/* put pseudo-netnumbers on the edge of this facet */
			for(realpp = realnp->firstportproto; realpp != NOPORTPROTO; realpp = realpp->nextportproto)
				realpp->temp1 = (INTBIG)NOPNET;
			for(net = realnp->firstnetwork; net != NONETWORK; net = net->nextnetwork)
				net->temp1 = (INTBIG)NOPNET;
			for(realpp = realnp->firstportproto; realpp != NOPORTPROTO; realpp = realpp->nextportproto)
			{
				if (realpp->temp1 != (INTBIG)NOPNET) continue;

				/* if there is an alternate contents facet, compute the port */
				if (realnp == anp) pp = realpp; else
				{
					pp = equivalentport(realnp, realpp, anp);
					if (pp == NOPORTPROTO) pp = anp->firstportproto;
				}

				/* see if an arc connects to the port */
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
					if (pi->proto->network == pp->network) break;
				if (pi != NOPORTARCINST)
				{
					realpp->temp1 = pi->conarcinst->network->temp1;
				} else
				{
					/* see if the port is an export */
					for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
						if (pe->proto->network == pp->network) break;
					if (pe != NOPORTEXPINST) realpp->temp1 = pe->exportproto->temp1; else
					{
						pn = net_newpnet(pnetlist);
						if (pn == NOPNET) return(0);
						pn->network = realpp->network;
						if ((realpp->userbits&STATEBITS) == PWRPORT)
						{
							pn->flags = POWERNET;
						} else if ((realpp->userbits&STATEBITS) == GNDPORT)
						{
							pn->flags = GROUNDNET;
						}
						realpp->temp1 = (INTBIG)pn;
					}
				}

				/* propagate net numbers to other connected ports */
				for(opp = realpp->nextportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
					if (opp->network == realpp->network)
						opp->temp1 = realpp->temp1;

				/* propagate export networks to nets inside facet */
				for(net = realnp->firstnetwork; net != NONETWORK; net = net->nextnetwork)
				{
					if (net != realpp->network) continue;
					net->temp1 = realpp->temp1;
					pn = (PNET *)realpp->temp1;
					outsidenet = pn->network;
					if (net->signals > 1 && outsidenet != NONETWORK &&
						outsidenet->signals == net->signals)
					{
						/* propagate bus information to individual nets */
						for(i=0; i<net->signals; i++)
						{
							subnet = net->networklist[i];
							subnet->temp1 = outsidenet->networklist[i]->temp1;
						}
					}
					break;
				}
			}

			/* recurse into the facet */
			downhierarchy(ni);
			initiallist = net_buildpseudo(realnp, initiallist, power, ground,
				components, pnetlist, compare_hierarchically);
			uphierarchy();
			if (initiallist == 0) return(0);
			continue;
		}

		/* nonflattenable component: add it to the pseudocomponent list */
		if (anp->primindex == 0) fun = NPUNKNOWN; else
		{
			fun = net_getfunction(ni);
			if (fun == NPCONNECT || fun == NPART || fun == NPUNKNOWN) continue;

			/* allow only one power and one ground component */
			if (fun == NPCONPOWER && *power != NOPNET) continue;
			if (fun == NPCONGROUND && *ground != NOPNET) continue;
		}

		/* create a pseudo-component */
		pcomp = net_allocpcomp();
		if (pcomp == NOPCOMP) return(0);
		pcomp->nextpcomp = initiallist;
		initiallist = pcomp;
		pcomp->function = (INTSML)fun;
		pcomp->hashreason = 0;
		gettraversalpath(&nilist, &pcomp->hierpathcount);
		if (pcomp->hierpathcount > 0)
		{
			pcomp->hierpath = (NODEINST **)emalloc(pcomp->hierpathcount *
				(sizeof (NODEINST *)), net_tool->cluster);
			if (pcomp->hierpath == 0) return(0);
			for(i=0; i<pcomp->hierpathcount; i++)
				pcomp->hierpath[i] = nilist[i];
		}
		pcomp->actuallist = (void *)ni;
		pcomp->numactual = 1;
		pcomp->topactual = net_toplevelinst;
		(*components)++;

		/* count the number of electrically distinct nets on the component */
		pcomp->wirecount = 0;
		cnp = contentsview(anp);
		for(pp = anp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			/* get real export on contents */
			if (cnp == NONODEPROTO) realpp = pp; else
				realpp = equivalentport(anp, pp, cnp);

			/* special case for isolated ports */
			if ((realpp->userbits&PORTISOLATED) != 0)
			{
				/* add one wire for each arc on the port */
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
					if (pi->proto == pp) pcomp->wirecount++;
				continue;
			}

			/* new port, add in the number of signals */
			if (pp->network->signals <= 1) pcomp->wirecount++; else
				pcomp->wirecount += pp->network->signals;
		}

		/* get parameters for the node */
		switch (fun)
		{
			case NPTRANMOS:  case NPTRA4NMOS:
			case NPTRADMOS:  case NPTRA4DMOS:
			case NPTRAPMOS:  case NPTRA4PMOS:
			case NPTRANJFET: case NPTRA4NJFET:
			case NPTRAPJFET: case NPTRA4PJFET:
			case NPTRADMES:  case NPTRA4DMES:
			case NPTRAEMES:  case NPTRA4EMES:
				/* transistors that have a length and a width */
				transistorsize(ni, &length, &width);
				lambda = facet->cell->lib->lambda[facet->tech->techindex];
				pcomp->length = (float)(length * WHOLE / lambda);
				pcomp->width = (float)(width * WHOLE / lambda);
				pcomp->flags |= COMPHASWIDLEN;
#ifdef IGNOREFOURPORT
				pcomp->wirecount = 3;
#endif
				break;

			case NPTRANPN:    case NPTRA4NPN:
			case NPTRAPNP:    case NPTRA4PNP:
				/* transistors that have an area */
				transistorsize(ni, &length, &width);
				lambda = facet->cell->lib->lambda[facet->tech->techindex];
				if (length < 0) pcomp->length = 0.0; else
					pcomp->length = (float)(length * WHOLE / lambda);
				pcomp->width = 0.0;
				pcomp->flags |= COMPHASAREA;
				break;

			case NPRESIST:
			case NPCAPAC:   case NPECAPAC:
			case NPDIODE:   case NPDIODEZ:
			case NPINDUCT:
				pcomp->length = net_getpartvalue(ni);
				pcomp->width = 0.0;
				pcomp->flags |= COMPHASAREA;
				break;
		}

		/* no further information if there are no wires */
		if (pcomp->wirecount == 0) continue;

		/* allocate the port and connection lists */
		pcomp->pconnarray = (PCOMP ***)emalloc(((sizeof (PCOMP **)) * pcomp->wirecount),
			net_tool->cluster);
		if (pcomp->pconnarray == 0) return(0);
		pcomp->portindexarray = (INTSML **)emalloc(((sizeof (INTSML *)) * pcomp->wirecount),
			net_tool->cluster);
		if (pcomp->portindexarray == 0) return(0);
		pcomp->portlist = (PORTPROTO **)emalloc(((sizeof (PORTPROTO *)) * pcomp->wirecount),
			net_tool->cluster);
		if (pcomp->portlist == 0) return(0);
		pcomp->count = (INTSML *)emalloc((SIZEOFINTSML * pcomp->wirecount),
			net_tool->cluster);
		if (pcomp->count == 0) return(0);
		pcomp->state = (INTSML *)emalloc((SIZEOFINTSML * pcomp->wirecount),
			net_tool->cluster);
		if (pcomp->state == 0) return(0);
		pcomp->portindices = (INTSML *)emalloc((SIZEOFINTSML * pcomp->wirecount),
			net_tool->cluster);
		if (pcomp->portindices == 0) return(0);
		pcomp->netnumbers = (PNET **)emalloc(((sizeof (PNET *)) * pcomp->wirecount),
			net_tool->cluster);
		if (pcomp->netnumbers == 0) return(0);
		for(i=0; i<pcomp->wirecount; i++)
		{
			pcomp->count[i] = 0;
			pcomp->state[i] = 0;
		}

		switch (pcomp->function)
		{
			case NPTRANMOS:
			case NPTRADMOS:
			case NPTRAPMOS:
			case NPTRADMES:
			case NPTRAEMES:
			case NPTRANPN:
			case NPTRAPNP:
			case NPTRANJFET:
			case NPTRAPJFET:
				/* transistors make the active ports equivalent */
				pcomp->portlist[0] = anp->firstportproto;
				pcomp->portlist[1] = pcomp->portlist[0]->nextportproto;
				pcomp->portlist[2] = pcomp->portlist[1]->nextportproto;
				if (anp != sch_transistorprim && anp != sch_transistor4prim)
					pcomp->portlist[2] = pcomp->portlist[2]->nextportproto;
				for(j=0; j<pcomp->wirecount; j++)
				{
					pp = pcomp->portlist[j];
					pcomp->portindices[j] = (INTSML)pp->network->temp2;
					if (net_getpnetandstate(ni, pp, NONETWORK, -1, &pcomp->netnumbers[j],
						&pcomp->state[j], pnetlist) != 0) return(0);
				}
				break;

			case NPTRA4NMOS:
			case NPTRA4DMOS:
			case NPTRA4PMOS:
			case NPTRA4DMES:
			case NPTRA4EMES:
			case NPTRA4NPN:
			case NPTRA4PNP:
			case NPTRA4NJFET:
			case NPTRA4PJFET:
				/* 4-port transistors make the active two equivalent */
				pcomp->portlist[0] = anp->firstportproto;
				pcomp->portlist[1] = pcomp->portlist[0]->nextportproto;
				pcomp->portlist[2] = pcomp->portlist[1]->nextportproto;
				pcomp->portlist[3] = pcomp->portlist[2]->nextportproto;
				for(j=0; j<pcomp->wirecount; j++)
				{
					pp = pcomp->portlist[j];
					pcomp->portindices[j] = (INTSML)pp->network->temp2;
					if (net_getpnetandstate(ni, pp, NONETWORK, -1, &pcomp->netnumbers[j],
						&pcomp->state[j], pnetlist) != 0) return(0);
				}
				break;

			default:
				j = 0;
				for(pp = anp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				{
					/* get real export on contents */
					if (cnp == NONODEPROTO) realpp = pp; else
						realpp = equivalentport(anp, pp, cnp);

					/* special case for isolated ports */
					if ((realpp->userbits&PORTISOLATED) != 0)
					{
						/* add one wire for each arc on the port */
						for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
						{
							if (pi->proto != pp) continue;
							pcomp->portlist[j] = pp;
							pcomp->portindices[j] = (INTSML)pp->network->temp2;
							if (net_getpnetandstate(ni, pp, pi->conarcinst->network, -1,
								&pcomp->netnumbers[j], &pcomp->state[j], pnetlist) != 0) return(0);
							j++;
						}
						continue;
					}

					if (realpp->network->signals <= 1)
					{
						/* single-wire port */
						pcomp->portlist[j] = pp;
						pcomp->portindices[j] = (INTSML)realpp->network->temp2;
						for(l=0; l<j; l++) if (pcomp->portindices[l] == pcomp->portindices[j]) break;
						if (l >= j)
						{
							if (net_getpnetandstate(ni, pp, NONETWORK, -1, &pcomp->netnumbers[j],
								&pcomp->state[j], pnetlist) != 0) return(0);
							j++;
						}
					} else
					{
						/* bus port */
						for(k=0; k<realpp->network->signals; k++)
						{
							pcomp->portlist[j] = pp;
							pcomp->portindices[j] = (INTSML)realpp->network->networklist[k]->temp2;
							for(l=0; l<j; l++) if (pcomp->portindices[l] == pcomp->portindices[j]) break;
							if (l >= j)
							{
								if (net_getpnetandstate(ni, pp, NONETWORK, k,
									&pcomp->netnumbers[j], &pcomp->state[j], pnetlist) != 0) return(0);
								j++;
							}
						}
					}
				}
				pcomp->wirecount = (INTSML)j;
				break;
		}

		/* for power or ground components, set the characteristics on the nets */
		if (fun == NPCONPOWER || fun == NPCONGROUND)
		{
			for(i=0; i<pcomp->wirecount; i++)
			{
				if (fun == NPCONPOWER)
				{
					if ((pcomp->netnumbers[i]->flags&GROUNDNET) != 0 && toldshort == 0)
					{
						ttyputerr(_("Power and ground are shorted in %s"), describenodeproto(facet));
						toldshort++;
					}
					pcomp->netnumbers[i]->flags |= POWERNET;
					*power = pcomp->netnumbers[i];
				}
				if (fun == NPCONGROUND)
				{
					if ((pcomp->netnumbers[i]->flags&POWERNET) != 0 && toldshort == 0)
					{
						ttyputerr(_("Power and ground are shorted in %s"), describenodeproto(facet));
						toldshort++;
					}
					pcomp->netnumbers[i]->flags |= GROUNDNET;
					*ground = pcomp->netnumbers[i];
				}
			}
		}
	}
	return(initiallist);
}

/*
 * Routine to examine node "ni", port "pp", and find the PNET that is connected to it.
 * If "forcenet" is not NONETWORK, use it as the arc site.
 * If "index" is not negative, look for that entry in a bus.
 * The PNET and its state are stored in "netnumber" and "state".  If nothing is connected,
 * a new PNET is allocated and saved in the list "pnetlist".
 * Returns nonzero on error.
 */
INTSML net_getpnetandstate(NODEINST *ni, PORTPROTO *pp, NETWORK *forcenet,
	INTBIG index, PNET **netnumber, INTSML *state, PNET **pnetlist)
{
	REGISTER ARCINST *ai;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER NETWORK *net;
	REGISTER PNET *pn;

	*netnumber = NOPNET;
	for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
	{
		if (forcenet != NONETWORK)
		{
			if (forcenet != pi->conarcinst->network) continue;
		} else
		{
			if (pi->proto->network != pp->network) continue;
		}

		/* pickup the network number of this connection */
		ai = pi->conarcinst;
		if (index < 0) *netnumber = (PNET *)ai->network->temp1; else
		{
			net = ai->network;
			if (index < net->signals) *netnumber = (PNET *)net->networklist[index]->temp1;
		}
		if ((ai->userbits&ISNEGATED) != 0)
		{
			if ((ai->end[0].portarcinst == pi && (ai->userbits&REVERSEEND) == 0) ||
				(ai->end[1].portarcinst == pi && (ai->userbits&REVERSEEND) != 0))
					*state = NEGATEDPORT;
		}
		if (*netnumber != NOPNET)
		{
			if (((*netnumber)->flags&EXPORTEDNET) != 0)
				*state |= EXPORTEDPORT;
		}
		break;
	}
	if (*netnumber == NOPNET)
	{
		for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
		{
			if (forcenet != NONETWORK)
			{
				if (forcenet != pe->proto->network) continue;
			} else
			{
				if (pe->proto->network != pp->network) continue;
			}
			if (index < 0) *netnumber = (PNET *)pe->exportproto->temp1; else
			{
				net = pe->exportproto->network;
				if (index < net->signals) *netnumber = (PNET *)net->networklist[index]->temp1;
			}
			*state |= EXPORTEDPORT;
			break;
		}
	}
	if (*netnumber == NOPNET)
	{
		pn = net_newpnet(pnetlist);
		if (pn == NOPNET) return(1);
		pn->network = forcenet;
		*netnumber = pn;
	}
	return(0);
}

/*
 * Routine to reduce the network in "pcomp" to merge parallel components.
 */
INTBIG net_mergeparallel(PCOMP **pcomp, INTBIG *components)
{
	REGISTER PCOMP *pc, *opc, *nextpc, *lastpc;
	REGISTER INTBIG i, j, newnum, mergecount, count;
	REGISTER NODEINST *ni, *oni, **newlist;

	lastpc = NOPCOMP;
	mergecount = 0;
	count = 0;
	for(pc = *pcomp; pc != NOPCOMP; pc = nextpc)
	{
		nextpc = pc->nextpcomp;
		if (pc->function == NPUNKNOWN) continue;

		if (((count++)%5) == 0)
		{
			if (stopping(STOPREASONNCC)) return(-1);
		}
		for(opc = pc->nextpcomp; opc != NOPCOMP; opc = opc->nextpcomp)
		{
			if (pc->function != opc->function) continue;

			/* both components must be in the same facet */
			if (pc->hierpathcount != opc->hierpathcount) continue;
			for(i=0; i<pc->hierpathcount; i++)
				if (pc->hierpath[i] != opc->hierpath[i]) break;
			if (i < pc->hierpathcount) continue;

			/* compare the wire lists */
			if (net_comparewirelist(pc, opc, 0) != 0) continue;

			/* components are equivalent: delete "pc" */
			mergecount++;
			net_removepcompfromlists(*pcomp, pc, opc);

			/* add "pc"s node pointer to "opc" */
			for(j=0; j<pc->numactual; j++)
			{
				if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
					ni = ((NODEINST **)pc->actuallist)[j];
				newnum = opc->numactual + 1;
				if (newnum == 1) opc->actuallist = (void *)ni; else
				{
					newlist = (NODEINST **)emalloc(newnum * (sizeof (NODEINST *)),
						net_tool->cluster);
					if (newlist == 0) return(0);
					for(i=0; i<opc->numactual; i++)
					{
						if (opc->numactual == 1) oni = (NODEINST *)opc->actuallist; else
							oni = ((NODEINST **)opc->actuallist)[i];
						newlist[i] = oni;
					}
					newlist[i] = ni;
					if (opc->numactual > 1) efree((char *)opc->actuallist);
					opc->actuallist = (void *)newlist;
				}
				opc->numactual = newnum;
			}

			/* combine sizes (as specified by Robert Bosnyak) */
			if ((pc->flags&(COMPHASWIDLEN|COMPHASAREA)) != 0 &&
				(opc->flags&(COMPHASWIDLEN|COMPHASAREA)) != 0)
			{
				switch (pc->function)
				{
					case NPTRANMOS:  case NPTRA4NMOS:
					case NPTRADMOS:  case NPTRA4DMOS:
					case NPTRAPMOS:  case NPTRA4PMOS:
						/* FET transistors in parallel depend on whether the length is the same */
						if (opc->length == pc->length)
						{
							/* same-length transistors: sum the width */
							opc->width += pc->width;
						} else
						{
							/* different-length transistors: more complex formula */
							if (pc->width + opc->width != 0.0)
							{
								opc->length = (pc->width * pc->length + opc->width * opc->length) /
									(pc->width + opc->width);
							}
							opc->width += pc->width;
						}
						break;
					case NPTRANPN:   case NPTRA4NPN:
					case NPTRAPNP:   case NPTRA4PNP:
					case NPTRANJFET: case NPTRA4NJFET:
					case NPTRAPJFET: case NPTRA4PJFET:
					case NPTRADMES:  case NPTRA4DMES:
					case NPTRAEMES:  case NPTRA4EMES:
						/* nonFET transistors in parallel sum the area */
						opc->length += pc->length;
						break;
					case NPRESIST:
					case NPINDUCT:
						/* resistance and capacitance in parallel take product over sum */
						if (pc->length + opc->length != 0.0)
							opc->length = (pc->length * opc->length) / (pc->length + opc->length);
						break;
					case NPCAPAC:  case NPECAPAC:
					case NPDIODE:  case NPDIODEZ:
						/* capacitance and diode in parallel sum the farads/area */
						opc->length += pc->length;
						break;
				}
			}

			/* unlink and free the duplicate component */
			if (lastpc == NOPCOMP) *pcomp = pc->nextpcomp; else
				lastpc->nextpcomp = pc->nextpcomp;
			net_freepcomp(pc);
			(*components)--;
			break;
		}
		if (opc != NOPCOMP) continue;
		lastpc = pc;
	}
	return(mergecount);
}

/*
 * Routine to add PCOMP "pc" to the list of serial transistors.
 */
INTSML net_addtoseriallist(PCOMP *pc)
{
	REGISTER INTBIG newtotal, i;
	REGISTER PCOMP **newlist;

	if (net_seriallistcount >= net_seriallisttotal)
	{
		newtotal = net_seriallisttotal * 2;
		if (newtotal <= net_seriallistcount)
			newtotal = net_seriallistcount+1;
		newlist = (PCOMP **)emalloc(newtotal * (sizeof (PCOMP *)), net_tool->cluster);
		if (newlist == 0) return(1);
		for(i=0; i<net_seriallistcount; i++)
			newlist[i] = net_seriallist[i];
		if (net_seriallisttotal > 0) efree((char *)net_seriallist);
		net_seriallist = newlist;
		net_seriallisttotal = newtotal;
	}
	net_seriallist[net_seriallistcount] = pc;
	net_seriallistcount++;
	return(0);
}

/*
 * Routine to gather a list of serial transistors that include "pc".  Returns
 * the list, and its length in "serialcount".  If "serialcount" is less than 2,
 * no chain of serial transistors has been found.
 */
PCOMP **net_gatherserial(PCOMP *pc, INTBIG *serialcount)
{
	net_seriallistcount = 0;
	net_serialcon1 = net_serialcon2 = -1;
	net_addtoseriallist(pc);
	net_crawlforserial(pc);
	*serialcount = net_seriallistcount;
	return(net_seriallist);
}

/*
 * Recursive helper routine for "net_gatherserial" to find transistors adjacent
 * to "pc" and add them to the serial chain if they are serial transistors.
 */
void net_crawlforserial(PCOMP *pc)
{
	REGISTER INTBIG i, j, badend;
	REGISTER PCOMP *opc;
	REGISTER PNET *pn;

	/* check source and drain connections */
	for(i=1; i<3; i++)
	{
		badend = 0;

		/* connection must be to exactly 1 other component */
		if (pc->count[i] != 1) badend = 1;

		/* other component must be the same as this */
		if (badend == 0)
		{
			opc = (pc->pconnarray[i])[0];
			if (opc->function != pc->function) badend = 1;
		}

		/* both components must be in the same facet */
		if (badend == 0)
		{
			if (pc->hierpathcount != opc->hierpathcount) continue;
			for(j=0; j<pc->hierpathcount; j++)
				if (pc->hierpath[j] != opc->hierpath[j]) break;
			if (j < pc->hierpathcount) badend = 1;
		}

		/* other component must point just to this on its source or drain */
		if (badend == 0)
		{
			pn = pc->netnumbers[i];
			if ((pn->flags&(EXPORTEDNET|POWERNET|GROUNDNET)) != 0) badend = 1;
		}

		/* network between components must not be exported */
		if (badend == 0)
		{
			for(j=1; j<3; j++)
			{
				if (opc->count[j] != 1) continue;
				if ((opc->pconnarray[j])[0] == pc) break;
			}
			if (j >= 3) badend = 1;
		}

		/* other component must not already be in the list */
		if (badend == 0)
		{
			for(j=0; j<net_seriallistcount; j++)
				if (net_seriallist[j] == opc) break;
			if (j < net_seriallistcount) badend = 2;
		}

		switch (badend)
		{
			case 0:		/* good end */
				/* another serial transistor found */
				net_addtoseriallist(opc);

				/* recursively search for others */
				net_crawlforserial(opc);
				break;
			case 1:		/* bad end: end of chain */
				/* add to the end list */
				if (net_serialcon1 < 0)
				{
					net_serialend1 = pc;
					net_serialcon1 = i;
				} else if (net_serialcon2 < 0)
				{
					net_serialend2 = pc;
					net_serialcon2 = i;
				}
				break;
		}
	}
}

/*
 * Routine to reduce the network in "pcomp" to merge serial transistors into more complex
 * single components.
 */
INTBIG net_mergeserial(PCOMP **pcomp, INTBIG *components)
{
	REGISTER PCOMP *pc, *opc, *oopc, *nextpc, *lastpc, **seriallist, *newpc, *npc;
	REGISTER INTBIG i, j, k, t, mergecount;
	INTBIG serialcount;
	REGISTER NODEINST *ni;

	/* clear flags on every component */
	for(pc = *pcomp; pc != NOPCOMP; pc = pc->nextpcomp) pc->timestamp = -1;

	/* scan for serial transistors */
	mergecount = 0;
	for(pc = *pcomp; pc != NOPCOMP; pc = nextpc)
	{
		nextpc = pc->nextpcomp;
		if (pc->function != NPTRANMOS && pc->function != NPTRADMOS &&
			pc->function != NPTRAPMOS) continue;

		/* look for adjacent single transistor */
		seriallist = net_gatherserial(pc, &serialcount);
		if (serialcount <= 1) continue;

		mergecount++;

		/* mark transistors */
		for(t=0; t<serialcount; t++) seriallist[t]->timestamp = t;

		/* create a new component with all the features of the gates and ends */
		newpc = net_allocpcomp();
		if (newpc == NOPCOMP) return(0);
		newpc->nextpcomp = *pcomp;
		*pcomp = newpc;
		newpc->topactual = pc->topactual;
		newpc->hierpathcount = pc->hierpathcount;
		if (newpc->hierpathcount > 0)
		{
			newpc->hierpath = (NODEINST **)emalloc(newpc->hierpathcount *
				(sizeof (NODEINST *)), net_tool->cluster);
			if (newpc->hierpath == 0) return(0);
			for(i=0; i<newpc->hierpathcount; i++)
				newpc->hierpath[i] = pc->hierpath[i];
		}
		newpc->flags = pc->flags;
		newpc->function = (INTSML)(pc->function * 1000 + serialcount);
		newpc->wirecount = serialcount + 2;
		newpc->timestamp = -1;
		newpc->hashreason = 0;

		/* length is the sum of all lengths, width is the average width */
		newpc->length = 0.0;
		newpc->width = 0.0;
		for(t=0; t<serialcount; t++)
		{
			newpc->length += seriallist[t]->length;
			newpc->width += seriallist[t]->width;
		}
		newpc->width /= serialcount;

		/* build "actual" list from all of the individual transistors */
		newpc->numactual = 0;
		for(t=0; t<serialcount; t++) newpc->numactual += seriallist[t]->numactual;
		newpc->actuallist = (NODEINST **)emalloc(newpc->numactual * (sizeof (NODEINST *)),
			net_tool->cluster);
		if (newpc->actuallist == 0) return(0);
		j = 0;
		for(t=0; t<serialcount; t++)
		{
			opc = seriallist[t];
			for(k=0; k < opc->numactual; k++)
			{
				if (opc->numactual == 1) ni = (NODEINST *)opc->actuallist; else
					ni = ((NODEINST **)opc->actuallist)[k];
				((NODEINST **)newpc->actuallist)[j++] = ni;
			}
		}

		/* allocate the pointer arrays */
		newpc->pconnarray = (PCOMP ***)emalloc(((sizeof (PCOMP **)) * newpc->wirecount),
			net_tool->cluster);
		if (newpc->pconnarray == 0) return(0);
		newpc->portindexarray = (INTSML **)emalloc(((sizeof (INTSML *)) * newpc->wirecount),
			net_tool->cluster);
		if (newpc->portindexarray == 0) return(0);
		newpc->portlist = (PORTPROTO **)emalloc(((sizeof (PORTPROTO *)) * newpc->wirecount),
			net_tool->cluster);
		if (newpc->portlist == 0) return(0);
		newpc->count = (INTSML *)emalloc((SIZEOFINTSML * newpc->wirecount),
			net_tool->cluster);
		if (newpc->count == 0) return(0);
		newpc->state = (INTSML *)emalloc((SIZEOFINTSML * newpc->wirecount),
			net_tool->cluster);
		if (newpc->state == 0) return(0);
		newpc->portindices = (INTSML *)emalloc((SIZEOFINTSML * newpc->wirecount),
			net_tool->cluster);
		if (newpc->portindices == 0) return(0);
		newpc->netnumbers = (PNET **)emalloc(((sizeof (PNET *)) * newpc->wirecount),
			net_tool->cluster);
		if (newpc->netnumbers == 0) return(0);

		/* pick up gates from all serial transistors */
		for(t=0; t<serialcount; t++)
		{
			newpc->portindices[t] = (INTSML)getprime(0);
			newpc->portlist[t] = seriallist[t]->portlist[0];
			newpc->netnumbers[t] = seriallist[t]->netnumbers[0];
			newpc->state[t] = seriallist[t]->state[0];
			newpc->count[t] = seriallist[t]->count[0];
			newpc->pconnarray[t] = (PCOMP **)emalloc(newpc->count[t] * (sizeof (PCOMP *)), net_tool->cluster);
			if (newpc->pconnarray[t] == 0) return(0);
			newpc->portindexarray[t] = (INTSML *)emalloc(newpc->count[t] * SIZEOFINTSML, net_tool->cluster);
			if (newpc->portindexarray[t] == 0) return(0);
			for(i=0; i<newpc->count[t]; i++)
			{
				newpc->pconnarray[t][i] = seriallist[t]->pconnarray[0][i];
				newpc->portindexarray[t][i] = seriallist[t]->portindexarray[0][i];
			}
		}

		/* add in new source and drain */
		newpc->portindices[t] = (INTSML)getprime(1);
		newpc->portlist[t] = net_serialend1->portlist[net_serialcon1];
		newpc->netnumbers[t] = net_serialend1->netnumbers[net_serialcon1];
		newpc->state[t] = net_serialend1->state[net_serialcon1];
		newpc->count[t] = net_serialend1->count[net_serialcon1];
		newpc->pconnarray[t] = (PCOMP **)emalloc(newpc->count[t] * (sizeof (PCOMP *)), net_tool->cluster);
		if (newpc->pconnarray[t] == 0) return(0);
		newpc->portindexarray[t] = (INTSML *)emalloc(newpc->count[t] * SIZEOFINTSML, net_tool->cluster);
		if (newpc->portindexarray[t] == 0) return(0);
		for(i=0; i<newpc->count[t]; i++)
		{
			newpc->pconnarray[t][i] = net_serialend1->pconnarray[net_serialcon1][i];
			newpc->portindexarray[t][i] = net_serialend1->portindexarray[net_serialcon1][i];
		}
		t++;
		newpc->portindices[t] = (INTSML)getprime(1);
		newpc->portlist[t] = net_serialend2->portlist[net_serialcon2];
		newpc->netnumbers[t] = net_serialend2->netnumbers[net_serialcon2];
		newpc->state[t] = net_serialend2->state[net_serialcon2];
		newpc->count[t] = net_serialend2->count[net_serialcon2];
		newpc->pconnarray[t] = (PCOMP **)emalloc(newpc->count[t] * (sizeof (PCOMP *)), net_tool->cluster);
		if (newpc->pconnarray[t] == 0) return(0);
		newpc->portindexarray[t] = (INTSML *)emalloc(newpc->count[t] * SIZEOFINTSML, net_tool->cluster);
		if (newpc->portindexarray[t] == 0) return(0);
		for(i=0; i<newpc->count[t]; i++)
		{
			newpc->pconnarray[t][i] = net_serialend2->pconnarray[net_serialcon2][i];
			newpc->portindexarray[t][i] = net_serialend2->portindexarray[net_serialcon2][i];
		}
		(*components)++;

		/* adjust pointers to the former transistors */
		for(opc = *pcomp; opc != NOPCOMP; opc = opc->nextpcomp)
		{
			/* ignore the removed ones */
			if (opc->timestamp != -1) continue;

			/* check every connection path from the component */
			for(i=0; i<opc->wirecount; i++)
			{
				/* look at every other component connected on this path */
				for(j=0; j<opc->count[i]; j++)
				{
					oopc = opc->pconnarray[i][j];
					if (oopc->timestamp == -1) continue;
					opc->pconnarray[i][j] = newpc;
				}
			}
		}

		/* fix pointer to the next transistor */
		while (nextpc != NOPCOMP && nextpc->timestamp != -1)
			nextpc = nextpc->nextpcomp;

		/* remove the serial transistors */
		lastpc = NOPCOMP;
		for(opc = *pcomp; opc != NOPCOMP; opc = npc)
		{
			npc = opc->nextpcomp;
			if (opc->timestamp != -1)
			{
				/* remove this */
				if (lastpc == NOPCOMP) *pcomp = npc; else
					lastpc->nextpcomp = npc;
				net_freepcomp(opc);
				(*components)--;
				continue;
			}
			lastpc = opc;
		}
	}
	return(mergecount);
}

/*
 * routine to compare pseudocomponent "p1" and "p2", returning nonzero if they
 * are different.
 */
INTSML net_comparewirelist(PCOMP *p1, PCOMP *p2, INTSML useportnames)
{
	REGISTER INTBIG i, j;

	/* simple test: number of wire lists must be equal */
	if (p1->wirecount != p2->wirecount) return(1);

	/* if ports must match in sequence, check is simpler */
	if (p1->function == NPTRANPN || p1->function == NPTRAPNP ||
		p1->function == NPDIODE || p1->function == NPDIODEZ ||
		p1->function == NPBUFFER || p1->function == NPFLIPFLOP ||
		p1->function == NPUNKNOWN)
	{
		for(i=0; i<p1->wirecount; i++)
		{
			if (p1->count[i] == 0) return(1);
			if (net_comparewires(p1, i, p2, i, useportnames) != 0)
				return(1);
		}
		return(0);
	}

	/* make sure there is memory for flags corresponding to component 2 */
	if (p2->wirecount > net_comparelistsize)
	{
		if (net_comparelistsize != 0) efree((char *)net_comparelist);
		net_comparelist = (INTSML *)emalloc((SIZEOFINTSML * p2->wirecount), net_tool->cluster);
		net_comparelistsize = p2->wirecount;
	}

	/* reset flags in list for component 2 */
	for(j=0; j<p2->wirecount; j++) net_comparelist[j] = 0;

	for(i=0; i<p1->wirecount; i++)
	{
		if (p1->count[i] == 0) return(1);
		for(j=0; j<p2->wirecount; j++)
		{
			if (net_comparelist[j] != 0) continue;
			if (p1->portindices[i] != p2->portindices[j]) continue;
			if (net_comparewires(p1, i, p2, j, useportnames) != 0) continue;
			net_comparelist[j] = 1;
			break;
		}
		if (j >= p2->wirecount) return(1);
	}
	return(0);
}

/*
 * routine to compare the entry "ind1" of component "p1" and
 * entry "ind2" of component "p2", returning nonzero if they are different.
 */
INTSML net_comparewires(PCOMP *p1, INTBIG ind1, PCOMP *p2, INTBIG ind2,
	INTSML useportnames)
{
	REGISTER INTBIG i1, i2, wirecount, f1, f2;
	REGISTER PNET *n1, *n2, *pnet1, *pnet2;
	REGISTER NETWORK *net1, *net2;

	/* simple test: number of wire lists must be equal */
	wirecount = p1->count[ind1];
	if (wirecount != p2->count[ind2]) return(1);

	/* simple test: negated factor must be equal */
	f1 = p1->state[ind1];
	f2 = p2->state[ind2];
	if ((f1&NEGATEDPORT) != (f2&NEGATEDPORT)) return(1);

	/* simple test: if an export, names must match */
	pnet1 = p1->netnumbers[ind1];
	pnet2 = p2->netnumbers[ind2];
	if (useportnames != 0)
	{
		if ((f1&EXPORTEDPORT) != 0 && (f2&EXPORTEDPORT) != 0)
		{
			net1 = pnet1->network;
			net2 = pnet2->network;
			if (net1 != NONETWORK && net1->namecount > 0 &&
				net2 != NONETWORK && net2->namecount > 0)
			{
				if (net_samenetworkname(net1, net2) == 0) return(1);
			}
		}
	}

	/* make sure there is memory for flags */
	if (wirecount > net_comparelistwsize)
	{
		if (net_comparelistwsize != 0) efree((char *)net_comparelistw);
		net_comparelistw = (INTSML *)emalloc((SIZEOFINTSML * wirecount), net_tool->cluster);
		net_comparelistwsize = wirecount;
	}

	/* erase flags */
	for(i2=0; i2<wirecount; i2++) net_comparelistw[i2] = 0;

	/* examine every element in list 1 */
	for(i1 = 0; i1 < wirecount; i1++)
	{
		n1 = &pnet1[i1];

		/* find the equivalent in list 2 */
		for(i2 = 0; i2 < wirecount; i2++)
		{
			if (net_comparelistw[i2] != 0) continue;
			n2 = &pnet2[i2];
			if (n1 != n2) continue;
			net_comparelistw[i2] = 1;
			break;
		}
		if (i2 >= wirecount) return(1);
	}
	return(0);
}

void net_removepcompfromlists(PCOMP *pcomp, PCOMP *removedpc, PCOMP *newpc)
{
	REGISTER PCOMP *pc;
	REGISTER INTBIG i, j, k;

	/* look through every component */
	for(pc = pcomp; pc != NOPCOMP; pc = pc->nextpcomp)
	{
		/* ignore the removed one */
		if (pc == removedpc) continue;

		/* check every connection path from the component */
		for(i=0; i<pc->wirecount; i++)
		{
			/* look at every other component connected on this path */
			for(j=0; j<pc->count[i]; j++)
			{
				if (pc->pconnarray[i][j] != removedpc) continue;

				/* found reference to old component */
				for(k=0; k<pc->count[i]; k++)
					if (pc->pconnarray[i][k] == newpc) break;
				if (k >= pc->count[i] && newpc != pc)
				{
					/* new component not referenced: substitute */
					pc->pconnarray[i][j] = newpc;
				} else
				{
					/* new component is referenced: remove old */
					for(k = j; k < pc->count[i]-1; k++)
					{
						pc->pconnarray[i][k] = pc->pconnarray[i][k+1];
						pc->portindexarray[i][k] = pc->portindexarray[i][k+1];
					}
					pc->count[i]--;
				}
				break;
			}
		}
	}
}

float net_getpartvalue(NODEINST *ni)
{
	REGISTER NODEPROTO *np;
	REGISTER VARIABLE *var;
	REGISTER INTBIG i;
	float value;
	REGISTER char *pt;

	np = ni->proto;
	if (np->primindex == 0) return(0.0);

	/* diodes have area on them */
	if (np == sch_diodeprim)
	{
		var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_diodekey);
		pt = describesimplevariable(var);
		return((float)atof(pt));
	}

	/* capacitors have Farads on them */
	if (np == sch_capacitorprim)
	{
		var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_capacitancekey);
		pt = describesimplevariable(var);
		value = (float)atof(pt);
		i = strlen(pt);
		if (i > 0)
		{
			if (tolower(pt[i-1]) == 'f')
			{
				value = value / 1000000000000.0f;
			} else if (tolower(pt[i-1]) == 'p')
			{
				value = value / 1000000000.0f;
			} else if (tolower(pt[i-1]) == 'u')
			{
				value = value / 1000000.0f;
			} else if (tolower(pt[i-1]) == 'm')
			{
				value = value / 1000.0f;
			}
		}
		return(value);
	}

	/* resistors have Ohms on them */
	if (np == sch_resistorprim)
	{
		var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_resistancekey);
		pt = describesimplevariable(var);
		value = (float)atof(pt);
		i = strlen(pt);
		if (i > 0)
		{
			if (tolower(pt[i-1]) == 'g')
			{
				value = value * 1000000000.0f;
			} else if (i > 2 && namesame(&pt[i-3], "meg") == 0)
			{
				value = value * 1000000.0f;
			} else if (tolower(pt[i-1]) == 'k')
			{
				value = value * 1000.0f;
			}
		}
		return(value);
	}

	/* inductors have Henrys on them */
	if (np == sch_inductorprim)
	{
		var = getvalkey((INTBIG)ni, VNODEINST, -1, sch_inductancekey);
		pt = describesimplevariable(var);
		value = (float)atof(pt);
		i = strlen(pt);
		if (i > 0)
		{
			if (tolower(pt[i-1]) == 'u')
			{
				value = value / 1000000.0f;
			} else if (tolower(pt[i-1]) == 'm')
			{
				value = value / 1000.0f;
			}
		}
		return(value);
	}
	return(0.0);
}

/*********************** ALLOCATION ***********************/

/*
 * routine to create a new PNET module and increase the net number count in the
 * global "net_pseudonode", add the module to the list in "pnetlist", and
 * return the module.  Returns NOPNET on error.
 */
PNET *net_newpnet(PNET **pnetlist)
{
	REGISTER PNET *pn;

	pn = net_allocpnet();
	if (pn == 0) return(NOPNET);
	net_pseudonode++;
	pn->nextpnet = *pnetlist;
	*pnetlist = pn;
	return(pn);
}

/*
 * routine to allocate a new pcomp module from the pool (if any) or memory
 */
PCOMP *net_allocpcomp(void)
{
	REGISTER PCOMP *pc;

	if (net_pcompfree == NOPCOMP)
	{
		pc = (PCOMP *)emalloc(sizeof (PCOMP), net_tool->cluster);
		if (pc == 0) return(NOPCOMP);
	} else
	{
		pc = net_pcompfree;
		net_pcompfree = (PCOMP *)pc->nextpcomp;
	}
	pc->flags = 0;
	return(pc);
}

/*
 * routine to return pcomp module "pc" to the pool of free modules
 */
void net_freepcomp(PCOMP *pc)
{
	REGISTER INTBIG i;

	for(i=0; i<pc->wirecount; i++)
	{
		if (pc->count[i] > 0)
		{
			efree((char *)pc->pconnarray[i]);
			efree((char *)pc->portindexarray[i]);
		}
	}
	if (pc->wirecount != 0)
	{
		efree((char *)pc->pconnarray);
		efree((char *)pc->portindexarray);
		efree((char *)pc->portlist);
		efree((char *)pc->count);
		efree((char *)pc->state);
		efree((char *)pc->netnumbers);
		efree((char *)pc->portindices);
	}
	if (pc->numactual > 1) efree((char *)pc->actuallist);
	if (pc->hashreason != 0) efree((char *)pc->hashreason);
	if (pc->hierpathcount != 0) efree((char *)pc->hierpath);

	pc->nextpcomp = net_pcompfree;
	net_pcompfree = pc;
}

/*
 * routine to allocate a new pnet module from the pool (if any) or memory
 */
PNET *net_allocpnet(void)
{
	REGISTER PNET *pn;

	if (net_pnetfree == NOPNET)
	{
		pn = (PNET *)emalloc(sizeof (PNET), net_tool->cluster);
		if (pn == 0) return(NOPNET);
	} else
	{
		pn = net_pnetfree;
		net_pnetfree = (PNET *)pn->nextpnet;
	}

	pn->flags = 0;
	pn->nodecount = 0;
	pn->realportcount = 0;
	return(pn);
}

/*
 * routine to return pnet module "pn" to the pool of free modules
 */
void net_freepnet(PNET *pn)
{
	if (pn->nodecount > 0)
	{
		efree((char *)pn->nodelist);
		efree((char *)pn->nodewire);
	}
	if (pn->realportcount > 1)
		efree((char *)pn->realportlist);

	pn->nextpnet = net_pnetfree;
	net_pnetfree = pn;
}

/*
 * routine to free all allocated structures in the list of pseudonets
 * headed by "pnlist"
 */
void net_freeallpnet(PNET *pnlist)
{
	REGISTER PNET *pn, *nextpn;

	for(pn = pnlist; pn != NOPNET; pn = nextpn)
	{
		nextpn = pn->nextpnet;
		net_freepnet(pn);
	}
}

/*
 * routine to free all allocated structures in the list of pseudocomponents
 * headed by "pclist"
 */
void net_freeallpcomp(PCOMP *pclist)
{
	REGISTER PCOMP *pc, *nextpc;

	for(pc = pclist; pc != NOPCOMP; pc = nextpc)
	{
		nextpc = pc->nextpcomp;
		net_freepcomp(pc);
	}
}

/*********************** COMPONENT FUNCTION ***********************/

#define NOTRANMODEL ((TRANMODEL *)-1)

typedef struct Itranmodel
{
	char *modelname;
	INTSML tmindex;
	struct Itranmodel *nexttranmodel;
} TRANMODEL;

static TRANMODEL *net_firsttranmodel = NOTRANMODEL;

/* must be larger than largest node function entry in "efunction.h" */
static INTSML net_tranmodelindex = 100;

/*
 * routine to return the function of node "ni"
 */
INTBIG net_getfunction(NODEINST *ni)
{
	REGISTER INTBIG fun;
	REGISTER PORTEXPINST *pe;
	REGISTER TRANMODEL *tm;

	fun = nodefunction(ni);
	switch (fun)
	{
		case NPTRANS:
			fun = NPTRANMOS;
			break;

		case NPTRANS4:
#ifdef IGNOREFOURPORT
			fun = NPTRANMOS;
#else
			fun = NPTRA4NMOS;
#endif
			break;

#ifdef IGNOREFOURPORT
		case NPTRA4NMOS:  fun = NPTRANMOS;   break;
		case NPTRA4DMOS:  fun = NPTRADMOS;   break;
		case NPTRA4PMOS:  fun = NPTRAPMOS;   break;
		case NPTRA4NJFET: fun = NPTRANJFET;  break;
		case NPTRA4PJFET: fun = NPTRAPJFET;  break;
		case NPTRA4DMES:  fun = NPTRADMES;   break;
		case NPTRA4EMES:  fun = NPTRAEMES;   break;
		case NPTRA4NPN:   fun = NPTRANPN;    break;
		case NPTRA4PNP:   fun = NPTRAPNP;    break;
#endif

		case NPTRANSREF:
			/* self-referential transistor: lookup the string in the table */
			for(tm = net_firsttranmodel; tm != NOTRANMODEL; tm = tm->nexttranmodel)
				if (namesame(tm->modelname, ni->proto->primname) == 0) break;
			if (tm == NOTRANMODEL)
			{
				/* new table entry */
				tm = (TRANMODEL *)emalloc(sizeof (TRANMODEL), net_tool->cluster);
				if (tm == 0) break;
				(void)allocstring(&tm->modelname, ni->proto->primname, net_tool->cluster);
				tm->tmindex = net_tranmodelindex++;
				tm->nexttranmodel = net_firsttranmodel;
				net_firsttranmodel = tm;
			}
			fun = tm->tmindex;
			break;

		case NPPIN:
		case NPNODE:
		case NPCONTACT:
		case NPWELL:
		case NPSUBSTRATE:
			fun = NPCONNECT;
			/* FALLTHROUGH */ 

		case NPCONNECT:
			/* unify the representations of power and ground */
			for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
			{
				if (portispower(pe->exportproto) != 0) fun = NPCONPOWER; else
					if (portisground(pe->exportproto) != 0) fun = NPCONGROUND;
			}
			break;
	}
	return(fun);
}

void net_setallexporttopology(void)
{
	REGISTER LIBRARY *lib;
	REGISTER NODEPROTO *np;
	INTBIG i;
	REGISTER PORTPROTO *pp;
	REGISTER TECHNOLOGY *tech;
	REGISTER NETWORK *net;

	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
	{
		/* clear all topology information */
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			for(net = np->firstnetwork; net != NONETWORK; net = net->nextnetwork)
				net->temp2 = 0;
		}

		/* assign topology information uniformly in each cell */
		i = 0;
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			{
				if (pp->network->temp2 != 0) continue;
				net_setthisexporttopology(pp, &i);
			}
			for(net = np->firstnetwork; net != NONETWORK; net = net->nextnetwork)
			{
				if (net->temp2 == 0)
					net->temp2 = getprime(i++);
			}
		}
	}
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
	{
		for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			switch ((np->userbits&NFUNCTION) >> NFUNCTIONSH)
			{
				case NPTRANMOS:  case NPTRADMOS: case NPTRAPMOS:
				case NPTRANJFET: case NPTRAPJFET:
				case NPTRADMES:  case NPTRAEMES:
					pp = np->firstportproto;   pp->network->temp2 = getprime(0);	/* poly */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(1);	/* active */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(0);	/* poly */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(1);	/* active */
					break;
				case NPTRANS:
					pp = np->firstportproto;   pp->network->temp2 = getprime(0);	/* poly */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(1);	/* active */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(1);	/* active */
					break;
				case NPTRANS4:
					pp = np->firstportproto;   pp->network->temp2 = getprime(0);	/* poly */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(1);	/* active */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(1);	/* active */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(2);	/* bias */
					break;
				case NPTRANPN:   case NPTRAPNP:
					pp = np->firstportproto;   pp->network->temp2 = getprime(0);	/* collector */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(1);	/* emitter */
					pp = pp->nextportproto;    pp->network->temp2 = getprime(2);	/* base */
					break;
				default:
					i = 0;
					for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					{
						pp->network->temp2 = getprime(i);
						i++;
					}
					break;
			}
		}
	}
}

void net_setthisexporttopology(PORTPROTO *pp, INTBIG *index)
{
	REGISTER NETWORK *net, *subnet;
	REGISTER INTBIG i;
	REGISTER char *name;

	/* assign the unique prime numbers to the export's network */
	net = pp->network;
	if (net->temp2 == 0)
	{
		net->temp2 = net_findotherexporttopology(pp->parent, pp->protoname);
		if (net->temp2 == 0) net->temp2 = getprime((*index)++);
	}
	if (net->signals > 1)
	{
		for(i=0; i<net->signals; i++)
		{
			subnet = net->networklist[i];
			if (subnet->temp2 == 0)
			{
				if (subnet->namecount > 0) name = subnet->netname; else
					name = pp->protoname;
				subnet->temp2 = net_findotherexporttopology(pp->parent, name);
				if (subnet->temp2 == 0) subnet->temp2 = getprime((*index)++);
			}
		}
	}
}

INTBIG net_findotherexporttopology(NODEPROTO *parent, char *exportname)
{
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *opp;
	REGISTER NETWORK *net;

	/* first look for an equivalent export in another facet */
	for(np = parent->cell->lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (np == parent) continue;
		if (np->cell != parent->cell) continue;

		/* facet from the same cell: look for an equivalent port */
		for(opp = np->firstportproto; opp != NOPORTPROTO; opp = opp->nextportproto)
		{
			if (namesame(exportname, opp->protoname) != 0) continue;
			if (opp->network->temp2 != 0) return(opp->network->temp2);
		}
	}

	/* next look for an equivalent network name in another facet */
	for(np = parent->cell->lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (np == parent) continue;
		if (np->cell != parent->cell) continue;

		/* facet from the same cell: look for an equivalent network name */
		net = getnetwork(exportname, np);
		if (net == NONETWORK) continue;
		if (net->temp2 != 0) return(net->temp2);
	}
	return(0);
}
