package Data::FR::Town;

use 5.006;
use strict;
use warnings;
use Carp;

=head1 NAME

Data::FR::Town - Provides data about french town (INSEE code, ZIP/postal code, name...)

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.02';

our %TOWNS;

=head1 SYNOPSIS

This module provides basic data about french towns.

    use Data::FR::Town;

    # Multiple ways to get the same town
    my $town1 = Data::FR::Town->new( {insee => "01001"});
    my $town2 = Data::FR::Town->new( {zip => "01400"});
    my $town3 = Data::FR::Town->find( {insee => "01001"});
    my $town3 = Data::FR::Town->find( {zip => "01400"});

    Now you can get data about the selected town, using the getters :

    my $zip = $town1->zip();
    my $name = $town1->name();
    my $cname = $town1->cname();
    ...

=head1 SUBROUTINES/METHODS

=head2 new ( \%params )

Return a town object selected through the specified parameters.

%params accept two valid keys : 'insee' and 'zip' to specfy the town.

=cut

sub new {
    my $class  = shift;
    my $params = shift;

    my $self = bless {}, $class;

    for my $line (<DATA>) {
        chomp $line;

        my ( $insee, $article, $name, $cname, $zip, $dep, $depname, $origin ) = split ';', $line;

        $class::TOWNS{insee}{$insee}{article} = $article;
        $class::TOWNS{insee}{$insee}{name}    = $name;
        $class::TOWNS{insee}{$insee}{cname}   = $cname;
        $class::TOWNS{insee}{$insee}{zip}     = $zip;
        $class::TOWNS{insee}{$insee}{dep}     = $dep;
        $class::TOWNS{insee}{$insee}{depname} = $depname;
        $class::TOWNS{insee}{$insee}{origin}  = $origin;
        $class::TOWNS{zip}{$zip}              = $class::TOWNS->{insee}{$insee};

    }

    for my $param ( keys %{$params} ) {
        if ( $param =~ /insee/ ) {
            return $class->find( { insee => $params->{$param} } );
        } elsif ( $param =~ /zip/ ) {
            return $class->find( { zip => $params->{$param} } );

        } else {
            croak "Unknown parameter : $param";
        }
    }

    return bless $self, $class;
}

=head2 find ( \%params )

Return a town object selected through the specified parameters.

%params accept two valid keys : 'insee' and 'zip' to specfy the town.

=cut

sub find {
    my $class  = shift;
    my $params = shift;

    $class = ref $class if ref $class;

    for my $param ( keys %$params ) {
        if ( $class::TOWNS{$param}{ $params->{$param} } ) {
            return bless $class::TOWNS{$param}{ $params->{$param} }, $class;
        } else {
            croak "Unknown parameter : $param" unless $param =~ /(zip|insee)/;
            return undef;
        }
    }
}

=head2 insee ()

Return the INSEE code associated to this town

=cut

sub insee {
    my $self = shift;

    return $self->{insee};
}

=head2 zip ()

Return the zip/postal code associated to this town

=cut

sub zip {
    my $self = shift;

    return $self->{zip};
}

=head2 article ()

Return the article associated to this town, as some french town may have one.
(La Courneuve)

=cut

sub article {
    my $self = shift;

    return $self->{article};
}

=head2 name ()

Return the name of the town
The first letter is uppercased the other ones are lowercase

=cut

sub name {
    my $self = shift;

    return $self->{name};
}

=head2 cname ()

Return the capitalized name of the town
All letters are (non accentuated) uppercase.

=cut

sub cname {
    my $self = shift;

    return $self->{cname};
}

=head2 dep ()

Return the department's number

=cut

sub dep {
    my $self = shift;

    return $self->{dep};
}

=head2 depname ()

Return the department's name

=cut

sub depname {
    my $self = shift;

    return $self->{depname};
}

=head1 DISCLAIMER

This version is based on the data from the INSEE site :
http://www.insee.fr/fr/methodes/nomenclatures/cog/telechargement/2012/txt/comsimp2012.zip

As this data is free and quite stable, the information about the INSEE
code, name are quite accurate.

The information about the postal/zip code is however much more difficult to
get AND subject to change.

I'll try to keep the postal/zip codes up to date, but anyone knowing a way to get
automatically these data could greatly contribute by telling it (or providing the data)

Don't hesitate to contact me on this subject.


Arnaud (Arhuman) ASSAD, C<< <arhuman at gmail.com> >>

=head1 AUTHOR

Arnaud (Arhuman) ASSAD, C<< <arhuman at gmail.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-data-fr-town at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Data-FR-Town>.  I will be notified, and then you'll

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Data::FR::Town


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Data-FR-Town>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Data-FR-Town>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Data-FR-Town>

=item * Search CPAN

L<http://search.cpan.org/dist/Data-FR-Town/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2012 Arnaud (Arhuman) ASSAD.

This program is free software; you can redistribute it and/or modify it
under the terms of either: the GNU General Public License as published
by the Free Software Foundation; or the Artistic License.

See http://dev.perl.org/licenses/ for more information.


=cut

1;    # End of Data::FR::Town

__DATA__
01001;L';Abergement-Clmenciat;ABERGEMENT-CLEMENCIAT;01400;01;AIN;1
01002;L';Abergement-de-Varey;ABERGEMENT-DE-VAREY;01640;01;AIN;1
01004;;Ambrieu-en-Bugey;AMBERIEU-EN-BUGEY;01500;01;AIN;1
01005;;Ambrieux-en-Dombes;AMBERIEUX-EN-DOMBES;01330;01;AIN;1
01006;;Amblon;AMBLEON;01300;01;AIN;1
01007;;Ambronay;AMBRONAY;01500;01;AIN;1
01008;;Ambutrix;AMBUTRIX;01500;01;AIN;1
01009;;Andert-et-Condon;ANDERT-ET-CONDON;01300;01;AIN;1
01010;;Anglefort;ANGLEFORT;01350;01;AIN;1
01011;;Apremont;APREMONT;01100;01;AIN;1
01012;;Aranc;ARANC;01110;01;AIN;1
01013;;Arandas;ARANDAS;01230;01;AIN;1
01014;;Arbent;ARBENT;01100;01;AIN;1
01015;;Arbignieu;ARBIGNIEU;01300;01;AIN;1
01016;;Arbigny;ARBIGNY;01190;01;AIN;1
01017;;Argis;ARGIS;01230;01;AIN;1
01019;;Armix;ARMIX;01510;01;AIN;1
01021;;Ars-sur-Formans;ARS-SUR-FORMANS;01480;01;AIN;1
01022;;Artemare;ARTEMARE;01510;01;AIN;1
01023;;Asnires-sur-Sane;ASNIERES-SUR-SAONE;01570;01;AIN;1
01024;;Attignat;ATTIGNAT;01340;01;AIN;1
01025;;Bg-la-Ville;BAGE-LA-VILLE;01380;01;AIN;1
01026;;Bg-le-Chtel;BAGE-LE-CHATEL;01380;01;AIN;1
01027;;Balan;BALAN;01360;01;AIN;1
01028;;Baneins;BANEINS;01990;01;AIN;1
01029;;Beaupont;BEAUPONT;01270;01;AIN;1
01030;;Beauregard;BEAUREGARD;01480;01;AIN;1
01031;;Bellignat;BELLIGNAT;01810;01;AIN;1
01032;;Bligneux;BELIGNEUX;01360;01;AIN;1
01033;;Bellegarde-sur-Valserine;BELLEGARDE-SUR-VALSERINE;01200;01;AIN;1
01034;;Belley;BELLEY;01300;01;AIN;1
01035;;Belleydoux;BELLEYDOUX;01130;01;AIN;1
01036;;Belmont-Luthzieu;BELMONT-LUTHEZIEU;01260;01;AIN;1
01037;;Bnonces;BENONCES;01470;01;AIN;1
01038;;Bny;BENY;01370;01;AIN;1
01039;;Bon;BEON;01350;01;AIN;1
01040;;Brziat;BEREZIAT;01340;01;AIN;1
01041;;Bettant;BETTANT;01500;01;AIN;1
01042;;Bey;BEY;01290;01;AIN;1
01043;;Beynost;BEYNOST;01700;01;AIN;1
01044;;Billiat;BILLIAT;01200;01;AIN;1
01045;;Birieux;BIRIEUX;01330;01;AIN;1
01046;;Biziat;BIZIAT;01290;01;AIN;1
01047;;Blyes;BLYES;01150;01;AIN;1
01049;La;Boisse;BOISSE;01120;01;AIN;1
01050;;Boissey;BOISSEY;01190;01;AIN;1
01051;;Bolozon;BOLOZON;01450;01;AIN;1
01052;;Bouligneux;BOULIGNEUX;01330;01;AIN;1
01053;;Bourg-en-Bresse;BOURG-EN-BRESSE;01000;01;AIN;1
01054;;Bourg-Saint-Christophe;BOURG-SAINT-CHRISTOPHE;01800;01;AIN;1
01056;;Boyeux-Saint-Jrme;BOYEUX-SAINT-JEROME;01640;01;AIN;1
01057;;Boz;BOZ;01190;01;AIN;1
01058;;Brgnier-Cordon;BREGNIER-CORDON;01300;01;AIN;1
01059;;Brnaz;BRENAZ;01260;01;AIN;1
01060;;Brnod;BRENOD;01110;01;AIN;1
01061;;Brens;BRENS;01300;01;AIN;1
01062;;Bressolles;BRESSOLLES;01360;01;AIN;1
01063;;Brion;BRION;01460;01;AIN;1
01064;;Briord;BRIORD;01470;01;AIN;1
01065;;Buellas;BUELLAS;01310;01;AIN;1
01066;La;Burbanche;BURBANCHE;01510;01;AIN;1
01067;;Ceignes;CEIGNES;01430;01;AIN;1
01068;;Cerdon;CERDON;01450;01;AIN;1
01069;;Certines;CERTINES;01240;01;AIN;1
01071;;Cessy;CESSY;01170;01;AIN;1
01072;;Ceyzriat;CEYZERIAT;01250;01;AIN;1
01073;;Ceyzrieu;CEYZERIEU;01350;01;AIN;1
01074;;Chalamont;CHALAMONT;01320;01;AIN;1
01075;;Chaleins;CHALEINS;01480;01;AIN;1
01076;;Chaley;CHALEY;01230;01;AIN;1
01077;;Challes-la-Montagne;CHALLES-LA-MONTAGNE;01450;01;AIN;1
01078;;Challex;CHALLEX;01630;01;AIN;1
01079;;Champagne-en-Valromey;CHAMPAGNE-EN-VALROMEY;01260;01;AIN;1
01080;;Champdor;CHAMPDOR;01110;01;AIN;1
01081;;Champfromier;CHAMPFROMIER;01410;01;AIN;1
01082;;Chanay;CHANAY;01420;01;AIN;1
01083;;Chaneins;CHANEINS;01990;01;AIN;1
01084;;Chanoz-Chtenay;CHANOZ-CHATENAY;01400;01;AIN;1
01085;La;Chapelle-du-Chtelard;CHAPELLE-DU-CHATELARD;01240;01;AIN;1
01087;;Charix;CHARIX;01130;01;AIN;1
01088;;Charnoz-sur-Ain;CHARNOZ-SUR-AIN;01800;01;AIN;1
01089;;Chteau-Gaillard;CHATEAU-GAILLARD;01500;01;AIN;1
01090;;Chtenay;CHATENAY;01320;01;AIN;1
01091;;Chtillon-en-Michaille;CHATILLON-EN-MICHAILLE;01200;01;AIN;1
01092;;Chtillon-la-Palud;CHATILLON-LA-PALUD;01320;01;AIN;1
01093;;Chtillon-sur-Chalaronne;CHATILLON-SUR-CHALARONNE;01400;01;AIN;1
01094;;Chavannes-sur-Reyssouze;CHAVANNES-SUR-REYSSOUZE;01190;01;AIN;1
01095;;Chavannes-sur-Suran;CHAVANNES-SUR-SURAN;01250;01;AIN;1
01096;;Chaveyriat;CHAVEYRIAT;01660;01;AIN;1
01097;;Chavornay;CHAVORNAY;01510;01;AIN;1
01098;;Chazey-Bons;CHAZEY-BONS;01300;01;AIN;1
01099;;Chazey-sur-Ain;CHAZEY-SUR-AIN;01150;01;AIN;1
01100;;Cheignieu-la-Balme;CHEIGNIEU-LA-BALME;01510;01;AIN;1
01101;;Chevillard;CHEVILLARD;01430;01;AIN;1
01102;;Chevroux;CHEVROUX;01190;01;AIN;1
01103;;Chevry;CHEVRY;01170;01;AIN;1
01104;;Chzery-Forens;CHEZERY-FORENS;01200;01;AIN;1
01105;;Civrieux;CIVRIEUX;01390;01;AIN;1
01106;;Cize;CIZE;01250;01;AIN;1
01107;;Cleyzieu;CLEYZIEU;01230;01;AIN;1
01108;;Coligny;COLIGNY;01270;01;AIN;1
01109;;Collonges;COLLONGES;01550;01;AIN;1
01110;;Colomieu;COLOMIEU;01300;01;AIN;1
01111;;Conand;CONAND;01230;01;AIN;1
01112;;Condamine;CONDAMINE;01430;01;AIN;1
01113;;Condeissiat;CONDEISSIAT;01400;01;AIN;1
01114;;Confort;CONFORT;01200;01;AIN;1
01115;;Confranon;CONFRANCON;01310;01;AIN;1
01116;;Contrevoz;CONTREVOZ;01300;01;AIN;1
01117;;Conzieu;CONZIEU;01300;01;AIN;1
01118;;Corbonod;CORBONOD;01420;01;AIN;1
01119;;Corcelles;CORCELLES;01110;01;AIN;1
01121;;Corlier;CORLIER;01110;01;AIN;1
01122;;Cormaranche-en-Bugey;CORMARANCHE-EN-BUGEY;01110;01;AIN;1
01123;;Cormoranche-sur-Sane;CORMORANCHE-SUR-SAONE;01290;01;AIN;1
01124;;Cormoz;CORMOZ;01560;01;AIN;1
01125;;Corveissiat;CORVEISSIAT;01250;01;AIN;1
01127;;Courmangoux;COURMANGOUX;01370;01;AIN;1
01128;;Courtes;COURTES;01560;01;AIN;1
01129;;Crans;CRANS;01320;01;AIN;1
01130;;Cras-sur-Reyssouze;CRAS-SUR-REYSSOUZE;01340;01;AIN;1
01133;;Cressin-Rochefort;CRESSIN-ROCHEFORT;01350;01;AIN;1
01134;;Crottet;CROTTET;01620;01;AIN;1
01135;;Crozet;CROZET;01170;01;AIN;1
01136;;Cruzilles-ls-Mpillat;CRUZILLES-LES-MEPILLAT;01290;01;AIN;1
01138;;Culoz;CULOZ;01350;01;AIN;1
01139;;Curciat-Dongalon;CURCIAT-DONGALON;01560;01;AIN;1
01140;;Curtafond;CURTAFOND;01310;01;AIN;1
01141;;Cuzieu;CUZIEU;01300;01;AIN;1
01142;;Dagneux;DAGNEUX;01120;01;AIN;1
01143;;Divonne-les-Bains;DIVONNE-LES-BAINS;01220;01;AIN;1
01144;;Dommartin;DOMMARTIN;01380;01;AIN;1
01145;;Dompierre-sur-Veyle;DOMPIERRE-SUR-VEYLE;01240;01;AIN;1
01146;;Dompierre-sur-Chalaronne;DOMPIERRE-SUR-CHALARONNE;01400;01;AIN;1
01147;;Domsure;DOMSURE;01270;01;AIN;1
01148;;Dortan;DORTAN;01590;01;AIN;1
01149;;Douvres;DOUVRES;01500;01;AIN;1
01150;;Drom;DROM;01250;01;AIN;1
01151;;Druillat;DRUILLAT;01160;01;AIN;1
01152;;challon;ECHALLON;01130;01;AIN;1
01153;;chenevex;ECHENEVEX;01170;01;AIN;1
01154;;trez;ETREZ;01340;01;AIN;1
01155;;vosges;EVOSGES;01230;01;AIN;1
01156;;Faramans;FARAMANS;01800;01;AIN;1
01157;;Fareins;FAREINS;01480;01;AIN;1
01158;;Farges;FARGES;01550;01;AIN;1
01159;;Feillens;FEILLENS;01570;01;AIN;1
01160;;Ferney-Voltaire;FERNEY-VOLTAIRE;01210;01;AIN;1
01162;;Flaxieu;FLAXIEU;01350;01;AIN;1
01163;;Foissiat;FOISSIAT;01340;01;AIN;1
01165;;Francheleins;FRANCHELEINS;01090;01;AIN;1
01166;;Frans;FRANS;01480;01;AIN;1
01167;;Garnerans;GARNERANS;01140;01;AIN;1
01169;;Genouilleux;GENOUILLEUX;01090;01;AIN;1
01170;;Bard-Govreissiat;BEARD-GEOVREISSIAT;01460;01;AIN;1
01171;;Govreisset;GEOVREISSET;01100;01;AIN;1
01172;;Germagnat;GERMAGNAT;01250;01;AIN;1
01173;;Gex;GEX;01170;01;AIN;1
01174;;Giron;GIRON;01130;01;AIN;1
01175;;Gorrevod;GORREVOD;01190;01;AIN;1
01176;Le;Grand-Abergement;GRAND-ABERGEMENT;01260;01;AIN;1
01177;;Grand-Corent;GRAND-CORENT;01250;01;AIN;1
01179;;Griges;GRIEGES;01290;01;AIN;1
01180;;Grilly;GRILLY;01220;01;AIN;1
01181;;Groissiat;GROISSIAT;01810;01;AIN;1
01182;;Grosle;GROSLEE;01680;01;AIN;1
01183;;Gureins;GUEREINS;01090;01;AIN;1
01184;;Hautecourt-Romanche;HAUTECOURT-ROMANECHE;01250;01;AIN;1
01185;;Hauteville-Lompnes;HAUTEVILLE-LOMPNES;01110;01;AIN;1
01186;;Hostiaz;HOSTIAZ;01110;01;AIN;1
01187;;Hotonnes;HOTONNES;01260;01;AIN;1
01188;;Illiat;ILLIAT;01140;01;AIN;1
01189;;Injoux-Gnissiat;INJOUX-GENISSIAT;01200;01;AIN;1
01190;;Innimond;INNIMOND;01680;01;AIN;1
01191;;Izenave;IZENAVE;01430;01;AIN;1
01192;;Izernore;IZERNORE;01580;01;AIN;1
01193;;Izieu;IZIEU;01300;01;AIN;1
01194;;Jassans-Riottier;JASSANS-RIOTTIER;01480;01;AIN;1
01195;;Jasseron;JASSERON;01250;01;AIN;1
01196;;Jayat;JAYAT;01340;01;AIN;1
01197;;Journans;JOURNANS;01250;01;AIN;1
01198;;Joyeux;JOYEUX;01800;01;AIN;1
01199;;Jujurieux;JUJURIEUX;01640;01;AIN;1
01200;;Labalme;LABALME;01450;01;AIN;1
01202;;Lagnieu;LAGNIEU;01150;01;AIN;1
01203;;Laiz;LAIZ;01290;01;AIN;1
01204;;Lalleyriat;LALLEYRIAT;01130;01;AIN;1
01205;;Lancrans;LANCRANS;01200;01;AIN;1
01206;;Lantenay;LANTENAY;01430;01;AIN;1
01207;;Lapeyrouse;LAPEYROUSE;01330;01;AIN;1
01208;;Lavours;LAVOURS;01350;01;AIN;1
01209;;Laz;LEAZ;01200;01;AIN;1
01210;;Llex;LELEX;01410;01;AIN;1
01211;;Lent;LENT;01240;01;AIN;1
01212;;Lescheroux;LESCHEROUX;01560;01;AIN;1
01213;;Leyment;LEYMENT;01150;01;AIN;1
01214;;Leyssard;LEYSSARD;01450;01;AIN;1
01215;;Lhpital;LHOPITAL;01420;01;AIN;1
01216;;Lhuis;LHUIS;01680;01;AIN;1
01218;;Lochieu;LOCHIEU;01260;01;AIN;1
01219;;Lompnas;LOMPNAS;01680;01;AIN;1
01221;;Lompnieu;LOMPNIEU;01260;01;AIN;1
01224;;Loyettes;LOYETTES;01360;01;AIN;1
01225;;Lurcy;LURCY;01090;01;AIN;1
01227;;Magnieu;MAGNIEU;01300;01;AIN;1
01228;;Maillat;MAILLAT;01430;01;AIN;1
01229;;Malafretaz;MALAFRETAZ;01340;01;AIN;1
01230;;Mantenay-Montlin;MANTENAY-MONTLIN;01560;01;AIN;1
01231;;Manziat;MANZIAT;01570;01;AIN;1
01232;;Marboz;MARBOZ;01851;01;AIN;1
01233;;Marchamp;MARCHAMP;01680;01;AIN;1
01234;;Marignieu;MARIGNIEU;01300;01;AIN;1
01235;;Marlieux;MARLIEUX;01240;01;AIN;1
01236;;Marsonnas;MARSONNAS;01340;01;AIN;1
01237;;Martignat;MARTIGNAT;01810;01;AIN;1
01238;;Massieux;MASSIEUX;01600;01;AIN;1
01239;;Massignieu-de-Rives;MASSIGNIEU-DE-RIVES;01300;01;AIN;1
01240;;Matafelon-Granges;MATAFELON-GRANGES;01580;01;AIN;1
01241;;Meillonnas;MEILLONNAS;01370;01;AIN;1
01242;;Mrignat;MERIGNAT;01450;01;AIN;1
01243;;Messimy-sur-Sane;MESSIMY-SUR-SAONE;01480;01;AIN;1
01244;;Meximieux;MEXIMIEUX;01800;01;AIN;1
01245;;Bohas-Meyriat-Rignat;BOHAS-MEYRIAT-RIGNAT;01250;01;AIN;1
01246;;Mzriat;MEZERIAT;01660;01;AIN;1
01247;;Mijoux;MIJOUX;01170;01;AIN;1
01248;;Mionnay;MIONNAY;01390;01;AIN;1
01249;;Miribel;MIRIBEL;01700;01;AIN;1
01250;;Misrieux;MISERIEUX;01600;01;AIN;1
01252;;Mogneneins;MOGNENEINS;01140;01;AIN;1
01254;;Montagnat;MONTAGNAT;01250;01;AIN;1
01255;;Montagnieu;MONTAGNIEU;01470;01;AIN;1
01257;;Montanges;MONTANGES;01200;01;AIN;1
01258;;Montceaux;MONTCEAUX;01090;01;AIN;1
01259;;Montcet;MONTCET;01310;01;AIN;1
01260;Le;Montellier;MONTELLIER;01800;01;AIN;1
01261;;Monthieux;MONTHIEUX;01390;01;AIN;1
01262;;Montluel;MONTLUEL;01120;01;AIN;1
01263;;Montmerle-sur-Sane;MONTMERLE-SUR-SAONE;01090;01;AIN;1
01264;;Montracol;MONTRACOL;01310;01;AIN;1
01265;;Montral-la-Cluse;MONTREAL-LA-CLUSE;01460;01;AIN;1
01266;;Montrevel-en-Bresse;MONTREVEL-EN-BRESSE;01340;01;AIN;1
01267;;Nurieux-Volognat;NURIEUX-VOLOGNAT;01460;01;AIN;1
01268;;Murs-et-Glignieux;MURS-ET-GELIGNIEUX;01300;01;AIN;1
01269;;Nantua;NANTUA;01130;01;AIN;1
01271;;Nattages;NATTAGES;01300;01;AIN;1
01272;;Neuville-les-Dames;NEUVILLE-LES-DAMES;01400;01;AIN;1
01273;;Neuville-sur-Ain;NEUVILLE-SUR-AIN;01160;01;AIN;1
01274;Les;Neyrolles;NEYROLLES;01130;01;AIN;1
01275;;Neyron;NEYRON;01700;01;AIN;1
01276;;Nivroz;NIEVROZ;01120;01;AIN;1
01277;;Nivollet-Montgriffon;NIVOLLET-MONTGRIFFON;01230;01;AIN;1
01279;;Oncieu;ONCIEU;01230;01;AIN;1
01280;;Ordonnaz;ORDONNAZ;01510;01;AIN;1
01281;;Ornex;ORNEX;01210;01;AIN;1
01282;;Outriaz;OUTRIAZ;01430;01;AIN;1
01283;;Oyonnax;OYONNAX;01100;01;AIN;1
01284;;Ozan;OZAN;01190;01;AIN;1
01285;;Parcieux;PARCIEUX;01600;01;AIN;1
01286;;Parves;PARVES;01300;01;AIN;1
01288;;Pron;PERON;01630;01;AIN;1
01289;;Pronnas;PERONNAS;01960;01;AIN;1
01290;;Prouges;PEROUGES;01800;01;AIN;1
01291;;Perrex;PERREX;01540;01;AIN;1
01292;Le;Petit-Abergement;PETIT-ABERGEMENT;01260;01;AIN;1
01293;;Peyriat;PEYRIAT;01430;01;AIN;1
01294;;Peyrieu;PEYRIEU;01300;01;AIN;1
01295;;Peyzieux-sur-Sane;PEYZIEUX-SUR-SAONE;01140;01;AIN;1
01296;;Pirajoux;PIRAJOUX;01270;01;AIN;1
01297;;Pizay;PIZAY;01120;01;AIN;1
01298;;Plagne;PLAGNE;01130;01;AIN;1
01299;Le;Plantay;PLANTAY;01330;01;AIN;1
01300;Le;Poizat;POIZAT;01130;01;AIN;1
01301;;Polliat;POLLIAT;01310;01;AIN;1
01302;;Pollieu;POLLIEU;01350;01;AIN;1
01303;;Poncin;PONCIN;01450;01;AIN;1
01304;;Pont-d'Ain;PONT-D'AIN;01160;01;AIN;1
01305;;Pont-de-Vaux;PONT-DE-VAUX;01190;01;AIN;1
01306;;Pont-de-Veyle;PONT-DE-VEYLE;01290;01;AIN;1
01307;;Port;PORT;01460;01;AIN;1
01308;;Pougny;POUGNY;01550;01;AIN;1
01309;;Pouillat;POUILLAT;01250;01;AIN;1
01310;;Prmeyzel;PREMEYZEL;01300;01;AIN;1
01311;;Prmillieu;PREMILLIEU;01510;01;AIN;1
01312;;Pressiat;PRESSIAT;01370;01;AIN;1
01313;;Prvessin-Mons;PREVESSIN-MOENS;01280;01;AIN;1
01314;;Priay;PRIAY;01160;01;AIN;1
01316;;Pugieu;PUGIEU;01510;01;AIN;1
01317;;Ramasse;RAMASSE;01250;01;AIN;1
01318;;Ranc;RANCE;01390;01;AIN;1
01319;;Relevant;RELEVANT;01990;01;AIN;1
01320;;Replonges;REPLONGES;01620;01;AIN;1
01321;;Revonnas;REVONNAS;01250;01;AIN;1
01322;;Reyrieux;REYRIEUX;01600;01;AIN;1
01323;;Reyssouze;REYSSOUZE;01190;01;AIN;1
01325;;Rignieux-le-Franc;RIGNIEUX-LE-FRANC;01800;01;AIN;1
01328;;Romans;ROMANS;01400;01;AIN;1
01329;;Rossillon;ROSSILLON;01510;01;AIN;1
01330;;Ruffieu;RUFFIEU;01260;01;AIN;1
01331;;Saint-Alban;SAINT-ALBAN;01450;01;AIN;1
01332;;Saint-Andr-de-Bg;SAINT-ANDRE-DE-BAGE;01380;01;AIN;1
01333;;Saint-Andr-de-Corcy;SAINT-ANDRE-DE-CORCY;01390;01;AIN;1
01334;;Saint-Andr-d'Huiriat;SAINT-ANDRE-D'HUIRIAT;01290;01;AIN;1
01335;;Saint-Andr-le-Bouchoux;SAINT-ANDRE-LE-BOUCHOUX;01240;01;AIN;1
01336;;Saint-Andr-sur-Vieux-Jonc;SAINT-ANDRE-SUR-VIEUX-JONC;01240;01;AIN;1
01337;;Saint-Bnigne;SAINT-BENIGNE;01190;01;AIN;1
01338;;Saint-Benot;SAINT-BENOIT;01300;01;AIN;1
01339;;Saint-Bernard;SAINT-BERNARD;01600;01;AIN;1
01340;;Saint-Bois;SAINT-BOIS;01300;01;AIN;1
01341;;Saint-Champ;SAINT-CHAMP;01300;01;AIN;1
01342;;Sainte-Croix;SAINTE-CROIX;01120;01;AIN;1
01343;;Saint-Cyr-sur-Menthon;SAINT-CYR-SUR-MENTHON;01380;01;AIN;1
01344;;Saint-Denis-ls-Bourg;SAINT-DENIS-LES-BOURG;01000;01;AIN;1
01345;;Saint-Denis-en-Bugey;SAINT-DENIS-EN-BUGEY;01500;01;AIN;1
01346;;Saint-Didier-d'Aussiat;SAINT-DIDIER-D'AUSSIAT;01340;01;AIN;1
01347;;Saint-Didier-de-Formans;SAINT-DIDIER-DE-FORMANS;01600;01;AIN;1
01348;;Saint-Didier-sur-Chalaronne;SAINT-DIDIER-SUR-CHALARONNE;01140;01;AIN;1
01349;;Saint-loi;SAINT-ELOI;01800;01;AIN;1
01350;;Saint-tienne-du-Bois;SAINT-ETIENNE-DU-BOIS;01370;01;AIN;1
01351;;Saint-tienne-sur-Chalaronne;SAINT-ETIENNE-SUR-CHALARONNE;01140;01;AIN;1
01352;;Saint-tienne-sur-Reyssouze;SAINT-ETIENNE-SUR-REYSSOUZE;01190;01;AIN;1
01353;;Sainte-Euphmie;SAINTE-EUPHEMIE;01600;01;AIN;1
01354;;Saint-Genis-Pouilly;SAINT-GENIS-POUILLY;01630;01;AIN;1
01355;;Saint-Genis-sur-Menthon;SAINT-GENIS-SUR-MENTHON;01380;01;AIN;1
01356;;Saint-Georges-sur-Renon;SAINT-GEORGES-SUR-RENON;01400;01;AIN;1
01357;;Saint-Germain-de-Joux;SAINT-GERMAIN-DE-JOUX;01130;01;AIN;1
01358;;Saint-Germain-les-Paroisses;SAINT-GERMAIN-LES-PAROISSES;01300;01;AIN;1
01359;;Saint-Germain-sur-Renon;SAINT-GERMAIN-SUR-RENON;01240;01;AIN;1
01360;;Saint-Jean-de-Gonville;SAINT-JEAN-DE-GONVILLE;01630;01;AIN;1
01361;;Saint-Jean-de-Niost;SAINT-JEAN-DE-NIOST;01800;01;AIN;1
01362;;Saint-Jean-de-Thurigneux;SAINT-JEAN-DE-THURIGNEUX;01390;01;AIN;1
01363;;Saint-Jean-le-Vieux;SAINT-JEAN-LE-VIEUX;01640;01;AIN;1
01364;;Saint-Jean-sur-Reyssouze;SAINT-JEAN-SUR-REYSSOUZE;01560;01;AIN;1
01365;;Saint-Jean-sur-Veyle;SAINT-JEAN-SUR-VEYLE;01290;01;AIN;1
01366;;Sainte-Julie;SAINTE-JULIE;01150;01;AIN;1
01367;;Saint-Julien-sur-Reyssouze;SAINT-JULIEN-SUR-REYSSOUZE;01560;01;AIN;1
01368;;Saint-Julien-sur-Veyle;SAINT-JULIEN-SUR-VEYLE;01540;01;AIN;1
01369;;Saint-Just;SAINT-JUST;01250;01;AIN;1
01370;;Saint-Laurent-sur-Sane;SAINT-LAURENT-SUR-SAONE;01620;01;AIN;1
01371;;Saint-Marcel;SAINT-MARCEL;01390;01;AIN;1
01372;;Saint-Martin-de-Bavel;SAINT-MARTIN-DE-BAVEL;01510;01;AIN;1
01373;;Saint-Martin-du-Frne;SAINT-MARTIN-DU-FRENE;01430;01;AIN;1
01374;;Saint-Martin-du-Mont;SAINT-MARTIN-DU-MONT;01160;01;AIN;1
01375;;Saint-Martin-le-Chtel;SAINT-MARTIN-LE-CHATEL;01310;01;AIN;1
01376;;Saint-Maurice-de-Beynost;SAINT-MAURICE-DE-BEYNOST;01700;01;AIN;1
01378;;Saint-Maurice-de-Gourdans;SAINT-MAURICE-DE-GOURDANS;01800;01;AIN;1
01379;;Saint-Maurice-de-Rmens;SAINT-MAURICE-DE-REMENS;01500;01;AIN;1
01380;;Saint-Nizier-le-Bouchoux;SAINT-NIZIER-LE-BOUCHOUX;01560;01;AIN;1
01381;;Saint-Nizier-le-Dsert;SAINT-NIZIER-LE-DESERT;01320;01;AIN;1
01382;;Sainte-Olive;SAINTE-OLIVE;01330;01;AIN;1
01383;;Saint-Paul-de-Varax;SAINT-PAUL-DE-VARAX;01240;01;AIN;1
01384;;Saint-Rambert-en-Bugey;SAINT-RAMBERT-EN-BUGEY;01230;01;AIN;1
01385;;Saint-Rmy;SAINT-REMY;01310;01;AIN;1
01386;;Saint-Sorlin-en-Bugey;SAINT-SORLIN-EN-BUGEY;01150;01;AIN;1
01387;;Saint-Sulpice;SAINT-SULPICE;01340;01;AIN;1
01388;;Saint-Trivier-de-Courtes;SAINT-TRIVIER-DE-COURTES;01560;01;AIN;1
01389;;Saint-Trivier-sur-Moignans;SAINT-TRIVIER-SUR-MOIGNANS;01990;01;AIN;1
01390;;Saint-Vulbas;SAINT-VULBAS;01150;01;AIN;1
01391;;Salavre;SALAVRE;01270;01;AIN;1
01392;;Samognat;SAMOGNAT;01580;01;AIN;1
01393;;Sandrans;SANDRANS;01400;01;AIN;1
01396;;Sault-Brnaz;SAULT-BRENAZ;01150;01;AIN;1
01397;;Sauverny;SAUVERNY;01220;01;AIN;1
01398;;Savigneux;SAVIGNEUX;01480;01;AIN;1
01399;;Sgny;SEGNY;01170;01;AIN;1
01400;;Seillonnaz;SEILLONNAZ;01470;01;AIN;1
01401;;Sergy;SERGY;01630;01;AIN;1
01402;;Sermoyer;SERMOYER;01190;01;AIN;1
01403;;Serrires-de-Briord;SERRIERES-DE-BRIORD;01470;01;AIN;1
01404;;Serrires-sur-Ain;SERRIERES-SUR-AIN;01450;01;AIN;1
01405;;Servas;SERVAS;01240;01;AIN;1
01406;;Servignat;SERVIGNAT;01560;01;AIN;1
01407;;Seyssel;SEYSSEL;01420;01;AIN;1
01408;;Simandre-sur-Suran;SIMANDRE-SUR-SURAN;01250;01;AIN;1
01409;;Songieu;SONGIEU;01260;01;AIN;1
01410;;Sonthonnax-la-Montagne;SONTHONNAX-LA-MONTAGNE;01580;01;AIN;1
01411;;Souclin;SOUCLIN;01150;01;AIN;1
01412;;Sulignat;SULIGNAT;01400;01;AIN;1
01413;;Surjoux;SURJOUX;01420;01;AIN;1
01414;;Sutrieu;SUTRIEU;01260;01;AIN;1
01415;;Talissieu;TALISSIEU;01510;01;AIN;1
01416;;Tenay;TENAY;01230;01;AIN;1
01417;;Thzillieu;THEZILLIEU;01110;01;AIN;1
01418;;Thil;THIL;01120;01;AIN;1
01419;;Thoiry;THOIRY;01710;01;AIN;1
01420;;Thoissey;THOISSEY;01140;01;AIN;1
01421;;Torcieu;TORCIEU;01230;01;AIN;1
01422;;Tossiat;TOSSIAT;01250;01;AIN;1
01423;;Toussieux;TOUSSIEUX;01600;01;AIN;1
01424;;Tramoyes;TRAMOYES;01390;01;AIN;1
01425;La;Tranclire;TRANCLIERE;01160;01;AIN;1
01426;;Treffort-Cuisiat;TREFFORT-CUISIAT;01370;01;AIN;1
01427;;Trvoux;TREVOUX;01600;01;AIN;1
01428;;Valeins;VALEINS;01140;01;AIN;1
01429;;Vandeins;VANDEINS;01660;01;AIN;1
01430;;Varambon;VARAMBON;01160;01;AIN;1
01431;;Vaux-en-Bugey;VAUX-EN-BUGEY;01150;01;AIN;1
01432;;Verjon;VERJON;01270;01;AIN;1
01433;;Vernoux;VERNOUX;01560;01;AIN;1
01434;;Versailleux;VERSAILLEUX;01330;01;AIN;1
01435;;Versonnex;VERSONNEX;01210;01;AIN;1
01436;;Vesancy;VESANCY;01170;01;AIN;1
01437;;Vescours;VESCOURS;01560;01;AIN;1
01439;;Vsines;VESINES;01570;01;AIN;1
01441;;Vieu-d'Izenave;VIEU-D'IZENAVE;01430;01;AIN;1
01442;;Vieu;VIEU;01260;01;AIN;1
01443;;Villars-les-Dombes;VILLARS-LES-DOMBES;01330;01;AIN;1
01444;;Villebois;VILLEBOIS;01150;01;AIN;1
01445;;Villemotier;VILLEMOTIER;01270;01;AIN;1
01446;;Villeneuve;VILLENEUVE;01480;01;AIN;1
01447;;Villereversure;VILLEREVERSURE;01250;01;AIN;1
01448;;Villes;VILLES;01200;01;AIN;1
01449;;Villette-sur-Ain;VILLETTE-SUR-AIN;01320;01;AIN;1
01450;;Villieu-Loyes-Mollon;VILLIEU-LOYES-MOLLON;01800;01;AIN;1
01451;;Viriat;VIRIAT;01440;01;AIN;1
01452;;Virieu-le-Grand;VIRIEU-LE-GRAND;01510;01;AIN;1
01453;;Virieu-le-Petit;VIRIEU-LE-PETIT;01260;01;AIN;1
01454;;Virignin;VIRIGNIN;01300;01;AIN;1
01456;;Vongnes;VONGNES;01350;01;AIN;1
01457;;Vonnas;VONNAS;01540;01;AIN;1
02001;;Abbcourt;ABBECOURT;02300;02;AISNE;1
02002;;Achery;ACHERY;02800;02;AISNE;1
02003;;Acy;ACY;02200;02;AISNE;1
02004;;Agnicourt-et-Schelles;AGNICOURT-ET-SECHELLES;02340;02;AISNE;1
02005;;Aguilcourt;AGUILCOURT;02190;02;AISNE;1
02006;;Aisonville-et-Bernoville;AISONVILLE-ET-BERNOVILLE;02110;02;AISNE;1
02007;;Aizelles;AIZELLES;02820;02;AISNE;1
02008;;Aizy-Jouy;AIZY-JOUY;02370;02;AISNE;1
02009;;Alaincourt;ALAINCOURT;02240;02;AISNE;1
02010;;Allemant;ALLEMANT;02320;02;AISNE;1
02011;;Ambleny;AMBLENY;02290;02;AISNE;1
02012;;Ambrief;AMBRIEF;02200;02;AISNE;1
02013;;Amifontaine;AMIFONTAINE;02190;02;AISNE;1
02014;;Amigny-Rouy;AMIGNY-ROUY;02700;02;AISNE;1
02015;;Ancienville;ANCIENVILLE;02600;02;AISNE;1
02016;;Andelain;ANDELAIN;02800;02;AISNE;1
02017;;Anguilcourt-le-Sart;ANGUILCOURT-LE-SART;02800;02;AISNE;1
02018;;Anizy-le-Chteau;ANIZY-LE-CHATEAU;02320;02;AISNE;1
02019;;Annois;ANNOIS;02480;02;AISNE;1
02020;;Any-Martin-Rieux;ANY-MARTIN-RIEUX;02500;02;AISNE;1
02021;;Archon;ARCHON;02360;02;AISNE;1
02022;;Arcy-Sainte-Restitue;ARCY-SAINTE-RESTITUE;02130;02;AISNE;1
02023;;Armentires-sur-Ourcq;ARMENTIERES-SUR-OURCQ;02210;02;AISNE;1
02024;;Arrancy;ARRANCY;02860;02;AISNE;1
02025;;Artemps;ARTEMPS;02480;02;AISNE;1
02026;;Artonges;ARTONGES;02330;02;AISNE;1
02027;;Assis-sur-Serre;ASSIS-SUR-SERRE;02270;02;AISNE;1
02028;;Athies-sous-Laon;ATHIES-SOUS-LAON;02840;02;AISNE;1
02029;;Attilly;ATTILLY;02490;02;AISNE;1
02030;;Aubencheul-aux-Bois;AUBENCHEUL-AUX-BOIS;02420;02;AISNE;1
02031;;Aubenton;AUBENTON;02500;02;AISNE;1
02032;;Aubigny-aux-Kaisnes;AUBIGNY-AUX-KAISNES;02590;02;AISNE;1
02033;;Aubigny-en-Laonnois;AUBIGNY-EN-LAONNOIS;02820;02;AISNE;1
02034;;Audignicourt;AUDIGNICOURT;02300;02;AISNE;1
02035;;Audigny;AUDIGNY;02120;02;AISNE;1
02036;;Augy;AUGY;02220;02;AISNE;1
02037;;Aulnois-sous-Laon;AULNOIS-SOUS-LAON;02000;02;AISNE;1
02038;Les;Autels;AUTELS;02360;02;AISNE;1
02039;;Autremencourt;AUTREMENCOURT;02250;02;AISNE;1
02040;;Autreppes;AUTREPPES;02580;02;AISNE;1
02041;;Autreville;AUTREVILLE;02300;02;AISNE;1
02042;;Azy-sur-Marne;AZY-SUR-MARNE;02400;02;AISNE;1
02043;;Bagneux;BAGNEUX;02290;02;AISNE;1
02044;;Bancigny;BANCIGNY;02140;02;AISNE;1
02046;;Barenton-Bugny;BARENTON-BUGNY;02000;02;AISNE;1
02047;;Barenton-Cel;BARENTON-CEL;02000;02;AISNE;1
02048;;Barenton-sur-Serre;BARENTON-SUR-SERRE;02270;02;AISNE;1
02049;;Barisis;BARISIS;02700;02;AISNE;1
02050;;Barzy-en-Thirache;BARZY-EN-THIERACHE;02170;02;AISNE;1
02051;;Barzy-sur-Marne;BARZY-SUR-MARNE;02850;02;AISNE;1
02052;;Bassoles-Aulers;BASSOLES-AULERS;02380;02;AISNE;1
02053;;Baulne-en-Brie;BAULNE-EN-BRIE;02330;02;AISNE;1
02054;;Bazoches-sur-Vesles;BAZOCHES-SUR-VESLES;02220;02;AISNE;1
02055;;Beaum;BEAUME;02500;02;AISNE;1
02056;;Beaumont-en-Beine;BEAUMONT-EN-BEINE;02300;02;AISNE;1
02057;;Beaurevoir;BEAUREVOIR;02110;02;AISNE;1
02058;;Beaurieux;BEAURIEUX;02160;02;AISNE;1
02059;;Beautor;BEAUTOR;02800;02;AISNE;1
02060;;Beauvois-en-Vermandois;BEAUVOIS-EN-VERMANDOIS;02590;02;AISNE;1
02061;;Becquigny;BECQUIGNY;02110;02;AISNE;1
02062;;Belleau;BELLEAU;02400;02;AISNE;1
02063;;Bellenglise;BELLENGLISE;02420;02;AISNE;1
02064;;Belleu;BELLEU;02200;02;AISNE;1
02065;;Bellicourt;BELLICOURT;02420;02;AISNE;1
02066;;Benay;BENAY;02440;02;AISNE;1
02067;;Bergues-sur-Sambre;BERGUES-SUR-SAMBRE;02450;02;AISNE;1
02068;;Berlancourt;BERLANCOURT;02250;02;AISNE;1
02069;;Berlise;BERLISE;02340;02;AISNE;1
02070;;Bernot;BERNOT;02120;02;AISNE;1
02071;;Berny-Rivire;BERNY-RIVIERE;02290;02;AISNE;1
02072;;Berrieux;BERRIEUX;02820;02;AISNE;1
02073;;Berry-au-Bac;BERRY-AU-BAC;02190;02;AISNE;1
02074;;Bertaucourt-Epourdon;BERTAUCOURT-EPOURDON;02800;02;AISNE;1
02075;;Berthenicourt;BERTHENICOURT;02240;02;AISNE;1
02076;;Bertricourt;BERTRICOURT;02190;02;AISNE;1
02077;;Berzy-le-Sec;BERZY-LE-SEC;02200;02;AISNE;1
02078;;Besm;BESME;02300;02;AISNE;1
02079;;Besmont;BESMONT;02500;02;AISNE;1
02080;;Besny-et-Loizy;BESNY-ET-LOIZY;02870;02;AISNE;1
02081;;Bthancourt-en-Vaux;BETHANCOURT-EN-VAUX;02300;02;AISNE;1
02082;;Beugneux;BEUGNEUX;02210;02;AISNE;1
02083;;Beuvardes;BEUVARDES;02130;02;AISNE;1
02084;;Bzu-le-Gury;BEZU-LE-GUERY;02310;02;AISNE;1
02085;;Bzu-Saint-Germain;BEZU-SAINT-GERMAIN;02400;02;AISNE;1
02086;;Bichancourt;BICHANCOURT;02300;02;AISNE;1
02087;;Bieuxy;BIEUXY;02290;02;AISNE;1
02088;;Bivres;BIEVRES;02860;02;AISNE;1
02089;;Billy-sur-Aisne;BILLY-SUR-AISNE;02200;02;AISNE;1
02090;;Billy-sur-Ourcq;BILLY-SUR-OURCQ;02210;02;AISNE;1
02091;;Blanzy-ls-Fismes;BLANZY-LES-FISMES;02160;02;AISNE;1
02093;;Blrancourt;BLERANCOURT;02300;02;AISNE;1
02094;;Blesmes;BLESMES;02400;02;AISNE;1
02095;;Bohain-en-Vermandois;BOHAIN-EN-VERMANDOIS;02110;02;AISNE;1
02096;;Bois-ls-Pargny;BOIS-LES-PARGNY;02270;02;AISNE;1
02097;;Boncourt;BONCOURT;02350;02;AISNE;1
02098;;Bonneil;BONNEIL;02400;02;AISNE;1
02099;;Bonnesvalyn;BONNESVALYN;02400;02;AISNE;1
02100;;Bony;BONY;02420;02;AISNE;1
02101;;Bosmont-sur-Serre;BOSMONT-SUR-SERRE;02250;02;AISNE;1
02102;;Bouconville-Vauclair;BOUCONVILLE-VAUCLAIR;02860;02;AISNE;1
02103;;Bou;BOUE;02450;02;AISNE;1
02104;;Bouffignereux;BOUFFIGNEREUX;02160;02;AISNE;1
02105;;Bouresches;BOURESCHES;02400;02;AISNE;1
02106;;Bourg-et-Comin;BOURG-ET-COMIN;02160;02;AISNE;1
02107;;Bourguignon-sous-Coucy;BOURGUIGNON-SOUS-COUCY;02300;02;AISNE;1
02108;;Bourguignon-sous-Montbavin;BOURGUIGNON-SOUS-MONTBAVIN;02000;02;AISNE;1
02109;La;Bouteille;BOUTEILLE;02140;02;AISNE;1
02110;;Braine;BRAINE;02220;02;AISNE;1
02111;;Brancourt-en-Laonnois;BRANCOURT-EN-LAONNOIS;02320;02;AISNE;1
02112;;Brancourt-le-Grand;BRANCOURT-LE-GRAND;02110;02;AISNE;1
02114;;Brasles;BRASLES;02400;02;AISNE;1
02115;;Braye-en-Laonnois;BRAYE-EN-LAONNOIS;02000;02;AISNE;1
02116;;Braye-en-Thirache;BRAYE-EN-THIERACHE;02140;02;AISNE;1
02117;;Bray-Saint-Christophe;BRAY-SAINT-CHRISTOPHE;02480;02;AISNE;1
02118;;Braye;BRAYE;02880;02;AISNE;1
02119;;Brcy;BRECY;02210;02;AISNE;1
02120;;Brenelle;BRENELLE;02220;02;AISNE;1
02121;;Breny;BRENY;02210;02;AISNE;1
02122;;Brie;BRIE;02870;02;AISNE;1
02123;;Brissay-Choigny;BRISSAY-CHOIGNY;02240;02;AISNE;1
02124;;Brissy-Hamgicourt;BRISSY-HAMEGICOURT;02240;02;AISNE;1
02125;;Brumetz;BRUMETZ;02810;02;AISNE;1
02126;;Brunehamel;BRUNEHAMEL;02360;02;AISNE;1
02127;;Bruyres-sur-Fre;BRUYERES-SUR-FERE;02130;02;AISNE;1
02128;;Bruyres-et-Montbrault;BRUYERES-ET-MONTBERAULT;02860;02;AISNE;1
02129;;Bruys;BRUYS;02220;02;AISNE;1
02130;;Bucilly;BUCILLY;02500;02;AISNE;1
02131;;Bucy-le-Long;BUCY-LE-LONG;02880;02;AISNE;1
02132;;Bucy-ls-Cerny;BUCY-LES-CERNY;02870;02;AISNE;1
02133;;Bucy-ls-Pierrepont;BUCY-LES-PIERREPONT;02350;02;AISNE;1
02134;;Buire;BUIRE;02500;02;AISNE;1
02135;;Buironfosse;BUIRONFOSSE;02620;02;AISNE;1
02136;;Burelles;BURELLES;02140;02;AISNE;1
02137;;Bussiares;BUSSIARES;02810;02;AISNE;1
02138;;Buzancy;BUZANCY;02200;02;AISNE;1
02139;;Cailloul-Crpigny;CAILLOUEL-CREPIGNY;02300;02;AISNE;1
02140;;Camelin;CAMELIN;02300;02;AISNE;1
02141;La;Capelle;CAPELLE;02260;02;AISNE;1
02142;;Castres;CASTRES;02680;02;AISNE;1
02143;Le;Catelet;CATELET;02420;02;AISNE;1
02144;;Caulaincourt;CAULAINCOURT;02490;02;AISNE;1
02145;;Caumont;CAUMONT;02300;02;AISNE;1
02146;;Celles-ls-Cond;CELLES-LES-CONDE;02330;02;AISNE;1
02147;La;Celle-sous-Montmirail;CELLE-SOUS-MONTMIRAIL;02540;02;AISNE;1
02148;;Celles-sur-Aisne;CELLES-SUR-AISNE;02370;02;AISNE;1
02149;;Cerizy;CERIZY;02240;02;AISNE;1
02150;;Cerny-en-Laonnois;CERNY-EN-LAONNOIS;02860;02;AISNE;1
02151;;Cerny-ls-Bucy;CERNY-LES-BUCY;02870;02;AISNE;1
02152;;Cerseuil;CERSEUIL;02220;02;AISNE;1
02153;;Cessires;CESSIERES;02320;02;AISNE;1
02154;;Chacrise;CHACRISE;02200;02;AISNE;1
02155;;Chaillevois;CHAILLEVOIS;02000;02;AISNE;1
02156;;Chalandry;CHALANDRY;02270;02;AISNE;1
02157;;Chambry;CHAMBRY;02000;02;AISNE;1
02158;;Chamouille;CHAMOUILLE;02860;02;AISNE;1
02159;;Champs;CHAMPS;02670;02;AISNE;1
02160;;Chaourse;CHAOURSE;02340;02;AISNE;1
02161;La;Chapelle-Monthodon;CHAPELLE-MONTHODON;02330;02;AISNE;1
02162;La;Chapelle-sur-Chzy;CHAPELLE-SUR-CHEZY;02570;02;AISNE;1
02163;;Charly-sur-Marne;CHARLY-SUR-MARNE;02310;02;AISNE;1
02164;Le;Charmel;CHARMEL;02850;02;AISNE;1
02165;;Charmes;CHARMES;02800;02;AISNE;1
02166;;Chartves;CHARTEVES;02400;02;AISNE;1
02167;;Chassemy;CHASSEMY;02370;02;AISNE;1
02168;;Chteau-Thierry;CHATEAU-THIERRY;02400;02;AISNE;1
02169;;Chtillon-ls-Sons;CHATILLON-LES-SONS;02270;02;AISNE;1
02170;;Chtillon-sur-Oise;CHATILLON-SUR-OISE;02240;02;AISNE;1
02171;;Chaudardes;CHAUDARDES;02160;02;AISNE;1
02172;;Chaudun;CHAUDUN;02200;02;AISNE;1
02173;;Chauny;CHAUNY;02300;02;AISNE;1
02174;;Chavignon;CHAVIGNON;02000;02;AISNE;1
02175;;Chavigny;CHAVIGNY;02880;02;AISNE;1
02176;;Chavonne;CHAVONNE;02370;02;AISNE;1
02177;;Chrt;CHERET;02860;02;AISNE;1
02178;;Chermizy-Ailles;CHERMIZY-AILLES;02860;02;AISNE;1
02179;;Chry-Chartreuve;CHERY-CHARTREUVE;02220;02;AISNE;1
02180;;Chry-ls-Pouilly;CHERY-LES-POUILLY;02000;02;AISNE;1
02181;;Chry-ls-Rozoy;CHERY-LES-ROZOY;02360;02;AISNE;1
02182;;Chevennes;CHEVENNES;02250;02;AISNE;1
02183;;Chevregny;CHEVREGNY;02000;02;AISNE;1
02184;;Chevresis-Monceau;CHEVRESIS-MONCEAU;02270;02;AISNE;1
02185;;Chzy-en-Orxois;CHEZY-EN-ORXOIS;02810;02;AISNE;1
02186;;Chzy-sur-Marne;CHEZY-SUR-MARNE;02570;02;AISNE;1
02187;;Chierry;CHIERRY;02400;02;AISNE;1
02188;;Chigny;CHIGNY;02120;02;AISNE;1
02189;;Chivres-en-Laonnois;CHIVRES-EN-LAONNOIS;02350;02;AISNE;1
02190;;Chivres-Val;CHIVRES-VAL;02880;02;AISNE;1
02191;;Chivy-ls-touvelles;CHIVY-LES-ETOUVELLES;02000;02;AISNE;1
02192;;Chouy;CHOUY;02210;02;AISNE;1
02193;;Cierges;CIERGES;02130;02;AISNE;1
02194;;Cilly;CILLY;02250;02;AISNE;1
02195;;Ciry-Salsogne;CIRY-SALSOGNE;02220;02;AISNE;1
02196;;Clacy-et-Thierret;CLACY-ET-THIERRET;02000;02;AISNE;1
02197;;Clairfontaine;CLAIRFONTAINE;02260;02;AISNE;1
02198;;Clamecy;CLAMECY;02880;02;AISNE;1
02199;;Clastres;CLASTRES;02440;02;AISNE;1
02200;;Clermont-les-Fermes;CLERMONT-LES-FERMES;02340;02;AISNE;1
02201;;Cuvres-et-Valsery;COEUVRES-ET-VALSERY;02600;02;AISNE;1
02203;;Coincy;COINCY;02210;02;AISNE;1
02204;;Coingt;COINGT;02360;02;AISNE;1
02205;;Colligis-Crandelain;COLLIGIS-CRANDELAIN;02860;02;AISNE;1
02206;;Colonfay;COLONFAY;02120;02;AISNE;1
02207;;Commenchon;COMMENCHON;02300;02;AISNE;1
02208;;Concevreux;CONCEVREUX;02160;02;AISNE;1
02209;;Cond-en-Brie;CONDE-EN-BRIE;02330;02;AISNE;1
02210;;Cond-sur-Aisne;CONDE-SUR-AISNE;02370;02;AISNE;1
02211;;Cond-sur-Suippe;CONDE-SUR-SUIPPE;02190;02;AISNE;1
02212;;Condren;CONDREN;02700;02;AISNE;1
02213;;Connigis;CONNIGIS;02330;02;AISNE;1
02214;;Contescourt;CONTESCOURT;02680;02;AISNE;1
02215;;Corbeny;CORBENY;02820;02;AISNE;1
02216;;Corcy;CORCY;02600;02;AISNE;1
02217;;Coucy-le-Chteau-Auffrique;COUCY-LE-CHATEAU-AUFFRIQUE;02380;02;AISNE;1
02218;;Coucy-ls-Eppes;COUCY-LES-EPPES;02840;02;AISNE;1
02219;;Coucy-la-Ville;COUCY-LA-VILLE;02380;02;AISNE;1
02220;;Coulonges-Cohan;COULONGES-COHAN;02130;02;AISNE;1
02221;;Coupru;COUPRU;02310;02;AISNE;1
02222;;Courbes;COURBES;02800;02;AISNE;1
02223;;Courboin;COURBOIN;02330;02;AISNE;1
02224;;Courcelles-sur-Vesle;COURCELLES-SUR-VESLE;02220;02;AISNE;1
02225;;Courchamps;COURCHAMPS;02810;02;AISNE;1
02226;;Courmelles;COURMELLES;02200;02;AISNE;1
02227;;Courmont;COURMONT;02130;02;AISNE;1
02228;;Courtemont-Varennes;COURTEMONT-VARENNES;02850;02;AISNE;1
02229;;Courtrizy-et-Fussigny;COURTRIZY-ET-FUSSIGNY;02820;02;AISNE;1
02230;;Couvrelles;COUVRELLES;02220;02;AISNE;1
02231;;Couvron-et-Aumencourt;COUVRON-ET-AUMENCOURT;02270;02;AISNE;1
02232;;Coyolles;COYOLLES;02600;02;AISNE;1
02233;;Cramaille;CRAMAILLE;02130;02;AISNE;1
02234;;Craonne;CRAONNE;02160;02;AISNE;1
02235;;Craonnelle;CRAONNELLE;02160;02;AISNE;1
02236;;Crcy-au-Mont;CRECY-AU-MONT;02380;02;AISNE;1
02237;;Crcy-sur-Serre;CRECY-SUR-SERRE;02270;02;AISNE;1
02238;;Crpy;CREPY;02870;02;AISNE;1
02239;;Crzancy;CREZANCY;02650;02;AISNE;1
02240;;Croix-Fonsomme;CROIX-FONSOMME;02110;02;AISNE;1
02241;La;Croix-sur-Ourcq;CROIX-SUR-OURCQ;02210;02;AISNE;1
02242;;Crouttes-sur-Marne;CROUTTES-SUR-MARNE;02310;02;AISNE;1
02243;;Crouy;CROUY;02880;02;AISNE;1
02244;;Crupilly;CRUPILLY;02120;02;AISNE;1
02245;;Cuffies;CUFFIES;02880;02;AISNE;1
02246;;Cugny;CUGNY;02480;02;AISNE;1
02248;;Cuirieux;CUIRIEUX;02350;02;AISNE;1
02249;;Cuiry-Housse;CUIRY-HOUSSE;02220;02;AISNE;1
02250;;Cuiry-ls-Chaudardes;CUIRY-LES-CHAUDARDES;02160;02;AISNE;1
02251;;Cuiry-ls-Iviers;CUIRY-LES-IVIERS;02360;02;AISNE;1
02252;;Cuissy-et-Geny;CUISSY-ET-GENY;02160;02;AISNE;1
02253;;Cuisy-en-Almont;CUISY-EN-ALMONT;02200;02;AISNE;1
02254;;Cutry;CUTRY;02600;02;AISNE;1
02255;;Cys-la-Commune;CYS-LA-COMMUNE;02220;02;AISNE;1
02256;;Dagny-Lambercy;DAGNY-LAMBERCY;02140;02;AISNE;1
02257;;Dallon;DALLON;02680;02;AISNE;1
02258;;Dammard;DAMMARD;02470;02;AISNE;1
02259;;Dampleux;DAMPLEUX;02600;02;AISNE;1
02260;;Danizy;DANIZY;02800;02;AISNE;1
02261;;Dercy;DERCY;02270;02;AISNE;1
02262;;Deuillet;DEUILLET;02700;02;AISNE;1
02263;;Dhuizel;DHUIZEL;02220;02;AISNE;1
02264;;Dizy-le-Gros;DIZY-LE-GROS;02340;02;AISNE;1
02265;;Dohis;DOHIS;02360;02;AISNE;1
02266;;Dolignon;DOLIGNON;02360;02;AISNE;1
02267;;Dommiers;DOMMIERS;02600;02;AISNE;1
02268;;Domptin;DOMPTIN;02310;02;AISNE;1
02269;;Dorengt;DORENGT;02450;02;AISNE;1
02270;;Douchy;DOUCHY;02590;02;AISNE;1
02271;;Dravegny;DRAVEGNY;02130;02;AISNE;1
02272;;Droizy;DROIZY;02210;02;AISNE;1
02273;;Dury;DURY;02480;02;AISNE;1
02274;;bouleau;EBOULEAU;02350;02;AISNE;1
02275;;Effry;EFFRY;02500;02;AISNE;1
02276;;Englancourt;ENGLANCOURT;02260;02;AISNE;1
02277;;pagny;EPAGNY;02290;02;AISNE;1
02278;;parcy;EPARCY;02500;02;AISNE;1
02279;;paux-Bzu;EPAUX-BEZU;02400;02;AISNE;1
02280;;pieds;EPIEDS;02400;02;AISNE;1
02281;L';pine-aux-Bois;EPINE-AUX-BOIS;02540;02;AISNE;1
02282;;Eppes;EPPES;02840;02;AISNE;1
02283;;Erlon;ERLON;02250;02;AISNE;1
02284;;Erloy;ERLOY;02260;02;AISNE;1
02286;;Esquhries;ESQUEHERIES;02170;02;AISNE;1
02287;;Essigny-le-Grand;ESSIGNY-LE-GRAND;02690;02;AISNE;1
02288;;Essigny-le-Petit;ESSIGNY-LE-PETIT;02100;02;AISNE;1
02289;;Essises;ESSISES;02570;02;AISNE;1
02290;;Essmes-sur-Marne;ESSOMES-SUR-MARNE;02400;02;AISNE;1
02291;;Estres;ESTREES;02420;02;AISNE;1
02292;;tampes-sur-Marne;ETAMPES-SUR-MARNE;02400;02;AISNE;1
02293;;taves-et-Bocquiaux;ETAVES-ET-BOCQUIAUX;02110;02;AISNE;1
02294;;touvelles;ETOUVELLES;02000;02;AISNE;1
02295;;traupont;ETREAUPONT;02580;02;AISNE;1
02296;;treillers;ETREILLERS;02590;02;AISNE;1
02297;;trpilly;ETREPILLY;02400;02;AISNE;1
02298;;treux;ETREUX;02510;02;AISNE;1
02299;;vergnicourt;EVERGNICOURT;02190;02;AISNE;1
02301;;Faucoucourt;FAUCOUCOURT;02320;02;AISNE;1
02302;;Faverolles;FAVEROLLES;02600;02;AISNE;1
02303;;Fayet;FAYET;02100;02;AISNE;1
02304;La;Fre;FERE;02800;02;AISNE;1
02305;;Fre-en-Tardenois;FERE-EN-TARDENOIS;02130;02;AISNE;1
02306;La;Fert-Chevresis;FERTE-CHEVRESIS;02270;02;AISNE;1
02307;La;Fert-Milon;FERTE-MILON;02460;02;AISNE;1
02308;;Fesmy-le-Sart;FESMY-LE-SART;02450;02;AISNE;1
02309;;Festieux;FESTIEUX;02840;02;AISNE;1
02310;;Fieulaine;FIEULAINE;02110;02;AISNE;1
02311;;Filain;FILAIN;02000;02;AISNE;1
02312;La;Flamengrie;FLAMENGRIE;02260;02;AISNE;1
02313;;Flavigny-le-Grand-et-Beaurain;FLAVIGNY-LE-GRAND-ET-BEAURAIN;02120;02;AISNE;1
02315;;Flavy-le-Martel;FLAVY-LE-MARTEL;02520;02;AISNE;1
02316;;Fleury;FLEURY;02600;02;AISNE;1
02317;;Fluquires;FLUQUIERES;02590;02;AISNE;1
02318;;Folembray;FOLEMBRAY;02670;02;AISNE;1
02319;;Fonsomme;FONSOMME;02110;02;AISNE;1
02320;;Fontaine-ls-Clercs;FONTAINE-LES-CLERCS;02680;02;AISNE;1
02321;;Fontaine-ls-Vervins;FONTAINE-LES-VERVINS;02140;02;AISNE;1
02322;;Fontaine-Notre-Dame;FONTAINE-NOTRE-DAME;02110;02;AISNE;1
02323;;Fontaine-Uterte;FONTAINE-UTERTE;02110;02;AISNE;1
02324;;Fontenelle;FONTENELLE;02170;02;AISNE;1
02325;;Fontenelle-en-Brie;FONTENELLE-EN-BRIE;02540;02;AISNE;1
02326;;Fontenoy;FONTENOY;02290;02;AISNE;1
02327;;Foreste;FORESTE;02590;02;AISNE;1
02328;;Fossoy;FOSSOY;02650;02;AISNE;1
02329;;Fourdrain;FOURDRAIN;02870;02;AISNE;1
02330;;Francilly-Selency;FRANCILLY-SELENCY;02760;02;AISNE;1
02331;;Franqueville;FRANQUEVILLE;02140;02;AISNE;1
02332;;Fresnes-en-Tardenois;FRESNES-EN-TARDENOIS;02130;02;AISNE;1
02333;;Fresnes;FRESNES;02380;02;AISNE;1
02334;;Fresnoy-le-Grand;FRESNOY-LE-GRAND;02230;02;AISNE;1
02335;;Fressancourt;FRESSANCOURT;02800;02;AISNE;1
02336;;Frires-Failloul;FRIERES-FAILLOUEL;02700;02;AISNE;1
02337;;Froidestres;FROIDESTREES;02260;02;AISNE;1
02338;;Froidmont-Cohartille;FROIDMONT-COHARTILLE;02270;02;AISNE;1
02339;;Gandelu;GANDELU;02810;02;AISNE;1
02340;;Gauchy;GAUCHY;02430;02;AISNE;1
02341;;Gercy;GERCY;02140;02;AISNE;1
02342;;Gergny;GERGNY;02260;02;AISNE;1
02343;;Germaine;GERMAINE;02590;02;AISNE;1
02344;;Gernicourt;GERNICOURT;02160;02;AISNE;1
02345;;Gibercourt;GIBERCOURT;02440;02;AISNE;1
02346;;Gizy;GIZY;02350;02;AISNE;1
02347;;Gland;GLAND;02400;02;AISNE;1
02348;;Glennes;GLENNES;02160;02;AISNE;1
02349;;Goudelancourt-ls-Berrieux;GOUDELANCOURT-LES-BERRIEUX;02820;02;AISNE;1
02350;;Goudelancourt-ls-Pierrepont;GOUDELANCOURT-LES-PIERREPONT;02350;02;AISNE;1
02351;;Goussancourt;GOUSSANCOURT;02130;02;AISNE;1
02352;;Gouy;GOUY;02420;02;AISNE;1
02353;;Grandlup-et-Fay;GRANDLUP-ET-FAY;02350;02;AISNE;1
02354;;Grandrieux;GRANDRIEUX;02360;02;AISNE;1
02355;;Gricourt;GRICOURT;02100;02;AISNE;1
02356;;Grisolles;GRISOLLES;02210;02;AISNE;1
02357;;Gronard;GRONARD;02140;02;AISNE;1
02358;;Grougis;GROUGIS;02110;02;AISNE;1
02359;;Grugies;GRUGIES;02680;02;AISNE;1
02360;;Guignicourt;GUIGNICOURT;02190;02;AISNE;1
02361;;Guise;GUISE;02120;02;AISNE;1
02362;;Guivry;GUIVRY;02300;02;AISNE;1
02363;;Guny;GUNY;02300;02;AISNE;1
02364;;Guyencourt;GUYENCOURT;02160;02;AISNE;1
02366;;Hannapes;HANNAPES;02510;02;AISNE;1
02367;;Happencourt;HAPPENCOURT;02480;02;AISNE;1
02368;;Haramont;HARAMONT;02600;02;AISNE;1
02369;;Harcigny;HARCIGNY;02140;02;AISNE;1
02370;;Hargicourt;HARGICOURT;02420;02;AISNE;1
02371;;Harly;HARLY;02100;02;AISNE;1
02372;;Hartennes-et-Taux;HARTENNES-ET-TAUX;02210;02;AISNE;1
02373;;Hary;HARY;02140;02;AISNE;1
02374;;Lehaucourt;LEHAUCOURT;02420;02;AISNE;1
02375;;Hautevesnes;HAUTEVESNES;02810;02;AISNE;1
02376;;Hauteville;HAUTEVILLE;02120;02;AISNE;1
02377;;Haution;HAUTION;02140;02;AISNE;1
02378;La;Hrie;HERIE;02500;02;AISNE;1
02379;Le;Hrie-la-Viville;HERIE-LA-VIEVILLE;02120;02;AISNE;1
02380;;Hinacourt;HINACOURT;02440;02;AISNE;1
02381;;Hirson;HIRSON;02500;02;AISNE;1
02382;;Holnon;HOLNON;02760;02;AISNE;1
02383;;Homblires;HOMBLIERES;02720;02;AISNE;1
02384;;Houry;HOURY;02140;02;AISNE;1
02385;;Housset;HOUSSET;02250;02;AISNE;1
02386;;Iron;IRON;02510;02;AISNE;1
02387;;Itancourt;ITANCOURT;02240;02;AISNE;1
02388;;Iviers;IVIERS;02360;02;AISNE;1
02389;;Jaulgonne;JAULGONNE;02850;02;AISNE;1
02390;;Jeancourt;JEANCOURT;02490;02;AISNE;1
02391;;Jeantes;JEANTES;02140;02;AISNE;1
02392;;Joncourt;JONCOURT;02420;02;AISNE;1
02393;;Jouaignes;JOUAIGNES;02220;02;AISNE;1
02395;;Jumencourt;JUMENCOURT;02380;02;AISNE;1
02396;;Jumigny;JUMIGNY;02160;02;AISNE;1
02397;;Jussy;JUSSY;02480;02;AISNE;1
02398;;Juvigny;JUVIGNY;02880;02;AISNE;1
02399;;Juvincourt-et-Damary;JUVINCOURT-ET-DAMARY;02190;02;AISNE;1
02400;;Laffaux;LAFFAUX;02880;02;AISNE;1
02401;;Laigny;LAIGNY;02140;02;AISNE;1
02402;;Lanchy;LANCHY;02590;02;AISNE;1
02403;;Landifay-et-Bertaignemont;LANDIFAY-ET-BERTAIGNEMONT;02120;02;AISNE;1
02404;;Landouzy-la-Cour;LANDOUZY-LA-COUR;02140;02;AISNE;1
02405;;Landouzy-la-Ville;LANDOUZY-LA-VILLE;02140;02;AISNE;1
02406;;Landricourt;LANDRICOURT;02380;02;AISNE;1
02407;;Laniscourt;LANISCOURT;02000;02;AISNE;1
02408;;Laon;LAON;02000;02;AISNE;1
02409;;Lappion;LAPPION;02150;02;AISNE;1
02410;;Largny-sur-Automne;LARGNY-SUR-AUTOMNE;02600;02;AISNE;1
02411;;Latilly;LATILLY;02210;02;AISNE;1
02412;;Launoy;LAUNOY;02210;02;AISNE;1
02413;;Laval-en-Laonnois;LAVAL-EN-LAONNOIS;02860;02;AISNE;1
02414;;Lavaqueresse;LAVAQUERESSE;02450;02;AISNE;1
02415;;Laversine;LAVERSINE;02600;02;AISNE;1
02416;;Lem;LEME;02140;02;AISNE;1
02417;;Lempire;LEMPIRE;02420;02;AISNE;1
02418;;Lerzy;LERZY;02260;02;AISNE;1
02419;;Leschelle;LESCHELLE;02170;02;AISNE;1
02420;;Lesdins;LESDINS;02100;02;AISNE;1
02421;;Lesges;LESGES;02220;02;AISNE;1
02422;;Lesquielles-Saint-Germain;LESQUIELLES-SAINT-GERMAIN;02120;02;AISNE;1
02423;;Leuilly-sous-Coucy;LEUILLY-SOUS-COUCY;02380;02;AISNE;1
02424;;Leury;LEURY;02880;02;AISNE;1
02425;;Leuze;LEUZE;02500;02;AISNE;1
02426;;Levergies;LEVERGIES;02420;02;AISNE;1
02427;;Lhuys;LHUYS;02220;02;AISNE;1
02428;;Licy-Clignon;LICY-CLIGNON;02810;02;AISNE;1
02429;;Lierval;LIERVAL;02860;02;AISNE;1
02430;;Liesse-Notre-Dame;LIESSE-NOTRE-DAME;02350;02;AISNE;1
02431;;Liez;LIEZ;02700;02;AISNE;1
02432;;Lim;LIME;02220;02;AISNE;1
02433;;Lislet;LISLET;02340;02;AISNE;1
02434;;Lizy;LIZY;02320;02;AISNE;1
02435;;Logny-ls-Aubenton;LOGNY-LES-AUBENTON;02500;02;AISNE;1
02438;;Longpont;LONGPONT;02600;02;AISNE;1
02439;;Longueval-Barbonval;LONGUEVAL-BARBONVAL;02160;02;AISNE;1
02440;;Lor;LOR;02190;02;AISNE;1
02441;;Loutre;LOUATRE;02600;02;AISNE;1
02442;;Loupeigne;LOUPEIGNE;02130;02;AISNE;1
02443;;Lucy-le-Bocage;LUCY-LE-BOCAGE;02400;02;AISNE;1
02444;;Lugny;LUGNY;02140;02;AISNE;1
02445;;Luzoir;LUZOIR;02500;02;AISNE;1
02446;;Ly-Fontaine;LY-FONTAINE;02440;02;AISNE;1
02447;;Maast-et-Violaine;MAAST-ET-VIOLAINE;02220;02;AISNE;1
02448;;Mchecourt;MACHECOURT;02350;02;AISNE;1
02449;;Macogny;MACOGNY;02470;02;AISNE;1
02450;;Macquigny;MACQUIGNY;02120;02;AISNE;1
02451;;Magny-la-Fosse;MAGNY-LA-FOSSE;02420;02;AISNE;1
02452;;Maissemy;MAISSEMY;02490;02;AISNE;1
02453;;Maizy;MAIZY;02160;02;AISNE;1
02454;La;Malmaison;MALMAISON;02190;02;AISNE;1
02455;;Malzy;MALZY;02120;02;AISNE;1
02456;;Manicamp;MANICAMP;02300;02;AISNE;1
02457;;Marchais;MARCHAIS;02350;02;AISNE;1
02458;;Marchais-en-Brie;MARCHAIS-EN-BRIE;02540;02;AISNE;1
02459;;Marcy;MARCY;02720;02;AISNE;1
02460;;Marcy-sous-Marle;MARCY-SOUS-MARLE;02250;02;AISNE;1
02461;;Marest-Dampcourt;MAREST-DAMPCOURT;02300;02;AISNE;1
02462;;Mareuil-en-Dle;MAREUIL-EN-DOLE;02130;02;AISNE;1
02463;;Marfontaine;MARFONTAINE;02140;02;AISNE;1
02464;;Margival;MARGIVAL;02880;02;AISNE;1
02465;;Marigny-en-Orxois;MARIGNY-EN-ORXOIS;02810;02;AISNE;1
02466;;Marizy-Sainte-Genevive;MARIZY-SAINTE-GENEVIEVE;02470;02;AISNE;1
02467;;Marizy-Saint-Mard;MARIZY-SAINT-MARD;02470;02;AISNE;1
02468;;Marle;MARLE;02250;02;AISNE;1
02469;;Marly-Gomont;MARLY-GOMONT;02120;02;AISNE;1
02470;;Martigny;MARTIGNY;02500;02;AISNE;1
02471;;Martigny-Courpierre;MARTIGNY-COURPIERRE;02860;02;AISNE;1
02472;;Mauregny-en-Haye;MAUREGNY-EN-HAYE;02820;02;AISNE;1
02473;;Mayot;MAYOT;02800;02;AISNE;1
02474;;Mennessis;MENNESSIS;02700;02;AISNE;1
02475;;Menneville;MENNEVILLE;02190;02;AISNE;1
02476;;Mennevret;MENNEVRET;02630;02;AISNE;1
02477;;Mercin-et-Vaux;MERCIN-ET-VAUX;02200;02;AISNE;1
02478;;Merlieux-et-Fouquerolles;MERLIEUX-ET-FOUQUEROLLES;02000;02;AISNE;1
02479;;Merval;MERVAL;02160;02;AISNE;1
02480;;Mesbrecourt-Richecourt;MESBRECOURT-RICHECOURT;02270;02;AISNE;1
02481;;Mesnil-Saint-Laurent;MESNIL-SAINT-LAURENT;02720;02;AISNE;1
02482;;Meurival;MEURIVAL;02160;02;AISNE;1
02483;;Mzires-sur-Oise;MEZIERES-SUR-OISE;02240;02;AISNE;1
02484;;Mzy-Moulins;MEZY-MOULINS;02650;02;AISNE;1
02485;;Missy-aux-Bois;MISSY-AUX-BOIS;02200;02;AISNE;1
02486;;Missy-ls-Pierrepont;MISSY-LES-PIERREPONT;02350;02;AISNE;1
02487;;Missy-sur-Aisne;MISSY-SUR-AISNE;02880;02;AISNE;1
02488;;Molain;MOLAIN;02110;02;AISNE;1
02489;;Molinchart;MOLINCHART;02000;02;AISNE;1
02490;;Monampteuil;MONAMPTEUIL;02000;02;AISNE;1
02491;;Monceau-le-Neuf-et-Faucouzy;MONCEAU-LE-NEUF-ET-FAUCOUZY;02270;02;AISNE;1
02492;;Monceau-ls-Leups;MONCEAU-LES-LEUPS;02270;02;AISNE;1
02493;;Monceau-le-Waast;MONCEAU-LE-WAAST;02840;02;AISNE;1
02494;;Monceau-sur-Oise;MONCEAU-SUR-OISE;02120;02;AISNE;1
02495;;Mondrepuis;MONDREPUIS;02500;02;AISNE;1
02496;;Monnes;MONNES;02470;02;AISNE;1
02497;;Mons-en-Laonnois;MONS-EN-LAONNOIS;02000;02;AISNE;1
02498;;Montaigu;MONTAIGU;02820;02;AISNE;1
02499;;Montbavin;MONTBAVIN;02000;02;AISNE;1
02500;;Montbrehain;MONTBREHAIN;02110;02;AISNE;1
02501;;Montchlons;MONTCHALONS;02860;02;AISNE;1
02502;;Montcornet;MONTCORNET;02340;02;AISNE;1
02503;;Mont-d'Origny;MONT-D'ORIGNY;02390;02;AISNE;1
02504;;Montescourt-Lizerolles;MONTESCOURT-LIZEROLLES;02440;02;AISNE;1
02505;;Montfaucon;MONTFAUCON;02540;02;AISNE;1
02506;;Montgobert;MONTGOBERT;02600;02;AISNE;1
02507;;Montgru-Saint-Hilaire;MONTGRU-SAINT-HILAIRE;02210;02;AISNE;1
02508;;Monthenault;MONTHENAULT;02860;02;AISNE;1
02509;;Monthiers;MONTHIERS;02400;02;AISNE;1
02510;;Monthurel;MONTHUREL;02330;02;AISNE;1
02511;;Montigny-en-Arrouaise;MONTIGNY-EN-ARROUAISE;02110;02;AISNE;1
02512;;Montigny-l'Allier;MONTIGNY-L'ALLIER;02810;02;AISNE;1
02513;;Montigny-le-Franc;MONTIGNY-LE-FRANC;02250;02;AISNE;1
02514;;Montigny-Lengrain;MONTIGNY-LENGRAIN;02290;02;AISNE;1
02515;;Montigny-ls-Cond;MONTIGNY-LES-CONDE;02330;02;AISNE;1
02516;;Montigny-sous-Marle;MONTIGNY-SOUS-MARLE;02250;02;AISNE;1
02517;;Montigny-sur-Crcy;MONTIGNY-SUR-CRECY;02270;02;AISNE;1
02518;;Montlevon;MONTLEVON;02330;02;AISNE;1
02519;;Montlou;MONTLOUE;02340;02;AISNE;1
02520;;Mont-Notre-Dame;MONT-NOTRE-DAME;02220;02;AISNE;1
02521;;Montreuil-aux-Lions;MONTREUIL-AUX-LIONS;02310;02;AISNE;1
02522;;Mont-Saint-Jean;MONT-SAINT-JEAN;02360;02;AISNE;1
02523;;Mont-Saint-Martin;MONT-SAINT-MARTIN;02220;02;AISNE;1
02524;;Mont-Saint-Pre;MONT-SAINT-PERE;02400;02;AISNE;1
02525;;Morcourt;MORCOURT;02100;02;AISNE;1
02526;;Morgny-en-Thirache;MORGNY-EN-THIERACHE;02360;02;AISNE;1
02527;;Morsain;MORSAIN;02290;02;AISNE;1
02528;;Mortefontaine;MORTEFONTAINE;02600;02;AISNE;1
02529;;Mortiers;MORTIERS;02270;02;AISNE;1
02530;;Moulins;MOULINS;02160;02;AISNE;1
02531;;Moussy-Verneuil;MOUSSY-VERNEUIL;02160;02;AISNE;1
02532;;Mo-de-l'Aisne;MOY-DE-L'AISNE;02610;02;AISNE;1
02533;;Muret-et-Crouttes;MURET-ET-CROUTTES;02210;02;AISNE;1
02534;;Muscourt;MUSCOURT;02160;02;AISNE;1
02535;;Nampcelles-la-Cour;NAMPCELLES-LA-COUR;02140;02;AISNE;1
02536;;Nampteuil-sous-Muret;NAMPTEUIL-SOUS-MURET;02200;02;AISNE;1
02537;;Nanteuil-la-Fosse;NANTEUIL-LA-FOSSE;02880;02;AISNE;1
02538;;Nanteuil-Notre-Dame;NANTEUIL-NOTRE-DAME;02210;02;AISNE;1
02539;;Nauroy;NAUROY;02420;02;AISNE;1
02540;;Nesles-la-Montagne;NESLES-LA-MONTAGNE;02400;02;AISNE;1
02541;;Neufchtel-sur-Aisne;NEUFCHATEL-SUR-AISNE;02190;02;AISNE;1
02542;;Neuflieux;NEUFLIEUX;02300;02;AISNE;1
02543;;Neuilly-Saint-Front;NEUILLY-SAINT-FRONT;02470;02;AISNE;1
02544;;Neuve-Maison;NEUVE-MAISON;02500;02;AISNE;1
02545;La;Neuville-Bosmont;NEUVILLE-BOSMONT;02250;02;AISNE;1
02546;La;Neuville-en-Beine;NEUVILLE-EN-BEINE;02300;02;AISNE;1
02547;La;Neuville-Housset;NEUVILLE-HOUSSET;02250;02;AISNE;1
02548;La;Neuville-ls-Dorengt;NEUVILLE-LES-DORENGT;02450;02;AISNE;1
02549;;Neuville-Saint-Amand;NEUVILLE-SAINT-AMAND;02100;02;AISNE;1
02550;;Neuville-sur-Ailette;NEUVILLE-SUR-AILETTE;02860;02;AISNE;1
02551;;Neuville-sur-Margival;NEUVILLE-SUR-MARGIVAL;02880;02;AISNE;1
02552;;Neuvillette;NEUVILLETTE;02390;02;AISNE;1
02553;;Nizy-le-Comte;NIZY-LE-COMTE;02150;02;AISNE;1
02554;;Nogentel;NOGENTEL;02400;02;AISNE;1
02555;;Nogent-l'Artaud;NOGENT-L'ARTAUD;02310;02;AISNE;1
02556;;Noircourt;NOIRCOURT;02340;02;AISNE;1
02557;;Noroy-sur-Ourcq;NOROY-SUR-OURCQ;02600;02;AISNE;1
02558;Le;Nouvion-en-Thirache;NOUVION-EN-THIERACHE;02170;02;AISNE;1
02559;;Nouvion-et-Catillon;NOUVION-ET-CATILLON;02270;02;AISNE;1
02560;;Nouvion-le-Comte;NOUVION-LE-COMTE;02800;02;AISNE;1
02561;;Nouvion-le-Vineux;NOUVION-LE-VINEUX;02860;02;AISNE;1
02562;;Nouvron-Vingr;NOUVRON-VINGRE;02290;02;AISNE;1
02563;;Noyales;NOYALES;02120;02;AISNE;1
02564;;Noyant-et-Aconin;NOYANT-ET-ACONIN;02200;02;AISNE;1
02565;;uilly;OEUILLY;02160;02;AISNE;1
02566;;Ognes;OGNES;02300;02;AISNE;1
02567;;Ohis;OHIS;02500;02;AISNE;1
02568;;Oigny-en-Valois;OIGNY-EN-VALOIS;02600;02;AISNE;1
02569;;Oisy;OISY;02450;02;AISNE;1
02570;;Ollezy;OLLEZY;02480;02;AISNE;1
02571;;Omissy;OMISSY;02100;02;AISNE;1
02572;;Orainville;ORAINVILLE;02190;02;AISNE;1
02573;;Orgeval;ORGEVAL;02860;02;AISNE;1
02574;;Origny-en-Thirache;ORIGNY-EN-THIERACHE;02550;02;AISNE;1
02575;;Origny-Sainte-Benoite;ORIGNY-SAINTE-BENOITE;02390;02;AISNE;1
02576;;Osly-Courtil;OSLY-COURTIL;02290;02;AISNE;1
02577;;Ostel;OSTEL;02370;02;AISNE;1
02578;;Oulches-la-Valle-Foulon;OULCHES-LA-VALLEE-FOULON;02160;02;AISNE;1
02579;;Oulchy-la-Ville;OULCHY-LA-VILLE;02210;02;AISNE;1
02580;;Oulchy-le-Chteau;OULCHY-LE-CHATEAU;02210;02;AISNE;1
02581;;Paars;PAARS;02220;02;AISNE;1
02582;;Paissy;PAISSY;02160;02;AISNE;1
02583;;Pancy-Courtecon;PANCY-COURTECON;02860;02;AISNE;1
02584;;Papleux;PAPLEUX;02260;02;AISNE;1
02585;;Parcy-et-Tigny;PARCY-ET-TIGNY;02210;02;AISNE;1
02586;;Parfondeval;PARFONDEVAL;02360;02;AISNE;1
02587;;Parfondru;PARFONDRU;02840;02;AISNE;1
02588;;Pargnan;PARGNAN;02160;02;AISNE;1
02589;;Pargny-Filain;PARGNY-FILAIN;02000;02;AISNE;1
02590;;Pargny-la-Dhuys;PARGNY-LA-DHUYS;02330;02;AISNE;1
02591;;Pargny-les-Bois;PARGNY-LES-BOIS;02270;02;AISNE;1
02592;;Parpeville;PARPEVILLE;02240;02;AISNE;1
02593;;Pasly;PASLY;02200;02;AISNE;1
02594;;Passy-en-Valois;PASSY-EN-VALOIS;02470;02;AISNE;1
02595;;Passy-sur-Marne;PASSY-SUR-MARNE;02850;02;AISNE;1
02596;;Pavant;PAVANT;02310;02;AISNE;1
02597;;Perles;PERLES;02160;02;AISNE;1
02598;;Pernant;PERNANT;02200;02;AISNE;1
02599;;Pierremande;PIERREMANDE;02300;02;AISNE;1
02600;;Pierrepont;PIERREPONT;02350;02;AISNE;1
02601;;Pignicourt;PIGNICOURT;02190;02;AISNE;1
02602;;Pinon;PINON;02320;02;AISNE;1
02604;;Pithon;PITHON;02480;02;AISNE;1
02605;;Pleine-Selve;PLEINE-SELVE;02240;02;AISNE;1
02606;Le;Plessier-Huleu;PLESSIER-HULEU;02210;02;AISNE;1
02607;;Ploisy;PLOISY;02200;02;AISNE;1
02608;;Plomion;PLOMION;02140;02;AISNE;1
02609;;Ployart-et-Vaurseine;PLOYART-ET-VAURSEINE;02860;02;AISNE;1
02610;;Pommiers;POMMIERS;02200;02;AISNE;1
02612;;Pont-Arcy;PONT-ARCY;02160;02;AISNE;1
02613;;Pontavert;PONTAVERT;02160;02;AISNE;1
02614;;Pontru;PONTRU;02490;02;AISNE;1
02615;;Pontruet;PONTRUET;02490;02;AISNE;1
02616;;Pont-Saint-Mard;PONT-SAINT-MARD;02380;02;AISNE;1
02617;;Pouilly-sur-Serre;POUILLY-SUR-SERRE;02270;02;AISNE;1
02618;;Prmont;PREMONT;02110;02;AISNE;1
02619;;Prmontr;PREMONTRE;02320;02;AISNE;1
02620;;Presles-et-Boves;PRESLES-ET-BOVES;02370;02;AISNE;1
02621;;Presles-et-Thierny;PRESLES-ET-THIERNY;02860;02;AISNE;1
02622;;Priez;PRIEZ;02470;02;AISNE;1
02623;;Prisces;PRISCES;02140;02;AISNE;1
02624;;Proisy;PROISY;02120;02;AISNE;1
02625;;Proix;PROIX;02120;02;AISNE;1
02626;;Prouvais;PROUVAIS;02190;02;AISNE;1
02627;;Proviseux-et-Plesnoy;PROVISEUX-ET-PLESNOY;02190;02;AISNE;1
02628;;Puiseux-en-Retz;PUISEUX-EN-RETZ;02600;02;AISNE;1
02629;;Puisieux-et-Clanlieu;PUISIEUX-ET-CLANLIEU;02120;02;AISNE;1
02631;;Quierzy;QUIERZY;02300;02;AISNE;1
02632;;Quincy-Basse;QUINCY-BASSE;02380;02;AISNE;1
02633;;Quincy-sous-le-Mont;QUINCY-SOUS-LE-MONT;02220;02;AISNE;1
02634;;Raillimont;RAILLIMONT;02360;02;AISNE;1
02635;;Ramicourt;RAMICOURT;02110;02;AISNE;1
02636;;Regny;REGNY;02240;02;AISNE;1
02637;;Remaucourt;REMAUCOURT;02100;02;AISNE;1
02638;;Remies;REMIES;02270;02;AISNE;1
02639;;Remigny;REMIGNY;02440;02;AISNE;1
02640;;Renansart;RENANSART;02240;02;AISNE;1
02641;;Renneval;RENNEVAL;02340;02;AISNE;1
02642;;Rsigny;RESIGNY;02360;02;AISNE;1
02643;;Ressons-le-Long;RESSONS-LE-LONG;02290;02;AISNE;1
02644;;Retheuil;RETHEUIL;02600;02;AISNE;1
02645;;Reuilly-Sauvigny;REUILLY-SAUVIGNY;02850;02;AISNE;1
02646;;Rvillon;REVILLON;02160;02;AISNE;1
02647;;Ribeauville;RIBEAUVILLE;02110;02;AISNE;1
02648;;Ribemont;RIBEMONT;02240;02;AISNE;1
02649;;Rocourt-Saint-Martin;ROCOURT-SAINT-MARTIN;02210;02;AISNE;1
02650;;Rocquigny;ROCQUIGNY;02260;02;AISNE;1
02651;;Rogcourt;ROGECOURT;02800;02;AISNE;1
02652;;Rogny;ROGNY;02140;02;AISNE;1
02653;;Romeny-sur-Marne;ROMENY-SUR-MARNE;02310;02;AISNE;1
02654;;Romery;ROMERY;02120;02;AISNE;1
02655;;Ronchres;RONCHERES;02130;02;AISNE;1
02656;;Roucy;ROUCY;02160;02;AISNE;1
02657;;Rougeries;ROUGERIES;02140;02;AISNE;1
02658;;Roupy;ROUPY;02590;02;AISNE;1
02659;;Rouvroy;ROUVROY;02100;02;AISNE;1
02660;;Rouvroy-sur-Serre;ROUVROY-SUR-SERRE;02360;02;AISNE;1
02661;;Royaucourt-et-Chailvet;ROYAUCOURT-ET-CHAILVET;02000;02;AISNE;1
02662;;Rozet-Saint-Albin;ROZET-SAINT-ALBIN;02210;02;AISNE;1
02663;;Rozires-sur-Crise;ROZIERES-SUR-CRISE;02200;02;AISNE;1
02664;;Rozoy-Bellevalle;ROZOY-BELLEVALLE;02540;02;AISNE;1
02665;;Grand-Rozoy;GRAND-ROZOY;02210;02;AISNE;1
02666;;Rozoy-sur-Serre;ROZOY-SUR-SERRE;02360;02;AISNE;1
02667;;Saconin-et-Breuil;SACONIN-ET-BREUIL;02200;02;AISNE;1
02668;;Sains-Richaumont;SAINS-RICHAUMONT;02120;02;AISNE;1
02669;;Saint-Agnan;SAINT-AGNAN;02330;02;AISNE;1
02670;;Saint-Algis;SAINT-ALGIS;02260;02;AISNE;1
02671;;Saint-Aubin;SAINT-AUBIN;02300;02;AISNE;1
02672;;Saint-Bandry;SAINT-BANDRY;02290;02;AISNE;1
02673;;Saint-Christophe--Berry;SAINT-CHRISTOPHE-A-BERRY;02290;02;AISNE;1
02674;;Saint-Clment;SAINT-CLEMENT;02360;02;AISNE;1
02675;;Sainte-Croix;SAINTE-CROIX;02820;02;AISNE;1
02676;;Saint-Erme-Outre-et-Ramecourt;SAINT-ERME-OUTRE-ET-RAMECOURT;02820;02;AISNE;1
02677;;Saint-Eugne;SAINT-EUGENE;02330;02;AISNE;1
02678;;Sainte-Genevive;SAINTE-GENEVIEVE;02340;02;AISNE;1
02679;;Saint-Gengoulph;SAINT-GENGOULPH;02810;02;AISNE;1
02680;;Saint-Gobain;SAINT-GOBAIN;02410;02;AISNE;1
02681;;Saint-Gobert;SAINT-GOBERT;02140;02;AISNE;1
02682;;Saint-Mard;SAINT-MARD;02220;02;AISNE;1
02683;;Saint-Martin-Rivire;SAINT-MARTIN-RIVIERE;02110;02;AISNE;1
02684;;Saint-Michel;SAINT-MICHEL;02830;02;AISNE;1
02685;;Saint-Nicolas-aux-Bois;SAINT-NICOLAS-AUX-BOIS;02410;02;AISNE;1
02686;;Saint-Paul-aux-Bois;SAINT-PAUL-AUX-BOIS;02300;02;AISNE;1
02687;;Saint-Pierre-Aigle;SAINT-PIERRE-AIGLE;02600;02;AISNE;1
02688;;Saint-Pierre-ls-Franqueville;SAINT-PIERRE-LES-FRANQUEVILLE;02140;02;AISNE;1
02689;;Saint-Pierremont;SAINT-PIERREMONT;02250;02;AISNE;1
02690;;Sainte-Preuve;SAINTE-PREUVE;02350;02;AISNE;1
02691;;Saint-Quentin;SAINT-QUENTIN;02100;02;AISNE;1
02693;;Saint-Rmy-Blanzy;SAINT-REMY-BLANZY;02210;02;AISNE;1
02694;;Saint-Simon;SAINT-SIMON;02640;02;AISNE;1
02695;;Saint-Thibaut;SAINT-THIBAUT;02220;02;AISNE;1
02696;;Saint-Thomas;SAINT-THOMAS;02820;02;AISNE;1
02697;;Samoussy;SAMOUSSY;02840;02;AISNE;1
02698;;Sancy-les-Cheminots;SANCY-LES-CHEMINOTS;02880;02;AISNE;1
02699;;Saponay;SAPONAY;02130;02;AISNE;1
02701;;Saulchery;SAULCHERY;02310;02;AISNE;1
02702;;Savy;SAVY;02590;02;AISNE;1
02703;;Seboncourt;SEBONCOURT;02110;02;AISNE;1
02704;;Selens;SELENS;02300;02;AISNE;1
02705;La;Selve;SELVE;02150;02;AISNE;1
02706;;Septmonts;SEPTMONTS;02200;02;AISNE;1
02707;;Septvaux;SEPTVAUX;02410;02;AISNE;1
02708;;Sequehart;SEQUEHART;02420;02;AISNE;1
02709;;Serain;SERAIN;02110;02;AISNE;1
02710;;Seraucourt-le-Grand;SERAUCOURT-LE-GRAND;02790;02;AISNE;1
02711;;Serches;SERCHES;02220;02;AISNE;1
02712;;Sergy;SERGY;02130;02;AISNE;1
02713;;Seringes-et-Nesles;SERINGES-ET-NESLES;02130;02;AISNE;1
02714;;Sermoise;SERMOISE;02220;02;AISNE;1
02715;;Serval;SERVAL;02160;02;AISNE;1
02716;;Servais;SERVAIS;02700;02;AISNE;1
02717;;Sry-ls-Mzires;SERY-LES-MEZIERES;02240;02;AISNE;1
02718;;Silly-la-Poterie;SILLY-LA-POTERIE;02460;02;AISNE;1
02719;;Sinceny;SINCENY;02300;02;AISNE;1
02720;;Sissonne;SISSONNE;02150;02;AISNE;1
02721;;Sissy;SISSY;02240;02;AISNE;1
02722;;Soissons;SOISSONS;02200;02;AISNE;1
02723;;Soize;SOIZE;02340;02;AISNE;1
02724;;Sommelans;SOMMELANS;02470;02;AISNE;1
02725;;Sommeron;SOMMERON;02260;02;AISNE;1
02726;;Sommette-Eaucourt;SOMMETTE-EAUCOURT;02480;02;AISNE;1
02727;;Sons-et-Ronchres;SONS-ET-RONCHERES;02270;02;AISNE;1
02728;;Sorbais;SORBAIS;02580;02;AISNE;1
02729;;Soucy;SOUCY;02600;02;AISNE;1
02730;;Soupir;SOUPIR;02160;02;AISNE;1
02731;Le;Sourd;SOURD;02140;02;AISNE;1
02732;;Surfontaine;SURFONTAINE;02240;02;AISNE;1
02733;;Suzy;SUZY;02320;02;AISNE;1
02734;;Taillefontaine;TAILLEFONTAINE;02600;02;AISNE;1
02735;;Tannires;TANNIERES;02220;02;AISNE;1
02736;;Tartiers;TARTIERS;02290;02;AISNE;1
02737;;Tavaux-et-Pontsricourt;TAVAUX-ET-PONTSERICOURT;02250;02;AISNE;1
02738;;Tergnier;TERGNIER;02700;02;AISNE;1
02739;;Terny-Sorny;TERNY-SORNY;02880;02;AISNE;1
02740;;Thenailles;THENAILLES;02140;02;AISNE;1
02741;;Thenelles;THENELLES;02390;02;AISNE;1
02742;;Thiernu;THIERNU;02250;02;AISNE;1
02743;Le;Thuel;THUEL;02340;02;AISNE;1
02744;;Torcy-en-Valois;TORCY-EN-VALOIS;02810;02;AISNE;1
02745;;Toulis-et-Attencourt;TOULIS-ET-ATTENCOURT;02250;02;AISNE;1
02746;;Travecy;TRAVECY;02800;02;AISNE;1
02747;;Trefcon;TREFCON;02490;02;AISNE;1
02748;;Trlou-sur-Marne;TRELOU-SUR-MARNE;02850;02;AISNE;1
02749;;Trosnes;TROESNES;02460;02;AISNE;1
02750;;Trosly-Loire;TROSLY-LOIRE;02300;02;AISNE;1
02751;;Trucy;TRUCY;02860;02;AISNE;1
02752;;Tugny-et-Pont;TUGNY-ET-PONT;02640;02;AISNE;1
02753;;Tupigny;TUPIGNY;02120;02;AISNE;1
02754;;Ugny-le-Gay;UGNY-LE-GAY;02300;02;AISNE;1
02755;;Urcel;URCEL;02000;02;AISNE;1
02756;;Urvillers;URVILLERS;02690;02;AISNE;1
02757;;Vadencourt;VADENCOURT;02120;02;AISNE;1
02758;;Vailly-sur-Aisne;VAILLY-SUR-AISNE;02370;02;AISNE;1
02759;La;Valle-au-Bl;VALLEE-AU-BLE;02140;02;AISNE;1
02760;La;Valle-Multre;VALLEE-MULATRE;02110;02;AISNE;1
02761;;Variscourt;VARISCOURT;02190;02;AISNE;1
02762;;Vassens;VASSENS;02290;02;AISNE;1
02763;;Vasseny;VASSENY;02220;02;AISNE;1
02764;;Vassogne;VASSOGNE;02160;02;AISNE;1
02765;;Vaucelles-et-Beffecourt;VAUCELLES-ET-BEFFECOURT;02000;02;AISNE;1
02766;;Vaudesson;VAUDESSON;02320;02;AISNE;1
02767;;Vauxrezis;VAUXREZIS;02200;02;AISNE;1
02768;;Vauxaillon;VAUXAILLON;02320;02;AISNE;1
02769;;Vaux-Andigny;VAUX-ANDIGNY;02110;02;AISNE;1
02770;;Vauxbuin;VAUXBUIN;02200;02;AISNE;1
02771;;Vauxcr;VAUXCERE;02160;02;AISNE;1
02772;;Vaux-en-Vermandois;VAUX-EN-VERMANDOIS;02590;02;AISNE;1
02773;;Vauxtin;VAUXTIN;02220;02;AISNE;1
02774;;Vendelles;VENDELLES;02490;02;AISNE;1
02775;;Vendeuil;VENDEUIL;02800;02;AISNE;1
02776;;Vendhuile;VENDHUILE;02420;02;AISNE;1
02777;;Vendires;VENDIERES;02540;02;AISNE;1
02778;;Vendresse-Beaulne;VENDRESSE-BEAULNE;02160;02;AISNE;1
02779;;Vnrolles;VENEROLLES;02510;02;AISNE;1
02780;;Venizel;VENIZEL;02200;02;AISNE;1
02781;;Verdilly;VERDILLY;02400;02;AISNE;1
02782;Le;Verguier;VERGUIER;02490;02;AISNE;1
02783;;Grand-Verly;GRAND-VERLY;02120;02;AISNE;1
02784;;Petit-Verly;PETIT-VERLY;02630;02;AISNE;1
02785;;Vermand;VERMAND;02490;02;AISNE;1
02786;;Verneuil-sous-Coucy;VERNEUIL-SOUS-COUCY;02380;02;AISNE;1
02787;;Verneuil-sur-Serre;VERNEUIL-SUR-SERRE;02000;02;AISNE;1
02788;;Versigny;VERSIGNY;02800;02;AISNE;1
02789;;Vervins;VERVINS;02140;02;AISNE;1
02790;;Vesles-et-Caumont;VESLES-ET-CAUMONT;02350;02;AISNE;1
02791;;Veslud;VESLUD;02840;02;AISNE;1
02792;;Veuilly-la-Poterie;VEUILLY-LA-POTERIE;02810;02;AISNE;1
02793;;Vzaponin;VEZAPONIN;02290;02;AISNE;1
02794;;Vzilly;VEZILLY;02130;02;AISNE;1
02795;;Vic-sur-Aisne;VIC-SUR-AISNE;02290;02;AISNE;1
02796;;Vichel-Nanteuil;VICHEL-NANTEUIL;02210;02;AISNE;1
02797;;Viel-Arcy;VIEL-ARCY;02160;02;AISNE;1
02798;;Viels-Maisons;VIELS-MAISONS;02540;02;AISNE;1
02799;;Vierzy;VIERZY;02210;02;AISNE;1
02800;;Viffort;VIFFORT;02540;02;AISNE;1
02801;;Vigneux-Hocquet;VIGNEUX-HOCQUET;02340;02;AISNE;1
02802;La;Ville-aux-Bois-ls-Dizy;VILLE-AUX-BOIS-LES-DIZY;02340;02;AISNE;1
02803;La;Ville-aux-Bois-ls-Pontavert;VILLE-AUX-BOIS-LES-PONTAVERT;02160;02;AISNE;1
02804;;Villemontoire;VILLEMONTOIRE;02210;02;AISNE;1
02805;;Villeneuve-Saint-Germain;VILLENEUVE-SAINT-GERMAIN;02200;02;AISNE;1
02806;;Villeneuve-sur-Fre;VILLENEUVE-SUR-FERE;02130;02;AISNE;1
02807;;Villequier-Aumont;VILLEQUIER-AUMONT;02300;02;AISNE;1
02808;;Villeret;VILLERET;02420;02;AISNE;1
02809;;Villers-Agron-Aiguizy;VILLERS-AGRON-AIGUIZY;02130;02;AISNE;1
02810;;Villers-Cotterts;VILLERS-COTTERETS;02600;02;AISNE;1
02811;;Villers-en-Prayres;VILLERS-EN-PRAYERES;02160;02;AISNE;1
02812;;Villers-Hlon;VILLERS-HELON;02600;02;AISNE;1
02813;;Villers-le-Sec;VILLERS-LE-SEC;02240;02;AISNE;1
02814;;Villers-ls-Guise;VILLERS-LES-GUISE;02120;02;AISNE;1
02815;;Villers-Saint-Christophe;VILLERS-SAINT-CHRISTOPHE;02590;02;AISNE;1
02816;;Villers-sur-Fre;VILLERS-SUR-FERE;02130;02;AISNE;1
02817;;Ville-Savoye;VILLE-SAVOYE;02220;02;AISNE;1
02818;;Villiers-Saint-Denis;VILLIERS-SAINT-DENIS;02310;02;AISNE;1
02819;;Vincy-Reuil-et-Magny;VINCY-REUIL-ET-MAGNY;02340;02;AISNE;1
02820;;Viry-Noureuil;VIRY-NOUREUIL;02300;02;AISNE;1
02821;;Vivaise;VIVAISE;02870;02;AISNE;1
02822;;Vivires;VIVIERES;02600;02;AISNE;1
02823;;Voharies;VOHARIES;02140;02;AISNE;1
02824;;Vorges;VORGES;02860;02;AISNE;1
02826;;Voulpaix;VOULPAIX;02140;02;AISNE;1
02827;;Voyenne;VOYENNE;02250;02;AISNE;1
02828;;Vregny;VREGNY;02880;02;AISNE;1
02829;;Vuillery;VUILLERY;02880;02;AISNE;1
02830;;Wassigny;WASSIGNY;02630;02;AISNE;1
02831;;Watigny;WATIGNY;02830;02;AISNE;1
02832;;Wige-Faty;WIEGE-FATY;02120;02;AISNE;1
02833;;Wimy;WIMY;02500;02;AISNE;1
02834;;Wissignicourt;WISSIGNICOURT;02320;02;AISNE;1
03001;;Abrest;ABREST;03200;03;ALLIER;1
03002;;Agonges;AGONGES;03210;03;ALLIER;1
03003;;Ainay-le-Chteau;AINAY-LE-CHATEAU;03360;03;ALLIER;1
03004;;Andelaroche;ANDELAROCHE;03120;03;ALLIER;1
03005;;Archignat;ARCHIGNAT;03380;03;ALLIER;1
03006;;Arfeuilles;ARFEUILLES;03640;03;ALLIER;1
03007;;Arpheuilles-Saint-Priest;ARPHEUILLES-SAINT-PRIEST;03420;03;ALLIER;1
03008;;Arronnes;ARRONNES;03250;03;ALLIER;1
03009;;Aubigny;AUBIGNY;03460;03;ALLIER;1
03010;;Audes;AUDES;03190;03;ALLIER;1
03011;;Aurour;AUROUER;03460;03;ALLIER;1
03012;;Autry-Issards;AUTRY-ISSARDS;03210;03;ALLIER;1
03013;;Avermes;AVERMES;03000;03;ALLIER;1
03014;;Avrilly;AVRILLY;03130;03;ALLIER;1
03015;;Bagneux;BAGNEUX;03460;03;ALLIER;1
03016;;Barberier;BARBERIER;03140;03;ALLIER;1
03017;;Barrais-Bussolles;BARRAIS-BUSSOLLES;03120;03;ALLIER;1
03018;;Bayet;BAYET;03500;03;ALLIER;1
03019;;Beaulon;BEAULON;03230;03;ALLIER;1
03020;;Beaune-d'Allier;BEAUNE-D'ALLIER;03390;03;ALLIER;1
03021;;Bgues;BEGUES;03800;03;ALLIER;1
03022;;Bellenaves;BELLENAVES;03330;03;ALLIER;1
03023;;Bellerive-sur-Allier;BELLERIVE-SUR-ALLIER;03700;03;ALLIER;1
03024;;Bert;BERT;03130;03;ALLIER;1
03025;;Bessay-sur-Allier;BESSAY-SUR-ALLIER;03340;03;ALLIER;1
03026;;Besson;BESSON;03210;03;ALLIER;1
03027;;Bzenet;BEZENET;03170;03;ALLIER;1
03028;;Billezois;BILLEZOIS;03120;03;ALLIER;1
03029;;Billy;BILLY;03260;03;ALLIER;1
03030;;Biozat;BIOZAT;03800;03;ALLIER;1
03031;;Bizeneuille;BIZENEUILLE;03170;03;ALLIER;1
03032;;Blomard;BLOMARD;03390;03;ALLIER;1
03033;;Bost;BOST;03300;03;ALLIER;1
03034;;Bouc;BOUCE;03150;03;ALLIER;1
03035;Le;Bouchaud;BOUCHAUD;03130;03;ALLIER;1
03036;;Bourbon-l'Archambault;BOURBON-L'ARCHAMBAULT;03160;03;ALLIER;1
03037;;Braize;BRAIZE;03360;03;ALLIER;1
03038;;Bransat;BRANSAT;03500;03;ALLIER;1
03039;;Bresnay;BRESNAY;03210;03;ALLIER;1
03040;;Bressolles;BRESSOLLES;03000;03;ALLIER;1
03041;Le;Brethon;BRETHON;03350;03;ALLIER;1
03042;Le;Breuil;BREUIL;03120;03;ALLIER;1
03043;;Brot-Vernet;BROUT-VERNET;03110;03;ALLIER;1
03044;;Brugheas;BRUGHEAS;03700;03;ALLIER;1
03045;;Busset;BUSSET;03270;03;ALLIER;1
03046;;Buxires-les-Mines;BUXIERES-LES-MINES;03440;03;ALLIER;1
03047;La;Celle;CELLE;03600;03;ALLIER;1
03048;;Crilly;CERILLY;03350;03;ALLIER;1
03049;;Cesset;CESSET;03500;03;ALLIER;1
03050;La;Chabanne;CHABANNE;03250;03;ALLIER;1
03051;;Chambrat;CHAMBERAT;03370;03;ALLIER;1
03052;;Chamblet;CHAMBLET;03170;03;ALLIER;1
03053;;Chantelle;CHANTELLE;03140;03;ALLIER;1
03054;;Chapeau;CHAPEAU;03340;03;ALLIER;1
03055;La;Chapelaude;CHAPELAUDE;03380;03;ALLIER;1
03056;La;Chapelle;CHAPELLE;03300;03;ALLIER;1
03057;La;Chapelle-aux-Chasses;CHAPELLE-AUX-CHASSES;03230;03;ALLIER;1
03058;;Chappes;CHAPPES;03390;03;ALLIER;1
03059;;Chareil-Cintrat;CHAREIL-CINTRAT;03140;03;ALLIER;1
03060;;Charmeil;CHARMEIL;03110;03;ALLIER;1
03061;;Charmes;CHARMES;03800;03;ALLIER;1
03062;;Charroux;CHARROUX;03140;03;ALLIER;1
03063;;Chassenard;CHASSENARD;03510;03;ALLIER;1
03064;;Chteau-sur-Allier;CHATEAU-SUR-ALLIER;03320;03;ALLIER;1
03065;;Chtel-de-Neuvre;CHATEL-DE-NEUVRE;03500;03;ALLIER;1
03066;;Chtel-Montagne;CHATEL-MONTAGNE;03250;03;ALLIER;1
03067;;Chtelperron;CHATELPERRON;03220;03;ALLIER;1
03068;;Chtelus;CHATELUS;03640;03;ALLIER;1
03069;;Chtillon;CHATILLON;03210;03;ALLIER;1
03070;;Chavenon;CHAVENON;03440;03;ALLIER;1
03071;;Chavroches;CHAVROCHES;03220;03;ALLIER;1
03072;;Chazemais;CHAZEMAIS;03370;03;ALLIER;1
03073;;Chemilly;CHEMILLY;03210;03;ALLIER;1
03074;;Chevagnes;CHEVAGNES;03230;03;ALLIER;1
03075;;Chezelle;CHEZELLE;03140;03;ALLIER;1
03076;;Chzy;CHEZY;03230;03;ALLIER;1
03077;;Chirat-l'glise;CHIRAT-L'EGLISE;03330;03;ALLIER;1
03078;;Chouvigny;CHOUVIGNY;03450;03;ALLIER;1
03079;;Cindr;CINDRE;03220;03;ALLIER;1
03080;;Cognat-Lyonne;COGNAT-LYONNE;03110;03;ALLIER;1
03081;;Colombier;COLOMBIER;03600;03;ALLIER;1
03082;;Commentry;COMMENTRY;03600;03;ALLIER;1
03083;;Contigny;CONTIGNY;03500;03;ALLIER;1
03084;;Cosne-d'Allier;COSNE-D'ALLIER;03430;03;ALLIER;1
03085;;Coulandon;COULANDON;03000;03;ALLIER;1
03086;;Coulanges;COULANGES;03470;03;ALLIER;1
03087;;Couleuvre;COULEUVRE;03320;03;ALLIER;1
03088;;Courais;COURCAIS;03370;03;ALLIER;1
03089;;Coutansouze;COUTANSOUZE;03330;03;ALLIER;1
03090;;Couzon;COUZON;03160;03;ALLIER;1
03091;;Crchy;CRECHY;03150;03;ALLIER;1
03092;;Cressanges;CRESSANGES;03240;03;ALLIER;1
03093;;Creuzier-le-Neuf;CREUZIER-LE-NEUF;03300;03;ALLIER;1
03094;;Creuzier-le-Vieux;CREUZIER-LE-VIEUX;03300;03;ALLIER;1
03095;;Cusset;CUSSET;03300;03;ALLIER;1
03096;;Deneuille-ls-Chantelle;DENEUILLE-LES-CHANTELLE;03140;03;ALLIER;1
03097;;Deneuille-les-Mines;DENEUILLE-LES-MINES;03170;03;ALLIER;1
03098;;Dsertines;DESERTINES;03630;03;ALLIER;1
03099;;Deux-Chaises;DEUX-CHAISES;03240;03;ALLIER;1
03100;;Diou;DIOU;03290;03;ALLIER;1
03101;;Domrat;DOMERAT;03410;03;ALLIER;1
03102;;Dompierre-sur-Besbre;DOMPIERRE-SUR-BESBRE;03290;03;ALLIER;1
03103;Le;Donjon;DONJON;03130;03;ALLIER;1
03104;;Doyet;DOYET;03170;03;ALLIER;1
03105;;Droiturier;DROITURIER;03120;03;ALLIER;1
03106;;Durdat-Larequille;DURDAT-LAREQUILLE;03310;03;ALLIER;1
03107;;breuil;EBREUIL;03450;03;ALLIER;1
03108;;chassires;ECHASSIERES;03330;03;ALLIER;1
03109;;Escurolles;ESCUROLLES;03110;03;ALLIER;1
03110;;Espinasse-Vozelle;ESPINASSE-VOZELLE;03110;03;ALLIER;1
03111;;Estivareilles;ESTIVAREILLES;03190;03;ALLIER;1
03112;;troussat;ETROUSSAT;03140;03;ALLIER;1
03113;;Ferrires-sur-Sichon;FERRIERES-SUR-SICHON;03250;03;ALLIER;1
03114;La;Fert-Hauterive;FERTE-HAUTERIVE;03340;03;ALLIER;1
03115;;Fleuriel;FLEURIEL;03140;03;ALLIER;1
03116;;Fourilles;FOURILLES;03140;03;ALLIER;1
03117;;Franchesse;FRANCHESSE;03160;03;ALLIER;1
03118;;Gannat;GANNAT;03800;03;ALLIER;1
03119;;Gannay-sur-Loire;GANNAY-SUR-LOIRE;03230;03;ALLIER;1
03120;;Garnat-sur-Engivre;GARNAT-SUR-ENGIEVRE;03230;03;ALLIER;1
03121;;Gennetines;GENNETINES;03400;03;ALLIER;1
03122;;Gipcy;GIPCY;03210;03;ALLIER;1
03123;;Givarlais;GIVARLAIS;03190;03;ALLIER;1
03124;;Gouise;GOUISE;03340;03;ALLIER;1
03125;La;Guillermie;GUILLERMIE;03250;03;ALLIER;1
03126;;Hauterive;HAUTERIVE;03270;03;ALLIER;1
03127;;Hrisson;HERISSON;03190;03;ALLIER;1
03128;;Huriel;HURIEL;03380;03;ALLIER;1
03129;;Hyds;HYDS;03600;03;ALLIER;1
03130;;Isle-et-Bardais;ISLE-ET-BARDAIS;03360;03;ALLIER;1
03131;;Isserpent;ISSERPENT;03120;03;ALLIER;1
03132;;Jaligny-sur-Besbre;JALIGNY-SUR-BESBRE;03220;03;ALLIER;1
03133;;Jenzat;JENZAT;03800;03;ALLIER;1
03134;;Lafline;LAFELINE;03500;03;ALLIER;1
03135;;Lalizolle;LALIZOLLE;03450;03;ALLIER;1
03136;;Lamaids;LAMAIDS;03380;03;ALLIER;1
03137;;Langy;LANGY;03150;03;ALLIER;1
03138;;Lapalisse;LAPALISSE;03120;03;ALLIER;1
03139;;Laprugne;LAPRUGNE;03250;03;ALLIER;1
03140;;Lavault-Sainte-Anne;LAVAULT-SAINTE-ANNE;03310;03;ALLIER;1
03141;;Lavoine;LAVOINE;03250;03;ALLIER;1
03142;;Lenax;LENAX;03130;03;ALLIER;1
03143;;Ltelon;LETELON;03360;03;ALLIER;1
03144;;Liernolles;LIERNOLLES;03130;03;ALLIER;1
03145;;Lignerolles;LIGNEROLLES;03410;03;ALLIER;1
03146;;Limoise;LIMOISE;03320;03;ALLIER;1
03147;;Loddes;LODDES;03130;03;ALLIER;1
03148;;Loriges;LORIGES;03500;03;ALLIER;1
03149;;Louchy-Montfand;LOUCHY-MONTFAND;03500;03;ALLIER;1
03150;;Louroux-Bourbonnais;LOUROUX-BOURBONNAIS;03350;03;ALLIER;1
03151;;Louroux-de-Beaune;LOUROUX-DE-BEAUNE;03600;03;ALLIER;1
03152;;Louroux-de-Bouble;LOUROUX-DE-BOUBLE;03330;03;ALLIER;1
03153;;Louroux-Hodement;LOUROUX-HODEMENT;03190;03;ALLIER;1
03154;;Luneau;LUNEAU;03130;03;ALLIER;1
03155;;Lurcy-Lvis;LURCY-LEVIS;03320;03;ALLIER;1
03156;;Lusigny;LUSIGNY;03230;03;ALLIER;1
03157;;Magnet;MAGNET;03260;03;ALLIER;1
03158;;Maillet;MAILLET;03190;03;ALLIER;1
03159;;Malicorne;MALICORNE;03600;03;ALLIER;1
03160;;Marcenat;MARCENAT;03260;03;ALLIER;1
03161;;Marcillat-en-Combraille;MARCILLAT-EN-COMBRAILLE;03420;03;ALLIER;1
03162;;Marigny;MARIGNY;03210;03;ALLIER;1
03163;;Mariol;MARIOL;03270;03;ALLIER;1
03164;Le;Mayet-d'cole;MAYET-D'ECOLE;03800;03;ALLIER;1
03165;Le;Mayet-de-Montagne;MAYET-DE-MONTAGNE;03250;03;ALLIER;1
03166;;Mazerier;MAZERIER;03800;03;ALLIER;1
03167;;Mazirat;MAZIRAT;03420;03;ALLIER;1
03168;;Meaulne;MEAULNE;03360;03;ALLIER;1
03169;;Meillard;MEILLARD;03500;03;ALLIER;1
03170;;Meillers;MEILLERS;03210;03;ALLIER;1
03171;;Mercy;MERCY;03340;03;ALLIER;1
03172;;Mesples;MESPLES;03370;03;ALLIER;1
03173;;Molinet;MOLINET;03510;03;ALLIER;1
03174;;Molles;MOLLES;03300;03;ALLIER;1
03175;;Monestier;MONESTIER;03140;03;ALLIER;1
03176;;Montay-sur-Allier;MONETAY-SUR-ALLIER;03500;03;ALLIER;1
03177;;Montay-sur-Loire;MONETAY-SUR-LOIRE;03470;03;ALLIER;1
03178;;Montaigut-en-Forez;MONTAIGUET-EN-FOREZ;03130;03;ALLIER;1
03179;;Montaigu-le-Blin;MONTAIGU-LE-BLIN;03150;03;ALLIER;1
03180;;Montbeugny;MONTBEUGNY;03340;03;ALLIER;1
03181;;Montcombroux-les-Mines;MONTCOMBROUX-LES-MINES;03130;03;ALLIER;1
03182;;Monteignet-sur-l'Andelot;MONTEIGNET-SUR-L'ANDELOT;03800;03;ALLIER;1
03183;Le;Montet;MONTET;03240;03;ALLIER;1
03184;;Montilly;MONTILLY;03000;03;ALLIER;1
03185;;Montluon;MONTLUCON;03100;03;ALLIER;1
03186;;Montmarault;MONTMARAULT;03390;03;ALLIER;1
03187;;Montoldre;MONTOLDRE;03150;03;ALLIER;1
03188;;Montord;MONTORD;03500;03;ALLIER;1
03189;;Montvicq;MONTVICQ;03170;03;ALLIER;1
03190;;Moulins;MOULINS;03000;03;ALLIER;1
03191;;Murat;MURAT;03390;03;ALLIER;1
03192;;Nades;NADES;03450;03;ALLIER;1
03193;;Nassigny;NASSIGNY;03190;03;ALLIER;1
03194;;Naves;NAVES;03330;03;ALLIER;1
03195;;Nris-les-Bains;NERIS-LES-BAINS;03310;03;ALLIER;1
03196;;Neuilly-en-Donjon;NEUILLY-EN-DONJON;03130;03;ALLIER;1
03197;;Neuilly-le-Ral;NEUILLY-LE-REAL;03340;03;ALLIER;1
03198;;Neure;NEURE;03320;03;ALLIER;1
03200;;Neuvy;NEUVY;03000;03;ALLIER;1
03201;;Nizerolles;NIZEROLLES;03250;03;ALLIER;1
03202;;Noyant-d'Allier;NOYANT-D'ALLIER;03210;03;ALLIER;1
03203;;Paray-le-Frsil;PARAY-LE-FRESIL;03230;03;ALLIER;1
03204;;Paray-sous-Briailles;PARAY-SOUS-BRIAILLES;03500;03;ALLIER;1
03205;;Prigny;PERIGNY;03120;03;ALLIER;1
03206;La;Petite-Marche;PETITE-MARCHE;03420;03;ALLIER;1
03207;;Pierrefitte-sur-Loire;PIERREFITTE-SUR-LOIRE;03470;03;ALLIER;1
03208;Le;Pin;PIN;03130;03;ALLIER;1
03209;;Pozat;POEZAT;03800;03;ALLIER;1
03210;;Pouzy-Msangy;POUZY-MESANGY;03320;03;ALLIER;1
03211;;Prmilhat;PREMILHAT;03410;03;ALLIER;1
03212;;Quinssaines;QUINSSAINES;03380;03;ALLIER;1
03213;;Reugny;REUGNY;03190;03;ALLIER;1
03214;;Rocles;ROCLES;03240;03;ALLIER;1
03215;;Rongres;RONGERES;03150;03;ALLIER;1
03216;;Ronnet;RONNET;03420;03;ALLIER;1
03217;;Saint-Angel;SAINT-ANGEL;03170;03;ALLIER;1
03218;;Saint-Aubin-le-Monial;SAINT-AUBIN-LE-MONIAL;03160;03;ALLIER;1
03219;;Saint-Bonnet-de-Four;SAINT-BONNET-DE-FOUR;03390;03;ALLIER;1
03220;;Saint-Bonnet-de-Rochefort;SAINT-BONNET-DE-ROCHEFORT;03800;03;ALLIER;1
03221;;Saint-Bonnet-Tronais;SAINT-BONNET-TRONCAIS;03360;03;ALLIER;1
03222;;Saint-Caprais;SAINT-CAPRAIS;03190;03;ALLIER;1
03223;;Saint-Christophe;SAINT-CHRISTOPHE;03120;03;ALLIER;1
03224;;Saint-Clment;SAINT-CLEMENT;03250;03;ALLIER;1
03225;;Saint-Dsir;SAINT-DESIRE;03370;03;ALLIER;1
03226;;Saint-Didier-en-Donjon;SAINT-DIDIER-EN-DONJON;03130;03;ALLIER;1
03227;;Saint-Didier-la-Fort;SAINT-DIDIER-LA-FORET;03110;03;ALLIER;1
03228;;Saint-loy-d'Allier;SAINT-ELOY-D'ALLIER;03370;03;ALLIER;1
03229;;Saint-Ennemond;SAINT-ENNEMOND;03400;03;ALLIER;1
03230;;Saint-tienne-de-Vicq;SAINT-ETIENNE-DE-VICQ;03300;03;ALLIER;1
03231;;Saint-Fargeol;SAINT-FARGEOL;03420;03;ALLIER;1
03232;;Saint-Flix;SAINT-FELIX;03260;03;ALLIER;1
03233;;Saint-Genest;SAINT-GENEST;03310;03;ALLIER;1
03234;;Saint-Grand-de-Vaux;SAINT-GERAND-DE-VAUX;03340;03;ALLIER;1
03235;;Saint-Grand-le-Puy;SAINT-GERAND-LE-PUY;03150;03;ALLIER;1
03236;;Saint-Germain-des-Fosss;SAINT-GERMAIN-DES-FOSSES;03260;03;ALLIER;1
03237;;Saint-Germain-de-Salles;SAINT-GERMAIN-DE-SALLES;03140;03;ALLIER;1
03238;;Saint-Hilaire;SAINT-HILAIRE;03440;03;ALLIER;1
03239;;Saint-Lger-sur-Vouzance;SAINT-LEGER-SUR-VOUZANCE;03130;03;ALLIER;1
03240;;Saint-Lon;SAINT-LEON;03220;03;ALLIER;1
03241;;Saint-Lopardin-d'Augy;SAINT-LEOPARDIN-D'AUGY;03160;03;ALLIER;1
03242;;Saint-Loup;SAINT-LOUP;03150;03;ALLIER;1
03243;;Saint-Marcel-en-Murat;SAINT-MARCEL-EN-MURAT;03390;03;ALLIER;1
03244;;Saint-Marcel-en-Marcillat;SAINT-MARCEL-EN-MARCILLAT;03420;03;ALLIER;1
03245;;Saint-Martin-des-Lais;SAINT-MARTIN-DES-LAIS;03230;03;ALLIER;1
03246;;Saint-Martinien;SAINT-MARTINIEN;03380;03;ALLIER;1
03247;;Saint-Menoux;SAINT-MENOUX;03210;03;ALLIER;1
03248;;Saint-Nicolas-des-Biefs;SAINT-NICOLAS-DES-BIEFS;03250;03;ALLIER;1
03249;;Saint-Palais;SAINT-PALAIS;03370;03;ALLIER;1
03250;;Saint-Pierre-Laval;SAINT-PIERRE-LAVAL;42620;03;LOIRE;1
03251;;Saint-Plaisir;SAINT-PLAISIR;03160;03;ALLIER;1
03252;;Saint-Pont;SAINT-PONT;03110;03;ALLIER;1
03253;;Saint-Pourain-sur-Besbre;SAINT-POURCAIN-SUR-BESBRE;03290;03;ALLIER;1
03254;;Saint-Pourain-sur-Sioule;SAINT-POURCAIN-SUR-SIOULE;03500;03;ALLIER;1
03255;;Saint-Priest-d'Andelot;SAINT-PRIEST-D'ANDELOT;03800;03;ALLIER;1
03256;;Saint-Priest-en-Murat;SAINT-PRIEST-EN-MURAT;03390;03;ALLIER;1
03257;;Saint-Prix;SAINT-PRIX;03120;03;ALLIER;1
03258;;Saint-Rmy-en-Rollat;SAINT-REMY-EN-ROLLAT;03110;03;ALLIER;1
03259;;Saint-Sauvier;SAINT-SAUVIER;03370;03;ALLIER;1
03260;;Saint-Sornin;SAINT-SORNIN;03240;03;ALLIER;1
03261;;Sainte-Thrence;SAINTE-THERENCE;03420;03;ALLIER;1
03262;;Saint-Victor;SAINT-VICTOR;03410;03;ALLIER;1
03263;;Saint-Voir;SAINT-VOIR;03220;03;ALLIER;1
03264;;Saint-Yorre;SAINT-YORRE;03270;03;ALLIER;1
03265;;Saligny-sur-Roudon;SALIGNY-SUR-ROUDON;03470;03;ALLIER;1
03266;;Sanssat;SANSSAT;03150;03;ALLIER;1
03267;;Saulcet;SAULCET;03500;03;ALLIER;1
03268;;Saulzet;SAULZET;03800;03;ALLIER;1
03269;;Sauvagny;SAUVAGNY;03430;03;ALLIER;1
03270;;Sazeret;SAZERET;03390;03;ALLIER;1
03271;;Serbannes;SERBANNES;03110;03;ALLIER;1
03272;;Servilly;SERVILLY;03120;03;ALLIER;1
03273;;Seuillet;SEUILLET;03260;03;ALLIER;1
03274;;Sorbier;SORBIER;03220;03;ALLIER;1
03275;;Souvigny;SOUVIGNY;03210;03;ALLIER;1
03276;;Sussat;SUSSAT;03450;03;ALLIER;1
03277;;Target;TARGET;03140;03;ALLIER;1
03278;;Taxat-Senat;TAXAT-SENAT;03140;03;ALLIER;1
03279;;Teillet-Argenty;TEILLET-ARGENTY;03410;03;ALLIER;1
03280;;Terjat;TERJAT;03420;03;ALLIER;1
03281;Le;Theil;THEIL;03240;03;ALLIER;1
03282;;Theneuille;THENEUILLE;03350;03;ALLIER;1
03283;;Thiel-sur-Acolin;THIEL-SUR-ACOLIN;03230;03;ALLIER;1
03284;;Thionne;THIONNE;03220;03;ALLIER;1
03285;;Tortezais;TORTEZAIS;03430;03;ALLIER;1
03286;;Toulon-sur-Allier;TOULON-SUR-ALLIER;03400;03;ALLIER;1
03287;;Treban;TREBAN;03240;03;ALLIER;1
03288;;Treignat;TREIGNAT;03380;03;ALLIER;1
03289;;Treteau;TRETEAU;03220;03;ALLIER;1
03290;;Trvol;TREVOL;03460;03;ALLIER;1
03291;;Trzelles;TREZELLES;03220;03;ALLIER;1
03292;;Tronget;TRONGET;03240;03;ALLIER;1
03293;;Uray;URCAY;03360;03;ALLIER;1
03294;;Ussel-d'Allier;USSEL-D'ALLIER;03140;03;ALLIER;1
03295;;Valignat;VALIGNAT;03330;03;ALLIER;1
03296;;Valigny;VALIGNY;03360;03;ALLIER;1
03297;;Vallon-en-Sully;VALLON-EN-SULLY;03190;03;ALLIER;1
03298;;Varennes-sur-Allier;VARENNES-SUR-ALLIER;03150;03;ALLIER;1
03299;;Varennes-sur-Tche;VARENNES-SUR-TECHE;03220;03;ALLIER;1
03300;;Vaumas;VAUMAS;03220;03;ALLIER;1
03301;;Vaux;VAUX;03190;03;ALLIER;1
03302;;Veauce;VEAUCE;03450;03;ALLIER;1
03303;;Venas;VENAS;03190;03;ALLIER;1
03304;;Vendat;VENDAT;03110;03;ALLIER;1
03305;;Verneix;VERNEIX;03190;03;ALLIER;1
03306;Le;Vernet;VERNET;03200;03;ALLIER;1
03307;;Verneuil-en-Bourbonnais;VERNEUIL-EN-BOURBONNAIS;03500;03;ALLIER;1
03308;;Vernusse;VERNUSSE;03390;03;ALLIER;1
03309;Le;Veurdre;VEURDRE;03320;03;ALLIER;1
03310;;Vichy;VICHY;03200;03;ALLIER;1
03311;;Vicq;VICQ;03450;03;ALLIER;1
03312;;Vieure;VIEURE;03430;03;ALLIER;1
03313;Le;Vilhain;VILHAIN;03350;03;ALLIER;1
03314;;Villebret;VILLEBRET;03310;03;ALLIER;1
03315;;Villefranche-d'Allier;VILLEFRANCHE-D'ALLIER;03430;03;ALLIER;1
03316;;Villeneuve-sur-Allier;VILLENEUVE-SUR-ALLIER;03460;03;ALLIER;1
03317;;Viplaix;VIPLAIX;03370;03;ALLIER;1
03318;;Vitray;VITRAY;03360;03;ALLIER;1
03319;;Voussac;VOUSSAC;03140;03;ALLIER;1
03320;;Ygrande;YGRANDE;03160;03;ALLIER;1
03321;;Yzeure;YZEURE;03400;03;ALLIER;1
04001;;Aiglun;AIGLUN;04510;04;ALPES DE HAUTE PROVENCE;1
04004;;Allemagne-en-Provence;ALLEMAGNE-EN-PROVENCE;04550;04;ALPES DE HAUTE PROVENCE;1
04005;;Allons;ALLONS;04170;04;ALPES DE HAUTE PROVENCE;1
04006;;Allos;ALLOS;04260;04;ALPES DE HAUTE PROVENCE;1
04007;;Angles;ANGLES;04170;04;ALPES DE HAUTE PROVENCE;1
04008;;Annot;ANNOT;04240;04;ALPES DE HAUTE PROVENCE;1
04009;;Archail;ARCHAIL;04420;04;ALPES DE HAUTE PROVENCE;1
04012;;Aubenas-les-Alpes;AUBENAS-LES-ALPES;04110;04;ALPES DE HAUTE PROVENCE;1
04013;;Aubignosc;AUBIGNOSC;04200;04;ALPES DE HAUTE PROVENCE;1
04016;;Authon;AUTHON;04200;04;ALPES DE HAUTE PROVENCE;1
04017;;Auzet;AUZET;04140;04;ALPES DE HAUTE PROVENCE;1
04018;;Banon;BANON;04150;04;ALPES DE HAUTE PROVENCE;1
04019;;Barcelonnette;BARCELONNETTE;04400;04;ALPES DE HAUTE PROVENCE;1
04020;;Barles;BARLES;04140;04;ALPES DE HAUTE PROVENCE;1
04021;;Barras;BARRAS;04380;04;ALPES DE HAUTE PROVENCE;1
04022;;Barrme;BARREME;04330;04;ALPES DE HAUTE PROVENCE;1
04023;;Bayons;BAYONS;04250;04;ALPES DE HAUTE PROVENCE;1
04024;;Beaujeu;BEAUJEU;04420;04;ALPES DE HAUTE PROVENCE;1
04025;;Beauvezer;BEAUVEZER;04370;04;ALPES DE HAUTE PROVENCE;1
04026;;Bellaffaire;BELLAFFAIRE;04250;04;ALPES DE HAUTE PROVENCE;1
04027;;Bevons;BEVONS;04200;04;ALPES DE HAUTE PROVENCE;1
04028;;Beynes;BEYNES;04270;04;ALPES DE HAUTE PROVENCE;1
04030;;Blieux;BLIEUX;04330;04;ALPES DE HAUTE PROVENCE;1
04031;;Bras-d'Asse;BRAS-D'ASSE;04270;04;ALPES DE HAUTE PROVENCE;1
04032;;Braux;BRAUX;04240;04;ALPES DE HAUTE PROVENCE;1
04033;La;Brole;BREOLE;04340;04;ALPES DE HAUTE PROVENCE;1
04034;La;Brillanne;BRILLANNE;04700;04;ALPES DE HAUTE PROVENCE;1
04035;;Brunet;BRUNET;04210;04;ALPES DE HAUTE PROVENCE;1
04036;Le;Brusquet;BRUSQUET;04420;04;ALPES DE HAUTE PROVENCE;1
04037;Le;Caire;CAIRE;04250;04;ALPES DE HAUTE PROVENCE;1
04039;;Castellane;CASTELLANE;04120;04;ALPES DE HAUTE PROVENCE;1
04040;Le;Castellard-Mlan;CASTELLARD-MELAN;04380;04;ALPES DE HAUTE PROVENCE;1
04041;Le;Castellet;CASTELLET;04700;04;ALPES DE HAUTE PROVENCE;1
04042;;Castellet-ls-Sausses;CASTELLET-LES-SAUSSES;04320;04;ALPES DE HAUTE PROVENCE;1
04043;;Val-de-Chalvagne;VAL-DE-CHALVAGNE;04320;04;ALPES DE HAUTE PROVENCE;1
04045;;Creste;CERESTE;04280;04;ALPES DE HAUTE PROVENCE;1
04046;Le;Chaffaut-Saint-Jurson;CHAFFAUT-SAINT-JURSON;04510;04;ALPES DE HAUTE PROVENCE;1
04047;;Champtercier;CHAMPTERCIER;04660;04;ALPES DE HAUTE PROVENCE;1
04049;;Chteau-Arnoux-Saint-Auban;CHATEAU-ARNOUX-SAINT-AUBAN;04160;04;ALPES DE HAUTE PROVENCE;1
04050;;Chteaufort;CHATEAUFORT;04250;04;ALPES DE HAUTE PROVENCE;1
04051;;Chteauneuf-Miravail;CHATEAUNEUF-MIRAVAIL;04200;04;ALPES DE HAUTE PROVENCE;1
04053;;Chteauneuf-Val-Saint-Donat;CHATEAUNEUF-VAL-SAINT-DONAT;04200;04;ALPES DE HAUTE PROVENCE;1
04054;;Chteauredon;CHATEAUREDON;04270;04;ALPES DE HAUTE PROVENCE;1
04055;;Chaudon-Norante;CHAUDON-NORANTE;04330;04;ALPES DE HAUTE PROVENCE;1
04057;;Clamensane;CLAMENSANE;04250;04;ALPES DE HAUTE PROVENCE;1
04058;;Claret;CLARET;05110;04;HAUTES ALPES;1
04059;;Clumanc;CLUMANC;04330;04;ALPES DE HAUTE PROVENCE;1
04061;;Colmars;COLMARS;04370;04;ALPES DE HAUTE PROVENCE;1
04062;La;Condamine-Chtelard;CONDAMINE-CHATELARD;04530;04;ALPES DE HAUTE PROVENCE;1
04063;;Corbires;CORBIERES;04220;04;ALPES DE HAUTE PROVENCE;1
04065;;Cruis;CRUIS;04230;04;ALPES DE HAUTE PROVENCE;1
04066;;Curbans;CURBANS;05110;04;HAUTES ALPES;1
04067;;Curel;CUREL;04200;04;ALPES DE HAUTE PROVENCE;1
04068;;Dauphin;DAUPHIN;04300;04;ALPES DE HAUTE PROVENCE;1
04069;;Demandolx;DEMANDOLX;04120;04;ALPES DE HAUTE PROVENCE;1
04070;;Digne-les-Bains;DIGNE-LES-BAINS;04000;04;ALPES DE HAUTE PROVENCE;1
04072;;Draix;DRAIX;04420;04;ALPES DE HAUTE PROVENCE;1
04073;;Enchastrayes;ENCHASTRAYES;04400;04;ALPES DE HAUTE PROVENCE;1
04074;;Entrages;ENTRAGES;04000;04;ALPES DE HAUTE PROVENCE;1
04075;;Entrepierres;ENTREPIERRES;04200;04;ALPES DE HAUTE PROVENCE;1
04076;;Entrevaux;ENTREVAUX;04320;04;ALPES DE HAUTE PROVENCE;1
04077;;Entrevennes;ENTREVENNES;04700;04;ALPES DE HAUTE PROVENCE;1
04079;L';Escale;ESCALE;04160;04;ALPES DE HAUTE PROVENCE;1
04081;;Esparron-de-Verdon;ESPARRON-DE-VERDON;04550;04;ALPES DE HAUTE PROVENCE;1
04084;;Estoublon;ESTOUBLON;04270;04;ALPES DE HAUTE PROVENCE;1
04085;;Faucon-du-Caire;FAUCON-DU-CAIRE;04250;04;ALPES DE HAUTE PROVENCE;1
04086;;Faucon-de-Barcelonnette;FAUCON-DE-BARCELONNETTE;04400;04;ALPES DE HAUTE PROVENCE;1
04087;;Fontienne;FONTIENNE;04230;04;ALPES DE HAUTE PROVENCE;1
04088;;Forcalquier;FORCALQUIER;04300;04;ALPES DE HAUTE PROVENCE;1
04090;Le;Fugeret;FUGERET;04240;04;ALPES DE HAUTE PROVENCE;1
04091;;Ganagobie;GANAGOBIE;04310;04;ALPES DE HAUTE PROVENCE;1
04092;La;Garde;GARDE;04120;04;ALPES DE HAUTE PROVENCE;1
04093;;Gigors;GIGORS;04250;04;ALPES DE HAUTE PROVENCE;1
04094;;Groux-les-Bains;GREOUX-LES-BAINS;04800;04;ALPES DE HAUTE PROVENCE;1
04095;L';Hospitalet;HOSPITALET;04150;04;ALPES DE HAUTE PROVENCE;1
04096;;Jausiers;JAUSIERS;04850;04;ALPES DE HAUTE PROVENCE;1
04097;La;Javie;JAVIE;04420;04;ALPES DE HAUTE PROVENCE;1
04099;;Lambruisse;LAMBRUISSE;04170;04;ALPES DE HAUTE PROVENCE;1
04100;;Larche;LARCHE;04540;04;ALPES DE HAUTE PROVENCE;1
04101;;Lardiers;LARDIERS;04230;04;ALPES DE HAUTE PROVENCE;1
04102;Le;Lauzet-Ubaye;LAUZET-UBAYE;04340;04;ALPES DE HAUTE PROVENCE;1
04104;;Limans;LIMANS;04300;04;ALPES DE HAUTE PROVENCE;1
04106;;Lurs;LURS;04700;04;ALPES DE HAUTE PROVENCE;1
04107;;Majastres;MAJASTRES;04270;04;ALPES DE HAUTE PROVENCE;1
04108;;Malijai;MALIJAI;04350;04;ALPES DE HAUTE PROVENCE;1
04109;;Mallefougasse-Augs;MALLEFOUGASSE-AUGES;04230;04;ALPES DE HAUTE PROVENCE;1
04110;;Mallemoisson;MALLEMOISSON;04510;04;ALPES DE HAUTE PROVENCE;1
04111;;Mane;MANE;04300;04;ALPES DE HAUTE PROVENCE;1
04112;;Manosque;MANOSQUE;04100;04;ALPES DE HAUTE PROVENCE;1
04113;;Marcoux;MARCOUX;04420;04;ALPES DE HAUTE PROVENCE;1
04115;;Mailles;MEAILLES;04240;04;ALPES DE HAUTE PROVENCE;1
04116;Les;Mes;MEES;04190;04;ALPES DE HAUTE PROVENCE;1
04118;;Melve;MELVE;04250;04;ALPES DE HAUTE PROVENCE;1
04120;;Meyronnes;MEYRONNES;04540;04;ALPES DE HAUTE PROVENCE;1
04121;;Mzel;MEZEL;04270;04;ALPES DE HAUTE PROVENCE;1
04122;;Mirabeau;MIRABEAU;04510;04;ALPES DE HAUTE PROVENCE;1
04123;;Mison;MISON;04200;04;ALPES DE HAUTE PROVENCE;1
04124;;Montagnac-Montpezat;MONTAGNAC-MONTPEZAT;04500;04;ALPES DE HAUTE PROVENCE;1
04126;;Montclar;MONTCLAR;04140;04;ALPES DE HAUTE PROVENCE;1
04127;;Montfort;MONTFORT;04600;04;ALPES DE HAUTE PROVENCE;1
04128;;Montfuron;MONTFURON;04110;04;ALPES DE HAUTE PROVENCE;1
04129;;Montjustin;MONTJUSTIN;04110;04;ALPES DE HAUTE PROVENCE;1
04130;;Montlaux;MONTLAUX;04230;04;ALPES DE HAUTE PROVENCE;1
04132;;Montsalier;MONTSALIER;04150;04;ALPES DE HAUTE PROVENCE;1
04133;;Moriez;MORIEZ;04170;04;ALPES DE HAUTE PROVENCE;1
04134;La;Motte-du-Caire;MOTTE-DU-CAIRE;04250;04;ALPES DE HAUTE PROVENCE;1
04135;;Moustiers-Sainte-Marie;MOUSTIERS-SAINTE-MARIE;04360;04;ALPES DE HAUTE PROVENCE;1
04136;La;Mure-Argens;MURE-ARGENS;04170;04;ALPES DE HAUTE PROVENCE;1
04137;;Nibles;NIBLES;04250;04;ALPES DE HAUTE PROVENCE;1
04138;;Niozelles;NIOZELLES;04300;04;ALPES DE HAUTE PROVENCE;1
04139;;Noyers-sur-Jabron;NOYERS-SUR-JABRON;04200;04;ALPES DE HAUTE PROVENCE;1
04140;Les;Omergues;OMERGUES;04200;04;ALPES DE HAUTE PROVENCE;1
04141;;Ongles;ONGLES;04230;04;ALPES DE HAUTE PROVENCE;1
04142;;Oppedette;OPPEDETTE;04110;04;ALPES DE HAUTE PROVENCE;1
04143;;Oraison;ORAISON;04700;04;ALPES DE HAUTE PROVENCE;1
04144;La;Palud-sur-Verdon;PALUD-SUR-VERDON;04120;04;ALPES DE HAUTE PROVENCE;1
04145;;Peipin;PEIPIN;04200;04;ALPES DE HAUTE PROVENCE;1
04148;;Peyroules;PEYROULES;04120;04;ALPES DE HAUTE PROVENCE;1
04149;;Peyruis;PEYRUIS;04310;04;ALPES DE HAUTE PROVENCE;1
04150;;Pigut;PIEGUT;05130;04;HAUTES ALPES;1
04151;;Pierrerue;PIERRERUE;04300;04;ALPES DE HAUTE PROVENCE;1
04152;;Pierrevert;PIERREVERT;04860;04;ALPES DE HAUTE PROVENCE;1
04154;;Pontis;PONTIS;05160;04;HAUTES ALPES;1
04155;;Prads-Haute-Blone;PRADS-HAUTE-BLEONE;04420;04;ALPES DE HAUTE PROVENCE;1
04156;;Puimichel;PUIMICHEL;04700;04;ALPES DE HAUTE PROVENCE;1
04157;;Puimoisson;PUIMOISSON;04410;04;ALPES DE HAUTE PROVENCE;1
04158;;Quinson;QUINSON;04500;04;ALPES DE HAUTE PROVENCE;1
04159;;Redortiers;REDORTIERS;04150;04;ALPES DE HAUTE PROVENCE;1
04160;;Reillanne;REILLANNE;04110;04;ALPES DE HAUTE PROVENCE;1
04161;;Molans-Revel;MEOLANS-REVEL;04340;04;ALPES DE HAUTE PROVENCE;1
04162;;Revest-des-Brousses;REVEST-DES-BROUSSES;04150;04;ALPES DE HAUTE PROVENCE;1
04163;;Revest-du-Bion;REVEST-DU-BION;04150;04;ALPES DE HAUTE PROVENCE;1
04164;;Revest-Saint-Martin;REVEST-SAINT-MARTIN;04230;04;ALPES DE HAUTE PROVENCE;1
04166;;Riez;RIEZ;04500;04;ALPES DE HAUTE PROVENCE;1
04167;La;Robine-sur-Galabre;ROBINE-SUR-GALABRE;04000;04;ALPES DE HAUTE PROVENCE;1
04169;La;Rochegiron;ROCHEGIRON;04150;04;ALPES DE HAUTE PROVENCE;1
04170;La;Rochette;ROCHETTE;06260;04;ALPES MARITIMES;1
04171;;Rougon;ROUGON;04120;04;ALPES DE HAUTE PROVENCE;1
04172;;Roumoules;ROUMOULES;04500;04;ALPES DE HAUTE PROVENCE;1
04173;;Saint-Andr-les-Alpes;SAINT-ANDRE-LES-ALPES;04170;04;ALPES DE HAUTE PROVENCE;1
04174;;Saint-Benot;SAINT-BENOIT;04240;04;ALPES DE HAUTE PROVENCE;1
04175;;Sainte-Croix--Lauze;SAINTE-CROIX-A-LAUZE;04110;04;ALPES DE HAUTE PROVENCE;1
04176;;Sainte-Croix-du-Verdon;SAINTE-CROIX-DU-VERDON;04500;04;ALPES DE HAUTE PROVENCE;1
04177;;Hautes-Duyes;HAUTES-DUYES;04380;04;ALPES DE HAUTE PROVENCE;1
04178;;Saint-tienne-les-Orgues;SAINT-ETIENNE-LES-ORGUES;04230;04;ALPES DE HAUTE PROVENCE;1
04179;;Saint-Geniez;SAINT-GENIEZ;04200;04;ALPES DE HAUTE PROVENCE;1
04180;;Saint-Jacques;SAINT-JACQUES;04330;04;ALPES DE HAUTE PROVENCE;1
04181;;Saint-Jeannet;SAINT-JEANNET;04270;04;ALPES DE HAUTE PROVENCE;1
04182;;Saint-Julien-d'Asse;SAINT-JULIEN-D'ASSE;04270;04;ALPES DE HAUTE PROVENCE;1
04183;;Saint-Julien-du-Verdon;SAINT-JULIEN-DU-VERDON;04170;04;ALPES DE HAUTE PROVENCE;1
04184;;Saint-Jurs;SAINT-JURS;04410;04;ALPES DE HAUTE PROVENCE;1
04186;;Saint-Laurent-du-Verdon;SAINT-LAURENT-DU-VERDON;04500;04;ALPES DE HAUTE PROVENCE;1
04187;;Saint-Lions;SAINT-LIONS;04330;04;ALPES DE HAUTE PROVENCE;1
04188;;Saint-Maime;SAINT-MAIME;04300;04;ALPES DE HAUTE PROVENCE;1
04189;;Saint-Martin-de-Brmes;SAINT-MARTIN-DE-BROMES;04800;04;ALPES DE HAUTE PROVENCE;1
04190;;Saint-Martin-les-Eaux;SAINT-MARTIN-LES-EAUX;04870;04;ALPES DE HAUTE PROVENCE;1
04191;;Saint-Martin-ls-Seyne;SAINT-MARTIN-LES-SEYNE;04460;04;ALPES DE HAUTE PROVENCE;1
04192;;Saint-Michel-l'Observatoire;SAINT-MICHEL-L'OBSERVATOIRE;04870;04;ALPES DE HAUTE PROVENCE;1
04193;;Saint-Paul-sur-Ubaye;SAINT-PAUL-SUR-UBAYE;04530;04;ALPES DE HAUTE PROVENCE;1
04194;;Saint-Pierre;SAINT-PIERRE;06260;04;ALPES MARITIMES;1
04195;;Saint-Pons;SAINT-PONS;04400;04;ALPES DE HAUTE PROVENCE;1
04197;;Sainte-Tulle;SAINTE-TULLE;04220;04;ALPES DE HAUTE PROVENCE;1
04198;;Saint-Vincent-les-Forts;SAINT-VINCENT-LES-FORTS;04340;04;ALPES DE HAUTE PROVENCE;1
04199;;Saint-Vincent-sur-Jabron;SAINT-VINCENT-SUR-JABRON;04200;04;ALPES DE HAUTE PROVENCE;1
04200;;Salignac;SALIGNAC;04290;04;ALPES DE HAUTE PROVENCE;1
04201;;Saumane;SAUMANE;04150;04;ALPES DE HAUTE PROVENCE;1
04202;;Sausses;SAUSSES;04320;04;ALPES DE HAUTE PROVENCE;1
04203;;Selonnet;SELONNET;04460;04;ALPES DE HAUTE PROVENCE;1
04204;;Senez;SENEZ;04330;04;ALPES DE HAUTE PROVENCE;1
04205;;Seyne;SEYNE;04140;04;ALPES DE HAUTE PROVENCE;1
04206;;Sigonce;SIGONCE;04300;04;ALPES DE HAUTE PROVENCE;1
04207;;Sigoyer;SIGOYER;04200;04;ALPES DE HAUTE PROVENCE;1
04208;;Simiane-la-Rotonde;SIMIANE-LA-ROTONDE;04150;04;ALPES DE HAUTE PROVENCE;1
04209;;Sisteron;SISTERON;04200;04;ALPES DE HAUTE PROVENCE;1
04210;;Soleilhas;SOLEILHAS;04120;04;ALPES DE HAUTE PROVENCE;1
04211;;Sourribes;SOURRIBES;04290;04;ALPES DE HAUTE PROVENCE;1
04214;;Tartonne;TARTONNE;04330;04;ALPES DE HAUTE PROVENCE;1
04216;;Thze;THEZE;04200;04;ALPES DE HAUTE PROVENCE;1
04217;;Thoard;THOARD;04380;04;ALPES DE HAUTE PROVENCE;1
04218;;Thorame-Basse;THORAME-BASSE;04170;04;ALPES DE HAUTE PROVENCE;1
04219;;Thorame-Haute;THORAME-HAUTE;04170;04;ALPES DE HAUTE PROVENCE;1
04220;Les;Thuiles;THUILES;04400;04;ALPES DE HAUTE PROVENCE;1
04222;;Turriers;TURRIERS;04250;04;ALPES DE HAUTE PROVENCE;1
04224;;Ubraye;UBRAYE;04240;04;ALPES DE HAUTE PROVENCE;1
04226;;Uvernet-Fours;UVERNET-FOURS;04400;04;ALPES DE HAUTE PROVENCE;1
04227;;Vachres;VACHERES;04110;04;ALPES DE HAUTE PROVENCE;1
04228;;Valavoire;VALAVOIRE;04250;04;ALPES DE HAUTE PROVENCE;1
04229;;Valbelle;VALBELLE;04200;04;ALPES DE HAUTE PROVENCE;1
04230;;Valensole;VALENSOLE;04210;04;ALPES DE HAUTE PROVENCE;1
04231;;Valernes;VALERNES;04200;04;ALPES DE HAUTE PROVENCE;1
04233;;Vaumeilh;VAUMEILH;04200;04;ALPES DE HAUTE PROVENCE;1
04234;;Venterol;VENTEROL;05130;04;HAUTES ALPES;1
04235;;Verdaches;VERDACHES;04140;04;ALPES DE HAUTE PROVENCE;1
04236;;Vergons;VERGONS;04170;04;ALPES DE HAUTE PROVENCE;1
04237;Le;Vernet;VERNET;04140;04;ALPES DE HAUTE PROVENCE;1
04240;;Villars-Colmars;VILLARS-COLMARS;04640;04;ALPES DE HAUTE PROVENCE;1
04241;;Villemus;VILLEMUS;04110;04;ALPES DE HAUTE PROVENCE;1
04242;;Villeneuve;VILLENEUVE;04180;04;ALPES DE HAUTE PROVENCE;1
04244;;Volonne;VOLONNE;04290;04;ALPES DE HAUTE PROVENCE;1
04245;;Volx;VOLX;04130;04;ALPES DE HAUTE PROVENCE;1
05001;;Abris;ABRIES;05460;05;HAUTES ALPES;1
05002;;Agnires-en-Dvoluy;AGNIERES-EN-DEVOLUY;05250;05;HAUTES ALPES;1
05003;;Aiguilles;AIGUILLES;05470;05;HAUTES ALPES;1
05004;;Ancelle;ANCELLE;05260;05;HAUTES ALPES;1
05005;;Antonaves;ANTONAVES;05300;05;HAUTES ALPES;1
05006;L';Argentire-la-Besse;ARGENTIERE-LA-BESSEE;05120;05;HAUTES ALPES;1
05007;;Arvieux;ARVIEUX;05350;05;HAUTES ALPES;1
05008;;Aspremont;ASPREMONT;05140;05;HAUTES ALPES;1
05009;;Aspres-ls-Corps;ASPRES-LES-CORPS;05800;05;HAUTES ALPES;1
05010;;Aspres-sur-Buch;ASPRES-SUR-BUECH;05140;05;HAUTES ALPES;1
05011;;Avanon;AVANCON;05230;05;HAUTES ALPES;1
05012;;Baratier;BARATIER;05200;05;HAUTES ALPES;1
05013;;Barcillonnette;BARCILLONNETTE;05110;05;HAUTES ALPES;1
05014;;Barret-sur-Mouge;BARRET-SUR-MEOUGE;05300;05;HAUTES ALPES;1
05016;La;Btie-Montsalon;BATIE-MONTSALEON;05700;05;HAUTES ALPES;1
05017;La;Btie-Neuve;BATIE-NEUVE;05230;05;HAUTES ALPES;1
05018;La;Btie-Vieille;BATIE-VIEILLE;05000;05;HAUTES ALPES;1
05019;La;Beaume;BEAUME;05140;05;HAUTES ALPES;1
05020;;Bnvent-et-Charbillac;BENEVENT-ET-CHARBILLAC;05500;05;HAUTES ALPES;1
05021;Le;Bersac;BERSAC;05700;05;HAUTES ALPES;1
05022;;Brziers;BREZIERS;05190;05;HAUTES ALPES;1
05023;;Brianon;BRIANCON;05100;05;HAUTES ALPES;1
05024;;Bruis;BRUIS;05150;05;HAUTES ALPES;1
05025;;Buissard;BUISSARD;05500;05;HAUTES ALPES;1
05026;;Ceillac;CEILLAC;05600;05;HAUTES ALPES;1
05027;;Cervires;CERVIERES;05100;05;HAUTES ALPES;1
05028;;Chabestan;CHABESTAN;05400;05;HAUTES ALPES;1
05029;;Chabottes;CHABOTTES;05260;05;HAUTES ALPES;1
05031;;Champcella;CHAMPCELLA;05310;05;HAUTES ALPES;1
05032;;Champolon;CHAMPOLEON;05260;05;HAUTES ALPES;1
05033;;Chanousse;CHANOUSSE;05700;05;HAUTES ALPES;1
05034;;Chteauneuf-de-Chabre;CHATEAUNEUF-DE-CHABRE;05300;05;HAUTES ALPES;1
05035;;Chteauneuf-d'Oze;CHATEAUNEUF-D'OZE;05400;05;HAUTES ALPES;1
05036;;Chteauroux-les-Alpes;CHATEAUROUX-LES-ALPES;05380;05;HAUTES ALPES;1
05037;;Chteauvieux;CHATEAUVIEUX;05000;05;HAUTES ALPES;1
05038;;Chteau-Ville-Vieille;CHATEAU-VILLE-VIEILLE;05460;05;HAUTES ALPES;1
05039;;Chauffayer;CHAUFFAYER;05800;05;HAUTES ALPES;1
05040;;Chorges;CHORGES;05230;05;HAUTES ALPES;1
05042;La;Cluse;CLUSE;05250;05;HAUTES ALPES;1
05043;Les;Costes;COSTES;05500;05;HAUTES ALPES;1
05044;;Crvoux;CREVOUX;05200;05;HAUTES ALPES;1
05045;;Crots;CROTS;05200;05;HAUTES ALPES;1
05046;;Embrun;EMBRUN;05200;05;HAUTES ALPES;1
05047;;ourres;EOURRES;26560;05;DROME;1
05048;L';pine;EPINE;05700;05;HAUTES ALPES;1
05049;;Esparron;ESPARRON;05110;05;HAUTES ALPES;1
05050;;Espinasses;ESPINASSES;05190;05;HAUTES ALPES;1
05051;;toile-Saint-Cyrice;ETOILE-SAINT-CYRICE;05700;05;HAUTES ALPES;1
05052;;Eygliers;EYGLIERS;05600;05;HAUTES ALPES;1
05053;;Eyguians;EYGUIANS;05300;05;HAUTES ALPES;1
05054;La;Fare-en-Champsaur;FARE-EN-CHAMPSAUR;05500;05;HAUTES ALPES;1
05055;La;Faurie;FAURIE;05140;05;HAUTES ALPES;1
05056;;Forest-Saint-Julien;FOREST-SAINT-JULIEN;05260;05;HAUTES ALPES;1
05057;;Fouillouse;FOUILLOUSE;05130;05;HAUTES ALPES;1
05058;;Freissinires;FREISSINIERES;05310;05;HAUTES ALPES;1
05059;La;Freissinouse;FREISSINOUSE;05000;05;HAUTES ALPES;1
05060;;Furmeyer;FURMEYER;05400;05;HAUTES ALPES;1
05061;;Gap;GAP;05000;05;HAUTES ALPES;1
05062;Le;Glaizil;GLAIZIL;05800;05;HAUTES ALPES;1
05063;La;Grave;GRAVE;05320;05;HAUTES ALPES;1
05064;La;Chapelle-en-Valgaudmar;CHAPELLE-EN-VALGAUDEMAR;05800;05;HAUTES ALPES;1
05065;;Guillestre;GUILLESTRE;05600;05;HAUTES ALPES;1
05066;La;Haute-Beaume;HAUTE-BEAUME;05140;05;HAUTES ALPES;1
05067;Les;Infournas;INFOURNAS;05500;05;HAUTES ALPES;1
05068;;Jarjayes;JARJAYES;05130;05;HAUTES ALPES;1
05069;;Lagrand;LAGRAND;05300;05;HAUTES ALPES;1
05070;;Laragne-Montglin;LARAGNE-MONTEGLIN;05300;05;HAUTES ALPES;1
05071;;Lardier-et-Valena;LARDIER-ET-VALENCA;05110;05;HAUTES ALPES;1
05072;;Laye;LAYE;05500;05;HAUTES ALPES;1
05073;;Lazer;LAZER;05300;05;HAUTES ALPES;1
05074;;Lettret;LETTRET;05130;05;HAUTES ALPES;1
05075;;Manteyer;MANTEYER;05400;05;HAUTES ALPES;1
05076;;Mreuil;MEREUIL;05700;05;HAUTES ALPES;1
05077;;Molines-en-Queyras;MOLINES-EN-QUEYRAS;05350;05;HAUTES ALPES;1
05078;;Montier-Allemont;MONETIER-ALLEMONT;05110;05;HAUTES ALPES;1
05079;Le;Montier-les-Bains;MONETIER-LES-BAINS;05220;05;HAUTES ALPES;1
05080;;Montbrand;MONTBRAND;05140;05;HAUTES ALPES;1
05081;;Montclus;MONTCLUS;05700;05;HAUTES ALPES;1
05082;;Mont-Dauphin;MONT-DAUPHIN;05600;05;HAUTES ALPES;1
05084;;Montgardin;MONTGARDIN;05230;05;HAUTES ALPES;1
05085;;Montgenvre;MONTGENEVRE;05100;05;HAUTES ALPES;1
05086;;Montjay;MONTJAY;05150;05;HAUTES ALPES;1
05087;;Montmaur;MONTMAUR;05400;05;HAUTES ALPES;1
05088;;Montmorin;MONTMORIN;05150;05;HAUTES ALPES;1
05089;;Montrond;MONTROND;05700;05;HAUTES ALPES;1
05090;La;Motte-en-Champsaur;MOTTE-EN-CHAMPSAUR;05500;05;HAUTES ALPES;1
05091;;Moydans;MOYDANS;05150;05;HAUTES ALPES;1
05092;;Neffes;NEFFES;05000;05;HAUTES ALPES;1
05093;;Nvache;NEVACHE;05100;05;HAUTES ALPES;1
05094;;Nossage-et-Bnvent;NOSSAGE-ET-BENEVENT;05700;05;HAUTES ALPES;1
05095;Le;Noyer;NOYER;05500;05;HAUTES ALPES;1
05096;;Orcires;ORCIERES;05170;05;HAUTES ALPES;1
05097;;Orpierre;ORPIERRE;05700;05;HAUTES ALPES;1
05098;Les;Orres;ORRES;05200;05;HAUTES ALPES;1
05099;;Oze;OZE;05400;05;HAUTES ALPES;1
05100;;Pelleautier;PELLEAUTIER;05000;05;HAUTES ALPES;1
05101;;Pelvoux;PELVOUX;05340;05;HAUTES ALPES;1
05102;La;Piarre;PIARRE;05700;05;HAUTES ALPES;1
05103;Le;Pot;POET;05300;05;HAUTES ALPES;1
05104;;Poligny;POLIGNY;05500;05;HAUTES ALPES;1
05106;;Prunires;PRUNIERES;05230;05;HAUTES ALPES;1
05107;;Puy-Saint-Andr;PUY-SAINT-ANDRE;05100;05;HAUTES ALPES;1
05108;;Puy-Saint-Eusbe;PUY-SAINT-EUSEBE;05200;05;HAUTES ALPES;1
05109;;Puy-Saint-Pierre;PUY-SAINT-PIERRE;05100;05;HAUTES ALPES;1
05110;;Puy-Saint-Vincent;PUY-SAINT-VINCENT;05290;05;HAUTES ALPES;1
05111;;Puy-Sanires;PUY-SANIERES;05200;05;HAUTES ALPES;1
05112;;Rabou;RABOU;05400;05;HAUTES ALPES;1
05113;;Rambaud;RAMBAUD;05000;05;HAUTES ALPES;1
05114;;Rallon;REALLON;05160;05;HAUTES ALPES;1
05115;;Remollon;REMOLLON;05190;05;HAUTES ALPES;1
05116;;Rotier;REOTIER;05600;05;HAUTES ALPES;1
05117;;Ribeyret;RIBEYRET;05150;05;HAUTES ALPES;1
05118;;Ribiers;RIBIERS;05300;05;HAUTES ALPES;1
05119;;Risoul;RISOUL;05600;05;HAUTES ALPES;1
05120;;Ristolas;RISTOLAS;05460;05;HAUTES ALPES;1
05121;;Rochebrune;ROCHEBRUNE;05190;05;HAUTES ALPES;1
05122;La;Roche-de-Rame;ROCHE-DE-RAME;05310;05;HAUTES ALPES;1
05123;La;Roche-des-Arnauds;ROCHE-DES-ARNAUDS;05400;05;HAUTES ALPES;1
05124;La;Rochette;ROCHETTE;05000;05;HAUTES ALPES;1
05126;;Rosans;ROSANS;05150;05;HAUTES ALPES;1
05127;;Rousset;ROUSSET;05190;05;HAUTES ALPES;1
05128;;Saint-Andr-d'Embrun;SAINT-ANDRE-D'EMBRUN;05200;05;HAUTES ALPES;1
05129;;Saint-Andr-de-Rosans;SAINT-ANDRE-DE-ROSANS;05150;05;HAUTES ALPES;1
05130;;Saint-Apollinaire;SAINT-APOLLINAIRE;05160;05;HAUTES ALPES;1
05131;;Saint-Auban-d'Oze;SAINT-AUBAN-D'OZE;05400;05;HAUTES ALPES;1
05132;;Saint-Bonnet-en-Champsaur;SAINT-BONNET-EN-CHAMPSAUR;05500;05;HAUTES ALPES;1
05133;;Saint-Chaffrey;SAINT-CHAFFREY;05330;05;HAUTES ALPES;1
05134;;Saint-Clment-sur-Durance;SAINT-CLEMENT-SUR-DURANCE;05600;05;HAUTES ALPES;1
05135;;Sainte-Colombe;SAINTE-COLOMBE;05700;05;HAUTES ALPES;1
05136;;Saint-Crpin;SAINT-CREPIN;05600;05;HAUTES ALPES;1
05138;;Saint-Disdier;SAINT-DISDIER;05250;05;HAUTES ALPES;1
05139;;Saint-tienne-en-Dvoluy;SAINT-ETIENNE-EN-DEVOLUY;05250;05;HAUTES ALPES;1
05140;;Saint-tienne-le-Laus;SAINT-ETIENNE-LE-LAUS;05130;05;HAUTES ALPES;1
05141;;Saint-Eusbe-en-Champsaur;SAINT-EUSEBE-EN-CHAMPSAUR;05500;05;HAUTES ALPES;1
05142;;Saint-Firmin;SAINT-FIRMIN;05800;05;HAUTES ALPES;1
05143;;Saint-Genis;SAINT-GENIS;05300;05;HAUTES ALPES;1
05144;;Saint-Jacques-en-Valgodemard;SAINT-JACQUES-EN-VALGODEMARD;05800;05;HAUTES ALPES;1
05145;;Saint-Jean-Saint-Nicolas;SAINT-JEAN-SAINT-NICOLAS;05260;05;HAUTES ALPES;1
05146;;Saint-Julien-en-Beauchne;SAINT-JULIEN-EN-BEAUCHENE;05140;05;HAUTES ALPES;1
05147;;Saint-Julien-en-Champsaur;SAINT-JULIEN-EN-CHAMPSAUR;05500;05;HAUTES ALPES;1
05148;;Saint-Laurent-du-Cros;SAINT-LAURENT-DU-CROS;05500;05;HAUTES ALPES;1
05149;;Saint-Lger-les-Mlzes;SAINT-LEGER-LES-MELEZES;05260;05;HAUTES ALPES;1
05150;;Sainte-Marie;SAINTE-MARIE;05150;05;HAUTES ALPES;1
05151;;Saint-Martin-de-Queyrires;SAINT-MARTIN-DE-QUEYRIERES;05120;05;HAUTES ALPES;1
05152;;Saint-Maurice-en-Valgodemard;SAINT-MAURICE-EN-VALGODEMARD;05800;05;HAUTES ALPES;1
05153;;Saint-Michel-de-Chaillol;SAINT-MICHEL-DE-CHAILLOL;05260;05;HAUTES ALPES;1
05154;;Saint-Pierre-d'Argenon;SAINT-PIERRE-D'ARGENCON;05140;05;HAUTES ALPES;1
05155;;Saint-Pierre-Avez;SAINT-PIERRE-AVEZ;05300;05;HAUTES ALPES;1
05156;;Saint-Sauveur;SAINT-SAUVEUR;05200;05;HAUTES ALPES;1
05157;;Saint-Vran;SAINT-VERAN;05350;05;HAUTES ALPES;1
05158;Le;Saix;SAIX;05400;05;HAUTES ALPES;1
05159;;Salon;SALEON;05300;05;HAUTES ALPES;1
05160;;Salrans;SALERANS;05300;05;HAUTES ALPES;1
05161;La;Salle-les-Alpes;SALLE-LES-ALPES;05240;05;HAUTES ALPES;1
05162;La;Saulce;SAULCE;05110;05;HAUTES ALPES;1
05163;Le;Sauze-du-Lac;SAUZE-DU-LAC;05160;05;HAUTES ALPES;1
05164;;Savines-le-Lac;SAVINES-LE-LAC;05160;05;HAUTES ALPES;1
05165;;Savournon;SAVOURNON;05700;05;HAUTES ALPES;1
05166;;Serres;SERRES;05700;05;HAUTES ALPES;1
05167;;Sigottier;SIGOTTIER;05700;05;HAUTES ALPES;1
05168;;Sigoyer;SIGOYER;05130;05;HAUTES ALPES;1
05169;;Sorbiers;SORBIERS;05150;05;HAUTES ALPES;1
05170;;Tallard;TALLARD;05130;05;HAUTES ALPES;1
05171;;Thus;THEUS;05190;05;HAUTES ALPES;1
05172;;Trescloux;TRESCLEOUX;05700;05;HAUTES ALPES;1
05173;;Upaix;UPAIX;05300;05;HAUTES ALPES;1
05174;;Val-des-Prs;VAL-DES-PRES;05100;05;HAUTES ALPES;1
05175;;Vallouise;VALLOUISE;05290;05;HAUTES ALPES;1
05176;;Valserres;VALSERRES;05130;05;HAUTES ALPES;1
05177;;Vars;VARS;05560;05;HAUTES ALPES;1
05178;;Ventavon;VENTAVON;05300;05;HAUTES ALPES;1
05179;;Veynes;VEYNES;05400;05;HAUTES ALPES;1
05180;Les;Vigneaux;VIGNEAUX;05120;05;HAUTES ALPES;1
05181;;Villar-d'Arne;VILLAR-D'ARENE;05480;05;HAUTES ALPES;1
05182;;Villar-Loubire;VILLAR-LOUBIERE;05800;05;HAUTES ALPES;1
05183;;Villar-Saint-Pancrace;VILLAR-SAINT-PANCRACE;05100;05;HAUTES ALPES;1
05184;;Vitrolles;VITROLLES;05110;05;HAUTES ALPES;1
06001;;Aiglun;AIGLUN;06910;06;ALPES MARITIMES;1
06002;;Amirat;AMIRAT;06910;06;ALPES MARITIMES;1
06003;;Andon;ANDON;06750;06;ALPES MARITIMES;1
06004;;Antibes;ANTIBES;06600;06;ALPES MARITIMES;1
06005;;Ascros;ASCROS;06260;06;ALPES MARITIMES;1
06006;;Aspremont;ASPREMONT;06790;06;ALPES MARITIMES;1
06007;;Auribeau-sur-Siagne;AURIBEAU-SUR-SIAGNE;06810;06;ALPES MARITIMES;1
06008;;Auvare;AUVARE;06260;06;ALPES MARITIMES;1
06009;;Bairols;BAIROLS;06420;06;ALPES MARITIMES;1
06010;Le;Bar-sur-Loup;BAR-SUR-LOUP;06620;06;ALPES MARITIMES;1
06011;;Beaulieu-sur-Mer;BEAULIEU-SUR-MER;06310;06;ALPES MARITIMES;1
06012;;Beausoleil;BEAUSOLEIL;06240;06;ALPES MARITIMES;1
06013;;Belvdre;BELVEDERE;06450;06;ALPES MARITIMES;1
06014;;Bendejun;BENDEJUN;06390;06;ALPES MARITIMES;1
06015;;Berre-les-Alpes;BERRE-LES-ALPES;06390;06;ALPES MARITIMES;1
06016;;Beuil;BEUIL;06470;06;ALPES MARITIMES;1
06017;;Bzaudun-les-Alpes;BEZAUDUN-LES-ALPES;06510;06;ALPES MARITIMES;1
06018;;Biot;BIOT;06410;06;ALPES MARITIMES;1
06019;;Blausasc;BLAUSASC;06440;06;ALPES MARITIMES;1
06020;La;Bollne-Vsubie;BOLLENE-VESUBIE;06450;06;ALPES MARITIMES;1
06021;;Bonson;BONSON;06830;06;ALPES MARITIMES;1
06022;;Bouyon;BOUYON;06510;06;ALPES MARITIMES;1
06023;;Breil-sur-Roya;BREIL-SUR-ROYA;06540;06;ALPES MARITIMES;1
06024;;Brianonnet;BRIANCONNET;06850;06;ALPES MARITIMES;1
06025;Le;Broc;BROC;06510;06;ALPES MARITIMES;1
06026;;Cabris;CABRIS;06530;06;ALPES MARITIMES;1
06027;;Cagnes-sur-Mer;CAGNES-SUR-MER;06800;06;ALPES MARITIMES;1
06028;;Caille;CAILLE;06750;06;ALPES MARITIMES;1
06029;;Cannes;CANNES;06400;06;ALPES MARITIMES;1
06030;Le;Cannet;CANNET;06110;06;ALPES MARITIMES;1
06031;;Cantaron;CANTARON;06340;06;ALPES MARITIMES;1
06032;;Cap-d'Ail;CAP-D'AIL;06320;06;ALPES MARITIMES;1
06033;;Carros;CARROS;06510;06;ALPES MARITIMES;1
06034;;Castagniers;CASTAGNIERS;06670;06;ALPES MARITIMES;1
06035;;Castellar;CASTELLAR;06500;06;ALPES MARITIMES;1
06036;;Castillon;CASTILLON;06500;06;ALPES MARITIMES;1
06037;;Caussols;CAUSSOLS;06460;06;ALPES MARITIMES;1
06038;;Chteauneuf-Grasse;CHATEAUNEUF-GRASSE;06740;06;ALPES MARITIMES;1
06039;;Chteauneuf-Villevieille;CHATEAUNEUF-VILLEVIEILLE;06390;06;ALPES MARITIMES;1
06040;;Chteauneuf-d'Entraunes;CHATEAUNEUF-D'ENTRAUNES;06470;06;ALPES MARITIMES;1
06041;;Cipires;CIPIERES;06620;06;ALPES MARITIMES;1
06042;;Clans;CLANS;06420;06;ALPES MARITIMES;1
06043;;Coaraze;COARAZE;06390;06;ALPES MARITIMES;1
06044;La;Colle-sur-Loup;COLLE-SUR-LOUP;06480;06;ALPES MARITIMES;1
06045;;Collongues;COLLONGUES;06910;06;ALPES MARITIMES;1
06046;;Colomars;COLOMARS;06670;06;ALPES MARITIMES;1
06047;;Consgudes;CONSEGUDES;06510;06;ALPES MARITIMES;1
06048;;Contes;CONTES;06390;06;ALPES MARITIMES;1
06049;;Courmes;COURMES;06620;06;ALPES MARITIMES;1
06050;;Coursegoules;COURSEGOULES;06140;06;ALPES MARITIMES;1
06051;La;Croix-sur-Roudoule;CROIX-SUR-ROUDOULE;06260;06;ALPES MARITIMES;1
06052;;Cubris;CUEBRIS;06910;06;ALPES MARITIMES;1
06053;;Daluis;DALUIS;06470;06;ALPES MARITIMES;1
06054;;Drap;DRAP;06340;06;ALPES MARITIMES;1
06055;;Duranus;DURANUS;06670;06;ALPES MARITIMES;1
06056;;Entraunes;ENTRAUNES;06470;06;ALPES MARITIMES;1
06057;L';Escarne;ESCARENE;06440;06;ALPES MARITIMES;1
06058;;Escragnolles;ESCRAGNOLLES;06460;06;ALPES MARITIMES;1
06059;;ze;EZE;06360;06;ALPES MARITIMES;1
06060;;Falicon;FALICON;06950;06;ALPES MARITIMES;1
06061;Les;Ferres;FERRES;06510;06;ALPES MARITIMES;1
06062;;Fontan;FONTAN;06540;06;ALPES MARITIMES;1
06063;;Gars;GARS;06850;06;ALPES MARITIMES;1
06064;;Gattires;GATTIERES;06510;06;ALPES MARITIMES;1
06065;La;Gaude;GAUDE;06610;06;ALPES MARITIMES;1
06066;;Gilette;GILETTE;06830;06;ALPES MARITIMES;1
06067;;Gorbio;GORBIO;06500;06;ALPES MARITIMES;1
06068;;Gourdon;GOURDON;06620;06;ALPES MARITIMES;1
06069;;Grasse;GRASSE;06130;06;ALPES MARITIMES;1
06070;;Grolires;GREOLIERES;06620;06;ALPES MARITIMES;1
06071;;Guillaumes;GUILLAUMES;06470;06;ALPES MARITIMES;1
06072;;Ilonse;ILONSE;06420;06;ALPES MARITIMES;1
06073;;Isola;ISOLA;06420;06;ALPES MARITIMES;1
06074;;Lantosque;LANTOSQUE;06450;06;ALPES MARITIMES;1
06075;;Levens;LEVENS;06670;06;ALPES MARITIMES;1
06076;;Lieuche;LIEUCHE;06260;06;ALPES MARITIMES;1
06077;;Lucram;LUCERAM;06440;06;ALPES MARITIMES;1
06078;;Malaussne;MALAUSSENE;06710;06;ALPES MARITIMES;1
06079;;Mandelieu-la-Napoule;MANDELIEU-LA-NAPOULE;06210;06;ALPES MARITIMES;1
06080;;Marie;MARIE;06420;06;ALPES MARITIMES;1
06081;Le;Mas;MAS;06910;06;ALPES MARITIMES;1
06082;;Massoins;MASSOINS;06710;06;ALPES MARITIMES;1
06083;;Menton;MENTON;06500;06;ALPES MARITIMES;1
06084;;Mouans-Sartoux;MOUANS-SARTOUX;06370;06;ALPES MARITIMES;1
06085;;Mougins;MOUGINS;06250;06;ALPES MARITIMES;1
06086;;Moulinet;MOULINET;06380;06;ALPES MARITIMES;1
06087;Les;Mujouls;MUJOULS;06910;06;ALPES MARITIMES;1
06088;;Nice;NICE;06000;06;ALPES MARITIMES;1
06089;;Opio;OPIO;06650;06;ALPES MARITIMES;1
06090;;Pgomas;PEGOMAS;06580;06;ALPES MARITIMES;1
06091;;Peille;PEILLE;06440;06;ALPES MARITIMES;1
06092;;Peillon;PEILLON;06440;06;ALPES MARITIMES;1
06093;La;Penne;PENNE;06260;06;ALPES MARITIMES;1
06094;;Pone;PEONE;06470;06;ALPES MARITIMES;1
06095;;Peymeinade;PEYMEINADE;06530;06;ALPES MARITIMES;1
06096;;Pierlas;PIERLAS;06260;06;ALPES MARITIMES;1
06097;;Pierrefeu;PIERREFEU;06910;06;ALPES MARITIMES;1
06098;;Puget-Rostang;PUGET-ROSTANG;06260;06;ALPES MARITIMES;1
06099;;Puget-Thniers;PUGET-THENIERS;06260;06;ALPES MARITIMES;1
06100;;Revest-les-Roches;REVEST-LES-ROCHES;06830;06;ALPES MARITIMES;1
06101;;Rigaud;RIGAUD;06260;06;ALPES MARITIMES;1
06102;;Rimplas;RIMPLAS;06420;06;ALPES MARITIMES;1
06103;;Roquebillire;ROQUEBILLIERE;06450;06;ALPES MARITIMES;1
06104;;Roquebrune-Cap-Martin;ROQUEBRUNE-CAP-MARTIN;06190;06;ALPES MARITIMES;1
06105;;Roquefort-les-Pins;ROQUEFORT-LES-PINS;06330;06;ALPES MARITIMES;1
06106;;Roquesteron;ROQUESTERON;06910;06;ALPES MARITIMES;1
06107;;Roquestron-Grasse;ROQUESTERON-GRASSE;06910;06;ALPES MARITIMES;1
06108;La;Roquette-sur-Siagne;ROQUETTE-SUR-SIAGNE;06550;06;ALPES MARITIMES;1
06109;La;Roquette-sur-Var;ROQUETTE-SUR-VAR;06670;06;ALPES MARITIMES;1
06110;;Roubion;ROUBION;06420;06;ALPES MARITIMES;1
06111;;Roure;ROURE;06420;06;ALPES MARITIMES;1
06112;Le;Rouret;ROURET;06650;06;ALPES MARITIMES;1
06113;;Sainte-Agns;SAINTE-AGNES;06500;06;ALPES MARITIMES;1
06114;;Saint-Andr-de-la-Roche;SAINT-ANDRE-DE-LA-ROCHE;06730;06;ALPES MARITIMES;1
06115;;Saint-Antonin;SAINT-ANTONIN;06260;06;ALPES MARITIMES;1
06116;;Saint-Auban;SAINT-AUBAN;06850;06;ALPES MARITIMES;1
06117;;Saint-Blaise;SAINT-BLAISE;06670;06;ALPES MARITIMES;1
06118;;Saint-Czaire-sur-Siagne;SAINT-CEZAIRE-SUR-SIAGNE;06780;06;ALPES MARITIMES;1
06119;;Saint-Dalmas-le-Selvage;SAINT-DALMAS-LE-SELVAGE;06660;06;ALPES MARITIMES;1
06120;;Saint-tienne-de-Tine;SAINT-ETIENNE-DE-TINEE;06660;06;ALPES MARITIMES;1
06121;;Saint-Jean-Cap-Ferrat;SAINT-JEAN-CAP-FERRAT;06230;06;ALPES MARITIMES;1
06122;;Saint-Jeannet;SAINT-JEANNET;06640;06;ALPES MARITIMES;1
06123;;Saint-Laurent-du-Var;SAINT-LAURENT-DU-VAR;06700;06;ALPES MARITIMES;1
06124;;Saint-Lger;SAINT-LEGER;06260;06;ALPES MARITIMES;1
06125;;Saint-Martin-d'Entraunes;SAINT-MARTIN-D'ENTRAUNES;06470;06;ALPES MARITIMES;1
06126;;Saint-Martin-du-Var;SAINT-MARTIN-DU-VAR;06670;06;ALPES MARITIMES;1
06127;;Saint-Martin-Vsubie;SAINT-MARTIN-VESUBIE;06450;06;ALPES MARITIMES;1
06128;;Saint-Paul-de-Vence;SAINT-PAUL-DE-VENCE;06570;06;ALPES MARITIMES;1
06129;;Saint-Sauveur-sur-Tine;SAINT-SAUVEUR-SUR-TINEE;06420;06;ALPES MARITIMES;1
06130;;Saint-Vallier-de-Thiey;SAINT-VALLIER-DE-THIEY;06460;06;ALPES MARITIMES;1
06131;;Sallagriffon;SALLAGRIFFON;06910;06;ALPES MARITIMES;1
06132;;Saorge;SAORGE;06540;06;ALPES MARITIMES;1
06133;;Sauze;SAUZE;06470;06;ALPES MARITIMES;1
06134;;Sranon;SERANON;06750;06;ALPES MARITIMES;1
06135;;Sigale;SIGALE;06910;06;ALPES MARITIMES;1
06136;;Sospel;SOSPEL;06380;06;ALPES MARITIMES;1
06137;;Spracdes;SPERACEDES;06530;06;ALPES MARITIMES;1
06138;;Thoule-sur-Mer;THEOULE-SUR-MER;06590;06;ALPES MARITIMES;1
06139;;Thiry;THIERY;06710;06;ALPES MARITIMES;1
06140;Le;Tignet;TIGNET;06530;06;ALPES MARITIMES;1
06141;;Toudon;TOUDON;06830;06;ALPES MARITIMES;1
06142;;Tout-de-l'Escarne;TOUET-DE-L'ESCARENE;06440;06;ALPES MARITIMES;1
06143;;Tout-sur-Var;TOUET-SUR-VAR;06710;06;ALPES MARITIMES;1
06144;La;Tour;TOUR;06710;06;ALPES MARITIMES;1
06145;;Tourette-du-Chteau;TOURETTE-DU-CHATEAU;06830;06;ALPES MARITIMES;1
06146;;Tournefort;TOURNEFORT;06710;06;ALPES MARITIMES;1
06147;;Tourrette-Levens;TOURRETTE-LEVENS;06690;06;ALPES MARITIMES;1
06148;;Tourrettes-sur-Loup;TOURRETTES-SUR-LOUP;06140;06;ALPES MARITIMES;1
06149;La;Trinit;TRINITE;06340;06;ALPES MARITIMES;1
06150;La;Turbie;TURBIE;06320;06;ALPES MARITIMES;1
06151;;Utelle;UTELLE;06450;06;ALPES MARITIMES;1
06152;;Valbonne;VALBONNE;06560;06;ALPES MARITIMES;1
06153;;Valdeblore;VALDEBLORE;06420;06;ALPES MARITIMES;1
06154;;Valderoure;VALDEROURE;06750;06;ALPES MARITIMES;1
06155;;Vallauris;VALLAURIS;06220;06;ALPES MARITIMES;1
06156;;Venanson;VENANSON;06450;06;ALPES MARITIMES;1
06157;;Vence;VENCE;06140;06;ALPES MARITIMES;1
06158;;Villars-sur-Var;VILLARS-SUR-VAR;06710;06;ALPES MARITIMES;1
06159;;Villefranche-sur-Mer;VILLEFRANCHE-SUR-MER;06230;06;ALPES MARITIMES;1
06160;;Villeneuve-d'Entraunes;VILLENEUVE-D'ENTRAUNES;06470;06;ALPES MARITIMES;1
06161;;Villeneuve-Loubet;VILLENEUVE-LOUBET;06270;06;ALPES MARITIMES;1
06162;La;Brigue;BRIGUE;06430;06;ALPES MARITIMES;1
06163;;Tende;TENDE;06430;06;ALPES MARITIMES;1
07001;;Accons;ACCONS;07160;07;ARDECHE;1
07002;;Ailhon;AILHON;07200;07;ARDECHE;1
07003;;Aizac;AIZAC;07530;07;ARDECHE;1
07004;;Ajoux;AJOUX;07000;07;ARDECHE;1
07005;;Alba-la-Romaine;ALBA-LA-ROMAINE;07400;07;ARDECHE;1
07006;;Albon-d'Ardche;ALBON-D'ARDECHE;07190;07;ARDECHE;1
07007;;Alboussire;ALBOUSSIERE;07440;07;ARDECHE;1
07008;;Alissas;ALISSAS;07210;07;ARDECHE;1
07009;;Andance;ANDANCE;07340;07;ARDECHE;1
07010;;Annonay;ANNONAY;07100;07;ARDECHE;1
07011;;Antraigues-sur-Volane;ANTRAIGUES-SUR-VOLANE;07530;07;ARDECHE;1
07012;;Arcens;ARCENS;07310;07;ARDECHE;1
07013;;Ardoix;ARDOIX;07290;07;ARDECHE;1
07014;;Arlebosc;ARLEBOSC;07410;07;ARDECHE;1
07015;;Arras-sur-Rhne;ARRAS-SUR-RHONE;07370;07;ARDECHE;1
07016;;Asperjoc;ASPERJOC;07600;07;ARDECHE;1
07017;Les;Assions;ASSIONS;07140;07;ARDECHE;1
07018;;Astet;ASTET;07330;07;ARDECHE;1
07019;;Aubenas;AUBENAS;07200;07;ARDECHE;1
07020;;Aubignas;AUBIGNAS;07400;07;ARDECHE;1
07022;;Baix;BAIX;07210;07;ARDECHE;1
07023;;Balazuc;BALAZUC;07120;07;ARDECHE;1
07024;;Banne;BANNE;07460;07;ARDECHE;1
07025;;Barnas;BARNAS;07330;07;ARDECHE;1
07026;Le;Bage;BEAGE;07630;07;ARDECHE;1
07027;;Beauchastel;BEAUCHASTEL;07800;07;ARDECHE;1
07028;;Beaulieu;BEAULIEU;07460;07;ARDECHE;1
07029;;Beaumont;BEAUMONT;07110;07;ARDECHE;1
07030;;Beauvne;BEAUVENE;07190;07;ARDECHE;1
07031;;Berrias-et-Casteljau;BERRIAS-ET-CASTELJAU;07460;07;ARDECHE;1
07032;;Berzme;BERZEME;07580;07;ARDECHE;1
07033;;Bessas;BESSAS;07150;07;ARDECHE;1
07034;;Bidon;BIDON;07700;07;ARDECHE;1
07035;;Boffres;BOFFRES;07440;07;ARDECHE;1
07036;;Bogy;BOGY;07340;07;ARDECHE;1
07037;;Bore;BOREE;07310;07;ARDECHE;1
07038;;Borne;BORNE;07590;07;ARDECHE;1
07039;;Bozas;BOZAS;07410;07;ARDECHE;1
07040;;Boucieu-le-Roi;BOUCIEU-LE-ROI;07270;07;ARDECHE;1
07041;;Boulieu-ls-Annonay;BOULIEU-LES-ANNONAY;07100;07;ARDECHE;1
07042;;Bourg-Saint-Andol;BOURG-SAINT-ANDEOL;07700;07;ARDECHE;1
07044;;Brossainc;BROSSAINC;07340;07;ARDECHE;1
07045;;Burzet;BURZET;07450;07;ARDECHE;1
07047;;Cellier-du-Luc;CELLIER-DU-LUC;07590;07;ARDECHE;1
07048;;Chalencon;CHALENCON;07240;07;ARDECHE;1
07049;Le;Chambon;CHAMBON;07160;07;ARDECHE;1
07050;;Chambonas;CHAMBONAS;07140;07;ARDECHE;1
07051;;Champagne;CHAMPAGNE;07340;07;ARDECHE;1
07052;;Champis;CHAMPIS;07440;07;ARDECHE;1
07053;;Chandolas;CHANDOLAS;07230;07;ARDECHE;1
07054;;Chanac;CHANEAC;07310;07;ARDECHE;1
07055;;Charmes-sur-Rhne;CHARMES-SUR-RHONE;07800;07;ARDECHE;1
07056;;Charnas;CHARNAS;07340;07;ARDECHE;1
07058;;Chassiers;CHASSIERS;07110;07;ARDECHE;1
07059;;Chteaubourg;CHATEAUBOURG;07130;07;ARDECHE;1
07060;;Chteauneuf-de-Vernoux;CHATEAUNEUF-DE-VERNOUX;07240;07;ARDECHE;1
07061;;Chauzon;CHAUZON;07120;07;ARDECHE;1
07062;;Chazeaux;CHAZEAUX;07110;07;ARDECHE;1
07063;;Cheminas;CHEMINAS;07300;07;ARDECHE;1
07064;Le;Cheylard;CHEYLARD;07160;07;ARDECHE;1
07065;;Chirols;CHIROLS;07380;07;ARDECHE;1
07066;;Chomrac;CHOMERAC;07210;07;ARDECHE;1
07067;;Colombier-le-Cardinal;COLOMBIER-LE-CARDINAL;07430;07;ARDECHE;1
07068;;Colombier-le-Jeune;COLOMBIER-LE-JEUNE;07270;07;ARDECHE;1
07069;;Colombier-le-Vieux;COLOMBIER-LE-VIEUX;07410;07;ARDECHE;1
07070;;Cornas;CORNAS;07130;07;ARDECHE;1
07071;;Coucouron;COUCOURON;07470;07;ARDECHE;1
07072;;Coux;COUX;07000;07;ARDECHE;1
07073;Le;Crestet;CRESTET;07270;07;ARDECHE;1
07074;;Creysseilles;CREYSSEILLES;07000;07;ARDECHE;1
07075;;Cros-de-Gorand;CROS-DE-GEORAND;07510;07;ARDECHE;1
07076;;Cruas;CRUAS;07350;07;ARDECHE;1
07077;;Darbres;DARBRES;07170;07;ARDECHE;1
07078;;Davzieux;DAVEZIEUX;07430;07;ARDECHE;1
07079;;Dsaignes;DESAIGNES;07570;07;ARDECHE;1
07080;;Devesset;DEVESSET;07320;07;ARDECHE;1
07081;;Dompnac;DOMPNAC;07260;07;ARDECHE;1
07082;;Dornas;DORNAS;07160;07;ARDECHE;1
07083;;Dunire-sur-Eyrieux;DUNIERE-SUR-EYRIEUX;07360;07;ARDECHE;1
07084;;Eclassan;ECLASSAN;07370;07;ARDECHE;1
07085;;Empurany;EMPURANY;07270;07;ARDECHE;1
07086;;tables;ETABLES;07300;07;ARDECHE;1
07087;;Fabras;FABRAS;07380;07;ARDECHE;1
07088;;Faugres;FAUGERES;07230;07;ARDECHE;1
07089;;Flines;FELINES;07340;07;ARDECHE;1
07090;;Flaviac;FLAVIAC;07000;07;ARDECHE;1
07091;;Fons;FONS;07200;07;ARDECHE;1
07092;;Freyssenet;FREYSSENET;07000;07;ARDECHE;1
07093;;Genestelle;GENESTELLE;07530;07;ARDECHE;1
07094;;Gilhac-et-Bruzac;GILHAC-ET-BRUZAC;07800;07;ARDECHE;1
07095;;Gilhoc-sur-Ormze;GILHOC-SUR-ORMEZE;07270;07;ARDECHE;1
07096;;Gluiras;GLUIRAS;07190;07;ARDECHE;1
07097;;Glun;GLUN;07300;07;ARDECHE;1
07098;;Gourdon;GOURDON;07000;07;ARDECHE;1
07099;;Gras;GRAS;07700;07;ARDECHE;1
07100;;Gravires;GRAVIERES;07140;07;ARDECHE;1
07101;;Grospierres;GROSPIERRES;07120;07;ARDECHE;1
07102;;Guilherand-Granges;GUILHERAND-GRANGES;07500;07;ARDECHE;1
07103;;Intres;INTRES;07310;07;ARDECHE;1
07104;;Issamoulenc;ISSAMOULENC;07190;07;ARDECHE;1
07105;;Issanlas;ISSANLAS;07510;07;ARDECHE;1
07106;;Issarls;ISSARLES;07470;07;ARDECHE;1
07107;;Jaujac;JAUJAC;07380;07;ARDECHE;1
07108;;Jaunac;JAUNAC;07160;07;ARDECHE;1
07109;;Joannas;JOANNAS;07110;07;ARDECHE;1
07110;;Joyeuse;JOYEUSE;07260;07;ARDECHE;1
07111;;Juvinas;JUVINAS;07600;07;ARDECHE;1
07112;;Labastide-sur-Bsorgues;LABASTIDE-SUR-BESORGUES;07600;07;ARDECHE;1
07113;;Labastide-de-Virac;LABASTIDE-DE-VIRAC;07150;07;ARDECHE;1
07114;;Labatie-d'Andaure;LABATIE-D'ANDAURE;07570;07;ARDECHE;1
07115;;Labeaume;LABEAUME;07120;07;ARDECHE;1
07116;;Labgude;LABEGUDE;07200;07;ARDECHE;1
07117;;Lablachre;LABLACHERE;07230;07;ARDECHE;1
07118;;Laboule;LABOULE;07110;07;ARDECHE;1
07119;Le;Lac-d'Issarls;LAC-D'ISSARLES;07470;07;ARDECHE;1
07120;;Lachamp-Raphal;LACHAMP-RAPHAEL;07530;07;ARDECHE;1
07121;;Lachapelle-Graillouse;LACHAPELLE-GRAILLOUSE;07470;07;ARDECHE;1
07122;;Lachapelle-sous-Aubenas;LACHAPELLE-SOUS-AUBENAS;07200;07;ARDECHE;1
07123;;Lachapelle-sous-Chanac;LACHAPELLE-SOUS-CHANEAC;07310;07;ARDECHE;1
07124;;Lafarre;LAFARRE;07520;07;ARDECHE;1
07126;;Lagorce;LAGORCE;07150;07;ARDECHE;1
07127;;Lalevade-d'Ardche;LALEVADE-D'ARDECHE;07380;07;ARDECHE;1
07128;;Lalouvesc;LALOUVESC;07520;07;ARDECHE;1
07129;;Lamastre;LAMASTRE;07270;07;ARDECHE;1
07130;;Lanarce;LANARCE;07660;07;ARDECHE;1
07131;;Lanas;LANAS;07200;07;ARDECHE;1
07132;;Largentire;LARGENTIERE;07110;07;ARDECHE;1
07133;;Larnas;LARNAS;07220;07;ARDECHE;1
07134;;Laurac-en-Vivarais;LAURAC-EN-VIVARAIS;07110;07;ARDECHE;1
07135;;Laval-d'Aurelle;LAVAL-D'AURELLE;07590;07;ARDECHE;1
07136;;Laveyrune;LAVEYRUNE;48250;07;LOZERE;1
07137;;Lavillatte;LAVILLATTE;07660;07;ARDECHE;1
07138;;Lavilledieu;LAVILLEDIEU;07170;07;ARDECHE;1
07139;;Laviolle;LAVIOLLE;07530;07;ARDECHE;1
07140;;Lemps;LEMPS;07300;07;ARDECHE;1
07141;;Lentillres;LENTILLERES;07200;07;ARDECHE;1
07142;;Lespron;LESPERON;07660;07;ARDECHE;1
07143;;Limony;LIMONY;07340;07;ARDECHE;1
07144;;Loubaresse;LOUBARESSE;07110;07;ARDECHE;1
07145;;Lussas;LUSSAS;07170;07;ARDECHE;1
07146;;Lyas;LYAS;07000;07;ARDECHE;1
07147;;Malarce-sur-la-Thines;MALARCE-SUR-LA-THINES;07140;07;ARDECHE;1
07148;;Malbosc;MALBOSC;07140;07;ARDECHE;1
07149;;Marcols-les-Eaux;MARCOLS-LES-EAUX;07190;07;ARDECHE;1
07150;;Mariac;MARIAC;07160;07;ARDECHE;1
07151;;Mars;MARS;07320;07;ARDECHE;1
07152;;Mauves;MAUVES;07300;07;ARDECHE;1
07153;;Mayres;MAYRES;07330;07;ARDECHE;1
07154;;Mazan-l'Abbaye;MAZAN-L'ABBAYE;07510;07;ARDECHE;1
07155;;Mercuer;MERCUER;07200;07;ARDECHE;1
07156;;Meyras;MEYRAS;07380;07;ARDECHE;1
07157;;Meysse;MEYSSE;07400;07;ARDECHE;1
07158;;Mzilhac;MEZILHAC;07530;07;ARDECHE;1
07159;;Mirabel;MIRABEL;07170;07;ARDECHE;1
07160;;Monestier;MONESTIER;07690;07;ARDECHE;1
07161;;Montpezat-sous-Bauzon;MONTPEZAT-SOUS-BAUZON;07560;07;ARDECHE;1
07162;;Montral;MONTREAL;07110;07;ARDECHE;1
07163;;Montselgues;MONTSELGUES;07140;07;ARDECHE;1
07165;;Nonires;NONIERES;07160;07;ARDECHE;1
07166;;Nozires;NOZIERES;07270;07;ARDECHE;1
07167;Les;Ollires-sur-Eyrieux;OLLIERES-SUR-EYRIEUX;07360;07;ARDECHE;1
07168;;Orgnac-l'Aven;ORGNAC-L'AVEN;07150;07;ARDECHE;1
07169;;Ozon;OZON;07370;07;ARDECHE;1
07170;;Pailhars;PAILHARES;07410;07;ARDECHE;1
07171;;Payzac;PAYZAC;07230;07;ARDECHE;1
07172;;Peaugres;PEAUGRES;07340;07;ARDECHE;1
07173;;Preyres;PEREYRES;07450;07;ARDECHE;1
07174;;Peyraud;PEYRAUD;07340;07;ARDECHE;1
07175;Le;Plagnal;PLAGNAL;07590;07;ARDECHE;1
07176;;Planzolles;PLANZOLLES;07230;07;ARDECHE;1
07177;;Plats;PLATS;07300;07;ARDECHE;1
07178;;Pont-de-Labeaume;PONT-DE-LABEAUME;07380;07;ARDECHE;1
07179;;Pourchres;POURCHERES;07000;07;ARDECHE;1
07181;Le;Pouzin;POUZIN;07250;07;ARDECHE;1
07182;;Prades;PRADES;07380;07;ARDECHE;1
07183;;Pradons;PRADONS;07120;07;ARDECHE;1
07184;;Pranles;PRANLES;07000;07;ARDECHE;1
07185;;Praux;PREAUX;07290;07;ARDECHE;1
07186;;Privas;PRIVAS;07000;07;ARDECHE;1
07187;;Prunet;PRUNET;07110;07;ARDECHE;1
07188;;Quintenas;QUINTENAS;07290;07;ARDECHE;1
07189;;Ribes;RIBES;07260;07;ARDECHE;1
07190;;Rochecolombe;ROCHECOLOMBE;07200;07;ARDECHE;1
07191;;Rochemaure;ROCHEMAURE;07400;07;ARDECHE;1
07192;;Rochepaule;ROCHEPAULE;07320;07;ARDECHE;1
07193;;Rocher;ROCHER;07110;07;ARDECHE;1
07194;;Rochessauve;ROCHESSAUVE;07210;07;ARDECHE;1
07195;La;Rochette;ROCHETTE;07310;07;ARDECHE;1
07196;;Rocles;ROCLES;07110;07;ARDECHE;1
07197;;Roiffieux;ROIFFIEUX;07100;07;ARDECHE;1
07198;;Rompon;ROMPON;07250;07;ARDECHE;1
07199;;Rosires;ROSIERES;07260;07;ARDECHE;1
07200;Le;Roux;ROUX;07560;07;ARDECHE;1
07201;;Ruoms;RUOMS;07120;07;ARDECHE;1
07202;;Sablires;SABLIERES;07260;07;ARDECHE;1
07203;;Sagnes-et-Goudoulet;SAGNES-ET-GOUDOULET;07450;07;ARDECHE;1
07204;;Saint-Agrve;SAINT-AGREVE;07320;07;ARDECHE;1
07205;;Saint-Alban-d'Ay;SAINT-ALBAN-D'AY;07790;07;ARDECHE;1
07206;;Saint-Alban-en-Montagne;SAINT-ALBAN-EN-MONTAGNE;07590;07;ARDECHE;1
07207;;Saint-Alban-Auriolles;SAINT-ALBAN-AURIOLLES;07120;07;ARDECHE;1
07208;;Saint-Andol-de-Berg;SAINT-ANDEOL-DE-BERG;07170;07;ARDECHE;1
07209;;Saint-Andol-de-Fourchades;SAINT-ANDEOL-DE-FOURCHADES;07160;07;ARDECHE;1
07210;;Saint-Andol-de-Vals;SAINT-ANDEOL-DE-VALS;07600;07;ARDECHE;1
07211;;Saint-Andr-de-Cruzires;SAINT-ANDRE-DE-CRUZIERES;07460;07;ARDECHE;1
07212;;Saint-Andr-en-Vivarais;SAINT-ANDRE-EN-VIVARAIS;07690;07;ARDECHE;1
07213;;Saint-Andr-Lachamp;SAINT-ANDRE-LACHAMP;07230;07;ARDECHE;1
07214;;Saint-Apollinaire-de-Rias;SAINT-APOLLINAIRE-DE-RIAS;07240;07;ARDECHE;1
07215;;Saint-Barthlemy-le-Meil;SAINT-BARTHELEMY-LE-MEIL;07160;07;ARDECHE;1
07216;;Saint-Barthlemy-Grozon;SAINT-BARTHELEMY-GROZON;07270;07;ARDECHE;1
07217;;Saint-Barthlemy-le-Plain;SAINT-BARTHELEMY-LE-PLAIN;07300;07;ARDECHE;1
07218;;Saint-Basile;SAINT-BASILE;07270;07;ARDECHE;1
07219;;Saint-Bauzile;SAINT-BAUZILE;07210;07;ARDECHE;1
07220;;Saint-Christol;SAINT-CHRISTOL;07160;07;ARDECHE;1
07221;;Saint-Cierge-la-Serre;SAINT-CIERGE-LA-SERRE;07800;07;ARDECHE;1
07222;;Saint-Cierge-sous-le-Cheylard;SAINT-CIERGE-SOUS-LE-CHEYLARD;07160;07;ARDECHE;1
07223;;Saint-Cirgues-de-Prades;SAINT-CIRGUES-DE-PRADES;07380;07;ARDECHE;1
07224;;Saint-Cirgues-en-Montagne;SAINT-CIRGUES-EN-MONTAGNE;07510;07;ARDECHE;1
07225;;Saint-Clair;SAINT-CLAIR;07430;07;ARDECHE;1
07226;;Saint-Clment;SAINT-CLEMENT;07310;07;ARDECHE;1
07227;;Saint-Cyr;SAINT-CYR;07430;07;ARDECHE;1
07228;;Saint-Dsirat;SAINT-DESIRAT;07340;07;ARDECHE;1
07229;;Saint-Didier-sous-Aubenas;SAINT-DIDIER-SOUS-AUBENAS;07200;07;ARDECHE;1
07230;;Saint-tienne-de-Boulogne;SAINT-ETIENNE-DE-BOULOGNE;07200;07;ARDECHE;1
07231;;Saint-tienne-de-Fontbellon;SAINT-ETIENNE-DE-FONTBELLON;07200;07;ARDECHE;1
07232;;Saint-tienne-de-Lugdars;SAINT-ETIENNE-DE-LUGDARES;07590;07;ARDECHE;1
07233;;Saint-tienne-de-Serre;SAINT-ETIENNE-DE-SERRE;07190;07;ARDECHE;1
07234;;Saint-tienne-de-Valoux;SAINT-ETIENNE-DE-VALOUX;07340;07;ARDECHE;1
07235;;Sainte-Eulalie;SAINTE-EULALIE;07510;07;ARDECHE;1
07236;;Saint-Flicien;SAINT-FELICIEN;07410;07;ARDECHE;1
07237;;Saint-Fortunat-sur-Eyrieux;SAINT-FORTUNAT-SUR-EYRIEUX;07360;07;ARDECHE;1
07238;;Saint-Genest-de-Beauzon;SAINT-GENEST-DE-BEAUZON;07230;07;ARDECHE;1
07239;;Saint-Genest-Lachamp;SAINT-GENEST-LACHAMP;07160;07;ARDECHE;1
07240;;Saint-Georges-les-Bains;SAINT-GEORGES-LES-BAINS;07800;07;ARDECHE;1
07241;;Saint-Germain;SAINT-GERMAIN;07170;07;ARDECHE;1
07242;;Saint-Gineis-en-Coiron;SAINT-GINEIS-EN-COIRON;07580;07;ARDECHE;1
07243;;Saint-Jacques-d'Atticieux;SAINT-JACQUES-D'ATTICIEUX;07340;07;ARDECHE;1
07244;;Saint-Jean-Chambre;SAINT-JEAN-CHAMBRE;07240;07;ARDECHE;1
07245;;Saint-Jean-de-Muzols;SAINT-JEAN-DE-MUZOLS;07300;07;ARDECHE;1
07247;;Saint-Jean-le-Centenier;SAINT-JEAN-LE-CENTENIER;07580;07;ARDECHE;1
07248;;Saint-Jean-Roure;SAINT-JEAN-ROURE;07160;07;ARDECHE;1
07249;;Saint-Jeure-d'Andaure;SAINT-JEURE-D'ANDAURE;07320;07;ARDECHE;1
07250;;Saint-Jeure-d'Ay;SAINT-JEURE-D'AY;07290;07;ARDECHE;1
07251;;Saint-Joseph-des-Bancs;SAINT-JOSEPH-DES-BANCS;07530;07;ARDECHE;1
07252;;Saint-Julien-Boutires;SAINT-JULIEN-BOUTIERES;07310;07;ARDECHE;1
07253;;Saint-Julien-du-Gua;SAINT-JULIEN-DU-GUA;07190;07;ARDECHE;1
07254;;Saint-Julien-du-Serre;SAINT-JULIEN-DU-SERRE;07200;07;ARDECHE;1
07255;;Saint-Julien-en-Saint-Alban;SAINT-JULIEN-EN-SAINT-ALBAN;07000;07;ARDECHE;1
07256;;Saint-Julien-Labrousse;SAINT-JULIEN-LABROUSSE;07160;07;ARDECHE;1
07257;;Saint-Julien-le-Roux;SAINT-JULIEN-LE-ROUX;07240;07;ARDECHE;1
07258;;Saint-Julien-Vocance;SAINT-JULIEN-VOCANCE;07690;07;ARDECHE;1
07259;;Saint-Just-d'Ardche;SAINT-JUST-D'ARDECHE;07700;07;ARDECHE;1
07260;;Saint-Lager-Bressac;SAINT-LAGER-BRESSAC;07210;07;ARDECHE;1
07261;;Saint-Laurent-du-Pape;SAINT-LAURENT-DU-PAPE;07800;07;ARDECHE;1
07262;;Saint-Laurent-les-Bains;SAINT-LAURENT-LES-BAINS;07590;07;ARDECHE;1
07263;;Saint-Laurent-sous-Coiron;SAINT-LAURENT-SOUS-COIRON;07170;07;ARDECHE;1
07264;;Saint-Marcel-d'Ardche;SAINT-MARCEL-D'ARDECHE;07700;07;ARDECHE;1
07265;;Saint-Marcel-ls-Annonay;SAINT-MARCEL-LES-ANNONAY;07100;07;ARDECHE;1
07266;;Sainte-Marguerite-Lafigre;SAINTE-MARGUERITE-LAFIGERE;07140;07;ARDECHE;1
07267;;Saint-Martial;SAINT-MARTIAL;07310;07;ARDECHE;1
07268;;Saint-Martin-d'Ardche;SAINT-MARTIN-D'ARDECHE;07700;07;ARDECHE;1
07269;;Saint-Martin-de-Valamas;SAINT-MARTIN-DE-VALAMAS;07310;07;ARDECHE;1
07270;;Saint-Martin-sur-Lavezon;SAINT-MARTIN-SUR-LAVEZON;07400;07;ARDECHE;1
07272;;Saint-Maurice-d'Ardche;SAINT-MAURICE-D'ARDECHE;07200;07;ARDECHE;1
07273;;Saint-Maurice-d'Ibie;SAINT-MAURICE-D'IBIE;07170;07;ARDECHE;1
07274;;Saint-Maurice-en-Chalencon;SAINT-MAURICE-EN-CHALENCON;07190;07;ARDECHE;1
07275;;Saint-Mlany;SAINT-MELANY;07260;07;ARDECHE;1
07276;;Saint-Michel-d'Aurance;SAINT-MICHEL-D'AURANCE;07160;07;ARDECHE;1
07277;;Saint-Michel-de-Boulogne;SAINT-MICHEL-DE-BOULOGNE;07200;07;ARDECHE;1
07278;;Saint-Michel-de-Chabrillanoux;SAINT-MICHEL-DE-CHABRILLANOUX;07360;07;ARDECHE;1
07279;;Saint-Montan;SAINT-MONTAN;07220;07;ARDECHE;1
07280;;Saint-Paul-le-Jeune;SAINT-PAUL-LE-JEUNE;07460;07;ARDECHE;1
07281;;Saint-Pray;SAINT-PERAY;07130;07;ARDECHE;1
07282;;Saint-Pierre-de-Colombier;SAINT-PIERRE-DE-COLOMBIER;07450;07;ARDECHE;1
07283;;Saint-Pierre-la-Roche;SAINT-PIERRE-LA-ROCHE;07400;07;ARDECHE;1
07284;;Saint-Pierre-Saint-Jean;SAINT-PIERRE-SAINT-JEAN;07140;07;ARDECHE;1
07285;;Saint-Pierre-sur-Doux;SAINT-PIERRE-SUR-DOUX;07520;07;ARDECHE;1
07286;;Saint-Pierreville;SAINT-PIERREVILLE;07190;07;ARDECHE;1
07287;;Saint-Pons;SAINT-PONS;07580;07;ARDECHE;1
07288;;Saint-Priest;SAINT-PRIEST;07000;07;ARDECHE;1
07289;;Saint-Privat;SAINT-PRIVAT;07200;07;ARDECHE;1
07290;;Saint-Prix;SAINT-PRIX;07270;07;ARDECHE;1
07291;;Saint-Remze;SAINT-REMEZE;07700;07;ARDECHE;1
07292;;Saint-Romain-d'Ay;SAINT-ROMAIN-D'AY;07290;07;ARDECHE;1
07293;;Saint-Romain-de-Lerps;SAINT-ROMAIN-DE-LERPS;07130;07;ARDECHE;1
07294;;Saint-Sauveur-de-Cruzires;SAINT-SAUVEUR-DE-CRUZIERES;07460;07;ARDECHE;1
07295;;Saint-Sauveur-de-Montagut;SAINT-SAUVEUR-DE-MONTAGUT;07190;07;ARDECHE;1
07296;;Saint-Sernin;SAINT-SERNIN;07200;07;ARDECHE;1
07297;;Saint-Sylvestre;SAINT-SYLVESTRE;07440;07;ARDECHE;1
07298;;Saint-Symphorien-sous-Chomrac;SAINT-SYMPHORIEN-SOUS-CHOMERAC;07210;07;ARDECHE;1
07299;;Saint-Symphorien-de-Mahun;SAINT-SYMPHORIEN-DE-MAHUN;07290;07;ARDECHE;1
07300;;Saint-Thom;SAINT-THOME;07220;07;ARDECHE;1
07301;;Saint-Victor;SAINT-VICTOR;07410;07;ARDECHE;1
07302;;Saint-Vincent-de-Barrs;SAINT-VINCENT-DE-BARRES;07210;07;ARDECHE;1
07303;;Saint-Vincent-de-Durfort;SAINT-VINCENT-DE-DURFORT;07360;07;ARDECHE;1
07304;;Salavas;SALAVAS;07150;07;ARDECHE;1
07305;Les;Salelles;SALELLES;07140;07;ARDECHE;1
07306;;Sampzon;SAMPZON;07120;07;ARDECHE;1
07307;;Sanilhac;SANILHAC;07110;07;ARDECHE;1
07308;;Sarras;SARRAS;07370;07;ARDECHE;1
07309;;Satillieu;SATILLIEU;07290;07;ARDECHE;1
07310;;Savas;SAVAS;07430;07;ARDECHE;1
07311;;Sceautres;SCEAUTRES;07400;07;ARDECHE;1
07312;;Scheras;SECHERAS;07610;07;ARDECHE;1
07313;;Serrires;SERRIERES;07340;07;ARDECHE;1
07314;;Silhac;SILHAC;07240;07;ARDECHE;1
07315;La;Souche;SOUCHE;07380;07;ARDECHE;1
07316;;Soyons;SOYONS;07130;07;ARDECHE;1
07317;;Talencieux;TALENCIEUX;07340;07;ARDECHE;1
07318;;Tauriers;TAURIERS;07110;07;ARDECHE;1
07319;Le;Teil;TEIL;07400;07;ARDECHE;1
07321;;Thorrenc;THORRENC;07340;07;ARDECHE;1
07322;;Thueyts;THUEYTS;07330;07;ARDECHE;1
07323;;Toulaud;TOULAUD;07130;07;ARDECHE;1
07324;;Tournon-sur-Rhne;TOURNON-SUR-RHONE;07300;07;ARDECHE;1
07325;;Ucel;UCEL;07200;07;ARDECHE;1
07326;;Usclades-et-Rieutord;USCLADES-ET-RIEUTORD;07510;07;ARDECHE;1
07327;;Uzer;UZER;07110;07;ARDECHE;1
07328;;Vagnas;VAGNAS;07150;07;ARDECHE;1
07329;;Valgorge;VALGORGE;07110;07;ARDECHE;1
07330;;Vallon-Pont-d'Arc;VALLON-PONT-D'ARC;07150;07;ARDECHE;1
07331;;Vals-les-Bains;VALS-LES-BAINS;07600;07;ARDECHE;1
07332;;Valvignres;VALVIGNERES;07400;07;ARDECHE;1
07333;;Vanosc;VANOSC;07690;07;ARDECHE;1
07334;Les;Vans;VANS;07140;07;ARDECHE;1
07335;;Vaudevant;VAUDEVANT;07410;07;ARDECHE;1
07336;;Vernon;VERNON;07260;07;ARDECHE;1
07337;;Vernosc-ls-Annonay;VERNOSC-LES-ANNONAY;07430;07;ARDECHE;1
07338;;Vernoux-en-Vivarais;VERNOUX-EN-VIVARAIS;07240;07;ARDECHE;1
07339;;Vesseaux;VESSEAUX;07200;07;ARDECHE;1
07340;;Veyras;VEYRAS;07000;07;ARDECHE;1
07341;;Villeneuve-de-Berg;VILLENEUVE-DE-BERG;07170;07;ARDECHE;1
07342;;Villevocance;VILLEVOCANCE;07690;07;ARDECHE;1
07343;;Vinezac;VINEZAC;07110;07;ARDECHE;1
07344;;Vinzieux;VINZIEUX;07340;07;ARDECHE;1
07345;;Vion;VION;07610;07;ARDECHE;1
07346;;Viviers;VIVIERS;07220;07;ARDECHE;1
07347;;Vocance;VOCANCE;07690;07;ARDECHE;1
07348;;Vog;VOGUE;07200;07;ARDECHE;1
07349;La;Voulte-sur-Rhne;VOULTE-SUR-RHONE;07800;07;ARDECHE;1
08001;;Acy-Romance;ACY-ROMANCE;08300;08;ARDENNES;1
08003;;Aiglemont;AIGLEMONT;08090;08;ARDENNES;1
08004;;Aire;AIRE;08190;08;ARDENNES;1
08005;;Alincourt;ALINCOURT;08310;08;ARDENNES;1
08006;;Alland'Huy-et-Sausseuil;ALLAND'HUY-ET-SAUSSEUIL;08130;08;ARDENNES;1
08007;Les;Alleux;ALLEUX;08400;08;ARDENNES;1
08008;;Amagne;AMAGNE;08300;08;ARDENNES;1
08009;;Amblimont;AMBLIMONT;08210;08;ARDENNES;1
08010;;Ambly-Fleury;AMBLY-FLEURY;08130;08;ARDENNES;1
08011;;Anchamps;ANCHAMPS;08500;08;ARDENNES;1
08013;;Angecourt;ANGECOURT;08450;08;ARDENNES;1
08014;;Annelles;ANNELLES;08310;08;ARDENNES;1
08015;;Antheny;ANTHENY;08260;08;ARDENNES;1
08016;;Aouste;AOUSTE;08290;08;ARDENNES;1
08017;;Apremont;APREMONT;08250;08;ARDENNES;1
08018;;Ardeuil-et-Montfauxelles;ARDEUIL-ET-MONTFAUXELLES;08400;08;ARDENNES;1
08019;Les;Grandes-Armoises;GRANDES-ARMOISES;08390;08;ARDENNES;1
08020;Les;Petites-Armoises;PETITES-ARMOISES;08390;08;ARDENNES;1
08021;;Arnicourt;ARNICOURT;08300;08;ARDENNES;1
08022;;Arreux;ARREUX;08090;08;ARDENNES;1
08023;;Artaise-le-Vivier;ARTAISE-LE-VIVIER;08390;08;ARDENNES;1
08024;;Asfeld;ASFELD;08190;08;ARDENNES;1
08025;;Attigny;ATTIGNY;08130;08;ARDENNES;1
08026;;Aubigny-les-Pothes;AUBIGNY-LES-POTHEES;08150;08;ARDENNES;1
08027;;Auboncourt-Vauzelles;AUBONCOURT-VAUZELLES;08270;08;ARDENNES;1
08028;;Aubrives;AUBRIVES;08320;08;ARDENNES;1
08029;;Auflance;AUFLANCE;08370;08;ARDENNES;1
08030;;Auge;AUGE;08380;08;ARDENNES;1
08031;;Aure;AURE;08400;08;ARDENNES;1
08032;;Aussonce;AUSSONCE;08310;08;ARDENNES;1
08033;;Authe;AUTHE;08240;08;ARDENNES;1
08034;;Autrecourt-et-Pourron;AUTRECOURT-ET-POURRON;08210;08;ARDENNES;1
08035;;Autruche;AUTRUCHE;08240;08;ARDENNES;1
08036;;Autry;AUTRY;08250;08;ARDENNES;1
08037;;Auvillers-les-Forges;AUVILLERS-LES-FORGES;08260;08;ARDENNES;1
08038;;Avanon;AVANCON;08300;08;ARDENNES;1
08039;;Avaux;AVAUX;08190;08;ARDENNES;1
08040;Les;Ayvelles;AYVELLES;08000;08;ARDENNES;1
08041;;Balons;BAALONS;08430;08;ARDENNES;1
08042;;Balaives-et-Butz;BALAIVES-ET-BUTZ;08160;08;ARDENNES;1
08043;;Balan;BALAN;08200;08;ARDENNES;1
08044;;Balham;BALHAM;08190;08;ARDENNES;1
08045;;Ballay;BALLAY;08400;08;ARDENNES;1
08046;;Banogne-Recouvrance;BANOGNE-RECOUVRANCE;08220;08;ARDENNES;1
08047;;Barbaise;BARBAISE;08430;08;ARDENNES;1
08048;;Barby;BARBY;08300;08;ARDENNES;1
08049;;Bar-ls-Buzancy;BAR-LES-BUZANCY;08240;08;ARDENNES;1
08052;;Bayonville;BAYONVILLE;08240;08;ARDENNES;1
08053;;Bazeilles;BAZEILLES;08140;08;ARDENNES;1
08055;;Beaumont-en-Argonne;BEAUMONT-EN-ARGONNE;08210;08;ARDENNES;1
08056;;Beffu-et-le-Morthomme;BEFFU-ET-LE-MORTHOMME;08250;08;ARDENNES;1
08057;;Belleville-et-Chtillon-sur-Bar;BELLEVILLE-ET-CHATILLON-SUR-BAR;08240;08;ARDENNES;1
08058;;Belval;BELVAL;08090;08;ARDENNES;1
08059;;Belval-Bois-des-Dames;BELVAL-BOIS-DES-DAMES;08240;08;ARDENNES;1
08060;;Bergnicourt;BERGNICOURT;08300;08;ARDENNES;1
08061;La;Berlire;BERLIERE;08240;08;ARDENNES;1
08062;;Bertoncourt;BERTONCOURT;08300;08;ARDENNES;1
08063;La;Besace;BESACE;08450;08;ARDENNES;1
08064;;Biermes;BIERMES;08300;08;ARDENNES;1
08065;;Bivres;BIEVRES;08370;08;ARDENNES;1
08066;;Bignicourt;BIGNICOURT;08310;08;ARDENNES;1
08067;;Blagny;BLAGNY;08110;08;ARDENNES;1
08069;;Blanchefosse-et-Bay;BLANCHEFOSSE-ET-BAY;08290;08;ARDENNES;1
08070;;Blanzy-la-Salonnaise;BLANZY-LA-SALONNAISE;08190;08;ARDENNES;1
08071;;Blombay;BLOMBAY;08260;08;ARDENNES;1
08072;;Bosseval-et-Briancourt;BOSSEVAL-ET-BRIANCOURT;08350;08;ARDENNES;1
08073;;Bossus-ls-Rumigny;BOSSUS-LES-RUMIGNY;08290;08;ARDENNES;1
08074;;Bouconville;BOUCONVILLE;08250;08;ARDENNES;1
08075;;Boult-aux-Bois;BOULT-AUX-BOIS;08240;08;ARDENNES;1
08076;;Boulzicourt;BOULZICOURT;08410;08;ARDENNES;1
08077;;Bourcq;BOURCQ;08400;08;ARDENNES;1
08078;;Bourg-Fidle;BOURG-FIDELE;08230;08;ARDENNES;1
08079;;Boutancourt;BOUTANCOURT;08160;08;ARDENNES;1
08080;;Bouvellemont;BOUVELLEMONT;08430;08;ARDENNES;1
08081;;Bogny-sur-Meuse;BOGNY-SUR-MEUSE;08120;08;ARDENNES;1
08082;;Brcy-Brires;BRECY-BRIERES;08400;08;ARDENNES;1
08083;;Brvilly;BREVILLY;08140;08;ARDENNES;1
08084;;Brienne-sur-Aisne;BRIENNE-SUR-AISNE;08190;08;ARDENNES;1
08085;;Brieulles-sur-Bar;BRIEULLES-SUR-BAR;08240;08;ARDENNES;1
08086;;Briquenay;BRIQUENAY;08240;08;ARDENNES;1
08087;;Brognon;BROGNON;08380;08;ARDENNES;1
08088;;Bulson;BULSON;08450;08;ARDENNES;1
08089;;Buzancy;BUZANCY;08240;08;ARDENNES;1
08090;;Carignan;CARIGNAN;08110;08;ARDENNES;1
08092;;Cauroy;CAUROY;08310;08;ARDENNES;1
08094;;Cernion;CERNION;08260;08;ARDENNES;1
08095;;Chagny;CHAGNY;08430;08;ARDENNES;1
08096;;Chalandry-Elaire;CHALANDRY-ELAIRE;08160;08;ARDENNES;1
08097;;Challerange;CHALLERANGE;08400;08;ARDENNES;1
08098;;Champigneulle;CHAMPIGNEULLE;08250;08;ARDENNES;1
08099;;Champigneul-sur-Vence;CHAMPIGNEUL-SUR-VENCE;08430;08;ARDENNES;1
08100;;Champlin;CHAMPLIN;08260;08;ARDENNES;1
08101;La;Chapelle;CHAPELLE;08200;08;ARDENNES;1
08102;;Chappes;CHAPPES;08220;08;ARDENNES;1
08103;;Charbogne;CHARBOGNE;08130;08;ARDENNES;1
08104;;Chardeny;CHARDENY;08400;08;ARDENNES;1
08105;;Charleville-Mzires;CHARLEVILLE-MEZIERES;08000;08;ARDENNES;1
08106;;Charnois;CHARNOIS;08600;08;ARDENNES;1
08107;;Chteau-Porcien;CHATEAU-PORCIEN;08360;08;ARDENNES;1
08109;;Chatel-Chhry;CHATEL-CHEHERY;08250;08;ARDENNES;1
08110;Le;Chtelet-sur-Sormonne;CHATELET-SUR-SORMONNE;08150;08;ARDENNES;1
08111;Le;Chtelet-sur-Retourne;CHATELET-SUR-RETOURNE;08300;08;ARDENNES;1
08113;;Chaumont-Porcien;CHAUMONT-PORCIEN;08220;08;ARDENNES;1
08114;;Chhry;CHEHERY;08350;08;ARDENNES;1
08115;;Chmery-sur-Bar;CHEMERY-SUR-BAR;08450;08;ARDENNES;1
08116;Le;Chesne;CHESNE;08390;08;ARDENNES;1
08117;;Chesnois-Auboncourt;CHESNOIS-AUBONCOURT;08270;08;ARDENNES;1
08119;;Cheveuges;CHEVEUGES;08350;08;ARDENNES;1
08120;;Chevires;CHEVIERES;08250;08;ARDENNES;1
08121;;Chilly;CHILLY;08260;08;ARDENNES;1
08122;;Chooz;CHOOZ;08600;08;ARDENNES;1
08123;;Chuffilly-Roche;CHUFFILLY-ROCHE;08130;08;ARDENNES;1
08124;;Clavy-Warby;CLAVY-WARBY;08560;08;ARDENNES;1
08125;;Cliron;CLIRON;08090;08;ARDENNES;1
08126;;Cond-ls-Herpy;CONDE-LES-HERPY;08360;08;ARDENNES;1
08128;;Cond-ls-Autry;CONDE-LES-AUTRY;08250;08;ARDENNES;1
08130;;Contreuve;CONTREUVE;08400;08;ARDENNES;1
08131;;Cornay;CORNAY;08250;08;ARDENNES;1
08132;;Corny-Machromnil;CORNY-MACHEROMENIL;08270;08;ARDENNES;1
08133;;Coucy;COUCY;08300;08;ARDENNES;1
08134;;Coulommes-et-Marqueny;COULOMMES-ET-MARQUENY;08130;08;ARDENNES;1
08135;La;Croix-aux-Bois;CROIX-AUX-BOIS;08400;08;ARDENNES;1
08136;;Daigny;DAIGNY;08140;08;ARDENNES;1
08137;;Damouzy;DAMOUZY;08090;08;ARDENNES;1
08138;Les;Deux-Villes;DEUX-VILLES;08110;08;ARDENNES;1
08139;;Deville;DEVILLE;08800;08;ARDENNES;1
08140;;Dom-le-Mesnil;DOM-LE-MESNIL;08160;08;ARDENNES;1
08141;;Dommery;DOMMERY;08460;08;ARDENNES;1
08142;;Donchery;DONCHERY;08350;08;ARDENNES;1
08143;;Doumely-Bgny;DOUMELY-BEGNY;08220;08;ARDENNES;1
08144;;Doux;DOUX;08300;08;ARDENNES;1
08145;;Douzy;DOUZY;08140;08;ARDENNES;1
08146;;Draize;DRAIZE;08220;08;ARDENNES;1
08147;;Dricourt;DRICOURT;08310;08;ARDENNES;1
08148;L';caille;ECAILLE;08300;08;ARDENNES;1
08149;L';chelle;ECHELLE;08150;08;ARDENNES;1
08150;;cly;ECLY;08300;08;ARDENNES;1
08151;;cordal;ECORDAL;08130;08;ARDENNES;1
08152;;lan;ELAN;08160;08;ARDENNES;1
08153;;Escombres-et-le-Chesnois;ESCOMBRES-ET-LE-CHESNOIS;08110;08;ARDENNES;1
08154;;Estrebay;ESTREBAY;08260;08;ARDENNES;1
08155;;talle;ETALLE;08260;08;ARDENNES;1
08156;;teignires;ETEIGNIERES;08260;08;ARDENNES;1
08158;;trpigny;ETREPIGNY;08160;08;ARDENNES;1
08159;;Euilly-et-Lombut;EUILLY-ET-LOMBUT;08210;08;ARDENNES;1
08160;;vigny;EVIGNY;08090;08;ARDENNES;1
08161;;Exermont;EXERMONT;08250;08;ARDENNES;1
08162;;Fagnon;FAGNON;08090;08;ARDENNES;1
08163;;Faissault;FAISSAULT;08270;08;ARDENNES;1
08164;;Falaise;FALAISE;08400;08;ARDENNES;1
08165;;Faux;FAUX;08270;08;ARDENNES;1
08166;;Fpin;FEPIN;08170;08;ARDENNES;1
08167;La;Fre;FEREE;08290;08;ARDENNES;1
08168;La;Fert-sur-Chiers;FERTE-SUR-CHIERS;08370;08;ARDENNES;1
08169;;Flaignes-Havys;FLAIGNES-HAVYS;08260;08;ARDENNES;1
08170;;Fleigneux;FLEIGNEUX;08200;08;ARDENNES;1
08171;;Flville;FLEVILLE;08250;08;ARDENNES;1
08172;;Fligny;FLIGNY;08380;08;ARDENNES;1
08173;;Flize;FLIZE;08160;08;ARDENNES;1
08174;;Floing;FLOING;08200;08;ARDENNES;1
08175;;Foisches;FOISCHES;08600;08;ARDENNES;1
08176;;Foss;FOSSE;08240;08;ARDENNES;1
08178;;Fraillicourt;FRAILLICOURT;08220;08;ARDENNES;1
08179;;Francheval;FRANCHEVAL;08140;08;ARDENNES;1
08180;La;Francheville;FRANCHEVILLE;08000;08;ARDENNES;1
08182;Le;Frty;FRETY;08290;08;ARDENNES;1
08183;;Fromelennes;FROMELENNES;08600;08;ARDENNES;1
08184;;Fromy;FROMY;08370;08;ARDENNES;1
08185;;Fumay;FUMAY;08170;08;ARDENNES;1
08186;;Germont;GERMONT;08240;08;ARDENNES;1
08187;;Gernelle;GERNELLE;08440;08;ARDENNES;1
08188;;Gespunsart;GESPUNSART;08700;08;ARDENNES;1
08189;;Girondelle;GIRONDELLE;08260;08;ARDENNES;1
08190;;Givet;GIVET;08600;08;ARDENNES;1
08191;;Givonne;GIVONNE;08200;08;ARDENNES;1
08192;;Givron;GIVRON;08220;08;ARDENNES;1
08193;;Givry;GIVRY;08130;08;ARDENNES;1
08194;;Glaire;GLAIRE;08200;08;ARDENNES;1
08195;;Gomont;GOMONT;08190;08;ARDENNES;1
08196;;Grandchamp;GRANDCHAMP;08270;08;ARDENNES;1
08197;;Grandham;GRANDHAM;08250;08;ARDENNES;1
08198;;Grandpr;GRANDPRE;08250;08;ARDENNES;1
08199;La;Grandville;GRANDVILLE;08700;08;ARDENNES;1
08200;;Grivy-Loisy;GRIVY-LOISY;08400;08;ARDENNES;1
08201;;Gruyres;GRUYERES;08430;08;ARDENNES;1
08202;;Gu-d'Hossus;GUE-D'HOSSUS;08230;08;ARDENNES;1
08203;;Guignicourt-sur-Vence;GUIGNICOURT-SUR-VENCE;08430;08;ARDENNES;1
08204;;Guincourt;GUINCOURT;08130;08;ARDENNES;1
08205;;Hagnicourt;HAGNICOURT;08430;08;ARDENNES;1
08206;;Ham-les-Moines;HAM-LES-MOINES;08090;08;ARDENNES;1
08207;;Ham-sur-Meuse;HAM-SUR-MEUSE;08600;08;ARDENNES;1
08208;;Hannappes;HANNAPPES;08290;08;ARDENNES;1
08209;;Hannogne-Saint-Martin;HANNOGNE-SAINT-MARTIN;08160;08;ARDENNES;1
08210;;Hannogne-Saint-Rmy;HANNOGNE-SAINT-REMY;08220;08;ARDENNES;1
08211;;Haraucourt;HARAUCOURT;08450;08;ARDENNES;1
08212;;Harcy;HARCY;08150;08;ARDENNES;1
08214;;Hargnies;HARGNIES;08170;08;ARDENNES;1
08215;;Harricourt;HARRICOURT;08240;08;ARDENNES;1
08216;;Haudrecy;HAUDRECY;08090;08;ARDENNES;1
08217;;Haulm;HAULME;08800;08;ARDENNES;1
08218;Les;Hautes-Rivires;HAUTES-RIVIERES;08800;08;ARDENNES;1
08219;;Hauteville;HAUTEVILLE;08300;08;ARDENNES;1
08220;;Hauvin;HAUVINE;08310;08;ARDENNES;1
08222;;Haybes;HAYBES;08170;08;ARDENNES;1
08223;;Herbeuval;HERBEUVAL;08370;08;ARDENNES;1
08225;;Herpy-l'Arlsienne;HERPY-L'ARLESIENNE;08360;08;ARDENNES;1
08226;;Hierges;HIERGES;08320;08;ARDENNES;1
08228;La;Horgne;HORGNE;08430;08;ARDENNES;1
08229;;Houdilcourt;HOUDILCOURT;08190;08;ARDENNES;1
08230;;Houldizy;HOULDIZY;08090;08;ARDENNES;1
08232;;Illy;ILLY;08200;08;ARDENNES;1
08233;;Imcourt;IMECOURT;08240;08;ARDENNES;1
08234;;Inaumont;INAUMONT;08300;08;ARDENNES;1
08235;;Issancourt-et-Rumel;ISSANCOURT-ET-RUMEL;08440;08;ARDENNES;1
08236;;Jandun;JANDUN;08430;08;ARDENNES;1
08237;;Joigny-sur-Meuse;JOIGNY-SUR-MEUSE;08700;08;ARDENNES;1
08238;;Jonval;JONVAL;08130;08;ARDENNES;1
08239;;Juniville;JUNIVILLE;08310;08;ARDENNES;1
08240;;Justine-Herbigny;JUSTINE-HERBIGNY;08270;08;ARDENNES;1
08242;;Laifour;LAIFOUR;08800;08;ARDENNES;1
08243;;Lalobbe;LALOBBE;08460;08;ARDENNES;1
08244;;Lametz;LAMETZ;08130;08;ARDENNES;1
08245;;Lanon;LANCON;08250;08;ARDENNES;1
08246;;Landres-et-Saint-Georges;LANDRES-ET-SAINT-GEORGES;08240;08;ARDENNES;1
08247;;Landrichamps;LANDRICHAMPS;08600;08;ARDENNES;1
08248;;Launois-sur-Vence;LAUNOIS-SUR-VENCE;08430;08;ARDENNES;1
08249;;Laval-Morency;LAVAL-MORENCY;08150;08;ARDENNES;1
08250;;Leffincourt;LEFFINCOURT;08310;08;ARDENNES;1
08251;;Lpron-les-Valles;LEPRON-LES-VALLEES;08150;08;ARDENNES;1
08252;;Ltanne;LETANNE;08210;08;ARDENNES;1
08254;;Liart;LIART;08290;08;ARDENNES;1
08255;;Linay;LINAY;08110;08;ARDENNES;1
08256;;Liry;LIRY;08400;08;ARDENNES;1
08257;;Logny-Bogny;LOGNY-BOGNY;08150;08;ARDENNES;1
08259;;Longw;LONGWE;08400;08;ARDENNES;1
08260;;Lonny;LONNY;08150;08;ARDENNES;1
08261;;Louvergny;LOUVERGNY;08390;08;ARDENNES;1
08262;;Lucquy;LUCQUY;08300;08;ARDENNES;1
08263;;Lumes;LUMES;08440;08;ARDENNES;1
08264;;Machault;MACHAULT;08310;08;ARDENNES;1
08267;;Mairy;MAIRY;08140;08;ARDENNES;1
08268;;Maisoncelle-et-Villers;MAISONCELLE-ET-VILLERS;08450;08;ARDENNES;1
08269;;Malandry;MALANDRY;08370;08;ARDENNES;1
08271;;Manre;MANRE;08400;08;ARDENNES;1
08272;;Maranwez;MARANWEZ;08460;08;ARDENNES;1
08273;;Marby;MARBY;08260;08;ARDENNES;1
08274;;Marcq;MARCQ;08250;08;ARDENNES;1
08275;;Margny;MARGNY;08370;08;ARDENNES;1
08276;;Margut;MARGUT;08370;08;ARDENNES;1
08277;;Marlemont;MARLEMONT;08290;08;ARDENNES;1
08278;;Marquigny;MARQUIGNY;08390;08;ARDENNES;1
08279;;Mars-sous-Bourcq;MARS-SOUS-BOURCQ;08400;08;ARDENNES;1
08280;;Marvaux-Vieux;MARVAUX-VIEUX;08400;08;ARDENNES;1
08281;;Matton-et-Clmency;MATTON-ET-CLEMENCY;08110;08;ARDENNES;1
08282;;Maubert-Fontaine;MAUBERT-FONTAINE;08260;08;ARDENNES;1
08283;;Mazerny;MAZERNY;08430;08;ARDENNES;1
08284;Les;Mazures;MAZURES;08500;08;ARDENNES;1
08286;;Mnil-Annelles;MENIL-ANNELLES;08310;08;ARDENNES;1
08287;;Mnil-Lpinois;MENIL-LEPINOIS;08310;08;ARDENNES;1
08288;;Mesmont;MESMONT;08270;08;ARDENNES;1
08289;;Messincourt;MESSINCOURT;08110;08;ARDENNES;1
08291;;Mogues;MOGUES;08110;08;ARDENNES;1
08293;;Moiry;MOIRY;08370;08;ARDENNES;1
08294;La;Moncelle;MONCELLE;08140;08;ARDENNES;1
08295;;Mondigny;MONDIGNY;08430;08;ARDENNES;1
08296;;Montcheutin;MONTCHEUTIN;08250;08;ARDENNES;1
08297;;Montcornet;MONTCORNET;08090;08;ARDENNES;1
08298;;Montcy-Notre-Dame;MONTCY-NOTRE-DAME;08090;08;ARDENNES;1
08300;Le;Mont-Dieu;MONT-DIEU;08390;08;ARDENNES;1
08301;;Montgon;MONTGON;08390;08;ARDENNES;1
08302;;Montherm;MONTHERME;08800;08;ARDENNES;1
08303;;Monthois;MONTHOIS;08400;08;ARDENNES;1
08304;;Montigny-sur-Meuse;MONTIGNY-SUR-MEUSE;08170;08;ARDENNES;1
08305;;Montigny-sur-Vence;MONTIGNY-SUR-VENCE;08430;08;ARDENNES;1
08306;;Mont-Laurent;MONT-LAURENT;08130;08;ARDENNES;1
08307;;Montmeillant;MONTMEILLANT;08220;08;ARDENNES;1
08308;;Mont-Saint-Martin;MONT-SAINT-MARTIN;08400;08;ARDENNES;1
08309;;Mont-Saint-Remy;MONT-SAINT-REMY;08310;08;ARDENNES;1
08310;;Mouron;MOURON;08250;08;ARDENNES;1
08311;;Mouzon;MOUZON;08210;08;ARDENNES;1
08312;;Murtin-et-Bogny;MURTIN-ET-BOGNY;08150;08;ARDENNES;1
08313;;Nanteuil-sur-Aisne;NANTEUIL-SUR-AISNE;08300;08;ARDENNES;1
08314;;Neuflize;NEUFLIZE;08300;08;ARDENNES;1
08315;;Neufmaison;NEUFMAISON;08560;08;ARDENNES;1
08316;;Neufmanil;NEUFMANIL;08700;08;ARDENNES;1
08317;La;Neuville--Maire;NEUVILLE-A-MAIRE;08450;08;ARDENNES;1
08318;La;Neuville-aux-Jotes;NEUVILLE-AUX-JOUTES;08380;08;ARDENNES;1
08319;;Neuville-lez-Beaulieu;NEUVILLE-LEZ-BEAULIEU;08380;08;ARDENNES;1
08320;La;Neuville-en-Tourne--Fuy;NEUVILLE-EN-TOURNE-A-FUY;08310;08;ARDENNES;1
08321;;Neuville-Day;NEUVILLE-DAY;08130;08;ARDENNES;1
08322;;Neuville-ls-This;NEUVILLE-LES-THIS;08090;08;ARDENNES;1
08323;La;Neuville-ls-Wasigny;NEUVILLE-LES-WASIGNY;08270;08;ARDENNES;1
08324;;Neuvizy;NEUVIZY;08430;08;ARDENNES;1
08325;;Noirval;NOIRVAL;08400;08;ARDENNES;1
08326;;Nouart;NOUART;08240;08;ARDENNES;1
08327;;Nouvion-sur-Meuse;NOUVION-SUR-MEUSE;08160;08;ARDENNES;1
08328;;Nouzonville;NOUZONVILLE;08700;08;ARDENNES;1
08329;;Novion-Porcien;NOVION-PORCIEN;08270;08;ARDENNES;1
08330;;Novy-Chevrires;NOVY-CHEVRIERES;08300;08;ARDENNES;1
08331;;Noyers-Pont-Maugis;NOYERS-PONT-MAUGIS;08350;08;ARDENNES;1
08332;;Oches;OCHES;08240;08;ARDENNES;1
08333;;Olizy-Primat;OLIZY-PRIMAT;08250;08;ARDENNES;1
08334;;Omicourt;OMICOURT;08450;08;ARDENNES;1
08335;;Omont;OMONT;08430;08;ARDENNES;1
08336;;Osnes;OSNES;08110;08;ARDENNES;1
08338;;Pauvres;PAUVRES;08310;08;ARDENNES;1
08339;;Perthes;PERTHES;08300;08;ARDENNES;1
08340;;Poilcourt-Sydney;POILCOURT-SYDNEY;08190;08;ARDENNES;1
08341;;Poix-Terron;POIX-TERRON;08430;08;ARDENNES;1
08342;;Pouru-aux-Bois;POURU-AUX-BOIS;08140;08;ARDENNES;1
08343;;Pouru-Saint-Remy;POURU-SAINT-REMY;08140;08;ARDENNES;1
08344;;Prez;PREZ;08290;08;ARDENNES;1
08346;;Prix-ls-Mzires;PRIX-LES-MEZIERES;08000;08;ARDENNES;1
08347;;Puilly-et-Charbeaux;PUILLY-ET-CHARBEAUX;08370;08;ARDENNES;1
08348;;Puiseux;PUISEUX;08270;08;ARDENNES;1
08349;;Pure;PURE;08110;08;ARDENNES;1
08350;;Quatre-Champs;QUATRE-CHAMPS;08400;08;ARDENNES;1
08351;;Quilly;QUILLY;08400;08;ARDENNES;1
08352;;Raillicourt;RAILLICOURT;08430;08;ARDENNES;1
08353;;Rancennes;RANCENNES;08600;08;ARDENNES;1
08354;;Raucourt-et-Flaba;RAUCOURT-ET-FLABA;08450;08;ARDENNES;1
08355;;Regniowez;REGNIOWEZ;08230;08;ARDENNES;1
08356;;Remaucourt;REMAUCOURT;08220;08;ARDENNES;1
08357;;Remilly-Aillicourt;REMILLY-AILLICOURT;08450;08;ARDENNES;1
08358;;Remilly-les-Pothes;REMILLY-LES-POTHEES;08150;08;ARDENNES;1
08360;;Renneville;RENNEVILLE;08220;08;ARDENNES;1
08361;;Renwez;RENWEZ;08150;08;ARDENNES;1
08362;;Rethel;RETHEL;08300;08;ARDENNES;1
08363;;Revin;REVIN;08500;08;ARDENNES;1
08364;;Rilly-sur-Aisne;RILLY-SUR-AISNE;08130;08;ARDENNES;1
08365;;Rimogne;RIMOGNE;08150;08;ARDENNES;1
08366;;Rocquigny;ROCQUIGNY;08220;08;ARDENNES;1
08367;;Rocroi;ROCROI;08230;08;ARDENNES;1
08368;;Roizy;ROIZY;08190;08;ARDENNES;1
08369;La;Romagne;ROMAGNE;08220;08;ARDENNES;1
08370;;Rouvroy-sur-Audry;ROUVROY-SUR-AUDRY;08150;08;ARDENNES;1
08371;;Rubcourt-et-Lamcourt;RUBECOURT-ET-LAMECOURT;08140;08;ARDENNES;1
08372;;Rubigny;RUBIGNY;08220;08;ARDENNES;1
08373;;Rumigny;RUMIGNY;08290;08;ARDENNES;1
08374;La;Sabotterie;SABOTTERIE;08130;08;ARDENNES;1
08375;;Sachy;SACHY;08110;08;ARDENNES;1
08376;;Sailly;SAILLY;08110;08;ARDENNES;1
08377;;Saint-Aignan;SAINT-AIGNAN;08350;08;ARDENNES;1
08378;;Saint-Clment--Arnes;SAINT-CLEMENT-A-ARNES;08310;08;ARDENNES;1
08379;;Saint-tienne--Arnes;SAINT-ETIENNE-A-ARNES;08310;08;ARDENNES;1
08380;;Saint-Fergeux;SAINT-FERGEUX;08360;08;ARDENNES;1
08381;;Saint-Germainmont;SAINT-GERMAINMONT;08190;08;ARDENNES;1
08382;;Saint-Jean-aux-Bois;SAINT-JEAN-AUX-BOIS;08220;08;ARDENNES;1
08383;;Saint-Juvin;SAINT-JUVIN;08250;08;ARDENNES;1
08384;;Saint-Lambert-et-Mont-de-Jeux;SAINT-LAMBERT-ET-MONT-DE-JEUX;08130;08;ARDENNES;1
08385;;Saint-Laurent;SAINT-LAURENT;08090;08;ARDENNES;1
08386;;Saint-Loup-en-Champagne;SAINT-LOUP-EN-CHAMPAGNE;08300;08;ARDENNES;1
08387;;Saint-Loup-Terrier;SAINT-LOUP-TERRIER;08130;08;ARDENNES;1
08388;;Saint-Marceau;SAINT-MARCEAU;08160;08;ARDENNES;1
08389;;Saint-Marcel;SAINT-MARCEL;08560;08;ARDENNES;1
08390;;Sainte-Marie;SAINTE-MARIE;08400;08;ARDENNES;1
08391;;Saint-Menges;SAINT-MENGES;08200;08;ARDENNES;1
08392;;Saint-Morel;SAINT-MOREL;08400;08;ARDENNES;1
08393;;Saint-Pierre--Arnes;SAINT-PIERRE-A-ARNES;08310;08;ARDENNES;1
08394;;Saint-Pierremont;SAINT-PIERREMONT;08240;08;ARDENNES;1
08395;;Saint-Pierre-sur-Vence;SAINT-PIERRE-SUR-VENCE;08430;08;ARDENNES;1
08396;;Saint-Quentin-le-Petit;SAINT-QUENTIN-LE-PETIT;08220;08;ARDENNES;1
08397;;Saint-Remy-le-Petit;SAINT-REMY-LE-PETIT;08300;08;ARDENNES;1
08398;;Sainte-Vaubourg;SAINTE-VAUBOURG;08130;08;ARDENNES;1
08399;;Sapogne-sur-Marche;SAPOGNE-SUR-MARCHE;08370;08;ARDENNES;1
08400;;Sapogne-et-Feuchres;SAPOGNE-ET-FEUCHERES;08160;08;ARDENNES;1
08401;;Saulces-Champenoises;SAULCES-CHAMPENOISES;08130;08;ARDENNES;1
08402;;Saulces-Monclin;SAULCES-MONCLIN;08270;08;ARDENNES;1
08403;;Sault-ls-Rethel;SAULT-LES-RETHEL;08300;08;ARDENNES;1
08404;;Sault-Saint-Remy;SAULT-SAINT-REMY;08190;08;ARDENNES;1
08405;;Sauville;SAUVILLE;08390;08;ARDENNES;1
08406;;Savigny-sur-Aisne;SAVIGNY-SUR-AISNE;08400;08;ARDENNES;1
08407;;Schault;SECHAULT;08250;08;ARDENNES;1
08408;;Scheval;SECHEVAL;08150;08;ARDENNES;1
08409;;Sedan;SEDAN;08200;08;ARDENNES;1
08410;;Semide;SEMIDE;08400;08;ARDENNES;1
08411;;Semuy;SEMUY;08130;08;ARDENNES;1
08412;;Senuc;SENUC;08250;08;ARDENNES;1
08413;;Seraincourt;SERAINCOURT;08220;08;ARDENNES;1
08415;;Sery;SERY;08270;08;ARDENNES;1
08416;;Seuil;SEUIL;08300;08;ARDENNES;1
08417;;Svigny-la-Fort;SEVIGNY-LA-FORET;08230;08;ARDENNES;1
08418;;Svigny-Waleppe;SEVIGNY-WALEPPE;08220;08;ARDENNES;1
08419;;Signy-l'Abbaye;SIGNY-L'ABBAYE;08460;08;ARDENNES;1
08420;;Signy-le-Petit;SIGNY-LE-PETIT;08380;08;ARDENNES;1
08421;;Signy-Montlibert;SIGNY-MONTLIBERT;08370;08;ARDENNES;1
08422;;Singly;SINGLY;08430;08;ARDENNES;1
08424;;Sommauthe;SOMMAUTHE;08240;08;ARDENNES;1
08425;;Sommerance;SOMMERANCE;08250;08;ARDENNES;1
08426;;Son;SON;08300;08;ARDENNES;1
08427;;Sorbon;SORBON;08300;08;ARDENNES;1
08428;;Sorcy-Bauthmont;SORCY-BAUTHEMONT;08270;08;ARDENNES;1
08429;;Sormonne;SORMONNE;08150;08;ARDENNES;1
08430;;Stonne;STONNE;08390;08;ARDENNES;1
08431;;Sugny;SUGNY;08400;08;ARDENNES;1
08432;;Sury;SURY;08090;08;ARDENNES;1
08433;;Suzanne;SUZANNE;08130;08;ARDENNES;1
08434;;Sy;SY;08390;08;ARDENNES;1
08435;;Tagnon;TAGNON;08300;08;ARDENNES;1
08436;;Taillette;TAILLETTE;08230;08;ARDENNES;1
08437;;Tailly;TAILLY;08240;08;ARDENNES;1
08438;;Taizy;TAIZY;08360;08;ARDENNES;1
08439;;Tannay;TANNAY;08390;08;ARDENNES;1
08440;;Tarzy;TARZY;08380;08;ARDENNES;1
08441;;Termes;TERMES;08250;08;ARDENNES;1
08443;;Terron-sur-Aisne;TERRON-SUR-AISNE;08400;08;ARDENNES;1
08444;;Ttaigne;TETAIGNE;08110;08;ARDENNES;1
08445;;Thelonne;THELONNE;08350;08;ARDENNES;1
08446;;Thnorgues;THENORGUES;08240;08;ARDENNES;1
08448;;Thilay;THILAY;08800;08;ARDENNES;1
08449;;Thin-le-Moutier;THIN-LE-MOUTIER;08460;08;ARDENNES;1
08450;;This;THIS;08090;08;ARDENNES;1
08451;Le;Thour;THOUR;08190;08;ARDENNES;1
08452;;Thugny-Trugny;THUGNY-TRUGNY;08300;08;ARDENNES;1
08453;;Toges;TOGES;08400;08;ARDENNES;1
08454;;Touligny;TOULIGNY;08430;08;ARDENNES;1
08455;;Tourcelles-Chaumont;TOURCELLES-CHAUMONT;08400;08;ARDENNES;1
08456;;Tournavaux;TOURNAVAUX;08800;08;ARDENNES;1
08457;;Tournes;TOURNES;08090;08;ARDENNES;1
08458;;Tourteron;TOURTERON;08130;08;ARDENNES;1
08459;;Tremblois-ls-Carignan;TREMBLOIS-LES-CARIGNAN;08110;08;ARDENNES;1
08460;;Tremblois-ls-Rocroi;TREMBLOIS-LES-ROCROI;08150;08;ARDENNES;1
08461;;Vandy;VANDY;08400;08;ARDENNES;1
08462;;Vaux-Champagne;VAUX-CHAMPAGNE;08130;08;ARDENNES;1
08463;;Vaux-en-Dieulet;VAUX-EN-DIEULET;08240;08;ARDENNES;1
08464;;Vaux-ls-Mouron;VAUX-LES-MOURON;08250;08;ARDENNES;1
08465;;Vaux-ls-Rubigny;VAUX-LES-RUBIGNY;08220;08;ARDENNES;1
08466;;Vaux-ls-Mouzon;VAUX-LES-MOUZON;08210;08;ARDENNES;1
08467;;Vaux-Montreuil;VAUX-MONTREUIL;08270;08;ARDENNES;1
08468;;Vaux-Villaine;VAUX-VILLAINE;08150;08;ARDENNES;1
08469;;Vendresse;VENDRESSE;08160;08;ARDENNES;1
08470;;Verpel;VERPEL;08240;08;ARDENNES;1
08471;;Verrires;VERRIERES;08390;08;ARDENNES;1
08472;;Viel-Saint-Remy;VIEL-SAINT-REMY;08270;08;ARDENNES;1
08473;;Vieux-ls-Asfeld;VIEUX-LES-ASFELD;08190;08;ARDENNES;1
08475;;Villers-Cernay;VILLERS-CERNAY;08140;08;ARDENNES;1
08476;;Villers-devant-le-Thour;VILLERS-DEVANT-LE-THOUR;08190;08;ARDENNES;1
08477;;Villers-devant-Mouzon;VILLERS-DEVANT-MOUZON;08210;08;ARDENNES;1
08478;;Villers-le-Tilleul;VILLERS-LE-TILLEUL;08430;08;ARDENNES;1
08479;;Villers-le-Tourneur;VILLERS-LE-TOURNEUR;08430;08;ARDENNES;1
08480;;Villers-Semeuse;VILLERS-SEMEUSE;08000;08;ARDENNES;1
08481;;Villers-sur-Bar;VILLERS-SUR-BAR;08350;08;ARDENNES;1
08482;;Villers-sur-le-Mont;VILLERS-SUR-LE-MONT;08430;08;ARDENNES;1
08483;;Ville-sur-Lumes;VILLE-SUR-LUMES;08440;08;ARDENNES;1
08484;;Ville-sur-Retourne;VILLE-SUR-RETOURNE;08310;08;ARDENNES;1
08485;;Villy;VILLY;08370;08;ARDENNES;1
08486;;Vireux-Molhain;VIREUX-MOLHAIN;08320;08;ARDENNES;1
08487;;Vireux-Wallerand;VIREUX-WALLERAND;08320;08;ARDENNES;1
08488;;Vivier-au-Court;VIVIER-AU-COURT;08440;08;ARDENNES;1
08489;;Voncq;VONCQ;08400;08;ARDENNES;1
08490;;Vouziers;VOUZIERS;08400;08;ARDENNES;1
08491;;Vrigne-aux-Bois;VRIGNE-AUX-BOIS;08330;08;ARDENNES;1
08492;;Vrigne-Meuse;VRIGNE-MEUSE;08350;08;ARDENNES;1
08493;;Vrizy;VRIZY;08400;08;ARDENNES;1
08494;;Wadelincourt;WADELINCOURT;08200;08;ARDENNES;1
08496;;Wagnon;WAGNON;08270;08;ARDENNES;1
08497;;Warcq;WARCQ;08000;08;ARDENNES;1
08498;;Warncourt;WARNECOURT;08090;08;ARDENNES;1
08499;;Wasigny;WASIGNY;08270;08;ARDENNES;1
08500;;Wignicourt;WIGNICOURT;08270;08;ARDENNES;1
08501;;Williers;WILLIERS;08110;08;ARDENNES;1
08502;;Yoncq;YONCQ;08210;08;ARDENNES;1
08503;;Yvernaumont;YVERNAUMONT;08430;08;ARDENNES;1
09001;;Aigues-Juntes;AIGUES-JUNTES;09240;09;ARIEGE;1
09002;;Aigues-Vives;AIGUES-VIVES;09600;09;ARIEGE;1
09003;L';Aiguillon;AIGUILLON;09300;09;ARIEGE;1
09004;;Albis;ALBIES;09310;09;ARIEGE;1
09005;;Aleu;ALEU;09320;09;ARIEGE;1
09006;;Alliat;ALLIAT;09400;09;ARIEGE;1
09007;;Allires;ALLIERES;09240;09;ARIEGE;1
09008;;Alos;ALOS;09200;09;ARIEGE;1
09009;;Alzen;ALZEN;09240;09;ARIEGE;1
09011;;Antras;ANTRAS;09800;09;ARIEGE;1
09012;;Appy;APPY;09250;09;ARIEGE;1
09013;;Arabaux;ARABAUX;09000;09;ARIEGE;1
09014;;Argein;ARGEIN;09800;09;ARIEGE;1
09015;;Arignac;ARIGNAC;09400;09;ARIEGE;1
09016;;Arnave;ARNAVE;09400;09;ARIEGE;1
09017;;Arrien-en-Bethmale;ARRIEN-EN-BETHMALE;09800;09;ARIEGE;1
09018;;Arrout;ARROUT;09800;09;ARIEGE;1
09019;;Artigat;ARTIGAT;09130;09;ARIEGE;1
09020;;Artigues;ARTIGUES;09460;09;ARIEGE;1
09021;;Artix;ARTIX;09120;09;ARIEGE;1
09022;;Arvigna;ARVIGNA;09100;09;ARIEGE;1
09023;;Ascou;ASCOU;09110;09;ARIEGE;1
09024;;Aston;ASTON;09310;09;ARIEGE;1
09025;;Aucazein;AUCAZEIN;09800;09;ARIEGE;1
09026;;Audressein;AUDRESSEIN;09800;09;ARIEGE;1
09027;;Augirein;AUGIREIN;09800;09;ARIEGE;1
09028;;Aulos;AULOS;09310;09;ARIEGE;1
09029;;Aulus-les-Bains;AULUS-LES-BAINS;09140;09;ARIEGE;1
09030;;Auzat;AUZAT;09220;09;ARIEGE;1
09031;;Axiat;AXIAT;09250;09;ARIEGE;1
09032;;Ax-les-Thermes;AX-LES-THERMES;09110;09;ARIEGE;1
09033;;Bagert;BAGERT;09160;09;ARIEGE;1
09034;;Balacet;BALACET;09800;09;ARIEGE;1
09035;;Balagures;BALAGUERES;09800;09;ARIEGE;1
09037;;Barjac;BARJAC;09190;09;ARIEGE;1
09038;La;Bastide-de-Besplas;BASTIDE-DE-BESPLAS;09350;09;ARIEGE;1
09039;La;Bastide-de-Bousignac;BASTIDE-DE-BOUSIGNAC;09500;09;ARIEGE;1
09040;La;Bastide-de-Lordat;BASTIDE-DE-LORDAT;09700;09;ARIEGE;1
09041;La;Bastide-du-Salat;BASTIDE-DU-SALAT;09160;09;ARIEGE;1
09042;La;Bastide-de-Srou;BASTIDE-DE-SEROU;09240;09;ARIEGE;1
09043;La;Bastide-sur-l'Hers;BASTIDE-SUR-L'HERS;09600;09;ARIEGE;1
09044;;Baulou;BAULOU;09000;09;ARIEGE;1
09045;;Bdeilhac-et-Aynat;BEDEILHAC-ET-AYNAT;09400;09;ARIEGE;1
09046;;Bdeille;BEDEILLE;09230;09;ARIEGE;1
09047;;Blesta;BELESTA;09300;09;ARIEGE;1
09048;;Belloc;BELLOC;09600;09;ARIEGE;1
09049;;Bnac;BENAC;09000;09;ARIEGE;1
09050;;Benagues;BENAGUES;09100;09;ARIEGE;1
09051;;Bnaix;BENAIX;09300;09;ARIEGE;1
09052;;Besset;BESSET;09500;09;ARIEGE;1
09053;;Bestiac;BESTIAC;09250;09;ARIEGE;1
09054;;Betchat;BETCHAT;09160;09;ARIEGE;1
09055;;Bethmale;BETHMALE;09800;09;ARIEGE;1
09056;;Bzac;BEZAC;09100;09;ARIEGE;1
09057;;Biert;BIERT;09320;09;ARIEGE;1
09058;;Bompas;BOMPAS;09400;09;ARIEGE;1
09059;;Bonac-Irazein;BONAC-IRAZEIN;09800;09;ARIEGE;1
09060;;Bonnac;BONNAC;09100;09;ARIEGE;1
09061;Les;Bordes-sur-Arize;BORDES-SUR-ARIZE;09350;09;ARIEGE;1
09062;Les;Bordes-sur-Lez;BORDES-SUR-LEZ;09800;09;ARIEGE;1
09063;Le;Bosc;BOSC;09000;09;ARIEGE;1
09064;;Bouan;BOUAN;09310;09;ARIEGE;1
09065;;Boussenac;BOUSSENAC;09320;09;ARIEGE;1
09066;;Brassac;BRASSAC;09000;09;ARIEGE;1
09067;;Brie;BRIE;09700;09;ARIEGE;1
09068;;Burret;BURRET;09000;09;ARIEGE;1
09069;;Buzan;BUZAN;09800;09;ARIEGE;1
09070;Les;Cabannes;CABANNES;09310;09;ARIEGE;1
09071;;Cadarcet;CADARCET;09240;09;ARIEGE;1
09072;;Calzan;CALZAN;09120;09;ARIEGE;1
09073;;Camarade;CAMARADE;09290;09;ARIEGE;1
09074;;Camon;CAMON;09500;09;ARIEGE;1
09075;;Campagne-sur-Arize;CAMPAGNE-SUR-ARIZE;09350;09;ARIEGE;1
09076;;Cant;CANTE;09700;09;ARIEGE;1
09077;;Capoulet-et-Junac;CAPOULET-ET-JUNAC;09400;09;ARIEGE;1
09078;;Carcanires;CARCANIERES;09460;09;ARIEGE;1
09079;;Carla-Bayle;CARLA-BAYLE;09130;09;ARIEGE;1
09080;;Carla-de-Roquefort;CARLA-DE-ROQUEFORT;09300;09;ARIEGE;1
09081;Le;Carlaret;CARLARET;09100;09;ARIEGE;1
09082;;Castelnau-Durban;CASTELNAU-DURBAN;09420;09;ARIEGE;1
09083;;Castras;CASTERAS;09130;09;ARIEGE;1
09084;;Castex;CASTEX;09350;09;ARIEGE;1
09085;;Castillon-en-Couserans;CASTILLON-EN-COUSERANS;09800;09;ARIEGE;1
09086;;Caumont;CAUMONT;09160;09;ARIEGE;1
09087;;Caussou;CAUSSOU;09250;09;ARIEGE;1
09088;;Caychax;CAYCHAX;09250;09;ARIEGE;1
09089;;Cazals-des-Bayls;CAZALS-DES-BAYLES;09500;09;ARIEGE;1
09090;;Cazaux;CAZAUX;09120;09;ARIEGE;1
09091;;Cazavet;CAZAVET;09160;09;ARIEGE;1
09092;;Cazenave-Serres-et-Allens;CAZENAVE-SERRES-ET-ALLENS;09400;09;ARIEGE;1
09093;;Celles;CELLES;09000;09;ARIEGE;1
09094;;Crizols;CERIZOLS;09230;09;ARIEGE;1
09095;;Cescau;CESCAU;09800;09;ARIEGE;1
09096;;Chteau-Verdun;CHATEAU-VERDUN;09310;09;ARIEGE;1
09097;;Clermont;CLERMONT;09420;09;ARIEGE;1
09098;;Contrazy;CONTRAZY;09230;09;ARIEGE;1
09099;;Cos;COS;09000;09;ARIEGE;1
09100;;Couflens;COUFLENS;09140;09;ARIEGE;1
09101;;Coussa;COUSSA;09120;09;ARIEGE;1
09102;;Coutens;COUTENS;09500;09;ARIEGE;1
09103;;Crampagna;CRAMPAGNA;09120;09;ARIEGE;1
09104;;Dalou;DALOU;09120;09;ARIEGE;1
09105;;Daumazan-sur-Arize;DAUMAZAN-SUR-ARIZE;09350;09;ARIEGE;1
09106;;Dreuilhe;DREUILHE;09300;09;ARIEGE;1
09107;;Dun;DUN;09600;09;ARIEGE;1
09108;;Durban-sur-Arize;DURBAN-SUR-ARIZE;09240;09;ARIEGE;1
09109;;Durfort;DURFORT;09130;09;ARIEGE;1
09110;;Encourtiech;ENCOURTIECH;09200;09;ARIEGE;1
09111;;Engomer;ENGOMER;09800;09;ARIEGE;1
09113;;Erc;ERCE;09140;09;ARIEGE;1
09114;;Erp;ERP;09200;09;ARIEGE;1
09115;;Esclagne;ESCLAGNE;09600;09;ARIEGE;1
09116;;Escosse;ESCOSSE;09100;09;ARIEGE;1
09117;;Esplas;ESPLAS;09700;09;ARIEGE;1
09118;;Esplas-de-Srou;ESPLAS-DE-SEROU;09420;09;ARIEGE;1
09119;;Eycheil;EYCHEIL;09200;09;ARIEGE;1
09120;;Fabas;FABAS;09230;09;ARIEGE;1
09121;;Ferrires-sur-Arige;FERRIERES-SUR-ARIEGE;09000;09;ARIEGE;1
09122;;Foix;FOIX;09000;09;ARIEGE;1
09123;;Fornex;FORNEX;09350;09;ARIEGE;1
09124;Le;Fossat;FOSSAT;09130;09;ARIEGE;1
09125;;Fougax-et-Barrineuf;FOUGAX-ET-BARRINEUF;09300;09;ARIEGE;1
09126;;Freychenet;FREYCHENET;09300;09;ARIEGE;1
09127;;Gabre;GABRE;09290;09;ARIEGE;1
09128;;Gajan;GAJAN;09190;09;ARIEGE;1
09129;;Galey;GALEY;09800;09;ARIEGE;1
09130;;Ganac;GANAC;09000;09;ARIEGE;1
09131;;Garanou;GARANOU;09250;09;ARIEGE;1
09132;;Gaudis;GAUDIES;09700;09;ARIEGE;1
09133;;Gnat;GENAT;09400;09;ARIEGE;1
09134;;Gestis;GESTIES;09220;09;ARIEGE;1
09135;;Goulier;GOULIER;09220;09;ARIEGE;1
09136;;Gourbit;GOURBIT;09400;09;ARIEGE;1
09137;;Gudas;GUDAS;09120;09;ARIEGE;1
09138;L';Herm;HERM;09000;09;ARIEGE;1
09139;L';Hospitalet-prs-l'Andorre;HOSPITALET-PRES-L'ANDORRE;09390;09;ARIEGE;1
09140;;Ignaux;IGNAUX;09110;09;ARIEGE;1
09141;;Illartein;ILLARTEIN;09800;09;ARIEGE;1
09142;;Ilhat;ILHAT;09300;09;ARIEGE;1
09143;;Illier-et-Laramade;ILLIER-ET-LARAMADE;09220;09;ARIEGE;1
09145;Les;Issards;ISSARDS;09100;09;ARIEGE;1
09146;;Justiniac;JUSTINIAC;09700;09;ARIEGE;1
09147;;Labatut;LABATUT;09700;09;ARIEGE;1
09148;;Lacave;LACAVE;09160;09;ARIEGE;1
09149;;Lacourt;LACOURT;09200;09;ARIEGE;1
09150;;Lagarde;LAGARDE;09500;09;ARIEGE;1
09151;;Lanoux;LANOUX;09130;09;ARIEGE;1
09152;;Lapge;LAPEGE;09400;09;ARIEGE;1
09153;;Lapenne;LAPENNE;09500;09;ARIEGE;1
09154;;Larbont;LARBONT;09240;09;ARIEGE;1
09155;;Larcat;LARCAT;09310;09;ARIEGE;1
09156;;Larnat;LARNAT;09310;09;ARIEGE;1
09157;;Laroque-d'Olmes;LAROQUE-D'OLMES;09600;09;ARIEGE;1
09158;;Lasserre;LASSERRE;09230;09;ARIEGE;1
09159;;Lassur;LASSUR;09310;09;ARIEGE;1
09160;;Lavelanet;LAVELANET;09300;09;ARIEGE;1
09161;;Lran;LERAN;09600;09;ARIEGE;1
09162;;Lercoul;LERCOUL;09220;09;ARIEGE;1
09163;;Lescousse;LESCOUSSE;09100;09;ARIEGE;1
09164;;Lescure;LESCURE;09420;09;ARIEGE;1
09165;;Lesparrou;LESPARROU;09300;09;ARIEGE;1
09166;;Leychert;LEYCHERT;09300;09;ARIEGE;1
09167;;Lzat-sur-Lze;LEZAT-SUR-LEZE;09210;09;ARIEGE;1
09168;;Lieurac;LIEURAC;09300;09;ARIEGE;1
09169;;Limbrassac;LIMBRASSAC;09600;09;ARIEGE;1
09170;;Lissac;LISSAC;09700;09;ARIEGE;1
09171;;Lordat;LORDAT;09250;09;ARIEGE;1
09172;;Loubaut;LOUBAUT;09350;09;ARIEGE;1
09173;;Loubens;LOUBENS;09120;09;ARIEGE;1
09174;;Loubires;LOUBIERES;09000;09;ARIEGE;1
09175;;Ludis;LUDIES;09100;09;ARIEGE;1
09176;;Luzenac;LUZENAC;09250;09;ARIEGE;1
09177;;Madire;MADIERE;09100;09;ARIEGE;1
09178;;Malegoude;MALEGOUDE;09500;09;ARIEGE;1
09179;;Mallon;MALLEON;09120;09;ARIEGE;1
09180;;Manses;MANSES;09500;09;ARIEGE;1
09181;Le;Mas-d'Azil;MAS-D'AZIL;09290;09;ARIEGE;1
09182;;Massat;MASSAT;09320;09;ARIEGE;1
09183;;Mauvezin-de-Prat;MAUVEZIN-DE-PRAT;09160;09;ARIEGE;1
09184;;Mauvezin-de-Sainte-Croix;MAUVEZIN-DE-SAINTE-CROIX;09230;09;ARIEGE;1
09185;;Mazres;MAZERES;09270;09;ARIEGE;1
09186;;Mras;MERAS;09350;09;ARIEGE;1
09187;;Mercenac;MERCENAC;09160;09;ARIEGE;1
09188;;Mercus-Garrabet;MERCUS-GARRABET;09400;09;ARIEGE;1
09189;;Mrens-les-Vals;MERENS-LES-VALS;09110;09;ARIEGE;1
09190;;Mrigon;MERIGON;09230;09;ARIEGE;1
09192;;Miglos;MIGLOS;09400;09;ARIEGE;1
09193;;Mijans;MIJANES;09460;09;ARIEGE;1
09194;;Mirepoix;MIREPOIX;09500;09;ARIEGE;1
09195;;Monesple;MONESPLE;09130;09;ARIEGE;1
09196;;Montagagne;MONTAGAGNE;09240;09;ARIEGE;1
09197;;Montaillou;MONTAILLOU;09110;09;ARIEGE;1
09198;;Montardit;MONTARDIT;09230;09;ARIEGE;1
09199;;Montaut;MONTAUT;09700;09;ARIEGE;1
09200;;Montbel;MONTBEL;09600;09;ARIEGE;1
09201;;Montgut-en-Couserans;MONTEGUT-EN-COUSERANS;09200;09;ARIEGE;1
09202;;Montgut-Plantaurel;MONTEGUT-PLANTAUREL;09120;09;ARIEGE;1
09203;;Montels;MONTELS;09240;09;ARIEGE;1
09204;;Montesquieu-Avants;MONTESQUIEU-AVANTES;09200;09;ARIEGE;1
09205;;Montfa;MONTFA;09350;09;ARIEGE;1
09206;;Montferrier;MONTFERRIER;09300;09;ARIEGE;1
09207;;Montgaillard;MONTGAILLARD;09330;09;ARIEGE;1
09208;;Montgauch;MONTGAUCH;09160;09;ARIEGE;1
09209;;Montjoie-en-Couserans;MONTJOIE-EN-COUSERANS;09200;09;ARIEGE;1
09210;;Montoulieu;MONTOULIEU;09000;09;ARIEGE;1
09211;;Montsgur;MONTSEGUR;09300;09;ARIEGE;1
09212;;Montseron;MONTSERON;09240;09;ARIEGE;1
09213;;Moulin-Neuf;MOULIN-NEUF;09500;09;ARIEGE;1
09214;;Moulis;MOULIS;09200;09;ARIEGE;1
09215;;Nalzen;NALZEN;09300;09;ARIEGE;1
09216;;Nescus;NESCUS;09240;09;ARIEGE;1
09217;;Niaux;NIAUX;09400;09;ARIEGE;1
09218;;Orgeix;ORGEIX;09110;09;ARIEGE;1
09219;;Orgibet;ORGIBET;09800;09;ARIEGE;1
09220;;Orlu;ORLU;09110;09;ARIEGE;1
09221;;Ornolac-Ussat-les-Bains;ORNOLAC-USSAT-LES-BAINS;09400;09;ARIEGE;1
09222;;Orus;ORUS;09220;09;ARIEGE;1
09223;;Oust;OUST;09140;09;ARIEGE;1
09224;;Pailhs;PAILHES;09130;09;ARIEGE;1
09225;;Pamiers;PAMIERS;09100;09;ARIEGE;1
09226;;Pech;PECH;09310;09;ARIEGE;1
09227;;Preille;PEREILLE;09300;09;ARIEGE;1
09228;;Perles-et-Castelet;PERLES-ET-CASTELET;09110;09;ARIEGE;1
09229;Le;Peyrat;PEYRAT;09600;09;ARIEGE;1
09230;Le;Pla;PLA;09460;09;ARIEGE;1
09231;Le;Port;PORT;09320;09;ARIEGE;1
09232;;Prades;PRADES;09110;09;ARIEGE;1
09233;;Pradettes;PRADETTES;09600;09;ARIEGE;1
09234;;Pradires;PRADIERES;09000;09;ARIEGE;1
09235;;Prat-Bonrepaux;PRAT-BONREPAUX;09160;09;ARIEGE;1
09236;;Prayols;PRAYOLS;09000;09;ARIEGE;1
09237;Le;Puch;PUCH;09460;09;ARIEGE;1
09238;Les;Pujols;PUJOLS;09100;09;ARIEGE;1
09239;;Qurigut;QUERIGUT;09460;09;ARIEGE;1
09240;;Qui;QUIE;09400;09;ARIEGE;1
09241;;Rabat-les-Trois-Seigneurs;RABAT-LES-TROIS-SEIGNEURS;09400;09;ARIEGE;1
09242;;Raissac;RAISSAC;09300;09;ARIEGE;1
09243;;Rgat;REGAT;09600;09;ARIEGE;1
09244;;Rieucros;RIEUCROS;09500;09;ARIEGE;1
09245;;Rieux-de-Pelleport;RIEUX-DE-PELLEPORT;09120;09;ARIEGE;1
09246;;Rimont;RIMONT;09420;09;ARIEGE;1
09247;;Rivrenert;RIVERENERT;09200;09;ARIEGE;1
09249;;Roquefixade;ROQUEFIXADE;09300;09;ARIEGE;1
09250;;Roquefort-les-Cascades;ROQUEFORT-LES-CASCADES;09300;09;ARIEGE;1
09251;;Roumengoux;ROUMENGOUX;09500;09;ARIEGE;1
09252;;Rouze;ROUZE;09460;09;ARIEGE;1
09253;;Sabarat;SABARAT;09350;09;ARIEGE;1
09254;;Saint-Amadou;SAINT-AMADOU;09100;09;ARIEGE;1
09255;;Saint-Amans;SAINT-AMANS;09100;09;ARIEGE;1
09256;;Saint-Bauzeil;SAINT-BAUZEIL;09120;09;ARIEGE;1
09257;;Sainte-Croix-Volvestre;SAINTE-CROIX-VOLVESTRE;09230;09;ARIEGE;1
09258;;Saint-Flix-de-Rieutord;SAINT-FELIX-DE-RIEUTORD;09120;09;ARIEGE;1
09259;;Saint-Flix-de-Tournegat;SAINT-FELIX-DE-TOURNEGAT;09500;09;ARIEGE;1
09260;;Sainte-Foi;SAINTE-FOI;09500;09;ARIEGE;1
09261;;Saint-Girons;SAINT-GIRONS;09200;09;ARIEGE;1
09262;;Saint-Jean-d'Aigues-Vives;SAINT-JEAN-D'AIGUES-VIVES;09300;09;ARIEGE;1
09263;;Saint-Jean-du-Castillonnais;SAINT-JEAN-DU-CASTILLONNAIS;09800;09;ARIEGE;1
09264;;Saint-Jean-de-Verges;SAINT-JEAN-DE-VERGES;09000;09;ARIEGE;1
09265;;Saint-Jean-du-Falga;SAINT-JEAN-DU-FALGA;09100;09;ARIEGE;1
09266;;Saint-Julien-de-Gras-Capou;SAINT-JULIEN-DE-GRAS-CAPOU;09500;09;ARIEGE;1
09267;;Saint-Lary;SAINT-LARY;09800;09;ARIEGE;1
09268;;Saint-Lizier;SAINT-LIZIER;09190;09;ARIEGE;1
09269;;Saint-Martin-de-Caralp;SAINT-MARTIN-DE-CARALP;09000;09;ARIEGE;1
09270;;Saint-Martin-d'Oydes;SAINT-MARTIN-D'OYDES;09100;09;ARIEGE;1
09271;;Saint-Michel;SAINT-MICHEL;09100;09;ARIEGE;1
09272;;Saint-Paul-de-Jarrat;SAINT-PAUL-DE-JARRAT;09000;09;ARIEGE;1
09273;;Saint-Pierre-de-Rivire;SAINT-PIERRE-DE-RIVIERE;09000;09;ARIEGE;1
09274;;Saint-Quentin-la-Tour;SAINT-QUENTIN-LA-TOUR;09500;09;ARIEGE;1
09275;;Saint-Quirc;SAINT-QUIRC;09700;09;ARIEGE;1
09276;;Saint-Victor-Rouzaud;SAINT-VICTOR-ROUZAUD;09100;09;ARIEGE;1
09277;;Saint-Ybars;SAINT-YBARS;09210;09;ARIEGE;1
09279;;Salsein;SALSEIN;09800;09;ARIEGE;1
09280;;Saurat;SAURAT;09400;09;ARIEGE;1
09281;;Sautel;SAUTEL;09300;09;ARIEGE;1
09282;;Saverdun;SAVERDUN;09700;09;ARIEGE;1
09283;;Savignac-les-Ormeaux;SAVIGNAC-LES-ORMEAUX;09110;09;ARIEGE;1
09284;;Sgura;SEGURA;09120;09;ARIEGE;1
09285;;Seix;SEIX;09140;09;ARIEGE;1
09286;;Sem;SEM;09220;09;ARIEGE;1
09287;;Senconac;SENCONAC;09250;09;ARIEGE;1
09289;;Lorp-Sentaraille;LORP-SENTARAILLE;09190;09;ARIEGE;1
09290;;Sentein;SENTEIN;09800;09;ARIEGE;1
09291;;Sentenac-d'Oust;SENTENAC-D'OUST;09140;09;ARIEGE;1
09292;;Sentenac-de-Srou;SENTENAC-DE-SEROU;09240;09;ARIEGE;1
09293;;Serres-sur-Arget;SERRES-SUR-ARGET;09000;09;ARIEGE;1
09294;;Sieuras;SIEURAS;09130;09;ARIEGE;1
09295;;Siguer;SIGUER;09220;09;ARIEGE;1
09296;;Sinsat;SINSAT;09310;09;ARIEGE;1
09297;;Sor;SOR;09800;09;ARIEGE;1
09298;;Sorgeat;SORGEAT;09110;09;ARIEGE;1
09299;;Soueix-Rogalle;SOUEIX-ROGALLE;09140;09;ARIEGE;1
09300;;Soula;SOULA;09000;09;ARIEGE;1
09301;;Soulan;SOULAN;09320;09;ARIEGE;1
09302;;Suc-et-Sentenac;SUC-ET-SENTENAC;09220;09;ARIEGE;1
09303;;Surba;SURBA;09400;09;ARIEGE;1
09304;;Suzan;SUZAN;09240;09;ARIEGE;1
09305;;Tabre;TABRE;09600;09;ARIEGE;1
09306;;Tarascon-sur-Arige;TARASCON-SUR-ARIEGE;09400;09;ARIEGE;1
09307;;Taurignan-Castet;TAURIGNAN-CASTET;09160;09;ARIEGE;1
09308;;Taurignan-Vieux;TAURIGNAN-VIEUX;09190;09;ARIEGE;1
09309;;Teilhet;TEILHET;09500;09;ARIEGE;1
09310;;Thouars-sur-Arize;THOUARS-SUR-ARIZE;09350;09;ARIEGE;1
09311;;Tignac;TIGNAC;09110;09;ARIEGE;1
09312;La;Tour-du-Crieu;TOUR-DU-CRIEU;09100;09;ARIEGE;1
09313;;Tourtouse;TOURTOUSE;09230;09;ARIEGE;1
09314;;Tourtrol;TOURTROL;09500;09;ARIEGE;1
09315;;Trmoulet;TREMOULET;09700;09;ARIEGE;1
09316;;Troye-d'Arige;TROYE-D'ARIEGE;09500;09;ARIEGE;1
09317;;Uchentein;UCHENTEIN;09800;09;ARIEGE;1
09318;;Unac;UNAC;09250;09;ARIEGE;1
09319;;Unzent;UNZENT;09100;09;ARIEGE;1
09320;;Urs;URS;09310;09;ARIEGE;1
09321;;Ussat;USSAT;09400;09;ARIEGE;1
09322;;Ustou;USTOU;09140;09;ARIEGE;1
09323;;Vals;VALS;09500;09;ARIEGE;1
09324;;Varilhes;VARILHES;09120;09;ARIEGE;1
09325;;Vaychis;VAYCHIS;09110;09;ARIEGE;1
09326;;Vbre;VEBRE;09310;09;ARIEGE;1
09327;;Ventenac;VENTENAC;09120;09;ARIEGE;1
09328;;Verdun;VERDUN;09310;09;ARIEGE;1
09329;;Vernajoul;VERNAJOUL;09000;09;ARIEGE;1
09330;;Vernaux;VERNAUX;09250;09;ARIEGE;1
09331;Le;Vernet;VERNET;09700;09;ARIEGE;1
09332;;Verniolle;VERNIOLLE;09340;09;ARIEGE;1
09334;;Vicdessos;VICDESSOS;09220;09;ARIEGE;1
09335;;Villeneuve;VILLENEUVE;09800;09;ARIEGE;1
09336;;Villeneuve-d'Olmes;VILLENEUVE-D'OLMES;09300;09;ARIEGE;1
09338;;Villeneuve-du-Latou;VILLENEUVE-DU-LATOU;09130;09;ARIEGE;1
09339;;Villeneuve-du-Parage;VILLENEUVE-DU-PAREAGE;09100;09;ARIEGE;1
09340;;Vira;VIRA;09120;09;ARIEGE;1
09341;;Vivis;VIVIES;09500;09;ARIEGE;1
09342;;Sainte-Suzanne;SAINTE-SUZANNE;09130;09;ARIEGE;1
10002;;Ailleville;AILLEVILLE;10200;10;AUBE;1
10003;;Aix-en-Othe;AIX-EN-OTHE;10160;10;AUBE;1
10004;;Allibaudires;ALLIBAUDIERES;10700;10;AUBE;1
10005;;Amance;AMANCE;10140;10;AUBE;1
10006;;Arcis-sur-Aube;ARCIS-SUR-AUBE;10700;10;AUBE;1
10007;;Arconville;ARCONVILLE;10200;10;AUBE;1
10008;;Arganon;ARGANCON;10140;10;AUBE;1
10009;;Arrelles;ARRELLES;10340;10;AUBE;1
10010;;Arrembcourt;ARREMBECOURT;10330;10;AUBE;1
10011;;Arrentires;ARRENTIERES;10200;10;AUBE;1
10012;;Arsonval;ARSONVAL;10200;10;AUBE;1
10013;;Assenay;ASSENAY;10320;10;AUBE;1
10014;;Assencires;ASSENCIERES;10220;10;AUBE;1
10015;;Aubeterre;AUBETERRE;10150;10;AUBE;1
10017;;Aulnay;AULNAY;10240;10;AUBE;1
10018;;Auxon;AUXON;10130;10;AUBE;1
10019;;Val-d'Auzon;VAL-D'AUZON;10220;10;AUBE;1
10020;;Avant-ls-Marcilly;AVANT-LES-MARCILLY;10400;10;AUBE;1
10021;;Avant-ls-Ramerupt;AVANT-LES-RAMERUPT;10240;10;AUBE;1
10022;;Avirey-Lingey;AVIREY-LINGEY;10340;10;AUBE;1
10023;;Avon-la-Pze;AVON-LA-PEZE;10290;10;AUBE;1
10024;;Avreuil;AVREUIL;10130;10;AUBE;1
10025;;Bagneux-la-Fosse;BAGNEUX-LA-FOSSE;10340;10;AUBE;1
10026;;Bailly-le-Franc;BAILLY-LE-FRANC;10330;10;AUBE;1
10027;;Balignicourt;BALIGNICOURT;10330;10;AUBE;1
10028;;Balnot-la-Grange;BALNOT-LA-GRANGE;10210;10;AUBE;1
10029;;Balnot-sur-Laignes;BALNOT-SUR-LAIGNES;10110;10;AUBE;1
10030;;Barberey-Saint-Sulpice;BARBEREY-SAINT-SULPICE;10600;10;AUBE;1
10031;;Barbuise;BARBUISE;10400;10;AUBE;1
10032;;Baroville;BAROVILLE;10200;10;AUBE;1
10033;;Bar-sur-Aube;BAR-SUR-AUBE;10200;10;AUBE;1
10034;;Bar-sur-Seine;BAR-SUR-SEINE;10110;10;AUBE;1
10035;;Bayel;BAYEL;10310;10;AUBE;1
10037;;Bercenay-en-Othe;BERCENAY-EN-OTHE;10190;10;AUBE;1
10038;;Bercenay-le-Hayer;BERCENAY-LE-HAYER;10290;10;AUBE;1
10039;;Bergres;BERGERES;10200;10;AUBE;1
10040;;Bernon;BERNON;10130;10;AUBE;1
10041;;Bertignolles;BERTIGNOLLES;10110;10;AUBE;1
10042;;Brulle;BERULLE;10160;10;AUBE;1
10043;;Bessy;BESSY;10170;10;AUBE;1
10044;;Btignicourt;BETIGNICOURT;10500;10;AUBE;1
10045;;Beurey;BEUREY;10140;10;AUBE;1
10046;;Blaincourt-sur-Aube;BLAINCOURT-SUR-AUBE;10500;10;AUBE;1
10047;;Blignicourt;BLIGNICOURT;10500;10;AUBE;1
10048;;Bligny;BLIGNY;10200;10;AUBE;1
10049;Les;Bordes-Aumont;BORDES-AUMONT;10800;10;AUBE;1
10050;;Bossancourt;BOSSANCOURT;10140;10;AUBE;1
10051;;Bouilly;BOUILLY;10320;10;AUBE;1
10052;;Boulages;BOULAGES;10380;10;AUBE;1
10053;;Bouranton;BOURANTON;10270;10;AUBE;1
10054;;Bourdenay;BOURDENAY;10290;10;AUBE;1
10055;;Bourguignons;BOURGUIGNONS;10110;10;AUBE;1
10056;;Bouy-Luxembourg;BOUY-LUXEMBOURG;10220;10;AUBE;1
10057;;Bouy-sur-Orvin;BOUY-SUR-ORVIN;10400;10;AUBE;1
10058;;Bragelogne-Beauvoir;BRAGELOGNE-BEAUVOIR;10340;10;AUBE;1
10059;;Braux;BRAUX;10500;10;AUBE;1
10060;;Brviandes;BREVIANDES;10800;10;AUBE;1
10061;;Brvonnes;BREVONNES;10220;10;AUBE;1
10062;;Briel-sur-Barse;BRIEL-SUR-BARSE;10140;10;AUBE;1
10063;;Brienne-la-Vieille;BRIENNE-LA-VIEILLE;10500;10;AUBE;1
10064;;Brienne-le-Chteau;BRIENNE-LE-CHATEAU;10500;10;AUBE;1
10065;;Brillecourt;BRILLECOURT;10240;10;AUBE;1
10066;;Bucey-en-Othe;BUCEY-EN-OTHE;10190;10;AUBE;1
10067;;Buchres;BUCHERES;10800;10;AUBE;1
10068;;Buxeuil;BUXEUIL;10110;10;AUBE;1
10069;;Buxires-sur-Arce;BUXIERES-SUR-ARCE;10110;10;AUBE;1
10070;;Celles-sur-Ource;CELLES-SUR-OURCE;10110;10;AUBE;1
10071;;Chacenay;CHACENAY;10110;10;AUBE;1
10072;La;Chaise;CHAISE;10500;10;AUBE;1
10073;;Chalette-sur-Voire;CHALETTE-SUR-VOIRE;10500;10;AUBE;1
10074;;Chamoy;CHAMOY;10130;10;AUBE;1
10075;;Champfleury;CHAMPFLEURY;10700;10;AUBE;1
10076;;Champignol-lez-Mondeville;CHAMPIGNOL-LEZ-MONDEVILLE;10200;10;AUBE;1
10077;;Champigny-sur-Aube;CHAMPIGNY-SUR-AUBE;10700;10;AUBE;1
10078;;Champ-sur-Barse;CHAMP-SUR-BARSE;10140;10;AUBE;1
10079;;Channes;CHANNES;10340;10;AUBE;1
10080;;Chaource;CHAOURCE;10210;10;AUBE;1
10081;La;Chapelle-Saint-Luc;CHAPELLE-SAINT-LUC;10600;10;AUBE;1
10082;;Chapelle-Vallon;CHAPELLE-VALLON;10700;10;AUBE;1
10083;;Chappes;CHAPPES;10260;10;AUBE;1
10084;;Charmont-sous-Barbuise;CHARMONT-SOUS-BARBUISE;10150;10;AUBE;1
10085;;Charmoy;CHARMOY;10290;10;AUBE;1
10086;;Charny-le-Bachot;CHARNY-LE-BACHOT;10380;10;AUBE;1
10087;;Chaserey;CHASEREY;10210;10;AUBE;1
10089;;Chtres;CHATRES;10510;10;AUBE;1
10090;;Chauchigny;CHAUCHIGNY;10170;10;AUBE;1
10091;;Chaudrey;CHAUDREY;10240;10;AUBE;1
10092;;Chauffour-ls-Bailly;CHAUFFOUR-LES-BAILLY;10110;10;AUBE;1
10093;;Chaumesnil;CHAUMESNIL;10500;10;AUBE;1
10094;;Chavanges;CHAVANGES;10330;10;AUBE;1
10095;Le;Chne;CHENE;10700;10;AUBE;1
10096;;Chennegy;CHENNEGY;10190;10;AUBE;1
10097;;Chervey;CHERVEY;10110;10;AUBE;1
10098;;Chesley;CHESLEY;10210;10;AUBE;1
10099;;Chessy-les-Prs;CHESSY-LES-PRES;10130;10;AUBE;1
10100;;Clrey;CLEREY;10390;10;AUBE;1
10101;;Coclois;COCLOIS;10240;10;AUBE;1
10102;;Colomb-la-Fosse;COLOMBE-LA-FOSSE;10200;10;AUBE;1
10103;;Colomb-le-Sec;COLOMBE-LE-SEC;10200;10;AUBE;1
10104;;Cormost;CORMOST;10800;10;AUBE;1
10105;;Courcelles-sur-Voire;COURCELLES-SUR-VOIRE;10500;10;AUBE;1
10106;;Courceroy;COURCEROY;10400;10;AUBE;1
10107;;Coursan-en-Othe;COURSAN-EN-OTHE;10130;10;AUBE;1
10108;;Courtaoult;COURTAOULT;10130;10;AUBE;1
10109;;Courtenot;COURTENOT;10260;10;AUBE;1
10110;;Courteranges;COURTERANGES;10270;10;AUBE;1
10111;;Courteron;COURTERON;10250;10;AUBE;1
10112;;Coussegrey;COUSSEGREY;10210;10;AUBE;1
10113;;Couvignon;COUVIGNON;10200;10;AUBE;1
10114;;Crancey;CRANCEY;10100;10;AUBE;1
10115;;Creney-prs-Troyes;CRENEY-PRES-TROYES;10150;10;AUBE;1
10116;;Crsantignes;CRESANTIGNES;10320;10;AUBE;1
10117;;Crespy-le-Neuf;CRESPY-LE-NEUF;10500;10;AUBE;1
10118;Les;Crotes;CROUTES;10130;10;AUBE;1
10119;;Cunfin;CUNFIN;10360;10;AUBE;1
10120;;Cussangy;CUSSANGY;10210;10;AUBE;1
10121;;Dampierre;DAMPIERRE;10240;10;AUBE;1
10122;;Davrey;DAVREY;10130;10;AUBE;1
10123;;Dienville;DIENVILLE;10500;10;AUBE;1
10124;;Dierrey-Saint-Julien;DIERREY-SAINT-JULIEN;10190;10;AUBE;1
10125;;Dierrey-Saint-Pierre;DIERREY-SAINT-PIERRE;10190;10;AUBE;1
10126;;Dolancourt;DOLANCOURT;10200;10;AUBE;1
10127;;Dommartin-le-Coq;DOMMARTIN-LE-COQ;10240;10;AUBE;1
10128;;Donnement;DONNEMENT;10330;10;AUBE;1
10129;;Dosches;DOSCHES;10220;10;AUBE;1
10130;;Dosnon;DOSNON;10700;10;AUBE;1
10131;;Droupt-Saint-Basle;DROUPT-SAINT-BASLE;10170;10;AUBE;1
10132;;Droupt-Sainte-Marie;DROUPT-SAINTE-MARIE;10170;10;AUBE;1
10133;;Eaux-Puiseaux;EAUX-PUISEAUX;10130;10;AUBE;1
10134;;chemines;ECHEMINES;10350;10;AUBE;1
10135;;clance;ECLANCE;10200;10;AUBE;1
10136;;guilly-sous-Bois;EGUILLY-SOUS-BOIS;10110;10;AUBE;1
10137;;Engente;ENGENTE;10200;10;AUBE;1
10138;;pagne;EPAGNE;10500;10;AUBE;1
10139;;pothmont;EPOTHEMONT;10500;10;AUBE;1
10140;;Ervy-le-Chtel;ERVY-LE-CHATEL;10130;10;AUBE;1
10141;;Essoyes;ESSOYES;10360;10;AUBE;1
10142;;Estissac;ESTISSAC;10190;10;AUBE;1
10143;;tourvy;ETOURVY;10210;10;AUBE;1
10144;;trelles-sur-Aube;ETRELLES-SUR-AUBE;10170;10;AUBE;1
10145;;Faux-Villecerf;FAUX-VILLECERF;10290;10;AUBE;1
10146;;Fay-ls-Marcilly;FAY-LES-MARCILLY;10290;10;AUBE;1
10147;;Fays-la-Chapelle;FAYS-LA-CHAPELLE;10320;10;AUBE;1
10148;;Ferreux-Quincey;FERREUX-QUINCEY;10400;10;AUBE;1
10149;;Feuges;FEUGES;10150;10;AUBE;1
10150;;Fontaine;FONTAINE;10200;10;AUBE;1
10151;;Fontaine-les-Grs;FONTAINE-LES-GRES;10280;10;AUBE;1
10153;;Fontaine-Mcon;FONTAINE-MACON;10400;10;AUBE;1
10154;;Fontenay-de-Bossery;FONTENAY-DE-BOSSERY;10400;10;AUBE;1
10155;;Fontette;FONTETTE;10360;10;AUBE;1
10156;;Fontvannes;FONTVANNES;10190;10;AUBE;1
10157;La;Fosse-Corduan;FOSSE-CORDUAN;10100;10;AUBE;1
10158;;Fouchres;FOUCHERES;10260;10;AUBE;1
10159;;Fralignes;FRALIGNES;10110;10;AUBE;1
10160;;Fravaux;FRAVAUX;10200;10;AUBE;1
10161;;Fresnay;FRESNAY;10200;10;AUBE;1
10162;;Fresnoy-le-Chteau;FRESNOY-LE-CHATEAU;10270;10;AUBE;1
10163;;Fuligny;FULIGNY;10200;10;AUBE;1
10164;;Glannes;GELANNES;10100;10;AUBE;1
10165;;Graudot;GERAUDOT;10220;10;AUBE;1
10166;Les;Grandes-Chapelles;GRANDES-CHAPELLES;10170;10;AUBE;1
10167;;Grandville;GRANDVILLE;10700;10;AUBE;1
10168;Les;Granges;GRANGES;10210;10;AUBE;1
10169;;Gumery;GUMERY;10400;10;AUBE;1
10170;;Gy-sur-Seine;GYE-SUR-SEINE;10250;10;AUBE;1
10171;;Hampigny;HAMPIGNY;10500;10;AUBE;1
10172;;Herbisse;HERBISSE;10700;10;AUBE;1
10173;;Isle-Aumont;ISLE-AUMONT;10800;10;AUBE;1
10174;;Isle-Aubigny;ISLE-AUBIGNY;10240;10;AUBE;1
10175;;Jasseines;JASSEINES;10330;10;AUBE;1
10176;;Jaucourt;JAUCOURT;10200;10;AUBE;1
10177;;Javernant;JAVERNANT;10320;10;AUBE;1
10178;;Jessains;JESSAINS;10140;10;AUBE;1
10179;;Jeugny;JEUGNY;10320;10;AUBE;1
10180;;Joncreuil;JONCREUIL;10330;10;AUBE;1
10181;;Jully-sur-Sarce;JULLY-SUR-SARCE;10260;10;AUBE;1
10182;;Juvancourt;JUVANCOURT;10310;10;AUBE;1
10183;;Juvanz;JUVANZE;10140;10;AUBE;1
10184;;Juzanvigny;JUZANVIGNY;10500;10;AUBE;1
10185;;Lagesse;LAGESSE;10210;10;AUBE;1
10186;;Laines-aux-Bois;LAINES-AUX-BOIS;10120;10;AUBE;1
10187;;Landreville;LANDREVILLE;10110;10;AUBE;1
10188;;Lantages;LANTAGES;10210;10;AUBE;1
10189;;Lassicourt;LASSICOURT;10500;10;AUBE;1
10190;;Laubressel;LAUBRESSEL;10270;10;AUBE;1
10191;;Lavau;LAVAU;10150;10;AUBE;1
10192;;Lentilles;LENTILLES;10330;10;AUBE;1
10193;;Lesmont;LESMONT;10500;10;AUBE;1
10194;;Lvigny;LEVIGNY;10200;10;AUBE;1
10195;;Lhutre;LHUITRE;10700;10;AUBE;1
10196;;Lignires;LIGNIERES;10130;10;AUBE;1
10197;;Lignol-le-Chteau;LIGNOL-LE-CHATEAU;10200;10;AUBE;1
10198;;Lirey;LIREY;10320;10;AUBE;1
10199;;Loches-sur-Ource;LOCHES-SUR-OURCE;10110;10;AUBE;1
10200;La;Loge-aux-Chvres;LOGE-AUX-CHEVRES;10140;10;AUBE;1
10201;La;Loge-Pomblin;LOGE-POMBLIN;10210;10;AUBE;1
10202;Les;Loges-Margueron;LOGES-MARGUERON;10210;10;AUBE;1
10203;;Longchamp-sur-Aujon;LONGCHAMP-SUR-AUJON;10310;10;AUBE;1
10204;;Longeville-sur-Mogne;LONGEVILLE-SUR-MOGNE;10320;10;AUBE;1
10205;;Longpr-le-Sec;LONGPRE-LE-SEC;10140;10;AUBE;1
10206;;Longsols;LONGSOLS;10240;10;AUBE;1
10207;;Longueville-sur-Aube;LONGUEVILLE-SUR-AUBE;10170;10;AUBE;1
10208;La;Louptire-Thnard;LOUPTIERE-THENARD;10400;10;AUBE;1
10209;;Lusigny-sur-Barse;LUSIGNY-SUR-BARSE;10270;10;AUBE;1
10210;;Luyres;LUYERES;10150;10;AUBE;1
10211;;Macey;MACEY;10300;10;AUBE;1
10212;;Machy;MACHY;10320;10;AUBE;1
10213;;Magnant;MAGNANT;10110;10;AUBE;1
10214;;Magnicourt;MAGNICOURT;10240;10;AUBE;1
10215;;Magny-Fouchard;MAGNY-FOUCHARD;10140;10;AUBE;1
10216;;Mailly-le-Camp;MAILLY-LE-CAMP;10230;10;AUBE;1
10217;;Maison-des-Champs;MAISON-DES-CHAMPS;10140;10;AUBE;1
10218;;Maisons-ls-Chaource;MAISONS-LES-CHAOURCE;10210;10;AUBE;1
10219;;Maisons-ls-Soulaines;MAISONS-LES-SOULAINES;10200;10;AUBE;1
10220;;Maizires-la-Grande-Paroisse;MAIZIERES-LA-GRANDE-PAROISSE;10510;10;AUBE;1
10221;;Maizires-ls-Brienne;MAIZIERES-LES-BRIENNE;10500;10;AUBE;1
10222;;Maraye-en-Othe;MARAYE-EN-OTHE;10160;10;AUBE;1
10223;;Marcilly-le-Hayer;MARCILLY-LE-HAYER;10290;10;AUBE;1
10224;;Marigny-le-Chtel;MARIGNY-LE-CHATEL;10350;10;AUBE;1
10225;;Marnay-sur-Seine;MARNAY-SUR-SEINE;10400;10;AUBE;1
10226;;Marolles-ls-Bailly;MAROLLES-LES-BAILLY;10110;10;AUBE;1
10227;;Marolles-sous-Lignires;MAROLLES-SOUS-LIGNIERES;10130;10;AUBE;1
10228;;Mathaux;MATHAUX;10500;10;AUBE;1
10229;;Maupas;MAUPAS;10320;10;AUBE;1
10230;;Mergey;MERGEY;10600;10;AUBE;1
10231;Le;Mriot;MERIOT;10400;10;AUBE;1
10232;;Merrey-sur-Arce;MERREY-SUR-ARCE;10110;10;AUBE;1
10233;;Mry-sur-Seine;MERY-SUR-SEINE;10170;10;AUBE;1
10234;;Mesgrigny;MESGRIGNY;10170;10;AUBE;1
10235;;Mesnil-la-Comtesse;MESNIL-LA-COMTESSE;10700;10;AUBE;1
10236;;Mesnil-Lettre;MESNIL-LETTRE;10240;10;AUBE;1
10237;;Mesnil-Saint-Loup;MESNIL-SAINT-LOUP;10190;10;AUBE;1
10238;;Mesnil-Saint-Pre;MESNIL-SAINT-PERE;10140;10;AUBE;1
10239;;Mesnil-Sellires;MESNIL-SELLIERES;10220;10;AUBE;1
10240;;Messon;MESSON;10190;10;AUBE;1
10241;;Metz-Robert;METZ-ROBERT;10210;10;AUBE;1
10242;;Meurville;MEURVILLE;10200;10;AUBE;1
10243;;Molins-sur-Aube;MOLINS-SUR-AUBE;10500;10;AUBE;1
10245;;Montaulin;MONTAULIN;10270;10;AUBE;1
10246;;Montceaux-ls-Vaudes;MONTCEAUX-LES-VAUDES;10260;10;AUBE;1
10247;;Montfey;MONTFEY;10130;10;AUBE;1
10248;;Montgueux;MONTGUEUX;10300;10;AUBE;1
10249;;Montiramey;MONTIERAMEY;10270;10;AUBE;1
10250;;Montier-en-l'Isle;MONTIER-EN-L'ISLE;10200;10;AUBE;1
10251;;Montigny-les-Monts;MONTIGNY-LES-MONTS;10130;10;AUBE;1
10252;;Montmartin-le-Haut;MONTMARTIN-LE-HAUT;10140;10;AUBE;1
10253;;Montmorency-Beaufort;MONTMORENCY-BEAUFORT;10330;10;AUBE;1
10254;;Montpothier;MONTPOTHIER;10400;10;AUBE;1
10255;;Montreuil-sur-Barse;MONTREUIL-SUR-BARSE;10270;10;AUBE;1
10256;;Montsuzain;MONTSUZAIN;10150;10;AUBE;1
10257;;Morembert;MOREMBERT;10240;10;AUBE;1
10258;;Morvilliers;MORVILLIERS;10500;10;AUBE;1
10259;La;Motte-Tilly;MOTTE-TILLY;10400;10;AUBE;1
10260;;Moussey;MOUSSEY;10800;10;AUBE;1
10261;;Mussy-sur-Seine;MUSSY-SUR-SEINE;10250;10;AUBE;1
10262;;Neuville-sur-Seine;NEUVILLE-SUR-SEINE;10250;10;AUBE;1
10263;;Neuville-sur-Vanne;NEUVILLE-SUR-VANNE;10190;10;AUBE;1
10264;;No-les-Mallets;NOE-LES-MALLETS;10360;10;AUBE;1
10265;Les;Nos-prs-Troyes;NOES-PRES-TROYES;10420;10;AUBE;1
10266;;Nogent-en-Othe;NOGENT-EN-OTHE;10160;10;AUBE;1
10267;;Nogent-sur-Aube;NOGENT-SUR-AUBE;10240;10;AUBE;1
10268;;Nogent-sur-Seine;NOGENT-SUR-SEINE;10400;10;AUBE;1
10269;;Nozay;NOZAY;10700;10;AUBE;1
10270;;Onjon;ONJON;10220;10;AUBE;1
10271;;Origny-le-Sec;ORIGNY-LE-SEC;10510;10;AUBE;1
10272;;Ormes;ORMES;10700;10;AUBE;1
10273;;Ortillon;ORTILLON;10700;10;AUBE;1
10274;;Orvilliers-Saint-Julien;ORVILLIERS-SAINT-JULIEN;10170;10;AUBE;1
10275;;Ossey-les-Trois-Maisons;OSSEY-LES-TROIS-MAISONS;10100;10;AUBE;1
10276;;Paisy-Cosdon;PAISY-COSDON;10160;10;AUBE;1
10277;;Palis;PALIS;10190;10;AUBE;1
10278;;Pargues;PARGUES;10210;10;AUBE;1
10279;;Pars-ls-Chavanges;PARS-LES-CHAVANGES;10330;10;AUBE;1
10280;;Pars-ls-Romilly;PARS-LES-ROMILLY;10100;10;AUBE;1
10281;Le;Pavillon-Sainte-Julie;PAVILLON-SAINTE-JULIE;10350;10;AUBE;1
10282;;Payns;PAYNS;10600;10;AUBE;1
10283;;Pel-et-Der;PEL-ET-DER;10500;10;AUBE;1
10284;;Prigny-la-Rose;PERIGNY-LA-ROSE;10400;10;AUBE;1
10285;;Perthes-ls-Brienne;PERTHES-LES-BRIENNE;10500;10;AUBE;1
10286;;Petit-Mesnil;PETIT-MESNIL;10500;10;AUBE;1
10287;;Piney;PINEY;10220;10;AUBE;1
10288;;Plaines-Saint-Lange;PLAINES-SAINT-LANGE;10250;10;AUBE;1
10289;;Plancy-l'Abbaye;PLANCY-L'ABBAYE;10380;10;AUBE;1
10290;;Planty;PLANTY;10160;10;AUBE;1
10291;;Plessis-Barbuise;PLESSIS-BARBUISE;10400;10;AUBE;1
10293;;Poivres;POIVRES;10700;10;AUBE;1
10294;;Poligny;POLIGNY;10110;10;AUBE;1
10295;;Polisot;POLISOT;10110;10;AUBE;1
10296;;Polisy;POLISY;10110;10;AUBE;1
10297;;Pont-Sainte-Marie;PONT-SAINTE-MARIE;10150;10;AUBE;1
10298;;Pont-sur-Seine;PONT-SUR-SEINE;10400;10;AUBE;1
10299;;Pouan-les-Valles;POUAN-LES-VALLEES;10700;10;AUBE;1
10300;;Pougy;POUGY;10240;10;AUBE;1
10301;;Pouy-sur-Vannes;POUY-SUR-VANNES;10290;10;AUBE;1
10302;;Praslin;PRASLIN;10210;10;AUBE;1
10303;;Prcy-Notre-Dame;PRECY-NOTRE-DAME;10500;10;AUBE;1
10304;;Prcy-Saint-Martin;PRECY-SAINT-MARTIN;10500;10;AUBE;1
10305;;Prmierfait;PREMIERFAIT;10170;10;AUBE;1
10306;;Proverville;PROVERVILLE;10200;10;AUBE;1
10307;;Prugny;PRUGNY;10190;10;AUBE;1
10308;;Prunay-Belleville;PRUNAY-BELLEVILLE;10350;10;AUBE;1
10309;;Prusy;PRUSY;10210;10;AUBE;1
10310;;Puits-et-Nuisement;PUITS-ET-NUISEMENT;10140;10;AUBE;1
10312;;Racines;RACINES;10130;10;AUBE;1
10313;;Radonvilliers;RADONVILLIERS;10500;10;AUBE;1
10314;;Ramerupt;RAMERUPT;10240;10;AUBE;1
10315;;Rances;RANCES;10500;10;AUBE;1
10316;;Rhges;RHEGES;10170;10;AUBE;1
10317;Les;Riceys;RICEYS;10340;10;AUBE;1
10318;;Rigny-la-Nonneuse;RIGNY-LA-NONNEUSE;10290;10;AUBE;1
10319;;Rigny-le-Ferron;RIGNY-LE-FERRON;10160;10;AUBE;1
10320;;Rilly-Sainte-Syre;RILLY-SAINTE-SYRE;10280;10;AUBE;1
10321;La;Rivire-de-Corps;RIVIERE-DE-CORPS;10440;10;AUBE;1
10323;;Romilly-sur-Seine;ROMILLY-SUR-SEINE;10100;10;AUBE;1
10324;;Roncenay;RONCENAY;10320;10;AUBE;1
10325;;Rosires-prs-Troyes;ROSIERES-PRES-TROYES;10430;10;AUBE;1
10326;;Rosnay-l'Hpital;ROSNAY-L'HOPITAL;10500;10;AUBE;1
10327;La;Rothire;ROTHIERE;10500;10;AUBE;1
10328;;Rouilly-Sacey;ROUILLY-SACEY;10220;10;AUBE;1
10329;;Rouilly-Saint-Loup;ROUILLY-SAINT-LOUP;10800;10;AUBE;1
10330;;Rouvres-les-Vignes;ROUVRES-LES-VIGNES;10200;10;AUBE;1
10331;;Rumilly-ls-Vaudes;RUMILLY-LES-VAUDES;10260;10;AUBE;1
10332;;Ruvigny;RUVIGNY;10410;10;AUBE;1
10333;;Saint-Andr-les-Vergers;SAINT-ANDRE-LES-VERGERS;10120;10;AUBE;1
10334;;Saint-Aubin;SAINT-AUBIN;10400;10;AUBE;1
10335;;Saint-Benoist-sur-Vanne;SAINT-BENOIST-SUR-VANNE;10160;10;AUBE;1
10336;;Saint-Benot-sur-Seine;SAINT-BENOIT-SUR-SEINE;10600;10;AUBE;1
10337;;Saint-Christophe-Dodinicourt;SAINT-CHRISTOPHE-DODINICOURT;10500;10;AUBE;1
10338;;Saint-tienne-sous-Barbuise;SAINT-ETIENNE-SOUS-BARBUISE;10700;10;AUBE;1
10339;;Saint-Flavy;SAINT-FLAVY;10350;10;AUBE;1
10340;;Saint-Germain;SAINT-GERMAIN;10120;10;AUBE;1
10341;;Saint-Hilaire-sous-Romilly;SAINT-HILAIRE-SOUS-ROMILLY;10100;10;AUBE;1
10342;;Saint-Jean-de-Bonneval;SAINT-JEAN-DE-BONNEVAL;10320;10;AUBE;1
10343;;Saint-Julien-les-Villas;SAINT-JULIEN-LES-VILLAS;10800;10;AUBE;1
10344;;Saint-Lger-prs-Troyes;SAINT-LEGER-PRES-TROYES;10800;10;AUBE;1
10345;;Saint-Lger-sous-Brienne;SAINT-LEGER-SOUS-BRIENNE;10500;10;AUBE;1
10346;;Saint-Lger-sous-Margerie;SAINT-LEGER-SOUS-MARGERIE;10330;10;AUBE;1
10347;;Saint-Loup-de-Buffigny;SAINT-LOUP-DE-BUFFIGNY;10100;10;AUBE;1
10348;;Saint-Lupien;SAINT-LUPIEN;10350;10;AUBE;1
10349;;Saint-Ly;SAINT-LYE;10600;10;AUBE;1
10350;;Saint-Mards-en-Othe;SAINT-MARDS-EN-OTHE;10160;10;AUBE;1
10351;;Saint-Martin-de-Bossenay;SAINT-MARTIN-DE-BOSSENAY;10100;10;AUBE;1
10352;;Sainte-Maure;SAINTE-MAURE;10150;10;AUBE;1
10353;;Saint-Mesmin;SAINT-MESMIN;10280;10;AUBE;1
10354;;Saint-Nabord-sur-Aube;SAINT-NABORD-SUR-AUBE;10700;10;AUBE;1
10355;;Saint-Nicolas-la-Chapelle;SAINT-NICOLAS-LA-CHAPELLE;10400;10;AUBE;1
10356;;Saint-Oulph;SAINT-OULPH;10170;10;AUBE;1
10357;;Saint-Parres-aux-Tertres;SAINT-PARRES-AUX-TERTRES;10410;10;AUBE;1
10358;;Saint-Parres-ls-Vaudes;SAINT-PARRES-LES-VAUDES;10260;10;AUBE;1
10359;;Saint-Phal;SAINT-PHAL;10130;10;AUBE;1
10360;;Saint-Pouange;SAINT-POUANGE;10120;10;AUBE;1
10361;;Saint-Remy-sous-Barbuise;SAINT-REMY-SOUS-BARBUISE;10700;10;AUBE;1
10362;;Sainte-Savine;SAINTE-SAVINE;10300;10;AUBE;1
10363;;Saint-Thibault;SAINT-THIBAULT;10800;10;AUBE;1
10364;;Saint-Usage;SAINT-USAGE;10360;10;AUBE;1
10365;;Salon;SALON;10700;10;AUBE;1
10366;;Saulcy;SAULCY;10200;10;AUBE;1
10367;La;Saulsotte;SAULSOTTE;10400;10;AUBE;1
10368;;Savires;SAVIERES;10600;10;AUBE;1
10369;;Semoine;SEMOINE;10700;10;AUBE;1
10370;;Soligny-les-tangs;SOLIGNY-LES-ETANGS;10400;10;AUBE;1
10371;;Sommeval;SOMMEVAL;10320;10;AUBE;1
10372;;Soulaines-Dhuys;SOULAINES-DHUYS;10200;10;AUBE;1
10373;;Souligny;SOULIGNY;10320;10;AUBE;1
10374;;Spoy;SPOY;10200;10;AUBE;1
10375;;Thennelires;THENNELIERES;10410;10;AUBE;1
10376;;Thieffrain;THIEFFRAIN;10140;10;AUBE;1
10377;;Thil;THIL;10200;10;AUBE;1
10378;;Thors;THORS;10200;10;AUBE;1
10379;;Torcy-le-Grand;TORCY-LE-GRAND;10700;10;AUBE;1
10380;;Torcy-le-Petit;TORCY-LE-PETIT;10700;10;AUBE;1
10381;;Torvilliers;TORVILLIERS;10440;10;AUBE;1
10382;;Tranel;TRAINEL;10400;10;AUBE;1
10383;;Trancault;TRANCAULT;10290;10;AUBE;1
10384;;Trannes;TRANNES;10140;10;AUBE;1
10386;;Trouans;TROUANS;10700;10;AUBE;1
10387;;Troyes;TROYES;10000;10;AUBE;1
10388;;Turgy;TURGY;10210;10;AUBE;1
10389;;Unienville;UNIENVILLE;10140;10;AUBE;1
10390;;Urville;URVILLE;10200;10;AUBE;1
10391;;Vailly;VAILLY;10150;10;AUBE;1
10392;;Vallant-Saint-Georges;VALLANT-SAINT-GEORGES;10170;10;AUBE;1
10393;;Vallentigny;VALLENTIGNY;10500;10;AUBE;1
10394;;Vallires;VALLIERES;10210;10;AUBE;1
10395;;Vanlay;VANLAY;10210;10;AUBE;1
10396;;Vauchassis;VAUCHASSIS;10190;10;AUBE;1
10397;;Vauchonvilliers;VAUCHONVILLIERS;10140;10;AUBE;1
10398;;Vaucogne;VAUCOGNE;10240;10;AUBE;1
10399;;Vaudes;VAUDES;10260;10;AUBE;1
10400;;Vaupoisson;VAUPOISSON;10700;10;AUBE;1
10401;;Vendeuvre-sur-Barse;VENDEUVRE-SUR-BARSE;10140;10;AUBE;1
10402;La;Vendue-Mignot;VENDUE-MIGNOT;10800;10;AUBE;1
10403;;Vernonvilliers;VERNONVILLIERS;10200;10;AUBE;1
10404;;Verpillires-sur-Ource;VERPILLIERES-SUR-OURCE;10360;10;AUBE;1
10405;;Verricourt;VERRICOURT;10240;10;AUBE;1
10406;;Verrires;VERRIERES;10390;10;AUBE;1
10408;;Vipres-le-Petit;VIAPRES-LE-PETIT;10380;10;AUBE;1
10409;;Villacerf;VILLACERF;10600;10;AUBE;1
10410;;Villadin;VILLADIN;10290;10;AUBE;1
10411;La;Ville-aux-Bois;VILLE-AUX-BOIS;10500;10;AUBE;1
10412;;Villechtif;VILLECHETIF;10410;10;AUBE;1
10414;;Villeloup;VILLELOUP;10350;10;AUBE;1
10415;;Villemaur-sur-Vanne;VILLEMAUR-SUR-VANNE;10190;10;AUBE;1
10416;;Villemereuil;VILLEMEREUIL;10800;10;AUBE;1
10417;;Villemoiron-en-Othe;VILLEMOIRON-EN-OTHE;10160;10;AUBE;1
10418;;Villemorien;VILLEMORIEN;10110;10;AUBE;1
10419;;Villemoyenne;VILLEMOYENNE;10260;10;AUBE;1
10420;;Villenauxe-la-Grande;VILLENAUXE-LA-GRANDE;10370;10;AUBE;1
10421;La;Villeneuve-au-Chtelot;VILLENEUVE-AU-CHATELOT;10400;10;AUBE;1
10422;;Villeneuve-au-Chemin;VILLENEUVE-AU-CHEMIN;10130;10;AUBE;1
10423;La;Villeneuve-au-Chne;VILLENEUVE-AU-CHENE;10140;10;AUBE;1
10424;;Villeret;VILLERET;10330;10;AUBE;1
10425;;Villery;VILLERY;10320;10;AUBE;1
10426;;Ville-sous-la-Fert;VILLE-SOUS-LA-FERTE;10310;10;AUBE;1
10427;;Ville-sur-Arce;VILLE-SUR-ARCE;10110;10;AUBE;1
10428;;Ville-sur-Terre;VILLE-SUR-TERRE;10200;10;AUBE;1
10429;;Villette-sur-Aube;VILLETTE-SUR-AUBE;10700;10;AUBE;1
10430;;Villiers-Herbisse;VILLIERS-HERBISSE;10700;10;AUBE;1
10431;;Villiers-le-Bois;VILLIERS-LE-BOIS;10210;10;AUBE;1
10432;;Villiers-sous-Praslin;VILLIERS-SOUS-PRASLIN;10210;10;AUBE;1
10433;;Villy-en-Trodes;VILLY-EN-TRODES;10140;10;AUBE;1
10434;;Villy-le-Bois;VILLY-LE-BOIS;10800;10;AUBE;1
10435;;Villy-le-Marchal;VILLY-LE-MARECHAL;10800;10;AUBE;1
10436;;Vinets;VINETS;10700;10;AUBE;1
10437;;Virey-sous-Bar;VIREY-SOUS-BAR;10260;10;AUBE;1
10438;;Vitry-le-Crois;VITRY-LE-CROISE;10110;10;AUBE;1
10439;;Viviers-sur-Artaut;VIVIERS-SUR-ARTAUT;10110;10;AUBE;1
10440;;Voigny;VOIGNY;10200;10;AUBE;1
10441;;Vosnon;VOSNON;10130;10;AUBE;1
10442;;Vou;VOUE;10150;10;AUBE;1
10443;;Vougrey;VOUGREY;10210;10;AUBE;1
10444;;Vulaines;VULAINES;10160;10;AUBE;1
10445;;Yvres-le-Petit;YEVRES-LE-PETIT;10500;10;AUBE;1
11001;;Aigues-Vives;AIGUES-VIVES;11800;11;AUDE;1
11002;;Airoux;AIROUX;11320;11;AUDE;1
11003;;Ajac;AJAC;11300;11;AUDE;1
11004;;Alaigne;ALAIGNE;11240;11;AUDE;1
11005;;Alairac;ALAIRAC;11290;11;AUDE;1
11006;;Albas;ALBAS;11360;11;AUDE;1
11007;;Albires;ALBIERES;11330;11;AUDE;1
11008;;Alet-les-Bains;ALET-LES-BAINS;11580;11;AUDE;1
11009;;Alzonne;ALZONNE;11170;11;AUDE;1
11010;;Antugnac;ANTUGNAC;11190;11;AUDE;1
11011;;Aragon;ARAGON;11600;11;AUDE;1
11012;;Argeliers;ARGELIERS;11120;11;AUDE;1
11013;;Argens-Minervois;ARGENS-MINERVOIS;11200;11;AUDE;1
11014;;Armissan;ARMISSAN;11110;11;AUDE;1
11015;;Arques;ARQUES;11190;11;AUDE;1
11016;;Arquettes-en-Val;ARQUETTES-EN-VAL;11220;11;AUDE;1
11017;;Artigues;ARTIGUES;11140;11;AUDE;1
11018;;Arzens;ARZENS;11290;11;AUDE;1
11019;;Aunat;AUNAT;11140;11;AUDE;1
11020;;Auriac;AURIAC;11330;11;AUDE;1
11021;;Axat;AXAT;11140;11;AUDE;1
11022;;Azille;AZILLE;11700;11;AUDE;1
11023;;Badens;BADENS;11800;11;AUDE;1
11024;;Bages;BAGES;11100;11;AUDE;1
11025;;Bagnoles;BAGNOLES;11600;11;AUDE;1
11026;;Baraigne;BARAIGNE;11410;11;AUDE;1
11027;;Barbaira;BARBAIRA;11800;11;AUDE;1
11028;;Belcaire;BELCAIRE;11340;11;AUDE;1
11029;;Belcastel-et-Buc;BELCASTEL-ET-BUC;11580;11;AUDE;1
11030;;Belflou;BELFLOU;11410;11;AUDE;1
11031;;Belfort-sur-Rebenty;BELFORT-SUR-REBENTY;11140;11;AUDE;1
11032;;Bellegarde-du-Razs;BELLEGARDE-DU-RAZES;11240;11;AUDE;1
11033;;Belpech;BELPECH;11420;11;AUDE;1
11034;;Belvze-du-Razs;BELVEZE-DU-RAZES;11240;11;AUDE;1
11035;;Belvianes-et-Cavirac;BELVIANES-ET-CAVIRAC;11500;11;AUDE;1
11036;;Belvis;BELVIS;11340;11;AUDE;1
11037;;Berriac;BERRIAC;11090;11;AUDE;1
11038;;Bessde-de-Sault;BESSEDE-DE-SAULT;11140;11;AUDE;1
11039;La;Bezole;BEZOLE;11300;11;AUDE;1
11040;;Bizanet;BIZANET;11200;11;AUDE;1
11041;;Bize-Minervois;BIZE-MINERVOIS;11120;11;AUDE;1
11042;;Blomac;BLOMAC;11700;11;AUDE;1
11043;;Bouilhonnac;BOUILHONNAC;11800;11;AUDE;1
11044;;Bouisse;BOUISSE;11190;11;AUDE;1
11045;;Bourige;BOURIEGE;11300;11;AUDE;1
11046;;Bourigeole;BOURIGEOLE;11300;11;AUDE;1
11047;Le;Bousquet;BOUSQUET;11140;11;AUDE;1
11048;;Boutenac;BOUTENAC;11200;11;AUDE;1
11049;;Bram;BRAM;11150;11;AUDE;1
11050;;Brenac;BRENAC;11500;11;AUDE;1
11051;;Brzilhac;BREZILHAC;11270;11;AUDE;1
11052;;Brousses-et-Villaret;BROUSSES-ET-VILLARET;11390;11;AUDE;1
11053;;Brugairolles;BRUGAIROLLES;11300;11;AUDE;1
11054;Les;Brunels;BRUNELS;11400;11;AUDE;1
11055;;Bugarach;BUGARACH;11190;11;AUDE;1
11056;;Cabrespine;CABRESPINE;11160;11;AUDE;1
11057;;Cahuzac;CAHUZAC;11420;11;AUDE;1
11058;;Cailhau;CAILHAU;11240;11;AUDE;1
11059;;Cailhavel;CAILHAVEL;11240;11;AUDE;1
11060;;Cailla;CAILLA;11140;11;AUDE;1
11061;;Cambieure;CAMBIEURE;11240;11;AUDE;1
11062;;Campagna-de-Sault;CAMPAGNA-DE-SAULT;11140;11;AUDE;1
11063;;Campagne-sur-Aude;CAMPAGNE-SUR-AUDE;11260;11;AUDE;1
11064;;Camplong-d'Aude;CAMPLONG-D'AUDE;11200;11;AUDE;1
11065;;Camps-sur-l'Agly;CAMPS-SUR-L'AGLY;11190;11;AUDE;1
11066;;Camurac;CAMURAC;11340;11;AUDE;1
11067;;Canet;CANET;11200;11;AUDE;1
11068;;Capendu;CAPENDU;11700;11;AUDE;1
11069;;Carcassonne;CARCASSONNE;11000;11;AUDE;1
11070;;Carlipa;CARLIPA;11170;11;AUDE;1
11071;;Cascastel-des-Corbires;CASCASTEL-DES-CORBIERES;11360;11;AUDE;1
11072;La;Cassaigne;CASSAIGNE;11270;11;AUDE;1
11073;;Cassaignes;CASSAIGNES;11190;11;AUDE;1
11074;Les;Casss;CASSES;11320;11;AUDE;1
11075;;Castans;CASTANS;11160;11;AUDE;1
11076;;Castelnaudary;CASTELNAUDARY;11400;11;AUDE;1
11077;;Castelnau-d'Aude;CASTELNAU-D'AUDE;11700;11;AUDE;1
11078;;Castelreng;CASTELRENG;11300;11;AUDE;1
11079;;Caudebronde;CAUDEBRONDE;11390;11;AUDE;1
11080;;Caudeval;CAUDEVAL;11230;11;AUDE;1
11081;;Caunes-Minervois;CAUNES-MINERVOIS;11160;11;AUDE;1
11082;;Caunette-sur-Lauquet;CAUNETTE-SUR-LAUQUET;11250;11;AUDE;1
11083;;Caunettes-en-Val;CAUNETTES-EN-VAL;11220;11;AUDE;1
11084;;Caux-et-Sauzens;CAUX-ET-SAUZENS;11170;11;AUDE;1
11085;;Cavanac;CAVANAC;11570;11;AUDE;1
11086;;Caves;CAVES;11510;11;AUDE;1
11087;;Cazalrenoux;CAZALRENOUX;11270;11;AUDE;1
11088;;Cazilhac;CAZILHAC;11570;11;AUDE;1
11089;;Cenne-Monestis;CENNE-MONESTIES;11170;11;AUDE;1
11090;;Cpie;CEPIE;11300;11;AUDE;1
11091;;Chalabre;CHALABRE;11230;11;AUDE;1
11092;;Citou;CITOU;11160;11;AUDE;1
11093;Le;Clat;CLAT;11140;11;AUDE;1
11094;;Clermont-sur-Lauquet;CLERMONT-SUR-LAUQUET;11250;11;AUDE;1
11095;;Comigne;COMIGNE;11700;11;AUDE;1
11096;;Comus;COMUS;11340;11;AUDE;1
11097;;Conilhac-de-la-Montagne;CONILHAC-DE-LA-MONTAGNE;11190;11;AUDE;1
11098;;Conilhac-Corbires;CONILHAC-CORBIERES;11200;11;AUDE;1
11099;;Conques-sur-Orbiel;CONQUES-SUR-ORBIEL;11600;11;AUDE;1
11100;;Corbires;CORBIERES;11230;11;AUDE;1
11101;;Coudons;COUDONS;11500;11;AUDE;1
11102;;Couffoulens;COUFFOULENS;11250;11;AUDE;1
11103;;Couiza;COUIZA;11190;11;AUDE;1
11104;;Counozouls;COUNOZOULS;11140;11;AUDE;1
11105;;Cournanel;COURNANEL;11300;11;AUDE;1
11106;;Coursan;COURSAN;11110;11;AUDE;1
11107;;Courtauly;COURTAULY;11230;11;AUDE;1
11108;La;Courtte;COURTETE;11240;11;AUDE;1
11109;;Coustaussa;COUSTAUSSA;11190;11;AUDE;1
11110;;Coustouge;COUSTOUGE;11220;11;AUDE;1
11111;;Cruscades;CRUSCADES;11200;11;AUDE;1
11112;;Cubires-sur-Cinoble;CUBIERES-SUR-CINOBLE;11190;11;AUDE;1
11113;;Cucugnan;CUCUGNAN;11350;11;AUDE;1
11114;;Cumis;CUMIES;11410;11;AUDE;1
11115;;Cuxac-Cabards;CUXAC-CABARDES;11390;11;AUDE;1
11116;;Cuxac-d'Aude;CUXAC-D'AUDE;11590;11;AUDE;1
11117;;Davejean;DAVEJEAN;11330;11;AUDE;1
11118;;Dernacueillette;DERNACUEILLETTE;11330;11;AUDE;1
11119;La;Digne-d'Amont;DIGNE-D'AMONT;11300;11;AUDE;1
11120;La;Digne-d'Aval;DIGNE-D'AVAL;11300;11;AUDE;1
11121;;Donazac;DONAZAC;11240;11;AUDE;1
11122;;Douzens;DOUZENS;11700;11;AUDE;1
11123;;Duilhac-sous-Peyrepertuse;DUILHAC-SOUS-PEYREPERTUSE;11350;11;AUDE;1
11124;;Durban-Corbires;DURBAN-CORBIERES;11360;11;AUDE;1
11125;;Embres-et-Castelmaure;EMBRES-ET-CASTELMAURE;11360;11;AUDE;1
11126;;Escales;ESCALES;11200;11;AUDE;1
11127;;Escouloubre;ESCOULOUBRE;11140;11;AUDE;1
11128;;Escueillens-et-Saint-Just-de-Blengard;ESCUEILLENS-ET-SAINT-JUST-DE-BELENGARD;11240;11;AUDE;1
11129;;Espraza;ESPERAZA;11260;11;AUDE;1
11130;;Espezel;ESPEZEL;11340;11;AUDE;1
11131;;Fa;FA;11260;11;AUDE;1
11132;;Fabrezan;FABREZAN;11200;11;AUDE;1
11133;;Fajac-en-Val;FAJAC-EN-VAL;11220;11;AUDE;1
11134;;Fajac-la-Relenque;FAJAC-LA-RELENQUE;11410;11;AUDE;1
11135;La;Fajolle;FAJOLLE;11140;11;AUDE;1
11136;;Fanjeaux;FANJEAUX;11270;11;AUDE;1
11137;;Flines-Termens;FELINES-TERMENES;11330;11;AUDE;1
11138;;Fendeille;FENDEILLE;11400;11;AUDE;1
11139;;Fenouillet-du-Razs;FENOUILLET-DU-RAZES;11240;11;AUDE;1
11140;;Ferrals-les-Corbires;FERRALS-LES-CORBIERES;11200;11;AUDE;1
11141;;Ferran;FERRAN;11240;11;AUDE;1
11142;;Festes-et-Saint-Andr;FESTES-ET-SAINT-ANDRE;11300;11;AUDE;1
11143;;Feuilla;FEUILLA;11510;11;AUDE;1
11144;;Fitou;FITOU;11510;11;AUDE;1
11145;;Fleury;FLEURY;11560;11;AUDE;1
11146;;Floure;FLOURE;11800;11;AUDE;1
11147;;Fontans-de-Sault;FONTANES-DE-SAULT;11140;11;AUDE;1
11148;;Fontcouverte;FONTCOUVERTE;11700;11;AUDE;1
11149;;Fonters-du-Razs;FONTERS-DU-RAZES;11400;11;AUDE;1
11150;;Fontiers-Cabards;FONTIERS-CABARDES;11310;11;AUDE;1
11151;;Fontis-d'Aude;FONTIES-D'AUDE;11800;11;AUDE;1
11152;;Fontjoncouse;FONTJONCOUSE;11360;11;AUDE;1
11153;La;Force;FORCE;11270;11;AUDE;1
11154;;Fournes-Cabards;FOURNES-CABARDES;11600;11;AUDE;1
11155;;Fourtou;FOURTOU;11190;11;AUDE;1
11156;;Fraisse-Cabards;FRAISSE-CABARDES;11600;11;AUDE;1
11157;;Fraiss-des-Corbires;FRAISSE-DES-CORBIERES;11360;11;AUDE;1
11158;;Gaja-et-Villedieu;GAJA-ET-VILLEDIEU;11300;11;AUDE;1
11159;;Gaja-la-Selve;GAJA-LA-SELVE;11270;11;AUDE;1
11160;;Galinagues;GALINAGUES;11140;11;AUDE;1
11161;;Gardie;GARDIE;11250;11;AUDE;1
11162;;Generville;GENERVILLE;11270;11;AUDE;1
11163;;Gincla;GINCLA;11140;11;AUDE;1
11164;;Ginestas;GINESTAS;11120;11;AUDE;1
11165;;Ginoles;GINOLES;11500;11;AUDE;1
11166;;Gourvieille;GOURVIEILLE;11410;11;AUDE;1
11167;;Gramazie;GRAMAZIE;11240;11;AUDE;1
11168;;Grans;GRANES;11500;11;AUDE;1
11169;;Greffeil;GREFFEIL;11250;11;AUDE;1
11170;;Gruissan;GRUISSAN;11430;11;AUDE;1
11171;;Gueytes-et-Labastide;GUEYTES-ET-LABASTIDE;11230;11;AUDE;1
11172;;Homps;HOMPS;11200;11;AUDE;1
11173;;Hounoux;HOUNOUX;11240;11;AUDE;1
11174;Les;Ilhes;ILHES;11380;11;AUDE;1
11175;;Issel;ISSEL;11400;11;AUDE;1
11176;;Jonquires;JONQUIERES;11220;11;AUDE;1
11177;;Joucou;JOUCOU;11140;11;AUDE;1
11178;;Labastide-d'Anjou;LABASTIDE-D'ANJOU;11320;11;AUDE;1
11179;;Labastide-en-Val;LABASTIDE-EN-VAL;11220;11;AUDE;1
11180;;Labastide-Esparbairenque;LABASTIDE-ESPARBAIRENQUE;11380;11;AUDE;1
11181;;Labcde-Lauragais;LABECEDE-LAURAGAIS;11400;11;AUDE;1
11182;;Lacombe;LACOMBE;11310;11;AUDE;1
11183;;Ladern-sur-Lauquet;LADERN-SUR-LAUQUET;11250;11;AUDE;1
11184;;Lafage;LAFAGE;11420;11;AUDE;1
11185;;Lagrasse;LAGRASSE;11220;11;AUDE;1
11186;;Lairire;LAIRIERE;11330;11;AUDE;1
11187;;Lanet;LANET;11330;11;AUDE;1
11188;La;Palme;PALME;11480;11;AUDE;1
11189;;Laprade;LAPRADE;11390;11;AUDE;1
11190;La;Redorte;REDORTE;11700;11;AUDE;1
11191;;Laroque-de-Fa;LAROQUE-DE-FA;11330;11;AUDE;1
11192;;Lasbordes;LASBORDES;11400;11;AUDE;1
11193;;Lasserre-de-Prouille;LASSERRE-DE-PROUILLE;11270;11;AUDE;1
11194;;Lastours;LASTOURS;11600;11;AUDE;1
11195;;Laurabuc;LAURABUC;11400;11;AUDE;1
11196;;Laurac;LAURAC;11270;11;AUDE;1
11197;;Lauraguel;LAURAGUEL;11300;11;AUDE;1
11198;;Laure-Minervois;LAURE-MINERVOIS;11800;11;AUDE;1
11199;;Lavalette;LAVALETTE;11290;11;AUDE;1
11200;;Lespinassire;LESPINASSIERE;11160;11;AUDE;1
11201;;Leuc;LEUC;11250;11;AUDE;1
11202;;Leucate;LEUCATE;11370;11;AUDE;1
11203;;Lzignan-Corbires;LEZIGNAN-CORBIERES;11200;11;AUDE;1
11204;;Lignairolles;LIGNAIROLLES;11240;11;AUDE;1
11205;;Limousis;LIMOUSIS;11600;11;AUDE;1
11206;;Limoux;LIMOUX;11300;11;AUDE;1
11207;;Loupia;LOUPIA;11300;11;AUDE;1
11208;La;Louvire-Lauragais;LOUVIERE-LAURAGAIS;11410;11;AUDE;1
11209;;Luc-sur-Aude;LUC-SUR-AUDE;11190;11;AUDE;1
11210;;Luc-sur-Orbieu;LUC-SUR-ORBIEU;11200;11;AUDE;1
11211;;Magrie;MAGRIE;11300;11;AUDE;1
11212;;Mailhac;MAILHAC;11120;11;AUDE;1
11213;;Maisons;MAISONS;11330;11;AUDE;1
11214;;Malras;MALRAS;11300;11;AUDE;1
11215;;Malves-en-Minervois;MALVES-EN-MINERVOIS;11600;11;AUDE;1
11216;;Malvis;MALVIES;11300;11;AUDE;1
11217;;Marcorignan;MARCORIGNAN;11120;11;AUDE;1
11218;;Marquein;MARQUEIN;11410;11;AUDE;1
11219;;Marsa;MARSA;11140;11;AUDE;1
11220;;Marseillette;MARSEILLETTE;11800;11;AUDE;1
11221;Les;Martys;MARTYS;11390;11;AUDE;1
11222;;Mas-Cabards;MAS-CABARDES;11380;11;AUDE;1
11223;;Mas-des-Cours;MAS-DES-COURS;11570;11;AUDE;1
11224;;Massac;MASSAC;11330;11;AUDE;1
11225;;Mas-Saintes-Puelles;MAS-SAINTES-PUELLES;11400;11;AUDE;1
11226;;Mayreville;MAYREVILLE;11420;11;AUDE;1
11227;;Mayronnes;MAYRONNES;11220;11;AUDE;1
11228;;Mazerolles-du-Razs;MAZEROLLES-DU-RAZES;11240;11;AUDE;1
11229;;Mazuby;MAZUBY;11140;11;AUDE;1
11230;;Mrial;MERIAL;11140;11;AUDE;1
11231;;Mzerville;MEZERVILLE;11410;11;AUDE;1
11232;;Miraval-Cabardes;MIRAVAL-CABARDES;11380;11;AUDE;1
11233;;Mirepeisset;MIREPEISSET;11120;11;AUDE;1
11234;;Mireval-Lauragais;MIREVAL-LAURAGAIS;11400;11;AUDE;1
11235;;Missgre;MISSEGRE;11580;11;AUDE;1
11236;;Molandier;MOLANDIER;11420;11;AUDE;1
11238;;Molleville;MOLLEVILLE;11410;11;AUDE;1
11239;;Montauriol;MONTAURIOL;11410;11;AUDE;1
11240;;Montazels;MONTAZELS;11190;11;AUDE;1
11241;;Montbrun-des-Corbires;MONTBRUN-DES-CORBIERES;11700;11;AUDE;1
11242;;Montclar;MONTCLAR;11250;11;AUDE;1
11243;;Montferrand;MONTFERRAND;11320;11;AUDE;1
11244;;Montfort-sur-Boulzane;MONTFORT-SUR-BOULZANE;11140;11;AUDE;1
11245;;Montgaillard;MONTGAILLARD;11330;11;AUDE;1
11246;;Montgradail;MONTGRADAIL;11240;11;AUDE;1
11247;;Monthaut;MONTHAUT;11240;11;AUDE;1
11248;;Montirat;MONTIRAT;11800;11;AUDE;1
11249;;Montjardin;MONTJARDIN;11230;11;AUDE;1
11250;;Montjoi;MONTJOI;11330;11;AUDE;1
11251;;Montlaur;MONTLAUR;11220;11;AUDE;1
11252;;Montmaur;MONTMAUR;11320;11;AUDE;1
11253;;Montolieu;MONTOLIEU;11170;11;AUDE;1
11254;;Montral;MONTREAL;11290;11;AUDE;1
11255;;Montredon-des-Corbires;MONTREDON-DES-CORBIERES;11100;11;AUDE;1
11256;;Montsret;MONTSERET;11200;11;AUDE;1
11257;;Monze;MONZE;11800;11;AUDE;1
11258;;Moussan;MOUSSAN;11120;11;AUDE;1
11259;;Moussoulens;MOUSSOULENS;11170;11;AUDE;1
11260;;Mouthoumet;MOUTHOUMET;11330;11;AUDE;1
11261;;Moux;MOUX;11700;11;AUDE;1
11262;;Narbonne;NARBONNE;11100;11;AUDE;1
11263;;Nbias;NEBIAS;11500;11;AUDE;1
11264;;Nvian;NEVIAN;11200;11;AUDE;1
11265;;Niort-de-Sault;NIORT-DE-SAULT;11140;11;AUDE;1
11266;;Port-la-Nouvelle;PORT-LA-NOUVELLE;11210;11;AUDE;1
11267;;Ornaisons;ORNAISONS;11200;11;AUDE;1
11268;;Orsans;ORSANS;11270;11;AUDE;1
11269;;Ouveillan;OUVEILLAN;11590;11;AUDE;1
11270;;Padern;PADERN;11350;11;AUDE;1
11271;;Palairac;PALAIRAC;11330;11;AUDE;1
11272;;Palaja;PALAJA;11570;11;AUDE;1
11273;;Paraza;PARAZA;11200;11;AUDE;1
11274;;Pauligne;PAULIGNE;11300;11;AUDE;1
11275;;Payra-sur-l'Hers;PAYRA-SUR-L'HERS;11410;11;AUDE;1
11276;;Paziols;PAZIOLS;11350;11;AUDE;1
11277;;Pcharic-et-le-Py;PECHARIC-ET-LE-PY;11420;11;AUDE;1
11278;;Pech-Luna;PECH-LUNA;11420;11;AUDE;1
11279;;Pennautier;PENNAUTIER;11610;11;AUDE;1
11280;;Ppieux;PEPIEUX;11700;11;AUDE;1
11281;;Pexiora;PEXIORA;11150;11;AUDE;1
11282;;Peyrefitte-du-Razs;PEYREFITTE-DU-RAZES;11230;11;AUDE;1
11283;;Peyrefitte-sur-l'Hers;PEYREFITTE-SUR-L'HERS;11410;11;AUDE;1
11284;;Peyrens;PEYRENS;11400;11;AUDE;1
11285;;Peyriac-de-Mer;PEYRIAC-DE-MER;11440;11;AUDE;1
11286;;Peyriac-Minervois;PEYRIAC-MINERVOIS;11160;11;AUDE;1
11287;;Peyrolles;PEYROLLES;11190;11;AUDE;1
11288;;Pezens;PEZENS;11170;11;AUDE;1
11289;;Pieusse;PIEUSSE;11300;11;AUDE;1
11290;;Plaigne;PLAIGNE;11420;11;AUDE;1
11291;;Plavilla;PLAVILLA;11270;11;AUDE;1
11292;La;Pomarde;POMAREDE;11400;11;AUDE;1
11293;;Pomas;POMAS;11250;11;AUDE;1
11294;;Pomy;POMY;11300;11;AUDE;1
11295;;Portel-des-Corbires;PORTEL-DES-CORBIERES;11490;11;AUDE;1
11296;;Pouzols-Minervois;POUZOLS-MINERVOIS;11120;11;AUDE;1
11297;;Pradelles-Cabards;PRADELLES-CABARDES;11380;11;AUDE;1
11298;;Pradelles-en-Val;PRADELLES-EN-VAL;11220;11;AUDE;1
11299;;Preixan;PREIXAN;11250;11;AUDE;1
11300;;Puginier;PUGINIER;11400;11;AUDE;1
11301;;Puichric;PUICHERIC;11700;11;AUDE;1
11302;;Puilaurens;PUILAURENS;11140;11;AUDE;1
11303;;Puivert;PUIVERT;11230;11;AUDE;1
11304;;Quillan;QUILLAN;11500;11;AUDE;1
11305;;Quintillan;QUINTILLAN;11360;11;AUDE;1
11306;;Quirbajou;QUIRBAJOU;11500;11;AUDE;1
11307;;Raissac-d'Aude;RAISSAC-D'AUDE;11200;11;AUDE;1
11308;;Raissac-sur-Lampy;RAISSAC-SUR-LAMPY;11170;11;AUDE;1
11309;;Rennes-le-Chteau;RENNES-LE-CHATEAU;11190;11;AUDE;1
11310;;Rennes-les-Bains;RENNES-LES-BAINS;11190;11;AUDE;1
11311;;Ribaute;RIBAUTE;11220;11;AUDE;1
11312;;Ribouisse;RIBOUISSE;11270;11;AUDE;1
11313;;Ricaud;RICAUD;11400;11;AUDE;1
11314;;Rieux-en-Val;RIEUX-EN-VAL;11220;11;AUDE;1
11315;;Rieux-Minervois;RIEUX-MINERVOIS;11160;11;AUDE;1
11316;;Rivel;RIVEL;11230;11;AUDE;1
11317;;Rodome;RODOME;11140;11;AUDE;1
11318;;Roquecourbe-Minervois;ROQUECOURBE-MINERVOIS;11700;11;AUDE;1
11319;;Roquefre;ROQUEFERE;11380;11;AUDE;1
11320;;Roquefeuil;ROQUEFEUIL;11340;11;AUDE;1
11321;;Roquefort-de-Sault;ROQUEFORT-DE-SAULT;11140;11;AUDE;1
11322;;Roquefort-des-Corbires;ROQUEFORT-DES-CORBIERES;11540;11;AUDE;1
11323;;Roquetaillade;ROQUETAILLADE;11300;11;AUDE;1
11324;;Roubia;ROUBIA;11200;11;AUDE;1
11325;;Rouffiac-d'Aude;ROUFFIAC-D'AUDE;11250;11;AUDE;1
11326;;Rouffiac-des-Corbires;ROUFFIAC-DES-CORBIERES;11350;11;AUDE;1
11327;;Roullens;ROULLENS;11290;11;AUDE;1
11328;;Routier;ROUTIER;11240;11;AUDE;1
11329;;Rouvenac;ROUVENAC;11260;11;AUDE;1
11330;;Rustiques;RUSTIQUES;11800;11;AUDE;1
11331;;Saint-Amans;SAINT-AMANS;11270;11;AUDE;1
11332;;Saint-Andr-de-Roquelongue;SAINT-ANDRE-DE-ROQUELONGUE;11200;11;AUDE;1
11333;;Saint-Benot;SAINT-BENOIT;11230;11;AUDE;1
11334;;Sainte-Camelle;SAINTE-CAMELLE;11410;11;AUDE;1
11335;;Sainte-Colombe-sur-Guette;SAINTE-COLOMBE-SUR-GUETTE;11140;11;AUDE;1
11336;;Sainte-Colombe-sur-l'Hers;SAINTE-COLOMBE-SUR-L'HERS;11230;11;AUDE;1
11337;;Saint-Couat-d'Aude;SAINT-COUAT-D'AUDE;11700;11;AUDE;1
11338;;Saint-Couat-du-Razs;SAINT-COUAT-DU-RAZES;11300;11;AUDE;1
11339;;Saint-Denis;SAINT-DENIS;11310;11;AUDE;1
11340;;Sainte-Eulalie;SAINTE-EULALIE;11170;11;AUDE;1
11341;;Saint-Ferriol;SAINT-FERRIOL;11500;11;AUDE;1
11342;;Saint-Frichoux;SAINT-FRICHOUX;11800;11;AUDE;1
11343;;Saint-Gaudric;SAINT-GAUDERIC;11270;11;AUDE;1
11344;;Saint-Hilaire;SAINT-HILAIRE;11250;11;AUDE;1
11345;;Saint-Jean-de-Barrou;SAINT-JEAN-DE-BARROU;11360;11;AUDE;1
11346;;Saint-Jean-de-Paracol;SAINT-JEAN-DE-PARACOL;11260;11;AUDE;1
11347;;Saint-Julia-de-Bec;SAINT-JULIA-DE-BEC;11500;11;AUDE;1
11348;;Saint-Julien-de-Briola;SAINT-JULIEN-DE-BRIOLA;11270;11;AUDE;1
11350;;Saint-Just-et-le-Bzu;SAINT-JUST-ET-LE-BEZU;11500;11;AUDE;1
11351;;Saint-Laurent-de-la-Cabrerisse;SAINT-LAURENT-DE-LA-CABRERISSE;11220;11;AUDE;1
11352;;Saint-Louis-et-Parahou;SAINT-LOUIS-ET-PARAHOU;11500;11;AUDE;1
11353;;Saint-Marcel-sur-Aude;SAINT-MARCEL-SUR-AUDE;11120;11;AUDE;1
11354;;Saint-Martin-des-Puits;SAINT-MARTIN-DES-PUITS;11220;11;AUDE;1
11355;;Saint-Martin-de-Villereglan;SAINT-MARTIN-DE-VILLEREGLAN;11300;11;AUDE;1
11356;;Saint-Martin-Lalande;SAINT-MARTIN-LALANDE;11400;11;AUDE;1
11357;;Saint-Martin-le-Vieil;SAINT-MARTIN-LE-VIEIL;11170;11;AUDE;1
11358;;Saint-Martin-Lys;SAINT-MARTIN-LYS;11500;11;AUDE;1
11359;;Saint-Michel-de-Lans;SAINT-MICHEL-DE-LANES;11410;11;AUDE;1
11360;;Saint-Nazaire-d'Aude;SAINT-NAZAIRE-D'AUDE;11120;11;AUDE;1
11361;;Saint-Papoul;SAINT-PAPOUL;11400;11;AUDE;1
11362;;Saint-Paulet;SAINT-PAULET;11320;11;AUDE;1
11363;;Saint-Pierre-des-Champs;SAINT-PIERRE-DES-CHAMPS;11220;11;AUDE;1
11364;;Saint-Polycarpe;SAINT-POLYCARPE;11300;11;AUDE;1
11365;;Saint-Sernin;SAINT-SERNIN;11420;11;AUDE;1
11366;;Sainte-Valire;SAINTE-VALIERE;11120;11;AUDE;1
11367;;Saissac;SAISSAC;11310;11;AUDE;1
11368;;Sallles-Cabards;SALLELES-CABARDES;11600;11;AUDE;1
11369;;Sallles-d'Aude;SALLELES-D'AUDE;11590;11;AUDE;1
11370;;Salles-d'Aude;SALLES-D'AUDE;11110;11;AUDE;1
11371;;Salles-sur-l'Hers;SALLES-SUR-L'HERS;11410;11;AUDE;1
11372;;Salsigne;SALSIGNE;11600;11;AUDE;1
11373;;Salvezines;SALVEZINES;11140;11;AUDE;1
11374;;Salza;SALZA;11330;11;AUDE;1
11375;;Seignalens;SEIGNALENS;11240;11;AUDE;1
11376;La;Serpent;SERPENT;11190;11;AUDE;1
11377;;Serres;SERRES;11190;11;AUDE;1
11378;;Servis-en-Val;SERVIES-EN-VAL;11220;11;AUDE;1
11379;;Sigean;SIGEAN;11130;11;AUDE;1
11380;;Sonnac-sur-l'Hers;SONNAC-SUR-L'HERS;11230;11;AUDE;1
11381;;Sougraigne;SOUGRAIGNE;11190;11;AUDE;1
11382;;Souilhanels;SOUILHANELS;11400;11;AUDE;1
11383;;Souilhe;SOUILHE;11400;11;AUDE;1
11384;;Soulatg;SOULATGE;11350;11;AUDE;1
11385;;Soupex;SOUPEX;11320;11;AUDE;1
11386;;Talairan;TALAIRAN;11220;11;AUDE;1
11387;;Taurize;TAURIZE;11220;11;AUDE;1
11388;;Termes;TERMES;11330;11;AUDE;1
11389;;Terroles;TERROLES;11580;11;AUDE;1
11390;;Thzan-des-Corbires;THEZAN-DES-CORBIERES;11200;11;AUDE;1
11391;La;Tourette-Cabards;TOURETTE-CABARDES;11380;11;AUDE;1
11392;;Tournissan;TOURNISSAN;11220;11;AUDE;1
11393;;Tourouzelle;TOUROUZELLE;11200;11;AUDE;1
11394;;Tourreilles;TOURREILLES;11300;11;AUDE;1
11395;;Trassanel;TRASSANEL;11160;11;AUDE;1
11396;;Trausse;TRAUSSE;11160;11;AUDE;1
11397;;Trbes;TREBES;11800;11;AUDE;1
11398;;Treilles;TREILLES;11510;11;AUDE;1
11399;;Trville;TREVILLE;11400;11;AUDE;1
11400;;Trziers;TREZIERS;11230;11;AUDE;1
11401;;Tuchan;TUCHAN;11350;11;AUDE;1
11402;;Valmigre;VALMIGERE;11580;11;AUDE;1
11404;;Ventenac-Cabards;VENTENAC-CABARDES;11610;11;AUDE;1
11405;;Ventenac-en-Minervois;VENTENAC-EN-MINERVOIS;11120;11;AUDE;1
11406;;Vraza;VERAZA;11580;11;AUDE;1
11407;;Verdun-en-Lauragais;VERDUN-EN-LAURAGAIS;11400;11;AUDE;1
11408;;Verzeille;VERZEILLE;11250;11;AUDE;1
11409;;Vignevieille;VIGNEVIEILLE;11330;11;AUDE;1
11410;;Villalier;VILLALIER;11600;11;AUDE;1
11411;;Villanire;VILLANIERE;11600;11;AUDE;1
11412;;Villardebelle;VILLARDEBELLE;11580;11;AUDE;1
11413;;Villardonnel;VILLARDONNEL;11600;11;AUDE;1
11414;;Villar-en-Val;VILLAR-EN-VAL;11220;11;AUDE;1
11415;;Villar-Saint-Anselme;VILLAR-SAINT-ANSELME;11250;11;AUDE;1
11416;;Villarzel-Cabards;VILLARZEL-CABARDES;11600;11;AUDE;1
11417;;Villarzel-du-Razs;VILLARZEL-DU-RAZES;11300;11;AUDE;1
11418;;Villasavary;VILLASAVARY;11150;11;AUDE;1
11419;;Villautou;VILLAUTOU;11420;11;AUDE;1
11420;;Villebazy;VILLEBAZY;11250;11;AUDE;1
11421;;Villedaigne;VILLEDAIGNE;11200;11;AUDE;1
11422;;Villedubert;VILLEDUBERT;11800;11;AUDE;1
11423;;Villefloure;VILLEFLOURE;11570;11;AUDE;1
11424;;Villefort;VILLEFORT;11230;11;AUDE;1
11425;;Villegailhenc;VILLEGAILHENC;11600;11;AUDE;1
11426;;Villegly;VILLEGLY;11600;11;AUDE;1
11427;;Villelongue-d'Aude;VILLELONGUE-D'AUDE;11300;11;AUDE;1
11428;;Villemagne;VILLEMAGNE;11310;11;AUDE;1
11429;;Villemoustaussou;VILLEMOUSTAUSSOU;11620;11;AUDE;1
11430;;Villeneuve-la-Comptal;VILLENEUVE-LA-COMPTAL;11400;11;AUDE;1
11431;;Villeneuve-les-Corbires;VILLENEUVE-LES-CORBIERES;11360;11;AUDE;1
11432;;Villeneuve-ls-Montral;VILLENEUVE-LES-MONTREAL;11290;11;AUDE;1
11433;;Villeneuve-Minervois;VILLENEUVE-MINERVOIS;11160;11;AUDE;1
11434;;Villepinte;VILLEPINTE;11150;11;AUDE;1
11435;;Villerouge-Termens;VILLEROUGE-TERMENES;11330;11;AUDE;1
11436;;Villesque-des-Corbires;VILLESEQUE-DES-CORBIERES;11360;11;AUDE;1
11437;;Villesquelande;VILLESEQUELANDE;11170;11;AUDE;1
11438;;Villesiscle;VILLESISCLE;11150;11;AUDE;1
11439;;Villespy;VILLESPY;11170;11;AUDE;1
11440;;Villetritouls;VILLETRITOULS;11220;11;AUDE;1
11441;;Vinassan;VINASSAN;11110;11;AUDE;1
12001;;Agen-d'Aveyron;AGEN-D'AVEYRON;12630;12;AVEYRON;1
12002;;Aguessac;AGUESSAC;12520;12;AVEYRON;1
12003;Les;Albres;ALBRES;12220;12;AVEYRON;1
12004;;Almont-les-Junies;ALMONT-LES-JUNIES;12300;12;AVEYRON;1
12005;;Alpuech;ALPUECH;12210;12;AVEYRON;1
12006;;Alrance;ALRANCE;12430;12;AVEYRON;1
12007;;Ambeyrac;AMBEYRAC;12260;12;AVEYRON;1
12008;;Anglars-Saint-Flix;ANGLARS-SAINT-FELIX;12390;12;AVEYRON;1
12009;;Arnac-sur-Dourdou;ARNAC-SUR-DOURDOU;12360;12;AVEYRON;1
12010;;Arques;ARQUES;12290;12;AVEYRON;1
12011;;Arvieu;ARVIEU;12120;12;AVEYRON;1
12012;;Asprires;ASPRIERES;12700;12;AVEYRON;1
12013;;Aubin;AUBIN;12110;12;AVEYRON;1
12014;;Aurelle-Verlac;AURELLE-VERLAC;12130;12;AVEYRON;1
12015;;Auriac-Lagast;AURIAC-LAGAST;12120;12;AVEYRON;1
12016;;Auzits;AUZITS;12390;12;AVEYRON;1
12017;;Ayssnes;AYSSENES;12430;12;AVEYRON;1
12018;;Balaguier-d'Olt;BALAGUIER-D'OLT;12260;12;AVEYRON;1
12019;;Balaguier-sur-Rance;BALAGUIER-SUR-RANCE;12380;12;AVEYRON;1
12020;;Balsac;BALSAC;12510;12;AVEYRON;1
12021;La;Bastide-l'vque;BASTIDE-L'EVEQUE;12200;12;AVEYRON;1
12022;La;Bastide-Pradines;BASTIDE-PRADINES;12490;12;AVEYRON;1
12023;La;Bastide-Solages;BASTIDE-SOLAGES;12550;12;AVEYRON;1
12024;;Belcastel;BELCASTEL;12390;12;AVEYRON;1
12025;;Belmont-sur-Rance;BELMONT-SUR-RANCE;12370;12;AVEYRON;1
12026;;Bertholne;BERTHOLENE;12310;12;AVEYRON;1
12027;;Bessujouls;BESSUEJOULS;12500;12;AVEYRON;1
12028;;Boisse-Penchot;BOISSE-PENCHOT;12300;12;AVEYRON;1
12029;;Bor-et-Bar;BOR-ET-BAR;12270;12;AVEYRON;1
12030;;Bouillac;BOUILLAC;12300;12;AVEYRON;1
12031;;Bournazel;BOURNAZEL;12390;12;AVEYRON;1
12032;;Boussac;BOUSSAC;12160;12;AVEYRON;1
12033;;Bozouls;BOZOULS;12340;12;AVEYRON;1
12034;;Brandonnet;BRANDONNET;12350;12;AVEYRON;1
12035;;Brasc;BRASC;12550;12;AVEYRON;1
12036;;Brommat;BROMMAT;12600;12;AVEYRON;1
12037;;Broquis;BROQUIES;12480;12;AVEYRON;1
12038;;Brousse-le-Chteau;BROUSSE-LE-CHATEAU;12480;12;AVEYRON;1
12039;;Brusque;BRUSQUE;12360;12;AVEYRON;1
12040;;Buzeins;BUZEINS;12150;12;AVEYRON;1
12041;;Cabans;CABANES;12800;12;AVEYRON;1
12042;;Calmels-et-le-Viala;CALMELS-ET-LE-VIALA;12400;12;AVEYRON;1
12043;;Calmont;CALMONT;12450;12;AVEYRON;1
12044;;Camars;CAMARES;12360;12;AVEYRON;1
12045;;Camboulazet;CAMBOULAZET;12160;12;AVEYRON;1
12046;;Camjac;CAMJAC;12800;12;AVEYRON;1
12047;;Campagnac;CAMPAGNAC;12560;12;AVEYRON;1
12048;;Campouriez;CAMPOURIEZ;12140;12;AVEYRON;1
12049;;Campuac;CAMPUAC;12580;12;AVEYRON;1
12050;;Canet-de-Salars;CANET-DE-SALARS;12290;12;AVEYRON;1
12051;;Cantoin;CANTOIN;12420;12;AVEYRON;1
12052;;Capdenac-Gare;CAPDENAC-GARE;12700;12;AVEYRON;1
12053;La;Capelle-Balaguier;CAPELLE-BALAGUIER;12260;12;AVEYRON;1
12054;La;Capelle-Bleys;CAPELLE-BLEYS;12240;12;AVEYRON;1
12055;La;Capelle-Bonance;CAPELLE-BONANCE;12130;12;AVEYRON;1
12056;;Baraqueville;BARAQUEVILLE;12160;12;AVEYRON;1
12057;;Cassagnes-Bgonhs;CASSAGNES-BEGONHES;12120;12;AVEYRON;1
12058;;Cassujouls;CASSUEJOULS;12210;12;AVEYRON;1
12059;;Castanet;CASTANET;12240;12;AVEYRON;1
12060;;Castelmary;CASTELMARY;12800;12;AVEYRON;1
12061;;Castelnau-de-Mandailles;CASTELNAU-DE-MANDAILLES;12500;12;AVEYRON;1
12062;;Castelnau-Pgayrols;CASTELNAU-PEGAYROLS;12620;12;AVEYRON;1
12063;La;Cavalerie;CAVALERIE;12230;12;AVEYRON;1
12064;Le;Cayrol;CAYROL;12500;12;AVEYRON;1
12065;;Centrs;CENTRES;12120;12;AVEYRON;1
12066;;Clairvaux-d'Aveyron;CLAIRVAUX-D'AVEYRON;12330;12;AVEYRON;1
12067;Le;Clapier;CLAPIER;12540;12;AVEYRON;1
12068;;Colombis;COLOMBIES;12240;12;AVEYRON;1
12069;;Combret;COMBRET;12370;12;AVEYRON;1
12070;;Compeyre;COMPEYRE;12520;12;AVEYRON;1
12071;;Compolibat;COMPOLIBAT;12350;12;AVEYRON;1
12072;;Comprgnac;COMPREGNAC;12100;12;AVEYRON;1
12073;;Comps-la-Grand-Ville;COMPS-LA-GRAND-VILLE;12120;12;AVEYRON;1
12074;;Condom-d'Aubrac;CONDOM-D'AUBRAC;12470;12;AVEYRON;1
12075;;Connac;CONNAC;12170;12;AVEYRON;1
12076;;Conques;CONQUES;12320;12;AVEYRON;1
12077;;Cornus;CORNUS;12540;12;AVEYRON;1
12078;Les;Costes-Gozon;COSTES-GOZON;12400;12;AVEYRON;1
12079;;Coubisou;COUBISOU;12190;12;AVEYRON;1
12080;;Coupiac;COUPIAC;12550;12;AVEYRON;1
12081;;Coussergues;COUSSERGUES;12310;12;AVEYRON;1
12082;La;Couvertoirade;COUVERTOIRADE;12230;12;AVEYRON;1
12083;;Cransac;CRANSAC;12110;12;AVEYRON;1
12084;;Creissels;CREISSELS;12100;12;AVEYRON;1
12085;;Crespin;CRESPIN;12800;12;AVEYRON;1
12086;La;Cresse;CRESSE;12640;12;AVEYRON;1
12087;;Crujouls;CRUEJOULS;12340;12;AVEYRON;1
12088;;Curires;CURIERES;12210;12;AVEYRON;1
12089;;Decazeville;DECAZEVILLE;12300;12;AVEYRON;1
12090;;Druelle;DRUELLE;12510;12;AVEYRON;1
12091;;Drulhe;DRULHE;12350;12;AVEYRON;1
12092;;Durenque;DURENQUE;12170;12;AVEYRON;1
12093;Le;Fel;FEL;12140;12;AVEYRON;1
12094;;Entraygues-sur-Truyre;ENTRAYGUES-SUR-TRUYERE;12140;12;AVEYRON;1
12095;;Escandolires;ESCANDOLIERES;12390;12;AVEYRON;1
12096;;Espalion;ESPALION;12500;12;AVEYRON;1
12097;;Espeyrac;ESPEYRAC;12140;12;AVEYRON;1
12098;;Estaing;ESTAING;12190;12;AVEYRON;1
12099;;Fayet;FAYET;12360;12;AVEYRON;1
12100;;Firmi;FIRMI;12300;12;AVEYRON;1
12101;;Flagnac;FLAGNAC;12300;12;AVEYRON;1
12102;;Flavin;FLAVIN;12450;12;AVEYRON;1
12103;;Florentin-la-Capelle;FLORENTIN-LA-CAPELLE;12140;12;AVEYRON;1
12104;;Foissac;FOISSAC;12260;12;AVEYRON;1
12105;La;Fouillade;FOUILLADE;12270;12;AVEYRON;1
12106;;Gabriac;GABRIAC;12340;12;AVEYRON;1
12107;;Gaillac-d'Aveyron;GAILLAC-D'AVEYRON;12310;12;AVEYRON;1
12108;;Galgan;GALGAN;12220;12;AVEYRON;1
12109;;Gissac;GISSAC;12360;12;AVEYRON;1
12110;;Golinhac;GOLINHAC;12140;12;AVEYRON;1
12111;;Goutrens;GOUTRENS;12390;12;AVEYRON;1
12112;;Graissac;GRAISSAC;12420;12;AVEYRON;1
12113;;Gramond;GRAMOND;12160;12;AVEYRON;1
12114;;Grand-Vabre;GRAND-VABRE;12320;12;AVEYRON;1
12115;L';Hospitalet-du-Larzac;HOSPITALET-DU-LARZAC;12230;12;AVEYRON;1
12116;;Huparlac;HUPARLAC;12460;12;AVEYRON;1
12117;;Lacalm;LACALM;12210;12;AVEYRON;1
12118;;Lacroix-Barrez;LACROIX-BARREZ;12600;12;AVEYRON;1
12119;;Laguiole;LAGUIOLE;12210;12;AVEYRON;1
12120;;Laissac;LAISSAC;12310;12;AVEYRON;1
12121;;Lanujouls;LANUEJOULS;12350;12;AVEYRON;1
12122;;Lapanouse-de-Cernon;LAPANOUSE-DE-CERNON;12230;12;AVEYRON;1
12123;;Lapanouse;LAPANOUSE;12150;12;AVEYRON;1
12124;;Lassouts;LASSOUTS;12500;12;AVEYRON;1
12125;;Laval-Roquecezire;LAVAL-ROQUECEZIERE;12380;12;AVEYRON;1
12126;;Lavernhe;LAVERNHE;12150;12;AVEYRON;1
12127;;Ldergues;LEDERGUES;12170;12;AVEYRON;1
12128;;Lescure-Jaoul;LESCURE-JAOUL;12440;12;AVEYRON;1
12129;;Lestrade-et-Thouels;LESTRADE-ET-THOUELS;12430;12;AVEYRON;1
12130;;Livinhac-le-Haut;LIVINHAC-LE-HAUT;12300;12;AVEYRON;1
12131;La;Loubire;LOUBIERE;12740;12;AVEYRON;1
12133;;Luc-la-Primaube;LUC-LA-PRIMAUBE;12450;12;AVEYRON;1
12134;;Lugan;LUGAN;12220;12;AVEYRON;1
12135;;Lunac;LUNAC;12270;12;AVEYRON;1
12136;;Maleville;MALEVILLE;12350;12;AVEYRON;1
12137;;Manhac;MANHAC;12160;12;AVEYRON;1
12138;;Marcillac-Vallon;MARCILLAC-VALLON;12330;12;AVEYRON;1
12139;;Marnhagues-et-Latour;MARNHAGUES-ET-LATOUR;12540;12;AVEYRON;1
12140;;Martiel;MARTIEL;12200;12;AVEYRON;1
12141;;Martrin;MARTRIN;12550;12;AVEYRON;1
12142;;Mayran;MAYRAN;12390;12;AVEYRON;1
12143;;Mlagues;MELAGUES;12360;12;AVEYRON;1
12144;;Meljac;MELJAC;12120;12;AVEYRON;1
12145;;Millau;MILLAU;12100;12;AVEYRON;1
12146;Le;Monastre;MONASTERE;12000;12;AVEYRON;1
12147;;Montagnol;MONTAGNOL;12360;12;AVEYRON;1
12148;;Montbazens;MONTBAZENS;12220;12;AVEYRON;1
12149;;Montclar;MONTCLAR;12550;12;AVEYRON;1
12150;;Monteils;MONTEILS;12200;12;AVEYRON;1
12151;;Montzic;MONTEZIC;12460;12;AVEYRON;1
12152;;Montfranc;MONTFRANC;12380;12;AVEYRON;1
12153;;Montjaux;MONTJAUX;12490;12;AVEYRON;1
12154;;Montlaur;MONTLAUR;12400;12;AVEYRON;1
12155;;Fondamente;FONDAMENTE;12540;12;AVEYRON;1
12156;;Montpeyroux;MONTPEYROUX;12210;12;AVEYRON;1
12157;;Montrozier;MONTROZIER;12630;12;AVEYRON;1
12158;;Montsals;MONTSALES;12260;12;AVEYRON;1
12159;;Morlhon-le-Haut;MORLHON-LE-HAUT;12200;12;AVEYRON;1
12160;;Mostujouls;MOSTUEJOULS;12720;12;AVEYRON;1
12161;;Mouret;MOURET;12330;12;AVEYRON;1
12162;;Moyrazs;MOYRAZES;12160;12;AVEYRON;1
12163;;Murasson;MURASSON;12370;12;AVEYRON;1
12164;;Mur-de-Barrez;MUR-DE-BARREZ;12600;12;AVEYRON;1
12165;;Muret-le-Chteau;MURET-LE-CHATEAU;12330;12;AVEYRON;1
12166;;Murols;MUROLS;12600;12;AVEYRON;1
12167;;Najac;NAJAC;12270;12;AVEYRON;1
12168;;Nant;NANT;12230;12;AVEYRON;1
12169;;Naucelle;NAUCELLE;12800;12;AVEYRON;1
12170;;Naussac;NAUSSAC;12700;12;AVEYRON;1
12171;;Nauviale;NAUVIALE;12330;12;AVEYRON;1
12172;Le;Nayrac;NAYRAC;12190;12;AVEYRON;1
12173;;Noailhac;NOAILHAC;12320;12;AVEYRON;1
12174;;Olemps;OLEMPS;12510;12;AVEYRON;1
12175;;Ols-et-Rinhodes;OLS-ET-RINHODES;12260;12;AVEYRON;1
12176;;Onet-le-Chteau;ONET-LE-CHATEAU;12850;12;AVEYRON;1
12177;;Palmas;PALMAS;12310;12;AVEYRON;1
12178;;Paulhe;PAULHE;12520;12;AVEYRON;1
12179;;Peux-et-Couffouleux;PEUX-ET-COUFFOULEUX;12360;12;AVEYRON;1
12180;;Peyreleau;PEYRELEAU;12720;12;AVEYRON;1
12181;;Peyrusse-le-Roc;PEYRUSSE-LE-ROC;12220;12;AVEYRON;1
12182;;Pierrefiche;PIERREFICHE;12130;12;AVEYRON;1
12183;;Plaisance;PLAISANCE;12550;12;AVEYRON;1
12184;;Pomayrols;POMAYROLS;12130;12;AVEYRON;1
12185;;Pont-de-Salars;PONT-DE-SALARS;12290;12;AVEYRON;1
12186;;Pousthomy;POUSTHOMY;12380;12;AVEYRON;1
12187;;Prades-d'Aubrac;PRADES-D'AUBRAC;12470;12;AVEYRON;1
12188;;Prades-Salars;PRADES-SALARS;12290;12;AVEYRON;1
12189;;Pradinas;PRADINAS;12240;12;AVEYRON;1
12190;;Prvinquires;PREVINQUIERES;12350;12;AVEYRON;1
12191;;Privezac;PRIVEZAC;12350;12;AVEYRON;1
12192;;Mounes-Prohencoux;MOUNES-PROHENCOUX;12370;12;AVEYRON;1
12193;;Pruines;PRUINES;12320;12;AVEYRON;1
12194;;Quins;QUINS;12800;12;AVEYRON;1
12195;;Rebourguil;REBOURGUIL;12400;12;AVEYRON;1
12196;;Recoules-Prvinquires;RECOULES-PREVINQUIERES;12150;12;AVEYRON;1
12197;;Rquista;REQUISTA;12170;12;AVEYRON;1
12198;;Rieupeyroux;RIEUPEYROUX;12240;12;AVEYRON;1
12199;;Rignac;RIGNAC;12390;12;AVEYRON;1
12200;;Rivire-sur-Tarn;RIVIERE-SUR-TARN;12640;12;AVEYRON;1
12201;;Rodelle;RODELLE;12340;12;AVEYRON;1
12202;;Rodez;RODEZ;12000;12;AVEYRON;1
12203;;Roquefort-sur-Soulzon;ROQUEFORT-SUR-SOULZON;12250;12;AVEYRON;1
12204;La;Roque-Sainte-Marguerite;ROQUE-SAINTE-MARGUERITE;12100;12;AVEYRON;1
12205;La;Rouquette;ROUQUETTE;12200;12;AVEYRON;1
12206;;Roussennac;ROUSSENNAC;12220;12;AVEYRON;1
12207;;Rullac-Saint-Cirq;RULLAC-SAINT-CIRQ;12120;12;AVEYRON;1
12208;;Saint-Affrique;SAINT-AFFRIQUE;12400;12;AVEYRON;1
12209;;Saint-Amans-des-Cots;SAINT-AMANS-DES-COTS;12460;12;AVEYRON;1
12210;;Saint-Andr-de-Najac;SAINT-ANDRE-DE-NAJAC;12270;12;AVEYRON;1
12211;;Saint-Andr-de-Vzines;SAINT-ANDRE-DE-VEZINES;12720;12;AVEYRON;1
12212;;Saint-Beaulize;SAINT-BEAULIZE;12540;12;AVEYRON;1
12213;;Saint-Beauzly;SAINT-BEAUZELY;12620;12;AVEYRON;1
12214;;Saint-Chly-d'Aubrac;SAINT-CHELY-D'AUBRAC;12470;12;AVEYRON;1
12215;;Saint-Christophe-Vallon;SAINT-CHRISTOPHE-VALLON;12330;12;AVEYRON;1
12216;;Saint-Cme-d'Olt;SAINT-COME-D'OLT;12500;12;AVEYRON;1
12217;;Sainte-Croix;SAINTE-CROIX;12260;12;AVEYRON;1
12218;;Saint-Cyprien-sur-Dourdou;SAINT-CYPRIEN-SUR-DOURDOU;12320;12;AVEYRON;1
12219;;Sainte-Eulalie-d'Olt;SAINTE-EULALIE-D'OLT;12130;12;AVEYRON;1
12220;;Sainte-Eulalie-de-Cernon;SAINTE-EULALIE-DE-CERNON;12230;12;AVEYRON;1
12221;;Saint-Flix-de-Lunel;SAINT-FELIX-DE-LUNEL;12320;12;AVEYRON;1
12222;;Saint-Flix-de-Sorgues;SAINT-FELIX-DE-SORGUES;12400;12;AVEYRON;1
12223;;Sainte-Genevive-sur-Argence;SAINTE-GENEVIEVE-SUR-ARGENCE;12420;12;AVEYRON;1
12224;;Saint-Geniez-d'Olt;SAINT-GENIEZ-D'OLT;12130;12;AVEYRON;1
12225;;Saint-Georges-de-Luzenon;SAINT-GEORGES-DE-LUZENCON;12100;12;AVEYRON;1
12226;;Saint-Hippolyte;SAINT-HIPPOLYTE;12140;12;AVEYRON;1
12227;;Saint-Igest;SAINT-IGEST;12260;12;AVEYRON;1
12228;;Saint-Izaire;SAINT-IZAIRE;12480;12;AVEYRON;1
12229;;Saint-Jean-d'Alcapis;SAINT-JEAN-D'ALCAPIES;12250;12;AVEYRON;1
12230;;Saint-Jean-Delnous;SAINT-JEAN-DELNOUS;12170;12;AVEYRON;1
12231;;Saint-Jean-du-Bruel;SAINT-JEAN-DU-BRUEL;12230;12;AVEYRON;1
12232;;Saint-Jean-et-Saint-Paul;SAINT-JEAN-ET-SAINT-PAUL;12250;12;AVEYRON;1
12233;;Saint-Jury;SAINT-JUERY;12550;12;AVEYRON;1
12234;;Sainte-Juliette-sur-Viaur;SAINTE-JULIETTE-SUR-VIAUR;12120;12;AVEYRON;1
12235;;Saint-Just-sur-Viaur;SAINT-JUST-SUR-VIAUR;12170;12;AVEYRON;1
12236;;Saint-Laurent-de-Lvzou;SAINT-LAURENT-DE-LEVEZOU;12620;12;AVEYRON;1
12237;;Saint-Laurent-d'Olt;SAINT-LAURENT-D'OLT;12560;12;AVEYRON;1
12238;;Saint-Lons;SAINT-LEONS;12780;12;AVEYRON;1
12239;;Saint-Martin-de-Lenne;SAINT-MARTIN-DE-LENNE;12130;12;AVEYRON;1
12240;;Saint-Parthem;SAINT-PARTHEM;12300;12;AVEYRON;1
12241;;Sainte-Radegonde;SAINTE-RADEGONDE;12850;12;AVEYRON;1
12242;;Saint-Rmy;SAINT-REMY;12200;12;AVEYRON;1
12243;;Saint-Rome-de-Cernon;SAINT-ROME-DE-CERNON;12490;12;AVEYRON;1
12244;;Saint-Rome-de-Tarn;SAINT-ROME-DE-TARN;12490;12;AVEYRON;1
12245;;Saint-Salvadou;SAINT-SALVADOU;12200;12;AVEYRON;1
12246;;Saint-Santin;SAINT-SANTIN;12300;12;AVEYRON;1
12247;;Saint-Saturnin-de-Lenne;SAINT-SATURNIN-DE-LENNE;12560;12;AVEYRON;1
12248;;Saint-Sernin-sur-Rance;SAINT-SERNIN-SUR-RANCE;12380;12;AVEYRON;1
12249;;Saint-Sever-du-Moustier;SAINT-SEVER-DU-MOUSTIER;12370;12;AVEYRON;1
12250;;Saint-Symphorien-de-Thnires;SAINT-SYMPHORIEN-DE-THENIERES;12460;12;AVEYRON;1
12251;;Saint-Victor-et-Melvieu;SAINT-VICTOR-ET-MELVIEU;12400;12;AVEYRON;1
12252;;Salles-Courbatis;SALLES-COURBATIES;12260;12;AVEYRON;1
12253;;Salles-Curan;SALLES-CURAN;12410;12;AVEYRON;1
12254;;Salles-la-Source;SALLES-LA-SOURCE;12330;12;AVEYRON;1
12255;;Salmiech;SALMIECH;12120;12;AVEYRON;1
12256;;Salvagnac-Cajarc;SALVAGNAC-CAJARC;12260;12;AVEYRON;1
12257;;Causse-et-Dige;CAUSSE-ET-DIEGE;12700;12;AVEYRON;1
12258;La;Salvetat-Peyrals;SALVETAT-PEYRALES;12440;12;AVEYRON;1
12259;;Sanvensa;SANVENSA;12200;12;AVEYRON;1
12260;;Sauclires;SAUCLIERES;12230;12;AVEYRON;1
12261;;Saujac;SAUJAC;12260;12;AVEYRON;1
12262;;Sauveterre-de-Rouergue;SAUVETERRE-DE-ROUERGUE;12800;12;AVEYRON;1
12263;;Savignac;SAVIGNAC;12200;12;AVEYRON;1
12264;;Sbazac-Concours;SEBAZAC-CONCOURES;12740;12;AVEYRON;1
12265;;Sbrazac;SEBRAZAC;12190;12;AVEYRON;1
12266;;Sgur;SEGUR;12290;12;AVEYRON;1
12267;La;Selve;SELVE;12170;12;AVEYRON;1
12268;;Snergues;SENERGUES;12320;12;AVEYRON;1
12269;La;Serre;SERRE;12380;12;AVEYRON;1
12270;;Svrac-le-Chteau;SEVERAC-LE-CHATEAU;12150;12;AVEYRON;1
12271;;Svrac-l'glise;SEVERAC-L'EGLISE;12310;12;AVEYRON;1
12272;;Sonnac;SONNAC;12700;12;AVEYRON;1
12273;;Soulages-Bonneval;SOULAGES-BONNEVAL;12210;12;AVEYRON;1
12274;;Sylvans;SYLVANES;12360;12;AVEYRON;1
12275;;Tauriac-de-Camars;TAURIAC-DE-CAMARES;12360;12;AVEYRON;1
12276;;Tauriac-de-Naucelle;TAURIAC-DE-NAUCELLE;12800;12;AVEYRON;1
12277;;Taussac;TAUSSAC;12600;12;AVEYRON;1
12278;;Tayrac;TAYRAC;12440;12;AVEYRON;1
12279;La;Terrisse;TERRISSE;12210;12;AVEYRON;1
12280;;Throndels;THERONDELS;12600;12;AVEYRON;1
12281;;Toulonjac;TOULONJAC;12200;12;AVEYRON;1
12282;;Tournemire;TOURNEMIRE;12250;12;AVEYRON;1
12283;;Trmouilles;TREMOUILLES;12290;12;AVEYRON;1
12284;Le;Truel;TRUEL;12430;12;AVEYRON;1
12285;;Vabre-Tizac;VABRE-TIZAC;12240;12;AVEYRON;1
12286;;Vabres-l'Abbaye;VABRES-L'ABBAYE;12400;12;AVEYRON;1
12287;;Vailhourles;VAILHOURLES;12200;12;AVEYRON;1
12288;;Valady;VALADY;12330;12;AVEYRON;1
12289;;Valzergues;VALZERGUES;12220;12;AVEYRON;1
12290;;Vaureilles;VAUREILLES;12220;12;AVEYRON;1
12291;;Verrires;VERRIERES;12520;12;AVEYRON;1
12292;;Versols-et-Lapeyre;VERSOLS-ET-LAPEYRE;12400;12;AVEYRON;1
12293;;Veyreau;VEYREAU;12720;12;AVEYRON;1
12294;;Vzins-de-Lvzou;VEZINS-DE-LEVEZOU;12780;12;AVEYRON;1
12295;;Viala-du-Pas-de-Jaux;VIALA-DU-PAS-DE-JAUX;12250;12;AVEYRON;1
12296;;Viala-du-Tarn;VIALA-DU-TARN;12490;12;AVEYRON;1
12297;Le;Vibal;VIBAL;12290;12;AVEYRON;1
12298;;Villecomtal;VILLECOMTAL;12580;12;AVEYRON;1
12299;;Villefranche-de-Panat;VILLEFRANCHE-DE-PANAT;12430;12;AVEYRON;1
12300;;Villefranche-de-Rouergue;VILLEFRANCHE-DE-ROUERGUE;12200;12;AVEYRON;1
12301;;Villeneuve;VILLENEUVE;12260;12;AVEYRON;1
12303;;Vimenet;VIMENET;12310;12;AVEYRON;1
12304;;Vitrac-en-Viadne;VITRAC-EN-VIADENE;12420;12;AVEYRON;1
12305;;Viviez;VIVIEZ;12110;12;AVEYRON;1
12307;;Curan;CURAN;12410;12;AVEYRON;1
13001;;Aix-en-Provence;AIX-EN-PROVENCE;13090;13;BOUCHES DU RHONE;1
13002;;Allauch;ALLAUCH;13190;13;BOUCHES DU RHONE;1
13003;;Alleins;ALLEINS;13980;13;BOUCHES DU RHONE;1
13004;;Arles;ARLES;13200;13;BOUCHES DU RHONE;1
13005;;Aubagne;AUBAGNE;13400;13;BOUCHES DU RHONE;1
13006;;Aureille;AUREILLE;13930;13;BOUCHES DU RHONE;1
13007;;Auriol;AURIOL;13390;13;BOUCHES DU RHONE;1
13008;;Aurons;AURONS;13121;13;BOUCHES DU RHONE;1
13009;La;Barben;BARBEN;13330;13;BOUCHES DU RHONE;1
13010;;Barbentane;BARBENTANE;13570;13;BOUCHES DU RHONE;1
13011;Les;Baux-de-Provence;BAUX-DE-PROVENCE;13520;13;BOUCHES DU RHONE;1
13012;;Beaurecueil;BEAURECUEIL;13100;13;BOUCHES DU RHONE;1
13013;;Belcodne;BELCODENE;13720;13;BOUCHES DU RHONE;1
13014;;Berre-l'tang;BERRE-L'ETANG;13130;13;BOUCHES DU RHONE;1
13015;;Bouc-Bel-Air;BOUC-BEL-AIR;13320;13;BOUCHES DU RHONE;1
13016;La;Bouilladisse;BOUILLADISSE;13720;13;BOUCHES DU RHONE;1
13017;;Boulbon;BOULBON;13150;13;BOUCHES DU RHONE;1
13018;;Cabannes;CABANNES;13440;13;BOUCHES DU RHONE;1
13019;;Cabris;CABRIES;13480;13;BOUCHES DU RHONE;1
13020;;Cadolive;CADOLIVE;13950;13;BOUCHES DU RHONE;1
13021;;Carry-le-Rouet;CARRY-LE-ROUET;13620;13;BOUCHES DU RHONE;1
13022;;Cassis;CASSIS;13260;13;BOUCHES DU RHONE;1
13023;;Ceyreste;CEYRESTE;13600;13;BOUCHES DU RHONE;1
13024;;Charleval;CHARLEVAL;13350;13;BOUCHES DU RHONE;1
13025;;Chteauneuf-le-Rouge;CHATEAUNEUF-LE-ROUGE;13790;13;BOUCHES DU RHONE;1
13026;;Chteauneuf-les-Martigues;CHATEAUNEUF-LES-MARTIGUES;13220;13;BOUCHES DU RHONE;1
13027;;Chteaurenard;CHATEAURENARD;13160;13;BOUCHES DU RHONE;1
13028;La;Ciotat;CIOTAT;13600;13;BOUCHES DU RHONE;1
13029;;Cornillon-Confoux;CORNILLON-CONFOUX;13250;13;BOUCHES DU RHONE;1
13030;;Cuges-les-Pins;CUGES-LES-PINS;13780;13;BOUCHES DU RHONE;1
13031;La;Destrousse;DESTROUSSE;13112;13;BOUCHES DU RHONE;1
13032;;guilles;EGUILLES;13510;13;BOUCHES DU RHONE;1
13033;;Ensus-la-Redonne;ENSUES-LA-REDONNE;13820;13;BOUCHES DU RHONE;1
13034;;Eygalires;EYGALIERES;13810;13;BOUCHES DU RHONE;1
13035;;Eyguires;EYGUIERES;13430;13;BOUCHES DU RHONE;1
13036;;Eyragues;EYRAGUES;13630;13;BOUCHES DU RHONE;1
13037;La;Fare-les-Oliviers;FARE-LES-OLIVIERS;13580;13;BOUCHES DU RHONE;1
13038;;Fontvieille;FONTVIEILLE;13990;13;BOUCHES DU RHONE;1
13039;;Fos-sur-Mer;FOS-SUR-MER;13270;13;BOUCHES DU RHONE;1
13040;;Fuveau;FUVEAU;13710;13;BOUCHES DU RHONE;1
13041;;Gardanne;GARDANNE;13120;13;BOUCHES DU RHONE;1
13042;;Gmenos;GEMENOS;13420;13;BOUCHES DU RHONE;1
13043;;Gignac-la-Nerthe;GIGNAC-LA-NERTHE;13180;13;BOUCHES DU RHONE;1
13044;;Grans;GRANS;13450;13;BOUCHES DU RHONE;1
13045;;Graveson;GRAVESON;13690;13;BOUCHES DU RHONE;1
13046;;Grasque;GREASQUE;13850;13;BOUCHES DU RHONE;1
13047;;Istres;ISTRES;13800;13;BOUCHES DU RHONE;1
13048;;Jouques;JOUQUES;13490;13;BOUCHES DU RHONE;1
13049;;Lamanon;LAMANON;13113;13;BOUCHES DU RHONE;1
13050;;Lambesc;LAMBESC;13410;13;BOUCHES DU RHONE;1
13051;;Lanon-Provence;LANCON-PROVENCE;13680;13;BOUCHES DU RHONE;1
13052;;Maillane;MAILLANE;13910;13;BOUCHES DU RHONE;1
13053;;Mallemort;MALLEMORT;13370;13;BOUCHES DU RHONE;1
13054;;Marignane;MARIGNANE;13700;13;BOUCHES DU RHONE;1
13055;;Marseille;MARSEILLE;;13;;
13056;;Martigues;MARTIGUES;13500;13;BOUCHES DU RHONE;1
13057;;Mas-Blanc-des-Alpilles;MAS-BLANC-DES-ALPILLES;13103;13;BOUCHES DU RHONE;1
13058;;Maussane-les-Alpilles;MAUSSANE-LES-ALPILLES;13520;13;BOUCHES DU RHONE;1
13059;;Meyrargues;MEYRARGUES;13650;13;BOUCHES DU RHONE;1
13060;;Meyreuil;MEYREUIL;13590;13;BOUCHES DU RHONE;1
13061;;Saint-Pierre-de-Mzoargues;SAINT-PIERRE-DE-MEZOARGUES;13150;13;BOUCHES DU RHONE;1
13062;;Mimet;MIMET;13105;13;BOUCHES DU RHONE;1
13063;;Miramas;MIRAMAS;13140;13;BOUCHES DU RHONE;1
13064;;Mollgs;MOLLEGES;13940;13;BOUCHES DU RHONE;1
13065;;Mouris;MOURIES;13890;13;BOUCHES DU RHONE;1
13066;;Noves;NOVES;13550;13;BOUCHES DU RHONE;1
13067;;Orgon;ORGON;13660;13;BOUCHES DU RHONE;1
13068;;Paradou;PARADOU;13520;13;BOUCHES DU RHONE;1
13069;;Plissanne;PELISSANNE;13330;13;BOUCHES DU RHONE;1
13070;La;Penne-sur-Huveaune;PENNE-SUR-HUVEAUNE;13821;13;BOUCHES DU RHONE;1
13071;Les;Pennes-Mirabeau;PENNES-MIRABEAU;13170;13;BOUCHES DU RHONE;1
13072;;Peynier;PEYNIER;13790;13;BOUCHES DU RHONE;1
13073;;Peypin;PEYPIN;13124;13;BOUCHES DU RHONE;1
13074;;Peyrolles-en-Provence;PEYROLLES-EN-PROVENCE;13860;13;BOUCHES DU RHONE;1
13075;;Plan-de-Cuques;PLAN-DE-CUQUES;13380;13;BOUCHES DU RHONE;1
13076;;Plan-d'Orgon;PLAN-D'ORGON;13750;13;BOUCHES DU RHONE;1
13077;;Port-de-Bouc;PORT-DE-BOUC;13110;13;BOUCHES DU RHONE;1
13078;;Port-Saint-Louis-du-Rhne;PORT-SAINT-LOUIS-DU-RHONE;13230;13;BOUCHES DU RHONE;1
13079;;Puyloubier;PUYLOUBIER;13114;13;BOUCHES DU RHONE;1
13080;Le;Puy-Sainte-Rparade;PUY-SAINTE-REPARADE;13610;13;BOUCHES DU RHONE;1
13081;;Rognac;ROGNAC;13340;13;BOUCHES DU RHONE;1
13082;;Rognes;ROGNES;13840;13;BOUCHES DU RHONE;1
13083;;Rognonas;ROGNONAS;13870;13;BOUCHES DU RHONE;1
13084;La;Roque-d'Anthron;ROQUE-D'ANTHERON;13640;13;BOUCHES DU RHONE;1
13085;;Roquefort-la-Bdoule;ROQUEFORT-LA-BEDOULE;13830;13;BOUCHES DU RHONE;1
13086;;Roquevaire;ROQUEVAIRE;13360;13;BOUCHES DU RHONE;1
13087;;Rousset;ROUSSET;13790;13;BOUCHES DU RHONE;1
13088;Le;Rove;ROVE;13740;13;BOUCHES DU RHONE;1
13089;;Saint-Andiol;SAINT-ANDIOL;13670;13;BOUCHES DU RHONE;1
13090;;Saint-Antonin-sur-Bayon;SAINT-ANTONIN-SUR-BAYON;13100;13;BOUCHES DU RHONE;1
13091;;Saint-Cannat;SAINT-CANNAT;13760;13;BOUCHES DU RHONE;1
13092;;Saint-Chamas;SAINT-CHAMAS;13250;13;BOUCHES DU RHONE;1
13093;;Saint-Estve-Janson;SAINT-ESTEVE-JANSON;13610;13;BOUCHES DU RHONE;1
13094;;Saint-tienne-du-Grs;SAINT-ETIENNE-DU-GRES;13103;13;BOUCHES DU RHONE;1
13095;;Saint-Marc-Jaumegarde;SAINT-MARC-JAUMEGARDE;13100;13;BOUCHES DU RHONE;1
13096;;Saintes-Maries-de-la-Mer;SAINTES-MARIES-DE-LA-MER;13460;13;BOUCHES DU RHONE;1
13097;;Saint-Martin-de-Crau;SAINT-MARTIN-DE-CRAU;13310;13;BOUCHES DU RHONE;1
13098;;Saint-Mitre-les-Remparts;SAINT-MITRE-LES-REMPARTS;13920;13;BOUCHES DU RHONE;1
13099;;Saint-Paul-ls-Durance;SAINT-PAUL-LES-DURANCE;13115;13;BOUCHES DU RHONE;1
13100;;Saint-Rmy-de-Provence;SAINT-REMY-DE-PROVENCE;13210;13;BOUCHES DU RHONE;1
13101;;Saint-Savournin;SAINT-SAVOURNIN;13119;13;BOUCHES DU RHONE;1
13102;;Saint-Victoret;SAINT-VICTORET;13730;13;BOUCHES DU RHONE;1
13103;;Salon-de-Provence;SALON-DE-PROVENCE;13300;13;BOUCHES DU RHONE;1
13104;;Sausset-les-Pins;SAUSSET-LES-PINS;13960;13;BOUCHES DU RHONE;1
13105;;Snas;SENAS;13560;13;BOUCHES DU RHONE;1
13106;;Septmes-les-Vallons;SEPTEMES-LES-VALLONS;13240;13;BOUCHES DU RHONE;1
13107;;Simiane-Collongue;SIMIANE-COLLONGUE;13109;13;BOUCHES DU RHONE;1
13108;;Tarascon;TARASCON;13150;13;BOUCHES DU RHONE;1
13109;Le;Tholonet;THOLONET;13100;13;BOUCHES DU RHONE;1
13110;;Trets;TRETS;13530;13;BOUCHES DU RHONE;1
13111;;Vauvenargues;VAUVENARGUES;13126;13;BOUCHES DU RHONE;1
13112;;Velaux;VELAUX;13880;13;BOUCHES DU RHONE;1
13113;;Venelles;VENELLES;13770;13;BOUCHES DU RHONE;1
13114;;Ventabren;VENTABREN;13122;13;BOUCHES DU RHONE;1
13115;;Verngues;VERNEGUES;13116;13;BOUCHES DU RHONE;1
13116;;Verquires;VERQUIERES;13670;13;BOUCHES DU RHONE;1
13117;;Vitrolles;VITROLLES;13127;13;BOUCHES DU RHONE;1
13118;;Coudoux;COUDOUX;13111;13;BOUCHES DU RHONE;1
13119;;Carnoux-en-Provence;CARNOUX-EN-PROVENCE;13470;13;BOUCHES DU RHONE;1
14001;;Ablon;ABLON;14600;14;CALVADOS;1
14002;;Acqueville;ACQUEVILLE;14220;14;CALVADOS;1
14003;;Agy;AGY;14400;14;CALVADOS;1
14004;;Aignerville;AIGNERVILLE;14710;14;CALVADOS;1
14005;;Airan;AIRAN;14370;14;CALVADOS;1
14006;;Amay-sur-Orne;AMAYE-SUR-ORNE;14210;14;CALVADOS;1
14007;;Amay-sur-Seulles;AMAYE-SUR-SEULLES;14310;14;CALVADOS;1
14008;;Amblie;AMBLIE;14480;14;CALVADOS;1
14009;;Amfreville;AMFREVILLE;14860;14;CALVADOS;1
14011;;Anctoville;ANCTOVILLE;14240;14;CALVADOS;1
14012;;Angerville;ANGERVILLE;14430;14;CALVADOS;1
14013;;Angoville;ANGOVILLE;14220;14;CALVADOS;1
14014;;Anguerny;ANGUERNY;14610;14;CALVADOS;1
14015;;Anisy;ANISY;14610;14;CALVADOS;1
14016;;Annebault;ANNEBAULT;14430;14;CALVADOS;1
14019;;Arganchy;ARGANCHY;14400;14;CALVADOS;1
14020;;Argences;ARGENCES;14370;14;CALVADOS;1
14021;;Arromanches-les-Bains;ARROMANCHES-LES-BAINS;14117;14;CALVADOS;1
14022;;Asnelles;ASNELLES;14960;14;CALVADOS;1
14023;;Asnires-en-Bessin;ASNIERES-EN-BESSIN;14710;14;CALVADOS;1
14024;;Auberville;AUBERVILLE;14640;14;CALVADOS;1
14025;;Aubigny;AUBIGNY;14700;14;CALVADOS;1
14026;;Audrieu;AUDRIEU;14250;14;CALVADOS;1
14027;;Aunay-sur-Odon;AUNAY-SUR-ODON;14260;14;CALVADOS;1
14028;;Auquainville;AUQUAINVILLE;14140;14;CALVADOS;1
14029;Les;Autels-Saint-Bazile;AUTELS-SAINT-BAZILE;14140;14;CALVADOS;1
14030;;Authie;AUTHIE;14280;14;CALVADOS;1
14031;Les;Authieux-Papion;AUTHIEUX-PAPION;14140;14;CALVADOS;1
14032;Les;Authieux-sur-Calonne;AUTHIEUX-SUR-CALONNE;14130;14;CALVADOS;1
14033;;Auvillars;AUVILLARS;14340;14;CALVADOS;1
14034;;Avenay;AVENAY;14210;14;CALVADOS;1
14035;;Balleroy;BALLEROY;14490;14;CALVADOS;1
14036;;Banneville-la-Campagne;BANNEVILLE-LA-CAMPAGNE;14940;14;CALVADOS;1
14037;;Banneville-sur-Ajon;BANNEVILLE-SUR-AJON;14260;14;CALVADOS;1
14038;;Banville;BANVILLE;14480;14;CALVADOS;1
14039;;Barbery;BARBERY;14220;14;CALVADOS;1
14040;;Barbeville;BARBEVILLE;14400;14;CALVADOS;1
14041;;Barneville-la-Bertran;BARNEVILLE-LA-BERTRAN;14600;14;CALVADOS;1
14042;;Baron-sur-Odon;BARON-SUR-ODON;14210;14;CALVADOS;1
14043;;Barou-en-Auge;BAROU-EN-AUGE;14620;14;CALVADOS;1
14044;;Basly;BASLY;14610;14;CALVADOS;1
14045;;Basseneville;BASSENEVILLE;14670;14;CALVADOS;1
14046;;Bavent;BAVENT;14860;14;CALVADOS;1
14047;;Bayeux;BAYEUX;14400;14;CALVADOS;1
14049;;Bazenville;BAZENVILLE;14480;14;CALVADOS;1
14050;La;Bazoque;BAZOQUE;14490;14;CALVADOS;1
14052;;Beaulieu;BEAULIEU;14350;14;CALVADOS;1
14053;;Beaumais;BEAUMAIS;14620;14;CALVADOS;1
14054;;Beaumesnil;BEAUMESNIL;14380;14;CALVADOS;1
14055;;Beaumont-en-Auge;BEAUMONT-EN-AUGE;14950;14;CALVADOS;1
14056;;Bauquay;BAUQUAY;14260;14;CALVADOS;1
14057;;Bellengreville;BELLENGREVILLE;14370;14;CALVADOS;1
14058;;Bellou;BELLOU;14140;14;CALVADOS;1
14059;;Benerville-sur-Mer;BENERVILLE-SUR-MER;14910;14;CALVADOS;1
14060;;Bnouville;BENOUVILLE;14970;14;CALVADOS;1
14061;Le;Bny-Bocage;BENY-BOCAGE;14350;14;CALVADOS;1
14062;;Bny-sur-Mer;BENY-SUR-MER;14440;14;CALVADOS;1
14063;;Bernesq;BERNESQ;14710;14;CALVADOS;1
14064;;Bernires-d'Ailly;BERNIERES-D'AILLY;14170;14;CALVADOS;1
14065;;Bernires-le-Patry;BERNIERES-LE-PATRY;14410;14;CALVADOS;1
14066;;Bernires-sur-Mer;BERNIERES-SUR-MER;14990;14;CALVADOS;1
14068;;Biville-Beuville;BIEVILLE-BEUVILLE;14112;14;CALVADOS;1
14069;;Beuvillers;BEUVILLERS;14100;14;CALVADOS;1
14070;;Beuvron-en-Auge;BEUVRON-EN-AUGE;14430;14;CALVADOS;1
14073;La;Bigne;BIGNE;14260;14;CALVADOS;1
14074;;Billy;BILLY;14370;14;CALVADOS;1
14075;;Bissires;BISSIERES;14370;14;CALVADOS;1
14076;;Blainville-sur-Orne;BLAINVILLE-SUR-ORNE;14550;14;CALVADOS;1
14077;;Blangy-le-Chteau;BLANGY-LE-CHATEAU;14130;14;CALVADOS;1
14078;;Blay;BLAY;14400;14;CALVADOS;1
14079;;Blonville-sur-Mer;BLONVILLE-SUR-MER;14910;14;CALVADOS;1
14080;Le;B;BO;14690;14;CALVADOS;1
14081;;Boissey;BOISSEY;14170;14;CALVADOS;1
14082;La;Boissire;BOISSIERE;14340;14;CALVADOS;1
14083;;Bonnebosq;BONNEBOSQ;14340;14;CALVADOS;1
14084;;Bonnemaison;BONNEMAISON;14260;14;CALVADOS;1
14085;;Bonneville-la-Louvet;BONNEVILLE-LA-LOUVET;14130;14;CALVADOS;1
14086;;Bonneville-sur-Touques;BONNEVILLE-SUR-TOUQUES;14800;14;CALVADOS;1
14087;;Bonnil;BONNOEIL;14700;14;CALVADOS;1
14088;;Bons-Tassilly;BONS-TASSILLY;14420;14;CALVADOS;1
14089;;Bougy;BOUGY;14210;14;CALVADOS;1
14090;;Boulon;BOULON;14220;14;CALVADOS;1
14091;;Bourgeauville;BOURGEAUVILLE;14430;14;CALVADOS;1
14092;;Bourgubus;BOURGUEBUS;14540;14;CALVADOS;1
14093;;Branville;BRANVILLE;14430;14;CALVADOS;1
14096;;Brmoy;BREMOY;14260;14;CALVADOS;1
14097;;Bretteville-le-Rabet;BRETTEVILLE-LE-RABET;14190;14;CALVADOS;1
14098;;Bretteville-l'Orgueilleuse;BRETTEVILLE-L'ORGUEILLEUSE;14740;14;CALVADOS;1
14099;;Bretteville-sur-Dives;BRETTEVILLE-SUR-DIVES;14170;14;CALVADOS;1
14100;;Bretteville-sur-Laize;BRETTEVILLE-SUR-LAIZE;14680;14;CALVADOS;1
14101;;Bretteville-sur-Odon;BRETTEVILLE-SUR-ODON;14760;14;CALVADOS;1
14102;Le;Breuil-en-Auge;BREUIL-EN-AUGE;14130;14;CALVADOS;1
14103;Le;Breuil-en-Bessin;BREUIL-EN-BESSIN;14330;14;CALVADOS;1
14104;Le;Brvedent;BREVEDENT;14130;14;CALVADOS;1
14105;La;Brvire;BREVIERE;14140;14;CALVADOS;1
14106;;Brville-les-Monts;BREVILLE-LES-MONTS;14860;14;CALVADOS;1
14107;;Bricqueville;BRICQUEVILLE;14710;14;CALVADOS;1
14109;;Brouay;BROUAY;14250;14;CALVADOS;1
14110;;Brucourt;BRUCOURT;14160;14;CALVADOS;1
14111;;Bucels;BUCEELS;14250;14;CALVADOS;1
14113;;Burcy;BURCY;14410;14;CALVADOS;1
14115;;Bures-les-Monts;BURES-LES-MONTS;14350;14;CALVADOS;1
14116;Le;B-sur-Rouvres;BU-SUR-ROUVRES;14190;14;CALVADOS;1
14117;;Cabourg;CABOURG;14390;14;CALVADOS;1
14118;;Caen;CAEN;14000;14;CALVADOS;1
14119;;Cagny;CAGNY;14630;14;CALVADOS;1
14120;;Cahagnes;CAHAGNES;14240;14;CALVADOS;1
14121;;Cahagnolles;CAHAGNOLLES;14490;14;CALVADOS;1
14122;La;Caine;CAINE;14210;14;CALVADOS;1
14123;;Cairon;CAIRON;14610;14;CALVADOS;1
14124;La;Cambe;CAMBE;14230;14;CALVADOS;1
14125;;Cambes-en-Plaine;CAMBES-EN-PLAINE;14610;14;CALVADOS;1
14126;;Cambremer;CAMBREMER;14340;14;CALVADOS;1
14127;;Campagnolles;CAMPAGNOLLES;14500;14;CALVADOS;1
14128;;Campandr-Valcongrain;CAMPANDRE-VALCONGRAIN;14260;14;CALVADOS;1
14129;;Campeaux;CAMPEAUX;14350;14;CALVADOS;1
14130;;Campigny;CAMPIGNY;14490;14;CALVADOS;1
14131;;Canapville;CANAPVILLE;14800;14;CALVADOS;1
14132;;Canchy;CANCHY;14230;14;CALVADOS;1
14134;;Canteloup;CANTELOUP;14370;14;CALVADOS;1
14135;;Carcagny;CARCAGNY;14740;14;CALVADOS;1
14136;;Cardonville;CARDONVILLE;14230;14;CALVADOS;1
14137;;Carpiquet;CARPIQUET;14650;14;CALVADOS;1
14138;;Cartigny-l'pinay;CARTIGNY-L'EPINAY;14330;14;CALVADOS;1
14139;;Carville;CARVILLE;14350;14;CALVADOS;1
14140;;Castillon;CASTILLON;14490;14;CALVADOS;1
14141;;Castillon-en-Auge;CASTILLON-EN-AUGE;14140;14;CALVADOS;1
14142;;Castilly;CASTILLY;14330;14;CALVADOS;1
14143;;Caumont-l'vent;CAUMONT-L'EVENTE;14240;14;CALVADOS;1
14144;;Caumont-sur-Orne;CAUMONT-SUR-ORNE;14220;14;CALVADOS;1
14145;;Cauvicourt;CAUVICOURT;14190;14;CALVADOS;1
14146;;Cauville;CAUVILLE;14770;14;CALVADOS;1
14147;;Cernay;CERNAY;14290;14;CALVADOS;1
14148;;Cerqueux;CERQUEUX;14290;14;CALVADOS;1
14149;;Cesny-aux-Vignes;CESNY-AUX-VIGNES;14270;14;CALVADOS;1
14150;;Cesny-Bois-Halbout;CESNY-BOIS-HALBOUT;14220;14;CALVADOS;1
14151;;Champ-du-Boult;CHAMP-DU-BOULT;14380;14;CALVADOS;1
14152;La;Chapelle-Engerbold;CHAPELLE-ENGERBOLD;14770;14;CALVADOS;1
14153;La;Chapelle-Haute-Grue;CHAPELLE-HAUTE-GRUE;14140;14;CALVADOS;1
14154;La;Chapelle-Yvon;CHAPELLE-YVON;14290;14;CALVADOS;1
14155;;Cheffreville-Tonnencourt;CHEFFREVILLE-TONNENCOURT;14140;14;CALVADOS;1
14156;;Chnedoll;CHENEDOLLE;14410;14;CALVADOS;1
14157;;Cheux;CHEUX;14210;14;CALVADOS;1
14158;;Chicheboville;CHICHEBOVILLE;14370;14;CALVADOS;1
14159;;Chouain;CHOUAIN;14250;14;CALVADOS;1
14160;;Cintheaux;CINTHEAUX;14680;14;CALVADOS;1
14161;;Clarbec;CLARBEC;14130;14;CALVADOS;1
14162;;Clcy;CLECY;14570;14;CALVADOS;1
14163;;Clville;CLEVILLE;14370;14;CALVADOS;1
14164;;Clinchamps-sur-Orne;CLINCHAMPS-SUR-ORNE;14320;14;CALVADOS;1
14165;;Colleville-sur-Mer;COLLEVILLE-SUR-MER;14710;14;CALVADOS;1
14166;;Colleville-Montgomery;COLLEVILLE-MONTGOMERY;14880;14;CALVADOS;1
14167;;Colombelles;COLOMBELLES;14460;14;CALVADOS;1
14168;;Colombires;COLOMBIERES;14710;14;CALVADOS;1
14169;;Colombiers-sur-Seulles;COLOMBIERS-SUR-SEULLES;14480;14;CALVADOS;1
14170;;Colomby-sur-Thaon;COLOMBY-SUR-THAON;14610;14;CALVADOS;1
14171;;Combray;COMBRAY;14220;14;CALVADOS;1
14172;;Commes;COMMES;14520;14;CALVADOS;1
14173;;Cond-sur-Ifs;CONDE-SUR-IFS;14270;14;CALVADOS;1
14174;;Cond-sur-Noireau;CONDE-SUR-NOIREAU;14110;14;CALVADOS;1
14175;;Cond-sur-Seulles;CONDE-SUR-SEULLES;14400;14;CALVADOS;1
14176;;Conteville;CONTEVILLE;14540;14;CALVADOS;1
14177;;Coquainvilliers;COQUAINVILLIERS;14130;14;CALVADOS;1
14178;;Corbon;CORBON;14340;14;CALVADOS;1
14179;;Cordebugle;CORDEBUGLE;14100;14;CALVADOS;1
14180;;Cordey;CORDEY;14700;14;CALVADOS;1
14181;;Cormelles-le-Royal;CORMELLES-LE-ROYAL;14123;14;CALVADOS;1
14182;;Cormolain;CORMOLAIN;14240;14;CALVADOS;1
14183;;Cossesseville;COSSESSEVILLE;14690;14;CALVADOS;1
14184;;Cottun;COTTUN;14400;14;CALVADOS;1
14185;;Coudray-Rabut;COUDRAY-RABUT;14130;14;CALVADOS;1
14186;;Coulombs;COULOMBS;14480;14;CALVADOS;1
14187;;Coulonces;COULONCES;14500;14;CALVADOS;1
14188;;Coulvain;COULVAIN;14310;14;CALVADOS;1
14189;;Coupesarte;COUPESARTE;14140;14;CALVADOS;1
14190;;Courcy;COURCY;14170;14;CALVADOS;1
14191;;Courseulles-sur-Mer;COURSEULLES-SUR-MER;14470;14;CALVADOS;1
14192;;Courson;COURSON;14380;14;CALVADOS;1
14193;;Courtonne-la-Meurdrac;COURTONNE-LA-MEURDRAC;14100;14;CALVADOS;1
14194;;Courtonne-les-Deux-glises;COURTONNE-LES-DEUX-EGLISES;14290;14;CALVADOS;1
14195;;Courvaudon;COURVAUDON;14260;14;CALVADOS;1
14196;;Crpon;CREPON;14480;14;CALVADOS;1
14197;;Cresserons;CRESSERONS;14440;14;CALVADOS;1
14198;;Cresseveuille;CRESSEVEUILLE;14430;14;CALVADOS;1
14200;;Creully;CREULLY;14480;14;CALVADOS;1
14201;;Crvecur-en-Auge;CREVECOEUR-EN-AUGE;14340;14;CALVADOS;1
14202;;Cricquebuf;CRICQUEBOEUF;14113;14;CALVADOS;1
14203;;Cricqueville-en-Auge;CRICQUEVILLE-EN-AUGE;14430;14;CALVADOS;1
14204;;Cricqueville-en-Bessin;CRICQUEVILLE-EN-BESSIN;14450;14;CALVADOS;1
14205;;Cristot;CRISTOT;14250;14;CALVADOS;1
14206;;Crocy;CROCY;14620;14;CALVADOS;1
14207;;Croisilles;CROISILLES;14220;14;CALVADOS;1
14208;;Croissanville;CROISSANVILLE;14370;14;CALVADOS;1
14209;;Crouay;CROUAY;14400;14;CALVADOS;1
14210;La;Croupte;CROUPTE;14140;14;CALVADOS;1
14211;;Culey-le-Patry;CULEY-LE-PATRY;14220;14;CALVADOS;1
14212;;Cully;CULLY;14480;14;CALVADOS;1
14213;;Curcy-sur-Orne;CURCY-SUR-ORNE;14220;14;CALVADOS;1
14214;;Cussy;CUSSY;14400;14;CALVADOS;1
14215;;Cuverville;CUVERVILLE;14840;14;CALVADOS;1
14216;;Damblainville;DAMBLAINVILLE;14620;14;CALVADOS;1
14217;;Dampierre;DAMPIERRE;14350;14;CALVADOS;1
14218;;Danestal;DANESTAL;14430;14;CALVADOS;1
14219;;Danvou-la-Ferrire;DANVOU-LA-FERRIERE;14770;14;CALVADOS;1
14220;;Deauville;DEAUVILLE;14800;14;CALVADOS;1
14221;;Dmouville;DEMOUVILLE;14840;14;CALVADOS;1
14222;Le;Dsert;DESERT;14350;14;CALVADOS;1
14223;Le;Dtroit;DETROIT;14690;14;CALVADOS;1
14224;;Deux-Jumeaux;DEUX-JUMEAUX;14230;14;CALVADOS;1
14225;;Dives-sur-Mer;DIVES-SUR-MER;14160;14;CALVADOS;1
14226;;Donnay;DONNAY;14220;14;CALVADOS;1
14227;;Douville-en-Auge;DOUVILLE-EN-AUGE;14430;14;CALVADOS;1
14228;;Douvres-la-Dlivrande;DOUVRES-LA-DELIVRANDE;14440;14;CALVADOS;1
14229;;Dozul;DOZULE;14430;14;CALVADOS;1
14230;;Drubec;DRUBEC;14130;14;CALVADOS;1
14231;;Beaufour-Druval;BEAUFOUR-DRUVAL;14340;14;CALVADOS;1
14232;;Ducy-Sainte-Marguerite;DUCY-SAINTE-MARGUERITE;14250;14;CALVADOS;1
14235;;crammeville;ECRAMMEVILLE;14710;14;CALVADOS;1
14236;;Ellon;ELLON;14250;14;CALVADOS;1
14237;;miville;EMIEVILLE;14630;14;CALVADOS;1
14238;;Englesqueville-en-Auge;ENGLESQUEVILLE-EN-AUGE;14800;14;CALVADOS;1
14239;;Englesqueville-la-Perce;ENGLESQUEVILLE-LA-PERCEE;14710;14;CALVADOS;1
14240;;paney;EPANEY;14170;14;CALVADOS;1
14241;;pinay-sur-Odon;EPINAY-SUR-ODON;14310;14;CALVADOS;1
14242;;pron;EPRON;14610;14;CALVADOS;1
14243;;quemauville;EQUEMAUVILLE;14600;14;CALVADOS;1
14244;;Eraines;ERAINES;14700;14;CALVADOS;1
14245;;Ernes;ERNES;14270;14;CALVADOS;1
14246;;Escoville;ESCOVILLE;14850;14;CALVADOS;1
14248;;Espins;ESPINS;14220;14;CALVADOS;1
14249;;Esquay-Notre-Dame;ESQUAY-NOTRE-DAME;14210;14;CALVADOS;1
14250;;Esquay-sur-Seulles;ESQUAY-SUR-SEULLES;14400;14;CALVADOS;1
14251;;Esson;ESSON;14220;14;CALVADOS;1
14252;;Estres-la-Campagne;ESTREES-LA-CAMPAGNE;14190;14;CALVADOS;1
14253;;Estry;ESTRY;14410;14;CALVADOS;1
14254;;terville;ETERVILLE;14930;14;CALVADOS;1
14255;;touvy;ETOUVY;14350;14;CALVADOS;1
14256;;trham;ETREHAM;14400;14;CALVADOS;1
14257;;vrecy;EVRECY;14210;14;CALVADOS;1
14258;;Falaise;FALAISE;14700;14;CALVADOS;1
14259;;Familly;FAMILLY;14290;14;CALVADOS;1
14260;;Fauguernon;FAUGUERNON;14100;14;CALVADOS;1
14261;Le;Faulq;FAULQ;14130;14;CALVADOS;1
14264;La;Ferrire-Harang;FERRIERE-HARANG;14350;14;CALVADOS;1
14265;;Fervaques;FERVAQUES;14140;14;CALVADOS;1
14266;;Feuguerolles-Bully;FEUGUEROLLES-BULLY;14320;14;CALVADOS;1
14268;;Fierville-Bray;FIERVILLE-BRAY;14190;14;CALVADOS;1
14269;;Fierville-les-Parcs;FIERVILLE-LES-PARCS;14130;14;CALVADOS;1
14270;;Firfol;FIRFOL;14100;14;CALVADOS;1
14271;;Fleury-sur-Orne;FLEURY-SUR-ORNE;14123;14;CALVADOS;1
14272;La;Folie;FOLIE;14710;14;CALVADOS;1
14273;La;Folletire-Abenon;FOLLETIERE-ABENON;14290;14;CALVADOS;1
14274;;Fontaine-toupefour;FONTAINE-ETOUPEFOUR;14790;14;CALVADOS;1
14275;;Fontaine-Henry;FONTAINE-HENRY;14610;14;CALVADOS;1
14276;;Fontaine-le-Pin;FONTAINE-LE-PIN;14190;14;CALVADOS;1
14277;;Fontenay-le-Marmion;FONTENAY-LE-MARMION;14320;14;CALVADOS;1
14278;;Fontenay-le-Pesnel;FONTENAY-LE-PESNEL;14250;14;CALVADOS;1
14279;;Fontenermont;FONTENERMONT;14380;14;CALVADOS;1
14280;;Formentin;FORMENTIN;14340;14;CALVADOS;1
14281;;Formigny;FORMIGNY;14710;14;CALVADOS;1
14282;;Foulognes;FOULOGNES;14240;14;CALVADOS;1
14283;;Fourches;FOURCHES;14620;14;CALVADOS;1
14284;;Fourneaux-le-Val;FOURNEAUX-LE-VAL;14700;14;CALVADOS;1
14285;Le;Fournet;FOURNET;14340;14;CALVADOS;1
14286;;Fourneville;FOURNEVILLE;14600;14;CALVADOS;1
14287;;Frnouville;FRENOUVILLE;14630;14;CALVADOS;1
14288;Le;Fresne-Camilly;FRESNE-CAMILLY;14480;14;CALVADOS;1
14289;;Fresn-la-Mre;FRESNE-LA-MERE;14700;14;CALVADOS;1
14290;;Fresney-le-Puceux;FRESNEY-LE-PUCEUX;14680;14;CALVADOS;1
14291;;Fresney-le-Vieux;FRESNEY-LE-VIEUX;14220;14;CALVADOS;1
14292;;Friardel;FRIARDEL;14290;14;CALVADOS;1
14293;;Fumichon;FUMICHON;14590;14;CALVADOS;1
14294;;Garcelles-Secqueville;GARCELLES-SECQUEVILLE;14540;14;CALVADOS;1
14296;Le;Gast;GAST;14380;14;CALVADOS;1
14297;;Gavrus;GAVRUS;14210;14;CALVADOS;1
14298;;Gfosse-Fontenay;GEFOSSE-FONTENAY;14230;14;CALVADOS;1
14299;;Genneville;GENNEVILLE;14600;14;CALVADOS;1
14300;;Gerrots;GERROTS;14430;14;CALVADOS;1
14301;;Giberville;GIBERVILLE;14730;14;CALVADOS;1
14302;;Glanville;GLANVILLE;14950;14;CALVADOS;1
14303;;Glos;GLOS;14100;14;CALVADOS;1
14304;;Gonneville-sur-Honfleur;GONNEVILLE-SUR-HONFLEUR;14600;14;CALVADOS;1
14305;;Gonneville-sur-Mer;GONNEVILLE-SUR-MER;14510;14;CALVADOS;1
14306;;Gonneville-en-Auge;GONNEVILLE-EN-AUGE;14810;14;CALVADOS;1
14307;;Goupillires;GOUPILLIERES;14210;14;CALVADOS;1
14308;;Goustranville;GOUSTRANVILLE;14430;14;CALVADOS;1
14309;;Gouvix;GOUVIX;14680;14;CALVADOS;1
14310;;Grainville-Langannerie;GRAINVILLE-LANGANNERIE;14190;14;CALVADOS;1
14311;;Grainville-sur-Odon;GRAINVILLE-SUR-ODON;14210;14;CALVADOS;1
14312;;Grandcamp-Maisy;GRANDCAMP-MAISY;14450;14;CALVADOS;1
14313;;Grandchamp-le-Chteau;GRANDCHAMP-LE-CHATEAU;14140;14;CALVADOS;1
14316;;Grangues;GRANGUES;14160;14;CALVADOS;1
14317;La;Graverie;GRAVERIE;14350;14;CALVADOS;1
14318;;Graye-sur-Mer;GRAYE-SUR-MER;14470;14;CALVADOS;1
14319;;Grentheville;GRENTHEVILLE;14540;14;CALVADOS;1
14320;;Grimbosq;GRIMBOSQ;14220;14;CALVADOS;1
14322;;Guron;GUERON;14400;14;CALVADOS;1
14324;;Hamars;HAMARS;14220;14;CALVADOS;1
14325;;Hermanville-sur-Mer;HERMANVILLE-SUR-MER;14880;14;CALVADOS;1
14326;;Hermival-les-Vaux;HERMIVAL-LES-VAUX;14100;14;CALVADOS;1
14327;;Hrouville-Saint-Clair;HEROUVILLE-SAINT-CLAIR;14200;14;CALVADOS;1
14328;;Hrouvillette;HEROUVILLETTE;14850;14;CALVADOS;1
14329;;Heuland;HEULAND;14430;14;CALVADOS;1
14330;;Heurtevent;HEURTEVENT;14140;14;CALVADOS;1
14331;;Hiville;HIEVILLE;14170;14;CALVADOS;1
14332;La;Hoguette;HOGUETTE;14700;14;CALVADOS;1
14333;;Honfleur;HONFLEUR;14600;14;CALVADOS;1
14334;L';Htellerie;HOTELLERIE;14100;14;CALVADOS;1
14335;;Hotot-en-Auge;HOTOT-EN-AUGE;14430;14;CALVADOS;1
14336;;Hottot-les-Bagues;HOTTOT-LES-BAGUES;14250;14;CALVADOS;1
14337;La;Houblonnire;HOUBLONNIERE;14340;14;CALVADOS;1
14338;;Houlgate;HOULGATE;14510;14;CALVADOS;1
14339;;Hubert-Folie;HUBERT-FOLIE;14540;14;CALVADOS;1
14341;;Ifs;IFS;14123;14;CALVADOS;1
14342;;Isigny-sur-Mer;ISIGNY-SUR-MER;14230;14;CALVADOS;1
14343;Les;Isles-Bardel;ISLES-BARDEL;14690;14;CALVADOS;1
14344;;Janville;JANVILLE;14670;14;CALVADOS;1
14345;;Jort;JORT;14170;14;CALVADOS;1
14346;;Juaye-Mondaye;JUAYE-MONDAYE;14250;14;CALVADOS;1
14347;;Jurques;JURQUES;14260;14;CALVADOS;1
14348;;Juvigny-sur-Seulles;JUVIGNY-SUR-SEULLES;14250;14;CALVADOS;1
14349;;Laize-la-Ville;LAIZE-LA-VILLE;14320;14;CALVADOS;1
14350;La;Lande-sur-Drme;LANDE-SUR-DROME;14240;14;CALVADOS;1
14352;;Landelles-et-Coupigny;LANDELLES-ET-COUPIGNY;14380;14;CALVADOS;1
14353;;Landes-sur-Ajon;LANDES-SUR-AJON;14310;14;CALVADOS;1
14354;;Langrune-sur-Mer;LANGRUNE-SUR-MER;14830;14;CALVADOS;1
14355;;Lantheuil;LANTHEUIL;14480;14;CALVADOS;1
14356;;Lasson;LASSON;14740;14;CALVADOS;1
14357;;Lassy;LASSY;14770;14;CALVADOS;1
14358;;Laupartie;LEAUPARTIE;14340;14;CALVADOS;1
14359;;Lcaude;LECAUDE;14140;14;CALVADOS;1
14360;;Leffard;LEFFARD;14700;14;CALVADOS;1
14361;;Lnault;LENAULT;14770;14;CALVADOS;1
14362;;Lessard-et-le-Chne;LESSARD-ET-LE-CHENE;14140;14;CALVADOS;1
14364;;Lingvres;LINGEVRES;14250;14;CALVADOS;1
14365;;Lion-sur-Mer;LION-SUR-MER;14780;14;CALVADOS;1
14366;;Lisieux;LISIEUX;14100;14;CALVADOS;1
14367;;Lison;LISON;14330;14;CALVADOS;1
14368;;Lisores;LISORES;14140;14;CALVADOS;1
14369;;Litteau;LITTEAU;14490;14;CALVADOS;1
14370;Le;Molay-Littry;MOLAY-LITTRY;14330;14;CALVADOS;1
14371;;Livarot;LIVAROT;14140;14;CALVADOS;1
14372;;Livry;LIVRY;14240;14;CALVADOS;1
14373;Le;Locheur;LOCHEUR;14210;14;CALVADOS;1
14374;Les;Loges;LOGES;14240;14;CALVADOS;1
14375;Les;Loges-Saulces;LOGES-SAULCES;14700;14;CALVADOS;1
14376;;Longraye;LONGRAYE;14250;14;CALVADOS;1
14377;;Longues-sur-Mer;LONGUES-SUR-MER;14400;14;CALVADOS;1
14378;;Longueville;LONGUEVILLE;14230;14;CALVADOS;1
14379;;Longvillers;LONGVILLERS;14310;14;CALVADOS;1
14380;;Loucelles;LOUCELLES;14250;14;CALVADOS;1
14381;;Louvagny;LOUVAGNY;14170;14;CALVADOS;1
14382;;Louvires;LOUVIERES;14710;14;CALVADOS;1
14383;;Louvigny;LOUVIGNY;14111;14;CALVADOS;1
14384;;Luc-sur-Mer;LUC-SUR-MER;14530;14;CALVADOS;1
14385;;Magny-en-Bessin;MAGNY-EN-BESSIN;14400;14;CALVADOS;1
14386;;Magny-la-Campagne;MAGNY-LA-CAMPAGNE;14270;14;CALVADOS;1
14387;;Magny-le-Freule;MAGNY-LE-FREULE;14270;14;CALVADOS;1
14388;;Maisoncelles-la-Jourdan;MAISONCELLES-LA-JOURDAN;14500;14;CALVADOS;1
14389;;Maisoncelles-Pelvey;MAISONCELLES-PELVEY;14310;14;CALVADOS;1
14390;;Maisoncelles-sur-Ajon;MAISONCELLES-SUR-AJON;14210;14;CALVADOS;1
14391;;Maisons;MAISONS;14400;14;CALVADOS;1
14393;;Maizet;MAIZET;14210;14;CALVADOS;1
14394;;Maizires;MAIZIERES;14190;14;CALVADOS;1
14395;;Mallou;MALLOUE;14350;14;CALVADOS;1
14396;;Maltot;MALTOT;14930;14;CALVADOS;1
14397;;Mandeville-en-Bessin;MANDEVILLE-EN-BESSIN;14710;14;CALVADOS;1
14398;;Manerbe;MANERBE;14340;14;CALVADOS;1
14399;;Manneville-la-Pipard;MANNEVILLE-LA-PIPARD;14130;14;CALVADOS;1
14400;Le;Manoir;MANOIR;14400;14;CALVADOS;1
14401;;Manvieux;MANVIEUX;14117;14;CALVADOS;1
14402;Le;Marais-la-Chapelle;MARAIS-LA-CHAPELLE;14620;14;CALVADOS;1
14403;;Marolles;MAROLLES;14100;14;CALVADOS;1
14404;;Martainville;MARTAINVILLE;14220;14;CALVADOS;1
14405;;Martigny-sur-l'Ante;MARTIGNY-SUR-L'ANTE;14700;14;CALVADOS;1
14406;;Martragny;MARTRAGNY;14740;14;CALVADOS;1
14407;;Mathieu;MATHIEU;14920;14;CALVADOS;1
14408;;May-sur-Orne;MAY-SUR-ORNE;14320;14;CALVADOS;1
14409;;Merville-Franceville-Plage;MERVILLE-FRANCEVILLE-PLAGE;14810;14;CALVADOS;1
14410;;Mry-Corbon;MERY-CORBON;14370;14;CALVADOS;1
14411;;Meslay;MESLAY;14220;14;CALVADOS;1
14412;Le;Mesnil-au-Grain;MESNIL-AU-GRAIN;14260;14;CALVADOS;1
14413;Le;Mesnil-Auzouf;MESNIL-AUZOUF;14260;14;CALVADOS;1
14414;Le;Mesnil-Bacley;MESNIL-BACLEY;14140;14;CALVADOS;1
14415;Le;Mesnil-Benoist;MESNIL-BENOIST;14380;14;CALVADOS;1
14416;Le;Mesnil-Caussois;MESNIL-CAUSSOIS;14380;14;CALVADOS;1
14417;;Mesnil-Clinchamps;MESNIL-CLINCHAMPS;14380;14;CALVADOS;1
14418;Le;Mesnil-Durand;MESNIL-DURAND;14140;14;CALVADOS;1
14419;Le;Mesnil-Eudes;MESNIL-EUDES;14100;14;CALVADOS;1
14420;Le;Mesnil-Germain;MESNIL-GERMAIN;14140;14;CALVADOS;1
14421;Le;Mesnil-Guillaume;MESNIL-GUILLAUME;14100;14;CALVADOS;1
14422;Le;Mesnil-Mauger;MESNIL-MAUGER;14270;14;CALVADOS;1
14423;Le;Mesnil-Patry;MESNIL-PATRY;14740;14;CALVADOS;1
14424;Le;Mesnil-Robert;MESNIL-ROBERT;14380;14;CALVADOS;1
14425;Le;Mesnil-Simon;MESNIL-SIMON;14140;14;CALVADOS;1
14426;Le;Mesnil-sur-Blangy;MESNIL-SUR-BLANGY;14130;14;CALVADOS;1
14427;Le;Mesnil-Villement;MESNIL-VILLEMENT;14690;14;CALVADOS;1
14429;;Meulles;MEULLES;14290;14;CALVADOS;1
14430;;Meuvaines;MEUVAINES;14960;14;CALVADOS;1
14431;;Mzidon-Canon;MEZIDON-CANON;14270;14;CALVADOS;1
14432;;Missy;MISSY;14210;14;CALVADOS;1
14433;;Mittois;MITTOIS;14170;14;CALVADOS;1
14435;Les;Monceaux;MONCEAUX;14100;14;CALVADOS;1
14436;;Monceaux-en-Bessin;MONCEAUX-EN-BESSIN;14400;14;CALVADOS;1
14437;;Mondeville;MONDEVILLE;14120;14;CALVADOS;1
14438;;Mondrainville;MONDRAINVILLE;14210;14;CALVADOS;1
14439;;Monfrville;MONFREVILLE;14230;14;CALVADOS;1
14440;;Montamy;MONTAMY;14260;14;CALVADOS;1
14441;;Mont-Bertrand;MONT-BERTRAND;14350;14;CALVADOS;1
14442;;Montchamp;MONTCHAMP;14350;14;CALVADOS;1
14443;;Montchauvet;MONTCHAUVET;14350;14;CALVADOS;1
14444;;Monteille;MONTEILLE;14270;14;CALVADOS;1
14445;;Montfiquet;MONTFIQUET;14490;14;CALVADOS;1
14446;;Montigny;MONTIGNY;14210;14;CALVADOS;1
14448;;Montreuil-en-Auge;MONTREUIL-EN-AUGE;14340;14;CALVADOS;1
14449;;Monts-en-Bessin;MONTS-EN-BESSIN;14310;14;CALVADOS;1
14450;;Montviette;MONTVIETTE;14140;14;CALVADOS;1
14452;;Morteaux-Coulibuf;MORTEAUX-COULIBOEUF;14620;14;CALVADOS;1
14453;;Mosles;MOSLES;14400;14;CALVADOS;1
14454;;Mouen;MOUEN;14790;14;CALVADOS;1
14455;;Moulines;MOULINES;14220;14;CALVADOS;1
14456;;Moult;MOULT;14370;14;CALVADOS;1
14457;Les;Moutiers-en-Auge;MOUTIERS-EN-AUGE;14620;14;CALVADOS;1
14458;Les;Moutiers-en-Cinglais;MOUTIERS-EN-CINGLAIS;14220;14;CALVADOS;1
14459;Les;Moutiers-Hubert;MOUTIERS-HUBERT;14140;14;CALVADOS;1
14460;;Moyaux;MOYAUX;14590;14;CALVADOS;1
14461;;Mutrcy;MUTRECY;14220;14;CALVADOS;1
14462;;Neuilly-la-Fort;NEUILLY-LA-FORET;14230;14;CALVADOS;1
14465;;Nonant;NONANT;14400;14;CALVADOS;1
14466;;Norolles;NOROLLES;14100;14;CALVADOS;1
14467;;Noron-l'Abbaye;NORON-L'ABBAYE;14700;14;CALVADOS;1
14468;;Noron-la-Poterie;NORON-LA-POTERIE;14490;14;CALVADOS;1
14469;;Norrey-en-Auge;NORREY-EN-AUGE;14620;14;CALVADOS;1
14471;;Notre-Dame-de-Courson;NOTRE-DAME-DE-COURSON;14140;14;CALVADOS;1
14473;;Notre-Dame-de-Livaye;NOTRE-DAME-DE-LIVAYE;14340;14;CALVADOS;1
14474;;Notre-Dame-d'Estres;NOTRE-DAME-D'ESTREES;14340;14;CALVADOS;1
14475;;Noyers-Bocage;NOYERS-BOCAGE;14210;14;CALVADOS;1
14476;;Olendon;OLENDON;14170;14;CALVADOS;1
14477;;Ondefontaine;ONDEFONTAINE;14260;14;CALVADOS;1
14478;;Orbec;ORBEC;14290;14;CALVADOS;1
14480;;Osmanville;OSMANVILLE;14230;14;CALVADOS;1
14481;Les;Oubeaux;OUBEAUX;14230;14;CALVADOS;1
14482;;Ouzy;OUEZY;14270;14;CALVADOS;1
14483;;Ouffires;OUFFIERES;14220;14;CALVADOS;1
14484;;Ouilly-du-Houley;OUILLY-DU-HOULEY;14590;14;CALVADOS;1
14486;;Ouilly-le-Tesson;OUILLY-LE-TESSON;14190;14;CALVADOS;1
14487;;Ouilly-le-Vicomte;OUILLY-LE-VICOMTE;14100;14;CALVADOS;1
14488;;Ouistreham;OUISTREHAM;14150;14;CALVADOS;1
14489;;Ouville-la-Bien-Tourne;OUVILLE-LA-BIEN-TOURNEE;14170;14;CALVADOS;1
14491;;Parfouru-sur-Odon;PARFOURU-SUR-ODON;14310;14;CALVADOS;1
14492;;Pennedepie;PENNEDEPIE;14600;14;CALVADOS;1
14493;;Percy-en-Auge;PERCY-EN-AUGE;14270;14;CALVADOS;1
14494;;Priers-en-Auge;PERIERS-EN-AUGE;14160;14;CALVADOS;1
14495;;Priers-sur-le-Dan;PERIERS-SUR-LE-DAN;14112;14;CALVADOS;1
14496;;Prigny;PERIGNY;14770;14;CALVADOS;1
14497;;Perrires;PERRIERES;14170;14;CALVADOS;1
14498;;Pertheville-Ners;PERTHEVILLE-NERS;14700;14;CALVADOS;1
14499;;Petiville;PETIVILLE;14390;14;CALVADOS;1
14500;;Pierrefitte-en-Auge;PIERREFITTE-EN-AUGE;14130;14;CALVADOS;1
14501;;Pierrefitte-en-Cinglais;PIERREFITTE-EN-CINGLAIS;14690;14;CALVADOS;1
14502;;Pierrepont;PIERREPONT;14690;14;CALVADOS;1
14503;;Pierres;PIERRES;14410;14;CALVADOS;1
14504;Le;Pin;PIN;14590;14;CALVADOS;1
14505;;Placy;PLACY;14220;14;CALVADOS;1
14506;;Planquery;PLANQUERY;14490;14;CALVADOS;1
14508;Le;Plessis-Grimoult;PLESSIS-GRIMOULT;14770;14;CALVADOS;1
14509;;Plumetot;PLUMETOT;14440;14;CALVADOS;1
14510;La;Pommeraye;POMMERAYE;14690;14;CALVADOS;1
14511;;Pont-Bellanger;PONT-BELLANGER;14380;14;CALVADOS;1
14512;;Pontcoulant;PONTECOULANT;14110;14;CALVADOS;1
14513;;Pont-Farcy;PONT-FARCY;14380;14;CALVADOS;1
14514;;Pont-l'vque;PONT-L'EVEQUE;14130;14;CALVADOS;1
14515;;Port-en-Bessin-Huppain;PORT-EN-BESSIN-HUPPAIN;14520;14;CALVADOS;1
14516;;Potigny;POTIGNY;14420;14;CALVADOS;1
14517;;Poussy-la-Campagne;POUSSY-LA-CAMPAGNE;14540;14;CALVADOS;1
14518;;Praux-Saint-Sbastien;PREAUX-SAINT-SEBASTIEN;14290;14;CALVADOS;1
14519;;Praux-Bocage;PREAUX-BOCAGE;14210;14;CALVADOS;1
14520;Le;Pr-d'Auge;PRE-D'AUGE;14340;14;CALVADOS;1
14521;;Presles;PRESLES;14410;14;CALVADOS;1
14522;;Prtreville;PRETREVILLE;14140;14;CALVADOS;1
14523;;Proussy;PROUSSY;14110;14;CALVADOS;1
14524;;Putot-en-Auge;PUTOT-EN-AUGE;14430;14;CALVADOS;1
14525;;Putot-en-Bessin;PUTOT-EN-BESSIN;14740;14;CALVADOS;1
14527;;Biville-Qutiville;BIEVILLE-QUETIEVILLE;14270;14;CALVADOS;1
14528;;Quetteville;QUETTEVILLE;14130;14;CALVADOS;1
14529;;Ranchy;RANCHY;14400;14;CALVADOS;1
14530;;Ranville;RANVILLE;14860;14;CALVADOS;1
14531;;Rapilly;RAPILLY;14690;14;CALVADOS;1
14532;Le;Reculey;RECULEY;14350;14;CALVADOS;1
14533;;Repentigny;REPENTIGNY;14340;14;CALVADOS;1
14534;;Reux;REUX;14130;14;CALVADOS;1
14535;;Reviers;REVIERS;14470;14;CALVADOS;1
14536;La;Rivire-Saint-Sauveur;RIVIERE-SAINT-SAUVEUR;14600;14;CALVADOS;1
14538;;Rocquancourt;ROCQUANCOURT;14540;14;CALVADOS;1
14539;La;Rocque;ROCQUE;14410;14;CALVADOS;1
14540;;Rocques;ROCQUES;14100;14;CALVADOS;1
14541;La;Roque-Baignard;ROQUE-BAIGNARD;14340;14;CALVADOS;1
14542;;Rosel;ROSEL;14740;14;CALVADOS;1
14543;;Rots;ROTS;14980;14;CALVADOS;1
14544;;Roucamps;ROUCAMPS;14260;14;CALVADOS;1
14545;;Roullours;ROULLOURS;14500;14;CALVADOS;1
14546;;Rouvres;ROUVRES;14190;14;CALVADOS;1
14547;;Rubercy;RUBERCY;14710;14;CALVADOS;1
14548;;Rucqueville;RUCQUEVILLE;14480;14;CALVADOS;1
14549;;Rully;RULLY;14410;14;CALVADOS;1
14550;;Rumesnil;RUMESNIL;14340;14;CALVADOS;1
14551;;Russy;RUSSY;14710;14;CALVADOS;1
14552;;Ryes;RYES;14400;14;CALVADOS;1
14553;;Saint-Agnan-le-Malherbe;SAINT-AGNAN-LE-MALHERBE;14260;14;CALVADOS;1
14554;;Saint-Aignan-de-Cramesnil;SAINT-AIGNAN-DE-CRAMESNIL;14540;14;CALVADOS;1
14555;;Saint-Andr-d'Hbertot;SAINT-ANDRE-D'HEBERTOT;14130;14;CALVADOS;1
14556;;Saint-Andr-sur-Orne;SAINT-ANDRE-SUR-ORNE;14320;14;CALVADOS;1
14557;;Saint-Arnoult;SAINT-ARNOULT;14800;14;CALVADOS;1
14558;;Saint-Aubin-d'Arquenay;SAINT-AUBIN-D'ARQUENAY;14970;14;CALVADOS;1
14559;;Saint-Aubin-des-Bois;SAINT-AUBIN-DES-BOIS;14380;14;CALVADOS;1
14562;;Saint-Aubin-sur-Mer;SAINT-AUBIN-SUR-MER;14750;14;CALVADOS;1
14563;;Saint-Benot-d'Hbertot;SAINT-BENOIT-D'HEBERTOT;14130;14;CALVADOS;1
14564;;Saint-Charles-de-Percy;SAINT-CHARLES-DE-PERCY;14350;14;CALVADOS;1
14565;;Saint-Cme-de-Fresn;SAINT-COME-DE-FRESNE;14960;14;CALVADOS;1
14566;;Saint-Contest;SAINT-CONTEST;14280;14;CALVADOS;1
14568;;Sainte-Croix-Grand-Tonne;SAINTE-CROIX-GRAND-TONNE;14740;14;CALVADOS;1
14569;;Sainte-Croix-sur-Mer;SAINTE-CROIX-SUR-MER;14480;14;CALVADOS;1
14570;;Saint-Cyr-du-Ronceray;SAINT-CYR-DU-RONCERAY;14290;14;CALVADOS;1
14571;;Saint-Denis-de-Mailloc;SAINT-DENIS-DE-MAILLOC;14100;14;CALVADOS;1
14572;;Saint-Denis-de-Mr;SAINT-DENIS-DE-MERE;14110;14;CALVADOS;1
14573;;Saint-Denis-Maisoncelles;SAINT-DENIS-MAISONCELLES;14350;14;CALVADOS;1
14574;;Saint-Dsir;SAINT-DESIR;14100;14;CALVADOS;1
14575;;Saint-tienne-la-Thillaye;SAINT-ETIENNE-LA-THILLAYE;14950;14;CALVADOS;1
14576;;Sainte-Foy-de-Montgommery;SAINTE-FOY-DE-MONTGOMMERY;14140;14;CALVADOS;1
14577;;Saint-Gabriel-Brcy;SAINT-GABRIEL-BRECY;14480;14;CALVADOS;1
14578;;Saint-Gatien-des-Bois;SAINT-GATIEN-DES-BOIS;14130;14;CALVADOS;1
14579;;Saint-Georges-d'Aunay;SAINT-GEORGES-D'AUNAY;14260;14;CALVADOS;1
14580;;Saint-Georges-en-Auge;SAINT-GEORGES-EN-AUGE;14140;14;CALVADOS;1
14581;;Saint-Germain-d'Ectot;SAINT-GERMAIN-D'ECTOT;14240;14;CALVADOS;1
14582;;Saint-Germain-de-Livet;SAINT-GERMAIN-DE-LIVET;14100;14;CALVADOS;1
14583;;Saint-Germain-de-Montgommery;SAINT-GERMAIN-DE-MONTGOMMERY;14140;14;CALVADOS;1
14584;;Saint-Germain-de-Tallevende-la-Lande-Vaumont;SAINT-GERMAIN-DE-TALLEVENDE-LA-LANDE-VAUMONT;14500;14;CALVADOS;1
14585;;Saint-Germain-du-Crioult;SAINT-GERMAIN-DU-CRIOULT;14110;14;CALVADOS;1
14586;;Saint-Germain-du-Pert;SAINT-GERMAIN-DU-PERT;14230;14;CALVADOS;1
14587;;Saint-Germain-la-Blanche-Herbe;SAINT-GERMAIN-LA-BLANCHE-HERBE;14280;14;CALVADOS;1
14588;;Saint-Germain-Langot;SAINT-GERMAIN-LANGOT;14700;14;CALVADOS;1
14589;;Saint-Germain-le-Vasson;SAINT-GERMAIN-LE-VASSON;14190;14;CALVADOS;1
14590;;Sainte-Honorine-de-Ducy;SAINTE-HONORINE-DE-DUCY;14240;14;CALVADOS;1
14591;;Sainte-Honorine-des-Pertes;SAINTE-HONORINE-DES-PERTES;14520;14;CALVADOS;1
14592;;Sainte-Honorine-du-Fay;SAINTE-HONORINE-DU-FAY;14210;14;CALVADOS;1
14593;;Saint-Hymer;SAINT-HYMER;14130;14;CALVADOS;1
14595;;Saint-Jean-de-Livet;SAINT-JEAN-DE-LIVET;14100;14;CALVADOS;1
14596;;Saint-Jean-des-Essartiers;SAINT-JEAN-DES-ESSARTIERS;14350;14;CALVADOS;1
14597;;Saint-Jean-le-Blanc;SAINT-JEAN-LE-BLANC;14770;14;CALVADOS;1
14598;;Saint-Jouin;SAINT-JOUIN;14430;14;CALVADOS;1
14599;;Saint-Julien-de-Mailloc;SAINT-JULIEN-DE-MAILLOC;14290;14;CALVADOS;1
14600;;Saint-Julien-le-Faucon;SAINT-JULIEN-LE-FAUCON;14140;14;CALVADOS;1
14601;;Saint-Julien-sur-Calonne;SAINT-JULIEN-SUR-CALONNE;14130;14;CALVADOS;1
14602;;Saint-Lambert;SAINT-LAMBERT;14570;14;CALVADOS;1
14603;;Saint-Laurent-de-Condel;SAINT-LAURENT-DE-CONDEL;14220;14;CALVADOS;1
14604;;Saint-Laurent-du-Mont;SAINT-LAURENT-DU-MONT;14340;14;CALVADOS;1
14605;;Saint-Laurent-sur-Mer;SAINT-LAURENT-SUR-MER;14710;14;CALVADOS;1
14606;;Saint-Lger-Dubosq;SAINT-LEGER-DUBOSQ;14430;14;CALVADOS;1
14607;;Saint-Louet-sur-Seulles;SAINT-LOUET-SUR-SEULLES;14310;14;CALVADOS;1
14608;;Saint-Loup-de-Fribois;SAINT-LOUP-DE-FRIBOIS;14340;14;CALVADOS;1
14609;;Saint-Loup-Hors;SAINT-LOUP-HORS;14400;14;CALVADOS;1
14610;;Saint-Manvieu-Norrey;SAINT-MANVIEU-NORREY;14740;14;CALVADOS;1
14611;;Saint-Manvieu-Bocage;SAINT-MANVIEU-BOCAGE;14380;14;CALVADOS;1
14613;;Saint-Marcouf;SAINT-MARCOUF;14330;14;CALVADOS;1
14614;;Sainte-Marguerite-d'Elle;SAINTE-MARGUERITE-D'ELLE;14330;14;CALVADOS;1
14615;;Sainte-Marguerite-des-Loges;SAINTE-MARGUERITE-DES-LOGES;14140;14;CALVADOS;1
14616;;Sainte-Marguerite-de-Viette;SAINTE-MARGUERITE-DE-VIETTE;14140;14;CALVADOS;1
14618;;Sainte-Marie-Laumont;SAINTE-MARIE-LAUMONT;14350;14;CALVADOS;1
14619;;Sainte-Marie-Outre-l'Eau;SAINTE-MARIE-OUTRE-L'EAU;14380;14;CALVADOS;1
14620;;Saint-Martin-aux-Chartrains;SAINT-MARTIN-AUX-CHARTRAINS;14130;14;CALVADOS;1
14621;;Saint-Martin-de-Bienfaite-la-Cressonnire;SAINT-MARTIN-DE-BIENFAITE-LA-CRESSONNIERE;14290;14;CALVADOS;1
14622;;Saint-Martin-de-Blagny;SAINT-MARTIN-DE-BLAGNY;14710;14;CALVADOS;1
14623;;Saint-Martin-de-Fontenay;SAINT-MARTIN-DE-FONTENAY;14320;14;CALVADOS;1
14625;;Saint-Martin-de-la-Lieue;SAINT-MARTIN-DE-LA-LIEUE;14100;14;CALVADOS;1
14626;;Saint-Martin-de-Mailloc;SAINT-MARTIN-DE-MAILLOC;14100;14;CALVADOS;1
14627;;Saint-Martin-de-Mieux;SAINT-MARTIN-DE-MIEUX;14700;14;CALVADOS;1
14628;;Saint-Martin-de-Sallen;SAINT-MARTIN-DE-SALLEN;14220;14;CALVADOS;1
14629;;Saint-Martin-des-Besaces;SAINT-MARTIN-DES-BESACES;14350;14;CALVADOS;1
14630;;Saint-Martin-des-Entres;SAINT-MARTIN-DES-ENTREES;14400;14;CALVADOS;1
14632;;Saint-Martin-Don;SAINT-MARTIN-DON;14350;14;CALVADOS;1
14633;;Saint-Martin-du-Mesnil-Oury;SAINT-MARTIN-DU-MESNIL-OURY;14140;14;CALVADOS;1
14634;;Saint-Michel-de-Livet;SAINT-MICHEL-DE-LIVET;14140;14;CALVADOS;1
14635;;Saint-Omer;SAINT-OMER;14220;14;CALVADOS;1
14636;;Saint-Ouen-des-Besaces;SAINT-OUEN-DES-BESACES;14350;14;CALVADOS;1
14637;;Saint-Ouen-du-Mesnil-Oger;SAINT-OUEN-DU-MESNIL-OGER;14670;14;CALVADOS;1
14638;;Saint-Ouen-le-Houx;SAINT-OUEN-LE-HOUX;14140;14;CALVADOS;1
14639;;Saint-Ouen-le-Pin;SAINT-OUEN-LE-PIN;14340;14;CALVADOS;1
14640;;Saint-Pair;SAINT-PAIR;14670;14;CALVADOS;1
14643;;Saint-Paul-du-Vernay;SAINT-PAUL-DU-VERNAY;14490;14;CALVADOS;1
14644;;Saint-Philbert-des-Champs;SAINT-PHILBERT-DES-CHAMPS;14130;14;CALVADOS;1
14645;;Saint-Pierre-Azif;SAINT-PIERRE-AZIF;14950;14;CALVADOS;1
14646;;Saint-Pierre-Canivet;SAINT-PIERRE-CANIVET;14700;14;CALVADOS;1
14647;;Saint-Pierre-de-Mailloc;SAINT-PIERRE-DE-MAILLOC;14290;14;CALVADOS;1
14648;;Saint-Pierre-des-Ifs;SAINT-PIERRE-DES-IFS;14100;14;CALVADOS;1
14649;;Saint-Pierre-du-B;SAINT-PIERRE-DU-BU;14700;14;CALVADOS;1
14650;;Saint-Pierre-du-Fresne;SAINT-PIERRE-DU-FRESNE;14260;14;CALVADOS;1
14651;;Saint-Pierre-du-Jonquet;SAINT-PIERRE-DU-JONQUET;14670;14;CALVADOS;1
14652;;Saint-Pierre-du-Mont;SAINT-PIERRE-DU-MONT;14450;14;CALVADOS;1
14653;;Saint-Pierre-la-Vieille;SAINT-PIERRE-LA-VIEILLE;14770;14;CALVADOS;1
14654;;Saint-Pierre-sur-Dives;SAINT-PIERRE-SUR-DIVES;14170;14;CALVADOS;1
14655;;Saint-Pierre-Tarentaine;SAINT-PIERRE-TARENTAINE;14350;14;CALVADOS;1
14656;;Saint-Rmy;SAINT-REMY;14570;14;CALVADOS;1
14657;;Saint-Samson;SAINT-SAMSON;14670;14;CALVADOS;1
14658;;Saint-Sever-Calvados;SAINT-SEVER-CALVADOS;14380;14;CALVADOS;1
14659;;Saint-Sylvain;SAINT-SYLVAIN;14190;14;CALVADOS;1
14660;;Saint-Vaast-en-Auge;SAINT-VAAST-EN-AUGE;14640;14;CALVADOS;1
14661;;Saint-Vaast-sur-Seulles;SAINT-VAAST-SUR-SEULLES;14250;14;CALVADOS;1
14662;;Saint-Vigor-des-Mzerets;SAINT-VIGOR-DES-MEZERETS;14770;14;CALVADOS;1
14663;;Saint-Vigor-le-Grand;SAINT-VIGOR-LE-GRAND;14400;14;CALVADOS;1
14664;;Sallen;SALLEN;14240;14;CALVADOS;1
14665;;Sallenelles;SALLENELLES;14121;14;CALVADOS;1
14666;;Sannerville;SANNERVILLE;14940;14;CALVADOS;1
14667;;Saon;SAON;14330;14;CALVADOS;1
14668;;Saonnet;SAONNET;14330;14;CALVADOS;1
14669;;Sassy;SASSY;14170;14;CALVADOS;1
14670;;Secqueville-en-Bessin;SECQUEVILLE-EN-BESSIN;14740;14;CALVADOS;1
14671;;Sept-Frres;SEPT-FRERES;14380;14;CALVADOS;1
14672;;Sept-Vents;SEPT-VENTS;14240;14;CALVADOS;1
14674;;Soignolles;SOIGNOLLES;14190;14;CALVADOS;1
14675;;Soliers;SOLIERS;14540;14;CALVADOS;1
14676;;Sommervieu;SOMMERVIEU;14400;14;CALVADOS;1
14677;;Soulangy;SOULANGY;14700;14;CALVADOS;1
14678;;Soumont-Saint-Quentin;SOUMONT-SAINT-QUENTIN;14420;14;CALVADOS;1
14679;;Subles;SUBLES;14400;14;CALVADOS;1
14680;;Sully;SULLY;14400;14;CALVADOS;1
14681;;Surrain;SURRAIN;14710;14;CALVADOS;1
14682;;Surville;SURVILLE;14130;14;CALVADOS;1
14684;;Tessel;TESSEL;14250;14;CALVADOS;1
14685;;Thaon;THAON;14610;14;CALVADOS;1
14686;Le;Theil-Bocage;THEIL-BOCAGE;14410;14;CALVADOS;1
14687;Le;Theil-en-Auge;THEIL-EN-AUGE;14130;14;CALVADOS;1
14688;;Thiville;THIEVILLE;14170;14;CALVADOS;1
14689;;Thury-Harcourt;THURY-HARCOURT;14220;14;CALVADOS;1
14690;;Tierceville;TIERCEVILLE;14480;14;CALVADOS;1
14691;;Tilly-la-Campagne;TILLY-LA-CAMPAGNE;14540;14;CALVADOS;1
14692;;Tilly-sur-Seulles;TILLY-SUR-SEULLES;14250;14;CALVADOS;1
14693;;Tordouet;TORDOUET;14290;14;CALVADOS;1
14694;Le;Torquesne;TORQUESNE;14130;14;CALVADOS;1
14695;;Torteval-Quesnay;TORTEVAL-QUESNAY;14240;14;CALVADOS;1
14696;;Tortisambert;TORTISAMBERT;14140;14;CALVADOS;1
14697;L';Oudon;OUDON;14170;14;CALVADOS;1
14698;;Touffrville;TOUFFREVILLE;14940;14;CALVADOS;1
14699;;Touques;TOUQUES;14800;14;CALVADOS;1
14700;;Tour-en-Bessin;TOUR-EN-BESSIN;14400;14;CALVADOS;1
14701;;Tourgville;TOURGEVILLE;14800;14;CALVADOS;1
14702;;Tournay-sur-Odon;TOURNAY-SUR-ODON;14310;14;CALVADOS;1
14703;;Tournebu;TOURNEBU;14220;14;CALVADOS;1
14704;Le;Tourneur;TOURNEUR;14350;14;CALVADOS;1
14705;;Tournires;TOURNIERES;14330;14;CALVADOS;1
14706;;Tourville-en-Auge;TOURVILLE-EN-AUGE;14130;14;CALVADOS;1
14707;;Tourville-sur-Odon;TOURVILLE-SUR-ODON;14210;14;CALVADOS;1
14708;;Tracy-Bocage;TRACY-BOCAGE;14310;14;CALVADOS;1
14709;;Tracy-sur-Mer;TRACY-SUR-MER;14117;14;CALVADOS;1
14710;;Trprel;TREPREL;14690;14;CALVADOS;1
14711;;Trvires;TREVIERES;14710;14;CALVADOS;1
14712;;Troarn;TROARN;14670;14;CALVADOS;1
14713;;Trois-Monts;TROIS-MONTS;14210;14;CALVADOS;1
14714;Le;Tronquay;TRONQUAY;14490;14;CALVADOS;1
14715;;Trouville-sur-Mer;TROUVILLE-SUR-MER;14360;14;CALVADOS;1
14716;;Trungy;TRUNGY;14490;14;CALVADOS;1
14717;;Truttemer-le-Grand;TRUTTEMER-LE-GRAND;14500;14;CALVADOS;1
14718;;Truttemer-le-Petit;TRUTTEMER-LE-PETIT;14500;14;CALVADOS;1
14719;;Urville;URVILLE;14190;14;CALVADOS;1
14720;;Ussy;USSY;14420;14;CALVADOS;1
14721;;Vacognes-Neuilly;VACOGNES-NEUILLY;14210;14;CALVADOS;1
14722;La;Vacquerie;VACQUERIE;14240;14;CALVADOS;1
14723;;Valsem;VALSEME;14340;14;CALVADOS;1
14724;;Varaville;VARAVILLE;14390;14;CALVADOS;1
14726;;Vassy;VASSY;14410;14;CALVADOS;1
14727;;Vaubadon;VAUBADON;14490;14;CALVADOS;1
14728;;Vaucelles;VAUCELLES;14400;14;CALVADOS;1
14729;;Vaudeloges;VAUDELOGES;14170;14;CALVADOS;1
14730;;Vaudry;VAUDRY;14500;14;CALVADOS;1
14731;;Vauville;VAUVILLE;14800;14;CALVADOS;1
14732;;Vaux-sur-Aure;VAUX-SUR-AURE;14400;14;CALVADOS;1
14733;;Vaux-sur-Seulles;VAUX-SUR-SEULLES;14400;14;CALVADOS;1
14734;;Vendes;VENDES;14250;14;CALVADOS;1
14735;;Vendeuvre;VENDEUVRE;14170;14;CALVADOS;1
14737;;Versainville;VERSAINVILLE;14700;14;CALVADOS;1
14738;;Verson;VERSON;14790;14;CALVADOS;1
14739;;Ver-sur-Mer;VER-SUR-MER;14114;14;CALVADOS;1
14740;La;Vespire;VESPIERE;14290;14;CALVADOS;1
14741;Le;Vey;VEY;14570;14;CALVADOS;1
14742;;Vicques;VICQUES;14170;14;CALVADOS;1
14743;;Victot-Pontfol;VICTOT-PONTFOL;14430;14;CALVADOS;1
14744;;Vienne-en-Bessin;VIENNE-EN-BESSIN;14400;14;CALVADOS;1
14745;;Vierville-sur-Mer;VIERVILLE-SUR-MER;14710;14;CALVADOS;1
14746;;Viessoix;VIESSOIX;14410;14;CALVADOS;1
14747;;Vieux;VIEUX;14930;14;CALVADOS;1
14748;;Vieux-Bourg;VIEUX-BOURG;14130;14;CALVADOS;1
14749;;Vieux-Fum;VIEUX-FUME;14270;14;CALVADOS;1
14750;;Vieux-Pont-en-Auge;VIEUX-PONT-EN-AUGE;14140;14;CALVADOS;1
14751;;Vignats;VIGNATS;14700;14;CALVADOS;1
14752;;Villers-Bocage;VILLERS-BOCAGE;14310;14;CALVADOS;1
14753;;Villers-Canivet;VILLERS-CANIVET;14420;14;CALVADOS;1
14754;;Villers-sur-Mer;VILLERS-SUR-MER;14640;14;CALVADOS;1
14755;;Villerville;VILLERVILLE;14113;14;CALVADOS;1
14756;La;Villette;VILLETTE;14570;14;CALVADOS;1
14757;;Villiers-le-Sec;VILLIERS-LE-SEC;14480;14;CALVADOS;1
14758;;Villons-les-Buissons;VILLONS-LES-BUISSONS;14610;14;CALVADOS;1
14759;;Villy-lez-Falaise;VILLY-LEZ-FALAISE;14700;14;CALVADOS;1
14760;;Villy-Bocage;VILLY-BOCAGE;14310;14;CALVADOS;1
14761;;Vimont;VIMONT;14370;14;CALVADOS;1
14762;;Vire;VIRE;14500;14;CALVADOS;1
14763;;Vouilly;VOUILLY;14230;14;CALVADOS;1
14764;;Pont-d'Ouilly;PONT-D'OUILLY;14690;14;CALVADOS;1
15001;;Allanche;ALLANCHE;15160;15;CANTAL;1
15002;;Alleuze;ALLEUZE;15100;15;CANTAL;1
15003;;Ally;ALLY;15700;15;CANTAL;1
15004;;Andelat;ANDELAT;15100;15;CANTAL;1
15005;;Anglards-de-Saint-Flour;ANGLARDS-DE-SAINT-FLOUR;15100;15;CANTAL;1
15006;;Anglards-de-Salers;ANGLARDS-DE-SALERS;15380;15;CANTAL;1
15007;;Anterrieux;ANTERRIEUX;15110;15;CANTAL;1
15008;;Antignac;ANTIGNAC;15240;15;CANTAL;1
15009;;Apchon;APCHON;15400;15;CANTAL;1
15010;;Arches;ARCHES;15200;15;CANTAL;1
15011;;Arnac;ARNAC;15150;15;CANTAL;1
15012;;Arpajon-sur-Cre;ARPAJON-SUR-CERE;15130;15;CANTAL;1
15013;;Auriac-l'glise;AURIAC-L'EGLISE;15500;15;CANTAL;1
15014;;Aurillac;AURILLAC;15000;15;CANTAL;1
15015;;Auzers;AUZERS;15240;15;CANTAL;1
15016;;Ayrens;AYRENS;15250;15;CANTAL;1
15017;;Badailhac;BADAILHAC;15800;15;CANTAL;1
15018;;Barriac-les-Bosquets;BARRIAC-LES-BOSQUETS;15700;15;CANTAL;1
15019;;Bassignac;BASSIGNAC;15240;15;CANTAL;1
15020;;Beaulieu;BEAULIEU;15270;15;CANTAL;1
15021;;Boisset;BOISSET;15600;15;CANTAL;1
15022;;Bonnac;BONNAC;15500;15;CANTAL;1
15024;;Brageac;BRAGEAC;15700;15;CANTAL;1
15025;;Albepierre-Bredons;ALBEPIERRE-BREDONS;15300;15;CANTAL;1
15026;;Brezons;BREZONS;15230;15;CANTAL;1
15027;;Calvinet;CALVINET;15340;15;CANTAL;1
15028;;Carlat;CARLAT;15130;15;CANTAL;1
15029;;Cassaniouze;CASSANIOUZE;15340;15;CANTAL;1
15030;;Cayrols;CAYROLS;15290;15;CANTAL;1
15031;;Celles;CELLES;15170;15;CANTAL;1
15032;;Celoux;CELOUX;15500;15;CANTAL;1
15033;;Czens;CEZENS;15230;15;CANTAL;1
15034;;Chaliers;CHALIERS;15320;15;CANTAL;1
15035;;Chalinargues;CHALINARGUES;15170;15;CANTAL;1
15036;;Chalvignac;CHALVIGNAC;15200;15;CANTAL;1
15037;;Champagnac;CHAMPAGNAC;15350;15;CANTAL;1
15038;;Champs-sur-Tarentaine-Marchal;CHAMPS-SUR-TARENTAINE-MARCHAL;15270;15;CANTAL;1
15040;;Chanterelle;CHANTERELLE;15190;15;CANTAL;1
15041;La;Chapelle-d'Alagnon;CHAPELLE-D'ALAGNON;15300;15;CANTAL;1
15042;La;Chapelle-Laurent;CHAPELLE-LAURENT;15500;15;CANTAL;1
15043;;Charmensac;CHARMENSAC;15500;15;CANTAL;1
15044;;Chastel-sur-Murat;CHASTEL-SUR-MURAT;15300;15;CANTAL;1
15045;;Chaudes-Aigues;CHAUDES-AIGUES;15110;15;CANTAL;1
15046;;Chaussenac;CHAUSSENAC;15700;15;CANTAL;1
15047;;Chavagnac;CHAVAGNAC;15300;15;CANTAL;1
15048;;Chazelles;CHAZELLES;15500;15;CANTAL;1
15049;;Cheylade;CHEYLADE;15400;15;CANTAL;1
15050;Le;Claux;CLAUX;15400;15;CANTAL;1
15051;;Clavires;CLAVIERES;15320;15;CANTAL;1
15052;;Collandres;COLLANDRES;15400;15;CANTAL;1
15053;;Coltines;COLTINES;15170;15;CANTAL;1
15054;;Condat;CONDAT;15190;15;CANTAL;1
15055;;Coren;COREN;15100;15;CANTAL;1
15056;;Crandelles;CRANDELLES;15250;15;CANTAL;1
15057;;Cros-de-Montvert;CROS-DE-MONTVERT;15150;15;CANTAL;1
15058;;Cros-de-Ronesque;CROS-DE-RONESQUE;15130;15;CANTAL;1
15059;;Cussac;CUSSAC;15430;15;CANTAL;1
15060;;Deux-Verges;DEUX-VERGES;15110;15;CANTAL;1
15061;;Dienne;DIENNE;15300;15;CANTAL;1
15063;;Drugeac;DRUGEAC;15140;15;CANTAL;1
15064;;Escorailles;ESCORAILLES;15700;15;CANTAL;1
15065;;Espinasse;ESPINASSE;15110;15;CANTAL;1
15066;Le;Falgoux;FALGOUX;15380;15;CANTAL;1
15067;Le;Fau;FAU;15140;15;CANTAL;1
15068;;Faverolles;FAVEROLLES;15390;15;CANTAL;1
15069;;Ferrires-Saint-Mary;FERRIERES-SAINT-MARY;15170;15;CANTAL;1
15070;;Fontanges;FONTANGES;15140;15;CANTAL;1
15071;;Fournouls;FOURNOULES;15600;15;CANTAL;1
15072;;Freix-Anglards;FREIX-ANGLARDS;15310;15;CANTAL;1
15073;;Fridefont;FRIDEFONT;15110;15;CANTAL;1
15074;;Giou-de-Mamou;GIOU-DE-MAMOU;15130;15;CANTAL;1
15075;;Girgols;GIRGOLS;15310;15;CANTAL;1
15076;;Glnat;GLENAT;15150;15;CANTAL;1
15077;;Gourdiges;GOURDIEGES;15230;15;CANTAL;1
15078;;Jabrun;JABRUN;15110;15;CANTAL;1
15079;;Jaleyrac;JALEYRAC;15200;15;CANTAL;1
15080;;Joursac;JOURSAC;15170;15;CANTAL;1
15081;;Jou-sous-Monjou;JOU-SOUS-MONJOU;15800;15;CANTAL;1
15082;;Junhac;JUNHAC;15120;15;CANTAL;1
15083;;Jussac;JUSSAC;15250;15;CANTAL;1
15084;;Labesserette;LABESSERETTE;15120;15;CANTAL;1
15085;;Labrousse;LABROUSSE;15130;15;CANTAL;1
15086;;Lacapelle-Barrs;LACAPELLE-BARRES;15230;15;CANTAL;1
15087;;Lacapelle-del-Fraisse;LACAPELLE-DEL-FRAISSE;15120;15;CANTAL;1
15088;;Lacapelle-Viescamp;LACAPELLE-VIESCAMP;15150;15;CANTAL;1
15089;;Ladinhac;LADINHAC;15120;15;CANTAL;1
15090;;Lafeuillade-en-Vzie;LAFEUILLADE-EN-VEZIE;15130;15;CANTAL;1
15091;;Landeyrat;LANDEYRAT;15160;15;CANTAL;1
15092;;Lanobre;LANOBRE;15270;15;CANTAL;1
15093;;Lapeyrugue;LAPEYRUGUE;15120;15;CANTAL;1
15094;;Laroquebrou;LAROQUEBROU;15150;15;CANTAL;1
15095;;Laroquevieille;LAROQUEVIEILLE;15250;15;CANTAL;1
15096;;Lascelle;LASCELLE;15590;15;CANTAL;1
15097;;Lastic;LASTIC;15500;15;CANTAL;1
15098;;Laurie;LAURIE;15500;15;CANTAL;1
15099;;Lavastrie;LAVASTRIE;15260;15;CANTAL;1
15100;;Laveissenet;LAVEISSENET;15300;15;CANTAL;1
15101;;Laveissire;LAVEISSIERE;15300;15;CANTAL;1
15102;;Lavigerie;LAVIGERIE;15300;15;CANTAL;1
15103;;Leucamp;LEUCAMP;15120;15;CANTAL;1
15104;;Leynhac;LEYNHAC;15600;15;CANTAL;1
15105;;Leyvaux;LEYVAUX;43450;15;HAUTE LOIRE;1
15106;;Lieutads;LIEUTADES;15110;15;CANTAL;1
15107;;Lorcires;LORCIERES;15320;15;CANTAL;1
15108;;Loubaresse;LOUBARESSE;15390;15;CANTAL;1
15110;;Lugarde;LUGARDE;15190;15;CANTAL;1
15111;;Madic;MADIC;15210;15;CANTAL;1
15112;;Malbo;MALBO;15230;15;CANTAL;1
15113;;Mandailles-Saint-Julien;MANDAILLES-SAINT-JULIEN;15590;15;CANTAL;1
15114;;Marcenat;MARCENAT;15190;15;CANTAL;1
15116;;Marchastel;MARCHASTEL;15400;15;CANTAL;1
15117;;Marcols;MARCOLES;15220;15;CANTAL;1
15118;;Marmanhac;MARMANHAC;15250;15;CANTAL;1
15119;;Massiac;MASSIAC;15500;15;CANTAL;1
15120;;Mauriac;MAURIAC;15200;15;CANTAL;1
15121;;Maurines;MAURINES;15110;15;CANTAL;1
15122;;Maurs;MAURS;15600;15;CANTAL;1
15123;;Mallet;MEALLET;15200;15;CANTAL;1
15124;;Menet;MENET;15400;15;CANTAL;1
15125;;Mentires;MENTIERES;15100;15;CANTAL;1
15126;;Moldes;MOLEDES;15500;15;CANTAL;1
15127;;Molompize;MOLOMPIZE;15500;15;CANTAL;1
15128;La;Monselie;MONSELIE;15240;15;CANTAL;1
15129;;Montboudif;MONTBOUDIF;15190;15;CANTAL;1
15130;;Montchamp;MONTCHAMP;15100;15;CANTAL;1
15131;Le;Monteil;MONTEIL;15240;15;CANTAL;1
15132;;Montgreleix;MONTGRELEIX;15190;15;CANTAL;1
15133;;Montmurat;MONTMURAT;15600;15;CANTAL;1
15134;;Montsalvy;MONTSALVY;15120;15;CANTAL;1
15135;;Montvert;MONTVERT;15150;15;CANTAL;1
15136;;Mourjou;MOURJOU;15340;15;CANTAL;1
15137;;Moussages;MOUSSAGES;15380;15;CANTAL;1
15138;;Murat;MURAT;15300;15;CANTAL;1
15139;;Narnhac;NARNHAC;15230;15;CANTAL;1
15140;;Naucelles;NAUCELLES;15000;15;CANTAL;1
15141;;Neussargues-Moissac;NEUSSARGUES-MOISSAC;15170;15;CANTAL;1
15142;;Neuvglise;NEUVEGLISE;15260;15;CANTAL;1
15143;;Nieudan;NIEUDAN;15150;15;CANTAL;1
15144;;Omps;OMPS;15290;15;CANTAL;1
15145;;Oradour;ORADOUR;15260;15;CANTAL;1
15146;;Pailherols;PAILHEROLS;15800;15;CANTAL;1
15147;;Parlan;PARLAN;15290;15;CANTAL;1
15148;;Paulhac;PAULHAC;15430;15;CANTAL;1
15149;;Paulhenc;PAULHENC;15230;15;CANTAL;1
15150;;Pers;PERS;15290;15;CANTAL;1
15151;;Peyrusse;PEYRUSSE;15170;15;CANTAL;1
15152;;Pierrefort;PIERREFORT;15230;15;CANTAL;1
15153;;Pleaux;PLEAUX;15700;15;CANTAL;1
15154;;Polminhac;POLMINHAC;15800;15;CANTAL;1
15155;;Pradiers;PRADIERS;15160;15;CANTAL;1
15156;;Prunet;PRUNET;15130;15;CANTAL;1
15157;;Quzac;QUEZAC;15600;15;CANTAL;1
15158;;Rageade;RAGEADE;15500;15;CANTAL;1
15159;;Raulhac;RAULHAC;15800;15;CANTAL;1
15160;;Reilhac;REILHAC;15250;15;CANTAL;1
15161;;Rzentires;REZENTIERES;15170;15;CANTAL;1
15162;;Riom-s-Montagnes;RIOM-ES-MONTAGNES;15400;15;CANTAL;1
15163;;Roannes-Saint-Mary;ROANNES-SAINT-MARY;15220;15;CANTAL;1
15164;;Roffiac;ROFFIAC;15100;15;CANTAL;1
15165;;Rouffiac;ROUFFIAC;15150;15;CANTAL;1
15166;;Roumgoux;ROUMEGOUX;15290;15;CANTAL;1
15167;;Rouziers;ROUZIERS;15600;15;CANTAL;1
15168;;Ruynes-en-Margeride;RUYNES-EN-MARGERIDE;15320;15;CANTAL;1
15169;;Saignes;SAIGNES;15240;15;CANTAL;1
15170;;Saint-Amandin;SAINT-AMANDIN;15190;15;CANTAL;1
15171;;Sainte-Anastasie;SAINTE-ANASTASIE;15170;15;CANTAL;1
15172;;Saint-Antoine;SAINT-ANTOINE;15220;15;CANTAL;1
15173;;Saint-Bonnet-de-Condat;SAINT-BONNET-DE-CONDAT;15190;15;CANTAL;1
15174;;Saint-Bonnet-de-Salers;SAINT-BONNET-DE-SALERS;15140;15;CANTAL;1
15175;;Saint-Cernin;SAINT-CERNIN;15310;15;CANTAL;1
15176;;Saint-Chamant;SAINT-CHAMANT;15140;15;CANTAL;1
15178;;Saint-Cirgues-de-Jordanne;SAINT-CIRGUES-DE-JORDANNE;15590;15;CANTAL;1
15179;;Saint-Cirgues-de-Malbert;SAINT-CIRGUES-DE-MALBERT;15140;15;CANTAL;1
15180;;Saint-Clment;SAINT-CLEMENT;15800;15;CANTAL;1
15181;;Saint-Constant;SAINT-CONSTANT;15600;15;CANTAL;1
15182;;Saint-tienne-Cantals;SAINT-ETIENNE-CANTALES;15150;15;CANTAL;1
15183;;Saint-tienne-de-Carlat;SAINT-ETIENNE-DE-CARLAT;15130;15;CANTAL;1
15184;;Saint-tienne-de-Maurs;SAINT-ETIENNE-DE-MAURS;15600;15;CANTAL;1
15185;;Saint-tienne-de-Chomeil;SAINT-ETIENNE-DE-CHOMEIL;15400;15;CANTAL;1
15186;;Sainte-Eulalie;SAINTE-EULALIE;15140;15;CANTAL;1
15187;;Saint-Flour;SAINT-FLOUR;15100;15;CANTAL;1
15188;;Saint-Georges;SAINT-GEORGES;15100;15;CANTAL;1
15189;;Saint-Grons;SAINT-GERONS;15150;15;CANTAL;1
15190;;Saint-Hippolyte;SAINT-HIPPOLYTE;15400;15;CANTAL;1
15191;;Saint-Illide;SAINT-ILLIDE;15310;15;CANTAL;1
15192;;Saint-Jacques-des-Blats;SAINT-JACQUES-DES-BLATS;15580;15;CANTAL;1
15194;;Saint-Julien-de-Toursac;SAINT-JULIEN-DE-TOURSAC;15600;15;CANTAL;1
15195;;Saint-Just;SAINT-JUST;15390;15;CANTAL;1
15196;;Saint-Mamet-la-Salvetat;SAINT-MAMET-LA-SALVETAT;15220;15;CANTAL;1
15197;;Saint-Marc;SAINT-MARC;15390;15;CANTAL;1
15198;;Sainte-Marie;SAINTE-MARIE;15230;15;CANTAL;1
15199;;Saint-Martial;SAINT-MARTIAL;15110;15;CANTAL;1
15200;;Saint-Martin-Cantals;SAINT-MARTIN-CANTALES;15140;15;CANTAL;1
15201;;Saint-Martin-sous-Vigouroux;SAINT-MARTIN-SOUS-VIGOUROUX;15230;15;CANTAL;1
15202;;Saint-Martin-Valmeroux;SAINT-MARTIN-VALMEROUX;15140;15;CANTAL;1
15203;;Saint-Mary-le-Plain;SAINT-MARY-LE-PLAIN;15500;15;CANTAL;1
15204;;Saint-Paul-des-Landes;SAINT-PAUL-DES-LANDES;15250;15;CANTAL;1
15205;;Saint-Paul-de-Salers;SAINT-PAUL-DE-SALERS;15140;15;CANTAL;1
15206;;Saint-Pierre;SAINT-PIERRE;15350;15;CANTAL;1
15207;;Saint-Poncy;SAINT-PONCY;15500;15;CANTAL;1
15208;;Saint-Projet-de-Salers;SAINT-PROJET-DE-SALERS;15140;15;CANTAL;1
15209;;Saint-Rmy-de-Chaudes-Aigues;SAINT-REMY-DE-CHAUDES-AIGUES;15110;15;CANTAL;1
15211;;Saint-Santin-Cantals;SAINT-SANTIN-CANTALES;15150;15;CANTAL;1
15212;;Saint-Santin-de-Maurs;SAINT-SANTIN-DE-MAURS;15600;15;CANTAL;1
15213;;Saint-Saturnin;SAINT-SATURNIN;15190;15;CANTAL;1
15214;;Saint-Saury;SAINT-SAURY;15290;15;CANTAL;1
15215;;Saint-Simon;SAINT-SIMON;15130;15;CANTAL;1
15216;;Saint-Urcize;SAINT-URCIZE;15110;15;CANTAL;1
15217;;Saint-Victor;SAINT-VICTOR;15150;15;CANTAL;1
15218;;Saint-Vincent-de-Salers;SAINT-VINCENT-DE-SALERS;15380;15;CANTAL;1
15219;;Salers;SALERS;15140;15;CANTAL;1
15220;;Salins;SALINS;15200;15;CANTAL;1
15221;;Sansac-de-Marmiesse;SANSAC-DE-MARMIESSE;15130;15;CANTAL;1
15222;;Sansac-Veinazs;SANSAC-VEINAZES;15120;15;CANTAL;1
15223;;Sauvat;SAUVAT;15240;15;CANTAL;1
15224;La;Sgalassire;SEGALASSIERE;15290;15;CANTAL;1
15225;;Sgur-les-Villas;SEGUR-LES-VILLAS;15300;15;CANTAL;1
15226;;Snezergues;SENEZERGUES;15340;15;CANTAL;1
15227;;Sriers;SERIERS;15100;15;CANTAL;1
15228;;Siran;SIRAN;15150;15;CANTAL;1
15229;;Soulages;SOULAGES;15100;15;CANTAL;1
15230;;Sourniac;SOURNIAC;15200;15;CANTAL;1
15231;;Talizat;TALIZAT;15170;15;CANTAL;1
15232;;Tanavelle;TANAVELLE;15100;15;CANTAL;1
15233;;Teissires-de-Cornet;TEISSIERES-DE-CORNET;15250;15;CANTAL;1
15234;;Teissires-ls-Boulis;TEISSIERES-LES-BOULIES;15130;15;CANTAL;1
15235;Les;Ternes;TERNES;15100;15;CANTAL;1
15236;;Thizac;THIEZAC;15450;15;CANTAL;1
15237;;Tiviers;TIVIERS;15100;15;CANTAL;1
15238;;Tournemire;TOURNEMIRE;15310;15;CANTAL;1
15240;;Trmouille;TREMOUILLE;15270;15;CANTAL;1
15241;La;Trinitat;TRINITAT;15110;15;CANTAL;1
15242;Le;Trioulou;TRIOULOU;15600;15;CANTAL;1
15243;;Trizac;TRIZAC;15400;15;CANTAL;1
15244;;Ussel;USSEL;15300;15;CANTAL;1
15245;;Vabres;VABRES;15100;15;CANTAL;1
15246;;Valette;VALETTE;15400;15;CANTAL;1
15247;;Valjouze;VALJOUZE;15170;15;CANTAL;1
15248;;Valujols;VALUEJOLS;15300;15;CANTAL;1
15249;Le;Vaulmier;VAULMIER;15380;15;CANTAL;1
15250;;Vebret;VEBRET;15240;15;CANTAL;1
15251;;Vdrines-Saint-Loup;VEDRINES-SAINT-LOUP;15100;15;CANTAL;1
15252;;Velzic;VELZIC;15590;15;CANTAL;1
15253;;Vernols;VERNOLS;15160;15;CANTAL;1
15254;;Veyrires;VEYRIERES;15350;15;CANTAL;1
15255;;Vzac;VEZAC;15130;15;CANTAL;1
15256;;Vze;VEZE;15160;15;CANTAL;1
15257;;Vezels-Roussy;VEZELS-ROUSSY;15130;15;CANTAL;1
15258;;Vic-sur-Cre;VIC-SUR-CERE;15800;15;CANTAL;1
15259;;Vieillespesse;VIEILLESPESSE;15500;15;CANTAL;1
15260;;Vieillevie;VIEILLEVIE;15120;15;CANTAL;1
15261;Le;Vigean;VIGEAN;15200;15;CANTAL;1
15262;;Villedieu;VILLEDIEU;15100;15;CANTAL;1
15263;;Virargues;VIRARGUES;15300;15;CANTAL;1
15264;;Vitrac;VITRAC;15220;15;CANTAL;1
15265;;Ydes;YDES;15210;15;CANTAL;1
15266;;Yolet;YOLET;15130;15;CANTAL;1
15267;;Ytrac;YTRAC;15000;15;CANTAL;1
15268;Le;Rouget;ROUGET;15290;15;CANTAL;1
15269;;Besse;BESSE;15140;15;CANTAL;1
16001;;Abzac;ABZAC;16500;16;CHARENTE;1
16002;Les;Adjots;ADJOTS;16700;16;CHARENTE;1
16003;;Agris;AGRIS;16110;16;CHARENTE;1
16004;;Aignes-et-Puyproux;AIGNES-ET-PUYPEROUX;16190;16;CHARENTE;1
16005;;Aigre;AIGRE;16140;16;CHARENTE;1
16007;;Alloue;ALLOUE;16490;16;CHARENTE;1
16008;;Ambrac;AMBERAC;16140;16;CHARENTE;1
16009;;Ambernac;AMBERNAC;16490;16;CHARENTE;1
16010;;Ambleville;AMBLEVILLE;16300;16;CHARENTE;1
16011;;Anais;ANAIS;16560;16;CHARENTE;1
16012;;Angeac-Champagne;ANGEAC-CHAMPAGNE;16130;16;CHARENTE;1
16013;;Angeac-Charente;ANGEAC-CHARENTE;16120;16;CHARENTE;1
16014;;Angeduc;ANGEDUC;16300;16;CHARENTE;1
16015;;Angoulme;ANGOULEME;16000;16;CHARENTE;1
16016;;Ansac-sur-Vienne;ANSAC-SUR-VIENNE;16500;16;CHARENTE;1
16017;;Anville;ANVILLE;16170;16;CHARENTE;1
16018;;Ars;ARS;16130;16;CHARENTE;1
16019;;Asnires-sur-Noure;ASNIERES-SUR-NOUERE;16290;16;CHARENTE;1
16020;;Aubeterre-sur-Dronne;AUBETERRE-SUR-DRONNE;16390;16;CHARENTE;1
16021;;Aubeville;AUBEVILLE;16250;16;CHARENTE;1
16023;;Aunac;AUNAC;16460;16;CHARENTE;1
16024;;Aussac-Vadalle;AUSSAC-VADALLE;16560;16;CHARENTE;1
16025;;Baignes-Sainte-Radegonde;BAIGNES-SAINTE-RADEGONDE;16360;16;CHARENTE;1
16026;;Balzac;BALZAC;16430;16;CHARENTE;1
16027;;Barbezires;BARBEZIERES;16140;16;CHARENTE;1
16028;;Barbezieux-Saint-Hilaire;BARBEZIEUX-SAINT-HILAIRE;16300;16;CHARENTE;1
16029;;Bardenac;BARDENAC;16210;16;CHARENTE;1
16030;;Barret;BARRET;16300;16;CHARENTE;1
16031;;Barro;BARRO;16700;16;CHARENTE;1
16032;;Bassac;BASSAC;16120;16;CHARENTE;1
16033;;Bayers;BAYERS;16460;16;CHARENTE;1
16034;;Bazac;BAZAC;16210;16;CHARENTE;1
16035;;Beaulieu-sur-Sonnette;BEAULIEU-SUR-SONNETTE;16450;16;CHARENTE;1
16036;;Bcheresse;BECHERESSE;16250;16;CHARENTE;1
16037;;Bellon;BELLON;16210;16;CHARENTE;1
16038;;Benest;BENEST;16350;16;CHARENTE;1
16039;;Bernac;BERNAC;16700;16;CHARENTE;1
16040;;Berneuil;BERNEUIL;16480;16;CHARENTE;1
16041;;Bessac;BESSAC;16250;16;CHARENTE;1
16042;;Bess;BESSE;16140;16;CHARENTE;1
16043;;Bignac;BIGNAC;16170;16;CHARENTE;1
16044;;Bioussac;BIOUSSAC;16700;16;CHARENTE;1
16045;;Birac;BIRAC;16120;16;CHARENTE;1
16046;;Blanzac-Porcheresse;BLANZAC-PORCHERESSE;16250;16;CHARENTE;1
16047;;Blanzaguet-Saint-Cybard;BLANZAGUET-SAINT-CYBARD;16320;16;CHARENTE;1
16048;;Boisbreteau;BOISBRETEAU;16480;16;CHARENTE;1
16049;;Bonnes;BONNES;16390;16;CHARENTE;1
16050;;Bonneuil;BONNEUIL;16120;16;CHARENTE;1
16051;;Bonneville;BONNEVILLE;16170;16;CHARENTE;1
16052;;(CANTON;BORS;16190;16;CHARENTE;1
16053;;(CANTON;BORS;16360;16;CHARENTE;1
16054;Le;Bouchage;BOUCHAGE;16350;16;CHARENTE;1
16055;;Boux;BOUEX;16410;16;CHARENTE;1
16056;;Bourg-Charente;BOURG-CHARENTE;16200;16;CHARENTE;1
16057;;Bouteville;BOUTEVILLE;16120;16;CHARENTE;1
16058;;Boutiers-Saint-Trojan;BOUTIERS-SAINT-TROJAN;16100;16;CHARENTE;1
16059;;Brettes;BRETTES;16240;16;CHARENTE;1
16060;;Brville;BREVILLE;16370;16;CHARENTE;1
16061;;Brie;BRIE;16590;16;CHARENTE;1
16062;;Brie-sous-Barbezieux;BRIE-SOUS-BARBEZIEUX;16300;16;CHARENTE;1
16063;;Brie-sous-Chalais;BRIE-SOUS-CHALAIS;16210;16;CHARENTE;1
16064;;Brigueuil;BRIGUEUIL;16420;16;CHARENTE;1
16065;;Brillac;BRILLAC;16500;16;CHARENTE;1
16066;;Brossac;BROSSAC;16480;16;CHARENTE;1
16067;;Bunzac;BUNZAC;16110;16;CHARENTE;1
16068;;Cellefrouin;CELLEFROUIN;16260;16;CHARENTE;1
16069;;Cellettes;CELLETTES;16230;16;CHARENTE;1
16070;;Chabanais;CHABANAIS;16150;16;CHARENTE;1
16071;;Chabrac;CHABRAC;16150;16;CHARENTE;1
16072;;Chadurie;CHADURIE;16250;16;CHARENTE;1
16073;;Chalais;CHALAIS;16210;16;CHARENTE;1
16074;;Challignac;CHALLIGNAC;16300;16;CHARENTE;1
16075;;Champagne-Vigny;CHAMPAGNE-VIGNY;16250;16;CHARENTE;1
16076;;Champagne-Mouton;CHAMPAGNE-MOUTON;16350;16;CHARENTE;1
16077;;Champmillon;CHAMPMILLON;16290;16;CHARENTE;1
16078;;Champniers;CHAMPNIERS;16430;16;CHARENTE;1
16079;;Chantillac;CHANTILLAC;16360;16;CHARENTE;1
16081;La;Chapelle;CHAPELLE;16140;16;CHARENTE;1
16082;;Charmant;CHARMANT;16320;16;CHARENTE;1
16083;;Charm;CHARME;16140;16;CHARENTE;1
16084;;Charras;CHARRAS;16380;16;CHARENTE;1
16085;;Chasseneuil-sur-Bonnieure;CHASSENEUIL-SUR-BONNIEURE;16260;16;CHARENTE;1
16086;;Chassenon;CHASSENON;16150;16;CHARENTE;1
16087;;Chassiecq;CHASSIECQ;16350;16;CHARENTE;1
16088;;Chassors;CHASSORS;16200;16;CHARENTE;1
16089;;Chteaubernard;CHATEAUBERNARD;16100;16;CHARENTE;1
16090;;Chteauneuf-sur-Charente;CHATEAUNEUF-SUR-CHARENTE;16120;16;CHARENTE;1
16091;;Chtignac;CHATIGNAC;16480;16;CHARENTE;1
16092;;Chavenat;CHAVENAT;16320;16;CHARENTE;1
16093;;Chazelles;CHAZELLES;16380;16;CHARENTE;1
16094;;Chenommet;CHENOMMET;16460;16;CHARENTE;1
16095;;Chenon;CHENON;16460;16;CHARENTE;1
16096;;Cherves-Chtelars;CHERVES-CHATELARS;16310;16;CHARENTE;1
16097;;Cherves-Richemont;CHERVES-RICHEMONT;16370;16;CHARENTE;1
16098;La;Chvrerie;CHEVRERIE;16240;16;CHARENTE;1
16099;;Chillac;CHILLAC;16480;16;CHARENTE;1
16100;;Chirac;CHIRAC;16150;16;CHARENTE;1
16101;;Claix;CLAIX;16440;16;CHARENTE;1
16102;;Cognac;COGNAC;16100;16;CHARENTE;1
16103;;Combiers;COMBIERS;16320;16;CHARENTE;1
16104;;Condac;CONDAC;16700;16;CHARENTE;1
16105;;Condon;CONDEON;16360;16;CHARENTE;1
16106;;Confolens;CONFOLENS;16500;16;CHARENTE;1
16107;;Coulgens;COULGENS;16560;16;CHARENTE;1
16108;;Coulonges;COULONGES;16330;16;CHARENTE;1
16109;;Courbillac;COURBILLAC;16200;16;CHARENTE;1
16110;;Courcme;COURCOME;16240;16;CHARENTE;1
16111;;Courgeac;COURGEAC;16190;16;CHARENTE;1
16112;;Courlac;COURLAC;16210;16;CHARENTE;1
16113;La;Couronne;COURONNE;16400;16;CHARENTE;1
16114;;Couture;COUTURE;16460;16;CHARENTE;1
16115;;Cressac-Saint-Genis;CRESSAC-SAINT-GENIS;16250;16;CHARENTE;1
16116;;Criteuil-la-Magdeleine;CRITEUIL-LA-MAGDELEINE;16300;16;CHARENTE;1
16117;;Curac;CURAC;16210;16;CHARENTE;1
16118;;Deviat;DEVIAT;16190;16;CHARENTE;1
16119;;Dignac;DIGNAC;16410;16;CHARENTE;1
16120;;Dirac;DIRAC;16410;16;CHARENTE;1
16121;;Douzat;DOUZAT;16290;16;CHARENTE;1
16122;;bron;EBREON;16140;16;CHARENTE;1
16123;;challat;ECHALLAT;16170;16;CHARENTE;1
16124;;curas;ECURAS;16220;16;CHARENTE;1
16125;;don;EDON;16320;16;CHARENTE;1
16127;;Empur;EMPURE;16240;16;CHARENTE;1
16128;;pende;EPENEDE;16490;16;CHARENTE;1
16129;;raville;ERAVILLE;16120;16;CHARENTE;1
16130;Les;Essards;ESSARDS;16210;16;CHARENTE;1
16131;;Esse;ESSE;16500;16;CHARENTE;1
16132;;tagnac;ETAGNAC;16150;16;CHARENTE;1
16133;;triac;ETRIAC;16250;16;CHARENTE;1
16134;;Exideuil;EXIDEUIL;16150;16;CHARENTE;1
16135;;Eymouthiers;EYMOUTHIERS;16220;16;CHARENTE;1
16136;La;Faye;FAYE;16700;16;CHARENTE;1
16137;;Feuillade;FEUILLADE;16380;16;CHARENTE;1
16138;;Flac;FLEAC;16730;16;CHARENTE;1
16139;;Fleurac;FLEURAC;16200;16;CHARENTE;1
16140;;Fontclaireau;FONTCLAIREAU;16230;16;CHARENTE;1
16141;;Fontenille;FONTENILLE;16230;16;CHARENTE;1
16142;La;Fort-de-Tess;FORET-DE-TESSE;16240;16;CHARENTE;1
16143;;Fouquebrune;FOUQUEBRUNE;16410;16;CHARENTE;1
16144;;Fouqueure;FOUQUEURE;16140;16;CHARENTE;1
16145;;Foussignac;FOUSSIGNAC;16200;16;CHARENTE;1
16146;;Garat;GARAT;16410;16;CHARENTE;1
16147;;Gardes-le-Pontaroux;GARDES-LE-PONTAROUX;16320;16;CHARENTE;1
16148;;Genac;GENAC;16170;16;CHARENTE;1
16149;;Genouillac;GENOUILLAC;16270;16;CHARENTE;1
16150;;Gensac-la-Pallue;GENSAC-LA-PALLUE;16130;16;CHARENTE;1
16151;;Gent;GENTE;16130;16;CHARENTE;1
16152;;Gimeux;GIMEUX;16130;16;CHARENTE;1
16153;;Gondeville;GONDEVILLE;16200;16;CHARENTE;1
16154;;Gond-Pontouvre;GOND-PONTOUVRE;16160;16;CHARENTE;1
16155;Les;Gours;GOURS;16140;16;CHARENTE;1
16156;;Gourville;GOURVILLE;16170;16;CHARENTE;1
16157;Le;Grand-Madieu;GRAND-MADIEU;16450;16;CHARENTE;1
16158;;Grassac;GRASSAC;16380;16;CHARENTE;1
16160;;Guimps;GUIMPS;16300;16;CHARENTE;1
16161;;Guizengeard;GUIZENGEARD;16480;16;CHARENTE;1
16162;;Gurat;GURAT;16320;16;CHARENTE;1
16163;;Hiersac;HIERSAC;16290;16;CHARENTE;1
16164;;Hiesse;HIESSE;16490;16;CHARENTE;1
16165;;Houlette;HOULETTE;16200;16;CHARENTE;1
16166;L';Isle-d'Espagnac;ISLE-D'ESPAGNAC;16340;16;CHARENTE;1
16167;;Jarnac;JARNAC;16200;16;CHARENTE;1
16168;;Jauldes;JAULDES;16560;16;CHARENTE;1
16169;;Javrezac;JAVREZAC;16100;16;CHARENTE;1
16170;;Juignac;JUIGNAC;16190;16;CHARENTE;1
16171;;Juillac-le-Coq;JUILLAC-LE-COQ;16130;16;CHARENTE;1
16172;;Juillaguet;JUILLAGUET;16320;16;CHARENTE;1
16173;;Juill;JUILLE;16230;16;CHARENTE;1
16174;;Julienne;JULIENNE;16200;16;CHARENTE;1
16175;;Jurignac;JURIGNAC;16250;16;CHARENTE;1
16176;;Lachaise;LACHAISE;16300;16;CHARENTE;1
16177;;Ladiville;LADIVILLE;16120;16;CHARENTE;1
16178;;Lagarde-sur-le-N;LAGARDE-SUR-LE-NE;16300;16;CHARENTE;1
16179;;Lamrac;LAMERAC;16300;16;CHARENTE;1
16180;;Laprade;LAPRADE;16390;16;CHARENTE;1
16181;;Lessac;LESSAC;16500;16;CHARENTE;1
16182;;Lesterps;LESTERPS;16420;16;CHARENTE;1
16183;;Lsignac-Durand;LESIGNAC-DURAND;16310;16;CHARENTE;1
16184;;Lichres;LICHERES;16460;16;CHARENTE;1
16185;;Lign;LIGNE;16140;16;CHARENTE;1
16186;;Lignires-Sonneville;LIGNIERES-SONNEVILLE;16130;16;CHARENTE;1
16187;;Linars;LINARS;16730;16;CHARENTE;1
16188;Le;Lindois;LINDOIS;16310;16;CHARENTE;1
16189;;Londigny;LONDIGNY;16700;16;CHARENTE;1
16190;;Longr;LONGRE;16240;16;CHARENTE;1
16191;;Lonnes;LONNES;16230;16;CHARENTE;1
16192;;Roumazires-Loubert;ROUMAZIERES-LOUBERT;16270;16;CHARENTE;1
16193;;Louzac-Saint-Andr;LOUZAC-SAINT-ANDRE;16100;16;CHARENTE;1
16194;;Lupsault;LUPSAULT;16140;16;CHARENTE;1
16195;;Lussac;LUSSAC;16450;16;CHARENTE;1
16196;;Lux;LUXE;16230;16;CHARENTE;1
16197;La;Magdeleine;MAGDELEINE;16240;16;CHARENTE;1
16198;;Magnac-Lavalette-Villars;MAGNAC-LAVALETTE-VILLARS;16320;16;CHARENTE;1
16199;;Magnac-sur-Touvre;MAGNAC-SUR-TOUVRE;16600;16;CHARENTE;1
16200;;Maine-de-Boixe;MAINE-DE-BOIXE;16230;16;CHARENTE;1
16201;;Mainfonds;MAINFONDS;16250;16;CHARENTE;1
16202;;Mainxe;MAINXE;16200;16;CHARENTE;1
16203;;Mainzac;MAINZAC;16380;16;CHARENTE;1
16204;;Malaville;MALAVILLE;16120;16;CHARENTE;1
16205;;Manot;MANOT;16500;16;CHARENTE;1
16206;;Mansle;MANSLE;16230;16;CHARENTE;1
16207;;Marcillac-Lanville;MARCILLAC-LANVILLE;16140;16;CHARENTE;1
16208;;Mareuil;MAREUIL;16170;16;CHARENTE;1
16209;;Marillac-le-Franc;MARILLAC-LE-FRANC;16110;16;CHARENTE;1
16210;;Marsac;MARSAC;16570;16;CHARENTE;1
16211;;Marthon;MARTHON;16380;16;CHARENTE;1
16212;;Massignac;MASSIGNAC;16310;16;CHARENTE;1
16213;;Mazerolles;MAZEROLLES;16310;16;CHARENTE;1
16214;;Mazires;MAZIERES;16270;16;CHARENTE;1
16215;;Mdillac;MEDILLAC;16210;16;CHARENTE;1
16216;;Mrignac;MERIGNAC;16200;16;CHARENTE;1
16217;;Merpins;MERPINS;16100;16;CHARENTE;1
16218;;Mesnac;MESNAC;16370;16;CHARENTE;1
16220;Les;Mtairies;METAIRIES;16200;16;CHARENTE;1
16221;;Mons;MONS;16140;16;CHARENTE;1
16222;;Montboyer;MONTBOYER;16620;16;CHARENTE;1
16223;;Montbron;MONTBRON;16220;16;CHARENTE;1
16224;;Montchaude;MONTCHAUDE;16300;16;CHARENTE;1
16225;;Montembuf;MONTEMBOEUF;16310;16;CHARENTE;1
16226;;Montignac-Charente;MONTIGNAC-CHARENTE;16330;16;CHARENTE;1
16227;;Montignac-le-Coq;MONTIGNAC-LE-COQ;16390;16;CHARENTE;1
16228;;Montign;MONTIGNE;16170;16;CHARENTE;1
16229;;Montjean;MONTJEAN;16240;16;CHARENTE;1
16230;;Montmoreau-Saint-Cybard;MONTMOREAU-SAINT-CYBARD;16190;16;CHARENTE;1
16231;;Montrollet;MONTROLLET;16420;16;CHARENTE;1
16232;;Mornac;MORNAC;16600;16;CHARENTE;1
16233;;Mosnac;MOSNAC;16120;16;CHARENTE;1
16234;;Moulidars;MOULIDARS;16290;16;CHARENTE;1
16236;;Mouthiers-sur-Bome;MOUTHIERS-SUR-BOEME;16440;16;CHARENTE;1
16237;;Mouton;MOUTON;16460;16;CHARENTE;1
16238;;Moutonneau;MOUTONNEAU;16460;16;CHARENTE;1
16239;;Mouzon;MOUZON;16310;16;CHARENTE;1
16240;;Nabinaud;NABINAUD;16390;16;CHARENTE;1
16241;;Nanclars;NANCLARS;16230;16;CHARENTE;1
16242;;Nanteuil-en-Valle;NANTEUIL-EN-VALLEE;16700;16;CHARENTE;1
16243;;Nercillac;NERCILLAC;16200;16;CHARENTE;1
16244;;Nersac;NERSAC;16440;16;CHARENTE;1
16245;;Nieuil;NIEUIL;16270;16;CHARENTE;1
16246;;Nonac;NONAC;16190;16;CHARENTE;1
16247;;Nonaville;NONAVILLE;16120;16;CHARENTE;1
16248;;Oradour;ORADOUR;16140;16;CHARENTE;1
16249;;Oradour-Fanais;ORADOUR-FANAIS;16500;16;CHARENTE;1
16250;;Orgedeuil;ORGEDEUIL;16220;16;CHARENTE;1
16251;;Oriolles;ORIOLLES;16480;16;CHARENTE;1
16252;;Orival;ORIVAL;16210;16;CHARENTE;1
16253;;Paizay-Naudouin-Embourie;PAIZAY-NAUDOUIN-EMBOURIE;16240;16;CHARENTE;1
16254;;Palluaud;PALLUAUD;16390;16;CHARENTE;1
16255;;Parzac;PARZAC;16450;16;CHARENTE;1
16256;;Passirac;PASSIRAC;16480;16;CHARENTE;1
16257;;Preuil;PEREUIL;16250;16;CHARENTE;1
16258;;Prignac;PERIGNAC;16250;16;CHARENTE;1
16259;La;Pruse;PERUSE;16270;16;CHARENTE;1
16260;;Pillac;PILLAC;16390;16;CHARENTE;1
16261;Les;Pins;PINS;16260;16;CHARENTE;1
16262;;Plaizac;PLAIZAC;16170;16;CHARENTE;1
16263;;Plassac-Rouffiac;PLASSAC-ROUFFIAC;16250;16;CHARENTE;1
16264;;Pleuville;PLEUVILLE;16490;16;CHARENTE;1
16267;;Poullignac;POULLIGNAC;16190;16;CHARENTE;1
16268;;Poursac;POURSAC;16700;16;CHARENTE;1
16269;;Pranzac;PRANZAC;16110;16;CHARENTE;1
16270;;Pressignac;PRESSIGNAC;16150;16;CHARENTE;1
16271;;Puymoyen;PUYMOYEN;16400;16;CHARENTE;1
16272;;Puyraux;PUYREAUX;16230;16;CHARENTE;1
16273;;Raix;RAIX;16240;16;CHARENTE;1
16274;;Rancogne;RANCOGNE;16110;16;CHARENTE;1
16275;;Ranville-Breuillaud;RANVILLE-BREUILLAUD;16140;16;CHARENTE;1
16276;;Reignac;REIGNAC;16360;16;CHARENTE;1
16277;;Rparsac;REPARSAC;16200;16;CHARENTE;1
16279;;Rioux-Martin;RIOUX-MARTIN;16210;16;CHARENTE;1
16280;;Rivires;RIVIERES;16110;16;CHARENTE;1
16281;La;Rochefoucauld;ROCHEFOUCAULD;16110;16;CHARENTE;1
16282;La;Rochette;ROCHETTE;16110;16;CHARENTE;1
16283;;Ronsenac;RONSENAC;16320;16;CHARENTE;1
16284;;Rouffiac;ROUFFIAC;16210;16;CHARENTE;1
16285;;Rougnac;ROUGNAC;16320;16;CHARENTE;1
16286;;Rouillac;ROUILLAC;16170;16;CHARENTE;1
16287;;Roullet-Saint-Estphe;ROULLET-SAINT-ESTEPHE;16440;16;CHARENTE;1
16289;;Roussines;ROUSSINES;16310;16;CHARENTE;1
16290;;Rouzde;ROUZEDE;16220;16;CHARENTE;1
16291;;Ruelle-sur-Touvre;RUELLE-SUR-TOUVRE;16600;16;CHARENTE;1
16292;;Ruffec;RUFFEC;16700;16;CHARENTE;1
16293;;Saint-Adjutory;SAINT-ADJUTORY;16310;16;CHARENTE;1
16294;;Saint-Amant;SAINT-AMANT;16190;16;CHARENTE;1
16295;;Saint-Amant-de-Boixe;SAINT-AMANT-DE-BOIXE;16330;16;CHARENTE;1
16296;;Saint-Amant-de-Bonnieure;SAINT-AMANT-DE-BONNIEURE;16230;16;CHARENTE;1
16297;;Graves-Saint-Amant;GRAVES-SAINT-AMANT;16120;16;CHARENTE;1
16298;;Saint-Amant-de-Noure;SAINT-AMANT-DE-NOUERE;16170;16;CHARENTE;1
16300;;Saint-Angeau;SAINT-ANGEAU;16230;16;CHARENTE;1
16301;;Saint-Aulais-la-Chapelle;SAINT-AULAIS-LA-CHAPELLE;16300;16;CHARENTE;1
16302;;Saint-Avit;SAINT-AVIT;16210;16;CHARENTE;1
16303;;Saint-Bonnet;SAINT-BONNET;16300;16;CHARENTE;1
16304;;Saint-Brice;SAINT-BRICE;16100;16;CHARENTE;1
16306;;Saint-Christophe;SAINT-CHRISTOPHE;16420;16;CHARENTE;1
16307;;Saint-Ciers-sur-Bonnieure;SAINT-CIERS-SUR-BONNIEURE;16230;16;CHARENTE;1
16308;;Saint-Claud;SAINT-CLAUD;16450;16;CHARENTE;1
16309;;Sainte-Colombe;SAINTE-COLOMBE;16230;16;CHARENTE;1
16310;;Saint-Coutant;SAINT-COUTANT;16350;16;CHARENTE;1
16312;;Saint-Cybardeaux;SAINT-CYBARDEAUX;16170;16;CHARENTE;1
16314;;Saint-Eutrope;SAINT-EUTROPE;16190;16;CHARENTE;1
16315;;Saint-Flix;SAINT-FELIX;16480;16;CHARENTE;1
16316;;Saint-Fort-sur-le-N;SAINT-FORT-SUR-LE-NE;16130;16;CHARENTE;1
16317;;Saint-Fraigne;SAINT-FRAIGNE;16140;16;CHARENTE;1
16318;;Saint-Front;SAINT-FRONT;16460;16;CHARENTE;1
16320;;Saint-Genis-d'Hiersac;SAINT-GENIS-D'HIERSAC;16570;16;CHARENTE;1
16321;;Saint-Georges;SAINT-GEORGES;16700;16;CHARENTE;1
16322;;Saint-Germain-de-Confolens;SAINT-GERMAIN-DE-CONFOLENS;16500;16;CHARENTE;1
16323;;Saint-Germain-de-Montbron;SAINT-GERMAIN-DE-MONTBRON;16380;16;CHARENTE;1
16325;;Saint-Gourson;SAINT-GOURSON;16700;16;CHARENTE;1
16326;;Saint-Groux;SAINT-GROUX;16230;16;CHARENTE;1
16328;;Saint-Laurent-de-Belzagot;SAINT-LAURENT-DE-BELZAGOT;16190;16;CHARENTE;1
16329;;Saint-Laurent-de-Cris;SAINT-LAURENT-DE-CERIS;16450;16;CHARENTE;1
16330;;Saint-Laurent-de-Cognac;SAINT-LAURENT-DE-COGNAC;16100;16;CHARENTE;1
16331;;Saint-Laurent-des-Combes;SAINT-LAURENT-DES-COMBES;16480;16;CHARENTE;1
16332;;Saint-Lger;SAINT-LEGER;16250;16;CHARENTE;1
16334;;Saint-Martial;SAINT-MARTIAL;16190;16;CHARENTE;1
16335;;Saint-Martin-du-Clocher;SAINT-MARTIN-DU-CLOCHER;16700;16;CHARENTE;1
16336;;Saint-Mary;SAINT-MARY;16260;16;CHARENTE;1
16337;;Saint-Maurice-des-Lions;SAINT-MAURICE-DES-LIONS;16500;16;CHARENTE;1
16338;;Saint-Mdard;SAINT-MEDARD;16300;16;CHARENTE;1
16339;;Auge-Saint-Mdard;AUGE-SAINT-MEDARD;16170;16;CHARENTE;1
16340;;Saint-Mme-les-Carrires;SAINT-MEME-LES-CARRIERES;16720;16;CHARENTE;1
16341;;Saint-Michel;SAINT-MICHEL;16470;16;CHARENTE;1
16342;;Saint-Palais-du-N;SAINT-PALAIS-DU-NE;16300;16;CHARENTE;1
16343;;Saint-Preuil;SAINT-PREUIL;16130;16;CHARENTE;1
16344;;Saint-Projet-Saint-Constant;SAINT-PROJET-SAINT-CONSTANT;16110;16;CHARENTE;1
16345;;Saint-Quentin-sur-Charente;SAINT-QUENTIN-SUR-CHARENTE;16150;16;CHARENTE;1
16346;;Saint-Quentin-de-Chalais;SAINT-QUENTIN-DE-CHALAIS;16210;16;CHARENTE;1
16347;;Saint-Romain;SAINT-ROMAIN;16210;16;CHARENTE;1
16348;;Saint-Saturnin;SAINT-SATURNIN;16290;16;CHARENTE;1
16349;;Sainte-Svre;SAINTE-SEVERE;16200;16;CHARENTE;1
16350;;Saint-Sverin;SAINT-SEVERIN;16390;16;CHARENTE;1
16351;;Saint-Simeux;SAINT-SIMEUX;16120;16;CHARENTE;1
16352;;Saint-Simon;SAINT-SIMON;16120;16;CHARENTE;1
16353;;Saint-Sornin;SAINT-SORNIN;16220;16;CHARENTE;1
16354;;Sainte-Souline;SAINTE-SOULINE;16480;16;CHARENTE;1
16355;;Saint-Sulpice-de-Cognac;SAINT-SULPICE-DE-COGNAC;16370;16;CHARENTE;1
16356;;Saint-Sulpice-de-Ruffec;SAINT-SULPICE-DE-RUFFEC;16460;16;CHARENTE;1
16357;;Saint-Vallier;SAINT-VALLIER;16480;16;CHARENTE;1
16358;;Saint-Yrieix-sur-Charente;SAINT-YRIEIX-SUR-CHARENTE;16710;16;CHARENTE;1
16359;;Salles-d'Angles;SALLES-D'ANGLES;16130;16;CHARENTE;1
16360;;Salles-de-Barbezieux;SALLES-DE-BARBEZIEUX;16300;16;CHARENTE;1
16361;;Salles-de-Villefagnan;SALLES-DE-VILLEFAGNAN;16700;16;CHARENTE;1
16362;;Salles-Lavalette;SALLES-LAVALETTE;16190;16;CHARENTE;1
16363;;Saulgond;SAULGOND;16420;16;CHARENTE;1
16364;;Sauvagnac;SAUVAGNAC;16310;16;CHARENTE;1
16365;;Sauvignac;SAUVIGNAC;16480;16;CHARENTE;1
16366;;Segonzac;SEGONZAC;16130;16;CHARENTE;1
16368;;Sers;SERS;16410;16;CHARENTE;1
16369;;Sigogne;SIGOGNE;16200;16;CHARENTE;1
16370;;Sireuil;SIREUIL;16440;16;CHARENTE;1
16371;;Sonneville;SONNEVILLE;16170;16;CHARENTE;1
16372;;Souffrignac;SOUFFRIGNAC;16380;16;CHARENTE;1
16373;;Souvign;SOUVIGNE;16240;16;CHARENTE;1
16374;;Soyaux;SOYAUX;16800;16;CHARENTE;1
16375;;Suaux;SUAUX;16260;16;CHARENTE;1
16376;;Suris;SURIS;16270;16;CHARENTE;1
16377;La;Tche;TACHE;16260;16;CHARENTE;1
16378;;Taiz-Aizie;TAIZE-AIZIE;16700;16;CHARENTE;1
16379;;Taponnat-Fleurignac;TAPONNAT-FLEURIGNAC;16110;16;CHARENTE;1
16380;Le;Ttre;TATRE;16360;16;CHARENTE;1
16381;;Theil-Rabier;THEIL-RABIER;16240;16;CHARENTE;1
16382;;Torsac;TORSAC;16410;16;CHARENTE;1
16383;;Tourriers;TOURRIERS;16560;16;CHARENTE;1
16384;;Touvrac;TOUVERAC;16360;16;CHARENTE;1
16385;;Touvre;TOUVRE;16600;16;CHARENTE;1
16386;;Touzac;TOUZAC;16120;16;CHARENTE;1
16387;;Triac-Lautrait;TRIAC-LAUTRAIT;16200;16;CHARENTE;1
16388;;Trois-Palis;TROIS-PALIS;16730;16;CHARENTE;1
16389;;Turgon;TURGON;16350;16;CHARENTE;1
16390;;Tusson;TUSSON;16140;16;CHARENTE;1
16391;;Tuzie;TUZIE;16700;16;CHARENTE;1
16392;;Valence;VALENCE;16460;16;CHARENTE;1
16393;;Vars;VARS;16330;16;CHARENTE;1
16394;;Vaux-Lavalette;VAUX-LAVALETTE;16320;16;CHARENTE;1
16395;;Vaux-Rouillac;VAUX-ROUILLAC;16170;16;CHARENTE;1
16396;;Ventouse;VENTOUSE;16460;16;CHARENTE;1
16397;;Verdille;VERDILLE;16140;16;CHARENTE;1
16398;;Verneuil;VERNEUIL;16310;16;CHARENTE;1
16399;;Verrires;VERRIERES;16130;16;CHARENTE;1
16400;;Verteuil-sur-Charente;VERTEUIL-SUR-CHARENTE;16510;16;CHARENTE;1
16401;;Vervant;VERVANT;16330;16;CHARENTE;1
16402;;Vibrac;VIBRAC;16120;16;CHARENTE;1
16403;Le;Vieux-Crier;VIEUX-CERIER;16350;16;CHARENTE;1
16404;;Vieux-Ruffec;VIEUX-RUFFEC;16350;16;CHARENTE;1
16405;;Vignolles;VIGNOLLES;16300;16;CHARENTE;1
16406;;Vilhonneur;VILHONNEUR;16220;16;CHARENTE;1
16408;;Villebois-Lavalette;VILLEBOIS-LAVALETTE;16320;16;CHARENTE;1
16409;;Villefagnan;VILLEFAGNAN;16240;16;CHARENTE;1
16410;;Villegats;VILLEGATS;16700;16;CHARENTE;1
16411;;Villejsus;VILLEJESUS;16140;16;CHARENTE;1
16412;;Villejoubert;VILLEJOUBERT;16560;16;CHARENTE;1
16413;;Villiers-le-Roux;VILLIERS-LE-ROUX;16240;16;CHARENTE;1
16414;;Villognon;VILLOGNON;16230;16;CHARENTE;1
16415;;Vindelle;VINDELLE;16430;16;CHARENTE;1
16416;;Vitrac-Saint-Vincent;VITRAC-SAINT-VINCENT;16310;16;CHARENTE;1
16417;;Viville;VIVILLE;16120;16;CHARENTE;1
16418;;Vuil-et-Giget;VOEUIL-ET-GIGET;16400;16;CHARENTE;1
16419;;Vouharte;VOUHARTE;16330;16;CHARENTE;1
16420;;Voulgzac;VOULGEZAC;16250;16;CHARENTE;1
16421;;Vouthon;VOUTHON;16220;16;CHARENTE;1
16422;;Vouzan;VOUZAN;16410;16;CHARENTE;1
16423;;Xambes;XAMBES;16330;16;CHARENTE;1
16424;;Yviers;YVIERS;16210;16;CHARENTE;1
16425;;Yvrac-et-Malleyrand;YVRAC-ET-MALLEYRAND;16110;16;CHARENTE;1
17002;;Agudelle;AGUDELLE;17500;17;CHARENTE MARITIME;1
17003;;Aigrefeuille-d'Aunis;AIGREFEUILLE-D'AUNIS;17290;17;CHARENTE MARITIME;1
17004;;le-d'Aix;ILE-D'AIX;17123;17;CHARENTE MARITIME;1
17005;;Allas-Bocage;ALLAS-BOCAGE;17150;17;CHARENTE MARITIME;1
17006;;Allas-Champagne;ALLAS-CHAMPAGNE;17500;17;CHARENTE MARITIME;1
17007;;Anais;ANAIS;17540;17;CHARENTE MARITIME;1
17008;;Andilly;ANDILLY;17230;17;CHARENTE MARITIME;1
17009;;Angliers;ANGLIERS;17540;17;CHARENTE MARITIME;1
17010;;Angoulins;ANGOULINS;17690;17;CHARENTE MARITIME;1
17011;;Annepont;ANNEPONT;17350;17;CHARENTE MARITIME;1
17012;;Annezay;ANNEZAY;17380;17;CHARENTE MARITIME;1
17013;;Antezant-la-Chapelle;ANTEZANT-LA-CHAPELLE;17400;17;CHARENTE MARITIME;1
17015;;Arces;ARCES;17120;17;CHARENTE MARITIME;1
17016;;Archiac;ARCHIAC;17520;17;CHARENTE MARITIME;1
17017;;Archingeay;ARCHINGEAY;17380;17;CHARENTE MARITIME;1
17018;;Ardillires;ARDILLIERES;17290;17;CHARENTE MARITIME;1
17019;;Ars-en-R;ARS-EN-RE;17590;17;CHARENTE MARITIME;1
17020;;Arthenac;ARTHENAC;17520;17;CHARENTE MARITIME;1
17021;;Arvert;ARVERT;17530;17;CHARENTE MARITIME;1
17022;;Asnires-la-Giraud;ASNIERES-LA-GIRAUD;17400;17;CHARENTE MARITIME;1
17023;;Aujac;AUJAC;17770;17;CHARENTE MARITIME;1
17024;;Aulnay;AULNAY;17470;17;CHARENTE MARITIME;1
17025;;Aumagne;AUMAGNE;17770;17;CHARENTE MARITIME;1
17026;;Authon-bon;AUTHON-EBEON;17770;17;CHARENTE MARITIME;1
17027;;Avy;AVY;17800;17;CHARENTE MARITIME;1
17028;;Aytr;AYTRE;17440;17;CHARENTE MARITIME;1
17029;;Bagnizeau;BAGNIZEAU;17160;17;CHARENTE MARITIME;1
17030;;Balanzac;BALANZAC;17600;17;CHARENTE MARITIME;1
17031;;Ballans;BALLANS;17160;17;CHARENTE MARITIME;1
17032;;Ballon;BALLON;17290;17;CHARENTE MARITIME;1
17033;La;Barde;BARDE;17360;17;CHARENTE MARITIME;1
17034;;Barzan;BARZAN;17120;17;CHARENTE MARITIME;1
17035;;Bazauges;BAZAUGES;17490;17;CHARENTE MARITIME;1
17036;;Beaugeay;BEAUGEAY;17620;17;CHARENTE MARITIME;1
17037;;Beauvais-sur-Matha;BEAUVAIS-SUR-MATHA;17490;17;CHARENTE MARITIME;1
17038;;Bedenac;BEDENAC;17210;17;CHARENTE MARITIME;1
17039;;Belluire;BELLUIRE;17800;17;CHARENTE MARITIME;1
17040;La;Bente;BENATE;17400;17;CHARENTE MARITIME;1
17041;;Benon;BENON;17170;17;CHARENTE MARITIME;1
17042;;Bercloux;BERCLOUX;17770;17;CHARENTE MARITIME;1
17043;;Bernay-Saint-Martin;BERNAY-SAINT-MARTIN;17330;17;CHARENTE MARITIME;1
17044;;Berneuil;BERNEUIL;17460;17;CHARENTE MARITIME;1
17045;;Beurlay;BEURLAY;17250;17;CHARENTE MARITIME;1
17046;;Bignay;BIGNAY;17400;17;CHARENTE MARITIME;1
17047;;Biron;BIRON;17800;17;CHARENTE MARITIME;1
17048;;Blanzac-ls-Matha;BLANZAC-LES-MATHA;17160;17;CHARENTE MARITIME;1
17049;;Blanzay-sur-Boutonne;BLANZAY-SUR-BOUTONNE;17470;17;CHARENTE MARITIME;1
17050;;Bois;BOIS;17240;17;CHARENTE MARITIME;1
17051;Le;Bois-Plage-en-R;BOIS-PLAGE-EN-RE;17580;17;CHARENTE MARITIME;1
17052;;Boisredon;BOISREDON;17150;17;CHARENTE MARITIME;1
17053;;Bords;BORDS;17430;17;CHARENTE MARITIME;1
17054;;Boresse-et-Martron;BORESSE-ET-MARTRON;17270;17;CHARENTE MARITIME;1
17055;;Boscamnant;BOSCAMNANT;17360;17;CHARENTE MARITIME;1
17056;;Bougneau;BOUGNEAU;17800;17;CHARENTE MARITIME;1
17057;;Bouhet;BOUHET;17540;17;CHARENTE MARITIME;1
17058;;Bourcefranc-le-Chapus;BOURCEFRANC-LE-CHAPUS;17560;17;CHARENTE MARITIME;1
17059;;Bourgneuf;BOURGNEUF;17220;17;CHARENTE MARITIME;1
17060;;Boutenac-Touvent;BOUTENAC-TOUVENT;17120;17;CHARENTE MARITIME;1
17061;;Bran;BRAN;17210;17;CHARENTE MARITIME;1
17062;;Bresdon;BRESDON;17490;17;CHARENTE MARITIME;1
17063;;Breuil-la-Rorte;BREUIL-LA-REORTE;17700;17;CHARENTE MARITIME;1
17064;;Breuillet;BREUILLET;17920;17;CHARENTE MARITIME;1
17065;;Breuil-Magn;BREUIL-MAGNE;17870;17;CHARENTE MARITIME;1
17066;;Brie-sous-Archiac;BRIE-SOUS-ARCHIAC;17520;17;CHARENTE MARITIME;1
17067;;Brie-sous-Matha;BRIE-SOUS-MATHA;17160;17;CHARENTE MARITIME;1
17068;;Brie-sous-Mortagne;BRIE-SOUS-MORTAGNE;17120;17;CHARENTE MARITIME;1
17069;;Brives-sur-Charente;BRIVES-SUR-CHARENTE;17800;17;CHARENTE MARITIME;1
17070;;Brizambourg;BRIZAMBOURG;17770;17;CHARENTE MARITIME;1
17071;La;Brousse;BROUSSE;17160;17;CHARENTE MARITIME;1
17072;;Burie;BURIE;17770;17;CHARENTE MARITIME;1
17073;;Bussac-sur-Charente;BUSSAC-SUR-CHARENTE;17100;17;CHARENTE MARITIME;1
17074;;Bussac-Fort;BUSSAC-FORET;17210;17;CHARENTE MARITIME;1
17075;;Cabariot;CABARIOT;17430;17;CHARENTE MARITIME;1
17076;;Celles;CELLES;17520;17;CHARENTE MARITIME;1
17077;;Cercoux;CERCOUX;17270;17;CHARENTE MARITIME;1
17078;;Chadenac;CHADENAC;17800;17;CHARENTE MARITIME;1
17079;;Chaillevette;CHAILLEVETTE;17890;17;CHARENTE MARITIME;1
17080;;Chambon;CHAMBON;17290;17;CHARENTE MARITIME;1
17081;;Chamouillac;CHAMOUILLAC;17130;17;CHARENTE MARITIME;1
17082;;Champagnac;CHAMPAGNAC;17500;17;CHARENTE MARITIME;1
17083;;Champagne;CHAMPAGNE;17620;17;CHARENTE MARITIME;1
17084;;Champagnolles;CHAMPAGNOLLES;17240;17;CHARENTE MARITIME;1
17085;;Champdolent;CHAMPDOLENT;17430;17;CHARENTE MARITIME;1
17086;;Chaniers;CHANIERS;17610;17;CHARENTE MARITIME;1
17087;;Chantemerle-sur-la-Soie;CHANTEMERLE-SUR-LA-SOIE;17380;17;CHARENTE MARITIME;1
17089;La;Chapelle-des-Pots;CHAPELLE-DES-POTS;17100;17;CHARENTE MARITIME;1
17091;;Charron;CHARRON;17230;17;CHARENTE MARITIME;1
17092;;Chartuzac;CHARTUZAC;17130;17;CHARENTE MARITIME;1
17093;Le;Chteau-d'Olron;CHATEAU-D'OLERON;17480;17;CHARENTE MARITIME;1
17094;;Chtelaillon-Plage;CHATELAILLON-PLAGE;17340;17;CHARENTE MARITIME;1
17095;;Chatenet;CHATENET;17210;17;CHARENTE MARITIME;1
17096;;Chaunac;CHAUNAC;17130;17;CHARENTE MARITIME;1
17097;Le;Chay;CHAY;17600;17;CHARENTE MARITIME;1
17098;;Chenac-Saint-Seurin-d'Uzet;CHENAC-SAINT-SEURIN-D'UZET;17120;17;CHARENTE MARITIME;1
17099;;Chepniers;CHEPNIERS;17210;17;CHARENTE MARITIME;1
17100;;Chrac;CHERAC;17610;17;CHARENTE MARITIME;1
17101;;Cherbonnires;CHERBONNIERES;17470;17;CHARENTE MARITIME;1
17102;;Chermignac;CHERMIGNAC;17460;17;CHARENTE MARITIME;1
17103;;Chervettes;CHERVETTES;17380;17;CHARENTE MARITIME;1
17104;;Chevanceaux;CHEVANCEAUX;17210;17;CHARENTE MARITIME;1
17105;;Chives;CHIVES;17510;17;CHARENTE MARITIME;1
17106;;Cierzac;CIERZAC;17520;17;CHARENTE MARITIME;1
17107;;Cir-d'Aunis;CIRE-D'AUNIS;17290;17;CHARENTE MARITIME;1
17108;;Clam;CLAM;17500;17;CHARENTE MARITIME;1
17109;;Clavette;CLAVETTE;17220;17;CHARENTE MARITIME;1
17110;;Clrac;CLERAC;17270;17;CHARENTE MARITIME;1
17111;;Clion;CLION;17240;17;CHARENTE MARITIME;1
17112;La;Clisse;CLISSE;17600;17;CHARENTE MARITIME;1
17113;La;Clotte;CLOTTE;17360;17;CHARENTE MARITIME;1
17114;;Coivert;COIVERT;17330;17;CHARENTE MARITIME;1
17115;;Colombiers;COLOMBIERS;17460;17;CHARENTE MARITIME;1
17116;;Consac;CONSAC;17150;17;CHARENTE MARITIME;1
17117;;Contr;CONTRE;17470;17;CHARENTE MARITIME;1
17118;;Corignac;CORIGNAC;17130;17;CHARENTE MARITIME;1
17119;;Corme-cluse;CORME-ECLUSE;17600;17;CHARENTE MARITIME;1
17120;;Corme-Royal;CORME-ROYAL;17600;17;CHARENTE MARITIME;1
17121;La;Couarde-sur-Mer;COUARDE-SUR-MER;17670;17;CHARENTE MARITIME;1
17122;;Coulonges;COULONGES;17800;17;CHARENTE MARITIME;1
17124;;Courant;COURANT;17330;17;CHARENTE MARITIME;1
17125;;Courcelles;COURCELLES;17400;17;CHARENTE MARITIME;1
17126;;Courcerac;COURCERAC;17160;17;CHARENTE MARITIME;1
17127;;Couron;COURCON;17170;17;CHARENTE MARITIME;1
17128;;Courcoury;COURCOURY;17100;17;CHARENTE MARITIME;1
17129;;Courpignac;COURPIGNAC;17130;17;CHARENTE MARITIME;1
17130;;Coux;COUX;17130;17;CHARENTE MARITIME;1
17131;;Cozes;COZES;17120;17;CHARENTE MARITIME;1
17132;;Cramchaban;CRAMCHABAN;17170;17;CHARENTE MARITIME;1
17133;;Cravans;CRAVANS;17260;17;CHARENTE MARITIME;1
17134;;Crazannes;CRAZANNES;17350;17;CHARENTE MARITIME;1
17135;;Cress;CRESSE;17160;17;CHARENTE MARITIME;1
17136;;Croix-Chapeau;CROIX-CHAPEAU;17220;17;CHARENTE MARITIME;1
17137;La;Croix-Comtesse;CROIX-COMTESSE;17330;17;CHARENTE MARITIME;1
17138;;Dampierre-sur-Boutonne;DAMPIERRE-SUR-BOUTONNE;17470;17;CHARENTE MARITIME;1
17139;;Duil-sur-le-Mignon;DOEUIL-SUR-LE-MIGNON;17330;17;CHARENTE MARITIME;1
17140;;Dolus-d'Olron;DOLUS-D'OLERON;17550;17;CHARENTE MARITIME;1
17141;;Dompierre-sur-Charente;DOMPIERRE-SUR-CHARENTE;17610;17;CHARENTE MARITIME;1
17142;;Dompierre-sur-Mer;DOMPIERRE-SUR-MER;17139;17;CHARENTE MARITIME;1
17143;Le;Douhet;DOUHET;17100;17;CHARENTE MARITIME;1
17145;;chebrune;ECHEBRUNE;17800;17;CHARENTE MARITIME;1
17146;;chillais;ECHILLAIS;17620;17;CHARENTE MARITIME;1
17147;;coyeux;ECOYEUX;17770;17;CHARENTE MARITIME;1
17148;;curat;ECURAT;17810;17;CHARENTE MARITIME;1
17149;Les;duts;EDUTS;17510;17;CHARENTE MARITIME;1
17150;Les;glises-d'Argenteuil;EGLISES-D'ARGENTEUIL;17400;17;CHARENTE MARITIME;1
17151;L';guille;EGUILLE;17600;17;CHARENTE MARITIME;1
17152;;pargnes;EPARGNES;17120;17;CHARENTE MARITIME;1
17153;;Esnandes;ESNANDES;17137;17;CHARENTE MARITIME;1
17154;Les;Essards;ESSARDS;17250;17;CHARENTE MARITIME;1
17155;;taules;ETAULES;17750;17;CHARENTE MARITIME;1
17156;;Expiremont;EXPIREMONT;17130;17;CHARENTE MARITIME;1
17157;;Fenioux;FENIOUX;17350;17;CHARENTE MARITIME;1
17158;;Ferrires;FERRIERES;17170;17;CHARENTE MARITIME;1
17159;;Flac-sur-Seugne;FLEAC-SUR-SEUGNE;17800;17;CHARENTE MARITIME;1
17160;;Floirac;FLOIRAC;17120;17;CHARENTE MARITIME;1
17161;La;Flotte;FLOTTE;17630;17;CHARENTE MARITIME;1
17162;;Fontaine-Chalendray;FONTAINE-CHALENDRAY;17510;17;CHARENTE MARITIME;1
17163;;Fontaines-d'Ozillac;FONTAINES-D'OZILLAC;17500;17;CHARENTE MARITIME;1
17164;;Fontcouverte;FONTCOUVERTE;17100;17;CHARENTE MARITIME;1
17165;;Fontenet;FONTENET;17400;17;CHARENTE MARITIME;1
17166;;Forges;FORGES;17290;17;CHARENTE MARITIME;1
17167;Le;Fouilloux;FOUILLOUX;17270;17;CHARENTE MARITIME;1
17168;;Fouras;FOURAS;17450;17;CHARENTE MARITIME;1
17169;La;Frdire;FREDIERE;17770;17;CHARENTE MARITIME;1
17171;;Geay;GEAY;17250;17;CHARENTE MARITIME;1
17172;;Gmozac;GEMOZAC;17260;17;CHARENTE MARITIME;1
17173;La;Gentouze;GENETOUZE;17360;17;CHARENTE MARITIME;1
17174;;Genouill;GENOUILLE;17430;17;CHARENTE MARITIME;1
17175;;Germignac;GERMIGNAC;17520;17;CHARENTE MARITIME;1
17176;;Gibourne;GIBOURNE;17160;17;CHARENTE MARITIME;1
17177;Le;Gicq;GICQ;17160;17;CHARENTE MARITIME;1
17178;;Givrezac;GIVREZAC;17260;17;CHARENTE MARITIME;1
17179;Les;Gonds;GONDS;17100;17;CHARENTE MARITIME;1
17180;;Gourvillette;GOURVILLETTE;17490;17;CHARENTE MARITIME;1
17181;;Grandjean;GRANDJEAN;17350;17;CHARENTE MARITIME;1
17182;La;Grve-sur-Mignon;GREVE-SUR-MIGNON;17170;17;CHARENTE MARITIME;1
17183;;Grzac;GREZAC;17120;17;CHARENTE MARITIME;1
17184;La;Gripperie-Saint-Symphorien;GRIPPERIE-SAINT-SYMPHORIEN;17620;17;CHARENTE MARITIME;1
17185;Le;Gua;GUA;17600;17;CHARENTE MARITIME;1
17186;Le;Gu-d'Aller;GUE-D'ALLERE;17540;17;CHARENTE MARITIME;1
17187;;Guitinires;GUITINIERES;17500;17;CHARENTE MARITIME;1
17188;;Haimps;HAIMPS;17160;17;CHARENTE MARITIME;1
17189;;Hiers-Brouage;HIERS-BROUAGE;17320;17;CHARENTE MARITIME;1
17190;L';Houmeau;HOUMEAU;17137;17;CHARENTE MARITIME;1
17191;La;Jard;JARD;17460;17;CHARENTE MARITIME;1
17192;;Jarnac-Champagne;JARNAC-CHAMPAGNE;17520;17;CHARENTE MARITIME;1
17193;La;Jarne;JARNE;17220;17;CHARENTE MARITIME;1
17194;La;Jarrie;JARRIE;17220;17;CHARENTE MARITIME;1
17195;La;Jarrie-Audouin;JARRIE-AUDOUIN;17330;17;CHARENTE MARITIME;1
17196;;Jazennes;JAZENNES;17260;17;CHARENTE MARITIME;1
17197;;Jonzac;JONZAC;17500;17;CHARENTE MARITIME;1
17198;;Juicq;JUICQ;17770;17;CHARENTE MARITIME;1
17199;;Jussas;JUSSAS;17130;17;CHARENTE MARITIME;1
17200;;Lagord;LAGORD;17140;17;CHARENTE MARITIME;1
17201;La;Laigne;LAIGNE;17170;17;CHARENTE MARITIME;1
17202;;Landes;LANDES;17380;17;CHARENTE MARITIME;1
17203;;Landrais;LANDRAIS;17290;17;CHARENTE MARITIME;1
17204;;Loville;LEOVILLE;17500;17;CHARENTE MARITIME;1
17205;;Loire-les-Marais;LOIRE-LES-MARAIS;17870;17;CHARENTE MARITIME;1
17206;;Loir-sur-Nie;LOIRE-SUR-NIE;17470;17;CHARENTE MARITIME;1
17207;;Loix;LOIX;17111;17;CHARENTE MARITIME;1
17208;;Longves;LONGEVES;17230;17;CHARENTE MARITIME;1
17209;;Lonzac;LONZAC;17520;17;CHARENTE MARITIME;1
17210;;Lorignac;LORIGNAC;17240;17;CHARENTE MARITIME;1
17211;;Loulay;LOULAY;17330;17;CHARENTE MARITIME;1
17212;;Louzignac;LOUZIGNAC;17160;17;CHARENTE MARITIME;1
17213;;Lozay;LOZAY;17330;17;CHARENTE MARITIME;1
17214;;Luchat;LUCHAT;17600;17;CHARENTE MARITIME;1
17215;;Lussac;LUSSAC;17500;17;CHARENTE MARITIME;1
17216;;Lussant;LUSSANT;17430;17;CHARENTE MARITIME;1
17217;;Macqueville;MACQUEVILLE;17490;17;CHARENTE MARITIME;1
17218;;Marans;MARANS;17230;17;CHARENTE MARITIME;1
17219;;Marennes;MARENNES;17320;17;CHARENTE MARITIME;1
17220;;Marignac;MARIGNAC;17800;17;CHARENTE MARITIME;1
17221;;Marsais;MARSAIS;17700;17;CHARENTE MARITIME;1
17222;;Marsilly;MARSILLY;17137;17;CHARENTE MARITIME;1
17223;;Massac;MASSAC;17490;17;CHARENTE MARITIME;1
17224;;Matha;MATHA;17160;17;CHARENTE MARITIME;1
17225;Les;Mathes;MATHES;17570;17;CHARENTE MARITIME;1
17226;;Mazeray;MAZERAY;17400;17;CHARENTE MARITIME;1
17227;;Mazerolles;MAZEROLLES;17800;17;CHARENTE MARITIME;1
17228;;Mdis;MEDIS;17600;17;CHARENTE MARITIME;1
17229;;Mrignac;MERIGNAC;17210;17;CHARENTE MARITIME;1
17230;;Meschers-sur-Gironde;MESCHERS-SUR-GIRONDE;17132;17;CHARENTE MARITIME;1
17231;;Messac;MESSAC;17130;17;CHARENTE MARITIME;1
17232;;Meursac;MEURSAC;17120;17;CHARENTE MARITIME;1
17233;;Meux;MEUX;17500;17;CHARENTE MARITIME;1
17234;;Migr;MIGRE;17330;17;CHARENTE MARITIME;1
17235;;Migron;MIGRON;17770;17;CHARENTE MARITIME;1
17236;;Mirambeau;MIRAMBEAU;17150;17;CHARENTE MARITIME;1
17237;;Moze;MOEZE;17780;17;CHARENTE MARITIME;1
17238;;Moings;MOINGS;17500;17;CHARENTE MARITIME;1
17239;;Mons;MONS;17160;17;CHARENTE MARITIME;1
17240;;Montendre;MONTENDRE;17130;17;CHARENTE MARITIME;1
17241;;Montguyon;MONTGUYON;17270;17;CHARENTE MARITIME;1
17242;;Montils;MONTILS;17800;17;CHARENTE MARITIME;1
17243;;Montlieu-la-Garde;MONTLIEU-LA-GARDE;17210;17;CHARENTE MARITIME;1
17244;;Montpellier-de-Mdillan;MONTPELLIER-DE-MEDILLAN;17260;17;CHARENTE MARITIME;1
17245;;Montroy;MONTROY;17220;17;CHARENTE MARITIME;1
17246;;Moragne;MORAGNE;17430;17;CHARENTE MARITIME;1
17247;;Mornac-sur-Seudre;MORNAC-SUR-SEUDRE;17113;17;CHARENTE MARITIME;1
17248;;Mortagne-sur-Gironde;MORTAGNE-SUR-GIRONDE;17120;17;CHARENTE MARITIME;1
17249;;Mortiers;MORTIERS;17500;17;CHARENTE MARITIME;1
17250;;Mosnac;MOSNAC;17240;17;CHARENTE MARITIME;1
17252;Le;Mung;MUNG;17350;17;CHARENTE MARITIME;1
17253;;Muron;MURON;17430;17;CHARENTE MARITIME;1
17254;;Nachamps;NACHAMPS;17380;17;CHARENTE MARITIME;1
17255;;Nancras;NANCRAS;17600;17;CHARENTE MARITIME;1
17256;;Nantill;NANTILLE;17770;17;CHARENTE MARITIME;1
17257;;Nr;NERE;17510;17;CHARENTE MARITIME;1
17258;;Neuillac;NEUILLAC;17520;17;CHARENTE MARITIME;1
17259;;Neulles;NEULLES;17500;17;CHARENTE MARITIME;1
17260;;Neuvicq;NEUVICQ;17270;17;CHARENTE MARITIME;1
17261;;Neuvicq-le-Chteau;NEUVICQ-LE-CHATEAU;17490;17;CHARENTE MARITIME;1
17262;;Nieul-ls-Saintes;NIEUL-LES-SAINTES;17810;17;CHARENTE MARITIME;1
17263;;Nieul-le-Virouil;NIEUL-LE-VIROUIL;17150;17;CHARENTE MARITIME;1
17264;;Nieul-sur-Mer;NIEUL-SUR-MER;17137;17;CHARENTE MARITIME;1
17265;;Nieulle-sur-Seudre;NIEULLE-SUR-SEUDRE;17600;17;CHARENTE MARITIME;1
17266;Les;Nouillers;NOUILLERS;17380;17;CHARENTE MARITIME;1
17267;;Nuaill-d'Aunis;NUAILLE-D'AUNIS;17540;17;CHARENTE MARITIME;1
17268;;Nuaill-sur-Boutonne;NUAILLE-SUR-BOUTONNE;17470;17;CHARENTE MARITIME;1
17269;;Orignolles;ORIGNOLLES;17210;17;CHARENTE MARITIME;1
17270;;Ozillac;OZILLAC;17500;17;CHARENTE MARITIME;1
17271;;Paill;PAILLE;17470;17;CHARENTE MARITIME;1
17272;;Pr;PERE;17700;17;CHARENTE MARITIME;1
17273;;Prignac;PERIGNAC;17800;17;CHARENTE MARITIME;1
17274;;Prigny;PERIGNY;17180;17;CHARENTE MARITIME;1
17275;;Pessines;PESSINES;17810;17;CHARENTE MARITIME;1
17276;Le;Pin;PIN;17210;17;CHARENTE MARITIME;1
17277;;Saint-Denis-du-Pin;SAINT-DENIS-DU-PIN;17400;17;CHARENTE MARITIME;1
17278;;Pisany;PISANY;17600;17;CHARENTE MARITIME;1
17279;;Plassac;PLASSAC;17240;17;CHARENTE MARITIME;1
17280;;Plassay;PLASSAY;17250;17;CHARENTE MARITIME;1
17281;;Polignac;POLIGNAC;17210;17;CHARENTE MARITIME;1
17282;;Pommiers-Moulons;POMMIERS-MOULONS;17130;17;CHARENTE MARITIME;1
17283;;Pons;PONS;17800;17;CHARENTE MARITIME;1
17284;;Pont-l'Abb-d'Arnoult;PONT-L'ABBE-D'ARNOULT;17250;17;CHARENTE MARITIME;1
17285;;Port-d'Envaux;PORT-D'ENVAUX;17350;17;CHARENTE MARITIME;1
17286;Les;Portes-en-R;PORTES-EN-RE;17880;17;CHARENTE MARITIME;1
17287;;Pouillac;POUILLAC;17210;17;CHARENTE MARITIME;1
17288;;Poursay-Garnaud;POURSAY-GARNAUD;17400;17;CHARENTE MARITIME;1
17289;;Prguillac;PREGUILLAC;17460;17;CHARENTE MARITIME;1
17290;;Prignac;PRIGNAC;17160;17;CHARENTE MARITIME;1
17291;;Puilboreau;PUILBOREAU;17138;17;CHARENTE MARITIME;1
17292;;Puy-du-Lac;PUY-DU-LAC;17380;17;CHARENTE MARITIME;1
17293;;Puyravault;PUYRAVAULT;17700;17;CHARENTE MARITIME;1
17294;;Puyrolland;PUYROLLAND;17380;17;CHARENTE MARITIME;1
17295;;Raux;REAUX;17500;17;CHARENTE MARITIME;1
17296;;Rtaud;RETAUD;17460;17;CHARENTE MARITIME;1
17297;;Rivedoux-Plage;RIVEDOUX-PLAGE;17940;17;CHARENTE MARITIME;1
17298;;Rioux;RIOUX;17460;17;CHARENTE MARITIME;1
17299;;Rochefort;ROCHEFORT;17300;17;CHARENTE MARITIME;1
17300;La;Rochelle;ROCHELLE;17000;17;CHARENTE MARITIME;1
17301;;Romazires;ROMAZIERES;17510;17;CHARENTE MARITIME;1
17302;;Romegoux;ROMEGOUX;17250;17;CHARENTE MARITIME;1
17303;La;Ronde;RONDE;17170;17;CHARENTE MARITIME;1
17304;;Rouffiac;ROUFFIAC;17800;17;CHARENTE MARITIME;1
17305;;Rouffignac;ROUFFIGNAC;17130;17;CHARENTE MARITIME;1
17306;;Royan;ROYAN;17200;17;CHARENTE MARITIME;1
17307;;Sablonceaux;SABLONCEAUX;17600;17;CHARENTE MARITIME;1
17308;;Saint-Agnant;SAINT-AGNANT;17620;17;CHARENTE MARITIME;1
17309;;Saint-Aigulin;SAINT-AIGULIN;17360;17;CHARENTE MARITIME;1
17310;;Saint-Andr-de-Lidon;SAINT-ANDRE-DE-LIDON;17260;17;CHARENTE MARITIME;1
17311;;Saint-Augustin;SAINT-AUGUSTIN;17570;17;CHARENTE MARITIME;1
17312;;Saint-Bonnet-sur-Gironde;SAINT-BONNET-SUR-GIRONDE;17150;17;CHARENTE MARITIME;1
17313;;Saint-Bris-des-Bois;SAINT-BRIS-DES-BOIS;17770;17;CHARENTE MARITIME;1
17314;;Saint-Csaire;SAINT-CESAIRE;17770;17;CHARENTE MARITIME;1
17315;;Saint-Christophe;SAINT-CHRISTOPHE;17220;17;CHARENTE MARITIME;1
17316;;Saint-Ciers-Champagne;SAINT-CIERS-CHAMPAGNE;17520;17;CHARENTE MARITIME;1
17317;;Saint-Ciers-du-Taillon;SAINT-CIERS-DU-TAILLON;17240;17;CHARENTE MARITIME;1
17318;;Saint-Clment-des-Baleines;SAINT-CLEMENT-DES-BALEINES;17590;17;CHARENTE MARITIME;1
17319;;Sainte-Colombe;SAINTE-COLOMBE;17210;17;CHARENTE MARITIME;1
17320;;Saint-Coutant-le-Grand;SAINT-COUTANT-LE-GRAND;17430;17;CHARENTE MARITIME;1
17321;;Saint-Crpin;SAINT-CREPIN;17380;17;CHARENTE MARITIME;1
17322;;Saint-Cyr-du-Doret;SAINT-CYR-DU-DORET;17170;17;CHARENTE MARITIME;1
17323;;Saint-Denis-d'Olron;SAINT-DENIS-D'OLERON;17650;17;CHARENTE MARITIME;1
17324;;Saint-Dizant-du-Bois;SAINT-DIZANT-DU-BOIS;17150;17;CHARENTE MARITIME;1
17325;;Saint-Dizant-du-Gua;SAINT-DIZANT-DU-GUA;17240;17;CHARENTE MARITIME;1
17326;;Saint-Eugne;SAINT-EUGENE;17520;17;CHARENTE MARITIME;1
17327;;Saint-Flix;SAINT-FELIX;17330;17;CHARENTE MARITIME;1
17328;;Saint-Fort-sur-Gironde;SAINT-FORT-SUR-GIRONDE;17240;17;CHARENTE MARITIME;1
17329;;Saint-Froult;SAINT-FROULT;17780;17;CHARENTE MARITIME;1
17330;;Sainte-Gemme;SAINTE-GEMME;17250;17;CHARENTE MARITIME;1
17331;;Saint-Genis-de-Saintonge;SAINT-GENIS-DE-SAINTONGE;17240;17;CHARENTE MARITIME;1
17332;;Saint-Georges-Antignac;SAINT-GEORGES-ANTIGNAC;17240;17;CHARENTE MARITIME;1
17333;;Saint-Georges-de-Didonne;SAINT-GEORGES-DE-DIDONNE;17110;17;CHARENTE MARITIME;1
17334;;Saint-Georges-de-Longuepierre;SAINT-GEORGES-DE-LONGUEPIERRE;17470;17;CHARENTE MARITIME;1
17335;;Saint-Georges-des-Agots;SAINT-GEORGES-DES-AGOUTS;17150;17;CHARENTE MARITIME;1
17336;;Saint-Georges-des-Coteaux;SAINT-GEORGES-DES-COTEAUX;17810;17;CHARENTE MARITIME;1
17337;;Saint-Georges-d'Olron;SAINT-GEORGES-D'OLERON;17190;17;CHARENTE MARITIME;1
17338;;Saint-Georges-du-Bois;SAINT-GEORGES-DU-BOIS;17700;17;CHARENTE MARITIME;1
17339;;Saint-Germain-de-Lusignan;SAINT-GERMAIN-DE-LUSIGNAN;17500;17;CHARENTE MARITIME;1
17340;;Saint-Germain-de-Marencennes;SAINT-GERMAIN-DE-MARENCENNES;17700;17;CHARENTE MARITIME;1
17341;;Saint-Germain-de-Vibrac;SAINT-GERMAIN-DE-VIBRAC;17500;17;CHARENTE MARITIME;1
17342;;Saint-Germain-du-Seudre;SAINT-GERMAIN-DU-SEUDRE;17240;17;CHARENTE MARITIME;1
17343;;Saint-Grgoire-d'Ardennes;SAINT-GREGOIRE-D'ARDENNES;17240;17;CHARENTE MARITIME;1
17344;;Saint-Hilaire-de-Villefranche;SAINT-HILAIRE-DE-VILLEFRANCHE;17770;17;CHARENTE MARITIME;1
17345;;Saint-Hilaire-du-Bois;SAINT-HILAIRE-DU-BOIS;17500;17;CHARENTE MARITIME;1
17346;;Saint-Hippolyte;SAINT-HIPPOLYTE;17430;17;CHARENTE MARITIME;1
17347;;Saint-Jean-d'Angly;SAINT-JEAN-D'ANGELY;17400;17;CHARENTE MARITIME;1
17348;;Saint-Jean-d'Angle;SAINT-JEAN-D'ANGLE;17620;17;CHARENTE MARITIME;1
17349;;Saint-Jean-de-Liversay;SAINT-JEAN-DE-LIVERSAY;17170;17;CHARENTE MARITIME;1
17350;;Saint-Julien-de-l'Escap;SAINT-JULIEN-DE-L'ESCAP;17400;17;CHARENTE MARITIME;1
17351;;Saint-Just-Luzac;SAINT-JUST-LUZAC;17320;17;CHARENTE MARITIME;1
17352;;Saint-Laurent-de-la-Barrire;SAINT-LAURENT-DE-LA-BARRIERE;17380;17;CHARENTE MARITIME;1
17353;;Saint-Laurent-de-la-Pre;SAINT-LAURENT-DE-LA-PREE;17450;17;CHARENTE MARITIME;1
17354;;Saint-Lger;SAINT-LEGER;17800;17;CHARENTE MARITIME;1
17355;;Sainte-Lheurine;SAINTE-LHEURINE;17520;17;CHARENTE MARITIME;1
17356;;Saint-Loup;SAINT-LOUP;17380;17;CHARENTE MARITIME;1
17357;;Saint-Maigrin;SAINT-MAIGRIN;17520;17;CHARENTE MARITIME;1
17358;;Saint-Mand-sur-Brdoire;SAINT-MANDE-SUR-BREDOIRE;17470;17;CHARENTE MARITIME;1
17359;;Saint-Mard;SAINT-MARD;17700;17;CHARENTE MARITIME;1
17360;;Sainte-Marie-de-R;SAINTE-MARIE-DE-RE;17740;17;CHARENTE MARITIME;1
17361;;Saint-Martial;SAINT-MARTIAL;17330;17;CHARENTE MARITIME;1
17362;;Saint-Martial-de-Mirambeau;SAINT-MARTIAL-DE-MIRAMBEAU;17150;17;CHARENTE MARITIME;1
17363;;Saint-Martial-de-Vitaterne;SAINT-MARTIAL-DE-VITATERNE;17500;17;CHARENTE MARITIME;1
17364;;Saint-Martial-sur-N;SAINT-MARTIAL-SUR-NE;17520;17;CHARENTE MARITIME;1
17365;;Saint-Martin-d'Ary;SAINT-MARTIN-D'ARY;17270;17;CHARENTE MARITIME;1
17366;;Saint-Martin-de-Coux;SAINT-MARTIN-DE-COUX;17360;17;CHARENTE MARITIME;1
17367;;Saint-Martin-de-Juillers;SAINT-MARTIN-DE-JUILLERS;17400;17;CHARENTE MARITIME;1
17369;;Saint-Martin-de-R;SAINT-MARTIN-DE-RE;17410;17;CHARENTE MARITIME;1
17371;;Saint-Maurice-de-Tavernole;SAINT-MAURICE-DE-TAVERNOLE;17500;17;CHARENTE MARITIME;1
17372;;Saint-Mdard;SAINT-MEDARD;17500;17;CHARENTE MARITIME;1
17373;;Saint-Mdard-d'Aunis;SAINT-MEDARD-D'AUNIS;17220;17;CHARENTE MARITIME;1
17374;;Sainte-Mme;SAINTE-MEME;17770;17;CHARENTE MARITIME;1
17375;;Saint-Nazaire-sur-Charente;SAINT-NAZAIRE-SUR-CHARENTE;17780;17;CHARENTE MARITIME;1
17376;;Saint-Ouen-d'Aunis;SAINT-OUEN-D'AUNIS;17230;17;CHARENTE MARITIME;1
17377;;Saint-Ouen;SAINT-OUEN;17490;17;CHARENTE MARITIME;1
17378;;Saint-Palais-de-Ngrignac;SAINT-PALAIS-DE-NEGRIGNAC;17210;17;CHARENTE MARITIME;1
17379;;Saint-Palais-de-Phiolin;SAINT-PALAIS-DE-PHIOLIN;17800;17;CHARENTE MARITIME;1
17380;;Saint-Palais-sur-Mer;SAINT-PALAIS-SUR-MER;17420;17;CHARENTE MARITIME;1
17381;;Saint-Pardoult;SAINT-PARDOULT;17400;17;CHARENTE MARITIME;1
17382;;Saint-Pierre-d'Amilly;SAINT-PIERRE-D'AMILLY;17700;17;CHARENTE MARITIME;1
17383;;Saint-Pierre-de-Juillers;SAINT-PIERRE-DE-JUILLERS;17400;17;CHARENTE MARITIME;1
17384;;Saint-Pierre-de-l'Isle;SAINT-PIERRE-DE-L'ISLE;17330;17;CHARENTE MARITIME;1
17385;;Saint-Pierre-d'Olron;SAINT-PIERRE-D'OLERON;17310;17;CHARENTE MARITIME;1
17386;;Saint-Pierre-du-Palais;SAINT-PIERRE-DU-PALAIS;17270;17;CHARENTE MARITIME;1
17387;;Saint-Porchaire;SAINT-PORCHAIRE;17250;17;CHARENTE MARITIME;1
17388;;Saint-Quantin-de-Rananne;SAINT-QUANTIN-DE-RANCANNE;17800;17;CHARENTE MARITIME;1
17389;;Sainte-Radegonde;SAINTE-RADEGONDE;17250;17;CHARENTE MARITIME;1
17390;;Sainte-Rame;SAINTE-RAMEE;17240;17;CHARENTE MARITIME;1
17391;;Saint-Rogatien;SAINT-ROGATIEN;17220;17;CHARENTE MARITIME;1
17392;;Saint-Romain-sur-Gironde;SAINT-ROMAIN-SUR-GIRONDE;17240;17;CHARENTE MARITIME;1
17393;;Saint-Romain-de-Benet;SAINT-ROMAIN-DE-BENET;17600;17;CHARENTE MARITIME;1
17394;;Saint-Saturnin-du-Bois;SAINT-SATURNIN-DU-BOIS;17700;17;CHARENTE MARITIME;1
17395;;Saint-Sauvant;SAINT-SAUVANT;17610;17;CHARENTE MARITIME;1
17396;;Saint-Sauveur-d'Aunis;SAINT-SAUVEUR-D'AUNIS;17540;17;CHARENTE MARITIME;1
17397;;Saint-Savinien;SAINT-SAVINIEN;17350;17;CHARENTE MARITIME;1
17398;;Saint-Seurin-de-Palenne;SAINT-SEURIN-DE-PALENNE;17800;17;CHARENTE MARITIME;1
17400;;Saint-Sever-de-Saintonge;SAINT-SEVER-DE-SAINTONGE;17800;17;CHARENTE MARITIME;1
17401;;Saint-Sverin-sur-Boutonne;SAINT-SEVERIN-SUR-BOUTONNE;17330;17;CHARENTE MARITIME;1
17402;;Saint-Sigismond-de-Clermont;SAINT-SIGISMOND-DE-CLERMONT;17240;17;CHARENTE MARITIME;1
17403;;Saint-Simon-de-Bordes;SAINT-SIMON-DE-BORDES;17500;17;CHARENTE MARITIME;1
17404;;Saint-Simon-de-Pellouaille;SAINT-SIMON-DE-PELLOUAILLE;17260;17;CHARENTE MARITIME;1
17405;;Saint-Sorlin-de-Conac;SAINT-SORLIN-DE-CONAC;17150;17;CHARENTE MARITIME;1
17406;;Saint-Sornin;SAINT-SORNIN;17600;17;CHARENTE MARITIME;1
17407;;Sainte-Soulle;SAINTE-SOULLE;17220;17;CHARENTE MARITIME;1
17408;;Saint-Sulpice-d'Arnoult;SAINT-SULPICE-D'ARNOULT;17250;17;CHARENTE MARITIME;1
17409;;Saint-Sulpice-de-Royan;SAINT-SULPICE-DE-ROYAN;17200;17;CHARENTE MARITIME;1
17410;;Saint-Thomas-de-Conac;SAINT-THOMAS-DE-CONAC;17150;17;CHARENTE MARITIME;1
17411;;Saint-Trojan-les-Bains;SAINT-TROJAN-LES-BAINS;17370;17;CHARENTE MARITIME;1
17412;;Saint-Vaize;SAINT-VAIZE;17100;17;CHARENTE MARITIME;1
17413;;Saint-Vivien;SAINT-VIVIEN;17220;17;CHARENTE MARITIME;1
17414;;Saint-Xandre;SAINT-XANDRE;17138;17;CHARENTE MARITIME;1
17415;;Saintes;SAINTES;17100;17;CHARENTE MARITIME;1
17416;;Saleignes;SALEIGNES;17510;17;CHARENTE MARITIME;1
17417;;Salignac-de-Mirambeau;SALIGNAC-DE-MIRAMBEAU;17130;17;CHARENTE MARITIME;1
17418;;Salignac-sur-Charente;SALIGNAC-SUR-CHARENTE;17800;17;CHARENTE MARITIME;1
17420;;Salles-sur-Mer;SALLES-SUR-MER;17220;17;CHARENTE MARITIME;1
17421;;Saujon;SAUJON;17600;17;CHARENTE MARITIME;1
17422;;Seign;SEIGNE;17510;17;CHARENTE MARITIME;1
17423;;Semillac;SEMILLAC;17150;17;CHARENTE MARITIME;1
17424;;Semoussac;SEMOUSSAC;17150;17;CHARENTE MARITIME;1
17425;;Semussac;SEMUSSAC;17120;17;CHARENTE MARITIME;1
17426;Le;Seure;SEURE;17770;17;CHARENTE MARITIME;1
17427;;Siecq;SIECQ;17490;17;CHARENTE MARITIME;1
17428;;Sonnac;SONNAC;17160;17;CHARENTE MARITIME;1
17429;;Soubise;SOUBISE;17780;17;CHARENTE MARITIME;1
17430;;Soubran;SOUBRAN;17150;17;CHARENTE MARITIME;1
17431;;Soulignonne;SOULIGNONNE;17250;17;CHARENTE MARITIME;1
17432;;Soumras;SOUMERAS;17130;17;CHARENTE MARITIME;1
17433;;Sousmoulins;SOUSMOULINS;17130;17;CHARENTE MARITIME;1
17434;;Surgres;SURGERES;17700;17;CHARENTE MARITIME;1
17435;;Taillant;TAILLANT;17350;17;CHARENTE MARITIME;1
17436;;Taillebourg;TAILLEBOURG;17350;17;CHARENTE MARITIME;1
17437;;Talmont-sur-Gironde;TALMONT-SUR-GIRONDE;17120;17;CHARENTE MARITIME;1
17438;;Tanzac;TANZAC;17260;17;CHARENTE MARITIME;1
17439;;Taugon;TAUGON;17170;17;CHARENTE MARITIME;1
17440;;Ternant;TERNANT;17400;17;CHARENTE MARITIME;1
17441;;Tesson;TESSON;17460;17;CHARENTE MARITIME;1
17442;;Thaims;THAIMS;17120;17;CHARENTE MARITIME;1
17443;;Thair;THAIRE;17290;17;CHARENTE MARITIME;1
17444;;Thnac;THENAC;17460;17;CHARENTE MARITIME;1
17445;;Thzac;THEZAC;17600;17;CHARENTE MARITIME;1
17446;;Thors;THORS;17160;17;CHARENTE MARITIME;1
17447;Le;Thou;THOU;17290;17;CHARENTE MARITIME;1
17448;;Tonnay-Boutonne;TONNAY-BOUTONNE;17380;17;CHARENTE MARITIME;1
17449;;Tonnay-Charente;TONNAY-CHARENTE;17430;17;CHARENTE MARITIME;1
17450;;Torx;TORXE;17380;17;CHARENTE MARITIME;1
17451;Les;Touches-de-Prigny;TOUCHES-DE-PERIGNY;17160;17;CHARENTE MARITIME;1
17452;La;Tremblade;TREMBLADE;17390;17;CHARENTE MARITIME;1
17453;;Trizay;TRIZAY;17250;17;CHARENTE MARITIME;1
17454;;Tugras-Saint-Maurice;TUGERAS-SAINT-MAURICE;17130;17;CHARENTE MARITIME;1
17455;La;Valle;VALLEE;17250;17;CHARENTE MARITIME;1
17457;;Vandr;VANDRE;17700;17;CHARENTE MARITIME;1
17458;;Vanzac;VANZAC;17500;17;CHARENTE MARITIME;1
17459;;Varaize;VARAIZE;17400;17;CHARENTE MARITIME;1
17460;;Varzay;VARZAY;17460;17;CHARENTE MARITIME;1
17461;;Vaux-sur-Mer;VAUX-SUR-MER;17640;17;CHARENTE MARITIME;1
17462;;Vnrand;VENERAND;17100;17;CHARENTE MARITIME;1
17463;;Vergeroux;VERGEROUX;17300;17;CHARENTE MARITIME;1
17464;;Vergn;VERGNE;17330;17;CHARENTE MARITIME;1
17465;La;Vergne;VERGNE;17400;17;CHARENTE MARITIME;1
17466;;Vrines;VERINES;17540;17;CHARENTE MARITIME;1
17467;;Vervant;VERVANT;17400;17;CHARENTE MARITIME;1
17468;;Vibrac;VIBRAC;17130;17;CHARENTE MARITIME;1
17469;;Villars-en-Pons;VILLARS-EN-PONS;17260;17;CHARENTE MARITIME;1
17470;;Villars-les-Bois;VILLARS-LES-BOIS;17770;17;CHARENTE MARITIME;1
17471;La;Villedieu;VILLEDIEU;17470;17;CHARENTE MARITIME;1
17472;;Villedoux;VILLEDOUX;17230;17;CHARENTE MARITIME;1
17473;;Villemorin;VILLEMORIN;17470;17;CHARENTE MARITIME;1
17474;;Villeneuve-la-Comtesse;VILLENEUVE-LA-COMTESSE;17330;17;CHARENTE MARITIME;1
17476;;Villexavier;VILLEXAVIER;17500;17;CHARENTE MARITIME;1
17477;;Villiers-Couture;VILLIERS-COUTURE;17510;17;CHARENTE MARITIME;1
17478;;Vinax;VINAX;17510;17;CHARENTE MARITIME;1
17479;;Virollet;VIROLLET;17260;17;CHARENTE MARITIME;1
17480;;Virson;VIRSON;17290;17;CHARENTE MARITIME;1
17481;;Voissay;VOISSAY;17400;17;CHARENTE MARITIME;1
17482;;Vouh;VOUHE;17700;17;CHARENTE MARITIME;1
17483;;Yves;YVES;17340;17;CHARENTE MARITIME;1
17484;;Port-des-Barques;PORT-DES-BARQUES;17730;17;CHARENTE MARITIME;1
17485;Le;Grand-Village-Plage;GRAND-VILLAGE-PLAGE;17370;17;CHARENTE MARITIME;1
17486;La;Bre-les-Bains;BREE-LES-BAINS;17840;17;CHARENTE MARITIME;1
18001;;Achres;ACHERES;18250;18;CHER;1
18002;;Ainay-le-Vieil;AINAY-LE-VIEIL;18200;18;CHER;1
18003;Les;Aix-d'Angillon;AIX-D'ANGILLON;18220;18;CHER;1
18004;;Allogny;ALLOGNY;18110;18;CHER;1
18005;;Allouis;ALLOUIS;18500;18;CHER;1
18006;;Annoix;ANNOIX;18340;18;CHER;1
18007;;Apremont-sur-Allier;APREMONT-SUR-ALLIER;18150;18;CHER;1
18008;;Aray;ARCAY;18340;18;CHER;1
18009;;Arcomps;ARCOMPS;18200;18;CHER;1
18010;;Ardenais;ARDENAIS;18170;18;CHER;1
18011;;Argent-sur-Sauldre;ARGENT-SUR-SAULDRE;18410;18;CHER;1
18012;;Argenvires;ARGENVIERES;18140;18;CHER;1
18013;;Arpheuilles;ARPHEUILLES;18200;18;CHER;1
18014;;Assigny;ASSIGNY;18260;18;CHER;1
18015;;Aubigny-sur-Nre;AUBIGNY-SUR-NERE;18700;18;CHER;1
18016;;Aubinges;AUBINGES;18220;18;CHER;1
18017;;Augy-sur-Aubois;AUGY-SUR-AUBOIS;18600;18;CHER;1
18018;;Avord;AVORD;18520;18;CHER;1
18019;;Azy;AZY;18220;18;CHER;1
18020;;Bannay;BANNAY;18300;18;CHER;1
18021;;Bannegon;BANNEGON;18210;18;CHER;1
18022;;Barlieu;BARLIEU;18260;18;CHER;1
18023;;Baugy;BAUGY;18800;18;CHER;1
18024;;Beddes;BEDDES;18370;18;CHER;1
18025;;Beffes;BEFFES;18320;18;CHER;1
18026;;Belleville-sur-Loire;BELLEVILLE-SUR-LOIRE;18240;18;CHER;1
18027;;Bengy-sur-Craon;BENGY-SUR-CRAON;18520;18;CHER;1
18028;;Berry-Bouy;BERRY-BOUY;18500;18;CHER;1
18029;;Bessais-le-Fromental;BESSAIS-LE-FROMENTAL;18210;18;CHER;1
18030;;Blancafort;BLANCAFORT;18410;18;CHER;1
18031;;Blet;BLET;18350;18;CHER;1
18032;;Boulleret;BOULLERET;18240;18;CHER;1
18033;;Bourges;BOURGES;18000;18;CHER;1
18034;;Bouzais;BOUZAIS;18200;18;CHER;1
18035;;Brcy;BRECY;18220;18;CHER;1
18036;;Brinay;BRINAY;18120;18;CHER;1
18037;;Brinon-sur-Sauldre;BRINON-SUR-SAULDRE;18410;18;CHER;1
18038;;Brure-Allichamps;BRUERE-ALLICHAMPS;18200;18;CHER;1
18039;;Bu;BUE;18300;18;CHER;1
18040;;Bussy;BUSSY;18130;18;CHER;1
18041;La;Celette;CELETTE;18360;18;CHER;1
18042;La;Celle;CELLE;18200;18;CHER;1
18043;La;Celle-Cond;CELLE-CONDE;18160;18;CHER;1
18044;;Cerbois;CERBOIS;18120;18;CHER;1
18045;;Chalivoy-Milon;CHALIVOY-MILON;18130;18;CHER;1
18046;;Chambon;CHAMBON;18190;18;CHER;1
18047;La;Chapelle-d'Angillon;CHAPELLE-D'ANGILLON;18380;18;CHER;1
18048;La;Chapelle-Hugon;CHAPELLE-HUGON;18150;18;CHER;1
18049;La;Chapelle-Montlinard;CHAPELLE-MONTLINARD;18140;18;CHER;1
18050;La;Chapelle-Saint-Ursin;CHAPELLE-SAINT-URSIN;18570;18;CHER;1
18051;La;Chapelotte;CHAPELOTTE;18250;18;CHER;1
18052;;Charenton-du-Cher;CHARENTON-DU-CHER;18210;18;CHER;1
18053;;Charentonnay;CHARENTONNAY;18140;18;CHER;1
18054;;Charly;CHARLY;18350;18;CHER;1
18055;;Chrost;CHAROST;18290;18;CHER;1
18056;;Chassy;CHASSY;18800;18;CHER;1
18057;;Chteaumeillant;CHATEAUMEILLANT;18370;18;CHER;1
18058;;Chteauneuf-sur-Cher;CHATEAUNEUF-SUR-CHER;18190;18;CHER;1
18059;Le;Chtelet;CHATELET;18170;18;CHER;1
18060;;Chaumont;CHAUMONT;18350;18;CHER;1
18061;;Chaumoux-Marcilly;CHAUMOUX-MARCILLY;18140;18;CHER;1
18062;Le;Chautay;CHAUTAY;18150;18;CHER;1
18063;;Chavannes;CHAVANNES;18190;18;CHER;1
18064;;Chry;CHERY;18120;18;CHER;1
18065;;Chezal-Benot;CHEZAL-BENOIT;18160;18;CHER;1
18066;;Civray;CIVRAY;18290;18;CHER;1
18067;;Clmont;CLEMONT;18410;18;CHER;1
18068;;Cogny;COGNY;18130;18;CHER;1
18069;;Colombiers;COLOMBIERS;18200;18;CHER;1
18070;;Concressault;CONCRESSAULT;18260;18;CHER;1
18071;;Contres;CONTRES;18130;18;CHER;1
18072;;Cornusse;CORNUSSE;18350;18;CHER;1
18073;;Corquoy;CORQUOY;18190;18;CHER;1
18074;;Couargues;COUARGUES;18300;18;CHER;1
18075;;Cours-les-Barres;COURS-LES-BARRES;18320;18;CHER;1
18076;;Coust;COUST;18210;18;CHER;1
18077;;Couy;COUY;18140;18;CHER;1
18078;;Crzanay-sur-Cher;CREZANCAY-SUR-CHER;18190;18;CHER;1
18079;;Crzancy-en-Sancerre;CREZANCY-EN-SANCERRE;18300;18;CHER;1
18080;;Croisy;CROISY;18350;18;CHER;1
18081;;Crosses;CROSSES;18340;18;CHER;1
18082;;Cuffy;CUFFY;18150;18;CHER;1
18083;;Culan;CULAN;18270;18;CHER;1
18084;;Dampierre-en-Crot;DAMPIERRE-EN-CROT;18260;18;CHER;1
18085;;Dampierre-en-Graay;DAMPIERRE-EN-GRACAY;18310;18;CHER;1
18086;;Drevant;DREVANT;18200;18;CHER;1
18087;;Dun-sur-Auron;DUN-SUR-AURON;18130;18;CHER;1
18088;;Ennordres;ENNORDRES;18380;18;CHER;1
18089;;pineuil-le-Fleuriel;EPINEUIL-LE-FLEURIEL;18360;18;CHER;1
18090;;trchy;ETRECHY;18800;18;CHER;1
18091;;Farges-Allichamps;FARGES-ALLICHAMPS;18200;18;CHER;1
18092;;Farges-en-Septaine;FARGES-EN-SEPTAINE;18800;18;CHER;1
18093;;Faverdines;FAVERDINES;18360;18;CHER;1
18094;;Feux;FEUX;18300;18;CHER;1
18095;;Flavigny;FLAVIGNY;18350;18;CHER;1
18096;;Focy;FOECY;18500;18;CHER;1
18097;;Fussy;FUSSY;18110;18;CHER;1
18098;;Gardefort;GARDEFORT;18300;18;CHER;1
18099;;Garigny;GARIGNY;18140;18;CHER;1
18100;;Genouilly;GENOUILLY;18310;18;CHER;1
18101;;Germigny-l'Exempt;GERMIGNY-L'EXEMPT;18150;18;CHER;1
18102;;Givardon;GIVARDON;18600;18;CHER;1
18103;;Graay;GRACAY;18310;18;CHER;1
18104;;Groises;GROISES;18140;18;CHER;1
18105;;Gron;GRON;18800;18;CHER;1
18106;;Grossouvre;GROSSOUVRE;18600;18;CHER;1
18107;La;Groutte;GROUTTE;18200;18;CHER;1
18108;La;Guerche-sur-l'Aubois;GUERCHE-SUR-L'AUBOIS;18150;18;CHER;1
18109;;Henrichemont;HENRICHEMONT;18250;18;CHER;1
18110;;Herry;HERRY;18140;18;CHER;1
18111;;Humbligny;HUMBLIGNY;18250;18;CHER;1
18112;;Ids-Saint-Roch;IDS-SAINT-ROCH;18170;18;CHER;1
18113;;Ignol;IGNOL;18350;18;CHER;1
18114;;Ineuil;INEUIL;18160;18;CHER;1
18115;;Ivoy-le-Pr;IVOY-LE-PRE;18380;18;CHER;1
18116;;Jalognes;JALOGNES;18300;18;CHER;1
18117;;Jars;JARS;18260;18;CHER;1
18118;;Jouet-sur-l'Aubois;JOUET-SUR-L'AUBOIS;18320;18;CHER;1
18119;;Jussy-Champagne;JUSSY-CHAMPAGNE;18130;18;CHER;1
18120;;Jussy-le-Chaudrier;JUSSY-LE-CHAUDRIER;18140;18;CHER;1
18121;;Lantan;LANTAN;18130;18;CHER;1
18122;;Lapan;LAPAN;18340;18;CHER;1
18123;;Laverdines;LAVERDINES;18800;18;CHER;1
18124;;Lazenay;LAZENAY;18120;18;CHER;1
18125;;Lr;LERE;18240;18;CHER;1
18126;;Levet;LEVET;18340;18;CHER;1
18127;;Lignires;LIGNIERES;18160;18;CHER;1
18128;;Limeux;LIMEUX;18120;18;CHER;1
18129;;Lissay-Lochy;LISSAY-LOCHY;18340;18;CHER;1
18130;;Loye-sur-Arnon;LOYE-SUR-ARNON;18170;18;CHER;1
18131;;Lugny-Bourbonnais;LUGNY-BOURBONNAIS;18350;18;CHER;1
18132;;Lugny-Champagne;LUGNY-CHAMPAGNE;18140;18;CHER;1
18133;;Lunery;LUNERY;18400;18;CHER;1
18134;;Lury-sur-Arnon;LURY-SUR-ARNON;18120;18;CHER;1
18135;;Maisonnais;MAISONNAIS;18170;18;CHER;1
18136;;Marais;MARCAIS;18170;18;CHER;1
18137;;Mareuil-sur-Arnon;MAREUIL-SUR-ARNON;18290;18;CHER;1
18138;;Marmagne;MARMAGNE;18500;18;CHER;1
18139;;Marseilles-ls-Aubigny;MARSEILLES-LES-AUBIGNY;18320;18;CHER;1
18140;;Massay;MASSAY;18120;18;CHER;1
18141;;Mehun-sur-Yvre;MEHUN-SUR-YEVRE;18500;18;CHER;1
18142;;Meillant;MEILLANT;18200;18;CHER;1
18143;;Menetou-Couture;MENETOU-COUTURE;18320;18;CHER;1
18144;;Menetou-Rtel;MENETOU-RATEL;18300;18;CHER;1
18145;;Menetou-Salon;MENETOU-SALON;18510;18;CHER;1
18146;;Mntrol-sous-Sancerre;MENETREOL-SOUS-SANCERRE;18300;18;CHER;1
18147;;Mntrol-sur-Sauldre;MENETREOL-SUR-SAULDRE;18700;18;CHER;1
18148;;Mreau;MEREAU;18120;18;CHER;1
18149;;Mry-s-Bois;MERY-ES-BOIS;18380;18;CHER;1
18150;;Mry-sur-Cher;MERY-SUR-CHER;18100;18;CHER;1
18151;;Montigny;MONTIGNY;18250;18;CHER;1
18152;;Montlouis;MONTLOUIS;18160;18;CHER;1
18153;;Morlac;MORLAC;18170;18;CHER;1
18154;;Mornay-Berry;MORNAY-BERRY;18350;18;CHER;1
18155;;Mornay-sur-Allier;MORNAY-SUR-ALLIER;18600;18;CHER;1
18156;;Morogues;MOROGUES;18220;18;CHER;1
18157;;Morthomiers;MORTHOMIERS;18570;18;CHER;1
18158;;Moulins-sur-Yvre;MOULINS-SUR-YEVRE;18390;18;CHER;1
18159;;Nanay;NANCAY;18330;18;CHER;1
18160;;Nrondes;NERONDES;18350;18;CHER;1
18161;;Neuilly-en-Dun;NEUILLY-EN-DUN;18600;18;CHER;1
18162;;Neuilly-en-Sancerre;NEUILLY-EN-SANCERRE;18250;18;CHER;1
18163;;Neuvy-Deux-Clochers;NEUVY-DEUX-CLOCHERS;18250;18;CHER;1
18164;;Neuvy-le-Barrois;NEUVY-LE-BARROIS;18600;18;CHER;1
18165;;Neuvy-sur-Barangeon;NEUVY-SUR-BARANGEON;18330;18;CHER;1
18166;;Nohant-en-Got;NOHANT-EN-GOUT;18390;18;CHER;1
18167;;Nohant-en-Graay;NOHANT-EN-GRACAY;18310;18;CHER;1
18168;Le;Noyer;NOYER;18260;18;CHER;1
18169;;Nozires;NOZIERES;18200;18;CHER;1
18170;;Oizon;OIZON;18700;18;CHER;1
18171;;Orcenais;ORCENAIS;18200;18;CHER;1
18172;;Orval;ORVAL;18200;18;CHER;1
18173;;Osmery;OSMERY;18130;18;CHER;1
18174;;Osmoy;OSMOY;18390;18;CHER;1
18175;;Ourouer-les-Bourdelins;OUROUER-LES-BOURDELINS;18350;18;CHER;1
18176;;Parassy;PARASSY;18220;18;CHER;1
18177;;Parnay;PARNAY;18130;18;CHER;1
18178;La;Perche;PERCHE;18200;18;CHER;1
18179;;Pigny;PIGNY;18110;18;CHER;1
18180;;Plaimpied-Givaudins;PLAIMPIED-GIVAUDINS;18340;18;CHER;1
18181;;Plou;PLOU;18290;18;CHER;1
18182;;Poisieux;POISIEUX;18290;18;CHER;1
18183;Le;Pondy;PONDY;18210;18;CHER;1
18184;;Prcy;PRECY;18140;18;CHER;1
18185;;Presly;PRESLY;18380;18;CHER;1
18186;;Preuilly;PREUILLY;18120;18;CHER;1
18187;;Prveranges;PREVERANGES;18370;18;CHER;1
18188;;Primelles;PRIMELLES;18400;18;CHER;1
18189;;Quantilly;QUANTILLY;18110;18;CHER;1
18190;;Quincy;QUINCY;18120;18;CHER;1
18191;;Raymond;RAYMOND;18130;18;CHER;1
18192;;Reigny;REIGNY;18270;18;CHER;1
18193;;Rezay;REZAY;18170;18;CHER;1
18194;;Rians;RIANS;18220;18;CHER;1
18195;;Sagonne;SAGONNE;18600;18;CHER;1
18196;;Saint-Aignan-des-Noyers;SAINT-AIGNAN-DES-NOYERS;18600;18;CHER;1
18197;;Saint-Amand-Montrond;SAINT-AMAND-MONTROND;18200;18;CHER;1
18198;;Saint-Ambroix;SAINT-AMBROIX;18290;18;CHER;1
18199;;Saint-Baudel;SAINT-BAUDEL;18160;18;CHER;1
18200;;Saint-Bouize;SAINT-BOUIZE;18300;18;CHER;1
18201;;Saint-Caprais;SAINT-CAPRAIS;18400;18;CHER;1
18202;;Saint-Cols;SAINT-CEOLS;18220;18;CHER;1
18203;;Saint-Christophe-le-Chaudry;SAINT-CHRISTOPHE-LE-CHAUDRY;18270;18;CHER;1
18204;;Saint-Denis-de-Palin;SAINT-DENIS-DE-PALIN;18130;18;CHER;1
18205;;Saint-Doulchard;SAINT-DOULCHARD;18230;18;CHER;1
18206;;Saint-loy-de-Gy;SAINT-ELOY-DE-GY;18110;18;CHER;1
18207;;Saint-Florent-sur-Cher;SAINT-FLORENT-SUR-CHER;18400;18;CHER;1
18208;;Sainte-Gemme-en-Sancerrois;SAINTE-GEMME-EN-SANCERROIS;18240;18;CHER;1
18209;;Saint-Georges-de-Poisieux;SAINT-GEORGES-DE-POISIEUX;18200;18;CHER;1
18210;;Saint-Georges-sur-la-Pre;SAINT-GEORGES-SUR-LA-PREE;18100;18;CHER;1
18211;;Saint-Georges-sur-Moulon;SAINT-GEORGES-SUR-MOULON;18110;18;CHER;1
18212;;Saint-Germain-des-Bois;SAINT-GERMAIN-DES-BOIS;18340;18;CHER;1
18213;;Saint-Germain-du-Puy;SAINT-GERMAIN-DU-PUY;18390;18;CHER;1
18214;;Saint-Hilaire-de-Court;SAINT-HILAIRE-DE-COURT;18100;18;CHER;1
18215;;Saint-Hilaire-de-Gondilly;SAINT-HILAIRE-DE-GONDILLY;18320;18;CHER;1
18216;;Saint-Hilaire-en-Lignires;SAINT-HILAIRE-EN-LIGNIERES;18160;18;CHER;1
18217;;Saint-Jeanvrin;SAINT-JEANVRIN;18370;18;CHER;1
18218;;Saint-Just;SAINT-JUST;18340;18;CHER;1
18219;;Saint-Laurent;SAINT-LAURENT;18330;18;CHER;1
18220;;Saint-Lger-le-Petit;SAINT-LEGER-LE-PETIT;18140;18;CHER;1
18221;;Saint-Loup-des-Chaumes;SAINT-LOUP-DES-CHAUMES;18190;18;CHER;1
18222;;Sainte-Lunaise;SAINTE-LUNAISE;18340;18;CHER;1
18223;;Saint-Martin-d'Auxigny;SAINT-MARTIN-D'AUXIGNY;18110;18;CHER;1
18224;;Saint-Martin-des-Champs;SAINT-MARTIN-DES-CHAMPS;18140;18;CHER;1
18225;;Saint-Maur;SAINT-MAUR;18270;18;CHER;1
18226;;Saint-Michel-de-Volangis;SAINT-MICHEL-DE-VOLANGIS;18390;18;CHER;1
18227;;Sainte-Montaine;SAINTE-MONTAINE;18700;18;CHER;1
18228;;Saint-Outrille;SAINT-OUTRILLE;18310;18;CHER;1
18229;;Saint-Palais;SAINT-PALAIS;18110;18;CHER;1
18230;;Saint-Pierre-les-Bois;SAINT-PIERRE-LES-BOIS;18170;18;CHER;1
18231;;Saint-Pierre-les-tieux;SAINT-PIERRE-LES-ETIEUX;18210;18;CHER;1
18232;;Saint-Priest-la-Marche;SAINT-PRIEST-LA-MARCHE;18370;18;CHER;1
18233;;Saint-Satur;SAINT-SATUR;18300;18;CHER;1
18234;;Saint-Saturnin;SAINT-SATURNIN;18370;18;CHER;1
18235;;Sainte-Solange;SAINTE-SOLANGE;18220;18;CHER;1
18236;;Saint-Symphorien;SAINT-SYMPHORIEN;18190;18;CHER;1
18237;;Sainte-Thorette;SAINTE-THORETTE;18500;18;CHER;1
18238;;Saint-Vitte;SAINT-VITTE;18360;18;CHER;1
18239;;Saligny-le-Vif;SALIGNY-LE-VIF;18800;18;CHER;1
18240;;Sancergues;SANCERGUES;18140;18;CHER;1
18241;;Sancerre;SANCERRE;18300;18;CHER;1
18242;;Sancoins;SANCOINS;18600;18;CHER;1
18243;;Santranges;SANTRANGES;18240;18;CHER;1
18244;;Saugy;SAUGY;18290;18;CHER;1
18245;;Saulzais-le-Potier;SAULZAIS-LE-POTIER;18360;18;CHER;1
18246;;Savigny-en-Sancerre;SAVIGNY-EN-SANCERRE;18240;18;CHER;1
18247;;Savigny-en-Septaine;SAVIGNY-EN-SEPTAINE;18390;18;CHER;1
18248;;Senneay;SENNECAY;18340;18;CHER;1
18249;;Sens-Beaujeu;SENS-BEAUJEU;18300;18;CHER;1
18250;;Serruelles;SERRUELLES;18190;18;CHER;1
18251;;Svry;SEVRY;18140;18;CHER;1
18252;;Sidiailles;SIDIAILLES;18270;18;CHER;1
18253;;Soulangis;SOULANGIS;18220;18;CHER;1
18254;;Soye-en-Septaine;SOYE-EN-SEPTAINE;18340;18;CHER;1
18255;Le;Subdray;SUBDRAY;18570;18;CHER;1
18256;;Subligny;SUBLIGNY;18260;18;CHER;1
18257;;Sury-prs-Lr;SURY-PRES-LERE;18240;18;CHER;1
18258;;Sury-en-Vaux;SURY-EN-VAUX;18300;18;CHER;1
18259;;Sury-s-Bois;SURY-ES-BOIS;18260;18;CHER;1
18260;;Tendron;TENDRON;18350;18;CHER;1
18261;;Thaumiers;THAUMIERS;18210;18;CHER;1
18262;;Thauvenay;THAUVENAY;18300;18;CHER;1
18263;;Thnioux;THENIOUX;18100;18;CHER;1
18264;;Thou;THOU;18260;18;CHER;1
18265;;Torteron;TORTERON;18320;18;CHER;1
18266;;Touchay;TOUCHAY;18160;18;CHER;1
18267;;Trouy;TROUY;18570;18;CHER;1
18268;;Uzay-le-Venon;UZAY-LE-VENON;18190;18;CHER;1
18269;;Vailly-sur-Sauldre;VAILLY-SUR-SAULDRE;18260;18;CHER;1
18270;;Vallenay;VALLENAY;18190;18;CHER;1
18271;;Vasselay;VASSELAY;18110;18;CHER;1
18272;;Veaugues;VEAUGUES;18300;18;CHER;1
18273;;Venesmes;VENESMES;18190;18;CHER;1
18274;;Verdigny;VERDIGNY;18300;18;CHER;1
18275;;Vereaux;VEREAUX;18600;18;CHER;1
18276;;Vernais;VERNAIS;18210;18;CHER;1
18277;;Verneuil;VERNEUIL;18210;18;CHER;1
18278;;Vesdun;VESDUN;18360;18;CHER;1
18279;;Vierzon;VIERZON;18100;18;CHER;1
18280;;Vignoux-sous-les-Aix;VIGNOUX-SOUS-LES-AIX;18110;18;CHER;1
18281;;Vignoux-sur-Barangeon;VIGNOUX-SUR-BARANGEON;18500;18;CHER;1
18282;;Villabon;VILLABON;18800;18;CHER;1
18283;;Villecelin;VILLECELIN;18160;18;CHER;1
18284;;Villegenon;VILLEGENON;18260;18;CHER;1
18285;;Villeneuve-sur-Cher;VILLENEUVE-SUR-CHER;18400;18;CHER;1
18286;;Villequiers;VILLEQUIERS;18800;18;CHER;1
18287;;Vinon;VINON;18300;18;CHER;1
18288;;Vorly;VORLY;18340;18;CHER;1
18289;;Vornay;VORNAY;18130;18;CHER;1
18290;;Vouzeron;VOUZERON;18330;18;CHER;1
19001;;Affieux;AFFIEUX;19260;19;CORREZE;1
19002;;Aix;AIX;19200;19;CORREZE;1
19003;;Albignac;ALBIGNAC;19190;19;CORREZE;1
19004;;Albussac;ALBUSSAC;19380;19;CORREZE;1
19005;;Allassac;ALLASSAC;19240;19;CORREZE;1
19006;;Alleyrat;ALLEYRAT;19200;19;CORREZE;1
19007;;Altillac;ALTILLAC;19120;19;CORREZE;1
19008;;Ambrugeat;AMBRUGEAT;19250;19;CORREZE;1
19009;Les;Angles-sur-Corrze;ANGLES-SUR-CORREZE;19000;19;CORREZE;1
19010;;Argentat;ARGENTAT;19400;19;CORREZE;1
19011;;Arnac-Pompadour;ARNAC-POMPADOUR;19230;19;CORREZE;1
19012;;Astaillac;ASTAILLAC;19120;19;CORREZE;1
19013;;Aubazines;AUBAZINES;19190;19;CORREZE;1
19014;;Auriac;AURIAC;19220;19;CORREZE;1
19015;;Ayen;AYEN;19310;19;CORREZE;1
19016;;Bar;BAR;19800;19;CORREZE;1
19017;;Bassignac-le-Bas;BASSIGNAC-LE-BAS;19430;19;CORREZE;1
19018;;Bassignac-le-Haut;BASSIGNAC-LE-HAUT;19220;19;CORREZE;1
19019;;Beaulieu-sur-Dordogne;BEAULIEU-SUR-DORDOGNE;19120;19;CORREZE;1
19020;;Beaumont;BEAUMONT;19390;19;CORREZE;1
19021;;Bellechassagne;BELLECHASSAGNE;19290;19;CORREZE;1
19022;;Benayes;BENAYES;19510;19;CORREZE;1
19023;;Beynat;BEYNAT;19190;19;CORREZE;1
19024;;Beyssac;BEYSSAC;19230;19;CORREZE;1
19025;;Beyssenac;BEYSSENAC;19230;19;CORREZE;1
19026;;Bilhac;BILHAC;19120;19;CORREZE;1
19027;;Bonnefond;BONNEFOND;19170;19;CORREZE;1
19028;;Bort-les-Orgues;BORT-LES-ORGUES;19110;19;CORREZE;1
19029;;Branceilles;BRANCEILLES;19500;19;CORREZE;1
19030;;Brignac-la-Plaine;BRIGNAC-LA-PLAINE;19310;19;CORREZE;1
19031;;Brive-la-Gaillarde;BRIVE-LA-GAILLARDE;19100;19;CORREZE;1
19032;;Brivezac;BRIVEZAC;19120;19;CORREZE;1
19033;;Bugeat;BUGEAT;19170;19;CORREZE;1
19034;;Camps-Saint-Mathurin-Lobazel;CAMPS-SAINT-MATHURIN-LEOBAZEL;19430;19;CORREZE;1
19035;;Chabrignac;CHABRIGNAC;19350;19;CORREZE;1
19036;;Chamberet;CHAMBERET;19370;19;CORREZE;1
19037;;Chamboulive;CHAMBOULIVE;19450;19;CORREZE;1
19038;;Chameyrat;CHAMEYRAT;19330;19;CORREZE;1
19039;;Champagnac-la-Noaille;CHAMPAGNAC-LA-NOAILLE;19320;19;CORREZE;1
19040;;Champagnac-la-Prune;CHAMPAGNAC-LA-PRUNE;19320;19;CORREZE;1
19041;;Chanac-les-Mines;CHANAC-LES-MINES;19150;19;CORREZE;1
19042;;Chanteix;CHANTEIX;19330;19;CORREZE;1
19043;La;Chapelle-aux-Brocs;CHAPELLE-AUX-BROCS;19360;19;CORREZE;1
19044;La;Chapelle-aux-Saints;CHAPELLE-AUX-SAINTS;19120;19;CORREZE;1
19045;La;Chapelle-Saint-Graud;CHAPELLE-SAINT-GERAUD;19430;19;CORREZE;1
19046;;Chapelle-Spinasse;CHAPELLE-SPINASSE;19300;19;CORREZE;1
19047;;Chartrier-Ferrire;CHARTRIER-FERRIERE;19600;19;CORREZE;1
19048;Le;Chastang;CHASTANG;19190;19;CORREZE;1
19049;;Chasteaux;CHASTEAUX;19600;19;CORREZE;1
19050;;Chauffour-sur-Vell;CHAUFFOUR-SUR-VELL;19500;19;CORREZE;1
19051;;Chaumeil;CHAUMEIL;19390;19;CORREZE;1
19052;;Chavanac;CHAVANAC;19290;19;CORREZE;1
19053;;Chaveroche;CHAVEROCHE;19200;19;CORREZE;1
19054;;Chenailler-Mascheix;CHENAILLER-MASCHEIX;19120;19;CORREZE;1
19055;;Chirac-Bellevue;CHIRAC-BELLEVUE;19160;19;CORREZE;1
19056;;Clergoux;CLERGOUX;19320;19;CORREZE;1
19057;;Collonges-la-Rouge;COLLONGES-LA-ROUGE;19500;19;CORREZE;1
19058;;Combressol;COMBRESSOL;19250;19;CORREZE;1
19059;;Concze;CONCEZE;19350;19;CORREZE;1
19060;;Condat-sur-Ganaveix;CONDAT-SUR-GANAVEIX;19140;19;CORREZE;1
19061;;Cornil;CORNIL;19150;19;CORREZE;1
19062;;Corrze;CORREZE;19800;19;CORREZE;1
19063;;Cosnac;COSNAC;19360;19;CORREZE;1
19064;;Couffy-sur-Sarsonne;COUFFY-SUR-SARSONNE;19340;19;CORREZE;1
19065;;Courteix;COURTEIX;19340;19;CORREZE;1
19066;;Cublac;CUBLAC;19520;19;CORREZE;1
19067;;Curemonte;CUREMONTE;19500;19;CORREZE;1
19068;;Dampniat;DAMPNIAT;19360;19;CORREZE;1
19069;;Darazac;DARAZAC;19220;19;CORREZE;1
19070;;Darnets;DARNETS;19300;19;CORREZE;1
19071;;Davignac;DAVIGNAC;19250;19;CORREZE;1
19072;;Donzenac;DONZENAC;19270;19;CORREZE;1
19073;;gletons;EGLETONS;19300;19;CORREZE;1
19074;L';glise-aux-Bois;EGLISE-AUX-BOIS;19170;19;CORREZE;1
19075;;Espagnac;ESPAGNAC;19150;19;CORREZE;1
19076;;Espartignac;ESPARTIGNAC;19140;19;CORREZE;1
19077;;Estivals;ESTIVALS;19600;19;CORREZE;1
19078;;Estivaux;ESTIVAUX;19410;19;CORREZE;1
19079;;Eyburie;EYBURIE;19140;19;CORREZE;1
19080;;Eygurande;EYGURANDE;19340;19;CORREZE;1
19081;;Eyrein;EYREIN;19800;19;CORREZE;1
19082;;Favars;FAVARS;19330;19;CORREZE;1
19083;;Feyt;FEYT;19340;19;CORREZE;1
19084;;Forgs;FORGES;19380;19;CORREZE;1
19085;;Gimel-les-Cascades;GIMEL-LES-CASCADES;19800;19;CORREZE;1
19086;;Goulles;GOULLES;19430;19;CORREZE;1
19087;;Gourdon-Murat;GOURDON-MURAT;19170;19;CORREZE;1
19088;;Grandsaigne;GRANDSAIGNE;19300;19;CORREZE;1
19089;;Gros-Chastang;GROS-CHASTANG;19320;19;CORREZE;1
19090;;Gumond;GUMOND;19320;19;CORREZE;1
19091;;Hautefage;HAUTEFAGE;19400;19;CORREZE;1
19092;Le;Jardin;JARDIN;19300;19;CORREZE;1
19093;;Jugeals-Nazareth;JUGEALS-NAZARETH;19500;19;CORREZE;1
19094;;Juillac;JUILLAC;19350;19;CORREZE;1
19095;;Lacelle;LACELLE;19170;19;CORREZE;1
19096;;Ladignac-sur-Rondelles;LADIGNAC-SUR-RONDELLES;19150;19;CORREZE;1
19097;;Lafage-sur-Sombre;LAFAGE-SUR-SOMBRE;19320;19;CORREZE;1
19098;;Lagarde-Enval;LAGARDE-ENVAL;19150;19;CORREZE;1
19099;;Lagleygeolle;LAGLEYGEOLLE;19500;19;CORREZE;1
19100;;Lagraulire;LAGRAULIERE;19700;19;CORREZE;1
19101;;Laguenne;LAGUENNE;19150;19;CORREZE;1
19102;;Lamazire-Basse;LAMAZIERE-BASSE;19160;19;CORREZE;1
19103;;Lamazire-Haute;LAMAZIERE-HAUTE;19340;19;CORREZE;1
19104;;Lamongerie;LAMONGERIE;19510;19;CORREZE;1
19105;;Lanteuil;LANTEUIL;19190;19;CORREZE;1
19106;;Lapleau;LAPLEAU;19550;19;CORREZE;1
19107;;Larche;LARCHE;19600;19;CORREZE;1
19108;;Laroche-prs-Feyt;LAROCHE-PRES-FEYT;19340;19;CORREZE;1
19109;;Lascaux;LASCAUX;19130;19;CORREZE;1
19110;;Latronche;LATRONCHE;19160;19;CORREZE;1
19111;;Laval-sur-Luzge;LAVAL-SUR-LUZEGE;19550;19;CORREZE;1
19112;;Lestards;LESTARDS;19170;19;CORREZE;1
19113;;Liginiac;LIGINIAC;19160;19;CORREZE;1
19114;;Lignareix;LIGNAREIX;19200;19;CORREZE;1
19115;;Ligneyrac;LIGNEYRAC;19500;19;CORREZE;1
19116;;Liourdres;LIOURDRES;19120;19;CORREZE;1
19117;;Lissac-sur-Couze;LISSAC-SUR-COUZE;19600;19;CORREZE;1
19118;Le;Lonzac;LONZAC;19470;19;CORREZE;1
19119;;Lostanges;LOSTANGES;19500;19;CORREZE;1
19120;;Louignac;LOUIGNAC;19310;19;CORREZE;1
19121;;Lubersac;LUBERSAC;19210;19;CORREZE;1
19122;;Madranges;MADRANGES;19470;19;CORREZE;1
19123;;Malemort-sur-Corrze;MALEMORT-SUR-CORREZE;19360;19;CORREZE;1
19124;;Mansac;MANSAC;19520;19;CORREZE;1
19125;;Marcillac-la-Croisille;MARCILLAC-LA-CROISILLE;19320;19;CORREZE;1
19126;;Marcillac-la-Croze;MARCILLAC-LA-CROZE;19500;19;CORREZE;1
19127;;Marc-la-Tour;MARC-LA-TOUR;19150;19;CORREZE;1
19128;;Margerides;MARGERIDES;19200;19;CORREZE;1
19129;;Masseret;MASSERET;19510;19;CORREZE;1
19130;;Maussac;MAUSSAC;19250;19;CORREZE;1
19131;;Meilhards;MEILHARDS;19510;19;CORREZE;1
19132;;Mnoire;MENOIRE;19190;19;CORREZE;1
19133;;Mercur;MERCOEUR;19430;19;CORREZE;1
19134;;Merlines;MERLINES;19340;19;CORREZE;1
19135;;Mestes;MESTES;19200;19;CORREZE;1
19136;;Meymac;MEYMAC;19250;19;CORREZE;1
19137;;Meyrignac-l'glise;MEYRIGNAC-L'EGLISE;19800;19;CORREZE;1
19138;;Meyssac;MEYSSAC;19500;19;CORREZE;1
19139;;Millevaches;MILLEVACHES;19290;19;CORREZE;1
19140;;Monceaux-sur-Dordogne;MONCEAUX-SUR-DORDOGNE;19400;19;CORREZE;1
19141;;Monestier-Merlines;MONESTIER-MERLINES;19340;19;CORREZE;1
19142;;Monestier-Port-Dieu;MONESTIER-PORT-DIEU;19110;19;CORREZE;1
19143;;Montaignac-Saint-Hippolyte;MONTAIGNAC-SAINT-HIPPOLYTE;19300;19;CORREZE;1
19144;;Montgibaud;MONTGIBAUD;19210;19;CORREZE;1
19145;;Moustier-Ventadour;MOUSTIER-VENTADOUR;19300;19;CORREZE;1
19146;;Naves;NAVES;19460;19;CORREZE;1
19147;;Nespouls;NESPOULS;19600;19;CORREZE;1
19148;;Neuvic;NEUVIC;19160;19;CORREZE;1
19149;;Neuville;NEUVILLE;19380;19;CORREZE;1
19150;;Noailhac;NOAILHAC;19500;19;CORREZE;1
19151;;Noailles;NOAILLES;19600;19;CORREZE;1
19152;;Nonards;NONARDS;19120;19;CORREZE;1
19153;;Objat;OBJAT;19130;19;CORREZE;1
19154;;Orgnac-sur-Vzre;ORGNAC-SUR-VEZERE;19410;19;CORREZE;1
19155;;Orliac-de-Bar;ORLIAC-DE-BAR;19390;19;CORREZE;1
19156;;Palazinges;PALAZINGES;19190;19;CORREZE;1
19157;;Palisse;PALISSE;19160;19;CORREZE;1
19158;;Pandrignes;PANDRIGNES;19150;19;CORREZE;1
19159;;Pret-Bel-Air;PERET-BEL-AIR;19300;19;CORREZE;1
19160;;Prols-sur-Vzre;PEROLS-SUR-VEZERE;19170;19;CORREZE;1
19161;;Perpezac-le-Blanc;PERPEZAC-LE-BLANC;19310;19;CORREZE;1
19162;;Perpezac-le-Noir;PERPEZAC-LE-NOIR;19410;19;CORREZE;1
19163;Le;Pescher;PESCHER;19190;19;CORREZE;1
19164;;Peyrelevade;PEYRELEVADE;19290;19;CORREZE;1
19165;;Peyrissac;PEYRISSAC;19260;19;CORREZE;1
19166;;Pierrefitte;PIERREFITTE;19450;19;CORREZE;1
19167;;Confolent-Port-Dieu;CONFOLENT-PORT-DIEU;19200;19;CORREZE;1
19168;;Pradines;PRADINES;19170;19;CORREZE;1
19169;;Puy-d'Arnac;PUY-D'ARNAC;19120;19;CORREZE;1
19170;;Queyssac-les-Vignes;QUEYSSAC-LES-VIGNES;19120;19;CORREZE;1
19171;;Reygade;REYGADE;19430;19;CORREZE;1
19172;;Rilhac-Treignac;RILHAC-TREIGNAC;19260;19;CORREZE;1
19173;;Rilhac-Xaintrie;RILHAC-XAINTRIE;19220;19;CORREZE;1
19174;La;Roche-Canillac;ROCHE-CANILLAC;19320;19;CORREZE;1
19175;;Roche-le-Peyroux;ROCHE-LE-PEYROUX;19160;19;CORREZE;1
19176;;Rosiers-d'gletons;ROSIERS-D'EGLETONS;19300;19;CORREZE;1
19177;;Rosiers-de-Juillac;ROSIERS-DE-JUILLAC;19350;19;CORREZE;1
19178;;Sadroc;SADROC;19270;19;CORREZE;1
19179;;Saillac;SAILLAC;19500;19;CORREZE;1
19180;;Saint-Angel;SAINT-ANGEL;19200;19;CORREZE;1
19181;;Saint-Augustin;SAINT-AUGUSTIN;19390;19;CORREZE;1
19182;;Saint-Aulaire;SAINT-AULAIRE;19130;19;CORREZE;1
19183;;Saint-Bazile-de-la-Roche;SAINT-BAZILE-DE-LA-ROCHE;19320;19;CORREZE;1
19184;;Saint-Bazile-de-Meyssac;SAINT-BAZILE-DE-MEYSSAC;19500;19;CORREZE;1
19185;;Saint-Bonnet-Avalouze;SAINT-BONNET-AVALOUZE;19150;19;CORREZE;1
19186;;Saint-Bonnet-Elvert;SAINT-BONNET-ELVERT;19380;19;CORREZE;1
19187;;Saint-Bonnet-la-Rivire;SAINT-BONNET-LA-RIVIERE;19130;19;CORREZE;1
19188;;Saint-Bonnet-l'Enfantier;SAINT-BONNET-L'ENFANTIER;19410;19;CORREZE;1
19189;;Saint-Bonnet-les-Tours-de-Merle;SAINT-BONNET-LES-TOURS-DE-MERLE;19430;19;CORREZE;1
19190;;Saint-Bonnet-prs-Bort;SAINT-BONNET-PRES-BORT;19200;19;CORREZE;1
19191;;Saint-Cernin-de-Larche;SAINT-CERNIN-DE-LARCHE;19600;19;CORREZE;1
19192;;Saint-Chamant;SAINT-CHAMANT;19380;19;CORREZE;1
19193;;Saint-Cirgues-la-Loutre;SAINT-CIRGUES-LA-LOUTRE;19220;19;CORREZE;1
19194;;Saint-Clment;SAINT-CLEMENT;19700;19;CORREZE;1
19195;;Saint-Cyprien;SAINT-CYPRIEN;19130;19;CORREZE;1
19196;;Saint-Cyr-la-Roche;SAINT-CYR-LA-ROCHE;19130;19;CORREZE;1
19198;;Saint-loy-les-Tuileries;SAINT-ELOY-LES-TUILERIES;19210;19;CORREZE;1
19199;;Saint-tienne-aux-Clos;SAINT-ETIENNE-AUX-CLOS;19200;19;CORREZE;1
19200;;Saint-tienne-la-Geneste;SAINT-ETIENNE-LA-GENESTE;19160;19;CORREZE;1
19201;;Saint-Exupry-les-Roches;SAINT-EXUPERY-LES-ROCHES;19200;19;CORREZE;1
19202;;Sainte-Frole;SAINTE-FEREOLE;19270;19;CORREZE;1
19203;;Sainte-Fortunade;SAINTE-FORTUNADE;19490;19;CORREZE;1
19204;;Saint-Frjoux;SAINT-FREJOUX;19200;19;CORREZE;1
19205;;Saint-Geniez--Merle;SAINT-GENIEZ-O-MERLE;19220;19;CORREZE;1
19206;;Saint-Germain-Lavolps;SAINT-GERMAIN-LAVOLPS;19290;19;CORREZE;1
19207;;Saint-Germain-les-Vergnes;SAINT-GERMAIN-LES-VERGNES;19330;19;CORREZE;1
19208;;Saint-Hilaire-Foissac;SAINT-HILAIRE-FOISSAC;19550;19;CORREZE;1
19209;;Saint-Hilaire-les-Courbes;SAINT-HILAIRE-LES-COURBES;19170;19;CORREZE;1
19210;;Saint-Hilaire-Luc;SAINT-HILAIRE-LUC;19160;19;CORREZE;1
19211;;Saint-Hilaire-Peyroux;SAINT-HILAIRE-PEYROUX;19560;19;CORREZE;1
19212;;Saint-Hilaire-Taurieux;SAINT-HILAIRE-TAURIEUX;19400;19;CORREZE;1
19213;;Saint-Jal;SAINT-JAL;19700;19;CORREZE;1
19214;;Saint-Julien-aux-Bois;SAINT-JULIEN-AUX-BOIS;19220;19;CORREZE;1
19215;;Saint-Julien-le-Plerin;SAINT-JULIEN-LE-PELERIN;19430;19;CORREZE;1
19216;;Saint-Julien-le-Vendmois;SAINT-JULIEN-LE-VENDOMOIS;19210;19;CORREZE;1
19217;;Saint-Julien-Maumont;SAINT-JULIEN-MAUMONT;19500;19;CORREZE;1
19218;;Saint-Julien-prs-Bort;SAINT-JULIEN-PRES-BORT;19110;19;CORREZE;1
19219;;Sainte-Marie-Lapanouze;SAINTE-MARIE-LAPANOUZE;19160;19;CORREZE;1
19220;;Saint-Martial-de-Gimel;SAINT-MARTIAL-DE-GIMEL;19150;19;CORREZE;1
19221;;Saint-Martial-Entraygues;SAINT-MARTIAL-ENTRAYGUES;19400;19;CORREZE;1
19222;;Saint-Martin-la-Manne;SAINT-MARTIN-LA-MEANNE;19320;19;CORREZE;1
19223;;Saint-Martin-Sepert;SAINT-MARTIN-SEPERT;19210;19;CORREZE;1
19225;;Saint-Merd-de-Lapleau;SAINT-MERD-DE-LAPLEAU;19320;19;CORREZE;1
19226;;Saint-Merd-les-Oussines;SAINT-MERD-LES-OUSSINES;19170;19;CORREZE;1
19227;;Saint-Mexant;SAINT-MEXANT;19330;19;CORREZE;1
19228;;Saint-Pantalon-de-Lapleau;SAINT-PANTALEON-DE-LAPLEAU;19160;19;CORREZE;1
19229;;Saint-Pantalon-de-Larche;SAINT-PANTALEON-DE-LARCHE;19600;19;CORREZE;1
19230;;Saint-Pardoux-Corbier;SAINT-PARDOUX-CORBIER;19210;19;CORREZE;1
19231;;Saint-Pardoux-la-Croisille;SAINT-PARDOUX-LA-CROISILLE;19320;19;CORREZE;1
19232;;Saint-Pardoux-le-Neuf;SAINT-PARDOUX-LE-NEUF;19200;19;CORREZE;1
19233;;Saint-Pardoux-le-Vieux;SAINT-PARDOUX-LE-VIEUX;19200;19;CORREZE;1
19234;;Saint-Pardoux-l'Ortigier;SAINT-PARDOUX-L'ORTIGIER;19270;19;CORREZE;1
19235;;Saint-Paul;SAINT-PAUL;19150;19;CORREZE;1
19236;;Saint-Priest-de-Gimel;SAINT-PRIEST-DE-GIMEL;19800;19;CORREZE;1
19237;;Saint-Privat;SAINT-PRIVAT;19220;19;CORREZE;1
19238;;Saint-Rmy;SAINT-REMY;19290;19;CORREZE;1
19239;;Saint-Robert;SAINT-ROBERT;19310;19;CORREZE;1
19240;;Saint-Salvadour;SAINT-SALVADOUR;19700;19;CORREZE;1
19241;;Saint-Setiers;SAINT-SETIERS;19290;19;CORREZE;1
19242;;Saint-Solve;SAINT-SOLVE;19130;19;CORREZE;1
19243;;Saint-Sornin-Lavolps;SAINT-SORNIN-LAVOLPS;19230;19;CORREZE;1
19244;;Saint-Sulpice-les-Bois;SAINT-SULPICE-LES-BOIS;19250;19;CORREZE;1
19245;;Saint-Sylvain;SAINT-SYLVAIN;19380;19;CORREZE;1
19246;;Saint-Viance;SAINT-VIANCE;19240;19;CORREZE;1
19247;;Saint-Victour;SAINT-VICTOUR;19200;19;CORREZE;1
19248;;Saint-Ybard;SAINT-YBARD;19140;19;CORREZE;1
19249;;Saint-Yrieix-le-Djalat;SAINT-YRIEIX-LE-DEJALAT;19300;19;CORREZE;1
19250;;Salon-la-Tour;SALON-LA-TOUR;19510;19;CORREZE;1
19251;;Sarran;SARRAN;19800;19;CORREZE;1
19252;;Sarroux;SARROUX;19110;19;CORREZE;1
19253;;Segonzac;SEGONZAC;19310;19;CORREZE;1
19254;;Sgur-le-Chteau;SEGUR-LE-CHATEAU;19230;19;CORREZE;1
19255;;Seilhac;SEILHAC;19700;19;CORREZE;1
19256;;Srandon;SERANDON;19160;19;CORREZE;1
19257;;Srilhac;SERILHAC;19190;19;CORREZE;1
19258;;Servires-le-Chteau;SERVIERES-LE-CHATEAU;19220;19;CORREZE;1
19259;;Sexcles;SEXCLES;19430;19;CORREZE;1
19260;;Sioniac;SIONIAC;19120;19;CORREZE;1
19261;;Sornac;SORNAC;19290;19;CORREZE;1
19262;;Soudaine-Lavinadire;SOUDAINE-LAVINADIERE;19370;19;CORREZE;1
19263;;Soudeilles;SOUDEILLES;19300;19;CORREZE;1
19264;;Soursac;SOURSAC;19550;19;CORREZE;1
19265;;Tarnac;TARNAC;19170;19;CORREZE;1
19266;;Thalamy;THALAMY;19200;19;CORREZE;1
19268;;Toy-Viam;TOY-VIAM;19170;19;CORREZE;1
19269;;Treignac;TREIGNAC;19260;19;CORREZE;1
19270;;Troche;TROCHE;19230;19;CORREZE;1
19271;;Tudeils;TUDEILS;19120;19;CORREZE;1
19272;;Tulle;TULLE;19000;19;CORREZE;1
19273;;Turenne;TURENNE;19500;19;CORREZE;1
19274;;Ussac;USSAC;19270;19;CORREZE;1
19275;;Ussel;USSEL;19200;19;CORREZE;1
19276;;Uzerche;UZERCHE;19140;19;CORREZE;1
19277;;Valiergues;VALIERGUES;19200;19;CORREZE;1
19278;;Varetz;VARETZ;19240;19;CORREZE;1
19279;;Vars-sur-Roseix;VARS-SUR-ROSEIX;19130;19;CORREZE;1
19280;;Vgennes;VEGENNES;19120;19;CORREZE;1
19281;;Veix;VEIX;19260;19;CORREZE;1
19282;;Venarsal;VENARSAL;19360;19;CORREZE;1
19283;;Veyrires;VEYRIERES;19200;19;CORREZE;1
19284;;Viam;VIAM;19170;19;CORREZE;1
19285;;Vigeois;VIGEOIS;19410;19;CORREZE;1
19286;;Vignols;VIGNOLS;19130;19;CORREZE;1
19287;;Vitrac-sur-Montane;VITRAC-SUR-MONTANE;19800;19;CORREZE;1
19288;;Voutezac;VOUTEZAC;19130;19;CORREZE;1
19289;;Yssandon;YSSANDON;19310;19;CORREZE;1
21001;;Agencourt;AGENCOURT;21700;21;COTE D'OR;1
21002;;Agey;AGEY;21410;21;COTE D'OR;1
21003;;Ahuy;AHUY;21121;21;COTE D'OR;1
21004;;Aignay-le-Duc;AIGNAY-LE-DUC;21510;21;COTE D'OR;1
21005;;Aiserey;AISEREY;21110;21;COTE D'OR;1
21006;;Aisey-sur-Seine;AISEY-SUR-SEINE;21400;21;COTE D'OR;1
21007;;Aisy-sous-Thil;AISY-SOUS-THIL;21390;21;COTE D'OR;1
21008;;Alise-Sainte-Reine;ALISE-SAINTE-REINE;21150;21;COTE D'OR;1
21009;;Allerey;ALLEREY;21230;21;COTE D'OR;1
21010;;Aloxe-Corton;ALOXE-CORTON;21420;21;COTE D'OR;1
21011;;Ampilly-les-Bordes;AMPILLY-LES-BORDES;21450;21;COTE D'OR;1
21012;;Ampilly-le-Sec;AMPILLY-LE-SEC;21400;21;COTE D'OR;1
21013;;Ancey;ANCEY;21410;21;COTE D'OR;1
21014;;Antheuil;ANTHEUIL;21360;21;COTE D'OR;1
21015;;Antigny-la-Ville;ANTIGNY-LA-VILLE;21230;21;COTE D'OR;1
21016;;Arceau;ARCEAU;21310;21;COTE D'OR;1
21017;;Arcenant;ARCENANT;21700;21;COTE D'OR;1
21018;;Arcey;ARCEY;21410;21;COTE D'OR;1
21020;;Arconcey;ARCONCEY;21320;21;COTE D'OR;1
21021;;Arc-sur-Tille;ARC-SUR-TILLE;21560;21;COTE D'OR;1
21022;;Argilly;ARGILLY;21700;21;COTE D'OR;1
21023;;Arnay-le-Duc;ARNAY-LE-DUC;21230;21;COTE D'OR;1
21024;;Arnay-sous-Vitteaux;ARNAY-SOUS-VITTEAUX;21350;21;COTE D'OR;1
21025;;Arrans;ARRANS;21500;21;COTE D'OR;1
21026;;Asnires-en-Montagne;ASNIERES-EN-MONTAGNE;21500;21;COTE D'OR;1
21027;;Asnires-ls-Dijon;ASNIERES-LES-DIJON;21380;21;COTE D'OR;1
21028;;Athe;ATHEE;21130;21;COTE D'OR;1
21029;;Athie;ATHIE;21500;21;COTE D'OR;1
21030;;Aubaine;AUBAINE;21360;21;COTE D'OR;1
21031;;Aubigny-en-Plaine;AUBIGNY-EN-PLAINE;21170;21;COTE D'OR;1
21032;;Aubigny-la-Ronce;AUBIGNY-LA-RONCE;21340;21;COTE D'OR;1
21033;;Aubigny-ls-Sombernon;AUBIGNY-LES-SOMBERNON;21540;21;COTE D'OR;1
21034;;Autricourt;AUTRICOURT;21570;21;COTE D'OR;1
21035;;Auvillars-sur-Sane;AUVILLARS-SUR-SAONE;21250;21;COTE D'OR;1
21036;;Auxant;AUXANT;21360;21;COTE D'OR;1
21037;;Auxey-Duresses;AUXEY-DURESSES;21190;21;COTE D'OR;1
21038;;Auxonne;AUXONNE;21130;21;COTE D'OR;1
21039;;Avelanges;AVELANGES;21120;21;COTE D'OR;1
21040;;Avosnes;AVOSNES;21350;21;COTE D'OR;1
21041;;Avot;AVOT;21580;21;COTE D'OR;1
21042;;Bagnot;BAGNOT;21700;21;COTE D'OR;1
21043;;Baigneux-les-Juifs;BAIGNEUX-LES-JUIFS;21450;21;COTE D'OR;1
21044;;Balot;BALOT;21330;21;COTE D'OR;1
21045;;Barbirey-sur-Ouche;BARBIREY-SUR-OUCHE;21410;21;COTE D'OR;1
21046;;Bard-le-Rgulier;BARD-LE-REGULIER;21430;21;COTE D'OR;1
21047;;Bard-ls-poisses;BARD-LES-EPOISSES;21460;21;COTE D'OR;1
21048;;Barges;BARGES;21910;21;COTE D'OR;1
21049;;Barjon;BARJON;21580;21;COTE D'OR;1
21050;;Baubigny;BAUBIGNY;21340;21;COTE D'OR;1
21051;;Baulme-la-Roche;BAULME-LA-ROCHE;21410;21;COTE D'OR;1
21052;;Beaulieu;BEAULIEU;21510;21;COTE D'OR;1
21053;;Beaumont-sur-Vingeanne;BEAUMONT-SUR-VINGEANNE;21310;21;COTE D'OR;1
21054;;Beaune;BEAUNE;21200;21;COTE D'OR;1
21055;;Beaunotte;BEAUNOTTE;21510;21;COTE D'OR;1
21056;;Beire-le-Chtel;BEIRE-LE-CHATEL;21310;21;COTE D'OR;1
21057;;Beire-le-Fort;BEIRE-LE-FORT;21110;21;COTE D'OR;1
21058;;Belan-sur-Ource;BELAN-SUR-OURCE;21570;21;COTE D'OR;1
21059;;Bellefond;BELLEFOND;21490;21;COTE D'OR;1
21060;;Belleneuve;BELLENEUVE;21310;21;COTE D'OR;1
21061;;Bellenod-sur-Seine;BELLENOD-SUR-SEINE;21510;21;COTE D'OR;1
21062;;Bellenot-sous-Pouilly;BELLENOT-SOUS-POUILLY;21320;21;COTE D'OR;1
21063;;Beneuvre;BENEUVRE;21290;21;COTE D'OR;1
21064;;Benoisey;BENOISEY;21500;21;COTE D'OR;1
21065;;Bessey-en-Chaume;BESSEY-EN-CHAUME;21360;21;COTE D'OR;1
21066;;Bessey-la-Cour;BESSEY-LA-COUR;21360;21;COTE D'OR;1
21067;;Bessey-ls-Cteaux;BESSEY-LES-CITEAUX;21110;21;COTE D'OR;1
21068;;Beurey-Bauguay;BEUREY-BAUGUAY;21320;21;COTE D'OR;1
21069;;Beurizot;BEURIZOT;21350;21;COTE D'OR;1
21070;;Bvy;BEVY;21220;21;COTE D'OR;1
21071;;Bze;BEZE;21310;21;COTE D'OR;1
21072;;Bzouotte;BEZOUOTTE;21310;21;COTE D'OR;1
21073;;Bierre-ls-Semur;BIERRE-LES-SEMUR;21390;21;COTE D'OR;1
21074;;Billey;BILLEY;21130;21;COTE D'OR;1
21075;;Billy-ls-Chanceaux;BILLY-LES-CHANCEAUX;21450;21;COTE D'OR;1
21076;;Binges;BINGES;21270;21;COTE D'OR;1
21077;;Bissey-la-Cte;BISSEY-LA-COTE;21520;21;COTE D'OR;1
21078;;Bissey-la-Pierre;BISSEY-LA-PIERRE;21330;21;COTE D'OR;1
21079;;Blagny-sur-Vingeanne;BLAGNY-SUR-VINGEANNE;21310;21;COTE D'OR;1
21080;;Blaisy-Bas;BLAISY-BAS;21540;21;COTE D'OR;1
21081;;Blaisy-Haut;BLAISY-HAUT;21540;21;COTE D'OR;1
21082;;Blancey;BLANCEY;21320;21;COTE D'OR;1
21083;;Blanot;BLANOT;21430;21;COTE D'OR;1
21084;;Source-Seine;SOURCE-SEINE;21690;21;COTE D'OR;1
21085;;Bligny-le-Sec;BLIGNY-LE-SEC;21440;21;COTE D'OR;1
21086;;Bligny-ls-Beaune;BLIGNY-LES-BEAUNE;21200;21;COTE D'OR;1
21087;;Bligny-sur-Ouche;BLIGNY-SUR-OUCHE;21360;21;COTE D'OR;1
21088;;Boncourt-le-Bois;BONCOURT-LE-BOIS;21700;21;COTE D'OR;1
21089;;Bonnencontre;BONNENCONTRE;21250;21;COTE D'OR;1
21090;;Boudreville;BOUDREVILLE;21520;21;COTE D'OR;1
21091;;Bouhey;BOUHEY;21360;21;COTE D'OR;1
21092;;Bouilland;BOUILLAND;21420;21;COTE D'OR;1
21093;;Bouix;BOUIX;21330;21;COTE D'OR;1
21094;;Bourberain;BOURBERAIN;21610;21;COTE D'OR;1
21095;;Bousselange;BOUSSELANGE;21250;21;COTE D'OR;1
21096;;Boussenois;BOUSSENOIS;21260;21;COTE D'OR;1
21097;;Boussey;BOUSSEY;21350;21;COTE D'OR;1
21098;;Boux-sous-Salmaise;BOUX-SOUS-SALMAISE;21690;21;COTE D'OR;1
21099;;Bouze-ls-Beaune;BOUZE-LES-BEAUNE;21200;21;COTE D'OR;1
21100;;Brain;BRAIN;21350;21;COTE D'OR;1
21101;;Braux;BRAUX;21390;21;COTE D'OR;1
21102;;Brazey-en-Morvan;BRAZEY-EN-MORVAN;21430;21;COTE D'OR;1
21103;;Brazey-en-Plaine;BRAZEY-EN-PLAINE;21470;21;COTE D'OR;1
21104;;Brmur-et-Vaurois;BREMUR-ET-VAUROIS;21400;21;COTE D'OR;1
21105;;Bressey-sur-Tille;BRESSEY-SUR-TILLE;21560;21;COTE D'OR;1
21106;;Bretenire;BRETENIERE;21110;21;COTE D'OR;1
21107;;Bretigny;BRETIGNY;21490;21;COTE D'OR;1
21108;;Brianny;BRIANNY;21390;21;COTE D'OR;1
21109;;Brion-sur-Ource;BRION-SUR-OURCE;21570;21;COTE D'OR;1
21110;;Brochon;BROCHON;21220;21;COTE D'OR;1
21111;;Brognon;BROGNON;21490;21;COTE D'OR;1
21112;;Broin;BROIN;21250;21;COTE D'OR;1
21113;;Broindon;BROINDON;21220;21;COTE D'OR;1
21114;;Buffon;BUFFON;21500;21;COTE D'OR;1
21115;;Buncey;BUNCEY;21400;21;COTE D'OR;1
21116;;Bure-les-Templiers;BURE-LES-TEMPLIERS;21290;21;COTE D'OR;1
21117;;Busseaut;BUSSEAUT;21510;21;COTE D'OR;1
21118;;Busserotte-et-Montenaille;BUSSEROTTE-ET-MONTENAILLE;21580;21;COTE D'OR;1
21119;;Bussires;BUSSIERES;21580;21;COTE D'OR;1
21120;La;Bussire-sur-Ouche;BUSSIERE-SUR-OUCHE;21360;21;COTE D'OR;1
21121;;Bussy-la-Pesle;BUSSY-LA-PESLE;21540;21;COTE D'OR;1
21122;;Bussy-le-Grand;BUSSY-LE-GRAND;21150;21;COTE D'OR;1
21123;;Buxerolles;BUXEROLLES;21290;21;COTE D'OR;1
21124;;Censerey;CENSEREY;21430;21;COTE D'OR;1
21125;;Crilly;CERILLY;21330;21;COTE D'OR;1
21126;;Cessey-sur-Tille;CESSEY-SUR-TILLE;21110;21;COTE D'OR;1
21127;;Chaignay;CHAIGNAY;21120;21;COTE D'OR;1
21128;;Chailly-sur-Armanon;CHAILLY-SUR-ARMANCON;21320;21;COTE D'OR;1
21129;;Chambain;CHAMBAIN;21290;21;COTE D'OR;1
21130;;Chambeire;CHAMBEIRE;21110;21;COTE D'OR;1
21131;;Chamblanc;CHAMBLANC;21250;21;COTE D'OR;1
21132;;Chambuf;CHAMBOEUF;21220;21;COTE D'OR;1
21133;;Chambolle-Musigny;CHAMBOLLE-MUSIGNY;21220;21;COTE D'OR;1
21134;;Chamesson;CHAMESSON;21400;21;COTE D'OR;1
21135;;Champagne-sur-Vingeanne;CHAMPAGNE-SUR-VINGEANNE;21310;21;COTE D'OR;1
21136;;Champagny;CHAMPAGNY;21440;21;COTE D'OR;1
21137;;Champ-d'Oiseau;CHAMP-D'OISEAU;21500;21;COTE D'OR;1
21138;;Champdtre;CHAMPDOTRE;21130;21;COTE D'OR;1
21139;;Champeau-en-Morvan;CHAMPEAU-EN-MORVAN;21210;21;COTE D'OR;1
21140;;Champignolles;CHAMPIGNOLLES;21230;21;COTE D'OR;1
21141;;Champrenault;CHAMPRENAULT;21690;21;COTE D'OR;1
21142;;Chanceaux;CHANCEAUX;21440;21;COTE D'OR;1
21143;;Channay;CHANNAY;21330;21;COTE D'OR;1
21144;;Charencey;CHARENCEY;21690;21;COTE D'OR;1
21145;;Charigny;CHARIGNY;21140;21;COTE D'OR;1
21146;;Charmes;CHARMES;21310;21;COTE D'OR;1
21147;;Charny;CHARNY;21350;21;COTE D'OR;1
21148;;Charrey-sur-Sane;CHARREY-SUR-SAONE;21170;21;COTE D'OR;1
21149;;Charrey-sur-Seine;CHARREY-SUR-SEINE;21400;21;COTE D'OR;1
21150;;Chassagne-Montrachet;CHASSAGNE-MONTRACHET;21190;21;COTE D'OR;1
21151;;Chassey;CHASSEY;21150;21;COTE D'OR;1
21152;;Chteauneuf;CHATEAUNEUF;21320;21;COTE D'OR;1
21153;;Chtellenot;CHATELLENOT;21320;21;COTE D'OR;1
21154;;Chtillon-sur-Seine;CHATILLON-SUR-SEINE;21400;21;COTE D'OR;1
21155;;Chaudenay-la-Ville;CHAUDENAY-LA-VILLE;21360;21;COTE D'OR;1
21156;;Chaudenay-le-Chteau;CHAUDENAY-LE-CHATEAU;21360;21;COTE D'OR;1
21157;;Chaugey;CHAUGEY;21290;21;COTE D'OR;1
21158;;Chaume-et-Courchamp;CHAUME-ET-COURCHAMP;21610;21;COTE D'OR;1
21159;La;Chaume;CHAUME;21520;21;COTE D'OR;1
21160;;Chaume-ls-Baigneux;CHAUME-LES-BAIGNEUX;21450;21;COTE D'OR;1
21161;;Chaumont-le-Bois;CHAUMONT-LE-BOIS;21400;21;COTE D'OR;1
21162;;Chaux;CHAUX;21700;21;COTE D'OR;1
21163;;Chazeuil;CHAZEUIL;21260;21;COTE D'OR;1
21164;;Chazilly;CHAZILLY;21320;21;COTE D'OR;1
21165;;Chemin-d'Aisey;CHEMIN-D'AISEY;21400;21;COTE D'OR;1
21166;;Chenve;CHENOVE;21300;21;COTE D'OR;1
21167;;Cheuge;CHEUGE;21310;21;COTE D'OR;1
21168;;Chevannay;CHEVANNAY;21540;21;COTE D'OR;1
21169;;Chevannes;CHEVANNES;21220;21;COTE D'OR;1
21170;;Chevigny-en-Valire;CHEVIGNY-EN-VALIERE;21200;21;COTE D'OR;1
21171;;Chevigny-Saint-Sauveur;CHEVIGNY-SAINT-SAUVEUR;21800;21;COTE D'OR;1
21172;;Chivres;CHIVRES;21820;21;COTE D'OR;1
21173;;Chorey-les-Beaune;CHOREY-LES-BEAUNE;21200;21;COTE D'OR;1
21175;;Cirey-ls-Pontailler;CIREY-LES-PONTAILLER;21270;21;COTE D'OR;1
21176;;Civry-en-Montagne;CIVRY-EN-MONTAGNE;21320;21;COTE D'OR;1
21177;;Clamerey;CLAMEREY;21390;21;COTE D'OR;1
21178;;Clmencey;CLEMENCEY;21220;21;COTE D'OR;1
21179;;Clnay;CLENAY;21490;21;COTE D'OR;1
21180;;Clry;CLERY;21270;21;COTE D'OR;1
21181;;Clomot;CLOMOT;21230;21;COTE D'OR;1
21182;;Collonges-ls-Bvy;COLLONGES-LES-BEVY;21220;21;COTE D'OR;1
21183;;Collonges-ls-Premires;COLLONGES-LES-PREMIERES;21110;21;COTE D'OR;1
21184;;Colombier;COLOMBIER;21360;21;COTE D'OR;1
21185;;Combertault;COMBERTAULT;21200;21;COTE D'OR;1
21186;;Comblanchien;COMBLANCHIEN;21700;21;COTE D'OR;1
21187;;Commarin;COMMARIN;21320;21;COTE D'OR;1
21189;;Corberon;CORBERON;21250;21;COTE D'OR;1
21190;;Corcelles-les-Arts;CORCELLES-LES-ARTS;21190;21;COTE D'OR;1
21191;;Corcelles-ls-Cteaux;CORCELLES-LES-CITEAUX;21910;21;COTE D'OR;1
21192;;Corcelles-les-Monts;CORCELLES-LES-MONTS;21160;21;COTE D'OR;1
21193;;Corgengoux;CORGENGOUX;21250;21;COTE D'OR;1
21194;;Corgoloin;CORGOLOIN;21700;21;COTE D'OR;1
21195;;Cormot-le-Grand;CORMOT-LE-GRAND;21340;21;COTE D'OR;1
21196;;Corpeau;CORPEAU;21190;21;COTE D'OR;1
21197;;Corpoyer-la-Chapelle;CORPOYER-LA-CHAPELLE;21150;21;COTE D'OR;1
21198;;Corrombles;CORROMBLES;21460;21;COTE D'OR;1
21199;;Corsaint;CORSAINT;21460;21;COTE D'OR;1
21200;;Couchey;COUCHEY;21160;21;COTE D'OR;1
21201;;Coulmier-le-Sec;COULMIER-LE-SEC;21400;21;COTE D'OR;1
21202;;Courban;COURBAN;21520;21;COTE D'OR;1
21203;;Courcelles-Frmoy;COURCELLES-FREMOY;21460;21;COTE D'OR;1
21204;;Courcelles-ls-Montbard;COURCELLES-LES-MONTBARD;21500;21;COTE D'OR;1
21205;;Courcelles-ls-Semur;COURCELLES-LES-SEMUR;21140;21;COTE D'OR;1
21207;;Courlon;COURLON;21580;21;COTE D'OR;1
21208;;Courtivron;COURTIVRON;21120;21;COTE D'OR;1
21209;;Couternon;COUTERNON;21560;21;COTE D'OR;1
21210;;Crancey;CREANCEY;21320;21;COTE D'OR;1
21211;;Crcey-sur-Tille;CRECEY-SUR-TILLE;21120;21;COTE D'OR;1
21212;;Crpand;CREPAND;21500;21;COTE D'OR;1
21213;;Crimolois;CRIMOLOIS;21800;21;COTE D'OR;1
21214;;Crugey;CRUGEY;21360;21;COTE D'OR;1
21215;;Cuiserey;CUISEREY;21310;21;COTE D'OR;1
21216;;Cultre;CULETRE;21230;21;COTE D'OR;1
21217;;Curley;CURLEY;21220;21;COTE D'OR;1
21218;;Curtil-Saint-Seine;CURTIL-SAINT-SEINE;21380;21;COTE D'OR;1
21219;;Curtil-Vergy;CURTIL-VERGY;21220;21;COTE D'OR;1
21220;;Cussey-les-Forges;CUSSEY-LES-FORGES;21580;21;COTE D'OR;1
21221;;Cussy-la-Colonne;CUSSY-LA-COLONNE;21360;21;COTE D'OR;1
21222;;Cussy-le-Chtel;CUSSY-LE-CHATEL;21230;21;COTE D'OR;1
21223;;Daix;DAIX;21121;21;COTE D'OR;1
21224;;Dampierre-en-Montagne;DAMPIERRE-EN-MONTAGNE;21350;21;COTE D'OR;1
21225;;Dampierre-et-Fle;DAMPIERRE-ET-FLEE;21310;21;COTE D'OR;1
21226;;Darcey;DARCEY;21150;21;COTE D'OR;1
21227;;Darois;DAROIS;21121;21;COTE D'OR;1
21228;;Dtain-et-Bruant;DETAIN-ET-BRUANT;21220;21;COTE D'OR;1
21229;;Diancey;DIANCEY;21430;21;COTE D'OR;1
21230;;Dinay;DIENAY;21120;21;COTE D'OR;1
21231;;Dijon;DIJON;21000;21;COTE D'OR;1
21232;;Dompierre-en-Morvan;DOMPIERRE-EN-MORVAN;21390;21;COTE D'OR;1
21233;;Drambon;DRAMBON;21270;21;COTE D'OR;1
21234;;Dre;DREE;21540;21;COTE D'OR;1
21235;;Duesme;DUESME;21510;21;COTE D'OR;1
21236;;baty;EBATY;21190;21;COTE D'OR;1
21237;;chalot;ECHALOT;21510;21;COTE D'OR;1
21238;;channay;ECHANNAY;21540;21;COTE D'OR;1
21239;;chenon;ECHENON;21170;21;COTE D'OR;1
21240;;chevannes;ECHEVANNES;21120;21;COTE D'OR;1
21241;;chevronne;ECHEVRONNE;21420;21;COTE D'OR;1
21242;;chigey;ECHIGEY;21110;21;COTE D'OR;1
21243;;cutigny;ECUTIGNY;21360;21;COTE D'OR;1
21244;;guilly;EGUILLY;21320;21;COTE D'OR;1
21245;;pagny;EPAGNY;21380;21;COTE D'OR;1
21246;;pernay-sous-Gevrey;EPERNAY-SOUS-GEVREY;21220;21;COTE D'OR;1
21247;;poisses;EPOISSES;21460;21;COTE D'OR;1
21248;;ringes;ERINGES;21500;21;COTE D'OR;1
21249;;Esbarres;ESBARRES;21170;21;COTE D'OR;1
21250;;Essarois;ESSAROIS;21290;21;COTE D'OR;1
21251;;Essey;ESSEY;21320;21;COTE D'OR;1
21252;;tais;ETAIS;21500;21;COTE D'OR;1
21253;;talante;ETALANTE;21510;21;COTE D'OR;1
21254;L';tang-Vergy;ETANG-VERGY;21220;21;COTE D'OR;1
21255;;taules;ETAULES;21121;21;COTE D'OR;1
21256;;tevaux;ETEVAUX;21270;21;COTE D'OR;1
21257;;tormay;ETORMAY;21450;21;COTE D'OR;1
21258;;trochey;ETROCHEY;21400;21;COTE D'OR;1
21259;;Fain-ls-Montbard;FAIN-LES-MONTBARD;21500;21;COTE D'OR;1
21260;;Fain-ls-Moutiers;FAIN-LES-MOUTIERS;21500;21;COTE D'OR;1
21261;;Fauverney;FAUVERNEY;21110;21;COTE D'OR;1
21262;;Faverolles-ls-Lucey;FAVEROLLES-LES-LUCEY;21290;21;COTE D'OR;1
21263;;Fnay;FENAY;21600;21;COTE D'OR;1
21264;Le;Fte;FETE;21230;21;COTE D'OR;1
21265;;Fixin;FIXIN;21220;21;COTE D'OR;1
21266;;Flacey;FLACEY;21490;21;COTE D'OR;1
21267;;Flagey-Echzeaux;FLAGEY-ECHEZEAUX;21640;21;COTE D'OR;1
21268;;Flagey-ls-Auxonne;FLAGEY-LES-AUXONNE;21130;21;COTE D'OR;1
21269;;Flammerans;FLAMMERANS;21130;21;COTE D'OR;1
21270;;Flavignerot;FLAVIGNEROT;21160;21;COTE D'OR;1
21271;;Flavigny-sur-Ozerain;FLAVIGNY-SUR-OZERAIN;21150;21;COTE D'OR;1
21272;;Fle;FLEE;21140;21;COTE D'OR;1
21273;;Fleurey-sur-Ouche;FLEUREY-SUR-OUCHE;21410;21;COTE D'OR;1
21274;;Foissy;FOISSY;21230;21;COTE D'OR;1
21275;;Foncegrive;FONCEGRIVE;21260;21;COTE D'OR;1
21276;;Fontaines-en-Duesmois;FONTAINES-EN-DUESMOIS;21450;21;COTE D'OR;1
21277;;Fontaine-Franaise;FONTAINE-FRANCAISE;21610;21;COTE D'OR;1
21278;;Fontaine-ls-Dijon;FONTAINE-LES-DIJON;21121;21;COTE D'OR;1
21279;;Fontaines-les-Sches;FONTAINES-LES-SECHES;21330;21;COTE D'OR;1
21280;;Fontangy;FONTANGY;21390;21;COTE D'OR;1
21281;;Fontenelle;FONTENELLE;21610;21;COTE D'OR;1
21282;;Forlans;FORLEANS;21460;21;COTE D'OR;1
21283;;Fraignot-et-Vesvrotte;FRAIGNOT-ET-VESVROTTE;21580;21;COTE D'OR;1
21284;;Francheville;FRANCHEVILLE;21440;21;COTE D'OR;1
21285;;Franxault;FRANXAULT;21170;21;COTE D'OR;1
21286;;Frnois;FRENOIS;21120;21;COTE D'OR;1
21287;;Fresnes;FRESNES;21500;21;COTE D'OR;1
21288;;Frlois;FROLOIS;21150;21;COTE D'OR;1
21289;;Fussey;FUSSEY;21700;21;COTE D'OR;1
21290;;Gemeaux;GEMEAUX;21120;21;COTE D'OR;1
21291;;Genay;GENAY;21140;21;COTE D'OR;1
21292;;Genlis;GENLIS;21110;21;COTE D'OR;1
21293;;Gergueil;GERGUEIL;21410;21;COTE D'OR;1
21294;;Gerland;GERLAND;21700;21;COTE D'OR;1
21295;;Gevrey-Chambertin;GEVREY-CHAMBERTIN;21220;21;COTE D'OR;1
21296;;Gevrolles;GEVROLLES;21520;21;COTE D'OR;1
21297;;Gilly-ls-Cteaux;GILLY-LES-CITEAUX;21640;21;COTE D'OR;1
21298;;Gissey-le-Vieil;GISSEY-LE-VIEIL;21350;21;COTE D'OR;1
21299;;Gissey-sous-Flavigny;GISSEY-SOUS-FLAVIGNY;21150;21;COTE D'OR;1
21300;;Gissey-sur-Ouche;GISSEY-SUR-OUCHE;21410;21;COTE D'OR;1
21301;;Glanon;GLANON;21250;21;COTE D'OR;1
21302;;Gommville;GOMMEVILLE;21400;21;COTE D'OR;1
21303;Les;Goulles;GOULLES;21520;21;COTE D'OR;1
21304;;Grancey-le-Chteau-Neuvelle;GRANCEY-LE-CHATEAU-NEUVELLE;21580;21;COTE D'OR;1
21305;;Grancey-sur-Ource;GRANCEY-SUR-OURCE;21570;21;COTE D'OR;1
21306;;Grenant-ls-Sombernon;GRENANT-LES-SOMBERNON;21540;21;COTE D'OR;1
21307;;Grsigny-Sainte-Reine;GRESIGNY-SAINTE-REINE;21150;21;COTE D'OR;1
21308;;Grignon;GRIGNON;21150;21;COTE D'OR;1
21309;;Griselles;GRISELLES;21330;21;COTE D'OR;1
21310;;Grosbois-en-Montagne;GROSBOIS-EN-MONTAGNE;21540;21;COTE D'OR;1
21311;;Grosbois-ls-Tichey;GROSBOIS-LES-TICHEY;21250;21;COTE D'OR;1
21312;;Gurgy-la-Ville;GURGY-LA-VILLE;21290;21;COTE D'OR;1
21313;;Gurgy-le-Chteau;GURGY-LE-CHATEAU;21290;21;COTE D'OR;1
21314;;Hauteroche;HAUTEROCHE;21150;21;COTE D'OR;1
21315;;Hauteville-ls-Dijon;HAUTEVILLE-LES-DIJON;21121;21;COTE D'OR;1
21316;;Heuilley-sur-Sane;HEUILLEY-SUR-SAONE;21270;21;COTE D'OR;1
21317;;Is-sur-Tille;IS-SUR-TILLE;21120;21;COTE D'OR;1
21318;;Ivry-en-Montagne;IVRY-EN-MONTAGNE;21340;21;COTE D'OR;1
21319;;Izeure;IZEURE;21110;21;COTE D'OR;1
21320;;Izier;IZIER;21110;21;COTE D'OR;1
21321;;Jailly-les-Moulins;JAILLY-LES-MOULINS;21150;21;COTE D'OR;1
21322;;Jallanges;JALLANGES;21250;21;COTE D'OR;1
21323;;Jancigny;JANCIGNY;21310;21;COTE D'OR;1
21324;;Jeux-ls-Bard;JEUX-LES-BARD;21460;21;COTE D'OR;1
21325;;Jouey;JOUEY;21230;21;COTE D'OR;1
21326;;Jours-ls-Baigneux;JOURS-LES-BAIGNEUX;21450;21;COTE D'OR;1
21327;;Jours-en-Vaux;JOURS-EN-VAUX;21340;21;COTE D'OR;1
21328;;Juillenay;JUILLENAY;21210;21;COTE D'OR;1
21329;;Juilly;JUILLY;21140;21;COTE D'OR;1
21330;;Labergement-Foigney;LABERGEMENT-FOIGNEY;21110;21;COTE D'OR;1
21331;;Labergement-ls-Auxonne;LABERGEMENT-LES-AUXONNE;21130;21;COTE D'OR;1
21332;;Labergement-ls-Seurre;LABERGEMENT-LES-SEURRE;21820;21;COTE D'OR;1
21333;;Labruyre;LABRUYERE;21250;21;COTE D'OR;1
21334;;Lacanche;LACANCHE;21230;21;COTE D'OR;1
21335;;Lacour-d'Arcenay;LACOUR-D'ARCENAY;21210;21;COTE D'OR;1
21336;;Laignes;LAIGNES;21330;21;COTE D'OR;1
21337;;Lamarche-sur-Sane;LAMARCHE-SUR-SAONE;21760;21;COTE D'OR;1
21338;;Lamargelle;LAMARGELLE;21440;21;COTE D'OR;1
21339;;Lantenay;LANTENAY;21370;21;COTE D'OR;1
21340;;Lanthes;LANTHES;21250;21;COTE D'OR;1
21341;;Lantilly;LANTILLY;21140;21;COTE D'OR;1
21342;;Laperrire-sur-Sane;LAPERRIERE-SUR-SAONE;21170;21;COTE D'OR;1
21343;;Larrey;LARREY;21330;21;COTE D'OR;1
21344;;Lechtelet;LECHATELET;21250;21;COTE D'OR;1
21345;;Lry;LERY;21440;21;COTE D'OR;1
21346;;Leuglay;LEUGLAY;21290;21;COTE D'OR;1
21347;;Levernois;LEVERNOIS;21200;21;COTE D'OR;1
21348;;Licey-sur-Vingeanne;LICEY-SUR-VINGEANNE;21610;21;COTE D'OR;1
21349;;Liernais;LIERNAIS;21430;21;COTE D'OR;1
21350;;Lignerolles;LIGNEROLLES;21520;21;COTE D'OR;1
21351;;Longchamp;LONGCHAMP;21110;21;COTE D'OR;1
21352;;Longeault;LONGEAULT;21110;21;COTE D'OR;1
21353;;Longecourt-en-Plaine;LONGECOURT-EN-PLAINE;21110;21;COTE D'OR;1
21354;;Longecourt-ls-Cultre;LONGECOURT-LES-CULETRE;21230;21;COTE D'OR;1
21355;;Longvic;LONGVIC;21600;21;COTE D'OR;1
21356;;Losne;LOSNE;21170;21;COTE D'OR;1
21357;;Louesme;LOUESME;21520;21;COTE D'OR;1
21358;;Lucenay-le-Duc;LUCENAY-LE-DUC;21150;21;COTE D'OR;1
21359;;Lucey;LUCEY;21290;21;COTE D'OR;1
21360;;Lusigny-sur-Ouche;LUSIGNY-SUR-OUCHE;21360;21;COTE D'OR;1
21361;;Lux;LUX;21120;21;COTE D'OR;1
21362;;Maconge;MACONGE;21320;21;COTE D'OR;1
21363;;Magnien;MAGNIEN;21230;21;COTE D'OR;1
21364;;Magny-Lambert;MAGNY-LAMBERT;21450;21;COTE D'OR;1
21365;;Magny-la-Ville;MAGNY-LA-VILLE;21140;21;COTE D'OR;1
21366;;Magny-ls-Aubigny;MAGNY-LES-AUBIGNY;21170;21;COTE D'OR;1
21367;;Magny-Montarlot;MAGNY-MONTARLOT;21130;21;COTE D'OR;1
21368;;Magny-ls-Villers;MAGNY-LES-VILLERS;21700;21;COTE D'OR;1
21369;;Magny-Saint-Mdard;MAGNY-SAINT-MEDARD;21310;21;COTE D'OR;1
21370;;Magny-sur-Tille;MAGNY-SUR-TILLE;21110;21;COTE D'OR;1
21371;Les;Maillys;MAILLYS;21130;21;COTE D'OR;1
21372;;Maisey-le-Duc;MAISEY-LE-DUC;21400;21;COTE D'OR;1
21373;;Mlain;MALAIN;21410;21;COTE D'OR;1
21374;;Maligny;MALIGNY;21230;21;COTE D'OR;1
21375;;Manlay;MANLAY;21430;21;COTE D'OR;1
21376;;Marandeuil;MARANDEUIL;21270;21;COTE D'OR;1
21377;;Marcellois;MARCELLOIS;21350;21;COTE D'OR;1
21378;;Marcenay;MARCENAY;21330;21;COTE D'OR;1
21379;;Marcheseuil;MARCHESEUIL;21430;21;COTE D'OR;1
21380;;Marcigny-sous-Thil;MARCIGNY-SOUS-THIL;21390;21;COTE D'OR;1
21381;;Marcilly-et-Dracy;MARCILLY-ET-DRACY;21350;21;COTE D'OR;1
21382;;Marcilly-Ogny;MARCILLY-OGNY;21320;21;COTE D'OR;1
21383;;Marcilly-sur-Tille;MARCILLY-SUR-TILLE;21120;21;COTE D'OR;1
21384;;Marey-ls-Fussey;MAREY-LES-FUSSEY;21700;21;COTE D'OR;1
21385;;Marey-sur-Tille;MAREY-SUR-TILLE;21120;21;COTE D'OR;1
21386;;Marigny-le-Cahout;MARIGNY-LE-CAHOUET;21150;21;COTE D'OR;1
21387;;Marigny-ls-Reulle;MARIGNY-LES-REULLEE;21200;21;COTE D'OR;1
21388;;Marliens;MARLIENS;21110;21;COTE D'OR;1
21389;;Marmagne;MARMAGNE;21500;21;COTE D'OR;1
21390;;Marsannay-la-Cte;MARSANNAY-LA-COTE;21160;21;COTE D'OR;1
21391;;Marsannay-le-Bois;MARSANNAY-LE-BOIS;21380;21;COTE D'OR;1
21392;;Martrois;MARTROIS;21320;21;COTE D'OR;1
21393;;Massingy;MASSINGY;21400;21;COTE D'OR;1
21394;;Massingy-ls-Semur;MASSINGY-LES-SEMUR;21140;21;COTE D'OR;1
21395;;Massingy-ls-Vitteaux;MASSINGY-LES-VITTEAUX;21350;21;COTE D'OR;1
21396;;Mauvilly;MAUVILLY;21510;21;COTE D'OR;1
21397;;Mavilly-Mandelot;MAVILLY-MANDELOT;21190;21;COTE D'OR;1
21398;;Maxilly-sur-Sane;MAXILLY-SUR-SAONE;21270;21;COTE D'OR;1
21399;;Meilly-sur-Rouvres;MEILLY-SUR-ROUVRES;21320;21;COTE D'OR;1
21400;Le;Meix;MEIX;21580;21;COTE D'OR;1
21401;;Meloisey;MELOISEY;21190;21;COTE D'OR;1
21402;;Menesble;MENESBLE;21290;21;COTE D'OR;1
21403;;Mnessaire;MENESSAIRE;21430;21;COTE D'OR;1
21404;;Mntreux-le-Pitois;MENETREUX-LE-PITOIS;21150;21;COTE D'OR;1
21405;;Merceuil;MERCEUIL;21190;21;COTE D'OR;1
21406;;Mesmont;MESMONT;21540;21;COTE D'OR;1
21407;;Messanges;MESSANGES;21220;21;COTE D'OR;1
21408;;Messigny-et-Vantoux;MESSIGNY-ET-VANTOUX;21380;21;COTE D'OR;1
21409;;Meuilley;MEUILLEY;21700;21;COTE D'OR;1
21410;;Meulson;MEULSON;21510;21;COTE D'OR;1
21411;;Meursanges;MEURSANGES;21200;21;COTE D'OR;1
21412;;Meursault;MEURSAULT;21190;21;COTE D'OR;1
21413;;Millery;MILLERY;21140;21;COTE D'OR;1
21414;;Mimeure;MIMEURE;21230;21;COTE D'OR;1
21415;;Minot;MINOT;21510;21;COTE D'OR;1
21416;;Mirebeau-sur-Bze;MIREBEAU-SUR-BEZE;21310;21;COTE D'OR;1
21417;;Missery;MISSERY;21210;21;COTE D'OR;1
21418;;Moitron;MOITRON;21510;21;COTE D'OR;1
21419;;Molesme;MOLESME;21330;21;COTE D'OR;1
21420;;Molinot;MOLINOT;21340;21;COTE D'OR;1
21421;;Moloy;MOLOY;21120;21;COTE D'OR;1
21422;;Molphey;MOLPHEY;21210;21;COTE D'OR;1
21423;;Montagny-ls-Beaune;MONTAGNY-LES-BEAUNE;21200;21;COTE D'OR;1
21424;;Montagny-ls-Seurre;MONTAGNY-LES-SEURRE;21250;21;COTE D'OR;1
21425;;Montbard;MONTBARD;21500;21;COTE D'OR;1
21426;;Montberthault;MONTBERTHAULT;21460;21;COTE D'OR;1
21427;;Montceau-et-charnant;MONTCEAU-ET-ECHARNANT;21360;21;COTE D'OR;1
21428;;Monthelie;MONTHELIE;21190;21;COTE D'OR;1
21429;;Montigny-Montfort;MONTIGNY-MONTFORT;21500;21;COTE D'OR;1
21430;;Montigny-Saint-Barthlemy;MONTIGNY-SAINT-BARTHELEMY;21390;21;COTE D'OR;1
21431;;Montigny-sur-Armanon;MONTIGNY-SUR-ARMANCON;21140;21;COTE D'OR;1
21432;;Montigny-sur-Aube;MONTIGNY-SUR-AUBE;21520;21;COTE D'OR;1
21433;;Montigny-Mornay-Villeneuve-sur-Vingeanne;MONTIGNY-MORNAY-VILLENEUVE-SUR-VINGEANNE;21610;21;COTE D'OR;1
21434;;Montlay-en-Auxois;MONTLAY-EN-AUXOIS;21210;21;COTE D'OR;1
21435;;Montliot-et-Courcelles;MONTLIOT-ET-COURCELLES;21400;21;COTE D'OR;1
21436;;Montmain;MONTMAIN;21250;21;COTE D'OR;1
21437;;Montmanon;MONTMANCON;21270;21;COTE D'OR;1
21438;;Montmoyen;MONTMOYEN;21290;21;COTE D'OR;1
21439;;Montoillot;MONTOILLOT;21540;21;COTE D'OR;1
21440;;Montot;MONTOT;21170;21;COTE D'OR;1
21441;;Mont-Saint-Jean;MONT-SAINT-JEAN;21320;21;COTE D'OR;1
21442;;Morey-Saint-Denis;MOREY-SAINT-DENIS;21220;21;COTE D'OR;1
21444;;Mosson;MOSSON;21400;21;COTE D'OR;1
21445;La;Motte-Ternant;MOTTE-TERNANT;21210;21;COTE D'OR;1
21446;;Moutiers-Saint-Jean;MOUTIERS-SAINT-JEAN;21500;21;COTE D'OR;1
21447;;Musigny;MUSIGNY;21230;21;COTE D'OR;1
21448;;Mussy-la-Fosse;MUSSY-LA-FOSSE;21150;21;COTE D'OR;1
21449;;Nan-sous-Thil;NAN-SOUS-THIL;21390;21;COTE D'OR;1
21450;;Nantoux;NANTOUX;21190;21;COTE D'OR;1
21451;;Nesle-et-Massoult;NESLE-ET-MASSOULT;21330;21;COTE D'OR;1
21452;;Neuilly-ls-Dijon;NEUILLY-LES-DIJON;21800;21;COTE D'OR;1
21454;;Nicey;NICEY;21330;21;COTE D'OR;1
21455;;Nod-sur-Seine;NOD-SUR-SEINE;21400;21;COTE D'OR;1
21456;;Nogent-ls-Montbard;NOGENT-LES-MONTBARD;21500;21;COTE D'OR;1
21457;;Noidan;NOIDAN;21390;21;COTE D'OR;1
21458;;Noiron-sous-Gevrey;NOIRON-SOUS-GEVREY;21910;21;COTE D'OR;1
21459;;Noiron-sur-Bze;NOIRON-SUR-BEZE;21310;21;COTE D'OR;1
21460;;Noiron-sur-Seine;NOIRON-SUR-SEINE;21400;21;COTE D'OR;1
21461;;Nolay;NOLAY;21340;21;COTE D'OR;1
21462;;Norges-la-Ville;NORGES-LA-VILLE;21490;21;COTE D'OR;1
21463;;Normier;NORMIER;21390;21;COTE D'OR;1
21464;;Nuits-Saint-Georges;NUITS-SAINT-GEORGES;21700;21;COTE D'OR;1
21465;;Obtre;OBTREE;21400;21;COTE D'OR;1
21466;;Oigny;OIGNY;21450;21;COTE D'OR;1
21467;;Oisilly;OISILLY;21310;21;COTE D'OR;1
21468;;Orain;ORAIN;21610;21;COTE D'OR;1
21469;;Orgeux;ORGEUX;21490;21;COTE D'OR;1
21470;;Origny;ORIGNY;21510;21;COTE D'OR;1
21471;;Orret;ORRET;21450;21;COTE D'OR;1
21472;;Orville;ORVILLE;21260;21;COTE D'OR;1
21473;;Ouges;OUGES;21600;21;COTE D'OR;1
21474;;Pagny-la-Ville;PAGNY-LA-VILLE;21250;21;COTE D'OR;1
21475;;Pagny-le-Chteau;PAGNY-LE-CHATEAU;21250;21;COTE D'OR;1
21476;;Painblanc;PAINBLANC;21360;21;COTE D'OR;1
21477;;Panges;PANGES;21540;21;COTE D'OR;1
21478;;Pasques;PASQUES;21370;21;COTE D'OR;1
21479;;Pellerey;PELLEREY;21440;21;COTE D'OR;1
21480;;Pernand-Vergelesses;PERNAND-VERGELESSES;21420;21;COTE D'OR;1
21481;;Perrigny-ls-Dijon;PERRIGNY-LES-DIJON;21160;21;COTE D'OR;1
21482;;Perrigny-sur-l'Ognon;PERRIGNY-SUR-L'OGNON;21270;21;COTE D'OR;1
21483;;Pichanges;PICHANGES;21120;21;COTE D'OR;1
21484;;Planay;PLANAY;21500;21;COTE D'OR;1
21485;;Plombires-ls-Dijon;PLOMBIERES-LES-DIJON;21370;21;COTE D'OR;1
21486;;Pluvault;PLUVAULT;21110;21;COTE D'OR;1
21487;;Pluvet;PLUVET;21110;21;COTE D'OR;1
21488;;Poinon-ls-Larrey;POINCON-LES-LARREY;21330;21;COTE D'OR;1
21489;;Poiseul-la-Grange;POISEUL-LA-GRANGE;21440;21;COTE D'OR;1
21490;;Poiseul-la-Ville-et-Laperrire;POISEUL-LA-VILLE-ET-LAPERRIERE;21450;21;COTE D'OR;1
21491;;Poiseul-ls-Saulx;POISEUL-LES-SAULX;21120;21;COTE D'OR;1
21492;;Pommard;POMMARD;21630;21;COTE D'OR;1
21493;;Poncey-ls-Athe;PONCEY-LES-ATHEE;21130;21;COTE D'OR;1
21494;;Poncey-sur-l'Ignon;PONCEY-SUR-L'IGNON;21440;21;COTE D'OR;1
21495;;Pont;PONT;21130;21;COTE D'OR;1
21496;;Pontailler-sur-Sane;PONTAILLER-SUR-SAONE;21270;21;COTE D'OR;1
21497;;Pont-et-Massne;PONT-ET-MASSENE;21140;21;COTE D'OR;1
21498;;Posanges;POSANGES;21350;21;COTE D'OR;1
21499;;Pothires;POTHIERES;21400;21;COTE D'OR;1
21500;;Pouillenay;POUILLENAY;21150;21;COTE D'OR;1
21501;;Pouilly-en-Auxois;POUILLY-EN-AUXOIS;21320;21;COTE D'OR;1
21502;;Pouilly-sur-Sane;POUILLY-SUR-SAONE;21250;21;COTE D'OR;1
21503;;Pouilly-sur-Vingeanne;POUILLY-SUR-VINGEANNE;21610;21;COTE D'OR;1
21504;;Prlon;PRALON;21410;21;COTE D'OR;1
21505;;Prcy-sous-Thil;PRECY-SOUS-THIL;21390;21;COTE D'OR;1
21506;;Premeaux-Prissey;PREMEAUX-PRISSEY;21700;21;COTE D'OR;1
21507;;Premires;PREMIERES;21110;21;COTE D'OR;1
21508;;Prenois;PRENOIS;21370;21;COTE D'OR;1
21510;;Prusly-sur-Ource;PRUSLY-SUR-OURCE;21400;21;COTE D'OR;1
21511;;Puits;PUITS;21400;21;COTE D'OR;1
21512;;Puligny-Montrachet;PULIGNY-MONTRACHET;21190;21;COTE D'OR;1
21513;;Quemigny-Poisot;QUEMIGNY-POISOT;21220;21;COTE D'OR;1
21514;;Quemigny-sur-Seine;QUEMIGNY-SUR-SEINE;21510;21;COTE D'OR;1
21515;;Quetigny;QUETIGNY;21800;21;COTE D'OR;1
21516;;Quincerot;QUINCEROT;21500;21;COTE D'OR;1
21517;;Quincey;QUINCEY;21700;21;COTE D'OR;1
21518;;Quincy-le-Vicomte;QUINCY-LE-VICOMTE;21500;21;COTE D'OR;1
21519;;Recey-sur-Ource;RECEY-SUR-OURCE;21290;21;COTE D'OR;1
21520;;Remilly-en-Montagne;REMILLY-EN-MONTAGNE;21540;21;COTE D'OR;1
21521;;Remilly-sur-Tille;REMILLY-SUR-TILLE;21560;21;COTE D'OR;1
21522;;Renve;RENEVE;21310;21;COTE D'OR;1
21523;;Reulle-Vergy;REULLE-VERGY;21220;21;COTE D'OR;1
21524;;Riel-les-Eaux;RIEL-LES-EAUX;21570;21;COTE D'OR;1
21525;La;Roche-en-Brenil;ROCHE-EN-BRENIL;21530;21;COTE D'OR;1
21526;;Rochefort-sur-Brvon;ROCHEFORT-SUR-BREVON;21510;21;COTE D'OR;1
21527;La;Rochepot;ROCHEPOT;21340;21;COTE D'OR;1
21528;La;Roche-Vanneau;ROCHE-VANNEAU;21150;21;COTE D'OR;1
21529;;Roilly;ROILLY;21390;21;COTE D'OR;1
21530;;Rougemont;ROUGEMONT;21500;21;COTE D'OR;1
21531;;Rouvray;ROUVRAY;21530;21;COTE D'OR;1
21532;;Rouvres-en-Plaine;ROUVRES-EN-PLAINE;21110;21;COTE D'OR;1
21533;;Rouvres-sous-Meilly;ROUVRES-SOUS-MEILLY;21320;21;COTE D'OR;1
21534;;Ruffey-ls-Beaune;RUFFEY-LES-BEAUNE;21200;21;COTE D'OR;1
21535;;Ruffey-ls-Echirey;RUFFEY-LES-ECHIREY;21490;21;COTE D'OR;1
21536;;Sacquenay;SACQUENAY;21260;21;COTE D'OR;1
21537;;Saffres;SAFFRES;21350;21;COTE D'OR;1
21538;;Saint-Andeux;SAINT-ANDEUX;21530;21;COTE D'OR;1
21539;;Saint-Anthot;SAINT-ANTHOT;21540;21;COTE D'OR;1
21540;;Saint-Apollinaire;SAINT-APOLLINAIRE;21850;21;COTE D'OR;1
21541;;Saint-Aubin;SAINT-AUBIN;21190;21;COTE D'OR;1
21542;;Saint-Bernard;SAINT-BERNARD;21700;21;COTE D'OR;1
21543;;Saint-Broing-les-Moines;SAINT-BROING-LES-MOINES;21290;21;COTE D'OR;1
21544;;Sainte-Colombe;SAINTE-COLOMBE;21350;21;COTE D'OR;1
21545;;Sainte-Colombe-sur-Seine;SAINTE-COLOMBE-SUR-SEINE;21400;21;COTE D'OR;1
21546;;Saint-Didier;SAINT-DIDIER;21210;21;COTE D'OR;1
21547;;Saint-Euphrne;SAINT-EUPHRONE;21140;21;COTE D'OR;1
21548;;Saint-Germain-de-Modon;SAINT-GERMAIN-DE-MODEON;21530;21;COTE D'OR;1
21549;;Saint-Germain-le-Rocheux;SAINT-GERMAIN-LE-ROCHEUX;21510;21;COTE D'OR;1
21550;;Saint-Germain-ls-Senailly;SAINT-GERMAIN-LES-SENAILLY;21500;21;COTE D'OR;1
21552;;Saint-Hlier;SAINT-HELIER;21690;21;COTE D'OR;1
21553;;Saint-Jean-de-Buf;SAINT-JEAN-DE-BOEUF;21410;21;COTE D'OR;1
21554;;Saint-Jean-de-Losne;SAINT-JEAN-DE-LOSNE;21170;21;COTE D'OR;1
21555;;Saint-Julien;SAINT-JULIEN;21490;21;COTE D'OR;1
21556;;Saint-Lger-Triey;SAINT-LEGER-TRIEY;21270;21;COTE D'OR;1
21557;;Saint-Marc-sur-Seine;SAINT-MARC-SUR-SEINE;21450;21;COTE D'OR;1
21558;;Sainte-Marie-la-Blanche;SAINTE-MARIE-LA-BLANCHE;21200;21;COTE D'OR;1
21559;;Sainte-Marie-sur-Ouche;SAINTE-MARIE-SUR-OUCHE;21410;21;COTE D'OR;1
21560;;Saint-Martin-de-la-Mer;SAINT-MARTIN-DE-LA-MER;21210;21;COTE D'OR;1
21561;;Saint-Martin-du-Mont;SAINT-MARTIN-DU-MONT;21440;21;COTE D'OR;1
21562;;Saint-Maurice-sur-Vingeanne;SAINT-MAURICE-SUR-VINGEANNE;21610;21;COTE D'OR;1
21563;;Saint-Mesmin;SAINT-MESMIN;21540;21;COTE D'OR;1
21564;;Saint-Nicolas-ls-Cteaux;SAINT-NICOLAS-LES-CITEAUX;21700;21;COTE D'OR;1
21565;;Saint-Philibert;SAINT-PHILIBERT;21220;21;COTE D'OR;1
21566;;Saint-Pierre-en-Vaux;SAINT-PIERRE-EN-VAUX;21230;21;COTE D'OR;1
21567;;Saint-Prix-ls-Arnay;SAINT-PRIX-LES-ARNAY;21230;21;COTE D'OR;1
21568;;Saint-Rmy;SAINT-REMY;21500;21;COTE D'OR;1
21569;;Saint-Romain;SAINT-ROMAIN;21190;21;COTE D'OR;1
21570;;Sainte-Sabine;SAINTE-SABINE;21320;21;COTE D'OR;1
21571;;Saint-Sauveur;SAINT-SAUVEUR;21270;21;COTE D'OR;1
21572;;Saint-Seine-en-Bche;SAINT-SEINE-EN-BACHE;21130;21;COTE D'OR;1
21573;;Saint-Seine-l'Abbaye;SAINT-SEINE-L'ABBAYE;21440;21;COTE D'OR;1
21574;;Saint-Seine-sur-Vingeanne;SAINT-SEINE-SUR-VINGEANNE;21610;21;COTE D'OR;1
21575;;Saint-Symphorien-sur-Sane;SAINT-SYMPHORIEN-SUR-SAONE;21170;21;COTE D'OR;1
21576;;Saint-Thibault;SAINT-THIBAULT;21350;21;COTE D'OR;1
21577;;Saint-Usage;SAINT-USAGE;21170;21;COTE D'OR;1
21578;;Saint-Victor-sur-Ouche;SAINT-VICTOR-SUR-OUCHE;21410;21;COTE D'OR;1
21579;;Salives;SALIVES;21580;21;COTE D'OR;1
21580;;Salmaise;SALMAISE;21690;21;COTE D'OR;1
21581;;Samerey;SAMEREY;21170;21;COTE D'OR;1
21582;;Santenay;SANTENAY;21590;21;COTE D'OR;1
21583;;Santosse;SANTOSSE;21340;21;COTE D'OR;1
21584;;Saulieu;SAULIEU;21210;21;COTE D'OR;1
21585;;Saulon-la-Chapelle;SAULON-LA-CHAPELLE;21910;21;COTE D'OR;1
21586;;Saulon-la-Rue;SAULON-LA-RUE;21910;21;COTE D'OR;1
21587;;Saulx-le-Duc;SAULX-LE-DUC;21120;21;COTE D'OR;1
21588;;Saussey;SAUSSEY;21360;21;COTE D'OR;1
21589;;Saussy;SAUSSY;21380;21;COTE D'OR;1
21590;;Savigny-ls-Beaune;SAVIGNY-LES-BEAUNE;21420;21;COTE D'OR;1
21591;;Savigny-le-Sec;SAVIGNY-LE-SEC;21380;21;COTE D'OR;1
21592;;Savigny-sous-Mlain;SAVIGNY-SOUS-MALAIN;21540;21;COTE D'OR;1
21593;;Savilly;SAVILLY;21430;21;COTE D'OR;1
21594;;Savoisy;SAVOISY;21500;21;COTE D'OR;1
21595;;Savolles;SAVOLLES;21310;21;COTE D'OR;1
21596;;Savouges;SAVOUGES;21910;21;COTE D'OR;1
21597;;Segrois;SEGROIS;21220;21;COTE D'OR;1
21598;;Seigny;SEIGNY;21150;21;COTE D'OR;1
21599;;Selongey;SELONGEY;21260;21;COTE D'OR;1
21600;;Semarey;SEMAREY;21320;21;COTE D'OR;1
21601;;Semezanges;SEMEZANGES;21220;21;COTE D'OR;1
21602;;Semond;SEMOND;21450;21;COTE D'OR;1
21603;;Semur-en-Auxois;SEMUR-EN-AUXOIS;21140;21;COTE D'OR;1
21604;;Senailly;SENAILLY;21500;21;COTE D'OR;1
21605;;Sennecey-ls-Dijon;SENNECEY-LES-DIJON;21800;21;COTE D'OR;1
21606;;Ladoix-Serrigny;LADOIX-SERRIGNY;21550;21;COTE D'OR;1
21607;;Seurre;SEURRE;21250;21;COTE D'OR;1
21608;;Sincey-ls-Rouvray;SINCEY-LES-ROUVRAY;21530;21;COTE D'OR;1
21609;;Soirans;SOIRANS;21110;21;COTE D'OR;1
21610;;Soissons-sur-Nacey;SOISSONS-SUR-NACEY;21270;21;COTE D'OR;1
21611;;Sombernon;SOMBERNON;21540;21;COTE D'OR;1
21612;;Souhey;SOUHEY;21140;21;COTE D'OR;1
21613;;Soussey-sur-Brionne;SOUSSEY-SUR-BRIONNE;21350;21;COTE D'OR;1
21614;;Spoy;SPOY;21120;21;COTE D'OR;1
21615;;Sussey;SUSSEY;21430;21;COTE D'OR;1
21616;;Tailly;TAILLY;21190;21;COTE D'OR;1
21617;;Talant;TALANT;21240;21;COTE D'OR;1
21618;;Talmay;TALMAY;21270;21;COTE D'OR;1
21619;;Tanay;TANAY;21310;21;COTE D'OR;1
21620;;Tarsul;TARSUL;21120;21;COTE D'OR;1
21621;;Tart-l'Abbaye;TART-L'ABBAYE;21110;21;COTE D'OR;1
21622;;Tart-le-Bas;TART-LE-BAS;21110;21;COTE D'OR;1
21623;;Tart-le-Haut;TART-LE-HAUT;21110;21;COTE D'OR;1
21624;;Tellecey;TELLECEY;21270;21;COTE D'OR;1
21625;;Ternant;TERNANT;21220;21;COTE D'OR;1
21626;;Terrefondre;TERREFONDREE;21290;21;COTE D'OR;1
21627;;Thenissey;THENISSEY;21150;21;COTE D'OR;1
21628;;Thoires;THOIRES;21570;21;COTE D'OR;1
21629;;Thoisy-la-Berchre;THOISY-LA-BERCHERE;21210;21;COTE D'OR;1
21630;;Thoisy-le-Dsert;THOISY-LE-DESERT;21320;21;COTE D'OR;1
21631;;Thomirey;THOMIREY;21360;21;COTE D'OR;1
21632;;Thorey-en-Plaine;THOREY-EN-PLAINE;21110;21;COTE D'OR;1
21633;;Thorey-sous-Charny;THOREY-SOUS-CHARNY;21350;21;COTE D'OR;1
21634;;Thorey-sur-Ouche;THOREY-SUR-OUCHE;21360;21;COTE D'OR;1
21635;;Thoste;THOSTE;21460;21;COTE D'OR;1
21636;;Thury;THURY;21340;21;COTE D'OR;1
21637;;Tichey;TICHEY;21250;21;COTE D'OR;1
21638;;Til-Chtel;TIL-CHATEL;21120;21;COTE D'OR;1
21639;;Tillenay;TILLENAY;21130;21;COTE D'OR;1
21640;;Torcy-et-Pouligny;TORCY-ET-POULIGNY;21460;21;COTE D'OR;1
21641;;Touillon;TOUILLON;21500;21;COTE D'OR;1
21642;;Toutry;TOUTRY;21460;21;COTE D'OR;1
21643;;Trclun;TRECLUN;21130;21;COTE D'OR;1
21644;;Trochres;TROCHERES;21310;21;COTE D'OR;1
21645;;Trouhans;TROUHANS;21170;21;COTE D'OR;1
21646;;Trouhaut;TROUHAUT;21440;21;COTE D'OR;1
21647;;Trugny;TRUGNY;21250;21;COTE D'OR;1
21648;;Turcey;TURCEY;21540;21;COTE D'OR;1
21649;;Uncey-le-Franc;UNCEY-LE-FRANC;21350;21;COTE D'OR;1
21650;;Urcy;URCY;21220;21;COTE D'OR;1
21651;;Val-Suzon;VAL-SUZON;21121;21;COTE D'OR;1
21652;;Vandenesse-en-Auxois;VANDENESSE-EN-AUXOIS;21320;21;COTE D'OR;1
21653;;Vannaire;VANNAIRE;21400;21;COTE D'OR;1
21655;;Vanvey;VANVEY;21400;21;COTE D'OR;1
21656;;Varanges;VARANGES;21110;21;COTE D'OR;1
21657;;Varois-et-Chaignot;VAROIS-ET-CHAIGNOT;21490;21;COTE D'OR;1
21658;;Vauchignon;VAUCHIGNON;21340;21;COTE D'OR;1
21659;;Vaux-Saules;VAUX-SAULES;21440;21;COTE D'OR;1
21660;;Veilly;VEILLY;21360;21;COTE D'OR;1
21661;;Velars-sur-Ouche;VELARS-SUR-OUCHE;21370;21;COTE D'OR;1
21662;;Velogny;VELOGNY;21350;21;COTE D'OR;1
21663;;Venarey-les-Laumes;VENAREY-LES-LAUMES;21150;21;COTE D'OR;1
21664;;Verdonnet;VERDONNET;21330;21;COTE D'OR;1
21665;;Vernois-ls-Vesvres;VERNOIS-LES-VESVRES;21260;21;COTE D'OR;1
21666;;Vernot;VERNOT;21120;21;COTE D'OR;1
21667;;Vronnes;VERONNES;21260;21;COTE D'OR;1
21669;;Verrey-sous-Dre;VERREY-SOUS-DREE;21540;21;COTE D'OR;1
21670;;Verrey-sous-Salmaise;VERREY-SOUS-SALMAISE;21690;21;COTE D'OR;1
21671;;Vertault;VERTAULT;21330;21;COTE D'OR;1
21672;;Vesvres;VESVRES;21350;21;COTE D'OR;1
21673;;Veuvey-sur-Ouche;VEUVEY-SUR-OUCHE;21360;21;COTE D'OR;1
21674;;Veuxhaulles-sur-Aube;VEUXHAULLES-SUR-AUBE;21520;21;COTE D'OR;1
21675;;Vianges;VIANGES;21430;21;COTE D'OR;1
21676;;Vic-de-Chassenay;VIC-DE-CHASSENAY;21140;21;COTE D'OR;1
21677;;Vic-des-Prs;VIC-DES-PRES;21360;21;COTE D'OR;1
21678;;Vic-sous-Thil;VIC-SOUS-THIL;21390;21;COTE D'OR;1
21679;;Vieilmoulin;VIEILMOULIN;21540;21;COTE D'OR;1
21680;;Vielverge;VIELVERGE;21270;21;COTE D'OR;1
21681;;Vieux-Chteau;VIEUX-CHATEAU;21460;21;COTE D'OR;1
21682;;Vivigne;VIEVIGNE;21310;21;COTE D'OR;1
21683;;Vivy;VIEVY;21230;21;COTE D'OR;1
21684;;Vignoles;VIGNOLES;21200;21;COTE D'OR;1
21685;;Villaines-en-Duesmois;VILLAINES-EN-DUESMOIS;21450;21;COTE D'OR;1
21686;;Villaines-les-Prvtes;VILLAINES-LES-PREVOTES;21500;21;COTE D'OR;1
21687;;Villargoix;VILLARGOIX;21210;21;COTE D'OR;1
21688;;Villars-Fontaine;VILLARS-FONTAINE;21700;21;COTE D'OR;1
21689;;Villars-et-Villenotte;VILLARS-ET-VILLENOTTE;21140;21;COTE D'OR;1
21690;;Villeberny;VILLEBERNY;21350;21;COTE D'OR;1
21691;;Villebichot;VILLEBICHOT;21700;21;COTE D'OR;1
21692;;Villecomte;VILLECOMTE;21120;21;COTE D'OR;1
21693;;Villedieu;VILLEDIEU;21330;21;COTE D'OR;1
21694;;Villeferry;VILLEFERRY;21350;21;COTE D'OR;1
21695;La;Villeneuve-les-Convers;VILLENEUVE-LES-CONVERS;21450;21;COTE D'OR;1
21696;;Villeneuve-sous-Charigny;VILLENEUVE-SOUS-CHARIGNY;21140;21;COTE D'OR;1
21698;;Villers-la-Faye;VILLERS-LA-FAYE;21700;21;COTE D'OR;1
21699;;Villers-les-Pots;VILLERS-LES-POTS;21130;21;COTE D'OR;1
21700;;Villers-Patras;VILLERS-PATRAS;21400;21;COTE D'OR;1
21701;;Villers-Rotin;VILLERS-ROTIN;21130;21;COTE D'OR;1
21702;;Villey-sur-Tille;VILLEY-SUR-TILLE;21120;21;COTE D'OR;1
21703;;Villiers-en-Morvan;VILLIERS-EN-MORVAN;21430;21;COTE D'OR;1
21704;;Villiers-le-Duc;VILLIERS-LE-DUC;21400;21;COTE D'OR;1
21705;;Villotte-Saint-Seine;VILLOTTE-SAINT-SEINE;21690;21;COTE D'OR;1
21706;;Villotte-sur-Ource;VILLOTTE-SUR-OURCE;21400;21;COTE D'OR;1
21707;;Villy-en-Auxois;VILLY-EN-AUXOIS;21350;21;COTE D'OR;1
21708;;Villy-le-Moutier;VILLY-LE-MOUTIER;21250;21;COTE D'OR;1
21709;;Viserny;VISERNY;21500;21;COTE D'OR;1
21710;;Vitteaux;VITTEAUX;21350;21;COTE D'OR;1
21711;;Vix;VIX;21400;21;COTE D'OR;1
21712;;Volnay;VOLNAY;21190;21;COTE D'OR;1
21713;;Vonges;VONGES;21270;21;COTE D'OR;1
21714;;Vosne-Romane;VOSNE-ROMANEE;21700;21;COTE D'OR;1
21715;;Voudenay;VOUDENAY;21230;21;COTE D'OR;1
21716;;Vougeot;VOUGEOT;21640;21;COTE D'OR;1
21717;;Voulaines-les-Templiers;VOULAINES-LES-TEMPLIERS;21290;21;COTE D'OR;1
22001;;Allineuc;ALLINEUC;22460;22;COTES D'ARMOR;1
22002;;Andel;ANDEL;22400;22;COTES D'ARMOR;1
22003;;Aucaleuc;AUCALEUC;22100;22;COTES D'ARMOR;1
22004;;Bgard;BEGARD;22140;22;COTES D'ARMOR;1
22005;;Belle-Isle-en-Terre;BELLE-ISLE-EN-TERRE;22810;22;COTES D'ARMOR;1
22006;;Berhet;BERHET;22140;22;COTES D'ARMOR;1
22007;;Binic;BINIC;22520;22;COTES D'ARMOR;1
22008;;Bobital;BOBITAL;22100;22;COTES D'ARMOR;1
22009;Le;Bodo;BODEO;22320;22;COTES D'ARMOR;1
22011;;Boqueho;BOQUEHO;22170;22;COTES D'ARMOR;1
22012;La;Bouillie;BOUILLIE;22240;22;COTES D'ARMOR;1
22013;;Bourbriac;BOURBRIAC;22390;22;COTES D'ARMOR;1
22014;;Bourseul;BOURSEUL;22130;22;COTES D'ARMOR;1
22015;;Brhand;BREHAND;22510;22;COTES D'ARMOR;1
22016;;le-de-Brhat;ILE-DE-BREHAT;22870;22;COTES D'ARMOR;1
22018;;Brlidy;BRELIDY;22140;22;COTES D'ARMOR;1
22019;;Bringolo;BRINGOLO;22170;22;COTES D'ARMOR;1
22020;;Broons;BROONS;22250;22;COTES D'ARMOR;1
22021;;Brusvily;BRUSVILY;22100;22;COTES D'ARMOR;1
22023;;Bulat-Pestivien;BULAT-PESTIVIEN;22160;22;COTES D'ARMOR;1
22024;;Calanhel;CALANHEL;22160;22;COTES D'ARMOR;1
22025;;Callac;CALLAC;22160;22;COTES D'ARMOR;1
22026;;Calorguen;CALORGUEN;22100;22;COTES D'ARMOR;1
22027;Le;Cambout;CAMBOUT;22210;22;COTES D'ARMOR;1
22028;;Camlez;CAMLEZ;22450;22;COTES D'ARMOR;1
22029;;Canihuel;CANIHUEL;22480;22;COTES D'ARMOR;1
22030;;Caounnec-Lanvzac;CAOUENNEC-LANVEZEAC;22300;22;COTES D'ARMOR;1
22031;;Carnot;CARNOET;22160;22;COTES D'ARMOR;1
22032;;Caulnes;CAULNES;22350;22;COTES D'ARMOR;1
22033;;Caurel;CAUREL;22530;22;COTES D'ARMOR;1
22034;;Cavan;CAVAN;22140;22;COTES D'ARMOR;1
22035;Les;Champs-Graux;CHAMPS-GERAUX;22630;22;COTES D'ARMOR;1
22036;La;Chapelle-Blanche;CHAPELLE-BLANCHE;22350;22;COTES D'ARMOR;1
22037;La;Chapelle-Neuve;CHAPELLE-NEUVE;22160;22;COTES D'ARMOR;1
22038;;Chtelaudren;CHATELAUDREN;22170;22;COTES D'ARMOR;1
22039;La;Chze;CHEZE;22210;22;COTES D'ARMOR;1
22040;;Coadout;COADOUT;22970;22;COTES D'ARMOR;1
22041;;Coatascorn;COATASCORN;22140;22;COTES D'ARMOR;1
22042;;Coatrven;COATREVEN;22450;22;COTES D'ARMOR;1
22043;;Cotlogon;COETLOGON;22210;22;COTES D'ARMOR;1
22044;;Cotmieux;COETMIEUX;22400;22;COTES D'ARMOR;1
22045;;Cohiniac;COHINIAC;22800;22;COTES D'ARMOR;1
22046;;Colline;COLLINEE;22330;22;COTES D'ARMOR;1
22047;;Corlay;CORLAY;22320;22;COTES D'ARMOR;1
22048;;Corseul;CORSEUL;22130;22;COTES D'ARMOR;1
22049;;Crhen;CREHEN;22130;22;COTES D'ARMOR;1
22050;;Dinan;DINAN;22100;22;COTES D'ARMOR;1
22051;;Dolo;DOLO;22270;22;COTES D'ARMOR;1
22052;;Duault;DUAULT;22160;22;COTES D'ARMOR;1
22053;;rac;EREAC;22250;22;COTES D'ARMOR;1
22054;;Erquy;ERQUY;22430;22;COTES D'ARMOR;1
22055;;tables-sur-Mer;ETABLES-SUR-MER;22680;22;COTES D'ARMOR;1
22056;;vran;EVRAN;22630;22;COTES D'ARMOR;1
22057;Le;Faout;FAOUET;22290;22;COTES D'ARMOR;1
22058;La;Ferrire;FERRIERE;22210;22;COTES D'ARMOR;1
22059;Le;Fil;FOEIL;22800;22;COTES D'ARMOR;1
22060;;Gausson;GAUSSON;22150;22;COTES D'ARMOR;1
22061;;Glomel;GLOMEL;22110;22;COTES D'ARMOR;1
22062;;Gomen;GOMENE;22230;22;COTES D'ARMOR;1
22063;;Gommenec'h;GOMMENEC'H;22290;22;COTES D'ARMOR;1
22064;;Gouarec;GOUAREC;22570;22;COTES D'ARMOR;1
22065;;Goudelin;GOUDELIN;22290;22;COTES D'ARMOR;1
22066;Le;Gouray;GOURAY;22330;22;COTES D'ARMOR;1
22067;;Grces;GRACES;22200;22;COTES D'ARMOR;1
22068;;Grce-Uzel;GRACE-UZEL;22460;22;COTES D'ARMOR;1
22069;;Guenroc;GUENROC;22350;22;COTES D'ARMOR;1
22070;;Guingamp;GUINGAMP;22200;22;COTES D'ARMOR;1
22071;;Guitt;GUITTE;22350;22;COTES D'ARMOR;1
22072;;Gurunhuel;GURUNHUEL;22390;22;COTES D'ARMOR;1
22073;La;Harmoye;HARMOYE;22320;22;COTES D'ARMOR;1
22074;Le;Haut-Corlay;HAUT-CORLAY;22320;22;COTES D'ARMOR;1
22075;;Hmonstoir;HEMONSTOIR;22600;22;COTES D'ARMOR;1
22076;;Hnanbihen;HENANBIHEN;22550;22;COTES D'ARMOR;1
22077;;Hnansal;HENANSAL;22400;22;COTES D'ARMOR;1
22078;;Hengoat;HENGOAT;22450;22;COTES D'ARMOR;1
22079;;Hnon;HENON;22150;22;COTES D'ARMOR;1
22080;L';Hermitage-Lorge;HERMITAGE-LORGE;22150;22;COTES D'ARMOR;1
22081;;Hillion;HILLION;22120;22;COTES D'ARMOR;1
22082;Le;Hingl;HINGLE;22100;22;COTES D'ARMOR;1
22083;;Illifaut;ILLIFAUT;22230;22;COTES D'ARMOR;1
22084;;Jugon-les-Lacs;JUGON-LES-LACS;22270;22;COTES D'ARMOR;1
22085;;Kerbors;KERBORS;22610;22;COTES D'ARMOR;1
22086;;Kerfot;KERFOT;22500;22;COTES D'ARMOR;1
22087;;Kergrist-Molou;KERGRIST-MOELOU;22110;22;COTES D'ARMOR;1
22088;;Kerien;KERIEN;22480;22;COTES D'ARMOR;1
22090;;Kermaria-Sulard;KERMARIA-SULARD;22450;22;COTES D'ARMOR;1
22091;;Kermoroc'h;KERMOROC'H;22140;22;COTES D'ARMOR;1
22092;;Kerpert;KERPERT;22480;22;COTES D'ARMOR;1
22093;;Lamballe;LAMBALLE;22400;22;COTES D'ARMOR;1
22094;;Lancieux;LANCIEUX;22770;22;COTES D'ARMOR;1
22095;;Landebaron;LANDEBAERON;22140;22;COTES D'ARMOR;1
22096;;Landbia;LANDEBIA;22130;22;COTES D'ARMOR;1
22097;La;Landec;LANDEC;22980;22;COTES D'ARMOR;1
22098;;Landhen;LANDEHEN;22400;22;COTES D'ARMOR;1
22099;;Lanfains;LANFAINS;22800;22;COTES D'ARMOR;1
22100;;Langast;LANGAST;22150;22;COTES D'ARMOR;1
22101;;Langoat;LANGOAT;22450;22;COTES D'ARMOR;1
22102;;Langourla;LANGOURLA;22330;22;COTES D'ARMOR;1
22103;;Langrolay-sur-Rance;LANGROLAY-SUR-RANCE;22490;22;COTES D'ARMOR;1
22104;;Langudias;LANGUEDIAS;22980;22;COTES D'ARMOR;1
22105;;Languenan;LANGUENAN;22130;22;COTES D'ARMOR;1
22106;;Langueux;LANGUEUX;22360;22;COTES D'ARMOR;1
22107;;Laniscat;LANISCAT;22570;22;COTES D'ARMOR;1
22108;;Lanleff;LANLEFF;22290;22;COTES D'ARMOR;1
22109;;Lanloup;LANLOUP;22580;22;COTES D'ARMOR;1
22110;;Lanmrin;LANMERIN;22300;22;COTES D'ARMOR;1
22111;;Lanmodez;LANMODEZ;22610;22;COTES D'ARMOR;1
22112;;Lannebert;LANNEBERT;22290;22;COTES D'ARMOR;1
22113;;Lannion;LANNION;22300;22;COTES D'ARMOR;1
22114;;Lanrelas;LANRELAS;22250;22;COTES D'ARMOR;1
22115;;Lanrivain;LANRIVAIN;22480;22;COTES D'ARMOR;1
22116;;Lanrodec;LANRODEC;22170;22;COTES D'ARMOR;1
22117;;Lantic;LANTIC;22410;22;COTES D'ARMOR;1
22118;;Lanvallay;LANVALLAY;22100;22;COTES D'ARMOR;1
22119;;Lanvellec;LANVELLEC;22420;22;COTES D'ARMOR;1
22121;;Lanvollon;LANVOLLON;22290;22;COTES D'ARMOR;1
22122;;Laurenan;LAURENAN;22230;22;COTES D'ARMOR;1
22123;;Lhon;LEHON;22100;22;COTES D'ARMOR;1
22124;;Lescout-Gouarec;LESCOUET-GOUAREC;22570;22;COTES D'ARMOR;1
22126;Le;Leslay;LESLAY;22800;22;COTES D'ARMOR;1
22127;;Lzardrieux;LEZARDRIEUX;22740;22;COTES D'ARMOR;1
22128;;Locarn;LOCARN;22340;22;COTES D'ARMOR;1
22129;;Loc-Envel;LOC-ENVEL;22810;22;COTES D'ARMOR;1
22131;;Loguivy-Plougras;LOGUIVY-PLOUGRAS;22780;22;COTES D'ARMOR;1
22132;;Lohuec;LOHUEC;22160;22;COTES D'ARMOR;1
22133;;Loscout-sur-Meu;LOSCOUET-SUR-MEU;22230;22;COTES D'ARMOR;1
22134;;Louannec;LOUANNEC;22700;22;COTES D'ARMOR;1
22135;;Louargat;LOUARGAT;22540;22;COTES D'ARMOR;1
22136;;Loudac;LOUDEAC;22600;22;COTES D'ARMOR;1
22137;;Mal-Carhaix;MAEL-CARHAIX;22340;22;COTES D'ARMOR;1
22138;;Mal-Pestivien;MAEL-PESTIVIEN;22160;22;COTES D'ARMOR;1
22139;;Magoar;MAGOAR;22480;22;COTES D'ARMOR;1
22140;La;Malhoure;MALHOURE;22640;22;COTES D'ARMOR;1
22141;;Mantallot;MANTALLOT;22450;22;COTES D'ARMOR;1
22143;;Matignon;MATIGNON;22550;22;COTES D'ARMOR;1
22144;La;Maugon;MEAUGON;22440;22;COTES D'ARMOR;1
22145;;Mgrit;MEGRIT;22270;22;COTES D'ARMOR;1
22146;;Mellionnec;MELLIONNEC;22110;22;COTES D'ARMOR;1
22147;;Merdrignac;MERDRIGNAC;22230;22;COTES D'ARMOR;1
22148;;Mrillac;MERILLAC;22230;22;COTES D'ARMOR;1
22149;;Merlac;MERLEAC;22460;22;COTES D'ARMOR;1
22150;Le;Merzer;MERZER;22200;22;COTES D'ARMOR;1
22151;;Meslin;MESLIN;22400;22;COTES D'ARMOR;1
22152;;Minihy-Trguier;MINIHY-TREGUIER;22220;22;COTES D'ARMOR;1
22153;;Moncontour;MONCONTOUR;22510;22;COTES D'ARMOR;1
22154;;Morieux;MORIEUX;22400;22;COTES D'ARMOR;1
22155;La;Motte;MOTTE;22600;22;COTES D'ARMOR;1
22156;;Moustru;MOUSTERU;22200;22;COTES D'ARMOR;1
22157;Le;Moustoir;MOUSTOIR;22340;22;COTES D'ARMOR;1
22158;;Mr-de-Bretagne;MUR-DE-BRETAGNE;22530;22;COTES D'ARMOR;1
22160;;Noyal;NOYAL;22400;22;COTES D'ARMOR;1
22161;;Pabu;PABU;22200;22;COTES D'ARMOR;1
22162;;Paimpol;PAIMPOL;22500;22;COTES D'ARMOR;1
22163;;Paule;PAULE;22340;22;COTES D'ARMOR;1
22164;;Pdernec;PEDERNEC;22540;22;COTES D'ARMOR;1
22165;;Penguily;PENGUILY;22510;22;COTES D'ARMOR;1
22166;;Penvnan;PENVENAN;22710;22;COTES D'ARMOR;1
22167;;Perret;PERRET;22570;22;COTES D'ARMOR;1
22168;;Perros-Guirec;PERROS-GUIREC;22700;22;COTES D'ARMOR;1
22169;;Peumerit-Quintin;PEUMERIT-QUINTIN;22480;22;COTES D'ARMOR;1
22170;;Plaine-Haute;PLAINE-HAUTE;22800;22;COTES D'ARMOR;1
22171;;Plaintel;PLAINTEL;22940;22;COTES D'ARMOR;1
22172;;Plancot;PLANCOET;22130;22;COTES D'ARMOR;1
22173;;Planguenoual;PLANGUENOUAL;22400;22;COTES D'ARMOR;1
22174;;Plboulle;PLEBOULLE;22550;22;COTES D'ARMOR;1
22175;;Pldliac;PLEDELIAC;22270;22;COTES D'ARMOR;1
22176;;Pldran;PLEDRAN;22960;22;COTES D'ARMOR;1
22177;;Plguien;PLEGUIEN;22290;22;COTES D'ARMOR;1
22178;;Plhdel;PLEHEDEL;22290;22;COTES D'ARMOR;1
22179;;Frhel;FREHEL;22240;22;COTES D'ARMOR;1
22180;;Pllan-le-Petit;PLELAN-LE-PETIT;22980;22;COTES D'ARMOR;1
22181;;Pllauff;PLELAUFF;22570;22;COTES D'ARMOR;1
22182;;Pllo;PLELO;22170;22;COTES D'ARMOR;1
22183;;Plmet;PLEMET;22210;22;COTES D'ARMOR;1
22184;;Plmy;PLEMY;22150;22;COTES D'ARMOR;1
22185;;Plne-Jugon;PLENEE-JUGON;22640;22;COTES D'ARMOR;1
22186;;Plneuf-Val-Andr;PLENEUF-VAL-ANDRE;22370;22;COTES D'ARMOR;1
22187;;Plrin;PLERIN;22190;22;COTES D'ARMOR;1
22188;;Plerneuf;PLERNEUF;22170;22;COTES D'ARMOR;1
22189;;Plsidy;PLESIDY;22720;22;COTES D'ARMOR;1
22190;;Pleslin-Trigavou;PLESLIN-TRIGAVOU;22490;22;COTES D'ARMOR;1
22191;;Plessala;PLESSALA;22330;22;COTES D'ARMOR;1
22192;;Plessix-Balisson;PLESSIX-BALISSON;22650;22;COTES D'ARMOR;1
22193;;Plestan;PLESTAN;22640;22;COTES D'ARMOR;1
22194;;Plestin-les-Grves;PLESTIN-LES-GREVES;22310;22;COTES D'ARMOR;1
22195;;Pleubian;PLEUBIAN;22610;22;COTES D'ARMOR;1
22196;;Pleudaniel;PLEUDANIEL;22740;22;COTES D'ARMOR;1
22197;;Pleudihen-sur-Rance;PLEUDIHEN-SUR-RANCE;22690;22;COTES D'ARMOR;1
22198;;Pleumeur-Bodou;PLEUMEUR-BODOU;22560;22;COTES D'ARMOR;1
22199;;Pleumeur-Gautier;PLEUMEUR-GAUTIER;22740;22;COTES D'ARMOR;1
22200;;Plven;PLEVEN;22130;22;COTES D'ARMOR;1
22201;;Plvenon;PLEVENON;22240;22;COTES D'ARMOR;1
22202;;Plvin;PLEVIN;22340;22;COTES D'ARMOR;1
22203;;Pluc-sur-Li;PLOEUC-SUR-LIE;22150;22;COTES D'ARMOR;1
22204;;Plozal;PLOEZAL;22260;22;COTES D'ARMOR;1
22205;;Plorec-sur-Arguenon;PLOREC-SUR-ARGUENON;22130;22;COTES D'ARMOR;1
22206;;Plouagat;PLOUAGAT;22170;22;COTES D'ARMOR;1
22207;;Plouaret;PLOUARET;22420;22;COTES D'ARMOR;1
22208;;Plouasne;PLOUASNE;22830;22;COTES D'ARMOR;1
22209;;Ploubalay;PLOUBALAY;22650;22;COTES D'ARMOR;1
22210;;Ploubazlanec;PLOUBAZLANEC;22620;22;COTES D'ARMOR;1
22211;;Ploubezre;PLOUBEZRE;22300;22;COTES D'ARMOR;1
22212;;Plouc-du-Trieux;PLOUEC-DU-TRIEUX;22260;22;COTES D'ARMOR;1
22213;;Plour-sur-Rance;PLOUER-SUR-RANCE;22490;22;COTES D'ARMOR;1
22214;;Plouzec;PLOUEZEC;22470;22;COTES D'ARMOR;1
22215;;Ploufragan;PLOUFRAGAN;22440;22;COTES D'ARMOR;1
22216;;Plougonver;PLOUGONVER;22810;22;COTES D'ARMOR;1
22217;;Plougras;PLOUGRAS;22780;22;COTES D'ARMOR;1
22218;;Plougrescant;PLOUGRESCANT;22820;22;COTES D'ARMOR;1
22219;;Plouguenast;PLOUGUENAST;22150;22;COTES D'ARMOR;1
22220;;Plouguernvel;PLOUGUERNEVEL;22110;22;COTES D'ARMOR;1
22221;;Plouguiel;PLOUGUIEL;22220;22;COTES D'ARMOR;1
22222;;Plouha;PLOUHA;22580;22;COTES D'ARMOR;1
22223;;Plouisy;PLOUISY;22200;22;COTES D'ARMOR;1
22224;;Ploulec'h;PLOULEC'H;22300;22;COTES D'ARMOR;1
22225;;Ploumagoar;PLOUMAGOAR;22970;22;COTES D'ARMOR;1
22226;;Ploumilliau;PLOUMILLIAU;22300;22;COTES D'ARMOR;1
22227;;Plounrin;PLOUNERIN;22780;22;COTES D'ARMOR;1
22228;;Plounvez-Modec;PLOUNEVEZ-MOEDEC;22810;22;COTES D'ARMOR;1
22229;;Plounvez-Quintin;PLOUNEVEZ-QUINTIN;22110;22;COTES D'ARMOR;1
22231;;Plourac'h;PLOURAC'H;22160;22;COTES D'ARMOR;1
22232;;Plourhan;PLOURHAN;22410;22;COTES D'ARMOR;1
22233;;Plourivo;PLOURIVO;22860;22;COTES D'ARMOR;1
22234;;Plouvara;PLOUVARA;22170;22;COTES D'ARMOR;1
22235;;Plouzlambre;PLOUZELAMBRE;22420;22;COTES D'ARMOR;1
22236;;Pludual;PLUDUAL;22290;22;COTES D'ARMOR;1
22237;;Pluduno;PLUDUNO;22130;22;COTES D'ARMOR;1
22238;;Plufur;PLUFUR;22310;22;COTES D'ARMOR;1
22239;;Plumaudan;PLUMAUDAN;22350;22;COTES D'ARMOR;1
22240;;Plumaugat;PLUMAUGAT;22250;22;COTES D'ARMOR;1
22241;;Plumieux;PLUMIEUX;22210;22;COTES D'ARMOR;1
22242;;Plurien;PLURIEN;22240;22;COTES D'ARMOR;1
22243;;Plusquellec;PLUSQUELLEC;22160;22;COTES D'ARMOR;1
22244;;Plussulien;PLUSSULIEN;22320;22;COTES D'ARMOR;1
22245;;Pluzunet;PLUZUNET;22140;22;COTES D'ARMOR;1
22246;;Pommeret;POMMERET;22120;22;COTES D'ARMOR;1
22247;;Pommerit-Jaudy;POMMERIT-JAUDY;22450;22;COTES D'ARMOR;1
22248;;Pommerit-le-Vicomte;POMMERIT-LE-VICOMTE;22200;22;COTES D'ARMOR;1
22249;;Pont-Melvez;PONT-MELVEZ;22390;22;COTES D'ARMOR;1
22250;;Pontrieux;PONTRIEUX;22260;22;COTES D'ARMOR;1
22251;;Pordic;PORDIC;22590;22;COTES D'ARMOR;1
22253;;Pouldouran;POULDOURAN;22450;22;COTES D'ARMOR;1
22254;;Prat;PRAT;22140;22;COTES D'ARMOR;1
22255;La;Prnessaye;PRENESSAYE;22210;22;COTES D'ARMOR;1
22256;;Quemper-Guzennec;QUEMPER-GUEZENNEC;22260;22;COTES D'ARMOR;1
22257;;Quemperven;QUEMPERVEN;22450;22;COTES D'ARMOR;1
22258;;Quessoy;QUESSOY;22120;22;COTES D'ARMOR;1
22259;;Quvert;QUEVERT;22100;22;COTES D'ARMOR;1
22260;Le;Quillio;QUILLIO;22460;22;COTES D'ARMOR;1
22261;;Quintenic;QUINTENIC;22400;22;COTES D'ARMOR;1
22262;;Quintin;QUINTIN;22800;22;COTES D'ARMOR;1
22263;Le;Quiou;QUIOU;22630;22;COTES D'ARMOR;1
22264;La;Roche-Derrien;ROCHE-DERRIEN;22450;22;COTES D'ARMOR;1
22265;;Rospez;ROSPEZ;22300;22;COTES D'ARMOR;1
22266;;Rostrenen;ROSTRENEN;22110;22;COTES D'ARMOR;1
22267;;Rouillac;ROUILLAC;22250;22;COTES D'ARMOR;1
22268;;Ruca;RUCA;22550;22;COTES D'ARMOR;1
22269;;Runan;RUNAN;22260;22;COTES D'ARMOR;1
22271;;Saint-Adrien;SAINT-ADRIEN;22390;22;COTES D'ARMOR;1
22272;;Saint-Agathon;SAINT-AGATHON;22200;22;COTES D'ARMOR;1
22273;;Saint-Alban;SAINT-ALBAN;22400;22;COTES D'ARMOR;1
22274;;Saint-Andr-des-Eaux;SAINT-ANDRE-DES-EAUX;22630;22;COTES D'ARMOR;1
22275;;Saint-Barnab;SAINT-BARNABE;22600;22;COTES D'ARMOR;1
22276;;Saint-Bihy;SAINT-BIHY;22800;22;COTES D'ARMOR;1
22277;;Saint-Brandan;SAINT-BRANDAN;22800;22;COTES D'ARMOR;1
22278;;Saint-Brieuc;SAINT-BRIEUC;22000;22;COTES D'ARMOR;1
22279;;Saint-Caradec;SAINT-CARADEC;22600;22;COTES D'ARMOR;1
22280;;Saint-Carn;SAINT-CARNE;22100;22;COTES D'ARMOR;1
22281;;Saint-Carreuc;SAINT-CARREUC;22150;22;COTES D'ARMOR;1
22282;;Saint-Cast-le-Guildo;SAINT-CAST-LE-GUILDO;22380;22;COTES D'ARMOR;1
22283;;Saint-Clet;SAINT-CLET;22260;22;COTES D'ARMOR;1
22284;;Saint-Connan;SAINT-CONNAN;22480;22;COTES D'ARMOR;1
22285;;Saint-Connec;SAINT-CONNEC;22530;22;COTES D'ARMOR;1
22286;;Saint-Denoual;SAINT-DENOUAL;22400;22;COTES D'ARMOR;1
22287;;Saint-Donan;SAINT-DONAN;22800;22;COTES D'ARMOR;1
22288;;Saint-tienne-du-Gu-de-l'Isle;SAINT-ETIENNE-DU-GUE-DE-L'ISLE;22210;22;COTES D'ARMOR;1
22289;;Saint-Fiacre;SAINT-FIACRE;22720;22;COTES D'ARMOR;1
22290;;Saint-Gelven;SAINT-GELVEN;22570;22;COTES D'ARMOR;1
22291;;Saint-Gildas;SAINT-GILDAS;22800;22;COTES D'ARMOR;1
22292;;Saint-Gilles-du-Men;SAINT-GILLES-DU-MENE;22330;22;COTES D'ARMOR;1
22293;;Saint-Gilles-les-Bois;SAINT-GILLES-LES-BOIS;22290;22;COTES D'ARMOR;1
22294;;Saint-Gilles-Pligeaux;SAINT-GILLES-PLIGEAUX;22480;22;COTES D'ARMOR;1
22295;;Saint-Gilles-Vieux-March;SAINT-GILLES-VIEUX-MARCHE;22530;22;COTES D'ARMOR;1
22296;;Saint-Glen;SAINT-GLEN;22510;22;COTES D'ARMOR;1
22297;;Saint-Gouno;SAINT-GOUENO;22330;22;COTES D'ARMOR;1
22298;;Saint-Guen;SAINT-GUEN;22530;22;COTES D'ARMOR;1
22299;;Saint-Hlen;SAINT-HELEN;22100;22;COTES D'ARMOR;1
22300;;Saint-Herv;SAINT-HERVE;22460;22;COTES D'ARMOR;1
22302;;Saint-Jacut-de-la-Mer;SAINT-JACUT-DE-LA-MER;22750;22;COTES D'ARMOR;1
22303;;Saint-Jacut-du-Men;SAINT-JACUT-DU-MENE;22330;22;COTES D'ARMOR;1
22304;;Saint-Jean-Kerdaniel;SAINT-JEAN-KERDANIEL;22170;22;COTES D'ARMOR;1
22305;;Saint-Jouan-de-l'Isle;SAINT-JOUAN-DE-L'ISLE;22350;22;COTES D'ARMOR;1
22306;;Saint-Judoce;SAINT-JUDOCE;22630;22;COTES D'ARMOR;1
22307;;Saint-Julien;SAINT-JULIEN;22940;22;COTES D'ARMOR;1
22308;;Saint-Juvat;SAINT-JUVAT;22630;22;COTES D'ARMOR;1
22309;;Saint-Launeuc;SAINT-LAUNEUC;22230;22;COTES D'ARMOR;1
22310;;Saint-Laurent;SAINT-LAURENT;22140;22;COTES D'ARMOR;1
22311;;Saint-Lormel;SAINT-LORMEL;22130;22;COTES D'ARMOR;1
22312;;Saint-Maden;SAINT-MADEN;22350;22;COTES D'ARMOR;1
22313;;Saint-Martin-des-Prs;SAINT-MARTIN-DES-PRES;22320;22;COTES D'ARMOR;1
22314;;Saint-Maudan;SAINT-MAUDAN;22600;22;COTES D'ARMOR;1
22315;;Saint-Maudez;SAINT-MAUDEZ;22980;22;COTES D'ARMOR;1
22316;;Saint-Mayeux;SAINT-MAYEUX;22320;22;COTES D'ARMOR;1
22317;;Saint-Mloir-des-Bois;SAINT-MELOIR-DES-BOIS;22980;22;COTES D'ARMOR;1
22318;;Saint-Michel-de-Pllan;SAINT-MICHEL-DE-PLELAN;22980;22;COTES D'ARMOR;1
22319;;Saint-Michel-en-Grve;SAINT-MICHEL-EN-GREVE;22300;22;COTES D'ARMOR;1
22320;;Saint-Nicodme;SAINT-NICODEME;22160;22;COTES D'ARMOR;1
22321;;Saint-Nicolas-du-Plem;SAINT-NICOLAS-DU-PELEM;22480;22;COTES D'ARMOR;1
22322;;Saint-Pver;SAINT-PEVER;22720;22;COTES D'ARMOR;1
22323;;Saint-Ptan;SAINT-POTAN;22550;22;COTES D'ARMOR;1
22324;;Saint-Quay-Perros;SAINT-QUAY-PERROS;22700;22;COTES D'ARMOR;1
22325;;Saint-Quay-Portrieux;SAINT-QUAY-PORTRIEUX;22410;22;COTES D'ARMOR;1
22326;;Saint-Rieul;SAINT-RIEUL;22270;22;COTES D'ARMOR;1
22327;;Saint-Samson-sur-Rance;SAINT-SAMSON-SUR-RANCE;22100;22;COTES D'ARMOR;1
22328;;Saint-Servais;SAINT-SERVAIS;22160;22;COTES D'ARMOR;1
22330;;Saint-Thlo;SAINT-THELO;22460;22;COTES D'ARMOR;1
22331;;Sainte-Trphine;SAINTE-TREPHINE;22480;22;COTES D'ARMOR;1
22332;;Saint-Trimol;SAINT-TRIMOEL;22510;22;COTES D'ARMOR;1
22333;;Saint-Vran;SAINT-VRAN;22230;22;COTES D'ARMOR;1
22334;;Saint-Igeaux;SAINT-IGEAUX;22570;22;COTES D'ARMOR;1
22335;;Senven-Lhart;SENVEN-LEHART;22720;22;COTES D'ARMOR;1
22337;;Svignac;SEVIGNAC;22250;22;COTES D'ARMOR;1
22338;;Squiffiec;SQUIFFIEC;22200;22;COTES D'ARMOR;1
22339;;Taden;TADEN;22100;22;COTES D'ARMOR;1
22340;;Tonqudec;TONQUEDEC;22140;22;COTES D'ARMOR;1
22341;;Tramain;TRAMAIN;22640;22;COTES D'ARMOR;1
22342;;Trbdan;TREBEDAN;22980;22;COTES D'ARMOR;1
22343;;Trbeurden;TREBEURDEN;22560;22;COTES D'ARMOR;1
22344;;Trbrivan;TREBRIVAN;22340;22;COTES D'ARMOR;1
22345;;Trbry;TREBRY;22510;22;COTES D'ARMOR;1
22346;;Trdaniel;TREDANIEL;22510;22;COTES D'ARMOR;1
22347;;Trdarzec;TREDARZEC;22220;22;COTES D'ARMOR;1
22348;;Trdias;TREDIAS;22250;22;COTES D'ARMOR;1
22349;;Trdrez-Locqumeau;TREDREZ-LOCQUEMEAU;22300;22;COTES D'ARMOR;1
22350;;Trduder;TREDUDER;22310;22;COTES D'ARMOR;1
22351;;Treffrin;TREFFRIN;22340;22;COTES D'ARMOR;1
22352;;Trfumel;TREFUMEL;22630;22;COTES D'ARMOR;1
22353;;Trgastel;TREGASTEL;22730;22;COTES D'ARMOR;1
22354;;Trglamus;TREGLAMUS;22540;22;COTES D'ARMOR;1
22356;;Trgomeur;TREGOMEUR;22590;22;COTES D'ARMOR;1
22357;;Trgon;TREGON;22650;22;COTES D'ARMOR;1
22358;;Trgonneau;TREGONNEAU;22200;22;COTES D'ARMOR;1
22359;;Trgrom;TREGROM;22420;22;COTES D'ARMOR;1
22360;;Trgueux;TREGUEUX;22950;22;COTES D'ARMOR;1
22361;;Trguidel;TREGUIDEL;22290;22;COTES D'ARMOR;1
22362;;Trguier;TREGUIER;22220;22;COTES D'ARMOR;1
22363;;Trlvern;TRELEVERN;22660;22;COTES D'ARMOR;1
22364;;Trlivan;TRELIVAN;22100;22;COTES D'ARMOR;1
22365;;Trmargat;TREMARGAT;22110;22;COTES D'ARMOR;1
22366;;Trmel;TREMEL;22310;22;COTES D'ARMOR;1
22367;;Trmloir;TREMELOIR;22590;22;COTES D'ARMOR;1
22368;;Trmreuc;TREMEREUC;22490;22;COTES D'ARMOR;1
22369;;Trmeur;TREMEUR;22250;22;COTES D'ARMOR;1
22370;;Trmven;TREMEVEN;22290;22;COTES D'ARMOR;1
22371;;Trmorel;TREMOREL;22230;22;COTES D'ARMOR;1
22372;;Trmuson;TREMUSON;22440;22;COTES D'ARMOR;1
22373;;Trogan;TREOGAN;22340;22;COTES D'ARMOR;1
22375;;Tressignaux;TRESSIGNAUX;22290;22;COTES D'ARMOR;1
22376;;Trv;TREVE;22600;22;COTES D'ARMOR;1
22377;;Trveneuc;TREVENEUC;22410;22;COTES D'ARMOR;1
22378;;Trvrec;TREVEREC;22290;22;COTES D'ARMOR;1
22379;;Trvou-Trguignec;TREVOU-TREGUIGNEC;22660;22;COTES D'ARMOR;1
22380;;Trvron;TREVRON;22100;22;COTES D'ARMOR;1
22381;;Trzny;TREZENY;22450;22;COTES D'ARMOR;1
22383;;Trogury;TROGUERY;22450;22;COTES D'ARMOR;1
22384;;Uzel;UZEL;22460;22;COTES D'ARMOR;1
22385;La;Vicomt-sur-Rance;VICOMTE-SUR-RANCE;22690;22;COTES D'ARMOR;1
22386;Le;Vieux-Bourg;VIEUX-BOURG;22800;22;COTES D'ARMOR;1
22387;Le;Vieux-March;VIEUX-MARCHE;22420;22;COTES D'ARMOR;1
22388;;Vild-Guingalan;VILDE-GUINGALAN;22980;22;COTES D'ARMOR;1
22389;;Yffiniac;YFFINIAC;22120;22;COTES D'ARMOR;1
22390;;Yvias;YVIAS;22930;22;COTES D'ARMOR;1
22391;;Yvignac-la-Tour;YVIGNAC-LA-TOUR;22350;22;COTES D'ARMOR;1
23001;;Ahun;AHUN;23150;23;CREUSE;1
23002;;Ajain;AJAIN;23380;23;CREUSE;1
23003;;Alleyrat;ALLEYRAT;23200;23;CREUSE;1
23004;;Anzme;ANZEME;23000;23;CREUSE;1
23005;;Arfeuille-Chtain;ARFEUILLE-CHATAIN;23700;23;CREUSE;1
23006;;Arrnes;ARRENES;23210;23;CREUSE;1
23007;;Ars;ARS;23480;23;CREUSE;1
23008;;Aubusson;AUBUSSON;23200;23;CREUSE;1
23009;;Auge;AUGE;23170;23;CREUSE;1
23010;;Augres;AUGERES;23210;23;CREUSE;1
23011;;Aulon;AULON;23210;23;CREUSE;1
23012;;Auriat;AURIAT;23400;23;CREUSE;1
23013;;Auzances;AUZANCES;23700;23;CREUSE;1
23014;;Azat-Chtenet;AZAT-CHATENET;23210;23;CREUSE;1
23015;;Azerables;AZERABLES;23160;23;CREUSE;1
23016;;Banize;BANIZE;23120;23;CREUSE;1
23017;;Basville;BASVILLE;23260;23;CREUSE;1
23018;;Bazelat;BAZELAT;23160;23;CREUSE;1
23019;;Beissat;BEISSAT;23260;23;CREUSE;1
23020;;Bellegarde-en-Marche;BELLEGARDE-EN-MARCHE;23190;23;CREUSE;1
23021;;Bnvent-l'Abbaye;BENEVENT-L'ABBAYE;23210;23;CREUSE;1
23022;;Btte;BETETE;23270;23;CREUSE;1
23023;;Blaudeix;BLAUDEIX;23140;23;CREUSE;1
23024;;Blessac;BLESSAC;23200;23;CREUSE;1
23025;;Bonnat;BONNAT;23220;23;CREUSE;1
23026;;Bord-Saint-Georges;BORD-SAINT-GEORGES;23230;23;CREUSE;1
23027;;Bosmoreau-les-Mines;BOSMOREAU-LES-MINES;23400;23;CREUSE;1
23028;;Bosroger;BOSROGER;23200;23;CREUSE;1
23029;Le;Bourg-d'Hem;BOURG-D'HEM;23220;23;CREUSE;1
23030;;Bourganeuf;BOURGANEUF;23400;23;CREUSE;1
23031;;Boussac;BOUSSAC;23600;23;CREUSE;1
23032;;Boussac-Bourg;BOUSSAC-BOURG;23600;23;CREUSE;1
23033;La;Brionne;BRIONNE;23000;23;CREUSE;1
23034;;Brousse;BROUSSE;23700;23;CREUSE;1
23035;;Budelire;BUDELIERE;23170;23;CREUSE;1
23036;;Bussire-Dunoise;BUSSIERE-DUNOISE;23320;23;CREUSE;1
23037;;Bussire-Nouvelle;BUSSIERE-NOUVELLE;23700;23;CREUSE;1
23038;;Bussire-Saint-Georges;BUSSIERE-SAINT-GEORGES;23600;23;CREUSE;1
23039;La;Celle-Dunoise;CELLE-DUNOISE;23800;23;CREUSE;1
23040;La;Celle-sous-Gouzon;CELLE-SOUS-GOUZON;23230;23;CREUSE;1
23041;La;Cellette;CELLETTE;23350;23;CREUSE;1
23042;;Ceyroux;CEYROUX;23210;23;CREUSE;1
23043;;Chamberaud;CHAMBERAUD;23480;23;CREUSE;1
23044;;Chambon-Sainte-Croix;CHAMBON-SAINTE-CROIX;23220;23;CREUSE;1
23045;;Chambon-sur-Voueize;CHAMBON-SUR-VOUEIZE;23170;23;CREUSE;1
23046;;Chambonchard;CHAMBONCHARD;23110;23;CREUSE;1
23047;;Chamborand;CHAMBORAND;23240;23;CREUSE;1
23048;;Champagnat;CHAMPAGNAT;23190;23;CREUSE;1
23049;;Champsanglard;CHAMPSANGLARD;23220;23;CREUSE;1
23050;La;Chapelle-Baloue;CHAPELLE-BALOUE;23160;23;CREUSE;1
23051;La;Chapelle-Saint-Martial;CHAPELLE-SAINT-MARTIAL;23250;23;CREUSE;1
23052;La;Chapelle-Taillefert;CHAPELLE-TAILLEFERT;23000;23;CREUSE;1
23053;;Chard;CHARD;23700;23;CREUSE;1
23054;;Charron;CHARRON;23700;23;CREUSE;1
23055;;Chtelard;CHATELARD;23700;23;CREUSE;1
23056;;Chtelus-le-Marcheix;CHATELUS-LE-MARCHEIX;23430;23;CREUSE;1
23057;;Chtelus-Malvaleix;CHATELUS-MALVALEIX;23270;23;CREUSE;1
23058;Le;Chauchet;CHAUCHET;23130;23;CREUSE;1
23059;La;Chaussade;CHAUSSADE;23200;23;CREUSE;1
23060;;Chavanat;CHAVANAT;23250;23;CREUSE;1
23061;;Chnrailles;CHENERAILLES;23130;23;CREUSE;1
23062;;Chniers;CHENIERS;23220;23;CREUSE;1
23063;;Clairavaux;CLAIRAVAUX;23500;23;CREUSE;1
23064;;Clugnat;CLUGNAT;23270;23;CREUSE;1
23065;;Colondannes;COLONDANNES;23800;23;CREUSE;1
23066;Le;Compas;COMPAS;23700;23;CREUSE;1
23067;La;Courtine;COURTINE;23100;23;CREUSE;1
23068;;Cressat;CRESSAT;23140;23;CREUSE;1
23069;;Crocq;CROCQ;23260;23;CREUSE;1
23070;;Crozant;CROZANT;23160;23;CREUSE;1
23071;;Croze;CROZE;23500;23;CREUSE;1
23072;;Domeyrot;DOMEYROT;23140;23;CREUSE;1
23073;;Dontreix;DONTREIX;23700;23;CREUSE;1
23074;Le;Donzeil;DONZEIL;23480;23;CREUSE;1
23075;;Dun-le-Palestel;DUN-LE-PALESTEL;23800;23;CREUSE;1
23076;;vaux-les-Bains;EVAUX-LES-BAINS;23110;23;CREUSE;1
23077;;Faux-la-Montagne;FAUX-LA-MONTAGNE;23340;23;CREUSE;1
23078;;Faux-Mazuras;FAUX-MAZURAS;23400;23;CREUSE;1
23079;;Felletin;FELLETIN;23500;23;CREUSE;1
23080;;Fniers;FENIERS;23100;23;CREUSE;1
23081;;Flayat;FLAYAT;23260;23;CREUSE;1
23082;;Fleurat;FLEURAT;23320;23;CREUSE;1
23083;;Fontanires;FONTANIERES;23110;23;CREUSE;1
23084;La;Fort-du-Temple;FORET-DU-TEMPLE;23360;23;CREUSE;1
23086;;Fransches;FRANSECHES;23480;23;CREUSE;1
23087;;Fresselines;FRESSELINES;23450;23;CREUSE;1
23088;;Gartempe;GARTEMPE;23320;23;CREUSE;1
23089;;Genouillac;GENOUILLAC;23350;23;CREUSE;1
23090;;Gentioux-Pigerolles;GENTIOUX-PIGEROLLES;23340;23;CREUSE;1
23091;;Gioux;GIOUX;23500;23;CREUSE;1
23092;;Glnic;GLENIC;23380;23;CREUSE;1
23093;;Gouzon;GOUZON;23230;23;CREUSE;1
23095;Le;Grand-Bourg;GRAND-BOURG;23240;23;CREUSE;1
23096;;Guret;GUERET;23000;23;CREUSE;1
23097;;Issoudun-Ltrieix;ISSOUDUN-LETRIEIX;23130;23;CREUSE;1
23098;;Jalesches;JALESCHES;23270;23;CREUSE;1
23099;;Janaillat;JANAILLAT;23250;23;CREUSE;1
23100;;Jarnages;JARNAGES;23140;23;CREUSE;1
23101;;Jouillat;JOUILLAT;23220;23;CREUSE;1
23102;;Ladapeyre;LADAPEYRE;23270;23;CREUSE;1
23103;;Lafat;LAFAT;23800;23;CREUSE;1
23104;;Lavaufranche;LAVAUFRANCHE;23600;23;CREUSE;1
23105;;Lavaveix-les-Mines;LAVAVEIX-LES-MINES;23150;23;CREUSE;1
23106;;Lpaud;LEPAUD;23170;23;CREUSE;1
23107;;Lpinas;LEPINAS;23150;23;CREUSE;1
23108;;Leyrat;LEYRAT;23600;23;CREUSE;1
23109;;Linard;LINARD;23220;23;CREUSE;1
23110;;Lioux-les-Monges;LIOUX-LES-MONGES;23700;23;CREUSE;1
23111;;Lizires;LIZIERES;23240;23;CREUSE;1
23112;;Lourdoueix-Saint-Pierre;LOURDOUEIX-SAINT-PIERRE;23360;23;CREUSE;1
23113;;Lupersat;LUPERSAT;23190;23;CREUSE;1
23114;;Lussat;LUSSAT;23170;23;CREUSE;1
23115;;Magnat-l'trange;MAGNAT-L'ETRANGE;23260;23;CREUSE;1
23116;;Mainsat;MAINSAT;23700;23;CREUSE;1
23117;;Maison-Feyne;MAISON-FEYNE;23800;23;CREUSE;1
23118;;Maisonnisses;MAISONNISSES;23150;23;CREUSE;1
23119;;Malleret;MALLERET;23260;23;CREUSE;1
23120;;Malleret-Boussac;MALLERET-BOUSSAC;23600;23;CREUSE;1
23121;;Malval;MALVAL;23220;23;CREUSE;1
23122;;Mansat-la-Courrire;MANSAT-LA-COURRIERE;23400;23;CREUSE;1
23123;Les;Mars;MARS;23700;23;CREUSE;1
23124;;Marsac;MARSAC;23210;23;CREUSE;1
23125;Le;Mas-d'Artige;MAS-D'ARTIGE;23100;23;CREUSE;1
23126;;Masbaraud-Mrignat;MASBARAUD-MERIGNAT;23400;23;CREUSE;1
23127;;Mautes;MAUTES;23190;23;CREUSE;1
23128;;Mazeirat;MAZEIRAT;23150;23;CREUSE;1
23129;La;Mazire-aux-Bons-Hommes;MAZIERE-AUX-BONS-HOMMES;23260;23;CREUSE;1
23130;;Masnes;MEASNES;23360;23;CREUSE;1
23131;;Mrinchal;MERINCHAL;23420;23;CREUSE;1
23132;;Montaigut-le-Blanc;MONTAIGUT-LE-BLANC;23320;23;CREUSE;1
23133;;Montboucher;MONTBOUCHER;23400;23;CREUSE;1
23134;Le;Monteil-au-Vicomte;MONTEIL-AU-VICOMTE;23460;23;CREUSE;1
23136;;Mortroux;MORTROUX;23220;23;CREUSE;1
23137;;Mourioux-Vieilleville;MOURIOUX-VIEILLEVILLE;23210;23;CREUSE;1
23138;;Moutier-d'Ahun;MOUTIER-D'AHUN;23150;23;CREUSE;1
23139;;Moutier-Malcard;MOUTIER-MALCARD;23220;23;CREUSE;1
23140;;Moutier-Rozeille;MOUTIER-ROZEILLE;23200;23;CREUSE;1
23141;;Naillat;NAILLAT;23800;23;CREUSE;1
23142;;Noux;NEOUX;23200;23;CREUSE;1
23143;;Noth;NOTH;23300;23;CREUSE;1
23144;La;Nouaille;NOUAILLE;23500;23;CREUSE;1
23145;;Nouhant;NOUHANT;23170;23;CREUSE;1
23146;;Nouzerines;NOUZERINES;23600;23;CREUSE;1
23147;;Nouzerolles;NOUZEROLLES;23360;23;CREUSE;1
23148;;Nouziers;NOUZIERS;23350;23;CREUSE;1
23149;;Parsac;PARSAC;23140;23;CREUSE;1
23150;;Peyrabout;PEYRABOUT;23000;23;CREUSE;1
23151;;Peyrat-la-Nonire;PEYRAT-LA-NONIERE;23130;23;CREUSE;1
23152;;Pierrefitte;PIERREFITTE;23130;23;CREUSE;1
23154;;Pionnat;PIONNAT;23140;23;CREUSE;1
23155;;Pontarion;PONTARION;23250;23;CREUSE;1
23156;;Pontcharraud;PONTCHARRAUD;23260;23;CREUSE;1
23157;La;Pouge;POUGE;23250;23;CREUSE;1
23158;;Poussanges;POUSSANGES;23500;23;CREUSE;1
23159;;Puy-Malsignat;PUY-MALSIGNAT;23130;23;CREUSE;1
23160;;Reterre;RETERRE;23110;23;CREUSE;1
23161;;Rimondeix;RIMONDEIX;23140;23;CREUSE;1
23162;;Roches;ROCHES;23270;23;CREUSE;1
23164;;Rougnat;ROUGNAT;23700;23;CREUSE;1
23165;;Royre-de-Vassivire;ROYERE-DE-VASSIVIERE;23460;23;CREUSE;1
23166;;Sagnat;SAGNAT;23800;23;CREUSE;1
23167;;Sannat;SANNAT;23110;23;CREUSE;1
23168;;Sardent;SARDENT;23250;23;CREUSE;1
23169;La;Saunire;SAUNIERE;23000;23;CREUSE;1
23170;;Savennes;SAVENNES;23000;23;CREUSE;1
23171;;Sermur;SERMUR;23700;23;CREUSE;1
23172;La;Serre-Bussire-Vieille;SERRE-BUSSIERE-VIEILLE;23190;23;CREUSE;1
23173;;Soubrebost;SOUBREBOST;23250;23;CREUSE;1
23174;;Soumans;SOUMANS;23600;23;CREUSE;1
23175;;Sous-Parsat;SOUS-PARSAT;23150;23;CREUSE;1
23176;La;Souterraine;SOUTERRAINE;23300;23;CREUSE;1
23177;;Saint-Agnant-de-Versillat;SAINT-AGNANT-DE-VERSILLAT;23300;23;CREUSE;1
23178;;Saint-Agnant-prs-Crocq;SAINT-AGNANT-PRES-CROCQ;23260;23;CREUSE;1
23179;;Saint-Alpinien;SAINT-ALPINIEN;23200;23;CREUSE;1
23180;;Saint-Amand;SAINT-AMAND;23200;23;CREUSE;1
23181;;Saint-Amand-Jartoudeix;SAINT-AMAND-JARTOUDEIX;23400;23;CREUSE;1
23182;;Saint-Avit-de-Tardes;SAINT-AVIT-DE-TARDES;23200;23;CREUSE;1
23183;;Saint-Avit-le-Pauvre;SAINT-AVIT-LE-PAUVRE;23480;23;CREUSE;1
23184;;Saint-Bard;SAINT-BARD;23260;23;CREUSE;1
23185;;Saint-Chabrais;SAINT-CHABRAIS;23130;23;CREUSE;1
23186;;Saint-Christophe;SAINT-CHRISTOPHE;23000;23;CREUSE;1
23187;;Saint-Dizier-la-Tour;SAINT-DIZIER-LA-TOUR;23130;23;CREUSE;1
23188;;Saint-Dizier-les-Domaines;SAINT-DIZIER-LES-DOMAINES;23270;23;CREUSE;1
23189;;Saint-Dizier-Leyrenne;SAINT-DIZIER-LEYRENNE;23400;23;CREUSE;1
23190;;Saint-Domet;SAINT-DOMET;23190;23;CREUSE;1
23191;;Saint-loi;SAINT-ELOI;23000;23;CREUSE;1
23192;;Saint-tienne-de-Fursac;SAINT-ETIENNE-DE-FURSAC;23290;23;CREUSE;1
23193;;Sainte-Feyre;SAINTE-FEYRE;23000;23;CREUSE;1
23194;;Sainte-Feyre-la-Montagne;SAINTE-FEYRE-LA-MONTAGNE;23500;23;CREUSE;1
23195;;Saint-Fiel;SAINT-FIEL;23000;23;CREUSE;1
23196;;Saint-Frion;SAINT-FRION;23500;23;CREUSE;1
23197;;Saint-Georges-la-Pouge;SAINT-GEORGES-LA-POUGE;23250;23;CREUSE;1
23198;;Saint-Georges-Nigremont;SAINT-GEORGES-NIGREMONT;23500;23;CREUSE;1
23199;;Saint-Germain-Beaupr;SAINT-GERMAIN-BEAUPRE;23160;23;CREUSE;1
23200;;Saint-Goussaud;SAINT-GOUSSAUD;23430;23;CREUSE;1
23201;;Saint-Hilaire-la-Plaine;SAINT-HILAIRE-LA-PLAINE;23150;23;CREUSE;1
23202;;Saint-Hilaire-le-Chteau;SAINT-HILAIRE-LE-CHATEAU;23250;23;CREUSE;1
23203;;Saint-Julien-la-Gente;SAINT-JULIEN-LA-GENETE;23110;23;CREUSE;1
23204;;Saint-Julien-le-Chtel;SAINT-JULIEN-LE-CHATEL;23130;23;CREUSE;1
23205;;Saint-Junien-la-Bregre;SAINT-JUNIEN-LA-BREGERE;23400;23;CREUSE;1
23206;;Saint-Laurent;SAINT-LAURENT;23000;23;CREUSE;1
23207;;Saint-Lger-Bridereix;SAINT-LEGER-BRIDEREIX;23300;23;CREUSE;1
23208;;Saint-Lger-le-Gurtois;SAINT-LEGER-LE-GUERETOIS;23000;23;CREUSE;1
23209;;Saint-Loup;SAINT-LOUP;23130;23;CREUSE;1
23210;;Saint-Maixant;SAINT-MAIXANT;23200;23;CREUSE;1
23211;;Saint-Marc--Frongier;SAINT-MARC-A-FRONGIER;23200;23;CREUSE;1
23212;;Saint-Marc--Loubaud;SAINT-MARC-A-LOUBAUD;23460;23;CREUSE;1
23213;;Saint-Marien;SAINT-MARIEN;23600;23;CREUSE;1
23214;;Saint-Martial-le-Mont;SAINT-MARTIAL-LE-MONT;23150;23;CREUSE;1
23215;;Saint-Martial-le-Vieux;SAINT-MARTIAL-LE-VIEUX;23100;23;CREUSE;1
23216;;Saint-Martin-Chteau;SAINT-MARTIN-CHATEAU;23460;23;CREUSE;1
23217;;Saint-Martin-Sainte-Catherine;SAINT-MARTIN-SAINTE-CATHERINE;23430;23;CREUSE;1
23218;;Saint-Maurice-prs-Crocq;SAINT-MAURICE-PRES-CROCQ;23260;23;CREUSE;1
23219;;Saint-Maurice-la-Souterraine;SAINT-MAURICE-LA-SOUTERRAINE;23300;23;CREUSE;1
23220;;Saint-Mdard-la-Rochette;SAINT-MEDARD-LA-ROCHETTE;23200;23;CREUSE;1
23221;;Saint-Merd-la-Breuille;SAINT-MERD-LA-BREUILLE;23100;23;CREUSE;1
23222;;Saint-Michel-de-Veisse;SAINT-MICHEL-DE-VEISSE;23480;23;CREUSE;1
23223;;Saint-Moreil;SAINT-MOREIL;23400;23;CREUSE;1
23224;;Saint-Oradoux-de-Chirouze;SAINT-ORADOUX-DE-CHIROUZE;23100;23;CREUSE;1
23225;;Saint-Oradoux-prs-Crocq;SAINT-ORADOUX-PRES-CROCQ;23260;23;CREUSE;1
23226;;Saint-Pardoux-d'Arnet;SAINT-PARDOUX-D'ARNET;23260;23;CREUSE;1
23227;;Saint-Pardoux-Morterolles;SAINT-PARDOUX-MORTEROLLES;23400;23;CREUSE;1
23228;;Saint-Pardoux-le-Neuf;SAINT-PARDOUX-LE-NEUF;23200;23;CREUSE;1
23229;;Saint-Pardoux-les-Cards;SAINT-PARDOUX-LES-CARDS;23150;23;CREUSE;1
23230;;Saint-Pierre-Chrignat;SAINT-PIERRE-CHERIGNAT;23430;23;CREUSE;1
23231;;Saint-Pierre-de-Fursac;SAINT-PIERRE-DE-FURSAC;23290;23;CREUSE;1
23232;;Saint-Pierre-Bellevue;SAINT-PIERRE-BELLEVUE;23460;23;CREUSE;1
23233;;Saint-Pierre-le-Bost;SAINT-PIERRE-LE-BOST;23600;23;CREUSE;1
23234;;Saint-Priest;SAINT-PRIEST;23110;23;CREUSE;1
23235;;Saint-Priest-la-Feuille;SAINT-PRIEST-LA-FEUILLE;23300;23;CREUSE;1
23236;;Saint-Priest-la-Plaine;SAINT-PRIEST-LA-PLAINE;23240;23;CREUSE;1
23237;;Saint-Priest-Palus;SAINT-PRIEST-PALUS;23400;23;CREUSE;1
23238;;Saint-Quentin-la-Chabanne;SAINT-QUENTIN-LA-CHABANNE;23500;23;CREUSE;1
23239;;Saint-Sbastien;SAINT-SEBASTIEN;23160;23;CREUSE;1
23240;;Saint-Silvain-Bas-le-Roc;SAINT-SILVAIN-BAS-LE-ROC;23600;23;CREUSE;1
23241;;Saint-Silvain-Bellegarde;SAINT-SILVAIN-BELLEGARDE;23190;23;CREUSE;1
23242;;Saint-Silvain-Montaigut;SAINT-SILVAIN-MONTAIGUT;23320;23;CREUSE;1
23243;;Saint-Silvain-sous-Toulx;SAINT-SILVAIN-SOUS-TOULX;23140;23;CREUSE;1
23244;;Saint-Sulpice-le-Dunois;SAINT-SULPICE-LE-DUNOIS;23800;23;CREUSE;1
23245;;Saint-Sulpice-le-Gurtois;SAINT-SULPICE-LE-GUERETOIS;23000;23;CREUSE;1
23246;;Saint-Sulpice-les-Champs;SAINT-SULPICE-LES-CHAMPS;23480;23;CREUSE;1
23247;;Saint-Vaury;SAINT-VAURY;23320;23;CREUSE;1
23248;;Saint-Victor-en-Marche;SAINT-VICTOR-EN-MARCHE;23000;23;CREUSE;1
23249;;Saint-Yrieix-la-Montagne;SAINT-YRIEIX-LA-MONTAGNE;23460;23;CREUSE;1
23250;;Saint-Yrieix-les-Bois;SAINT-YRIEIX-LES-BOIS;23150;23;CREUSE;1
23251;;Tardes;TARDES;23170;23;CREUSE;1
23252;;Tercillat;TERCILLAT;23350;23;CREUSE;1
23253;;Thauron;THAURON;23250;23;CREUSE;1
23254;;Toulx-Sainte-Croix;TOULX-SAINTE-CROIX;23600;23;CREUSE;1
23255;;Trois-Fonds;TROIS-FONDS;23230;23;CREUSE;1
23257;;Vallire;VALLIERE;23120;23;CREUSE;1
23258;;Vareilles;VAREILLES;23300;23;CREUSE;1
23259;;Verneiges;VERNEIGES;23170;23;CREUSE;1
23260;;Vidaillat;VIDAILLAT;23250;23;CREUSE;1
23261;;Viersat;VIERSAT;23170;23;CREUSE;1
23262;;Vigeville;VIGEVILLE;23140;23;CREUSE;1
23263;;Villard;VILLARD;23800;23;CREUSE;1
23264;La;Villedieu;VILLEDIEU;23340;23;CREUSE;1
23265;La;Villeneuve;VILLENEUVE;23260;23;CREUSE;1
23266;La;Villetelle;VILLETELLE;23260;23;CREUSE;1
24001;;Abjat-sur-Bandiat;ABJAT-SUR-BANDIAT;24300;24;DORDOGNE;1
24002;;Agonac;AGONAC;24460;24;DORDOGNE;1
24004;;Ajat;AJAT;24210;24;DORDOGNE;1
24005;;Alles-sur-Dordogne;ALLES-SUR-DORDOGNE;24480;24;DORDOGNE;1
24006;;Allas-les-Mines;ALLAS-LES-MINES;24220;24;DORDOGNE;1
24007;;Allemans;ALLEMANS;24600;24;DORDOGNE;1
24008;;Angoisse;ANGOISSE;24270;24;DORDOGNE;1
24009;;Anlhiac;ANLHIAC;24160;24;DORDOGNE;1
24010;;Annesse-et-Beaulieu;ANNESSE-ET-BEAULIEU;24430;24;DORDOGNE;1
24011;;Antonne-et-Trigonant;ANTONNE-ET-TRIGONANT;24420;24;DORDOGNE;1
24012;;Archignac;ARCHIGNAC;24590;24;DORDOGNE;1
24013;;Atur;ATUR;24750;24;DORDOGNE;1
24014;;Aubas;AUBAS;24290;24;DORDOGNE;1
24015;;Audrix;AUDRIX;24260;24;DORDOGNE;1
24016;;Augignac;AUGIGNAC;24300;24;DORDOGNE;1
24018;;Auriac-du-Prigord;AURIAC-DU-PERIGORD;24290;24;DORDOGNE;1
24019;;Azerat;AZERAT;24210;24;DORDOGNE;1
24020;La;Bachellerie;BACHELLERIE;24210;24;DORDOGNE;1
24021;;Badefols-d'Ans;BADEFOLS-D'ANS;24390;24;DORDOGNE;1
24022;;Badefols-sur-Dordogne;BADEFOLS-SUR-DORDOGNE;24150;24;DORDOGNE;1
24023;;Baneuil;BANEUIL;24150;24;DORDOGNE;1
24024;;Bardou;BARDOU;24560;24;DORDOGNE;1
24025;;Bars;BARS;24210;24;DORDOGNE;1
24026;;Bassillac;BASSILLAC;24330;24;DORDOGNE;1
24027;;Bayac;BAYAC;24150;24;DORDOGNE;1
24028;;Beaumont-du-Prigord;BEAUMONT-DU-PERIGORD;24440;24;DORDOGNE;1
24029;;Beaupouyet;BEAUPOUYET;24400;24;DORDOGNE;1
24030;;Beauregard-de-Terrasson;BEAUREGARD-DE-TERRASSON;24120;24;DORDOGNE;1
24031;;Beauregard-et-Bassac;BEAUREGARD-ET-BASSAC;24140;24;DORDOGNE;1
24032;;Beauronne;BEAURONNE;24400;24;DORDOGNE;1
24033;;Beaussac;BEAUSSAC;24340;24;DORDOGNE;1
24034;;Beleymas;BELEYMAS;24140;24;DORDOGNE;1
24035;;Belvs;BELVES;24170;24;DORDOGNE;1
24036;;Berbiguires;BERBIGUIERES;24220;24;DORDOGNE;1
24037;;Bergerac;BERGERAC;24100;24;DORDOGNE;1
24038;;Bertric-Bure;BERTRIC-BUREE;24320;24;DORDOGNE;1
24039;;Besse;BESSE;24550;24;DORDOGNE;1
24040;;Beynac-et-Cazenac;BEYNAC-ET-CAZENAC;24220;24;DORDOGNE;1
24041;;Bzenac;BEZENAC;24220;24;DORDOGNE;1
24042;;Biras;BIRAS;24310;24;DORDOGNE;1
24043;;Biron;BIRON;24540;24;DORDOGNE;1
24044;;Blis-et-Born;BLIS-ET-BORN;24330;24;DORDOGNE;1
24045;;Boisse;BOISSE;24560;24;DORDOGNE;1
24046;;Boisseuilh;BOISSEUILH;24390;24;DORDOGNE;1
24047;La;Boissire-d'Ans;BOISSIERE-D'ANS;24640;24;DORDOGNE;1
24048;;Bonneville-et-Saint-Avit-de-Fumadires;BONNEVILLE-ET-SAINT-AVIT-DE-FUMADIERES;24230;24;DORDOGNE;1
24050;;Borrze;BORREZE;24590;24;DORDOGNE;1
24051;;Bosset;BOSSET;24130;24;DORDOGNE;1
24052;;Bouillac;BOUILLAC;24480;24;DORDOGNE;1
24053;;Boulazac;BOULAZAC;24750;24;DORDOGNE;1
24054;;Bouniagues;BOUNIAGUES;24560;24;DORDOGNE;1
24055;;Bourdeilles;BOURDEILLES;24310;24;DORDOGNE;1
24056;Le;Bourdeix;BOURDEIX;24300;24;DORDOGNE;1
24057;;Bourg-des-Maisons;BOURG-DES-MAISONS;24320;24;DORDOGNE;1
24058;;Bourg-du-Bost;BOURG-DU-BOST;24600;24;DORDOGNE;1
24059;;Bourgnac;BOURGNAC;24400;24;DORDOGNE;1
24060;;Bourniquel;BOURNIQUEL;24150;24;DORDOGNE;1
24061;;Bourrou;BOURROU;24110;24;DORDOGNE;1
24062;;Bouteilles-Saint-Sbastien;BOUTEILLES-SAINT-SEBASTIEN;24320;24;DORDOGNE;1
24063;;Bouzic;BOUZIC;24250;24;DORDOGNE;1
24064;;Brantme;BRANTOME;24310;24;DORDOGNE;1
24065;;Breuilh;BREUILH;24380;24;DORDOGNE;1
24066;;Brouchaud;BROUCHAUD;24210;24;DORDOGNE;1
24067;Le;Bugue;BUGUE;24260;24;DORDOGNE;1
24068;Le;Buisson-de-Cadouin;BUISSON-DE-CADOUIN;24480;24;DORDOGNE;1
24069;;Bussac;BUSSAC;24350;24;DORDOGNE;1
24070;;Busserolles;BUSSEROLLES;24360;24;DORDOGNE;1
24071;;Bussire-Badil;BUSSIERE-BADIL;24360;24;DORDOGNE;1
24073;;Cals;CALES;24150;24;DORDOGNE;1
24074;;Calviac-en-Prigord;CALVIAC-EN-PERIGORD;24370;24;DORDOGNE;1
24075;;Campagnac-ls-Quercy;CAMPAGNAC-LES-QUERCY;24550;24;DORDOGNE;1
24076;;Campagne;CAMPAGNE;24260;24;DORDOGNE;1
24077;;Campsegret;CAMPSEGRET;24140;24;DORDOGNE;1
24079;;Cantillac;CANTILLAC;24530;24;DORDOGNE;1
24080;;Capdrot;CAPDROT;24540;24;DORDOGNE;1
24081;;Carlux;CARLUX;24370;24;DORDOGNE;1
24082;;Carsac-Aillac;CARSAC-AILLAC;24200;24;DORDOGNE;1
24083;;Carsac-de-Gurson;CARSAC-DE-GURSON;24610;24;DORDOGNE;1
24084;;Carves;CARVES;24170;24;DORDOGNE;1
24085;La;Cassagne;CASSAGNE;24120;24;DORDOGNE;1
24086;;Castelnaud-la-Chapelle;CASTELNAUD-LA-CHAPELLE;24250;24;DORDOGNE;1
24087;;Castels;CASTELS;24220;24;DORDOGNE;1
24088;;Cause-de-Clrans;CAUSE-DE-CLERANS;24150;24;DORDOGNE;1
24089;;Cazouls;CAZOULES;24370;24;DORDOGNE;1
24090;;Celles;CELLES;24600;24;DORDOGNE;1
24091;;Cnac-et-Saint-Julien;CENAC-ET-SAINT-JULIEN;24250;24;DORDOGNE;1
24092;;Cendrieux;CENDRIEUX;24380;24;DORDOGNE;1
24093;;Cercles;CERCLES;24320;24;DORDOGNE;1
24094;;Chalagnac;CHALAGNAC;24380;24;DORDOGNE;1
24095;;Chalais;CHALAIS;24800;24;DORDOGNE;1
24096;;Champagnac-de-Belair;CHAMPAGNAC-DE-BELAIR;24530;24;DORDOGNE;1
24097;;Champagne-et-Fontaine;CHAMPAGNE-ET-FONTAINE;24320;24;DORDOGNE;1
24098;;Champcevinel;CHAMPCEVINEL;24750;24;DORDOGNE;1
24099;;Champeaux-et-la-Chapelle-Pommier;CHAMPEAUX-ET-LA-CHAPELLE-POMMIER;24340;24;DORDOGNE;1
24100;;Champniers-et-Reilhac;CHAMPNIERS-ET-REILHAC;24360;24;DORDOGNE;1
24101;;Champs-Romain;CHAMPS-ROMAIN;24470;24;DORDOGNE;1
24102;;Chancelade;CHANCELADE;24650;24;DORDOGNE;1
24103;Le;Change;CHANGE;24640;24;DORDOGNE;1
24104;;Chantrac;CHANTERAC;24190;24;DORDOGNE;1
24105;;Chapdeuil;CHAPDEUIL;24320;24;DORDOGNE;1
24106;La;Chapelle-Aubareil;CHAPELLE-AUBAREIL;24290;24;DORDOGNE;1
24107;La;Chapelle-Faucher;CHAPELLE-FAUCHER;24530;24;DORDOGNE;1
24108;La;Chapelle-Gonaguet;CHAPELLE-GONAGUET;24350;24;DORDOGNE;1
24109;La;Chapelle-Grsignac;CHAPELLE-GRESIGNAC;24320;24;DORDOGNE;1
24110;La;Chapelle-Montabourlet;CHAPELLE-MONTABOURLET;24320;24;DORDOGNE;1
24111;La;Chapelle-Montmoreau;CHAPELLE-MONTMOREAU;24300;24;DORDOGNE;1
24113;La;Chapelle-Saint-Jean;CHAPELLE-SAINT-JEAN;24390;24;DORDOGNE;1
24114;;Chassaignes;CHASSAIGNES;24600;24;DORDOGNE;1
24115;;Chteau-l'vque;CHATEAU-L'EVEQUE;24460;24;DORDOGNE;1
24116;;Chtres;CHATRES;24120;24;DORDOGNE;1
24117;;Chavagnac;CHAVAGNAC;24120;24;DORDOGNE;1
24118;;Chenaud;CHENAUD;24410;24;DORDOGNE;1
24119;;Cherval;CHERVAL;24320;24;DORDOGNE;1
24120;;Cherveix-Cubas;CHERVEIX-CUBAS;24390;24;DORDOGNE;1
24121;;Chourgnac;CHOURGNAC;24640;24;DORDOGNE;1
24122;;Cladech;CLADECH;24170;24;DORDOGNE;1
24123;;Clermont-de-Beauregard;CLERMONT-DE-BEAUREGARD;24140;24;DORDOGNE;1
24124;;Clermont-d'Excideuil;CLERMONT-D'EXCIDEUIL;24160;24;DORDOGNE;1
24126;;Colombier;COLOMBIER;24560;24;DORDOGNE;1
24127;;Coly;COLY;24120;24;DORDOGNE;1
24128;;Comberanche-et-peluche;COMBERANCHE-ET-EPELUCHE;24600;24;DORDOGNE;1
24129;;Condat-sur-Trincou;CONDAT-SUR-TRINCOU;24530;24;DORDOGNE;1
24130;;Condat-sur-Vzre;CONDAT-SUR-VEZERE;24570;24;DORDOGNE;1
24131;;Connezac;CONNEZAC;24300;24;DORDOGNE;1
24132;;Conne-de-Labarde;CONNE-DE-LABARDE;24560;24;DORDOGNE;1
24133;La;Coquille;COQUILLE;24450;24;DORDOGNE;1
24134;;Corgnac-sur-l'Isle;CORGNAC-SUR-L'ISLE;24800;24;DORDOGNE;1
24135;;Cornille;CORNILLE;24750;24;DORDOGNE;1
24136;;Coubjours;COUBJOURS;24390;24;DORDOGNE;1
24137;;Coulaures;COULAURES;24420;24;DORDOGNE;1
24138;;Coulounieix-Chamiers;COULOUNIEIX-CHAMIERS;24660;24;DORDOGNE;1
24139;;Coursac;COURSAC;24430;24;DORDOGNE;1
24140;;Cours-de-Pile;COURS-DE-PILE;24520;24;DORDOGNE;1
24141;;Coutures;COUTURES;24320;24;DORDOGNE;1
24142;;Coux-et-Bigaroque;COUX-ET-BIGAROQUE;24220;24;DORDOGNE;1
24143;;Couze-et-Saint-Front;COUZE-ET-SAINT-FRONT;24150;24;DORDOGNE;1
24144;;Creyssac;CREYSSAC;24350;24;DORDOGNE;1
24145;;Creysse;CREYSSE;24100;24;DORDOGNE;1
24146;;Creyssensac-et-Pissot;CREYSSENSAC-ET-PISSOT;24380;24;DORDOGNE;1
24147;;Cubjac;CUBJAC;24640;24;DORDOGNE;1
24148;;Cunges;CUNEGES;24240;24;DORDOGNE;1
24150;;Daglan;DAGLAN;24250;24;DORDOGNE;1
24151;;Doissat;DOISSAT;24170;24;DORDOGNE;1
24152;;Domme;DOMME;24250;24;DORDOGNE;1
24153;La;Dornac;DORNAC;24120;24;DORDOGNE;1
24154;;Douchapt;DOUCHAPT;24350;24;DORDOGNE;1
24155;;Douville;DOUVILLE;24140;24;DORDOGNE;1
24156;La;Douze;DOUZE;24330;24;DORDOGNE;1
24157;;Douzillac;DOUZILLAC;24190;24;DORDOGNE;1
24158;;Dussac;DUSSAC;24270;24;DORDOGNE;1
24159;;chourgnac;ECHOURGNAC;24410;24;DORDOGNE;1
24160;;glise-Neuve-de-Vergt;EGLISE-NEUVE-DE-VERGT;24380;24;DORDOGNE;1
24161;;glise-Neuve-d'Issac;EGLISE-NEUVE-D'ISSAC;24400;24;DORDOGNE;1
24162;;Escoire;ESCOIRE;24420;24;DORDOGNE;1
24163;;touars;ETOUARS;24360;24;DORDOGNE;1
24164;;Excideuil;EXCIDEUIL;24160;24;DORDOGNE;1
24165;;Eygurande-et-Gardedeuil;EYGURANDE-ET-GARDEDEUIL;24700;24;DORDOGNE;1
24166;;Eyliac;EYLIAC;24330;24;DORDOGNE;1
24167;;Eymet;EYMET;24500;24;DORDOGNE;1
24168;;Plaisance;PLAISANCE;24560;24;DORDOGNE;1
24170;;Eyvirat;EYVIRAT;24460;24;DORDOGNE;1
24171;;Eyzerac;EYZERAC;24800;24;DORDOGNE;1
24172;Les;Eyzies-de-Tayac-Sireuil;EYZIES-DE-TAYAC-SIREUIL;24620;24;DORDOGNE;1
24174;;Fanlac;FANLAC;24290;24;DORDOGNE;1
24175;Les;Farges;FARGES;24290;24;DORDOGNE;1
24176;;Faurilles;FAURILLES;24560;24;DORDOGNE;1
24177;;Faux;FAUX;24560;24;DORDOGNE;1
24178;;Festalemps;FESTALEMPS;24410;24;DORDOGNE;1
24179;La;Feuillade;FEUILLADE;19600;24;CORREZE;1
24180;;Firbeix;FIRBEIX;24450;24;DORDOGNE;1
24181;;Flaugeac;FLAUGEAC;24240;24;DORDOGNE;1
24182;Le;Fleix;FLEIX;24130;24;DORDOGNE;1
24183;;Fleurac;FLEURAC;24580;24;DORDOGNE;1
24184;;Florimont-Gaumier;FLORIMONT-GAUMIER;24250;24;DORDOGNE;1
24186;;Fonroque;FONROQUE;24500;24;DORDOGNE;1
24188;;Fossemagne;FOSSEMAGNE;24210;24;DORDOGNE;1
24189;;Fougueyrolles;FOUGUEYROLLES;33220;24;GIRONDE;1
24190;;Fouleix;FOULEIX;24380;24;DORDOGNE;1
24191;;Fraisse;FRAISSE;24130;24;DORDOGNE;1
24192;;Gabillou;GABILLOU;24210;24;DORDOGNE;1
24193;;Gageac-et-Rouillac;GAGEAC-ET-ROUILLAC;24240;24;DORDOGNE;1
24194;;Gardonne;GARDONNE;24680;24;DORDOGNE;1
24195;;Gaugeac;GAUGEAC;24540;24;DORDOGNE;1
24196;;Gnis;GENIS;24160;24;DORDOGNE;1
24197;;Ginestet;GINESTET;24130;24;DORDOGNE;1
24198;La;Gonterie-Boulouneix;GONTERIE-BOULOUNEIX;24310;24;DORDOGNE;1
24199;;Gout-Rossignol;GOUT-ROSSIGNOL;24320;24;DORDOGNE;1
24200;;Grand-Brassac;GRAND-BRASSAC;24350;24;DORDOGNE;1
24202;;Granges-d'Ans;GRANGES-D'ANS;24390;24;DORDOGNE;1
24203;Les;Graulges;GRAULGES;24340;24;DORDOGNE;1
24204;;Grzes;GREZES;24120;24;DORDOGNE;1
24205;;Grignols;GRIGNOLS;24110;24;DORDOGNE;1
24206;;Grives;GRIVES;24170;24;DORDOGNE;1
24207;;Groljac;GROLEJAC;24250;24;DORDOGNE;1
24208;;Grun-Bordas;GRUN-BORDAS;24380;24;DORDOGNE;1
24209;;Hautefaye;HAUTEFAYE;24300;24;DORDOGNE;1
24210;;Hautefort;HAUTEFORT;24390;24;DORDOGNE;1
24211;;Issac;ISSAC;24400;24;DORDOGNE;1
24212;;Issigeac;ISSIGEAC;24560;24;DORDOGNE;1
24213;;Jaure;JAURE;24140;24;DORDOGNE;1
24214;;Javerlhac-et-la-Chapelle-Saint-Robert;JAVERLHAC-ET-LA-CHAPELLE-SAINT-ROBERT;24300;24;DORDOGNE;1
24215;;Jayac;JAYAC;24590;24;DORDOGNE;1
24216;La;Jemaye;JEMAYE;24410;24;DORDOGNE;1
24217;;Journiac;JOURNIAC;24260;24;DORDOGNE;1
24218;;Jumilhac-le-Grand;JUMILHAC-LE-GRAND;24630;24;DORDOGNE;1
24219;;Labouquerie;LABOUQUERIE;24440;24;DORDOGNE;1
24220;;Lacropte;LACROPTE;24380;24;DORDOGNE;1
24221;;Rudeau-Ladosse;RUDEAU-LADOSSE;24340;24;DORDOGNE;1
24222;La;Force;FORCE;24130;24;DORDOGNE;1
24223;;Lalinde;LALINDE;24150;24;DORDOGNE;1
24224;;Lamonzie-Montastruc;LAMONZIE-MONTASTRUC;24520;24;DORDOGNE;1
24225;;Lamonzie-Saint-Martin;LAMONZIE-SAINT-MARTIN;24680;24;DORDOGNE;1
24226;;Lamothe-Montravel;LAMOTHE-MONTRAVEL;24230;24;DORDOGNE;1
24227;;Lanouaille;LANOUAILLE;24270;24;DORDOGNE;1
24228;;Lanquais;LANQUAIS;24150;24;DORDOGNE;1
24229;Le;Lardin-Saint-Lazare;LARDIN-SAINT-LAZARE;24570;24;DORDOGNE;1
24230;;Larzac;LARZAC;24170;24;DORDOGNE;1
24231;;Lavalade;LAVALADE;24540;24;DORDOGNE;1
24232;;Lavaur;LAVAUR;24550;24;DORDOGNE;1
24233;;Laveyssire;LAVEYSSIERE;24130;24;DORDOGNE;1
24234;Les;Lches;LECHES;24400;24;DORDOGNE;1
24235;;Lguillac-de-Cercles;LEGUILLAC-DE-CERCLES;24340;24;DORDOGNE;1
24236;;Lguillac-de-l'Auche;LEGUILLAC-DE-L'AUCHE;24110;24;DORDOGNE;1
24237;;Lembras;LEMBRAS;24100;24;DORDOGNE;1
24238;;Lempzours;LEMPZOURS;24800;24;DORDOGNE;1
24239;;Ligueux;LIGUEUX;24460;24;DORDOGNE;1
24240;;Limeuil;LIMEUIL;24510;24;DORDOGNE;1
24241;;Limeyrat;LIMEYRAT;24210;24;DORDOGNE;1
24242;;Liorac-sur-Louyre;LIORAC-SUR-LOUYRE;24520;24;DORDOGNE;1
24243;;Lisle;LISLE;24350;24;DORDOGNE;1
24244;;Lolme;LOLME;24540;24;DORDOGNE;1
24245;;Loubejac;LOUBEJAC;24550;24;DORDOGNE;1
24246;;Lunas;LUNAS;24130;24;DORDOGNE;1
24247;;Lusignac;LUSIGNAC;24320;24;DORDOGNE;1
24248;;Lussas-et-Nontronneau;LUSSAS-ET-NONTRONNEAU;24300;24;DORDOGNE;1
24249;;Manaurie;MANAURIE;24620;24;DORDOGNE;1
24251;;Manzac-sur-Vern;MANZAC-SUR-VERN;24110;24;DORDOGNE;1
24252;;Marcillac-Saint-Quentin;MARCILLAC-SAINT-QUENTIN;24200;24;DORDOGNE;1
24253;;Mareuil;MAREUIL;24340;24;DORDOGNE;1
24254;;Marnac;MARNAC;24220;24;DORDOGNE;1
24255;;Marquay;MARQUAY;24620;24;DORDOGNE;1
24256;;Marsac-sur-l'Isle;MARSAC-SUR-L'ISLE;24430;24;DORDOGNE;1
24257;;Marsals;MARSALES;24540;24;DORDOGNE;1
24258;;Marsaneix;MARSANEIX;24750;24;DORDOGNE;1
24259;;Maurens;MAURENS;24140;24;DORDOGNE;1
24260;;Mauzac-et-Grand-Castang;MAUZAC-ET-GRAND-CASTANG;24150;24;DORDOGNE;1
24261;;Mauzens-et-Miremont;MAUZENS-ET-MIREMONT;24260;24;DORDOGNE;1
24262;;Mayac;MAYAC;24420;24;DORDOGNE;1
24263;;Mazeyrolles;MAZEYROLLES;24550;24;DORDOGNE;1
24264;;Mnesplet;MENESPLET;24700;24;DORDOGNE;1
24266;;Mensignac;MENSIGNAC;24350;24;DORDOGNE;1
24267;;Mescoules;MESCOULES;24240;24;DORDOGNE;1
24268;;Meyrals;MEYRALS;24220;24;DORDOGNE;1
24269;;Mialet;MIALET;24450;24;DORDOGNE;1
24270;;Milhac-d'Auberoche;MILHAC-D'AUBEROCHE;24330;24;DORDOGNE;1
24271;;Milhac-de-Nontron;MILHAC-DE-NONTRON;24470;24;DORDOGNE;1
24272;;Minzac;MINZAC;24610;24;DORDOGNE;1
24273;;Molires;MOLIERES;24480;24;DORDOGNE;1
24274;;Monbazillac;MONBAZILLAC;24240;24;DORDOGNE;1
24276;;Monestier;MONESTIER;24240;24;DORDOGNE;1
24277;;Monfaucon;MONFAUCON;24130;24;DORDOGNE;1
24278;;Monmadals;MONMADALES;24560;24;DORDOGNE;1
24279;;Monmarvs;MONMARVES;24560;24;DORDOGNE;1
24280;;Monpazier;MONPAZIER;24540;24;DORDOGNE;1
24281;;Monsac;MONSAC;24440;24;DORDOGNE;1
24282;;Monsaguel;MONSAGUEL;24560;24;DORDOGNE;1
24283;;Monsec;MONSEC;24340;24;DORDOGNE;1
24284;;Montagnac-d'Auberoche;MONTAGNAC-D'AUBEROCHE;24210;24;DORDOGNE;1
24285;;Montagnac-la-Crempse;MONTAGNAC-LA-CREMPSE;24140;24;DORDOGNE;1
24286;;Montagrier;MONTAGRIER;24350;24;DORDOGNE;1
24287;;Montaut;MONTAUT;24560;24;DORDOGNE;1
24288;;Montazeau;MONTAZEAU;24230;24;DORDOGNE;1
24289;;Montcaret;MONTCARET;24230;24;DORDOGNE;1
24290;;Montferrand-du-Prigord;MONTFERRAND-DU-PERIGORD;24440;24;DORDOGNE;1
24291;;Montignac;MONTIGNAC;24290;24;DORDOGNE;1
24292;;Montpeyroux;MONTPEYROUX;24610;24;DORDOGNE;1
24293;;Monplaisant;MONPLAISANT;24170;24;DORDOGNE;1
24294;;Montpon-Mnestrol;MONTPON-MENESTEROL;24700;24;DORDOGNE;1
24295;;Montrem;MONTREM;24110;24;DORDOGNE;1
24296;;Mouleydier;MOULEYDIER;24520;24;DORDOGNE;1
24297;;Moulin-Neuf;MOULIN-NEUF;24700;24;DORDOGNE;1
24298;;Mouzens;MOUZENS;24220;24;DORDOGNE;1
24299;;Mussidan;MUSSIDAN;24400;24;DORDOGNE;1
24300;;Nabirat;NABIRAT;24250;24;DORDOGNE;1
24301;;Nadaillac;NADAILLAC;24590;24;DORDOGNE;1
24302;;Nailhac;NAILHAC;24390;24;DORDOGNE;1
24303;;Nanteuil-Auriac-de-Bourzac;NANTEUIL-AURIAC-DE-BOURZAC;24320;24;DORDOGNE;1
24304;;Nantheuil;NANTHEUIL;24800;24;DORDOGNE;1
24305;;Nanthiat;NANTHIAT;24800;24;DORDOGNE;1
24306;;Nastringues;NASTRINGUES;24230;24;DORDOGNE;1
24307;;Naussannes;NAUSSANNES;24440;24;DORDOGNE;1
24308;;Ngrondes;NEGRONDES;24460;24;DORDOGNE;1
24309;;Neuvic;NEUVIC;24190;24;DORDOGNE;1
24310;;Nojals-et-Clotte;NOJALS-ET-CLOTTE;24440;24;DORDOGNE;1
24311;;Nontron;NONTRON;24300;24;DORDOGNE;1
24312;;Notre-Dame-de-Sanilhac;NOTRE-DAME-DE-SANILHAC;24660;24;DORDOGNE;1
24313;;Orliac;ORLIAC;24170;24;DORDOGNE;1
24314;;Orliaguet;ORLIAGUET;24370;24;DORDOGNE;1
24316;;Parcoul;PARCOUL;24410;24;DORDOGNE;1
24317;;Paulin;PAULIN;24590;24;DORDOGNE;1
24318;;Paunat;PAUNAT;24510;24;DORDOGNE;1
24319;;Paussac-et-Saint-Vivien;PAUSSAC-ET-SAINT-VIVIEN;24310;24;DORDOGNE;1
24320;;Payzac;PAYZAC;24270;24;DORDOGNE;1
24321;;Pazayac;PAZAYAC;24120;24;DORDOGNE;1
24322;;Prigueux;PERIGUEUX;24000;24;DORDOGNE;1
24323;;Petit-Bersac;PETIT-BERSAC;24600;24;DORDOGNE;1
24324;;Peyrignac;PEYRIGNAC;24210;24;DORDOGNE;1
24325;;Peyrillac-et-Millac;PEYRILLAC-ET-MILLAC;24370;24;DORDOGNE;1
24326;;Peyzac-le-Moustier;PEYZAC-LE-MOUSTIER;24620;24;DORDOGNE;1
24327;;Pezuls;PEZULS;24510;24;DORDOGNE;1
24328;;Pigut-Pluviers;PIEGUT-PLUVIERS;24360;24;DORDOGNE;1
24329;Le;Pizou;PIZOU;24700;24;DORDOGNE;1
24330;;Plazac;PLAZAC;24580;24;DORDOGNE;1
24331;;Pomport;POMPORT;24240;24;DORDOGNE;1
24333;;Ponteyraud;PONTEYRAUD;24410;24;DORDOGNE;1
24334;;Pontours;PONTOURS;24150;24;DORDOGNE;1
24335;;Port-Sainte-Foy-et-Ponchapt;PORT-SAINTE-FOY-ET-PONCHAPT;33220;24;GIRONDE;1
24336;;Prats-de-Carlux;PRATS-DE-CARLUX;24370;24;DORDOGNE;1
24337;;Prats-du-Prigord;PRATS-DU-PERIGORD;24550;24;DORDOGNE;1
24338;;Pressignac-Vicq;PRESSIGNAC-VICQ;24150;24;DORDOGNE;1
24339;;Preyssac-d'Excideuil;PREYSSAC-D'EXCIDEUIL;24160;24;DORDOGNE;1
24340;;Prigonrieux;PRIGONRIEUX;24130;24;DORDOGNE;1
24341;;Proissans;PROISSANS;24200;24;DORDOGNE;1
24343;;Puymangou;PUYMANGOU;24410;24;DORDOGNE;1
24344;;Puyrenier;PUYRENIER;24340;24;DORDOGNE;1
24345;;Queyssac;QUEYSSAC;24140;24;DORDOGNE;1
24346;;Quinsac;QUINSAC;24530;24;DORDOGNE;1
24347;;Rampieux;RAMPIEUX;24440;24;DORDOGNE;1
24348;;Razac-d'Eymet;RAZAC-D'EYMET;24500;24;DORDOGNE;1
24349;;Razac-de-Saussignac;RAZAC-DE-SAUSSIGNAC;24240;24;DORDOGNE;1
24350;;Razac-sur-l'Isle;RAZAC-SUR-L'ISLE;24430;24;DORDOGNE;1
24351;;Ribagnac;RIBAGNAC;24240;24;DORDOGNE;1
24352;;Ribrac;RIBERAC;24600;24;DORDOGNE;1
24353;La;Rochebeaucourt-et-Argentine;ROCHEBEAUCOURT-ET-ARGENTINE;24340;24;DORDOGNE;1
24354;La;Roche-Chalais;ROCHE-CHALAIS;24490;24;DORDOGNE;1
24355;La;Roque-Gageac;ROQUE-GAGEAC;24250;24;DORDOGNE;1
24356;;Rouffignac-Saint-Cernin-de-Reilhac;ROUFFIGNAC-SAINT-CERNIN-DE-REILHAC;24580;24;DORDOGNE;1
24357;;Rouffignac-de-Sigouls;ROUFFIGNAC-DE-SIGOULES;24240;24;DORDOGNE;1
24359;;Sadillac;SADILLAC;24500;24;DORDOGNE;1
24360;;Sagelat;SAGELAT;24170;24;DORDOGNE;1
24361;;Saint-Agne;SAINT-AGNE;24520;24;DORDOGNE;1
24362;;Sainte-Alvre;SAINTE-ALVERE;24510;24;DORDOGNE;1
24363;;Saint-Amand-de-Belvs;SAINT-AMAND-DE-BELVES;24170;24;DORDOGNE;1
24364;;Saint-Amand-de-Coly;SAINT-AMAND-DE-COLY;24290;24;DORDOGNE;1
24365;;Saint-Amand-de-Vergt;SAINT-AMAND-DE-VERGT;24380;24;DORDOGNE;1
24366;;Saint-Andr-d'Allas;SAINT-ANDRE-D'ALLAS;24200;24;DORDOGNE;1
24367;;Saint-Andr-de-Double;SAINT-ANDRE-DE-DOUBLE;24190;24;DORDOGNE;1
24368;;Saint-Antoine-Cumond;SAINT-ANTOINE-CUMOND;24410;24;DORDOGNE;1
24369;;Saint-Antoine-d'Auberoche;SAINT-ANTOINE-D'AUBEROCHE;24330;24;DORDOGNE;1
24370;;Saint-Antoine-de-Breuilh;SAINT-ANTOINE-DE-BREUILH;24230;24;DORDOGNE;1
24371;;Saint-Aquilin;SAINT-AQUILIN;24110;24;DORDOGNE;1
24372;;Saint-Astier;SAINT-ASTIER;24110;24;DORDOGNE;1
24373;;Saint-Aubin-de-Cadelech;SAINT-AUBIN-DE-CADELECH;24500;24;DORDOGNE;1
24374;;Saint-Aubin-de-Lanquais;SAINT-AUBIN-DE-LANQUAIS;24560;24;DORDOGNE;1
24375;;Saint-Aubin-de-Nabirat;SAINT-AUBIN-DE-NABIRAT;24250;24;DORDOGNE;1
24376;;Saint-Aulaye;SAINT-AULAYE;24410;24;DORDOGNE;1
24377;;Saint-Avit-de-Vialard;SAINT-AVIT-DE-VIALARD;24260;24;DORDOGNE;1
24378;;Saint-Avit-Rivire;SAINT-AVIT-RIVIERE;24540;24;DORDOGNE;1
24379;;Saint-Avit-Snieur;SAINT-AVIT-SENIEUR;24440;24;DORDOGNE;1
24380;;Saint-Barthlemy-de-Bellegarde;SAINT-BARTHELEMY-DE-BELLEGARDE;24700;24;DORDOGNE;1
24381;;Saint-Barthlemy-de-Bussire;SAINT-BARTHELEMY-DE-BUSSIERE;24360;24;DORDOGNE;1
24382;;Saint-Capraise-de-Lalinde;SAINT-CAPRAISE-DE-LALINDE;24150;24;DORDOGNE;1
24383;;Saint-Capraise-d'Eymet;SAINT-CAPRAISE-D'EYMET;24500;24;DORDOGNE;1
24384;;Saint-Cassien;SAINT-CASSIEN;24540;24;DORDOGNE;1
24385;;Saint-Cernin-de-Labarde;SAINT-CERNIN-DE-LABARDE;24560;24;DORDOGNE;1
24386;;Saint-Cernin-de-l'Herm;SAINT-CERNIN-DE-L'HERM;24550;24;DORDOGNE;1
24388;;Saint-Chamassy;SAINT-CHAMASSY;24260;24;DORDOGNE;1
24389;;Saint-Cirq;SAINT-CIRQ;24260;24;DORDOGNE;1
24390;;Saint-Crpin-d'Auberoche;SAINT-CREPIN-D'AUBEROCHE;24330;24;DORDOGNE;1
24391;;Saint-Crpin-de-Richemont;SAINT-CREPIN-DE-RICHEMONT;24310;24;DORDOGNE;1
24392;;Saint-Crpin-et-Carlucet;SAINT-CREPIN-ET-CARLUCET;24590;24;DORDOGNE;1
24393;;Sainte-Croix;SAINTE-CROIX;24440;24;DORDOGNE;1
24394;;Sainte-Croix-de-Mareuil;SAINTE-CROIX-DE-MAREUIL;24340;24;DORDOGNE;1
24395;;Saint-Cybranet;SAINT-CYBRANET;24250;24;DORDOGNE;1
24396;;Saint-Cyprien;SAINT-CYPRIEN;24220;24;DORDOGNE;1
24397;;Saint-Cyr-les-Champagnes;SAINT-CYR-LES-CHAMPAGNES;24270;24;DORDOGNE;1
24398;;Saint-Estphe;SAINT-ESTEPHE;24360;24;DORDOGNE;1
24399;;Saint-tienne-de-Puycorbier;SAINT-ETIENNE-DE-PUYCORBIER;24400;24;DORDOGNE;1
24401;;Sainte-Eulalie-d'Ans;SAINTE-EULALIE-D'ANS;24640;24;DORDOGNE;1
24402;;Sainte-Eulalie-d'Eymet;SAINTE-EULALIE-D'EYMET;24500;24;DORDOGNE;1
24403;;Saint-Flix-de-Bourdeilles;SAINT-FELIX-DE-BOURDEILLES;24340;24;DORDOGNE;1
24404;;Saint-Flix-de-Reillac-et-Mortemart;SAINT-FELIX-DE-REILLAC-ET-MORTEMART;24260;24;DORDOGNE;1
24405;;Saint-Flix-de-Villadeix;SAINT-FELIX-DE-VILLADEIX;24510;24;DORDOGNE;1
24406;;Sainte-Foy-de-Belvs;SAINTE-FOY-DE-BELVES;24170;24;DORDOGNE;1
24407;;Sainte-Foy-de-Longas;SAINTE-FOY-DE-LONGAS;24510;24;DORDOGNE;1
24408;;Saint-Front-d'Alemps;SAINT-FRONT-D'ALEMPS;24460;24;DORDOGNE;1
24409;;Saint-Front-de-Pradoux;SAINT-FRONT-DE-PRADOUX;24400;24;DORDOGNE;1
24410;;Saint-Front-la-Rivire;SAINT-FRONT-LA-RIVIERE;24300;24;DORDOGNE;1
24411;;Saint-Front-sur-Nizonne;SAINT-FRONT-SUR-NIZONNE;24300;24;DORDOGNE;1
24412;;Saint-Genis;SAINT-GENIES;24590;24;DORDOGNE;1
24413;;Saint-Georges-Blancaneix;SAINT-GEORGES-BLANCANEIX;24130;24;DORDOGNE;1
24414;;Saint-Georges-de-Montclard;SAINT-GEORGES-DE-MONTCLARD;24140;24;DORDOGNE;1
24415;;Saint-Graud-de-Corps;SAINT-GERAUD-DE-CORPS;24700;24;DORDOGNE;1
24416;;Saint-Germain-de-Belvs;SAINT-GERMAIN-DE-BELVES;24170;24;DORDOGNE;1
24417;;Saint-Germain-des-Prs;SAINT-GERMAIN-DES-PRES;24160;24;DORDOGNE;1
24418;;Saint-Germain-du-Salembre;SAINT-GERMAIN-DU-SALEMBRE;24190;24;DORDOGNE;1
24419;;Saint-Germain-et-Mons;SAINT-GERMAIN-ET-MONS;24520;24;DORDOGNE;1
24420;;Saint-Gry;SAINT-GERY;24400;24;DORDOGNE;1
24421;;Saint-Geyrac;SAINT-GEYRAC;24330;24;DORDOGNE;1
24422;;Saint-Hilaire-d'Estissac;SAINT-HILAIRE-D'ESTISSAC;24140;24;DORDOGNE;1
24423;;Sainte-Innocence;SAINTE-INNOCENCE;24500;24;DORDOGNE;1
24424;;Saint-Jean-d'Ataux;SAINT-JEAN-D'ATAUX;24190;24;DORDOGNE;1
24425;;Saint-Jean-de-Cle;SAINT-JEAN-DE-COLE;24800;24;DORDOGNE;1
24426;;Saint-Jean-d'Estissac;SAINT-JEAN-D'ESTISSAC;24140;24;DORDOGNE;1
24427;;Saint-Jean-d'Eyraud;SAINT-JEAN-D'EYRAUD;24140;24;DORDOGNE;1
24428;;Saint-Jory-de-Chalais;SAINT-JORY-DE-CHALAIS;24800;24;DORDOGNE;1
24429;;Saint-Jory-las-Bloux;SAINT-JORY-LAS-BLOUX;24160;24;DORDOGNE;1
24430;;Saint-Julien-de-Bourdeilles;SAINT-JULIEN-DE-BOURDEILLES;24310;24;DORDOGNE;1
24431;;Saint-Julien-de-Crempse;SAINT-JULIEN-DE-CREMPSE;24140;24;DORDOGNE;1
24432;;Saint-Julien-de-Lampon;SAINT-JULIEN-DE-LAMPON;24370;24;DORDOGNE;1
24433;;Saint-Julien-d'Eymet;SAINT-JULIEN-D'EYMET;24500;24;DORDOGNE;1
24434;;Saint-Just;SAINT-JUST;24320;24;DORDOGNE;1
24435;;Saint-Laurent-des-Btons;SAINT-LAURENT-DES-BATONS;24510;24;DORDOGNE;1
24436;;Saint-Laurent-des-Hommes;SAINT-LAURENT-DES-HOMMES;24400;24;DORDOGNE;1
24437;;Saint-Laurent-des-Vignes;SAINT-LAURENT-DES-VIGNES;24100;24;DORDOGNE;1
24438;;Saint-Laurent-la-Valle;SAINT-LAURENT-LA-VALLEE;24170;24;DORDOGNE;1
24439;;Saint-Laurent-sur-Manoire;SAINT-LAURENT-SUR-MANOIRE;24330;24;DORDOGNE;1
24441;;Saint-Lon-d'Issigeac;SAINT-LEON-D'ISSIGEAC;24560;24;DORDOGNE;1
24442;;Saint-Lon-sur-l'Isle;SAINT-LEON-SUR-L'ISLE;24110;24;DORDOGNE;1
24443;;Saint-Lon-sur-Vzre;SAINT-LEON-SUR-VEZERE;24290;24;DORDOGNE;1
24444;;Saint-Louis-en-l'Isle;SAINT-LOUIS-EN-L'ISLE;24400;24;DORDOGNE;1
24445;;Saint-Marcel-du-Prigord;SAINT-MARCEL-DU-PERIGORD;24510;24;DORDOGNE;1
24446;;Saint-Marcory;SAINT-MARCORY;24540;24;DORDOGNE;1
24447;;Sainte-Marie-de-Chignac;SAINTE-MARIE-DE-CHIGNAC;24330;24;DORDOGNE;1
24448;;Saint-Martial-d'Albarde;SAINT-MARTIAL-D'ALBAREDE;24160;24;DORDOGNE;1
24449;;Saint-Martial-d'Artenset;SAINT-MARTIAL-D'ARTENSET;24700;24;DORDOGNE;1
24450;;Saint-Martial-de-Nabirat;SAINT-MARTIAL-DE-NABIRAT;24250;24;DORDOGNE;1
24451;;Saint-Martial-de-Valette;SAINT-MARTIAL-DE-VALETTE;24300;24;DORDOGNE;1
24452;;Saint-Martial-Viveyrol;SAINT-MARTIAL-VIVEYROL;24320;24;DORDOGNE;1
24453;;Saint-Martin-de-Fressengeas;SAINT-MARTIN-DE-FRESSENGEAS;24800;24;DORDOGNE;1
24454;;Saint-Martin-de-Gurson;SAINT-MARTIN-DE-GURSON;24610;24;DORDOGNE;1
24455;;Saint-Martin-de-Ribrac;SAINT-MARTIN-DE-RIBERAC;24600;24;DORDOGNE;1
24456;;Saint-Martin-des-Combes;SAINT-MARTIN-DES-COMBES;24140;24;DORDOGNE;1
24457;;Saint-Martin-l'Astier;SAINT-MARTIN-L'ASTIER;24400;24;DORDOGNE;1
24458;;Saint-Martin-le-Pin;SAINT-MARTIN-LE-PIN;24300;24;DORDOGNE;1
24459;;Saint-Maime-de-Preyrol;SAINT-MAIME-DE-PEREYROL;24380;24;DORDOGNE;1
24460;;Saint-Mard-de-Drne;SAINT-MEARD-DE-DRONE;24600;24;DORDOGNE;1
24461;;Saint-Mard-de-Guron;SAINT-MEARD-DE-GURCON;24610;24;DORDOGNE;1
24462;;Saint-Mdard-de-Mussidan;SAINT-MEDARD-DE-MUSSIDAN;24400;24;DORDOGNE;1
24463;;Saint-Mdard-d'Excideuil;SAINT-MEDARD-D'EXCIDEUIL;24160;24;DORDOGNE;1
24464;;Saint-Mesmin;SAINT-MESMIN;24270;24;DORDOGNE;1
24465;;Saint-Michel-de-Double;SAINT-MICHEL-DE-DOUBLE;24400;24;DORDOGNE;1
24466;;Saint-Michel-de-Montaigne;SAINT-MICHEL-DE-MONTAIGNE;24230;24;DORDOGNE;1
24468;;Saint-Michel-de-Villadeix;SAINT-MICHEL-DE-VILLADEIX;24380;24;DORDOGNE;1
24470;;Sainte-Mondane;SAINTE-MONDANE;24370;24;DORDOGNE;1
24471;;Sainte-Nathalne;SAINTE-NATHALENE;24200;24;DORDOGNE;1
24472;;Saint-Nexans;SAINT-NEXANS;24520;24;DORDOGNE;1
24473;;Sainte-Orse;SAINTE-ORSE;24210;24;DORDOGNE;1
24474;;Saint-Pancrace;SAINT-PANCRACE;24530;24;DORDOGNE;1
24475;;Saint-Pantaly-d'Ans;SAINT-PANTALY-D'ANS;24640;24;DORDOGNE;1
24476;;Saint-Pantaly-d'Excideuil;SAINT-PANTALY-D'EXCIDEUIL;24160;24;DORDOGNE;1
24477;;Saint-Pardoux-de-Drne;SAINT-PARDOUX-DE-DRONE;24600;24;DORDOGNE;1
24478;;Saint-Pardoux-et-Vielvic;SAINT-PARDOUX-ET-VIELVIC;24170;24;DORDOGNE;1
24479;;Saint-Pardoux-la-Rivire;SAINT-PARDOUX-LA-RIVIERE;24470;24;DORDOGNE;1
24480;;Saint-Paul-de-Serre;SAINT-PAUL-DE-SERRE;24380;24;DORDOGNE;1
24481;;Saint-Paul-la-Roche;SAINT-PAUL-LA-ROCHE;24800;24;DORDOGNE;1
24482;;Saint-Paul-Lizonne;SAINT-PAUL-LIZONNE;24320;24;DORDOGNE;1
24483;;Saint-Perdoux;SAINT-PERDOUX;24560;24;DORDOGNE;1
24484;;Saint-Pierre-de-Chignac;SAINT-PIERRE-DE-CHIGNAC;24330;24;DORDOGNE;1
24485;;Saint-Pierre-de-Cle;SAINT-PIERRE-DE-COLE;24800;24;DORDOGNE;1
24486;;Saint-Pierre-de-Frugie;SAINT-PIERRE-DE-FRUGIE;24450;24;DORDOGNE;1
24487;;Saint-Pierre-d'Eyraud;SAINT-PIERRE-D'EYRAUD;24130;24;DORDOGNE;1
24488;;Saint-Pompont;SAINT-POMPONT;24170;24;DORDOGNE;1
24489;;Saint-Priest-les-Fougres;SAINT-PRIEST-LES-FOUGERES;24450;24;DORDOGNE;1
24490;;Saint-Privat-des-Prs;SAINT-PRIVAT-DES-PRES;24410;24;DORDOGNE;1
24491;;Saint-Rabier;SAINT-RABIER;24210;24;DORDOGNE;1
24492;;Sainte-Radegonde;SAINTE-RADEGONDE;24560;24;DORDOGNE;1
24493;;Saint-Raphal;SAINT-RAPHAEL;24160;24;DORDOGNE;1
24494;;Saint-Rmy;SAINT-REMY;24700;24;DORDOGNE;1
24495;;Saint-Romain-de-Monpazier;SAINT-ROMAIN-DE-MONPAZIER;24540;24;DORDOGNE;1
24496;;Saint-Romain-et-Saint-Clment;SAINT-ROMAIN-ET-SAINT-CLEMENT;24800;24;DORDOGNE;1
24497;;Sainte-Sabine-Born;SAINTE-SABINE-BORN;24440;24;DORDOGNE;1
24498;;Saint-Saud-Lacoussire;SAINT-SAUD-LACOUSSIERE;24470;24;DORDOGNE;1
24499;;Saint-Sauveur;SAINT-SAUVEUR;24520;24;DORDOGNE;1
24500;;Saint-Sauveur-Lalande;SAINT-SAUVEUR-LALANDE;24700;24;DORDOGNE;1
24501;;Saint-Seurin-de-Prats;SAINT-SEURIN-DE-PRATS;24230;24;DORDOGNE;1
24502;;Saint-Sverin-d'Estissac;SAINT-SEVERIN-D'ESTISSAC;24190;24;DORDOGNE;1
24503;;Saint-Sulpice-de-Mareuil;SAINT-SULPICE-DE-MAREUIL;24340;24;DORDOGNE;1
24504;;Saint-Sulpice-de-Roumagnac;SAINT-SULPICE-DE-ROUMAGNAC;24600;24;DORDOGNE;1
24505;;Saint-Sulpice-d'Excideuil;SAINT-SULPICE-D'EXCIDEUIL;24800;24;DORDOGNE;1
24507;;Sainte-Trie;SAINTE-TRIE;24160;24;DORDOGNE;1
24508;;Saint-Victor;SAINT-VICTOR;24350;24;DORDOGNE;1
24509;;Saint-Vincent-de-Connezac;SAINT-VINCENT-DE-CONNEZAC;24190;24;DORDOGNE;1
24510;;Saint-Vincent-de-Cosse;SAINT-VINCENT-DE-COSSE;24220;24;DORDOGNE;1
24511;;Saint-Vincent-Jalmoutiers;SAINT-VINCENT-JALMOUTIERS;24410;24;DORDOGNE;1
24512;;Saint-Vincent-le-Paluel;SAINT-VINCENT-LE-PALUEL;24200;24;DORDOGNE;1
24513;;Saint-Vincent-sur-l'Isle;SAINT-VINCENT-SUR-L'ISLE;24420;24;DORDOGNE;1
24514;;Saint-Vivien;SAINT-VIVIEN;24230;24;DORDOGNE;1
24515;;Salagnac;SALAGNAC;24160;24;DORDOGNE;1
24516;;Salignac-Eyvigues;SALIGNAC-EYVIGUES;24590;24;DORDOGNE;1
24517;;Salles-de-Belvs;SALLES-DE-BELVES;24170;24;DORDOGNE;1
24518;;Salon;SALON;24380;24;DORDOGNE;1
24519;;Sarlande;SARLANDE;24270;24;DORDOGNE;1
24520;;Sarlat-la-Canda;SARLAT-LA-CANEDA;24200;24;DORDOGNE;1
24521;;Sarliac-sur-l'Isle;SARLIAC-SUR-L'ISLE;24420;24;DORDOGNE;1
24522;;Sarrazac;SARRAZAC;24800;24;DORDOGNE;1
24523;;Saussignac;SAUSSIGNAC;24240;24;DORDOGNE;1
24524;;Savignac-de-Miremont;SAVIGNAC-DE-MIREMONT;24260;24;DORDOGNE;1
24525;;Savignac-de-Nontron;SAVIGNAC-DE-NONTRON;24300;24;DORDOGNE;1
24526;;Savignac-Ldrier;SAVIGNAC-LEDRIER;24270;24;DORDOGNE;1
24527;;Savignac-les-glises;SAVIGNAC-LES-EGLISES;24420;24;DORDOGNE;1
24528;;Sceau-Saint-Angel;SCEAU-SAINT-ANGEL;24300;24;DORDOGNE;1
24529;;Segonzac;SEGONZAC;24600;24;DORDOGNE;1
24530;;Sencenac-Puy-de-Fourches;SENCENAC-PUY-DE-FOURCHES;24310;24;DORDOGNE;1
24531;;Sergeac;SERGEAC;24290;24;DORDOGNE;1
24532;;Serres-et-Montguyard;SERRES-ET-MONTGUYARD;24500;24;DORDOGNE;1
24533;;Servanches;SERVANCHES;24410;24;DORDOGNE;1
24534;;Sigouls;SIGOULES;24240;24;DORDOGNE;1
24535;;Simeyrols;SIMEYROLS;24370;24;DORDOGNE;1
24536;;Singleyrac;SINGLEYRAC;24500;24;DORDOGNE;1
24537;;Siorac-de-Ribrac;SIORAC-DE-RIBERAC;24600;24;DORDOGNE;1
24538;;Siorac-en-Prigord;SIORAC-EN-PERIGORD;24170;24;DORDOGNE;1
24540;;Sorges;SORGES;24420;24;DORDOGNE;1
24541;;Soudat;SOUDAT;24360;24;DORDOGNE;1
24542;;Soulaures;SOULAURES;24540;24;DORDOGNE;1
24543;;Sourzac;SOURZAC;24400;24;DORDOGNE;1
24544;;Tamnis;TAMNIES;24620;24;DORDOGNE;1
24545;;Teillots;TEILLOTS;24390;24;DORDOGNE;1
24546;;Temple-Laguyon;TEMPLE-LAGUYON;24390;24;DORDOGNE;1
24547;;Terrasson-Lavilledieu;TERRASSON-LAVILLEDIEU;24120;24;DORDOGNE;1
24548;;Teyjat;TEYJAT;24300;24;DORDOGNE;1
24549;;Thnac;THENAC;24240;24;DORDOGNE;1
24550;;Thenon;THENON;24210;24;DORDOGNE;1
24551;;Thiviers;THIVIERS;24800;24;DORDOGNE;1
24552;;Thonac;THONAC;24290;24;DORDOGNE;1
24553;;Tocane-Saint-Apre;TOCANE-SAINT-APRE;24350;24;DORDOGNE;1
24554;La;Tour-Blanche;TOUR-BLANCHE;24320;24;DORDOGNE;1
24555;;Tourtoirac;TOURTOIRAC;24390;24;DORDOGNE;1
24557;;Trlissac;TRELISSAC;24750;24;DORDOGNE;1
24558;;Trmolat;TREMOLAT;24510;24;DORDOGNE;1
24559;;Tursac;TURSAC;24620;24;DORDOGNE;1
24560;;Urval;URVAL;24480;24;DORDOGNE;1
24561;;Valeuil;VALEUIL;24310;24;DORDOGNE;1
24562;;Vallereuil;VALLEREUIL;24190;24;DORDOGNE;1
24563;;Valojoulx;VALOJOULX;24290;24;DORDOGNE;1
24564;;Vanxains;VANXAINS;24600;24;DORDOGNE;1
24565;;Varaignes;VARAIGNES;24360;24;DORDOGNE;1
24566;;Varennes;VARENNES;24150;24;DORDOGNE;1
24567;;Vaunac;VAUNAC;24800;24;DORDOGNE;1
24568;;Vlines;VELINES;24230;24;DORDOGNE;1
24569;;Vendoire;VENDOIRE;24320;24;DORDOGNE;1
24570;;Verdon;VERDON;24520;24;DORDOGNE;1
24571;;Vergt;VERGT;24380;24;DORDOGNE;1
24572;;Vergt-de-Biron;VERGT-DE-BIRON;24540;24;DORDOGNE;1
24573;;Verteillac;VERTEILLAC;24320;24;DORDOGNE;1
24574;;Veyrignac;VEYRIGNAC;24370;24;DORDOGNE;1
24575;;Veyrines-de-Domme;VEYRINES-DE-DOMME;24250;24;DORDOGNE;1
24576;;Veyrines-de-Vergt;VEYRINES-DE-VERGT;24380;24;DORDOGNE;1
24577;;Vzac;VEZAC;24220;24;DORDOGNE;1
24579;;Vieux-Mareuil;VIEUX-MAREUIL;24340;24;DORDOGNE;1
24580;;Villac;VILLAC;24120;24;DORDOGNE;1
24581;;Villamblard;VILLAMBLARD;24140;24;DORDOGNE;1
24582;;Villars;VILLARS;24530;24;DORDOGNE;1
24584;;Villefranche-de-Lonchat;VILLEFRANCHE-DE-LONCHAT;24610;24;DORDOGNE;1
24585;;Villefranche-du-Prigord;VILLEFRANCHE-DU-PERIGORD;24550;24;DORDOGNE;1
24586;;Villetoureix;VILLETOUREIX;24600;24;DORDOGNE;1
24587;;Vitrac;VITRAC;24200;24;DORDOGNE;1
25001;;Abbans-Dessous;ABBANS-DESSOUS;25320;25;DOUBS;1
25002;;Abbans-Dessus;ABBANS-DESSUS;25440;25;DOUBS;1
25003;;Abbenans;ABBENANS;25340;25;DOUBS;1
25004;;Abbvillers;ABBEVILLERS;25310;25;DOUBS;1
25005;;Accolans;ACCOLANS;25250;25;DOUBS;1
25006;;Adam-ls-Passavant;ADAM-LES-PASSAVANT;25360;25;DOUBS;1
25007;;Adam-ls-Vercel;ADAM-LES-VERCEL;25530;25;DOUBS;1
25008;;Aibre;AIBRE;25750;25;DOUBS;1
25009;;Assey;AISSEY;25360;25;DOUBS;1
25011;;Allenjoie;ALLENJOIE;25490;25;DOUBS;1
25012;Les;Allis;ALLIES;25300;25;DOUBS;1
25013;;Allondans;ALLONDANS;25550;25;DOUBS;1
25014;;Amagney;AMAGNEY;25220;25;DOUBS;1
25015;;Amancey;AMANCEY;25330;25;DOUBS;1
25016;;Amathay-Vsigneux;AMATHAY-VESIGNEUX;25330;25;DOUBS;1
25017;;Amondans;AMONDANS;25330;25;DOUBS;1
25018;;Anteuil;ANTEUIL;25340;25;DOUBS;1
25019;;Appenans;APPENANS;25250;25;DOUBS;1
25020;;Arbouans;ARBOUANS;25400;25;DOUBS;1
25021;;Arc-et-Senans;ARC-ET-SENANS;25610;25;DOUBS;1
25022;;Arcey;ARCEY;25750;25;DOUBS;1
25024;;Aron;ARCON;25300;25;DOUBS;1
25025;;Arc-sous-Cicon;ARC-SOUS-CICON;25520;25;DOUBS;1
25026;;Arc-sous-Montenot;ARC-SOUS-MONTENOT;25270;25;DOUBS;1
25027;;Arguel;ARGUEL;25720;25;DOUBS;1
25028;;Athose;ATHOSE;25580;25;DOUBS;1
25029;;Aubonne;AUBONNE;25520;25;DOUBS;1
25030;;Audeux;AUDEUX;25170;25;DOUBS;1
25031;;Audincourt;AUDINCOURT;25400;25;DOUBS;1
25032;;Autechaux;AUTECHAUX;25110;25;DOUBS;1
25033;;Autechaux-Roide;AUTECHAUX-ROIDE;25150;25;DOUBS;1
25034;;Auxon-Dessous;AUXON-DESSOUS;25870;25;DOUBS;1
25035;;Auxon-Dessus;AUXON-DESSUS;25870;25;DOUBS;1
25036;;Avanne-Aveney;AVANNE-AVENEY;25720;25;DOUBS;1
25038;;Avilley;AVILLEY;25680;25;DOUBS;1
25039;;Avoudrey;AVOUDREY;25690;25;DOUBS;1
25040;;Badevel;BADEVEL;25490;25;DOUBS;1
25041;;Bannans;BANNANS;25560;25;DOUBS;1
25042;Le;Barboux;BARBOUX;25210;25;DOUBS;1
25043;;Bart;BART;25420;25;DOUBS;1
25044;;Bartherans;BARTHERANS;25440;25;DOUBS;1
25045;;Battenans-les-Mines;BATTENANS-LES-MINES;25640;25;DOUBS;1
25046;;Battenans-Varin;BATTENANS-VARIN;25380;25;DOUBS;1
25047;;Baume-les-Dames;BAUME-LES-DAMES;25110;25;DOUBS;1
25048;;Bavans;BAVANS;25550;25;DOUBS;1
25049;;Belfays;BELFAYS;25470;25;DOUBS;1
25050;Le;Blieu;BELIEU;25500;25;DOUBS;1
25051;;Belleherbe;BELLEHERBE;25380;25;DOUBS;1
25052;;Belmont;BELMONT;25530;25;DOUBS;1
25053;;Belvoir;BELVOIR;25430;25;DOUBS;1
25054;;Berche;BERCHE;25420;25;DOUBS;1
25055;;Berthelange;BERTHELANGE;25410;25;DOUBS;1
25056;;Besanon;BESANCON;25000;25;DOUBS;1
25057;;Bethoncourt;BETHONCOURT;25200;25;DOUBS;1
25058;;Beure;BEURE;25720;25;DOUBS;1
25059;;Beutal;BEUTAL;25250;25;DOUBS;1
25060;;Bians-les-Usiers;BIANS-LES-USIERS;25520;25;DOUBS;1
25061;;Bief;BIEF;25190;25;DOUBS;1
25062;Le;Bizot;BIZOT;25210;25;DOUBS;1
25063;;Blamont;BLAMONT;25310;25;DOUBS;1
25065;;Blarians;BLARIANS;25640;25;DOUBS;1
25066;;Blussangeaux;BLUSSANGEAUX;25250;25;DOUBS;1
25067;;Blussans;BLUSSANS;25250;25;DOUBS;1
25070;;Bolandoz;BOLANDOZ;25330;25;DOUBS;1
25071;;Bondeval;BONDEVAL;25230;25;DOUBS;1
25072;;Bonnal;BONNAL;25680;25;DOUBS;1
25073;;Bonnay;BONNAY;25870;25;DOUBS;1
25074;;Bonntage;BONNETAGE;25210;25;DOUBS;1
25075;;Bonnevaux;BONNEVAUX;25560;25;DOUBS;1
25076;;Bonnevaux-le-Prieur;BONNEVAUX-LE-PRIEURE;25620;25;DOUBS;1
25077;La;Bosse;BOSSE;25210;25;DOUBS;1
25078;;Bouclans;BOUCLANS;25360;25;DOUBS;1
25079;;Boujailles;BOUJAILLES;25560;25;DOUBS;1
25082;;Bourguignon;BOURGUIGNON;25150;25;DOUBS;1
25083;;Bournois;BOURNOIS;25250;25;DOUBS;1
25084;;Boussires;BOUSSIERES;25320;25;DOUBS;1
25085;;Bouverans;BOUVERANS;25560;25;DOUBS;1
25086;;Braillans;BRAILLANS;25640;25;DOUBS;1
25087;;Branne;BRANNE;25340;25;DOUBS;1
25088;;Breconchaux;BRECONCHAUX;25640;25;DOUBS;1
25089;;Bremondans;BREMONDANS;25530;25;DOUBS;1
25090;;Brres;BRERES;25440;25;DOUBS;1
25091;Les;Brseux;BRESEUX;25120;25;DOUBS;1
25092;La;Bretenire;BRETENIERE;25640;25;DOUBS;1
25093;;Bretigney;BRETIGNEY;25250;25;DOUBS;1
25094;;Bretigney-Notre-Dame;BRETIGNEY-NOTRE-DAME;25110;25;DOUBS;1
25095;;Bretonvillers;BRETONVILLERS;25380;25;DOUBS;1
25096;;Brey-et-Maison-du-Bois;BREY-ET-MAISON-DU-BOIS;25240;25;DOUBS;1
25097;;Brognard;BROGNARD;25600;25;DOUBS;1
25098;;Buffard;BUFFARD;25440;25;DOUBS;1
25099;;Bugny;BUGNY;25520;25;DOUBS;1
25100;;Bulle;BULLE;25560;25;DOUBS;1
25101;;Burgille;BURGILLE;25170;25;DOUBS;1
25102;;Burnevillers;BURNEVILLERS;25470;25;DOUBS;1
25103;;Busy;BUSY;25320;25;DOUBS;1
25104;;By;BY;25440;25;DOUBS;1
25105;;Byans-sur-Doubs;BYANS-SUR-DOUBS;25320;25;DOUBS;1
25106;;Cademne;CADEMENE;25290;25;DOUBS;1
25107;;Cendrey;CENDREY;25640;25;DOUBS;1
25108;;Cernay-l'glise;CERNAY-L'EGLISE;25120;25;DOUBS;1
25109;;Cessey;CESSEY;25440;25;DOUBS;1
25110;;Chaffois;CHAFFOIS;25300;25;DOUBS;1
25111;;Chalze;CHALEZE;25220;25;DOUBS;1
25112;;Chalezeule;CHALEZEULE;25220;25;DOUBS;1
25113;;Chamesey;CHAMESEY;25380;25;DOUBS;1
25114;;Chamesol;CHAMESOL;25190;25;DOUBS;1
25115;;Champagney;CHAMPAGNEY;25170;25;DOUBS;1
25116;;Champlive;CHAMPLIVE;25360;25;DOUBS;1
25117;;Champoux;CHAMPOUX;25640;25;DOUBS;1
25119;;Champvans-les-Moulins;CHAMPVANS-LES-MOULINS;25170;25;DOUBS;1
25120;;Chantrans;CHANTRANS;25330;25;DOUBS;1
25121;;Chapelle-des-Bois;CHAPELLE-DES-BOIS;25240;25;DOUBS;1
25122;;Chapelle-d'Huin;CHAPELLE-D'HUIN;25270;25;DOUBS;1
25123;;Charbonnires-les-Sapins;CHARBONNIERES-LES-SAPINS;25620;25;DOUBS;1
25124;;Charmauvillers;CHARMAUVILLERS;25470;25;DOUBS;1
25125;;Charmoille;CHARMOILLE;25380;25;DOUBS;1
25126;;Charnay;CHARNAY;25440;25;DOUBS;1
25127;;Charquemont;CHARQUEMONT;25140;25;DOUBS;1
25128;;Chasnans;CHASNANS;25580;25;DOUBS;1
25129;;Chassagne-Saint-Denis;CHASSAGNE-SAINT-DENIS;25290;25;DOUBS;1
25130;;Chteauvieux-les-Fosss;CHATEAUVIEUX-LES-FOSSES;25840;25;DOUBS;1
25131;;Chtelblanc;CHATELBLANC;25240;25;DOUBS;1
25132;;Chtillon-Guyotte;CHATILLON-GUYOTTE;25640;25;DOUBS;1
25133;;Chtillon-le-Duc;CHATILLON-LE-DUC;25870;25;DOUBS;1
25134;;Chtillon-sur-Lison;CHATILLON-SUR-LISON;25440;25;DOUBS;1
25136;;Chaucenne;CHAUCENNE;25170;25;DOUBS;1
25137;;Chaudefontaine;CHAUDEFONTAINE;25640;25;DOUBS;1
25138;Les;Terres-de-Chaux;TERRES-DE-CHAUX;25190;25;DOUBS;1
25139;La;Chaux;CHAUX;25650;25;DOUBS;1
25140;;Chaux-ls-Clerval;CHAUX-LES-CLERVAL;25340;25;DOUBS;1
25141;;Chaux-ls-Passavant;CHAUX-LES-PASSAVANT;25530;25;DOUBS;1
25142;;Chaux-Neuve;CHAUX-NEUVE;25240;25;DOUBS;1
25143;;Chay;CHAY;25440;25;DOUBS;1
25145;;Chazot;CHAZOT;25430;25;DOUBS;1
25147;;Chemaudin;CHEMAUDIN;25320;25;DOUBS;1
25148;La;Chenalotte;CHENALOTTE;25500;25;DOUBS;1
25149;;Chenecey-Buillon;CHENECEY-BUILLON;25440;25;DOUBS;1
25150;;Chevigney-sur-l'Ognon;CHEVIGNEY-SUR-L'OGNON;25170;25;DOUBS;1
25151;;Chevigney-ls-Vercel;CHEVIGNEY-LES-VERCEL;25530;25;DOUBS;1
25152;La;Chevillotte;CHEVILLOTTE;25620;25;DOUBS;1
25153;;Chevroz;CHEVROZ;25870;25;DOUBS;1
25154;;Chouzelot;CHOUZELOT;25440;25;DOUBS;1
25155;;Clron;CLERON;25330;25;DOUBS;1
25156;;Clerval;CLERVAL;25340;25;DOUBS;1
25157;La;Cluse-et-Mijoux;CLUSE-ET-MIJOUX;25300;25;DOUBS;1
25159;;Colombier-Fontaine;COLOMBIER-FONTAINE;25260;25;DOUBS;1
25160;Les;Combes;COMBES;25500;25;DOUBS;1
25161;;Consolation-Maisonnettes;CONSOLATION-MAISONNETTES;25390;25;DOUBS;1
25162;;Corcelles-Ferrires;CORCELLES-FERRIERES;25410;25;DOUBS;1
25163;;Corcelle-Mieslot;CORCELLE-MIESLOT;25640;25;DOUBS;1
25164;;Corcondray;CORCONDRAY;25410;25;DOUBS;1
25166;;Ctebrune;COTEBRUNE;25360;25;DOUBS;1
25170;;Courcelles-ls-Montbliard;COURCELLES-LES-MONTBELIARD;25420;25;DOUBS;1
25171;;Courcelles;COURCELLES;25440;25;DOUBS;1
25172;;Courchapon;COURCHAPON;25170;25;DOUBS;1
25173;;Cour-Saint-Maurice;COUR-SAINT-MAURICE;25380;25;DOUBS;1
25174;;Courtefontaine;COURTEFONTAINE;25470;25;DOUBS;1
25175;;Courtetain-et-Salans;COURTETAIN-ET-SALANS;25530;25;DOUBS;1
25176;;Courvires;COURVIERES;25560;25;DOUBS;1
25177;;Crosey-le-Grand;CROSEY-LE-GRAND;25340;25;DOUBS;1
25178;;Crosey-le-Petit;CROSEY-LE-PETIT;25340;25;DOUBS;1
25179;Le;Crouzet;CROUZET;25240;25;DOUBS;1
25180;;Crouzet-Migette;CROUZET-MIGETTE;25270;25;DOUBS;1
25181;;Cubrial;CUBRIAL;25680;25;DOUBS;1
25182;;Cubry;CUBRY;25680;25;DOUBS;1
25183;;Cusance;CUSANCE;25110;25;DOUBS;1
25184;;Cuse-et-Adrisans;CUSE-ET-ADRISANS;25680;25;DOUBS;1
25185;;Cussey-sur-Lison;CUSSEY-SUR-LISON;25440;25;DOUBS;1
25186;;Cussey-sur-l'Ognon;CUSSEY-SUR-L'OGNON;25870;25;DOUBS;1
25187;;Dambelin;DAMBELIN;25150;25;DOUBS;1
25188;;Dambenois;DAMBENOIS;25600;25;DOUBS;1
25189;;Dammartin-les-Templiers;DAMMARTIN-LES-TEMPLIERS;25110;25;DOUBS;1
25190;;Dampierre-les-Bois;DAMPIERRE-LES-BOIS;25490;25;DOUBS;1
25191;;Dampierre-sur-le-Doubs;DAMPIERRE-SUR-LE-DOUBS;25420;25;DOUBS;1
25192;;Dampjoux;DAMPJOUX;25190;25;DOUBS;1
25193;;Damprichard;DAMPRICHARD;25450;25;DOUBS;1
25194;;Dannemarie;DANNEMARIE;25310;25;DOUBS;1
25195;;Dannemarie-sur-Crte;DANNEMARIE-SUR-CRETE;25410;25;DOUBS;1
25196;;Dasle;DASLE;25230;25;DOUBS;1
25197;;Deluz;DELUZ;25960;25;DOUBS;1
25198;;Dsandans;DESANDANS;25750;25;DOUBS;1
25199;;Dservillers;DESERVILLERS;25330;25;DOUBS;1
25200;;Devecey;DEVECEY;25870;25;DOUBS;1
25201;;Dommartin;DOMMARTIN;25300;25;DOUBS;1
25202;;Dompierre-les-Tilleuls;DOMPIERRE-LES-TILLEULS;25560;25;DOUBS;1
25203;;Domprel;DOMPREL;25510;25;DOUBS;1
25204;;Doubs;DOUBS;25300;25;DOUBS;1
25207;;Dung;DUNG;25550;25;DOUBS;1
25208;;Durnes;DURNES;25580;25;DOUBS;1
25209;;chay;ECHAY;25440;25;DOUBS;1
25210;;chenans;ECHENANS;25550;25;DOUBS;1
25211;;chevannes;ECHEVANNES;25580;25;DOUBS;1
25212;;cole-Valentin;ECOLE-VALENTIN;25480;25;DOUBS;1
25213;Les;corces;ECORCES;25140;25;DOUBS;1
25214;;cot;ECOT;25150;25;DOUBS;1
25215;L';couvotte;ECOUVOTTE;25640;25;DOUBS;1
25216;;curcey;ECURCEY;25150;25;DOUBS;1
25217;;magny;EMAGNY;25170;25;DOUBS;1
25218;;penouse;EPENOUSE;25530;25;DOUBS;1
25219;;penoy;EPENOY;25800;25;DOUBS;1
25220;;peugney;EPEUGNEY;25290;25;DOUBS;1
25221;;Esnans;ESNANS;25110;25;DOUBS;1
25222;;talans;ETALANS;25580;25;DOUBS;1
25223;;ternoz;ETERNOZ;25330;25;DOUBS;1
25224;;touvans;ETOUVANS;25260;25;DOUBS;1
25225;;trabonne;ETRABONNE;25170;25;DOUBS;1
25226;;trappe;ETRAPPE;25250;25;DOUBS;1
25227;;tray;ETRAY;25800;25;DOUBS;1
25228;;tupes;ETUPES;25460;25;DOUBS;1
25229;;villers;EVILLERS;25520;25;DOUBS;1
25230;;Exincourt;EXINCOURT;25400;25;DOUBS;1
25231;;Eysson;EYSSON;25530;25;DOUBS;1
25232;;Faimbe;FAIMBE;25250;25;DOUBS;1
25233;;Fallerans;FALLERANS;25580;25;DOUBS;1
25234;;Ferrires-le-Lac;FERRIERES-LE-LAC;25470;25;DOUBS;1
25235;;Ferrires-les-Bois;FERRIERES-LES-BOIS;25410;25;DOUBS;1
25236;;Fertans;FERTANS;25330;25;DOUBS;1
25237;;Fesches-le-Chtel;FESCHES-LE-CHATEL;25490;25;DOUBS;1
25238;;Fessevillers;FESSEVILLERS;25470;25;DOUBS;1
25239;;Feule;FEULE;25190;25;DOUBS;1
25240;Les;Fins;FINS;25500;25;DOUBS;1
25241;;Flagey;FLAGEY;25330;25;DOUBS;1
25242;;Flagey-Rigney;FLAGEY-RIGNEY;25640;25;DOUBS;1
25243;;Flangebouche;FLANGEBOUCHE;25690;25;DOUBS;1
25244;;Fleurey;FLEUREY;25190;25;DOUBS;1
25245;;Fontain;FONTAIN;25660;25;DOUBS;1
25246;;Fontaine-ls-Clerval;FONTAINE-LES-CLERVAL;25340;25;DOUBS;1
25247;;Fontenelle-Montby;FONTENELLE-MONTBY;25340;25;DOUBS;1
25248;Les;Fontenelles;FONTENELLES;25210;25;DOUBS;1
25249;;Fontenotte;FONTENOTTE;25110;25;DOUBS;1
25250;;Foucherans;FOUCHERANS;25620;25;DOUBS;1
25251;;Fourbanne;FOURBANNE;25110;25;DOUBS;1
25252;;Fourcatier-et-Maison-Neuve;FOURCATIER-ET-MAISON-NEUVE;25370;25;DOUBS;1
25253;;Fourg;FOURG;25440;25;DOUBS;1
25254;Les;Fourgs;FOURGS;25300;25;DOUBS;1
25255;;Fournet-Blancheroche;FOURNET-BLANCHEROCHE;25140;25;DOUBS;1
25256;;Frambouhans;FRAMBOUHANS;25140;25;DOUBS;1
25257;;Franey;FRANEY;25170;25;DOUBS;1
25258;;Franois;FRANOIS;25770;25;DOUBS;1
25259;;Frasne;FRASNE;25560;25;DOUBS;1
25261;;Froidevaux;FROIDEVAUX;25190;25;DOUBS;1
25262;;Fuans;FUANS;25390;25;DOUBS;1
25263;;Gellin;GELLIN;25240;25;DOUBS;1
25264;;Gmonval;GEMONVAL;25250;25;DOUBS;1
25265;;Geneuille;GENEUILLE;25870;25;DOUBS;1
25266;;Geney;GENEY;25250;25;DOUBS;1
25267;;Gennes;GENNES;25660;25;DOUBS;1
25268;;Germfontaine;GERMEFONTAINE;25510;25;DOUBS;1
25269;;Germondans;GERMONDANS;25640;25;DOUBS;1
25270;;Gevresin;GEVRESIN;25270;25;DOUBS;1
25271;;Gilley;GILLEY;25650;25;DOUBS;1
25273;;Glamondans;GLAMONDANS;25360;25;DOUBS;1
25274;;Glay;GLAY;25310;25;DOUBS;1
25275;;Glre;GLERE;25190;25;DOUBS;1
25276;;Gondenans-Montby;GONDENANS-MONTBY;25340;25;DOUBS;1
25277;;Gondenans-les-Moulins;GONDENANS-LES-MOULINS;25680;25;DOUBS;1
25278;;Gonsans;GONSANS;25360;25;DOUBS;1
25279;;Gouhelans;GOUHELANS;25680;25;DOUBS;1
25280;;Goumois;GOUMOIS;25470;25;DOUBS;1
25281;;Goux-ls-Dambelin;GOUX-LES-DAMBELIN;25150;25;DOUBS;1
25282;;Goux-les-Usiers;GOUX-LES-USIERS;25520;25;DOUBS;1
25283;;Goux-sous-Landet;GOUX-SOUS-LANDET;25440;25;DOUBS;1
25284;;Grand-Charmont;GRAND-CHARMONT;25200;25;DOUBS;1
25285;;Grand'Combe-Chteleu;GRAND'COMBE-CHATELEU;25570;25;DOUBS;1
25286;;Grand'Combe-des-Bois;GRAND'COMBE-DES-BOIS;25210;25;DOUBS;1
25287;;Grandfontaine;GRANDFONTAINE;25320;25;DOUBS;1
25288;;Fournets-Luisans;FOURNETS-LUISANS;25390;25;DOUBS;1
25289;;Grandfontaine-sur-Creuse;GRANDFONTAINE-SUR-CREUSE;25510;25;DOUBS;1
25290;La;Grange;GRANGE;25380;25;DOUBS;1
25293;;Granges-Narboz;GRANGES-NARBOZ;25300;25;DOUBS;1
25295;Les;Grangettes;GRANGETTES;25160;25;DOUBS;1
25296;Les;Gras;GRAS;25790;25;DOUBS;1
25297;Le;Gratteris;GRATTERIS;25620;25;DOUBS;1
25298;;Grosbois;GROSBOIS;25110;25;DOUBS;1
25299;;Guillon-les-Bains;GUILLON-LES-BAINS;25110;25;DOUBS;1
25300;;Guyans-Durnes;GUYANS-DURNES;25580;25;DOUBS;1
25301;;Guyans-Vennes;GUYANS-VENNES;25390;25;DOUBS;1
25302;;Hautepierre-le-Chtelet;HAUTEPIERRE-LE-CHATELET;25580;25;DOUBS;1
25303;;Hauterive-la-Fresse;HAUTERIVE-LA-FRESSE;25650;25;DOUBS;1
25304;;Hrimoncourt;HERIMONCOURT;25310;25;DOUBS;1
25305;L';Hpital-du-Grosbois;HOPITAL-DU-GROSBOIS;25620;25;DOUBS;1
25306;L';Hpital-Saint-Lieffroy;HOPITAL-SAINT-LIEFFROY;25340;25;DOUBS;1
25307;Les;Hpitaux-Neufs;HOPITAUX-NEUFS;25370;25;DOUBS;1
25308;Les;Hpitaux-Vieux;HOPITAUX-VIEUX;25370;25;DOUBS;1
25309;;Houtaud;HOUTAUD;25300;25;DOUBS;1
25310;;Huanne-Montmartin;HUANNE-MONTMARTIN;25680;25;DOUBS;1
25311;;Hymondans;HYEMONDANS;25250;25;DOUBS;1
25312;;Hyvre-Magny;HYEVRE-MAGNY;25110;25;DOUBS;1
25313;;Hyvre-Paroisse;HYEVRE-PAROISSE;25110;25;DOUBS;1
25314;;Indevillers;INDEVILLERS;25470;25;DOUBS;1
25315;L';Isle-sur-le-Doubs;ISLE-SUR-LE-DOUBS;25250;25;DOUBS;1
25316;;Issans;ISSANS;25550;25;DOUBS;1
25317;;Jallerange;JALLERANGE;25170;25;DOUBS;1
25318;;Jougne;JOUGNE;25370;25;DOUBS;1
25319;;Labergement-du-Navois;LABERGEMENT-DU-NAVOIS;25270;25;DOUBS;1
25320;;Labergement-Sainte-Marie;LABERGEMENT-SAINTE-MARIE;25160;25;DOUBS;1
25321;;Villers-le-Lac;VILLERS-LE-LAC;25130;25;DOUBS;1
25322;;Laire;LAIRE;25550;25;DOUBS;1
25323;;Laissey;LAISSEY;25820;25;DOUBS;1
25324;;Lanans;LANANS;25360;25;DOUBS;1
25325;;Landresse;LANDRESSE;25530;25;DOUBS;1
25326;;Lantenne-Vertire;LANTENNE-VERTIERE;25170;25;DOUBS;1
25327;;Lanthenans;LANTHENANS;25250;25;DOUBS;1
25328;;Larnod;LARNOD;25720;25;DOUBS;1
25329;;Laval-le-Prieur;LAVAL-LE-PRIEURE;25210;25;DOUBS;1
25330;;Lavans-Quingey;LAVANS-QUINGEY;25440;25;DOUBS;1
25331;;Lavans-Vuillafans;LAVANS-VUILLAFANS;25580;25;DOUBS;1
25332;;Lavernay;LAVERNAY;25170;25;DOUBS;1
25333;;Laviron;LAVIRON;25510;25;DOUBS;1
25334;;Levier;LEVIER;25270;25;DOUBS;1
25335;;Liebvillers;LIEBVILLERS;25190;25;DOUBS;1
25336;;Liesle;LIESLE;25440;25;DOUBS;1
25338;;Lizine;LIZINE;25330;25;DOUBS;1
25339;;Lods;LODS;25930;25;DOUBS;1
25340;;Lombard;LOMBARD;25440;25;DOUBS;1
25341;;Lomont-sur-Crte;LOMONT-SUR-CRETE;25110;25;DOUBS;1
25342;;Longechaux;LONGECHAUX;25690;25;DOUBS;1
25343;;Longemaison;LONGEMAISON;25690;25;DOUBS;1
25344;;Longevelle-ls-Russey;LONGEVELLE-LES-RUSSEY;25380;25;DOUBS;1
25345;;Longevelle-sur-Doubs;LONGEVELLE-SUR-DOUBS;25260;25;DOUBS;1
25346;;Longeville;LONGEVILLE;25330;25;DOUBS;1
25347;La;Longeville;LONGEVILLE;25650;25;DOUBS;1
25348;;Longevilles-Mont-d'Or;LONGEVILLES-MONT-D'OR;25370;25;DOUBS;1
25349;;Loray;LORAY;25390;25;DOUBS;1
25350;;Lougres;LOUGRES;25260;25;DOUBS;1
25351;Le;Luhier;LUHIER;25210;25;DOUBS;1
25354;;Luxiol;LUXIOL;25110;25;DOUBS;1
25355;;Magny-Chtelard;MAGNY-CHATELARD;25360;25;DOUBS;1
25356;;Mache;MAICHE;25120;25;DOUBS;1
25357;;Maisons-du-Bois-Livremont;MAISONS-DU-BOIS-LIEVREMONT;25650;25;DOUBS;1
25359;;Malans;MALANS;25330;25;DOUBS;1
25360;;Malbrans;MALBRANS;25620;25;DOUBS;1
25361;;Malbuisson;MALBUISSON;25160;25;DOUBS;1
25362;;Malpas;MALPAS;25160;25;DOUBS;1
25364;;Mamirolle;MAMIROLLE;25620;25;DOUBS;1
25365;;Mancenans;MANCENANS;25250;25;DOUBS;1
25366;;Mancenans-Lizerne;MANCENANS-LIZERNE;25120;25;DOUBS;1
25367;;Mandeure;MANDEURE;25350;25;DOUBS;1
25368;;Marchaux;MARCHAUX;25640;25;DOUBS;1
25369;;Marvelise;MARVELISE;25250;25;DOUBS;1
25370;;Mathay;MATHAY;25700;25;DOUBS;1
25371;;Mazerolles-le-Salin;MAZEROLLES-LE-SALIN;25170;25;DOUBS;1
25372;;Mdire;MEDIERE;25250;25;DOUBS;1
25373;Le;Mmont;MEMONT;25210;25;DOUBS;1
25374;;Mercey-le-Grand;MERCEY-LE-GRAND;25410;25;DOUBS;1
25375;;Mrey-sous-Montrond;MEREY-SOUS-MONTROND;25660;25;DOUBS;1
25376;;Mrey-Vieilley;MEREY-VIEILLEY;25870;25;DOUBS;1
25377;;Msandans;MESANDANS;25680;25;DOUBS;1
25378;;Meslires;MESLIERES;25310;25;DOUBS;1
25379;;Mesmay;MESMAY;25440;25;DOUBS;1
25380;;Mtabief;METABIEF;25370;25;DOUBS;1
25381;;Miserey-Salines;MISEREY-SALINES;25480;25;DOUBS;1
25382;;Moncey;MONCEY;25870;25;DOUBS;1
25383;;Moncley;MONCLEY;25170;25;DOUBS;1
25384;;Mondon;MONDON;25680;25;DOUBS;1
25385;;Montagney-Servigney;MONTAGNEY-SERVIGNEY;25680;25;DOUBS;1
25386;;Montancy;MONTANCY;25190;25;DOUBS;1
25387;;Montandon;MONTANDON;25190;25;DOUBS;1
25388;;Montbliard;MONTBELIARD;25200;25;DOUBS;1
25389;;Montbliardot;MONTBELIARDOT;25210;25;DOUBS;1
25390;;Montbenot;MONTBENOIT;25650;25;DOUBS;1
25391;;Mont-de-Laval;MONT-DE-LAVAL;25210;25;DOUBS;1
25392;;Mont-de-Vougney;MONT-DE-VOUGNEY;25120;25;DOUBS;1
25393;;Montcheroux;MONTECHEROUX;25190;25;DOUBS;1
25394;;Montenois;MONTENOIS;25260;25;DOUBS;1
25395;;Montfaucon;MONTFAUCON;25660;25;DOUBS;1
25397;;Montferrand-le-Chteau;MONTFERRAND-LE-CHATEAU;25320;25;DOUBS;1
25398;;Montflovin;MONTFLOVIN;25650;25;DOUBS;1
25399;;Montfort;MONTFORT;25440;25;DOUBS;1
25400;;Montgesoye;MONTGESOYE;25111;25;DOUBS;1
25401;;Montivernage;MONTIVERNAGE;25110;25;DOUBS;1
25402;;Montjoie-le-Chteau;MONTJOIE-LE-CHATEAU;25190;25;DOUBS;1
25403;;Montlebon;MONTLEBON;25500;25;DOUBS;1
25404;;Montmahoux;MONTMAHOUX;25270;25;DOUBS;1
25405;;Montperreux;MONTPERREUX;25160;25;DOUBS;1
25406;;Montrond-le-Chteau;MONTROND-LE-CHATEAU;25660;25;DOUBS;1
25408;;Montussaint;MONTUSSAINT;25680;25;DOUBS;1
25410;;Morre;MORRE;25660;25;DOUBS;1
25411;;Morteau;MORTEAU;25500;25;DOUBS;1
25413;;Mouthe;MOUTHE;25240;25;DOUBS;1
25414;Le;Moutherot;MOUTHEROT;25170;25;DOUBS;1
25415;;Mouthier-Haute-Pierre;MOUTHIER-HAUTE-PIERRE;25920;25;DOUBS;1
25416;;Myon;MYON;25440;25;DOUBS;1
25417;;Naisey-les-Granges;NAISEY-LES-GRANGES;25360;25;DOUBS;1
25418;;Nancray;NANCRAY;25360;25;DOUBS;1
25419;;Nans;NANS;25680;25;DOUBS;1
25420;;Nans-sous-Sainte-Anne;NANS-SOUS-SAINTE-ANNE;25330;25;DOUBS;1
25421;;Narbief;NARBIEF;25210;25;DOUBS;1
25422;;Neuchtel-Urtire;NEUCHATEL-URTIERE;25150;25;DOUBS;1
25424;;Nods;NODS;25580;25;DOUBS;1
25425;;Nol-Cerneux;NOEL-CERNEUX;25500;25;DOUBS;1
25426;;Noirefontaine;NOIREFONTAINE;25190;25;DOUBS;1
25427;;Noironte;NOIRONTE;25170;25;DOUBS;1
25428;;Nommay;NOMMAY;25600;25;DOUBS;1
25429;;Novillars;NOVILLARS;25220;25;DOUBS;1
25430;;Ollans;OLLANS;25640;25;DOUBS;1
25431;;Onans;ONANS;25250;25;DOUBS;1
25432;;Orchamps-Vennes;ORCHAMPS-VENNES;25390;25;DOUBS;1
25433;;Orgeans-Blanchefontaine;ORGEANS-BLANCHEFONTAINE;25120;25;DOUBS;1
25434;;Ornans;ORNANS;25290;25;DOUBS;1
25435;;Orsans;ORSANS;25530;25;DOUBS;1
25436;;Orve;ORVE;25430;25;DOUBS;1
25437;;Osse;OSSE;25360;25;DOUBS;1
25438;;Osselle;OSSELLE;25320;25;DOUBS;1
25439;;Ougney-Douvot;OUGNEY-DOUVOT;25640;25;DOUBS;1
25440;;Ouhans;OUHANS;25520;25;DOUBS;1
25441;;Ouvans;OUVANS;25530;25;DOUBS;1
25442;;Oye-et-Pallet;OYE-ET-PALLET;25160;25;DOUBS;1
25443;;Palantine;PALANTINE;25440;25;DOUBS;1
25444;;Palise;PALISE;25870;25;DOUBS;1
25445;;Paroy;PAROY;25440;25;DOUBS;1
25446;;Passavant;PASSAVANT;25360;25;DOUBS;1
25447;;Passonfontaine;PASSONFONTAINE;25690;25;DOUBS;1
25448;;Pelousey;PELOUSEY;25170;25;DOUBS;1
25449;;Pseux;PESEUX;25190;25;DOUBS;1
25450;;Pessans;PESSANS;25440;25;DOUBS;1
25451;;Petite-Chaux;PETITE-CHAUX;25240;25;DOUBS;1
25452;;Pierrefontaine-ls-Blamont;PIERREFONTAINE-LES-BLAMONT;25310;25;DOUBS;1
25453;;Pierrefontaine-les-Varans;PIERREFONTAINE-LES-VARANS;25510;25;DOUBS;1
25454;;Pirey;PIREY;25480;25;DOUBS;1
25455;;Placey;PLACEY;25170;25;DOUBS;1
25456;;Plaimbois-du-Miroir;PLAIMBOIS-DU-MIROIR;25210;25;DOUBS;1
25457;;Plaimbois-Vennes;PLAIMBOIS-VENNES;25390;25;DOUBS;1
25458;Les;Plains-et-Grands-Essarts;PLAINS-ET-GRANDS-ESSARTS;25470;25;DOUBS;1
25459;La;Plane;PLANEE;25160;25;DOUBS;1
25460;;Pointvillers;POINTVILLERS;25440;25;DOUBS;1
25461;;Pompierre-sur-Doubs;POMPIERRE-SUR-DOUBS;25340;25;DOUBS;1
25462;;Pontarlier;PONTARLIER;25300;25;DOUBS;1
25463;;Pont-de-Roide;PONT-DE-ROIDE;25150;25;DOUBS;1
25464;Les;Pontets;PONTETS;25240;25;DOUBS;1
25465;;Pont-les-Moulins;PONT-LES-MOULINS;25110;25;DOUBS;1
25466;;Pouilley-Franais;POUILLEY-FRANCAIS;25410;25;DOUBS;1
25467;;Pouilley-les-Vignes;POUILLEY-LES-VIGNES;25115;25;DOUBS;1
25468;;Pouligney-Lusans;POULIGNEY-LUSANS;25640;25;DOUBS;1
25469;;Prsentevillers;PRESENTEVILLERS;25550;25;DOUBS;1
25470;La;Prtire;PRETIERE;25250;25;DOUBS;1
25471;;Provenchre;PROVENCHERE;25380;25;DOUBS;1
25472;;Puessans;PUESSANS;25680;25;DOUBS;1
25473;;Pugey;PUGEY;25720;25;DOUBS;1
25474;Le;Puy;PUY;25640;25;DOUBS;1
25475;;Quingey;QUINGEY;25440;25;DOUBS;1
25476;;Rahon;RAHON;25430;25;DOUBS;1
25477;;Rancenay;RANCENAY;25320;25;DOUBS;1
25478;;Randevillers;RANDEVILLERS;25430;25;DOUBS;1
25479;;Rang;RANG;25250;25;DOUBS;1
25480;;Rantechaux;RANTECHAUX;25580;25;DOUBS;1
25481;;Raynans;RAYNANS;25550;25;DOUBS;1
25482;;Recologne;RECOLOGNE;25170;25;DOUBS;1
25483;;Reculfoz;RECULFOZ;25240;25;DOUBS;1
25485;;Rmondans-Vaivre;REMONDANS-VAIVRE;25150;25;DOUBS;1
25486;;Remoray-Boujeons;REMORAY-BOUJEONS;25160;25;DOUBS;1
25487;;Rendale;RENEDALE;25520;25;DOUBS;1
25488;;Rennes-sur-Loue;RENNES-SUR-LOUE;25440;25;DOUBS;1
25489;;Reugney;REUGNEY;25330;25;DOUBS;1
25490;;Rigney;RIGNEY;25640;25;DOUBS;1
25491;;Rignosot;RIGNOSOT;25640;25;DOUBS;1
25492;;Rillans;RILLANS;25110;25;DOUBS;1
25493;La;Rivire-Drugeon;RIVIERE-DRUGEON;25560;25;DOUBS;1
25494;;Rochejean;ROCHEJEAN;25370;25;DOUBS;1
25495;;Roche-lez-Beaupr;ROCHE-LEZ-BEAUPRE;25220;25;DOUBS;1
25496;;Roche-ls-Clerval;ROCHE-LES-CLERVAL;25340;25;DOUBS;1
25497;;Roches-ls-Blamont;ROCHES-LES-BLAMONT;25310;25;DOUBS;1
25498;;Rognon;ROGNON;25680;25;DOUBS;1
25499;;Romain;ROMAIN;25680;25;DOUBS;1
25500;;Ronchaux;RONCHAUX;25440;25;DOUBS;1
25501;;Rondefontaine;RONDEFONTAINE;25240;25;DOUBS;1
25502;;Roset-Fluans;ROSET-FLUANS;25410;25;DOUBS;1
25503;;Rosires-sur-Barbche;ROSIERES-SUR-BARBECHE;25190;25;DOUBS;1
25504;;Rosureux;ROSUREUX;25380;25;DOUBS;1
25505;;Rougemont;ROUGEMONT;25680;25;DOUBS;1
25506;;Rougemontot;ROUGEMONTOT;25640;25;DOUBS;1
25507;;Rouhe;ROUHE;25440;25;DOUBS;1
25508;;Roulans;ROULANS;25640;25;DOUBS;1
25509;;Routelle;ROUTELLE;25410;25;DOUBS;1
25510;;Ruffey-le-Chteau;RUFFEY-LE-CHATEAU;25170;25;DOUBS;1
25511;;Rurey;RUREY;25290;25;DOUBS;1
25512;Le;Russey;RUSSEY;25210;25;DOUBS;1
25513;;Sainte-Anne;SAINTE-ANNE;25270;25;DOUBS;1
25514;;Saint-Antoine;SAINT-ANTOINE;25370;25;DOUBS;1
25515;;Sainte-Colombe;SAINTE-COLOMBE;25300;25;DOUBS;1
25516;;Saint-Georges-Armont;SAINT-GEORGES-ARMONT;25340;25;DOUBS;1
25517;;Saint-Gorgon-Main;SAINT-GORGON-MAIN;25520;25;DOUBS;1
25518;;Saint-Hilaire;SAINT-HILAIRE;25640;25;DOUBS;1
25519;;Saint-Hippolyte;SAINT-HIPPOLYTE;25190;25;DOUBS;1
25520;;Saint-Juan;SAINT-JUAN;25360;25;DOUBS;1
25521;;Saint-Julien-ls-Montbliard;SAINT-JULIEN-LES-MONTBELIARD;25550;25;DOUBS;1
25522;;Saint-Julien-ls-Russey;SAINT-JULIEN-LES-RUSSEY;25210;25;DOUBS;1
25523;;Sainte-Marie;SAINTE-MARIE;25113;25;DOUBS;1
25524;;Saint-Maurice-Colombier;SAINT-MAURICE-COLOMBIER;25260;25;DOUBS;1
25525;;Saint-Point-Lac;SAINT-POINT-LAC;25160;25;DOUBS;1
25526;;Sainte-Suzanne;SAINTE-SUZANNE;25630;25;DOUBS;1
25527;;Saint-Vit;SAINT-VIT;25410;25;DOUBS;1
25528;;Samson;SAMSON;25440;25;DOUBS;1
25529;;Sancey-le-Grand;SANCEY-LE-GRAND;25430;25;DOUBS;1
25530;;Sancey-le-Long;SANCEY-LE-LONG;25430;25;DOUBS;1
25531;;Santoche;SANTOCHE;25340;25;DOUBS;1
25532;;Sane;SAONE;25660;25;DOUBS;1
25533;;Saraz;SARAZ;25330;25;DOUBS;1
25534;;Sarrageois;SARRAGEOIS;25240;25;DOUBS;1
25535;;Saules;SAULES;25580;25;DOUBS;1
25536;;Sauvagney;SAUVAGNEY;25170;25;DOUBS;1
25537;;Scey-Maisires;SCEY-MAISIERES;25290;25;DOUBS;1
25538;;Schin;SECHIN;25110;25;DOUBS;1
25539;;Seloncourt;SELONCOURT;25230;25;DOUBS;1
25540;;Semondans;SEMONDANS;25750;25;DOUBS;1
25541;;Septfontaines;SEPTFONTAINES;25270;25;DOUBS;1
25542;;Serre-les-Sapins;SERRE-LES-SAPINS;25770;25;DOUBS;1
25544;;Servin;SERVIN;25430;25;DOUBS;1
25545;;Silley-Amancey;SILLEY-AMANCEY;25330;25;DOUBS;1
25546;;Silley-Blfond;SILLEY-BLEFOND;25110;25;DOUBS;1
25547;;Sochaux;SOCHAUX;25600;25;DOUBS;1
25548;;Solemont;SOLEMONT;25190;25;DOUBS;1
25549;;Sombacour;SOMBACOUR;25520;25;DOUBS;1
25550;La;Sommette;SOMMETTE;25510;25;DOUBS;1
25551;;Soulce-Cernay;SOULCE-CERNAY;25190;25;DOUBS;1
25552;;Sourans;SOURANS;25250;25;DOUBS;1
25553;;Soye;SOYE;25250;25;DOUBS;1
25554;;Surmont;SURMONT;25380;25;DOUBS;1
25555;;Taillecourt;TAILLECOURT;25400;25;DOUBS;1
25556;;Tallans;TALLANS;25680;25;DOUBS;1
25557;;Tallenay;TALLENAY;25870;25;DOUBS;1
25558;;Tarcenay;TARCENAY;25620;25;DOUBS;1
25559;;Thibouhans;THIEBOUHANS;25470;25;DOUBS;1
25560;;Thise;THISE;25220;25;DOUBS;1
25561;;Thoraise;THORAISE;25320;25;DOUBS;1
25562;;Thulay;THULAY;25310;25;DOUBS;1
25563;;Thurey-le-Mont;THUREY-LE-MONT;25870;25;DOUBS;1
25564;;Torpes;TORPES;25320;25;DOUBS;1
25565;;Touillon-et-Loutelet;TOUILLON-ET-LOUTELET;25370;25;DOUBS;1
25566;La;Tour-de-Say;TOUR-DE-SCAY;25640;25;DOUBS;1
25567;;Tournans;TOURNANS;25680;25;DOUBS;1
25569;;Trpot;TREPOT;25620;25;DOUBS;1
25570;;Tressandans;TRESSANDANS;25680;25;DOUBS;1
25571;;Trvillers;TREVILLERS;25470;25;DOUBS;1
25572;;Trouvans;TROUVANS;25680;25;DOUBS;1
25573;;Urtire;URTIERE;25470;25;DOUBS;1
25574;;Uzelle;UZELLE;25340;25;DOUBS;1
25575;;Vaire-Arcier;VAIRE-ARCIER;25220;25;DOUBS;1
25576;;Vaire-le-Petit;VAIRE-LE-PETIT;25220;25;DOUBS;1
25578;;Valdahon;VALDAHON;25800;25;DOUBS;1
25579;;Val-de-Roulans;VAL-DE-ROULANS;25640;25;DOUBS;1
25580;;Valentigney;VALENTIGNEY;25700;25;DOUBS;1
25582;;Valleroy;VALLEROY;25870;25;DOUBS;1
25583;;Valonne;VALONNE;25190;25;DOUBS;1
25584;;Valoreille;VALOREILLE;25190;25;DOUBS;1
25585;;Vanclans;VANCLANS;25580;25;DOUBS;1
25586;;Vandoncourt;VANDONCOURT;25230;25;DOUBS;1
25587;;Vauchamps;VAUCHAMPS;25360;25;DOUBS;1
25588;;Vaucluse;VAUCLUSE;25380;25;DOUBS;1
25589;;Vauclusotte;VAUCLUSOTTE;25380;25;DOUBS;1
25590;;Vaudrivillers;VAUDRIVILLERS;25360;25;DOUBS;1
25591;;Vaufrey;VAUFREY;25190;25;DOUBS;1
25592;;Vaux-et-Chantegrue;VAUX-ET-CHANTEGRUE;25160;25;DOUBS;1
25593;;Vaux-les-Prs;VAUX-LES-PRES;25770;25;DOUBS;1
25594;;Velesmes-Essarts;VELESMES-ESSARTS;25410;25;DOUBS;1
25595;;Vellerot-ls-Belvoir;VELLEROT-LES-BELVOIR;25430;25;DOUBS;1
25596;;Vellerot-ls-Vercel;VELLEROT-LES-VERCEL;25530;25;DOUBS;1
25597;;Vellevans;VELLEVANS;25430;25;DOUBS;1
25598;;Venise;VENISE;25870;25;DOUBS;1
25599;;Vennans;VENNANS;25640;25;DOUBS;1
25600;;Vennes;VENNES;25390;25;DOUBS;1
25601;;Vercel-Villedieu-le-Camp;VERCEL-VILLEDIEU-LE-CAMP;25530;25;DOUBS;1
25602;;Vergranne;VERGRANNE;25110;25;DOUBS;1
25604;;Verne;VERNE;25110;25;DOUBS;1
25605;;Vernierfontaine;VERNIERFONTAINE;25580;25;DOUBS;1
25607;;Vernois-ls-Belvoir;VERNOIS-LES-BELVOIR;25430;25;DOUBS;1
25608;Le;Vernoy;VERNOY;25750;25;DOUBS;1
25609;;Verrires-de-Joux;VERRIERES-DE-JOUX;25300;25;DOUBS;1
25610;;Verrires-du-Grosbois;VERRIERES-DU-GROSBOIS;25580;25;DOUBS;1
25611;La;Vze;VEZE;25660;25;DOUBS;1
25612;;Vieilley;VIEILLEY;25870;25;DOUBS;1
25613;;Vithorey;VIETHOREY;25340;25;DOUBS;1
25614;;Vieux-Charmont;VIEUX-CHARMONT;25600;25;DOUBS;1
25615;;Villars-ls-Blamont;VILLARS-LES-BLAMONT;25310;25;DOUBS;1
25616;;Villars-Saint-Georges;VILLARS-SAINT-GEORGES;25410;25;DOUBS;1
25617;;Villars-sous-Dampjoux;VILLARS-SOUS-DAMPJOUX;25190;25;DOUBS;1
25618;;Villars-sous-cot;VILLARS-SOUS-ECOT;25150;25;DOUBS;1
25619;Les;Villedieu;VILLEDIEU;25240;25;DOUBS;1
25620;;Ville-du-Pont;VILLE-DU-PONT;25650;25;DOUBS;1
25621;;Villeneuve-d'Amont;VILLENEUVE-D'AMONT;25270;25;DOUBS;1
25622;;Villers-Buzon;VILLERS-BUZON;25170;25;DOUBS;1
25623;;Villers-Chief;VILLERS-CHIEF;25530;25;DOUBS;1
25624;;Villers-Grlot;VILLERS-GRELOT;25640;25;DOUBS;1
25625;;Villers-la-Combe;VILLERS-LA-COMBE;25510;25;DOUBS;1
25626;;Villers-Saint-Martin;VILLERS-SAINT-MARTIN;25110;25;DOUBS;1
25627;;Villers-sous-Chalamont;VILLERS-SOUS-CHALAMONT;25270;25;DOUBS;1
25628;;Villers-sous-Montrond;VILLERS-SOUS-MONTROND;25620;25;DOUBS;1
25629;;Voillans;VOILLANS;25110;25;DOUBS;1
25630;;Voires;VOIRES;25580;25;DOUBS;1
25631;;Vorges-les-Pins;VORGES-LES-PINS;25320;25;DOUBS;1
25632;;Voujeaucourt;VOUJEAUCOURT;25420;25;DOUBS;1
25633;;Vuillafans;VUILLAFANS;25840;25;DOUBS;1
25634;;Vuillecin;VUILLECIN;25300;25;DOUBS;1
25635;;Vyt-ls-Belvoir;VYT-LES-BELVOIR;25430;25;DOUBS;1
26001;;Aix-en-Diois;AIX-EN-DIOIS;26150;26;DROME;1
26002;;Albon;ALBON;26140;26;DROME;1
26003;;Aleyrac;ALEYRAC;26770;26;DROME;1
26004;;Alixan;ALIXAN;26300;26;DROME;1
26005;;Allan;ALLAN;26780;26;DROME;1
26006;;Allex;ALLEX;26400;26;DROME;1
26007;;Ambonil;AMBONIL;26800;26;DROME;1
26008;;Ancne;ANCONE;26200;26;DROME;1
26009;;Andancette;ANDANCETTE;26140;26;DROME;1
26010;;Anneyron;ANNEYRON;26140;26;DROME;1
26011;;Aouste-sur-Sye;AOUSTE-SUR-SYE;26400;26;DROME;1
26012;;Arnayon;ARNAYON;26470;26;DROME;1
26013;;Arpavon;ARPAVON;26110;26;DROME;1
26014;;Arthmonay;ARTHEMONAY;26260;26;DROME;1
26015;;Aubenasson;AUBENASSON;26340;26;DROME;1
26016;;Aubres;AUBRES;26110;26;DROME;1
26017;;Aucelon;AUCELON;26340;26;DROME;1
26018;;Aulan;AULAN;26570;26;DROME;1
26019;;Aurel;AUREL;26340;26;DROME;1
26020;La;Rpara-Auriples;REPARA-AURIPLES;26400;26;DROME;1
26021;;Autichamp;AUTICHAMP;26400;26;DROME;1
26022;;Ballons;BALLONS;26560;26;DROME;1
26023;;Barbires;BARBIERES;26300;26;DROME;1
26024;;Barcelonne;BARCELONNE;26120;26;DROME;1
26025;;Barnave;BARNAVE;26310;26;DROME;1
26026;;Barret-de-Lioure;BARRET-DE-LIOURE;26570;26;DROME;1
26027;;Barsac;BARSAC;26150;26;DROME;1
26028;;Bathernay;BATHERNAY;26260;26;DROME;1
26030;La;Btie-des-Fonds;BATIE-DES-FONDS;26310;26;DROME;1
26031;La;Btie-Rolland;BATIE-ROLLAND;26160;26;DROME;1
26032;La;Baume-Cornillane;BAUME-CORNILLANE;26120;26;DROME;1
26033;La;Baume-de-Transit;BAUME-DE-TRANSIT;26790;26;DROME;1
26034;La;Baume-d'Hostun;BAUME-D'HOSTUN;26730;26;DROME;1
26035;;Beaufort-sur-Gervanne;BEAUFORT-SUR-GERVANNE;26400;26;DROME;1
26036;;Beaumont-en-Diois;BEAUMONT-EN-DIOIS;26310;26;DROME;1
26037;;Beaumont-ls-Valence;BEAUMONT-LES-VALENCE;26760;26;DROME;1
26038;;Beaumont-Monteux;BEAUMONT-MONTEUX;26600;26;DROME;1
26039;;Beauregard-Baret;BEAUREGARD-BARET;26300;26;DROME;1
26040;;Beaurires;BEAURIERES;26310;26;DROME;1
26041;;Beausemblant;BEAUSEMBLANT;26240;26;DROME;1
26042;;Beauvallon;BEAUVALLON;26800;26;DROME;1
26043;;Beauvoisin;BEAUVOISIN;26170;26;DROME;1
26045;La;Bgude-de-Mazenc;BEGUDE-DE-MAZENC;26160;26;DROME;1
26046;;Bellecombe-Tarendol;BELLECOMBE-TARENDOL;26110;26;DROME;1
26047;;Bellegarde-en-Diois;BELLEGARDE-EN-DIOIS;26470;26;DROME;1
26048;;Bnivay-Ollon;BENIVAY-OLLON;26170;26;DROME;1
26049;;Bsayes;BESAYES;26300;26;DROME;1
26050;;Bsignan;BESIGNAN;26110;26;DROME;1
26051;;Bzaudun-sur-Bne;BEZAUDUN-SUR-BINE;26460;26;DROME;1
26052;;Bonlieu-sur-Roubion;BONLIEU-SUR-ROUBION;26160;26;DROME;1
26054;;Bouchet;BOUCHET;26790;26;DROME;1
26055;;Boulc;BOULC;26410;26;DROME;1
26056;;Bourdeaux;BOURDEAUX;26460;26;DROME;1
26057;;Bourg-de-Page;BOURG-DE-PEAGE;26300;26;DROME;1
26058;;Bourg-ls-Valence;BOURG-LES-VALENCE;26500;26;DROME;1
26059;;Bouvante;BOUVANTE;26190;26;DROME;1
26060;;Bouvires;BOUVIERES;26460;26;DROME;1
26061;;Bren;BREN;26260;26;DROME;1
26062;;Brette;BRETTE;26340;26;DROME;1
26063;;Buis-les-Baronnies;BUIS-LES-BARONNIES;26170;26;DROME;1
26064;;Chabeuil;CHABEUIL;26120;26;DROME;1
26065;;Chabrillan;CHABRILLAN;26400;26;DROME;1
26066;Le;Chaffal;CHAFFAL;26190;26;DROME;1
26067;;Chalancon;CHALANCON;26340;26;DROME;1
26068;Le;Chalon;CHALON;26350;26;DROME;1
26069;;Chamaloc;CHAMALOC;26150;26;DROME;1
26070;;Chamaret;CHAMARET;26230;26;DROME;1
26071;;Chanos-Curson;CHANOS-CURSON;26600;26;DROME;1
26072;;Chantemerle-les-Bls;CHANTEMERLE-LES-BLES;26600;26;DROME;1
26073;;Chantemerle-ls-Grignan;CHANTEMERLE-LES-GRIGNAN;26230;26;DROME;1
26074;La;Chapelle-en-Vercors;CHAPELLE-EN-VERCORS;26420;26;DROME;1
26075;La;Charce;CHARCE;26470;26;DROME;1
26076;;Charens;CHARENS;26310;26;DROME;1
26077;;Charmes-sur-l'Herbasse;CHARMES-SUR-L'HERBASSE;26260;26;DROME;1
26078;;Charols;CHAROLS;26450;26;DROME;1
26079;;Charpey;CHARPEY;26300;26;DROME;1
26080;;Chastel-Arnaud;CHASTEL-ARNAUD;26340;26;DROME;1
26081;;Chteaudouble;CHATEAUDOUBLE;26120;26;DROME;1
26082;;Chteauneuf-de-Bordette;CHATEAUNEUF-DE-BORDETTE;26110;26;DROME;1
26083;;Chteauneuf-de-Galaure;CHATEAUNEUF-DE-GALAURE;26330;26;DROME;1
26084;;Chteauneuf-sur-Isre;CHATEAUNEUF-SUR-ISERE;26300;26;DROME;1
26085;;Chteauneuf-du-Rhne;CHATEAUNEUF-DU-RHONE;26780;26;DROME;1
26086;;Chtillon-en-Diois;CHATILLON-EN-DIOIS;26410;26;DROME;1
26087;;Chtillon-Saint-Jean;CHATILLON-SAINT-JEAN;26750;26;DROME;1
26088;;Chatuzange-le-Goubet;CHATUZANGE-LE-GOUBET;26300;26;DROME;1
26089;;Chaudebonne;CHAUDEBONNE;26110;26;DROME;1
26090;La;Chaudire;CHAUDIERE;26340;26;DROME;1
26091;;Chauvac-Laux-Montaux;CHAUVAC-LAUX-MONTAUX;26510;26;DROME;1
26092;;Chavannes;CHAVANNES;26260;26;DROME;1
26093;;Clansayes;CLANSAYES;26130;26;DROME;1
26094;;Claveyson;CLAVEYSON;26240;26;DROME;1
26095;;Clon-d'Andran;CLEON-D'ANDRAN;26450;26;DROME;1
26096;;Clrieux;CLERIEUX;26260;26;DROME;1
26097;;Cliousclat;CLIOUSCLAT;26270;26;DROME;1
26098;;Cobonne;COBONNE;26400;26;DROME;1
26099;;Colonzelle;COLONZELLE;26230;26;DROME;1
26100;;Combovin;COMBOVIN;26120;26;DROME;1
26101;;Comps;COMPS;26220;26;DROME;1
26102;;Condillac;CONDILLAC;26740;26;DROME;1
26103;;Condorcet;CONDORCET;26110;26;DROME;1
26104;;Cornillac;CORNILLAC;26510;26;DROME;1
26105;;Cornillon-sur-l'Oule;CORNILLON-SUR-L'OULE;26510;26;DROME;1
26106;La;Coucourde;COUCOURDE;26740;26;DROME;1
26107;;Crpol;CREPOL;26350;26;DROME;1
26108;;Crest;CREST;26400;26;DROME;1
26110;;Crozes-Hermitage;CROZES-HERMITAGE;26600;26;DROME;1
26111;;Crupies;CRUPIES;26460;26;DROME;1
26112;;Curnier;CURNIER;26110;26;DROME;1
26113;;Die;DIE;26150;26;DROME;1
26114;;Dieulefit;DIEULEFIT;26220;26;DROME;1
26115;;Divajeu;DIVAJEU;26400;26;DROME;1
26116;;Donzre;DONZERE;26290;26;DROME;1
26117;;chevis;ECHEVIS;26190;26;DROME;1
26118;;pinouze;EPINOUZE;26210;26;DROME;1
26119;;rme;EROME;26600;26;DROME;1
26121;;Espeluche;ESPELUCHE;26780;26;DROME;1
26122;;Espenel;ESPENEL;26340;26;DROME;1
26123;;Establet;ESTABLET;26470;26;DROME;1
26124;;toile-sur-Rhne;ETOILE-SUR-RHONE;26800;26;DROME;1
26125;;Eurre;EURRE;26400;26;DROME;1
26126;;Eygalayes;EYGALAYES;26560;26;DROME;1
26127;;Eygaliers;EYGALIERS;26170;26;DROME;1
26128;;Eygluy-Escoulin;EYGLUY-ESCOULIN;26400;26;DROME;1
26129;;Eymeux;EYMEUX;26730;26;DROME;1
26130;;Eyroles;EYROLES;26110;26;DROME;1
26131;;Eyzahut;EYZAHUT;26160;26;DROME;1
26133;;Fay-le-Clos;FAY-LE-CLOS;26240;26;DROME;1
26134;;Flines-sur-Rimandoule;FELINES-SUR-RIMANDOULE;26160;26;DROME;1
26135;;Ferrassires;FERRASSIERES;26570;26;DROME;1
26136;;Val-Maravel;VAL-MARAVEL;26310;26;DROME;1
26137;;Francillon-sur-Roubion;FRANCILLON-SUR-ROUBION;26400;26;DROME;1
26138;La;Garde-Adhmar;GARDE-ADHEMAR;26700;26;DROME;1
26139;;Gnissieux;GENISSIEUX;26750;26;DROME;1
26140;;Geyssans;GEYSSANS;26750;26;DROME;1
26141;;Gigors-et-Lozeron;GIGORS-ET-LOZERON;26400;26;DROME;1
26142;;Glandage;GLANDAGE;26410;26;DROME;1
26143;Le;Grand-Serre;GRAND-SERRE;26530;26;DROME;1
26144;;Grane;GRANE;26400;26;DROME;1
26145;Les;Granges-Gontardes;GRANGES-GONTARDES;26290;26;DROME;1
26146;;Grignan;GRIGNAN;26230;26;DROME;1
26147;;Gumiane;GUMIANE;26470;26;DROME;1
26148;;Hauterives;HAUTERIVES;26390;26;DROME;1
26149;;Hostun;HOSTUN;26730;26;DROME;1
26150;;Izon-la-Bruisse;IZON-LA-BRUISSE;26560;26;DROME;1
26152;;Jonchres;JONCHERES;26310;26;DROME;1
26153;;Laborel;LABOREL;26560;26;DROME;1
26154;;Lachau;LACHAU;26560;26;DROME;1
26155;;Lapeyrouse-Mornay;LAPEYROUSE-MORNAY;26210;26;DROME;1
26156;;Larnage;LARNAGE;26600;26;DROME;1
26157;La;Laupie;LAUPIE;26740;26;DROME;1
26159;;Laval-d'Aix;LAVAL-D'AIX;26150;26;DROME;1
26160;;Laveyron;LAVEYRON;26240;26;DROME;1
26161;;Lemps;LEMPS;26510;26;DROME;1
26162;;Lens-Lestang;LENS-LESTANG;26210;26;DROME;1
26163;;Loncel;LEONCEL;26190;26;DROME;1
26164;;Lesches-en-Diois;LESCHES-EN-DIOIS;26310;26;DROME;1
26165;;Livron-sur-Drme;LIVRON-SUR-DROME;26250;26;DROME;1
26166;;Loriol-sur-Drme;LORIOL-SUR-DROME;26270;26;DROME;1
26167;;Luc-en-Diois;LUC-EN-DIOIS;26310;26;DROME;1
26168;;Lus-la-Croix-Haute;LUS-LA-CROIX-HAUTE;26620;26;DROME;1
26169;;Malataverne;MALATAVERNE;26780;26;DROME;1
26170;;Malissard;MALISSARD;26120;26;DROME;1
26171;;Manas;MANAS;26160;26;DROME;1
26172;;Manthes;MANTHES;26210;26;DROME;1
26173;;Marches;MARCHES;26300;26;DROME;1
26174;;Margs;MARGES;26260;26;DROME;1
26175;;Marignac-en-Diois;MARIGNAC-EN-DIOIS;26150;26;DROME;1
26176;;Marsanne;MARSANNE;26740;26;DROME;1
26177;;Marsaz;MARSAZ;26260;26;DROME;1
26178;;Menglon;MENGLON;26410;26;DROME;1
26179;;Mercurol;MERCUROL;26600;26;DROME;1
26180;;Mrindol-les-Oliviers;MERINDOL-LES-OLIVIERS;26170;26;DROME;1
26181;;Mvouillon;MEVOUILLON;26560;26;DROME;1
26182;;Mirabel-aux-Baronnies;MIRABEL-AUX-BARONNIES;26110;26;DROME;1
26183;;Mirabel-et-Blacons;MIRABEL-ET-BLACONS;26400;26;DROME;1
26184;;Miribel;MIRIBEL;26350;26;DROME;1
26185;;Mirmande;MIRMANDE;26270;26;DROME;1
26186;;Miscon;MISCON;26310;26;DROME;1
26187;;Molires-Glandaz;MOLIERES-GLANDAZ;26150;26;DROME;1
26188;;Mollans-sur-Ouvze;MOLLANS-SUR-OUVEZE;26170;26;DROME;1
26189;;Montauban-sur-l'Ouvze;MONTAUBAN-SUR-L'OUVEZE;26170;26;DROME;1
26190;;Montaulieu;MONTAULIEU;26110;26;DROME;1
26191;;Montboucher-sur-Jabron;MONTBOUCHER-SUR-JABRON;26740;26;DROME;1
26192;;Montbrison-sur-Lez;MONTBRISON-SUR-LEZ;26770;26;DROME;1
26193;;Montbrun-les-Bains;MONTBRUN-LES-BAINS;26570;26;DROME;1
26194;;Montchenu;MONTCHENU;26350;26;DROME;1
26195;;Montclar-sur-Gervanne;MONTCLAR-SUR-GERVANNE;26400;26;DROME;1
26196;;Montlger;MONTELEGER;26760;26;DROME;1
26197;;Montlier;MONTELIER;26120;26;DROME;1
26198;;Montlimar;MONTELIMAR;26200;26;DROME;1
26199;;Montferrand-la-Fare;MONTFERRAND-LA-FARE;26510;26;DROME;1
26200;;Montfroc;MONTFROC;26560;26;DROME;1
26201;;Montguers;MONTGUERS;26170;26;DROME;1
26202;;Montjoux;MONTJOUX;26220;26;DROME;1
26203;;Montjoyer;MONTJOYER;26230;26;DROME;1
26204;;Montlaur-en-Diois;MONTLAUR-EN-DIOIS;26310;26;DROME;1
26205;;Montmaur-en-Diois;MONTMAUR-EN-DIOIS;26150;26;DROME;1
26206;;Montmeyran;MONTMEYRAN;26120;26;DROME;1
26207;;Montmiral;MONTMIRAL;26750;26;DROME;1
26208;;Montoison;MONTOISON;26800;26;DROME;1
26209;;Montral-les-Sources;MONTREAL-LES-SOURCES;26510;26;DROME;1
26210;;Montrigaud;MONTRIGAUD;26350;26;DROME;1
26211;;Montsgur-sur-Lauzon;MONTSEGUR-SUR-LAUZON;26130;26;DROME;1
26212;;Montvendre;MONTVENDRE;26120;26;DROME;1
26213;;Moras-en-Valloire;MORAS-EN-VALLOIRE;26210;26;DROME;1
26214;;Mornans;MORNANS;26460;26;DROME;1
26215;La;Motte-Chalancon;MOTTE-CHALANCON;26470;26;DROME;1
26216;La;Motte-de-Galaure;MOTTE-DE-GALAURE;26240;26;DROME;1
26217;La;Motte-Fanjas;MOTTE-FANJAS;26190;26;DROME;1
26218;;Mours-Saint-Eusbe;MOURS-SAINT-EUSEBE;26540;26;DROME;1
26219;;Mureils;MUREILS;26240;26;DROME;1
26220;;Nyons;NYONS;26110;26;DROME;1
26221;;Omblze;OMBLEZE;26400;26;DROME;1
26222;;Orcinas;ORCINAS;26220;26;DROME;1
26223;;Oriol-en-Royans;ORIOL-EN-ROYANS;26190;26;DROME;1
26224;;Ourches;OURCHES;26120;26;DROME;1
26225;;Parnans;PARNANS;26750;26;DROME;1
26226;Le;Pgue;PEGUE;26770;26;DROME;1
26227;;Pelonne;PELONNE;26510;26;DROME;1
26228;;Pennes-le-Sec;PENNES-LE-SEC;26340;26;DROME;1
26229;La;Penne-sur-l'Ouvze;PENNE-SUR-L'OUVEZE;26170;26;DROME;1
26231;;Peyrins;PEYRINS;26380;26;DROME;1
26232;;Peyrus;PEYRUS;26120;26;DROME;1
26233;;Pigon;PIEGON;26110;26;DROME;1
26234;;Pigros-la-Clastre;PIEGROS-LA-CLASTRE;26400;26;DROME;1
26235;;Pierrelatte;PIERRELATTE;26700;26;DROME;1
26236;;Pierrelongue;PIERRELONGUE;26170;26;DROME;1
26238;Les;Pilles;PILLES;26110;26;DROME;1
26239;;Plaisians;PLAISIANS;26170;26;DROME;1
26240;;Plan-de-Baix;PLAN-DE-BAIX;26400;26;DROME;1
26241;Le;Pot-Clard;POET-CELARD;26460;26;DROME;1
26242;Le;Pot-en-Percip;POET-EN-PERCIP;26170;26;DROME;1
26243;Le;Pot-Laval;POET-LAVAL;26160;26;DROME;1
26244;Le;Pot-Sigillat;POET-SIGILLAT;26110;26;DROME;1
26245;;Pommerol;POMMEROL;26470;26;DROME;1
26246;;Ponet-et-Saint-Auban;PONET-ET-SAINT-AUBAN;26150;26;DROME;1
26247;;Ponsas;PONSAS;26240;26;DROME;1
26248;;Pontaix;PONTAIX;26150;26;DROME;1
26249;;Pont-de-Barret;PONT-DE-BARRET;26160;26;DROME;1
26250;;Pont-de-l'Isre;PONT-DE-L'ISERE;26600;26;DROME;1
26251;;Portes-en-Valdaine;PORTES-EN-VALDAINE;26160;26;DROME;1
26252;;Portes-ls-Valence;PORTES-LES-VALENCE;26800;26;DROME;1
26253;;Poyols;POYOLS;26310;26;DROME;1
26254;;Pradelle;PRADELLE;26340;26;DROME;1
26255;Les;Prs;PRES;26310;26;DROME;1
26256;;Propiac;PROPIAC;26170;26;DROME;1
26257;;Puygiron;PUYGIRON;26160;26;DROME;1
26258;;Puy-Saint-Martin;PUY-SAINT-MARTIN;26450;26;DROME;1
26259;;Ratires;RATIERES;26330;26;DROME;1
26261;;Rauville;REAUVILLE;26230;26;DROME;1
26262;;Recoubeau-Jansac;RECOUBEAU-JANSAC;26310;26;DROME;1
26263;;Reilhanette;REILHANETTE;26570;26;DROME;1
26264;;Rmuzat;REMUZAT;26510;26;DROME;1
26266;;Rimon-et-Savel;RIMON-ET-SAVEL;26340;26;DROME;1
26267;;Rioms;RIOMS;26170;26;DROME;1
26268;;Rochebaudin;ROCHEBAUDIN;26160;26;DROME;1
26269;;Rochebrune;ROCHEBRUNE;26110;26;DROME;1
26270;;Rochechinard;ROCHECHINARD;26190;26;DROME;1
26271;La;Roche-de-Glun;ROCHE-DE-GLUN;26600;26;DROME;1
26272;;Rochefort-en-Valdaine;ROCHEFORT-EN-VALDAINE;26160;26;DROME;1
26273;;Rochefort-Samson;ROCHEFORT-SAMSON;26300;26;DROME;1
26274;;Rochefourchat;ROCHEFOURCHAT;26340;26;DROME;1
26275;;Rochegude;ROCHEGUDE;26790;26;DROME;1
26276;;Roche-Saint-Secret-Bconne;ROCHE-SAINT-SECRET-BECONNE;26770;26;DROME;1
26277;La;Roche-sur-Grane;ROCHE-SUR-GRANE;26400;26;DROME;1
26278;La;Roche-sur-le-Buis;ROCHE-SUR-LE-BUIS;26170;26;DROME;1
26279;La;Rochette-du-Buis;ROCHETTE-DU-BUIS;26170;26;DROME;1
26281;;Romans-sur-Isre;ROMANS-SUR-ISERE;26100;26;DROME;1
26282;;Romeyer;ROMEYER;26150;26;DROME;1
26283;;Rottier;ROTTIER;26470;26;DROME;1
26284;;Roussas;ROUSSAS;26230;26;DROME;1
26285;;Rousset-les-Vignes;ROUSSET-LES-VIGNES;26770;26;DROME;1
26286;;Roussieux;ROUSSIEUX;26510;26;DROME;1
26287;;Roynac;ROYNAC;26450;26;DROME;1
26288;;Sahune;SAHUNE;26510;26;DROME;1
26289;;Saillans;SAILLANS;26340;26;DROME;1
26290;;Saint-Agnan-en-Vercors;SAINT-AGNAN-EN-VERCORS;26420;26;DROME;1
26291;;Saint-Andol;SAINT-ANDEOL;26150;26;DROME;1
26292;;Saint-Auban-sur-l'Ouvze;SAINT-AUBAN-SUR-L'OUVEZE;26170;26;DROME;1
26293;;Saint-Avit;SAINT-AVIT;26330;26;DROME;1
26294;;Saint-Bardoux;SAINT-BARDOUX;26260;26;DROME;1
26295;;Saint-Barthlemy-de-Vals;SAINT-BARTHELEMY-DE-VALS;26240;26;DROME;1
26296;;Saint-Benoit-en-Diois;SAINT-BENOIT-EN-DIOIS;26340;26;DROME;1
26297;;Saint-Bonnet-de-Valclrieux;SAINT-BONNET-DE-VALCLERIEUX;26350;26;DROME;1
26298;;Saint-Christophe-et-le-Laris;SAINT-CHRISTOPHE-ET-LE-LARIS;26350;26;DROME;1
26299;;Sainte-Croix;SAINTE-CROIX;26150;26;DROME;1
26300;;Saint-Dizier-en-Diois;SAINT-DIZIER-EN-DIOIS;26310;26;DROME;1
26301;;Saint-Donat-sur-l'Herbasse;SAINT-DONAT-SUR-L'HERBASSE;26260;26;DROME;1
26302;;Sainte-Eulalie-en-Royans;SAINTE-EULALIE-EN-ROYANS;26190;26;DROME;1
26303;;Sainte-Euphmie-sur-Ouvze;SAINTE-EUPHEMIE-SUR-OUVEZE;26170;26;DROME;1
26304;;Saint-Ferrol-Trente-Pas;SAINT-FERREOL-TRENTE-PAS;26110;26;DROME;1
26305;;Saint-Gervais-sur-Roubion;SAINT-GERVAIS-SUR-ROUBION;26160;26;DROME;1
26306;;Sainte-Jalle;SAINTE-JALLE;26110;26;DROME;1
26307;;Saint-Jean-en-Royans;SAINT-JEAN-EN-ROYANS;26190;26;DROME;1
26308;;Saint-Julien-en-Quint;SAINT-JULIEN-EN-QUINT;26150;26;DROME;1
26309;;Saint-Julien-en-Vercors;SAINT-JULIEN-EN-VERCORS;26420;26;DROME;1
26310;;Saint-Laurent-d'Onay;SAINT-LAURENT-D'ONAY;26350;26;DROME;1
26311;;Saint-Laurent-en-Royans;SAINT-LAURENT-EN-ROYANS;26190;26;DROME;1
26312;;Saint-Marcel-ls-Sauzet;SAINT-MARCEL-LES-SAUZET;26740;26;DROME;1
26313;;Saint-Marcel-ls-Valence;SAINT-MARCEL-LES-VALENCE;26320;26;DROME;1
26314;;Saint-Martin-d'Aot;SAINT-MARTIN-D'AOUT;26330;26;DROME;1
26315;;Saint-Martin-en-Vercors;SAINT-MARTIN-EN-VERCORS;26420;26;DROME;1
26316;;Saint-Martin-le-Colonel;SAINT-MARTIN-LE-COLONEL;26190;26;DROME;1
26317;;Saint-Maurice-sur-Eygues;SAINT-MAURICE-SUR-EYGUES;26110;26;DROME;1
26318;;Saint-May;SAINT-MAY;26510;26;DROME;1
26319;;Saint-Michel-sur-Savasse;SAINT-MICHEL-SUR-SAVASSE;26750;26;DROME;1
26320;;Saint-Nazaire-en-Royans;SAINT-NAZAIRE-EN-ROYANS;26190;26;DROME;1
26321;;Saint-Nazaire-le-Dsert;SAINT-NAZAIRE-LE-DESERT;26340;26;DROME;1
26322;;Saint-Pantalon-les-Vignes;SAINT-PANTALEON-LES-VIGNES;26770;26;DROME;1
26323;;Saint-Paul-ls-Romans;SAINT-PAUL-LES-ROMANS;26750;26;DROME;1
26324;;Saint-Paul-Trois-Chteaux;SAINT-PAUL-TROIS-CHATEAUX;26130;26;DROME;1
26325;;Saint-Rambert-d'Albon;SAINT-RAMBERT-D'ALBON;26140;26;DROME;1
26326;;Saint-Restitut;SAINT-RESTITUT;26130;26;DROME;1
26327;;Saint-Roman;SAINT-ROMAN;26410;26;DROME;1
26328;;Saint-Sauveur-en-Diois;SAINT-SAUVEUR-EN-DIOIS;26340;26;DROME;1
26329;;Saint-Sauveur-Gouvernet;SAINT-SAUVEUR-GOUVERNET;26110;26;DROME;1
26330;;Saint-Sorlin-en-Valloire;SAINT-SORLIN-EN-VALLOIRE;26210;26;DROME;1
26331;;Saint-Thomas-en-Royans;SAINT-THOMAS-EN-ROYANS;26190;26;DROME;1
26332;;Saint-Uze;SAINT-UZE;26240;26;DROME;1
26333;;Saint-Vallier;SAINT-VALLIER;26240;26;DROME;1
26334;;Salettes;SALETTES;26160;26;DROME;1
26335;;Salles-sous-Bois;SALLES-SOUS-BOIS;26770;26;DROME;1
26336;;Saou;SAOU;26400;26;DROME;1
26337;;Saulce-sur-Rhne;SAULCE-SUR-RHONE;26270;26;DROME;1
26338;;Sauzet;SAUZET;26740;26;DROME;1
26339;;Savasse;SAVASSE;26740;26;DROME;1
26340;;Sderon;SEDERON;26560;26;DROME;1
26341;;Serves-sur-Rhne;SERVES-SUR-RHONE;26600;26;DROME;1
26342;;Solrieux;SOLERIEUX;26130;26;DROME;1
26343;;Souspierre;SOUSPIERRE;26160;26;DROME;1
26344;;Soyans;SOYANS;26400;26;DROME;1
26345;;Suze-la-Rousse;SUZE-LA-ROUSSE;26790;26;DROME;1
26346;;Suze;SUZE;26400;26;DROME;1
26347;;Tain-l'Hermitage;TAIN-L'HERMITAGE;26600;26;DROME;1
26348;;Taulignan;TAULIGNAN;26770;26;DROME;1
26349;;Tersanne;TERSANNE;26390;26;DROME;1
26350;;Teyssires;TEYSSIERES;26220;26;DROME;1
26351;Les;Tonils;TONILS;26460;26;DROME;1
26352;La;Touche;TOUCHE;26160;26;DROME;1
26353;Les;Tourrettes;TOURRETTES;26740;26;DROME;1
26354;;Treschenu-Creyers;TRESCHENU-CREYERS;26410;26;DROME;1
26355;;Triors;TRIORS;26750;26;DROME;1
26356;;Truinas;TRUINAS;26460;26;DROME;1
26357;;Tulette;TULETTE;26790;26;DROME;1
26358;;Upie;UPIE;26120;26;DROME;1
26359;;Vachres-en-Quint;VACHERES-EN-QUINT;26150;26;DROME;1
26360;;Valaurie;VALAURIE;26230;26;DROME;1
26361;;Valdrme;VALDROME;26310;26;DROME;1
26362;;Valence;VALENCE;26000;26;DROME;1
26363;;Valouse;VALOUSE;26110;26;DROME;1
26364;;Vassieux-en-Vercors;VASSIEUX-EN-VERCORS;26420;26;DROME;1
26365;;Vaunaveys-la-Rochette;VAUNAVEYS-LA-ROCHETTE;26400;26;DROME;1
26366;;Veaunes;VEAUNES;26600;26;DROME;1
26367;;Venterol;VENTEROL;26110;26;DROME;1
26368;;Vercheny;VERCHENY;26340;26;DROME;1
26369;;Verclause;VERCLAUSE;26510;26;DROME;1
26370;;Vercoiran;VERCOIRAN;26170;26;DROME;1
26371;;Vronne;VERONNE;26340;26;DROME;1
26372;;Vers-sur-Mouge;VERS-SUR-MEOUGE;26560;26;DROME;1
26373;;Vesc;VESC;26220;26;DROME;1
26374;;Villebois-les-Pins;VILLEBOIS-LES-PINS;05700;26;HAUTES ALPES;1
26375;;Villefranche-le-Chteau;VILLEFRANCHE-LE-CHATEAU;26560;26;DROME;1
26376;;Villeperdrix;VILLEPERDRIX;26510;26;DROME;1
26377;;Vinsobres;VINSOBRES;26110;26;DROME;1
26378;;Volvent;VOLVENT;26470;26;DROME;1
26379;;Granges-les-Beaumont;GRANGES-LES-BEAUMONT;26600;26;DROME;1
26380;;Gervans;GERVANS;26600;26;DROME;1
26381;;Jaillans;JAILLANS;26300;26;DROME;1
26382;;Saint-Vincent-la-Commanderie;SAINT-VINCENT-LA-COMMANDERIE;26300;26;DROME;1
27001;;Aclou;ACLOU;27800;27;EURE;1
27002;;Acon;ACON;27570;27;EURE;1
27003;;Acquigny;ACQUIGNY;27400;27;EURE;1
27004;;Aigleville;AIGLEVILLE;27120;27;EURE;1
27005;;Ailly;AILLY;27600;27;EURE;1
27006;;Aizier;AIZIER;27500;27;EURE;1
27007;;Ajou;AJOU;27410;27;EURE;1
27008;;Alizay;ALIZAY;27460;27;EURE;1
27009;;Ambenay;AMBENAY;27250;27;EURE;1
27010;;Amcourt;AMECOURT;27140;27;EURE;1
27011;;Amfreville-la-Campagne;AMFREVILLE-LA-CAMPAGNE;27370;27;EURE;1
27012;;Amfreville-les-Champs;AMFREVILLE-LES-CHAMPS;27380;27;EURE;1
27013;;Amfreville-sous-les-Monts;AMFREVILLE-SOUS-LES-MONTS;27380;27;EURE;1
27014;;Amfreville-sur-Iton;AMFREVILLE-SUR-ITON;27400;27;EURE;1
27015;;And;ANDE;27430;27;EURE;1
27016;Les;Andelys;ANDELYS;27700;27;EURE;1
27017;;Angerville-la-Campagne;ANGERVILLE-LA-CAMPAGNE;27930;27;EURE;1
27018;;Appeville-Annebault;APPEVILLE-ANNEBAULT;27290;27;EURE;1
27019;;Armentires-sur-Avre;ARMENTIERES-SUR-AVRE;27820;27;EURE;1
27020;;Arnires-sur-Iton;ARNIERES-SUR-ITON;27180;27;EURE;1
27021;;Asnires;ASNIERES;27260;27;EURE;1
27022;;Aubevoye;AUBEVOYE;27940;27;EURE;1
27023;;Aulnay-sur-Iton;AULNAY-SUR-ITON;27180;27;EURE;1
27024;Le;Roncenay-Authenay;RONCENAY-AUTHENAY;27240;27;EURE;1
27025;;Autheuil-Authouillet;AUTHEUIL-AUTHOUILLET;27490;27;EURE;1
27026;;Authevernes;AUTHEVERNES;27420;27;EURE;1
27027;Les;Authieux;AUTHIEUX;27220;27;EURE;1
27028;;Authou;AUTHOU;27290;27;EURE;1
27031;;Aviron;AVIRON;27930;27;EURE;1
27032;;Avrilly;AVRILLY;27240;27;EURE;1
27033;;Bacquepuis;BACQUEPUIS;27930;27;EURE;1
27034;;Bacqueville;BACQUEVILLE;27440;27;EURE;1
27035;;Bailleul-la-Valle;BAILLEUL-LA-VALLEE;27260;27;EURE;1
27036;;Blines;BALINES;27130;27;EURE;1
27037;;Barc;BARC;27170;27;EURE;1
27038;Les;Barils;BARILS;27130;27;EURE;1
27039;;Barneville-sur-Seine;BARNEVILLE-SUR-SEINE;27310;27;EURE;1
27040;;Barquet;BARQUET;27170;27;EURE;1
27041;La;Barre-en-Ouche;BARRE-EN-OUCHE;27330;27;EURE;1
27042;;Barville;BARVILLE;27230;27;EURE;1
27043;Les;Baux-de-Breteuil;BAUX-DE-BRETEUIL;27160;27;EURE;1
27044;Les;Baux-Sainte-Croix;BAUX-SAINTE-CROIX;27180;27;EURE;1
27045;;Bazincourt-sur-Epte;BAZINCOURT-SUR-EPTE;27140;27;EURE;1
27046;;Bazoques;BAZOQUES;27230;27;EURE;1
27047;;Beaubray;BEAUBRAY;27190;27;EURE;1
27048;;Beauficel-en-Lyons;BEAUFICEL-EN-LYONS;27480;27;EURE;1
27049;;Beaumesnil;BEAUMESNIL;27410;27;EURE;1
27050;;Beaumontel;BEAUMONTEL;27170;27;EURE;1
27051;;Beaumont-le-Roger;BEAUMONT-LE-ROGER;27170;27;EURE;1
27052;Le;Bec-Hellouin;BEC-HELLOUIN;27800;27;EURE;1
27053;Le;Bec-Thomas;BEC-THOMAS;27370;27;EURE;1
27054;;Bmcourt;BEMECOURT;27160;27;EURE;1
27055;;Brengeville-la-Campagne;BERENGEVILLE-LA-CAMPAGNE;27110;27;EURE;1
27056;;Bernay;BERNAY;27300;27;EURE;1
27057;;Bernienville;BERNIENVILLE;27180;27;EURE;1
27058;;Bernires-sur-Seine;BERNIERES-SUR-SEINE;27700;27;EURE;1
27059;;Bernouville;BERNOUVILLE;27660;27;EURE;1
27060;;Berthenonville;BERTHENONVILLE;27630;27;EURE;1
27061;;Berthouville;BERTHOUVILLE;27800;27;EURE;1
27062;;Berville-en-Roumois;BERVILLE-EN-ROUMOIS;27520;27;EURE;1
27063;;Berville-la-Campagne;BERVILLE-LA-CAMPAGNE;27170;27;EURE;1
27064;;Berville-sur-Mer;BERVILLE-SUR-MER;27210;27;EURE;1
27065;;Beuzeville;BEUZEVILLE;27210;27;EURE;1
27066;;Bzu-la-Fort;BEZU-LA-FORET;27480;27;EURE;1
27067;;Bzu-Saint-loi;BEZU-SAINT-ELOI;27660;27;EURE;1
27068;;Bois-Anzeray;BOIS-ANZERAY;27330;27;EURE;1
27069;;Bois-Arnault;BOIS-ARNAULT;27250;27;EURE;1
27070;;Boisemont;BOISEMONT;27150;27;EURE;1
27071;Le;Bois-Hellain;BOIS-HELLAIN;27260;27;EURE;1
27072;;Bois-Jrme-Saint-Ouen;BOIS-JEROME-SAINT-OUEN;27620;27;EURE;1
27073;;Bois-le-Roi;BOIS-LE-ROI;27220;27;EURE;1
27074;;Boisney;BOISNEY;27800;27;EURE;1
27075;;Bois-Normand-prs-Lyre;BOIS-NORMAND-PRES-LYRE;27330;27;EURE;1
27076;;Boisset-les-Prvanches;BOISSET-LES-PREVANCHES;27120;27;EURE;1
27077;;Boissey-le-Chtel;BOISSEY-LE-CHATEL;27520;27;EURE;1
27078;La;Boissire;BOISSIERE;27220;27;EURE;1
27079;;Boissy-Lamberville;BOISSY-LAMBERVILLE;27300;27;EURE;1
27081;;Boncourt;BONCOURT;27120;27;EURE;1
27082;La;Bonneville-sur-Iton;BONNEVILLE-SUR-ITON;27190;27;EURE;1
27083;;Bonneville-Aptot;BONNEVILLE-APTOT;27290;27;EURE;1
27084;;Bosc-Bnard-Commin;BOSC-BENARD-COMMIN;27520;27;EURE;1
27085;;Bosc-Bnard-Crescy;BOSC-BENARD-CRESCY;27310;27;EURE;1
27088;;Bosc-Renoult-en-Ouche;BOSC-RENOULT-EN-OUCHE;27330;27;EURE;1
27089;;Bosc-Renoult-en-Roumois;BOSC-RENOULT-EN-ROUMOIS;27520;27;EURE;1
27090;Le;Bosc-Roger-en-Roumois;BOSC-ROGER-EN-ROUMOIS;27670;27;EURE;1
27091;;Bosgouet;BOSGOUET;27310;27;EURE;1
27092;;Bosgurard-de-Marcouville;BOSGUERARD-DE-MARCOUVILLE;27520;27;EURE;1
27093;;Bosnormand;BOSNORMAND;27670;27;EURE;1
27094;;Bosquentin;BOSQUENTIN;27480;27;EURE;1
27095;;Bosrobert;BOSROBERT;27800;27;EURE;1
27096;Les;Bottereaux;BOTTEREAUX;27250;27;EURE;1
27097;;Bouafles;BOUAFLES;27700;27;EURE;1
27098;;Bouchevilliers;BOUCHEVILLIERS;27150;27;EURE;1
27099;Le;Boulay-Morin;BOULAY-MORIN;27930;27;EURE;1
27100;;Boulleville;BOULLEVILLE;27210;27;EURE;1
27101;;Bouquelon;BOUQUELON;27500;27;EURE;1
27102;;Bouquetot;BOUQUETOT;27310;27;EURE;1
27103;;Bourg-Achard;BOURG-ACHARD;27310;27;EURE;1
27104;;Bourg-Beaudouin;BOURG-BEAUDOUIN;27380;27;EURE;1
27105;;Bourgtheroulde-Infreville;BOURGTHEROULDE-INFREVILLE;27520;27;EURE;1
27106;;Bournainville-Faverolles;BOURNAINVILLE-FAVEROLLES;27230;27;EURE;1
27107;;Bourneville;BOURNEVILLE;27500;27;EURE;1
27108;;Bourth;BOURTH;27580;27;EURE;1
27109;;Bray;BRAY;27170;27;EURE;1
27110;;Brestot;BRESTOT;27350;27;EURE;1
27111;;Bretagnolles;BRETAGNOLLES;27220;27;EURE;1
27112;;Breteuil;BRETEUIL;27160;27;EURE;1
27113;;Brtigny;BRETIGNY;27800;27;EURE;1
27114;;Breuilpont;BREUILPONT;27640;27;EURE;1
27115;;Breux-sur-Avre;BREUX-SUR-AVRE;27570;27;EURE;1
27116;;Brionne;BRIONNE;27800;27;EURE;1
27117;;Broglie;BROGLIE;27270;27;EURE;1
27118;;Brosville;BROSVILLE;27930;27;EURE;1
27119;;Bueil;BUEIL;27730;27;EURE;1
27120;;Burey;BUREY;27190;27;EURE;1
27121;;Bus-Saint-Rmy;BUS-SAINT-REMY;27630;27;EURE;1
27122;;Cahaignes;CAHAIGNES;27420;27;EURE;1
27123;;Caillouet-Orgeville;CAILLOUET-ORGEVILLE;27120;27;EURE;1
27124;;Cailly-sur-Eure;CAILLY-SUR-EURE;27490;27;EURE;1
27125;;Calleville;CALLEVILLE;27800;27;EURE;1
27126;;Campigny;CAMPIGNY;27500;27;EURE;1
27127;;Canappeville;CANAPPEVILLE;27400;27;EURE;1
27128;;Cantiers;CANTIERS;27420;27;EURE;1
27129;;Caorches-Saint-Nicolas;CAORCHES-SAINT-NICOLAS;27300;27;EURE;1
27130;;Capelle-les-Grands;CAPELLE-LES-GRANDS;27270;27;EURE;1
27131;;Carsix;CARSIX;27300;27;EURE;1
27132;;Caug;CAUGE;27180;27;EURE;1
27133;;Caumont;CAUMONT;27310;27;EURE;1
27134;;Cauverville-en-Roumois;CAUVERVILLE-EN-ROUMOIS;27350;27;EURE;1
27135;;Cesseville;CESSEVILLE;27110;27;EURE;1
27136;;Chaignes;CHAIGNES;27120;27;EURE;1
27137;;Chaise-Dieu-du-Theil;CHAISE-DIEU-DU-THEIL;27580;27;EURE;1
27138;;Chamblac;CHAMBLAC;27270;27;EURE;1
27139;;Chambord;CHAMBORD;27250;27;EURE;1
27140;;Chambray;CHAMBRAY;27120;27;EURE;1
27141;;Champ-Dolent;CHAMP-DOLENT;27190;27;EURE;1
27142;;Champenard;CHAMPENARD;27600;27;EURE;1
27143;;Champignolles;CHAMPIGNOLLES;27330;27;EURE;1
27144;;Champigny-la-Futelaye;CHAMPIGNY-LA-FUTELAYE;27220;27;EURE;1
27145;;Chanteloup;CHANTELOUP;27240;27;EURE;1
27146;La;Chapelle-Bayvel;CHAPELLE-BAYVEL;27260;27;EURE;1
27147;La;Chapelle-du-Bois-des-Faulx;CHAPELLE-DU-BOIS-DES-FAULX;27930;27;EURE;1
27148;La;Chapelle-Gauthier;CHAPELLE-GAUTHIER;27270;27;EURE;1
27149;La;Chapelle-Hareng;CHAPELLE-HARENG;27230;27;EURE;1
27150;La;Chapelle-Ranville;CHAPELLE-REANVILLE;27950;27;EURE;1
27151;;Charleval;CHARLEVAL;27380;27;EURE;1
27152;;Chteau-sur-Epte;CHATEAU-SUR-EPTE;27420;27;EURE;1
27153;;Chauvincourt-Provemont;CHAUVINCOURT-PROVEMONT;27150;27;EURE;1
27154;;Chavigny-Bailleul;CHAVIGNY-BAILLEUL;27220;27;EURE;1
27155;;Chennebrun;CHENNEBRUN;27820;27;EURE;1
27156;;Chronvilliers;CHERONVILLIERS;27250;27;EURE;1
27157;Le;Chesne;CHESNE;27160;27;EURE;1
27158;;Cierrey;CIERREY;27930;27;EURE;1
27159;;Cintray;CINTRAY;27160;27;EURE;1
27160;;Civires;CIVIERES;27630;27;EURE;1
27161;;Claville;CLAVILLE;27180;27;EURE;1
27162;;Collandres-Quincarnon;COLLANDRES-QUINCARNON;27190;27;EURE;1
27163;;Colletot;COLLETOT;27500;27;EURE;1
27164;;Combon;COMBON;27170;27;EURE;1
27165;;Conches-en-Ouche;CONCHES-EN-OUCHE;27190;27;EURE;1
27166;;Cond-sur-Iton;CONDE-SUR-ITON;27160;27;EURE;1
27167;;Cond-sur-Risle;CONDE-SUR-RISLE;27290;27;EURE;1
27168;;Connelles;CONNELLES;27430;27;EURE;1
27169;;Conteville;CONTEVILLE;27210;27;EURE;1
27170;;Cormeilles;CORMEILLES;27260;27;EURE;1
27171;Le;Cormier;CORMIER;27120;27;EURE;1
27172;;Corneuil;CORNEUIL;27240;27;EURE;1
27173;;Corneville-la-Fouquetire;CORNEVILLE-LA-FOUQUETIERE;27300;27;EURE;1
27174;;Corneville-sur-Risle;CORNEVILLE-SUR-RISLE;27500;27;EURE;1
27175;;Corny;CORNY;27700;27;EURE;1
27176;;Coudray;COUDRAY;27150;27;EURE;1
27177;;Coudres;COUDRES;27220;27;EURE;1
27179;;Courbpine;COURBEPINE;27300;27;EURE;1
27180;;Courcelles-sur-Seine;COURCELLES-SUR-SEINE;27940;27;EURE;1
27181;;Courdemanche;COURDEMANCHE;27320;27;EURE;1
27182;;Courteilles;COURTEILLES;27130;27;EURE;1
27183;La;Couture-Boussey;COUTURE-BOUSSEY;27750;27;EURE;1
27184;;Crasville;CRASVILLE;27400;27;EURE;1
27185;;Crestot;CRESTOT;27110;27;EURE;1
27187;;Criquebeuf-la-Campagne;CRIQUEBEUF-LA-CAMPAGNE;27110;27;EURE;1
27188;;Criquebeuf-sur-Seine;CRIQUEBEUF-SUR-SEINE;27340;27;EURE;1
27189;La;Croisille;CROISILLE;27190;27;EURE;1
27190;;Croisy-sur-Eure;CROISY-SUR-EURE;27120;27;EURE;1
27191;La;Croix-Saint-Leufroy;CROIX-SAINT-LEUFROY;27490;27;EURE;1
27192;;Crosville-la-Vieille;CROSVILLE-LA-VIEILLE;27110;27;EURE;1
27193;;Croth;CROTH;28520;27;EURE ET LOIR;1
27194;;Cuverville;CUVERVILLE;27700;27;EURE;1
27195;;Dame-Marie;DAME-MARIE;27160;27;EURE;1
27196;Les;Damps;DAMPS;27340;27;EURE;1
27197;;Dampsmesnil;DAMPSMESNIL;27630;27;EURE;1
27198;;Damville;DAMVILLE;27240;27;EURE;1
27199;;Dangu;DANGU;27720;27;EURE;1
27200;;Dardez;DARDEZ;27930;27;EURE;1
27201;;Daubeuf-la-Campagne;DAUBEUF-LA-CAMPAGNE;27110;27;EURE;1
27202;;Daubeuf-prs-Vatteville;DAUBEUF-PRES-VATTEVILLE;27430;27;EURE;1
27203;;Douains;DOUAINS;27120;27;EURE;1
27204;;Doudeauville-en-Vexin;DOUDEAUVILLE-EN-VEXIN;27150;27;EURE;1
27205;;Douville-sur-Andelle;DOUVILLE-SUR-ANDELLE;27380;27;EURE;1
27206;;Droisy;DROISY;27320;27;EURE;1
27207;;Drucourt;DRUCOURT;27230;27;EURE;1
27208;;Duranville;DURANVILLE;27230;27;EURE;1
27209;;caquelon;ECAQUELON;27290;27;EURE;1
27210;;cardenville-la-Campagne;ECARDENVILLE-LA-CAMPAGNE;27170;27;EURE;1
27211;;cardenville-sur-Eure;ECARDENVILLE-SUR-EURE;27490;27;EURE;1
27212;;cauville;ECAUVILLE;27110;27;EURE;1
27213;;cos;ECOS;27630;27;EURE;1
27214;;couis;ECOUIS;27440;27;EURE;1
27215;;Ecquetot;ECQUETOT;27110;27;EURE;1
27216;;malleville;EMALLEVILLE;27930;27;EURE;1
27217;;manville;EMANVILLE;27190;27;EURE;1
27218;;paignes;EPAIGNES;27260;27;EURE;1
27219;;pgard;EPEGARD;27110;27;EURE;1
27220;;pieds;EPIEDS;27730;27;EURE;1
27221;;pinay;EPINAY;27330;27;EURE;1
27222;;preville-en-Lieuvin;EPREVILLE-EN-LIEUVIN;27560;27;EURE;1
27223;;preville-en-Roumois;EPREVILLE-EN-ROUMOIS;27310;27;EURE;1
27224;;preville-prs-le-Neubourg;EPREVILLE-PRES-LE-NEUBOURG;27110;27;EURE;1
27225;Les;Essarts;ESSARTS;27240;27;EURE;1
27226;;trpagny;ETREPAGNY;27150;27;EURE;1
27227;;trville;ETREVILLE;27350;27;EURE;1
27228;;turqueraye;ETURQUERAYE;27350;27;EURE;1
27229;;vreux;EVREUX;27000;27;EURE;1
27230;;zy-sur-Eure;EZY-SUR-EURE;27530;27;EURE;1
27231;;Fains;FAINS;27120;27;EURE;1
27232;;Farceaux;FARCEAUX;27150;27;EURE;1
27233;;Fatouville-Grestain;FATOUVILLE-GRESTAIN;27210;27;EURE;1
27234;;Fauville;FAUVILLE;27930;27;EURE;1
27235;;Faverolles-la-Campagne;FAVEROLLES-LA-CAMPAGNE;27190;27;EURE;1
27237;Le;Favril;FAVRIL;27230;27;EURE;1
27238;;Ferrires-Haut-Clocher;FERRIERES-HAUT-CLOCHER;27190;27;EURE;1
27239;;Ferrires-Saint-Hilaire;FERRIERES-SAINT-HILAIRE;27270;27;EURE;1
27240;La;Ferrire-sur-Risle;FERRIERE-SUR-RISLE;27760;27;EURE;1
27241;;Feuguerolles;FEUGUEROLLES;27110;27;EURE;1
27242;Le;Fidelaire;FIDELAIRE;27190;27;EURE;1
27243;;Fiquefleur-quainville;FIQUEFLEUR-EQUAINVILLE;27210;27;EURE;1
27244;;Flancourt-Catelon;FLANCOURT-CATELON;27310;27;EURE;1
27245;;Fleury-la-Fort;FLEURY-LA-FORET;27480;27;EURE;1
27246;;Fleury-sur-Andelle;FLEURY-SUR-ANDELLE;27380;27;EURE;1
27247;;Flipou;FLIPOU;27380;27;EURE;1
27248;;Folleville;FOLLEVILLE;27230;27;EURE;1
27249;;Fontaine-Bellenger;FONTAINE-BELLENGER;27600;27;EURE;1
27250;;Fontaine-Heudebourg;FONTAINE-HEUDEBOURG;27490;27;EURE;1
27251;;Fontaine-l'Abb;FONTAINE-L'ABBE;27300;27;EURE;1
27252;;Fontaine-la-Louvet;FONTAINE-LA-LOUVET;27230;27;EURE;1
27253;;Fontaine-la-Soret;FONTAINE-LA-SORET;27550;27;EURE;1
27254;;Fontaine-sous-Jouy;FONTAINE-SOUS-JOUY;27120;27;EURE;1
27255;;Fontenay;FONTENAY;27510;27;EURE;1
27256;La;Fort-du-Parc;FORET-DU-PARC;27220;27;EURE;1
27257;;Fort-la-Folie;FORET-LA-FOLIE;27510;27;EURE;1
27258;;Fort-Moville;FORT-MOVILLE;27210;27;EURE;1
27259;;Foucrainville;FOUCRAINVILLE;27220;27;EURE;1
27260;;Foulbec;FOULBEC;27210;27;EURE;1
27261;;Fouqueville;FOUQUEVILLE;27370;27;EURE;1
27262;;Fourges;FOURGES;27630;27;EURE;1
27263;;Fourmetot;FOURMETOT;27500;27;EURE;1
27264;;Fours-en-Vexin;FOURS-EN-VEXIN;27630;27;EURE;1
27265;;Francheville;FRANCHEVILLE;27160;27;EURE;1
27266;;Franqueville;FRANQUEVILLE;27800;27;EURE;1
27267;;Freneuse-sur-Risle;FRENEUSE-SUR-RISLE;27290;27;EURE;1
27268;Le;Fresne;FRESNE;27190;27;EURE;1
27269;;Fresne-Cauverville;FRESNE-CAUVERVILLE;27260;27;EURE;1
27270;;Fresne-l'Archevque;FRESNE-L'ARCHEVEQUE;27700;27;EURE;1
27271;;Fresney;FRESNEY;27220;27;EURE;1
27273;;Gadencourt;GADENCOURT;27120;27;EURE;1
27274;;Gaillardbois-Cressenville;GAILLARDBOIS-CRESSENVILLE;27440;27;EURE;1
27275;;Gaillon;GAILLON;27600;27;EURE;1
27276;;Gamaches-en-Vexin;GAMACHES-EN-VEXIN;27150;27;EURE;1
27277;;Garencires;GARENCIERES;27220;27;EURE;1
27278;;Garennes-sur-Eure;GARENNES-SUR-EURE;27780;27;EURE;1
27279;;Gasny;GASNY;27620;27;EURE;1
27280;;Gauciel;GAUCIEL;27930;27;EURE;1
27281;;Gaudreville-la-Rivire;GAUDREVILLE-LA-RIVIERE;27190;27;EURE;1
27282;;Gauville-la-Campagne;GAUVILLE-LA-CAMPAGNE;27930;27;EURE;1
27283;;Gisay-la-Coudre;GISAY-LA-COUDRE;27330;27;EURE;1
27284;;Gisors;GISORS;27140;27;EURE;1
27285;;Giverny;GIVERNY;27620;27;EURE;1
27286;;Giverville;GIVERVILLE;27560;27;EURE;1
27287;;Glisolles;GLISOLLES;27190;27;EURE;1
27288;;Glos-sur-Risle;GLOS-SUR-RISLE;27290;27;EURE;1
27289;La;Goulafrire;GOULAFRIERE;27390;27;EURE;1
27290;;Goupillires;GOUPILLIERES;27170;27;EURE;1
27291;;Gournay-le-Gurin;GOURNAY-LE-GUERIN;27580;27;EURE;1
27292;;Gouttires;GOUTTIERES;27410;27;EURE;1
27293;;Gouville;GOUVILLE;27240;27;EURE;1
27294;;Grainville;GRAINVILLE;27380;27;EURE;1
27295;;Grand-Camp;GRAND-CAMP;27270;27;EURE;1
27296;;Granchain;GRANCHAIN;27410;27;EURE;1
27297;;Grandvilliers;GRANDVILLIERS;27240;27;EURE;1
27298;;Graveron-Smerville;GRAVERON-SEMERVILLE;27110;27;EURE;1
27299;;Gravigny;GRAVIGNY;27930;27;EURE;1
27300;;Grosley-sur-Risle;GROSLEY-SUR-RISLE;27170;27;EURE;1
27301;;Grossuvre;GROSSOEUVRE;27220;27;EURE;1
27302;Le;Gros-Theil;GROS-THEIL;27370;27;EURE;1
27303;;Guernanville;GUERNANVILLE;27160;27;EURE;1
27304;;Guerny;GUERNY;27720;27;EURE;1
27305;La;Guroulde;GUEROULDE;27160;27;EURE;1
27306;;Guichainville;GUICHAINVILLE;27930;27;EURE;1
27307;;Guiseniers;GUISENIERS;27700;27;EURE;1
27308;;Guitry;GUITRY;27510;27;EURE;1
27309;L';Habit;HABIT;27220;27;EURE;1
27310;;Hacqueville;HACQUEVILLE;27150;27;EURE;1
27311;;Harcourt;HARCOURT;27800;27;EURE;1
27312;;Hardencourt-Cocherel;HARDENCOURT-COCHEREL;27120;27;EURE;1
27313;La;Harengre;HARENGERE;27370;27;EURE;1
27315;;Harquency;HARQUENCY;27700;27;EURE;1
27316;;Hauville;HAUVILLE;27350;27;EURE;1
27317;La;Haye-Aubre;HAYE-AUBREE;27350;27;EURE;1
27318;La;Haye-de-Calleville;HAYE-DE-CALLEVILLE;27800;27;EURE;1
27319;La;Haye-de-Routot;HAYE-DE-ROUTOT;27350;27;EURE;1
27320;La;Haye-du-Theil;HAYE-DU-THEIL;27370;27;EURE;1
27321;La;Haye-le-Comte;HAYE-LE-COMTE;27400;27;EURE;1
27322;La;Haye-Malherbe;HAYE-MALHERBE;27400;27;EURE;1
27323;La;Haye-Saint-Sylvestre;HAYE-SAINT-SYLVESTRE;27330;27;EURE;1
27324;;Hbcourt;HEBECOURT;27150;27;EURE;1
27325;;Hecmanville;HECMANVILLE;27800;27;EURE;1
27326;;Hcourt;HECOURT;27120;27;EURE;1
27327;;Hectomare;HECTOMARE;27110;27;EURE;1
27329;;Hennezis;HENNEZIS;27700;27;EURE;1
27330;;Herqueville;HERQUEVILLE;27430;27;EURE;1
27331;;Heubcourt-Haricourt;HEUBECOURT-HARICOURT;27630;27;EURE;1
27332;;Heudebouville;HEUDEBOUVILLE;27400;27;EURE;1
27333;;Heudicourt;HEUDICOURT;27860;27;EURE;1
27334;;Heudreville-en-Lieuvin;HEUDREVILLE-EN-LIEUVIN;27230;27;EURE;1
27335;;Heudreville-sur-Eure;HEUDREVILLE-SUR-EURE;27400;27;EURE;1
27336;La;Heunire;HEUNIERE;27950;27;EURE;1
27337;;Heuqueville;HEUQUEVILLE;27700;27;EURE;1
27338;Les;Hogues;HOGUES;27910;27;EURE;1
27339;;Hondouville;HONDOUVILLE;27400;27;EURE;1
27340;;Honguemare-Guenouville;HONGUEMARE-GUENOUVILLE;27310;27;EURE;1
27341;L';Hosmes;HOSMES;27570;27;EURE;1
27342;;Houetteville;HOUETTEVILLE;27400;27;EURE;1
27343;;Houlbec-Cocherel;HOULBEC-COCHEREL;27120;27;EURE;1
27344;;Houlbec-prs-le-Gros-Theil;HOULBEC-PRES-LE-GROS-THEIL;27370;27;EURE;1
27345;La;Houssaye;HOUSSAYE;27410;27;EURE;1
27346;;Houville-en-Vexin;HOUVILLE-EN-VEXIN;27440;27;EURE;1
27347;;Huest;HUEST;27930;27;EURE;1
27348;;Igoville;IGOVILLE;27460;27;EURE;1
27349;;Illeville-sur-Montfort;ILLEVILLE-SUR-MONTFORT;27290;27;EURE;1
27350;;Illiers-l'vque;ILLIERS-L'EVEQUE;27770;27;EURE;1
27351;;Incarville;INCARVILLE;27400;27;EURE;1
27353;;Irreville;IRREVILLE;27930;27;EURE;1
27354;;Iville;IVILLE;27110;27;EURE;1
27355;;Ivry-la-Bataille;IVRY-LA-BATAILLE;27540;27;EURE;1
27356;;Jonquerets-de-Livet;JONQUERETS-DE-LIVET;27410;27;EURE;1
27358;;Jouy-sur-Eure;JOUY-SUR-EURE;27120;27;EURE;1
27359;;Juignettes;JUIGNETTES;27250;27;EURE;1
27360;;Jumelles;JUMELLES;27220;27;EURE;1
27361;La;Lande-Saint-Lger;LANDE-SAINT-LEGER;27210;27;EURE;1
27362;;Landepreuse;LANDEPEREUSE;27410;27;EURE;1
27363;Le;Landin;LANDIN;27350;27;EURE;1
27364;;Launay;LAUNAY;27470;27;EURE;1
27365;;Lry;LERY;27690;27;EURE;1
27366;;Letteguives;LETTEGUIVES;27910;27;EURE;1
27367;;Lieurey;LIEUREY;27560;27;EURE;1
27368;;Lignerolles;LIGNEROLLES;27220;27;EURE;1
27369;;Lilly;LILLY;27480;27;EURE;1
27370;;Lisors;LISORS;27440;27;EURE;1
27371;;Livet-sur-Authou;LIVET-SUR-AUTHOU;27800;27;EURE;1
27372;;Longchamps;LONGCHAMPS;27150;27;EURE;1
27373;;Lorleau;LORLEAU;27480;27;EURE;1
27374;;Louversey;LOUVERSEY;27190;27;EURE;1
27375;;Louviers;LOUVIERS;27400;27;EURE;1
27376;;Louye;LOUYE;27650;27;EURE;1
27377;;Lyons-la-Fort;LYONS-LA-FORET;27480;27;EURE;1
27378;La;Madeleine-de-Nonancourt;MADELEINE-DE-NONANCOURT;27320;27;EURE;1
27379;;Mainneville;MAINNEVILLE;27150;27;EURE;1
27380;;Malleville-sur-le-Bec;MALLEVILLE-SUR-LE-BEC;27800;27;EURE;1
27381;;Malouy;MALOUY;27300;27;EURE;1
27382;;Mandeville;MANDEVILLE;27370;27;EURE;1
27383;;Mandres;MANDRES;27130;27;EURE;1
27384;;Manneville-la-Raoult;MANNEVILLE-LA-RAOULT;27210;27;EURE;1
27385;;Manneville-sur-Risle;MANNEVILLE-SUR-RISLE;27500;27;EURE;1
27386;Le;Manoir;MANOIR;27460;27;EURE;1
27387;;Manthelon;MANTHELON;27240;27;EURE;1
27388;;Marais-Vernier;MARAIS-VERNIER;27680;27;EURE;1
27389;;Marbeuf;MARBEUF;27110;27;EURE;1
27390;;Marcilly-la-Campagne;MARCILLY-LA-CAMPAGNE;27320;27;EURE;1
27391;;Marcilly-sur-Eure;MARCILLY-SUR-EURE;27810;27;EURE;1
27392;;Martagny;MARTAGNY;27150;27;EURE;1
27393;;Martainville;MARTAINVILLE;27210;27;EURE;1
27394;;Martot;MARTOT;27340;27;EURE;1
27395;;Mlicourt;MELICOURT;27390;27;EURE;1
27396;;Mnesqueville;MENESQUEVILLE;27850;27;EURE;1
27397;;Mnilles;MENILLES;27120;27;EURE;1
27398;;Menneval;MENNEVAL;27300;27;EURE;1
27399;;Mercey;MERCEY;27950;27;EURE;1
27400;;Merey;MEREY;27640;27;EURE;1
27401;Le;Mesnil-Fuguet;MESNIL-FUGUET;27930;27;EURE;1
27402;Le;Mesnil-Hardray;MESNIL-HARDRAY;27190;27;EURE;1
27403;Le;Mesnil-Jourdain;MESNIL-JOURDAIN;27400;27;EURE;1
27404;;Mesnil-Rousset;MESNIL-ROUSSET;27390;27;EURE;1
27405;;Mesnil-sous-Vienne;MESNIL-SOUS-VIENNE;27150;27;EURE;1
27406;;Mesnil-sur-l'Estre;MESNIL-SUR-L'ESTREE;27650;27;EURE;1
27407;;Mesnil-Verclives;MESNIL-VERCLIVES;27440;27;EURE;1
27408;;Mzires-en-Vexin;MEZIERES-EN-VEXIN;27510;27;EURE;1
27410;;Miserey;MISEREY;27930;27;EURE;1
27411;;Moisville;MOISVILLE;27320;27;EURE;1
27412;;Montaure;MONTAURE;27400;27;EURE;1
27413;;Montfort-sur-Risle;MONTFORT-SUR-RISLE;27290;27;EURE;1
27414;;Montreuil-l'Argill;MONTREUIL-L'ARGILLE;27390;27;EURE;1
27415;;Morainville-Jouveaux;MORAINVILLE-JOUVEAUX;27260;27;EURE;1
27416;;Buis-sur-Damville;BUIS-SUR-DAMVILLE;27240;27;EURE;1
27417;;Morgny;MORGNY;27150;27;EURE;1
27418;;Morsan;MORSAN;27800;27;EURE;1
27419;;Mouettes;MOUETTES;27220;27;EURE;1
27420;;Mouflaines;MOUFLAINES;27420;27;EURE;1
27421;;Mousseaux-Neuville;MOUSSEAUX-NEUVILLE;27220;27;EURE;1
27422;;Muids;MUIDS;27430;27;EURE;1
27423;;Muzy;MUZY;27650;27;EURE;1
27424;;Nagel-Sez-Mesnil;NAGEL-SEEZ-MESNIL;27190;27;EURE;1
27425;;Nassandres;NASSANDRES;27550;27;EURE;1
27426;;Neaufles-Saint-Martin;NEAUFLES-SAINT-MARTIN;27830;27;EURE;1
27427;;Neaufles-Auvergny;NEAUFLES-AUVERGNY;27250;27;EURE;1
27428;Le;Neubourg;NEUBOURG;27110;27;EURE;1
27429;;Neuilly;NEUILLY;27730;27;EURE;1
27430;La;Neuve-Grange;NEUVE-GRANGE;27150;27;EURE;1
27431;La;Neuve-Lyre;NEUVE-LYRE;27330;27;EURE;1
27432;La;Neuville-du-Bosc;NEUVILLE-DU-BOSC;27890;27;EURE;1
27433;;Neuville-sur-Authou;NEUVILLE-SUR-AUTHOU;27800;27;EURE;1
27434;;Noards;NOARDS;27560;27;EURE;1
27435;La;No-Poulain;NOE-POULAIN;27560;27;EURE;1
27436;;Nogent-le-Sec;NOGENT-LE-SEC;27190;27;EURE;1
27437;;Nojeon-en-Vexin;NOJEON-EN-VEXIN;27150;27;EURE;1
27438;;Nonancourt;NONANCOURT;27320;27;EURE;1
27439;;Normanville;NORMANVILLE;27930;27;EURE;1
27440;;Notre-Dame-de-l'Isle;NOTRE-DAME-DE-L'ISLE;27940;27;EURE;1
27441;;Notre-Dame-d'pine;NOTRE-DAME-D'EPINE;27800;27;EURE;1
27442;;Notre-Dame-du-Hamel;NOTRE-DAME-DU-HAMEL;27390;27;EURE;1
27444;Le;Noyer-en-Ouche;NOYER-EN-OUCHE;27410;27;EURE;1
27445;;Noyers;NOYERS;27720;27;EURE;1
27446;;Ormes;ORMES;27190;27;EURE;1
27447;;Orvaux;ORVAUX;27190;27;EURE;1
27448;;Pacy-sur-Eure;PACY-SUR-EURE;27120;27;EURE;1
27449;;Panilleuse;PANILLEUSE;27510;27;EURE;1
27451;;Parville;PARVILLE;27180;27;EURE;1
27452;;Perriers-la-Campagne;PERRIERS-LA-CAMPAGNE;27170;27;EURE;1
27453;;Perriers-sur-Andelle;PERRIERS-SUR-ANDELLE;27910;27;EURE;1
27454;;Perruel;PERRUEL;27910;27;EURE;1
27455;;Piencourt;PIENCOURT;27230;27;EURE;1
27456;;Pinterville;PINTERVILLE;27400;27;EURE;1
27457;;Piseux;PISEUX;27130;27;EURE;1
27458;;Ptres;PITRES;27590;27;EURE;1
27459;Les;Places;PLACES;27230;27;EURE;1
27460;;Plainville;PLAINVILLE;27300;27;EURE;1
27462;Le;Planquay;PLANQUAY;27230;27;EURE;1
27463;;Plasnes;PLASNES;27300;27;EURE;1
27464;Le;Plessis-Grohan;PLESSIS-GROHAN;27180;27;EURE;1
27465;Le;Plessis-Hbert;PLESSIS-HEBERT;27120;27;EURE;1
27466;Le;Plessis-Sainte-Opportune;PLESSIS-SAINTE-OPPORTUNE;27170;27;EURE;1
27467;;Pont-Audemer;PONT-AUDEMER;27500;27;EURE;1
27468;;Pont-Authou;PONT-AUTHOU;27290;27;EURE;1
27469;;Pont-de-l'Arche;PONT-DE-L'ARCHE;27340;27;EURE;1
27470;;Pont-Saint-Pierre;PONT-SAINT-PIERRE;27360;27;EURE;1
27471;;Porte-Joie;PORTE-JOIE;27430;27;EURE;1
27472;;Portes;PORTES;27190;27;EURE;1
27473;;Port-Mort;PORT-MORT;27940;27;EURE;1
27474;;Poses;POSES;27740;27;EURE;1
27475;La;Poterie-Mathieu;POTERIE-MATHIEU;27560;27;EURE;1
27476;Les;Praux;PREAUX;27500;27;EURE;1
27477;;Pressagny-l'Orgueilleux;PRESSAGNY-L'ORGUEILLEUX;27510;27;EURE;1
27478;;Prey;PREY;27220;27;EURE;1
27480;;Puchay;PUCHAY;27150;27;EURE;1
27481;;Pullay;PULLAY;27130;27;EURE;1
27482;La;Pyle;PYLE;27370;27;EURE;1
27483;;Quatremare;QUATREMARE;27400;27;EURE;1
27484;;Quessigny;QUESSIGNY;27220;27;EURE;1
27485;;Quillebeuf-sur-Seine;QUILLEBEUF-SUR-SEINE;27680;27;EURE;1
27486;;Quittebeuf;QUITTEBEUF;27110;27;EURE;1
27487;;Radepont;RADEPONT;27380;27;EURE;1
27488;;Renneville;RENNEVILLE;27910;27;EURE;1
27489;;Reuilly;REUILLY;27930;27;EURE;1
27490;;Richeville;RICHEVILLE;27420;27;EURE;1
27491;;Roman;ROMAN;27240;27;EURE;1
27492;;Romilly-la-Puthenaye;ROMILLY-LA-PUTHENAYE;27170;27;EURE;1
27493;;Romilly-sur-Andelle;ROMILLY-SUR-ANDELLE;27610;27;EURE;1
27495;La;Roquette;ROQUETTE;27700;27;EURE;1
27496;;Rosay-sur-Lieure;ROSAY-SUR-LIEURE;27790;27;EURE;1
27497;;Rougemontiers;ROUGEMONTIERS;27350;27;EURE;1
27498;;Rouge-Perriers;ROUGE-PERRIERS;27110;27;EURE;1
27499;La;Roussire;ROUSSIERE;27270;27;EURE;1
27500;;Routot;ROUTOT;27350;27;EURE;1
27501;;Rouvray;ROUVRAY;27120;27;EURE;1
27502;;Rugles;RUGLES;27250;27;EURE;1
27503;Le;Sacq;SACQ;27240;27;EURE;1
27504;;Sacquenville;SACQUENVILLE;27930;27;EURE;1
27505;;Saint-Agnan-de-Cernires;SAINT-AGNAN-DE-CERNIERES;27390;27;EURE;1
27506;;Saint-Amand-des-Hautes-Terres;SAINT-AMAND-DES-HAUTES-TERRES;27370;27;EURE;1
27507;;Saint-Andr-de-l'Eure;SAINT-ANDRE-DE-L'EURE;27220;27;EURE;1
27508;;Saint-Antonin-de-Sommaire;SAINT-ANTONIN-DE-SOMMAIRE;27250;27;EURE;1
27510;;Saint-Aquilin-de-Pacy;SAINT-AQUILIN-DE-PACY;27120;27;EURE;1
27511;;Saint-Aubin-d'crosville;SAINT-AUBIN-D'ECROSVILLE;27110;27;EURE;1
27512;;Saint-Aubin-de-Scellon;SAINT-AUBIN-DE-SCELLON;27230;27;EURE;1
27513;;Saint-Aubin-des-Hayes;SAINT-AUBIN-DES-HAYES;27410;27;EURE;1
27514;;Saint-Aubin-du-Thenney;SAINT-AUBIN-DU-THENNEY;27270;27;EURE;1
27515;;Saint-Aubin-le-Guichard;SAINT-AUBIN-LE-GUICHARD;27410;27;EURE;1
27516;;Saint-Aubin-le-Vertueux;SAINT-AUBIN-LE-VERTUEUX;27300;27;EURE;1
27517;;Saint-Aubin-sur-Gaillon;SAINT-AUBIN-SUR-GAILLON;27600;27;EURE;1
27518;;Saint-Aubin-sur-Quillebeuf;SAINT-AUBIN-SUR-QUILLEBEUF;27680;27;EURE;1
27519;;Sainte-Barbe-sur-Gaillon;SAINTE-BARBE-SUR-GAILLON;27600;27;EURE;1
27520;;Saint-Benot-des-Ombres;SAINT-BENOIT-DES-OMBRES;27450;27;EURE;1
27521;;Saint-Christophe-sur-Avre;SAINT-CHRISTOPHE-SUR-AVRE;27820;27;EURE;1
27522;;Saint-Christophe-sur-Cond;SAINT-CHRISTOPHE-SUR-CONDE;27450;27;EURE;1
27523;;Saint-Clair-d'Arcey;SAINT-CLAIR-D'ARCEY;27300;27;EURE;1
27524;;Sainte-Colombe-la-Commanderie;SAINTE-COLOMBE-LA-COMMANDERIE;27110;27;EURE;1
27525;;Sainte-Colombe-prs-Vernon;SAINTE-COLOMBE-PRES-VERNON;27950;27;EURE;1
27526;;Sainte-Croix-sur-Aizier;SAINTE-CROIX-SUR-AIZIER;27500;27;EURE;1
27527;;Saint-Cyr-de-Salerne;SAINT-CYR-DE-SALERNE;27800;27;EURE;1
27528;Le;Vaudreuil;VAUDREUIL;27100;27;EURE;1
27529;;Saint-Cyr-la-Campagne;SAINT-CYR-LA-CAMPAGNE;27370;27;EURE;1
27530;;Saint-Denis-d'Augerons;SAINT-DENIS-D'AUGERONS;27390;27;EURE;1
27531;;Saint-Denis-des-Monts;SAINT-DENIS-DES-MONTS;27520;27;EURE;1
27532;;Saint-Denis-du-Bhlan;SAINT-DENIS-DU-BEHELAN;27160;27;EURE;1
27533;;Saint-Denis-le-Ferment;SAINT-DENIS-LE-FERMENT;27140;27;EURE;1
27534;;Saint-Didier-des-Bois;SAINT-DIDIER-DES-BOIS;27370;27;EURE;1
27535;;Saint-lier;SAINT-ELIER;27190;27;EURE;1
27536;;Saint-loi-de-Fourques;SAINT-ELOI-DE-FOURQUES;27800;27;EURE;1
27537;;Saint-tienne-du-Vauvray;SAINT-ETIENNE-DU-VAUVRAY;27430;27;EURE;1
27538;;Saint-tienne-l'Allier;SAINT-ETIENNE-L'ALLIER;27450;27;EURE;1
27539;;Saint-tienne-sous-Bailleul;SAINT-ETIENNE-SOUS-BAILLEUL;27920;27;EURE;1
27540;;Sainte-Genevive-ls-Gasny;SAINTE-GENEVIEVE-LES-GASNY;27620;27;EURE;1
27541;;Saint-Georges-du-Mesnil;SAINT-GEORGES-DU-MESNIL;27560;27;EURE;1
27542;;Saint-Georges-du-Vivre;SAINT-GEORGES-DU-VIEVRE;27450;27;EURE;1
27543;;Saint-Georges-Motel;SAINT-GEORGES-MOTEL;27710;27;EURE;1
27544;;Saint-Germain-de-Fresney;SAINT-GERMAIN-DE-FRESNEY;27220;27;EURE;1
27545;;Saint-Germain-de-Pasquier;SAINT-GERMAIN-DE-PASQUIER;27370;27;EURE;1
27546;;Saint-Germain-des-Angles;SAINT-GERMAIN-DES-ANGLES;27930;27;EURE;1
27547;;Saint-Germain-la-Campagne;SAINT-GERMAIN-LA-CAMPAGNE;27230;27;EURE;1
27548;;Saint-Germain-sur-Avre;SAINT-GERMAIN-SUR-AVRE;27320;27;EURE;1
27549;;Saint-Germain-Village;SAINT-GERMAIN-VILLAGE;27500;27;EURE;1
27550;;Saint-Grgoire-du-Vivre;SAINT-GREGOIRE-DU-VIEVRE;27450;27;EURE;1
27551;;Saint-Jean-de-la-Lqueraye;SAINT-JEAN-DE-LA-LEQUERAYE;27560;27;EURE;1
27552;;Saint-Jean-du-Thenney;SAINT-JEAN-DU-THENNEY;27270;27;EURE;1
27553;;Saint-Julien-de-la-Ligue;SAINT-JULIEN-DE-LA-LIEGUE;27600;27;EURE;1
27554;;Saint-Just;SAINT-JUST;27950;27;EURE;1
27555;;Saint-Laurent-des-Bois;SAINT-LAURENT-DES-BOIS;27220;27;EURE;1
27556;;Saint-Laurent-du-Tencement;SAINT-LAURENT-DU-TENCEMENT;27390;27;EURE;1
27557;;Saint-Lger-de-Rtes;SAINT-LEGER-DE-ROTES;27300;27;EURE;1
27558;;Saint-Lger-du-Gennetey;SAINT-LEGER-DU-GENNETEY;27520;27;EURE;1
27560;;Saint-Luc;SAINT-LUC;27930;27;EURE;1
27561;;Saint-Maclou;SAINT-MACLOU;27210;27;EURE;1
27562;;Saint-Marcel;SAINT-MARCEL;27950;27;EURE;1
27563;;Saint-Mards-de-Blacarville;SAINT-MARDS-DE-BLACARVILLE;27500;27;EURE;1
27564;;Saint-Mards-de-Fresne;SAINT-MARDS-DE-FRESNE;27230;27;EURE;1
27565;;Sainte-Marguerite-de-l'Autel;SAINTE-MARGUERITE-DE-L'AUTEL;27160;27;EURE;1
27566;;Sainte-Marguerite-en-Ouche;SAINTE-MARGUERITE-EN-OUCHE;27410;27;EURE;1
27567;;Sainte-Marie-de-Vatimesnil;SAINTE-MARIE-DE-VATIMESNIL;27150;27;EURE;1
27568;;Sainte-Marthe;SAINTE-MARTHE;27190;27;EURE;1
27569;;Saint-Martin-du-Tilleul;SAINT-MARTIN-DU-TILLEUL;27300;27;EURE;1
27570;;Saint-Martin-la-Campagne;SAINT-MARTIN-LA-CAMPAGNE;27930;27;EURE;1
27571;;Saint-Martin-Saint-Firmin;SAINT-MARTIN-SAINT-FIRMIN;27450;27;EURE;1
27572;;Saint-Meslin-du-Bosc;SAINT-MESLIN-DU-BOSC;27370;27;EURE;1
27573;;Saint-Nicolas-d'Attez;SAINT-NICOLAS-D'ATTEZ;27160;27;EURE;1
27574;;Saint-Nicolas-du-Bosc;SAINT-NICOLAS-DU-BOSC;27370;27;EURE;1
27576;;Sainte-Opportune-du-Bosc;SAINTE-OPPORTUNE-DU-BOSC;27110;27;EURE;1
27577;;Sainte-Opportune-la-Mare;SAINTE-OPPORTUNE-LA-MARE;27680;27;EURE;1
27578;;Saint-Ouen-d'Attez;SAINT-OUEN-D'ATTEZ;27160;27;EURE;1
27579;;Saint-Ouen-de-Pontcheuil;SAINT-OUEN-DE-PONTCHEUIL;27370;27;EURE;1
27580;;Saint-Ouen-de-Thouberville;SAINT-OUEN-DE-THOUBERVILLE;27310;27;EURE;1
27581;;Saint-Ouen-des-Champs;SAINT-OUEN-DES-CHAMPS;27680;27;EURE;1
27582;;Saint-Ouen-du-Tilleul;SAINT-OUEN-DU-TILLEUL;27670;27;EURE;1
27584;;Saint-Paul-de-Fourques;SAINT-PAUL-DE-FOURQUES;27800;27;EURE;1
27586;;Saint-Philbert-sur-Boissey;SAINT-PHILBERT-SUR-BOISSEY;27520;27;EURE;1
27587;;Saint-Philbert-sur-Risle;SAINT-PHILBERT-SUR-RISLE;27290;27;EURE;1
27588;;Saint-Pierre-d'Autils;SAINT-PIERRE-D'AUTILS;27950;27;EURE;1
27589;;Saint-Pierre-de-Bailleul;SAINT-PIERRE-DE-BAILLEUL;27920;27;EURE;1
27590;;Saint-Pierre-de-Cernires;SAINT-PIERRE-DE-CERNIERES;27390;27;EURE;1
27591;;Saint-Pierre-de-Cormeilles;SAINT-PIERRE-DE-CORMEILLES;27260;27;EURE;1
27592;;Saint-Pierre-de-Salerne;SAINT-PIERRE-DE-SALERNE;27800;27;EURE;1
27593;;Saint-Pierre-des-Fleurs;SAINT-PIERRE-DES-FLEURS;27370;27;EURE;1
27594;;Saint-Pierre-des-Ifs;SAINT-PIERRE-DES-IFS;27450;27;EURE;1
27595;;Saint-Pierre-du-Bosgurard;SAINT-PIERRE-DU-BOSGUERARD;27370;27;EURE;1
27596;;Saint-Pierre-du-Mesnil;SAINT-PIERRE-DU-MESNIL;27330;27;EURE;1
27597;;Saint-Pierre-du-Val;SAINT-PIERRE-DU-VAL;27210;27;EURE;1
27598;;Saint-Pierre-du-Vauvray;SAINT-PIERRE-DU-VAUVRAY;27430;27;EURE;1
27599;;Saint-Pierre-la-Garenne;SAINT-PIERRE-LA-GARENNE;27600;27;EURE;1
27600;;Saint-Quentin-des-Isles;SAINT-QUENTIN-DES-ISLES;27270;27;EURE;1
27601;;Saint-Samson-de-la-Roque;SAINT-SAMSON-DE-LA-ROQUE;27680;27;EURE;1
27602;;Saint-Sbastien-de-Morsent;SAINT-SEBASTIEN-DE-MORSENT;27180;27;EURE;1
27603;;Saint-Simon;SAINT-SIMEON;27560;27;EURE;1
27604;;Saint-Sulpice-de-Grimbouville;SAINT-SULPICE-DE-GRIMBOUVILLE;27210;27;EURE;1
27605;;Saint-Sylvestre-de-Cormeilles;SAINT-SYLVESTRE-DE-CORMEILLES;27260;27;EURE;1
27606;;Saint-Symphorien;SAINT-SYMPHORIEN;27500;27;EURE;1
27607;;Saint-Thurien;SAINT-THURIEN;27680;27;EURE;1
27608;;Saint-Victor-de-Chrtienville;SAINT-VICTOR-DE-CHRETIENVILLE;27300;27;EURE;1
27609;;Saint-Victor-d'pine;SAINT-VICTOR-D'EPINE;27800;27;EURE;1
27610;;Saint-Victor-sur-Avre;SAINT-VICTOR-SUR-AVRE;27130;27;EURE;1
27611;;Saint-Vigor;SAINT-VIGOR;27930;27;EURE;1
27612;;Saint-Vincent-des-Bois;SAINT-VINCENT-DES-BOIS;27950;27;EURE;1
27613;;Saint-Vincent-du-Boulay;SAINT-VINCENT-DU-BOULAY;27230;27;EURE;1
27614;;Sancourt;SANCOURT;27150;27;EURE;1
27615;;Sassey;SASSEY;27930;27;EURE;1
27616;La;Saussaye;SAUSSAYE;27370;27;EURE;1
27617;;Saussay-la-Campagne;SAUSSAY-LA-CAMPAGNE;27150;27;EURE;1
27618;;Sbcourt;SEBECOURT;27190;27;EURE;1
27620;;Selles;SELLES;27500;27;EURE;1
27621;;Serez;SEREZ;27220;27;EURE;1
27622;;Serquigny;SERQUIGNY;27470;27;EURE;1
27623;;Surtauville;SURTAUVILLE;27400;27;EURE;1
27624;;Surville;SURVILLE;27400;27;EURE;1
27625;;Suzay;SUZAY;27420;27;EURE;1
27626;;Theillement;THEILLEMENT;27520;27;EURE;1
27627;Le;Theil-Nolent;THEIL-NOLENT;27230;27;EURE;1
27628;;Thevray;THEVRAY;27330;27;EURE;1
27629;;Thiberville;THIBERVILLE;27230;27;EURE;1
27630;;Thibouville;THIBOUVILLE;27800;27;EURE;1
27631;;Thierville;THIERVILLE;27290;27;EURE;1
27632;Le;Thil;THIL;27150;27;EURE;1
27633;Les;Thilliers-en-Vexin;THILLIERS-EN-VEXIN;27420;27;EURE;1
27634;;Thomer-la-Sgne;THOMER-LA-SOGNE;27240;27;EURE;1
27635;Le;Thuit;THUIT;27700;27;EURE;1
27636;Le;Thuit-Anger;THUIT-ANGER;27370;27;EURE;1
27637;;Thuit-Hbert;THUIT-HEBERT;27520;27;EURE;1
27638;Le;Thuit-Signol;THUIT-SIGNOL;27370;27;EURE;1
27639;Le;Thuit-Simer;THUIT-SIMER;27370;27;EURE;1
27640;;Tilleul-Dame-Agns;TILLEUL-DAME-AGNES;27170;27;EURE;1
27641;Le;Tilleul-Lambert;TILLEUL-LAMBERT;27110;27;EURE;1
27642;Le;Tilleul-Othon;TILLEUL-OTHON;27170;27;EURE;1
27643;;Tillires-sur-Avre;TILLIERES-SUR-AVRE;27570;27;EURE;1
27644;;Tilly;TILLY;27510;27;EURE;1
27645;;Tocqueville;TOCQUEVILLE;27500;27;EURE;1
27646;Le;Torpt;TORPT;27210;27;EURE;1
27647;;Tosny;TOSNY;27700;27;EURE;1
27648;;Tostes;TOSTES;27340;27;EURE;1
27649;;Touffreville;TOUFFREVILLE;27440;27;EURE;1
27650;;Tournedos-Bois-Hubert;TOURNEDOS-BOIS-HUBERT;27180;27;EURE;1
27651;;Tournedos-sur-Seine;TOURNEDOS-SUR-SEINE;27100;27;EURE;1
27652;;Tourneville;TOURNEVILLE;27930;27;EURE;1
27653;;Tourny;TOURNY;27510;27;EURE;1
27654;;Tourville-la-Campagne;TOURVILLE-LA-CAMPAGNE;27370;27;EURE;1
27655;;Tourville-sur-Pont-Audemer;TOURVILLE-SUR-PONT-AUDEMER;27500;27;EURE;1
27656;;Toutainville;TOUTAINVILLE;27500;27;EURE;1
27657;;Touville;TOUVILLE;27290;27;EURE;1
27658;Le;Tremblay-Omonville;TREMBLAY-OMONVILLE;27110;27;EURE;1
27659;La;Trinit;TRINITE;27930;27;EURE;1
27660;La;Trinit-de-Rville;TRINITE-DE-REVILLE;27270;27;EURE;1
27661;La;Trinit-de-Thouberville;TRINITE-DE-THOUBERVILLE;27310;27;EURE;1
27662;;Triqueville;TRIQUEVILLE;27500;27;EURE;1
27663;Le;Troncq;TRONCQ;27110;27;EURE;1
27664;Le;Tronquay;TRONQUAY;27480;27;EURE;1
27665;;Trouville-la-Haule;TROUVILLE-LA-HAULE;27680;27;EURE;1
27666;La;Vacherie;VACHERIE;27400;27;EURE;1
27667;;Valailles;VALAILLES;27300;27;EURE;1
27668;Le;Val-David;VAL-DAVID;27120;27;EURE;1
27669;;Valletot;VALLETOT;27350;27;EURE;1
27670;;Vandrimare;VANDRIMARE;27380;27;EURE;1
27671;;Vannecrocq;VANNECROCQ;27210;27;EURE;1
27672;;Vascuil;VASCOEUIL;27910;27;EURE;1
27673;;Vatteville;VATTEVILLE;27430;27;EURE;1
27674;;Vaux-sur-Eure;VAUX-SUR-EURE;27120;27;EURE;1
27676;;Venables;VENABLES;27940;27;EURE;1
27677;;Venon;VENON;27110;27;EURE;1
27678;Les;Ventes;VENTES;27180;27;EURE;1
27679;;Verneuil-sur-Avre;VERNEUIL-SUR-AVRE;27130;27;EURE;1
27680;;Verneusses;VERNEUSSES;27390;27;EURE;1
27681;;Vernon;VERNON;27200;27;EURE;1
27682;;Vesly;VESLY;27870;27;EURE;1
27683;;Vzillon;VEZILLON;27700;27;EURE;1
27684;Le;Vieil-vreux;VIEIL-EVREUX;27930;27;EURE;1
27685;La;Vieille-Lyre;VIEILLE-LYRE;27330;27;EURE;1
27686;;Vieux-Port;VIEUX-PORT;27680;27;EURE;1
27687;;Vieux-Villez;VIEUX-VILLEZ;27600;27;EURE;1
27688;;Villalet;VILLALET;27240;27;EURE;1
27689;;Villegats;VILLEGATS;27120;27;EURE;1
27690;;Villers-en-Vexin;VILLERS-EN-VEXIN;27420;27;EURE;1
27691;;Villers-sur-le-Roule;VILLERS-SUR-LE-ROULE;27940;27;EURE;1
27692;;Villettes;VILLETTES;27110;27;EURE;1
27693;;Sylvains-les-Moulins;SYLVAINS-LES-MOULINS;27240;27;EURE;1
27694;;Villez-sous-Bailleul;VILLEZ-SOUS-BAILLEUL;27950;27;EURE;1
27695;;Villez-sur-le-Neubourg;VILLEZ-SUR-LE-NEUBOURG;27110;27;EURE;1
27696;;Villiers-en-Dsuvre;VILLIERS-EN-DESOEUVRE;27640;27;EURE;1
27697;;Vironvay;VIRONVAY;27400;27;EURE;1
27698;;Vitot;VITOT;27110;27;EURE;1
27699;;Voiscreville;VOISCREVILLE;27520;27;EURE;1
27700;;Vraiville;VRAIVILLE;27370;27;EURE;1
27701;;Val-de-Reuil;VAL-DE-REUIL;27100;27;EURE;1
28001;;Abondant;ABONDANT;28570;28;EURE ET LOIR;1
28002;;Allaines-Mervilliers;ALLAINES-MERVILLIERS;28310;28;EURE ET LOIR;1
28003;;Allainville;ALLAINVILLE;28500;28;EURE ET LOIR;1
28004;;Allonnes;ALLONNES;28150;28;EURE ET LOIR;1
28005;;Alluyes;ALLUYES;28800;28;EURE ET LOIR;1
28006;;Amilly;AMILLY;28300;28;EURE ET LOIR;1
28007;;Anet;ANET;28260;28;EURE ET LOIR;1
28008;;Ardelles;ARDELLES;28170;28;EURE ET LOIR;1
28009;;Ardelu;ARDELU;28700;28;EURE ET LOIR;1
28010;;Argenvilliers;ARGENVILLIERS;28420;28;EURE ET LOIR;1
28012;;Arrou;ARROU;28290;28;EURE ET LOIR;1
28013;;Aunay-sous-Auneau;AUNAY-SOUS-AUNEAU;28700;28;EURE ET LOIR;1
28014;;Aunay-sous-Crcy;AUNAY-SOUS-CRECY;28500;28;EURE ET LOIR;1
28015;;Auneau;AUNEAU;28700;28;EURE ET LOIR;1
28016;Les;Autels-Villevillon;AUTELS-VILLEVILLON;28330;28;EURE ET LOIR;1
28017;;Autheuil;AUTHEUIL;28220;28;EURE ET LOIR;1
28018;;Authon-du-Perche;AUTHON-DU-PERCHE;28330;28;EURE ET LOIR;1
28019;;Baigneaux;BAIGNEAUX;28140;28;EURE ET LOIR;1
28020;;Baignolet;BAIGNOLET;28150;28;EURE ET LOIR;1
28021;;Bailleau-le-Pin;BAILLEAU-LE-PIN;28120;28;EURE ET LOIR;1
28022;;Bailleau-l'vque;BAILLEAU-L'EVEQUE;28300;28;EURE ET LOIR;1
28023;;Bailleau-Armenonville;BAILLEAU-ARMENONVILLE;28320;28;EURE ET LOIR;1
28024;;Barjouville;BARJOUVILLE;28630;28;EURE ET LOIR;1
28025;;Barmainville;BARMAINVILLE;28310;28;EURE ET LOIR;1
28026;;Baudreville;BAUDREVILLE;28310;28;EURE ET LOIR;1
28027;La;Bazoche-Gouet;BAZOCHE-GOUET;28330;28;EURE ET LOIR;1
28028;;Bazoches-en-Dunois;BAZOCHES-EN-DUNOIS;28140;28;EURE ET LOIR;1
28029;;Bazoches-les-Hautes;BAZOCHES-LES-HAUTES;28140;28;EURE ET LOIR;1
28030;;Beauche;BEAUCHE;28270;28;EURE ET LOIR;1
28031;;Beaumont-les-Autels;BEAUMONT-LES-AUTELS;28420;28;EURE ET LOIR;1
28032;;Beauvilliers;BEAUVILLIERS;28150;28;EURE ET LOIR;1
28033;;Belhomert-Guhouville;BELHOMERT-GUEHOUVILLE;28240;28;EURE ET LOIR;1
28034;;Berchres-Saint-Germain;BERCHERES-SAINT-GERMAIN;28300;28;EURE ET LOIR;1
28035;;Berchres-les-Pierres;BERCHERES-LES-PIERRES;28630;28;EURE ET LOIR;1
28036;;Berchres-sur-Vesgre;BERCHERES-SUR-VESGRE;28560;28;EURE ET LOIR;1
28037;;Brou-la-Mulotire;BEROU-LA-MULOTIERE;28270;28;EURE ET LOIR;1
28038;;Bthonvilliers;BETHONVILLIERS;28330;28;EURE ET LOIR;1
28039;;Bville-le-Comte;BEVILLE-LE-COMTE;28700;28;EURE ET LOIR;1
28040;;Billancelles;BILLANCELLES;28190;28;EURE ET LOIR;1
28041;;Blandainville;BLANDAINVILLE;28120;28;EURE ET LOIR;1
28044;;Boisgasson;BOISGASSON;28220;28;EURE ET LOIR;1
28045;;Boissy-en-Drouais;BOISSY-EN-DROUAIS;28500;28;EURE ET LOIR;1
28046;;Boissy-ls-Perche;BOISSY-LES-PERCHE;28340;28;EURE ET LOIR;1
28047;;Boisville-la-Saint-Pre;BOISVILLE-LA-SAINT-PERE;28150;28;EURE ET LOIR;1
28048;La;Bourdinire-Saint-Loup;BOURDINIERE-SAINT-LOUP;28360;28;EURE ET LOIR;1
28049;;Bonc;BONCE;28150;28;EURE ET LOIR;1
28050;;Boncourt;BONCOURT;28260;28;EURE ET LOIR;1
28051;;Bonneval;BONNEVAL;28800;28;EURE ET LOIR;1
28052;;Bouglainval;BOUGLAINVAL;28130;28;EURE ET LOIR;1
28053;Le;Boullay-les-Deux-glises;BOULLAY-LES-DEUX-EGLISES;28170;28;EURE ET LOIR;1
28054;Le;Boullay-Mivoye;BOULLAY-MIVOYE;28210;28;EURE ET LOIR;1
28055;Le;Boullay-Thierry;BOULLAY-THIERRY;28210;28;EURE ET LOIR;1
28056;;Boutigny-Prouais;BOUTIGNY-PROUAIS;28410;28;EURE ET LOIR;1
28057;;Bouville;BOUVILLE;28800;28;EURE ET LOIR;1
28058;;Brchamps;BRECHAMPS;28210;28;EURE ET LOIR;1
28059;;Brezolles;BREZOLLES;28270;28;EURE ET LOIR;1
28060;;Briconville;BRICONVILLE;28300;28;EURE ET LOIR;1
28061;;Brou;BROU;28160;28;EURE ET LOIR;1
28062;;Brou;BROUE;28410;28;EURE ET LOIR;1
28063;;Brunelles;BRUNELLES;28400;28;EURE ET LOIR;1
28064;;B;BU;28410;28;EURE ET LOIR;1
28065;;Bullainville;BULLAINVILLE;28800;28;EURE ET LOIR;1
28066;;Bullou;BULLOU;28160;28;EURE ET LOIR;1
28067;;Cernay;CERNAY;28120;28;EURE ET LOIR;1
28068;;Challet;CHALLET;28300;28;EURE ET LOIR;1
28069;;Champagne;CHAMPAGNE;28410;28;EURE ET LOIR;1
28070;;Champhol;CHAMPHOL;28300;28;EURE ET LOIR;1
28071;;Champrond-en-Gtine;CHAMPROND-EN-GATINE;28240;28;EURE ET LOIR;1
28072;;Champrond-en-Perchet;CHAMPROND-EN-PERCHET;28400;28;EURE ET LOIR;1
28073;;Champseru;CHAMPSERU;28700;28;EURE ET LOIR;1
28074;La;Chapelle-d'Aunainville;CHAPELLE-D'AUNAINVILLE;28700;28;EURE ET LOIR;1
28075;La;Chapelle-du-Noyer;CHAPELLE-DU-NOYER;28200;28;EURE ET LOIR;1
28076;La;Chapelle-Forainvilliers;CHAPELLE-FORAINVILLIERS;28500;28;EURE ET LOIR;1
28077;La;Chapelle-Fortin;CHAPELLE-FORTIN;28340;28;EURE ET LOIR;1
28078;;Chapelle-Guillaume;CHAPELLE-GUILLAUME;28330;28;EURE ET LOIR;1
28079;;Chapelle-Royale;CHAPELLE-ROYALE;28290;28;EURE ET LOIR;1
28080;;Charbonnires;CHARBONNIERES;28330;28;EURE ET LOIR;1
28081;;Charonville;CHARONVILLE;28120;28;EURE ET LOIR;1
28082;;Charpont;CHARPONT;28500;28;EURE ET LOIR;1
28083;;Charray;CHARRAY;28220;28;EURE ET LOIR;1
28084;;Chartainvilliers;CHARTAINVILLIERS;28130;28;EURE ET LOIR;1
28085;;Chartres;CHARTRES;28000;28;EURE ET LOIR;1
28086;;Chassant;CHASSANT;28480;28;EURE ET LOIR;1
28087;;Chtaincourt;CHATAINCOURT;28270;28;EURE ET LOIR;1
28088;;Chteaudun;CHATEAUDUN;28200;28;EURE ET LOIR;1
28089;;Chteauneuf-en-Thymerais;CHATEAUNEUF-EN-THYMERAIS;28170;28;EURE ET LOIR;1
28090;Les;Chtelets;CHATELETS;28270;28;EURE ET LOIR;1
28091;Les;Chtelliers-Notre-Dame;CHATELLIERS-NOTRE-DAME;28120;28;EURE ET LOIR;1
28092;;Chtenay;CHATENAY;28700;28;EURE ET LOIR;1
28093;;Chtillon-en-Dunois;CHATILLON-EN-DUNOIS;28290;28;EURE ET LOIR;1
28094;;Chaudon;CHAUDON;28210;28;EURE ET LOIR;1
28095;;Chauffours;CHAUFFOURS;28120;28;EURE ET LOIR;1
28096;La;Chausse-d'Ivry;CHAUSSEE-D'IVRY;28260;28;EURE ET LOIR;1
28098;;Cherisy;CHERISY;28500;28;EURE ET LOIR;1
28099;;Chuisnes;CHUISNES;28190;28;EURE ET LOIR;1
28100;;Cintray;CINTRAY;28300;28;EURE ET LOIR;1
28101;;Civry;CIVRY;28200;28;EURE ET LOIR;1
28102;;Clvilliers;CLEVILLIERS;28300;28;EURE ET LOIR;1
28103;;Cloyes-sur-le-Loir;CLOYES-SUR-LE-LOIR;28220;28;EURE ET LOIR;1
28104;;Coltainville;COLTAINVILLE;28300;28;EURE ET LOIR;1
28105;;Combres;COMBRES;28480;28;EURE ET LOIR;1
28106;;Conie-Molitard;CONIE-MOLITARD;28200;28;EURE ET LOIR;1
28107;;Corancez;CORANCEZ;28630;28;EURE ET LOIR;1
28108;;Cormainville;CORMAINVILLE;28140;28;EURE ET LOIR;1
28109;Les;Corves-les-Yys;CORVEES-LES-YYS;28240;28;EURE ET LOIR;1
28110;Le;Coudray;COUDRAY;28630;28;EURE ET LOIR;1
28111;;Coudray-au-Perche;COUDRAY-AU-PERCHE;28330;28;EURE ET LOIR;1
28112;;Coudreceau;COUDRECEAU;28400;28;EURE ET LOIR;1
28113;;Coulombs;COULOMBS;28210;28;EURE ET LOIR;1
28114;;Courbehaye;COURBEHAYE;28140;28;EURE ET LOIR;1
28115;;Courtalain;COURTALAIN;28290;28;EURE ET LOIR;1
28116;;Courville-sur-Eure;COURVILLE-SUR-EURE;28190;28;EURE ET LOIR;1
28117;;Crcy-Couv;CRECY-COUVE;28500;28;EURE ET LOIR;1
28118;;Croisilles;CROISILLES;28210;28;EURE ET LOIR;1
28119;La;Croix-du-Perche;CROIX-DU-PERCHE;28480;28;EURE ET LOIR;1
28120;;Crucey-Villages;CRUCEY-VILLAGES;28270;28;EURE ET LOIR;1
28121;;Dambron;DAMBRON;28140;28;EURE ET LOIR;1
28122;;Dammarie;DAMMARIE;28360;28;EURE ET LOIR;1
28123;;Dampierre-sous-Brou;DAMPIERRE-SOUS-BROU;28160;28;EURE ET LOIR;1
28124;;Dampierre-sur-Avre;DAMPIERRE-SUR-AVRE;28350;28;EURE ET LOIR;1
28126;;Dancy;DANCY;28800;28;EURE ET LOIR;1
28127;;Dangeau;DANGEAU;28160;28;EURE ET LOIR;1
28128;;Dangers;DANGERS;28190;28;EURE ET LOIR;1
28129;;Denonville;DENONVILLE;28700;28;EURE ET LOIR;1
28130;;Digny;DIGNY;28250;28;EURE ET LOIR;1
28132;;Donnemain-Saint-Mams;DONNEMAIN-SAINT-MAMES;28200;28;EURE ET LOIR;1
28133;;Douy;DOUY;28220;28;EURE ET LOIR;1
28134;;Dreux;DREUX;28100;28;EURE ET LOIR;1
28135;;Droue-sur-Drouette;DROUE-SUR-DROUETTE;28230;28;EURE ET LOIR;1
28136;;cluzelles;ECLUZELLES;28500;28;EURE ET LOIR;1
28137;;crosnes;ECROSNES;28320;28;EURE ET LOIR;1
28139;;peautrolles;EPEAUTROLLES;28120;28;EURE ET LOIR;1
28140;;pernon;EPERNON;28230;28;EURE ET LOIR;1
28141;;Ermenonville-la-Grande;ERMENONVILLE-LA-GRANDE;28120;28;EURE ET LOIR;1
28142;;Ermenonville-la-Petite;ERMENONVILLE-LA-PETITE;28120;28;EURE ET LOIR;1
28143;;Escorpain;ESCORPAIN;28270;28;EURE ET LOIR;1
28144;Les;tilleux;ETILLEUX;28330;28;EURE ET LOIR;1
28145;;Fains-la-Folie;FAINS-LA-FOLIE;28150;28;EURE ET LOIR;1
28146;;Faverolles;FAVEROLLES;28210;28;EURE ET LOIR;1
28147;;Favires;FAVIERES;28170;28;EURE ET LOIR;1
28148;Le;Favril;FAVRIL;28190;28;EURE ET LOIR;1
28149;La;Fert-Vidame;FERTE-VIDAME;28340;28;EURE ET LOIR;1
28150;La;Fert-Villeneuil;FERTE-VILLENEUIL;28220;28;EURE ET LOIR;1
28151;;Fessanvilliers-Mattanvilliers;FESSANVILLIERS-MATTANVILLIERS;28270;28;EURE ET LOIR;1
28153;;Flacey;FLACEY;28800;28;EURE ET LOIR;1
28154;;Fontaine-la-Guyon;FONTAINE-LA-GUYON;28190;28;EURE ET LOIR;1
28155;;Fontaine-les-Ribouts;FONTAINE-LES-RIBOUTS;28170;28;EURE ET LOIR;1
28156;;Fontaine-Simon;FONTAINE-SIMON;28240;28;EURE ET LOIR;1
28157;;Fontenay-sur-Conie;FONTENAY-SUR-CONIE;28140;28;EURE ET LOIR;1
28158;;Fontenay-sur-Eure;FONTENAY-SUR-EURE;28630;28;EURE ET LOIR;1
28159;La;Framboisire;FRAMBOISIERE;28250;28;EURE ET LOIR;1
28160;;Francourville;FRANCOURVILLE;28700;28;EURE ET LOIR;1
28161;;Fraz;FRAZE;28160;28;EURE ET LOIR;1
28162;;Fresnay-le-Comte;FRESNAY-LE-COMTE;28360;28;EURE ET LOIR;1
28163;;Fresnay-le-Gilmert;FRESNAY-LE-GILMERT;28300;28;EURE ET LOIR;1
28164;;Fresnay-l'vque;FRESNAY-L'EVEQUE;28310;28;EURE ET LOIR;1
28165;;Frtigny;FRETIGNY;28480;28;EURE ET LOIR;1
28166;;Friaize;FRIAIZE;28240;28;EURE ET LOIR;1
28167;;Frunc;FRUNCE;28190;28;EURE ET LOIR;1
28168;;Gallardon;GALLARDON;28320;28;EURE ET LOIR;1
28169;;Garancires-en-Beauce;GARANCIERES-EN-BEAUCE;28700;28;EURE ET LOIR;1
28170;;Garancires-en-Drouais;GARANCIERES-EN-DROUAIS;28500;28;EURE ET LOIR;1
28171;;Garnay;GARNAY;28500;28;EURE ET LOIR;1
28172;;Gas;GAS;28320;28;EURE ET LOIR;1
28173;;Gasville-Oisme;GASVILLE-OISEME;28300;28;EURE ET LOIR;1
28175;La;Gaudaine;GAUDAINE;28400;28;EURE ET LOIR;1
28176;Le;Gault-Saint-Denis;GAULT-SAINT-DENIS;28800;28;EURE ET LOIR;1
28177;;Gellainville;GELLAINVILLE;28630;28;EURE ET LOIR;1
28178;;Germainville;GERMAINVILLE;28500;28;EURE ET LOIR;1
28179;;Germignonville;GERMIGNONVILLE;28140;28;EURE ET LOIR;1
28180;;Gilles;GILLES;28260;28;EURE ET LOIR;1
28182;;Gohory;GOHORY;28160;28;EURE ET LOIR;1
28183;;Gommerville;GOMMERVILLE;28310;28;EURE ET LOIR;1
28184;;Gouillons;GOUILLONS;28310;28;EURE ET LOIR;1
28185;;Goussainville;GOUSSAINVILLE;28410;28;EURE ET LOIR;1
28187;;Guainville;GUAINVILLE;28260;28;EURE ET LOIR;1
28188;Le;Gu-de-Longroi;GUE-DE-LONGROI;28700;28;EURE ET LOIR;1
28189;;Guilleville;GUILLEVILLE;28310;28;EURE ET LOIR;1
28190;;Guillonville;GUILLONVILLE;28140;28;EURE ET LOIR;1
28191;;Hanches;HANCHES;28130;28;EURE ET LOIR;1
28192;;Happonvilliers;HAPPONVILLIERS;28480;28;EURE ET LOIR;1
28193;;Havelu;HAVELU;28410;28;EURE ET LOIR;1
28194;;Houville-la-Branche;HOUVILLE-LA-BRANCHE;28700;28;EURE ET LOIR;1
28195;;Houx;HOUX;28130;28;EURE ET LOIR;1
28196;;Illiers-Combray;ILLIERS-COMBRAY;28120;28;EURE ET LOIR;1
28197;;Intrville;INTREVILLE;28310;28;EURE ET LOIR;1
28198;;Jallans;JALLANS;28200;28;EURE ET LOIR;1
28199;;Janville;JANVILLE;28310;28;EURE ET LOIR;1
28200;;Jaudrais;JAUDRAIS;28250;28;EURE ET LOIR;1
28201;;Jouy;JOUY;28300;28;EURE ET LOIR;1
28202;;Lamblore;LAMBLORE;28340;28;EURE ET LOIR;1
28203;;Landelles;LANDELLES;28190;28;EURE ET LOIR;1
28204;;Langey;LANGEY;28220;28;EURE ET LOIR;1
28205;;Lanneray;LANNERAY;28200;28;EURE ET LOIR;1
28206;;Laons;LAONS;28270;28;EURE ET LOIR;1
28207;;Lthuin;LETHUIN;28700;28;EURE ET LOIR;1
28208;;Levainville;LEVAINVILLE;28700;28;EURE ET LOIR;1
28209;;Lves;LEVES;28300;28;EURE ET LOIR;1
28210;;Levesville-la-Chenard;LEVESVILLE-LA-CHENARD;28310;28;EURE ET LOIR;1
28211;;Logron;LOGRON;28200;28;EURE ET LOIR;1
28212;;Loigny-la-Bataille;LOIGNY-LA-BATAILLE;28140;28;EURE ET LOIR;1
28213;;Lormaye;LORMAYE;28210;28;EURE ET LOIR;1
28214;La;Loupe;LOUPE;28240;28;EURE ET LOIR;1
28215;;Louville-la-Chenard;LOUVILLE-LA-CHENARD;28150;28;EURE ET LOIR;1
28216;;Louvilliers-en-Drouais;LOUVILLIERS-EN-DROUAIS;28500;28;EURE ET LOIR;1
28217;;Louvilliers-ls-Perche;LOUVILLIERS-LES-PERCHE;28250;28;EURE ET LOIR;1
28218;;Luc;LUCE;28110;28;EURE ET LOIR;1
28219;;Luigny;LUIGNY;28420;28;EURE ET LOIR;1
28220;;Luisant;LUISANT;28600;28;EURE ET LOIR;1
28221;;Lumeau;LUMEAU;28140;28;EURE ET LOIR;1
28222;;Luplant;LUPLANTE;28360;28;EURE ET LOIR;1
28223;;Luray;LURAY;28500;28;EURE ET LOIR;1
28224;;Lutz-en-Dunois;LUTZ-EN-DUNOIS;28200;28;EURE ET LOIR;1
28225;;Magny;MAGNY;28120;28;EURE ET LOIR;1
28226;;Maillebois;MAILLEBOIS;28170;28;EURE ET LOIR;1
28227;;Maintenon;MAINTENON;28130;28;EURE ET LOIR;1
28229;;Mainvilliers;MAINVILLIERS;28300;28;EURE ET LOIR;1
28230;;Maisons;MAISONS;28700;28;EURE ET LOIR;1
28231;La;Mancelire;MANCELIERE;28270;28;EURE ET LOIR;1
28232;;Manou;MANOU;28240;28;EURE ET LOIR;1
28233;;Marbou;MARBOUE;28200;28;EURE ET LOIR;1
28234;;Marchville;MARCHEVILLE;28120;28;EURE ET LOIR;1
28235;;Marchezais;MARCHEZAIS;28410;28;EURE ET LOIR;1
28236;;Margon;MARGON;28400;28;EURE ET LOIR;1
28237;;Marolles-les-Buis;MAROLLES-LES-BUIS;28400;28;EURE ET LOIR;1
28239;;Marville-Moutiers-Brl;MARVILLE-MOUTIERS-BRULE;28500;28;EURE ET LOIR;1
28240;;Meauc;MEAUCE;28240;28;EURE ET LOIR;1
28241;Le;Me;MEE;28220;28;EURE ET LOIR;1
28242;;Mrglise;MEREGLISE;28120;28;EURE ET LOIR;1
28243;;Mrouville;MEROUVILLE;28310;28;EURE ET LOIR;1
28245;;Meslay-le-Grenet;MESLAY-LE-GRENET;28120;28;EURE ET LOIR;1
28246;;Meslay-le-Vidame;MESLAY-LE-VIDAME;28360;28;EURE ET LOIR;1
28247;Le;Mesnil-Simon;MESNIL-SIMON;28260;28;EURE ET LOIR;1
28248;Le;Mesnil-Thomas;MESNIL-THOMAS;28250;28;EURE ET LOIR;1
28249;;Mvoisins;MEVOISINS;28130;28;EURE ET LOIR;1
28250;;Mzires-au-Perche;MEZIERES-AU-PERCHE;28160;28;EURE ET LOIR;1
28251;;Mzires-en-Drouais;MEZIERES-EN-DROUAIS;28500;28;EURE ET LOIR;1
28252;;Miermaigne;MIERMAIGNE;28420;28;EURE ET LOIR;1
28253;;Mignires;MIGNIERES;28630;28;EURE ET LOIR;1
28254;;Mittainvilliers;MITTAINVILLIERS;28190;28;EURE ET LOIR;1
28255;;Moinville-la-Jeulin;MOINVILLE-LA-JEULIN;28700;28;EURE ET LOIR;1
28256;;Molans;MOLEANS;28200;28;EURE ET LOIR;1
28257;;Mondonville-Saint-Jean;MONDONVILLE-SAINT-JEAN;28700;28;EURE ET LOIR;1
28258;;Montainville;MONTAINVILLE;28150;28;EURE ET LOIR;1
28259;;Montboissier;MONTBOISSIER;28800;28;EURE ET LOIR;1
28260;;Montharville;MONTHARVILLE;28800;28;EURE ET LOIR;1
28261;;Montigny-le-Chartif;MONTIGNY-LE-CHARTIF;28120;28;EURE ET LOIR;1
28262;;Montigny-le-Gannelon;MONTIGNY-LE-GANNELON;28220;28;EURE ET LOIR;1
28263;;Montigny-sur-Avre;MONTIGNY-SUR-AVRE;28270;28;EURE ET LOIR;1
28264;;Montireau;MONTIREAU;28240;28;EURE ET LOIR;1
28265;;Montlandon;MONTLANDON;28240;28;EURE ET LOIR;1
28267;;Montreuil;MONTREUIL;28500;28;EURE ET LOIR;1
28268;;Morainville;MORAINVILLE;28700;28;EURE ET LOIR;1
28269;;Morancez;MORANCEZ;28630;28;EURE ET LOIR;1
28270;;Moriers;MORIERS;28800;28;EURE ET LOIR;1
28271;;Morvilliers;MORVILLIERS;28340;28;EURE ET LOIR;1
28272;;Mottereau;MOTTEREAU;28160;28;EURE ET LOIR;1
28273;;Moulhard;MOULHARD;28160;28;EURE ET LOIR;1
28274;;Moutiers;MOUTIERS;28150;28;EURE ET LOIR;1
28275;;Nron;NERON;28210;28;EURE ET LOIR;1
28276;;Neuvy-en-Beauce;NEUVY-EN-BEAUCE;28310;28;EURE ET LOIR;1
28277;;Neuvy-en-Dunois;NEUVY-EN-DUNOIS;28800;28;EURE ET LOIR;1
28278;;Nogent-le-Phaye;NOGENT-LE-PHAYE;28630;28;EURE ET LOIR;1
28279;;Nogent-le-Roi;NOGENT-LE-ROI;28210;28;EURE ET LOIR;1
28280;;Nogent-le-Rotrou;NOGENT-LE-ROTROU;28400;28;EURE ET LOIR;1
28281;;Nogent-sur-Eure;NOGENT-SUR-EURE;28120;28;EURE ET LOIR;1
28282;;Nonvilliers-Grandhoux;NONVILLIERS-GRANDHOUX;28120;28;EURE ET LOIR;1
28283;;Nottonville;NOTTONVILLE;28140;28;EURE ET LOIR;1
28284;;Oinville-Saint-Liphard;OINVILLE-SAINT-LIPHARD;28310;28;EURE ET LOIR;1
28285;;Oinville-sous-Auneau;OINVILLE-SOUS-AUNEAU;28700;28;EURE ET LOIR;1
28286;;Oll;OLLE;28120;28;EURE ET LOIR;1
28287;;Orgres-en-Beauce;ORGERES-EN-BEAUCE;28140;28;EURE ET LOIR;1
28288;;Orlu;ORLU;28700;28;EURE ET LOIR;1
28289;;Ormoy;ORMOY;28210;28;EURE ET LOIR;1
28290;;Orrouer;ORROUER;28190;28;EURE ET LOIR;1
28291;;Ouarville;OUARVILLE;28150;28;EURE ET LOIR;1
28292;;Ouerre;OUERRE;28500;28;EURE ET LOIR;1
28293;;Oulins;OULINS;28260;28;EURE ET LOIR;1
28294;;Oysonville;OYSONVILLE;28700;28;EURE ET LOIR;1
28295;;Ozoir-le-Breuil;OZOIR-LE-BREUIL;28200;28;EURE ET LOIR;1
28296;;Pronville;PERONVILLE;28140;28;EURE ET LOIR;1
28297;;Pzy;PEZY;28150;28;EURE ET LOIR;1
28298;;Pierres;PIERRES;28130;28;EURE ET LOIR;1
28299;Les;Pinthires;PINTHIERES;28210;28;EURE ET LOIR;1
28300;;Poinville;POINVILLE;28310;28;EURE ET LOIR;1
28301;;Poisvilliers;POISVILLIERS;28300;28;EURE ET LOIR;1
28302;;Pontgouin;PONTGOUIN;28190;28;EURE ET LOIR;1
28303;;Poupry;POUPRY;28140;28;EURE ET LOIR;1
28304;;Prasville;PRASVILLE;28150;28;EURE ET LOIR;1
28305;;Pr-Saint-vroult;PRE-SAINT-EVROULT;28800;28;EURE ET LOIR;1
28306;;Pr-Saint-Martin;PRE-SAINT-MARTIN;28800;28;EURE ET LOIR;1
28308;;Prudemanche;PRUDEMANCHE;28270;28;EURE ET LOIR;1
28309;;Prunay-le-Gillon;PRUNAY-LE-GILLON;28360;28;EURE ET LOIR;1
28310;La;Puisaye;PUISAYE;28250;28;EURE ET LOIR;1
28311;Le;Puiset;PUISET;28310;28;EURE ET LOIR;1
28312;;Puiseux;PUISEUX;28170;28;EURE ET LOIR;1
28313;;Rclainville;RECLAINVILLE;28150;28;EURE ET LOIR;1
28314;Les;Ressuintes;RESSUINTES;28340;28;EURE ET LOIR;1
28315;;Revercourt;REVERCOURT;28270;28;EURE ET LOIR;1
28316;;Rohaire;ROHAIRE;28340;28;EURE ET LOIR;1
28317;;Roinville;ROINVILLE;28700;28;EURE ET LOIR;1
28318;;Romilly-sur-Aigre;ROMILLY-SUR-AIGRE;28220;28;EURE ET LOIR;1
28319;;Rouvray-Saint-Denis;ROUVRAY-SAINT-DENIS;28310;28;EURE ET LOIR;1
28320;;Rouvray-Saint-Florentin;ROUVRAY-SAINT-FLORENTIN;28150;28;EURE ET LOIR;1
28321;;Rouvres;ROUVRES;28260;28;EURE ET LOIR;1
28322;;Rueil-la-Gadelire;RUEIL-LA-GADELIERE;28270;28;EURE ET LOIR;1
28323;;Saint-Ange-et-Toray;SAINT-ANGE-ET-TORCAY;28170;28;EURE ET LOIR;1
28324;;Saint-Arnoult-des-Bois;SAINT-ARNOULT-DES-BOIS;28190;28;EURE ET LOIR;1
28325;;Saint-Aubin-des-Bois;SAINT-AUBIN-DES-BOIS;28300;28;EURE ET LOIR;1
28326;;Saint-Avit-les-Guespires;SAINT-AVIT-LES-GUESPIERES;28120;28;EURE ET LOIR;1
28327;;Saint-Bomer;SAINT-BOMER;28330;28;EURE ET LOIR;1
28329;;Saint-Christophe;SAINT-CHRISTOPHE;28200;28;EURE ET LOIR;1
28330;;Saint-Cloud-en-Dunois;SAINT-CLOUD-EN-DUNOIS;28200;28;EURE ET LOIR;1
28331;;Saint-Denis-d'Authou;SAINT-DENIS-D'AUTHOU;28480;28;EURE ET LOIR;1
28332;;Sainte-Gemme-Moronval;SAINTE-GEMME-MORONVAL;28500;28;EURE ET LOIR;1
28333;;Saint-Denis-des-Puits;SAINT-DENIS-DES-PUITS;28240;28;EURE ET LOIR;1
28334;;Saint-Denis-les-Ponts;SAINT-DENIS-LES-PONTS;28200;28;EURE ET LOIR;1
28335;;Saint-liph;SAINT-ELIPH;28240;28;EURE ET LOIR;1
28336;;Saint-man;SAINT-EMAN;28120;28;EURE ET LOIR;1
28337;;Saint-Georges-sur-Eure;SAINT-GEORGES-SUR-EURE;28190;28;EURE ET LOIR;1
28339;;Saint-Germain-le-Gaillard;SAINT-GERMAIN-LE-GAILLARD;28190;28;EURE ET LOIR;1
28340;;Saint-Hilaire-sur-Yerre;SAINT-HILAIRE-SUR-YERRE;28220;28;EURE ET LOIR;1
28341;;Saint-Jean-de-Rebervilliers;SAINT-JEAN-DE-REBERVILLIERS;28170;28;EURE ET LOIR;1
28342;;Saint-Jean-Pierre-Fixte;SAINT-JEAN-PIERRE-FIXTE;28400;28;EURE ET LOIR;1
28343;;Saint-Laurent-la-Gtine;SAINT-LAURENT-LA-GATINE;28210;28;EURE ET LOIR;1
28344;;Saint-Lger-des-Aubes;SAINT-LEGER-DES-AUBEES;28700;28;EURE ET LOIR;1
28346;;Saint-Lubin-de-Cravant;SAINT-LUBIN-DE-CRAVANT;28270;28;EURE ET LOIR;1
28347;;Saint-Lubin-de-la-Haye;SAINT-LUBIN-DE-LA-HAYE;28410;28;EURE ET LOIR;1
28348;;Saint-Lubin-des-Joncherets;SAINT-LUBIN-DES-JONCHERETS;28350;28;EURE ET LOIR;1
28349;;Saint-Lucien;SAINT-LUCIEN;28210;28;EURE ET LOIR;1
28350;;Saint-Luperce;SAINT-LUPERCE;28190;28;EURE ET LOIR;1
28351;;Saint-Maixme-Hauterive;SAINT-MAIXME-HAUTERIVE;28170;28;EURE ET LOIR;1
28352;;Saint-Martin-de-Nigelles;SAINT-MARTIN-DE-NIGELLES;28130;28;EURE ET LOIR;1
28353;;Saint-Maur-sur-le-Loir;SAINT-MAUR-SUR-LE-LOIR;28800;28;EURE ET LOIR;1
28354;;Saint-Maurice-Saint-Germain;SAINT-MAURICE-SAINT-GERMAIN;28240;28;EURE ET LOIR;1
28355;;Saint-Ouen-Marchefroy;SAINT-OUEN-MARCHEFROY;28560;28;EURE ET LOIR;1
28356;;Saint-Pellerin;SAINT-PELLERIN;28290;28;EURE ET LOIR;1
28357;;Saint-Piat;SAINT-PIAT;28130;28;EURE ET LOIR;1
28358;;Saint-Prest;SAINT-PREST;28300;28;EURE ET LOIR;1
28359;;Saint-Rmy-sur-Avre;SAINT-REMY-SUR-AVRE;28380;28;EURE ET LOIR;1
28360;;Saint-Sauveur-Marville;SAINT-SAUVEUR-MARVILLE;28170;28;EURE ET LOIR;1
28361;;-;BLEURY;28700;28;EURE ET LOIR;1
28362;;Saint-Victor-de-Buthon;SAINT-VICTOR-DE-BUTHON;28240;28;EURE ET LOIR;1
28363;;Sainville;SAINVILLE;28700;28;EURE ET LOIR;1
28364;;Sancheville;SANCHEVILLE;28800;28;EURE ET LOIR;1
28365;;Sandarville;SANDARVILLE;28120;28;EURE ET LOIR;1
28366;;Santeuil;SANTEUIL;28700;28;EURE ET LOIR;1
28367;;Santilly;SANTILLY;28310;28;EURE ET LOIR;1
28368;La;Saucelle;SAUCELLE;28250;28;EURE ET LOIR;1
28369;;Saulnires;SAULNIERES;28500;28;EURE ET LOIR;1
28370;;Saumeray;SAUMERAY;28800;28;EURE ET LOIR;1
28371;;Saussay;SAUSSAY;28260;28;EURE ET LOIR;1
28372;;Senantes;SENANTES;28210;28;EURE ET LOIR;1
28373;;Senonches;SENONCHES;28250;28;EURE ET LOIR;1
28374;;Serazereux;SERAZEREUX;28170;28;EURE ET LOIR;1
28375;;Serville;SERVILLE;28410;28;EURE ET LOIR;1
28376;;Soiz;SOIZE;28330;28;EURE ET LOIR;1
28377;;Sorel-Moussel;SOREL-MOUSSEL;28520;28;EURE ET LOIR;1
28378;;Souanc-au-Perche;SOUANCE-AU-PERCHE;28400;28;EURE ET LOIR;1
28379;;Soulaires;SOULAIRES;28130;28;EURE ET LOIR;1
28380;;Sours;SOURS;28630;28;EURE ET LOIR;1
28382;;Terminiers;TERMINIERS;28140;28;EURE ET LOIR;1
28383;;Theuville;THEUVILLE;28360;28;EURE ET LOIR;1
28385;Le;Thieulin;THIEULIN;28240;28;EURE ET LOIR;1
28386;;Thimert-Gtelles;THIMERT-GATELLES;28170;28;EURE ET LOIR;1
28387;;Thiron-Gardais;THIRON-GARDAIS;28480;28;EURE ET LOIR;1
28388;;Thivars;THIVARS;28630;28;EURE ET LOIR;1
28389;;Thiville;THIVILLE;28200;28;EURE ET LOIR;1
28390;;Tillay-le-Pneux;TILLAY-LE-PENEUX;28140;28;EURE ET LOIR;1
28391;;Toury;TOURY;28390;28;EURE ET LOIR;1
28392;;Trancrainville;TRANCRAINVILLE;28310;28;EURE ET LOIR;1
28393;;Tremblay-les-Villages;TREMBLAY-LES-VILLAGES;28170;28;EURE ET LOIR;1
28394;;Tron;TREON;28500;28;EURE ET LOIR;1
28395;;Trizay-Coutretot-Saint-Serge;TRIZAY-COUTRETOT-SAINT-SERGE;28400;28;EURE ET LOIR;1
28396;;Trizay-ls-Bonneval;TRIZAY-LES-BONNEVAL;28800;28;EURE ET LOIR;1
28397;;Umpeau;UMPEAU;28700;28;EURE ET LOIR;1
28398;;Unverre;UNVERRE;28160;28;EURE ET LOIR;1
28400;;Varize;VARIZE;28140;28;EURE ET LOIR;1
28401;;Vaupillon;VAUPILLON;28240;28;EURE ET LOIR;1
28402;;Vrigny;VERIGNY;28190;28;EURE ET LOIR;1
28403;;Ver-ls-Chartres;VER-LES-CHARTRES;28630;28;EURE ET LOIR;1
28404;;Vernouillet;VERNOUILLET;28500;28;EURE ET LOIR;1
28405;;Vert-en-Drouais;VERT-EN-DROUAIS;28500;28;EURE ET LOIR;1
28406;;Viabon;VIABON;28150;28;EURE ET LOIR;1
28407;;Vichres;VICHERES;28420;28;EURE ET LOIR;1
28408;;Vierville;VIERVILLE;28700;28;EURE ET LOIR;1
28409;;Vieuvicq;VIEUVICQ;28120;28;EURE ET LOIR;1
28410;;Villampuy;VILLAMPUY;28200;28;EURE ET LOIR;1
28411;;Villars;VILLARS;28150;28;EURE ET LOIR;1
28412;;Villeau;VILLEAU;28150;28;EURE ET LOIR;1
28414;;Villebon;VILLEBON;28190;28;EURE ET LOIR;1
28415;;Villemeux-sur-Eure;VILLEMEUX-SUR-EURE;28210;28;EURE ET LOIR;1
28416;;Villeneuve-Saint-Nicolas;VILLENEUVE-SAINT-NICOLAS;28150;28;EURE ET LOIR;1
28417;;Villiers-le-Morhier;VILLIERS-LE-MORHIER;28130;28;EURE ET LOIR;1
28418;;Villiers-Saint-Orien;VILLIERS-SAINT-ORIEN;28800;28;EURE ET LOIR;1
28419;;Vitray-en-Beauce;VITRAY-EN-BEAUCE;28360;28;EURE ET LOIR;1
28421;;Voise;VOISE;28700;28;EURE ET LOIR;1
28422;;Voves;VOVES;28150;28;EURE ET LOIR;1
28423;;Yermenonville;YERMENONVILLE;28130;28;EURE ET LOIR;1
28424;;Yvres;YEVRES;28160;28;EURE ET LOIR;1
28425;;Ymeray;YMERAY;28320;28;EURE ET LOIR;1
28426;;Ymonville;YMONVILLE;28150;28;EURE ET LOIR;1
29001;;Argol;ARGOL;29560;29;FINISTERE;1
29002;;Arzano;ARZANO;29300;29;FINISTERE;1
29003;;Audierne;AUDIERNE;29770;29;FINISTERE;1
29004;;Bannalec;BANNALEC;29380;29;FINISTERE;1
29005;;Baye;BAYE;29300;29;FINISTERE;1
29006;;Bnodet;BENODET;29950;29;FINISTERE;1
29007;;Berrien;BERRIEN;29690;29;FINISTERE;1
29008;;Beuzec-Cap-Sizun;BEUZEC-CAP-SIZUN;29790;29;FINISTERE;1
29010;;Bodilis;BODILIS;29400;29;FINISTERE;1
29011;;Bohars;BOHARS;29820;29;FINISTERE;1
29012;;Bolazec;BOLAZEC;29640;29;FINISTERE;1
29013;;Botmeur;BOTMEUR;29690;29;FINISTERE;1
29014;;Botsorhel;BOTSORHEL;29650;29;FINISTERE;1
29015;;Bourg-Blanc;BOURG-BLANC;29860;29;FINISTERE;1
29016;;Brasparts;BRASPARTS;29190;29;FINISTERE;1
29017;;Brls;BRELES;29810;29;FINISTERE;1
29018;;Brennilis;BRENNILIS;29690;29;FINISTERE;1
29019;;Brest;BREST;29200;29;FINISTERE;1
29020;;Briec;BRIEC;29510;29;FINISTERE;1
29021;;Brignogan-Plage;BRIGNOGAN-PLAGE;29890;29;FINISTERE;1
29022;;Camaret-sur-Mer;CAMARET-SUR-MER;29570;29;FINISTERE;1
29023;;Carantec;CARANTEC;29660;29;FINISTERE;1
29024;;Carhaix-Plouguer;CARHAIX-PLOUGUER;29270;29;FINISTERE;1
29025;;Cast;CAST;29150;29;FINISTERE;1
29026;;Chteaulin;CHATEAULIN;29150;29;FINISTERE;1
29027;;Chteauneuf-du-Faou;CHATEAUNEUF-DU-FAOU;29520;29;FINISTERE;1
29028;;Clden-Cap-Sizun;CLEDEN-CAP-SIZUN;29770;29;FINISTERE;1
29029;;Clden-Poher;CLEDEN-POHER;29270;29;FINISTERE;1
29030;;Clder;CLEDER;29233;29;FINISTERE;1
29031;;Clohars-Carnot;CLOHARS-CARNOET;29360;29;FINISTERE;1
29032;;Clohars-Fouesnant;CLOHARS-FOUESNANT;29950;29;FINISTERE;1
29033;Le;Clotre-Pleyben;CLOITRE-PLEYBEN;29190;29;FINISTERE;1
29034;Le;Clotre-Saint-Thgonnec;CLOITRE-SAINT-THEGONNEC;29410;29;FINISTERE;1
29035;;Coat-Mal;COAT-MEAL;29870;29;FINISTERE;1
29036;;Collorec;COLLOREC;29530;29;FINISTERE;1
29037;;Combrit;COMBRIT;29120;29;FINISTERE;1
29038;;Commana;COMMANA;29450;29;FINISTERE;1
29039;;Concarneau;CONCARNEAU;29900;29;FINISTERE;1
29040;Le;Conquet;CONQUET;29217;29;FINISTERE;1
29041;;Coray;CORAY;29370;29;FINISTERE;1
29042;;Crozon;CROZON;29160;29;FINISTERE;1
29043;;Daoulas;DAOULAS;29460;29;FINISTERE;1
29044;;Dinault;DINEAULT;29150;29;FINISTERE;1
29045;;Dirinon;DIRINON;29460;29;FINISTERE;1
29046;;Douarnenez;DOUARNENEZ;29100;29;FINISTERE;1
29047;Le;Drennec;DRENNEC;29860;29;FINISTERE;1
29048;;Edern;EDERN;29510;29;FINISTERE;1
29049;;Elliant;ELLIANT;29370;29;FINISTERE;1
29051;;Ergu-Gabric;ERGUE-GABERIC;29500;29;FINISTERE;1
29052;;Esquibien;ESQUIBIEN;29770;29;FINISTERE;1
29053;Le;Faou;FAOU;29590;29;FINISTERE;1
29054;La;Feuille;FEUILLEE;29690;29;FINISTERE;1
29055;Le;Folgot;FOLGOET;29260;29;FINISTERE;1
29056;La;Forest-Landerneau;FOREST-LANDERNEAU;29800;29;FINISTERE;1
29057;La;Fort-Fouesnant;FORET-FOUESNANT;29940;29;FINISTERE;1
29058;;Fouesnant;FOUESNANT;29170;29;FINISTERE;1
29059;;Garlan;GARLAN;29610;29;FINISTERE;1
29060;;Gouesnach;GOUESNACH;29950;29;FINISTERE;1
29061;;Gouesnou;GOUESNOU;29850;29;FINISTERE;1
29062;;Gouzec;GOUEZEC;29190;29;FINISTERE;1
29063;;Goulien;GOULIEN;29770;29;FINISTERE;1
29064;;Goulven;GOULVEN;29890;29;FINISTERE;1
29065;;Gourlizon;GOURLIZON;29710;29;FINISTERE;1
29066;;Guengat;GUENGAT;29180;29;FINISTERE;1
29067;;Guerlesquin;GUERLESQUIN;29650;29;FINISTERE;1
29068;;Guiclan;GUICLAN;29410;29;FINISTERE;1
29069;;Guilers;GUILERS;29820;29;FINISTERE;1
29070;;Guiler-sur-Goyen;GUILER-SUR-GOYEN;29710;29;FINISTERE;1
29071;;Guilligomarc'h;GUILLIGOMARC'H;29300;29;FINISTERE;1
29072;;Guilvinec;GUILVINEC;29730;29;FINISTERE;1
29073;;Guimac;GUIMAEC;29620;29;FINISTERE;1
29074;;Guimiliau;GUIMILIAU;29400;29;FINISTERE;1
29075;;Guipavas;GUIPAVAS;29490;29;FINISTERE;1
29076;;Guipronvel;GUIPRONVEL;29290;29;FINISTERE;1
29077;;Guissny;GUISSENY;29880;29;FINISTERE;1
29078;;Hanvec;HANVEC;29460;29;FINISTERE;1
29079;;Henvic;HENVIC;29670;29;FINISTERE;1
29080;;Hpital-Camfrout;HOPITAL-CAMFROUT;29460;29;FINISTERE;1
29081;;Huelgoat;HUELGOAT;29690;29;FINISTERE;1
29082;;le-de-Batz;ILE-DE-BATZ;29253;29;FINISTERE;1
29083;;le-de-Sein;ILE-DE-SEIN;29990;29;FINISTERE;1
29084;;le-Molne;ILE-MOLENE;29259;29;FINISTERE;1
29085;;le-Tudy;ILE-TUDY;29980;29;FINISTERE;1
29086;;Irvillac;IRVILLAC;29460;29;FINISTERE;1
29087;Le;Juch;JUCH;29100;29;FINISTERE;1
29089;;Kergloff;KERGLOFF;29270;29;FINISTERE;1
29090;;Kerlaz;KERLAZ;29100;29;FINISTERE;1
29091;;Kerlouan;KERLOUAN;29890;29;FINISTERE;1
29093;;Kernilis;KERNILIS;29260;29;FINISTERE;1
29094;;Kernous;KERNOUES;29260;29;FINISTERE;1
29095;;Kersaint-Plabennec;KERSAINT-PLABENNEC;29860;29;FINISTERE;1
29097;;Lampaul-Guimiliau;LAMPAUL-GUIMILIAU;29400;29;FINISTERE;1
29098;;Lampaul-Plouarzel;LAMPAUL-PLOUARZEL;29810;29;FINISTERE;1
29099;;Lampaul-Ploudalmzeau;LAMPAUL-PLOUDALMEZEAU;29830;29;FINISTERE;1
29100;;Lanarvily;LANARVILY;29260;29;FINISTERE;1
29101;;Landda;LANDEDA;29870;29;FINISTERE;1
29102;;Landeleau;LANDELEAU;29530;29;FINISTERE;1
29103;;Landerneau;LANDERNEAU;29800;29;FINISTERE;1
29104;;Landvennec;LANDEVENNEC;29560;29;FINISTERE;1
29105;;Landivisiau;LANDIVISIAU;29400;29;FINISTERE;1
29106;;Landrvarzec;LANDREVARZEC;29510;29;FINISTERE;1
29107;;Landudal;LANDUDAL;29510;29;FINISTERE;1
29108;;Landudec;LANDUDEC;29710;29;FINISTERE;1
29109;;Landunvez;LANDUNVEZ;29840;29;FINISTERE;1
29110;;Langolen;LANGOLEN;29510;29;FINISTERE;1
29111;;Lanhouarneau;LANHOUARNEAU;29430;29;FINISTERE;1
29112;;Lanildut;LANILDUT;29840;29;FINISTERE;1
29113;;Lanmeur;LANMEUR;29620;29;FINISTERE;1
29114;;Lannanou;LANNEANOU;29640;29;FINISTERE;1
29115;;Lanndern;LANNEDERN;29190;29;FINISTERE;1
29116;;Lanneuffret;LANNEUFFRET;29400;29;FINISTERE;1
29117;;Lannilis;LANNILIS;29870;29;FINISTERE;1
29119;;Lanrivoar;LANRIVOARE;29290;29;FINISTERE;1
29120;;Lanvoc;LANVEOC;29160;29;FINISTERE;1
29122;;Laz;LAZ;29520;29;FINISTERE;1
29123;;Lennon;LENNON;29190;29;FINISTERE;1
29124;;Lesneven;LESNEVEN;29260;29;FINISTERE;1
29125;;Leuhan;LEUHAN;29390;29;FINISTERE;1
29126;;Loc-Brvalaire;LOC-BREVALAIRE;29260;29;FINISTERE;1
29127;;Loc-Eguiner-Saint-Thgonnec;LOC-EGUINER-SAINT-THEGONNEC;29410;29;FINISTERE;1
29128;;Loc-Eguiner;LOC-EGUINER;29400;29;FINISTERE;1
29129;;Locmaria-Berrien;LOCMARIA-BERRIEN;29690;29;FINISTERE;1
29130;;Locmaria-Plouzan;LOCMARIA-PLOUZANE;29280;29;FINISTERE;1
29131;;Locmlar;LOCMELAR;29400;29;FINISTERE;1
29132;;Locqunol;LOCQUENOLE;29670;29;FINISTERE;1
29133;;Locquirec;LOCQUIREC;29241;29;FINISTERE;1
29134;;Locronan;LOCRONAN;29180;29;FINISTERE;1
29135;;Loctudy;LOCTUDY;29750;29;FINISTERE;1
29136;;Locunol;LOCUNOLE;29310;29;FINISTERE;1
29137;;Logonna-Daoulas;LOGONNA-DAOULAS;29460;29;FINISTERE;1
29139;;Loprec;LOPEREC;29590;29;FINISTERE;1
29140;;Loperhet;LOPERHET;29470;29;FINISTERE;1
29141;;Loqueffret;LOQUEFFRET;29530;29;FINISTERE;1
29142;;Lothey;LOTHEY;29190;29;FINISTERE;1
29143;;Mahalon;MAHALON;29790;29;FINISTERE;1
29144;La;Martyre;MARTYRE;29800;29;FINISTERE;1
29145;;Confort-Meilars;CONFORT-MEILARS;29790;29;FINISTERE;1
29146;;Melgven;MELGVEN;29140;29;FINISTERE;1
29147;;Mellac;MELLAC;29300;29;FINISTERE;1
29148;;Mespaul;MESPAUL;29420;29;FINISTERE;1
29149;;Milizac;MILIZAC;29290;29;FINISTERE;1
29150;;Molan-sur-Mer;MOELAN-SUR-MER;29350;29;FINISTERE;1
29151;;Morlaix;MORLAIX;29600;29;FINISTERE;1
29152;;Motreff;MOTREFF;29270;29;FINISTERE;1
29153;;Nvez;NEVEZ;29920;29;FINISTERE;1
29155;;Ouessant;OUESSANT;29242;29;FINISTERE;1
29156;;Pencran;PENCRAN;29800;29;FINISTERE;1
29158;;Penmarch;PENMARCH;29760;29;FINISTERE;1
29159;;Peumrit;PEUMERIT;29710;29;FINISTERE;1
29160;;Plabennec;PLABENNEC;29860;29;FINISTERE;1
29161;;Pleuven;PLEUVEN;29170;29;FINISTERE;1
29162;;Pleyben;PLEYBEN;29190;29;FINISTERE;1
29163;;Pleyber-Christ;PLEYBER-CHRIST;29410;29;FINISTERE;1
29165;;Plobannalec-Lesconil;PLOBANNALEC-LESCONIL;29740;29;FINISTERE;1
29166;;Ploven;PLOEVEN;29550;29;FINISTERE;1
29167;;Plogastel-Saint-Germain;PLOGASTEL-SAINT-GERMAIN;29710;29;FINISTERE;1
29168;;Plogoff;PLOGOFF;29770;29;FINISTERE;1
29169;;Plogonnec;PLOGONNEC;29180;29;FINISTERE;1
29170;;Plomelin;PLOMELIN;29700;29;FINISTERE;1
29171;;Plomeur;PLOMEUR;29120;29;FINISTERE;1
29172;;Plomodiern;PLOMODIERN;29550;29;FINISTERE;1
29173;;Plonis;PLONEIS;29710;29;FINISTERE;1
29174;;Plonour-Lanvern;PLONEOUR-LANVERN;29720;29;FINISTERE;1
29175;;Plonvez-du-Faou;PLONEVEZ-DU-FAOU;29530;29;FINISTERE;1
29176;;Plonvez-Porzay;PLONEVEZ-PORZAY;29550;29;FINISTERE;1
29177;;Plouarzel;PLOUARZEL;29810;29;FINISTERE;1
29178;;Ploudalmzeau;PLOUDALMEZEAU;29830;29;FINISTERE;1
29179;;Ploudaniel;PLOUDANIEL;29260;29;FINISTERE;1
29180;;Ploudiry;PLOUDIRY;29800;29;FINISTERE;1
29181;;Ploudern;PLOUEDERN;29800;29;FINISTERE;1
29182;;Plougat-Gurand;PLOUEGAT-GUERAND;29620;29;FINISTERE;1
29183;;Plougat-Moysan;PLOUEGAT-MOYSAN;29650;29;FINISTERE;1
29184;;Plounan;PLOUENAN;29420;29;FINISTERE;1
29185;;Plouescat;PLOUESCAT;29430;29;FINISTERE;1
29186;;Plouezoc'h;PLOUEZOC'H;29252;29;FINISTERE;1
29187;;Plougar;PLOUGAR;29440;29;FINISTERE;1
29188;;Plougasnou;PLOUGASNOU;29630;29;FINISTERE;1
29189;;Plougastel-Daoulas;PLOUGASTEL-DAOULAS;29470;29;FINISTERE;1
29190;;Plougonvelin;PLOUGONVELIN;29217;29;FINISTERE;1
29191;;Plougonven;PLOUGONVEN;29640;29;FINISTERE;1
29192;;Plougoulm;PLOUGOULM;29250;29;FINISTERE;1
29193;;Plougourvest;PLOUGOURVEST;29400;29;FINISTERE;1
29195;;Plouguerneau;PLOUGUERNEAU;29880;29;FINISTERE;1
29196;;Plouguin;PLOUGUIN;29830;29;FINISTERE;1
29197;;Plouhinec;PLOUHINEC;29780;29;FINISTERE;1
29198;;Plouider;PLOUIDER;29260;29;FINISTERE;1
29199;;Plouigneau;PLOUIGNEAU;29610;29;FINISTERE;1
29201;;Ploumoguer;PLOUMOGUER;29810;29;FINISTERE;1
29202;;Plounour-Mnez;PLOUNEOUR-MENEZ;29410;29;FINISTERE;1
29203;;Plounour-Trez;PLOUNEOUR-TREZ;29890;29;FINISTERE;1
29204;;Plounventer;PLOUNEVENTER;29400;29;FINISTERE;1
29205;;Plounvzel;PLOUNEVEZEL;29270;29;FINISTERE;1
29206;;Plounvez-Lochrist;PLOUNEVEZ-LOCHRIST;29430;29;FINISTERE;1
29207;;Plourin-ls-Morlaix;PLOURIN-LES-MORLAIX;29600;29;FINISTERE;1
29208;;Plourin;PLOURIN;29830;29;FINISTERE;1
29209;;Plouvien;PLOUVIEN;29860;29;FINISTERE;1
29210;;Plouvorn;PLOUVORN;29420;29;FINISTERE;1
29211;;Plouy;PLOUYE;29690;29;FINISTERE;1
29212;;Plouzan;PLOUZANE;29280;29;FINISTERE;1
29213;;Plouzvd;PLOUZEVEDE;29440;29;FINISTERE;1
29214;;Plovan;PLOVAN;29720;29;FINISTERE;1
29215;;Plozvet;PLOZEVET;29710;29;FINISTERE;1
29216;;Pluguffan;PLUGUFFAN;29700;29;FINISTERE;1
29217;;Pont-Aven;PONT-AVEN;29930;29;FINISTERE;1
29218;;Pont-Croix;PONT-CROIX;29790;29;FINISTERE;1
29219;Le;Ponthou;PONTHOU;29650;29;FINISTERE;1
29220;;Pont-l'Abb;PONT-L'ABBE;29120;29;FINISTERE;1
29221;;Porspoder;PORSPODER;29840;29;FINISTERE;1
29222;;Port-Launay;PORT-LAUNAY;29150;29;FINISTERE;1
29224;;Pouldergat;POULDERGAT;29100;29;FINISTERE;1
29225;;Pouldreuzic;POULDREUZIC;29710;29;FINISTERE;1
29226;;Poullan-sur-Mer;POULLAN-SUR-MER;29100;29;FINISTERE;1
29227;;Poullaouen;POULLAOUEN;29246;29;FINISTERE;1
29228;;Primelin;PRIMELIN;29770;29;FINISTERE;1
29229;;Qumnven;QUEMENEVEN;29180;29;FINISTERE;1
29230;;Querrien;QUERRIEN;29310;29;FINISTERE;1
29232;;Quimper;QUIMPER;29000;29;FINISTERE;1
29233;;Quimperl;QUIMPERLE;29300;29;FINISTERE;1
29234;;Rden;REDENE;29300;29;FINISTERE;1
29235;Le;Relecq-Kerhuon;RELECQ-KERHUON;29480;29;FINISTERE;1
29236;;Riec-sur-Belon;RIEC-SUR-BELON;29340;29;FINISTERE;1
29237;La;Roche-Maurice;ROCHE-MAURICE;29800;29;FINISTERE;1
29238;;Roscanvel;ROSCANVEL;29570;29;FINISTERE;1
29239;;Roscoff;ROSCOFF;29680;29;FINISTERE;1
29240;;Rosnon;ROSNOEN;29590;29;FINISTERE;1
29241;;Rosporden;ROSPORDEN;29140;29;FINISTERE;1
29243;;Saint-Coulitz;SAINT-COULITZ;29150;29;FINISTERE;1
29244;;Saint-Derrien;SAINT-DERRIEN;29440;29;FINISTERE;1
29245;;Saint-Divy;SAINT-DIVY;29800;29;FINISTERE;1
29246;;Saint-Eloy;SAINT-ELOY;29460;29;FINISTERE;1
29247;;Saint-varzec;SAINT-EVARZEC;29170;29;FINISTERE;1
29248;;Saint-Frgant;SAINT-FREGANT;29260;29;FINISTERE;1
29249;;Saint-Goazec;SAINT-GOAZEC;29520;29;FINISTERE;1
29250;;Saint-Hernin;SAINT-HERNIN;29270;29;FINISTERE;1
29251;;Saint-Jean-du-Doigt;SAINT-JEAN-DU-DOIGT;29630;29;FINISTERE;1
29252;;Saint-Jean-Trolimon;SAINT-JEAN-TROLIMON;29120;29;FINISTERE;1
29254;;Saint-Martin-des-Champs;SAINT-MARTIN-DES-CHAMPS;29600;29;FINISTERE;1
29255;;Saint-Men;SAINT-MEEN;29260;29;FINISTERE;1
29256;;Saint-Nic;SAINT-NIC;29550;29;FINISTERE;1
29257;;Saint-Pabu;SAINT-PABU;29830;29;FINISTERE;1
29259;;Saint-Pol-de-Lon;SAINT-POL-DE-LEON;29250;29;FINISTERE;1
29260;;Saint-Renan;SAINT-RENAN;29290;29;FINISTERE;1
29261;;Saint-Rivoal;SAINT-RIVOAL;29190;29;FINISTERE;1
29262;;Saint-Sauveur;SAINT-SAUVEUR;29400;29;FINISTERE;1
29263;;Saint-Sgal;SAINT-SEGAL;29590;29;FINISTERE;1
29264;;Saint-Servais;SAINT-SERVAIS;29400;29;FINISTERE;1
29265;;Sainte-Sve;SAINTE-SEVE;29600;29;FINISTERE;1
29266;;Saint-Thgonnec;SAINT-THEGONNEC;29410;29;FINISTERE;1
29267;;Saint-Thois;SAINT-THOIS;29520;29;FINISTERE;1
29268;;Saint-Thonan;SAINT-THONAN;29800;29;FINISTERE;1
29269;;Saint-Thurien;SAINT-THURIEN;29380;29;FINISTERE;1
29270;;Saint-Urbain;SAINT-URBAIN;29800;29;FINISTERE;1
29271;;Saint-Vougay;SAINT-VOUGAY;29440;29;FINISTERE;1
29272;;Saint-Yvi;SAINT-YVI;29140;29;FINISTERE;1
29273;;Santec;SANTEC;29250;29;FINISTERE;1
29274;;Scar;SCAER;29390;29;FINISTERE;1
29275;;Scrignac;SCRIGNAC;29640;29;FINISTERE;1
29276;;Sibiril;SIBIRIL;29250;29;FINISTERE;1
29277;;Sizun;SIZUN;29450;29;FINISTERE;1
29278;;Spzet;SPEZET;29540;29;FINISTERE;1
29279;;Taul;TAULE;29670;29;FINISTERE;1
29280;;Telgruc-sur-Mer;TELGRUC-SUR-MER;29560;29;FINISTERE;1
29281;;Tourch;TOURCH;29140;29;FINISTERE;1
29282;;Trbabu;TREBABU;29217;29;FINISTERE;1
29284;;Treffiagat;TREFFIAGAT;29730;29;FINISTERE;1
29285;;Trflaounan;TREFLAOUENAN;29440;29;FINISTERE;1
29286;;Trflvnez;TREFLEVENEZ;29800;29;FINISTERE;1
29287;;Trflez;TREFLEZ;29430;29;FINISTERE;1
29288;;Trgarantec;TREGARANTEC;29260;29;FINISTERE;1
29289;;Trgarvan;TREGARVAN;29560;29;FINISTERE;1
29290;;Trglonou;TREGLONOU;29870;29;FINISTERE;1
29291;;Trgourez;TREGOUREZ;29970;29;FINISTERE;1
29292;;Trguennec;TREGUENNEC;29720;29;FINISTERE;1
29293;;Trgunc;TREGUNC;29910;29;FINISTERE;1
29294;Le;Trhou;TREHOU;29450;29;FINISTERE;1
29295;;Trmaouzan;TREMAOUEZAN;29800;29;FINISTERE;1
29296;;Trmoc;TREMEOC;29120;29;FINISTERE;1
29297;;Trmven;TREMEVEN;29300;29;FINISTERE;1
29298;;Trogat;TREOGAT;29720;29;FINISTERE;1
29299;;Trouergat;TREOUERGAT;29290;29;FINISTERE;1
29300;Le;Trvoux;TREVOUX;29380;29;FINISTERE;1
29301;;Trzilid;TREZILIDE;29440;29;FINISTERE;1
29302;;Pont-de-Buis-ls-Quimerch;PONT-DE-BUIS-LES-QUIMERCH;29590;29;FINISTERE;1
30001;;Aigaliers;AIGALIERS;30700;30;GARD;1
30002;;Aigremont;AIGREMONT;30350;30;GARD;1
30003;;Aigues-Mortes;AIGUES-MORTES;30220;30;GARD;1
30004;;Aigues-Vives;AIGUES-VIVES;30670;30;GARD;1
30005;;Aiguze;AIGUEZE;30760;30;GARD;1
30006;;Aimargues;AIMARGUES;30470;30;GARD;1
30007;;Als;ALES;30100;30;GARD;1
30008;;Allgre-les-Fumades;ALLEGRE-LES-FUMADES;30500;30;GARD;1
30009;;Alzon;ALZON;30770;30;GARD;1
30010;;Anduze;ANDUZE;30140;30;GARD;1
30011;Les;Angles;ANGLES;30133;30;GARD;1
30012;;Aramon;ARAMON;30390;30;GARD;1
30013;;Argilliers;ARGILLIERS;30210;30;GARD;1
30014;;Arpaillargues-et-Aureillac;ARPAILLARGUES-ET-AUREILLAC;30700;30;GARD;1
30015;;Arphy;ARPHY;30120;30;GARD;1
30016;;Arre;ARRE;30120;30;GARD;1
30017;;Arrigas;ARRIGAS;30770;30;GARD;1
30018;;Aspres;ASPERES;30250;30;GARD;1
30019;;Aubais;AUBAIS;30250;30;GARD;1
30020;;Aubord;AUBORD;30620;30;GARD;1
30021;;Aubussargues;AUBUSSARGUES;30190;30;GARD;1
30022;;Aujac;AUJAC;30450;30;GARD;1
30023;;Aujargues;AUJARGUES;30250;30;GARD;1
30024;;Aulas;AULAS;30120;30;GARD;1
30025;;Aumessas;AUMESSAS;30770;30;GARD;1
30026;;Avze;AVEZE;30120;30;GARD;1
30027;;Bagard;BAGARD;30140;30;GARD;1
30028;;Bagnols-sur-Cze;BAGNOLS-SUR-CEZE;30200;30;GARD;1
30029;;Barjac;BARJAC;30430;30;GARD;1
30030;;Baron;BARON;30700;30;GARD;1
30031;La;Bastide-d'Engras;BASTIDE-D'ENGRAS;30330;30;GARD;1
30032;;Beaucaire;BEAUCAIRE;30300;30;GARD;1
30033;;Beauvoisin;BEAUVOISIN;30640;30;GARD;1
30034;;Bellegarde;BELLEGARDE;30127;30;GARD;1
30035;;Belvzet;BELVEZET;30580;30;GARD;1
30036;;Bernis;BERNIS;30620;30;GARD;1
30037;;Bessges;BESSEGES;30160;30;GARD;1
30038;;Bez-et-Esparon;BEZ-ET-ESPARON;30120;30;GARD;1
30039;;Bezouce;BEZOUCE;30320;30;GARD;1
30040;;Blandas;BLANDAS;30770;30;GARD;1
30041;;Blauzac;BLAUZAC;30700;30;GARD;1
30042;;Boisset-et-Gaujac;BOISSET-ET-GAUJAC;30140;30;GARD;1
30043;;Boissires;BOISSIERES;30114;30;GARD;1
30044;;Bonnevaux;BONNEVAUX;30450;30;GARD;1
30045;;Bordezac;BORDEZAC;30160;30;GARD;1
30046;;Boucoiran-et-Nozires;BOUCOIRAN-ET-NOZIERES;30190;30;GARD;1
30047;;Bouillargues;BOUILLARGUES;30230;30;GARD;1
30048;;Bouquet;BOUQUET;30580;30;GARD;1
30049;;Bourdic;BOURDIC;30190;30;GARD;1
30050;;Bragassargues;BRAGASSARGUES;30260;30;GARD;1
30051;;Branoux-les-Taillades;BRANOUX-LES-TAILLADES;30110;30;GARD;1
30052;;Brau-et-Salagosse;BREAU-ET-SALAGOSSE;30120;30;GARD;1
30053;;Brignon;BRIGNON;30190;30;GARD;1
30054;;Brouzet-ls-Quissac;BROUZET-LES-QUISSAC;30260;30;GARD;1
30055;;Brouzet-ls-Als;BROUZET-LES-ALES;30580;30;GARD;1
30056;La;Bruguire;BRUGUIERE;30580;30;GARD;1
30057;;Cabrires;CABRIERES;30210;30;GARD;1
30058;La;Cadire-et-Cambo;CADIERE-ET-CAMBO;30170;30;GARD;1
30059;Le;Cailar;CAILAR;30740;30;GARD;1
30060;;Caissargues;CAISSARGUES;30132;30;GARD;1
30061;La;Calmette;CALMETTE;30190;30;GARD;1
30062;;Calvisson;CALVISSON;30420;30;GARD;1
30064;;Campestre-et-Luc;CAMPESTRE-ET-LUC;30770;30;GARD;1
30065;;Canaules-et-Argentires;CANAULES-ET-ARGENTIERES;30350;30;GARD;1
30066;;Cannes-et-Clairan;CANNES-ET-CLAIRAN;30260;30;GARD;1
30067;La;Capelle-et-Masmolne;CAPELLE-ET-MASMOLENE;30700;30;GARD;1
30068;;Cardet;CARDET;30350;30;GARD;1
30069;;Carnas;CARNAS;30260;30;GARD;1
30070;;Carsan;CARSAN;30130;30;GARD;1
30071;;Cassagnoles;CASSAGNOLES;30350;30;GARD;1
30072;;Castelnau-Valence;CASTELNAU-VALENCE;30190;30;GARD;1
30073;;Castillon-du-Gard;CASTILLON-DU-GARD;30210;30;GARD;1
30074;;Causse-Bgon;CAUSSE-BEGON;30750;30;GARD;1
30075;;Caveirac;CAVEIRAC;30820;30;GARD;1
30076;;Cavillargues;CAVILLARGUES;30330;30;GARD;1
30077;;Cendras;CENDRAS;30480;30;GARD;1
30079;;Chambon;CHAMBON;30450;30;GARD;1
30080;;Chamborigaud;CHAMBORIGAUD;30530;30;GARD;1
30081;;Chusclan;CHUSCLAN;30200;30;GARD;1
30082;;Clarensac;CLARENSAC;30870;30;GARD;1
30083;;Codognan;CODOGNAN;30920;30;GARD;1
30084;;Codolet;CODOLET;30200;30;GARD;1
30085;;Collias;COLLIAS;30210;30;GARD;1
30086;;Collorgues;COLLORGUES;30190;30;GARD;1
30087;;Colognac;COLOGNAC;30460;30;GARD;1
30088;;Combas;COMBAS;30250;30;GARD;1
30089;;Comps;COMPS;30300;30;GARD;1
30090;;Concoules;CONCOULES;30450;30;GARD;1
30091;;Congnies;CONGENIES;30111;30;GARD;1
30092;;Connaux;CONNAUX;30330;30;GARD;1
30093;;Conqueyrac;CONQUEYRAC;30170;30;GARD;1
30094;;Corbs;CORBES;30140;30;GARD;1
30095;;Corconne;CORCONNE;30260;30;GARD;1
30096;;Cornillon;CORNILLON;30630;30;GARD;1
30097;;Courry;COURRY;30500;30;GARD;1
30098;;Crespian;CRESPIAN;30260;30;GARD;1
30099;;Cros;CROS;30170;30;GARD;1
30100;;Cruviers-Lascours;CRUVIERS-LASCOURS;30360;30;GARD;1
30101;;Deaux;DEAUX;30360;30;GARD;1
30102;;Dions;DIONS;30190;30;GARD;1
30103;;Domazan;DOMAZAN;30390;30;GARD;1
30104;;Domessargues;DOMESSARGUES;30350;30;GARD;1
30105;;Dourbies;DOURBIES;30750;30;GARD;1
30106;;Durfort-et-Saint-Martin-de-Sossenac;DURFORT-ET-SAINT-MARTIN-DE-SOSSENAC;30170;30;GARD;1
30107;;Estzargues;ESTEZARGUES;30390;30;GARD;1
30108;L';Estrchure;ESTRECHURE;30124;30;GARD;1
30109;;Euzet;EUZET;30360;30;GARD;1
30110;;Flaux;FLAUX;30700;30;GARD;1
30111;;Foissac;FOISSAC;30700;30;GARD;1
30112;;Fons;FONS;30730;30;GARD;1
30113;;Fons-sur-Lussan;FONS-SUR-LUSSAN;30580;30;GARD;1
30114;;Fontans;FONTANES;30250;30;GARD;1
30115;;Fontarches;FONTARECHES;30580;30;GARD;1
30116;;Fourns;FOURNES;30210;30;GARD;1
30117;;Fourques;FOURQUES;30300;30;GARD;1
30119;;Fressac;FRESSAC;30170;30;GARD;1
30120;;Gagnires;GAGNIERES;30160;30;GARD;1
30121;;Gailhan;GAILHAN;30260;30;GARD;1
30122;;Gajan;GAJAN;30730;30;GARD;1
30123;;Gallargues-le-Montueux;GALLARGUES-LE-MONTUEUX;30660;30;GARD;1
30124;Le;Garn;GARN;30760;30;GARD;1
30125;;Garons;GARONS;30128;30;GARD;1
30126;;Garrigues-Sainte-Eulalie;GARRIGUES-SAINTE-EULALIE;30190;30;GARD;1
30127;;Gaujac;GAUJAC;30330;30;GARD;1
30128;;Gnrac;GENERAC;30510;30;GARD;1
30129;;Gnrargues;GENERARGUES;30140;30;GARD;1
30130;;Gnolhac;GENOLHAC;30450;30;GARD;1
30131;;Goudargues;GOUDARGUES;30630;30;GARD;1
30132;La;Grand-Combe;GRAND-COMBE;30110;30;GARD;1
30133;Le;Grau-du-Roi;GRAU-DU-ROI;30240;30;GARD;1
30134;;Issirac;ISSIRAC;30760;30;GARD;1
30135;;Jonquires-Saint-Vincent;JONQUIERES-SAINT-VINCENT;30300;30;GARD;1
30136;;Junas;JUNAS;30250;30;GARD;1
30137;;Lamelouze;LAMELOUZE;30110;30;GARD;1
30138;;Langlade;LANGLADE;30980;30;GARD;1
30139;;Lanujols;LANUEJOLS;30750;30;GARD;1
30140;;Lasalle;LASALLE;30460;30;GARD;1
30141;;Laudun-l'Ardoise;LAUDUN-L'ARDOISE;30290;30;GARD;1
30142;;Laval-Pradel;LAVAL-PRADEL;30110;30;GARD;1
30143;;Laval-Saint-Roman;LAVAL-SAINT-ROMAN;30760;30;GARD;1
30144;;Lecques;LECQUES;30250;30;GARD;1
30145;;Ldenon;LEDENON;30210;30;GARD;1
30146;;Ldignan;LEDIGNAN;30350;30;GARD;1
30147;;Lzan;LEZAN;30350;30;GARD;1
30148;;Liouc;LIOUC;30260;30;GARD;1
30149;;Lirac;LIRAC;30126;30;GARD;1
30150;;Logrian-Florian;LOGRIAN-FLORIAN;30610;30;GARD;1
30151;;Lussan;LUSSAN;30580;30;GARD;1
30152;Les;Mages;MAGES;30960;30;GARD;1
30153;;Malons-et-Elze;MALONS-ET-ELZE;30450;30;GARD;1
30154;;Mandagout;MANDAGOUT;30120;30;GARD;1
30155;;Manduel;MANDUEL;30129;30;GARD;1
30156;;Marguerittes;MARGUERITTES;30320;30;GARD;1
30157;;Mars;MARS;30120;30;GARD;1
30158;;Martignargues;MARTIGNARGUES;30360;30;GARD;1
30159;Le;Martinet;MARTINET;30960;30;GARD;1
30160;;Marujols-ls-Gardon;MARUEJOLS-LES-GARDON;30350;30;GARD;1
30161;;Massanes;MASSANES;30350;30;GARD;1
30162;;Massillargues-Attuech;MASSILLARGUES-ATTUECH;30140;30;GARD;1
30163;;Mauressargues;MAURESSARGUES;30350;30;GARD;1
30164;;Mjannes-le-Clap;MEJANNES-LE-CLAP;30430;30;GARD;1
30165;;Mjannes-ls-Als;MEJANNES-LES-ALES;30340;30;GARD;1
30166;;Meynes;MEYNES;30840;30;GARD;1
30167;;Meyrannes;MEYRANNES;30410;30;GARD;1
30168;;Mialet;MIALET;30140;30;GARD;1
30169;;Milhaud;MILHAUD;30540;30;GARD;1
30170;;Molires-Cavaillac;MOLIERES-CAVAILLAC;30120;30;GARD;1
30171;;Molires-sur-Cze;MOLIERES-SUR-CEZE;30410;30;GARD;1
30172;;Monoblet;MONOBLET;30170;30;GARD;1
30173;;Mons;MONS;30340;30;GARD;1
30174;;Montaren-et-Saint-Mdiers;MONTAREN-ET-SAINT-MEDIERS;30700;30;GARD;1
30175;;Montclus;MONTCLUS;30630;30;GARD;1
30176;;Montdardier;MONTDARDIER;30120;30;GARD;1
30177;;Monteils;MONTEILS;30360;30;GARD;1
30178;;Montfaucon;MONTFAUCON;30150;30;GARD;1
30179;;Montfrin;MONTFRIN;30490;30;GARD;1
30180;;Montignargues;MONTIGNARGUES;30190;30;GARD;1
30181;;Montmirat;MONTMIRAT;30260;30;GARD;1
30182;;Montpezat;MONTPEZAT;30730;30;GARD;1
30183;;Moulzan;MOULEZAN;30350;30;GARD;1
30184;;Moussac;MOUSSAC;30190;30;GARD;1
30185;;Mus;MUS;30121;30;GARD;1
30186;;Nages-et-Solorgues;NAGES-ET-SOLORGUES;30114;30;GARD;1
30187;;Navacelles;NAVACELLES;30580;30;GARD;1
30188;;Ners;NERS;30360;30;GARD;1
30189;;Nmes;NIMES;30000;30;GARD;1
30190;;Notre-Dame-de-la-Rouvire;NOTRE-DAME-DE-LA-ROUVIERE;30570;30;GARD;1
30191;;Orsan;ORSAN;30200;30;GARD;1
30192;;Orthoux-Srignac-Quilhan;ORTHOUX-SERIGNAC-QUILHAN;30260;30;GARD;1
30193;;Parignargues;PARIGNARGUES;30730;30;GARD;1
30194;;Peyremale;PEYREMALE;30160;30;GARD;1
30195;;Peyrolles;PEYROLLES;30124;30;GARD;1
30196;Le;Pin;PIN;30330;30;GARD;1
30197;Les;Plans;PLANS;30340;30;GARD;1
30198;Les;Plantiers;PLANTIERS;30122;30;GARD;1
30199;;Pommiers;POMMIERS;30120;30;GARD;1
30200;;Pompignan;POMPIGNAN;30170;30;GARD;1
30201;;Ponteils-et-Brsis;PONTEILS-ET-BRESIS;30450;30;GARD;1
30202;;Pont-Saint-Esprit;PONT-SAINT-ESPRIT;30130;30;GARD;1
30203;;Portes;PORTES;30530;30;GARD;1
30204;;Potelires;POTELIERES;30500;30;GARD;1
30205;;Pougnadoresse;POUGNADORESSE;30330;30;GARD;1
30206;;Poulx;POULX;30320;30;GARD;1
30207;;Pouzilhac;POUZILHAC;30210;30;GARD;1
30208;;Puechredon;PUECHREDON;30610;30;GARD;1
30209;;Pujaut;PUJAUT;30131;30;GARD;1
30210;;Quissac;QUISSAC;30260;30;GARD;1
30211;;Redessan;REDESSAN;30129;30;GARD;1
30212;;Remoulins;REMOULINS;30210;30;GARD;1
30213;;Revens;REVENS;30750;30;GARD;1
30214;;Ribaute-les-Tavernes;RIBAUTE-LES-TAVERNES;30720;30;GARD;1
30215;;Rivires;RIVIERES;30430;30;GARD;1
30216;;Robiac-Rochessadoule;ROBIAC-ROCHESSADOULE;30160;30;GARD;1
30217;;Rochefort-du-Gard;ROCHEFORT-DU-GARD;30650;30;GARD;1
30218;;Rochegude;ROCHEGUDE;30430;30;GARD;1
30219;;Rogues;ROGUES;30120;30;GARD;1
30220;;Roquedur;ROQUEDUR;30440;30;GARD;1
30221;;Roquemaure;ROQUEMAURE;30150;30;GARD;1
30222;La;Roque-sur-Cze;ROQUE-SUR-CEZE;30200;30;GARD;1
30223;;Rousson;ROUSSON;30340;30;GARD;1
30224;La;Rouvire;ROUVIERE;30190;30;GARD;1
30225;;Sabran;SABRAN;30200;30;GARD;1
30226;;Saint-Alexandre;SAINT-ALEXANDRE;30130;30;GARD;1
30227;;Saint-Ambroix;SAINT-AMBROIX;30500;30;GARD;1
30228;;Sainte-Anastasie;SAINTE-ANASTASIE;30190;30;GARD;1
30229;;Saint-Andr-de-Majencoules;SAINT-ANDRE-DE-MAJENCOULES;30570;30;GARD;1
30230;;Saint-Andr-de-Roquepertuis;SAINT-ANDRE-DE-ROQUEPERTUIS;30630;30;GARD;1
30231;;Saint-Andr-de-Valborgne;SAINT-ANDRE-DE-VALBORGNE;30940;30;GARD;1
30232;;Saint-Andr-d'Olrargues;SAINT-ANDRE-D'OLERARGUES;30330;30;GARD;1
30233;;Saint-Bauzly;SAINT-BAUZELY;30730;30;GARD;1
30234;;Saint-Bnzet;SAINT-BENEZET;30350;30;GARD;1
30235;;Saint-Bonnet-du-Gard;SAINT-BONNET-DU-GARD;30210;30;GARD;1
30236;;Saint-Bonnet-de-Salendrinque;SAINT-BONNET-DE-SALENDRINQUE;30460;30;GARD;1
30237;;Saint-Brs;SAINT-BRES;30500;30;GARD;1
30238;;Saint-Bresson;SAINT-BRESSON;30440;30;GARD;1
30239;;Sainte-Ccile-d'Andorge;SAINTE-CECILE-D'ANDORGE;30110;30;GARD;1
30240;;Saint-Csaire-de-Gauzignan;SAINT-CESAIRE-DE-GAUZIGNAN;30360;30;GARD;1
30241;;Saint-Chaptes;SAINT-CHAPTES;30190;30;GARD;1
30242;;Saint-Christol-de-Rodires;SAINT-CHRISTOL-DE-RODIERES;30760;30;GARD;1
30243;;Saint-Christol-ls-Als;SAINT-CHRISTOL-LES-ALES;30380;30;GARD;1
30244;;Saint-Clment;SAINT-CLEMENT;30260;30;GARD;1
30245;;Saint-Cme-et-Marujols;SAINT-COME-ET-MARUEJOLS;30870;30;GARD;1
30246;;Sainte-Croix-de-Caderle;SAINTE-CROIX-DE-CADERLE;30460;30;GARD;1
30247;;Saint-Denis;SAINT-DENIS;30500;30;GARD;1
30248;;Saint-Dzry;SAINT-DEZERY;30190;30;GARD;1
30249;;Saint-Dionisy;SAINT-DIONISY;30980;30;GARD;1
30250;;Saint-tienne-de-l'Olm;SAINT-ETIENNE-DE-L'OLM;30360;30;GARD;1
30251;;Saint-tienne-des-Sorts;SAINT-ETIENNE-DES-SORTS;30200;30;GARD;1
30252;;Saint-Flix-de-Pallires;SAINT-FELIX-DE-PALLIERES;30140;30;GARD;1
30253;;Saint-Florent-sur-Auzonnet;SAINT-FLORENT-SUR-AUZONNET;30960;30;GARD;1
30254;;Saint-Genis-de-Comolas;SAINT-GENIES-DE-COMOLAS;30150;30;GARD;1
30255;;Saint-Genis-de-Malgoirs;SAINT-GENIES-DE-MALGOIRES;30190;30;GARD;1
30256;;Saint-Gervais;SAINT-GERVAIS;30200;30;GARD;1
30257;;Saint-Gervasy;SAINT-GERVASY;30320;30;GARD;1
30258;;Saint-Gilles;SAINT-GILLES;30800;30;GARD;1
30259;;Saint-Hilaire-de-Brethmas;SAINT-HILAIRE-DE-BRETHMAS;30560;30;GARD;1
30260;;Saint-Hilaire-d'Ozilhan;SAINT-HILAIRE-D'OZILHAN;30210;30;GARD;1
30261;;Saint-Hippolyte-de-Caton;SAINT-HIPPOLYTE-DE-CATON;30360;30;GARD;1
30262;;Saint-Hippolyte-de-Montaigu;SAINT-HIPPOLYTE-DE-MONTAIGU;30700;30;GARD;1
30263;;Saint-Hippolyte-du-Fort;SAINT-HIPPOLYTE-DU-FORT;30170;30;GARD;1
30264;;Saint-Jean-de-Ceyrargues;SAINT-JEAN-DE-CEYRARGUES;30360;30;GARD;1
30265;;Saint-Jean-de-Crieulon;SAINT-JEAN-DE-CRIEULON;30610;30;GARD;1
30266;;Saint-Jean-de-Marujols-et-Avjan;SAINT-JEAN-DE-MARUEJOLS-ET-AVEJAN;30430;30;GARD;1
30267;;Saint-Jean-de-Serres;SAINT-JEAN-DE-SERRES;30350;30;GARD;1
30268;;Saint-Jean-de-Valriscle;SAINT-JEAN-DE-VALERISCLE;30960;30;GARD;1
30269;;Saint-Jean-du-Gard;SAINT-JEAN-DU-GARD;30270;30;GARD;1
30270;;Saint-Jean-du-Pin;SAINT-JEAN-DU-PIN;30140;30;GARD;1
30271;;Saint-Julien-de-Cassagnas;SAINT-JULIEN-DE-CASSAGNAS;30500;30;GARD;1
30272;;Saint-Julien-de-la-Nef;SAINT-JULIEN-DE-LA-NEF;30440;30;GARD;1
30273;;Saint-Julien-de-Peyrolas;SAINT-JULIEN-DE-PEYROLAS;30760;30;GARD;1
30274;;Saint-Julien-les-Rosiers;SAINT-JULIEN-LES-ROSIERS;30340;30;GARD;1
30275;;Saint-Just-et-Vacquires;SAINT-JUST-ET-VACQUIERES;30580;30;GARD;1
30276;;Saint-Laurent-d'Aigouze;SAINT-LAURENT-D'AIGOUZE;30220;30;GARD;1
30277;;Saint-Laurent-de-Carnols;SAINT-LAURENT-DE-CARNOLS;30200;30;GARD;1
30278;;Saint-Laurent-des-Arbres;SAINT-LAURENT-DES-ARBRES;30126;30;GARD;1
30279;;Saint-Laurent-la-Vernde;SAINT-LAURENT-LA-VERNEDE;30330;30;GARD;1
30280;;Saint-Laurent-le-Minier;SAINT-LAURENT-LE-MINIER;30440;30;GARD;1
30281;;Saint-Mamert-du-Gard;SAINT-MAMERT-DU-GARD;30730;30;GARD;1
30282;;Saint-Marcel-de-Careiret;SAINT-MARCEL-DE-CAREIRET;30330;30;GARD;1
30283;;Saint-Martial;SAINT-MARTIAL;30440;30;GARD;1
30284;;Saint-Martin-de-Valgalgues;SAINT-MARTIN-DE-VALGALGUES;30520;30;GARD;1
30285;;Saint-Maurice-de-Cazevieille;SAINT-MAURICE-DE-CAZEVIEILLE;30360;30;GARD;1
30286;;Saint-Maximin;SAINT-MAXIMIN;30700;30;GARD;1
30287;;Saint-Michel-d'Euzet;SAINT-MICHEL-D'EUZET;30200;30;GARD;1
30288;;Saint-Nazaire;SAINT-NAZAIRE;30200;30;GARD;1
30289;;Saint-Nazaire-des-Gardies;SAINT-NAZAIRE-DES-GARDIES;30610;30;GARD;1
30290;;Saint-Paulet-de-Caisson;SAINT-PAULET-DE-CAISSON;30130;30;GARD;1
30291;;Saint-Paul-la-Coste;SAINT-PAUL-LA-COSTE;30480;30;GARD;1
30292;;Saint-Pons-la-Calm;SAINT-PONS-LA-CALM;30330;30;GARD;1
30293;;Saint-Privat-de-Champclos;SAINT-PRIVAT-DE-CHAMPCLOS;30430;30;GARD;1
30294;;Saint-Privat-des-Vieux;SAINT-PRIVAT-DES-VIEUX;30340;30;GARD;1
30295;;Saint-Quentin-la-Poterie;SAINT-QUENTIN-LA-POTERIE;30700;30;GARD;1
30296;;Saint-Roman-de-Codires;SAINT-ROMAN-DE-CODIERES;30440;30;GARD;1
30297;;Saint-Sauveur-Camprieu;SAINT-SAUVEUR-CAMPRIEU;30750;30;GARD;1
30298;;Saint-Sbastien-d'Aigrefeuille;SAINT-SEBASTIEN-D'AIGREFEUILLE;30140;30;GARD;1
30299;;Saint-Siffret;SAINT-SIFFRET;30700;30;GARD;1
30300;;Saint-Thodorit;SAINT-THEODORIT;30260;30;GARD;1
30301;;Saint-Victor-des-Oules;SAINT-VICTOR-DES-OULES;30700;30;GARD;1
30302;;Saint-Victor-la-Coste;SAINT-VICTOR-LA-COSTE;30290;30;GARD;1
30303;;Saint-Victor-de-Malcap;SAINT-VICTOR-DE-MALCAP;30500;30;GARD;1
30304;;Salazac;SALAZAC;30760;30;GARD;1
30305;;Salindres;SALINDRES;30340;30;GARD;1
30306;;Salinelles;SALINELLES;30250;30;GARD;1
30307;Les;Salles-du-Gardon;SALLES-DU-GARDON;30110;30;GARD;1
30308;;Sanilhac-Sagris;SANILHAC-SAGRIES;30700;30;GARD;1
30309;;Sardan;SARDAN;30260;30;GARD;1
30310;;Saumane;SAUMANE;30125;30;GARD;1
30311;;Sauve;SAUVE;30610;30;GARD;1
30312;;Sauveterre;SAUVETERRE;30150;30;GARD;1
30313;;Sauzet;SAUZET;30190;30;GARD;1
30314;;Savignargues;SAVIGNARGUES;30350;30;GARD;1
30315;;Saze;SAZE;30650;30;GARD;1
30316;;Snchas;SENECHAS;30450;30;GARD;1
30317;;Sernhac;SERNHAC;30210;30;GARD;1
30318;;Servas;SERVAS;30340;30;GARD;1
30319;;Serviers-et-Labaume;SERVIERS-ET-LABAUME;30700;30;GARD;1
30320;;Seynes;SEYNES;30580;30;GARD;1
30321;;Sommires;SOMMIERES;30250;30;GARD;1
30322;;Soudorgues;SOUDORGUES;30460;30;GARD;1
30323;;Soustelle;SOUSTELLE;30110;30;GARD;1
30324;;Souvignargues;SOUVIGNARGUES;30250;30;GARD;1
30325;;Sumne;SUMENE;30440;30;GARD;1
30326;;Tavel;TAVEL;30126;30;GARD;1
30327;;Tharaux;THARAUX;30430;30;GARD;1
30328;;Thziers;THEZIERS;30390;30;GARD;1
30329;;Thoiras;THOIRAS;30140;30;GARD;1
30330;;Tornac;TORNAC;30140;30;GARD;1
30331;;Tresques;TRESQUES;30330;30;GARD;1
30332;;Trves;TREVES;30750;30;GARD;1
30333;;Uchaud;UCHAUD;30620;30;GARD;1
30334;;Uzs;UZES;30700;30;GARD;1
30335;;Vabres;VABRES;30460;30;GARD;1
30336;;Vallabrgues;VALLABREGUES;30300;30;GARD;1
30337;;Vallabrix;VALLABRIX;30700;30;GARD;1
30338;;Vallrargues;VALLERARGUES;30580;30;GARD;1
30339;;Valleraugue;VALLERAUGUE;30570;30;GARD;1
30340;;Valliguires;VALLIGUIERES;30210;30;GARD;1
30341;;Vauvert;VAUVERT;30600;30;GARD;1
30342;;Vnjan;VENEJAN;30200;30;GARD;1
30343;;Verfeuil;VERFEUIL;30630;30;GARD;1
30344;;Vergze;VERGEZE;30310;30;GARD;1
30345;La;Vernarde;VERNAREDE;30530;30;GARD;1
30346;;Vers-Pont-du-Gard;VERS-PONT-DU-GARD;30210;30;GARD;1
30347;;Vestric-et-Candiac;VESTRIC-ET-CANDIAC;30600;30;GARD;1
30348;;Vznobres;VEZENOBRES;30360;30;GARD;1
30349;;Vic-le-Fesq;VIC-LE-FESQ;30260;30;GARD;1
30350;Le;Vigan;VIGAN;30120;30;GARD;1
30351;;Villeneuve-ls-Avignon;VILLENEUVE-LES-AVIGNON;30400;30;GARD;1
30352;;Villevieille;VILLEVIEILLE;30250;30;GARD;1
30353;;Vissec;VISSEC;30770;30;GARD;1
30354;;Montagnac;MONTAGNAC;30350;30;GARD;1
30355;;Saint-Paul-les-Fonts;SAINT-PAUL-LES-FONTS;30330;30;GARD;1
30356;;Rodilhan;RODILHAN;30230;30;GARD;1
31001;;Agassac;AGASSAC;31230;31;HAUTE GARONNE;1
31002;;Aignes;AIGNES;31550;31;HAUTE GARONNE;1
31003;;Aigrefeuille;AIGREFEUILLE;31280;31;HAUTE GARONNE;1
31004;;Ayguesvives;AYGUESVIVES;31450;31;HAUTE GARONNE;1
31005;;Alan;ALAN;31420;31;HAUTE GARONNE;1
31006;;Albiac;ALBIAC;31460;31;HAUTE GARONNE;1
31007;;Ambax;AMBAX;31230;31;HAUTE GARONNE;1
31008;;Anan;ANAN;31230;31;HAUTE GARONNE;1
31009;;Antichan-de-Frontignes;ANTICHAN-DE-FRONTIGNES;31510;31;HAUTE GARONNE;1
31010;;Antignac;ANTIGNAC;31110;31;HAUTE GARONNE;1
31011;;Arbas;ARBAS;31160;31;HAUTE GARONNE;1
31012;;Arbon;ARBON;31160;31;HAUTE GARONNE;1
31013;;Ardige;ARDIEGE;31210;31;HAUTE GARONNE;1
31014;;Arguenos;ARGUENOS;31160;31;HAUTE GARONNE;1
31015;;Argut-Dessous;ARGUT-DESSOUS;31440;31;HAUTE GARONNE;1
31017;;Arlos;ARLOS;31440;31;HAUTE GARONNE;1
31018;;Arnaud-Guilhem;ARNAUD-GUILHEM;31360;31;HAUTE GARONNE;1
31019;;Artigue;ARTIGUE;31110;31;HAUTE GARONNE;1
31020;;Aspet;ASPET;31160;31;HAUTE GARONNE;1
31021;;Aspret-Sarrat;ASPRET-SARRAT;31800;31;HAUTE GARONNE;1
31022;;Aucamville;AUCAMVILLE;31140;31;HAUTE GARONNE;1
31023;;Aulon;AULON;31420;31;HAUTE GARONNE;1
31024;;Auragne;AURAGNE;31190;31;HAUTE GARONNE;1
31025;;Aureville;AUREVILLE;31320;31;HAUTE GARONNE;1
31026;;Auriac-sur-Vendinelle;AURIAC-SUR-VENDINELLE;31460;31;HAUTE GARONNE;1
31027;;Auribail;AURIBAIL;31190;31;HAUTE GARONNE;1
31028;;Aurignac;AURIGNAC;31420;31;HAUTE GARONNE;1
31029;;Aurin;AURIN;31570;31;HAUTE GARONNE;1
31030;;Ausseing;AUSSEING;31260;31;HAUTE GARONNE;1
31031;;Ausson;AUSSON;31210;31;HAUTE GARONNE;1
31032;;Aussonne;AUSSONNE;31840;31;HAUTE GARONNE;1
31033;;Auterive;AUTERIVE;31190;31;HAUTE GARONNE;1
31034;;Auzas;AUZAS;31360;31;HAUTE GARONNE;1
31035;;Auzeville-Tolosane;AUZEVILLE-TOLOSANE;31320;31;HAUTE GARONNE;1
31036;;Auzielle;AUZIELLE;31650;31;HAUTE GARONNE;1
31037;;Avignonet-Lauragais;AVIGNONET-LAURAGAIS;31290;31;HAUTE GARONNE;1
31038;;Azas;AZAS;31380;31;HAUTE GARONNE;1
31039;;Bachas;BACHAS;31420;31;HAUTE GARONNE;1
31040;;Bachos;BACHOS;31440;31;HAUTE GARONNE;1
31041;;Bagiry;BAGIRY;31510;31;HAUTE GARONNE;1
31042;;Bagnres-de-Luchon;BAGNERES-DE-LUCHON;31110;31;HAUTE GARONNE;1
31043;;Balesta;BALESTA;31580;31;HAUTE GARONNE;1
31044;;Balma;BALMA;31130;31;HAUTE GARONNE;1
31045;;Barbazan;BARBAZAN;31510;31;HAUTE GARONNE;1
31046;;Baren;BAREN;31440;31;HAUTE GARONNE;1
31047;;Bax;BAX;31310;31;HAUTE GARONNE;1
31048;;Bazige;BAZIEGE;31450;31;HAUTE GARONNE;1
31049;;Bazus;BAZUS;31380;31;HAUTE GARONNE;1
31050;;Beauchalot;BEAUCHALOT;31360;31;HAUTE GARONNE;1
31051;;Beaufort;BEAUFORT;31370;31;HAUTE GARONNE;1
31052;;Beaumont-sur-Lze;BEAUMONT-SUR-LEZE;31870;31;HAUTE GARONNE;1
31053;;Beaupuy;BEAUPUY;31850;31;HAUTE GARONNE;1
31054;;Beauteville;BEAUTEVILLE;31290;31;HAUTE GARONNE;1
31055;;Beauville;BEAUVILLE;31460;31;HAUTE GARONNE;1
31056;;Beauzelle;BEAUZELLE;31700;31;HAUTE GARONNE;1
31057;;Belberaud;BELBERAUD;31450;31;HAUTE GARONNE;1
31058;;Belbze-de-Lauragais;BELBEZE-DE-LAURAGAIS;31450;31;HAUTE GARONNE;1
31059;;Belbze-en-Comminges;BELBEZE-EN-COMMINGES;31260;31;HAUTE GARONNE;1
31060;;Blesta-en-Lauragais;BELESTA-EN-LAURAGAIS;31540;31;HAUTE GARONNE;1
31061;;Bellegarde-Sainte-Marie;BELLEGARDE-SAINTE-MARIE;31530;31;HAUTE GARONNE;1
31062;;Bellesserre;BELLESSERRE;31480;31;HAUTE GARONNE;1
31063;;Benque;BENQUE;31420;31;HAUTE GARONNE;1
31064;;Benque-Dessous-et-Dessus;BENQUE-DESSOUS-ET-DESSUS;31110;31;HAUTE GARONNE;1
31065;;Brat;BERAT;31370;31;HAUTE GARONNE;1
31066;;Bessires;BESSIERES;31660;31;HAUTE GARONNE;1
31067;;Bezins-Garraux;BEZINS-GARRAUX;31440;31;HAUTE GARONNE;1
31068;;Billire;BILLIERE;31110;31;HAUTE GARONNE;1
31069;;Blagnac;BLAGNAC;31700;31;HAUTE GARONNE;1
31070;;Blajan;BLAJAN;31350;31;HAUTE GARONNE;1
31071;;Bois-de-la-Pierre;BOIS-DE-LA-PIERRE;31390;31;HAUTE GARONNE;1
31072;;Boissde;BOISSEDE;31230;31;HAUTE GARONNE;1
31073;;Bondigoux;BONDIGOUX;31340;31;HAUTE GARONNE;1
31074;;Bonrepos-Riquet;BONREPOS-RIQUET;31590;31;HAUTE GARONNE;1
31075;;Bonrepos-sur-Aussonnelle;BONREPOS-SUR-AUSSONNELLE;31470;31;HAUTE GARONNE;1
31076;;Bordes-de-Rivire;BORDES-DE-RIVIERE;31210;31;HAUTE GARONNE;1
31077;Le;Born;BORN;31340;31;HAUTE GARONNE;1
31078;;Boudrac;BOUDRAC;31580;31;HAUTE GARONNE;1
31079;;Bouloc;BOULOC;31620;31;HAUTE GARONNE;1
31080;;Boulogne-sur-Gesse;BOULOGNE-SUR-GESSE;31350;31;HAUTE GARONNE;1
31081;;Bourg-d'Oueil;BOURG-D'OUEIL;31110;31;HAUTE GARONNE;1
31082;;Bourg-Saint-Bernard;BOURG-SAINT-BERNARD;31570;31;HAUTE GARONNE;1
31083;;Boussan;BOUSSAN;31420;31;HAUTE GARONNE;1
31084;;Boussens;BOUSSENS;31360;31;HAUTE GARONNE;1
31085;;Boutx;BOUTX;31440;31;HAUTE GARONNE;1
31086;;Bouzin;BOUZIN;31420;31;HAUTE GARONNE;1
31087;;Bragayrac;BRAGAYRAC;31470;31;HAUTE GARONNE;1
31088;;Brax;BRAX;31490;31;HAUTE GARONNE;1
31089;;Bretx;BRETX;31530;31;HAUTE GARONNE;1
31090;;Brignemont;BRIGNEMONT;31480;31;HAUTE GARONNE;1
31091;;Bruguires;BRUGUIERES;31150;31;HAUTE GARONNE;1
31092;;Burgalays;BURGALAYS;31440;31;HAUTE GARONNE;1
31093;Le;Burgaud;BURGAUD;31330;31;HAUTE GARONNE;1
31094;;Buzet-sur-Tarn;BUZET-SUR-TARN;31660;31;HAUTE GARONNE;1
31095;;Cabanac-Cazaux;CABANAC-CAZAUX;31160;31;HAUTE GARONNE;1
31096;;Cabanac-Sguenville;CABANAC-SEGUENVILLE;31480;31;HAUTE GARONNE;1
31097;Le;Cabanial;CABANIAL;31460;31;HAUTE GARONNE;1
31098;;Cadours;CADOURS;31480;31;HAUTE GARONNE;1
31099;;Caignac;CAIGNAC;31560;31;HAUTE GARONNE;1
31100;;Calmont;CALMONT;31560;31;HAUTE GARONNE;1
31101;;Cambernard;CAMBERNARD;31470;31;HAUTE GARONNE;1
31102;;Cambiac;CAMBIAC;31460;31;HAUTE GARONNE;1
31103;;Canens;CANENS;31310;31;HAUTE GARONNE;1
31104;;Capens;CAPENS;31410;31;HAUTE GARONNE;1
31105;;Caragoudes;CARAGOUDES;31460;31;HAUTE GARONNE;1
31106;;Caraman;CARAMAN;31460;31;HAUTE GARONNE;1
31107;;Carbonne;CARBONNE;31390;31;HAUTE GARONNE;1
31108;;Cardeilhac;CARDEILHAC;31350;31;HAUTE GARONNE;1
31109;;Cassagnabre-Tournas;CASSAGNABERE-TOURNAS;31420;31;HAUTE GARONNE;1
31110;;Cassagne;CASSAGNE;31260;31;HAUTE GARONNE;1
31111;;Castagnac;CASTAGNAC;31310;31;HAUTE GARONNE;1
31112;;Castagnde;CASTAGNEDE;31260;31;HAUTE GARONNE;1
31113;;Castanet-Tolosan;CASTANET-TOLOSAN;31320;31;HAUTE GARONNE;1
31114;;Castelbiague;CASTELBIAGUE;31160;31;HAUTE GARONNE;1
31115;;Castelgaillard;CASTELGAILLARD;31230;31;HAUTE GARONNE;1
31116;;Castelginest;CASTELGINEST;31780;31;HAUTE GARONNE;1
31117;;Castelmaurou;CASTELMAUROU;31180;31;HAUTE GARONNE;1
31118;;Castelnau-d'Estrtefonds;CASTELNAU-D'ESTRETEFONDS;31620;31;HAUTE GARONNE;1
31119;;Castelnau-Picampeau;CASTELNAU-PICAMPEAU;31430;31;HAUTE GARONNE;1
31120;Le;Castra;CASTERA;31530;31;HAUTE GARONNE;1
31121;;Castra-Vignoles;CASTERA-VIGNOLES;31350;31;HAUTE GARONNE;1
31122;;Casties-Labrande;CASTIES-LABRANDE;31430;31;HAUTE GARONNE;1
31123;;Castillon-de-Larboust;CASTILLON-DE-LARBOUST;31110;31;HAUTE GARONNE;1
31124;;Castillon-de-Saint-Martory;CASTILLON-DE-SAINT-MARTORY;31360;31;HAUTE GARONNE;1
31125;;Cathervielle;CATHERVIELLE;31110;31;HAUTE GARONNE;1
31126;;Caubiac;CAUBIAC;31480;31;HAUTE GARONNE;1
31127;;Caubous;CAUBOUS;31110;31;HAUTE GARONNE;1
31128;;Caujac;CAUJAC;31190;31;HAUTE GARONNE;1
31129;;Cazaril-Laspnes;CAZARIL-LASPENES;31110;31;HAUTE GARONNE;1
31130;;Cazaril-Tambours;CAZARIL-TAMBOURES;31580;31;HAUTE GARONNE;1
31131;;Cazaunous;CAZAUNOUS;31160;31;HAUTE GARONNE;1
31132;;Cazaux-Layrisse;CAZAUX-LAYRISSE;31440;31;HAUTE GARONNE;1
31133;;Cazeaux-de-Larboust;CAZEAUX-DE-LARBOUST;31110;31;HAUTE GARONNE;1
31134;;Cazeneuve-Montaut;CAZENEUVE-MONTAUT;31420;31;HAUTE GARONNE;1
31135;;Cazres;CAZERES;31220;31;HAUTE GARONNE;1
31136;;Cpet;CEPET;31620;31;HAUTE GARONNE;1
31137;;Cessales;CESSALES;31290;31;HAUTE GARONNE;1
31138;;Charlas;CHARLAS;31350;31;HAUTE GARONNE;1
31139;;Chaum;CHAUM;31440;31;HAUTE GARONNE;1
31140;;Chein-Dessus;CHEIN-DESSUS;31160;31;HAUTE GARONNE;1
31141;;Ciadoux;CIADOUX;31350;31;HAUTE GARONNE;1
31142;;Cier-de-Luchon;CIER-DE-LUCHON;31110;31;HAUTE GARONNE;1
31143;;Cier-de-Rivire;CIER-DE-RIVIERE;31510;31;HAUTE GARONNE;1
31144;;Cierp-Gaud;CIERP-GAUD;31440;31;HAUTE GARONNE;1
31145;;Cintegabelle;CINTEGABELLE;31550;31;HAUTE GARONNE;1
31146;;Cirs;CIRES;31110;31;HAUTE GARONNE;1
31147;;Clarac;CLARAC;31210;31;HAUTE GARONNE;1
31148;;Clermont-le-Fort;CLERMONT-LE-FORT;31810;31;HAUTE GARONNE;1
31149;;Colomiers;COLOMIERS;31770;31;HAUTE GARONNE;1
31150;;Cornebarrieu;CORNEBARRIEU;31700;31;HAUTE GARONNE;1
31151;;Corronsac;CORRONSAC;31450;31;HAUTE GARONNE;1
31152;;Coueilles;COUEILLES;31230;31;HAUTE GARONNE;1
31153;;Couladre;COULADERE;31220;31;HAUTE GARONNE;1
31155;;Couret;COURET;31160;31;HAUTE GARONNE;1
31156;;Cox;COX;31480;31;HAUTE GARONNE;1
31157;;Cugnaux;CUGNAUX;31270;31;HAUTE GARONNE;1
31158;;Cuguron;CUGURON;31210;31;HAUTE GARONNE;1
31159;Le;Cuing;CUING;31210;31;HAUTE GARONNE;1
31160;;Daux;DAUX;31700;31;HAUTE GARONNE;1
31161;;Deyme;DEYME;31450;31;HAUTE GARONNE;1
31162;;Donneville;DONNEVILLE;31450;31;HAUTE GARONNE;1
31163;;Drmil-Lafage;DREMIL-LAFAGE;31280;31;HAUTE GARONNE;1
31164;;Drudas;DRUDAS;31480;31;HAUTE GARONNE;1
31165;;Eaunes;EAUNES;31600;31;HAUTE GARONNE;1
31166;;Empeaux;EMPEAUX;31470;31;HAUTE GARONNE;1
31167;;Encausse-les-Thermes;ENCAUSSE-LES-THERMES;31160;31;HAUTE GARONNE;1
31168;;Eoux;EOUX;31420;31;HAUTE GARONNE;1
31169;;Escalquens;ESCALQUENS;31750;31;HAUTE GARONNE;1
31170;;Escanecrabe;ESCANECRABE;31350;31;HAUTE GARONNE;1
31171;;Espans;ESPANES;31450;31;HAUTE GARONNE;1
31172;;Esparron;ESPARRON;31420;31;HAUTE GARONNE;1
31173;;Esperce;ESPERCE;31190;31;HAUTE GARONNE;1
31174;;Estadens;ESTADENS;31160;31;HAUTE GARONNE;1
31175;;Estancarbon;ESTANCARBON;31800;31;HAUTE GARONNE;1
31176;;Estnos;ESTENOS;31440;31;HAUTE GARONNE;1
31177;;Eup;EUP;31440;31;HAUTE GARONNE;1
31178;;Fabas;FABAS;31230;31;HAUTE GARONNE;1
31179;Le;Faget;FAGET;31460;31;HAUTE GARONNE;1
31180;;Falga;FALGA;31540;31;HAUTE GARONNE;1
31181;Le;Fauga;FAUGA;31410;31;HAUTE GARONNE;1
31182;;Fenouillet;FENOUILLET;31150;31;HAUTE GARONNE;1
31183;;Figarol;FIGAROL;31260;31;HAUTE GARONNE;1
31184;;Flourens;FLOURENS;31130;31;HAUTE GARONNE;1
31185;;Folcarde;FOLCARDE;31290;31;HAUTE GARONNE;1
31186;;Fonbeauzard;FONBEAUZARD;31140;31;HAUTE GARONNE;1
31187;;Fonsorbes;FONSORBES;31470;31;HAUTE GARONNE;1
31188;;Fontenilles;FONTENILLES;31470;31;HAUTE GARONNE;1
31189;;Forgues;FORGUES;31370;31;HAUTE GARONNE;1
31190;;Fos;FOS;31440;31;HAUTE GARONNE;1
31191;;Fougaron;FOUGARON;31160;31;HAUTE GARONNE;1
31192;;Fourquevaux;FOURQUEVAUX;31450;31;HAUTE GARONNE;1
31193;Le;Fousseret;FOUSSERET;31430;31;HAUTE GARONNE;1
31194;;Francarville;FRANCARVILLE;31460;31;HAUTE GARONNE;1
31195;;Francazal;FRANCAZAL;31260;31;HAUTE GARONNE;1
31196;;Francon;FRANCON;31420;31;HAUTE GARONNE;1
31197;;Franquevielle;FRANQUEVIELLE;31210;31;HAUTE GARONNE;1
31198;Le;Frchet;FRECHET;31360;31;HAUTE GARONNE;1
31199;;Fronsac;FRONSAC;31440;31;HAUTE GARONNE;1
31200;;Frontignan-de-Comminges;FRONTIGNAN-DE-COMMINGES;31510;31;HAUTE GARONNE;1
31201;;Frontignan-Savs;FRONTIGNAN-SAVES;31230;31;HAUTE GARONNE;1
31202;;Fronton;FRONTON;31620;31;HAUTE GARONNE;1
31203;;Frouzins;FROUZINS;31270;31;HAUTE GARONNE;1
31204;;Fustignac;FUSTIGNAC;31430;31;HAUTE GARONNE;1
31205;;Gagnac-sur-Garonne;GAGNAC-SUR-GARONNE;31150;31;HAUTE GARONNE;1
31206;;Gaillac-Toulza;GAILLAC-TOULZA;31550;31;HAUTE GARONNE;1
31207;;Gali;GALIE;31510;31;HAUTE GARONNE;1
31208;;Ganties;GANTIES;31160;31;HAUTE GARONNE;1
31209;;Garac;GARAC;31480;31;HAUTE GARONNE;1
31210;;Gardouch;GARDOUCH;31290;31;HAUTE GARONNE;1
31211;;Gargas;GARGAS;31620;31;HAUTE GARONNE;1
31212;;Garidech;GARIDECH;31380;31;HAUTE GARONNE;1
31213;;Garin;GARIN;31110;31;HAUTE GARONNE;1
31215;;Gaur;GAURE;31590;31;HAUTE GARONNE;1
31216;;Gmil;GEMIL;31380;31;HAUTE GARONNE;1
31217;;Gnos;GENOS;31510;31;HAUTE GARONNE;1
31218;;Gensac-de-Boulogne;GENSAC-DE-BOULOGNE;31350;31;HAUTE GARONNE;1
31219;;Gensac-sur-Garonne;GENSAC-SUR-GARONNE;31310;31;HAUTE GARONNE;1
31220;;Gibel;GIBEL;31560;31;HAUTE GARONNE;1
31221;;Gouaux-de-Larboust;GOUAUX-DE-LARBOUST;31110;31;HAUTE GARONNE;1
31222;;Gouaux-de-Luchon;GOUAUX-DE-LUCHON;31110;31;HAUTE GARONNE;1
31223;;Goudex;GOUDEX;31230;31;HAUTE GARONNE;1
31224;;Gourdan-Polignan;GOURDAN-POLIGNAN;31210;31;HAUTE GARONNE;1
31225;;Goutevernisse;GOUTEVERNISSE;31310;31;HAUTE GARONNE;1
31226;;Gouzens;GOUZENS;31310;31;HAUTE GARONNE;1
31227;;Goyrans;GOYRANS;31120;31;HAUTE GARONNE;1
31228;;Gragnague;GRAGNAGUE;31380;31;HAUTE GARONNE;1
31229;;Gratens;GRATENS;31430;31;HAUTE GARONNE;1
31230;;Gratentour;GRATENTOUR;31150;31;HAUTE GARONNE;1
31231;;Grazac;GRAZAC;31190;31;HAUTE GARONNE;1
31232;;Grenade;GRENADE;31330;31;HAUTE GARONNE;1
31233;;Grpiac;GREPIAC;31190;31;HAUTE GARONNE;1
31234;Le;Grs;GRES;31480;31;HAUTE GARONNE;1
31235;;Guran;GURAN;31440;31;HAUTE GARONNE;1
31236;;Herran;HERRAN;31160;31;HAUTE GARONNE;1
31237;;His;HIS;31260;31;HAUTE GARONNE;1
31238;;Huos;HUOS;31210;31;HAUTE GARONNE;1
31239;L';Isle-en-Dodon;ISLE-EN-DODON;31230;31;HAUTE GARONNE;1
31240;;Issus;ISSUS;31450;31;HAUTE GARONNE;1
31241;;Izaut-de-l'Htel;IZAUT-DE-L'HOTEL;31160;31;HAUTE GARONNE;1
31242;;Jurvielle;JURVIELLE;31110;31;HAUTE GARONNE;1
31243;;Juzes;JUZES;31540;31;HAUTE GARONNE;1
31244;;Juzet-de-Luchon;JUZET-DE-LUCHON;31110;31;HAUTE GARONNE;1
31245;;Juzet-d'Izaut;JUZET-D'IZAUT;31160;31;HAUTE GARONNE;1
31246;;Labarthe-Inard;LABARTHE-INARD;31800;31;HAUTE GARONNE;1
31247;;Labarthe-Rivire;LABARTHE-RIVIERE;31800;31;HAUTE GARONNE;1
31248;;Labarthe-sur-Lze;LABARTHE-SUR-LEZE;31860;31;HAUTE GARONNE;1
31249;;Labastide-Beauvoir;LABASTIDE-BEAUVOIR;31450;31;HAUTE GARONNE;1
31250;;Labastide-Clermont;LABASTIDE-CLERMONT;31370;31;HAUTE GARONNE;1
31251;;Labastide-Paums;LABASTIDE-PAUMES;31230;31;HAUTE GARONNE;1
31252;;Labastide-Saint-Sernin;LABASTIDE-SAINT-SERNIN;31620;31;HAUTE GARONNE;1
31253;;Labastidette;LABASTIDETTE;31600;31;HAUTE GARONNE;1
31254;;Labge;LABEGE;31670;31;HAUTE GARONNE;1
31255;;Labroqure;LABROQUERE;31510;31;HAUTE GARONNE;1
31256;;Labruyre-Dorsa;LABRUYERE-DORSA;31190;31;HAUTE GARONNE;1
31258;;Lacaugne;LACAUGNE;31390;31;HAUTE GARONNE;1
31259;;Lacroix-Falgarde;LACROIX-FALGARDE;31120;31;HAUTE GARONNE;1
31260;;Laffite-Toupire;LAFFITE-TOUPIERE;31360;31;HAUTE GARONNE;1
31261;;Lafitte-Vigordane;LAFITTE-VIGORDANE;31390;31;HAUTE GARONNE;1
31262;;Lagarde;LAGARDE;31290;31;HAUTE GARONNE;1
31263;;Lagardelle-sur-Lze;LAGARDELLE-SUR-LEZE;31870;31;HAUTE GARONNE;1
31264;;Lagrce-Dieu;LAGRACE-DIEU;31190;31;HAUTE GARONNE;1
31265;;Lagraulet-Saint-Nicolas;LAGRAULET-SAINT-NICOLAS;31480;31;HAUTE GARONNE;1
31266;;Lahage;LAHAGE;31370;31;HAUTE GARONNE;1
31267;;Lahitre;LAHITERE;31310;31;HAUTE GARONNE;1
31268;;Lalouret-Laffiteau;LALOURET-LAFFITEAU;31800;31;HAUTE GARONNE;1
31269;;Lamasqure;LAMASQUERE;31600;31;HAUTE GARONNE;1
31270;;Landorthe;LANDORTHE;31800;31;HAUTE GARONNE;1
31271;;Lanta;LANTA;31570;31;HAUTE GARONNE;1
31272;;Lapeyrre;LAPEYRERE;31310;31;HAUTE GARONNE;1
31273;;Lapeyrouse-Fossat;LAPEYROUSE-FOSSAT;31180;31;HAUTE GARONNE;1
31274;;Larcan;LARCAN;31800;31;HAUTE GARONNE;1
31275;;Larole;LAREOLE;31480;31;HAUTE GARONNE;1
31276;;Larroque;LARROQUE;31580;31;HAUTE GARONNE;1
31277;;Lasserre;LASSERRE;31530;31;HAUTE GARONNE;1
31278;;Latoue;LATOUE;31800;31;HAUTE GARONNE;1
31279;;Latour;LATOUR;31310;31;HAUTE GARONNE;1
31280;;Latrape;LATRAPE;31310;31;HAUTE GARONNE;1
31281;;Launac;LAUNAC;31330;31;HAUTE GARONNE;1
31282;;Launaguet;LAUNAGUET;31140;31;HAUTE GARONNE;1
31283;;Lautignac;LAUTIGNAC;31370;31;HAUTE GARONNE;1
31284;;Lauzerville;LAUZERVILLE;31650;31;HAUTE GARONNE;1
31285;;Lavalette;LAVALETTE;31590;31;HAUTE GARONNE;1
31286;;Lavelanet-de-Comminges;LAVELANET-DE-COMMINGES;31220;31;HAUTE GARONNE;1
31287;;Lavernose-Lacasse;LAVERNOSE-LACASSE;31410;31;HAUTE GARONNE;1
31288;;Layrac-sur-Tarn;LAYRAC-SUR-TARN;31340;31;HAUTE GARONNE;1
31289;;Lcussan;LECUSSAN;31580;31;HAUTE GARONNE;1
31290;;Lge;LEGE;31440;31;HAUTE GARONNE;1
31291;;Lguevin;LEGUEVIN;31490;31;HAUTE GARONNE;1
31292;;Lescuns;LESCUNS;31220;31;HAUTE GARONNE;1
31293;;Lespinasse;LESPINASSE;31150;31;HAUTE GARONNE;1
31294;;Lespiteau;LESPITEAU;31160;31;HAUTE GARONNE;1
31295;;Lespugue;LESPUGUE;31350;31;HAUTE GARONNE;1
31296;;Lestelle-de-Saint-Martory;LESTELLE-DE-SAINT-MARTORY;31360;31;HAUTE GARONNE;1
31297;;Lvignac;LEVIGNAC;31530;31;HAUTE GARONNE;1
31298;;Lez;LEZ;31440;31;HAUTE GARONNE;1
31299;;Lherm;LHERM;31600;31;HAUTE GARONNE;1
31300;;Lieoux;LIEOUX;31800;31;HAUTE GARONNE;1
31301;;Lilhac;LILHAC;31230;31;HAUTE GARONNE;1
31302;;Lodes;LODES;31800;31;HAUTE GARONNE;1
31303;;Longages;LONGAGES;31410;31;HAUTE GARONNE;1
31304;;Loubens-Lauragais;LOUBENS-LAURAGAIS;31460;31;HAUTE GARONNE;1
31305;;Loudet;LOUDET;31580;31;HAUTE GARONNE;1
31306;;Lourde;LOURDE;31510;31;HAUTE GARONNE;1
31307;;Lunax;LUNAX;31350;31;HAUTE GARONNE;1
31308;;Luscan;LUSCAN;31510;31;HAUTE GARONNE;1
31309;;Lussan-Adeilhac;LUSSAN-ADEILHAC;31430;31;HAUTE GARONNE;1
31310;;Lux;LUX;31290;31;HAUTE GARONNE;1
31311;La;Magdelaine-sur-Tarn;MAGDELAINE-SUR-TARN;31340;31;HAUTE GARONNE;1
31312;;Mailholas;MAILHOLAS;31310;31;HAUTE GARONNE;1
31313;;Malvezie;MALVEZIE;31510;31;HAUTE GARONNE;1
31314;;Mancioux;MANCIOUX;31360;31;HAUTE GARONNE;1
31315;;Mane;MANE;31260;31;HAUTE GARONNE;1
31316;;Marignac;MARIGNAC;31440;31;HAUTE GARONNE;1
31317;;Marignac-Lasclares;MARIGNAC-LASCLARES;31430;31;HAUTE GARONNE;1
31318;;Marignac-Laspeyres;MARIGNAC-LASPEYRES;31220;31;HAUTE GARONNE;1
31319;;Marliac;MARLIAC;31550;31;HAUTE GARONNE;1
31320;;Marquefave;MARQUEFAVE;31390;31;HAUTE GARONNE;1
31321;;Marsoulas;MARSOULAS;31260;31;HAUTE GARONNE;1
31322;;Martisserre;MARTISSERRE;31230;31;HAUTE GARONNE;1
31323;;Martres-de-Rivire;MARTRES-DE-RIVIERE;31210;31;HAUTE GARONNE;1
31324;;Martres-Tolosane;MARTRES-TOLOSANE;31220;31;HAUTE GARONNE;1
31325;;Mascarville;MASCARVILLE;31460;31;HAUTE GARONNE;1
31326;;Massabrac;MASSABRAC;31310;31;HAUTE GARONNE;1
31327;;Mauran;MAURAN;31220;31;HAUTE GARONNE;1
31328;;Mauremont;MAUREMONT;31290;31;HAUTE GARONNE;1
31329;;Maurens;MAURENS;31540;31;HAUTE GARONNE;1
31330;;Mauressac;MAURESSAC;31190;31;HAUTE GARONNE;1
31331;;Maureville;MAUREVILLE;31460;31;HAUTE GARONNE;1
31332;;Mauvaisin;MAUVAISIN;31190;31;HAUTE GARONNE;1
31333;;Mauvezin;MAUVEZIN;31230;31;HAUTE GARONNE;1
31334;;Mauzac;MAUZAC;31410;31;HAUTE GARONNE;1
31335;;Mayrgne;MAYREGNE;31110;31;HAUTE GARONNE;1
31336;;Mazres-sur-Salat;MAZERES-SUR-SALAT;31260;31;HAUTE GARONNE;1
31337;;Melles;MELLES;31440;31;HAUTE GARONNE;1
31338;;Menville;MENVILLE;31530;31;HAUTE GARONNE;1
31339;;Mrenvielle;MERENVIELLE;31530;31;HAUTE GARONNE;1
31340;;Mervilla;MERVILLA;31320;31;HAUTE GARONNE;1
31341;;Merville;MERVILLE;31330;31;HAUTE GARONNE;1
31342;;Milhas;MILHAS;31160;31;HAUTE GARONNE;1
31343;;Mirambeau;MIRAMBEAU;31230;31;HAUTE GARONNE;1
31344;;Miramont-de-Comminges;MIRAMONT-DE-COMMINGES;31800;31;HAUTE GARONNE;1
31345;;Miremont;MIREMONT;31190;31;HAUTE GARONNE;1
31346;;Mirepoix-sur-Tarn;MIREPOIX-SUR-TARN;31340;31;HAUTE GARONNE;1
31347;;Molas;MOLAS;31230;31;HAUTE GARONNE;1
31348;;Moncaup;MONCAUP;31160;31;HAUTE GARONNE;1
31349;;Mondavezan;MONDAVEZAN;31220;31;HAUTE GARONNE;1
31350;;Mondilhan;MONDILHAN;31350;31;HAUTE GARONNE;1
31351;;Mondonville;MONDONVILLE;31700;31;HAUTE GARONNE;1
31352;;Mondouzil;MONDOUZIL;31850;31;HAUTE GARONNE;1
31353;;Mons;MONES;31370;31;HAUTE GARONNE;1
31354;;Monestrol;MONESTROL;31560;31;HAUTE GARONNE;1
31355;;Mons;MONS;31280;31;HAUTE GARONNE;1
31356;;Montaigut-sur-Save;MONTAIGUT-SUR-SAVE;31530;31;HAUTE GARONNE;1
31357;;Montastruc-de-Salies;MONTASTRUC-DE-SALIES;31160;31;HAUTE GARONNE;1
31358;;Montastruc-la-Conseillre;MONTASTRUC-LA-CONSEILLERE;31380;31;HAUTE GARONNE;1
31359;;Montastruc-Savs;MONTASTRUC-SAVES;31370;31;HAUTE GARONNE;1
31360;;Montauban-de-Luchon;MONTAUBAN-DE-LUCHON;31110;31;HAUTE GARONNE;1
31361;;Montaut;MONTAUT;31410;31;HAUTE GARONNE;1
31362;;Montberaud;MONTBERAUD;31220;31;HAUTE GARONNE;1
31363;;Montbernard;MONTBERNARD;31230;31;HAUTE GARONNE;1
31364;;Montberon;MONTBERON;31140;31;HAUTE GARONNE;1
31365;;Montbrun-Bocage;MONTBRUN-BOCAGE;31310;31;HAUTE GARONNE;1
31366;;Montbrun-Lauragais;MONTBRUN-LAURAGAIS;31450;31;HAUTE GARONNE;1
31367;;Montclar-de-Comminges;MONTCLAR-DE-COMMINGES;31220;31;HAUTE GARONNE;1
31368;;Montclar-Lauragais;MONTCLAR-LAURAGAIS;31290;31;HAUTE GARONNE;1
31369;;Mont-de-Gali;MONT-DE-GALIE;31510;31;HAUTE GARONNE;1
31370;;Montgut-Bourjac;MONTEGUT-BOURJAC;31430;31;HAUTE GARONNE;1
31371;;Montgut-Lauragais;MONTEGUT-LAURAGAIS;31540;31;HAUTE GARONNE;1
31372;;Montespan;MONTESPAN;31260;31;HAUTE GARONNE;1
31373;;Montesquieu-Guittaut;MONTESQUIEU-GUITTAUT;31230;31;HAUTE GARONNE;1
31374;;Montesquieu-Lauragais;MONTESQUIEU-LAURAGAIS;31450;31;HAUTE GARONNE;1
31375;;Montesquieu-Volvestre;MONTESQUIEU-VOLVESTRE;31310;31;HAUTE GARONNE;1
31376;;Montgaillard-de-Salies;MONTGAILLARD-DE-SALIES;31260;31;HAUTE GARONNE;1
31377;;Montgaillard-Lauragais;MONTGAILLARD-LAURAGAIS;31290;31;HAUTE GARONNE;1
31378;;Montgaillard-sur-Save;MONTGAILLARD-SUR-SAVE;31350;31;HAUTE GARONNE;1
31379;;Montgazin;MONTGAZIN;31410;31;HAUTE GARONNE;1
31380;;Montgeard;MONTGEARD;31560;31;HAUTE GARONNE;1
31381;;Montgiscard;MONTGISCARD;31450;31;HAUTE GARONNE;1
31382;;Montgras;MONTGRAS;31370;31;HAUTE GARONNE;1
31383;;Montjoire;MONTJOIRE;31380;31;HAUTE GARONNE;1
31384;;Montlaur;MONTLAUR;31450;31;HAUTE GARONNE;1
31385;;Montmaurin;MONTMAURIN;31350;31;HAUTE GARONNE;1
31386;;Montoulieu-Saint-Bernard;MONTOULIEU-SAINT-BERNARD;31420;31;HAUTE GARONNE;1
31387;;Montoussin;MONTOUSSIN;31430;31;HAUTE GARONNE;1
31388;;Montpitol;MONTPITOL;31380;31;HAUTE GARONNE;1
31389;;Montrab;MONTRABE;31850;31;HAUTE GARONNE;1
31390;;Montrjeau;MONTREJEAU;31210;31;HAUTE GARONNE;1
31391;;Montsauns;MONTSAUNES;31260;31;HAUTE GARONNE;1
31392;;Mourvilles-Basses;MOURVILLES-BASSES;31460;31;HAUTE GARONNE;1
31393;;Mourvilles-Hautes;MOURVILLES-HAUTES;31540;31;HAUTE GARONNE;1
31394;;Moustajon;MOUSTAJON;31110;31;HAUTE GARONNE;1
31395;;Muret;MURET;31600;31;HAUTE GARONNE;1
31396;;Nailloux;NAILLOUX;31560;31;HAUTE GARONNE;1
31397;;Nnigan;NENIGAN;31350;31;HAUTE GARONNE;1
31398;;Nizan-Gesse;NIZAN-GESSE;31350;31;HAUTE GARONNE;1
31399;;No;NOE;31410;31;HAUTE GARONNE;1
31400;;Nogaret;NOGARET;31540;31;HAUTE GARONNE;1
31401;;Noueilles;NOUEILLES;31450;31;HAUTE GARONNE;1
31402;;Odars;ODARS;31450;31;HAUTE GARONNE;1
31403;;Ondes;ONDES;31330;31;HAUTE GARONNE;1
31404;;O;OO;31110;31;HAUTE GARONNE;1
31405;;Ore;ORE;31510;31;HAUTE GARONNE;1
31406;;Palaminy;PALAMINY;31220;31;HAUTE GARONNE;1
31407;;Paulhac;PAULHAC;31380;31;HAUTE GARONNE;1
31408;;Payssous;PAYSSOUS;31510;31;HAUTE GARONNE;1
31409;;Pchabou;PECHABOU;31320;31;HAUTE GARONNE;1
31410;;Pechbonnieu;PECHBONNIEU;31140;31;HAUTE GARONNE;1
31411;;Pechbusque;PECHBUSQUE;31320;31;HAUTE GARONNE;1
31412;;Pguilhan;PEGUILHAN;31350;31;HAUTE GARONNE;1
31413;;Pelleport;PELLEPORT;31480;31;HAUTE GARONNE;1
31414;;Peyrissas;PEYRISSAS;31420;31;HAUTE GARONNE;1
31415;;Peyrouzet;PEYROUZET;31420;31;HAUTE GARONNE;1
31416;;Peyssies;PEYSSIES;31390;31;HAUTE GARONNE;1
31417;;Pibrac;PIBRAC;31820;31;HAUTE GARONNE;1
31418;;Pin-Balma;PIN-BALMA;31130;31;HAUTE GARONNE;1
31419;Le;Pin-Murelet;PIN-MURELET;31370;31;HAUTE GARONNE;1
31420;;Pinsaguel;PINSAGUEL;31120;31;HAUTE GARONNE;1
31421;;Pins-Justaret;PINS-JUSTARET;31860;31;HAUTE GARONNE;1
31422;;Plagne;PLAGNE;31220;31;HAUTE GARONNE;1
31423;;Plagnole;PLAGNOLE;31370;31;HAUTE GARONNE;1
31424;;Plaisance-du-Touch;PLAISANCE-DU-TOUCH;31830;31;HAUTE GARONNE;1
31425;Le;Plan;PLAN;31220;31;HAUTE GARONNE;1
31426;;Pointis-de-Rivire;POINTIS-DE-RIVIERE;31210;31;HAUTE GARONNE;1
31427;;Pointis-Inard;POINTIS-INARD;31800;31;HAUTE GARONNE;1
31428;;Polastron;POLASTRON;31430;31;HAUTE GARONNE;1
31429;;Pompertuzat;POMPERTUZAT;31450;31;HAUTE GARONNE;1
31430;;Ponlat-Taillebourg;PONLAT-TAILLEBOURG;31210;31;HAUTE GARONNE;1
31431;;Portet-d'Aspet;PORTET-D'ASPET;31160;31;HAUTE GARONNE;1
31432;;Portet-de-Luchon;PORTET-DE-LUCHON;31110;31;HAUTE GARONNE;1
31433;;Portet-sur-Garonne;PORTET-SUR-GARONNE;31120;31;HAUTE GARONNE;1
31434;;Poubeau;POUBEAU;31110;31;HAUTE GARONNE;1
31435;;Poucharramet;POUCHARRAMET;31370;31;HAUTE GARONNE;1
31436;;Pouy-de-Touges;POUY-DE-TOUGES;31430;31;HAUTE GARONNE;1
31437;;Pouze;POUZE;31450;31;HAUTE GARONNE;1
31438;;Pradre-les-Bourguets;PRADERE-LES-BOURGUETS;31530;31;HAUTE GARONNE;1
31439;;Prserville;PRESERVILLE;31570;31;HAUTE GARONNE;1
31440;;Proupiary;PROUPIARY;31360;31;HAUTE GARONNE;1
31441;;Prunet;PRUNET;31460;31;HAUTE GARONNE;1
31442;;Puydaniel;PUYDANIEL;31190;31;HAUTE GARONNE;1
31443;;Puymaurin;PUYMAURIN;31230;31;HAUTE GARONNE;1
31444;;Puyssgur;PUYSSEGUR;31480;31;HAUTE GARONNE;1
31445;;Quint-Fonsegrives;QUINT-FONSEGRIVES;31130;31;HAUTE GARONNE;1
31446;;Ramonville-Saint-Agne;RAMONVILLE-SAINT-AGNE;31520;31;HAUTE GARONNE;1
31447;;Razecueill;RAZECUEILLE;31160;31;HAUTE GARONNE;1
31448;;Rebigue;REBIGUE;31320;31;HAUTE GARONNE;1
31449;;Rgades;REGADES;31800;31;HAUTE GARONNE;1
31450;;Renneville;RENNEVILLE;31290;31;HAUTE GARONNE;1
31451;;Revel;REVEL;31250;31;HAUTE GARONNE;1
31452;;Rieucaz;RIEUCAZE;31800;31;HAUTE GARONNE;1
31453;;Rieumajou;RIEUMAJOU;31290;31;HAUTE GARONNE;1
31454;;Rieumes;RIEUMES;31370;31;HAUTE GARONNE;1
31455;;Rieux-Volvestre;RIEUX-VOLVESTRE;31310;31;HAUTE GARONNE;1
31456;;Riolas;RIOLAS;31230;31;HAUTE GARONNE;1
31457;;Roquefort-sur-Garonne;ROQUEFORT-SUR-GARONNE;31360;31;HAUTE GARONNE;1
31458;;Roques;ROQUES;31120;31;HAUTE GARONNE;1
31459;;Roquesrire;ROQUESERIERE;31380;31;HAUTE GARONNE;1
31460;;Roquettes;ROQUETTES;31120;31;HAUTE GARONNE;1
31461;;Roude;ROUEDE;31160;31;HAUTE GARONNE;1
31462;;Rouffiac-Tolosan;ROUFFIAC-TOLOSAN;31180;31;HAUTE GARONNE;1
31463;;Roumens;ROUMENS;31540;31;HAUTE GARONNE;1
31464;;Sabonnres;SABONNERES;31370;31;HAUTE GARONNE;1
31465;;Saccourvielle;SACCOURVIELLE;31110;31;HAUTE GARONNE;1
31466;;Saigude;SAIGUEDE;31470;31;HAUTE GARONNE;1
31467;;Saint-Alban;SAINT-ALBAN;31140;31;HAUTE GARONNE;1
31468;;Saint-Andr;SAINT-ANDRE;31420;31;HAUTE GARONNE;1
31469;;Saint-Araille;SAINT-ARAILLE;31430;31;HAUTE GARONNE;1
31470;;Saint-Aventin;SAINT-AVENTIN;31110;31;HAUTE GARONNE;1
31471;;Saint-Bat;SAINT-BEAT;31440;31;HAUTE GARONNE;1
31472;;Saint-Bertrand-de-Comminges;SAINT-BERTRAND-DE-COMMINGES;31510;31;HAUTE GARONNE;1
31473;;Saint-Czert;SAINT-CEZERT;31330;31;HAUTE GARONNE;1
31474;;Saint-Christaud;SAINT-CHRISTAUD;31310;31;HAUTE GARONNE;1
31475;;Saint-Clar-de-Rivire;SAINT-CLAR-DE-RIVIERE;31600;31;HAUTE GARONNE;1
31476;;Saint-lix-le-Chteau;SAINT-ELIX-LE-CHATEAU;31430;31;HAUTE GARONNE;1
31477;;Saint-lix-Sglan;SAINT-ELIX-SEGLAN;31420;31;HAUTE GARONNE;1
31478;;Saint-Flix-Lauragais;SAINT-FELIX-LAURAGAIS;31540;31;HAUTE GARONNE;1
31479;;Saint-Ferrol-de-Comminges;SAINT-FERREOL-DE-COMMINGES;31350;31;HAUTE GARONNE;1
31480;;Sainte-Foy-d'Aigrefeuille;SAINTE-FOY-D'AIGREFEUILLE;31570;31;HAUTE GARONNE;1
31481;;Sainte-Foy-de-Peyrolires;SAINTE-FOY-DE-PEYROLIERES;31470;31;HAUTE GARONNE;1
31482;;Saint-Frajou;SAINT-FRAJOU;31230;31;HAUTE GARONNE;1
31483;;Saint-Gaudens;SAINT-GAUDENS;31800;31;HAUTE GARONNE;1
31484;;Saint-Genis-Bellevue;SAINT-GENIES-BELLEVUE;31180;31;HAUTE GARONNE;1
31485;;Saint-Germier;SAINT-GERMIER;31290;31;HAUTE GARONNE;1
31486;;Saint-Hilaire;SAINT-HILAIRE;31410;31;HAUTE GARONNE;1
31487;;Saint-Ignan;SAINT-IGNAN;31800;31;HAUTE GARONNE;1
31488;;Saint-Jean;SAINT-JEAN;31240;31;HAUTE GARONNE;1
31489;;Saint-Jean-Lherm;SAINT-JEAN-LHERM;31380;31;HAUTE GARONNE;1
31490;;Saint-Jory;SAINT-JORY;31790;31;HAUTE GARONNE;1
31491;;Saint-Julia;SAINT-JULIA;31540;31;HAUTE GARONNE;1
31492;;Saint-Julien-sur-Garonne;SAINT-JULIEN-SUR-GARONNE;31220;31;HAUTE GARONNE;1
31493;;Saint-Lary-Boujean;SAINT-LARY-BOUJEAN;31350;31;HAUTE GARONNE;1
31494;;Saint-Laurent;SAINT-LAURENT;31230;31;HAUTE GARONNE;1
31495;;Saint-Lon;SAINT-LEON;31560;31;HAUTE GARONNE;1
31496;;Sainte-Livrade;SAINTE-LIVRADE;31530;31;HAUTE GARONNE;1
31497;;Saint-Loup-Cammas;SAINT-LOUP-CAMMAS;31140;31;HAUTE GARONNE;1
31498;;Saint-Loup-en-Comminges;SAINT-LOUP-EN-COMMINGES;31350;31;HAUTE GARONNE;1
31499;;Saint-Lys;SAINT-LYS;31470;31;HAUTE GARONNE;1
31500;;Saint-Mamet;SAINT-MAMET;31110;31;HAUTE GARONNE;1
31501;;Saint-Marcel-Paulel;SAINT-MARCEL-PAULEL;31590;31;HAUTE GARONNE;1
31502;;Saint-Marcet;SAINT-MARCET;31800;31;HAUTE GARONNE;1
31503;;Saint-Martory;SAINT-MARTORY;31360;31;HAUTE GARONNE;1
31504;;Saint-Mdard;SAINT-MEDARD;31360;31;HAUTE GARONNE;1
31505;;Saint-Michel;SAINT-MICHEL;31220;31;HAUTE GARONNE;1
31506;;Saint-Orens-de-Gameville;SAINT-ORENS-DE-GAMEVILLE;31650;31;HAUTE GARONNE;1
31507;;Saint-Paul-sur-Save;SAINT-PAUL-SUR-SAVE;31530;31;HAUTE GARONNE;1
31508;;Saint-Paul-d'Oueil;SAINT-PAUL-D'OUEIL;31110;31;HAUTE GARONNE;1
31509;;Saint-P-d'Ardet;SAINT-PE-D'ARDET;31510;31;HAUTE GARONNE;1
31510;;Saint-P-Delbosc;SAINT-PE-DELBOSC;31350;31;HAUTE GARONNE;1
31511;;Saint-Pierre;SAINT-PIERRE;31590;31;HAUTE GARONNE;1
31512;;Saint-Pierre-de-Lages;SAINT-PIERRE-DE-LAGES;31570;31;HAUTE GARONNE;1
31513;;Saint-Plancard;SAINT-PLANCARD;31580;31;HAUTE GARONNE;1
31514;;Saint-Rome;SAINT-ROME;31290;31;HAUTE GARONNE;1
31515;;Saint-Rustice;SAINT-RUSTICE;31620;31;HAUTE GARONNE;1
31516;;Saint-Sauveur;SAINT-SAUVEUR;31790;31;HAUTE GARONNE;1
31517;;Saint-Sulpice-sur-Lze;SAINT-SULPICE-SUR-LEZE;31410;31;HAUTE GARONNE;1
31518;;Saint-Thomas;SAINT-THOMAS;31470;31;HAUTE GARONNE;1
31519;;Saint-Vincent;SAINT-VINCENT;31290;31;HAUTE GARONNE;1
31520;;Sajas;SAJAS;31370;31;HAUTE GARONNE;1
31521;;Saleich;SALEICH;31260;31;HAUTE GARONNE;1
31522;;Salerm;SALERM;31230;31;HAUTE GARONNE;1
31523;;Salies-du-Salat;SALIES-DU-SALAT;31260;31;HAUTE GARONNE;1
31524;;Salles-et-Pratviel;SALLES-ET-PRATVIEL;31110;31;HAUTE GARONNE;1
31525;;Salles-sur-Garonne;SALLES-SUR-GARONNE;31390;31;HAUTE GARONNE;1
31526;La;Salvetat-Saint-Gilles;SALVETAT-SAINT-GILLES;31880;31;HAUTE GARONNE;1
31527;La;Salvetat-Lauragais;SALVETAT-LAURAGAIS;31460;31;HAUTE GARONNE;1
31528;;Saman;SAMAN;31350;31;HAUTE GARONNE;1
31529;;Samouillan;SAMOUILLAN;31420;31;HAUTE GARONNE;1
31530;;Sana;SANA;31220;31;HAUTE GARONNE;1
31531;;Sarrecave;SARRECAVE;31350;31;HAUTE GARONNE;1
31532;;Sarremezan;SARREMEZAN;31350;31;HAUTE GARONNE;1
31533;;Saubens;SAUBENS;31600;31;HAUTE GARONNE;1
31534;;Saussens;SAUSSENS;31460;31;HAUTE GARONNE;1
31535;;Sauveterre-de-Comminges;SAUVETERRE-DE-COMMINGES;31510;31;HAUTE GARONNE;1
31536;;Saux-et-Pomarde;SAUX-ET-POMAREDE;31800;31;HAUTE GARONNE;1
31537;;Savarths;SAVARTHES;31800;31;HAUTE GARONNE;1
31538;;Savres;SAVERES;31370;31;HAUTE GARONNE;1
31539;;Sdeilhac;SEDEILHAC;31580;31;HAUTE GARONNE;1
31540;;Sgreville;SEGREVILLE;31460;31;HAUTE GARONNE;1
31541;;Seilh;SEILH;31840;31;HAUTE GARONNE;1
31542;;Seilhan;SEILHAN;31510;31;HAUTE GARONNE;1
31543;;Snarens;SENARENS;31430;31;HAUTE GARONNE;1
31544;;Sengouagnet;SENGOUAGNET;31160;31;HAUTE GARONNE;1
31545;;Sepx;SEPX;31360;31;HAUTE GARONNE;1
31546;;Seyre;SEYRE;31560;31;HAUTE GARONNE;1
31547;;Seysses;SEYSSES;31600;31;HAUTE GARONNE;1
31548;;Signac;SIGNAC;31440;31;HAUTE GARONNE;1
31549;;Sode;SODE;31110;31;HAUTE GARONNE;1
31550;;Soueich;SOUEICH;31160;31;HAUTE GARONNE;1
31551;;Tarabel;TARABEL;31570;31;HAUTE GARONNE;1
31552;;Terrebasse;TERREBASSE;31420;31;HAUTE GARONNE;1
31553;;Thil;THIL;31530;31;HAUTE GARONNE;1
31554;;Touille;TOUILLE;31260;31;HAUTE GARONNE;1
31555;;Toulouse;TOULOUSE;31000;31;HAUTE GARONNE;1
31556;Les;Tourreilles;TOURREILLES;31210;31;HAUTE GARONNE;1
31557;;Tournefeuille;TOURNEFEUILLE;31170;31;HAUTE GARONNE;1
31558;;Toutens;TOUTENS;31460;31;HAUTE GARONNE;1
31559;;Trbons-de-Luchon;TREBONS-DE-LUCHON;31110;31;HAUTE GARONNE;1
31560;;Trbons-sur-la-Grasse;TREBONS-SUR-LA-GRASSE;31290;31;HAUTE GARONNE;1
31561;L';Union;UNION;31240;31;HAUTE GARONNE;1
31562;;Urau;URAU;31260;31;HAUTE GARONNE;1
31563;;Vacquiers;VACQUIERS;31340;31;HAUTE GARONNE;1
31564;;Valcabrre;VALCABRERE;31510;31;HAUTE GARONNE;1
31565;;Valentine;VALENTINE;31800;31;HAUTE GARONNE;1
31566;;Vallgue;VALLEGUE;31290;31;HAUTE GARONNE;1
31567;;Vallesvilles;VALLESVILLES;31570;31;HAUTE GARONNE;1
31568;;Varennes;VARENNES;31450;31;HAUTE GARONNE;1
31569;;Vaudreuille;VAUDREUILLE;31250;31;HAUTE GARONNE;1
31570;;Vaux;VAUX;31540;31;HAUTE GARONNE;1
31571;;Vendine;VENDINE;31460;31;HAUTE GARONNE;1
31572;;Venerque;VENERQUE;31810;31;HAUTE GARONNE;1
31573;;Verfeil;VERFEIL;31590;31;HAUTE GARONNE;1
31574;;Vernet;VERNET;31810;31;HAUTE GARONNE;1
31575;;Vieille-Toulouse;VIEILLE-TOULOUSE;31320;31;HAUTE GARONNE;1
31576;;Vieillevigne;VIEILLEVIGNE;31290;31;HAUTE GARONNE;1
31577;;Vignaux;VIGNAUX;31480;31;HAUTE GARONNE;1
31578;;Vigoulet-Auzil;VIGOULET-AUZIL;31320;31;HAUTE GARONNE;1
31579;;Villaris;VILLARIES;31380;31;HAUTE GARONNE;1
31580;;Villate;VILLATE;31860;31;HAUTE GARONNE;1
31581;;Villaudric;VILLAUDRIC;31620;31;HAUTE GARONNE;1
31582;;Villefranche-de-Lauragais;VILLEFRANCHE-DE-LAURAGAIS;31290;31;HAUTE GARONNE;1
31583;;Villematier;VILLEMATIER;31340;31;HAUTE GARONNE;1
31584;;Villemur-sur-Tarn;VILLEMUR-SUR-TARN;31340;31;HAUTE GARONNE;1
31585;;Villeneuve-de-Rivire;VILLENEUVE-DE-RIVIERE;31800;31;HAUTE GARONNE;1
31586;;Villeneuve-Lcussan;VILLENEUVE-LECUSSAN;31580;31;HAUTE GARONNE;1
31587;;Villeneuve-ls-Bouloc;VILLENEUVE-LES-BOULOC;31620;31;HAUTE GARONNE;1
31588;;Villeneuve-Tolosane;VILLENEUVE-TOLOSANE;31270;31;HAUTE GARONNE;1
31589;;Villenouvelle;VILLENOUVELLE;31290;31;HAUTE GARONNE;1
31590;;Binos;BINOS;31440;31;HAUTE GARONNE;1
31591;;Escoulis;ESCOULIS;31260;31;HAUTE GARONNE;1
31592;;Larra;LARRA;31330;31;HAUTE GARONNE;1
31593;;Cazac;CAZAC;31230;31;HAUTE GARONNE;1
32001;;Aignan;AIGNAN;32290;32;GERS;1
32002;;Ansan;ANSAN;32270;32;GERS;1
32003;;Antras;ANTRAS;32360;32;GERS;1
32004;;Arblade-le-Bas;ARBLADE-LE-BAS;32720;32;GERS;1
32005;;Arblade-le-Haut;ARBLADE-LE-HAUT;32110;32;GERS;1
32007;;Ardizas;ARDIZAS;32430;32;GERS;1
32008;;Armentieux;ARMENTIEUX;32230;32;GERS;1
32009;;Armous-et-Cau;ARMOUS-ET-CAU;32230;32;GERS;1
32010;;Arroude;ARROUEDE;32140;32;GERS;1
32012;;Aubiet;AUBIET;32270;32;GERS;1
32013;;Auch;AUCH;32000;32;GERS;1
32014;;Augnax;AUGNAX;32120;32;GERS;1
32015;;Aujan-Mournde;AUJAN-MOURNEDE;32300;32;GERS;1
32016;;Aurad;AURADE;32600;32;GERS;1
32017;;Aurensan;AURENSAN;32400;32;GERS;1
32018;;Aurimont;AURIMONT;32450;32;GERS;1
32019;;Auterive;AUTERIVE;32550;32;GERS;1
32020;;Aux-Aussat;AUX-AUSSAT;32170;32;GERS;1
32021;;Avensac;AVENSAC;32120;32;GERS;1
32022;;Avron-Bergelle;AVERON-BERGELLE;32290;32;GERS;1
32023;;Avezan;AVEZAN;32380;32;GERS;1
32024;;Ayguetinte;AYGUETINTE;32410;32;GERS;1
32025;;Ayzieu;AYZIEU;32800;32;GERS;1
32026;;Bajonnette;BAJONNETTE;32120;32;GERS;1
32027;;Barcelonne-du-Gers;BARCELONNE-DU-GERS;32720;32;GERS;1
32028;;Barcugnan;BARCUGNAN;32170;32;GERS;1
32029;;Barran;BARRAN;32350;32;GERS;1
32030;;Bars;BARS;32300;32;GERS;1
32031;;Bascous;BASCOUS;32190;32;GERS;1
32032;;Bassoues;BASSOUES;32320;32;GERS;1
32033;;Bazian;BAZIAN;32320;32;GERS;1
32034;;Bazugues;BAZUGUES;32170;32;GERS;1
32035;;Beaucaire;BEAUCAIRE;32410;32;GERS;1
32036;;Beaumarchs;BEAUMARCHES;32160;32;GERS;1
32037;;Beaumont;BEAUMONT;32100;32;GERS;1
32038;;Beaupuy;BEAUPUY;32600;32;GERS;1
32039;;Beccas;BECCAS;32730;32;GERS;1
32040;;Bdchan;BEDECHAN;32450;32;GERS;1
32041;;Bellegarde;BELLEGARDE;32140;32;GERS;1
32042;;Belloc-Saint-Clamens;BELLOC-SAINT-CLAMENS;32300;32;GERS;1
32043;;Belmont;BELMONT;32190;32;GERS;1
32044;;Braut;BERAUT;32100;32;GERS;1
32045;;Berdoues;BERDOUES;32300;32;GERS;1
32046;;Bernde;BERNEDE;32400;32;GERS;1
32047;;Berrac;BERRAC;32480;32;GERS;1
32048;;Betcave-Aguin;BETCAVE-AGUIN;32420;32;GERS;1
32049;;Btous;BETOUS;32110;32;GERS;1
32050;;Betplan;BETPLAN;32730;32;GERS;1
32051;;Bzril;BEZERIL;32130;32;GERS;1
32052;;Bezolles;BEZOLLES;32310;32;GERS;1
32053;;Bzues-Bajon;BEZUES-BAJON;32140;32;GERS;1
32054;;Biran;BIRAN;32350;32;GERS;1
32055;;Bivs;BIVES;32380;32;GERS;1
32056;;Blanquefort;BLANQUEFORT;32270;32;GERS;1
32057;;Blaziert;BLAZIERT;32100;32;GERS;1
32058;;Blousson-Srian;BLOUSSON-SERIAN;32230;32;GERS;1
32059;;Bonas;BONAS;32410;32;GERS;1
32060;;Boucagnres;BOUCAGNERES;32550;32;GERS;1
32061;;Boulaur;BOULAUR;32450;32;GERS;1
32062;;Bourrouillan;BOURROUILLAN;32370;32;GERS;1
32063;;Bouzon-Gellenave;BOUZON-GELLENAVE;32290;32;GERS;1
32064;;Bretagne-d'Armagnac;BRETAGNE-D'ARMAGNAC;32800;32;GERS;1
32065;Le;Brouilh-Monbert;BROUILH-MONBERT;32350;32;GERS;1
32066;;Brugnens;BRUGNENS;32500;32;GERS;1
32067;;Cabas-Loumasss;CABAS-LOUMASSES;32140;32;GERS;1
32068;;Cadeilhan;CADEILHAN;32380;32;GERS;1
32069;;Cadeillan;CADEILLAN;32220;32;GERS;1
32070;;Cahuzac-sur-Adour;CAHUZAC-SUR-ADOUR;32400;32;GERS;1
32071;;Caillavet;CAILLAVET;32190;32;GERS;1
32072;;Callian;CALLIAN;32190;32;GERS;1
32073;;Campagne-d'Armagnac;CAMPAGNE-D'ARMAGNAC;32800;32;GERS;1
32074;;Cannet;CANNET;32400;32;GERS;1
32075;;Cassaigne;CASSAIGNE;32100;32;GERS;1
32076;;Castelnau-Barbarens;CASTELNAU-BARBARENS;32450;32;GERS;1
32077;;Castelnau-d'Angls;CASTELNAU-D'ANGLES;32320;32;GERS;1
32078;;Castelnau-d'Arbieu;CASTELNAU-D'ARBIEU;32500;32;GERS;1
32079;;Castelnau-d'Auzan;CASTELNAU-D'AUZAN;32440;32;GERS;1
32080;;Castelnau-sur-l'Auvignon;CASTELNAU-SUR-L'AUVIGNON;32100;32;GERS;1
32081;;Castelnavet;CASTELNAVET;32290;32;GERS;1
32082;;Castra-Lectourois;CASTERA-LECTOUROIS;32700;32;GERS;1
32083;;Castra-Verduzan;CASTERA-VERDUZAN;32410;32;GERS;1
32084;;Castron;CASTERON;32380;32;GERS;1
32085;;Castet-Arrouy;CASTET-ARROUY;32340;32;GERS;1
32086;;Castex;CASTEX;32170;32;GERS;1
32087;;Castex-d'Armagnac;CASTEX-D'ARMAGNAC;32240;32;GERS;1
32088;;Castillon-Debats;CASTILLON-DEBATS;32190;32;GERS;1
32089;;Castillon-Massas;CASTILLON-MASSAS;32360;32;GERS;1
32090;;Castillon-Savs;CASTILLON-SAVES;32490;32;GERS;1
32091;;Castin;CASTIN;32810;32;GERS;1
32092;;Catonvielle;CATONVIELLE;32200;32;GERS;1
32093;;Caumont;CAUMONT;32400;32;GERS;1
32094;;Caupenne-d'Armagnac;CAUPENNE-D'ARMAGNAC;32110;32;GERS;1
32095;;Caussens;CAUSSENS;32100;32;GERS;1
32096;;Cazaubon;CAZAUBON;32150;32;GERS;1
32097;;Cazaux-d'Angls;CAZAUX-D'ANGLES;32190;32;GERS;1
32098;;Cazaux-Savs;CAZAUX-SAVES;32130;32;GERS;1
32099;;Cazaux-Villecomtal;CAZAUX-VILLECOMTAL;32230;32;GERS;1
32100;;Cazeneuve;CAZENEUVE;32800;32;GERS;1
32101;;Cran;CERAN;32500;32;GERS;1
32102;;Czan;CEZAN;32410;32;GERS;1
32103;;Chlan;CHELAN;32140;32;GERS;1
32104;;Clermont-Pouyguills;CLERMONT-POUYGUILLES;32300;32;GERS;1
32105;;Clermont-Savs;CLERMONT-SAVES;32600;32;GERS;1
32106;;Cologne;COLOGNE;32430;32;GERS;1
32107;;Condom;CONDOM;32100;32;GERS;1
32108;;Corneillan;CORNEILLAN;32400;32;GERS;1
32109;;Couloum-Mondebat;COULOUME-MONDEBAT;32160;32;GERS;1
32110;;Courrensan;COURRENSAN;32330;32;GERS;1
32111;;Courties;COURTIES;32230;32;GERS;1
32112;;Crastes;CRASTES;32270;32;GERS;1
32113;;Cravencres;CRAVENCERES;32110;32;GERS;1
32114;;Culas;CUELAS;32300;32;GERS;1
32115;;Dmu;DEMU;32190;32;GERS;1
32116;;Duffort;DUFFORT;32170;32;GERS;1
32117;;Duran;DURAN;32810;32;GERS;1
32118;;Durban;DURBAN;32260;32;GERS;1
32119;;Eauze;EAUZE;32800;32;GERS;1
32120;;Encausse;ENCAUSSE;32430;32;GERS;1
32121;;Endoufielle;ENDOUFIELLE;32600;32;GERS;1
32122;;Esclassan-Labastide;ESCLASSAN-LABASTIDE;32140;32;GERS;1
32123;;Escornebuf;ESCORNEBOEUF;32200;32;GERS;1
32124;;Espaon;ESPAON;32220;32;GERS;1
32125;;Espas;ESPAS;32370;32;GERS;1
32126;;Estampes;ESTAMPES;32170;32;GERS;1
32127;;Estang;ESTANG;32240;32;GERS;1
32128;;Estipouy;ESTIPOUY;32300;32;GERS;1
32129;;Estramiac;ESTRAMIAC;32380;32;GERS;1
32130;;Faget-Abbatial;FAGET-ABBATIAL;32450;32;GERS;1
32131;;Flamarens;FLAMARENS;32340;32;GERS;1
32132;;Fleurance;FLEURANCE;32500;32;GERS;1
32133;;Fourcs;FOURCES;32250;32;GERS;1
32134;;Frgouville;FREGOUVILLE;32490;32;GERS;1
32135;;Fustrouau;FUSTEROUAU;32400;32;GERS;1
32136;;Galiax;GALIAX;32160;32;GERS;1
32138;;Garravet;GARRAVET;32220;32;GERS;1
32139;;Gaudonville;GAUDONVILLE;32380;32;GERS;1
32140;;Gaujac;GAUJAC;32220;32;GERS;1
32141;;Gaujan;GAUJAN;32420;32;GERS;1
32142;;Gavarret-sur-Aulouste;GAVARRET-SUR-AULOUSTE;32390;32;GERS;1
32143;;Gazaupouy;GAZAUPOUY;32480;32;GERS;1
32144;;Gazax-et-Baccarisse;GAZAX-ET-BACCARISSE;32230;32;GERS;1
32145;;Ge-Rivire;GEE-RIVIERE;32720;32;GERS;1
32146;;Gimbrde;GIMBREDE;32340;32;GERS;1
32147;;Gimont;GIMONT;32200;32;GERS;1
32148;;Giscaro;GISCARO;32200;32;GERS;1
32149;;Gondrin;GONDRIN;32330;32;GERS;1
32150;;Goutz;GOUTZ;32500;32;GERS;1
32151;;Goux;GOUX;32400;32;GERS;1
32152;;Haget;HAGET;32730;32;GERS;1
32153;;Haulies;HAULIES;32550;32;GERS;1
32154;;Homps;HOMPS;32120;32;GERS;1
32155;Le;Houga;HOUGA;32460;32;GERS;1
32156;;Idrac-Respaills;IDRAC-RESPAILLES;32300;32;GERS;1
32157;L';Isle-Arn;ISLE-ARNE;32270;32;GERS;1
32158;L';Isle-Bouzon;ISLE-BOUZON;32380;32;GERS;1
32159;L';Isle-de-No;ISLE-DE-NOE;32300;32;GERS;1
32160;L';Isle-Jourdain;ISLE-JOURDAIN;32600;32;GERS;1
32161;;Izotges;IZOTGES;32400;32;GERS;1
32162;;Jegun;JEGUN;32360;32;GERS;1
32163;;J-Belloc;JU-BELLOC;32160;32;GERS;1
32164;;Juillac;JUILLAC;32230;32;GERS;1
32165;;Juilles;JUILLES;32200;32;GERS;1
32166;;Justian;JUSTIAN;32190;32;GERS;1
32167;;Laas;LAAS;32170;32;GERS;1
32168;;Labarrre;LABARRERE;32250;32;GERS;1
32169;;Labarthe;LABARTHE;32260;32;GERS;1
32170;;Labarthte;LABARTHETE;32400;32;GERS;1
32171;;Labastide-Savs;LABASTIDE-SAVES;32130;32;GERS;1
32172;;Labjan;LABEJAN;32300;32;GERS;1
32173;;Labrihe;LABRIHE;32120;32;GERS;1
32174;;Ladevze-Rivire;LADEVEZE-RIVIERE;32230;32;GERS;1
32175;;Ladevze-Ville;LADEVEZE-VILLE;32230;32;GERS;1
32176;;Lagarde;LAGARDE;32700;32;GERS;1
32177;;Lagarde-Hachan;LAGARDE-HACHAN;32300;32;GERS;1
32178;;Lagardre;LAGARDERE;32310;32;GERS;1
32180;;Lagraulet-du-Gers;LAGRAULET-DU-GERS;32330;32;GERS;1
32181;;Laguian-Mazous;LAGUIAN-MAZOUS;32170;32;GERS;1
32182;;Lahas;LAHAS;32130;32;GERS;1
32183;;Lahitte;LAHITTE;32810;32;GERS;1
32184;;Lalanne;LALANNE;32500;32;GERS;1
32185;;Lalanne-Arqu;LALANNE-ARQUE;32140;32;GERS;1
32186;;Lamagure;LAMAGUERE;32260;32;GERS;1
32187;;Lamazre;LAMAZERE;32300;32;GERS;1
32188;;Lamothe-Goas;LAMOTHE-GOAS;32500;32;GERS;1
32189;;Lannemaignan;LANNEMAIGNAN;32240;32;GERS;1
32190;;Lannepax;LANNEPAX;32190;32;GERS;1
32191;;Lanne-Soubiran;LANNE-SOUBIRAN;32110;32;GERS;1
32192;;Lannux;LANNUX;32400;32;GERS;1
32193;;Lare;LAREE;32150;32;GERS;1
32194;;Larressingle;LARRESSINGLE;32100;32;GERS;1
32195;;Larroque-Engalin;LARROQUE-ENGALIN;32480;32;GERS;1
32196;;Larroque-Saint-Sernin;LARROQUE-SAINT-SERNIN;32410;32;GERS;1
32197;;Larroque-sur-l'Osse;LARROQUE-SUR-L'OSSE;32100;32;GERS;1
32198;;Lartigue;LARTIGUE;32450;32;GERS;1
32199;;Lasserade;LASSERADE;32160;32;GERS;1
32200;;Lassran;LASSERAN;32550;32;GERS;1
32201;;Lasseube-Propre;LASSEUBE-PROPRE;32550;32;GERS;1
32202;;Laujuzan;LAUJUZAN;32110;32;GERS;1
32203;;Laurat;LAURAET;32330;32;GERS;1
32204;;Lavardens;LAVARDENS;32360;32;GERS;1
32205;;Laverat;LAVERAET;32230;32;GERS;1
32206;;Laymont;LAYMONT;32220;32;GERS;1
32207;;Leboulin;LEBOULIN;32810;32;GERS;1
32208;;Lectoure;LECTOURE;32700;32;GERS;1
32209;;Lelin-Lapujolle;LELIN-LAPUJOLLE;32400;32;GERS;1
32210;;Lias;LIAS;32600;32;GERS;1
32211;;Lias-d'Armagnac;LIAS-D'ARMAGNAC;32240;32;GERS;1
32212;;Ligardes;LIGARDES;32480;32;GERS;1
32213;;Lombez;LOMBEZ;32220;32;GERS;1
32214;;Loubdat;LOUBEDAT;32110;32;GERS;1
32215;;Loubersan;LOUBERSAN;32300;32;GERS;1
32216;;Lourties-Monbrun;LOURTIES-MONBRUN;32140;32;GERS;1
32217;;Louslitges;LOUSLITGES;32230;32;GERS;1
32218;;Loussous-Dbat;LOUSSOUS-DEBAT;32290;32;GERS;1
32219;;Lupiac;LUPIAC;32290;32;GERS;1
32220;;Lupp-Violles;LUPPE-VIOLLES;32110;32;GERS;1
32221;;Lussan;LUSSAN;32270;32;GERS;1
32222;;Magnan;MAGNAN;32110;32;GERS;1
32223;;Magnas;MAGNAS;32380;32;GERS;1
32224;;Maignaut-Tauzia;MAIGNAUT-TAUZIA;32310;32;GERS;1
32225;;Malabat;MALABAT;32730;32;GERS;1
32226;;Manas-Bastanous;MANAS-BASTANOUS;32170;32;GERS;1
32227;;Manciet;MANCIET;32370;32;GERS;1
32228;;Manent-Montan;MANENT-MONTANE;32140;32;GERS;1
32229;;Mansempuy;MANSEMPUY;32120;32;GERS;1
32230;;Mansencme;MANSENCOME;32310;32;GERS;1
32231;;Marambat;MARAMBAT;32190;32;GERS;1
32232;;Maravat;MARAVAT;32120;32;GERS;1
32233;;Marciac;MARCIAC;32230;32;GERS;1
32234;;Marestaing;MARESTAING;32490;32;GERS;1
32235;;Margout-Meymes;MARGOUET-MEYMES;32290;32;GERS;1
32236;;Marguestau;MARGUESTAU;32150;32;GERS;1
32237;;Marsan;MARSAN;32270;32;GERS;1
32238;;Marseillan;MARSEILLAN;32170;32;GERS;1
32239;;Marsolan;MARSOLAN;32700;32;GERS;1
32240;;Mascaras;MASCARAS;32230;32;GERS;1
32241;;Mas-d'Auvignon;MAS-D'AUVIGNON;32700;32;GERS;1
32242;;Masseube;MASSEUBE;32140;32;GERS;1
32243;;Maulon-d'Armagnac;MAULEON-D'ARMAGNAC;32240;32;GERS;1
32244;;Maulichres;MAULICHERES;32400;32;GERS;1
32245;;Maumusson-Laguian;MAUMUSSON-LAGUIAN;32400;32;GERS;1
32246;;Maupas;MAUPAS;32240;32;GERS;1
32247;;Maurens;MAURENS;32200;32;GERS;1
32248;;Mauroux;MAUROUX;32380;32;GERS;1
32249;;Mauvezin;MAUVEZIN;32120;32;GERS;1
32250;;Meilhan;MEILHAN;32420;32;GERS;1
32251;;Mrens;MERENS;32360;32;GERS;1
32252;;Milan;MIELAN;32170;32;GERS;1
32253;;Miradoux;MIRADOUX;32340;32;GERS;1
32254;;Miramont-d'Astarac;MIRAMONT-D'ASTARAC;32300;32;GERS;1
32255;;Miramont-Latour;MIRAMONT-LATOUR;32390;32;GERS;1
32256;;Mirande;MIRANDE;32300;32;GERS;1
32257;;Mirannes;MIRANNES;32350;32;GERS;1
32258;;Mirepoix;MIREPOIX;32390;32;GERS;1
32260;;Monbardon;MONBARDON;32420;32;GERS;1
32261;;Monblanc;MONBLANC;32130;32;GERS;1
32262;;Monbrun;MONBRUN;32600;32;GERS;1
32263;;Moncassin;MONCASSIN;32300;32;GERS;1
32264;;Monclar;MONCLAR;32150;32;GERS;1
32265;;Monclar-sur-Losse;MONCLAR-SUR-LOSSE;32300;32;GERS;1
32266;;Moncorneil-Grazan;MONCORNEIL-GRAZAN;32260;32;GERS;1
32267;;Monferran-Plavs;MONFERRAN-PLAVES;32260;32;GERS;1
32268;;Monferran-Savs;MONFERRAN-SAVES;32490;32;GERS;1
32269;;Monfort;MONFORT;32120;32;GERS;1
32270;;Mongausy;MONGAUSY;32220;32;GERS;1
32271;;Monguilhem;MONGUILHEM;32240;32;GERS;1
32272;;Monlaur-Bernet;MONLAUR-BERNET;32140;32;GERS;1
32273;;Monlezun;MONLEZUN;32230;32;GERS;1
32274;;Monlezun-d'Armagnac;MONLEZUN-D'ARMAGNAC;32240;32;GERS;1
32275;;Monpardiac;MONPARDIAC;32170;32;GERS;1
32276;;Montadet;MONTADET;32220;32;GERS;1
32277;;Montamat;MONTAMAT;32220;32;GERS;1
32278;;Montaut;MONTAUT;32300;32;GERS;1
32279;;Montaut-les-Crneaux;MONTAUT-LES-CRENEAUX;32810;32;GERS;1
32280;;Mont-d'Astarac;MONT-D'ASTARAC;32140;32;GERS;1
32281;;Mont-de-Marrast;MONT-DE-MARRAST;32170;32;GERS;1
32282;;Montgut;MONTEGUT;32550;32;GERS;1
32283;;Montgut-Arros;MONTEGUT-ARROS;32730;32;GERS;1
32284;;Montgut-Savs;MONTEGUT-SAVES;32220;32;GERS;1
32285;;Montesquiou;MONTESQUIOU;32320;32;GERS;1
32286;;Montestruc-sur-Gers;MONTESTRUC-SUR-GERS;32390;32;GERS;1
32287;;Monties;MONTIES;32420;32;GERS;1
32288;;Montiron;MONTIRON;32200;32;GERS;1
32289;;Montpzat;MONTPEZAT;32220;32;GERS;1
32290;;Montral;MONTREAL;32250;32;GERS;1
32291;;Morms;MORMES;32240;32;GERS;1
32292;;Mouchan;MOUCHAN;32330;32;GERS;1
32293;;Mouchs;MOUCHES;32300;32;GERS;1
32294;;Mourde;MOUREDE;32190;32;GERS;1
32295;;Nizas;NIZAS;32130;32;GERS;1
32296;;Nogaro;NOGARO;32110;32;GERS;1
32297;;Noilhan;NOILHAN;32130;32;GERS;1
32298;;Nougaroulet;NOUGAROULET;32270;32;GERS;1
32299;;Noulens;NOULENS;32800;32;GERS;1
32300;;Orbessan;ORBESSAN;32260;32;GERS;1
32301;;Ordan-Larroque;ORDAN-LARROQUE;32350;32;GERS;1
32302;;Ornzan;ORNEZAN;32260;32;GERS;1
32303;;Pallanne;PALLANNE;32230;32;GERS;1
32304;;Panassac;PANASSAC;32140;32;GERS;1
32305;;Panjas;PANJAS;32110;32;GERS;1
32306;;Pauilhac;PAUILHAC;32500;32;GERS;1
32307;;Pavie;PAVIE;32550;32;GERS;1
32308;;Pbes;PEBEES;32130;32;GERS;1
32309;;Pellefigue;PELLEFIGUE;32420;32;GERS;1
32310;;Perchde;PERCHEDE;32460;32;GERS;1
32311;;Pergain-Taillac;PERGAIN-TAILLAC;32700;32;GERS;1
32312;;Pessan;PESSAN;32550;32;GERS;1
32313;;Pessoulens;PESSOULENS;32380;32;GERS;1
32314;;Peyrecave;PEYRECAVE;32340;32;GERS;1
32315;;Peyrusse-Grande;PEYRUSSE-GRANDE;32320;32;GERS;1
32316;;Peyrusse-Massas;PEYRUSSE-MASSAS;32360;32;GERS;1
32317;;Peyrusse-Vieille;PEYRUSSE-VIEILLE;32230;32;GERS;1
32318;;Pis;PIS;32500;32;GERS;1
32319;;Plaisance;PLAISANCE;32160;32;GERS;1
32320;;Plieux;PLIEUX;32340;32;GERS;1
32321;;Polastron;POLASTRON;32130;32;GERS;1
32322;;Pompiac;POMPIAC;32130;32;GERS;1
32323;;Ponsampre;PONSAMPERE;32300;32;GERS;1
32324;;Ponsan-Soubiran;PONSAN-SOUBIRAN;32300;32;GERS;1
32325;;Pouydraguin;POUYDRAGUIN;32290;32;GERS;1
32326;;Pouylebon;POUYLEBON;32320;32;GERS;1
32327;;Pouy-Loubrin;POUY-LOUBRIN;32260;32;GERS;1
32328;;Pouy-Roquelaure;POUY-ROQUELAURE;32480;32;GERS;1
32329;;Prchac;PRECHAC;32390;32;GERS;1
32330;;Prchac-sur-Adour;PRECHAC-SUR-ADOUR;32160;32;GERS;1
32331;;Preignan;PREIGNAN;32810;32;GERS;1
32332;;Prneron;PRENERON;32190;32;GERS;1
32333;;Projan;PROJAN;32400;32;GERS;1
32334;;Pujaudran;PUJAUDRAN;32600;32;GERS;1
32335;;Puycasquier;PUYCASQUIER;32120;32;GERS;1
32336;;Puylausic;PUYLAUSIC;32220;32;GERS;1
32337;;Puysgur;PUYSEGUR;32390;32;GERS;1
32338;;Ramouzens;RAMOUZENS;32800;32;GERS;1
32339;;Razengues;RAZENGUES;32600;32;GERS;1
32340;;Rans;REANS;32800;32;GERS;1
32341;;Rjaumont;REJAUMONT;32390;32;GERS;1
32342;;Ricourt;RICOURT;32230;32;GERS;1
32343;;Riguepeu;RIGUEPEU;32320;32;GERS;1
32344;;Riscle;RISCLE;32400;32;GERS;1
32345;La;Romieu;ROMIEU;32480;32;GERS;1
32346;;Roquebrune;ROQUEBRUNE;32190;32;GERS;1
32347;;Roquefort;ROQUEFORT;32390;32;GERS;1
32348;;Roquelaure;ROQUELAURE;32810;32;GERS;1
32349;;Roquelaure-Saint-Aubin;ROQUELAURE-SAINT-AUBIN;32430;32;GERS;1
32350;;Roquepine;ROQUEPINE;32100;32;GERS;1
32351;;Roques;ROQUES;32310;32;GERS;1
32352;;Rozs;ROZES;32190;32;GERS;1
32353;;Sabaillan;SABAILLAN;32420;32;GERS;1
32354;;Sabazan;SABAZAN;32290;32;GERS;1
32355;;Sadeillan;SADEILLAN;32170;32;GERS;1
32356;;Saint-Andr;SAINT-ANDRE;32200;32;GERS;1
32357;;Sainte-Anne;SAINTE-ANNE;32430;32;GERS;1
32358;;Saint-Antoine;SAINT-ANTOINE;32340;32;GERS;1
32359;;Saint-Antonin;SAINT-ANTONIN;32120;32;GERS;1
32360;;Saint-Arailles;SAINT-ARAILLES;32350;32;GERS;1
32361;;Saint-Arroman;SAINT-ARROMAN;32300;32;GERS;1
32362;;Saint-Aunix-Lengros;SAINT-AUNIX-LENGROS;32160;32;GERS;1
32363;;Sainte-Aurence-Cazaux;SAINTE-AURENCE-CAZAUX;32300;32;GERS;1
32364;;Saint-Avit-Frandat;SAINT-AVIT-FRANDAT;32700;32;GERS;1
32365;;Saint-Blancard;SAINT-BLANCARD;32140;32;GERS;1
32366;;Saint-Brs;SAINT-BRES;32120;32;GERS;1
32367;;Saint-Christaud;SAINT-CHRISTAUD;32320;32;GERS;1
32368;;Sainte-Christie;SAINTE-CHRISTIE;32390;32;GERS;1
32369;;Sainte-Christie-d'Armagnac;SAINTE-CHRISTIE-D'ARMAGNAC;32370;32;GERS;1
32370;;Saint-Clar;SAINT-CLAR;32380;32;GERS;1
32371;;Saint-Crac;SAINT-CREAC;32380;32;GERS;1
32372;;Saint-Cricq;SAINT-CRICQ;32430;32;GERS;1
32373;;Sainte-Dode;SAINTE-DODE;32170;32;GERS;1
32374;;Saint-lix;SAINT-ELIX;32450;32;GERS;1
32375;;Saint-lix-Theux;SAINT-ELIX-THEUX;32300;32;GERS;1
32376;;Sainte-Gemme;SAINTE-GEMME;32120;32;GERS;1
32377;;Saint-Georges;SAINT-GEORGES;32430;32;GERS;1
32378;;Saint-Germ;SAINT-GERME;32400;32;GERS;1
32379;;Saint-Germier;SAINT-GERMIER;32200;32;GERS;1
32380;;Saint-Gride;SAINT-GRIEDE;32110;32;GERS;1
32381;;Saint-Jean-le-Comtal;SAINT-JEAN-LE-COMTAL;32550;32;GERS;1
32382;;Saint-Jean-Poutge;SAINT-JEAN-POUTGE;32190;32;GERS;1
32383;;Saint-Justin;SAINT-JUSTIN;32230;32;GERS;1
32384;;Saint-Lary;SAINT-LARY;32360;32;GERS;1
32385;;Saint-Lonard;SAINT-LEONARD;32380;32;GERS;1
32386;;Saint-Lizier-du-Plant;SAINT-LIZIER-DU-PLANTE;32220;32;GERS;1
32387;;Saint-Loube;SAINT-LOUBE;32220;32;GERS;1
32388;;Sainte-Marie;SAINTE-MARIE;32200;32;GERS;1
32389;;Saint-Martin;SAINT-MARTIN;32300;32;GERS;1
32390;;Saint-Martin-d'Armagnac;SAINT-MARTIN-D'ARMAGNAC;32110;32;GERS;1
32391;;Saint-Martin-de-Goyne;SAINT-MARTIN-DE-GOYNE;32480;32;GERS;1
32392;;Saint-Martin-Gimois;SAINT-MARTIN-GIMOIS;32450;32;GERS;1
32393;;Saint-Maur;SAINT-MAUR;32300;32;GERS;1
32394;;Saint-Mdard;SAINT-MEDARD;32300;32;GERS;1
32395;;Sainte-Mre;SAINTE-MERE;32700;32;GERS;1
32396;;Saint-Mzard;SAINT-MEZARD;32700;32;GERS;1
32397;;Saint-Michel;SAINT-MICHEL;32300;32;GERS;1
32398;;Saint-Mont;SAINT-MONT;32400;32;GERS;1
32399;;Saint-Orens;SAINT-ORENS;32120;32;GERS;1
32400;;Saint-Orens-Pouy-Petit;SAINT-ORENS-POUY-PETIT;32100;32;GERS;1
32401;;Saint-Ost;SAINT-OST;32300;32;GERS;1
32402;;Saint-Paul-de-Base;SAINT-PAUL-DE-BAISE;32190;32;GERS;1
32403;;Saint-Pierre-d'Aubzies;SAINT-PIERRE-D'AUBEZIES;32290;32;GERS;1
32404;;Saint-Puy;SAINT-PUY;32310;32;GERS;1
32405;;Sainte-Radegonde;SAINTE-RADEGONDE;32500;32;GERS;1
32406;;Saint-Sauvy;SAINT-SAUVY;32270;32;GERS;1
32407;;Saint-Soulan;SAINT-SOULAN;32220;32;GERS;1
32408;;Salles-d'Armagnac;SALLES-D'ARMAGNAC;32370;32;GERS;1
32409;;Samaran;SAMARAN;32140;32;GERS;1
32410;;Samatan;SAMATAN;32130;32;GERS;1
32411;;Sansan;SANSAN;32260;32;GERS;1
32412;;Saramon;SARAMON;32450;32;GERS;1
32413;;Sarcos;SARCOS;32420;32;GERS;1
32414;;Sarragachies;SARRAGACHIES;32400;32;GERS;1
32415;;Sarraguzan;SARRAGUZAN;32170;32;GERS;1
32416;;Sarrant;SARRANT;32120;32;GERS;1
32417;La;Sauvetat;SAUVETAT;32500;32;GERS;1
32418;;Sauveterre;SAUVETERRE;32220;32;GERS;1
32419;;Sauviac;SAUVIAC;32300;32;GERS;1
32420;;Sauvimont;SAUVIMONT;32220;32;GERS;1
32421;;Savignac-Mona;SAVIGNAC-MONA;32130;32;GERS;1
32422;;Scieurac-et-Flours;SCIEURAC-ET-FLOURES;32230;32;GERS;1
32423;;Sailles;SEAILLES;32190;32;GERS;1
32424;;Sgos;SEGOS;32400;32;GERS;1
32425;;Sgoufielle;SEGOUFIELLE;32600;32;GERS;1
32426;;Seissan;SEISSAN;32260;32;GERS;1
32427;;Sembous;SEMBOUES;32230;32;GERS;1
32428;;Smzies-Cachan;SEMEZIES-CACHAN;32450;32;GERS;1
32429;;Sempesserre;SEMPESSERRE;32700;32;GERS;1
32430;;Sre;SERE;32140;32;GERS;1
32431;;Srempuy;SEREMPUY;32120;32;GERS;1
32432;;Seysses-Savs;SEYSSES-SAVES;32130;32;GERS;1
32433;;Simorre;SIMORRE;32420;32;GERS;1
32434;;Sion;SION;32110;32;GERS;1
32435;;Sirac;SIRAC;32430;32;GERS;1
32436;;Solomiac;SOLOMIAC;32120;32;GERS;1
32437;;Sorbets;SORBETS;32110;32;GERS;1
32438;;Tachoires;TACHOIRES;32260;32;GERS;1
32439;;Tarsac;TARSAC;32400;32;GERS;1
32440;;Tasque;TASQUE;32160;32;GERS;1
32441;;Taybosc;TAYBOSC;32120;32;GERS;1
32442;;Terraube;TERRAUBE;32700;32;GERS;1
32443;;Termes-d'Armagnac;TERMES-D'ARMAGNAC;32400;32;GERS;1
32444;;Thoux;THOUX;32430;32;GERS;1
32445;;Tieste-Uragnoux;TIESTE-URAGNOUX;32160;32;GERS;1
32446;;Tillac;TILLAC;32170;32;GERS;1
32447;;Tirent-Pontjac;TIRENT-PONTEJAC;32450;32;GERS;1
32448;;Touget;TOUGET;32430;32;GERS;1
32449;;Toujouse;TOUJOUSE;32240;32;GERS;1
32450;;Tourdun;TOURDUN;32230;32;GERS;1
32451;;Tournan;TOURNAN;32420;32;GERS;1
32452;;Tournecoupe;TOURNECOUPE;32380;32;GERS;1
32453;;Tourrenquets;TOURRENQUETS;32390;32;GERS;1
32454;;Traversres;TRAVERSERES;32450;32;GERS;1
32455;;Troncens;TRONCENS;32230;32;GERS;1
32456;;Tudelle;TUDELLE;32190;32;GERS;1
32457;;Urdens;URDENS;32500;32;GERS;1
32458;;Urgosse;URGOSSE;32110;32;GERS;1
32459;;Valence-sur-Base;VALENCE-SUR-BAISE;32310;32;GERS;1
32460;;Vergoignan;VERGOIGNAN;32720;32;GERS;1
32461;;Verlus;VERLUS;32400;32;GERS;1
32462;;Vic-Fezensac;VIC-FEZENSAC;32190;32;GERS;1
32463;;Viella;VIELLA;32400;32;GERS;1
32464;;Villecomtal-sur-Arros;VILLECOMTAL-SUR-ARROS;32730;32;GERS;1
32465;;Villefranche;VILLEFRANCHE;32420;32;GERS;1
32466;;Viozan;VIOZAN;32300;32;GERS;1
32467;;Saint-Caprais;SAINT-CAPRAIS;32200;32;GERS;1
32468;;Aussos;AUSSOS;32140;32;GERS;1
33001;;Abzac;ABZAC;33230;33;GIRONDE;1
33002;;Aillas;AILLAS;33124;33;GIRONDE;1
33003;;Ambars-et-Lagrave;AMBARES-ET-LAGRAVE;33440;33;GIRONDE;1
33004;;Ambs;AMBES;33810;33;GIRONDE;1
33005;;Andernos-les-Bains;ANDERNOS-LES-BAINS;33510;33;GIRONDE;1
33006;;Anglade;ANGLADE;33390;33;GIRONDE;1
33007;;Arbanats;ARBANATS;33640;33;GIRONDE;1
33008;;Arbis;ARBIS;33760;33;GIRONDE;1
33009;;Arcachon;ARCACHON;33120;33;GIRONDE;1
33010;;Arcins;ARCINS;33460;33;GIRONDE;1
33011;;Ars;ARES;33740;33;GIRONDE;1
33012;;Arsac;ARSAC;33460;33;GIRONDE;1
33013;;Artigues-prs-Bordeaux;ARTIGUES-PRES-BORDEAUX;33370;33;GIRONDE;1
33014;Les;Artigues-de-Lussac;ARTIGUES-DE-LUSSAC;33570;33;GIRONDE;1
33015;;Arveyres;ARVEYRES;33500;33;GIRONDE;1
33016;;Asques;ASQUES;33240;33;GIRONDE;1
33017;;Aubiac;AUBIAC;33430;33;GIRONDE;1
33018;;Aubie-et-Espessas;AUBIE-ET-ESPESSAS;33240;33;GIRONDE;1
33019;;Audenge;AUDENGE;33980;33;GIRONDE;1
33020;;Auriolles;AURIOLLES;33790;33;GIRONDE;1
33021;;Auros;AUROS;33124;33;GIRONDE;1
33022;;Avensan;AVENSAN;33480;33;GIRONDE;1
33023;;Ayguemorte-les-Graves;AYGUEMORTE-LES-GRAVES;33640;33;GIRONDE;1
33024;;Bagas;BAGAS;33190;33;GIRONDE;1
33025;;Baigneaux;BAIGNEAUX;33760;33;GIRONDE;1
33026;;Balizac;BALIZAC;33730;33;GIRONDE;1
33027;;Barie;BARIE;33190;33;GIRONDE;1
33028;;Baron;BARON;33750;33;GIRONDE;1
33029;Le;Barp;BARP;33114;33;GIRONDE;1
33030;;Barsac;BARSAC;33720;33;GIRONDE;1
33031;;Bassanne;BASSANNE;33190;33;GIRONDE;1
33032;;Bassens;BASSENS;33530;33;GIRONDE;1
33033;;Baurech;BAURECH;33880;33;GIRONDE;1
33034;;Bayas;BAYAS;33230;33;GIRONDE;1
33035;;Bayon-sur-Gironde;BAYON-SUR-GIRONDE;33710;33;GIRONDE;1
33036;;Bazas;BAZAS;33430;33;GIRONDE;1
33037;;Beautiran;BEAUTIRAN;33640;33;GIRONDE;1
33038;;Bgadan;BEGADAN;33340;33;GIRONDE;1
33039;;Bgles;BEGLES;33130;33;GIRONDE;1
33040;;Bguey;BEGUEY;33410;33;GIRONDE;1
33042;;Belin-Bliet;BELIN-BELIET;33830;33;GIRONDE;1
33043;;Bellebat;BELLEBAT;33760;33;GIRONDE;1
33044;;Bellefond;BELLEFOND;33760;33;GIRONDE;1
33045;;Belvs-de-Castillon;BELVES-DE-CASTILLON;33350;33;GIRONDE;1
33046;;Bernos-Beaulac;BERNOS-BEAULAC;33430;33;GIRONDE;1
33047;;Berson;BERSON;33390;33;GIRONDE;1
33048;;Berthez;BERTHEZ;33124;33;GIRONDE;1
33049;;Beychac-et-Caillau;BEYCHAC-ET-CAILLAU;33750;33;GIRONDE;1
33050;;Bieujac;BIEUJAC;33210;33;GIRONDE;1
33051;;Biganos;BIGANOS;33380;33;GIRONDE;1
33052;Les;Billaux;BILLAUX;33500;33;GIRONDE;1
33053;;Birac;BIRAC;33430;33;GIRONDE;1
33054;;Blaignac;BLAIGNAC;33190;33;GIRONDE;1
33055;;Blaignan;BLAIGNAN;33340;33;GIRONDE;1
33056;;Blanquefort;BLANQUEFORT;33290;33;GIRONDE;1
33057;;Blasimon;BLASIMON;33540;33;GIRONDE;1
33058;;Blaye;BLAYE;33390;33;GIRONDE;1
33059;;Blsignac;BLESIGNAC;33670;33;GIRONDE;1
33060;;Bommes;BOMMES;33210;33;GIRONDE;1
33061;;Bonnetan;BONNETAN;33370;33;GIRONDE;1
33062;;Bonzac;BONZAC;33910;33;GIRONDE;1
33063;;Bordeaux;BORDEAUX;33000;33;GIRONDE;1
33064;;Bossugan;BOSSUGAN;33350;33;GIRONDE;1
33065;;Bouliac;BOULIAC;33270;33;GIRONDE;1
33066;;Bourdelles;BOURDELLES;33190;33;GIRONDE;1
33067;;Bourg;BOURG;33710;33;GIRONDE;1
33068;;Bourideys;BOURIDEYS;33113;33;GIRONDE;1
33069;Le;Bouscat;BOUSCAT;33110;33;GIRONDE;1
33070;;Brach;BRACH;33480;33;GIRONDE;1
33071;;Branne;BRANNE;33420;33;GIRONDE;1
33072;;Brannens;BRANNENS;33124;33;GIRONDE;1
33073;;Braud-et-Saint-Louis;BRAUD-ET-SAINT-LOUIS;33820;33;GIRONDE;1
33074;;Brouqueyran;BROUQUEYRAN;33124;33;GIRONDE;1
33075;;Bruges;BRUGES;33520;33;GIRONDE;1
33076;;Budos;BUDOS;33720;33;GIRONDE;1
33077;;Cabanac-et-Villagrains;CABANAC-ET-VILLAGRAINS;33650;33;GIRONDE;1
33078;;Cabara;CABARA;33420;33;GIRONDE;1
33079;;Cadarsac;CADARSAC;33750;33;GIRONDE;1
33080;;Cadaujac;CADAUJAC;33140;33;GIRONDE;1
33081;;Cadillac;CADILLAC;33410;33;GIRONDE;1
33082;;Cadillac-en-Fronsadais;CADILLAC-EN-FRONSADAIS;33240;33;GIRONDE;1
33083;;Camarsac;CAMARSAC;33750;33;GIRONDE;1
33084;;Cambes;CAMBES;33880;33;GIRONDE;1
33085;;Camblanes-et-Meynac;CAMBLANES-ET-MEYNAC;33360;33;GIRONDE;1
33086;;Camiac-et-Saint-Denis;CAMIAC-ET-SAINT-DENIS;33420;33;GIRONDE;1
33087;;Camiran;CAMIRAN;33190;33;GIRONDE;1
33088;;Camps-sur-l'Isle;CAMPS-SUR-L'ISLE;33660;33;GIRONDE;1
33089;;Campugnan;CAMPUGNAN;33390;33;GIRONDE;1
33090;;Canjan;CANEJAN;33610;33;GIRONDE;1
33091;;Cantenac;CANTENAC;33460;33;GIRONDE;1
33092;;Cantois;CANTOIS;33760;33;GIRONDE;1
33093;;Capian;CAPIAN;33550;33;GIRONDE;1
33094;;Caplong;CAPLONG;33220;33;GIRONDE;1
33095;;Captieux;CAPTIEUX;33840;33;GIRONDE;1
33096;;Carbon-Blanc;CARBON-BLANC;33560;33;GIRONDE;1
33097;;Carcans;CARCANS;33121;33;GIRONDE;1
33098;;Cardan;CARDAN;33410;33;GIRONDE;1
33099;;Carignan-de-Bordeaux;CARIGNAN-DE-BORDEAUX;33360;33;GIRONDE;1
33100;;Cars;CARS;33390;33;GIRONDE;1
33101;;Cartelgue;CARTELEGUE;33390;33;GIRONDE;1
33102;;Casseuil;CASSEUIL;33190;33;GIRONDE;1
33103;;Castelmoron-d'Albret;CASTELMORON-D'ALBRET;33540;33;GIRONDE;1
33104;;Castelnau-de-Mdoc;CASTELNAU-DE-MEDOC;33480;33;GIRONDE;1
33105;;Castelviel;CASTELVIEL;33540;33;GIRONDE;1
33106;;Castets-en-Dorthe;CASTETS-EN-DORTHE;33210;33;GIRONDE;1
33107;;Castillon-de-Castets;CASTILLON-DE-CASTETS;33210;33;GIRONDE;1
33108;;Castillon-la-Bataille;CASTILLON-LA-BATAILLE;33350;33;GIRONDE;1
33109;;Castres-Gironde;CASTRES-GIRONDE;33640;33;GIRONDE;1
33111;;Caudrot;CAUDROT;33490;33;GIRONDE;1
33112;;Caumont;CAUMONT;33540;33;GIRONDE;1
33113;;Cauvignac;CAUVIGNAC;33690;33;GIRONDE;1
33114;;Cavignac;CAVIGNAC;33620;33;GIRONDE;1
33115;;Cazalis;CAZALIS;33113;33;GIRONDE;1
33116;;Cazats;CAZATS;33430;33;GIRONDE;1
33117;;Cazaugitat;CAZAUGITAT;33790;33;GIRONDE;1
33118;;Cnac;CENAC;33360;33;GIRONDE;1
33119;;Cenon;CENON;33150;33;GIRONDE;1
33120;;Crons;CERONS;33720;33;GIRONDE;1
33121;;Cessac;CESSAC;33760;33;GIRONDE;1
33122;;Cestas;CESTAS;33610;33;GIRONDE;1
33123;;Czac;CEZAC;33620;33;GIRONDE;1
33124;;Chamadelle;CHAMADELLE;33230;33;GIRONDE;1
33125;;Cissac-Mdoc;CISSAC-MEDOC;33250;33;GIRONDE;1
33126;;Civrac-de-Blaye;CIVRAC-DE-BLAYE;33920;33;GIRONDE;1
33127;;Civrac-sur-Dordogne;CIVRAC-SUR-DORDOGNE;33350;33;GIRONDE;1
33128;;Civrac-en-Mdoc;CIVRAC-EN-MEDOC;33340;33;GIRONDE;1
33129;;Cleyrac;CLEYRAC;33540;33;GIRONDE;1
33130;;Coimres;COIMERES;33210;33;GIRONDE;1
33131;;Coirac;COIRAC;33540;33;GIRONDE;1
33132;;Comps;COMPS;33710;33;GIRONDE;1
33133;;Coubeyrac;COUBEYRAC;33890;33;GIRONDE;1
33134;;Couquques;COUQUEQUES;33340;33;GIRONDE;1
33135;;Courpiac;COURPIAC;33760;33;GIRONDE;1
33136;;Cours-de-Monsgur;COURS-DE-MONSEGUR;33580;33;GIRONDE;1
33137;;Cours-les-Bains;COURS-LES-BAINS;33690;33;GIRONDE;1
33138;;Coutras;COUTRAS;33230;33;GIRONDE;1
33139;;Coutures;COUTURES;33580;33;GIRONDE;1
33140;;Cron;CREON;33670;33;GIRONDE;1
33141;;Croignon;CROIGNON;33750;33;GIRONDE;1
33142;;Cubnezais;CUBNEZAIS;33620;33;GIRONDE;1
33143;;Cubzac-les-Ponts;CUBZAC-LES-PONTS;33240;33;GIRONDE;1
33144;;Cudos;CUDOS;33430;33;GIRONDE;1
33145;;Cursan;CURSAN;33670;33;GIRONDE;1
33146;;Cussac-Fort-Mdoc;CUSSAC-FORT-MEDOC;33460;33;GIRONDE;1
33147;;Daignac;DAIGNAC;33420;33;GIRONDE;1
33148;;Dardenac;DARDENAC;33420;33;GIRONDE;1
33149;;Daubze;DAUBEZE;33540;33;GIRONDE;1
33150;;Dieulivol;DIEULIVOL;33580;33;GIRONDE;1
33151;;Donnezac;DONNEZAC;33860;33;GIRONDE;1
33152;;Donzac;DONZAC;33410;33;GIRONDE;1
33153;;Doulezon;DOULEZON;33350;33;GIRONDE;1
33154;Les;glisottes-et-Chalaures;EGLISOTTES-ET-CHALAURES;33230;33;GIRONDE;1
33155;;Escaudes;ESCAUDES;33840;33;GIRONDE;1
33156;;Escoussans;ESCOUSSANS;33760;33;GIRONDE;1
33157;;Espiet;ESPIET;33420;33;GIRONDE;1
33158;Les;Esseintes;ESSEINTES;33190;33;GIRONDE;1
33159;;tauliers;ETAULIERS;33820;33;GIRONDE;1
33160;;Eynesse;EYNESSE;33220;33;GIRONDE;1
33161;;Eyrans;EYRANS;33390;33;GIRONDE;1
33162;;Eysines;EYSINES;33320;33;GIRONDE;1
33163;;Faleyras;FALEYRAS;33760;33;GIRONDE;1
33164;;Fargues;FARGUES;33210;33;GIRONDE;1
33165;;Fargues-Saint-Hilaire;FARGUES-SAINT-HILAIRE;33370;33;GIRONDE;1
33166;Le;Fieu;FIEU;33230;33;GIRONDE;1
33167;;Floirac;FLOIRAC;33270;33;GIRONDE;1
33168;;Flaujagues;FLAUJAGUES;33350;33;GIRONDE;1
33169;;Flouds;FLOUDES;33190;33;GIRONDE;1
33170;;Fontet;FONTET;33190;33;GIRONDE;1
33171;;Fosss-et-Baleyssac;FOSSES-ET-BALEYSSAC;33190;33;GIRONDE;1
33172;;Fours;FOURS;33390;33;GIRONDE;1
33173;;Francs;FRANCS;33570;33;GIRONDE;1
33174;;Fronsac;FRONSAC;33126;33;GIRONDE;1
33175;;Frontenac;FRONTENAC;33760;33;GIRONDE;1
33176;;Gabarnac;GABARNAC;33410;33;GIRONDE;1
33177;;Gaillan-en-Mdoc;GAILLAN-EN-MEDOC;33340;33;GIRONDE;1
33178;;Gajac;GAJAC;33430;33;GIRONDE;1
33179;;Galgon;GALGON;33133;33;GIRONDE;1
33180;;Gans;GANS;33430;33;GIRONDE;1
33181;;Gardegan-et-Tourtirac;GARDEGAN-ET-TOURTIRAC;33350;33;GIRONDE;1
33182;;Gauriac;GAURIAC;33710;33;GIRONDE;1
33183;;Gauriaguet;GAURIAGUET;33240;33;GIRONDE;1
33184;;Gnrac;GENERAC;33920;33;GIRONDE;1
33185;;Gnissac;GENISSAC;33420;33;GIRONDE;1
33186;;Gensac;GENSAC;33890;33;GIRONDE;1
33187;;Gironde-sur-Dropt;GIRONDE-SUR-DROPT;33190;33;GIRONDE;1
33188;;Giscos;GISCOS;33840;33;GIRONDE;1
33189;;Gornac;GORNAC;33540;33;GIRONDE;1
33190;;Goualade;GOUALADE;33840;33;GIRONDE;1
33191;;Gours;GOURS;33660;33;GIRONDE;1
33192;;Gradignan;GRADIGNAN;33170;33;GIRONDE;1
33193;;Grayan-et-l'Hpital;GRAYAN-ET-L'HOPITAL;33590;33;GIRONDE;1
33194;;Grzillac;GREZILLAC;33420;33;GIRONDE;1
33195;;Grignols;GRIGNOLS;33690;33;GIRONDE;1
33196;;Guillac;GUILLAC;33420;33;GIRONDE;1
33197;;Guillos;GUILLOS;33720;33;GIRONDE;1
33198;;Gutres;GUITRES;33230;33;GIRONDE;1
33199;;Gujan-Mestras;GUJAN-MESTRAS;33470;33;GIRONDE;1
33200;Le;Haillan;HAILLAN;33185;33;GIRONDE;1
33201;;Haux;HAUX;33550;33;GIRONDE;1
33202;;Hostens;HOSTENS;33125;33;GIRONDE;1
33203;;Hourtin;HOURTIN;33990;33;GIRONDE;1
33204;;Hure;HURE;33190;33;GIRONDE;1
33205;;Illats;ILLATS;33720;33;GIRONDE;1
33206;;Isle-Saint-Georges;ISLE-SAINT-GEORGES;33640;33;GIRONDE;1
33207;;Izon;IZON;33450;33;GIRONDE;1
33208;;Jau-Dignac-et-Loirac;JAU-DIGNAC-ET-LOIRAC;33590;33;GIRONDE;1
33209;;Jugazan;JUGAZAN;33420;33;GIRONDE;1
33210;;Juillac;JUILLAC;33890;33;GIRONDE;1
33211;;Labarde;LABARDE;33460;33;GIRONDE;1
33212;;Labescau;LABESCAU;33690;33;GIRONDE;1
33213;La;Brde;BREDE;33650;33;GIRONDE;1
33214;;Lacanau;LACANAU;33680;33;GIRONDE;1
33215;;Ladaux;LADAUX;33760;33;GIRONDE;1
33216;;Lados;LADOS;33124;33;GIRONDE;1
33218;;Lagorce;LAGORCE;33230;33;GIRONDE;1
33219;La;Lande-de-Fronsac;LANDE-DE-FRONSAC;33240;33;GIRONDE;1
33220;;Lamarque;LAMARQUE;33460;33;GIRONDE;1
33221;;Lamothe-Landerron;LAMOTHE-LANDERRON;33190;33;GIRONDE;1
33222;;Lalande-de-Pomerol;LALANDE-DE-POMEROL;33500;33;GIRONDE;1
33223;;Landerrouat;LANDERROUAT;33790;33;GIRONDE;1
33224;;Landerrouet-sur-Sgur;LANDERROUET-SUR-SEGUR;33540;33;GIRONDE;1
33225;;Landiras;LANDIRAS;33720;33;GIRONDE;1
33226;;Langoiran;LANGOIRAN;33550;33;GIRONDE;1
33227;;Langon;LANGON;33210;33;GIRONDE;1
33228;;Lansac;LANSAC;33710;33;GIRONDE;1
33229;;Lanton;LANTON;33138;33;GIRONDE;1
33230;;Lapouyade;LAPOUYADE;33620;33;GIRONDE;1
33231;;Laroque;LAROQUE;33410;33;GIRONDE;1
33232;;Lartigue;LARTIGUE;33840;33;GIRONDE;1
33233;;Laruscade;LARUSCADE;33620;33;GIRONDE;1
33234;;Latresne;LATRESNE;33360;33;GIRONDE;1
33235;;Lavazan;LAVAZAN;33690;33;GIRONDE;1
33236;;Lge-Cap-Ferret;LEGE-CAP-FERRET;33950;33;GIRONDE;1
33237;;Logeats;LEOGEATS;33210;33;GIRONDE;1
33238;;Lognan;LEOGNAN;33850;33;GIRONDE;1
33239;;Lerm-et-Musset;LERM-ET-MUSSET;33840;33;GIRONDE;1
33240;;Lesparre-Mdoc;LESPARRE-MEDOC;33340;33;GIRONDE;1
33241;;Lestiac-sur-Garonne;LESTIAC-SUR-GARONNE;33550;33;GIRONDE;1
33242;Les;Lves-et-Thoumeyragues;LEVES-ET-THOUMEYRAGUES;33220;33;GIRONDE;1
33243;;Libourne;LIBOURNE;33500;33;GIRONDE;1
33244;;Lignan-de-Bazas;LIGNAN-DE-BAZAS;33430;33;GIRONDE;1
33245;;Lignan-de-Bordeaux;LIGNAN-DE-BORDEAUX;33360;33;GIRONDE;1
33246;;Ligueux;LIGUEUX;33220;33;GIRONDE;1
33247;;Listrac-de-Durze;LISTRAC-DE-DUREZE;33790;33;GIRONDE;1
33248;;Listrac-Mdoc;LISTRAC-MEDOC;33480;33;GIRONDE;1
33249;;Lormont;LORMONT;33310;33;GIRONDE;1
33250;;Loubens;LOUBENS;33190;33;GIRONDE;1
33251;;Louchats;LOUCHATS;33125;33;GIRONDE;1
33252;;Loupes;LOUPES;33370;33;GIRONDE;1
33253;;Loupiac;LOUPIAC;33410;33;GIRONDE;1
33254;;Loupiac-de-la-Role;LOUPIAC-DE-LA-REOLE;33190;33;GIRONDE;1
33255;;Lucmau;LUCMAU;33840;33;GIRONDE;1
33256;;Ludon-Mdoc;LUDON-MEDOC;33290;33;GIRONDE;1
33257;;Lugaignac;LUGAIGNAC;33420;33;GIRONDE;1
33258;;Lugasson;LUGASSON;33760;33;GIRONDE;1
33259;;Lugon-et-l'le-du-Carnay;LUGON-ET-L'ILE-DU-CARNAY;33240;33;GIRONDE;1
33260;;Lugos;LUGOS;33830;33;GIRONDE;1
33261;;Lussac;LUSSAC;33570;33;GIRONDE;1
33262;;Macau;MACAU;33460;33;GIRONDE;1
33263;;Madirac;MADIRAC;33670;33;GIRONDE;1
33264;;Maransin;MARANSIN;33230;33;GIRONDE;1
33266;;Marcenais;MARCENAIS;33620;33;GIRONDE;1
33267;;Marcillac;MARCILLAC;33860;33;GIRONDE;1
33268;;Margaux;MARGAUX;33460;33;GIRONDE;1
33269;;Margueron;MARGUERON;33220;33;GIRONDE;1
33270;;Marimbault;MARIMBAULT;33430;33;GIRONDE;1
33271;;Marions;MARIONS;33690;33;GIRONDE;1
33272;;Marsas;MARSAS;33620;33;GIRONDE;1
33273;;Martignas-sur-Jalle;MARTIGNAS-SUR-JALLE;33127;33;GIRONDE;1
33274;;Martillac;MARTILLAC;33650;33;GIRONDE;1
33275;;Martres;MARTRES;33760;33;GIRONDE;1
33276;;Masseilles;MASSEILLES;33690;33;GIRONDE;1
33277;;Massugas;MASSUGAS;33790;33;GIRONDE;1
33278;;Mauriac;MAURIAC;33540;33;GIRONDE;1
33279;;Mazres;MAZERES;33210;33;GIRONDE;1
33280;;Mazion;MAZION;33390;33;GIRONDE;1
33281;;Mrignac;MERIGNAC;33700;33;GIRONDE;1
33282;;Mrignas;MERIGNAS;33350;33;GIRONDE;1
33283;;Mesterrieux;MESTERRIEUX;33540;33;GIRONDE;1
33284;;Mios;MIOS;33380;33;GIRONDE;1
33285;;Mombrier;MOMBRIER;33710;33;GIRONDE;1
33287;;Mongauzy;MONGAUZY;33190;33;GIRONDE;1
33288;;Monprimblanc;MONPRIMBLANC;33410;33;GIRONDE;1
33289;;Monsgur;MONSEGUR;33580;33;GIRONDE;1
33290;;Montagne;MONTAGNE;33570;33;GIRONDE;1
33291;;Montagoudin;MONTAGOUDIN;33190;33;GIRONDE;1
33292;;Montignac;MONTIGNAC;33760;33;GIRONDE;1
33293;;Montussan;MONTUSSAN;33450;33;GIRONDE;1
33294;;Morizs;MORIZES;33190;33;GIRONDE;1
33295;;Mouillac;MOUILLAC;33240;33;GIRONDE;1
33296;;Mouliets-et-Villemartin;MOULIETS-ET-VILLEMARTIN;33350;33;GIRONDE;1
33297;;Moulis-en-Mdoc;MOULIS-EN-MEDOC;33480;33;GIRONDE;1
33298;;Moulon;MOULON;33420;33;GIRONDE;1
33299;;Mourens;MOURENS;33410;33;GIRONDE;1
33300;;Naujac-sur-Mer;NAUJAC-SUR-MER;33990;33;GIRONDE;1
33301;;Naujan-et-Postiac;NAUJAN-ET-POSTIAC;33420;33;GIRONDE;1
33302;;Nac;NEAC;33500;33;GIRONDE;1
33303;;Nrigean;NERIGEAN;33750;33;GIRONDE;1
33304;;Neuffons;NEUFFONS;33580;33;GIRONDE;1
33305;Le;Nizan;NIZAN;33430;33;GIRONDE;1
33306;;Noaillac;NOAILLAC;33190;33;GIRONDE;1
33307;;Noaillan;NOAILLAN;33730;33;GIRONDE;1
33308;;Omet;OMET;33410;33;GIRONDE;1
33309;;Ordonnac;ORDONNAC;33340;33;GIRONDE;1
33310;;Origne;ORIGNE;33113;33;GIRONDE;1
33311;;Paillet;PAILLET;33550;33;GIRONDE;1
33312;;Parempuyre;PAREMPUYRE;33290;33;GIRONDE;1
33314;;Pauillac;PAUILLAC;33250;33;GIRONDE;1
33315;Les;Peintures;PEINTURES;33230;33;GIRONDE;1
33316;;Pellegrue;PELLEGRUE;33790;33;GIRONDE;1
33317;;Prissac;PERISSAC;33240;33;GIRONDE;1
33318;;Pessac;PESSAC;33600;33;GIRONDE;1
33319;;Pessac-sur-Dordogne;PESSAC-SUR-DORDOGNE;33890;33;GIRONDE;1
33320;;Petit-Palais-et-Cornemps;PETIT-PALAIS-ET-CORNEMPS;33570;33;GIRONDE;1
33321;;Peujard;PEUJARD;33240;33;GIRONDE;1
33322;Le;Pian-Mdoc;PIAN-MEDOC;33290;33;GIRONDE;1
33323;Le;Pian-sur-Garonne;PIAN-SUR-GARONNE;33490;33;GIRONDE;1
33324;;Pineuilh;PINEUILH;33220;33;GIRONDE;1
33325;;Plassac;PLASSAC;33390;33;GIRONDE;1
33326;;Pleine-Selve;PLEINE-SELVE;33820;33;GIRONDE;1
33327;;Podensac;PODENSAC;33720;33;GIRONDE;1
33328;;Pomerol;POMEROL;33500;33;GIRONDE;1
33329;;Pompjac;POMPEJAC;33730;33;GIRONDE;1
33330;;Pompignac;POMPIGNAC;33370;33;GIRONDE;1
33331;;Pondaurat;PONDAURAT;33190;33;GIRONDE;1
33332;;Porchres;PORCHERES;33660;33;GIRONDE;1
33333;Le;Porge;PORGE;33680;33;GIRONDE;1
33334;;Portets;PORTETS;33640;33;GIRONDE;1
33335;Le;Pout;POUT;33670;33;GIRONDE;1
33336;;Prchac;PRECHAC;33730;33;GIRONDE;1
33337;;Preignac;PREIGNAC;33210;33;GIRONDE;1
33338;;Prignac-en-Mdoc;PRIGNAC-EN-MEDOC;33340;33;GIRONDE;1
33339;;Prignac-et-Marcamps;PRIGNAC-ET-MARCAMPS;33710;33;GIRONDE;1
33341;;Pugnac;PUGNAC;33710;33;GIRONDE;1
33342;;Puisseguin;PUISSEGUIN;33570;33;GIRONDE;1
33343;;Pujols-sur-Ciron;PUJOLS-SUR-CIRON;33210;33;GIRONDE;1
33344;;Pujols;PUJOLS;33350;33;GIRONDE;1
33345;Le;Puy;PUY;33580;33;GIRONDE;1
33346;;Puybarban;PUYBARBAN;33190;33;GIRONDE;1
33347;;Puynormand;PUYNORMAND;33660;33;GIRONDE;1
33348;;Queyrac;QUEYRAC;33340;33;GIRONDE;1
33349;;Quinsac;QUINSAC;33360;33;GIRONDE;1
33350;;Rauzan;RAUZAN;33420;33;GIRONDE;1
33351;;Reignac;REIGNAC;33860;33;GIRONDE;1
33352;La;Role;REOLE;33190;33;GIRONDE;1
33353;;Rimons;RIMONS;33580;33;GIRONDE;1
33354;;Riocaud;RIOCAUD;33220;33;GIRONDE;1
33355;;Rions;RIONS;33410;33;GIRONDE;1
33356;La;Rivire;RIVIERE;33126;33;GIRONDE;1
33357;;Roaillan;ROAILLAN;33210;33;GIRONDE;1
33358;;Romagne;ROMAGNE;33760;33;GIRONDE;1
33359;;Roquebrune;ROQUEBRUNE;33580;33;GIRONDE;1
33360;La;Roquille;ROQUILLE;33220;33;GIRONDE;1
33361;;Ruch;RUCH;33350;33;GIRONDE;1
33362;;Sablons;SABLONS;33910;33;GIRONDE;1
33363;;Sadirac;SADIRAC;33670;33;GIRONDE;1
33364;;Saillans;SAILLANS;33141;33;GIRONDE;1
33365;;Saint-Aignan;SAINT-AIGNAN;33126;33;GIRONDE;1
33366;;Saint-Andr-de-Cubzac;SAINT-ANDRE-DE-CUBZAC;33240;33;GIRONDE;1
33367;;Saint-Andr-du-Bois;SAINT-ANDRE-DU-BOIS;33490;33;GIRONDE;1
33369;;Saint-Andr-et-Appelles;SAINT-ANDRE-ET-APPELLES;33220;33;GIRONDE;1
33370;;Saint-Androny;SAINT-ANDRONY;33390;33;GIRONDE;1
33371;;Saint-Antoine;SAINT-ANTOINE;33240;33;GIRONDE;1
33372;;Saint-Antoine-du-Queyret;SAINT-ANTOINE-DU-QUEYRET;33790;33;GIRONDE;1
33373;;Saint-Antoine-sur-l'Isle;SAINT-ANTOINE-SUR-L'ISLE;33660;33;GIRONDE;1
33374;;Saint-Aubin-de-Blaye;SAINT-AUBIN-DE-BLAYE;33820;33;GIRONDE;1
33375;;Saint-Aubin-de-Branne;SAINT-AUBIN-DE-BRANNE;33420;33;GIRONDE;1
33376;;Saint-Aubin-de-Mdoc;SAINT-AUBIN-DE-MEDOC;33160;33;GIRONDE;1
33377;;Saint-Avit-de-Soulge;SAINT-AVIT-DE-SOULEGE;33220;33;GIRONDE;1
33378;;Saint-Avit-Saint-Nazaire;SAINT-AVIT-SAINT-NAZAIRE;33220;33;GIRONDE;1
33379;;Saint-Brice;SAINT-BRICE;33540;33;GIRONDE;1
33380;;Saint-Caprais-de-Blaye;SAINT-CAPRAIS-DE-BLAYE;33820;33;GIRONDE;1
33381;;Saint-Caprais-de-Bordeaux;SAINT-CAPRAIS-DE-BORDEAUX;33880;33;GIRONDE;1
33382;;Saint-Christoly-de-Blaye;SAINT-CHRISTOLY-DE-BLAYE;33920;33;GIRONDE;1
33383;;Saint-Christoly-Mdoc;SAINT-CHRISTOLY-MEDOC;33340;33;GIRONDE;1
33384;;Saint-Christophe-des-Bardes;SAINT-CHRISTOPHE-DES-BARDES;33330;33;GIRONDE;1
33385;;Saint-Christophe-de-Double;SAINT-CHRISTOPHE-DE-DOUBLE;33230;33;GIRONDE;1
33386;;Saint-Cibard;SAINT-CIBARD;33570;33;GIRONDE;1
33387;;Saint-Ciers-d'Abzac;SAINT-CIERS-D'ABZAC;33910;33;GIRONDE;1
33388;;Saint-Ciers-de-Canesse;SAINT-CIERS-DE-CANESSE;33710;33;GIRONDE;1
33389;;Saint-Ciers-sur-Gironde;SAINT-CIERS-SUR-GIRONDE;33820;33;GIRONDE;1
33390;;Sainte-Colombe;SAINTE-COLOMBE;33350;33;GIRONDE;1
33391;;Saint-Cme;SAINT-COME;33430;33;GIRONDE;1
33392;;Sainte-Croix-du-Mont;SAINTE-CROIX-DU-MONT;33410;33;GIRONDE;1
33393;;Saint-Denis-de-Pile;SAINT-DENIS-DE-PILE;33910;33;GIRONDE;1
33394;;Saint-milion;SAINT-EMILION;33330;33;GIRONDE;1
33395;;Saint-Estphe;SAINT-ESTEPHE;33180;33;GIRONDE;1
33396;;Saint-tienne-de-Lisse;SAINT-ETIENNE-DE-LISSE;33330;33;GIRONDE;1
33397;;Sainte-Eulalie;SAINTE-EULALIE;33560;33;GIRONDE;1
33398;;Saint-Exupry;SAINT-EXUPERY;33190;33;GIRONDE;1
33399;;Saint-Flix-de-Foncaude;SAINT-FELIX-DE-FONCAUDE;33540;33;GIRONDE;1
33400;;Saint-Ferme;SAINT-FERME;33580;33;GIRONDE;1
33401;;Sainte-Florence;SAINTE-FLORENCE;33350;33;GIRONDE;1
33402;;Sainte-Foy-la-Grande;SAINTE-FOY-LA-GRANDE;33220;33;GIRONDE;1
33403;;Sainte-Foy-la-Longue;SAINTE-FOY-LA-LONGUE;33490;33;GIRONDE;1
33404;;Sainte-Gemme;SAINTE-GEMME;33580;33;GIRONDE;1
33405;;Saint-Gens-de-Blaye;SAINT-GENES-DE-BLAYE;33390;33;GIRONDE;1
33406;;Saint-Gens-de-Castillon;SAINT-GENES-DE-CASTILLON;33350;33;GIRONDE;1
33407;;Saint-Gens-de-Fronsac;SAINT-GENES-DE-FRONSAC;33240;33;GIRONDE;1
33408;;Saint-Gens-de-Lombaud;SAINT-GENES-DE-LOMBAUD;33670;33;GIRONDE;1
33409;;Saint-Genis-du-Bois;SAINT-GENIS-DU-BOIS;33760;33;GIRONDE;1
33411;;Saint-Germain-de-Grave;SAINT-GERMAIN-DE-GRAVE;33490;33;GIRONDE;1
33412;;Saint-Germain-d'Esteuil;SAINT-GERMAIN-D'ESTEUIL;33340;33;GIRONDE;1
33413;;Saint-Germain-du-Puch;SAINT-GERMAIN-DU-PUCH;33750;33;GIRONDE;1
33414;;Saint-Germain-de-la-Rivire;SAINT-GERMAIN-DE-LA-RIVIERE;33240;33;GIRONDE;1
33415;;Saint-Gervais;SAINT-GERVAIS;33240;33;GIRONDE;1
33416;;Saint-Girons-d'Aiguevives;SAINT-GIRONS-D'AIGUEVIVES;33920;33;GIRONDE;1
33417;;Sainte-Hlne;SAINTE-HELENE;33480;33;GIRONDE;1
33418;;Saint-Hilaire-de-la-Noaille;SAINT-HILAIRE-DE-LA-NOAILLE;33190;33;GIRONDE;1
33419;;Saint-Hilaire-du-Bois;SAINT-HILAIRE-DU-BOIS;33540;33;GIRONDE;1
33420;;Saint-Hippolyte;SAINT-HIPPOLYTE;33330;33;GIRONDE;1
33421;;Saint-Jean-de-Blaignac;SAINT-JEAN-DE-BLAIGNAC;33420;33;GIRONDE;1
33422;;Saint-Jean-d'Illac;SAINT-JEAN-D'ILLAC;33127;33;GIRONDE;1
33423;;Saint-Julien-Beychevelle;SAINT-JULIEN-BEYCHEVELLE;33250;33;GIRONDE;1
33424;;Saint-Laurent-Mdoc;SAINT-LAURENT-MEDOC;33112;33;GIRONDE;1
33425;;Saint-Laurent-d'Arce;SAINT-LAURENT-D'ARCE;33240;33;GIRONDE;1
33426;;Saint-Laurent-des-Combes;SAINT-LAURENT-DES-COMBES;33330;33;GIRONDE;1
33427;;Saint-Laurent-du-Bois;SAINT-LAURENT-DU-BOIS;33540;33;GIRONDE;1
33428;;Saint-Laurent-du-Plan;SAINT-LAURENT-DU-PLAN;33190;33;GIRONDE;1
33429;;Saint-Lger-de-Balson;SAINT-LEGER-DE-BALSON;33113;33;GIRONDE;1
33431;;Saint-Lon;SAINT-LEON;33670;33;GIRONDE;1
33432;;Saint-Loubert;SAINT-LOUBERT;33210;33;GIRONDE;1
33433;;Saint-Loubs;SAINT-LOUBES;33450;33;GIRONDE;1
33434;;Saint-Louis-de-Montferrand;SAINT-LOUIS-DE-MONTFERRAND;33440;33;GIRONDE;1
33435;;Saint-Macaire;SAINT-MACAIRE;33490;33;GIRONDE;1
33436;;Saint-Magne;SAINT-MAGNE;33125;33;GIRONDE;1
33437;;Saint-Magne-de-Castillon;SAINT-MAGNE-DE-CASTILLON;33350;33;GIRONDE;1
33438;;Saint-Maixant;SAINT-MAIXANT;33490;33;GIRONDE;1
33439;;Saint-Mariens;SAINT-MARIENS;33620;33;GIRONDE;1
33440;;Saint-Martial;SAINT-MARTIAL;33490;33;GIRONDE;1
33441;;Saint-Martin-Lacaussade;SAINT-MARTIN-LACAUSSADE;33390;33;GIRONDE;1
33442;;Saint-Martin-de-Laye;SAINT-MARTIN-DE-LAYE;33910;33;GIRONDE;1
33443;;Saint-Martin-de-Lerm;SAINT-MARTIN-DE-LERM;33540;33;GIRONDE;1
33444;;Saint-Martin-de-Sescas;SAINT-MARTIN-DE-SESCAS;33490;33;GIRONDE;1
33445;;Saint-Martin-du-Bois;SAINT-MARTIN-DU-BOIS;33910;33;GIRONDE;1
33446;;Saint-Martin-du-Puy;SAINT-MARTIN-DU-PUY;33540;33;GIRONDE;1
33447;;Saint-Mdard-de-Guizires;SAINT-MEDARD-DE-GUIZIERES;33230;33;GIRONDE;1
33448;;Saint-Mdard-d'Eyrans;SAINT-MEDARD-D'EYRANS;33650;33;GIRONDE;1
33449;;Saint-Mdard-en-Jalles;SAINT-MEDARD-EN-JALLES;33160;33;GIRONDE;1
33450;;Saint-Michel-de-Castelnau;SAINT-MICHEL-DE-CASTELNAU;33840;33;GIRONDE;1
33451;;Saint-Michel-de-Fronsac;SAINT-MICHEL-DE-FRONSAC;33126;33;GIRONDE;1
33452;;Saint-Michel-de-Rieufret;SAINT-MICHEL-DE-RIEUFRET;33720;33;GIRONDE;1
33453;;Saint-Michel-de-Lapujade;SAINT-MICHEL-DE-LAPUJADE;33190;33;GIRONDE;1
33454;;Saint-Morillon;SAINT-MORILLON;33650;33;GIRONDE;1
33456;;Saint-Palais;SAINT-PALAIS;33820;33;GIRONDE;1
33457;;Saint-Pardon-de-Conques;SAINT-PARDON-DE-CONQUES;33210;33;GIRONDE;1
33458;;Saint-Paul;SAINT-PAUL;33390;33;GIRONDE;1
33459;;Saint-Pey-d'Armens;SAINT-PEY-D'ARMENS;33330;33;GIRONDE;1
33460;;Saint-Pey-de-Castets;SAINT-PEY-DE-CASTETS;33350;33;GIRONDE;1
33461;;Saint-Philippe-d'Aiguille;SAINT-PHILIPPE-D'AIGUILLE;33350;33;GIRONDE;1
33462;;Saint-Philippe-du-Seignal;SAINT-PHILIPPE-DU-SEIGNAL;33220;33;GIRONDE;1
33463;;Saint-Pierre-d'Aurillac;SAINT-PIERRE-D'AURILLAC;33490;33;GIRONDE;1
33464;;Saint-Pierre-de-Bat;SAINT-PIERRE-DE-BAT;33760;33;GIRONDE;1
33465;;Saint-Pierre-de-Mons;SAINT-PIERRE-DE-MONS;33210;33;GIRONDE;1
33466;;Saint-Quentin-de-Baron;SAINT-QUENTIN-DE-BARON;33750;33;GIRONDE;1
33467;;Saint-Quentin-de-Caplong;SAINT-QUENTIN-DE-CAPLONG;33220;33;GIRONDE;1
33468;;Sainte-Radegonde;SAINTE-RADEGONDE;33350;33;GIRONDE;1
33470;;Saint-Romain-la-Virve;SAINT-ROMAIN-LA-VIRVEE;33240;33;GIRONDE;1
33471;;Saint-Sauveur;SAINT-SAUVEUR;33250;33;GIRONDE;1
33472;;Saint-Sauveur-de-Puynormand;SAINT-SAUVEUR-DE-PUYNORMAND;33660;33;GIRONDE;1
33473;;Saint-Savin;SAINT-SAVIN;33920;33;GIRONDE;1
33474;;Saint-Selve;SAINT-SELVE;33650;33;GIRONDE;1
33475;;Saint-Seurin-de-Bourg;SAINT-SEURIN-DE-BOURG;33710;33;GIRONDE;1
33476;;Saint-Seurin-de-Cadourne;SAINT-SEURIN-DE-CADOURNE;33180;33;GIRONDE;1
33477;;Saint-Seurin-de-Cursac;SAINT-SEURIN-DE-CURSAC;33390;33;GIRONDE;1
33478;;Saint-Seurin-sur-l'Isle;SAINT-SEURIN-SUR-L'ISLE;33660;33;GIRONDE;1
33479;;Saint-Sve;SAINT-SEVE;33190;33;GIRONDE;1
33480;;Saint-Sulpice-de-Faleyrens;SAINT-SULPICE-DE-FALEYRENS;33330;33;GIRONDE;1
33481;;Saint-Sulpice-de-Guilleragues;SAINT-SULPICE-DE-GUILLERAGUES;33580;33;GIRONDE;1
33482;;Saint-Sulpice-de-Pommiers;SAINT-SULPICE-DE-POMMIERS;33540;33;GIRONDE;1
33483;;Saint-Sulpice-et-Cameyrac;SAINT-SULPICE-ET-CAMEYRAC;33450;33;GIRONDE;1
33484;;Saint-Symphorien;SAINT-SYMPHORIEN;33113;33;GIRONDE;1
33485;;Sainte-Terre;SAINTE-TERRE;33350;33;GIRONDE;1
33486;;Saint-Trojan;SAINT-TROJAN;33710;33;GIRONDE;1
33487;;Saint-Vincent-de-Paul;SAINT-VINCENT-DE-PAUL;33440;33;GIRONDE;1
33488;;Saint-Vincent-de-Pertignas;SAINT-VINCENT-DE-PERTIGNAS;33420;33;GIRONDE;1
33489;;Saint-Vivien-de-Blaye;SAINT-VIVIEN-DE-BLAYE;33920;33;GIRONDE;1
33490;;Saint-Vivien-de-Mdoc;SAINT-VIVIEN-DE-MEDOC;33590;33;GIRONDE;1
33491;;Saint-Vivien-de-Monsgur;SAINT-VIVIEN-DE-MONSEGUR;33580;33;GIRONDE;1
33492;;Saint-Yzan-de-Soudiac;SAINT-YZAN-DE-SOUDIAC;33920;33;GIRONDE;1
33493;;Saint-Yzans-de-Mdoc;SAINT-YZANS-DE-MEDOC;33340;33;GIRONDE;1
33494;;Salaunes;SALAUNES;33160;33;GIRONDE;1
33495;;Salignac;SALIGNAC;33240;33;GIRONDE;1
33496;;Sallebuf;SALLEBOEUF;33370;33;GIRONDE;1
33498;;Salles;SALLES;33770;33;GIRONDE;1
33499;Les;Salles-de-Castillon;SALLES-DE-CASTILLON;33350;33;GIRONDE;1
33500;;Samonac;SAMONAC;33710;33;GIRONDE;1
33501;;Saucats;SAUCATS;33650;33;GIRONDE;1
33502;;Saugon;SAUGON;33920;33;GIRONDE;1
33503;;Saumos;SAUMOS;33680;33;GIRONDE;1
33504;;Sauternes;SAUTERNES;33210;33;GIRONDE;1
33505;La;Sauve;SAUVE;33670;33;GIRONDE;1
33506;;Sauveterre-de-Guyenne;SAUVETERRE-DE-GUYENNE;33540;33;GIRONDE;1
33507;;Sauviac;SAUVIAC;33430;33;GIRONDE;1
33508;;Savignac;SAVIGNAC;33124;33;GIRONDE;1
33509;;Savignac-de-l'Isle;SAVIGNAC-DE-L'ISLE;33910;33;GIRONDE;1
33510;;Semens;SEMENS;33490;33;GIRONDE;1
33511;;Sendets;SENDETS;33690;33;GIRONDE;1
33512;;Sigalens;SIGALENS;33690;33;GIRONDE;1
33513;;Sillas;SILLAS;33690;33;GIRONDE;1
33514;;Soulac-sur-Mer;SOULAC-SUR-MER;33780;33;GIRONDE;1
33515;;Soulignac;SOULIGNAC;33760;33;GIRONDE;1
33516;;Soussac;SOUSSAC;33790;33;GIRONDE;1
33517;;Soussans;SOUSSANS;33460;33;GIRONDE;1
33518;;Tabanac;TABANAC;33550;33;GIRONDE;1
33519;Le;Taillan-Mdoc;TAILLAN-MEDOC;33320;33;GIRONDE;1
33520;;Taillecavat;TAILLECAVAT;33580;33;GIRONDE;1
33521;;Talais;TALAIS;33590;33;GIRONDE;1
33522;;Talence;TALENCE;33400;33;GIRONDE;1
33523;;Targon;TARGON;33760;33;GIRONDE;1
33524;;Tarns;TARNES;33240;33;GIRONDE;1
33525;;Tauriac;TAURIAC;33710;33;GIRONDE;1
33526;;Tayac;TAYAC;33570;33;GIRONDE;1
33527;Le;Teich;TEICH;33470;33;GIRONDE;1
33528;Le;Temple;TEMPLE;33680;33;GIRONDE;1
33529;La;Teste-de-Buch;TESTE-DE-BUCH;33260;33;GIRONDE;1
33530;;Teuillac;TEUILLAC;33710;33;GIRONDE;1
33531;;Tizac-de-Curton;TIZAC-DE-CURTON;33420;33;GIRONDE;1
33532;;Tizac-de-Lapouyade;TIZAC-DE-LAPOUYADE;33620;33;GIRONDE;1
33533;;Toulenne;TOULENNE;33210;33;GIRONDE;1
33534;Le;Tourne;TOURNE;33550;33;GIRONDE;1
33535;;Tresses;TRESSES;33370;33;GIRONDE;1
33536;Le;Tuzan;TUZAN;33125;33;GIRONDE;1
33537;;Uzeste;UZESTE;33730;33;GIRONDE;1
33538;;Valeyrac;VALEYRAC;33340;33;GIRONDE;1
33539;;Vayres;VAYRES;33870;33;GIRONDE;1
33540;;Vendays-Montalivet;VENDAYS-MONTALIVET;33930;33;GIRONDE;1
33541;;Vensac;VENSAC;33590;33;GIRONDE;1
33542;;Vrac;VERAC;33240;33;GIRONDE;1
33543;;Verdelais;VERDELAIS;33490;33;GIRONDE;1
33544;Le;Verdon-sur-Mer;VERDON-SUR-MER;33123;33;GIRONDE;1
33545;;Vertheuil;VERTHEUIL;33180;33;GIRONDE;1
33546;;Vignonet;VIGNONET;33330;33;GIRONDE;1
33547;;Villandraut;VILLANDRAUT;33730;33;GIRONDE;1
33548;;Villegouge;VILLEGOUGE;33141;33;GIRONDE;1
33549;;Villenave-de-Rions;VILLENAVE-DE-RIONS;33550;33;GIRONDE;1
33550;;Villenave-d'Ornon;VILLENAVE-D'ORNON;33140;33;GIRONDE;1
33551;;Villeneuve;VILLENEUVE;33710;33;GIRONDE;1
33552;;Virelade;VIRELADE;33720;33;GIRONDE;1
33553;;Virsac;VIRSAC;33240;33;GIRONDE;1
33554;;Yvrac;YVRAC;33370;33;GIRONDE;1
33555;;Marcheprime;MARCHEPRIME;33380;33;GIRONDE;1
34001;;Abeilhan;ABEILHAN;34290;34;HERAULT;1
34002;;Adissan;ADISSAN;34230;34;HERAULT;1
34003;;Agde;AGDE;34300;34;HERAULT;1
34004;;Agel;AGEL;34210;34;HERAULT;1
34005;;Agons;AGONES;34190;34;HERAULT;1
34006;;Aigne;AIGNE;34210;34;HERAULT;1
34007;;Aigues-Vives;AIGUES-VIVES;34210;34;HERAULT;1
34008;Les;Aires;AIRES;34600;34;HERAULT;1
34009;;Alignan-du-Vent;ALIGNAN-DU-VENT;34290;34;HERAULT;1
34010;;Aniane;ANIANE;34150;34;HERAULT;1
34011;;Arboras;ARBORAS;34150;34;HERAULT;1
34012;;Argelliers;ARGELLIERS;34380;34;HERAULT;1
34013;;Aspiran;ASPIRAN;34800;34;HERAULT;1
34014;;Assas;ASSAS;34820;34;HERAULT;1
34015;;Assignan;ASSIGNAN;34360;34;HERAULT;1
34016;;Aumelas;AUMELAS;34230;34;HERAULT;1
34017;;Aumes;AUMES;34530;34;HERAULT;1
34018;;Autignac;AUTIGNAC;34480;34;HERAULT;1
34019;;Avne;AVENE;34260;34;HERAULT;1
34020;;Azillanet;AZILLANET;34210;34;HERAULT;1
34021;;Babeau-Bouldoux;BABEAU-BOULDOUX;34360;34;HERAULT;1
34022;;Baillargues;BAILLARGUES;34670;34;HERAULT;1
34023;;Balaruc-les-Bains;BALARUC-LES-BAINS;34540;34;HERAULT;1
34024;;Balaruc-le-Vieux;BALARUC-LE-VIEUX;34540;34;HERAULT;1
34025;;Bassan;BASSAN;34290;34;HERAULT;1
34026;;Beaufort;BEAUFORT;34210;34;HERAULT;1
34027;;Beaulieu;BEAULIEU;34160;34;HERAULT;1
34028;;Bdarieux;BEDARIEUX;34600;34;HERAULT;1
34029;;Blarga;BELARGA;34230;34;HERAULT;1
34030;;Berlou;BERLOU;34360;34;HERAULT;1
34031;;Bessan;BESSAN;34550;34;HERAULT;1
34032;;Bziers;BEZIERS;34500;34;HERAULT;1
34033;;Boisseron;BOISSERON;34160;34;HERAULT;1
34034;;Boisset;BOISSET;34220;34;HERAULT;1
34035;La;Boissire;BOISSIERE;34150;34;HERAULT;1
34036;Le;Bosc;BOSC;34700;34;HERAULT;1
34037;;Boujan-sur-Libron;BOUJAN-SUR-LIBRON;34760;34;HERAULT;1
34038;Le;Bousquet-d'Orb;BOUSQUET-D'ORB;34260;34;HERAULT;1
34039;;Bouzigues;BOUZIGUES;34140;34;HERAULT;1
34040;;Brenas;BRENAS;34650;34;HERAULT;1
34041;;Brignac;BRIGNAC;34800;34;HERAULT;1
34042;;Brissac;BRISSAC;34190;34;HERAULT;1
34043;;Buzignargues;BUZIGNARGUES;34160;34;HERAULT;1
34044;;Cabrerolles;CABREROLLES;34480;34;HERAULT;1
34045;;Cabrires;CABRIERES;34800;34;HERAULT;1
34046;;Cambon-et-Salvergues;CAMBON-ET-SALVERGUES;34330;34;HERAULT;1
34047;;Campagnan;CAMPAGNAN;34230;34;HERAULT;1
34048;;Campagne;CAMPAGNE;34160;34;HERAULT;1
34049;;Camplong;CAMPLONG;34260;34;HERAULT;1
34050;;Candillargues;CANDILLARGUES;34130;34;HERAULT;1
34051;;Canet;CANET;34800;34;HERAULT;1
34052;;Capestang;CAPESTANG;34310;34;HERAULT;1
34053;;Carlencas-et-Levas;CARLENCAS-ET-LEVAS;34600;34;HERAULT;1
34054;;Cassagnoles;CASSAGNOLES;34210;34;HERAULT;1
34055;;Castanet-le-Haut;CASTANET-LE-HAUT;34610;34;HERAULT;1
34056;;Castelnau-de-Guers;CASTELNAU-DE-GUERS;34120;34;HERAULT;1
34057;;Castelnau-le-Lez;CASTELNAU-LE-LEZ;34170;34;HERAULT;1
34058;;Castries;CASTRIES;34160;34;HERAULT;1
34059;La;Caunette;CAUNETTE;34210;34;HERAULT;1
34060;;Causse-de-la-Selle;CAUSSE-DE-LA-SELLE;34380;34;HERAULT;1
34061;;Causses-et-Veyran;CAUSSES-ET-VEYRAN;34490;34;HERAULT;1
34062;;Caussiniojouls;CAUSSINIOJOULS;34600;34;HERAULT;1
34063;;Caux;CAUX;34720;34;HERAULT;1
34064;Le;Caylar;CAYLAR;34520;34;HERAULT;1
34065;;Cazedarnes;CAZEDARNES;34460;34;HERAULT;1
34066;;Cazevieille;CAZEVIEILLE;34270;34;HERAULT;1
34067;;Cazilhac;CAZILHAC;34190;34;HERAULT;1
34068;;Cazouls-d'Hrault;CAZOULS-D'HERAULT;34120;34;HERAULT;1
34069;;Cazouls-ls-Bziers;CAZOULS-LES-BEZIERS;34370;34;HERAULT;1
34070;;Cbazan;CEBAZAN;34360;34;HERAULT;1
34071;;Ceilhes-et-Rocozels;CEILHES-ET-ROCOZELS;34260;34;HERAULT;1
34072;;Celles;CELLES;34800;34;HERAULT;1
34073;;Cers;CERS;34420;34;HERAULT;1
34074;;Cessenon-sur-Orb;CESSENON-SUR-ORB;34460;34;HERAULT;1
34075;;Cesseras;CESSERAS;34210;34;HERAULT;1
34076;;Ceyras;CEYRAS;34800;34;HERAULT;1
34077;;Clapiers;CLAPIERS;34830;34;HERAULT;1
34078;;Claret;CLARET;34270;34;HERAULT;1
34079;;Clermont-l'Hrault;CLERMONT-L'HERAULT;34800;34;HERAULT;1
34080;;Colombires-sur-Orb;COLOMBIERES-SUR-ORB;34390;34;HERAULT;1
34081;;Colombiers;COLOMBIERS;34440;34;HERAULT;1
34082;;Combaillaux;COMBAILLAUX;34980;34;HERAULT;1
34083;;Combes;COMBES;34240;34;HERAULT;1
34084;;Corneilhan;CORNEILHAN;34490;34;HERAULT;1
34085;;Coulobres;COULOBRES;34290;34;HERAULT;1
34086;;Courniou;COURNIOU;34220;34;HERAULT;1
34087;;Cournonsec;COURNONSEC;34660;34;HERAULT;1
34088;;Cournonterral;COURNONTERRAL;34660;34;HERAULT;1
34089;;Creissan;CREISSAN;34370;34;HERAULT;1
34090;Le;Crs;CRES;34920;34;HERAULT;1
34091;Le;Cros;CROS;34520;34;HERAULT;1
34092;;Cruzy;CRUZY;34310;34;HERAULT;1
34093;;Dio-et-Valquires;DIO-ET-VALQUIERES;34650;34;HERAULT;1
34094;;Espondeilhan;ESPONDEILHAN;34290;34;HERAULT;1
34095;;Fabrgues;FABREGUES;34690;34;HERAULT;1
34096;;Faugres;FAUGERES;34600;34;HERAULT;1
34097;;Flines-Minervois;FELINES-MINERVOIS;34210;34;HERAULT;1
34098;;Ferrals-les-Montagnes;FERRALS-LES-MONTAGNES;34210;34;HERAULT;1
34099;;Ferrires-les-Verreries;FERRIERES-LES-VERRERIES;34190;34;HERAULT;1
34100;;Ferrires-Poussarou;FERRIERES-POUSSAROU;34360;34;HERAULT;1
34101;;Florensac;FLORENSAC;34510;34;HERAULT;1
34102;;Fontans;FONTANES;34270;34;HERAULT;1
34103;;Fonts;FONTES;34320;34;HERAULT;1
34104;;Fos;FOS;34320;34;HERAULT;1
34105;;Fouzilhon;FOUZILHON;34480;34;HERAULT;1
34106;;Fozires;FOZIERES;34700;34;HERAULT;1
34107;;Fraisse-sur-Agout;FRAISSE-SUR-AGOUT;34330;34;HERAULT;1
34108;;Frontignan;FRONTIGNAN;34110;34;HERAULT;1
34109;;Gabian;GABIAN;34320;34;HERAULT;1
34110;;Galargues;GALARGUES;34160;34;HERAULT;1
34111;;Ganges;GANGES;34190;34;HERAULT;1
34112;;Garrigues;GARRIGUES;34160;34;HERAULT;1
34113;;Gigean;GIGEAN;34770;34;HERAULT;1
34114;;Gignac;GIGNAC;34150;34;HERAULT;1
34115;;Gornis;GORNIES;34190;34;HERAULT;1
34116;;Grabels;GRABELS;34790;34;HERAULT;1
34117;;Graissessac;GRAISSESSAC;34260;34;HERAULT;1
34118;;Guzargues;GUZARGUES;34820;34;HERAULT;1
34119;;Hrpian;HEREPIAN;34600;34;HERAULT;1
34120;;Jacou;JACOU;34830;34;HERAULT;1
34121;;Joncels;JONCELS;34650;34;HERAULT;1
34122;;Jonquires;JONQUIERES;34725;34;HERAULT;1
34123;;Juvignac;JUVIGNAC;34990;34;HERAULT;1
34124;;Lacoste;LACOSTE;34800;34;HERAULT;1
34125;;Lagamas;LAGAMAS;34150;34;HERAULT;1
34126;;Lamalou-les-Bains;LAMALOU-LES-BAINS;34240;34;HERAULT;1
34127;;Lansargues;LANSARGUES;34130;34;HERAULT;1
34128;;Laroque;LAROQUE;34190;34;HERAULT;1
34129;;Lattes;LATTES;34970;34;HERAULT;1
34130;;Laurens;LAURENS;34480;34;HERAULT;1
34131;;Lauret;LAURET;34270;34;HERAULT;1
34132;;Lauroux;LAUROUX;34700;34;HERAULT;1
34133;;Lavalette;LAVALETTE;34700;34;HERAULT;1
34134;;Lavrune;LAVERUNE;34880;34;HERAULT;1
34135;;Lespignan;LESPIGNAN;34710;34;HERAULT;1
34136;;Lzignan-la-Cbe;LEZIGNAN-LA-CEBE;34120;34;HERAULT;1
34137;;Liausson;LIAUSSON;34800;34;HERAULT;1
34138;;Lieuran-Cabrires;LIEURAN-CABRIERES;34800;34;HERAULT;1
34139;;Lieuran-ls-Bziers;LIEURAN-LES-BEZIERS;34290;34;HERAULT;1
34140;;Lignan-sur-Orb;LIGNAN-SUR-ORB;34490;34;HERAULT;1
34141;La;Livinire;LIVINIERE;34210;34;HERAULT;1
34142;;Lodve;LODEVE;34700;34;HERAULT;1
34143;;Loupian;LOUPIAN;34140;34;HERAULT;1
34144;;Lunas;LUNAS;34650;34;HERAULT;1
34145;;Lunel;LUNEL;34400;34;HERAULT;1
34146;;Lunel-Viel;LUNEL-VIEL;34400;34;HERAULT;1
34147;;Magalas;MAGALAS;34480;34;HERAULT;1
34148;;Maraussan;MARAUSSAN;34370;34;HERAULT;1
34149;;Margon;MARGON;34320;34;HERAULT;1
34150;;Marseillan;MARSEILLAN;34340;34;HERAULT;1
34151;;Marsillargues;MARSILLARGUES;34590;34;HERAULT;1
34152;;Mas-de-Londres;MAS-DE-LONDRES;34380;34;HERAULT;1
34153;Les;Matelles;MATELLES;34270;34;HERAULT;1
34154;;Mauguio;MAUGUIO;34130;34;HERAULT;1
34155;;Maureilhan;MAUREILHAN;34370;34;HERAULT;1
34156;;Mrifons;MERIFONS;34800;34;HERAULT;1
34157;;Mze;MEZE;34140;34;HERAULT;1
34158;;Minerve;MINERVE;34210;34;HERAULT;1
34159;;Mireval;MIREVAL;34110;34;HERAULT;1
34160;;Mons;MONS;34390;34;HERAULT;1
34161;;Montady;MONTADY;34310;34;HERAULT;1
34162;;Montagnac;MONTAGNAC;34530;34;HERAULT;1
34163;;Montarnaud;MONTARNAUD;34570;34;HERAULT;1
34164;;Montaud;MONTAUD;34160;34;HERAULT;1
34165;;Montbazin;MONTBAZIN;34560;34;HERAULT;1
34166;;Montblanc;MONTBLANC;34290;34;HERAULT;1
34167;;Montels;MONTELS;34310;34;HERAULT;1
34168;;Montesquieu;MONTESQUIEU;34320;34;HERAULT;1
34169;;Montferrier-sur-Lez;MONTFERRIER-SUR-LEZ;34980;34;HERAULT;1
34170;;Montouliers;MONTOULIERS;34310;34;HERAULT;1
34171;;Montoulieu;MONTOULIEU;34190;34;HERAULT;1
34172;;Montpellier;MONTPELLIER;34000;34;HERAULT;1
34173;;Montpeyroux;MONTPEYROUX;34150;34;HERAULT;1
34174;;Mouls-et-Baucels;MOULES-ET-BAUCELS;34190;34;HERAULT;1
34175;;Mourze;MOUREZE;34800;34;HERAULT;1
34176;;Mudaison;MUDAISON;34130;34;HERAULT;1
34177;;Murles;MURLES;34980;34;HERAULT;1
34178;;Murviel-ls-Bziers;MURVIEL-LES-BEZIERS;34490;34;HERAULT;1
34179;;Murviel-ls-Montpellier;MURVIEL-LES-MONTPELLIER;34570;34;HERAULT;1
34180;;Nbian;NEBIAN;34800;34;HERAULT;1
34181;;Neffis;NEFFIES;34320;34;HERAULT;1
34182;;Nzignan-l'vque;NEZIGNAN-L'EVEQUE;34120;34;HERAULT;1
34183;;Nissan-lez-Enserune;NISSAN-LEZ-ENSERUNE;34440;34;HERAULT;1
34184;;Nizas;NIZAS;34320;34;HERAULT;1
34185;;Notre-Dame-de-Londres;NOTRE-DAME-DE-LONDRES;34380;34;HERAULT;1
34186;;Octon;OCTON;34800;34;HERAULT;1
34187;;Olargues;OLARGUES;34390;34;HERAULT;1
34188;;Olmet-et-Villecun;OLMET-ET-VILLECUN;34700;34;HERAULT;1
34189;;Olonzac;OLONZAC;34210;34;HERAULT;1
34190;;Oupia;OUPIA;34210;34;HERAULT;1
34191;;Pailhs;PAILHES;34490;34;HERAULT;1
34192;;Palavas-les-Flots;PALAVAS-LES-FLOTS;34250;34;HERAULT;1
34193;;Pardailhan;PARDAILHAN;34360;34;HERAULT;1
34194;;Paulhan;PAULHAN;34230;34;HERAULT;1
34195;;Pgairolles-de-Buges;PEGAIROLLES-DE-BUEGES;34380;34;HERAULT;1
34196;;Pgairolles-de-l'Escalette;PEGAIROLLES-DE-L'ESCALETTE;34700;34;HERAULT;1
34197;;Pret;PERET;34800;34;HERAULT;1
34198;;Prols;PEROLS;34470;34;HERAULT;1
34199;;Pzenas;PEZENAS;34120;34;HERAULT;1
34200;;Pznes-les-Mines;PEZENES-LES-MINES;34600;34;HERAULT;1
34201;;Pierrerue;PIERRERUE;34360;34;HERAULT;1
34202;;Pignan;PIGNAN;34570;34;HERAULT;1
34203;;Pinet;PINET;34850;34;HERAULT;1
34204;;Plaissan;PLAISSAN;34230;34;HERAULT;1
34205;Les;Plans;PLANS;34700;34;HERAULT;1
34206;;Poilhes;POILHES;34310;34;HERAULT;1
34207;;Pomrols;POMEROLS;34810;34;HERAULT;1
34208;;Popian;POPIAN;34230;34;HERAULT;1
34209;;Portiragnes;PORTIRAGNES;34420;34;HERAULT;1
34210;Le;Pouget;POUGET;34230;34;HERAULT;1
34211;Le;Poujol-sur-Orb;POUJOL-SUR-ORB;34600;34;HERAULT;1
34212;;Poujols;POUJOLS;34700;34;HERAULT;1
34213;;Poussan;POUSSAN;34560;34;HERAULT;1
34214;;Pouzolles;POUZOLLES;34480;34;HERAULT;1
34215;;Pouzols;POUZOLS;34230;34;HERAULT;1
34216;Le;Pradal;PRADAL;34600;34;HERAULT;1
34217;;Prades-le-Lez;PRADES-LE-LEZ;34730;34;HERAULT;1
34218;;Prades-sur-Vernazobre;PRADES-SUR-VERNAZOBRE;34360;34;HERAULT;1
34219;;Prmian;PREMIAN;34390;34;HERAULT;1
34220;Le;Puech;PUECH;34700;34;HERAULT;1
34221;;Puchabon;PUECHABON;34150;34;HERAULT;1
34222;;Puilacher;PUILACHER;34230;34;HERAULT;1
34223;;Puimisson;PUIMISSON;34480;34;HERAULT;1
34224;;Puissalicon;PUISSALICON;34480;34;HERAULT;1
34225;;Puisserguier;PUISSERGUIER;34620;34;HERAULT;1
34226;;Quarante;QUARANTE;34310;34;HERAULT;1
34227;;Restinclires;RESTINCLIERES;34160;34;HERAULT;1
34228;;Rieussec;RIEUSSEC;34220;34;HERAULT;1
34229;;Riols;RIOLS;34220;34;HERAULT;1
34230;Les;Rives;RIVES;34520;34;HERAULT;1
34231;;Romiguires;ROMIGUIERES;34650;34;HERAULT;1
34232;;Roquebrun;ROQUEBRUN;34460;34;HERAULT;1
34233;;Roqueredonde;ROQUEREDONDE;34650;34;HERAULT;1
34234;;Roquessels;ROQUESSELS;34320;34;HERAULT;1
34235;;Rosis;ROSIS;34610;34;HERAULT;1
34236;;Rouet;ROUET;34380;34;HERAULT;1
34237;;Roujan;ROUJAN;34320;34;HERAULT;1
34238;;Saint-Andr-de-Buges;SAINT-ANDRE-DE-BUEGES;34190;34;HERAULT;1
34239;;Saint-Andr-de-Sangonis;SAINT-ANDRE-DE-SANGONIS;34725;34;HERAULT;1
34240;;Saint-Auns;SAINT-AUNES;34130;34;HERAULT;1
34241;;Saint-Bauzille-de-la-Sylve;SAINT-BAUZILLE-DE-LA-SYLVE;34230;34;HERAULT;1
34242;;Saint-Bauzille-de-Montmel;SAINT-BAUZILLE-DE-MONTMEL;34160;34;HERAULT;1
34243;;Saint-Bauzille-de-Putois;SAINT-BAUZILLE-DE-PUTOIS;34190;34;HERAULT;1
34244;;Saint-Brs;SAINT-BRES;34670;34;HERAULT;1
34245;;Saint-Chinian;SAINT-CHINIAN;34360;34;HERAULT;1
34246;;Saint-Christol;SAINT-CHRISTOL;34400;34;HERAULT;1
34247;;Saint-Clment-de-Rivire;SAINT-CLEMENT-DE-RIVIERE;34980;34;HERAULT;1
34248;;Sainte-Croix-de-Quintillargues;SAINTE-CROIX-DE-QUINTILLARGUES;34270;34;HERAULT;1
34249;;Saint-Drzry;SAINT-DREZERY;34160;34;HERAULT;1
34250;;Saint-tienne-d'Albagnan;SAINT-ETIENNE-D'ALBAGNAN;34390;34;HERAULT;1
34251;;Saint-tienne-de-Gourgas;SAINT-ETIENNE-DE-GOURGAS;34700;34;HERAULT;1
34252;;Saint-tienne-Estrchoux;SAINT-ETIENNE-ESTRECHOUX;34260;34;HERAULT;1
34253;;Saint-Flix-de-l'Hras;SAINT-FELIX-DE-L'HERAS;34520;34;HERAULT;1
34254;;Saint-Flix-de-Lodez;SAINT-FELIX-DE-LODEZ;34725;34;HERAULT;1
34255;;Saint-Gly-du-Fesc;SAINT-GELY-DU-FESC;34980;34;HERAULT;1
34256;;Saint-Genis-des-Mourgues;SAINT-GENIES-DES-MOURGUES;34160;34;HERAULT;1
34257;;Saint-Genis-de-Varensal;SAINT-GENIES-DE-VARENSAL;34610;34;HERAULT;1
34258;;Saint-Genis-de-Fontedit;SAINT-GENIES-DE-FONTEDIT;34480;34;HERAULT;1
34259;;Saint-Georges-d'Orques;SAINT-GEORGES-D'ORQUES;34680;34;HERAULT;1
34260;;Saint-Gervais-sur-Mare;SAINT-GERVAIS-SUR-MARE;34610;34;HERAULT;1
34261;;Saint-Guilhem-le-Dsert;SAINT-GUILHEM-LE-DESERT;34150;34;HERAULT;1
34262;;Saint-Guiraud;SAINT-GUIRAUD;34725;34;HERAULT;1
34263;;Saint-Hilaire-de-Beauvoir;SAINT-HILAIRE-DE-BEAUVOIR;34160;34;HERAULT;1
34264;;Saint-Jean-de-Buges;SAINT-JEAN-DE-BUEGES;34380;34;HERAULT;1
34265;;Saint-Jean-de-Cornies;SAINT-JEAN-DE-CORNIES;34160;34;HERAULT;1
34266;;Saint-Jean-de-Cuculles;SAINT-JEAN-DE-CUCULLES;34270;34;HERAULT;1
34267;;Saint-Jean-de-Fos;SAINT-JEAN-DE-FOS;34150;34;HERAULT;1
34268;;Saint-Jean-de-la-Blaquire;SAINT-JEAN-DE-LA-BLAQUIERE;34700;34;HERAULT;1
34269;;Saint-Jean-de-Minervois;SAINT-JEAN-DE-MINERVOIS;34360;34;HERAULT;1
34270;;Saint-Jean-de-Vdas;SAINT-JEAN-DE-VEDAS;34430;34;HERAULT;1
34271;;Saint-Julien;SAINT-JULIEN;34390;34;HERAULT;1
34272;;Saint-Just;SAINT-JUST;34400;34;HERAULT;1
34273;;Saint-Martin-de-l'Aron;SAINT-MARTIN-DE-L'ARCON;34390;34;HERAULT;1
34274;;Saint-Martin-de-Londres;SAINT-MARTIN-DE-LONDRES;34380;34;HERAULT;1
34276;;Saint-Mathieu-de-Trviers;SAINT-MATHIEU-DE-TREVIERS;34270;34;HERAULT;1
34277;;Saint-Maurice-Navacelles;SAINT-MAURICE-NAVACELLES;34520;34;HERAULT;1
34278;;Saint-Michel;SAINT-MICHEL;34520;34;HERAULT;1
34279;;Saint-Nazaire-de-Ladarez;SAINT-NAZAIRE-DE-LADAREZ;34490;34;HERAULT;1
34280;;Saint-Nazaire-de-Pzan;SAINT-NAZAIRE-DE-PEZAN;34400;34;HERAULT;1
34281;;Saint-Pargoire;SAINT-PARGOIRE;34230;34;HERAULT;1
34282;;Saint-Paul-et-Valmalle;SAINT-PAUL-ET-VALMALLE;34570;34;HERAULT;1
34283;;Saint-Pierre-de-la-Fage;SAINT-PIERRE-DE-LA-FAGE;34520;34;HERAULT;1
34284;;Saint-Pons-de-Thomires;SAINT-PONS-DE-THOMIERES;34220;34;HERAULT;1
34285;;Saint-Pons-de-Mauchiens;SAINT-PONS-DE-MAUCHIENS;34230;34;HERAULT;1
34286;;Saint-Privat;SAINT-PRIVAT;34700;34;HERAULT;1
34287;;Saint-Saturnin-de-Lucian;SAINT-SATURNIN-DE-LUCIAN;34725;34;HERAULT;1
34288;;Saint-Sris;SAINT-SERIES;34400;34;HERAULT;1
34289;;Saint-Thibry;SAINT-THIBERY;34630;34;HERAULT;1
34290;;Saint-Vincent-de-Barbeyrargues;SAINT-VINCENT-DE-BARBEYRARGUES;34730;34;HERAULT;1
34291;;Saint-Vincent-d'Olargues;SAINT-VINCENT-D'OLARGUES;34390;34;HERAULT;1
34292;;Salasc;SALASC;34800;34;HERAULT;1
34293;La;Salvetat-sur-Agout;SALVETAT-SUR-AGOUT;34330;34;HERAULT;1
34294;;Saturargues;SATURARGUES;34400;34;HERAULT;1
34295;;Saussan;SAUSSAN;34570;34;HERAULT;1
34296;;Saussines;SAUSSINES;34160;34;HERAULT;1
34297;;Sauteyrargues;SAUTEYRARGUES;34270;34;HERAULT;1
34298;;Sauvian;SAUVIAN;34410;34;HERAULT;1
34299;;Srignan;SERIGNAN;34410;34;HERAULT;1
34300;;Servian;SERVIAN;34290;34;HERAULT;1
34301;;Ste;SETE;34200;34;HERAULT;1
34302;;Siran;SIRAN;34210;34;HERAULT;1
34303;;Sorbs;SORBS;34520;34;HERAULT;1
34304;;Soubs;SOUBES;34700;34;HERAULT;1
34305;Le;Souli;SOULIE;34330;34;HERAULT;1
34306;;Soumont;SOUMONT;34700;34;HERAULT;1
34307;;Sussargues;SUSSARGUES;34160;34;HERAULT;1
34308;;Taussac-la-Billire;TAUSSAC-LA-BILLIERE;34600;34;HERAULT;1
34309;;Teyran;TEYRAN;34820;34;HERAULT;1
34310;;Thzan-ls-Bziers;THEZAN-LES-BEZIERS;34490;34;HERAULT;1
34311;;Tourbes;TOURBES;34120;34;HERAULT;1
34312;La;Tour-sur-Orb;TOUR-SUR-ORB;34260;34;HERAULT;1
34313;;Tressan;TRESSAN;34230;34;HERAULT;1
34314;Le;Triadou;TRIADOU;34270;34;HERAULT;1
34315;;Usclas-d'Hrault;USCLAS-D'HERAULT;34230;34;HERAULT;1
34316;;Usclas-du-Bosc;USCLAS-DU-BOSC;34700;34;HERAULT;1
34317;La;Vacquerie-et-Saint-Martin-de-Castries;VACQUERIE-ET-SAINT-MARTIN-DE-CASTRIES;34520;34;HERAULT;1
34318;;Vacquires;VACQUIERES;34270;34;HERAULT;1
34319;;Vailhan;VAILHAN;34320;34;HERAULT;1
34320;;Vailhauqus;VAILHAUQUES;34570;34;HERAULT;1
34321;;Valergues;VALERGUES;34130;34;HERAULT;1
34322;;Valflauns;VALFLAUNES;34270;34;HERAULT;1
34323;;Valmascle;VALMASCLE;34800;34;HERAULT;1
34324;;Valras-Plage;VALRAS-PLAGE;34350;34;HERAULT;1
34325;;Valros;VALROS;34290;34;HERAULT;1
34326;;Vlieux;VELIEUX;34220;34;HERAULT;1
34327;;Vendargues;VENDARGUES;34740;34;HERAULT;1
34328;;Vendmian;VENDEMIAN;34230;34;HERAULT;1
34329;;Vendres;VENDRES;34350;34;HERAULT;1
34330;;Vrargues;VERARGUES;34400;34;HERAULT;1
34331;;Verreries-de-Moussans;VERRERIES-DE-MOUSSANS;34220;34;HERAULT;1
34332;;Vias;VIAS;34450;34;HERAULT;1
34333;;Vic-la-Gardiole;VIC-LA-GARDIOLE;34110;34;HERAULT;1
34334;;Vieussan;VIEUSSAN;34390;34;HERAULT;1
34335;;Villemagne-l'Argentire;VILLEMAGNE-L'ARGENTIERE;34600;34;HERAULT;1
34336;;Villeneuve-ls-Bziers;VILLENEUVE-LES-BEZIERS;34420;34;HERAULT;1
34337;;Villeneuve-ls-Maguelone;VILLENEUVE-LES-MAGUELONE;34750;34;HERAULT;1
34338;;Villeneuvette;VILLENEUVETTE;34800;34;HERAULT;1
34339;;Villespassans;VILLESPASSANS;34360;34;HERAULT;1
34340;;Villetelle;VILLETELLE;34400;34;HERAULT;1
34341;;Villeveyrac;VILLEVEYRAC;34560;34;HERAULT;1
34342;;Viols-en-Laval;VIOLS-EN-LAVAL;34380;34;HERAULT;1
34343;;Viols-le-Fort;VIOLS-LE-FORT;34380;34;HERAULT;1
34344;La;Grande-Motte;GRANDE-MOTTE;34280;34;HERAULT;1
35001;;Acign;ACIGNE;35690;35;ILLE ET VILAINE;1
35002;;Amanlis;AMANLIS;35150;35;ILLE ET VILAINE;1
35003;;Andouill-Neuville;ANDOUILLE-NEUVILLE;35250;35;ILLE ET VILAINE;1
35004;;Antrain;ANTRAIN;35560;35;ILLE ET VILAINE;1
35005;;Arbrissel;ARBRISSEL;35130;35;ILLE ET VILAINE;1
35006;;Argentr-du-Plessis;ARGENTRE-DU-PLESSIS;35370;35;ILLE ET VILAINE;1
35007;;Aubign;AUBIGNE;35250;35;ILLE ET VILAINE;1
35008;;Availles-sur-Seiche;AVAILLES-SUR-SEICHE;35130;35;ILLE ET VILAINE;1
35009;;Baguer-Morvan;BAGUER-MORVAN;35120;35;ILLE ET VILAINE;1
35010;;Baguer-Pican;BAGUER-PICAN;35120;35;ILLE ET VILAINE;1
35011;;Baill;BAILLE;35460;35;ILLE ET VILAINE;1
35012;;Bain-de-Bretagne;BAIN-DE-BRETAGNE;35470;35;ILLE ET VILAINE;1
35013;;Bains-sur-Oust;BAINS-SUR-OUST;35600;35;ILLE ET VILAINE;1
35014;;Bais;BAIS;35680;35;ILLE ET VILAINE;1
35015;;Balaz;BALAZE;35500;35;ILLE ET VILAINE;1
35016;;Baulon;BAULON;35580;35;ILLE ET VILAINE;1
35017;La;Baussaine;BAUSSAINE;35190;35;ILLE ET VILAINE;1
35018;La;Bazouge-du-Dsert;BAZOUGE-DU-DESERT;35420;35;ILLE ET VILAINE;1
35019;;Bazouges-la-Prouse;BAZOUGES-LA-PEROUSE;35560;35;ILLE ET VILAINE;1
35021;;Beauc;BEAUCE;35133;35;ILLE ET VILAINE;1
35022;;Bcherel;BECHEREL;35190;35;ILLE ET VILAINE;1
35023;;Bde;BEDEE;35137;35;ILLE ET VILAINE;1
35024;;Betton;BETTON;35830;35;ILLE ET VILAINE;1
35025;;Bill;BILLE;35133;35;ILLE ET VILAINE;1
35026;;Blruais;BLERUAIS;35750;35;ILLE ET VILAINE;1
35027;;Boisgervilly;BOISGERVILLY;35360;35;ILLE ET VILAINE;1
35028;;Boistrudan;BOISTRUDAN;35150;35;ILLE ET VILAINE;1
35029;;Bonnemain;BONNEMAIN;35270;35;ILLE ET VILAINE;1
35030;La;Bosse-de-Bretagne;BOSSE-DE-BRETAGNE;35320;35;ILLE ET VILAINE;1
35031;La;Bouxire;BOUEXIERE;35340;35;ILLE ET VILAINE;1
35032;;Bourgbarr;BOURGBARRE;35230;35;ILLE ET VILAINE;1
35033;;Bourg-des-Comptes;BOURG-DES-COMPTES;35890;35;ILLE ET VILAINE;1
35034;La;Boussac;BOUSSAC;35120;35;ILLE ET VILAINE;1
35035;;Bovel;BOVEL;35330;35;ILLE ET VILAINE;1
35037;;Bral-sous-Montfort;BREAL-SOUS-MONTFORT;35310;35;ILLE ET VILAINE;1
35038;;Bral-sous-Vitr;BREAL-SOUS-VITRE;35370;35;ILLE ET VILAINE;1
35039;;Brc;BRECE;35530;35;ILLE ET VILAINE;1
35040;;Breteil;BRETEIL;35160;35;ILLE ET VILAINE;1
35041;;Brie;BRIE;35150;35;ILLE ET VILAINE;1
35042;;Brielles;BRIELLES;35370;35;ILLE ET VILAINE;1
35044;;Broualan;BROUALAN;35120;35;ILLE ET VILAINE;1
35045;;Bruc-sur-Aff;BRUC-SUR-AFF;35550;35;ILLE ET VILAINE;1
35046;Les;Brulais;BRULAIS;35330;35;ILLE ET VILAINE;1
35047;;Bruz;BRUZ;35170;35;ILLE ET VILAINE;1
35048;;Campel;CAMPEL;35330;35;ILLE ET VILAINE;1
35049;;Cancale;CANCALE;35260;35;ILLE ET VILAINE;1
35050;;Cardroc;CARDROC;35190;35;ILLE ET VILAINE;1
35051;;Cesson-Svign;CESSON-SEVIGNE;35510;35;ILLE ET VILAINE;1
35052;;Champeaux;CHAMPEAUX;35500;35;ILLE ET VILAINE;1
35053;;Chanc;CHANCE;35680;35;ILLE ET VILAINE;1
35054;;Chanteloup;CHANTELOUP;35150;35;ILLE ET VILAINE;1
35055;;Chantepie;CHANTEPIE;35135;35;ILLE ET VILAINE;1
35056;La;Chapelle-aux-Filtzmens;CHAPELLE-AUX-FILTZMEENS;35190;35;ILLE ET VILAINE;1
35057;La;Chapelle-Bouxic;CHAPELLE-BOUEXIC;35330;35;ILLE ET VILAINE;1
35058;La;Chapelle-Chausse;CHAPELLE-CHAUSSEE;35630;35;ILLE ET VILAINE;1
35059;La;Chapelle-des-Fougeretz;CHAPELLE-DES-FOUGERETZ;35520;35;ILLE ET VILAINE;1
35060;La;Chapelle-du-Lou;CHAPELLE-DU-LOU;35360;35;ILLE ET VILAINE;1
35061;La;Chapelle-Erbre;CHAPELLE-ERBREE;35500;35;ILLE ET VILAINE;1
35062;La;Chapelle-Janson;CHAPELLE-JANSON;35133;35;ILLE ET VILAINE;1
35063;La;Chapelle-Saint-Aubert;CHAPELLE-SAINT-AUBERT;35140;35;ILLE ET VILAINE;1
35064;La;Chapelle-de-Brain;CHAPELLE-DE-BRAIN;35660;35;ILLE ET VILAINE;1
35065;La;Chapelle-Thouarault;CHAPELLE-THOUARAULT;35590;35;ILLE ET VILAINE;1
35066;;Chartres-de-Bretagne;CHARTRES-DE-BRETAGNE;35131;35;ILLE ET VILAINE;1
35067;;Chasn-sur-Illet;CHASNE-SUR-ILLET;35250;35;ILLE ET VILAINE;1
35068;;Chteaubourg;CHATEAUBOURG;35220;35;ILLE ET VILAINE;1
35069;;Chteaugiron;CHATEAUGIRON;35410;35;ILLE ET VILAINE;1
35070;;Chteauneuf-d'Ille-et-Vilaine;CHATEAUNEUF-D'ILLE-ET-VILAINE;35430;35;ILLE ET VILAINE;1
35071;Le;Chtellier;CHATELLIER;35133;35;ILLE ET VILAINE;1
35072;;Chtillon-en-Vendelais;CHATILLON-EN-VENDELAIS;35210;35;ILLE ET VILAINE;1
35075;;Chauvign;CHAUVIGNE;35490;35;ILLE ET VILAINE;1
35076;;Chavagne;CHAVAGNE;35310;35;ILLE ET VILAINE;1
35077;;Chelun;CHELUN;35640;35;ILLE ET VILAINE;1
35078;;Cherrueix;CHERRUEIX;35120;35;ILLE ET VILAINE;1
35079;;Chevaign;CHEVAIGNE;35250;35;ILLE ET VILAINE;1
35080;;Cintr;CINTRE;35310;35;ILLE ET VILAINE;1
35081;;Clayes;CLAYES;35590;35;ILLE ET VILAINE;1
35082;;Cosmes;COESMES;35134;35;ILLE ET VILAINE;1
35083;;Cogls;COGLES;35460;35;ILLE ET VILAINE;1
35084;;Comblessac;COMBLESSAC;35330;35;ILLE ET VILAINE;1
35085;;Combourg;COMBOURG;35270;35;ILLE ET VILAINE;1
35086;;Combourtill;COMBOURTILLE;35210;35;ILLE ET VILAINE;1
35087;;Cornill;CORNILLE;35500;35;ILLE ET VILAINE;1
35088;;Corps-Nuds;CORPS-NUDS;35150;35;ILLE ET VILAINE;1
35089;La;Couyre;COUYERE;35320;35;ILLE ET VILAINE;1
35090;;Crevin;CREVIN;35320;35;ILLE ET VILAINE;1
35091;Le;Crouais;CROUAIS;35290;35;ILLE ET VILAINE;1
35092;;Cuguen;CUGUEN;35270;35;ILLE ET VILAINE;1
35093;;Dinard;DINARD;35800;35;ILLE ET VILAINE;1
35094;;Ding;DINGE;35440;35;ILLE ET VILAINE;1
35095;;Dol-de-Bretagne;DOL-DE-BRETAGNE;35120;35;ILLE ET VILAINE;1
35096;;Domagn;DOMAGNE;35113;35;ILLE ET VILAINE;1
35097;;Domalain;DOMALAIN;35680;35;ILLE ET VILAINE;1
35098;La;Dominelais;DOMINELAIS;35390;35;ILLE ET VILAINE;1
35099;;Domloup;DOMLOUP;35410;35;ILLE ET VILAINE;1
35100;;Dompierre-du-Chemin;DOMPIERRE-DU-CHEMIN;35210;35;ILLE ET VILAINE;1
35101;;Dourdain;DOURDAIN;35450;35;ILLE ET VILAINE;1
35102;;Drouges;DROUGES;35130;35;ILLE ET VILAINE;1
35103;;Eanc;EANCE;35640;35;ILLE ET VILAINE;1
35104;;Epiniac;EPINIAC;35120;35;ILLE ET VILAINE;1
35105;;Erbre;ERBREE;35500;35;ILLE ET VILAINE;1
35106;;Erc-en-Lame;ERCE-EN-LAMEE;35620;35;ILLE ET VILAINE;1
35107;;Erc-prs-Liffr;ERCE-PRES-LIFFRE;35340;35;ILLE ET VILAINE;1
35108;;Ess;ESSE;35150;35;ILLE ET VILAINE;1
35109;;trelles;ETRELLES;35370;35;ILLE ET VILAINE;1
35110;;Feins;FEINS;35440;35;ILLE ET VILAINE;1
35111;Le;Ferr;FERRE;35420;35;ILLE ET VILAINE;1
35112;;Fleurign;FLEURIGNE;35133;35;ILLE ET VILAINE;1
35113;La;Fontenelle;FONTENELLE;35560;35;ILLE ET VILAINE;1
35114;;Forges-la-Fort;FORGES-LA-FORET;35640;35;ILLE ET VILAINE;1
35115;;Fougres;FOUGERES;35300;35;ILLE ET VILAINE;1
35116;La;Fresnais;FRESNAIS;35111;35;ILLE ET VILAINE;1
35117;;Gal;GAEL;35290;35;ILLE ET VILAINE;1
35118;;Gahard;GAHARD;35490;35;ILLE ET VILAINE;1
35119;;Gennes-sur-Seiche;GENNES-SUR-SEICHE;35370;35;ILLE ET VILAINE;1
35120;;Gvez;GEVEZE;35850;35;ILLE ET VILAINE;1
35121;;Gosn;GOSNE;35140;35;ILLE ET VILAINE;1
35122;La;Gouesnire;GOUESNIERE;35350;35;ILLE ET VILAINE;1
35123;;Goven;GOVEN;35580;35;ILLE ET VILAINE;1
35124;;Grand-Fougeray;GRAND-FOUGERAY;35390;35;ILLE ET VILAINE;1
35125;La;Guerche-de-Bretagne;GUERCHE-DE-BRETAGNE;35130;35;ILLE ET VILAINE;1
35126;;Guichen;GUICHEN;35580;35;ILLE ET VILAINE;1
35127;;Guignen;GUIGNEN;35580;35;ILLE ET VILAINE;1
35128;;Guipel;GUIPEL;35440;35;ILLE ET VILAINE;1
35129;;Guipry;GUIPRY;35480;35;ILLE ET VILAINE;1
35130;;Hd-Bazouges;HEDE-BAZOUGES;35630;35;ILLE ET VILAINE;1
35131;L';Hermitage;HERMITAGE;35590;35;ILLE ET VILAINE;1
35132;;Hirel;HIREL;35120;35;ILLE ET VILAINE;1
35133;;Iffendic;IFFENDIC;35750;35;ILLE ET VILAINE;1
35134;Les;Iffs;IFFS;35630;35;ILLE ET VILAINE;1
35135;;Irodour;IRODOUER;35850;35;ILLE ET VILAINE;1
35136;;Janz;JANZE;35150;35;ILLE ET VILAINE;1
35137;;Javen;JAVENE;35133;35;ILLE ET VILAINE;1
35138;;Laignelet;LAIGNELET;35133;35;ILLE ET VILAINE;1
35139;;Laill;LAILLE;35890;35;ILLE ET VILAINE;1
35140;;Lalleu;LALLEU;35320;35;ILLE ET VILAINE;1
35141;;Landavran;LANDAVRAN;35450;35;ILLE ET VILAINE;1
35142;;Landan;LANDEAN;35133;35;ILLE ET VILAINE;1
35143;;Landujan;LANDUJAN;35360;35;ILLE ET VILAINE;1
35144;;Langan;LANGAN;35850;35;ILLE ET VILAINE;1
35145;;Langon;LANGON;35660;35;ILLE ET VILAINE;1
35146;;Langouet;LANGOUET;35630;35;ILLE ET VILAINE;1
35147;;Lanhlin;LANHELIN;35720;35;ILLE ET VILAINE;1
35148;;Lanrigan;LANRIGAN;35270;35;ILLE ET VILAINE;1
35149;;Lassy;LASSY;35580;35;ILLE ET VILAINE;1
35150;;Lcousse;LECOUSSE;35133;35;ILLE ET VILAINE;1
35151;;Lieuron;LIEURON;35550;35;ILLE ET VILAINE;1
35152;;Liffr;LIFFRE;35340;35;ILLE ET VILAINE;1
35153;;Lillemer;LILLEMER;35111;35;ILLE ET VILAINE;1
35154;;Livr-sur-Changeon;LIVRE-SUR-CHANGEON;35450;35;ILLE ET VILAINE;1
35155;;Lohac;LOHEAC;35550;35;ILLE ET VILAINE;1
35156;;Longaulnay;LONGAULNAY;35190;35;ILLE ET VILAINE;1
35157;Le;Loroux;LOROUX;35133;35;ILLE ET VILAINE;1
35158;Le;Lou-du-Lac;LOU-DU-LAC;35360;35;ILLE ET VILAINE;1
35159;;Lourmais;LOURMAIS;35270;35;ILLE ET VILAINE;1
35160;;Loutehel;LOUTEHEL;35330;35;ILLE ET VILAINE;1
35161;;Louvign-de-Bais;LOUVIGNE-DE-BAIS;35680;35;ILLE ET VILAINE;1
35162;;Louvign-du-Dsert;LOUVIGNE-DU-DESERT;35420;35;ILLE ET VILAINE;1
35163;;Luitr;LUITRE;35133;35;ILLE ET VILAINE;1
35164;;Marcill-Raoul;MARCILLE-RAOUL;35560;35;ILLE ET VILAINE;1
35165;;Marcill-Robert;MARCILLE-ROBERT;35240;35;ILLE ET VILAINE;1
35166;;Marpir;MARPIRE;35220;35;ILLE ET VILAINE;1
35167;;Martign-Ferchaud;MARTIGNE-FERCHAUD;35640;35;ILLE ET VILAINE;1
35168;;Maure-de-Bretagne;MAURE-DE-BRETAGNE;35330;35;ILLE ET VILAINE;1
35169;;Maxent;MAXENT;35380;35;ILLE ET VILAINE;1
35170;;Mec;MECE;35450;35;ILLE ET VILAINE;1
35171;;Mdrac;MEDREAC;35360;35;ILLE ET VILAINE;1
35172;;Meillac;MEILLAC;35270;35;ILLE ET VILAINE;1
35173;;Melesse;MELESSE;35520;35;ILLE ET VILAINE;1
35174;;Mell;MELLE;35420;35;ILLE ET VILAINE;1
35175;;Mernel;MERNEL;35330;35;ILLE ET VILAINE;1
35176;;Messac;MESSAC;35480;35;ILLE ET VILAINE;1
35177;La;Mzire;MEZIERE;35520;35;ILLE ET VILAINE;1
35178;;Mzires-sur-Couesnon;MEZIERES-SUR-COUESNON;35140;35;ILLE ET VILAINE;1
35179;;Miniac-Morvan;MINIAC-MORVAN;35540;35;ILLE ET VILAINE;1
35180;;Miniac-sous-Bcherel;MINIAC-SOUS-BECHEREL;35190;35;ILLE ET VILAINE;1
35181;Le;Minihic-sur-Rance;MINIHIC-SUR-RANCE;35870;35;ILLE ET VILAINE;1
35183;;Mondevert;MONDEVERT;35370;35;ILLE ET VILAINE;1
35184;;Montauban-de-Bretagne;MONTAUBAN-DE-BRETAGNE;35360;35;ILLE ET VILAINE;1
35185;;Montautour;MONTAUTOUR;35210;35;ILLE ET VILAINE;1
35186;;Mont-Dol;MONT-DOL;35120;35;ILLE ET VILAINE;1
35187;;Monterfil;MONTERFIL;35160;35;ILLE ET VILAINE;1
35188;;Montfort-sur-Meu;MONTFORT-SUR-MEU;35160;35;ILLE ET VILAINE;1
35189;;Montgermont;MONTGERMONT;35760;35;ILLE ET VILAINE;1
35190;;Monthault;MONTHAULT;35420;35;ILLE ET VILAINE;1
35191;;Montours;MONTOURS;35460;35;ILLE ET VILAINE;1
35192;;Montreuil-des-Landes;MONTREUIL-DES-LANDES;35210;35;ILLE ET VILAINE;1
35193;;Montreuil-le-Gast;MONTREUIL-LE-GAST;35520;35;ILLE ET VILAINE;1
35194;;Montreuil-sous-Prouse;MONTREUIL-SOUS-PEROUSE;35500;35;ILLE ET VILAINE;1
35195;;Montreuil-sur-Ille;MONTREUIL-SUR-ILLE;35440;35;ILLE ET VILAINE;1
35196;;Mordelles;MORDELLES;35310;35;ILLE ET VILAINE;1
35197;;Mouaz;MOUAZE;35250;35;ILLE ET VILAINE;1
35198;;Moulins;MOULINS;35680;35;ILLE ET VILAINE;1
35199;;Mouss;MOUSSE;35130;35;ILLE ET VILAINE;1
35200;;Moutiers;MOUTIERS;35130;35;ILLE ET VILAINE;1
35201;;Muel;MUEL;35290;35;ILLE ET VILAINE;1
35202;La;No-Blanche;NOE-BLANCHE;35470;35;ILLE ET VILAINE;1
35203;La;Nouaye;NOUAYE;35137;35;ILLE ET VILAINE;1
35204;;Nouvoitou;NOUVOITOU;35410;35;ILLE ET VILAINE;1
35205;;Noyal-sous-Bazouges;NOYAL-SOUS-BAZOUGES;35560;35;ILLE ET VILAINE;1
35206;;Noyal-Chtillon-sur-Seiche;NOYAL-CHATILLON-SUR-SEICHE;35230;35;ILLE ET VILAINE;1
35207;;Noyal-sur-Vilaine;NOYAL-SUR-VILAINE;35530;35;ILLE ET VILAINE;1
35208;;Orgres;ORGERES;35230;35;ILLE ET VILAINE;1
35209;;Oss;OSSE;35410;35;ILLE ET VILAINE;1
35210;;Pac;PACE;35740;35;ILLE ET VILAINE;1
35211;;Paimpont;PAIMPONT;35380;35;ILLE ET VILAINE;1
35212;;Panc;PANCE;35320;35;ILLE ET VILAINE;1
35214;;Parc;PARCE;35210;35;ILLE ET VILAINE;1
35215;;Parign;PARIGNE;35133;35;ILLE ET VILAINE;1
35216;;Parthenay-de-Bretagne;PARTHENAY-DE-BRETAGNE;35850;35;ILLE ET VILAINE;1
35217;Le;Pertre;PERTRE;35370;35;ILLE ET VILAINE;1
35218;Le;Petit-Fougeray;PETIT-FOUGERAY;35320;35;ILLE ET VILAINE;1
35219;;Pipriac;PIPRIAC;35550;35;ILLE ET VILAINE;1
35220;;Pir-sur-Seiche;PIRE-SUR-SEICHE;35150;35;ILLE ET VILAINE;1
35221;;Plchtel;PLECHATEL;35470;35;ILLE ET VILAINE;1
35222;;Pleine-Fougres;PLEINE-FOUGERES;35610;35;ILLE ET VILAINE;1
35223;;Pllan-le-Grand;PLELAN-LE-GRAND;35380;35;ILLE ET VILAINE;1
35224;;Plerguer;PLERGUER;35540;35;ILLE ET VILAINE;1
35225;;Plesder;PLESDER;35720;35;ILLE ET VILAINE;1
35226;;Pleugueneuc;PLEUGUENEUC;35720;35;ILLE ET VILAINE;1
35227;;Pleumeleuc;PLEUMELEUC;35137;35;ILLE ET VILAINE;1
35228;;Pleurtuit;PLEURTUIT;35730;35;ILLE ET VILAINE;1
35229;;Poc-les-Bois;POCE-LES-BOIS;35500;35;ILLE ET VILAINE;1
35230;;Poilley;POILLEY;35420;35;ILLE ET VILAINE;1
35231;;Polign;POLIGNE;35320;35;ILLE ET VILAINE;1
35232;;Princ;PRINCE;35210;35;ILLE ET VILAINE;1
35233;;Qubriac;QUEBRIAC;35190;35;ILLE ET VILAINE;1
35234;;Qudillac;QUEDILLAC;35290;35;ILLE ET VILAINE;1
35235;;Ranne;RANNEE;35130;35;ILLE ET VILAINE;1
35236;;Redon;REDON;35600;35;ILLE ET VILAINE;1
35237;;Renac;RENAC;35660;35;ILLE ET VILAINE;1
35238;;Rennes;RENNES;35000;35;ILLE ET VILAINE;1
35239;;Retiers;RETIERS;35240;35;ILLE ET VILAINE;1
35240;Le;Rheu;RHEU;35650;35;ILLE ET VILAINE;1
35241;La;Richardais;RICHARDAIS;35780;35;ILLE ET VILAINE;1
35242;;Rimou;RIMOU;35560;35;ILLE ET VILAINE;1
35243;;Romagn;ROMAGNE;35133;35;ILLE ET VILAINE;1
35244;;Romazy;ROMAZY;35490;35;ILLE ET VILAINE;1
35245;;Romill;ROMILLE;35850;35;ILLE ET VILAINE;1
35246;;Roz-Landrieux;ROZ-LANDRIEUX;35120;35;ILLE ET VILAINE;1
35247;;Roz-sur-Couesnon;ROZ-SUR-COUESNON;35610;35;ILLE ET VILAINE;1
35248;;Sains;SAINS;35610;35;ILLE ET VILAINE;1
35249;;Sainte-Anne-sur-Vilaine;SAINTE-ANNE-SUR-VILAINE;35390;35;ILLE ET VILAINE;1
35250;;Saint-Armel;SAINT-ARMEL;35230;35;ILLE ET VILAINE;1
35251;;Saint-Aubin-d'Aubign;SAINT-AUBIN-D'AUBIGNE;35250;35;ILLE ET VILAINE;1
35252;;Saint-Aubin-des-Landes;SAINT-AUBIN-DES-LANDES;35500;35;ILLE ET VILAINE;1
35253;;Saint-Aubin-du-Cormier;SAINT-AUBIN-DU-CORMIER;35140;35;ILLE ET VILAINE;1
35254;;Saint-Aubin-du-Pavail;SAINT-AUBIN-DU-PAVAIL;35410;35;ILLE ET VILAINE;1
35255;;Saint-Benot-des-Ondes;SAINT-BENOIT-DES-ONDES;35114;35;ILLE ET VILAINE;1
35256;;Saint-Briac-sur-Mer;SAINT-BRIAC-SUR-MER;35800;35;ILLE ET VILAINE;1
35257;;Saint-Brice-en-Cogls;SAINT-BRICE-EN-COGLES;35460;35;ILLE ET VILAINE;1
35258;;Saint-Brieuc-des-Iffs;SAINT-BRIEUC-DES-IFFS;35630;35;ILLE ET VILAINE;1
35259;;Saint-Broladre;SAINT-BROLADRE;35120;35;ILLE ET VILAINE;1
35260;;Saint-Christophe-des-Bois;SAINT-CHRISTOPHE-DES-BOIS;35210;35;ILLE ET VILAINE;1
35261;;Saint-Christophe-de-Valains;SAINT-CHRISTOPHE-DE-VALAINS;35140;35;ILLE ET VILAINE;1
35262;;Sainte-Colombe;SAINTE-COLOMBE;35134;35;ILLE ET VILAINE;1
35263;;Saint-Coulomb;SAINT-COULOMB;35350;35;ILLE ET VILAINE;1
35264;;Saint-Didier;SAINT-DIDIER;35220;35;ILLE ET VILAINE;1
35265;;Saint-Domineuc;SAINT-DOMINEUC;35190;35;ILLE ET VILAINE;1
35266;;Saint-Erblon;SAINT-ERBLON;35230;35;ILLE ET VILAINE;1
35267;;Saint-tienne-en-Cogls;SAINT-ETIENNE-EN-COGLES;35460;35;ILLE ET VILAINE;1
35268;;Saint-Ganton;SAINT-GANTON;35550;35;ILLE ET VILAINE;1
35269;;Saint-Georges-de-Chesn;SAINT-GEORGES-DE-CHESNE;35140;35;ILLE ET VILAINE;1
35270;;Saint-Georges-de-Grhaigne;SAINT-GEORGES-DE-GREHAIGNE;35610;35;ILLE ET VILAINE;1
35271;;Saint-Georges-de-Reintembault;SAINT-GEORGES-DE-REINTEMBAULT;35420;35;ILLE ET VILAINE;1
35272;;Saint-Germain-du-Pinel;SAINT-GERMAIN-DU-PINEL;35370;35;ILLE ET VILAINE;1
35273;;Saint-Germain-en-Cogls;SAINT-GERMAIN-EN-COGLES;35133;35;ILLE ET VILAINE;1
35274;;Saint-Germain-sur-Ille;SAINT-GERMAIN-SUR-ILLE;35250;35;ILLE ET VILAINE;1
35275;;Saint-Gilles;SAINT-GILLES;35590;35;ILLE ET VILAINE;1
35276;;Saint-Gondran;SAINT-GONDRAN;35630;35;ILLE ET VILAINE;1
35277;;Saint-Gonlay;SAINT-GONLAY;35750;35;ILLE ET VILAINE;1
35278;;Saint-Grgoire;SAINT-GREGOIRE;35760;35;ILLE ET VILAINE;1
35279;;Saint-Guinoux;SAINT-GUINOUX;35430;35;ILLE ET VILAINE;1
35280;;Saint-Hilaire-des-Landes;SAINT-HILAIRE-DES-LANDES;35140;35;ILLE ET VILAINE;1
35281;;Saint-Jacques-de-la-Lande;SAINT-JACQUES-DE-LA-LANDE;35136;35;ILLE ET VILAINE;1
35282;;Saint-Jean-sur-Couesnon;SAINT-JEAN-SUR-COUESNON;35140;35;ILLE ET VILAINE;1
35283;;Saint-Jean-sur-Vilaine;SAINT-JEAN-SUR-VILAINE;35220;35;ILLE ET VILAINE;1
35284;;Saint-Jouan-des-Gurets;SAINT-JOUAN-DES-GUERETS;35430;35;ILLE ET VILAINE;1
35285;;Saint-Just;SAINT-JUST;35550;35;ILLE ET VILAINE;1
35286;;Saint-Lger-des-Prs;SAINT-LEGER-DES-PRES;35270;35;ILLE ET VILAINE;1
35287;;Saint-Lunaire;SAINT-LUNAIRE;35800;35;ILLE ET VILAINE;1
35288;;Saint-Malo;SAINT-MALO;35400;35;ILLE ET VILAINE;1
35289;;Saint-Malo-de-Phily;SAINT-MALO-DE-PHILY;35480;35;ILLE ET VILAINE;1
35290;;Saint-Malon-sur-Mel;SAINT-MALON-SUR-MEL;35750;35;ILLE ET VILAINE;1
35291;;Saint-Marcan;SAINT-MARCAN;35120;35;ILLE ET VILAINE;1
35292;;Saint-Marc-le-Blanc;SAINT-MARC-LE-BLANC;35460;35;ILLE ET VILAINE;1
35293;;Saint-Marc-sur-Couesnon;SAINT-MARC-SUR-COUESNON;35140;35;ILLE ET VILAINE;1
35294;;Sainte-Marie;SAINTE-MARIE;35600;35;ILLE ET VILAINE;1
35295;;Saint-Maugan;SAINT-MAUGAN;35750;35;ILLE ET VILAINE;1
35296;;Saint-Mdard-sur-Ille;SAINT-MEDARD-SUR-ILLE;35250;35;ILLE ET VILAINE;1
35297;;Saint-Men-le-Grand;SAINT-MEEN-LE-GRAND;35290;35;ILLE ET VILAINE;1
35299;;Saint-Mloir-des-Ondes;SAINT-MELOIR-DES-ONDES;35350;35;ILLE ET VILAINE;1
35300;;Saint-M'Herv;SAINT-M'HERVE;35500;35;ILLE ET VILAINE;1
35301;;Saint-M'Hervon;SAINT-M'HERVON;35360;35;ILLE ET VILAINE;1
35302;;Saint-Onen-la-Chapelle;SAINT-ONEN-LA-CHAPELLE;35290;35;ILLE ET VILAINE;1
35303;;Saint-Ouen-la-Rourie;SAINT-OUEN-LA-ROUERIE;35460;35;ILLE ET VILAINE;1
35304;;Saint-Ouen-des-Alleux;SAINT-OUEN-DES-ALLEUX;35140;35;ILLE ET VILAINE;1
35305;;Saint-Pran;SAINT-PERAN;35380;35;ILLE ET VILAINE;1
35306;;Saint-Pre;SAINT-PERE;35430;35;ILLE ET VILAINE;1
35307;;Saint-Pern;SAINT-PERN;35190;35;ILLE ET VILAINE;1
35308;;Saint-Pierre-de-Plesguen;SAINT-PIERRE-DE-PLESGUEN;35720;35;ILLE ET VILAINE;1
35309;;Saint-Rmy-du-Plain;SAINT-REMY-DU-PLAIN;35560;35;ILLE ET VILAINE;1
35310;;Saint-Sauveur-des-Landes;SAINT-SAUVEUR-DES-LANDES;35133;35;ILLE ET VILAINE;1
35311;;Saint-Sglin;SAINT-SEGLIN;35330;35;ILLE ET VILAINE;1
35312;;Saint-Senoux;SAINT-SENOUX;35580;35;ILLE ET VILAINE;1
35314;;Saint-Suliac;SAINT-SULIAC;35430;35;ILLE ET VILAINE;1
35315;;Saint-Sulpice-la-Fort;SAINT-SULPICE-LA-FORET;35250;35;ILLE ET VILAINE;1
35316;;Saint-Sulpice-des-Landes;SAINT-SULPICE-DES-LANDES;35390;35;ILLE ET VILAINE;1
35317;;Saint-Symphorien;SAINT-SYMPHORIEN;35630;35;ILLE ET VILAINE;1
35318;;Saint-Thual;SAINT-THUAL;35190;35;ILLE ET VILAINE;1
35319;;Saint-Thurial;SAINT-THURIAL;35310;35;ILLE ET VILAINE;1
35320;;Saint-Uniac;SAINT-UNIAC;35360;35;ILLE ET VILAINE;1
35321;;Saulnires;SAULNIERES;35320;35;ILLE ET VILAINE;1
35322;Le;Sel-de-Bretagne;SEL-DE-BRETAGNE;35320;35;ILLE ET VILAINE;1
35323;La;Selle-en-Cogls;SELLE-EN-COGLES;35460;35;ILLE ET VILAINE;1
35324;La;Selle-en-Luitr;SELLE-EN-LUITRE;35133;35;ILLE ET VILAINE;1
35325;La;Selle-Guerchaise;SELLE-GUERCHAISE;35130;35;ILLE ET VILAINE;1
35326;;Sens-de-Bretagne;SENS-DE-BRETAGNE;35490;35;ILLE ET VILAINE;1
35327;;Servon-sur-Vilaine;SERVON-SUR-VILAINE;35530;35;ILLE ET VILAINE;1
35328;;Sixt-sur-Aff;SIXT-SUR-AFF;35550;35;ILLE ET VILAINE;1
35329;;Sougal;SOUGEAL;35610;35;ILLE ET VILAINE;1
35330;;Taillis;TAILLIS;35500;35;ILLE ET VILAINE;1
35331;;Talensac;TALENSAC;35160;35;ILLE ET VILAINE;1
35332;;Teillay;TEILLAY;35620;35;ILLE ET VILAINE;1
35333;Le;Theil-de-Bretagne;THEIL-DE-BRETAGNE;35240;35;ILLE ET VILAINE;1
35334;;Thorign-Fouillard;THORIGNE-FOUILLARD;35235;35;ILLE ET VILAINE;1
35335;;Thourie;THOURIE;35134;35;ILLE ET VILAINE;1
35336;Le;Tiercent;TIERCENT;35460;35;ILLE ET VILAINE;1
35337;;Tintniac;TINTENIAC;35190;35;ILLE ET VILAINE;1
35338;;Torc;TORCE;35370;35;ILLE ET VILAINE;1
35339;;Trans-la-Fort;TRANS-LA-FORET;35610;35;ILLE ET VILAINE;1
35340;;Treffendel;TREFFENDEL;35380;35;ILLE ET VILAINE;1
35341;;Tremblay;TREMBLAY;35460;35;ILLE ET VILAINE;1
35342;;Trmeheuc;TREMEHEUC;35270;35;ILLE ET VILAINE;1
35343;;Tresbuf;TRESBOEUF;35320;35;ILLE ET VILAINE;1
35344;;Tress;TRESSE;35720;35;ILLE ET VILAINE;1
35345;;Trvrien;TREVERIEN;35190;35;ILLE ET VILAINE;1
35346;;Trimer;TRIMER;35190;35;ILLE ET VILAINE;1
35347;;Val-d'Iz;VAL-D'IZE;35450;35;ILLE ET VILAINE;1
35348;;Vendel;VENDEL;35140;35;ILLE ET VILAINE;1
35350;;Vergal;VERGEAL;35680;35;ILLE ET VILAINE;1
35351;Le;Verger;VERGER;35160;35;ILLE ET VILAINE;1
35352;;Vern-sur-Seiche;VERN-SUR-SEICHE;35770;35;ILLE ET VILAINE;1
35353;;Vezin-le-Coquet;VEZIN-LE-COQUET;35132;35;ILLE ET VILAINE;1
35354;;Vieux-Viel;VIEUX-VIEL;35610;35;ILLE ET VILAINE;1
35355;;Vieux-Vy-sur-Couesnon;VIEUX-VY-SUR-COUESNON;35490;35;ILLE ET VILAINE;1
35356;;Vignoc;VIGNOC;35630;35;ILLE ET VILAINE;1
35357;;Villame;VILLAMEE;35420;35;ILLE ET VILAINE;1
35358;La;Ville-s-Nonais;VILLE-ES-NONAIS;35430;35;ILLE ET VILAINE;1
35359;;Visseiche;VISSEICHE;35130;35;ILLE ET VILAINE;1
35360;;Vitr;VITRE;35500;35;ILLE ET VILAINE;1
35361;Le;Vivier-sur-Mer;VIVIER-SUR-MER;35960;35;ILLE ET VILAINE;1
35362;Le;Tronchet;TRONCHET;35540;35;ILLE ET VILAINE;1
35363;;Pont-Pan;PONT-PEAN;35131;35;ILLE ET VILAINE;1
36001;;Aigurande;AIGURANDE;36140;36;INDRE;1
36002;;Aize;AIZE;36150;36;INDRE;1
36003;;Ambrault;AMBRAULT;36120;36;INDRE;1
36004;;Anjouin;ANJOUIN;36210;36;INDRE;1
36005;;Ardentes;ARDENTES;36120;36;INDRE;1
36006;;Argenton-sur-Creuse;ARGENTON-SUR-CREUSE;36200;36;INDRE;1
36007;;Argy;ARGY;36500;36;INDRE;1
36008;;Arpheuilles;ARPHEUILLES;36700;36;INDRE;1
36009;;Arthon;ARTHON;36330;36;INDRE;1
36010;;Azay-le-Ferron;AZAY-LE-FERRON;36290;36;INDRE;1
36011;;Bagneux;BAGNEUX;36210;36;INDRE;1
36012;;Baraize;BARAIZE;36270;36;INDRE;1
36013;;Baudres;BAUDRES;36110;36;INDRE;1
36014;;Bazaiges;BAZAIGES;36270;36;INDRE;1
36015;;Beaulieu;BEAULIEU;36310;36;INDRE;1
36016;;Blbre;BELABRE;36370;36;INDRE;1
36017;La;Berthenoux;BERTHENOUX;36400;36;INDRE;1
36018;Le;Blanc;BLANC;36300;36;INDRE;1
36019;;Bommiers;BOMMIERS;36120;36;INDRE;1
36020;;Bonneuil;BONNEUIL;36310;36;INDRE;1
36021;Les;Bordes;BORDES;36100;36;INDRE;1
36022;;Bouesse;BOUESSE;36200;36;INDRE;1
36023;;Bouges-le-Chteau;BOUGES-LE-CHATEAU;36110;36;INDRE;1
36024;;Bretagne;BRETAGNE;36110;36;INDRE;1
36025;;Briantes;BRIANTES;36400;36;INDRE;1
36026;;Brion;BRION;36110;36;INDRE;1
36027;;Brives;BRIVES;36100;36;INDRE;1
36028;La;Buxerette;BUXERETTE;36140;36;INDRE;1
36029;;Buxeuil;BUXEUIL;36150;36;INDRE;1
36030;;Buxires-d'Aillac;BUXIERES-D'AILLAC;36230;36;INDRE;1
36031;;Buzanais;BUZANCAIS;36500;36;INDRE;1
36032;;Ceaulmont;CEAULMONT;36200;36;INDRE;1
36033;;Celon;CELON;36200;36;INDRE;1
36034;;Chabris;CHABRIS;36210;36;INDRE;1
36035;;Chaillac;CHAILLAC;36310;36;INDRE;1
36036;;Chalais;CHALAIS;36370;36;INDRE;1
36037;La;Champenoise;CHAMPENOISE;36100;36;INDRE;1
36038;;Champillet;CHAMPILLET;36160;36;INDRE;1
36040;La;Chapelle-Orthemale;CHAPELLE-ORTHEMALE;36500;36;INDRE;1
36041;La;Chapelle-Saint-Laurian;CHAPELLE-SAINT-LAURIAN;36150;36;INDRE;1
36042;;Chasseneuil;CHASSENEUIL;36800;36;INDRE;1
36043;;Chassignolles;CHASSIGNOLLES;36400;36;INDRE;1
36044;;Chteauroux;CHATEAUROUX;36000;36;INDRE;1
36045;;Chtillon-sur-Indre;CHATILLON-SUR-INDRE;36700;36;INDRE;1
36046;La;Chtre;CHATRE;36400;36;INDRE;1
36047;La;Chtre-Langlin;CHATRE-LANGLIN;36170;36;INDRE;1
36048;;Chavin;CHAVIN;36200;36;INDRE;1
36049;;Chazelet;CHAZELET;36170;36;INDRE;1
36050;;Chezelles;CHEZELLES;36500;36;INDRE;1
36051;;Chitray;CHITRAY;36800;36;INDRE;1
36052;;Chouday;CHOUDAY;36100;36;INDRE;1
36053;;Ciron;CIRON;36300;36;INDRE;1
36054;;Clr-du-Bois;CLERE-DU-BOIS;36700;36;INDRE;1
36055;;Clion;CLION;36700;36;INDRE;1
36056;;Cluis;CLUIS;36340;36;INDRE;1
36057;;Coings;COINGS;36130;36;INDRE;1
36058;;Concremiers;CONCREMIERS;36300;36;INDRE;1
36059;;Cond;CONDE;36100;36;INDRE;1
36060;;Crevant;CREVANT;36140;36;INDRE;1
36061;;Crozon-sur-Vauvre;CROZON-SUR-VAUVRE;36140;36;INDRE;1
36062;;Cuzion;CUZION;36190;36;INDRE;1
36063;;Dols;DEOLS;36130;36;INDRE;1
36064;;Diors;DIORS;36130;36;INDRE;1
36065;;Diou;DIOU;36260;36;INDRE;1
36066;;Douadic;DOUADIC;36300;36;INDRE;1
36067;;Dunet;DUNET;36310;36;INDRE;1
36068;;Dun-le-Polier;DUN-LE-POELIER;36210;36;INDRE;1
36069;;cueill;ECUEILLE;36240;36;INDRE;1
36070;;guzon-Chantme;EGUZON-CHANTOME;36270;36;INDRE;1
36071;;trechet;ETRECHET;36120;36;INDRE;1
36072;;Faverolles;FAVEROLLES;36360;36;INDRE;1
36073;;Feusines;FEUSINES;36160;36;INDRE;1
36074;;Flr-la-Rivire;FLERE-LA-RIVIERE;36700;36;INDRE;1
36075;;Fontenay;FONTENAY;36150;36;INDRE;1
36076;;Fontgombault;FONTGOMBAULT;36220;36;INDRE;1
36077;;Fontguenand;FONTGUENAND;36600;36;INDRE;1
36078;;Fougerolles;FOUGEROLLES;36230;36;INDRE;1
36079;;Francillon;FRANCILLON;36110;36;INDRE;1
36080;;Frdille;FREDILLE;36180;36;INDRE;1
36081;;Gargilesse-Dampierre;GARGILESSE-DAMPIERRE;36190;36;INDRE;1
36082;;Gehe;GEHEE;36240;36;INDRE;1
36083;;Giroux;GIROUX;36150;36;INDRE;1
36084;;Gournay;GOURNAY;36230;36;INDRE;1
36085;;Guilly;GUILLY;36150;36;INDRE;1
36086;;Heugnes;HEUGNES;36180;36;INDRE;1
36087;;Ingrandes;INGRANDES;36300;36;INDRE;1
36088;;Issoudun;ISSOUDUN;36100;36;INDRE;1
36089;;Jeu-les-Bois;JEU-LES-BOIS;36120;36;INDRE;1
36090;;Jeu-Maloches;JEU-MALOCHES;36240;36;INDRE;1
36091;;Lacs;LACS;36400;36;INDRE;1
36092;;Lang;LANGE;36600;36;INDRE;1
36093;;Levroux;LEVROUX;36110;36;INDRE;1
36094;;Lignac;LIGNAC;36370;36;INDRE;1
36095;;Lignerolles;LIGNEROLLES;36160;36;INDRE;1
36096;;Ling;LINGE;36220;36;INDRE;1
36097;;Liniez;LINIEZ;36150;36;INDRE;1
36098;;Lizeray;LIZERAY;36100;36;INDRE;1
36099;;Lourdoueix-Saint-Michel;LOURDOUEIX-SAINT-MICHEL;36140;36;INDRE;1
36100;;Lourouer-Saint-Laurent;LOUROUER-SAINT-LAURENT;36400;36;INDRE;1
36101;;Luant;LUANT;36350;36;INDRE;1
36102;;Luay-le-Libre;LUCAY-LE-LIBRE;36150;36;INDRE;1
36103;;Luay-le-Mle;LUCAY-LE-MALE;36360;36;INDRE;1
36104;;Lurais;LURAIS;36220;36;INDRE;1
36105;;Lureuil;LUREUIL;36220;36;INDRE;1
36106;;Luzeret;LUZERET;36800;36;INDRE;1
36107;;Lye;LYE;36600;36;INDRE;1
36108;;Lys-Saint-Georges;LYS-SAINT-GEORGES;36230;36;INDRE;1
36109;Le;Magny;MAGNY;36400;36;INDRE;1
36110;;Maillet;MAILLET;36340;36;INDRE;1
36111;;Malicornay;MALICORNAY;36340;36;INDRE;1
36112;;Mron;MARON;36120;36;INDRE;1
36113;;Martizay;MARTIZAY;36220;36;INDRE;1
36114;;Mauvires;MAUVIERES;36370;36;INDRE;1
36115;;Menetou-sur-Nahon;MENETOU-SUR-NAHON;36210;36;INDRE;1
36116;;Mntrols-sous-Vatan;MENETREOLS-SOUS-VATAN;36150;36;INDRE;1
36117;Le;Menoux;MENOUX;36200;36;INDRE;1
36118;;Mobecq;MEOBECQ;36500;36;INDRE;1
36119;;Mrigny;MERIGNY;36220;36;INDRE;1
36120;;Mers-sur-Indre;MERS-SUR-INDRE;36230;36;INDRE;1
36121;;Meunet-Planches;MEUNET-PLANCHES;36100;36;INDRE;1
36122;;Meunet-sur-Vatan;MEUNET-SUR-VATAN;36150;36;INDRE;1
36123;;Mzires-en-Brenne;MEZIERES-EN-BRENNE;36290;36;INDRE;1
36124;;Mign;MIGNE;36800;36;INDRE;1
36125;;Migny;MIGNY;36260;36;INDRE;1
36126;;Montchevrier;MONTCHEVRIER;36140;36;INDRE;1
36127;;Montgivray;MONTGIVRAY;36400;36;INDRE;1
36128;;Montierchaume;MONTIERCHAUME;36130;36;INDRE;1
36129;;Montipouret;MONTIPOURET;36230;36;INDRE;1
36130;;Montlevicq;MONTLEVICQ;36400;36;INDRE;1
36131;;Mosnay;MOSNAY;36200;36;INDRE;1
36132;La;Motte-Feuilly;MOTTE-FEUILLY;36160;36;INDRE;1
36133;;Mouhers;MOUHERS;36340;36;INDRE;1
36134;;Mouhet;MOUHET;36170;36;INDRE;1
36135;;Moulins-sur-Cphons;MOULINS-SUR-CEPHONS;36110;36;INDRE;1
36136;;Murs;MURS;36700;36;INDRE;1
36137;;Nons-sur-Creuse;NEONS-SUR-CREUSE;36220;36;INDRE;1
36138;;Nret;NERET;36400;36;INDRE;1
36139;;Neuillay-les-Bois;NEUILLAY-LES-BOIS;36500;36;INDRE;1
36140;;Neuvy-Pailloux;NEUVY-PAILLOUX;36100;36;INDRE;1
36141;;Neuvy-Saint-Spulchre;NEUVY-SAINT-SEPULCHRE;36230;36;INDRE;1
36142;;Niherne;NIHERNE;36250;36;INDRE;1
36143;;Nohant-Vic;NOHANT-VIC;36400;36;INDRE;1
36144;;Nuret-le-Ferron;NURET-LE-FERRON;36800;36;INDRE;1
36145;;Obterre;OBTERRE;36290;36;INDRE;1
36146;;Orsennes;ORSENNES;36190;36;INDRE;1
36147;;Orville;ORVILLE;36210;36;INDRE;1
36148;;Oulches;OULCHES;36800;36;INDRE;1
36149;;Palluau-sur-Indre;PALLUAU-SUR-INDRE;36500;36;INDRE;1
36150;;Parnac;PARNAC;36170;36;INDRE;1
36151;;Parpeay;PARPECAY;36210;36;INDRE;1
36152;;Paudy;PAUDY;36260;36;INDRE;1
36153;;Paulnay;PAULNAY;36290;36;INDRE;1
36154;Le;Pchereau;PECHEREAU;36200;36;INDRE;1
36155;;Pellevoisin;PELLEVOISIN;36180;36;INDRE;1
36156;;Prassay;PERASSAY;36160;36;INDRE;1
36157;La;Prouille;PEROUILLE;36350;36;INDRE;1
36158;;Badecon-le-Pin;BADECON-LE-PIN;36200;36;INDRE;1
36159;Le;Poinonnet;POINCONNET;36330;36;INDRE;1
36160;;Pommiers;POMMIERS;36190;36;INDRE;1
36161;Le;Pont-Chrtien-Chabenet;PONT-CHRETIEN-CHABENET;36800;36;INDRE;1
36162;;Poulaines;POULAINES;36210;36;INDRE;1
36163;;Pouligny-Notre-Dame;POULIGNY-NOTRE-DAME;36160;36;INDRE;1
36164;;Pouligny-Saint-Martin;POULIGNY-SAINT-MARTIN;36160;36;INDRE;1
36165;;Pouligny-Saint-Pierre;POULIGNY-SAINT-PIERRE;36300;36;INDRE;1
36166;;Praux;PREAUX;36240;36;INDRE;1
36167;;Preuilly-la-Ville;PREUILLY-LA-VILLE;36220;36;INDRE;1
36168;;Prissac;PRISSAC;36370;36;INDRE;1
36169;;Pruniers;PRUNIERS;36120;36;INDRE;1
36170;;Reboursin;REBOURSIN;36150;36;INDRE;1
36171;;Reuilly;REUILLY;36260;36;INDRE;1
36172;;Rivarennes;RIVARENNES;36800;36;INDRE;1
36173;;Rosnay;ROSNAY;36300;36;INDRE;1
36174;;Roussines;ROUSSINES;36170;36;INDRE;1
36175;;Rouvres-les-Bois;ROUVRES-LES-BOIS;36110;36;INDRE;1
36176;;Ruffec;RUFFEC;36300;36;INDRE;1
36177;;Sacierges-Saint-Martin;SACIERGES-SAINT-MARTIN;36170;36;INDRE;1
36178;;Saint-Aigny;SAINT-AIGNY;36300;36;INDRE;1
36179;;Saint-Aoustrille;SAINT-AOUSTRILLE;36100;36;INDRE;1
36180;;Saint-Aot;SAINT-AOUT;36120;36;INDRE;1
36181;;Saint-Aubin;SAINT-AUBIN;36100;36;INDRE;1
36182;;Saint-Benot-du-Sault;SAINT-BENOIT-DU-SAULT;36170;36;INDRE;1
36183;;Sainte-Ccile;SAINTE-CECILE;36210;36;INDRE;1
36184;;Saint-Chartier;SAINT-CHARTIER;36400;36;INDRE;1
36185;;Saint-Christophe-en-Bazelle;SAINT-CHRISTOPHE-EN-BAZELLE;36210;36;INDRE;1
36186;;Saint-Christophe-en-Boucherie;SAINT-CHRISTOPHE-EN-BOUCHERIE;36400;36;INDRE;1
36187;;Saint-Civran;SAINT-CIVRAN;36170;36;INDRE;1
36188;;Saint-Cyran-du-Jambot;SAINT-CYRAN-DU-JAMBOT;36700;36;INDRE;1
36189;;Saint-Denis-de-Jouhet;SAINT-DENIS-DE-JOUHET;36230;36;INDRE;1
36190;;Sainte-Fauste;SAINTE-FAUSTE;36100;36;INDRE;1
36191;;Saint-Florentin;SAINT-FLORENTIN;36150;36;INDRE;1
36192;;Saint-Gaultier;SAINT-GAULTIER;36800;36;INDRE;1
36193;;Sainte-Gemme;SAINTE-GEMME;36500;36;INDRE;1
36194;;Saint-Genou;SAINT-GENOU;36500;36;INDRE;1
36195;;Saint-Georges-sur-Arnon;SAINT-GEORGES-SUR-ARNON;36100;36;INDRE;1
36196;;Saint-Gilles;SAINT-GILLES;36170;36;INDRE;1
36197;;Saint-Hilaire-sur-Benaize;SAINT-HILAIRE-SUR-BENAIZE;36370;36;INDRE;1
36198;;Saint-Lactencin;SAINT-LACTENCIN;36500;36;INDRE;1
36199;;Sainte-Lizaigne;SAINTE-LIZAIGNE;36260;36;INDRE;1
36200;;Saint-Marcel;SAINT-MARCEL;36200;36;INDRE;1
36201;;Saint-Martin-de-Lamps;SAINT-MARTIN-DE-LAMPS;36110;36;INDRE;1
36202;;Saint-Maur;SAINT-MAUR;36250;36;INDRE;1
36203;;Saint-Mdard;SAINT-MEDARD;36700;36;INDRE;1
36204;;Saint-Michel-en-Brenne;SAINT-MICHEL-EN-BRENNE;36290;36;INDRE;1
36205;;Saint-Pierre-de-Jards;SAINT-PIERRE-DE-JARDS;36260;36;INDRE;1
36206;;Saint-Pierre-de-Lamps;SAINT-PIERRE-DE-LAMPS;36110;36;INDRE;1
36207;;Saint-Plantaire;SAINT-PLANTAIRE;36190;36;INDRE;1
36208;;Sainte-Svre-sur-Indre;SAINTE-SEVERE-SUR-INDRE;36160;36;INDRE;1
36209;;Saint-Valentin;SAINT-VALENTIN;36100;36;INDRE;1
36210;;Sarzay;SARZAY;36230;36;INDRE;1
36211;;Sassierges-Saint-Germain;SASSIERGES-SAINT-GERMAIN;36120;36;INDRE;1
36212;;Saulnay;SAULNAY;36290;36;INDRE;1
36213;;Sauzelles;SAUZELLES;36220;36;INDRE;1
36214;;Sazeray;SAZERAY;36160;36;INDRE;1
36215;;Sgry;SEGRY;36100;36;INDRE;1
36216;;Selles-sur-Nahon;SELLES-SUR-NAHON;36180;36;INDRE;1
36217;;Sembleay;SEMBLECAY;36210;36;INDRE;1
36218;;Soug;SOUGE;36500;36;INDRE;1
36219;;Tendu;TENDU;36200;36;INDRE;1
36220;;Thenay;THENAY;36800;36;INDRE;1
36221;;Thevet-Saint-Julien;THEVET-SAINT-JULIEN;36400;36;INDRE;1
36222;;Thizay;THIZAY;36100;36;INDRE;1
36223;;Tilly;TILLY;36310;36;INDRE;1
36224;;Tournon-Saint-Martin;TOURNON-SAINT-MARTIN;36220;36;INDRE;1
36225;Le;Tranger;TRANGER;36700;36;INDRE;1
36226;;Tranzault;TRANZAULT;36230;36;INDRE;1
36227;;Urciers;URCIERS;36160;36;INDRE;1
36228;;Valenay;VALENCAY;36600;36;INDRE;1
36229;;Varennes-sur-Fouzon;VARENNES-SUR-FOUZON;36210;36;INDRE;1
36230;;Vatan;VATAN;36150;36;INDRE;1
36231;;Velles;VELLES;36330;36;INDRE;1
36232;;Venduvres;VENDOEUVRES;36500;36;INDRE;1
36233;La;Vernelle;VERNELLE;36600;36;INDRE;1
36234;;Verneuil-sur-Igneraie;VERNEUIL-SUR-IGNERAIE;36400;36;INDRE;1
36235;;Veuil;VEUIL;36600;36;INDRE;1
36236;;Vicq-Exemplet;VICQ-EXEMPLET;36400;36;INDRE;1
36237;;Vicq-sur-Nahon;VICQ-SUR-NAHON;36600;36;INDRE;1
36238;;Vigoulant;VIGOULANT;36160;36;INDRE;1
36239;;Vigoux;VIGOUX;36170;36;INDRE;1
36240;;Vijon;VIJON;36160;36;INDRE;1
36241;;Villedieu-sur-Indre;VILLEDIEU-SUR-INDRE;36320;36;INDRE;1
36242;;Villegongis;VILLEGONGIS;36110;36;INDRE;1
36243;;Villegouin;VILLEGOUIN;36500;36;INDRE;1
36244;;Villentrois;VILLENTROIS;36600;36;INDRE;1
36245;;Villers-les-Ormes;VILLERS-LES-ORMES;36250;36;INDRE;1
36246;;Villiers;VILLIERS;36290;36;INDRE;1
36247;;Vineuil;VINEUIL;36110;36;INDRE;1
36248;;Vouillon;VOUILLON;36100;36;INDRE;1
37001;;Abilly;ABILLY;37160;37;INDRE ET LOIRE;1
37002;;Ambillou;AMBILLOU;37340;37;INDRE ET LOIRE;1
37003;;Amboise;AMBOISE;37400;37;INDRE ET LOIRE;1
37004;;Anch;ANCHE;37500;37;INDRE ET LOIRE;1
37005;;Antogny-le-Tillac;ANTOGNY-LE-TILLAC;37800;37;INDRE ET LOIRE;1
37006;;Artannes-sur-Indre;ARTANNES-SUR-INDRE;37260;37;INDRE ET LOIRE;1
37007;;Assay;ASSAY;37120;37;INDRE ET LOIRE;1
37008;;Athe-sur-Cher;ATHEE-SUR-CHER;37270;37;INDRE ET LOIRE;1
37009;;Autrche;AUTRECHE;37110;37;INDRE ET LOIRE;1
37010;;Auzouer-en-Touraine;AUZOUER-EN-TOURAINE;37110;37;INDRE ET LOIRE;1
37011;;Avoine;AVOINE;37420;37;INDRE ET LOIRE;1
37012;;Avon-les-Roches;AVON-LES-ROCHES;37220;37;INDRE ET LOIRE;1
37013;;Avrill-les-Ponceaux;AVRILLE-LES-PONCEAUX;37340;37;INDRE ET LOIRE;1
37014;;Azay-le-Rideau;AZAY-LE-RIDEAU;37190;37;INDRE ET LOIRE;1
37015;;Azay-sur-Cher;AZAY-SUR-CHER;37270;37;INDRE ET LOIRE;1
37016;;Azay-sur-Indre;AZAY-SUR-INDRE;37310;37;INDRE ET LOIRE;1
37018;;Ballan-Mir;BALLAN-MIRE;37510;37;INDRE ET LOIRE;1
37019;;Barrou;BARROU;37350;37;INDRE ET LOIRE;1
37020;;Beaulieu-ls-Loches;BEAULIEU-LES-LOCHES;37600;37;INDRE ET LOIRE;1
37021;;Beaumont-la-Ronce;BEAUMONT-LA-RONCE;37360;37;INDRE ET LOIRE;1
37022;;Beaumont-en-Vron;BEAUMONT-EN-VERON;37420;37;INDRE ET LOIRE;1
37023;;Beaumont-Village;BEAUMONT-VILLAGE;37460;37;INDRE ET LOIRE;1
37024;;Benais;BENAIS;37140;37;INDRE ET LOIRE;1
37025;;Berthenay;BERTHENAY;37510;37;INDRE ET LOIRE;1
37026;;Betz-le-Chteau;BETZ-LE-CHATEAU;37600;37;INDRE ET LOIRE;1
37027;;Blr;BLERE;37150;37;INDRE ET LOIRE;1
37028;;Bossay-sur-Claise;BOSSAY-SUR-CLAISE;37290;37;INDRE ET LOIRE;1
37029;;Bosse;BOSSEE;37240;37;INDRE ET LOIRE;1
37030;Le;Boulay;BOULAY;37110;37;INDRE ET LOIRE;1
37031;;Bourgueil;BOURGUEIL;37140;37;INDRE ET LOIRE;1
37032;;Bournan;BOURNAN;37240;37;INDRE ET LOIRE;1
37033;;Boussay;BOUSSAY;37290;37;INDRE ET LOIRE;1
37034;;Braslou;BRASLOU;37120;37;INDRE ET LOIRE;1
37035;;Braye-sous-Faye;BRAYE-SOUS-FAYE;37120;37;INDRE ET LOIRE;1
37036;;Braye-sur-Maulne;BRAYE-SUR-MAULNE;37330;37;INDRE ET LOIRE;1
37037;;Brches;BRECHES;37330;37;INDRE ET LOIRE;1
37038;;Brhmont;BREHEMONT;37130;37;INDRE ET LOIRE;1
37039;;Bridor;BRIDORE;37600;37;INDRE ET LOIRE;1
37040;;Brizay;BRIZAY;37220;37;INDRE ET LOIRE;1
37041;;Bueil-en-Touraine;BUEIL-EN-TOURAINE;37370;37;INDRE ET LOIRE;1
37042;;Candes-Saint-Martin;CANDES-SAINT-MARTIN;37500;37;INDRE ET LOIRE;1
37043;;Cangey;CANGEY;37530;37;INDRE ET LOIRE;1
37044;La;Celle-Guenand;CELLE-GUENAND;37350;37;INDRE ET LOIRE;1
37045;La;Celle-Saint-Avant;CELLE-SAINT-AVANT;37160;37;INDRE ET LOIRE;1
37046;;Cr-la-Ronde;CERE-LA-RONDE;37460;37;INDRE ET LOIRE;1
37047;;Cerelles;CERELLES;37390;37;INDRE ET LOIRE;1
37048;;Chambon;CHAMBON;37290;37;INDRE ET LOIRE;1
37049;;Chambourg-sur-Indre;CHAMBOURG-SUR-INDRE;37310;37;INDRE ET LOIRE;1
37050;;Chambray-ls-Tours;CHAMBRAY-LES-TOURS;37170;37;INDRE ET LOIRE;1
37051;;Champigny-sur-Veude;CHAMPIGNY-SUR-VEUDE;37120;37;INDRE ET LOIRE;1
37052;;Chanay;CHANCAY;37210;37;INDRE ET LOIRE;1
37053;;Chanceaux-prs-Loches;CHANCEAUX-PRES-LOCHES;37600;37;INDRE ET LOIRE;1
37054;;Chanceaux-sur-Choisille;CHANCEAUX-SUR-CHOISILLE;37390;37;INDRE ET LOIRE;1
37055;;Channay-sur-Lathan;CHANNAY-SUR-LATHAN;37330;37;INDRE ET LOIRE;1
37056;La;Chapelle-aux-Naux;CHAPELLE-AUX-NAUX;37130;37;INDRE ET LOIRE;1
37057;La;Chapelle-Blanche-Saint-Martin;CHAPELLE-BLANCHE-SAINT-MARTIN;37240;37;INDRE ET LOIRE;1
37058;La;Chapelle-sur-Loire;CHAPELLE-SUR-LOIRE;37140;37;INDRE ET LOIRE;1
37059;;Charentilly;CHARENTILLY;37390;37;INDRE ET LOIRE;1
37060;;Charg;CHARGE;37530;37;INDRE ET LOIRE;1
37061;;Charnizay;CHARNIZAY;37290;37;INDRE ET LOIRE;1
37062;;Chteau-la-Vallire;CHATEAU-LA-VALLIERE;37330;37;INDRE ET LOIRE;1
37063;;Chteau-Renault;CHATEAU-RENAULT;37110;37;INDRE ET LOIRE;1
37064;;Chaumussay;CHAUMUSSAY;37350;37;INDRE ET LOIRE;1
37065;;Chaveignes;CHAVEIGNES;37120;37;INDRE ET LOIRE;1
37066;;Chdigny;CHEDIGNY;37310;37;INDRE ET LOIRE;1
37067;;Cheill;CHEILLE;37190;37;INDRE ET LOIRE;1
37068;;Chemill-sur-Dme;CHEMILLE-SUR-DEME;37370;37;INDRE ET LOIRE;1
37069;;Chemill-sur-Indrois;CHEMILLE-SUR-INDROIS;37460;37;INDRE ET LOIRE;1
37070;;Chenonceaux;CHENONCEAUX;37150;37;INDRE ET LOIRE;1
37071;;Chezelles;CHEZELLES;37220;37;INDRE ET LOIRE;1
37072;;Chinon;CHINON;37500;37;INDRE ET LOIRE;1
37073;;Chisseaux;CHISSEAUX;37150;37;INDRE ET LOIRE;1
37074;;Chouz-sur-Loire;CHOUZE-SUR-LOIRE;37140;37;INDRE ET LOIRE;1
37075;;Cigogn;CIGOGNE;37310;37;INDRE ET LOIRE;1
37076;;Cinais;CINAIS;37500;37;INDRE ET LOIRE;1
37077;;Cinq-Mars-la-Pile;CINQ-MARS-LA-PILE;37130;37;INDRE ET LOIRE;1
37078;;Ciran;CIRAN;37240;37;INDRE ET LOIRE;1
37079;;Civray-de-Touraine;CIVRAY-DE-TOURAINE;37150;37;INDRE ET LOIRE;1
37080;;Civray-sur-Esves;CIVRAY-SUR-ESVES;37160;37;INDRE ET LOIRE;1
37081;;Clr-les-Pins;CLERE-LES-PINS;37340;37;INDRE ET LOIRE;1
37082;;Continvoir;CONTINVOIR;37340;37;INDRE ET LOIRE;1
37083;;Cormery;CORMERY;37320;37;INDRE ET LOIRE;1
37084;;Couesmes;COUESMES;37330;37;INDRE ET LOIRE;1
37085;;Couray;COURCAY;37310;37;INDRE ET LOIRE;1
37086;;Courcelles-de-Touraine;COURCELLES-DE-TOURAINE;37330;37;INDRE ET LOIRE;1
37087;;Courcou;COURCOUE;37120;37;INDRE ET LOIRE;1
37088;;Couziers;COUZIERS;37500;37;INDRE ET LOIRE;1
37089;;Cravant-les-Cteaux;CRAVANT-LES-COTEAUX;37500;37;INDRE ET LOIRE;1
37090;;Crissay-sur-Manse;CRISSAY-SUR-MANSE;37220;37;INDRE ET LOIRE;1
37091;La;Croix-en-Touraine;CROIX-EN-TOURAINE;37150;37;INDRE ET LOIRE;1
37092;;Crotelles;CROTELLES;37380;37;INDRE ET LOIRE;1
37093;;Crouzilles;CROUZILLES;37220;37;INDRE ET LOIRE;1
37094;;Cussay;CUSSAY;37240;37;INDRE ET LOIRE;1
37095;;Dame-Marie-les-Bois;DAME-MARIE-LES-BOIS;37110;37;INDRE ET LOIRE;1
37096;;Dierre;DIERRE;37150;37;INDRE ET LOIRE;1
37097;;Dolus-le-Sec;DOLUS-LE-SEC;37310;37;INDRE ET LOIRE;1
37098;;Drach;DRACHE;37800;37;INDRE ET LOIRE;1
37099;;Druye;DRUYE;37190;37;INDRE ET LOIRE;1
37100;;peign-les-Bois;EPEIGNE-LES-BOIS;37150;37;INDRE ET LOIRE;1
37101;;peign-sur-Dme;EPEIGNE-SUR-DEME;37370;37;INDRE ET LOIRE;1
37102;Les;Essards;ESSARDS;37130;37;INDRE ET LOIRE;1
37103;;Esves-le-Moutier;ESVES-LE-MOUTIER;37240;37;INDRE ET LOIRE;1
37104;;Esvres;ESVRES;37320;37;INDRE ET LOIRE;1
37105;;Faye-la-Vineuse;FAYE-LA-VINEUSE;37120;37;INDRE ET LOIRE;1
37106;La;Ferrire;FERRIERE;37110;37;INDRE ET LOIRE;1
37107;;Ferrire-Laron;FERRIERE-LARCON;37350;37;INDRE ET LOIRE;1
37108;;Ferrire-sur-Beaulieu;FERRIERE-SUR-BEAULIEU;37600;37;INDRE ET LOIRE;1
37109;;Fondettes;FONDETTES;37230;37;INDRE ET LOIRE;1
37110;;Francueil;FRANCUEIL;37150;37;INDRE ET LOIRE;1
37111;;Genill;GENILLE;37460;37;INDRE ET LOIRE;1
37112;;Gizeux;GIZEUX;37340;37;INDRE ET LOIRE;1
37113;Le;Grand-Pressigny;GRAND-PRESSIGNY;37350;37;INDRE ET LOIRE;1
37114;La;Guerche;GUERCHE;37350;37;INDRE ET LOIRE;1
37115;;Descartes;DESCARTES;37160;37;INDRE ET LOIRE;1
37116;Les;Hermites;HERMITES;37110;37;INDRE ET LOIRE;1
37117;;Hommes;HOMMES;37340;37;INDRE ET LOIRE;1
37118;;Huismes;HUISMES;37420;37;INDRE ET LOIRE;1
37119;L';le-Bouchard;ILE-BOUCHARD;37220;37;INDRE ET LOIRE;1
37120;;Ingrandes-de-Touraine;INGRANDES-DE-TOURAINE;37140;37;INDRE ET LOIRE;1
37121;;Jaulnay;JAULNAY;37120;37;INDRE ET LOIRE;1
37122;;Jou-ls-Tours;JOUE-LES-TOURS;37300;37;INDRE ET LOIRE;1
37123;;Langeais;LANGEAIS;37130;37;INDRE ET LOIRE;1
37124;;Laray;LARCAY;37270;37;INDRE ET LOIRE;1
37125;;Lmer;LEMERE;37120;37;INDRE ET LOIRE;1
37126;;Lern;LERNE;37500;37;INDRE ET LOIRE;1
37127;Le;Lige;LIEGE;37460;37;INDRE ET LOIRE;1
37128;;Lignires-de-Touraine;LIGNIERES-DE-TOURAINE;37130;37;INDRE ET LOIRE;1
37129;;Ligr;LIGRE;37500;37;INDRE ET LOIRE;1
37130;;Ligueil;LIGUEIL;37240;37;INDRE ET LOIRE;1
37131;;Limeray;LIMERAY;37530;37;INDRE ET LOIRE;1
37132;;Loches;LOCHES;37600;37;INDRE ET LOIRE;1
37133;;Loch-sur-Indrois;LOCHE-SUR-INDROIS;37460;37;INDRE ET LOIRE;1
37134;;Louans;LOUANS;37320;37;INDRE ET LOIRE;1
37135;;Louestault;LOUESTAULT;37370;37;INDRE ET LOIRE;1
37136;Le;Louroux;LOUROUX;37240;37;INDRE ET LOIRE;1
37137;;Lubl;LUBLE;37330;37;INDRE ET LOIRE;1
37138;;Lussault-sur-Loire;LUSSAULT-SUR-LOIRE;37400;37;INDRE ET LOIRE;1
37139;;Luynes;LUYNES;37230;37;INDRE ET LOIRE;1
37140;;Luz;LUZE;37120;37;INDRE ET LOIRE;1
37141;;Luzill;LUZILLE;37150;37;INDRE ET LOIRE;1
37142;;Maill;MAILLE;37800;37;INDRE ET LOIRE;1
37143;;Manthelan;MANTHELAN;37240;37;INDRE ET LOIRE;1
37144;;Maray;MARCAY;37500;37;INDRE ET LOIRE;1
37145;;Marc-sur-Esves;MARCE-SUR-ESVES;37160;37;INDRE ET LOIRE;1
37146;;Marcilly-sur-Maulne;MARCILLY-SUR-MAULNE;37330;37;INDRE ET LOIRE;1
37147;;Marcilly-sur-Vienne;MARCILLY-SUR-VIENNE;37800;37;INDRE ET LOIRE;1
37148;;Marigny-Marmande;MARIGNY-MARMANDE;37120;37;INDRE ET LOIRE;1
37149;;Marray;MARRAY;37370;37;INDRE ET LOIRE;1
37150;;Mazires-de-Touraine;MAZIERES-DE-TOURAINE;37130;37;INDRE ET LOIRE;1
37151;La;Membrolle-sur-Choisille;MEMBROLLE-SUR-CHOISILLE;37390;37;INDRE ET LOIRE;1
37152;;Mettray;METTRAY;37390;37;INDRE ET LOIRE;1
37153;;Monnaie;MONNAIE;37380;37;INDRE ET LOIRE;1
37154;;Montbazon;MONTBAZON;37250;37;INDRE ET LOIRE;1
37155;;Monthodon;MONTHODON;37110;37;INDRE ET LOIRE;1
37156;;Montlouis-sur-Loire;MONTLOUIS-SUR-LOIRE;37270;37;INDRE ET LOIRE;1
37157;;Montrsor;MONTRESOR;37460;37;INDRE ET LOIRE;1
37158;;Montreuil-en-Touraine;MONTREUIL-EN-TOURAINE;37530;37;INDRE ET LOIRE;1
37159;;Monts;MONTS;37260;37;INDRE ET LOIRE;1
37160;;Morand;MORAND;37110;37;INDRE ET LOIRE;1
37161;;Mosnes;MOSNES;37530;37;INDRE ET LOIRE;1
37162;;Mouzay;MOUZAY;37600;37;INDRE ET LOIRE;1
37163;;Nazelles-Ngron;NAZELLES-NEGRON;37530;37;INDRE ET LOIRE;1
37165;;Neuil;NEUIL;37190;37;INDRE ET LOIRE;1
37166;;Neuill-le-Lierre;NEUILLE-LE-LIERRE;37380;37;INDRE ET LOIRE;1
37167;;Neuill-Pont-Pierre;NEUILLE-PONT-PIERRE;37360;37;INDRE ET LOIRE;1
37168;;Neuilly-le-Brignon;NEUILLY-LE-BRIGNON;37160;37;INDRE ET LOIRE;1
37169;;Neuville-sur-Brenne;NEUVILLE-SUR-BRENNE;37110;37;INDRE ET LOIRE;1
37170;;Neuvy-le-Roi;NEUVY-LE-ROI;37370;37;INDRE ET LOIRE;1
37171;;Noizay;NOIZAY;37210;37;INDRE ET LOIRE;1
37172;;Notre-Dame-d'O;NOTRE-DAME-D'OE;37390;37;INDRE ET LOIRE;1
37173;;Nouans-les-Fontaines;NOUANS-LES-FONTAINES;37460;37;INDRE ET LOIRE;1
37174;;Noutre;NOUATRE;37800;37;INDRE ET LOIRE;1
37175;;Nouzilly;NOUZILLY;37380;37;INDRE ET LOIRE;1
37176;;Noyant-de-Touraine;NOYANT-DE-TOURAINE;37800;37;INDRE ET LOIRE;1
37177;;Orbigny;ORBIGNY;37460;37;INDRE ET LOIRE;1
37178;;Panzoult;PANZOULT;37220;37;INDRE ET LOIRE;1
37179;;Paray-Meslay;PARCAY-MESLAY;37210;37;INDRE ET LOIRE;1
37180;;Paray-sur-Vienne;PARCAY-SUR-VIENNE;37220;37;INDRE ET LOIRE;1
37181;;Paulmy;PAULMY;37350;37;INDRE ET LOIRE;1
37182;;Pernay;PERNAY;37230;37;INDRE ET LOIRE;1
37183;;Perrusson;PERRUSSON;37600;37;INDRE ET LOIRE;1
37184;Le;Petit-Pressigny;PETIT-PRESSIGNY;37350;37;INDRE ET LOIRE;1
37185;;Poc-sur-Cisse;POCE-SUR-CISSE;37530;37;INDRE ET LOIRE;1
37186;;Pont-de-Ruan;PONT-DE-RUAN;37260;37;INDRE ET LOIRE;1
37187;;Ports;PORTS;37800;37;INDRE ET LOIRE;1
37188;;Pouzay;POUZAY;37800;37;INDRE ET LOIRE;1
37189;;Preuilly-sur-Claise;PREUILLY-SUR-CLAISE;37290;37;INDRE ET LOIRE;1
37190;;Pussigny;PUSSIGNY;37800;37;INDRE ET LOIRE;1
37191;;Razines;RAZINES;37120;37;INDRE ET LOIRE;1
37192;;Reignac-sur-Indre;REIGNAC-SUR-INDRE;37310;37;INDRE ET LOIRE;1
37193;;Restign;RESTIGNE;37140;37;INDRE ET LOIRE;1
37194;;Reugny;REUGNY;37380;37;INDRE ET LOIRE;1
37195;La;Riche;RICHE;37520;37;INDRE ET LOIRE;1
37196;;Richelieu;RICHELIEU;37120;37;INDRE ET LOIRE;1
37197;;Rigny-Uss;RIGNY-USSE;37420;37;INDRE ET LOIRE;1
37198;;Rill;RILLE;37340;37;INDRE ET LOIRE;1
37199;;Rilly-sur-Vienne;RILLY-SUR-VIENNE;37220;37;INDRE ET LOIRE;1
37200;;Rivarennes;RIVARENNES;37190;37;INDRE ET LOIRE;1
37201;;Rivire;RIVIERE;37500;37;INDRE ET LOIRE;1
37202;La;Roche-Clermault;ROCHE-CLERMAULT;37500;37;INDRE ET LOIRE;1
37203;;Rochecorbon;ROCHECORBON;37210;37;INDRE ET LOIRE;1
37204;;Rouziers-de-Touraine;ROUZIERS-DE-TOURAINE;37360;37;INDRE ET LOIRE;1
37205;;Sach;SACHE;37190;37;INDRE ET LOIRE;1
37206;;Saint-Antoine-du-Rocher;SAINT-ANTOINE-DU-ROCHER;37360;37;INDRE ET LOIRE;1
37207;;Saint-Aubin-le-Dpeint;SAINT-AUBIN-LE-DEPEINT;37370;37;INDRE ET LOIRE;1
37208;;Saint-Avertin;SAINT-AVERTIN;37550;37;INDRE ET LOIRE;1
37209;;Saint-Bauld;SAINT-BAULD;37310;37;INDRE ET LOIRE;1
37210;;Saint-Benot-la-Fort;SAINT-BENOIT-LA-FORET;37500;37;INDRE ET LOIRE;1
37211;;Saint-Branchs;SAINT-BRANCHS;37320;37;INDRE ET LOIRE;1
37212;;Sainte-Catherine-de-Fierbois;SAINTE-CATHERINE-DE-FIERBOIS;37800;37;INDRE ET LOIRE;1
37213;;Saint-Christophe-sur-le-Nais;SAINT-CHRISTOPHE-SUR-LE-NAIS;37370;37;INDRE ET LOIRE;1
37214;;Saint-Cyr-sur-Loire;SAINT-CYR-SUR-LOIRE;37540;37;INDRE ET LOIRE;1
37216;;Saint-pain;SAINT-EPAIN;37800;37;INDRE ET LOIRE;1
37217;;Saint-tienne-de-Chigny;SAINT-ETIENNE-DE-CHIGNY;37230;37;INDRE ET LOIRE;1
37218;;Saint-Flovier;SAINT-FLOVIER;37600;37;INDRE ET LOIRE;1
37219;;Saint-Genouph;SAINT-GENOUPH;37510;37;INDRE ET LOIRE;1
37220;;Saint-Germain-sur-Vienne;SAINT-GERMAIN-SUR-VIENNE;37500;37;INDRE ET LOIRE;1
37221;;Saint-Hippolyte;SAINT-HIPPOLYTE;37600;37;INDRE ET LOIRE;1
37222;;Saint-Jean-Saint-Germain;SAINT-JEAN-SAINT-GERMAIN;37600;37;INDRE ET LOIRE;1
37223;;Saint-Laurent-de-Lin;SAINT-LAURENT-DE-LIN;37330;37;INDRE ET LOIRE;1
37224;;Saint-Laurent-en-Gtines;SAINT-LAURENT-EN-GATINES;37380;37;INDRE ET LOIRE;1
37225;;Saint-Martin-le-Beau;SAINT-MARTIN-LE-BEAU;37270;37;INDRE ET LOIRE;1
37226;;Sainte-Maure-de-Touraine;SAINTE-MAURE-DE-TOURAINE;37800;37;INDRE ET LOIRE;1
37227;;Saint-Michel-sur-Loire;SAINT-MICHEL-SUR-LOIRE;37130;37;INDRE ET LOIRE;1
37228;;Saint-Nicolas-de-Bourgueil;SAINT-NICOLAS-DE-BOURGUEIL;37140;37;INDRE ET LOIRE;1
37229;;Saint-Nicolas-des-Motets;SAINT-NICOLAS-DES-MOTETS;37110;37;INDRE ET LOIRE;1
37230;;Saint-Ouen-les-Vignes;SAINT-OUEN-LES-VIGNES;37530;37;INDRE ET LOIRE;1
37231;;Saint-Paterne-Racan;SAINT-PATERNE-RACAN;37370;37;INDRE ET LOIRE;1
37232;;Saint-Patrice;SAINT-PATRICE;37130;37;INDRE ET LOIRE;1
37233;;Saint-Pierre-des-Corps;SAINT-PIERRE-DES-CORPS;37700;37;INDRE ET LOIRE;1
37234;;Saint-Quentin-sur-Indrois;SAINT-QUENTIN-SUR-INDROIS;37310;37;INDRE ET LOIRE;1
37236;;Saint-Rgle;SAINT-REGLE;37530;37;INDRE ET LOIRE;1
37237;;Saint-Roch;SAINT-ROCH;37390;37;INDRE ET LOIRE;1
37238;;Saint-Senoch;SAINT-SENOCH;37600;37;INDRE ET LOIRE;1
37240;;Saunay;SAUNAY;37110;37;INDRE ET LOIRE;1
37241;;Savign-sur-Lathan;SAVIGNE-SUR-LATHAN;37340;37;INDRE ET LOIRE;1
37242;;Savigny-en-Vron;SAVIGNY-EN-VERON;37420;37;INDRE ET LOIRE;1
37243;;Savonnires;SAVONNIERES;37510;37;INDRE ET LOIRE;1
37244;;Sazilly;SAZILLY;37220;37;INDRE ET LOIRE;1
37245;;Semblanay;SEMBLANCAY;37360;37;INDRE ET LOIRE;1
37246;;Sennevires;SENNEVIERES;37600;37;INDRE ET LOIRE;1
37247;;Sepmes;SEPMES;37800;37;INDRE ET LOIRE;1
37248;;Seuilly;SEUILLY;37500;37;INDRE ET LOIRE;1
37249;;Sonzay;SONZAY;37360;37;INDRE ET LOIRE;1
37250;;Sorigny;SORIGNY;37250;37;INDRE ET LOIRE;1
37251;;Souvign;SOUVIGNE;37330;37;INDRE ET LOIRE;1
37252;;Souvigny-de-Touraine;SOUVIGNY-DE-TOURAINE;37530;37;INDRE ET LOIRE;1
37253;;Sublaines;SUBLAINES;37310;37;INDRE ET LOIRE;1
37254;;Tauxigny;TAUXIGNY;37310;37;INDRE ET LOIRE;1
37255;;Tavant;TAVANT;37220;37;INDRE ET LOIRE;1
37256;;Theneuil;THENEUIL;37220;37;INDRE ET LOIRE;1
37257;;Thilouze;THILOUZE;37260;37;INDRE ET LOIRE;1
37258;;Thizay;THIZAY;37500;37;INDRE ET LOIRE;1
37259;;Tournon-Saint-Pierre;TOURNON-SAINT-PIERRE;37290;37;INDRE ET LOIRE;1
37260;La;Tour-Saint-Gelin;TOUR-SAINT-GELIN;37120;37;INDRE ET LOIRE;1
37261;;Tours;TOURS;37000;37;INDRE ET LOIRE;1
37262;;Trogues;TROGUES;37220;37;INDRE ET LOIRE;1
37263;;Truyes;TRUYES;37320;37;INDRE ET LOIRE;1
37264;;Vallres;VALLERES;37190;37;INDRE ET LOIRE;1
37265;;Varennes;VARENNES;37600;37;INDRE ET LOIRE;1
37266;;Veign;VEIGNE;37250;37;INDRE ET LOIRE;1
37267;;Vretz;VERETZ;37270;37;INDRE ET LOIRE;1
37268;;Verneuil-le-Chteau;VERNEUIL-LE-CHATEAU;37120;37;INDRE ET LOIRE;1
37269;;Verneuil-sur-Indre;VERNEUIL-SUR-INDRE;37600;37;INDRE ET LOIRE;1
37270;;Vernou-sur-Brenne;VERNOU-SUR-BRENNE;37210;37;INDRE ET LOIRE;1
37271;;Villaines-les-Rochers;VILLAINES-LES-ROCHERS;37190;37;INDRE ET LOIRE;1
37272;;Villandry;VILLANDRY;37510;37;INDRE ET LOIRE;1
37273;La;Ville-aux-Dames;VILLE-AUX-DAMES;37700;37;INDRE ET LOIRE;1
37274;;Villebourg;VILLEBOURG;37370;37;INDRE ET LOIRE;1
37275;;Villedmain;VILLEDOMAIN;37460;37;INDRE ET LOIRE;1
37276;;Villedmer;VILLEDOMER;37110;37;INDRE ET LOIRE;1
37277;;Villeloin-Coulang;VILLELOIN-COULANGE;37460;37;INDRE ET LOIRE;1
37278;;Villeperdue;VILLEPERDUE;37260;37;INDRE ET LOIRE;1
37279;;Villiers-au-Bouin;VILLIERS-AU-BOUIN;37330;37;INDRE ET LOIRE;1
37280;;Vou;VOU;37240;37;INDRE ET LOIRE;1
37281;;Vouvray;VOUVRAY;37210;37;INDRE ET LOIRE;1
37282;;Yzeures-sur-Creuse;YZEURES-SUR-CREUSE;37290;37;INDRE ET LOIRE;1
38001;Les;Abrets;ABRETS;38490;38;ISERE;1
38002;Les;Adrets;ADRETS;38190;38;ISERE;1
38003;;Agnin;AGNIN;38150;38;ISERE;1
38004;L';Albenc;ALBENC;38470;38;ISERE;1
38005;;Allemond;ALLEMOND;38114;38;ISERE;1
38006;;Allevard;ALLEVARD;38580;38;ISERE;1
38008;;Ambel;AMBEL;38970;38;ISERE;1
38009;;Anjou;ANJOU;38150;38;ISERE;1
38010;;Annoisin-Chatelans;ANNOISIN-CHATELANS;38460;38;ISERE;1
38011;;Anthon;ANTHON;38280;38;ISERE;1
38012;;Aoste;AOSTE;38490;38;ISERE;1
38013;;Apprieu;APPRIEU;38140;38;ISERE;1
38014;;Arandon;ARANDON;38510;38;ISERE;1
38015;;Artas;ARTAS;38440;38;ISERE;1
38016;;Arzay;ARZAY;38260;38;ISERE;1
38017;;Assieu;ASSIEU;38150;38;ISERE;1
38018;;Auberives-en-Royans;AUBERIVES-EN-ROYANS;38680;38;ISERE;1
38019;;Auberives-sur-Varze;AUBERIVES-SUR-VAREZE;38550;38;ISERE;1
38020;;Auris;AURIS;38142;38;ISERE;1
38021;;Autrans;AUTRANS;38880;38;ISERE;1
38022;Les;Avenires;AVENIERES;38630;38;ISERE;1
38023;;Avignonet;AVIGNONET;38650;38;ISERE;1
38024;;Badinires;BADINIERES;38300;38;ISERE;1
38025;;Balbins;BALBINS;38260;38;ISERE;1
38026;La;Balme-les-Grottes;BALME-LES-GROTTES;38390;38;ISERE;1
38027;;Barraux;BARRAUX;38530;38;ISERE;1
38028;La;Btie-Divisin;BATIE-DIVISIN;38490;38;ISERE;1
38029;La;Btie-Montgascon;BATIE-MONTGASCON;38110;38;ISERE;1
38030;;Beaucroissant;BEAUCROISSANT;38140;38;ISERE;1
38031;;Beaufin;BEAUFIN;38970;38;ISERE;1
38032;;Beaufort;BEAUFORT;38270;38;ISERE;1
38033;;Beaulieu;BEAULIEU;38470;38;ISERE;1
38034;;Beaurepaire;BEAUREPAIRE;38270;38;ISERE;1
38035;;Beauvoir-de-Marc;BEAUVOIR-DE-MARC;38440;38;ISERE;1
38036;;Beauvoir-en-Royans;BEAUVOIR-EN-ROYANS;38160;38;ISERE;1
38037;;Bellegarde-Poussieu;BELLEGARDE-POUSSIEU;38270;38;ISERE;1
38038;;Belmont;BELMONT;38690;38;ISERE;1
38039;;Bernin;BERNIN;38190;38;ISERE;1
38040;;Besse;BESSE;38142;38;ISERE;1
38041;;Bessins;BESSINS;38160;38;ISERE;1
38042;;Bvenais;BEVENAIS;38690;38;ISERE;1
38043;;Bilieu;BILIEU;38850;38;ISERE;1
38044;;Biol;BIOL;38690;38;ISERE;1
38045;;Biviers;BIVIERS;38330;38;ISERE;1
38046;;Bizonnes;BIZONNES;38690;38;ISERE;1
38047;;Blandin;BLANDIN;38730;38;ISERE;1
38048;;Bonnefamille;BONNEFAMILLE;38090;38;ISERE;1
38049;;Bossieu;BOSSIEU;38260;38;ISERE;1
38050;Le;Bouchage;BOUCHAGE;38510;38;ISERE;1
38051;;Boug-Chambalud;BOUGE-CHAMBALUD;38150;38;ISERE;1
38052;Le;Bourg-d'Oisans;BOURG-D'OISANS;38520;38;ISERE;1
38053;;Bourgoin-Jallieu;BOURGOIN-JALLIEU;38300;38;ISERE;1
38054;;Bouvesse-Quirieu;BOUVESSE-QUIRIEU;38390;38;ISERE;1
38055;;Brangues;BRANGUES;38510;38;ISERE;1
38056;;Bressieux;BRESSIEUX;38870;38;ISERE;1
38057;;Bresson;BRESSON;38320;38;ISERE;1
38058;;Brzins;BREZINS;38590;38;ISERE;1
38059;;Bri-et-Angonnes;BRIE-ET-ANGONNES;38320;38;ISERE;1
38060;;Brion;BRION;38590;38;ISERE;1
38061;La;Buisse;BUISSE;38500;38;ISERE;1
38062;La;Buissire;BUISSIERE;38530;38;ISERE;1
38063;;Burcin;BURCIN;38690;38;ISERE;1
38064;;Cessieu;CESSIEU;38110;38;ISERE;1
38065;;Chbons;CHABONS;38690;38;ISERE;1
38066;;Chlons;CHALONS;38122;38;ISERE;1
38067;;Chamagnieu;CHAMAGNIEU;38460;38;ISERE;1
38068;;Champagnier;CHAMPAGNIER;38800;38;ISERE;1
38069;;Champier;CHAMPIER;38260;38;ISERE;1
38070;Le;Champ-prs-Froges;CHAMP-PRES-FROGES;38190;38;ISERE;1
38071;;Champ-sur-Drac;CHAMP-SUR-DRAC;38560;38;ISERE;1
38072;;Chanas;CHANAS;38150;38;ISERE;1
38073;;Chantelouve;CHANTELOUVE;38740;38;ISERE;1
38074;;Chantesse;CHANTESSE;38470;38;ISERE;1
38075;;Chapareillan;CHAPAREILLAN;38530;38;ISERE;1
38076;La;Chapelle-de-la-Tour;CHAPELLE-DE-LA-TOUR;38110;38;ISERE;1
38077;La;Chapelle-de-Surieu;CHAPELLE-DE-SURIEU;38150;38;ISERE;1
38078;La;Chapelle-du-Bard;CHAPELLE-DU-BARD;38580;38;ISERE;1
38080;;Charancieu;CHARANCIEU;38490;38;ISERE;1
38081;;Charantonnay;CHARANTONNAY;38790;38;ISERE;1
38082;;Charavines;CHARAVINES;38850;38;ISERE;1
38083;;Charette;CHARETTE;38390;38;ISERE;1
38084;;Charncles;CHARNECLES;38140;38;ISERE;1
38085;;Charvieu-Chavagneux;CHARVIEU-CHAVAGNEUX;38230;38;ISERE;1
38086;;Chasselay;CHASSELAY;38470;38;ISERE;1
38087;;Chasse-sur-Rhne;CHASSE-SUR-RHONE;38670;38;ISERE;1
38089;;Chassignieu;CHASSIGNIEU;38730;38;ISERE;1
38090;;Chteau-Bernard;CHATEAU-BERNARD;38650;38;ISERE;1
38091;;Chteauvilain;CHATEAUVILAIN;38300;38;ISERE;1
38092;;Chtelus;CHATELUS;38680;38;ISERE;1
38093;;Chtenay;CHATENAY;38980;38;ISERE;1
38094;;Chtonnay;CHATONNAY;38440;38;ISERE;1
38095;;Chatte;CHATTE;38160;38;ISERE;1
38097;;Chavanoz;CHAVANOZ;38230;38;ISERE;1
38098;;Chlieu;CHELIEU;38730;38;ISERE;1
38099;;Chevrires;CHEVRIERES;38160;38;ISERE;1
38100;Le;Cheylas;CHEYLAS;38570;38;ISERE;1
38101;;Cheyssieu;CHEYSSIEU;38550;38;ISERE;1
38102;;Chzeneuve;CHEZENEUVE;38300;38;ISERE;1
38103;;Chichilianne;CHICHILIANNE;38930;38;ISERE;1
38104;;Chimilin;CHIMILIN;38490;38;ISERE;1
38105;;Chirens;CHIRENS;38850;38;ISERE;1
38106;;Cholonge;CHOLONGE;38220;38;ISERE;1
38107;;Chonas-l'Amballan;CHONAS-L'AMBALLAN;38121;38;ISERE;1
38108;;Choranche;CHORANCHE;38680;38;ISERE;1
38109;;Chozeau;CHOZEAU;38460;38;ISERE;1
38110;;Chuzelles;CHUZELLES;38200;38;ISERE;1
38111;;Claix;CLAIX;38640;38;ISERE;1
38112;;Clavans-en-Haut-Oisans;CLAVANS-EN-HAUT-OISANS;38142;38;ISERE;1
38113;;Clelles;CLELLES;38930;38;ISERE;1
38114;;Clonas-sur-Varze;CLONAS-SUR-VAREZE;38550;38;ISERE;1
38115;;Saint-Martin-de-la-Cluze;SAINT-MARTIN-DE-LA-CLUZE;38650;38;ISERE;1
38116;;Cognet;COGNET;38350;38;ISERE;1
38117;;Cognin-les-Gorges;COGNIN-LES-GORGES;38470;38;ISERE;1
38118;;Colombe;COLOMBE;38690;38;ISERE;1
38120;La;Combe-de-Lancey;COMBE-DE-LANCEY;38190;38;ISERE;1
38121;;Commelle;COMMELLE;38260;38;ISERE;1
38124;;Corbelin;CORBELIN;38630;38;ISERE;1
38125;;Cordac;CORDEAC;38710;38;ISERE;1
38126;;Corenc;CORENC;38700;38;ISERE;1
38127;;Cornillon-en-Trives;CORNILLON-EN-TRIEVES;38710;38;ISERE;1
38128;;Corps;CORPS;38970;38;ISERE;1
38129;;Correnon-en-Vercors;CORRENCON-EN-VERCORS;38250;38;ISERE;1
38130;La;Cte-Saint-Andr;COTE-SAINT-ANDRE;38260;38;ISERE;1
38131;Les;Ctes-d'Arey;COTES-D'AREY;38138;38;ISERE;1
38132;Les;Ctes-de-Corps;COTES-DE-CORPS;38970;38;ISERE;1
38133;;Coublevie;COUBLEVIE;38500;38;ISERE;1
38134;;Cour-et-Buis;COUR-ET-BUIS;38122;38;ISERE;1
38135;;Courtenay;COURTENAY;38510;38;ISERE;1
38136;;Crachier;CRACHIER;38300;38;ISERE;1
38137;;Cras;CRAS;38210;38;ISERE;1
38138;;Crmieu;CREMIEU;38460;38;ISERE;1
38139;;Creys-Mpieu;CREYS-MEPIEU;38510;38;ISERE;1
38140;;Crolles;CROLLES;38190;38;ISERE;1
38141;;Culin;CULIN;38300;38;ISERE;1
38144;;Dimoz;DIEMOZ;38790;38;ISERE;1
38145;;Dionay;DIONAY;38160;38;ISERE;1
38146;;Dizimieu;DIZIMIEU;38460;38;ISERE;1
38147;;Doissin;DOISSIN;38730;38;ISERE;1
38148;;Dolomieu;DOLOMIEU;38110;38;ISERE;1
38149;;Domarin;DOMARIN;38300;38;ISERE;1
38150;;Domne;DOMENE;38420;38;ISERE;1
38151;;chirolles;ECHIROLLES;38130;38;ISERE;1
38152;;Eclose;ECLOSE;38300;38;ISERE;1
38153;;Engins;ENGINS;38360;38;ISERE;1
38154;;Entraigues;ENTRAIGUES;38740;38;ISERE;1
38155;;Entre-deux-Guiers;ENTRE-DEUX-GUIERS;38380;38;ISERE;1
38156;Les;parres;EPARRES;38300;38;ISERE;1
38157;;Estrablin;ESTRABLIN;38780;38;ISERE;1
38158;;Eybens;EYBENS;38320;38;ISERE;1
38159;;Eydoche;EYDOCHE;38690;38;ISERE;1
38160;;Eyzin-Pinet;EYZIN-PINET;38780;38;ISERE;1
38161;;Faramans;FARAMANS;38260;38;ISERE;1
38162;;Faverges-de-la-Tour;FAVERGES-DE-LA-TOUR;38110;38;ISERE;1
38163;La;Ferrire;FERRIERE;38580;38;ISERE;1
38165;;Fitilieu;FITILIEU;38490;38;ISERE;1
38166;La;Flachre;FLACHERE;38530;38;ISERE;1
38167;;Flachres;FLACHERES;38690;38;ISERE;1
38169;;Fontaine;FONTAINE;38600;38;ISERE;1
38170;;Fontanil-Cornillon;FONTANIL-CORNILLON;38120;38;ISERE;1
38171;La;Forteresse;FORTERESSE;38590;38;ISERE;1
38172;;Four;FOUR;38080;38;ISERE;1
38173;Le;Freney-d'Oisans;FRENEY-D'OISANS;38142;38;ISERE;1
38174;La;Frette;FRETTE;38260;38;ISERE;1
38175;;Froges;FROGES;38190;38;ISERE;1
38176;;Frontonas;FRONTONAS;38290;38;ISERE;1
38177;La;Garde;GARDE;38520;38;ISERE;1
38179;;Gires;GIERES;38610;38;ISERE;1
38180;;Gillonnay;GILLONNAY;38260;38;ISERE;1
38181;;Goncelin;GONCELIN;38570;38;ISERE;1
38182;Le;Grand-Lemps;GRAND-LEMPS;38690;38;ISERE;1
38183;;Granieu;GRANIEU;38490;38;ISERE;1
38184;;Grenay;GRENAY;38540;38;ISERE;1
38185;;Grenoble;GRENOBLE;38000;38;ISERE;1
38186;;Gresse-en-Vercors;GRESSE-EN-VERCORS;38650;38;ISERE;1
38187;Le;Gua;GUA;38450;38;ISERE;1
38188;;Herbeys;HERBEYS;38320;38;ISERE;1
38189;;Heyrieux;HEYRIEUX;38540;38;ISERE;1
38190;;Hires-sur-Amby;HIERES-SUR-AMBY;38118;38;ISERE;1
38191;;Huez;HUEZ;38750;38;ISERE;1
38192;;Hurtires;HURTIERES;38570;38;ISERE;1
38193;L';Isle-d'Abeau;ISLE-D'ABEAU;38080;38;ISERE;1
38194;;Izeaux;IZEAUX;38140;38;ISERE;1
38195;;Izeron;IZERON;38160;38;ISERE;1
38197;;Janneyrias;JANNEYRIAS;38280;38;ISERE;1
38198;;Jarcieu;JARCIEU;38270;38;ISERE;1
38199;;Jardin;JARDIN;38200;38;ISERE;1
38200;;Jarrie;JARRIE;38560;38;ISERE;1
38203;;Laffrey;LAFFREY;38220;38;ISERE;1
38204;;Lalley;LALLEY;38930;38;ISERE;1
38205;;Lans-en-Vercors;LANS-EN-VERCORS;38250;38;ISERE;1
38206;;Laval;LAVAL;38190;38;ISERE;1
38207;;Lavaldens;LAVALDENS;38350;38;ISERE;1
38208;;Lavars;LAVARS;38710;38;ISERE;1
38209;;Lentiol;LENTIOL;38270;38;ISERE;1
38210;;Leyrieu;LEYRIEU;38460;38;ISERE;1
38211;;Lieudieu;LIEUDIEU;38440;38;ISERE;1
38212;;Livet-et-Gavet;LIVET-ET-GAVET;38220;38;ISERE;1
38213;;Longechenal;LONGECHENAL;38690;38;ISERE;1
38214;;Lumbin;LUMBIN;38660;38;ISERE;1
38215;;Luzinay;LUZINAY;38200;38;ISERE;1
38216;;Malleval-en-Vercors;MALLEVAL-EN-VERCORS;38470;38;ISERE;1
38217;;Marcieu;MARCIEU;38350;38;ISERE;1
38218;;Marcilloles;MARCILLOLES;38260;38;ISERE;1
38219;;Marcollin;MARCOLLIN;38270;38;ISERE;1
38221;;Marnans;MARNANS;38980;38;ISERE;1
38222;;Massieu;MASSIEU;38620;38;ISERE;1
38223;;Maubec;MAUBEC;38300;38;ISERE;1
38224;;Mayres-Savel;MAYRES-SAVEL;38350;38;ISERE;1
38225;;Maudre;MEAUDRE;38112;38;ISERE;1
38226;;Mens;MENS;38710;38;ISERE;1
38228;;Merlas;MERLAS;38620;38;ISERE;1
38229;;Meylan;MEYLAN;38240;38;ISERE;1
38230;;Meyri;MEYRIE;38300;38;ISERE;1
38231;;Meyrieu-les-tangs;MEYRIEU-LES-ETANGS;38440;38;ISERE;1
38232;;Meyssis;MEYSSIES;38440;38;ISERE;1
38235;;Miribel-Lanchtre;MIRIBEL-LANCHATRE;38450;38;ISERE;1
38236;;Miribel-les-chelles;MIRIBEL-LES-ECHELLES;38380;38;ISERE;1
38237;;Mizon;MIZOEN;38142;38;ISERE;1
38238;;Moidieu-Dtourbe;MOIDIEU-DETOURBE;38440;38;ISERE;1
38239;;Moirans;MOIRANS;38430;38;ISERE;1
38240;;Moissieu-sur-Dolon;MOISSIEU-SUR-DOLON;38270;38;ISERE;1
38241;;Monestier-d'Ambel;MONESTIER-D'AMBEL;38970;38;ISERE;1
38242;;Monestier-de-Clermont;MONESTIER-DE-CLERMONT;38650;38;ISERE;1
38243;Le;Monestier-du-Percy;MONESTIER-DU-PERCY;38930;38;ISERE;1
38244;;Monsteroux-Milieu;MONSTEROUX-MILIEU;38122;38;ISERE;1
38245;;Montagne;MONTAGNE;38160;38;ISERE;1
38246;;Montagnieu;MONTAGNIEU;38110;38;ISERE;1
38247;;Montalieu-Vercieu;MONTALIEU-VERCIEU;38390;38;ISERE;1
38248;;Montaud;MONTAUD;38210;38;ISERE;1
38249;;Montbonnot-Saint-Martin;MONTBONNOT-SAINT-MARTIN;38330;38;ISERE;1
38250;;Montcarra;MONTCARRA;38890;38;ISERE;1
38252;;Montchaboud;MONTCHABOUD;38220;38;ISERE;1
38253;;Mont-de-Lans;MONT-DE-LANS;38860;38;ISERE;1
38254;;Monteynard;MONTEYNARD;38770;38;ISERE;1
38255;;Montfalcon;MONTFALCON;38940;38;ISERE;1
38256;;Montferrat;MONTFERRAT;38620;38;ISERE;1
38257;;Montrevel;MONTREVEL;38690;38;ISERE;1
38258;;Mont-Saint-Martin;MONT-SAINT-MARTIN;38120;38;ISERE;1
38259;;Montseveroux;MONTSEVEROUX;38122;38;ISERE;1
38260;;Moras;MORAS;38460;38;ISERE;1
38261;;Morestel;MORESTEL;38510;38;ISERE;1
38262;;Mortel-de-Mailles;MORETEL-DE-MAILLES;38570;38;ISERE;1
38263;;Morette;MORETTE;38210;38;ISERE;1
38264;La;Morte;MORTE;38350;38;ISERE;1
38265;La;Motte-d'Aveillans;MOTTE-D'AVEILLANS;38770;38;ISERE;1
38266;La;Motte-Saint-Martin;MOTTE-SAINT-MARTIN;38770;38;ISERE;1
38267;;Mottier;MOTTIER;38260;38;ISERE;1
38268;Le;Moutaret;MOUTARET;38580;38;ISERE;1
38269;La;Mure;MURE;38350;38;ISERE;1
38270;La;Murette;MURETTE;38140;38;ISERE;1
38271;;Murianette;MURIANETTE;38420;38;ISERE;1
38272;;Murinais;MURINAIS;38160;38;ISERE;1
38273;;Nantes-en-Ratier;NANTES-EN-RATIER;38350;38;ISERE;1
38274;;Nantoin;NANTOIN;38260;38;ISERE;1
38275;;Serre-Nerpol;SERRE-NERPOL;38470;38;ISERE;1
38276;;Nivolas-Vermelle;NIVOLAS-VERMELLE;38300;38;ISERE;1
38277;;Notre-Dame-de-Commiers;NOTRE-DAME-DE-COMMIERS;38450;38;ISERE;1
38278;;Notre-Dame-de-l'Osier;NOTRE-DAME-DE-L'OSIER;38470;38;ISERE;1
38279;;Notre-Dame-de-Msage;NOTRE-DAME-DE-MESAGE;38220;38;ISERE;1
38280;;Notre-Dame-de-Vaulx;NOTRE-DAME-DE-VAULX;38144;38;ISERE;1
38281;;Noyarey;NOYAREY;38360;38;ISERE;1
38282;;Optevoz;OPTEVOZ;38460;38;ISERE;1
38283;;Oris-en-Rattier;ORIS-EN-RATTIER;38350;38;ISERE;1
38284;;Ornacieux;ORNACIEUX;38260;38;ISERE;1
38285;;Ornon;ORNON;38520;38;ISERE;1
38286;;Oulles;OULLES;38520;38;ISERE;1
38287;;Oyeu;OYEU;38690;38;ISERE;1
38288;;Oytier-Saint-Oblas;OYTIER-SAINT-OBLAS;38780;38;ISERE;1
38289;;Oz;OZ;38114;38;ISERE;1
38290;;Pact;PACT;38270;38;ISERE;1
38291;;Pajay;PAJAY;38260;38;ISERE;1
38292;;Paladru;PALADRU;38137;38;ISERE;1
38293;;Panissage;PANISSAGE;38730;38;ISERE;1
38294;;Panossas;PANOSSAS;38460;38;ISERE;1
38295;;Parmilieu;PARMILIEU;38390;38;ISERE;1
38296;Le;Passage;PASSAGE;38490;38;ISERE;1
38297;;Passins;PASSINS;38510;38;ISERE;1
38298;Le;Page-de-Roussillon;PEAGE-DE-ROUSSILLON;38550;38;ISERE;1
38299;;Pellafol;PELLAFOL;38970;38;ISERE;1
38300;;Penol;PENOL;38260;38;ISERE;1
38301;;Percy;PERCY;38930;38;ISERE;1
38302;Le;Prier;PERIER;38740;38;ISERE;1
38303;La;Pierre;PIERRE;38570;38;ISERE;1
38304;;Pierre-Chtel;PIERRE-CHATEL;38119;38;ISERE;1
38305;Le;Pin;PIN;38730;38;ISERE;1
38306;;Pinsot;PINSOT;38580;38;ISERE;1
38307;;Pisieu;PISIEU;38270;38;ISERE;1
38308;;Plan;PLAN;38590;38;ISERE;1
38309;;Poisat;POISAT;38320;38;ISERE;1
38310;;Polinas;POLIENAS;38210;38;ISERE;1
38311;;Pommier-de-Beaurepaire;POMMIER-DE-BEAUREPAIRE;38260;38;ISERE;1
38312;;Pommiers-la-Placette;POMMIERS-LA-PLACETTE;38340;38;ISERE;1
38313;;Ponsonnas;PONSONNAS;38350;38;ISERE;1
38314;;Pontcharra;PONTCHARRA;38530;38;ISERE;1
38315;Le;Pont-de-Beauvoisin;PONT-DE-BEAUVOISIN;38480;38;ISERE;1
38316;;Pont-de-Chruy;PONT-DE-CHERUY;38230;38;ISERE;1
38317;Le;Pont-de-Claix;PONT-DE-CLAIX;38800;38;ISERE;1
38318;;Pont-vque;PONT-EVEQUE;38780;38;ISERE;1
38319;;Pont-en-Royans;PONT-EN-ROYANS;38680;38;ISERE;1
38320;;Porcieu-Amblagnieu;PORCIEU-AMBLAGNIEU;38390;38;ISERE;1
38321;;Prbois;PREBOIS;38710;38;ISERE;1
38322;;Presles;PRESLES;38680;38;ISERE;1
38323;;Pressins;PRESSINS;38480;38;ISERE;1
38324;;Primarette;PRIMARETTE;38270;38;ISERE;1
38325;;Proveysieux;PROVEYSIEUX;38120;38;ISERE;1
38326;;Prunires;PRUNIERES;38350;38;ISERE;1
38328;;Quaix-en-Chartreuse;QUAIX-EN-CHARTREUSE;38950;38;ISERE;1
38329;;Quet-en-Beaumont;QUET-EN-BEAUMONT;38970;38;ISERE;1
38330;;Quincieu;QUINCIEU;38470;38;ISERE;1
38331;;Raumont;REAUMONT;38140;38;ISERE;1
38332;;Renage;RENAGE;38140;38;ISERE;1
38333;;Rencurel;RENCUREL;38680;38;ISERE;1
38334;;Revel;REVEL;38420;38;ISERE;1
38335;;Revel-Tourdan;REVEL-TOURDAN;38270;38;ISERE;1
38336;;Reventin-Vaugris;REVENTIN-VAUGRIS;38121;38;ISERE;1
38337;;Rives;RIVES;38140;38;ISERE;1
38338;La;Rivire;RIVIERE;38210;38;ISERE;1
38339;;Roche;ROCHE;38090;38;ISERE;1
38340;Les;Roches-de-Condrieu;ROCHES-DE-CONDRIEU;38370;38;ISERE;1
38341;;Rochetoirin;ROCHETOIRIN;38110;38;ISERE;1
38342;;Roissard;ROISSARD;38650;38;ISERE;1
38343;;Romagnieu;ROMAGNIEU;38480;38;ISERE;1
38344;;Roussillon;ROUSSILLON;38150;38;ISERE;1
38345;;Rovon;ROVON;38470;38;ISERE;1
38346;;Royas;ROYAS;38440;38;ISERE;1
38347;;Roybon;ROYBON;38940;38;ISERE;1
38348;;Ruy;RUY;38300;38;ISERE;1
38349;;Sablons;SABLONS;38550;38;ISERE;1
38350;;Sainte-Agns;SAINTE-AGNES;38190;38;ISERE;1
38351;;Saint-Agnin-sur-Bion;SAINT-AGNIN-SUR-BION;38300;38;ISERE;1
38352;;Saint-Alban-de-Roche;SAINT-ALBAN-DE-ROCHE;38300;38;ISERE;1
38353;;Saint-Alban-du-Rhne;SAINT-ALBAN-DU-RHONE;38370;38;ISERE;1
38354;;Saint-Albin-de-Vaulserre;SAINT-ALBIN-DE-VAULSERRE;38480;38;ISERE;1
38355;;Saint-Andol;SAINT-ANDEOL;38650;38;ISERE;1
38356;;Saint-Andr-en-Royans;SAINT-ANDRE-EN-ROYANS;38680;38;ISERE;1
38357;;Saint-Andr-le-Gaz;SAINT-ANDRE-LE-GAZ;38490;38;ISERE;1
38358;;Sainte-Anne-sur-Gervonde;SAINTE-ANNE-SUR-GERVONDE;38440;38;ISERE;1
38359;;Saint-Antoine-l'Abbaye;SAINT-ANTOINE-L'ABBAYE;38160;38;ISERE;1
38360;;Saint-Appolinard;SAINT-APPOLINARD;38160;38;ISERE;1
38361;;Saint-Arey;SAINT-AREY;38350;38;ISERE;1
38362;;Saint-Aupre;SAINT-AUPRE;38960;38;ISERE;1
38363;;Saint-Barthlemy;SAINT-BARTHELEMY;38270;38;ISERE;1
38364;;Saint-Barthlemy-de-Schilienne;SAINT-BARTHELEMY-DE-SECHILIENNE;38220;38;ISERE;1
38365;;Saint-Baudille-de-la-Tour;SAINT-BAUDILLE-DE-LA-TOUR;38118;38;ISERE;1
38366;;Saint-Baudille-et-Pipet;SAINT-BAUDILLE-ET-PIPET;38710;38;ISERE;1
38367;;Saint-Bernard;SAINT-BERNARD;38660;38;ISERE;1
38368;;Saint-Blaise-du-Buis;SAINT-BLAISE-DU-BUIS;38140;38;ISERE;1
38369;;Sainte-Blandine;SAINTE-BLANDINE;38110;38;ISERE;1
38370;;Saint-Bonnet-de-Chavagne;SAINT-BONNET-DE-CHAVAGNE;38840;38;ISERE;1
38372;;Saint-Bueil;SAINT-BUEIL;38620;38;ISERE;1
38373;;Saint-Cassien;SAINT-CASSIEN;38500;38;ISERE;1
38374;;Saint-Chef;SAINT-CHEF;38890;38;ISERE;1
38375;;Saint-Christophe-en-Oisans;SAINT-CHRISTOPHE-EN-OISANS;38520;38;ISERE;1
38376;;Saint-Christophe-sur-Guiers;SAINT-CHRISTOPHE-SUR-GUIERS;38380;38;ISERE;1
38377;;Saint-Clair-de-la-Tour;SAINT-CLAIR-DE-LA-TOUR;38110;38;ISERE;1
38378;;Saint-Clair-du-Rhne;SAINT-CLAIR-DU-RHONE;38370;38;ISERE;1
38379;;Saint-Clair-sur-Galaure;SAINT-CLAIR-SUR-GALAURE;38940;38;ISERE;1
38380;;Saint-Didier-de-Bizonnes;SAINT-DIDIER-DE-BIZONNES;38690;38;ISERE;1
38381;;Saint-Didier-de-la-Tour;SAINT-DIDIER-DE-LA-TOUR;38110;38;ISERE;1
38382;;Saint-grve;SAINT-EGREVE;38120;38;ISERE;1
38383;;Saint-tienne-de-Crossey;SAINT-ETIENNE-DE-CROSSEY;38960;38;ISERE;1
38384;;Saint-tienne-de-Saint-Geoirs;SAINT-ETIENNE-DE-SAINT-GEOIRS;38590;38;ISERE;1
38386;;Saint-Geoire-en-Valdaine;SAINT-GEOIRE-EN-VALDAINE;38620;38;ISERE;1
38387;;Saint-Geoirs;SAINT-GEOIRS;38590;38;ISERE;1
38388;;Saint-Georges-de-Commiers;SAINT-GEORGES-DE-COMMIERS;38450;38;ISERE;1
38389;;Saint-Georges-d'Espranche;SAINT-GEORGES-D'ESPERANCHE;38790;38;ISERE;1
38390;;Saint-Gervais;SAINT-GERVAIS;38470;38;ISERE;1
38391;;Saint-Guillaume;SAINT-GUILLAUME;38650;38;ISERE;1
38392;;Saint-Hilaire-de-Brens;SAINT-HILAIRE-DE-BRENS;38460;38;ISERE;1
38393;;Saint-Hilaire-de-la-Cte;SAINT-HILAIRE-DE-LA-COTE;38260;38;ISERE;1
38394;;Saint-Hilaire-du-Rosier;SAINT-HILAIRE-DU-ROSIER;38840;38;ISERE;1
38395;;Saint-Hilaire;SAINT-HILAIRE;38660;38;ISERE;1
38396;;Saint-Honor;SAINT-HONORE;38350;38;ISERE;1
38397;;Saint-Ismier;SAINT-ISMIER;38330;38;ISERE;1
38398;;Saint-Jean-d'Avelanne;SAINT-JEAN-D'AVELANNE;38480;38;ISERE;1
38399;;Saint-Jean-de-Bournay;SAINT-JEAN-DE-BOURNAY;38440;38;ISERE;1
38400;;Saint-Jean-de-Moirans;SAINT-JEAN-DE-MOIRANS;38430;38;ISERE;1
38401;;Saint-Jean-de-Soudain;SAINT-JEAN-DE-SOUDAIN;38110;38;ISERE;1
38402;;Saint-Jean-de-Vaulx;SAINT-JEAN-DE-VAULX;38220;38;ISERE;1
38403;;Saint-Jean-d'Hrans;SAINT-JEAN-D'HERANS;38710;38;ISERE;1
38404;;Saint-Jean-le-Vieux;SAINT-JEAN-LE-VIEUX;38420;38;ISERE;1
38405;;Saint-Joseph-de-Rivire;SAINT-JOSEPH-DE-RIVIERE;38134;38;ISERE;1
38406;;Saint-Julien-de-l'Herms;SAINT-JULIEN-DE-L'HERMS;38122;38;ISERE;1
38407;;Saint-Julien-de-Raz;SAINT-JULIEN-DE-RAZ;38134;38;ISERE;1
38408;;Saint-Just-Chaleyssin;SAINT-JUST-CHALEYSSIN;38540;38;ISERE;1
38409;;Saint-Just-de-Claix;SAINT-JUST-DE-CLAIX;38680;38;ISERE;1
38410;;Saint-Lattier;SAINT-LATTIER;38840;38;ISERE;1
38412;;Saint-Laurent-du-Pont;SAINT-LAURENT-DU-PONT;38380;38;ISERE;1
38413;;Saint-Laurent-en-Beaumont;SAINT-LAURENT-EN-BEAUMONT;38350;38;ISERE;1
38414;;Sainte-Luce;SAINTE-LUCE;38970;38;ISERE;1
38415;;Saint-Marcel-Bel-Accueil;SAINT-MARCEL-BEL-ACCUEIL;38080;38;ISERE;1
38416;;Saint-Marcellin;SAINT-MARCELLIN;38160;38;ISERE;1
38417;;Sainte-Marie-d'Alloix;SAINTE-MARIE-D'ALLOIX;38660;38;ISERE;1
38418;;Sainte-Marie-du-Mont;SAINTE-MARIE-DU-MONT;38660;38;ISERE;1
38419;;Saint-Martin-de-Clelles;SAINT-MARTIN-DE-CLELLES;38930;38;ISERE;1
38420;;Saint-Martin-de-Vaulserre;SAINT-MARTIN-DE-VAULSERRE;38480;38;ISERE;1
38421;;Saint-Martin-d'Hres;SAINT-MARTIN-D'HERES;38400;38;ISERE;1
38422;;Saint-Martin-d'Uriage;SAINT-MARTIN-D'URIAGE;38410;38;ISERE;1
38423;;Saint-Martin-le-Vinoux;SAINT-MARTIN-LE-VINOUX;38950;38;ISERE;1
38424;;Saint-Maurice-en-Trives;SAINT-MAURICE-EN-TRIEVES;38930;38;ISERE;1
38425;;Saint-Maurice-l'Exil;SAINT-MAURICE-L'EXIL;38550;38;ISERE;1
38426;;Saint-Maximin;SAINT-MAXIMIN;38530;38;ISERE;1
38427;;Saint-Michel-de-Saint-Geoirs;SAINT-MICHEL-DE-SAINT-GEOIRS;38590;38;ISERE;1
38428;;Saint-Michel-en-Beaumont;SAINT-MICHEL-EN-BEAUMONT;38350;38;ISERE;1
38429;;Saint-Michel-les-Portes;SAINT-MICHEL-LES-PORTES;38650;38;ISERE;1
38430;;Saint-Mury-Monteymond;SAINT-MURY-MONTEYMOND;38190;38;ISERE;1
38431;;Saint-Nazaire-les-Eymes;SAINT-NAZAIRE-LES-EYMES;38330;38;ISERE;1
38432;;Saint-Nicolas-de-Macherin;SAINT-NICOLAS-DE-MACHERIN;38500;38;ISERE;1
38433;;Saint-Nizier-du-Moucherotte;SAINT-NIZIER-DU-MOUCHEROTTE;38250;38;ISERE;1
38434;;Saint-Ondras;SAINT-ONDRAS;38490;38;ISERE;1
38435;;Saint-Pancrasse;SAINT-PANCRASSE;38660;38;ISERE;1
38436;;Saint-Paul-de-Varces;SAINT-PAUL-DE-VARCES;38760;38;ISERE;1
38437;;Saint-Paul-d'Izeaux;SAINT-PAUL-D'IZEAUX;38140;38;ISERE;1
38438;;Saint-Paul-ls-Monestier;SAINT-PAUL-LES-MONESTIER;38650;38;ISERE;1
38439;;Saint-Pierre-d'Allevard;SAINT-PIERRE-D'ALLEVARD;38830;38;ISERE;1
38440;;Saint-Pierre-de-Bressieux;SAINT-PIERRE-DE-BRESSIEUX;38870;38;ISERE;1
38442;;Saint-Pierre-de-Chartreuse;SAINT-PIERRE-DE-CHARTREUSE;38380;38;ISERE;1
38443;;Saint-Pierre-de-Chrennes;SAINT-PIERRE-DE-CHERENNES;38160;38;ISERE;1
38444;;Saint-Pierre-de-Maroz;SAINT-PIERRE-DE-MEAROZ;38350;38;ISERE;1
38445;;Saint-Pierre-de-Msage;SAINT-PIERRE-DE-MESAGE;38220;38;ISERE;1
38446;;Saint-Pierre-d'Entremont;SAINT-PIERRE-D'ENTREMONT;73670;38;SAVOIE;1
38448;;Saint-Prim;SAINT-PRIM;38370;38;ISERE;1
38449;;Saint-Quentin-Fallavier;SAINT-QUENTIN-FALLAVIER;38070;38;ISERE;1
38450;;Saint-Quentin-sur-Isre;SAINT-QUENTIN-SUR-ISERE;38210;38;ISERE;1
38451;;Saint-Romain-de-Jalionas;SAINT-ROMAIN-DE-JALIONAS;38460;38;ISERE;1
38452;;Saint-Romain-de-Surieu;SAINT-ROMAIN-DE-SURIEU;38150;38;ISERE;1
38453;;Saint-Romans;SAINT-ROMANS;38160;38;ISERE;1
38454;;Saint-Sauveur;SAINT-SAUVEUR;38160;38;ISERE;1
38455;;Saint-Savin;SAINT-SAVIN;38300;38;ISERE;1
38456;;Saint-Sbastien;SAINT-SEBASTIEN;38710;38;ISERE;1
38457;;Saint-Simon-de-Bressieux;SAINT-SIMEON-DE-BRESSIEUX;38870;38;ISERE;1
38458;;Saint-Sorlin-de-Morestel;SAINT-SORLIN-DE-MORESTEL;38510;38;ISERE;1
38459;;Saint-Sorlin-de-Vienne;SAINT-SORLIN-DE-VIENNE;38200;38;ISERE;1
38460;;Saint-Sulpice-des-Rivoires;SAINT-SULPICE-DES-RIVOIRES;38620;38;ISERE;1
38462;;Saint-Thoffrey;SAINT-THEOFFREY;38119;38;ISERE;1
38463;;Saint-Vrand;SAINT-VERAND;38160;38;ISERE;1
38464;;Saint-Victor-de-Cessieu;SAINT-VICTOR-DE-CESSIEU;38110;38;ISERE;1
38465;;Saint-Victor-de-Morestel;SAINT-VICTOR-DE-MORESTEL;38510;38;ISERE;1
38466;;Saint-Vincent-de-Mercuze;SAINT-VINCENT-DE-MERCUZE;38660;38;ISERE;1
38467;;Salagnon;SALAGNON;38890;38;ISERE;1
38468;;Salaise-sur-Sanne;SALAISE-SUR-SANNE;38150;38;ISERE;1
38469;La;Salette-Fallavaux;SALETTE-FALLAVAUX;38970;38;ISERE;1
38470;La;Salle-en-Beaumont;SALLE-EN-BEAUMONT;38350;38;ISERE;1
38471;Le;Sappey-en-Chartreuse;SAPPEY-EN-CHARTREUSE;38700;38;ISERE;1
38472;;Sarcenas;SARCENAS;38700;38;ISERE;1
38473;;Sardieu;SARDIEU;38260;38;ISERE;1
38474;;Sassenage;SASSENAGE;38360;38;ISERE;1
38475;;Satolas-et-Bonce;SATOLAS-ET-BONCE;38290;38;ISERE;1
38476;;Savas-Mpin;SAVAS-MEPIN;38440;38;ISERE;1
38478;;Schilienne;SECHILIENNE;38220;38;ISERE;1
38479;;Semons;SEMONS;38260;38;ISERE;1
38480;;Septme;SEPTEME;38780;38;ISERE;1
38481;;Srzin-de-la-Tour;SEREZIN-DE-LA-TOUR;38300;38;ISERE;1
38483;;Sermrieu;SERMERIEU;38510;38;ISERE;1
38484;;Serpaize;SERPAIZE;38200;38;ISERE;1
38485;;Seyssinet-Pariset;SEYSSINET-PARISET;38170;38;ISERE;1
38486;;Seyssins;SEYSSINS;38180;38;ISERE;1
38487;;Seyssuel;SEYSSUEL;38200;38;ISERE;1
38488;;Siccieu-Saint-Julien-et-Carisieu;SICCIEU-SAINT-JULIEN-ET-CARISIEU;38460;38;ISERE;1
38489;;Sivoz;SIEVOZ;38350;38;ISERE;1
38490;;Sillans;SILLANS;38590;38;ISERE;1
38492;;Sinard;SINARD;38650;38;ISERE;1
38494;;Soleymieu;SOLEYMIEU;38460;38;ISERE;1
38495;La;Sne;SONE;38840;38;ISERE;1
38496;;Sonnay;SONNAY;38150;38;ISERE;1
38497;;Sousville;SOUSVILLE;38350;38;ISERE;1
38498;;Succieu;SUCCIEU;38300;38;ISERE;1
38499;;Susville;SUSVILLE;38350;38;ISERE;1
38500;;Tche;TECHE;38470;38;ISERE;1
38501;;Tencin;TENCIN;38570;38;ISERE;1
38503;La;Terrasse;TERRASSE;38660;38;ISERE;1
38504;;Theys;THEYS;38570;38;ISERE;1
38505;;Thodure;THODURE;38260;38;ISERE;1
38507;;Tignieu-Jameyzieu;TIGNIEU-JAMEYZIEU;38230;38;ISERE;1
38508;;Torchefelon;TORCHEFELON;38690;38;ISERE;1
38509;La;Tour-du-Pin;TOUR-DU-PIN;38110;38;ISERE;1
38511;Le;Touvet;TOUVET;38660;38;ISERE;1
38512;;Tramol;TRAMOLE;38300;38;ISERE;1
38513;;Treffort;TREFFORT;38650;38;ISERE;1
38514;;Trminis;TREMINIS;38710;38;ISERE;1
38515;;Trept;TREPT;38460;38;ISERE;1
38516;La;Tronche;TRONCHE;38700;38;ISERE;1
38517;;Tullins;TULLINS;38210;38;ISERE;1
38518;;Valbonnais;VALBONNAIS;38740;38;ISERE;1
38519;;Valencin;VALENCIN;38540;38;ISERE;1
38520;;Valencogne;VALENCOGNE;38730;38;ISERE;1
38521;La;Valette;VALETTE;38350;38;ISERE;1
38522;;Valjouffrey;VALJOUFFREY;38740;38;ISERE;1
38523;;Varacieux;VARACIEUX;38470;38;ISERE;1
38524;;Varces-Allires-et-Risset;VARCES-ALLIERES-ET-RISSET;38760;38;ISERE;1
38525;;Vasselin;VASSELIN;38890;38;ISERE;1
38526;;Vatilieu;VATILIEU;38470;38;ISERE;1
38527;;Vaujany;VAUJANY;38114;38;ISERE;1
38528;;Vaulnaveys-le-Bas;VAULNAVEYS-LE-BAS;38410;38;ISERE;1
38529;;Vaulnaveys-le-Haut;VAULNAVEYS-LE-HAUT;38410;38;ISERE;1
38530;;Vaulx-Milieu;VAULX-MILIEU;38090;38;ISERE;1
38531;;Velanne;VELANNE;38620;38;ISERE;1
38532;;Vnrieu;VENERIEU;38460;38;ISERE;1
38533;;Venon;VENON;38610;38;ISERE;1
38534;;Vnosc;VENOSC;38520;38;ISERE;1
38535;;Vernas;VERNAS;38460;38;ISERE;1
38536;;Vernioz;VERNIOZ;38150;38;ISERE;1
38537;La;Verpillire;VERPILLIERE;38290;38;ISERE;1
38538;Le;Versoud;VERSOUD;38420;38;ISERE;1
38539;;Vertrieu;VERTRIEU;38390;38;ISERE;1
38540;;Veurey-Voroize;VEUREY-VOROIZE;38113;38;ISERE;1
38541;;Veyrins-Thuellin;VEYRINS-THUELLIN;38630;38;ISERE;1
38542;;Veyssilieu;VEYSSILIEU;38460;38;ISERE;1
38543;;Vzeronce-Curtin;VEZERONCE-CURTIN;38510;38;ISERE;1
38544;;Vienne;VIENNE;38200;38;ISERE;1
38545;;Vif;VIF;38450;38;ISERE;1
38546;;Vignieu;VIGNIEU;38890;38;ISERE;1
38547;;Villard-Bonnot;VILLARD-BONNOT;38190;38;ISERE;1
38548;;Villard-de-Lans;VILLARD-DE-LANS;38250;38;ISERE;1
38549;;Villard-Notre-Dame;VILLARD-NOTRE-DAME;38520;38;ISERE;1
38550;;Villard-Reculas;VILLARD-RECULAS;38114;38;ISERE;1
38551;;Villard-Reymond;VILLARD-REYMOND;38520;38;ISERE;1
38552;;Villard-Saint-Christophe;VILLARD-SAINT-CHRISTOPHE;38119;38;ISERE;1
38553;;Villefontaine;VILLEFONTAINE;38090;38;ISERE;1
38554;;Villemoirieu;VILLEMOIRIEU;38460;38;ISERE;1
38555;;Villeneuve-de-Marc;VILLENEUVE-DE-MARC;38440;38;ISERE;1
38556;;Ville-sous-Anjou;VILLE-SOUS-ANJOU;38150;38;ISERE;1
38557;;Villette-d'Anthon;VILLETTE-D'ANTHON;38280;38;ISERE;1
38558;;Villette-de-Vienne;VILLETTE-DE-VIENNE;38200;38;ISERE;1
38559;;Vinay;VINAY;38470;38;ISERE;1
38560;;Virieu;VIRIEU;38730;38;ISERE;1
38561;;Viriville;VIRIVILLE;38980;38;ISERE;1
38562;;Vizille;VIZILLE;38220;38;ISERE;1
38563;;Voiron;VOIRON;38500;38;ISERE;1
38564;;Voissant;VOISSANT;38620;38;ISERE;1
38565;;Voreppe;VOREPPE;38340;38;ISERE;1
38566;;Vourey;VOUREY;38210;38;ISERE;1
38567;;Chamrousse;CHAMROUSSE;38410;38;ISERE;1
39001;;Abergement-la-Ronce;ABERGEMENT-LA-RONCE;39500;39;JURA;1
39002;;Abergement-le-Grand;ABERGEMENT-LE-GRAND;39600;39;JURA;1
39003;;Abergement-le-Petit;ABERGEMENT-LE-PETIT;39800;39;JURA;1
39004;;Abergement-ls-Thsy;ABERGEMENT-LES-THESY;39110;39;JURA;1
39006;;Aiglepierre;AIGLEPIERRE;39110;39;JURA;1
39007;;Alize;ALIEZE;39270;39;JURA;1
39008;;Amange;AMANGE;39700;39;JURA;1
39009;;Andelot-en-Montagne;ANDELOT-EN-MONTAGNE;39110;39;JURA;1
39010;;Andelot-Morval;ANDELOT-MORVAL;39320;39;JURA;1
39011;;Annoire;ANNOIRE;39120;39;JURA;1
39013;;Arbois;ARBOIS;39600;39;JURA;1
39014;;Archelange;ARCHELANGE;39290;39;JURA;1
39015;;Ardon;ARDON;39300;39;JURA;1
39016;;Arinthod;ARINTHOD;39240;39;JURA;1
39017;;Arlay;ARLAY;39140;39;JURA;1
39018;;Aromas;AROMAS;39240;39;JURA;1
39019;Les;Arsures;ARSURES;39600;39;JURA;1
39020;;Arsure-Arsurette;ARSURE-ARSURETTE;39250;39;JURA;1
39021;;Arthenas;ARTHENAS;39270;39;JURA;1
39022;;Asnans-Beauvoisin;ASNANS-BEAUVOISIN;39120;39;JURA;1
39023;L';Aubpin;AUBEPIN;39160;39;JURA;1
39024;;Audelange;AUDELANGE;39700;39;JURA;1
39025;;Augea;AUGEA;39190;39;JURA;1
39026;;Augerans;AUGERANS;39380;39;JURA;1
39027;;Augisey;AUGISEY;39270;39;JURA;1
39028;;Aumont;AUMONT;39800;39;JURA;1
39029;;Aumur;AUMUR;39410;39;JURA;1
39030;;Authume;AUTHUME;39100;39;JURA;1
39031;;Auxange;AUXANGE;39700;39;JURA;1
39032;;Avignon-ls-Saint-Claude;AVIGNON-LES-SAINT-CLAUDE;39200;39;JURA;1
39034;;Balaiseaux;BALAISEAUX;39120;39;JURA;1
39035;;Balanod;BALANOD;39160;39;JURA;1
39036;La;Balme-d'py;BALME-D'EPY;39320;39;JURA;1
39037;;Bans;BANS;39380;39;JURA;1
39038;;Barsia-sur-l'Ain;BARESIA-SUR-L'AIN;39130;39;JURA;1
39039;La;Barre;BARRE;39700;39;JURA;1
39040;;Barretaine;BARRETAINE;39800;39;JURA;1
39041;;Baume-les-Messieurs;BAUME-LES-MESSIEURS;39210;39;JURA;1
39042;;Baverans;BAVERANS;39100;39;JURA;1
39043;;Beaufort;BEAUFORT;39190;39;JURA;1
39045;;Beffia;BEFFIA;39270;39;JURA;1
39046;;Bellecombe;BELLECOMBE;39310;39;JURA;1
39047;;Bellefontaine;BELLEFONTAINE;39400;39;JURA;1
39048;;Belmont;BELMONT;39380;39;JURA;1
39049;;Bersaillin;BERSAILLIN;39800;39;JURA;1
39050;;Besain;BESAIN;39800;39;JURA;1
39051;;Biarne;BIARNE;39290;39;JURA;1
39052;;Bief-des-Maisons;BIEF-DES-MAISONS;39150;39;JURA;1
39053;;Bief-du-Fourg;BIEF-DU-FOURG;39250;39;JURA;1
39054;;Biefmorin;BIEFMORIN;39800;39;JURA;1
39055;;Billecul;BILLECUL;39250;39;JURA;1
39056;;Bletterans;BLETTERANS;39140;39;JURA;1
39057;;Blois-sur-Seille;BLOIS-SUR-SEILLE;39210;39;JURA;1
39058;;Blye;BLYE;39130;39;JURA;1
39059;;Bois-d'Amont;BOIS-D'AMONT;39220;39;JURA;1
39060;;Bois-de-Gand;BOIS-DE-GAND;39230;39;JURA;1
39061;;Boissia;BOISSIA;39130;39;JURA;1
39062;La;Boissire;BOISSIERE;39240;39;JURA;1
39063;;Bonlieu;BONLIEU;39130;39;JURA;1
39064;;Bonnaud;BONNAUD;39190;39;JURA;1
39065;;Bonnefontaine;BONNEFONTAINE;39800;39;JURA;1
39066;;Bornay;BORNAY;39570;39;JURA;1
39068;Les;Bouchoux;BOUCHOUX;39370;39;JURA;1
39069;;Bourcia;BOURCIA;39320;39;JURA;1
39070;;Bourg-de-Sirod;BOURG-DE-SIROD;39300;39;JURA;1
39072;;Bracon;BRACON;39110;39;JURA;1
39073;;Brainans;BRAINANS;39800;39;JURA;1
39074;;Brans;BRANS;39290;39;JURA;1
39075;;Brry;BRERY;39230;39;JURA;1
39076;La;Bretenire;BRETENIERE;39700;39;JURA;1
39077;;Bretenires;BRETENIERES;39120;39;JURA;1
39078;;Brevans;BREVANS;39100;39;JURA;1
39079;;Briod;BRIOD;39570;39;JURA;1
39080;;Broissia;BROISSIA;39320;39;JURA;1
39081;;Buvilly;BUVILLY;39800;39;JURA;1
39083;;Censeau;CENSEAU;39250;39;JURA;1
39084;;Cernans;CERNANS;39110;39;JURA;1
39085;;Cernibaud;CERNIEBAUD;39250;39;JURA;1
39086;;Cernon;CERNON;39240;39;JURA;1
39088;;Cesancey;CESANCEY;39570;39;JURA;1
39089;;Czia;CEZIA;39240;39;JURA;1
39090;;Chane-des-Coupis;CHAINEE-DES-COUPIS;39120;39;JURA;1
39091;Les;Chalesmes;CHALESMES;39150;39;JURA;1
39092;;Chambria;CHAMBERIA;39270;39;JURA;1
39093;;Chamblay;CHAMBLAY;39380;39;JURA;1
39094;;Chamole;CHAMOLE;39800;39;JURA;1
39095;;Champagne-sur-Loue;CHAMPAGNE-SUR-LOUE;39600;39;JURA;1
39096;;Champagney;CHAMPAGNEY;39290;39;JURA;1
39097;;Champagnole;CHAMPAGNOLE;39300;39;JURA;1
39099;;Champdivers;CHAMPDIVERS;39500;39;JURA;1
39100;;Champrougier;CHAMPROUGIER;39230;39;JURA;1
39101;;Champvans;CHAMPVANS;39100;39;JURA;1
39102;;Chancia;CHANCIA;01590;39;AIN;1
39103;La;Chapelle-sur-Furieuse;CHAPELLE-SUR-FURIEUSE;39110;39;JURA;1
39104;;Chapelle-Voland;CHAPELLE-VOLAND;39140;39;JURA;1
39105;;Chapois;CHAPOIS;39300;39;JURA;1
39106;;Charchilla;CHARCHILLA;39260;39;JURA;1
39107;;Charcier;CHARCIER;39130;39;JURA;1
39108;;Charency;CHARENCY;39250;39;JURA;1
39109;;Charzier;CHAREZIER;39130;39;JURA;1
39110;La;Charme;CHARME;39230;39;JURA;1
39111;;Charnod;CHARNOD;39240;39;JURA;1
39112;La;Chassagne;CHASSAGNE;39230;39;JURA;1
39113;;Chassal;CHASSAL;39360;39;JURA;1
39114;;Chteau-Chalon;CHATEAU-CHALON;39210;39;JURA;1
39115;;Chteau-des-Prs;CHATEAU-DES-PRES;39150;39;JURA;1
39116;La;Chtelaine;CHATELAINE;39600;39;JURA;1
39117;;Chatelay;CHATELAY;39380;39;JURA;1
39118;;Chtel-de-Joux;CHATEL-DE-JOUX;39130;39;JURA;1
39119;Le;Chateley;CHATELEY;39230;39;JURA;1
39120;;Chtelneuf;CHATELNEUF;39300;39;JURA;1
39121;;Chtenois;CHATENOIS;39700;39;JURA;1
39122;;Chtillon;CHATILLON;39130;39;JURA;1
39123;;Chatonnay;CHATONNAY;39240;39;JURA;1
39124;;Chaumergy;CHAUMERGY;39230;39;JURA;1
39126;La;Chaumusse;CHAUMUSSE;39150;39;JURA;1
39127;;Chaussenans;CHAUSSENANS;39800;39;JURA;1
39128;;Chaussin;CHAUSSIN;39120;39;JURA;1
39129;;Chaux-des-Crotenay;CHAUX-DES-CROTENAY;39150;39;JURA;1
39130;;Chaux-des-Prs;CHAUX-DES-PRES;39150;39;JURA;1
39131;La;Chaux-du-Dombief;CHAUX-DU-DOMBIEF;39150;39;JURA;1
39132;La;Chaux-en-Bresse;CHAUX-EN-BRESSE;39230;39;JURA;1
39133;;Chaux-Champagny;CHAUX-CHAMPAGNY;39110;39;JURA;1
39134;;Chavria;CHAVERIA;39270;39;JURA;1
39135;;Chazelles;CHAZELLES;39160;39;JURA;1
39136;;Chemenot;CHEMENOT;39230;39;JURA;1
39137;;Chemilla;CHEMILLA;39240;39;JURA;1
39138;;Chemin;CHEMIN;39120;39;JURA;1
39139;;Chne-Bernard;CHENE-BERNARD;39120;39;JURA;1
39140;;Chne-Sec;CHENE-SEC;39230;39;JURA;1
39141;;Chevigny;CHEVIGNY;39290;39;JURA;1
39142;;Chevreaux;CHEVREAUX;39190;39;JURA;1
39143;;Chevrotaine;CHEVROTAINE;39130;39;JURA;1
39145;;Chille;CHILLE;39570;39;JURA;1
39146;;Chilly-le-Vignoble;CHILLY-LE-VIGNOBLE;39570;39;JURA;1
39147;;Chilly-sur-Salins;CHILLY-SUR-SALINS;39110;39;JURA;1
39148;;Chissria;CHISSERIA;39240;39;JURA;1
39149;;Chissey-sur-Loue;CHISSEY-SUR-LOUE;39380;39;JURA;1
39150;;Choisey;CHOISEY;39100;39;JURA;1
39151;;Choux;CHOUX;39370;39;JURA;1
39153;;Cize;CIZE;39300;39;JURA;1
39154;;Clairvaux-les-Lacs;CLAIRVAUX-LES-LACS;39130;39;JURA;1
39155;;Clucy;CLUCY;39110;39;JURA;1
39156;;Cogna;COGNA;39130;39;JURA;1
39157;;Coiserette;COISERETTE;39200;39;JURA;1
39158;;Coisia;COISIA;39240;39;JURA;1
39159;;Colonne;COLONNE;39800;39;JURA;1
39160;;Commenailles;COMMENAILLES;39140;39;JURA;1
39161;;Communailles-en-Montagne;COMMUNAILLES-EN-MONTAGNE;39250;39;JURA;1
39162;;Condamine;CONDAMINE;39570;39;JURA;1
39163;;Condes;CONDES;39240;39;JURA;1
39164;;Conlige;CONLIEGE;39570;39;JURA;1
39165;;Conte;CONTE;39300;39;JURA;1
39166;;Cornod;CORNOD;39240;39;JURA;1
39167;;Cosges;COSGES;39140;39;JURA;1
39168;;Courbette;COURBETTE;39570;39;JURA;1
39169;;Courbouzon;COURBOUZON;39570;39;JURA;1
39170;;Courlans;COURLANS;39570;39;JURA;1
39171;;Courlaoux;COURLAOUX;39570;39;JURA;1
39172;;Courtefontaine;COURTEFONTAINE;39700;39;JURA;1
39173;;Cousance;COUSANCE;39190;39;JURA;1
39174;;Coyrire;COYRIERE;39200;39;JURA;1
39175;;Coyron;COYRON;39260;39;JURA;1
39176;;Cramans;CRAMANS;39600;39;JURA;1
39177;;Cranot;CRANCOT;39570;39;JURA;1
39178;;Crans;CRANS;39300;39;JURA;1
39179;;Crenans;CRENANS;39260;39;JURA;1
39180;;Cressia;CRESSIA;39270;39;JURA;1
39182;;Crissey;CRISSEY;39100;39;JURA;1
39183;;Crotenay;CROTENAY;39300;39;JURA;1
39184;Les;Crozets;CROZETS;39260;39;JURA;1
39185;;Cuisia;CUISIA;39190;39;JURA;1
39186;;Cuttura;CUTTURA;39170;39;JURA;1
39187;;Cuvier;CUVIER;39250;39;JURA;1
39188;;Dammartin-Marpain;DAMMARTIN-MARPAIN;39290;39;JURA;1
39189;;Damparis;DAMPARIS;39500;39;JURA;1
39190;;Dampierre;DAMPIERRE;39700;39;JURA;1
39191;;Darbonnay;DARBONNAY;39230;39;JURA;1
39192;;Denezires;DENEZIERES;39130;39;JURA;1
39193;Le;Deschaux;DESCHAUX;39120;39;JURA;1
39194;;Desnes;DESNES;39140;39;JURA;1
39195;;Dessia;DESSIA;39320;39;JURA;1
39196;Les;Deux-Fays;DEUX-FAYS;39230;39;JURA;1
39197;;Digna;DIGNA;39190;39;JURA;1
39198;;Dole;DOLE;39100;39;JURA;1
39199;;Domblans;DOMBLANS;39210;39;JURA;1
39200;;Dompierre-sur-Mont;DOMPIERRE-SUR-MONT;39270;39;JURA;1
39201;;Doucier;DOUCIER;39130;39;JURA;1
39202;;Dournon;DOURNON;39110;39;JURA;1
39203;;Doye;DOYE;39250;39;JURA;1
39204;;Dramelay;DRAMELAY;39240;39;JURA;1
39205;;clans-Nenon;ECLANS-NENON;39700;39;JURA;1
39206;;cleux;ECLEUX;39600;39;JURA;1
39207;;crille;ECRILLE;39270;39;JURA;1
39208;;Entre-deux-Monts;ENTRE-DEUX-MONTS;39150;39;JURA;1
39209;;Val-d'py;VAL-D'EPY;39160;39;JURA;1
39210;;quevillon;EQUEVILLON;39300;39;JURA;1
39211;Les;Essards-Taignevaux;ESSARDS-TAIGNEVAUX;39120;39;JURA;1
39213;;Esserval-Combe;ESSERVAL-COMBE;39250;39;JURA;1
39214;;Esserval-Tartre;ESSERVAL-TARTRE;39250;39;JURA;1
39215;;Essia;ESSIA;39270;39;JURA;1
39216;;tival;ETIVAL;39130;39;JURA;1
39217;L';toile;ETOILE;39570;39;JURA;1
39218;;trepigney;ETREPIGNEY;39700;39;JURA;1
39219;;vans;EVANS;39700;39;JURA;1
39220;;Falletans;FALLETANS;39700;39;JURA;1
39221;La;Favire;FAVIERE;39250;39;JURA;1
39222;;Fay-en-Montagne;FAY-EN-MONTAGNE;39800;39;JURA;1
39223;La;Fert;FERTE;39600;39;JURA;1
39224;;Ftigny;FETIGNY;39240;39;JURA;1
39225;Le;Fied;FIED;39800;39;JURA;1
39226;;Florentia;FLORENTIA;39320;39;JURA;1
39227;;Foncine-le-Bas;FONCINE-LE-BAS;39520;39;JURA;1
39228;;Foncine-le-Haut;FONCINE-LE-HAUT;39460;39;JURA;1
39229;;Fontainebrux;FONTAINEBRUX;39140;39;JURA;1
39230;;Fontenu;FONTENU;39130;39;JURA;1
39232;;Fort-du-Plasne;FORT-DU-PLASNE;39150;39;JURA;1
39233;;Foucherans;FOUCHERANS;39100;39;JURA;1
39234;;Foulenay;FOULENAY;39230;39;JURA;1
39235;;Fraisans;FRAISANS;39700;39;JURA;1
39236;;Francheville;FRANCHEVILLE;39230;39;JURA;1
39237;;Fraroz;FRAROZ;39250;39;JURA;1
39238;;Frasne-les-Meulires;FRASNE-LES-MEULIERES;39290;39;JURA;1
39239;La;Frasne;FRASNEE;39130;39;JURA;1
39240;Le;Frasnois;FRASNOIS;39130;39;JURA;1
39241;;Frbuans;FREBUANS;39570;39;JURA;1
39243;;Froideville;FROIDEVILLE;39230;39;JURA;1
39244;;Frontenay;FRONTENAY;39210;39;JURA;1
39245;;Gatey;GATEY;39120;39;JURA;1
39246;;Gendrey;GENDREY;39350;39;JURA;1
39247;;Genod;GENOD;39240;39;JURA;1
39248;;Geraise;GERAISE;39110;39;JURA;1
39249;;Germigney;GERMIGNEY;39380;39;JURA;1
39250;;Geruge;GERUGE;39570;39;JURA;1
39251;;Gevingey;GEVINGEY;39570;39;JURA;1
39252;;Gevry;GEVRY;39100;39;JURA;1
39253;;Gigny;GIGNY;39320;39;JURA;1
39254;;Gillois;GILLOIS;39250;39;JURA;1
39255;;Gizia;GIZIA;39190;39;JURA;1
39258;;Grande-Rivire;GRANDE-RIVIERE;39150;39;JURA;1
39259;;Grange-de-Vaivre;GRANGE-DE-VAIVRE;39600;39;JURA;1
39260;;Granges-sur-Baume;GRANGES-SUR-BAUME;39210;39;JURA;1
39261;;Graye-et-Charnay;GRAYE-ET-CHARNAY;39320;39;JURA;1
39262;;Gredisans;GREDISANS;39290;39;JURA;1
39263;;Grozon;GROZON;39800;39;JURA;1
39264;;Grusse;GRUSSE;39190;39;JURA;1
39265;;Hautecour;HAUTECOUR;39130;39;JURA;1
39266;Les;Hays;HAYS;39120;39;JURA;1
39267;;Ivory;IVORY;39110;39;JURA;1
39268;;Ivrey;IVREY;39110;39;JURA;1
39269;;Jeurre;JEURRE;39360;39;JURA;1
39270;;Jouhe;JOUHE;39100;39;JURA;1
39271;;Lac-des-Rouges-Truites;LAC-DES-ROUGES-TRUITES;39150;39;JURA;1
39272;;Ladoye-sur-Seille;LADOYE-SUR-SEILLE;39210;39;JURA;1
39273;;Lains;LAINS;39320;39;JURA;1
39274;;Lajoux;LAJOUX;;39;JURA;1
39275;;Lamoura;LAMOURA;39310;39;JURA;1
39277;Le;Larderet;LARDERET;39300;39;JURA;1
39278;;Largillay-Marsonnay;LARGILLAY-MARSONNAY;39130;39;JURA;1
39279;;Larnaud;LARNAUD;39140;39;JURA;1
39280;;Larrivoire;LARRIVOIRE;39360;39;JURA;1
39281;Le;Latet;LATET;39300;39;JURA;1
39282;La;Latette;LATETTE;39250;39;JURA;1
39283;;Lavancia-Epercy;LAVANCIA-EPERCY;01590;39;AIN;1
39284;;Lavangeot;LAVANGEOT;39700;39;JURA;1
39285;;Lavans-ls-Dole;LAVANS-LES-DOLE;39700;39;JURA;1
39286;;Lavans-ls-Saint-Claude;LAVANS-LES-SAINT-CLAUDE;39170;39;JURA;1
39287;;Lavans-sur-Valouse;LAVANS-SUR-VALOUSE;39240;39;JURA;1
39288;;Lavigny;LAVIGNY;39210;39;JURA;1
39289;;Lect;LECT;39260;39;JURA;1
39290;;Lgna;LEGNA;39240;39;JURA;1
39291;;Lemuy;LEMUY;39110;39;JURA;1
39292;;Lent;LENT;39300;39;JURA;1
39293;;Leschres;LESCHERES;39170;39;JURA;1
39294;;Lzat;LEZAT;39400;39;JURA;1
39295;;Loisia;LOISIA;39320;39;JURA;1
39296;;Lombard;LOMBARD;39230;39;JURA;1
39297;;Longchaumois;LONGCHAUMOIS;39400;39;JURA;1
39298;;Longcochon;LONGCOCHON;39250;39;JURA;1
39299;;Longwy-sur-le-Doubs;LONGWY-SUR-LE-DOUBS;39120;39;JURA;1
39300;;Lons-le-Saunier;LONS-LE-SAUNIER;39000;39;JURA;1
39301;;Loulle;LOULLE;39300;39;JURA;1
39302;;Louvatange;LOUVATANGE;39350;39;JURA;1
39303;;Louvenne;LOUVENNE;39320;39;JURA;1
39304;Le;Louverot;LOUVEROT;39210;39;JURA;1
39305;La;Loye;LOYE;39380;39;JURA;1
39306;;Macornay;MACORNAY;39570;39;JURA;1
39307;;Maisod;MAISOD;39260;39;JURA;1
39308;;Malange;MALANGE;39700;39;JURA;1
39309;;Mallerey;MALLEREY;39190;39;JURA;1
39310;;Mantry;MANTRY;39230;39;JURA;1
39312;;Marigna-sur-Valouse;MARIGNA-SUR-VALOUSE;39240;39;JURA;1
39313;;Marigny;MARIGNY;39130;39;JURA;1
39314;;Marnzia;MARNEZIA;39270;39;JURA;1
39315;;Marnoz;MARNOZ;39110;39;JURA;1
39317;La;Marre;MARRE;39210;39;JURA;1
39318;;Martigna;MARTIGNA;39260;39;JURA;1
39319;;Mathenay;MATHENAY;39600;39;JURA;1
39320;;Maynal;MAYNAL;39190;39;JURA;1
39321;;Mentru-le-Vignoble;MENETRU-LE-VIGNOBLE;39210;39;JURA;1
39322;;Mentrux-en-Joux;MENETRUX-EN-JOUX;39130;39;JURA;1
39323;;Menotey;MENOTEY;39290;39;JURA;1
39324;;Mrona;MERONA;39270;39;JURA;1
39325;;Mesnay;MESNAY;39600;39;JURA;1
39326;;Mesnois;MESNOIS;39130;39;JURA;1
39327;;Messia-sur-Sorne;MESSIA-SUR-SORNE;39570;39;JURA;1
39328;;Meussia;MEUSSIA;39260;39;JURA;1
39329;;Miges;MIEGES;39250;39;JURA;1
39330;;Miry;MIERY;39800;39;JURA;1
39331;;Mignovillard;MIGNOVILLARD;39250;39;JURA;1
39332;;Mirebel;MIREBEL;39570;39;JURA;1
39333;;Moirans-en-Montagne;MOIRANS-EN-MONTAGNE;39260;39;JURA;1
39334;;Moiron;MOIRON;39570;39;JURA;1
39335;;Moissey;MOISSEY;39290;39;JURA;1
39336;;Molain;MOLAIN;39800;39;JURA;1
39337;;Molamboz;MOLAMBOZ;39600;39;JURA;1
39338;;Molay;MOLAY;39500;39;JURA;1
39339;;Molinges;MOLINGES;39360;39;JURA;1
39340;;Molpr;MOLPRE;39250;39;JURA;1
39341;Les;Molunes;MOLUNES;39310;39;JURA;1
39342;;Monay;MONAY;39230;39;JURA;1
39343;;Monnetay;MONNETAY;39320;39;JURA;1
39344;;Monnet-la-Ville;MONNET-LA-VILLE;39300;39;JURA;1
39345;;Monnires;MONNIERES;39100;39;JURA;1
39346;;Montagna-le-Reconduit;MONTAGNA-LE-RECONDUIT;39160;39;JURA;1
39347;;Montagna-le-Templier;MONTAGNA-LE-TEMPLIER;39320;39;JURA;1
39348;;Montaigu;MONTAIGU;39570;39;JURA;1
39349;;Montain;MONTAIN;39210;39;JURA;1
39350;;Montbarrey;MONTBARREY;39380;39;JURA;1
39351;;Montcusel;MONTCUSEL;39260;39;JURA;1
39352;;Monteplain;MONTEPLAIN;39700;39;JURA;1
39353;;Montfleur;MONTFLEUR;39320;39;JURA;1
39354;;Montholier;MONTHOLIER;39800;39;JURA;1
39355;;Montigny-ls-Arsures;MONTIGNY-LES-ARSURES;39600;39;JURA;1
39356;;Montigny-sur-l'Ain;MONTIGNY-SUR-L'AIN;39300;39;JURA;1
39359;;Montmarlon;MONTMARLON;39110;39;JURA;1
39360;;Montmirey-la-Ville;MONTMIREY-LA-VILLE;39290;39;JURA;1
39361;;Montmirey-le-Chteau;MONTMIREY-LE-CHATEAU;39290;39;JURA;1
39362;;Montmorot;MONTMOROT;39570;39;JURA;1
39363;;Montrevel;MONTREVEL;39320;39;JURA;1
39364;;Montrond;MONTROND;39300;39;JURA;1
39365;;Mont-sous-Vaudrey;MONT-SOUS-VAUDREY;39380;39;JURA;1
39366;;Mont-sur-Monnet;MONT-SUR-MONNET;39300;39;JURA;1
39367;;Morbier;MORBIER;39400;39;JURA;1
39368;;Morez;MOREZ;39400;39;JURA;1
39370;;Mouchard;MOUCHARD;39330;39;JURA;1
39371;La;Mouille;MOUILLE;39400;39;JURA;1
39372;;Mournans-Charbonny;MOURNANS-CHARBONNY;39250;39;JURA;1
39373;Les;Moussires;MOUSSIERES;39310;39;JURA;1
39375;;Moutonne;MOUTONNE;39270;39;JURA;1
39376;;Moutoux;MOUTOUX;39300;39;JURA;1
39377;;Mutigney;MUTIGNEY;39290;39;JURA;1
39378;;Nanc-ls-Saint-Amour;NANC-LES-SAINT-AMOUR;39160;39;JURA;1
39379;;Nance;NANCE;39140;39;JURA;1
39380;;Nancuise;NANCUISE;39270;39;JURA;1
39381;Les;Nans;NANS;39300;39;JURA;1
39382;;Nantey;NANTEY;39160;39;JURA;1
39385;;Neublans-Abergement;NEUBLANS-ABERGEMENT;39120;39;JURA;1
39386;;Neuvilley;NEUVILLEY;39800;39;JURA;1
39387;;Nevy-ls-Dole;NEVY-LES-DOLE;39380;39;JURA;1
39388;;Nevy-sur-Seille;NEVY-SUR-SEILLE;39210;39;JURA;1
39389;;Ney;NEY;39300;39;JURA;1
39390;;Nogna;NOGNA;39570;39;JURA;1
39391;;Nozeroy;NOZEROY;39250;39;JURA;1
39392;;Offlanges;OFFLANGES;39290;39;JURA;1
39393;;Onglires;ONGLIERES;39250;39;JURA;1
39394;;Onoz;ONOZ;39270;39;JURA;1
39395;;Orbagna;ORBAGNA;39190;39;JURA;1
39396;;Orchamps;ORCHAMPS;39700;39;JURA;1
39397;;Orgelet;ORGELET;39270;39;JURA;1
39398;;Ougney;OUGNEY;39350;39;JURA;1
39399;;Ounans;OUNANS;39380;39;JURA;1
39400;;Our;OUR;39700;39;JURA;1
39401;;Oussires;OUSSIERES;39800;39;JURA;1
39402;;Pagney;PAGNEY;39350;39;JURA;1
39403;;Pagnoz;PAGNOZ;39330;39;JURA;1
39404;;Pannessires;PANNESSIERES;39570;39;JURA;1
39405;;Parcey;PARCEY;39100;39;JURA;1
39406;Le;Pasquier;PASQUIER;39300;39;JURA;1
39407;;Passenans;PASSENANS;39230;39;JURA;1
39408;;Patornay;PATORNAY;39130;39;JURA;1
39409;;Peintre;PEINTRE;39290;39;JURA;1
39411;;Perrigny;PERRIGNY;39570;39;JURA;1
39412;;Peseux;PESEUX;39120;39;JURA;1
39413;La;Pesse;PESSE;39370;39;JURA;1
39414;Le;Petit-Mercey;PETIT-MERCEY;39350;39;JURA;1
39415;;Petit-Noir;PETIT-NOIR;39120;39;JURA;1
39417;Les;Piards;PIARDS;39150;39;JURA;1
39418;;Picarreau;PICARREAU;39800;39;JURA;1
39419;;Pillemoine;PILLEMOINE;39300;39;JURA;1
39420;;Pimorin;PIMORIN;39270;39;JURA;1
39421;Le;Pin;PIN;39210;39;JURA;1
39422;;Plainoiseau;PLAINOISEAU;39210;39;JURA;1
39423;;Plaisia;PLAISIA;39270;39;JURA;1
39424;Les;Planches-en-Montagne;PLANCHES-EN-MONTAGNE;39150;39;JURA;1
39425;Les;Planches-prs-Arbois;PLANCHES-PRES-ARBOIS;39600;39;JURA;1
39426;;Plasne;PLASNE;39210;39;JURA;1
39427;;Plnise;PLENISE;39250;39;JURA;1
39428;;Plnisette;PLENISETTE;39250;39;JURA;1
39429;;Pleure;PLEURE;39120;39;JURA;1
39430;;Plumont;PLUMONT;39700;39;JURA;1
39431;;Poids-de-Fiole;POIDS-DE-FIOLE;39570;39;JURA;1
39432;;Pointre;POINTRE;39290;39;JURA;1
39434;;Poligny;POLIGNY;39800;39;JURA;1
39435;;Pont-de-Poitte;PONT-DE-POITTE;39130;39;JURA;1
39436;;Pont-d'Hry;PONT-D'HERY;39110;39;JURA;1
39437;;Pont-du-Navoy;PONT-DU-NAVOY;39300;39;JURA;1
39438;;Ponthoux;PONTHOUX;39170;39;JURA;1
39439;;Port-Lesney;PORT-LESNEY;39330;39;JURA;1
39440;;Pratz;PRATZ;39170;39;JURA;1
39441;;Prmanon;PREMANON;39220;39;JURA;1
39442;;Prnovel;PRENOVEL;39150;39;JURA;1
39443;;Prsilly;PRESILLY;39270;39;JURA;1
39444;;Pretin;PRETIN;39110;39;JURA;1
39445;;Publy;PUBLY;39570;39;JURA;1
39446;;Pupillin;PUPILLIN;39600;39;JURA;1
39447;;Quintigny;QUINTIGNY;39570;39;JURA;1
39448;;Rahon;RAHON;39120;39;JURA;1
39449;;Rainans;RAINANS;39290;39;JURA;1
39451;;Ranchot;RANCHOT;39700;39;JURA;1
39452;;Rans;RANS;39700;39;JURA;1
39453;;Ravilloles;RAVILLOLES;39170;39;JURA;1
39454;;Recanoz;RECANOZ;39230;39;JURA;1
39455;;Reithouse;REITHOUSE;39270;39;JURA;1
39456;;Relans;RELANS;39140;39;JURA;1
39457;Les;Repts;REPOTS;39140;39;JURA;1
39458;;Revigny;REVIGNY;39570;39;JURA;1
39460;La;Rixouse;RIXOUSE;39200;39;JURA;1
39461;;Rix;RIX;39250;39;JURA;1
39462;;Rochefort-sur-Nenon;ROCHEFORT-SUR-NENON;39700;39;JURA;1
39463;;Rogna;ROGNA;39360;39;JURA;1
39464;;Romain;ROMAIN;39350;39;JURA;1
39465;;Romange;ROMANGE;39700;39;JURA;1
39466;;Rosay;ROSAY;39190;39;JURA;1
39467;;Rotalier;ROTALIER;39190;39;JURA;1
39468;;Rothonay;ROTHONAY;39270;39;JURA;1
39469;;Rouffange;ROUFFANGE;39350;39;JURA;1
39470;Les;Rousses;ROUSSES;39220;39;JURA;1
39471;;Ruffey-sur-Seille;RUFFEY-SUR-SEILLE;39140;39;JURA;1
39472;;Rye;RYE;39230;39;JURA;1
39473;;Saffloz;SAFFLOZ;39130;39;JURA;1
39474;;Sainte-Agns;SAINTE-AGNES;39190;39;JURA;1
39475;;Saint-Amour;SAINT-AMOUR;39160;39;JURA;1
39476;;Saint-Aubin;SAINT-AUBIN;39410;39;JURA;1
39477;;Saint-Baraing;SAINT-BARAING;39120;39;JURA;1
39478;;Saint-Claude;SAINT-CLAUDE;39200;39;JURA;1
39479;;Saint-Cyr-Montmalin;SAINT-CYR-MONTMALIN;39600;39;JURA;1
39480;;Saint-Didier;SAINT-DIDIER;39570;39;JURA;1
39481;;Saint-Germain-en-Montagne;SAINT-GERMAIN-EN-MONTAGNE;39300;39;JURA;1
39482;;Saint-Germain-ls-Arlay;SAINT-GERMAIN-LES-ARLAY;39210;39;JURA;1
39483;;Saint-Hymetire;SAINT-HYMETIERE;39240;39;JURA;1
39484;;Saint-Jean-d'treux;SAINT-JEAN-D'ETREUX;39160;39;JURA;1
39485;;Saint-Julien;SAINT-JULIEN;39320;39;JURA;1
39486;;Saint-Lamain;SAINT-LAMAIN;39230;39;JURA;1
39487;;Saint-Laurent-en-Grandvaux;SAINT-LAURENT-EN-GRANDVAUX;39150;39;JURA;1
39488;;Saint-Laurent-la-Roche;SAINT-LAURENT-LA-ROCHE;39570;39;JURA;1
39489;;Saint-Lothain;SAINT-LOTHAIN;39230;39;JURA;1
39490;;Saint-Loup;SAINT-LOUP;39120;39;JURA;1
39491;;Saint-Lupicin;SAINT-LUPICIN;39170;39;JURA;1
39492;;Saint-Maur;SAINT-MAUR;39570;39;JURA;1
39493;;Saint-Maurice-Crillat;SAINT-MAURICE-CRILLAT;39130;39;JURA;1
39494;;Saint-Pierre;SAINT-PIERRE;39150;39;JURA;1
39495;;Saint-Thibaud;SAINT-THIEBAUD;39110;39;JURA;1
39497;;Saizenay;SAIZENAY;39110;39;JURA;1
39498;;Salans;SALANS;39700;39;JURA;1
39499;;Saligney;SALIGNEY;39350;39;JURA;1
39500;;Salins-les-Bains;SALINS-LES-BAINS;39110;39;JURA;1
39501;;Sampans;SAMPANS;39100;39;JURA;1
39502;;Santans;SANTANS;39380;39;JURA;1
39503;;Sapois;SAPOIS;39300;39;JURA;1
39504;;Sarrogna;SARROGNA;39270;39;JURA;1
39505;;Saugeot;SAUGEOT;39130;39;JURA;1
39506;;Savigna;SAVIGNA;39240;39;JURA;1
39507;;Sligney;SELIGNEY;39120;39;JURA;1
39508;;Sellires;SELLIERES;39230;39;JURA;1
39509;;Senaud;SENAUD;39160;39;JURA;1
39510;;Septmoncel;SEPTMONCEL;39310;39;JURA;1
39511;;Sergenaux;SERGENAUX;39230;39;JURA;1
39512;;Sergenon;SERGENON;39120;39;JURA;1
39513;;Sermange;SERMANGE;39700;39;JURA;1
39514;;Serre-les-Moulires;SERRE-LES-MOULIERES;39700;39;JURA;1
39517;;Sirod;SIROD;39300;39;JURA;1
39518;;Songeson;SONGESON;39130;39;JURA;1
39519;;Soucia;SOUCIA;39130;39;JURA;1
39520;;Souvans;SOUVANS;39380;39;JURA;1
39522;;Supt;SUPT;39300;39;JURA;1
39523;;Syam;SYAM;39300;39;JURA;1
39525;;Tassenires;TASSENIERES;39120;39;JURA;1
39526;;Tavaux;TAVAUX;39500;39;JURA;1
39527;;Taxenne;TAXENNE;39350;39;JURA;1
39528;;Thervay;THERVAY;39290;39;JURA;1
39529;;Thsy;THESY;39110;39;JURA;1
39530;;Thoirette;THOIRETTE;39240;39;JURA;1
39531;;Thoiria;THOIRIA;39130;39;JURA;1
39532;;Thoissia;THOISSIA;39160;39;JURA;1
39533;;Toulouse-le-Chteau;TOULOUSE-LE-CHATEAU;39230;39;JURA;1
39534;La;Tour-du-Meix;TOUR-DU-MEIX;39270;39;JURA;1
39535;;Tourmont;TOURMONT;39800;39;JURA;1
39537;;Trenal;TRENAL;39570;39;JURA;1
39538;;Uxelles;UXELLES;39130;39;JURA;1
39539;;Vadans;VADANS;39600;39;JURA;1
39540;;Valempoulires;VALEMPOULIERES;39300;39;JURA;1
39542;;Valfin-sur-Valouse;VALFIN-SUR-VALOUSE;39240;39;JURA;1
39543;;Vannoz;VANNOZ;39300;39;JURA;1
39544;;Varessia;VARESSIA;39270;39;JURA;1
39545;Le;Vaudioux;VAUDIOUX;39300;39;JURA;1
39546;;Vaudrey;VAUDREY;39380;39;JURA;1
39547;;Vaux-ls-Saint-Claude;VAUX-LES-SAINT-CLAUDE;39360;39;JURA;1
39548;;Vaux-sur-Poligny;VAUX-SUR-POLIGNY;39800;39;JURA;1
39549;;Vercia;VERCIA;39190;39;JURA;1
39550;;Verges;VERGES;39570;39;JURA;1
39551;;Vria;VERIA;39160;39;JURA;1
39552;;Vernantois;VERNANTOIS;39570;39;JURA;1
39553;Le;Vernois;VERNOIS;39210;39;JURA;1
39554;;Vers-en-Montagne;VERS-EN-MONTAGNE;39300;39;JURA;1
39555;;Vers-sous-Sellires;VERS-SOUS-SELLIERES;39230;39;JURA;1
39556;;Vertamboz;VERTAMBOZ;39130;39;JURA;1
39557;;Vescles;VESCLES;39240;39;JURA;1
39558;;Vevy;VEVY;39570;39;JURA;1
39559;La;Vieille-Loye;VIEILLE-LOYE;39380;39;JURA;1
39560;;Villard-Saint-Sauveur;VILLARD-SAINT-SAUVEUR;39200;39;JURA;1
39561;;Villards-d'Hria;VILLARDS-D'HERIA;39260;39;JURA;1
39562;;Villard-sur-Bienne;VILLARD-SUR-BIENNE;39200;39;JURA;1
39564;;Villechantria;VILLECHANTRIA;39320;39;JURA;1
39565;;Villeneuve-d'Aval;VILLENEUVE-D'AVAL;39600;39;JURA;1
39566;;Villeneuve-ls-Charnod;VILLENEUVE-LES-CHARNOD;39240;39;JURA;1
39567;;Villeneuve-sous-Pymont;VILLENEUVE-SOUS-PYMONT;39570;39;JURA;1
39568;;Villerserine;VILLERSERINE;39800;39;JURA;1
39569;;Villers-Farlay;VILLERS-FARLAY;39600;39;JURA;1
39570;;Villers-les-Bois;VILLERS-LES-BOIS;39120;39;JURA;1
39571;;Villers-Robert;VILLERS-ROBERT;39120;39;JURA;1
39572;;Villette-ls-Arbois;VILLETTE-LES-ARBOIS;39600;39;JURA;1
39573;;Villette-ls-Dole;VILLETTE-LES-DOLE;39100;39;JURA;1
39574;;Villevieux;VILLEVIEUX;39140;39;JURA;1
39575;Le;Villey;VILLEY;39230;39;JURA;1
39576;;Vincelles;VINCELLES;39190;39;JURA;1
39577;;Vincent;VINCENT;39230;39;JURA;1
39579;;Viry;VIRY;39360;39;JURA;1
39581;;Vitreux;VITREUX;39350;39;JURA;1
39582;;Voiteur;VOITEUR;39210;39;JURA;1
39583;;Vosbles;VOSBLES;39240;39;JURA;1
39584;;Vriange;VRIANGE;39700;39;JURA;1
39585;;Vulvoz;VULVOZ;39360;39;JURA;1
39586;;Aresches;ARESCHES;39110;39;JURA;1
40001;;Aire-sur-l'Adour;AIRE-SUR-L'ADOUR;40800;40;LANDES;1
40002;;Amou;AMOU;40330;40;LANDES;1
40003;;Angoum;ANGOUME;40990;40;LANDES;1
40004;;Angresse;ANGRESSE;40150;40;LANDES;1
40005;;Arboucave;ARBOUCAVE;40320;40;LANDES;1
40006;;Arengosse;ARENGOSSE;40110;40;LANDES;1
40007;;Argelos;ARGELOS;40700;40;LANDES;1
40008;;Argelouse;ARGELOUSE;40430;40;LANDES;1
40009;;Arjuzanx;ARJUZANX;40110;40;LANDES;1
40011;;Arsague;ARSAGUE;40330;40;LANDES;1
40012;;Artassenx;ARTASSENX;40090;40;LANDES;1
40013;;Arthez-d'Armagnac;ARTHEZ-D'ARMAGNAC;40190;40;LANDES;1
40014;;Arue;ARUE;40120;40;LANDES;1
40015;;Arx;ARX;40310;40;LANDES;1
40016;;Aubagnan;AUBAGNAN;40700;40;LANDES;1
40017;;Audignon;AUDIGNON;40500;40;LANDES;1
40018;;Audon;AUDON;40400;40;LANDES;1
40019;;Aureilhan;AUREILHAN;40200;40;LANDES;1
40020;;Aurice;AURICE;40500;40;LANDES;1
40021;;Azur;AZUR;40140;40;LANDES;1
40022;;Bahus-Soubiran;BAHUS-SOUBIRAN;40320;40;LANDES;1
40023;;Baigts;BAIGTS;40380;40;LANDES;1
40024;;Banos;BANOS;40500;40;LANDES;1
40025;;Bascons;BASCONS;40090;40;LANDES;1
40026;;Bas-Mauco;BAS-MAUCO;40500;40;LANDES;1
40027;;Bassercles;BASSERCLES;40700;40;LANDES;1
40028;;Bastennes;BASTENNES;40360;40;LANDES;1
40029;;Bats;BATS;40320;40;LANDES;1
40030;;Baudignan;BAUDIGNAN;40310;40;LANDES;1
40031;;Bgaar;BEGAAR;40400;40;LANDES;1
40032;;Belhade;BELHADE;40410;40;LANDES;1
40033;;Blis;BELIS;40120;40;LANDES;1
40034;;Blus;BELUS;40300;40;LANDES;1
40035;;Bnesse-ls-Dax;BENESSE-LES-DAX;40180;40;LANDES;1
40036;;Bnesse-Maremne;BENESSE-MAREMNE;40230;40;LANDES;1
40037;;Benquet;BENQUET;40280;40;LANDES;1
40038;;Bergouey;BERGOUEY;40250;40;LANDES;1
40039;;Betbezer-d'Armagnac;BETBEZER-D'ARMAGNAC;40240;40;LANDES;1
40040;;Beylongue;BEYLONGUE;40370;40;LANDES;1
40041;;Beyries;BEYRIES;40700;40;LANDES;1
40042;;Biarrotte;BIARROTTE;40390;40;LANDES;1
40043;;Bias;BIAS;40170;40;LANDES;1
40044;;Biaudos;BIAUDOS;40390;40;LANDES;1
40046;;Biscarrosse;BISCARROSSE;40600;40;LANDES;1
40047;;Bonnegarde;BONNEGARDE;40330;40;LANDES;1
40048;;Boos;BOOS;40370;40;LANDES;1
40049;;Bordres-et-Lamensans;BORDERES-ET-LAMENSANS;40270;40;LANDES;1
40050;;Bostens;BOSTENS;40090;40;LANDES;1
40051;;Bougue;BOUGUE;40090;40;LANDES;1
40052;;Bourdalat;BOURDALAT;40190;40;LANDES;1
40053;;Bourriot-Bergonce;BOURRIOT-BERGONCE;40120;40;LANDES;1
40054;;Brassempouy;BRASSEMPOUY;40330;40;LANDES;1
40055;;Bretagne-de-Marsan;BRETAGNE-DE-MARSAN;40280;40;LANDES;1
40056;;Brocas;BROCAS;40420;40;LANDES;1
40057;;Buanes;BUANES;40320;40;LANDES;1
40058;;Cachen;CACHEN;40120;40;LANDES;1
40059;;Cagnotte;CAGNOTTE;40300;40;LANDES;1
40060;;Callen;CALLEN;40430;40;LANDES;1
40061;;Campagne;CAMPAGNE;40090;40;LANDES;1
40062;;Campet-et-Lamolre;CAMPET-ET-LAMOLERE;40090;40;LANDES;1
40063;;Candresse;CANDRESSE;40180;40;LANDES;1
40064;;Canenx-et-Raut;CANENX-ET-REAUT;40090;40;LANDES;1
40065;;Capbreton;CAPBRETON;40130;40;LANDES;1
40066;;Carcars-Sainte-Croix;CARCARES-SAINTE-CROIX;40400;40;LANDES;1
40067;;Carcen-Ponson;CARCEN-PONSON;40400;40;LANDES;1
40068;;Cassen;CASSEN;40380;40;LANDES;1
40069;;Castaignos-Souslens;CASTAIGNOS-SOUSLENS;40700;40;LANDES;1
40070;;Castandet;CASTANDET;40270;40;LANDES;1
40071;;Castelnau-Chalosse;CASTELNAU-CHALOSSE;40360;40;LANDES;1
40072;;Castelnau-Tursan;CASTELNAU-TURSAN;40320;40;LANDES;1
40073;;Castelner;CASTELNER;40700;40;LANDES;1
40074;;Castel-Sarrazin;CASTEL-SARRAZIN;40330;40;LANDES;1
40075;;Castets;CASTETS;40260;40;LANDES;1
40076;;Cauna;CAUNA;40500;40;LANDES;1
40077;;Cauneille;CAUNEILLE;40300;40;LANDES;1
40078;;Caupenne;CAUPENNE;40250;40;LANDES;1
40079;;Cazalis;CAZALIS;40700;40;LANDES;1
40080;;Cazres-sur-l'Adour;CAZERES-SUR-L'ADOUR;40270;40;LANDES;1
40081;;Cre;CERE;40090;40;LANDES;1
40082;;Classun;CLASSUN;40320;40;LANDES;1
40083;;Cldes;CLEDES;40320;40;LANDES;1
40084;;Clermont;CLERMONT;40180;40;LANDES;1
40085;;Commensacq;COMMENSACQ;40210;40;LANDES;1
40086;;Coudures;COUDURES;40500;40;LANDES;1
40087;;Cron-d'Armagnac;CREON-D'ARMAGNAC;40240;40;LANDES;1
40088;;Dax;DAX;40100;40;LANDES;1
40089;;Doazit;DOAZIT;40700;40;LANDES;1
40090;;Donzacq;DONZACQ;40360;40;LANDES;1
40091;;Duhort-Bachen;DUHORT-BACHEN;40800;40;LANDES;1
40092;;Dumes;DUMES;40500;40;LANDES;1
40093;;Escalans;ESCALANS;40310;40;LANDES;1
40094;;Escource;ESCOURCE;40210;40;LANDES;1
40095;;Estibeaux;ESTIBEAUX;40290;40;LANDES;1
40096;;Estigarde;ESTIGARDE;40240;40;LANDES;1
40097;;Eugnie-les-Bains;EUGENIE-LES-BAINS;40320;40;LANDES;1
40098;;Eyres-Moncube;EYRES-MONCUBE;40500;40;LANDES;1
40099;;Fargues;FARGUES;40500;40;LANDES;1
40100;Le;Frche;FRECHE;40190;40;LANDES;1
40101;;Gaas;GAAS;40350;40;LANDES;1
40102;;Gabarret;GABARRET;40310;40;LANDES;1
40103;;Gaillres;GAILLERES;40090;40;LANDES;1
40104;;Gamarde-les-Bains;GAMARDE-LES-BAINS;40380;40;LANDES;1
40105;;Garein;GAREIN;40420;40;LANDES;1
40106;;Garrey;GARREY;40180;40;LANDES;1
40107;;Garrosse;GARROSSE;40110;40;LANDES;1
40108;;Gastes;GASTES;40160;40;LANDES;1
40109;;Gaujacq;GAUJACQ;40330;40;LANDES;1
40110;;Geaune;GEAUNE;40320;40;LANDES;1
40111;;Geloux;GELOUX;40090;40;LANDES;1
40112;;Gibret;GIBRET;40380;40;LANDES;1
40113;;Goos;GOOS;40180;40;LANDES;1
40114;;Gourbera;GOURBERA;40990;40;LANDES;1
40115;;Gousse;GOUSSE;40465;40;LANDES;1
40116;;Gouts;GOUTS;40400;40;LANDES;1
40117;;Grenade-sur-l'Adour;GRENADE-SUR-L'ADOUR;40270;40;LANDES;1
40118;;Habas;HABAS;40290;40;LANDES;1
40119;;Hagetmau;HAGETMAU;40700;40;LANDES;1
40120;;Hastingues;HASTINGUES;40300;40;LANDES;1
40121;;Hauriet;HAURIET;40250;40;LANDES;1
40122;;Haut-Mauco;HAUT-MAUCO;40280;40;LANDES;1
40123;;Herm;HERM;40990;40;LANDES;1
40124;;Herr;HERRE;40310;40;LANDES;1
40125;;Heugas;HEUGAS;40180;40;LANDES;1
40126;;Hinx;HINX;40180;40;LANDES;1
40127;;Hontanx;HONTANX;40190;40;LANDES;1
40128;;Horsarrieu;HORSARRIEU;40700;40;LANDES;1
40129;;Josse;JOSSE;40230;40;LANDES;1
40130;;Labastide-Chalosse;LABASTIDE-CHALOSSE;40700;40;LANDES;1
40131;;Labastide-d'Armagnac;LABASTIDE-D'ARMAGNAC;40240;40;LANDES;1
40132;;Labatut;LABATUT;40300;40;LANDES;1
40133;;Labenne;LABENNE;40530;40;LANDES;1
40134;;Labouheyre;LABOUHEYRE;40210;40;LANDES;1
40135;;Labrit;LABRIT;40420;40;LANDES;1
40136;;Lacajunte;LACAJUNTE;40320;40;LANDES;1
40137;;Lacquy;LACQUY;40120;40;LANDES;1
40138;;Lacrabe;LACRABE;40700;40;LANDES;1
40139;;Laglorieuse;LAGLORIEUSE;40090;40;LANDES;1
40140;;Lagrange;LAGRANGE;40240;40;LANDES;1
40141;;Lahosse;LAHOSSE;40250;40;LANDES;1
40142;;Laluque;LALUQUE;40465;40;LANDES;1
40143;;Lamothe;LAMOTHE;40250;40;LANDES;1
40144;;Larbey;LARBEY;40250;40;LANDES;1
40145;;Larrivire-Saint-Savin;LARRIVIERE-SAINT-SAVIN;40270;40;LANDES;1
40146;;Latrille;LATRILLE;40800;40;LANDES;1
40147;;Laurde;LAUREDE;40250;40;LANDES;1
40148;;Lauret;LAURET;40320;40;LANDES;1
40149;;Lencouacq;LENCOUACQ;40120;40;LANDES;1
40150;;Lon;LEON;40550;40;LANDES;1
40151;;Lesgor;LESGOR;40400;40;LANDES;1
40152;;Lesperon;LESPERON;40260;40;LANDES;1
40153;Le;Leuy;LEUY;40250;40;LANDES;1
40154;;Lvignacq;LEVIGNACQ;40170;40;LANDES;1
40155;;Linxe;LINXE;40260;40;LANDES;1
40156;;Liposthey;LIPOSTHEY;40410;40;LANDES;1
40157;;Lit-et-Mixe;LIT-ET-MIXE;40170;40;LANDES;1
40158;;Losse;LOSSE;40240;40;LANDES;1
40159;;Louer;LOUER;40380;40;LANDES;1
40160;;Lourquen;LOURQUEN;40250;40;LANDES;1
40161;;Lubbon;LUBBON;40240;40;LANDES;1
40162;;Lucbardez-et-Bargues;LUCBARDEZ-ET-BARGUES;40090;40;LANDES;1
40163;;Le;LUE;40210;40;LANDES;1
40164;;Retjons;RETJONS;40120;40;LANDES;1
40165;;Luglon;LUGLON;40630;40;LANDES;1
40166;;Lussagnet;LUSSAGNET;40270;40;LANDES;1
40167;;Luxey;LUXEY;40430;40;LANDES;1
40168;;Magescq;MAGESCQ;40140;40;LANDES;1
40169;;Maillas;MAILLAS;40120;40;LANDES;1
40170;;Maillres;MAILLERES;40120;40;LANDES;1
40171;;Mano;MANO;40410;40;LANDES;1
40172;;Mant;MANT;40700;40;LANDES;1
40173;;Marpaps;MARPAPS;40330;40;LANDES;1
40174;;Mauries;MAURIES;40320;40;LANDES;1
40175;;Maurrin;MAURRIN;40270;40;LANDES;1
40176;;Mauvezin-d'Armagnac;MAUVEZIN-D'ARMAGNAC;40240;40;LANDES;1
40177;;Maylis;MAYLIS;40250;40;LANDES;1
40178;;Mazerolles;MAZEROLLES;40090;40;LANDES;1
40179;;Mes;MEES;40990;40;LANDES;1
40180;;Meilhan;MEILHAN;40400;40;LANDES;1
40181;;Messanges;MESSANGES;40660;40;LANDES;1
40182;;Mzos;MEZOS;40170;40;LANDES;1
40183;;Mimbaste;MIMBASTE;40350;40;LANDES;1
40184;;Mimizan;MIMIZAN;40200;40;LANDES;1
40185;;Miramont-Sensacq;MIRAMONT-SENSACQ;40320;40;LANDES;1
40186;;Misson;MISSON;40290;40;LANDES;1
40187;;Moliets-et-Maa;MOLIETS-ET-MAA;40660;40;LANDES;1
40188;;Momuy;MOMUY;40700;40;LANDES;1
40189;;Monget;MONGET;40700;40;LANDES;1
40190;;Monsgur;MONSEGUR;40700;40;LANDES;1
40191;;Montaut;MONTAUT;40500;40;LANDES;1
40192;;Mont-de-Marsan;MONT-DE-MARSAN;40000;40;LANDES;1
40193;;Montgut;MONTEGUT;40190;40;LANDES;1
40194;;Montfort-en-Chalosse;MONTFORT-EN-CHALOSSE;40380;40;LANDES;1
40195;;Montgaillard;MONTGAILLARD;40500;40;LANDES;1
40196;;Montsou;MONTSOUE;40500;40;LANDES;1
40197;;Morcenx;MORCENX;40110;40;LANDES;1
40198;;Morganx;MORGANX;40700;40;LANDES;1
40199;;Mouscards;MOUSCARDES;40290;40;LANDES;1
40200;;Moustey;MOUSTEY;40410;40;LANDES;1
40201;;Mugron;MUGRON;40250;40;LANDES;1
40202;;Narrosse;NARROSSE;40180;40;LANDES;1
40203;;Nassiet;NASSIET;40330;40;LANDES;1
40204;;Nerbis;NERBIS;40250;40;LANDES;1
40205;;Nousse;NOUSSE;40380;40;LANDES;1
40206;;Oeyregave;OEYREGAVE;40300;40;LANDES;1
40207;;Oeyreluy;OEYRELUY;40180;40;LANDES;1
40208;;Onard;ONARD;40380;40;LANDES;1
40209;;Ondres;ONDRES;40440;40;LANDES;1
40210;;Onesse-et-Laharie;ONESSE-ET-LAHARIE;40110;40;LANDES;1
40211;;Orist;ORIST;40300;40;LANDES;1
40212;;Orthevielle;ORTHEVIELLE;40300;40;LANDES;1
40213;;Orx;ORX;40230;40;LANDES;1
40214;;Ossages;OSSAGES;40290;40;LANDES;1
40215;;Ousse-Suzan;OUSSE-SUZAN;40110;40;LANDES;1
40216;;Ozourt;OZOURT;40380;40;LANDES;1
40217;;Parentis-en-Born;PARENTIS-EN-BORN;40160;40;LANDES;1
40218;;Parleboscq;PARLEBOSCQ;40310;40;LANDES;1
40219;;Payros-Cazautets;PAYROS-CAZAUTETS;40320;40;LANDES;1
40220;;Pcorade;PECORADE;40320;40;LANDES;1
40221;;Perquie;PERQUIE;40190;40;LANDES;1
40222;;Pey;PEY;40300;40;LANDES;1
40223;;Peyre;PEYRE;40700;40;LANDES;1
40224;;Peyrehorade;PEYREHORADE;40300;40;LANDES;1
40225;;Philondenx;PHILONDENX;40320;40;LANDES;1
40226;;Pimbo;PIMBO;40320;40;LANDES;1
40227;;Pissos;PISSOS;40410;40;LANDES;1
40228;;Pomarez;POMAREZ;40360;40;LANDES;1
40229;;Pontenx-les-Forges;PONTENX-LES-FORGES;40200;40;LANDES;1
40230;;Pontonx-sur-l'Adour;PONTONX-SUR-L'ADOUR;40465;40;LANDES;1
40231;;Port-de-Lanne;PORT-DE-LANNE;40300;40;LANDES;1
40232;;Poudenx;POUDENX;40700;40;LANDES;1
40233;;Pouillon;POUILLON;40350;40;LANDES;1
40234;;Pouydesseaux;POUYDESSEAUX;40120;40;LANDES;1
40235;;Poyanne;POYANNE;40380;40;LANDES;1
40236;;Poyartin;POYARTIN;40380;40;LANDES;1
40237;;Prchacq-les-Bains;PRECHACQ-LES-BAINS;40465;40;LANDES;1
40238;;Pujo-le-Plan;PUJO-LE-PLAN;40190;40;LANDES;1
40239;;Puyol-Cazalet;PUYOL-CAZALET;40320;40;LANDES;1
40240;;Renung;RENUNG;40270;40;LANDES;1
40242;;Rimbez-et-Baudiets;RIMBEZ-ET-BAUDIETS;40310;40;LANDES;1
40243;;Rion-des-Landes;RION-DES-LANDES;40370;40;LANDES;1
40244;;Rivire-Saas-et-Gourby;RIVIERE-SAAS-ET-GOURBY;40180;40;LANDES;1
40245;;Roquefort;ROQUEFORT;40120;40;LANDES;1
40246;;Sabres;SABRES;40630;40;LANDES;1
40247;;Saint-Agnet;SAINT-AGNET;40800;40;LANDES;1
40248;;Saint-Andr-de-Seignanx;SAINT-ANDRE-DE-SEIGNANX;40390;40;LANDES;1
40249;;Saint-Aubin;SAINT-AUBIN;40250;40;LANDES;1
40250;;Saint-Avit;SAINT-AVIT;40090;40;LANDES;1
40251;;Saint-Barthlemy;SAINT-BARTHELEMY;40390;40;LANDES;1
40252;;Sainte-Colombe;SAINTE-COLOMBE;40700;40;LANDES;1
40253;;Saint-Cricq-Chalosse;SAINT-CRICQ-CHALOSSE;40700;40;LANDES;1
40254;;Saint-Cricq-du-Gave;SAINT-CRICQ-DU-GAVE;40300;40;LANDES;1
40255;;Saint-Cricq-Villeneuve;SAINT-CRICQ-VILLENEUVE;40190;40;LANDES;1
40256;;Saint-tienne-d'Orthe;SAINT-ETIENNE-D'ORTHE;40300;40;LANDES;1
40257;;Sainte-Eulalie-en-Born;SAINTE-EULALIE-EN-BORN;40200;40;LANDES;1
40258;;Sainte-Foy;SAINTE-FOY;40190;40;LANDES;1
40259;;Saint-Gein;SAINT-GEIN;40190;40;LANDES;1
40260;;Saint-Geours-d'Auribat;SAINT-GEOURS-D'AURIBAT;40380;40;LANDES;1
40261;;Saint-Geours-de-Maremne;SAINT-GEOURS-DE-MAREMNE;40230;40;LANDES;1
40262;;Saint-Gor;SAINT-GOR;40120;40;LANDES;1
40263;;Saint-Jean-de-Lier;SAINT-JEAN-DE-LIER;40380;40;LANDES;1
40264;;Saint-Jean-de-Marsacq;SAINT-JEAN-DE-MARSACQ;40230;40;LANDES;1
40265;;Saint-Julien-d'Armagnac;SAINT-JULIEN-D'ARMAGNAC;40240;40;LANDES;1
40266;;Saint-Julien-en-Born;SAINT-JULIEN-EN-BORN;40170;40;LANDES;1
40267;;Saint-Justin;SAINT-JUSTIN;40240;40;LANDES;1
40268;;Saint-Laurent-de-Gosse;SAINT-LAURENT-DE-GOSSE;40390;40;LANDES;1
40269;;Saint-Lon-les-Mines;SAINT-LON-LES-MINES;40300;40;LANDES;1
40270;;Saint-Loubouer;SAINT-LOUBOUER;40320;40;LANDES;1
40271;;Sainte-Marie-de-Gosse;SAINTE-MARIE-DE-GOSSE;40390;40;LANDES;1
40272;;Saint-Martin-de-Hinx;SAINT-MARTIN-DE-HINX;40390;40;LANDES;1
40273;;Saint-Martin-de-Seignanx;SAINT-MARTIN-DE-SEIGNANX;40390;40;LANDES;1
40274;;Saint-Martin-d'Oney;SAINT-MARTIN-D'ONEY;40090;40;LANDES;1
40275;;Saint-Maurice-sur-Adour;SAINT-MAURICE-SUR-ADOUR;40270;40;LANDES;1
40276;;Saint-Michel-Escalus;SAINT-MICHEL-ESCALUS;40550;40;LANDES;1
40277;;Saint-Pandelon;SAINT-PANDELON;40180;40;LANDES;1
40278;;Saint-Paul-en-Born;SAINT-PAUL-EN-BORN;40200;40;LANDES;1
40279;;Saint-Paul-ls-Dax;SAINT-PAUL-LES-DAX;40990;40;LANDES;1
40280;;Saint-Perdon;SAINT-PERDON;40090;40;LANDES;1
40281;;Saint-Pierre-du-Mont;SAINT-PIERRE-DU-MONT;40280;40;LANDES;1
40282;;Saint-Sever;SAINT-SEVER;40500;40;LANDES;1
40283;;Saint-Vincent-de-Paul;SAINT-VINCENT-DE-PAUL;40990;40;LANDES;1
40284;;Saint-Vincent-de-Tyrosse;SAINT-VINCENT-DE-TYROSSE;40230;40;LANDES;1
40285;;Saint-Yaguen;SAINT-YAGUEN;40400;40;LANDES;1
40286;;Samadet;SAMADET;40320;40;LANDES;1
40287;;Sanguinet;SANGUINET;40460;40;LANDES;1
40288;;Sarbazan;SARBAZAN;40120;40;LANDES;1
40289;;Sarraziet;SARRAZIET;40500;40;LANDES;1
40290;;Sarron;SARRON;40800;40;LANDES;1
40291;;Saubion;SAUBION;40230;40;LANDES;1
40292;;Saubrigues;SAUBRIGUES;40230;40;LANDES;1
40293;;Saubusse;SAUBUSSE;40180;40;LANDES;1
40294;;Saugnac-et-Cambran;SAUGNAC-ET-CAMBRAN;40180;40;LANDES;1
40295;;Saugnacq-et-Muret;SAUGNACQ-ET-MURET;40410;40;LANDES;1
40296;;Seignosse;SEIGNOSSE;40510;40;LANDES;1
40297;Le;Sen;SEN;40420;40;LANDES;1
40298;;Serres-Gaston;SERRES-GASTON;40700;40;LANDES;1
40299;;Serreslous-et-Arribans;SERRESLOUS-ET-ARRIBANS;40700;40;LANDES;1
40300;;Seyresse;SEYRESSE;40180;40;LANDES;1
40301;;Siest;SIEST;40180;40;LANDES;1
40302;;Sindres;SINDERES;40110;40;LANDES;1
40303;;Solfrino;SOLFERINO;40210;40;LANDES;1
40304;;Soorts-Hossegor;SOORTS-HOSSEGOR;40150;40;LANDES;1
40305;;Sorbets;SORBETS;40320;40;LANDES;1
40306;;Sorde-l'Abbaye;SORDE-L'ABBAYE;40300;40;LANDES;1
40307;;Sore;SORE;40430;40;LANDES;1
40308;;Sort-en-Chalosse;SORT-EN-CHALOSSE;40180;40;LANDES;1
40309;;Souprosse;SOUPROSSE;40250;40;LANDES;1
40310;;Soustons;SOUSTONS;40140;40;LANDES;1
40311;;Taller;TALLER;40260;40;LANDES;1
40312;;Tarnos;TARNOS;40220;40;LANDES;1
40313;;Tartas;TARTAS;40400;40;LANDES;1
40314;;Tercis-les-Bains;TERCIS-LES-BAINS;40180;40;LANDES;1
40315;;Tthieu;TETHIEU;40990;40;LANDES;1
40316;;Tilh;TILH;40360;40;LANDES;1
40317;;Tosse;TOSSE;40230;40;LANDES;1
40318;;Toulouzette;TOULOUZETTE;40250;40;LANDES;1
40319;;Trensacq;TRENSACQ;40630;40;LANDES;1
40320;;Uchacq-et-Parentis;UCHACQ-ET-PARENTIS;40090;40;LANDES;1
40321;;Urgons;URGONS;40320;40;LANDES;1
40322;;Uza;UZA;40170;40;LANDES;1
40323;;Vert;VERT;40420;40;LANDES;1
40324;;Vicq-d'Auribat;VICQ-D'AURIBAT;40380;40;LANDES;1
40325;;Vielle-Tursan;VIELLE-TURSAN;40320;40;LANDES;1
40326;;Vielle-Saint-Girons;VIELLE-SAINT-GIRONS;40560;40;LANDES;1
40327;;Vielle-Soubiran;VIELLE-SOUBIRAN;40240;40;LANDES;1
40328;;Vieux-Boucau-les-Bains;VIEUX-BOUCAU-LES-BAINS;40480;40;LANDES;1
40329;Le;Vignau;VIGNAU;40270;40;LANDES;1
40330;;Villenave;VILLENAVE;40110;40;LANDES;1
40331;;Villeneuve-de-Marsan;VILLENEUVE-DE-MARSAN;40190;40;LANDES;1
40332;;Ychoux;YCHOUX;40160;40;LANDES;1
40333;;Ygos-Saint-Saturnin;YGOS-SAINT-SATURNIN;40110;40;LANDES;1
40334;;Yzosse;YZOSSE;40180;40;LANDES;1
41001;;Ambloy;AMBLOY;41310;41;LOIR ET CHER;1
41002;;Ang;ANGE;41400;41;LOIR ET CHER;1
41003;;Areines;AREINES;41100;41;LOIR ET CHER;1
41004;;Artins;ARTINS;41800;41;LOIR ET CHER;1
41005;;Arville;ARVILLE;41170;41;LOIR ET CHER;1
41006;;Autainville;AUTAINVILLE;41240;41;LOIR ET CHER;1
41007;;Authon;AUTHON;41310;41;LOIR ET CHER;1
41008;;Avaray;AVARAY;41500;41;LOIR ET CHER;1
41009;;Averdon;AVERDON;41330;41;LOIR ET CHER;1
41010;;Az;AZE;41100;41;LOIR ET CHER;1
41011;;Baigneaux;BAIGNEAUX;41290;41;LOIR ET CHER;1
41012;;Baillou;BAILLOU;41170;41;LOIR ET CHER;1
41013;;Bauzy;BAUZY;41250;41;LOIR ET CHER;1
41014;;Beauchne;BEAUCHENE;41170;41;LOIR ET CHER;1
41015;;Beauvilliers;BEAUVILLIERS;41290;41;LOIR ET CHER;1
41016;;Billy;BILLY;41130;41;LOIR ET CHER;1
41017;;Binas;BINAS;41240;41;LOIR ET CHER;1
41018;;Blois;BLOIS;41000;41;LOIR ET CHER;1
41019;;Boisseau;BOISSEAU;41290;41;LOIR ET CHER;1
41020;;Bonneveau;BONNEVEAU;41800;41;LOIR ET CHER;1
41022;;Bouffry;BOUFFRY;41270;41;LOIR ET CHER;1
41023;;Bourr;BOURRE;41400;41;LOIR ET CHER;1
41024;;Boursay;BOURSAY;41270;41;LOIR ET CHER;1
41025;;Bracieux;BRACIEUX;41250;41;LOIR ET CHER;1
41026;;Brvainville;BREVAINVILLE;41160;41;LOIR ET CHER;1
41027;;Briou;BRIOU;41370;41;LOIR ET CHER;1
41028;;Busloup;BUSLOUP;41160;41;LOIR ET CHER;1
41029;;Cand-sur-Beuvron;CANDE-SUR-BEUVRON;41120;41;LOIR ET CHER;1
41030;;Cell;CELLE;41360;41;LOIR ET CHER;1
41031;;Cellettes;CELLETTES;41120;41;LOIR ET CHER;1
41032;;Chailles;CHAILLES;41120;41;LOIR ET CHER;1
41033;;Chambon-sur-Cisse;CHAMBON-SUR-CISSE;41190;41;LOIR ET CHER;1
41034;;Chambord;CHAMBORD;41250;41;LOIR ET CHER;1
41035;;Champigny-en-Beauce;CHAMPIGNY-EN-BEAUCE;41330;41;LOIR ET CHER;1
41036;;Chaon;CHAON;41600;41;LOIR ET CHER;1
41037;La;Chapelle-Enchrie;CHAPELLE-ENCHERIE;41290;41;LOIR ET CHER;1
41038;La;Chapelle-Montmartin;CHAPELLE-MONTMARTIN;41320;41;LOIR ET CHER;1
41039;La;Chapelle-Saint-Martin-en-Plaine;CHAPELLE-SAINT-MARTIN-EN-PLAINE;41500;41;LOIR ET CHER;1
41040;La;Chapelle-Vendmoise;CHAPELLE-VENDOMOISE;41330;41;LOIR ET CHER;1
41041;La;Chapelle-Vicomtesse;CHAPELLE-VICOMTESSE;41270;41;LOIR ET CHER;1
41042;;Chteauvieux;CHATEAUVIEUX;41110;41;LOIR ET CHER;1
41043;;Chtillon-sur-Cher;CHATILLON-SUR-CHER;41130;41;LOIR ET CHER;1
41044;;Chtres-sur-Cher;CHATRES-SUR-CHER;41320;41;LOIR ET CHER;1
41045;;Chaumont-sur-Loire;CHAUMONT-SUR-LOIRE;41150;41;LOIR ET CHER;1
41046;;Chaumont-sur-Tharonne;CHAUMONT-SUR-THARONNE;41600;41;LOIR ET CHER;1
41047;La;Chausse-Saint-Victor;CHAUSSEE-SAINT-VICTOR;41260;41;LOIR ET CHER;1
41048;;Chauvigny-du-Perche;CHAUVIGNY-DU-PERCHE;41270;41;LOIR ET CHER;1
41049;;Chmery;CHEMERY;41700;41;LOIR ET CHER;1
41050;;Cheverny;CHEVERNY;41700;41;LOIR ET CHER;1
41051;;Chissay-en-Touraine;CHISSAY-EN-TOURAINE;41400;41;LOIR ET CHER;1
41052;;Chitenay;CHITENAY;41120;41;LOIR ET CHER;1
41053;;Choue;CHOUE;41170;41;LOIR ET CHER;1
41054;;Choussy;CHOUSSY;41700;41;LOIR ET CHER;1
41055;;Chouzy-sur-Cisse;CHOUZY-SUR-CISSE;41150;41;LOIR ET CHER;1
41056;La;Colombe;COLOMBE;41160;41;LOIR ET CHER;1
41057;;Conan;CONAN;41290;41;LOIR ET CHER;1
41058;;Concriers;CONCRIERS;41370;41;LOIR ET CHER;1
41059;;Contres;CONTRES;41700;41;LOIR ET CHER;1
41060;;Cormenon;CORMENON;41170;41;LOIR ET CHER;1
41061;;Cormeray;CORMERAY;41120;41;LOIR ET CHER;1
41062;;Couddes;COUDDES;41700;41;LOIR ET CHER;1
41063;;Couffy;COUFFY;41110;41;LOIR ET CHER;1
41064;;Coulanges;COULANGES;41150;41;LOIR ET CHER;1
41065;;Coulommiers-la-Tour;COULOMMIERS-LA-TOUR;41100;41;LOIR ET CHER;1
41066;;Courbouzon;COURBOUZON;41500;41;LOIR ET CHER;1
41067;;Cour-Cheverny;COUR-CHEVERNY;41700;41;LOIR ET CHER;1
41068;;Courmemin;COURMEMIN;41230;41;LOIR ET CHER;1
41069;;Cour-sur-Loire;COUR-SUR-LOIRE;41500;41;LOIR ET CHER;1
41070;;Couture-sur-Loir;COUTURE-SUR-LOIR;41800;41;LOIR ET CHER;1
41071;;Crouy-sur-Cosson;CROUY-SUR-COSSON;41220;41;LOIR ET CHER;1
41072;;Crucheray;CRUCHERAY;41100;41;LOIR ET CHER;1
41073;;Danz;DANZE;41160;41;LOIR ET CHER;1
41074;;Dhuizon;DHUIZON;41220;41;LOIR ET CHER;1
41075;;Drou;DROUE;41270;41;LOIR ET CHER;1
41077;;piais;EPIAIS;41290;41;LOIR ET CHER;1
41078;;puisay;EPUISAY;41360;41;LOIR ET CHER;1
41079;Les;Essarts;ESSARTS;41800;41;LOIR ET CHER;1
41080;;Faverolles-sur-Cher;FAVEROLLES-SUR-CHER;41400;41;LOIR ET CHER;1
41081;;Faye;FAYE;41100;41;LOIR ET CHER;1
41082;;Feings;FEINGS;41120;41;LOIR ET CHER;1
41083;La;Fert-Beauharnais;FERTE-BEAUHARNAIS;41210;41;LOIR ET CHER;1
41084;La;Fert-Imbault;FERTE-IMBAULT;41300;41;LOIR ET CHER;1
41085;La;Fert-Saint-Cyr;FERTE-SAINT-CYR;41220;41;LOIR ET CHER;1
41086;;Fontaines-en-Sologne;FONTAINES-EN-SOLOGNE;41250;41;LOIR ET CHER;1
41087;;Fontaine-les-Coteaux;FONTAINE-LES-COTEAUX;41800;41;LOIR ET CHER;1
41088;;Fontaine-Raoul;FONTAINE-RAOUL;41270;41;LOIR ET CHER;1
41089;La;Fontenelle;FONTENELLE;41270;41;LOIR ET CHER;1
41090;;Fortan;FORTAN;41360;41;LOIR ET CHER;1
41091;;Foss;FOSSE;41330;41;LOIR ET CHER;1
41092;;Fougres-sur-Bivre;FOUGERES-SUR-BIEVRE;41120;41;LOIR ET CHER;1
41093;;Franay;FRANCAY;41190;41;LOIR ET CHER;1
41094;;Fresnes;FRESNES;41700;41;LOIR ET CHER;1
41095;;Frteval;FRETEVAL;41160;41;LOIR ET CHER;1
41096;Le;Gault-Perche;GAULT-PERCHE;41270;41;LOIR ET CHER;1
41097;;Givres;GIEVRES;41130;41;LOIR ET CHER;1
41098;;Gombergean;GOMBERGEAN;41310;41;LOIR ET CHER;1
41099;;Gy-en-Sologne;GY-EN-SOLOGNE;41230;41;LOIR ET CHER;1
41100;Les;Hayes;HAYES;41800;41;LOIR ET CHER;1
41101;;Herbault;HERBAULT;41190;41;LOIR ET CHER;1
41102;;Houssay;HOUSSAY;41800;41;LOIR ET CHER;1
41103;;Huisseau-en-Beauce;HUISSEAU-EN-BEAUCE;41310;41;LOIR ET CHER;1
41104;;Huisseau-sur-Cosson;HUISSEAU-SUR-COSSON;41350;41;LOIR ET CHER;1
41105;;Josnes;JOSNES;41370;41;LOIR ET CHER;1
41106;;Lamotte-Beuvron;LAMOTTE-BEUVRON;41600;41;LOIR ET CHER;1
41107;;Lanc;LANCE;41310;41;LOIR ET CHER;1
41108;;Lancme;LANCOME;41190;41;LOIR ET CHER;1
41109;;Landes-le-Gaulois;LANDES-LE-GAULOIS;41190;41;LOIR ET CHER;1
41110;;Langon;LANGON;41320;41;LOIR ET CHER;1
41112;;Lassay-sur-Croisne;LASSAY-SUR-CROISNE;41230;41;LOIR ET CHER;1
41113;;Lavardin;LAVARDIN;41800;41;LOIR ET CHER;1
41114;;Lestiou;LESTIOU;41500;41;LOIR ET CHER;1
41115;;Lignires;LIGNIERES;41160;41;LOIR ET CHER;1
41116;;Lisle;LISLE;41100;41;LOIR ET CHER;1
41118;;Loreux;LOREUX;41200;41;LOIR ET CHER;1
41119;;Lorges;LORGES;41370;41;LOIR ET CHER;1
41120;;Lunay;LUNAY;41360;41;LOIR ET CHER;1
41121;La;Madeleine-Villefrouin;MADELEINE-VILLEFROUIN;41370;41;LOIR ET CHER;1
41122;;Maray;MARAY;41320;41;LOIR ET CHER;1
41123;;Marchenoir;MARCHENOIR;41370;41;LOIR ET CHER;1
41124;;Marcilly-en-Beauce;MARCILLY-EN-BEAUCE;41100;41;LOIR ET CHER;1
41125;;Marcilly-en-Gault;MARCILLY-EN-GAULT;41210;41;LOIR ET CHER;1
41126;;Mareuil-sur-Cher;MAREUIL-SUR-CHER;41110;41;LOIR ET CHER;1
41127;La;Marolle-en-Sologne;MAROLLE-EN-SOLOGNE;41210;41;LOIR ET CHER;1
41128;;Marolles;MAROLLES;41330;41;LOIR ET CHER;1
41129;;Maslives;MASLIVES;41250;41;LOIR ET CHER;1
41130;;Maves;MAVES;41500;41;LOIR ET CHER;1
41131;;Mazang;MAZANGE;41100;41;LOIR ET CHER;1
41132;;Mhers;MEHERS;41140;41;LOIR ET CHER;1
41133;;Membrolles;MEMBROLLES;41240;41;LOIR ET CHER;1
41134;;Menars;MENARS;41500;41;LOIR ET CHER;1
41135;;Mennetou-sur-Cher;MENNETOU-SUR-CHER;41320;41;LOIR ET CHER;1
41136;;Mer;MER;41500;41;LOIR ET CHER;1
41137;;Mesland;MESLAND;41150;41;LOIR ET CHER;1
41138;;Meslay;MESLAY;41100;41;LOIR ET CHER;1
41139;;Meusnes;MEUSNES;41130;41;LOIR ET CHER;1
41140;;Millanay;MILLANCAY;41200;41;LOIR ET CHER;1
41141;;Moisy;MOISY;41160;41;LOIR ET CHER;1
41142;;Molineuf;MOLINEUF;41190;41;LOIR ET CHER;1
41143;;Mondoubleau;MONDOUBLEAU;41170;41;LOIR ET CHER;1
41144;;Monteaux;MONTEAUX;41150;41;LOIR ET CHER;1
41145;;Monthou-sur-Bivre;MONTHOU-SUR-BIEVRE;41120;41;LOIR ET CHER;1
41146;;Monthou-sur-Cher;MONTHOU-SUR-CHER;41400;41;LOIR ET CHER;1
41147;Les;Montils;MONTILS;41120;41;LOIR ET CHER;1
41148;;Montlivault;MONTLIVAULT;41350;41;LOIR ET CHER;1
41149;;Montoire-sur-le-Loir;MONTOIRE-SUR-LE-LOIR;41800;41;LOIR ET CHER;1
41150;;Mont-prs-Chambord;MONT-PRES-CHAMBORD;41250;41;LOIR ET CHER;1
41151;;Montrichard;MONTRICHARD;41400;41;LOIR ET CHER;1
41152;;Montrieux-en-Sologne;MONTRIEUX-EN-SOLOGNE;41210;41;LOIR ET CHER;1
41153;;Montrouveau;MONTROUVEAU;41800;41;LOIR ET CHER;1
41154;;More;MOREE;41160;41;LOIR ET CHER;1
41155;;Muides-sur-Loire;MUIDES-SUR-LOIRE;41500;41;LOIR ET CHER;1
41156;;Mulsans;MULSANS;41500;41;LOIR ET CHER;1
41157;;Mur-de-Sologne;MUR-DE-SOLOGNE;41230;41;LOIR ET CHER;1
41158;;Naveil;NAVEIL;41100;41;LOIR ET CHER;1
41159;;Neung-sur-Beuvron;NEUNG-SUR-BEUVRON;41210;41;LOIR ET CHER;1
41160;;Neuvy;NEUVY;41250;41;LOIR ET CHER;1
41161;;Nouan-le-Fuzelier;NOUAN-LE-FUZELIER;41600;41;LOIR ET CHER;1
41163;;Nourray;NOURRAY;41310;41;LOIR ET CHER;1
41164;;Noyers-sur-Cher;NOYERS-SUR-CHER;41140;41;LOIR ET CHER;1
41165;;Oigny;OIGNY;41170;41;LOIR ET CHER;1
41166;;Oisly;OISLY;41700;41;LOIR ET CHER;1
41167;;Onzain;ONZAIN;41150;41;LOIR ET CHER;1
41168;;Oray;ORCAY;41300;41;LOIR ET CHER;1
41169;;Orchaise;ORCHAISE;41190;41;LOIR ET CHER;1
41170;;Ouchamps;OUCHAMPS;41120;41;LOIR ET CHER;1
41171;;Oucques;OUCQUES;41290;41;LOIR ET CHER;1
41172;;Ouzouer-le-Doyen;OUZOUER-LE-DOYEN;41160;41;LOIR ET CHER;1
41173;;Ouzouer-le-March;OUZOUER-LE-MARCHE;41240;41;LOIR ET CHER;1
41174;;Prigny;PERIGNY;41100;41;LOIR ET CHER;1
41175;;Pezou;PEZOU;41100;41;LOIR ET CHER;1
41176;;Pierrefitte-sur-Sauldre;PIERREFITTE-SUR-SAULDRE;41300;41;LOIR ET CHER;1
41177;Le;Plessis-Dorin;PLESSIS-DORIN;41170;41;LOIR ET CHER;1
41178;Le;Plessis-l'chelle;PLESSIS-L'ECHELLE;41370;41;LOIR ET CHER;1
41179;Le;Poislay;POISLAY;41270;41;LOIR ET CHER;1
41180;;Pontlevoy;PONTLEVOY;41400;41;LOIR ET CHER;1
41181;;Pouill;POUILLE;41110;41;LOIR ET CHER;1
41182;;Pray;PRAY;41190;41;LOIR ET CHER;1
41183;;Prnouvellon;PRENOUVELLON;41240;41;LOIR ET CHER;1
41184;;Prunay-Cassereau;PRUNAY-CASSEREAU;41310;41;LOIR ET CHER;1
41185;;Pruniers-en-Sologne;PRUNIERS-EN-SOLOGNE;41200;41;LOIR ET CHER;1
41186;;Rahart;RAHART;41160;41;LOIR ET CHER;1
41187;;Renay;RENAY;41100;41;LOIR ET CHER;1
41188;;Rhodon;RHODON;41290;41;LOIR ET CHER;1
41189;;Rilly-sur-Loire;RILLY-SUR-LOIRE;41150;41;LOIR ET CHER;1
41190;;Roc;ROCE;41100;41;LOIR ET CHER;1
41191;;Roches;ROCHES;41370;41;LOIR ET CHER;1
41192;Les;Roches-l'vque;ROCHES-L'EVEQUE;41800;41;LOIR ET CHER;1
41193;;Romilly;ROMILLY;41270;41;LOIR ET CHER;1
41194;;Romorantin-Lanthenay;ROMORANTIN-LANTHENAY;41200;41;LOIR ET CHER;1
41195;;Rougeou;ROUGEOU;41230;41;LOIR ET CHER;1
41196;;Ruan-sur-Egvonne;RUAN-SUR-EGVONNE;41270;41;LOIR ET CHER;1
41197;;Saint-Agil;SAINT-AGIL;41170;41;LOIR ET CHER;1
41198;;Saint-Aignan;SAINT-AIGNAN;41110;41;LOIR ET CHER;1
41199;;Saint-Amand-Longpr;SAINT-AMAND-LONGPRE;41310;41;LOIR ET CHER;1
41200;;Sainte-Anne;SAINTE-ANNE;41100;41;LOIR ET CHER;1
41201;;Saint-Arnoult;SAINT-ARNOULT;41800;41;LOIR ET CHER;1
41202;;Saint-Avit;SAINT-AVIT;41170;41;LOIR ET CHER;1
41203;;Saint-Bohaire;SAINT-BOHAIRE;41330;41;LOIR ET CHER;1
41204;;Saint-Claude-de-Diray;SAINT-CLAUDE-DE-DIRAY;41350;41;LOIR ET CHER;1
41205;;Saint-Cyr-du-Gault;SAINT-CYR-DU-GAULT;41190;41;LOIR ET CHER;1
41206;;Saint-Denis-sur-Loire;SAINT-DENIS-SUR-LOIRE;41000;41;LOIR ET CHER;1
41207;;Saint-Dy-sur-Loire;SAINT-DYE-SUR-LOIRE;41500;41;LOIR ET CHER;1
41208;;Saint-tienne-des-Gurets;SAINT-ETIENNE-DES-GUERETS;41190;41;LOIR ET CHER;1
41209;;Saint-Firmin-des-Prs;SAINT-FIRMIN-DES-PRES;41100;41;LOIR ET CHER;1
41210;;Sainte-Gemmes;SAINTE-GEMMES;41290;41;LOIR ET CHER;1
41211;;Saint-Georges-sur-Cher;SAINT-GEORGES-SUR-CHER;41400;41;LOIR ET CHER;1
41212;;Saint-Gervais-la-Fort;SAINT-GERVAIS-LA-FORET;41350;41;LOIR ET CHER;1
41213;;Saint-Gourgon;SAINT-GOURGON;41310;41;LOIR ET CHER;1
41214;;Saint-Hilaire-la-Gravelle;SAINT-HILAIRE-LA-GRAVELLE;41160;41;LOIR ET CHER;1
41215;;Saint-Jacques-des-Gurets;SAINT-JACQUES-DES-GUERETS;41800;41;LOIR ET CHER;1
41216;;Saint-Jean-Froidmentel;SAINT-JEAN-FROIDMENTEL;41160;41;LOIR ET CHER;1
41217;;Saint-Julien-de-Chdon;SAINT-JULIEN-DE-CHEDON;41400;41;LOIR ET CHER;1
41218;;Saint-Julien-sur-Cher;SAINT-JULIEN-SUR-CHER;41320;41;LOIR ET CHER;1
41219;;Saint-Laurent-des-Bois;SAINT-LAURENT-DES-BOIS;41240;41;LOIR ET CHER;1
41220;;Saint-Laurent-Nouan;SAINT-LAURENT-NOUAN;41220;41;LOIR ET CHER;1
41221;;Saint-Lonard-en-Beauce;SAINT-LEONARD-EN-BEAUCE;41370;41;LOIR ET CHER;1
41222;;Saint-Loup;SAINT-LOUP;41320;41;LOIR ET CHER;1
41223;;Saint-Lubin-en-Vergonnois;SAINT-LUBIN-EN-VERGONNOIS;41190;41;LOIR ET CHER;1
41224;;Saint-Marc-du-Cor;SAINT-MARC-DU-COR;41170;41;LOIR ET CHER;1
41225;;Saint-Martin-des-Bois;SAINT-MARTIN-DES-BOIS;41800;41;LOIR ET CHER;1
41226;;Saint-Ouen;SAINT-OUEN;41100;41;LOIR ET CHER;1
41228;;Saint-Rimay;SAINT-RIMAY;41800;41;LOIR ET CHER;1
41229;;Saint-Romain-sur-Cher;SAINT-ROMAIN-SUR-CHER;41140;41;LOIR ET CHER;1
41230;;Saint-Sulpice-de-Pommeray;SAINT-SULPICE-DE-POMMERAY;41000;41;LOIR ET CHER;1
41231;;Saint-Vitre;SAINT-VIATRE;41210;41;LOIR ET CHER;1
41232;;Salbris;SALBRIS;41300;41;LOIR ET CHER;1
41233;;Sambin;SAMBIN;41120;41;LOIR ET CHER;1
41234;;Santenay;SANTENAY;41190;41;LOIR ET CHER;1
41235;;Sarg-sur-Braye;SARGE-SUR-BRAYE;41170;41;LOIR ET CHER;1
41236;;Sasnires;SASNIERES;41310;41;LOIR ET CHER;1
41237;;Sassay;SASSAY;41700;41;LOIR ET CHER;1
41238;;Savigny-sur-Braye;SAVIGNY-SUR-BRAYE;41360;41;LOIR ET CHER;1
41239;;Seigy;SEIGY;41110;41;LOIR ET CHER;1
41240;;Seillac;SEILLAC;41150;41;LOIR ET CHER;1
41241;;Selles-Saint-Denis;SELLES-SAINT-DENIS;41300;41;LOIR ET CHER;1
41242;;Selles-sur-Cher;SELLES-SUR-CHER;41130;41;LOIR ET CHER;1
41243;;Selommes;SELOMMES;41100;41;LOIR ET CHER;1
41244;;Semerville;SEMERVILLE;41160;41;LOIR ET CHER;1
41245;;Sris;SERIS;41500;41;LOIR ET CHER;1
41246;;Seur;SEUR;41120;41;LOIR ET CHER;1
41247;;Soings-en-Sologne;SOINGS-EN-SOLOGNE;41230;41;LOIR ET CHER;1
41248;;Souday;SOUDAY;41170;41;LOIR ET CHER;1
41249;;Souesmes;SOUESMES;41300;41;LOIR ET CHER;1
41250;;Soug;SOUGE;41800;41;LOIR ET CHER;1
41251;;Souvigny-en-Sologne;SOUVIGNY-EN-SOLOGNE;41600;41;LOIR ET CHER;1
41252;;Suvres;SUEVRES;41500;41;LOIR ET CHER;1
41253;;Talcy;TALCY;41370;41;LOIR ET CHER;1
41254;Le;Temple;TEMPLE;41170;41;LOIR ET CHER;1
41255;;Ternay;TERNAY;41800;41;LOIR ET CHER;1
41256;;Theillay;THEILLAY;41300;41;LOIR ET CHER;1
41257;;Thenay;THENAY;41400;41;LOIR ET CHER;1
41258;;Thse;THESEE;41140;41;LOIR ET CHER;1
41259;;Thor-la-Rochette;THORE-LA-ROCHETTE;41100;41;LOIR ET CHER;1
41260;;Thoury;THOURY;41220;41;LOIR ET CHER;1
41261;;Tourailles;TOURAILLES;41190;41;LOIR ET CHER;1
41262;;Tour-en-Sologne;TOUR-EN-SOLOGNE;41250;41;LOIR ET CHER;1
41263;;Trhet;TREHET;41800;41;LOIR ET CHER;1
41264;;Tripleville;TRIPLEVILLE;41240;41;LOIR ET CHER;1
41265;;Troo;TROO;41800;41;LOIR ET CHER;1
41266;;Valaire;VALAIRE;41120;41;LOIR ET CHER;1
41267;;Vallires-les-Grandes;VALLIERES-LES-GRANDES;41400;41;LOIR ET CHER;1
41268;;Veilleins;VEILLEINS;41230;41;LOIR ET CHER;1
41269;;Vendme;VENDOME;41100;41;LOIR ET CHER;1
41270;;Verdes;VERDES;41240;41;LOIR ET CHER;1
41271;;Vernou-en-Sologne;VERNOU-EN-SOLOGNE;41230;41;LOIR ET CHER;1
41272;;Veuves;VEUVES;41150;41;LOIR ET CHER;1
41273;;Vievy-le-Ray;VIEVY-LE-RAYE;41290;41;LOIR ET CHER;1
41274;;Villavard;VILLAVARD;41800;41;LOIR ET CHER;1
41275;La;Ville-aux-Clercs;VILLE-AUX-CLERCS;41160;41;LOIR ET CHER;1
41276;;Villebarou;VILLEBAROU;41000;41;LOIR ET CHER;1
41277;;Villebout;VILLEBOUT;41270;41;LOIR ET CHER;1
41278;;Villechauve;VILLECHAUVE;41310;41;LOIR ET CHER;1
41279;;Villedieu-le-Chteau;VILLEDIEU-LE-CHATEAU;41800;41;LOIR ET CHER;1
41280;;Villefranche-sur-Cher;VILLEFRANCHE-SUR-CHER;41200;41;LOIR ET CHER;1
41281;;Villefrancur;VILLEFRANCOEUR;41330;41;LOIR ET CHER;1
41282;;Villeherviers;VILLEHERVIERS;41200;41;LOIR ET CHER;1
41283;;Villemardy;VILLEMARDY;41100;41;LOIR ET CHER;1
41284;;Villeneuve-Frouville;VILLENEUVE-FROUVILLE;41290;41;LOIR ET CHER;1
41285;;Villeny;VILLENY;41220;41;LOIR ET CHER;1
41286;;Villeporcher;VILLEPORCHER;41310;41;LOIR ET CHER;1
41287;;Villerable;VILLERABLE;41100;41;LOIR ET CHER;1
41288;;Villerbon;VILLERBON;41000;41;LOIR ET CHER;1
41289;;Villermain;VILLERMAIN;41240;41;LOIR ET CHER;1
41290;;Villeromain;VILLEROMAIN;41100;41;LOIR ET CHER;1
41291;;Villetrun;VILLETRUN;41100;41;LOIR ET CHER;1
41292;;Villexanton;VILLEXANTON;41500;41;LOIR ET CHER;1
41293;;Villiersfaux;VILLIERSFAUX;41100;41;LOIR ET CHER;1
41294;;Villiers-sur-Loir;VILLIERS-SUR-LOIR;41100;41;LOIR ET CHER;1
41295;;Vineuil;VINEUIL;41350;41;LOIR ET CHER;1
41296;;Vouzon;VOUZON;41600;41;LOIR ET CHER;1
41297;;Yvoy-le-Marron;YVOY-LE-MARRON;41600;41;LOIR ET CHER;1
42001;;Abon;ABOEN;42380;42;LOIRE;1
42002;;Ailleux;AILLEUX;42130;42;LOIRE;1
42003;;Ambierle;AMBIERLE;42820;42;LOIRE;1
42004;;Amions;AMIONS;42260;42;LOIRE;1
42005;;Andrzieux-Bouthon;ANDREZIEUX-BOUTHEON;42160;42;LOIRE;1
42006;;Apinac;APINAC;42550;42;LOIRE;1
42007;;Arcinges;ARCINGES;42460;42;LOIRE;1
42008;;Arcon;ARCON;42370;42;LOIRE;1
42009;;Arthun;ARTHUN;42130;42;LOIRE;1
42010;;Aveizieux;AVEIZIEUX;42330;42;LOIRE;1
42011;;Balbigny;BALBIGNY;42510;42;LOIRE;1
42012;;Bard;BARD;42600;42;LOIRE;1
42013;;Bellegarde-en-Forez;BELLEGARDE-EN-FOREZ;42210;42;LOIRE;1
42014;;Belleroche;BELLEROCHE;42670;42;LOIRE;1
42015;;Belmont-de-la-Loire;BELMONT-DE-LA-LOIRE;42670;42;LOIRE;1
42016;La;Bnisson-Dieu;BENISSON-DIEU;42720;42;LOIRE;1
42017;Le;Bessat;BESSAT;42660;42;LOIRE;1
42018;;Bessey;BESSEY;42520;42;LOIRE;1
42019;;Bon;BOEN;42130;42;LOIRE;1
42020;;Boisset-ls-Montrond;BOISSET-LES-MONTROND;42210;42;LOIRE;1
42021;;Boisset-Saint-Priest;BOISSET-SAINT-PRIEST;42560;42;LOIRE;1
42022;;Bonson;BONSON;42160;42;LOIRE;1
42023;;Bourg-Argental;BOURG-ARGENTAL;42220;42;LOIRE;1
42025;;Boyer;BOYER;42460;42;LOIRE;1
42026;;Briennon;BRIENNON;42720;42;LOIRE;1
42027;;Bully;BULLY;42260;42;LOIRE;1
42028;;Burdignes;BURDIGNES;42220;42;LOIRE;1
42029;;Bussires;BUSSIERES;42510;42;LOIRE;1
42030;;Bussy-Albieux;BUSSY-ALBIEUX;42260;42;LOIRE;1
42031;;Caloire;CALOIRE;42240;42;LOIRE;1
42032;;Cellieu;CELLIEU;42320;42;LOIRE;1
42033;Le;Cergne;CERGNE;42460;42;LOIRE;1
42034;;Cervires;CERVIERES;42440;42;LOIRE;1
42035;;Cezay;CEZAY;42130;42;LOIRE;1
42036;;Chagnon;CHAGNON;42800;42;LOIRE;1
42037;;Chalain-d'Uzore;CHALAIN-D'UZORE;42600;42;LOIRE;1
42038;;Chalain-le-Comtal;CHALAIN-LE-COMTAL;42600;42;LOIRE;1
42039;;Chalmazel;CHALMAZEL;42920;42;LOIRE;1
42040;La;Chamba;CHAMBA;42440;42;LOIRE;1
42041;;Chambon;CHAMBEON;42110;42;LOIRE;1
42042;;Chambles;CHAMBLES;42170;42;LOIRE;1
42043;;Chambuf;CHAMBOEUF;42330;42;LOIRE;1
42044;Le;Chambon-Feugerolles;CHAMBON-FEUGEROLLES;42500;42;LOIRE;1
42045;La;Chambonie;CHAMBONIE;42440;42;LOIRE;1
42046;;Champdieu;CHAMPDIEU;42600;42;LOIRE;1
42047;;Champoly;CHAMPOLY;42430;42;LOIRE;1
42048;;Chandon;CHANDON;42190;42;LOIRE;1
42049;;Changy;CHANGY;42310;42;LOIRE;1
42050;La;Chapelle-en-Lafaye;CHAPELLE-EN-LAFAYE;42380;42;LOIRE;1
42051;La;Chapelle-Villars;CHAPELLE-VILLARS;42410;42;LOIRE;1
42052;;Charlieu;CHARLIEU;42190;42;LOIRE;1
42053;;Chteauneuf;CHATEAUNEUF;42800;42;LOIRE;1
42054;;Chtelneuf;CHATELNEUF;42940;42;LOIRE;1
42055;;Chtelus;CHATELUS;42140;42;LOIRE;1
42056;;Chavanay;CHAVANAY;42410;42;LOIRE;1
42058;;Chazelles-sur-Lavieu;CHAZELLES-SUR-LAVIEU;42560;42;LOIRE;1
42059;;Chazelles-sur-Lyon;CHAZELLES-SUR-LYON;42140;42;LOIRE;1
42060;;Chenereilles;CHENEREILLES;42560;42;LOIRE;1
42061;;Cherier;CHERIER;42430;42;LOIRE;1
42062;;Chevrires;CHEVRIERES;42140;42;LOIRE;1
42063;;Chirassimont;CHIRASSIMONT;42114;42;LOIRE;1
42064;;Chuyer;CHUYER;42410;42;LOIRE;1
42065;;Civens;CIVENS;42110;42;LOIRE;1
42066;;Clepp;CLEPPE;42110;42;LOIRE;1
42067;;Colombier;COLOMBIER;42220;42;LOIRE;1
42068;;Combre;COMBRE;42840;42;LOIRE;1
42069;;Commelle-Vernay;COMMELLE-VERNAY;42120;42;LOIRE;1
42070;;Cordelle;CORDELLE;42123;42;LOIRE;1
42071;Le;Coteau;COTEAU;42120;42;LOIRE;1
42072;La;Cte-en-Couzan;COTE-EN-COUZAN;42111;42;LOIRE;1
42073;;Cottance;COTTANCE;42360;42;LOIRE;1
42074;;Coutouvre;COUTOUVRE;42460;42;LOIRE;1
42075;;Craintilleux;CRAINTILLEUX;42210;42;LOIRE;1
42076;;Cremeaux;CREMEAUX;42260;42;LOIRE;1
42077;;Croizet-sur-Gand;CROIZET-SUR-GAND;42540;42;LOIRE;1
42078;Le;Crozet;CROZET;42310;42;LOIRE;1
42079;;Cuinzier;CUINZIER;42460;42;LOIRE;1
42081;;Cuzieu;CUZIEU;42330;42;LOIRE;1
42082;;Danc;DANCE;42260;42;LOIRE;1
42083;;Dargoire;DARGOIRE;42800;42;LOIRE;1
42084;;Dbats-Rivire-d'Orpra;DEBATS-RIVIERE-D'ORPRA;42130;42;LOIRE;1
42085;;Doizieux;DOIZIEUX;42740;42;LOIRE;1
42086;;coche;ECOCHE;42670;42;LOIRE;1
42087;;cotay-l'Olme;ECOTAY-L'OLME;42600;42;LOIRE;1
42088;;percieux-Saint-Paul;EPERCIEUX-SAINT-PAUL;42110;42;LOIRE;1
42089;;Essertines-en-Chtelneuf;ESSERTINES-EN-CHATELNEUF;42600;42;LOIRE;1
42090;;Essertines-en-Donzy;ESSERTINES-EN-DONZY;42360;42;LOIRE;1
42091;;Estivareilles;ESTIVAREILLES;42380;42;LOIRE;1
42092;L';trat;ETRAT;42580;42;LOIRE;1
42093;;Farnay;FARNAY;42320;42;LOIRE;1
42094;;Feurs;FEURS;42110;42;LOIRE;1
42095;;Firminy;FIRMINY;42700;42;LOIRE;1
42096;;Fontans;FONTANES;42140;42;LOIRE;1
42097;La;Fouillouse;FOUILLOUSE;42480;42;LOIRE;1
42098;;Fourneaux;FOURNEAUX;42470;42;LOIRE;1
42099;;Fraisses;FRAISSES;42490;42;LOIRE;1
42100;La;Gimond;GIMOND;42140;42;LOIRE;1
42101;;Graix;GRAIX;42220;42;LOIRE;1
42102;;Grammond;GRAMMOND;42140;42;LOIRE;1
42103;La;Grand-Croix;GRAND-CROIX;42320;42;LOIRE;1
42104;La;Gresle;GRESLE;42460;42;LOIRE;1
42105;;Grzieux-le-Fromental;GREZIEUX-LE-FROMENTAL;42600;42;LOIRE;1
42106;;Grzolles;GREZOLLES;42260;42;LOIRE;1
42107;;Gumires;GUMIERES;42560;42;LOIRE;1
42108;L';Hpital-le-Grand;HOPITAL-LE-GRAND;42210;42;LOIRE;1
42109;L';Hpital-sous-Rochefort;HOPITAL-SOUS-ROCHEFORT;42130;42;LOIRE;1
42110;L';Horme;HORME;42152;42;LOIRE;1
42112;;Jarnosse;JARNOSSE;42460;42;LOIRE;1
42113;;Jas;JAS;42110;42;LOIRE;1
42114;;Jeansagnire;JEANSAGNIERE;42920;42;LOIRE;1
42115;;Jonzieux;JONZIEUX;42660;42;LOIRE;1
42116;;Jur;JURE;42430;42;LOIRE;1
42117;;Lavieu;LAVIEU;42560;42;LOIRE;1
42118;;Lay;LAY;42470;42;LOIRE;1
42119;;Leigneux;LEIGNEUX;42130;42;LOIRE;1
42120;;Lentigny;LENTIGNY;42155;42;LOIRE;1
42121;;Lrigneux;LERIGNEUX;42600;42;LOIRE;1
42122;;Lzigneux;LEZIGNEUX;42600;42;LOIRE;1
42123;;Lorette;LORETTE;42420;42;LOIRE;1
42124;;Lup;LUPE;42520;42;LOIRE;1
42125;;Lur;LURE;42260;42;LOIRE;1
42126;;Luriecq;LURIECQ;42380;42;LOIRE;1
42127;;Mably;MABLY;42300;42;LOIRE;1
42128;;Machzal;MACHEZAL;42114;42;LOIRE;1
42129;;Maclas;MACLAS;42520;42;LOIRE;1
42130;;Magneux-Haute-Rive;MAGNEUX-HAUTE-RIVE;42600;42;LOIRE;1
42131;;Maizilly;MAIZILLY;42750;42;LOIRE;1
42132;;Malleval;MALLEVAL;42520;42;LOIRE;1
42133;;Marcenod;MARCENOD;42140;42;LOIRE;1
42134;;Marcilly-le-Chtel;MARCILLY-LE-CHATEL;42130;42;LOIRE;1
42135;;Marclopt;MARCLOPT;42210;42;LOIRE;1
42136;;Marcoux;MARCOUX;42130;42;LOIRE;1
42137;;Margerie-Chantagret;MARGERIE-CHANTAGRET;42560;42;LOIRE;1
42138;;Maringes;MARINGES;42140;42;LOIRE;1
42139;;Marlhes;MARLHES;42660;42;LOIRE;1
42140;;Marols;MAROLS;42560;42;LOIRE;1
42141;;Mars;MARS;42750;42;LOIRE;1
42142;;Merle-Leignec;MERLE-LEIGNEC;42380;42;LOIRE;1
42143;;Mizrieux;MIZERIEUX;42110;42;LOIRE;1
42145;;Montagny;MONTAGNY;42840;42;LOIRE;1
42146;;Montarcher;MONTARCHER;42380;42;LOIRE;1
42147;;Montbrison;MONTBRISON;42600;42;LOIRE;1
42148;;Montchal;MONTCHAL;42360;42;LOIRE;1
42149;;Montrond-les-Bains;MONTROND-LES-BAINS;42210;42;LOIRE;1
42150;;Montverdun;MONTVERDUN;42130;42;LOIRE;1
42151;;Mornand-en-Forez;MORNAND-EN-FOREZ;42600;42;LOIRE;1
42152;;Nandax;NANDAX;42720;42;LOIRE;1
42153;;Neaux;NEAUX;42470;42;LOIRE;1
42154;;Nronde;NERONDE;42510;42;LOIRE;1
42155;;Nervieux;NERVIEUX;42510;42;LOIRE;1
42156;;Neulise;NEULISE;42590;42;LOIRE;1
42157;;Noailly;NOAILLY;42640;42;LOIRE;1
42158;Les;Nos;NOES;42370;42;LOIRE;1
42159;;Noirtable;NOIRETABLE;42440;42;LOIRE;1
42160;;Nollieux;NOLLIEUX;42260;42;LOIRE;1
42161;;Notre-Dame-de-Boisset;NOTRE-DAME-DE-BOISSET;42120;42;LOIRE;1
42162;;Ouches;OUCHES;42155;42;LOIRE;1
42163;La;Pacaudire;PACAUDIERE;42310;42;LOIRE;1
42164;;Palogneux;PALOGNEUX;42890;42;LOIRE;1
42165;;Panissires;PANISSIERES;42360;42;LOIRE;1
42166;;Parigny;PARIGNY;42120;42;LOIRE;1
42167;;Pavezin;PAVEZIN;42410;42;LOIRE;1
42168;;Plussin;PELUSSIN;42410;42;LOIRE;1
42169;;Prigneux;PERIGNEUX;42380;42;LOIRE;1
42170;;Perreux;PERREUX;42120;42;LOIRE;1
42171;;Pinay;PINAY;42590;42;LOIRE;1
42172;;Planfoy;PLANFOY;42660;42;LOIRE;1
42173;;Pommiers;POMMIERS;42260;42;LOIRE;1
42174;;Poncins;PONCINS;42110;42;LOIRE;1
42175;;Pouilly-ls-Feurs;POUILLY-LES-FEURS;42110;42;LOIRE;1
42176;;Pouilly-les-Nonains;POUILLY-LES-NONAINS;42155;42;LOIRE;1
42177;;Pouilly-sous-Charlieu;POUILLY-SOUS-CHARLIEU;42720;42;LOIRE;1
42178;;Pradines;PRADINES;42630;42;LOIRE;1
42179;;Pralong;PRALONG;42600;42;LOIRE;1
42180;;Prcieux;PRECIEUX;42600;42;LOIRE;1
42181;;Rgny;REGNY;42630;42;LOIRE;1
42182;;Renaison;RENAISON;42370;42;LOIRE;1
42183;La;Ricamarie;RICAMARIE;42150;42;LOIRE;1
42184;;Riorges;RIORGES;42153;42;LOIRE;1
42185;;Rivas;RIVAS;42340;42;LOIRE;1
42186;;Rive-de-Gier;RIVE-DE-GIER;42800;42;LOIRE;1
42187;;Roanne;ROANNE;42300;42;LOIRE;1
42188;;Roche;ROCHE;42600;42;LOIRE;1
42189;;Roche-la-Molire;ROCHE-LA-MOLIERE;42230;42;LOIRE;1
42191;;Roisey;ROISEY;42520;42;LOIRE;1
42192;;Rozier-Ctes-d'Aurec;ROZIER-COTES-D'AUREC;42380;42;LOIRE;1
42193;;Rozier-en-Donzy;ROZIER-EN-DONZY;42810;42;LOIRE;1
42194;;Sail-les-Bains;SAIL-LES-BAINS;42310;42;LOIRE;1
42195;;Sail-sous-Couzan;SAIL-SOUS-COUZAN;42890;42;LOIRE;1
42196;;Sainte-Agathe-en-Donzy;SAINTE-AGATHE-EN-DONZY;42510;42;LOIRE;1
42197;;Sainte-Agathe-la-Bouteresse;SAINTE-AGATHE-LA-BOUTERESSE;42130;42;LOIRE;1
42198;;Saint-Alban-les-Eaux;SAINT-ALBAN-LES-EAUX;42370;42;LOIRE;1
42199;;Saint-Andr-d'Apchon;SAINT-ANDRE-D'APCHON;42370;42;LOIRE;1
42200;;Saint-Andr-le-Puy;SAINT-ANDRE-LE-PUY;42210;42;LOIRE;1
42201;;Saint-Appolinard;SAINT-APPOLINARD;42520;42;LOIRE;1
42202;;Saint-Barthlemy-Lestra;SAINT-BARTHELEMY-LESTRA;42110;42;LOIRE;1
42203;;Saint-Bonnet-des-Quarts;SAINT-BONNET-DES-QUARTS;42310;42;LOIRE;1
42204;;Saint-Bonnet-le-Chteau;SAINT-BONNET-LE-CHATEAU;42380;42;LOIRE;1
42205;;Saint-Bonnet-le-Courreau;SAINT-BONNET-LE-COURREAU;42940;42;LOIRE;1
42206;;Saint-Bonnet-les-Oules;SAINT-BONNET-LES-OULES;42330;42;LOIRE;1
42207;;Saint-Chamond;SAINT-CHAMOND;42400;42;LOIRE;1
42208;;Saint-Christo-en-Jarez;SAINT-CHRISTO-EN-JAREZ;42320;42;LOIRE;1
42209;;Sainte-Colombe-sur-Gand;SAINTE-COLOMBE-SUR-GAND;42540;42;LOIRE;1
42210;;Sainte-Croix-en-Jarez;SAINTE-CROIX-EN-JAREZ;42800;42;LOIRE;1
42211;;Saint-Cyprien;SAINT-CYPRIEN;42160;42;LOIRE;1
42212;;Saint-Cyr-de-Favires;SAINT-CYR-DE-FAVIERES;42132;42;LOIRE;1
42213;;Saint-Cyr-de-Valorges;SAINT-CYR-DE-VALORGES;42114;42;LOIRE;1
42214;;Saint-Cyr-les-Vignes;SAINT-CYR-LES-VIGNES;42210;42;LOIRE;1
42215;;Saint-Denis-de-Cabanne;SAINT-DENIS-DE-CABANNE;42750;42;LOIRE;1
42216;;Saint-Denis-sur-Coise;SAINT-DENIS-SUR-COISE;42140;42;LOIRE;1
42217;;Saint-Didier-sur-Rochefort;SAINT-DIDIER-SUR-ROCHEFORT;42111;42;LOIRE;1
42218;;Saint-tienne;SAINT-ETIENNE;42000;42;LOIRE;1
42219;;Saint-tienne-le-Molard;SAINT-ETIENNE-LE-MOLARD;42130;42;LOIRE;1
42220;;Saint-Forgeux-Lespinasse;SAINT-FORGEUX-LESPINASSE;42640;42;LOIRE;1
42221;;Sainte-Foy-Saint-Sulpice;SAINTE-FOY-SAINT-SULPICE;42110;42;LOIRE;1
42222;;Saint-Galmier;SAINT-GALMIER;42330;42;LOIRE;1
42223;;Saint-Genest-Lerpt;SAINT-GENEST-LERPT;42530;42;LOIRE;1
42224;;Saint-Genest-Malifaux;SAINT-GENEST-MALIFAUX;42660;42;LOIRE;1
42225;;Genilac;GENILAC;42800;42;LOIRE;1
42226;;Saint-Georges-de-Baroille;SAINT-GEORGES-DE-BAROILLE;42510;42;LOIRE;1
42227;;Saint-Georges-en-Couzan;SAINT-GEORGES-EN-COUZAN;42990;42;LOIRE;1
42228;;Saint-Georges-Haute-Ville;SAINT-GEORGES-HAUTE-VILLE;42610;42;LOIRE;1
42229;;Saint-Germain-la-Montagne;SAINT-GERMAIN-LA-MONTAGNE;42670;42;LOIRE;1
42230;;Saint-Germain-Laval;SAINT-GERMAIN-LAVAL;42260;42;LOIRE;1
42231;;Saint-Germain-Lespinasse;SAINT-GERMAIN-LESPINASSE;42640;42;LOIRE;1
42232;;Saint-Haon-le-Chtel;SAINT-HAON-LE-CHATEL;42370;42;LOIRE;1
42233;;Saint-Haon-le-Vieux;SAINT-HAON-LE-VIEUX;42370;42;LOIRE;1
42234;;Saint-Hand;SAINT-HEAND;42570;42;LOIRE;1
42235;;Saint-Hilaire-Cusson-la-Valmitte;SAINT-HILAIRE-CUSSON-LA-VALMITTE;42380;42;LOIRE;1
42236;;Saint-Hilaire-sous-Charlieu;SAINT-HILAIRE-SOUS-CHARLIEU;42190;42;LOIRE;1
42237;;Saint-Jean-Bonnefonds;SAINT-JEAN-BONNEFONDS;42650;42;LOIRE;1
42238;;Saint-Jean-la-Vtre;SAINT-JEAN-LA-VETRE;42440;42;LOIRE;1
42239;;Saint-Jean-Saint-Maurice-sur-Loire;SAINT-JEAN-SAINT-MAURICE-SUR-LOIRE;42155;42;LOIRE;1
42240;;Saint-Jean-Soleymieux;SAINT-JEAN-SOLEYMIEUX;42560;42;LOIRE;1
42241;;Saint-Jodard;SAINT-JODARD;42590;42;LOIRE;1
42242;;Saint-Joseph;SAINT-JOSEPH;42800;42;LOIRE;1
42243;;Saint-Julien-d'Oddes;SAINT-JULIEN-D'ODDES;42260;42;LOIRE;1
42245;;Saint-Julien-la-Vtre;SAINT-JULIEN-LA-VETRE;42440;42;LOIRE;1
42246;;Saint-Julien-Molin-Molette;SAINT-JULIEN-MOLIN-MOLETTE;42220;42;LOIRE;1
42247;;Saint-Just-en-Bas;SAINT-JUST-EN-BAS;42136;42;LOIRE;1
42248;;Saint-Just-en-Chevalet;SAINT-JUST-EN-CHEVALET;42430;42;LOIRE;1
42249;;Saint-Just-la-Pendue;SAINT-JUST-LA-PENDUE;42540;42;LOIRE;1
42251;;Saint-Laurent-la-Conche;SAINT-LAURENT-LA-CONCHE;42210;42;LOIRE;1
42252;;Saint-Laurent-Rochefort;SAINT-LAURENT-ROCHEFORT;42130;42;LOIRE;1
42253;;Saint-Lger-sur-Roanne;SAINT-LEGER-SUR-ROANNE;42155;42;LOIRE;1
42254;;Saint-Marcel-de-Flines;SAINT-MARCEL-DE-FELINES;42122;42;LOIRE;1
42255;;Saint-Marcel-d'Urf;SAINT-MARCEL-D'URFE;42430;42;LOIRE;1
42256;;Saint-Marcellin-en-Forez;SAINT-MARCELLIN-EN-FOREZ;42680;42;LOIRE;1
42257;;Saint-Martin-d'Estraux;SAINT-MARTIN-D'ESTREAUX;42620;42;LOIRE;1
42259;;Saint-Martin-la-Plaine;SAINT-MARTIN-LA-PLAINE;42800;42;LOIRE;1
42260;;Saint-Martin-la-Sauvet;SAINT-MARTIN-LA-SAUVETE;42260;42;LOIRE;1
42261;;Saint-Martin-Lestra;SAINT-MARTIN-LESTRA;42110;42;LOIRE;1
42262;;Saint-Maurice-en-Gourgois;SAINT-MAURICE-EN-GOURGOIS;42240;42;LOIRE;1
42264;;Saint-Mdard-en-Forez;SAINT-MEDARD-EN-FOREZ;42330;42;LOIRE;1
42265;;Saint-Michel-sur-Rhne;SAINT-MICHEL-SUR-RHONE;42410;42;LOIRE;1
42266;;Saint-Nizier-de-Fornas;SAINT-NIZIER-DE-FORNAS;42380;42;LOIRE;1
42267;;Saint-Nizier-sous-Charlieu;SAINT-NIZIER-SOUS-CHARLIEU;42190;42;LOIRE;1
42268;;Saint-Paul-de-Vzelin;SAINT-PAUL-DE-VEZELIN;42590;42;LOIRE;1
42269;;Saint-Paul-d'Uzore;SAINT-PAUL-D'UZORE;42600;42;LOIRE;1
42270;;Saint-Paul-en-Cornillon;SAINT-PAUL-EN-CORNILLON;42240;42;LOIRE;1
42271;;Saint-Paul-en-Jarez;SAINT-PAUL-EN-JAREZ;42740;42;LOIRE;1
42272;;Saint-Pierre-de-Buf;SAINT-PIERRE-DE-BOEUF;42520;42;LOIRE;1
42273;;Saint-Pierre-la-Noaille;SAINT-PIERRE-LA-NOAILLE;42190;42;LOIRE;1
42274;;Saint-Polgues;SAINT-POLGUES;42260;42;LOIRE;1
42275;;Saint-Priest-en-Jarez;SAINT-PRIEST-EN-JAREZ;42270;42;LOIRE;1
42276;;Saint-Priest-la-Prugne;SAINT-PRIEST-LA-PRUGNE;42830;42;LOIRE;1
42277;;Saint-Priest-la-Roche;SAINT-PRIEST-LA-ROCHE;42590;42;LOIRE;1
42278;;Saint-Priest-la-Vtre;SAINT-PRIEST-LA-VETRE;42440;42;LOIRE;1
42279;;Saint-Just-Saint-Rambert;SAINT-JUST-SAINT-RAMBERT;42170;42;LOIRE;1
42280;;Saint-Rgis-du-Coin;SAINT-REGIS-DU-COIN;42660;42;LOIRE;1
42281;;Saint-Rirand;SAINT-RIRAND;42370;42;LOIRE;1
42282;;Saint-Romain-d'Urf;SAINT-ROMAIN-D'URFE;42430;42;LOIRE;1
42283;;Saint-Romain-en-Jarez;SAINT-ROMAIN-EN-JAREZ;42800;42;LOIRE;1
42284;;Saint-Romain-la-Motte;SAINT-ROMAIN-LA-MOTTE;42640;42;LOIRE;1
42285;;Saint-Romain-le-Puy;SAINT-ROMAIN-LE-PUY;42610;42;LOIRE;1
42286;;Saint-Romain-les-Atheux;SAINT-ROMAIN-LES-ATHEUX;42660;42;LOIRE;1
42287;;Saint-Sauveur-en-Rue;SAINT-SAUVEUR-EN-RUE;42220;42;LOIRE;1
42288;;Saint-Sixte;SAINT-SIXTE;42130;42;LOIRE;1
42289;;Saint-Symphorien-de-Lay;SAINT-SYMPHORIEN-DE-LAY;42470;42;LOIRE;1
42290;;Saint-Thomas-la-Garde;SAINT-THOMAS-LA-GARDE;42600;42;LOIRE;1
42291;;Saint-Thurin;SAINT-THURIN;42111;42;LOIRE;1
42293;;Saint-Victor-sur-Rhins;SAINT-VICTOR-SUR-RHINS;42630;42;LOIRE;1
42294;;Saint-Vincent-de-Boisset;SAINT-VINCENT-DE-BOISSET;42120;42;LOIRE;1
42295;Les;Salles;SALLES;42440;42;LOIRE;1
42296;;Salt-en-Donzy;SALT-EN-DONZY;42110;42;LOIRE;1
42297;;Salvizinet;SALVIZINET;42110;42;LOIRE;1
42298;;Sauvain;SAUVAIN;42990;42;LOIRE;1
42299;;Savigneux;SAVIGNEUX;42600;42;LOIRE;1
42300;;Sevelinges;SEVELINGES;42460;42;LOIRE;1
42301;;Soleymieux;SOLEYMIEUX;42560;42;LOIRE;1
42302;;Sorbiers;SORBIERS;42290;42;LOIRE;1
42303;;Souternon;SOUTERNON;42260;42;LOIRE;1
42304;;Sury-le-Comtal;SURY-LE-COMTAL;42450;42;LOIRE;1
42305;La;Talaudire;TALAUDIERE;42350;42;LOIRE;1
42306;;Tarentaise;TARENTAISE;42660;42;LOIRE;1
42307;;Tartaras;TARTARAS;42800;42;LOIRE;1
42308;La;Terrasse-sur-Dorlay;TERRASSE-SUR-DORLAY;42740;42;LOIRE;1
42310;;Thlis-la-Combe;THELIS-LA-COMBE;42220;42;LOIRE;1
42311;La;Tour-en-Jarez;TOUR-EN-JAREZ;42580;42;LOIRE;1
42312;La;Tourette;TOURETTE;42380;42;LOIRE;1
42313;;Trelins;TRELINS;42130;42;LOIRE;1
42314;La;Tuilire;TUILIERE;42830;42;LOIRE;1
42315;;Unias;UNIAS;42210;42;LOIRE;1
42316;;Unieux;UNIEUX;42240;42;LOIRE;1
42317;;Urbise;URBISE;42310;42;LOIRE;1
42318;;Usson-en-Forez;USSON-EN-FOREZ;42550;42;LOIRE;1
42319;;Valeille;VALEILLE;42110;42;LOIRE;1
42320;;Valfleury;VALFLEURY;42320;42;LOIRE;1
42321;La;Valla-sur-Rochefort;VALLA-SUR-ROCHEFORT;42111;42;LOIRE;1
42322;La;Valla-en-Gier;VALLA-EN-GIER;42131;42;LOIRE;1
42323;;Veauche;VEAUCHE;42340;42;LOIRE;1
42324;;Veauchette;VEAUCHETTE;42340;42;LOIRE;1
42325;;Vendranges;VENDRANGES;42590;42;LOIRE;1
42326;;Vranne;VERANNE;42520;42;LOIRE;1
42327;;Vrin;VERIN;42410;42;LOIRE;1
42328;;Verrires-en-Forez;VERRIERES-EN-FOREZ;42600;42;LOIRE;1
42329;La;Versanne;VERSANNE;42220;42;LOIRE;1
42330;;Villars;VILLARS;42390;42;LOIRE;1
42331;;Villemontais;VILLEMONTAIS;42155;42;LOIRE;1
42332;;Villerest;VILLEREST;42300;42;LOIRE;1
42333;;Villers;VILLERS;42460;42;LOIRE;1
42334;;Violay;VIOLAY;42780;42;LOIRE;1
42335;;Viricelles;VIRICELLES;42140;42;LOIRE;1
42336;;Virigneux;VIRIGNEUX;42140;42;LOIRE;1
42337;;Vivans;VIVANS;42310;42;LOIRE;1
42338;;Vougy;VOUGY;42720;42;LOIRE;1
42339;;Chausseterre;CHAUSSETERRE;42430;42;LOIRE;1
43001;;Agnat;AGNAT;43100;43;HAUTE LOIRE;1
43002;;Aiguilhe;AIGUILHE;43000;43;HAUTE LOIRE;1
43003;;Allgre;ALLEGRE;43270;43;HAUTE LOIRE;1
43004;;Alleyrac;ALLEYRAC;43150;43;HAUTE LOIRE;1
43005;;Alleyras;ALLEYRAS;43580;43;HAUTE LOIRE;1
43006;;Ally;ALLY;43380;43;HAUTE LOIRE;1
43007;;Araules;ARAULES;43200;43;HAUTE LOIRE;1
43008;;Arlempdes;ARLEMPDES;43490;43;HAUTE LOIRE;1
43009;;Arlet;ARLET;43380;43;HAUTE LOIRE;1
43010;;Arsac-en-Velay;ARSAC-EN-VELAY;43700;43;HAUTE LOIRE;1
43011;;Aubazat;AUBAZAT;43380;43;HAUTE LOIRE;1
43012;;Aurec-sur-Loire;AUREC-SUR-LOIRE;43110;43;HAUTE LOIRE;1
43013;;Vissac-Auteyrac;VISSAC-AUTEYRAC;43300;43;HAUTE LOIRE;1
43014;;Autrac;AUTRAC;43450;43;HAUTE LOIRE;1
43015;;Auvers;AUVERS;43300;43;HAUTE LOIRE;1
43016;;Auzon;AUZON;43390;43;HAUTE LOIRE;1
43017;;Azrat;AZERAT;43390;43;HAUTE LOIRE;1
43018;;Bains;BAINS;43370;43;HAUTE LOIRE;1
43019;;Barges;BARGES;43340;43;HAUTE LOIRE;1
43020;;Bas-en-Basset;BAS-EN-BASSET;43210;43;HAUTE LOIRE;1
43021;;Beaulieu;BEAULIEU;43800;43;HAUTE LOIRE;1
43022;;Beaumont;BEAUMONT;43100;43;HAUTE LOIRE;1
43023;;Beaune-sur-Arzon;BEAUNE-SUR-ARZON;43500;43;HAUTE LOIRE;1
43024;;Beaux;BEAUX;43200;43;HAUTE LOIRE;1
43025;;Beauzac;BEAUZAC;43590;43;HAUTE LOIRE;1
43026;;Bellevue-la-Montagne;BELLEVUE-LA-MONTAGNE;43350;43;HAUTE LOIRE;1
43027;;Berbezit;BERBEZIT;43160;43;HAUTE LOIRE;1
43028;;Bessamorel;BESSAMOREL;43200;43;HAUTE LOIRE;1
43029;La;Besseyre-Saint-Mary;BESSEYRE-SAINT-MARY;43170;43;HAUTE LOIRE;1
43030;;Blanzac;BLANZAC;43350;43;HAUTE LOIRE;1
43031;;Blassac;BLASSAC;43380;43;HAUTE LOIRE;1
43032;;Blavozy;BLAVOZY;43700;43;HAUTE LOIRE;1
43033;;Blesle;BLESLE;43450;43;HAUTE LOIRE;1
43034;;Boisset;BOISSET;43500;43;HAUTE LOIRE;1
43035;;Bonneval;BONNEVAL;43160;43;HAUTE LOIRE;1
43036;;Borne;BORNE;43350;43;HAUTE LOIRE;1
43037;Le;Bouchet-Saint-Nicolas;BOUCHET-SAINT-NICOLAS;43510;43;HAUTE LOIRE;1
43038;;Bournoncle-Saint-Pierre;BOURNONCLE-SAINT-PIERRE;43360;43;HAUTE LOIRE;1
43039;Le;Brignon;BRIGNON;43370;43;HAUTE LOIRE;1
43040;;Brioude;BRIOUDE;43100;43;HAUTE LOIRE;1
43041;;Brives-Charensac;BRIVES-CHARENSAC;43700;43;HAUTE LOIRE;1
43042;;Cayres;CAYRES;43510;43;HAUTE LOIRE;1
43043;;Caux-d'Allgre;CEAUX-D'ALLEGRE;43270;43;HAUTE LOIRE;1
43044;;Cerzat;CERZAT;43380;43;HAUTE LOIRE;1
43045;;Ceyssac;CEYSSAC;43000;43;HAUTE LOIRE;1
43046;;Chadrac;CHADRAC;43770;43;HAUTE LOIRE;1
43047;;Chadron;CHADRON;43150;43;HAUTE LOIRE;1
43048;La;Chaise-Dieu;CHAISE-DIEU;43160;43;HAUTE LOIRE;1
43049;;Chamalires-sur-Loire;CHAMALIERES-SUR-LOIRE;43800;43;HAUTE LOIRE;1
43050;;Chambezon;CHAMBEZON;43410;43;HAUTE LOIRE;1
43051;Le;Chambon-sur-Lignon;CHAMBON-SUR-LIGNON;43400;43;HAUTE LOIRE;1
43052;;Champagnac-le-Vieux;CHAMPAGNAC-LE-VIEUX;43440;43;HAUTE LOIRE;1
43053;;Champclause;CHAMPCLAUSE;43260;43;HAUTE LOIRE;1
43054;;Chanaleilles;CHANALEILLES;43170;43;HAUTE LOIRE;1
43055;;Chaniat;CHANIAT;43100;43;HAUTE LOIRE;1
43056;;Chanteuges;CHANTEUGES;43300;43;HAUTE LOIRE;1
43057;La;Chapelle-Bertin;CHAPELLE-BERTIN;43270;43;HAUTE LOIRE;1
43058;La;Chapelle-d'Aurec;CHAPELLE-D'AUREC;43120;43;HAUTE LOIRE;1
43059;La;Chapelle-Geneste;CHAPELLE-GENESTE;43160;43;HAUTE LOIRE;1
43060;;Charraix;CHARRAIX;43300;43;HAUTE LOIRE;1
43061;;Chaspinhac;CHASPINHAC;43700;43;HAUTE LOIRE;1
43062;;Chaspuzac;CHASPUZAC;43320;43;HAUTE LOIRE;1
43063;;Chassagnes;CHASSAGNES;43230;43;HAUTE LOIRE;1
43064;;Chassignolles;CHASSIGNOLLES;43440;43;HAUTE LOIRE;1
43065;;Chastel;CHASTEL;43300;43;HAUTE LOIRE;1
43066;;Chaudeyrolles;CHAUDEYROLLES;43430;43;HAUTE LOIRE;1
43067;;Chavaniac-Lafayette;CHAVANIAC-LAFAYETTE;43230;43;HAUTE LOIRE;1
43068;;Chazelles;CHAZELLES;43300;43;HAUTE LOIRE;1
43069;;Chenereilles;CHENEREILLES;43190;43;HAUTE LOIRE;1
43070;;Chilhac;CHILHAC;43380;43;HAUTE LOIRE;1
43071;;Chomelix;CHOMELIX;43500;43;HAUTE LOIRE;1
43072;La;Chomette;CHOMETTE;43230;43;HAUTE LOIRE;1
43073;;Cistrires;CISTRIERES;43160;43;HAUTE LOIRE;1
43074;;Cohade;COHADE;43100;43;HAUTE LOIRE;1
43075;;Collat;COLLAT;43230;43;HAUTE LOIRE;1
43076;;Connangles;CONNANGLES;43160;43;HAUTE LOIRE;1
43077;;Costaros;COSTAROS;43490;43;HAUTE LOIRE;1
43078;;Coubon;COUBON;43700;43;HAUTE LOIRE;1
43079;;Couteuges;COUTEUGES;43230;43;HAUTE LOIRE;1
43080;;Craponne-sur-Arzon;CRAPONNE-SUR-ARZON;43500;43;HAUTE LOIRE;1
43081;;Croisances;CROISANCES;43580;43;HAUTE LOIRE;1
43082;;Cronce;CRONCE;43300;43;HAUTE LOIRE;1
43083;;Cubelles;CUBELLES;43170;43;HAUTE LOIRE;1
43084;;Cussac-sur-Loire;CUSSAC-SUR-LOIRE;43370;43;HAUTE LOIRE;1
43085;;Desges;DESGES;43300;43;HAUTE LOIRE;1
43086;;Domeyrat;DOMEYRAT;43230;43;HAUTE LOIRE;1
43087;;Dunires;DUNIERES;43220;43;HAUTE LOIRE;1
43088;;Espalem;ESPALEM;43450;43;HAUTE LOIRE;1
43089;;Espaly-Saint-Marcel;ESPALY-SAINT-MARCEL;43000;43;HAUTE LOIRE;1
43090;;Esplantas;ESPLANTAS;43170;43;HAUTE LOIRE;1
43091;Les;Estables;ESTABLES;43150;43;HAUTE LOIRE;1
43092;;Fay-sur-Lignon;FAY-SUR-LIGNON;43430;43;HAUTE LOIRE;1
43093;;Flines;FELINES;43160;43;HAUTE LOIRE;1
43094;;Ferrussac;FERRUSSAC;43300;43;HAUTE LOIRE;1
43095;;Fix-Saint-Geneys;FIX-SAINT-GENEYS;43320;43;HAUTE LOIRE;1
43096;;Fontannes;FONTANNES;43100;43;HAUTE LOIRE;1
43097;;Freycenet-la-Cuche;FREYCENET-LA-CUCHE;43150;43;HAUTE LOIRE;1
43098;;Freycenet-la-Tour;FREYCENET-LA-TOUR;43150;43;HAUTE LOIRE;1
43099;;Frugers-les-Mines;FRUGERES-LES-MINES;43250;43;HAUTE LOIRE;1
43100;;Frugires-le-Pin;FRUGIERES-LE-PIN;43230;43;HAUTE LOIRE;1
43101;;Goudet;GOUDET;43150;43;HAUTE LOIRE;1
43102;;Grazac;GRAZAC;43200;43;HAUTE LOIRE;1
43103;;Grenier-Montgon;GRENIER-MONTGON;43450;43;HAUTE LOIRE;1
43104;;Grzes;GREZES;43170;43;HAUTE LOIRE;1
43105;;Javaugues;JAVAUGUES;43100;43;HAUTE LOIRE;1
43106;;Jax;JAX;43230;43;HAUTE LOIRE;1
43107;;Josat;JOSAT;43230;43;HAUTE LOIRE;1
43108;;Jullianges;JULLIANGES;43500;43;HAUTE LOIRE;1
43109;;Lafarre;LAFARRE;43490;43;HAUTE LOIRE;1
43110;;Lamothe;LAMOTHE;43100;43;HAUTE LOIRE;1
43111;;Landos;LANDOS;43340;43;HAUTE LOIRE;1
43112;;Langeac;LANGEAC;43300;43;HAUTE LOIRE;1
43113;;Lantriac;LANTRIAC;43260;43;HAUTE LOIRE;1
43114;;Lapte;LAPTE;43200;43;HAUTE LOIRE;1
43115;;Laussonne;LAUSSONNE;43150;43;HAUTE LOIRE;1
43116;;Laval-sur-Doulon;LAVAL-SUR-DOULON;43440;43;HAUTE LOIRE;1
43117;;Lavaudieu;LAVAUDIEU;43100;43;HAUTE LOIRE;1
43118;;Lavote-Chilhac;LAVOUTE-CHILHAC;43380;43;HAUTE LOIRE;1
43119;;Lavote-sur-Loire;LAVOUTE-SUR-LOIRE;43800;43;HAUTE LOIRE;1
43120;;Lempdes-sur-Allagnon;LEMPDES-SUR-ALLAGNON;43410;43;HAUTE LOIRE;1
43121;;Lotoing;LEOTOING;43410;43;HAUTE LOIRE;1
43122;;Lissac;LISSAC;43350;43;HAUTE LOIRE;1
43123;;Lorlanges;LORLANGES;43360;43;HAUTE LOIRE;1
43124;;Loudes;LOUDES;43320;43;HAUTE LOIRE;1
43125;;Lubilhac;LUBILHAC;43100;43;HAUTE LOIRE;1
43126;;Malrevers;MALREVERS;43800;43;HAUTE LOIRE;1
43127;;Malvalette;MALVALETTE;43210;43;HAUTE LOIRE;1
43128;;Malvires;MALVIERES;43160;43;HAUTE LOIRE;1
43129;Le;Mas-de-Tence;MAS-DE-TENCE;43190;43;HAUTE LOIRE;1
43130;;Mazet-Saint-Voy;MAZET-SAINT-VOY;43520;43;HAUTE LOIRE;1
43131;;Mazerat-Aurouze;MAZERAT-AUROUZE;43230;43;HAUTE LOIRE;1
43132;;Mazeyrat-d'Allier;MAZEYRAT-D'ALLIER;43300;43;HAUTE LOIRE;1
43133;;Mercur;MERCOEUR;43100;43;HAUTE LOIRE;1
43134;;Mzres;MEZERES;43800;43;HAUTE LOIRE;1
43135;Le;Monastier-sur-Gazeille;MONASTIER-SUR-GAZEILLE;43150;43;HAUTE LOIRE;1
43136;;Monistrol-d'Allier;MONISTROL-D'ALLIER;43580;43;HAUTE LOIRE;1
43137;;Monistrol-sur-Loire;MONISTROL-SUR-LOIRE;43120;43;HAUTE LOIRE;1
43138;;Monlet;MONLET;43270;43;HAUTE LOIRE;1
43139;;Montclard;MONTCLARD;43230;43;HAUTE LOIRE;1
43140;Le;Monteil;MONTEIL;43700;43;HAUTE LOIRE;1
43141;;Montfaucon-en-Velay;MONTFAUCON-EN-VELAY;43290;43;HAUTE LOIRE;1
43142;;Montregard;MONTREGARD;43290;43;HAUTE LOIRE;1
43143;;Montusclat;MONTUSCLAT;43260;43;HAUTE LOIRE;1
43144;;Moudeyres;MOUDEYRES;43150;43;HAUTE LOIRE;1
43145;;Ouides;OUIDES;43510;43;HAUTE LOIRE;1
43147;;Paulhac;PAULHAC;43100;43;HAUTE LOIRE;1
43148;;Paulhaguet;PAULHAGUET;43230;43;HAUTE LOIRE;1
43149;;Pbrac;PEBRAC;43300;43;HAUTE LOIRE;1
43150;Le;Pertuis;PERTUIS;43200;43;HAUTE LOIRE;1
43151;;Pinols;PINOLS;43300;43;HAUTE LOIRE;1
43152;;Polignac;POLIGNAC;43770;43;HAUTE LOIRE;1
43153;;Pont-Salomon;PONT-SALOMON;43330;43;HAUTE LOIRE;1
43154;;Pradelles;PRADELLES;43420;43;HAUTE LOIRE;1
43155;;Prades;PRADES;43300;43;HAUTE LOIRE;1
43156;;Prsailles;PRESAILLES;43150;43;HAUTE LOIRE;1
43157;Le;Puy-en-Velay;PUY-EN-VELAY;43000;43;HAUTE LOIRE;1
43158;;Queyrires;QUEYRIERES;43260;43;HAUTE LOIRE;1
43159;;Raucoules;RAUCOULES;43290;43;HAUTE LOIRE;1
43160;;Rauret;RAURET;43340;43;HAUTE LOIRE;1
43162;;Retournac;RETOURNAC;43130;43;HAUTE LOIRE;1
43163;;Riotord;RIOTORD;43220;43;HAUTE LOIRE;1
43164;;Roche-en-Rgnier;ROCHE-EN-REGNIER;43130;43;HAUTE LOIRE;1
43165;;Rosires;ROSIERES;43800;43;HAUTE LOIRE;1
43166;;Saint-Andr-de-Chalencon;SAINT-ANDRE-DE-CHALENCON;43130;43;HAUTE LOIRE;1
43167;;Saint-Arcons-d'Allier;SAINT-ARCONS-D'ALLIER;43300;43;HAUTE LOIRE;1
43168;;Saint-Arcons-de-Barges;SAINT-ARCONS-DE-BARGES;43420;43;HAUTE LOIRE;1
43169;;Saint-Austremoine;SAINT-AUSTREMOINE;43380;43;HAUTE LOIRE;1
43170;;Saint-Beauzire;SAINT-BEAUZIRE;43100;43;HAUTE LOIRE;1
43171;;Saint-Brain;SAINT-BERAIN;43300;43;HAUTE LOIRE;1
43172;;Saint-Bonnet-le-Froid;SAINT-BONNET-LE-FROID;43290;43;HAUTE LOIRE;1
43173;;Saint-Christophe-d'Allier;SAINT-CHRISTOPHE-D'ALLIER;43340;43;HAUTE LOIRE;1
43174;;Saint-Christophe-sur-Dolaison;SAINT-CHRISTOPHE-SUR-DOLAISON;43370;43;HAUTE LOIRE;1
43175;;Saint-Cirgues;SAINT-CIRGUES;43380;43;HAUTE LOIRE;1
43176;;Saint-Didier-d'Allier;SAINT-DIDIER-D'ALLIER;43580;43;HAUTE LOIRE;1
43177;;Saint-Didier-en-Velay;SAINT-DIDIER-EN-VELAY;43140;43;HAUTE LOIRE;1
43178;;Saint-Didier-sur-Doulon;SAINT-DIDIER-SUR-DOULON;43440;43;HAUTE LOIRE;1
43180;;Saint-tienne-du-Vigan;SAINT-ETIENNE-DU-VIGAN;43420;43;HAUTE LOIRE;1
43181;;Saint-tienne-Lardeyrol;SAINT-ETIENNE-LARDEYROL;43260;43;HAUTE LOIRE;1
43182;;Saint-tienne-sur-Blesle;SAINT-ETIENNE-SUR-BLESLE;43450;43;HAUTE LOIRE;1
43183;;Sainte-Eugnie-de-Villeneuve;SAINTE-EUGENIE-DE-VILLENEUVE;43230;43;HAUTE LOIRE;1
43184;;Saint-Ferrol-d'Auroure;SAINT-FERREOL-D'AUROURE;43330;43;HAUTE LOIRE;1
43185;;Sainte-Florine;SAINTE-FLORINE;43250;43;HAUTE LOIRE;1
43186;;Saint-Front;SAINT-FRONT;43550;43;HAUTE LOIRE;1
43187;;Saint-Geneys-prs-Saint-Paulien;SAINT-GENEYS-PRES-SAINT-PAULIEN;43350;43;HAUTE LOIRE;1
43188;;Saint-Georges-d'Aurac;SAINT-GEORGES-D'AURAC;43230;43;HAUTE LOIRE;1
43189;;Saint-Georges-Lagricol;SAINT-GEORGES-LAGRICOL;43500;43;HAUTE LOIRE;1
43190;;Saint-Germain-Laprade;SAINT-GERMAIN-LAPRADE;43700;43;HAUTE LOIRE;1
43191;;Saint-Gron;SAINT-GERON;43360;43;HAUTE LOIRE;1
43192;;Saint-Haon;SAINT-HAON;43340;43;HAUTE LOIRE;1
43193;;Saint-Hilaire;SAINT-HILAIRE;43390;43;HAUTE LOIRE;1
43194;;Saint-Hostien;SAINT-HOSTIEN;43260;43;HAUTE LOIRE;1
43195;;Saint-Ilpize;SAINT-ILPIZE;43380;43;HAUTE LOIRE;1
43196;;Saint-Jean-d'Aubrigoux;SAINT-JEAN-D'AUBRIGOUX;43500;43;HAUTE LOIRE;1
43197;;Saint-Jean-de-Nay;SAINT-JEAN-DE-NAY;43320;43;HAUTE LOIRE;1
43198;;Saint-Jean-Lachalm;SAINT-JEAN-LACHALM;43510;43;HAUTE LOIRE;1
43199;;Saint-Jeures;SAINT-JEURES;43200;43;HAUTE LOIRE;1
43200;;Saint-Julien-Chapteuil;SAINT-JULIEN-CHAPTEUIL;43260;43;HAUTE LOIRE;1
43201;;Saint-Julien-d'Ance;SAINT-JULIEN-D'ANCE;43500;43;HAUTE LOIRE;1
43202;;Saint-Julien-des-Chazes;SAINT-JULIEN-DES-CHAZES;43300;43;HAUTE LOIRE;1
43203;;Saint-Julien-du-Pinet;SAINT-JULIEN-DU-PINET;43200;43;HAUTE LOIRE;1
43204;;Saint-Julien-Molhesabate;SAINT-JULIEN-MOLHESABATE;43220;43;HAUTE LOIRE;1
43205;;Saint-Just-Malmont;SAINT-JUST-MALMONT;43240;43;HAUTE LOIRE;1
43206;;Saint-Just-prs-Brioude;SAINT-JUST-PRES-BRIOUDE;43100;43;HAUTE LOIRE;1
43207;;Saint-Laurent-Chabreuges;SAINT-LAURENT-CHABREUGES;43100;43;HAUTE LOIRE;1
43208;;Sainte-Marguerite;SAINTE-MARGUERITE;43230;43;HAUTE LOIRE;1
43210;;Saint-Martin-de-Fugres;SAINT-MARTIN-DE-FUGERES;43150;43;HAUTE LOIRE;1
43211;;Saint-Maurice-de-Lignon;SAINT-MAURICE-DE-LIGNON;43200;43;HAUTE LOIRE;1
43212;;Saint-Pal-de-Chalencon;SAINT-PAL-DE-CHALENCON;43500;43;HAUTE LOIRE;1
43213;;Saint-Pal-de-Mons;SAINT-PAL-DE-MONS;43620;43;HAUTE LOIRE;1
43214;;Saint-Pal-de-Senouire;SAINT-PAL-DE-SENOUIRE;43160;43;HAUTE LOIRE;1
43215;;Saint-Paul-de-Tartas;SAINT-PAUL-DE-TARTAS;43420;43;HAUTE LOIRE;1
43216;;Saint-Paulien;SAINT-PAULIEN;43350;43;HAUTE LOIRE;1
43217;;Saint-Pierre-du-Champ;SAINT-PIERRE-DU-CHAMP;43130;43;HAUTE LOIRE;1
43218;;Saint-Pierre-Eynac;SAINT-PIERRE-EYNAC;43260;43;HAUTE LOIRE;1
43219;;Saint-Prjet-Armandon;SAINT-PREJET-ARMANDON;43230;43;HAUTE LOIRE;1
43220;;Saint-Prjet-d'Allier;SAINT-PREJET-D'ALLIER;43580;43;HAUTE LOIRE;1
43221;;Saint-Privat-d'Allier;SAINT-PRIVAT-D'ALLIER;43580;43;HAUTE LOIRE;1
43222;;Saint-Privat-du-Dragon;SAINT-PRIVAT-DU-DRAGON;43380;43;HAUTE LOIRE;1
43223;;Saint-Romain-Lachalm;SAINT-ROMAIN-LACHALM;43620;43;HAUTE LOIRE;1
43224;;Sainte-Sigolne;SAINTE-SIGOLENE;43600;43;HAUTE LOIRE;1
43225;;Saint-Vnrand;SAINT-VENERAND;43580;43;HAUTE LOIRE;1
43226;;Saint-Vert;SAINT-VERT;43440;43;HAUTE LOIRE;1
43227;;Saint-Victor-Malescours;SAINT-VICTOR-MALESCOURS;43140;43;HAUTE LOIRE;1
43228;;Saint-Victor-sur-Arlanc;SAINT-VICTOR-SUR-ARLANC;43500;43;HAUTE LOIRE;1
43229;;Saint-Vidal;SAINT-VIDAL;43320;43;HAUTE LOIRE;1
43230;;Saint-Vincent;SAINT-VINCENT;43800;43;HAUTE LOIRE;1
43231;;Salettes;SALETTES;43150;43;HAUTE LOIRE;1
43232;;Salzuit;SALZUIT;43230;43;HAUTE LOIRE;1
43233;;Sanssac-l'glise;SANSSAC-L'EGLISE;43320;43;HAUTE LOIRE;1
43234;;Saugues;SAUGUES;43170;43;HAUTE LOIRE;1
43236;La;Sauve-sur-Semne;SEAUVE-SUR-SEMENE;43140;43;HAUTE LOIRE;1
43237;;Sembadel;SEMBADEL;43160;43;HAUTE LOIRE;1
43238;;Sneujols;SENEUJOLS;43510;43;HAUTE LOIRE;1
43239;;Siaugues-Sainte-Marie;SIAUGUES-SAINTE-MARIE;43300;43;HAUTE LOIRE;1
43240;;Solignac-sous-Roche;SOLIGNAC-SOUS-ROCHE;43130;43;HAUTE LOIRE;1
43241;;Solignac-sur-Loire;SOLIGNAC-SUR-LOIRE;43370;43;HAUTE LOIRE;1
43242;;Tailhac;TAILHAC;43300;43;HAUTE LOIRE;1
43244;;Tence;TENCE;43190;43;HAUTE LOIRE;1
43245;;Thoras;THORAS;43170;43;HAUTE LOIRE;1
43246;;Tiranges;TIRANGES;43130;43;HAUTE LOIRE;1
43247;;Torsiac;TORSIAC;43450;43;HAUTE LOIRE;1
43249;;Valprivas;VALPRIVAS;43210;43;HAUTE LOIRE;1
43250;;Vals-le-Chastel;VALS-LE-CHASTEL;43230;43;HAUTE LOIRE;1
43251;;Vals-prs-le-Puy;VALS-PRES-LE-PUY;43750;43;HAUTE LOIRE;1
43252;;Varennes-Saint-Honorat;VARENNES-SAINT-HONORAT;43270;43;HAUTE LOIRE;1
43253;Les;Vastres;VASTRES;43430;43;HAUTE LOIRE;1
43254;;Vazeilles-Limandre;VAZEILLES-LIMANDRE;43320;43;HAUTE LOIRE;1
43255;;Vazeilles-prs-Saugues;VAZEILLES-PRES-SAUGUES;43580;43;HAUTE LOIRE;1
43256;;Venteuges;VENTEUGES;43170;43;HAUTE LOIRE;1
43257;;Vergezac;VERGEZAC;43320;43;HAUTE LOIRE;1
43258;;Vergongheon;VERGONGHEON;43360;43;HAUTE LOIRE;1
43259;;Vernassal;VERNASSAL;43270;43;HAUTE LOIRE;1
43260;Le;Vernet;VERNET;43320;43;HAUTE LOIRE;1
43261;;Vzzoux;VEZEZOUX;43390;43;HAUTE LOIRE;1
43262;;Vieille-Brioude;VIEILLE-BRIOUDE;43100;43;HAUTE LOIRE;1
43263;;Vielprat;VIELPRAT;43490;43;HAUTE LOIRE;1
43264;;Villeneuve-d'Allier;VILLENEUVE-D'ALLIER;43380;43;HAUTE LOIRE;1
43265;Les;Villettes;VILLETTES;43600;43;HAUTE LOIRE;1
43267;;Vorey;VOREY;43800;43;HAUTE LOIRE;1
43268;;Yssingeaux;YSSINGEAUX;43200;43;HAUTE LOIRE;1
44001;;Abbaretz;ABBARETZ;44170;44;LOIRE ATLANTIQUE;1
44002;;Aigrefeuille-sur-Maine;AIGREFEUILLE-SUR-MAINE;44140;44;LOIRE ATLANTIQUE;1
44003;;Ancenis;ANCENIS;44150;44;LOIRE ATLANTIQUE;1
44004;;Anetz;ANETZ;44150;44;LOIRE ATLANTIQUE;1
44005;;Arthon-en-Retz;ARTHON-EN-RETZ;44320;44;LOIRE ATLANTIQUE;1
44006;;Assrac;ASSERAC;44410;44;LOIRE ATLANTIQUE;1
44007;;Avessac;AVESSAC;44460;44;LOIRE ATLANTIQUE;1
44008;;Barbechat;BARBECHAT;44450;44;LOIRE ATLANTIQUE;1
44009;;Basse-Goulaine;BASSE-GOULAINE;44115;44;LOIRE ATLANTIQUE;1
44010;;Batz-sur-Mer;BATZ-SUR-MER;44740;44;LOIRE ATLANTIQUE;1
44011;;Bellign;BELLIGNE;44370;44;LOIRE ATLANTIQUE;1
44012;La;Bernerie-en-Retz;BERNERIE-EN-RETZ;44760;44;LOIRE ATLANTIQUE;1
44013;;Besn;BESNE;44160;44;LOIRE ATLANTIQUE;1
44014;Le;Bignon;BIGNON;44140;44;LOIRE ATLANTIQUE;1
44015;;Blain;BLAIN;44130;44;LOIRE ATLANTIQUE;1
44016;La;Boissire-du-Dor;BOISSIERE-DU-DORE;44430;44;LOIRE ATLANTIQUE;1
44017;;Bonnuvre;BONNOEUVRE;44540;44;LOIRE ATLANTIQUE;1
44018;;Bouaye;BOUAYE;44830;44;LOIRE ATLANTIQUE;1
44019;;Boue;BOUEE;44260;44;LOIRE ATLANTIQUE;1
44020;;Bouguenais;BOUGUENAIS;44340;44;LOIRE ATLANTIQUE;1
44021;;Bourgneuf-en-Retz;BOURGNEUF-EN-RETZ;44580;44;LOIRE ATLANTIQUE;1
44022;;Boussay;BOUSSAY;44190;44;LOIRE ATLANTIQUE;1
44023;;Bouvron;BOUVRON;44130;44;LOIRE ATLANTIQUE;1
44024;;Brains;BRAINS;44830;44;LOIRE ATLANTIQUE;1
44025;;Campbon;CAMPBON;44750;44;LOIRE ATLANTIQUE;1
44026;;Carquefou;CARQUEFOU;44470;44;LOIRE ATLANTIQUE;1
44027;;Casson;CASSON;44390;44;LOIRE ATLANTIQUE;1
44028;Le;Cellier;CELLIER;44850;44;LOIRE ATLANTIQUE;1
44029;La;Chapelle-Basse-Mer;CHAPELLE-BASSE-MER;44450;44;LOIRE ATLANTIQUE;1
44030;La;Chapelle-des-Marais;CHAPELLE-DES-MARAIS;44410;44;LOIRE ATLANTIQUE;1
44031;La;Chapelle-Glain;CHAPELLE-GLAIN;44670;44;LOIRE ATLANTIQUE;1
44032;La;Chapelle-Heulin;CHAPELLE-HEULIN;44330;44;LOIRE ATLANTIQUE;1
44033;La;Chapelle-Launay;CHAPELLE-LAUNAY;44260;44;LOIRE ATLANTIQUE;1
44034;La;Chapelle-Saint-Sauveur;CHAPELLE-SAINT-SAUVEUR;44370;44;LOIRE ATLANTIQUE;1
44035;La;Chapelle-sur-Erdre;CHAPELLE-SUR-ERDRE;44240;44;LOIRE ATLANTIQUE;1
44036;;Chteaubriant;CHATEAUBRIANT;44110;44;LOIRE ATLANTIQUE;1
44037;;Chteau-Thbaud;CHATEAU-THEBAUD;44690;44;LOIRE ATLANTIQUE;1
44038;;Chauv;CHAUVE;44320;44;LOIRE ATLANTIQUE;1
44039;;Cheix-en-Retz;CHEIX-EN-RETZ;44640;44;LOIRE ATLANTIQUE;1
44040;;Chmr;CHEMERE;44680;44;LOIRE ATLANTIQUE;1
44041;La;Chevrolire;CHEVROLIERE;44118;44;LOIRE ATLANTIQUE;1
44043;;Clisson;CLISSON;44190;44;LOIRE ATLANTIQUE;1
44044;;Conquereuil;CONQUEREUIL;44290;44;LOIRE ATLANTIQUE;1
44045;;Cordemais;CORDEMAIS;44360;44;LOIRE ATLANTIQUE;1
44046;;Corsept;CORSEPT;44560;44;LOIRE ATLANTIQUE;1
44047;;Couron;COUERON;44220;44;LOIRE ATLANTIQUE;1
44048;;Couff;COUFFE;44521;44;LOIRE ATLANTIQUE;1
44049;Le;Croisic;CROISIC;44490;44;LOIRE ATLANTIQUE;1
44050;;Crossac;CROSSAC;44160;44;LOIRE ATLANTIQUE;1
44051;;Derval;DERVAL;44590;44;LOIRE ATLANTIQUE;1
44052;;Donges;DONGES;44480;44;LOIRE ATLANTIQUE;1
44053;;Dreffac;DREFFEAC;44530;44;LOIRE ATLANTIQUE;1
44054;;Erbray;ERBRAY;44110;44;LOIRE ATLANTIQUE;1
44055;La;Baule-Escoublac;BAULE-ESCOUBLAC;44500;44;LOIRE ATLANTIQUE;1
44056;;Fay-de-Bretagne;FAY-DE-BRETAGNE;44130;44;LOIRE ATLANTIQUE;1
44057;;Fgrac;FEGREAC;44460;44;LOIRE ATLANTIQUE;1
44058;;Ferc;FERCE;44660;44;LOIRE ATLANTIQUE;1
44059;;Fresnay-en-Retz;FRESNAY-EN-RETZ;44580;44;LOIRE ATLANTIQUE;1
44060;Le;Fresne-sur-Loire;FRESNE-SUR-LOIRE;49123;44;MAINE ET LOIRE;1
44061;;Frossay;FROSSAY;44320;44;LOIRE ATLANTIQUE;1
44062;Le;Gvre;GAVRE;44130;44;LOIRE ATLANTIQUE;1
44063;;Gtign;GETIGNE;44190;44;LOIRE ATLANTIQUE;1
44064;;Gorges;GORGES;44190;44;LOIRE ATLANTIQUE;1
44065;;Grand-Auvern;GRAND-AUVERNE;44520;44;LOIRE ATLANTIQUE;1
44066;;Grandchamps-des-Fontaines;GRANDCHAMPS-DES-FONTAINES;44119;44;LOIRE ATLANTIQUE;1
44067;;Gumen-Penfao;GUEMENE-PENFAO;44290;44;LOIRE ATLANTIQUE;1
44068;;Guenrouet;GUENROUET;44530;44;LOIRE ATLANTIQUE;1
44069;;Gurande;GUERANDE;44350;44;LOIRE ATLANTIQUE;1
44070;La;Haie-Fouassire;HAIE-FOUASSIERE;44690;44;LOIRE ATLANTIQUE;1
44071;;Haute-Goulaine;HAUTE-GOULAINE;44115;44;LOIRE ATLANTIQUE;1
44072;;Herbignac;HERBIGNAC;44410;44;LOIRE ATLANTIQUE;1
44073;;Hric;HERIC;44810;44;LOIRE ATLANTIQUE;1
44074;;Indre;INDRE;44610;44;LOIRE ATLANTIQUE;1
44075;;Iss;ISSE;44520;44;LOIRE ATLANTIQUE;1
44076;;Jans;JANS;44170;44;LOIRE ATLANTIQUE;1
44077;;Jou-sur-Erdre;JOUE-SUR-ERDRE;44440;44;LOIRE ATLANTIQUE;1
44078;;Juign-des-Moutiers;JUIGNE-DES-MOUTIERS;44670;44;LOIRE ATLANTIQUE;1
44079;Le;Landreau;LANDREAU;44430;44;LOIRE ATLANTIQUE;1
44080;;Lavau-sur-Loire;LAVAU-SUR-LOIRE;44260;44;LOIRE ATLANTIQUE;1
44081;;Leg;LEGE;44650;44;LOIRE ATLANTIQUE;1
44082;;Lign;LIGNE;44850;44;LOIRE ATLANTIQUE;1
44083;La;Limouzinire;LIMOUZINIERE;44310;44;LOIRE ATLANTIQUE;1
44084;Le;Loroux-Bottereau;LOROUX-BOTTEREAU;44430;44;LOIRE ATLANTIQUE;1
44085;;Louisfert;LOUISFERT;44110;44;LOIRE ATLANTIQUE;1
44086;;Lusanger;LUSANGER;44590;44;LOIRE ATLANTIQUE;1
44087;;Machecoul;MACHECOUL;44270;44;LOIRE ATLANTIQUE;1
44088;;Maisdon-sur-Svre;MAISDON-SUR-SEVRE;44690;44;LOIRE ATLANTIQUE;1
44089;;Malville;MALVILLE;44260;44;LOIRE ATLANTIQUE;1
44090;La;Marne;MARNE;44270;44;LOIRE ATLANTIQUE;1
44091;;Marsac-sur-Don;MARSAC-SUR-DON;44170;44;LOIRE ATLANTIQUE;1
44092;;Massrac;MASSERAC;44290;44;LOIRE ATLANTIQUE;1
44093;;Maumusson;MAUMUSSON;44540;44;LOIRE ATLANTIQUE;1
44094;;Mauves-sur-Loire;MAUVES-SUR-LOIRE;44470;44;LOIRE ATLANTIQUE;1
44095;La;Meilleraye-de-Bretagne;MEILLERAYE-DE-BRETAGNE;44520;44;LOIRE ATLANTIQUE;1
44096;;Msanger;MESANGER;44522;44;LOIRE ATLANTIQUE;1
44097;;Mesquer;MESQUER;44420;44;LOIRE ATLANTIQUE;1
44098;;Missillac;MISSILLAC;44780;44;LOIRE ATLANTIQUE;1
44099;;Moisdon-la-Rivire;MOISDON-LA-RIVIERE;44520;44;LOIRE ATLANTIQUE;1
44100;;Monnires;MONNIERES;44690;44;LOIRE ATLANTIQUE;1
44101;La;Montagne;MONTAGNE;44620;44;LOIRE ATLANTIQUE;1
44102;;Montbert;MONTBERT;44140;44;LOIRE ATLANTIQUE;1
44103;;Montoir-de-Bretagne;MONTOIR-DE-BRETAGNE;44550;44;LOIRE ATLANTIQUE;1
44104;;Montrelais;MONTRELAIS;44370;44;LOIRE ATLANTIQUE;1
44105;;Mouais;MOUAIS;44590;44;LOIRE ATLANTIQUE;1
44106;Les;Moutiers-en-Retz;MOUTIERS-EN-RETZ;44580;44;LOIRE ATLANTIQUE;1
44107;;Mouzeil;MOUZEIL;44850;44;LOIRE ATLANTIQUE;1
44108;;Mouzillon;MOUZILLON;44330;44;LOIRE ATLANTIQUE;1
44109;;Nantes;NANTES;44000;44;LOIRE ATLANTIQUE;1
44110;;Nort-sur-Erdre;NORT-SUR-ERDRE;44390;44;LOIRE ATLANTIQUE;1
44111;;Notre-Dame-des-Landes;NOTRE-DAME-DES-LANDES;44130;44;LOIRE ATLANTIQUE;1
44112;;Noyal-sur-Brutz;NOYAL-SUR-BRUTZ;44110;44;LOIRE ATLANTIQUE;1
44113;;Nozay;NOZAY;44170;44;LOIRE ATLANTIQUE;1
44114;;Orvault;ORVAULT;44700;44;LOIRE ATLANTIQUE;1
44115;;Oudon;OUDON;44521;44;LOIRE ATLANTIQUE;1
44116;;Paimbuf;PAIMBOEUF;44560;44;LOIRE ATLANTIQUE;1
44117;Le;Pallet;PALLET;44330;44;LOIRE ATLANTIQUE;1
44118;;Pannec;PANNECE;44440;44;LOIRE ATLANTIQUE;1
44119;;Paulx;PAULX;44270;44;LOIRE ATLANTIQUE;1
44120;Le;Pellerin;PELLERIN;44640;44;LOIRE ATLANTIQUE;1
44121;;Petit-Auvern;PETIT-AUVERNE;44670;44;LOIRE ATLANTIQUE;1
44122;;Petit-Mars;PETIT-MARS;44390;44;LOIRE ATLANTIQUE;1
44123;;Pierric;PIERRIC;44290;44;LOIRE ATLANTIQUE;1
44124;Le;Pin;PIN;44540;44;LOIRE ATLANTIQUE;1
44125;;Piriac-sur-Mer;PIRIAC-SUR-MER;44420;44;LOIRE ATLANTIQUE;1
44126;La;Plaine-sur-Mer;PLAINE-SUR-MER;44770;44;LOIRE ATLANTIQUE;1
44127;La;Planche;PLANCHE;44140;44;LOIRE ATLANTIQUE;1
44128;;Pless;PLESSE;44630;44;LOIRE ATLANTIQUE;1
44129;;Pontchteau;PONTCHATEAU;44160;44;LOIRE ATLANTIQUE;1
44130;;Pont-Saint-Martin;PONT-SAINT-MARTIN;44860;44;LOIRE ATLANTIQUE;1
44131;;Pornic;PORNIC;44210;44;LOIRE ATLANTIQUE;1
44132;;Pornichet;PORNICHET;44380;44;LOIRE ATLANTIQUE;1
44133;;Port-Saint-Pre;PORT-SAINT-PERE;44710;44;LOIRE ATLANTIQUE;1
44134;;Pouill-les-Cteaux;POUILLE-LES-COTEAUX;44522;44;LOIRE ATLANTIQUE;1
44135;Le;Pouliguen;POULIGUEN;44510;44;LOIRE ATLANTIQUE;1
44136;;Prfailles;PREFAILLES;44770;44;LOIRE ATLANTIQUE;1
44137;;Prinquiau;PRINQUIAU;44260;44;LOIRE ATLANTIQUE;1
44138;;Puceul;PUCEUL;44390;44;LOIRE ATLANTIQUE;1
44139;;Quilly;QUILLY;44750;44;LOIRE ATLANTIQUE;1
44140;La;Regrippire;REGRIPPIERE;44330;44;LOIRE ATLANTIQUE;1
44141;La;Remaudire;REMAUDIERE;44430;44;LOIRE ATLANTIQUE;1
44142;;Remouill;REMOUILLE;44140;44;LOIRE ATLANTIQUE;1
44143;;Rez;REZE;44400;44;LOIRE ATLANTIQUE;1
44144;;Riaill;RIAILLE;44440;44;LOIRE ATLANTIQUE;1
44145;;Rouans;ROUANS;44640;44;LOIRE ATLANTIQUE;1
44146;;Roug;ROUGE;44660;44;LOIRE ATLANTIQUE;1
44147;La;Rouxire;ROUXIERE;44370;44;LOIRE ATLANTIQUE;1
44148;;Ruffign;RUFFIGNE;44660;44;LOIRE ATLANTIQUE;1
44149;;Saffr;SAFFRE;44390;44;LOIRE ATLANTIQUE;1
44150;;Saint-Aignan-Grandlieu;SAINT-AIGNAN-GRANDLIEU;44860;44;LOIRE ATLANTIQUE;1
44151;;Saint-Andr-des-Eaux;SAINT-ANDRE-DES-EAUX;44117;44;LOIRE ATLANTIQUE;1
44152;;Sainte-Anne-sur-Brivet;SAINTE-ANNE-SUR-BRIVET;44160;44;LOIRE ATLANTIQUE;1
44153;;Saint-Aubin-des-Chteaux;SAINT-AUBIN-DES-CHATEAUX;44110;44;LOIRE ATLANTIQUE;1
44154;;Saint-Brevin-les-Pins;SAINT-BREVIN-LES-PINS;44250;44;LOIRE ATLANTIQUE;1
44155;;Saint-Colomban;SAINT-COLOMBAN;44310;44;LOIRE ATLANTIQUE;1
44156;;Corcou-sur-Logne;CORCOUE-SUR-LOGNE;44650;44;LOIRE ATLANTIQUE;1
44157;;Saint-tienne-de-Mer-Morte;SAINT-ETIENNE-DE-MER-MORTE;44270;44;LOIRE ATLANTIQUE;1
44158;;Saint-tienne-de-Montluc;SAINT-ETIENNE-DE-MONTLUC;44360;44;LOIRE ATLANTIQUE;1
44159;;Saint-Fiacre-sur-Maine;SAINT-FIACRE-SUR-MAINE;44690;44;LOIRE ATLANTIQUE;1
44160;;Saint-Gron;SAINT-GEREON;44150;44;LOIRE ATLANTIQUE;1
44161;;Saint-Gildas-des-Bois;SAINT-GILDAS-DES-BOIS;44530;44;LOIRE ATLANTIQUE;1
44162;;Saint-Herblain;SAINT-HERBLAIN;44800;44;LOIRE ATLANTIQUE;1
44163;;Saint-Herblon;SAINT-HERBLON;44150;44;LOIRE ATLANTIQUE;1
44164;;Saint-Hilaire-de-Chalons;SAINT-HILAIRE-DE-CHALEONS;44680;44;LOIRE ATLANTIQUE;1
44165;;Saint-Hilaire-de-Clisson;SAINT-HILAIRE-DE-CLISSON;44190;44;LOIRE ATLANTIQUE;1
44166;;Saint-Jean-de-Boiseau;SAINT-JEAN-DE-BOISEAU;44640;44;LOIRE ATLANTIQUE;1
44168;;Saint-Joachim;SAINT-JOACHIM;44720;44;LOIRE ATLANTIQUE;1
44169;;Saint-Julien-de-Concelles;SAINT-JULIEN-DE-CONCELLES;44450;44;LOIRE ATLANTIQUE;1
44170;;Saint-Julien-de-Vouvantes;SAINT-JULIEN-DE-VOUVANTES;44670;44;LOIRE ATLANTIQUE;1
44171;;Saint-Lger-les-Vignes;SAINT-LEGER-LES-VIGNES;44710;44;LOIRE ATLANTIQUE;1
44172;;Sainte-Luce-sur-Loire;SAINTE-LUCE-SUR-LOIRE;44980;44;LOIRE ATLANTIQUE;1
44173;;Saint-Lumine-de-Clisson;SAINT-LUMINE-DE-CLISSON;44190;44;LOIRE ATLANTIQUE;1
44174;;Saint-Lumine-de-Coutais;SAINT-LUMINE-DE-COUTAIS;44310;44;LOIRE ATLANTIQUE;1
44175;;Saint-Lyphard;SAINT-LYPHARD;44410;44;LOIRE ATLANTIQUE;1
44176;;Saint-Malo-de-Guersac;SAINT-MALO-DE-GUERSAC;44550;44;LOIRE ATLANTIQUE;1
44178;;Saint-Mars-de-Coutais;SAINT-MARS-DE-COUTAIS;44680;44;LOIRE ATLANTIQUE;1
44179;;Saint-Mars-du-Dsert;SAINT-MARS-DU-DESERT;44850;44;LOIRE ATLANTIQUE;1
44180;;Saint-Mars-la-Jaille;SAINT-MARS-LA-JAILLE;44540;44;LOIRE ATLANTIQUE;1
44181;;Saint-Mme-le-Tenu;SAINT-MEME-LE-TENU;44270;44;LOIRE ATLANTIQUE;1
44182;;Saint-Michel-Chef-Chef;SAINT-MICHEL-CHEF-CHEF;44730;44;LOIRE ATLANTIQUE;1
44183;;Saint-Molf;SAINT-MOLF;44350;44;LOIRE ATLANTIQUE;1
44184;;Saint-Nazaire;SAINT-NAZAIRE;44600;44;LOIRE ATLANTIQUE;1
44185;;Saint-Nicolas-de-Redon;SAINT-NICOLAS-DE-REDON;44460;44;LOIRE ATLANTIQUE;1
44186;;Sainte-Pazanne;SAINTE-PAZANNE;44680;44;LOIRE ATLANTIQUE;1
44187;;Saint-Pre-en-Retz;SAINT-PERE-EN-RETZ;44320;44;LOIRE ATLANTIQUE;1
44188;;Saint-Philbert-de-Grand-Lieu;SAINT-PHILBERT-DE-GRAND-LIEU;44310;44;LOIRE ATLANTIQUE;1
44189;;Sainte-Reine-de-Bretagne;SAINTE-REINE-DE-BRETAGNE;44160;44;LOIRE ATLANTIQUE;1
44190;;Saint-Sbastien-sur-Loire;SAINT-SEBASTIEN-SUR-LOIRE;44230;44;LOIRE ATLANTIQUE;1
44191;;Saint-Sulpice-des-Landes;SAINT-SULPICE-DES-LANDES;44540;44;LOIRE ATLANTIQUE;1
44192;;Saint-Viaud;SAINT-VIAUD;44320;44;LOIRE ATLANTIQUE;1
44193;;Saint-Vincent-des-Landes;SAINT-VINCENT-DES-LANDES;44590;44;LOIRE ATLANTIQUE;1
44194;;Sautron;SAUTRON;44880;44;LOIRE ATLANTIQUE;1
44195;;Savenay;SAVENAY;44260;44;LOIRE ATLANTIQUE;1
44196;;Svrac;SEVERAC;44530;44;LOIRE ATLANTIQUE;1
44197;;Sion-les-Mines;SION-LES-MINES;44590;44;LOIRE ATLANTIQUE;1
44198;Les;Sorinires;SORINIERES;44840;44;LOIRE ATLANTIQUE;1
44199;;Soudan;SOUDAN;44110;44;LOIRE ATLANTIQUE;1
44200;;Soulvache;SOULVACHE;44660;44;LOIRE ATLANTIQUE;1
44201;;Suc-sur-Erdre;SUCE-SUR-ERDRE;44240;44;LOIRE ATLANTIQUE;1
44202;;Teill;TEILLE;44440;44;LOIRE ATLANTIQUE;1
44203;Le;Temple-de-Bretagne;TEMPLE-DE-BRETAGNE;44360;44;LOIRE ATLANTIQUE;1
44204;;Thouar-sur-Loire;THOUARE-SUR-LOIRE;44470;44;LOIRE ATLANTIQUE;1
44205;Les;Touches;TOUCHES;44390;44;LOIRE ATLANTIQUE;1
44206;;Touvois;TOUVOIS;44650;44;LOIRE ATLANTIQUE;1
44207;;Trans-sur-Erdre;TRANS-SUR-ERDRE;44440;44;LOIRE ATLANTIQUE;1
44208;;Treffieux;TREFFIEUX;44170;44;LOIRE ATLANTIQUE;1
44209;;Treillires;TREILLIERES;44119;44;LOIRE ATLANTIQUE;1
44210;;Trignac;TRIGNAC;44570;44;LOIRE ATLANTIQUE;1
44211;La;Turballe;TURBALLE;44420;44;LOIRE ATLANTIQUE;1
44212;;Vallet;VALLET;44330;44;LOIRE ATLANTIQUE;1
44213;;Varades;VARADES;44370;44;LOIRE ATLANTIQUE;1
44214;;Vay;VAY;44170;44;LOIRE ATLANTIQUE;1
44215;;Vertou;VERTOU;44120;44;LOIRE ATLANTIQUE;1
44216;;Vieillevigne;VIEILLEVIGNE;44116;44;LOIRE ATLANTIQUE;1
44217;;Vigneux-de-Bretagne;VIGNEUX-DE-BRETAGNE;44360;44;LOIRE ATLANTIQUE;1
44218;;Villepot;VILLEPOT;44110;44;LOIRE ATLANTIQUE;1
44219;;Vritz;VRITZ;49440;44;MAINE ET LOIRE;1
44220;;Vue;VUE;44640;44;LOIRE ATLANTIQUE;1
44221;La;Chevallerais;CHEVALLERAIS;44810;44;LOIRE ATLANTIQUE;1
44222;La;Roche-Blanche;ROCHE-BLANCHE;44522;44;LOIRE ATLANTIQUE;1
44223;;Geneston;GENESTON;44140;44;LOIRE ATLANTIQUE;1
44224;La;Grigonnais;GRIGONNAIS;44170;44;LOIRE ATLANTIQUE;1
45001;;Adon;ADON;45230;45;LOIRET;1
45002;;Aillant-sur-Milleron;AILLANT-SUR-MILLERON;45230;45;LOIRET;1
45004;;Amilly;AMILLY;45200;45;LOIRET;1
45005;;Andonville;ANDONVILLE;45480;45;LOIRET;1
45006;;Ardon;ARDON;45160;45;LOIRET;1
45008;;Artenay;ARTENAY;45410;45;LOIRET;1
45009;;Aschres-le-March;ASCHERES-LE-MARCHE;45170;45;LOIRET;1
45010;;Ascoux;ASCOUX;45300;45;LOIRET;1
45011;;Attray;ATTRAY;45170;45;LOIRET;1
45012;;Audeville;AUDEVILLE;45300;45;LOIRET;1
45013;;Augerville-la-Rivire;AUGERVILLE-LA-RIVIERE;45330;45;LOIRET;1
45014;;Aulnay-la-Rivire;AULNAY-LA-RIVIERE;45390;45;LOIRET;1
45015;;Autruy-sur-Juine;AUTRUY-SUR-JUINE;45480;45;LOIRET;1
45016;;Autry-le-Chtel;AUTRY-LE-CHATEL;45500;45;LOIRET;1
45017;;Auvilliers-en-Gtinais;AUVILLIERS-EN-GATINAIS;45270;45;LOIRET;1
45018;;Auxy;AUXY;45340;45;LOIRET;1
45019;;Baccon;BACCON;45130;45;LOIRET;1
45020;Le;Bardon;BARDON;45130;45;LOIRET;1
45021;;Barville-en-Gtinais;BARVILLE-EN-GATINAIS;45340;45;LOIRET;1
45022;;Batilly-en-Gtinais;BATILLY-EN-GATINAIS;45340;45;LOIRET;1
45023;;Batilly-en-Puisaye;BATILLY-EN-PUISAYE;45420;45;LOIRET;1
45024;;Baule;BAULE;45130;45;LOIRET;1
45025;;Bazoches-les-Gallerandes;BAZOCHES-LES-GALLERANDES;45480;45;LOIRET;1
45026;;Bazoches-sur-le-Betz;BAZOCHES-SUR-LE-BETZ;45210;45;LOIRET;1
45027;;Beauchamps-sur-Huillard;BEAUCHAMPS-SUR-HUILLARD;45270;45;LOIRET;1
45028;;Beaugency;BEAUGENCY;45190;45;LOIRET;1
45029;;Beaulieu-sur-Loire;BEAULIEU-SUR-LOIRE;45630;45;LOIRET;1
45030;;Beaune-la-Rolande;BEAUNE-LA-ROLANDE;45340;45;LOIRET;1
45031;;Bellegarde;BELLEGARDE;45270;45;LOIRET;1
45032;Le;Bignon-Mirabeau;BIGNON-MIRABEAU;45210;45;LOIRET;1
45033;;Bosses;BOESSES;45390;45;LOIRET;1
45034;;Boigny-sur-Bionne;BOIGNY-SUR-BIONNE;45760;45;LOIRET;1
45035;;Boiscommun;BOISCOMMUN;45340;45;LOIRET;1
45036;;Boismorand;BOISMORAND;45290;45;LOIRET;1
45037;;Boisseaux;BOISSEAUX;45480;45;LOIRET;1
45038;;Bondaroy;BONDAROY;45300;45;LOIRET;1
45039;;Bonne;BONNEE;45460;45;LOIRET;1
45040;;Bonny-sur-Loire;BONNY-SUR-LOIRE;45420;45;LOIRET;1
45041;;Bordeaux-en-Gtinais;BORDEAUX-EN-GATINAIS;45340;45;LOIRET;1
45042;Les;Bordes;BORDES;45460;45;LOIRET;1
45043;;Bou;BOU;45430;45;LOIRET;1
45044;;Bougy-lez-Neuville;BOUGY-LEZ-NEUVILLE;45170;45;LOIRET;1
45045;;Bouilly-en-Gtinais;BOUILLY-EN-GATINAIS;45300;45;LOIRET;1
45046;;Boulay-les-Barres;BOULAY-LES-BARRES;45140;45;LOIRET;1
45047;;Bouzonville-aux-Bois;BOUZONVILLE-AUX-BOIS;45300;45;LOIRET;1
45049;;Bouzy-la-Fort;BOUZY-LA-FORET;45460;45;LOIRET;1
45050;;Boynes;BOYNES;45300;45;LOIRET;1
45051;;Bray-en-Val;BRAY-EN-VAL;45460;45;LOIRET;1
45052;;Breteau;BRETEAU;45250;45;LOIRET;1
45053;;Briare;BRIARE;45250;45;LOIRET;1
45054;;Briarres-sur-Essonne;BRIARRES-SUR-ESSONNE;45390;45;LOIRET;1
45055;;Bricy;BRICY;45310;45;LOIRET;1
45056;;Bromeilles;BROMEILLES;45390;45;LOIRET;1
45057;;Labrosse;LABROSSE;45330;45;LOIRET;1
45058;;Bucy-le-Roi;BUCY-LE-ROI;45410;45;LOIRET;1
45059;;Bucy-Saint-Liphard;BUCY-SAINT-LIPHARD;45140;45;LOIRET;1
45060;La;Bussire;BUSSIERE;45230;45;LOIRET;1
45061;;Cepoy;CEPOY;45120;45;LOIRET;1
45062;;Cercottes;CERCOTTES;45520;45;LOIRET;1
45063;;Cerdon;CERDON;45620;45;LOIRET;1
45064;;Cernoy-en-Berry;CERNOY-EN-BERRY;45360;45;LOIRET;1
45065;;Csarville-Dossainville;CESARVILLE-DOSSAINVILLE;45300;45;LOIRET;1
45066;;Chailly-en-Gtinais;CHAILLY-EN-GATINAIS;45260;45;LOIRET;1
45067;;Chaingy;CHAINGY;45380;45;LOIRET;1
45068;;Chlette-sur-Loing;CHALETTE-SUR-LOING;45120;45;LOIRET;1
45069;;Chambon-la-Fort;CHAMBON-LA-FORET;45340;45;LOIRET;1
45070;;Champoulet;CHAMPOULET;45420;45;LOIRET;1
45072;;Chanteau;CHANTEAU;45400;45;LOIRET;1
45073;;Chantecoq;CHANTECOQ;45320;45;LOIRET;1
45074;La;Chapelle-Onzerain;CHAPELLE-ONZERAIN;45310;45;LOIRET;1
45075;La;Chapelle-Saint-Mesmin;CHAPELLE-SAINT-MESMIN;45380;45;LOIRET;1
45076;La;Chapelle-Saint-Spulcre;CHAPELLE-SAINT-SEPULCRE;45210;45;LOIRET;1
45077;La;Chapelle-sur-Aveyron;CHAPELLE-SUR-AVEYRON;45230;45;LOIRET;1
45078;;Chapelon;CHAPELON;45270;45;LOIRET;1
45079;Le;Charme;CHARME;45230;45;LOIRET;1
45080;;Charmont-en-Beauce;CHARMONT-EN-BEAUCE;45480;45;LOIRET;1
45081;;Charsonville;CHARSONVILLE;45130;45;LOIRET;1
45082;;Chteauneuf-sur-Loire;CHATEAUNEUF-SUR-LOIRE;45110;45;LOIRET;1
45083;;Chteau-Renard;CHATEAU-RENARD;45220;45;LOIRET;1
45084;;Chtenoy;CHATENOY;45260;45;LOIRET;1
45085;;Chtillon-Coligny;CHATILLON-COLIGNY;45230;45;LOIRET;1
45086;;Chtillon-le-Roi;CHATILLON-LE-ROI;45480;45;LOIRET;1
45087;;Chtillon-sur-Loire;CHATILLON-SUR-LOIRE;45360;45;LOIRET;1
45088;;Chaussy;CHAUSSY;45480;45;LOIRET;1
45089;;Chcy;CHECY;45430;45;LOIRET;1
45091;;Chevannes;CHEVANNES;45210;45;LOIRET;1
45092;;Chevillon-sur-Huillard;CHEVILLON-SUR-HUILLARD;45700;45;LOIRET;1
45093;;Chevilly;CHEVILLY;45520;45;LOIRET;1
45094;;Chevry-sous-le-Bignon;CHEVRY-SOUS-LE-BIGNON;45210;45;LOIRET;1
45095;;Chilleurs-aux-Bois;CHILLEURS-AUX-BOIS;45170;45;LOIRET;1
45096;Les;Choux;CHOUX;45290;45;LOIRET;1
45097;;Chuelles;CHUELLES;45220;45;LOIRET;1
45098;;Clry-Saint-Andr;CLERY-SAINT-ANDRE;45370;45;LOIRET;1
45099;;Coinces;COINCES;45310;45;LOIRET;1
45100;;Combleux;COMBLEUX;45800;45;LOIRET;1
45101;;Combreux;COMBREUX;45530;45;LOIRET;1
45102;;Conflans-sur-Loing;CONFLANS-SUR-LOING;45700;45;LOIRET;1
45103;;Corbeilles;CORBEILLES;45490;45;LOIRET;1
45104;;Corquilleroy;CORQUILLEROY;45120;45;LOIRET;1
45105;;Cortrat;CORTRAT;45700;45;LOIRET;1
45106;;Coudray;COUDRAY;45330;45;LOIRET;1
45107;;Coudroy;COUDROY;45260;45;LOIRET;1
45108;;Coullons;COULLONS;45720;45;LOIRET;1
45109;;Coulmiers;COULMIERS;45130;45;LOIRET;1
45110;;Courcelles;COURCELLES;45300;45;LOIRET;1
45111;;Courcy-aux-Loges;COURCY-AUX-LOGES;45300;45;LOIRET;1
45112;La;Cour-Marigny;COUR-MARIGNY;45260;45;LOIRET;1
45113;;Courtemaux;COURTEMAUX;45320;45;LOIRET;1
45114;;Courtempierre;COURTEMPIERRE;45490;45;LOIRET;1
45115;;Courtenay;COURTENAY;45320;45;LOIRET;1
45116;;Cravant;CRAVANT;45190;45;LOIRET;1
45118;;Crottes-en-Pithiverais;CROTTES-EN-PITHIVERAIS;45170;45;LOIRET;1
45119;;Dadonville;DADONVILLE;45300;45;LOIRET;1
45120;;Dammarie-en-Puisaye;DAMMARIE-EN-PUISAYE;45420;45;LOIRET;1
45121;;Dammarie-sur-Loing;DAMMARIE-SUR-LOING;45230;45;LOIRET;1
45122;;Dampierre-en-Burly;DAMPIERRE-EN-BURLY;45570;45;LOIRET;1
45123;;Darvoy;DARVOY;45150;45;LOIRET;1
45124;;Desmonts;DESMONTS;45390;45;LOIRET;1
45125;;Dimancheville;DIMANCHEVILLE;45390;45;LOIRET;1
45126;;Donnery;DONNERY;45450;45;LOIRET;1
45127;;Dordives;DORDIVES;45680;45;LOIRET;1
45129;;Douchy;DOUCHY;45220;45;LOIRET;1
45130;;Dry;DRY;45370;45;LOIRET;1
45131;;chilleuses;ECHILLEUSES;45390;45;LOIRET;1
45132;;gry;EGRY;45340;45;LOIRET;1
45133;;Engenville;ENGENVILLE;45300;45;LOIRET;1
45134;;pieds-en-Beauce;EPIEDS-EN-BEAUCE;45130;45;LOIRET;1
45135;;Erceville;ERCEVILLE;45480;45;LOIRET;1
45136;;Ervauville;ERVAUVILLE;45320;45;LOIRET;1
45137;;Escrennes;ESCRENNES;45300;45;LOIRET;1
45138;;Escrignelles;ESCRIGNELLES;45250;45;LOIRET;1
45139;;Estouy;ESTOUY;45300;45;LOIRET;1
45141;;Faverelles;FAVERELLES;45420;45;LOIRET;1
45142;;Fay-aux-Loges;FAY-AUX-LOGES;45450;45;LOIRET;1
45143;;Feins-en-Gtinais;FEINS-EN-GATINAIS;45230;45;LOIRET;1
45144;;Frolles;FEROLLES;45150;45;LOIRET;1
45145;;Ferrires-en-Gtinais;FERRIERES-EN-GATINAIS;45210;45;LOIRET;1
45146;La;Fert-Saint-Aubin;FERTE-SAINT-AUBIN;45240;45;LOIRET;1
45147;;Fleury-les-Aubrais;FLEURY-LES-AUBRAIS;45400;45;LOIRET;1
45148;;Fontenay-sur-Loing;FONTENAY-SUR-LOING;45210;45;LOIRET;1
45149;;Foucherolles;FOUCHEROLLES;45320;45;LOIRET;1
45150;;Frville-du-Gtinais;FREVILLE-DU-GATINAIS;45270;45;LOIRET;1
45151;;Gaubertin;GAUBERTIN;45340;45;LOIRET;1
45152;;Gmigny;GEMIGNY;45310;45;LOIRET;1
45153;;Germigny-des-Prs;GERMIGNY-DES-PRES;45110;45;LOIRET;1
45154;;Gidy;GIDY;45520;45;LOIRET;1
45155;;Gien;GIEN;45500;45;LOIRET;1
45156;;Girolles;GIROLLES;45120;45;LOIRET;1
45157;;Givraines;GIVRAINES;45300;45;LOIRET;1
45158;;Gondreville;GONDREVILLE;45490;45;LOIRET;1
45159;;Grangermont;GRANGERMONT;45390;45;LOIRET;1
45160;;Greneville-en-Beauce;GRENEVILLE-EN-BEAUCE;45480;45;LOIRET;1
45161;;Griselles;GRISELLES;45210;45;LOIRET;1
45162;;Guigneville;GUIGNEVILLE;45300;45;LOIRET;1
45164;;Guilly;GUILLY;45600;45;LOIRET;1
45165;;Gy-les-Nonains;GY-LES-NONAINS;45220;45;LOIRET;1
45166;;Hutre;HUETRE;45520;45;LOIRET;1
45167;;Huisseau-sur-Mauves;HUISSEAU-SUR-MAUVES;45130;45;LOIRET;1
45168;;Ingrannes;INGRANNES;45450;45;LOIRET;1
45169;;Ingr;INGRE;45140;45;LOIRET;1
45170;;Intville-la-Gutard;INTVILLE-LA-GUETARD;45300;45;LOIRET;1
45171;;Isdes;ISDES;45620;45;LOIRET;1
45173;;Jargeau;JARGEAU;45150;45;LOIRET;1
45174;;Jouy-en-Pithiverais;JOUY-EN-PITHIVERAIS;45480;45;LOIRET;1
45175;;Jouy-le-Potier;JOUY-LE-POTIER;45370;45;LOIRET;1
45176;;Juranville;JURANVILLE;45340;45;LOIRET;1
45177;;Laas;LAAS;45300;45;LOIRET;1
45178;;Ladon;LADON;45270;45;LOIRET;1
45179;;Lailly-en-Val;LAILLY-EN-VAL;45740;45;LOIRET;1
45180;;Langesse;LANGESSE;45290;45;LOIRET;1
45181;;Louville;LEOUVILLE;45480;45;LOIRET;1
45182;;Ligny-le-Ribault;LIGNY-LE-RIBAULT;45240;45;LOIRET;1
45183;;Lion-en-Beauce;LION-EN-BEAUCE;45410;45;LOIRET;1
45184;;Lion-en-Sullias;LION-EN-SULLIAS;45600;45;LOIRET;1
45185;;Lombreuil;LOMBREUIL;45700;45;LOIRET;1
45186;;Lorcy;LORCY;45490;45;LOIRET;1
45187;;Lorris;LORRIS;45260;45;LOIRET;1
45188;;Loury;LOURY;45470;45;LOIRET;1
45189;;Louzouer;LOUZOUER;45210;45;LOIRET;1
45190;;Mainvilliers;MAINVILLIERS;45330;45;LOIRET;1
45191;;Malesherbes;MALESHERBES;45330;45;LOIRET;1
45192;;Manchecourt;MANCHECOURT;45300;45;LOIRET;1
45193;;Marcilly-en-Villette;MARCILLY-EN-VILLETTE;45240;45;LOIRET;1
45194;;Mardi;MARDIE;45430;45;LOIRET;1
45195;;Mareau-aux-Bois;MAREAU-AUX-BOIS;45300;45;LOIRET;1
45196;;Mareau-aux-Prs;MAREAU-AUX-PRES;45370;45;LOIRET;1
45197;;Marigny-les-Usages;MARIGNY-LES-USAGES;45760;45;LOIRET;1
45198;;Marsainvilliers;MARSAINVILLIERS;45300;45;LOIRET;1
45199;;Melleroy;MELLEROY;45220;45;LOIRET;1
45200;;Mnestreau-en-Villette;MENESTREAU-EN-VILLETTE;45240;45;LOIRET;1
45201;;Mrinville;MERINVILLE;45210;45;LOIRET;1
45202;;Messas;MESSAS;45190;45;LOIRET;1
45203;;Meung-sur-Loire;MEUNG-SUR-LOIRE;45130;45;LOIRET;1
45204;;Mzires-lez-Clry;MEZIERES-LEZ-CLERY;45370;45;LOIRET;1
45205;;Mzires-en-Gtinais;MEZIERES-EN-GATINAIS;45270;45;LOIRET;1
45206;;Mignres;MIGNERES;45490;45;LOIRET;1
45207;;Mignerette;MIGNERETTE;45490;45;LOIRET;1
45208;;Montargis;MONTARGIS;45200;45;LOIRET;1
45209;;Montbarrois;MONTBARROIS;45340;45;LOIRET;1
45210;;Montbouy;MONTBOUY;45230;45;LOIRET;1
45211;;Montcorbon;MONTCORBON;45220;45;LOIRET;1
45212;;Montcresson;MONTCRESSON;45700;45;LOIRET;1
45213;;Montereau;MONTEREAU;45260;45;LOIRET;1
45214;;Montigny;MONTIGNY;45170;45;LOIRET;1
45215;;Montliard;MONTLIARD;45340;45;LOIRET;1
45216;;Mormant-sur-Vernisson;MORMANT-SUR-VERNISSON;45700;45;LOIRET;1
45217;;Morville-en-Beauce;MORVILLE-EN-BEAUCE;45300;45;LOIRET;1
45218;Le;Moulinet-sur-Solin;MOULINET-SUR-SOLIN;45290;45;LOIRET;1
45219;;Moulon;MOULON;45270;45;LOIRET;1
45220;;Nancray-sur-Rimarde;NANCRAY-SUR-RIMARDE;45340;45;LOIRET;1
45221;;Nangeville;NANGEVILLE;45330;45;LOIRET;1
45222;;Nargis;NARGIS;45210;45;LOIRET;1
45223;;Nesploy;NESPLOY;45270;45;LOIRET;1
45224;;Neuville-aux-Bois;NEUVILLE-AUX-BOIS;45170;45;LOIRET;1
45225;La;Neuville-sur-Essonne;NEUVILLE-SUR-ESSONNE;45390;45;LOIRET;1
45226;;Neuvy-en-Sullias;NEUVY-EN-SULLIAS;45510;45;LOIRET;1
45227;;Nevoy;NEVOY;45500;45;LOIRET;1
45228;;Nibelle;NIBELLE;45340;45;LOIRET;1
45229;;Nogent-sur-Vernisson;NOGENT-SUR-VERNISSON;45290;45;LOIRET;1
45230;;Noyers;NOYERS;45260;45;LOIRET;1
45231;;Oison;OISON;45170;45;LOIRET;1
45232;;Olivet;OLIVET;45160;45;LOIRET;1
45233;;Ondreville-sur-Essonne;ONDREVILLE-SUR-ESSONNE;45390;45;LOIRET;1
45234;;Orlans;ORLEANS;45000;45;LOIRET;1
45235;;Ormes;ORMES;45140;45;LOIRET;1
45236;;Orveau-Bellesauve;ORVEAU-BELLESAUVE;45330;45;LOIRET;1
45237;;Orville;ORVILLE;45390;45;LOIRET;1
45238;;Ousson-sur-Loire;OUSSON-SUR-LOIRE;45250;45;LOIRET;1
45239;;Oussoy-en-Gtinais;OUSSOY-EN-GATINAIS;45290;45;LOIRET;1
45240;;Outarville;OUTARVILLE;45480;45;LOIRET;1
45241;;Ouvrouer-les-Champs;OUVROUER-LES-CHAMPS;45150;45;LOIRET;1
45242;;Ouzouer-des-Champs;OUZOUER-DES-CHAMPS;45290;45;LOIRET;1
45243;;Ouzouer-sous-Bellegarde;OUZOUER-SOUS-BELLEGARDE;45270;45;LOIRET;1
45244;;Ouzouer-sur-Loire;OUZOUER-SUR-LOIRE;45570;45;LOIRET;1
45245;;Ouzouer-sur-Trze;OUZOUER-SUR-TREZEE;45250;45;LOIRET;1
45246;;Pannecires;PANNECIERES;45300;45;LOIRET;1
45247;;Pannes;PANNES;45700;45;LOIRET;1
45248;;Patay;PATAY;45310;45;LOIRET;1
45249;;Paucourt;PAUCOURT;45200;45;LOIRET;1
45250;;Pers-en-Gtinais;PERS-EN-GATINAIS;45210;45;LOIRET;1
45251;;Pierrefitte-s-Bois;PIERREFITTE-ES-BOIS;45360;45;LOIRET;1
45252;;Pithiviers;PITHIVIERS;45300;45;LOIRET;1
45253;;Pithiviers-le-Vieil;PITHIVIERS-LE-VIEIL;45300;45;LOIRET;1
45254;;Poilly-lez-Gien;POILLY-LEZ-GIEN;45500;45;LOIRET;1
45255;;Prfontaines;PREFONTAINES;45490;45;LOIRET;1
45256;;Presnoy;PRESNOY;45260;45;LOIRET;1
45257;;Pressigny-les-Pins;PRESSIGNY-LES-PINS;45290;45;LOIRET;1
45258;;Puiseaux;PUISEAUX;45390;45;LOIRET;1
45259;;Quiers-sur-Bzonde;QUIERS-SUR-BEZONDE;45270;45;LOIRET;1
45260;;Ramoulu;RAMOULU;45300;45;LOIRET;1
45261;;Rebrchien;REBRECHIEN;45470;45;LOIRET;1
45262;;Rouvray-Sainte-Croix;ROUVRAY-SAINTE-CROIX;45310;45;LOIRET;1
45263;;Rouvres-Saint-Jean;ROUVRES-SAINT-JEAN;45300;45;LOIRET;1
45264;;Rozires-en-Beauce;ROZIERES-EN-BEAUCE;45130;45;LOIRET;1
45265;;Rozoy-le-Vieil;ROZOY-LE-VIEIL;45210;45;LOIRET;1
45266;;Ruan;RUAN;45410;45;LOIRET;1
45267;;Saint-Aignan-des-Gus;SAINT-AIGNAN-DES-GUES;45460;45;LOIRET;1
45268;;Saint-Aignan-le-Jaillard;SAINT-AIGNAN-LE-JAILLARD;45600;45;LOIRET;1
45269;;Saint-Ay;SAINT-AY;45130;45;LOIRET;1
45270;;Saint-Benot-sur-Loire;SAINT-BENOIT-SUR-LOIRE;45730;45;LOIRET;1
45271;;Saint-Brisson-sur-Loire;SAINT-BRISSON-SUR-LOIRE;45500;45;LOIRET;1
45272;;Saint-Cyr-en-Val;SAINT-CYR-EN-VAL;45590;45;LOIRET;1
45273;;Saint-Denis-de-l'Htel;SAINT-DENIS-DE-L'HOTEL;45550;45;LOIRET;1
45274;;Saint-Denis-en-Val;SAINT-DENIS-EN-VAL;45560;45;LOIRET;1
45275;;Saint-Firmin-des-Bois;SAINT-FIRMIN-DES-BOIS;45220;45;LOIRET;1
45276;;Saint-Firmin-sur-Loire;SAINT-FIRMIN-SUR-LOIRE;45360;45;LOIRET;1
45277;;Saint-Florent;SAINT-FLORENT;45600;45;LOIRET;1
45278;;Sainte-Genevive-des-Bois;SAINTE-GENEVIEVE-DES-BOIS;45230;45;LOIRET;1
45279;;Saint-Germain-des-Prs;SAINT-GERMAIN-DES-PRES;45220;45;LOIRET;1
45280;;Saint-Gondon;SAINT-GONDON;45500;45;LOIRET;1
45281;;Saint-Hilaire-les-Andrsis;SAINT-HILAIRE-LES-ANDRESIS;45320;45;LOIRET;1
45282;;Saint-Hilaire-Saint-Mesmin;SAINT-HILAIRE-SAINT-MESMIN;45160;45;LOIRET;1
45283;;Saint-Hilaire-sur-Puiseaux;SAINT-HILAIRE-SUR-PUISEAUX;45700;45;LOIRET;1
45284;;Saint-Jean-de-Braye;SAINT-JEAN-DE-BRAYE;45800;45;LOIRET;1
45285;;Saint-Jean-de-la-Ruelle;SAINT-JEAN-DE-LA-RUELLE;45140;45;LOIRET;1
45286;;Saint-Jean-le-Blanc;SAINT-JEAN-LE-BLANC;45650;45;LOIRET;1
45287;;Saint-Loup-de-Gonois;SAINT-LOUP-DE-GONOIS;45210;45;LOIRET;1
45288;;Saint-Loup-des-Vignes;SAINT-LOUP-DES-VIGNES;45340;45;LOIRET;1
45289;;Saint-Ly-la-Fort;SAINT-LYE-LA-FORET;45170;45;LOIRET;1
45290;;Saint-Martin-d'Abbat;SAINT-MARTIN-D'ABBAT;45110;45;LOIRET;1
45291;;Saint-Martin-sur-Ocre;SAINT-MARTIN-SUR-OCRE;45500;45;LOIRET;1
45292;;Saint-Maurice-sur-Aveyron;SAINT-MAURICE-SUR-AVEYRON;45230;45;LOIRET;1
45293;;Saint-Maurice-sur-Fessard;SAINT-MAURICE-SUR-FESSARD;45700;45;LOIRET;1
45294;;Saint-Michel;SAINT-MICHEL;45340;45;LOIRET;1
45296;;Saint-Pravy-la-Colombe;SAINT-PERAVY-LA-COLOMBE;45310;45;LOIRET;1
45297;;Saint-Pre-sur-Loire;SAINT-PERE-SUR-LOIRE;45600;45;LOIRET;1
45298;;Saint-Pryv-Saint-Mesmin;SAINT-PRYVE-SAINT-MESMIN;45750;45;LOIRET;1
45299;;Saint-Sigismond;SAINT-SIGISMOND;45310;45;LOIRET;1
45300;;Sandillon;SANDILLON;45640;45;LOIRET;1
45301;;Santeau;SANTEAU;45170;45;LOIRET;1
45302;;Saran;SARAN;45770;45;LOIRET;1
45303;;Sceaux-du-Gtinais;SCEAUX-DU-GATINAIS;45490;45;LOIRET;1
45305;;Seichebrires;SEICHEBRIERES;45530;45;LOIRET;1
45306;La;Selle-en-Hermoy;SELLE-EN-HERMOY;45210;45;LOIRET;1
45307;La;Selle-sur-le-Bied;SELLE-SUR-LE-BIED;45210;45;LOIRET;1
45308;;Semoy;SEMOY;45400;45;LOIRET;1
45309;;Sennely;SENNELY;45240;45;LOIRET;1
45310;;Sermaises;SERMAISES;45300;45;LOIRET;1
45311;;Sigloy;SIGLOY;45110;45;LOIRET;1
45312;;Solterre;SOLTERRE;45700;45;LOIRET;1
45313;;Sougy;SOUGY;45410;45;LOIRET;1
45314;;Sully-la-Chapelle;SULLY-LA-CHAPELLE;45450;45;LOIRET;1
45315;;Sully-sur-Loire;SULLY-SUR-LOIRE;45600;45;LOIRET;1
45316;;Sury-aux-Bois;SURY-AUX-BOIS;45530;45;LOIRET;1
45317;;Tavers;TAVERS;45190;45;LOIRET;1
45320;;Thignonville;THIGNONVILLE;45300;45;LOIRET;1
45321;;Thimory;THIMORY;45260;45;LOIRET;1
45322;;Thorailles;THORAILLES;45210;45;LOIRET;1
45323;;Thou;THOU;45420;45;LOIRET;1
45324;;Tigy;TIGY;45510;45;LOIRET;1
45325;;Tivernon;TIVERNON;45170;45;LOIRET;1
45326;;Tournoisis;TOURNOISIS;45310;45;LOIRET;1
45327;;Tranou;TRAINOU;45470;45;LOIRET;1
45328;;Treilles-en-Gtinais;TREILLES-EN-GATINAIS;45490;45;LOIRET;1
45329;;Trigures;TRIGUERES;45220;45;LOIRET;1
45330;;Trinay;TRINAY;45410;45;LOIRET;1
45331;;Vannes-sur-Cosson;VANNES-SUR-COSSON;45510;45;LOIRET;1
45332;;Varennes-Changy;VARENNES-CHANGY;45290;45;LOIRET;1
45333;;Vennecy;VENNECY;45760;45;LOIRET;1
45334;;Vieilles-Maisons-sur-Joudry;VIEILLES-MAISONS-SUR-JOUDRY;45260;45;LOIRET;1
45335;;Vienne-en-Val;VIENNE-EN-VAL;45510;45;LOIRET;1
45336;;Viglain;VIGLAIN;45600;45;LOIRET;1
45337;;Villamblain;VILLAMBLAIN;45310;45;LOIRET;1
45338;;Villemandeur;VILLEMANDEUR;45700;45;LOIRET;1
45339;;Villemoutiers;VILLEMOUTIERS;45270;45;LOIRET;1
45340;;Villemurlin;VILLEMURLIN;45600;45;LOIRET;1
45341;;Villeneuve-sur-Conie;VILLENEUVE-SUR-CONIE;45310;45;LOIRET;1
45342;;Villereau;VILLEREAU;45170;45;LOIRET;1
45343;;Villevoques;VILLEVOQUES;45700;45;LOIRET;1
45344;;Villorceau;VILLORCEAU;45190;45;LOIRET;1
45345;;Vimory;VIMORY;45700;45;LOIRET;1
45346;;Vitry-aux-Loges;VITRY-AUX-LOGES;45530;45;LOIRET;1
45347;;Vrigny;VRIGNY;45300;45;LOIRET;1
45348;;Yvre-la-Ville;YEVRE-LA-VILLE;45300;45;LOIRET;1
46001;;Albas;ALBAS;46140;46;LOT;1
46002;;Albiac;ALBIAC;46500;46;LOT;1
46003;;Alvignac;ALVIGNAC;46500;46;LOT;1
46004;;Anglars;ANGLARS;46120;46;LOT;1
46005;;Anglars-Juillac;ANGLARS-JUILLAC;46140;46;LOT;1
46006;;Anglars-Nozac;ANGLARS-NOZAC;46300;46;LOT;1
46007;;Arcambal;ARCAMBAL;46090;46;LOT;1
46008;Les;Arques;ARQUES;46250;46;LOT;1
46009;;Assier;ASSIER;46320;46;LOT;1
46010;;Aujols;AUJOLS;46090;46;LOT;1
46011;;Autoire;AUTOIRE;46400;46;LOT;1
46012;;Aynac;AYNAC;46120;46;LOT;1
46013;;Bach;BACH;46230;46;LOT;1
46014;;Bagat-en-Quercy;BAGAT-EN-QUERCY;46800;46;LOT;1
46015;;Bagnac-sur-Cl;BAGNAC-SUR-CELE;46270;46;LOT;1
46016;;Baladou;BALADOU;46600;46;LOT;1
46017;;Bannes;BANNES;46400;46;LOT;1
46018;Le;Bastit;BASTIT;46500;46;LOT;1
46019;;Beaumat;BEAUMAT;46240;46;LOT;1
46020;;Beauregard;BEAUREGARD;46260;46;LOT;1
46021;;Bduer;BEDUER;46100;46;LOT;1
46022;;Blaye;BELAYE;46140;46;LOT;1
46023;;Belfort-du-Quercy;BELFORT-DU-QUERCY;46230;46;LOT;1
46024;;Belmont-Bretenoux;BELMONT-BRETENOUX;46130;46;LOT;1
46025;;Belmontet;BELMONTET;46800;46;LOT;1
46026;;Belmont-Sainte-Foi;BELMONT-SAINTE-FOI;46230;46;LOT;1
46027;;Berganty;BERGANTY;46090;46;LOT;1
46028;;Btaille;BETAILLE;46110;46;LOT;1
46029;;Biars-sur-Cre;BIARS-SUR-CERE;46130;46;LOT;1
46030;;Bio;BIO;46500;46;LOT;1
46031;;Blars;BLARS;46330;46;LOT;1
46032;;Boissires;BOISSIERES;46150;46;LOT;1
46033;Le;Boulv;BOULVE;46800;46;LOT;1
46034;Le;Bourg;BOURG;46120;46;LOT;1
46035;;Boussac;BOUSSAC;46100;46;LOT;1
46036;Le;Bouyssou;BOUYSSOU;46120;46;LOT;1
46037;;Bouzis;BOUZIES;46330;46;LOT;1
46038;;Bretenoux;BRETENOUX;46130;46;LOT;1
46039;;Brengues;BRENGUES;46320;46;LOT;1
46040;;Cabrerets;CABRERETS;46330;46;LOT;1
46041;;Cadrieu;CADRIEU;46160;46;LOT;1
46042;;Cahors;CAHORS;46000;46;LOT;1
46043;;Cahus;CAHUS;46130;46;LOT;1
46044;;Caillac;CAILLAC;46140;46;LOT;1
46045;;Cajarc;CAJARC;46160;46;LOT;1
46046;;Calamane;CALAMANE;46150;46;LOT;1
46047;;Cals;CALES;46350;46;LOT;1
46048;;Calviac;CALVIAC;46190;46;LOT;1
46049;;Calvignac;CALVIGNAC;46160;46;LOT;1
46050;;Cambayrac;CAMBAYRAC;46140;46;LOT;1
46051;;Cambes;CAMBES;46100;46;LOT;1
46052;;Camboulit;CAMBOULIT;46100;46;LOT;1
46053;;Camburat;CAMBURAT;46100;46;LOT;1
46054;;Caniac-du-Causse;CANIAC-DU-CAUSSE;46240;46;LOT;1
46055;;Capdenac;CAPDENAC;46100;46;LOT;1
46056;;Carayac;CARAYAC;46160;46;LOT;1
46057;;Cardaillac;CARDAILLAC;46100;46;LOT;1
46058;;Carennac;CARENNAC;46110;46;LOT;1
46059;;Carlucet;CARLUCET;46500;46;LOT;1
46060;;Carnac-Rouffiac;CARNAC-ROUFFIAC;46140;46;LOT;1
46061;;Cassagnes;CASSAGNES;46700;46;LOT;1
46062;;Castelfranc;CASTELFRANC;46140;46;LOT;1
46063;;Castelnau-Montratier;CASTELNAU-MONTRATIER;46170;46;LOT;1
46064;;Catus;CATUS;46150;46;LOT;1
46065;;Cavagnac;CAVAGNAC;46110;46;LOT;1
46066;;Cazals;CAZALS;46250;46;LOT;1
46067;;Cazillac;CAZILLAC;46600;46;LOT;1
46068;;Cnevires;CENEVIERES;46330;46;LOT;1
46069;;Czac;CEZAC;46170;46;LOT;1
46070;;Cieurac;CIEURAC;46230;46;LOT;1
46071;;Comiac;COMIAC;46190;46;LOT;1
46072;;Concors;CONCORES;46310;46;LOT;1
46073;;Concots;CONCOTS;46260;46;LOT;1
46074;;Condat;CONDAT;46110;46;LOT;1
46075;;Corn;CORN;46100;46;LOT;1
46076;;Cornac;CORNAC;46130;46;LOT;1
46077;;Cours;COURS;46090;46;LOT;1
46078;;Couzou;COUZOU;46500;46;LOT;1
46079;;Cras;CRAS;46360;46;LOT;1
46080;;Crayssac;CRAYSSAC;46150;46;LOT;1
46081;;Crgols;CREGOLS;46330;46;LOT;1
46082;;Cremps;CREMPS;46230;46;LOT;1
46083;;Cressensac;CRESSENSAC;46600;46;LOT;1
46084;;Creysse;CREYSSE;46600;46;LOT;1
46085;;Cuzac;CUZAC;46270;46;LOT;1
46086;;Cuzance;CUZANCE;46600;46;LOT;1
46087;;Dgagnac;DEGAGNAC;46340;46;LOT;1
46088;;Douelle;DOUELLE;46140;46;LOT;1
46089;;Duravel;DURAVEL;46700;46;LOT;1
46090;;Durbans;DURBANS;46320;46;LOT;1
46091;;Escamps;ESCAMPS;46230;46;LOT;1
46092;;Esclauzels;ESCLAUZELS;46090;46;LOT;1
46093;;Espagnac-Sainte-Eulalie;ESPAGNAC-SAINTE-EULALIE;46320;46;LOT;1
46094;;Espdaillac;ESPEDAILLAC;46320;46;LOT;1
46095;;Espre;ESPERE;46090;46;LOT;1
46096;;Espeyroux;ESPEYROUX;46120;46;LOT;1
46097;;Estal;ESTAL;46130;46;LOT;1
46098;;Fajoles;FAJOLES;46300;46;LOT;1
46099;;Fargues;FARGUES;46800;46;LOT;1
46100;;Faycelles;FAYCELLES;46100;46;LOT;1
46101;;Felzins;FELZINS;46270;46;LOT;1
46102;;Figeac;FIGEAC;46100;46;LOT;1
46103;;Flaugnac;FLAUGNAC;46170;46;LOT;1
46104;;Flaujac-Gare;FLAUJAC-GARE;46320;46;LOT;1
46105;;Flaujac-Poujols;FLAUJAC-POUJOLS;46090;46;LOT;1
46106;;Floirac;FLOIRAC;46600;46;LOT;1
46107;;Floressas;FLORESSAS;46700;46;LOT;1
46108;;Fons;FONS;46100;46;LOT;1
46109;;Fontanes;FONTANES;46230;46;LOT;1
46110;;Fontanes-du-Causse;FONTANES-DU-CAUSSE;46240;46;LOT;1
46111;;Fourmagnac;FOURMAGNAC;46100;46;LOT;1
46112;;Francouls;FRANCOULES;46090;46;LOT;1
46113;;Frayssinet;FRAYSSINET;46310;46;LOT;1
46114;;Frayssinet-le-Glat;FRAYSSINET-LE-GELAT;46250;46;LOT;1
46115;;Frayssinhes;FRAYSSINHES;46400;46;LOT;1
46116;;Frontenac;FRONTENAC;46160;46;LOT;1
46117;;Gagnac-sur-Cre;GAGNAC-SUR-CERE;46130;46;LOT;1
46118;;Gignac;GIGNAC;46600;46;LOT;1
46119;;Gigouzac;GIGOUZAC;46150;46;LOT;1
46120;;Gindou;GINDOU;46250;46;LOT;1
46121;;Ginouillac;GINOUILLAC;46300;46;LOT;1
46122;;Gintrac;GINTRAC;46130;46;LOT;1
46123;;Girac;GIRAC;46130;46;LOT;1
46124;;Glanes;GLANES;46130;46;LOT;1
46125;;Gorses;GORSES;46210;46;LOT;1
46126;;Goujounac;GOUJOUNAC;46250;46;LOT;1
46127;;Gourdon;GOURDON;46300;46;LOT;1
46128;;Gramat;GRAMAT;46500;46;LOT;1
46129;;Gralou;GREALOU;46160;46;LOT;1
46130;;Grzels;GREZELS;46700;46;LOT;1
46131;;Grzes;GREZES;46320;46;LOT;1
46132;;Issendolus;ISSENDOLUS;46500;46;LOT;1
46133;;Issepts;ISSEPTS;46320;46;LOT;1
46134;Les;Junies;JUNIES;46150;46;LOT;1
46135;;Labastide-du-Haut-Mont;LABASTIDE-DU-HAUT-MONT;46210;46;LOT;1
46136;;Labastide-du-Vert;LABASTIDE-DU-VERT;46150;46;LOT;1
46137;;Labastide-Marnhac;LABASTIDE-MARNHAC;46090;46;LOT;1
46138;;Labastide-Murat;LABASTIDE-MURAT;46240;46;LOT;1
46139;;Labathude;LABATHUDE;46120;46;LOT;1
46140;;Laburgade;LABURGADE;46230;46;LOT;1
46141;;Lacam-d'Ourcet;LACAM-D'OURCET;46190;46;LOT;1
46142;;Lacapelle-Cabanac;LACAPELLE-CABANAC;46700;46;LOT;1
46143;;Lacapelle-Marival;LACAPELLE-MARIVAL;46120;46;LOT;1
46144;;Lacave;LACAVE;46200;46;LOT;1
46145;;Lachapelle-Auzac;LACHAPELLE-AUZAC;46200;46;LOT;1
46146;;Ladirat;LADIRAT;46400;46;LOT;1
46147;;Lagardelle;LAGARDELLE;46220;46;LOT;1
46148;;Lalbenque;LALBENQUE;46230;46;LOT;1
46149;;Lamagdelaine;LAMAGDELAINE;46090;46;LOT;1
46150;;Lamativie;LAMATIVIE;46190;46;LOT;1
46151;;Lamothe-Cassel;LAMOTHE-CASSEL;46240;46;LOT;1
46152;;Lamothe-Fnelon;LAMOTHE-FENELON;46350;46;LOT;1
46153;;Lanzac;LANZAC;46200;46;LOT;1
46154;;Laramire;LARAMIERE;46260;46;LOT;1
46155;;Larnagol;LARNAGOL;46160;46;LOT;1
46156;;Laroque-des-Arcs;LAROQUE-DES-ARCS;46090;46;LOT;1
46157;;Larroque-Toirac;LARROQUE-TOIRAC;46160;46;LOT;1
46158;;Lascabanes;LASCABANES;46800;46;LOT;1
46159;;Latouille-Lentillac;LATOUILLE-LENTILLAC;46400;46;LOT;1
46160;;Latronquire;LATRONQUIERE;46210;46;LOT;1
46161;;Lauresses;LAURESSES;46210;46;LOT;1
46162;;Lauzs;LAUZES;46360;46;LOT;1
46163;;Laval-de-Cre;LAVAL-DE-CERE;46130;46;LOT;1
46164;;Lavercantire;LAVERCANTIERE;46340;46;LOT;1
46165;;Lavergne;LAVERGNE;46500;46;LOT;1
46166;;Lebreil;LEBREIL;46800;46;LOT;1
46167;;Lentillac-du-Causse;LENTILLAC-DU-CAUSSE;46330;46;LOT;1
46168;;Lentillac-Saint-Blaise;LENTILLAC-SAINT-BLAISE;46100;46;LOT;1
46169;;Lobard;LEOBARD;46300;46;LOT;1
46170;;Leyme;LEYME;46120;46;LOT;1
46171;;Lherm;LHERM;46150;46;LOT;1
46172;;Lhospitalet;LHOSPITALET;46170;46;LOT;1
46173;;Limogne-en-Quercy;LIMOGNE-EN-QUERCY;46260;46;LOT;1
46174;;Linac;LINAC;46270;46;LOT;1
46175;;Lissac-et-Mouret;LISSAC-ET-MOURET;46100;46;LOT;1
46176;;Livernon;LIVERNON;46320;46;LOT;1
46177;;Loubressac;LOUBRESSAC;46130;46;LOT;1
46178;;Loupiac;LOUPIAC;46350;46;LOT;1
46179;;Lugagnac;LUGAGNAC;46260;46;LOT;1
46180;;Lunan;LUNAN;46100;46;LOT;1
46181;;Lunegarde;LUNEGARDE;46240;46;LOT;1
46182;;Luzech;LUZECH;46140;46;LOT;1
46183;;Marcilhac-sur-Cl;MARCILHAC-SUR-CELE;46160;46;LOT;1
46184;;Marminiac;MARMINIAC;46250;46;LOT;1
46185;;Martel;MARTEL;46600;46;LOT;1
46186;;Masclat;MASCLAT;46350;46;LOT;1
46187;;Mauroux;MAUROUX;46700;46;LOT;1
46188;;Maxou;MAXOU;46090;46;LOT;1
46189;;Mayrinhac-Lentour;MAYRINHAC-LENTOUR;46500;46;LOT;1
46190;;Mechmont;MECHMONT;46150;46;LOT;1
46191;;Mercus;MERCUES;46090;46;LOT;1
46192;;Meyronne;MEYRONNE;46200;46;LOT;1
46193;;Miers;MIERS;46500;46;LOT;1
46194;;Milhac;MILHAC;46300;46;LOT;1
46195;;Molires;MOLIERES;46120;46;LOT;1
46196;;Montamel;MONTAMEL;46310;46;LOT;1
46197;Le;Montat;MONTAT;46090;46;LOT;1
46198;;Montbrun;MONTBRUN;46160;46;LOT;1
46199;;Montcabrier;MONTCABRIER;46700;46;LOT;1
46200;;Montclra;MONTCLERA;46250;46;LOT;1
46201;;Montcuq;MONTCUQ;46800;46;LOT;1
46202;;Montdoumerc;MONTDOUMERC;46230;46;LOT;1
46203;;Montet-et-Bouxal;MONTET-ET-BOUXAL;46210;46;LOT;1
46204;;Montfaucon;MONTFAUCON;46240;46;LOT;1
46205;;Montgesty;MONTGESTY;46150;46;LOT;1
46206;;Montlauzun;MONTLAUZUN;46800;46;LOT;1
46207;;Montredon;MONTREDON;46270;46;LOT;1
46208;;Montvalent;MONTVALENT;46600;46;LOT;1
46209;;Nadaillac-de-Rouge;NADAILLAC-DE-ROUGE;46350;46;LOT;1
46210;;Nadillac;NADILLAC;46360;46;LOT;1
46211;;Nuzjouls;NUZEJOULS;46150;46;LOT;1
46212;;Orniac;ORNIAC;46330;46;LOT;1
46213;;Padirac;PADIRAC;46500;46;LOT;1
46214;;Parnac;PARNAC;46140;46;LOT;1
46215;;Payrac;PAYRAC;46350;46;LOT;1
46216;;Payrignac;PAYRIGNAC;46300;46;LOT;1
46217;;Pern;PERN;46170;46;LOT;1
46218;;Pescadoires;PESCADOIRES;46220;46;LOT;1
46219;;Peyrilles;PEYRILLES;46310;46;LOT;1
46220;;Pinsac;PINSAC;46200;46;LOT;1
46221;;Planioles;PLANIOLES;46100;46;LOT;1
46222;;Pomarde;POMAREDE;46250;46;LOT;1
46223;;Pontcirq;PONTCIRQ;46150;46;LOT;1
46224;;Pradines;PRADINES;46090;46;LOT;1
46225;;Prayssac;PRAYSSAC;46220;46;LOT;1
46226;;Prendeignes;PRENDEIGNES;46270;46;LOT;1
46227;;Promilhanes;PROMILHANES;46260;46;LOT;1
46228;;Prudhomat;PRUDHOMAT;46130;46;LOT;1
46229;;Puybrun;PUYBRUN;46130;46;LOT;1
46230;;Puyjourdes;PUYJOURDES;46260;46;LOT;1
46231;;Puy-l'vque;PUY-L'EVEQUE;46700;46;LOT;1
46232;Les;Quatre-Routes-du-Lot;QUATRE-ROUTES-DU-LOT;46110;46;LOT;1
46233;;Quissac;QUISSAC;46320;46;LOT;1
46234;;Rampoux;RAMPOUX;46340;46;LOT;1
46235;;Reilhac;REILHAC;46500;46;LOT;1
46236;;Reilhaguet;REILHAGUET;46350;46;LOT;1
46237;;Reyrevignes;REYREVIGNES;46320;46;LOT;1
46238;;Rignac;RIGNAC;46500;46;LOT;1
46239;Le;Roc;ROC;46200;46;LOT;1
46240;;Rocamadour;ROCAMADOUR;46500;46;LOT;1
46241;;Rouffilhac;ROUFFILHAC;46300;46;LOT;1
46242;;Rudelle;RUDELLE;46120;46;LOT;1
46243;;Rueyres;RUEYRES;46120;46;LOT;1
46244;;Sabadel-Latronquire;SABADEL-LATRONQUIERE;46210;46;LOT;1
46245;;Sabadel-Lauzs;SABADEL-LAUZES;46360;46;LOT;1
46246;;Saignes;SAIGNES;46500;46;LOT;1
46247;;Saillac;SAILLAC;46260;46;LOT;1
46248;;Sainte-Alauzie;SAINTE-ALAUZIE;46170;46;LOT;1
46249;;Saint-Bressou;SAINT-BRESSOU;46120;46;LOT;1
46250;;Saint-Caprais;SAINT-CAPRAIS;46250;46;LOT;1
46251;;Saint-Cr;SAINT-CERE;46400;46;LOT;1
46252;;Saint-Cernin;SAINT-CERNIN;46360;46;LOT;1
46253;;Saint-Chamarand;SAINT-CHAMARAND;46310;46;LOT;1
46254;;Saint-Chels;SAINT-CHELS;46160;46;LOT;1
46255;;Saint-Cirgues;SAINT-CIRGUES;46210;46;LOT;1
46256;;Saint-Cirq-Lapopie;SAINT-CIRQ-LAPOPIE;46330;46;LOT;1
46257;;Saint-Cirq-Madelon;SAINT-CIRQ-MADELON;46300;46;LOT;1
46258;;Saint-Cirq-Souillaguet;SAINT-CIRQ-SOUILLAGUET;46300;46;LOT;1
46259;;Saint-Clair;SAINT-CLAIR;46300;46;LOT;1
46260;;Sainte-Colombe;SAINTE-COLOMBE;46120;46;LOT;1
46261;;Sainte-Croix;SAINTE-CROIX;46800;46;LOT;1
46262;;Saint-Cyprien;SAINT-CYPRIEN;46800;46;LOT;1
46263;;Saint-Dauns;SAINT-DAUNES;46800;46;LOT;1
46264;;Saint-Denis-Catus;SAINT-DENIS-CATUS;46150;46;LOT;1
46265;;Saint-Denis-ls-Martel;SAINT-DENIS-LES-MARTEL;46600;46;LOT;1
46266;;Saint-Flix;SAINT-FELIX;46100;46;LOT;1
46267;;Saint-Germain-du-Bel-Air;SAINT-GERMAIN-DU-BEL-AIR;46310;46;LOT;1
46268;;Saint-Gry;SAINT-GERY;46330;46;LOT;1
46269;;Saint-Hilaire;SAINT-HILAIRE;46210;46;LOT;1
46270;;Saint-Jean-de-Laur;SAINT-JEAN-DE-LAUR;46260;46;LOT;1
46271;;Saint-Jean-Lespinasse;SAINT-JEAN-LESPINASSE;46400;46;LOT;1
46272;;Saint-Jean-Mirabel;SAINT-JEAN-MIRABEL;46270;46;LOT;1
46273;;Saint-Laurent-les-Tours;SAINT-LAURENT-LES-TOURS;46400;46;LOT;1
46274;;Saint-Laurent-Lolmie;SAINT-LAURENT-LOLMIE;46800;46;LOT;1
46275;;Saint-Martin-de-Vers;SAINT-MARTIN-DE-VERS;46360;46;LOT;1
46276;;Saint-Martin-Labouval;SAINT-MARTIN-LABOUVAL;46330;46;LOT;1
46277;;Saint-Martin-le-Redon;SAINT-MARTIN-LE-REDON;46700;46;LOT;1
46278;;Saint-Matr;SAINT-MATRE;46800;46;LOT;1
46279;;Saint-Maurice-en-Quercy;SAINT-MAURICE-EN-QUERCY;46120;46;LOT;1
46280;;Saint-Mdard;SAINT-MEDARD;46150;46;LOT;1
46281;;Saint-Mdard-de-Presque;SAINT-MEDARD-DE-PRESQUE;46400;46;LOT;1
46282;;Saint-Mdard-Nicourby;SAINT-MEDARD-NICOURBY;46210;46;LOT;1
46283;;Saint-Michel-de-Bannires;SAINT-MICHEL-DE-BANNIERES;46110;46;LOT;1
46284;;Saint-Michel-Loubjou;SAINT-MICHEL-LOUBEJOU;46130;46;LOT;1
46285;;Saint-Pantalon;SAINT-PANTALEON;46800;46;LOT;1
46286;;Saint-Paul-de-Vern;SAINT-PAUL-DE-VERN;46400;46;LOT;1
46287;;Saint-Paul-de-Loubressac;SAINT-PAUL-DE-LOUBRESSAC;46170;46;LOT;1
46288;;Saint-Perdoux;SAINT-PERDOUX;46100;46;LOT;1
46289;;Saint-Pierre-Toirac;SAINT-PIERRE-TOIRAC;46160;46;LOT;1
46290;;Saint-Projet;SAINT-PROJET;46300;46;LOT;1
46291;;Saint-Sauveur-la-Valle;SAINT-SAUVEUR-LA-VALLEE;46240;46;LOT;1
46292;;Saint-Simon;SAINT-SIMON;46320;46;LOT;1
46293;;Saint-Sozy;SAINT-SOZY;46200;46;LOT;1
46294;;Saint-Sulpice;SAINT-SULPICE;46160;46;LOT;1
46295;;Saint-Vincent-du-Pendit;SAINT-VINCENT-DU-PENDIT;46400;46;LOT;1
46296;;Saint-Vincent-Rive-d'Olt;SAINT-VINCENT-RIVE-D'OLT;46140;46;LOT;1
46297;;Salviac;SALVIAC;46340;46;LOT;1
46298;;Sarrazac;SARRAZAC;46600;46;LOT;1
46299;;Sauliac-sur-Cl;SAULIAC-SUR-CELE;46330;46;LOT;1
46300;;Saux;SAUX;46800;46;LOT;1
46301;;Sauzet;SAUZET;46140;46;LOT;1
46302;;Snaillac-Latronquire;SENAILLAC-LATRONQUIERE;46210;46;LOT;1
46303;;Snaillac-Lauzs;SENAILLAC-LAUZES;46360;46;LOT;1
46304;;Sniergues;SENIERGUES;46240;46;LOT;1
46305;;Srignac;SERIGNAC;46700;46;LOT;1
46306;;Sonac;SONAC;46320;46;LOT;1
46307;;Soturac;SOTURAC;46700;46;LOT;1
46308;;Soucirac;SOUCIRAC;46300;46;LOT;1
46309;;Souillac;SOUILLAC;46200;46;LOT;1
46310;;Souloms;SOULOMES;46240;46;LOT;1
46311;;Sousceyrac;SOUSCEYRAC;46190;46;LOT;1
46312;;Strenquels;STRENQUELS;46110;46;LOT;1
46313;;Tauriac;TAURIAC;46130;46;LOT;1
46314;;Terrou;TERROU;46120;46;LOT;1
46315;;Teyssieu;TEYSSIEU;46190;46;LOT;1
46316;;Thdirac;THEDIRAC;46150;46;LOT;1
46317;;Thgra;THEGRA;46500;46;LOT;1
46318;;Thmines;THEMINES;46120;46;LOT;1
46319;;Thminettes;THEMINETTES;46120;46;LOT;1
46320;;Tour-de-Faure;TOUR-DE-FAURE;46330;46;LOT;1
46321;;Touzac;TOUZAC;46700;46;LOT;1
46322;;Trespoux-Rassiels;TRESPOUX-RASSIELS;46090;46;LOT;1
46323;;Ussel;USSEL;46240;46;LOT;1
46324;;Uzech;UZECH;46310;46;LOT;1
46325;;Vaillac;VAILLAC;46240;46;LOT;1
46326;;Valprionde;VALPRIONDE;46800;46;LOT;1
46327;;Valroufi;VALROUFIE;46090;46;LOT;1
46328;;Varaire;VARAIRE;46260;46;LOT;1
46329;;Vaylats;VAYLATS;46230;46;LOT;1
46330;;Vayrac;VAYRAC;46110;46;LOT;1
46331;;Vers;VERS;46090;46;LOT;1
46332;;Viazac;VIAZAC;46100;46;LOT;1
46333;;Vidaillac;VIDAILLAC;46260;46;LOT;1
46334;Le;Vigan;VIGAN;46300;46;LOT;1
46335;;Villesque;VILLESEQUE;46090;46;LOT;1
46336;;Vire-sur-Lot;VIRE-SUR-LOT;46700;46;LOT;1
46337;;Mayrac;MAYRAC;46200;46;LOT;1
46338;;Bessonies;BESSONIES;46210;46;LOT;1
46339;;Saint-Jean-Lagineste;SAINT-JEAN-LAGINESTE;46400;46;LOT;1
46340;;Saint-Pierre-Lafeuille;SAINT-PIERRE-LAFEUILLE;46090;46;LOT;1
47001;;Agen;AGEN;47000;47;LOT ET GARONNE;1
47002;;Agm;AGME;47350;47;LOT ET GARONNE;1
47003;;Agnac;AGNAC;47800;47;LOT ET GARONNE;1
47004;;Aiguillon;AIGUILLON;47190;47;LOT ET GARONNE;1
47005;;Allemans-du-Dropt;ALLEMANS-DU-DROPT;47800;47;LOT ET GARONNE;1
47006;;Allez-et-Cazeneuve;ALLEZ-ET-CAZENEUVE;47110;47;LOT ET GARONNE;1
47007;;Allons;ALLONS;47420;47;LOT ET GARONNE;1
47008;;Ambrus;AMBRUS;47160;47;LOT ET GARONNE;1
47009;;Andiran;ANDIRAN;47170;47;LOT ET GARONNE;1
47010;;Antagnac;ANTAGNAC;47700;47;LOT ET GARONNE;1
47011;;Anth;ANTHE;47370;47;LOT ET GARONNE;1
47012;;Anzex;ANZEX;47700;47;LOT ET GARONNE;1
47013;;Argenton;ARGENTON;47250;47;LOT ET GARONNE;1
47014;;Armillac;ARMILLAC;47800;47;LOT ET GARONNE;1
47015;;Astaffort;ASTAFFORT;47220;47;LOT ET GARONNE;1
47016;;Aubiac;AUBIAC;47310;47;LOT ET GARONNE;1
47017;;Auradou;AURADOU;47140;47;LOT ET GARONNE;1
47018;;Auriac-sur-Dropt;AURIAC-SUR-DROPT;47120;47;LOT ET GARONNE;1
47019;;Bajamont;BAJAMONT;47480;47;LOT ET GARONNE;1
47020;;Baleyssagues;BALEYSSAGUES;47120;47;LOT ET GARONNE;1
47021;;Barbaste;BARBASTE;47230;47;LOT ET GARONNE;1
47022;;Bazens;BAZENS;47130;47;LOT ET GARONNE;1
47023;;Beaugas;BEAUGAS;47290;47;LOT ET GARONNE;1
47024;;Beaupuy;BEAUPUY;47200;47;LOT ET GARONNE;1
47025;;Beauville;BEAUVILLE;47470;47;LOT ET GARONNE;1
47026;;Beauziac;BEAUZIAC;47700;47;LOT ET GARONNE;1
47027;;Bias;BIAS;47300;47;LOT ET GARONNE;1
47028;;Birac-sur-Trec;BIRAC-SUR-TREC;47200;47;LOT ET GARONNE;1
47029;;Blanquefort-sur-Briolance;BLANQUEFORT-SUR-BRIOLANCE;47500;47;LOT ET GARONNE;1
47030;;Blaymont;BLAYMONT;47470;47;LOT ET GARONNE;1
47031;;Bo;BOE;47550;47;LOT ET GARONNE;1
47032;;Bon-Encontre;BON-ENCONTRE;47240;47;LOT ET GARONNE;1
47033;;Boudy-de-Beauregard;BOUDY-DE-BEAUREGARD;47290;47;LOT ET GARONNE;1
47034;;Bouglon;BOUGLON;47250;47;LOT ET GARONNE;1
47035;;Bourgougnague;BOURGOUGNAGUE;47410;47;LOT ET GARONNE;1
47036;;Bourlens;BOURLENS;47370;47;LOT ET GARONNE;1
47037;;Bournel;BOURNEL;47210;47;LOT ET GARONNE;1
47038;;Bourran;BOURRAN;47320;47;LOT ET GARONNE;1
47039;;Bousss;BOUSSES;47420;47;LOT ET GARONNE;1
47040;;Brax;BRAX;47310;47;LOT ET GARONNE;1
47041;;Bruch;BRUCH;47130;47;LOT ET GARONNE;1
47042;;Brugnac;BRUGNAC;47260;47;LOT ET GARONNE;1
47043;;Buzet-sur-Base;BUZET-SUR-BAISE;47160;47;LOT ET GARONNE;1
47044;;Cahuzac;CAHUZAC;47330;47;LOT ET GARONNE;1
47045;;Calignac;CALIGNAC;47600;47;LOT ET GARONNE;1
47046;;Calonges;CALONGES;47430;47;LOT ET GARONNE;1
47047;;Cambes;CAMBES;47350;47;LOT ET GARONNE;1
47048;;Cancon;CANCON;47290;47;LOT ET GARONNE;1
47049;;Casseneuil;CASSENEUIL;47440;47;LOT ET GARONNE;1
47050;;Cassignas;CASSIGNAS;47340;47;LOT ET GARONNE;1
47051;;Castelculier;CASTELCULIER;47240;47;LOT ET GARONNE;1
47052;;Casteljaloux;CASTELJALOUX;47700;47;LOT ET GARONNE;1
47053;;Castella;CASTELLA;47340;47;LOT ET GARONNE;1
47054;;Castelmoron-sur-Lot;CASTELMORON-SUR-LOT;47260;47;LOT ET GARONNE;1
47055;;Castelnaud-de-Gratecambe;CASTELNAUD-DE-GRATECAMBE;47290;47;LOT ET GARONNE;1
47056;;Castelnau-sur-Gupie;CASTELNAU-SUR-GUPIE;47200;47;LOT ET GARONNE;1
47057;;Castillonns;CASTILLONNES;47330;47;LOT ET GARONNE;1
47058;;Caubeyres;CAUBEYRES;47160;47;LOT ET GARONNE;1
47059;;Caubon-Saint-Sauveur;CAUBON-SAINT-SAUVEUR;47120;47;LOT ET GARONNE;1
47060;;Caudecoste;CAUDECOSTE;47220;47;LOT ET GARONNE;1
47061;;Caumont-sur-Garonne;CAUMONT-SUR-GARONNE;47430;47;LOT ET GARONNE;1
47062;;Cauzac;CAUZAC;47470;47;LOT ET GARONNE;1
47063;;Cavarc;CAVARC;47330;47;LOT ET GARONNE;1
47064;;Cazideroque;CAZIDEROQUE;47370;47;LOT ET GARONNE;1
47065;;Clairac;CLAIRAC;47320;47;LOT ET GARONNE;1
47066;;Clermont-Dessous;CLERMONT-DESSOUS;47130;47;LOT ET GARONNE;1
47067;;Clermont-Soubiran;CLERMONT-SOUBIRAN;47270;47;LOT ET GARONNE;1
47068;;Cocumont;COCUMONT;47250;47;LOT ET GARONNE;1
47069;;Colayrac-Saint-Cirq;COLAYRAC-SAINT-CIRQ;47450;47;LOT ET GARONNE;1
47070;;Condezaygues;CONDEZAYGUES;47500;47;LOT ET GARONNE;1
47071;;Coulx;COULX;47260;47;LOT ET GARONNE;1
47072;;Courbiac;COURBIAC;47370;47;LOT ET GARONNE;1
47073;;Cours;COURS;47360;47;LOT ET GARONNE;1
47074;;Couthures-sur-Garonne;COUTHURES-SUR-GARONNE;47200;47;LOT ET GARONNE;1
47075;La;Croix-Blanche;CROIX-BLANCHE;47340;47;LOT ET GARONNE;1
47076;;Cuq;CUQ;47220;47;LOT ET GARONNE;1
47077;;Cuzorn;CUZORN;47500;47;LOT ET GARONNE;1
47078;;Damazan;DAMAZAN;47160;47;LOT ET GARONNE;1
47079;;Dausse;DAUSSE;47140;47;LOT ET GARONNE;1
47080;;Dvillac;DEVILLAC;47210;47;LOT ET GARONNE;1
47081;;Dolmayrac;DOLMAYRAC;47110;47;LOT ET GARONNE;1
47082;;Dondas;DONDAS;47470;47;LOT ET GARONNE;1
47083;;Doudrac;DOUDRAC;47210;47;LOT ET GARONNE;1
47084;;Douzains;DOUZAINS;47330;47;LOT ET GARONNE;1
47085;;Durance;DURANCE;47420;47;LOT ET GARONNE;1
47086;;Duras;DURAS;47120;47;LOT ET GARONNE;1
47087;;Engayrac;ENGAYRAC;47470;47;LOT ET GARONNE;1
47088;;Escassefort;ESCASSEFORT;47350;47;LOT ET GARONNE;1
47089;;Esclottes;ESCLOTTES;47120;47;LOT ET GARONNE;1
47090;;Espiens;ESPIENS;47600;47;LOT ET GARONNE;1
47091;;Estillac;ESTILLAC;47310;47;LOT ET GARONNE;1
47092;;Fals;FALS;47220;47;LOT ET GARONNE;1
47093;;Fargues-sur-Ourbise;FARGUES-SUR-OURBISE;47700;47;LOT ET GARONNE;1
47094;;Fauguerolles;FAUGUEROLLES;47400;47;LOT ET GARONNE;1
47095;;Fauillet;FAUILLET;47400;47;LOT ET GARONNE;1
47096;;Ferrensac;FERRENSAC;47330;47;LOT ET GARONNE;1
47097;;Feugarolles;FEUGAROLLES;47230;47;LOT ET GARONNE;1
47098;;Fieux;FIEUX;47600;47;LOT ET GARONNE;1
47099;;Fongrave;FONGRAVE;47260;47;LOT ET GARONNE;1
47100;;Foulayronnes;FOULAYRONNES;47510;47;LOT ET GARONNE;1
47101;;Fourques-sur-Garonne;FOURQUES-SUR-GARONNE;47200;47;LOT ET GARONNE;1
47102;;Francescas;FRANCESCAS;47600;47;LOT ET GARONNE;1
47103;;Frchou;FRECHOU;47600;47;LOT ET GARONNE;1
47104;;Frgimont;FREGIMONT;47360;47;LOT ET GARONNE;1
47105;;Frespech;FRESPECH;47140;47;LOT ET GARONNE;1
47106;;Fumel;FUMEL;47500;47;LOT ET GARONNE;1
47107;;Galapian;GALAPIAN;47190;47;LOT ET GARONNE;1
47108;;Gaujac;GAUJAC;47200;47;LOT ET GARONNE;1
47109;;Gavaudun;GAVAUDUN;47150;47;LOT ET GARONNE;1
47110;;Gontaud-de-Nogaret;GONTAUD-DE-NOGARET;47400;47;LOT ET GARONNE;1
47111;;Granges-sur-Lot;GRANGES-SUR-LOT;47260;47;LOT ET GARONNE;1
47112;;Grateloup-Saint-Gayrand;GRATELOUP-SAINT-GAYRAND;47400;47;LOT ET GARONNE;1
47113;;Grayssas;GRAYSSAS;47270;47;LOT ET GARONNE;1
47114;;Grzet-Cavagnan;GREZET-CAVAGNAN;47250;47;LOT ET GARONNE;1
47115;;Gurin;GUERIN;47250;47;LOT ET GARONNE;1
47117;;Hautefage-la-Tour;HAUTEFAGE-LA-TOUR;47340;47;LOT ET GARONNE;1
47118;;Hautesvignes;HAUTESVIGNES;47400;47;LOT ET GARONNE;1
47119;;Houeills;HOUEILLES;47420;47;LOT ET GARONNE;1
47120;;Jusix;JUSIX;47200;47;LOT ET GARONNE;1
47121;;Labastide-Castel-Amouroux;LABASTIDE-CASTEL-AMOUROUX;47250;47;LOT ET GARONNE;1
47122;;Labretonie;LABRETONIE;47350;47;LOT ET GARONNE;1
47123;;Lacapelle-Biron;LACAPELLE-BIRON;47150;47;LOT ET GARONNE;1
47124;;Lacaussade;LACAUSSADE;47150;47;LOT ET GARONNE;1
47125;;Lacpde;LACEPEDE;47360;47;LOT ET GARONNE;1
47126;;Lachapelle;LACHAPELLE;47350;47;LOT ET GARONNE;1
47127;;Lafitte-sur-Lot;LAFITTE-SUR-LOT;47320;47;LOT ET GARONNE;1
47128;;Lafox;LAFOX;47240;47;LOT ET GARONNE;1
47129;;Lagarrigue;LAGARRIGUE;47190;47;LOT ET GARONNE;1
47130;;Lagrure;LAGRUERE;47400;47;LOT ET GARONNE;1
47131;;Lagupie;LAGUPIE;47200;47;LOT ET GARONNE;1
47132;;Lalandusse;LALANDUSSE;47330;47;LOT ET GARONNE;1
47133;;Lamontjoie;LAMONTJOIE;47310;47;LOT ET GARONNE;1
47134;;Lannes;LANNES;47170;47;LOT ET GARONNE;1
47135;;Laparade;LAPARADE;47260;47;LOT ET GARONNE;1
47136;;Laperche;LAPERCHE;47800;47;LOT ET GARONNE;1
47137;;Laplume;LAPLUME;47310;47;LOT ET GARONNE;1
47138;;Laroque-Timbaut;LAROQUE-TIMBAUT;47340;47;LOT ET GARONNE;1
47139;;Lasserre;LASSERRE;47600;47;LOT ET GARONNE;1
47140;;Laugnac;LAUGNAC;47360;47;LOT ET GARONNE;1
47141;;Laussou;LAUSSOU;47150;47;LOT ET GARONNE;1
47142;;Lauzun;LAUZUN;47410;47;LOT ET GARONNE;1
47143;;Lavardac;LAVARDAC;47230;47;LOT ET GARONNE;1
47144;;Lavergne;LAVERGNE;47800;47;LOT ET GARONNE;1
47145;;Layrac;LAYRAC;47390;47;LOT ET GARONNE;1
47146;;Ldat;LEDAT;47300;47;LOT ET GARONNE;1
47147;;Lvignac-de-Guyenne;LEVIGNAC-DE-GUYENNE;47120;47;LOT ET GARONNE;1
47148;;Leyritz-Moncassin;LEYRITZ-MONCASSIN;47700;47;LOT ET GARONNE;1
47150;;Longueville;LONGUEVILLE;47200;47;LOT ET GARONNE;1
47151;;Loubs-Bernac;LOUBES-BERNAC;47120;47;LOT ET GARONNE;1
47152;;Lougratte;LOUGRATTE;47290;47;LOT ET GARONNE;1
47154;;Lusignan-Petit;LUSIGNAN-PETIT;47360;47;LOT ET GARONNE;1
47155;;Madaillan;MADAILLAN;47360;47;LOT ET GARONNE;1
47156;;Marcellus;MARCELLUS;47200;47;LOT ET GARONNE;1
47157;;Marmande;MARMANDE;47200;47;LOT ET GARONNE;1
47158;;Marmont-Pachas;MARMONT-PACHAS;47220;47;LOT ET GARONNE;1
47159;Le;Mas-d'Agenais;MAS-D'AGENAIS;47430;47;LOT ET GARONNE;1
47160;;Masquires;MASQUIERES;47370;47;LOT ET GARONNE;1
47161;;Massels;MASSELS;47140;47;LOT ET GARONNE;1
47162;;Massouls;MASSOULES;47140;47;LOT ET GARONNE;1
47163;;Mauvezin-sur-Gupie;MAUVEZIN-SUR-GUPIE;47200;47;LOT ET GARONNE;1
47164;;Mazires-Naresse;MAZIERES-NARESSE;47210;47;LOT ET GARONNE;1
47165;;Meilhan-sur-Garonne;MEILHAN-SUR-GARONNE;47200;47;LOT ET GARONNE;1
47167;;Mzin;MEZIN;47170;47;LOT ET GARONNE;1
47168;;Miramont-de-Guyenne;MIRAMONT-DE-GUYENNE;47800;47;LOT ET GARONNE;1
47169;;Moirax;MOIRAX;47310;47;LOT ET GARONNE;1
47170;;Monbahus;MONBAHUS;47290;47;LOT ET GARONNE;1
47171;;Monbalen;MONBALEN;47340;47;LOT ET GARONNE;1
47172;;Moncaut;MONCAUT;47310;47;LOT ET GARONNE;1
47173;;Monclar;MONCLAR;47380;47;LOT ET GARONNE;1
47174;;Moncrabeau;MONCRABEAU;47600;47;LOT ET GARONNE;1
47175;;Monflanquin;MONFLANQUIN;47150;47;LOT ET GARONNE;1
47176;;Mongaillard;MONGAILLARD;47230;47;LOT ET GARONNE;1
47177;;Monheurt;MONHEURT;47160;47;LOT ET GARONNE;1
47178;;Monsgur;MONSEGUR;47150;47;LOT ET GARONNE;1
47179;;Monsempron-Libos;MONSEMPRON-LIBOS;47500;47;LOT ET GARONNE;1
47180;;Montagnac-sur-Auvignon;MONTAGNAC-SUR-AUVIGNON;47600;47;LOT ET GARONNE;1
47181;;Montagnac-sur-Lde;MONTAGNAC-SUR-LEDE;47150;47;LOT ET GARONNE;1
47182;;Montastruc;MONTASTRUC;47380;47;LOT ET GARONNE;1
47183;;Montauriol;MONTAURIOL;47330;47;LOT ET GARONNE;1
47184;;Montaut;MONTAUT;47210;47;LOT ET GARONNE;1
47185;;Montayral;MONTAYRAL;47500;47;LOT ET GARONNE;1
47186;;Montesquieu;MONTESQUIEU;47130;47;LOT ET GARONNE;1
47187;;Monteton;MONTETON;47120;47;LOT ET GARONNE;1
47188;;Montignac-de-Lauzun;MONTIGNAC-DE-LAUZUN;47800;47;LOT ET GARONNE;1
47189;;Montignac-Toupinerie;MONTIGNAC-TOUPINERIE;47350;47;LOT ET GARONNE;1
47190;;Montpezat;MONTPEZAT;47360;47;LOT ET GARONNE;1
47191;;Montpouillan;MONTPOUILLAN;47200;47;LOT ET GARONNE;1
47192;;Monviel;MONVIEL;47290;47;LOT ET GARONNE;1
47193;;Moulinet;MOULINET;47290;47;LOT ET GARONNE;1
47194;;Moustier;MOUSTIER;47800;47;LOT ET GARONNE;1
47195;;Nrac;NERAC;47600;47;LOT ET GARONNE;1
47196;;Nicole;NICOLE;47190;47;LOT ET GARONNE;1
47197;;Nomdieu;NOMDIEU;47600;47;LOT ET GARONNE;1
47198;;Pailloles;PAILLOLES;47440;47;LOT ET GARONNE;1
47199;;Pardaillan;PARDAILLAN;47120;47;LOT ET GARONNE;1
47200;;Parranquet;PARRANQUET;47210;47;LOT ET GARONNE;1
47201;Le;Passage;PASSAGE;47520;47;LOT ET GARONNE;1
47202;;Paulhiac;PAULHIAC;47150;47;LOT ET GARONNE;1
47203;;Penne-d'Agenais;PENNE-D'AGENAIS;47140;47;LOT ET GARONNE;1
47204;;Peyrire;PEYRIERE;47350;47;LOT ET GARONNE;1
47205;;Pindres;PINDERES;47700;47;LOT ET GARONNE;1
47206;;Pinel-Hauterive;PINEL-HAUTERIVE;47380;47;LOT ET GARONNE;1
47207;;Pompiey;POMPIEY;47230;47;LOT ET GARONNE;1
47208;;Pompogne;POMPOGNE;47420;47;LOT ET GARONNE;1
47209;;Pont-du-Casse;PONT-DU-CASSE;47480;47;LOT ET GARONNE;1
47210;;Port-Sainte-Marie;PORT-SAINTE-MARIE;47130;47;LOT ET GARONNE;1
47211;;Poudenas;POUDENAS;47170;47;LOT ET GARONNE;1
47212;;Poussignac;POUSSIGNAC;47700;47;LOT ET GARONNE;1
47213;;Prayssas;PRAYSSAS;47360;47;LOT ET GARONNE;1
47214;;Puch-d'Agenais;PUCH-D'AGENAIS;47160;47;LOT ET GARONNE;1
47215;;Pujols;PUJOLS;47300;47;LOT ET GARONNE;1
47216;;Puymiclan;PUYMICLAN;47350;47;LOT ET GARONNE;1
47217;;Puymirol;PUYMIROL;47270;47;LOT ET GARONNE;1
47218;;Puysserampion;PUYSSERAMPION;47800;47;LOT ET GARONNE;1
47219;;Rayet;RAYET;47210;47;LOT ET GARONNE;1
47220;;Razimet;RAZIMET;47160;47;LOT ET GARONNE;1
47221;;Raup-Lisse;REAUP-LISSE;47170;47;LOT ET GARONNE;1
47222;La;Runion;REUNION;47700;47;LOT ET GARONNE;1
47223;;Rives;RIVES;47210;47;LOT ET GARONNE;1
47224;;Romestaing;ROMESTAING;47250;47;LOT ET GARONNE;1
47225;;Roquefort;ROQUEFORT;47310;47;LOT ET GARONNE;1
47226;;Roumagne;ROUMAGNE;47800;47;LOT ET GARONNE;1
47227;;Ruffiac;RUFFIAC;47700;47;LOT ET GARONNE;1
47228;;Saint-Antoine-de-Ficalba;SAINT-ANTOINE-DE-FICALBA;47340;47;LOT ET GARONNE;1
47229;;Saint-Astier;SAINT-ASTIER;47120;47;LOT ET GARONNE;1
47230;;Saint-Aubin;SAINT-AUBIN;47150;47;LOT ET GARONNE;1
47231;;Saint-Avit;SAINT-AVIT;47350;47;LOT ET GARONNE;1
47232;;Saint-Barthlemy-d'Agenais;SAINT-BARTHELEMY-D'AGENAIS;47350;47;LOT ET GARONNE;1
47233;;Sainte-Bazeille;SAINTE-BAZEILLE;47200;47;LOT ET GARONNE;1
47234;;Saint-Caprais-de-Lerm;SAINT-CAPRAIS-DE-LERM;47270;47;LOT ET GARONNE;1
47235;;Saint-Colomb-de-Lauzun;SAINT-COLOMB-DE-LAUZUN;47410;47;LOT ET GARONNE;1
47236;;Sainte-Colombe-de-Duras;SAINTE-COLOMBE-DE-DURAS;47120;47;LOT ET GARONNE;1
47237;;Sainte-Colombe-de-Villeneuve;SAINTE-COLOMBE-DE-VILLENEUVE;47300;47;LOT ET GARONNE;1
47238;;Sainte-Colombe-en-Bruilhois;SAINTE-COLOMBE-EN-BRUILHOIS;47310;47;LOT ET GARONNE;1
47239;;Saint-tienne-de-Fougres;SAINT-ETIENNE-DE-FOUGERES;47380;47;LOT ET GARONNE;1
47240;;Saint-tienne-de-Villeral;SAINT-ETIENNE-DE-VILLEREAL;47210;47;LOT ET GARONNE;1
47241;;Saint-Eutrope-de-Born;SAINT-EUTROPE-DE-BORN;47210;47;LOT ET GARONNE;1
47242;;Saint-Front-sur-Lmance;SAINT-FRONT-SUR-LEMANCE;47500;47;LOT ET GARONNE;1
47244;;Sainte-Gemme-Martaillac;SAINTE-GEMME-MARTAILLAC;47250;47;LOT ET GARONNE;1
47245;;Saint-Graud;SAINT-GERAUD;47120;47;LOT ET GARONNE;1
47246;;Saint-Hilaire-de-Lusignan;SAINT-HILAIRE-DE-LUSIGNAN;47450;47;LOT ET GARONNE;1
47247;;Saint-Jean-de-Duras;SAINT-JEAN-DE-DURAS;47120;47;LOT ET GARONNE;1
47248;;Saint-Jean-de-Thurac;SAINT-JEAN-DE-THURAC;47270;47;LOT ET GARONNE;1
47249;;Saint-Laurent;SAINT-LAURENT;47130;47;LOT ET GARONNE;1
47250;;Saint-Lger;SAINT-LEGER;47160;47;LOT ET GARONNE;1
47251;;Saint-Lon;SAINT-LEON;47160;47;LOT ET GARONNE;1
47252;;Sainte-Livrade-sur-Lot;SAINTE-LIVRADE-SUR-LOT;47110;47;LOT ET GARONNE;1
47253;;Sainte-Marthe;SAINTE-MARTHE;47430;47;LOT ET GARONNE;1
47254;;Saint-Martin-Curton;SAINT-MARTIN-CURTON;47700;47;LOT ET GARONNE;1
47255;;Saint-Martin-de-Beauville;SAINT-MARTIN-DE-BEAUVILLE;47270;47;LOT ET GARONNE;1
47256;;Saint-Martin-de-Villeral;SAINT-MARTIN-DE-VILLEREAL;47210;47;LOT ET GARONNE;1
47257;;Saint-Martin-Petit;SAINT-MARTIN-PETIT;47200;47;LOT ET GARONNE;1
47258;;Sainte-Maure-de-Peyriac;SAINTE-MAURE-DE-PEYRIAC;47170;47;LOT ET GARONNE;1
47259;;Saint-Maurice-de-Lestapel;SAINT-MAURICE-DE-LESTAPEL;47290;47;LOT ET GARONNE;1
47260;;Saint-Maurin;SAINT-MAURIN;47270;47;LOT ET GARONNE;1
47262;;Saint-Nicolas-de-la-Balerme;SAINT-NICOLAS-DE-LA-BALERME;47220;47;LOT ET GARONNE;1
47263;;Saint-Pardoux-du-Breuil;SAINT-PARDOUX-DU-BREUIL;47200;47;LOT ET GARONNE;1
47264;;Saint-Pardoux-Isaac;SAINT-PARDOUX-ISAAC;47800;47;LOT ET GARONNE;1
47265;;Saint-Pastour;SAINT-PASTOUR;47290;47;LOT ET GARONNE;1
47266;;Saint-P-Saint-Simon;SAINT-PE-SAINT-SIMON;47170;47;LOT ET GARONNE;1
47267;;Saint-Pierre-de-Buzet;SAINT-PIERRE-DE-BUZET;47160;47;LOT ET GARONNE;1
47269;;Saint-Pierre-de-Clairac;SAINT-PIERRE-DE-CLAIRAC;47270;47;LOT ET GARONNE;1
47271;;Saint-Pierre-sur-Dropt;SAINT-PIERRE-SUR-DROPT;47120;47;LOT ET GARONNE;1
47272;;Saint-Quentin-du-Dropt;SAINT-QUENTIN-DU-DROPT;47330;47;LOT ET GARONNE;1
47273;;Saint-Robert;SAINT-ROBERT;47340;47;LOT ET GARONNE;1
47274;;Saint-Romain-le-Noble;SAINT-ROMAIN-LE-NOBLE;47270;47;LOT ET GARONNE;1
47275;;Saint-Salvy;SAINT-SALVY;47360;47;LOT ET GARONNE;1
47276;;Saint-Sardos;SAINT-SARDOS;47360;47;LOT ET GARONNE;1
47277;;Saint-Sauveur-de-Meilhan;SAINT-SAUVEUR-DE-MEILHAN;47200;47;LOT ET GARONNE;1
47278;;Saint-Sernin;SAINT-SERNIN;47120;47;LOT ET GARONNE;1
47279;;Saint-Sixte;SAINT-SIXTE;47220;47;LOT ET GARONNE;1
47280;;Saint-Sylvestre-sur-Lot;SAINT-SYLVESTRE-SUR-LOT;47140;47;LOT ET GARONNE;1
47281;;Saint-Urcisse;SAINT-URCISSE;47270;47;LOT ET GARONNE;1
47282;;Saint-Vincent-de-Lamontjoie;SAINT-VINCENT-DE-LAMONTJOIE;47310;47;LOT ET GARONNE;1
47283;;Saint-Vite;SAINT-VITE;47500;47;LOT ET GARONNE;1
47284;;Salles;SALLES;47150;47;LOT ET GARONNE;1
47285;;Samazan;SAMAZAN;47250;47;LOT ET GARONNE;1
47286;;Saumjan;SAUMEJAN;47420;47;LOT ET GARONNE;1
47287;;Saumont;SAUMONT;47600;47;LOT ET GARONNE;1
47288;;Sauvagnas;SAUVAGNAS;47340;47;LOT ET GARONNE;1
47289;La;Sauvetat-de-Savres;SAUVETAT-DE-SAVERES;47270;47;LOT ET GARONNE;1
47290;La;Sauvetat-du-Dropt;SAUVETAT-DU-DROPT;47800;47;LOT ET GARONNE;1
47291;La;Sauvetat-sur-Lde;SAUVETAT-SUR-LEDE;47150;47;LOT ET GARONNE;1
47292;;Sauveterre-la-Lmance;SAUVETERRE-LA-LEMANCE;47500;47;LOT ET GARONNE;1
47293;;Sauveterre-Saint-Denis;SAUVETERRE-SAINT-DENIS;47220;47;LOT ET GARONNE;1
47294;;Savignac-de-Duras;SAVIGNAC-DE-DURAS;47120;47;LOT ET GARONNE;1
47295;;Savignac-sur-Leyze;SAVIGNAC-SUR-LEYZE;47150;47;LOT ET GARONNE;1
47296;;Sgalas;SEGALAS;47410;47;LOT ET GARONNE;1
47297;;Sembas;SEMBAS;47360;47;LOT ET GARONNE;1
47298;;Snestis;SENESTIS;47430;47;LOT ET GARONNE;1
47299;;Srignac-Pboudou;SERIGNAC-PEBOUDOU;47410;47;LOT ET GARONNE;1
47300;;Srignac-sur-Garonne;SERIGNAC-SUR-GARONNE;47310;47;LOT ET GARONNE;1
47301;;Seyches;SEYCHES;47350;47;LOT ET GARONNE;1
47302;;Sos;SOS;47170;47;LOT ET GARONNE;1
47303;;Soumensac;SOUMENSAC;47120;47;LOT ET GARONNE;1
47304;;Taillebourg;TAILLEBOURG;47200;47;LOT ET GARONNE;1
47305;;Tayrac;TAYRAC;47270;47;LOT ET GARONNE;1
47306;Le;Temple-sur-Lot;TEMPLE-SUR-LOT;47110;47;LOT ET GARONNE;1
47307;;Thzac;THEZAC;47370;47;LOT ET GARONNE;1
47308;;Thouars-sur-Garonne;THOUARS-SUR-GARONNE;47230;47;LOT ET GARONNE;1
47309;;Tombebuf;TOMBEBOEUF;47380;47;LOT ET GARONNE;1
47310;;Tonneins;TONNEINS;47400;47;LOT ET GARONNE;1
47311;;Tourliac;TOURLIAC;47210;47;LOT ET GARONNE;1
47312;;Tournon-d'Agenais;TOURNON-D'AGENAIS;47370;47;LOT ET GARONNE;1
47313;;Tourtrs;TOURTRES;47380;47;LOT ET GARONNE;1
47314;;Trmons;TREMONS;47140;47;LOT ET GARONNE;1
47315;;Trentels;TRENTELS;47140;47;LOT ET GARONNE;1
47316;;Vars;VARES;47400;47;LOT ET GARONNE;1
47317;;Verteuil-d'Agenais;VERTEUIL-D'AGENAIS;47260;47;LOT ET GARONNE;1
47318;;Vianne;VIANNE;47230;47;LOT ET GARONNE;1
47319;;Villebramar;VILLEBRAMAR;47380;47;LOT ET GARONNE;1
47320;;Villefranche-du-Queyran;VILLEFRANCHE-DU-QUEYRAN;47160;47;LOT ET GARONNE;1
47321;;Villeneuve-de-Duras;VILLENEUVE-DE-DURAS;47120;47;LOT ET GARONNE;1
47323;;Villeneuve-sur-Lot;VILLENEUVE-SUR-LOT;47300;47;LOT ET GARONNE;1
47324;;Villeral;VILLEREAL;47210;47;LOT ET GARONNE;1
47325;;Villeton;VILLETON;47400;47;LOT ET GARONNE;1
47326;;Virazeil;VIRAZEIL;47200;47;LOT ET GARONNE;1
47327;;Xaintrailles;XAINTRAILLES;47230;47;LOT ET GARONNE;1
47328;;Saint-Georges;SAINT-GEORGES;47370;47;LOT ET GARONNE;1
48001;;Albaret-le-Comtal;ALBARET-LE-COMTAL;48310;48;LOZERE;1
48002;;Albaret-Sainte-Marie;ALBARET-SAINTE-MARIE;48200;48;LOZERE;1
48003;;Allenc;ALLENC;48190;48;LOZERE;1
48004;;Altier;ALTIER;48800;48;LOZERE;1
48005;;Antrenas;ANTRENAS;48100;48;LOZERE;1
48007;;Arzenc-d'Apcher;ARZENC-D'APCHER;48310;48;LOZERE;1
48008;;Arzenc-de-Randon;ARZENC-DE-RANDON;48170;48;LOZERE;1
48009;;Aumont-Aubrac;AUMONT-AUBRAC;48130;48;LOZERE;1
48010;;Auroux;AUROUX;48600;48;LOZERE;1
48012;Les;Monts-Verts;MONTS-VERTS;48200;48;LOZERE;1
48013;;Badaroux;BADAROUX;48000;48;LOZERE;1
48014;;Bagnols-les-Bains;BAGNOLS-LES-BAINS;48190;48;LOZERE;1
48015;;Pied-de-Borne;PIED-DE-BORNE;48800;48;LOZERE;1
48016;;Balsiges;BALSIEGES;48000;48;LOZERE;1
48017;;Banassac;BANASSAC;48500;48;LOZERE;1
48018;;Barjac;BARJAC;48000;48;LOZERE;1
48019;;Barre-des-Cvennes;BARRE-DES-CEVENNES;48400;48;LOZERE;1
48020;;Bassurels;BASSURELS;48400;48;LOZERE;1
48021;La;Bastide-Puylaurent;BASTIDE-PUYLAURENT;48250;48;LOZERE;1
48022;;Bdous;BEDOUES;48400;48;LOZERE;1
48023;;Belvezet;BELVEZET;48170;48;LOZERE;1
48025;Les;Bessons;BESSONS;48200;48;LOZERE;1
48026;;Blavignac;BLAVIGNAC;48200;48;LOZERE;1
48027;Le;Bleymard;BLEYMARD;48190;48;LOZERE;1
48028;Les;Bondons;BONDONS;48400;48;LOZERE;1
48029;Le;Born;BORN;48000;48;LOZERE;1
48030;;Brenoux;BRENOUX;48000;48;LOZERE;1
48031;;Brion;BRION;48310;48;LOZERE;1
48032;Le;Buisson;BUISSON;48100;48;LOZERE;1
48033;;Canilhac;CANILHAC;48500;48;LOZERE;1
48034;La;Canourgue;CANOURGUE;48500;48;LOZERE;1
48036;;Cassagnas;CASSAGNAS;48400;48;LOZERE;1
48037;;Chadenet;CHADENET;48190;48;LOZERE;1
48038;;Chambon-le-Chteau;CHAMBON-LE-CHATEAU;48600;48;LOZERE;1
48039;;Chanac;CHANAC;48230;48;LOZERE;1
48040;;Chasserads;CHASSERADES;48250;48;LOZERE;1
48041;;Chastanier;CHASTANIER;48300;48;LOZERE;1
48042;;Chastel-Nouvel;CHASTEL-NOUVEL;48000;48;LOZERE;1
48043;;Chteauneuf-de-Randon;CHATEAUNEUF-DE-RANDON;48170;48;LOZERE;1
48044;;Chauchailles;CHAUCHAILLES;48310;48;LOZERE;1
48045;;Chaudeyrac;CHAUDEYRAC;48170;48;LOZERE;1
48046;;Chaulhac;CHAULHAC;48140;48;LOZERE;1
48047;La;Chaze-de-Peyre;CHAZE-DE-PEYRE;48130;48;LOZERE;1
48048;;Cheylard-l'vque;CHEYLARD-L'EVEQUE;48300;48;LOZERE;1
48049;;Chirac;CHIRAC;48100;48;LOZERE;1
48050;;Cocurs;COCURES;48400;48;LOZERE;1
48051;Le;Collet-de-Dze;COLLET-DE-DEZE;48160;48;LOZERE;1
48053;;Cubires;CUBIERES;48190;48;LOZERE;1
48054;;Cubirettes;CUBIERETTES;48190;48;LOZERE;1
48055;;Cultures;CULTURES;48230;48;LOZERE;1
48056;;Esclandes;ESCLANEDES;48230;48;LOZERE;1
48057;;Estables;ESTABLES;48700;48;LOZERE;1
48058;La;Fage-Montivernoux;FAGE-MONTIVERNOUX;48310;48;LOZERE;1
48059;La;Fage-Saint-Julien;FAGE-SAINT-JULIEN;48200;48;LOZERE;1
48060;;Fau-de-Peyre;FAU-DE-PEYRE;48130;48;LOZERE;1
48061;;Florac;FLORAC;48400;48;LOZERE;1
48062;;Fontanes;FONTANES;48300;48;LOZERE;1
48063;;Fontans;FONTANS;48700;48;LOZERE;1
48064;;Fournels;FOURNELS;48310;48;LOZERE;1
48065;;Fraissinet-de-Fourques;FRAISSINET-DE-FOURQUES;48400;48;LOZERE;1
48066;;Fraissinet-de-Lozre;FRAISSINET-DE-LOZERE;48220;48;LOZERE;1
48067;;Gabriac;GABRIAC;48110;48;LOZERE;1
48068;;Gabrias;GABRIAS;48100;48;LOZERE;1
48069;;Gatuzires;GATUZIERES;48150;48;LOZERE;1
48070;;Grandrieu;GRANDRIEU;48600;48;LOZERE;1
48071;;Grandvals;GRANDVALS;48260;48;LOZERE;1
48072;;Grzes;GREZES;48100;48;LOZERE;1
48073;Les;Hermaux;HERMAUX;48340;48;LOZERE;1
48074;;Hures-la-Parade;HURES-LA-PARADE;48150;48;LOZERE;1
48075;;Ispagnac;ISPAGNAC;48320;48;LOZERE;1
48076;;Javols;JAVOLS;48130;48;LOZERE;1
48077;;Julianges;JULIANGES;48140;48;LOZERE;1
48078;;Lachamp;LACHAMP;48100;48;LOZERE;1
48079;;Lajo;LAJO;48120;48;LOZERE;1
48080;;Langogne;LANGOGNE;48300;48;LOZERE;1
48081;;Lanujols;LANUEJOLS;48000;48;LOZERE;1
48082;;Laubert;LAUBERT;48170;48;LOZERE;1
48083;Les;Laubies;LAUBIES;48700;48;LOZERE;1
48084;;Laval-Atger;LAVAL-ATGER;48600;48;LOZERE;1
48085;;Laval-du-Tarn;LAVAL-DU-TARN;48500;48;LOZERE;1
48086;;Luc;LUC;48250;48;LOZERE;1
48087;;Malbouzon;MALBOUZON;48270;48;LOZERE;1
48088;La;Malne;MALENE;48210;48;LOZERE;1
48089;Le;Malzieu-Forain;MALZIEU-FORAIN;48140;48;LOZERE;1
48090;Le;Malzieu-Ville;MALZIEU-VILLE;48140;48;LOZERE;1
48091;;Marchastel;MARCHASTEL;48260;48;LOZERE;1
48092;;Marvejols;MARVEJOLS;48100;48;LOZERE;1
48093;;Mas-d'Orcires;MAS-D'ORCIERES;48190;48;LOZERE;1
48094;Le;Massegros;MASSEGROS;48500;48;LOZERE;1
48095;;Mende;MENDE;48000;48;LOZERE;1
48096;;Meyrueis;MEYRUEIS;48150;48;LOZERE;1
48097;;Moissac-Valle-Franaise;MOISSAC-VALLEE-FRANCAISE;48110;48;LOZERE;1
48098;;Molezon;MOLEZON;48110;48;LOZERE;1
48099;Le;Monastier-Pin-Moris;MONASTIER-PIN-MORIES;48100;48;LOZERE;1
48100;;Montbel;MONTBEL;48170;48;LOZERE;1
48101;;Montbrun;MONTBRUN;48210;48;LOZERE;1
48103;;Montrodat;MONTRODAT;48100;48;LOZERE;1
48104;;Nasbinals;NASBINALS;48260;48;LOZERE;1
48105;;Naussac;NAUSSAC;48300;48;LOZERE;1
48106;;Noalhac;NOALHAC;48310;48;LOZERE;1
48107;;Palhers;PALHERS;48100;48;LOZERE;1
48108;La;Panouse;PANOUSE;48600;48;LOZERE;1
48110;;Paulhac-en-Margeride;PAULHAC-EN-MARGERIDE;48140;48;LOZERE;1
48111;;Pelouse;PELOUSE;48000;48;LOZERE;1
48112;;Pierrefiche;PIERREFICHE;48300;48;LOZERE;1
48115;Le;Pompidou;POMPIDOU;48110;48;LOZERE;1
48116;Le;Pont-de-Montvert;PONT-DE-MONTVERT;48220;48;LOZERE;1
48117;;Pourcharesses;POURCHARESSES;48800;48;LOZERE;1
48119;;Prvenchres;PREVENCHERES;48800;48;LOZERE;1
48120;;Prinsujols;PRINSUEJOLS;48100;48;LOZERE;1
48121;;Prunires;PRUNIERES;48200;48;LOZERE;1
48122;;Quzac;QUEZAC;48320;48;LOZERE;1
48123;;Recoules-d'Aubrac;RECOULES-D'AUBRAC;48260;48;LOZERE;1
48124;;Recoules-de-Fumas;RECOULES-DE-FUMAS;48100;48;LOZERE;1
48125;Le;Recoux;RECOUX;48500;48;LOZERE;1
48126;;Ribennes;RIBENNES;48700;48;LOZERE;1
48127;;Rieutort-de-Randon;RIEUTORT-DE-RANDON;48700;48;LOZERE;1
48128;;Rimeize;RIMEIZE;48200;48;LOZERE;1
48129;;Rocles;ROCLES;48300;48;LOZERE;1
48130;;Rousses;ROUSSES;48400;48;LOZERE;1
48131;Le;Rozier;ROZIER;48150;48;LOZERE;1
48132;;Saint-Alban-sur-Limagnole;SAINT-ALBAN-SUR-LIMAGNOLE;48120;48;LOZERE;1
48133;;Saint-Amans;SAINT-AMANS;48700;48;LOZERE;1
48134;;Saint-Andol-de-Clerguemort;SAINT-ANDEOL-DE-CLERGUEMORT;48160;48;LOZERE;1
48135;;Saint-Andr-Capcze;SAINT-ANDRE-CAPCEZE;48800;48;LOZERE;1
48136;;Saint-Andr-de-Lancize;SAINT-ANDRE-DE-LANCIZE;48240;48;LOZERE;1
48137;;Saint-Bauzile;SAINT-BAUZILE;48000;48;LOZERE;1
48138;;Saint-Bonnet-de-Chirac;SAINT-BONNET-DE-CHIRAC;48100;48;LOZERE;1
48139;;Saint-Bonnet-de-Montauroux;SAINT-BONNET-DE-MONTAUROUX;48600;48;LOZERE;1
48140;;Saint-Chly-d'Apcher;SAINT-CHELY-D'APCHER;48200;48;LOZERE;1
48141;;Mas-Saint-Chly;MAS-SAINT-CHELY;48210;48;LOZERE;1
48142;;Sainte-Colombe-de-Peyre;SAINTE-COLOMBE-DE-PEYRE;48130;48;LOZERE;1
48144;;Sainte-Croix-Valle-Franaise;SAINTE-CROIX-VALLEE-FRANCAISE;48110;48;LOZERE;1
48145;;Saint-Denis-en-Margeride;SAINT-DENIS-EN-MARGERIDE;48700;48;LOZERE;1
48146;;Sainte-Enimie;SAINTE-ENIMIE;48210;48;LOZERE;1
48147;;Saint-tienne-du-Valdonnez;SAINT-ETIENNE-DU-VALDONNEZ;48000;48;LOZERE;1
48148;;Saint-tienne-Valle-Franaise;SAINT-ETIENNE-VALLEE-FRANCAISE;48330;48;LOZERE;1
48149;;Sainte-Eulalie;SAINTE-EULALIE;48120;48;LOZERE;1
48150;;Saint-Flour-de-Mercoire;SAINT-FLOUR-DE-MERCOIRE;48300;48;LOZERE;1
48151;;Saint-Frzal-d'Albuges;SAINT-FREZAL-D'ALBUGES;48170;48;LOZERE;1
48152;;Saint-Frzal-de-Ventalon;SAINT-FREZAL-DE-VENTALON;48240;48;LOZERE;1
48153;;Saint-Gal;SAINT-GAL;48700;48;LOZERE;1
48154;;Saint-Georges-de-Lvjac;SAINT-GEORGES-DE-LEVEJAC;48500;48;LOZERE;1
48155;;Saint-Germain-de-Calberte;SAINT-GERMAIN-DE-CALBERTE;48370;48;LOZERE;1
48156;;Saint-Germain-du-Teil;SAINT-GERMAIN-DU-TEIL;48340;48;LOZERE;1
48157;;Sainte-Hlne;SAINTE-HELENE;48190;48;LOZERE;1
48158;;Saint-Hilaire-de-Lavit;SAINT-HILAIRE-DE-LAVIT;48160;48;LOZERE;1
48160;;Saint-Jean-la-Fouillouse;SAINT-JEAN-LA-FOUILLOUSE;48170;48;LOZERE;1
48161;;Saint-Jury;SAINT-JUERY;48310;48;LOZERE;1
48162;;Saint-Julien-d'Arpaon;SAINT-JULIEN-D'ARPAON;48400;48;LOZERE;1
48163;;Saint-Julien-des-Points;SAINT-JULIEN-DES-POINTS;48160;48;LOZERE;1
48164;;Saint-Julien-du-Tournel;SAINT-JULIEN-DU-TOURNEL;48190;48;LOZERE;1
48165;;Saint-Laurent-de-Muret;SAINT-LAURENT-DE-MURET;48100;48;LOZERE;1
48166;;Saint-Laurent-de-Trves;SAINT-LAURENT-DE-TREVES;48400;48;LOZERE;1
48167;;Saint-Laurent-de-Veyrs;SAINT-LAURENT-DE-VEYRES;48310;48;LOZERE;1
48168;;Saint-Lger-de-Peyre;SAINT-LEGER-DE-PEYRE;48100;48;LOZERE;1
48169;;Saint-Lger-du-Malzieu;SAINT-LEGER-DU-MALZIEU;48140;48;LOZERE;1
48170;;Saint-Martin-de-Boubaux;SAINT-MARTIN-DE-BOUBAUX;48160;48;LOZERE;1
48171;;Saint-Martin-de-Lansuscle;SAINT-MARTIN-DE-LANSUSCLE;48110;48;LOZERE;1
48172;;Saint-Maurice-de-Ventalon;SAINT-MAURICE-DE-VENTALON;48220;48;LOZERE;1
48173;;Saint-Michel-de-Dze;SAINT-MICHEL-DE-DEZE;48160;48;LOZERE;1
48174;;Saint-Paul-le-Froid;SAINT-PAUL-LE-FROID;48600;48;LOZERE;1
48175;;Saint-Pierre-de-Nogaret;SAINT-PIERRE-DE-NOGARET;48340;48;LOZERE;1
48176;;Saint-Pierre-des-Tripiers;SAINT-PIERRE-DES-TRIPIERS;48150;48;LOZERE;1
48177;;Saint-Pierre-le-Vieux;SAINT-PIERRE-LE-VIEUX;48200;48;LOZERE;1
48178;;Saint-Privat-de-Vallongue;SAINT-PRIVAT-DE-VALLONGUE;48240;48;LOZERE;1
48179;;Saint-Privat-du-Fau;SAINT-PRIVAT-DU-FAU;48140;48;LOZERE;1
48180;;Saint-Rome-de-Dolan;SAINT-ROME-DE-DOLAN;48500;48;LOZERE;1
48181;;Saint-Saturnin;SAINT-SATURNIN;48500;48;LOZERE;1
48182;;Saint-Sauveur-de-Ginestoux;SAINT-SAUVEUR-DE-GINESTOUX;48170;48;LOZERE;1
48183;;Saint-Sauveur-de-Peyre;SAINT-SAUVEUR-DE-PEYRE;48130;48;LOZERE;1
48184;;Saint-Symphorien;SAINT-SYMPHORIEN;48600;48;LOZERE;1
48185;Les;Salelles;SALELLES;48230;48;LOZERE;1
48186;La;Salle-Prunet;SALLE-PRUNET;48400;48;LOZERE;1
48187;Les;Salces;SALCES;48100;48;LOZERE;1
48188;;Serverette;SERVERETTE;48700;48;LOZERE;1
48189;;Servires;SERVIERES;48000;48;LOZERE;1
48190;;Termes;TERMES;48310;48;LOZERE;1
48191;La;Tieule;TIEULE;48500;48;LOZERE;1
48192;;Trlans;TRELANS;48340;48;LOZERE;1
48193;;Vebron;VEBRON;48400;48;LOZERE;1
48194;;Vialas;VIALAS;48220;48;LOZERE;1
48195;Les;Vignes;VIGNES;48210;48;LOZERE;1
48197;La;Villedieu;VILLEDIEU;48700;48;LOZERE;1
48198;;Villefort;VILLEFORT;48800;48;LOZERE;1
49001;Les;Alleuds;ALLEUDS;49320;49;MAINE ET LOIRE;1
49002;;Allonnes;ALLONNES;49650;49;MAINE ET LOIRE;1
49003;;Ambillou-Chteau;AMBILLOU-CHATEAU;49700;49;MAINE ET LOIRE;1
49004;;Andard;ANDARD;49800;49;MAINE ET LOIRE;1
49005;;Andign;ANDIGNE;49220;49;MAINE ET LOIRE;1
49006;;Andrez;ANDREZE;49600;49;MAINE ET LOIRE;1
49007;;Angers;ANGERS;49000;49;MAINE ET LOIRE;1
49008;;Angrie;ANGRIE;49440;49;MAINE ET LOIRE;1
49009;;Antoign;ANTOIGNE;49260;49;MAINE ET LOIRE;1
49010;;Armaill;ARMAILLE;49420;49;MAINE ET LOIRE;1
49011;;Artannes-sur-Thouet;ARTANNES-SUR-THOUET;49260;49;MAINE ET LOIRE;1
49012;;Aubign-sur-Layon;AUBIGNE-SUR-LAYON;49540;49;MAINE ET LOIRE;1
49013;;Auverse;AUVERSE;49490;49;MAINE ET LOIRE;1
49014;;Avir;AVIRE;49500;49;MAINE ET LOIRE;1
49015;;Avrill;AVRILLE;49240;49;MAINE ET LOIRE;1
49017;;Barac;BARACE;49430;49;MAINE ET LOIRE;1
49018;;Baug;BAUGE;49150;49;MAINE ET LOIRE;1
49019;;Baun;BAUNE;49140;49;MAINE ET LOIRE;1
49020;;Beaucouz;BEAUCOUZE;49070;49;MAINE ET LOIRE;1
49021;;Beaufort-en-Valle;BEAUFORT-EN-VALLEE;49250;49;MAINE ET LOIRE;1
49022;;Beaulieu-sur-Layon;BEAULIEU-SUR-LAYON;49750;49;MAINE ET LOIRE;1
49023;;Beauprau;BEAUPREAU;49600;49;MAINE ET LOIRE;1
49024;;Beausse;BEAUSSE;49410;49;MAINE ET LOIRE;1
49025;;Beauvau;BEAUVAU;49140;49;MAINE ET LOIRE;1
49026;;Bcon-les-Granits;BECON-LES-GRANITS;49370;49;MAINE ET LOIRE;1
49027;;Bgrolles-en-Mauges;BEGROLLES-EN-MAUGES;49122;49;MAINE ET LOIRE;1
49028;;Bhuard;BEHUARD;49170;49;MAINE ET LOIRE;1
49029;;Blaison-Gohier;BLAISON-GOHIER;49320;49;MAINE ET LOIRE;1
49030;;Blou;BLOU;49160;49;MAINE ET LOIRE;1
49031;;Boc;BOCE;49150;49;MAINE ET LOIRE;1
49032;La;Bohalle;BOHALLE;49800;49;MAINE ET LOIRE;1
49033;La;Boissire-sur-vre;BOISSIERE-SUR-EVRE;49110;49;MAINE ET LOIRE;1
49034;;Botz-en-Mauges;BOTZ-EN-MAUGES;49110;49;MAINE ET LOIRE;1
49035;;Bouchemaine;BOUCHEMAINE;49080;49;MAINE ET LOIRE;1
49036;;Bouill-Mnard;BOUILLE-MENARD;49520;49;MAINE ET LOIRE;1
49037;Le;Bourg-d'Ir;BOURG-D'IRE;49520;49;MAINE ET LOIRE;1
49038;;Bourg-l'vque;BOURG-L'EVEQUE;49520;49;MAINE ET LOIRE;1
49039;;Bourgneuf-en-Mauges;BOURGNEUF-EN-MAUGES;49290;49;MAINE ET LOIRE;1
49040;;Bouzill;BOUZILLE;49530;49;MAINE ET LOIRE;1
49041;;Brain-sur-Allonnes;BRAIN-SUR-ALLONNES;49650;49;MAINE ET LOIRE;1
49042;;Brain-sur-l'Authion;BRAIN-SUR-L'AUTHION;49800;49;MAINE ET LOIRE;1
49043;;Brain-sur-Longuene;BRAIN-SUR-LONGUENEE;49220;49;MAINE ET LOIRE;1
49044;;Breil;BREIL;49490;49;MAINE ET LOIRE;1
49045;La;Breille-les-Pins;BREILLE-LES-PINS;49390;49;MAINE ET LOIRE;1
49046;;Brz;BREZE;49260;49;MAINE ET LOIRE;1
49047;;Brign;BRIGNE;49700;49;MAINE ET LOIRE;1
49048;;Briollay;BRIOLLAY;49125;49;MAINE ET LOIRE;1
49049;;Brion;BRION;49250;49;MAINE ET LOIRE;1
49050;;Brissac-Quinc;BRISSAC-QUINCE;49320;49;MAINE ET LOIRE;1
49051;;Brissarthe;BRISSARTHE;49330;49;MAINE ET LOIRE;1
49052;;Broc;BROC;49490;49;MAINE ET LOIRE;1
49053;;Brossay;BROSSAY;49700;49;MAINE ET LOIRE;1
49054;;Cand;CANDE;49440;49;MAINE ET LOIRE;1
49055;;Cantenay-pinard;CANTENAY-EPINARD;49460;49;MAINE ET LOIRE;1
49056;;Carbay;CARBAY;49420;49;MAINE ET LOIRE;1
49057;;Cernusson;CERNUSSON;49310;49;MAINE ET LOIRE;1
49058;Les;Cerqueux;CERQUEUX;49360;49;MAINE ET LOIRE;1
49059;Les;Cerqueux-sous-Passavant;CERQUEUX-SOUS-PASSAVANT;49310;49;MAINE ET LOIRE;1
49060;;Chac;CHACE;49400;49;MAINE ET LOIRE;1
49061;;Challain-la-Potherie;CHALLAIN-LA-POTHERIE;49440;49;MAINE ET LOIRE;1
49062;;Chalonnes-sous-le-Lude;CHALONNES-SOUS-LE-LUDE;49490;49;MAINE ET LOIRE;1
49063;;Chalonnes-sur-Loire;CHALONNES-SUR-LOIRE;49290;49;MAINE ET LOIRE;1
49064;;Chambellay;CHAMBELLAY;49220;49;MAINE ET LOIRE;1
49065;;Champign;CHAMPIGNE;49330;49;MAINE ET LOIRE;1
49066;;Champ-sur-Layon;CHAMP-SUR-LAYON;49380;49;MAINE ET LOIRE;1
49067;;Champteuss-sur-Baconne;CHAMPTEUSSE-SUR-BACONNE;49220;49;MAINE ET LOIRE;1
49068;;Champtoc-sur-Loire;CHAMPTOCE-SUR-LOIRE;49123;49;MAINE ET LOIRE;1
49069;;Champtoceaux;CHAMPTOCEAUX;49270;49;MAINE ET LOIRE;1
49070;;Chanteloup-les-Bois;CHANTELOUP-LES-BOIS;49340;49;MAINE ET LOIRE;1
49071;;Chanzeaux;CHANZEAUX;49750;49;MAINE ET LOIRE;1
49072;La;Chapelle-du-Gent;CHAPELLE-DU-GENET;49600;49;MAINE ET LOIRE;1
49073;La;Chapelle-Hullin;CHAPELLE-HULLIN;49860;49;MAINE ET LOIRE;1
49074;La;Chapelle-Rousselin;CHAPELLE-ROUSSELIN;49120;49;MAINE ET LOIRE;1
49075;La;Chapelle-Saint-Florent;CHAPELLE-SAINT-FLORENT;49410;49;MAINE ET LOIRE;1
49076;La;Chapelle-Saint-Laud;CHAPELLE-SAINT-LAUD;49140;49;MAINE ET LOIRE;1
49077;La;Chapelle-sur-Oudon;CHAPELLE-SUR-OUDON;49500;49;MAINE ET LOIRE;1
49078;;Charc-Saint-Ellier-sur-Aubance;CHARCE-SAINT-ELLIER-SUR-AUBANCE;49320;49;MAINE ET LOIRE;1
49079;;Chartren;CHARTRENE;49150;49;MAINE ET LOIRE;1
49080;;Chteauneuf-sur-Sarthe;CHATEAUNEUF-SUR-SARTHE;49330;49;MAINE ET LOIRE;1
49081;;Chtelais;CHATELAIS;49520;49;MAINE ET LOIRE;1
49082;;Chaudefonds-sur-Layon;CHAUDEFONDS-SUR-LAYON;49290;49;MAINE ET LOIRE;1
49083;;Chaudron-en-Mauges;CHAUDRON-EN-MAUGES;49110;49;MAINE ET LOIRE;1
49084;;Chaumont-d'Anjou;CHAUMONT-D'ANJOU;49140;49;MAINE ET LOIRE;1
49085;La;Chaussaire;CHAUSSAIRE;49600;49;MAINE ET LOIRE;1
49086;;Chavagnes;CHAVAGNES;49380;49;MAINE ET LOIRE;1
49087;;Chavaignes;CHAVAIGNES;49490;49;MAINE ET LOIRE;1
49088;;Chaz-Henry;CHAZE-HENRY;49860;49;MAINE ET LOIRE;1
49089;;Chaz-sur-Argos;CHAZE-SUR-ARGOS;49500;49;MAINE ET LOIRE;1
49090;;Cheffes;CHEFFES;49125;49;MAINE ET LOIRE;1
49091;;Chemellier;CHEMELLIER;49320;49;MAINE ET LOIRE;1
49092;;Chemill;CHEMILLE;49120;49;MAINE ET LOIRE;1
49093;;Chemir-sur-Sarthe;CHEMIRE-SUR-SARTHE;49640;49;MAINE ET LOIRE;1
49094;;Chnehutte-Trves-Cunault;CHENEHUTTE-TREVES-CUNAULT;49350;49;MAINE ET LOIRE;1
49095;;Chenill-Chang;CHENILLE-CHANGE;49220;49;MAINE ET LOIRE;1
49096;;Cherr;CHERRE;49330;49;MAINE ET LOIRE;1
49097;;Chevir-le-Rouge;CHEVIRE-LE-ROUGE;49150;49;MAINE ET LOIRE;1
49098;;Chign;CHIGNE;49490;49;MAINE ET LOIRE;1
49099;;Cholet;CHOLET;49300;49;MAINE ET LOIRE;1
49100;;Cizay-la-Madeleine;CIZAY-LA-MADELEINE;49700;49;MAINE ET LOIRE;1
49101;;Clefs;CLEFS;49150;49;MAINE ET LOIRE;1
49102;;Clr-sur-Layon;CLERE-SUR-LAYON;49560;49;MAINE ET LOIRE;1
49103;;Combre;COMBREE;49520;49;MAINE ET LOIRE;1
49104;;Concourson-sur-Layon;CONCOURSON-SUR-LAYON;49700;49;MAINE ET LOIRE;1
49105;;Contign;CONTIGNE;49330;49;MAINE ET LOIRE;1
49106;;Corn;CORNE;49250;49;MAINE ET LOIRE;1
49107;;Cornill-les-Caves;CORNILLE-LES-CAVES;49140;49;MAINE ET LOIRE;1
49108;La;Cornuaille;CORNUAILLE;49440;49;MAINE ET LOIRE;1
49109;;Coron;CORON;49690;49;MAINE ET LOIRE;1
49110;;Corz;CORZE;49140;49;MAINE ET LOIRE;1
49111;;Coss-d'Anjou;COSSE-D'ANJOU;49120;49;MAINE ET LOIRE;1
49112;Le;Coudray-Macouard;COUDRAY-MACOUARD;49260;49;MAINE ET LOIRE;1
49113;;Courchamps;COURCHAMPS;49260;49;MAINE ET LOIRE;1
49114;;Courlon;COURLEON;49390;49;MAINE ET LOIRE;1
49115;;Coutures;COUTURES;49320;49;MAINE ET LOIRE;1
49116;;Cuon;CUON;49150;49;MAINE ET LOIRE;1
49117;La;Daguenire;DAGUENIERE;49800;49;MAINE ET LOIRE;1
49119;;Daumeray;DAUMERAY;49640;49;MAINE ET LOIRE;1
49120;;Dene;DENEE;49190;49;MAINE ET LOIRE;1
49121;;Dnez-sous-Dou;DENEZE-SOUS-DOUE;49700;49;MAINE ET LOIRE;1
49122;;Dnez-sous-le-Lude;DENEZE-SOUS-LE-LUDE;49490;49;MAINE ET LOIRE;1
49123;;Distr;DISTRE;49400;49;MAINE ET LOIRE;1
49125;;Dou-la-Fontaine;DOUE-LA-FONTAINE;49700;49;MAINE ET LOIRE;1
49126;;Drain;DRAIN;49530;49;MAINE ET LOIRE;1
49127;;Durtal;DURTAL;49430;49;MAINE ET LOIRE;1
49128;;chemir;ECHEMIRE;49150;49;MAINE ET LOIRE;1
49129;;couflant;ECOUFLANT;49000;49;MAINE ET LOIRE;1
49130;;cuill;ECUILLE;49460;49;MAINE ET LOIRE;1
49131;;pieds;EPIEDS;49260;49;MAINE ET LOIRE;1
49132;;trich;ETRICHE;49330;49;MAINE ET LOIRE;1
49133;;Faveraye-Mchelles;FAVERAYE-MACHELLES;49380;49;MAINE ET LOIRE;1
49134;;Faye-d'Anjou;FAYE-D'ANJOU;49380;49;MAINE ET LOIRE;1
49135;;Feneu;FENEU;49460;49;MAINE ET LOIRE;1
49136;La;Ferrire-de-Fle;FERRIERE-DE-FLEE;49500;49;MAINE ET LOIRE;1
49137;Le;Fief-Sauvin;FIEF-SAUVIN;49600;49;MAINE ET LOIRE;1
49138;;Fontaine-Gurin;FONTAINE-GUERIN;49250;49;MAINE ET LOIRE;1
49139;;Fontaine-Milon;FONTAINE-MILON;49140;49;MAINE ET LOIRE;1
49140;;Fontevraud-l'Abbaye;FONTEVRAUD-L'ABBAYE;49590;49;MAINE ET LOIRE;1
49141;;Forges;FORGES;49700;49;MAINE ET LOIRE;1
49142;La;Fosse-de-Tign;FOSSE-DE-TIGNE;49540;49;MAINE ET LOIRE;1
49143;;Fouger;FOUGERE;49150;49;MAINE ET LOIRE;1
49144;;Freign;FREIGNE;49440;49;MAINE ET LOIRE;1
49145;Le;Fuilet;FUILET;49270;49;MAINE ET LOIRE;1
49147;;Ge;GEE;49250;49;MAINE ET LOIRE;1
49148;;Gen;GENE;49220;49;MAINE ET LOIRE;1
49149;;Gennes;GENNES;49350;49;MAINE ET LOIRE;1
49150;;Genneteil;GENNETEIL;49490;49;MAINE ET LOIRE;1
49151;;Gest;GESTE;49600;49;MAINE ET LOIRE;1
49153;;Valanjou;VALANJOU;49670;49;MAINE ET LOIRE;1
49154;;Grzill;GREZILLE;49320;49;MAINE ET LOIRE;1
49155;;Grez-Neuville;GREZ-NEUVILLE;49220;49;MAINE ET LOIRE;1
49156;;Grug-l'Hpital;GRUGE-L'HOPITAL;49520;49;MAINE ET LOIRE;1
49157;Le;Gudeniau;GUEDENIAU;49150;49;MAINE ET LOIRE;1
49158;L';Htellerie-de-Fle;HOTELLERIE-DE-FLEE;49500;49;MAINE ET LOIRE;1
49159;;Huill;HUILLE;49430;49;MAINE ET LOIRE;1
49160;;Ingrandes;INGRANDES;49123;49;MAINE ET LOIRE;1
49161;La;Jaille-Yvon;JAILLE-YVON;49220;49;MAINE ET LOIRE;1
49162;;Jallais;JALLAIS;49510;49;MAINE ET LOIRE;1
49163;;Jarz;JARZE;49140;49;MAINE ET LOIRE;1
49165;La;Jubaudire;JUBAUDIERE;49510;49;MAINE ET LOIRE;1
49167;;Juign-sur-Loire;JUIGNE-SUR-LOIRE;49610;49;MAINE ET LOIRE;1
49169;La;Jumellire;JUMELLIERE;49120;49;MAINE ET LOIRE;1
49170;;Juvardeil;JUVARDEIL;49330;49;MAINE ET LOIRE;1
49171;La;Lande-Chasles;LANDE-CHASLES;49150;49;MAINE ET LOIRE;1
49172;;Landemont;LANDEMONT;49270;49;MAINE ET LOIRE;1
49173;;Lasse;LASSE;49490;49;MAINE ET LOIRE;1
49174;;Lzign;LEZIGNE;49430;49;MAINE ET LOIRE;1
49175;;Linires-Bouton;LINIERES-BOUTON;49490;49;MAINE ET LOIRE;1
49176;Le;Lion-d'Angers;LION-D'ANGERS;49220;49;MAINE ET LOIRE;1
49177;;Lir;LIRE;49530;49;MAINE ET LOIRE;1
49178;;Loir;LOIRE;49440;49;MAINE ET LOIRE;1
49179;Le;Longeron;LONGERON;49710;49;MAINE ET LOIRE;1
49180;;Longu-Jumelles;LONGUE-JUMELLES;49160;49;MAINE ET LOIRE;1
49181;;Louerre;LOUERRE;49700;49;MAINE ET LOIRE;1
49182;;Louresse-Rochemenier;LOURESSE-ROCHEMENIER;49700;49;MAINE ET LOIRE;1
49183;Le;Louroux-Bconnais;LOUROUX-BECONNAIS;49370;49;MAINE ET LOIRE;1
49184;;Louvaines;LOUVAINES;49500;49;MAINE ET LOIRE;1
49185;;Lu-en-Baugeois;LUE-EN-BAUGEOIS;49140;49;MAINE ET LOIRE;1
49186;;Luign;LUIGNE;49320;49;MAINE ET LOIRE;1
49187;;Marans;MARANS;49500;49;MAINE ET LOIRE;1
49188;;Marc;MARCE;49140;49;MAINE ET LOIRE;1
49189;;Marign;MARIGNE;49330;49;MAINE ET LOIRE;1
49190;Le;Marillais;MARILLAIS;49410;49;MAINE ET LOIRE;1
49191;;Martign-Briand;MARTIGNE-BRIAND;49540;49;MAINE ET LOIRE;1
49192;;Maulvrier;MAULEVRIER;49360;49;MAINE ET LOIRE;1
49193;Le;May-sur-vre;MAY-SUR-EVRE;49122;49;MAINE ET LOIRE;1
49194;;Maz;MAZE;49250;49;MAINE ET LOIRE;1
49195;;Mazires-en-Mauges;MAZIERES-EN-MAUGES;49280;49;MAINE ET LOIRE;1
49196;La;Meignanne;MEIGNANNE;49220;49;MAINE ET LOIRE;1
49197;;Meign-le-Vicomte;MEIGNE-LE-VICOMTE;49490;49;MAINE ET LOIRE;1
49198;;Meign;MEIGNE;49700;49;MAINE ET LOIRE;1
49199;;Melay;MELAY;49120;49;MAINE ET LOIRE;1
49200;La;Membrolle-sur-Longuene;MEMBROLLE-SUR-LONGUENEE;49220;49;MAINE ET LOIRE;1
49201;La;Mnitr;MENITRE;49250;49;MAINE ET LOIRE;1
49202;;Mon;MEON;49490;49;MAINE ET LOIRE;1
49204;Le;Mesnil-en-Valle;MESNIL-EN-VALLEE;49410;49;MAINE ET LOIRE;1
49205;;Mir;MIRE;49330;49;MAINE ET LOIRE;1
49206;;Montfaucon-Montign;MONTFAUCON-MONTIGNE;49230;49;MAINE ET LOIRE;1
49207;;Montfort;MONTFORT;49700;49;MAINE ET LOIRE;1
49208;;Montguillon;MONTGUILLON;49500;49;MAINE ET LOIRE;1
49209;;Montign-ls-Rairies;MONTIGNE-LES-RAIRIES;49430;49;MAINE ET LOIRE;1
49211;;Montilliers;MONTILLIERS;49310;49;MAINE ET LOIRE;1
49212;;Montjean-sur-Loire;MONTJEAN-SUR-LOIRE;49570;49;MAINE ET LOIRE;1
49213;;Montpollin;MONTPOLLIN;49150;49;MAINE ET LOIRE;1
49214;;Montreuil-Juign;MONTREUIL-JUIGNE;49460;49;MAINE ET LOIRE;1
49215;;Montreuil-Bellay;MONTREUIL-BELLAY;49260;49;MAINE ET LOIRE;1
49216;;Montreuil-sur-Loir;MONTREUIL-SUR-LOIR;49140;49;MAINE ET LOIRE;1
49217;;Montreuil-sur-Maine;MONTREUIL-SUR-MAINE;49220;49;MAINE ET LOIRE;1
49218;;Montrevault;MONTREVAULT;49110;49;MAINE ET LOIRE;1
49219;;Montsoreau;MONTSOREAU;49730;49;MAINE ET LOIRE;1
49220;;Morannes;MORANNES;49640;49;MAINE ET LOIRE;1
49221;;Mouliherne;MOULIHERNE;49390;49;MAINE ET LOIRE;1
49222;;Moz-sur-Louet;MOZE-SUR-LOUET;49610;49;MAINE ET LOIRE;1
49223;;Mrs-Erign;MURS-ERIGNE;49130;49;MAINE ET LOIRE;1
49224;;Neuill;NEUILLE;49680;49;MAINE ET LOIRE;1
49225;;Neuvy-en-Mauges;NEUVY-EN-MAUGES;49120;49;MAINE ET LOIRE;1
49226;;Nollet;NOELLET;49520;49;MAINE ET LOIRE;1
49227;;Notre-Dame-d'Allenon;NOTRE-DAME-D'ALLENCON;49380;49;MAINE ET LOIRE;1
49228;;Noyant;NOYANT;49490;49;MAINE ET LOIRE;1
49229;;Noyant-la-Gravoyre;NOYANT-LA-GRAVOYERE;49520;49;MAINE ET LOIRE;1
49230;;Noyant-la-Plaine;NOYANT-LA-PLAINE;49700;49;MAINE ET LOIRE;1
49231;;Nuaill;NUAILLE;49340;49;MAINE ET LOIRE;1
49232;;Nueil-sur-Layon;NUEIL-SUR-LAYON;49560;49;MAINE ET LOIRE;1
49233;;Nyoiseau;NYOISEAU;49500;49;MAINE ET LOIRE;1
49234;;Paray-les-Pins;PARCAY-LES-PINS;49390;49;MAINE ET LOIRE;1
49235;;Parnay;PARNAY;49730;49;MAINE ET LOIRE;1
49236;;Passavant-sur-Layon;PASSAVANT-SUR-LAYON;49560;49;MAINE ET LOIRE;1
49237;La;Pellerine;PELLERINE;49490;49;MAINE ET LOIRE;1
49238;;Pellouailles-les-Vignes;PELLOUAILLES-LES-VIGNES;49112;49;MAINE ET LOIRE;1
49239;Le;Pin-en-Mauges;PIN-EN-MAUGES;49110;49;MAINE ET LOIRE;1
49240;La;Plaine;PLAINE;49360;49;MAINE ET LOIRE;1
49241;Le;Plessis-Grammoire;PLESSIS-GRAMMOIRE;49124;49;MAINE ET LOIRE;1
49242;Le;Plessis-Mac;PLESSIS-MACE;49220;49;MAINE ET LOIRE;1
49243;La;Poitevinire;POITEVINIERE;49510;49;MAINE ET LOIRE;1
49244;La;Pommeraye;POMMERAYE;49620;49;MAINE ET LOIRE;1
49245;;Pontign;PONTIGNE;49150;49;MAINE ET LOIRE;1
49246;Les;Ponts-de-C;PONTS-DE-CE;49130;49;MAINE ET LOIRE;1
49247;La;Possonnire;POSSONNIERE;49170;49;MAINE ET LOIRE;1
49248;;Pouanc;POUANCE;49420;49;MAINE ET LOIRE;1
49249;La;Pouze;POUEZE;49370;49;MAINE ET LOIRE;1
49250;La;Prvire;PREVIERE;49420;49;MAINE ET LOIRE;1
49251;;Pruill;PRUILLE;49220;49;MAINE ET LOIRE;1
49252;Le;Puiset-Dor;PUISET-DORE;49600;49;MAINE ET LOIRE;1
49253;Le;Puy-Notre-Dame;PUY-NOTRE-DAME;49260;49;MAINE ET LOIRE;1
49254;;Querr;QUERRE;49330;49;MAINE ET LOIRE;1
49256;;Rablay-sur-Layon;RABLAY-SUR-LAYON;49750;49;MAINE ET LOIRE;1
49257;Les;Rairies;RAIRIES;49430;49;MAINE ET LOIRE;1
49258;La;Renaudire;RENAUDIERE;49450;49;MAINE ET LOIRE;1
49259;;Rochefort-sur-Loire;ROCHEFORT-SUR-LOIRE;49190;49;MAINE ET LOIRE;1
49260;La;Romagne;ROMAGNE;49740;49;MAINE ET LOIRE;1
49261;Les;Rosiers-sur-Loire;ROSIERS-SUR-LOIRE;49350;49;MAINE ET LOIRE;1
49262;;Rou-Marson;ROU-MARSON;49400;49;MAINE ET LOIRE;1
49263;;Roussay;ROUSSAY;49450;49;MAINE ET LOIRE;1
49264;;Saint-Andr-de-la-Marche;SAINT-ANDRE-DE-LA-MARCHE;49450;49;MAINE ET LOIRE;1
49265;;Saint-Aubin-de-Luign;SAINT-AUBIN-DE-LUIGNE;49190;49;MAINE ET LOIRE;1
49266;;Saint-Augustin-des-Bois;SAINT-AUGUSTIN-DES-BOIS;49170;49;MAINE ET LOIRE;1
49267;;Saint-Barthlemy-d'Anjou;SAINT-BARTHELEMY-D'ANJOU;49124;49;MAINE ET LOIRE;1
49268;;Sainte-Christine;SAINTE-CHRISTINE;49120;49;MAINE ET LOIRE;1
49269;;Saint-Christophe-du-Bois;SAINT-CHRISTOPHE-DU-BOIS;49280;49;MAINE ET LOIRE;1
49270;;Saint-Christophe-la-Couperie;SAINT-CHRISTOPHE-LA-COUPERIE;49270;49;MAINE ET LOIRE;1
49271;;Saint-Clment-de-la-Place;SAINT-CLEMENT-DE-LA-PLACE;49370;49;MAINE ET LOIRE;1
49272;;Saint-Clment-des-Leves;SAINT-CLEMENT-DES-LEVEES;49350;49;MAINE ET LOIRE;1
49273;;Saint-Crespin-sur-Moine;SAINT-CRESPIN-SUR-MOINE;49230;49;MAINE ET LOIRE;1
49274;;Saint-Cyr-en-Bourg;SAINT-CYR-EN-BOURG;49260;49;MAINE ET LOIRE;1
49276;;Saint-Florent-le-Vieil;SAINT-FLORENT-LE-VIEIL;49410;49;MAINE ET LOIRE;1
49277;;Sainte-Gemmes-d'Andign;SAINTE-GEMMES-D'ANDIGNE;49500;49;MAINE ET LOIRE;1
49278;;Sainte-Gemmes-sur-Loire;SAINTE-GEMMES-SUR-LOIRE;49130;49;MAINE ET LOIRE;1
49279;;Saint-Georges-des-Sept-Voies;SAINT-GEORGES-DES-SEPT-VOIES;49350;49;MAINE ET LOIRE;1
49280;;Saint-Georges-du-Bois;SAINT-GEORGES-DU-BOIS;49250;49;MAINE ET LOIRE;1
49281;;Saint-Georges-des-Gardes;SAINT-GEORGES-DES-GARDES;49120;49;MAINE ET LOIRE;1
49282;;Saint-Georges-sur-Layon;SAINT-GEORGES-SUR-LAYON;49700;49;MAINE ET LOIRE;1
49283;;Saint-Georges-sur-Loire;SAINT-GEORGES-SUR-LOIRE;49170;49;MAINE ET LOIRE;1
49284;;Saint-Germain-des-Prs;SAINT-GERMAIN-DES-PRES;49170;49;MAINE ET LOIRE;1
49285;;Saint-Germain-sur-Moine;SAINT-GERMAIN-SUR-MOINE;49230;49;MAINE ET LOIRE;1
49288;;Saint-Jean-de-la-Croix;SAINT-JEAN-DE-LA-CROIX;49130;49;MAINE ET LOIRE;1
49289;;Saint-Jean-de-Linires;SAINT-JEAN-DE-LINIERES;49070;49;MAINE ET LOIRE;1
49290;;Saint-Jean-des-Mauvrets;SAINT-JEAN-DES-MAUVRETS;49320;49;MAINE ET LOIRE;1
49291;;Saint-Just-sur-Dive;SAINT-JUST-SUR-DIVE;49260;49;MAINE ET LOIRE;1
49292;;Saint-Lambert-du-Lattay;SAINT-LAMBERT-DU-LATTAY;49750;49;MAINE ET LOIRE;1
49294;;Saint-Lambert-la-Potherie;SAINT-LAMBERT-LA-POTHERIE;49070;49;MAINE ET LOIRE;1
49295;;Saint-Laurent-de-la-Plaine;SAINT-LAURENT-DE-LA-PLAINE;49290;49;MAINE ET LOIRE;1
49296;;Saint-Laurent-des-Autels;SAINT-LAURENT-DES-AUTELS;49270;49;MAINE ET LOIRE;1
49297;;Saint-Laurent-du-Mottay;SAINT-LAURENT-DU-MOTTAY;49410;49;MAINE ET LOIRE;1
49298;;Saint-Lger-des-Bois;SAINT-LEGER-DES-BOIS;49170;49;MAINE ET LOIRE;1
49299;;Saint-Lger-sous-Cholet;SAINT-LEGER-SOUS-CHOLET;49280;49;MAINE ET LOIRE;1
49300;;Saint-Lzin;SAINT-LEZIN;49120;49;MAINE ET LOIRE;1
49301;;Saint-Macaire-en-Mauges;SAINT-MACAIRE-EN-MAUGES;49450;49;MAINE ET LOIRE;1
49302;;Saint-Macaire-du-Bois;SAINT-MACAIRE-DU-BOIS;49260;49;MAINE ET LOIRE;1
49303;;Saint-Martin-d'Arc;SAINT-MARTIN-D'ARCE;49150;49;MAINE ET LOIRE;1
49304;;Saint-Martin-de-la-Place;SAINT-MARTIN-DE-LA-PLACE;49160;49;MAINE ET LOIRE;1
49305;;Saint-Martin-du-Bois;SAINT-MARTIN-DU-BOIS;49500;49;MAINE ET LOIRE;1
49306;;Saint-Martin-du-Fouilloux;SAINT-MARTIN-DU-FOUILLOUX;49170;49;MAINE ET LOIRE;1
49307;;Saint-Mathurin-sur-Loire;SAINT-MATHURIN-SUR-LOIRE;49250;49;MAINE ET LOIRE;1
49308;;Saint-Melaine-sur-Aubance;SAINT-MELAINE-SUR-AUBANCE;49610;49;MAINE ET LOIRE;1
49309;;Saint-Michel-et-Chanveaux;SAINT-MICHEL-ET-CHANVEAUX;49420;49;MAINE ET LOIRE;1
49310;;Saint-Paul-du-Bois;SAINT-PAUL-DU-BOIS;49310;49;MAINE ET LOIRE;1
49311;;Saint-Philbert-du-Peuple;SAINT-PHILBERT-DU-PEUPLE;49160;49;MAINE ET LOIRE;1
49312;;Saint-Philbert-en-Mauges;SAINT-PHILBERT-EN-MAUGES;49600;49;MAINE ET LOIRE;1
49313;;Saint-Pierre-Montlimart;SAINT-PIERRE-MONTLIMART;49110;49;MAINE ET LOIRE;1
49314;;Saint-Quentin-en-Mauges;SAINT-QUENTIN-EN-MAUGES;49110;49;MAINE ET LOIRE;1
49315;;Saint-Quentin-ls-Beaurepaire;SAINT-QUENTIN-LES-BEAUREPAIRE;49150;49;MAINE ET LOIRE;1
49316;;Saint-Rmy-en-Mauges;SAINT-REMY-EN-MAUGES;49110;49;MAINE ET LOIRE;1
49317;;Saint-Rmy-la-Varenne;SAINT-REMY-LA-VARENNE;49250;49;MAINE ET LOIRE;1
49318;;Saint-Saturnin-sur-Loire;SAINT-SATURNIN-SUR-LOIRE;49320;49;MAINE ET LOIRE;1
49319;;Saint-Sauveur-de-Fle;SAINT-SAUVEUR-DE-FLEE;49500;49;MAINE ET LOIRE;1
49320;;Saint-Sauveur-de-Landemont;SAINT-SAUVEUR-DE-LANDEMONT;49270;49;MAINE ET LOIRE;1
49321;;Saint-Sigismond;SAINT-SIGISMOND;49123;49;MAINE ET LOIRE;1
49322;;Saint-Sulpice;SAINT-SULPICE;49320;49;MAINE ET LOIRE;1
49323;;Saint-Sylvain-d'Anjou;SAINT-SYLVAIN-D'ANJOU;49480;49;MAINE ET LOIRE;1
49324;La;Salle-et-Chapelle-Aubry;SALLE-ET-CHAPELLE-AUBRY;49110;49;MAINE ET LOIRE;1
49325;La;Salle-de-Vihiers;SALLE-DE-VIHIERS;49310;49;MAINE ET LOIRE;1
49326;;Sarrign;SARRIGNE;49800;49;MAINE ET LOIRE;1
49327;;Saulg-l'Hpital;SAULGE-L'HOPITAL;49320;49;MAINE ET LOIRE;1
49328;;Saumur;SAUMUR;49400;49;MAINE ET LOIRE;1
49329;;Savennires;SAVENNIERES;49170;49;MAINE ET LOIRE;1
49330;;Sceaux-d'Anjou;SCEAUX-D'ANJOU;49330;49;MAINE ET LOIRE;1
49331;;Segr;SEGRE;49500;49;MAINE ET LOIRE;1
49332;La;Sguinire;SEGUINIERE;49280;49;MAINE ET LOIRE;1
49333;;Seiches-sur-le-Loir;SEICHES-SUR-LE-LOIR;49140;49;MAINE ET LOIRE;1
49334;;Sermaise;SERMAISE;49140;49;MAINE ET LOIRE;1
49335;;Surdres;SOEURDRES;49330;49;MAINE ET LOIRE;1
49336;;Somloire;SOMLOIRE;49360;49;MAINE ET LOIRE;1
49337;;Soucelles;SOUCELLES;49140;49;MAINE ET LOIRE;1
49338;;Soulaines-sur-Aubance;SOULAINES-SUR-AUBANCE;49610;49;MAINE ET LOIRE;1
49339;;Soulaire-et-Bourg;SOULAIRE-ET-BOURG;49460;49;MAINE ET LOIRE;1
49341;;Souzay-Champigny;SOUZAY-CHAMPIGNY;49400;49;MAINE ET LOIRE;1
49342;;Tancoign;TANCOIGNE;49310;49;MAINE ET LOIRE;1
49343;La;Tessoualle;TESSOUALLE;49280;49;MAINE ET LOIRE;1
49344;;Thorign-d'Anjou;THORIGNE-D'ANJOU;49220;49;MAINE ET LOIRE;1
49345;;Thouarc;THOUARCE;49380;49;MAINE ET LOIRE;1
49346;Le;Thoureil;THOUREIL;49350;49;MAINE ET LOIRE;1
49347;;Tierc;TIERCE;49125;49;MAINE ET LOIRE;1
49348;;Tign;TIGNE;49540;49;MAINE ET LOIRE;1
49349;;Tillires;TILLIERES;49230;49;MAINE ET LOIRE;1
49350;;Torfou;TORFOU;49660;49;MAINE ET LOIRE;1
49351;La;Tourlandry;TOURLANDRY;49120;49;MAINE ET LOIRE;1
49352;;Toutlemonde;TOUTLEMONDE;49360;49;MAINE ET LOIRE;1
49353;;Trlaz;TRELAZE;49800;49;MAINE ET LOIRE;1
49354;Le;Tremblay;TREMBLAY;49520;49;MAINE ET LOIRE;1
49355;;Trmentines;TREMENTINES;49340;49;MAINE ET LOIRE;1
49356;;Trmont;TREMONT;49310;49;MAINE ET LOIRE;1
49358;;Turquant;TURQUANT;49730;49;MAINE ET LOIRE;1
49359;Les;Ulmes;ULMES;49700;49;MAINE ET LOIRE;1
49360;La;Varenne;VARENNE;49270;49;MAINE ET LOIRE;1
49361;;Varennes-sur-Loire;VARENNES-SUR-LOIRE;49730;49;MAINE ET LOIRE;1
49362;;Varrains;VARRAINS;49400;49;MAINE ET LOIRE;1
49363;;Vauchrtien;VAUCHRETIEN;49320;49;MAINE ET LOIRE;1
49364;;Vaudelnay;VAUDELNAY;49260;49;MAINE ET LOIRE;1
49365;Les;Verchers-sur-Layon;VERCHERS-SUR-LAYON;49700;49;MAINE ET LOIRE;1
49366;;Vergonnes;VERGONNES;49420;49;MAINE ET LOIRE;1
49367;;Vern-d'Anjou;VERN-D'ANJOU;49220;49;MAINE ET LOIRE;1
49368;;Vernantes;VERNANTES;49390;49;MAINE ET LOIRE;1
49369;;Vernoil-le-Fourrier;VERNOIL-LE-FOURRIER;49390;49;MAINE ET LOIRE;1
49370;;Verrie;VERRIE;49400;49;MAINE ET LOIRE;1
49371;;Vezins;VEZINS;49340;49;MAINE ET LOIRE;1
49372;Le;Vieil-Baug;VIEIL-BAUGE;49150;49;MAINE ET LOIRE;1
49373;;Vihiers;VIHIERS;49310;49;MAINE ET LOIRE;1
49374;;Villebernier;VILLEBERNIER;49400;49;MAINE ET LOIRE;1
49375;;Villedieu-la-Bloure;VILLEDIEU-LA-BLOUERE;49450;49;MAINE ET LOIRE;1
49376;;Villemoisan;VILLEMOISAN;49370;49;MAINE ET LOIRE;1
49377;;Villevque;VILLEVEQUE;49140;49;MAINE ET LOIRE;1
49378;;Vivy;VIVY;49680;49;MAINE ET LOIRE;1
49380;;Vaulandry;VAULANDRY;49150;49;MAINE ET LOIRE;1
49381;;Yzernay;YZERNAY;49360;49;MAINE ET LOIRE;1
50001;;Acqueville;ACQUEVILLE;50440;50;MANCHE;1
50002;;Agneaux;AGNEAUX;50180;50;MANCHE;1
50003;;Agon-Coutainville;AGON-COUTAINVILLE;50230;50;MANCHE;1
50004;;Airel;AIREL;50680;50;MANCHE;1
50005;;Amfreville;AMFREVILLE;50480;50;MANCHE;1
50006;;Amigny;AMIGNY;50620;50;MANCHE;1
50007;;Ancteville;ANCTEVILLE;50200;50;MANCHE;1
50008;;Anctoville-sur-Boscq;ANCTOVILLE-SUR-BOSCQ;50400;50;MANCHE;1
50009;;Angey;ANGEY;50530;50;MANCHE;1
50010;;Angoville-au-Plain;ANGOVILLE-AU-PLAIN;50480;50;MANCHE;1
50012;;Angoville-sur-Ay;ANGOVILLE-SUR-AY;50430;50;MANCHE;1
50013;;Anneville-en-Saire;ANNEVILLE-EN-SAIRE;50760;50;MANCHE;1
50014;;Anneville-sur-Mer;ANNEVILLE-SUR-MER;50560;50;MANCHE;1
50015;;Annoville;ANNOVILLE;50660;50;MANCHE;1
50016;;Appeville;APPEVILLE;50500;50;MANCHE;1
50018;;Argouges;ARGOUGES;50240;50;MANCHE;1
50019;;Aucey-la-Plaine;AUCEY-LA-PLAINE;50170;50;MANCHE;1
50020;;Auderville;AUDERVILLE;50440;50;MANCHE;1
50021;;Audouville-la-Hubert;AUDOUVILLE-LA-HUBERT;50480;50;MANCHE;1
50022;;Aumeville-Lestre;AUMEVILLE-LESTRE;50630;50;MANCHE;1
50023;;Auvers;AUVERS;50500;50;MANCHE;1
50024;;Auxais;AUXAIS;50500;50;MANCHE;1
50025;;Avranches;AVRANCHES;50300;50;MANCHE;1
50026;;Azeville;AZEVILLE;50310;50;MANCHE;1
50027;;Bacilly;BACILLY;50530;50;MANCHE;1
50028;La;Baleine;BALEINE;50450;50;MANCHE;1
50029;;Barenton;BARENTON;50720;50;MANCHE;1
50030;;Barfleur;BARFLEUR;50760;50;MANCHE;1
50031;;Barneville-Carteret;BARNEVILLE-CARTERET;50270;50;MANCHE;1
50032;La;Barre-de-Semilly;BARRE-DE-SEMILLY;50810;50;MANCHE;1
50033;;Baubigny;BAUBIGNY;50270;50;MANCHE;1
50034;;Baudre;BAUDRE;50000;50;MANCHE;1
50035;;Baudreville;BAUDREVILLE;50250;50;MANCHE;1
50036;;Baupte;BAUPTE;50500;50;MANCHE;1
50037;La;Bazoge;BAZOGE;50520;50;MANCHE;1
50038;;Beauchamps;BEAUCHAMPS;50320;50;MANCHE;1
50039;;Beaucoudray;BEAUCOUDRAY;50420;50;MANCHE;1
50040;;Beauficel;BEAUFICEL;50150;50;MANCHE;1
50041;;Beaumont-Hague;BEAUMONT-HAGUE;50440;50;MANCHE;1
50042;;Beauvoir;BEAUVOIR;50170;50;MANCHE;1
50043;;Bellefontaine;BELLEFONTAINE;50520;50;MANCHE;1
50044;;Belval;BELVAL;50210;50;MANCHE;1
50045;;Benotville;BENOITVILLE;50340;50;MANCHE;1
50046;;Brigny;BERIGNY;50810;50;MANCHE;1
50048;;Beslon;BESLON;50800;50;MANCHE;1
50049;;Besneville;BESNEVILLE;50390;50;MANCHE;1
50050;;Beuvrigny;BEUVRIGNY;50420;50;MANCHE;1
50051;;Beuzeville-au-Plain;BEUZEVILLE-AU-PLAIN;50480;50;MANCHE;1
50052;;Beuzeville-la-Bastille;BEUZEVILLE-LA-BASTILLE;50360;50;MANCHE;1
50054;;Biville;BIEVILLE;50160;50;MANCHE;1
50055;;Biniville;BINIVILLE;50390;50;MANCHE;1
50056;;Bion;BION;50140;50;MANCHE;1
50057;;Biville;BIVILLE;50440;50;MANCHE;1
50058;;Blainville-sur-Mer;BLAINVILLE-SUR-MER;50560;50;MANCHE;1
50059;;Blosville;BLOSVILLE;50480;50;MANCHE;1
50060;La;Bloutire;BLOUTIERE;50800;50;MANCHE;1
50061;;Boisroger;BOISROGER;50200;50;MANCHE;1
50062;;Boisyvon;BOISYVON;50800;50;MANCHE;1
50063;;Bolleville;BOLLEVILLE;50250;50;MANCHE;1
50064;La;Bonneville;BONNEVILLE;50360;50;MANCHE;1
50066;;Jullouville;JULLOUVILLE;50610;50;MANCHE;1
50069;;Bourguenolles;BOURGUENOLLES;50800;50;MANCHE;1
50070;;Boutteville;BOUTTEVILLE;50480;50;MANCHE;1
50071;;Braffais;BRAFFAIS;50870;50;MANCHE;1
50072;;Brainville;BRAINVILLE;50200;50;MANCHE;1
50073;;Branville-Hague;BRANVILLE-HAGUE;50440;50;MANCHE;1
50074;;Brcey;BRECEY;50370;50;MANCHE;1
50075;;Brectouville;BRECTOUVILLE;50160;50;MANCHE;1
50076;;Brhal;BREHAL;50290;50;MANCHE;1
50077;;Bretteville;BRETTEVILLE;50110;50;MANCHE;1
50078;;Bretteville-sur-Ay;BRETTEVILLE-SUR-AY;50430;50;MANCHE;1
50079;;Breuville;BREUVILLE;50260;50;MANCHE;1
50080;;Brvands;BREVANDS;50500;50;MANCHE;1
50081;;Brville-sur-Mer;BREVILLE-SUR-MER;50290;50;MANCHE;1
50082;;Bricquebec;BRICQUEBEC;50260;50;MANCHE;1
50083;;Bricquebosq;BRICQUEBOSQ;50340;50;MANCHE;1
50084;;Bricqueville-la-Blouette;BRICQUEVILLE-LA-BLOUETTE;50200;50;MANCHE;1
50085;;Bricqueville-sur-Mer;BRICQUEVILLE-SUR-MER;50290;50;MANCHE;1
50086;;Brillevast;BRILLEVAST;50330;50;MANCHE;1
50087;;Brix;BRIX;50700;50;MANCHE;1
50088;;Brouains;BROUAINS;50150;50;MANCHE;1
50089;;Brucheville;BRUCHEVILLE;50480;50;MANCHE;1
50090;;Buais;BUAIS;50640;50;MANCHE;1
50092;;Cambernon;CAMBERNON;50200;50;MANCHE;1
50093;;Cametours;CAMETOURS;50570;50;MANCHE;1
50094;;Camprond;CAMPROND;50210;50;MANCHE;1
50095;;Canisy;CANISY;50750;50;MANCHE;1
50096;;Canteloup;CANTELOUP;50330;50;MANCHE;1
50097;;Canville-la-Rocque;CANVILLE-LA-ROCQUE;50580;50;MANCHE;1
50098;;Carantilly;CARANTILLY;50570;50;MANCHE;1
50099;;Carentan;CARENTAN;50500;50;MANCHE;1
50100;;Carnet;CARNET;50240;50;MANCHE;1
50101;;Carneville;CARNEVILLE;50330;50;MANCHE;1
50102;;Carolles;CAROLLES;50740;50;MANCHE;1
50103;;Carquebut;CARQUEBUT;50480;50;MANCHE;1
50105;;Catteville;CATTEVILLE;50390;50;MANCHE;1
50106;;Cavigny;CAVIGNY;50620;50;MANCHE;1
50107;;Catz;CATZ;50500;50;MANCHE;1
50108;;Caux;CEAUX;50220;50;MANCHE;1
50109;;Crences;CERENCES;50510;50;MANCHE;1
50110;;Cerisy-la-Fort;CERISY-LA-FORET;50680;50;MANCHE;1
50111;;Cerisy-la-Salle;CERISY-LA-SALLE;50210;50;MANCHE;1
50112;La;Chaise-Baudouin;CHAISE-BAUDOUIN;50370;50;MANCHE;1
50114;Les;Chambres;CHAMBRES;50320;50;MANCHE;1
50115;;Champcervon;CHAMPCERVON;50320;50;MANCHE;1
50116;;Champcey;CHAMPCEY;50530;50;MANCHE;1
50117;;Champeaux;CHAMPEAUX;50530;50;MANCHE;1
50118;;Champrepus;CHAMPREPUS;50800;50;MANCHE;1
50119;Les;Champs-de-Losque;CHAMPS-DE-LOSQUE;50620;50;MANCHE;1
50120;;Chanteloup;CHANTELOUP;50510;50;MANCHE;1
50121;La;Chapelle-Ccelin;CHAPELLE-CECELIN;50800;50;MANCHE;1
50123;La;Chapelle-en-Juger;CHAPELLE-EN-JUGER;50570;50;MANCHE;1
50124;La;Chapelle-Ure;CHAPELLE-UREE;50370;50;MANCHE;1
50125;;Chasseguey;CHASSEGUEY;50520;50;MANCHE;1
50126;;Chavoy;CHAVOY;50870;50;MANCHE;1
50127;;Chef-du-Pont;CHEF-DU-PONT;50480;50;MANCHE;1
50128;Le;Chefresne;CHEFRESNE;50410;50;MANCHE;1
50129;;Cherbourg-Octeville;CHERBOURG-OCTEVILLE;50100;50;MANCHE;1
50130;;Chrenc-le-Hron;CHERENCE-LE-HERON;50800;50;MANCHE;1
50131;;Chrenc-le-Roussel;CHERENCE-LE-ROUSSEL;50520;50;MANCHE;1
50132;Les;Chris;CHERIS;50220;50;MANCHE;1
50133;;Chvreville;CHEVREVILLE;50600;50;MANCHE;1
50134;;Chevry;CHEVRY;50420;50;MANCHE;1
50135;;Clitourps;CLITOURPS;50330;50;MANCHE;1
50136;;Coigny;COIGNY;50250;50;MANCHE;1
50137;La;Colombe;COLOMBE;50800;50;MANCHE;1
50138;;Colomby;COLOMBY;50700;50;MANCHE;1
50139;;Cond-sur-Vire;CONDE-SUR-VIRE;50890;50;MANCHE;1
50140;;Contrires;CONTRIERES;50660;50;MANCHE;1
50142;;Cosqueville;COSQUEVILLE;50330;50;MANCHE;1
50143;;Coudeville-sur-Mer;COUDEVILLE-SUR-MER;50290;50;MANCHE;1
50144;;Coulouvray-Boisbentre;COULOUVRAY-BOISBENATRE;50670;50;MANCHE;1
50145;;Courcy;COURCY;50200;50;MANCHE;1
50146;;Courtils;COURTILS;50220;50;MANCHE;1
50147;;Coutances;COUTANCES;50200;50;MANCHE;1
50148;;Couvains;COUVAINS;50680;50;MANCHE;1
50149;;Couville;COUVILLE;50690;50;MANCHE;1
50150;;Crasville;CRASVILLE;50630;50;MANCHE;1
50151;;Crances;CREANCES;50710;50;MANCHE;1
50152;Les;Cresnays;CRESNAYS;50370;50;MANCHE;1
50153;;Cretteville;CRETTEVILLE;50250;50;MANCHE;1
50154;La;Croix-Avranchin;CROIX-AVRANCHIN;50240;50;MANCHE;1
50155;;Crollon;CROLLON;50220;50;MANCHE;1
50156;;Crosville-sur-Douve;CROSVILLE-SUR-DOUVE;50360;50;MANCHE;1
50158;;Cuves;CUVES;50670;50;MANCHE;1
50159;;Dangy;DANGY;50750;50;MANCHE;1
50160;;Denneville;DENNEVILLE;50580;50;MANCHE;1
50161;Le;Dzert;DEZERT;50620;50;MANCHE;1
50162;;Digosville;DIGOSVILLE;50110;50;MANCHE;1
50163;;Digulleville;DIGULLEVILLE;50440;50;MANCHE;1
50164;;Domjean;DOMJEAN;50420;50;MANCHE;1
50165;;Donville-les-Bains;DONVILLE-LES-BAINS;50350;50;MANCHE;1
50166;;Doville;DOVILLE;50250;50;MANCHE;1
50167;;Dragey-Ronthon;DRAGEY-RONTHON;50530;50;MANCHE;1
50168;;Ducey;DUCEY;50220;50;MANCHE;1
50169;;causseville;ECAUSSEVILLE;50310;50;MANCHE;1
50170;;coquenauville;ECOQUENEAUVILLE;50480;50;MANCHE;1
50171;;culleville;ECULLEVILLE;50440;50;MANCHE;1
50172;;mondeville;EMONDEVILLE;50310;50;MANCHE;1
50173;;queurdreville-Hainneville;EQUEURDREVILLE-HAINNEVILLE;50120;50;MANCHE;1
50174;;quilly;EQUILLY;50320;50;MANCHE;1
50175;;roudeville;EROUDEVILLE;50310;50;MANCHE;1
50176;L';tang-Bertrand;ETANG-BERTRAND;50260;50;MANCHE;1
50177;;tienville;ETIENVILLE;50360;50;MANCHE;1
50178;;Fermanville;FERMANVILLE;50840;50;MANCHE;1
50179;;Ferrires;FERRIERES;50640;50;MANCHE;1
50180;;Fervaches;FERVACHES;50420;50;MANCHE;1
50181;;Feugres;FEUGERES;50190;50;MANCHE;1
50182;La;Feuillie;FEUILLIE;50190;50;MANCHE;1
50183;;Fierville-les-Mines;FIERVILLE-LES-MINES;50580;50;MANCHE;1
50184;;Flamanville;FLAMANVILLE;50340;50;MANCHE;1
50185;;Fleury;FLEURY;50800;50;MANCHE;1
50186;;Flottemanville;FLOTTEMANVILLE;50700;50;MANCHE;1
50187;;Flottemanville-Hague;FLOTTEMANVILLE-HAGUE;50690;50;MANCHE;1
50188;;Folligny;FOLLIGNY;50320;50;MANCHE;1
50189;;Fontenay;FONTENAY;50140;50;MANCHE;1
50190;;Fontenay-sur-Mer;FONTENAY-SUR-MER;50310;50;MANCHE;1
50191;;Foucarville;FOUCARVILLE;50480;50;MANCHE;1
50192;;Fourneaux;FOURNEAUX;50420;50;MANCHE;1
50193;Le;Fresne-Poret;FRESNE-PORET;50850;50;MANCHE;1
50194;;Fresville;FRESVILLE;50310;50;MANCHE;1
50195;;Gathemo;GATHEMO;50150;50;MANCHE;1
50196;;Gatteville-le-Phare;GATTEVILLE-LE-PHARE;50760;50;MANCHE;1
50197;;Gavray;GAVRAY;50450;50;MANCHE;1
50198;;Geffosses;GEFFOSSES;50560;50;MANCHE;1
50199;;Gents;GENETS;50530;50;MANCHE;1
50200;;Ger;GER;50850;50;MANCHE;1
50202;;Giville;GIEVILLE;50160;50;MANCHE;1
50203;La;Glacerie;GLACERIE;50470;50;MANCHE;1
50204;;Glatigny;GLATIGNY;50250;50;MANCHE;1
50205;La;Godefroy;GODEFROY;50300;50;MANCHE;1
50206;La;Gohannire;GOHANNIERE;50300;50;MANCHE;1
50207;;Golleville;GOLLEVILLE;50390;50;MANCHE;1
50208;;Gonfreville;GONFREVILLE;50190;50;MANCHE;1
50209;;Gonneville;GONNEVILLE;50330;50;MANCHE;1
50210;;Gorges;GORGES;50190;50;MANCHE;1
50211;;Gouberville;GOUBERVILLE;50330;50;MANCHE;1
50212;;Gourbesville;GOURBESVILLE;50480;50;MANCHE;1
50213;;Gourfaleur;GOURFALEUR;50750;50;MANCHE;1
50214;;Gouvets;GOUVETS;50420;50;MANCHE;1
50215;;Gouville-sur-Mer;GOUVILLE-SUR-MER;50560;50;MANCHE;1
50216;;Graignes-Mesnil-Angot;GRAIGNES-MESNIL-ANGOT;50620;50;MANCHE;1
50217;Le;Grand-Celland;GRAND-CELLAND;50370;50;MANCHE;1
50218;;Granville;GRANVILLE;50400;50;MANCHE;1
50219;;Gratot;GRATOT;50200;50;MANCHE;1
50220;;Grville-Hague;GREVILLE-HAGUE;50440;50;MANCHE;1
50221;;Grimesnil;GRIMESNIL;50450;50;MANCHE;1
50222;;Grosville;GROSVILLE;50340;50;MANCHE;1
50223;;Guhbert;GUEHEBERT;50210;50;MANCHE;1
50224;;Guilberville;GUILBERVILLE;50160;50;MANCHE;1
50225;Le;Guislain;GUISLAIN;50410;50;MANCHE;1
50227;Le;Ham;HAM;50310;50;MANCHE;1
50228;;Hambye;HAMBYE;50450;50;MANCHE;1
50229;;Hamelin;HAMELIN;50730;50;MANCHE;1
50230;;Hardinvast;HARDINVAST;50690;50;MANCHE;1
50231;;Hauteville-sur-Mer;HAUTEVILLE-SUR-MER;50590;50;MANCHE;1
50232;;Hauteville-la-Guichard;HAUTEVILLE-LA-GUICHARD;50570;50;MANCHE;1
50233;;Hautteville-Bocage;HAUTTEVILLE-BOCAGE;50390;50;MANCHE;1
50234;La;Haye-Bellefond;HAYE-BELLEFOND;50410;50;MANCHE;1
50235;La;Haye-d'Ectot;HAYE-D'ECTOT;50270;50;MANCHE;1
50236;La;Haye-du-Puits;HAYE-DU-PUITS;50250;50;MANCHE;1
50237;La;Haye-Pesnel;HAYE-PESNEL;50320;50;MANCHE;1
50238;;Hauville;HEAUVILLE;50340;50;MANCHE;1
50239;;Hbcrevon;HEBECREVON;50180;50;MANCHE;1
50240;;Helleville;HELLEVILLE;50340;50;MANCHE;1
50241;;Hmevez;HEMEVEZ;50700;50;MANCHE;1
50242;;Herqueville;HERQUEVILLE;50440;50;MANCHE;1
50243;;Heugueville-sur-Sienne;HEUGUEVILLE-SUR-SIENNE;50200;50;MANCHE;1
50244;;Hrenguerville;HERENGUERVILLE;50660;50;MANCHE;1
50245;;Heuss;HEUSSE;50640;50;MANCHE;1
50246;;Hiesville;HIESVILLE;50480;50;MANCHE;1
50247;;Hocquigny;HOCQUIGNY;50320;50;MANCHE;1
50248;Le;Hommet-d'Arthenay;HOMMET-D'ARTHENAY;50620;50;MANCHE;1
50249;;Houesville;HOUESVILLE;50480;50;MANCHE;1
50250;;Houtteville;HOUTTEVILLE;50250;50;MANCHE;1
50251;;Huberville;HUBERVILLE;50700;50;MANCHE;1
50252;;Hudimesnil;HUDIMESNIL;50510;50;MANCHE;1
50253;;Huisnes-sur-Mer;HUISNES-SUR-MER;50170;50;MANCHE;1
50254;;Husson;HUSSON;50640;50;MANCHE;1
50255;;Hyenville;HYENVILLE;50660;50;MANCHE;1
50256;;Isigny-le-Buat;ISIGNY-LE-BUAT;50540;50;MANCHE;1
50257;;Jobourg;JOBOURG;50440;50;MANCHE;1
50258;;Joganville;JOGANVILLE;50310;50;MANCHE;1
50259;;Juilley;JUILLEY;50220;50;MANCHE;1
50260;;Juvigny-le-Tertre;JUVIGNY-LE-TERTRE;50520;50;MANCHE;1
50261;;Lamberville;LAMBERVILLE;50160;50;MANCHE;1
50262;La;Lande-d'Airou;LANDE-D'AIROU;50800;50;MANCHE;1
50263;;Lapenty;LAPENTY;50600;50;MANCHE;1
50265;;Laulne;LAULNE;50430;50;MANCHE;1
50266;;Lengronne;LENGRONNE;50510;50;MANCHE;1
50267;;Lessay;LESSAY;50430;50;MANCHE;1
50268;;Lestre;LESTRE;50310;50;MANCHE;1
50269;;Liesville-sur-Douve;LIESVILLE-SUR-DOUVE;50480;50;MANCHE;1
50270;;Lieusaint;LIEUSAINT;50700;50;MANCHE;1
50271;;Lingeard;LINGEARD;50670;50;MANCHE;1
50272;;Lingreville;LINGREVILLE;50660;50;MANCHE;1
50273;;Lithaire;LITHAIRE;50250;50;MANCHE;1
50274;Les;Loges-Marchis;LOGES-MARCHIS;50600;50;MANCHE;1
50275;Les;Loges-sur-Brcey;LOGES-SUR-BRECEY;50370;50;MANCHE;1
50276;;Lolif;LOLIF;50530;50;MANCHE;1
50277;;Longueville;LONGUEVILLE;50290;50;MANCHE;1
50278;Le;Loreur;LOREUR;50510;50;MANCHE;1
50279;Le;Lorey;LOREY;50570;50;MANCHE;1
50280;;Lozon;LOZON;50570;50;MANCHE;1
50281;La;Lucerne-d'Outremer;LUCERNE-D'OUTREMER;50320;50;MANCHE;1
50282;Le;Luot;LUOT;50870;50;MANCHE;1
50283;La;Luzerne;LUZERNE;50680;50;MANCHE;1
50284;;Macey;MACEY;50170;50;MANCHE;1
50285;;Magneville;MAGNEVILLE;50260;50;MANCHE;1
50287;La;Mancellire-sur-Vire;MANCELLIERE-SUR-VIRE;50750;50;MANCHE;1
50288;;Marcey-les-Grves;MARCEY-LES-GREVES;50300;50;MANCHE;1
50289;;Marchsieux;MARCHESIEUX;50190;50;MANCHE;1
50290;;Marcilly;MARCILLY;50220;50;MANCHE;1
50291;;Margueray;MARGUERAY;50410;50;MANCHE;1
50292;;Marigny;MARIGNY;50570;50;MANCHE;1
50293;;Martigny;MARTIGNY;50600;50;MANCHE;1
50294;;Martinvast;MARTINVAST;50690;50;MANCHE;1
50295;;Maupertuis;MAUPERTUIS;50410;50;MANCHE;1
50296;;Maupertus-sur-Mer;MAUPERTUS-SUR-MER;50330;50;MANCHE;1
50297;La;Meauffe;MEAUFFE;50880;50;MANCHE;1
50298;;Mautis;MEAUTIS;50500;50;MANCHE;1
50299;Le;Mesnil;MESNIL;50580;50;MANCHE;1
50300;Le;Mesnil-Adele;MESNIL-ADELEE;50520;50;MANCHE;1
50301;Le;Mesnil-Amand;MESNIL-AMAND;50450;50;MANCHE;1
50302;Le;Mesnil-Amey;MESNIL-AMEY;50570;50;MANCHE;1
50304;Le;Mesnil-Aubert;MESNIL-AUBERT;50510;50;MANCHE;1
50305;Le;Mesnil-au-Val;MESNIL-AU-VAL;50110;50;MANCHE;1
50308;Le;Mesnilbus;MESNILBUS;50490;50;MANCHE;1
50310;Le;Mesnil-Eury;MESNIL-EURY;50570;50;MANCHE;1
50311;Le;Mesnil-Garnier;MESNIL-GARNIER;50450;50;MANCHE;1
50312;Le;Mesnil-Gilbert;MESNIL-GILBERT;50670;50;MANCHE;1
50313;Le;Mesnil-Herman;MESNIL-HERMAN;50750;50;MANCHE;1
50315;Le;Mesnillard;MESNILLARD;50600;50;MANCHE;1
50316;Le;Mesnil-Opac;MESNIL-OPAC;50860;50;MANCHE;1
50317;Le;Mesnil-Ozenne;MESNIL-OZENNE;50220;50;MANCHE;1
50318;Le;Mesnil-Rainfray;MESNIL-RAINFRAY;50520;50;MANCHE;1
50319;Le;Mesnil-Raoult;MESNIL-RAOULT;50420;50;MANCHE;1
50320;Le;Mesnil-Rogues;MESNIL-ROGUES;50450;50;MANCHE;1
50321;Le;Mesnil-Rouxelin;MESNIL-ROUXELIN;50000;50;MANCHE;1
50323;Le;Mesnil-Tve;MESNIL-TOVE;50520;50;MANCHE;1
50324;Le;Mesnil-Vneron;MESNIL-VENERON;50620;50;MANCHE;1
50325;Le;Mesnil-Vigot;MESNIL-VIGOT;50570;50;MANCHE;1
50326;Le;Mesnil-Villeman;MESNIL-VILLEMAN;50450;50;MANCHE;1
50327;La;Meurdraquire;MEURDRAQUIERE;50510;50;MANCHE;1
50328;;Millires;MILLIERES;50190;50;MANCHE;1
50329;;Milly;MILLY;50600;50;MANCHE;1
50330;;Mobecq;MOBECQ;50250;50;MANCHE;1
50332;Les;Moitiers-d'Allonne;MOITIERS-D'ALLONNE;50270;50;MANCHE;1
50333;Les;Moitiers-en-Bauptois;MOITIERS-EN-BAUPTOIS;50360;50;MANCHE;1
50334;;Montabot;MONTABOT;50410;50;MANCHE;1
50335;;Montaigu-la-Brisette;MONTAIGU-LA-BRISETTE;50700;50;MANCHE;1
50336;;Montaigu-les-Bois;MONTAIGU-LES-BOIS;50450;50;MANCHE;1
50337;;Montanel;MONTANEL;50240;50;MANCHE;1
50338;;Montbray;MONTBRAY;50410;50;MANCHE;1
50339;;Montchaton;MONTCHATON;50660;50;MANCHE;1
50340;;Montcuit;MONTCUIT;50490;50;MANCHE;1
50341;;Montebourg;MONTEBOURG;50310;50;MANCHE;1
50342;;Montfarville;MONTFARVILLE;50760;50;MANCHE;1
50343;;Montgardon;MONTGARDON;50250;50;MANCHE;1
50345;;Monthuchon;MONTHUCHON;50200;50;MANCHE;1
50347;;Montjoie-Saint-Martin;MONTJOIE-SAINT-MARTIN;50240;50;MANCHE;1
50348;;Montmartin-en-Graignes;MONTMARTIN-EN-GRAIGNES;50620;50;MANCHE;1
50349;;Montmartin-sur-Mer;MONTMARTIN-SUR-MER;50590;50;MANCHE;1
50350;;Montpinchon;MONTPINCHON;50210;50;MANCHE;1
50351;;Montrabot;MONTRABOT;50810;50;MANCHE;1
50352;;Montreuil-sur-Lozon;MONTREUIL-SUR-LOZON;50570;50;MANCHE;1
50353;Le;Mont-Saint-Michel;MONT-SAINT-MICHEL;50170;50;MANCHE;1
50354;;Montsurvent;MONTSURVENT;50200;50;MANCHE;1
50355;;Montviron;MONTVIRON;50530;50;MANCHE;1
50356;;Moon-sur-Elle;MOON-SUR-ELLE;50680;50;MANCHE;1
50357;;Morigny;MORIGNY;50410;50;MANCHE;1
50358;;Morsalines;MORSALINES;50630;50;MANCHE;1
50359;;Mortain;MORTAIN;50140;50;MANCHE;1
50360;;Morville;MORVILLE;50700;50;MANCHE;1
50361;La;Mouche;MOUCHE;50320;50;MANCHE;1
50362;;Moulines;MOULINES;50600;50;MANCHE;1
50363;;Moyon;MOYON;50860;50;MANCHE;1
50364;;Muneville-le-Bingard;MUNEVILLE-LE-BINGARD;50490;50;MANCHE;1
50365;;Muneville-sur-Mer;MUNEVILLE-SUR-MER;50290;50;MANCHE;1
50368;;Nay;NAY;50190;50;MANCHE;1
50369;;Ngreville;NEGREVILLE;50260;50;MANCHE;1
50370;;Nhou;NEHOU;50390;50;MANCHE;1
50371;Le;Neufbourg;NEUFBOURG;50140;50;MANCHE;1
50372;;Neufmesnil;NEUFMESNIL;50250;50;MANCHE;1
50373;;Neuville-au-Plain;NEUVILLE-AU-PLAIN;50480;50;MANCHE;1
50374;;Neuville-en-Beaumont;NEUVILLE-EN-BEAUMONT;50250;50;MANCHE;1
50375;;Nville-sur-Mer;NEVILLE-SUR-MER;50330;50;MANCHE;1
50376;;Nicorps;NICORPS;50200;50;MANCHE;1
50378;;Notre-Dame-de-Cenilly;NOTRE-DAME-DE-CENILLY;50210;50;MANCHE;1
50379;;Notre-Dame-de-Livoye;NOTRE-DAME-DE-LIVOYE;50370;50;MANCHE;1
50380;;Notre-Dame-d'Elle;NOTRE-DAME-D'ELLE;50810;50;MANCHE;1
50381;;Notre-Dame-du-Touchet;NOTRE-DAME-DU-TOUCHET;50140;50;MANCHE;1
50382;;Nouainville;NOUAINVILLE;50690;50;MANCHE;1
50384;;Octeville-l'Avenel;OCTEVILLE-L'AVENEL;50630;50;MANCHE;1
50385;;Omonville-la-Petite;OMONVILLE-LA-PETITE;50440;50;MANCHE;1
50386;;Omonville-la-Rogue;OMONVILLE-LA-ROGUE;50440;50;MANCHE;1
50387;;Orglandes;ORGLANDES;50390;50;MANCHE;1
50388;;Orval;ORVAL;50660;50;MANCHE;1
50389;;Ouville;OUVILLE;50210;50;MANCHE;1
50390;;Ozeville;OZEVILLE;50310;50;MANCHE;1
50391;;Parigny;PARIGNY;50600;50;MANCHE;1
50393;;Percy;PERCY;50410;50;MANCHE;1
50394;;Priers;PERIERS;50190;50;MANCHE;1
50395;La;Pernelle;PERNELLE;50630;50;MANCHE;1
50396;Les;Perques;PERQUES;50260;50;MANCHE;1
50397;;Perriers-en-Beauficel;PERRIERS-EN-BEAUFICEL;50150;50;MANCHE;1
50398;Le;Perron;PERRON;50160;50;MANCHE;1
50399;Le;Petit-Celland;PETIT-CELLAND;50370;50;MANCHE;1
50400;;Picauville;PICAUVILLE;50360;50;MANCHE;1
50401;;Pierreville;PIERREVILLE;50340;50;MANCHE;1
50402;Les;Pieux;PIEUX;50340;50;MANCHE;1
50403;;Pirou;PIROU;50770;50;MANCHE;1
50404;;Placy-Montaigu;PLACY-MONTAIGU;50160;50;MANCHE;1
50405;Le;Plessis-Lastelle;PLESSIS-LASTELLE;50250;50;MANCHE;1
50406;;Plomb;PLOMB;50870;50;MANCHE;1
50407;;Poilley;POILLEY;50220;50;MANCHE;1
50408;;Pontaubault;PONTAUBAULT;50220;50;MANCHE;1
50409;;Pont-Hbert;PONT-HEBERT;50880;50;MANCHE;1
50410;;Pontorson;PONTORSON;50170;50;MANCHE;1
50411;;Ponts;PONTS;50300;50;MANCHE;1
50412;;Portbail;PORTBAIL;50580;50;MANCHE;1
50413;;Prcey;PRECEY;50220;50;MANCHE;1
50414;;Prcorbin;PRECORBIN;50810;50;MANCHE;1
50415;;Prtot-Sainte-Suzanne;PRETOT-SAINTE-SUZANNE;50250;50;MANCHE;1
50416;;Querqueville;QUERQUEVILLE;50460;50;MANCHE;1
50417;;Quettehou;QUETTEHOU;50630;50;MANCHE;1
50418;;Quettetot;QUETTETOT;50260;50;MANCHE;1
50419;;Quettreville-sur-Sienne;QUETTREVILLE-SUR-SIENNE;50660;50;MANCHE;1
50420;;Quibou;QUIBOU;50750;50;MANCHE;1
50421;;Quinville;QUINEVILLE;50310;50;MANCHE;1
50422;;Raids;RAIDS;50500;50;MANCHE;1
50423;;Rampan;RAMPAN;50000;50;MANCHE;1
50425;;Rauville-la-Bigot;RAUVILLE-LA-BIGOT;50260;50;MANCHE;1
50426;;Rauville-la-Place;RAUVILLE-LA-PLACE;50390;50;MANCHE;1
50427;;Ravenoville;RAVENOVILLE;50480;50;MANCHE;1
50428;;Reffuveille;REFFUVEILLE;50520;50;MANCHE;1
50429;;Regnville-sur-Mer;REGNEVILLE-SUR-MER;50590;50;MANCHE;1
50430;;Reigneville-Bocage;REIGNEVILLE-BOCAGE;50390;50;MANCHE;1
50431;;Remilly-sur-Lozon;REMILLY-SUR-LOZON;50570;50;MANCHE;1
50432;;Rthoville;RETHOVILLE;50330;50;MANCHE;1
50433;;Rville;REVILLE;50760;50;MANCHE;1
50434;La;Rochelle-Normande;ROCHELLE-NORMANDE;50530;50;MANCHE;1
50435;;Rocheville;ROCHEVILLE;50260;50;MANCHE;1
50436;;Romagny;ROMAGNY;50140;50;MANCHE;1
50437;;Roncey;RONCEY;50210;50;MANCHE;1
50438;La;Ronde-Haye;RONDE-HAYE;50490;50;MANCHE;1
50440;;Rouffigny;ROUFFIGNY;50800;50;MANCHE;1
50441;;Rouxeville;ROUXEVILLE;50810;50;MANCHE;1
50442;Le;Rozel;ROZEL;50340;50;MANCHE;1
50443;;Sacey;SACEY;50170;50;MANCHE;1
50444;;Saint-Amand;SAINT-AMAND;50160;50;MANCHE;1
50445;;Saint-Andr-de-Bohon;SAINT-ANDRE-DE-BOHON;50500;50;MANCHE;1
50446;;Saint-Andr-de-l'pine;SAINT-ANDRE-DE-L'EPINE;50680;50;MANCHE;1
50447;;Saint-Aubin-des-Praux;SAINT-AUBIN-DES-PREAUX;50380;50;MANCHE;1
50448;;Saint-Aubin-de-Terregatte;SAINT-AUBIN-DE-TERREGATTE;50111;50;MANCHE;1
50449;;Saint-Aubin-du-Perron;SAINT-AUBIN-DU-PERRON;50490;50;MANCHE;1
50450;;Saint-Barthlemy;SAINT-BARTHELEMY;50140;50;MANCHE;1
50451;;Saint-Brice;SAINT-BRICE;50300;50;MANCHE;1
50452;;Saint-Brice-de-Landelles;SAINT-BRICE-DE-LANDELLES;50730;50;MANCHE;1
50453;;Sainte-Ccile;SAINTE-CECILE;50800;50;MANCHE;1
50454;;Saint-Christophe-du-Foc;SAINT-CHRISTOPHE-DU-FOC;50340;50;MANCHE;1
50455;;Saint-Clair-sur-l'Elle;SAINT-CLAIR-SUR-L'ELLE;50680;50;MANCHE;1
50456;;Saint-Clment-Rancoudray;SAINT-CLEMENT-RANCOUDRAY;50140;50;MANCHE;1
50457;;Sainte-Colombe;SAINTE-COLOMBE;50390;50;MANCHE;1
50458;;Saint-Cme-du-Mont;SAINT-COME-DU-MONT;50500;50;MANCHE;1
50460;;Sainte-Croix-Hague;SAINTE-CROIX-HAGUE;50440;50;MANCHE;1
50461;;Saint-Cyr;SAINT-CYR;50310;50;MANCHE;1
50462;;Saint-Cyr-du-Bailleul;SAINT-CYR-DU-BAILLEUL;50720;50;MANCHE;1
50463;;Saint-Denis-le-Gast;SAINT-DENIS-LE-GAST;50450;50;MANCHE;1
50464;;Saint-Denis-le-Vtu;SAINT-DENIS-LE-VETU;50210;50;MANCHE;1
50465;;Saint-bremond-de-Bonfoss;SAINT-EBREMOND-DE-BONFOSSE;50750;50;MANCHE;1
50467;;Saint-Floxel;SAINT-FLOXEL;50310;50;MANCHE;1
50468;;Saint-Fromond;SAINT-FROMOND;50620;50;MANCHE;1
50469;;Sainte-Genevive;SAINTE-GENEVIEVE;50760;50;MANCHE;1
50470;;Saint-Georges-de-Bohon;SAINT-GEORGES-DE-BOHON;50500;50;MANCHE;1
50471;;Saint-Georges-de-la-Rivire;SAINT-GEORGES-DE-LA-RIVIERE;50270;50;MANCHE;1
50472;;Saint-Georges-de-Livoye;SAINT-GEORGES-DE-LIVOYE;50370;50;MANCHE;1
50473;;Saint-Georges-d'Elle;SAINT-GEORGES-D'ELLE;50680;50;MANCHE;1
50474;;Saint-Georges-de-Rouelley;SAINT-GEORGES-DE-ROUELLEY;50720;50;MANCHE;1
50475;;Saint-Georges-Montcocq;SAINT-GEORGES-MONTCOCQ;50000;50;MANCHE;1
50476;;Saint-Germain-d'Elle;SAINT-GERMAIN-D'ELLE;50810;50;MANCHE;1
50477;;Saint-Germain-des-Vaux;SAINT-GERMAIN-DES-VAUX;50440;50;MANCHE;1
50478;;Saint-Germain-de-Tournebut;SAINT-GERMAIN-DE-TOURNEBUT;50700;50;MANCHE;1
50479;;Saint-Germain-de-Varreville;SAINT-GERMAIN-DE-VARREVILLE;50480;50;MANCHE;1
50480;;Saint-Germain-le-Gaillard;SAINT-GERMAIN-LE-GAILLARD;50340;50;MANCHE;1
50481;;Saint-Germain-sur-Ay;SAINT-GERMAIN-SUR-AY;50430;50;MANCHE;1
50482;;Saint-Germain-sur-Sves;SAINT-GERMAIN-SUR-SEVES;50190;50;MANCHE;1
50483;;Saint-Gilles;SAINT-GILLES;50180;50;MANCHE;1
50484;;Saint-Hilaire-du-Harcout;SAINT-HILAIRE-DU-HARCOUET;50600;50;MANCHE;1
50485;;Saint-Hilaire-Petitville;SAINT-HILAIRE-PETITVILLE;50500;50;MANCHE;1
50486;;Saint-Jacques-de-Nhou;SAINT-JACQUES-DE-NEHOU;50390;50;MANCHE;1
50487;;Saint-James;SAINT-JAMES;50240;50;MANCHE;1
50488;;Saint-Jean-de-Daye;SAINT-JEAN-DE-DAYE;50620;50;MANCHE;1
50489;;Saint-Jean-de-la-Haize;SAINT-JEAN-DE-LA-HAIZE;50300;50;MANCHE;1
50490;;Saint-Jean-de-la-Rivire;SAINT-JEAN-DE-LA-RIVIERE;50270;50;MANCHE;1
50491;;Saint-Jean-de-Savigny;SAINT-JEAN-DE-SAVIGNY;50680;50;MANCHE;1
50492;;Saint-Jean-des-Baisants;SAINT-JEAN-DES-BAISANTS;50810;50;MANCHE;1
50493;;Saint-Jean-des-Champs;SAINT-JEAN-DES-CHAMPS;50320;50;MANCHE;1
50494;;Saint-Jean-du-Corail;SAINT-JEAN-DU-CORAIL;50140;50;MANCHE;1
50495;;Saint-Jean-du-Corail-des-Bois;SAINT-JEAN-DU-CORAIL-DES-BOIS;50370;50;MANCHE;1
50496;;Saint-Jean-le-Thomas;SAINT-JEAN-LE-THOMAS;50530;50;MANCHE;1
50497;;Saint-Jores;SAINT-JORES;50250;50;MANCHE;1
50498;;Saint-Joseph;SAINT-JOSEPH;50700;50;MANCHE;1
50499;;Saint-Laurent-de-Cuves;SAINT-LAURENT-DE-CUVES;50670;50;MANCHE;1
50500;;Saint-Laurent-de-Terregatte;SAINT-LAURENT-DE-TERREGATTE;50111;50;MANCHE;1
50502;;Saint-L;SAINT-LO;50000;50;MANCHE;1
50503;;Saint-L-d'Ourville;SAINT-LO-D'OURVILLE;50580;50;MANCHE;1
50504;;Saint-Louet-sur-Vire;SAINT-LOUET-SUR-VIRE;50420;50;MANCHE;1
50505;;Saint-Loup;SAINT-LOUP;50300;50;MANCHE;1
50506;;Saint-Malo-de-la-Lande;SAINT-MALO-DE-LA-LANDE;50200;50;MANCHE;1
50507;;Saint-Marcouf;SAINT-MARCOUF;50310;50;MANCHE;1
50508;;Sainte-Marie-du-Bois;SAINTE-MARIE-DU-BOIS;50640;50;MANCHE;1
50509;;Sainte-Marie-du-Mont;SAINTE-MARIE-DU-MONT;50480;50;MANCHE;1
50510;;Saint-Martin-d'Aubigny;SAINT-MARTIN-D'AUBIGNY;50190;50;MANCHE;1
50511;;Saint-Martin-d'Audouville;SAINT-MARTIN-D'AUDOUVILLE;50310;50;MANCHE;1
50512;;Saint-Martin-de-Bonfoss;SAINT-MARTIN-DE-BONFOSSE;50750;50;MANCHE;1
50513;;Saint-Martin-de-Cenilly;SAINT-MARTIN-DE-CENILLY;50210;50;MANCHE;1
50514;;Chaulieu;CHAULIEU;50150;50;MANCHE;1
50515;;Saint-Martin-de-Landelles;SAINT-MARTIN-DE-LANDELLES;50730;50;MANCHE;1
50516;;Saint-Martin-des-Champs;SAINT-MARTIN-DES-CHAMPS;50300;50;MANCHE;1
50517;;Saint-Martin-de-Varreville;SAINT-MARTIN-DE-VARREVILLE;50480;50;MANCHE;1
50518;;Saint-Martin-le-Bouillant;SAINT-MARTIN-LE-BOUILLANT;50800;50;MANCHE;1
50519;;Saint-Martin-le-Grard;SAINT-MARTIN-LE-GREARD;50690;50;MANCHE;1
50520;;Saint-Martin-le-Hbert;SAINT-MARTIN-LE-HEBERT;;50;;
50521;;Saint-Maur-des-Bois;SAINT-MAUR-DES-BOIS;50800;50;MANCHE;1
50522;;Saint-Maurice-en-Cotentin;SAINT-MAURICE-EN-COTENTIN;50270;50;MANCHE;1
50523;;Sainte-Mre-glise;SAINTE-MERE-EGLISE;50480;50;MANCHE;1
50524;;Saint-Michel-de-la-Pierre;SAINT-MICHEL-DE-LA-PIERRE;50490;50;MANCHE;1
50525;;Saint-Michel-de-Montjoie;SAINT-MICHEL-DE-MONTJOIE;50670;50;MANCHE;1
50528;;Saint-Nicolas-de-Pierrepont;SAINT-NICOLAS-DE-PIERREPONT;50250;50;MANCHE;1
50529;;Saint-Nicolas-des-Bois;SAINT-NICOLAS-DES-BOIS;50370;50;MANCHE;1
50531;;Saint-Ovin;SAINT-OVIN;50300;50;MANCHE;1
50532;;Saint-Pair-sur-Mer;SAINT-PAIR-SUR-MER;50380;50;MANCHE;1
50533;;Saint-Patrice-de-Claids;SAINT-PATRICE-DE-CLAIDS;50190;50;MANCHE;1
50534;;Saint-Pellerin;SAINT-PELLERIN;50500;50;MANCHE;1
50535;;Sainte-Pience;SAINTE-PIENCE;50870;50;MANCHE;1
50536;;Saint-Pierre-d'Arthglise;SAINT-PIERRE-D'ARTHEGLISE;50270;50;MANCHE;1
50537;;Saint-Pierre-de-Coutances;SAINT-PIERRE-DE-COUTANCES;50200;50;MANCHE;1
50538;;Saint-Pierre-de-Semilly;SAINT-PIERRE-DE-SEMILLY;50810;50;MANCHE;1
50539;;Saint-Pierre-glise;SAINT-PIERRE-EGLISE;50330;50;MANCHE;1
50540;;Saint-Pierre-Langers;SAINT-PIERRE-LANGERS;50530;50;MANCHE;1
50541;;Saint-Planchers;SAINT-PLANCHERS;50400;50;MANCHE;1
50542;;Saint-Pois;SAINT-POIS;50670;50;MANCHE;1
50543;;Saint-Quentin-sur-le-Homme;SAINT-QUENTIN-SUR-LE-HOMME;50220;50;MANCHE;1
50544;;Saint-Rmy-des-Landes;SAINT-REMY-DES-LANDES;50580;50;MANCHE;1
50545;;Saint-Romphaire;SAINT-ROMPHAIRE;50750;50;MANCHE;1
50546;;Saint-Samson-de-Bonfoss;SAINT-SAMSON-DE-BONFOSSE;50750;50;MANCHE;1
50548;;Saint-Sauveur-de-Pierrepont;SAINT-SAUVEUR-DE-PIERREPONT;50250;50;MANCHE;1
50549;;Saint-Sauveur-la-Pommeraye;SAINT-SAUVEUR-LA-POMMERAYE;50510;50;MANCHE;1
50550;;Saint-Sauveur-Lendelin;SAINT-SAUVEUR-LENDELIN;50490;50;MANCHE;1
50551;;Saint-Sauveur-le-Vicomte;SAINT-SAUVEUR-LE-VICOMTE;50390;50;MANCHE;1
50552;;Saint-Sbastien-de-Raids;SAINT-SEBASTIEN-DE-RAIDS;50190;50;MANCHE;1
50553;;Saint-Senier-de-Beuvron;SAINT-SENIER-DE-BEUVRON;50240;50;MANCHE;1
50554;;Saint-Senier-sous-Avranches;SAINT-SENIER-SOUS-AVRANCHES;50300;50;MANCHE;1
50556;;Sainte-Suzanne-sur-Vire;SAINTE-SUZANNE-SUR-VIRE;50750;50;MANCHE;1
50557;;Saint-Symphorien-des-Monts;SAINT-SYMPHORIEN-DES-MONTS;50640;50;MANCHE;1
50558;;Saint-Symphorien-le-Valois;SAINT-SYMPHORIEN-LE-VALOIS;50250;50;MANCHE;1
50562;;Saint-Vaast-la-Hougue;SAINT-VAAST-LA-HOUGUE;50550;50;MANCHE;1
50563;;Saint-Vigor-des-Monts;SAINT-VIGOR-DES-MONTS;50420;50;MANCHE;1
50564;;Sainteny;SAINTENY;50500;50;MANCHE;1
50565;;Sartilly;SARTILLY;50530;50;MANCHE;1
50567;;Saussemesnil;SAUSSEMESNIL;50700;50;MANCHE;1
50568;;Saussey;SAUSSEY;50200;50;MANCHE;1
50569;;Savigny;SAVIGNY;50210;50;MANCHE;1
50570;;Savigny-le-Vieux;SAVIGNY-LE-VIEUX;50640;50;MANCHE;1
50571;;Sbeville;SEBEVILLE;50480;50;MANCHE;1
50572;;Snoville;SENOVILLE;50270;50;MANCHE;1
50573;;Servigny;SERVIGNY;50200;50;MANCHE;1
50574;;Servon;SERVON;50170;50;MANCHE;1
50575;;Sideville;SIDEVILLE;50690;50;MANCHE;1
50576;;Siouville-Hague;SIOUVILLE-HAGUE;50340;50;MANCHE;1
50577;;Sortosville-en-Beaumont;SORTOSVILLE-EN-BEAUMONT;50270;50;MANCHE;1
50578;;Sortosville;SORTOSVILLE;50310;50;MANCHE;1
50579;;Sottevast;SOTTEVAST;50260;50;MANCHE;1
50580;;Sotteville;SOTTEVILLE;50340;50;MANCHE;1
50581;;Soulles;SOULLES;50750;50;MANCHE;1
50582;;Sourdeval;SOURDEVAL;50150;50;MANCHE;1
50583;;Sourdeval-les-Bois;SOURDEVAL-LES-BOIS;50450;50;MANCHE;1
50584;;Subligny;SUBLIGNY;50870;50;MANCHE;1
50585;;Surtainville;SURTAINVILLE;50270;50;MANCHE;1
50586;;Surville;SURVILLE;50250;50;MANCHE;1
50587;;Taillepied;TAILLEPIED;50390;50;MANCHE;1
50588;;Tamerville;TAMERVILLE;50700;50;MANCHE;1
50589;;Tanis;TANIS;50170;50;MANCHE;1
50590;Le;Tanu;TANU;50320;50;MANCHE;1
50591;Le;Teilleul;TEILLEUL;50640;50;MANCHE;1
50592;;Tessy-sur-Vire;TESSY-SUR-VIRE;50420;50;MANCHE;1
50593;;Teurthville-Bocage;TEURTHEVILLE-BOCAGE;50630;50;MANCHE;1
50594;;Teurthville-Hague;TEURTHEVILLE-HAGUE;50690;50;MANCHE;1
50595;Le;Theil;THEIL;50330;50;MANCHE;1
50596;;Thville;THEVILLE;50330;50;MANCHE;1
50597;;Tirepied;TIREPIED;50870;50;MANCHE;1
50598;;Tocqueville;TOCQUEVILLE;50330;50;MANCHE;1
50599;;Tollevast;TOLLEVAST;50470;50;MANCHE;1
50600;;Tonneville;TONNEVILLE;50460;50;MANCHE;1
50601;;Torigni-sur-Vire;TORIGNI-SUR-VIRE;50160;50;MANCHE;1
50602;;Tourlaville;TOURLAVILLE;50110;50;MANCHE;1
50603;;Tourville-sur-Sienne;TOURVILLE-SUR-SIENNE;50200;50;MANCHE;1
50604;;Trauville;TREAUVILLE;50340;50;MANCHE;1
50605;;Trelly;TRELLY;50660;50;MANCHE;1
50606;;Tribehou;TRIBEHOU;50620;50;MANCHE;1
50607;La;Trinit;TRINITE;50800;50;MANCHE;1
50608;;Troisgots;TROISGOTS;50420;50;MANCHE;1
50609;;Turqueville;TURQUEVILLE;50480;50;MANCHE;1
50610;;Urville;URVILLE;50700;50;MANCHE;1
50611;;Urville-Nacqueville;URVILLE-NACQUEVILLE;50460;50;MANCHE;1
50612;;Vains;VAINS;50300;50;MANCHE;1
50613;;Valcanville;VALCANVILLE;50760;50;MANCHE;1
50614;Le;Valdcie;VALDECIE;50260;50;MANCHE;1
50615;;Valognes;VALOGNES;50700;50;MANCHE;1
50616;Le;Val-Saint-Pre;VAL-SAINT-PERE;50300;50;MANCHE;1
50617;;Varenguebec;VARENGUEBEC;50250;50;MANCHE;1
50618;;Varouville;VAROUVILLE;50330;50;MANCHE;1
50619;Le;Vast;VAST;50630;50;MANCHE;1
50620;;Vasteville;VASTEVILLE;50440;50;MANCHE;1
50621;;Vaudreville;VAUDREVILLE;50310;50;MANCHE;1
50622;;Vaudrimesnil;VAUDRIMESNIL;50490;50;MANCHE;1
50623;;Vauville;VAUVILLE;50440;50;MANCHE;1
50624;La;Vendele;VENDELEE;50200;50;MANCHE;1
50625;;Vengeons;VENGEONS;50150;50;MANCHE;1
50626;;Ver;VER;50450;50;MANCHE;1
50627;;Vergoncey;VERGONCEY;50240;50;MANCHE;1
50628;;Vernix;VERNIX;50370;50;MANCHE;1
50629;;Vesly;VESLY;50430;50;MANCHE;1
50630;;Vessey;VESSEY;50170;50;MANCHE;1
50631;Les;Veys;VEYS;50500;50;MANCHE;1
50633;Le;Vicel;VICEL;50760;50;MANCHE;1
50634;;Videcosville;VIDECOSVILLE;50630;50;MANCHE;1
50635;;Vidouville;VIDOUVILLE;50810;50;MANCHE;1
50636;;Vierville;VIERVILLE;50480;50;MANCHE;1
50637;;Villebaudon;VILLEBAUDON;50410;50;MANCHE;1
50638;;Villechien;VILLECHIEN;50140;50;MANCHE;1
50639;;Villedieu-les-Poles;VILLEDIEU-LES-POELES;50800;50;MANCHE;1
50640;;Villiers-le-Pr;VILLIERS-LE-PRE;50240;50;MANCHE;1
50641;;Villiers-Fossard;VILLIERS-FOSSARD;50680;50;MANCHE;1
50642;;Vindefontaine;VINDEFONTAINE;50250;50;MANCHE;1
50643;;Virandeville;VIRANDEVILLE;50690;50;MANCHE;1
50644;;Virey;VIREY;50600;50;MANCHE;1
50646;Le;Vrtot;VRETOT;50260;50;MANCHE;1
50647;;Yquelon;YQUELON;50400;50;MANCHE;1
50648;;Yvetot-Bocage;YVETOT-BOCAGE;50700;50;MANCHE;1
51001;;Ablancourt;ABLANCOURT;51240;51;MARNE;1
51002;;Saint-Martin-d'Ablois;SAINT-MARTIN-D'ABLOIS;51200;51;MARNE;1
51003;;Aigny;AIGNY;51150;51;MARNE;1
51004;;Allemanche-Launay-et-Soyer;ALLEMANCHE-LAUNAY-ET-SOYER;51260;51;MARNE;1
51005;;Allemant;ALLEMANT;51120;51;MARNE;1
51006;;Alliancelles;ALLIANCELLES;51250;51;MARNE;1
51007;;Ambonnay;AMBONNAY;51150;51;MARNE;1
51008;;Ambrires;AMBRIERES;51290;51;MARNE;1
51009;;Anglure;ANGLURE;51260;51;MARNE;1
51010;;Angluzelles-et-Courcelles;ANGLUZELLES-ET-COURCELLES;51230;51;MARNE;1
51012;;Anthenay;ANTHENAY;51700;51;MARNE;1
51013;;Aougny;AOUGNY;51170;51;MARNE;1
51014;;Arcis-le-Ponsart;ARCIS-LE-PONSART;51170;51;MARNE;1
51015;;Argers;ARGERS;51800;51;MARNE;1
51016;;Arrigny;ARRIGNY;51290;51;MARNE;1
51017;;Arzillires-Neuville;ARZILLIERES-NEUVILLE;51290;51;MARNE;1
51018;;Athis;ATHIS;51150;51;MARNE;1
51019;;Aubrive;AUBERIVE;51600;51;MARNE;1
51020;;Aubilly;AUBILLY;51170;51;MARNE;1
51022;;Aulnay-l'Atre;AULNAY-L'AITRE;51240;51;MARNE;1
51023;;Aulnay-sur-Marne;AULNAY-SUR-MARNE;51150;51;MARNE;1
51025;;Aumnancourt;AUMENANCOURT;51110;51;MARNE;1
51027;;Auve;AUVE;51800;51;MARNE;1
51028;;Avenay-Val-d'Or;AVENAY-VAL-D'OR;51160;51;MARNE;1
51029;;Avize;AVIZE;51190;51;MARNE;1
51030;;Ay;AY;51160;51;MARNE;1
51031;;Baconnes;BACONNES;51400;51;MARNE;1
51032;;Bagneux;BAGNEUX;51260;51;MARNE;1
51033;Le;Baizil;BAIZIL;51270;51;MARNE;1
51034;;Bannay;BANNAY;51270;51;MARNE;1
51035;;Bannes;BANNES;51230;51;MARNE;1
51036;;Barbonne-Fayel;BARBONNE-FAYEL;51120;51;MARNE;1
51037;;Baslieux-ls-Fismes;BASLIEUX-LES-FISMES;51170;51;MARNE;1
51038;;Baslieux-sous-Chtillon;BASLIEUX-SOUS-CHATILLON;51700;51;MARNE;1
51039;;Bassu;BASSU;51300;51;MARNE;1
51040;;Bassuet;BASSUET;51300;51;MARNE;1
51041;;Baudement;BAUDEMENT;51260;51;MARNE;1
51042;;Baye;BAYE;51270;51;MARNE;1
51043;;Bazancourt;BAZANCOURT;51110;51;MARNE;1
51044;;Beaumont-sur-Vesle;BEAUMONT-SUR-VESLE;51360;51;MARNE;1
51045;;Beaunay;BEAUNAY;51270;51;MARNE;1
51046;;Beine-Nauroy;BEINE-NAUROY;51490;51;MARNE;1
51047;;Belval-en-Argonne;BELVAL-EN-ARGONNE;51330;51;MARNE;1
51048;;Belval-sous-Chtillon;BELVAL-SOUS-CHATILLON;51480;51;MARNE;1
51049;;Bergres-ls-Vertus;BERGERES-LES-VERTUS;51130;51;MARNE;1
51050;;Bergres-sous-Montmirail;BERGERES-SOUS-MONTMIRAIL;51210;51;MARNE;1
51051;;Bermricourt;BERMERICOURT;51220;51;MARNE;1
51052;;Berru;BERRU;51420;51;MARNE;1
51053;;Berzieux;BERZIEUX;51800;51;MARNE;1
51054;;Btheniville;BETHENIVILLE;51490;51;MARNE;1
51055;;Btheny;BETHENY;51450;51;MARNE;1
51056;;Bethon;BETHON;51260;51;MARNE;1
51057;;Bettancourt-la-Longue;BETTANCOURT-LA-LONGUE;51330;51;MARNE;1
51058;;Bezannes;BEZANNES;51430;51;MARNE;1
51059;;Bignicourt-sur-Marne;BIGNICOURT-SUR-MARNE;51300;51;MARNE;1
51060;;Bignicourt-sur-Saulx;BIGNICOURT-SUR-SAULX;51340;51;MARNE;1
51061;;Billy-le-Grand;BILLY-LE-GRAND;51400;51;MARNE;1
51062;;Binarville;BINARVILLE;51800;51;MARNE;1
51063;;Binson-et-Orquigny;BINSON-ET-ORQUIGNY;51700;51;MARNE;1
51064;;Bisseuil;BISSEUIL;51150;51;MARNE;1
51065;;Blacy;BLACY;51300;51;MARNE;1
51066;;Blaise-sous-Arzillires;BLAISE-SOUS-ARZILLIERES;51300;51;MARNE;1
51068;;Blesme;BLESME;51340;51;MARNE;1
51069;;Bligny;BLIGNY;51170;51;MARNE;1
51070;;Boissy-le-Repos;BOISSY-LE-REPOS;51210;51;MARNE;1
51071;;Bouchy-Saint-Genest;BOUCHY-SAINT-GENEST;51310;51;MARNE;1
51072;;Bouilly;BOUILLY;51390;51;MARNE;1
51073;;Bouleuse;BOULEUSE;51170;51;MARNE;1
51074;;Boult-sur-Suippe;BOULT-SUR-SUIPPE;51110;51;MARNE;1
51075;;Bourgogne;BOURGOGNE;51110;51;MARNE;1
51076;;Boursault;BOURSAULT;51480;51;MARNE;1
51077;;Bouvancourt;BOUVANCOURT;51140;51;MARNE;1
51078;;Bouy;BOUY;51400;51;MARNE;1
51079;;Bouzy;BOUZY;51150;51;MARNE;1
51080;;Brandonvillers;BRANDONVILLERS;51290;51;MARNE;1
51081;;Branscourt;BRANSCOURT;51140;51;MARNE;1
51082;;Braux-Sainte-Cohire;BRAUX-SAINTE-COHIERE;51800;51;MARNE;1
51083;;Braux-Saint-Remy;BRAUX-SAINT-REMY;51800;51;MARNE;1
51084;;Brban;BREBAN;51320;51;MARNE;1
51085;Le;Breuil;BREUIL;51210;51;MARNE;1
51086;;Breuil;BREUIL;51140;51;MARNE;1
51087;;Breuvery-sur-Coole;BREUVERY-SUR-COOLE;51240;51;MARNE;1
51088;;Brimont;BRIMONT;51220;51;MARNE;1
51089;;Brouillet;BROUILLET;51170;51;MARNE;1
51090;;Broussy-le-Grand;BROUSSY-LE-GRAND;51230;51;MARNE;1
51091;;Broussy-le-Petit;BROUSSY-LE-PETIT;51230;51;MARNE;1
51092;;Broyes;BROYES;51120;51;MARNE;1
51093;;Brugny-Vaudancourt;BRUGNY-VAUDANCOURT;51200;51;MARNE;1
51094;;Brusson;BRUSSON;51300;51;MARNE;1
51095;Le;Buisson;BUISSON;51300;51;MARNE;1
51097;;Bussy-le-Chteau;BUSSY-LE-CHATEAU;51600;51;MARNE;1
51098;;Bussy-le-Repos;BUSSY-LE-REPOS;51330;51;MARNE;1
51099;;Bussy-Lettre;BUSSY-LETTREE;51320;51;MARNE;1
51100;La;Caure;CAURE;51270;51;MARNE;1
51101;;Caurel;CAUREL;51110;51;MARNE;1
51102;;Cauroy-ls-Hermonville;CAUROY-LES-HERMONVILLE;51220;51;MARNE;1
51103;La;Celle-sous-Chantemerle;CELLE-SOUS-CHANTEMERLE;51260;51;MARNE;1
51104;;Cernay-en-Dormois;CERNAY-EN-DORMOIS;51800;51;MARNE;1
51105;;Cernay-ls-Reims;CERNAY-LES-REIMS;51420;51;MARNE;1
51106;;Cernon;CERNON;51240;51;MARNE;1
51107;;Chaintrix-Bierges;CHAINTRIX-BIERGES;51130;51;MARNE;1
51108;;Chlons-en-Champagne;CHALONS-EN-CHAMPAGNE;51000;51;MARNE;1
51109;;Chlons-sur-Vesle;CHALONS-SUR-VESLE;51140;51;MARNE;1
51110;;Chaltrait;CHALTRAIT;51130;51;MARNE;1
51111;;Chambrecy;CHAMBRECY;51170;51;MARNE;1
51112;;Chamery;CHAMERY;51500;51;MARNE;1
51113;;Champaubert;CHAMPAUBERT;51270;51;MARNE;1
51115;;Champfleury;CHAMPFLEURY;51500;51;MARNE;1
51116;;Champguyon;CHAMPGUYON;51310;51;MARNE;1
51117;;Champigneul-Champagne;CHAMPIGNEUL-CHAMPAGNE;51150;51;MARNE;1
51118;;Champigny;CHAMPIGNY;51370;51;MARNE;1
51119;;Champillon;CHAMPILLON;51160;51;MARNE;1
51120;;Champlat-et-Boujacourt;CHAMPLAT-ET-BOUJACOURT;51480;51;MARNE;1
51121;;Champvoisy;CHAMPVOISY;51700;51;MARNE;1
51122;;Changy;CHANGY;51300;51;MARNE;1
51124;;Chantemerle;CHANTEMERLE;51260;51;MARNE;1
51125;;Chapelaine;CHAPELAINE;51290;51;MARNE;1
51126;La;Chapelle-Felcourt;CHAPELLE-FELCOURT;51800;51;MARNE;1
51127;La;Chapelle-Lasson;CHAPELLE-LASSON;51260;51;MARNE;1
51128;La;Chapelle-sous-Orbais;CHAPELLE-SOUS-ORBAIS;51270;51;MARNE;1
51129;;Charleville;CHARLEVILLE;51120;51;MARNE;1
51130;;Charmont;CHARMONT;51330;51;MARNE;1
51132;Les;Charmontois;CHARMONTOIS;51330;51;MARNE;1
51133;Le;Chtelier;CHATELIER;51330;51;MARNE;1
51134;;Chtelraould-Saint-Louvent;CHATELRAOULD-SAINT-LOUVENT;51300;51;MARNE;1
51135;;Chtillon-sur-Brou;CHATILLON-SUR-BROUE;51290;51;MARNE;1
51136;;Chtillon-sur-Marne;CHATILLON-SUR-MARNE;51700;51;MARNE;1
51137;;Chtillon-sur-Morin;CHATILLON-SUR-MORIN;51310;51;MARNE;1
51138;;Chtrices;CHATRICES;51800;51;MARNE;1
51139;;Chaudefontaine;CHAUDEFONTAINE;51800;51;MARNE;1
51140;;Chaumuzy;CHAUMUZY;51170;51;MARNE;1
51141;La;Chausse-sur-Marne;CHAUSSEE-SUR-MARNE;51240;51;MARNE;1
51142;;Chavot-Courcourt;CHAVOT-COURCOURT;51200;51;MARNE;1
51143;Le;Chemin;CHEMIN;51800;51;MARNE;1
51144;;Cheminon;CHEMINON;51250;51;MARNE;1
51145;;Chenay;CHENAY;51140;51;MARNE;1
51146;;Cheniers;CHENIERS;51510;51;MARNE;1
51147;La;Cheppe;CHEPPE;51600;51;MARNE;1
51148;;Cheppes-la-Prairie;CHEPPES-LA-PRAIRIE;51240;51;MARNE;1
51149;;Chepy;CHEPY;51240;51;MARNE;1
51150;;Cherville;CHERVILLE;51150;51;MARNE;1
51151;;Chichey;CHICHEY;51120;51;MARNE;1
51152;;Chigny-les-Roses;CHIGNY-LES-ROSES;51500;51;MARNE;1
51153;;Chouilly;CHOUILLY;51200;51;MARNE;1
51154;;Clamanges;CLAMANGES;51130;51;MARNE;1
51155;;Clesles;CLESLES;51260;51;MARNE;1
51156;;Cloyes-sur-Marne;CLOYES-SUR-MARNE;51300;51;MARNE;1
51157;;Coizard-Joches;COIZARD-JOCHES;51270;51;MARNE;1
51158;;Val-des-Marais;VAL-DES-MARAIS;51130;51;MARNE;1
51160;;Compertrix;COMPERTRIX;51510;51;MARNE;1
51161;;Cond-sur-Marne;CONDE-SUR-MARNE;51150;51;MARNE;1
51162;;Conflans-sur-Seine;CONFLANS-SUR-SEINE;51260;51;MARNE;1
51163;;Congy;CONGY;51270;51;MARNE;1
51164;;Connantray-Vaurefroy;CONNANTRAY-VAUREFROY;51230;51;MARNE;1
51165;;Connantre;CONNANTRE;51230;51;MARNE;1
51166;;Contault;CONTAULT;51330;51;MARNE;1
51167;;C