#include <malloc.h>

#ifdef MOMO
#include <momo/kpse-api.h>
#  define KPSE_API WINAPI
#  define IS_KPSE_EXCEPTION(e) ((KPSE_EXCEPTION(0)&(e)) == KPSE_EXCEPTION(0))
#  define KPSE_CHECK_EXCEPTION(e) \
    IS_KPSE_EXCEPTION(e) ? \
      EXCEPTION_EXECUTE_HANDLER : EXCEPTION_CONTINUE_SEARCH
#  define BEGIN_KPSE __try {
#  define END_KPSE(r) \
      } _except(KPSE_CHECK_EXCEPTION(GetExceptionCode())) { \
        return (r); \
      }
#else
#include <kpathsea/kpathsea.h>
#  define KPSE_API /* */
#  define BEGIN_KPSE /* */
#  define END_KPSE(r)   /* */
#endif
#include <win32dll/win32lib.h>

HANDLE hDLL = NULL;
HANDLE hMakeTeX = NULL;
LPFMakeTeXPKProc MakeTeXPK;
LPFMakeTeXTFMProc MakeTeXTFM;
LPFMakeTeXMFProc MakeTeXMF;

VOID (KPSE_API *set_progname)(LPCSTR );
LPSTR (KPSE_API *find_file)(LPCSTR, kpse_file_format_type, BOOL );
LPSTR (KPSE_API *path_search)(LPCSTR, LPCSTR, BOOL );
LPSTR (KPSE_API *var_value)(LPCSTR );
LPSTR (KPSE_API *var_expand)(LPCSTR );
LPSTR (KPSE_API *path_expand)(LPCSTR );
VOID (KPSE_API *init_prog)(LPCSTR,  DWORD, LPCSTR, LPCSTR );
VOID (KPSE_API *set_program_enabled)(kpse_file_format_type,
				     BOOL, kpse_src_type );
LPSTR (KPSE_API *find_glyph)(LPCTSTR font_name, UINT dpi, 
			     kpse_file_format_type format,
			     kpse_glyph_file_type *glyph_file);
BOOL (KPSE_API *tex_hush)(LPCSTR );
UINT (KPSE_API *magstep_fix)(UINT, UINT, LPINT);
BOOL (KPSE_API *bitmap_tolerance)(double, double);
#ifdef MOMO
VOID (WINAPI *set_callback)(KpseCallBack*, DWORD );
BOOL (WINAPI *set_curdir)(LPCSTR dir);
#endif

char **version_string;
char **bug_address;

#define GET_ADDRESS(v, n) \
  (FARPROC)v = GetProcAddress(hDLL, n); \
  if (!v) { FreeLibrary(hDLL); hDLL = NULL; return FALSE; }

typedef struct {
  BOOL* lpMakeTeXDiscardErrors;
  UINT* lpDebug;
  LPCSTR* lpFallbackResolutionsString;
  kpse_format_info_type* lpFormatInfo;
} texdll_search_info;


BOOL APIENTRY DllMain(HANDLE hModule,
                      DWORD ul_reason_for_call, 
		      LPVOID lpReserved)
{
  switch(ul_reason_for_call) {
  case DLL_PROCESS_ATTACH:
    win32_init_module(hModule);
    {
      LPCSTR dllname = "kpathsea.dll";
      hDLL = win32_load_library_just_in_same_directory(dllname);
#ifdef MOMO
      if (!hDLL)
	hDLL = win32_load_just_registry_library("Software\\momo\\Kpathsea",
						"Path", dllname);
#endif
      if (!hDLL) hDLL =  win32_load_tex_library(dllname);
    }
    if (!hDLL) return FALSE;
    GET_ADDRESS(set_progname, "kpse_set_progname");
    GET_ADDRESS(find_file, "kpse_find_file");
    GET_ADDRESS(path_search, "kpse_path_search");
    GET_ADDRESS(var_value, "kpse_var_value");
    GET_ADDRESS(var_expand, "kpse_var_expand");
    GET_ADDRESS(path_expand, "kpse_path_expand");
    GET_ADDRESS(init_prog, "kpse_init_prog");
    GET_ADDRESS(set_program_enabled, "kpse_set_program_enabled");
    GET_ADDRESS(version_string, "kpathsea_version_string");
    GET_ADDRESS(bug_address, "kpse_bug_address");
    GET_ADDRESS(find_glyph, "kpse_find_glyph");
    GET_ADDRESS(tex_hush, "kpse_tex_hush");
    GET_ADDRESS(magstep_fix, "kpse_magstep_fix");
    GET_ADDRESS(bitmap_tolerance, "kpse_bitmap_tolerance");
#ifdef MOMO
    GET_ADDRESS(set_callback, "kpse_set_callback");
    GET_ADDRESS(set_curdir, "kpse_change_dir");
#endif
    break;
  case DLL_PROCESS_DETACH:
    if (hDLL) FreeLibrary(hDLL);
    if (hMakeTeX) FreeLibrary(hMakeTeX);
    hDLL = NULL;
    hMakeTeX = NULL;
    break;
  }
  return TRUE;
}

LPVOID WINAPI KpseInitialize(VOID)
{
  BOOL code = TRUE;
  texdll_search_info* pInfo = malloc(sizeof(texdll_search_info));
  if (!pInfo) return NULL;
#ifdef MOMO
  {
    BOOL* (WINAPI *get)();
    (FARPROC)get = GetProcAddress(hDLL, "kpse_make_tex_discard_errors_get");
    if (get) pInfo->lpMakeTeXDiscardErrors = (*get)();
  }
  {
    LPCSTR (WINAPI *get)();
    (FARPROC)get = GetProcAddress(hDLL, "kpse_fallback_resolutions_string_get");
    pInfo->lpFallbackResolutionsString = get ? (*get)() : NULL;
  }
  {
    kpse_format_info_type* (WINAPI *get)();
    (FARPROC)get = GetProcAddress(hDLL, "kpse_format_info_get");
    pInfo->lpFormatInfo = get ? (*get)() : NULL;
  }
#else
  (FARPROC)pInfo->lpMakeTeXDiscardErrors =
    GetProcAddress(hDLL, "kpse_make_tex_discard_errors");
  {
    LPCSTR *p;
    (FARPROC)p = GetProcAddress(hDLL, "kpse_fallback_resolutions_string");
    pInfo->lpFallbackResolutionsString = p;
  }
  {
    kpse_format_info_type *p;
    (FARPROC)p = GetProcAddress(hDLL, "kpse_format_info");
    pInfo->lpFormatInfo = p;
  }
#endif
  (FARPROC)pInfo->lpDebug = GetProcAddress(hDLL, "kpathsea_debug");
  if (!pInfo->lpMakeTeXDiscardErrors ||
      !pInfo->lpDebug ||
      !pInfo->lpFormatInfo) {
    free(pInfo);
    return NULL;
  }
  return pInfo;
}

BOOL WINAPI KpseUninitialize(LPVOID pContext)
{
  free(pContext);
  return TRUE;
}

BOOL WINAPI KpseSetProgName(LPVOID pContext, LPCSTR name)
{
  BEGIN_KPSE
  (*set_progname)(name);
  END_KPSE(FALSE)
  return TRUE;
}

LPSTR WINAPI KpseFindFile(LPVOID pContext, LPCSTR name,
			  DWORD fmt, BOOL must_exist)
{
  LPSTR p;
  BEGIN_KPSE
  p = (*find_file)(name, fmt, must_exist);
  END_KPSE(NULL)
  return p;
}

LPSTR WINAPI KpsePathSearch(LPVOID pContext,
			    LPCSTR path, LPCTSTR name, BOOL must_exist)
{
  LPSTR p;
  BEGIN_KPSE
  p = (*path_search)(path, name, must_exist);
  END_KPSE(NULL)
  return p;
}

LPSTR WINAPI KpseVarValue(LPVOID pContext, LPCSTR name)
{
  LPSTR p;
  BEGIN_KPSE
  p = (*var_value)(name);
  END_KPSE(NULL)
  return p;
}

LPSTR WINAPI KpseVarExpand(LPVOID pContext, LPCSTR name)
{
  LPSTR p;
  BEGIN_KPSE
  p = (*var_expand)(name);
  END_KPSE(NULL)
  return p;
}

LPSTR WINAPI KpsePathExpand(LPVOID pContext, LPCSTR name)
{
  LPSTR p;
  BEGIN_KPSE
  p = (*path_expand)(name);
  END_KPSE(NULL)
  return p;
}

BOOL WINAPI KpseInitProg(LPVOID pContext,
			 LPCSTR pfx, ULONG dpi, LPCSTR mode, LPCSTR fb)
{
  BEGIN_KPSE
  (*init_prog)(pfx, dpi, mode, fb);
  END_KPSE(FALSE)
  return TRUE;
}

BOOL WINAPI KpseSetProgramEnabled(LPVOID pContext,
				  DWORD fmt, BOOL bydefault, DWORD srctype)
{
  BEGIN_KPSE
  (*set_program_enabled)(fmt, bydefault, srctype);
  END_KPSE(FALSE)
  return TRUE;
}

LPSTR WINAPI KpseFindGlyph(LPVOID pContext,
			   LPCTSTR font_name, UINT dpi, 
			   kpse_file_format_type format,
			   kpse_glyph_file_type *glyph_file)
{
  LPSTR p;
  BEGIN_KPSE
  p = (*find_glyph)(font_name, dpi, format, glyph_file);
  END_KPSE(NULL)
  return p;
}

BOOL KpseTeXHush(LPVOID pContext, LPCSTR what)
{
  BOOL r;
  BEGIN_KPSE
  r = (*tex_hush)(what);
  END_KPSE(FALSE)
  return r;
}

UINT KpseMagstepFix(LPVOID pContext, UINT dpi, UINT bdpi, LPINT pret)
{
  UINT r;
  BEGIN_KPSE
  r = (*magstep_fix)(dpi, bdpi, pret);
  END_KPSE(0)
}

BOOL KpseBitmapTolerance(LPVOID pContext, double dpi1, double dpi2)
{
  BOOL r;
  BEGIN_KPSE
  r = (*bitmap_tolerance)(dpi1, dpi2);
  END_KPSE(FALSE)
  return r;
}

#ifdef MOMO

typedef struct {
  LPFTeXCallbackProc lpCallerCallback;
  LPVOID lpCallerCallbackData;
} CallbackInfo;

static LONG WINAPI maketex_callback(TeXDLLMessage msg,
				    DWORD param1, DWORD param2,
				    LPVOID app_data)
{
  CallbackInfo *lpInfo = (CallbackInfo *)app_data;
  
  if (msg == TeXDLL_STDOUT_WRITE)
    msg = TeXDLL_CONSOLE_WRITE;
  return lpInfo->lpCallerCallback(msg, param1, param2,
				  lpInfo->lpCallerCallbackData);
}

static LONG CALLBACK callback(UINT msg, DWORD wParam,
			      DWORD lParam, DWORD user_param)
{
  Win32Context* pContext = (Win32Context*)user_param;
  LONG code = 0;
  if (!pContext) return 0;
  
  switch (msg) {
  case KPSE_CONSOLE_WRITE:
    (*pContext->lpfCallback)(TeXDLL_CONSOLE_WRITE, wParam, lParam,
			     pContext->lpvCallbackData);
    return 1;
  case KPSE_CONSOLE_FLUSH:
    return 1;
  case KPSE_MAKETEX:
    code = (*pContext->lpfCallback)(TeXDLL_BEGIN_MAKETEX, lParam, 0,
				    pContext->lpvCallbackData);
    if (code > 0) return 1;
    if (code < 0) return -1;
    
    if (!hMakeTeX) {
      hMakeTeX = win32_load_tex_library("MakeTeX.dll");
      if (!hMakeTeX) return 0;
      MakeTeXPK =
	(LPFMakeTeXPKProc)GetProcAddress(hMakeTeX, "MakeTeXPKMain");
      MakeTeXTFM =
	(LPFMakeTeXTFMProc)GetProcAddress(hMakeTeX, "MakeTeXTFMMain");
      MakeTeXMF =
	(LPFMakeTeXMFProc)GetProcAddress(hMakeTeX, "MakeTeXMFMain");
    }
    {
      kpse_maketex_type* pMakeTeX = (kpse_maketex_type*)lParam;
      CallbackInfo *lpInfo = malloc(sizeof(CallbackInfo));
      if (!lpInfo) return -1;
      lpInfo->lpCallerCallback = pContext->lpfCallback;
      lpInfo->lpCallerCallbackData = pContext->lpvCallbackData;
      switch (pMakeTeX->format) {
      case kpse_pk_format:
      case kpse_any_glyph_format:
	code = (*MakeTeXPK)(pMakeTeX->lpszPathName,
			    pMakeTeX->dwLen,
			    maketex_callback,
			    lpInfo,
			    pMakeTeX->lpszFontName,
			    pMakeTeX->lpszMag,
			    pMakeTeX->dwModeDpi,
			    pMakeTeX->lpszMode) ? 1 : -1;
	break;
      case kpse_tfm_format:
	code = (*MakeTeXTFM)(pMakeTeX->lpszPathName,
			     pMakeTeX->dwLen,
			     maketex_callback,
			     lpInfo,
			     pMakeTeX->lpszFontName) ? 1 : -1;
	break;
      case kpse_mf_format:
	code = (*MakeTeXMF)(pMakeTeX->lpszPathName,
			    pMakeTeX->dwLen,
			    maketex_callback,
			    lpInfo,
			    pMakeTeX->lpszFontName) ? 1 : -1;
	break;
      default:
	code = -1;
	break;
      }
      free(lpInfo);
    }
    (*pContext->lpfCallback)(TeXDLL_END_MAKETEX, lParam, 0,
			     pContext->lpvCallbackData);
    break;
  }
  return code; /* msg was not processed. */
}
#endif

BOOL WINAPI KpseSetCallback(LPVOID pKpseContext, Win32Context* pContext)
{
#ifdef MOMO
  BEGIN_KPSE
  (*set_callback)(pContext ? callback : 0, (DWORD)pContext);
  END_KPSE(FALSE)
#endif
  return TRUE;
}

BOOL WINAPI KpseFree(LPVOID pContext, LPVOID ptr)
{
  return TRUE;
}

LPCSTR WINAPI KpseGetVersionString(LPVOID pContext)
{
  return *version_string;
}

LPCSTR WINAPI KpseGetBugAddress(LPVOID pContext)
{
  return *bug_address;
}

BOOL* WINAPI KpseGetMakeTeXDiscardErrors(LPVOID pContext)
{
  texdll_search_info* pInfo = pContext;
  return pInfo->lpMakeTeXDiscardErrors;
}

UINT* WINAPI KpseGetDebug(LPVOID pContext)
{
  texdll_search_info* pInfo = pContext;
  return pInfo->lpDebug;
}

LPCSTR* WINAPI KpseGetFallbackResolutionsString(LPVOID pContext)
{
  texdll_search_info* pInfo = pContext;
  return pInfo->lpFallbackResolutionsString;
}

kpse_format_info_type* WINAPI KpseGetFormatInfo(LPVOID pContext)
{
  texdll_search_info* pInfo = pContext;
  return pInfo->lpFormatInfo;
}

BOOL WINAPI KpseSetCurrentDirectory(LPVOID pContext, LPCSTR dir)
{
#ifdef MOMO
  BOOL code;
  BEGIN_KPSE
  code = (*set_curdir)(dir);
  END_KPSE(FALSE)
  return code;
#else
  return TRUE;
#endif
}
