#include "gs.h"

#define PLUGIN_VERSION "0.1"
#define PLUGIN_DATE    "1998/01/23"

HMODULE g_hModule;
CGS* g_lpGS = 0;
LPCSTR g_lpszPlugInType = "00IN";
LPCSTR g_lpszCopyright =
  "Susie Plug-In for PostScript with Ghostscript\n"
  "Version " PLUGIN_VERSION "(" PLUGIN_DATE ")\n"
  "Copyright(C) 1998, Kazunori Asayama.";

static CRITICAL_SECTION g_csGS;

struct CFormatInfo {
  LPCSTR lpszExt;
  LPCSTR lpszName;
};

CFormatInfo g_fiFormats[] = {
  { "*.PS;*.PSF;*.EPS;*.EPSF;*.EPSI", "PostScript", },
};

#define NUM_FORMATS (sizeof(g_fiFormats) / sizeof(CFormatInfo))

// Beginning of SPI API
struct PictureInfo
{
  long left;
  long top;
  long width;
  long height;
  WORD x_density;
  WORD y_density;
  short colorDepth;
  HLOCAL hInfo;
};

// End of SPI API


/////////////////////////////// Internal ////////////////////////////////

//////////////////////////////// Exports ////////////////////////////////

BOOL WINAPI DllMain(HMODULE hModule, ULONG uReason, LPVOID lpReserved)
{
  switch (uReason) {
  case DLL_PROCESS_ATTACH:
    g_hModule = hModule;
    g_lpGS = new CGS;
    ::InitializeCriticalSection(&g_csGS);
    break;
  case DLL_PROCESS_DETACH:
    if (g_lpGS) delete g_lpGS;
    ::DeleteCriticalSection(&g_csGS);
    break;
  }
  return TRUE;
}

extern "C" int WINAPI ConfigurationDlg(HWND hParent, int nFunc)
{
  int code;
  
  switch (nFunc) {
  case 1:
    code = g_lpGS->DoSetup(hParent);
    break;
  case 0: // About Plug-In
  default:
    code = SPI_ERROR_NOTIMPL;
    break;    
  }
  return code;
}

extern "C" int WINAPI GetPluginInfo(int nInfoNo, LPSTR lpszBuf, int nBufLen)
{
  LPCSTR lpszStr = 0;
  if (nInfoNo == 0) { // API Version
    lpszStr = g_lpszPlugInType;
  }
  else if (nInfoNo == 1) { // Plug-In Version, Name, Copyright, etc.
    lpszStr = g_lpszCopyright;
  }
  else if (nInfoNo >= 2) {
    int idx = nInfoNo / 2 - 1;
    if (idx >= NUM_FORMATS) // Invalid index
      return 0;
    if (nInfoNo % 2) {
      lpszStr = g_fiFormats[idx].lpszName;
    }
    else {
      lpszStr = g_fiFormats[idx].lpszExt;
    }
  }
  
  if (lpszStr) {
    int len = strlen(lpszStr) + 1;
    if (nBufLen < len) return 0;
    memcpy(lpszBuf, lpszStr, len);
    return len;
  }
  else {
    return 0;
  }
}

extern "C" int WINAPI IsSupported(LPSTR lpszFileName, DWORD dw)
{
  CFileMapping map;
  LPCSTR lpData;
  DWORD dwLen;
  if (HIWORD(dw) == 0) {
    int code =
      map.Map((HANDLE)dw, ::SetFilePointer((HANDLE)dw, 0, 0, FILE_CURRENT));
    if (code != SPI_ERROR_NORMAL)
      return code;
    lpData = (LPCSTR)map.GetView();
    dwLen = map.GetSize();
  }
  else {
    lpData = (LPCSTR)dw;
    dwLen = 2000;
  }
  
  return g_lpGS->IsPS(lpData, lpData + dwLen);
}

int OpenPicture(CFileMapping*lpMap,
		LPSTR lpszBuf,
		LONG nLen,
		UINT uFlag,
		struct PictureInfo *lpInfo,
		CPSFInfo* lpPSI)
{
  UINT uDataType = uFlag & 0x00000007;
  LPCSTR lpData;
  DWORD dwSize;
  if (uDataType == 0) { // File
    int code = lpMap->Map(lpszBuf, nLen);
    if (code != SPI_ERROR_NORMAL)
      return code;
    lpData = (LPCSTR)lpMap->GetView();
    dwSize = lpMap->GetSize();
  }
  else if (uDataType == 1) {
    lpData = lpszBuf;
    dwSize = nLen;
  }
  else {
    return SPI_ERROR_UNKNOWN_FORMAT;
  }
  
  LPCSTR lpEnd = lpData + dwSize;
  if (!g_lpGS->IsPS(lpData, lpEnd))
    return SPI_ERROR_UNKNOWN_FORMAT;
  
  lpData = PSNextLine(lpData, lpEnd);
  
  PictureInfo pi;
  ZeroMemory(&pi, sizeof(pi));
  pi.left = 0;
  pi.top = 0;
  
  CPSFInfo psi;
  if (lpData >= lpEnd ||
      !g_lpGS->ParsePSF(lpData, lpEnd, psi, 0) ||
      !g_lpGS->GetClip()) {
    psi.bb = g_lpGS->GetDefaultBoundingBox();
  }
  else {
    if (psi.bb.urx <= psi.bb.llx)
      psi.bb.urx = psi.bb.llx + 1;
    if (psi.bb.ury <= psi.bb.lly)
      psi.bb.ury = psi.bb.lly + 1;
  }
  
  pi.x_density = g_lpGS->GetXResolution();
  pi.y_density = g_lpGS->GetYResolution();
  
  pi.width = (psi.bb.urx - psi.bb.llx + 1) * pi.x_density / 72;
  pi.height = (psi.bb.ury - psi.bb.lly + 1) * pi.y_density / 72;
  pi.colorDepth = g_lpGS->GetDepth();
  
  *lpInfo = pi;
  *lpPSI = psi;
  
  return SPI_ERROR_NORMAL;
}

extern "C" int WINAPI GetPictureInfo(LPSTR lpszBuf,
				     LONG nLen,
				     UINT uFlag,
				     struct PictureInfo *lpInfo)
{
  CFileMapping map;
  CPSFInfo psi;
  int code = OpenPicture(&map, lpszBuf, nLen, uFlag, lpInfo, &psi);
  if (code != SPI_ERROR_NORMAL)
    return code;

  return SPI_ERROR_NORMAL;
}

static int GSDibToSusieDib(LPBITMAPINFO lpDib,
			   HLOCAL* lphHeader,
			   HLOCAL* lphBits)
{
  // Check Header size
  int nHeaderSize = sizeof(BITMAPINFOHEADER);
  DWORD nPallete = lpDib->bmiHeader.biClrUsed;
  DWORD nPalleteReq = 0;
  switch (lpDib->bmiHeader.biBitCount) {
  case 1:
    if (!nPallete) nPallete = 2;
    nPalleteReq = 2;
    break;
  case 4:
    if (!nPallete) nPallete = 16;
    nPalleteReq = 16;
    break;
  case 8:
    if (!nPallete) nPallete = 256;
    nPalleteReq = 256;
    break;
  case 24: // no Pallete or Mask
    nPallete = 0;
    break;
  case 16:
  case 32:
    nPallete = 3; // RGB Mask
    nPalleteReq = 3;
    break;
  default: // unknown format
    return SPI_ERROR_INTERNAL;
  }
  nHeaderSize += nPallete * sizeof(RGBQUAD);
  
  DWORD dwBitsSize;
  if ((lpDib->bmiHeader.biCompression == BI_RGB ||
       lpDib->bmiHeader.biCompression == BI_BITFIELDS) &&
      lpDib->bmiHeader.biSizeImage == 0) {
    // No Compression.
    // Scanlines are padded by 32bits.
    dwBitsSize =
      ((DWORD)lpDib->bmiHeader.biWidth * lpDib->bmiHeader.biBitCount + 31) /
	32 * 4 * lpDib->bmiHeader.biHeight;
  }
  else {
    dwBitsSize = lpDib->bmiHeader.biSizeImage;
  }
  if (dwBitsSize == 0) {
    return SPI_ERROR_INTERNAL;
  }
  
  HLOCAL hHdr =
    ::LocalAlloc(LMEM_MOVEABLE,
		 sizeof(BITMAPINFOHEADER) + nPalleteReq * sizeof(RGBQUAD));
  if (!hHdr) return SPI_ERROR_NO_MEMORY;
  LPBITMAPINFO lpHdr = (LPBITMAPINFO)::LocalLock(hHdr);
  if (!lpHdr) {
    ::LocalFree(hHdr);
    return SPI_ERROR_MEMORY;
  }
  ZeroMemory(lpHdr, sizeof(BITMAPINFOHEADER) + nPalleteReq * sizeof(RGBQUAD));
  memcpy(lpHdr, lpDib, nHeaderSize);
  lpHdr->bmiHeader.biClrUsed = 0;
  ::LocalUnlock(hHdr);
  
  HLOCAL hBits = ::LocalAlloc(LMEM_MOVEABLE, dwBitsSize);
  if (!hBits) {
    ::LocalFree(hHdr);
    return SPI_ERROR_NO_MEMORY;
  }
  LPVOID lpBits = ::LocalLock(hBits);
  if (!lpBits) {
    ::LocalFree(hHdr);
    ::LocalFree(hBits);
    return SPI_ERROR_MEMORY;
  }
  memcpy(lpBits, (LPBYTE)lpDib + nHeaderSize, dwBitsSize);
  ::LocalUnlock(hBits);
  
  *lphHeader = hHdr;
  *lphBits = hBits;
  
  return SPI_ERROR_NORMAL;
}

extern "C" int WINAPI GetPicture(LPSTR lpszBuf,
				 LONG lLen,
				 UINT uFlag,
				 HANDLE *pHBInfo,
				 HANDLE *pHBm,
				 FARPROC lpProgressCallback,
				 LONG lProgressCallbackParam)
{
  CFileMapping map;
  PictureInfo pi;
  CPSFInfo psi;
  
  int code = OpenPicture(&map, lpszBuf, lLen, uFlag, &pi, &psi);
  if (code != SPI_ERROR_NORMAL)
    return code;

  HGLOBAL hDib;
  ::EnterCriticalSection(&g_csGS);
  g_lpGS->BeginPage();
  code = g_lpGS->DoPage((LPCSTR)map.GetView(), map.GetSize(),
			pi.width, pi.height, psi,
			lpProgressCallback, lProgressCallbackParam,
			&hDib);
  g_lpGS->EndPage();
  ::LeaveCriticalSection(&g_csGS);
  if (code != SPI_ERROR_NORMAL) return code;
  
  LPBITMAPINFO lpDib = (LPBITMAPINFO)::GlobalLock(hDib);
  code = GSDibToSusieDib(lpDib, pHBInfo, pHBm);
  ::GlobalUnlock(hDib);
  ::GlobalFree(hDib);
  
  return code;
}

#define THUMBNAIL_MAX 64

extern "C" int WINAPI GetPreview(LPSTR lpszBuf,
				 LONG lLen,
				 UINT uFlag,
				 HANDLE *pHBInfo,
				 HANDLE *pHBm,
				 FARPROC lpProgressCallback,
				 LONG lProgressCallbackParam)
{
  CFileMapping map;
  PictureInfo pi;
  CPSFInfo psi;
  
  int code = OpenPicture(&map, lpszBuf, lLen, uFlag, &pi, &psi);
  if (code != SPI_ERROR_NORMAL)
    return code;

  if (pi.width > THUMBNAIL_MAX || pi.height > THUMBNAIL_MAX) {
    DWORD dwPSWidth = psi.bb.urx - psi.bb.llx;
    DWORD dwPSHeight = psi.bb.ury - psi.bb.lly;
    if (dwPSWidth > dwPSHeight) {
      pi.width = THUMBNAIL_MAX;
      pi.height =
	THUMBNAIL_MAX *
	  (dwPSHeight * g_lpGS->GetYResolution()) /
	    (dwPSWidth * g_lpGS->GetXResolution());
    }
    else {
      pi.height = THUMBNAIL_MAX;
      pi.width =
	THUMBNAIL_MAX *
	  (dwPSWidth * g_lpGS->GetXResolution()) /
	    (dwPSHeight * g_lpGS->GetYResolution());
    }
    if (pi.width == 0) pi.width = 1;
    if (pi.height == 0) pi.height = 1;
  }
  
  HGLOBAL hDib;
  ::EnterCriticalSection(&g_csGS);
  g_lpGS->BeginPage();
  code = g_lpGS->DoPage((LPCSTR)map.GetView(), map.GetSize(),
			pi.width, pi.height, psi,
			lpProgressCallback, lProgressCallbackParam,
			&hDib);
  g_lpGS->EndPage();
  ::LeaveCriticalSection(&g_csGS);
  if (code != SPI_ERROR_NORMAL) return code;
  
  LPBITMAPINFO lpDib = (LPBITMAPINFO)::GlobalLock(hDib);
  code = GSDibToSusieDib(lpDib, pHBInfo, pHBm);
  ::GlobalUnlock(hDib);
  ::GlobalFree(hDib);
  
  return code;
}

////////////////////////////////// EOF //////////////////////////////////
