#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <mbstring.h>
#include <io.h>
#include <signal.h>
#include <win32dll/texdll.h>
#include <win32dll/win32lib.h>

HINSTANCE hDLL;
LPCTeXProcedures Proc;

DWORD wm_texdll;

#define REG_CUI_KEY "Software\\Kazunori Asayama\\TeX CUI"
#define REG_EDITOR "editor"
#define REG_DISPLAY_DLL "displayDll"
#define REG_NODISPLAY  "noDisplay"
#define REG_REPEAT "debugRepeat" /* for debug of DLL */

#define DEFAULT_DISPLAY_DLL "cuimfdsp.dll";
HINSTANCE hMFDisp = NULL;
typedef LPVOID WINAPI DispInitializeProc(HWND );
typedef HDC WINAPI DispBeginCharProc(LPVOID);
typedef VOID WINAPI DispEndCharProc(LPVOID);
typedef VOID WINAPI DispUninitializeProc(LPVOID);
DispInitializeProc *MFDispInit;
DispBeginCharProc *MFDispBeginChar;
DispEndCharProc *MFDispEndChar;
DispUninitializeProc *MFDispUninit;

/* Data type for callback function. */
typedef struct {
  LPVOID MFDispInfo;
  HTeX hTeX;
} callback_info;

/* Registry */

BOOL APIENTRY DllMain(HANDLE hModule,
                      DWORD ul_reason_for_call, 
		      LPVOID lpReserved)
{
  switch(ul_reason_for_call) {
  case DLL_PROCESS_ATTACH:
    hDLL = NULL;
    hMFDisp = NULL;
    wm_texdll = RegisterWindowMessage(TeXDLL_WM_TEXDLL);
    break;
  case DLL_PROCESS_DETACH:
    if (hMFDisp) FreeLibrary(hMFDisp);
    if (hDLL) FreeLibrary(hDLL);
    break;
  }
  return TRUE;
}

/* Initializing display routine for Virgin METAFONT */
int init_mf_display(callback_info *pinfo)
{
  int c = 0;

  if (win32_get_registry_int(REG_CUI_KEY "\\METAFONT", REG_NODISPLAY, 0))
    return 0;
  
  if (!hMFDisp) {
    LPCTSTR dll_filename =
      win32_get_registry_string(REG_CUI_KEY "\\METAFONT",
				REG_DISPLAY_DLL, NULL);
    if (dll_filename == NULL) dll_filename = DEFAULT_DISPLAY_DLL;
    hMFDisp = LoadLibrary(dll_filename);
    if (!hMFDisp) {
      win32_console_printf("%s: Could not load.\n", dll_filename);
    }
    else {
      MFDispInit =
	(DispInitializeProc*)GetProcAddress(hMFDisp, "MFDisplayInitialize");
      MFDispBeginChar =
	(DispBeginCharProc*)GetProcAddress(hMFDisp, "MFDisplayBeginChar");
      MFDispEndChar =
	(DispEndCharProc*)GetProcAddress(hMFDisp, "MFDisplayEndChar");
      MFDispUninit =
	(DispUninitializeProc*)GetProcAddress(hMFDisp, "MFDisplayUninitialize");
      if (!MFDispInit || !MFDispBeginChar || !MFDispEndChar || !MFDispUninit) {
	win32_console_printf("Could not find the entry point in `%s'.\n",
			     dll_filename);
	FreeLibrary(hMFDisp);
	hMFDisp = NULL;
      }
    }
  }
  if (MFDispInit && MFDispBeginChar && MFDispEndChar && MFDispUninit) {
    HWND hWnd = NULL;
    if (Proc->GetWindow)
      hWnd = (*Proc->GetWindow)(pinfo->hTeX);
    pinfo->MFDispInfo = (*MFDispInit)(hWnd);
    if (pinfo->MFDispInfo) c = 1;
    else {
      win32_console_printf("Could not initialize DLL.\n");
    }
  }
  return c;
}

/* Editor */
void invoke_editor(LPCTSTR filename, DWORD lineno)
{
  BOOL quote_f = FALSE;
  STARTUPINFO si;
  PROCESS_INFORMATION pi;
  LPTSTR editor = win32_get_registry_string(REG_CUI_KEY, REG_EDITOR, NULL);
  LPTSTR command, p;
  int c;
  
  if (editor == NULL) return; /* editor does not exist in the registry. */
  
  p = command = malloc (strlen (editor) + strlen(filename) + 13);
  
  while ((c = *editor++) != 0) {
    if (c == '"') quote_f = !quote_f;
    else if (c == '%') {
      switch (c = *editor++) {
      case 'd':
	sprintf (p, "%u", lineno);
	p += strlen(p);
	break;
      case 's':
	if (!quote_f) *p++ = '"';
	strcpy(p, filename);
	p += strlen(p);
	if (!quote_f) *p++ = '"';
	break;
      case '\0':
	*p++ = '%';
	/* Back up to the null to force termination.  */
	editor--;
	break;
      default:
	*p++ = '%';
	*p++ = c;
	break;
      }
    }
    else *p++ = c;
  }
  *p = '\0';
  
  ZeroMemory(&si, sizeof(STARTUPINFO));
  ZeroMemory(&pi, sizeof(PROCESS_INFORMATION));
  si.cb = sizeof(STARTUPINFO);
  if (!CreateProcess(NULL, command, NULL, NULL, FALSE,
		     CREATE_NEW_CONSOLE, NULL, NULL, &si, &pi))
    win32_console_printf ("! Trouble executing `%s'.\n", command);
  
  free(command);
}

/* Callback function */
static LONG WINAPI callback(TeXDLLMessage msg,
			    DWORD param1, DWORD param2,
			    LPVOID app_data)
{
  LONG c = 0;
  callback_info* pcallback_data = (callback_info*)app_data;
  DWORD dwLen;
      
  switch (msg) {
  case TeXDLL_STDOUT_WRITE:
  case TeXDLL_CONSOLE_WRITE:
  case TeXDLL_CONSOLE_GET:
    return win32_standard_cui_callback(msg, param1, param2, app_data);
  case TeXDLL_INVOKE_EDITOR:
    invoke_editor((LPCTSTR)param1, param2);
    break;
  case TeXDLL_MFDISPLAY_INITIALIZE:
    c = init_mf_display(pcallback_data);
    break;
  case TeXDLL_MFDISPLAY_BEGINCHAR:
    c = (LONG)(*MFDispBeginChar)(pcallback_data->MFDispInfo);
    break;
  case TeXDLL_MFDISPLAY_ENDCHAR:
    (*MFDispEndChar)(pcallback_data->MFDispInfo);
    break;
  case TeXDLL_MFDISPLAY_UNINITIALIZE:
    (*MFDispUninit)(pcallback_data->MFDispInfo);
    break;
  default:
    /* do nothing */
    break;
  }
  return c;
}

/* Get default format/base name */
LPTSTR get_format_name(LPTSTR progname)
{
  LPTSTR base, ext;
  struct _finddata_t fileinfo;
  long hfind;
  if ((hfind = _findfirst(progname, &fileinfo)) != -1) {
    progname = fileinfo.name;
    _findclose(hfind);
  }
  
  base = _mbsrchr(progname, '\\');
  if (base) base++;
  else if (strlen(progname) >= 2 && *(progname+1) == ':')
    base = progname + 2;
  else base = progname;

  ext = _mbsrchr(base, '.');
  if (!ext) ext = base + strlen(base);
  
  *ext = 0;

  return strdup(base);
}

static HWND hMsgWnd;

static void catch_interrupt(int arg)
{
  PostMessage(hMsgWnd, wm_texdll, TeXDLL_MESSAGE_INTERRUPT, 0);
}

static DWORD get_version(LPCSTR filename)
{
  DWORD reserved;
  DWORD apiVer = 0;
  DWORD verInfoSize = GetFileVersionInfoSize(filename, &reserved);
  LPVOID verInfo = malloc(verInfoSize);
  if (verInfo) {
    if (GetFileVersionInfo(filename, 0, verInfoSize, verInfo)) {
      VS_FIXEDFILEINFO* pFileInfo;
      UINT len;
      if (VerQueryValue(verInfo, "\\", &pFileInfo, &len)) {
	apiVer = pFileInfo->dwFileVersionLS & 0xffff;
      }
    }
    free(verInfo);
  }
  return apiVer;
}

typedef struct {
  LPCTeXProcedures Proc;
  LPCTSTR *argv;
  LONG argc;
} ThreadInfo;

UINT WINAPI ThreadFunc(LPVOID ptr)
{
  ThreadInfo* pInfo = (ThreadInfo*)ptr;
  static callback_info callback_data;
  HTeX hTeX;
  UINT code = 0;
  
  hTeX = pInfo->Proc->Initialize(callback, &callback_data, NULL);
  if (hTeX) {
    callback_data.hTeX = hTeX;
    if (Proc->GetWindow) {
      hMsgWnd = pInfo->Proc->GetWindow(hTeX);
      if (hMsgWnd) signal(SIGINT, catch_interrupt);
    }
    code = pInfo->Proc->Main(hTeX, pInfo->argc, pInfo->argv);
    Proc->Uninitialize(hTeX);
  }

  _endthreadex(code);
}

LONG WINAPI
CUITeXMain(LPCTSTR DLLName, LPCTSTR GetProcName, LONG argc, LPTSTR* argv)
{
  LONG code = 1;
  LPFTeXGetProceduresProc GetProcs;
  DWORD api_ver;
  LPCTSTR progpath = argv[0];
  char errbuf[150];
  
  win32_init_cui_callback();
  
  if (!hDLL) {
    char progdir[_MAX_PATH + 1];
    LPTSTR base;
    HINSTANCE hInstance = GetModuleHandle(NULL);
    GetModuleFileName(hInstance, progdir, _MAX_PATH+1);
    base = _mbsrchr(progdir, '\\');
    base[1] = '\0';
    hDLL = win32_load_tex_library(DLLName);
  }
  if (!hDLL) {
    sprintf(errbuf, "Could not load DLL `%s'.\n", DLLName);
    win32_write_stderr_text(errbuf, -1);
    return 1;
  }
  GetProcs = (LPFTeXGetProceduresProc)GetProcAddress(hDLL, GetProcName);
  if (!GetProcs) {
    sprintf(errbuf, "Could not find entry point `%s' in `%s'.\n",
	    GetProcName, DLLName);
    win32_write_stderr_text(errbuf, -1);
    return 1;
  }
  
  Proc = GetProcs();
  api_ver = get_version(Proc->GetLibraryFileName());
  if (api_ver == TeXDLL_API_VERSION) {
    int i;
    LONG debug_repeat_count =
      win32_get_registry_int(REG_CUI_KEY, REG_REPEAT, 1);
    ThreadInfo* pInfo = malloc(sizeof(ThreadInfo));
    char **heap_argv = malloc(sizeof(char*)*argc);
    heap_argv[0] = get_format_name(argv[0]);
    for (i=1;i<argc;i++)
      heap_argv[i] = strdup(argv[i]);
    pInfo->argv = heap_argv;
    pInfo->argc = argc;
    pInfo->Proc = Proc;
    while (debug_repeat_count-- > 0) {
      UINT idThread;
      HANDLE hThread = (HANDLE)_beginthreadex(0, 0, ThreadFunc, pInfo,
					    0, &idThread);
      if (hThread) {
	WaitForSingleObject(hThread, INFINITE);
	GetExitCodeThread(hThread, &code);
	CloseHandle(hThread);
      }
    }
    for (i=0;i<argc;i++)
      free(heap_argv[i]);
    free(heap_argv);
    free(pInfo);
  }
  else {
    sprintf(errbuf, "DLL API mismatch (%lu in DLL, %lu in Application)\n",
	    api_ver, TeXDLL_API_VERSION);
    win32_write_stderr_text(errbuf, -1);
  }
  
  return code;
}
