/*
 * zkfwin32.c --- Kanji font operator for MS-Windows API.
 * Copyright (C) ASAYAMA Kazunori  1997.
 */

#include "math_.h"
#include "memory_.h"
#include "ghost.h"
#include "gp.h"
#include "oper.h"
#include "errors.h"
#include "gspath.h"
#include "gsmatrix.h"
#include "gscoord.h"
#include "gsstate.h"
#include "igstate.h"
#include "stream.h"
#include "files.h"
#include "windows_.h"

#define BOX_SIZE 1000
#define B_X_CL   500 /* horizontal center in BuildChar */
#define B_Y_CL   400 /* vertical center in BuildChar */

/* Forward declaration */
extern int kf_is_vchar(P1(int));
extern int kf_vmatrix(P5(int, floatp, floatp, gs_rect *, gs_matrix *));
private int draw_font(P2(const char *, int));
private int open_font(P2(const char *, HFONT* ));

int
zkfwin32(register os_ptr op)
{
  char *font_name;
  int code, jis_code, wmode;
  
  check_type(op[-2], t_integer);                /* JIS Code */
  check_type(op[-1], t_integer);                /* WMode */
  check_type(op[0], t_string);          /* Font Name */ 
  
  jis_code = op[-2].value.intval;
  wmode = op[-1].value.intval;
  font_name = gs_malloc(r_size(op) + 2, 1, "zkfwin32");
  if (font_name == NULL)
    return e_VMerror;
  
  memcpy(font_name, op->value.bytes, r_size(op));
  font_name[r_size(op)] = 0;
  
  if(wmode && kf_is_vchar(jis_code)) {
    /* vertical kanji character */
    gs_matrix smat, vmat;
    gs_rect bbox;
    if((code = gs_gsave(igs)) >= 0 &&
       (code = draw_font(font_name, jis_code)) >= 0 &&
       (code = gs_flattenpath(igs)) >= 0 &&
       (code = gs_pathbbox(igs, &bbox)) >= 0 &&
       (code = gs_grestore(igs)) >= 0) {
      gs_currentmatrix(igs, &smat);
      kf_vmatrix(jis_code, 
		 (floatp)B_X_CL, (floatp)B_Y_CL, &bbox, &vmat);
      if((code = gs_concat(igs, &vmat)) >= 0 &&
	 (code = draw_font(font_name, jis_code)) >= 0)
	gs_setmatrix(igs, &smat);
    }
  }
  else
    code = draw_font(font_name, jis_code);
  
  gs_free(font_name, r_size(op) + 2, 1, "zkfwin32");
  
  if (code >= 0)
    pop(3);
  
  return code;
}


/* -------- Initialization procedure -------- */

BEGIN_OP_DEFS(zkfwin32_op_defs) {
  {"3kfwin32", zkfwin32},
END_OP_DEFS(0) }

/* -------- Internal routines -------- */
#define fixed2double(p) ( (double)*(LONG*)&(p) / 0x10000 )

/* data coordination -> character coordination */
#define map_x(x) ( fixed2double(x) / BOX_SIZE * 1000 )
#define map_y(y) ( fixed2double(y) / BOX_SIZE * 1000 )

/* drawing */
#define moveto(p) gs_moveto(igs, map_x((p)->x), map_y((p)->y))
#define lineto(p) gs_lineto(igs, map_x((p)->x), map_y((p)->y))
#define curveto(p0, p1, p2) \
  gs_curveto(igs, \
	     (map_x((p0)->x) + map_x((p1)->x)) / 2, \
	     (map_y((p0)->y) + map_y((p1)->y)) / 2, \
             (map_x((p1)->x) + map_x((p2)->x)) / 2, \
	     (map_y((p1)->y) + map_y((p2)->y)) / 2, \
	     map_x((p2)->x), map_y((p2)->y))
#define closepath() gs_closepath(igs)

/* encoding */
#define jistosjis(jis) _mbcjistojms(jis)

/*
 * Open font 
 */

private LOGFONT g_lfPrev;
private HFONT g_hPrevFont = NULL;

private int
open_font(const char *font_name, HFONT* phFont)
{
  LOGFONT lf;
  
  if (g_hPrevFont) {
    if (strcmp(g_lfPrev.lfFaceName, font_name) == 0) {
      *phFont = g_hPrevFont;
      return 0;
    }
    DeleteObject(g_hPrevFont);
  }
  
  lf.lfHeight = - BOX_SIZE;
  lf.lfWidth = 0;
  lf.lfEscapement = 0;
  lf.lfOrientation = 0;
  lf.lfWeight = FW_NORMAL;
  lf.lfItalic = 0;
  lf.lfUnderline = 0;
  lf.lfStrikeOut = 0;
  lf.lfCharSet = SHIFTJIS_CHARSET;
  lf.lfOutPrecision = OUT_TT_ONLY_PRECIS;
  lf.lfClipPrecision = CLIP_TT_ALWAYS;
  lf.lfQuality = PROOF_QUALITY;
  lf.lfPitchAndFamily = DEFAULT_PITCH;
  strcpy(lf.lfFaceName, font_name);
  
  g_hPrevFont = CreateFontIndirect(&lf);
  if (!g_hPrevFont) {
    eprintf1("win32(open_font): Cannot open the font `%s'.\n", font_name);
    return e_ioerror;
  }
  
  *phFont = g_hPrevFont;
  strcpy(g_lfPrev.lfFaceName, font_name);
  
  return 0;
}

private int
draw_contour(TTPOLYGONHEADER *lpHdr)
{
  TTPOLYCURVE *lpCrv = (TTPOLYCURVE*)((LPBYTE)lpHdr + sizeof(TTPOLYGONHEADER));
  TTPOLYCURVE *lpCrvEnd = (TTPOLYCURVE*)((LPBYTE)lpHdr + lpHdr->cb);
  POINTFX pt;
  int code;
  
  if (lpHdr->dwType != TT_POLYGON_TYPE) {
    /* unsupported type of outline */
    return e_unknownerror;
  }
  
  pt = lpHdr->pfxStart;
  if ((code = moveto(&pt)) < 0)
    return code;
  while (lpCrv < lpCrvEnd) {
    if (lpCrv->wType == TT_PRIM_LINE) {
      int i;
      for (i=0;i<lpCrv->cpfx;i++) {
	if ((code = lineto(lpCrv->apfx + i)) < 0)
	  return code;
      }
      pt = lpCrv->apfx[lpCrv->cpfx-1];
    }
    else if (lpCrv->wType == TT_PRIM_QSPLINE) {
      int i;
      for (i=0;i<lpCrv->cpfx - 1;i++) {
	POINTFX ptc, ptn;
	ptc = lpCrv->apfx[i];
	ptn = lpCrv->apfx[i+1];
	if (i + 2 < lpCrv->cpfx) {
	  *(LONG*)&ptn.x = (*(LONG*)&ptc.x + *(LONG*)&ptn.x) / 2;
	  *(LONG*)&ptn.y = (*(LONG*)&ptc.y + *(LONG*)&ptn.y) / 2;
	}
	if ((code = curveto(&pt, &ptc, &ptn)) < 0)
	  return code;
	pt = ptn;
      }
    }
    else {
      /* unknown type of curve */
      return e_unknownerror;
    }
    lpCrv = (TTPOLYCURVE*)(lpCrv->apfx + lpCrv->cpfx);
  }
  return closepath();
}

private int 
draw_font(const char *font_name, int jis)
{
  int code = 0;
  HFONT hFont, hOldFont;
  HWND hWnd = HWND_DESKTOP;
  HDC hDC;
  DWORD len;
  GLYPHMETRICS met;
  MAT2 mat;
  byte *glyph;
  int sjis;
  
  code = open_font(font_name, &hFont);
  if (code < 0) return code;

  hDC = GetDC(hWnd);
  if (!hDC) return e_unknownerror;

  memset(&mat, 0, sizeof(mat));
  mat.eM11.value = 1;
  mat.eM22.value = 1;
  sjis = jistosjis(jis);
  hOldFont = SelectObject(hDC, hFont);
  len = GetGlyphOutline(hDC, sjis, GGO_NATIVE, &met, 0, NULL, &mat);
  glyph = gs_malloc(len, 1, "GetGlyphOutline(draw_font)");
  if (!glyph) {
    code = e_VMerror;
  }
  else {
    int n = GetGlyphOutline(hDC, sjis, GGO_NATIVE, &met, len, glyph, &mat);
    if (n == GDI_ERROR) {
      code = e_unknownerror;
    }
    else {
      LPBYTE p = glyph;
      while (p < glyph + len) {
	code = draw_contour((TTPOLYGONHEADER*)p);
	if (code < 0) break;
	p += ((TTPOLYGONHEADER*)p)->cb;
      }
    }
    gs_free(glyph, len, 1, "GetGlyphOutline(draw_font)");
  }
  SelectObject(hDC, hOldFont);
  
  ReleaseDC(hWnd, hDC);
  
  return code;
}
