#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <malloc.h>
#include <process.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <dos.h>
#include <fcntl.h>
#include <time.h>

/* DJGPP */
#include <stubinfo.h>
#include <crt0.h>
#include <libc/dosio.h>

/* kpathsea */
#include <kpathsea/cnf.h>
#include <kpathsea/lib.h>
#include <kpathsea/progname.h>
#include <kpathsea/tex-file.h>
#include <kpathsea/variable.h>

#define STUBINFO_OFFSET (512 + STUBINFO)
char **__crt0_glob_function(char *a) { return (char **)0; }
int _crt0_startup_flags = _CRT0_FLAG_DISALLOW_RESPONSE_FILES;

#define PROGRAM_VERSION "0.4"

typedef struct{
  char *format_name;
  char *format_extention;
  char *input_extention;
  char *input_environment;
  int pool_format;
  char *example_format;
  char *example_input;
} format_t;

format_t tex_format_info = {
  "TeX formats",
  ".fmt",
  ".tex",
  "TEXINPUTS",
  kpse_texpool_format,
  "tex",
  "plain.tex",
};

format_t mf_base_info = {
  "METAFONT bases",
  ".base",
  ".mf",
  "MFINPUTS",
  kpse_mfpool_format,
  "mf",
  "plain.mf modes.mf",
};

format_t *format_info = &tex_format_info;

#define DEFAULT_MIN_STACK (512L<<10) /* 512KB */
#define MIN_MIN_STACK     (256L<<10) /* 256KB */

typedef struct {
  char *initex;
  char *virtex;
  char *tex;
  char *tex_path;
  char *format;
  char *format_dir;

  char **input;
  int input_count;
} tex_info;

static char *prog,*prog_base;

static char **path_list;

static int gen_only_f = 0;
static int quiet_f = 0;
static char *cwd;
static unsigned long min_stack = DEFAULT_MIN_STACK;

static unsigned char stub[] = {
#  include "stub.c"
};

int yes(void)
{
  union REGS r;
  int a = 0;
  for ( ; ; ) {
    r.h.ah = 0x07;
    intdos(&r,&r);
    if (r.h.al == 'Y' || r.h.al == 'y') {
      a = 1; break;
    }
    if (r.h.al == 'N' || r.h.al == 'n')
      break;
    putc('\a',stderr);
  }
  putc(r.h.al,stdout);
  fflush(stdout);
  return a;
}

void *xmalloc(unsigned int s)
{
  void *p = malloc(s);
  if (p  == NULL) {
    fprintf(stderr,"Memory exhausted.\n");
    exit(1);
  }
  return p;
}

char *xstrdup(const char *s)
{ char *p = xmalloc(strlen(s)+1);
  return strcpy(p,s);
}

char **parse_path(const char *path)
{ const char *p = path;
  const char *q;
  char **path_list;
  int c = 0;
  if (path == NULL) {
    path_list = xmalloc(sizeof(char *));
    path_list[0] = NULL;
    return path_list;
  }
  while (*p) {
    if (*p == ';') c++;
    p++;
  }
  path_list = xmalloc(sizeof(char *) * (c+2));
  p = q = path;
  c = 0;
  while (*p) {
    if (*p == ';') {
      if (p != q) {
	int l = (int)(p-q);
	path_list[c] = xmalloc(l+1);
	strncpy(path_list[c],q,l);
	*(path_list[c]+l) = '\0';
	c++;
      }
      p++;
      q = p;
    }
    else p++;
  }
  if (p != q) {
    int l = (int)(p-q);
    path_list[c] = xmalloc(l+1);
    strncpy(path_list[c],q,l);
    *(path_list[c]+l) = '\0';
    c++;
  }
  path_list[c] = NULL;
  return path_list;
}

#define file_exist(f) (access(f,0) == 0)

char *find_prog(const char *prog,char **path_list)
{
  char **p;

  if (file_exist(prog)) return cwd;
  for (p = path_list;*p;p++) {
    char *f = xmalloc(strlen(*p)+strlen(prog)+2);
    strcpy(f,*p);
    strcat(f,"\\");
    strcat(f,prog);
    if (file_exist(f)) { free(f); return *p; }
    free(f);
  }
  return NULL;
}

void decompose_path(char *path,char **b,char **e)
{
  char *last_slash = NULL;
  char *last_dot = NULL;
  char *p = path;
#ifdef SJIS_FN
  int kanji_f;
#endif
  while (*p) {
#ifdef SJIS_FN
    if (kanji_f) kanji_f = 0;
    else if (IS_KANJI(*p)) kanji_f = 1;
    else
#endif
    if (*p == ':' || *p == '\\' || *p == '/') last_slash = p+1;
    else if (*p == '.') last_dot = p;
    p++;
  }
  if (last_slash == NULL)
    last_slash = path;
  if (last_slash > last_dot)
    last_dot = path + strlen(path);
  if (b) *b = last_slash;
  if (e) *e = last_dot;
}

int GenStub(const char *f)
{
  FILE *fp = fopen(f,"wb");
  int c;
  if (fp == NULL) return -1;
  *(unsigned long *)(stub + STUBINFO_OFFSET + STUBINFO_MINSTACK) = min_stack;
  c = fwrite(stub,sizeof(stub),1,fp);
  fclose(fp);
  return c < 1 ? -1 : 0;
}

int StubEdit(const char *f,char *r,char *a0)
{
  char buf[20];
  FILE *fp;
  fp = fopen(f,"r+b");
  if (fp == NULL) return -1;
  memset(buf,0,8);
  memcpy(buf,r,strlen(r)>8 ? 8 : strlen(r));
  fseek(fp,STUBINFO_OFFSET+STUBINFO_BASENAME,0);
  fwrite(buf,8,1,fp);
  memset(buf,0,16);
  memcpy(buf,a0,strlen(a0)>16 ? 16 : strlen(a0));
  fseek(fp,STUBINFO_OFFSET+STUBINFO_ARGV0,0);
  fwrite(buf,16,1,fp);
  fclose(fp);
  return 0;
}

void chdisk(int n)
{
  union REGS r;
  r.h.ah = 0x0e;
  r.h.dl = n;
  intdos(&r,&r);
}

int MakeFormat(tex_info *pinfo)
{
  FILE *fp;
  char temp_input[32];
  char temp_input_base[32];
  char temp_format[32];
  char temp_log[32];
  char *fmt_disk_cwd;
  int i;
  
  if (strlen(pinfo->format_dir) > 2 &&
      isalpha(pinfo->format_dir[0]) &&
      pinfo->format_dir[1] == ':') {
    chdisk(tolower(pinfo->format_dir[0])-'a');
  }
  fmt_disk_cwd = getcwd(NULL,512);
  chdir(pinfo->format_dir);

  {
    time_t t;
    time(&t); srand(t); t = rand();
    sprintf(temp_input_base,"tmp%05x",(unsigned int)(t&0xfffff));
  }
  strcpy(temp_input,temp_input_base);
  strcat(temp_input,format_info->input_extention);
  strcpy(temp_format,temp_input_base);
  strcat(temp_format,format_info->format_extention);
  strcpy(temp_log,temp_input_base);
  strcat(temp_log,".log");

  fp = fopen(temp_input,"w");
  if (fp == NULL) {
    fprintf(stderr,"cannot open temporary file.\n");
    i = -1;
    goto ex;
  }
  for (i=0;i<pinfo->input_count;i++) {
    char *e;
    fputs("\\input ",fp);
    decompose_path(pinfo->input[i],NULL,&e);
    if (strcmp(e,format_info->input_extention) == 0)
      fwrite(pinfo->input[i],e-pinfo->input[i],1,fp);
    else
      fputs(pinfo->input[i],fp);
    fputs("\n",fp);
  }
  fprintf(fp,"\\dump\n");
  fclose(fp);
  
  {
    int orig_stdin, fd;
    if (quiet_f) {
      orig_stdin = dup(0);
      fd = open("nul",O_RDONLY);
      dup2(fd,0);
      close(fd);
    }
    i = spawnl(P_WAIT,pinfo->tex_path,pinfo->tex_path,
			"\\input ",temp_input_base,NULL);
    if (quiet_f) {
      dup2(orig_stdin,0);
      close(orig_stdin);
    }
  }

  if (i == 0) {
    if (file_exist(pinfo->format)) unlink(pinfo->format);
    rename(temp_format,pinfo->format);
  }
  else if (file_exist(temp_format)) {
    unlink(temp_format);
  }

  unlink(temp_input);

ex:
  if (file_exist(temp_log)) {
    char log[32];
    strcpy(log,pinfo->tex);
    strcat(log,".log");
    if (file_exist(log)) unlink(log);
    rename(temp_log,log);
  }
  chdir(fmt_disk_cwd);
  chdisk(tolower(cwd[0])-'a');
  chdir(cwd);
  return i ? -1 : 0;
}

void usage(void)
{
  FILE *fp = stderr;
  fprintf(fp,"\n%s version " PROGRAM_VERSION "\n",prog_base);
  fprintf(fp,"\nUsage:\n");
  fprintf(fp,
	"  %s [options] <basename> <input-1> [<input-2> ... <input-N>]\n",prog_base);
  fprintf(fp,
	"    Generate %s and executables in the proper directories.\n",
	format_info->format_name);
  fprintf(fp,"  %s [options] -g <basename>\n",prog_base);
  fprintf(fp,"    Generate only executables in the current directory.\n");
  fprintf(fp,"\nOptions:\n");
  fprintf(fp,"  -q               Quiet mode.\n");
  fprintf(fp,"  -s <stack-size>  Amount of stack space.\n");
  fprintf(fp,"\nExamples:\n");
  fprintf(fp,"  %s -s 256K %s %s\n",
	prog_base,format_info->example_format,format_info->example_input);
  fprintf(fp,"  %s -g %s\n",prog_base,format_info->example_format);
  exit(1);
}

static unsigned long a2uint(const char *str)
{
  unsigned long n = 0;
  const char *s = str;
  while (*s) {
    if (isdigit(*s)) n = n * 10 + (*s - '0');
	else break;
	s++;
  }
  if (strcmp("K",s) == 0 || strcmp("k",s) == 0) n <<= 10;
  else if (strcmp("M",s) == 0 || strcmp("m",s) == 0) n <<= 20;
  else if (strcmp("",s) == 0) ;
  else {
    fprintf(stderr,"%s: invalid value.\n",str);
    exit(1);
  }
  return n;
}

int main(int argc,char *argv[])
{
  tex_info info;
  char tex_exe_buf[30];
  char *pool;
  char *tex_exe;
  int i;
  
  prog = argv[0];
  {
    char *prog_ext;
    decompose_path(prog,&prog_base,&prog_ext);
    if (prog_ext-prog_base == 6 && strncmp(prog_base,"dumpmf",6) == 0)
      format_info = &mf_base_info;
  }
  for (i=1;i<argc;i++) {
    char *p = argv[i];
    if (*p != '-') break;
    p++;
    while (*p) {
      switch (*p) {
      case 'g': gen_only_f = 1; break;
      case 'q': quiet_f = 1; break;
	  case 's':
		if (strlen(p) > 1) {
		  min_stack = a2uint(p+1);
		  goto next_arg;
		}
		else if (i < argc-1) {
		  min_stack = a2uint(argv[i+1]);
		  i++;
		}
		else {
		  fprintf(stderr,"-s: requires just one argument.\n");
		  exit(1);
	    }
	    break;
      case 'v': quiet_f = 0; break;
      default:
        fprintf(stderr,"-%c: unknown flag.\n",(int)*p);
        exit(1);
      }
      p++;
    }
    next_arg: ;
  }
  if ((gen_only_f && argc-i != 1) || (!gen_only_f && argc-i < 2)) usage();
  if (min_stack < MIN_MIN_STACK) {
    fprintf(stderr,"%lu: user-specific amount of stack is too small.\n",
		min_stack);
    fprintf(stderr,"`%lu' is used instead.\n",MIN_MIN_STACK);
    min_stack = MIN_MIN_STACK;
  }
  path_list = parse_path(getenv("PATH"));
  info.tex = argv[i++];
  if (!gen_only_f) {
    info.input = argv+i;
    info.input_count = argc-i;
  }
  cwd = getcwd(NULL,512);

  { char *b,*base;
    decompose_path(info.tex,&base,&b);
    if (info.tex != base || strcmp(b,"")) {
      fprintf(stderr,"%s: can not include directory or extension.\n",
		info.tex);
	  exit(1);
    }
    if ((_use_lfn() && strlen(info.tex) > 16) ||
        (!_use_lfn() && strlen(info.tex) > 8)) {
      fprintf(stderr,"%s: too long basename.\n",info.tex);
      exit(1);
    }
    decompose_path(prog,&b,NULL);
    if (strncmp(b,"dump",4)) {
      fprintf(stderr,"%s: invalid program name.\n",prog);
      exit(1);
    }
    b += 4;
    base = xstrdup(b);
    decompose_path(base,NULL,&b);
    *b = '\0';
    info.initex = xmalloc(strlen(base)+8);
    strcpy(info.initex,"ini");
    strcat(info.initex,base);
    strcat(info.initex,".exe");
    info.virtex = xstrdup(info.initex);
    memcpy(info.virtex,"vir",3);
    pool = xmalloc(strlen(base)+6);
    strcpy(pool,base);
    strcat(pool,".pool");

    /* search initex and virtex */
    b = info.initex;
    if (!find_prog(b,path_list)) {
    cannot_find_prog:
      fprintf(stderr,"%s: cannot find.\n",b);
      exit(1);
    }
    b = info.virtex;
    if (!(b = find_prog(b,path_list))) goto cannot_find_prog;

    info.tex_path = xmalloc(strlen(b)+strlen(info.tex)+6);
    strcpy(info.tex_path,b);
    strcat(info.tex_path,"\\");
    strcat(info.tex_path,info.tex);
    strcat(info.tex_path,".exe");

    decompose_path(info.initex,NULL,&b);
    *b = '\0';
    decompose_path(info.virtex,NULL,&b);
    *b = '\0';
  }
  info.format = xmalloc(strlen(info.tex)+6);
  strcpy(info.format,info.tex);
  strcat(info.format,format_info->format_extention);

  strcpy(tex_exe_buf,info.tex);
  strcat(tex_exe_buf,".exe");
  tex_exe = gen_only_f ? tex_exe_buf : info.tex_path;

  signal(SIGINT,SIG_IGN);

  if (!gen_only_f) {
    char *p;
    kpse_set_progname(info.tex);
    p = kpse_find_file(pool,format_info->pool_format,1);
    if (p == NULL) {
      perror(pool);
      exit(1);
    }
    info.format_dir = p;
    decompose_path(info.format_dir,&p,NULL);
    *p = '\0';
    if (strlen(info.format_dir) >= 4 && p[-1] == '\\') *--p = '\0';
  }

  i = GenStub(tex_exe);
  if (i < 0) return 1;

  StubEdit(tex_exe,info.initex,info.tex);

  if (!gen_only_f) {
    char *env;
    env = getenv(format_info->input_environment);
    if (!env) env = kpse_cnf_get(format_info->input_environment);
    if (env) {
      int len = 0;
      char **p,**input_path_list;
      input_path_list = parse_path(kpse_var_expand(env));
      for (p=input_path_list;*p;p++) {
        if (strcmp(*p,".") == 0) *p = cwd;
        len += (strlen(*p) + 1);
      }
      env = xmalloc(len+2);
      strcpy(env,".");
      for (p=input_path_list;*p;p++) {
        strcat(env,";");
        strcat(env,*p);
      }
      xputenv(format_info->input_environment,env);
    }
    i = MakeFormat(&info);
  }

  StubEdit(tex_exe,info.virtex,info.tex);
  
  return i < 0 ? 1 : 0;
}
