/*
  (c) Copyright 1992 Eric Backus

  This software may be used freely so long as this copyright notice is
  left intact.  There is no warrantee on this software.
*/

/*
  Modified by K.Asayama to support filenames containing SHIFT JIS code.
  If you need that feature, define SJIS_FILENAME, else do not.
 */

#define SJIS_FILENAME

#include <dos.h>		/* For intdos() */
#include <errno.h>		/* For errno */
#include <string.h>		/* For strlen() */
#include <ctype.h>		/* for tolower() */

#ifdef SJIS_FILENAME

#define IS_KANJI1(c) \
	( ( (c) >= 0x81 && (c) <= 0x9f) || ( (c) >= 0xe0 && (c) <= 0xfc) )

static int kanji_f;

static char *sjstrchr(str,c)
  char *str;
  int c;
{ unsigned char *s;
  s = (unsigned char *)str;
  kanji_f = 0;
  while (*s != '\0') {
    if (kanji_f) kanji_f = 0;
    else if (IS_KANJI1(*s)) kanji_f = 1;
    else if (*s == c) return (char *)s;
    s++;
  }
  return NULL;
}

static char *sjstrrchr(str,c)
  char *str;
  int c;
{
  char *last = NULL;
  char *x;
  while (*str != '\0') {
    str = sjstrchr(str,c);
    if (str == NULL) return last;
    last = str;
    str++;
  }
  return last;
}

static char *sjstrlwr(str)
  char *str;
{
  unsigned char *s;
  s = (unsigned char *)str;
  kanji_f = 0;
  while (*s != '\0') {
    if (kanji_f) kanji_f = 0;
    else if (IS_KANJI1(*s)) kanji_f = 1;
    else *s = tolower(*s);
    s++;
  }
  return str;
}
#endif

int
_get_default_drive(void)
{
    union REGS	regs;

    regs.h.ah = 0x19;		/* DOS Get Default Drive call */
    regs.h.al = 0;
    (void) intdos(&regs, &regs);
    return regs.h.al;
}

static char *
get_current_directory(char *out, int drive_number)
{
    union REGS	regs;

    regs.h.ah = 0x47;
    regs.h.dl = drive_number + 1;
    regs.x.si = (unsigned long) (out + 1);
    (void) intdos(&regs, &regs);
    if (regs.x.cflag != 0)
    {
	errno = regs.x.ax;
	return out;
    }
    else
    {
	/* Root path, don't insert "/", it'll be added later */
	if (*(out + 1) != '\0')
	    *out = '/';
	else
	    *out = '\0';
	return out + strlen(out);
    }
}

static int
is_slash(int c)
{
    return c == '/' || c == '\\';
}

static int
is_term(int c)
{
    return c == '/' || c == '\\' || c == '\0';
}
  
static int
is_sep(int c)
{
    return c == '/' || c == '\\' || c == '.' || c == '\0';
}

/* Takes as input an arbitrary path.  Fixes up the path by:
   1. Removing consecutive slashes
   2. Removing trailing slashes
   3. Making the path absolute if it wasn't already
   4. Removing "." in the path
   5. Removing ".." entries in the path (and the directory above them)
   6. Adding a drive specification if one wasn't there
   7. Converting all slashes to '/'
   8. Converting to lowercase
   9. Stripping characters that would be ignored by MS-DOS
 */
void
_fixpath(const char *in, char *out)
{
    int		drive_number, count;
    const char	*ip = in;
    char	*op = out;
    char	*p;
#ifdef SJIS_FILENAME
	int kanji_f;
#endif

    /* Add drive specification to output string */
    if (*(ip + 1) == ':' && ((*ip >= 'a' && *ip <= 'z') ||
			     (*ip >= 'A' && *ip <= 'Z')))
    {
	if (*ip >= 'a' && *ip <= 'z')
	    drive_number = *ip - 'a';
	else
	    drive_number = *ip - 'A';
	*op++ = tolower(*ip++);
	*op++ = tolower(*ip++);
    }
    else
    {
	drive_number = _get_default_drive();
	*op++ = drive_number + 'a';
	*op++ = ':';
    }

    /* Convert relative path to absolute */
    if (!is_slash(*ip))
    {
	p = op;
  	op = get_current_directory(op, drive_number);
	/* Convert path to lowercase, in case it wasn't already */
#ifdef SJIS_FILENAME
    sjstrlwr(p);
#else
	while (p < op)
	{
	    *p = tolower(*p);
	    p++;
	}
#endif
    }

    /* Step through the input path */
#ifdef SJIS_FILENAME
    kanji_f = 0;
#endif
    while (*ip != '\0')
    {
	/* Skip input slashes */
	if (is_slash(*ip))
	{
	    ip++;
	    continue;
	}

	/* Skip "." and output nothing */
	if (*ip == '.' && is_term(*(ip + 1)))
	{
	    ip++;
	    continue;
	}

	/* Skip ".." and remove previous output directory */
	if (*ip == '.' && *(ip + 1) == '.' && is_term(*(ip + 2)))
	{
	    ip += 2;
	    /* Don't back up over drive spec */
	    if (op > out + 2)
		/* This requires "/" to follow drive spec */
#ifdef SJIS_FILENAME
      { char *x,*y;
		*op = '\0';
        x = sjstrrchr(out,'/');
        y = sjstrrchr(out,'\\');
        op = x > y ? x : y;
      }
#else
		while (!is_slash(*--op));
#endif
	    continue;
	}

	/* Copy path component from in to out */
	*op++ = '/';
	count = 0;
	/* Copy up to eight characters in filename */
#ifdef SJIS_FILENAME
    kanji_f = 0;
	while ((!is_sep(*ip) || kanji_f) && *ip)
	{
		if (kanji_f) {
		    if (count++ < 8)
				*op++ = *ip;
			kanji_f = 0;
		}
		else if (IS_KANJI1(*(unsigned char *)ip)) {
		    if (count++ < 7)
				*op++ = *ip;
			kanji_f  = 1;
		}
		else {
		    if (count++ < 8)
				*op++ = tolower(*ip);
		}
	    ip++;
	}
#else
	while (!is_sep(*ip))
	{
	    if (count++ < 8)
		*op++ = tolower(*ip);
	    ip++;
	}
#endif
	/* Copy dot */
	if (*ip == '.')
	    *op++ = *ip++;
	count = 0;
	/* Copy up to three characters in extension */
#ifdef SJIS_FILENAME
    kanji_f = 0;
	while ((!is_sep(*ip) || kanji_f) && *ip)
	{
		if (kanji_f) {
		    if (count++ < 3)
				*op++ = *ip;
			kanji_f = 0;
		}
		else if (IS_KANJI1(*(unsigned char *)ip)) {
		    if (count++ < 2)
				*op++ = *ip;
			kanji_f  = 1;
		}
		else {
		    if (count++ < 3)
				*op++ = tolower(*ip);
		}
	    ip++;
	}
#else
	while (!is_sep(*ip))
	{
	    if (count++ < 3)
		*op++ = tolower(*ip);
	    ip++;
	}
#endif
	/* Copy dot */
	if (*ip == '.')
	    *op++ = *ip++;
	/* Skip over extra garbage in the input path */
#ifdef SJIS_FILENAME
	kanji_f = 0;
	while ((!is_term(*ip) || kanji_f) && *ip) {
		if (kanji_f) kanji_f = 0;
		else if (IS_KANJI1(*(unsigned char *)ip)) kanji_f = 1;
		ip++;
	}
#else
	while (!is_term(*ip)) ip++;
#endif
    }

    /* If root directory, insert trailing slash */
    if (op == out + 2) *op++ = '/';

    /* Null terminate the output */
    *op = '\0';
}

#ifdef	TEST
#include <stdio.h>

int
main(int argc, char **argv)
{
    char	path[90];
    int		i;

    for (i = 1; i < argc; i++)
    {
	_fixpath(argv[i], path);
	(void) printf("'%s' -> '%s'\n", argv[i], path);
    }

    return 0;
}
#endif
