/*
    Copyright (C) 2013 Nicola L.C. Talbot
    www.dickimaw-books.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package com.dickimawbooks.texparserlib.latex.lipsum;

import java.io.IOException;
import java.text.CharacterIterator;
import java.text.StringCharacterIterator;

import com.dickimawbooks.texparserlib.*;
import com.dickimawbooks.texparserlib.latex.*;

public class Lipsum extends Command
{
   public Lipsum(LipsumSty sty)
   {
      this("lipsum", sty);
   }

   public Lipsum(String name, LipsumSty sty)
   {
      super(name);
      this.sty = sty;
   }

   public Object clone()
   {
      return new Lipsum(getName(), sty);
   }

   public TeXObjectList expandonce(TeXParser parser)
     throws IOException
   {
      return expandonce(parser, parser);
   }

   public TeXObjectList expandonce(TeXParser parser, TeXObjectList stack)
     throws IOException
   {
      TeXObject next = (parser == stack ? parser.popStack() : stack.pop());

      boolean isStar = false;

      if (next != null && next instanceof CharObject
       && ((CharObject)next).getCharCode() == (int)'*')
      {
         isStar = true;
      }
      else
      {
         stack.push(next);
      }

      TeXObject optArg = (parser == stack ?
       parser.popNextArg('[', ']') :
       stack.popArg(parser, '[', ']'));

      TeXObjectList expanded = null;
      String refs;

      if (optArg != null)
      {
         if (optArg instanceof Expandable)
         {
            if (parser == stack)
            {
               expanded = ((Expandable)optArg).expandfully(parser);
            }
            else
            {
               expanded = ((Expandable)optArg).expandfully(parser, stack);
            }
         }

         if (expanded == null)
         {
            refs = optArg.toString(parser);
         }
         else
         {
            refs = expanded.toString(parser);
         }
      }
      else
      {
         refs = sty.getDefaultRange();
      }

      expanded = new TeXObjectList();

      String[] split = refs.split(",");

      for (int j = 0; j < split.length; j++)
      {
         String[] range = split[j].split("-");

         if (range == null || range.length == 0)
         {
            continue;
         }

         try
         {
            int firstIdx = Integer.parseInt(range[0]);

            int lastIdx = (range.length == 1 ? firstIdx : 
              Integer.parseInt(range[1]));

            for (int i = firstIdx-1; i < lastIdx; i++)
            {
               expanded.addAll(
                  parser.getListener().createString(LIPSUM_TEXT[i]));

               if (!isStar)
               {
                  expanded.add(new TeXCsRef("par"));
               }
            }
         }
         catch (NumberFormatException e)
         {
            throw new TeXSyntaxException(parser,
              TeXSyntaxException.ERROR_NUMBER_EXPECTED);
         }
      }

      return expanded;
   }

   private void lipsum(TeXParser parser, TeXObjectList stack,
     TeXObject optArg, boolean isStar)
     throws IOException
   {
      String refs;

      if (optArg != null)
      {
         TeXObjectList expanded = null;

         if (optArg instanceof Expandable)
         {
            if (parser == stack)
            {
               expanded = ((Expandable)optArg).expandfully(parser);
            }
            else
            {
               expanded = ((Expandable)optArg).expandfully(parser, stack);
            }
         }

         if (expanded == null)
         {
            refs = optArg.toString(parser);
         }
         else
         {
            refs = expanded.toString(parser);
         }
      }
      else
      {
         refs = sty.getDefaultRange();
      }

      Writeable writeable = parser.getListener().getWriteable();

      String[] split = refs.split(",");

      for (int j = 0; j < split.length; j++)
      {
         String[] range = split[j].split("-");

         if (range == null || range.length == 0)
         {
            continue;
         }

         try
         {
            int firstIdx = Integer.parseInt(range[0]);

            int lastIdx = (range.length == 1 ? firstIdx : 
              Integer.parseInt(range[1]));

            for (int i = firstIdx-1; i < lastIdx; i++)
            {
               writeable.write(LIPSUM_TEXT[i]);

               if (!isStar)
               {
                  parser.getListener().getPar().process(parser);
               }
            }
         }
         catch (NumberFormatException e)
         {
            throw new TeXSyntaxException(parser,
              TeXSyntaxException.ERROR_NUMBER_EXPECTED);
         }
      }

   }

   public void process(TeXParser parser, TeXObjectList list)
     throws IOException
   {
      TeXObject next = list.pop();

      boolean isStar = false;

      if (next != null && next instanceof CharObject
       && ((CharObject)next).getCharCode() == (int)'*')
      {
         isStar = true;
      }
      else
      {
         list.push(next);
      }

      TeXObject optArg = list.popArg(parser, '[', ']');

      lipsum(parser, list, optArg, isStar);
   }

   public void process(TeXParser parser)
     throws IOException
   {
      TeXObject next = parser.popStack();

      boolean isStar = false;

      if (next != null && next instanceof CharObject
       && ((CharObject)next).getCharCode() == (int)'*')
      {
         isStar = true;
      }
      else if (next != null)
      {
         parser.push(next);
      }

      TeXObject optArg = parser.popNextArg('[', ']');

      lipsum(parser, parser, optArg, isStar);
   }

   private LipsumSty sty;

   public static final String[] LIPSUM_TEXT = new String[]
   {
      "Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Ut purus elit, vestibulum ut, placerat ac, adipiscing vitae, felis. Curabitur dictum gravida mauris. Nam arcu libero, nonummy eget, consectetuer id, vulputate a, magna. Donec vehicula augue eu neque. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Mauris ut leo. Cras viverra metus rhoncus sem. Nulla et lectus vestibulum urna fringilla ultrices.  Phasellus eu tellus sit amet tortor gravida placerat. Integer sapien est, iaculis in, pretium quis, viverra ac, nunc. Praesent eget sem vel leo ultrices bibendum. Aenean faucibus.  Morbi dolor nulla, malesuada eu, pulvinar at, mollis ac, nulla.  Curabitur auctor semper nulla. Donec varius orci eget risus. Duis nibh mi, congue eu, accumsan eleifend, sagittis quis, diam. Duis eget orci sit amet orci dignissim rutrum.",
      "Nam dui ligula, fringilla a, euismod sodales, sollicitudin vel, wisi. Morbi auctor lorem non justo. Nam lacus libero, pretium at, lobortis vitae, ultricies et, tellus. Donec aliquet, tortor sed accumsan bibendum, erat ligula aliquet magna, vitae ornare odio metus a mi. Morbi ac orci et nisl hendrerit mollis. Suspendisse ut massa. Cras nec ante. Pellentesque a nulla.  Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Aliquam tincidunt urna. Nulla ullamcorper vestibulum turpis. Pellentesque cursus luctus mauris.",
      "Nulla malesuada porttitor diam. Donec felis erat, congue non, volutpat at, tincidunt tristique, libero. Vivamus viverra fermentum felis. Donec nonummy pellentesque ante. Phasellus adipiscing semper elit. Proin fermentum massa ac quam. Sed diam turpis, molestie vitae, placerat a, molestie nec, leo. Maecenas lacinia. Nam ipsum ligula, eleifend at, accumsan nec, suscipit a, ipsum. Morbi blandit ligula feugiat magna. Nunc eleifend consequat lorem. Sed lacinia nulla vitae enim. Pellentesque tincidunt purus vel magna. Integer non enim. Praesent euismod nunc eu purus. Donec bibendum quam in tellus. Nullam cursus pulvinar lectus. Donec et mi.  Nam vulputate metus eu enim. Vestibulum pellentesque felis eu massa.",
      "Quisque ullamcorper placerat ipsum. Cras nibh.  Morbi vel justo vitae lacus tincidunt ultrices. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. In hac habitasse platea dictumst. Integer tempus convallis augue. Etiam facilisis. Nunc elementum fermentum wisi. Aenean placerat. Ut imperdiet, enim sed gravida sollicitudin, felis odio placerat quam, ac pulvinar elit purus eget enim. Nunc vitae tortor. Proin tempus nibh sit amet nisl.  Vivamus quis tortor vitae risus porta vehicula.",
      "Fusce mauris. Vestibulum luctus nibh at lectus.  Sed bibendum, nulla a faucibus semper, leo velit ultricies tellus, ac venenatis arcu wisi vel nisl. Vestibulum diam. Aliquam pellentesque, augue quis sagittis posuere, turpis lacus congue quam, in hendrerit risus eros eget felis. Maecenas eget erat in sapien mattis porttitor. Vestibulum porttitor. Nulla facilisi. Sed a turpis eu lacus commodo facilisis. Morbi fringilla, wisi in dignissim interdum, justo lectus sagittis dui, et vehicula libero dui cursus dui. Mauris tempor ligula sed lacus. Duis cursus enim ut augue. Cras ac magna. Cras nulla. Nulla egestas. Curabitur a leo. Quisque egestas wisi eget nunc. Nam feugiat lacus vel est. Curabitur consectetuer.",
      "Suspendisse vel felis. Ut lorem lorem, interdum eu, tincidunt sit amet, laoreet vitae, arcu. Aenean faucibus pede eu ante. Praesent enim elit, rutrum at, molestie non, nonummy vel, nisl. Ut lectus eros, malesuada sit amet, fermentum eu, sodales cursus, magna. Donec eu purus. Quisque vehicula, urna sed ultricies auctor, pede lorem egestas dui, et convallis elit erat sed nulla.  Donec luctus. Curabitur et nunc. Aliquam dolor odio, commodo pretium, ultricies non, pharetra in, velit. Integer arcu est, nonummy in, fermentum faucibus, egestas vel, odio.",
     "Sed commodo posuere pede. Mauris ut est. Ut quis purus. Sed ac odio. Sed vehicula hendrerit sem. Duis non odio.  Morbi ut dui. Sed accumsan risus eget odio. In hac habitasse platea dictumst. Pellentesque non elit. Fusce sed justo eu urna porta tincidunt. Mauris felis odio, sollicitudin sed, volutpat a, ornare ac, erat. Morbi quis dolor. Donec pellentesque, erat ac sagittis semper, nunc dui lobortis purus, quis congue purus metus ultricies tellus. Proin et quam. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Praesent sapien turpis, fermentum vel, eleifend faucibus, vehicula eu, lacus.",
      "Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Donec odio elit, dictum in, hendrerit sit amet, egestas sed, leo. Praesent feugiat sapien aliquet odio. Integer vitae justo. Aliquam vestibulum fringilla lorem. Sed neque lectus, consectetuer at, consectetuer sed, eleifend ac, lectus. Nulla facilisi. Pellentesque eget lectus.  Proin eu metus. Sed porttitor. In hac habitasse platea dictumst.  Suspendisse eu lectus. Ut mi mi, lacinia sit amet, placerat et, mollis vitae, dui. Sed ante tellus, tristique ut, iaculis eu, malesuada ac, dui.  Mauris nibh leo, facilisis non, adipiscing quis, ultrices a, dui.",
      "Morbi luctus, wisi viverra faucibus pretium, nibh est placerat odio, nec commodo wisi enim eget quam. Quisque libero justo, consectetuer a, feugiat vitae, porttitor eu, libero.  Suspendisse sed mauris vitae elit sollicitudin malesuada. Maecenas ultricies eros sit amet ante. Ut venenatis velit. Maecenas sed mi eget dui varius euismod. Phasellus aliquet volutpat odio. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Pellentesque sit amet pede ac sem eleifend consectetuer. Nullam elementum, urna vel imperdiet sodales, elit ipsum pharetra ligula, ac pretium ante justo a nulla. Curabitur tristique arcu eu metus. Vestibulum lectus. Proin mauris. Proin eu nunc eu urna hendrerit faucibus. Aliquam auctor, pede consequat laoreet varius, eros tellus scelerisque quam, pellentesque hendrerit ipsum dolor sed augue. Nulla nec lacus.",
      "Suspendisse vitae elit. Aliquam arcu neque, ornare in, ullamcorper quis, commodo eu, libero. Fusce sagittis erat at erat tristique mollis. Maecenas sapien libero, molestie et, lobortis in, sodales eget, dui. Morbi ultrices rutrum lorem. Nam elementum ullamcorper leo. Morbi dui. Aliquam sagittis. Nunc placerat.  Pellentesque tristique sodales est. Maecenas imperdiet lacinia velit. Cras non urna. Morbi eros pede, suscipit ac, varius vel, egestas non, eros. Praesent malesuada, diam id pretium elementum, eros sem dictum tortor, vel consectetuer odio sem sed wisi.",
      "Sed feugiat. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Ut pellentesque augue sed urna. Vestibulum diam eros, fringilla et, consectetuer eu, nonummy id, sapien. Nullam at lectus. In sagittis ultrices mauris. Curabitur malesuada erat sit amet massa. Fusce blandit. Aliquam erat volutpat.  Aliquam euismod. Aenean vel lectus.  Nunc imperdiet justo nec dolor.",
    "Etiam euismod. Fusce facilisis lacinia dui.  Suspendisse potenti. In mi erat, cursus id, nonummy sed, ullamcorper eget, sapien. Praesent pretium, magna in eleifend egestas, pede pede pretium lorem, quis consectetuer tortor sapien facilisis magna.  Mauris quis magna varius nulla scelerisque imperdiet. Aliquam non quam. Aliquam porttitor quam a lacus. Praesent vel arcu ut tortor cursus volutpat. In vitae pede quis diam bibendum placerat. Fusce elementum convallis neque. Sed dolor orci, scelerisque ac, dapibus nec, ultricies ut, mi. Duis nec dui quis leo sagittis commodo.",
     "Aliquam lectus. Vivamus leo. Quisque ornare tellus ullamcorper nulla. Mauris porttitor pharetra tortor. Sed fringilla justo sed mauris. Mauris tellus. Sed non leo. Nullam elementum, magna in cursus sodales, augue est scelerisque sapien, venenatis congue nulla arcu et pede. Ut suscipit enim vel sapien.  Donec congue. Maecenas urna mi, suscipit in, placerat ut, vestibulum ut, massa. Fusce ultrices nulla et nisl.",
      "Etiam ac leo a risus tristique nonummy. Donec dignissim tincidunt nulla. Vestibulum rhoncus molestie odio. Sed lobortis, justo et pretium lobortis, mauris turpis condimentum augue, nec ultricies nibh arcu pretium enim. Nunc purus neque, placerat id, imperdiet sed, pellentesque nec, nisl. Vestibulum imperdiet neque non sem accumsan laoreet. In hac habitasse platea dictumst. Etiam condimentum facilisis libero. Suspendisse in elit quis nisl aliquam dapibus. Pellentesque auctor sapien. Sed egestas sapien nec lectus. Pellentesque vel dui vel neque bibendum viverra.  Aliquam porttitor nisl nec pede. Proin mattis libero vel turpis.  Donec rutrum mauris et libero. Proin euismod porta felis. Nam lobortis, metus quis elementum commodo, nunc lectus elementum mauris, eget vulputate ligula tellus eu neque. Vivamus eu dolor.",
      "Nulla in ipsum. Praesent eros nulla, congue vitae, euismod ut, commodo a, wisi. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.  Aenean nonummy magna non leo. Sed felis erat, ullamcorper in, dictum non, ultricies ut, lectus. Proin vel arcu a odio lobortis euismod.  Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Proin ut est. Aliquam odio. Pellentesque massa turpis, cursus eu, euismod nec, tempor congue, nulla. Duis viverra gravida mauris. Cras tincidunt. Curabitur eros ligula, varius ut, pulvinar in, cursus faucibus, augue.",
      "Nulla mattis luctus nulla. Duis commodo velit at leo.  Aliquam vulputate magna et leo. Nam vestibulum ullamcorper leo.  Vestibulum condimentum rutrum mauris. Donec id mauris. Morbi molestie justo et pede. Vivamus eget turpis sed nisl cursus tempor.  Curabitur mollis sapien condimentum nunc. In wisi nisl, malesuada at, dignissim sit amet, lobortis in, odio. Aenean consequat arcu a ante. Pellentesque porta elit sit amet orci. Etiam at turpis nec elit ultricies imperdiet. Nulla facilisi. In hac habitasse platea dictumst. Suspendisse viverra aliquam risus. Nullam pede justo, molestie nonummy, scelerisque eu, facilisis vel, arcu.",
      "Curabitur tellus magna, porttitor a, commodo a, commodo in, tortor. Donec interdum. Praesent scelerisque.  Maecenas posuere sodales odio. Vivamus metus lacus, varius quis, imperdiet quis, rhoncus a, turpis. Etiam ligula arcu, elementum a, venenatis quis, sollicitudin sed, metus. Donec nunc pede, tincidunt in, venenatis vitae, faucibus vel, nibh. Pellentesque wisi. Nullam malesuada. Morbi ut tellus ut pede tincidunt porta. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Etiam congue neque id dolor.",
      "Donec et nisl at wisi luctus bibendum. Nam interdum tellus ac libero. Sed sem justo, laoreet vitae, fringilla at, adipiscing ut, nibh. Maecenas non sem quis tortor eleifend fermentum. Etiam id tortor ac mauris porta vulputate. Integer porta neque vitae massa. Maecenas tempus libero a libero posuere dictum.  Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Aenean quis mauris sed elit commodo placerat.  Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Vivamus rhoncus tincidunt libero. Etiam elementum pretium justo. Vivamus est. Morbi a tellus eget pede tristique commodo. Nulla nisl. Vestibulum sed nisl eu sapien cursus rutrum.",
      "Nulla non mauris vitae wisi posuere convallis.  Sed eu nulla nec eros scelerisque pharetra. Nullam varius. Etiam dignissim elementum metus. Vestibulum faucibus, metus sit amet mattis rhoncus, sapien dui laoreet odio, nec ultricies nibh augue a enim. Fusce in ligula. Quisque at magna et nulla commodo consequat.  Proin accumsan imperdiet sem. Nunc porta. Donec feugiat mi at justo.  Phasellus facilisis ipsum quis ante. In ac elit eget ipsum pharetra faucibus.  Maecenas viverra nulla in massa.",
     "Nulla ac nisl. Nullam urna nulla, ullamcorper in, interdum sit amet, gravida ut, risus. Aenean ac enim. In luctus.  Phasellus eu quam vitae turpis viverra pellentesque. Duis feugiat felis ut enim. Phasellus pharetra, sem id porttitor sodales, magna nunc aliquet nibh, nec blandit nisl mauris at pede. Suspendisse risus risus, lobortis eget, semper at, imperdiet sit amet, quam.  Quisque scelerisque dapibus nibh. Nam enim. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Nunc ut metus. Ut metus justo, auctor at, ultrices eu, sagittis ut, purus. Aliquam aliquam.",
      "Etiam pede massa, dapibus vitae, rhoncus in, placerat posuere, odio. Vestibulum luctus commodo lacus. Morbi lacus dui, tempor sed, euismod eget, condimentum at, tortor. Phasellus aliquet odio ac lacus tempor faucibus. Praesent sed sem. Praesent iaculis.  Cras rhoncus tellus sed justo ullamcorper sagittis. Donec quis orci.  Sed ut tortor quis tellus euismod tincidunt. Suspendisse congue nisl eu elit. Aliquam tortor diam, tempus id, tristique eget, sodales vel, nulla. Praesent tellus mi, condimentum sed, viverra at, consectetuer quis, lectus. In auctor vehicula orci. Sed pede sapien, euismod in, suscipit in, pharetra placerat, metus. Vivamus commodo dui non odio. Donec et felis.",
      "Etiam suscipit aliquam arcu. Aliquam sit amet est ac purus bibendum congue. Sed in eros. Morbi non orci.  Pellentesque mattis lacinia elit. Fusce molestie velit in ligula.  Nullam et orci vitae nibh vulputate auctor. Aliquam eget purus.  Nulla auctor wisi sed ipsum. Morbi porttitor tellus ac enim. Fusce ornare. Proin ipsum enim, tincidunt in, ornare venenatis, molestie a, augue. Donec vel pede in lacus sagittis porta. Sed hendrerit ipsum quis nisl.  Suspendisse quis massa ac nibh pretium cursus. Sed sodales. Nam eu neque quis pede dignissim ornare. Maecenas eu purus ac urna tincidunt congue.",
      "Donec et nisl id sapien blandit mattis.  Aenean dictum odio sit amet risus. Morbi purus. Nulla a est sit amet purus venenatis iaculis. Vivamus viverra purus vel magna. Donec in justo sed odio malesuada dapibus. Nunc ultrices aliquam nunc.  Vivamus facilisis pellentesque velit. Nulla nunc velit, vulputate dapibus, vulputate id, mattis ac, justo. Nam mattis elit dapibus purus.  Quisque enim risus, congue non, elementum ut, mattis quis, sem.  Quisque elit.",
      "Maecenas non massa. Vestibulum pharetra nulla at lorem. Duis quis quam id lacus dapibus interdum. Nulla lorem.  Donec ut ante quis dolor bibendum condimentum. Etiam egestas tortor vitae lacus. Praesent cursus. Mauris bibendum pede at elit. Morbi et felis a lectus interdum facilisis. Sed suscipit gravida turpis.  Nulla at lectus. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Praesent nonummy luctus nibh. Proin turpis nunc, congue eu, egestas ut, fringilla at, tellus. In hac habitasse platea dictumst.",
      "Vivamus eu tellus sed tellus consequat suscipit. Nam orci orci, malesuada id, gravida nec, ultricies vitae, erat. Donec risus turpis, luctus sit amet, interdum quis, porta sed, ipsum.  Suspendisse condimentum, tortor at egestas posuere, neque metus tempor orci, et tincidunt urna nunc a purus. Sed facilisis blandit tellus. Nunc risus sem, suscipit nec, eleifend quis, cursus quis, libero. Curabitur et dolor. Sed vitae sem. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus.  Maecenas ante. Duis ullamcorper enim. Donec tristique enim eu leo.  Nullam molestie elit eu dolor. Nullam bibendum, turpis vitae tristique gravida, quam sapien tempor lectus, quis pretium tellus purus ac quam. Nulla facilisi.",
      "Duis aliquet dui in est. Donec eget est. Nunc lectus odio, varius at, fermentum in, accumsan non, enim. Aliquam erat volutpat. Proin sit amet nulla ut eros consectetuer cursus.  Phasellus dapibus aliquam justo. Nunc laoreet. Donec consequat placerat magna. Duis pretium tincidunt justo. Sed sollicitudin vestibulum quam. Nam quis ligula. Vivamus at metus. Etiam imperdiet imperdiet pede. Aenean turpis. Fusce augue velit, scelerisque sollicitudin, dictum vitae, tempor et, pede. Donec wisi sapien, feugiat in, fermentum ut, sollicitudin adipiscing, metus.",
      "Donec vel nibh ut felis consectetuer laoreet.  Donec pede. Sed id quam id wisi laoreet suscipit. Nulla lectus dolor, aliquam ac, fringilla eget, mollis ut, orci. In pellentesque justo in ligula. Maecenas turpis. Donec eleifend leo at felis tincidunt consequat. Aenean turpis metus, malesuada sed, condimentum sit amet, auctor a, wisi. Pellentesque sapien elit, bibendum ac, posuere et, congue eu, felis. Vestibulum mattis libero quis metus scelerisque ultrices. Sed purus.",
      "Donec molestie, magna ut luctus ultrices, tellus arcu nonummy velit, sit amet pulvinar elit justo et mauris.  In pede.  Maecenas euismod elit eu erat. Aliquam augue wisi, facilisis congue, suscipit in, adipiscing et, ante. In justo. Cras lobortis neque ac ipsum. Nunc fermentum massa at ante. Donec orci tortor, egestas sit amet, ultrices eget, venenatis eget, mi.  Maecenas vehicula leo semper est. Mauris vel metus. Aliquam erat volutpat. In rhoncus sapien ac tellus. Pellentesque ligula.",
      "Cras dapibus, augue quis scelerisque ultricies, felis dolor placerat sem, id porta velit odio eu elit.  Aenean interdum nibh sed wisi. Praesent sollicitudin vulputate dui.  Praesent iaculis viverra augue. Quisque in libero. Aenean gravida lorem vitae sem ullamcorper cursus. Nunc adipiscing rutrum ante.  Nunc ipsum massa, faucibus sit amet, viverra vel, elementum semper, orci. Cras eros sem, vulputate et, tincidunt id, ultrices eget, magna. Nulla varius ornare odio. Donec accumsan mauris sit amet augue. Sed ligula lacus, laoreet non, aliquam sit amet, iaculis tempor, lorem. Suspendisse eros. Nam porta, leo sed congue tempor, felis est ultrices eros, id mattis velit felis non metus. Curabitur vitae elit non mauris varius pretium. Aenean lacus sem, tincidunt ut, consequat quis, porta vitae, turpis. Nullam laoreet fermentum urna. Proin iaculis lectus.",
      "Sed mattis, erat sit amet gravida malesuada, elit augue egestas diam, tempus scelerisque nunc nisl vitae libero.  Sed consequat feugiat massa. Nunc porta, eros in eleifend varius, erat leo rutrum dui, non convallis lectus orci ut nibh. Sed lorem massa, nonummy quis, egestas id, condimentum at, nisl. Maecenas at nibh.  Aliquam et augue at nunc pellentesque ullamcorper. Duis nisl nibh, laoreet suscipit, convallis ut, rutrum id, enim. Phasellus odio.  Nulla nulla elit, molestie non, scelerisque at, vestibulum eu, nulla. Ut odio nisl, facilisis id, mollis et, scelerisque nec, enim.  Aenean sem leo, pellentesque sit amet, scelerisque sit amet, vehicula pellentesque, sapien.",
      "Sed consequat tellus et tortor. Ut tempor laoreet quam. Nullam id wisi a libero tristique semper. Nullam nisl massa, rutrum ut, egestas semper, mollis id, leo. Nulla ac massa eu risus blandit mattis. Mauris ut nunc. In hac habitasse platea dictumst.  Aliquam eget tortor. Quisque dapibus pede in erat. Nunc enim. In dui nulla, commodo at, consectetuer nec, malesuada nec, elit. Aliquam ornare tellus eu urna. Sed nec metus. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus.  Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.",
      "Phasellus id magna. Duis malesuada interdum arcu.  Integer metus. Morbi pulvinar pellentesque mi. Suspendisse sed est eu magna molestie egestas. Quisque mi lorem, pulvinar eget, egestas quis, luctus at, ante. Proin auctor vehicula purus. Fusce ac nisl aliquam ante hendrerit pellentesque. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Morbi wisi. Etiam arcu mauris, facilisis sed, eleifend non, nonummy ut, pede. Cras ut lacus tempor metus mollis placerat.  Vivamus eu tortor vel metus interdum malesuada.",
      "Sed eleifend, eros sit amet faucibus elementum, urna sapien consectetuer mauris, quis egestas leo justo non risus.  Morbi non felis ac libero vulputate fringilla. Mauris libero eros, lacinia non, sodales quis, dapibus porttitor, pede.  Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Morbi dapibus mauris condimentum nulla. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Etiam sit amet erat. Nulla varius. Etiam tincidunt dui vitae turpis. Donec leo. Morbi vulputate convallis est. Integer aliquet. Pellentesque aliquet sodales urna.",
      "Nullam eleifend justo in nisl. In hac habitasse platea dictumst. Morbi nonummy. Aliquam ut felis. In velit leo, dictum vitae, posuere id, vulputate nec, ante. Maecenas vitae pede nec dui dignissim suscipit. Morbi magna. Vestibulum id purus eget velit laoreet laoreet. Praesent sed leo vel nibh convallis blandit.  Ut rutrum. Donec nibh. Donec interdum. Fusce sed pede sit amet elit rhoncus ultrices. Nullam at enim vitae pede vehicula iaculis.",
      "Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Aenean nonummy turpis id odio. Integer euismod imperdiet turpis. Ut nec leo nec diam imperdiet lacinia. Etiam eget lacus eget mi ultricies posuere.  In placerat tristique tortor. Sed porta vestibulum metus. Nulla iaculis sollicitudin pede. Fusce luctus tellus in dolor. Curabitur auctor velit a sem. Morbi sapien. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Donec adipiscing urna vehicula nunc. Sed ornare leo in leo. In rhoncus leo ut dui. Aenean dolor quam, volutpat nec, fringilla id, consectetuer vel, pede.",
      "Nulla malesuada risus ut urna. Aenean pretium velit sit amet metus. Duis iaculis. In hac habitasse platea dictumst.  Nullam molestie turpis eget nisl. Duis a massa id pede dapibus ultricies. Sed eu leo. In at mauris sit amet tortor bibendum varius.  Phasellus justo risus, posuere in, sagittis ac, varius vel, tortor.  Quisque id enim. Phasellus consequat, libero pretium nonummy fringilla, tortor lacus vestibulum nunc, ut rhoncus ligula neque id justo. Nullam accumsan euismod nunc. Proin vitae ipsum ac metus dictum tempus. Nam ut wisi. Quisque tortor felis, interdum ac, sodales a, semper a, sem. Curabitur in velit sit amet dui tristique sodales. Vivamus mauris pede, lacinia eget, pellentesque quis, scelerisque eu, est. Aliquam risus. Quisque bibendum pede eu dolor.",
      "Donec tempus neque vitae est. Aenean egestas odio sed risus ullamcorper ullamcorper. Sed in nulla a tortor tincidunt egestas. Nam sapien tortor, elementum sit amet, aliquam in, porttitor faucibus, enim. Nullam congue suscipit nibh. Quisque convallis. Praesent arcu nibh, vehicula eget, accumsan eu, tincidunt a, nibh. Suspendisse vulputate, tortor quis adipiscing viverra, lacus nibh dignissim tellus, eu suscipit risus ante fringilla diam.  Quisque a libero vel pede imperdiet aliquet. Pellentesque nunc nibh, eleifend a, consequat consequat, hendrerit nec, diam. Sed urna.  Maecenas laoreet eleifend neque. Vivamus purus odio, eleifend non, iaculis a, ultrices sit amet, urna. Mauris faucibus odio vitae risus. In nisl. Praesent purus. Integer iaculis, sem eu egestas lacinia, lacus pede scelerisque augue, in ullamcorper dolor eros ac lacus. Nunc in libero.",
      "Fusce suscipit cursus sem. Vivamus risus mi, egestas ac, imperdiet varius, faucibus quis, leo. Aenean tincidunt.  Donec suscipit. Cras id justo quis nibh scelerisque dignissim.  Aliquam sagittis elementum dolor. Aenean consectetuer justo in pede.  Curabitur ullamcorper ligula nec orci. Aliquam purus turpis, aliquam id, ornare vitae, porttitor non, wisi. Maecenas luctus porta lorem.  Donec vitae ligula eu ante pretium varius.  Proin tortor metus, convallis et, hendrerit non, scelerisque in, urna. Cras quis libero eu ligula bibendum tempor. Vivamus tellus quam, malesuada eu, tempus sed, tempor sed, velit. Donec lacinia auctor libero.",
      "Praesent sed neque id pede mollis rutrum.  Vestibulum iaculis risus. Pellentesque lacus. Ut quis nunc sed odio malesuada egestas. Duis a magna sit amet ligula tristique pretium.  Ut pharetra. Vestibulum imperdiet magna nec wisi. Mauris convallis.  Sed accumsan sollicitudin massa. Sed id enim. Nunc pede enim, lacinia ut, pulvinar quis, suscipit semper, elit. Cras accumsan erat vitae enim. Cras sollicitudin. Vestibulum rutrum blandit massa.",
      "Sed gravida lectus ut purus. Morbi laoreet magna.  Pellentesque eu wisi. Proin turpis. Integer sollicitudin augue nec dui. Fusce lectus. Vivamus faucibus nulla nec lacus.  Integer diam.  Pellentesque sodales, enim feugiat cursus volutpat, sem mauris dignissim mauris, quis consequat sem est fermentum ligula. Nullam justo lectus, condimentum sit amet, posuere a, fringilla mollis, felis. Morbi nulla nibh, pellentesque at, nonummy eu, sollicitudin nec, ipsum. Cras neque. Nunc augue. Nullam vitae quam id quam pulvinar blandit. Nunc sit amet orci. Aliquam erat elit, pharetra nec, aliquet a, gravida in, mi. Quisque urna enim, viverra quis, suscipit quis, tincidunt ut, sapien. Cras placerat consequat sem.  Curabitur ac diam. Curabitur diam tortor, mollis et, viverra ac, tempus vel, metus.",
      "Curabitur ac lorem. Vivamus non justo in dui mattis posuere. Etiam accumsan ligula id pede. Maecenas tincidunt diam nec velit. Praesent convallis sapien ac est. Aliquam ullamcorper euismod nulla. Integer mollis enim vel tortor. Nulla sodales placerat nunc.  Sed tempus rutrum wisi. Duis accumsan gravida purus. Nunc nunc.  Etiam facilisis dui eu sem. Vestibulum semper. Praesent eu eros.  Vestibulum tellus nisl, dapibus id, vestibulum sit amet, placerat ac, mauris. Maecenas et elit ut erat placerat dictum. Nam feugiat, turpis et sodales volutpat, wisi quam rhoncus neque, vitae aliquam ipsum sapien vel enim. Maecenas suscipit cursus mi.",
      "Quisque consectetuer. In suscipit mauris a dolor pellentesque consectetuer. Mauris convallis neque non erat. In lacinia. Pellentesque leo eros, sagittis quis, fermentum quis, tincidunt ut, sapien. Maecenas sem. Curabitur eros odio, interdum eu, feugiat eu, porta ac, nisl. Curabitur nunc. Etiam fermentum convallis velit. Pellentesque laoreet lacus. Quisque sed elit. Nam quis tellus. Aliquam tellus arcu, adipiscing non, tincidunt eleifend, adipiscing quis, augue. Vivamus elementum placerat enim.  Suspendisse ut tortor. Integer faucibus adipiscing felis. Aenean consectetuer mattis lectus. Morbi malesuada faucibus dolor. Nam lacus. Etiam arcu libero, malesuada vitae, aliquam vitae, blandit tristique, nisl.",
      "Maecenas accumsan dapibus sapien. Duis pretium iaculis arcu. Curabitur ut lacus. Aliquam vulputate.  Suspendisse ut purus sed sem tempor rhoncus. Ut quam dui, fringilla at, dictum eget, ultricies quis, quam. Etiam sem est, pharetra non, vulputate in, pretium at, ipsum. Nunc semper sagittis orci. Sed scelerisque suscipit diam. Ut volutpat, dolor at ullamcorper tristique, eros purus mollis quam, sit amet ornare ante nunc et enim.",
      "Phasellus fringilla, metus id feugiat consectetuer, lacus wisi ultrices tellus, quis lobortis nibh lorem quis tortor. Donec egestas ornare nulla. Mauris mi tellus, porta faucibus, dictum vel, nonummy in, est. Aliquam erat volutpat. In tellus magna, porttitor lacinia, molestie vitae, pellentesque eu, justo. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Sed orci nibh, scelerisque sit amet, suscipit sed, placerat vel, diam. Vestibulum nonummy vulputate orci.  Donec et velit ac arcu interdum semper. Morbi pede orci, cursus ac, elementum non, vehicula ut, lacus. Cras volutpat. Nam vel wisi quis libero venenatis placerat. Aenean sed odio. Quisque posuere purus ac orci. Vivamus odio. Vivamus varius, nulla sit amet semper viverra, odio mauris consequat lacus, at vestibulum neque arcu eu tortor.  Donec iaculis tincidunt tellus. Aliquam erat volutpat. Curabitur magna lorem, dignissim volutpat, viverra et, adipiscing nec, dolor.  Praesent lacus mauris, dapibus vitae, sollicitudin sit amet, nonummy eget, ligula.",
      "Cras egestas ipsum a nisl. Vivamus varius dolor ut dolor. Fusce vel enim. Pellentesque accumsan ligula et eros. Cras id lacus non tortor facilisis facilisis. Etiam nisl elit, cursus sed, fringilla in, congue nec, urna. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Integer at turpis. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Duis fringilla, ligula sed porta fringilla, ligula wisi commodo felis, ut adipiscing felis dui in enim. Suspendisse malesuada ultrices ante. Pellentesque scelerisque augue sit amet urna. Nulla volutpat aliquet tortor. Cras aliquam, tellus at aliquet pellentesque, justo sapien commodo leo, id rhoncus sapien quam at erat. Nulla commodo, wisi eget sollicitudin pretium, orci orci aliquam orci, ut cursus turpis justo et lacus. Nulla vel tortor. Quisque erat elit, viverra sit amet, sagittis eget, porta sit amet, lacus.",
      "In hac habitasse platea dictumst. Proin at est. Curabitur tempus vulputate elit. Pellentesque sem. Praesent eu sapien. Duis elit magna, aliquet at, tempus sed, vehicula non, enim.  Morbi viverra arcu nec purus. Vivamus fringilla, enim et commodo malesuada, tortor metus elementum ligula, nec aliquet est sapien ut lectus. Aliquam mi. Ut nec elit. Fusce euismod luctus tellus.  Curabitur scelerisque. Nullam purus. Nam ultricies accumsan magna.  Morbi pulvinar lorem sit amet ipsum. Donec ut justo vitae nibh mollis congue. Fusce quis diam. Praesent tempus eros ut quam.",
      "Donec in nisl. Fusce vitae est. Vivamus ante ante, mattis laoreet, posuere eget, congue vel, nunc. Fusce sem. Nam vel orci eu eros viverra luctus. Pellentesque sit amet augue. Nunc sit amet ipsum et lacus varius nonummy. Integer rutrum sem eget wisi. Aenean eu sapien. Quisque ornare dignissim mi. Duis a urna vel risus pharetra imperdiet. Suspendisse potenti.",
      "Morbi justo. Aenean nec dolor. In hac habitasse platea dictumst. Proin nonummy porttitor velit. Sed sit amet leo nec metus rhoncus varius. Cras ante. Vestibulum commodo sem tincidunt massa. Nam justo. Aenean luctus, felis et condimentum lacinia, lectus enim pulvinar purus, non porta velit nisl sed eros.  Suspendisse consequat. Mauris a dui et tortor mattis pretium. Sed nulla metus, volutpat id, aliquam eget, ullamcorper ut, ipsum. Morbi eu nunc. Praesent pretium. Duis aliquam pulvinar ligula. Ut blandit egestas justo. Quisque posuere metus viverra pede.",
      "Vivamus sodales elementum neque. Vivamus dignissim accumsan neque. Sed at enim. Vestibulum nonummy interdum purus. Mauris ornare velit id nibh pretium ultricies. Fusce tempor pellentesque odio. Vivamus augue purus, laoreet in, scelerisque vel, commodo id, wisi. Duis enim. Nulla interdum, nunc eu semper eleifend, enim dolor pretium elit, ut commodo ligula nisl a est.  Vivamus ante. Nulla leo massa, posuere nec, volutpat vitae, rhoncus eu, magna.",
      "Quisque facilisis auctor sapien. Pellentesque gravida hendrerit lectus. Mauris rutrum sodales sapien. Fusce hendrerit sem vel lorem. Integer pellentesque massa vel augue.  Integer elit tortor, feugiat quis, sagittis et, ornare non, lacus.  Vestibulum posuere pellentesque eros. Quisque venenatis ipsum dictum nulla. Aliquam quis quam non metus eleifend interdum. Nam eget sapien ac mauris malesuada adipiscing. Etiam eleifend neque sed quam. Nulla facilisi. Proin a ligula. Sed id dui eu nibh egestas tincidunt. Suspendisse arcu.",
      "Maecenas dui. Aliquam volutpat auctor lorem.  Cras placerat est vitae lectus. Curabitur massa lectus, rutrum euismod, dignissim ut, dapibus a, odio. Ut eros erat, vulputate ut, interdum non, porta eu, erat. Cras fermentum, felis in porta congue, velit leo facilisis odio, vitae consectetuer lorem quam vitae orci.  Sed ultrices, pede eu placerat auctor, ante ligula rutrum tellus, vel posuere nibh lacus nec nibh. Maecenas laoreet dolor at enim.  Donec molestie dolor nec metus. Vestibulum libero. Sed quis erat.  Sed tristique. Duis pede leo, fermentum quis, consectetuer eget, vulputate sit amet, erat.",
      "Donec vitae velit. Suspendisse porta fermentum mauris. Ut vel nunc non mauris pharetra varius. Duis consequat libero quis urna. Maecenas at ante. Vivamus varius, wisi sed egestas tristique, odio wisi luctus nulla, lobortis dictum dolor ligula in lacus. Vivamus aliquam, urna sed interdum porttitor, metus orci interdum odio, sit amet euismod lectus felis et leo. Praesent ac wisi. Nam suscipit vestibulum sem. Praesent eu ipsum vitae pede cursus venenatis. Duis sed odio. Vestibulum eleifend. Nulla ut massa. Proin rutrum mattis sapien. Curabitur dictum gravida ante.",
      "Phasellus placerat vulputate quam. Maecenas at tellus. Pellentesque neque diam, dignissim ac, venenatis vitae, consequat ut, lacus. Nam nibh. Vestibulum fringilla arcu mollis arcu. Sed et turpis. Donec sem tellus, volutpat et, varius eu, commodo sed, lectus. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Quisque enim arcu, suscipit nec, tempus at, imperdiet vel, metus. Morbi volutpat purus at erat. Donec dignissim, sem id semper tempus, nibh massa eleifend turpis, sed pellentesque wisi purus sed libero. Nullam lobortis tortor vel risus.  Pellentesque consequat nulla eu tellus. Donec velit. Aliquam fermentum, wisi ac rhoncus iaculis, tellus nunc malesuada orci, quis volutpat dui magna id mi. Nunc vel ante. Duis vitae lacus. Cras nec ipsum.",
      "Morbi nunc. Aliquam consectetuer varius nulla.  Phasellus eros. Cras dapibus porttitor risus. Maecenas ultrices mi sed diam. Praesent gravida velit at elit vehicula porttitor.  Phasellus nisl mi, sagittis ac, pulvinar id, gravida sit amet, erat.  Vestibulum est. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Curabitur id sem elementum leo rutrum hendrerit. Ut at mi.  Donec tincidunt faucibus massa. Sed turpis quam, sollicitudin a, hendrerit eget, pretium ut, nisl. Duis hendrerit ligula. Nunc pulvinar congue urna.",
      "Nunc velit. Nullam elit sapien, eleifend eu, commodo nec, semper sit amet, elit. Nulla lectus risus, condimentum ut, laoreet eget, viverra nec, odio. Proin lobortis. Curabitur dictum arcu vel wisi. Cras id nulla venenatis tortor congue ultrices. Pellentesque eget pede. Sed eleifend sagittis elit. Nam sed tellus sit amet lectus ullamcorper tristique. Mauris enim sem, tristique eu, accumsan at, scelerisque vulputate, neque. Quisque lacus. Donec et ipsum sit amet elit nonummy aliquet. Sed viverra nisl at sem. Nam diam. Mauris ut dolor. Curabitur ornare tortor cursus velit.",
      "Morbi tincidunt posuere arcu. Cras venenatis est vitae dolor. Vivamus scelerisque semper mi. Donec ipsum arcu, consequat scelerisque, viverra id, dictum at, metus. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Ut pede sem, tempus ut, porttitor bibendum, molestie eu, elit. Suspendisse potenti. Sed id lectus sit amet purus faucibus vehicula. Praesent sed sem non dui pharetra interdum. Nam viverra ultrices magna.",
      "Aenean laoreet aliquam orci. Nunc interdum elementum urna. Quisque erat. Nullam tempor neque. Maecenas velit nibh, scelerisque a, consequat ut, viverra in, enim. Duis magna.  Donec odio neque, tristique et, tincidunt eu, rhoncus ac, nunc.  Mauris malesuada malesuada elit. Etiam lacus mauris, pretium vel, blandit in, ultricies id, libero. Phasellus bibendum erat ut diam.  In congue imperdiet lectus.",
      "Aenean scelerisque. Fusce pretium porttitor lorem. In hac habitasse platea dictumst. Nulla sit amet nisl at sapien egestas pretium. Nunc non tellus. Vivamus aliquet. Nam adipiscing euismod dolor. Aliquam erat volutpat. Nulla ut ipsum.  Quisque tincidunt auctor augue. Nunc imperdiet ipsum eget elit.  Aliquam quam leo, consectetuer non, ornare sit amet, tristique quis, felis. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Pellentesque interdum quam sit amet mi. Pellentesque mauris dui, dictum a, adipiscing ac, fermentum sit amet, lorem.",
      "Ut quis wisi. Praesent quis massa. Vivamus egestas risus eget lacus. Nunc tincidunt, risus quis bibendum facilisis, lorem purus rutrum neque, nec porta tortor urna quis orci. Aenean aliquet, libero semper volutpat luctus, pede erat lacinia augue, quis rutrum sem ipsum sit amet pede. Vestibulum aliquet, nibh sed iaculis sagittis, odio dolor blandit augue, eget mollis urna tellus id tellus. Aenean aliquet aliquam nunc. Nulla ultricies justo eget orci. Phasellus tristique fermentum leo. Sed massa metus, sagittis ut, semper ut, pharetra vel, erat. Aliquam quam turpis, egestas vel, elementum in, egestas sit amet, lorem.  Duis convallis, wisi sit amet mollis molestie, libero mauris porta dui, vitae aliquam arcu turpis ac sem. Aliquam aliquet dapibus metus.",
      "Vivamus commodo eros eleifend dui. Vestibulum in leo eu erat tristique mattis. Cras at elit. Cras pellentesque.  Nullam id lacus sit amet libero aliquet hendrerit. Proin placerat, mi non elementum laoreet, eros elit tincidunt magna, a rhoncus sem arcu id odio. Nulla eget leo a leo egestas facilisis. Curabitur quis velit. Phasellus aliquam, tortor nec ornare rhoncus, purus urna posuere velit, et commodo risus tellus quis tellus. Vivamus leo turpis, tempus sit amet, tristique vitae, laoreet quis, odio. Proin scelerisque bibendum ipsum. Etiam nisl. Praesent vel dolor.  Pellentesque vel magna. Curabitur urna. Vivamus congue urna in velit. Etiam ullamcorper elementum dui. Praesent non urna. Sed placerat quam non mi. Pellentesque diam magna, ultricies eget, ultrices placerat, adipiscing rutrum, sem.",
      "Morbi sem. Nulla facilisi. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Nulla facilisi. Morbi sagittis ultrices libero. Praesent eu ligula sed sapien auctor sagittis. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Donec vel nunc. Nunc fermentum, lacus id aliquam porta, dui tortor euismod eros, vel molestie ipsum purus eu lacus. Vivamus pede arcu, euismod ac, tempus id, pretium et, lacus. Curabitur sodales dapibus urna.  Nunc eu sapien. Donec eget nunc a pede dictum pretium. Proin mauris.  Vivamus luctus libero vel nibh.",
      "Fusce tristique risus id wisi. Integer molestie massa id sem. Vestibulum vel dolor. Pellentesque vel urna vel risus ultricies elementum. Quisque sapien urna, blandit nec, iaculis ac, viverra in, odio. In hac habitasse platea dictumst.  Morbi neque lacus, convallis vitae, commodo ac, fermentum eu, velit.  Sed in orci. In fringilla turpis non arcu. Donec in ante. Phasellus tempor feugiat velit. Aenean varius massa non turpis. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae;",
      "Aliquam tortor. Morbi ipsum massa, imperdiet non, consectetuer vel, feugiat vel, lorem. Quisque eget lorem nec elit malesuada vestibulum. Quisque sollicitudin ipsum vel sem. Nulla enim. Proin nonummy felis vitae felis. Nullam pellentesque. Duis rutrum feugiat felis. Mauris vel pede sed libero tincidunt mollis.  Phasellus sed urna rhoncus diam euismod bibendum. Phasellus sed nisl. Integer condimentum justo id orci iaculis varius. Quisque et lacus. Phasellus elementum, justo at dignissim auctor, wisi odio lobortis arcu, sed sollicitudin felis felis eu neque. Praesent at lacus.",
      "Vivamus sit amet pede. Duis interdum, nunc eget rutrum dignissim, nisl diam luctus leo, et tincidunt velit nisl id tellus. In lorem tellus, aliquet vitae, porta in, aliquet sed, lectus. Phasellus sodales. Ut varius scelerisque erat. In vel nibh eu eros imperdiet rutrum. Donec ac odio nec neque vulputate suscipit. Nam nec magna. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Nullam porta, odio et sagittis iaculis, wisi neque fringilla sapien, vel commodo lorem lorem id elit. Ut sem lectus, scelerisque eget, placerat et, tincidunt scelerisque, ligula. Pellentesque non orci.",
      "Etiam vel ipsum. Morbi facilisis vestibulum nisl. Praesent cursus laoreet felis. Integer adipiscing pretium orci. Nulla facilisi. Quisque posuere bibendum purus. Nulla quam mauris, cursus eget, convallis ac, molestie non, enim. Aliquam congue. Quisque sagittis nonummy sapien. Proin molestie sem vitae urna. Maecenas lorem. Vivamus viverra consequat enim.",
      "Nunc sed pede. Praesent vitae lectus. Praesent neque justo, vehicula eget, interdum id, facilisis et, nibh.  Phasellus at purus et libero lacinia dictum. Fusce aliquet. Nulla eu ante placerat leo semper dictum. Mauris metus. Curabitur lobortis.  Curabitur sollicitudin hendrerit nunc. Donec ultrices lacus id ipsum.",
      "Donec a nibh ut elit vestibulum tristique.  Integer at pede. Cras volutpat varius magna. Phasellus eu wisi.  Praesent risus justo, lobortis eget, scelerisque ac, aliquet in, dolor. Proin id leo. Nunc iaculis, mi vitae accumsan commodo, neque sem lacinia nulla, quis vestibulum justo sem in eros. Quisque sed massa. Morbi lectus ipsum, vulputate a, mollis ut, accumsan placerat, tellus. Nullam in wisi. Vivamus eu ligula a nunc accumsan congue. Suspendisse ac libero. Aliquam erat volutpat. Donec augue.  Nunc venenatis fringilla nibh. Fusce accumsan pulvinar justo. Nullam semper, dui ut dignissim auctor, orci libero fringilla massa, blandit pulvinar pede tortor id magna. Nunc adipiscing justo sed velit tincidunt fermentum.",
      "Integer placerat. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Sed in massa. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Phasellus tempus aliquam risus. Aliquam rutrum purus at metus. Donec posuere odio at erat. Nam non nibh. Phasellus ligula. Quisque venenatis lectus in augue. Sed vestibulum dapibus neque.",
      "Mauris tempus eros at nulla. Sed quis dui dignissim mauris pretium tincidunt. Mauris ac purus. Phasellus ac libero. Etiam dapibus iaculis nunc. In lectus wisi, elementum eu, sollicitudin nec, imperdiet quis, dui. Nulla viverra neque ac libero. Mauris urna leo, adipiscing eu, ultrices non, blandit eu, dui. Maecenas dui neque, suscipit sit amet, rutrum a, laoreet in, eros. Ut eu nibh. Fusce nec erat tempus urna fringilla tempus.  Curabitur id enim. Sed ante. Cras sodales enim sit amet wisi. Nunc fermentum consequat quam.",
      "Ut auctor, augue porta dignissim vestibulum, arcu diam lobortis velit, vel scelerisque risus augue sagittis risus. Maecenas eu justo. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Mauris congue ligula eget tortor. Nullam laoreet urna sed enim. Donec eget eros ut eros volutpat convallis. Praesent turpis. Integer mauris diam, elementum quis, egestas ac, rutrum vel, orci. Nulla facilisi.  Quisque adipiscing, nulla vitae elementum porta, sem urna volutpat leo, sed porta enim risus sed massa. Integer ac enim quis diam sodales luctus. Ut eget eros a ligula commodo ultricies. Donec eu urna viverra dolor hendrerit feugiat. Aliquam ac orci vel eros congue pharetra. Quisque rhoncus, justo eu volutpat faucibus, augue leo posuere lacus, a rhoncus purus pede vel est. Proin ultrices enim.",
      "Aenean tincidunt laoreet dui. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Integer ipsum lectus, fermentum ac, malesuada in, eleifend ut, lorem. Vivamus ipsum turpis, elementum vel, hendrerit ut, semper at, metus. Vivamus sapien tortor, eleifend id, dapibus in, egestas et, pede. Pellentesque faucibus. Praesent lorem neque, dignissim in, facilisis nec, hendrerit vel, odio. Nam at diam ac neque aliquet viverra. Morbi dapibus ligula sagittis magna. In lobortis. Donec aliquet ultricies libero. Nunc dictum vulputate purus. Morbi varius.  Lorem ipsum dolor sit amet, consectetuer adipiscing elit. In tempor.  Phasellus commodo porttitor magna. Curabitur vehicula odio vel dolor.",
      "Praesent facilisis, augue a adipiscing venenatis, libero risus molestie odio, pulvinar consectetuer felis erat ac mauris. Nam vestibulum rhoncus quam. Sed velit urna, pharetra eu, eleifend eu, viverra at, wisi. Maecenas ultrices nibh at turpis. Aenean quam. Nulla ipsum. Aliquam posuere luctus erat.  Curabitur magna felis, lacinia et, tristique id, ultrices ut, mauris. Suspendisse feugiat. Cras eleifend wisi vitae tortor.  Phasellus leo purus, mattis sit amet, auctor in, rutrum in, magna.  In hac habitasse platea dictumst. Phasellus imperdiet metus in sem.  Vestibulum ac enim non sem ultricies sagittis. Sed vel diam.",
      "Integer vel enim sed turpis adipiscing bibendum. Vestibulum pede dolor, laoreet nec, posuere in, nonummy in, sem. Donec imperdiet sapien placerat erat. Donec viverra.  Aliquam eros. Nunc consequat massa id leo. Sed ullamcorper, lorem in sodales dapibus, risus metus sagittis lorem, non porttitor purus odio nec odio. Sed tincidunt posuere elit. Quisque eu enim. Donec libero risus, feugiat ac, dapibus eget, posuere a, felis. Quisque vel lectus ut metus tincidunt eleifend. Duis ut pede. Duis velit erat, venenatis vitae, vulputate a, pharetra sit amet, est. Etiam fringilla faucibus augue.",
      "Aenean velit sem, viverra eu, tempus id, rutrum id, mi. Nullam nec nibh. Proin ullamcorper, dolor in cursus tristique, eros augue tempor nibh, at gravida diam wisi at purus.  Donec mattis ullamcorper tellus. Phasellus vel nulla. Praesent interdum, eros in sodales sollicitudin, nunc nulla pulvinar justo, a euismod eros sem nec nibh. Nullam sagittis dapibus lectus. Nullam eget ipsum eu tortor lobortis sodales. Etiam purus leo, pretium nec, feugiat non, ullamcorper vel, nibh. Sed vel elit et quam accumsan facilisis. Nunc leo. Suspendisse faucibus lacus.",
      "Pellentesque interdum sapien sed nulla. Proin tincidunt. Aliquam volutpat est vel massa. Sed dolor lacus, imperdiet non, ornare non, commodo eu, neque. Integer pretium semper justo. Proin risus. Nullam id quam. Nam neque. Duis vitae wisi ullamcorper diam congue ultricies. Quisque ligula. Mauris vehicula.",
      "Curabitur nunc magna, posuere eget, venenatis eu, vehicula ac, velit. Aenean ornare, massa a accumsan pulvinar, quam lorem laoreet purus, eu sodales magna risus molestie lorem.  Nunc erat velit, hendrerit quis, malesuada ut, aliquam vitae, wisi.  Sed posuere. Suspendisse ipsum arcu, scelerisque nec, aliquam eu, molestie tincidunt, justo. Phasellus iaculis. Sed posuere lorem non ipsum. Pellentesque dapibus. Suspendisse quam libero, laoreet a, tincidunt eget, consequat at, est. Nullam ut lectus non enim consequat facilisis. Mauris leo. Quisque pede ligula, auctor vel, pellentesque vel, posuere id, turpis. Cras ipsum sem, cursus et, facilisis ut, tempus euismod, quam. Suspendisse tristique dolor eu orci. Mauris mattis. Aenean semper. Vivamus tortor magna, facilisis id, varius mattis, hendrerit in, justo. Integer purus.",
      "Vivamus adipiscing. Curabitur imperdiet tempus turpis. Vivamus sapien dolor, congue venenatis, euismod eget, porta rhoncus, magna. Proin condimentum pretium enim. Fusce fringilla, libero et venenatis facilisis, eros enim cursus arcu, vitae facilisis odio augue vitae orci. Aliquam varius nibh ut odio.  Sed condimentum condimentum nunc. Pellentesque eget massa.  Pellentesque quis mauris. Donec ut ligula ac pede pulvinar lobortis.  Pellentesque euismod. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Praesent elit.  Ut laoreet ornare est. Phasellus gravida vulputate nulla. Donec sit amet arcu ut sem tempor malesuada. Praesent hendrerit augue in urna.  Proin enim ante, ornare vel, consequat ut, blandit in, justo. Donec felis elit, dignissim sed, sagittis ut, ullamcorper a, nulla. Aenean pharetra vulputate odio.",
      "Quisque enim. Proin velit neque, tristique eu, eleifend eget, vestibulum nec, lacus. Vivamus odio. Duis odio urna, vehicula in, elementum aliquam, aliquet laoreet, tellus. Sed velit. Sed vel mi ac elit aliquet interdum. Etiam sapien neque, convallis et, aliquet vel, auctor non, arcu. Aliquam suscipit aliquam lectus. Proin tincidunt magna sed wisi. Integer blandit lacus ut lorem. Sed luctus justo sed enim.",
     "Morbi malesuada hendrerit dui. Nunc mauris leo, dapibus sit amet, vestibulum et, commodo id, est. Pellentesque purus. Pellentesque tristique, nunc ac pulvinar adipiscing, justo eros consequat lectus, sit amet posuere lectus neque vel augue. Cras consectetuer libero ac eros. Ut eget massa. Fusce sit amet enim eleifend sem dictum auctor. In eget risus luctus wisi convallis pulvinar. Vivamus sapien risus, tempor in, viverra in, aliquet pellentesque, eros. Aliquam euismod libero a sem.",
      "Nunc velit augue, scelerisque dignissim, lobortis et, aliquam in, risus. In eu eros. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Curabitur vulputate elit viverra augue. Mauris fringilla, tortor sit amet malesuada mollis, sapien mi dapibus odio, ac imperdiet ligula enim eget nisl. Quisque vitae pede a pede aliquet suscipit.  Phasellus tellus pede, viverra vestibulum, gravida id, laoreet in, justo. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Integer commodo luctus lectus. Mauris justo.  Duis varius eros. Sed quam. Cras lacus eros, rutrum eget, varius quis, convallis iaculis, velit. Mauris imperdiet, metus at tristique venenatis, purus neque pellentesque mauris, a ultrices elit lacus nec tortor. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Praesent malesuada. Nam lacus lectus, auctor sit amet, malesuada vel, elementum eget, metus.  Duis neque pede, facilisis eget, egestas elementum, nonummy id, neque.",
      "Proin non sem. Donec nec erat. Proin libero.  Aliquam viverra arcu. Donec vitae purus. Donec felis mi, semper id, scelerisque porta, sollicitudin sed, turpis. Nulla in urna. Integer varius wisi non elit. Etiam nec sem. Mauris consequat, risus nec congue condimentum, ligula ligula suscipit urna, vitae porta odio erat quis sapien. Proin luctus leo id erat. Etiam massa metus, accumsan pellentesque, sagittis sit amet, venenatis nec, mauris.  Praesent urna eros, ornare nec, vulputate eget, cursus sed, justo.  Phasellus nec lorem. Nullam ligula ligula, mollis sit amet, faucibus vel, eleifend ac, dui. Aliquam erat volutpat.",
      "Fusce vehicula, tortor et gravida porttitor, metus nibh congue lorem, ut tempus purus mauris a pede. Integer tincidunt orci sit amet turpis. Aenean a metus. Aliquam vestibulum lobortis felis. Donec gravida. Sed sed urna. Mauris et orci. Integer ultrices feugiat ligula. Sed dignissim nibh a massa. Donec orci dui, tempor sed, tincidunt nonummy, viverra sit amet, turpis. Quisque lobortis. Proin venenatis tortor nec wisi. Vestibulum placerat. In hac habitasse platea dictumst. Aliquam porta mi quis risus. Donec sagittis luctus diam. Nam ipsum elit, imperdiet vitae, faucibus nec, fringilla eget, leo. Etiam quis dolor in sapien porttitor imperdiet.",
      "Cras pretium. Nulla malesuada ipsum ut libero. Suspendisse gravida hendrerit tellus. Maecenas quis lacus.  Morbi fringilla. Vestibulum odio turpis, tempor vitae, scelerisque a, dictum non, massa. Praesent erat felis, porta sit amet, condimentum sit amet, placerat et, turpis. Praesent placerat lacus a enim. Vestibulum non eros. Ut congue. Donec tristique varius tortor.  Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Nam dictum dictum urna.",
      "Phasellus vestibulum orci vel mauris. Fusce quam leo, adipiscing ac, pulvinar eget, molestie sit amet, erat. Sed diam. Suspendisse eros leo, tempus eget, dapibus sit amet, tempus eu, arcu. Vestibulum wisi metus, dapibus vel, luctus sit amet, condimentum quis, leo. Suspendisse molestie. Duis in ante. Ut sodales sem sit amet mauris. Suspendisse ornare pretium orci. Fusce tristique enim eget mi. Vestibulum eros elit, gravida ac, pharetra sed, lobortis in, massa. Proin at dolor. Duis accumsan accumsan pede. Nullam blandit elit in magna lacinia hendrerit. Ut nonummy luctus eros. Fusce eget tortor.",
      "Ut sit amet magna. Cras a ligula eu urna dignissim viverra. Nullam tempor leo porta ipsum. Praesent purus.  Nullam consequat. Mauris dictum sagittis dui. Vestibulum sollicitudin consectetuer wisi. In sit amet diam. Nullam malesuada pharetra risus. Proin lacus arcu, eleifend sed, vehicula at, congue sit amet, sem. Sed sagittis pede a nisl. Sed tincidunt odio a pede.  Sed dui. Nam eu enim. Aliquam sagittis lacus eget libero.  Pellentesque diam sem, sagittis molestie, tristique et, fermentum ornare, nibh. Nulla et tellus non felis imperdiet mattis. Aliquam erat volutpat.",
      "Vestibulum sodales ipsum id augue. Integer ipsum pede, convallis sit amet, tristique vitae, tempor ut, nunc.  Nam non ligula non lorem convallis hendrerit. Maecenas hendrerit.  Sed magna odio, aliquam imperdiet, porta ac, aliquet eget, mi. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Vestibulum nisl sem, dignissim vel, euismod quis, egestas ut, orci. Nunc vitae risus vel metus euismod laoreet. Cras sit amet neque a turpis lobortis auctor. Sed aliquam sem ac elit.  Cras velit lectus, facilisis id, dictum sed, porta rutrum, nisl. Nam hendrerit ipsum sed augue. Nullam scelerisque hendrerit wisi.  Vivamus egestas arcu sed purus. Ut ornare lectus sed eros.  Suspendisse potenti. Mauris sollicitudin pede vel velit. In hac habitasse platea dictumst.",
      "Suspendisse erat mauris, nonummy eget, pretium eget, consequat vel, justo. Pellentesque consectetuer erat sed lacus. Nullam egestas nulla ac dui. Donec cursus rhoncus ipsum.  Nunc et sem eu magna egestas malesuada. Vivamus dictum massa at dolor. Morbi est nulla, faucibus ac, posuere in, interdum ut, sapien. Proin consectetuer pretium urna. Donec sit amet nibh nec purus dignissim mattis. Phasellus vehicula elit at lacus. Nulla facilisi. Cras ut arcu. Sed consectetuer. Integer tristique elit quis felis consectetuer eleifend. Cras et lectus.",
      "Ut congue malesuada justo. Curabitur congue, felis at hendrerit faucibus, mauris lacus porttitor pede, nec aliquam turpis diam feugiat arcu. Nullam rhoncus ipsum at risus.  Vestibulum a dolor sed dolor fermentum vulputate. Sed nec ipsum dapibus urna bibendum lobortis. Vestibulum elit. Nam ligula arcu, volutpat eget, lacinia eu, lobortis ac, urna. Nam mollis ultrices nulla. Cras vulputate. Suspendisse at risus at metus pulvinar malesuada. Nullam lacus. Aliquam tempus magna. Aliquam ut purus.  Proin tellus.",
      "Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Donec scelerisque metus. Maecenas non mi ut metus porta hendrerit. Nunc semper. Cras quis wisi ut lorem posuere tristique. Nunc vestibulum scelerisque nulla. Suspendisse pharetra sollicitudin ante. Praesent at augue sit amet ante interdum porta. Nunc bibendum augue luctus diam. Etiam nec sem. Sed eros turpis, facilisis nec, vehicula vitae, aliquam sed, nulla. Curabitur justo leo, vestibulum eget, tristique ut, tempus at, nisl.",
      "Nulla venenatis lorem id arcu. Morbi cursus urna a ipsum. Donec porttitor. Integer eleifend, est non mattis malesuada, mi nulla convallis mi, et auctor lectus sapien ut purus.  Aliquam nulla augue, pharetra sit amet, faucibus semper, molestie vel, nibh. Pellentesque vestibulum magna et mi. Sed fringilla dolor vel tellus. Nunc libero nunc, venenatis eget, convallis hendrerit, iaculis elementum, mi. Nullam aliquam, felis et accumsan vehicula, magna justo vehicula diam, eu condimentum nisl felis et nunc.  Quisque volutpat mauris a velit. Pellentesque massa. Integer at lorem. Nam metus erat, lacinia id, convallis ut, pulvinar non, wisi.  Cras iaculis mauris ut neque. Cras sodales, sem vitae imperdiet consequat, pede purus sollicitudin urna, ac aliquam metus orci in leo. Ut molestie ultrices mauris. Vivamus vitae sem. Aliquam erat volutpat. Praesent commodo, nisl ac dapibus aliquet, tortor orci sodales lorem, non ornare nulla lorem quis nisl.",
      "Sed at sem vitae purus ultrices vestibulum.  Vestibulum tincidunt lacus et ligula. Pellentesque vitae elit.  Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Duis ornare, erat eget laoreet vulputate, lacus ipsum suscipit turpis, et bibendum nisl orci non lectus.  Vestibulum nec risus nec libero fermentum fringilla. Morbi non velit in magna gravida hendrerit. Pellentesque quis lectus. Vestibulum eleifend lobortis leo. Vestibulum non augue. Vivamus dictum tempor dui. Maecenas at ligula id felis congue porttitor. Nulla leo magna, egestas quis, vulputate sit amet, viverra id, velit.",
      "Ut lectus lectus, ultricies sit amet, semper eget, laoreet non, ante. Proin at massa quis nunc rhoncus mattis.  Aliquam lorem. Curabitur pharetra dui at neque. Aliquam eu tellus.  Aenean tempus, felis vitae vulputate iaculis, est dolor faucibus urna, in viverra wisi neque non risus. Fusce vel dolor nec sapien pretium nonummy. Integer faucibus massa ac nulla ornare venenatis.  Nulla quis sapien. Sed tortor. Phasellus eget mi. Cras nunc. Cras a enim.",
      "Quisque nisl. In dignissim dapibus massa.  Aenean sem magna, scelerisque nec, ullamcorper quis, porttitor ut, lectus. Fusce dignissim facilisis tortor. Vivamus gravida felis sit amet nunc. Nam pulvinar odio vel enim. Pellentesque sit amet est.  Vivamus pulvinar leo non sapien. Aliquam erat volutpat. Ut elementum auctor metus. Mauris vestibulum neque vitae eros. Pellentesque aliquam quam. Donec venenatis tristique purus. In nisl. Nulla velit libero, fermentum at, porta a, feugiat vitae, urna. Etiam aliquet ornare ipsum. Proin non dolor. Aenean nunc ligula, venenatis suscipit, porttitor sit amet, mattis suscipit, magna. Vivamus egestas viverra est. Morbi at risus sed sapien sodales pretium.",
      "Morbi congue congue metus. Aenean sed purus.  Nam pede magna, tristique nec, porta id, sollicitudin quis, sapien.  Vestibulum blandit. Suspendisse ut augue ac nibh ullamcorper posuere. Integer euismod, neque at eleifend fringilla, augue elit ornare dolor, vel tincidunt purus est id lacus. Vivamus lorem dui, commodo quis, scelerisque eu, tincidunt non, magna. Cras sodales.  Quisque vestibulum pulvinar diam. Phasellus tincidunt, leo vitae tristique facilisis, ipsum wisi interdum sem, dapibus semper nulla velit vel lectus. Cras dapibus mauris et augue. Quisque cursus nulla in libero. Suspendisse et lorem sit amet mauris malesuada mollis.  Nullam id justo. Maecenas venenatis. Donec lacus arcu, egestas ac, fermentum consectetuer, tempus eu, metus. Proin sodales, sem in pretium fermentum, arcu sapien commodo mauris, venenatis consequat augue urna in wisi. Quisque sapien nunc, varius eget, condimentum quis, lacinia in, est. Fusce facilisis. Praesent nec ipsum.",
      "Suspendisse a dolor. Nam erat eros, congue eget, sagittis a, lacinia in, pede. Maecenas in elit. Proin molestie varius nibh. Vivamus tristique purus sed augue. Proin egestas semper tortor. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos.  Vestibulum orci enim, sagittis ornare, eleifend ut, mattis at, ligula. Nulla molestie convallis arcu. Ut eros tellus, condimentum at, sodales in, ultrices vel, nulla.",
      "Duis magna ante, bibendum eget, eleifend eget, suscipit sed, neque. Vestibulum in mi sed massa cursus cursus.  Pellentesque pulvinar mollis neque. Fusce ut enim vitae mauris malesuada tincidunt. Vivamus a neque. Mauris pulvinar, sapien id condimentum dictum, quam arcu rhoncus dui, id tempor lacus justo et justo. Proin sit amet orci eu diam eleifend blandit. Nunc erat massa, luctus ac, fermentum lacinia, tincidunt ultrices, sapien.  Praesent sed orci vitae dolor sollicitudin adipiscing. Cras a neque.  Ut risus dui, interdum at, placerat id, tristique eu, enim.  Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Etiam adipiscing eros vestibulum dolor.  Pellentesque aliquam, diam eget eleifend posuere, augue eros porttitor lectus, ac dignissim dui metus nec felis. Quisque lacinia.  Vestibulum tellus. Suspendisse nec wisi. Aenean ac felis. Aliquam ultrices metus et nulla.",
      "Praesent sed est non nibh tempus venenatis.  Praesent rhoncus. Curabitur sagittis est sit amet neque. Sed commodo malesuada lectus. Phasellus enim tellus, tempor ut, tristique eu, aliquam eu, quam. Aenean quis quam quis wisi gravida vehicula.  Pellentesque a massa a leo pretium rhoncus. Suspendisse ultrices.  Donec lacinia malesuada massa. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Donec pretium ornare mauris. Phasellus auctor erat eget enim. Integer scelerisque, felis eu consequat fringilla, lorem wisi ultricies velit, id vehicula purus nulla eget odio. Nullam mattis, diam a rutrum fermentum, odio sapien tristique quam, id mollis tellus quam in odio. Mauris eu sapien. Donec aliquam lorem sit amet lorem pharetra lobortis.",
      "Donec ac velit. Sed convallis vestibulum sapien. Vivamus tempor lacus sed lacus. Nunc ut lorem. Ut et tortor.  Nullam varius wisi at diam. Etiam ultricies, dolor sit amet fermentum vulputate, neque libero vestibulum orci, vitae fringilla neque arcu aliquet ante. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Quisque venenatis lobortis augue. Sed tempor, tellus iaculis pellentesque pharetra, pede dui malesuada mauris, vel ultrices urna mauris ac nibh. Etiam nibh odio, ultricies vehicula, vestibulum vitae, feugiat eleifend, felis. Vivamus pulvinar. Aliquam erat volutpat. Nulla egestas venenatis metus. Nam feugiat nunc quis elit egestas sagittis. Sed vitae felis. In libero arcu, rhoncus in, commodo eget, auctor in, enim. Vivamus suscipit est. Nulla dapibus, magna vel aliquet egestas, massa massa hendrerit lacus, ac rutrum tellus tellus sit amet felis. Cras viverra.",
      "Suspendisse eu nunc. Aliquam dignissim urna sit amet mauris. Cras commodo, urna ut porttitor venenatis, arcu metus sodales risus, vitae gravida sapien ligula in est. Donec vulputate sollicitudin wisi. Donec vehicula, est id interdum ornare, nibh tellus consectetuer justo, a ultrices felis erat at lectus. In est massa, malesuada non, suscipit at, ullamcorper eu, elit. Nam nulla lacus, bibendum sit amet, sagittis sed, tempor eget, libero.  Praesent ligula. Suspendisse nulla. Etiam diam. Nulla ante diam, vestibulum et, aliquet ac, imperdiet vitae, urna. Fusce tincidunt lacus vel elit. Maecenas dictum, tortor non euismod bibendum, pede nibh pretium tellus, at dignissim leo eros eget pede. Nulla venenatis eleifend eros. Aenean ut odio dignissim augue rutrum faucibus. Fusce posuere, tellus eget viverra mattis, erat tellus porta mi, at facilisis sem nibh non urna. Phasellus quis turpis quis mauris suscipit vulputate. Sed interdum lacus non velit. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae;",
      "Vivamus vehicula leo a justo. Quisque nec augue.  Morbi mauris wisi, aliquet vitae, dignissim eget, sollicitudin molestie, ligula. In dictum enim sit amet risus. Curabitur vitae velit eu diam rhoncus hendrerit. Vivamus ut elit. Praesent mattis ipsum quis turpis. Curabitur rhoncus neque eu dui. Etiam vitae magna. Nam ullamcorper. Praesent interdum bibendum magna. Quisque auctor aliquam dolor. Morbi eu lorem et est porttitor fermentum.  Nunc egestas arcu at tortor varius viverra. Fusce eu nulla ut nulla interdum consectetuer. Vestibulum gravida. Morbi mattis libero sed est.",
      "Nam quis enim. Quisque ornare dui a tortor.  Fusce consequat lacus pellentesque metus. Duis euismod. Duis non quam. Maecenas vitae dolor in ipsum auctor vehicula. Vivamus nec nibh eget wisi varius pulvinar. Cras a lacus. Etiam et massa. Donec in nisl sit amet dui imperdiet vestibulum. Duis porttitor nibh id eros.",
      "Mauris consectetuer, wisi eu lobortis scelerisque, urna nibh feugiat quam, id congue eros justo eget orci.  Ut tellus. Maecenas mattis sapien sed eros. Aliquam quis lectus.  Donec nec massa ac turpis semper cursus. Etiam consectetuer ante vel odio. Aliquam tincidunt felis non dolor. Cras id augue ut nisl pretium placerat. Phasellus sapien sapien, pharetra sed, aliquam nec, suscipit a, nibh. Suspendisse risus. Nulla ut mi eget tellus sollicitudin euismod. Vestibulum malesuada malesuada dui. Ut at est ac dui aliquam sagittis. Aliquam erat volutpat.",
      "Curabitur ullamcorper est in mauris. Praesent ac massa. Quisque enim odio, lobortis nec, mattis ut, luctus et, mauris. Mauris eu risus. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Duis eu ligula. Nulla vehicula leo tincidunt erat. Maecenas et nunc. Sed ut sapien.  Vestibulum in est. Vestibulum rhoncus.",
      "Donec metus metus, condimentum eu, accumsan nec, vulputate non, purus. Vestibulum ullamcorper vehicula sapien.  Mauris risus odio, hendrerit ac, congue ac, ullamcorper at, odio.  Aenean leo justo, commodo vitae, placerat blandit, malesuada vel, sem. Donec sit amet ante eget mauris adipiscing sollicitudin.  Curabitur posuere sem et leo. Nulla ultricies mauris. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Fusce sollicitudin augue vel tellus. Vivamus mauris eros, pharetra vel, lacinia pretium, egestas a, nibh. Morbi a ligula.",
      "Donec vitae turpis. Suspendisse porttitor.  Mauris aliquam purus vitae tellus. Morbi metus diam, tempus ac, cursus ut, ultricies quis, nulla. Praesent nec justo. In lobortis.  Donec nec lectus a neque laoreet rhoncus. Quisque in risus nec wisi lacinia ullamcorper. In placerat. Proin facilisis sollicitudin libero. Integer eget neque et pede placerat aliquet. Aliquam purus nulla, pulvinar ut, facilisis quis, sodales sed, magna. Curabitur nulla lectus, rutrum id, bibendum ut, sagittis eget, diam. Sed porta dolor eget est. Integer hendrerit orci. In hac habitasse platea dictumst.",
      "Ut facilisis. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Sed pellentesque, turpis sit amet aliquet porta, risus odio venenatis felis, at porta tellus lacus vitae nisl. Donec augue. Quisque consequat, pede laoreet pellentesque posuere, urna sapien tempor justo, eu aliquam tortor nunc id mauris. Fusce pretium, purus facilisis consequat mattis, ligula leo pretium mauris, ac suscipit augue sapien sit amet ipsum.  Praesent et ligula eget tortor dapibus blandit. Duis rutrum felis eget dolor. Vestibulum quis elit. Integer dignissim, velit at scelerisque congue, ipsum nulla dignissim dolor, lacinia scelerisque neque erat a mi. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Quisque ipsum lectus, euismod et, lacinia eu, iaculis eu, pede. Etiam justo quam, cursus ut, vulputate vel, feugiat ut, eros.  Fusce eleifend mollis ipsum.",
      "Nulla facilisi. Nunc nec elit. Integer ornare convallis tortor. Proin ac diam. In est sapien, laoreet euismod, mattis a, tincidunt at, risus. Vivamus risus. Vestibulum aliquam, urna aliquam porttitor accumsan, nulla tortor ullamcorper elit, ut consequat augue purus sit amet libero. Vivamus nisl lacus, commodo vel, dignissim ut, vestibulum id, pede. Curabitur malesuada hendrerit libero. Mauris quis dolor in tellus varius posuere. Sed vulputate elit at wisi. Fusce vitae neque. Nulla consectetuer, nunc ac eleifend laoreet, mi nulla commodo wisi, vel faucibus ligula lectus ut arcu. Vivamus hendrerit.",
      "Sed varius, nulla vitae tincidunt lobortis, nibh ipsum sollicitudin libero, et commodo tellus massa in neque.  Nulla facilisi. Aenean nec lectus. Aliquam fermentum. Duis ut magna et augue interdum gravida. Morbi elit. Fusce malesuada tempus ipsum.  Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Mauris iaculis enim non metus. Nullam dui magna, congue et, suscipit sed, aliquam vel, turpis. Quisque ultricies.",
      "Suspendisse feugiat sapien laoreet ante.  Integer fringilla, erat eget adipiscing ultrices, nibh dui sollicitudin nunc, in lobortis arcu odio vitae erat. Fusce bibendum ultricies lacus. Mauris eleifend ligula a ante. Etiam faucibus cursus pede. Mauris enim eros, malesuada eu, mattis sit amet, blandit in, nulla. Fusce sit amet purus id mi posuere tincidunt.  Mauris sit amet quam vitae quam semper accumsan. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Nam a justo at quam accumsan euismod. Duis tincidunt tristique risus. Ut vel nibh vel libero varius malesuada. In hac habitasse platea dictumst. Morbi sagittis mattis lorem. Pellentesque metus tellus, rutrum vitae, malesuada et, pharetra accumsan, ante. Quisque ac metus ac nisl gravida pellentesque. Sed dapibus feugiat sapien. Vestibulum nec nunc eget sem aliquam lobortis. Suspendisse aliquam quam quis metus.",
     "Suspendisse in odio. In elit diam, cursus vitae, venenatis in, molestie in, leo. Cras ornare. Nulla libero. Phasellus feugiat mattis libero. Sed vehicula aliquam ligula. Nullam lacinia, felis vel dignissim sodales, enim lectus lobortis diam, quis nonummy mauris odio auctor tortor. Integer in dui nec lacus bibendum ultrices. Etiam odio elit, aliquam et, porttitor id, interdum cursus, elit. Nulla eleifend tempor mauris. In vel arcu quis pede laoreet vulputate.",
     "Morbi pharetra magna a lorem. Cras sapien. Duis porttitor vehicula urna. Phasellus iaculis, mi vitae varius consequat, purus nibh sollicitudin mauris, quis aliquam felis dolor vel elit. Quisque neque mi, bibendum non, tristique convallis, congue eu, quam. Etiam vel felis. Quisque ac ligula at orci pulvinar rutrum. Donec mi eros, sagittis eu, consectetuer sed, sagittis sed, lorem. Nunc sed eros. Nullam pellentesque ante quis lectus. Vivamus lacinia, sapien vel fermentum placerat, purus nisl aliquet odio, et porta wisi dui nec nunc. Fusce porta cursus libero.",
      "Quisque eu mi a augue mollis posuere. Donec tincidunt, lorem at vestibulum pulvinar, felis purus nonummy urna, at accumsan purus dui nec leo. Praesent tortor turpis, vehicula in, aliquet ut, dignissim ac, leo. Curabitur sagittis mi id eros. In magna. Sed vitae elit facilisis elit semper sollicitudin. Curabitur convallis tempor nulla. Nullam non turpis a pede sagittis ultrices.  Etiam vulputate pede in ligula. Sed a ante id metus pellentesque suscipit. Sed adipiscing justo vitae sapien. Nunc posuere, pede ullamcorper gravida egestas, justo libero tincidunt arcu, vitae pellentesque arcu leo ut mauris. Pellentesque auctor mauris sit amet elit luctus fringilla. Cras sed wisi. Morbi luctus enim vitae tellus. Vivamus venenatis sodales libero.",
      "In hac habitasse platea dictumst. Suspendisse potenti. Nulla pretium sem sit amet nisl. Nulla facilisi. Sed aliquam, turpis sed hendrerit gravida, nunc metus aliquam urna, eget pharetra nibh urna nec lectus. Duis in nisl a nisl commodo facilisis. Nunc placerat risus sed leo. Duis pellentesque porta libero. Praesent et enim. Aenean ullamcorper, ante sit amet fermentum mollis, ligula metus laoreet magna, accumsan accumsan nibh wisi at wisi. Nam tincidunt tempor neque. Maecenas dolor. Donec interdum nisl. Aliquam quam libero, interdum quis, volutpat sed, semper ut, eros. Pellentesque sodales auctor quam. Nullam suscipit massa nec elit. Nullam vulputate.",
     "Aliquam a nulla. Suspendisse suscipit. Etiam lectus ante, interdum sit amet, euismod venenatis, condimentum eu, urna. Etiam at turpis. Cras quis ligula. Cras varius, sapien non pellentesque bibendum, mauris wisi sodales sem, ac commodo mauris neque non felis. Sed sollicitudin tincidunt arcu. Nullam vel lectus sit amet magna tincidunt tempor. Phasellus a ante. Donec et diam.",
      "Proin sit amet augue. Praesent lacus. Donec a leo. Ut turpis ante, condimentum sed, sagittis a, blandit sit amet, enim. Integer sed elit. In ultricies blandit libero. Proin molestie erat dignissim nulla convallis ultrices. Aliquam in magna. Etiam sollicitudin, eros a sagittis pellentesque, lacus odio volutpat elit, vel tincidunt felis dui vitae lorem. Etiam leo. Nulla et justo.",
      "Integer interdum varius diam. Nam aliquam velit a pede. Vivamus dictum nulla et wisi. Vestibulum a massa.  Donec vulputate nibh vitae risus dictum varius. Nunc suscipit, nunc nec facilisis convallis, lacus ligula bibendum nulla, ac sollicitudin sapien nisl fermentum velit. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Nullam commodo dui ut augue molestie scelerisque. Sed aliquet rhoncus tortor. Fusce laoreet, turpis a facilisis tristique, leo mauris accumsan tellus, vitae ornare lacus pede sit amet purus. Sed dignissim velit vitae ligula.  Sed sit amet diam sit amet arcu luctus ullamcorper.",
      "Duis quis velit id elit facilisis luctus.  Donec nec elit. Quisque ullamcorper arcu ac felis. Phasellus leo.  Pellentesque consequat consequat purus. Ut vel justo at pede facilisis tempor. Integer tempus blandit dolor. Donec eget neque sed elit ultricies molestie. Cras cursus viverra tortor. Cras commodo condimentum diam. Pellentesque interdum malesuada wisi. Suspendisse eu quam. Donec consectetuer. Suspendisse wisi purus, vestibulum at, vehicula vel, congue a, eros. Nulla vulputate dolor at purus.",
      "Suspendisse ac diam sed dui adipiscing pretium. Donec ullamcorper, sapien nec tempor venenatis, enim felis euismod pede, ut auctor lacus lectus sit amet diam. Vestibulum rutrum sem ut ante. Nulla eros. Quisque vitae nisl eget tellus feugiat volutpat. Nam id neque eu quam sodales vehicula. Nam dapibus, nulla eu iaculis placerat, pede est volutpat purus, id iaculis elit elit vel mauris. Donec dui. In hac habitasse platea dictumst. Nunc non quam. Proin euismod egestas eros. Mauris nisl.  Sed neque. Phasellus bibendum. Proin ut purus in eros faucibus auctor.",
      "Fusce mollis dui eu leo. Sed sapien augue, porta at, posuere ut, ultrices molestie, est. Vivamus quis pede nec erat placerat tincidunt. Aenean odio dui, facilisis non, vehicula et, bibendum a, libero. Etiam leo turpis, venenatis eleifend, nonummy sit amet, aliquam non, mi. Maecenas eget mi. Sed nec diam.  Integer orci tellus, pellentesque nec, bibendum quis, sodales ut, nibh. Duis laoreet aliquet orci. Curabitur sit amet sem sit amet nibh fermentum faucibus. Donec adipiscing, ipsum id fringilla convallis, elit massa cursus augue, at lobortis massa augue nec ligula. Proin ac lacus.",
      "Nunc id nulla nec mauris iaculis rutrum. Nunc nisl. Integer mi. Praesent lorem neque, egestas at, molestie in, faucibus et, eros. Sed rutrum, ante vitae aliquet tincidunt, diam elit auctor risus, eu elementum purus turpis eu elit. Proin ac orci.  Integer varius, urna non sollicitudin consequat, massa libero pharetra erat, et venenatis dui orci eget purus. Aliquam iaculis est eget ipsum. Ut volutpat velit. Phasellus fringilla. Aliquam mollis tellus vel odio. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Vestibulum gravida sapien sed diam dictum pharetra. Nulla ac odio. Duis vitae metus ut purus feugiat interdum. Duis eros enim, tincidunt ac, venenatis et, dignissim id, lacus. Curabitur sagittis dolor nec augue. Sed ultricies mauris. Donec semper, enim eu vestibulum placerat, justo risus eleifend quam, ac semper velit pede convallis arcu.",
      "Pellentesque tempus. Fusce tempor euismod nulla. Integer metus quam, semper sit amet, pellentesque sed, ornare sit amet, pede. Sed viverra. Aliquam erat volutpat. Donec tristique.  In ac pede ut tortor mattis blandit. Phasellus a nunc. Integer metus. Sed malesuada gravida arcu. Lorem ipsum dolor sit amet, consectetuer adipiscing elit.",
      "Phasellus suscipit placerat neque. Duis rutrum. Quisque enim. Proin et erat at augue aliquam aliquam. Mauris porttitor imperdiet lectus. Proin egestas faucibus risus. Praesent pharetra consequat odio. Fusce sed felis et nulla tempor elementum.  Nulla eu turpis. Proin posuere. Nullam nonummy nulla sed nulla volutpat consectetuer. Vivamus vehicula accumsan eros. Fusce ullamcorper. Phasellus vehicula consequat mauris. Sed vitae purus.  Sed accumsan, felis suscipit auctor fermentum, odio turpis vestibulum risus, vitae mattis metus neque non pede.",
      "Suspendisse mollis erat et risus. Vestibulum et odio eu nisl malesuada dapibus. Morbi ac tortor et magna tincidunt ullamcorper. Ut pellentesque fermentum mi. Etiam sed neque sit amet leo consectetuer sagittis. Nulla facilisi. Sed lobortis erat vitae nulla. Duis bibendum ipsum et mi scelerisque dapibus.  Fusce nonummy vestibulum orci. Donec a nisl. Integer ac nibh.  Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Aenean nec nunc sed dui lobortis vestibulum. Praesent metus ligula, auctor vitae, lacinia sed, hendrerit a, felis. Etiam sapien. Proin et sem vitae dolor sodales venenatis. Integer luctus aliquam risus.",
      "Maecenas mi massa, fermentum eu, venenatis et, cursus id, ipsum. Morbi vehicula justo faucibus mauris. Donec non neque. Fusce id mi ut neque tincidunt posuere. Suspendisse quis enim. Cras porttitor. Sed quis velit. Aliquam vel augue at wisi blandit suscipit. Duis ut justo. Class aptent taciti sociosqu ad litora torquent per conubia nostra, per inceptos hymenaeos. Etiam bibendum wisi quis augue. Nulla lorem odio, sollicitudin vitae, vehicula nec, dapibus ultricies, purus. In vitae tellus at odio cursus congue. Quisque tincidunt tempus metus. Aenean et nulla nec dolor dapibus ultricies. Phasellus commodo vulputate arcu. Sed enim.  Phasellus quis leo. Aliquam iaculis, turpis nec aliquet rutrum, pede risus porta diam, id ullamcorper erat est sed eros. Fusce ornare.",
      "Suspendisse porta, dolor sed fringilla ultrices, augue mauris gravida dolor, vel sollicitudin magna dui sit amet nunc. Mauris mollis condimentum risus. Integer ipsum. Quisque malesuada, erat ac dictum pulvinar, magna nisl fermentum ligula, quis euismod mauris felis non diam. Nullam sapien turpis, rutrum vel, condimentum ac, bibendum vulputate, nulla. Vestibulum tortor ipsum, fermentum egestas, placerat ut, vulputate et, wisi. Aliquam erat volutpat. Donec consequat, ligula sit amet tincidunt aliquam, nunc lorem sagittis nunc, a ullamcorper erat ante ac felis. Donec eleifend. Nullam quam leo, lobortis non, condimentum at, tempus consectetuer, orci. Quisque ut lorem. Donec nisl. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Donec porta, libero eget feugiat posuere, felis arcu pulvinar odio, vel dapibus enim dui nec turpis.",
      "Duis leo. Cras nec odio. Nullam pretium lacinia est. Fusce aliquet, metus et vestibulum lobortis, ante erat vestibulum eros, eu sodales eros turpis id massa. Quisque est.  Vivamus eu lacus. Nulla nisl. Nam eros. Aliquam sit amet neque vel magna dictum ultricies. Praesent magna mauris, sollicitudin ac, commodo eu, bibendum sit amet, lectus. Suspendisse potenti. Fusce congue leo quis libero nonummy adipiscing. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Nunc a orci. Ut at erat sit amet nunc scelerisque malesuada.  Phasellus odio nisl, porta eget, laoreet nec, vehicula non, risus.  Etiam dolor mauris, consectetuer eget, tincidunt sed, egestas quis, neque. Ut egestas ante ac libero. Proin mattis volutpat metus.",
      "Sed tempor metus eget wisi. Duis cursus. Nam nunc. Nulla placerat wisi sed est. Aenean risus. Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Proin erat dolor, ultricies a, rutrum sed, posuere eget, metus. Donec sagittis nunc ac tortor.  Aliquam erat volutpat. Curabitur consectetuer, augue nec viverra eleifend, dolor dolor volutpat orci, dapibus pellentesque eros pede a arcu. Nullam augue. Etiam eget nulla vel mi porta hendrerit.  Phasellus cursus scelerisque tortor. Maecenas ut leo.",
      "Donec libero. Quisque vitae est quis dui bibendum suscipit. Fusce leo felis, sagittis non, vehicula ac, ultricies vitae, diam. Aenean congue libero et metus. Nulla convallis libero a lacus. Donec hendrerit lorem sit amet leo. Mauris libero. Pellentesque pulvinar molestie dolor. Proin nibh mauris, ornare at, pretium sit amet, porttitor vel, mi. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.",
      "Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Aliquam interdum porttitor tortor. Donec ultricies justo eget sapien. Proin ac est. Aliquam erat volutpat. In tempus scelerisque ligula. Morbi scelerisque urna.  Duis ac nisl. Donec sed leo. Fusce posuere orci mollis nunc. Sed arcu enim, pharetra nec, aliquam eu, consectetuer sit amet, eros.  Sed id enim. Etiam mattis est at elit. Pellentesque est risus, pellentesque nec, dignissim vitae, egestas vitae, sapien. Maecenas et eros non libero iaculis facilisis. Mauris porttitor tempor justo.  Sed sollicitudin neque nec libero.",
      "Mauris ac ipsum. Duis ultrices erat ac felis.  Donec dignissim luctus orci. Fusce pede odio, feugiat sit amet, aliquam eu, viverra eleifend, ipsum. Fusce arcu massa, posuere id, nonummy eu, pulvinar ut, wisi. Sed dui. Vestibulum nunc nisl, rutrum quis, pharetra eget, congue sed, dui. Donec justo neque, euismod eget, nonummy adipiscing, iaculis eu, leo. Duis lectus. Morbi pellentesque nonummy dui.",
      "Aenean sem dolor, fermentum nec, gravida hendrerit, mattis eget, felis. Nullam non diam vitae mi lacinia consectetuer. Fusce non massa eget quam luctus posuere. Aenean vulputate velit. Quisque et dolor. Donec ipsum tortor, rutrum quis, mollis eu, mollis a, pede. Donec nulla. Duis molestie. Duis lobortis commodo purus. Pellentesque vel quam. Ut congue congue risus. Sed ligula. Aenean dictum pede vitae felis. Donec sit amet nibh.  Maecenas eu orci. Quisque gravida quam sed massa.",
      "Nunc euismod, mauris luctus adipiscing pellentesque, augue ligula pellentesque lectus, vitae posuere purus velit a pede. Phasellus leo mi, egestas imperdiet, blandit non, sollicitudin pharetra, enim. Nullam faucibus tellus non enim. Sed egestas nunc eu eros. Nunc euismod venenatis urna. Phasellus ullamcorper. Vivamus varius est ac lorem. In id pede eleifend nibh consectetuer faucibus. Phasellus accumsan euismod elit. Etiam vitae elit. Integer imperdiet nibh. Morbi imperdiet orci euismod mi.",
      "Donec tincidunt tempor metus. Aenean egestas cursus nulla. Fusce ac metus at enim viverra lacinia.  Vestibulum in magna non eros varius suscipit. Nullam cursus nibh.  Mauris neque. In nunc quam, convallis vitae, posuere in, consequat sed, wisi. Phasellus bibendum consectetuer massa. Curabitur quis urna. Pellentesque a justo.",
      "In sit amet dui eget lacus rutrum accumsan.  Phasellus ac metus sed massa varius auctor. Curabitur velit elit, pellentesque eget, molestie nec, congue at, pede. Maecenas quis tellus non lorem vulputate ornare. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Etiam magna arcu, vulputate egestas, aliquet ut, facilisis ut, nisl. Donec vulputate wisi ac dolor. Aliquam feugiat nibh id tellus. Morbi eget massa sit amet purus accumsan dictum. Aenean a lorem. Fusce semper porta sapien.",
      "Curabitur sit amet libero eget enim eleifend lacinia. Vivamus sagittis volutpat dui. Suspendisse potenti. Morbi a nibh eu augue fermentum posuere. Curabitur elit augue, porta quis, congue aliquam, rutrum non, massa. Integer mattis mollis ipsum. Sed tellus enim, mattis id, feugiat sed, eleifend in, elit. Phasellus non purus sed elit viverra rhoncus. Vestibulum id tellus vel sem imperdiet congue. Aenean in arcu. Nullam urna justo, imperdiet eget, volutpat vitae, semper eu, quam. Sed turpis dui, porttitor ut, egestas ac, condimentum non, wisi. Fusce iaculis turpis eget dui.  Quisque pulvinar est pellentesque leo. Ut nulla elit, mattis vel, scelerisque vel, blandit ut, justo. Nulla feugiat risus in erat.",
      "Curabitur hendrerit. Morbi fringilla enim quis nunc. Phasellus at dui. Donec commodo augue at nunc. Nunc in sapien et magna mollis sagittis. Morbi eu elit. Phasellus lacus.  Donec a quam. Etiam pulvinar sapien. Sed nibh magna, viverra vitae, auctor eget, eleifend nec, lorem. Curabitur fringilla dui a odio.  Nunc semper condimentum arcu. Curabitur vitae lectus sit amet turpis pretium condimentum. Nullam imperdiet mattis neque. Proin eget magna porta erat rhoncus consectetuer. Aenean pulvinar erat vitae mi.",
      "In eget turpis non tortor iaculis tincidunt. Ut egestas feugiat magna. Duis enim nibh, gravida vel, nonummy non, sollicitudin ac, enim. Sed in tortor sed tellus eleifend cursus. Morbi a dolor at nibh molestie faucibus. Duis sed wisi. Duis fringilla, dui et malesuada dignissim, elit eros dictum lacus, rhoncus imperdiet pede elit nec tellus. Aenean at ligula eget nulla imperdiet faucibus. Quisque vestibulum lorem ac leo. Sed laoreet neque ut magna.",
      "Aenean eget justo id lorem congue tristique. Maecenas sit amet nunc. Aenean bibendum risus. Nam convallis, mi sed ultrices sodales, metus nibh placerat dui, eu hendrerit erat enim vel libero. Duis placerat sem vitae wisi imperdiet condimentum. Aliquam pellentesque dui ac diam eleifend venenatis. Nulla facilisis posuere sapien. Cras euismod. Praesent ut enim. Aliquam ut ipsum quis urna interdum vehicula. Fusce eget sem.  Nullam accumsan ullamcorper turpis.",
      "Integer posuere, metus ac rhoncus auctor, mi tellus scelerisque nunc, venenatis elementum tortor lorem eu erat.  Sed consectetuer risus vitae orci. Nullam tortor mauris, interdum at, imperdiet in, convallis eget, massa. Aliquam suscipit, magna nec blandit volutpat, lectus neque suscipit nunc, sit amet cursus nisl erat eget risus. Vestibulum leo lectus, accumsan ut, pharetra vel, elementum sed, quam. Maecenas condimentum orci at enim. Maecenas ut nunc. Vivamus pede. Integer vel purus vel mi mollis vestibulum. Sed laoreet ultricies nibh. Suspendisse non nisl quis ligula fermentum facilisis. Vestibulum sem nibh, porttitor et, fermentum a, ultricies id, augue.",
      "In accumsan convallis metus. Aenean est. Donec pharetra porta odio. Duis nunc nisl, imperdiet ac, tincidunt vitae, varius sit amet, felis. Curabitur wisi. Ut iaculis, nunc in lacinia egestas, elit enim tincidunt turpis, at luctus ipsum augue condimentum metus. Aenean lorem wisi, cursus sit amet, mollis nec, porta ac, augue. Vivamus massa. Praesent rhoncus imperdiet orci.  Aenean pharetra dolor ut sapien. Maecenas egestas augue semper dolor.",
      "Vestibulum at lectus. Vestibulum dapibus placerat magna. Suspendisse dolor urna, condimentum sit amet, euismod a, adipiscing a, enim. Aliquam erat volutpat. Donec imperdiet dolor non mi. Phasellus magna metus, dictum sit amet, laoreet non, dictum vel, dui. Suspendisse potenti. Nunc turpis risus, porta vel, pharetra id, eleifend vitae, justo. Duis pulvinar dolor sit amet urna. Integer eu eros. Nulla facilisi. Duis dui.  Nullam vitae quam. Morbi a nunc in elit sodales euismod. Nunc sed orci. Etiam malesuada metus vitae felis. Suspendisse imperdiet velit in tellus.",
      "Nullam elit orci, condimentum vitae, accumsan quis, gravida non, velit. Morbi pellentesque accumsan elit. Aenean est purus, eleifend ac, dictum at, dignissim sed, dolor. Vestibulum volutpat sapien quis augue. Maecenas vulputate accumsan sapien. Nam mattis, lacus non iaculis aliquet, mi elit varius lectus, eu malesuada dolor nunc at wisi. Aliquam ligula. Mauris nisl elit, molestie vitae, gravida sit amet, facilisis convallis, enim. Sed urna. Praesent et augue. Fusce pellentesque. Maecenas varius orci eget nisl. Donec tempor rhoncus turpis. Integer nibh. Cras metus erat, tincidunt et, scelerisque quis, bibendum sed, dui. Suspendisse potenti.",
      "Integer ac diam. Nullam porttitor dolor eget metus. Nulla sed metus quis tortor lacinia tempor. Mauris mauris dui, faucibus vitae, aliquet sit amet, placerat a, ante. Nunc placerat tincidunt neque. Mauris egestas dolor ut ipsum cursus malesuada. Curabitur odio. Nunc lobortis. Sed mattis tempor felis.  Mauris dolor quam, facilisis at, bibendum sit amet, rutrum ornare, pede. Suspendisse accumsan sagittis velit. Pellentesque varius laoreet lorem. Vivamus egestas sapien id diam.",
      "Integer viverra, felis ac tempus cursus, neque risus interdum turpis, eget venenatis tellus velit in neque.  Nulla feugiat luctus tellus. Nam pulvinar lacus id leo. Vestibulum at ligula. Duis laoreet tincidunt enim. Suspendisse at nisl molestie est laoreet laoreet. Suspendisse euismod metus vel nisl. Aenean ullamcorper imperdiet massa. Aliquam nibh. Donec quis erat. Nunc sodales auctor ante.",
      "Nam quis ante. Nullam interdum quam in eros.  Sed eleifend libero eu tellus consequat fermentum. Nullam pellentesque risus ut augue. Vestibulum eu tellus. Integer eleifend suscipit urna. Fusce porttitor leo et odio. Vivamus vehicula justo a nisl. In rutrum, purus ut dictum auctor, dolor velit accumsan dolor, eu convallis augue dui ac lectus. Nullam eleifend pellentesque ligula. Nam quis magna. Donec elementum dapibus erat. Pellentesque vel ipsum nec orci fermentum accumsan. Nunc porta magna eu neque.  Nam id erat eu mi aliquet cursus. Morbi ut felis. Vestibulum in ipsum.",
      "Donec vel augue. Morbi a turpis sed libero consequat porta. Quisque lacinia consequat odio. Sed vehicula sollicitudin purus. Vestibulum eget est. In hac habitasse platea dictumst. Sed blandit, tortor a auctor imperdiet, wisi nibh ornare leo, ac dictum nibh enim eu orci. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas.  Aliquam tincidunt ullamcorper justo. Etiam accumsan lacus nec ante.  Ut dictum luctus mauris. Ut metus. Maecenas gravida. Proin iaculis.  Integer convallis, justo iaculis ullamcorper sollicitudin, lectus neque tincidunt mi, at condimentum sem quam vel diam. Aenean sit amet purus.",
      "Sed justo. Maecenas lacinia, turpis sed commodo congue, odio urna elementum nunc, vitae molestie velit nunc eu sem. Maecenas enim. Proin quis neque nec tortor sollicitudin volutpat. Sed at ante. Sed vitae mauris non ante egestas hendrerit.  Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. In venenatis facilisis magna. Phasellus purus. Cras quis mauris. Aliquam eget magna. Donec rutrum sagittis mi. Morbi elementum, est sit amet sollicitudin feugiat, orci magna semper risus, eu congue nulla metus vel elit. Nunc tempor ornare mi.  Integer justo odio, suscipit tincidunt, fermentum eu, tincidunt et, libero. Vestibulum vestibulum, urna et suscipit imperdiet, nulla ante fermentum erat, at laoreet lorem lectus sed metus. Fusce ante sem, posuere in, vehicula a, posuere sed, ante. Phasellus magna.  Maecenas sit amet diam. Nunc at nibh sit amet augue tristique gravida.",
      "Aenean adipiscing auctor est. Morbi quam arcu, malesuada sed, volutpat et, elementum sit amet, libero. Duis accumsan. Curabitur urna. In sed ipsum. Donec lobortis nibh. Duis mattis. Sed cursus lectus quis odio. Phasellus arcu. Praesent imperdiet dui in sapien. Vestibulum tellus pede, auctor a, pellentesque sit amet, vulputate sed, purus. Nunc pulvinar, dui at eleifend adipiscing, tellus nulla placerat massa, sed condimentum nulla tellus sed ligula. Nulla vitae odio sit amet leo imperdiet blandit. In vel massa. Maecenas varius dui at turpis. Sed odio.",
      "Quisque aliquam ipsum sed turpis.  Pellentesque laoreet velit nec justo. Nam sed augue. Maecenas rutrum quam eu dolor. Fusce consectetuer. Proin tellus est, luctus vitae, molestie a, mattis et, mauris. Donec tempor. Pellentesque habitant morbi tristique senectus et netus et malesuada fames ac turpis egestas. Duis ante felis, dignissim id, blandit in, suscipit vel, dolor. Pellentesque tincidunt cursus felis. Proin rhoncus semper nulla. Ut et est. Vivamus ipsum erat, gravida in, venenatis ac, fringilla in, quam. Nunc ac augue. Fusce pede erat, ultrices non, consequat et, semper sit amet, urna.",
      "Fusce adipiscing justo nec ante. Nullam in enim.  Pellentesque felis orci, sagittis ac, malesuada et, facilisis in, ligula. Nunc non magna sit amet mi aliquam dictum. In mi. Curabitur sollicitudin justo sed quam. Aenean imperdiet. Vestibulum ante ipsum primis in faucibus orci luctus et ultrices posuere cubilia Curae; Donec lacinia nonummy lectus. Proin vel urna. Fusce sit amet orci ac magna iaculis pharetra. Duis sagittis massa in tellus. Aenean vel velit vel felis consectetuer pharetra."
   };
}
