/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - integrating C++ with Tcl
 * 
 *	FILE: "list_v.h"
 *									  created: 24/12/95 {2:20:12 am} 
 *								  last update: 05/08/98 {09:35:48 AM} 
 *	Author:	Vince Darley
 *	E-mail:	<darley@fas.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www:	<http://www.fas.harvard.edu/~darley/>
 *	
 *	Description: doubly-linked list of 'void *' around which we
 *	  can wrap any pointer class very cheaply.  This is memory
 *	  efficient and type-safe, if done correctly. 
 * 
 *	History
 * 
 *	modified by	 rev reason
 *	-------- --- --- -----------
 *	30/1/97	 VMD 1.0 original
 * ###################################################################
 */

#ifndef _cpptcl_plist_
#define _cpptcl_plist_
#include "cpptclInt.h"
#include <stdlib.h>

// Some forward references.  We're defining three
// mutually-using classes.
class list_v;
class list_pos_v;

//ostream&  operator << (ostream&, const list_v&);

/* 
 * -------------------------------------------------------------------------
 *   
 * "list_node_v" --
 *  
 *  list_nodes are little more than cells containing a T-value 
 *  and pointers to the next and previous nodes.
 *  They are entirely private, except that the public classes 
 *  can refer to them.
 *  
 * -------------------------------------------------------------------------
 */
DLL_IMPORT_EXPORT
struct list_node_v {
	friend class list_v;
	friend class list_pos_v;
	//friend class ostream& operator <<(ostream&, const list_v&);
  protected:
	void* _t;
	list_node_v *_next, *_prev;
	
	list_node_v(list_node_v *prev, const void* tt,  list_node_v *next)
		:_t((void*)tt),_next(next),_prev(prev) {}
	virtual ~list_node_v(void) {}  // Just delete this node
	
};


/* 
 * -------------------------------------------------------------------------
 *   
 * "class list_v" --
 *  
 *	A class	of doubly-linked lists,	suitable for holding pointers to things.
 *	We make	the	assumption that	copying	T's	is cheap and sensible.
 *	This is	a good class for Animal*'s,	for	example; it	is
 *	nigh-useless for Animals.
 * 
 *	It is your basic classical form	abstraction.  In particular, lists
 *	may	be passed around by	value freely, and assigned fairly casually.
 *	Be careful,	though:	 copying a list	costs linear time (plus	the	time
 *	required to	copy the elements).
 * 
 *	This isn't the complete	abstract data type of lists.  There	are	any
 *	number of functions	that should	be in the abstraction -- including
 *	some things	as simple as computing the length of the list.
 *	(Actually, it would	be sensible	and	easy to	maintain a count of	the
 *	length of the list as a	data, so that computing	the	length was
 *	constant-time.)	 Feel free to add more list	functions as you wish.
 * 
 *	lists are mainly manipulated via list_pos<T>'s.	  These	capture	most
 *	of the uses	of pointers	to elements	of the list; e.g., you can use them
 *	as iterators to	loop over the list;	as indices to reference, modify,
 *	or delete elements;	and	so on.	Again, the class isn't complete; it	lacks
 *	such basic functions as	operator==.	 Feel free to add them.
 * 
 *	WARNING: list_pos's	are	safe as	long as	there are no deletions on the list.
 *	However, a list_pos	to a deleted element is	NOT	told that the element
 *	is deleted.
 *  
 * -------------------------------------------------------------------------
 */
DLL_IMPORT_EXPORT
class list_v {
  public:

	// Create lists with 0-4 elements....
	list_v ():_headPtr(NULL),_tailPtr(NULL) {}
	/*
	list_v (void* t1); 
	list_v (void* t1, void* t2); 
	list_v (void* t1, void* t2, void* t3); 
	list_v (void* t1, void* t2, void* t3, void* t4); 
	*/
	// Usual Classical Form stuff...
	virtual ~list_v(void) {trashlist();}
	
	
	// Copy and assignment.  Warning; these duplicate list structure,
	// so be a bit careful about calling them; they can be expensive.
	list_v(const list_v&);
	list_v& operator=(const list_v& l);
	
	// The following constructor creates a list from a length n array of void*'s
	//list_v(void* *at, int  n);
	
	// MUTATORS
	
	// Adding and removing things to lists
	
	
	void append(void* t);  // Add t at the end of *this.
	void remove(void* t);        //rem t from *this.
	// Tests...
	bool isEmpty() const { return (_headPtr == NULL ? true : false); }
	bool isNonEmpty() const {return (_headPtr == NULL ? false : true); }
	// Return true iff *this contains an element == to t.
	// Requires T::operator== or the like.
	bool contains(const void* const& t) const;
		
	// Returns the position of 't' in the list, or -1 if not there
	int positionOf(const void* const& t) const;	
	
	// OPERATIONS: 
	
	const list_v& operator+(const list_v& lt) const;
	list_v& operator+=(const list_v& lt);
	// list concatenation.  Copies both *this and lt, so it's not 
	// cheap.
	
	void appendAndDestroy(list_v& lt);
	// Append lt to *this.  Modifies *this (making it have lt at the end),
	// and sets lt to a null list.  This is quite cheap.
	// Hopefully, the alarming name will warn users that
	// it's got an unusual side effect.
		
	// Reverses the list so that the old head element is the new tail
	// element and vis versa
	void reverseInPlace();
	
	// toArray creates an array of T's from list. 
	//void* *toArray() const;
	// returns n'th element currently performs no error checking
	void* nth_element(int n) const;
	
	// The length of the list (duh).
	int length() const;

	/// Empty the list safely
	void remove_all_contents(void) {trashlist();initToNulllist();}
	//friend ostream& operator<<(ostream& o, const list_v& l);
  protected: 
	friend class list_node_v;
	friend class list_pos_v;
	
	// Head pointer: NULL for empty list, non-NULL for non-empty list.
	list_node_v *_headPtr;
	
	// Tail pointer: NULL for empty list, non-NULL for non-empty list.
	list_node_v *_tailPtr;
	
	// ABSTRACTION FUNCTION:
	// This implements a mutable list of T's.
	// The elements of the list are given, in order,
	// by starting with this->headPtr's T element, and following
	// next-pointers until the NULL marking the end of the list.
	
	// REP INVARIANT:
	// If the list is empty, both headPtr and tailPtr are null.
	// Otherwise, headPtr points to the head and tailPtr
	// to the tail.
		
	void initToNulllist(void) {_headPtr = NULL;_tailPtr = NULL;}
	
	// UNSAFE.
	// Turns *this into a null list, losing track of any previous
	// contents.  That is, it NULLs both pointers.
	// Afterwards, the list is a well-represented empty
	// list, and (e.g.) it is safe to iterate over and use listpos's on.
	
	void create_first_element(void* t);
	// UNSAFE.
	// Turns an empty list into a singleton containing t.
	// It does no checking, and will do bad things (probably
	// losing previous list contents) if the list is nonempty.
	
	void trashlist();
	// UNSAFE.
	// Destructor body.
	// Deletes the elements of *this,
	// leaving *this in an inconsistent state.
	
	void copyToNullFrom(const list_v& l);
	// UNSAFE.
	// copy constructor and assignment component.
	// Assumes that *this is a proper null list, and copies
	// l into it.
	virtual list_v& _copy_me(void) const=0;
	virtual list_node_v* _alloc_node(list_node_v *prev, void* t, 
									list_node_v *next) const=0;
	// Returns a new list_node_v containing t, doing appropriate
	// memory allocation checks.  It doesn't really concern *this,
	// but we want to hide it inside the abstraction barrier, so it's
	// here.
};


/* 
 * -------------------------------------------------------------------------
 *   
 * "list_pos_v" --
 *  
 *  
 * -------------------------------------------------------------------------
 */
DLL_IMPORT_EXPORT
class list_pos_v {
  protected:
	friend class list_v;
	
	list_pos_v(bool consty);
	// Construct a list_pos to node n of list l, with
	// constant-ness consty.
	// Use list_pos_v lp = l.head()   (or l.tail() if that's what you mean)
	// for internal use only.
	list_node_v* _cursor;
	list_v* _tolist;
	
	bool isConstPos; // True iff the list_pos should not be used for mutators.

	// Assignment and copying is normal.
	list_pos_v(const list_pos_v& lp);
	void set_equal(const list_pos_v& l);
	// Constructor to head of list
	list_pos_v(list_v& l);
	// Constructor to head of const list
	list_pos_v(const list_v& l);
  public:	
	// Returns true if this list_pos is actually on the list it thinks it is.
	bool reallyOnlist() const;
	
	// Operations
	bool operator<(const list_pos_v& b) const;
	bool operator==(const list_pos_v& b) const {return (_cursor == b._cursor);}
			
	// Destroying the listpos doesn't do anything to the list
	virtual ~list_pos_v(void) {}
	
	// get the element at this position.
	// We provide two names because (A) each made sense in
	// some phrasing, and (B) we couldn't think of a good
	// single name when we wrote that part of the code.
	// Bad style, really.
	// These functions blow up if the list_pos is offlist.
	virtual void* here_v() const;
	virtual void* item_v() const {return here_v();}
	operator void* () const { return here_v();}
	bool operator==(const void* const& t) const { return (here_v() == t);}
	bool operator==(void* & t) const { return (here_v() == t);}
	// Get back to the list that this list_pos points to.  
	virtual list_v* mylist_v() const {return _tolist;}
	
	// The main test is, does the list_pos point somewhere
	// on the list.  The guarantee is,
	// if it is onlist, then it's here() will work right.
	// e.g., if mylist is empty, mylist.head() is offlist().
	// Note also that if lp points to an element of a list, 
	// and that element is deleted, lp will still register as
	// onlist.
	bool offlist() const {return (_cursor == NULL);}
	bool onlist() const {return  (_cursor != NULL);}
	
	// so a list_pos can be used just like a pointer in a for loop
	operator bool () const  { return onlist();};
	// old version which is better without ANSI bool
	//operator int () const  { return onlist();};
	const list_pos_v& operator ++ (int) {goNext(); return *this;};
	list_pos_v& operator ++ () {goNext(); return *this;};
	const list_pos_v& operator -- (int) {goPrev(); return *this;};
	list_pos_v& operator -- () {goPrev(); return *this;};
	// return true if *this will refuse to allow modifying
	// operations on the underlying list to be used through it. 
	bool isConst() const {return this->isConstPos;};
	
	// Is the cursor at the head of the list (empty or not)?
	// if this->atHead() is true, this->insertPrev(t) will
	// make t the new head element of the list.
	// For example, if the list is empty, its listPos's are atHead.
	bool atHead() const {return (mylist_v()->_headPtr == _cursor);}
	
	// Dual to atHead();
	bool atTail() const {return (mylist_v()->_tailPtr == _cursor);}
	
	// MOVING THE LISTPOS AROUND
	
	// go forward or backward in the list.
	void goNext() {if (onlist()) _cursor = _cursor->_next;}
	void goPrev() {if (onlist()) _cursor = _cursor->_prev;}
	
	// MUTATING THE LIST ITSELF
	
	// Insert an element t after the current position.
	// Leave this list_pos pointed at the new element.
	// I'm not sure what the right behavior should be when
	// *this is off the list; so it causes an error.
	// However, when the list is empty, this adds a single element to it.
	void insertNext(void* t);
	
	// Insert t before the current position; leave *this
	// pointing to new position.
	// Like insertNext, mutatis mutandis.
	void insertPrev(void* t);
	
	void replaceWith(void* t);
	// replace referenced list element with t.
	// Doom, if the list_pos doesn't refer to an element
	// of a non-constant list.
	
	void delete0();
	// Delete the node of the list *this is pointing to.
	// Leaves *this pointing to nowhere; that's what the 
	// "0" in "delete0" means.  
	
	// WARNING: Dangling list_pos's are dangerous: 
	// if you delete a list_pos to element x, you might
	// have other list_pos's to x -- and that's trouble.
	// MISSING: isReallyOnlist() -- scan list and make sure 
	// that *this is a list_pos to somewhere on the list.
	
	void deleteAndGoNext();
	// like delete0, but moves to the next list_pos.
	
	// Returns a list consisting of this through b.
	//list_v through(const list_pos_v b) const;
	
};  


#endif

  
