/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - Integrating C++ with Tcl
 * 
 *	FILE: "meta_object.h"
 *									  created: 28/5/96 {8:42:15 pm} 
 *								  last update: 05/08/98 {09:48:35 AM} 
 *	Author:	Vince Darley
 *	E-mail:	<mailto:darley@fas.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www: <http://www.fas.harvard.edu/~darley/>
 *	
 * ===================================================================
 * Copyright (c) 1997  Vince Darley
 *  
 * See the file "license.terms" for information on usage and 
 * redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * ===================================================================
 *	Description: 
 * 
 * 
 *	History
 * 
 *	modified by	 rev reason
 *	-------- --- --- -----------
 *	19/12/95 VMD 1.0 original
 * ###################################################################
 */

#ifndef _Cpptcl_object_
#define _Cpptcl_object_

#include "cpptclInt.h"
#include "cpplist.h"
#include "cpptcl_metaobject.h"
#include "cpptcl_member_base.h"
#include "cpptcl_config_mem.h"
#include "meta_type.h"

//@Section: Cpptcl library
//@Man: 
/** 
 * -------------------------------------------------------------------------
 * 
 * "class cpptcl_reader" --
 * 
 *  Base class for reading objects from an argument list.  Any object
 *  which is registered with the Cpptcl system is probably best read
 *  using one of the descendants of this class.  This allows error-
 *  checking and type-conversion to be performed automatically.
 *  
 *  For instance:
 *  
 *  	object<cpptcl_test> t;
 *		arg >> t >> done;
 *		cpptcl_test* obj = t;
 *
 *  will read a single object, of type cpptcl_test (or any of its
 *  descendants) from the argument stream, and fail with an 
 *  appropriate error message if the read or conversion failed.
 *  
 *  After the 'done' manipulator, we are guaranteed that 'obj'
 *  is a pointer to a cpptcl_test object.
 *  
 *  There are three types of information that can be read using
 *  descendants of this class: objects (as above), class members,
 *  and type information.  These are usually read using objects
 *  of type object<T>, member<T> and meta<T> respectively.
 *  
 *  See those classes for further information
 * 
 * --Version--Author------------------Changes-------------------------------
 *    1.0     <darley@fas.harvard.edu> original
 * -------------------------------------------------------------------------
 */
class cpptcl_reader {
  public:
	///
	const meta_object* required;
	///
	cpptcl_reader(const char* type) {
		required = tcl_interaction::metaobject->find_meta_info(type);
	}
	///
	cpptcl_reader(const meta_object& type):required(&type) {}
	///
	virtual const char* type(void) const=0;
	///
	virtual ~cpptcl_reader(void) {}
	///
	virtual tcl_args& read_name(tcl_args&) = 0;
	///
	virtual tcl_args& write(tcl_args&) = 0;
	///
	virtual bool have_item(void) const=0;
	///
	virtual void clear(void)=0;
	
};

//@Section: Cpptcl library
//@Man: 
class meta_reader : public cpptcl_reader {
  public:
	///
	const meta_object* meta;
	///
	bool have_item(void) const {return (meta ? true : false);}
	///
	operator const meta_object* (void) const { return meta;}
	///
	meta_reader(const char* type):cpptcl_reader(type),meta(0) {}
	///
	meta_reader(const meta_object& type):cpptcl_reader(type),meta(0) {}
	///
	const char* type(void) const {
		return (meta ? meta->type : 0);
	}
	///
	tcl_args& read_name(tcl_args&);
	///
	tcl_args& write(tcl_args&);
	///
	~meta_reader(void) {}
	///
	void clear(void) {meta = 0;}
	
};

//@Section: Cpptcl library
//@Man: 
template <class T>
class ometa : public meta_reader {
  public:
	///
	ometa(void):meta_reader(T::_type) {}
	///
	~ometa(void) {}
};

//@Section: Cpptcl library
//@Man: 
class object_reader : public cpptcl_reader  {
  public:
	///
    tcl_interaction* obj;
	///
	object_reader& operator = (tcl_interaction* o) { obj = o; return *this;}
	///
	bool have_item(void) const {return (obj ? true : false);}
	///
	operator tcl_interaction* (void) const { return obj;}
	///
	object_reader(const char* type)
		:cpptcl_reader(type),obj(0) {}
	///
	object_reader(const meta_object& type)
		:cpptcl_reader(type),obj(0) {}
	///
	const char* type(void) const {
		return (obj ? (const char*)(obj->type()) : (const char*)0);
	}
	///
	tcl_args& read_name(tcl_args&);
	///
	tcl_args& write(tcl_args&);
	///
	~object_reader(void) {}
	///
	void clear(void) {obj = 0;}
};

//@Section: Cpptcl library
//@Man: 
template <class T>
class object : public object_reader {
  public:
	///
    object(void):object_reader(T::_type) {}
	///
	~object(void) {}
	///
	operator T* (void) const {return (T*) obj;}
	///
	operator T& (void) const {return (T&) *obj;}
};

//@Section: Cpptcl library
//@Man: 
class member_reader : public cpptcl_reader  {
  public:
	///
	cpp_mem* obj;
	///
	member_reader(tcl_base* b, const char* type)
		:cpptcl_reader(type),obj(0),container(b) {}	
	///
	member_reader(tcl_base* b, const meta_object& type)
		:cpptcl_reader(type),obj(0),container(b) {}	
	///
	bool have_item(void) const {return (obj && container ? true : false);}
	///
	const char* type(void) const {
		return (obj ? (const char*)(obj->type()) : (const char*)0);
	}
	///
	tcl_args& read_name(tcl_args&);
	///
	tcl_args& write(tcl_args&);
	///
	~member_reader(void) {}
	///
	mutable tcl_base* container;
	///
	void clear(void) {obj = 0;}
};

class config_opt : public member_reader {
  public:
	///
    config_opt(tcl_base* b):member_reader(b,cpp_config_mem::_type) {}
	///
	~config_opt(void) {}
	///
	config_opt& operator () (tcl_base* b) {
		container = b; return *this;
	}
	///
	operator cpp_config_mem* (void) const { 
		obj->operator()((tcl_object*)container); 
		return (cpp_config_mem*) obj;
	}
	///
	tcl_args& write(tcl_args&);
};

//@Section: Cpptcl library
//@Man: 
template <class T>
class member : public member_reader {
  public:
	///
	member(tcl_base* b):member_reader(b,T::_type) {}
	///
	~member(void) {}
	///
	operator T* (void) const { obj->operator()((tcl_object*)container); return (T*) obj;}
	///
	member<T>& operator () (tcl_base* b) {
		container = b; return *this;
	}	
};

//@Section: Cpptcl library
//@Man: 
tcl_args& operator>> (tcl_args& arg, cpptcl_reader& into);

inline tcl_args& operator>> (tcl_args& arg, cpptcl_reader& into) {
	return (arg.state() ? arg : 
	  (into.have_item() ? into.write(arg) : into.read_name(arg)));
}

#include "cpptcl_config_mem.h"
//@Section: Cpptcl library
//@Man: 
const Tcl_Obj* operator >> (const Tcl_Obj* o, member_reader& mem);

#endif

