/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains functions common to the device drivers.
 */

#include "types.h"
#include "drivutil.h"

/*
 *	Routine 'Init_GState' initilizes a state structure for
 *	solid lines, color black:
 */

Init_GState (GState, Default_Line_Width)
struct Graphics_State *GState;
unsigned int Default_Line_Width;
{
	GState->Line_Width = (unsigned long) Default_Line_Width << 10;
	GState->Dashline_Len = 0;
	GState->Dashline_Phase = 0;
	GState->Color.Red_Component = 0;
	GState->Color.Green_Component = 0;
	GState->Color.Blue_Component = 0;
	GState->Flags = 0;
}

/*
 *	Routine 'Save_GState' saves the current graphics state
 *	on the graphics stack:
 */

Save_GState (GState, Identifier)
struct Graphics_State *GState;
pointer Identifier;
{
	auto   struct Graphics_State *State_Ptr;
	extern char *Mem_Alloc();

	State_Ptr = (struct Graphics_State *) Mem_Alloc (sizeof (struct Graphics_State));
	*State_Ptr = *GState;
	State_Ptr->Pred = Graphics_SP;
	State_Ptr->Ident = Identifier;
	Graphics_SP = State_Ptr;
}

/*
 *	Routine 'Restore_GState' restores the top graphics state to
 *	the current state:
 */

int Restore_GState (GState, Identifier)
struct Graphics_State *GState;
pointer Identifier;
{
	auto   struct Graphics_State *State_Ptr;
	auto   int Status;
	msgcode DVIOUT_GSTACKUNDERFLOW;

	if ((State_Ptr = Graphics_SP) == 0 || (Identifier != 0 && State_Ptr->Ident != Identifier)) {
		Message (DVIOUT_GSTACKUNDERFLOW);
		Status = 0;
	} else {
		*GState = *State_Ptr;
		Graphics_SP = State_Ptr->Pred;
		Mem_Free (State_Ptr);
		Status = 1;
	}
	return (Status);
}

Collapse_GStack ()
{
	auto   struct Graphics_State *State_Ptr;

	while ((State_Ptr = Graphics_SP) != 0) {
		Graphics_SP = State_Ptr->Pred;
		Mem_Free (State_Ptr);
	}
}

/*
 *	Routine 'Compare_Linestyle' compares two line styles and
 *	returns 0 if they are equal, 1 otherwise:
 */

int Compare_Linestyle (Len1, Desc1, Len2, Desc2)
unsigned int Len1, Desc1[], Len2, Desc2[];
{
	auto   int Index;

	if (Len1 != Len2)
		return (1);
	for (Index = 0; Index < Len1; Index++)
		if (Desc1[Index] != Desc2[Index])
			return (1);
	return (0);
}

/*
 *	Routine Scale_Linestyle scales a dashline specification to the
 *	line width. The function returns the scaled phase of the line.
 */

unsigned long Scale_Linestyle (Len, In_Spec, Width, Phase, Out_Spec)
unsigned int Len, In_Spec[], Phase;
unsigned long Width, Out_Spec[];
{
	auto   unsigned int Index, Size, Seg_Len, White_Count;

	White_Count = 0;
	Seg_Len = 0;
	for (Index = 0; Index < Len; Index++) {
		Size = In_Spec[Index];
		Seg_Len += Size;
		if ((Index & 0x01) != 0) {	 /* Invisible segment, increase length slightly */
			Size++;
			if (Phase >= Seg_Len)
				White_Count++;
		}
		Out_Spec[Index] = Size * Width;
	}
	return ((Phase + White_Count) * Width);
}
