// Copyright (c) 1991 by Parag Patel.  All Rights Reserved.
static const char rcsid[] = "$Header: dvi.C,v 1.24 91/02/22 15:57:33 hmgr Exp $";

// dvi scanning routines
//
// by Parag Patel

#include "defs.h"


static long mag = 1000;
static double magval = 1.0;


// print the specified page from the file "fp" at the specified
// magnification - this is called for any page in the DVI file provided
// that the "fp" has been set to the start of a page in the DVI file -
// thus pages may be printed in any random order
// 
static void dopage(FILE *fp, long page)
{
    // clear and initialize the global state for a new page
    cleardvi();
    clearfonts();

    // if this is not the first page, then eject the previous
    // page from the printer
    // 
    static first = TRUE;
    dev_newpage((page & 0x01) || page == 0, first);
    first = FALSE;

    // now go through the DVI file executing opcodes...
    register int opcode;
    while ((opcode = (int)getuval(1, fp)) != EOP)
    {
	debug(7, "    H=%f V=%f  W=%f X=%f  Y=%f Z=%f", H, V, W, X, Y, Z);
	debug(6, "opcode = %d", opcode);

	if (opcode >= SET0 && opcode <= SET127)
	{
	    // typeset (print) a character
	    typeset(opcode, TRUE, magval);
	    continue;
	}
	if (opcode >= FONT0 && opcode <= FONT63)
	{
	    // switch to a new font
	    newfont(opcode - FONT0);
	    continue;
	}
	switch (opcode)
	{
	    // print (typeset) a character and move the cursor
	Case SET1: 
	    typeset(getuval(1, fp), TRUE, magval);
	Case SET2: 
	    typeset(getuval(2, fp), TRUE, magval);
	Case SET3: 
	    typeset(getuval(3, fp), TRUE, magval);
	Case SET4: 
	    typeset(getsval(4, fp), TRUE, magval);

	    // just typeset a character
	Case PUT1: 
	    typeset(getuval(1, fp), FALSE, magval);
	Case PUT2: 
	    typeset(getuval(2, fp), FALSE, magval);
	Case PUT3: 
	    typeset(getuval(3, fp), FALSE, magval);
	Case PUT4: 
	    typeset(getsval(4, fp), FALSE, magval);

	    // print a rule
	Case SETRULE: 
	    typerule(fp, TRUE, magval);
	Case PUTRULE: 
	    typerule(fp, FALSE, magval);

	Case NOOP: 

	Case PUSH: 
	    pushdvi();
	Case POP: 
	    popdvi();

	    // horizontal moves (negative == move left)
	Case RIGHT1: 
	    moveright(Getsval(1, fp) * magval);
	Case RIGHT2: 
	    moveright(Getsval(2, fp) * magval);
	Case RIGHT3: 
	    moveright(Getsval(3, fp) * magval);
	Case RIGHT4: 
	    moveright(Getsval(4, fp) * magval);

	    // move horizontal based on the var W
	Case W0: 
	    moveright(W);
	Case W1: 
	    moveright(W = Getsval(1, fp) * magval);
	Case W2: 
	    moveright(W = Getsval(2, fp) * magval);
	Case W3: 
	    moveright(W = Getsval(3, fp) * magval);
	Case W4: 
	    moveright(W = Getsval(4, fp) * magval);

	    // move horizontal based on the var X
	Case X0: 
	    moveright(X);
	Case X1: 
	    moveright(X = Getsval(1, fp) * magval);
	Case X2: 
	    moveright(X = Getsval(2, fp) * magval);
	Case X3: 
	    moveright(X = Getsval(3, fp) * magval);
	Case X4: 
	    moveright(X = Getsval(4, fp) * magval);

	    // move vertically (negative == towards top of page)
	Case DOWN1: 
	    movedown(Getsval(1, fp) * magval);
	Case DOWN2: 
	    movedown(Getsval(2, fp) * magval);
	Case DOWN3: 
	    movedown(Getsval(3, fp) * magval);
	Case DOWN4: 
	    movedown(Getsval(4, fp) * magval);

	    // move vertically based on the var Y
	Case Y0: 
	    movedown(Y);
	Case Y1: 
	    movedown(Y = Getsval(1, fp) * magval);
	Case Y2: 
	    movedown(Y = Getsval(2, fp) * magval);
	Case Y3: 
	    movedown(Y = Getsval(3, fp) * magval);
	Case Y4: 
	    movedown(Y = Getsval(4, fp) * magval);

	    // move vertically based on the var Z
	Case Z0: 
	    movedown(Z);
	Case Z1: 
	    movedown(Z = Getsval(1, fp) * magval);
	Case Z2: 
	    movedown(Z = Getsval(2, fp) * magval);
	Case Z3: 
	    movedown(Z = Getsval(3, fp) * magval);
	Case Z4: 
	    movedown(Z = Getsval(4, fp) * magval);

	    // change current font
	Case FNT1: 
	    newfont(getuval(1, fp));
	Case FNT2: 
	    newfont(getuval(2, fp));
	Case FNT3: 
	    newfont(getuval(3, fp));
	Case FNT4: 
	    newfont(getsval(4, fp));

	    // special - used definable
	Case XXX1: 
	    special(fp, getuval(1, fp));
	Case XXX2: 
	    special(fp, getuval(2, fp));
	Case XXX3: 
	    special(fp, getuval(3, fp));
	Case XXX4: 
	    special(fp, getsval(4, fp));

	    // define a font - note that this will be a redefinition
	Case FNTDEF1: 
	    definefont(fp, getuval(1, fp), mag);
	Case FNTDEF2: 
	    definefont(fp, getuval(2, fp), mag);
	Case FNTDEF3: 
	    definefont(fp, getuval(3, fp), mag);
	Case FNTDEF4: 
	    definefont(fp, getsval(4, fp), mag);

	Default: 
	    quit("Unexpected opcode %d", opcode);
	}
    }
}


// read the DVI file and create a list with pointers to every
// page in the file - also calculate a section number for each page
//
static void loadpages(FILE *fp, long pageloc, long numpages, 
		long &sections, Pageinfo &pages)
{
    long loc = pageloc;
    long lastpage = MAXLONG;
    long i;

    sections = 1;

    // search backwards through the DVI file
    for (i = numpages - 1; loc >= 0 && i >= 0; i--)
    {
	(void)fseek(fp, loc, SEEK_SET);
	if (getuval(1, fp) != BOP)
	    quit("Expected BOP");

	// get the values of the internal TeX variables
	// - the first is a page number (c[0])
	long c[10];
	for (int j = 0; j < 10; j++)
	    c[j] = getsval(4, fp);
	loc = getsval(4, fp);	// location of previous page in file

	// new section if we see a page larger than the previous one
	if (c[0] >= lastpage)
	    sections++;
	lastpage = c[0];

	// save the "real" page number and the offset in the file
	pages[i].page = c[0];
	pages[i].section = sections;
	pages[i].loc = ftell(fp);
    }

    // now change the section numbers to be in the proper order
    for (i = 0; i < numpages; i++)
	pages[i].section = sections - pages[i].section + 1;
}


// dump only the specified pages from the DVI file
//
static void dumppages(FILE *fp, Pagespec &pages, 
			Pageinfo &allpages, long sections)
{
    long numpages = allpages.size();

    for (int i = 0; i < pages.size(); i++)
    {
	// get the page spec entry in normal or reverse order
	long pent = reverse ? pages.size() - i - 1 : i;

	// if the end is not specified, take the current maximum
	long endsect = pages[pent].endsection == MAXLONG ? sections
		: pages[pent].endsection;
	long endpage = pages[pent].endpage == MAXLONG ? numpages
		: pages[pent].endpage;

	// print only the last section? make sure we go through loop once
	long startsect = pages[pent].section;
	if (startsect == MAXLONG)
	    startsect = endsect;

	for (long s = startsect; s <= endsect; s++)
	{
	    long snum = reverse ? endsect - s + startsect : s;
	    for (long p = pages[pent].page; p <= endpage; p++)
	    {
		long pnum = reverse ? endpage - p + pages[pent].page : p;

		// look for this page/section in the list of allpages
		// - if section == MAXLONG - 1, then use the last section
		long ent;
		for (ent = numpages - 1; ent >= 0; ent--)
		    if (allpages[ent].page == pnum
			    && allpages[ent].section == snum)
			break;

		if (ent < 0)
		{
		    // print an error only if the user explicitly specified
		    // a page that does not exist - do not print an error
		    // if we are going through to the end
		    if (pages[pent].section == MAXLONG
			    || pages[pent].endsection <= pages[pent].section)
			if (pages[pent].endpage == MAXLONG
				|| pages[pent].endpage < pages[pent].page)
			    if (pages[pent].section <= sections)
				continue;

		    error("No page/section %ld.%ld in DVI file", pnum, snum);
		    continue;
		}

		// seek to that page and spit it out
		(void)fseek(fp, allpages[ent].loc, SEEK_SET);
		mesg(" [%ld.%ld", allpages[ent].page, allpages[ent].section);
		debug(1, "Page: %ld.%ld (%ld)", allpages[ent].page,
			allpages[ent].section, ent + 1);
		dopage(fp, allpages[ent].page);
		mesg("]");
	    }
	}
    }
}


// dump all the pages in the DVI file in either forward or reverse order
//
static void dumpall(FILE *fp, Pageinfo &allpages)
{
    long numpages = allpages.size();
    for (int i = 0; i < numpages; i++)
    {
	long p = reverse ? numpages - i - 1 : i;
	(void)fseek(fp, allpages[p].loc, SEEK_SET);
	mesg(" [%ld.%ld", allpages[p].page, allpages[p].section);
	debug(1, "Page: %ld.%ld (%ld)", allpages[p].page,
		allpages[p].section, p + 1);
	dopage(fp, allpages[p].page);
	mesg("]");
    }
}


// process the DVI file "fp" - verify the file, setup the fonts, then
// print pages in the file either backwards or forwards - print only
// that "pages" specified, or everything if none were specified
// 
void dodvi(FILE *fp, Pagespec &pages)
{
    // verify the preamble and id value
    if (getuval(1, fp) != PRE)
	quit("Not a DVI file");
    long version = getuval(1, fp);
    if (version != ID)
    {
	debug(7, "DVI file version = %ld instead of %d", version, ID);
	quit("Incorrect DVI file version");
    }

    // if we cannot seek, then we cannot do anything
    if (fseek(fp, 0, SEEK_END) != 0)
	quit("Cannot seek to end of DVI file");

    long floc = ftell(fp);
    debug(2, "DVI file len = %ld", floc);

    // now look for the real end of the DVI file (after the POSTPOST)
    register int opcode = FILLER;
    while (opcode == FILLER && floc > 0)
    {
	(void)fseek(fp, --floc, SEEK_SET);
	opcode = (int)getuval(1, fp);
    }
    if (opcode != ID)
    {
	debug(7, "DVI file version = %d instead of %d", opcode, ID);
	quit("Incorrect DVI file version");
    }

    // get the pointer to the start of the postamble
    (void)fseek(fp, floc -= 4, SEEK_SET);
    long postloc = getuval(4, fp);
    (void)fseek(fp, postloc, SEEK_SET);

    // verify that we are where we think we are
    if (getuval(1, fp) != POST)
	quit("Expected POST");

    // pointer to the last page in the DVI file
    long pageloc = getsval(4, fp);

    // read in the other values in the postamble
    double numerator = Getsval(4, fp);
    if (numerator <= 0)
	quit("Illegal numerator");

    double denominator = Getsval(4, fp);
    if (denominator <= 0)
	quit("Illegal denominator");

    mag = getsval(4, fp);
    if (usermag > 0)
	mag = usermag;
    magval = (double)mag;
    if (mag <= 0)
	quit("Illegal magnification");
    magval *= numerator / denominator / 25400000.0 * 473628672.0 / 1000.0;
    debug(3, "num=%f denom=%f mag=%ld/%f", numerator, denominator,
	    mag, magval);

    // worst-case page sizes - we ignore these for now
    long tallest = getsval(4, fp);
    long widest = getsval(4, fp);
    long maxdepth = getsval(2, fp);
    long numpages = getuval(2, fp);
    debug(3, "tallest=%ld widest=%ld maxdepth=%ld", tallest, widest, maxdepth);
    debug(1, "numpages=%ld", numpages);

    // scan the opcodes in the rest of the postamble - only font
    // definitions and specials are allowed here
    while ((opcode = (int)getuval(1, fp)) != POSTPOST)
	switch (opcode)
	{
	Case FNTDEF1: 
	    definefont(fp, getuval(1, fp), mag);
	Case FNTDEF2: 
	    definefont(fp, getuval(2, fp), mag);
	Case FNTDEF3: 
	    definefont(fp, getuval(3, fp), mag);
	Case FNTDEF4: 
	    definefont(fp, getsval(4, fp), mag);

	Case XXX1: 
	    special(fp, getuval(1, fp));
	Case XXX2: 
	    special(fp, getuval(2, fp));
	Case XXX3: 
	    special(fp, getuval(3, fp));
	Case XXX4: 
	    special(fp, getsval(4, fp));

	Case NOOP: 

	Default: 
	    quit("Illegal opcode = %d in font definitions", opcode);
	}

    // initialize the page pointer table
    Pageinfo allpages(numpages);
    long sections;
    loadpages(fp, pageloc, numpages, sections, allpages);

    // process the pages as specified
    if (pages.size() < 1)
	dumpall(fp, allpages);
    else
	dumppages(fp, pages, allpages, sections);

    mesg(NULL);
}
