/*
  usage: hzview [-s <spec file>] file
  
  Version 1.0
  
  Copyright (C) 1989  Fung Fung Lee (leeumunhum.stanford.edu)
  
  A program to let you view a HZ file on a tty screen.
  HZ is a 7-bit data format for mixed Chinese (GB) and ASCII text.
  
  The font files and spec files should all be under the directory
  HZDIR (setenv HZDIR the-hanzi-directory).
  
  This program is free for general distribution.  

  This program runs on UNIX. You are welcome to port it to other operating
  systems.

*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#define true 1
#define false 0

char *getenv();

extern char *HZbitmap();
extern loadHZfont();
int hzbyte;	/* number of bytes per hanzi image */

char hzdir[80], libdir[100], fontdir[100];
char pmode;
float leftmargin, rightmargin, topmargin, bottommargin;
float hpos, vpos;
int HZbm;
float HZpoint, HZspacing, HZskip;
float Epoint, Espacing, Eskip, Evskip, Ekerning, Espace;
float vskip, hskip;

static char HZDIR[] = "HZDIR";
static char SPEC[] = "hzview.spec";
static char nullName[] = "";
char *progname, *inputName;
char headerName[100], specName[100];
char hzfont[30], hzName[30];
char engfont[30];
FILE *spec, *fin;

#define MAXROW 24	/* maximum hanzi point size */
#define MAXCOL 132	/* maximum number of columns on terminal */
char screen[MAXROW][MAXCOL];
char cmark = '*';	/* default Chinese display marker */

#define DB(hi,lo)	(((hi)&0xFF) << 8 | (lo)&0xFF)
#define isGB(c)		((c)>=0x21 && (c)<=0x7E)


warning()
{
  fprintf(stderr, "usage: %s [-s <spec-file>] hz-file\n", progname); 
  exit(1);
}


#define LEN 100

main(argc, argv)
     int argc;
     char *argv[];
{
  char s[LEN];
  
  if (getenv(HZDIR) != NULL)
    {
      strcpy(hzdir, getenv(HZDIR));
      strcpy(libdir, hzdir);
      strcat(libdir, "/lib/");
      strcpy(fontdir, hzdir);
      strcat(fontdir, "/font/");
    }
  else
    {
      fprintf(stderr, "Please setenv %s appropriately.\n", HZDIR);
      exit(1);
    }
#if 0
  strcpy(headerName, libdir);
  strcat(headerName, HEADER);
#endif
  strcpy(specName, libdir);
  strcat(specName, SPEC);
  inputName = nullName;
  
  progname = argv[0];
  switch (argc)
    {
    case 1: break;
    case 2: inputName = argv[1]; break;
    case 4: inputName = argv[3];
    case 3: 
      if (strcmp(argv[1], "-s") != 0) warning();
      strcpy(specName, libdir);
      strcat(specName, argv[2]);
      break;
    default: warning(); break;
    }
  
  if ((spec = fopen(specName, "r")) == 0)
    {
      fprintf(stderr, "Cannot open spec file %s\n", specName);
      exit(1);
    }
  
  fgets(s, LEN, spec);
  sscanf(s, "%c", &pmode);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &leftmargin);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &rightmargin);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &topmargin);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &bottommargin);
  fgets(s, LEN, spec);
  sscanf(s, "%s %d", hzName, &HZbm);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &HZpoint);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &HZspacing);
  HZskip = HZpoint + HZspacing;
  fgets(s, LEN, spec);
  sscanf(s, "%f", &vskip);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &hskip);
  fgets(s, LEN, spec);
  sscanf(s, "%s", engfont);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Epoint);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Eskip);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Espace);
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Espacing);
  Evskip = Epoint + Espacing;
  fgets(s, LEN, spec);
  sscanf(s, "%f", &Ekerning);
  
  sprintf(s, "%d", HZbm);
  strcpy(hzfont, hzName);
  strcat(hzfont, s);
  
  hzbyte = HZbm * HZbm / 8;
  loadHZfont();
  
  if (strcmp(inputName, nullName) == 0)
    fin = stdin;
  else if ((fin = fopen(inputName, "r")) == 0)
    {
      fprintf(stderr, "Cannot open input file %s\n", inputName);
      exit(1);
    }
  ttyInit();
  if (pmode == 'h') 
    {     
      hpos = leftmargin;
      vpos = topmargin;
/*    fputNewLine(stdout); */
      scanner(fin, stdout);
    }  
  else
    {
      fprintf(stderr, "unknown printing mode %c\n", pmode);
      exit(1);
    }
}


GBrange(hi, lo)
int hi, lo;
{
  if (!isGB(hi) || !isGB(lo))
    {
      fprintf(stderr, "GB code out of range: %c%c (hex: %2x%2x)\n",
	      hi, lo, hi, lo);
      exit(1);
    }
}

#define SLEN 80		/* maximum English string length */

scanner(fin, fout)
     FILE *fin, *fout;
{
  int c1, c2, code, i;
  char *image, s[SLEN];
  int GBmode = false;
  
  while ((c1=fgetc(fin)) != EOF)
    {
      if (GBmode)
	{
	  c2 = fgetc(fin);
	  GBrange(c1, c2);
	  code = DB(c1, c2);
	  switch (code)
	    {
	    case 0x7E7D: /* DB('~','}') */
	      GBmode = false; ttyShow(fout); break; /* for hzview only */
#if 0
	    case 0x212A: fputDash(0.5, fout); break;
	    case 0x235F: fputDash(0.0, fout); break;
	    case 0x237E: fputDash(1.0, fout); break;
#endif
	    default:
	      image = HZbitmap(code);
	      fputHanzi(image, fout);
	      break;
	    }
	}
      /* not in GBmode */
      else if (c1 == '~')
	{
	  c2 = fgetc(fin);
	  switch (c2)
	    {
	    case '~': fputAscii('~', fout); break;
	    case '{': GBmode = true;
	      fputNewLine(fout);  /* for hzview only */
	      break;
	    case '\n': break;	/* line-continuation marker */
	    default:
	      fprintf(stderr, "unexpected escape sequence: ~%c\n", c2);
	      /* optional error recovery: */
	      fputAscii('~', fout); ungetc(c2, fin);
	      break;
	    }
	}
      else if (isalnum(c1))
	{
	  s[0] = c1;
	  i = 1;
	  while (i<SLEN-1 && (c1=fgetc(fin)) && isalnum(c1))
	    s[i++] = c1;
	  s[i++] = '\0';
	  if (i<SLEN)
	    ungetc(c1, fin);
	  fputWord(s, fout);
	}
      else
	  fputAscii(c1, fout);
    }
#if 0
  fprintf(fout, "printpage\n");
#endif
}

fputNewPage(f)
     FILE *f;
{
#if 0
  fprintf(f, "printpage\n");
  hpos = leftmargin;
  vpos = topmargin;
  fputNewLine(f);
#endif
}

fputNewLine(f)
     FILE *f;
{
  fprintf(f, "\n");
  hpos = leftmargin;
#if 0
  vpos -= vskip;
  if (vpos < bottommargin)
    fputNewPage(f);
  else
    fprintf(f, "%6.2f %6.2f moveto\n", hpos, vpos);
#endif
}

fputHanzi(image, f)
     char *image;	/* image points to hzbyte bytes of bitmap */
     FILE *f;
{
  int row, col, n=0, w;
  char *p = image;

  if (hpos + HZskip > rightmargin)
      ttyShow(f);
  for (row = 0; row < HZbm; row++)
    {
      for (col = hpos; col < hpos + HZbm; col++)
	{
	  if (n==0) {w = *p++; w &= 0xFF; n = 8;}
	  screen[row][col] = (w & 0x80) ? cmark : ' ';
	  w <<= 1; n--;
	}
    }
  hpos += HZskip;
}

ttyInit()
{
  int row, col, k;

  for (row = 0; row < MAXROW; row++)
    for (col = 0; col < MAXCOL; col++)
      screen[row][col] = ' ';
}

ttyShow(f)
FILE *f;
{
  int row, col, k;

  if (hpos == leftmargin) return;
  for (row = 0; row < HZbm; row++)
    {
      for (col = 0; col < hpos; col++)
	fprintf(f, "%c", screen[row][col]);
      fprintf(f, "\n");
    }	
  fputNewLine(f);
}

fputWord(s, f)
     char *s;
     FILE *f;
{
  if (hpos + strlen(s) > rightmargin)
    fputNewLine(f);
  fprintf(f, "%s", s);
  hpos += strlen(s);
}

fputAscii(c, f)
     char c;
     FILE *f;
{
  if (c=='\n' || c=='\r') 
    fputNewLine(f);
  else if (c=='\f')
    fprintf(f, "\n\n\n\n");
  else
    fputc(c, f);
#if 0
  if (c=='\n' || c=='\r')
    {
      fputNewLine(f);
      return;
    }
  if (hpos + Eskip > rightmargin)
    fputNewLine(f);
  if (c==' ' || c=='\t')
    {
      fprintf(f, "( ) Eshow\n");
      hpos += Eskip;
    }
  else if (c=='(' ||  c==')')
    {
      fprintf(f,  "(\\%c) Eshow\n", c);
      hpos += Eskip;
    }
  else if (c=='\\')
    {
      fprintf(f,  "(\\\\) Eshow\n");
      hpos += Eskip;
    }
  else if (isgraph(c))
    {
      fprintf(f, "(%c) Eshow\n", c);
      hpos += Eskip;
    }
  /* ignore other unprintable ascii characters */	
#endif
}

#if 0
fputDash(scale, f)
     float scale;
     FILE *f;
{
  if (hpos + HZskip > rightmargin)
    fputNewLine(f);
  fprintf(f, "%3f dash\n", scale);
  hpos += HZskip;
}
#endif
