/* Functions for W32 as window system.
   Copyright (C) 1989, 1992, 1993, 1994 Free Software Foundation.

This file is part of GNU Emacs.

GNU Emacs is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by Kevin Gallo */
/* Modified by H.Miyashita (himi@bird.scphys.kyoto-u.ac.jp)*/


#include <signal.h>
#include <config.h>
#include <stdio.h>

#include "lisp.h"
#include "frame.h"

#include "mw32font.h"
#include "mw32term.h"
#include "mw32reg.h"
#include "window.h"
#include "buffer.h"
#include "dispextern.h"
#include "keyboard.h"
#include "blockinput.h"
#ifdef MEADOW
#include "mw32sync.h"
#endif
#include <shellapi.h>

#include "coding.h"

extern void abort ();

extern void free_frame_menubar ();

extern HINSTANCE hinst;
extern HINSTANCE hprevinst;
extern LPSTR lpCmdLine;
extern int nCmdShow;
    
#define EMACS_RESOURCE_CLASS "Emacs"

/* Title name and application name for X stuff. */
extern char *w32_id_name;

/* The background and shape of the mouse pointer, and shape when not
   over text or in the modeline. */
Lisp_Object Vw32_pointer_shape, Vw32_nontext_pointer_shape, Vw32_mode_pointer_shape;

/* The shape when over mouse-sensitive text.  */
Lisp_Object Vw32_sensitive_text_pointer_shape;

/* Color of chars displayed in cursor box. */
Lisp_Object Vw32_cursor_fore_pixel;

/* The colormap for converting color names to RGB values */
Lisp_Object Vw32_color_map;

/* The name we're using in resource queries.  */
Lisp_Object Vx_resource_name;

/* Height of this X screen in pixels. */
int w32_screen_height;

/* Width of this X screen in pixels.  */
int w32_screen_width;

/* Height of this X screen in inches.  */
int w32_screen_height_in;

/* Width of this X screen in inches.  */
int w32_screen_width_in;

/* Number of planes for this screen.  */
int w32_screen_planes;

/* `t' if a mouse button is depressed.  */
Lisp_Object Vmouse_depressed;

/* W32 Psudo(?) terminal connection flag.  */
int w32gui_open = 0;

/* The last 23 bits of the timestamp of the last mouse button event. */
Time mouse_timestamp;

/* Evaluate this expression to rebuild the section of syms_of_xfns
   that initializes and staticpros the symbols declared below.  Note
   that Emacs 18 has a bug that keeps C-x C-e from being able to
   evaluate this expression.

(progn
  ;; Accumulate a list of the symbols we want to initialize from the
  ;; declarations at the top of the file.
  (goto-char (point-min))
  (search-forward "/\*&&& symbols declared here &&&*\/\n")
  (let (symbol-list)
    (while (looking-at "Lisp_Object \\(Q[a-z_]+\\)")
      (setq symbol-list
	    (cons (buffer-substring (match-beginning 1) (match-end 1))
		  symbol-list))
      (forward-line 1))
    (setq symbol-list (nreverse symbol-list))
    ;; Delete the section of syms_of_... where we initialize the symbols.
    (search-forward "\n  /\*&&& init symbols here &&&*\/\n")
    (let ((start (point)))
      (while (looking-at "^  Q")
	(forward-line 2))
      (kill-region start (point)))
    ;; Write a new symbol initialization section.
    (while symbol-list
      (insert (format "  %s = intern (\"" (car symbol-list)))
      (let ((start (point)))
	(insert (substring (car symbol-list) 1))
	(subst-char-in-region start (point) ?_ ?-))
      (insert (format "\");\n  staticpro (&%s);\n" (car symbol-list)))
      (setq symbol-list (cdr symbol-list)))))

  */        

/*&&& symbols declared here &&&*/
Lisp_Object Qauto_raise;
Lisp_Object Qauto_lower;
Lisp_Object Qbackground_color;
Lisp_Object Qbar, Qcaret, Qcheckered_caret, Qhairline_caret;
Lisp_Object Qborder_color;
Lisp_Object Qborder_width;
Lisp_Object Qbox;
Lisp_Object Qcursor_color;
Lisp_Object Qcursor_type;
Lisp_Object Qcursor_height;
Lisp_Object Qfont;
Lisp_Object Qforeground_color;
Lisp_Object Qgeometry;
Lisp_Object Qicon_left;
Lisp_Object Qicon_top;
Lisp_Object Qicon_type;
Lisp_Object Qinternal_border_width;
Lisp_Object Qleft;
Lisp_Object Qright;
Lisp_Object Qmouse_color;
Lisp_Object Qnone;
Lisp_Object Qparent_id;
Lisp_Object Qsuppress_icon;
Lisp_Object Qtop;
Lisp_Object Qundefined_color;
Lisp_Object Qvertical_scroll_bars;
Lisp_Object Qvisibility;
Lisp_Object Qwindow_id;
Lisp_Object Qx_frame_parameter;
Lisp_Object Qx_resource_name;
Lisp_Object Quser_position;
Lisp_Object Quser_size;
#ifdef MULE
Lisp_Object Qline_space;
#endif  /* MULE */
#ifdef MEADOW
Lisp_Object Qime_font;
#endif

Lisp_Object Qfontset;
Lisp_Object Qfont_property;

/* The below are defined in frame.c. */
extern Lisp_Object Qheight, Qminibuffer, Qname, Qonly, Qwidth;
extern Lisp_Object Qunsplittable, Qmenu_bar_lines;

extern Lisp_Object Vwindow_system_version;

void
check_w32gui ()
{
  if (! w32gui_open)
    error ("Windows have not been initialized yet.");
}

int
have_menus_p ()
{
  return w32gui_open;
}

/* Extract a frame as a FRAME_PTR, defaulting to the selected frame
   and checking validity for W32.  */

FRAME_PTR
check_x_frame (frame)
     Lisp_Object frame;
{
  FRAME_PTR f;

  if (NILP (frame))
    f = selected_frame;
  else
    {
      CHECK_LIVE_FRAME (frame, 0);
      f = XFRAME (frame);
    }
  if (! FRAME_W32_P (f))
    error ("non-w32 frame used");
  return f;
}

/* The default colors for the color map */
typedef struct ColorMap_t {
    char * name ;
    COLORREF colorref ;
} ColorMap_t ;

ColorMap_t w32_color_map[] = {
#include "mw32rgb.h"
} ;

DEFUN ("w32-default-color-map", Fw32_default_color_map, Sw32_default_color_map,
       0, 0, 0, "Return the default color map.")
  ()
{
    int i ;
    ColorMap_t * pc = w32_color_map ;
    Lisp_Object cmap ;

    BLOCK_INPUT;

    cmap = Qnil ;

    for (i = 0; i < sizeof (w32_color_map) / sizeof (w32_color_map[0]); pc++,i++)
	cmap = Fcons(Fcons (build_string(pc->name),
			    make_number (pc->colorref)),
		     cmap) ;

    UNBLOCK_INPUT ;

    return (cmap) ;
}

Lisp_Object w32_to_x_color(rgb)
    Lisp_Object rgb ;
{
    Lisp_Object color ;

    CHECK_NUMBER (rgb, 0);

    BLOCK_INPUT;

    color = Frassq(rgb,Vw32_color_map) ;

    UNBLOCK_INPUT ;

    if (!NILP(color))
	return (Fcar(color)) ;
    else
	return Qnil ;
}

static
int get_hex1(int numstr)
{
  if (isalpha(numstr))
    {
      numstr = toupper(numstr);
      if (!(numstr >= 'A' && numstr <= 'F'))
	return -1;
      numstr -= ('A' - 10);
      return numstr;
    }
  if (isdigit(numstr))
    {
      numstr -= '0';
      return numstr;
    }
  return -1;
}

static
int color_radix_change(TCHAR *colstr, int size)
{
  int i;
  int ret, tmp;

  ret = 0;
  if ((size <= 0) || (size > 4)) return -1;
  for(i = 0;i < size;i++)
    {
      ret <<= 4;
      tmp = get_hex1(*colstr);
      if (tmp == -1) 
	return -1;
      ret += tmp;
      colstr++;
    }
  return ret;
}

static
Lisp_Object x_old_rgb_names(TCHAR *colstr, int len)
{
  Lisp_Object cdef;
  int red, green, blue;

  switch(len)
    {
    case 4:			/* #RGB */
      red = color_radix_change(colstr+1, 1) << 4;
      green = color_radix_change(colstr+2, 1) << 4;
      blue = color_radix_change(colstr+3, 1) << 4;
      break;

    case 7:			/* #RRGGBB */
      red = color_radix_change(colstr+1, 2);
      green = color_radix_change(colstr+3, 2);
      blue = color_radix_change(colstr+5, 2);
      break;
    case 10:			/* #RRRGGGBBB */
      red = color_radix_change(colstr+1, 3) >> 4;
      green = color_radix_change(colstr+4, 3) >> 4;
      blue = color_radix_change(colstr+7, 3) >> 4;
      break;
    case 13:			/* #RRRRGGGGBBBB */
      red = color_radix_change(colstr+1, 4) >> 8;
      green = color_radix_change(colstr+5, 4) >> 8;
      blue = color_radix_change(colstr+9, 4) >> 8;
      break;

    default :
      return Qnil;
    }
  if ((red == -1) || (green == -1) || (blue == -1))
    return Qnil;

  return RGB(red, green, blue);
}

static
Lisp_Object x_rgb_names(TCHAR *colstr, int len)
{
  TCHAR *colstrend;
  Lisp_Object ret;
  int red, green, blue;
  int redchars, greenchars, bluechars;

  colstr += 4;
  len -= 4;
  colstrend = memchr(colstr, '/', len);
  if (!colstrend)
    return Qnil;
  redchars = (int) (colstrend - colstr);
  red = color_radix_change(colstr, redchars);
  if (red == -1) return Qnil;

  len -= (int)(colstrend - colstr + 1);
  colstr = colstrend + 1;
  colstrend = memchr(colstr, '/', len);
  if (!colstrend)
    return Qnil;
  greenchars = (int) (colstrend - colstr);
  green = color_radix_change(colstr, greenchars);
  if (green == -1) return Qnil;

  len -= (int)(colstrend - colstr + 1);
  colstr = colstrend + 1;
  bluechars = len;
  blue = color_radix_change(colstr, bluechars);
  if (blue == -1) return Qnil;

  /* 
     We rescale color values in 4bit * <number of chars>,
     then normalize them in 8bit.
  */
  if (redchars > 2)
    red >>= -(8 - redchars * 4);
  else if (redchars < 2)
    red <<= (8 - redchars * 4);
  if (greenchars > 2)
    green >>= -(8 - greenchars * 4);
  else if (greenchars < 2)
    green <<= (8 - greenchars * 4);
  if (bluechars > 2)
    blue >>= -(8 - bluechars * 4);
  else if (bluechars < 2)
    blue <<= (8 - bluechars * 4);

  XSETINT(ret, RGB(red, green, blue));
  return ret;
}

Lisp_Object x_to_w32_color(colorname)
    Lisp_Object colorname ;
{
    TCHAR *colstr;
    int colstrlen;
    register Lisp_Object tail,ret = Qnil;

    CHECK_STRING (colorname,0);

    BLOCK_INPUT;

    colstr = XSTRING(colorname)->data;
    colstrlen = LISPY_STRING_BYTES(colorname);
    if (*colstr == '#')
      {
	return x_old_rgb_names(colstr, colstrlen);
      }
    
    if (colstrlen > 4) {
      if (memcmp(colstr, "rgb:", 4) == 0)
	{
	  return x_rgb_names(colstr, colstrlen);
	}
#if 0
      if (memcmp(colstr, "rgbi:", 5) == 0)
	{
	  return x_rgbi_names(arg, colstr, colstrlen);
	}
#endif
    }
    

    for (tail = Vw32_color_map; !NILP (tail); tail = Fcdr (tail))
      {
	register Lisp_Object elt, tem;

	elt = Fcar (tail);
	if (!CONSP (elt)) continue;

	tem = Fcar (elt);

/* 96.10.12 modified by himi */
	if ((XSTRING(colorname) -> size) == (XSTRING(tem) -> size)){
	  if (memicmp(XSTRING(colorname) -> data, XSTRING(tem) -> data,
		      XSTRING(colorname) -> size) == 0) {
	    ret = Fcdr (elt);
	    break;
	  }
	}

	QUIT ;
      }

    UNBLOCK_INPUT ;

    return ret ;
}

/* Return the Emacs frame-object corresponding to an X window.
   It could be the frame's main window or an icon window.  */

/* This function can be called during GC, so use XGCTYPE.  */

struct frame *
w32_window_to_frame (wdesc)
    HWND wdesc;
{
  Lisp_Object tail, frame;
  struct frame *f;

  if (wdesc == 0) return 0;

  for (tail = Vframe_list; CONSP (tail);
       tail = XCONS (tail)->cdr)
    {
      frame = XCONS (tail)->car;
      if (!FRAMEP(frame))
        continue;
      f = XFRAME (frame);
      if (f->output_data.nothing == 1) 
	return 0;
      if (FRAME_W32_WINDOW (f) == wdesc
          || f->output_data.w32->icon_desc == wdesc)
        return f;
    }
  return 0;
}

/* Connect the frame-parameter names for X frames
   to the ways of passing the parameter values to the window system.

   The name of a parameter, as a Lisp symbol,
   has an `x-frame-parameter' property which is an integer in Lisp
   but can be interpreted as an `enum w32_frame_parm' in C.  */

enum w32_frame_parm
{
  W32_PARM_FOREGROUND_COLOR,
  W32_PARM_BACKGROUND_COLOR,
  W32_PARM_MOUSE_COLOR,
  W32_PARM_CURSOR_COLOR,
  W32_PARM_BORDER_COLOR,
  W32_PARM_ICON_TYPE,
  W32_PARM_FONT,
#ifdef MW32_FONT
  W32_PARM_FONTSET,
  W32_PARM_FONT_PROPERTY,
#endif
#ifdef MULE
  W32_PARM_LINE_SPACE,
#endif  /* MULE */
  W32_PARM_BORDER_WIDTH,
  W32_PARM_INTERNAL_BORDER_WIDTH,
  W32_PARM_NAME,
  W32_PARM_AUTORAISE,
  W32_PARM_AUTOLOWER,
  W32_PARM_VERT_SCROLL_BAR,
  W32_PARM_VISIBILITY,
  W32_PARM_MENU_BAR_LINES
};


struct w32_frame_parm_table
{
  char *name;
  void (*setter)( /* struct frame *frame, Lisp_Object val, oldval */ );
};

void w32_set_foreground_color ();
void w32_set_background_color ();
void w32_set_mouse_color ();
void w32_set_cursor_color ();
void w32_set_border_color ();
void w32_set_cursor_type ();
void w32_set_cursor_height ();
void w32_set_font ();
#ifdef MEADOW
void w32_set_frame_ime_font ();
#endif
#ifdef MULE
void w32_set_line_space ();
#endif  /* MULE */
void w32_set_border_width ();
void w32_set_internal_border_width ();
void x_explicitly_set_name ();
void w32_set_autoraise ();
void w32_set_autolower ();
void w32_set_vertical_scroll_bars ();
void w32_set_visibility ();
void x_set_menu_bar_lines ();
void w32_set_scroll_bar_width ();
void x_set_unsplittable ();

static struct w32_frame_parm_table w32_frame_parms[] =
{
  "foreground-color", w32_set_foreground_color,
  "background-color", w32_set_background_color,
  "mouse-color", w32_set_mouse_color,
  "cursor-color", w32_set_cursor_color,
  "border-color", w32_set_border_color,
  "cursor-type", w32_set_cursor_type,
  "cursor-height", w32_set_cursor_height,
  "font", w32_set_font,
#ifdef IME_CONTROL
  "ime-font", w32_set_frame_ime_font,
#endif
#ifdef MULE
  "line-space", w32_set_line_space,
#endif  /* MULE */
  "border-width", w32_set_border_width,
  "internal-border-width", w32_set_internal_border_width,
  "name", x_explicitly_set_name,
  "auto-raise", w32_set_autoraise,
  "auto-lower", w32_set_autolower,
  "vertical-scroll-bars", w32_set_vertical_scroll_bars,
  "visibility", w32_set_visibility,
  "menu-bar-lines", x_set_menu_bar_lines,
  "scroll-bar-width", w32_set_scroll_bar_width,
  "unsplittable", x_set_unsplittable,
};

/* Attach the `x-frame-parameter' properties to
   the Lisp symbol names of parameters relevant to X.  */

init_w32_parm_symbols ()
{
  int i;

  for (i = 0; i < sizeof (w32_frame_parms) / sizeof (w32_frame_parms[0]); i++)
    Fput (intern (w32_frame_parms[i].name), Qx_frame_parameter,
	  make_number (i));
}

/* Change the parameters of FRAME as specified by ALIST.
   If a parameter is not specially recognized, do nothing;
   otherwise call the `w32_set_...' function for that parameter.  */
/* x_set_... is a temporal action to adapt the common Emacs modules.  */
/* to correspond to other emacs module....*/
void
x_set_frame_parameters (f, alist)
     FRAME_PTR f;
     Lisp_Object alist;
{
  Lisp_Object tail;

  /* If both of these parameters are present, it's more efficient to
     set them both at once.  So we wait until we've looked at the
     entire list before we set them.  */
  Lisp_Object width, height;

  /* Same here.  */
  Lisp_Object left, top;

  /* Record in these vectors all the parms specified.  */
  Lisp_Object *parms;
  Lisp_Object *values;
  int i;
  
  i = 0;
  for (tail = alist; CONSP (tail); tail = Fcdr (tail))
    i++;

  parms = (Lisp_Object *) alloca (i * sizeof (Lisp_Object));
  values = (Lisp_Object *) alloca (i * sizeof (Lisp_Object));

  /* Extract parm names and values into those vectors.  */

  i = 0;
  for (tail = alist; CONSP (tail); tail = Fcdr (tail))
    {
      Lisp_Object elt, prop, val;

      elt = Fcar (tail);
      parms[i] = Fcar (elt);
      values[i] = Fcdr (elt);
      i++;
    }

  width = height = top = left = Qunbound;

  /* Now process them in reverse of specified order.  */
  for (i--; i >= 0; i--)
    {
      Lisp_Object prop, val;

      prop = parms[i];
      val = values[i];

      if (EQ (prop, Qwidth))
	width = val;
      else if (EQ (prop, Qheight))
	height = val;
      else if (EQ (prop, Qtop))
	top = val;
      else if (EQ (prop, Qleft))
	left = val;
      else
	{
	  register Lisp_Object param_index, old_value;

	  param_index = Fget (prop, Qx_frame_parameter);
	  old_value = get_frame_param (f, prop);
	  store_frame_param (f, prop, val);
	  if (INTEGERP(param_index)
	      && XINT (param_index) >= 0
	      && (XINT (param_index)
		  < sizeof (w32_frame_parms)/sizeof (w32_frame_parms[0])))
	    (*w32_frame_parms[XINT (param_index)].setter)(f, val, old_value);
	}
    }

  /* Don't die if just one of these was set.  */
  if (EQ (left, Qunbound))
    XSETINT (left, f->output_data.w32->left_pos);
  if (EQ (top, Qunbound))
    XSETINT (top, f->output_data.w32->top_pos);

  /* Don't die if just one of these was set.  */
  if (EQ (width, Qunbound))
    XSETINT (width, FRAME_WIDTH (f));
  if (EQ (height, Qunbound))
    XSETINT (height, FRAME_HEIGHT (f));

  /* Don't set these parameters these unless they've been explicitly
     specified.  The window might be mapped or resized while we're in
     this function, and we don't want to override that unless the lisp
     code has asked for it.

     Don't set these parameters unless they actually differ from the
     window's current parameters; the window may not actually exist
     yet.  */
  {
    Lisp_Object frame;

    check_frame_size (f, &height, &width);

    XSETFRAME (frame, f);

    if ((NUMBERP (width) && XINT (width) != FRAME_WIDTH (f))
	|| (NUMBERP (height) && XINT (height) != FRAME_HEIGHT (f)))
      Fset_frame_size (frame, width, height);

    if ((!NILP (left) || !NILP (top))
	&& ! (NUMBERP (left) && XINT (left) == f->output_data.w32->left_pos
	      && NUMBERP (top) && XINT (top) == f->output_data.w32->top_pos))
      {
	int leftpos = (NUMBERP (left) ? XINT (left) : 0);
	int toppos = (NUMBERP (top) ? XINT (top) : 0);

	/* Store the numeric value of the position.  */
	f->output_data.w32->top_pos = toppos;
	f->output_data.w32->left_pos = leftpos;

	/* Record the signs.  */
#if 0
	f->output_data.w32->size_hint_flags &= ~ (XNegative | YNegative);
	if (EQ (left, Qminus) || (NUMBERP (left) && XINT (left) < 0))
	  f->output_data.w32->size_hint_flags |= XNegative;
	if (EQ (top, Qminus) || (NUMBERP (top) && XINT (top) < 0))
	  f->output_data.w32->size_hint_flags |= YNegative;
	f->output_data.w32->win_gravity = NorthWestGravity;
#endif

	/* Actually set that position, and convert to absolute.  */
	x_set_offset (f, leftpos, toppos, 0);
      }
  }
}

#ifndef MW32_FONT
/* The font conversion stuff between x and w32 */

/* X font string is as follows (from faces.el)
 * (let ((- 		"[-?]")
 *      (foundry	"[^-]+")
 *      (family 	"[^-]+")
 *      (weight		"\\(bold\\|demibold\\|medium\\)")		; 1
 *      (weight\?	"\\([^-]*\\)")					; 1
 *      (slant		"\\([ior]\\)")					; 2
 *      (slant\?	"\\([^-]?\\)")					; 2
 *      (swidth		"\\([^-]*\\)")					; 3
 *      (adstyle	"[^-]*")					; 4
 *      (pixelsize	"[0-9]+")
 *      (pointsize	"[0-9][0-9]+")
 *      (resx		"[0-9][0-9]+")
 *      (resy		"[0-9][0-9]+")
 *      (spacing	"[cmp?*]")
 *      (avgwidth	"[0-9]+")
 *      (registry	"[^-]+")
 *      (encoding	"[^-]+")
 *      )
 *  (setq x-font-regexp
 *	(concat "\\`\\*?[-?*]"
 *		foundry - family - weight\? - slant\? - swidth - adstyle -
 *		pixelsize - pointsize - resx - resy - spacing - registry -
 *		encoding "[-?*]\\*?\\'"
 *		))
 *  (setq x-font-regexp-head
 *	(concat "\\`[-?*]" foundry - family - weight\? - slant\?
 *		"\\([-*?]\\|\\'\\)"))
 *  (setq x-font-regexp-slant (concat - slant -))
 *  (setq x-font-regexp-weight (concat - weight -))
 * nil)	    
 */
    
#define FONT_START       "[-?]"
#define FONT_FOUNDRY     "[^-]+"
#define FONT_FAMILY      "\\([^-]+\\)"                      /* 1 */
#define FONT_WEIGHT      "\\(bold\\|demibold\\|medium\\)"   /* 2 */
#define FONT_WEIGHT_Q    "\\([^-]*\\)"                      /* 2 */
#define FONT_SLANT       "\\([ior]\\)"                      /* 3 */
#define FONT_SLANT_Q     "\\([^-]?\\)"                      /* 3 */
#define FONT_SWIDTH      "\\([^-]*\\)"                      /* 4 */
#define FONT_ADSTYLE     "[^-]*"
#define FONT_PIXELSIZE   "[^-]*"
#define FONT_POINTSIZE   "\\([0-9][0-9]+\\|\\*\\)"          /* 5 */
#define FONT_RESX        "[0-9][0-9]+"
#define FONT_RESY        "[0-9][0-9]+"
#define FONT_SPACING     "[cmp?*]"
#define FONT_AVGWIDTH    "[0-9]+"
#define FONT_REGISTRY    "[^-]+"
#define FONT_ENCODING    "[^-]+"

#define FONT_REGEXP      ("\\`\\*?[-?*]"     \
			  FONT_FOUNDRY   "-" \
			  FONT_FAMILY    "-" \
			  FONT_WEIGHT_Q  "-" \
			  FONT_SLANT_Q   "-" \
			  FONT_SWIDTH    "-" \
			  FONT_ADSTYLE   "-" \
			  FONT_PIXELSIZE "-" \
			  FONT_POINTSIZE "-" \
			  "[-?*]\\|\\'")

#define FONT_REGEXP_HEAD ("\\`[-?*]"        \
			  FONT_FOUNDRY  "-" \
			  FONT_FAMILY   "-" \
			  FONT_WEIGHT_Q "-" \
			  FONT_SLANT_Q      \
			  "\\([-*?]\\|\\'\\)")

#define FONT_REGEXP_SLANT  "-" FONT_SLANT  "-"
#define FONT_REGEXP_WEIGHT "-" FONT_WEIGHT "-"

LONG x_to_w32_weight (lpw)
    char * lpw ;
{
    if (!lpw) return (FW_DONTCARE) ;

    if (stricmp (lpw,"bold") == 0)
	return (FW_BOLD) ;
    else if (stricmp (lpw,"demibold") == 0)
	return (FW_SEMIBOLD) ;
    else if (stricmp (lpw,"medium") == 0)
	return (FW_MEDIUM) ;
    else if (stricmp (lpw,"normal") == 0)
	return (FW_NORMAL) ;
    else
	return (FW_DONTCARE) ;
}

char * w32_to_x_weight (fnweight)
    int fnweight ;
{
    if (fnweight >= FW_BOLD) 
	return ("bold") ;
    else if (fnweight >= FW_SEMIBOLD) 
	return ("demibold") ;
    else if (fnweight >= FW_MEDIUM) 
	return ("medium") ;
    else  
	return ("normal") ;
}

BOOL w32_to_x_font(lplogfont,lpxstr,len)
    LOGFONT * lplogfont ;
    char * lpxstr ;
    int len ;
{
    if (!lpxstr) return (FALSE) ;

    if (lplogfont) {
	_snprintf (lpxstr,len - 1,
		   "-*-%s-%s-%c-%s-%s-*-%d-*-*-%c-%d-*-*-",
		   lplogfont->lfFaceName,
		   w32_to_x_weight(lplogfont->lfWeight),
		   lplogfont->lfItalic?'i':'r',
		   "*","*",
		   (lplogfont->lfHeight * 1440) / w32_screen_height_in,
		   ((lplogfont->lfPitchAndFamily & 0x3) == VARIABLE_PITCH)?'p':'c',
		   (lplogfont->lfWidth * 1440) / w32_screen_width_in) ;
    } else {
	strncpy (lpxstr,"-*-*-*-*-*-*-*-*-*-*-*-*-*-*-",len - 1) ;
    }

    lpxstr[len - 1] = 0 ; /* just to be sure */
    return (TRUE) ;
}

BOOL x_to_w32_font(lpxstr,lplogfont)
    char * lpxstr ;
    LOGFONT * lplogfont ;
{
    if (!lplogfont) return (FALSE) ;

    memset(lplogfont,0,sizeof(*lplogfont)) ;

#ifdef MULE
    lplogfont->lfCharSet = SHIFTJIS_CHARSET;
#else
    lplogfont->lfCharSet = OEM_CHARSET ;
#endif
    lplogfont->lfOutPrecision = OUT_DEFAULT_PRECIS ;
    lplogfont->lfClipPrecision = CLIP_DEFAULT_PRECIS ;
    lplogfont->lfQuality = DEFAULT_QUALITY ;
    
    if (lpxstr && *lpxstr == '-') lpxstr++ ;

    {
	int fields ;
	char name[50], weight[20], slant, pitch , height[10], width[10] ;
	
	fields = (lpxstr
		  ?sscanf(lpxstr,
			  "%*[^-]-%[^-]-%[^-]-%c-%*[^-]-%*[^-]-%*[^-]-%[^-]-%*[^-]-%*[^-]-%c-%[^-]",
			  name,weight,&slant,height,&pitch,width)
		  :0) ;

	if (fields == EOF) return (FALSE) ;

	if (fields > 0 && name[0] != '*') {
	    strncpy (lplogfont->lfFaceName,name,LF_FACESIZE) ;
	} else {
	    lplogfont->lfFaceName[0] = 0 ;
	}

	fields-- ;

	lplogfont->lfWeight = 0; /* x_to_w32_weight((fields > 0?weight:"")); */

	fields--;

	lplogfont->lfItalic = (fields > 0 && slant == 'i');

	fields--;

	if (fields > 0 && height[0] != '*')
	    lplogfont->lfHeight = (atoi(height) * w32_screen_height_in) / 1440;

	fields--;

	lplogfont->lfPitchAndFamily = (fields > 0 && pitch == 'p')?VARIABLE_PITCH:FIXED_PITCH;

	fields--;

	if (fields > 0 && width[0] != '*')
	    lplogfont->lfWidth = (atoi(width) * w32_screen_width_in) / 1440;
    }

    return TRUE;
}

BOOL w32_font_match (lpszfont1,lpszfont2)
    char * lpszfont1;
    char * lpszfont2;
{
    char * s1 = lpszfont1, *e1;
    char * s2 = lpszfont2, *e2;

    if (s1 == NULL || s2 == NULL) return FALSE;

    if (*s1 == '-') s1++;
    if (*s2 == '-') s2++;

    while (1) {
	int len1, len2;

	e1 = strchr (s1,'-');
	e2 = strchr (s2,'-');

	if (e1 == NULL || e2 == NULL) return TRUE;

	len1 = e1 - s1;
	len2 = e2 - s2;

	if (*s1 != '*' && *s2 != '*' && 
	    (len1 != len2 || strnicmp(s1,s2,len1) != 0))
	    return FALSE;

	s1 = e1 + 1;
	s2 = e2 + 1;
    }
}
#endif /* not MW32_FONT */

/* Store the positions of frame F into XPTR and YPTR.
   These are the positions of the containing window manager window,
   not Emacs's own window.  */

void
w32_real_positions (f, xptr, yptr)
     FRAME_PTR f;
     int *xptr, *yptr;
{
  POINT pt;

  BLOCK_INPUT;

  {
      RECT rect;
      
      GetClientRect (FRAME_W32_WINDOW(f), &rect);
      AdjustWindowRectEx (&rect, f->output_data.w32->dwStyle, 
			  FRAME_EXTERNAL_MENU_BAR(f),
			  f->output_data.w32->dwStyleEx);
      
      pt.x = rect.left;
      pt.y = rect.top;
  }

  ClientToScreen (FRAME_W32_WINDOW(f), &pt);

  UNBLOCK_INPUT;

  *xptr = pt.x;
  *yptr = pt.y;
}

/* Insert a description of internally-recorded parameters of frame X
   into the parameter alist *ALISTPTR that is to be given to the user.
   Only parameters that are specific to the X window system
   and whose values are not correctly recorded in the frame's
   param_alist need to be considered here.  */
/* to correspond to other emacs module....*/
x_report_frame_params (f, alistptr)
     struct frame *f;
     Lisp_Object *alistptr;
{
  char buf[16];

  store_in_alist (alistptr, Qleft, make_number (f->output_data.w32->left_pos));
  store_in_alist (alistptr, Qtop, make_number (f->output_data.w32->top_pos));

  store_in_alist (alistptr, Qborder_width,
		  make_number (f->output_data.w32->border_width));
  store_in_alist (alistptr, Qinternal_border_width,
		  make_number (f->output_data.w32->internal_border_width));
  sprintf (buf, "%d", FRAME_W32_WINDOW (f));
  store_in_alist (alistptr, Qwindow_id,
		  build_string (buf));
  FRAME_SAMPLE_VISIBILITY (f);
  store_in_alist (alistptr, Qvisibility,
		  (FRAME_VISIBLE_P (f) ? Qt :
		   (FRAME_ICONIFIED_P (f) ? Qicon : Qnil)));
}

/* Decide if color named COLOR is valid for the display associated with
   the selected frame; if so, return the rgb values in COLOR_DEF.
   If ALLOC is nonzero, allocate a new colormap cell.  */

int
defined_color (color, color_def)
     Lisp_Object color;
     Color *color_def;
{
  register Lisp_Object tem;

  tem = x_to_w32_color (color);

  if (! NILP (tem)) {
    /* Modified by himi */
      *color_def = XUINT(tem);
      return 1;
  } else {
      return 0;
  }
}

/* Given a string ARG naming a color, compute a pixel value from it
   suitable for screen F.
   If F is not a color screen, return DEF (default) regardless of what
   ARG says.  */
/* 96.10.1 modified by himi */
int
w32_decode_color (arg, def)
     Lisp_Object arg;
     int def;
{
  Color cdef;
  TCHAR *colstr;

  CHECK_STRING (arg, 0);

  colstr = XSTRING(arg) -> data;

  if (strcmp (colstr, "black") == 0)
    return BLACK_PIX_DEFAULT;
  else if (strcmp (colstr, "white") == 0)
    return WHITE_PIX_DEFAULT;

  if (w32_screen_planes == 1)
    return def;

  if (defined_color (arg, &cdef, 1))
    return cdef;
  else
    Fsignal (Qundefined_color, Fcons (arg, Qnil));

  return 0;
}

/* Functions called only from `x_set_frame_param'
   to set individual parameters.

   If FRAME_W32_WINDOW (f) is 0,
   the frame is being created and its X-window does not exist yet.
   In that case, just record the parameter's new value
   in the standard place; do not attempt to change the window.  */

void
w32_set_foreground_color (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  f->output_data.w32->foreground_pixel = w32_decode_color (arg, BLACK_PIX_DEFAULT);
  if (FRAME_W32_WINDOW (f) != 0)
    {
      recompute_basic_faces (f);
      if (FRAME_VISIBLE_P (f))
        redraw_frame (f);
    }
}

void
w32_set_background_color (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  f->output_data.w32->background_pixel = w32_decode_color (arg, WHITE_PIX_DEFAULT);

  if (FRAME_W32_WINDOW (f) != 0)
    {
      recompute_basic_faces (f);

      if (FRAME_VISIBLE_P (f))
        redraw_frame (f);
    }
}

void
w32_set_mouse_color (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
#if 0
  Cursor cursor, nontext_cursor, mode_cursor, cross_cursor;
#endif
  int count;
  int mask_color;

  if (!EQ (Qnil, arg))
    f->output_data.w32->mouse_pixel = w32_decode_color (arg, BLACK_PIX_DEFAULT);
  mask_color = f->output_data.w32->background_pixel;
				/* No invisible pointers. */
  if (mask_color == f->output_data.w32->mouse_pixel
	&& mask_color == f->output_data.w32->background_pixel)
    f->output_data.w32->mouse_pixel = f->output_data.w32->foreground_pixel;

  BLOCK_INPUT;
#if 0

  /* It's not okay to crash if the user selects a screwy cursor.  */
  w32_catch_errors ();

  if (!EQ (Qnil, Vw32_pointer_shape))
    {
      CHECK_NUMBER (Vw32_pointer_shape, 0);
      cursor = XCreateFontCursor (x_current_display, XINT (Vw32_pointer_shape));
    }
  else
    cursor = XCreateFontCursor (dsp, XC_xterm);
  count = w32_check_errors ("bad text pointer cursor: %s");

  if (!EQ (Qnil, Vw32_nontext_pointer_shape))
    {
      CHECK_NUMBER (Vw32_nontext_pointer_shape, 0);
      nontext_cursor = XCreateFontCursor (dsp,
					  XINT (Vw32_nontext_pointer_shape));
    }
  else
    nontext_cursor = XCreateFontCursor (dsp, XC_left_ptr);
  w32_check_errors ("bad nontext pointer cursor: %s");

  if (!EQ (Qnil, Vw32_mode_pointer_shape))
    {
      CHECK_NUMBER (Vw32_mode_pointer_shape, 0);
      mode_cursor = XCreateFontCursor (dsp,
					  XINT (Vw32_mode_pointer_shape));
    }
  else
    mode_cursor = XCreateFontCursor (dsp, XC_xterm);
  w32_check_errors ("bad modeline pointer cursor: %s");

  if (!EQ (Qnil, Vw32_sensitive_text_pointer_shape))
    {
      CHECK_NUMBER (Vw32_sensitive_text_pointer_shape, 0);
      cross_cursor
	= XCreateFontCursor (dsp,
			     XINT (Vw32_sensitive_text_pointer_shape));
    }
  else
    cross_cursor = XCreateFontCursor (dsp, XC_crosshair);

  /* Check and report errors with the above calls.  */
  w32_check_errors ("can't set cursor shape: %s");
  w32_uncatch_errors (FRAME_W32_DISPLAY (f), count);
  {
    XColor fore_color, back_color;

    fore_color.pixel = f->output_data.w32->mouse_pixel;
    back_color.pixel = mask_color;
    XQueryColor (dsp,
		 DefaultColormap (dsp,
				  DefaultScreen (dsp)),
		 &fore_color);
    XQueryColor (dsp,
		 DefaultColormap (dsp,
				  DefaultScreen (dsp)),
		 &back_color);
    XRecolorCursor (dsp, cursor,
		    &fore_color, &back_color);
    XRecolorCursor (dsp, nontext_cursor,
		    &fore_color, &back_color);
    XRecolorCursor (dsp, mode_cursor,
		    &fore_color, &back_color);
    XRecolorCursor (dsp, cross_cursor,
                    &fore_color, &back_color);
  }
  if (FRAME_W32_WINDOW (f) != 0)
    {
      XDefineCursor (XDISPLAY FRAME_W32_WINDOW (f), cursor);
    }

  if (cursor != f->output_data.w32->text_cursor && f->output_data.w32->text_cursor != 0)
      XFreeCursor (XDISPLAY f->output_data.w32->text_cursor);
  f->output_data.w32->text_cursor = cursor;
#ifdef HAVE_X11
  if (nontext_cursor != f->output_data.w32->nontext_cursor
      && f->output_data.w32->nontext_cursor != 0)
      XFreeCursor (XDISPLAY f->output_data.w32->nontext_cursor);
  f->output_data.w32->nontext_cursor = nontext_cursor;

  if (mode_cursor != f->output_data.w32->modeline_cursor
      && f->output_data.w32->modeline_cursor != 0)
      XFreeCursor (XDISPLAY f->output_data.w32->modeline_cursor);
  f->output_data.w32->modeline_cursor = mode_cursor;
  if (cross_cursor != f->output_data.w32->cross_cursor
      && f->output_data.w32->cross_cursor != 0)
      XFreeCursor (XDISPLAY f->output_data.w32->cross_cursor);
  f->output_data.w32->cross_cursor = cross_cursor;
#endif
#endif

  UNBLOCK_INPUT;
}

void
w32_set_cursor_color (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  unsigned long fore_pixel;

  if (!EQ (Vw32_cursor_fore_pixel, Qnil))
    fore_pixel = w32_decode_color (Vw32_cursor_fore_pixel, WHITE_PIX_DEFAULT);
  else
    fore_pixel = f->output_data.w32->background_pixel;
  f->output_data.w32->cursor_pixel = w32_decode_color (arg, BLACK_PIX_DEFAULT);
  
  /* Make sure that the cursor color differs from the background color.  */
  if (f->output_data.w32->cursor_pixel == f->output_data.w32->background_pixel)
    {
      f->output_data.w32->cursor_pixel = f->output_data.w32->mouse_pixel;
      if (f->output_data.w32->cursor_pixel == fore_pixel)
	fore_pixel = f->output_data.w32->background_pixel;
    }
  f->output_data.w32->cursor_foreground_pixel = fore_pixel;

  if (FRAME_W32_WINDOW (f) != 0)
    {
      if (FRAME_VISIBLE_P (f))
	{
	  w32_display_cursor (f, 0);
	  w32_display_cursor (f, 1);
	}
    }
}

/* Set the border-color of frame F to pixel value PIX.
   Note that this does not fully take effect if done before
   F has an x-window.  */

void
w32_set_border_pixel (f, pix)
     struct frame *f;
     int pix;
{
  f->output_data.w32->border_pixel = pix;

  if (FRAME_W32_WINDOW (f) != 0 && f->output_data.w32->border_width > 0)
    {
      if (FRAME_VISIBLE_P (f))
        redraw_frame (f);
    }
}

/* Set the border-color of frame F to value described by ARG.
   ARG can be a string naming a color.
   The border-color is used for the border that is drawn by the X server.
   Note that this does not fully take effect if done before
   F has an x-window; it must be redone when the window is created.

   Note: this is done in two routines because of the way X10 works.

   Note: under X11, this is normally the province of the window manager,
   and so emacs' border colors may be overridden. */

void
w32_set_border_color (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  CHECK_STRING (arg, 0);

  w32_set_border_pixel (f, w32_decode_color (arg, BLACK_PIX_DEFAULT));
}

void
w32_set_cursor_type (f, arg, oldval)
     FRAME_PTR f;
     Lisp_Object arg, oldval;
{
  if (EQ (arg, Qbar))
    FRAME_DESIRED_CURSOR (f) = bar_cursor;
  else if (EQ (arg, Qcaret))
    FRAME_DESIRED_CURSOR (f) = caret_cursor;
  else if (EQ (arg, Qcheckered_caret))
    FRAME_DESIRED_CURSOR (f) = checkered_caret_cursor;
  else if (EQ (arg, Qhairline_caret))
    FRAME_DESIRED_CURSOR (f) = hairline_caret_cursor;
  else
    FRAME_DESIRED_CURSOR (f) = filled_box_cursor;
  /* Error messages commented out because people have trouble fixing
     .Xdefaults with Emacs, when it has something bad in it.  */

  /* Make sure the cursor gets redrawn.  This is overkill, but how
     often do people change cursor types?  */
  update_mode_lines++;
}

void
w32_set_cursor_height (f, arg, oldval)
     FRAME_PTR f;
     Lisp_Object arg, oldval;
{
  if(XFASTINT (arg) >= 0 && XFASTINT (arg) < 5)
    FRAME_CURSOR_HEIGHT (f) = XFASTINT (arg);
  else
    FRAME_CURSOR_HEIGHT (f) = 4;	/* full height as default */
}

extern Lisp_Object w32_new_font ();

void
w32_set_font (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  Lisp_Object result;
  Lisp_Object fontset_name;

  CHECK_STRING (arg, 1);

  fontset_name = Fquery_fontset (arg, Qt);

  BLOCK_INPUT;

  result = (STRINGP (fontset_name)
	    ? w32_new_fontset (f, XSTRING (fontset_name)->data)
	    : w32_new_font (f, XSTRING (arg)->data));

  UNBLOCK_INPUT;
  
  if (EQ (result, Qnil))
    error ("Font `%s' is not defined", XSTRING (arg)->data);
  else if (EQ (result, Qt))
    error ("the characters of the given font have varying widths");
  else if (STRINGP (result))
    {
      recompute_basic_faces (f);
      store_frame_param (f, Qfont, result);
    }
  else
    abort ();
}

#ifdef IME_CONTROL
void
w32_set_frame_ime_font (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  fs_fonts *fs;
  MSG msg;
  CHECK_STRING (arg, 0);

  fs = mw32_get_font(XSTRING(arg)->data);
  if (!fs)
    {
      error("Font`%s' does not exist.", XSTRING(arg)->data);
    }
  SEND_INFORM_MESSAGE(FRAME_W32_WINDOW(f),
		      WM_MULE_IMM_SET_COMPOSITION_FONT,
		      (WPARAM) f,
		      (LPARAM) get_logfont(fs, 0));
#if 0
  WAIT_REPLY_MESSAGE(&msg, WM_MULE_IMM_SET_COMPOSITION_FONT_REPLY);
#endif
}
  
#endif

#ifdef MULE
void
w32_set_line_space (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  int upper, lower;
  char *s, *p;

  CHECK_STRING (arg, 0);
  s = (char *) XSTRING (arg)->data;

  upper = lower = 0;
  if (*s) {
    if (*s != '+')
      upper = atoi(s);
    if (p = (char *) index (s, '+')) {
      lower = atoi(p);
    } else {
      lower = upper - (upper / 2);
      upper /= 2;
    }
  }
  if (upper < 0) upper = 0;
  if (lower < 0) lower = 0;

  if (f->output_data.w32->upper_space != upper
      || f->output_data.w32->lower_space != lower) {
    f->output_data.w32->line_height +=
      upper - f->output_data.w32->upper_space +
	lower - f->output_data.w32->lower_space;
    f->output_data.w32->upper_space = upper;
    f->output_data.w32->lower_space = lower;
    if (FRAME_W32_WINDOW (f) != 0) {
      BLOCK_INPUT;
      x_set_window_size (f, 0, f->width, f->height);
      UNBLOCK_INPUT;
      SET_FRAME_GARBAGED (f);
    }
  }
}
#endif /* MULE */

void
w32_set_border_width (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  CHECK_NUMBER (arg, 0);

  if (XINT (arg) == f->output_data.w32->border_width)
    return;

  if (FRAME_W32_WINDOW (f) != 0)
    error ("Cannot change the border width of a window");

  f->output_data.w32->border_width = XINT (arg);
}

void
w32_set_internal_border_width (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  int mask;
  int old = f->output_data.w32->internal_border_width;

  CHECK_NUMBER (arg, 0);
  f->output_data.w32->internal_border_width = XINT (arg);
  if (f->output_data.w32->internal_border_width < 0)
    f->output_data.w32->internal_border_width = 0;

  if (f->output_data.w32->internal_border_width == old)
    return;

  if (FRAME_W32_WINDOW (f) != 0)
    {
      BLOCK_INPUT;
      x_set_window_size (f, 0, f->width, f->height);
      UNBLOCK_INPUT;
      SET_FRAME_GARBAGED (f);
    }
}

void
w32_set_visibility (f, value, oldval)
     struct frame *f;
     Lisp_Object value, oldval;
{
  Lisp_Object frame;
  XSETFRAME (frame, f);

  if (NILP (value))
    Fmake_frame_invisible (frame, Qt);
  else if (EQ (value, Qicon))
    Ficonify_frame (frame);
  else
    Fmake_frame_visible (frame);
}

/* to correspond to other emacs module....*/
void
x_set_menu_bar_lines (f, value, oldval)
     struct frame *f;
     Lisp_Object value, oldval;
{
  int nlines;
  int olines = FRAME_MENU_BAR_LINES (f);

  /* Right now, menu bars don't work properly in minibuf-only frames;
     most of the commands try to apply themselves to the minibuffer
     frame itslef, and get an error because you can't switch buffers
     in or split the minibuffer window.  */
  if (FRAME_MINIBUF_ONLY_P (f))
    return;

  if (INTEGERP(value))
    nlines = XINT (value);
  else
    nlines = 0;

  FRAME_MENU_BAR_LINES (f) = 0;
  if (nlines)
    {
      if (FRAME_EXTERNAL_MENU_BAR (f) == 0)
	set_frame_menubar (f, 1, 0);
      FRAME_EXTERNAL_MENU_BAR (f) = 1;
    }
  else
    {
      if (FRAME_EXTERNAL_MENU_BAR (f) == 1)
	free_frame_menubar (f);
      FRAME_EXTERNAL_MENU_BAR (f) = 0;
    }
}

/* Change the name of frame F to NAME.  If NAME is nil, set F's name to
       w32_id_name.

   If EXPLICIT is non-zero, that indicates that lisp code is setting the
       name; if NAME is a string, set F's name to NAME and set
       F->explicit_name; if NAME is Qnil, then clear F->explicit_name.

   If EXPLICIT is zero, that indicates that Emacs redisplay code is
       suggesting a new name, which lisp code should override; if
       F->explicit_name is set, ignore the new name; otherwise, set it.  */

void
w32_set_name (f, name, explicit)
     struct frame *f;
     Lisp_Object name;
     int explicit;
{
  /* Make sure that requests from lisp code override requests from 
     Emacs redisplay code.  */
  if (explicit)
    {
      /* If we're switching from explicit to implicit, we had better
	 update the mode lines and thereby update the title.  */
      if (f->explicit_name && NILP (name))
	update_mode_lines = 1;

      f->explicit_name = ! NILP (name);
    }
  else if (f->explicit_name)
    return;

  /* If NAME is nil, set the name to the w32_id_name.  */
  if (NILP (name))
    {
      /* Check for no change needed in this very common case
	 before we do any consing.  */
      if (!strcmp (w32_id_name, XSTRING (f->name)->data))
	return;
      name = build_string (w32_id_name);
    }
  else
    CHECK_STRING (name, 0);

  /* Don't change the name if it's already NAME.  */
  if (! NILP (Fstring_equal (name, f->name)))
    return;

  if (FRAME_W32_WINDOW (f))
    {
#ifdef MEADOW /* 96.10.12 modified by himi */
      MEADOW_ENCODE_ALLOC_PREDEFINE;
      int size;

      MEADOW_ENCODE_ALLOC(LISPY_STRING_BYTES(name) + 1);
      MEADOW_ENCODE(XSTRING(name) -> data,
		    LISPY_STRING_BYTES(name));
      size = MEADOW_ENCODE_PRODUCED;
      MEADOW_ENCODE_BUF[size] = (TCHAR) NULL;

#endif
      BLOCK_INPUT;
#ifdef MEADOW
      SetWindowText(FRAME_W32_WINDOW (f), MEADOW_ENCODE_BUF);
#else     
      SetWindowText(FRAME_W32_WINDOW (f), XSTRING(name)->data);
#endif
      UNBLOCK_INPUT;
    }

  f->name = name;
}

/* This function should be called when the user's lisp code has
   specified a name for the frame; the name will override any set by the
   redisplay code.  */
/* to correspond to other emacs module....*/
void
x_explicitly_set_name (f, arg, oldval)
     FRAME_PTR f;
     Lisp_Object arg, oldval;
{
  w32_set_name (f, arg, 1);
}

/* This function should be called by Emacs redisplay code to set the
   name; names set this way will never override names set by the user's
   lisp code.  */
/* to correspond to other emacs module....*/
void
x_implicitly_set_name (f, arg, oldval)
     FRAME_PTR f;
     Lisp_Object arg, oldval;
{
  w32_set_name (f, arg, 0);
}

void
w32_set_autoraise (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  f->auto_raise = !EQ (Qnil, arg);
}

void
w32_set_autolower (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  f->auto_lower = !EQ (Qnil, arg);
}

void
x_set_unsplittable (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  f->no_split = !NILP (arg);
}


void
w32_set_vertical_scroll_bars (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  if ((EQ (arg, Qleft) && FRAME_HAS_VERTICAL_SCROLL_BARS_ON_RIGHT (f))
      || (EQ (arg, Qright) && FRAME_HAS_VERTICAL_SCROLL_BARS_ON_LEFT (f))
      || (NILP (arg) && FRAME_HAS_VERTICAL_SCROLL_BARS (f))
      || (!NILP (arg) && ! FRAME_HAS_VERTICAL_SCROLL_BARS (f)))
    {
#ifdef W32_SCROLLBAR
      FRAME_VERTICAL_SCROLL_BAR_TYPE (f)
	= (NILP (arg)
	   ? vertical_scroll_bar_none
	   : EQ (Qright, arg)
	   ? vertical_scroll_bar_right 
	   : vertical_scroll_bar_left);

      /* We set this parameter before creating the X window for the
	 frame, so we can get the geometry right from the start.
	 However, if the window hasn't been created yet, we shouldn't
	 call x_set_window_size.  */
      if (FRAME_W32_WINDOW (f))
	x_set_window_size (f, 0, FRAME_WIDTH (f), FRAME_HEIGHT (f));
#endif
    }
}

void
w32_set_scroll_bar_width (f, arg, oldval)
     struct frame *f;
     Lisp_Object arg, oldval;
{
  if (NILP (arg))
    {
      FRAME_SCROLL_BAR_PIXEL_WIDTH (f) = 0;
      FRAME_SCROLL_BAR_COLS (f) = 3;
    }
  else if (INTEGERP (arg) && XINT (arg) > 0
	   && XFASTINT (arg) != FRAME_SCROLL_BAR_PIXEL_WIDTH (f))
    {
      int wid = FONT_WIDTH (f->output_data.w32->font);

      FRAME_SCROLL_BAR_PIXEL_WIDTH (f) = XFASTINT (arg);
      FRAME_SCROLL_BAR_COLS (f) = (XFASTINT (arg) + wid-1) / wid;
#ifdef W32_SCROLLBAR
      if (FRAME_W32_WINDOW (f))
	x_set_window_size (f, 0, FRAME_WIDTH (f), FRAME_HEIGHT (f));
#endif
    }
}


/* Subroutines of creating a frame.  */

/* Make sure that Vx_resource_name is set to a reasonable value.
   Fix it up, or set it to w32_id_name if it is too hopeless.  */

static void
validate_x_resource_name ()
{
  int len;
  /* Number of valid characters in the resource name.  */
  int good_count = 0;
  /* Number of invalid characters in the resource name.  */
  int bad_count = 0;
  Lisp_Object new;
  int i;

  if (STRINGP (Vx_resource_name))
    {
      unsigned char *p = XSTRING (Vx_resource_name)->data;
      int i;

      len = LISPY_STRING_BYTES(Vx_resource_name);

      /* Only letters, digits, - and _ are valid in resource names.
	 Count the valid characters and count the invalid ones.  */
      for (i = 0; i < len; i++)
	{
	  int c = p[i];
	  if (! ((c >= 'a' && c <= 'z')
		 || (c >= 'A' && c <= 'Z')
		 || (c >= '0' && c <= '9')
		 || c == '-' || c == '_'))
	    bad_count++;
	  else
	    good_count++;
	}
    }
  else
    /* Not a string => completely invalid.  */
    bad_count = 5, good_count = 0;

  /* If name is valid already, return.  */
  if (bad_count == 0)
    return;

  /* If name is entirely invalid, or nearly so, use `emacs'.  */
  if (good_count == 0
      || (good_count == 1 && bad_count > 0))
    {
      Vx_resource_name = build_string (w32_id_name);
      return;
    }

  /* Name is partly valid.  Copy it and replace the invalid characters
     with underscores.  */

  Vx_resource_name = new = Fcopy_sequence (Vx_resource_name);

  for (i = 0; i < len; i++)
    {
      int c = XSTRING (new)->data[i];
      if (! ((c >= 'a' && c <= 'z')
	     || (c >= 'A' && c <= 'Z')
	     || (c >= '0' && c <= '9')
	     || c == '-' || c == '_'))
	XSTRING (new)->data[i] = '_';
    }
}

DEFUN ("x-get-resource", Fx_get_resource, Sx_get_resource, 2, 4, 0,
  "Return the value of ATTRIBUTE, of class CLASS, from the X defaults database.\n\
This uses `INSTANCE.ATTRIBUTE' as the key and `Emacs.CLASS' as the\n\
class, where INSTANCE is the name under which Emacs was invoked, or\n\
the name specified by the `-name' or `-rn' command-line arguments.\n\
\n\
The optional arguments COMPONENT and SUBCLASS add to the key and the\n\
class, respectively.  You must specify both of them or neither.\n\
If you specify them, the key is `INSTANCE.COMPONENT.ATTRIBUTE'\n\
and the class is `Emacs.CLASS.SUBCLASS'.")
  (attribute, class, component, subclass)
     Lisp_Object attribute, class, component, subclass;
{
  register char *value;
  char *name_key;
  char *class_key;

  CHECK_STRING (attribute, 0);
  CHECK_STRING (class, 0);

  if (!NILP (component))
    CHECK_STRING (component, 1);
  if (!NILP (subclass))
    CHECK_STRING (subclass, 2);
  if (NILP (component) != NILP (subclass))
    error ("x-get-resource: must specify both COMPONENT and SUBCLASS or neither");

  validate_x_resource_name ();

  /* Allocate space for the components, the dots which separate them,
     and the final '\0'.  Make them big enough for the worst case.  */
  name_key = (char *) alloca (LISPY_STRING_BYTES (Vx_resource_name)
			      + (STRINGP (component)
				 ? LISPY_STRING_BYTES (component) : 0)
			      + LISPY_STRING_BYTES (attribute)
			      + 3);

  class_key = (char *) alloca ((sizeof (EMACS_RESOURCE_CLASS) - 1)
			       + LISPY_STRING_BYTES(class)
			       + (STRINGP (subclass)
				  ? LISPY_STRING_BYTES (subclass) : 0)
			       + 3);

  /* Start with emacs.FRAMENAME for the name (the specific one)
     and with `Emacs' for the class key (the general one).  */
  strcpy (name_key, XSTRING (Vx_resource_name)->data);
  strcpy (class_key, EMACS_RESOURCE_CLASS);

  strcat (class_key, ".");
  strcat (class_key, XSTRING (class)->data);

  if (!NILP (component))
    {
      strcat (class_key, ".");
      strcat (class_key, XSTRING (subclass)->data);

      strcat (name_key, ".");
      strcat (name_key, XSTRING (component)->data);
    }

  strcat (name_key, ".");
  strcat (name_key, XSTRING (attribute)->data);

  value = x_get_string_resource (Qnil,
				 name_key, class_key);

  if (value != (char *) 0)
    return build_string (value);
  else
    return Qnil;
}

/* Types we might convert a resource string into.  */
enum resource_types
  {
    number, eboolean, string, symbol
  };

/* Return the value of parameter PARAM.

   First search ALIST, then Vdefault_frame_alist, then the X defaults
   database, using ATTRIBUTE as the attribute name and CLASS as its class.

   Convert the resource to the type specified by desired_type.

   If no default is specified, return Qunbound.  If you call
   w32_get_arg, make sure you deal with Qunbound in a reasonable way,
   and don't let it get stored in any lisp-visible variables!  */

static Lisp_Object
w32_get_arg (alist, param, attribute, class, type)
     Lisp_Object alist, param;
     char *attribute;
     char *class;
     enum resource_types type;
{
  register Lisp_Object tem;

  tem = Fassq (param, alist);
  if (EQ (tem, Qnil))
    tem = Fassq (param, Vdefault_frame_alist);
  if (EQ (tem, Qnil))
    {

      if (attribute)
	{
	  tem = Fx_get_resource (build_string (attribute),
				   build_string (class),
				   Qnil, Qnil);

	  if (NILP (tem))
	    return Qunbound;

	  switch (type)
	    {
	    case number:
	      return make_number (atoi (XSTRING (tem)->data));

	    case eboolean:
	      tem = Fdowncase (tem);
	      if (!strcmp (XSTRING (tem)->data, "on")
		  || !strcmp (XSTRING (tem)->data, "true"))
		return Qt;
	      else 
		return Qnil;

	    case string:
	      return tem;

	    case symbol:
	      /* As a special case, we map the values `true' and `on'
		 to Qt, and `false' and `off' to Qnil.  */
	      {
		Lisp_Object lower;
		lower = Fdowncase (tem);
		if (!strcmp (XSTRING (lower)->data, "on")
		    || !strcmp (XSTRING (lower)->data, "true"))
		  return Qt;
		else if (!strcmp (XSTRING (lower)->data, "off")
		      || !strcmp (XSTRING (lower)->data, "false"))
		  return Qnil;
		else
		  return Fintern (tem, Qnil);
	      }

	    default:
	      abort ();
	    }
	}
      else
	return Qunbound;
    }
  return Fcdr (tem);
}

/* Record in frame F the specified or default value according to ALIST
   of the parameter named PARAM (a Lisp symbol).
   If no value is specified for PARAM, look for an X default for XPROP
   on the frame named NAME.
   If that is not found either, use the value DEFLT.  */

static Lisp_Object
w32_default_parameter (f, alist, prop, deflt, xprop, xclass, type)
     struct frame *f;
     Lisp_Object alist;
     Lisp_Object prop;
     Lisp_Object deflt;
     char *xprop;
     char *xclass;
     enum resource_types type;
{
  Lisp_Object tem;

  tem = w32_get_arg (alist, prop, xprop, xclass, type);
  if (EQ (tem, Qunbound))
    tem = deflt;
  x_set_frame_parameters (f, Fcons (Fcons (prop, tem), Qnil));
  return tem;
}

/* Calculate the desired size and position of this window,
   and return the flags saying which aspects were specified.

   This function does not make the coordinates positive.  */

#define DEFAULT_ROWS 40
#define DEFAULT_COLS 80

static int
w32_figure_window_size (f, parms)
     struct frame *f;
     Lisp_Object parms;
{
  register Lisp_Object tem0, tem1, tem2;
  int height, width, left, top;
  register int geometry;
  long window_prompting = 0;

  /* Default values if we fall through.
     Actually, if that happens we should get
     window manager prompting. */
  f->width = DEFAULT_COLS;
  f->height = DEFAULT_ROWS;
  /* Window managers expect that if program-specified
     positions are not (0,0), they're intentional, not defaults.  */
  f->output_data.w32->top_pos = 0;
  f->output_data.w32->left_pos = 0;

  tem0 = w32_get_arg (parms, Qheight, 0, 0, number);
  tem1 = w32_get_arg (parms, Qwidth, 0, 0, number);
  tem2 = w32_get_arg (parms, Quser_size, 0, 0, number);
  if (! EQ (tem0, Qunbound) || ! EQ (tem1, Qunbound))
    {
      if (!EQ (tem0, Qunbound))
	{
	  CHECK_NUMBER (tem0, 0);
	  f->height = XINT (tem0);
	}
      if (!EQ (tem1, Qunbound))
	{
	  CHECK_NUMBER (tem1, 0);
	  f->width = XINT (tem1);
	}
#if 0
      if (!NILP (tem2) && !EQ (tem2, Qunbound))
	window_prompting |= USSize;
      else
	window_prompting |= PSize;
#endif
    }

  f->output_data.w32->vertical_scroll_bar_extra
    = (!FRAME_HAS_VERTICAL_SCROLL_BARS (f)
       ? 0
       : FRAME_SCROLL_BAR_PIXEL_WIDTH (f) > 0
       ? FRAME_SCROLL_BAR_PIXEL_WIDTH (f)
       : (FRAME_SCROLL_BAR_COLS (f) * FONT_WIDTH (f->output_data.w32->font)));
  f->output_data.w32->pixel_width = CHAR_TO_PIXEL_WIDTH (f, f->width);
  f->output_data.w32->pixel_height = CHAR_TO_PIXEL_HEIGHT (f, f->height);

  tem0 = w32_get_arg (parms, Qtop, 0, 0, number);
  tem1 = w32_get_arg (parms, Qleft, 0, 0, number);
  tem2 = w32_get_arg (parms, Quser_position, 0, 0, number);
  if (! EQ (tem0, Qunbound) || ! EQ (tem1, Qunbound))
    {
      if (EQ (tem0, Qminus))
	{
	  f->output_data.w32->top_pos = 0;
#if 0
	  window_prompting |= YNegative;
#endif
	}
      else if (EQ (tem0, Qunbound))
	f->output_data.w32->top_pos = 0;
      else
	{
	  CHECK_NUMBER (tem0, 0);
	  f->output_data.w32->top_pos = XINT (tem0);
#if 0
	  if (f->output_data.w32->top_pos < 0)
	    window_prompting |= YNegative;
#endif
	}

      if (EQ (tem1, Qminus))
	{
	  f->output_data.w32->left_pos = 0;
#if 0
	  window_prompting |= XNegative;
#endif
	}
      else if (EQ (tem1, Qunbound))
	f->output_data.w32->left_pos = 0;
      else
	{
	  CHECK_NUMBER (tem1, 0);
	  f->output_data.w32->left_pos = XINT (tem1);
#if 0
	  if (f->output_data.w32->left_pos < 0)
	    window_prompting |= XNegative;
#endif
	}

#if 0
      if (!NILP (tem2))
	window_prompting |= USPosition;
      else
	window_prompting |= PPosition;
#endif
    }

  return window_prompting;
}

#if 0
HDC MapMode(hdc)
    HDC hdc;
{
    if (hdc) {
#if 0
	/* Make mapping mode be in 1/20 of point */
	
	SetMapMode (hdc, MM_ANISOTROPIC);
	SetWindowExtEx (hdc, 1440, 1440, NULL);
	SetViewportExtEx (hdc,
			  GetDeviceCaps (hdc, LOGPIXELSX),
			  GetDeviceCaps (hdc, LOGPIXELSY),
			  NULL);
#endif
    }

    return (hdc);
}
#endif

void w32_create_frame_window (struct frame *f, LPSTR title)
{
  HWND hwnd;

  hwnd = CreateWindowEx (f->output_data.w32->dwStyleEx,
			 EMACS_CLASS,
			 (LPSTR)title,
			 f->output_data.w32->dwStyle,
			 f->output_data.w32->left_pos,
			 f->output_data.w32->top_pos,
			 PIXEL_WIDTH(f),
			 PIXEL_HEIGHT(f),
			 NULL,
			 NULL,
			 hinst,
			 NULL);
  POST_THREAD_INFORM_MESSAGE(main_thread_id, WM_EMACS_CREATE_FRAME_REPLY,
			     (WPARAM) hwnd, (LPARAM) 0);
  DragAcceptFiles(hwnd, TRUE);

  if (w32_frame_window == INVALID_HANDLE_VALUE)
    w32_frame_window = hwnd;
}

static void w32_destroy_frame (HWND hwnd)
{
  Lisp_Object tail, frame;
  HWND hnextwnd;
  struct frame *f;

  ReplyMessage (1);
  DestroyWindow (hwnd);
  if (hwnd == w32_frame_window)
    {
      w32_frame_window = INVALID_HANDLE_VALUE;
      for (tail = Vframe_list;CONSP (tail);
	   tail = XCONS (tail)->cdr)
	{
	  frame = XCONS(tail)->car;
	  if (!FRAMEP (frame)) continue;
	  f = XFRAME (frame);
	  if (f->output_data.nothing == 1)
	    break;
	  hnextwnd = FRAME_W32_WINDOW (f);
	  if ((hnextwnd != hwnd)
	      && IsWindow(hnextwnd))
	    {
	      w32_frame_window = hnextwnd;
	      break;
	    }
	}
    }

  POST_THREAD_INFORM_MESSAGE (main_thread_id, 
			      WM_EMACS_DESTROY_FRAME_REPLY,
			      (WPARAM) 0, (LPARAM) 0);
  return;
}

static void w32_create_scrollbar (HWND hwnd_parent,
				  LPRECT lprect, HINSTANCE hinst)
{
  HWND hwnd;

  hwnd = CreateWindowEx(0L,
			"SCROLLBAR",
			(LPSTR) NULL,
			WS_CHILD | WS_VISIBLE | SBS_VERT,
			lprect->left,
			lprect->top,
			lprect->right,
			lprect->bottom,
			hwnd_parent,
			(HMENU) NULL,
			hinst, (LPVOID) NULL); 
  POST_THREAD_INFORM_MESSAGE (main_thread_id,
			      WM_EMACS_CREATE_SCROLLBAR_REPLY,
			      (WPARAM) hwnd, (LPARAM) 0);
  return;
}

static void w32_track_popup_menu (HWND parent, HANDLE hmenu, LPPOINT lppos)
{
  int flag;
  UINT track_flag;
  MSG msg2;
  extern unsigned int w32_mouse_grabbed;  /* defined in mw32term.c */

  track_flag = TPM_LEFTALIGN;
  if (GetAsyncKeyState(VK_LBUTTON)&0x8000) track_flag |= TPM_LEFTBUTTON;
  if (GetAsyncKeyState(VK_RBUTTON)&0x8000) track_flag |= TPM_RIGHTBUTTON;

  /* Menu tracking steales mouse button message so W32read_socket
     cannot reset w32_mouse_grabbed on releasing button when popup
     menu is activated. */
  w32_mouse_grabbed &= ~7;   /* clear all grabbing flags */
  
  lock_mouse_cursor_visible(TRUE);

  flag = TrackPopupMenu (hmenu,
			 track_flag,
			 lppos->x, lppos->y,
			 0,
			 parent,
			 NULL);

  lock_mouse_cursor_visible(FALSE);

  if (!flag)
    POST_THREAD_INFORM_MESSAGE (main_thread_id,
				WM_EMACS_POPUP_MENU_REPLY,
				(WPARAM) 0, (LPARAM) 0);
  else
    {
      flag = PeekMessage(&msg2, parent, WM_COMMAND, WM_COMMAND, PM_REMOVE);
      if (flag
	  && ((msg2.message == WM_COMMAND)
	      && (HIWORD(msg2.wParam) == 0)))
	{
	  POST_THREAD_INFORM_MESSAGE (main_thread_id,
				      WM_EMACS_POPUP_MENU_REPLY,
				      (WPARAM) msg2.wParam, (LPARAM) 0);
	}
      else
	{
	  POST_THREAD_INFORM_MESSAGE (main_thread_id,
				      WM_EMACS_POPUP_MENU_REPLY,
				      (WPARAM) 0, (LPARAM) 0);
	}
    }
}

void w32_ime_create_agent()
{
  HWND hwnd;
  hwnd = CreateWindowEx (0L,
			 CONVAGENT_CLASS, "Agent",
			 0,                          /* STYLE */
			 0, 0, 0, 0,
			 NULL,
			 NULL,
			 hinst,
			 NULL);
  POST_THREAD_INFORM_MESSAGE(main_thread_id, WM_MULE_IME_CREATE_AGENT_REPLY,
			     (WPARAM) hwnd, (LPARAM) 0);
}

static void w32_ime_destroy_agent(HWND hwnd)
{
  DestroyWindow(hwnd);
  POST_THREAD_INFORM_MESSAGE (main_thread_id, 
			      WM_MULE_IME_DESTROY_AGENT_REPLY,
			      (WPARAM) 0, (LPARAM) 0);
}

#ifndef W32_VER4
static HANDLE w32_ime_string_handle(HANDLE hStr)
{
  HANDLE hw32_ir_string;
  LPTSTR lpStr;
  LPTSTR lpCode;

  if (!hStr) return 0;
  lpStr = GlobalLock (hStr);
  if (!lpStr) return 0;
  hw32_ir_string = 
    GlobalAlloc (GMEM_MOVEABLE | GMEM_SHARE, 
		 strlen(lpStr) + 1);
  if (!hw32_ir_string) abort();
  lpCode = GlobalLock (hw32_ir_string);
  strcpy (lpCode, lpStr);
  GlobalUnlock (hw32_ir_string);
  GlobalUnlock (hStr);
  return hw32_ir_string;
}
#endif

/* Create and set up the X widget for frame F.  */

LRESULT CALLBACK w32_WndProc (hwnd, msg, wParam, lParam)
    HWND hwnd;
    UINT msg;
    WPARAM wParam;
    LPARAM lParam;
{
#ifdef IME_CONTROL
  extern int IME_event_off_count;
#endif
#ifdef W32_SCROLLBAR
  extern void w32_scroll_bar_store_event (WPARAM wParam, LPARAM lParam);
#endif
  extern void w32_menu_bar_store_activate_event (struct frame *f);

  struct frame *f;
  WPARAM SCParam;
  LRESULT ret = 1;

  f = w32_window_to_frame (hwnd);
  if (!f && !IS_EMACS_PRIVATE_MESSAGE(msg))
    return DefWindowProc (hwnd, msg, wParam, lParam);

  switch (msg) {
  case WM_ERASEBKGND:
    {
      RECT rect;

      GetClientRect (FRAME_W32_WINDOW (f), &rect);
      w32_clear_rect (f, MapMode ((HDC)wParam), &rect);
      ret = 0;

      if (ret == 0) return 0;
    }
    goto dflt;

  case WM_PAINT:
    {
      PAINTSTRUCT paintStruct;

      if (f->async_visible == 0)
	{
	  f->async_visible = 1;
	  f->async_iconified = 0;
	  SET_FRAME_GARBAGED (f);
	} 
      else
	{
	  RECT rect;

	  BeginPaint (hwnd, &paintStruct);
	  rect = paintStruct.rcPaint;
	  EndPaint (hwnd, &paintStruct);

	  W32_BLOCK_INPUT;
	  dumprectangle (f,
			 rect.left,
			 rect.top,
			 rect.right-rect.left+1,
			 rect.bottom-rect.top+1);
	  W32_UNBLOCK_INPUT;
	  ret = 0;
	}
    }
		    
    if (ret == 0) return 0;

    goto dflt;

  case WM_SETFOCUS:

    w32_new_focus_frame (f);

    /* This code is for switching selected-frame.  In order to
       generate no events before Emacs is set up, check whether
       selected_frame is initial terminal frame.  This is because
       input_pending should be false at the startup to show startup
       message. */
    if (selected_frame
	&& (selected_frame != XFRAME (Vterminal_frame))
	&& (f != selected_frame))
      PostMessage (hwnd, WM_EMACS_ACTIVATE, 
		   (WPARAM) 0, (LPARAM) 0);

    if ((FRAME_CURRENT_CURSOR (f) == caret_cursor ||
	 FRAME_CURRENT_CURSOR (f) == checkered_caret_cursor ||
	 FRAME_CURRENT_CURSOR (f) == hairline_caret_cursor))
      {
	if (f->output_data.w32->caret_state == 0)
	  {
	    CreateCaret(hwnd,
			(HBITMAP) (FRAME_CURRENT_CURSOR (f)
				   == checkered_caret_cursor ? 1 : 0),
			FRAME_CURRENT_CURSOR (f) 
			== hairline_caret_cursor ?
			0 : w32_get_glyph_width(f, f->phys_cursor_glyph),
			(FRAME_FONT_HEIGHT (f) * FRAME_CURSOR_HEIGHT (f)) / 4);
	    f->output_data.w32->caret_state = 1;
	  }

	if (f->phys_cursor_x >= 0)
	  {
	    int cx, cy;

	    cx = f->phys_cursor_x;
	    cy = f->phys_cursor_y;
	    SetCaretPos(CHAR_TO_PIXEL_COL(f, cx),
			CHAR_TO_PIXEL_ROW(f, cy + 1) -
			(FRAME_FONT_HEIGHT (f) * FRAME_CURSOR_HEIGHT (f)) / 4);
	    if (f->output_data.w32->caret_state == 1)
	      {
		ShowCaret(hwnd);
		f->output_data.w32->caret_state = 2;
	      }
	  }
      }

#if 0
#ifdef IME_CONTROL
    w32_set_ime_font(hwnd, 
		     get_logfont(f->output_data.w32->font, 0));
#endif
#endif

    return 0;

  case WM_KILLFOCUS:

    if (f->output_data.w32->caret_state > 0)
      {
	f->output_data.w32->caret_state = 0;
	DestroyCaret();
      }

    if (f == w32_focus_frame)
      w32_new_focus_frame (NULL);

    PostMessage (hwnd, WM_EMACS_CLEAR_MOUSE_FACE, 
		 (WPARAM) 0, (LPARAM) 0);
    return 0;

  case WM_MOVE:
	
    if (!f->async_iconified && f->async_visible)
      {
	RECT rect;

	GetWindowRect(hwnd, &rect);

	f->output_data.w32->left_pos = rect.left;
	f->output_data.w32->top_pos = rect.top;
      }

    break;

  case WM_SIZE:

    switch (wParam)
      {
      case SIZE_MINIMIZED:
	f->async_visible = 0;
	f->async_iconified = 1;
	break;
      case SIZE_MAXIMIZED:
      case SIZE_RESTORED:
	{
	  RECT rect;

	  GetWindowRect(hwnd, &rect);
 
	  f->async_visible = 1;
	  f->async_iconified = 0;
	  f->output_data.w32->left_pos = rect.left;
	  f->output_data.w32->top_pos = rect.top;
	  SET_FRAME_GARBAGED (f);
	}
      }

    if (!f->async_iconified && f->async_visible)
      {
	RECT rect;
	int rows;
	int columns;
	int width;
	int height;
	    
	GetClientRect(hwnd, &rect);

	height = rect.bottom - rect.top;
	width = rect.right - rect.left;

	rows = PIXEL_TO_CHAR_HEIGHT (f, height);
	columns = PIXEL_TO_CHAR_WIDTH (f, width);

	/* Even if the number of character rows and columns has
	   not changed, the font size may have changed, so we need
	   to check the pixel dimensions as well.  */
	    
	if (columns != f->width
	    || rows != f->height
	    || width != f->output_data.w32->pixel_width
	    || height != f->output_data.w32->pixel_height)
	  {
	    //          change_frame_size (f, rows, columns, 0, 0);
	    change_frame_size (f, rows, columns, 0, 1);
	    SET_FRAME_GARBAGED (f);
	    
	    f->output_data.w32->pixel_width = width;
	    f->output_data.w32->pixel_height = height;
	  }
      }

    break;

  case WM_NCMOUSEMOVE:
    PostMessage (hwnd, WM_EMACS_CLEAR_MOUSE_FACE, 
		 (WPARAM) 0, (LPARAM) 0);
    goto dflt;

#if 0

  case WM_WINDOWPOSCHANGING:
	
    if (!f->async_iconified && f->async_visible &&
	!f->output_data.w32->frame_change_state)
      {
	LPWINDOWPOS lppos = (LPWINDOWPOS) lParam;

	f->output_data.w32->frame_change_state = 1;

	if (!(lppos->flags & SWP_NOSIZE))
	  {
	    RECT rect;
	    int wdiff;
	    int hdiff;

	    memset (&rect, 0, sizeof(rect));
	    AdjustWindowRectEx (&rect,
				f->output_data.w32->dwStyle,
				FRAME_EXTERNAL_MENU_BAR (f),
				f->output-h_data.w32->dwStyleEx);

	    /* All windows have an extra pixel so subtract 1 */

#ifdef MW32_FONT
	    wdiff = (lppos->cx - (rect.right - rect.left) - 1 -
		     f->output_data.w32->vertical_scroll_bar_extra) %
		       FRAME_FONT_WIDTH(f);
#else
	    wdiff = (lppos->cx - (rect.right - rect.left) - 1) %
	      FONT_WIDTH (f->output_data.w32->font);
#endif
	    hdiff = (lppos->cy - (rect.bottom - rect.top) - 1) %
	      f->output_data.w32->line_height;

	    if (wdiff || hdiff)
	      {
		lppos->cx -= wdiff;
		lppos->cy -= hdiff;
		return 0;
	      }
	  }
      }

    goto dflt;

#else
  case WM_WINDOWPOSCHANGED:
	
    if (!f->async_iconified && f->async_visible &&
	f->output_data.w32->frame_change_state < 2)
      /* To adjust window correctly,
	 we must check size of the window twice
	 (strictly speaking number of dimention),
	 thus, width and height. */
      {
	LPWINDOWPOS lppos = (LPWINDOWPOS) lParam;

	f->output_data.w32->frame_change_state++;

	if ((!(lppos->flags & SWP_NOSIZE)) ||
	    (lppos->flags & SWP_DRAWFRAME))
	  {
	    RECT rect;
	    int wdiff;
	    int hdiff;

	    GetClientRect (hwnd, &rect);
#if 0
	    fprintf(stderr, "CHANGED-A:WindowRect:%d, %d, %d, %d\n",
		    rect.left, rect.top, rect.right, rect.bottom);
#endif


#ifdef MW32_FONT
	    wdiff = ((rect.right - rect.left)
		     - 2 * FRAME_INTERNAL_BORDER_WIDTH(f)
		     - f->output_data.w32->vertical_scroll_bar_extra) 
	      % FRAME_FONT_WIDTH(f);
#else
	    wdiff = ((rect.right - rect.left)
		     - 2 * FRAME_INTERNAL_BORDER_WIDTH(f)
		     - f->output_data.w32->vertical_scroll_bar_extra)
	      % FONT_WIDTH(f->output_data.w32->font);
#endif
	    hdiff = ((rect.bottom - rect.top)
		     - 2 * FRAME_INTERNAL_BORDER_WIDTH(f))
	      % f->output_data.w32->line_height;

	    if (wdiff || hdiff)
	      {
		if ((2 * wdiff) > FRAME_FONT_WIDTH(f))
		  wdiff -= FRAME_FONT_WIDTH(f);
		if ((2 * hdiff) > f->output_data.w32->line_height)
		  hdiff -= f->output_data.w32->line_height;

		rect.right = lppos->cx - wdiff;
		rect.bottom = lppos->cy - hdiff;
		SetWindowPos (hwnd, NULL, 0, 0,
			      rect.right, rect.bottom,
			      SWP_NOZORDER | SWP_NOMOVE);
		goto dflt;
	      }
	  }
      }
    f->output_data.w32->frame_change_state = 0;

    goto dflt;
#endif

  case WM_CLOSE:
    PostMessage (hwnd, WM_EMACS_DESTROY, wParam, lParam);
    return 0;

  case WM_CREATE:
    {
      HDC hdc = MyGetDC(hwnd);

      /* Make mapping mode be in 1/20 of point */

      MapMode (hdc);

      ReleaseDC (hwnd, hdc);
    }
    return 0;

  case WM_INITMENU:
    {
      if (f->output_data.w32->menubar_handle == (HMENU) wParam)
	w32_menu_bar_store_activate_event (f);
      return 0;
    }
  case WM_EXITMENULOOP:
    {
      if (!wParam) /* not track popup menu */
	f->output_data.w32->disable_reconstruct_menubar = 0;
      lock_mouse_cursor_visible (FALSE);
      return 0;
    }

#ifdef W32_SCROLLBAR
  case WM_VSCROLL:
    {
      w32_scroll_bar_store_event(wParam, lParam);
      return 0;
    }
#endif
  case WM_EMACS_FLASH_WINDOW:
    {
      FlashWindow (hwnd, TRUE);
      Sleep(100);
      FlashWindow (hwnd, FALSE);
      break;
    }
#if defined(MEADOW) && defined(IME_CONTROL)
#ifdef W32_VER4
  case WM_IME_NOTIFY:
    if (wParam == IMN_SETOPENSTATUS)
      {
	if (!IME_event_off_count)
	  PostMessage (hwnd, WM_MULE_IME_STATUS, 0, 0);
	else
	  IME_event_off_count--;
      }
    goto dflt;

  case WM_IME_COMPOSITION:
    {
      if (lParam & GCS_RESULTSTR) {
	extern BOOL w32_get_ime_composion_string (HWND);
	if (w32_get_ime_composition_string (hwnd))
	  return 0;
	else 
	  break;
      }
      goto dflt;
    }
#else /* not W32_VER4 */

  case WM_IME_REPORT:
    switch (wParam)
      {
      case IR_STRING:
	{
	  HANDLE himestr;

	  himestr = w32_ime_string_handle((HANDLE)lParam);
	  if (!himestr) break;

	  PostMessage(NULL, WM_MULE_IME_REPORT, 
		      (WPARAM) himestr, (LPARAM) f);
	  return 1;
	}
      case IR_OPENCONVERT:
      case IR_CLOSECONVERT:
	if (!IME_event_off_count)
	  PostMessage(hwnd, WM_MULE_IME_STATUS, 0, 0);
	else
	  IME_event_off_count--;
	return 0;
      }
#endif /* not W32_VER4 */
  case WM_MULE_IMM_SET_COMPOSITION_FONT:
    w32_set_ime_font(hwnd, (LPLOGFONT) lParam);
    POST_THREAD_INFORM_MESSAGE(main_thread_id,
			       WM_MULE_IMM_SET_COMPOSITION_FONT_REPLY,
			       (WPARAM) 0, (LPARAM) 0);
    break;
#endif /* not MEADOW and IME_CONTROL */

    /* Emacs private message entries. */
  case WM_EMACS_CREATE_FRAME:
    w32_create_frame_window((struct frame*)wParam, (LPSTR)lParam);
    break;

#ifdef W32_SCROLLBAR	  
  case WM_EMACS_CREATE_SCROLLBAR:
    w32_create_scrollbar(hwnd, (LPRECT)wParam, (HINSTANCE)lParam);
    break;
#endif
#ifdef IME_CONTROL
  case WM_MULE_IME_CREATE_AGENT:
    w32_ime_create_agent ();
    break;
  case WM_MULE_IME_DESTROY_AGENT:
    w32_ime_destroy_agent (hwnd);
    break;
#endif
  case WM_EMACS_DESTROY_FRAME:
    w32_destroy_frame (hwnd);
    break;

  case WM_EMACS_POPUP_MENU:
    w32_track_popup_menu (hwnd, (HANDLE)wParam, (LPPOINT)lParam);
    break;

  case WM_EMACS_SETCARET:
    if (f->phys_cursor_x >= 0)
      {
	if (f->output_data.w32->caret_state > 0)
	  {
	    HideCaret(hwnd);
	    DestroyCaret();
	    f->output_data.w32->caret_state = 0;
	    if (wParam == 0) break;
	  }
	if ((f == w32_focus_frame)
	    && (f->output_data.w32->caret_state == 0))
	  {
	    CreateCaret(hwnd,
		      (HBITMAP) (FRAME_CURRENT_CURSOR (f)
				 == checkered_caret_cursor ? 1 : 0),
			FRAME_CURRENT_CURSOR (f) == hairline_caret_cursor ?
			0 : w32_get_glyph_width(f, f->phys_cursor_glyph),
			(FRAME_FONT_HEIGHT (f) * FRAME_CURSOR_HEIGHT (f)) / 4);
	    f->output_data.w32->caret_state = 1;
	  }
	if (f->output_data.w32->caret_state > 0)
	  {
	    int cx, cy;
	    cx = f->phys_cursor_x;
	    cy = f->phys_cursor_y;
	    SetCaretPos(CHAR_TO_PIXEL_COL(f, cx),
			CHAR_TO_PIXEL_ROW(f, cy + 1) -
			(FRAME_FONT_HEIGHT (f) * FRAME_CURSOR_HEIGHT (f)) / 4);
	    if ((wParam == 0)
		&& (f->output_data.w32->caret_state == 2))
	      {
		HideCaret(hwnd);
		f->output_data.w32->caret_state = 1;
	      }
	    else if ((wParam == 1)
		     && (f->output_data.w32->caret_state == 1))
	      {
		ShowCaret(hwnd);
		f->output_data.w32->caret_state = 2;
	      }
	  }
      }
    else
      {
	if (f->output_data.w32->caret_state == 2)
	  {
	    HideCaret(hwnd);
	    f->output_data.w32->caret_state = 1;
	  }
      }
    break;

    /* end of Emacs private message entries. */

  default:
#if defined(MEADOW) && defined(IME_CONTROL) && defined(W32_VER4)
    {
      extern LRESULT CALLBACK
	conversion_agent_wndproc(HWND hwnd, UINT message,
				 WPARAM wparam, LPARAM lparam);
      if (MESSAGE_IMM_COM_P(msg))
	return conversion_agent_wndproc(hwnd, msg, wParam, lParam);
    }
#endif

  dflt:
    return DefWindowProc (hwnd, msg, wParam, lParam);
  }

  return 1;
}

BOOL w32_init_app (hinst)
    HINSTANCE hinst;
{
    WNDCLASS wc;

    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = (WNDPROC) w32_WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hinst;
    wc.hIcon = LoadIcon (hinst, EMACS_CLASS);
    wc.hCursor = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject (WHITE_BRUSH);
    wc.lpszMenuName = NULL;
    wc.lpszClassName = EMACS_CLASS;

    return (RegisterClass (&wc));
}

static void
w32_window (f, window_prompting, minibuffer_only)
     struct frame *f;
     long window_prompting;
     int minibuffer_only;
{
  HWND hwnd;
  char* name;
  MSG msg;

  BLOCK_INPUT;

  if (STRINGP (f->name))
    name = (char*) XSTRING (f->name)->data;
  else
    name = w32_id_name;

  SEND_MSGTHREAD_INFORM_MESSAGE (WM_EMACS_CREATE_FRAME,
				 (WPARAM)f, (LPARAM)name);
  WAIT_REPLY_MESSAGE (&msg, WM_EMACS_CREATE_FRAME_REPLY);

  hwnd = (HWND)msg.wParam;
  FRAME_W32_WINDOW (f) = hwnd;

  f->output_data.w32->mainthread_to_frame_handle
    = CreateEvent(0, TRUE, TRUE, NULL);

  validate_x_resource_name ();

  if (!minibuffer_only && FRAME_EXTERNAL_MENU_BAR (f))
    initialize_frame_menubar (f);

  w32_calc_absolute_position (f);

  /* w32_set_name normally ignores requests to set the name if the
     requested name is the same as the current name.  This is the one
     place where that assumption isn't correct; f->name is set, but
     the X server hasn't been told.  */
  {
    Lisp_Object name;
    int explicit = f->explicit_name;

    f->explicit_name = 0;
    name = f->name;
    f->name = Qnil;
    w32_set_name (f, name, explicit);
  }

#if 0
  XDefineCursor (XDISPLAY FRAME_W32_WINDOW (f),
		 f->output_data.w32->text_cursor);
#endif

  UNBLOCK_INPUT;

  if (FRAME_W32_WINDOW (f) == 0)
    error ("Unable to create window");

}

DEFUN ("x-create-frame", Fx_create_frame, Sx_create_frame,
       1, 1, 0,
  "Make a new window, which is called a \"frame\" in Emacs terms.\n\
Return an Emacs frame object representing the window.\n\
ALIST is an alist of frame parameters.\n\
If the parameters specify that the frame should not have a minibuffer,\n\
and do not specify a specific minibuffer window to use,\n\
then `default-minibuffer-frame' must be a frame whose minibuffer can\n\
be shared by the new frame.")
  (parms)
     Lisp_Object parms;
{
  struct frame *f;
  Lisp_Object frame, tem;
  Lisp_Object name;
  int minibuffer_only = 0;
  long window_prompting = 0;
  int width, height;
  int count = specpdl_ptr - specpdl;
  struct gcpro gcpro1;
  Lisp_Object display;
  struct kboard *kb;

  /* #ifdef MULTI_KBOARD */
#if MULTI_KBOARD
  kb = dpyinfo->kboard;
#else
  kb = &the_only_kboard;
#endif

  /* Use this general default value to start with
     until we know if this frame has a specified name.  */
  Vx_resource_name = Vinvocation_name;

#if 0
  display = x_get_arg (dpyinfo, parms, Qdisplay, 0, 0, string);
  if (EQ (display, Qunbound))
    display = Qnil;
  dpyinfo = check_x_display_info (display);
#else
  display = Qnil;
#endif

  name = w32_get_arg (parms, Qname, "name", "Name", string);
  if (!STRINGP (name)
      && ! EQ (name, Qunbound)
      && ! NILP (name))
    error ("x-create-frame: name parameter must be a string");

  if (STRINGP (name))
    Vx_resource_name = name;

  tem = w32_get_arg (parms, Qminibuffer, 0, 0, symbol);
  if (EQ (tem, Qnone) || NILP (tem))
    f = make_frame_without_minibuffer (Qnil, kb, display);
  else if (EQ (tem, Qonly))
    {
      f = make_minibuffer_frame ();
      minibuffer_only = 1;
    }
  else if (WINDOWP(tem))
    f = make_frame_without_minibuffer (tem, kb, display);
  else
    f = make_frame (1);

#ifdef W32_SCROLLBAR
  FRAME_CAN_HAVE_SCROLL_BARS (f) = 1;
  FRAME_VERTICAL_SCROLL_BAR_TYPE (f) = vertical_scroll_bar_right;
#else
  FRAME_CAN_HAVE_SCROLL_BARS (f) = 0;
#endif
  f->scroll_bar_pixel_width = GetSystemMetrics (SM_CXVSCROLL);

  /* Set the name; the functions to which we pass f expect the name to
     be set.  */
  if (EQ (name, Qunbound) || NILP (name))
    {
      f->name = build_string (w32_id_name);
      f->explicit_name = 0;
    }
  else
    {
      f->name = name;
      f->explicit_name = 1;
      /* use the frame's title when getting resources for this frame.  */
      specbind (Qx_resource_name, name);
    }

  XSETFRAME (frame, f);
  GCPRO1 (frame);

  f->output_method = output_w32;
  f->output_data.w32 = (struct w32_output *) xmalloc (sizeof (struct w32_output));
  bzero (f->output_data.w32, sizeof (struct w32_output));

  /* all handles must be set to INVALID_HANLE_VALUE.  */
  f->output_data.w32->menubar_handle = INVALID_HANDLE_VALUE;

  /* Note that the frame has no physical cursor right now.  */
  f->phys_cursor_x = -1;

  /* Set the name; the functions to which we pass f expect the name to
     be set.  */
  if (EQ (name, Qunbound) || NILP (name))
    {
      f->name = build_string (w32_id_name);
      f->explicit_name = 0;
    }
  else
    {
      f->name = name;
      f->explicit_name = 1;
      /* use the frame's title when getting resources for this frame.  */
      specbind (Qx_resource_name, name);
    }

  /* Create fontsets from `global_fontset_alist' before handling fonts.  */
  for (tem = Vglobal_fontset_alist; CONSP (tem); tem = XCONS (tem)->cdr)
    fs_register_fontset (f, XCONS (tem)->car);

  /* Extract the window parameters from the supplied values
     that are needed to determine window geometry.  */
  {
    Lisp_Object font, fontset;

    font = w32_get_arg (parms, Qfont, "font", "Font", string);
    if (! STRINGP (font))
      font = build_string ("default");

    fontset = Fquery_fontset (font, Qt);
    if (STRINGP (fontset))
      font = w32_new_fontset (f, XSTRING (fontset)->data);
    else
      font = w32_new_font (f, XSTRING (font)->data);

    if (!STRINGP(font))
      {
	/* Initial font cannot be created,
	   we should display detail informations
	   as far as possible. */
	if (SYMBOLP(font))
	  error ("Cannot select initial font:%s",
		 XSYMBOL(font)->name);
	else
	  error ("Cannot select initial font");
      }

    w32_default_parameter (f, parms, Qfont, font, 
			   "font", "Font", string);

#ifdef IME_CONTROL
    w32_default_parameter (f, parms, Qime_font,
			   build_string(FRAME_FONT(f)->name),
			   "ime-font", "IME-Font", string);
#endif
  }

#ifdef MULE
  w32_default_parameter (f, parms, Qline_space, build_string ("2"),
		       "linespace", "LineSpace", string);
#endif  /* MULE */

  w32_default_parameter (f, parms, Qborder_width, make_number (0),
		       "borderwidth", "BorderWidth", number);
  /* This defaults to 0 in order to match w32.  We recognize either
     internalBorderWidth or internalBorder (which is what xterm calls
     it).  */
  if (NILP (Fassq (Qinternal_border_width, parms)))
    {
      Lisp_Object value;

      value = w32_get_arg (parms, Qinternal_border_width,
			 "internalBorder", "BorderWidth", number);
      if (! EQ (value, Qunbound))
	parms = Fcons (Fcons (Qinternal_border_width, value),
		       parms);
    }
  w32_default_parameter (f, parms, Qinternal_border_width, make_number (1),
			 "internalBorderWidth", "BorderWidth", number);
  w32_default_parameter (f, parms, Qvertical_scroll_bars, Qt,
			 "verticalScrollBars", "ScrollBars", eboolean);

  /* Also do the stuff which must be set before the window exists. */
  w32_default_parameter (f, parms, Qforeground_color, build_string ("black"),
			 "foreground", "Foreground", string);
  w32_default_parameter (f, parms, Qbackground_color, build_string ("white"),
			 "background", "Background", string);
  w32_default_parameter (f, parms, Qmouse_color, build_string ("black"),
			 "pointerColor", "Foreground", string);
  w32_default_parameter (f, parms, Qcursor_color, build_string ("black"),
			 "cursorColor", "Foreground", string);
  w32_default_parameter (f, parms, Qborder_color, build_string ("black"),
			 "borderColor", "BorderColor", string);

  w32_default_parameter (f, parms, Qmenu_bar_lines, make_number (1),
			 "menuBarLines", "MenuBarLines", number);

  f->output_data.w32->parent_desc = ROOT_WINDOW;

  f->output_data.w32->dwStyle = WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN;
  f->output_data.w32->dwStyleEx = WS_EX_CLIENTEDGE;
  
  window_prompting = w32_figure_window_size (f, parms);

#if 0
  if (window_prompting & XNegative)
    {
      if (window_prompting & YNegative)
	f->output_data.w32->win_gravity = SouthEastGravity;
      else
	f->output_data.w32->win_gravity = NorthEastGravity;
    }
  else
    {
      if (window_prompting & YNegative)
	f->output_data.w32->win_gravity = SouthWestGravity;
      else
	f->output_data.w32->win_gravity = NorthWestGravity;
    }

  f->output_data.w32->size_hint_flags = window_prompting;
#endif

  w32_window (f, window_prompting, minibuffer_only);
  init_frame_faces (f);

  /* We need to do this after creating the X window, so that the
     icon-creation functions can say whose icon they're describing.  */
  w32_default_parameter (f, parms, Qauto_raise, Qnil,
			 "autoRaise", "AutoRaiseLower", eboolean);
  w32_default_parameter (f, parms, Qauto_lower, Qnil,
			 "autoLower", "AutoRaiseLower", eboolean);
  w32_default_parameter (f, parms, Qcursor_type, Qcaret,
			 "cursorType", "CursorType", symbol);
  w32_default_parameter (f, parms, Qcursor_height, make_number (4),
			 "cursorHeight", "CursorHeight", number);

  /* Dimensions, especially f->height, must be done via change_frame_size.
     Change will not be effected unless different from the current
     f->height. */
  width = f->width;
  height = f->height;
  f->height = f->width = 0;
  change_frame_size (f, height, width, 1, 0);
  x_set_window_size (f, 0, width, height);

  tem = w32_get_arg (parms, Qunsplittable, 0, 0, eboolean);
  f->no_split = minibuffer_only || EQ (tem, Qt);

  UNGCPRO;

  /* It is now ok to make the frame official
     even if we get an error below.
     And the frame needs to be on Vframe_list
     or making it visible won't work.  */
  Vframe_list = Fcons (frame, Vframe_list);

  /* Make the window appear on the frame and enable display,
     unless the caller says not to.  */
  {
    Lisp_Object visibility;
    int exist_frames = CONSP (Vframe_list);

    visibility = w32_get_arg (parms, Qvisibility, 0, 0, symbol);
    if (EQ (visibility, Qunbound))
      visibility = Qt;

    if (EQ (visibility, Qicon))
      x_iconify_frame (f);
    else
      {
	x_make_frame_visible (f);
	w32_new_focus_frame (f);
	SetForegroundWindow (FRAME_W32_WINDOW(f));
      }
  }

  return unbind_to (count, frame);
}

/* to correspond to other emacs module....*/
Lisp_Object
x_get_focus_frame ()
{
  Lisp_Object xfocus;
  if (! w32_focus_frame)
    return Qnil;

  XSETFRAME (xfocus, w32_focus_frame);
  return xfocus;
}

DEFUN ("w32-focus-frame", Fw32_focus_frame, Sw32_focus_frame, 1, 1, 0,
  "Give FRAME input focus, raising to foreground if necessary.")
  (frame)
     Lisp_Object frame;
{
  CHECK_LIVE_FRAME (frame, 0);

  x_focus_on_frame (XFRAME (frame));
  return Qnil;
}



DEFUN ("x-color-defined-p", Fx_color_defined_p, Sx_color_defined_p, 1, 1, 0,
  "Return non-nil if the X display supports the color named COLOR.")
  (color)
     Lisp_Object color;
{
  Color foo;
  
  CHECK_STRING (color, 0);

  if (defined_color (color, &foo, 0))
    return Qt;
  else
    return Qnil;
}

DEFUN ("w32-color-values", Fw32_color_values, Sw32_color_values, 1, 1, 0,
  "Return a description of the color named COLOR.\n\
The value is a list of integer RGB values--(RED GREEN BLUE).\n\
These values appear to range from 0 to 255; white is (255 255 255).")
  (color)
     Lisp_Object color;
{
  Color foo;

  CHECK_STRING (color, 0);

  if (defined_color (color, &foo, 0))
    {
      Lisp_Object rgb[3];

      rgb[0] = make_number (GetRValue(foo));
      rgb[1] = make_number (GetGValue(foo));
      rgb[2] = make_number (GetBValue(foo));
      return Flist (3, rgb);
    }
  else
    return Qnil;
}

DEFUN ("x-display-color-p", Fx_display_color_p, Sx_display_color_p, 0, 1, 0,
  "Return t if the X screen currently in use supports color.")
  (display)
     Lisp_Object display;
{
    HDC hdc;
    int cap;

    hdc = MyGetDC (ROOT_WINDOW);

    cap = GetDeviceCaps(hdc, BITSPIXEL) *
      GetDeviceCaps(hdc, PLANES);

    ReleaseDC (ROOT_WINDOW, hdc);
    
    return (cap <= 2)? Qnil : Qt;
}

DEFUN ("x-display-grayscale-p", Fx_display_grayscale_p, Sx_display_grayscale_p,
  0, 1, 0,
  "Return t if this system supports shades of gray.\n\
Note that color displays do support shades of gray.\n\
The optional argument DISPLAY is neglected.")
  (display)
     Lisp_Object display;
{
  if (w32_screen_planes <= 1)
    return Qnil;

  return Qt;
}

DEFUN ("x-display-pixel-width", Fx_display_pixel_width, Sx_display_pixel_width,
  0, 1, 0,
  "Returns the width in pixels of the X display DISPLAY.\n\
The optional argument DISPLAY specifies which display to ask about.\n\
DISPLAY should be either a frame or a display name (a string).\n\
If omitted or nil, that stands for the selected frame's display.\n\
On Windows, DISPLAY is ignored.")
  (display)
     Lisp_Object display;
{
  int width;
  HDC hdc;
  HWND hwnd = ROOT_WINDOW;
  hdc = MyGetDC (hwnd);
  width = GetDeviceCaps (hdc, HORZRES);
  ReleaseDC (hwnd, hdc);
  return make_number (width);
}

DEFUN ("x-display-pixel-height", Fx_display_pixel_height,
  Sx_display_pixel_height, 0, 1, 0,
  "Returns the height in pixels of the X display DISPLAY.\n\
The optional argument DISPLAY specifies which display to ask about.\n\
DISPLAY should be either a frame or a display name (a string).\n\
If omitted or nil, that stands for the selected frame's display.\n\
On Windows, DISPLAY is ignored.")
  (display)
     Lisp_Object display;
{
  int height;
  HDC hdc;
  HWND hwnd = ROOT_WINDOW;
  hdc = MyGetDC (hwnd);
  height = GetDeviceCaps (hdc, VERTRES);
  ReleaseDC (hwnd, hdc);
  return make_number (height);
}

DEFUN ("x-display-planes", Fx_display_planes, Sx_display_planes,
  0, 1, 0,
  "Returns the number of bitplanes of the display DISPLAY.\n\
The optional argument DISPLAY specifies which display to ask about.\n\
DISPLAY should be either a frame or a display name (a string).\n\
If omitted or nil, that stands for the selected frame's display.\n\
On Windows, DISPLAY is ignored.")
  (display)
{
    int planes;
    HWND hwnd = ROOT_WINDOW;
    HDC hdc;
    hdc = MyGetDC (hwnd);
    planes = GetDeviceCaps(hdc, PLANES) *
      GetDeviceCaps(hdc, BITSPIXEL);
    /* 96.4.27 by himi */
    ReleaseDC (hwnd, hdc);
    return make_number (planes);
}

DEFUN ("x-display-color-cells", Fx_display_color_cells, Sx_display_color_cells,
  0, 1, 0,
  "Returns the number of color cells of the display DISPLAY.\n\
The optional argument DISPLAY specifies which display to ask about.\n\
DISPLAY should be either a frame or a display name (a string).\n\
If omitted or nil, that stands for the selected frame's display.\n\
On Windows, DISPLAY is ignored.")
  (display)
     Lisp_Object display;
{
  int colors;
  HDC hdc;
  HWND hwnd = ROOT_WINDOW;
  hdc = MyGetDC (hwnd);
  colors = GetDeviceCaps (hdc, NUMCOLORS);
  ReleaseDC (hwnd, hdc);
  return make_number (colors);
}

DEFUN ("x-display-mm-height", Fx_display_mm_height, Sx_display_mm_height, 0, 1, 0,
  "Returns the height in millimeters of the X display DISPLAY.\n\
The optional argument DISPLAY specifies which display to ask about.\n\
DISPLAY should be either a frame or a display name (a string).\n\
If omitted or nil, that stands for the selected frame's display.\n\
On Windows, DISPLAY is ignored.")
  (display)
     Lisp_Object display;
{
  HDC hdc;
  int display_height;
  HWND hwnd = ROOT_WINDOW;
  hdc = MyGetDC (hwnd);
  display_height = GetDeviceCaps (hdc, VERTSIZE);
  ReleaseDC (hwnd, hdc);
  return make_number (display_height);
}

DEFUN ("x-display-mm-width", Fx_display_mm_width, Sx_display_mm_width, 0, 1, 0,
  "Returns the width in millimeters of the X display DISPLAY.\n\
The optional argument DISPLAY specifies which display to ask about.\n\
DISPLAY should be either a frame or a display name (a string).\n\
If omitted or nil, that stands for the selected frame's display.\n\
On Windows, DISPLAY is ignored.")
  (display)
     Lisp_Object display;
{
    HDC hdc;
    int display_width;
    HWND hwnd = ROOT_WINDOW;
    hdc = MyGetDC (hwnd);
    display_width = GetDeviceCaps (hdc,HORZSIZE);
    ReleaseDC (hwnd, hdc);
    return make_number (display_width);
}


/* to correspond to other emacs module....*/
x_pixel_width (f)
     register struct frame *f;
{
  return PIXEL_WIDTH (f);
}

x_pixel_height (f)
     register struct frame *f;
{
  return PIXEL_HEIGHT (f);
}

x_char_width (f)
     register struct frame *f;
{
#ifdef MW32_FONT
  return FRAME_FONT_WIDTH(f);
#else
  return FONT_WIDTH (f->output_data.w32->font);
#endif
}

x_char_height (f)
     register struct frame *f;
{
  return f->output_data.w32->line_height;
}


DEFUN ("x-open-connection", Fx_open_connection, Sx_open_connection,
       1, 3, 0, "Open a connection to an X server.\n\
DISPLAY is the name of the display to connect to.\n\
(on Windows, it is neglected.)\n\
Optional second arg XRM_STRING is a string of resources in xrdb format.\n\
On Meadow, It is bogus. It Only initializes system.")
  (display, xrm_string, must_succeed)
     Lisp_Object display, xrm_string, must_succeed;
{
  HDC hdc;

  /* 95.10.20 Modified by himi */
  w32_init_app(hinst);
    
  /* This is what opens the connection and sets dsp.
     This also initializes many symbols, such as those used for input. */

  w32_term_init ();

  Vw32_color_map = Fw32_default_color_map();

  w32gui_open = 1;

  hdc = MyGetDC (ROOT_WINDOW);
  
  w32_screen_planes = GetDeviceCaps (hdc, PLANES) *
    GetDeviceCaps (hdc, BITSPIXEL);
  w32_screen_height = GetDeviceCaps(hdc, VERTRES);
  w32_screen_width  = GetDeviceCaps(hdc, HORZRES);
  w32_screen_height_in = GetDeviceCaps(hdc, LOGPIXELSX);
  w32_screen_width_in  = GetDeviceCaps(hdc, LOGPIXELSY);

  ReleaseDC (ROOT_WINDOW, hdc);

  Vx_resource_name = Vinvocation_name;

  return Qnil;
}

DEFUN ("x-close-current-connection", Fx_close_current_connection,
       Sx_close_current_connection,
       0, 0, 0, "Close the current connection. It is noon on Meadow.")
  ()
{
  w32gui_open = 0;
  POST_THREAD_INFORM_MESSAGE (msg_thread_id, 
			      WM_EMACS_CLOSE_CONNECTION, 
			      (WPARAM) 0, (LPARAM) 0);
  return Qnil;
}

DEFUN ("w32-send-sys-command", Fw32_send_sys_command, Sw32_send_sys_command, 1, 2, 0,
   "Send frame a Windows WM_SYSCOMMAND message of type COMMAND.\n\
Some useful values for command are 0xf030 to maximise frame (0xf020\n\
to minimize), 0xf120 to restore frame to original size, and 0xf100\n\
to activate the menubar for keyboard access.  0xf140 activates the\n\
screen saver if defined.\n\
\n\
If optional parameter FRAME is not specified, use selected frame.")
  (command, frame)
     Lisp_Object command, frame;
{
  WPARAM code;
  FRAME_PTR f = check_x_frame (frame);

  CHECK_NUMBER (command, 0);

  PostMessage (FRAME_W32_WINDOW (f), WM_SYSCOMMAND, XINT (command), 0);

  return Qnil;
}


syms_of_w32fns ()
{
  /* The section below is built by the lisp expression at the top of the file,
     just above where these variables are declared.  */
  /*&&& init symbols here &&&*/
  Qauto_raise = intern ("auto-raise");
  staticpro (&Qauto_raise);
  Qauto_lower = intern ("auto-lower");
  staticpro (&Qauto_lower);
  Qbackground_color = intern ("background-color");
  staticpro (&Qbackground_color);
  Qbar = intern ("bar");
  staticpro (&Qbar);
  Qcaret = intern ("caret");
  staticpro (&Qcaret);
  Qcheckered_caret = intern ("checkered-caret");
  staticpro (&Qcheckered_caret);
  Qhairline_caret = intern ("hairline-caret");
  staticpro (&Qhairline_caret);
  Qborder_color = intern ("border-color");
  staticpro (&Qborder_color);
  Qborder_width = intern ("border-width");
  staticpro (&Qborder_width);
  Qbox = intern ("box");
  staticpro (&Qbox);
  Qcursor_color = intern ("cursor-color");
  staticpro (&Qcursor_color);
  Qcursor_type = intern ("cursor-type");
  staticpro (&Qcursor_type);
  Qcursor_height = intern ("cursor-height");
  staticpro (&Qcursor_height);
  Qfont = intern ("font");
  staticpro (&Qfont);
  Qforeground_color = intern ("foreground-color");
  staticpro (&Qforeground_color);
  Qgeometry = intern ("geometry");
  staticpro (&Qgeometry);
  Qicon_left = intern ("icon-left");
  staticpro (&Qicon_left);
  Qicon_top = intern ("icon-top");
  staticpro (&Qicon_top);
  Qicon_type = intern ("icon-type");
  staticpro (&Qicon_type);
  Qinternal_border_width = intern ("internal-border-width");
  staticpro (&Qinternal_border_width);
  Qleft = intern ("left");
  staticpro (&Qleft);
  Qright = intern ("right");
  staticpro (&Qright);
  Qmouse_color = intern ("mouse-color");
  staticpro (&Qmouse_color);
  Qnone = intern ("none");
  staticpro (&Qnone);
  Qparent_id = intern ("parent-id");
  staticpro (&Qparent_id);
  Qsuppress_icon = intern ("suppress-icon");
  staticpro (&Qsuppress_icon);
  Qtop = intern ("top");
  staticpro (&Qtop);
  Qundefined_color = intern ("undefined-color");
  staticpro (&Qundefined_color);
  Qvertical_scroll_bars = intern ("vertical-scroll-bars");
  staticpro (&Qvertical_scroll_bars);
  Qvisibility = intern ("visibility");
  staticpro (&Qvisibility);
  Qwindow_id = intern ("window-id");
  staticpro (&Qwindow_id);
  Qx_frame_parameter = intern ("x-frame-parameter");
  staticpro (&Qx_frame_parameter);
  Qx_resource_name = intern ("x-resource-name");
  staticpro (&Qx_resource_name);
  Quser_position = intern ("user-position");
  staticpro (&Quser_position);
  Quser_size = intern ("user-size");
  staticpro (&Quser_size);
#ifdef MULE
  Qline_space = intern ("line-space");
  staticpro (&Qline_space);
#endif  /* MULE */
  Qime_font = intern ("ime-font");
  staticpro (&Qime_font);
  /* This is the end of symbol initialization.  */

  Fput (Qundefined_color, Qerror_conditions,
	Fcons (Qundefined_color, Fcons (Qerror, Qnil)));
  Fput (Qundefined_color, Qerror_message,
	build_string ("Undefined color"));

  init_w32_parm_symbols ();

  DEFVAR_LISP ("w32-pointer-shape", &Vw32_pointer_shape,
    "The shape of the pointer when over text.\n\
Changing the value does not affect existing frames\n\
unless you set the mouse color.");
  Vw32_pointer_shape = Qnil;

  Vw32_nontext_pointer_shape = Qnil;

  Vw32_mode_pointer_shape = Qnil;

  DEFVAR_INT ("w32-sensitive-text-pointer-shape",
	      &Vw32_sensitive_text_pointer_shape,
	      "The shape of the pointer when over mouse-sensitive text.\n\
This variable takes effect when you create a new frame\n\
or when you set the mouse color.");
  Vw32_sensitive_text_pointer_shape = Qnil;

  DEFVAR_LISP ("w32-cursor-fore-pixel", &Vw32_cursor_fore_pixel,
	       "A string indicating the foreground color of the cursor box.");
  Vw32_cursor_fore_pixel = Qnil;

  DEFVAR_LISP ("w32-color-map", &Vw32_color_map,
	       "A array of color name mappings for windows.");
  Vw32_color_map = Qnil;

  DEFVAR_LISP ("x-resource-name", &Vx_resource_name,
    "The name Emacs uses to look up resources; for internal use only.\n\
`x-get-resource' uses this as the first component of the instance name\n\
when requesting resource values.\n\
Emacs initially sets `x-resource-name' to the name under which Emacs\n\
was invoked, or to the value specified with the `-name' or `-rn'\n\
switches, if present.");
  Vx_resource_name = Qnil;

  defsubr (&Sx_get_resource);
  defsubr (&Sx_display_color_p);

  defsubr (&Sx_color_defined_p);
  defsubr (&Sw32_color_values);
  defsubr (&Sx_display_pixel_width);
  defsubr (&Sx_display_pixel_height);
  defsubr (&Sx_display_mm_width);
  defsubr (&Sx_display_mm_height);
  defsubr (&Sx_display_planes);
  defsubr (&Sx_display_color_cells);
  defsubr (&Sx_create_frame);
  defsubr (&Sx_open_connection);
  defsubr (&Sx_close_current_connection);

  defsubr (&Sw32_default_color_map);
  defsubr (&Sw32_focus_frame);
  defsubr (&Sw32_send_sys_command);

}

#undef abort

void w32_abort()
{
    MessageBox (NULL,
		"A fatal error has occurred - aborting!",
		"Emacs Abort Dialog",
		MB_OK|MB_ICONEXCLAMATION);
    RaiseException(100, EXCEPTION_NONCONTINUABLE, 0, NULL);
    abort();
}

