;; -*- Mode: Emacs-Lisp -*-
;;
;; elscreen-gf.el
;;
(defconst elscreen-gf-version "1.4.0 (November 19, 2005)")
;;
;; Author:   Naoto Morishima <naoto@morishima.net>
;; Based on: grep-family.el
;;              by Youki Kadobayashi <youki-k@is.aist-nara.ac.jp>
;; Created:  June 23, 1996
;; Revised:  November 19, 2005

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(provide 'elscreen-gf)
(require 'elscreen)
(eval-when-compile
  (require 'static))

(static-unless (fboundp 'read-directory-name)
  (require 'read-directory-name))

;;; User Customizable Variables:

(defgroup elscreen-gf nil
  "ElScreen-GF -- Grep Family on ElScreen"
  :tag "ElScreen-GF"
  :group 'tools)

(defcustom elscreen-gf-truncate-lines t
  "If non-nil, truncate each line in ElScreen-GF mode."
  :type 'boolean
  :tag "Truncate lines in ElScreen-GF mode."
  :group 'elscreen-gf)

(defface elscreen-gf-selected-entry-face
  '((t (:underline t :overline)))
  "Face used for the selected entry in ElScreen-GF mode."
  :group 'elscreen-gf)

(defface elscreen-gf-filename-face
  '((((class color) (background dark))
     (:foreground "khaki" :bold t))
    (((class color) (background light))
     (:foreground "DarkSlateBlue" :bold t))
    (t (:bold t)))
  "Face used for filename in ElScreen-GF mode."
  :group 'elscreen-gf)

(defface elscreen-gf-line-number-face
  '((((class color) (background dark))
     (:foreground "gray" :bold t))
    (((class color) (background light))
     (:foreground "gray50" :bold t))
    (t (:bold t)))
  "Face used for line number in ElScreen-GF mode."
  :group 'elscreen-gf)

(defface elscreen-gf-pattern-face
  '((((class color) (background dark))
     (:foreground "tomato" :bold t))
    (((class color) (background light))
     (:foreground "tomato" :bold t))
    (t (:bold t)))
  "Face used to emphasize the specified keyword in ElScreen-GF mode."
  :group 'elscreen-gf)

(defface elscreen-gf-emphasis-after-jump-face
  '((((class color) (background light))
     (:background "LightSteelBlue1"))
    (((class color) (background dark))
     (:background "LightSteelBlue1"))
    (t (:bold t)))
  "Face used to emphasize the selected line after jump."
  :group 'elscreen-gf)

;;; Key bindings:

(defvar elscreen-gf-map (make-sparse-keymap)
  "*Keymap for elscreen-gf.")
(define-key elscreen-gf-map  "G" 'elscreen-gf-execute-grep)
(define-key elscreen-gf-map  "g" 'elscreen-gf-execute-gid)
(define-key elscreen-gf-map  "m" 'elscreen-gf-execute-mkid)
(define-key elscreen-gf-map  "v" 'elscreen-gf-display-version)

(define-key elscreen-map  "\C-g" elscreen-gf-map)

(defvar elscreen-gf-mode-map (make-sparse-keymap)
  "keymap used in elscreen-gf mode.")
(define-key elscreen-gf-mode-map "n"    'elscreen-gf-mode-next-line)
(define-key elscreen-gf-mode-map "p"    'elscreen-gf-mode-previous-line)
(define-key elscreen-gf-mode-map " "    'elscreen-gf-mode-scroll-up)
(define-key elscreen-gf-mode-map "\177" 'elscreen-gf-mode-scroll-down)
(define-key elscreen-gf-mode-map "<"    'elscreen-gf-mode-beginning-of-buffer)
(define-key elscreen-gf-mode-map ">"    'elscreen-gf-mode-end-of-buffer)
(define-key elscreen-gf-mode-map "N"    'elscreen-gf-mode-next-file)
(define-key elscreen-gf-mode-map "P"    'elscreen-gf-mode-previous-file)
(define-key elscreen-gf-mode-map "t"    'elscreen-gf-mode-truncate-lines-toggle)
(define-key elscreen-gf-mode-map "o"    'elscreen-gf-mode-jump-to-entry)
(define-key elscreen-gf-mode-map "v"    'elscreen-gf-display-version)


;;; Code:

(defun-maybe line-number-at-pos (&optional pos)
  (let ((opoint (or pos (point))))
    (save-excursion
      (goto-char opoint)
      (forward-line 0)
      (1+ (count-lines (point-min) (point))))))


(defsubst elscreen-gf-overlay-create (start end face)
  (let ((overlay (make-overlay start end)))
    (overlay-put overlay 'face face)
    (overlay-put overlay 'evaporate t)
    overlay))

(defsubst elscreen-gf-move-overlay-create (overlay-symbol start end face)
  (let ((overlay (condition-case nil
		     (symbol-value overlay-symbol)
		   (error nil))))
    (if (overlayp overlay)
	(move-overlay overlay start end (current-buffer))
      (set overlay-symbol (elscreen-gf-overlay-create start end face))
      (setq overlay (symbol-value overlay-symbol)))
    overlay))

;;; ElScreen-GF mode

(defvar elscreen-gf-mode-buffer nil)
(defun elscreen-gf-goto-screen-create (target-directory)
  (unless (buffer-live-p elscreen-gf-mode-buffer)
    (setq elscreen-gf-mode-buffer (get-buffer-create "ElScreen-GF")))
  (elscreen-find-and-goto-by-buffer elscreen-gf-mode-buffer 'create)
  (switch-to-buffer elscreen-gf-mode-buffer)
  (elscreen-gf-mode target-directory)
  (let ((buffer-read-only nil))
    (erase-buffer)))

(defvar elscreen-gf-pattern)
(defvar elscreen-gf-selected-entry-overlay)
(defun elscreen-gf-mode (target-directory)
  "Major mode for jumping to the entries.

Key bindings:
\\{elscreen-gf-mode-map}"
  (setq major-mode 'elscreen-gf-mode)
  (setq mode-name "ElScreen-GF")
  (use-local-map elscreen-gf-mode-map)

  (setq buffer-read-only t)
  (setq truncate-lines elscreen-gf-truncate-lines)
  (set (make-local-variable 'elscreen-gf-pattern) nil)
  (set (make-local-variable 'elscreen-gf-selected-entry-overlay) nil)
  (setq default-directory target-directory)

  (auto-fill-mode nil))

(defun elscreen-gf-mode-selected-entry-overlay ()
  (elscreen-gf-move-overlay-create
   'elscreen-gf-selected-entry-overlay
   (line-beginning-position) (line-end-position)
   'elscreen-gf-selected-entry-face))

(defun elscreen-gf-mode-next-line ()
  "Move the current entry vertically down."
  (interactive)
  (let ((current-line (line-number-at-pos)))
    (cond
     ((< current-line 4)
      (goto-line 4))
     ((< current-line (line-number-at-pos (point-max)))
      (next-line 1)))
    (elscreen-gf-mode-selected-entry-overlay)))

(defun elscreen-gf-mode-previous-line ()
  "Move the current entry vertically up."
  (interactive)
  (let ((current-line (line-number-at-pos)))
    (cond
     ((< 4 current-line)
      (previous-line 1))
     (t
      (goto-line 4)))
    (elscreen-gf-mode-selected-entry-overlay)))

(defun elscreen-gf-mode-scroll-up ()
  "Scroll entries upward full screen."
  (interactive)
  (scroll-up)
  (elscreen-gf-mode-selected-entry-overlay))

(defun elscreen-gf-mode-scroll-down ()
  "Scroll entries downward full screen."
  (interactive)
  (scroll-down)
  (elscreen-gf-mode-selected-entry-overlay))

(defun elscreen-gf-mode-beginning-of-buffer ()
  "Move the current entry to the beginning of the entries."
  (interactive)
  (goto-line 4)
  (elscreen-gf-mode-selected-entry-overlay))

(defun elscreen-gf-mode-end-of-buffer ()
  "Move the current entry to the end of the entries."
  (interactive)
  (goto-char (point-max))
  (forward-line -1)
  (elscreen-gf-mode-selected-entry-overlay))

(defun elscreen-gf-mode-next-file ()
  (interactive)
  (cond
   ((save-excursion
      (beginning-of-line)
      (looking-at "^\\([^:\n]+\\):\\([0-9]+\\):"))
    (let ((filename (match-string 1)))
      (goto-char (point-max))
      (re-search-backward (concat "^" filename ":"))
      (forward-line)))
   (t
    (re-search-forward "^\\([^:\n]+\\):\\([0-9]+\\):" nil t)
    (beginning-of-line)))
  (elscreen-gf-mode-selected-entry-overlay))

(defun elscreen-gf-mode-previous-file ()
  (interactive)
  (cond
   ((save-excursion
      (beginning-of-line)
      (looking-at "^\\([^:\n]+\\):\\([0-9]+\\):"))
    (let ((filename (match-string 1)))
      (goto-char (point-min))
      (re-search-forward (concat "^" filename ":"))
      (forward-line -1)
      (beginning-of-line)))
   (t
    (re-search-backward "^\\([^:\n]+\\):\\([0-9]+\\):" nil t)))
  (when (looking-at "^\\([^:\n]+\\):\\([0-9]+\\):")
    (let ((filename (match-string 1)))
      (goto-char (point-min))
      (re-search-forward (concat "^" filename ":"))
      (beginning-of-line)))
  (elscreen-gf-mode-selected-entry-overlay))

(defun elscreen-gf-mode-truncate-lines-toggle ()
  "Toggle truncated lines."
  (interactive)
  (setq truncate-lines (not truncate-lines))
  (recenter))

(defvar elscreen-gf-emphasis-after-jump-overlay nil)
(defun elscreen-gf-mode-jump-to-entry ()
  "Jump to the current entry."
  (interactive)
  (when (save-excursion
	  (beginning-of-line)
	  (looking-at "^\\([^:\n]+\\):\\([0-9]+\\):"))
    (let ((filename (match-string 1))
	  (line (string-to-number (match-string 2)))
	  (pattern elscreen-gf-pattern))
      (isearch-update-ring pattern)
      (elscreen-find-file filename)
      (goto-line line)
      (re-search-forward
       (format "[^a-zA-Z0-9_]\\(%s\\)[^a-zA-Z0-9_]" pattern)
       (line-end-position) t)
      (goto-char (match-beginning 1))
      (elscreen-gf-move-overlay-create
       'elscreen-gf-emphasis-after-jump-overlay
       (line-beginning-position) (line-end-position)
       'elscreen-gf-emphasis-after-jump-face))))

;;; mkid / gid / grep

(defvar elscreen-gf-running-process nil)
(defun elscreen-gf-execute-exclusive-p ()
  (cond
   ((and (processp elscreen-gf-running-process)
	 (eq (process-status elscreen-gf-running-process) 'run))
    (message "Sorry, %s is running now. Try again later."
	     (process-name elscreen-gf-running-process))
    nil)
   (t
    (setq elscreen-gf-running-process nil)
    t)))

(defun elscreen-gf-execute-mkid ()
  "Run mkid, with user-specified args."
  (interactive)
  (when (elscreen-gf-execute-exclusive-p)
    (let* ((default-directory (read-directory-name "Run mkid (target): "
						   nil default-directory t))
	   (mkid-command "mkid"))
      (message "Running mkid...")
      (setq elscreen-gf-running-process
	    (start-process "mkid" nil "sh" "-c" mkid-command))
      (set-process-sentinel elscreen-gf-running-process
			    'elscreen-gf-mkid-sentinel))))

(defun elscreen-gf-mkid-sentinel (process event)
  (message "Running mkid... %s"
	   (or (and (string-match "finished" event) "done") "error"))
  (setq elscreen-gf-running-process nil))

(defun elscreen-gf-grep/gid-token-at-point ()
  (let ((thing (or (get major-mode 'elscreen-gf-grep/gid-thing) 'word)))
    (thing-at-point thing)))

(put 'c-mode 'elscreen-gf-grep/gid-thing 'c-token)
(defvar elscreen-gf-token-chars-c-mode "[:alnum:]_")
(put 'c-token 'end-op
     (lambda ()
       (re-search-forward
	(concat "\\=[" elscreen-gf-token-chars-c-mode "]*") nil t)))
(put 'c-token 'beginning-op
     (lambda ()
       (if (re-search-backward
	    (concat "[^" elscreen-gf-token-chars-c-mode "]") nil t)
	   (forward-char)
	 (goto-char (point-min)))))

(put 'emacs-lisp-mode 'elscreen-gf-grep/gid-thing 'emacs-lisp-token)
(defvar elscreen-gf-token-chars-emacs-lisp-mode "[:alnum:]/+:-")
(put 'emacs-lisp-token 'end-op
     (lambda ()
       (re-search-forward
	(concat "\\=[" elscreen-gf-token-chars-emacs-lisp-mode "]*") nil t)))
(put 'emacs-lisp-token 'beginning-op
     (lambda ()
       (if (re-search-backward
	    (concat "[^" elscreen-gf-token-chars-emacs-lisp-mode "]") nil t)
	   (forward-char)
	 (goto-char (point-min)))))

(defun elscreen-gf-execute-grep ()
  "Run grep, with user-specified args, and collect output
in the ElScreen-GF buffer."
  (interactive)
  (when (elscreen-gf-execute-exclusive-p)
    (let* ((buffer-file-name (buffer-file-name))
	   (file-name-re
	    (or
	     (and buffer-file-name
		  (string-match "\.[^.]+$" buffer-file-name)
		  (concat "*" (match-string 0 buffer-file-name)))
	     "*"))
	   (pattern (read-string
		     "Run grep (pattern): "
		     (cons (or (elscreen-gf-grep/gid-token-at-point) "") 1)))
	   (file-name-re (read-string
			  "Run grep (files): "
			  (cons file-name-re 1) nil "*")))
      (setq elscreen-gf-pattern pattern)
      (elscreen-gf-goto-screen-create default-directory)
      (elscreen-gf-grep/gid-run-command
       (list "grep" "-n" pattern file-name-re)))))

(defun elscreen-gf-execute-gid ()
  "Run gid, with user-specified args, and collect output
in the ElScreen-GF buffer."
  (interactive)
  (when (elscreen-gf-execute-exclusive-p)
    (let ((pattern (read-string
		    "Run gid (pattern): "
		    (cons (or (elscreen-gf-grep/gid-token-at-point) "") 1)))
	  (thing (or (get major-mode 'elscreen-gf-grep/gid-thing) 'word)))
      (elscreen-gf-goto-screen-create default-directory)
      (setq elscreen-gf-pattern pattern)
      (put 'elscreen-gf-mode 'elscreen-gf-grep/gid-thing thing)
      (elscreen-gf-grep/gid-run-command (list "gid" pattern)))))

(defun elscreen-gf-grep/gid-run-command (command-list)
  (let ((command-name (car command-list))
	(command-line (mapconcat (lambda (v) v) command-list " "))
	(buffer-read-only nil))
    (message "Running %s..." command-name)
    (insert "DIR: " (abbreviate-file-name default-directory) "\n"
	    "CMD: " command-line "\n\n")
    (setq elscreen-gf-running-process
	  (start-process command-name (current-buffer)
			 "sh" "-c" command-line "2> /dev/null"))
    (set-process-filter elscreen-gf-running-process
			'elscreen-gf-grep/gid-filter)
    (set-process-sentinel elscreen-gf-running-process
			  'elscreen-gf-grep/gid-sentinel)))

(defun elscreen-gf-grep/gid-filter (process string)
  (set-buffer (process-buffer process))
  (save-excursion
    (let ((buffer-read-only nil)
	  (start (point-max)))
      (goto-char start)
      (save-excursion (insert string))
      (while (not (eobp))
	(beginning-of-line)
	(when (looking-at "^\\([^:\n]+\\):\\([0-9]+\\):")
	  (elscreen-gf-overlay-create (match-beginning 1) (match-end 1)
				      'elscreen-gf-filename-face)
	  (elscreen-gf-overlay-create (match-beginning 2) (match-end 2)
				      'elscreen-gf-line-number-face)
	  (goto-char (match-end 2))
	  (while (re-search-forward
		  (format "[^a-zA-Z0-9_]\\(%s\\)[^a-zA-Z0-9_]"
			  elscreen-gf-pattern)
		  (line-end-position) t)
	    (elscreen-gf-overlay-create (match-beginning 1) (match-end 1)
					'elscreen-gf-pattern-face)))
	(forward-line)))))

(defun elscreen-gf-grep/gid-sentinel (process event)
  (set-buffer (process-buffer process))
  (message "Running %s... %s"
	   (process-name elscreen-gf-running-process)
	   (or (and (string-match "finished" event) "done") "error"))
  (setq elscreen-gf-running-process nil)
  (elscreen-gf-mode-selected-entry-overlay))

;;; Help

(defvar elscreen-gf-help "ElScreen-GF keys:
       \\[elscreen-gf-execute-grep]    Run grep
       \\[elscreen-gf-execute-gid]    Run gid
       \\[elscreen-gf-execute-mkid]    Run mkid
       \\[elscreen-gf-display-version]    Display ElScreen-GF version")
(elscreen-set-help 'elscreen-gf-help)

(defun elscreen-gf-display-version ()
  "Display ElScreen-GF version."
  (interactive)
  (elscreen-message (concat "ElScreen-GF version " elscreen-gf-version)))
