;;; pukiwiki-mode.el -- Major mode for Pukiwiki editing -*- coding: iso-2022-7bit -*-

;; Copyright (C) 2003 Hideaki Hori

;; Author: Hideaki Hori <yowaken@cool.ne.jp>

;; $Id: pukiwiki-mode.el,v 1.22 2004/06/06 13:48:15 akihisa Exp $
;;
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or (at
;; your option) any later version.

;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; usage:
;;
;; Put the following in ~/.emacs
;;
;;  (setq pukiwiki-site-list
;;        '(("Meadow" "http://www.bookshelf.jp/pukiwiki/pukiwiki.php" nil euc-jp-dos)
;;          ))
;;  (setq pukiwiki-auto-insert t)
;;  (autoload 'pukiwiki-edit "pukiwiki-mode" nil t)
;;  (autoload 'pukiwiki-index "pukiwiki-mode" nil t)
;;  (autoload 'pukiwiki-edit-url "pukiwiki-mode" nil t)

;; $B$3$N%b!<%I$O(B hiki-mode $B$r%Y!<%9$KB?$/$NJQ99$,2C$($i$l$F$$$^$9!%%P%0$J$I$N:]$K(B
;; $B$O(B akihisa@mail.ne.jp $B$+(B
;; http://www.bookshelf.jp/pukiwiki/pukiwiki.php?%A5%A2%A5%A4%A5%C7%A5%A2%BD%B8%2Fpukiwiki-mode
;; $B$KJs9p$r$*4j$$$7$^$9!%(BHideaki Hori $BMM$K?R$M$k$3$H$O$7$J$$$G$/$@$5$$!%(B

;; Hi-lock: (("^;;; .*" (0 (quote hi-black-hb) t)))
;; Hi-lock: ( ("make-variable-buffer-\\(local\\)" (0 font-lock-keyword-face)(1 'italic append)))))
;; Hi-lock: end

(defvar pukiwiki-site-list
      '(("Meadow" "http://www.bookshelf.jp/pukiwiki/pukiwiki.php" nil euc-jp-dos)))
;;(setq pukiwiki-browser-function 'browse-url)

;;; Variable:

(require 'http)

(if (or (featurep 'xemacs)
        (not (boundp 'emacs-major-version))
        (< emacs-major-version 21))
    (progn
      (require 'poe)
      (require 'poem)))
(require 'derived)

(defconst pukiwiki-mode-version
  (let ((revision "$Revision: 1.22 $"))
    (string-match "\\([0-9.]+\\)" revision)
    (match-string 1 revision)))

(defvar pukiwiki-site-list nil
  "List of Pukiwiki list.
Each element looks like (NAME URL STYLE). STYLE is optional.")

(defvar pukiwiki-list nil
  "`pukiwiki-list' is OBSOLETE; use pukiwiki-site-list.")

(defvar pukiwiki-auto-insert nil)

(defvar pukiwiki-auto-face
  (if (locate-library "facemenu")
      t
    nil))

(defvar pukiwiki-index-wiki-name 50)

(defvar pukiwiki-site-info nil)

(defvar pukiwiki-index-page-info-list nil)
(defvar pukiwiki-index-attach-list nil)

(defvar pukiwiki-index-sort-key nil)

(defvar pukiwiki-pagename nil)

(defvar pukiwiki-prev-buffer nil)

(defvar pukiwiki-pagetitle nil)

(defvar pukiwiki-md5hex nil)

(defvar pukiwiki-edit-newpage nil)

(defvar pukiwiki-password-alist nil)

(defvar pukiwiki-browser-function nil
  "Function to call browser.
If non-nil, `pukiwiki-edit-save-page' calls this function.
The function is expected to accept only one argument(URL).")

(defvar pukiwiki-init-file "~/.pukiwiki"
  "Init file for pukiwiki-mode.")

(defvar pukiwiki-non-wikiname-regexp-string "[^A-Za-z0-9]")
(defvar pukiwiki-wikiname-regexp-string
  "\\([A-Z][a-z0-9]+\\([A-Z][a-z0-9]+\\)+\\)")
(defvar pukiwiki-wikiname-regexp-list
  (list
   (cons (concat pukiwiki-non-wikiname-regexp-string
                 pukiwiki-wikiname-regexp-string
                 pukiwiki-non-wikiname-regexp-string) 1)
   (cons (concat "^"
                 pukiwiki-wikiname-regexp-string
                 pukiwiki-non-wikiname-regexp-string) 1)
   (cons (concat pukiwiki-non-wikiname-regexp-string
                 pukiwiki-wikiname-regexp-string "$") 1)
   (cons (concat "^" pukiwiki-wikiname-regexp-string "$") 1)))
(defvar pukiwiki-bracket-name-regexp '("\\[\\[\\([^]:|]+\\)\\]\\]" . 1))
(defvar pukiwiki-rd+-bracket-name-regexp '("((<\\([^>:|]+\\)>))" . 1))

(defvar pukiwiki-style-anchor-regexp-alist
  (list
   (cons 'default
         (cons pukiwiki-bracket-name-regexp
               pukiwiki-wikiname-regexp-list))
   (cons 'rd+ (list pukiwiki-rd+-bracket-name-regexp)))
  "Alist of regexp for anchor.")

(defvar pukiwiki-anchor-regexp-alist
  (cdr (assoc 'default pukiwiki-style-anchor-regexp-alist)))

(defvar pukiwiki-anchor-face
  (copy-face 'underline 'pukiwiki-anchor-face)
  "Face for Pukiwiki anchor." )

(defvar pukiwiki-keywords
  '("add" "aname" "article" "attach" "back"
    "backup" "bugtrack" "bugtrack_list"
    "calendar" "calendar_edit" "calendar_read"
    "calendar_viewer" "calendar2" "color"
    "contents" "comment" "counter" "deleted" "diff"
    "edit" "filelist" "freeze" "img" "include"
    "includesubmenu" "insert" "interwiki"
    "links" "list" "lookup" "ls" "ls2"
    "map" "md5" "memo" "navi" "new"
    "newpage" "norelated" "online"
    "paint" "pcomment" "popular" "random"
    "read" "recent" "ref" "rename" "rss"
    "rss10" "ruby" "search" "server"
    "showrss" "size" "source" "tb" "template"
    "touchgraph" "tracker" "unfreeze" "version"
    "versionlist" "vote" "yetlist"))

(defvar pukiwiki-font-lock-keywords
  (list
   '("^[ ]*_ \\([0-9]+-[0-9]+-[0-9]+[ ]*([^)]+)[ ]*[0-9]+:[0-9]+:[0-9]+ \\[[^\r\n]+$\\)" 1 font-lock-comment-face)
   '("\\(^//[^\n\r]+$\\)" 1 font-lock-comment-face)
   '("\\(^[>]+[^\n\r]+$\\)" 1 font-lock-reference-face)
   '("^\\([ ]*-+[^\n\r]+$\\)" 1 font-lock-keyword-face)
   (cons
    (concat "^#\\("
            (mapconcat 'identity pukiwiki-keywords "\\|")
            "\\)")
    (list 0 'font-lock-keyword-face))
   (cons
    (concat "^#\\("
            (mapconcat 'identity pukiwiki-keywords "\\|")
            "\\)"
            "(\\([^\n\r]+\\))")
    (list 2 'font-lock-doc-face))
   '("\\(^[*]+[^\n\r]+\\)" 1 font-lock-function-name-face)
   '("\\(^ [^\n\r]+$\\)" 1 font-lock-constant-face)
   ))

(defface pukiwiki-added-face
  '((((class color)
      (background dark))
     (:background "navy" :foreground "honeydew"))
    (((class color)
      (background light))
     (:background "alice blue" :foreground "black"))
    (t
     ())) nil)

(defface pukiwiki-removed-face
  '((((class color)
      (background dark))
     (:background "firebrick4" :foreground "snow3"))
    (((class color)
      (background light))
     (:background "misty rose" :foreground "black"))
    (t
     ())) nil)
(defvar pukiwiki-added-face 'pukiwiki-added-face)
(defvar pukiwiki-removed-face 'pukiwiki-removed-face)

(defvar pukiwiki-diff-font-lock-keywords
  (list
   '("\\(^+[^\n\r]*$\\)" 1 pukiwiki-added-face)
   '("\\(^-[^\n\r]*$\\)" 1 pukiwiki-removed-face)
   '("\\(^ //[^\n\r]+$\\)" 1 font-lock-comment-face)
   '("\\(^ [>]+[^\n\r]+$\\)" 1 font-lock-reference-face)
   '("\\(^  [^\n\r]+$\\)" 1 font-lock-constant-face)
   '("^ \\(-+[^\n\r]+$\\)" 1 font-lock-keyword-face)
   (cons
    (concat "^ #\\("
            (mapconcat 'identity pukiwiki-keywords "\\|")
            "\\)")
    (list 0 'font-lock-keyword-face))
   (cons
    (concat "^ #\\("
            (mapconcat 'identity pukiwiki-keywords "\\|")
            "\\)"
            "(\\([^\n\r]+\\))")
    (list 2 'font-lock-doc-face))
   '("\\(^ [*]+[^\n\r]+\\)" 1 font-lock-function-name-face)))

(defvar pukiwiki-site-name-history nil
  "History of Pukiwiki site name." )

(defvar pukiwiki-pagename-history nil
  "History of Pukiwiki page name." )

(defvar pukiwiki-diff-buffer-name "*Pukiwiki diff*")

(defvar pukiwiki-page-buffer-alist nil)

(defvar pukiwiki-init nil)

(defvar pukiwiki-search-word nil)

;;; $BHFMQ4X?t(B

(defun pukiwiki-mode-version ()
  (interactive)
  (message (format "pukiwiki-mode (Revision: %s)"
                   pukiwiki-mode-version)))

(defun pukiwiki-initialize ()
  (unless pukiwiki-init
    (pukiwiki-load-init-file)
    (setq pukiwiki-init t)))

;;; $BJT=8%b!<%I(B (pukiwiki-edit-*)
(defun pukiwiki-mode-set-variable ()
  (setq tab-width 4)
  (make-local-variable 'fill-prefix)
  (make-local-variable 'paragraph-start)
  (make-local-variable 'require-final-newline)
  (make-local-variable 'pukiwiki-site-info)
  (make-local-variable 'pukiwiki-newpage)
  (make-local-variable 'pukiwiki-pagename)
  (make-local-variable 'pukiwiki-prev-buffer)
  (make-local-variable 'pukiwiki-md5hex)
  (setq require-final-newline t
        indent-tabs-mode nil))

(defun pukiwiki-mode-set-font-lock (mode)
  (setq pukiwiki-anchor-regexp-alist
        (cdr (assoc (pukiwiki-site-style pukiwiki-site-info)
                    pukiwiki-style-anchor-regexp-alist)))
  (put mode 'font-lock-defaults
       '(text-font-lock-keywords nil t))
  (when (and (featurep 'font-lock)
             (fboundp 'font-lock-add-keywords))
    (let ((case-fold-search nil))
      (font-lock-add-keywords
       mode
       (mapcar (lambda (cell)
                 (list (car cell) (cdr cell)
                       'pukiwiki-anchor-face t))
               pukiwiki-anchor-regexp-alist)))
    (turn-on-font-lock)))

(define-derived-mode pukiwiki-edit-mode text-mode "Pukiwiki Edit"
  "Major mode for Pukiwiki editing.

\\{pukiwiki-edit-mode-map}"
  (pukiwiki-mode-set-variable)
  (pukiwiki-edit-setup-keys)
  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults
        '(pukiwiki-font-lock-keywords
          nil nil ((?_ . "w")) nil
          (font-lock-comment-start-regexp . "//")))
  (pukiwiki-mode-set-font-lock 'pukiwiki-edit-mode)
  (run-hooks 'pukiwiki-edit-mode-hook))

(defun pukiwiki-edit-setup-keys ()
  "Set up keymap for pukiwiki-edit-mode.
If you want to set up your own key bindings, use `pukiwiki-edit-mode-hook'."
  (define-key pukiwiki-edit-mode-map "\C-c\C-i" 'pukiwiki-edit-next-anchor)
  (define-key pukiwiki-edit-mode-map "\C-c\C-r" 'pukiwiki-edit-reload)
  (define-key pukiwiki-edit-mode-map "\C-c\C-e" 'pukiwiki-edit)
  (define-key pukiwiki-edit-mode-map "\C-c\C-c" 'pukiwiki-edit-save-page)
  (define-key pukiwiki-edit-mode-map "\C-c\C-p" 'pukiwiki-edit-preview)
  (define-key pukiwiki-edit-mode-map "\C-c\C-q" 'pukiwiki-edit-quit)
  (define-key pukiwiki-edit-mode-map "\C-m" 'pukiwiki-edit-new-line)
  (define-key pukiwiki-edit-mode-map "\C-xv=" 'pukiwiki-index-show-diff)
  )

(defun pukiwiki-load-init-file ()
  "Load init file."
  (when pukiwiki-init-file
    (let ((init-file (expand-file-name pukiwiki-init-file)))
      (when (file-readable-p init-file)
        (load init-file t t))
      (pukiwiki-obsolete-check))))

(defun pukiwiki-obsolete-check ()
  (when pukiwiki-list
    (message "pukiwiki-list is OBSOLETE. Use pukiwiki-site-list.")
    (sit-for 30)
    (setq pukiwiki-site-list pukiwiki-list)))

(defun pukiwiki-read-site-name (&optional string)
  "$B%5%$%HL>$r%_%K%P%C%U%!$+$iFI$_!"%5%$%H>pJs$N%j%9%H$rJV$9!#(B

STRING $B$,(B non-nil $B$J$i!"$=$l$r%5%$%HL>$H$9$k!#(B"
  (let* ((selected (car pukiwiki-site-list))
         (default (or (pukiwiki-site-name) (car selected))))
    (assoc
     (or
      (completing-read
       (format "Select SITE (%s): " default) pukiwiki-site-list
       nil t nil 'pukiwiki-site-name-history default) default)
     pukiwiki-site-list)))

(defun pukiwiki-password-read (sitename pagename)
  (cdr (assoc (cons sitename pagename) pukiwiki-password-alist)))

(defun pukiwiki-password-store (sitename pagename password)
  (let (key unit)
    (setq key (cons sitename pagename)
          unit (assoc key pukiwiki-password-alist))
    (if unit
        (if password
            (setcdr unit password)
          (setq pukiwiki-password-alist
                (delete unit pukiwiki-password-alist)))
      (if password
          (setq pukiwiki-password-alist
                (cons (cons (cons sitename pagename)
                            password) pukiwiki-password-alist))))))

(defun pukiwiki-read-pagename (arg site-name)
  (completing-read (format "Page name for %s: " site-name)
                   (cdr (assoc site-name pukiwiki-pagename-history))
                   nil nil arg nil arg))

(defun pukiwiki-read-char (prompt)
  "PROMPT (non-nil $B$N>l9g(B) $B$rI=<($7$F(B `read-char' $B$r8F$S=P$9!#(B"
  ;;; navi2ch-read-char $B$r;29M$K$7$F$^$9!#(B
  (let ((cursor-in-echo-area t)
        c)
    (if prompt
        (message "%s" prompt))
    (setq c (read-char))
    (if prompt
        (message "%s%c" prompt c))
    c))

(defun pukiwiki-read-char-with-retry (prompt retry-prompt list)
  ;;; navi2ch-read-char-with-retry $B$r;29M$K$7$F$^$9!#(B
  (let ((retry t) c)
    (while retry
      (setq c (pukiwiki-read-char prompt))
      (cond ((memq c list) (setq retry nil))
            ((eq c 12) (recenter))
            (t
             (ding)
             (setq prompt (or retry-prompt prompt)))))
    c))

(defun pukiwiki-http-request (mode cmd pagename site-url coding-system &optional post-data)
  (let (url buf)
    (cond
     ((eq mode 'post)
      (setq url (concat
                 (format "%s?cmd=%s&page=%s"
                         site-url "edit"
                         (http-url-hexify-string
                          pagename
                          coding-system)))))
     ((eq mode 'search)
      (setq url (concat
                 (format "%s?cmd=search"
                         site-url)))
      (setq mode 'post))
     ((eq mode 'raw)
      (setq mode 'get)
      (setq url site-url))
     (t
      (setq url (concat
                 (format "%s?cmd=%s" site-url cmd)
                 (if pagename
                     (format "&page=%s"
                             (http-url-hexify-string
                              pagename
                              coding-system)))
                 ""))))
    (setq buf (pukiwiki-http-fetch
               url mode nil nil
               (http-url-hexify-alist
                post-data coding-system)))
    (if (bufferp buf)
        (save-excursion
          (set-buffer buf)
          (decode-coding-region (point-min) (point-max)
                                coding-system)
          (goto-char (point-min))
          buf)
      (error (format "pukiwiki get: %s - %s"
                     (car buf) (cdr buf))))))

(defun pukiwiki-current-anchor-string ()
  "Return anchor string at current point."
  (let (str result pos (point (point)))
    (save-excursion
      (beginning-of-line)
      (setq pos (point))
      (while (and (setq result (pukiwiki-search-anchor pos))
                  (<= (cdr result) point))
        (setq pos (cdr result)))
      (when (and result (<= (car result) point))
        (setq str (buffer-substring-no-properties
                   (car result) (cdr result)))))
    str))

(defun pukiwiki-edit-next-anchor (&optional prev)
  "$B<!$N%"%s%+!<$X0\F0$9$k!#(B

PREV $B$,(B non-nil $B$J$i$P!"A0$N%"%s%+!<$X0\F0$9$k!#(B"
  (interactive "P")
  (goto-char (or (car (pukiwiki-search-anchor (point) prev))
                 (point))))

(defun pukiwiki-search-anchor (point &optional prev)
  "POINT $B$+$i:G$b6a$$%"%s%+!<$rC5$9!#(B

$B8+$D$+$C$?$i(B (beginning . end) $B$r!"8+$D$+$i$J$+$C$?$i(B nil $B$r(B $BJV$9(B"
  (let ((case-fold-search nil)
        (alist pukiwiki-anchor-regexp-alist)
        result)
    (save-excursion
      (while alist
        (goto-char point)
        (if (if prev
                (re-search-backward (car (car alist)) nil t nil)
              (re-search-forward (car (car alist)) nil t nil))
            (when (or (null result)
                      (> (car result)
                         (match-beginning (cdr (car alist)))))
              (setq result
                    (cons (match-beginning (cdr (car alist)))
                          (match-end (cdr (car alist)))))))
        (setq alist (cdr alist))))
    result))

(defun pukiwiki-edit-rename-buffer (sitename pagename pagetitle frozenp)
  (let ((name
         (format "[%s%s] %s%s"
                 sitename
                 (if (string= pagename pagetitle)
                     "" (concat ":" pagename))
                 pagetitle (if frozenp " (frozen)" ""))))
    (or (string= name (buffer-name))
        (rename-buffer name t))))

(defun pukiwiki-edit-url (str &optional url-encoded)
  "URL $B$r;XDj$7$FJT=8$9$k!#(B"
  (interactive "sURL: ")
  (let (url pagename site-info)
    (or (string-match "^\\(http://[^?]+\\)\\?\\(.+\\)$" str)
        (error "Illegal URL. (%s)" str))
    (setq url (match-string 1 str))
    (setq pagename (match-string 2 str))
    (when (string-match "=" pagename)
      (if (string-match "\\(^\\|[?&;]\\)p=\\(.+\\)" pagename)
          (setq pagename (match-string 2 pagename))
        (error "Illegal URL. (%s)" str)))
    (setq site-info (list url url))
    (pukiwiki-edit-page
     (http-url-unhexify-string pagename
                               (pukiwiki-site-coding-system site-info))
     site-info)))

(defun pukiwiki-edit-quit ()
  (interactive)
  (let ((site-info pukiwiki-site-info)
        (pagename pukiwiki-pagename)
        win cancelled)
    (setq buffer-read-only t)
    (when (buffer-modified-p)
      (if (y-or-n-p "Buffer is modified. Really quit?")
          (progn
            (kill-buffer (current-buffer))
            (setq pukiwiki-page-buffer-alist
                  (remassoc
                   (list (pukiwiki-site-name site-info) pagename)
                   pukiwiki-page-buffer-alist))
            (delete-other-windows))
        (setq cancelled t)))
    (when (not cancelled)
      (cond
       ((setq win (get-buffer-window
                   (pukiwiki-index-buffer-name site-info)))
        (select-window win))
       (t (delete-other-windows)))
      (pukiwiki-index site-info t pagename))))

(defun pukiwiki-edit-reload ()
  "$B8=:_JT=8Cf$N%Z!<%8$r%j%m!<%I$9$k!#(B"
  (interactive)
  (let ((selected-pagename pukiwiki-pagename))
    (pukiwiki-edit)))

(defun pukiwiki-edit (&optional select-site)
  "$B%Z!<%8L>$r;XDj$7$FJT=8$9$k!#(B

SELECT-SITE $B$,(B non-nil $B$N;~$O!"(BSITE$BL>$b;XDj$9$k!#(B"
  (interactive "P")
  (pukiwiki-initialize)
  (let ((point (point))
        (start (window-start))
        site-info pagename (same-site t) same-page)
    ;; site-name input (if required)
    (cond
     ((and (pukiwiki-site-name) (not select-site))
      (setq site-info pukiwiki-site-info))
     (t
      (setq site-info (pukiwiki-read-site-name))
      (when (not (string=
                  (pukiwiki-site-name site-info)
                  (pukiwiki-site-name)))
        (setq same-site nil))))
    ;; pagename input
    (setq pagename
          (if (boundp 'selected-pagename)
              selected-pagename
            (pukiwiki-read-pagename
             (or (pukiwiki-current-anchor-string)
                 pukiwiki-pagename "FrontPage")
             (pukiwiki-site-name site-info))))
    (if (string= pagename pukiwiki-pagename) (setq same-page t))
    ;; edit
    (pukiwiki-edit-page pagename site-info)
    ;; restore point (if required)
    (when (and same-site same-page)
      (set-window-start (selected-window) start)
      (goto-char point))))

(defvar pukiwiki-post-data nil)
(defun pukiwiki-edit-save-page (&optional toggle)
  (interactive "P")
  (let (buf contents post-data pagetitle password freeze keywords result
            (pbuf (current-buffer)))
    (message "Sending... ")

    ;; $BJ8;z%3!<%I$,JQ99$5$l$F$$$?;~$KLa$7$F$*$/(B
    (if (string= buffer-file-coding-system (pukiwiki-site-coding-system))
        ()
      (set-buffer-file-coding-system (pukiwiki-site-coding-system)))
    ;; $BH>3Q%+%?%+%J$OA43Q$KJQ49$7$FAw?.(B
    (pukiwiki-check-jisx0201)
    ;; $BJ8;z%3!<%I$N$*$+$7$$J8;z$,L5$$$+3NG'(B
    (pukiwiki-check-encode-able (point-min) (point-max))

    (sit-for 30)
    (setq password (pukiwiki-password-read
                    (car pukiwiki-site-info) pukiwiki-pagename)
          freeze (if toggle (not password) (if password t)))
    (if (or (eq t password) (and freeze (not password)))
        (setq password
              (read-passwd
               (format "Password for [%s] %s: "
                       (car pukiwiki-site-info) pukiwiki-pagename))))
    (setq pagetitle pukiwiki-pagename)
    (setq contents
          (buffer-substring-no-properties (point-min) (point-max)))
    (add-to-list 'post-data (cons "cmd" "edit"))
    (add-to-list 'post-data (cons "encode_hint" "$B$W(B"))
    (add-to-list 'post-data (cons "template_page" ""))
    (add-to-list 'post-data (cons "write" "$B%Z!<%8$N99?7(B"))
    (add-to-list 'post-data (cons "page" pagetitle))
    (add-to-list 'post-data (cons "digest" pukiwiki-md5hex))
    (add-to-list 'post-data (cons "password" password))
    (add-to-list 'post-data (cons "msg" contents))
    (setq buf
          (pukiwiki-http-request 'post nil pukiwiki-pagename
                                 (pukiwiki-site-url)
                                 (pukiwiki-site-coding-system)
                                 post-data))
    (setq pukiwiki-post-data post-data)
    (when (bufferp buf)
      (save-excursion
        (set-buffer buf)
        ;;(decode-coding-region (point-min) (point-max) pukiwiki-coding-system)
        (setq result
              (cond
               ((progn
                  (goto-char (point-min))
                  (re-search-forward
                   "textarea[^\n\r]*name=[^\n\r]*original[^\n\r]*" nil t))
                (let ((md5hex nil) (cbuf (current-buffer)))
                  (goto-char (point-min))
                  (re-search-forward
                   "name=\"digest\" value=\"\\([^ ]+\\)\" />" nil t nil)
                  (setq md5hex (match-string-no-properties 1))
                  (set-buffer pbuf)
                  (setq pukiwiki-md5hex md5hex)
                  (set-buffer cbuf))
                'conflict)
               (t 'success))))
      (cond
       ((equal result 'conflict)
        (pukiwiki-conflict-show-diff)
        (error "Conflict! (--- server, +++ yours)"))
       ((equal result 'wrong-pass)
        (error "Password is wrong!")))
      (message "Sending... done.")

      ;; $BI=<(2hLL$r@07A(B
      (pukiwiki-view-reformating)

      (set-buffer-modified-p nil)
      (setq buffer-read-only t)
      (pukiwiki-password-store
       (car pukiwiki-site-info) pukiwiki-pagename (if freeze password nil))
      (pukiwiki-edit-rename-buffer
       (car pukiwiki-site-info) pukiwiki-pagename
       (setq pukiwiki-pagetitle pagetitle) freeze)
      (and (functionp pukiwiki-browser-function)
           (funcall pukiwiki-browser-function
                    (format "%s?%s" (pukiwiki-site-url) pukiwiki-pagename)))
      (pukiwiki-edit-quit)
      (delete-other-windows))))

;;;; $B%W%l%S%e!<(B
(defvar pukiwiki-preview-file-list nil)
(defun pukiwiki-edit-preview (&optional toggle)
  (interactive "P")
  (let ((site-url (pukiwiki-site-url))
        (coding-system (pukiwiki-site-coding-system))
        (filename (concat
                   (expand-file-name
                    (concat (make-temp-name "pw") ".html")
                    temporary-file-directory)))
        ;;(getenv "TEMP")
        buf contents post-data pagetitle password freeze keywords result
        (pbuf (current-buffer)))
    (message "Sending... ")
    (sit-for 30)
    (setq pagetitle pukiwiki-pagename)
    (setq contents (buffer-substring-no-properties (point-min) (point-max)))
    (add-to-list 'post-data (cons "cmd" "edit"))
    (add-to-list 'post-data (cons "encode_hint" "$B$W(B"))
    (add-to-list 'post-data (cons "template_page" ""))
    (add-to-list 'post-data (cons "preview" "$B%W%l%S%e!<(B"))
    (add-to-list 'post-data (cons "page" pagetitle))
    (add-to-list 'post-data (cons "digest" pukiwiki-md5hex))
    (add-to-list 'post-data (cons "password" password))
    (add-to-list 'post-data (cons "msg" contents))
    (setq buf
          (pukiwiki-http-request 'post nil pukiwiki-pagename
                                 site-url
                                 coding-system
                                 post-data))
    (setq pukiwiki-post-data post-data)
    (message "Sending... done.")
    (when (bufferp buf)
      (save-excursion
        (set-buffer buf)
        ;;(decode-coding-region (point-min) (point-max) pukiwiki-coding-system)
        (goto-char (point-min))
        (re-search-forward "^<" nil t)
        (beginning-of-line)
        (delete-region (point-min) (point))
        (re-search-forward "<head>" nil t)
        (insert (concat
                 "<base href=\"" site-url "\">"))
        (goto-char (point-max))
        (re-search-backward "</html>" nil t)
        (end-of-line)
        (delete-region (point) (point-max))
        (set-buffer-file-coding-system coding-system)
        (write-region (point-min) (point-max) filename)
        (setq pukiwiki-preview-file-list
              (cons
               filename
               pukiwiki-preview-file-list))
        (if (functionp pukiwiki-browser-function)
            (funcall pukiwiki-browser-function
                     filename)
          (browse-url filename))
        ))))

(defun pukiwiki-conflict-show-diff ()
  "$B8=:_$N%P%C%U%!$H%5!<%P$N%G!<%?$rHf3S$7!"I=<($9$k!#(B"
  (let
      ((file1 (expand-file-name (make-temp-name "pukiwiki")
                                temporary-file-directory))
       (file2 (expand-file-name (make-temp-name "pukiwiki")
                                temporary-file-directory))
       (str (buffer-substring-no-properties (point-min) (point-max)))
       (pagename pukiwiki-pagename)
       (site-url (pukiwiki-site-url))
       (coding-system (pukiwiki-site-coding-system))
       diff-process diff-switches lines)
    (if (get-buffer pukiwiki-diff-buffer-name)
        (kill-buffer pukiwiki-diff-buffer-name))
    (with-temp-file file1
      (insert (cdr (assoc 'body
                          (pukiwiki-fetch-source pagename site-url coding-system))))
      (pukiwiki-replace-entity-refs)
      (if (> (current-column) 0) (insert "\n"))
      (setq lines (count-lines (point-min) (point-max))))
    (with-temp-file file2
      (insert str)
      (if (> (current-column) 0) (insert "\n")))
    (setq diff-process
          (start-process
           "diff" pukiwiki-diff-buffer-name
           diff-command "-U" (format "%d" lines) file1 file2))
    (set-process-sentinel diff-process (lambda (process event)))
    (save-excursion
      (set-buffer pukiwiki-diff-buffer-name)
      (setq buffer-read-only nil)
      (erase-buffer)
      (pukiwiki-diff-mode)
      (while
          (progn
            (accept-process-output diff-process 1)
            (not (equal (process-status (current-buffer)) 'exit))))
      (setq buffer-read-only nil)
      ;; delete header
      (goto-char (point-min))
      (forward-line 3)
      (delete-region (point-min) (point))
      (pop-to-buffer (process-buffer diff-process))
      (setq buffer-read-only t))
    (delete-file file1)
    (delete-file file2)))

;;; $BI=<(%b!<%I(B (pukiwiki-view)
(define-derived-mode pukiwiki-view-mode text-mode "Pukiwiki View"
  "Major mode for Pukiwiki editing.

\\{pukiwiki-edit-mode-map}"
  (pukiwiki-mode-set-variable)
  (pukiwiki-view-setup-keys)
  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults
        '(pukiwiki-font-lock-keywords
          nil nil ((?_ . "w")) nil
          (font-lock-comment-start-regexp . "//")))
  (pukiwiki-mode-set-font-lock 'pukiwiki-view-mode)
  (run-hooks 'pukiwiki-view-mode-hook))

(defun pukiwiki-view-setup-keys ()
  "Set up keymap for pukiwiki-edit-mode.
If you want to set up your own key bindings, use `pukiwiki-edit-mode-hook'."
  (local-set-key "q" 'pukiwiki-diff-exit)
  (local-set-key "b" 'scroll-down)
  (local-set-key " " 'scroll-up)
  (local-set-key "e" 'pukiwiki-view-edit-current-page)
  (local-set-key "\C-m" 'pukiwiki-edit-new-line)
  (local-set-key "\C-xv=" 'pukiwiki-index-show-diff)
  )

(defun pukiwiki-view-edit-current-page ()
  "$B8=:_9T$N%Z!<%8$rJT=8$9$k!#(B"
  (interactive)
  (if (and
       pukiwiki-pagename
       pukiwiki-site-info)
      (pukiwiki-edit-page pukiwiki-pagename pukiwiki-site-info)))


;;; $B0lMw%b!<%I(B(pukiwiki-index-*)
(make-variable-buffer-local 'pukiwiki-site-info)
(make-variable-buffer-local 'pukiwiki-index-page-info-list)
(make-variable-buffer-local 'pukiwiki-index-attach-list)
(make-variable-buffer-local 'pukiwiki-index-sort-key)
(define-derived-mode pukiwiki-index-mode text-mode "Pukiwiki Index"
  "Major mode for Pukiwiki index.

\\{pukiwiki-index-mode-map}"
  (make-local-variable 'pukiwiki-site-info)
  (make-local-variable 'pukiwiki-index-page-info-list)
  (make-local-variable 'pukiwiki-index-attach-list)
  (make-local-variable 'pukiwiki-index-sort-key)
  (pukiwiki-index-setup-keys)
  (run-hooks 'pukiwiki-index-mode-hook))

(defun pukiwiki-index-setup-keys ()
  "Set up keymap for pukiwiki-index-mode.
If you want to set up your own key bindings, use `pukiwiki-index-mode-hook'."
  (define-key pukiwiki-index-mode-map "\r" 'pukiwiki-index-edit-page-current-line)
  (define-key pukiwiki-index-mode-map "e" 'pukiwiki-index-edit-page)
  (define-key pukiwiki-index-mode-map "." 'pukiwiki-index-display-page)
  (define-key pukiwiki-index-mode-map " " 'pukiwiki-index-display-page-next)
  (define-key pukiwiki-index-mode-map "b" 'pukiwiki-index-display-page-prev)
  (define-key pukiwiki-index-mode-map "S" 'pukiwiki-index-sort)
  (define-key pukiwiki-index-mode-map "s" 'pukiwiki-index-sort)
  (define-key pukiwiki-index-mode-map "R" 'pukiwiki-index-refetch-index)
  (define-key pukiwiki-index-mode-map "q" 'pukiwiki-index-suspend)
  (define-key pukiwiki-index-mode-map "Q" 'pukiwiki-index-quit)
  (define-key pukiwiki-index-mode-map "n" 'pukiwiki-index-next-page)
  (define-key pukiwiki-index-mode-map "p" 'pukiwiki-index-prev-page)
  ;;(define-key pukiwiki-index-mode-map "j" 'pukiwiki-index-jump-chapter)
  (define-key pukiwiki-index-mode-map "j" 'pukiwiki-index-next-page)
  (define-key pukiwiki-index-mode-map "k" 'pukiwiki-index-prev-page)
  (define-key pukiwiki-index-mode-map "v" 'pukiwiki-index-view-page-by-browser)
  (define-key pukiwiki-index-mode-map "=" 'pukiwiki-index-show-diff)
  (define-key pukiwiki-index-mode-map ">" 'pukiwiki-index-end-of-buffer)
  (define-key pukiwiki-index-mode-map "<" 'pukiwiki-index-beginning-of-buffer)
  (define-key pukiwiki-index-mode-map "t=" 'pukiwiki-index-show-today-diff)
  (define-key pukiwiki-index-mode-map "/" 'pukiwiki-search)
  (define-key pukiwiki-index-mode-map "i" 'pukiwiki-index-isearch-article)
  (define-key pukiwiki-index-mode-map "ta" 'pukiwiki-index-show-today-changed))

(defun pukiwiki-index (&optional site-info refetch pagename)
  "$B0lMw%b!<%I$KF~$k!#(B

SITE-INFO $B$,;XDj$5$l$F$$$J$1$l$P!"%_%K%P%C%U%!$+$iFI$_9~$`!#(B
REFETCH $B$,(B nil $B$G$9$G$K%P%C%U%!$,B8:_$9$k$J$i!"(BHTTP GET $B$7$J$$!#(B"
  (interactive "P")
  (pukiwiki-initialize)
  (let (buf)
    ;; site-name input (if required)
    (when (null site-info)
      (setq site-info (pukiwiki-read-site-name)))
    (setq buf (pukiwiki-display-index site-info refetch pagename))
    (switch-to-buffer buf)
    (pukiwiki-index-sort nil ?d)
    (unless pagename (delete-other-windows))))

(defun pukiwiki-display-index (site-info &optional refetch pagename)
  "$B0lMw$rI=<($7!"%P%C%U%!$rJV$9!#(B

REFETCH $B$,(B nil $B$G4{$K%P%C%U%!$,B8:_$9$k$J$i!"(BHTTP GET $B$7$J$$!#(B
PAGENAME $B$KBP1~$7$?9T$,$"$l$P!"%+!<%=%k$r$=$3$K0\F0$9$k!#(B "
  (let ((old-buf (current-buffer))
        (buf (pukiwiki-index-get-buffer-create site-info)))
    (switch-to-buffer buf)
    (setq buffer-read-only nil)
    (erase-buffer)
    (when (or refetch (null pukiwiki-index-page-info-list))
      (setq pukiwiki-index-page-info-list nil)
      (message "Loading...")
      (setq pukiwiki-index-page-info-list
            (pukiwiki-fetch-index site-info))
      (setq pukiwiki-index-attach-list
            (pukiwiki-fetch-attach-index site-info))
      (message "Loading... done."))
    (mapcar (lambda (page-info)
              (insert (pukiwiki-index-page-info-string
                       page-info site-info)))
            pukiwiki-index-page-info-list)
    (set-buffer-modified-p nil)
    (pukiwiki-index-sort-by)
    (setq buffer-read-only t)
    (goto-char (point-min))
    (when pagename
      (dolist (elm pukiwiki-index-page-info-list)
        (when (string= (nth 1 elm) pagename)
          (re-search-forward (format "^%4d" (nth 0 elm)))
          (beginning-of-line)
          (recenter))))
    (switch-to-buffer old-buf)
    buf))

(defun pukiwiki-index-get-buffer-create (site-info)
  "$B0lMwI=<(MQ$N%P%C%U%!$rJV$9!#(B"
  (let ((buf-name (pukiwiki-index-buffer-name site-info)))
    (or (get-buffer buf-name)
        (progn
          (save-excursion
            (get-buffer-create buf-name)
            (set-buffer buf-name)
            (pukiwiki-index-mode)
            (setq pukiwiki-site-info site-info)
            (get-buffer buf-name))))))

(defun pukiwiki-index-page-info-string (page-info site-info)
  (let ((num (nth 0 page-info))
        (name (nth 1 page-info))
        (title (nth 2 page-info))
        (extra (nth 3 page-info)))
    (format "%4d %s %s %s\n" num
            (if (pukiwiki-page-buffer-name name site-info)
                "V" " ")
            (pukiwiki-prefix
             (concat title
                     (if (string= title name)
                         ""
                       (format " <%s>" name)))
             pukiwiki-index-wiki-name)
            extra)))

(defun pukiwiki-index-display-page (&optional refetch)
  "$B8=:_9T$N%Z!<%8$rI=<($9$k!#(B

REFETCH $B$,(B nil $B$G$9$G$K%P%C%U%!$,B8:_$9$k$J$i!"(BHTTP GET $B$7$J$$!#(B"
  (interactive)
  (let ((point (point))
        (search-word pukiwiki-search-word)
        (page-info (pukiwiki-index-page-info-current-line))
        (site-info pukiwiki-site-info)
        (info-list pukiwiki-index-page-info-list)
        (attach-list pukiwiki-index-attach-list)
        (cbuf (current-buffer))
        pagename)
    (when page-info
      (setq pagename (nth 1 page-info))
      (delete-other-windows)
      (split-window nil 10)
      (recenter t)
      (other-window 1)
      (pukiwiki-display-page pagename pukiwiki-site-info refetch)
      (pukiwiki-view-mode)
      (setq pukiwiki-prev-buffer nil)
      (setq pukiwiki-pagename pagename)
      (setq pukiwiki-site-info site-info)
      (setq pukiwiki-index-page-info-list info-list)
      (setq pukiwiki-index-attach-list attach-list)
      (condition-case err
          (if hi-lock-mode
              ()
            (hi-lock-mode 1))
        (error
         ()))
      (if (and
           search-word
           (functionp 'hi-lock-face-buffer))
          (hi-lock-face-buffer search-word 'region))

      ;; $BI=<(2hLL$r@07A(B
      (pukiwiki-view-reformating)

      (setq buffer-read-only t)
      (other-window 1))
    (pukiwiki-index pukiwiki-site-info nil pagename)
    (switch-to-buffer cbuf)
    (goto-char point)))

(defun pukiwiki-index-display-page-next (&optional refetch)
  "$B8=:_9T$N%Z!<%8$rI=<($9$k!#$9$G$KI=<($5$l$F$$$k;~$O%9%/%m!<%k$9$k!#(B

REFETCH $B$,(B nil $B$G$9$G$K%P%C%U%!$,B8:_$9$k$J$i!"(BHTTP GET $B$7$J$$!#(B"
  (interactive)
  (let ((page-info (pukiwiki-index-page-info-current-line))
        (old-win (selected-window))
        pagename buf win)
    (when page-info
      (setq pagename (nth 1 page-info))
      (setq buf (cdr (assoc
                      (list (pukiwiki-site-name pukiwiki-site-info)
                            pagename)
                      pukiwiki-page-buffer-alist)))
      (if (or (null buf) (null (setq win (get-buffer-window buf))))
          (pukiwiki-index-display-page refetch)
        (let ((other-window-scroll-buffer buf)
              (start (window-start win)))
          (scroll-other-window)
          ;; $B%9%/%m!<%k=PMh$J$$;~$O<!9T$K0\$k!#(B
          ;;   (when (= (window-start win) start)
          ;;     (forward-line)
          ;;     (pukiwiki-index-display-page refetch))
          )))))

(defun pukiwiki-index-display-page-prev (&optional refetch)
  "$B8=:_9T$N%Z!<%8$rI=<($9$k!#$9$G$KI=<($5$l$F$$$k;~$O%9%/%m!<%k$9$k!#(B

REFETCH $B$,(B nil $B$G$9$G$K%P%C%U%!$,B8:_$9$k$J$i!"(BHTTP GET $B$7$J$$!#(B"
  (interactive)
  (let ((page-info (pukiwiki-index-page-info-current-line))
        (old-win (selected-window))
        pagename buf win)
    (when page-info
      (setq pagename (nth 1 page-info))
      (setq buf (cdr (assoc (list (pukiwiki-site-name
                                   pukiwiki-site-info) pagename)
                            pukiwiki-page-buffer-alist)))
      (if (or (null buf) (null (setq win (get-buffer-window buf))))
          (pukiwiki-index-display-page refetch)
        (let ((other-window-scroll-buffer buf)
              (start (window-start win)))
          (scroll-other-window-down nil)
          ;; $B%9%/%m!<%k=PMh$J$$;~$O<!9T$K0\$k!#(B
          ;;   (when (= (window-start win) start)
          ;;     (forward-line)
          ;;     (pukiwiki-index-display-page refetch))
          )))))

(defun pukiwiki-index-edit-page-current-line ()
  "$B8=:_9T$N%Z!<%8$rJT=8$9$k!#(B"
  (interactive)
  (pukiwiki-index-edit-page
   (nth 1 (pukiwiki-index-page-info-current-line))))

(defun pukiwiki-index-edit-page (&optional pagename)
  "$B%Z!<%8$rJT=8$9$k!#(B"
  (interactive)
  (let ((index-buf (current-buffer))
        edit-buf start)
    (unless pagename
      (setq pagename
            (pukiwiki-read-pagename
             (nth 1 (pukiwiki-index-page-info-current-line))
             (pukiwiki-site-name))))
    (when (and pagename
               (setq edit-buf
                     (pukiwiki-edit-page pagename pukiwiki-site-info)))
      (switch-to-buffer index-buf)
      (delete-other-windows)
      (split-window nil 10)
      (setq start (window-start))
      (recenter)
      (pukiwiki-display-index pukiwiki-site-info nil pagename)
      (set-window-start (selected-window) start)
      (other-window 1)
      (switch-to-buffer edit-buf))))

(defun pukiwiki-index-page-info-current-line ()
  "$B8=:_9T$N%Z!<%8>pJs(B(list)$B$rJV$9!#(B"
  (let (num)
    (save-excursion
      (beginning-of-line)
      (re-search-forward "\\([0-9]+\\)" nil t nil)
      (setq num (match-string 1)))
    (cond
     (num (nth (1- (string-to-number num))
               pukiwiki-index-page-info-list))
     (t nil))))

(defun pukiwiki-index-page-info (pagename)
  "PAGENAME $B$N(B page-info $B$rJV$9!#CN$i$J$1$l$P(B nil$B!#(B"
  (let (result)
    (dolist (page-info pukiwiki-index-page-info-list)
      (when (string= (nth 1 page-info) pagename)
        (setq result page-info)))
    result))

(defun pukiwiki-index-refetch-index ()
  "$B0lMw$N:FFI$_9~$_$r9T$&!#(B"
  (interactive)
  (condition-case err
      (pukiwiki-index
       pukiwiki-site-info t
       (nth 1 (pukiwiki-index-page-info-current-line)))
    (error
     (pukiwiki-index pukiwiki-site-info t
                     ""))))

(defun pukiwiki-index-sort (&optional rev sortkey)
  "$B0lMw$N%=!<%H$r9T$&(B"
  (interactive "P")
  (message "Sorting...")
  (pukiwiki-index-sort-by
   (if sortkey
       sortkey
     (pukiwiki-read-char-with-retry
      "Sort by n)umber or d)ate? " nil '(?n ?d)))
   rev)
  (message "Sorting... done."))

(defun pukiwiki-index-suspend ()
  "pukiwiki-index $B$r0l;~CfCG$9$k!#(B"
  (interactive)
  (delete-other-windows)
  (dolist (elm pukiwiki-page-buffer-alist)
    (bury-buffer (cdr elm)))
  (replace-buffer-in-windows (current-buffer)))

(defun pukiwiki-index-quit ()
  "pukiwiki-index $B$r=*N;$9$k!#(B"
  (interactive)
  (let ((tmp pukiwiki-page-buffer-alist))
    (delete-other-windows)
    (dolist (elm pukiwiki-page-buffer-alist)
      (when (string= (nth 0 (car elm)) (pukiwiki-site-name))
        (kill-buffer (cdr elm))
        (setq tmp (remassoc (car elm) tmp))))
    (setq pukiwiki-page-buffer-alist tmp)
    (kill-buffer (current-buffer))))

(defun pukiwiki-index-next-page (arg)
  (interactive "P")
  (if arg
      (next-line arg)
    (next-line 1))
  (pukiwiki-index-display-page))

(defun pukiwiki-index-prev-page (arg)
  (interactive "P")
  (if arg
      (pukiwiki-index-next-page (* -1 arg))
    (pukiwiki-index-next-page -1)))

(defun pukiwiki-index-jump-chapter ()
  (interactive)
  (pukiwiki-index-display-page-next)
  (let ((page-info (pukiwiki-index-page-info-current-line))
        pagename buf chap)
    (when page-info
      (setq pagename (nth 1 page-info))
      (setq buf (cdr (assoc
                      (list (pukiwiki-site-name pukiwiki-site-info)
                            pagename)
                      pukiwiki-page-buffer-alist)))
      (set-buffer buf)
      (goto-char (point-min))
      (re-search-forward "^#contents" nil t)
      (setq chap
            (read-from-minibuffer
             "$B%8%c%s%W$9$k>OHV8e$rF~NO(B: "))
      (goto-char (point-min))
      (if (= 0 (length chap))
          (setq chap "1"))
      (if (re-search-forward
           (concat "^[*]+[ ]*" chap) nil t)
          ()
        (re-search-forward "^[*]+" nil t)))))

(defun pukiwiki-index-view-page-by-browser (arg)
  (interactive "P")
  (let ((pagename
         (nth 1 (pukiwiki-index-page-info-current-line))))
    (if (functionp pukiwiki-browser-function)
        (funcall pukiwiki-browser-function
                 (format "%s?%s" (pukiwiki-site-url)
                         (http-url-hexify-string pagename
                                                 (pukiwiki-site-coding-system))))
      (browse-url
       (format "%s?%s" (pukiwiki-site-url)
               (http-url-hexify-string pagename
                                       (pukiwiki-site-coding-system)))))))

(defun pukiwiki-index-show-diff (arg &optional day)
  "$B8=:_$N%P%C%U%!$H%5!<%P$N%G!<%?$rHf3S$7!"I=<($9$k!#(B"
  (interactive "P")
  (let* ((cbuf (current-buffer))
         (pagename
          (if (string= major-mode 'pukiwiki-index-mode)
              (nth 1 (pukiwiki-index-page-info-current-line))
            pukiwiki-pagename))
         (url nil)
         (age-last "1")
         (age "1")
         (site-info pukiwiki-site-info)
         (diff-index
          (format "%s?cmd=backup&page=%s"
                  (pukiwiki-site-url)
                  (http-url-hexify-string
                   pagename
                   (pukiwiki-site-coding-system site-info))))
         (site-url (pukiwiki-site-url))
         (diff-contents nil)
         (site-name (car pukiwiki-site-info))
         buf)
    (message "Checking backup...")
    (setq buf (pukiwiki-http-request
               'raw "edit" pagename diff-index
               (pukiwiki-site-coding-system site-info)))
    (when buf
      (set-buffer buf)
      (goto-char (point-max))
      (cond
       (day
        (let ((time
               (format-time-string "20%y-%02m-%02d"
                                   (pukiwiki-date-convert day))))
          (goto-char (point-min))
          (if (re-search-forward
               (concat
                "age=[^(\n\r>]+>\\([0-9]+\\)[ ]*("
                time) nil t)
              (setq age-last
                    (buffer-substring-no-properties
                     (match-beginning 1) (match-end 1)))
            (goto-char (point-max))
            (if (re-search-backward
                 "http://[^\n\r]+age=\\([0-9]+\\)\"" nil t)
                (setq age-last
                      (number-to-string
                       (+ (string-to-number
                           (buffer-substring-no-properties
                            (match-beginning 1) (match-end 1)))
                          1)))))
          (if (< (string-to-number age-last) 2)
              (setq age-last "1")
            (setq age-last
                  (number-to-string (- (string-to-number age-last) 1))))))
       (t
        (if (re-search-backward
             "http://[^\n\r]+age=\\([0-9]+\\)\"" nil t)
            (setq age-last
                  (buffer-substring-no-properties
                   (match-beginning 1) (match-end 1)))))))
    (if arg
        (setq age (read-from-minibuffer
                   (format "Diff to (Default %s) : " age-last)
                   nil nil nil nil age-last t))
      (setq age age-last))
    (if (= (string-to-number age) 0)
        (setq age age-last))
    (setq url
          (format "%s?cmd=backup&page=%s&age=%s&action=nowdiff"
                  site-url
                  (http-url-hexify-string pagename
                                          (pukiwiki-site-coding-system site-info))
                  age))
    (message "Creating diff...")
    (setq diff-contents (pukiwiki-fetch-source
                         pagename url
                         (pukiwiki-site-coding-system site-info) t))
    (setq buf (generate-new-buffer "*pukiwiki tmp*"))
    (switch-to-buffer buf)
    (pukiwiki-edit-rename-buffer
     site-name pagename (format "diff-to-%s" age) nil)
    (insert (cdr (assoc 'body diff-contents)))
    (replace-regexp
     "^ $" "" nil (point-min) (point-max))
    (replace-string
     "<span class=\"diff_added\">" "+" nil (point-min) (point-max))
    (replace-string
     "<span class=\"diff_removed\">" "-" nil (point-min) (point-max))
    (replace-string
     "</span>" "" nil (point-min) (point-max))
    (goto-char (point-min))
    (pukiwiki-replace-entity-refs)
    (setq pukiwiki-prev-buffer cbuf)
    (pukiwiki-diff-mode)
    (set-buffer-modified-p nil)
    ;;(view-mode)
    (delete-other-windows)
    (goto-char (point-min))
    (message "Creating diff... done!")
    ))

(defun pukiwiki-index-end-of-buffer ()
  (interactive)
  (pukiwiki-index-display-page)
  (end-of-buffer-other-window nil))

(defun pukiwiki-index-beginning-of-buffer ()
  (interactive)
  (pukiwiki-index-display-page)
  (beginning-of-buffer-other-window nil))

(defun pukiwiki-index-isearch-article (&optional regexp-p)
  "Do incremental search forward on the current article.
If REGEXP-P (the prefix) is non-nil, do regexp isearch."
  (interactive "P")
  (let ((page-info (pukiwiki-index-page-info-current-line))
        (old-win (selected-window))
        pagename buf win)
    (when page-info
      (setq pagename (nth 1 page-info))
      (setq buf (cdr (assoc
                      (list (pukiwiki-site-name pukiwiki-site-info)
                            pagename)
                      pukiwiki-page-buffer-alist)))
      (when (or (null buf) (null (setq win (get-buffer-window buf))))
        (pukiwiki-index-display-page nil)
        (setq buf (cdr (assoc
                        (list (pukiwiki-site-name pukiwiki-site-info)
                              pagename)
                        pukiwiki-page-buffer-alist))))
      (pukiwiki-eval-in-buffer-window
       buf
       (save-restriction
         (widen)
         (goto-char (point-min))
         (isearch-forward regexp-p))))))

(defmacro pukiwiki-eval-in-buffer-window (buffer &rest forms)
  (let ((tempvar (make-symbol "PWStartBufferWindow"))
        (w (make-symbol "w"))
        (buf (make-symbol "buf"))
        (frame (make-symbol "frame")))
    `(let* ((,tempvar (selected-window))
            (,buf ,buffer)
            (,w (get-buffer-window ,buf 'visible))
            ,frame)
       (unwind-protect
           (progn
             (if ,w
                 (progn
                   (select-window ,w)
                   (set-buffer (window-buffer ,w)))
               (pop-to-buffer ,buf))
             ,@forms)
         (setq ,frame (selected-frame))
         (select-window ,tempvar)
         (select-frame ,frame)))))

;;; $B8!:wMQ4X?t(B ($B$^$@L$40(B)
(define-derived-mode pukiwiki-search-mode text-mode "Pukiwiki Search"
  "Major mode for Pukiwiki index.

\\{pukiwiki-index-mode-map}"
  (make-local-variable 'pukiwiki-site-info)
  (make-local-variable 'pukiwiki-index-page-info-list)
  (make-local-variable 'pukiwiki-index-attach-list)
  (make-local-variable 'pukiwiki-index-sort-key)
  (pukiwiki-search-setup-keys)
  (run-hooks 'pukiwiki-search-mode-hook))

(defun pukiwiki-search-setup-keys ()
  "Set up keymap for pukiwiki-index-mode.
If you want to set up your own key bindings, use `pukiwiki-search-mode-hook'."
  (define-key pukiwiki-search-mode-map "\r" 'pukiwiki-index-edit-page-current-line)
  (define-key pukiwiki-search-mode-map "e" 'pukiwiki-index-edit-page)
  (define-key pukiwiki-search-mode-map "." 'pukiwiki-index-display-page)
  (define-key pukiwiki-search-mode-map " " 'pukiwiki-index-display-page-next)
  (define-key pukiwiki-search-mode-map "b" 'pukiwiki-index-display-page-prev)
  (define-key pukiwiki-search-mode-map "S" 'pukiwiki-index-sort)
  (define-key pukiwiki-search-mode-map "s" 'pukiwiki-index-sort)
  (define-key pukiwiki-search-mode-map "R" 'pukiwiki-index-refetch-index)
  (define-key pukiwiki-search-mode-map "q" 'pukiwiki-search-quit)
  (define-key pukiwiki-search-mode-map "Q" 'pukiwiki-search-quit)
  (define-key pukiwiki-search-mode-map "n" 'pukiwiki-index-next-page)
  (define-key pukiwiki-search-mode-map "p" 'pukiwiki-index-prev-page)
  ;;(define-key pukiwiki-search-mode-map "j" 'pukiwiki-index-jump-chapter)
  (define-key pukiwiki-search-mode-map "j" 'pukiwiki-index-next-page)
  (define-key pukiwiki-search-mode-map "k" 'pukiwiki-index-prev-page)
  (define-key pukiwiki-search-mode-map "v" 'pukiwiki-index-view-page-by-browser)
  (define-key pukiwiki-search-mode-map "=" 'pukiwiki-index-show-diff)
  (define-key pukiwiki-search-mode-map ">" 'pukiwiki-index-end-of-buffer)
  (define-key pukiwiki-search-mode-map "<" 'pukiwiki-index-beginning-of-buffer)
  (define-key pukiwiki-search-mode-map "t=" 'pukiwiki-index-show-today-diff)
  (define-key pukiwiki-search-mode-map "/" 'pukiwiki-search)
  (define-key pukiwiki-search-mode-map "ta" 'pukiwiki-index-show-today-changed))

(defun pukiwiki-search-quit ()
  (interactive)
  (delete-other-windows)
  (kill-buffer (current-buffer))
  (switch-to-buffer (pukiwiki-index-buffer-name pukiwiki-site-info)))

(defun pukiwiki-fetch-search (&optional site-info)
  (interactive "P")
  (let (buf contents post-data pagetitle password freeze keywords result
            (pbuf (current-buffer)) (indexes nil)
            (search-word nil) (i 1) (lst nil))
    (message "Sending... ")
    (setq search-word
          (read-from-minibuffer
           "$B8!:w8l$rF~NO(B: "))
    (setq pagetitle pukiwiki-pagename)

    (add-to-list 'post-data (cons "encode_hint" "$B$W(B"))
    (add-to-list 'post-data (cons "word" search-word))
    (add-to-list 'post-data (cons "type" "AND"))
    (setq buf
          (pukiwiki-http-request 'search nil ""
                                 (pukiwiki-site-url site-info)
                                 (pukiwiki-site-coding-system site-info)
                                 post-data))

    (setq lst (split-string search-word " "))
    (setq search-word (concat
                       "\\(" (car lst)))
    (setq lst (cdr lst))
    (while lst
      (setq search-word
            (concat search-word
                    "\\|"
                    (car lst)))
      (setq lst (cdr lst)))
    (setq search-word (concat
                       search-word "\\)"))

    (setq pukiwiki-post-data post-data)
    (when (bufferp buf)
      (save-excursion
        (set-buffer buf)
        (goto-char (point-min))
        (while (re-search-forward
                "<li><a[^?]*\\?cmd=read&amp;page=\\([^\"]*\\)&amp;word=[^\"]*\"[^>]*>\\([^\n\r]*\\)</a>[ ]*\\([^<\n\r]*\\)" nil t)
          (setq indexes
                (cons
                 (list i (http-url-unhexify-string
                          (match-string 1)
                          (pukiwiki-site-coding-system site-info))
                       ;;(pukiwiki-replace-entity-refs (match-string 2))
                       (http-url-unhexify-string
                        (match-string 1)
                        (pukiwiki-site-coding-system site-info))
                       (pukiwiki-replace-entity-refs
                        (format-time-string
                         "%y/%m/%d"
                         (pukiwiki-index-date (match-string 3)))))
                 indexes))
          (setq i (1+ i)))))
    (cons search-word (reverse indexes))))

(defun pukiwiki-search-buffer-name (&optional site-info)
  (format "Pukiwiki search <%s>" (pukiwiki-site-name site-info)))

(defun pukiwiki-search-get-buffer-create (site-info)
  "$B0lMwI=<(MQ$N%P%C%U%!$rJV$9!#(B"
  (let ((buf-name (pukiwiki-search-buffer-name site-info)))
    (or (get-buffer buf-name)
        (progn
          (save-excursion
            (get-buffer-create buf-name)
            (set-buffer buf-name)
            (pukiwiki-search-mode)
            (setq pukiwiki-site-info site-info)
            (get-buffer buf-name))))))

(defun pukiwiki-search-index (site-info)
  "$B0lMw$rI=<($7!"%P%C%U%!$rJV$9!#(B

REFETCH $B$,(B nil $B$G4{$K%P%C%U%!$,B8:_$9$k$J$i!"(BHTTP GET $B$7$J$$!#(B
PAGENAME $B$KBP1~$7$?9T$,$"$l$P!"%+!<%=%k$r$=$3$K0\F0$9$k!#(B "
  (let ((old-buf (current-buffer))
        (buf (pukiwiki-search-get-buffer-create site-info))
        (site-info pukiwiki-site-info)
        (info-list pukiwiki-index-page-info-list)
        (attach-list pukiwiki-index-attach-list)
        lst)
    (switch-to-buffer buf)
    (setq buffer-read-only nil)
    (erase-buffer)
    (message "Loading...")
    (make-local-variable 'pukiwiki-search-word)
    (condition-case err
        (setq lst (pukiwiki-fetch-search site-info))
      (error
       (setq lst 'error)))
    (when (or
           (and (not (listp lst))
                (string= lst 'error)))
      (pukiwiki-search-quit)
      (error "$B%?%$%`%"%&%H$7$^$7$?!%(B"))
    (setq pukiwiki-search-word (car lst))
    (setq lst (cdr lst))
    (when (not lst)
      (pukiwiki-search-quit)
      (error "$B8+$D$+$j$^$;$s$G$7$?!%(B"))
    (setq pukiwiki-site-info site-info)
    (setq pukiwiki-index-page-info-list lst)
    (setq pukiwiki-index-attach-list attach-list)
    (make-local-variable 'pukiwiki-index-sort-key)
    (message "Loading... done.")
    (mapcar (lambda (page-info)
              (insert (pukiwiki-index-page-info-string
                       page-info site-info)))
            lst)
    (set-buffer-modified-p nil)
    (pukiwiki-index-sort-by)
    (setq buffer-read-only t)
    (goto-char (point-min))
    (switch-to-buffer old-buf)
    buf))

(defun pukiwiki-search ()
  "$B0lMw%b!<%I$KF~$k!#(B

SITE-INFO $B$,;XDj$5$l$F$$$J$1$l$P!"%_%K%P%C%U%!$+$iFI$_9~$`!#(B
REFETCH $B$,(B nil $B$G$9$G$K%P%C%U%!$,B8:_$9$k$J$i!"(BHTTP GET $B$7$J$$!#(B"
  (interactive)
  (pukiwiki-initialize)
  (let (buf (site-info pukiwiki-site-info))
    ;; site-name input (if required)
    (when (null site-info)
      (setq site-info (pukiwiki-read-site-name)))
    (setq buf (pukiwiki-search-index site-info))
    (switch-to-buffer buf)
    (pukiwiki-index-sort nil ?d)))

;;; $BJQ992U=j$rI=<($5$;$k4X?t(B
(defun pukiwiki-index-show-today-diff (&optional arg)
  "$B:#F|JQ99$5$l$?2U=j$r8+$k!%%j!<%8%g%s$,$"$l$P$=$NHO0O$N(B
$BJ8=q$NJQ992U=j$r0lMwI=<($9$k!%(B"
  (interactive "P")
  (let ((wikilist nil)
        (cbuf (current-buffer))
        (site-info pukiwiki-site-info)
        (today-changed-text nil) buf pt end-pt
        (site-name (car pukiwiki-site-info)))
    (if (listp arg)
        (setq arg (car arg)))
    (if (not arg)
        (setq arg 0))
    (cond
     ((or (and (boundp 'mark-active) mark-active)
          (and (fboundp 'region-exists-p) (region-exists-p)))
      (setq pt (region-beginning))
      (setq end-pt (region-end))
      (save-excursion
        (goto-char pt)
        (while (< (point) end-pt)
          (setq wikilist
                (cons (pukiwiki-index-page-info-current-line)
                      wikilist))
          (forward-line 1))
        (setq wikilist (reverse wikilist))
        (setq today-changed-text
              (pukiwiki-index-make-changed-text wikilist arg site-info))
        (setq buf (generate-new-buffer "*pukiwiki tmp*"))
        (switch-to-buffer buf)
        (pukiwiki-edit-rename-buffer
         site-name "" "region-changed" nil)
        (insert today-changed-text)
        (goto-char (point-min))
        (pukiwiki-replace-entity-refs)
        (setq pukiwiki-prev-buffer cbuf)
        (pukiwiki-diff-mode)
        (set-buffer-modified-p nil)
        (delete-other-windows)
        (goto-char (point-min))
        (message "Creating ... done!")))
     (t
      (pukiwiki-index-show-diff nil arg)))))

(defun pukiwiki-index-make-changed-text (wikilist day site-info)
  (with-temp-buffer
    (while wikilist
      (let* ((pukiwiki-site-info site-info)
             (contents nil)
             (pukiwiki-pagename (nth 1 (car wikilist)))
             (stringwidth (string-width (nth 1 (car wikilist))))
             (str
              (if (> (/ (- fill-column (+ 2 stringwidth)) 2) 0)
                  (make-string
                   (floor (/ (- fill-column (+ 2 stringwidth)) 2))
                   ?*)
                "")))
        (insert
         (concat
          str " " (nth 1 (car wikilist)) " " str
          "\n"))
        (save-current-buffer
          (pukiwiki-index-show-diff nil day)
          (goto-char (point-min))
          (while (re-search-forward "^[+-]" nil t)
            (setq contents
                  (cons
                   (buffer-substring-no-properties
                    (line-beginning-position)
                    (line-end-position))
                   contents)))
          (kill-buffer (current-buffer)))
        (setq contents (reverse contents))
        (while contents
          (insert
           (concat
            (car contents)
            "\n"))
          (setq contents (cdr contents)))
        (insert "\n")
        (setq wikilist (cdr wikilist))))
    (buffer-substring-no-properties
     (point-min) (point-max))))

(defun pukiwiki-index-show-today-changed (&optional arg)
  "$B:#F|JQ99$5$l$?2U=j$r0lMwI=<((B"
  (interactive "P")
  (let ((wikilist nil)
        (cbuf (current-buffer))
        (site-info pukiwiki-site-info)
        (today-changed-text nil) buf
        days
        (site-name (car pukiwiki-site-info)))
    (if (listp arg)
        (setq arg (car arg)))
    (if (not arg)
        (setq arg 0))
    (save-excursion
      ;;       (while (re-search-forward
      ;;               (format-time-string
      ;;                "%y/%m/%d" (pukiwiki-date-convert arg)) nil t)
      (save-excursion
        (beginning-of-line)
        (if (re-search-forward "[0-9]+/[0-9]+/[0-9]+" nil t)
            (setq days
                  (buffer-substring-no-properties
                   (match-beginning 0) (match-end 0)))
          (setq days
                (format-time-string
                 "%y/%m/%d" (pukiwiki-date-convert arg)) nil t)))
      (goto-char (point-min))
      (while (re-search-forward days nil t)
        (setq wikilist
              (cons (pukiwiki-index-page-info-current-line)
                    wikilist))))
    (setq wikilist (reverse wikilist))
    (setq today-changed-text
          (pukiwiki-index-make-changed-text wikilist arg site-info))
    (setq buf (generate-new-buffer "*pukiwiki tmp*"))
    (switch-to-buffer buf)
    (pukiwiki-edit-rename-buffer
     site-name "" "today-changed" nil)
    (insert today-changed-text)
    (goto-char (point-min))
    (pukiwiki-replace-entity-refs)
    (setq pukiwiki-prev-buffer cbuf)
    (pukiwiki-diff-mode)
    (set-buffer-modified-p nil)
    (delete-other-windows)
    (goto-char (point-min))
    (message "Creating ... done!")
    ))

;;; pukiwiki-diff-mode
(define-derived-mode pukiwiki-diff-mode text-mode "Pukiwiki Diff"
  "Major mode for Pukiwiki editing.

\\{pukiwiki-edit-mode-map}"
  (pukiwiki-mode-set-variable)
  (pukiwiki-edit-setup-keys)
  (view-mode)
  (local-set-key " " 'scroll-up)
  (local-set-key "q" 'pukiwiki-diff-exit)
  (local-set-key "\M-n" 'pukiwiki-diff-next-change)
  (local-set-key "\M-p" 'pukiwiki-diff-prev-change)
  (local-set-key "b" 'scroll-down)
  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults
        '(pukiwiki-diff-font-lock-keywords
          nil nil ((?_ . "w")) nil
          (font-lock-comment-start-regexp . "//")))
  (pukiwiki-mode-set-font-lock 'pukiwiki-diff-mode)
  (run-hooks 'pukiwiki-edit-mode-hook))

(defun pukiwiki-diff-exit ()
  (interactive)
  (let ((site-info pukiwiki-site-info)
        (prev-buffer pukiwiki-prev-buffer))
    (kill-buffer (current-buffer))
    (delete-other-windows)
    (if (and prev-buffer
             (buffer-live-p prev-buffer))
        (switch-to-buffer prev-buffer)
      (if (and
           site-info
           (pukiwiki-index-buffer-name site-info)
           (buffer-live-p
            (get-buffer
             (pukiwiki-index-buffer-name site-info))))
          (switch-to-buffer
           (get-buffer
            (pukiwiki-index-buffer-name site-info)))))))

(defun pukiwiki-diff-next-change ()
  (interactive)
  (end-of-line)
  (if (re-search-forward
       "^[+-]" nil t)
      (forward-char -1)
    (goto-char (point-max))))

(defun pukiwiki-diff-prev-change ()
  (interactive)
  (beginning-of-line)
  (if (re-search-backward
       "^[+-]" nil t)
      ()
    (goto-char (point-min))))


;;; func

(defun pukiwiki-display-page (pagename site-info &optional refetch)
  "$B%Z!<%8$rI=<($9$k(B

REFETCH $B$,(B nil $B$G$9$G$K%P%C%U%!$,B8:_$9$k$J$i!"(BHTTP GET $B$7$J$$!#(B"
  (let ((not-cancelled t)
        result body keyword pagetitle
        password history point buf new-page)
    (setq buf (cdr (assoc
                    (list (pukiwiki-site-name site-info) pagename)
                    pukiwiki-page-buffer-alist)))
    (if (and buf (buffer-name buf) (not refetch))
        (progn
          (switch-to-buffer buf)
          (goto-char (point-min)))
      (and buf (kill-buffer buf))
      (message "Loading...")
      (condition-case err
          (setq result (pukiwiki-fetch-source
                        pagename (pukiwiki-site-url site-info)
                        (pukiwiki-site-coding-system site-info)))
        (error
         (setq result (pukiwiki-fetch-source
                       pagename
                       (format "%s?cmd=%s&page=%s"
                               (pukiwiki-site-url site-info)
                               "diff"
                               (http-url-hexify-string
                                pagename
                                (pukiwiki-site-coding-system site-info)))
                       (pukiwiki-site-coding-system site-info)
                       'diff))
         (setq result (delete (list 'password) result))
         (setq result
               (cons
                (cons 'password t)
                result))))
      (setq body (cdr (assoc 'body result)))
      (setq pagetitle (cdr (assoc 'pagetitle result)))
      (setq password (cdr (assq 'password result)))
      (when
          (and body
               (or (> (length body) 0)
                   (progn
                     (setq not-cancelled
                           (y-or-n-p
                            (format
                             "Page %s is not exist. Create new page?"
                             pagename)))
                     (if not-cancelled
                         (setq new-page t)
                       (setq result nil))
                     not-cancelled)))
        (setq buf (generate-new-buffer "*pukiwiki tmp*"))
        (switch-to-buffer buf)
        (pukiwiki-edit-rename-buffer
         (pukiwiki-site-name site-info) pagename pagetitle password)
        (save-excursion
          (setq point (point))
          (insert body)
          (goto-char (point-min))
          (pukiwiki-replace-entity-refs)
          (pukiwiki-edit-mode)
          (setq pukiwiki-edit-newpage new-page)
          (set-buffer-modified-p nil)
          (add-to-list
           'pukiwiki-page-buffer-alist
           (cons (list (pukiwiki-site-name site-info) pagename)
                 (current-buffer)))
          (message "Loading... done."))
        (goto-char point))
      result)))

(defun pukiwiki-edit-page (pagename site-info)
  "PAGENAME $B$NJT=8%b!<%I$KF~$k!#%P%C%U%!$rJV$9!#(B"
  (let ((result (pukiwiki-display-page pagename site-info t))
        password history)
    (when result
      (if (setq password (cdr (assq 'password result)))
          (or (pukiwiki-password-read (car site-info) pagename)
              (pukiwiki-password-store  (car site-info) pagename t))
        (pukiwiki-password-store (car site-info) pagename nil))
      (when (cdr (assq 'password result))
        (kill-buffer (current-buffer))
        (error "$B$3$N%Z!<%8$OJT=8$G$-$^$;$s(B"))
      ;;       (if (setq password (cdr (assq 'password result)))
      ;;           (or (pukiwiki-password-read (car site-info) pagename)
      ;;               (pukiwiki-password-store  (car site-info) pagename t))
      ;;         (pukiwiki-password-store (car site-info) pagename nil))
      (setq pukiwiki-md5hex (cdr (assq 'md5hex result)))
      (setq pukiwiki-pagename pagename)
      (setq pukiwiki-pagetitle (or (cdr (assq 'pagetitle result)) pagename))
      (setq pukiwiki-site-info site-info)
      (or (setq history (assoc (pukiwiki-site-name) pukiwiki-pagename-history))
          (setq pukiwiki-pagename-history
                (cons (setq history (cons (pukiwiki-site-name) nil))
                      pukiwiki-pagename-history)))
      (or (member pukiwiki-pagename (cdr history))
          (setcdr history (cons (cons pukiwiki-pagename nil) (cdr history))))
      (set-buffer-modified-p nil)
      (current-buffer))))

(defun pukiwiki-fetch-index (site-info)
  "$B%Z!<%80lMw$r<hF@$9$k!#(B"
  (let (indexes history (i 1)
                (buf (pukiwiki-http-request
                      'get "list" nil (pukiwiki-site-url site-info)
                      (pukiwiki-site-coding-system site-info))))
    (when (bufferp buf)
      (save-excursion
        (set-buffer buf)
        (re-search-forward "<ul>" nil t nil)
        (while (re-search-forward
                ;;"<li><a[^?]*\\?cmd=read&amp;page=\\([^\"]*\\)\"[^>]*>\\([^<]*\\)</a><small>\\([^<]*\\)</small></li>"
                "<li><a[^?]*\\?cmd=read&amp;page=\\([^\"]*\\)\"[^>]*>\\([^<]*\\)</a>[<small>]*\\([^<]*\\)[</small>]*</li>"
                nil t nil)
          (setq indexes
                (cons
                 (list i (http-url-unhexify-string
                          (match-string 1) (pukiwiki-site-coding-system site-info))

                       (pukiwiki-replace-entity-refs (match-string 2))
                       (pukiwiki-replace-entity-refs
                        (format-time-string
                         "%y/%m/%d"
                         (pukiwiki-index-date (match-string 3)))))
                 indexes))
          (setq i (1+ i))))
      (or (setq history
                (assoc
                 (pukiwiki-site-name site-info) pukiwiki-pagename-history))
          (setq pukiwiki-pagename-history
                (cons (setq history (cons (pukiwiki-site-name site-info) nil))
                      pukiwiki-pagename-history)))
      (setcdr history (mapcar (lambda (elm) (cons (nth 1 elm) nil)) indexes))
      (reverse indexes))))

(defun pukiwiki-fetch-attach-index (site-info)
  "$BE:IU%U%!%$%k$N0lMw$rI=<($9$k!#(B"
  (let (indexes
        history (i 1)
        (list-regexp
         "^<li><a[^?]*\\?\\([- .%0-9a-zA-Z]*\\)\"[^>]*>\\([^<]*\\)</a>")
        beg end pagename
        attach-list
        (buf (pukiwiki-http-request
              'raw "list" nil
              (concat
               (pukiwiki-site-url site-info)
               "?plugin=attach&pcmd=list")
              (pukiwiki-site-coding-system site-info))))
    (when (bufferp buf)
      (save-excursion
        (set-buffer buf)
        (re-search-forward "<ul>" nil t nil)
        (while (re-search-forward
                list-regexp nil t)
          (setq pagename (match-string 2))
          (setq beg (point))
          (if (re-search-forward
               list-regexp nil t nil)
              (setq end (point))
            (setq end (point-max)))
          (goto-char beg)
          (setq attach-list nil)
          (while (re-search-forward
                  "<li><a[^?]*\\?plugin=attach\\([^\"]*\\)\"[^>]*>\\([^()<]*\\)</a>"
                  end t)
            (setq attach-list
                  (cons
                   (match-string 2)
                   attach-list)))
          (setq indexes
                (cons
                 (cons
                  pagename
                  attach-list)
                 indexes)))))
    indexes))

(defun pukiwiki-fetch-source (pagename site-url coding-system &optional raw)
  "Pukiwiki $B$N(B $B%=!<%9$r<hF@$9$k!#(B

'((md5hex . \"...\")
  (body . \"...\")
  (pagetitle . (...))
  (password . t/nil)) $B$rJV$9!#(B"
  (let (buf start end pt result
            (win (current-window-configuration)))
    (if raw
        (setq buf (pukiwiki-http-request 'raw "edit" pagename site-url coding-system))
      (setq buf (pukiwiki-http-request 'get "edit" pagename site-url coding-system)))
    (save-current-buffer
      (when (bufferp buf)
        (set-buffer buf)
        (save-excursion
          (goto-char (point-min))
          (if (or
               (re-search-forward "<pre>" nil t nil)
               (re-search-forward
                "<textarea [^>]*name=\"msg\"[^>]+>" nil t nil))
              ()
            (set-window-configuration win)
            (error "PukiWiki $B$N%F%-%9%H$,8+$D$+$j$^$;$s(B")))
        ;; md5hex
        (goto-char (point-min))
        (re-search-forward
         "name=\"digest\" value=\"\\([^ ]+\\)\" />" nil t nil)
        (setq result
              (cons (cons
                     'md5hex (match-string-no-properties 1)) result))
        (setq pt (point))
        ;; textarea
        (when (string= raw 'diff)
          (goto-char (point-min))
          (re-search-forward "<pre>" nil t nil)
          (while (re-search-forward "^ " nil t)
            (delete-region
             (match-beginning 0)
             (match-end 0)))
          (goto-char (point-min))
          (re-search-forward "<pre>" nil t nil)
          (while (re-search-forward ">\\( \\)" nil t)
            (delete-region
             (match-beginning 1)
             (match-end 1))))
        (goto-char (point-min))
        (cond
         (raw
          (re-search-forward "<pre>" nil t nil)
          (setq start (match-end 0))
          (re-search-forward "</pre>" nil t nil)
          (setq end (match-beginning 0))
          (setq result
                (cons (cons
                       'body
                       (buffer-substring-no-properties start end)) result)))
         ((re-search-forward
           "<textarea [^>]*name=\"msg\"[^>]+>" nil t nil)
          (setq start (match-end 0))
          (re-search-forward
           "</textarea>" nil t nil)
          (setq end (match-beginning 0))
          (setq result
                (cons (cons
                       'body
                       (buffer-substring-no-properties start end)) result))))
        ;; page_title
        (goto-char (point-min))
        (re-search-forward
         "<h1[^>]+class[^>]+\"title\"><a[^>]+href[^>]+>\\([^<>\"]*\\)</a>" nil t nil)
        (setq result
              (cons (cons 'pagetitle
                          (pukiwiki-replace-entity-refs
                           (match-string-no-properties 1))) result))
        (setq result (cons (cons 'password nil) result))))
    result))

(defun pukiwiki-check-encode-able (beg end)
  (interactive "r")
  (save-excursion
    (let* (
           (mycodingsystem buffer-file-coding-system)
           mychar
           mycharname mycharsetname
           (mycount 0)
           ;;;encoding $B$KBP1~$9$k(B charset $B$N%j%9%H$r<hF@$9$k!#(B
           ;;;Meadow2 (Emacs21) $B$G$bF0$/$+$I$&$+L$3NG'(B
           ;;;$B$&$^$/$$$+$J$1$l$P!"<+J,$GBP1~$rDj5A$9$l$PNI$$(B
           (mycharsetlist (coding-system-get mycodingsystem 'safe-charsets))
           )
      (goto-char beg) ;;;$B%j!<%8%g%s$N@hF,$K0\F0(B
      (while (< (point) end) ;;;$B%j!<%8%g%sFb$r=g$KD4$Y$k(B
        (setq mychar (following-char))
        (setq mycharsetname (char-charset mychar))
        ;;$B9g@.J8;z$KBP$9$k=hM}!#(B Meadow2 (Emacs21) $B$G$OITMW$+$b(B????
        (if (equal 'composition mycharsetname)
            (setq mycharsetname
                  (char-charset (string-to-char
                                 (decompose-string (char-to-string mychar))))))
        ;;encode $B$G$-$J$$J8;z$@$C$?$i?'$r$D$1$k(B
        (if (or (equal mycharsetlist t) (member mycharsetname mycharsetlist))
            nil ;;;encode $B$G$-$k;~$O2?$b$7$J$$!#(B encode $B$G$-$J$$;~"-(B
          (overlay-put (make-overlay (point) (1+ (point))) 'face 'region)
          (setq mycount (1+ mycount)))
        (forward-char) ;;;$B<!$NJ8;z$X(B
        )
      ;;$B7k2L$NI=<((B
      (if (< 0 mycount)
          (error "$BJ8;z%3!<%I$N0[$J$kJ8;z$,4^$^$l$F$$$^$9(B"))
      (if transient-mark-mode
          (setq deactivate-mark t)) ;;;region $B$r?'$D$1$7$F$$$k;~!"?'$r2r=|(B
      )))

(defun pukiwiki-check-jisx0201 ()
  (interactive)
  (let ((beg (point-min))
        (end (point-max))
        debug-on-quit)
    (if (memq 'katakana-jisx0201 (find-charset-region beg end))
        (let ((buffer-read-only nil)
              (inhibit-read-only t))
          (japanese-zenkaku-region beg end t)
          t)
      t)))

;;; Util

(defun pukiwiki-site-name (&optional site-info)
  (nth 0 (or site-info pukiwiki-site-info)))

(defun pukiwiki-site-url (&optional site-info)
  (nth 1 (or site-info pukiwiki-site-info)))

(defun pukiwiki-site-style (&optional site-info)
  (or (nth 2 (or site-info pukiwiki-site-info))
      'default))

(defun pukiwiki-site-coding-system (&optional site-info)
  (or (nth 3 (or site-info pukiwiki-site-info))
    'euc-jp-dos))

(defun pukiwiki-page-buffer-name (pagename site-info)
  (let ((buf (cdr (assoc (list (pukiwiki-site-name site-info) pagename)
                         pukiwiki-page-buffer-alist))))
    (and buf (buffer-name buf))))

(defun pukiwiki-index-buffer-name (&optional site-info)
  (format "Pukiwiki index <%s>" (pukiwiki-site-name site-info)))

(defun pukiwiki-index-sort-by (&optional key arg)
  (unless key
    (unless pukiwiki-index-sort-key
      (setq pukiwiki-index-sort-key '(?n nil)))
    (setq key (nth 0 pukiwiki-index-sort-key))
    (setq arg (nth 1 pukiwiki-index-sort-key)))
  (setq pukiwiki-index-sort-key (list key arg))
  (setq buffer-read-only nil)
  (save-excursion
    (goto-char (point-min))
    (cond
     ((eq key ?n) (pukiwiki-index-sort-subr arg 0))
     ((eq key ?d) (pukiwiki-index-sort-subr (not arg) 3))))
  (set-buffer-modified-p nil)
  (setq buffer-read-only t))

(defun pukiwiki-index-sort-subr (rev num)
  (sort-subr rev
             'forward-line 'end-of-line
             (lambda () (nth num (pukiwiki-index-page-info-current-line)))))

(defun pukiwiki-replace-entity-refs (&optional str)
  "Replace entity references.

If STR is a string, replace entity references within the string.
Otherwise replace all entity references within current buffer."
  (pukiwiki-do-replace-entity-ref
   "&amp;" "&"
   (pukiwiki-do-replace-entity-ref
    "&lt;" "<"
    (pukiwiki-do-replace-entity-ref
     "&gt;" ">"
     (pukiwiki-do-replace-entity-ref
      "&quot;" "\""
      (pukiwiki-do-replace-entity-ref
       "&#39;" "'" str))))))

(defun pukiwiki-do-replace-entity-ref (from to &optional str)
  (save-match-data
    (save-excursion
      (goto-char (point-min))
      (if (stringp str)
          (progn
            (while (string-match from str)
              (setq str (replace-match to nil nil str)))
            str)
        (while (search-forward from nil t)
          (replace-match to nil nil))))))

(defun pukiwiki-propertize (string &rest properties)
  "Return a copy of STRING with text PROPERTIES added."
  (prog1
      (setq string (copy-sequence string))
    (add-text-properties 0 (length string) properties string)))

(defun pukiwiki-prefix (str width)
  "STR $B$N@hF,$r(B WIDTH$BJ8;z$r<h$j=P$9!#(B

WIDTH $B$KK~$?$J$$>l9g$O!"KvHx$K6uGr$,%Q%G%#%s%0$5$l$k!#(B"

  (let (l (result "")
          (w (string-width str)))
    (if (< w width)
        (setq str (concat str (make-string (1- width) ? ))
              w (string-width str)))
    (setq l (string-to-list str))
    (while (< (char-width (car l)) width)
      (setq result (concat result (char-to-string (car l))))
      (setq width (- width (char-width (car l))))
      (setq l (cdr l)))
    (concat result (make-string width ? ))))

;;; Util (http-*)

(defun http-url-unhexify-string (str coding)
  "Unescape characters in a string."
  (save-match-data
    (let ((result (string-as-unibyte str)) (pos -1))
      (while (setq pos (string-match "+" result (1+ pos)))
        (setq result (replace-match " " nil nil result)))
      (setq pos -1)
      (while (setq pos
                   (string-match
                    "%\\([0-9a-fA-F][0-9a-fA-F]\\)" result (1+ pos)))
        (setq result
              (replace-match
               (format "%c"
                       (eval (read (concat "?\\x"
                                           (match-string 1 result)))))
               t t result)))
      (decode-coding-string result coding))))

(defun http-url-hexify-alist (alist coding)
  (mapcar
   (lambda (c)
     (cons (car c) (and (cdr c) (http-url-hexify-string (cdr c) coding))))
   alist))


;;; for pukiwiki

;;;; $B%Z!<%8I=<($N@07A4XO"(B
(defun pukiwiki-insert-comment-str (str)
  (let ((contents nil) (lth nil) (end nil))
    (with-temp-buffer
      (text-mode)
      (insert str)
      (goto-char (point-min))
      (keep-lines "^-")
      (goto-char (point-min))
      (replace-string "&amp;new{" "" nil (point-min) (point-max))
      (replace-string "&new{" "" nil (point-min) (point-max))
      (replace-string "SIZE(10){" "" nil (point-min) (point-max))
      (replace-regexp "}[;]*" "" nil (point-min) (point-max))
      (replace-regexp
       "^\\([-]+\\)\\([^\n\r]+\\)[ ]*--[ ]*\\[\\[\\([^]\n\r]+\\)\\]\\] \\([^\n\r]+\\)[ ]*$"
       "\\1 \\4 [\\3]\n    \\2"
       nil (point-min) (point-max))
      (replace-regexp
       "^\\([-]+\\)\\([^\n\r]+\\)[ ]*--[ ]*\\([^[\n\r]+\\)[ ]*$"
       "\\1 \\3 [$BL>L5$7(B]\n    \\2"
       nil (point-min) (point-max))
      (replace-regexp "[ ]+$"
                      ""
                      nil (point-min) (point-max))
      (replace-regexp "\n"
                      "\n\n"
                      nil (point-min) (point-max))
      (goto-char (point-min))
      (while (re-search-forward
              "^[-]+" nil t nil)
        (setq lth (length (match-string 0)))
        (delete-region (match-beginning 0) (match-end 0))
        (insert
         (concat
          (if (not (= lth 1))
              (make-string (* 2 lth) ? ))
          "_"))
        (when (not (= lth 1))
          (save-excursion
            (if (re-search-forward "^[-]+" nil t nil)
                (setq end (point))
              (setq end (progn (goto-char (point-max))
                               (line-beginning-position)))))
          (save-excursion
            (forward-line 1)
            (beginning-of-line)
            (while (> end (point))
              (insert
               (format "%s" (make-string (* 2 lth) ? )))
              (forward-line 1)
              (beginning-of-line)))))
      ;;       (replace-regexp "^[-]+"
      ;;                       "_"
      ;;                       nil (point-min) (point-max))
      (fill-region (point-min) (point-max))
      (replace-regexp "\n[ ]*\n"
                      "\n"
                      nil (point-min) (point-max))
      (setq contents (buffer-substring-no-properties
                      (point-min) (point-max))))
    (insert contents)))

(defun pukiwiki-insert-comment ()
  "pcomment $B$K$h$k%3%a%s%H$rA^F~$9$k(B"
  (let ((comment nil))
    (save-excursion
      (save-current-buffer
        (goto-char (point-min))
        (when (re-search-forward "^#pcomment\\((reply)\\|\\)" nil t)
          (message "Inserting pcomment...")
          (setq comment
                (pukiwiki-fetch-source
                 (concat "$B%3%a%s%H(B/" pukiwiki-pagename)
                 (pukiwiki-site-url pukiwiki-site-info)
                 (pukiwiki-site-coding-system pukiwiki-site-info)))
          (forward-line 1)
          (if (cdr (assoc 'body comment))
              (pukiwiki-insert-comment-str (cdr (assoc 'body comment))))
          (pukiwiki-replace-entity-refs))
        (goto-char (point-min))
        (when (re-search-forward "^#pcomment(reply[ ]*,\\([^),]+\\))" nil t)
          (message "Inserting comment...")
          (setq comment
                (pukiwiki-fetch-source
                 (buffer-substring-no-properties
                  (match-beginning 1) (match-end 1))
                 (pukiwiki-site-url pukiwiki-site-info)
                 (pukiwiki-site-coding-system pukiwiki-site-info)))
          (forward-line 1)
          (if (cdr (assoc 'body comment))
              (pukiwiki-insert-comment-str (cdr (assoc 'body comment))))
          (pukiwiki-replace-entity-refs)
          )))
    (message "Inserting comment...done!")))

(defun pukiwiki-insert-contents ()
  "$BL\<!$rA^F~$9$k(B"
  (let ((chapter nil) (lst nil))
    (save-excursion
      (save-current-buffer
        (goto-char (point-min))
        (while (re-search-forward "^[*]+[^\n\r]+" nil t)
          (setq chapter
                (cons
                 (buffer-substring-no-properties
                  (match-beginning 0)
                  (match-end 0))
                 chapter)))
        (setq chapter (reverse chapter))
        (goto-char (point-min))
        (while (re-search-forward
                "^#contents" nil t)
          (forward-line 1)
          (setq lst chapter)
          (while lst
            (insert
             (concat
              " "
              (car lst)
              "\n"))
            (setq lst (cdr lst))))))))

(defun pukiwiki-insert-ls2 ()
  (goto-char (point-min))
  (when pukiwiki-index-page-info-list
    ;;#ls2($B@_Dj:Q$_(BMeadow/)
    ;;#ls2
    ;;#ls2($B%Q%?!<%s(B,$B%Q%i%a!<%?(B,link$B$NJLL>I=<((B)
    (let ((pagename nil))
      (while (re-search-forward "^#ls2" nil t)
        (beginning-of-line)
        (cond
         ((re-search-forward "(\\([^,]+\\))" (line-end-position) t)
          (setq pagename
                (buffer-substring-no-properties
                 (match-beginning 1) (match-end 1))))
         ((re-search-forward "(\\([^,]+\\),.+)" (line-end-position) t)
          (setq pagename
                (buffer-substring-no-properties
                 (match-beginning 1) (match-end 1))))
         (t
          (setq pagename
                (concat
                 "^" pukiwiki-pagename "/"))))
        (end-of-line)
        (insert "\n")
        (let ((ls2 nil))
          (setq ls2
                (delete
                 nil
                 (mapcar
                  (lambda (elm)
                    (if (string-match
                         pagename (nth 1 elm))
                        (nth 1 elm)))
                  pukiwiki-index-page-info-list)))
          (while ls2
            (insert
             (concat "\n- [[" (car ls2) "]]"))
            (setq ls2 (cdr ls2))))))))

(defun pukiwiki-text-reformating ()
  "$BI=<($N;~$KFI$_$d$9$$$h$&$KE,Ev$J$H$3$m$G@^$jJV$9$J$I$N=hM}$r9T$&(B"
  (let ((pt) (end-pt) (reg "^[^ *->#\n\r]+[^\n\r]*$") str)
    (goto-char (point-min))
    (while (re-search-forward "[ ]*[~]+[ ]*$" nil t)
      (delete-region (match-beginning 0)
                     (match-end 0))
      (insert "\n"))

    ;; $BDj5A$rFI$_$d$9$/=hM}(B
    (goto-char (point-min))
    (replace-regexp "^:\\([^|\n\r]+\\)|\\([^\n\r]+\\)$"
                    "\\1\n\n\t\\2\n"
                    nil (point-min) (point-max))

    ;; $B%3%a%s%H$rFI$_$d$9$/=hM}(B
    (goto-char (point-min))
    (while (re-search-forward
            "^[-]+\\([^\n\r]+\\)[ ]*--[ ]*[^\n\r]+[0-9]+-[0-9]+-[0-9]+[^\n\r]+"
            nil t)
      (setq pt (line-beginning-position))
      (if (re-search-forward
           "\\(^$\\|^[^-\n\r]+\\)" nil t)
          (setq end-pt (- (line-beginning-position) 1))
        (setq end-pt (point-max)))
      (setq str
            (buffer-substring-no-properties
             pt end-pt))
      (delete-region pt end-pt)
      (insert "\n")
      (pukiwiki-insert-comment-str str)
      (insert "\n"))

    (goto-char (point-min))
    (while (re-search-forward
            "^*" nil t)
      (save-excursion
        (forward-line -1)
        (if (string=
             ""
             (buffer-substring
              (line-beginning-position)
              (line-end-position)))
            ()
          (end-of-line)
          (insert "\n"))))

    (goto-char (point-min))
    (while (re-search-forward
            "^*" nil t)
      (save-excursion
        (forward-line 1)
        (if (string=
             ""
             (buffer-substring
              (line-beginning-position)
              (line-end-position)))
            ()
          (beginning-of-line)
          (insert "\n"))))

    ;; $B9T$rE,Ev$J$H$3$m$G@^$jJV$9(B
    (goto-char (point-min))
    (while (re-search-forward
            "^[^ |*->#\n\r]+[^\n\r]*$" nil t)
      ;;(beginning-of-line)
      ;;(setq pt (point))
      (setq pt (line-beginning-position))
      (if (re-search-forward
           "^[ |*->#]+[^\n\r]*$" nil t)
          ;;(setq end-pt (point))
          (setq end-pt (line-beginning-position)) ;;(line-end-position))
        (setq end-pt (point-max))
        (goto-char (point-max)))
      (fill-region pt end-pt))))

(defun pukiwiki-insert-attach-file-list ()
  (when pukiwiki-index-attach-list
    (let ((lst nil))
      (if (assoc pukiwiki-pagename
                 pukiwiki-index-attach-list)
          (setq lst
                (cdr
                 (assoc pukiwiki-pagename
                        pukiwiki-index-attach-list))))
      (when lst
        (goto-char (point-max))
        (insert "\n* $BE:IU%U%!%$%k(B\n\n")
        (while lst
          (insert "#ref(" (car lst) ")\n")
          (setq lst (cdr lst)))))))

(defun pukiwiki-set-face-color ()
  (let ((color nil) (str nil) (ov nil))
    (goto-char (point-min))
    (while (re-search-forward
            "COLOR(\\([^)]+\\)){\\([^\n\r]+\\)}" nil t)
      (setq color (match-string 1))
      (setq str (match-string 2))
      (delete-region (match-beginning 0)
                     (match-end 0))
      (insert str)
      (when (x-color-defined-p color)
        (setq ov (make-overlay (line-beginning-position)
                               (point)))
        (overlay-put ov 'face (cons
                               (cons 'background-color "white")
                               (cons 'foreground-color color)))
        (overlay-put ov 'priority 0)))))

(defun pukiwiki-set-bold ()
  (goto-char (point-min))
  (let ((start nil) (str nil) (ov nil))
    (while (re-search-forward
            "\\([']+\\)\\([^'\n\r]+\\)\\([']+\\)" nil t)
      (setq start (match-string 1))
      (setq str (match-string 2))
      (delete-region (match-beginning 0)
                     (match-end 0))
      (insert str)
      (setq ov (make-overlay (line-beginning-position)
                             (point)))
      (if (= 2 (length start))
          (overlay-put ov 'face 'bold)
        (overlay-put ov 'face 'italic))
      (overlay-put ov 'priority 0))))

(defun pukiwiki-set-justification ()
  (interactive)
  (let ((just nil) (str nil) (ov nil) beg)
    (goto-char (point-min))
    (while (re-search-forward
            "\\(CENTER:\\|RIGHT:\\)\\([^\r\n]+\\)" nil t)
      (setq just (match-string 1))
      (setq str (match-string 2))
      (delete-region (match-beginning 0)
                     (match-end 0))
      (setq beg (point))
      (insert str)
      (save-excursion
        (forward-line 1)
        (if (string= (buffer-substring-no-properties
                      (line-beginning-position)
                      (line-end-position))
                     "")
            ()
          (insert "\n")
          (forward-char -1)))
      (cond
       ((string= just "CENTER:")
        (set-justification beg
                           (point)
                           'center t))
       ((string= just "RIGHT:")
        (set-justification beg
                           (point)
                           'right t)))
      (beginning-of-line))))

(defun pukiwiki-insert-cite ()
  (interactive)
  (goto-char (point-min))
  (let ((str nil)
        (page-delimiter "^")
        (paragraph-start (concat page-delimiter "\\|[ \t]*$"))
        (paragraph-start
         (if (eq ?^ (aref paragraph-start 0))
             (substring paragraph-start 1)))
        (paragraph-separate paragraph-start))
  (while (re-search-forward "\\(^[>]+\\)" nil t)
    (setq str (length (match-string-no-properties 1)))
    (delete-region (match-beginning 0)
                   (match-end 0))
    (while (and str (not (= str 0)))
      (insert "  ")
      (setq str (- str 1)))
    (beginning-of-line)
    (insert "\n")
    (fill-paragraph nil))))

(defun pukiwiki-insert-list ()
  (message "list...")
  (goto-char (point-min))
  (let ((str nil) (level 1) (num 1))
    (while (re-search-forward "\\(^[-+]+\\)" nil t)
      (setq str (length (buffer-substring-no-properties
                         (match-beginning 0)
                         (match-end 0))))
      (delete-region (match-beginning 0)
                     (match-end 0))
      (while (not (= str 1))
        (insert "  ")
        (setq str (- str 1)))
      (insert "-")
      (let* ((fill-prefix "  ")
             (paragraph-start
              "^\\($\\|[ ]*[+-]+\\)")
             (paragraph-separate "^[ ]*[+- <\t\n\f]"))
        (fill-region (line-beginning-position)
                     (line-end-position))
        ;;(fill-paragraph nil)
        ))))

(defun pukiwiki-delete-blank-line ()
  (goto-char (point-min))
  (while (re-search-forward
          "^\\([ \t]+\\)$" nil t)
    (delete-region (match-beginning 1)
                   (match-end 1)))
  (goto-char (point-min))
  (while (re-search-forward
          "^\n[\n]+" nil t)
    (delete-region (match-beginning 0)
                   (match-end 0))
    (insert "\n")))

(defun pukiwiki-set-auto-face ()
  (when pukiwiki-auto-face
    (pukiwiki-set-bold)
    (pukiwiki-set-face-color)
    (pukiwiki-set-justification)))

(defun pukiwiki-mode-auto-insert ()
  (save-excursion
    (pukiwiki-text-reformating)
    (pukiwiki-insert-cite)
    (pukiwiki-insert-comment)
    (pukiwiki-insert-list)
    (pukiwiki-insert-ls2)
    (pukiwiki-insert-attach-file-list)
    (pukiwiki-set-auto-face)
    (pukiwiki-delete-blank-line)
    ))

(defun pukiwiki-outline-renumber ()
  "Renumber headings in buffer."
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (let ((stack nil)
          (whole 1))
      (while (re-search-forward "^[*]+ \\([0-9]+\\(\\.[0-9]+\\)*\\)" nil t)
        (let ((level (1+ (loop for c across
                               (match-string whole) count (eq c ?.)))))
          (while (< level (length stack)) (pop stack))
          (while (> level (length stack)) (push 0 stack))
          (incf (car stack)))
        (delete-region (match-beginning whole) (match-end whole))
        (apply #'insert
               (nreverse (butlast
                          (mapcan #'(lambda (x)
                                      (list (number-to-string x) "."))
                                  stack))))))))

(defun pukiwiki-view-renumber ()
  "Renumber headings in buffer."
  (interactive)
  (let ((header nil))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward
              "^\\([*]+\\)" nil t)
        (setq header (length (match-string 1)))
        (insert " ")
        (while (not (= header 0))
          (insert "1.")
          (setq header (- header 1))))
      (goto-char (point-min))
      (if (re-search-forward "^#contents" nil t)
          ()
        (goto-char (point-max))
        (insert "\n\n* 1. $B$b$/$8(B\n\n#contents\n"))
      (goto-char (point-min))
      (pukiwiki-outline-renumber)
      )))

(defun pukiwiki-view-delete-html-tag ()
  (let (pt end-pt)
    (goto-char (point-min))
    (while (re-search-forward
            "<span class=\"diff_removed\">" nil t)
      (setq pt (match-beginning 0))
      (if (re-search-forward "</span>" nil t)
          (setq end-pt (point)))
      (if end-pt
          (delete-region pt end-pt))
      (setq pt nil
            end-pt nil))
    (goto-char (point-min))
    (while (re-search-forward
            "\\(<span class=\"[^\"]+\">\\|</span>\\)" nil t)
      (delete-region (match-beginning 0)
                     (match-end 0)))
    (goto-char (point-min))))

(defun pukiwiki-view-reformating ()
  (when (not buffer-read-only)
    (pukiwiki-view-delete-html-tag)
    (when (and pukiwiki-auto-insert
               (not buffer-read-only))
      (pukiwiki-mode-auto-insert))
    (pukiwiki-view-renumber)
    (pukiwiki-insert-contents)))

;;;; http.el $BFb$N4X?t$r(B pukiwiki $BMQ$K2~B$$7$?$b$N$?$A(B
(defvar pukiwiki-http-fetch-running nil)
(defun pukiwiki-http-fetch-sentinel (proc str)
  (setq pukiwiki-http-fetch-running nil))

(defun pukiwiki-http-fetch (url method &optional user pass data)
  "Fetch via HTTP.

URL is a url to be POSTed.
METHOD is 'get or 'post.
USER and PASS must be a valid username and password, if required.
DATA is an alist, each element is in the form of (FIELD . DATA).

If no error, return a buffer which contains output from the web server.
If error, return a cons cell (ERRCODE . DESCRIPTION)."
  (setq pukiwiki-http-fetch-running t)
  (let (connection server port path buf str len)
    (string-match
     "^http://\\([^/:]+\\)\\(:\\([0-9]+\\)\\)?\\(/.*$\\)" url)
    (setq server (match-string 1 url)
          port (string-to-int (or (match-string 3 url) "80"))
          path (if http-proxy-server url (match-string 4 url)))
    (setq str (mapconcat
               '(lambda (x)
                  (concat (car x) "=" (cdr x)))
               data "&"))
    (setq len (length str))
    (save-excursion
      (setq buf (get-buffer-create
                 (concat "*result from " server "*")))
      (set-buffer buf)
      (erase-buffer)
      (setq connection
            (as-binary-process
             (open-network-stream
              (concat "*request to " server "*")
              buf
              (or http-proxy-server server)
              (or http-proxy-port port))))
      (set-process-coding-system connection 'binary 'binary)
      ;;(set-process-sentinel connection 'ignore)
      (set-process-sentinel connection 'pukiwiki-http-fetch-sentinel)
      (process-send-string
       connection
       (concat (if (eq method 'post)
                   (concat "POST " path)
                 (concat "GET " path (if (> len 0)
                                         (concat "?" str))))
               " HTTP/1.0\r\n"
               (concat "Host: " server "\r\n")
               "Connection: close\r\n"
               "Content-type: application/x-www-form-urlencoded\r\n"
               (if (and user pass)
                   (concat "Authorization: Basic "
                           (base64-encode-string
                            (concat user ":" pass))
                           "\r\n"))
               (if (eq method 'post)
                   (concat "Content-length: " (int-to-string len) "\r\n"
                           "\r\n"
                           str))
               "\r\n"))
      (while pukiwiki-http-fetch-running
        (unless (accept-process-output connection http-timeout)
          (error "HTTP fetch: Connection timeout!"))
        (sit-for 30))
      (delete-process connection)
      (goto-char (point-min))
      (save-excursion
        (if (re-search-forward
             "HTTP/1.[01] \\([0-9][0-9][0-9]\\) \\(.*\\)" nil t)
            (let ((code (match-string 1))
                  (desc (match-string 2)))
              (cond ((or (equal code "200")
                         (equal code "302"))
                     buf)
                    ;; support authorization
                    ((equal code "401")
                     (let (user pass)
                       (setq user (read-from-minibuffer "Username: "))
                       (unless (equal user "")
                         (setq pass (read-passwd "Password: "))
                         (setq buf (pukiwiki-http-fetch
                                    url method user pass data)))))
                    (t
                     (cons code desc))))
          buf)))))

(defvar pukiwiki-download-running nil)
(defun pukiwiki-download-sentinel (proc str)
  (setq pukiwiki-download-running nil))

(defun pukiwiki-http-download (url filename)
  "Fetch via HTTP and save to the filename."
  (message "Downloading file...")
  (let (connection server port path buf str len
                   (coding-system-for-write 'binary)
                   (coding-system-for-read 'binary))
    (setq pukiwiki-download-running t)
    (string-match
     "^http://\\([^/:]+\\)\\(:\\([0-9]+\\)\\)?\\(/.*$\\)" url)
    (setq server (match-string 1 url)
          port (string-to-int (or (match-string 3 url) "80"))
          path (if http-proxy-server url (match-string 4 url)))
    (save-excursion
      (setq buf (get-buffer-create
                 (concat "*result from " server "*")))
      (set-buffer buf)
      (erase-buffer)
      (setq connection
            (open-network-stream
             (concat "*request to " server "*")
             buf
             (or http-proxy-server server)
             (or http-proxy-port port)))
      (set-process-coding-system connection 'binary 'binary)
      (set-process-sentinel connection 'pukiwiki-download-sentinel)
      (process-send-string
       connection
       (concat
        (concat "GET " path
                " HTTP/1.0\r\n"
                (concat "Host: " server "\r\n")
                "Connection: close\r\n"
                "Content-type: application/x-www-form-urlencoded\r\n"
                "\r\n")))
      (goto-char (point-min))
      (while pukiwiki-download-running
        (sit-for 30))
      (delete-process connection)
      )
    (set-buffer buf)
    (goto-char (point-min))
    ;; header $B$N8e$K0\F0(B
    (re-search-forward "\r\n\r?\n" nil t)
    (write-region
     (point) (point-max) filename))
  (message "Downloading file... done!"))

;;;; $B%$%s%G%C%/%9$NF|IU$1=hM}4XO"(B
(defun pukiwiki-date-convert (time)
  "time $B$G;XDj$7$?F|$@$1A0$N;~4V$rJV$9(B"
  (let ((now (current-time)) (days (* -1 time))
        dateh datel daysec daysh daysl dir
        (offset 0))
    (setq daysec (* -1.0 days 60 60 24))
    (setq daysh (floor (/ daysec 65536.0)))
    (setq daysl (round (- daysec (* daysh 65536.0))))
    (setq dateh (- (nth 0 now) daysh))
    (setq datel (- (nth 1 now) (* offset 3600) daysl))
    (if (< datel 0)
        (progn
          (setq datel (+ datel 65536))
          (setq dateh (- dateh 1)))) ;;(floor (/ offset 24))))))
    (if (< dateh 0)
        (setq dateh 0))
    ;;(insert (concat (int-to-string dateh) ":"))
    (list dateh datel)))

(defun pukiwiki-last-date (time)
  (let ((now (current-time)) (days (* -1 time)) dateh datel daysec daysh daysl dir
        (offset 0))
    (setq daysec (* -1.0 days 60 60 24))
    (setq daysh (floor (/ daysec 65536.0)))
    (setq daysl (round (- daysec (* daysh 65536.0))))
    (setq dateh (- (nth 0 now) daysh))
    (setq datel (- (nth 1 now) (* offset 3600) daysl))
    (if (< datel 0)
        (progn
          (setq datel (+ datel 65536))
          (setq dateh (- dateh 1)))) ;;(floor (/ offset 24))))))
    (if (< dateh 0)
        (setq dateh 0))
    (list dateh datel)))

(defun pukiwiki-index-date (time-str)
  (let ((current-hour
         (string-to-int (format-time-string "%H" (current-time)))))
    (if (and
         (string-match "(\\([0-9]+\\)\\($B;~4VA0(B\\|h\\))" time-str)
         (> (string-to-int (substring
                            time-str
                            1
                            (string-match "[^0-9]" time-str 1)))
            current-hour))
        (pukiwiki-last-date 1)
      (if (or
           (string-match "(\\([0-9]+\\)\\($B;~4VA0(B\\|h\\))" time-str)
           (string-match "(\\([0-9]+\\)\\($BJ,A0(B\\|m\\))" time-str))
          (pukiwiki-last-date 0)
        (string-match "(\\([0-9]+\\)\\($BF|A0(B\\|d\\))" time-str)
        (pukiwiki-last-date
         (string-to-int (substring
                         time-str
                         1
                         (string-match "[^0-9]" time-str 1))))))))

;;;; $B%"%s%+!<$G%j%?!<%s$7$?>l9g$N=hM}(B
(defun pukiwiki-get-face (pt)
  (if (get-text-property pt 'face)
      (get-text-property pt 'face)
    (get-char-property pt 'face)))

(defun pukiwiki-edit-new-line ()
  (interactive)
  (let ((cursorface
         (pukiwiki-get-face (point)))
        (prevb pukiwiki-prev-buffer)
        (wikiname nil)
        filename url (buf-name "*pukiwiki download*")
        end start)
    (cond
     ((string= cursorface 'pukiwiki-anchor-face)
      (if (not
           (string= 'pukiwiki-anchor-face
                    (pukiwiki-get-face (- (point) 1))))
          (setq start (point)))
      (if (not
           (string= 'pukiwiki-anchor-face
                    (pukiwiki-get-face (+ (point) 1))))
          (setq end (+ (point) 1)))
      (if start
          ()
        (setq start (previous-property-change (point))))
      (if end
          ()
        (setq end (next-property-change (point))))
      (setq wikiname (buffer-substring-no-properties start end))
      (let ((buffer-read buffer-read-only))
        (pukiwiki-index-edit-page wikiname)
        (setq buffer-read-only buffer-read))
      (delete-other-windows))

     ((string-match
       "^#ref(\\([^,)]+\\)[,)]"
       (buffer-substring-no-properties
        (line-beginning-position) (line-end-position)))
      (setq filename
            (match-string
             1
             (buffer-substring-no-properties
              (line-beginning-position) (line-end-position))))
      (setq url
            (concat
             (nth 1 pukiwiki-site-info)
             "?plugin=attach&pcmd=open&file="
             filename
             "&refer="
             (http-url-hexify-string
              pukiwiki-pagename
              (pukiwiki-site-coding-system))))
      (setq filename
            (concat
             (expand-file-name
              (concat (make-temp-name "pw")
                      filename)
              temporary-file-directory)))
      (pukiwiki-http-download url filename)
      (setq pukiwiki-preview-file-list
            (cons
             filename
             pukiwiki-preview-file-list))
      (let ((history file-name-history))
        (find-file filename)
        (setq file-name-history history))
      (pukiwiki-diff-mode)
      (setq pukiwiki-prev-buffer prevb)
      (set-buffer-modified-p nil))
     (t
      (newline-and-indent)))))

;;;; $B=*N;=hM}(B
(defadvice save-buffers-kill-emacs
  (before delete-temp-file-of-pukiwiki activate)
  (if pukiwiki-preview-file-list
      (while pukiwiki-preview-file-list
        (if (file-exists-p (car pukiwiki-preview-file-list))
            (delete-file (car pukiwiki-preview-file-list)))
        (setq pukiwiki-preview-file-list
              (cdr pukiwiki-preview-file-list)))))

(provide 'pukiwiki-mode)
;;; pukiwiki-mode.el ends here
