;;; cgreek-util.el --- utilities for classical Greek -*- coding: iso-2022-7bit; -*-

;; Copyright (C) 2002, 2003
;;   National Institute of Advanced Industrial Science and Technology (AIST)
;;   Registration Number H14PRO020

;; Author: TAKAHASHI Naoto <ntakahas@m17n.org>
;; Keywords: multilingual, classical Greek

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;; Last modified: 4 February 2003

;;; Code:

(defvar cgreek-menu-map
  (let ((map (make-sparse-keymap)))
    (define-key map [print] (cons "Print" (make-sparse-keymap)))
    (define-key map [print region] '("Region" . ps-print-region-with-faces))
    (define-key map [print buffer] '("Buffer" . ps-print-buffer-with-faces))

    (define-key map [restained] '("Unhighlight Greek" . cgreek-restained-mode))
    (define-key map [flamboyant] '("Highligh Greek" . cgreek-flamboyant-mode))

    (define-key map [convert] (cons "Convert Region" (make-sparse-keymap)))
    (define-key map [convert tex] '("TeX to CGreek" . tex-to-cgreek-latin1-region))
    (define-key map [convert tlg] '("TLG to CGreek" . tlg-to-cgreek-region))

    (define-key map [save] (cons "Save Buffer" (make-sparse-keymap)))
    (define-key map [save wingreek] '("WinGreek Format" . cgreek-write-file-wingreek))
    (define-key map [save iso-2022-7bit] '("CGreek Format" . cgreek-write-file-iso-2022-7bit))
    (define-key map [save tex] '("TeX Format" . cgreek-write-file-tex))

    (define-key map [open] (cons "Open File" (make-sparse-keymap)))
    (define-key map [open wingreek] '("WinGreek Format" . cgreek-find-file-wingreek))
    (define-key map [open iso-2022-7bit] '("CGreek Format" . cgreek-find-file-iso-2022-7bit))
    (define-key map [open tex] '("TeX Format" . cgreek-find-file-tex))
    (define-key map [open tlg] '("TLG Format" . cgreek-tlg-parse-authtab))

    map))

;;;###autoload
(defun setup-cgreek-environment ()
  "Setup multilingual environment for classical Greek."
  (interactive)

  ;; menu in cgreek environment
  (define-key-after (lookup-key global-map [menu-bar])
    [cgreek] (cons "CGreek" cgreek-menu-map) t)

  (add-hook 'find-file-hooks
	    '(lambda ()
	       (cgreek-highlight-greek-chars-in-buffer 0)
	       (set-buffer-modified-p nil)))
  (add-hook 'post-command-hook 'cgreek-highlight-preceding-greek-char))

;;;###autoload
(defun exit-cgreek-environment ()
  "Exit classical Greek environment."
  (global-set-key [menu-bar cgreek] nil)

  (remove-hook 'find-file-hooks
	       '(lambda ()
		  (cgreek-highlight-greek-chars-in-buffer 0)
		  (set-buffer-modified-p nil)))
  (remove-hook 'post-command-hook 'cgreek-highlight-preceding-greek-char))

;;; highlighting

(defvar cgreek-greek-text-property 'highlight
  "*Text property used to highlight Greek characters.")

(defvar cgreek-highlight-switch nil
  "*Non-nil means highlight Greek characters.")

(make-variable-buffer-local 'cgreek-highlight-switch)

(defun cgreek-highlight-greek-chars-in-buffer (arg)
  "Change text property of Greek characters in current buffer.
If ARG is positive, highlight.
If ARG is negative, unhighlight.
If ARG is zero and `cgreek-highlight-switch' is non-nil, highlight.
If ARG is zero and `cgreek-highlight-switch' is nil, unhighlight.
`cgreek-greek-text-proprety' defines how to highlight."

  (interactive "p")
  (let ((plist (list 'face cgreek-greek-text-property 'rear-nonsticky t)))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward "\\cg+" nil t)
	(cond
	 ((> arg 0)
	  (add-text-properties (match-beginning 0) (match-end 0) plist))
	 ((< arg 0)
	  (remove-text-properties (match-beginning 0) (match-end 0) plist))
	 (cgreek-highlight-switch
	  (add-text-properties (match-beginning 0) (match-end 0) plist))
	 (t
	  (remove-text-properties (match-beginning 0) (match-end 0) plist)))))))

(defun cgreek-flamboyant-mode nil
  (interactive)
  (cgreek-highlight-greek-chars-in-buffer 1)
  (setq cgreek-highlight-switch t))

(defun cgreek-restained-mode nil
  (interactive)
  (cgreek-highlight-greek-chars-in-buffer -1)
  (setq cgreek-highlight-switch nil))

(defun cgreek-highlight-preceding-greek-char nil
  (if (and cgreek-highlight-switch
	   (eq (char-charset (preceding-char)) 'cgreek)
	   (null (get-text-property (1- (point)) 'face)))
      (add-text-properties
       (point) (1- (point))
       (list 'face cgreek-greek-text-property 'rear-nonsticky t))))

;;; cgreek to tex

(defconst
 cgreek-to-tex-table
 [
  nil      nil      nil      nil      nil      nil      nil      nil
  nil      nil      nil      nil      nil      nil      nil      nil
  nil      nil      nil      nil      nil      nil      nil      nil
  nil      nil      nil      nil      nil      nil      nil      nil
  " "      " "      "s+"     "v"      " "      "k+"     "K+"     nil
  "{((}"   "{))}"   "*"      nil      ","      "-"      "."      " "
  "0"      "1"      "2"      "3"      "4"      "5"      "6"      "7"
  "8"      "9"      ";"      "?"      "<"      " "      ">"      " "
  "+"      "A"      "B"      "X"      "D"      "E"      "F"      "G"
  "H"      "I"      "w|"     "K"      "L"      "M"      "N"      "O"
  "P"      "Q"      "R"      "S"      "T"      "U"      "h|"     "W"
  "C"      "Y"      "Z"      "["      " "      "]"      "="      "_"
  nil      "a"      "b"      "x"      "d"      "e"      "f"      "g"
  "h"      "i"      "j"      "k"      "l"      "m"      "n"      "o"
  "p"      "q"      "r"      "s|"     "t"      "u"      "a|"     "w"
  "c"      "y"      "z"      "\\{"    " "      "\\}"    "~"      " "
  " "					; 128
  " "					; 129
  " "					; 130
  "i("					; 131
  "i)"					; 132
  "i'"					; 133
  "i('"					; 134
  "i)'"					; 135
  "i`"					; 136
  "i(`"					; 137
  "i)`"					; 138
  "i="					; 139
  "i(="					; 140
  "i)="					; 141
  "i+"					; 142
  "i+'"					; 143
  "i+`"					; 144
  "(="					; 145
  ")="					; 146
  "('"					; 147
  ")'"					; 148
  "(`"					; 149
  ")`"					; 150
  "{+'}"				; 151
  "e("					; 152
  "e)"					; 153
  "e'"					; 154
  "e('"					; 155
  "e)'"					; 156
  "e`"					; 157
  "e(`"					; 158
  "e)`"					; 159
  "{+`}"				; 160
  "a("					; 161
  "a)"					; 162
  "a'"					; 163
  "a('"					; 164
  "a)'"					; 165
  "a`"					; 166
  "a(`"					; 167
  "a)`"					; 168
  "a="					; 169
  "a(="					; 170
  "a)="					; 171
  "a(|"					; 172
  "a)|"					; 173
  "a'|"					; 174
  "a('|"				; 175
  "a)'|"				; 176
  "a`|"					; 177
  "a(`|"				; 178
  "a)`|"				; 179
  "a=|"					; 180
  "a(=|"				; 181
  "a)=|"				; 182
  "r("					; 183
  "r)"					; 184
  "h("					; 185
  "h)"					; 186
  "h'"					; 187
  "h('"					; 188
  "h)'"					; 189
  "h`"					; 190
  "h(`"					; 191
  "h)`"					; 192
  "h="					; 193
  "h(="					; 194
  "h)="					; 195
  "{|}"					; 196
  "h(|"					; 197
  "h)|"					; 198
  "h'|"					; 199
  "h('|"				; 200
  "h)'|"				; 201
  "h`|"					; 202
  "h(`|"				; 203
  "h)`|"				; 204
  "h=|"					; 205
  "h(=|"				; 206
  "h)=|"				; 207
  "o("					; 208
  "o)"					; 209
  "o'"					; 210
  "o('"					; 211
  "o)'"					; 212
  "o`"					; 213
  "o(`"					; 214
  "o)`"					; 215
  "u("					; 216
  "u)"					; 217
  "u'"					; 218
  "u('"					; 219
  "u)'"					; 220
  "u`"					; 221
  "u(`"					; 222
  "u)`"					; 223
  "u="					; 224
  "u(="					; 225
  "u)="					; 226
  "u+"					; 227
  "u+'"					; 228
  "u+`"					; 229
  "w("					; 230
  "w)"					; 231
  "w'"					; 232
  "w('"					; 233
  "w)'"					; 234
  "w`"					; 235
  "w(`"					; 236
  "w)`"					; 237
  "w="					; 238
  "w(="					; 239
  "w)="					; 240
  "w(|"					; 241
  "w)|"					; 242
  "w'|"					; 243
  "w('|"				; 244
  "w)'|"				; 245
  "w`|"					; 246
  "w(`|"				; 247
  "w)`|"				; 248
  "w=|"					; 249
  "w(=|"				; 250
  "w)=|"				; 251
  " "					; 252
  " "					; 253
  " "					; 254
  " "					; 255
  ])

(defun cgreek-char-to-codepoint (ch)
  "Receive a cgreek char CH and return its codepoint in the MuleCGreek font."
  (+ (* 96 (- (nth 1 (split-char ch)) 32)) (nth 2 (split-char ch)) -32))

(defun cgreek-to-tex-region (beg end)
  "Convert cgreek characters in the region into TeX notation."
  (interactive "*r")
  (save-restriction
    (narrow-to-region beg end)
    (cgreek-to-tex-buffer)))

(defun cgreek-to-tex-buffer nil
  "Convert cgreek characters in the current buffer into TeX notation."
  (interactive "*")
  (let (ch0 ch1)
    (goto-char (point-min))

    ;; regexp matcher overflows for a long Greek sequence, so we first
    ;; divide a Greek sequence into lines, then remove redundant pairs
    ;; of \end{greek} and \begin{greek}

    (while (re-search-forward "\\cg" nil t)
      (save-restriction
	;; narrowing to a region "\cg ... \cg"
	(narrow-to-region
	 (match-beginning 0)
	 (progn
	   (if (looking-at "\\(\\cg\\|[\t ]\\)*\\cg")
	       (goto-char (match-end 0)))
	   (point)))

	(goto-char (point-min))
	(insert "\\begin{greek}")

	(while (not (eobp))
	  (setq ch0 (following-char)
		ch1 (char-after (1+ (point))))

	  (cond

	   ((eq ch0 ?$,4 G(B)
	    (cond
	     ((memq ch1 '(?$,4 a(B ?$,4 e(B ?$,4 h(B ?$,4 i(B ?$,4 o(B ?$,4 u(B ?$,4 w(B))
	      (insert ")")
	      (delete-char 1))
	     ((eq ch1 ?$,4 G(B)
	      (insert "{''}")
	      (delete-char 2))
	     (t
	      (insert "{'}")
	      (delete-char 1))))

	   ((eq ch0 ?$,4! (B)
	    (cond
	     ((memq ch1 '(?$,4 a(B ?$,4 e(B ?$,4 h(B ?$,4 i(B ?$,4 o(B ?$,4 u(B ?$,4 w(B ?$,4 r(B))
	      (insert "(")
	      (delete-char 1))
	     ((eq ch1 ?$,4! (B)
	      (insert "{``}")
	      (delete-char 2))
	     (t
	      (insert "{`}")
	      (delete-char 1))))

	   ((eq ch0 ?$,4 K(B)
	    (cond
	     ((eq ch1 ?$,4 K(B)
	      (insert "{\\ddag}")
	      (delete-char 2))
	     (t
	      (insert "{\\dag}")
	      (delete-char 1))))

	   ((looking-at "\\cg")
	    (insert
	     (aref cgreek-to-tex-table (cgreek-char-to-codepoint ch0)))
	    (delete-char 1))

	   (t
	    ;; must be looking at a space or a tab
	    (forward-char 1))))

	(insert "\\end{greek}")))

    ;; remove redundant "\end{greek}, \begin{greek}" pairs
    (goto-char (point-min))
    (while (re-search-forward "\\\\end{greek}\\([ \t\n]*\\)\\\\begin{greek}"
			      nil t)
      (replace-match "\\1"))))

(defun cgreek-post-read-conversion (len)
  "Perform tex-to-cgreek-buffer and tex-to-latin1-buffer in this order."
  (save-excursion
    (save-restriction
      (narrow-to-region (point) (+ (point) len))
      (let ((buffer-modified-p (buffer-modified-p)))
	(tex-to-cgreek-buffer)
	(tex-to-latin1-buffer)
	(set-buffer-modified-p buffer-modified-p)
	(- (point-max) (point-min))))))

(defun cgreek-pre-write-conversion (from to)
  "Perform cgreek-to-tex-buffer and latin1-to-tex-buffer in this order."
  (let ((old-buffer (current-buffer)))
    (set-buffer (get-buffer-create " *cgreek-tmp*"))
    (erase-buffer)
    (if (stringp from)
	(insert from)
      (insert-buffer-substring old-buffer from to))
    (cgreek-to-tex-buffer)
    (latin1-to-tex-buffer)
    ;; return nil as annotation
    nil))

;;; tex to cgreek

(defconst cgreek-command-table
  (let ((vec
	 (make-vector
	  128
	  '(progn
	     (insert (aref cgreek-simple-table (following-char)))
	     (delete-char 1))))
	(l
	 '(?a ?e ?h ?i ?k ?o ?r ?s ?u ?w ?K ?\( ?\) ?\\ ?{)))
    (while l
      (aset vec (car l) '(cgreek-complex-transcription))
      (setq l (cdr l)))
    vec))

(defconst cgreek-simple-table
  [
   ;;   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F
   32  32  32  32  32  32  32  32  32  ?\t ?\n 32  32  32  32  32  ; 0
   32  32  32  32  32  32  32  32  32  32  32  32  32  32  32  32  ; 1
   32  32  ?\" ?#  ?$  ?%  ?&  ?$,4 G(B  nil nil ?$,4 J(B  ?$,4 `(B  ?$,4 L(B  ?$,4 M(B  ?$,4 N(B  ?/  ; 2
   ?$,4 P(B  ?$,4 Q(B  ?$,4 R(B  ?$,4 S(B  ?$,4 T(B  ?$,4 U(B  ?$,4 V(B  ?$,4 W(B  ?$,4 X(B  ?$,4 Y(B  ?:  ?$,4 Z(B  ?$,4 \(B  ?$,4 ~(B  ?$,4 ^(B  ?$,4 [(B  ; 3
   ?@  ?$,4 a(B  ?$,4 b(B  ?$,4 x(B  ?$,4 d(B  ?$,4 e(B  ?$,4 f(B  ?$,4 g(B  ?$,4 h(B  ?$,4 i(B  32  nil ?$,4 l(B  ?$,4 m(B  ?$,4 n(B  ?$,4 o(B  ; 4
   ?$,4 p(B  ?$,4 q(B  ?$,4 r(B  ?$,4 s(B  ?$,4 t(B  ?$,4 u(B  32  ?$,4 w(B  ?$,4 c(B  ?$,4 y(B  ?$,4 z(B  ?$,4 {(B  nil ?$,4 }(B  ?^  ?_  ; 5
   ?$,4! (B  nil ?$,4!"(B  ?$,4!8(B  ?$,4!$(B  nil ?$,4!&(B  ?$,4!'(B  nil nil ?$,4!*(B  nil ?$,4!,(B  ?$,4!-(B  ?$,4!.(B  nil ; 6
   ?$,4!0(B  ?$,4!1(B  nil nil ?$,4!4(B  nil ?$,4 C(B  nil ?$,4!#(B  ?$,4!9(B  ?$,4!:(B  nil ?$,4"$(B  ?}  ?~  32  ; 7
   ]
  "One-character to one-character mapping table from tex to cgreek.")

;;; Ugly, but more straightforward and (hopefully) faster
;;; than the previous version.
(defun cgreek-complex-transcription ()
  (let ((ch0 (following-char))
	(ch1 (char-after (1+ (point))))
	(ch2 (char-after (+ (point) 2)))
	(ch3 (char-after (+ (point) 3))))
    (cond

     ((eq ch0 ?a)
      (cond
       ((eq ch1 ?\()
	(cond
	 ((eq ch2 ?')
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4!o(B))
	   (t (delete-char 3) (insert ?$,4!d(B))))
	 ((eq ch2 ?`)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4!r(B))
	   (t (delete-char 3) (insert ?$,4!g(B))))
	 ((eq ch2 ?=)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4!u(B))
	   (t (delete-char 3) (insert ?$,4!j(B))))
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4!l(B))
	 (t (delete-char 2) (insert ?$,4!a(B))))
       ((eq ch1 ?\))
	(cond
	 ((eq ch2 ?')
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4!p(B))
	   (t (delete-char 3) (insert ?$,4!e(B))))
	 ((eq ch2 ?`)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4!s(B))
	   (t (delete-char 3) (insert ?$,4!h(B))))
	 ((eq ch2 ?=)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4!v(B))
	   (t (delete-char 3) (insert ?$,4!k(B))))
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4!m(B))
	 (t (delete-char 2) (insert ?$,4!b(B))))
       ((eq ch1 ?')
	(cond
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4!n(B))
	 (t (delete-char 2) (insert ?$,4!c(B))))
       ((eq ch1 ?`)
	(cond
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4!q(B))
	 (t (delete-char 2) (insert ?$,4!f(B))))
       ((eq ch1 ?=)
	(cond
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4!t(B))
	 (t (delete-char 2) (insert ?$,4!i(B))))
       ((eq ch1 ?|) (delete-char 2) (insert ?$,4!6(B))
       (t (delete-char 1) (insert ?$,4!!(B))))

     ((eq ch0 ?e)
      (cond
       ((eq ch1 ?\()
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4![(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4!^(B))
	 (t (delete-char 2) (insert ?$,4!X(B))))
       ((eq ch1 ?\))
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4!\(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4!_(B))
	 (t (delete-char 2) (insert ?$,4!Y(B))))
       ((eq ch1 ?') (delete-char 2) (insert ?$,4!Z(B))
       ((eq ch1 ?`) (delete-char 2) (insert ?$,4!](B))
       ((eq ch1 ?=) (delete-char 2) (insert ?$,4"\(B))
       (t (delete-char 1) (insert ?$,4!%(B))))

     ((eq ch0 ?h)
      (cond
       ((eq ch1 ?\()
	(cond
	 ((eq ch2 ?')
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4"((B))
	   (t (delete-char 3) (insert ?$,4!|(B))))
	 ((eq ch2 ?`)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4"+(B))
	   (t (delete-char 3) (insert ?$,4!(B))))
	 ((eq ch2 ?=)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4".(B))
	   (t (delete-char 3) (insert ?$,4""(B))))
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"%(B))
	 (t (delete-char 2) (insert ?$,4!y(B))))
       ((eq ch1 ?\))
	(cond
	 ((eq ch2 ?')
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4")(B))
	   (t (delete-char 3) (insert ?$,4!}(B))))
	 ((eq ch2 ?`)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4",(B))
	   (t (delete-char 3) (insert ?$,4" (B))))
	 ((eq ch2 ?=)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4"/(B))
	   (t (delete-char 3) (insert ?$,4"#(B))))
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"&(B))
	 (t (delete-char 2) (insert ?$,4!z(B))))
       ((eq ch1 ?')
	(cond
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"'(B))
	 (t (delete-char 2) (insert ?$,4!{(B))))
       ((eq ch1 ?`)
	(cond
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"*(B))
	 (t (delete-char 2) (insert ?$,4!~(B))))
       ((eq ch1 ?=)
	(cond
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"-(B))
	 (t (delete-char 2) (insert ?$,4"!(B))))
       ((eq ch1 ?|) (delete-char 2) (insert ?$,4 v(B))
       (t (delete-char 1) (insert ?$,4!((B))))

     ((eq ch0 ?i)
      (cond
       ((eq ch1 ?\()
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4!F(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4!I(B))
	 ((eq ch2 ?=) (delete-char 3) (insert ?$,4!L(B))
	 (t (delete-char 2) (insert ?$,4!C(B))))
       ((eq ch1 ?\))
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4!G(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4!J(B))
	 ((eq ch2 ?=) (delete-char 3) (insert ?$,4!M(B))
	 (t (delete-char 2) (insert ?$,4!D(B))))
       ((eq ch1 ?') (delete-char 2) (insert ?$,4!E(B))
       ((eq ch1 ?`) (delete-char 2) (insert ?$,4!H(B))
       ((eq ch1 ?=) (delete-char 2) (insert ?$,4!K(B))
       ((eq ch1 ?+)
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4!O(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4!P(B))
	 (t (delete-char 2) (insert ?$,4!N(B))))
       (t (delete-char 1) (insert ?$,4!)(B))))

     ((eq ch0 ?k)
      (cond
       ((eq ch1 ?+) (delete-char 2) (insert ?$,4 E(B))
       (t (delete-char 1) (insert ?$,4!+(B))))

     ((eq ch0 ?o)
      (cond
       ((eq ch1 ?\()
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4"3(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4"6(B))
	 (t (delete-char 2) (insert ?$,4"0(B))))
       ((eq ch1 ?\))
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4"4(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4"7(B))
	 (t (delete-char 2) (insert ?$,4"1(B))))
       ((eq ch1 ?') (delete-char 2) (insert ?$,4"2(B))
       ((eq ch1 ?`) (delete-char 2) (insert ?$,4"5(B))
       ((eq ch1 ?=) (delete-char 2) (insert ?$,4"](B))
       (t (delete-char 1) (insert ?$,4!/(B))))

     ((eq ch0 ?r)
      (cond
       ((eq ch1 ?\() (delete-char 2) (insert ?$,4!w(B))
       ((eq ch1 ?\)) (delete-char 2) (insert ?$,4!x(B))
       (t (delete-char 1) (insert ?$,4!2(B))))

     ((eq ch0 ?s)
      (cond
       ((eq ch1 ?+) (delete-char 2) (insert ?$,4 B(B))
       ((eq ch1 ?|) (delete-char 2) (insert ?$,4!3(B))
       ((eq ch1 ?') (delete-char 1) (insert ?$,4!3(B))
       ((and (>= ch1 ?a) (<= ch1 ?z)) (delete-char 1) (insert ?$,4!3(B))
       ((and (>= ch1 ?A) (<= ch1 ?Z)) (delete-char 1) (insert ?$,4!3(B))
       (t (delete-char 1) (insert ?$,4!*(B))))

     ((eq ch0 ?u)
      (cond
       ((eq ch1 ?\()
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4";(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4">(B))
	 ((eq ch2 ?=) (delete-char 3) (insert ?$,4"A(B))
	 (t (delete-char 2) (insert ?$,4"8(B))))
       ((eq ch1 ?\))
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4"<(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4"?(B))
	 ((eq ch2 ?=) (delete-char 3) (insert ?$,4"B(B))
	 (t (delete-char 2) (insert ?$,4"9(B))))
       ((eq ch1 ?') (delete-char 2) (insert ?$,4":(B))
       ((eq ch1 ?`) (delete-char 2) (insert ?$,4"=(B))
       ((eq ch1 ?=) (delete-char 2) (insert ?$,4"@(B))
       ((eq ch1 ?+)
	(cond
	 ((eq ch2 ?') (delete-char 3) (insert ?$,4"D(B))
	 ((eq ch2 ?`) (delete-char 3) (insert ?$,4"E(B))
	 (t (delete-char 2) (insert ?$,4"C(B))))
       (t (delete-char 1) (insert ?$,4!5(B))))

     ((eq ch0 ?w)
      (cond
       ((eq ch1 ?\()
	(cond
	 ((eq ch2 ?')
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4"T(B))
	   (t (delete-char 3) (insert ?$,4"I(B))))
	 ((eq ch2 ?`)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4"W(B))
	   (t (delete-char 3) (insert ?$,4"L(B))))
	 ((eq ch2 ?=)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4"Z(B))
	   (t (delete-char 3) (insert ?$,4"O(B))))
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"Q(B))
	 (t (delete-char 2) (insert ?$,4"F(B))))
       ((eq ch1 ?\))
	(cond
	 ((eq ch2 ?')
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4"U(B))
	   (t (delete-char 3) (insert ?$,4"J(B))))
	 ((eq ch2 ?`)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4"X(B))
	   (t (delete-char 3) (insert ?$,4"M(B))))
	 ((eq ch2 ?=)
	  (cond
	   ((eq ch3 ?|) (delete-char 4) (insert ?$,4"[(B))
	   (t (delete-char 3) (insert ?$,4"P(B))))
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"R(B))
	 (t (delete-char 2) (insert ?$,4"G(B))))
       ((eq ch1 ?')
	(cond
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"S(B))
	 (t (delete-char 2) (insert ?$,4"H(B))))
       ((eq ch1 ?`)
	(cond
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"V(B))
	 (t (delete-char 2) (insert ?$,4"K(B))))
       ((eq ch1 ?=)
	(cond
	 ((eq ch2 ?|) (delete-char 3) (insert ?$,4"Y(B))
	 (t (delete-char 2) (insert ?$,4"N(B))))
       ((eq ch1 ?|) (delete-char 2) (insert ?$,4 j(B))
       (t (delete-char 1) (insert ?$,4!7(B))))

     ((eq ch0 ?K)
      (cond
       ((eq ch1 ?+) (delete-char 2) (insert ?$,4 F(B))
       (t (delete-char 1) (insert ?$,4 k(B))))

     ((eq ch0 ?\()
      (cond
       ((eq ch1 ?') (delete-char 2) (insert ?$,4!S(B))
       ((eq ch1 ?`) (delete-char 2) (insert ?$,4!U(B))
       ((eq ch1 ?=) (delete-char 2) (insert ?$,4!Q(B))
       ((eq ch1 ?\() (delete-char 2) (insert ?$,4 H(B))
       (t (delete-char 1) (insert ?$,4! (B))))

     ((eq ch0 ?\))
      (cond
       ((eq ch1 ?') (delete-char 2) (insert ?$,4!T(B))
       ((eq ch1 ?`) (delete-char 2) (insert ?$,4!V(B))
       ((eq ch1 ?=) (delete-char 2) (insert ?$,4!R(B))
       ((eq ch1 ?\)) (delete-char 2) (insert ?$,4 I(B))
       (t (delete-char 1) (insert ?$,4 G(B))))
	  
     ((eq ch0 ?\\)
      (cond
       ((eq ch1 ?{) (delete-char 2) (insert ?$,4!;(B))
       ((eq ch1 ?}) (delete-char 2) (insert ?$,4!=(B))
       (t (error "Unrecognised cgreek macro \\%c" ch1))))

     ;; now ch0 must be ?{
     ;; don't forget to remove the closing ?}

     ((eq ch1 ?+)
      (cond
       ((eq ch2 ?') (delete-char 4) (insert ?$,4!W(B))
       ((eq ch2 ?`) (delete-char 4) (insert ?$,4!`(B))
       ((eq ch2 ?}) (delete-char 3) (insert ?$,4 `(B))
       (t (error "Unrecognised cgreek macro {+%c" ch2))))

     ((eq ch1 ?')
      (cond
       ((eq ch2 ?') (delete-char 4) (insert "$,4 G G(B"))
       ((eq ch2 ?}) (delete-char 3) (insert ?$,4 G(B))
       (t (error "Unrecognised cgreek macro {'%c" ch2))))

     ((eq ch1 ?`)
      (cond
       ((eq ch2 ?`) (delete-char 4) (insert "$,4! ! (B"))
       ((eq ch2 ?}) (delete-char 3) (insert ?$,4! (B))
       (t (error "Unrecognised cgreek macro {`%c" ch2))))

     ((eq ch1 ?|) (delete-char 3) (insert ?$,4"$(B))

     ((looking-at "{((}")
      (delete-region (point) (match-end 0))
      (insert ?$,4 H(B))

     ((looking-at "{))}")
      (delete-region (point) (match-end 0))
      (insert ?$,4 I(B))

     ((looking-at "{\\\\dag}")
      (delete-region (point) (match-end 0))
      (insert ?$,4 K(B))

     ((looking-at "{\\\\ddag}")
      (delete-region (point) (match-end 0))
      (insert "$,4 K K(B"))

     (t
      (error "Unrecognised cgreek macro {%c" ch1)))))

(defun tex-to-cgreek-region (beg end)
  "Convert the region from tex notation into cgreek characters."
  (interactive "*r")
  (let ((case-fold-search nil))
    (save-restriction
      (narrow-to-region beg end)
      (goto-char (point-min))
      (while (not (eobp))
	(eval (aref cgreek-command-table (following-char)))))))

(defun tex-to-cgreek-buffer nil
  "Convert Greek regions in the current buffer from tex notation into cgreek characters.
A Greek region means a region surrounded by \"\\begin{greek}\" and
\"\\end{greek}\"."
  (interactive "*")

  (goto-char (point-min))
  (while (search-forward "\\begin{greek}" nil t)
    (replace-match "")
    (tex-to-cgreek-region
     (point)
     (progn
       (if (search-forward "\\end{greek}" nil 0)
	   (replace-match ""))
       (point))))

  (cgreek-highlight-greek-chars-in-buffer 0))

;;; file I/O

(defun cgreek-find-file-iso-2022-7bit (filename)
  "Edit file FILENAME encoded in iso-2022-7bit.
Switch to a buffer visiting file FILENAME,
creating one if none already exists."
  (interactive "FFind file: ")
  (let ((coding-system-for-read 'iso-2022-7bit))
    (find-file filename)))

(defun cgreek-find-file-wingreek (filename)
  "Edit file FILENAME encoded in WinGreek charset.
Switch to a buffer visiting file FILENAME,
creating one if none already exists."
  (interactive "FFind file: ")
  (let ((coding-system-for-read 'greek-WinGreek))
    (find-file filename)))

(defun cgreek-find-file-tex (filename)
  "Edit file FILENAME encoded in the TeX format.
Switch to a buffer visiting file FILENAME,
creating one if none already exists."
  (interactive "FFind file: ")
  (let ((coding-system-for-read cgreek-tex-coding-system))
    (find-file filename)))

(defun cgreek-write-file-iso-2022-7bit (filename)
  "Write current buffer into file FILENAME in iso-2022-7bit.
Makes buffer visit that file, and marks it not modified.
If the buffer is already visiting a file, you can specify
a directory name as FILENAME, to write a file of the same
old name in that directory."
  (interactive
   (list (if buffer-file-truename
	     (read-file-name "Write file: " buffer-file-truename)
	   (read-file-name "Write file: "
			   (cdr (assq 'default-directory
				      (buffer-local-variables)))
			   nil nil (buffer-name)))))
  (let ((coding-system-for-write 'iso-2022-7bit))
    (write-file (expand-file-name filename))))

(defun cgreek-write-file-wingreek (filename)
  "Write current buffer into file FILENAME in WinGreek charset.
Makes buffer visit that file, and marks it not modified.
If the buffer is already visiting a file, you can specify
a directory name as FILENAME, to write a file of the same
old name in that directory."
  (interactive
   (list (if buffer-file-truename
	     (read-file-name "Write file: " buffer-file-truename)
	   (read-file-name "Write file: "
			   (cdr (assq 'default-directory
				      (buffer-local-variables)))
			   nil nil (buffer-name)))))
  (let ((coding-system-for-write 'greek-WinGreek))
    (write-file (expand-file-name filename))))

(defun cgreek-write-file-tex (filename)
  "Write current buffer into file FILENAME in the TeX format
Makes buffer visit that file, and marks it not modified.
If the buffer is already visiting a file, you can specify
a directory name as FILENAME, to write a file of the same
old name in that directory."
  (interactive
   (list (if buffer-file-truename
	     (read-file-name "Write file: " buffer-file-truename)
	   (read-file-name "Write file: "
			   (cdr (assq 'default-directory
				      (buffer-local-variables)))
			   nil nil (buffer-name)))))
  (let ((coding-system-for-write cgreek-tex-coding-system))
    (write-file (expand-file-name filename))))

;;; latin-1 support

;;; Users may want to change these four TeX commands.

(defvar texcommand-feminine-ordinal-indicator
  "$\\mathrm{^{\\underline{a}}}$"
  "*TeX command to make `,A*(B'.")

(defvar texcommand-masculine-ordinal-indicator
  "$\\mathrm{^{\\underline{o}}}$"
  "*TeX command to make `,A:(B'.")

(defvar texcommand-left-pointing-guillemet "$\\ll$"
  "*TeX command to make `,A+(B'.")

(defvar texcommand-right-pointing-guillemet "$\\gg$"
  "*TeX command to make `,A;(B'.")

(defconst cgreek-latin1-to-tex-table
  [
   " "     "!`"    " "     " "     " "        " "        " "        "{\\S}"
   " "     " "     10      11      " "        " "        " "        " "
   " "     " "     " "     " "     " "        " "        "{\\P}"    " "
   " "     " "     26      27      " "        " "        " "        "?`"
   "\\`A"  "\\'A"  "\\^A"  "\\~A"  "\\\"A"    "{\\AA}"  "{\\AE}"    "\\c{C}"
   "\\`E"  "\\'E"  "\\^E"  "\\\"E" "\\`I"     "\\'I"     "\\^I"     "\\\"I"
   " "     "\\~N"  "\\`O"  "\\'O"  "\\^O"     "\\~O"     "\\\"O"    " "
   "{\\O}" "\\`U"  "\\'U"  "\\^U"  "\\\"U"    "\\'Y"     " "        "{\\ss}"
   "\\`a"  "\\'a"  "\\^a"  "\\~a"  "\\\"a"    "{\\aa}"   "{\\ae}"   "\\c{c}"
   "\\`e"  "\\'e"  "\\^e"  "\\\"e" "\\`{\\i}" "\\'{\\i}" "\\^{\\i}" "\\\"{\\i}"
   " "     "\\~n"  "\\`o"  "\\'o"  "\\^o"     "\\~o"     "\\\"o"    " "
   "{\\o}" "\\`u"  "\\'u"  "\\^u"  "\\\"u"    "\\'y"     " "        "\\\"y"
   ])

(defun latin1-to-tex-region (beg end)
  "Convert latin-1 characters in the region into TeX macros."
  (interactive "*r")
  (save-restriction
    (narrow-to-region beg end)
    (latin1-to-tex-buffer)))

(defun latin1-to-tex-buffer nil
  "Convert latin-1 characters in the current buffer into TeX macros."
  (interactive "*")
  (let ((case-fold-search nil)
	prev)

    ;; user may have changed these variables, so make it sure
    (aset cgreek-latin1-to-tex-table 10 texcommand-feminine-ordinal-indicator)
    (aset cgreek-latin1-to-tex-table 26 texcommand-masculine-ordinal-indicator)
    (aset cgreek-latin1-to-tex-table 11 texcommand-left-pointing-guillemet)
    (aset cgreek-latin1-to-tex-table 27 texcommand-right-pointing-guillemet)

    (goto-char (point-min))
    (while (re-search-forward "[,A!(B-,A(B]" nil t)
      (replace-match
       (aref cgreek-latin1-to-tex-table
	     (- (nth 1 (split-char (preceding-char))) 32))
       t t))))

(defun tex-to-latin1-region (beg end)
  "Convert TeX macros in the region into latin-1 characters."
  (interactive "*r")
  (save-restriction
    (narrow-to-region beg end)
    (tex-to-latin1-buffer)))

(defun tex-to-latin1-buffer nil
  "Convert TeX macros in the current buffer into latin-1 characters."
  (interactive "*")
  (let ((case-fold-search nil)
	ch1 ch2)

    ;; 1st step:  replacing {\AA}, {\o}, {\ss}, etc.
    (goto-char (point-min))
    (while
	(re-search-forward
	 "{\\\\\\(AA\\|AE\\|O\\|ss\\|aa\\|ae\\|o\\|S\\|P\\)}" nil t)
      (setq ch1 (char-before (match-end 1)))
      (replace-match
       (cond
	((= ch1 ?A) ",AE(B")
	((= ch1 ?E) ",AF(B")
	((= ch1 ?O) ",AX(B")
	((= ch1 ?s) ",A_(B")
	((= ch1 ?a) ",Ae(B")
	((= ch1 ?e) ",Af(B")
	((= ch1 ?o) ",Ax(B")
	((= ch1 ?S) ",A'(B")
	((= ch1 ?P) ",A6(B")
	(t ""))))

    ;; 2nd step: replacing \`a, \~n, \c{C}, \"{\i}, etc.
    (goto-char (point-min))
    (while (re-search-forward "\\\\[`'^~\"c][AEIOUYCNaeouycn{]" nil t)
      (setq ch1 (char-before (1- (point)))
	    ch2 (preceding-char))
      (cond
       ((= ch1 ?`)
	(cond
	 ((= ch2 ?A) (replace-match ",A@(B"))
	 ((= ch2 ?E) (replace-match ",AH(B"))
	 ((= ch2 ?I) (replace-match ",AL(B"))
	 ((= ch2 ?O) (replace-match ",AR(B"))
	 ((= ch2 ?U) (replace-match ",AY(B"))
	 ((= ch2 ?a) (replace-match ",A`(B"))
	 ((= ch2 ?e) (replace-match ",Ah(B"))
	 ((= ch2 ?o) (replace-match ",Ar(B"))
	 ((= ch2 ?u) (replace-match ",Ay(B"))
	 ((and (= ch2 ?{) (looking-at "\\\\i}")) ; \`{-!-\i}
	  (goto-char (match-end 0))	; \`{\i}-!-
	  (delete-backward-char 6)
	  (insert ",Al(B"))
	 (t nil)))
       ((= ch1 ?')
	(cond
	 ((= ch2 ?A) (replace-match ",AA(B"))
	 ((= ch2 ?E) (replace-match ",AI(B"))
	 ((= ch2 ?I) (replace-match ",AM(B"))
	 ((= ch2 ?O) (replace-match ",AS(B"))
	 ((= ch2 ?U) (replace-match ",AZ(B"))
	 ((= ch2 ?Y) (replace-match ",A](B"))
	 ((= ch2 ?a) (replace-match ",Aa(B"))
	 ((= ch2 ?e) (replace-match ",Ai(B"))
	 ((= ch2 ?o) (replace-match ",As(B"))
	 ((= ch2 ?u) (replace-match ",Az(B"))
	 ((= ch2 ?y) (replace-match ",A}(B"))
	 ((and (= ch2 ?{) (looking-at "\\\\i}"))
	  (goto-char (match-end 0))
	  (delete-backward-char 6)
	  (insert ",Am(B"))
	 (t nil)))
       ((= ch1 ?^)
	(cond
	 ((= ch2 ?A) (replace-match ",AB(B"))
	 ((= ch2 ?E) (replace-match ",AJ(B"))
	 ((= ch2 ?I) (replace-match ",AN(B"))
	 ((= ch2 ?O) (replace-match ",AT(B"))
	 ((= ch2 ?U) (replace-match ",A[(B"))
	 ((= ch2 ?a) (replace-match ",Ab(B"))
	 ((= ch2 ?e) (replace-match ",Aj(B"))
	 ((= ch2 ?o) (replace-match ",At(B"))
	 ((= ch2 ?u) (replace-match ",A{(B"))
	 ((and (= ch2 ?{) (looking-at "\\\\i}"))
	  (goto-char (match-end 0))
	  (delete-backward-char 6)
	  (insert ",An(B"))
	 (t nil)))
       ((= ch1 ?~)
	(cond
	 ((= ch2 ?A) (replace-match ",AC(B"))
	 ((= ch2 ?O) (replace-match ",AU(B"))
	 ((= ch2 ?N) (replace-match ",AQ(B"))
	 ((= ch2 ?a) (replace-match ",Ac(B"))
	 ((= ch2 ?o) (replace-match ",Au(B"))
	 ((= ch2 ?n) (replace-match ",Aq(B"))
	 (t nil)))
       ((= ch1 ?\")
	(cond
	 ((= ch2 ?A) (replace-match ",AD(B"))
	 ((= ch2 ?E) (replace-match ",AK(B"))
	 ((= ch2 ?I) (replace-match ",AO(B"))
	 ((= ch2 ?O) (replace-match ",AV(B"))
	 ((= ch2 ?U) (replace-match ",A\(B"))
	 ((= ch2 ?a) (replace-match ",Ad(B"))
	 ((= ch2 ?e) (replace-match ",Ak(B"))
	 ((= ch2 ?o) (replace-match ",Av(B"))
	 ((= ch2 ?u) (replace-match ",A|(B"))
	 ((= ch2 ?y) (replace-match ",A(B"))
	 ((and (= ch2 ?{) (looking-at "\\\\i}"))
	  (goto-char (match-end 0))
	  (delete-backward-char 6)
	  (insert ",Ao(B"))
	 (t nil)))
       (t				; ch1 == ?c
	(cond
	 ((and (= ch2 ?{) (looking-at "C}"))
	  (goto-char (match-end 0))
	  (delete-backward-char 5)
	  (insert ",AG(B"))
	 ((and (= ch2 ?{) (looking-at "c}"))
	  (goto-char (match-end 0))
	  (delete-backward-char 5)
	  (insert ",Ag(B"))
	 (t nil)))))

    ;; 3rd step: replacing !` and ?`
    (goto-char (point-min))
    (while (re-search-forward "[!\\?]`" nil t)
      (replace-match (if (= (char-before (1- (point))) ?!) ",A!(B" ",A?(B")))

    ;; 4th step: replacing feminine/masculine ordinal indicators
    (goto-char (point-min))
    (while (search-forward texcommand-feminine-ordinal-indicator nil t)
      (replace-match ",A*(B"))

    (goto-char (point-min))
    (while (search-forward texcommand-masculine-ordinal-indicator nil t)
      (replace-match ",A:(B"))

    ;; 5th step: replacing guillemets
    (goto-char (point-min))
    (while (search-forward texcommand-left-pointing-guillemet nil t)
      (replace-match ",A+(B"))

    (goto-char (point-min))
    (while (search-forward texcommand-right-pointing-guillemet nil t)
      (replace-match ",A;(B"))
    ))

;;; Shortcuts

(defun tex-to-cgreek-latin1-region (beg end)
  (interactive "*r")
  (save-excursion
    (save-restriction
      (narrow-to-region beg end)
      (tex-to-cgreek-buffer)
      (tex-to-latin1-buffer))))

(defun tex-to-cgreek-latin1-buffer nil
  (interactive "*")
  (tex-to-cgreek-buffer)
  (tex-to-latin1-buffer))

;;; TLG conversion

(defvar cgreek-tlg-directory
  (if (memq system-type' (ms-dos windows-nt)) "d:/" "/cdrom/")
  "*Directory in which TLG files reside.
Do not forget a directory separator at the end.")

(defvar cgreek-tlg-cdrom-version "E"
  "*Set this variable to \"D\" to avoid the error in Disk D.")

(defvar cgreek-tlg-my-authtab-file nil
  "*Absolute pathname to a file containing a customised author list.
Each line in the file begins with the string \"TLG\" directly followed
by four digits.  Rest of the line can be anything.")

(defvar cgreek-tlg2emacs-program
  (expand-file-name
   (if (featurep 'meadow)
       "~/cgreek-meadow/tlg/tlg2emacs"
     "~/cgreek-emacs20/tlg/tlg2emacs"))
  "*Program name for invoking a TLG-to-CGreek converter.")

(if (null (assoc cgreek-tlg2emacs-program process-coding-system-alist))
    (setq process-coding-system-alist
	  (cons (cons cgreek-tlg2emacs-program
		      (cons 'emacs-mule 'no-conversion))
		process-coding-system-alist)))

(defvar cgreek-tlg2citation-program
  (expand-file-name
   (if (featurep 'meadow)
       "~/cgreek-meadow/tlg/tlg2cit"
     "~/cgreek-emacs20/tlg/tlg2cit"))
  "*Program name to extract citation information from tlg txt file.")

(if (null (assoc cgreek-tlg2citation-program process-coding-system-alist))
    (setq process-coding-system-alist
	  (cons (cons cgreek-tlg2citation-program
		      (cons 'no-conversion 'no-conversion))
		process-coding-system-alist)))

(defconst cgreek-tlg-end-of-file 0
  "Type code for end-of-file in TLG IDT file.")

(defconst cgreek-tlg-new-author 1
  "Type code for `new author' in TLG IDT file.")

(defconst cgreek-tlg-new-work 2
  "Type code for `new work' in TLG IDT file.")

(defconst cgreek-tlg-author-work 16
  "Type code for `description of ID fields a..b' in TLG IDT file.
If followed by numeric value 0, means ID level a; if 1, means b.")

(defconst cgreek-tlg-citation 17
  "Type code for `description of ID fields n, v..z' in TLG IDT file.
In case of document, followed by numeric value 0 and means ID level n.
Otherwise, followed by 4..0 meaning v..z, respectively.")

(defconst cgreek-tlg-citation-regexp
  "\\([^\t]*\\)\t\\([^\t]*\\)\t\\([^\t]*\\)\t\\([^\t]*\\)\t\\([^\t]*\\)\t\\([^\t]*\\)\t"
  "Regular expression used to get citation ID value in a tlg file.
There are 6 strings separated by tab, of which
(match-string 1) corresponds to level n citation,
(match-string 2) corresponds to level v citation,
(match-string 3) corresponds to level w citation,
(match-string 4) corresponds to level x citation,
(match-string 5) corresponds to level y citation and
(match-string 6) corresponds to level z citation.")

(defvar cgreek-tlg-authtab-buffer nil
  "Buffer used for selecting an author.")

(defvar cgreek-tlg-temporary-file
  (expand-file-name "cgreek-tlg-temporary-file" temporary-file-directory)
  "Temporary file for citation extraction.")

(defvar cgreek-tlg-temporary-buffer
  (get-buffer-create " cgreek-tlg-temporary-buffer")
  "Temporary file for beta code conversion.")

(defvar cgreek-tlg-author-name nil
  "Athor name of the work displayed in the current buffer.")

(make-variable-buffer-local 'cgreek-tlg-author-name)

(defvar cgreek-tlg-work-name nil
  "Work name displayed in the current buffer.")

(make-variable-buffer-local 'cgreek-tlg-work-name)

(defvar cgreek-tlg-citation-buffer nil
  "Buffer containing the citation information of the current buffer.")

(make-variable-buffer-local 'cgreek-tlg-citation-buffer)

(defvar cgreek-tlg-citation-description nil
  "Citation descriptions of the current buffer.
Plist of the form (level-1 desc-1 level-2 desc-2 ...) where
level-x is integer from 2 (level v) to 6 (level z or n) and
desc-x is string like \"line\", \"section\", etc.")

(make-variable-buffer-local 'cgreek-tlg-citation-description)

(defvar cgreek-tlg-idt-buffer nil
  "Buffer containing the work list of the same author.")

(make-variable-buffer-local 'cgreek-tlg-idt-buffer)

(defun cgreek-tlg-parse-authtab ()
  "Parse and display the \"authtab.dir\" file.
The parsed result is shown in the *TLG authtab* buffer.
Hitting RET opens the file for that author."
  (interactive)
  (cond

   ;; already created
   ((buffer-live-p cgreek-tlg-authtab-buffer)
    (switch-to-buffer cgreek-tlg-authtab-buffer))

   ;; customised file is set
   (cgreek-tlg-my-authtab-file
    (find-file cgreek-tlg-my-authtab-file)

    ;; a little make-up
    (goto-char (point-min))
    (while (looking-at "TLG[0-9][0-9][0-9][0-9]")
      (forward-char 7)
      (if (looking-at "[ \t]+")
	  (goto-char (match-end 0)))
      (add-text-properties
       (point)
       (progn (end-of-line) (point))
       '(mouse-face highlight))
      (beginning-of-line 2))		; go to next line's beginning

    ;; local bindings
    (local-set-key "q" '(lambda () (interactive) (kill-buffer nil)))
    (local-set-key [mouse-2] 'cgreek-tlg-parse-idt-mouse)
    (local-set-key [RET] 'cgreek-tlg-parse-idt-keyboard)
    (local-set-key "\C-m" 'cgreek-tlg-parse-idt-keyboard)

    (goto-char (point-min))
    (set-buffer-modified-p nil)
    (setq cgreek-tlg-authtab-buffer (current-buffer)))

   ;; newly create
   (t
    (let ((raw (generate-new-buffer "authtab.dir"))
	  (cooked (get-buffer-create "*TLG authtab*"))
	  (str nil))
      (message "Reading authtab ... ")

      ;; initialise temporary buffer
      (set-buffer cgreek-tlg-temporary-buffer)
      (erase-buffer)

      ;; initialise cooked (user visible) authtab
      (set-buffer cooked)
      (setq buffer-read-only nil)
      (erase-buffer)

      ;; prepare the original file contents
      (set-buffer raw)
      (set-buffer-multibyte nil)
      (insert-file-contents-literally
       (concat cgreek-tlg-directory "authtab.dir"))
      (re-search-forward ".TLG....[^\xff]+\xff") ; skip header

      (while (not (looking-at "\*END[\0\xff]+"))
	(cond

	 ;; plain text
	 ((or (looking-at "TLG[0-9][0-9][0-9][0-9] ")
	      (looking-at "DOCCAN[0-9]+ "))
	  (setq str (buffer-substring (match-beginning 0) (match-end 0)))
	  (goto-char (match-end 0)))

	 ;; Beta codes go to cgreek-tlg-temporary-buffer,
	 ;; separated by \0, for later conversion.
	 ;; For the mean time \0 is inserted in cooked as position marker.
	 ((looking-at "[\x20-\x7e]+")
	  (save-excursion
	    (set-buffer cgreek-tlg-temporary-buffer)
	    (insert-buffer-substring raw (match-beginning 0) (match-end 0))
	    (insert 0))
	  (setq str (char-to-string 0))
	  (goto-char (match-end 0)))

	 ;; special codes
	 ((= (following-char) ?\x80)
	  (setq str ", AKA ")
	  (forward-char 1))

	 ((= (following-char) ?\x81)
	  (setq str ". Remarks: ")
	  (forward-char 2))

	 ((= (following-char) ?\x82)
	  (setq str ". Size: ")
	  (forward-char 2))

	 ((= (following-char) ?\x83)
	  (setq str
		(concat ". Language: "
			(char-to-string (char-after (1+ (point))))))
	  (forward-char 2))

	 ;; field separator
	 ((looking-at "[\xff]+")
	  (setq str "\n")
	  (goto-char (match-end 0)))

	 (t
	  (error "Unexpected code %c at %d" (following-char) (point))))

	;; insert cooked string
	(save-excursion
	  (set-buffer cooked)
	  (insert str)))

      ;; doccan files are not supported yet
      (let ((kill-whole-line t))
	(goto-char (point-min))
	(while (re-search-forward "^DOCCAN[0-9]+ " nil t)
	  (beginning-of-line)
	  (kill-line)))

      ;; convert beta code in cgreek-tlg-temporary-buffer
      ;; note that \0 becomes \n
      (set-buffer cgreek-tlg-temporary-buffer)
      (call-process-region
       (point-min) (point-max)
       cgreek-tlg2emacs-program t t nil
       "-l" (format "%d" (charset-id 'cgreek)))
      (goto-char (point-min))

      ;; replace each \0 in cooked with a line in cgreek-tlg-temporary-buffer
      (set-buffer cooked)
      (goto-char (point-min))
      (let (start end)
	(while (search-forward (char-to-string 0) nil t)
	  (save-excursion
	    (set-buffer cgreek-tlg-temporary-buffer)
	    (setq start (point))
	    (end-of-line)
	    (setq end (point))
	    (forward-char 1))		; go to next line's beginning
	  (replace-match "")
	  (add-text-properties
	   (point)
	   (progn
	     (insert-buffer-substring cgreek-tlg-temporary-buffer start end)
	     (point))
	   '(mouse-face highlight))))

      (goto-char (point-min))

      ;; local bindings
      (local-set-key "q" '(lambda () (interactive) (kill-buffer nil)))
      (local-set-key [mouse-2] 'cgreek-tlg-parse-idt-mouse)
      (local-set-key [RET] 'cgreek-tlg-parse-idt-keyboard)
      (local-set-key "\C-m" 'cgreek-tlg-parse-idt-keyboard)
      (set-buffer-modified-p nil)
      (setq buffer-read-only t)

      ;; clean up
      (kill-buffer raw)
      (switch-to-buffer cooked)
      (setq cgreek-tlg-authtab-buffer (current-buffer))
      (message nil)))))

(defun cgreek-tlg-parse-idt-mouse (click)
  "Wrapper of cgreek-tlg-parse-idt for interactive use.
In the \"*TLG authtab*\" buffer, open line-initial tlg???? file."
  (interactive "e")
  (let* ((start (event-start click))
	 (window (car start))
	 (pos (car (cdr start))))
    (select-window window)
    (goto-char pos))
  (cgreek-tlg-parse-idt-keyboard))

(defun cgreek-tlg-parse-idt-keyboard ()
  "Wrapper of cgreek-tlg-parse-idt for interactive use.
In the \"*TLG authtab*\" buffer, open line-initial tlg???? file."
  (interactive)

  (cond
   ((eq (current-buffer) cgreek-tlg-authtab-buffer)
    (beginning-of-line)
    (if (not (looking-at "TLG[0-9][0-9][0-9][0-9]"))
	(message "N/A for this file")
      (switch-to-buffer
       (cgreek-tlg-parse-idt
	(concat
	 (downcase (buffer-substring (match-beginning 0) (match-end 0)))
	 ".idt")))))

   (cgreek-tlg-idt-buffer
    (switch-to-buffer
     (cgreek-tlg-parse-idt
      (substring cgreek-tlg-idt-buffer
		 1 (1- (length cgreek-tlg-idt-buffer))))))

   (t					; should never happen
    (error "cgreek-tlg-parse-idt-keyboard called from an unexpected buffer"))))

(defun cgreek-tlg-parse-idt (filename)
  "Parse the TLG IDT file FILENAME.
The result is stored in a buffer, which will be the return value."
  (let ((raw nil)
	(cooked (get-buffer-create (concat "*" filename "*"))))

    (set-buffer cooked)
    (if (/= (point-min) (point-max))

	;; OK, already created.  Just return it.
	cooked

      ;; not created yet.  Make it.
      (message "Reading %s ..." filename)

      ;; probability of "cooked being empty but read-only" is not zero
      (setq buffer-read-only nil)

      ;; initialise temporary buffer
      (set-buffer cgreek-tlg-temporary-buffer)
      (erase-buffer)

      ;; prepare the original file contents
      (setq raw (generate-new-buffer filename))
      (set-buffer raw)
      (set-buffer-multibyte nil)
      (insert-file-contents-literally
       (concat cgreek-tlg-directory filename))

      ;; extract citation description from FILENAME

      ;; Beta code goes to cgreek-tlg-temporary-buffer,
      ;; separated by \0, for later conversion.
      ;; For the mean time \0 is inserted in cooked as position marker
      (while (eq (char-after (point)) cgreek-tlg-new-author)
	(cgreek-tlg-parse-author cooked))
      (if (/= (char-after (point)) cgreek-tlg-end-of-file)
	  (error "Unexpected type code %s at %d"
		 (buffer-file-name)
		 (point)))

      ;; convert beta code in cgreek-tlg-temporary-buffer
      ;; note that \0 becomes \n
      (set-buffer cgreek-tlg-temporary-buffer)
      (call-process-region
       (point-min) (point-max)
       cgreek-tlg2emacs-program t t nil
       "-l" (format "%d" (charset-id 'cgreek)))
      (goto-char (point-min))

      ;; replace each \0 in cooked with a line in cgreek-tlg-temporary-buffer
      ;; do not forget to inherit text properties
      (set-buffer cooked)
      (goto-char (point-min))
      (let (start end prop)
	(while (search-forward (char-to-string 0) nil t)
	  (setq prop (text-properties-at (match-beginning 0)))
	  (replace-match "")
	  (save-excursion
	    (set-buffer cgreek-tlg-temporary-buffer)
	    (setq start (point))
	    (end-of-line)
	    (setq end (point))
	    (forward-char 1))		; go to next line's beginning
	  (add-text-properties
	   (point)
	   (progn
	     (insert-buffer-substring cgreek-tlg-temporary-buffer start end)
	     (point))
	   prop)))

      ;; local bindings
      (local-set-key "q" '(lambda () (interactive) (kill-buffer nil)))
      (local-set-key "a" 'cgreek-tlg-parse-authtab)
      (local-set-key [mouse-2] 'cgreek-tlg-parse-txt-mouse)
      (local-set-key [RET] 'cgreek-tlg-parse-txt-keyboard)
      (local-set-key "\C-m" 'cgreek-tlg-parse-txt-keyboard)
      (goto-char (point-min))
      (set-buffer-modified-p nil)
      (setq buffer-read-only t)

      ;; clean up
      (kill-buffer raw)
      (message nil)

      cooked)))

(defun cgreek-tlg-parse-author (cooked)
  "Process an author entry in TLG IDT file."

  (let (id desc next-author)

    ;; should be looking at:
    ;; ?\x01		type code 1 marks new author
    ;; [2 bytes]	length of the author section in bytes
    ;; [2 bytes]	author location in blocks
    ;; ?\xef		left nibble = 1110 : escape code
    ;;			right nibble = 1111 : ASCII string
    ;; ?\x80		escape level = 0 (author)
    ;; [n bytes]	ID value in string (with MSB on)
    ;; ?\xff		ASCII string terminator

    ;; just make sure
    (if (or (/= (char-after (+ (point) 5)) ?\xef)
	    (/= (char-after (+ (point) 6)) ?\x80))
      (error "Unexpected author ID value introducer %02x%02x"
	     (char-after (+ (point) 5))
	     (char-after (+ (point) 6))))

    (setq next-author
	  (+ (point)
	     (* 256 (char-after (1+ (point))))
	     (char-after (+ 2 (point)))))
    (forward-char 7)
    (setq id (cgreek-tlg-read-string))

    ;; now should be looking at:
    ;; ?\x10		type code 16 is author/work description
    ;; ?\x00		level = 0 (author)
    ;; [1 byte]		length of the following description
    ;; [n bytes]	author description

    ;; just make sure
    (if (or (/= (following-char) ?\x10)
	    (/= (char-after (1+ (point))) ?\x00))
      (error "Unexpected author description introducer %02x%02x"
	     (following-char)
	     (char-after (1+ (point)))))

    (forward-char 3)
    (setq desc (cgreek-tlg-read-string (preceding-char)))

    ;; print out author description
    ;; author id value is given as property
    (save-excursion
      (set-buffer cooked)
      (if (/= (point) (point-min))
	  (insert "\n"))
      (add-text-properties
       (point)
       (progn (insert (format "%s\n" desc)) (point))
       (list 'tlg-author-id id)))

    ;; process each work written by this author
    (while (= (following-char) cgreek-tlg-new-work)
      (cgreek-tlg-parse-work cooked))

    (goto-char next-author)))

(defun cgreek-tlg-parse-work (cooked)
  "Process a work entry in TLG IDT file."

  (let (id desc next-work block cit-desc plist)

    ;; should be looking at:
    ;; ?\x02		type code 2 marks new work
    ;; [2 bytes]	length of the work section in bytes
    ;; [2 bytes]	work location in blocks
    ;; ?\xef		left nibble = 1110 : escape code
    ;;			right nibble = 1111 : ASCII string
    ;; ?\x81		escape level = 1 (work)
    ;; [n bytes]	ID value in string (with MSB on)
    ;; ?\xff		ASCII string terminator

    ;; just make sure
    (if (or (/= (char-after (+ (point) 5)) ?\xef)
	    (/= (char-after (+ (point) 6)) ?\x81))
	(error "Unexpected work ID value introducer %02x%02x"
	       (char-after (+ (point) 5))
	       (char-after (+ (point) 6))))
    
    (setq next-work
	  (+ (point)
	     (* (char-after (1+ (point))) 256)
	     (char-after (+ (point) 2))))
    (setq block
	  (+ (* (char-after (+ (point) 3)) 256)
	     (char-after (+ (point) 4))))
    (forward-char 7)
    (setq id (cgreek-tlg-read-string))

    ;; now should be looking at:
    ;; ?\x10		type code 16 is author/work description
    ;; ?\x01		level = 1 (work)
    ;; [1 byte]		length of the following description
    ;; [n bytes]	work description

    ;; just make sure
    (if (or (/= (following-char) ?\x10)
	    (/= (char-after (1+ (point))) ?\x01))
	(error "Unexpected work description introducer %02x%02x"
	       (following-char)
	       (char-after (1+ (point)))))

    (forward-char 3)
    (setq desc (cgreek-tlg-read-string (preceding-char)))
    
    (if (= (following-char) cgreek-tlg-citation)

	;; citation descriptions are given for this work; set them
	(while (= (following-char) cgreek-tlg-citation)
	  (setq cit-desc (append (cgreek-tlg-parse-citation) cit-desc)))

      ;; citation descriptions are not given for this work;
      ;; inherit previous work's ones
      (save-excursion
	(set-buffer cooked)
	(save-excursion
	  (previous-line 1)
	  (setq cit-desc
		(get-text-property (point) 'tlg-citation-description)))))

    ;; set property list
    (setq plist
	  (list 'tlg-work-id id
		'tlg-citation-description cit-desc
		'tlg-starting-block block
		'mouse-face 'highlight))

    ;; special consideration for Bekker pages
    ;; Bekker page is, if present, always at level y (level 5)
    (when (and (string= (plist-get cit-desc 5) "Bekker page")
	       (not (string= id "052"))
	       (not (string= id "053")))
      (setq plist
	    (append
	     (list 'tlg-first-bekker-page
		   (cgreek-tlg-get-bekker-page))
	     plist)))

    ;; print out work description
    ;; necessary information is given as properties
    (save-excursion
      (set-buffer cooked)
      (add-text-properties
       (point)
       (progn (insert (format "\t%s" desc)) (point))
       plist)
      (insert "\n"))

    ;; jump to the starting point of the next work and return
    (goto-char next-work)))

(defun cgreek-tlg-parse-citation ()
  "Process a citation description entry in TLG IDT file.
Return a list of the form (LEVEL DESCRIPTION).
LEVEL is an integer varying from 1 (level v) to 5 (level z, or n).
DESCRIPTION is a string."

  (let (level length)

  ;; should be looking at cgreek-tlg-citation, followed by 1-byte
  ;; level, followed by 1-byte length
  ;; ?\x11		type code 17 is citation description
  ;; [1 byte]		level (0 = n or z, 1 = y, 2 = x, 3 = w, 4 = v)
  ;; [1 byte]		length of the following description
  ;; [n bytes]		citation description
  
    (setq level (- 6 (char-after (1+ (point)))) ; 2 = v, ... , 6 = z or n
	  length (char-after (+ (point) 2)))
    (list level
	  (buffer-substring
	   (+ (point) 3)
	   (progn (goto-char (+ (point) 3 length)) (point))))))

(defun cgreek-tlg-read-string (&optional length)
  "Return TLG string starting at point.  Point is moved.

Without the optional argument LENGTH, read up to \xff, stripping each MSB.

With the optional argument LENGTH, read that long beta code and return \0.
The beta code read will be stored in cgreek-tlg-temporary-buffer
for later conversion.
Each beta code fragment is separated by \0."

  (if length

      (let ((str (buffer-substring (point) (+ (point) length))))
	(forward-char length)
	(save-excursion
	  (set-buffer cgreek-tlg-temporary-buffer)

	  ;; work around for the error in Disk D
	  (if (string= cgreek-tlg-cdrom-version "D")
	      (insert "&"))

	  (insert str 0))
	(char-to-string 0))

    (let ((p (point))
	  (i 0)
	  str)
      (search-forward (char-to-string ?\xff))
      (setq str (buffer-substring p (1- (point))))
      (while (< i (length str))
	(aset str i (- (aref str i) 128))
	(setq i (1+ i)))
      str)))

(defun cgreek-tlg-parse-txt-mouse (click)
  "Invoke cgreek-tlg-parse-txt with parameters given to the clicked line."
  (interactive "e")
  (let* ((start (event-start click))
	 (window (car start))
	 (pos (car (cdr start))))
    (select-window window)
    (goto-char pos))
  (cgreek-tlg-parse-txt-keyboard))

(defun cgreek-tlg-parse-txt-keyboard ()
  "Invoke cgreek-tlg-parse-txt with parameters given to the current line."
  (interactive)
  (if (get-text-property (point) 'tlg-work-id) ; on a valid line?

      (let (author-id author-name (cit (current-buffer)))

	;; get author-id and author-name given as text properties
	;; work name in a line is preceded by a tab; author name is not
	(save-excursion
	  (re-search-backward "^[^\t]")	; should be found
	  (setq author-id (get-text-property (point) 'tlg-author-id))
	  (setq author-name
		(buffer-substring-no-properties
		 (progn (beginning-of-line) (point))
		 (progn (end-of-line) (point)))))

	;; parse and display
	(switch-to-buffer
	 (cgreek-tlg-parse-txt
	  (buffer-name)
	  author-id
	  author-name
	  (get-text-property (point) 'tlg-work-id)
	  (save-excursion
	    (buffer-substring-no-properties ; this is work-name
	     (progn (beginning-of-line)
		    (1+ (point)))	; skipping line-initail tab
	     (progn (end-of-line) (point))))
	  (get-text-property (point) 'tlg-citation-description)
	  (get-text-property (point) 'tlg-starting-block))))))

(defun cgreek-tlg-parse-txt (idt-buffer author-id author-name work-id work-name
				     cit-desc &optional block)
  "Parse the work WORK-ID of the author AUTHOR-ID from a tlg txt file.

The tlg txt file name \"/.../tlg????.txt\" is derived from IDT-BUFFER,
which is of the form \"*tlg????.idt*\".

Both AUTHOR-ID and WORK-ID are strings although they consist of digits.

AUTHOR-NAME and WORK-NAME are descriptive string.

CIT-DESC is a plist of the form (level-1 desc-1 level-2 desc-2 ...) where
level-x is an integer from 1 (level v) to 5 (level z or n) and
desc-x is a string like \"line\", \"section\", etc.

BLOCK is an integer representing the starting block of WORK-ID.

Return the buffer in which parsed result is contained."

  (let ((txt (get-buffer work-name))
	(cit (get-buffer (concat work-name " citation"))))

    ;; Am I already created?
    (if (and txt
	     cit
	     (save-excursion
	       (set-buffer txt)
	       (and
		(string= author-name cgreek-tlg-author-name)
		(string= work-name cgreek-tlg-work-name))))

	;; OK, already created
	txt

      ;; no, not created yet
      (message "Reading %s ..." work-name)
      (setq txt (generate-new-buffer work-name))
      (setq cit (generate-new-buffer (concat work-name " citation")))

      ;; cgreek-tlg2citation-program writes citation information to stdout
      ;; and beta code text to stderr
      (call-process
       cgreek-tlg2citation-program
       (concat cgreek-tlg-directory
	       (substring idt-buffer 1 (- (length idt-buffer) 4))
	       "txt")			; *tlg????.idt* -> /.../tlg????.txt
       (list cit cgreek-tlg-temporary-file)
       nil
       "-a" author-id "-w" work-id "-s" (format "%d" block))

      ;; beta code text just written is decoded and inserted into txt
      (call-process
       cgreek-tlg2emacs-program
       cgreek-tlg-temporary-file
       txt
       nil
       "-l" (format "%d" (charset-id 'cgreek)))

      ;; buffer local bindings
      (set-buffer txt)
      (setq cgreek-tlg-author-name author-name
	    cgreek-tlg-work-name work-name
	    cgreek-tlg-citation-buffer cit
	    cgreek-tlg-citation-description cit-desc
	    cgreek-tlg-idt-buffer idt-buffer)
      (local-set-key "q" '(lambda () (interactive) (kill-buffer nil)))
      (local-set-key "a" 'cgreek-tlg-parse-authtab)
      (local-set-key "w" 'cgreek-tlg-parse-idt-keyboard)
      (local-set-key "j" 'cgreek-tlg-jump)
      (make-local-hook 'post-command-hook)
      (add-hook 'post-command-hook
		'cgreek-tlg-show-citation
		nil t)
      (make-local-hook 'kill-buffer-hook)
      (add-hook 'kill-buffer-hook
		'(lambda ()
		   (if (buffer-live-p cgreek-tlg-citation-buffer)
		       (kill-buffer cgreek-tlg-citation-buffer))
		   (let ((win (get-buffer-window "*TLG citation*" t)))
		     (if (window-live-p win)
			 (save-selected-window
			   (select-window win)
			   (if (not (one-window-p t))
			       (delete-window win)))))
		   (if (buffer-live-p (get-buffer "*TLG citation*"))
		       (kill-buffer "*TLG citation*")))
		nil t)

      ;; It is better to remove the final newline so that the cursor
      ;; always stays on a `real' line.
      (goto-char (point-max))
      (if (= (preceding-char) ?\n)
	  (delete-backward-char 1))

      (goto-char (point-min))
      (set-buffer-modified-p nil)
      (setq buffer-read-only t)
      (message nil)

      ;; return value
      txt)))

;; why is this not supplied by default?
(defun current-line ()
  "Return current line, beginning with 1 (like in goto-line)."
  (+ (count-lines (point-min) (point))
     (if (= (current-column) 0) 1 0)))

(defun cgreek-tlg-show-citation ()
  "Display current buffer's tlg citation in buffer \"*TLG citation*\""

  (let ((author cgreek-tlg-author-name)
	(work cgreek-tlg-work-name)
	(cit-desc cgreek-tlg-citation-description)
	(mesg nil)
	(line (current-line)))

    ;; AUTHOR should be non-nil if called from a tlg txt buffer
    (when author

      (save-excursion
	(set-buffer cgreek-tlg-citation-buffer) ; current buffer's citation
	(goto-line line)
	(beginning-of-line)

	(when (looking-at cgreek-tlg-citation-regexp)

	  ;; (match-string 1) = level n citation ID value
	  ;; (match-string 2) = level v citation ID value
	  ;; (match-string 3) = level w citation ID value
	  ;; (match-string 4) = level x citation ID value
	  ;; (match-string 5) = level y citation ID value
	  ;; (match-string 6) = level z citation ID value

	  (if (or (string= (match-string 1) "")
		  (null (match-string 1)))

	      ;; no level n, thus this work is hierarchical.
	      ;; level z is always present.  level v..y may or may not.
	      (setq mesg
		    (mapcar
		     '(lambda (match)
			(if (or (string= (match-string match) "")
				(null (match-string match)))
			    ""
			  (format "%s %s   "
				  (plist-get cit-desc match)
				  (match-string match))))
		     '(2 3 4 5 6)))

	    ;; level n is present, thus this work is a document.
	    ;; level v..z may exist, buf their description is not available.
	    (setq mesg
		  (mapcar
		   '(lambda (match)
		      (if (or (string= (match-string match) "")
			      (null (match-string match)))
			  ""	
			(format "%s   " (match-string match))))
		   '(2 3 4 5 6)))

	    ;; only level n has description, but it locates at level 6!
	    (setq mesg
		  (cons (format "%s "
				(plist-get cit-desc 6))
			(cons (format "%s   " (match-string 1))
			      mesg))))

	  ;; display citation
	  (set-buffer (get-buffer-create "*TLG citation*"))
	  (erase-buffer)
	  (setq truncate-lines t)
	  (insert author "\n" work "\n" (apply 'concat mesg))
	  (unless (get-buffer-window (current-buffer) 'visible)
	    (switch-to-buffer-other-window (current-buffer))
	    (shrink-window-if-larger-than-buffer (selected-window)))))

      ;; put cursor back to the txt buffer
      (select-window (get-buffer-window (current-buffer))))))

(defun cgreek-tlg-jump ()
  "Jump to the specified position in the current tlg txt buffer"
  (interactive)
  (let ((txt-buf (current-buffer))
	(cit-buf cgreek-tlg-citation-buffer)
	(cit-desc cgreek-tlg-citation-description)
	level value matched desc line)

    ;; this function is meaningful only in a tlg txt buffer,
    ;; in that case cit-desc should be non-nil
    (when cit-desc

      (set-buffer cit-buf)
      (goto-char (point-min))
      (looking-at cgreek-tlg-citation-regexp)
      (if (string= (match-string 1) "")	; 1 means level n

	  ;; level n id does not exist, that means this is a hierarchical text
	  (progn

	    (setq level 2)		; 2 means level v
	    (while (<= level 6)		; 6 means level z
	      (setq desc (plist-get cit-desc level))

	      ;; prompt only for levels with description
	      (when desc

		;; read id
		(setq value (read-from-minibuffer (concat desc ": ")))
		(if (string= value "")
		    (error ""))

		;; start searching
		(setq matched nil)
		(while (not matched)
		  (looking-at cgreek-tlg-citation-regexp)
		  (if (string= (match-string level) value)
		      (setq matched t)
		    (if (/= (forward-line 1) 0)
			(setq matched 'eob))))

		;; special handling for Bekker page even not found in this work
		(when (and (eq matched 'eob)
			   (string= desc "Bekker page"))
		  (cgreek-tlg-open-bekker-page value)
		  (setq txt-buf (current-buffer)
			cit-buf cgreek-tlg-citation-buffer
			cit-desc cgreek-tlg-citation-description)
		  (set-buffer cit-buf)
		  (goto-char (point-min))
			  
		  ;; search again
		  (setq matched nil)
		  (while (not matched)
		    (looking-at cgreek-tlg-citation-regexp)
		    (if (string= (match-string level) value)
			(setq matched t)
		      (if (/= (forward-line 1) 0)
			  (setq matched 'eob)))))

		;; really found?
		(if (eq matched 'eob)

		  (error "Invalid value for this work")

		  ;; really found!
		  (setq line (current-line))
		  (switch-to-buffer txt-buf) ; show it to the user
		  (goto-line line)
		  (recenter 0)
		  (cgreek-tlg-show-citation)
		  (set-buffer cit-buf)))

	      (setq level (1+ level))))

	;; level n id exist, thus this is a `document' (non-hierarchical)
	(setq value
	      (read-from-minibuffer
	       ;; level n DESCRIPTION is tagged as 6, but ... (see below)
	       (concat (plist-get cit-desc 6) ": ")))
	(if (string= value "")
	    (error ""))

	;; start searching
	(setq matched nil)
	(while (not matched)
	  (looking-at cgreek-tlg-citation-regexp)
	  (if (string= (match-string 1) value) ; level n CITATION id is at 1!
	      (setq matched t)
	    (if (/= (forward-line 1) 0)
		(setq matched 'eob))))

	;; really found?
	(if (eq matched 'eob)
	    (error "Invalid value for this work")

	  (setq line (current-line))
	  (switch-to-buffer txt-buf)	; show it to the user
	  (goto-line line)
	  (recenter 0)
	  (cgreek-tlg-show-citation))))))

(defun cgreek-tlg-open-bekker-page (page)
  "Open the work in which Bekker page PAGE resides."
  (let ((distance 9999)
	candidate
	firstpage)
    (set-buffer (cgreek-tlg-parse-idt "tlg0086.idt")) ; Aristotle's work list
    (goto-char (point-min))
    (while (not (eobp))
      (if (and (setq firstpage
		     (plist-get (text-properties-at (point))
				'tlg-first-bekker-page))
	       (cgreek-tlg-bekker-<= firstpage page)
	       (< (cgreek-tlg-bekker-distance page firstpage) distance))
	  (setq candidate (current-line)
		distance (cgreek-tlg-bekker-distance page firstpage)))
      (beginning-of-line 2))		; next line's beginning
    (goto-line candidate)
    (cgreek-tlg-parse-txt-keyboard)))

(defun cgreek-tlg-get-bekker-page ()
  "Return the Bekker page number in a starting citation ID"

  ;; should be looking at:
  ;; ?\x03		type code 3 marks new section
  ;; [2 bytes]		the text block in which this section is located
  ;; ?\x08		type code 8 marks the section starting ID

  ;; just make sure
  (if (or (/= (following-char) ?\x03)
	  (/= (char-after (+ (point) 3)) ?\x08))
      (error "Unexpected subdivision %02x, %02x"
	     (following-char)
	     (char-after (+ (point) 3))))

  (forward-char 4)
  (if (= (following-char) ?\xa1)	; this happens only in two works
      (forward-char 1))

  ;; in TLG CD-ROM volume E, there are only two cases listed below
  (cond

   ;; left nibble = 1001 : level y
   ;; right nibble = 1001 : 7-bit numeric value + single ASCII character
   ((= (following-char) ?\x99)
    (format "%d%c"
	    (logand (char-after (1+ (point))) ?\x7f)
	    (logand (char-after (+ (point) 2)) ?\x7f)))

   ;; left nibble = 1001 : level y
   ;; right nibble = 1100 : 14-bit numeric value + single ASCII character
   ((= (following-char) ?\x9c)
    (format "%d%c"
	    (+ (* (logand (char-after (1+ (point))) ?\x7f) 128)
	       (logand (char-after (+ (point) 2)) ?\x7f))
	    (logand (char-after (+ (point) 3)) ?\x7f)))

   ;; should never happen
   (t
    (error "Unexpected starting citation code %02x" (following-char)))))

(defun cgreek-tlg-bekker-<= (x y)
  "Return t if Bekker page X is `less than' or `equal to' Bekker page Y."
  (let ((x-num (string-to-number x))
	(y-num (string-to-number y)))
    (cond
     ((< x-num y-num) t)
     ((> x-num y-num) nil)
     ((or (= (aref x (1- (length x))) ?a) 
	  (= (aref y (1- (length y))) ?b))
      t)
     (t nil))))
    
(defun cgreek-tlg-bekker-distance (x y)
  "Return Bekker page X `minus' Bekker page Y.  Both X and Y are strings.
A Bekker page consists of a numeric part and a character, either a or b.
The characters `a' and `b' are regarded as .0 and .1, respectively.
For example, 45a - 23b = 45.0 - 23.1 = 21.9."
  (- (+ (string-to-number x) (if (= (aref x (1- (length x))) ?a) 0.0 0.1))
     (+ (string-to-number y) (if (= (aref y (1- (length y))) ?a) 0.0 0.1))))

(defun tlg-to-cgreek-region (beg end)
  (interactive "*r")
  (call-process-region beg end cgreek-tlg2emacs-program t t nil))

(provide 'cgreek-util)

;;; cgreek-util.el ends here
