;;; cgreek-quail.el --- Quail package for classical Greek -*- coding: iso-2022-7bit; -*-

;; Copyright (C) 2002, 2003
;;   National Institute of Advanced Industrial Science and Technology (AIST)
;;   Registration Number H14PRO020

;; Author: TAKAHASHI Naoto <ntakahas@m17n.org>
;; Keywords: multilingual, classical Greek

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;; Last modified: 4 February 2003

;;; Code:

(require 'quail)

(quail-define-package "cgreek" "CGREEK" "CG" t "
     ---------------------------------------------------------------
        ASCII(Key Top)    Greek (Shift)      Greek (Non Shift)
     ---------------------------------------------------------------
         A                 [ALPHA]            [alpha]
         B                 [BETA]             [beta]
         C                 [CHI]              [chi]
         D                 [DELTA]            [delta]
         E                 [EPSILON]          [epsilon]
         F                 [PHI]              [phi]
         G                 [GAMMA]            [gamma]
         H                 [ETA]              [eta]
         I                 [IOTA]             [iota]
         J                 [iota subscriptum] [sigma (word end)]
         K                 [KAPPA]            [kappa]
         L                 [LAMBDA]           [lambda]
         M                 [MU]               [mu]
         N                 [NU]               [nu]
         O                 [OMICRON]          [omicron]
         P                 [PI]               [pi]
         Q                 [THETA]            [theta]
         R                 [RHO]              [rho]
         S                 [SIGMA]            [sigma]
         T                 [TAU]              [tau]
         U                 [YPSILON]          [ypsilon]
         V                 [rough breathing]  [smooth breathing]
         W                 [OMEGA]            [omega]
         X                 [XI]               [xi]
         Y                 [PSI]              [psi]
         Z                 [ZETA]             [zeta]

     -------------------------------------------------------------------
        ASCII(=Key Top)   Greek Letter        Code in Greek font Set
     -------------------------------------------------------------------
           !               [sampi]             {22h}
           #               [DIGAMMA]           {23h}
           $               [stigma]            {24h}
           %               [koppa]             {25h}
           &               [KOPPA]             {26h}

           :               [upper dot]         {3Ah}
           ;               ; [Gr.question mark]{3Bh}
           \"               [trema]             {40h}
           `       {60h}   [rough breating]    {60h}
           '               [smooth breathing]  {27h}
           /               [accute accent]     {2Fh}
           ?               [grave accent]      {5Ch}
           ^               [circumflex accent] {5Eh}
           \               [circumflex accent] {5Eh}

           ,               ,                   {2Ch}
           .               .                   {2Eh}
           -               -                   {2Dh}
           (               (                   {28h}
           )               )                   {29h}
           <               <                   {3Ch}
           >               >                   {3Eh}
           [               [                   {5Bh}
           ]               ]                   {5Dh}
           +               [KOME-JIRUSHI]      {2Bh}
"
nil t t)

(quail-define-rules

 ;; (" " ?$,4 A(B)
 ("!" ?$,4 B(B)
 ("#" ?$,4 C(B)
 ("$" ?$,4 D(B)
 ("%" ?$,4 E(B)
 ("&" ?$,4 F(B)
 ("'" ?$,4 G(B) ("v" ?$,4 G(B)
 ("(" ?$,4 H(B)
 (")" ?$,4 I(B)
 ;; ("*" ?$,4 J(B)
 ("+" ?$,4 K(B)
 ("," ?$,4 L(B)
 ("-" ?$,4 M(B)
 ("." ?$,4 N(B)
 ("/" ?$,4 O(B)

 ;; ("0" ?$,4 P(B)
 ;; ("1" ?$,4 Q(B)
 ;; ("2" ?$,4 R(B)
 ;; ("3" ?$,4 S(B)
 ;; ("4" ?$,4 T(B)
 ;; ("5" ?$,4 U(B)
 ;; ("6" ?$,4 V(B)
 ;; ("7" ?$,4 W(B)
 ;; ("8" ?$,4 X(B)
 ;; ("9" ?$,4 Y(B)
 (":" ?$,4 Z(B)
 (";" ?$,4 [(B)
 ("<" ?$,4 \(B)
 ;; ("=" ?$,4 ](B)
 (">" ?$,4 ^(B)
 ;; ("?" ?$,4 _(B)

 ("\"" ?$,4 `(B)
 ("A" ?$,4 a(B)
 ("B" ?$,4 b(B)
 ("C" ?$,4 c(B)
 ("D" ?$,4 d(B)
 ("E" ?$,4 e(B)
 ("F" ?$,4 f(B)
 ("G" ?$,4 g(B)
 ("H" ?$,4 h(B)
 ("I" ?$,4 i(B)
 ("wJ" ?$,4 j(B)
 ("K" ?$,4 k(B)
 ("L" ?$,4 l(B)
 ("M" ?$,4 m(B)
 ("N" ?$,4 n(B)
 ("O" ?$,4 o(B)

 ("P" ?$,4 p(B)
 ("Q" ?$,4 q(B)
 ("R" ?$,4 r(B)
 ("S" ?$,4 s(B)
 ("T" ?$,4 t(B)
 ("U" ?$,4 u(B)
 ("hJ" ?$,4 v(B)
 ("W" ?$,4 w(B)
 ("X" ?$,4 x(B)
 ("Y" ?$,4 y(B)
 ("Z" ?$,4 z(B)
 ("[" ?$,4 {(B)
 ("?" ?$,4 |(B)
 ("]" ?$,4 }(B)
 ("\\" ?$,4 ~(B) ("^" ?$,4 ~(B)
 ;; ("_" ?$,4 (B)

 ("`" ?$,4! (B) ("V" ?$,4! (B)
 ("a" ?$,4!!(B)
 ("b" ?$,4!"(B)
 ("c" ?$,4!#(B)
 ("d" ?$,4!$(B)
 ("e" ?$,4!%(B)
 ("f" ?$,4!&(B)
 ("g" ?$,4!'(B)
 ("h" ?$,4!((B)
 ("i" ?$,4!)(B)
 ("j" ?$,4!*(B)
 ("k" ?$,4!+(B)
 ("l" ?$,4!,(B)
 ("m" ?$,4!-(B)
 ("n" ?$,4!.(B)
 ("o" ?$,4!/(B)
 
 ("p" ?$,4!0(B)
 ("q" ?$,4!1(B)
 ("r" ?$,4!2(B)
 ("s" ?$,4!3(B)
 ("t" ?$,4!4(B)
 ("u" ?$,4!5(B)
 ("aJ" ?$,4!6(B)
 ("w" ?$,4!7(B)
 ("x" ?$,4!8(B)
 ("y" ?$,4!9(B)
 ("z" ?$,4!:(B)
 ;; ("{" ?$,4!;(B)
 ;; ("|" ?$,4!<(B)
 ;; ("}" ?$,4!=(B)
 ;; ("~" ?$,4!>(B)

 ("i`" ?$,4!C(B) ("iV" ?$,4!C(B)
 ("i'" ?$,4!D(B) ("iv" ?$,4!D(B)
 ("i/" ?$,4!E(B)
 ("i`/" ?$,4!F(B) ("iV/" ?$,4!F(B) ("i/`" ?$,4!F(B) ("i/V" ?$,4!F(B)
 ("i'/" ?$,4!G(B) ("iv/" ?$,4!G(B) ("i/'" ?$,4!G(B) ("i/v" ?$,4!G(B)
 ("i?" ?$,4!H(B)
 ("i`?" ?$,4!I(B) ("iV?" ?$,4!I(B) ("i?`" ?$,4!I(B) ("i?V" ?$,4!I(B)
 ("i'?" ?$,4!J(B) ("iv?" ?$,4!J(B) ("i?'" ?$,4!J(B) ("i?v" ?$,4!J(B)
 ("i^"  ?$,4!K(B) ("i\\" ?$,4!K(B)
 ("i`^"  ?$,4!L(B) ("i`\\" ?$,4!L(B) ("iV^"  ?$,4!L(B) ("iV\\" ?$,4!L(B)
 ("i^`"  ?$,4!L(B) ("i\\`" ?$,4!L(B) ("i^V"  ?$,4!L(B) ("i\\V" ?$,4!L(B)
 ("i'^"  ?$,4!M(B) ("i'\\" ?$,4!M(B) ("iv^"  ?$,4!M(B) ("iv\\" ?$,4!M(B)
 ("i^'"  ?$,4!M(B) ("i\\'" ?$,4!M(B) ("i^v"  ?$,4!M(B) ("i\\v" ?$,4!M(B)
 ("i\"" ?$,4!N(B)
 ("i/\"" ?$,4!O(B) ("i\"/" ?$,4!O(B)

 ("i?\"" ?$,4!P(B) ("i\"?" ?$,4!P(B)
 ("^`"  ?$,4!Q(B) ("^V"  ?$,4!Q(B) ("\\`" ?$,4!Q(B) ("\\V" ?$,4!Q(B)
 ("`^"  ?$,4!Q(B) ("V^"  ?$,4!Q(B) ("`\\" ?$,4!Q(B) ("V\\" ?$,4!Q(B)
 ("^'"  ?$,4!R(B) ("^v"  ?$,4!R(B) ("\\'" ?$,4!R(B) ("\\v" ?$,4!R(B)
 ("'^"  ?$,4!R(B) ("v^"  ?$,4!R(B) ("'\\" ?$,4!R(B) ("v\\" ?$,4!R(B)
 ("/`" ?$,4!S(B) ("/V" ?$,4!S(B) ("`/" ?$,4!S(B) ("V/" ?$,4!S(B)
 ("/'" ?$,4!T(B) ("/v" ?$,4!T(B) ("'/" ?$,4!T(B) ("v/" ?$,4!T(B)
 ("?`" ?$,4!U(B) ("?V" ?$,4!U(B) ("`?" ?$,4!U(B) ("V?" ?$,4!U(B)
 ("?'" ?$,4!V(B) ("?v" ?$,4!V(B) ("'?" ?$,4!V(B) ("v?" ?$,4!V(B)
 ("\"/" ?$,4!W(B) ("/\"" ?$,4!W(B)
 ("e`" ?$,4!X(B) ("eV" ?$,4!X(B)
 ("e'" ?$,4!Y(B) ("ev" ?$,4!Y(B)
 ("e/" ?$,4!Z(B)
 ("e/`" ?$,4![(B) ("e/V" ?$,4![(B) ("e`/" ?$,4![(B) ("eV/" ?$,4![(B)
 ("e/'" ?$,4!\(B) ("e/v" ?$,4!\(B) ("e'/" ?$,4!\(B) ("ev/" ?$,4!\(B)
 ("e?" ?$,4!](B)
 ("e?`" ?$,4!^(B) ("e?V" ?$,4!^(B) ("e`?" ?$,4!^(B) ("eV?" ?$,4!^(B)
 ("e?'" ?$,4!_(B) ("e?v" ?$,4!_(B) ("e'?" ?$,4!_(B) ("ev?" ?$,4!_(B)

 ("\"?" ?$,4!`(B) ("?\"" ?$,4!`(B)
 ("a`" ?$,4!a(B) ("aV" ?$,4!a(B)
 ("a'" ?$,4!b(B) ("av" ?$,4!b(B)
 ("a/" ?$,4!c(B)
 ("a/`" ?$,4!d(B) ("a/V" ?$,4!d(B) ("a`/" ?$,4!d(B) ("aV/" ?$,4!d(B)
 ("a/'" ?$,4!e(B) ("a/v" ?$,4!e(B) ("a'/" ?$,4!e(B) ("av/" ?$,4!e(B)
 ("a?" ?$,4!f(B)
 ("a?`" ?$,4!g(B) ("a?V" ?$,4!g(B) ("a`?" ?$,4!g(B) ("aV?" ?$,4!g(B)
 ("a?'" ?$,4!h(B) ("a?v" ?$,4!h(B) ("a'?" ?$,4!h(B) ("av?" ?$,4!h(B)
 ("a^"  ?$,4!i(B) ("a\\" ?$,4!i(B)
 ("a^`"  ?$,4!j(B) ("a^V"  ?$,4!j(B) ("a\\`" ?$,4!j(B) ("a\\V" ?$,4!j(B)
 ("a`^"  ?$,4!j(B) ("aV^"  ?$,4!j(B) ("a`\\" ?$,4!j(B) ("aV\\" ?$,4!j(B)
 ("a^'"  ?$,4!k(B) ("a^v"  ?$,4!k(B) ("a\\'" ?$,4!k(B) ("a\\v" ?$,4!k(B)
 ("a'^"  ?$,4!k(B) ("av^"  ?$,4!k(B) ("a'\\" ?$,4!k(B) ("av\\" ?$,4!k(B)
 ("aJ`" ?$,4!l(B) ("aJV" ?$,4!l(B)
 ("aJ'" ?$,4!m(B) ("aJv" ?$,4!m(B)
 ("aJ/" ?$,4!n(B)
 ("aJ/`" ?$,4!o(B) ("aJ/V" ?$,4!o(B) ("aJ`/" ?$,4!o(B) ("aJV/" ?$,4!o(B)

 ("aJ/'" ?$,4!p(B) ("aJ/v" ?$,4!p(B) ("aJ'/" ?$,4!p(B) ("aJv/" ?$,4!p(B)
 ("aJ?" ?$,4!q(B)
 ("aJ?`" ?$,4!r(B) ("aJ?V" ?$,4!r(B) ("aJ`?" ?$,4!r(B) ("aJV?" ?$,4!r(B)
 ("aJ?'" ?$,4!s(B) ("aJ?v" ?$,4!s(B) ("aJ'?" ?$,4!s(B) ("aJv?" ?$,4!s(B)
 ("aJ^"  ?$,4!t(B) ("aJ\\" ?$,4!t(B)
 ("aJ^`"  ?$,4!u(B) ("aJ^V"  ?$,4!u(B) ("aJ\\`" ?$,4!u(B) ("aJ\\V" ?$,4!u(B)
 ("aJ`^"  ?$,4!u(B) ("aJV^"  ?$,4!u(B) ("aJ`\\" ?$,4!u(B) ("aJV\\" ?$,4!u(B)
 ("aJ^'"  ?$,4!v(B) ("aJ^v"  ?$,4!v(B) ("aJ\\'" ?$,4!v(B) ("aJ\\v" ?$,4!v(B)
 ("aJ'^"  ?$,4!v(B) ("aJv^"  ?$,4!v(B) ("aJ'\\" ?$,4!v(B) ("aJv\\" ?$,4!v(B)
 ("r`" ?$,4!w(B) ("rV" ?$,4!w(B)
 ("r'" ?$,4!x(B) ("rv" ?$,4!x(B)
 ("h`" ?$,4!y(B) ("hV" ?$,4!y(B)
 ("h'" ?$,4!z(B) ("hv" ?$,4!z(B)
 ("h/" ?$,4!{(B)
 ("h/`" ?$,4!|(B) ("h/V" ?$,4!|(B) ("h`/" ?$,4!|(B) ("hV/" ?$,4!|(B)
 ("h/'" ?$,4!}(B) ("h/v" ?$,4!}(B) ("h'/" ?$,4!}(B) ("hv/" ?$,4!}(B)
 ("h?" ?$,4!~(B)
 ("h?`" ?$,4!(B) ("h?V" ?$,4!(B) ("h`?" ?$,4!(B) ("hV?" ?$,4!(B)

 ("h?'" ?$,4" (B) ("h?v" ?$,4" (B) ("h'?" ?$,4" (B) ("hv?" ?$,4" (B)
 ("h^"  ?$,4"!(B) ("h\\" ?$,4"!(B)
 ("h^`"  ?$,4""(B) ("h^V"  ?$,4""(B) ("h\\`" ?$,4""(B) ("h\\V" ?$,4""(B)
 ("h`^"  ?$,4""(B) ("h`\\" ?$,4""(B) ("hV^"  ?$,4""(B) ("hV\\" ?$,4""(B)
 ("h^'"  ?$,4"#(B) ("h^v"  ?$,4"#(B) ("h\\'" ?$,4"#(B) ("h\\v" ?$,4"#(B)
 ("h'^"  ?$,4"#(B) ("h'\\" ?$,4"#(B) ("hv^"  ?$,4"#(B) ("hv\\" ?$,4"#(B)
 ("J" ?$,4"$(B)
 ("hJ`" ?$,4"%(B) ("hJV" ?$,4"%(B)
 ("hJ'" ?$,4"&(B) ("hJv" ?$,4"&(B)
 ("hJ/" ?$,4"'(B)
 ("hJ`/" ?$,4"((B) ("hJV/" ?$,4"((B) ("hJ/`" ?$,4"((B) ("hJ/V" ?$,4"((B)
 ("hJ'/" ?$,4")(B) ("hJv/" ?$,4")(B) ("hJ/'" ?$,4")(B) ("hJ/v" ?$,4")(B)
 ("hJ?" ?$,4"*(B)
 ("hJ`?" ?$,4"+(B) ("hJV?" ?$,4"+(B) ("hJ?`" ?$,4"+(B) ("hJ?V" ?$,4"+(B)
 ("hJ'?" ?$,4",(B) ("hJv?" ?$,4",(B) ("hJ?'" ?$,4",(B) ("hJ?v" ?$,4",(B)
 ("hJ^"  ?$,4"-(B) ("hJ\\" ?$,4"-(B)
 ("hJ`^"  ?$,4".(B) ("hJ`\\" ?$,4".(B) ("hJV^"  ?$,4".(B) ("hJV\\" ?$,4".(B)
 ("hJ^`"  ?$,4".(B) ("hJ\\`" ?$,4".(B) ("hJ^V"  ?$,4".(B) ("hJ\\V" ?$,4".(B)
 ("hJ'^"  ?$,4"/(B) ("hJ'\\" ?$,4"/(B) ("hJv^"  ?$,4"/(B) ("hJv\\" ?$,4"/(B)
 ("hJ^'"  ?$,4"/(B) ("hJ\\'" ?$,4"/(B) ("hJ^v"  ?$,4"/(B) ("hJ\\v" ?$,4"/(B)

 ("o`" ?$,4"0(B) ("oV" ?$,4"0(B)
 ("o'" ?$,4"1(B) ("ov" ?$,4"1(B)
 ("o/" ?$,4"2(B)
 ("o/`" ?$,4"3(B) ("o/V" ?$,4"3(B) ("o`/" ?$,4"3(B) ("oV/" ?$,4"3(B)
 ("o/'" ?$,4"4(B) ("o/v" ?$,4"4(B) ("o'/" ?$,4"4(B) ("ov/" ?$,4"4(B)
 ("o?" ?$,4"5(B)
 ("o?`" ?$,4"6(B) ("o?V" ?$,4"6(B) ("o`?" ?$,4"6(B) ("oV?" ?$,4"6(B)
 ("o?'" ?$,4"7(B) ("o?v" ?$,4"7(B) ("o'?" ?$,4"7(B) ("ov?" ?$,4"7(B)
 ("u`" ?$,4"8(B) ("uV" ?$,4"8(B)
 ("u'" ?$,4"9(B) ("uv" ?$,4"9(B)
 ("u/" ?$,4":(B)
 ("u/`" ?$,4";(B) ("u/V" ?$,4";(B) ("u`/" ?$,4";(B) ("uV/" ?$,4";(B)
 ("u/'" ?$,4"<(B) ("u/v" ?$,4"<(B) ("u'/" ?$,4"<(B) ("uv/" ?$,4"<(B)
 ("u?" ?$,4"=(B)
 ("u?`" ?$,4">(B) ("u?V" ?$,4">(B) ("u`?" ?$,4">(B) ("uV?" ?$,4">(B)
 ("u?'" ?$,4"?(B) ("u?v" ?$,4"?(B) ("u'?" ?$,4"?(B) ("uv?" ?$,4"?(B)

 ("u^"  ?$,4"@(B) ("u\\" ?$,4"@(B)
 ("u^`"  ?$,4"A(B) ("u^V"  ?$,4"A(B) ("u\\`" ?$,4"A(B) ("u\\V" ?$,4"A(B)
 ("u`^"  ?$,4"A(B) ("uV^"  ?$,4"A(B) ("u`\\" ?$,4"A(B) ("uV\\" ?$,4"A(B)
 ("u^'"  ?$,4"B(B) ("u^v"  ?$,4"B(B) ("u\\'" ?$,4"B(B) ("u\\v" ?$,4"B(B)
 ("u'^"  ?$,4"B(B) ("uv^"  ?$,4"B(B) ("u'\\" ?$,4"B(B) ("uv\\" ?$,4"B(B)
 ("u\"" ?$,4"C(B)
 ("u\"/" ?$,4"D(B) ("u/\"" ?$,4"D(B)
 ("u\"?" ?$,4"E(B) ("u?\"" ?$,4"E(B)
 ("w`" ?$,4"F(B) ("wV" ?$,4"F(B)
 ("w'" ?$,4"G(B) ("wv" ?$,4"G(B)
 ("w/" ?$,4"H(B)
 ("w/`" ?$,4"I(B) ("w/V" ?$,4"I(B) ("w`/" ?$,4"I(B) ("wV/" ?$,4"I(B)
 ("w/'" ?$,4"J(B) ("w/v" ?$,4"J(B) ("w'/" ?$,4"J(B) ("wv/" ?$,4"J(B)
 ("w?" ?$,4"K(B)
 ("w?`" ?$,4"L(B) ("w?V" ?$,4"L(B) ("w`?" ?$,4"L(B) ("wV?" ?$,4"L(B)
 ("w?'" ?$,4"M(B) ("w?v" ?$,4"M(B) ("w'?" ?$,4"M(B) ("wv?" ?$,4"M(B)
 ("w^"  ?$,4"N(B) ("w\\" ?$,4"N(B)
 ("w^`"  ?$,4"O(B) ("w^V"  ?$,4"O(B) ("w\\`" ?$,4"O(B) ("w\\V" ?$,4"O(B)
 ("w`^"  ?$,4"O(B) ("wV^"  ?$,4"O(B) ("w`\\" ?$,4"O(B) ("wV\\" ?$,4"O(B)

 ("w^'"  ?$,4"P(B) ("w^v"  ?$,4"P(B) ("w\\'" ?$,4"P(B) ("w\\v" ?$,4"P(B)
 ("w'^"  ?$,4"P(B) ("wv^"  ?$,4"P(B) ("w'\\" ?$,4"P(B) ("wv\\" ?$,4"P(B)
 ("wJ`" ?$,4"Q(B) ("wJV" ?$,4"Q(B)
 ("wJ'" ?$,4"R(B) ("wJv" ?$,4"R(B)
 ("wJ/" ?$,4"S(B)
 ("wJ/`" ?$,4"T(B) ("wJ/V" ?$,4"T(B) ("wJ`/" ?$,4"T(B) ("wJV/" ?$,4"T(B)
 ("wJ/'" ?$,4"U(B) ("wJ/v" ?$,4"U(B) ("wJ'/" ?$,4"U(B) ("wJv/" ?$,4"U(B)
 ("wJ?" ?$,4"V(B)
 ("wJ?`" ?$,4"W(B) ("wJ?V" ?$,4"W(B) ("wJ`?" ?$,4"W(B) ("wJV?" ?$,4"W(B)
 ("wJ?'" ?$,4"X(B) ("wJ?v" ?$,4"X(B) ("wJ'?" ?$,4"X(B) ("wJv?" ?$,4"X(B)
 ("wJ^"  ?$,4"Y(B) ("wJ\\" ?$,4"Y(B)
 ("wJ^`"  ?$,4"Z(B) ("wJ^V"  ?$,4"Z(B) ("wJ\\`" ?$,4"Z(B) ("wJ\\V" ?$,4"Z(B)
 ("wJ`^"  ?$,4"Z(B) ("wJV^"  ?$,4"Z(B) ("wJ`\\" ?$,4"Z(B) ("wJV\\" ?$,4"Z(B)
 ("wJ^'"  ?$,4"[(B) ("wJ^v"  ?$,4"[(B) ("wJ\\'" ?$,4"[(B) ("wJ\\v" ?$,4"[(B)
 ("wJ'^"  ?$,4"[(B) ("wJv^"  ?$,4"[(B) ("wJ'\\" ?$,4"[(B) ("wJv\\" ?$,4"[(B)
 ("e^"  ?$,4"\(B) ("e\\" ?$,4"\(B)
 ("o^"  ?$,4"](B) ("o\\" ?$,4"](B)
 )

(quail-define-package "greek-ibycus4" "Greek" "IB" t
"The Ibycus4 input method for Classical Greek."
nil t t)

(quail-define-rules

 ("\(\(" ?$,4 H(B)
 ("\)\)" ?$,4 I(B)

 ("|" ?$,4"$(B)
 ("?" ?$,4 [(B)
 (";" ?$,4 Z(B)
 ("A" ?$,4 a(B)
 ("B" ?$,4 b(B)
 ("G" ?$,4 g(B)
 ("D" ?$,4 d(B)
 ("E" ?$,4 e(B)
 ("Z" ?$,4 z(B)
 ("H" ?$,4 h(B)
 ("Q" ?$,4 q(B)
 ("I" ?$,4 i(B)
 ("K" ?$,4 k(B)
 ("L" ?$,4 l(B)
 ("M" ?$,4 m(B)
 ("N" ?$,4 n(B)
 ("C" ?$,4 x(B)
 ("O" ?$,4 o(B)
 ("P" ?$,4 p(B)
 ("R" ?$,4 r(B)
 ("S" ?$,4 s(B)
 ("T" ?$,4 t(B)
 ("U" ?$,4 u(B)
 ("F" ?$,4 f(B)
 ("X" ?$,4 c(B)
 ("Y" ?$,4 y(B)
 ("W" ?$,4 w(B)
 ("a" ?$,4!!(B)
 ("b" ?$,4!"(B)
 ("g" ?$,4!'(B)
 ("d" ?$,4!$(B)
 ("e" ?$,4!%(B)
 ("z" ?$,4!:(B)
 ("h" ?$,4!((B)
 ("q" ?$,4!1(B)
 ("i" ?$,4!)(B)
 ("k" ?$,4!+(B)
 ("l" ?$,4!,(B)
 ("m" ?$,4!-(B)
 ("n" ?$,4!.(B)
 ("c" ?$,4!8(B)
 ("o" ?$,4!/(B)
 ("p" ?$,4!0(B)
 ("r" ?$,4!2(B)
 ("j" ?$,4!*(B) ("s " ["$,4!*(B "]) ("s," ["$,4!* L(B"]) ("s." ["$,4!* N(B"]) ("s?" ["$,4!* [(B"]) ("s;" ["$,4!* Z(B"])
 ("s|" ?$,4!3(B) ("s" ?$,4!3(B)
 ("t" ?$,4!4(B)
 ("u" ?$,4!5(B)
 ("f" ?$,4!&(B)
 ("x" ?$,4!#(B)
 ("y" ?$,4!9(B)
 ("w" ?$,4!7(B)
 ("i+" ?$,4!N(B)
 ("u+" ?$,4"C(B)

 ("s+" ?$,4 B(B) ; sampi
 ("V" ?$,4 C(B) ; DIGAMMA
; ("" ?$,4 D(B) ; stigma
 ("K+" ?$,4 F(B) ; KOPPA
 ("k+" ?$,4 E(B) ; koppa

 ("a\)" ?$,4!b(B)
 ("a\(" ?$,4!a(B)
 ("a\)`" ?$,4!h(B)
 ("a\(`" ?$,4!g(B)
 ("a\)'" ?$,4!e(B)
 ("a\('" ?$,4!d(B)
 ("a\)='" ?$,4!k(B)
 ("a\(=" ?$,4!j(B)
 ("e\)" ?$,4!Y(B)
 ("e\(" ?$,4!X(B)
 ("e\)`" ?$,4!_(B)
 ("e\(`" ?$,4!^(B)
 ("e\)'" ?$,4!\(B)
 ("e\('" ?$,4![(B)
 ("h\)" ?$,4!z(B)
 ("h\(" ?$,4!y(B)
 ("h\)`" ?$,4" (B)
 ("h\(`" ?$,4!(B)
 ("h\)'" ?$,4!}(B)
 ("h\('" ?$,4!|(B)
 ("h\)=" ?$,4"#(B)
 ("h\(=" ?$,4""(B)
 ("i\)" ?$,4!D(B)
 ("i\(" ?$,4!C(B)
 ("i\)`" ?$,4!J(B)
 ("i\(`" ?$,4!I(B)
 ("i\)'" ?$,4!G(B)
 ("i\('" ?$,4!F(B)
 ("i\)=" ?$,4!M(B)
 ("i\(=" ?$,4!L(B)
 ("o\)" ?$,4"1(B)
 ("o\(" ?$,4"0(B)
 ("o\)`" ?$,4"7(B)
 ("o\(`" ?$,4"6(B)
 ("o\)'" ?$,4"4(B)
 ("o\('" ?$,4"3(B)
 ("u\)" ?$,4"9(B)
 ("u\(" ?$,4"8(B)
 ("u\)`" ?$,4"?(B)
 ("u\(`" ?$,4">(B)
 ("u\)'" ?$,4"<(B)
 ("u\('" ?$,4";(B)
 ("u\)=" ?$,4"B(B)
 ("u\(=" ?$,4"A(B)
 ("w\)" ?$,4"G(B)
 ("w\(" ?$,4"F(B)
 ("w\)`" ?$,4"M(B)
 ("w\(`" ?$,4"L(B)
 ("w\)'" ?$,4"J(B)
 ("w\('" ?$,4"I(B)
 ("w\)=" ?$,4"P(B)
 ("w\(=" ?$,4"O(B)
 ("a`" ?$,4!f(B)
 ("a'" ?$,4!c(B)
 ("e`" ?$,4!](B)
 ("e'" ?$,4!Z(B)
 ("h`" ?$,4!~(B)
 ("h'" ?$,4!{(B)
 ("i`" ?$,4!H(B)
 ("i'" ?$,4!E(B)
 ("o`" ?$,4"5(B)
 ("o'" ?$,4"2(B)
 ("u`" ?$,4"=(B)
 ("u'" ?$,4":(B)
 ("w`" ?$,4"K(B)
 ("w'" ?$,4"H(B)
 ("a\)|" ?$,4!m(B)
 ("a\(|" ?$,4!l(B)
 ("a\)`|" ?$,4!s(B)
 ("a\(`|" ?$,4!r(B)
 ("a\)'|" ?$,4!p(B)
 ("a\('|" ?$,4!o(B)
 ("a\)=|" ?$,4!v(B)
 ("a\(=|" ?$,4!u(B)
 ("h\)|" ?$,4"&(B)
 ("h\(|" ?$,4"%(B)
 ("h\)`|" ?$,4",(B)
 ("h\(`|" ?$,4"+(B)
 ("h\)'|" ?$,4")(B)
 ("h\('|" ?$,4"((B)
 ("h\)=|" ?$,4"/(B)
 ("h\(=|" ?$,4".(B)
 ("w\)|" ?$,4"R(B)
 ("w\(|" ?$,4"Q(B)
 ("w\)`|" ?$,4"X(B)
 ("w\(`|" ?$,4"W(B)
 ("w\)'|" ?$,4"U(B)
 ("w\('|" ?$,4"T(B)
 ("w\)=|" ?$,4"[(B)
 ("w\(=|" ?$,4"Z(B)
 ("a`|" ?$,4!q(B)
 ("a|" ?$,4!6(B)
 ("a'|" ?$,4!n(B)
 ("a=" ?$,4!i(B)
 ("a=|" ?$,4!t(B)
 ("|" ?$,4"$(B)
 ("\)" ?$,4 G(B)
 ("=" ?$,4 ~(B)
 ("h`|" ?$,4"*(B)
 ("h|" ?$,4 v(B)
 ("h'|" ?$,4"'(B)
 ("h=" ?$,4"!(B)
 ("h=|" ?$,4"-(B)
 ("\)`" ?$,4!V(B)
 ("\)'" ?$,4!T(B)
 ("\)=" ?$,4!R(B)
 ("i+`" ?$,4!P(B)
 ("i+'" ?$,4!O(B)
 ("i=" ?$,4!K(B)
 ("\(`" ?$,4!U(B)
 ("\('" ?$,4!S(B)
 ("\(=" ?$,4!Q(B)
 ("u+`" ?$,4"E(B)
 ("u+'" ?$,4"D(B)
 ("r\)" ?$,4!x(B)
 ("r\(" ?$,4!w(B)
 ("u=" ?$,4"@(B)
 ("+`" ?$,4!`(B)
 ("+'" ?$,4!W(B)
 ("`" ?$,4 |(B)
 ("w`|" ?$,4"V(B)
 ("w|" ?$,4 j(B)
 ("w'|" ?$,4"S(B)
 ("w=" ?$,4"N(B)
 ("w=|" ?$,4"Y(B)
 ("'" ?$,4 O(B)
 ("\(" ?$,4! (B)
)

(quail-define-package "latin-1-latex" "Latin-1" "LL" t
"The LaTeX input method for Latin-1 characters, e.g.,
,Ar(B \\`{o}        ,As(B \\'{o}        ,At(B \\^{o}        ,Av(B \\\"{o}
,Au(B \\~{o}        ,Ag(B \\c{c}

Be careful with i's.
,Al(B \\`{\\i}       ,Am(B \\'{\\i}       ,An(B \\^{\\i}       ,Ao(B \\\"{\\i}

Use {} when the command is purely alphabetical.
,Af(B {\\ae}        ,Ae(B {\\aa}        ,Ax(B {\\o}         ,A_(B {\\ss}
,AF(B {\\AE}        ,AE(B {\\AA}        ,AX(B {\\O}
,A'(B {\\S}         ,A6(B {\\P}         ,A#(B {\\pounds}    ,A)(B {\\copyright}

Spanish punctuations do not require {}'s.
,A?(B ?`           ,A!(B !`"
nil t t)

(quail-define-rules
 ("!`" ?,A!(B)
 ("{\\pounds}" ?,A#(B)
 ("{\\S}" ?,A'(B)
 ("{\\copyright}" ?,A)(B)
 ("{\\P}" ?,A6(B)
 ("?`" ?,A?(B)

 ("\\`{A}" ?,A@(B)
 ("\\'{A}" ?,AA(B)
 ("\\^{A}" ?,AB(B)
 ("\\~{A}" ?,AC(B)
 ("\\\"{A}" ?,AD(B)
 ("{\\AA}" ?,AE(B)
 ("{\\AE}" ?,AF(B)
 ("\\c{C}" ?,AG(B)
 ("\\`{E}" ?,AH(B)
 ("\\'{E}" ?,AI(B)
 ("\\^{E}" ?,AJ(B)
 ("\\\"{E}" ?,AK(B)
 ("\\`{I}" ?,AL(B)
 ("\\'{I}" ?,AM(B)
 ("\\^{I}" ?,AN(B)
 ("\\\"{I}" ?,AO(B)

 ("\\~{N}" ?,AQ(B)
 ("\\`{O}" ?,AR(B)
 ("\\'{O}" ?,AS(B)
 ("\\^{O}" ?,AT(B)
 ("\\~{O}" ?,AU(B)
 ("\\\"{O}" ?,AV(B)
 ("{\\O}" ?,AX(B)
 ("\\`{U}" ?,AY(B)
 ("\\'{U}" ?,AZ(B)
 ("\\^{U}" ?,A[(B)
 ("\\\"{U}" ?,A\(B)
 ("\\'{Y}" ?,A](B)
 ("{\\ss}" ?,A_(B)

 ("\\`{a}" ?,A`(B)
 ("\\'{a}" ?,Aa(B)
 ("\\^{a}" ?,Ab(B)
 ("\\~{a}" ?,Ac(B)
 ("\\\"{a}" ?,Ad(B)
 ("{\\aa}" ?,Ae(B)
 ("{\\ae}" ?,Af(B)
 ("\\c{c}" ?,Ag(B)
 ("\\`{e}" ?,Ah(B)
 ("\\'{e}" ?,Ai(B)
 ("\\^{e}" ?,Aj(B)
 ("\\\"{e}" ?,Ak(B)
 ("\\`{\\i}" ?,Al(B)
 ("\\'{\\i}" ?,Am(B)
 ("\\^{\\i}" ?,An(B)
 ("\\\"{\\i}" ?,Ao(B)

 ("\\~{n}" ?,Aq(B)
 ("\\`{o}" ?,Ar(B)
 ("\\'{o}" ?,As(B)
 ("\\^{o}" ?,At(B)
 ("\\~{o}" ?,Au(B)
 ("\\\"{o}" ?,Av(B)
 ("{\\o}" ?,Ax(B)
 ("\\`{u}" ?,Ay(B)
 ("\\'{u}" ?,Az(B)
 ("\\^{u}" ?,A{(B)
 ("\\\"{u}" ?,A|(B)
 ("\\'{y}" ?,A}(B)
 ("\\\"{y}" ?,A(B)
)

(provide 'cgreek-quail)

;;; for your info

;;     | 0 1 2 3 4 5 6 7 8 9 A B C D E F
;;  ---+--------------------------------
;;   2 | $,4 @(B $,4 A(B $,4 B(B $,4 C(B $,4 D(B $,4 E(B $,4 F(B $,4 G(B $,4 H(B $,4 I(B $,4 J(B $,4 K(B $,4 L(B $,4 M(B $,4 N(B $,4 O(B
;;   3 | $,4 P(B $,4 Q(B $,4 R(B $,4 S(B $,4 T(B $,4 U(B $,4 V(B $,4 W(B $,4 X(B $,4 Y(B $,4 Z(B $,4 [(B $,4 \(B $,4 ](B $,4 ^(B $,4 _(B
;;   4 | $,4 `(B $,4 a(B $,4 b(B $,4 c(B $,4 d(B $,4 e(B $,4 f(B $,4 g(B $,4 h(B $,4 i(B $,4 j(B $,4 k(B $,4 l(B $,4 m(B $,4 n(B $,4 o(B
;;   5 | $,4 p(B $,4 q(B $,4 r(B $,4 s(B $,4 t(B $,4 u(B $,4 v(B $,4 w(B $,4 x(B $,4 y(B $,4 z(B $,4 {(B $,4 |(B $,4 }(B $,4 ~(B $,4 (B
;;   6 | $,4! (B $,4!!(B $,4!"(B $,4!#(B $,4!$(B $,4!%(B $,4!&(B $,4!'(B $,4!((B $,4!)(B $,4!*(B $,4!+(B $,4!,(B $,4!-(B $,4!.(B $,4!/(B
;;   7 | $,4!0(B $,4!1(B $,4!2(B $,4!3(B $,4!4(B $,4!5(B $,4!6(B $,4!7(B $,4!8(B $,4!9(B $,4!:(B $,4!;(B $,4!<(B $,4!=(B $,4!>(B $,4!?(B
;;   8 | $,4!@(B $,4!A(B $,4!B(B $,4!C(B $,4!D(B $,4!E(B $,4!F(B $,4!G(B $,4!H(B $,4!I(B $,4!J(B $,4!K(B $,4!L(B $,4!M(B $,4!N(B $,4!O(B
;;   9 | $,4!P(B $,4!Q(B $,4!R(B $,4!S(B $,4!T(B $,4!U(B $,4!V(B $,4!W(B $,4!X(B $,4!Y(B $,4!Z(B $,4![(B $,4!\(B $,4!](B $,4!^(B $,4!_(B
;;   A | $,4!`(B $,4!a(B $,4!b(B $,4!c(B $,4!d(B $,4!e(B $,4!f(B $,4!g(B $,4!h(B $,4!i(B $,4!j(B $,4!k(B $,4!l(B $,4!m(B $,4!n(B $,4!o(B
;;   B | $,4!p(B $,4!q(B $,4!r(B $,4!s(B $,4!t(B $,4!u(B $,4!v(B $,4!w(B $,4!x(B $,4!y(B $,4!z(B $,4!{(B $,4!|(B $,4!}(B $,4!~(B $,4!(B
;;   C | $,4" (B $,4"!(B $,4""(B $,4"#(B $,4"$(B $,4"%(B $,4"&(B $,4"'(B $,4"((B $,4")(B $,4"*(B $,4"+(B $,4",(B $,4"-(B $,4".(B $,4"/(B
;;   D | $,4"0(B $,4"1(B $,4"2(B $,4"3(B $,4"4(B $,4"5(B $,4"6(B $,4"7(B $,4"8(B $,4"9(B $,4":(B $,4";(B $,4"<(B $,4"=(B $,4">(B $,4"?(B
;;   E | $,4"@(B $,4"A(B $,4"B(B $,4"C(B $,4"D(B $,4"E(B $,4"F(B $,4"G(B $,4"H(B $,4"I(B $,4"J(B $,4"K(B $,4"L(B $,4"M(B $,4"N(B $,4"O(B
;;   F | $,4"P(B $,4"Q(B $,4"R(B $,4"S(B $,4"T(B $,4"U(B $,4"V(B $,4"W(B $,4"X(B $,4"Y(B $,4"Z(B $,4"[(B $,4"\(B $,4"](B $,4"^(B $,4"_(B

;;; cgreek-quail.el ends here
