/*
 * Copyright (c) Eric D. Friedman 1998. All Rights Reserved.
 * Copyright (c) Paul Kinnucan 1998. All Rights Reserved.
 *
 * $Revision: 1.9 $ 
 * $Date: 2002/05/14 06:38:44 $ 
 *
 * InterfaceFactory is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * InterfaceFactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * To obtain a copy of the GNU General Public License write to the
 * Free Software Foundation, Inc.,  59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

package jde.wizards;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.io.PrintWriter;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;

/**
 * Defines a factory for creating skeleton implementations of 
 * Java interfaces. The factory can be invoked from the command line
 * or from another program. The factory can generate implementations for
 * multiple interfaces when invoked from the command line.
 *
 * @author Eric D. Friedman and Paul Kinnucan
 * @version $Revision: 1.9 $
 */

public class InterfaceFactory extends MethodFactory
{

  /** A table w/ declaring classes as keys and vectors of method
   * signatures as values */
  protected Hashtable interfaces = new Hashtable();

  /** The interface factory. */
  static InterfaceFactory interfaceFactory;
  
  public InterfaceFactory() {}

  /** 
   * Creates an InterfaceFactory that uses the specified NameFactory
   * for generating parameter names 
   *
   * @param factory Factory for generating parameter names
   */
  public InterfaceFactory(NameFactory factory)
  {
    super(factory);
  }

  /** 
   * Adds a signature to the signature table.
   * The signatures are stored in Vectors with their declaring
   * Classes as keys in a hashtable.  This allows them to be pulled
   * out in groups when we print, and keeping the declaring Class info
   * lets us put in comments about where the method (or method group)
   * comes from.  Signatures are not added if they're already registered
   * because we do not want duplicate method implementations even though
   * a class might implement interfaces that inherit from some common
   * super-interface. 
   *
   * @param sig Signature to be stored in the signature table.
   */
  protected final void sortByDeclaringClass(Signature sig)
  {
    String declaring = sig.getDeclaringClass().getName();
    if (interfaces.containsKey( declaring ))
    {
      Vector v = (Vector)interfaces.get( declaring );
      if (! v.contains( sig ) ) // "There can be only one" - the Highlander
        v.addElement(sig);
    } // end of if (interfaces.containsKey( dec ))
    else
    {
      Vector v = new Vector();
      v.addElement(sig);
      interfaces.put( declaring, v );
    } // end of else
  }


  /** 
   * Clears the import and interface hashtables for this factory so they
   * can be re-used to process a new set of interfaces.
   */
  public void flush()
  {
    super.flush();
    interfaces.clear();
  }

  
  /**
   * Generates signatures based on introspection of the specified interface. 
   * Strips package specifiers from generated signatures.
   *
   * @param name the interface to process for signatures.
   * @exception NotAnInterfaceException the requested class isn't an interface
   * @exception java.lang.ClassNotFoundException the requested class cannot
   be loaded 
  */
  public void process(String interfaceName)
    throws ClassNotFoundException, NotAnInterfaceException
  {
    process(interfaceName, true);
  }  
  
  /**
   * Generates signatures based on introspection of the specified class. 
   *
   * @param name the interface to process for signatures.
   * @param truncate toggles truncation of package specifiers in signatures..
   *
   * @exception NotAnInterfaceException the requested class isn't an interface
   * @exception java.lang.ClassNotFoundException the requested class cannot
   * be loaded 
  */
  public void process(String name, boolean truncate)
    throws ClassNotFoundException, NotAnInterfaceException
  {
    if (null == namefactory)
      namefactory = new DefaultNameFactory();
    
    Class aclass = Class.forName( name );
    if (false == aclass.isInterface())
      throw new NotAnInterfaceException(name);
    
    Method[] methods = aclass.getMethods();
    for (int i = 0; i < methods.length; i++)
      sortByDeclaringClass( new Signature( methods[i], this, truncate ) );
  }


  /**
   * Makes an expression for generating the implementation of an
   * interface. This method delegates the creation of the
   * implementation to makeInterfaceExpressionInternal.
   *
   * @param name Name of interface to be implemented.
   * @param truncate If <code>true</code>, truncate package specifier
   * when generating code.
   */

  public static void makeInterfaceExpression
      (String name, boolean truncate) {

    if (interfaceFactory == null)
      interfaceFactory = new InterfaceFactory();

    interfaceFactory.flush();
    interfaceFactory.makeInterfaceExpressionInternal(name, truncate);

  }

 /**
   * Makes an implementation of an interface.
   *
   * @param name Name of interface to be implemented.
   * @param truncate If <code>true</code>, truncate package specifier
   * when generating code.
   */
  protected void makeInterfaceExpressionInternal 
      (String name, boolean truncate) {
    try {
      process(name, truncate);
    }
    catch (ClassNotFoundException e) {
      println("(error \"Error: could not find interface named: " + name + ". "
	      + "Note: name must be qualified.\")");
      return;
    }
    catch (NotAnInterfaceException e) {
      println("(error \"Error: " + name + " is not an interface.\")");
      return;
    }
    catch (Exception e) {
      println("(error \"Error: unknown type.\")");
      return;
    }

    dumpExpression(new PrintWriter( System.out, true), truncate);
  }


  public static void getImportedClasses() {
    println(interfaceFactory.getImportsAsList());
  }


  public void dumpExpression(PrintWriter out, boolean truncate)
  {
    StringBuffer buf = new StringBuffer
	("(jde-wiz-gen-implementation-methods (list ");

    Enumeration declaring_classes = interfaces.keys();
    while (declaring_classes.hasMoreElements())
    {
      String interf = (String)declaring_classes.nextElement();
      Vector v = (Vector)interfaces.get(interf);
      buf.append ("(quote ");
      buf.append("\"Implementation of ");
      buf.append(interf);
      buf.append("\")");
      Enumeration e = v.elements();
      while (e.hasMoreElements())
      {
        Signature sig = (Signature)e.nextElement();
	buf.append ("(quote ");
        buf.append(getMethodSkeletonExpression(sig));
	buf.append (")");
      } // end of while (e.hasMoreElements())
    }
    buf.append("))");
    println(buf.toString());
  }
} // SignatureFactory

class NotAnInterfaceException extends Exception
{
  NotAnInterfaceException (String name)
  {
    super(name);
  }
}

/*
 * $Log: InterfaceFactory.java,v $
 * Revision 1.9  2002/05/14 06:38:44  paulk
 * Enhances code generation wizards for implementing interfaces, abstract
 * classes, etc., to use customizable templates to generate skeleton methods
 * instead of hard-wired skeletons. Thanks to "Dr. Michael Lipp" <lipp@danet.de>
 * for proposing and implementing this improvement.
 *
 * Revision 1.8  2001/10/21 14:06:41  paulk
 * Changed some methods from private to protected to permit overriding by descendent classes.
 *
 * Revision 1.7  2001/08/04 03:24:10  paulk
 * DefaultNameFactory.java
 *
 * Revision 1.6  2000/08/01 08:23:22  paulk
 * Fixes bug in dump method. Thanks to Eric Friedman, eric@hfriedman.rdsl.lmi.net.
 *
 *
 */

// End of InterfaceFactory.java
