/* W32 Communication module.
   Copyright (C) 1989, 1993, 1994 Free Software Foundation, Inc.

This file is part of GNU Emacs.

GNU Emacs is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <signal.h>

#include <config.h>

#include <stdio.h>

#include "lisp.h"
#include "blockinput.h"
#ifdef MEADOW
#include "mw32sync.h"
#endif
#include <shellapi.h>

#include "frame.h"
#include "mw32term.h"

#include "charset.h"
#include "mw32font.h"
#include "coding.h"
#ifdef IME_CONTROL
extern void w32_ime_control_init(void);
extern void w32_set_ime_conv_window(struct frame *f);
#endif

#include "systty.h"
#include "systime.h"

#include <ctype.h>
#include <errno.h>
#include <setjmp.h>
#include <sys/stat.h>
#include <sys/param.h>

#include <assert.h>

#include "dispextern.h"
#include "termhooks.h"
#include "termopts.h"
#include "termchar.h"
#include "gnu.h"

#include "disptab.h"
#include "buffer.h"
#include "window.h"
#include "keyboard.h"

/* To handle C-g quickly, this event handle is
   pulsed when C-g is detected. */
HANDLE interrupt_handle;

/* The id of a bitmap used for icon windows.
   One such map is shared by all Emacs icon windows.
   This is zero if we have not yet had a need to create the bitmap.  */

static HBITMAP icon_bitmap;

/* Stuff for dealing with the main icon title. */

extern Lisp_Object Vcommand_line_args, Vsystem_name;
char *w32_id_name;

/* for special input (monopolistic & exclusive input use only) by himi */
int w32_special_input;

/* If the system has already created frame, this variable is set to
   its window handle.  This variable is used for sending message to
   window procedure. */
HWND w32_frame_window;

/* Initial values of argv and argc.  */
extern char **initial_argv;
extern int initial_argc;

/* The cursor to use for vertical scroll bars on w32_current_display.  */
static HCURSOR w32_vertical_scroll_bar_cursor;

/* Frame being updated by update_frame.  This is declared in term.c.
   This is set by update_begin and looked at by all the
   XT functions.  It is zero while not inside an update.
   In that case, the XT functions assume that `selected_frame'
   is the frame to apply to.  */
extern struct frame *updating_frame;

/* The frame (if any) which has the X window that has keyboard focus.
   Zero if none.  This is examined by Ffocus_frame in frame.c.  Note
   that a mere EnterNotify event can set this; if you need to know the
   last frame specified in a FocusIn or FocusOut event, use
   w32_focus_event_frame.  */
struct frame *w32_focus_frame;

/* This is a frame waiting to be autoraised, within W32read_socket.  */
struct frame *pending_autoraise_frame;

/* The last frame mentioned in a FocusIn or FocusOut event.  This is
   separate from w32_focus_frame, because whether or not LeaveNotify
   events cause us to lose focus depends on whether or not we have
   received a FocusIn event for it.  */
struct frame *w32_focus_event_frame;

/* The frame which currently has the visual highlight, and should get
   keyboard input (other sorts of input have the frame encoded in the
   event).  It points to the X focus frame's selected window's
   frame.  It differs from w32_focus_frame when we're using a global
   minibuffer.  */
static struct frame *w32_highlight_frame;

/* From .Xdefaults, the value of "emacs.WarpMouse".  If non-zero,
   mouse is moved to inside of frame when frame is de-iconified.  */

static int warp_mouse_on_deiconify;

/* During an update, maximum vpos for ins/del line operations to affect.  */

static int flexlines;

/* During an update, nonzero if chars output now should be highlighted.  */

static int highlight;

/* Modifier Key list.  ....
   0 is a normal key, 1 is a neglected key, 2 is meta modifier,
   4 is ctrl modifier, 8 is shift modifier, 16 is alt modifier,
   32 is super modifier. 64 is hyper modifier.  
   128 is a special key that is translated by windows. 
   (ToAscii translate it to normal key).
   It is noted to prevent it.  */
/*0   1   2   3   4   5   6   7   8   9   a   b   c   d   e   f   */
static BYTE keymodifier[256] =
{
  0,  0,  0,  0,  0,  0,  0,  0,128,128,  0,  0,  0,  0,  0,  0,  /* 0x00-0x0f */
  8,  4,  2,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  /* 0x10-0x1f */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  /* 0x20-0x2f */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0x30-0x3f */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0x40-0x4f */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0x50-0x5f */
128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,  /* 0x60-0x6f */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0x70-0x7f */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0x80-0x8f */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0x90-0x9f */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0xa0-0xaf */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0xb0-0xbf */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0xc0-0xcf */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0xd0-0xdf */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0xe0-0xef */
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 0xf0-0xff */
};

/* Nominal cursor position -- where to draw output.
   During an update, these are different from the cursor-box position.  */

static int curs_x;
static int curs_y;

/* Mouse movement.

   In order to avoid asking for motion events and then throwing most
   of them away or busy-polling the server for mouse positions, we ask
   the server for pointer motion hints.  This means that we get only
   one event per group of mouse movements.  "Groups" are delimited by
   other kinds of events (focus changes and button clicks, for
   example), or by XQueryPointer calls; when one of these happens, we
   get another MotionNotify event the next time the mouse moves.  This
   is at least as efficient as getting motion events when mouse
   tracking is on, and I suspect only negligibly worse when tracking
   is off.

   The silly O'Reilly & Associates Nutshell guides barely document
   pointer motion hints at all (I think you have to infer how they
   work from an example), and the description of XQueryPointer doesn't
   mention that calling it causes you to get another motion hint from
   the server, which is very important.  */

/* Where the mouse was last time we reported a mouse event.  */
static FRAME_PTR last_mouse_frame;
static RECT last_mouse_glyph;

#if 0 /* modified by himi controled by Windows */
/* The scroll bar in which the last X motion event occurred.

   If the last X motion event occurred in a scroll bar, we set this
   so XTmouse_position can know whether to report a scroll bar motion or
   an ordinary motion.

   If the last X motion event didn't occur in a scroll bar, we set this
   to Qnil, to tell XTmouse_position to return an ordinary motion event.  */
static Lisp_Object last_mouse_scroll_bar;
#endif

#ifdef W32_INTELLIMOUSE
static UINT w32_wheel_message = WM_MOUSEWHEEL;
#endif

/* to prevent thread from operating messages.  */
CRITICAL_SECTION critsec_message;

/* to prevent thread from accessing events.  */
CRITICAL_SECTION critsec_access_event;

/* make thread waiting for gobbling input.  */
HANDLE next_message_block_event;

/* flag if mw32term is blocked in message loop. */
int message_loop_blocked_p;

/* Record which buttons are currently pressed.  */
unsigned int w32_mouse_grabbed;

/* 0 means mouse cursor is visible, -1 is invisible, 1 means locked visible. */
static int mouse_cursor_stat = 0;

/* Message Handling thread & its id.  */
HANDLE msg_thread;
DWORD msg_thread_id;

/* Main thread & its id. */
HANDLE main_thread;
DWORD main_thread_id;


/* This is a hack.  We would really prefer that XTmouse_position would
   return the time associated with the position it returns, but there
   doesn't seem to be any way to wrest the timestamp from the server
   along with the position query.  So, we just keep track of the time
   of the last movement we received, and return that in hopes that
   it's somewhat accurate.  */
static Time last_mouse_movement_time;

/* For synchronous note_mouse_movement.  */
/* last_mouse_movement_frame maintains the frame where a mouse moves last time.
   Its value is set to NULL when a mouse is not moved.  */
static FRAME_PTR last_mouse_movement_frame = NULL;

/* last_mouse_movement_message describes the mouse movement last time.  */
static MSG last_mouse_movement_message;

/* These variables describe the range of text currently shown
   in its mouse-face, together with the window they apply to.
   As long as the mouse stays within this range, we need not
   redraw anything on its account.  */
static int mouse_face_beg_row, mouse_face_beg_col;
static int mouse_face_end_row, mouse_face_end_col;
static int mouse_face_past_end;
static Lisp_Object mouse_face_window;
static int mouse_face_face_id;

/* 1 if a mouse motion event came and we didn't handle it right away because
   gc was in progress.  */
static int mouse_face_deferred_gc;

/* FRAME and X, Y position of mouse when last checked for
   highlighting.  X and Y can be negative or out of range for the frame.  */
static FRAME_PTR mouse_face_mouse_frame;
static int mouse_face_mouse_x, mouse_face_mouse_y;

/* Nonzero means defer mouse-motion highlighting.  */
static int mouse_face_defer;

/* `t' if a mouse button is depressed. */

extern Lisp_Object Vmouse_depressed;

/* Tells if a window manager is present or not. */

extern Lisp_Object Vw32_no_window_manager;

/* Timestamp that we requested selection data was made. */
extern Time requestor_time;

extern Lisp_Object Qface, Qmouse_face;

/* From time to time we get info on an Emacs window, here.  */

extern int errno;

/* A mask of extra modifier bits to put into every keyboard char.  */
extern int extra_keyboard_modifiers;

static void flashback ();
static void redraw_previous_char ();
static void redraw_following_char ();
static unsigned int x_w32_to_emacs_modifiers ();

static int fast_find_position ();
static void note_mouse_highlight ();
static void clear_mouse_face ();
static void show_mouse_face ();

void dumpborder ();
static void W32cursor_to ();
static void W32clear_end_of_line ();

void w32_display_cursor ();
void x_set_window_size ();
static void w32_raise_frame ();
static void w32_lower_frame ();

void lock_mouse_cursor_visible(lock)
     int lock;	/* 0 : unlock, 1 : lock visible */
{
  if (lock)
    {
      if (mouse_cursor_stat < 0)
	ShowCursor (TRUE);
      mouse_cursor_stat = 1;
    }
  else
    if (mouse_cursor_stat == 1)
      mouse_cursor_stat = 0;
}

void w32_fill_rect (f, _hdc, pix, lprect)
     FRAME_PTR f;
     HDC _hdc;
     PIX_TYPE pix;
     RECT * lprect;
{
  HDC hdc;
  HBRUSH hb;
  HANDLE oldobj;
  RECT rect;

  int restore_caret = 0;

  /* First of all, hide caret. */
  if (f->output_data.w32->caret_state == 2)
    {
      SEND_INFORM_MESSAGE (FRAME_W32_WINDOW(f), WM_EMACS_SETCARET, 0, 0);
      restore_caret = 1;
    }

  if (_hdc)
    hdc = _hdc;
  else
    {
      if (!f) return;
      hdc = MyGetDC (FRAME_W32_WINDOW (f));
    }

  hb = CreateSolidBrush (pix);
  oldobj = SelectObject (hdc, hb);
    
  FillRect (hdc, lprect, hb);
    
  SelectObject (hdc, oldobj);
    
  DeleteObject (hb);

  if (!_hdc)
    ReleaseDC (FRAME_W32_WINDOW(f), hdc);

  if (restore_caret)
    SEND_INFORM_MESSAGE (FRAME_W32_WINDOW(f), WM_EMACS_SETCARET, 1, 0);
}


/* Starting and ending updates. 

   These hooks are called by update_frame at the beginning and end
   of a frame update.  We record in `updating_frame' the identity
   of the frame being updated, so that the XT... functions do not
   need to take a frame as argument.  Most of the XT... functions
   should never be called except during an update, the only exceptions
   being W32cursor_to, W32write_glyphs and W32reassert_line_highlight.  */

extern int mouse_track_top, mouse_track_left, mouse_track_width;

static void
W32update_begin (f)
     struct frame *f;
{	
  int mask;

  if (f == 0)
    abort ();

  flexlines = f->height;
  highlight = 0;

  BLOCK_INPUT;

  /* Before updating, hide caret. */
  if (f->output_data.w32->caret_state == 2)
    SEND_INFORM_MESSAGE (FRAME_W32_WINDOW(f), WM_EMACS_SETCARET, 0, 0);

  if (f == mouse_face_mouse_frame)
    {
      /* Don't do highlighting for mouse motion during the update.  */
      mouse_face_defer = 1;
      if (!NILP (mouse_face_window))
	{
	  int firstline, lastline, i;
	  struct window *w = XWINDOW (mouse_face_window);

	  /* Find the first, and the last+1, lines affected by redisplay.  */
	  for (firstline = 0; firstline < f->height; firstline++)
	    if (FRAME_DESIRED_GLYPHS (f)->enable[firstline])
	      break;

	  lastline = f->height;
	  for (i = f->height - 1; i >= 0; i--)
	    {
	      if (FRAME_DESIRED_GLYPHS (f)->enable[i])
		break;
	      else
		lastline = i;
	    }

	  /* Can we tell that this update does not affect the window
	     where the mouse highlight is?  If so, no need to turn off.  */
	  if (! (firstline > (XFASTINT (w->top) + window_internal_height (w))
		 || lastline < XFASTINT (w->top)))
	    /* Otherwise turn off the mouse highlight now.  */
	    clear_mouse_face ();
	}
    }
#if 0
  dumpqueue ();
#endif /* HAVE_X11 */
  UNBLOCK_INPUT;
  W32_BLOCK_INPUT;
}

#if 0
static void w32_do_pending_expose ();
#endif

static void
W32update_end (f)
     struct frame *f;
{	
  int mask;
  W32_UNBLOCK_INPUT;

  BLOCK_INPUT;
#if 0
  dumpqueue ();
  w32_do_pending_expose ();
#endif /* HAVE_X11 */

  w32_display_cursor (f, 1);

  /* restore caret state if caret_cursor is enabled. */
  if (FRAME_CURRENT_CURSOR (f) == caret_cursor ||
      FRAME_CURRENT_CURSOR (f) == checkered_caret_cursor ||
      FRAME_CURRENT_CURSOR (f) == hairline_caret_cursor)
    SEND_INFORM_MESSAGE (FRAME_W32_WINDOW(f), WM_EMACS_SETCARET, 1, 0);

  if (f == mouse_face_mouse_frame)
    mouse_face_defer = 0;

  UNBLOCK_INPUT;
}

/* This is called after a redisplay on frame F.  */

static void
W32frame_up_to_date (f)
     FRAME_PTR f;
{
  if (mouse_face_deferred_gc || f == mouse_face_mouse_frame)
    {
      note_mouse_highlight (mouse_face_mouse_frame,
			    mouse_face_mouse_x, mouse_face_mouse_y);
      mouse_face_deferred_gc = 0;
    }
}

/* External interface to control of standout mode.
   Call this when about to modify line at position VPOS
   and not change whether it is highlighted.  */

static void
W32reassert_line_highlight (new, vpos)
     int new, vpos;
{
  highlight = new;
}

/* Call this when about to modify line at position VPOS
   and change whether it is highlighted.  */

static void
W32change_line_highlight (new_highlight, vpos, first_unused_hpos)
     int new_highlight, vpos, first_unused_hpos;
{
  highlight = new_highlight;
  W32cursor_to (vpos, 0);
  W32clear_end_of_line (updating_frame->width);
}

/* This is used when starting Emacs and when restarting after suspend.
   When starting Emacs, no X window is mapped.  And nothing must be done
   to Emacs's own window if it is suspended (though that rarely happens).  */

static void
W32set_terminal_modes ()
{
}

/* This is called when exiting or suspending Emacs.
   Exiting will make the X-windows go away, and suspending
   requires no action.  */

static void
W32reset_terminal_modes ()
{
/*  W32clear_frame ();  */
}

/* Set the nominal cursor position of the frame.
   This is where display update commands will take effect.
   This does not affect the place where the cursor-box is displayed.  */

static void
W32cursor_to (row, col)
     register int row, col;
{
  curs_x = col;
  curs_y = row;

  if (updating_frame == 0)
    {
      BLOCK_INPUT;
      w32_display_cursor (selected_frame, 1);
      UNBLOCK_INPUT;
    }
}


#define PER_CHAR_METRIC_TTF(lpgm, hdc, buf, charset)                       \
((CHARSET_DIMENSION(charset) == 1)?                                        \
 (GetGlyphOutline((hdc), (UINT) *((unsigned char*)(buf)),                  \
		  GGO_METRICS, (lpgm), 0, NULL, NULL)) :                   \
 (GetGlyphOutline((hdc), (UINT) MAKELENDSHORT((buf)[0], (buf)[1]),         \
		  GGO_METRICS, (lpgm), 0, NULL, NULL)))

#define PER_CHAR_METRIC_BDF(lpglyph, fontp, buf, charset)                  \
((CHARSET_DIMENSION(charset) == 1)?                                        \
 (mw32_get_bdf_glyph((fontp), *((unsigned char*)(buf)), 0, (lpglyph))) :   \
 (mw32_get_bdf_glyph((fontp), MAKELENDSHORT((buf)[1], (buf)[0]), 0, (lpglyph))))

/* Display a sequence of N glyphs found at GP.
   WINDOW is the x-window to output to.  LEFT and TOP are starting coords.
   HL is 1 if this text is highlighted, 2 if the cursor is on it,
   3 if should appear in its mouse-face.
   JUST_FOREGROUND if 1 means draw only the foreground;
   don't alter the background.

   FONT is the default font to use (for glyphs whose font-code is 0).

   Since the display generation code is responsible for calling
   compute_char_face and compute_glyph_face on everything it puts in
   the display structure, we can assume that the face code on each
   glyph is a valid index into FRAME_COMPUTED_FACES (f), and the one
   to which we can actually apply intern_face.
   Call this function with input blocked.  */

/* This is the multi-face code.  */

#define W32_MAX_BYTE_PER_CHARACTER 4

static void
dumpglyphs (f, left, top, gp, n, hl, just_foreground, cmpcharp)
     struct frame *f;
     int left, top;
     register GLYPH *gp; /* Points to first GLYPH. */
     register int n;  /* Number of glyphs to display. */
     int hl;
     int just_foreground;
     struct cmpchar_info *cmpcharp;
{
  /* Holds characters to be displayed.  */
  BYTE *bufs = (TCHAR *) alloca (FRAME_WINDOW_WIDTH (f) *
				 W32_MAX_BYTE_PER_CHARACTER);
  BYTE *bufd = (TCHAR *) alloca (FRAME_WINDOW_WIDTH (f) *
				 W32_MAX_BYTE_PER_CHARACTER);
  BYTE *cpend;
  register BYTE *cps, *cpd;		/* Steps through buf?[]. */
  register int tlen = GLYPH_TABLE_LENGTH;
  register Lisp_Object *tbase = GLYPH_TABLE_BASE;
  HWND hwnd = FRAME_W32_WINDOW (f);
  HDC hdc = INVALID_HANDLE_VALUE;
  HFONT hfont;
  int orig_left = left;
  int gidx = 0;
  int pixel_width;

  while (n > 0)
    {
      /* Get the face-code of the next GLYPH.  */
      int cf, len;
      GLYPH g = *gp;
      int ch, charset;
      Lisp_Object first_ch;
      /* HIGHEST and LOWEST are used while drawing a composite
         character.  The meanings are described later.  */
      int highest, lowest;

      GLYPH_FOLLOW_ALIASES (tbase, tlen, g);
      cf = (cmpcharp ? cmpcharp->face_work : FAST_GLYPH_FACE (g));
      ch = FAST_GLYPH_CHAR (g);
      if (gidx == 0) XSETFASTINT (first_ch, ch);
      charset = CHAR_CHARSET (ch);
      if (charset == CHARSET_COMPOSITION)
	{
	  /* We must draw components of the composite character on the
             same column.  */
	  cmpcharp = cmpchar_table[COMPOSITE_CHAR_ID (ch)];

	  /* Set the face in the slot for work.  */
	  cmpcharp->face_work = cf;

	  /* We don't need the return value ... */
	  dumpglyphs (f, left, top, cmpcharp->glyph, cmpcharp->glyph_len,
		      hl, just_foreground, cmpcharp);
	  /* ... because the width of just drawn text can be
             calculated as follows.  */
	  left += FRAME_FONT_WIDTH (f) * cmpcharp->width;

	  ++gp, --n;
	  while (gp && (*gp & GLYPH_MASK_PADDING)) ++gp, --n;
	  cmpcharp = NULL;
	  continue;
	}

      /* Find the run of consecutive glyphs which can be drawn with
	 the same Device Context (i.e. the same charset and the same face-code).
	 Extract their character codes into BUF.
	 If CMPCHARP is not NULL, face-code is not checked because we
	 use only the face specified in `cmpcharp->face_work'.  */
      /* In the case of ccl translation, a ccl program chanages
	 number of bytes 1 character use.  So we need to prepare two buffers,
	 one is used for raw glyph character code,
	 the other is used for result ccl program translates.  */
      cps = bufs;
      cpd = bufd;
      len = 0;
      while (n > 0)
	{
	  int this_charset, c1, c2;

	  g = *gp;
	  GLYPH_FOLLOW_ALIASES (tbase, tlen, g);
	  ch = FAST_GLYPH_CHAR (g);
	  SPLIT_CHAR (ch, this_charset, c1, c2);
	  if (this_charset != charset
	      || (cmpcharp == NULL && FAST_GLYPH_FACE (g) != cf))
	    break;

	  *cps++ = c1;
	  if (c2 > 0)
	    *cps++ = c2;
	  ++len, ++gp, --n;
	  while (gp && (*gp & GLYPH_MASK_PADDING))
	    ++gp, --n;
	}
      cpend = cps;

      /* PIXEL_WIDTH get the pixcel width of the run.  */
      pixel_width
	= (FRAME_FONT_WIDTH (f)
	   * (cmpcharp ? cmpcharp->width : len * CHARSET_WIDTH (charset)));

      /* Now output this run of chars, with the font and pixel values
	 determined by the face code CF.  */
      {
	struct face *face = FRAME_DEFAULT_FACE (f);
	W32FontStruct *font = NULL;
	int stippled = 0;
	/* A flag to tell if we have already filled background.  We
	   fill background in advance in the following cases:
	   1) A face has stipple.
	   2) A height of font is different from that of the current line.
	   3) Drawing a composite character.
	   4) Font has non-zero _MULE_BASELINE_OFFSET property.
	   After filling background, we draw glyphs by XDrawString16.  */
	/* Baseline position of a character, offset from TOP.  */
	int baseline;
	/* The property value of `_MULE_RELATIVE_COMPOSE' and
           `_MULE_DEFAULT_ASCENT'.  */
	int relative_compose = 0, default_ascent = 0;
	PIX_TYPE fg;
	PIX_TYPE bg;
	/* to indicate whether the selected font is truetype or not.  */
	int ttfflag = 0;
	/* points to bdffont structure if the selected font is BDF font.  */
	bdffont *bdffontp = NULL;
	/* to indicate text area to ExtTextOut */
	RECT r;

	/* charset dimension */
	int charset_dimension = CHARSET_DIMENSION(charset);

	/* number of encoding bytes of font;
	   If this is 0, the length of unit bytes is variable. */
	int font_unit_bytes = charset_dimension;

#ifdef MEADOW
	int property;
#endif 
	int encoding = 0;

	if (hdc == INVALID_HANDLE_VALUE)
	  hdc = MyGetDC(hwnd);

	/* HL = 3 means use a mouse face previously chosen.  */
	if (hl == 3)
#ifdef MEADOW
	  cf = mouse_face_face_id;
#else
	  cf = FRAME_X_DISPLAY_INFO (f)->mouse_face_face_id;
#endif

	/* First look at the face of the text itself.  */
	if (cf != 0)
	  {
	    /* It's possible for the display table to specify
	       a face code that is out of range.  Use 0 in that case.  */
	    if (cf < 0 || cf >= FRAME_N_COMPUTED_FACES (f)
		|| FRAME_COMPUTED_FACES (f) [cf] == 0)
	      cf = 0;

	    if (cf == 1)
	      face = FRAME_MODE_LINE_FACE (f);
	    else
	      face = intern_face (f, FRAME_COMPUTED_FACES (f) [cf]);
	    if (FACE_STIPPLE (face))
	      stippled = 1;
	  }

	/* Then comes the distinction between modeline and normal text.  */
	else if (hl == 0)
	  ;
	else if (hl == 1)
	  {
	    face = FRAME_MODE_LINE_FACE (f);
	    if (FACE_STIPPLE (face))
	      stippled = 1;
	  }
#ifdef MEADOW
	property = FACE_PROPERTY(face);
#endif	
	fg = face->foreground;
	bg = face->background;	

	/* Setting appropriate font and Device Context for this charset.  */
	if (charset != CHARSET_ASCII)
	  {
	    int font_id;
	    int fontset = FACE_FONTSET (face);
	    struct font_info *fontp;

	    if ((fontset < 0 && (fontset = FRAME_FONTSET (f)) < 0)
		|| !(fontp = FS_LOAD_FONT (f, W32_FONT_INFO_TABLE,
					   charset, NULL, fontset)))
	      goto font_not_found;

	    font = (W32FontStruct *) (fontp->font);
	    encoding = fontp->encoding[charset];

	    baseline = FONT_BASE(font);
	    if (cmpcharp && cmpcharp->cmp_rule == NULL)
	      {
		relative_compose = fontp->relative_compose;
		default_ascent = fontp->default_ascent;
	      }

	    /* We have to change code points in the following cases.  */
	    if (fontp->font_encoder)
	      {
		/* This font requires CCL program to calculate code
                   point of characters.  */
		struct ccl_program *ccl = fontp->font_encoder;
		int i;
		
		if (encoding == ENCODING_BYTE1MSB1)
		  font_unit_bytes = 1;
		else if ((encoding == ENCODING_BYTE2MSB1) ||
			 (encoding == ENCODING_BYTE2LSB1) ||
			 (encoding == ENCODING_UNICODE))
		  font_unit_bytes = 2;

		if (charset_dimension == 1)
		  for (cps = bufs;cps < cpend;cps++)
		    {
		      ccl->reg[0] = charset;
		      ccl->reg[1] = cps[0];
		      ccl_driver (ccl, NULL, NULL, 0, 0, NULL);
		      for (i = 1;i <= font_unit_bytes;i++)
			{
			  *cpd++ = ccl->reg[i];
			}
		    }
		else
		  for (cps = bufs;cps < cpend;)
		    {
		      ccl->reg[0] = charset;
		      ccl->reg[1] = cps[0], ccl->reg[2] = cps[1];
		      ccl_driver (ccl, NULL, NULL, 0, 0, NULL);
		      cps += 2;
		      for (i = 1;i <= font_unit_bytes;i++)
			{
			  *cpd++ = ccl->reg[i];
			}
		    }

		cpend = cpd;
		cps = bufd;
	      }
	    else
	      {
		switch(encoding)
		  {
		  case ENCODING_BYTE1MSB1:
		    font_unit_bytes = 1;
		    for (cps = bufs; cps < cpend; cps++)
		      *cps |= 0x80;
		    cps = bufs;
		    break;
		  case ENCODING_BYTE2MSB1:
		    font_unit_bytes = 2;
		    if (charset_dimension == 2)
		      {
			for (cps = bufs; cps < cpend;)
			  {
			    *cps |= 0x80;
			    cps += 2;
			  }
		      }
		    cps = bufs;
		    break;
		  case ENCODING_BYTE2LSB1:
		    font_unit_bytes = 2;
		    if (charset_dimension == 2)
		      {
			for (cps = bufs + 1; cps < cpend;)
			  {
			    *cps |= 0x80;
			    cps += 2;
			  }
		      }
		    cps = bufs;
		    break;

		  case ENCODING_SHIFTJIS:
		    if (charset_dimension == 2)
		      {
			int s1, s2;
			for (cps = bufs; cps < cpend;)
			  {
			    ENCODE_SJIS(cps[0], cps[1], s1, s2);
			    *cps++ = s1;
			    *cps++ = s2;
			  }
		      }
		    else
		      for (cps = bufs; cps < cpend; cps++)
			*cps |= 0x80;
		    cps = bufs;
		    break;
		  case ENCODING_UNICODE:
		    font_unit_bytes = 2;
		    if (charset_dimension == 2)
		      {
			int fc;
			for (cps = bufs; cps < cpend;)
			  {
			    fc = ((cps[0] - 32) * CHARSET_CHARS(charset)
				  + (cps[1] - 32));
			    *cps++ = ((fc >> 8) & 0xFF);
			    *cps++ = (fc & 0xFF);
			  }
			cps = bufs;
		      }
		    else
		      {
			for (cpd = bufd, cps = bufs; cps < cpend;)
			  {
			    *cpd++ = 0;
			    *cpd++ = *cps++ - 32;
			  }
			cpend = cpd;
			cps = bufd;
		      }
		  default:
		    cps = bufs;
		    break;
		  }
	      }
	  }
	else
	  {
	    font = FACE_FONT (face);
	    baseline = FONT_BASE (font);
	  font_not_found:;
	    cps = bufs;
	  }

#define FACE_DEFAULT (~0)

	/* Now override that if the cursor's on this character.  */
	if (hl == 2)
	  {
	    unsigned long mask;
	    /* The cursor overrides stippling.  */
	    stippled = 0;
	    
	    bg = f->output_data.w32->cursor_pixel;
	    fg = face->background;
	    /* If the glyph would be invisible,
	       try a different foreground.  */
	    if (fg == bg)
	      fg = face->foreground;
	    if (fg == bg)
	      fg = f->output_data.w32->cursor_foreground_pixel;
	    if (fg == bg)
	      fg = face->foreground;
	    /* Make sure the cursor is distinct from text in this face.  */
	    if (bg == face->background
		&& fg == face->foreground)
	      {
		bg = face->foreground;
		fg = face->background;
	      }
#if 0
	    /* If this code is restored, it must also reset to the default stipple
	       if necessary. */
	    if (face->stipple && face->stipple != FACE_DEFAULT)
	      XSetStipple (FRAME_X_DISPLAY (f), gc, face->stipple);
#endif
	  }
	if (font == (W32FontStruct *) FACE_DEFAULT)
	  font = f->output_data.w32->font;

	if (font)
	  {
#ifdef MEADOW
	    hfont = FONT_HFONT(font, property);
#else
	    hfont = FONT_HFONT(font, 0);
#endif
	    if (hfont == INVALID_HANDLE_VALUE)
	      {
		bdffontp = (bdffont*) get_logfont(font, property);
	      }
	    else
	      SelectObject(hdc, hfont);
	  }

	SetTextColor(hdc, fg);
	SetBkColor(hdc, bg);
	SetTextAlign(hdc, TA_BASELINE | TA_LEFT);
	SetBkMode (hdc, TRANSPARENT);

	r.left = left;
	r.top = top;
	r.bottom = top + f->output_data.w32->line_height - 1;
	r.right = left + pixel_width + FRAME_FONT_OVERHANG(f) - 1;

	if (!(just_foreground || (cmpcharp && gidx > 0)))
	  {
	    POINT points[6];
	    TEXTMETRIC tm;
	    LOGBRUSH logpenbrush;
	    HPEN hpen, holdpen;
	    HBRUSH hbrush, holdbrush;
	    logpenbrush.lbStyle = BS_SOLID;
	    logpenbrush.lbColor = bg;
	    hpen = ExtCreatePen(PS_COSMETIC | PS_SOLID, 1, &logpenbrush,
				0, NULL);
	    hbrush = CreateSolidBrush(bg);
	    
	    GetTextMetrics(hdc, &tm);
	    /* this is the edge of the window. */
	    if (left < FRAME_FONT_WIDTH(f))
	      {
		points[0].x = left;
		points[1].x = points[0].x + pixel_width - 1;
	      }
	    else
	      {
		if (bdffontp)
		  {
		    points[0].x = left;
		    points[1].x = points[0].x + pixel_width - 1;
		  }
		/* this font is TrueType and has propotional width. */
		else if ((tm.tmPitchAndFamily & (TMPF_TRUETYPE | TMPF_FIXED_PITCH))
			 == (TMPF_TRUETYPE | TMPF_FIXED_PITCH))
		  {
		    ABC abcwidth_first;
		    ABC abcwidth_last;
		    int first_char, last_char;

		    ttfflag = 1;
		    
		    if ((font_unit_bytes == 2)
			&& ((cpend - cps) >= 2))
		      {
			first_char = MAKELENDSHORT(cps[1], cps[0]);
			last_char = MAKELENDSHORT(*(cpend - 1), *(cpend - 2));
		      }
		    else
		      {
			first_char = cps[0];
			last_char = *(cpend - 1);
		      }
		    GetCharABCWidths(hdc, first_char, first_char, &abcwidth_first);
		    GetCharABCWidths(hdc, last_char, last_char, &abcwidth_last);
		    points[0].x = left + abcwidth_first.abcA - 1;
		    if (abcwidth_last.abcB <= 1) /* whitespace */
		      points[1].x = left + pixel_width - 1;
		    else
		      points[1].x = left + pixel_width - abcwidth_last.abcC - 1;
		  }
		else
		  {
		    points[0].x = left + tm.tmOverhang;
		    points[1].x = points[0].x + pixel_width - 1;
		  }
	      }
	    points[0].y = top;
	    points[1].y = top;
	    points[2].x = left + pixel_width - 1;
	    if (font)
	      points[2].y = top + FONT_HEIGHT(font) - 1;
	    else
	      points[2].y = top + FRAME_FONT_HEIGHT(f) - 1;
	    points[3].x = points[2].x;
	    points[3].y = r.bottom;
	    points[4].x = left;
	    points[4].y = r.bottom;
	    if (points[2].y > r.bottom)
	      points[2].y = r.bottom;
	    points[5].x = left;
	    points[5].y = points[2].y;
	    holdpen = SelectObject(hdc, hpen);
	    holdbrush = SelectObject(hdc, hbrush);
	    Polygon(hdc, points, 6);
	    SelectObject(hdc, holdpen);
	    SelectObject(hdc, holdbrush);
	    DeleteObject(hpen);
	    DeleteObject(hbrush);
	    if (cmpcharp)
	      /* To assure not to fill background while drawing
		 remaining components.  */
	      just_foreground = 1;
	  }

	if (font)
	  {
	    if (cmpcharp)
	      {
		GLYPHMETRICS gm; /* Pointer to per char metric info.  */
		glyph_struct glyph;
		int i;

		if ((!cmpcharp->cmp_rule) && (relative_compose < 0))
		  {
		    /* On this character,
		       we entrast all rendering to system. */
		    int *lpdx;
		    int i;
		    int bytelen = cpend - cps;

		    lpdx = alloca(sizeof(int) * bytelen);
		    for(i = 0;i < bytelen;i++) lpdx[i] = 0;
		    if (encoding == ENCODING_UNICODE)
		      ExtTextOutW(hdc, left, top + baseline,
				  0, &r, cps, bytelen, lpdx);
		    else
		      ExtTextOutA(hdc, left, top + baseline,
				  0, &r, cps, bytelen, lpdx);
		  }
		else
		  {
		    if ((cmpcharp->cmp_rule || relative_compose)
			&& gidx == 0)
		      {
			/* This is the first character.  Initialize variables.
			   HIGHEST is the highest position of glyphs ever
			   written, LOWEST the lowest position.  */
			int x_offset = 0;

			if (default_ascent
			    && CHAR_TABLE_P (Vuse_default_ascent)
			    && !NILP (Faref (Vuse_default_ascent, first_ch)))
			  {
			    highest = default_ascent;
			    lowest = 0;
			  }
			else
			  {
			    if (ttfflag)
			      {
				PER_CHAR_METRIC_TTF (&gm, hdc, cps, charset);
				highest = gm.gmptGlyphOrigin.y;
				lowest = highest - gm.gmBlackBoxY;
				highest++;
			      }
			    else if ((bdffontp) &&
				     PER_CHAR_METRIC_BDF (&glyph, 
							  bdffontp, 
							  cps,
							  charset))
			      {
				highest = glyph.metric.bbh + glyph.metric.bboy + 1;
				lowest = glyph.metric.bboy;
			      }
			    else
			      {
				highest = FONT_BASE(font) + 1;
				lowest = -(FONT_HEIGHT(font) - FONT_BASE(font));
			      }
			  }

			if (cmpcharp->cmp_rule)
			  x_offset = (int)(cmpcharp->col_offset[0]
					   * FRAME_FONT_WIDTH (f));
			/* Draw the first character at the normal position.  */
			if (bdffontp)
			  mw32_BDF_TextOut(bdffontp, hdc, left + x_offset,
					   top + baseline, cps,
					   font_unit_bytes,
					   font_unit_bytes, 0);
			else
			  if (encoding == ENCODING_UNICODE)
			    ExtTextOutW(hdc, left + x_offset, top + baseline,
					0, &r, cps, font_unit_bytes, NULL);
			  else
			    ExtTextOutA(hdc, left + x_offset, top + baseline,
					0, &r, cps, font_unit_bytes, NULL);
			i = 1;
			cps += font_unit_bytes;
			gidx++;
		      }
		    else
		      i = 0;

		    for (;i < len; i++, gidx++)
		      {
			int x_offset = 0, y_offset = 0;

			/* these operations is to detect subscript char, but 
			   it is no use on good designed fonts */
			if (relative_compose &&
			    bdffontp &&
			    PER_CHAR_METRIC_BDF (&glyph,
						 bdffontp,
						 cps,
						 charset))
			  {
			    if (glyph.metric.bboy >= relative_compose)
			      {
				/* Draw above the current glyphs.  */
				y_offset = highest - glyph.metric.bboy;
				highest += glyph.metric.bbh;
			      }
			    else if ((glyph.metric.bbh + glyph.metric.bboy) <= 0)
			      {
				/* Draw beneath the current glyphs.  */
				y_offset = lowest - (glyph.metric.bbh + glyph.metric.bboy);
				lowest -= glyph.metric.bbh;
			      }
			  }
			else if (cmpcharp->cmp_rule)
			  {
			    int gref = (cmpcharp->cmp_rule[gidx] - 0xA0) / 9;
			    int nref = (cmpcharp->cmp_rule[gidx] - 0xA0) % 9;
			    int bottom, top, ascent, descent, height;

			    /* Re-encode GREF and NREF so that they specify
			       only Y-axis information:
			       0:top, 1:base, 2:bottom, 3:center  */
			    gref = gref / 3 + (gref == 4) * 2;
			    nref = nref / 3 + (nref == 4) * 2;

			    if (ttfflag)
			      {
				PER_CHAR_METRIC_TTF (&gm, hdc, cps, charset);
				ascent = gm.gmptGlyphOrigin.y;
				height = gm.gmBlackBoxY;
			      }
			    else if (bdffontp &&
				     PER_CHAR_METRIC_BDF (&glyph,
							  bdffontp,
							  cps,
							  charset))
			      {
				ascent = glyph.metric.bbh + glyph.metric.bboy;
				height = glyph.metric.bbh;
			      }
			    else
			      {
				ascent = FONT_BASE(font);
				height = FONT_HEIGHT(font);
			      }
			    descent = height - ascent;
			    bottom = ((gref == 0 ? highest : gref == 1 ? 0
				       : gref == 2 ? lowest
				       : (height) / 2)
				      - (nref == 0 ? height
					 : nref == 1 ? descent : nref == 2 ? 0
					 : (height) / 2));
			    top = bottom + (height);
			    if (top > highest)
			      highest = top;
			    if (bottom < lowest)
			      lowest = bottom;
			    y_offset = bottom + descent;
			    x_offset = (int) (cmpcharp->col_offset[gidx]
					      * FRAME_FONT_WIDTH (f));
			  }
			if (bdffontp)
			  mw32_BDF_TextOut(bdffontp, hdc, left + x_offset,
					   top + baseline - y_offset,
					   cps, font_unit_bytes,
					   font_unit_bytes, 0);
			else
			  if (encoding == ENCODING_UNICODE)
			    ExtTextOutW(hdc, left + x_offset,
					top + baseline - y_offset,
					0, &r, cps, font_unit_bytes, NULL);
			  else
			    ExtTextOutA(hdc, left + x_offset,
					top + baseline - y_offset,
					0, &r, cps, font_unit_bytes, NULL);

			cps += font_unit_bytes;
		      }
		  }
	      }
	    else
	      {
		if (bdffontp)
		  mw32_BDF_TextOut(bdffontp, hdc, left, top + baseline,
				   cps, font_unit_bytes, (cpend - cps),
				   CHARSET_WIDTH (charset) *
				   FRAME_FONT_WIDTH(f));
		else
		  {
		    int *lpdx;
		    int i;
		    int bytelen = cpend - cps;

		    lpdx = alloca(sizeof(int) * bytelen);
		    for(i = 0;i < bytelen;i++) lpdx[i] = FRAME_FONT_WIDTH(f);
		    if (encoding == ENCODING_UNICODE)
		      ExtTextOutW(hdc, left, top + baseline,
				  0, &r, cps, bytelen, lpdx);
		    else
		      ExtTextOutA(hdc, left, top + baseline,
				  0, &r, cps, bytelen, lpdx);

		  }
	      }

#if 0 /* Doesn't work, because it uses FRAME_CURRENT_GLYPHS,
	 which often is not up to date yet.  */
	    if (!just_foreground)
	      {
		if (left == orig_left)
		  redraw_previous_char (f, PIXEL_TO_CHAR_COL (f, left),
					PIXEL_TO_CHAR_ROW (f, top), hl == 1);
		if (n == 0)
		  redraw_following_char (f, PIXEL_TO_CHAR_COL (f, left + len * FONT_WIDTH (font)),
					 PIXEL_TO_CHAR_ROW (f, top), hl == 1);
	      }
#endif
	  }
	else
	  {
	    /* There's no appropriate font for this glyph.  Just show
               rectangles.  */

	    if (cmpcharp)
	      Rectangle(hdc, left, top + 1,
			left + pixel_width - 3,
			top + f->output_data.w32->line_height - 3);
	    else
	      {
		int left_offset;
		int left_skip_step = (FRAME_FONT_WIDTH (f)
				      * CHARSET_WIDTH (charset));

		for (left_offset = 0; left_offset < pixel_width;
		     left_offset += left_skip_step)
		  Rectangle(hdc, left + left_offset, top + 1,
			    left + left_offset + left_skip_step - 2,
			    top + f->output_data.w32->line_height - 2);
	      }
	  }

	/* We should probably check for XA_UNDERLINE_POSITION and
	   XA_UNDERLINE_THICKNESS properties on the font, but let's
	   just get the thing working, and come back to that.  */
	if (FACE_UNDERLINE_P(face))
	  {
	    /* Setting underline position based on the metric of the
	       current font results in shaky underline if it strides
	       over different fonts.  So, we set the position based only
	       on the default font of this frame.  */
	    int underline_position = FRAME_FONT_BASELINE(f) + 1;

	    if (underline_position >= f->output_data.w32->line_height)
	      underline_position = f->output_data.w32->line_height - 1;

	    w32_fill_area(f, hdc, fg, left,
			    top + underline_position, pixel_width, 1);
	  }

	if (!cmpcharp)
	  left += pixel_width;
      }
      ReleaseDC(hwnd, hdc);
      hdc = INVALID_HANDLE_VALUE;
    }

/*  return (left - orig_left); */
}


/* Output some text at the nominal frame cursor position.
   Advance the cursor over the text.
   Output LEN glyphs at START.

   `highlight', set up by XTreassert_line_highlight or W32change_line_highlight,
   controls the pixel values used for foreground and background.  */

static void
W32write_glyphs (start, len)
     register GLYPH *start;
     int len;
{
  register int temp_length;
  int mask;
  struct frame *f;

  BLOCK_INPUT;

  f = updating_frame;

  if (f == 0)
    {
      f = selected_frame;
      /* If not within an update,
	 output at the frame's visible cursor.  */
      curs_x = f->cursor_x;
      curs_y = f->cursor_y;
      /* hide caret if there is no updating frame. */
      SEND_INFORM_MESSAGE (FRAME_W32_WINDOW(f), WM_EMACS_SETCARET, 0, 0);
    }

  dumpglyphs (f,
	      CHAR_TO_PIXEL_COL (f, curs_x),
	      CHAR_TO_PIXEL_ROW (f, curs_y),
	      start, len, highlight, 0, NULL);

  /* If we drew on top of the cursor, note that it is turned off.  */
  if (curs_y == f->phys_cursor_y
      && curs_x <= f->phys_cursor_x
      && curs_x + len > f->phys_cursor_x)
    f->phys_cursor_x = -1;
  
  if (updating_frame == 0)
    {
      f->cursor_x += len;
      w32_display_cursor (f, 1);
      f->cursor_x -= len;
    }
  else
    curs_x += len;

  UNBLOCK_INPUT;
}

/* Clear to the end of the line.
   Erase the current text line from the nominal cursor position (inclusive)
   to column FIRST_UNUSED (exclusive).  The idea is that everything
   from FIRST_UNUSED onward is already erased.  */
  
static void
W32clear_end_of_line (first_unused)
     register int first_unused;
{
  struct frame *f = updating_frame;
  int mask;

  if (f == 0)
    abort ();

  if (curs_y < 0 || curs_y >= f->height)
    return;
  if (first_unused <= 0)
    return;

  if (first_unused >= f->width)
    first_unused = f->width;

  first_unused += FRAME_LEFT_SCROLL_BAR_WIDTH (f);

  BLOCK_INPUT;

  /* Notice if the cursor will be cleared by this operation.  */
  if (curs_y == f->phys_cursor_y
      && curs_x <= f->phys_cursor_x
      && f->phys_cursor_x < first_unused)
    f->phys_cursor_x = -1;

#ifdef MW32_FONT
  w32_clear_area (f, NULL,
		  CHAR_TO_PIXEL_COL (f, curs_x),
		  CHAR_TO_PIXEL_ROW (f, curs_y),
		  FRAME_FONT_WIDTH(f) * (first_unused - curs_x) +
		  FRAME_FONT_OVERHANG(f),
		  f->output_data.w32->line_height);
#else
  w32_clear_area (f, NULL,
		  CHAR_TO_PIXEL_COL (f, curs_x),
		  CHAR_TO_PIXEL_ROW (f, curs_y),
		  FONT_WIDTH (f->output_data.w32->font) * (first_unused - curs_x),
		  f->output_data.w32->line_height);
#endif

  UNBLOCK_INPUT;
}

void w32_scroll_bar_expose(FRAME_PTR f);

static void
W32clear_frame ()
{
  int mask;
  struct frame *f = updating_frame;

  if (f == 0)
    f = selected_frame;

  f->phys_cursor_x = -1;	/* Cursor not visible.  */
  curs_x = 0;			/* Nominal cursor position is top left.  */
  curs_y = 0;
  
  BLOCK_INPUT;

  {
    RECT rect;

    GetClientRect (FRAME_W32_WINDOW(f), &rect);
    w32_clear_rect (f, NULL, &rect);
  }

#if 0
  /* We have to clear the scroll bars, too.  If we have changed
     colors or something like that, then they should be notified.  */
  w32_scroll_bar_clear (f);
#endif

#if 0
  dumpborder (f, 0);
#endif /* HAVE_X11 */

  UNBLOCK_INPUT;
}

extern unsigned int sound_type;

/* Make audible or visible bell.  */
static void
W32ring_bell ()
{
  extern void w32_sys_ring_bell (void);

  if (visible_bell)
    {
      if (w32_focus_frame)
	{
	  POST_INFORM_MESSAGE (FRAME_W32_WINDOW (w32_focus_frame),
			       WM_EMACS_FLASH_WINDOW, 0, 0);
	}
    }
  else
    w32_sys_ring_bell ();
}

#ifndef HAVE_NTGUI
DEFUN ("set-message-beep", Fset_message_beep, Sset_message_beep, 1, 1, 0,
       "Set the sound generated when the bell is rung.\n\
SOUND is 'asterisk, 'exclamation, 'hand, 'question, or 'ok\n\
to use the corresponding system sound for the bell.\n\
SOUND is nil to use the normal beep.")
     (sound)
     Lisp_Object sound;
{
  CHECK_SYMBOL (sound, 0);

  if (NILP (sound)) 
      sound_type = 0xFFFFFFFF;
  else if (EQ (sound, intern ("asterisk")))
      sound_type = MB_ICONASTERISK;
  else if (EQ (sound, intern ("exclamation"))) 
      sound_type = MB_ICONEXCLAMATION;
  else if (EQ (sound, intern ("hand"))) 
      sound_type = MB_ICONHAND;
  else if (EQ (sound, intern ("question"))) 
      sound_type = MB_ICONQUESTION;
  else if (EQ (sound, intern ("ok"))) 
      sound_type = MB_OK;
  else
      sound_type = 0xFFFFFFFF;

  return sound;
}
#endif


/* Insert and delete character.
   These are not supposed to be used because we are supposed to turn
   off the feature of using them.  */

static void
W32insert_glyphs (start, len)
     register char *start;
     register int len;
{
  abort ();
}

static void
W32delete_glyphs (n)
     register int n;
{
  abort ();
}

/* Specify how many text lines, from the top of the window,
   should be affected by insert-lines and delete-lines operations.
   This, and those operations, are used only within an update
   that is bounded by calls to W32update_begin and W32update_end.  */

static void
W32set_terminal_window (n)
     register int n;
{
  if (updating_frame == 0)
    abort ();

  if ((n <= 0) || (n > updating_frame->height))
    flexlines = updating_frame->height;
  else
    flexlines = n;
}

/* Perform an insert-lines operation.
   Insert N lines at a vertical position curs_y.  */

static void
stufflines (n)
     register int n;
{
  register int topregion, bottomregion;
  register int length, newtop, mask;
  register struct frame *f = updating_frame;
  int intborder = f->output_data.w32->internal_border_width;

  if (curs_y >= flexlines)
    return;

  topregion = curs_y;
  bottomregion = flexlines - (n + 1);
  newtop = topregion + n;
  length = (bottomregion - topregion) + 1;

#if 0
  dumpqueue ();
#endif /* HAVE_X11 */

  if ((length > 0) && (newtop <= flexlines))
    {
	HDC hdc = MyGetDC (FRAME_W32_WINDOW (f));

#ifdef MW32_FONT
	BitBlt (hdc,
		intborder, CHAR_TO_PIXEL_ROW (f, newtop),
		f->width * FRAME_FONT_WIDTH (f) + FRAME_FONT_OVERHANG(f),
		length * f->output_data.w32->line_height, 
		hdc,
		intborder, CHAR_TO_PIXEL_ROW (f, topregion),
		SRCCOPY);
#else
	BitBlt (hdc,
		intborder, CHAR_TO_PIXEL_ROW (f, newtop),
		f->width * FONT_WIDTH (f->output_data.w32->font),
		length * f->output_data.w32->line_height, 
		hdc,
		intborder, CHAR_TO_PIXEL_ROW (f, topregion),
		SRCCOPY);
#endif

	ReleaseDC (FRAME_W32_WINDOW(f), hdc);
    }

  newtop = min (newtop, (flexlines - 1));
  length = newtop - topregion;
  if (length > 0)
    {
#ifdef MW32_FONT
	w32_clear_area (f, NULL,
			intborder, 
			CHAR_TO_PIXEL_ROW (f, topregion),
			f->width * FRAME_FONT_WIDTH(f)
			+ FRAME_FONT_OVERHANG(f),
			n * f->output_data.w32->line_height);
#else
	w32_clear_area (f, NULL,
			intborder, 
			CHAR_TO_PIXEL_ROW (f, topregion),
			f->width * FONT_WIDTH (f->output_data.w32->font),
			n * f->output_data.w32->line_height);
#endif
    }
}

/* Perform a delete-lines operation, deleting N lines
   at a vertical position curs_y.  */

static void
scraplines (n)
     register int n;
{
  int mask;
  register struct frame *f = updating_frame;
  int intborder = f->output_data.w32->internal_border_width;

  if (curs_y >= flexlines)
    return;

#if 0
  dumpqueue ();
#endif /* HAVE_X11 */

  if ((curs_y + n) >= flexlines)
    {
      if (flexlines >= (curs_y + 1))
	{
#ifdef MW32_FONT
	    w32_clear_area (f, NULL,
			    intborder,
			    CHAR_TO_PIXEL_ROW (f, curs_y),
			    f->width * FRAME_FONT_WIDTH(f) 
			    + FRAME_FONT_OVERHANG(f),
			    (flexlines - curs_y) * f->output_data.w32->line_height);
#else
	    w32_clear_area (f, NULL,
			    intborder,
			    CHAR_TO_PIXEL_ROW (f, curs_y),
			    f->width * FONT_WIDTH (f->output_data.w32->font),
			    (flexlines - curs_y) * f->output_data.w32->line_height);
#endif
	}
    }
  else
    {
	HDC hdc = MyGetDC (FRAME_W32_WINDOW (f));

#ifdef MW32_FONT
	BitBlt (hdc,
		intborder, CHAR_TO_PIXEL_ROW (f, curs_y),
		f->width * FRAME_FONT_WIDTH(f) + FRAME_FONT_OVERHANG(f),
		(flexlines - (curs_y + n)) * f->output_data.w32->line_height,
		hdc,
		intborder, CHAR_TO_PIXEL_ROW (f, curs_y + n),
		SRCCOPY);

	w32_clear_area (f, hdc,
			intborder,
			CHAR_TO_PIXEL_ROW (f, flexlines - n),
			f->width * FRAME_FONT_WIDTH(f)
			+ FRAME_FONT_OVERHANG(f),
			n * f->output_data.w32->line_height);
#else
	BitBlt (hdc,
		intborder, CHAR_TO_PIXEL_ROW (f, curs_y),
		f->width * FONT_WIDTH (f->output_data.w32->font),
		(flexlines - (curs_y + n)) * f->output_data.w32->line_height,
		hdc,
		intborder, CHAR_TO_PIXEL_ROW (f, curs_y + n),
		SRCCOPY);

	w32_clear_area (f, hdc,
			intborder,
			CHAR_TO_PIXEL_ROW (f, flexlines - n),
			f->width * FONT_WIDTH (f->output_data.w32->font),
			n * f->output_data.w32->line_height);
#endif

	ReleaseDC (FRAME_W32_WINDOW(f), hdc);
    }
}

/* Perform an insert-lines or delete-lines operation,
   inserting N lines or deleting -N lines at vertical position VPOS.  */

static void
W32ins_del_lines (vpos, n)
     int vpos, n;
{
  if (updating_frame == 0)
    abort ();

  /* Hide the cursor.  */
  w32_display_cursor (updating_frame, 0);

  W32cursor_to (vpos, 0);

  BLOCK_INPUT;
  if (n >= 0)
    stufflines (n);
  else
    scraplines (-n);
  UNBLOCK_INPUT;
}

/* Support routines for exposure events.  */
static void clear_cursor ();

/* Output into a rectangle of an X-window (for frame F)
   the characters in f->phys_lines that overlap that rectangle.
   TOP and LEFT are the position of the upper left corner of the rectangle.
   ROWS and COLS are the size of the rectangle.
   Call this function with input blocked.  */

void
dumprectangle (f, left, top, cols, rows)
     struct frame *f;
     register int left, top, cols, rows;
{
  register struct frame_glyphs *active_frame = FRAME_CURRENT_GLYPHS (f);
  int cursor_cleared = 0;
  int bottom, right;
  register int y;

#if 0
  if (FRAME_GARBAGED_P (f))
    return;
#endif

  /* Express rectangle as four edges, instead of position-and-size.  */
  bottom = top + rows;
  right = left + cols;

#if 0		/* Window manger does this for X11. */
  {
    int intborder = f->output_data.w32->internal_border_width;

    /* If the rectangle includes any of the internal border area,
       redisplay the border emphasis.  */
    if (top < intborder || left < intborder
	|| bottom > intborder + f->height * f->output_data.w32->line_height
	|| right > intborder + f->width * f->output_data.w32->line_height)
      dumpborder (f, 0);
  }
#endif /* not HAVE_X11		Window manger does this for X11. */
  
  /* Convert rectangle edges in pixels to edges in chars.
     Round down for left and top, up for right and bottom.  */
  top  = PIXEL_TO_CHAR_ROW (f, top);
  left = PIXEL_TO_CHAR_COL (f, left);
  bottom += (f->output_data.w32->line_height - 1);
#ifdef MW32_FONT
  right += (FRAME_FONT_WIDTH(f) - 1);
#else
  right += (FONT_WIDTH (f->output_data.w32->font) - 1);
#endif
  bottom = PIXEL_TO_CHAR_ROW (f, bottom);
  right = PIXEL_TO_CHAR_COL (f, right);

  /* Clip the rectangle to what can be visible.  */
  if (left < 0)
    left = 0;
  if (top < 0)
    top = 0;
  if (right > f->width)
    right = f->width;
  if (bottom > f->height)
    bottom = f->height;

  /* Get size in chars of the rectangle.  */
  cols = right - left;
  rows = bottom - top;

  /* If rectangle has zero area, return.  */
  if (rows <= 0) return;
  if (cols <= 0) return;

  /* Turn off the cursor if it is in the rectangle.
     We will turn it back on afterward.  */
  if ((f->phys_cursor_x >= left) && (f->phys_cursor_x < right)
      && (f->phys_cursor_y >= top) && (f->phys_cursor_y < bottom))
    {
      clear_cursor (f);
      cursor_cleared = 1;
    }

  /* Display the text in the rectangle, one text line at a time.  */

  for (y = top; y < bottom; y++)
    {
      GLYPH *line = &active_frame->glyphs[y][left];

      if (! active_frame->enable[y] || left > active_frame->used[y])
	continue;

      while (*line & GLYPH_MASK_PADDING)
	{
	  /* We must display the whole glyph of a wide-column
             character.  */
	  left--;
	  line--;
	  cols++;
	}
      dumpglyphs (f,
		  CHAR_TO_PIXEL_COL (f, left),
		  CHAR_TO_PIXEL_ROW (f, y),
		  line, min (cols, active_frame->used[y] - left),
		  active_frame->highlight[y], 0, NULL);
    }

  /* Turn the cursor on if we turned it off.  */

  if (cursor_cleared)
    w32_display_cursor (f, 1);
}



static void
frame_highlight (frame)
     struct frame *frame;
{
  w32_display_cursor (frame, 1);
}

static void
frame_unhighlight (frame)
     struct frame *frame;
{
  w32_display_cursor (frame, 1);
}

#if 0
/* Dump the border-emphasis of frame F.
   If F is selected, this is a lining of the same color as the border,
   just within the border, occupying a portion of the internal border.
   If F is not selected, it is background in the same place.
   If ALWAYS is 0, don't bother explicitly drawing if it's background.

   ALWAYS = 1 is used when a frame becomes selected or deselected.
   In that case, we also turn the cursor off and on again
   so it will appear in the proper shape (solid if selected; else hollow.)  */

static void
dumpborder (f, always)
     struct frame *f;
     int always;
{
  int thickness = f->output_data.w32->internal_border_width / 2;
  int width = PIXEL_WIDTH (f);
  int height = PIXEL_HEIGHT (f);
  int pixel;

  if (f != selected_frame)
    {
      if (!always)
	return;

      pixel = f->output_data.w32->background_pixel;
    }
  else
    {
      pixel = f->output_data.w32->border_pixel;
    }

  XPixSet (FRAME_W32_WINDOW (f), 0, 0, width, thickness, pixel);
  XPixSet (FRAME_W32_WINDOW (f), 0, 0, thickness, height, pixel);
  XPixSet (FRAME_W32_WINDOW (f), 0, height - thickness, width,
	   thickness, pixel);
  XPixSet (FRAME_W32_WINDOW (f), width - thickness, 0, thickness,
	   height, pixel);

  if (always)
    w32_display_cursor (f, 1);
}
#endif /* ! defined (HAVE_X11) */

static void W32frame_rehighlight (struct frame *frame);

/* The focus has changed.  Update the frames as necessary to reflect
   the new situation.  Note that we can't change the selected frame
   here, because the lisp code we are interrupting might become confused.
   Each event gets marked with the frame in which it occurred, so the
   lisp code can tell when the switch took place by examining the events.  */

void
w32_new_focus_frame (frame)
     struct frame *frame;
{
  struct frame *old_focus = w32_focus_frame;
  int events_enqueued = 0;

  if (frame != w32_focus_frame)
    {
      /* Set this before calling other routines, so that they see 
	 the correct value of w32_focus_frame.  */
      w32_focus_frame = frame;

      if (old_focus && old_focus->auto_lower)
	w32_lower_frame (old_focus);

      if (w32_focus_frame && w32_focus_frame->auto_raise)
	pending_autoraise_frame = w32_focus_frame;
      else
	pending_autoraise_frame = 0;
    }

  W32frame_rehighlight (frame);
}



/* The focus has changed, or we have redirected a frame's focus to
   another frame (this happens when a frame uses a surrogate
   minibuffer frame).  Shift the highlight as appropriate.

   The FRAME argument doesn't necessarily have anything to do with which
   frame is being highlighted or unhighlighted; we only use it to find
   the appropriate display info.  */
static void
W32frame_rehighlight (frame)
     struct frame *frame;
{
  struct frame *old_highlight = w32_highlight_frame;

  if (w32_focus_frame)
    {
      w32_highlight_frame =
	((GC_FRAMEP(FRAME_FOCUS_FRAME (w32_focus_frame)))
	 ? XFRAME (FRAME_FOCUS_FRAME (w32_focus_frame))
	 : w32_focus_frame);
      if (! FRAME_LIVE_P (w32_highlight_frame))
	{
	  FRAME_FOCUS_FRAME (w32_focus_frame) = Qnil;
	  w32_highlight_frame = w32_focus_frame;
	}
    }
  else
    w32_highlight_frame = 0;

  if (w32_highlight_frame != old_highlight)
    {
      if (old_highlight)
	frame_unhighlight (old_highlight);
      if (w32_highlight_frame)
	frame_highlight (w32_highlight_frame);
    }
}

/* Keyboard processing - modifier keys, etc. */

/* Convert a keysym to its name.  */

char *
x_get_keysym_name (keysym)
    int keysym;
{
  /* Make static so we can always return it */
  static char value[100];

  BLOCK_INPUT;
  GetKeyNameText (keysym, value, 100);
  UNBLOCK_INPUT;

  return value;
}


/* Given a pixel position (PIX_X, PIX_Y) on the frame F, return
   glyph co-ordinates in (*X, *Y).  Set *BOUNDS to the rectangle
   that the glyph at X, Y occupies, if BOUNDS != 0.
   If NOCLIP is nonzero, do not force the value into range.  */

void
pixel_to_glyph_coords (f, pix_x, pix_y, x, y, bounds, noclip)
     FRAME_PTR f;
     register int pix_x, pix_y;
     register int *x, *y;
     RECT *bounds;
     int noclip;
{
  /* Arrange for the division in PIXEL_TO_CHAR_COL etc. to round down
     even for negative values.  */
  if (!inhibit_window_system)
    {
      if (pix_x < 0)
#ifdef MW32_FONT
	pix_x -= FRAME_FONT_WIDTH (f) - 1;
#else
	pix_x -= FONT_WIDTH ((f)->output_data.w32->font) - 1;
#endif
      if (pix_y < 0)
	pix_y -= (f)->output_data.w32->line_height - 1;

      pix_x = PIXEL_TO_CHAR_COL (f, pix_x);
      pix_y = PIXEL_TO_CHAR_ROW (f, pix_y);

      if (bounds)
	{
#ifdef MW32_FONT
	  bounds->right  = ((bounds->left = CHAR_TO_PIXEL_COL (f, pix_x)) + 
			    FRAME_FONT_WIDTH(f));
#else
	  bounds->right  = ((bounds->left = CHAR_TO_PIXEL_COL (f, pix_x)) + 
			    FONT_WIDTH  (f->output_data.w32->font));
#endif
	  bounds->top = ((bounds->bottom = CHAR_TO_PIXEL_ROW (f, pix_y)) +
			 f->output_data.w32->line_height);
	}

      if (!noclip)
	{
	  if (pix_x < 0)
	    pix_x = 0;
	  else if (pix_x > f->width)
	    pix_x = f->width;

	  if (pix_y < 0)
	    pix_y = 0;
	  else if (pix_y > f->height)
	    pix_y = f->height;
	}
    }
  *x = pix_x;
  *y = pix_y;
}

void
glyph_to_pixel_coords (f, x, y, pix_x, pix_y)
     FRAME_PTR f;
     register int x, y;
     register int *pix_x, *pix_y;
{
  if (inhibit_window_system)
    {
      *pix_x = x;
      *pix_y = y;
    }
  else
    {
      *pix_x = CHAR_TO_PIXEL_COL (f, x);
      *pix_y = CHAR_TO_PIXEL_ROW (f, y);
    }
}

#if 0
/* Prepare a mouse-event in *RESULT for placement in the input queue.

   If the event is a button press, then note that we have grabbed
   the mouse.  */

static Lisp_Object
construct_mouse_click (result, event, f)
     struct input_event *result;
     XButtonEvent *event;
     struct frame *f;
{
  /* Make the event type no_event; we'll change that when we decide
     otherwise.  */
  result->kind = mouse_click;
  result->code = event->button - Button1;
  result->timestamp = event->time;
  result->modifiers = (x_w32_to_emacs_modifiers (event->state)
		       | (event->type == ButtonRelease
			  ? up_modifier 
			  : down_modifier));

  {
    int row, column;

    XSETINT (result->x, event->x);
    XSETINT (result->y, event->y);
    XSETFRAME (result->frame_or_window, f);
  }
}

/* Prepare a menu-event in *RESULT for placement in the input queue.  */

static Lisp_Object
construct_menu_click (result, event, f)
     struct input_event *result;
     XButtonEvent *event;
     struct frame *f;
{
  /* Make the event type no_event; we'll change that when we decide
     otherwise.  */
  result->kind = mouse_click;
  XSETINT (result->code, event->button - Button1);
  result->timestamp = event->time;
  result->modifiers = (x_w32_to_emacs_modifiers (event->state)
		       | (event->type == ButtonRelease
			  ? up_modifier 
			  : down_modifier));

  XSETINT (result->x, event->x);
  XSETINT (result->y, -1);
  XSETFRAME (result->frame_or_window, f);
}
#endif

/* Function to report a mouse movement to the mainstream Emacs code.
   The input handler calls this.

   We have received a mouse movement event, which is given in *event.
   If the mouse is over a different glyph than it was last time, tell
   the mainstream emacs code by setting mouse_moved.  If not, ask for
   another motion event, so we can check again the next time it moves.  */

static void
note_mouse_movement (frame, msg)
     FRAME_PTR frame;
     MSG * msg;
{
  last_mouse_movement_time = msg->time;

  if (msg->hwnd != FRAME_W32_WINDOW (frame))
    {
      frame->mouse_moved = 1;
#if 0
      last_mouse_scroll_bar = Qnil;
#endif

      note_mouse_highlight (frame, -1, -1);

    }

  /* Has the mouse moved off the glyph it was on at the last sighting?  */
  else if (LOWORD(msg->lParam) < last_mouse_glyph.left
	   || LOWORD(msg->lParam) > last_mouse_glyph.right
	   || HIWORD(msg->lParam) < last_mouse_glyph.top
	   || HIWORD(msg->lParam) > last_mouse_glyph.bottom)
    {
      frame->mouse_moved = 1;
#if 0 
      last_mouse_scroll_bar = Qnil;
#endif

      note_mouse_highlight (frame, LOWORD(msg->lParam), HIWORD(msg->lParam));
    }
}


/* If you want to add synchronized operation for event,
   you may do in this.  */

void
note_sync_event()
{
  if (!(inhibit_window_system || noninteractive))
    {
      if (last_mouse_movement_frame)
	{
	  note_mouse_movement (last_mouse_movement_frame, 
			       &last_mouse_movement_message);
	  last_mouse_movement_frame = NULL;
	}
    }

  return;
}

/* This is used for debugging, to turn off note_mouse_highlight.  */
static int disable_mouse_highlight;

/* Take proper action when the mouse has moved to position X, Y on frame F
   as regards highlighting characters that have mouse-face properties.
   Also dehighlighting chars where the mouse was before.
   X and Y can be negative or out of range.  */

static void
note_mouse_highlight (f, x, y)
     FRAME_PTR f;
{
  int row, column, portion;
  RECT new_glyph;
  Lisp_Object window;
  struct window *w;

  if (disable_mouse_highlight)
    return;

  mouse_face_mouse_x = x;
  mouse_face_mouse_y = y;
  mouse_face_mouse_frame = f;

  if (mouse_face_defer)
    return;

  if (gc_in_progress)
    {
      mouse_face_deferred_gc = 1;
      return;
    }
  
  if (x < 0 && y < 0 || mouse_cursor_stat < 0)
    {
      clear_mouse_face();
      return;
    }

  /* Find out which glyph the mouse is on.  */
  pixel_to_glyph_coords (f, x, y, &column, &row,
			 &new_glyph, w32_mouse_grabbed);

  /* Which window is that in?  */
  window = window_from_coordinates (f, column, row, &portion);
  w = XWINDOW (window);

  /* If we were displaying active text in another window, clear that.  */
  if (! EQ (window, mouse_face_window))
    clear_mouse_face ();

  /* Are we in a window whose display is up to date?
     And verify the buffer's text has not changed.  */
  if (WINDOWP (window) && portion == 0 && row >= 0 && column >= 0
      && row < FRAME_HEIGHT (f) && column < FRAME_WIDTH (f)
      && EQ (w->window_end_valid, w->buffer)
      && XFASTINT (w->last_modified) == BUF_MODIFF (XBUFFER (w->buffer))
      && (XFASTINT (w->last_overlay_modified)
	  == BUF_OVERLAY_MODIFF (XBUFFER (w->buffer))))

    {
      int *ptr = FRAME_CURRENT_GLYPHS (f)->charstarts[row];
      int i, pos;

      /* Find which buffer position the mouse corresponds to.  */
      for (i = column; i >= 0; i--)
	if (ptr[i] > 0)
	  break;
      pos = ptr[i];
      /* Is it outside the displayed active region (if any)?  */
      if (pos <= 0)
	clear_mouse_face ();
      else if (! (EQ (window, mouse_face_window)
		  && row >= mouse_face_beg_row
		  && row <= mouse_face_end_row
		  && (row > mouse_face_beg_row || column >= mouse_face_beg_col)
		  && (row < mouse_face_end_row || column < mouse_face_end_col
		      || mouse_face_past_end)))
	{
	  Lisp_Object mouse_face, overlay, position;
	  Lisp_Object *overlay_vec;
	  int len, noverlays;
	  struct buffer *obuf;
	  int obegv, ozv;

	  /* If we get an out-of-range value, return now; avoid an error.  */
	  if (pos > BUF_Z (XBUFFER (w->buffer)))
	    return;

	  /* Make the window's buffer temporarily current for
	     overlays_at and compute_char_face.  */
	  obuf = current_buffer;
	  current_buffer = XBUFFER (w->buffer);
	  obegv = BEGV;
	  ozv = ZV;
	  BEGV = BEG;
	  ZV = Z;

	  /* Yes.  Clear the display of the old active region, if any.  */
	  clear_mouse_face ();

	  /* Is this char mouse-active?  */
	  XSETINT (position, pos);

	  len = 10;

	  overlay_vec = (Lisp_Object *) xmalloc (len * sizeof (Lisp_Object));

	  /* Put all the overlays we want in a vector in overlay_vec.
	     Store the length in len.  */
	  noverlays = overlays_at (pos, 1, &overlay_vec, &len,
				   NULL, NULL);
	  noverlays = sort_overlays (overlay_vec, noverlays, w);

	  /* Find the highest priority overlay that has a mouse-face prop.  */
	  overlay = Qnil;
	  for (i = 0; i < noverlays; i++)
	    {
	      mouse_face = Foverlay_get (overlay_vec[i], Qmouse_face);
	      if (!NILP (mouse_face))
		{
		  overlay = overlay_vec[i];
		  break;
		}
	    }
	  free (overlay_vec);
	  /* If no overlay applies, get a text property.  */
	  if (NILP (overlay))
	    mouse_face = Fget_text_property (position, Qmouse_face, w->buffer);

	  /* Handle the overlay case.  */
	  if (! NILP (overlay))
	    {
	      /* Find the range of text around this char that
		 should be active.  */
	      Lisp_Object before, after;
	      int ignore;

	      before = Foverlay_start (overlay);
	      after = Foverlay_end (overlay);
	      /* Record this as the current active region.  */
	      fast_find_position (window, before, &mouse_face_beg_col,
				  &mouse_face_beg_row);
	      mouse_face_past_end
		= !fast_find_position (window, after, &mouse_face_end_col,
				       &mouse_face_end_row);
	      mouse_face_window = window;
	      mouse_face_face_id = compute_char_face (f, w, pos, 0, 0,
						      &ignore, pos + 1, 1);

	      /* Display it as active.  */
	      show_mouse_face (1);
	    }
	  /* Handle the text property case.  */
	  else if (! NILP (mouse_face))
	    {
	      /* Find the range of text around this char that
		 should be active.  */
	      Lisp_Object before, after, beginning, end;
	      int ignore;

	      beginning = Fmarker_position (w->start);
	      XSETINT (end, (BUF_Z (XBUFFER (w->buffer))
			     - XFASTINT (w->window_end_pos)));
	      before
		= Fprevious_single_property_change (make_number (pos + 1),
						    Qmouse_face,
						    w->buffer, beginning);
	      after
		= Fnext_single_property_change (position, Qmouse_face,
						w->buffer, end);
	      /* Record this as the current active region.  */
	      fast_find_position (window, before, &mouse_face_beg_col,
				  &mouse_face_beg_row);
	      mouse_face_past_end
		= !fast_find_position (window, after, &mouse_face_end_col,
				       &mouse_face_end_row);
	      mouse_face_window = window;
	      mouse_face_face_id
		= compute_char_face (f, w, pos, 0, 0,
				     &ignore, pos + 1, 1);

	      /* Display it as active.  */
	      show_mouse_face (1);
	    }
	  BEGV = obegv;
	  ZV = ozv;
	  current_buffer = obuf;
	}
    }
}

/* Find the row and column of position POS in window WINDOW.
   Store them in *COLUMNP and *ROWP.
   This assumes display in WINDOW is up to date.
   If POS is above start of WINDOW, return coords
   of start of first screen line.
   If POS is after end of WINDOW, return coords of end of last screen line.

   Value is 1 if POS is in range, 0 if it was off screen.  */

static int
fast_find_position (window, pos, columnp, rowp)
     Lisp_Object window;
     int pos;
     int *columnp, *rowp;
{
  struct window *w = XWINDOW (window);
  FRAME_PTR f = XFRAME (WINDOW_FRAME (w));
  int i;
  int row = 0;
  int left = WINDOW_LEFT_MARGIN (w);
  int top = XFASTINT (w->top);
  int height = XFASTINT (w->height) - ! MINI_WINDOW_P (w);
  int width = window_internal_width (w);
  int *charstarts;
  int lastcol;
  int maybe_next_line = 0;

  /* Find the right row.  */
  for (i = 0;
       i < height;
       i++)
    {
      int linestart = FRAME_CURRENT_GLYPHS (f)->charstarts[top + i][left];
      if (linestart > pos)
	break;
      /* If the position sought is the end of the buffer,
	 don't include the blank lines at the bottom of the window.  */
      if (linestart == pos && pos == BUF_ZV (XBUFFER (w->buffer)))
	{
	  maybe_next_line = 1;
	  break;
	}
      if (linestart > 0)
	row = i;
    }

  /* Find the right column with in it.  */
  charstarts = FRAME_CURRENT_GLYPHS (f)->charstarts[top + row];
  lastcol = left;
  for (i = 0; i < width; i++)
    {
      if (charstarts[left + i] == pos)
	{
	  *rowp = row + top;
	  *columnp = i + left;
	  return 1;
	}
      else if (charstarts[left + i] > pos)
	break;
      else if (charstarts[left + i] > 0)
	lastcol = left + i + 1;
    }

  /* If we're looking for the end of the buffer,
     and we didn't find it in the line we scanned,
     use the start of the following line.  */
  if (maybe_next_line)
    {
      row++;
      lastcol = left;
    }

  *rowp = row + top;
  *columnp = lastcol;
  return 0;
}

/* Display the active region described by mouse_face_*
   in its mouse-face if HL > 0, in its normal face if HL = 0.  */

static void
show_mouse_face (hl)
     int hl;
{
  struct window *w = XWINDOW (mouse_face_window);
  int width = window_internal_width (w);
  FRAME_PTR f = XFRAME (WINDOW_FRAME (w));
  int i;
  int cursor_off = 0;
  int old_curs_x = curs_x;
  int old_curs_y = curs_y;

  /* Set these variables temporarily
     so that if we have to turn the cursor off and on again
     we will put it back at the same place.  */
  curs_x = f->phys_cursor_x;
  curs_y = f->phys_cursor_y;

  for (i = mouse_face_beg_row; i <= mouse_face_end_row; i++)
    {
      int column = (i == mouse_face_beg_row ? mouse_face_beg_col
		    : WINDOW_LEFT_MARGIN (w));
      int endcolumn = (i == mouse_face_end_row ? mouse_face_end_col
		       : WINDOW_LEFT_MARGIN (w) + width);
      endcolumn = min (endcolumn, FRAME_CURRENT_GLYPHS (f)->used[i]);

      /* If the cursor's in the text we are about to rewrite,
	 turn the cursor off.  */
      if (i == curs_y
	  && curs_x >= column - 1
	  && curs_x <= endcolumn)
	{
	  w32_display_cursor (f, 0);
	  cursor_off = 1;
	}

      dumpglyphs (f,
		  CHAR_TO_PIXEL_COL (f, column),
		  CHAR_TO_PIXEL_ROW (f, i),
		  FRAME_CURRENT_GLYPHS (f)->glyphs[i] + column,
		  endcolumn - column,
		  /* Highlight with mouse face if hl > 0.  */
		  hl > 0 ? 3 : 0, 0, NULL);
    }

  /* If we turned the cursor off, turn it back on.  */
  if (cursor_off)
    w32_display_cursor (f, 1);

  curs_x = old_curs_x;
  curs_y = old_curs_y;

  /* Change the mouse cursor according to the value of HL.  */
  if (hl > 0)
    SetCursor (f->output_data.w32->cross_cursor);
  else
    SetCursor (f->output_data.w32->text_cursor);
}

/* Clear out the mouse-highlighted active region.
   Redraw it unhighlighted first.  */

static void
clear_mouse_face ()
{
  if (! NILP (mouse_face_window))
    show_mouse_face (0);

  mouse_face_beg_row = mouse_face_beg_col = -1;
  mouse_face_end_row = mouse_face_end_col = -1;
  mouse_face_window = Qnil;
}

/* Just discard the mouse face information for frame F, if any.
   This is used when the size of F is changed.  */

cancel_mouse_face (f)
     FRAME_PTR f;
{
  Lisp_Object window;

  window = mouse_face_window;
  if (! NILP (window) && XFRAME (XWINDOW (window)->frame) == f)
    {
      mouse_face_beg_row = mouse_face_beg_col = -1;
      mouse_face_end_row = mouse_face_end_col = -1;
      mouse_face_window = Qnil;
    }
}


struct scroll_bar *w32_window_to_scroll_bar ();
static void w32_scroll_bar_report_motion ();

/* Return the current position of the mouse.
   *fp should be a frame which indicates which display to ask about.

   If the mouse movement started in a scroll bar, set *fp, *bar_window,
   and *part to the frame, window, and scroll bar part that the mouse
   is over.  Set *x and *y to the portion and whole of the mouse's
   position on the scroll bar.

   If the mouse movement started elsewhere, set *fp to the frame the
   mouse is on, *bar_window to nil, and *x and *y to the character cell
   the mouse is over.

   Set *time to the server timestamp for the time at which the mouse
   was at this position.

   Don't store anything if we don't have a valid set of values to report.

   This clears the mouse_moved flag, so we can wait for the next mouse
   movement.  This also calls XQueryPointer, which will cause the
   server to give us another MotionNotify when the mouse moves
   again. */

static void
W32mouse_position (f, insist, bar_window, part, x, y, time)
     FRAME_PTR *f;
     int insist;
     Lisp_Object *bar_window;
     enum scroll_bar_part *part;
     Lisp_Object *x, *y;
     unsigned long *time;
{
  FRAME_PTR f1;

  BLOCK_INPUT;

#if 0
  if (! NILP (last_mouse_scroll_bar))
    w32_scroll_bar_report_motion (f, bar_window, part, x, y, time);
  else
#endif
    {
      POINT pt;

      Lisp_Object frame, tail;

      /* Clear the mouse-moved flag for every frame on this display.  */
      FOR_EACH_FRAME (tail, frame)
	XFRAME (frame)->mouse_moved = 0;
#if 0
      last_mouse_scroll_bar = Qnil;
#endif
      
      GetCursorPos (&pt);
      /* Now we have a position on the root; find the innermost window
	 containing the pointer.  */
      {
	if (w32_mouse_grabbed && last_mouse_frame
	    && FRAME_LIVE_P (last_mouse_frame))
	  {
	    f1 = last_mouse_frame;
	  }
	else
	  {
	    /* Is win one of our frames?  */
	    f1 = w32_window_to_frame (WindowFromPoint(pt));
	  }
      
#if 0
	/* If not, is it one of our scroll bars?  */
	if (! f1)
	  {
	    struct scroll_bar *bar = w32_window_to_scroll_bar (win);

	    if (bar)
	      {
		f1 = XFRAME (WINDOW_FRAME (XWINDOW (bar->window)));
#if 0
		win_x = parent_x;
		win_y = parent_y;
#endif
	      }
	  }

#endif
	if (f1 == 0 && insist)
	  f1 = selected_frame;

	if (f1)
	  {
	    int ignore1, ignore2;

	    ScreenToClient (FRAME_W32_WINDOW(f1),&pt);

	    /* Ok, we found a frame.  Store all the values.  */

	    pixel_to_glyph_coords (f1, pt.x, pt.y, &ignore1, &ignore2,
				   &last_mouse_glyph, w32_mouse_grabbed || insist);

	    *bar_window = Qnil;
	    *part = 0;
	    *f = f1;
	    XSETINT (*x, pt.x);
	    XSETINT (*y, pt.y);
	    *time = last_mouse_movement_time;
	  }
      }
    }

  UNBLOCK_INPUT;
}



/* Drop file Support */

int 
w32_drop_file_handler (frame, msg, emacs_event)
     FRAME_PTR frame;
     MSG* msg;
     struct input_event* emacs_event;
{
  HDROP hDrop;
  POINT pt;

  hDrop = (HANDLE)msg->wParam;
  DragQueryPoint(hDrop, &pt);
  /* DragQueryPoint returns position based on window coordination */
  emacs_event->kind = drag_n_drop;
  emacs_event->code = (int) hDrop;
  emacs_event->modifiers = W32GETMODIFIER;
  XSETINT (emacs_event->x, pt.x);
  XSETINT (emacs_event->y, pt.y);
  XSETFRAME (emacs_event->frame_or_window, frame);
  emacs_event->timestamp = msg->time;

  return 1;
}



/* IntelliMouse Support */

#ifdef W32_INTELLIMOUSE
int 
w32_mouse_wheel_handler (frame, msg, emacs_event)
     FRAME_PTR frame;
     MSG* msg;
     struct input_event* emacs_event;
{
  POINT pt;

  /* Sony VAIO Jog Dial Utility sends WM_MOUSEWHEEL with the posotion
     guessed by active window. Here replace it by last mouse position
     stored as last_mouse_movement_message. There is no side effect,
     maybe.. 2001/03/16 K. Horiguchi */

  pt.x = (signed short) LOWORD(last_mouse_movement_message.lParam);
  pt.y = (signed short) HIWORD(last_mouse_movement_message.lParam);

/*
  pt.x = (signed short) LOWORD(msg->lParam);
  pt.y = (signed short) HIWORD(msg->lParam);
  ScreenToClient(msg->hwnd, &pt);
*/

  emacs_event->kind = w32_mouse_wheel;
  emacs_event->code = (signed short) HIWORD(msg->wParam);
  emacs_event->modifiers = W32GETMODIFIER;
  XSETINT (emacs_event->x, pt.x);
  XSETINT (emacs_event->y, pt.y);
  XSETFRAME (emacs_event->frame_or_window, frame);
  emacs_event->timestamp = msg->time;

  return 1;
}
#endif


/* Scroll bar support.  */

/* Given an X window ID, find the struct scroll_bar which manages it.
   This can be called in GC, so we have to make sure to strip off mark
   bits.  */
struct scroll_bar *
w32_window_to_scroll_bar (window_id)
     HWND window_id;
{
#ifdef W32_SCROLLBAR
  Lisp_Object tail, frame;

  for (tail = Vframe_list;
       XGCTYPE (tail) == Lisp_Cons;
       tail = XCONS (tail)->cdr)
    {
      Lisp_Object frame, bar, condemned;

      frame = XCONS (tail)->car;
      /* All elements of Vframe_list should be frames.  */
      if (! GC_FRAMEP(frame))
	abort ();

      /* Scan this frame's scroll bar list for a scroll bar with the
         right window ID.  */
      condemned = FRAME_CONDEMNED_SCROLL_BARS (XFRAME (frame));
      for (bar = FRAME_SCROLL_BARS (XFRAME (frame));
	   /* This trick allows us to search both the ordinary and
              condemned scroll bar lists with one loop.  */
	   ! GC_NILP (bar) || (bar = condemned,
			       condemned = Qnil,
			       ! GC_NILP (bar));
	   bar = XSCROLL_BAR (bar)->next)
	if (SCROLL_BAR_W32_WINDOW (XSCROLL_BAR (bar)) == window_id)
	  return XSCROLL_BAR (bar);
    }
  return 0;
#endif
}

/* Open a new X window to serve as a scroll bar, and return the
   scroll bar vector for it.  */
static struct scroll_bar *
w32_scroll_bar_create (window, top, left, width, height)
     struct window *window;
     int top, left, width, height;
{
#ifdef W32_SCROLLBAR
  FRAME_PTR frame = XFRAME (WINDOW_FRAME (window));
  struct scroll_bar *bar =
    XSCROLL_BAR (Fmake_vector (make_number (SCROLL_BAR_VEC_SIZE), Qnil));

  {
    extern HINSTANCE hinst;
    MSG msg;
    RECT rect;
    HWND hwndScroll;
    SCROLLINFO scinfo;

    rect.top = top;
    rect.left = left;
    rect.bottom = height;
    rect.right = width;
    SEND_INFORM_MESSAGE (FRAME_W32_WINDOW(frame),
			 WM_EMACS_CREATE_SCROLLBAR,
			 (WPARAM)&rect, (LPARAM)hinst);
    WAIT_REPLY_MESSAGE(&msg, WM_EMACS_CREATE_SCROLLBAR_REPLY);

    scinfo.cbSize = sizeof(SCROLLINFO);
    scinfo.fMask = SIF_ALL;
    scinfo.nMin = 0;
    scinfo.nMax = VERTICAL_SCROLL_BAR_TOP_RANGE(height);
    scinfo.nPage = 0;
    scinfo.nPos = 0;
    SetScrollInfo((HWND)msg.wParam, SB_CTL, &scinfo, TRUE);

    SET_SCROLL_BAR_W32_WINDOW(bar, (HWND)msg.wParam);

  }

  XSETWINDOW (bar->window, window);
  XSETINT (bar->top, top);
  XSETINT (bar->left, left);
  XSETINT (bar->width, width);
  XSETINT (bar->height, height);
  XSETINT (bar->start, 0);
  XSETINT (bar->end, 0);
  bar->dragging = Qnil;

  /* Add bar to its frame's list of scroll bars.  */
  bar->next = FRAME_SCROLL_BARS (frame);
  bar->prev = Qnil;
  XSETVECTOR (FRAME_SCROLL_BARS (frame), bar);
  if (! NILP (bar->next))
    XSETVECTOR (XSCROLL_BAR (bar->next)->prev, bar);

  return bar;
#endif
}

/* Draw BAR's handle in the proper position.
   If the handle is already drawn from START to END, don't bother
   redrawing it, unless REBUILD is non-zero; in that case, always
   redraw it.  (REBUILD is handy for drawing the handle after expose
   events.)  

   Normally, we want to constrain the start and end of the handle to
   fit inside its rectangle, but if the user is dragging the scroll bar
   handle, we want to let them drag it down all the way, so that the
   bar's top is as far down as it goes; otherwise, there's no way to
   move to the very end of the buffer.  */
static void
w32_scroll_bar_set_handle (bar, start, end, rebuild)
     struct scroll_bar *bar;
     int start, end;
     int rebuild;
{
#ifdef W32_SCROLLBAR
  int dragging;
  HWND hwndScroll;
  SCROLLINFO scinfo;

  hwndScroll = SCROLL_BAR_W32_WINDOW (bar);

  dragging = ! NILP (bar->dragging);

  /* If the display is already accurate, do nothing.  */
  if (! rebuild
      && start == XINT (bar->start)
      && end == XINT (bar->end))
    return;

  scinfo.cbSize = sizeof(SCROLLINFO);
  scinfo.fMask = SIF_PAGE | SIF_POS;
  scinfo.nPage = end - start;
  scinfo.nPos = start;
  SetScrollInfo(hwndScroll, SB_CTL, &scinfo, TRUE);

  /* Store the adjusted setting in the scroll bar.  */
  XSETINT (bar->start, start);
  XSETINT (bar->end, end);
#endif
}

/* Move a scroll bar around on the screen, to accommodate changing
   window configurations.  */
static void
w32_scroll_bar_move (bar, top, left, width, height)
     struct scroll_bar *bar;
     int top, left, width, height;
{
#ifdef W32_SCROLLBAR
  HWND hwndScroll;

  hwndScroll = SCROLL_BAR_W32_WINDOW(bar);

  MoveWindow(hwndScroll, left, top, width, height, TRUE);

  XSETINT (bar->left,   left);
  XSETINT (bar->top,    top);
  XSETINT (bar->width,  width);
  XSETINT (bar->height, height);
#endif
}

/* Destroy the X window for BAR, and set its Emacs window's scroll bar
   to nil.  */
static void
w32_scroll_bar_remove (bar)
     struct scroll_bar *bar;
{
  MSG msg;

#ifdef W32_SCROLLBAR
  /* Destroy the window.  */
  SEND_INFORM_MESSAGE (SCROLL_BAR_W32_WINDOW(bar), WM_CLOSE, 0, 0);

  /* Disassociate this scroll bar from its window.  */
  XWINDOW (bar->window)->vertical_scroll_bar = Qnil;
#endif
}

/* Set the handle of the vertical scroll bar for WINDOW to indicate
   that we are displaying PORTION characters out of a total of WHOLE
   characters, starting at POSITION.  If WINDOW has no scroll bar,
   create one.  */
static void
W32set_vertical_scroll_bar (window, portion, whole, position)
     struct window *window;
     int portion, whole, position;
{
#ifdef W32_SCROLLBAR
  FRAME_PTR f = XFRAME (WINDOW_FRAME (window));
  int top = XINT (window->top);
  int left = WINDOW_VERTICAL_SCROLL_BAR_COLUMN (window);
  int height = WINDOW_VERTICAL_SCROLL_BAR_HEIGHT (window);

  /* Where should this scroll bar be, pixelwise?  */
  int pixel_top  = CHAR_TO_PIXEL_ROW (f, top);
  int pixel_left = CHAR_TO_PIXEL_COL (f, left);
  int pixel_width
    = (FRAME_SCROLL_BAR_PIXEL_WIDTH (f) > 0
       ? FRAME_SCROLL_BAR_PIXEL_WIDTH (f)
       : (FRAME_SCROLL_BAR_COLS (f) * FONT_WIDTH (f->output_data.w32->font)));
  int pixel_height = VERTICAL_SCROLL_BAR_PIXEL_HEIGHT (f, height);

  struct scroll_bar *bar;

  /* Does the scroll bar exist yet?  */
  if (NILP (window->vertical_scroll_bar))
    bar = w32_scroll_bar_create (window,
				   pixel_top, pixel_left,
				   pixel_width, pixel_height);
  else
    {
      /* It may just need to be moved and resized.  */
      bar = XSCROLL_BAR (window->vertical_scroll_bar);
      w32_scroll_bar_move (bar, pixel_top, pixel_left,
			     pixel_width, pixel_height);
    }

  /* Set the scroll bar's current state, unless we're currently being
     dragged.  */
  if (NILP (bar->dragging))
    {
      int top_range =
	VERTICAL_SCROLL_BAR_TOP_RANGE (pixel_height);

      if (whole == 0)
	w32_scroll_bar_set_handle (bar, 0, top_range, 0);
      else
	{
	  int start = (int)(((double) position * top_range) / whole);
	  int end = (int)(((double) (position + portion) * top_range) / whole);

	  w32_scroll_bar_set_handle (bar, start, end, 0);
	}
    }

  XSETVECTOR (window->vertical_scroll_bar, bar);
#endif
}


/* The following three hooks are used when we're doing a thorough
   redisplay of the frame.  We don't explicitly know which scroll bars
   are going to be deleted, because keeping track of when windows go
   away is a real pain - "Can you say set-window-configuration, boys
   and girls?"  Instead, we just assert at the beginning of redisplay
   that *all* scroll bars are to be removed, and then save a scroll bar
   from the fiery pit when we actually redisplay its window.  */

/* Arrange for all scroll bars on FRAME to be removed at the next call
   to `*judge_scroll_bars_hook'.  A scroll bar may be spared if
   `*redeem_scroll_bar_hook' is applied to its window before the judgement.  */
static void 
W32condemn_scroll_bars (frame)
     FRAME_PTR frame;
{
#ifdef W32_SCROLLBAR
  /* Transfer all the scroll bars to FRAME_CONDEMNED_SCROLL_BARS.  */
  while (! NILP (FRAME_SCROLL_BARS (frame)))
    {
      Lisp_Object bar;
      bar = FRAME_SCROLL_BARS (frame);
      FRAME_SCROLL_BARS (frame) = XSCROLL_BAR (bar)->next;
      XSCROLL_BAR (bar)->next = FRAME_CONDEMNED_SCROLL_BARS (frame);
      XSCROLL_BAR (bar)->prev = Qnil;
      if (! NILP (FRAME_CONDEMNED_SCROLL_BARS (frame)))
	XSCROLL_BAR (FRAME_CONDEMNED_SCROLL_BARS (frame))->prev = bar;
      FRAME_CONDEMNED_SCROLL_BARS (frame) = bar;
    }
#if 0
  /* The condemned list should be empty at this point; if it's not,
     then the rest of Emacs isn't using the condemn/redeem/judge
     protocol correctly.  */
  if (! NILP (FRAME_CONDEMNED_SCROLL_BARS (frame)))
    abort ();

  /* Move them all to the "condemned" list.  */
  FRAME_CONDEMNED_SCROLL_BARS (frame) = FRAME_SCROLL_BARS (frame);
  FRAME_SCROLL_BARS (frame) = Qnil;
#endif
#endif
}

/* Unmark WINDOW's scroll bar for deletion in this judgement cycle.
   Note that WINDOW isn't necessarily condemned at all.  */
static void
W32redeem_scroll_bar (window)
     struct window *window;
{
#ifdef W32_SCROLLBAR
  struct scroll_bar *bar;

  /* We can't redeem this window's scroll bar if it doesn't have one.  */
  if (NILP (window->vertical_scroll_bar))
    abort ();

  bar = XSCROLL_BAR (window->vertical_scroll_bar);

  /* Unlink it from the condemned list.  */
  {
    FRAME_PTR f = XFRAME (WINDOW_FRAME (window));

    if (NILP (bar->prev))
      {
	/* If the prev pointer is nil, it must be the first in one of
           the lists.  */
	if (EQ (FRAME_SCROLL_BARS (f), window->vertical_scroll_bar))
	  /* It's not condemned.  Everything's fine.  */
	  return;
	else if (EQ (FRAME_CONDEMNED_SCROLL_BARS (f),
		     window->vertical_scroll_bar))
	  FRAME_CONDEMNED_SCROLL_BARS (f) = bar->next;
	else
	  /* If its prev pointer is nil, it must be at the front of
             one or the other!  */
	  abort ();
      }
    else
      XSCROLL_BAR (bar->prev)->next = bar->next;

    if (! NILP (bar->next))
      XSCROLL_BAR (bar->next)->prev = bar->prev;

    bar->next = FRAME_SCROLL_BARS (f);
    bar->prev = Qnil;
    XSETVECTOR (FRAME_SCROLL_BARS (f), bar);
    if (! NILP (bar->next))
      XSETVECTOR (XSCROLL_BAR (bar->next)->prev, bar);
  }
#endif
}

/* Remove all scroll bars on FRAME that haven't been saved since the
   last call to `*condemn_scroll_bars_hook'.  */
static void
W32judge_scroll_bars (f)
     FRAME_PTR f;
{
#ifdef W32_SCROLLBAR
  Lisp_Object bar, next;

  bar = FRAME_CONDEMNED_SCROLL_BARS (f);

  /* Clear out the condemned list now so we won't try to process any
     more events on the hapless scroll bars.  */
  FRAME_CONDEMNED_SCROLL_BARS (f) = Qnil;

  for (; ! NILP (bar); bar = next)
    {
      struct scroll_bar *b = XSCROLL_BAR (bar);

      w32_scroll_bar_remove (b);

      next = b->next;
      b->next = b->prev = Qnil;
    }

  /* Now there should be no references to the condemned scroll bars,
     and they should get garbage-collected.  */
#endif
}


/* Handle an Expose or GraphicsExpose event on a scroll bar.

   This may be called from a signal handler, so we have to ignore GC
   mark bits.  */
void
w32_scroll_bar_expose (FRAME_PTR f)
{
#ifdef W32_SCROLLBAR
  Lisp_Object bar;

  for (bar = FRAME_SCROLL_BARS (f); VECTORP (bar);
       bar = XSCROLL_BAR (bar)->next)
    {
      HWND hwndScroll = SCROLL_BAR_W32_WINDOW (XSCROLL_BAR (bar));
      UpdateWindow(hwndScroll);
    }

#endif
}

/* Handle a mouse click on the scroll bar BAR.  If *EMACS_EVENT's kind
   is set to something other than no_event, it is enqueued.

   This may be called from a signal handler, so we have to ignore GC
   mark bits.  */
static int
w32_scroll_bar_handle_click (bar, msg, emacs_event)
     struct scroll_bar *bar;
     MSG * msg;
     struct input_event *emacs_event;
{
#ifdef W32_SCROLLBAR
  if (!GC_WINDOWP (bar->window))
    abort ();

  emacs_event->kind = w32_scroll_bar_click;
  emacs_event->code = 0;
  emacs_event->modifiers = W32GETMODIFIER;
  emacs_event->frame_or_window = bar -> window;
  emacs_event->timestamp = msg -> time;
  {
    int y;
    int top_range =
      VERTICAL_SCROLL_BAR_TOP_RANGE (XINT (bar->height));

    y = HIWORD(msg -> wParam);
   /* copy from emacs 19.34 ....(himi) */
    switch (LOWORD (msg -> wParam))
      {
      case SB_THUMBTRACK:
	emacs_event->part = scroll_bar_handle;
#if 0
	printf("Scroll Bar Tracking...%d\n", y);
	fflush(stdout);
#endif
	break;
      case SB_LINEDOWN:
	emacs_event->part = scroll_bar_down_arrow;
	break;
      case SB_LINEUP:
	emacs_event->part = scroll_bar_up_arrow;
	break;
      case SB_PAGEUP:
	emacs_event->part = scroll_bar_above_handle;
	break;
      case SB_PAGEDOWN:
	emacs_event->part = scroll_bar_below_handle;
	break;
      case SB_TOP:
	emacs_event->part = scroll_bar_handle;
	y = 0;
	break;
      case SB_BOTTOM:
	emacs_event->part = scroll_bar_handle;
	y = top_range;
	break;
      case SB_THUMBPOSITION:
	emacs_event->part = scroll_bar_handle;
#if 0
	printf("Scroll Bar Tracking...%d\n", y);
	fflush(stdout);
#endif
	break;
      case SB_ENDSCROLL:
      default:
	return 0;
	break;
      }
    XSETINT (emacs_event->x, y);

    XSETINT (emacs_event->y, top_range);
  }
  return 1;
#endif
}

#ifdef W32_SCROLLBAR
void w32_scroll_bar_store_event(WPARAM wParam, LPARAM lParam)
{
  struct input_event event;
  MSG msg;
  struct scroll_bar *bar;

  bar = w32_window_to_scroll_bar ((HWND)lParam);
  msg.time = GetTickCount();
  msg.wParam = wParam;
  msg.lParam = lParam;
  if (bar && w32_scroll_bar_handle_click (bar, &msg, &event))
    {
      kbd_buffer_store_event(&event);
      SetEvent(keyboard_handle);
    }
}
#endif

/* The screen has been cleared so we may have changed foreground or
   background colors, and the scroll bars may need to be redrawn.
   Clear out the scroll bars, and ask for expose events, so we can
   redraw them.  */

#if 0
w32_scroll_bar_clear (f)
     FRAME_PTR f;
{
  Lisp_Object bar;

  for (bar = FRAME_SCROLL_BARS (f);
       XTYPE (bar) == Lisp_Vector;
       bar = XSCROLL_BAR (bar)->next)
    XClearArea (dsp, SCROLL_BAR_W32_WINDOW (XSCROLL_BAR (bar)),
		0, 0, 0, 0, True);
}
#endif

/* We must construct menu structure, but only main thread
   can do it.  We store event menu_bar_activate_event to
   request main thread to reconstruct menu sturucture.
   Then, this thread must wait for the finish, but
   if main thread is busy, it cannot deal with the request.
   Nevertheless, we caputre any event mainly in order
   to store keyboard quit.
   Thus, I set timeout to 500 millisecond. */
void
w32_menu_bar_store_activate_event (struct frame *f)
{
  extern Lisp_Object Vmenu_updating_frame;
  int result;

  struct input_event emacs_event;
  HANDLE ev = f->output_data.w32->mainthread_to_frame_handle;

  emacs_event.kind = menu_bar_activate_event;
  XSETFRAME (emacs_event.frame_or_window, f);
  emacs_event.timestamp = GetTickCount();

  ResetEvent (ev);
  kbd_buffer_store_event (&emacs_event);
  SetEvent (keyboard_handle);

  message_loop_blocked_p = 1;
  result = WaitForSingleObject (ev, 500);
  /* main thread have already entered updating phase.
     Freeze until the finish! */
  if (!NILP(Vmenu_updating_frame))
    result = WaitForSingleObject (ev, 1000);
  message_loop_blocked_p = 0;

#if 0
  if (result == WAIT_OBJECT_0)
    fprintf(stderr, "Success!!\n");
  else if (result == WAIT_TIMEOUT)
    fprintf(stderr, "Timeout!!\n");
  else
    fprintf(stderr, "Unknown:%d!!\n", result);
#endif

  lock_mouse_cursor_visible (TRUE);
  DrawMenuBar (FRAME_W32_WINDOW (f));

  f->output_data.w32->disable_reconstruct_menubar = 1;

  return;
}

#if 0
/* This processes Expose events from the menubar specific X event
   loop in menubar.c.  This allows to redisplay the frame if necessary
   when handling menubar or popup items.  */

void
process_expose_from_menu (event)
     XEvent event;
{
  FRAME_PTR f;

  BLOCK_INPUT;

  f = w32_window_to_frame (event.xexpose.window);
  if (f)
    {
      if (f->async_visible == 0)
	{
	  f->async_visible = 1;
	  f->async_iconified = 0;
	  SET_FRAME_GARBAGED (f);
	  cancel_mouse_face (f);
	}
      else
	{
	  dumprectangle (w32_window_to_frame (event.xexpose.window),
			 event.xexpose.x, event.xexpose.y,
			 event.xexpose.width, event.xexpose.height);
	}
    }
  else
    {
      struct scroll_bar *bar
	= w32_window_to_scroll_bar (event.xexpose.window);
      
      if (bar)
	w32_scroll_bar_expose (bar, &event);
    }

  UNBLOCK_INPUT;
}
#endif

int w32_rbutton_to_emacs_button;
int w32_mbutton_to_emacs_button;
int w32_lbutton_to_emacs_button;
int w32_hide_mouse_timeout;
int w32_hide_mouse_on_key;

static void w32_mouse_button_cc(UINT mouse_event,
				int *button,
				int *up,
				int *modifier)
{
  Lisp_Object modp = Qnil;

  switch(mouse_event){
  case WM_LBUTTONUP:
    *button = w32_lbutton_to_emacs_button;
    *up = 1;
    break;
  case WM_LBUTTONDOWN:
    *button = w32_lbutton_to_emacs_button;
    *up = 0;
    break;
  case WM_MBUTTONUP:
    *button = w32_mbutton_to_emacs_button;
    *up = 1;
    break;
  case WM_MBUTTONDOWN:
    *button = w32_mbutton_to_emacs_button;
    *up = 0;
    break;
  case WM_RBUTTONUP:
    *button = w32_rbutton_to_emacs_button;
    *up = 1;
    break;
  case WM_RBUTTONDOWN:
    *button = w32_rbutton_to_emacs_button;
    *up = 0;
    break;
  }

  *modifier = W32GETMOUSEMODIFIER(*up);
}

/* for Emacs Translate Message */
int
GetKeyModifierState()
{
  BYTE keystate[256];
  int i;
  int mod_result = 0;
  PBYTE pks, pkm;
  BYTE tmp;

  pks = keystate;
  pkm = keymodifier;
  GetKeyboardState(keystate);
  for (i = 0;i < 256;i++, pks++, pkm++)
    {
      if ((*pkm == 0) || (*pkm == 128)) continue;    /* for speed.  */
      tmp = *pks >> 7;
      if (tmp & (*pkm >> 1)) mod_result |= meta_modifier;
      else if (tmp & (*pkm >> 2)) mod_result |= ctrl_modifier;
      else if (tmp & (*pkm >> 3)) mod_result |= shift_modifier;
      else if (tmp & (*pkm >> 4)) mod_result |= alt_modifier;
      else if (tmp & (*pkm >> 5)) mod_result |= super_modifier;
      else if (tmp & (*pkm >> 6)) mod_result |= hyper_modifier;
    }
  return mod_result;
}

/* 96.10.26 by himi */
int EmacsTranslateMessage(UINT type, UINT virtkey,
			  UINT *keycode, int *modifier)
{
  BYTE keystate[256];
  static BYTE ansi_code[4];
  static int isdead = 0;
  extern char *lispy_function_keys[];

  static int cur_mod;

  if (!type) {
    if (isdead) {
      *modifier = cur_mod;
      isdead = 0;
      *keycode = ansi_code[2];
      return 1;
    }
    return 0;
  }

  cur_mod = W32GETMODIFIER;
  *modifier = cur_mod;

  if (keymodifier[virtkey] == 128) {
    *keycode = virtkey;
    return 2;
  }
  if (keymodifier[virtkey] == 1) return 3;
  if (keymodifier[virtkey]) return 0;

  GetKeyboardState(keystate);
  if (cur_mod)
    keystate[VK_KANA] = 0;
  keystate[VK_CONTROL] = 0;
  keystate[VK_MENU] = 0;
  keystate[VK_RCONTROL] = 0;
  keystate[VK_LCONTROL] = 0;
  keystate[VK_RMENU] = 0;
  keystate[VK_LMENU] = 0;
  if (keymodifier[VK_CAPITAL])
    keystate[VK_CAPITAL] = 0;
  if (cur_mod & shift_modifier)
    keystate[VK_SHIFT] = 0x80;
  else
    keystate[VK_SHIFT] = 0;

  /* On some platform(at least Windows NT4.0),
     ToAscii does not work well.  But ToAsciiEx() is
     not supported by all W32 Platoform...*/
#if 1
  isdead = ToAscii(virtkey, 0, keystate, (LPWORD) ansi_code, 0);
#else
  isdead = ToAsciiEx(virtkey, 0, keystate, (LPWORD) ansi_code,
		     0, GetKeyboardLayout(0));
#endif
  if (isdead < 0) return 0;
  if (isdead >= 1) {
    isdead--;
    *keycode = ansi_code[0];
    return 1;
  }
  if (lispy_function_keys[virtkey]) {
    *keycode = virtkey;
    return 2;
  }
  return 3;
}

/* The main X event-reading loop - W32read_socket.  */

/* Timestamp of enter window event.  This is only used by W32read_socket,
   but we have to put it out here, since static variables within functions
   sometimes don't work.  */
static Time enter_timestamp;

/* Read events coming from the X server.
   This routine is called by the SIGIO handler.
   We return as soon as there are no more events to be read.

   Events representing keys are stored in buffer BUFP,
   which can hold up to NUMCHARS characters.
   We return the number of characters stored into the buffer,
   thus pretending to be `read'.

   EXPECTED is nonzero if the caller knows input is available.  */

int
W32read_socket (sd, bufp, numchars, waitp, expected)
     register int sd;
     register struct input_event *bufp;
     register int numchars;
     int waitp;
     int expected;
{
  int count = 0;
  MSG msg;
  struct frame *f;
  static int lastmsgp = 0;
  static MSG lastmsg = {INVALID_HANDLE_VALUE, 0, 0, 0, 0, {0, 0}};

  int exit_message_loop = 0;

  if ((interrupt_input_blocked) && (w32_special_input == 0))
    {
      interrupt_input_pending = 1;
      return -1;
    }

  interrupt_input_pending = 0;
	
#if 0				/* 95.10.9 Modified by himi */
  if (numchars <= 0)
    abort ();			/* Don't think this happens. */
#else
  assert(numchars > 0);
#endif

  if (lastmsgp)
    {
      msg = lastmsg;
    }
  while (1)
    {
      int button;
      int up;
      int modifier;

      if (lastmsgp) lastmsgp = 0;
      else if (waitp)
	{
	  GetMessage(&msg, NULL, 0, 0);
	  waitp = 0;
	}
      else if (!PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) break;

      if (msg.hwnd)
	{
	  f = w32_window_to_frame (msg.hwnd);
	  if ((!f) && (!IS_EMACS_PRIVATE_MESSAGE(msg.message)))
	    {
	      TranslateMessage (&msg);
	      DispatchMessage (&msg);
	      continue;
	    }
	}
      else
	f = NULL;

      switch (msg.message)
	{
	case WM_KEYDOWN:
	  if (w32_hide_mouse_on_key && mouse_cursor_stat == 0)
	    {
	      mouse_cursor_stat = -1;
	      ShowCursor (FALSE);
	    }

	case WM_SYSKEYDOWN:

	  if (f && !f->iconified && f->visible)
	    {
	      UINT keycode;
	      int keymod;
	      int keyflag;
	      int ocount = count;

	      keyflag = EmacsTranslateMessage(1, msg.wParam, 
					      &keycode, &keymod);
	      while (1)
		{
		  switch(keyflag)
		    {
		    case 0:
		      goto dflt_no_translate;
		    case 1:
		      bufp->kind = ascii_keystroke;
		      break;
		    case 2:
		      bufp->kind = non_ascii_keystroke;
		      break;
		    case 3:
		      goto dflt;
		    }
		  bufp->code = keycode;
		  bufp->modifiers = keymod;
		  XSETFRAME (bufp->frame_or_window, f);
		  bufp->timestamp = msg.time;
		  bufp++;
		  numchars--;
		  count++;

		  keyflag = EmacsTranslateMessage(0, 0, &keycode, &keymod);
		  if (!keyflag) break;

		  if (numchars == 0)
		    {
		      count = ocount;
		      exit_message_loop = 1; /* no space, quit! */
		      break;
		    }
		}
	    }
	  break;

	case WM_KEYUP:
	case WM_SYSKEYUP:
	  f = w32_window_to_frame (msg.hwnd);
	  if (!f) goto dflt;

	    {
	      int keyflag;
	      UINT keycode;
	      int keymod;
	      keyflag = EmacsTranslateMessage(1, msg.wParam, 
					      &keycode, &keymod);
	      if (keyflag == 0) goto dflt_no_translate;
	      goto dflt;
	    }
	    break;

	case WM_MULE_IME_STATUS:
	  if (f && !f->iconified && f->visible && numchars > 0)
	    {
	      bufp->kind = non_ascii_keystroke;
	      bufp->code = VK_KANJI;
	      bufp->modifiers = 0;
	      XSETFRAME (bufp->frame_or_window, f);
	      bufp->timestamp = msg.time;
	      bufp++;
	      numchars--;
	      count++;
	    }
	  break;
	case WM_MULE_IME_REPORT:
	  {
	    LPTSTR lpStr;
	    struct input_event buf;
	    HANDLE hw32_ime_string = (HANDLE) msg.wParam;

	    if (count != 0) {
	      exit_message_loop = 1;
	      break;
	    }
	    f = (struct frame *) msg.lParam;
	    if (f && !f->iconified && f->visible)
	      {
		lpStr = GlobalLock(hw32_ime_string);
		while(1)
		  {
		    XSETFRAME (buf.frame_or_window, f);
		    buf.timestamp = msg.time;
		    buf.modifiers = 0;
		    if (*lpStr)
		      {
			buf.kind = ascii_keystroke;
			buf.code = *lpStr;
			kbd_buffer_store_event(&buf);
			lpStr++;
		      }
		    else
		      {
			buf.kind = non_ascii_keystroke;
			buf.code = VK_COMPEND;
			kbd_buffer_store_event(&buf);
			break;
		      }
		  }
		GlobalUnlock(hw32_ime_string);
		GlobalFree(hw32_ime_string);
	      }
	  }
	  break;

	case WM_MOUSEMOVE:
	  if (f)
	    last_mouse_movement_frame = f;
	  else
	    clear_mouse_face ();

	  /* fall through*/
	case WM_NCMOUSEMOVE:
	  if (last_mouse_movement_message.lParam != msg.lParam)
	    {
	      last_mouse_movement_message = msg;
	      if (mouse_cursor_stat < 0)
		{
		  ShowCursor (TRUE);
		  mouse_cursor_stat = 0;
		}
	    }
	  else
	    {
	      if (XFASTINT (w32_hide_mouse_timeout) > 0 &&
		  msg.time - last_mouse_movement_message.time
		  > XFASTINT (w32_hide_mouse_timeout))
		if (mouse_cursor_stat == 0)
		  {
		    ShowCursor (FALSE);
		    mouse_cursor_stat = -1;
		  }
	    }

	  if (w32_mouse_grabbed && last_mouse_frame
	      && FRAME_LIVE_P (last_mouse_frame))
	    f = last_mouse_frame;

	  goto dflt;

	case WM_LBUTTONDOWN:
	case WM_MBUTTONDOWN:
	case WM_RBUTTONDOWN:
	  if (mouse_cursor_stat < 0)
	    ShowCursor (TRUE);
	  mouse_cursor_stat = 2;

	  /* fall through */
	case WM_LBUTTONUP:
	case WM_MBUTTONUP:
	case WM_RBUTTONUP:
	  if (mouse_cursor_stat >0)
	    mouse_cursor_stat--; /* 2 -> 1,  1 -> 0 */
	      
	  w32_mouse_button_cc(msg.message, &button, &up, &modifier);
	  {
	    /* If we decide we want to generate an event to be seen
	       by the rest of Emacs, we put it here.  */
		
	    f = w32_window_to_frame (msg.hwnd);
		
	    if (f == w32_focus_frame)
	      {
		if (numchars >= 1)
		  {
		    bufp->kind = mouse_click;
		    bufp->code = button;
		    bufp->timestamp = msg.time;
		    bufp->modifiers = modifier;
		    XSETINT (bufp->x, LOWORD(msg.lParam));
		    XSETINT (bufp->y, HIWORD(msg.lParam));
		    XSETFRAME (bufp->frame_or_window, f);
		    bufp++;
		    count++;
		    numchars--;
		  }
	      }
		
	    if (up == 1)
	      {
		w32_mouse_grabbed &= ~(1 << button);
		if (!w32_mouse_grabbed) {
		  if (!NILP(Vmouse_depressed))
		    {
		      ReleaseCapture ();
		      Vmouse_depressed = Qnil;
		    }
		}
	      }
	    else if (up == 0)
	      {
		w32_mouse_grabbed |= (1 << button);
		Vmouse_depressed = Qt;
		last_mouse_frame = f;
		SetCapture (msg.hwnd);
	      }
	    else
	      break;
	  }
	  goto dflt;

	case WM_DROPFILES:
         {
           if (f && !f->iconified && f->visible &&
	       (w32_drop_file_handler(f, &msg, bufp)))
	     {
	       bufp++;
	       count++;
	       numchars--;
	     }
         }
	 break;

#ifdef W32_INTELLIMOUSE
	case WM_MOUSEWHEEL:
          {
            if (f && !f->iconified && f->visible &&
	        (w32_mouse_wheel_handler (f, &msg, bufp)))
	      {
		bufp++;
		count++;
		numchars--;
	      }
          }
          break;
#endif

	case WM_EMACS_CLEAR_MOUSE_FACE:
	  note_mouse_highlight(f, -1, -1);
	  break;
	  
	case WM_EMACS_DESTROY:
	  assert (numchars > 0);
		
	  bufp->kind = delete_window_event;
	  XSETFRAME (bufp->frame_or_window, f);
	  bufp++;
	  count++;
	  numchars--;
	  break;

	case WM_EMACS_CLOSE_CONNECTION:
	  ExitThread(1);

	case WM_EMACS_CREATE_FRAME:
	  w32_create_frame_window ((struct frame*)msg.wParam,
				   (LPSTR)msg.lParam);
	  break;
#ifdef IME_CONTROL
	case WM_MULE_IME_CREATE_AGENT:
	  w32_ime_create_agent ();
	  break;
#endif

	case WM_EMACS_ACTIVATE:
	  if (f && numchars > 0)
	    {
	      bufp->kind = meadow_private_event;
	      bufp->code = 0;
	      bufp->modifiers = 0;
	      XSETFRAME (bufp->frame_or_window, f);
	      bufp->timestamp = msg.time;
	      bufp++;
	      numchars--;
	      count++;
	    }
	  break;

	case WM_COMMAND:
	  if ((msg.lParam == 0) && (HIWORD(msg.wParam) == 0))
	    {
	      /* Came from window menu */

	      if (count != 0)
		{
		  exit_message_loop = 1;
		  break;
		}
	      menubar_selection_callback (msg.hwnd,
					  LOWORD(msg.wParam));
	    }
	  break;

	default:
	dflt:

#ifdef W32_INTELLIMOUSE
	    if (msg.message == w32_wheel_message)
	      {
		WORD wp;

		wp = LOWORD(msg.wParam);
		msg.wParam = MAKELONG(0, wp);
		if (f && !f->iconified && f->visible &&
		    (w32_mouse_wheel_handler (f, &msg, bufp))){
		  bufp++;
		  count++;
		  numchars--;
		}
		break;
	      }
#endif

	  TranslateMessage (&msg);
	  DispatchMessage (&msg);

	dflt_no_translate:	  
	  continue;
	}
      if (exit_message_loop){
	lastmsg = msg;
	lastmsgp = 1;
	break;
      }
    }

#if 0
  if (updating_frame == 0)
    w32_do_pending_expose ();
#endif

  /* If the focus was just given to an autoraising frame,
     raise it now.  */
  if (pending_autoraise_frame)
    {
      w32_raise_frame (pending_autoraise_frame);
      pending_autoraise_frame = 0;
    }

  return count;
}



/* Drawing the cursor.  */


/* Draw a hollow box cursor.  Don't change the inside of the box.  */

static void
w32_draw_box (f)
     struct frame *f;
{
  RECT rect;
  HBRUSH hb;
  HDC hdc;
  int c = FAST_GLYPH_CHAR (f->phys_cursor_glyph);
  int charset = CHAR_CHARSET (c);

  hdc = MyGetDC(FRAME_W32_WINDOW (f));

  hb = CreateSolidBrush (f->output_data.w32->cursor_pixel);

  rect.left = CHAR_TO_PIXEL_COL (f, curs_x);
  rect.top  = CHAR_TO_PIXEL_ROW (f, curs_y);
    
  /* 96.4.2 by himi */
  rect.right = 
    rect.left + 
    (FRAME_FONT_WIDTH(f) *
     (charset == CHARSET_COMPOSITION
      ? cmpchar_table[COMPOSITE_CHAR_ID (c)]->width
      : CHARSET_WIDTH (charset)));

  rect.bottom = rect.top + f->output_data.w32->line_height;

  rect.left++;
  rect.top++;
  rect.right--;
  rect.bottom--;

  FrameRect (hdc, &rect, hb);

  DeleteObject (hb);

  ReleaseDC (FRAME_W32_WINDOW(f), hdc);
}

/* Clear the cursor of frame F to background color,
   and mark the cursor as not shown.
   This is used when the text where the cursor is
   is about to be rewritten.  */

static void
clear_cursor (f)
     struct frame *f;
{
  if (! FRAME_VISIBLE_P (f)
      || f->phys_cursor_x < 0)
    return;

  w32_display_cursor (f, 0);
  f->phys_cursor_x = -1;
}

/* Redraw the glyph at ROW, COLUMN on frame F, in the style
   HIGHLIGHT.  HIGHLIGHT is as defined for dumpglyphs.  Return the
   glyph drawn.  */

static void
w32_draw_single_glyph (f, row, column, glyph, highlight)
     struct frame *f;
     int row, column;
     GLYPH glyph;
     int highlight;
{
  dumpglyphs (f,
	      CHAR_TO_PIXEL_COL (f, column),
	      CHAR_TO_PIXEL_ROW (f, row),
	      &glyph, 1, highlight, 0, NULL);
}

static void
w32_display_bar_cursor (f, on)
     struct frame *f;
     int on;
{
  struct frame_glyphs *current_glyphs = FRAME_CURRENT_GLYPHS (f);

  /* This is pointless on invisible frames, and dangerous on garbaged
     frames; in the latter case, the frame may be in the midst of
     changing its size, and curs_x and curs_y may be off the frame.  */
  if (! FRAME_VISIBLE_P (f) || FRAME_GARBAGED_P (f))
    return;

  if (! on && f->phys_cursor_x < 0)
    return;

  /* If we're not updating, then we want to use the current frame's
     cursor position, not our local idea of where the cursor ought to be.  */
  if (f != updating_frame)
    {
      curs_x = FRAME_CURSOR_X (f);
      curs_y = FRAME_CURSOR_Y (f);
    }

  /* If there is anything wrong with the current cursor state, remove it.  */
  if (f->phys_cursor_x >= 0
      && (!on
	  || f->phys_cursor_x != curs_x
	  || f->phys_cursor_y != curs_y
	  || FRAME_CURRENT_CURSOR (f) != bar_cursor))
    {
      /* Erase the cursor by redrawing the character underneath it.  */
      w32_draw_single_glyph (f, f->phys_cursor_y, f->phys_cursor_x,
			     f->phys_cursor_glyph,
			     current_glyphs->highlight[f->phys_cursor_y]);
      f->phys_cursor_x = -1;
    }

  /* If we now need a cursor in the new place or in the new form, do it so.  */
  if (on
      && (f->phys_cursor_x < 0
	  || (FRAME_CURRENT_CURSOR (f) != bar_cursor)))
    {
      f->phys_cursor_glyph
	= ((current_glyphs->enable[curs_y]
	    && curs_x < current_glyphs->used[curs_y])
	   ? current_glyphs->glyphs[curs_y][curs_x]
	   : SPACEGLYPH);
      w32_fill_area (f, NULL, f->output_data.w32->cursor_pixel,
		     CHAR_TO_PIXEL_COL (f, curs_x),
		     CHAR_TO_PIXEL_ROW (f, curs_y),
		     1, f->output_data.w32->line_height);

      f->phys_cursor_x = curs_x;
      f->phys_cursor_y = curs_y;

      FRAME_CURRENT_CURSOR (f) = bar_cursor;
    }
}

static void
w32_display_caret_cursor (f, on, caret_type)
     struct frame *f;
     int on, caret_type;
{
  struct frame_glyphs *current_glyphs = FRAME_CURRENT_GLYPHS (f);

  /* This is pointless on invisible frames, and dangerous on garbaged
     frames; in the latter case, the frame may be in the midst of
     changing its size, and curs_x and curs_y may be off the frame.  */
  if (! FRAME_VISIBLE_P (f) || FRAME_GARBAGED_P (f))
    return;

  if (! on && f->phys_cursor_x < 0)
    return;

  /* If we're not updating, then we want to use the current frame's
     cursor position, not our local idea of where the cursor ought to be.  */
  if (f != updating_frame)
    {
      curs_x = FRAME_CURSOR_X (f);
      curs_y = FRAME_CURSOR_Y (f);
    }

  /* If there is anything wrong with the current cursor state, remove it.  */
  if (f->phys_cursor_x >= 0
      && (!on
	  || f->phys_cursor_x != curs_x
	  || f->phys_cursor_y != curs_y
	  || (FRAME_CURRENT_CURSOR (f) != caret_cursor &&
	      FRAME_CURRENT_CURSOR (f) != checkered_caret_cursor &&
	      FRAME_CURRENT_CURSOR (f) != hairline_caret_cursor)))
    {
      /* Erase the cursor by redrawing the character underneath it.  */
      SEND_INFORM_MESSAGE (FRAME_W32_WINDOW(f), WM_EMACS_SETCARET, 0, 0);
      if (FRAME_CURRENT_CURSOR (f) != caret_cursor &&
	  FRAME_CURRENT_CURSOR (f) != checkered_caret_cursor &&
	  FRAME_CURRENT_CURSOR (f) != hairline_caret_cursor)
	w32_draw_single_glyph (f, f->phys_cursor_y, f->phys_cursor_x,
			       f->phys_cursor_glyph,
			       current_glyphs->highlight[f->phys_cursor_y]);
      f->phys_cursor_x = -1;
    }

  /* If we now need a cursor in the new place or in the new form, do it so.  */
  if (on
      && (f->phys_cursor_x < 0
	  || (FRAME_CURRENT_CURSOR (f) != caret_cursor &&
	      FRAME_CURRENT_CURSOR (f) != checkered_caret_cursor &&
	      FRAME_CURRENT_CURSOR (f) != hairline_caret_cursor)))
    {
      FRAME_CURRENT_CURSOR (f) = caret_type;
      f->phys_cursor_glyph
	= ((current_glyphs->enable[curs_y]
	    && curs_x < current_glyphs->used[curs_y])
	   ? current_glyphs->glyphs[curs_y][curs_x]
	   : SPACEGLYPH);
      f->phys_cursor_x = curs_x;
      f->phys_cursor_y = curs_y;
      SEND_INFORM_MESSAGE (FRAME_W32_WINDOW(f), WM_EMACS_SETCARET, 1, 0);
    }
}

/* Turn the displayed cursor of frame F on or off according to ON.
   If ON is nonzero, where to put the cursor is specified
   by F->cursor_x and F->cursor_y.  */

static void
w32_display_box_cursor (f, on)
     struct frame *f;
     int on;
{
  struct frame_glyphs *current_glyphs = FRAME_CURRENT_GLYPHS (f);

  /* This is pointless on invisible frames, and dangerous on garbaged
     frames; in the latter case, the frame may be in the midst of
     changing its size, and curs_x and curs_y may be off the frame.  */
  if (! FRAME_VISIBLE_P (f) || FRAME_GARBAGED_P (f))
    return;

  /* If cursor is off and we want it off, return quickly.  */
  if (!on && f->phys_cursor_x < 0)
    return;

  /* If we're not updating, then we want to use the current frame's
     cursor position, not our local idea of where the cursor ought to be.  */
  if (f != updating_frame)
    {
      curs_x = FRAME_CURSOR_X (f);
      curs_y = FRAME_CURSOR_Y (f);
    }

  /* If cursor is currently being shown and we don't want it to be
     or it is in the wrong place,
     or we want a hollow box and it's not so, (pout!)
     erase it.  */
  if (f->phys_cursor_x >= 0
      && (!on
	  || f->phys_cursor_x != curs_x
	  || f->phys_cursor_y != curs_y
	  || (FRAME_CURRENT_CURSOR (f) != hollow_box_cursor
	      && (f != w32_highlight_frame))))
    {
      int mouse_face_here = 0;

      /* If the cursor is in the mouse face area, redisplay that when
	 we clear the cursor.  */
      if (f == mouse_face_mouse_frame
	  &&
	  (f->phys_cursor_y > mouse_face_beg_row
	   || (f->phys_cursor_y == mouse_face_beg_row
	       && f->phys_cursor_x >= mouse_face_beg_col))
	  &&
	  (f->phys_cursor_y < mouse_face_end_row
	   || (f->phys_cursor_y == mouse_face_end_row
	       && f->phys_cursor_x < mouse_face_end_col)))
	mouse_face_here = 1;

      /* If the font is not as tall as a whole line,
	 we must explicitly clear the line's whole height.  */
#ifdef MW32_FONT
      w32_draw_single_glyph (f, f->phys_cursor_y, f->phys_cursor_x,
			     f->phys_cursor_glyph,
			     (mouse_face_here
			      ? 3
			      : current_glyphs->
			      highlight[f->phys_cursor_y]));
#else
      if (FONT_HEIGHT (f->output_data.w32->font) != f->output_data.w32->line_height)
	w32_clear_area (f, NULL,
			CHAR_TO_PIXEL_COL (f, f->phys_cursor_x),
			CHAR_TO_PIXEL_ROW (f, f->phys_cursor_y),
			FONT_WIDTH (f->output_data.w32->font),
			f->output_data.w32->line_height);
      /* Erase the cursor by redrawing the character underneath it.  */
      w32_draw_single_glyph (f, f->phys_cursor_y, f->phys_cursor_x,
			     f->phys_cursor_glyph,
			     (mouse_face_here
			      ? 3
			      : current_glyphs->
			      highlight[f->phys_cursor_y]));
#endif
      f->phys_cursor_x = -1;
    }

  /* If we want to show a cursor,
     or we want a box cursor and it's not so,
     write it in the right place.  */
  if (on
      && (f->phys_cursor_x < 0
	  || (FRAME_CURRENT_CURSOR (f) != filled_box_cursor
	      && f == w32_highlight_frame)))
    {
#if 0
      if (current_glyphs->enable[curs_y]) {
	if (curs_x > 0) {
	  f->phys_cursor_glyphs[0]
	    = ((curs_x - 1) < current_glyphs->used[curs_y])
	      ? current_glyphs->glyphs[curs_y][curs_x - 1]
		: SPACEGLYPH;
	}
	f->phys_cursor_glyphs[1]
	  = (curs_x < current_glyphs->used[curs_y])
	    ? current_glyphs->glyphs[curs_y][curs_x]
	      : SPACEGLYPH;
	f->phys_cursor_glyphs[2]
	  =((curs_x + 1) < current_glyphs->used[curs_y])
	    ? current_glyphs->glyphs[curs_y][curs_x + 1]
	      : SPACEGLYPH;
      } else {
	f->phys_cursor_glyphs[0] = SPACEGLYPH;
	f->phys_cursor_glyphs[1] = SPACEGLYPH;
	f->phys_cursor_glyphs[2] = SPACEGLYPH;
      }
#else
      f->phys_cursor_glyph
	= ((current_glyphs->enable[curs_y]
	    && curs_x < current_glyphs->used[curs_y])
	   ? current_glyphs->glyphs[curs_y][curs_x]
	   : SPACEGLYPH);
#endif
      if (f != w32_highlight_frame)
	{
	  w32_draw_box (f);
	  FRAME_CURRENT_CURSOR (f) = hollow_box_cursor;
	}
      else
	{
	  w32_draw_single_glyph (f, curs_y, curs_x,
				 f->phys_cursor_glyph, 2);
	  f->output_data.w32->current_cursor = filled_box_cursor;
	}

      f->phys_cursor_x = curs_x;
      f->phys_cursor_y = curs_y;
    }
}

void
w32_display_cursor (f, on)
     struct frame *f;
     int on;
{
  BLOCK_INPUT;

  if (FRAME_DESIRED_CURSOR (f) == filled_box_cursor)
    w32_display_box_cursor (f, on);
  else if (FRAME_DESIRED_CURSOR (f) == bar_cursor)
    w32_display_bar_cursor (f, on);
  else if (FRAME_DESIRED_CURSOR (f) == caret_cursor ||
	   FRAME_DESIRED_CURSOR (f) == checkered_caret_cursor ||
	   FRAME_DESIRED_CURSOR (f) == hairline_caret_cursor)
    w32_display_caret_cursor (f, on, FRAME_DESIRED_CURSOR (f));
  else
    /* Those are the only two we have implemented!  */
    abort ();

#if defined(MEADOW) && defined(IME_CONTROL)
  if ((f == w32_highlight_frame) && on)
    w32_set_ime_conv_window(f);
#endif

  UNBLOCK_INPUT;
}

/* Changing the font of the frame.  */

Lisp_Object
w32_new_font (struct frame *f, unsigned char *fontname)
{
  HWND hwnd;
  struct font_info *fontp;

  fontp = fs_load_font (f, W32_FONT_INFO_TABLE, CHARSET_ASCII, fontname, -1);

  if (!fontp) return Qnil;

  FRAME_FONT(f) = (W32FontStruct *)fontp->font;
  FRAME_FONT_BASELINE(f) = FONT_BASE((W32FontStruct *)(fontp->font));
  FRAME_FONTSET(f) = -1;

  /* Compute the scroll bar width in character columns.  */
  if (f->scroll_bar_pixel_width > 0)
    {
      int wid = FRAME_FONT_WIDTH (f);
      f->scroll_bar_cols = (f->scroll_bar_pixel_width + wid-1) / wid;
    }
  else
    f->scroll_bar_cols = 2;

  hwnd = FRAME_W32_WINDOW (f);
  if (hwnd)
    {
      frame_update_line_height (f);
      x_set_window_size (f, 0, f->width, f->height);
#if 0
      w32_set_ime_font(hwnd, get_logfont(FRAME_FONT(f), 0));
#endif

    }
  else
    /* If we are setting a new frame's font for the first time,
       there are no faces yet, so this font's height is the line height.  */
    f->output_data.w32->line_height = FRAME_FONT_HEIGHT(f);

  return build_string(fontp->full_name);
}

/* Give frame F the fontset named FONTSETNAME as its default font, and
   return the full name of that fontset.  FONTSETNAME may be a wildcard
   pattern; in that case, we choose some font that fits the pattern.
   The return value shows which font we chose.  */

Lisp_Object
w32_new_fontset (struct frame *f, char *fontsetname)
{
  int fontset = fs_query_fontset (f, fontsetname);
  struct fontset_info *fontsetp;
  Lisp_Object result;

  if (fontset < 0)
    return Qnil;

  fontsetp = FRAME_FONTSET_DATA (f)->fontset_table[fontset];

  if (!fontsetp->fontname[CHARSET_ASCII])
    /* This fontset doesn't contain ASCII font.  */
    return Qnil;

  result = w32_new_font (f, fontsetp->fontname[CHARSET_ASCII]);

  if (!STRINGP (result))
    /* Can't load ASCII font.  */
    return Qnil;

  /* Since x_new_font doesn't update any fontset information, do it now.  */
  FRAME_FONTSET(f) = fontset;
  FS_LOAD_FONT (f, W32_FONT_INFO_TABLE,
		CHARSET_ASCII, XSTRING (result)->data, fontset);

  return build_string (fontsetname);
}

int 
w32_get_glyph_width(struct frame *f, GLYPH g)
{
  int tlen = GLYPH_TABLE_LENGTH;
  Lisp_Object *tbase = GLYPH_TABLE_BASE;
  int cf, ch, charset;
  struct cmpchar_info *cmpcharp;
  int unit;

  GLYPH_FOLLOW_ALIASES (tbase, tlen, g);
  ch = FAST_GLYPH_CHAR (g);
  charset = CHAR_CHARSET (ch);
  if (charset == CHARSET_COMPOSITION)
    {
      cmpcharp = cmpchar_table[COMPOSITE_CHAR_ID (ch)];
      unit = cmpcharp->width;
    }
  else
    {
      unit = CHARSET_WIDTH(charset);
    }

  return FRAME_FONT_WIDTH(f) * unit;
}

w32_calc_absolute_position (f)
     struct frame *f;
{
#if 0
  Window win, child;
  int win_x = 0, win_y = 0;
  int flags = f->output_data.w32->size_hint_flags;

  /* Find the position of the outside upper-left corner of
     the inner window, with respect to the outer window.  */
  if (f->output_data.w32->parent_desc != ROOT_WINDOW)
    {
      BLOCK_INPUT;
      W32ranslateCoordinates (dsp,
			       
			     /* From-window, to-window.  */
			     f->output_data.w32->window_desc,
			     f->output_data.w32->parent_desc,

			     /* From-position, to-position.  */
			     0, 0, &win_x, &win_y,

			     /* Child of win.  */
			     &child);
      UNBLOCK_INPUT;
    }

  /* Treat negative positions as relative to the leftmost bottommost
     position that fits on the screen.  */
  if (flags & XNegative)
    f->output_data.w32->left_pos = (w32_screen_width 
			      - 2 * f->output_data.w32->border_width - win_x
			      - PIXEL_WIDTH (f)
			      + f->output_data.w32->left_pos);

  if (flags & YNegative)
    f->output_data.w32->top_pos = (w32_screen_height
			     - 2 * f->output_data.w32->border_width - win_y
			     - PIXEL_HEIGHT (f)
			     + f->output_data.w32->top_pos);
  /* The left_pos and top_pos
     are now relative to the top and left screen edges,
     so the flags should correspond.  */
  f->output_data.w32->size_hint_flags &= ~ (XNegative | YNegative);
#endif
}

/* CHANGE_GRAVITY is 1 when calling from Fset_frame_position,
   to really change the position, and 0 when calling from
   w32_make_frame_visible (in that case, XOFF and YOFF are the current
   position values).  */

/* to correspond to other emacs module....*/
x_set_offset (f, xoff, yoff, change_gravity)
     struct frame *f;
     register int xoff, yoff;
     int change_gravity;
{
  if (change_gravity)
    {
      f->output_data.w32->top_pos = yoff;
      f->output_data.w32->left_pos = xoff;
#if 0
      f->output_data.w32->size_hint_flags &= ~ (XNegative | YNegative);
      if (xoff < 0)
	f->output_data.w32->size_hint_flags |= XNegative;
      if (yoff < 0)
	f->output_data.w32->size_hint_flags |= YNegative;
      f->output_data.w32->win_gravity = NorthWestGravity;
#endif /* 0 */
    }

  w32_calc_absolute_position (f);

  BLOCK_INPUT;

  SetWindowPos (FRAME_W32_WINDOW (f),
		HWND_NOTOPMOST,
		f->output_data.w32->left_pos, f->output_data.w32->top_pos,
		0,0,
		SWP_NOZORDER | SWP_NOSIZE);
  UNBLOCK_INPUT;
}

/* Call this to change the size of frame F's x-window.
   If CHANGE_GRAVITY is 1, we change to top-left-corner window gravity
   for this size change and subsequent size changes.
   Otherwise we leave the window gravity unchanged.  */

void
x_set_window_size (f, change_gravity, cols, rows)
     struct frame *f;
     int change_gravity;
     int cols, rows;
{
  int pixelwidth, pixelheight;
  RECT rect;

  BLOCK_INPUT;

  check_frame_size (f, &rows, &cols);
  f->output_data.w32->vertical_scroll_bar_extra
    = (!FRAME_HAS_VERTICAL_SCROLL_BARS (f)
       ? 0
       : FRAME_SCROLL_BAR_PIXEL_WIDTH (f) > 0
       ? FRAME_SCROLL_BAR_PIXEL_WIDTH (f)
       : (FRAME_SCROLL_BAR_COLS (f) * FONT_WIDTH (f->output_data.w32->font)));

  rect.left = rect.top = 0;
  rect.right = CHAR_TO_PIXEL_WIDTH (f, cols);
  rect.bottom = CHAR_TO_PIXEL_HEIGHT (f, rows);

  AdjustWindowRectEx(&rect, f->output_data.w32->dwStyle,
		     FRAME_EXTERNAL_MENU_BAR (f),
		     f->output_data.w32->dwStyleEx);

  /* All windows have an extra pixel */

  SetWindowPos (FRAME_W32_WINDOW (f),
		HWND_NOTOPMOST,
		0, 0,
		rect.right - rect.left,
		rect.bottom - rect.top,
		SWP_NOZORDER | SWP_NOMOVE);

  /* Now, strictly speaking, we can't be sure that this is accurate,
     but the window manager will get around to dealing with the size
     change request eventually, and we'll hear how it went when the
     ConfigureNotify event gets here.

     We could just not bother storing any of this information here,
     and let the ConfigureNotify event set everything up, but that
     might be kind of confusing to the lisp code, since size changes
     wouldn't be reported in the frame parameters until some random
     point in the future when the ConfigureNotify event arrives.  */
  change_frame_size (f, rows, cols, 0, 0);
  PIXEL_WIDTH (f) = CHAR_TO_PIXEL_WIDTH (f, cols);
  PIXEL_HEIGHT (f) = CHAR_TO_PIXEL_HEIGHT (f, rows);

  /* If cursor was outside the new size, mark it as off.  */
  if (f->phys_cursor_y >= rows
      || f->phys_cursor_x >= cols)
    {
      f->phys_cursor_x = -1;
      f->phys_cursor_y = -1;
    }

  /* We've set {FRAME,PIXEL}_{WIDTH,HEIGHT} to the values we hope to
     receive in the ConfigureNotify event; if we get what we asked
     for, then the event won't cause the screen to become garbaged, so
     we have to make sure to do it here.  */
  SET_FRAME_GARBAGED (f);

  /* Clear out any recollection of where the mouse highlighting was,
     since it might be in a place that's outside the new frame size. 
     Actually checking whether it is outside is a pain in the neck,
     so don't try--just let the highlighting be done afresh with new size.  */
  cancel_mouse_face (f);

  UNBLOCK_INPUT;
}

/* Mouse warping, focus shifting, raising and lowering.  */
/* to correspond to other emacs module....*/
void
x_set_mouse_position (f, x, y)
     struct frame *f;
     int x, y;
{
  POINT pt;

#ifdef MW32_FONT
  pt.x = CHAR_TO_PIXEL_COL (f, x) + FRAME_FONT_WIDTH(f) / 2;
#else
  pt.x = CHAR_TO_PIXEL_COL (f, x) + FONT_WIDTH  (f->output_data.w32->font) / 2;
#endif
  pt.y = CHAR_TO_PIXEL_ROW (f, y) + f->output_data.w32->line_height / 2;

  if (pt.x < 0) pt.x = 0;
  if (pt.x > PIXEL_WIDTH (f)) pt.x = PIXEL_WIDTH (f);

  if (pt.y < 0) pt.y = 0;
  if (pt.y > PIXEL_HEIGHT (f)) pt.y = PIXEL_HEIGHT (f);

  BLOCK_INPUT;

  ClientToScreen(FRAME_W32_WINDOW(f), &pt);
  SetCursorPos (pt.x, pt.y);
  UNBLOCK_INPUT;
}

/* Move the mouse to position pixel PIX_X, PIX_Y relative to frame F.  */
/* to correspond to other emacs module....*/
void
x_set_mouse_pixel_position (f, pix_x, pix_y)
     struct frame *f;
     int pix_x, pix_y;
{
  POINT org;
 
  BLOCK_INPUT;
  org.x = org.y = 0;
  ClientToScreen (FRAME_W32_WINDOW(f), &org);
  SetCursorPos (pix_x + org.x, pix_y + org.y);
  UNBLOCK_INPUT;
}


/* focus shifting, raising and lowering.  */

x_focus_on_frame (f)
     struct frame *f;
{
  /* preserve Z-order, select frame */
}

x_unfocus_frame (f)
     struct frame *f;
{
  /* preserve Z-order, deselect frame */
}

/* Raise frame F.  */

static void
w32_raise_frame (f)
     struct frame *f;
{
  if (f->async_visible)
    {
      BLOCK_INPUT;
      SetForegroundWindow(FRAME_W32_WINDOW (f));
      UNBLOCK_INPUT;
    }
}

/* Lower frame F.  */

static void
w32_lower_frame (f)
     struct frame *f;
{
  if (f->async_visible)
    {
      BLOCK_INPUT;
      SetWindowPos(FRAME_W32_WINDOW (f),
		   HWND_BOTTOM,
		   0,0,0,0,
		   SWP_NOSIZE | SWP_NOMOVE);
      UNBLOCK_INPUT;
    }
}

static void
W32frame_raise_lower (f, raise)
     FRAME_PTR f;
     int raise;
{
  if (raise)
    w32_raise_frame (f);
  else
    w32_lower_frame (f);
}


/* Change from withdrawn state to mapped state,
   or deiconify. */
/* to correspond to other emacs module....*/
x_make_frame_visible (f)
     struct frame *f;
{
  int mask;

  BLOCK_INPUT;

  if (! FRAME_VISIBLE_P (f))
    {
      if (! FRAME_ICONIFIED_P (f))
	x_set_offset (f, f->output_data.w32->left_pos,
		      f->output_data.w32->top_pos, 0);
#if 1
      ShowWindow(FRAME_W32_WINDOW(f), SW_RESTORE);
      ShowWindow(FRAME_W32_WINDOW(f), SW_SHOW);
#else
      PostMessage(FRAME_W32_WINDOW(f), WM_EMACS_SHOWWINDOW,
		  (WPARAM)SW_SHOWNORMAL, 0);
#endif
    }

  UNBLOCK_INPUT;

  /* Synchronize to ensure Emacs knows the frame is visible
     before we do anything else.  We do this loop with input not blocked
     so that incoming events are handled.  */
  {
    Lisp_Object frame;
    XSETFRAME (frame, f);
#if 0
    while (! f->async_visible)
      {
	/* Machines that do polling rather than SIGIO have been observed
	   to go into a busy-wait here.  So we'll fake an alarm signal
	   to let the handler know that there's something to be read.
	   We used to raise a real alarm, but it seems that the handler
	   isn't always enabled here.  This is probably a bug.  */
	if (input_polling_used ())
	  {
	    /* It could be confusing if a real alarm arrives while processing
	       the fake one.  Turn it off and let the handler reset it.  */
	    alarm (0);
	    input_poll_signal ();
	  }
      }
#endif
    FRAME_SAMPLE_VISIBILITY (f);
  }
}

/* Change from mapped state to withdrawn state. */
/* to correspond to other emacs module....*/
x_make_frame_invisible (f)
     struct frame *f;
{
  /* Don't keep the highlight on an invisible frame.  */
  if (w32_highlight_frame == f)
    w32_highlight_frame = 0;

  BLOCK_INPUT;

#if 1
  ShowWindow(FRAME_W32_WINDOW(f), SW_HIDE);
#else
  PostMessage(FRAME_W32_WINDOW(f), WM_EMACS_SHOWWINDOW, (WPARAM)SW_HIDE, 0);
#endif

  /* We can't distinguish this from iconification
     just by the event that we get from the server.
     So we can't win using the usual strategy of letting
     FRAME_SAMPLE_VISIBILITY set this.  So do it by hand,
     and synchronize with the server to make sure we agree.  */
  f->visible = 0;
  FRAME_ICONIFIED_P (f) = 0;
  f->async_visible = 0;
  f->async_iconified = 0;

  UNBLOCK_INPUT;
}

/* Change window state from mapped to iconified. */
/* to correspond to other emacs module....*/
void
x_iconify_frame (f)
     struct frame *f;
{
  /* Don't keep the highlight on an invisible frame.  */
  if (w32_highlight_frame == f)
    w32_highlight_frame = 0;

  if (f->async_iconified)
    return;

  BLOCK_INPUT;

  /* Make sure the X server knows where the window should be positioned,
     in case the user deiconifies with the window manager.  */
  if (! FRAME_VISIBLE_P (f) && !FRAME_ICONIFIED_P (f))
    x_set_offset (f, f->output_data.w32->left_pos, f->output_data.w32->top_pos, 0);

#if 0
  ShowWindow(FRAME_W32_WINDOW(f), SW_SHOWMINIMIZED);
#else
  PostMessage(FRAME_W32_WINDOW(f), WM_SYSCOMMAND, SC_ICON, 0);
#endif

  cancel_mouse_face(f);
  f->async_iconified = 1;
  f->async_visible = 0;

  UNBLOCK_INPUT;

  return;
}

/* Destroy the X window of frame F.  */
/* to correspond to other emacs module....*/
x_destroy_window (f)
     struct frame *f;
{
  MSG msg;
  BLOCK_INPUT;

  SEND_INFORM_MESSAGE (FRAME_W32_WINDOW(f),
		       WM_EMACS_DESTROY_FRAME, 0, 0);
  WAIT_REPLY_MESSAGE (&msg, WM_EMACS_DESTROY_FRAME_REPLY);

  free_frame_menubar (f);
  free_frame_faces (f);

  CloseHandle(f->output_data.w32->mainthread_to_frame_handle);

  xfree (f->output_data.w32);
  f->output_data.w32 = 0;
  if (f == w32_focus_frame)
    w32_focus_frame = 0;
  if (f == w32_highlight_frame)
    w32_highlight_frame = 0;
  if (f == last_mouse_movement_frame)
    last_mouse_movement_frame = 0;

  if (f == mouse_face_mouse_frame)
    {
      mouse_face_beg_row = mouse_face_beg_col = -1;
      mouse_face_end_row = mouse_face_end_col = -1;
      mouse_face_window = Qnil;
    }

  UNBLOCK_INPUT;
}

/* Manage event queues for X10.  */

#if 0

/* Manage event queues.

   This code is only used by the X10 support.

   We cannot leave events in the X queue and get them when we are ready
   because X does not provide a subroutine to get only a certain kind
   of event but not block if there are no queued events of that kind.

   Therefore, we must examine events as they come in and copy events
   of certain kinds into our private queues.

   All ExposeRegion events are put in w32_expose_queue.
   All ButtonPress and ButtonRelease events are put in w32_mouse_queue.  */


/* Write the event *P_XREP into the event queue *QUEUE.
   If the queue is full, do nothing, but return nonzero.  */

int
enqueue_event (p_xrep, queue)
     register XEvent *p_xrep;
     register struct event_queue *queue;
{
  int newindex = queue->windex + 1;
  if (newindex == EVENT_BUFFER_SIZE)
    newindex = 0;
  if (newindex == queue->rindex)
    return -1;
  queue->xrep[queue->windex] = *p_xrep;
  queue->windex = newindex;
  return 0;
}

/* Fetch the next event from queue *QUEUE and store it in *P_XREP.
   If *QUEUE is empty, do nothing and return 0.  */

int
dequeue_event (p_xrep, queue)
     register XEvent *p_xrep;
     register struct event_queue *queue;
{
  if (queue->windex == queue->rindex)
    return 0;
  *p_xrep = queue->xrep[queue->rindex++];
  if (queue->rindex == EVENT_BUFFER_SIZE)
    queue->rindex = 0;
  return 1;
}

/* Return the number of events buffered in *QUEUE.  */

int
queue_event_count (queue)
     register struct event_queue *queue;
{
  int tem = queue->windex - queue->rindex;
  if (tem >= 0)
    return tem;
  return EVENT_BUFFER_SIZE + tem;
}

/* Return nonzero if mouse input is pending.  */

int
mouse_event_pending_p ()
{
  return queue_event_count (&w32_mouse_queue);
}
#endif /* HAVE_X11 */

void
w32_term_init (display_name, xrm_option, resource_name)
     char *display_name;
     char *xrm_option;
     char *resource_name;
{

  extern DWORD WINAPI async_handle_message(void*);

  Lisp_Object frame;
  char *defaultvalue;
  int argc = 0;
  char** argv = 0;
  
  w32_focus_frame = w32_highlight_frame = 0;

#if 0  
  if ((!noninteractive) && (!inhibit_window_system) && (initialized))
  {
    /* detach an old console (by himi) */
    FreeConsole();
    AllocConsole();
  }
#endif
  w32_id_name
    = (char *) xmalloc (LISPY_STRING_BYTES (Vinvocation_name)
			+ LISPY_STRING_BYTES (Vsystem_name)
			+ 2);
  sprintf (w32_id_name, "%s@%s",
	   XSTRING (Vinvocation_name)->data, XSTRING (Vsystem_name)->data);

  clear_frame_hook = W32clear_frame;
  clear_end_of_line_hook = W32clear_end_of_line;
  ins_del_lines_hook = W32ins_del_lines;
  change_line_highlight_hook = W32change_line_highlight;
  insert_glyphs_hook = W32insert_glyphs;
  write_glyphs_hook = W32write_glyphs;
  delete_glyphs_hook = W32delete_glyphs;
  ring_bell_hook = W32ring_bell;
  reset_terminal_modes_hook = W32reset_terminal_modes;
  set_terminal_modes_hook = W32set_terminal_modes;
  update_begin_hook = W32update_begin;
  update_end_hook = W32update_end;
  set_terminal_window_hook = W32set_terminal_window;
  read_socket_hook = W32read_socket;
  frame_up_to_date_hook = W32frame_up_to_date;
  cursor_to_hook = W32cursor_to;
  reassert_line_highlight_hook = W32reassert_line_highlight;
  mouse_position_hook = W32mouse_position;
  frame_rehighlight_hook = W32frame_rehighlight;
  frame_raise_lower_hook = W32frame_raise_lower;
  set_vertical_scroll_bar_hook = W32set_vertical_scroll_bar;
  condemn_scroll_bars_hook = W32condemn_scroll_bars;
  redeem_scroll_bar_hook = W32redeem_scroll_bar;
  judge_scroll_bars_hook = W32judge_scroll_bars;

  w32_mouse_grabbed = 0;
  scroll_region_ok = 1;		/* we'll scroll partial frames */
  char_ins_del_ok = 0;		/* just as fast to write the line */
  line_ins_del_ok = 1;		/* we'll just blt 'em */
  fast_clear_end_of_line = 1;	/* X does this well */
  memory_below_frame = 0;	/* we don't remember what scrolls 
				   off the bottom */
  baud_rate = 19200;

  /* Try to use interrupt input; if we can't, then start polling.  */
  Fset_input_mode (Qt, Qnil, Qt, Qnil);

#ifdef IME_CONTROL
  w32_ime_control_init();
#endif

#ifdef W32_INTELLIMOUSE
  w32_wheel_message = RegisterWindowMessage("MSWHEEL_ROLLMSG");
#endif

  next_message_block_event = CreateEvent(0, TRUE, TRUE, NULL);
  keyboard_handle = CreateEvent(0, TRUE, TRUE, NULL);

  main_thread = GetCurrentThread();
  main_thread_id = GetCurrentThreadId();

  /* Create message thread.  */
  /* Caution!!!!! inherited thread can't make Lisp Object directly 
     in stack.  */
  msg_thread = CreateThread(NULL, 0, async_handle_message,
			    0, 0, &msg_thread_id);
  SetThreadPriority(msg_thread, THREAD_PRIORITY_ABOVE_NORMAL);
  message_loop_blocked_p = 0;
  w32_frame_window = INVALID_HANDLE_VALUE;
}

/* to obtain system infomation. 96.10.10 by himi */
DEFUN ("w32-get-system-metrics", Fw32_get_system_metrics,
       Sw32_get_system_metrics,
       1, 1, 0, 
"Retrieve system metrics. This function only calls GetSystemMetrics.")
  (index)
{
  Lisp_Object ret;
  CHECK_NUMBER(index, 0);
  XSETINT(ret, GetSystemMetrics(XFASTINT(index)));
  return ret;
}

DEFUN ("w32-set-modifier-key", Fw32_set_modifier_key,
       Sw32_set_modifier_key, 2, 2, 0,
       "Set modifier key. \n\
KEY must be a virtual key code or string that describe a key.\n\
MODIFIER is one of these.\n\
'nil....normal key. 'none...neglected key. \n\
'meta...meta modifier. 'ctrl...ctrl modifier. \n\
'shift...shift modifier(it works modifier key only.)\n\
'alt...alt modifier. 'super...super modifier.\n\
'hyper...hyper modifier.")
     (key, modifier)
{
  int virtkey;

  if (NUMBERP(key))
    {
      virtkey = XFASTINT(key);
      if (!((key <= 255) && (key >= 0)))
	error("invalid key %d", virtkey);
    }
  else if (STRINGP(key))
    {
      extern char *lispy_function_keys[];
      int i;

      virtkey = -1;
      for (i = 0;i < 256;i++)
	{
	  if (lispy_function_keys[i] &&
	      (strcmp (lispy_function_keys[i],
		       XSTRING(key)->data) == 0))
	    {
	      virtkey = i;
	      break;
	    }
	}

      if (virtkey == -1)
	error ("Can't find the key:%s", XSTRING(key)->data);
    }
  else
    error ("KEY must be a string or number.");
      

  if (EQ(modifier, Qnil))
    {
      keymodifier[virtkey] &= 0x80;
    }
  else if (EQ(modifier, intern("none")))
    {
      keymodifier[virtkey] |= 0x01;
    }
  else if (EQ(modifier, intern("meta")))
    {
      keymodifier[virtkey] |= 0x02;
    }
  else if (EQ(modifier, intern("ctrl")))
    {
      keymodifier[virtkey] |= 0x04;
    }
  else if (EQ(modifier, intern("shift")))
    {
      keymodifier[virtkey] |= 0x08;
    }
  else if (EQ(modifier, intern("alt")))
    {
      keymodifier[virtkey] |= 0x10;
    }
  else if (EQ(modifier, intern("super")))
    {
      keymodifier[virtkey] |= 0x20;
    }
  else if (EQ(modifier, intern("hyper")))
    {
      keymodifier[virtkey] |= 0x40;
    }
  else
    {
      error("unknown modifier type!!");
    }
  return Qnil;
}

DEFUN ("w32-get-key-state", Fw32_get_key_state, Sw32_get_key_state,
       1, 1, 0, 
"Retrieve a key state when the previous message was received;\n\
not the current state. KEY is a virtual key code to get a state.")
  (key)
{
  int state;
  Lisp_Object ret;
  CHECK_NUMBER(key, 0);
  state = GetKeyState(XFASTINT(key));
  if (state & 0x8000) ret = Qt;
  else ret = Qnil;
  return ret;
}

DEFUN ("w32-get-mouse-wheel-scroll-lines", 
       Fw32_get_mouse_wheel_scroll_lines,
       Sw32_get_mouse_wheel_scroll_lines,
       1, 1, 0, 
       "Retrieve a number of scroll lines from delta number of mouse wheel.")
  (delta)
{
#ifdef W32_INTELLIMOUSE
  UINT lines;
  int dt;

  CHECK_NUMBER(delta, 0);

  dt = XINT(delta);
  if (!SystemParametersInfo(SPI_GETWHEELSCROLLLINES, 0, &lines, 0))
    return Qnil;

  if (lines == WHEEL_PAGESCROLL)
    if (dt > 0) return intern("above-handle");
    else if (dt < 0) return intern("below-handle");
    else dt = 0;

  return make_number(-((signed int)(lines) * dt / WHEEL_DELTA));
#else
  return make_number (0);
#endif
}

DEFUN ("w32-keyboard-type", 
       Fw32_keyboard_type,
       Sw32_keyboard_type,
       0, 0, 0, 
       "Retrieve w32 keyboard type with string.")
     ()
{
  int keyboardtypenum, oemkeyboardtype, functionkeys;
  char keyboardlayout[KL_NAMELENGTH];
  char buf[KL_NAMELENGTH + 20 + 20 + 20 +1];
  char *oemtype;
  static char* keyboardtype[] =
    {
      "unknown",
      "PC/XT#83",
      "Olivetti/ICO#102",
      "PC/AT#84",
      "PC/AT#101",
      "Nokia1050",
      "Nokia9140",
      "Japanese",
    };
  static char* japanoemname[] =
    {
      "Microsoft",
      "AX", 0, 0,
      "EPSON",
      "Fujitsu", 0,
      "IBM@Japan", 0, 0,
      "Matsushita", 0, 0,
      "NEC", 0, 0, 0, 0,
      "Toshiba",
    };

  keyboardtypenum = GetKeyboardType(0);
  functionkeys    = GetKeyboardType(2);
  oemkeyboardtype = GetKeyboardType(1);
  if (!GetKeyboardLayoutName(keyboardlayout)) return Qnil;
#if 0
  if (!((functionkeytype >= 1) && (functionkeytype <= 6)))
    functionkeytype = 0;
#endif
  if (!((keyboardtypenum >= 1) && (keyboardtypenum <= 7)))
    keyboardtypenum = 0;

  if (keyboardtypenum == 7)
    {
      if (!((oemkeyboardtype >= 0) && (oemkeyboardtype <= 18)))
	oemtype = 0;
      else
	oemtype = japanoemname[oemkeyboardtype];
    }
  else
    oemtype = 0;

  if (oemtype)
    sprintf(buf, "%s-%s-%d-%s",
	    keyboardtype[keyboardtypenum],
	    oemtype,
	    functionkeys,
	    keyboardlayout);
  else
    sprintf(buf, "%s-OEMNo.0x%08x-%d-%s",
	    keyboardtype[keyboardtypenum],
	    oemkeyboardtype,
	    functionkeys,
	    keyboardlayout);

  return build_string(buf);
}

#ifdef MEADOW

DEFUN ("Meadow-version",
       FMeadow_version,
       SMeadow_version,
       0, 1, 0, 
       "return the Meadow's version in string. \n\
The optional argument DUMMY is not currently used.")
     (dummy)
{
  return (build_string (MEADOW_VERSION_STRING));
}
#endif

void
syms_of_w32term ()
{
  DEFVAR_INT ("w32-lbutton-to-emacs-button", &w32_lbutton_to_emacs_button,
    "Position of a mouse button sent to emacs, when the w32 left button \n\
is changed.");
  DEFVAR_INT ("w32-mbutton-to-emacs-button", &w32_mbutton_to_emacs_button,
    "Position of a mouse button sent to emacs, when the w32 middle button \n\
is changed.");
  DEFVAR_INT ("w32-rbutton-to-emacs-button", &w32_rbutton_to_emacs_button,
    "Position of a mouse button sent to emacs, when the w32 right button \n\
is changed.");
  DEFVAR_INT ("w32-hide-mouse-timeout", &w32_hide_mouse_timeout,
    "Mouse cursor will hide after some rest. (in milliseconds) \n\
Cursor will not hide if 0. (default) \n\
This works correctly on Windows 98, 2000 or later.");
  DEFVAR_BOOL ("w32-hide-mouse-on-key", &w32_hide_mouse_on_key,
    "Non nil means mouse will hide on key input.");

  w32_hide_mouse_timeout = 0;    /* infinite */
  w32_hide_mouse_on_key = 0;
  w32_lbutton_to_emacs_button = 0;
  w32_mbutton_to_emacs_button = 2;
  w32_rbutton_to_emacs_button = 1;

#if 0
  staticpro (&last_mouse_scroll_bar);
  last_mouse_scroll_bar = Qnil;
#endif
  staticpro (&mouse_face_window);
  mouse_face_window = Qnil;
#ifndef HAVE_NTGUI
  defsubr (&Sset_message_beep);
#endif
  defsubr (&Sw32_get_system_metrics);
  defsubr (&Sw32_set_modifier_key);
  defsubr (&Sw32_get_key_state);
  defsubr (&Sw32_get_mouse_wheel_scroll_lines);
  defsubr (&Sw32_keyboard_type);
  defsubr (&SMeadow_version);
}
