#!/bin/bash
set -e
set -u

script_path=$(readlink -e "$0")
script_name=$(basename "$script_path")
script_dir=$(dirname "$script_path")

# Upstream Debian source package name, version, and release number.
DPN="ucspi-tcp"
DVER="0.88"
DREL="3"

# Vars for the Debian filenames.
orig_tar_gz="${DPN}_$DVER.orig.tar.gz"
diff_gz="${DPN}_$DVER-$DREL.diff.gz"
dsc="${DPN}_$DVER-$DREL.dsc"
patched_tar_gz="${DPN}_$DVER-$DREL.patched.tar.gz"

# Important for cygport to function:
export SRC_URI="file://./$patched_tar_gz"
export SRC_DIR="$DPN-$DVER"

# Parse the script name to deduce my PN, VER, and REL.
PN="$script_name"
ext="build.sh"
PN=${PN%%.$ext}
REL=${PN##*-}
PN=${PN%%-$REL}
VER=${PN##*-}
PN=${PN%%-$VER}
#echo "PN='$PN' VER='$VER' REL='$REL'"

# Convenience vars.
PKG="${PN}-${VER}-${REL}"
CYP="${PKG}.cygport"
SRC="${PKG}-src.tar.bz2"

usage_and_exit() {
    
    cat <<EOF 1>&2
Usage: $0 <beginstage> [ <endstage> (default=5) ]

Some cygport equivalences:

    'cygport $CYP prep'
        <==>        '$0 1 1'

    'cygport $CYP almostall'
        <==>        '$0 1'
        <==>        '$0 1 5'

    'cygport $CYP finish'
        <==>        '$0 6 6'

    'cygport $CYP all'
        <==>        '$0 1 6'
EOF
    exit 1
}

to_run_stage() {
    (( $1 >= beginstage )) && (( $1 <= endstage ))
}

# Get the build stages to begin and end at.

# The begin stage must be specified and in the range 1-6.
beginstage=${1:-99}
[[ $beginstage =~ ^[1-6]$ ]] || usage_and_exit

endstage=${2:-5}
[[ $endstage   =~ ^[1-6]$ ]] || usage_and_exit
(( beginstage <= endstage )) || usage_and_exit

# Now begin.

cd "$script_dir"

to_run_stage 1 && {
    
	#
	# Take the original source and apply the Debian patches to create
	# the patched source tarball that we consider our upstream release.
	#
	
    # Unpack the orig tarball downloaded from Debian.
    # Assume it creates a dir named $SRC_DIR.
    tar -xzf "$orig_tar_gz"
    cd "$SRC_DIR"
    
    # Apply the Debian patch.
    echo "Applying $diff_gz :"
    zcat "$script_dir/$diff_gz" | patch -p1
    
    # Apply additional patches for IPv6 capabilities.
    for dif in ucspi-tcp-ipv6/diff/* ; do
        echo "Applying" $(basename "$dif") ":"
        patch -p1 < "$dif"
    done
    
    # Roll the tarball and remove the source dir.
    cd "$script_dir"
    tar -czf "$patched_tar_gz" "$SRC_DIR"
    rm -rf "$SRC_DIR"
    
    # Now hand over to cygport, which will use $patched_tar_gz as its
    # source tarball, since SRC_URI and SRC_DIR are defined appropriately.
    cygport "$CYP" prep
}

to_run_stage 2 && cygport "$CYP" compile
to_run_stage 3 && cygport "$CYP" check
to_run_stage 4 && cygport "$CYP" install
to_run_stage 5 && {

    cygport "$CYP" package
    
    # To the source tarball, we need to
    #   remove: $patched_tar_gz
    #      add: $orig_tar_gz, $diff_gz, $dsc, and this script
    mytmp="$(mktemp -d)"
    cd       "$mytmp"
    tar -xjf "$script_dir/$SRC"
    rm       "$patched_tar_gz"
    cp -p    "$script_dir/$orig_tar_gz" \
             "$script_dir/$diff_gz" \
             "$script_dir/$dsc" \
             "$script_path" \
             .
    tar -cjf "$script_dir/$SRC" *
    cd       "$script_dir"
    rm -rf   "$mytmp"
    cp -p    "$SRC" "$PKG/dist/$PN"
}

to_run_stage 6 && {
    
    cygport "$CYP" finish
    rm -f "$patched_tar_gz"
}
