package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See L<http://www.perl.com/perl/misc/Artistic.html>

=cut


__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``3`$*
M``>RC54``````````.``+@,+`0(9`%(```#8````$@```!`````0````<```
M``!````0`````@``!`````$````$``````````!P`0``!```5G4!``,``(``
M`"```!``````$```$````````!``````$`$`_@$````@`0"<"0```#`!`+`H
M`````````````````````````&`!`%0#````T```'```````````````````
M``````````````````````````````````````#((0$`9`$`````````````
M`````````````````````"YT97AT````-%`````0````4@````0`````````
M`````````&``4&`N9&%T80```$0`````<`````(```!6````````````````
M``!``&#`+G)D871A``#$10```(````!&````6```````````````````0``P
M0"YB=6EL9&ED-0````#0`````@```)X``````````````````$``,$`N96A?
M9G)A;:P'````X`````@```"@``````````````````!``#!`+F)S<P````!P
M$0```/``````````````````````````````@`!PP"YE9&%T80``_@$````0
M`0```@```*@``````````````````$``,$`N:61A=&$``)P)````(`$```H`
M``"J``````````````````!``##`+G)S<F,```"P*````#`!```D````M```
M````````````````0``PP"YR96QO8P``5`,```!@`0``!````-X`````````
M`````````$``,$(`````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````%6)Y8/L&(/D\,<$
M)"!>0`#HZTL``,=$)`@`````QT0D!`````#'!"0`````Z"]-``#'1"0(````
M`,=$)`0`````QP0D`````.@C30``QT0D"`````#'1"0$`````,<$)`````#H
M%TT``,=$)`@`````QT0D!`````#'!"0`````Z`M-``#)PY"0D)"0D)"0D%6)
MY5=64X/L+(LU("-!`,<$)`"`0`#_UH/L!(7`B<,/A*T```"+%20C00#'1"0$
M#X!``(D$)(E5Y/_2@^P(B<>+5>3'1"0$)8!``(D<)/_2@^P(HP!P0`"%_W01
MQT0D!`#P0`#'!"0XX$``_]>A0'!``(7`=#C'!"0]@$``_]:#[`2%P+H`````
M=!;'1"0$2X!``(D$)/\5)"-!`(/L"(G"A=)T"<<$)$!P0`#_TL=$)`@$<$``
MQT0D!`````#'!"2`$4``Z$I&``"-9?1;7E]=P\<%`'!```````"_`````.ER
M____C78`C;PG`````*$`<$``A<!T$%6)Y8/L&,<$)#C@0`#_T,GSPY"0D)"0
M55<Q[593@>P,`P``B40D:(/`.(ET)`B+%:@B00")E"3\`@``,=*)!"2)?"0,
MC78`BP0DQT3L?`````#'A.R$`````````(M4J`2+!*B)P8G&B=?!X0C!YAB!
MX0``_P`)\8G&#ZS6"('F`/\```G.B<&)T`^LT1@QTH'A_P````GQB?Z)3.QX
MB?G!YAC!X0B!X0``_P`)SHGY#ZS1"`^LT!B!X0#_```)\27_````"<B)A.R`
M````@\4"@_T0#X5M____C80DX````(V4).`"``"+6-B+2-R#P`@S6/@S2/PS
M6*`S2*0S6)`S2)3'0!0`````B<^)V8G>#ZSY'XG[B0PDB?'!ZQ\!\0L,)(E<
M)`2)2!`YT'6ZBT0D:(M(#(M8$(MP%(LXBU`(BV@$B5PD"(E,)`2+6!R+2!B)
M="0,B10DB6PD+(E,)#")7"0TBT@@BU@DB5PD5(G>B<L#7"1XB4PD4!-T)'R)
MV`69>8):B?*)_H/2`(G!B?@/K.X;B40D*,'@!8G/"<:+!"2)TS'2`?>)W8E\
M)!"+?"0P$=6)QHEL)!3'1"04`````/?6B?.+="0((?N+?"00(<:)]0G=`>^)
M?"0@B7PD$(L$)(M4)`2+3"0HQT0D'`````#'1"0\`````,=$)$0`````B<>)
MQL=$)"0`````#ZS7`L'F'HM4)!2)_8M\)#`)]0.\)(````")SHEL)#CWUHML
M)#B)^(M\)"@%F7F"6B-T)`@AS0GUB>F+;"0L`<&+1"00B<;!X`4/K-8;BU0D
M.`G&B?")SHGY`<;!YQX/K.D"B70D&(ET)%B)S8M,)`@)_0.,)(@```"+?"04
MB6PD0(ML)!#WU8'!F7F"6B'JBVPD("-L)$")Z(G5BU0D'`G%BT0D&`'IB<N)
MQL'@!0^LUAL)Q@'SBW0D$(E<)&")7"0@B?'!YAX/K/D"B<T)]8EL)$B+5"0X
M`Y0DD````(ML)!C'1"04`````,=$)$P`````QT0D/`````#'1"1$`````/?5
MB=&+5"1`@<&9>8):(>J+;"18(VPD2(GHB=6+5"0D"<6+1"0@`>F)SXG&P>`%
M#ZS6&XM4)$`)Q@'WBW0D&(E\)!")?"18BWPD',=$)!P`````B?'!YAX/K/D"
MBWPD)(G-"?4#E"28````B6PD.(ML)"")T8M4)$CWU8'!F7F"6B'JBVPD8"-L
M)#B)Z(G5BU0D%`G%BT0D$`'IB<N)QL'@!0^LUAL)Q@'SBW0D((E<)&")7"08
MB?'!YAX/K/D"B<T)]8EL)$"+5"1(`Y0DH````(ML)!#'1"0D`````,=$)$P`
M````QT0D/`````#WU8G1BU0D.('!F7F"6B'JBVPD6"-L)$")Z(G5BU0D'`G%
MBT0D&`'IB<^)QL'@!0^LUAN+5"0X"<8!]XMT)!")?"0@B7PD6(M\)!3'1"04
M`````(GQP>8>#ZSY`HM\)!R)S0GU`Y0DJ````(EL)$B+;"08B=&+5"1`]]6!
MP9EY@EHAZHML)&`C;"1(B>B)U8M4)"0)Q8M$)"`!Z8G+B<;!X`4/K-8;BU0D
M0`G&`?.+="08B5PD8(E<)!")\<'F'@^L^0*)S0GUB6PD.`.4)+````"+;"0@
MQT0D'`````#'1"1$`````,=$)$P`````B=&+5"1(]]6!P9EY@EHAZHML)%@C
M;"0XB>B)U8M4)!0)Q8M$)!`!Z8G/B<;!X`4/K-8;BU0D2`G&`?>+="0@B7PD
M&(E\)%B+?"0DQT0D)`````")\<'F'@^L^0*+?"04B<T)]0.4)+@```");"1`
MBVPD$(G1BU0D./?5@<&9>8):(>J+;"1@(VPD0(GHB=6+5"0<"<6+1"08`>F)
MRXG&P>`%#ZS6&XM4)#@)Q@'SBW0D$(E<)&")7"0@B?'!YAX/K/D"B<T)]8EL
M)$@#E"3`````BVPD&,=$)!0`````QT0D/`````#'1"1$`````(G1BU0D0/?5
M@<&9>8):(>J+;"18(VPD2(GHB=6+5"0D"<6+1"0@`>F)SXG&P>`%#ZS6&XM4
M)$`)Q@'WBW0D&(E\)!")?"18BWPD',=$)!P`````B?'!YAX/K/D"BWPD)(G-
M"?4#E"3(````B6PD.(ML)"")T8M4)$CWU8'!F7F"6B'JBVPD8"-L)#B)Z(G5
MBU0D%`G%BT0D$`'IB<N)QL'@!0^LUAN+5"1("<8!\XMT)"")7"1@B5PD&(GQ
MP>8>#ZSY`HG-"?4#E"30````B6PD0(ML)!#'1"0D`````,=$)$P`````QT0D
M/`````")T8M4)#CWU8'!F7F"6B'JBVPD6"-L)$")Z(G5BU0D'`G%BT0D&`'I
MB<^)QL'@!0^LUAN+5"0X"<8!]XMT)!")?"0@B7PD6(M\)!3'1"04`````(GQ
MP>8>#ZSY`HM\)!R)S0GU`Y0DV````(EL)$B+;"08B=&+5"1`]]6!P9EY@EHA
MZHML)&`C;"1(B>B)U8M4)"0)Q8M$)"`!Z8G+B<;!X`4/K-8;BU0D0`G&`?.+
M="08B5PD8(E<)!")\<'F'@^L^0*)S0GU`Y0DX````(EL)#B+;"0@B=&+5"1(
M]]6!P9EY@EK'1"0<`````,=$)$0`````QT0D3``````AZHML)%@C;"0XB>B)
MU8M4)!0)Q8M$)!`!Z8G/B<;!X`4/K-8;BU0D2`G&`?>+="0@B7PD&(E\)%B+
M?"0DQT0D)`````")\<'F'@^L^0*+?"04B<T)]0.4).@```");"1`BVPD$(G1
MBU0D./?5@<&9>8):(>J+;"1@(VPD0(GHB=6+5"0<"<6+1"08`>F)RXG&P>`%
M#ZS6&XM4)#@)Q@'SBW0D$(E<)&")7"0@B?'!YAX/K/D"B<T)]0.4)/````")
M;"1(BVPD&(G1BU0D0/?5@<&9>8):(>J+;"18(VPD2,=$)!0`````QT0D/```
M``#'1"1$`````(GHB=6+5"0D"<6+1"0@`>F)SXG&P>`%#ZS6&XM4)$`)Q@'W
MBW0D&(E\)!")?"18BWPD',=$)!P`````B?'!YAX/K/D"BWPD)(G-"?4#E"3X
M````B6PD.(ML)"")T8M4)$CWU8'!F7F"6B'JBVPD8"-L)#B)Z(G5BU0D%`G%
MBT0D$`'IB<N)QL'@!0^LUAN+5"1("<8!\XMT)"")7"1@B5PD&(GQP>8>#ZSY
M`HG-"?4#E"0``0``B6PD0(ML)!")T8M4)#CWU8'!F7F"6B'JBVPD6"-L)$#'
M1"0D`````,=$)$P`````B>B)U8M4)!P)Q8M$)!@!Z8G/B<;!X`4/K-8;BU0D
M.`G&`?>+="00B7PD((E\)%B+?"04B?'!YAX/K/D"B<V+3"0\QT0D/``````)
M]0.4)`@!```3C"0,`0``B6PD2(ML)!B)RXG1BU0D0('!F7F"6O?5@],`,?\A
MZHML)&`C;"1(B>B)U8M4)"0)Q8M$)"`!Z1'[BWPD',=$)!P`````B<;!X`4/
MK-8;,=()QHGPB=Z)RP'#$=:+5"1`B5PD8(ET)&2+="08B5PD.(GQP>8>#ZSY
M`HG-"?4#E"00`0``B6PD;(EL)!B)T8M4)""+="18@<&9>8):BT0D.(M\)"3'
M1"04`````/?2(U0D2"'NBVPD;`GRB<8!T8M4)#S!X`4/K-8;BU0D2`G&B?")
MS@'&B70D0(ET)!"+="0@B?'!YAX/K/D"B?")S@G&`Y0D&`$``(ET)"")T(M4
M)!0%H>O9;C'UB>Z+;"1@,?6)Z0'!BT0D$(G&P>`%B40D2(M$)#B+;"1(#ZS6
M&XM4)#P)]8G.B<$![L'@'C'_#ZS1`HM4)!B+;"1`B<N)="1("<,#E"0@`0``
MB5PD.(G0!:'KV6XS7"0@,=V)Z0'!B?`/K/@;B40D&(GPBVPD&,'@!8E$)$`+
M;"1``>F)3"1`BU0D$(E,)!B+3"04BT0D(,=$)!P`````#ZS*`HM,)!")U8N4
M)"@!``#!X1X)S8N,)"P!``"!PJ'KV6Z);"0@@]$`B<N)T3'2`<@1VC-L)#B)
M1"00B50D%(M4)$B+1"08QT0D%``````QZHG1BU0D'`-,)!`/K-`;B?*)1"00
MBT0D&(ML)!#!X`6)1"1("VPD2(M$)#@!Z<'F'@^L^@*)3"00B4PD2(G5BU0D
M%`GUB[0D,`$``(EL)#B!QJ'KV6X!\(MT)"`Q[HML)$`Q]8GI`<&+1"00B<;!
MX`6)1"1`BVPD0(M$)!@/K-8;BU0D',=$)!P`````"?6)SHG!`>[!X!XQ_P^L
MT0*+E"0X`0``B70D0(G+"<.+1"0@B=V)7"0@BUPD.('"H>O9;@'0BU0D%#'K
MBVPD2#'=B>D!P8GP#ZSX&XE$)!B)\(ML)!C!X`6)1"1("VPD2(M$)!`!Z8E,
M)!B)3"1(B<$/K-$"BY0D0`$``,'@'HG-BXPD1`$```G%BT0D.('"H>O9;HEL
M)#B#T0")RXG1,=(!R!':,VPD((E$)!")5"04BU0D0(M$)!@QZHG1BU0D'`-,
M)!`/K-`;B?*)1"00BT0D&,'@!8E$)$"+;"00"VPD0(M$)"#'1"04``````'I
MP>8>#ZSZ`HE,)$")3"00B=6+5"04"?6+M"1(`0``B6PD(('&H>O9;@'PBW0D
M.#'NBVPD2#'UB>D!P8M$)!")QL'@!8E$)$B+1"08BVPD2`^LUAN+5"0<QT0D
M'``````)]8G.B<$![L'@'C'_#ZS1`HN4)%`!``")="1(B<L)PXM$)#B)W8E<
M)#B+7"0@@<*AZ]EN`="+5"04,>N+;"1`,=V)Z0'!B?`/K/@;B40D&(GPBVPD
M&,'@!8E$)$`+;"1`BT0D$`'IB4PD0(E,)!B)P0^LT0*+E"18`0``P>`>B<V+
MC"1<`0``"<6+1"0@@<*AZ]ENB6PD((/1`(G+B=$QT@'($=HS;"0XB40D$(E4
M)!2+5"1(BT0D&,=$)!0`````,>J)T8M4)!P#3"00#ZS0&XGRB40D$(M$)!B+
M;"00P>`%B40D2`ML)$B+1"0X`>G!YAX/K/H"B4PD$(E,)$B)U8M4)!0)]8NT
M)&`!``");"0X@<:AZ]EN`?"+="0@,>Z+;"1`,?6)Z0'!BT0D$(G&P>`%B40D
M0(M$)!B+;"1`#ZS6&XM4)!P)]8G.B<$![L'@'C'_#ZS1`HN4)&@!``")="1`
MB<L)PXM$)"")W8E<)""+7"0X@<*AZ]ENQT0D'``````!T(M4)!0QZXML)$@Q
MW8GI`<&)\`^L^!N)1"08B?"+;"08P>`%B40D2`ML)$B+1"00`>F)3"08B4PD
M2(G!#ZS1`HN4)'`!``#!X!Z)S8N,)'0!```)Q8M$)#B!PJ'KV6Z);"0X@]$`
MB<N)T3'2`<@1VC-L)"")1"00B50D%(M4)$"+1"08QT0D%``````QZHG1BU0D
M'`-,)!`/K-`;B?*)1"00BT0D&(ML)!#!X`6)1"1`"VPD0(M$)"`!Z<'F'@^L
M^@*)3"1`B4PD$(G5"?6+M"1X`0``@<:AZ]EN`?"+="0XB6PD((M4)!0Q[HML
M)$@Q]8GI`<&+1"00B<;!X`6)1"1(BT0D&(ML)$@/K-8;BU0D',=$)!P`````
M"?6)SHG!`>[!X!XQ_P^LT0*+E"2``0``B70D2(G+"<.+1"0XB=V)7"0XBUPD
M(('"H>O9;@'0BU0D%#'KBVPD0#'=B>D!P8GP#ZSX&XE$)!B)\(ML)!C!X`6)
M1"1`"VPD0(M$)!`!Z8E,)$")3"08B<$/K-$"BY0DB`$``,'@'HG-BXPDC`$`
M``G%BT0D(('"H>O9;HEL)""#T0")RXG1,=(!R!':,VPD.(E$)!")5"04BU0D
M2#'JB=$#3"00BT0D&(M4)!S'1"04``````^LT!N)\HE$)!"+1"08BVPD$,'@
M!8E$)$@+;"1(BT0D.`'IP>8>#ZSZ`HE,)$B)3"00B=6+E"20`0``"?6+="0@
MB6PD.('"H>O9;@'0,>Z+;"1`BU0D%#'UB>D!P8M$)!")QL'@!8E$)$"+1"08
MBVPD0`^LUAN+5"0<"?6)SHG!`>[!X!XQ_P^LT0*+E"28`0``B70D0(G+"<.+
M1"0@B=V)7"0@BUPD.('"H>O9;@'0,>N+;"1(,=V)Z0'!B?`/K/@;B40D&(GP
MBVPD&,'@!8E$)$@+;"1(`>F)3"1(B4PD&(M$)!"+5"04QT0D'`````")P<'@
M'@^LT0*+E"2@`0``B<V+C"2D`0``"<6+1"0X@<*AZ]ENB6PD.(/1`(G+B=$Q
MT@'($=HS;"0@B40D$(E4)!2+5"1`BT0D&,=$)!0`````,>J)T8M4)!P#3"00
M#ZS0&XGRB40D$(M$)!B+;"00P>`%B40D0`ML)$"+1"0@`>G!YAX/K/H"B4PD
M$(E,)%B)U8N4)*@!```)]8MT)#B);"1`@<*AZ]EN`=`Q[HML)$B+5"04,?6)
MZ0'!BT0D$(G&P>`%#ZS6&XM4)!P)QHGPB<X!QHM$)!@Q_XET)$B)P<'@'@^L
MT0*+E"2P`0``B<V+C"2T`0``"<6+1"0X@<*AZ]ENB6PD((/1`(G+B=$QT@'(
M$=HS;"1`B40D&(E4)!R+5"18B?#'1"0<`````#'JB=$#3"08BU0D%`^L^!N)
M1"08B?"+;"08P>`%B40D.`ML)#B+1"00`>F)3"08B4PD6(G!#ZS1`HN4)+@!
M``#!X!Z)S0G%BT0D0('"W+P;CXEL)#B)T8M4)"`!P8G0(U0D.`GH(T0D2(E4
M)!"+5"0<B<4+;"00BT0D&`'I#ZS0&XGRB40D$(M$)!B+;"00P>`%B40D0`ML
M)$"+1"0@QT0D%`````#'1"0D``````'IP>8>#ZSZ`HE,)$B)3"00B=>+E"3`
M`0``"?>)_HE\)$"+7"1`@<+<O!N/B=&)^HM\)!P!P8M$)#C'1"0<``````G&
M(W0D6"'"BT0D$(GUB<8)U8M4)!0!Z<'@!8E$)""+;"0@#ZS6&XN4),@!```)
M]8G.`>Z)="0@B70D6(MT)!B)\<'F'@^L^0*)\(G.BTPD2`G&BT0D.('"W+P;
MCXGWB70D.`'0"=\A^8GWB<(AWXG-BT0D(`G]BUPD.`'JB=&+5"0DB<;!X`6)
MSP^LUAN+E"30`0``"<:+1"1``?>+="00B7PD2(E\)!B+?"04B?'!YAX/K/D"
MB<V+3"18BWPD)`GU@<+<O!N/B=X!T`GN(>LA\8G"BT0D&`G9B6PD0,=$)!0`
M`````<K'1"0D`````(G1BU0D'(G&P>`%B<N)1"00BVPD$`^LUAN+E"38`0``
MBT0D.`GUBW0D(`'KBVPD0(E<)!")7"18B?'!YAX/K/D"B<^+3"1("?>!PMR\
M&X^)_@'0B7PD.`GNB<*+1"00(?&)_HM\)!PA[HG-BUPD.`GUB<8!ZL'@!8G1
MBU0D%(E$)""+;"0@#ZS6&XN4).`!```)]8MT)!@!Z8E,)"")3"1(B?'!YAX/
MK/D"B?")SHM,)%@)QHM$)$"!PMR\&X^)]8EL)$`!T`G>(?&)[HG"(=Z)S8M$
M)"`)]0'JB=&+5"0DB<;!X`6)SP^LUAL)Q@'WB7PD6(E\)!B+="00BWPD%(M<
M)$"+E"3H`0``BT0D.,=$)!P`````B?'!YA['1"04``````^L^0*+?"0DQT0D
M)`````")S8M,)$@)]8'"W+P;CXG>`=`)[B'K(?&)PHM$)!@)V8EL)#@!RHG1
MBU0D'(G&P>`%B<N)1"00BVPD$`^LUAN+E"3P`0``BT0D0`GUBW0D(`'KBVPD
M.(E<)!")7"1(B?'!YAX/K/D"B<^+3"18"?>!PMR\&X^)_@'0B7PD0`GNB<*+
M1"00(?&)_HM\)!PA[HG-"?6)Q@'JP>`%B=&+5"04B40D((ML)"`/K-8;BY0D
M^`$```GUBW0D&`'IB4PD6(E,)"")\<'F'@^L^0*)\(G."<:+1"0XBUPD0(M,
M)$B!PMR\&X^)]8EL)#C'1"0<``````'0"=XA\8GNB<(AWHG-BT0D(`GUBUPD
M.`'JB=&+5"0DB<;!X`6)SP^LUAN+E"0``@``"<:+1"1``?>+="00B7PD2(E\
M)!B+?"04QT0D%`````")\<'F'@^L^0*+?"0DB<V+3"18"?6!PMR\&X^)W@'0
M">XAZR'QB<*+1"08"=F);"1``<J)T8M4)!R)QL'@!8G+B40D$(ML)!`/K-8;
MBY0D"`(``(M$)#@)]8MT)"`!ZXML)$")7"00B5PD6(GQP>8>#ZSY`HG/BTPD
M2`GW@<+<O!N/B?X!T(E\)#@)[HG"BT0D$"'QB?XA[HG-"?6)Q@'JP>`%B=&+
M5"04B40D((ML)"`/K-8;"?4!Z8E,)$B+="08BWPD'(E,)""+E"00`@``BUPD
M.(GQP>8>QT0D)``````/K/D"B?#'1"0<`````(G.BTPD6`G&BT0D0('"W+P;
MCXGUB6PD0`'0"=XA\8GNB<(AWHG-BT0D(`GUBUPD0`'JB=&+5"0DB<;!X`6)
MSP^LUAN+E"08`@``"<:+1"0X`?>+="00B7PD&(E\)%B+?"04QT0D%`````")
M\<'F'@^L^0*+?"0DB<V+3"1("?6!PMR\&X^)W@'0">XAZR'QB<*+1"08"=F)
M;"0X`<J)T8M4)!R)QL'@!8G+B40D$(ML)!`/K-8;BY0D(`(``(M$)$`)]8MT
M)"`!ZXML)#B)7"1(B5PD$(GQP>8>#ZSY`HG/BTPD6,=$)"0`````"?>!PMR\
M&X^)_@'0B7PD0`GNB<*+1"00(?&)_HM\)!PA[HG-BUPD0`GUB<;'1"0<````
M``'JP>`%B=&+5"04B40D((ML)"`/K-8;BY0D*`(```GUBW0D&`'IB4PD6(E,
M)"")\<'F'@^L^0*)\(G.BTPD2`G&BT0D.('"W+P;CXGUB6PD.`'0"=XA\8GN
MB<(AWHG-BT0D(`GUBUPD.`'JB=&+5"0DB<;!X`6)SP^LUAN+E"0P`@``"<:+
M1"1``?>+="00B7PD&(E\)$B+?"04B?'!YAX/K/D"B<V+3"18"?6!PMR\&X^)
MW@'0">XAZR'QB<*+1"08"=F);"1``<J)T8M4)!R)QL'@!8G+B40D$(ML)!`/
MK-8;"?4!ZXE<)%B)7"00BW0D((M\)"2+E"0X`@``BVPD0(M$)#C'1"04````
M`(GQP>8>QT0D)``````/K/D"B<^+3"1("?>!PMR\&X^)_@'0B7PD.`GNB<*+
M1"00(?&)_HM\)!PA[HG-BUPD.`GUB<;'1"0<``````'JP>`%B=&+5"04B40D
M((ML)"`/K-8;BY0D0`(```GUBW0D&`'IB4PD2(E,)"")\<'F'@^L^0*)\(G.
MBTPD6`G&BT0D0('"W+P;CXGUB6PD0`'0"=XA\8GNB<(AWHG-BT0D(`GUBUPD
M0`'JB=&+5"0DB<;!X`6)SP^LUAN+E"1(`@``"<:+1"0X`?>+="00B7PD6(E\
M)!B+?"04B?'!YAX/K/D"B<V+3"1("?6!PMR\&X^)W@'0">XAZR'QB6PD.(G"
M"=F+1"08BWPD)`'*QT0D%`````#'1"0D`````(G1BU0D'(G&P>`%B<N)1"00
MBVPD$`^LUAN+E"10`@``BT0D0`GUBW0D(`'KBVPD.(E<)$B)7"00B?'!YAX/
MK/D"B<^+3"18"?>!PMR\&X^)_@'0B?L)[B'KB<(A\8M$)!")?"1`B<V+?"0<
M"=T!ZHG&P>`%B=&+5"04B40D((ML)"`/K-8;BY0D6`(```GUB<X![HML)$B)
M="0@B70D6(MT)!B)\<'F'@^L^0*)\(G."<:+1"0X@<+6P6+*B70D&`'0,W0D
M0(M4)"0Q]8GI`<&+1"0@B<^)QL'@!0^LUAN+5"04"<:+1"00`?>)?"1(B?XQ
M_XG!P>`>#ZS1`HN4)&`"``"+7"08B<W'1"04`````,=$)!P`````"<6+1"1`
M@<+6P6+*B6PD.`'0,>N+;"18BU0D)#'=B>D!P8GP#ZSX&XE$)!")\(ML)!#!
MX`6)1"1`"VPD0(M$)"`!Z8E,)!")3"1`B<$/K-$"P>`>BY0D:`(``(G-"<6+
M1"08@<+6P6+*B6PD(`'0,VPD.(M4)!0S;"1(B>D!P8M$)!`/K-`;B?*)1"08
MBT0D$(ML)!C!X`6)1"1("VPD2`'IP>8>#ZSZ`HGPB4PD2(E,)!B)UHN4)'`"
M```)QHM$)#B)="0X@<+6P6+*`=`S="0@BVPD0(M4)!R+7"0X,?6)Z0'!BT0D
M&(G&P>`%#ZS6&XM4)!3'1"04``````G&B?")S@'&BT0D$#'_B70D0(G!P>`>
M#ZS1`HN4)'@"``")S0G%BT0D(('"UL%BRHEL)"`!T#'KBVPD2(M4)!PQW8GI
M`<&)\`^L^!N)1"00B?"+;"00P>`%B40D2`ML)$B+1"08`>F)3"00B4PD2(G!
M#ZS1`L'@'HN4)(`"``")S0G%BT0D.('"UL%BRHEL)#@!T#-L)""+5"04,VPD
M0(GI`<&+1"00#ZS0&XGRB40D&(M$)!"+;"08P>`%B40D0`ML)$`!Z<'F'@^L
M^@*)\(E,)$")3"08B=:+E"2(`@``BVPD2`G&BT0D(,=$)!P`````B70D((M<
M)""!PM;!8LH!T#-T)#B+5"0<,?6)Z0'!BT0D&(G&P>`%#ZS6&XM4)!3'1"04
M``````G&B?")S@'&BT0D$#'_B70D2(G!P>`>#ZS1`HN4))`"``")S0G%BT0D
M.('"UL%BRHEL)#@!T#'KBVPD0(M4)!PQW8GI`<&)\`^L^!N)1"00B?"+;"00
MP>`%B40D0`ML)$"+1"08`>F)3"00B4PD0(G!#ZS1`L'@'HN4))@"``")S0G%
MBT0D(('"UL%BRHEL)"`!T#-L)#B+5"04,VPD2(GI`<&+1"00#ZS0&XGRB40D
M&(M$)!"+;"08QT0D'`````#!X`6)1"1("VPD2`'IP>8>BVPD0`^L^@*)\(E,
M)$B)3"08B=:+E"2@`@``"<:+1"0XB70D.(M<)#B!PM;!8LH!T#-T)""+5"0<
M,?6)Z0'!BT0D&(G&P>`%#ZS6&XM4)!3'1"04``````G&B?")S@'&BT0D$#'_
MB70D0(G!P>`>#ZS1`HN4)*@"``")S0G%BT0D(('"UL%BRHEL)"`!T#'KBVPD
M2(M4)!PQW8GI`<&)\`^L^!N)1"00B?"+;"00P>`%B40D2`ML)$B+1"08`>F)
M3"1(B4PD$(G!#ZS1`L'@'HN4)+`"``")S0G%BT0D.('"UL%BRHEL)#C'1"0<
M``````'0,VPD((M4)!0S;"1`B>D!P8M$)!`/K-`;B?*)1"08BT0D$(ML)!C!
MX`6)1"1`"VPD0`'IP>8>BVPD2`^L^@*)\(E,)$")3"08B=:+E"2X`@``"<:+
M1"0@B70D((M<)""!PM;!8LH!T#-T)#B+5"0<,?6)Z0'!BT0D&(G&P>`%#ZS6
M&XM4)!0)QHGPB<X!QHM$)!`Q_XET)$B)P<'@'@^LT0*+E"3``@``B<T)Q8M$
M)#B!PM;!8LJ);"0X`=`QZXML)$`QW8GI`<&)\`^L^!N)1"00B?"+;"00P>`%
MB40D0`ML)$`!Z8E,)$")3"00QT0D%`````"+1"08BU0D',=$)!P`````B<'!
MX!X/K-$"BY0DR`(``(G-"<6+1"0@@<+6P6+*B6PD(`'0,VPD.(M4)!0S;"1(
MB>D!P8M$)!`/K-`;B?*)1"08BT0D$(ML)!C!X`6)1"1("VPD2`'IP>8>BVPD
M0`^L^@*)\(E,)$B)3"08B=:+E"30`@``"<:+1"0XB70D.('"UL%BR@'0,W0D
M((M4)!PQ]8GI`<&+1"08B<;!X`4/K-8;BU0D%`G&B?")S@'&BT0D$#'_B70D
M6(G!P>`>#ZS1`HN4)-@"``")S8N,)-P"```)Q8M$)""!PM;!8LJ);"1`@]$`
MB<N)T3'2`<@1VHM<)#@QZXML)$@QW3';B>D!P8GP$=,QT@^L^!N)1"00B?"+
M;"00P>`%QT0D)`````#'1"04`````(E$)"`+;"0@BT0D&`'I$=.)RHG!B50D
M((E4)$C!X!Z+5"0<B5PD3`^LT0*+E"3@`@``B<V+C"3D`@``"<6+1"0X@<+6
MP6+*B6PD&(/1`(G+B=$QT@'($=HS;"1`,=LS;"18B>D!P8M$)"`1TXM4)"0/
MK-`;BU0D)(E$)!"+1"0@BVPD$`^DP@7!X`6)1"0X"VPD.(E4)#PQT@'IBVPD
M2!'3B<J)5"18B50D$(GR#ZSZ`L'F'HE<)%R)\(G6BY0DZ`(```G&BT0D0(ET
M)#B!PM;!8LH!T#-T)!@Q]8GI`<&+1"00BU0D%(M<)!2)QL'@!0^LUAN+5"0D
M"<:)\`'(,?^)QHM$)"")P<'@'@^LT0*+5"00B<V+3"04"<6+1"08B6PD2`^L
MR@+!Z0*)3"0DBTPD$(E4)""+E"3P`@``#Z3+'L'A'H'"UL%BRHE<)!2+7"0X
M`=")3"00B?HQZXML)%@QW8GI`<&)\,'J&P^L^!N)5"0<B?J)1"08B?"+;"08
MP>`%B40D0`ML)$`/I/(%B50D1(GHBVPD(`'(B<$#3"0HBT0D:(G#QT`$````
M`(D(BP0DQT,,``````'P"VPD$(E#"(GH`T0D",=#%`````#'0QP`````QT,D
M`````(E#$(M$)$@#1"0PB4,8BT0D.`-$)%")0R"+A"3\`@``,P6H(D$`=0N!
MQ`P#``!;7E]=P^BJ(```C78`C;PG`````(7`#X27````55=64X/L+(L-A`!!
M`(7)B4PD&'17#[8(@/D]=%N$R717B<.#PP$/M@N`^3UT!(3)=?$IPXM,)!B+
M*87M="Z)5"0<B<:)SV:0B5PD"(ET)`2)+"3H?"$``(7`=0>`?!T`/709@\<$
MBR^%[77<@\0L,<!;7E]=PV:0,=OKN(GZ*U0D&(M$)!S!^@*)$(/$+(U$'0%;
M7E]=PS'`PXVT)@````"-O"<`````4XG#@^P8H:@B00")1"0,,<"-5"0(B=CH
M-/___X7`=#"AA`!!`(M4)`B-%)"+2@2%R8D*=-R#P@2+2@2%R8D*=?2-5"0(
MB=CH!/___X7`=="+1"0,,P6H(D$`=06#Q!A;P^B*'P``C78`C;PG`````%57
MB<=64XG3@^P\H:@B00")1"0L,<`QP(`Z/0^4P`'#B1PDZ(P@``"-5"0HB<6)
M^.BK_O__A<")Q@^$Y````(D$).AM(```.<4/AL$```"+1"0HC12%``````^V
M!SP]#X1G`0``A,`/A%\!``")^.L$A,ET"X/``0^V"(#Y/77Q*?B+-80`00"-
M1"@"B00D`=;HEA\``(7`B08/A#\!``"AA`!!`(M4)"B-3P&+%)`/M@>$P(@"
M=1CK&HVV`````(/!`0^V0?^#P@&$P(@"=`0\/77LQ@(],<"-M"8`````#[8,
M`XA,`@&#P`&$R77Q,<"+?"0L,SVH(D$`#X7H````@\0\6UY?7<.-M"8`````
M@\,!#[9#_X/&`83`B$;_=>_KS(L-A`!!`(L!A<`/A*`````Q]H/&`8L$L87`
M=?:-!+4(````BQ5``$$`A=)T,(E$)`2)#"3H(!\``(7`HX0`00!T>HT4M0``
M``#'1+`$`````(ET)"CIW?[__XUV`(D$)(E,)!C'!4``00`!````Z)8>``"%
MP'1%BTPD&(T4M0````")!"2)1"08B50D"(E4)!R)3"0$Z(<>``"+1"08BU0D
M'*.$`$$`ZZ`QP.FO_O__N`@````Q]NEG____N/_____I"/___^B2'0``9I!5
M5XG%5E.![)P```"AJ")!`(F$)(P````QP(7M=`:`?0``=1Z+A"2,````,P6H
M(D$`#X4"`0``@<2<````6UY?7<.)+"3H(AX``(7`B40D''32BT0D'(D$).@>
M'@``A<`/A*H```"-6!2_8(!``+D"````B=[SIG39OV*`0`"Y`P```(G>\Z9T
MR8DL).@S'@``B1PDB<;H*1X``(U$!@*)!"3HE1T``(E<)`R);"0(B<;'1"0$
M98!``(D$).C;'0``C40D*(DT)(E$)`3HTQT``(/X_W00BT0D."4`\```/0!`
M``!T2HDT).C^'0``B30DZ!8=``"+1"0<B00DZ'H=``"%P`^%7/___XVV````
M`(M$)!R)!"3HT!P``(DL).AP'0``Z?O^__^-M"8`````B?#HR?[__^NUZ%(<
M``!FD%93@^P4H;0B00")!"3H-AP``(N`A````(MP!*&T(D$`B00DZ"`<```/
MMEAIH;0B00")!"3H#QP``(M`!(/C?P^W0!#!X`<)PV:)7@*AM")!`(D$).CO
M&P``BX"$````BU@$H;0B00")!"3HV1L``,=$)`@`````QT0D!&N`0`")!"3H
M:1L``(E#'*&T(D$`B00DZ+$;``"+@(0```"+6`2AM")!`(D$).B;&P``BX!T
M!```B4,@H;0B00")!"3HA1L``(N`A````(M8!*&T(D$`B00DZ&\;``"+@#P!
M```QTH7`=!:AM")!`(D$).A6&P``BX`\`0``BU!PH;0B00")4R2)!"3H/1L`
M`(N`A````(M`!,=`*`````"AM")!`(D$).@@&P``BX"$````BT`$QT`8````
M`*&T(D$`B00DZ`,;``"+@(0```"+6`2AM")!`(D$).CM&@``BX!(`@``B4,L
M@\046U[#C;8`````4XG#@^P8H;0B00")!"3HQ1H``(V02`$``(D<)(E4)`3H
M7QL``(7`=`>#Q!@QP%O#H;0B00")!"3HFQH``(N`6`$``"4`\```/0!```!T
M'Z&T(D$`B00DZ'P:``"+@%@!```E`/```#T`H```=;N)'"3'1"0$`@```.BN
M&@``A<`/E,"#Q!@/ML!;PY"-="8`55=64U"X*!```.A)&@``*<2)U8L$!(UT
M)!S'1"0(_P\``,<$)&R`0`")="0$B<>AJ")!`(F$)!P0```QP.@"&P``A<!^
M:HU8_X!\'!P`=%6)PXE<)`C'1"0$+P```(DT).B>&@``A<!T1J&T(D$`B00D
MZ-$9``")7"0,B70D"(E\)`2)!"3H+1D``(N$)!P0```S!:@B00!U-8'$+!``
M`%M>7UW#A=MUJ8VT)@````"AM")!`(D$).B+&0``B6PD"(E\)`2)!"3H\Q@`
M`.N\Z'P9``"-M@````"-OP````"#[!RAM")!`(D$).A8&0``QT0D#+>#0`#'
M1"0(P%9``,=$)`1[@$``B00DZ-@8``"#Q!S#C;8`````C;\`````@^PLH;0B
M00")!"3H&!D``(D$).BH____H;0B00")!"3H`QD``,=$)!0`````QT0D$&N`
M0`#'1"0,:X!``,=$)`B@44``QT0D!)>`0`")!"3H:Q@``(/$+,.-M"8`````
M,<##C;8`````C;PG`````%57N$P0``!64^BZ&```*<2+G"1@$```C50D.(N\
M)&00``"AJ")!`(F$)#P0```QP+BL@$``Z.;W___'1"0$+P```(D<)(G&Z)`9
M``"%P'0LB=JXM8!``.CD^/__B=B+C"0\$```,PVH(D$`#X5*`0``@<1,$```
M6UY?7<.)'"3H<!D``(D\),=$)`3"@$``C7PD/(E$)"SH>!D``(G%BT0D+(/`
M`8E$)"R-M"8`````A>UTG(!]``"X8(!```]$Z(7V=!2);"0$B30DZ!09``"%
MP`^$FP```(DL).@4&0``@'P%_R]U&(GVC;PG`````(/H`8!\!?\OQD0%``!T
M\8M,)"R-%`&!^O\/```/CS[___^)7"00QT0D#,2`0`");"0(QT0D!,:`0`")
M/"3HFA@``*&T(D$`B00DZ'D7```%2`$``(D\)(E$)`3HA!@``(7`=1^AM")!
M`(D$).A7%P``BX!8`0``)0#P```]`(```'0;QT0D!,*`0`#'!"0`````Z(T8
M``")Q>DB____QT0D!`$```")/"3H;A<``(7`==&)^KBU@$``Z)+W__^)/"3H
M-A@``.FC_O__Z``7``"+1"0$#[8(A,ET%8U0`6:0@/DO#[8*#T3"@\(!A,EU
M\//#C;0F`````(V\)P````!64X/L%(M<)""%VP^$GP````^V,XGPA,`/A)(`
M``")'"3HWA<``(U$`_\YPW()ZQR#Z`$YV'05@#@O=/3K!V:0@#@O=#"#Z`$Y
MPW+T.<-U)8GP/"]T6L=$)"#$@$``@\046U[IE!<``)"-M"8`````@#@O=0>#
MZ`$YPW+T*=B-2`*)PC'`@?D`$```=QR-0@&)7"0$QP0D0/!``(E$)`CH<!<`
M`+A`\$``@\046U[#D,=$)"!@@$``@\046U[I.A<``(VT)@````"-O"<`````
M5U:#[#2AJ")!`(E$)"PQP*&8@4``B40D"J&<@4``B40D#J&@@4``B40D$J&D
M@4``B40D%J&H@4``B40D&J&L@4``B40D'J&P@4``B40D(J&T@4``B40D)@^W
M!;B!0`!FB40D*KC-@$``Z*+U__^XUH!``.B8]?__N-Z`0`#HCO7__[CG@$``
MZ(3U__^X[H!``.AZ]?__N/Z`0`#H</7__[BL@$``Z&;U__^X"H%``.A<]?__
MN!2!0`#H4O7__[@>@4``Z$CU__^XM8!``.@^]?__N"B!0`#H-/7__[@Q@4``
MZ"KU__^-5"0$N#R!0`#H;/3__X7`=`R)PK@4@4``Z'SU__^-5"0$N$V!0`#H
M3O3__X7`=`R)PKA?@4``Z%[U__^-5"0$N&J!0`#H,/3__X7`="V)PKBL@$``
MZ$#U__^ZEH%``+CN@$``Z#'U__^+1"0L,P6H(D$`=4B#Q#1>7\.-5"0$N'J!
M0`#H\?/__X7`=`Z)PK@*@4``Z`'U___KOXUT)!N_BX%``+D*````\Z9UK8U4
M)"6X"H%``.C?]/__ZYWH6!0``)"-M"8`````@^P<H:@B00")1"0,,<"A"'!`
M`(/X_W08BTPD##,-J")!`'5`@\0<PY"-M"8`````C50D"+@*@4``Z'+S__^%
MP'0>#[80A-(/E<"`^C`/E<(/MM(AT*,(<$``Z[R-="8`,<#K\>CG$P``C;0F
M`````(/L'.AL%```B00DZ%P4``"%P'0,BP"#Q!S#C;8`````,<#K\HVV````
M`(V_`````%57O\*!0`!64[NZ@4``@^QLBZPD@````,=$)$#"@4``C70D2*&H
M(D$`B40D7#'`QT0D1+J!0`#'1"1(TH%``,=$)$S?@4``QT0D4.2!0`#'1"14
M]H%``,=$)%@`````ZR&);"0$B00DB40D*.B+%```A<!T2X7;="J+!HG?@\8$
MB<.-5"0\B?CH@_+__X7`=`6`.`!URXGJB?CHD?/__X7;==:+1"1<,P6H(D$`
M=7&#Q&Q;7E]=PXGVC;PG`````(DL).@<%```BTPD*(E$)"R)#"3H#!0``(M4
M)"R-1`("B00DZ'03``"+3"0HQT0D#,*`0`");"0(QT0D!,:`0`")!"2)1"0H
MB4PD$.BL$P``BU0D*(GXZ!7S___I5?___^B+$@``C70F`(V\)P````!55[@\
M@0``5E/H>A(``"G$C50D-(NL)%"!``#'1"0X#8)``*&H(D$`B80D+($``#'`
MN*R`0`#'1"0\8(!``,=$)$``````QT0D4%^!0`#'1"14$H)``,=$)%@9@D``
MQT0D7"&"0`#'1"1@)H)``,=$)&0`````QT0D1`&"0`#'1"1(*H)``,=$)$P`
M````Z$WQ__^%P'0)@#@`#X4P!```Z/O]__^%P(G##X0^!```B1PDZ/T2``"-
M1``!B00DZ&D2``"`.P")QHG'=":-="8`#[8#B3PD@\,!QT0D!#."0`"#QP*)
M1"0(Z)\2``"`.P!UWHU\)%2X7X%``.L-C78`@\<$BT?\A<!T*XU4)#3HS?#_
M_X7`B<-TYX`X`'3BZ'WV__^%P'39B1PDZ'T2``"%P(G!=4R+7"0XA=L/A',!
M``"`.P"_`0```'4=Z60!``"+7+PXA=L/A%@!``"#QP&`.P`/A$P!``")V.@Q
M]O__A<!TW8D<).@Q$@``A<!TT8G!B<B+&(/`!(V3__[^_O?3(=J!XH"`@(!T
MZ8G3P>L0]\*`@```#T33C5@"#T3#`-*)\X/8`RG(BSN#PP2-E__^_O[WUR'Z
M@>*`@("`=.F)UXE,)"#![Q#WPH"````/1->->P(/1-\`TH/;`RGSC9P8!`0`
M`(D<).@P$0``BTPD((ET)!2)QL=$)!`X@D``QT0D#,2`0`#'1"0$/8)``(D$
M)(E,)`B)1"0HZ%X1``#'1"0$P`$``(DT).@.$0``@_C_='BAM")!`(D$).@H
M$```!4@!``")1"0$BT0D*(D$).B_$```A<`/A),```"+70#H\P\``(M,)"C'
M1"0$P()``(E<)`B)3"0,BT`,B00DZ&`0```QP(N,)"R!```S#:@B00`/A?<)
M``"!Q#R!``!;7E]=PS')Z<;^___H_`\``(,X$0^$>O___^CN#P``BUT`BS#H
MD`\``(M,)"C'1"0$2()``(ET)!")7"0(B4PD#(M`#(D$).CY#P``,<#KEY"A
MM")!`(D$).AK#P``BX!8`0``)0#P```]`$````^%2O___Z&T(D$`B00DZ$@/
M``"+L&`!``#HV0\``#G&#X4J____H;0B00")!"3H*`\``(N`6`$``"7_`0``
M/<`!```/A0?___^)'"3HO@\``(U4)#2)1"0LN-^!0`#H8.[__X7`#X3X"```
MB00DZ!P0``")1"0$BT4`B00DZ"'V__^%P(G##X37"```B1PDZ`,0``"#^`=^
M*HUT`_B_@X)``+D)````\Z9U&(/X"`^$)P<``(!\`_<O#X0<!P``C70F`.A+
M^O__A<!U',=$)`0```$`B1PDZ%L/``"%P(E$)"`/A3<!``"ZEH%``+@*@4``
M,=OHZN[__^CI#@``BWPD*(MT)"S'1"04:X!``(E$)!#'1"0,Q(!``,=$)`2@
M@D``B7PD"(DT).A!#P``ZT&-=@#H9PX``(,X$75)@\,!Z*(.``#'1"08:X!`
M`(E<)!2)1"00QT0D#,2`0`")?"0(QT0D!*Z"0`")-"3H_@X``,=$)`3``0``
MB30DZ*X.``"#^/]TK8M$)"B)!"3H30X``(MT)"RXK(!``(GRZ$'N__^)-"3H
M"?K__XGPZ<C]__]FD(D$)(E$)"CH]/G__XM$)"B)!"3HQ`X``.FI_?__N`&"
M0`"^`0```.L2C;0F`````(M$M$2#Q@&%P'0EC50D-.C,[/__A<!TZ(`X`'3C
MB00DZ(<.``"%P'37B</I?_O__[`-NP:"0`#I?_O__\=$)`P"````QT0D!.[_
M__^)QL=$)`C_____B00DZ+X-``"-A"0L`0``QT0D"`8```")-"2)1"0$Z.L-
M``"`O"0L`0```'4.@+PD+0$``$,/A&0&``"+1"0@QT0D#`````#'1"0$````
M`,=$)`@`````B00DZ&<-``#'1"1H`2-%9\=$)&P`````QT0D<(FKS>_'1"1T
M`````,=$)'C^W+J8QT0D?`````#'A"2`````=E0R$,>$)(0`````````QX0D
MB````/#ATL/'A"2,`````````,>$))``````````QX0DE`````````#'A"28
M`````````,>$))P`````````QX0DX`````````"0C;0F`````(V$)"P!``#'
M1"0(`(```(E$)`2+1"0@B00DZ/4,``"%P(G%#X[P`0``BXPDD````(N<))0`
M``")Z(GOP>`#B>[!_Q\!R(/[``^'L@$``#G!#X>J`0``BXPDF````(N<))P`
M```/K/X=B80DD````(N$).````#![QT!\<>$))0`````````$?N)C"28````
MA<")G"2<````#X13`0``OT````"-C"0L`0``*<>-A`2@````.?V)3"0$#T[]
MB00DB7PD".@D#```BX0DX`````'X@_A`B80DX`````^%'____XV$)"P!```I
M_0''C40D:.AKQ?__@_T_#X[0````B>Z)^XL#@^Y`@\-`B80DH````(M#Q(F$
M)*0```"+0\B)A"2H````BT/,B80DK````(M#T(F$)+````"+0]2)A"2T````
MBT/8B80DN````(M#W(F$)+P```"+0^")A"3`````BT/DB80DQ````(M#Z(F$
M),@```"+0^R)A"3,````BT/PB80DT````(M#](F$)-0```"+0_B)A"38````
MBT/\B80DW````(U$)&CHL,3__X/^/P^/2?___X/M0(GHP>@&C5`!P>`&*<7!
MX@8!UXV$)*````");"0(B7PD!(D$).@'"P``B:PDX````.D/_O__C;PD+`$`
M`.GX_O__C78`BXPDF````(N<))P```"#P0&#TP#I2_[__XM$)"")!"3H7PH`
M`(N$))@```"+G"20````B[0DE````(N4))P```")1"0@B=@/K/`#B50D)(G!
M@^$_C4$!QH0,H````("#^#@/CW0"``"Z-P```"G*C8P,F0```(/Z!(U!"`^#
MHP(``(72=`WVP@+&00@`#X6E`P``BWPD((ML)"2(G"3?````B?@/K.@8B(0D
MV````(GX#ZSH$(B$)-D```")^`^LZ`B-K"0#`0``B(0DV@```(GXC;PD\```
M`(B$)-L```")V`^L\!B(A"3<````B=@/K/`0B(0DW0```(G8#ZSP"(B$)-X`
M``"-1"1HZ%##__^+1"1HBU0D;(M<)'2)P8B$)/(````/K-$8B<Z(C"3O````
MB<$/K-$0B(PD\````(G!#ZS1"(B,)/$```"+3"1PB<B(C"3V````#ZS8&(B$
M)/,```")R`^LV!"(A"3T````B<B+3"1X#ZS8"(M<)'R(A"3U````B<B(C"3Z
M````#ZS8&(B$)/<```")R`^LV!"(A"3X````B<B+C"2`````#ZS8"(N<)(0`
M``"(A"3Y````B<B(C"3^````#ZS8&(B$)/L```")R`^LV!"(A"3\````B<B+
MC"2(````#ZS8"(N<)(P```"(A"3]````B<B(C"0"`0``#ZS8&(B$)/\```")
MR`^LV!"(A"0``0``B<@/K-@(C9PD*P$``(B$)`$!``")\.D,````C;8`````
M#[8'@\<!#[;`B2PDQT0D!#."0`")1"0(@\4"Z-P(```YW77<C80D`P$``,:$
M)"L!````QT0D%&N`0`")1"00BT0D*,=$)`S$@$``QT0D!)&"0`")1"0(BT0D
M+(D$).B9"```Z:OY__^+;02%[0^$W?C__XDL).BI"```@_@##X;,^/__C70%
M_+^,@D``N04```#SI@]$W>FT^/__N#\````IR(G"C80,F0```(/Z!(U("'-P
MA=)T#?;"`L9`"``/A3\!``"-1"1HZ%+!__^-E"2@````N0X````QP(G7\ZOI
M;OW__\=!"`````"#P0S'1`+\`````(/A_"G(`="#X/R#^`0/@DG]__^#X/R)
MPC'`QP0!`````(/`!#G0<O+I+_W__XUX#,=`"`````")R,=$"OP`````@^?\
M*?@!T(/@_(/X!`^">O___X/@_(G",<#'!`<`````@\`$.=!R\NE@____@+PD
M+@$``$$/A8[Y__^`O"0O`0``0P^%@/G__X"\)#`!``!(#X5R^?__@+PD,0$`
M`$4/A63Y__^+="0@QT0D#`(```#'1"0$QO___\=$)`C_____B30DZ,L&``"-
MA"0L`0``QT0D""D```")-"2)1"0$Z/@&``#'1"04:X!``(V$)"P!``#I2_[_
M_XM=`.DA]___,?]FB7P"_NE/_/__,>UFB6P*_NFU_O__Z-L%``"-="8`C;PG
M`````%575E.#["RAJ")!`(E$)!PQP*&T(D$`B00DZ*D%``"-5"08QD!E`;CN
M@$``Z/_D__^%P'0)@#@Q#X2"`@``Z(WO__^AM")!`(D$).AX!0``QT0D#`L`
M``#'1"0(`0```,=$)`03@T``B00DZ!`%``"%P`^$R````(,]2`!!``&+0!"+
M>!@/CK4````Q]KL!````ZRZ+11")1"0$H40`00"+!)B)!"3H208``(7`#X5]
M````H4@`00"#PP&#Q@$YV'YSH;0B00")!"3H^00``,=$)`P`````B70D"(E\
M)`2)!"3HN00``(7`=$6+*(M%""4`!"``/0`$``!TF*&T(D$`B00DZ+\$``#'
M1"0,`@```,=$)`@`````B6PD!(D$).@C!```Z6W___^-M@````"A2`!!`(U0
M_SGR#X1``0``H;0B00")!"3H>P0``,=$)`P#````QT0D"`$```#'1"0$&(-`
M`(D$).@3!```A<")PW0UBT`0BS"AM")!`(D$).A#!```QT0D"!J#0`")!"2)
M="0$Z*<#``"+0Q"+&/9#"D`/A2D"``"AM")!`(D$).@3!```QT0D#`,```#'
M1"0(`0```,=$)`0?@T``B00DZ*L#``"%P(G#=#J-5"08N+6`0`#H1^/__X7`
MB<8/A%X!``"+0Q"+.*&T(D$`B00DZ,,#``")="0(B7PD!(D$).@K`P``H;0B
M00")!"3HI@,``(U4)!C&0&4`N*R`0`#H_.+__X7`HTP`00`/A(\```#'1"0$
MP`$``(D$).A+!```C5`!@_H!=@F#^!$/A9X!``"+1"0<,P6H(D$`#X6)`0``
M@\0L6UY?7<.-M@````"#^`$/CK?^__^[`0```&:0H;0B00"#PP&)!"3H*`,`
M`(E\)`2)!"3H[`(``#D=2`!!`'_<Z8?^__^-M"8`````@'@!``^$>?W__^EO
M_?__D*%$`$$`B00DZ'/P__^%P*-,`$$`#X0%`0``BT,0BSU$`$$`BS"+1@@E
M``0@`#T`!```#X6;````BT80B7PD!(D$).@=`P``B<:+0Q"+&(M3"('B``0@
M`('Z``0``'5)BT,0B40D"*%$`$$`B70D#(L`QP0D2(-``(E$)`3H/P(``*%$
M`$$`BSB+0Q"+,*&T(D$`B00DZ%X"``")^HGPZ`WH___IG?[__Z&T(D$`B00D
MZ$,"``#'1"0,`@```,=$)`@`````B5PD!(D$).BG`0``ZX^AM")!`(D$).@8
M`@``QT0D#`(```#'1"0(`````(ET)`2)!"3H?`$``.DZ____H;0B00")!"3H
MZ@$``(E<)`2)!"3HC@$``.FY_?__QP0D)(-``.B5`0``Z-`!``")1"0,H4P`
M00")1"0(H40`00"+`,<$)'"#0`")1"0$Z&P!``"-M@````"-OP````!64X/L
M%(MT)"")-"3HO^K__P^V"(G#A,ET%HU0`8UV`(#Y+P^V"@]$VH/"`83)=?#H
M&^W__X7`=!V%]G09@#X`=!3'1"0$.()``(D<).B:`@``.=AT!H/$%%M>PX/$
M%(GP6U[IJ./__Y"-M"8`````@^P<H;0B00")!"3H&`$``,=$)`3_____B00D
MZ,@```"AM")!`(D$).C[````QT0D"`````#'1"0$(#M``(D$).AK````@\0<
MPY"0D)"0D)#_)1@C00"0D/\E%"-!`)"0_R40(T$`D)#_)0PC00"0D/\E""-!
M`)"0_R4$(T$`D)#_)0`C00"0D/\E_")!`)"0_R7X(D$`D)#_)?0B00"0D/\E
M\")!`)"0_R7L(D$`D)#_)>@B00"0D/\EY")!`)"0_R7@(D$`D)#_)=PB00"0
MD/\EV")!`)"0_R74(D$`D)#_)=`B00"0D/\ES")!`)"0_R7((D$`D)#_)<0B
M00"0D/\EP")!`)"0_R6\(D$`D)#_);@B00"0D/\ET"%!`)"0_R58(D$`D)#_
M)5PB00"0D/\EI")!`)"045`]`!```(U,)`QR%8'I`!```(,)`"T`$```/0`0
M``!WZRG!@PD`6%G#D)#_)<@A00"0D/\ES"%!`)"0_R74(4$`D)#_)?@A00"0
MD/\E`")!`)"0_R4$(D$`D)#_)10B00"0D/\E&")!`)"0_R4<(D$`D)#_)2`B
M00"0D/\E)")!`)"0_R7<(4$`D)#_)>`A00"0D/\EZ"%!`)"0_R7L(4$`D)#_
M)3@B00"0D/\E/")!`)"0_R5`(D$`D)#_)40B00"0D/\E2")!`)"0_R7P(4$`
MD)#_)5`B00"0D/\E8")!`)"0_R5D(D$`D)#_)6@B00"0D/\E;")!`)"0_R5P
M(D$`D)#_)70B00"0D/\E]"%!`)"0_R5\(D$`D)#_)8`B00"0D/\EA")!`)"0
M_R6((D$`D)#_)8PB00"0D/\ED")!`)"0_R64(D$`D)#_)9@B00"0D/\EG")!
M`)"09I!FD&:09I!FD&:058GE4X/L%(M=#,<$)`@```#HR@0``(7;#X3(`0``
MBQ7D(4$`A<"+30C'0P2H````QT,(T`<``(E3$(M5`,=##`0```#'@X``````
M````QX.$````'P$``,=#+"!@0`#'0S`L8$``QT,4A`!!`(E+*,=#2&!=0`#'
M0TQP74``QT-0@%U``,=#5)!=0`#'0R2``$$`B1/'0Q@<6$``QT,<[%=``,=#
M(&Q80`#'0T2P74``QX.0````R%U```^$P0```(M0>(72#X2V````BPV(`$$`
MA<D/A:@```"A('!``(7`#X0I`@``H21P0`"%P`^$#`(``*$H<$``A<`/A.\!
M``"A+'!``(7`#X32`0``H3!P0`"%P`^$M0$``*$T<$``A<`/A)@!``"+#3AP
M0`"%R0^$>@$``*$\<$``A<`/A%T!``"A('!``(D"H21P0`")0@2A*'!``(E"
M"*$L<$``B4(,H3!P0`")0A"A-'!``(E"%*$X<$``B4(8H3QP0`")0AS'!"0`
M````Z*H#``#'0S0`<$``QT,X1'!``,=#/`#P0`#'0T!P`4$`QX.4````6,)`
M`(/L!(E#?,>#F````,3%0`#'@YP``````$``QP0D`````.CF`@``N`$```"+
M7?S)P@@`BXBD````A<"+%>0A00#'0`2H````QT`(T`<``,=`#`0```")"HM5
M`(M-",>`@`````````#'@(0````?`0``QT`L(&!``,=`,"Q@0`#'0!2$`$$`
MB4@HQT!(8%U``,=`3'!=0`#'0%"`74``QT!4D%U``,=`)(``00")$,=`&!Q8
M0`#'0!SL5T``QT`@;%A``,=`1+!=0`#'@)````#(74``#X3T_O__BU!XB<.%
MT@^%,?[__^GB_O__BT(<HSQP0`#IEO[__XUV`(M"&*,X<$``Z7G^__^-=@"+
M0A2C-'!``.E;_O__C78`BT(0HS!P0`#I/O[__XUV`(M"#*,L<$``Z2'^__^-
M=@"+0@BC*'!``.D$_O__C78`BT($HR1P0`#IY_W__XUV`(L"HR!P0`#IR_W_
M_Y"0D)!5B>6#[!B+10C'1"0$`````(D$).C'_/__@^P(_Q78(4$`D)"0D)"0
MD)"0D)"0D)!5B>6#[!BAD`!!`(E$)`BAE`!!`(E$)`2AF`!!`(D$).C[````
M@^P,R<.-M@````!5B>6#[!B+10R+50B+31"#^`%T?'([@_@"=!J#^`-FD'4F
MB4T0QT4,`P```(E5",GINP```(E-$,=%#`(```")50C)Z:@```"X`0```,G"
M#`")3"0(QT0D!`````")%"3HBP```(/L#(7`=.&+%8P`00")1?2)%"3HVP``
M`(M%],<%C`!!`/_____KP8UT)@`QP(7)#Y3`QT0D!#!<0`")%"2)%9@`00#'
M!90`00`!````B0V0`$$`HX@`00#HK@```(/X_Z.,`$$`#Y7`#[;`R<(,`(VV
M`````(V_`````%6)Y<GI%____Y"0D)"0D)"X`0```,(,`)"0D)"0D)"0\\.0
MD)"0D)"0D)"0D)"0D//#D)"0D)"0D)"0D)"0D)#SPY"0D)"0D)"0D)"0D)"0
M\\.0D)"0D)"0D)"0D)"0D//#D)"0D)"0D)"0D)"0D)#_)?PA00"0D/\E"")!
M`)"0_R4,(D$`D)#_)50B00"0D%6)Y5.#[!2+10S'1"0$H`!!`(M="(D$).CS
M^O__B5T(BUW\QT4,H`!!`(/L",GI#0```)"0D)"0D)"0D)"0D)#_)1`B00"0
MD/\E("-!`)"058GE5E.#Y/"#[$"+10R)1"0LBT40B40D*.AV^?__H:@B00")
M1"0\,<"-1"0HB40D"(U$)"R)1"0$C44(B00DZ&SX___'1"0(2%=``,=$)`1(
M5T``QP0D4%=``.C@^/__@#VP(D$``'4NZ"KX__^%P*-0`$$`#X1<`0``B00D
MZ`WX__^AM")!`(D$).BX^/__QH#%!0```*&T(D$`B00DZ*3X__^`B%8#```"
MH;0B00")!"3HD/C__X"(5@,```&+10B-!(44````B00DZ`?X__^+="0LBUT(
MHT0`00#'!4@`00`$````BQ:#^P''0`2Q@T``QT`,M(-``(D0BQ4,<$``B5`(
MN@$```!^$HUT)@"+#):)3)`,@\(!.=IU\L=$F`P`````B40D#(/#`Z%0`$$`
MB5PD",=$)!``````QT0D!$`^0`")!"3H*O?__X7`B<-T1J%0`$$`B00DZ"?W
M__^-5"0XN/Z`0`#H0=?__X7`=#BA4`!!`(D$).@`]___Z"/W__^+="0\,S6H
M(D$`B=AU4(UE^%M>7<.A4`!!`(D$).C)]O__B</KJ:%,`$$`A<!T"HD$).CT
M]?__Z[6-5"0XN*R`0`#HY-;__X7`HTP`00!TGNO=QP0D`0```.C9]___Z&CW
M__^0D)"0D)"0D%6)Y5WI=[#__Y"0D)"0D)#_____$&!```````#_____````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````/_____$@T``````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````&-Y9V=C8U]S+3$N9&QL`%]?
M<F5G:7-T97)?9G)A;65?:6YF;P!?7V1E<F5G:7-T97)?9G)A;65?:6YF;P!C
M>6=G8VHM,34N9&QL`%]*=E]296=I<W1E<D-L87-S97,``"X`+BX`)7,O)7,`
M`"]P<F]C+W-E;&8O97AE`$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!)
M;G1E<FYA;',Z.E!!4CHZ0D]/5`!005)?5$5-4`!005)?4%)/1TY!344`.@`O
M`"5S)7,E<P!015),-4Q)0@!015),3$E"`%!%4DPU3U!4`%!%4DQ)3P!005)?
M24Y)5$E!3$E:140`4$%27U-005=.140`4$%27T-,14%.`%!!4E]$14)51P!0
M05)?0T%#2$4`4$%27T%21T,`4$%27T%21U9?,`!005)?1TQ/0D%,7T1%0E5'
M`%!!4E]'3$]"04Q?5$U01$E2`%!!4E]435!$25(`4$%27T=,3T)!3%]414U0
M`%!!4E]'3$]"04Q?0TQ%04X`4$%27T-,14%./0`Q`%]?14Y67U!!4E]#3$5!
M3E]?("`@("`@("`@("`@("`@``!,24)0051(`$Q$7TQ)0E)!4EE?4$%42`!,
M24)205)97U!!5$@`4$%42`!$64Q$7TQ)0E)!4EE?4$%42`!32$Q)0E]0051(
M`%5315(`4UE35$5-`"]T;7``5$U01$E2`%1%35!$25(`5$5-4`!435``55-%
M4DY!344`)3`R>`!P87(M`"5S)7,E<R5S````)7,Z(&-R96%T:6]N(&]F('!R
M:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@!P87)L
M+F5X90`N<&%R`"5S)7-C86-H92TE<R5S`"5S)7-T96UP+25U)7,`)7,E<W1E
M;7`M)74M)74E<P``)7,Z('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N
M<V%F92`H<&QE87-E(')E;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I
M;VXI"@!!4D=6`!@`<&5R;``P`````%5N86)L92!T;R!C<F5A=&4@8V%C:&4@
M9&ER96-T;W)Y`````"5S.B!E>&5C=71I;VX@;V8@)7,@9F%I;&5D("AE<G)N
M;STE:2D*```E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E
M8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*`"UE`"TM`'!E<FQX<VDN8P``
M``!E=F%L("=E>&5C("]U<W(O8FEN+W!E<FP@("U3("0P("1[,2LB)$`B?2<*
M:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E<B!S;VUE('-H96QL"G!A8VMA9V4@
M7U]P87)?<&P["FUY("@D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I;&4I
M.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]*2!["G)E<75I<F4@1FEL93HZ
M5&5M<#L*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*<F5Q=6ER92!&:6QE.CI3
M<&5C.PIM>2`D=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'!A
M<E]T96UP*3L*;W5T<RAQ<7M296UO=FEN9R!F:6QE<R!I;B`B)'!A<E]T96UP
M(GTI.PI&:6QE.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@*"`M9"`I(#\@<FUD
M:7(@.B!U;FQI;FL@?2P@)'!A<E]T96UP*3L*<FUD:7(@)'!A<E]T96UP.PII
M9B`H+60@)'!A<E]T96UP("8F("1>3R!N92`G35-7:6XS,B<I('L*;7D@)'1M
M<"`](&YE=R!&:6QE.CI496UP*`I414U03$%412`]/B`G=&UP6%A86%@G+`I$
M25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'1O<&1I<BDL"E-51D9)
M6"`]/B`G+F-M9"<L"E5.3$E.2R`]/B`P+`HI.PIP<FEN="`D=&UP("(C(2]B
M:6XO<V@*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D
M<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A
M:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G(B`N("1T
M;7`M/F9I;&5N86UE("X@(B<*(CL*8VAM;V0@,#<P,"PD=&UP+3YF:6QE;F%M
M93L*;7D@)&-M9"`]("1T;7`M/F9I;&5N86UE("X@)R`^+V1E=B]N=6QL(#(^
M)C$@)B<["F-L;W-E("1T;7`["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<2A3<&%W
M;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("D*
M+B`D=&UP+3YF:6QE;F%M92D["GT*?2!]"D)%1TE.('L*26YT97)N86QS.CI0
M05(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E<FYA;',Z.E!!4CHZ0D]/5#L*
M979A;"!["E]P87)?:6YI=%]E;G8H*3L*:68@*&5X:7-T<R`D14Y6>U!!4E]!
M4D=67S!](&%N9"`D14Y6>U!!4E]!4D=67S!]("D@>PI`05)'5B`](&UA<"`D
M14Y6>R)005)?05)'5E\D7R)]+"`H,2`N+B`D14Y6>U!!4E]!4D=#?2`M(#$I
M.PHD,"`]("1%3E9[4$%27T%21U9?,'T["GT*96QS92!["F9O<B`H:V5Y<R`E
M14Y6*2!["F1E;&5T92`D14Y6>R1??2!I9B`O7E!!4E]!4D=67R\["GT*?0IM
M>2`D<75I970@/2`A)$5.5GM005)?1$5"54=].PIM>2`E0V]N9FEG(#T@*`IP
M871H7W-E<"`@("`]/B`H)%Y/(#U^("]>35-7:6XO(#\@)SLG(#H@)SHG*2P*
M7V5X92`@("`@("`@/3X@*"1>3R`]?B`O7B@_.DU35VEN?$]3,GQC>6=W:6XI
M+R`_("<N97AE)R`Z("<G*2P*7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O7DU3
M5VEN?$]3,B\@/R`G7%PG(#H@)R\G*2P**3L*7W-E=%]P<F]G;F%M92@I.PI?
M<V5T7W!A<E]T96UP*"D["FUY("@D<W1A<G1?<&]S+"`D9&%T85]P;W,I.PI[
M"FQO8V%L("1324=[7U]705).7U]](#T@<W5B('M].PIO<&5N(%]&2"P@)SPG
M+"`D<')O9VYA;64@;W(@;&%S=#L*8FEN;6]D92A?1D@I.PIM>2`D8G5F.PIS
M965K(%]&2"P@+3@L(#(["G)E860@7T9(+"`D8G5F+"`X.PIL87-T('5N;&5S
M<R`D8G5F(&5Q(")<;E!!4BYP;5QN(CL*<V5E:R!?1D@L("TQ,BP@,CL*<F5A
M9"!?1D@L("1B=68L(#0["G-E96L@7T9(+"`M,3(@+2!U;G!A8VLH(DXB+"`D
M8G5F*2P@,CL*<F5A9"!?1D@L("1B=68L(#0["B1D871A7W!O<R`]("AT96QL
M(%]&2"D@+2`T.PIM>2`E<F5Q=6ER95]L:7-T.PIW:&EL92`H)&)U9B!E<2`B
M1DE,12(I('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U
M;G!A8VLH(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*;W5T<RAQ
M<2A5;G!A8VMI;F<@9FEL92`B)&9U;&QN86UE(BXN+BDI.PIM>2`D8W)C(#T@
M*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z('5N
M9&5F.PIM>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\*#\Z+BHO
M*3\H+BHI*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@
M)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E>'0I(&%N
M9"`D97AT("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`H)&]U="P@
M)&9I;&5N86UE*2`](%]T96UP9FEL92@D97AT+"`D8W)C*3L*:68@*"1O=70I
M('L*8FEN;6]D92@D;W5T*3L*<')I;G0@)&]U="`D8G5F.PIC;&]S92`D;W5T
M.PIC:&UO9"`P-S4U+"`D9FEL96YA;64["GT*)%!!4CHZ2&5A=GDZ.D9U;&Q#
M86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE.PHD4$%2.CI(96%V>3HZ1G5L
M;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA;64["GT*96QS:68@*"`D9G5L
M;&YA;64@/7X@;7Q>+S]S:&QI8B]\(&%N9"!D969I;F5D("1%3E9[4$%27U1%
M35!]("D@>PIM>2`D9FEL96YA;64@/2`B)$5.5GM005)?5$5-4'TO)&)A<V5N
M86UE)&5X="(["F]U=',H(E-(3$E".B`D9FEL96YA;65<;B(I.PIO<&5N(&UY
M("1O=70L("<^)RP@)&9I;&5N86UE(&]R(&1I92`D(3L*8FEN;6]D92@D;W5T
M*3L*<')I;G0@)&]U="`D8G5F.PIC;&]S92`D;W5T.PI]"F5L<V4@>PHD<F5Q
M=6ER95]L:7-T>R1F=6QL;F%M97T@/0HD4$%2.CI(96%V>3HZ36]D=6QE0V%C
M:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN
M86UE(#T^("1F=6QL;F%M92P*?3L*?0IR96%D(%]&2"P@)&)U9BP@-#L*?0IL
M;V-A;"!`24Y#(#T@*'-U8B!["FUY("@D<V5L9BP@)&UO9'5L92D@/2!`7SL*
M<F5T=7)N(&EF(')E9B`D;6]D=6QE(&]R("$D;6]D=6QE.PIM>2`D9FEL96YA
M;64@/2!D96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?2!\?"!D;R!["FUY
M("1K97D["F9O<F5A8V@@*&ME>7,@)7)E<75I<F5?;&ES="D@>PIN97AT('5N
M;&5S<R`O7%$D;6]D=6QE7$4D+SL*)&ME>2`]("1?.R!L87-T.PI]"F1E;&5T
M92`D<F5Q=6ER95]L:7-T>R1K97E](&EF(&1E9FEN960H)&ME>2D["GT@;W(@
M<F5T=7)N.PHD24Y#>R1M;V1U;&5](#T@(B]L;V%D97(O)&9I;&5N86UE+R1M
M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ
M1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@
M;W(@9&EE("0A.PIB:6YM;V1E*"1F:"D["G!R:6YT("1F:"`D9FEL96YA;64M
M/GMB=69].PIS965K*"1F:"P@,"P@,"D["G)E='5R;B`D9F@["GT*96QS92![
M"FUY("@D;W5T+"`D;F%M92D@/2!?=&5M<&9I;&4H)RYP;2<L("1F:6QE;F%M
M92T^>V-R8WTI.PII9B`H)&]U="D@>PIB:6YM;V1E*"1O=70I.PIP<FEN="`D
M;W5T("1F:6QE;F%M92T^>V)U9GT["F-L;W-E("1O=70["GT*;W!E;B!M>2`D
M9F@L("<\)RP@)&YA;64@;W(@9&EE("0A.PIB:6YM;V1E*"1F:"D["G)E='5R
M;B`D9F@["GT*9&EE(")";V]T<W1R87!P:6YG(&9A:6QE9#H@8V%N;F]T(&9I
M;F0@)&UO9'5L92%<;B(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR
M97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I
M<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A
M9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@
M/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.
M0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H
M)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64[
M"GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S
M:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[
M)&9I;&5N86UE?3L*?0IL87-T('5N;&5S<R`D8G5F(&5Q(")02UPP,#-<,#`T
M(CL*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.PI]"FUY($!P87)?87)G
M<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID
M96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@
M;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*)'%U:65T(#T@,"!U;FQE<W,@
M)$5.5GM005)?1$5"54=].PII9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=
M(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD
M(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N
M:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF
M("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!
M4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS
M92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW
M:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4
M=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?
M=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D
M,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@
M)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@
M*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E
M<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I
M('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO
M9V9H+"`B/CXB+"`D,B!O<B!D:64@(EA86#H@0V%N;F]T(&]P96X@;&]G.B`D
M(2(["GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]
M"G-H:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2![
M"F1E;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U:7)E
M(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%2
M1U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X:70[
M"GT*?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I
M('L*>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*
M?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE
M9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G/"<L("1P87(@;W(@9&EE(")#86YN
M;W0@9FEN9"`G)'!A<B<Z("0A(CL*8FEN;6]D92@D9F@I.PIB;&5S<R@D9F@L
M("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II
M<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ
M6FEP.CI!6E]/2R@I("D*;W(@9&EE(")296%D("<D<&%R)R!E<G)O<CH@)"$B
M.PI]"FUY("5E;G8@/2!D;R!["FEF("@D>FEP(&%N9"!M>2`D;65T82`]("1Z
M:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R
M.B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;
M7CI=*RDZ("@N*RDD+VUG.PI]"GT["FQO8V%L("0O(#T@7#0["FEF("AD969I
M;F5D("1P87(I('L*;W!E;B!005(L("<\)RP@)'!A<B!O<B!D:64@(B0A.B`D
M<&%R(CL*8FEN;6]D92A005(I.PID:64@(B1P87(@:7,@;F]T(&$@4$%2(&9I
M;&4B('5N;&5S<R`\4$%2/B!E<2`B4$M<,#`S7#`P-"(["GT*0W)E871E4&%T
M:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&
M:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]74D].3%DH*2!\($E/.CI&
M:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE("0A.PIB:6YM;V1E*"1F
M:"D["B0O(#T@*&1E9FEN960@)&1A=&%?<&]S*2`_(%PD9&%T85]P;W,@.B!U
M;F1E9CL*<V5E:R!?1D@L(#`L(#`["FUY("1L;V%D97(@/2!S8V%L87(@/%]&
M2#X["FEF("@A)$5.5GM005)?5D520D%424U](&%N9"`D;&]A9&5R(#U^("]>
M*#\Z(R%\7$!R96TI+RD@>PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I
M<#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YN97<M/F%P<&QY*%PD;&]A9&5R
M+"`D,"D*?0IF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E96YV*2!["FUY
M("1V86P@/2`D96YV>R1K97E](&]R(&YE>'0["B1V86P@/2!E=F%L("1V86P@
M:68@)'9A;"`]?B`O7ELG(ETO.PIM>2`D;6%G:6,@/2`B7U]%3E9?4$%27R(@
M+B!U8R@D:V5Y*2`N(")?7R(["FUY("1S970@/2`B4$%27R(@+B!U8R@D:V5Y
M*2`N("(])'9A;"(["B1L;V%D97(@/7X@<WLD;6%G:6,H("LI?7L*)&UA9VEC
M("X@)'-E="`N("@G("<@>"`H;&5N9W1H*"0Q*2`M(&QE;F=T:"@D<V5T*2DI
M"GUE9SL*?0HD9F@M/G!R:6YT*"1L;V%D97(I.PHD+R`]('5N9&5F.PII9B`H
M)&)U;F1L92D@>PIR97%U:7)E(%!!4CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I
M;FET7V1Y;F%L;V%D97(H*3L*:6YI=%]I;F,H*3L*<F5Q=6ER95]M;V1U;&5S
M*"D["FUY($!I;F,@/2!S;W)T('L*;&5N9W1H*"1B*2`\/3X@;&5N9W1H*"1A
M*0I](&=R97`@>PHA+T)31%!!3B\*?2!G<F5P('L**"1B=6YD;&4@;F4@)W-I
M=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N
M9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,[
M"FUY("5F:6QE<SL*+UY?/"@N*RDD+R!A;F0@+64@)#$@86YD("1F:6QE<WLD
M,7TK*R!F;W(@:V5Y<R`E.CH["B1F:6QE<WLD7WTK*R!F;W(@=F%L=65S("5)
M3D,["FUY("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG>VQI8E]E>'1].PIM
M>2`E=W)I='1E;CL*9F]R96%C:"`H<V]R="!K97ES("5F:6QE<RD@>PIM>2`H
M)&YA;64L("1F:6QE*3L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["FEF("@D
M;F%M92`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)%]]*2!["B1F:6QE(#T@
M)%\["FQA<W0["GT*96QS:68@*"]>*%Q1)&1I<EQ%7"\H+BI;7D-C72DI7%HO
M:2D@>PHH)&9I;&4L("1N86UE*2`]("@D,2P@)#(I.PIL87-T.PI]"F5L<VEF
M("AM(5XO;&]A9&5R+UM>+UTK+R@N*EM>0V-=*5Q:(2D@>PII9B`H;7D@)')E
M9B`]("1005(Z.DAE879Y.CI-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D
M;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"F5L<VEF("@M9B`B)&1I<B\D
M,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0[
M"GT*?0I]"FYE>'0@=6YL97-S(&1E9FEN960@)&YA;64@86YD(&YO="`D=W)I
M='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@
M/7X@+UPN7%$D;&EB7V5X=%Q%)"\["F]U=',H(&IO:6X@(B(L"G%Q*%!A8VMI
M;F<@(BDL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@.B`D9FEL92P*<7$H
M(BXN+BD**3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT
M96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIO<&5N($9)3$4L("<\)RP@
M)&9I;&4@;W(@9&EE(")#86XG="!O<&5N("1F:6QE.B`D(2(["F)I;FUO9&4H
M1DE,12D["B1C;VYT96YT(#T@/$9)3$4^.PIC;&]S92!&24Q%.PI005(Z.D9I
M;'1E<CHZ4&]D4W1R:7`M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D9FEL92D*
M:68@(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z<&U\
M:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P
M<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*?0IO=71S*'%Q*%=R:71T
M96X@87,@(B1N86UE(BDI.PHD9F@M/G!R:6YT*")&24Q%(BD["B1F:"T^<')I
M;G0H<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2D["B1F:"T^<')I;G0H
M<W!R:6YT9B@*(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S
M,B@D8V]N=&5N="DL("1N86UE"BDI.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"!L
M96YG=&@H)&-O;G1E;G0I*2D["B1F:"T^<')I;G0H)&-O;G1E;G0I.PI]"GT*
M)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2!I9B`D>FEP.PHD8V%C:&5?
M;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@-#`["FEF("@A)&-A8VAE
M7VYA;64@86YD(&UY("1M=&EM92`]("AS=&%T*"1O=70I*5LY72D@>PIM>2`D
M8W1X(#T@979A;"![(')E<75I<F4@1&EG97-T.CI32$$[($1I9V5S=#HZ4TA!
M+3YN97<H,2D@?0I\?"!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z.E-(03$[($1I
M9V5S=#HZ4TA!,2T^;F5W('T*?'P@979A;"![(')E<75I<F4@1&EG97-T.CI-
M1#4[($1I9V5S=#HZ340U+3YN97<@?3L*;7D@)'-H85]V97)S:6]N(#T@979A
M;"![("1$:6=E<W0Z.E-(03HZ5D524TE/3B!]('Q\(#`["FEF("@D<VAA7W9E
M<G-I;VX@97$@)S4N,S@G(&]R("1S:&%?=F5R<VEO;B!E<2`G-2XS.2<I('L*
M)&-T>"T^861D9FEL92@D;W5T+"`B8B(I(&EF("@D8W1X*3L*?0IE;'-E('L*
M:68@*"1C='@@86YD(&]P96XH;7D@)&9H+"`B/"1O=70B*2D@>PIB:6YM;V1E
M*"1F:"D["B1C='@M/F%D9&9I;&4H)&9H*3L*8VQO<V4H)&9H*3L*?0I]"B1C
M86-H95]N86UE(#T@)&-T>"`_("1C='@M/FAE>&1I9V5S="`Z("1M=&EM93L*
M?0HD8V%C:&5?;F%M92`N/2`B7#`B('@@*#0Q("T@;&5N9W1H("1C86-H95]N
M86UE*3L*)&-A8VAE7VYA;64@+CT@(D-!0TA%(CL*)&9H+3YP<FEN="@D8V%C
M:&5?;F%M92D["B1F:"T^<')I;G0H<&%C:R@G3B<L("1F:"T^=&5L;"`M(&QE
M;F=T:"@D;&]A9&5R*2DI.PHD9F@M/G!R:6YT*")<;E!!4BYP;5QN(BD["B1F
M:"T^8VQO<V4["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N
M;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q
M=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR
M97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY
M("1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL*;7D@)&9H(#T@24\Z.D9I;&4M
M/FYE=SL*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE("(D
M(3H@)$`B.PHD>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA
M;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2!O<B!D:64@(B0A.B`D0"([
M"G!U<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`["B1005(Z.DQI8D-A8VAE>R1P
M<F]G;F%M97T@/2`D>FEP.PHD<75I970@/2`A)$5.5GM005)?1$5"54=].PIO
M=71S*'%Q*%PD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!4E]414U0?2(I*3L*
M:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI('L@(R!S:&]U;&0@8F4@<V5T
M(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z:7`M/FUE
M;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^:7-$:7)E8W1O<GD["FUY("1M
M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.86UE.PIN97AT('5N;&5S<R`D
M;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB+PHH/SHD0V]N9FEG.CI#;VYF
M:6=[=F5R<VEO;GTO*3\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I
M/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E
M<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL
M("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]
M("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q*%-K:7!P
M:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T
M("(D9&5S=%]N86UE(BDI.PI](&5L<V4@>PIO=71S*'%Q*$5X=')A8W1I;F<@
M(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B*2D["B1M96UB97(M/F5X
M=')A8W14;T9I;&5.86UE9"@D9&5S=%]N86UE*3L*8VAM;V0H,#4U-2P@)&1E
M<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*?0IU;FQE<W,@*"10
M05(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\("(N(B!U;FQE<W,@
M0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-
M;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L(%T*)#`@6R`M0GPM
M8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!4E]04D]'3D%-17T@
M/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!4D=6*3L*?0IS=6(@0W)E871E
M4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&:6QE.CI"87-E;F%M
M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@1FEL93HZ0F%S96YA
M;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR97%U:7)E($9I;&4Z
M.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@=6YL97-S*"UE("1P
M871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*<W5B(')E<75I<F5?
M;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U:7)E($1Y;F%,;V%D97(["G)E
M<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I8W0["G)E<75I<F4@=V%R;FEN
M9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR97%U:7)E($-A<G`Z
M.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U:7)E($5X<&]R=&5R.CI(96%V
M>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT;#L*<F5Q=6ER92!&
M:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P96,["G)E<75I<F4@6%-,;V%D
M97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(86YD;&4["G)E<75I
M<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL:6(["G)E<75I<F4@
M07)C:&EV93HZ6FEP.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y
M.PIR97%U:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D
M4W1R:7`["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q
M=6ER92!A='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*979A;"![
M(')E<75I<F4@5VEN,S(@?3L*979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL
M('T["F5V86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE
M=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!].PIE=F%L
M('L@<F5Q=6ER92!097)L24\[(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*
M?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[4$%27U1%
M35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP
M(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA<"`D14Y6
M>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@
M*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE<W,@9&5F
M:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM>2`D=7-E
M<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D
M/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I
M;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE
M;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W=6ED.PI]
M"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U53
M15)]('Q\("=365-414TG.PI]"B1U<V5R;F%M92`]?B!S+UQ7+U\O9SL*;7D@
M)'-T;7!D:7(@/2`B)'!A=&@D0V]N9FEG>U]D96QI;7UP87(M(BYU;G!A8VLH
M(D@J(BP@)'5S97)N86UE*3L*;6MD:7(@)'-T;7!D:7(L(#`W-34["FEF("@A
M)$5.5GM005)?0TQ%04Y](&%N9"!M>2`D;71I;64@/2`H<W1A="@D<')O9VYA
M;64I*5LY72D@>PIO<&5N("AM>2`D9F@L("(\(BX@)'!R;V=N86UE*3L*<V5E
M:R`D9F@L("TQ."P@,CL*<WES<F5A9"`D9F@L(&UY("1B=68L(#8["FEF("@D
M8G5F(&5Q(")<,$-!0TA%(BD@>PIS965K("1F:"P@+34X+"`R.PIS>7-R96%D
M("1F:"P@)&)U9BP@-#$["B1B=68@/7X@<R]<,"\O9SL*)'-T;7!D:7(@+CT@
M(B1#;VYF:6=[7V1E;&EM?6-A8VAE+2(@+B`D8G5F.PI]"F5L<V4@>PIM>2`D
M8W1X(#T@979A;"![(')E<75I<F4@1&EG97-T.CI32$$[($1I9V5S=#HZ4TA!
M+3YN97<H,2D@?0I\?"!E=F%L('L@<F5Q=6ER92!$:6=E<W0Z.E-(03$[($1I
M9V5S=#HZ4TA!,2T^;F5W('T*?'P@979A;"![(')E<75I<F4@1&EG97-T.CI-
M1#4[($1I9V5S=#HZ340U+3YN97<@?3L*;7D@)'-H85]V97)S:6]N(#T@979A
M;"![("1$:6=E<W0Z.E-(03HZ5D524TE/3B!]('Q\(#`["FEF("@D<VAA7W9E
M<G-I;VX@97$@)S4N,S@G(&]R("1S:&%?=F5R<VEO;B!E<2`G-2XS.2<I('L*
M)&-T>"T^861D9FEL92@D<')O9VYA;64L(")B(BD@:68@*"1C='@I.PI]"F5L
M<V4@>PII9B`H)&-T>"!A;F0@;W!E;BAM>2`D9F@L("(\)'!R;V=N86UE(BDI
M('L*8FEN;6]D92@D9F@I.PHD8W1X+3YA9&1F:6QE*"1F:"D["F-L;W-E*"1F
M:"D["GT*?0HD<W1M<&1I<B`N/2`B)$-O;F9I9WM?9&5L:6U]8V%C:&4M(B`N
M("@@)&-T>"`_("1C='@M/FAE>&1I9V5S="`Z("1M=&EM92`I.PI]"F-L;W-E
M*"1F:"D["GT*96QS92!["B1%3E9[4$%27T-,14%.?2`](#$["B1S=&UP9&ER
M("X]("(D0V]N9FEG>U]D96QI;7UT96UP+20D(CL*?0HD14Y6>U!!4E]414U0
M?2`]("1S=&UP9&ER.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*;&%S=#L*?0HD
M<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?
M5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?=&5M<&9I;&4@>PIM>2`H)&5X="P@
M)&-R8RD@/2!`7SL*;7D@*"1F:"P@)&9I;&5N86UE*3L*)&9I;&5N86UE(#T@
M(B1P87)?=&5M<"\D8W)C)&5X="(["FEF("@D14Y6>U!!4E]#3$5!3GTI('L*
M=6YL:6YK("1F:6QE;F%M92!I9B`M92`D9FEL96YA;64["G!U<V@@0'1M<&9I
M;&4L("1F:6QE;F%M93L*?0IE;'-E('L*<F5T=7)N("AU;F1E9BP@)&9I;&5N
M86UE*2!I9B`H+7(@)&9I;&5N86UE*3L*?0IO<&5N("1F:"P@)SXG+"`D9FEL
M96YA;64@;W(@9&EE("0A.PIB:6YM;V1E*"1F:"D["G)E='5R;B@D9F@L("1F
M:6QE;F%M92D["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O
M*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*
M:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[
M4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA
M;64L(')I;F1E>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@*R`Q*3L*
M?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L
M("1#;VYF:6=[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\
M)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D
M<')O9VYA;64D0V]N9FEG>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)$-O;F9I
M9WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O
M7%$D0V]N9FEG>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF
M(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]4
M14U0?3L*)&1I<B`]?B!S+UQ1)$-O;F9I9WM?9&5L:6U]7$4D+R\["B@H)'!R
M;V=N86UE(#T@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA;64D0V]N9FEG
M>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R
M;V=N86UE)$-O;F9I9WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D0V]N
M9FEG>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D0V]N
M9FEG>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@
M>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H
M:6YD97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI(#P@,"D@>PHD<')O
M9VYA;64@/2`B+B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W
M9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@
M*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z
M(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN
M/UQ1)$-O;F9I9WM?9&5L:6U]7$4I+R1P=V0D0V]N9FEG>U]D96QI;7TO.PHD
M14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI
M=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2![
M"G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*
M?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/
M1TY!344@05)'0R!!4D=67S`@*2`I('L*9&5L971E("1%3E9[)U!!4E\G+B1?
M?3L*?0IF;W(@*'%W+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B1%
M3E9[)U!!4E\G+B1??2`]("1%3E9[)U!!4E]'3$]"04Q?)RXD7WT@:68@97AI
M<W1S("1%3E9[)U!!4E]'3$]"04Q?)RXD7WT["GT*;7D@)'!A<E]C;&5A;B`]
M(")?7T5.5E]005)?0TQ%04Y?7R`@("`@("`@("`@("`@("(["FEF("@D14Y6
M>U!!4E]414U0?2D@>PID96QE=&4@)$5.5GM005)?0TQ%04Y].PI]"F5L<VEF
M("@A97AI<W1S("1%3E9[4$%27T=,3T)!3%]#3$5!3GTI('L*;7D@)'9A;'5E
M(#T@<W5B<W1R*"1P87)?8VQE86XL(#$R("L@;&5N9W1H*")#3$5!3B(I*3L*
M)$5.5GM005)?0TQ%04Y](#T@)#$@:68@)'9A;'5E(#U^("]>4$%27T-,14%.
M/2A<4RLI+SL*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@
M*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN
M="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*
M<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[
M)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I
M=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K
M86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF
M:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q*'!A<BYP
M;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C
M:"!F:6QE(&]R(&1I<F5C=&]R>5QN*0IU;FQE<W,@+64@)'!R;V=N86UE.PID
M;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58
M251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]
M("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^
M+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ
M7U]%4E)/4CL*,3L*7U]%3D1?7PH`H%T``$=#0SH@*$=.52D@-"XX+C,@,C`Q
M-#`U,C(@*$9E9&]R82!#>6=W:6X@-"XX+C,M-BD```!'0T,Z("A'3E4I(#0N
M.2XR`````$=#0SH@*$=.52D@-"XY+C(`````1T-#.B`H1TY5*2`T+C@N,R`R
M,#$T,#4R,B`H1F5D;W)A($-Y9W=I;B`T+C@N,RTV*0```$=#0SH@*$=.52D@
M-"XX+C,@,C`Q-#`U,C(@*$9E9&]R82!#>6=W:6X@-"XX+C,M-BD```!'0T,Z
M("A'3E4I(#0N."XS(#(P,30P-3(R("A&961O<F$@0WEG=VEN(#0N."XS+38I
M````1T-#.B`H1TY5*2`T+C@N,R`R,#$T,#4R,B`H1F5D;W)A($-Y9W=I;B`T
M+C@N,RTV*0```$=#0SH@*$=.52D@-"XX+C,@,C`Q-#`U,C(@*$9E9&]R82!#
M>6=W:6X@-"XX+C,M-BD```!'0T,Z("A'3E4I(#0N."XS(#(P,30P-3(R("A&
M961O<F$@0WEG=VEN(#0N."XS+38I````1T-#.B`H1TY5*2`T+C@N,R`R,#$T
M,#4R,B`H1F5D;W)A($-Y9W=I;B`T+C@N,RTV*0```$=#0SH@*$=.52D@-"XX
M+C,@,C`Q-#`U,C(@*$9E9&]R82!#>6=W:6X@-"XX+C,M-BD```!'0T,Z("A'
M3E4I(#0N."XS(#(P,30P-3(R("A&961O<F$@0WEG=VEN(#0N."XS+38I````
M1T-#.B`H1TY5*2`T+C@N,R`R,#$T,#4R,B`H1F5D;W)A($-Y9W=I;B`T+C@N
M,RTV*0```$=#0SH@*$=.52D@-"XY+C(`````1T-#.B`H1TY5*2`T+CDN,@``
M``````````````$```"H(@$`N1$``"````"H(@$`L#8``"````"H(@$`AS<`
M`"````"H(@$`UC<``"````"H(@$`_#<``"````"H(@$`YS@``"````"H(@$`
M[3D``"````"H(@$`#3H``"````"H(@$`4#T``"````"H(@$`M#T``"````"H
M(@$`TSX``"````"H(@$`%S\``"````"H(@$`=D$``"````"H(@$`Q4(``"``
M``"H(@$`)$,``"````"H(@$`/D,``"````"H(@$`Y4,``"````"H(@$`:40`
M`"````"H(@$`%$4``"````"H(@$`CT<``"````"H(@$`J%$``"````"H(@$`
M"%0``"````"H(@$`/UX``"````"H(@$`JU\``"````"T(@$`)CL``"````"T
M(@$`/#L``"````"T(@$`33L``"````"T(@$`;3L``"````"T(@$`@SL``"``
M``"T(@$`JSL``"````"T(@$`P3L``"````"T(@$`USL``"````"T(@$`[3L`
M`"````"T(@$`!CP``"````"T(@$`'#P``"````"T(@$`/#P``"````"T(@$`
M63P``"````"T(@$`;SP``"````"T(@$`ESP``"````"T(@$`P3P``"````"T
M(@$`X#P``"````"T(@$`BST``"````"T(@$`T3T``"````"T(@$`!#X``"``
M``"T(@$`1#X``"````"T(@$`63X``"````"T(@$`XS\``"````"T(@$`!4``
M`"````"T(@$`-$<``"````"T(@$`\4<``"````"T(@$`%$@``"````"T(@$`
M-$@``"````"T(@$`LU$``"````"T(@$`Y%$``"````"T(@$`8U(``"````"T
M(@$`G5(``"````"T(@$`X5(``"````"T(@$`&5,``"````"T(@$`25,``"``
M``"T(@$`F5,``"````"T(@$`ME,``"````"T(@$`,50``"````"T(@$`_E0`
M`"````"T(@$`&54``"````"T(@$`1%4``"````"T(@$`<E4``"````"T(@$`
M1%8``"````"T(@$`858``"````"T(@$`I%X``"````"T(@$`N%X``"````"T
M(@$`S%X``"````"P(@$`@EX``"``````````````````````````````````
M````````````````````````````````````````````````````````````
M```````"````&0```!S0```<G@``4E-$4YM*I_L`R+&1Y=0?+9VGZD,!````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````!0``````````7I2``%\"`$;#`0$B`$``!P````<````X"__
M_X<`````00X(A0)"#04"@\4,!`0`%``````````!>E(``7P(`1L,!`2(`0``
M+````!P````X,/__Y@````!!#@B%`D(-!4:'`X8$@P4"Q0K#0<9!QT'%#`0$
M00L`'````$P```#X,/__&P````!*#@B%`D(-!4W%#`0$```4``````````%Z
M4@`!?`@!&PP$!(@!``!`````'````.`P__\F)0```$$."(4"00X,AP-##A"&
M!$$.%(,%1@Z@!@,0)0H.%$'##A!!Q@X,0<<."$'%#@1!"P```%````!@````
MS%7__Z(`````20X(A0)!#@R'`T$.$(8$00X4@P5##D`":`H.%$/##A!!Q@X,
M0<<."$'%#@1#"U8.%$7##A!!Q@X,0<<."$'%#@0``"````"T````*%;__V8`
M````00X(@P)%#B`"60H."$'##@1!"SP```#8````=%;__^X!````00X(A0)!
M#@R'`T,.$(8$00X4@P5%#E`"^0H.%$'##A!!Q@X,0<<."$'%#@1("P`\````
M&`$``"18__\^`0```$$."(4"00X,AP-##A"&!$$.%(,%1@ZP`7$*#A1!PPX0
M0<8.#$''#@A!Q0X$00L`*````%@!```D6?__:@$```!!#@B&`D$.#(,#0PX@
M`V(!#@Q!PPX(0<8.!``H````A`$``&A:__^+`````$$."(,"10X@9@H."$/#
M#@1!"P)6#@A$PPX$`$0```"P`0``S%K__]0`````00X(A0)!#@R'`T$.$(8$
M00X4@P5!#AB`!DP.P"`"CPH.%$'##A!!Q@X,0<<."$'%#@1!"P```!0```#X
M`0``9%O__S0`````0PX@<`X$`!0````0`@``C%O__UD`````0PXP`E4.!!``
M```H`@``U%O__P,`````````/````#P"``#06___P`$```!!#@B%`D$.#(<#
M1@X0A@1!#A2#!4<.X"`"9PH.%$'##A!!Q@X,0<<."$'%#@1!"Q````!\`@``
M4%W__R(`````````1````)`"``!L7?__P@````!!#@B&`D$.#(,#0PX@`EP*
M#@Q!PPX(0<8.!$T+?`H.#$'##@A!Q@X$0@M+#@Q!PPX(0<8.!```+````-@"
M``#T7?__J`$```!!#@B'`D$.#(8#0PY``UD!"@X,0<8."$''#@1!"P``&```
M``@#``!T7___:0````!##B!D"@X$20L``!@````D`P``R%___R0`````0PX@
M5@H.!$<+```\````0`,``-Q?__\E`0```$$."(4"00X,AP-&#A"&!$$.%(,%
M2`Z``0*A"@X40<,.$$'&#@Q!QPX(0<4.!$H+0````(`#``#,8/__I0P```!!
M#@B%`D$.#(<#1@X0A@1!#A2#!4<.T(("`Y\""@X40<,.$$'&#@Q!QPX(0<4.
M!$$+```\````Q`,``#AM__\D!````$$."(4"00X,AP-!#A"&!$$.%(,%0PY`
M`VX""@X40<,.$$'&#@Q!QPX(0<4.!$<+-`````0$```H<?__:`````!!#@B&
M`D$.#(,#0PX@`E0*#@Q!PPX(0<8.!$$+0PX,0\,."$'&#@04````/`0``&!Q
M__])`````$,.(`)%#@0H````5`0``"AY___H`0```$$."(4"0@T%2(8#@P0#
MC`$*PT'&0<4,!`1!"Q0``````````7I2``%\"`$;#`0$B`$``"0````<````
MI'/__QP#````00X(A0)"#05$@P,#W`$*Q<,,!`1#"P`4``````````%Z4@`!
M?`@!&PP$!(@!```8````'````(1V__\B`````$$."(4"0@T%````%```````
M```!>E(``7P(`1L,!`2(`0``'````!P```"`=O__*@````!!#@B%`D(-!6;%
M#`0$```\````/````)!V___4`````$$."(4"0@T%;0K%#`0$11,"0`M."L4,
M!`1%$P)`"T8*Q0P$!$,+`H#%#`0$````(````'P````P=___"0````!!#@B%
M`D(-!4'%#`0$11,"````%``````````!>E(``7P(`1L,!`2(`0``$````!P`
M```$=___"``````````4``````````%Z4@`!?`@!&PP$!(@!```0````'```
M`.AV__\"`````````!0``````````7I2``%\"`$;#`0$B`$``!`````<````
MS';__P(`````````%``````````!>E(``7P(`1L,!`2(`0``$````!P```"P
M=O__`@`````````4``````````%Z4@`!?`@!&PP$!(@!```0````'````)1V
M__\"`````````!0``````````7I2``%\"`$;#`0$B`$``!`````<````>';_
M_P(`````````%``````````!>E(``7P(`1L,!`2(`0``(````!P```!\=O__
M,P````!!#@B%`D(-!42#`V?%PPP$!````````!0``````````7I2``%\"`$;
M#`0$B`$``!P````<````?'C__PD`````00X(A0)"#05!Q0P$!```````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````>RC54`````
MW!`!``$````2````$@```"@0`0!P$`$`N!`!`*!1``!`5@``^"4!``@F`0#D
M)0$`D$,``"!>```,<```<$```-!5``"@/@``H$```"!#``"P/@``<$$``/!$
M``#`0P```#X``.00`0#Z$`$`%A$!`"@1`0`X$0$`3Q$!`&H1`0!O$0$`>1$!
M`(81`0"2$0$`HQ$!`*\1`0"]$0$`RA$!`-<1`0#D$0$`]A$!`````0`"``,`
M!``%``8`!P`(``D`"@`+``P`#0`.``\`$``1`'!A<BYE>&4`6%-?26YT97)N
M86QS7U!!4E]"3T]4`%A37TEN=&5R;F%L<U]005)?0TQ%05)35$%#2P!?;FU?
M7U!,7V1O7W5N9'5M<`!?;FU?7U!,7W1H<E]K97D`7VYM7U]?7W-T86-K7V-H
M:U]G=6%R9`!G971?=7-E<FYA;65?9G)O;5]G971P=W5I9`!M86EN`&UY7W!A
M<E]P;`!P87)?8F%S96YA;64`<&%R7V-L96%N=7``<&%R7V-U<G)E;G1?97AE
M8P!P87)?9&ER;F%M90!P87)?96YV7V-L96%N`'!A<E]F:6YD<')O9P!P87)?
M:6YI=%]E;G8`<&%R7VUK=&UP9&ER`'!A<E]S971U<%]L:6)P871H`'AS7VEN
M:70```!D(`$```````````#D*`$`R"$!`$`A`0```````````/@H`0"D(@$`
M3"$!````````````="D!`+`B`0"\(0$```````````",*0$`(",!````````
M````````````````````+",!`#PC`0!((P$`6",!`&0C`0!T(P$`A",!`)`C
M`0"@(P$`K",!`+@C`0#$(P$`T",!`-PC`0#H(P$`\",!`/PC`0`0)`$`)"0!
M`#0D`0!`)`$`2"0!`%0D`0!<)`$`:"0!`'0D`0"`)`$`B"0!`)`D`0"<)`$`
MJ"0!`+0D`0#`)`$`R"0!`-`D`0#<)`$`\"0!``0E`0`<)0$`)"4!`#`E`0`\
M)0$`2"4!`%`E`0!<)0$`9"4!`'`E`0!\)0$`B"4!`)0E`0"@)0$`K"4!`+@E
M`0#$)0$``````-`E`0#D)0$``````/@E`0`()@$`&"8!`"PF`0!$)@$`5"8!
M`&0F`0!\)@$`C"8!`*`F`0"P)@$`P"8!`-`F`0#D)@$`_"8!`!@G`0`L)P$`
M/"<!`%`G`0!D)P$`="<!`(@G`0"8)P$`K"<!`+PG`0#()P$`V"<!``````#D
M)P$`^"<!```````L(P$`/",!`$@C`0!8(P$`9",!`'0C`0"$(P$`D",!`*`C
M`0"L(P$`N",!`,0C`0#0(P$`W",!`.@C`0#P(P$`_",!`!`D`0`D)`$`-"0!
M`$`D`0!()`$`5"0!`%PD`0!H)`$`="0!`(`D`0"()`$`D"0!`)PD`0"H)`$`
MM"0!`,`D`0#()`$`T"0!`-PD`0#P)`$`!"4!`!PE`0`D)0$`,"4!`#PE`0!(
M)0$`4"4!`%PE`0!D)0$`<"4!`'PE`0"()0$`E"4!`*`E`0"L)0$`N"4!`,0E
M`0``````T"4!`.0E`0``````^"4!``@F`0`8)@$`+"8!`$0F`0!4)@$`9"8!
M`'PF`0",)@$`H"8!`+`F`0#`)@$`T"8!`.0F`0#\)@$`&"<!`"PG`0`\)P$`
M4"<!`&0G`0!T)P$`B"<!`)@G`0"L)P$`O"<!`,@G`0#8)P$``````.0G`0#X
M)P$``````!D`7U]C>&%?871E>&ET```G`%]?97)R;F\```!%`%]?9V5T<F5E
M;G0`````30!?7VUA:6X`````R`!?9&QL7V-R=#!`,````&T!7V=E='!W=6ED
M,S(```!T`5]G971U:60S,@"!`5]I;7!U<F5?<'1R````O`%?;'-E96LV-```
MO@%?;'-T870V-```WP%?;W!E;C8T````1@)?<W1A=#8T````K@)A8V-E<W,`
M````]`)C86QL;V,`````(P-C;&]S90`D`V-L;W-E9&ER``!=`V-Y9W=I;E]D
M971A8VA?9&QL`%\#8WEG=VEN7VEN=&5R;F%L````=`-D;&Q?9&QL8W)T,```
M`*@#97AE8W9P`````*H#97AI=```!@1F<')I;G1F````$@1F<F5E``!U!&=E
M='!I9`````"!!&=E='!W=6ED``".!&=E='5I9``````K!6QS965K`"T%;'-T
M870`,05M86QL;V,`````005M96UC:'(`````0P5M96UC<'D`````105M96UM
M;W9E````205M:V1I<@!\!6]P96X``(`%;W!E;F1I<@```),%<&]S:7A?;65M
M86QI9VX`````N@5P=&AR96%D7V%T9F]R:P````#F!7!T:')E861?9V5T<W!E
M8VEF:6,````S!G)E860``#0&<F5A9&1I<@```#8&<F5A9&QI;FL``#D&<F5A
M;&QO8P```&,&<FUD:7(`[P9S<')I;G1F````^`9S=&%T```!!W-T<F-H<@``
M```#!W-T<F-M<``````'!W-T<F1U<``````.!W-T<FQE;@`````2!W-T<FYC
M;7`````3!W-T<FYC<'D````<!W-T<G-T<@`````A!W-T<G1O:P````!N!W5N
M;&EN:P`````*`%]?<W1A8VM?8VAK7V9A:6P```P`7U]S=&%C:U]C:&M?9W5A
M<F0`#P!03%]D;U]U;F1U;7```%,`4$Q?=&AR7VME>0````!*`5!E<FQ?871F
M;W)K7VQO8VL``$L!4&5R;%]A=&9O<FM?=6YL;V-K`````%0!4&5R;%]A=E]F
M971C:`!<`5!E<FQ?879?<VAI9G0`N@%097)L7V-R;V%K7VYO8V]N=&5X=```
M$`)097)L7V1O=6YW:6YD`'`"4&5R;%]G=E]F971C:'!V````6@-097)L7VUG
M7W-E=````+P#4&5R;%]N97=35G!V``#*`U!E<FQ?;F5W6%,`````S`-097)L
M7VYE=UA37V9L86=S``")!5!E<FQ?<V%F97-Y<VUA;&QO8P````"8!5!E<FQ?
M<V%V95]D97-T<G5C=&]R7W@`````Z`5097)L7W-V7S)P=E]F;&%G<P!9!E!E
M<FQ?<W9?<V5T<'8`809097)L7W-V7W-E='!V;@````",!E!E<FQ?<WES7VEN
M:70S`````(T&4&5R;%]S>7-?=&5R;0`<!V)O;W1?1'EN84QO861E<@```"8'
M<&5R;%]A;&QO8P`````H!W!E<FQ?8V]N<W1R=6-T`````"D'<&5R;%]D97-T
M<G5C=``J!W!E<FQ?9G)E90`K!W!E<FQ?<&%R<V4`````+`=P97)L7W)U;@``
M%0)'971-;V1U;&5(86YD;&5!``!%`D=E=%!R;V-!9&1R97-S```````@`0``
M(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@
M`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!
M```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$`
M`"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!```@`0``
M(`$``"`!```@`0``(`$``"`!```@`0``(`$``"`!`&-Y9W=I;C$N9&QL`!0@
M`0`4(`$`8WEG<W-P+3`N9&QL`````"@@`0`H(`$`*"`!`"@@`0`H(`$`*"`!
M`"@@`0`H(`$`*"`!`"@@`0`H(`$`*"`!`"@@`0`H(`$`*"`!`"@@`0`H(`$`
M*"`!`"@@`0`H(`$`*"`!`"@@`0`H(`$`*"`!`"@@`0`H(`$`*"`!`&-Y9W!E
M<FPU7S(R+F1L;``\(`$`/"`!`$M%4DY%3#,R+F1L;```````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````$``,````P``"`#@```,```(`0````\```@!@````@
M`0"````````````````````$``$```!@``"``@```'@``(`#````D```@`0`
M``"H``"````````````````````!``D$``!0`0`````````````````````!
M``D$``!@`0`````````````````````!``D$``!P`0``````````````````
M```!``D$``"``0```````````````````0```,`!`(#8``"`````````````
M```````!``D$``"0`0`````````````````````!``$````(`0"`````````
M```````````!``D$``"@`0`````````````````````!``$````X`0"`````
M```````````````!``D$``"P`0``T#$!`&@%`````````````#@W`0"H"```
M``````````#@/P$`:`4`````````````2$4!`*@(`````````````/!-`0`^
M```````````````P3@$`6`(`````````````B%`!`#X#``````````````8`
M5P!)`$X`10!8`$4````H````$````"`````!``@````````!````````````
M```!`````0``!@8&``0$!``"`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`
M____`'-S<P"?GY\`6%A8`+*RL@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6
MEI8`CHZ.`$9&1@#+R\L`"PL+`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3
MDP"SL[,`[N[N`/3T]`#CX^,`Y>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U
M`/7U]0")B8D`BHJ*`#0T-`#HZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`
M#`P,`(^/CP"^OKX`;6UM``D)"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$
M1$0`4%!0`#P\/`!D9&0`PL+"`#(R,@`("`@`#0T-``\/#P``````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````%!
M04%!04%!04%!04%!00$!`35`(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!
M`0$!`0$!%0$!)A4!`0\!`0$!`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!
M(`$!`0$!`0$!$`H@(A45+B$N`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0
M$!`0%T4!`0%!$!`0$!`0$!`0$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?
M$!`0$!`0!P$!`0$@$!`0`0\0$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!
M`0$!`0%!!0\!`0$!`0$!`0$!`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``
M\`,``/`#``#@`P``P`$``(`!``"``0``@`,``(`#````!P````<``(X/``#_
M'P``*````"````!``````0`(````````!````````````````0````$`````
M```!`0$``@("``,#`P`$!`0`!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+
M``P,#``-#0T`#@X.``\/#P`0$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`
M%Q<7`!@8&``9&1D`&AH:`!L;&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B
M(B(`(R,C`"0D)``E)24`)B8F`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM
M+0`N+BX`+R\O`#`P,``Q,3$`,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X
M`#DY.0`Z.CH`.SL[`#P\/``]/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`
M1$1$`$5%10!&1D8`1T='`$A(2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/
M3T\`4%!0`%%140!24E(`4U-3`%145`!5554`5E96`%=75P!86%@`65E9`%I:
M6@!;6UL`7%Q<`%U=70!>7EX`7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E
M`&9F9@!G9V<`:&AH`&EI:0!J:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``
M<7%Q`')R<@!S<W,`='1T`'5U=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\
M?'P`?7U]`'Y^?@!_?W\`@("``(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'
MAP"(B(@`B8F)`(J*B@"+BXL`C(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2
M`).3DP"4E)0`E965`):6E@"7EY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`
MGIZ>`)^?GP"@H*``H:&A`**BH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"I
MJ:D`JJJJ`*NKJP"LK*P`K:VM`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2T
MM`"UM;4`MK:V`+>WMP"XN+@`N;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_
M`,#`P`#!P<$`PL+"`,/#PP#$Q,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`
MR\O+`,S,S`#-S<T`SL[.`,_/SP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6
MUM8`U]?7`-C8V`#9V=D`VMK:`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'A
MX0#BXN(`X^/C`.3DY`#EY>4`YN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL
M`.WM[0#N[NX`[^_O`/#P\`#Q\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`
M^/CX`/GY^0#Z^OH`^_O[`/S\_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\
M/#P\/#P\/#P\/#PL#`0$!`0$!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T
M400$!`0$9,SL]/3T_/3\]/3T_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T
M]/3T]/3T]/3T]/3T]/3T[.R\-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L
M[.SLY+PT45%1400$9+SD[.RLC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401D
MO-SD[.3$492L9.2D3#0L7,3,4;34Y-R\-%%145%1462\W.3D[.3D481TY(QL
MI)QLC+QTK-3DW+PT!`0$!`0$9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$
M!`0$!`1DO-SDY.3DY-R,?)RT?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3D
MU)RD7)2,S#R49,R\C%'<U+0T!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4
MI-S4M#0$!`0$!`1<M-3<W.3<W'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2T
MU-S<W.3<=%'T=(3LS&3D?&RDM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/ST
MM/R\9,RLA%&\K#0$!`0$!`1<M,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$
M!`0$!%RLS,QLG/3\_/S\_/S\_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\
M_/S\_/S\_/S\_/SLS*Q1K#0$!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\
M4<2L-`0$!`0$!%RLM$S4_/STM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$
MI%'\_/RTG/3\_/S\_/S\_/244<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\
M_/S\U&RLS+RD-`0$!`0$!%RD9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$
M!`0$7*1LC.3T_/R<4<Q1E.S\_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&L
MS,1TE.3T_.11O,S,Q+R<-`0$!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$
MO)PL!`0$!`0$7)R\Q,3$Q,3$Q,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$
MQ,3$Q,3$Q,3$Q,3$Q,3$Q,3$O+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\
MO+R\O+R\M)PL!`0$!`0$7)RTQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$
M!`14A)2<G)24E)24E)24E)24E)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<
M7%Q<7%Q<7%Q<7%Q<7%0T!`0$!`0$````/P```!\````/````!P````,````!
M`````````#\````_````/P```#\````_````/P```#\````_````/P```#\`
M```_````/P```#\````_````/P```#\````_````/P```#\````_````/P``
M`#\````_````/P```#\H````$````"`````!``@````````!````````````
M```!`````0``!@8&``0$!``"`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`
M____`'-S<P"?GY\`6%A8`+*RL@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6
MEI8`CHZ.`$9&1@#+R\L`"PL+`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3
MDP"SL[,`[N[N`/3T]`#CX^,`Y>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U
M`/7U]0")B8D`BHJ*`#0T-`#HZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`
M#`P,`(^/CP"^OKX`;6UM``D)"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$
M1$0`4%!0`#P\/`!D9&0`PL+"`#(R,@`("`@`#0T-``\/#P``````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````%!
M04%!04%!04%!04%!00$!`35`(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!
M`0$!`0$!%0$!)A4!`0\!`0$!`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!
M(`$!`0$!`0$!$`H@(A45+B$N`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0
M$!`0%T4!`0%!$!`0$!`0$!`0$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?
M$!`0$!`0!P$!`0$@$!`0`0\0$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!
M`0$!`0%!!0\!`0$!`0$!`0$!`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``
M\`,``/`#``#@`P``P`$``(`!``"``0``@`,``(`#````!P````<``(X/``#_
M'P``*````"````!``````0`(````````!````````````````0````$`````
M```!`0$``@("``,#`P`$!`0`!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+
M``P,#``-#0T`#@X.``\/#P`0$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`
M%Q<7`!@8&``9&1D`&AH:`!L;&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B
M(B(`(R,C`"0D)``E)24`)B8F`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM
M+0`N+BX`+R\O`#`P,``Q,3$`,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X
M`#DY.0`Z.CH`.SL[`#P\/``]/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`
M1$1$`$5%10!&1D8`1T='`$A(2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/
M3T\`4%!0`%%140!24E(`4U-3`%145`!5554`5E96`%=75P!86%@`65E9`%I:
M6@!;6UL`7%Q<`%U=70!>7EX`7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E
M`&9F9@!G9V<`:&AH`&EI:0!J:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``
M<7%Q`')R<@!S<W,`='1T`'5U=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\
M?'P`?7U]`'Y^?@!_?W\`@("``(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'
MAP"(B(@`B8F)`(J*B@"+BXL`C(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2
M`).3DP"4E)0`E965`):6E@"7EY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`
MGIZ>`)^?GP"@H*``H:&A`**BH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"I
MJ:D`JJJJ`*NKJP"LK*P`K:VM`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2T
MM`"UM;4`MK:V`+>WMP"XN+@`N;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_
M`,#`P`#!P<$`PL+"`,/#PP#$Q,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`
MR\O+`,S,S`#-S<T`SL[.`,_/SP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6
MUM8`U]?7`-C8V`#9V=D`VMK:`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'A
MX0#BXN(`X^/C`.3DY`#EY>4`YN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL
M`.WM[0#N[NX`[^_O`/#P\`#Q\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`
M^/CX`/GY^0#Z^OH`^_O[`/S\_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\
M/#P\/#P\/#P\/#PL#`0$!`0$!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T
M400$!`0$9,SL]/3T_/3\]/3T_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T
M]/3T]/3T]/3T]/3T]/3T[.R\-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L
M[.SLY+PT45%1400$9+SD[.RLC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401D
MO-SD[.3$492L9.2D3#0L7,3,4;34Y-R\-%%145%1462\W.3D[.3D481TY(QL
MI)QLC+QTK-3DW+PT!`0$!`0$9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$
M!`0$!`1DO-SDY.3DY-R,?)RT?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3D
MU)RD7)2,S#R49,R\C%'<U+0T!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4
MI-S4M#0$!`0$!`1<M-3<W.3<W'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2T
MU-S<W.3<=%'T=(3LS&3D?&RDM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/ST
MM/R\9,RLA%&\K#0$!`0$!`1<M,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$
M!`0$!%RLS,QLG/3\_/S\_/S\_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\
M_/S\_/S\_/S\_/SLS*Q1K#0$!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\
M4<2L-`0$!`0$!%RLM$S4_/STM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$
MI%'\_/RTG/3\_/S\_/S\_/244<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\
M_/S\U&RLS+RD-`0$!`0$!%RD9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$
M!`0$7*1LC.3T_/R<4<Q1E.S\_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&L
MS,1TE.3T_.11O,S,Q+R<-`0$!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$
MO)PL!`0$!`0$7)R\Q,3$Q,3$Q,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$
MQ,3$Q,3$Q,3$Q,3$Q,3$Q,3$O+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\
MO+R\O+R\M)PL!`0$!`0$7)RTQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$
M!`14A)2<G)24E)24E)24E)24E)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<
M7%Q<7%Q<7%Q<7%Q<7%0T!`0$!`0$````/P```!\````/````!P````,````!
M`````````#\````_````/P```#\````_````/P```#\````_````/P```#\`
M```_````/P```#\````_````/P```#\````_````/P```#\````_````/P``
M`#\````_````/P```#\```$`!``0$````0`(`&@%```!`"`@```!``@`J`@`
M``(`$!````$`"`!H!0```P`@(````0`(`*@(```$````6`(T````5@!3`%\`
M5@!%`%(`4P!)`$\`3@!?`$D`3@!&`$\``````+T$[_X```$`````````````
M`````````#\`````````!``$``$```````````````````"X`0```0!3`'0`
M<@!I`&X`9P!&`&D`;`!E`$D`;@!F`&\```"4`0```0`P`#``,``P`#``-`!"
M`#`````D``(``0!#`&\`;0!P`&$`;@!Y`$X`80!M`&4``````"`````L``(`
M`0!&`&D`;`!E`$0`90!S`&,`<@!I`'``=`!I`&\`;@``````(````#``"``!
M`$8`:0!L`&4`5@!E`'(`<P!I`&\`;@``````,``N`#``+@`P`"X`,````"0`
M`@`!`$D`;@!T`&4`<@!N`&$`;`!.`&$`;0!E````(````"@``@`!`$P`90!G
M`&$`;`!#`&\`<`!Y`'(`:0!G`&@`=````"`````L``(``0!,`&4`9P!A`&P`
M5`!R`&$`9`!E`&T`80!R`&L`<P``````(````"P``@`!`$\`<@!I`&<`:0!N
M`&$`;`!&`&D`;`!E`&X`80!M`&4````@````)``"``$`4`!R`&\`9`!U`&,`
M=`!.`&$`;0!E```````@````-``(``$`4`!R`&\`9`!U`&,`=`!6`&4`<@!S
M`&D`;P!N````,``N`#``+@`P`"X`,````$0````!`%8`80!R`$8`:0!L`&4`
M20!N`&8`;P``````)``$````5`!R`&$`;@!S`&P`80!T`&D`;P!N````````
M`+`$/#]X;6P@=F5R<VEO;CTB,2XP(B!E;F-O9&EN9STB551&+3@B('-T86YD
M86QO;F4](GEE<R(_/@T*/&%S<V5M8FQY('AM;&YS/2)U<FXZ<V-H96UA<RUM
M:6-R;W-O9G0M8V]M.F%S;2YV,2(@;6%N:69E<W1697)S:6]N/2(Q+C`B/@T*
M("`@(#QA<W-E;6)L>4ED96YT:71Y('9E<G-I;VX](C`N,"XP+C`B('1Y<&4]
M(G=I;C,R(B!N86UE/2)005(M4&%C:V5R+D%P<&QI8V%T:6]N(B`O/@T*("`@
M(#QD97-C<FEP=&EO;CY005(M4&%C:V5R($%P<&QI8V%T:6]N/"]D97-C<FEP
M=&EO;CX-"B`@("`\9&5P96YD96YC>3X-"B`@("`@("`@/&1E<&5N9&5N=$%S
M<V5M8FQY/@T*("`@("`@("`@("`@/&%S<V5M8FQY261E;G1I='D@='EP93TB
M=VEN,S(B(&YA;64](DUI8W)O<V]F="Y7:6YD;W=S+D-O;6UO;BU#;VYT<F]L
M<R(@=F5R<VEO;CTB-BXP+C`N,"(@<'5B;&EC2V5Y5&]K96X](C8U.35B-C0Q
M-#1C8V8Q9&8B(&QA;F=U86=E/2(J(B!P<F]C97-S;W)!<F-H:71E8W1U<F4]
M(BHB("\^#0H@("`@("`@(#PO9&5P96YD96YT07-S96UB;'D^#0H@("`@/"]D
M97!E;F1E;F-Y/@T*("`@(#QT<G5S=$EN9F\@>&UL;G,](G5R;CIS8VAE;6%S
M+6UI8W)O<V]F="UC;VTZ87-M+G8S(CX-"B`@("`@("`@/'-E8W5R:71Y/@T*
M("`@("`@("`@("`@/')E<75E<W1E9%!R:79I;&5G97,^#0H@("`@("`@("`@
M("`@("`@/')E<75E<W1E9$5X96-U=&EO;DQE=F5L(&QE=F5L/2)A<TEN=F]K
M97(B('5I06-C97-S/2)F86QS92(@+SX-"B`@("`@("`@("`@(#PO<F5Q=65S
M=&5D4')I=FEL96=E<SX-"B`@("`@("`@/"]S96-U<FET>3X-"B`@("`\+W1R
M=7-T26YF;SX-"CPO87-S96UB;'D^#0H`````````````````````````````
M`````````````````````````````````````````````````````````0``
M````2````"!4`0"/!``````````````\/WAM;"!V97)S:6]N/2(Q+C`B(&5N
M8V]D:6YG/2)55$8M."(@<W1A;F1A;&]N93TB>65S(C\^"CQA<W-E;6)L>2!X
M;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IA<VTN=C$B(&UA;FEF
M97-T5F5R<VEO;CTB,2XP(CX*("`\=')U<W1);F9O('AM;&YS/2)U<FXZ<V-H
M96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV,R(^"B`@("`\<V5C=7)I='D^"B`@
M("`@(#QR97%U97-T9610<FEV:6QE9V5S/@H@("`@("`@(#QR97%U97-T961%
M>&5C=71I;VY,979E;"!L979E;#TB87-);G9O:V5R(B\^"B`@("`@(#PO<F5Q
M=65S=&5D4')I=FEL96=E<SX*("`@(#PO<V5C=7)I='D^"B`@/"]T<G5S=$EN
M9F\^"B`@/&-O;7!A=&EB:6QI='D@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O
M<V]F="UC;VTZ8V]M<&%T:6)I;&ET>2YV,2(^"B`@("`\87!P;&EC871I;VX^
M"B`@("`@(#PA+2U4:&4@240@8F5L;W<@:6YD:6-A=&5S(&%P<&QI8V%T:6]N
M('-U<'!O<G0@9F]R(%=I;F1O=W,@5FES=&$@+2T^"B`@("`@(#QS=7!P;W)T
M961/4R!)9#TB>V4R,#$Q-#4W+3$U-#8M-#-C-2UA-69E+3`P.&1E964S9#-F
M,'TB+SX*("`@("`@/"$M+51H92!)1"!B96QO=R!I;F1I8V%T97,@87!P;&EC
M871I;VX@<W5P<&]R="!F;W(@5VEN9&]W<R`W("TM/@H@("`@("`\<W5P<&]R
M=&5D3U,@260](GLS-3$S.&(Y82TU9#DV+31F8F0M.&4R9"UA,C0T,#(R-68Y
M,V%](B\^"B`@("`@(#PA+2U4:&4@240@8F5L;W<@:6YD:6-A=&5S(&%P<&QI
M8V%T:6]N('-U<'!O<G0@9F]R(%=I;F1O=W,@."`M+3X*("`@("`@/'-U<'!O
M<G1E9$]3($ED/2)[-&$R9C(X93,M-3-B.2TT-#0Q+6)A.6,M9#8Y9#1A-&$V
M93,X?2(O/@H@("`@("`\(2TM5&AE($E$(&)E;&]W(&EN9&EC871E<R!A<'!L
M:6-A=&EO;B!S=7!P;W)T(&9O<B!7:6YD;W=S(#@N,2`M+3X*("`@("`@/'-U
M<'!O<G1E9$]3($ED/2)[,68V-S9C-S8M.#!E,2TT,C,Y+3DU8F(M.#-D,&8V
M9#!D83<X?2(O/B`*("`@("`@/"$M+51H92!)1"!B96QO=R!I;F1I8V%T97,@
M87!P;&EC871I;VX@<W5P<&]R="!F;W(@5VEN9&]W<R`Q,"`M+3X*("`@("`@
M/'-U<'!O<G1E9$]3($ED/2)[.&4P9C=A,3(M8F9B,RTT9F4X+6(Y834M-#AF
M9#4P83$U83EA?2(O/B`*("`@(#PO87!P;&EC871I;VX^"B`@/"]C;VUP871I
M8FEL:71Y/@H\+V%S<V5M8FQY/@H`````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````````!``
M`#`````,,)LPHC"W,+\PUS#D,/`P]S#^,`DQ'S$H,3@Q0C%1,60Q@3&2,;DQ
M`#```(0```"P-N$VAS>A-]8W_#=T.)$XYS@5.3@Y3SEY.;8Y[3D-.DHZ6CJ5
M.B8[/#M-.VT[@SN;.ZL[P3O7.^T[!CP</#P\63QO/)<\P3S@/$4]4#V+/;0]
MT3T$/A0^'#XD/D0^63YQ/GD^@3Z)/M,^X3X#/Q<_.S]I/\H_UC_C/P```$``
M`-`````%,"<P53#Z,#<Q13%4,78Q@3&*,9,QG#&E,:XQMS'`,<LQU3'?,>DQ
M\S'],0<R$3(;,B4R+S(Y,D,R33);,FLR>3*),I<RIS*Q,K8RQ3+6,N8R]C((
M,R0S+S,^,U4S=C/#,\HSW#/E,_,S^S,#-`LT$S1I-+`TO#0/-10U(C4J-3HU
M0C5*-5(U6C5J-7(US37G-?HV`C<*-S0W;3>/-]`W\3<4.#0X9SBK./8X^S@7
M.2,Y*SE6.68Y<CFD.=XY(SJ0/[<_QS_//P!0``"X`0``"3!@,:@QLS'(,>0Q
M!#(:,CPR5#)C,ITRT3+A,@$S&3,I,TDS:3.`,YDSMC/+,]<S"#0Q-$XT<32`
M-(\TUS3D-/(T_C09-40U<C62-:4UKC6W-18V1#9A-GDVDC::-J(VJC:R-KHV
MPC;*-M(VVC;B-NHV\C;Z-@(W"C<2-QHW(C<J-S(W.C="-THW4C=:-V(W:C=R
M-Z8WKC>V-[XWQC?.-]8WWC?F-^XW]C?^-P8X#C@6.!XX)C@N.#8X/CA&.$XX
M5CA>.&8X;CAV.'XXACB..)8XGCBF.*XXMCB^.,8XSC@`.3LY0CE).5,Y6CEA
M.6@Y;SEX.7\YACF-.9<YKCF[.<@YU3GB.>\Y_#D*.A<Z)#HK.C,Z.SI#.DLZ
M4SI;.G$Z>#I_.H8ZD#J@.JHZT#H(.P\[%CL@.R<[+CLU.SP[13M,.U,[6CMD
M.X0[E#ND.[0[Q#O4.^0[\SL>/#<\0#Q)/,X\XCS[/`0]"CT4/1D])CVR/;H]
MPCW*/=X]]CT2/AH^/SYH/G`^=SZ"/I$^I#ZX/LP^^3[_/@\_%C\>/TX_8C]U
M/X8_E#^K/[L_S#_C/^\_`&````P````D,````'````P````,,```````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
?````````````````````````````````````````````
