/*
 * Copyright (c) 2008-2016 Stefan Krah. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#ifndef MPDECIMAL_H
#define MPDECIMAL_H


#ifdef __cplusplus
extern "C" {
  #ifndef __STDC_LIMIT_MACROS
    #define __STDC_LIMIT_MACROS
    #define MPD_CLEAR_STDC_LIMIT_MACROS
  #endif
#endif


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <assert.h>
#include <stdint.h>
#include <inttypes.h>


#ifndef __GNUC_STDC_INLINE__
  #define __GNUC_STDC_INLINE__ 1
#endif
#if defined(__GNUC__) && !defined(__INTEL_COMPILER)
  #define UNUSED __attribute__((unused))
#else
  #define UNUSED
#endif
#if (defined(__linux__) || defined(__FreeBSD__) || defined(__APPLE__)) && \
    defined(__GNUC__) && __GNUC__ >= 4 && !defined(__INTEL_COMPILER)
  #define MPD_PRAGMA(x) _Pragma(x)
  #define MPD_HIDE_SYMBOLS_START "GCC visibility push(hidden)"
  #define MPD_HIDE_SYMBOLS_END "GCC visibility pop"
#else
  #define MPD_PRAGMA(x)
  #define MPD_HIDE_SYMBOLS_START
  #define MPD_HIDE_SYMBOLS_END
#endif


#if !defined(LEGACY_COMPILER)
  #if !defined(UINT64_MAX)
    /* The following #error is just a warning. If the compiler indeed does
     * not have uint64_t, it is perfectly safe to comment out the #error. */
    #error "Warning: Compiler without uint64_t. Comment out this line."
    #define LEGACY_COMPILER
  #endif
#endif

#if _CRTBLD
#define __MANGLE_ME(x) __mingw_dfp_##x
#define _mpd_baseadd __MANGLE_ME(_mpd_baseadd)
#define _mpd_baseaddto __MANGLE_ME(_mpd_baseaddto)
#define _mpd_basedivmod __MANGLE_ME(_mpd_basedivmod)
#define _mpd_baseincr __MANGLE_ME(_mpd_baseincr)
#define _mpd_basemul __MANGLE_ME(_mpd_basemul)
#define _mpd_baseshiftl __MANGLE_ME(_mpd_baseshiftl)
#define _mpd_baseshiftr __MANGLE_ME(_mpd_baseshiftr)
#define _mpd_basesub __MANGLE_ME(_mpd_basesub)
#define _mpd_basesubfrom __MANGLE_ME(_mpd_basesubfrom)
#define _mpd_getkernel __MANGLE_ME(_mpd_getkernel)
#define _mpd_init_fnt_params __MANGLE_ME(_mpd_init_fnt_params)
#define _mpd_init_w3table __MANGLE_ME(_mpd_init_w3table)
#define _mpd_shortadd __MANGLE_ME(_mpd_shortadd)
#define _mpd_shortadd_b __MANGLE_ME(_mpd_shortadd_b)
#define _mpd_shortdiv __MANGLE_ME(_mpd_shortdiv)
#define _mpd_shortdiv_b __MANGLE_ME(_mpd_shortdiv_b)
#define _mpd_shortmul __MANGLE_ME(_mpd_shortmul)
#define _mpd_shortmul_b __MANGLE_ME(_mpd_shortmul_b)
#define _mpd_shortmul_c __MANGLE_ME(_mpd_shortmul_c)
#define crt3 __MANGLE_ME(crt3)
#define fnt_autoconvolute __MANGLE_ME(fnt_autoconvolute)
#define fnt_convolute __MANGLE_ME(fnt_convolute)
#define fnt_dif2 __MANGLE_ME(fnt_dif2)
#define four_step_fnt __MANGLE_ME(four_step_fnt)
#define inv_four_step_fnt __MANGLE_ME(inv_four_step_fnt)
#define inv_six_step_fnt __MANGLE_ME(inv_six_step_fnt)
#define mpd_abs __MANGLE_ME(mpd_abs)
#define mpd_abs_uint __MANGLE_ME(mpd_abs_uint)
#define mpd_add __MANGLE_ME(mpd_add)
#define mpd_add_i32 __MANGLE_ME(mpd_add_i32)
#define mpd_add_i64 __MANGLE_ME(mpd_add_i64)
#define mpd_add_ssize __MANGLE_ME(mpd_add_ssize)
#define mpd_add_u32 __MANGLE_ME(mpd_add_u32)
#define mpd_add_u64 __MANGLE_ME(mpd_add_u64)
#define mpd_add_uint __MANGLE_ME(mpd_add_uint)
#define mpd_addstatus_raise __MANGLE_ME(mpd_addstatus_raise)
#define mpd_adjexp __MANGLE_ME(mpd_adjexp)
#define mpd_alloc __MANGLE_ME(mpd_alloc)
#define mpd_and __MANGLE_ME(mpd_and)
#define mpd_arith_sign __MANGLE_ME(mpd_arith_sign)
#define mpd_basiccontext __MANGLE_ME(mpd_basiccontext)
#define mpd_calloc __MANGLE_ME(mpd_calloc)
#define mpd_callocfunc_em __MANGLE_ME(mpd_callocfunc_em)
#define mpd_canonical __MANGLE_ME(mpd_canonical)
#define mpd_ceil __MANGLE_ME(mpd_ceil)
#define mpd_check_nan __MANGLE_ME(mpd_check_nan)
#define mpd_check_nans __MANGLE_ME(mpd_check_nans)
#define mpd_class __MANGLE_ME(mpd_class)
#define mpd_clear_flags __MANGLE_ME(mpd_clear_flags)
#define mpd_cmp __MANGLE_ME(mpd_cmp)
#define mpd_cmp_total __MANGLE_ME(mpd_cmp_total)
#define mpd_cmp_total_mag __MANGLE_ME(mpd_cmp_total_mag)
#define mpd_compare __MANGLE_ME(mpd_compare)
#define mpd_compare_signal __MANGLE_ME(mpd_compare_signal)
#define mpd_compare_total __MANGLE_ME(mpd_compare_total)
#define mpd_compare_total_mag __MANGLE_ME(mpd_compare_total_mag)
#define mpd_copy __MANGLE_ME(mpd_copy)
#define mpd_copy_abs __MANGLE_ME(mpd_copy_abs)
#define mpd_copy_flags __MANGLE_ME(mpd_copy_flags)
#define mpd_copy_negate __MANGLE_ME(mpd_copy_negate)
#define mpd_copy_sign __MANGLE_ME(mpd_copy_sign)
#define mpd_defaultcontext __MANGLE_ME(mpd_defaultcontext)
#define mpd_del __MANGLE_ME(mpd_del)
#define mpd_dflt_traphandler __MANGLE_ME(mpd_dflt_traphandler)
#define mpd_digits_to_size __MANGLE_ME(mpd_digits_to_size)
#define mpd_div __MANGLE_ME(mpd_div)
#define mpd_div_i32 __MANGLE_ME(mpd_div_i32)
#define mpd_div_i64 __MANGLE_ME(mpd_div_i64)
#define mpd_div_ssize __MANGLE_ME(mpd_div_ssize)
#define mpd_div_u32 __MANGLE_ME(mpd_div_u32)
#define mpd_div_u64 __MANGLE_ME(mpd_div_u64)
#define mpd_div_uint __MANGLE_ME(mpd_div_uint)
#define mpd_divint __MANGLE_ME(mpd_divint)
#define mpd_divmod __MANGLE_ME(mpd_divmod)
#define mpd_etiny __MANGLE_ME(mpd_etiny)
#define mpd_etop __MANGLE_ME(mpd_etop)
#define mpd_exp __MANGLE_ME(mpd_exp)
#define mpd_exp_digits __MANGLE_ME(mpd_exp_digits)
#define mpd_export_u16 __MANGLE_ME(mpd_export_u16)
#define mpd_export_u32 __MANGLE_ME(mpd_export_u32)
#define mpd_finalize __MANGLE_ME(mpd_finalize)
#define mpd_floor __MANGLE_ME(mpd_floor)
#define mpd_fma __MANGLE_ME(mpd_fma)
#define mpd_format __MANGLE_ME(mpd_format)
#define mpd_fprint __MANGLE_ME(mpd_fprint)
#define mpd_get_i32 __MANGLE_ME(mpd_get_i32)
#define mpd_get_i64 __MANGLE_ME(mpd_get_i64)
#define mpd_get_ssize __MANGLE_ME(mpd_get_ssize)
#define mpd_get_u32 __MANGLE_ME(mpd_get_u32)
#define mpd_get_u64 __MANGLE_ME(mpd_get_u64)
#define mpd_get_uint __MANGLE_ME(mpd_get_uint)
#define mpd_getclamp __MANGLE_ME(mpd_getclamp)
#define mpd_getcr __MANGLE_ME(mpd_getcr)
#define mpd_getemax __MANGLE_ME(mpd_getemax)
#define mpd_getemin __MANGLE_ME(mpd_getemin)
#define mpd_getprec __MANGLE_ME(mpd_getprec)
#define mpd_getround __MANGLE_ME(mpd_getround)
#define mpd_getstatus __MANGLE_ME(mpd_getstatus)
#define mpd_gettraps __MANGLE_ME(mpd_gettraps)
#define mpd_ieee_context __MANGLE_ME(mpd_ieee_context)
#define mpd_import_u16 __MANGLE_ME(mpd_import_u16)
#define mpd_import_u32 __MANGLE_ME(mpd_import_u32)
#define mpd_init __MANGLE_ME(mpd_init)
#define mpd_invert __MANGLE_ME(mpd_invert)
#define mpd_invroot __MANGLE_ME(mpd_invroot)
#define mpd_iscanonical __MANGLE_ME(mpd_iscanonical)
#define mpd_isconst_data __MANGLE_ME(mpd_isconst_data)
#define mpd_isdynamic __MANGLE_ME(mpd_isdynamic)
#define mpd_isdynamic_data __MANGLE_ME(mpd_isdynamic_data)
#define mpd_iseven __MANGLE_ME(mpd_iseven)
#define mpd_isfinite __MANGLE_ME(mpd_isfinite)
#define mpd_isinfinite __MANGLE_ME(mpd_isinfinite)
#define mpd_isinteger __MANGLE_ME(mpd_isinteger)
#define mpd_isnan __MANGLE_ME(mpd_isnan)
#define mpd_isnegative __MANGLE_ME(mpd_isnegative)
#define mpd_isnormal __MANGLE_ME(mpd_isnormal)
#define mpd_isodd __MANGLE_ME(mpd_isodd)
#define mpd_isoddcoeff __MANGLE_ME(mpd_isoddcoeff)
#define mpd_isoddword __MANGLE_ME(mpd_isoddword)
#define mpd_ispositive __MANGLE_ME(mpd_ispositive)
#define mpd_isqnan __MANGLE_ME(mpd_isqnan)
#define mpd_isshared_data __MANGLE_ME(mpd_isshared_data)
#define mpd_issigned __MANGLE_ME(mpd_issigned)
#define mpd_issnan __MANGLE_ME(mpd_issnan)
#define mpd_isspecial __MANGLE_ME(mpd_isspecial)
#define mpd_isstatic __MANGLE_ME(mpd_isstatic)
#define mpd_isstatic_data __MANGLE_ME(mpd_isstatic_data)
#define mpd_issubnormal __MANGLE_ME(mpd_issubnormal)
#define mpd_iszero __MANGLE_ME(mpd_iszero)
#define mpd_iszerocoeff __MANGLE_ME(mpd_iszerocoeff)
#define mpd_ln __MANGLE_ME(mpd_ln)
#define mpd_log10 __MANGLE_ME(mpd_log10)
#define mpd_logb __MANGLE_ME(mpd_logb)
#define mpd_lsd __MANGLE_ME(mpd_lsd)
#define mpd_lsnprint_flags __MANGLE_ME(mpd_lsnprint_flags)
#define mpd_lsnprint_signals __MANGLE_ME(mpd_lsnprint_signals)
#define mpd_max __MANGLE_ME(mpd_max)
#define mpd_max_mag __MANGLE_ME(mpd_max_mag)
#define mpd_maxcoeff __MANGLE_ME(mpd_maxcoeff)
#define mpd_maxcontext __MANGLE_ME(mpd_maxcontext)
#define mpd_min __MANGLE_ME(mpd_min)
#define mpd_min_mag __MANGLE_ME(mpd_min_mag)
#define mpd_minalloc __MANGLE_ME(mpd_minalloc)
#define mpd_minus __MANGLE_ME(mpd_minus)
#define mpd_msd __MANGLE_ME(mpd_msd)
#define mpd_msword __MANGLE_ME(mpd_msword)
#define mpd_mul __MANGLE_ME(mpd_mul)
#define mpd_mul_i32 __MANGLE_ME(mpd_mul_i32)
#define mpd_mul_i64 __MANGLE_ME(mpd_mul_i64)
#define mpd_mul_ssize __MANGLE_ME(mpd_mul_ssize)
#define mpd_mul_u32 __MANGLE_ME(mpd_mul_u32)
#define mpd_mul_u64 __MANGLE_ME(mpd_mul_u64)
#define mpd_mul_uint __MANGLE_ME(mpd_mul_uint)
#define mpd_new __MANGLE_ME(mpd_new)
#define mpd_next_minus __MANGLE_ME(mpd_next_minus)
#define mpd_next_plus __MANGLE_ME(mpd_next_plus)
#define mpd_next_toward __MANGLE_ME(mpd_next_toward)
#define mpd_or __MANGLE_ME(mpd_or)
#define mpd_parse_fmt_str __MANGLE_ME(mpd_parse_fmt_str)
#define mpd_plus __MANGLE_ME(mpd_plus)
#define mpd_pow __MANGLE_ME(mpd_pow)
#define mpd_powmod __MANGLE_ME(mpd_powmod)
#define mpd_print __MANGLE_ME(mpd_print)
#define mpd_qabs __MANGLE_ME(mpd_qabs)
#define mpd_qabs_uint __MANGLE_ME(mpd_qabs_uint)
#define mpd_qadd __MANGLE_ME(mpd_qadd)
#define mpd_qadd_i32 __MANGLE_ME(mpd_qadd_i32)
#define mpd_qadd_i64 __MANGLE_ME(mpd_qadd_i64)
#define mpd_qadd_ssize __MANGLE_ME(mpd_qadd_ssize)
#define mpd_qadd_u32 __MANGLE_ME(mpd_qadd_u32)
#define mpd_qadd_u64 __MANGLE_ME(mpd_qadd_u64)
#define mpd_qadd_uint __MANGLE_ME(mpd_qadd_uint)
#define mpd_qand __MANGLE_ME(mpd_qand)
#define mpd_qceil __MANGLE_ME(mpd_qceil)
#define mpd_qcheck_nan __MANGLE_ME(mpd_qcheck_nan)
#define mpd_qcheck_nans __MANGLE_ME(mpd_qcheck_nans)
#define mpd_qcmp __MANGLE_ME(mpd_qcmp)
#define mpd_qcompare __MANGLE_ME(mpd_qcompare)
#define mpd_qcompare_signal __MANGLE_ME(mpd_qcompare_signal)
#define mpd_qcopy __MANGLE_ME(mpd_qcopy)
#define mpd_qcopy_abs __MANGLE_ME(mpd_qcopy_abs)
#define mpd_qcopy_negate __MANGLE_ME(mpd_qcopy_negate)
#define mpd_qcopy_sign __MANGLE_ME(mpd_qcopy_sign)
#define mpd_qdiv __MANGLE_ME(mpd_qdiv)
#define mpd_qdiv_i32 __MANGLE_ME(mpd_qdiv_i32)
#define mpd_qdiv_i64 __MANGLE_ME(mpd_qdiv_i64)
#define mpd_qdiv_ssize __MANGLE_ME(mpd_qdiv_ssize)
#define mpd_qdiv_u32 __MANGLE_ME(mpd_qdiv_u32)
#define mpd_qdiv_u64 __MANGLE_ME(mpd_qdiv_u64)
#define mpd_qdiv_uint __MANGLE_ME(mpd_qdiv_uint)
#define mpd_qdivint __MANGLE_ME(mpd_qdivint)
#define mpd_qdivmod __MANGLE_ME(mpd_qdivmod)
#define mpd_qexp __MANGLE_ME(mpd_qexp)
#define mpd_qexport_u16 __MANGLE_ME(mpd_qexport_u16)
#define mpd_qexport_u32 __MANGLE_ME(mpd_qexport_u32)
#define mpd_qfinalize __MANGLE_ME(mpd_qfinalize)
#define mpd_qfloor __MANGLE_ME(mpd_qfloor)
#define mpd_qfma __MANGLE_ME(mpd_qfma)
#define mpd_qformat __MANGLE_ME(mpd_qformat)
#define mpd_qformat_spec __MANGLE_ME(mpd_qformat_spec)
#define mpd_qget_i32 __MANGLE_ME(mpd_qget_i32)
#define mpd_qget_i64 __MANGLE_ME(mpd_qget_i64)
#define mpd_qget_ssize __MANGLE_ME(mpd_qget_ssize)
#define mpd_qget_u32 __MANGLE_ME(mpd_qget_u32)
#define mpd_qget_u64 __MANGLE_ME(mpd_qget_u64)
#define mpd_qget_uint __MANGLE_ME(mpd_qget_uint)
#define mpd_qimport_u16 __MANGLE_ME(mpd_qimport_u16)
#define mpd_qimport_u32 __MANGLE_ME(mpd_qimport_u32)
#define mpd_qinvert __MANGLE_ME(mpd_qinvert)
#define mpd_qinvroot __MANGLE_ME(mpd_qinvroot)
#define mpd_qln __MANGLE_ME(mpd_qln)
#define mpd_qln10 __MANGLE_ME(mpd_qln10)
#define mpd_qlog10 __MANGLE_ME(mpd_qlog10)
#define mpd_qlogb __MANGLE_ME(mpd_qlogb)
#define mpd_qmax __MANGLE_ME(mpd_qmax)
#define mpd_qmax_mag __MANGLE_ME(mpd_qmax_mag)
#define mpd_qmaxcoeff __MANGLE_ME(mpd_qmaxcoeff)
#define mpd_qmin __MANGLE_ME(mpd_qmin)
#define mpd_qmin_mag __MANGLE_ME(mpd_qmin_mag)
#define mpd_qminus __MANGLE_ME(mpd_qminus)
#define mpd_qmul __MANGLE_ME(mpd_qmul)
#define mpd_qmul_i32 __MANGLE_ME(mpd_qmul_i32)
#define mpd_qmul_i64 __MANGLE_ME(mpd_qmul_i64)
#define mpd_qmul_ssize __MANGLE_ME(mpd_qmul_ssize)
#define mpd_qmul_u32 __MANGLE_ME(mpd_qmul_u32)
#define mpd_qmul_u64 __MANGLE_ME(mpd_qmul_u64)
#define mpd_qmul_uint __MANGLE_ME(mpd_qmul_uint)
#define mpd_qncopy __MANGLE_ME(mpd_qncopy)
#define mpd_qnew __MANGLE_ME(mpd_qnew)
#define mpd_qnew_size __MANGLE_ME(mpd_qnew_size)
#define mpd_qnext_minus __MANGLE_ME(mpd_qnext_minus)
#define mpd_qnext_plus __MANGLE_ME(mpd_qnext_plus)
#define mpd_qnext_toward __MANGLE_ME(mpd_qnext_toward)
#define mpd_qor __MANGLE_ME(mpd_qor)
#define mpd_qplus __MANGLE_ME(mpd_qplus)
#define mpd_qpow __MANGLE_ME(mpd_qpow)
#define mpd_qpowmod __MANGLE_ME(mpd_qpowmod)
#define mpd_qquantize __MANGLE_ME(mpd_qquantize)
#define mpd_qreduce __MANGLE_ME(mpd_qreduce)
#define mpd_qrem __MANGLE_ME(mpd_qrem)
#define mpd_qrem_near __MANGLE_ME(mpd_qrem_near)
#define mpd_qrescale __MANGLE_ME(mpd_qrescale)
#define mpd_qrescale_fmt __MANGLE_ME(mpd_qrescale_fmt)
#define mpd_qresize __MANGLE_ME(mpd_qresize)
#define mpd_qresize_zero __MANGLE_ME(mpd_qresize_zero)
#define mpd_qrotate __MANGLE_ME(mpd_qrotate)
#define mpd_qround_to_int __MANGLE_ME(mpd_qround_to_int)
#define mpd_qround_to_intx __MANGLE_ME(mpd_qround_to_intx)
#define mpd_qscaleb __MANGLE_ME(mpd_qscaleb)
#define mpd_qset_i32 __MANGLE_ME(mpd_qset_i32)
#define mpd_qset_i64 __MANGLE_ME(mpd_qset_i64)
#define mpd_qset_ssize __MANGLE_ME(mpd_qset_ssize)
#define mpd_qset_string __MANGLE_ME(mpd_qset_string)
#define mpd_qset_u32 __MANGLE_ME(mpd_qset_u32)
#define mpd_qset_u64 __MANGLE_ME(mpd_qset_u64)
#define mpd_qset_uint __MANGLE_ME(mpd_qset_uint)
#define mpd_qsetclamp __MANGLE_ME(mpd_qsetclamp)
#define mpd_qsetcr __MANGLE_ME(mpd_qsetcr)
#define mpd_qsetemax __MANGLE_ME(mpd_qsetemax)
#define mpd_qsetemin __MANGLE_ME(mpd_qsetemin)
#define mpd_qsetprec __MANGLE_ME(mpd_qsetprec)
#define mpd_qsetround __MANGLE_ME(mpd_qsetround)
#define mpd_qsetstatus __MANGLE_ME(mpd_qsetstatus)
#define mpd_qsettraps __MANGLE_ME(mpd_qsettraps)
#define mpd_qshift __MANGLE_ME(mpd_qshift)
#define mpd_qshiftl __MANGLE_ME(mpd_qshiftl)
#define mpd_qshiftn __MANGLE_ME(mpd_qshiftn)
#define mpd_qshiftr __MANGLE_ME(mpd_qshiftr)
#define mpd_qshiftr_inplace __MANGLE_ME(mpd_qshiftr_inplace)
#define mpd_qsqrt __MANGLE_ME(mpd_qsqrt)
#define mpd_qsset_i32 __MANGLE_ME(mpd_qsset_i32)
#define mpd_qsset_i64 __MANGLE_ME(mpd_qsset_i64)
#define mpd_qsset_ssize __MANGLE_ME(mpd_qsset_ssize)
#define mpd_qsset_u32 __MANGLE_ME(mpd_qsset_u32)
#define mpd_qsset_u64 __MANGLE_ME(mpd_qsset_u64)
#define mpd_qsset_uint __MANGLE_ME(mpd_qsset_uint)
#define mpd_qsub __MANGLE_ME(mpd_qsub)
#define mpd_qsub_i32 __MANGLE_ME(mpd_qsub_i32)
#define mpd_qsub_i64 __MANGLE_ME(mpd_qsub_i64)
#define mpd_qsub_ssize __MANGLE_ME(mpd_qsub_ssize)
#define mpd_qsub_u32 __MANGLE_ME(mpd_qsub_u32)
#define mpd_qsub_u64 __MANGLE_ME(mpd_qsub_u64)
#define mpd_qsub_uint __MANGLE_ME(mpd_qsub_uint)
#define mpd_qtrunc __MANGLE_ME(mpd_qtrunc)
#define mpd_quantize __MANGLE_ME(mpd_quantize)
#define mpd_qxor __MANGLE_ME(mpd_qxor)
#define mpd_radix __MANGLE_ME(mpd_radix)
#define mpd_realloc __MANGLE_ME(mpd_realloc)
#define mpd_realloc_dyn __MANGLE_ME(mpd_realloc_dyn)
#define mpd_reduce __MANGLE_ME(mpd_reduce)
#define mpd_rem __MANGLE_ME(mpd_rem)
#define mpd_rem_near __MANGLE_ME(mpd_rem_near)
#define mpd_rescale __MANGLE_ME(mpd_rescale)
#define mpd_resize __MANGLE_ME(mpd_resize)
#define mpd_resize_zero __MANGLE_ME(mpd_resize_zero)
#define mpd_rotate __MANGLE_ME(mpd_rotate)
#define mpd_round_to_int __MANGLE_ME(mpd_round_to_int)
#define mpd_round_to_intx __MANGLE_ME(mpd_round_to_intx)
#define mpd_same_quantum __MANGLE_ME(mpd_same_quantum)
#define mpd_scaleb __MANGLE_ME(mpd_scaleb)
#define mpd_set_const_data __MANGLE_ME(mpd_set_const_data)
#define mpd_set_dynamic __MANGLE_ME(mpd_set_dynamic)
#define mpd_set_dynamic_data __MANGLE_ME(mpd_set_dynamic_data)
#define mpd_set_flags __MANGLE_ME(mpd_set_flags)
#define mpd_set_i32 __MANGLE_ME(mpd_set_i32)
#define mpd_set_i64 __MANGLE_ME(mpd_set_i64)
#define mpd_set_infinity __MANGLE_ME(mpd_set_infinity)
#define mpd_set_negative __MANGLE_ME(mpd_set_negative)
#define mpd_set_positive __MANGLE_ME(mpd_set_positive)
#define mpd_set_qnan __MANGLE_ME(mpd_set_qnan)
#define mpd_set_shared_data __MANGLE_ME(mpd_set_shared_data)
#define mpd_set_sign __MANGLE_ME(mpd_set_sign)
#define mpd_set_snan __MANGLE_ME(mpd_set_snan)
#define mpd_set_ssize __MANGLE_ME(mpd_set_ssize)
#define mpd_set_static __MANGLE_ME(mpd_set_static)
#define mpd_set_static_data __MANGLE_ME(mpd_set_static_data)
#define mpd_set_string __MANGLE_ME(mpd_set_string)
#define mpd_set_u32 __MANGLE_ME(mpd_set_u32)
#define mpd_set_u64 __MANGLE_ME(mpd_set_u64)
#define mpd_set_uint __MANGLE_ME(mpd_set_uint)
#define mpd_setdigits __MANGLE_ME(mpd_setdigits)
#define mpd_seterror __MANGLE_ME(mpd_seterror)
#define mpd_setminalloc __MANGLE_ME(mpd_setminalloc)
#define mpd_setspecial __MANGLE_ME(mpd_setspecial)
#define mpd_sh_alloc __MANGLE_ME(mpd_sh_alloc)
#define mpd_shift __MANGLE_ME(mpd_shift)
#define mpd_shiftl __MANGLE_ME(mpd_shiftl)
#define mpd_shiftn __MANGLE_ME(mpd_shiftn)
#define mpd_shiftr __MANGLE_ME(mpd_shiftr)
#define mpd_sign __MANGLE_ME(mpd_sign)
#define mpd_signcpy __MANGLE_ME(mpd_signcpy)
#define mpd_sizeinbase __MANGLE_ME(mpd_sizeinbase)
#define mpd_snprint_flags __MANGLE_ME(mpd_snprint_flags)
#define mpd_sqrt __MANGLE_ME(mpd_sqrt)
#define mpd_sset_i32 __MANGLE_ME(mpd_sset_i32)
#define mpd_sset_i64 __MANGLE_ME(mpd_sset_i64)
#define mpd_sset_ssize __MANGLE_ME(mpd_sset_ssize)
#define mpd_sset_u32 __MANGLE_ME(mpd_sset_u32)
#define mpd_sset_u64 __MANGLE_ME(mpd_sset_u64)
#define mpd_sset_uint __MANGLE_ME(mpd_sset_uint)
#define mpd_sub __MANGLE_ME(mpd_sub)
#define mpd_sub_i32 __MANGLE_ME(mpd_sub_i32)
#define mpd_sub_i64 __MANGLE_ME(mpd_sub_i64)
#define mpd_sub_ssize __MANGLE_ME(mpd_sub_ssize)
#define mpd_sub_u32 __MANGLE_ME(mpd_sub_u32)
#define mpd_sub_u64 __MANGLE_ME(mpd_sub_u64)
#define mpd_sub_uint __MANGLE_ME(mpd_sub_uint)
#define mpd_switch_to_dyn __MANGLE_ME(mpd_switch_to_dyn)
#define mpd_switch_to_dyn_zero __MANGLE_ME(mpd_switch_to_dyn_zero)
#define mpd_to_eng __MANGLE_ME(mpd_to_eng)
#define mpd_to_eng_size __MANGLE_ME(mpd_to_eng_size)
#define mpd_to_sci __MANGLE_ME(mpd_to_sci)
#define mpd_to_sci_size __MANGLE_ME(mpd_to_sci_size)
#define mpd_trail_zeros __MANGLE_ME(mpd_trail_zeros)
#define mpd_trunc __MANGLE_ME(mpd_trunc)
#define mpd_uint_zero __MANGLE_ME(mpd_uint_zero)
#define mpd_validate_lconv __MANGLE_ME(mpd_validate_lconv)
#define mpd_version __MANGLE_ME(mpd_version)
#define mpd_word_digits __MANGLE_ME(mpd_word_digits)
#define mpd_xor __MANGLE_ME(mpd_xor)
#define mpd_zerocoeff __MANGLE_ME(mpd_zerocoeff)
#define six_step_fnt __MANGLE_ME(six_step_fnt)
#define std_fnt __MANGLE_ME(std_fnt)
#define std_inv_fnt __MANGLE_ME(std_inv_fnt)
#define std_trans __MANGLE_ME(std_trans)
#define transpose_pow2 __MANGLE_ME(transpose_pow2)
#endif


/******************************************************************************/
/*                                  Version                                   */
/******************************************************************************/

#define MPD_MAJOR_VERSION 2
#define MPD_MINOR_VERSION 4
#define MPD_MICRO_VERSION 0

#define MPD_VERSION "2.4.0"

const char *mpd_version(void);


/******************************************************************************/
/*                              Configuration                                 */
/******************************************************************************/

/* ABI: 32-bit */
#ifdef CONFIG_64
  #error "cannot use CONFIG_64 with 32-bit header."
#endif

#ifndef CONFIG_32
  #define CONFIG_32
#endif


/* BEGIN CONFIG_64 */
#if defined(CONFIG_64)
/* types for modular and base arithmetic */
#define MPD_UINT_MAX UINT64_MAX
#define MPD_BITS_PER_UINT 64
typedef uint64_t mpd_uint_t;  /* unsigned mod type */

#define MPD_SIZE_MAX SIZE_MAX
typedef size_t mpd_size_t; /* unsigned size type */

/* type for exp, digits, len, prec */
#define MPD_SSIZE_MAX INT64_MAX
#define MPD_SSIZE_MIN INT64_MIN
typedef int64_t mpd_ssize_t;
#define _mpd_strtossize strtoll

/* decimal arithmetic */
#define MPD_RADIX 10000000000000000000ULL  /* 10**19 */
#define MPD_RDIGITS 19
#define MPD_MAX_POW10 19
#define MPD_EXPDIGITS 19  /* MPD_EXPDIGITS <= MPD_RDIGITS+1 */

#define MPD_MAXTRANSFORM_2N 4294967296ULL      /* 2**32 */
#define MPD_MAX_PREC 999999999999999999LL
#define MPD_MAX_PREC_LOG2 64
#define MPD_ELIMIT  1000000000000000000LL
#define MPD_MAX_EMAX   999999999999999999LL    /* ELIMIT-1 */
#define MPD_MIN_EMIN  (-999999999999999999LL)  /* -EMAX */
#define MPD_MIN_ETINY (MPD_MIN_EMIN-(MPD_MAX_PREC-1))
#define MPD_EXP_INF 2000000000000000001LL
#define MPD_EXP_CLAMP (-4000000000000000001LL)
#define MPD_MAXIMPORT 105263157894736842L /* ceil((2*MPD_MAX_PREC)/MPD_RDIGITS) */

/* conversion specifiers */
#define PRI_mpd_uint_t PRIu64
#define PRI_mpd_ssize_t PRIi64
/* END CONFIG_64 */


/* BEGIN CONFIG_32 */
#elif defined(CONFIG_32)
/* types for modular and base arithmetic */
#define MPD_UINT_MAX UINT32_MAX
#define MPD_BITS_PER_UINT 32
typedef uint32_t mpd_uint_t;  /* unsigned mod type */

#ifndef LEGACY_COMPILER
#define MPD_UUINT_MAX UINT64_MAX
typedef uint64_t mpd_uuint_t; /* double width unsigned mod type */
#endif

#define MPD_SIZE_MAX SIZE_MAX
typedef size_t mpd_size_t; /* unsigned size type */

/* type for dec->len, dec->exp, ctx->prec */
#define MPD_SSIZE_MAX INT32_MAX
#define MPD_SSIZE_MIN INT32_MIN
typedef int32_t mpd_ssize_t;
#define _mpd_strtossize strtol

/* decimal arithmetic */
#define MPD_RADIX 1000000000UL  /* 10**9 */
#define MPD_RDIGITS 9
#define MPD_MAX_POW10 9
#define MPD_EXPDIGITS 10 /* MPD_EXPDIGITS <= MPD_RDIGITS+1 */

#define MPD_MAXTRANSFORM_2N 33554432UL /* 2**25 */
#define MPD_MAX_PREC 425000000L
#define MPD_MAX_PREC_LOG2 32
#define MPD_ELIMIT 425000001L
#define MPD_MAX_EMAX 425000000L        /* ELIMIT-1 */
#define MPD_MIN_EMIN (-425000000L)     /* -EMAX */
#define MPD_MIN_ETINY (MPD_MIN_EMIN-(MPD_MAX_PREC-1))
#define MPD_EXP_INF 1000000001L      /* allows for emax=999999999 in the tests */
#define MPD_EXP_CLAMP (-2000000001L) /* allows for emin=-999999999 in the tests */
#define MPD_MAXIMPORT 94444445L      /* ceil((2*MPD_MAX_PREC)/MPD_RDIGITS) */

/* conversion specifiers */
#define PRI_mpd_uint_t PRIu32
#define PRI_mpd_ssize_t PRIi32
/* END CONFIG_32 */

#else
  #error "define CONFIG_64 or CONFIG_32"
#endif
/* END CONFIG */


#if MPD_SIZE_MAX != MPD_UINT_MAX
  #error "unsupported platform: need mpd_size_t == mpd_uint_t"
#endif


/******************************************************************************/
/*                                Context                                     */
/******************************************************************************/

enum {
    MPD_ROUND_UP,          /* round away from 0               */
    MPD_ROUND_DOWN,        /* round toward 0 (truncate)       */
    MPD_ROUND_CEILING,     /* round toward +infinity          */
    MPD_ROUND_FLOOR,       /* round toward -infinity          */
    MPD_ROUND_HALF_UP,     /* 0.5 is rounded up               */
    MPD_ROUND_HALF_DOWN,   /* 0.5 is rounded down             */
    MPD_ROUND_HALF_EVEN,   /* 0.5 is rounded to even          */
    MPD_ROUND_05UP,        /* round zero or five away from 0  */
    MPD_ROUND_TRUNC,       /* truncate, but set infinity      */
    MPD_ROUND_GUARD
};

enum { MPD_CLAMP_DEFAULT, MPD_CLAMP_IEEE_754, MPD_CLAMP_GUARD };

extern const char *mpd_round_string[MPD_ROUND_GUARD];
extern const char *mpd_clamp_string[MPD_CLAMP_GUARD];


typedef struct mpd_context_t {
    mpd_ssize_t prec;   /* precision */
    mpd_ssize_t emax;   /* max positive exp */
    mpd_ssize_t emin;   /* min negative exp */
    uint32_t traps;     /* status events that should be trapped */
    uint32_t status;    /* status flags */
    uint32_t newtrap;   /* set by mpd_addstatus_raise() */
    int      round;     /* rounding mode */
    int      clamp;     /* clamp mode */
    int      allcr;     /* all functions correctly rounded */
} mpd_context_t;


/* Status flags */
#define MPD_Clamped             0x00000001U
#define MPD_Conversion_syntax   0x00000002U
#define MPD_Division_by_zero    0x00000004U
#define MPD_Division_impossible 0x00000008U
#define MPD_Division_undefined  0x00000010U
#define MPD_Fpu_error           0x00000020U
#define MPD_Inexact             0x00000040U
#define MPD_Invalid_context     0x00000080U
#define MPD_Invalid_operation   0x00000100U
#define MPD_Malloc_error        0x00000200U
#define MPD_Not_implemented     0x00000400U
#define MPD_Overflow            0x00000800U
#define MPD_Rounded             0x00001000U
#define MPD_Subnormal           0x00002000U
#define MPD_Underflow           0x00004000U
#define MPD_Max_status         (0x00008000U-1U)

/* Conditions that result in an IEEE 754 exception */
#define MPD_IEEE_Invalid_operation (MPD_Conversion_syntax |   \
                                    MPD_Division_impossible | \
                                    MPD_Division_undefined |  \
                                    MPD_Fpu_error |           \
                                    MPD_Invalid_context |     \
                                    MPD_Invalid_operation |   \
                                    MPD_Malloc_error)         \

/* Errors that require the result of an operation to be set to NaN */
#define MPD_Errors (MPD_IEEE_Invalid_operation | \
                    MPD_Division_by_zero)

/* Default traps */
#define MPD_Traps (MPD_IEEE_Invalid_operation | \
                   MPD_Division_by_zero |       \
                   MPD_Overflow |               \
                   MPD_Underflow)

/* Official name */
#define MPD_Insufficient_storage MPD_Malloc_error

/* IEEE 754 interchange format contexts */
#define MPD_IEEE_CONTEXT_MAX_BITS 512 /* 16*(log2(MPD_MAX_EMAX / 3)-3) */
#define MPD_DECIMAL32 32
#define MPD_DECIMAL64 64
#define MPD_DECIMAL128 128


#define MPD_MINALLOC_MIN 2
#define MPD_MINALLOC_MAX 64
extern mpd_ssize_t MPD_MINALLOC;
extern void (* mpd_traphandler)(mpd_context_t *);
void mpd_dflt_traphandler(mpd_context_t *);

void mpd_setminalloc(mpd_ssize_t n);
void mpd_init(mpd_context_t *ctx, mpd_ssize_t prec);

void mpd_maxcontext(mpd_context_t *ctx);
void mpd_defaultcontext(mpd_context_t *ctx);
void mpd_basiccontext(mpd_context_t *ctx);
int mpd_ieee_context(mpd_context_t *ctx, int bits);

mpd_ssize_t mpd_getprec(const mpd_context_t *ctx);
mpd_ssize_t mpd_getemax(const mpd_context_t *ctx);
mpd_ssize_t mpd_getemin(const mpd_context_t *ctx);
int mpd_getround(const mpd_context_t *ctx);
uint32_t mpd_gettraps(const mpd_context_t *ctx);
uint32_t mpd_getstatus(const mpd_context_t *ctx);
int mpd_getclamp(const mpd_context_t *ctx);
int mpd_getcr(const mpd_context_t *ctx);

int mpd_qsetprec(mpd_context_t *ctx, mpd_ssize_t prec);
int mpd_qsetemax(mpd_context_t *ctx, mpd_ssize_t emax);
int mpd_qsetemin(mpd_context_t *ctx, mpd_ssize_t emin);
int mpd_qsetround(mpd_context_t *ctx, int newround);
int mpd_qsettraps(mpd_context_t *ctx, uint32_t flags);
int mpd_qsetstatus(mpd_context_t *ctx, uint32_t flags);
int mpd_qsetclamp(mpd_context_t *ctx, int c);
int mpd_qsetcr(mpd_context_t *ctx, int c);
void mpd_addstatus_raise(mpd_context_t *ctx, uint32_t flags);


/******************************************************************************/
/*                           Decimal Arithmetic                               */
/******************************************************************************/

/* mpd_t flags */
#define MPD_POS                 ((uint8_t)0)
#define MPD_NEG                 ((uint8_t)1)
#define MPD_INF                 ((uint8_t)2)
#define MPD_NAN                 ((uint8_t)4)
#define MPD_SNAN                ((uint8_t)8)
#define MPD_SPECIAL (MPD_INF|MPD_NAN|MPD_SNAN)
#define MPD_STATIC              ((uint8_t)16)
#define MPD_STATIC_DATA         ((uint8_t)32)
#define MPD_SHARED_DATA         ((uint8_t)64)
#define MPD_CONST_DATA          ((uint8_t)128)
#define MPD_DATAFLAGS (MPD_STATIC_DATA|MPD_SHARED_DATA|MPD_CONST_DATA)

/* mpd_t */
typedef struct mpd_t {
    uint8_t flags;
    mpd_ssize_t exp;
    mpd_ssize_t digits;
    mpd_ssize_t len;
    mpd_ssize_t alloc;
    mpd_uint_t *data;
} mpd_t;


typedef unsigned char uchar;


/******************************************************************************/
/*                       Quiet, thread-safe functions                         */
/******************************************************************************/

/* format specification */
typedef struct mpd_spec_t {
    mpd_ssize_t min_width; /* minimum field width */
    mpd_ssize_t prec;      /* fraction digits or significant digits */
    char type;             /* conversion specifier */
    char align;            /* alignment */
    char sign;             /* sign printing/alignment */
    char fill[5];          /* fill character */
    const char *dot;       /* decimal point */
    const char *sep;       /* thousands separator */
    const char *grouping;  /* grouping of digits */
} mpd_spec_t;

/* output to a string */
char *mpd_to_sci(const mpd_t *dec, int fmt);
char *mpd_to_eng(const mpd_t *dec, int fmt);
mpd_ssize_t mpd_to_sci_size(char **res, const mpd_t *dec, int fmt);
mpd_ssize_t mpd_to_eng_size(char **res, const mpd_t *dec, int fmt);
int mpd_validate_lconv(mpd_spec_t *spec);
int mpd_parse_fmt_str(mpd_spec_t *spec, const char *fmt, int caps);
char *mpd_qformat_spec(const mpd_t *dec, const mpd_spec_t *spec, const mpd_context_t *ctx, uint32_t *status);
char *mpd_qformat(const mpd_t *dec, const char *fmt, const mpd_context_t *ctx, uint32_t *status);

#define MPD_NUM_FLAGS 15
#define MPD_MAX_FLAG_STRING 208
#define MPD_MAX_FLAG_LIST (MPD_MAX_FLAG_STRING+18)
#define MPD_MAX_SIGNAL_LIST 121
int mpd_snprint_flags(char *dest, int nmemb, uint32_t flags);
int mpd_lsnprint_flags(char *dest, int nmemb, uint32_t flags, const char *flag_string[]);
int mpd_lsnprint_signals(char *dest, int nmemb, uint32_t flags, const char *signal_string[]);

/* output to a file */
void mpd_fprint(FILE *file, const mpd_t *dec);
void mpd_print(const mpd_t *dec);

/* assignment from a string */
void mpd_qset_string(mpd_t *dec, const char *s, const mpd_context_t *ctx, uint32_t *status);

/* set to NaN with error flags */
void mpd_seterror(mpd_t *result, uint32_t flags, uint32_t *status);
/* set a special with sign and type */
void mpd_setspecial(mpd_t *dec, uint8_t sign, uint8_t type);
/* set coefficient to zero or all nines */
void mpd_zerocoeff(mpd_t *result);
void mpd_qmaxcoeff(mpd_t *result, const mpd_context_t *ctx, uint32_t *status);

/* quietly assign a C integer type to an mpd_t */
void mpd_qset_ssize(mpd_t *result, mpd_ssize_t a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qset_i32(mpd_t *result, int32_t a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qset_uint(mpd_t *result, mpd_uint_t a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qset_u32(mpd_t *result, uint32_t a, const mpd_context_t *ctx, uint32_t *status);
#ifndef LEGACY_COMPILER
void mpd_qset_i64(mpd_t *result, int64_t a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qset_u64(mpd_t *result, uint64_t a, const mpd_context_t *ctx, uint32_t *status);
#endif

/* quietly assign a C integer type to an mpd_t with a static coefficient */
void mpd_qsset_ssize(mpd_t *result, mpd_ssize_t a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsset_i32(mpd_t *result, int32_t a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsset_uint(mpd_t *result, mpd_uint_t a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsset_u32(mpd_t *result, uint32_t a, const mpd_context_t *ctx, uint32_t *status);

/* quietly get a C integer type from an mpd_t */
mpd_ssize_t mpd_qget_ssize(const mpd_t *dec, uint32_t *status);
mpd_uint_t mpd_qget_uint(const mpd_t *dec, uint32_t *status);
mpd_uint_t mpd_qabs_uint(const mpd_t *dec, uint32_t *status);

int32_t mpd_qget_i32(const mpd_t *dec, uint32_t *status);
uint32_t mpd_qget_u32(const mpd_t *dec, uint32_t *status);
#ifndef LEGACY_COMPILER
int64_t mpd_qget_i64(const mpd_t *dec, uint32_t *status);
uint64_t mpd_qget_u64(const mpd_t *dec, uint32_t *status);
#endif

/* quiet functions */
int mpd_qcheck_nan(mpd_t *nanresult, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
int mpd_qcheck_nans(mpd_t *nanresult, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qfinalize(mpd_t *result, const mpd_context_t *ctx, uint32_t *status);

const char *mpd_class(const mpd_t *a, const mpd_context_t *ctx);

int mpd_qcopy(mpd_t *result, const mpd_t *a,  uint32_t *status);
mpd_t *mpd_qncopy(const mpd_t *a);
int mpd_qcopy_abs(mpd_t *result, const mpd_t *a, uint32_t *status);
int mpd_qcopy_negate(mpd_t *result, const mpd_t *a, uint32_t *status);
int mpd_qcopy_sign(mpd_t *result, const mpd_t *a, const mpd_t *b, uint32_t *status);

void mpd_qand(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qinvert(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qlogb(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qor(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qscaleb(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qxor(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
int mpd_same_quantum(const mpd_t *a, const mpd_t *b);

void mpd_qrotate(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
int mpd_qshiftl(mpd_t *result, const mpd_t *a, mpd_ssize_t n, uint32_t *status);
mpd_uint_t mpd_qshiftr(mpd_t *result, const mpd_t *a, mpd_ssize_t n, uint32_t *status);
mpd_uint_t mpd_qshiftr_inplace(mpd_t *result, mpd_ssize_t n);
void mpd_qshift(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qshiftn(mpd_t *result, const mpd_t *a, mpd_ssize_t n, const mpd_context_t *ctx, uint32_t *status);

int mpd_qcmp(const mpd_t *a, const mpd_t *b, uint32_t *status);
int mpd_qcompare(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
int mpd_qcompare_signal(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
int mpd_cmp_total(const mpd_t *a, const mpd_t *b);
int mpd_cmp_total_mag(const mpd_t *a, const mpd_t *b);
int mpd_compare_total(mpd_t *result, const mpd_t *a, const mpd_t *b);
int mpd_compare_total_mag(mpd_t *result, const mpd_t *a, const mpd_t *b);

void mpd_qround_to_intx(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qround_to_int(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qtrunc(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qfloor(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qceil(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);

void mpd_qabs(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmax(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmax_mag(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmin(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmin_mag(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qminus(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qplus(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qnext_minus(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qnext_plus(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qnext_toward(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qquantize(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qrescale(mpd_t *result, const mpd_t *a, mpd_ssize_t exp, const mpd_context_t *ctx, uint32_t *status);
void mpd_qrescale_fmt(mpd_t *result, const mpd_t *a, mpd_ssize_t exp, const mpd_context_t *ctx, uint32_t *status);
void mpd_qreduce(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qadd(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qadd_ssize(mpd_t *result, const mpd_t *a, mpd_ssize_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qadd_i32(mpd_t *result, const mpd_t *a, int32_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qadd_uint(mpd_t *result, const mpd_t *a, mpd_uint_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qadd_u32(mpd_t *result, const mpd_t *a, uint32_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsub(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsub_ssize(mpd_t *result, const mpd_t *a, mpd_ssize_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsub_i32(mpd_t *result, const mpd_t *a, int32_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsub_uint(mpd_t *result, const mpd_t *a, mpd_uint_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsub_u32(mpd_t *result, const mpd_t *a, uint32_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmul(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmul_ssize(mpd_t *result, const mpd_t *a, mpd_ssize_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmul_i32(mpd_t *result, const mpd_t *a, int32_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmul_uint(mpd_t *result, const mpd_t *a, mpd_uint_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmul_u32(mpd_t *result, const mpd_t *a, uint32_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qfma(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_t *c, const mpd_context_t *ctx, uint32_t *status);
void mpd_qdiv(mpd_t *q, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qdiv_ssize(mpd_t *result, const mpd_t *a, mpd_ssize_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qdiv_i32(mpd_t *result, const mpd_t *a, int32_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qdiv_uint(mpd_t *result, const mpd_t *a, mpd_uint_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qdiv_u32(mpd_t *result, const mpd_t *a, uint32_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qdivint(mpd_t *q, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qrem(mpd_t *r, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qrem_near(mpd_t *r, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qdivmod(mpd_t *q, mpd_t *r, const mpd_t *a, const mpd_t *b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qpow(mpd_t *result, const mpd_t *base, const mpd_t *exp, const mpd_context_t *ctx, uint32_t *status);
void mpd_qpowmod(mpd_t *result, const mpd_t *base, const mpd_t *exp, const mpd_t *mod, const mpd_context_t *ctx, uint32_t *status);
void mpd_qexp(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qln10(mpd_t *result, mpd_ssize_t prec, uint32_t *status);
void mpd_qln(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qlog10(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsqrt(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qinvroot(mpd_t *result, const mpd_t *a, const mpd_context_t *ctx, uint32_t *status);

#ifndef LEGACY_COMPILER
void mpd_qadd_i64(mpd_t *result, const mpd_t *a, int64_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qadd_u64(mpd_t *result, const mpd_t *a, uint64_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsub_i64(mpd_t *result, const mpd_t *a, int64_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsub_u64(mpd_t *result, const mpd_t *a, uint64_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmul_i64(mpd_t *result, const mpd_t *a, int64_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qmul_u64(mpd_t *result, const mpd_t *a, uint64_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qdiv_i64(mpd_t *result, const mpd_t *a, int64_t b, const mpd_context_t *ctx, uint32_t *status);
void mpd_qdiv_u64(mpd_t *result, const mpd_t *a, uint64_t b, const mpd_context_t *ctx, uint32_t *status);
#endif


size_t mpd_sizeinbase(const mpd_t *a, uint32_t base);
void mpd_qimport_u16(mpd_t *result, const uint16_t *srcdata, size_t srclen,
                     uint8_t srcsign, uint32_t srcbase,
                     const mpd_context_t *ctx, uint32_t *status);
void mpd_qimport_u32(mpd_t *result, const uint32_t *srcdata, size_t srclen,
                     uint8_t srcsign, uint32_t srcbase,
                     const mpd_context_t *ctx, uint32_t *status);
size_t mpd_qexport_u16(uint16_t **rdata, size_t rlen, uint32_t base,
                       const mpd_t *src, uint32_t *status);
size_t mpd_qexport_u32(uint32_t **rdata, size_t rlen, uint32_t base,
                       const mpd_t *src, uint32_t *status);


/******************************************************************************/
/*                           Signalling functions                             */
/******************************************************************************/

char *mpd_format(const mpd_t *dec, const char *fmt, mpd_context_t *ctx);
void mpd_import_u16(mpd_t *result, const uint16_t *srcdata, size_t srclen, uint8_t srcsign, uint32_t base, mpd_context_t *ctx);
void mpd_import_u32(mpd_t *result, const uint32_t *srcdata, size_t srclen, uint8_t srcsign, uint32_t base, mpd_context_t *ctx);
size_t mpd_export_u16(uint16_t **rdata, size_t rlen, uint32_t base, const mpd_t *src, mpd_context_t *ctx);
size_t mpd_export_u32(uint32_t **rdata, size_t rlen, uint32_t base, const mpd_t *src, mpd_context_t *ctx);
void mpd_finalize(mpd_t *result, mpd_context_t *ctx);
int mpd_check_nan(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
int mpd_check_nans(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_set_string(mpd_t *result, const char *s, mpd_context_t *ctx);
void mpd_maxcoeff(mpd_t *result, mpd_context_t *ctx);
void mpd_sset_ssize(mpd_t *result, mpd_ssize_t a, mpd_context_t *ctx);
void mpd_sset_i32(mpd_t *result, int32_t a, mpd_context_t *ctx);
void mpd_sset_uint(mpd_t *result, mpd_uint_t a, mpd_context_t *ctx);
void mpd_sset_u32(mpd_t *result, uint32_t a, mpd_context_t *ctx);
void mpd_set_ssize(mpd_t *result, mpd_ssize_t a, mpd_context_t *ctx);
void mpd_set_i32(mpd_t *result, int32_t a, mpd_context_t *ctx);
void mpd_set_uint(mpd_t *result, mpd_uint_t a, mpd_context_t *ctx);
void mpd_set_u32(mpd_t *result, uint32_t a, mpd_context_t *ctx);
#ifndef LEGACY_COMPILER
void mpd_set_i64(mpd_t *result, int64_t a, mpd_context_t *ctx);
void mpd_set_u64(mpd_t *result, uint64_t a, mpd_context_t *ctx);
#endif
mpd_ssize_t mpd_get_ssize(const mpd_t *a, mpd_context_t *ctx);
mpd_uint_t mpd_get_uint(const mpd_t *a, mpd_context_t *ctx);
mpd_uint_t mpd_abs_uint(const mpd_t *a, mpd_context_t *ctx);
int32_t mpd_get_i32(const mpd_t *a, mpd_context_t *ctx);
uint32_t mpd_get_u32(const mpd_t *a, mpd_context_t *ctx);
#ifndef LEGACY_COMPILER
int64_t mpd_get_i64(const mpd_t *a, mpd_context_t *ctx);
uint64_t mpd_get_u64(const mpd_t *a, mpd_context_t *ctx);
#endif
void mpd_and(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_copy(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_canonical(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_copy_abs(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_copy_negate(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_copy_sign(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_invert(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_logb(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_or(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_rotate(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_scaleb(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_shiftl(mpd_t *result, const mpd_t *a, mpd_ssize_t n, mpd_context_t *ctx);
mpd_uint_t mpd_shiftr(mpd_t *result, const mpd_t *a, mpd_ssize_t n, mpd_context_t *ctx);
void mpd_shiftn(mpd_t *result, const mpd_t *a, mpd_ssize_t n, mpd_context_t *ctx);
void mpd_shift(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_xor(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_abs(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
int mpd_cmp(const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
int mpd_compare(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
int mpd_compare_signal(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_add(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_add_ssize(mpd_t *result, const mpd_t *a, mpd_ssize_t b, mpd_context_t *ctx);
void mpd_add_i32(mpd_t *result, const mpd_t *a, int32_t b, mpd_context_t *ctx);
void mpd_add_uint(mpd_t *result, const mpd_t *a, mpd_uint_t b, mpd_context_t *ctx);
void mpd_add_u32(mpd_t *result, const mpd_t *a, uint32_t b, mpd_context_t *ctx);
void mpd_sub(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_sub_ssize(mpd_t *result, const mpd_t *a, mpd_ssize_t b, mpd_context_t *ctx);
void mpd_sub_i32(mpd_t *result, const mpd_t *a, int32_t b, mpd_context_t *ctx);
void mpd_sub_uint(mpd_t *result, const mpd_t *a, mpd_uint_t b, mpd_context_t *ctx);
void mpd_sub_u32(mpd_t *result, const mpd_t *a, uint32_t b, mpd_context_t *ctx);
void mpd_div(mpd_t *q, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_div_ssize(mpd_t *result, const mpd_t *a, mpd_ssize_t b, mpd_context_t *ctx);
void mpd_div_i32(mpd_t *result, const mpd_t *a, int32_t b, mpd_context_t *ctx);
void mpd_div_uint(mpd_t *result, const mpd_t *a, mpd_uint_t b, mpd_context_t *ctx);
void mpd_div_u32(mpd_t *result, const mpd_t *a, uint32_t b, mpd_context_t *ctx);
void mpd_divmod(mpd_t *q, mpd_t *r, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_divint(mpd_t *q, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_exp(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_fma(mpd_t *result, const mpd_t *a, const mpd_t *b, const mpd_t *c, mpd_context_t *ctx);
void mpd_ln(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_log10(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_max(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_max_mag(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_min(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_min_mag(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_minus(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_mul(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_mul_ssize(mpd_t *result, const mpd_t *a, mpd_ssize_t b, mpd_context_t *ctx);
void mpd_mul_i32(mpd_t *result, const mpd_t *a, int32_t b, mpd_context_t *ctx);
void mpd_mul_uint(mpd_t *result, const mpd_t *a, mpd_uint_t b, mpd_context_t *ctx);
void mpd_mul_u32(mpd_t *result, const mpd_t *a, uint32_t b, mpd_context_t *ctx);
void mpd_next_minus(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_next_plus(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_next_toward(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_plus(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_pow(mpd_t *result, const mpd_t *base, const mpd_t *exp, mpd_context_t *ctx);
void mpd_powmod(mpd_t *result, const mpd_t *base, const mpd_t *exp, const mpd_t *mod, mpd_context_t *ctx);
void mpd_quantize(mpd_t *result, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_rescale(mpd_t *result, const mpd_t *a, mpd_ssize_t exp, mpd_context_t *ctx);
void mpd_reduce(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_rem(mpd_t *r, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_rem_near(mpd_t *r, const mpd_t *a, const mpd_t *b, mpd_context_t *ctx);
void mpd_round_to_intx(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_round_to_int(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_trunc(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_floor(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_ceil(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_sqrt(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);
void mpd_invroot(mpd_t *result, const mpd_t *a, mpd_context_t *ctx);

#ifndef LEGACY_COMPILER
void mpd_add_i64(mpd_t *result, const mpd_t *a, int64_t b, mpd_context_t *ctx);
void mpd_add_u64(mpd_t *result, const mpd_t *a, uint64_t b, mpd_context_t *ctx);
void mpd_sub_i64(mpd_t *result, const mpd_t *a, int64_t b, mpd_context_t *ctx);
void mpd_sub_u64(mpd_t *result, const mpd_t *a, uint64_t b, mpd_context_t *ctx);
void mpd_div_i64(mpd_t *result, const mpd_t *a, int64_t b, mpd_context_t *ctx);
void mpd_div_u64(mpd_t *result, const mpd_t *a, uint64_t b, mpd_context_t *ctx);
void mpd_mul_i64(mpd_t *result, const mpd_t *a, int64_t b, mpd_context_t *ctx);
void mpd_mul_u64(mpd_t *result, const mpd_t *a, uint64_t b, mpd_context_t *ctx);
#endif


/******************************************************************************/
/*                          Configuration specific                            */
/******************************************************************************/

#ifdef CONFIG_64
void mpd_qsset_i64(mpd_t *result, int64_t a, const mpd_context_t *ctx, uint32_t *status);
void mpd_qsset_u64(mpd_t *result, uint64_t a, const mpd_context_t *ctx, uint32_t *status);
void mpd_sset_i64(mpd_t *result, int64_t a, mpd_context_t *ctx);
void mpd_sset_u64(mpd_t *result, uint64_t a, mpd_context_t *ctx);
#endif


/******************************************************************************/
/*                       Get attributes of a decimal                          */
/******************************************************************************/

mpd_ssize_t mpd_adjexp(const mpd_t *dec);
mpd_ssize_t mpd_etiny(const mpd_context_t *ctx);
mpd_ssize_t mpd_etop(const mpd_context_t *ctx);
mpd_uint_t mpd_msword(const mpd_t *dec);
int mpd_word_digits(mpd_uint_t word);
/* most significant digit of a word */
mpd_uint_t mpd_msd(mpd_uint_t word);
/* least significant digit of a word */
mpd_uint_t mpd_lsd(mpd_uint_t word);
/* coefficient size needed to store 'digits' */
mpd_ssize_t mpd_digits_to_size(mpd_ssize_t digits);
/* number of digits in the exponent, undefined for MPD_SSIZE_MIN */
int mpd_exp_digits(mpd_ssize_t exp);
int mpd_iscanonical(const mpd_t *dec UNUSED);
int mpd_isfinite(const mpd_t *dec);
int mpd_isinfinite(const mpd_t *dec);
int mpd_isinteger(const mpd_t *dec);
int mpd_isnan(const mpd_t *dec);
int mpd_isnegative(const mpd_t *dec);
int mpd_ispositive(const mpd_t *dec);
int mpd_isqnan(const mpd_t *dec);
int mpd_issigned(const mpd_t *dec);
int mpd_issnan(const mpd_t *dec);
int mpd_isspecial(const mpd_t *dec);
int mpd_iszero(const mpd_t *dec);
/* undefined for special numbers */
int mpd_iszerocoeff(const mpd_t *dec);
int mpd_isnormal(const mpd_t *dec, const mpd_context_t *ctx);
int mpd_issubnormal(const mpd_t *dec, const mpd_context_t *ctx);
/* odd word */
int mpd_isoddword(mpd_uint_t word);
/* odd coefficient */
int mpd_isoddcoeff(const mpd_t *dec);
/* odd decimal, only defined for integers */
int mpd_isodd(const mpd_t *dec);
/* even decimal, only defined for integers */
int mpd_iseven(const mpd_t *dec);
/* 0 if dec is positive, 1 if dec is negative */
uint8_t mpd_sign(const mpd_t *dec);
/* 1 if dec is positive, -1 if dec is negative */
int mpd_arith_sign(const mpd_t *dec);
long mpd_radix(void);
int mpd_isdynamic(const mpd_t *dec);
int mpd_isstatic(const mpd_t *dec);
int mpd_isdynamic_data(const mpd_t *dec);
int mpd_isstatic_data(const mpd_t *dec);
int mpd_isshared_data(const mpd_t *dec);
int mpd_isconst_data(const mpd_t *dec);
mpd_ssize_t mpd_trail_zeros(const mpd_t *dec);


/******************************************************************************/
/*                       Set attributes of a decimal                          */
/******************************************************************************/

/* set number of decimal digits in the coefficient */
void mpd_setdigits(mpd_t *result);
void mpd_set_sign(mpd_t *result, uint8_t sign);
/* copy sign from another decimal */
void mpd_signcpy(mpd_t *result, const mpd_t *a);
void mpd_set_infinity(mpd_t *result);
void mpd_set_qnan(mpd_t *result);
void mpd_set_snan(mpd_t *result);
void mpd_set_negative(mpd_t *result);
void mpd_set_positive(mpd_t *result);
void mpd_set_dynamic(mpd_t *result);
void mpd_set_static(mpd_t *result);
void mpd_set_dynamic_data(mpd_t *result);
void mpd_set_static_data(mpd_t *result);
void mpd_set_shared_data(mpd_t *result);
void mpd_set_const_data(mpd_t *result);
void mpd_clear_flags(mpd_t *result);
void mpd_set_flags(mpd_t *result, uint8_t flags);
void mpd_copy_flags(mpd_t *result, const mpd_t *a);


/******************************************************************************/
/*                              Error Macros                                  */
/******************************************************************************/

#define mpd_err_fatal(...) \
    do {fprintf(stderr, "%s:%d: error: ", __FILE__, __LINE__); \
        fprintf(stderr, __VA_ARGS__);  fputc('\n', stderr);    \
        abort();                                               \
    } while (0)
#define mpd_err_warn(...) \
    do {fprintf(stderr, "%s:%d: warning: ", __FILE__, __LINE__); \
        fprintf(stderr, __VA_ARGS__); fputc('\n', stderr);       \
    } while (0)


/******************************************************************************/
/*                            Memory handling                                 */
/******************************************************************************/

extern void *(* const mpd_mallocfunc)(size_t size);
extern void *(* const mpd_callocfunc)(size_t nmemb, size_t size);
extern void *(* const mpd_reallocfunc)(void *ptr, size_t size);
extern void (* const mpd_free)(void *ptr);

void *mpd_callocfunc_em(size_t nmemb, size_t size);

void *mpd_alloc(mpd_size_t nmemb, mpd_size_t size);
void *mpd_calloc(mpd_size_t nmemb, mpd_size_t size);
void *mpd_realloc(void *ptr, mpd_size_t nmemb, mpd_size_t size, uint8_t *err);
void *mpd_sh_alloc(mpd_size_t struct_size, mpd_size_t nmemb, mpd_size_t size);

mpd_t *mpd_qnew(void);
mpd_t *mpd_new(mpd_context_t *ctx);
mpd_t *mpd_qnew_size(mpd_ssize_t size);
void mpd_del(mpd_t *dec);

void mpd_uint_zero(mpd_uint_t *dest, mpd_size_t len);
int mpd_qresize(mpd_t *result, mpd_ssize_t size, uint32_t *status);
int mpd_qresize_zero(mpd_t *result, mpd_ssize_t size, uint32_t *status);
void mpd_minalloc(mpd_t *result);

int mpd_resize(mpd_t *result, mpd_ssize_t size, mpd_context_t *ctx);
int mpd_resize_zero(mpd_t *result, mpd_ssize_t size, mpd_context_t *ctx);


#ifdef __cplusplus
  #ifdef MPD_CLEAR_STDC_LIMIT_MACROS
    #undef MPD_CLEAR_STDC_LIMIT_MACROS
    #undef __STDC_LIMIT_MACROS
  #endif
} /* END extern "C" */
#endif


#endif /* MPDECIMAL_H */



