#!/bin/bash
set -e
set -u

# The following must be set to the extension of the UPSTREAM source tarball.
SRC_EXT="tar.bz2"

# The following must be set to the extension of a GPG signature file.
SIG_EXT="asc"

# Deduce some important paths.
script_path=$(readlink -e "$0")
script_name=$(basename "$script_path")
script_dir=$(dirname "$script_path")

# Parse the script name to deduce my PN, VER, and REL.
PN="$script_name"
ext="build.sh"
PN=${PN%%.$ext}
REL=${PN##*-}
PN=${PN%%-$REL}
VER=${PN##*-}
PN=${PN%%-$VER}
#echo "PN='$PN' VER='$VER' REL='$REL'"

# Important for cygport to function:
export SRC_DIR="${PN}-${VER}"
export SRC_URI="file://./${SRC_DIR}.${SRC_EXT}"

# Convenience vars.
PKG="${PN}-${VER}-${REL}"
CYP="${PKG}.cygport"
SRC="${PKG}-src.tar.bz2"
SIG="${PN}-${VER}.${SRC_EXT}.${SIG_EXT}"

usage_and_exit() {
    
    cat <<EOF 1>&2
Usage: $0 <beginstage> [ <endstage> (default=5) ]

Some cygport equivalences:

    'cygport $CYP prep'
        <==>        '$0 1 1'

    'cygport $CYP almostall'
        <==>        '$0 1'
        <==>        '$0 1 5'

    'cygport $CYP finish'
        <==>        '$0 6 6'

    'cygport $CYP all'
        <==>        '$0 1 6'
EOF
    exit 1
}

to_run_stage() {
    (( $1 >= beginstage )) && (( $1 <= endstage ))
}

# Get the build stages to begin and end at.

# The begin stage must be specified and in the range 1-6.
beginstage=${1:-99}
[[ $beginstage =~ ^[1-6]$ ]] || usage_and_exit

endstage=${2:-5}
[[ $endstage   =~ ^[1-6]$ ]] || usage_and_exit
(( beginstage <= endstage )) || usage_and_exit

# Now begin.

cd "$script_dir"

to_run_stage 1 && cygport "$CYP" prep
to_run_stage 2 && {
	  
	  #
	  # Create the source patches now and stash them away to include in
	  # the source tarball at the packaging stage.
	  #
	  
    cygport "$CYP" mkpatch
    
    rm -rf "$PKG/patch-real"
    cp -a  "$PKG/patch" "$PKG/patch-real"
}
to_run_stage 3 && cygport "$CYP" compile
to_run_stage 4 && cygport "$CYP" install
to_run_stage 5 && {

    cygport "$CYP" package
    
    #
    # Massage the source package tarball:
    # Put the REAL patches in; also add this script and the
    # upstream source tarball signature file.
    #
    
    rm -rf "$PKG/patch"
    cp -a  "$PKG/patch-real" "$PKG/patch"
    
    mytmp="$(mktemp -d)"
    
    cd       "$mytmp"
    tar -xjf "$script_dir/$SRC"
    rm       *.patch
    cp -p    "$script_dir/$PKG/patch/"* \
             "$script_dir/$SIG" \
             "$script_path" \
             .
    tar -cjf "$script_dir/$SRC" *
    
    cd       "$script_dir"
    rm -rf   "$mytmp"
    cp -p    "$SRC" "$PKG/dist/$PN"
}
to_run_stage 6 && cygport "$CYP" finish
