#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
Functions for reducing terms
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Reduction function
=============================================

---------------------------------------------
Types
---------------------------------------------

The reduction functions know the following data structures:

map: Something that represents a map

rnf: rnf stands for "root normal form". An rnf is the same as a map, but cannot represent bottom and one can tell immediately whether or not the given map equals T.

term: Something that represents a term and which equals a map when all variables in the term are bound to maps.

env: env stands for "environment". An env is a list of maps. When evaluating a term in the context of an environment, the variable with deBruijn index n in the term is bound to the n'th element of the environment (counting from zero).

Syntax:

tnf ::= nil           ; tnf means "truth normal form".
                      ; T of map theory is represented by Nil in Lisp.
                      ; Nil is chosen because it is easy to test for
                      ; nil-ness in Lisp.

fnf ::= (mtag . args) ; fnf means "function normal form".
                      ; (mtag . args) represents the function normal
                      ; form which, when applied to a map x, yields
                      ; (apply mtag x args) which must return an rnf.

mtag ::= symbol       ; mtag means "map tag".
                      ; The symbol must be a non-keyword Lisp symbol.

rnf ::= tnf | fnf     ; rnf means "root normal form".

indir ::= (:I . rnf)  ; Equals the given rnf. I stands for 'identity' or
                      ; 'indirection'.

closure ::=
(:A term env)         ; Unevaluated closure. Equals the given term when the
                      ; variables of the term are bound as specified in the
                      ; given environment. A stands for "apply" (i.e. "apply
                      ; term to environment").

map ::=
rnf | indir | closure

term ::= (ttag . args); Represents a term which has value (apply ttag e args)
                      ; in environment e.

ttag ::= symbol       ; ttag means "term tag".
                      ; The symbol must be a non-keyword Lisp symbol.

env ::=
nil | (map . env)     ; The enviroment that maps deBruijn index 0 to the given
                      ; map and otherwise maps deBruijn indices to maps as
                      ; specified by env.

---------------------------------------------
General functions
---------------------------------------------

(ifnil x y z)
This function (Lisp macro, actually) equals (if x z y).

(term2closure term env): term x env -> closure
Construct a closure from the given term and environment. Satisfies
  (term2closure (closure2term closure) (closure2env closure)) = closure.

(closure2term closure): closure -> term
Destructs the given closure. Satisfies
  (closure2term (term2closure term env)) = term.

(closure2env closure): closure -> env
Destructs the given closure. Satisfies
  (closure2env (term2closure term env)) = env.

(term*2closure* term* env): term* x env -> closure*
Coordinatewise application of term2closure to term*. Note that closure* is a subset of env. term*2closure* is the environment forming operation in a lazy call. An eager call would use a function named something like term*2rnf* instead.

(closure-overwrite closure rnf): closure x rnf -> rnf
Destructively overwrite the given closure with an indirection to the given rnf and return the rnf.

(map2rnf map): map -> rnf
Reduce the given map to rnf.

(term2rnf term env): term x env -> rnf
Apply the given term to the given environment and reduce to rnf. Satisfies
   (map2rnf (term2closure term env)) = (term2rnf term env)

|#

(defmacro ifnil (x y z) `(if ,x ,z ,y))

(deff term2closure (term env)
 (list :A term env))

(deff closure2term (closure)
 (second closure))

(deff closure2env (closure)
 (third closure))

(deff term*2closure* (term* env)
 (if (null term*) nil
  (cons
   (term2closure (car term*) env)
   (term*2closure* (cdr term*) env))))

(deff closure-overwrite (closure rnf) ; returns rnf
 (setf (car closure) :I)
 (setf (cdr closure) rnf))

(deff map2rnf (map)
 (case (car map)
  ((nil) nil)
  ((:A) (closure-overwrite map (term2rnf (closure2term map) (closure2env map))))
  ((:I) (cdr map))
  (t map)))

(deff term2rnf (term env) ; term x env -> rnf
 (apply (car term) env (cdr term)))

#|
---------------------------------------------
Apply function
---------------------------------------------

(map-apply fct arg) : map x map -> rnf
Applies the given fct to the given arg and reduces to rnf.

(ttag-apply env fct arg) : env x term x term -> rnf
Applies the given fct and arg to the given environment and then applies the resulting fct to the resulting arg.

(term-apply fct arg) : term x term -> term
Applies the first term to the second.
|#

(deff map-apply (fct arg)
 (:let rnf (map2rnf fct))
 (:when (null rnf) nil)
 (apply (car rnf) arg (cdr rnf)))

(deff ttag-apply (env fct arg)
 (:let rnf (term2rnf fct env))
 (:when (null rnf) nil)
 (apply (car rnf) (term2closure arg env) (cdr rnf)))

(deff term-apply (fct arg)
 (list 'ttag-apply fct arg))

#|
---------------------------------------------
Selection function
---------------------------------------------

(map-if cond then else) : map x map x map -> rnf
Reduces the given condition to rnf and, depending on the result,
returns the 'then' or 'else' map reduced to rnf.

(ttag-if env cond then else) : env x term x term x term -> rnf
Applies the given condition to the given environment and, depending on
the result, applies either the 'then' or 'else' term to the given
environment.

(term-if cond then else) : term x term x term -> term
Applies the if-construct to the three given terms.
|#

(deff map-if (cond then else)
 (if (null (map2rnf cond)) (map2rnf then) (map2rnf else)))

(deff ttag-if (env cond then else)
 (ifnil (term2rnf cond env) (term2rnf then env) (term2rnf else env)))

(deff term-if (cond then else)
 (list 'ttag-if cond then else))

#|
---------------------------------------------
True functions
---------------------------------------------

(map-true) : rnf
Returns the atom T.

(ttag-true env) : env -> rnf
Returns the atom T independently of the given environment.

term-true : term
The term that denotes T.

|#

(deff map-true ()
 nil)

(deff ttag-true (:env)
 nil)

(defc term-true (list 'ttag-true))

#|
---------------------------------------------
Variable functions
---------------------------------------------

(ttag-var env index) : env x card -> rnf
Return the value of the given variable in the given environment.

(term-var index) : card -> term
Returns the variable with the given index.
|#

(deff ttag-var (env index)
 (map2rnf (nth index env)))

(deff term-var (index)
 (list 'ttag-var index))

#|
---------------------------------------------
Lambda functions
---------------------------------------------

(mtag-lambda arg env term) : map x env x term -> rnf
Effectively applies \x.term to the given arg where x is the variable with deBruijn index 0. Then decrements all remaining deBruijn indices in the term. Then applies the result to the given environment and reduces to rnf. In reallity, mtag-lambda simply conses the given arg onto the given env and reduces the given term in that environment.

(ttag-lambda env term) : env x term -> rnf
Applies \x.term to the given environment where x is the variable with deBruijn index 0.

(term-lambda term) : term -> term
Returns \x.term where x is the variable with deBruijn index 0.

|#

(deff mtag-lambda (arg env term)
 (term2rnf term (cons arg env)))

(deff ttag-lambda (env term)
 (list 'mtag-lambda env term))

(deff term-lambda (term)
 (list 'ttag-lambda term))

#|
---------------------------------------------
Evaluation in environment
---------------------------------------------
(term-eval term &rest term*) : Compute the value of the given term in the environement defined by the given term*. The terms of term* are computed in the empty environment.
|#

(deff term-eval (term &rest term*)
 (term2rnf term (term*2closure* term* nil)))

#|
---------------------------------------------
Test of truth and falsehood
---------------------------------------------
|#

(ntst (term2rnf term-true nil))
(test (term2rnf (term-lambda term-true) nil))

(ntst (term-eval term-true))
(test (term-eval (term-lambda term-true)))

(defc term-t term-true)
(defc term-f (term-lambda term-t))

(ntst (term-eval term-t))
(test (term-eval term-f))

#|
---------------------------------------------
Test of variables
---------------------------------------------
|#

(defc var-0 (term-var 0))
(defc var-1 (term-var 1))
(defc var-2 (term-var 2))

(ntst (term-eval var-0 term-t term-t))
(ntst (term-eval var-0 term-t term-f))
(test (term-eval var-0 term-f term-t))
(test (term-eval var-0 term-f term-f))

(ntst (term-eval var-1 term-t term-t))
(test (term-eval var-1 term-t term-f))
(ntst (term-eval var-1 term-f term-t))
(test (term-eval var-1 term-f term-f))

(ntst (term-eval var-2 term-t term-t))
(ntst (term-eval var-2 term-t term-f))
(ntst (term-eval var-2 term-f term-t))
(ntst (term-eval var-2 term-f term-f))

#|
---------------------------------------------
Test of selection
---------------------------------------------
|#

(ntst (term-eval (term-if term-t term-t term-t)))
(ntst (term-eval (term-if term-t term-t term-f)))
(test (term-eval (term-if term-t term-f term-t)))
(test (term-eval (term-if term-t term-f term-f)))
(ntst (term-eval (term-if term-f term-t term-t)))
(test (term-eval (term-if term-f term-t term-f)))
(ntst (term-eval (term-if term-f term-f term-t)))
(test (term-eval (term-if term-f term-f term-f)))

#|
---------------------------------------------
Test of selection/variables
---------------------------------------------
|#

(ntst (term-eval (term-if var-0 var-1 var-2) term-t term-t term-t))
(ntst (term-eval (term-if var-0 var-1 var-2) term-t term-t term-f))
(test (term-eval (term-if var-0 var-1 var-2) term-t term-f term-t))
(test (term-eval (term-if var-0 var-1 var-2) term-t term-f term-f))
(ntst (term-eval (term-if var-0 var-1 var-2) term-f term-t term-t))
(test (term-eval (term-if var-0 var-1 var-2) term-f term-t term-f))
(ntst (term-eval (term-if var-0 var-1 var-2) term-f term-f term-t))
(test (term-eval (term-if var-0 var-1 var-2) term-f term-f term-f))

#|
---------------------------------------------
Test of apply/lambda
---------------------------------------------
|#

(ntst (term-eval (term-apply (term-lambda var-0) term-t) term-t))
(ntst (term-eval (term-apply (term-lambda var-0) term-t) term-f))
(test (term-eval (term-apply (term-lambda var-0) term-f) term-t))
(test (term-eval (term-apply (term-lambda var-0) term-f) term-f))

(ntst (term-eval (term-apply (term-lambda term-t) term-t) term-t))
(ntst (term-eval (term-apply (term-lambda term-t) term-t) term-f))
(ntst (term-eval (term-apply (term-lambda term-t) term-f) term-t))
(ntst (term-eval (term-apply (term-lambda term-t) term-f) term-f))

(test (term-eval (term-apply (term-lambda term-f) term-t) term-t))
(test (term-eval (term-apply (term-lambda term-f) term-t) term-f))
(test (term-eval (term-apply (term-lambda term-f) term-f) term-t))
(test (term-eval (term-apply (term-lambda term-f) term-f) term-f))

(ntst (term-eval (term-apply (term-lambda var-1) term-t) term-t))
(test (term-eval (term-apply (term-lambda var-1) term-t) term-f))
(ntst (term-eval (term-apply (term-lambda var-1) term-f) term-t))
(test (term-eval (term-apply (term-lambda var-1) term-f) term-f))

#|
---------------------------------------------
Test of Curried functions
---------------------------------------------
|#

(defc term-k (term-lambda (term-lambda var-1)))

(ntst (term-eval (term-apply (term-apply term-k term-t) term-t)))
(ntst (term-eval (term-apply (term-apply term-k term-t) term-f)))
(test (term-eval (term-apply (term-apply term-k term-f) term-t)))
(test (term-eval (term-apply (term-apply term-k term-f) term-f)))

(defc term-ki (term-lambda (term-lambda var-0)))

(ntst (term-eval (term-apply (term-apply term-ki term-t) term-t)))
(test (term-eval (term-apply (term-apply term-ki term-t) term-f)))
(ntst (term-eval (term-apply (term-apply term-ki term-f) term-t)))
(test (term-eval (term-apply (term-apply term-ki term-f) term-f)))

#|
---------------------------------------------
Test of unoptimized pair function
---------------------------------------------
|#

(deff term-p (head tail)
 (term-lambda (term-if var-0 head tail)))
(deff term-hd (pair)
 (term-apply pair term-t))
(deff term-tl (pair)
 (term-apply pair term-f))

(ntst (term-eval (term-hd (term-p term-t term-t))))
(ntst (term-eval (term-hd (term-p term-t term-f))))
(test (term-eval (term-hd (term-p term-f term-t))))
(test (term-eval (term-hd (term-p term-f term-f))))
(ntst (term-eval (term-tl (term-p term-t term-t))))
(test (term-eval (term-tl (term-p term-t term-f))))
(ntst (term-eval (term-tl (term-p term-f term-t))))
(test (term-eval (term-tl (term-p term-f term-f))))

#|
---------------------------------------------
Test of fixed point combinator
---------------------------------------------
|#

(defc term-y1 (term-lambda (term-apply var-1 (term-apply var-0 var-0))))
(defc term-y (term-lambda (term-apply term-y1 term-y1)))
(defc term-yk (term-apply term-y term-k))
(defc term-yki (term-apply term-y term-ki))

(ntst (term-eval (term-apply term-y term-t)))
(ntst (term-eval (term-apply term-y term-f)))
(test (term-eval term-yk))
(test (term-eval (term-apply term-yk term-t)))
(test (term-eval (term-apply term-yk term-f)))
(test (term-eval (term-apply (term-apply term-yk term-t) term-t)))
(test (term-eval (term-apply (term-apply term-yk term-f) term-f)))
(test (term-eval term-yki))
(ntst (term-eval (term-apply term-yki term-t)))
(test (term-eval (term-apply term-yki term-f)))
















