#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
Logiweb
=============================================
Ripemd-160
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Ripemd-160.
=============================================

Ripemd-160 is described in: H. Dobbertin, A. Bosselaers, B. Preneel, ``RIPEMD-160, a strengthened version of RIPEMD'', which is available on the net in ps and pdf. The article is an updated and corrected version of the article published in Fast Software Encryption, LNCS 1039, D. Gollmann, Ed., Springer-Verlag, 1996, pp. 71-82.

See also http://www.esat.kuleuven.ac.be/~bosselae/ripemd160.html

---------------

(ripemd x) returns the ripemd-160 hash value of the byte-list x.

---------------

(ripemd-mask x) masks x to 32 bit.

(ripemd-rol n x) rotates the 32-bit word x n places left where "left" means "in the carry direction".

(ripemd-sum x y) adds x and y modulo 2^32.

(ripemd-f j x y z) does as specified in the RIPEMD-160 specification.

(ripemd-k j) does as specified in the RIPEMD-160 specification.

(ripemd-k-prime j) does as specified in the RIPEMD-160 specification.

*ripemd-init-word* is the initial value for the hash computation.

(ripemd-byte*-2-byte x) returns the first byte of the list x of bytes if such a byte exists and returns 0 if x is empty.

(ripemd-byte*-2-word n x) converts the first n bytes of the list x of bytes to an integer where the first byte in x is the least significant.

(ripemd-byte*-2-word* n x) converts the first 4*n bytes of x to a list of 32 bit words.

(ripemd-byte*-2-word** x y) converts the list x of bytes to a list of 16 element lists of 32 bit words and appends the result in reverse order to y.

(ripemd-add-zero-block x) if the last two words of x (the last two words of the car of x) are zero, then x has room for a 64 bit length field and the function returns x unmodified. Otherwise, the function adds a block of 16 zeros.

(ripemd-add-length n x) sets the last two words of x to n modulo 2^64.

(ripemd-pad x) pads the byte list x to a sequence of lists with 16 32-bit words in each list.

|#

(defc *ripemd-32-bit-mask* (- (ash 1 32) 1))

(etst *ripemd-32-bit-mask* #xFFFFFFFF)

(deff ripemd-mask (x)
 (logand *ripemd-32-bit-mask* x))

(etst (ripemd-mask #x123456789) #x23456789)

(deff ripemd-rol (n x)
 (ripemd-mask (logior (ash x n) (ash x (- n 32)))))

(etst (ripemd-rol 8 #x12345678) #x34567812)

(deff ripemd-sum (x y)
 (ripemd-mask (+ x y)))

(etst (ripemd-sum 10 100) 110)
(etst (ripemd-sum 10 #xFFFFFFFF) 9)

(deff ripemd-f (j x y z)
 (case (floor j 16)
  (0 (logxor x y z))
  (1 (logior (logand x y) (logandc1 x z)))
  (2 (logxor (ripemd-mask (logorc2 x y)) z))
  (3 (logior (logand x z) (logandc2 y z)))
  (4 (logxor x (ripemd-mask (logorc2 y z))))))

(etst (ripemd-f  0 #b00001111 #b00110011 #b01010101) #b01101001)
(etst (ripemd-f 16 #b00001111 #b00110011 #b01010101) #b01010011)
(etst (ripemd-f 32 #b00001111 #b00110011 #b01010101) #xFFFFFF9A)
(etst (ripemd-f 48 #b00001111 #b00110011 #b01010101) #b00100111)
(etst (ripemd-f 64 #b00001111 #b00110011 #b01010101) #xFFFFFFB4)

(deff ripemd-k (j)
 (case (floor j 16)
  (0 #x00000000)
  (1 #x5A827999)
  (2 #x6ED9EBA1)
  (3 #x8F1BBCDC)
  (4 #xA953FD4E)))

(etst (ripemd-k 64) #xa953fd4e)

(deff ripemd-k-prime (j)
 (case (floor j 16)
  (0 #x50A28BE6)
  (1 #x5C4DD124)
  (2 #x6D703EF3)
  (3 #x7A6D76E9)
  (4 #x00000000)))

(etst (ripemd-k-prime 48) #x7a6d76e9)

(deff list-2-vector (x)
 (coerce x 'vector))

(defc ripemd-r-array
 (list-2-vector '(
   0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15
   7  4 13  1 10  6 15  3 12  0  9  5  2 14 11  8
   3 10 14  4  9 15  8  1  2  7  0  6 13 11  5 12
   1  9 11 10  0  8 12  4 13  3  7 15 14  5  6  2
   4  0  5  9  7 12  2 10 14  1  3  8 11  6 15 13)))
(defc ripemd-r-prime-array
 (list-2-vector '(
   5 14  7  0  9  2 11  4 13  6 15  8  1 10  3 12
   6 11  3  7  0 13  5 10 14 15  8 12  4  9  1  2
  15  5  1  3  7 14  6  9 11  8 12  2 10  0  4 13
   8  6  4  1  3 11 15  0  5 12  2 13  9  7 10 14
  12 15 10  4  1  5  8  7  6  2 13 14  0  3  9 11)))
(defc ripemd-s-array
 (list-2-vector '(
   11 14 15 12  5  8  7  9 11 13 14 15  6  7  9  8
    7  6  8 13 11  9  7 15  7 12 15  9 11  7 13 12
   11 13  6  7 14  9 13 15 14  8 13  6  5 12  7  5
   11 12 14 15 14 15  9  8  9 14  5  6  8  6  5 12
    9 15  5 11  6  8 13 12  5 12 13 14 11  8  5  6)))
(defc ripemd-s-prime-array
 (list-2-vector '(
    8  9  9 11 13 15 15  5  7  7  8 11 14 14 12  6
    9 13 15  7 12  8  9 11  7  7 12  7  6 15 13 11
    9  7 15 11  8  6  6 14 12 13  5 14 13 13  7  5
   15  5  8 11 14 14  6 14  6  9 12  9 12  5 15  8
    8  5 12  9 12  5 14  6  8 13  6  5 15 13 11 11)))

(deff ripemd-r (j)
 (aref ripemd-r-array j))

(etst (ripemd-r 64) 4)

(deff ripemd-r-prime (j)
 (aref ripemd-r-prime-array j))

(etst (ripemd-r-prime 64) 12)

(deff ripemd-s (j)
 (aref ripemd-s-array j))

(etst (ripemd-s 64) 9)

(deff ripemd-s-prime (j)
 (aref ripemd-s-prime-array j))

(etst (ripemd-s-prime 64) 8)

(defc ripemd-init-hash '(
  #x67452301
  #xEFCDAB89
  #x98BADCFE
  #x10325476
  #xC3D2E1F0))

(deff ripemd-byte*-2-byte (x)
 (if x (car x) 0))

(etst (ripemd-byte*-2-byte (list #x12)) #x12)
(etst (ripemd-byte*-2-byte nil) 0)

(deff ripemd-byte*-2-word (n x)
 (:when (= n 0) 0)
 (+ (ripemd-byte*-2-byte x)
  (* 256 (ripemd-byte*-2-word (- n 1) (cdr x)))))

(etst (ripemd-byte*-2-word 0 (list #x12 #x34 #x56)) 0)
(etst (ripemd-byte*-2-word 1 (list #x12 #x34 #x56)) #x12)
(etst (ripemd-byte*-2-word 2 (list #x12 #x34 #x56)) #x3412)
(etst (ripemd-byte*-2-word 3 (list #x12 #x34 #x56)) #x563412)
(etst (ripemd-byte*-2-word 4 (list #x12 #x34 #x56)) #x563412)

(deff ripemd-byte*-2-word* (n x)
 (:when (= n 0) nil)
 (cons
  (ripemd-byte*-2-word 4 x)
  (ripemd-byte*-2-word* (- n 1) (nthcdr 4 x))))

(etst (ripemd-byte*-2-word* 0 (list #x10 #x32 #x54 #x76 #x98)) nil)
(etst (ripemd-byte*-2-word* 1 (list #x10 #x32 #x54 #x76 #x98)) '(#x76543210))
(etst (ripemd-byte*-2-word* 2 (list #x10 #x32 #x54 #x76 #x98))
 '(#x76543210 #x00000098))

(deff ripemd-byte*-2-word** (x result)
 (:when (null x) result)
 (ripemd-byte*-2-word** (nthcdr 64 x)
  (cons (ripemd-byte*-2-word* 16 x) result)))

(etst
 (ripemd-byte*-2-word**
  '(#x11 #x22 #x33 #x44
    #x55 #x11 #x22 #x33
    #x44 #x55 #x11 #x22
    #x33 #x44 #x55 #x11

    #x22 #x33 #x44 #x55
    #x11 #x22 #x33 #x44
    #x55 #x11 #x22 #x33
    #x44 #x55 #x11 #x22

    #x33 #x44 #x55 #x11
    #x22 #x33 #x44 #x55
    #x11 #x22 #x33 #x44
    #x55 #x11 #x22 #x33

    #x44 #x55 #x11 #x22
    #x33 #x44 #x55 #x11
    #x22 #x33 #x44 #x55
    #x11 #x22 #x33 #x44


    #x55 #x11 #x22 #x33
    #x44 #x55 #x11 #x22
    #x33 #x44 #x55 #x11
    #x22 #x33 #x44 #x55

    #x11)
  nil)
 '((#x33221155 #x22115544 #x11554433 #x55443322
    #x00000011 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000)
   (#x44332211 #x33221155 #x22115544 #x11554433
    #x55443322 #x44332211 #x33221155 #x22115544
    #x11554433 #x55443322 #x44332211 #x33221155
    #x22115544 #x11554433 #x55443322 #x44332211)))

(deff ripemd-add-zero-block (x)
 (:when (equalp (last (car x) 2) '(0 0)) x)
 (cons (ripemd-byte*-2-word* 16 nil) x))

(etst
 (ripemd-add-zero-block
  '((#x33221155 #x22115544 #x11554433 #x55443322
     #x00000011 #x00000000 #x00000000 #x00000000
     #x00000000 #x00000000 #x00000000 #x00000000
     #x00000000 #x00000000 #x00000000 #x00000000)))
 '((#x33221155 #x22115544 #x11554433 #x55443322
    #x00000011 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000)))
(etst
 (ripemd-add-zero-block
  '((#x33221155 #x22115544 #x11554433 #x55443322
     #x00000011 #x00000000 #x00000000 #x00000000
     #x00000000 #x00000000 #x00000000 #x00000000
     #x00000000 #x00000000 #x00000010 #x00000000)))
 '((#x00000000 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000)
   (#x33221155 #x22115544 #x11554433 #x55443322
    #x00000011 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000010 #x00000000)))

(deff ripemd-add-length (n x)
 (cons
  (append
   (butlast (car x) 2)
   (list (ripemd-mask n) (ripemd-mask (ash n -32))))
  (cdr x)))

(etst
 (ripemd-add-length #x1111222233334444
  '((#x33221155 #x22115544 #x11554433 #x55443322
     #x00000011 #x00000000 #x00000000 #x00000000
     #x00000000 #x00000000 #x00000000 #x00000000
     #x00000000 #x00000000 #x00000000 #x00000000)))
 '((#x33221155 #x22115544 #x11554433 #x55443322
    #x00000011 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x33334444 #x11112222)))

(deff ripemd-pad (x)
 (reverse
  (ripemd-add-length (* 8 (length x))
   (ripemd-add-zero-block
    (ripemd-byte*-2-word** (append x '(128)) nil)))))

(etst
 (ripemd-pad
  '(#x11 #x22 #x33 #x44
    #x55 #x11 #x22 #x33
    #x44 #x55 #x11 #x22
    #x33 #x44 #x55 #x11

    #x22 #x33 #x44 #x55
    #x11 #x22 #x33 #x44
    #x55 #x11 #x22 #x33
    #x44 #x55 #x11 #x22

    #x33 #x44 #x55 #x11
    #x22 #x33 #x44 #x55
    #x11 #x22 #x33 #x44
    #x55 #x11 #x22 #x33

    #x44 #x55 #x11 #x22
    #x33 #x44 #x55 #x11
    #x22 #x33 #x44 #x55
    #x11 #x22 #x33 #x44


    #x55 #x11 #x22 #x33
    #x44 #x55 #x11 #x22
    #x33 #x44 #x55 #x11
    #x22 #x33 #x44 #x55

    #x11))
 `((#x44332211 #x33221155 #x22115544 #x11554433
    #x55443322 #x44332211 #x33221155 #x22115544
    #x11554433 #x55443322 #x44332211 #x33221155
    #x22115544 #x11554433 #x55443322 #x44332211)
   (#x33221155 #x22115544 #x11554433 #x55443322
    #x00008011 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 #x00000000 #x00000000
    #x00000000 #x00000000 ,(* 8 81)  #x00000000)))

(deff ripemd-line-1 (j words a b c d e)
 (:when (= j 80) (list a b c d e))
 (ripemd-line-1
  (+ j 1)
  words
  e
  (ripemd-sum e
   (ripemd-rol (ripemd-s j)
    (ripemd-sum a
     (ripemd-sum (ripemd-k j)
      (ripemd-sum
       (ripemd-f j b c d)
       (aref words (ripemd-r j)))))))
  b
  (ripemd-rol 10 c)
  d))

(deff ripemd-line-2 (j words a b c d e)
 (:when (= j 80) (list a b c d e))
 (ripemd-line-2
  (+ j 1)
  words
  e
  (ripemd-sum e
   (ripemd-rol (ripemd-s-prime j)
    (ripemd-sum a
     (ripemd-sum (ripemd-k-prime j)
      (ripemd-sum
       (ripemd-f (- 79 j) b c d)
       (aref words (ripemd-r-prime j)))))))
  b
  (ripemd-rol 10 c)
  d))

(deff ripemd-rotate-list-left (n list)
 (append (subseq list n) (subseq list 0 n)))

(etst (ripemd-rotate-list-left 3 '(1 2 3 4 5)) '(4 5 1 2 3))
(etst (ripemd-rotate-list-left 0 '(1 2 3 4 5)) '(1 2 3 4 5))

(deff ripemd-sum-list (a b)
 (:when (null a) nil)
 (cons (ripemd-sum (car a) (car b)) (ripemd-sum-list (cdr a) (cdr b))))

(etst (ripemd-sum-list '(10 100) '(1000 #xFFFFFFFF)) '(1010 99))

(deff ripemd-combine-hash (a b c)
 (ripemd-sum-list
  (ripemd-rotate-list-left 1 a)
  (ripemd-sum-list
   (ripemd-rotate-list-left 2 b)
   (ripemd-rotate-list-left 3 c))))

(etst
 (ripemd-combine-hash
  '(#x0 #x1 #x2 #x3 #x4)
  '(#x0 #x0 #x0 #x0 #x0)
  '(#x0 #x0 #x0 #x0 #x0))
 '(#x1 #x2 #x3 #x4 #x0))
(etst
 (ripemd-combine-hash
  '(#x0 #x0 #x0 #x0 #x0)
  '(#xA #xB #xC #xD #xE)
  '(#x0 #x0 #x0 #x0 #x0))
 '(#xC #xD #xE #xA #xB))
(etst
 (ripemd-combine-hash
  '(#x0 #x0 #x0 #x0 #x0)
  '(#x0 #x0 #x0 #x0 #x0)
  '(#xA #xB #xC #xD #xE))
 '(#xD #xE #xA #xB #xC))

(deff ripemd-steps (file hash)
 (:when (null file) hash)
 (:let words (list-2-vector (car file)))
 (ripemd-steps (cdr file)
  (ripemd-combine-hash hash
   (apply 'ripemd-line-1 0 words hash)
   (apply 'ripemd-line-2 0 words hash))))

(deff word-2-byte* (n word)
 (:when (= n 0) nil)
 (cons (logand #xFF word) (word-2-byte* (- n 1) (ash word -8))))

(etst (word-2-byte* 4 #x12345678) '(#x78 #x56 #x34 #x12))

(deff word*-2-byte* (list)
 (:when (null list) nil)
 (append
  (word-2-byte* 4 (car list))
  (word*-2-byte* (cdr list))))

(etst
 (word*-2-byte* '(#x11223344 #x55667788))
 '(#x44 #x33 #x22 #x11 #x88 #x77 #x66 #x55))

(deff ripemd (x)
 (word*-2-byte* (ripemd-steps (ripemd-pad x) ripemd-init-hash)))

(deff string-2-ripemd (x)
 (ripemd
  (map 'list 'char-code
   (coerce x 'list))))

(etst
 (string-2-ripemd "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ123")
 '(#x71 #xe0 #xb6 #x23 #x22
   #xd3 #xa0 #x31 #xc0 #xfb
   #x62 #x1b #xe4 #xd9 #x07
   #x73 #xb7 #xe6 #x6b #xb2))
(etst
 (string-2-ripemd "")
 '(#x9c #x11 #x85 #xa5 #xc5
   #xe9 #xfc #x54 #x61 #x28
   #x08 #x97 #x7e #xe8 #xf5
   #x48 #xb2 #x25 #x8d #x31))
(etst
 (string-2-ripemd "a")
 '(#x0b #xdc #x9d #x2d #x25
   #x6b #x3e #xe9 #xda #xae
   #x34 #x7b #xe6 #xf4 #xdc
   #x83 #x5a #x46 #x7f #xfe))
(etst
 (string-2-ripemd
  "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789")
 '(#xb0 #xe2 #x0b #x6e #x31
   #x16 #x64 #x02 #x86 #xed
   #x3a #x87 #xa5 #x71 #x30
   #x79 #xb2 #x1f #x51 #x89))

(deff print-hex-list (x)
 (when x
  (format t "~2x" (car x))
  (print-hex-list (cdr x))))

(deff print-ripemd
 (&optional (x "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ123"))
 (print-hex-list (string-2-ripemd x)))

