#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb System
=============================================
File system interface
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Read file
=============================================

(slurp filename) is a Perl'ish file reading function that reads an entire file and returns it as a list of cardinals. Each cardinal is in the range 0..255 and represents one byte from the file. The filename can be given as a cardinal tree (ct).

|#

(deff slurp (filename)
 (:let filename (ct2string filename))
 (:when (directory (slash filename)) :directory)
 (with-open-file
  (stream filename
   :direction :input
   :element-type 'unsigned-byte
   :if-does-not-exist nil)
  (if (null stream) :does-not-exist
   (slurp1 stream nil))))

(deff slurp1 (stream result)
 (:let card (read-byte stream nil nil))
 (:when (null card) (reverse result))
 (:let result (cons card result))
 (slurp1 stream result))

(deff path2vector (path)
 (:let path (ct2string path))
 (:when (directory (slash path)) :directory)
 (with-open-file
  (stream path
   :direction :input
   :element-type 'unsigned-byte
   :if-does-not-exist nil)
  (path2vector1 stream)))

(deff path2vector1 (stream)
 (:when (null stream) :does-not-exist)
 (:let length (file-length stream))
 (:when (null length) (card*2vector (slurp1 stream nil)))
 (:let vector (make-vector length))
 (dotimes (i length) (setf (aref vector i) (read-byte stream)))
 vector)

#|
=============================================
Convert to reference
=============================================

(vector2ref vector) extracts the reference from the given Logiweb page
Returns zero if the RIPEMD-160 code does not match the contents.

(stream2ref stream) and (file2ref filename) call vector2ref on the vector read from the given stream or file.
|#

(deff file2ref (filename &optional ignore)
 (vector2ref (slurp filename) ignore))

(deff stream2ref (stream &optional ignore)
 (vector2ref (slurp1 stream nil) ignore))

(deff vector2septets (vector)
 (:let position (position 128 vector :test '>))
 (:when (null position) (format t "End of file in cardinal~%") (raise))
 (:let position (+ position 1))
 (:let septets (subseq vector 0 position))
 (:let vector (subseq vector position))
 (cons septets vector))

(etst (vector2septets '(129 128 127 126 125)) '((129 128 127) . (126 125)))
(etst (vector2septets '(129 128 127)) '((129 128 127) . ()))
(xtst (vector2septets '(129 128)))

(deff vector2ripemd (vector n)
 (:let v1 (nthcdr n vector))
 (:when (null v1) (format t "End of file in Ripemd code~%") (raise))
 (cons (subseq vector 0 n) v1))

(etst (vector2ripemd '(11 12 13 14 15) 2) '((11 12) . (13 14 15)))
(etst (vector2ripemd '(11 12 13 14 15) 4) '((11 12 13 14) . (15)))
(xtst (vector2ripemd '(11 12 13 14 15) 5))
(xtst (vector2ripemd '(11 12 13 14 15) 6))

(deff valid-id (id)
 (:when (null id) nil)
 (:when (equalp id '(1)) t)
 (:let (card . id) id)
 (:when (unequal card 129) nil)
 (valid-id1 id))

(deff valid-id1 (id)
 (:when (null id) nil)
 (:when (equalp id '(0)) t)
 (:let (card . id) id)
 (:when (unequal card 128) nil)
 (valid-id1 id))

(ntst (valid-id '(0)))
(ttst (valid-id '(1)))
(ntst (valid-id '(2)))
(ntst (valid-id '(128)))
(ntst (valid-id '(129)))
(ntst (valid-id '(130)))
(ttst (valid-id '(129 0)))
(ntst (valid-id '(129 1)))
(ntst (valid-id '(129 128)))
(ntst (valid-id '(129 129)))
(ttst (valid-id '(129 128 0)))

(deff file-septet*2card (septet*)
 (:when (null septet*) 0)
 (:let (septet . septet*) septet*)
 (+ (mod septet 128) (* 128 (file-septet*2card septet*))))

(etst   0 (file-septet*2card '(0)))
(etst   1 (file-septet*2card '(1)))
(etst   2 (file-septet*2card '(2)))
(etst   0 (file-septet*2card '(128)))
(etst   1 (file-septet*2card '(129)))
(etst   2 (file-septet*2card '(130)))
(etst   1 (file-septet*2card '(129 0)))
(etst 129 (file-septet*2card '(129 1)))
(etst   1 (file-septet*2card '(129 128)))
(etst 129 (file-septet*2card '(129 129)))
(etst   1 (file-septet*2card '(129 128 0)))

(deff vector2ref (vector &optional ignore)
 (:let (length . vector) (vector2septets vector))
 (:let length (file-septet*2card length))
 (:let (id . vector) (vector2septets vector))
 (:unless (valid-id id) (format t "Invalid id: ~s~%" id) (raise))
 (:let (ripemd . vector) (vector2ripemd vector 20))
 (:when (and (null ignore) (unequal ripemd (ripemd vector)))
  (format t "Ripemd mismatch~%") (raise))
 (:let (mantissa . vector) (vector2septets vector))
 (:let (exponent . :vector) (vector2septets vector))
 (:let ref (append id ripemd mantissa exponent))
 (:when (equalp length (length ref)) ref)
 (format t "Malformed reference~%") (raise))

(xtst (vector2ref '(0)))
(xtst (vector2ref '(1)))
(xtst (vector2ref '(1 1)))
(etst
 '(  1
    85  95  21 132  76
   160 148  46 137 199
   106 227 218   7 215
   132 142 205 121  79
   128 127   6        )
 (vector2ref
  '( 24
      1
     85  95  21 132  76
    160 148  46 137 199
    106 227 218   7 215
    132 142 205 121  79
    128 127 6 1        )))
(xtst
 (vector2ref
  '( 24
      1
     85  95  21 132  76
    160 148  46 137 199
    106 227 218   8 215
    132 142 205 121  79
    128 127 6 1        )))
(xtst
 (vector2ref
  '( 23
      1
     88  37 210  16 131
    245 218 198 114  96
    205 173  28  32 161
    154 144 190 154  82
    128 127)))

#|
=============================================
Read bibliography part of file
=============================================
Fast routine for reading the bibliography part of a file and returning it as a vector.
|#

(defc *path2bib* nil)

(deff path2bib (path)
 (:let path (ct2string path))
 (:when (directory (slash path)) :directory)
 (with-open-file
  (stream path
   :direction :input
   :element-type 'unsigned-byte
   :if-does-not-exist nil)
  (path2bib1 stream)))

(deff path2bib1 (stream)
 (:when (null stream) :does-not-exist)
 (setq *path2bib* nil)
 (:catch () :malformed)
 (path2bib2 stream))

(deff path2bib2 (stream)
 (:let id (path2bib-card stream))
 (:when (equalp id 0) (card*2vector (reverse *path2bib*)))
 (:when (unequal id 1) (raise))
 (dotimes (n 20) (path2bib-byte stream))
 (path2bib-card stream)
 (path2bib-card stream)
 (path2bib2 stream))

(deff path2bib-card (stream)
 (:let byte (read-byte stream))
 (:when (null byte) (raise))
 (push byte *path2bib*)
 (:when (<debug byte 128) byte)
 (+ byte -128 (* 128 (path2bib-card stream))))

(deff path2bib-byte (stream)
 (:let byte (read-byte stream))
 (:when (null byte) (raise))
 (push byte *path2bib*))

#|
=============================================
Write cardinal tree (ct) to file
=============================================
(ct2file filename ct) writes the given ct to the given file.

Lurking bug: ct2file is able to write strings, byte-vectors, characters, and integers to the file. But an integer may either be interpreted as a character or a string according to two different representation schemes used. The two representation schemes have *one* common value: the cardinal 1 represents a Unicode Start Of Header when interpreted as a character and the empty string when interpreted as a string. In the long run, there may be a need to split ct2file into vt2file (vector-tree to file) and ct2file (cardinal-tree to file) where the former interprets 1 as the empty string. But the entire system probably needs to be overhawled to overcome this problem. The function below interprets the cardinal 1 as a Unicode Start Of Header in the hope that it will be ignored by external systems if it should have been interpreted as an empty string.
|#

(deff ct2file (filename ct)
 (:let filename (ct2string filename))
 (when (verbose '> 1) (format t "Writing ~s~%" filename))
 (ensure-directories-exist filename)
 (with-open-file
  (stream filename
   :direction :output
   :element-type 'unsigned-byte
   :if-exists :supersede)
  (ct2stream stream ct)))

(deff ct2stream (stream ct)
 (:when (consp ct)
  (ct2stream stream (car ct))
  (ct2stream stream (cdr ct)))
 (:when (stringp ct)
  (dotimes (n (length ct)) (write-byte (char-code (aref ct n)) stream)))
 (:when (arrayp ct)
  (dotimes (n (length ct)) (write-byte (aref ct n) stream)))
 (:when (characterp ct)
  (write-byte (char-code ct) stream))
 (:when (not (integerp ct)) nil)
 (:when (<debug ct 256) (write-byte ct stream))
 (:when (equalp (mod (integer-length ct) 8) 1)
  (:let ct (card2vector ct))
  (dotimes (n (length ct)) (write-byte (aref ct n) stream)))
 (format t "Warning: ~d translated to '###'~%" ct)
 (write-byte (char-code #\#) stream)
 (write-byte (char-code #\#) stream)
 (write-byte (char-code #\#) stream))




