#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
Functions for optimized reduction
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Untagged pairs, truth, and falsehood
=============================================

(mtag-pair arg head tail) : map x map x map -> rnf
Applies the pair of the given head and tail to the given argument.

(map-pair head tail) : map x map -> rnf
Constructs the pair of the given maps.

map-t represents truth.

map-f represents falsehood.

(map-head pair) : map -> rnf
Returns the first component of the given pair.

(map-tail pair) : map -> rnf
Returns the first component of the given pair.

(map-equal x y) : map x map -> bool
Tests x and y for equality using map-head and map-tail.
|#

(deff mtag-pair (arg head tail)
 (map2rnf (ifnil (map2rnf arg) head tail)))

(deff map-pair (head tail)
 (list 'mtag-pair head tail))

(defc map-t nil)

(defc map-f (map-pair map-t map-t))

(deff map-head (map)
 (map-apply map map-t))

(deff map-tail (map)
 (map-apply map map-f))

(deff map-equal (x y)
 (:let x (map2rnf x))
 (:let y (map2rnf y))
 (:when (null x) (null y))
 (:when (null y) (null x))
 (:when (not (map-equal (map-head x) (map-head y))) nil)
 (:when (not (map-equal (map-tail x) (map-tail y))) nil)
 t)

(ntst (map-apply (map-pair map-t map-t) map-t))
(ntst (map-apply (map-pair map-t map-t) map-f))
(ntst (map-apply (map-pair map-t map-f) map-t))
(test (map-apply (map-pair map-t map-f) map-f))
(test (map-apply (map-pair map-f map-t) map-t))
(ntst (map-apply (map-pair map-f map-t) map-f))
(test (map-apply (map-pair map-f map-f) map-t))
(test (map-apply (map-pair map-f map-f) map-f))

(test (map-pair map-t map-t))
(test (map-pair map-t map-f))
(test (map-pair map-f map-t))
(test (map-pair map-f map-f))

(ntst (map-head (map-pair map-t map-t)))
(ntst (map-tail (map-pair map-t map-t)))
(ntst (map-head (map-pair map-t map-f)))
(test (map-tail (map-pair map-t map-f)))
(test (map-head (map-pair map-f map-t)))
(ntst (map-tail (map-pair map-f map-t)))
(test (map-head (map-pair map-f map-f)))
(test (map-tail (map-pair map-f map-f)))

(ttst (map-equal map-t map-t))
(ntst (map-equal map-t map-f))
(ntst (map-equal map-f map-t))
(ttst (map-equal map-f map-f))

(ttst (map-equal (map-pair map-t map-t) (map-pair map-t map-t)))
(ntst (map-equal (map-pair map-t map-t) (map-pair map-t map-f)))
(ntst (map-equal (map-pair map-t map-t) (map-pair map-f map-t)))
(ntst (map-equal (map-pair map-t map-t) (map-pair map-f map-f)))
(ntst (map-equal (map-pair map-t map-f) (map-pair map-t map-t)))
(ttst (map-equal (map-pair map-t map-f) (map-pair map-t map-f)))
(ntst (map-equal (map-pair map-t map-f) (map-pair map-f map-t)))
(ntst (map-equal (map-pair map-t map-f) (map-pair map-f map-f)))
(ntst (map-equal (map-pair map-f map-t) (map-pair map-t map-t)))
(ntst (map-equal (map-pair map-f map-t) (map-pair map-t map-f)))
(ttst (map-equal (map-pair map-f map-t) (map-pair map-f map-t)))
(ntst (map-equal (map-pair map-f map-t) (map-pair map-f map-f)))
(ntst (map-equal (map-pair map-f map-f) (map-pair map-t map-t)))
(ntst (map-equal (map-pair map-f map-f) (map-pair map-t map-f)))
(ntst (map-equal (map-pair map-f map-f) (map-pair map-f map-t)))
(ttst (map-equal (map-pair map-f map-f) (map-pair map-f map-f)))

#|
=============================================
The *spy* variable
=============================================
The *spy* variable is set by the spy function spy(x) which sets *spy* to x and returns 'true'. The spy variable can be printed out from the main-event-loop by a call to (spy) or it can be promptet for by the 'spy' command-line argument. Furthermore, (spy) is called by optimized bottom.
|#

(defc *spy* nil)
(defc *spy-state* nil)
(defc *spypath* nil)
(defc *trace* nil)
(defc *spy-depth* 10)
(defc *spy-length* 20)

(deff spy (&key (depth *spy-depth*) (length *spy-length*))
 (:let spy (spy-access (reverse *spypath*) *spy*))
 (spy1 (default -1 depth) length spy *spy-state*)
 (values))

(deff spy-access (path spy)
 (:when (atom path) spy)
 (:let (fct . path) path)
 (spy-access path (spy-access1 fct spy)))

(deff spy-access1 (fct spy)
 (:when (atom fct) (spy-access2 fct spy))
 (:let (fct . args) fct)
 (apply fct spy args))

(deff spy-access2 (fct spy)
 (:when (integerp fct) (nth fct spy))
 (funcall fct spy))

(defmacro spycd (&rest fct)
 `(setq *spypath* (append ',fct *spypath*)))

(defmacro spyls (&rest fct)
 `(spyls1 ',fct))

(deff spyls1 (fct)
 (:let spy (spy-access (revappend *spypath* fct) *spy*))
 (spy1 *spy-depth* *spy-length* spy *spy-state*)
 (values))

(deff spy-set-depth (depth)
 (setq *spy-depth* depth))

(deff spy-set-length (length)
 (setq *spy-length* length))

(deff spy/ ()
 (setq *spypath* nil))

(deff spy.. ()
 (pop *spypath*)
 *spypath*)

(deff spypwd ()
 *spypath*)

; looks-like-tree not used anymore. Functionality moved to
; c-looks-like-tree in codify.lisp
(deff looks-like-tree (tree)
 (:when (atom tree) nil)
 (:let ((ref idx) . tree*) tree)
 (:when (equalp ref 0) (and (null tree*) (or (integerp idx) (arrayp idx))))
 (:when (not (integerp ref)) nil)
 (:when (null (aget *spy-state* (id-cache) ref)) nil)
 (looks-like-tree* tree*))

(deff looks-like-tree* (tree)
 (:when (null tree) t)
 (:when (atom tree) nil)
 (:let (tree . tree*) tree)
 (and (looks-like-tree tree) (looks-like-tree* tree*)))

#|
depth indicates how deep the function should look into the cons/tree structure.

A depth of -1 means infinity.

length indicates how much of a string should be printed.

A length of NIL means infinity.
|#

(deff o-print-tree (indent depth length val state)
 (funcall 'c-print-tree indent depth length val state))

(deff spy1 (depth length val state)
 (o-print-tree 1 depth length val state))

(deff trace1 (d s v)
 (:when (= d 0) "...")
 (:let d (- d 1))
 (:when (consp v) (cons (trace1 d s (head v)) (trace1 d s (tail v))))
 (:when (arrayp v) (vector2string (safe-subseq v 0 s)))
 (:when (tagmap-p v) (make-tagmap :map "..."))
 (:when (object-p v)
  (:let ref (object-ref v))
  (:let idx (object-idx v))
  (:let val (trace1 d s (object-val v)))
  (make-object :ref ref :idx idx :val val))
 v)

#|
=============================================
Optimized bottom
=============================================

(ttag-bottom env) : env -> rnf
Never returns.

Do not abuse optimized bottom: It would be *very* *very* tempting to introduce a "bottom-ness" test which returns T when applied to optimized bottom and F otherwise. But such a function would ruin the principle of "substitution of equals" and would kill referential transparency which are the very reasons for preferring functional programming over other kinds of programming. Use exceptions and exception catching instead.
|#

(deff map-bottom ()
 (format t "Evaluation of 'bottom' takes forever~%")
 (format t "Value of *spy*:~%")
 (spy)
 (error "You are now at the End of the Universe"))

(deff ttag-bottom (:env)
 (map-bottom))

(defc term-b '(ttag-bottom))

(defc map-b (list :A term-b nil))

;(test (map2rnf map-b))
(test (map-pair map-b map-b))
;(test (map-head (map-pair map-b map-b)))

#|
=============================================
Delayed application
=============================================
(ttag-delay-apply) : env -> rnf
Applies the deBruijn variable 0 to deBruijn variable 1.

(delay-apply f x) : map x map -> map
Applies f to x without reducing to rnf.

(delay-tail x) : map -> map
Applies x to map-f without reducing to rnf
|#

(deff ttag-delay-apply (env)
 (:let (f x) env)
 (map-apply f x))

(deff delay-apply (f x)
 (term2closure '(ttag-delay-apply) (list f x)))

(deff delay-tail (x)
 (delay-apply x map-f))

(ntst (map2rnf (delay-apply (map-pair map-t map-t) map-t)))
(ntst (map2rnf (delay-apply (map-pair map-t map-t) map-f)))
(ntst (map2rnf (delay-apply (map-pair map-t map-f) map-t)))
(test (map2rnf (delay-apply (map-pair map-t map-f) map-f)))
(test (map2rnf (delay-apply (map-pair map-f map-t) map-t)))
(ntst (map2rnf (delay-apply (map-pair map-f map-t) map-f)))
(test (map2rnf (delay-apply (map-pair map-f map-f) map-t)))
(test (map2rnf (delay-apply (map-pair map-f map-f) map-f)))

(test (delay-apply (map-pair map-t map-b) map-f))
;(test (map2rnf (delay-apply (map-pair map-t map-b) map-f)))

#|
=============================================
Untagged cardinals
=============================================

(card2rnf card) : card -> rnf
Converts the given cardinal to a map.

(rnf2card map) : rnf -> card
Converts the given map to a cardinal.
|#

(deff bool2map (x) (if x map-t map-f))

(deff card2rnf (card)
 (:when (= card 0) map-t)
 (map-pair (bool2map (evenp card)) (card2rnf (ash card -1))))

(deff rnf2card (map)
 (:let rnf (map2rnf map))
 (:when (null rnf) 0)
 (+ (if (null (map-head rnf)) 0 1) (ash (rnf2card (map-tail rnf)) 1)))

(defc map-0 (card2rnf 0))
(defc map-1 (card2rnf 1))
(defc map-2 (card2rnf 2))
(defc map-3 (card2rnf 3))
(defc map-4 (card2rnf 4))

(etst (rnf2card map-t) 0)
(etst (rnf2card (map-pair map-t map-t)) 0)
(etst (rnf2card (map-pair map-f map-t)) 1)
(etst (rnf2card (map-pair map-t (map-pair map-t map-t))) 0)
(etst (rnf2card (map-pair map-f (map-pair map-t map-t))) 1)
(etst (rnf2card (map-pair map-t (map-pair map-f map-t))) 2)
(etst (rnf2card (map-pair map-f (map-pair map-f map-t))) 3)

(ntst (card2rnf 0))
(test (card2rnf 1))
(test (map-head (card2rnf 1)))
(ntst (map-tail (card2rnf 1)))
(test (card2rnf 2))
(ntst (map-head (card2rnf 2)))
(test (map-head (map-tail (card2rnf 2))))
(ntst (map-tail (map-tail (card2rnf 2))))

(etst (rnf2card map-0) 0)
(etst (rnf2card map-1) 1)
(etst (rnf2card map-2) 2)
(etst (rnf2card map-3) 3)
(etst (rnf2card map-4) 4)

(ttst (map-equal map-0 map-0))
(ntst (map-equal map-0 map-1))
(ntst (map-equal map-0 map-2))
(ntst (map-equal map-0 map-3))
(ntst (map-equal map-0 map-4))
(ntst (map-equal map-1 map-0))
(ttst (map-equal map-1 map-1))
(ntst (map-equal map-1 map-2))
(ntst (map-equal map-1 map-3))
(ntst (map-equal map-1 map-4))
(ntst (map-equal map-2 map-0))
(ntst (map-equal map-2 map-1))
(ttst (map-equal map-2 map-2))
(ntst (map-equal map-2 map-3))
(ntst (map-equal map-2 map-4))
(ntst (map-equal map-3 map-0))
(ntst (map-equal map-3 map-1))
(ntst (map-equal map-3 map-2))
(ttst (map-equal map-3 map-3))
(ntst (map-equal map-3 map-4))
(ntst (map-equal map-4 map-0))
(ntst (map-equal map-4 map-1))
(ntst (map-equal map-4 map-2))
(ntst (map-equal map-4 map-3))
(ttst (map-equal map-4 map-4))



#|
=============================================
TAGGED VALUES
=============================================
Tagged values (tv's) are Lisp structures built up from nil, :false, integers, vectors, cons, fct-structures, tagmap-structures and object-structures. fct-structures, tagmap-structures and object-structures are introduced later.

Tagged maps (tm's) are maps which represent tagged values. The representation is thus:
  Tagged value (tv)         Tagged map (tm)
  nil                       map-t
  :false                    map-f
  int                       (map-0::map-1)::(sign::magnitude)
  vector                    (map-0::map-1)::(map-t::magniture)
  pair                      (map-0::map-2)::(head::tail)
  tagmap                    (map-0::map-4)::tagmap
  fct                       (map-0::map-4)::fct
  object                    (ref::idx)::val

Maps of form (map-0::map-1)::val represent integers; vectors (i.e. byte vectors) are thought of as an alternative way of representing integers.

Maps of form (map-0::map-4)::val represent arbitrary maps; fct-structures are thought of as an alternative way of representing maps.

Maps of form (map-0::map-3)::val represent exceptions. Functions which convert tm's to tv's can raise exceptions. Functions which convert tv's to tm's cannot catch exceptions.
|#

#|
=============================================
Definitions of truth and falsehood
=============================================
|#

(defconstant true nil)
(defconstant false :false)
(defmacro iftrue (x y z) `(if ,x ,z ,y))
(defmacro true (x) `(null ,x))

#|
=============================================
Declaration of structures
=============================================
|#

(defstruct tagmap    map)
(defstruct fctstruct data)
(defstruct object    ref idx val)

(deff fct-p (x)
 (fctstruct-p x))

#|
=============================================
Conversion of tv's to tm's
=============================================
(mtag-tv arg tv) : map x tv -> rnf
Applies the tv to the given argument.

(map-tv tv) : tv -> rnf
Converts the given tv to a tm.

(ttag-tv env tv) : env x tv -> rnf
Applies the given tv to the given environment.

(term-tv tv) : tv -> term
Converts the given tv to a term.

(tv2tm tv) : tv -> rnf
Converts the given tv to a tm like (map-tv tv) does, but does so more eagerly: (tv2tm tv) converts the root of the tv to a map, delaying conversion of sub-tv's, if any. (map-tv tv) delays the conversion as much as possible.

We have (tm2tv (map-tv tv))=tv and (tm2tv (tv2tm tv))=tv but the former is slightly faster because map-tv is is lazy as possible and because tm2tv checks whether or not its argument has been constructed by map-tv.
|#

(deff mtag-tv (arg tv)
 (map-apply (tv2tm tv) arg))

(deff map-tv (tv)
 (:when (true tv) true)
 (list 'mtag-tv tv))

(deff ttag-tv (:env tv)
 (map-tv tv))

(deff term-tv (tv)
 (list 'ttag-tv tv))

(deff tv2tm (tv)
 (:when (equalp tv false) map-f)
 (:when (integerp tv) (int2tm tv))
 (:when (arrayp tv) (int2tm (vector2card tv)))
 (:when (consp tv) (pair2tm tv))
 (:when (fct-p tv) (fct2tm tv))
 (:when (tagmap-p tv) (tagmap2tm tv))
 (:when (object-p tv) (object2tm tv))
 (error "Internal error: Unknown tagged value: ~s" tv))

(deff int2tm (int)
 (:let tag (map-pair map-0 map-1))
 (:let sign (bool2map (>= int 0)))
 (:let mag (card2rnf (abs int)))
 (map-pair tag (map-pair sign mag)))

(deff pair2tm (pair)
 (:let (head . tail) pair)
 (:let tag (map-pair map-0 map-2))
 (:let head (map-tv head))
 (:let tail (map-tv tail))
 (map-pair tag (map-pair head tail)))

(deff fct2tm (fct)
 (:let tag (map-pair map-0 map-4))
 (:let map (fct2map fct))
 (map-pair tag map))

(deff tagmap2tm (tagmap)
 (:let tag (map-pair map-0 map-4))
 (:let map (tagmap-map tagmap))
 (map-pair tag map))

(deff object2tm (object)
 (:let ref (object-ref object))
 (:let idx (object-idx object))
 (:let val (object-val object))
 (:let ref (card2rnf ref))
 (:let idx (card2rnf idx))
 (:let val (map-tv val))
 (map-pair (map-pair ref idx) val))

(ttst (map-equal (term-eval (term-tv true)) map-t))
(ttst (map-equal (term-eval (term-tv false)) map-f))
(ttst (map-equal (term-eval (term-tv 2))
 (map-pair (map-pair map-0 map-1) (map-pair map-t map-2))))
(ttst (map-equal (term-eval (term-tv -2))
 (map-pair (map-pair map-0 map-1) (map-pair map-f map-2))))
(ttst (map-equal (term-eval (term-tv 0))
 (map-pair (map-pair map-0 map-1) (map-pair map-t map-0))))
(ttst (map-equal (term-eval (term-tv #(3 2)))
 (map-pair (map-pair map-0 map-1) (map-pair map-t (card2rnf #x010203)))))
(ttst (map-equal (term-eval (term-tv (cons true false)))
 (map-pair (map-pair map-0 map-2) (map-pair map-t map-f))))
(ttst (map-equal (term-eval (term-tv (make-tagmap :map map-2)))
 (map-pair (map-pair map-0 map-4) map-2)))
(ttst (map-equal (term-eval (term-tv (make-object :ref 1 :idx 2 :val false)))
 (map-pair (map-pair map-1 map-2) map-f)))

#|
=============================================
Conversion of tm's to tv's
=============================================
(tm2tv tm) converts the given tagged map (tm) to a tagged value (tv).

Conversion is done eagerly, left to right, depth first, but stops if an exception is encountered in which case tm2tv raises the exception.

Exceptions take an argument. If conversion of the argument itself raises an exception, then the latter exception is raised. As an example,
  (raise (pair (raise 1) (raise 2)))
gives the same result as
  (raise 1)

(tm*2tv* tm*) converts the given list of tagged maps to a list of tagged values. Conversion is done left to right but stops if an exception is encountered in which case tm*2tv* raises the exception.
|#

(deff tm*2tv* (tm*)
 (:when (atom tm*) true)
 (:let (tm . tm*) tm*)
 (:let tv (tm2tv tm))
 (:let tv* (tm*2tv* tm*))
 (cons tv tv*))

(deff tm2tv (tm)
 (:let tm (map2rnf tm))
 (:when (true tm) true)
 (:let (tag tv) tm)
 (:when (equalp tag 'mtag-tv) tv)
 (:let tag (map-head tm))
 (:let ref (rnf2card (map-head tag)))
 (:let idx (rnf2card (map-tail tag)))
 (:when (unequal ref 0) (tm2object ref idx tm))
 (:when (equalp idx 0) false)
 (:when (equalp idx 1) (tm2int tm))
 (:when (equalp idx 2) (tm2cons tm))
 (:when (equalp idx 3) (tm2ex tm))
 (:when (equalp idx 4) (tm2tagmap tm))
 (tm2object ref idx tm))

(deff tm2int (tm)
 (:let val (map-tail tm))
 (:let mag (rnf2card (map-tail val)))
 (iftrue (map-head val) mag (- mag)))

(deff tm2cons (tm)
 (:let val (map-tail tm))
 (:let head (tm2tv (map-head val)))
 (:let tail (tm2tv (map-tail val)))
 (cons head tail))

(deff tm2ex (tm)
 (:let val (map-tail tm))
 (:let tv (tm2tv val))
 (raise tv))

(deff tm2tagmap (tm)
 (:let (mtag :head tail) tm)
 (:when (equalp mtag 'mtag-pair) (make-tagmap :map tail))
 (make-tagmap :map (delay-tail tm)))

(deff tm2object (ref idx tm)
 (:let val (map-tail tm))
 (make-object :ref ref :idx idx :val (tm2tv val)))

(ntst (tm2tv map-t))
(etst false (tm2tv map-f))
(etst false (tm2tv (map-pair (map-pair map-0 map-0) map-4)))
(etst 2 (tm2tv (map-pair (map-pair map-0 map-1) (map-pair map-t map-2))))
(etst -2 (tm2tv (map-pair (map-pair map-0 map-1) (map-pair map-f map-2))))
(etst 0 (tm2tv (map-pair (map-pair map-0 map-1) (map-pair map-t map-0))))
(etst 0 (tm2tv (map-pair (map-pair map-0 map-1) (map-pair map-f map-0))))
(etst (cons true false)
 (tm2tv (map-pair (map-pair map-0 map-2) (map-pair map-t map-f))))
(xtst (tm2tv (map-pair (map-pair map-0 map-3) (map-pair map-t map-f))))
(etst (make-tagmap :map map-b)
 (tm2tv (map-pair (map-pair map-0 map-4) map-b)))
(test (tm2tv (map-apply (term-eval term-k) (map-pair map-0 map-4))))
(ttst
 (map-equal
  (map-pair map-0 map-4)
  (tagmap-map (tm2tv (map-apply (term-eval term-k) (map-pair map-0 map-4))))))

(etst (make-object :ref 1 :idx 2 :val false)
 (tm2tv (map-pair (map-pair map-1 map-2) map-f)))

#|
=============================================
Big tagged values
=============================================
(term-big-tv prefix tv) is the same term as (term-tv tv) except that the given tv occurs directly in (term-tv tv) but indirectly as the symbol-value of a gensym in (term-big-tv tv). For that reason, (term-tv tv) is slightly faster to evaluate than (term-big-tv prefix tv), but *extremely* expensive to compile if the tv is big. The name of the gensym is constructed using the given prefix.
|#

(deff ttag-symbol (:env symbol)
 (map-tv (symbol-value symbol)))

(deff term-symbol (symbol)
 (list 'ttag-symbol symbol))

(deff value2gensym (prefix value)
 (:let gensym (gen-sym prefix))
 (setf (symbol-value gensym) value)
 gensym)

(deff term-big-tv (prefix value)
 (term-symbol (value2gensym prefix value)))

(ttst (map-equal (term-eval (term-big-tv "TEST-" true)) map-t))
(ttst (map-equal (term-eval (term-big-tv "TEST-" false)) map-f))
(ttst (map-equal (term-eval (term-big-tv "TEST-" 2))
 (map-pair (map-pair map-0 map-1) (map-pair map-t map-2))))
(ttst (map-equal (term-eval (term-big-tv "TEST-" -2))
 (map-pair (map-pair map-0 map-1) (map-pair map-f map-2))))
(ttst (map-equal (term-eval (term-big-tv "TEST-" 0))
 (map-pair (map-pair map-0 map-1) (map-pair map-t map-0))))
(ttst (map-equal (term-eval (term-big-tv "TEST-" #(3 2)))
 (map-pair (map-pair map-0 map-1) (map-pair map-t (card2rnf #x010203)))))
(ttst (map-equal (term-eval (term-big-tv "TEST-" (cons true false)))
 (map-pair (map-pair map-0 map-2) (map-pair map-t map-f))))
(ttst (map-equal (term-eval (term-big-tv "TEST-" (make-tagmap :map map-2)))
 (map-pair (map-pair map-0 map-4) map-2)))
(ttst (map-equal (term-eval (term-big-tv "TEST-" (make-object :ref 1 :idx 2 :val false)))
 (map-pair (map-pair map-1 map-2) map-f)))



#|
=============================================
FCT-STRUCTURES
=============================================
We shall represent operators by fct-structures.

(fct symbol type arity) defined below constructs an fct-structure.

An fct-structure represents an operator f(x_1,...,x_n) with the given arity.

When used as a map, the fct-structure represents the Curried operator \x_1:...\x_n:f(x_1,...,x_n).

The symbol may be

- one of the atoms ttag-apply, ttag-true, and ttag-if, which denote proclaimable computable constructs,

- a Lisp symbol which represents an introduced function, or

- a gensym, which represents a user defined function.

The type can be

- :lazy indicating that (symbol-function s) is such that (s . args) is a term in the sense of reduce.lisp.

- :eager indicating that (symbol-function s) operates on tagged values. An eager fct applied to an insufficient number of arguments returns a lambda and does not evaluate any of the arguments. An eager fct applied to a sufficient number of arguments evaluates its arguments from left to right until an argument raises an exception or all arguments are evaluated.

During eagerness analysis, the type can take on one further value:

- :unknown indicating that nothing is known about the type of the function.

The life cycle of fct-structures that represent user defined operators is thus: The fct-structure is born with a type of :unknown. If eagerness analysis proves that the associated construct is lazy then the type is changed to :lazy. After that all constructs that are still :unknown have their type changed to :eager.
|#

#|
=============================================
Making and destructuring fct-structures"
=============================================
|#

(deff fct (symbol type arity)
 (make-fctstruct :data (list symbol type arity)))

(deff thefct (fct)
 (fctstruct-data fct))

(deff no-fct (x)
 (not (fct-p x)))

(etst (list 1 2 3) (thefct (fct 1 2 3)))
(ttst (fct-p (fct 1 2 3)))
(ntst (fct-p 2))
(ntst (no-fct (fct 1 2 3)))
(ttst (no-fct 2))

#|
=============================================
Fct-structures of proclaimable constructs
=============================================
fct-apply, fct-if, and fct-true are fct-structures which represent the proclaimable concepts of application, selection, and truth, respectively.

Fct-structures can represent the value aspect of any construct except lambda constructs, quoting constructs, and constructs with no value definition. In codify.lisp these three cases are represented by the atoms 0, 1, and nil, respectively. To support the convention that 0 represents lambda abstraction and 1 represents quoting we define fct-lambda and fct-quote to represent 0 and 1, respectively.
|#

(defc fct-apply (fct 'ttag-apply :lazy 2))

(defc fct-if (fct 'ttag-if :lazy 3))

(defc fct-true (fct 'ttag-true :lazy 0))

(defc fct-lambda 0)

(defc fct-quote 1)

#|
=============================================
Application of fct-structures to terms
=============================================
(apply-fct fct term*) constructs the term fct(term_1,...,term_n).

(funcall-fct fct &rest term*) also constructs fct(term_1,...,term_n).
|#

(deff ttag-fct (env symbol &rest term*)
 (:catch (tv) (map-pair (map-pair map-0 map-3) (map-tv tv)))
 (:let map* (term*2closure* term* env))
 (:let tv* (tm*2tv* map*))
 (:let tv (apply symbol tv*))
 (:let map (map-tv tv))
 map)

(deff apply-fct (fct term*)
 (:let (symbol type arity) (thefct fct))
 (:when (unequal (length term*) arity) (error "Internal error: Arity mismatch"))
 (:when (equalp type :eager) (list* 'ttag-fct symbol term*))
 (:when (equalp type :lazy) (cons symbol term*))
 (error "Internal error: Unknown type of ~s" (thefct fct)))

#|
Never used. Was introduced for the sake of testing, but it is easier
to test apply-fct by testing tv2tm which calls apply-fct thus:
tv2tm -> fct2tm -> fct2map -> fct2term -> apply-fct
(deff funcall-fct (fct &rest term*)
 (apply-fct fct term*))
|#

#|
=============================================
Conversion of fct-structures to rnf
=============================================
(fct2term fct), (fct2map fct), and (fct2rnf fct) convert an fct to a term, map, and rnf, respectively, which represent the Curried function \x_1:...\x_n:fct(x_1,...,x_n).
|#

(deff varlist (n)
 (:when (= n 0) nil)
 (:let n (- n 1))
 (cons (term-var n) (varlist n)))

(etst (varlist 3) (list (term-var 2) (term-var 1) (term-var 0)))

(deff nlambda (n term)
 (:when (= n 0) term)
 (nlambda (- n 1) (term-lambda term)))

(etst (nlambda 3 (term-var 0))
 (term-lambda (term-lambda (term-lambda (term-var 0)))))

(deff fct2term (fct)
 (:let (:symbol :type arity) (thefct fct))
 (nlambda arity (apply-fct fct (varlist arity))))

(deff fct2map (fct)
 (term2closure (fct2term fct) nil))

(deff fct2rnf (fct)
 (term2rnf (fct2term fct) nil))

(ttst
 (map-equal (tv2tm 90)
  (map-apply
   (map-apply
    (map-tail (tv2tm (fct '- :eager 2)))
    (tv2tm 100))
   (tv2tm 10))))
(ntst
 (map-equal (tv2tm 91)
  (map-apply
   (map-apply
    (map-tail (tv2tm (fct '- :eager 2)))
    (tv2tm 100))
   (tv2tm 10))))
(ttst
 (map-equal (tv2tm -90)
  (map-apply
   (map-apply
    (map-tail (tv2tm (fct '- :eager 2)))
    (tv2tm 10))
   (tv2tm 100))))
(ttst
 (map-equal (map-pair (map-pair map-0 map-3) map-f)
  (map-apply (map-tail (tv2tm (fct 'raise :eager 1))) (tv2tm false))))

#|
=============================================
Registration functions
=============================================
(lazy symbol &rest fingerprint) and (eager symbol &rest fingerprint) associates the given fingerprint with the symbol-function of the given symbol. The symbol-function is interpreted differently in the two cases.
|#

(defvar *opti* :undefined) ; set in fingerprint.lisp

(defmacro register (symbol fingerprint &rest ignored)
 (declare (ignore ignored))
 `(register1 ',symbol ',fingerprint))

(deff register1 (symbol fingerprint)
 (:let ((arity)) fingerprint)
 (:let type (symbol-value symbol))
 (:let fct (fct symbol type arity))
 (push (cons fingerprint fct) *opti*))

(defmacro lazy (symbol &rest args)
 `(progn
   (defc ,symbol :lazy)
   (deff ,symbol ,@args)))

(defmacro eager (symbol &rest args)
 `(progn
   (defc ,symbol :eager)
   (deff ,symbol ,@args)))

(defmacro nodef (symbol)
 `(progn
   (defc ,symbol :lazy)
   (fmakunbound ,symbol)))

(defmacro eager-nodef (symbol)
 `(progn
   (defc ,symbol :eager)
   (fmakunbound ,symbol)))

#|
=============================================
Auxilliary definition for optimized functions
=============================================
|#

(deff bool2val (x) (if x true false))

(deff boolp (x)
 (or (true x) (equalp x false)))

(deff intp (x)
 (or (numberp x) (arrayp x)))

(deff pairp (x)
 (consp x))

(deff mapp (x)
 (or (fct-p x) (tagmap-p x)))

(deff objectp (x)
 (object-p x))

(deff boolg (x)
 (:when (boolp x) true)
 (raise true))

(deff intg (x)
 (:when (intp x) true)
 (raise true))

(deff pairg (x)
 (:when (pairp x) true)
 (raise true))

(deff mapg (x)
 (:when (mapp x) true)
 (raise true))

(deff objectg (x)
 (:when (objectp x) true)
 (raise true))

(deff getbyte (int idx)
 (ldb (byte 8 (* 8 idx)) int))

(etst (getbyte #x123456 0) #x56)
(etst (getbyte #x123456 1) #x34)
(etst (getbyte #x123456 2) #x12)
(etst (getbyte #x123456 3) 0)

(deff eqintvec (x y)
 (:let i (length y))
 (and (= (getbyte x i) 1) (eqintvec1 x y (- i 1))))

(deff eqintvec1 (x y i)
 (:when (< i 0) t)
 (and (= (getbyte x i) (aref y i)) (eqintvec1 x y (- i 1))))

(ttst (eqintvec #x01636261 (string2vector "abc")))
(ntst (eqintvec   #x636261 (string2vector "abc")))
(ntst (eqintvec #x02636261 (string2vector "abc")))
(ntst (eqintvec #x01646261 (string2vector "abc")))
(ntst (eqintvec #x01636262 (string2vector "abc")))

(deff eqint (x y)
 (if (integerp x)
  (if (integerp y) (= x y) (eqintvec x y))
  (if (integerp y) (eqintvec y x) (equalp x y))))

(deff eqval (x y)
 (:when (intp x) (and (intp y) (eqint x y)))
 (:when (pairp x)
  (and (pairp y) (eqval (car x) (car y)) (eqval (cdr x) (cdr y))))
 (:when (mapp x) (mapp y))
 (:when (objectp x)
  (and
   (objectp y)
   (equalp (object-ref x) (object-ref y))
   (equalp (object-idx x) (object-idx y))
   (equalp (object-val x) (object-val y))))
 (equalp x y))

(deff theint (x)
 (if (integerp x) x (vector2card x)))

(defc empty-vector (card*2vector nil))

(deff thevector (x)
 (:when (arrayp x) x)
 (:when (< x 0) empty-vector)
 (card2vector x))

(deff vector-norm (x)
 (:when (arrayp x) x)
 (:when (< x 0) true)
 (:when (/= (mod (integer-length x) 8) 1) true)
 (card2vector x))

#|
=============================================
Definitions of optimized functions
=============================================
|#

#|
=============================================
Parentheses
=============================================
The |!"| function covers the following constructs:
!"
( " )
hide " end hide
hiding name " end name
name " end name
newline "

The |"!| function occurs in fingerprint.lisp,
optimize.lisp, and codify.lisp. Remember to
change all of them if changing one of them
e.g. when generating fingerprint.lisp using
-optidump.
|#

(nodef |!"|)

#|
=============================================
Bottom
=============================================
|#

(eager |bottom| ()
 (map-bottom))

;(lazy |bottom| (env)
; (ttag-bottom env))

#|
=============================================
Falsehood
=============================================
|#

(eager |false| () false)

#|
=============================================
Spy function
=============================================
|#

(eager |spy ( " )| (x)
 (setq *spy* x)
 (spycd)
 card-spy)

#|
=============================================
Trace function
=============================================
|#

(eager |trace ( " )| (x)
 (format t "~%~%Trace:~%")
 (setq *trace* x)
 (:let depth (option "spydepth"))
 (:let depth (case depth (-2 10) (-1 -1) (t depth)))
 (:let length (option "spylength"))
 (:let length (case length (-2 20) (-1 nil) (t length)))
 (spy1 depth length x *spy-state*)
 card-trace)

#|
=============================================
Trace function
=============================================
|#

#|
(deff o-princ-tree (indent depth length val state)
 (funcall 'c-princ-tree indent depth length val state))

(eager |print ( " )| (x)
 (setq *trace* x)
 (:let depth (option "spydepth"))
 (:let depth (case depth (-2 10) (-1 -1) (t depth)))
 (:let length (option "spylength"))
 (:let length (case length (-2 20) (-1 nil) (t length)))
 (o-princ-tree 1 depth length x *spy-state*)
 (format t "~%")
 card-print)
|#

(deff o-princ (v)
 (:when (consp v) (o-princ (head v)) (o-princ (tail v)))
 (:when (arrayp v) (princ (vector2string v)))
 (:when (not (numberp v)) nil)
 (:when (< v 0) nil)
 (princ (card2string v)))

(eager |print ( " )| (x)
 (setq *trace* x)
 (o-princ x)
 card-print)

#|
=============================================
Timer function
=============================================
|#

(defc *timers* (make-hash-table :test 'equalp))
(defc *current-timer* nil)

(deff clear-timers ()
 (setq *current-timer* nil)
 (clrhash *timers*))

(eager |timer ( " )| (x)
 (timer1 x)
 card-timer)

(deff timer1 (x)
 (:let time (unix-time))
 (:let (count . timer) (gethash *current-timer* *timers* '(0 0 0)))
 (:let timer (time+ timer time))
 (setf (gethash *current-timer* *timers*) (cons count timer))
 (setq *current-timer* x)
 (:let (count . timer) (gethash *current-timer* *timers* '(0 0 0)))
 (:let timer (time- timer time))
 (:let count (+ 1 count))
 (setf (gethash *current-timer* *timers*) (cons count timer)))

;(general-hash2assoc hash) converts the given hash table to an association list.

(deff general-hash2assoc (hash)
 (let* ((assoc nil))
  (maphash
   #'(lambda (key val) (setq assoc (acons key val assoc)))
   hash)
  assoc))

(deff print-timers ()
 (timer1 nil)
 (remhash nil *timers*)
 (:let assoc (general-hash2assoc *timers*))
 (:when (null assoc) nil)
 (:let assoc (sort assoc 'time< :key 'cddr))
 (format t "~%Timers:~%")
 (dolist (item assoc) (report-item item)))

(deff report-item (item)
 (:let (timer c m e) item)
 (format t "~11:de-~d ~d~%" m e c)
 (:let depth (option "spydepth"))
 (:let depth (case depth (-2 10) (-1 -1) (t depth)))
 (:let indent 12)
 (:let depth (+ depth indent -1))
 (:let length (option "spylength"))
 (:let length (case length (-2 20) (-1 nil) (t length)))
 (o-print-tree indent depth length timer *spy-state*)
 (format t "~%"))

#|
=============================================
Clear all trace information
=============================================
|#

(deff clear-trace-info ()
 (clear-timers)
 (setq *spy* nil))

#|
=============================================
Normalization
=============================================
|#

(eager |" norm| (x)
 x)

(eager |norm "| (x)
 x)

#|
=============================================
Guards
=============================================
Only one of the following is used, but it is unpredictable which one
|#

(eager |" is val : "| (:x y)
 y)

(eager |" .then. "| (:x y)
 y)

#|
=============================================
Guards
=============================================
|#

(eager |" is bool : "| (x y)
 (boolg x)
 y)

(eager |" is int : "| (x y)
 (intg x)
 y)

(eager |" is pair : "| (x y)
 (pairg x)
 y)

(eager |" is map : "| (x y)
 (mapg x)
 y)

(eager |" is object : "| (x y)
 (objectg x)
 y)

#|
=============================================
Booleans
=============================================
|#

(eager |" boolp| (x)
 (bool2val (boolp x)))

(nodef |if " then " else "|)

(eager |.not. "| (x)
;(boolg x)
 (iftrue x false true))

(eager |notnot "| (x)
;(boolg x)
 (iftrue x true false))

(nodef |" .and. "|)

(nodef |" .or. "|)

(eager |" = "| (x y)
 (bool2val (eqval x y)))

#|
=============================================
Integers
=============================================
|#

(eager |" intp| (x)
 (bool2val (intp x)))

(eager |" + "| (x y)
 (intg x)
 (intg y)
 (+ (theint x) (theint y)))

(eager |+ "| (x)
 (intg x)
 (theint x))

(eager |+"| (x)
 (intg x)
 (theint x))

(eager |- "| (x)
 (intg x)
 (- (theint x)))

(eager |-"| (x)
 (intg x)
 (- (theint x)))

(eager |" - "| (x y)
 (intg x)
 (intg y)
 (- (theint x) (theint y)))

(eager |" * "| (x y)
 (intg x)
 (intg y)
 (* (theint x) (theint y)))

(eager |" < "| (x y)
 (intg x)
 (intg y)
 (bool2val (< (theint x) (theint y))))

(eager |" <= "| (x y)
 (intg x)
 (intg y)
 (bool2val (<= (theint x) (theint y))))

(eager |" > "| (x y)
 (intg x)
 (intg y)
 (bool2val (> (theint x) (theint y))))

(eager |" >= "| (x y)
 (intg x)
 (intg y)
 (bool2val (>= (theint x) (theint y))))

(eager |evenp ( " )| (x)
 (intg x)
 (bool2val (evenp (theint x))))

(eager |half ( " )| (x)
 (intg x)
 (ash (theint x) -1))

(eager |" small| (x)
 (intg x)
 (bool2val (<= -1 (theint x) 0)))

(eager |Base| () 10)

(eager %% () 0)

(eager |" %0| (x)
 (intg x)
 (* (theint x) 10))

(eager |" %1| (x)
 (intg x)
 (+ 1 (* (theint x) 10)))

(eager |" %2| (x)
 (intg x)
 (+ 2 (* (theint x) 10)))

(eager |" %3| (x)
 (intg x)
 (+ 3 (* (theint x) 10)))

(eager |" %4| (x)
 (intg x)
 (+ 4 (* (theint x) 10)))

(eager |" %5| (x)
 (intg x)
 (+ 5 (* (theint x) 10)))

(eager |" %6| (x)
 (intg x)
 (+ 6 (* (theint x) 10)))

(eager |" %7| (x)
 (intg x)
 (+ 7 (* (theint x) 10)))

(eager |" %8| (x)
 (intg x)
 (+ 8 (* (theint x) 10)))

(eager |" %9| (x)
 (intg x)
 (+ 9 (* (theint x) 10)))

(eager |floor ( " , " )| (x y)
 (intg x)
 (intg y)
 (:let x (theint x))
 (:let y (theint y))
 (:when (<= y 0) (raise))
 (:mlet (q r) (floor x y))
 (cons q r))

(eager |ceiling ( " , " )| (x y)
 (intg x)
 (intg y)
 (:let x (theint x))
 (:let y (theint y))
 (:when (<= y 0) (raise))
 (:mlet (q r) (ceiling x y))
 (cons q r))

(eager |truncate ( " , " )| (x y)
 (intg x)
 (intg y)
 (:let x (theint x))
 (:let y (theint y))
 (:when (<= y 0) (raise))
 (:mlet (q r) (truncate x y))
 (cons q r))

(eager |round ( " , " )| (x y)
 (intg x)
 (intg y)
 (:let x (theint x))
 (:let y (theint y))
 (:when (<= y 0) (raise))
 (:mlet (q r) (round x y))
 (cons q r))

(eager |" div "| (x y)
 (intg x)
 (intg y)
 (:let x (theint x))
 (:let y (theint y))
 (:when (<= y 0) (raise))
 (floor x y))

(eager |" mod "| (x y)
 (intg x)
 (intg y)
 (:let x (theint x))
 (:let y (theint y))
 (:when (<= y 0) (raise))
 (mod x y))

#|
=============================================
Pairs
=============================================
|#

(eager |" pairp| (x)
 (bool2val (pairp x)))

(eager |" :: "| (x y)
 (cons x y))

(eager |" head| (x)
 (head x))

(eager |" tail| (x)
 (tail x))

#|
=============================================
Registrations concerning exceptions
=============================================
|#

(eager |" raise| (x)
 (raise x))

(eager |exception| ()
 (raise true))

(nodef |" catch|)

#|
=============================================
Registrations concerning maps
=============================================
|#

(eager |" mapp| (x)
 (bool2val (mapp x)))

(nodef |map ( " )|)

(nodef |" catching maptag|)

(deff maptag (x)
 (make-tagmap :map (map-tv x)))

(eager |" maptag| (x)
 (maptag x))

(deff untag (x)
 (:unless (mapp x) (error "Internal error: untag applied to non-map"))
 (tm2tv (map-tail (map-tv x))))

(eager |" untag| (x)
 (mapg x)
 (untag x))

(eager |" apply "| (x y)
 (mapg x)
 (mapg y)
 (make-tagmap :map
  (delay-apply (delay-tail (map-tv x)) (delay-tail (map-tv y)))))

(eager |" root| (x)
 (mapg x)
 (:let root (map-tail (map-tv x)))
 (iftrue root true false))

#|
=============================================
Registrations concerning objects
=============================================
|#

(eager |" objectp| (x)
 (bool2val (objectp x)))

(eager |object ( " )| (x)
 (pairg x)
 (:let (tag . val) x)
 (pairg tag)
 (:let (ref . idx) tag)
 (intg ref)
 (intg idx)
 (:let ref (theint ref))
 (:let idx (theint idx))
 (:when (< ref 0) (raise))
 (:when (and (= ref 0) (< idx 4)) (raise))
 (make-object :ref ref :idx idx :val val))

(eager |destruct ( " )| (x)
 (objectg x)
 (acons (object-ref x) (object-idx x) (object-val x)))

#|
=============================================
Let construct
=============================================
|#

(nodef |LET " BE "|)

#|
=============================================
Logical operations on integers
=============================================
|#

(eager |integer-length ( " )| (x)
 (intg x)
 (:when (integerp x) (integer-length x))
 (+ 1 (* 8 (length x))))

(eager |logcount ( " )| (x)
 (intg x)
 (logcount (theint x)))

(eager |logbitp ( " , " )| ( index integer )
 (intg index)
 (intg integer)
 (:let index (theint index))
 (:when (< index 0) (raise))
 (:when (integerp integer) (bool2val (logbitp index integer)))
 (:mlet (div modulo) (floor index 8))
 (:let length (length integer))
 (:let byte (if (< div length) (aref integer div) (if (= div length) 1 0)))
 (bool2val (logbitp modulo byte)))

(eager |ash ( " , " )| (integer count)
 (intg integer)
 (intg count)
 (:let count (theint count))
 (:when (integerp integer) (ash integer count))
 (:mlet (div modulo) (floor count 8))
 (:when (or (/= modulo 0) (> div 0)) (ash (theint integer) count))
 (subvector integer (min (length integer) (- div))))

(eager |logtest ( " , " )| (x y)
 (intg x)
 (intg y)
 (bool2val (logtest (theint x) (theint y))))

(eager |logorc2 ( " , " )| (x y)
 (intg x)
 (intg y)
 (logorc2 (theint x) (theint y)))

(eager |logorc1 ( " , " )| (x y)
 (intg x)
 (intg y)
 (logorc1 (theint x) (theint y)))

(eager |logandc2 ( " , " )| (x y)
 (intg x)
 (intg y)
 (logandc2 (theint x) (theint y)))

(eager |logandc1 ( " , " )| (x y)
 (intg x)
 (intg y)
 (logandc1 (theint x) (theint y)))

(eager |lognor ( " , " )| (x y)
 (intg x)
 (intg y)
 (lognor (theint x) (theint y)))

(eager |lognand ( " , " )| (x y)
 (intg x)
 (intg y)
 (lognand (theint x) (theint y)))

(eager |logeqv ( " , " )| (x y)
 (intg x)
 (intg y)
 (logeqv (theint x) (theint y)))

(eager |logand ( " , " )| (x y)
 (intg x)
 (intg y)
 (logand (theint x) (theint y)))

(eager |logxor ( " , " )| (x y)
 (intg x)
 (intg y)
 (logxor (theint x) (theint y)))

(eager |logior ( " , " )| (x y)
 (intg x)
 (intg y)
 (logior (theint x) (theint y)))

(eager |lognot ( " )| (x)
 (intg x)
 (lognot (theint x)))

#|
=============================================
Operations on vectors
=============================================
|#

(eager |vector ( " )| (x)
 (intg x)
 (thevector x))

(eager |vector-norm ( " )| (x)
 (intg x)
 (vector-norm x))

(deff vector-empty-p (x)
 (:when (integerp x) (< x 256))
 (equalp (length x) 0))

(eager |vector-empty ( " )| (x)
 (intg x)
 (bool2val (vector-empty-p x)))

(eager |vector-suffix ( " , " )| (x y)
 (intg x)
 (intg y)
 (:let vector (thevector x))
 (:let index (theint y))
 (:when (< index 0) vector)
 (:when (<= (length vector) index) empty-vector)
 (subseq vector index))

(eager |vector-prefix ( " , " )| (x y)
 (intg x)
 (intg y)
 (:let vector (thevector x))
 (:let index (theint y))
 (:when (< index 0) empty-vector)
 (:when (<= (length vector) index) vector)
 (subseq vector 0 index))

(eager |vector-subseq ( " , " , " )| (x y z)
 (intg x)
 (intg y)
 (intg z)
 (:let vector (thevector x))
 (:let begin (max 0 (theint y)))
 (:let end (min (length vector) (theint z)))
 (:when (<= end begin) empty-vector)
 (subseq vector begin end))

(eager |vector-length ( " )| (x)
 (intg x)
 (length (thevector x)))

(eager |vector-index ( " , " )| (x y)
 (intg x)
 (intg y)
 (:let vector (thevector x))
 (:let index (theint y))
 (:when (< index 0) (raise))
 (:when (<= (length vector) index) (raise))
 (aref vector index))

(eager |vector2byte* ( " )| (x)
 (intg x)
 (vector2card* (thevector x)))

(eager |vector2vector* ( " )| (x)
 (intg x)
 (vector2vector* (thevector x)))

(deff bt-length (bt result)
 (:when (and (numberp bt) (<= 0 bt 255)) (+ 1 result))
 (:when (and (arrayp bt) (= (length bt) 0)) (+ 1 result))
 (:when (pairp bt) (bt-length (cdr bt) (bt-length (car bt) result)))
 result)

(deff bt2vector1 (bt v i)
 (:when (and (numberp bt) (<= 0 bt 255)) (setf (aref v i) bt) (+ i 1))
 (:when (and (arrayp bt) (= (length bt) 0)) (setf (aref v i) 1) (+ i 1))
 (:when (pairp bt) (bt2vector1 (cdr bt) v (bt2vector1 (car bt) v i)))
 i)

(eager |bt2vector ( " )| (bt)
 (:let length (bt-length bt 0))
 (:let vector (null-vector length))
 (:when (unequal length (bt2vector1 bt vector 0)) (error "Internal error"))
 vector)

(eager |bt2byte* ( " )| (bt)
 (vector2card* (|bt2vector ( " )| bt)))

(eager |bt2vector* ( " )| (bt)
 (vector2vector* (|bt2vector ( " )| bt)))

(deff vt-length (bt result)
 (:when (intp bt) (+ (length (thevector bt)) result))
 (:when (pairp bt) (vt-length (cdr bt) (vt-length (car bt) result)))
 result)

(deff vt2vector1 (vt v i)
 (:when (pairp vt) (vt2vector1 (cdr vt) v (vt2vector1 (car vt) v i)))
 (:unless (intp vt) i)
 (:let w (thevector vt))
 (:let length (length w))
 (dotimes (j length) (setf (aref v (+ i j)) (aref w j)))
 (+ i length))

(eager |vt2vector ( " )| (vt)
 (:let length (vt-length vt 0))
 (:let vector (null-vector length))
 (:when (unequal length (vt2vector1 vt vector 0)) (error "Internal error"))
 vector)

(eager |vt2byte* ( " )| (vt)
 (vector2card* (|vt2vector ( " )| vt)))

(eager |vt2vector* ( " )| (vt)
 (vector2vector* (|vt2vector ( " )| vt)))

#|
=============================================
Fixed point operator
=============================================
|#

(deff ttag-yy1 (env)
 (:let (fct) env)
 (:let env (list :to-be-replaced fct))
 (:let closure (term2closure (term-apply var-1 var-0) env))
 (setf (car env) closure)
 (map2rnf closure))

(lazy |YY| (env)
 (list 'mtag-lambda env (list 'ttag-yy1)))

(deff term-yy ()
 (list '|YY|))

(defc term-yy (term-yy))

(defc term-yyk (term-apply term-yy term-k))
(defc term-yyki (term-apply term-yy term-ki))
(test (term-eval term-yy))
(ntst (term-eval (term-apply term-yy term-t)))
(ntst (term-eval (term-apply term-yy term-f)))
(test (term-eval term-yyk))
(test (term-eval (term-apply term-yyk term-t)))
(test (term-eval (term-apply term-yyk term-f)))
(test (term-eval (term-apply (term-apply term-yyk term-t) term-t)))
(test (term-eval (term-apply (term-apply term-yyk term-f) term-f)))
(test (term-eval term-yyki))
(ntst (term-eval (term-apply term-yyki term-t)))
(test (term-eval (term-apply term-yyki term-f)))

#|
=============================================
Compilation
=============================================
|#

;(eager-nodef |compile ( " )|)

(eager |compile ( " )| (c)
 (funcall 'c-compile-verify-cache c))

#|
=============================================
Ripemd-160
=============================================
|#

(eager |ripemd ( " )| (vt)
 (|bt2vector* ( " )| (ripemd (|vt2byte* ( " )| vt))))

(etst
 (|ripemd ( " )| (list #(97) #(98) #(99)))
 (list
  #(#x8E) #(#xB2) #(#x08) #(#xF7) #(#xE0)
  #(#x5D) #(#x98) #(#x7A) #(#x9B) #(#x04)
  #(#x4A) #(#x8E) #(#x98) #(#xC6) #(#xB0)
  #(#x87) #(#xF1) #(#x5A) #(#x0B) #(#xFC)))
(etst
 (|ripemd ( " )| (list (+ 97 256) (+ 98 256) (+ 99 256)))
 (list
  #(#x8E) #(#xB2) #(#x08) #(#xF7) #(#xE0)
  #(#x5D) #(#x98) #(#x7A) #(#x9B) #(#x04)
  #(#x4A) #(#x8E) #(#x98) #(#xC6) #(#xB0)
  #(#x87) #(#xF1) #(#x5A) #(#x0B) #(#xFC)))
(etst
 (|ripemd ( " )| (cons (cons #(97) #(98)) (cons nil #(99))))
 (list
  #(#x8E) #(#xB2) #(#x08) #(#xF7) #(#xE0)
  #(#x5D) #(#x98) #(#x7A) #(#x9B) #(#x04)
  #(#x4A) #(#x8E) #(#x98) #(#xC6) #(#xB0)
  #(#x87) #(#xF1) #(#x5A) #(#x0B) #(#xFC)))

#|
=============================================
Convert list of singletons to rack
=============================================
The definition of load-rack in codify.lisp
was copied here and then adapted for the task
at hand.
|#

(defc *o-rack* nil)
(defc *o-hash* (make-hash-table :test 'equalp))
(defc *o-pnt* 0)

(deff sing2card (sing)
 (:when (arrayp sing) (if (= (length sing) 1) (aref sing 0) (raise true)))
 (:when (numberp sing) (if (< 255 sing 512) (- sing 256) (raise true)))
 (raise true))

(deff o-read-byte ()
 (:unless (listp *o-rack*) (raise true))
 (sing2card (pop *o-rack*)))

(deff o-write-byte (byte)
 (push byte *o-rack*))

(deff o-get-hash (index)
 (gethash index *o-hash*))

(deff o-put-hash (index value)
 (setf (gethash index *o-hash*) value))

(deff o-push-hash (value)
 (o-put-hash *o-pnt* value)
 (incf *o-pnt*))

(deff o-pop-hash ()
 (decf *o-pnt*)
 (o-get-hash *o-pnt*))

(deff o-read-card ()
 (:let card (o-read-byte))
 (:when (< card 128) card)
 (+ card -128 (ash (o-read-card) 7)))

(deff o-read-vector (length result)
 (:when (= length 0) (card*2vector (reverse result)))
 (:let card (o-read-byte))
 (o-read-vector (- length 1) (cons card result)))

(deff o-read-sharing ()
 (:let card0 (o-read-card))
 (:when (= card0 *o-pnt*) (o-pop-hash))
 (:let card1 (o-read-card))
 (o-read-sharing1 card0 card1)
 (o-read-sharing))

(deff o-read-sharing1 (card0 card1)
 (:when (= card0 0) (o-push-hash card1))
 (:when (= card0 2) (o-push-hash (o-read-vector card1 nil)))
 (o-push-hash (cons (o-get-hash card0) (o-get-hash card1))))

(eager |sl2rack ( " )| (v)
;(:when (null v) nil)
 (setq *o-rack* v)
 (clrhash *o-hash*)
 (setq *o-pnt* 0)
 (o-push-hash nil)
 (o-push-hash nil)
 (o-push-hash nil)
 (o-read-sharing))

(deff o-v* (&rest x)
 (card*2vector* x))

(etst (o-v* 2 3 4) '(#(2) #(3) #(4)))
(etst (|sl2rack ( " )| (o-v* 3)) nil)
(etst (|sl2rack ( " )| (o-v* 1 1 4)) (cons nil nil))
(etst (|sl2rack ( " )| (o-v* 0 7 4)) 7)
(etst (|sl2rack ( " )| (o-v* 2 3 7 8 9 4)) #(7 8 9))
(etst (|sl2rack ( " )| (o-v* 0 6 2 3 7 8 9 3 4 6)) (cons 6 #(7 8 9)))
(etst (|sl2rack ( " )| (o-v* 0 6 2 3 7 8 9 3 4 6 0)) (cons 6 #(7 8 9)))
(etst (|sl2rack ( " )| (append (o-v* 0 6 2 3 7 8 9 3 4 6) (list false)))
 (cons 6 #(7 8 9)))
(xtst (|sl2rack ( " )| (o-v* 0 6 2 3 7 8 9 3 4)))

#|
=============================================
Convert rack to list of singletons
=============================================
|#

(deff o-write-card (card)
 (:when (< card 128) (o-write-byte card))
 (:mlet (div modulo) (floor card 128))
 (o-write-byte (+ 128 modulo))
 (o-write-card div))

(deff o-write-shared-card (card)
 (:when (< card 0) (raise true))
 (:let index (o-get-hash card))
 (:when index index)
 (:when (and (> card 0) (= 1 (floor (integer-length card) 8)))
  (o-write-shared-vector (card2vector card)))
 (incf *o-pnt*)
 (o-put-hash card *o-pnt*)
 (o-write-card 0)
 (o-write-card card)
 *o-pnt*)

(deff o-write-shared-vector (vector)
 (:let index (o-get-hash vector))
 (:when index index)
 (incf *o-pnt*)
 (o-put-hash vector *o-pnt*)
 (:let length (length vector))
 (o-write-card 2)
 (o-write-card length)
 (dotimes (n length) (o-write-byte (aref vector n)))
 *o-pnt*)

(deff o-write-shared-cons (cons)
 (:let (head . tail) cons)
 (:let head (o-write-sharing head))
 (:let tail (o-write-sharing tail))
 (:let cons (cons head tail))
 (:let index (o-get-hash cons))
 (:when index index)
 (incf *o-pnt*)
 (o-put-hash cons *o-pnt*)
 (o-write-card head)
 (o-write-card tail)
 *o-pnt*)

(deff o-write-sharing (r)
 (:when (null r) 1)
 (:when (numberp r) (o-write-shared-card r))
 (:when (arrayp r) (o-write-shared-vector r))
 (:when (consp r) (o-write-shared-cons r))
 (raise true))

(eager |rack2sl ( " )| (r)
;(:when (null r) (list #(3)))
 (setq *o-rack* nil)
 (clrhash *o-hash*)
 (setq *o-pnt* 2)
 (o-write-sharing r)
 (o-write-card (+ *o-pnt* 1))
 (card*2vector* (reverse *o-rack*)))

(etst (o-v* 3) (|rack2sl ( " )| nil))
(etst (o-v* 1 1 4) (|rack2sl ( " )| (cons nil nil)))
(etst (o-v* 0 7 4) (|rack2sl ( " )| 7))
(etst (o-v* 2 3 7 8 9 4) (|rack2sl ( " )| #(7 8 9)))
(etst (o-v* 0 6 2 3 7 8 9 3 4 6) (|rack2sl ( " )| (cons 6 #(7 8 9))))
(etst (o-v* 0 7 3 3 5) (|rack2sl ( " )| (cons 7 7)))
(etst (o-v* 1 1 3 3 5) (|rack2sl ( " )| (cons (cons nil nil) (cons nil nil))))
(xtst (|rack2sl ( " )| (cons (cons nil nil) (cons nil -1))))
(xtst (|rack2sl ( " )| (cons (cons nil nil) (cons nil false))))



