#|  Logiweb, a system for electronic distribution of mathematics
    Copyright (C) 2004-2010 Klaus Grue

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact: Klaus Grue, DIKU, Universitetsparken 1, DK2100 Copenhagen,
    Denmark, grue@diku.dk, http://logiweb.eu/, http://www.diku.dk/~grue/

    Logiweb is a system for distribution of mathematical definitions,
    lemmas, and proofs. For more on Logiweb, consult http://logiweb.eu/.
|#

#|
=============================================
The Logiweb Compiler
=============================================
Front end
=============================================
|#

(in-package "COMMON-LISP-USER")

#|
=============================================
Parse functions
=============================================

Parse functions move a 'pivot' across the text in the reading direction (left-to-right in left-to-right reading countries, as one should expect).

A pivot is a cardinal which indexes the next, unparsed character. A value of zero indicates that no characters have been parsed.

The text to be parsed is in *source* which is an array of cardinals.

A parse function takes a pivot as input and either raises an exception or delivers a pair (item . pivot1) as output. Normally, a parse function parses one syntactical item, translates it, and returns it together with a new pivot.

When a parse function cannot parse one syntactical item, it either makes a 'hard' or a 'soft' complaint. It makes a hard complaint if the situation indicates that the file definitely is in error and, furthermore, the situation allows to formulate a precise error message. In that case, the parse function reports the error to standard output and raises an exception.

If it is unclear, however, whether or not the file is in error, or if the situation does not allow to formulate a precise error message, then the parse function makes a soft complaint: it returns the pair (:no-item . pivot) where pivot is the pivot with which the parse function was called.

Text from lgs files is pre-prosessed before it reaches *source* where pre-processing may involve recursive inclusion of other files. The *sourcemap* variable is set to a value which allows to locate errors in include files. We shall refer to the main lgs file and all recursively included files as 'source files'.

The value of *sourcemap* has form ((i_1 n_1 l_1 c_1) (i_2 n_2 l_2 c_2) ...) where n_k is the name of a source file, i_k is the index of a character in *source*, and l_k and c_k are the line and character numbers, respectively, of the associated character in the source file. The names n_k are expressed as strings. We have i_1 > i_2 > ...

As an example, if *sourcemap* is ((100 "main.lgs" 1 35) (5 "include.lgs" 1 1) (0 "main.lgs" 1 1)) then characters 0..4 of *source* come from "main.lgs", characters 5..99 come from "include.lgs", and characters 100 and up come from "main.lgs".

In ((i_1 n_1 l_1 c_1) (i_2 n_2 l_2 c_2) ...), the values of i_k count in UTF-8 bytes starting with zero. l_k counts lines, starting with one, and c_k count characters starting with one at the beginning of each line. One character may correspond to several UTF-8 bytes.

The text to be parsed comes from one of three sources: test cases, lgs files, and CGI-scripts. Source text from test cases and CGI-scripts is not pre-preposessed and *sourcemap* is set to ((0 name 1 1)) where 'name' is "testfile" for test cases and "Form input" for CGI-scripts.

Whenever a source file starts with "";; followed by a hex number, that
hex number is stored in *sourceref* as a card* and the name of the file
is stored in *sourcename*. If a source file does not
start with "";; then *sourceref* is set to :none.
|#

(defc *sourcemap* nil)
(defc *source* (make-array 0))
(defc *pivot* 0)
(defc *max-pivot* 0)
(defc *sourceref* :none)
(defc *sourcename* :none)

(deff set-source
 (source &key
  (filename "testfile")
  (pivot 0)
  (silent nil)
  (sourcemap nil))
 (setq *pivot* pivot)
 (setq *sourcemap* (default (list (list 0 filename 1 1)) sourcemap))
 (setq *source* (ct2vector source))
 (when (and (not silent) *ok* *testp*)
  (format t "-----------------------------------------------------~%")
  (format t "~a~%" (vector2string *source*))
  (format t "-----------------------------------------------------~%")))

(deff no-item (item)
 (equalp item :no-item))

(ttst (no-item :no-item))
(ntst (no-item nil))

(deff is-item (item)
 (not (no-item item)))

(ntst (is-item :no-item))
(ttst (is-item nil))

(deff soft-complaint (pivot)
 (setq *pivot* pivot)
 :no-item)

(setq *pivot* 5)
(ttst (no-item (soft-complaint 2)))
(etst *pivot* 2)

#|
=============================================
Add and remove file extensions
=============================================
(add-extension string1 string2) concatenates the given strings unless string1 contains a period in which case the function returns string1.

(remove-extension string1 string2) removes string2 from string1 if string2 occurs at the end of string1.
|#

(deff add-extension (file extension)
;(:when (position #\. file) file)
 (:when (pathname-type file) file)
 (cat file extension))

(etst (add-extension "abc" ".def") "abc.def")
(etst (add-extension "abc.xxx" ".def") "abc.xxx")

(deff remove-extension (file extension)
 (:let file0 (length file))
 (:let ext0 (length extension))
 (:when (<= file0 ext0) file)
 (:let pivot (- file0 ext0))
 (:when (equal (subseq file pivot) extension) (subseq file 0 pivot))
 file)

(etst (remove-extension "abc.def" ".def") "abc")
(etst (remove-extension "abc.def" ".DEF") "abc.def")

#|
=============================================
UTF-8 line and character counting
=============================================
(f-utf8-length vector start end) counts the number of UTF-8 characters from start to end in vector.

(f-line-char vector start end line char) returns (list line1 char1) where line1 and char1 are the line and char numbers at 'end' in vector provided the line and char numbers at 'start' is (list line char).

f-line-char1 does the same as f-line-char except that it is assumed that there are no newline characters between the given start and end.

(f-move vector position line char delta) moves up to 'line' lines / 'char' utf8 characters in the direction indicated by delta and returns the new position. The function moves forward if delta is +1 and backwards if delta is -1. The function never moves past the end of the vector. If the function hits the beginning of the vector it returns zero. If it hits the end it returns (length vector).
|#

(deff f-utf8-nochar (x)
 (<= #x80 x #xBF))

(deff f-utf8-length (vector start end)
 (count-if-not 'f-utf8-nochar vector :start start :end end))

(deff f-line-char (vector start end line char)
 (:let newlines (count f-newline vector :start start :end end))
 (:when (= newlines 0) (f-line-char1 vector start end line char))
 (:let line (+ line newlines))
 (:let start (position f-newline vector :start start :end end :from-end t))
 (:let char (f-utf8-length vector start end))
 (list line char))

(deff f-line-char1 (vector start end line char)
 (:let char (+ char (f-utf8-length vector start end)))
 (list line char))

(deff f-move (vector position line char delta)
 (:when (equal line 0) position)
 (:when (equal char 0) position)
 (:let position1 (+ position delta))
 (:unless (< -1 position1 (length vector)) position)
 (:let char (aref vector position1))
 (:when (equalp char f-newline) (f-move vector position1 (- line 1) char delta))
 (:when (f-utf8-nochar char) (f-move vector position1 line char delta))
 (f-move vector position1 line (- char 1) delta))

#|
=============================================
Generate error message and raise exception
=============================================

(f-parse-error message) reports the given message to standard output and indicates the location of the error using information from the pivot.

(f-error message) is similar, but skips leading spaces in the given pivot first.

The location of the error is reported in a TeX-like style: Up to *max-error-char* characters / max-error-line* lines before and after the pivot is printed out together with a line number.
|#

(defc *max-error-char* 200)
(defc *max-error-line* 3)

(deff f-error (format &rest args)
 (f-skip-space)
 (apply 'f-parse-error format args))

(deff f-parse-error (format &rest args)
 (:let left (f-move *source* *pivot* *max-error-line* *max-error-char* -1))
 (:let right (f-move *source* *pivot* *max-error-line* *max-error-char* 1))
 (:let (index name line char) (assoc *pivot* *sourcemap* :test '>=))
 (:let (line char) (f-line-char *source* index *pivot* line char))
 (format t "---~%")
 (format t "~a|~%" (utf2string (subseq *source* left *pivot*)))
 (format t "---~%")
 (format t "~a~%" (utf2string (subseq *source* *pivot* right)))
 (format t "---~%")
 (format t "File ~a around line ~d char ~d:~%" name line char)
 (apply 'format t format args)
 (terpri)
 (raise))

(defc *intended-error*
 "This error message is intensional and indicates that a test has succeeded")

(set-source
"abc
defghi
jkl")
(setq *pivot* 7)
(xtst (f-parse-error *intended-error*))

#|
=============================================
Preprocessor
=============================================
(f-preprocess name) preprocesses the file with the given name, returns the processed file as a vector, and sets *sourcemap*. The name must be a string.

(f-preprocess1 index map name option) filters the file with the given name using the filter specified by the given option. The option may be (option "filter")
or (string2list string). f-preprocess1 returns a list (ct index map) where ct is the result of the preprocessing, index is the total byte length of the ct measured in bytes, and map is the thing to be installed in *sourcemap*.

(f-include vector index map name line char option) processes include directives in the given vector and returns a list (ct index map) like f-preprocess1 does. The 'index' indicates which position in the final vector returned by f-preprocess we have reached. The name, line, and char indicates which file the vector comes from and the line and char number of the first byte in the vector. Hence, the index refers to the output of the preprocessor whereas the name, line, and  char refers to the input. The built up map is an association list from indices to (name line char) lists.
|#

(deff f-preprocess (name)
 (:let option (option "filter"))
 (:let (ct :index map) (f-preprocess1 0 nil name option))
 (set-source ct :filename name :sourcemap map :silent t)
 nil)

(defc f-magic-code "\"\";;")

(deff digit-code-p (code)
 (or
  (<= (char-code #\0) code (char-code #\9))
  (<= (char-code #\a) code (char-code #\f))
  (<= (char-code #\A) code (char-code #\F))))

(deff f-source2ref (vector)
 (:when (unequal (safe-subseq vector 0 4) (ct2vector f-magic-code)) :none)
 (:let position (position-if-not 'digit-code-p vector :start 4))
 (:catch () nil)
 (edge2hex (ct2card* (subseq vector 4 position))))

(deff f-remove-header (vector)
 (:let position (position-if-not 'digit-code-p vector :start 4))
 (:when (null position) (ct2vector ""))
 (subseq vector position))

(etst (f-source2ref (ct2vector "abcd")) :none)
(etst (f-source2ref (ct2vector "\"\";;90afAFxyz")) '(#x90 #xAF #xAF))
(etst (f-source2ref (ct2vector "\"\";;90afAF")) '(#x90 #xAF #xAF))
(etst (f-source2ref (ct2vector "\"\";;90afAF")) '(#x90 #xAF #xAF))
(etst (f-source2ref (ct2vector "\"\";;90afAx")) nil)

(etst (f-remove-header (ct2vector "\"\";;90afAFxyz")) (ct2vector "xyz"))
(etst (f-remove-header (ct2vector "\"\";;90afAF")) (ct2vector ""))
(etst (f-remove-header (ct2vector "\"\";;90afAF")) (ct2vector ""))
(etst (f-remove-header (ct2vector "\"\";;90afAx")) (ct2vector "x"))

(deff f-ref2hex (ref)
 (if (equalp ref :none) ":none"
  (ct2string (b-byte*2hex* *sourceref*))))

(deff f-preprocess1 (index map name option)
 (format t "Frontend: reading ~s~%" name)
 (:let vector (path2vector name))
 (:when (equalp vector :does-not-exist) (c-error "File not found: ~a~%" name))
 (:let ref1 (f-source2ref vector))
 (:catch () (c-error "Error while processing file ~a~%" name))
 (when (verbose '> 0) (progress "Frontend: ensure source to be utf-8"))
 (:let vector (apply-filter option vector))
 (:let ref2 (f-source2ref vector))
 (:let ref3 (if (equalp ref1 ref2) ref1 :none))
 (when (equalp index 0)
  (when (verbose '> 0) (format t "Frontend: sourceref=~a~%" (f-ref2hex ref3)))
  (setq *sourceref* ref3)
  (setq *sourcename* name))
 (when (verbose '> 0) (progress "Frontend: process include directives"))
 (f-include vector index map name 1 1 option))

(defc f-lpar "(") (defc f-rpar ")")
;(defc f-quot (char-code #\"))
(defc include-directive (string2vector "\"\"#include"))

(deff f-include-usage (item)
 (:when (is-item item) nil)
 (f-parse-error "Usage: \"\"#include(\"filter\",\"filename\")"))

(deff f-parse-include ()
 (f-include-usage (f-parse-separator f-lpar))
 (:let x (f-parse-soft-string))
 (f-include-usage x)
 (:let x (card*2string x))
 (:when (is-item (f-parse-separator f-rpar)) (cons x ""))
 (f-include-usage (f-parse-separator ","))
 (:let y (f-parse-soft-string))
 (f-include-usage y)
 (:let y (card*2string y))
 (f-include-usage (f-parse-separator f-rpar))
 (cons x y))

(deff f-include (vector index map name line char option)
 (:let map (cons (list index name line char) map))
 (:let pos (search include-directive vector))
 (:when (null pos) (list vector (+ index (length vector)) map))
 (:let ct1 (subvector vector 0 pos))
 (:let index (+ index pos))
 (:let pos (+ pos (length include-directive)))
 (set-source vector :filename name :pivot pos :silent t :sourcemap map)
 (:let (name2 . filter) (f-parse-include))
 (:let filter (string2list filter))
 (:let filter (default option filter))
 (:let pos *pivot*)
 (:let (ct2 index map) (f-preprocess1 index map name2 filter))
 (:let (line char) (f-line-char vector 0 pos line char))
 (:let vector (subvector vector pos))
 (:let (ct3 index map) (f-include vector index map name line char option))
 (:let ct (list ct1 ct2 ct3))
 (list ct index map))

#|
=============================================
Process keyword option
=============================================
|#

(defc *f-keyword* nil)
(defc *f-name* nil)
(defc *f-prio* nil)

(defc *f-standard-keyword*
 '("Name" "Priority"
   "PAGE" "BIBLIOGRAPHY" "PREASSOCIATIVE" "POSTASSOCIATIVE" "BODY"))

(deff keyword-zip (a b)
 (keyword-zip1 a b nil))

(deff keyword-zip1 (a b result)
 (:when (and (atom a) (atom b)) (reverse result))
 (:when (or (atom a) (atom b)) :mismatch)
 (:let (a1 . a) a)
 (:let (b1 . b) b)
 (keyword-zip1 a b (acons (coerce (ct2ct a1) 'vector) (ct2ct b1) result)))

(deff f-process-keyword-option ()
 (f-process-keyword-option1 (option "keyword")))

(deff f-process-keyword-option1 (keyword)
 (:let keyword (default *f-standard-keyword* keyword))
 (:let (name prio . rest) keyword)
 (:let f-keyword (keyword-zip rest (cddr *f-standard-keyword*)))
 (:when (equalp f-keyword :mismatch)
  (error "The 'keyword' option must contain seven elements"))
 (setq *f-keyword* f-keyword)
 (setq *f-name* name)
 (setq *f-prio* prio))

(f-process-keyword-option1 nil)

#|
=============================================
Process options
=============================================
|#

(deff f-process-options ()
 (f-process-keyword-option))

#|
=============================================
Parse one character
=============================================

f-parse-char parses one character and makes a soft complaint at end of file.
|#

(deff f-parse-char ()
 (:when (>= *pivot* (length *source*)) :no-item)
 (:let result (aref *source* *pivot*))
 (incf *pivot*)
 result)

(set-source "abc")
(etst (f-parse-char) 97)
(etst (f-parse-char) 98)
(etst (f-parse-char) 99)
(ttst (no-item (f-parse-char)))
(etst *pivot* 3)

#|
=============================================
Skip spaces and comments
=============================================

(f-skip-space pivot) returns the given pivot with initial spaces and comments removed. f-skip-space is not a genuine parse function since it just returns a pivot, not a pair of an item and a pivot.

f-skip-space recognizes 'long' and 'short' comments. Long comments start with a left brace, ends with a right brace, and may span several lines. Short comments start with a semicolon and end with the end of line/end of file.

f-skip-space makes a hard complaint if it encounters the end of the file while scanning a long comment.

(f-parse-space pivot) is the parse-function-equivalent of f-skip-space. It makes a soft complaint if the given pivot does not start with a space and returns a space character otherwise.

(f-parse-space-or-char pivot). Parse one character. A sequence of spaces/comments counts as a single space character. The function makes a soft complaint at end of file.

f-short is the character that begins a short comments, i.e. a semicolon.

f-long-begin is the character that begins a long comments, i.e. a left brace.

f-long-end is the character that ends a long comments, i.e. a right brace.

f-spaces is the set of characters that count as spaces.

|#

(defc f-short (char-code #\;))

(defc f-long-begin (char-code #\{))

(defc f-long-end (char-code #\}))

(defc f-spaces (list f-space f-newline f-tab))

(defc f-quote (char-code #\"))

(deff f-parse-space ()
 (:let pivot *pivot*)
 (:let code (f-parse-char))
 (:when (no-item code) (soft-complaint pivot))
 (:when (member code f-spaces) (f-skip-space) f-space)
 (:when (equalp code f-quote) (f-parse-comment pivot))
 (soft-complaint pivot))

(deff f-parse-comment (pivot)
 (:when (unequal (f-parse-char) f-quote) (soft-complaint pivot))
 (f-parse-comment1 pivot))

(deff f-parse-comment1 (pivot)
 (:let code (f-parse-char))
 (:when (equalp code f-quote) (f-parse-comment1 pivot))
 (:when (equalp code f-short) (f-skip-comment f-newline) f-space)
 (:when (equalp code f-long-begin)
  (f-skip-long-comment f-quote f-long-end) f-space)
 (soft-complaint pivot))

(deff f-skip-space ()
 (:when (is-item (f-parse-space)) (f-skip-space)))

(deff f-skip-comment (end)
 (:let code (f-parse-char))
 (:when (no-item code) nil)
 (:when (equalp code end) (f-skip-space))
 (f-skip-comment end))

(deff f-skip-long-comment (escape end)
 (:let code (f-parse-char))
 (:when (no-item code) (f-parse-error "End of file in comment"))
 (:when (unequal code escape) (f-skip-long-comment escape end))
 (:let code (f-parse-char))
 (:when (no-item code) (f-parse-error "End of file in comment"))
 (:when (unequal code escape) (f-skip-long-comment escape end))
 (:let code (f-parse-char))
 (:when (no-item code) (f-parse-error "End of file in comment"))
 (:when (unequal code end) (f-skip-long-comment escape end))
 (f-skip-space))

(deff f-parse-space-or-char ()
 (:let pivot *pivot*)
 (:let result (f-parse-space))
 (:when (is-item result) result)
 (setq *pivot* pivot)
 (f-parse-char))

(set-source "abc")
(etst (f-parse-char) 97)
(etst *pivot* 1)
(f-skip-space)
(etst *pivot* 1)

(set-source "a bc")
(etst (f-parse-char) 97)
(etst *pivot* 1)
(f-skip-space)
(etst *pivot* 2)

(set-source "a
bc")
(etst (f-parse-char) 97)
(etst *pivot* 1)
(f-skip-space)
(etst *pivot* 2)

(set-source "a
 bc")
(etst (f-parse-char) 97)
(etst *pivot* 1)
(f-skip-space)
(etst *pivot* 3)

(set-source "a\"\"{xy\"\"}b")
(etst (f-parse-char) 97)
(f-skip-space)
(etst (f-parse-char) 98)

(set-source "a\"\"\"{xy\"\"}b")
(etst (f-parse-char) 97)
(f-skip-space)
(etst (f-parse-char) 98)

(set-source "a\"\"\"\"{xy\"\"}b")
(etst (f-parse-char) 97)
(f-skip-space)
(etst (f-parse-char) 98)

(set-source "a\"\"\"\"\"{xy\"\"}b")
(etst (f-parse-char) 97)
(f-skip-space)
(etst (f-parse-char) 98)

(set-source "a\"\";xy
b")
(etst (f-parse-char) 97)
(f-skip-space)
(etst (f-parse-char) 98)

(set-source "a
 \"\";{xy
\"\"{xy;xy\"\"}b")
(etst (f-parse-char) 97)
(f-skip-space)
(etst (f-parse-char) 98)

(set-source "a\"\"{")
(etst (f-parse-char) 97)
(xtst (f-skip-space))



(set-source "abc")
(etst (f-parse-char) 97)
(etst *pivot* 1)
(ttst (no-item (f-parse-space)))
(etst *pivot* 1)

(set-source "a bc")
(etst (f-parse-char) 97)
(etst *pivot* 1)
(ttst (is-item (f-parse-space)))
(etst *pivot* 2)

(set-source "a
bc")
(etst (f-parse-char) 97)
(etst *pivot* 1)
(ttst (is-item (f-parse-space)))
(etst *pivot* 2)

(set-source "a
 bc")
(etst (f-parse-char) 97)
(etst *pivot* 1)
(ttst (is-item (f-parse-space)))
(etst *pivot* 3)

(set-source "a\"\"{xy\"\"}b")
(etst (f-parse-char) 97)
(ttst (is-item (f-parse-space)))
(etst (f-parse-char) 98)

(set-source "a\"\";xy
b")
(etst (f-parse-char) 97)
(ttst (is-item (f-parse-space)))
(etst (f-parse-char) 98)

(set-source "a
 \"\";{xy
\"\"{xy;xy\"\"}b")
(etst (f-parse-char) 97)
(ttst (is-item (f-parse-space)))
(etst (f-parse-char) 98)

(set-source "a\"\"{")
(etst (f-parse-char) 97)
(xtst (f-parse-space))

#|
=============================================
Parse particular character
=============================================

f-parse-separator skips leading spaces in search for the given separator. If the separator is parsed and returned. Otherwise f-parse-separator makes a soft complaint.

|#

(deff f-parse-separator (separator)
 (:let pivot *pivot*)
 (f-skip-space)
 (:let char (f-parse-char))
 (:when (and (is-item char) (position (code-char char) separator)) char)
 (soft-complaint pivot))

(set-source "a ,b")
(etst (f-parse-char) 97)
(etst (char-code #\,) (f-parse-separator ".,;"))
(etst (f-parse-char) 98)

(set-source "a , b")
(etst (f-parse-char) 97)
(ttst (no-item (f-parse-separator ".;")))
(etst *pivot* 1)

#|
=============================================
Parse keyword
=============================================
|#

(deff f-parse-keyword ()
 (:let pivot *pivot*)
 (f-skip-space)
 (:let keyword (f-parse-keyword1 *f-keyword*))
 (:when keyword keyword)
 (soft-complaint pivot))

(deff f-parse-keyword-or-eof ()
 (:let pivot *pivot*)
 (f-skip-space)
 (:when (>= *pivot* (length *source*)) :eof)
 (:let keyword (f-parse-keyword1 *f-keyword*))
 (:when keyword keyword)
 (soft-complaint pivot))

(deff f-parse-keyword1 (keyword*)
 (:when (atom keyword*) nil)
 (:let ((key . val) . keyword*) keyword*)
 (:let end (+ *pivot* (length key)))
 (:when (< (length *source*) end) (f-parse-keyword1 keyword*))
 (:when (unequal key (subseq *source* *pivot* end)) (f-parse-keyword1 keyword*))
 (setq *pivot* end)
 val)

(set-source "a \"\"{Comment\"\"} PAGEb")
(etst (f-parse-char) 97)
(etst (f-parse-keyword) (ct2ct "PAGE"))
(etst (f-parse-char) 98)

(set-source "a \"\"{Comment\"\"} b")
(etst (f-parse-char) 97)
(ttst (no-item (f-parse-keyword)))
(etst *pivot* 1)

#|
=============================================
Parse string
=============================================

(f-parse-string pivot) parses a string that starts with an ascii quotation mark (").

At any time while parsing the string, one particular character is the 'escape' character. The escape character is the ascii quotation mark, unless it is changed by an escape sequence.

While scanning the string, an escape character followed by a space marks the end of the string. Hence, "abc" denotes a three-letter-string since it ends with the escape character (a quotation mark) followed by a space. In this context, a space can be an ascii space character, an ascii newline character, a semicolon (which initiates a comment which acts as a space), a left brace (which also initiates a comment), or the end of the file. An escape character followed by a comma or a right bracket also marks the end of the string.

An escape character followed by an exclamation mark denotes one occurrence of the escape character itself. Hence, "ab"!" denotes a three-letter-string consisting of an ascii small a, an ascii small b, and an ascii quotation mark.

An escape character followed by a question mark changes the escape character to be the character after the quotation mark. Hence, "ab"?+c+ denotes a three-letter-string consisting of a, b, and c.

An escape character followed by an ascii small n denotes an ascii newline character. (A newline character inside a string also denotes a newline character, so the newline character is available in two, different ways).

An escape character followed by a minus sign denotes no character.

An escape character followed by a plus sign makes the scanner skip spaces until a non-space is found. In this context, comments and newline characters act as spaces. Hence "ab"+ {this is a comment} c" denotes the three-letter-string consisting of a, b, and c. "ab"+ {comment} "-c" also denotes this three-letter-string, but may be more readable because of the escape-minus sequence. "ab"+ {comment} "- " denotes the three-letter-string consisting of an ascii small a, an ascii small b, and an ascii space character.

An escape character followed by a semicolon makes the scanner ignore all characters until the end of the line.

An escape character followed by a left brace makes the scanner ignore all characters until the first right brace.

A sequence of escape characters acts as a single escape character.
|#

(deff f-parse-soft-string ()
 (:let pivot *pivot*)
 (:when (no-item (f-parse-separator "\"")) (soft-complaint pivot))
 (f-parse-string-body (char-code #\")))

(deff f-parse-string ()
 (:let result (f-parse-soft-string))
 (:when (is-item result) result)
 (f-error "String expected"))

(deff f-parse-string-body (escape)
 (:let pivot *pivot*)
 (:let x (f-parse-char))
 (:let y (f-parse-char))
 (:when (and (equal x (char-code #\")) (equal y (char-code #\.))) nil)
 (setq *pivot* pivot)
 (f-parse-string-body1 escape nil))

; changed from (< 0 card 128) because of trouble with tabulation characters
; in codify.lisp. Hence, strings cannot contain tabulation characters.
; Might need another iteration some time.
(deff f-ascii-char-p (card)
 (or (<= 32 card 127) (= card f-newline)))

(deff f-parse-string-body1 (escape result)
 (:let char (f-parse-char))
 (:when (no-item char)
  (f-parse-error "End of file in string: missing ~s" escape))
 (:unless (f-ascii-char-p char)
  (f-parse-error "Illegal character in string: code ~d" char))
 (:when (unequal char escape) (f-parse-string-body1 escape (cons char result)))
 (f-parse-string-body2 escape result))

(deff f-parse-string-body2 (escape result)
 (:let pivot *pivot*)
 (:let char (f-parse-char))
 (:when (no-item char) (setq *pivot* pivot) (reverse result))
 (:when (member char f-spaces) (setq *pivot* pivot) (reverse result))
 (:when (member char (ct2ct " ,.[]()<>")) (setq *pivot* pivot) (reverse result))
 (:when (equalp char (ct2ct #\n))
  (f-parse-string-body1 escape (cons f-newline result)))
 (:when (equalp char (ct2ct #\!))
  (f-parse-string-body1 escape (cons escape result)))
 (:when (equalp char (ct2ct #\?))
  (:let char (f-parse-char))
  (:when (no-item char)
   (f-parse-error "End of file in string: missing ~s" escape))
  (f-parse-string-body1 char result))
 (:when (equalp char (ct2ct #\+))
  (f-skip-space)
  (f-parse-string-body1 escape result))
 (:when (equalp char (ct2ct #\-)) (f-parse-string-body1 escape result))
 (:when (equalp char f-short)
  (f-skip-comment f-newline)
  (f-parse-string-body1 escape result))
 (:when (equalp char f-long-begin)
  (f-skip-long-comment escape f-long-end)
  (f-parse-string-body1 escape result))
 (:when (equalp char (ct2ct #\")) (f-parse-string-body2 escape result))
 (:when (equalp char (ct2ct #\/))
  (:let safe* (f-parse-string-binary escape nil))
  (f-parse-string-body2 escape (revappend-safe* safe* result)))
 (f-parse-error "Unknown escape code: ~s" char))

(deff f-parse-string-binary (escape result)
 (:let char (f-parse-char))
 (:when (no-item char)
  (f-parse-error "End of file in binary string."))
 (:when (equalp char escape) (reverse result))
 (f-parse-string-binary escape (cons char result)))

(set-source "\"xyz\"")
(etst (f-parse-string) (ct2ct "xyz"))
(set-source "\"\".")
(etst (f-parse-string) (ct2ct ""))
(ttst (no-item (f-parse-char)))

(set-source "\"xyz\" ")
(etst (f-parse-string) (ct2ct "xyz"))
(ttst (is-item (f-parse-space)))

(set-source "\"xyz\" a")
(etst (f-parse-string) (ct2ct "xyz"))
(ttst (is-item (f-parse-space)))
(etst (f-parse-char) 97)

(set-source "\"ab\"+\"\"{comment\"\"}c\"")
(etst (f-parse-string) (ct2ct "abc"))

(set-source "\"ab\"+\"\"{comment\"\"}\"-c\"")
(etst (f-parse-string) (ct2ct "abc"))

(set-source "\"\"?+abc+")
(etst (f-parse-string) (ct2ct "abc"))

(set-source "\"\"?+a+!+")
(etst (f-parse-string) (ct2ct "a+"))

(set-source "\"\"?++n+")
(etst (f-parse-string) (list f-newline))

(set-source "\"\"")
(ntst (f-parse-string))

(set-source "abcd")
(xtst (f-parse-string))

(set-source "\"abc")
(xtst (f-parse-string))

(set-source "\"ab\"{de\"\"}fg\"")
(etst (f-parse-string) (ct2ct "abfg"))

(set-source "\"ab\"\"{de\"\"}fg\"")
(etst (f-parse-string) (ct2ct "abfg"))

(set-source "\"ab\"\"\"{de\"\"}fg\"")
(etst (f-parse-string) (ct2ct "abfg"))

(set-source "\"ab\"\"\"\"{de\"\"}fg\"")
(etst (f-parse-string) (ct2ct "abfg"))

(set-source "\"ab\"/AA AAIA\"-de\"")
(etst (f-parse-string) (ct2card* '("ab" 0 0 0 8 "de")))

#|
=============================================
Parse decimal number
=============================================

(f-digit char) checks whether or not char is a digit; it relies on the ASCII character encoding which assigns consequtive codes to digits.

(f-parse-decimal pivot) parses one decimal number. Absense of a number is represented by nil.

|#

(deff f-digit (char)
 (:unless (integerp char) nil)
 (:let digit (- char (char-code #\0)))
 (when (<= 0 digit 9) digit))

(deff f-parse-decimal ()
 (f-skip-space)
 (f-parse-decimal1 nil))

(deff f-parse-decimal1 (result)
 (:let pivot *pivot*)
 (:let char (f-parse-char))
 (:let digit (f-digit char))
 (:when (null digit) (setq *pivot* pivot) result)
 (:let result (default 0 result))
 (:let result (+ digit (* 10 result)))
 (f-parse-decimal1 result))

(set-source "123a")
(etst (f-parse-decimal) 123)
(etst (f-parse-char) 97)

(set-source "a")
(ntst (f-parse-decimal))
(etst (f-parse-char) 97)

#|
=============================================
Check for being a small English letter
=============================================

(f-small-letter char) checks whether or not char is a small English letter; it relies on the ASCII character encoding which assigns consequtive codes to small letters. The definition of 'small letter' has been ad hoc extended by all characters that are not used for other purposes at present. The reserved characters are: Capital letters (keywords), digits (construct ids), double quotes (strings), asterisks (placeholders in constructs), commas (separator between constructs), semicolons (comment initiators), and braces (long comments).
|#

(deff f-small-letter (char)
 (and (integerp char)
  (or
   (<= (char-code #\a) char (char-code #\z))
   (position char "!#$%&'()+-./:<=>?@[\]^_`|~"))))

(ttst (f-small-letter (char-code #\a)))
(ttst (f-small-letter (char-code #\b)))
(ttst (f-small-letter (char-code #\z)))
(ntst (f-small-letter (char-code #\A)))
(ntst (f-small-letter (char-code #\.)))
(ntst (f-small-letter :no-item))

#|
=============================================
Parse kana
=============================================
(f-parse-kana pivot) parses kana bytes until encountering a character which is not a small letter. Spaces are allowed between but not inside kana bytes.
|#

(deff f-parse-kana-consonant ()
 (:let char (f-parse-char))
 (:let card (consonant2card char))
 (:when (null card) (f-parse-error "Expected kana consonant (ntsk)"))
 card)

(deff f-parse-kana-vovel ()
 (:let char (f-parse-char))
 (:let card (vovel2card char))
 (:when (null card) (f-parse-error "Expected kana vovel (aiue)"))
 card)

(deff f-parse-kana-sylable ()
 (:let x (f-parse-kana-consonant))
 (:let y (f-parse-kana-vovel))
 (+ (* 4 x) y))

(deff f-parse-kana-byte ()
 (:let x (f-parse-kana-sylable))
 (:let y (f-parse-kana-sylable))
 (+ (* 16 x) y))

(deff f-parse-kana ()
 (:let url (f-parse-soft-string))
 (:when (is-item url) (card2ref (c-url2ref url)))
 (f-parse-kana1 nil))

(deff f-parse-kana1 (result)
 (f-skip-space)
 (:let pivot *pivot*)
 (:let char (f-parse-char))
 (setq *pivot* pivot)
 (:unless (f-small-letter char) (reverse result))
 (:let byte (f-parse-kana-byte))
 (f-parse-kana1 (cons byte result)))

(set-source "nana nani nata nina tana")
(etst (f-parse-kana) '(0 1 4 16 64))

(set-source "na na")
(xtst (f-parse-kana))

(set-source "naan")
(xtst (f-parse-kana))

(set-source "x")
(xtst (f-parse-kana))

(set-source "X")
(ntst (f-parse-kana))

#|
=============================================
Parse line
=============================================
|#

(deff f-parse-line ()
 (f-parse-line1 nil))

(deff f-parse-line1 (result)
 (:let char (f-parse-char))
 (:when (no-item char) (reverse result))
 (:when (equalp char f-newline) (reverse result))
 (f-parse-line1 (cons char result)))

(set-source "a b c")
(etst (f-parse-line) (ct2ct "a b c"))
(etst (f-parse-line) (ct2ct ""))
(set-source "a b c
d")
(etst (f-parse-line) (ct2ct "a b c"))
(etst (f-parse-line) (ct2ct "d"))
(etst (f-parse-line) (ct2ct ""))

#|
=============================================
Parse page
=============================================
|#

(deff f-parse-page-decl ()
 (:let keyword (f-parse-keyword))
 (:when (unequal keyword (ct2ct "PAGE"))
  (f-error "File must start with page declaration"))
 (:let page (f-parse-line))
 (:let page (f-internalize page))
 (:when (null page) (f-error "Missing page name"))
 (:when (unequal (f-arity page) 0) (f-error "Page name must be nulary"))
 page)

(set-source "  PAGE  a b c 
")
(etst (f-parse-page-decl) (ct2ct "a b c"))

(set-source "page a b c 
")
(xtst (f-parse-page-decl))

(set-source "PAGE 
")
(xtst (f-parse-page-decl))

(set-source "PAGE a\"b
")
(xtst (f-parse-page-decl))

#|
=============================================
Parse bib
=============================================

(f-parse-bib pivot) parses a bib structure such as
  BIBLIOGRAPHY local name "filename" kana, another page "" kata kana
and returns it as a bibliography structure such as
  ((" local name" "filename" " kana") (" another page" "" " kata kana"))

A bib structure may be empty

|#

(deff f-print-bib (bib)
 (format t "~%BIBLIOGRAPHY~%")
 (f-print-bib1 bib))

(deff f-print-bib1 (bib)
 (:when (null bib) nil)
 (:let (bib-item . bib) bib)
 (f-print-bib-item bib-item)
 (f-print-bib1 bib))

(deff f-print-bib-item (bib-item)
 (:let (page filename reference) bib-item)
 (format t "~%~a ~s~%" (utf2string page) (card*2string filename))
 (f-print-kana reference -1))

(deff f-print-kana (reference index)
 (:when (null reference) (terpri))
 (when (= index 0) (terpri))
 (when (= index 5) (format t " "))
 (format t " ~a" (ct2string (card2kana (car reference))))
 (f-print-kana (cdr reference) (mod (+ index 1) 10)))

(deff f-parse-bib ()
 (:let keyword (f-parse-keyword))
 (:when (unequal keyword (ct2ct "BIBLIOGRAPHY"))
  (f-error "Missing 'BIBLIOGRAPHY' section"))
 (:let bib (f-parse-bib1 nil))
 (:let name* (map 'list 'first bib))
 (:let ref* (map 'list 'second bib))
 (list name* ref*))

(deff f-parse-bib1 (result)
 (:let pivot *pivot*)
 (:let keyword (f-parse-keyword-or-eof))
 (setq *pivot* pivot)
 (:when (is-item keyword) (reverse result))
 (:let bib-item (f-parse-bib-item))
 (:let result (cons bib-item result))
 (:when (no-item (f-parse-separator ".,")) (reverse result))
 (f-parse-bib1 result))

(deff f-parse-bib-item ()
 (:let name (f-parse-soft-string))
 (:when (no-item name)
  (f-error "Missing page name in bibliographic entry"))
 (:let name (f-internalize name))
 (:let reference (f-parse-kana))
 (:let bib-item (list name reference))
 bib-item)

(set-source "  BIBLIOGRAPHY  \"a\" nana, \"b  cd\" nani nata")
(etst (f-parse-bib) (ct2ct '(("a" "b cd") ((0) (1 4)))))

(set-source "  BIBLIOGRAPHY  BODY")
(etst '(nil nil) (f-parse-bib))

(set-source "page a b c X")
(xtst (f-parse-bib))

(set-source "BIBLIOGRAPHY \"x\" a, \"y\" aa BODY")
(xtst (f-parse-bib))

(set-source "BIBLIOGRAPHY 117 \"a\" nani BODY")
(xtst (f-parse-bib))

#|
=============================================
Assoc structures
=============================================

Assoc structures are structures of form

    ((card* name id ref pre post)
     ...
     (card* name id ref pre post))

The card* is the name of the construct in question.

The name is the name of the home page of the construct

The id can be a cardinal or NIL. The id is NIL for constructs introduced on the page being parsed in case the user does not specify an id. In that case, f-parse-assoc assigns an id before adding the construct to the grammar.

The ref is a relative reference, i.e. an index into the bibliography of the page.

Pre and post are cardinals or nil and encode priority, associativity, and fixity.

Priorities are encoded by even cardinals p where a small number indicates a high priority.

Pre is nil if the construct is preclosed. If the construct is preopen and has priority p then pre has value p+1 if the construct is preassociative and p otherwise.

Post is nil if the construct is protclosed. If the construct is postopen and has priority p then post has value p if the construct is preassociative and p+1 otherwise.

(f-print-assoc assoc) pretty prints an assoc structure.

|#

(deff f-print-assoc (assoc)
 (format t "  id  ref prio  pre post name:construct~%")
 (f-print-assoc1 assoc))

(deff f-print-assoc1 (assoc)
 (:when (null assoc) nil)
 (:let ((construct name prio id ref pre post) . assoc) assoc)
 (:let name (utf2string (f-externalize name)))
 (:let construct (utf2string (f-externalize construct)))
 (format t "~4@a ~4@a ~4@a ~4@a ~4@a ~a:~a~%"
  id ref prio pre post name construct)
 (f-print-assoc1 assoc))

#|
=============================================
Check assoc
=============================================
(check-assoc assoc ref* state) checks that all constructs defined in referenced pages are imported. If not, prints a warning.

Warnings may be suppressed by option header=nowarn.

Option header=suggest makes the compiler suggest a header which imports all constructs.

Option header=html is like header=suggest but wraps the output in html <textarea> blocks
|#

(deff f-check-assoc (assoc ref* pivot state)
 (:let header (option "header"))
 (:when (equalp header "nowarn") nil)
 (:when (equalp header "suggest") (f-suggest assoc ref* state))
 (:when (equalp header "html") (f-suggest-html assoc ref* pivot state))
 (:let missing (f-missing assoc ref* state))
 (:let ((ref . id)) missing)
 (:when (null ref) nil)
 (:let symbol (utf2string (c-symbol2name ref id state)))
 (:let page (utf2string (c-symbol2name ref 0 state)))
 (format t "Unimported ~s construct:~%~a~%" page symbol))

#|
(deff f-ref*2classes (ref* state result)
 (:when (atom ref*) result)
 (:let (ref . ref*) ref*)
 (:let symbol** (f-get-prio ref state))
 (:let result (f-symbol**2classes ref symbol** state result))
 (f-ref*2classes ref* state result))

(deff f-symbol**2classes (ref symbol** state result)
 (:when (atom symbol**) result)
 (:let ((:direct . symbol*) . symbol**) symbol**)
 (:let result (f-symbol*2classes ref symbol* symbol* state result))
 (f-symbol**2classes ref symbol** state result))

(deff f-symbol*2classes (ref symbols1 symbols2 state result)
 (:when (atom symbols1) result)
 (:let ((sref . sid) . symbols1) symbols1)
 (:let result (aput result symbols2 ref sref sid))
 (f-symbol*2classes ref symbols1 symbols2 state result))

(deff f-prune-missing (missing classes result)
 (:when (atom missing) result)
 (:let (symbol . missing) missing)
 (:let result (cons symbol result))
 (:let (ref . id) symbol)
 (:let class (aget classes ref ref id))
 (:let missing (set-difference missing class :test 'equalp))
 (f-prune-missing missing classes result))

(deff f-array2index (array)
 (:when (atom array) nil)
 (:let (left . right) array)
 (:when (integerp left) left)
 (:when (atom left) (f-array2index right))
 (f-array2index left))

(ntst (f-array2index nil))
(etst (f-array2index (aput nil t 1)) 1)
(etst (f-array2index (aput nil t 2)) 2)
(etst (f-array2index (aput (aput nil t 1) t 2)) 2)
(etst (f-array2index (aput (aput nil t 0) t 2)) 0)

(deff f-array2index* (array n)
 (f-array2index*1 array n nil))

(deff f-array2index*1 (array n result)
 (:when (= n 0) (reverse result))
 (:let index (f-array2index array))
 (:when (null index) nil)
 (f-array2index*1 (aget array index) (- n 1) (cons index result)))

(etst (f-array2index* (aput nil t 1 2 3) 3) '(1 2 3))
|#

#|
=============================================
Identify missing constructs
=============================================
(f-missing assoc ref* state) returns the list of constructs defined on pages pointed out by ref* which do not occur in assoc.

For each assoc item, (f-item2array item result) adds the symbol associated to the item to the result. The result is a set represented as an array which maps symbols to T.

(f-assoc2array assoc nil) returns the set of all symbols mentioned in the given assoc.
|#

(deff f-item2array (item result)
 (:let (:construct :page :section id ref) item)
 (:unless (integerp ref) result)
 (:unless (integerp id) result)
 (aput result t ref id))

(deff f-assoc2array (assoc result)
 (:when (atom assoc) result)
 (:let (item . assoc) assoc)
 (:let result (f-item2array item result))
 (f-assoc2array assoc result))

(deff f-missing (assoc ref* state)
 (f-missing0 (f-assoc2array assoc nil) ref* 1 state nil))

(deff f-missing0 (array ref* relref state result)
 (:when (atom ref*) result)
 (:let (ref . ref*) ref*)
 (:let dictionary (rack-get state ref card-dictionary))
 (:let result (f-missing1 (aget array relref) ref (adom dictionary) result))
 (f-missing0 array ref* (+ relref 1) state result))

(deff f-missing1 (array ref id* result)
 (:when (atom id*) result)
 (:let (id . id*) id*)
 (:when (aget array id) (f-missing1 array ref id* result))
 (:let result (acons ref id result))
 (f-missing1 array ref id* result))

#|
=============================================
Identify missing constructs
=============================================
(f-missing* import* ref* state result) returns the list of constructs defined on pages pointed out by ref* which do not occur in import*. Contrary to f-missing, import* is a list of symbols and the return value has form
  ((direct . (ref . id)) ...
where direct is :pre or :post. f-missing* is called from f-suggest.
|#

(deff f-missing* (import* ref* state result)
 (:when (atom ref*) result)
 (:let (ref . ref*) ref*)
 (:let symbol** (f-get-prio ref state))
 (:let result (f-missing*1 import* ref symbol** result))
 (f-missing* import* ref* state result))

(deff f-missing*1 (import* ref symbol** result)
 (:when (atom symbol**) result)
 (:let ((direct . symbol*) . symbol**) symbol**)
 (:let symbol (find ref symbol* :key 'car :test 'equalp))
 (:when (or (null symbol) (member symbol import* :test 'equal))
  (f-missing*1 import* ref symbol** result))
 (f-missing*1 import* ref symbol** (acons direct symbol result)))

#|
=============================================
Construct mapping to canonical elements
=============================================
Each associativity section may be seen as a class in a class division. Each symbol is introduced in exactly one associativity section, and for each symbol we shall refer to the first introduced symbol of that section as the 'canonical representation' of the class. (f-ref*2norm ref* state nil) returns an array 'norm' for which (aget norm ref id) returns the canonical representation of the class to which (ref . id) belongs. (f-ref*2norm ref* state nil) treats all symbols introduced on pages transitively referenced from ref*.

(f-ref2norm ref state result) is like (f-ref*2norm ref* state result) except that it only treats pages transitively referenced from the given ref. f-ref2norm returns 'result' unchanged if 'result' already has an entry for the given ref (indicating that the given ref has already been processed).
|#

(deff f-ref*2norm (ref* state result)
 (:when (atom ref*) result)
 (:let (ref . ref*) ref*)
 (:let result (f-ref2norm ref state result))
 (f-ref*2norm ref* state result))

(deff f-ref2norm (ref state result)
 (:when (aget result ref) result)
 (:let symbol** (f-get-prio ref state))
 (:let result (f-symbol**2norm ref symbol** state result))
 (:let (:ref . ref*) (rack-get state ref card-bibliography))
 (f-ref*2norm ref* state result))

(deff f-symbol**2norm (ref symbol** state result)
 (:when (atom symbol**) result)
 (:let ((:direct . symbol*) . symbol**) symbol**)
 (:let symbol (find ref symbol* :key 'car :test 'equalp))
 (:let result (f-symbol*2norm ref symbol* symbol state result))
 (f-symbol**2norm ref symbol** state result))

(deff f-symbol*2norm (ref symbol* symbol state result)
 (:when (atom symbol*) result)
 (:let ((ref1 . id) . symbol*) symbol*)
 (:when (unequal ref ref1) (f-symbol*2norm ref symbol* symbol state result))
 (:let result (aput result symbol ref id))
 (f-symbol*2norm ref symbol* symbol state result))

#|
=============================================
Construct ordering relation
=============================================
|#

(deff f-ref*2order (ref* norm state result)
 (:when (atom ref*) result)
 (:let (ref . ref*) ref*)
 (:let result (f-ref2order ref norm state result))
 (f-ref*2order ref* norm state result))

(deff f-print-canon** (canon** state)
 (format t "~%canon**=~%")
 (dolist (canon* canon**) (f-print-canon* canon* state)))

(deff f-print-canon* (canon* state)
 (format t "ASSOCIATIVE~%")
 (dolist (canon canon*) (f-print-canon canon state)))

(deff f-print-canon (canon state)
 (:let (ref . id) canon)
 (:let page (utf2string (c-symbol2name ref 0 state)))
 (:let construct (utf2string (c-symbol2name ref id state)))
 (format t "~s ~a~%" page construct))

(deff f-canon**2order (smaller greater result)
 (:when (atom greater) result)
 (:let (current . greater) greater)
 (:let smaller1 (apply 'append (cons current smaller)))
 (:let greater1 (apply 'append (cons current greater)))
 (:let result (f-canon*2order current smaller1 greater1 result))
 (:let smaller (cons current smaller))
 (f-canon**2order smaller greater result))

(deff f-canon*2order (canon* smaller greater result)
 (:when (atom canon*) result)
 (:let ((ref . id) . canon*) canon*)
 (:let (smaller1 . greater1) (aget result ref id))
 (:let smaller (set-difference smaller smaller1 :test 'equal))
 (:let greater (set-difference greater greater1 :test 'equal))
 (:let smaller (append smaller smaller1))
 (:let greater (append greater greater1))
 (:let result (aput result (cons smaller greater) ref id))
 (f-canon*2order canon* smaller greater result))

(deff f-ref2order (ref norm state result)
 (:when (aget result ref) result)
 (:let (:ref . ref*) (rack-get state ref card-bibliography))
 (:let result (f-ref*2order ref* norm state result))
 (:let symbol** (f-get-prio ref state))
 (:let canon** (f-symbol**2canon** ref symbol** norm nil))
 (:let result (f-canon**2order nil canon** result))
 result)

(deff f-add-symbol-2-canon (symbol canon)
 (:when (null symbol) canon)
 (:when (member symbol canon) canon)
 (cons symbol canon))

(deff f-symbol**2canon** (ref symbol** norm result)
 (:when (atom symbol**) (reverse result))
 (:let ((:direct . symbol*) . symbol**) symbol**)
 (:let symbol (find ref symbol* :key 'car :test 'equalp))
 (:let canon* (when symbol (list symbol)))
 (:let canon* (f-symbol*2canon* ref symbol* norm canon*))
 (:let result (cons canon* result))
 (f-symbol**2canon** ref symbol** norm result))

(deff f-symbol*2canon* (ref symbol* norm result)
 (:when (atom symbol*) result)
 (:let ((ref1 . id) . symbol*) symbol*)
 (:when (equalp ref ref1) (f-symbol*2canon* ref symbol* norm result))
 (:let symbol (aget norm ref1 id))
 (:when (member symbol result :test 'equalp) result)
 (cons symbol result))

#|
=============================================
Reorganize assoc
=============================================
(f-assoc-reorg assoc ref* norm result) reorganizes assoc from form ((construct page section id ref) ...) to ((associativity import* . local*) ...). Each (associativity import* . local*) item represents one associativity section. associativiey is :pre or :post. import* has form ((ref . id) ...). local* has form (string ...).
|#

; Split assoc in first section and rest
(deff f-split-assoc (assoc section result)
 (:when (atom assoc) (cons (reverse result) nil))
 (:let (item . assoc1) assoc)
 (:let (:construct :page section1) item)
 (:when (unequal section section1) (cons (reverse result) assoc))
 (f-split-assoc assoc1 section (cons item result)))

; Split assoc section in imports and local constructs
(deff f-split-assoc1 (assoc ref* norm import* local*)
 (:when (atom assoc) (cons (remove-duplicates import* :test 'equalp) local*))
 (:let (item . assoc) assoc)
 (:let (construct :page :section id ref) item)
 (:when (= ref 0)
  (:let construct (card*2string construct))
  (f-split-assoc1 assoc ref* norm import* (cons construct local*)))
 (:let ref (nth (- ref 1) ref*))
 (:let symbol (aget norm ref id))
 (f-split-assoc1 assoc ref* norm (cons symbol import*) local*))

(deff f-assoc-reorg (assoc ref* norm result)
 (:when (atom assoc) result)
 (:let ((:construct :page section) . assoc0) assoc)
 (:unless (integerp section) (f-assoc-reorg assoc0 ref* norm result))
 (:let (assoc1 . assoc2) (f-split-assoc assoc section nil))
 (:let associativity (if (evenp section) :post :pre))
 (:let sect (cons associativity (f-split-assoc1 assoc1 ref* norm nil nil)))
 (f-assoc-reorg assoc2 ref* norm (cons sect result)))

#|
=============================================
Print suggestion
=============================================
(f-print-suggestion assoc state) prints the given reorganized assoc which has format ((:pre|:post import* . local*) ...).
|#

(deff f-print-import* (import* state)
 (:when (atom import*) nil)
 (:let ((ref . id) . import*) import*)
 (:let construct (utf2string (c-symbol2name ref id state)))
 (:let page (utf2string (c-symbol2name ref 0 state)))
 (format t "~s ~a~%" page construct)
 (f-print-import* import* state))

(deff f-print-suggestion (assoc state)
 (:when (atom assoc) nil)
 (:let ((associativity import* . local*) . assoc) assoc)
 (format t "~aASSOCIATIVE~%" associativity)
 (f-print-import* import* state)
 (dolist (local local*) (format t "~s ~a~%" "" (utf2string local)))
 (format t "~%")
 (f-print-suggestion assoc state))

#|
=============================================
Add missing constructs
=============================================
|#

(deff f-find-smaller (ref id order result)
 (:when (aget result ref id) result)
 (:let result (aput result t ref id))
 (:let (smaller . :greater) (aget order ref id))
 (f-find-smaller* smaller order result))

(deff f-find-smaller* (symbol* order result)
 (:when (atom symbol*) result)
 (:let ((ref . id) . symbol*) symbol*)
 (:let result (f-find-smaller ref id order result))
 (f-find-smaller* symbol* order result))

(deff f-find-greater (ref id order result)
 (:when (aget result ref id) result)
 (:let result (aput result t ref id))
 (:let (:smaller . greater) (aget order ref id))
 (f-find-greater* greater order result))

(deff f-find-greater* (symbol* order result)
 (:when (atom symbol*) result)
 (:let ((ref . id) . symbol*) symbol*)
 (:let result (f-find-greater ref id order result))
 (f-find-greater* symbol* order result))

(deff f-add-missing* (missing* order assoc)
 (:when (atom missing*) assoc)
 (:let ((direct . missing) . missing*) missing*)
 (:let (ref . id) missing)
 (:let smaller (f-find-smaller ref id order nil))
 (:let greater (f-find-greater ref id order nil))
 (:let assoc (f-add-missing direct missing smaller greater assoc))
 (f-add-missing* missing* order assoc))

(deff f-disjoint (symbol* array)
 (:when (atom symbol*) t)
 (:let ((ref . id) . symbol*) symbol*)
 (:when (aget array ref id) nil)
 (f-disjoint symbol* array))

(deff f-add-missing (direct missing smaller greater assoc)
 (:when (atom assoc) (list (list :pre (list missing))))
 (:let (item . assoc1) assoc)
 (:let (associativity import* . local*) item)
 (:when (f-disjoint import* greater)
  (cons item (f-add-missing direct missing smaller greater assoc1)))
 (:when (f-disjoint import* smaller) (cons (list direct (list missing)) assoc))
 (cons (list* associativity (cons missing import*) local*) assoc1))

#|
=============================================
Print ordering
=============================================
|#

(deff f-print-order (order state)
 (format t "order=~%")
 (dolist (ref (adom order)) (f-print-order1 ref (aget order ref) state)))

(deff f-print-order1 (ref order state)
 (dolist (id (adom order)) (f-print-order2 ref id (aget order id) state)))

(deff f-print-order2 (ref id order state)
 (:let (smaller . greater) order)
 (format t "~%")
 (format t "Order for ~a~%" (utf2string (c-symbol2name ref id state)))
 (format t "Smaller:~%")
 (f-print-symbol* smaller state)
 (format t "Greater:~%")
 (f-print-symbol* greater state))

(deff f-print-symbol* (symbol* state)
 (:when (atom symbol*) nil)
 (:let ((ref . id) . symbol*) symbol*)
 (format t "  ~a~%" (utf2string (c-symbol2name ref id state)))
 (f-print-symbol* symbol* state))

#|
=============================================
Print normalisation
=============================================
Print the mapping from symbols to canonical symbols
|#

(deff f-print-norm (norm state)
 (f-print-norm1 norm (adom norm) state))

(deff f-print-norm1 (norm ref* state)
 (:when (atom ref*) nil)
 (:let (ref . ref*) ref*)
 (f-print-norm2 (aget norm ref) ref state)
 (f-print-norm1 norm ref* state))

(deff f-print-norm2 (norm ref state)
 (f-print-norm3 norm ref (adom norm) state))

(deff f-print-norm3 (norm ref id* state)
 (:when (atom id*) nil)
 (:let (id . id*) id*)
 (f-print-norm4 (aget norm id) ref id state)
 (f-print-norm3 norm ref id* state))

(deff f-print-norm4 (norm ref id state)
 (:let name1 (utf2string (c-symbol2name ref id state)))
 (:let (ref . id) norm)
 (:let name2 (utf2string (c-symbol2name ref id state)))
 (format t "~a -> ~a~%" name1 name2))

#|
=============================================
Construct suggestion
=============================================
(f-suggest assoc ref* state) extends the given assoc with missing constructs from pages directly referenced from ref*. It does so based on priorities and associativities of all pages transitively referenced from ref*.

f-suggest first computes an array norm such that (aget norm ref id) returns the canonical element of the priority class containing the symbol with the given ref and id. Two symbols belong to the same priority class iff they have the same associated canonical element. Two symbols belong to the same priority class iff they were introduced on the same page and in the same associativity section.

f-suggest then computes and 'order' such that (aget order ref id) is a pair (smaller* . greater*) where smaller* is a list of constructs with weakly smaller priority than the symbol with the given ref and id (and similar for greater*).

f-suggest then reorganizes assoc to have form
  ((direct import* . local*) ...)
where direct is :pre or :post, import* is a list of imports and local* is a list of local constructs.

f-suggest then cleans assoc for duplicate imports. Duplicate imports may occur if distinct symbols from the same priority class are imported with different priority. Simultaneously, f-suggest constructs the list import* of all imports.

f-suggest then finds out which constructs are missing by subtracting import* from the set of all constructs to be imported.

Finally, f-suggest adds the missing constructs.
|#

(deff f-assoc-remove-duplicates (assoc result1 result2)
 (:when (atom assoc) (cons (reverse result1) result2))
 (:let ((associativity import* . local*) . assoc) assoc)
 (:let import* (set-difference import* result2 :test 'equal))
 (:let result2 (append import* result2))
 (:let result1 (cons (list* associativity import* local*) result1))
 (f-assoc-remove-duplicates assoc result1 result2))

; (lgc "dump" :url "http://thor/logiweb/page" :header :suggest)
; (lgc "dump1" :url "http://thor/logiweb/page" :header :suggest)

(deff f-suggest (assoc ref* state)
 (:let norm (f-ref*2norm ref* state nil))
 (:let order (f-ref*2order ref* norm state nil))
 (:let assoc (f-assoc-reorg assoc ref* norm nil))
 (:let (assoc . import*) (f-assoc-remove-duplicates assoc nil nil))
 (:let missing* (f-missing* import* ref* state nil))
 (:let assoc (f-add-missing* missing* order assoc))
 (format t "~%")
 (f-print-suggestion assoc state)
 (raise))

(deff f-suggest-html (assoc ref* pivot state)
 (:let norm (f-ref*2norm ref* state nil))
 (:let order (f-ref*2order ref* norm state nil))
 (:let assoc (f-assoc-reorg assoc ref* norm nil))
 (:let (assoc . import*) (f-assoc-remove-duplicates assoc nil nil))
 (:let missing* (f-missing* import* ref* state nil))
 (:let assoc (f-add-missing* missing* order assoc))
 (format t "</pre>~%")
 (format t "~%")
 (format t "<textarea name='pane1' rows=20 cols=80>~%")
 (format t "~a~%~%" (utf2string (subseq *source* 0 pivot)))
 (f-print-suggestion assoc state)
 (format t "</textarea>~%")
 (format t "~%")
 (format t "<br>~%")
 (format t "~%")
 (format t "<textarea name='pane2' rows=20 cols=80>~%")
 (format t "~a~%" (utf2string (subseq *source* *pivot*)))
 (format t "</textarea>~%")
 (format t "<pre>~%")
 (raise))

#|
=============================================
Parse assoc
=============================================

(f-parse-assoc name* ref* state) parses an assoc structure such as
  PREASSOCIATIVE "" " head
  117 "" " tail
  POSTASSOCIATIVE "base" " pair "
  PREASSOCIATIVE 17 "base" " equals "
and returns it as a grammar and an arity table.

Above '" head' indicates that the construct '" head' belongs to the page being parsed (relative reference zero) and that the Logiweb compiler should assign an identifier.

'117 "" " tail' indicates that the construct '" tail' belongs to the page being parsed and that its identifier is 117.

'"base" " pair "' imports the construct named '" pair "' from the page that is called "base" in the bibliography.

'17 "base" " equals "' imports construct with identifier 17 from the page that is called "base" in the bibliography and assigns name '" equals "' to it.

When importing a construct with a specified id, only that particular construct is imported.

When importing a construct without a specified id, all constructs from the given page that have the same priority on the referenced page are also imported and get the same priority as the imported construct on the referencing page.
|#

; Parse all associativity sections and convert to grammar/arity table
(deff f-parse-assoc (name* ref* state)
 (:let pivot *pivot*)
 (:let assoc (f-parse-assoc1 name* ref* state 0 nil))
 (f-check-assoc assoc ref* pivot state)
 (:let grammar (f-extract-grammar assoc nil))
 (:let arity (f-extract-arity assoc nil))
 (list grammar arity assoc))

(deff f-extract-grammar (assoc grammar)
 (:when (null assoc) grammar)
 (:let ((construct name :prio . node) . assoc) assoc)
 (:let node (cons 0 node))
 (:let grammar (grammar-addx construct name node grammar))
 (f-extract-grammar assoc grammar))

(deff grammar-addx (construct name node grammar)
 (:let grammar (grammar-add0 construct node grammar))
 (:when (null name) grammar)
 (:let node (cons (car node) (cdr node))) ; equal, not eql, for 'grammar' option
 (grammar-add0 (f-add-page-name name construct) node grammar))

(deff f-extract-arity (assoc result)
 (:when (null assoc) result)
 (:let ((construct :name :prio id ref) . assoc) assoc)
 (:when (null ref) (f-extract-arity assoc result))
 (:let arity (f-arity construct))
 (:when (equalp id 0) (f-extract-arity assoc result))
 (:let result (aput result arity ref id))
 (f-extract-arity assoc result))

(defc f-pre-assoc
 `((,(ct2ct "PREASSOCIATIVE") . 1)
   (,(ct2ct "POSTASSOCIATIVE") . 0)))

; Parse all associativity sections and return assoc
(deff f-parse-assoc1 (name* ref* state prio assoc)
 (:let pivot *pivot*)
 (:let keyword (f-parse-keyword))
 (:let pre-delta (assoc-get keyword f-pre-assoc))
 (:when (null pre-delta) (setq *pivot* pivot) (f-finish-assoc name* assoc))
 (:let pre (+ prio pre-delta))
 (:let post (- prio pre-delta -1))
 (:let assoc (f-parse-assoc2 name* ref* state pre post assoc))
 (f-parse-assoc1 name* ref* state (+ prio 2) assoc))

; Parse one associativity section and add to assoc
(deff f-parse-assoc2 (name* ref* state pre post assoc)
 (:let assoc (f-parse-assoc3 name* ref* state pre post assoc))
 (:let pivot *pivot*)
 (:let keyword (f-parse-keyword))
 (setq *pivot* pivot)
 (:when (is-item keyword) assoc)
 (f-parse-assoc2 name* ref* state pre post assoc))

(deff f-pre-open-p (char*)
 (equalp (car char*) f-placeholder))

(deff f-post-open-p (char*)
 (equalp (car (last char*)) f-placeholder))

(deff f-add-page-name (name construct)
 (:when (unequal (car construct) f-placeholder)
  (append name (cons f-space construct)))
 (list* f-placeholder f-space (append name (cdr construct))))

; Parse one associativity entry and add to assoc
(deff f-parse-assoc3 (name* ref* state pre post assoc)
 (:let id (f-parse-decimal)) ; nil if no id given
 (:let page (f-parse-soft-string)) ; nil if string empty
 (:when (no-item page) (f-error "Page name (string) expected"))
 (:let page (f-internalize page))
 (:let construct (f-internalize (f-parse-line)))
 (:when (null construct)
  (f-parse-error "Empty construct in associativity section"))
 (:when (null page) (f-define-construct construct name* id pre post assoc))
 (f-import-construct id page construct name* ref* state pre post assoc))

; Add new construct to assoc
(deff f-define-construct (construct name* id pre post assoc)
 (:let ref 0)
 (:let pre1 (when (f-pre-open-p construct) pre))
 (:let post (when (f-post-open-p construct) post))
 (cons (list construct (car name*) pre id ref pre1 post) assoc))

; Add imported constructs to assoc
(deff f-import-construct (id page construct name* ref* state pre post assoc)
 (:let position (position page (cdr name*) :test 'equalp))
 (:when (null position)
  (f-parse-error "In associativity section: unknown page: ~a"
   (f-externalize page)))
 (:let ref (nth position ref*))
 (:let relref (+ position 1))
 (:when id
  (f-import-one-construct id page construct ref relref state pre post assoc))
 (f-import-construct1 page construct ref relref state pre post assoc))

; Import one construct whose id is given
(deff f-import-one-construct (id page construct ref relref state pre post assoc)
 (:let arity (arity-get state ref id))
 (:when (unequal arity (f-arity construct))
  (format t "ref=~s~%" ref)
  (format t "id=~s~%" id)
  (format t "arity=~s~%" arity)
  (format t "arity2=~s~%" (f-arity construct))
  (f-parse-error "Arity mismatch in imported construct"))
 (:let pre1 (when (f-pre-open-p construct) pre))
 (:let post (when (f-post-open-p construct) post))
 (cons (list construct page pre id relref pre1 post) assoc))

; Import all constructs with same priority as given construct
(deff f-import-construct1 (page construct ref relref state pre post assoc)
 (:let id (f-name2id ref construct state))
 (:when (null id)
  (f-parse-error "Unknown construct: ~s" (f-externalize construct)))
 (:when (equalp id :ambiguous)
  (f-parse-error "Ambiguous construct: ~s" (f-externalize construct)))
 (:let id* (f-get-equal-prio ref id state))
 (:when (null id*)
  (f-parse-error "Construct not found in priority section of given page"))
 (f-add-id* page relref ref id* pre post state assoc))

(deff f-add-id* (page relref ref id* pre post state assoc)
 (:when (null id*) assoc)
 (:let (id . id*) id*)
 (:let definition (codex-get state ref id 0 card-name))
 (:when (null definition)
  (f-parse-error "Imported construct with id ~s has no name" id))
 (:let rhs (fourth definition))
 (:let construct (f-internalize (c-tree2card* rhs)))
 (:when (null construct)
  (f-parse-error "Imported construct with id ~s has empty name" id))
 (:when (equalp construct (ct2ct "*"))
  (f-parse-error "Imported construct with id ~s has illegal name" id))
 (:let arity (arity-get state ref id))
 (:when (unequal arity (f-arity construct))
  (format t "ref=~s~%" ref)
  (format t "id=~s~%" id)
  (format t "arity=~s~%" arity)
  (format t "arity2=~s~%" (f-arity construct))
  (f-parse-error "Arity mismatch for imported construct ~a"
   (f-externalize construct)))
 (:let pre1 (when (f-pre-open-p construct) pre))
 (:let post1 (when (f-post-open-p construct) post))
 (:let assoc (cons (list construct page pre id relref pre1 post1) assoc))
 (f-add-id* page relref ref id* pre post state assoc))

#|
=============================================
Convert name to id using state
=============================================
|#

(deff f-name2id (ref construct state)
 (:let codex (rack-get state ref card-codex))
 (:let assoc (aget codex ref))
 (f-name2id1 assoc construct))

(deff f-name2id1 (assoc construct)
 (:when (null assoc) nil)
 (:let (a1 . a2) assoc)
 (:when (numberp a1)
  (:let definition (aget a2 0 card-name))
  (:let rhs (fourth definition))
  (:let card* (c-tree2card* rhs))
  (:let construct1 (f-internalize card*))
  (when (equalp construct construct1) a1))
 (:let id1 (f-name2id1 a1 construct))
 (:let id2 (f-name2id1 a2 construct))
 (:when (null id1) id2)
 (:when (null id2) id1)
 :ambiguous)

#|
=============================================
Manipulate priority definitions
=============================================
|#

; Check whether or not the given ref/id occur in the given priority section
(deff f-prio-member (ref id tree)
 (:when (null tree) nil)
 (:let (:root tree1 tree2) tree)
 (:let ((ref1 id1)) tree1)
 (:when (and (equalp ref ref1) (equalp id id1)) t)
 (f-prio-member ref id tree2))

; Extract identifiers from given priority section
(deff f-prio2id* (ref tree result)
 (:when (null tree) (reverse result))
 (:let (:root tree1 tree2) tree)
 (:let ((ref1 id)) tree1)
 (:when (unequal ref ref1) (f-prio2id* ref tree2 result))
 (f-prio2id* ref tree2 (cons id result)))

; Convert ref/id to list of id's of all constructs with same priority
(deff f-get-equal-prio (ref id state)
 (:let prio (codex-get state ref 0 0 card-priority))
 (:let rhs (fourth prio))
 (f-get-equal-prio1 ref id rhs))

(deff f-get-equal-prio1 (ref id tree)
 (:let (:root tree1 tree2) tree)
 (:when (null tree1) nil)
 (:when (f-prio-member ref id tree1) (f-prio2id* ref tree1 nil))
 (f-get-equal-prio1 ref id tree2))

(deff f-get-prio (ref state)
 (:let prio (codex-get state ref 0 0 card-priority))
 (:let rhs (fourth prio))
 (f-get-prio1 rhs state nil))

(deff f-symbol2associativity (ref id state)
 (:let prio (codex-get state ref id 0 card-priority))
 (:when (null prio)
  (format t "Warning when scanning priority definition.~%")
  (format t "Expected pre/postassociativity construct~%")
  (format t "Found ~a~%" (c-symbol2name ref id state)))
 (:let rhs (fourth prio))
 (:let ((:ref id)) rhs)
 (when (null id) (format t "f-get-prio1 of null argument~%"))
 (:let vector (card2vector id))
 (:let string (vector2string vector))
 (:let direct (assoc-get string '(("pre" . :pre) ("post" . :post))))
 direct)

(deff f-get-prio1 (tree state result)
 (:let ((ref id) tree1 tree2) tree)
 (:when (null tree1) (reverse result))
 (:let direct (f-symbol2associativity ref id state))
 (:when (null direct) (reverse result))
 (:let result (acons direct (f-get-prio2 tree1 nil) result))
 (f-get-prio1 tree2 state result))

(deff f-get-prio2 (tree result)
 (:when (null tree) (reverse result))
 (:let (:root tree1 tree2) tree)
 (:let ((ref id)) tree1)
 (:let result (acons ref id result))
 (f-get-prio2 tree2 result))

#|
=============================================
Finish assoc
=============================================
(f-finish-assoc bib assoc) finishes the given assoc by
- adding the page symbol
- adding the *,* and *[*]* constructs
- assigning identifiers to constructs that have no identifier
|#

(deff f-finish-assoc (bib assoc)
 (:let assoc (f-add-id (reverse assoc)))
 (:let name (car bib))
`((,name   ,name           nil 0 0)
  (,(f-internalize (ct2ct *f-name* )) nil nil :name)
  (,(f-internalize (ct2ct *f-prio*)) nil nil :prio)
  ,@assoc))

(deff f-add-id (assoc)
 (:let id* (cons 0 (f-extract-id assoc nil)))
 (:let id* (safe-sort '< id*))
 (f-check-for-duplicates id*)
 (:let id* (my-remove-duplicates id*))
 (f-add-id1 assoc 0 id* nil))

(deff f-check-for-duplicates (id*)
 (:let (id . id*) id*)
 (:when (null id*) nil)
 (:when (unequal id (car id*)) (f-check-for-duplicates id*))
 (complain "Identifier specified more than once: ~d" id))

(deff f-add-id1 (assoc id id* result)
 (:when (null assoc) result)
 (:let (item . assoc) assoc)
 (:let (:construct :name :prio id1) item)
 (:when id1 (f-add-id1 assoc id id* (cons item result)))
 (f-add-id2 item assoc id id* result))

(deff f-add-id2 (item assoc id id* result)
 (:when (equal id (car id*)) (f-add-id2 item assoc (+ id 1) (cdr id*) result))
 (:let (construct name prio :id ref pre post) item)
 (f-add-id1 assoc (+ id 1) id*
  (cons (list construct name prio id ref pre post) result)))

(deff f-extract-id (assoc result)
 (:when (null assoc) result)
 (:let ((:construct :name :prio id ref) . assoc) assoc)
 (:when (null id) (f-extract-id assoc result))
 (:when (unequal ref 0) (f-extract-id assoc result))
 (f-extract-id assoc (cons id result)))

#|
=============================================
Flatten assoc structure
=============================================

An assoc is a list of priority equivalence classes where a priority equivalence class is a list of labelled constructs that have the same associativity and priority.

(flatten-assoc ref name assoc) changes the given assoc from a list of lists to a single list in an information lossless way. It does so by assigning a class id to each priority equivalence class and then distributing that class id to all members of the class. flatten-assoc uses the cardinals for class ids, reserving class id 0 for the page symbol.

Furthermore, flatten-assoc distributes the associativity (:preassociative or :postassociative) to each member of each class.

flatten-assoc does other things as well:

It distributes the given ref to allow to merge assocs of different pages without loss of information.

It distributes the given page name to each member of each class to allow generation of readable error messages and to disambiguate constructs.

It assigns ids to constructs that do not yet have one.

The return value from flatten-assoc has form
  (construct ...)
where constructs have form
  ((id ref name associativity class-id) char ...)

|#

(deff list2duplicates (id*)
 (list2duplicates1 id* nil))

(deff list2duplicates1 (id* result)
 (:let (id . id*) id*)
 (:when (null id*) (my-remove-duplicates (reverse result)))
 (:let result (if (equalp id (car id*)) (cons id result) result))
 (list2duplicates1 id* result))

(etst (list2duplicates '(1 2 2 2 3 4 4)) '(2 4))

(deff construct*2id* (construct*)
 (construct*2id*1 construct* nil))

(deff construct*2id*1 (construct* result)
 (:when (null construct*) (safe-sort '< result))
 (:let (((id)) . construct*) construct*)
 (:let result (if id (cons id result) result))
 (construct*2id*1 construct* result))

(etst
 (construct*2id*
  '(((  3 :ref :name :preassociative  1) . x)
    ((nil :ref :name :preassociative  1) . y)
    ((  1 :ref :name :postassociative 2) . z)
    ((  3 :ref :name :postassociative 2) . u)))
 '(1 3 3))

(deff flatten-assoc (ref name assoc)
 (:let construct* (flatten-assoc1 ref name 1 assoc nil))
 (:let id* (construct*2id* construct*))
 (:let duplicates (list2duplicates (cons 0 id*)))
 (:when duplicates
  (complain "Overloaded identifiers in ~a: ~s"
   (f-externalize name) duplicates))
 (:let construct* (add-ids-to-construct* construct* id* 1 nil))
 ; add page symbol
 (:let attributes (list 0 ref name :preassociative 0))
 (:let construct (cons attributes name))
 (cons construct construct*))

(deff add-ids-to-construct* (construct* id* index result)
 (:when (null construct*) (reverse result))
 (:when (caaar construct*)
  (:let (construct . construct*) construct*)
  (:let result (cons construct result))
  (add-ids-to-construct* construct* id* index result))
 (:when (equalp (car id*) index)
  (add-ids-to-construct* construct* (cdr id*) (+ index 1) result))
 (:let (((nil . attributes) . construct) . construct*) construct*)
 (:let result `(((,index . ,attributes) . ,construct) . ,result))
 (add-ids-to-construct* construct* id* (+ index 1) result))

(deff flatten-assoc1 (ref name index assoc result)
 (:when (null assoc) (reverse result))
 (:let ((direction . construct*) . assoc) assoc)
 (:let attributes (list ref name direction index))
 (:let result (flatten-construct* attributes construct* result))
 (flatten-assoc1 ref name (+ index 1) assoc result))

(deff flatten-construct* (attributes construct* result)
 (:when (null construct*) result)
 (:let ((id . construct) . construct*) construct*)
 (:let construct (acons id attributes construct))
 (:let result (cons construct result))
 (flatten-construct* attributes construct* result))

(etst
 (flatten-assoc :ref :name
  '((:preassociative (1 . x) (nil . y)) (:postassociative (nil . z) (3 . u))))
 '(((0 :ref :name :preassociative  0) . :name)
   ((1 :ref :name :preassociative  1) . x)
   ((2 :ref :name :preassociative  1) . y)
   ((4 :ref :name :postassociative 2) . z)
   ((3 :ref :name :postassociative 2) . u)))

(deff flatten-assoc* (name* assoc*)
 (flatten-assoc*1 0 name* assoc* nil))

(deff flatten-assoc*1 (ref name* assoc* result)
 (:when (null name*) result) 
 (:let construct* (flatten-assoc ref (car name*) (car assoc*)))
 (:let result (append construct* result))
 (flatten-assoc*1 (+ ref 1) (cdr name*) (cdr assoc*) result))

(etst
 (flatten-assoc* '(:name)
  '(((:preassociative (1 . x) (nil . y)) (:postassociative (nil . z) (3 . u)))))
 '(((0 0 :name :preassociative  0) . :name)
   ((1 0 :name :preassociative  1) . x)
   ((2 0 :name :preassociative  1) . y)
   ((4 0 :name :postassociative 2) . z)
   ((3 0 :name :postassociative 2) . u)))

#|
=============================================
Grammars
=============================================

A grammar is a list (chars parm . node) with the following properties:

chars is an association list from cardinals to grammars which indicates how to parse characters.

parm is a grammar which indicates how to continue after parsing a substructure.

node is a data structure that describes the "current" construct. One may see a grammar as a tree whose edges are labelled by characters (the keys in the 'chars' association list), and the "current" construct is the list of labels from the root to the present node of the grammar.

A node may have the following forms:



node = (0 id ref left right)

id is the id of the given construct (if any, and null otherwise)

ref is the relative reference of the given construct. The page indexed by ref in the bibliography is the home page of the construct.

left is the priority of the given construct (if any). left is nil if the construct is left-closed. It is one greater than right if the construct is left-associative.

right is analogous to left.



node = (1 id1 ref1 id2 ref2)

The current construct is ambiguous and denotes at least id1 at page ref1 and id2 at page ref2.



A grammar needs not contain trailing nil elements.

|#

#|
=============================================
Print grammar
=============================================
|#

(deff f-print-grammar (grammar)
 (format t "~%GRAMMAR~%")
 (f-print-grammar1 grammar nil))

(deff f-print-grammar1 (grammar construct)
 (:when (null grammar) nil)
 (:let (chars parm . node) grammar)
 (when node (f-print-grammar3 node construct))
 (f-print-grammar2 chars construct)
 (f-print-grammar1 parm (cons f-placeholder construct)))

(deff f-print-grammar2 (chars construct)
 (:when (null chars) nil)
 (:let ((key . grammar) . chars) chars)
 (f-print-grammar1 grammar (cons key construct))
 (f-print-grammar2 chars construct))

(deff f-print-grammar3 (node construct)
 (:let (type . attribute) node)
 (:let construct (utf2string (f-externalize (reverse construct))))
 (:when (equalp type 1)
  (format t "Ambiguous    ~a~%" construct))
 (:let (id ref left right) attribute)
 (format t "~4d ~3d ~3d ~3d ~a~%" id ref left right construct))

#|
=============================================
Print grammar visits
=============================================
Print the number of times each node in the given grammar has been visited.

*sample-hash* is a hash table from subgrammars to cardinals which indicates how often each subgrammar has been visited.

*node-hash* is a hash table from nodes to lists of subgrammars which indicates which path through the grammar leads to the given node.

(f-print-grammar-visits grammar tree*) calls f-make-node-hash to build up *node-hash*. Then it calls f-downcount-tree* to remove counts associated with successful parsing. Finally it calls f-print-grammar-visits1 to convert the counts into an association list from viable prefixes to counts. The reason why this is so complicated is that the grammar is optimized for parsing, not for producing the present statistics.
|#

(defc *sample-hash* (make-hash-table :test 'eql))
(defc *node-hash* (make-hash-table :test 'eql))
(defc *measure-grammar* nil)

(deff f-print-grammar-visits (grammar tree*)
 (format t "~%GRAMMAR VISITS~%")
 (f-make-node-hash grammar)
 (f-downcount-tree* (append tree* '(:dummy-pivot)))
 (:let assoc (f-print-grammar-visits1 grammar nil nil))
 (:let assoc (reverse assoc))
 (:let assoc (stable-sort assoc '> :key 'cdr))
 (dolist (x assoc)
  (when (>= (cdr x) *measure-grammar*)
   (format t "~7d |~a|~%" (cdr x) (utf2string (car x)))))
 (raise))

(deff f-make-node-hash (grammar)
 (clrhash *node-hash*)
 (f-make-node-hash1 grammar nil))

(deff f-make-node-hash1 (grammar path)
 (:when (null grammar) nil)
 (:let (chars parm . node) grammar)
 (:let path (cons grammar path))
 (when node (setf (gethash node *node-hash*) path))
 (f-make-node-hash1 parm path)
 (f-make-node-hash2 chars path))

(deff f-make-node-hash2 (chars path)
 (:when (null chars) nil)
 (:let ((:key . grammar) . chars) chars)
 (f-make-node-hash1 grammar path)
 (f-make-node-hash2 chars path))

(deff f-downcount-tree* (tree*)
 (:let (tree . tree*) tree*)
 (:when (atom tree*) nil);last element of list is a pivot into the source
 (f-downcount-tree tree)
 (f-downcount-tree* tree*))

(deff f-downcount-tree (tree)
 (:let (node . tree*) tree)
 (:let grammar* (gethash node *node-hash*))
 (f-downcount-grammar* grammar*)
 (f-downcount-tree* tree*))

(deff f-downcount-grammar* (grammar*)
 (:when (atom grammar*) nil)
 (:let (grammar . grammar*) grammar*)
 (decf (gethash grammar *sample-hash* 0))
 (f-downcount-grammar* grammar*))

(deff f-print-grammar-visits1 (grammar construct result)
 (:when (null grammar) result)
 (:let visits (gethash grammar *sample-hash* 0))
 (:let string (card*2string (reverse construct)))
 (:let result (acons string visits result))
 (:let (chars parm . :node) grammar)
 (:let result (f-print-grammar-visits2 chars construct result))
 (f-print-grammar-visits1 parm (cons f-placeholder construct) result))

(deff f-print-grammar-visits2 (chars construct result)
 (:when (null chars) result)
 (:let ((key . grammar) . chars) chars)
 (:let result (f-print-grammar-visits1 grammar (cons key construct) result))
 (f-print-grammar-visits2 chars construct result))

#|
=============================================
Search grammar
=============================================
|#

(deff f-grammar-get (card* grammar)
 (:let (chars parm . node) grammar)
 (:when (null card*) node)
 (:let (card . card*) card*)
 (:when (equalp card f-placeholder) (f-grammar-get card* parm))
 (f-grammar-get card* (assoc-get card chars)))

(deff f-construct2root (construct grammar)
 (:let card* (ct2card* construct))
 (:let card* (f-internalize card*))
 (:let (type id ref) (f-grammar-get card* grammar))
 (:when (unequal type 0) (error "Ambiguous: ~s" construct))
 (list id ref))

#|
=============================================
Construct grammar
=============================================
(make-prio-assoc prio* construct*) looks up all constructs in prio* in construct* and finds their page name/class id pairs. Then it computes the left and right priority of the construct and returns an association list that allows to translate page name/class id pairs to left and right priorities.

(grammar-add0 char* node grammar) adds the construct char* to the grammar and associates the node to it.
|#

(deff make-prio-assoc (prio* construct*)
 (make-prio-assoc1 prio* construct* 0 nil nil))

(deff make-prio-assoc1 (prio* construct* priority associativity1 result)
 (:let (prio . prio*) prio*)
 (:let construct (find prio construct* :test 'equal :key 'cdr))
 (:when (null construct)
  (complain "Unknown construct in priority table: ~s"
   (f-externalize prio)))
 (:let ((nil nil name associativity class-id) . char*) construct)
 (:when (and associativity1 (unequal associativity1 associativity))
  (complain "Contradicting associativities for ~s"
   (f-externalize prio)))
 (:let left (if (equalp associativity :preassociative) (+ priority 1) priority))
 (:let right
  (if (equalp associativity :postassociative) (+ priority 1) priority))
 (:let key (cons name class-id))
 (:let value (list left right))
 (:let value1 (assoc-get key result))
 (:when (and value1 (unequal value value1))
  (if (equalp (first value) (second value1))
   (complain "Ambiguous associativity: ~s" (f-externalize char*))
   (complain "Ambiguous priority: ~s" (f-externalize char*))))
 (:let result (acons key value result))
 (:when (null prio*) result)
 (:let (relation . prio*) prio*)
 (if (equalp relation (ct2ct #\>))
  (make-prio-assoc1 prio* construct* (+ priority 2) nil result)
  (make-prio-assoc1 prio* construct* priority associativity result)))

(deff grammar-add0 (char* node grammar)
 (:when (null char*) (grammar-add2 node grammar))
 (:let (chars parm . node1) grammar)
 (:let (char . char*) char*)
 (:when (equalp char f-placeholder)
  (:let parm (grammar-add0 char* node parm))
  (list* chars parm node1))
 (:let chars (grammar-add1 char char* node chars))
 (list* chars parm node1))

(deff grammar-add1 (char char* node chars)
 (:when (null chars) (acons char (grammar-add0 char* node nil) nil))
 (:when (unequal (caar chars) char)
  (cons (car chars) (grammar-add1 char char* node (cdr chars))))
 (acons char (grammar-add0 char* node (cdar chars)) (cdr chars)))

(deff grammar-add2 (node grammar)
 (:let (chars parm . node1) grammar)
 (:when (null node1) (list* chars parm node))
 (:when (unequal (car node1) 0) grammar)
 (:let (nil id1 ref1) node)
 (:let (nil id2 ref2) node1)
 (:let node (list 1 id1 ref1 id2 ref2))
 (list* chars parm node))

#|
=============================================
Parse trees
=============================================
A parse tree has form (root tree tree ...) where root has form (id ref . attribute*). Ref is a relative reference number (i.e. an index into the bibliography of the page). Id is the id of the given construct on the given page. Attribute* contains attached attributes.

As a special case, ref may be nil in which case id is a list of cardinals which represents a string.

The attributes attached to trees vary with time. The trees returned from f-parse-body have attributes (left right). Left and right are priorities.

The trees returned from f-reorganize have no attributes, so attribute* is NIL for those trees. f-reorganize uses the left and right priorities to get understood parentheses right and consumes the priorities in its course.

(f-print-tree tree) prints the given tree with one node per line and indentation.

|#

(deff f-print-tree (tree)
 (format t "~%BODY~%")
 (f-print-tree1 tree 0))

(deff f-print-tree1 (tree indent)
 (:let (root . tree*) tree)
 (:let (id ref . rest) root)
 (:let id (if (atom id) id (safe-subseq id 0 3)))
 (indent indent)
 (format t "ref ~3d id ~3d rest ~s~%" ref id rest)
 (f-print-tree* tree* (+ indent 1)))

(deff f-print-tree* (tree* indent)
 (:when (null tree*) nil)
 (:let (tree . tree*) tree*)
 (f-print-tree1 tree indent)
 (f-print-tree* tree* indent))

#|
=============================================
Pretty print trees
=============================================
|#

(deff f-grammar2hash (grammar)
 (f-grammar2hash1 grammar nil (make-hash-table :test 'equal)))

(deff f-grammar2hash1 (grammar construct result)
 (:when (null grammar) result)
 (:let (assoc parm . node) grammar)
 (:let result (f-grammar2hash2 assoc construct result))
 (:let result (f-grammar2hash1 parm (cons f-placeholder construct) result))
 (:let result (f-grammar2hash3 node construct result))
 result)

(deff f-grammar2hash2 (assoc construct result)
 (:when (null assoc) result)
 (:let ((char . grammar) . assoc) assoc)
 (:let result (f-grammar2hash1 grammar (cons char construct) result))
 (:let result (f-grammar2hash2 assoc construct result))
 result)

(deff f-grammar2hash3 (node construct result)
 (:when (null node) result)
 (:let (kind id ref) node)
 (:when (unequal kind 0) result)
 (:let key (list id ref))
 (:let newvalue (reverse construct))
 (:let oldvalue (gethash key result))
 (when (or (null oldvalue) (< (length newvalue) (length oldvalue)))
  (setf (gethash key result) newvalue))
 result)

(deff f-tree2ct (tree grammar depth length)
 (:let hash (f-grammar2hash grammar))
 (:let ct (f-tree2ct1 tree hash depth length))
 ct)

#|
(deff f-remove-initial-space (construct)
 (:let (char . construct1) construct)
 (if (equalp char f-space) construct1 construct))
|#

(deff f-tree2ct1 (tree hash depth length)
 (:when (= depth 0) "#")
 (:let (node . tree*) tree)
 (:let (id ref) node)
 (:when (null ref)
  (:when (equalp id :name) *f-name*)
  (:when (equalp id :prio) *f-prio*)
  (:let ellipsis (when (and length (> (length id) length)) "..."))
  (:let id (safe-subseq id 0 length))
  (:let id (substitute f-space f-newline id))
  (list f-quote id ellipsis f-quote))
 (:let construct (gethash (list id ref) hash))
 (:when (null construct) (format t "node=~s~%" node) "#")
;(:let construct (f-remove-initial-space construct))
 (f-tree*2ct construct tree* hash depth length))

(deff f-tree*2ct (construct tree* hash depth length)
 (:when (null construct) nil)
 (:let (char . construct) construct)
 (:when (unequal char f-placeholder)
  (cons char (f-tree*2ct construct tree* hash depth length)))
;(:let construct (f-remove-initial-space construct))
 (:let (tree . tree*) tree*)
 (:let tree (f-tree2ct1 tree hash (- depth 1) length))
 (:let tree* (f-tree*2ct construct tree* hash depth length))
 (list* "{" tree "}" tree*))

#|
=============================================
Reverse reversed trees
=============================================
f-parse-all defined later returns 'reversed trees' whose argument lists are in reverse order. Furthermore, the argument lists end with a pivot which indicates the location of the construct in the source text. (f-translate-tree revtree) translates such a reversed tree to an ordinary one. Hence, f-revtree recursively reverses the argument lists of trees and also extract char and line numbers from pivots and install them in the nodes of the tree.

Reversed trees have form
  (node arg ... arg pivot)
The node may denote a string in which case it has form
  (0 (card ... card) nil left right)
or it may be taken directly from the grammar and has form
  (0 id ref left right)
or
  (1 id1 ref1 id2 ref2)

(f-translate-tree revtree) complains if the given revtree contains a node of form (1 id1 ref1 id2 ref2). Otherwise, it recursively reverses the argument lists and removes the initial zero in each node. Furthermore, f-translate-tree builds an association list which translates tree addresses to pivots. The function then returns a list of form (tree assoc).

(f-revtree revtree) recursively reverses the arguments, complains about (1 page1 page2), removes the initial zero, and conses each pivot on the associated node.

(f-make-map tree) builds the assoc.

(f-remove-pivot tree) removes the pivots from the tree.

|#

(deff f-translate-tree (revtree)
 (f-revtree revtree))

(deff f-revtree (revtree)
 (:let (root . arg*) revtree)
 (f-revtree* root arg* nil))

(deff f-revtree* (root arg* result)
 (:let (arg . arg*) arg*)
 (:when (null arg*) (f-revtree1 root arg result))
 (:let arg (f-revtree arg))
 (:let result (cons arg result))
 (f-revtree* root arg* result))

(deff f-revtree1 (root pivot result)
 (:let (kind . root) root)
 (:when (unequal kind 0) (f-revtree2 root pivot))
 (cons root result))

(deff f-revtree2 (root pivot)
 (:let (id1 ref1 id2 ref2) root)
 (format t "Symbol ~d of page ~d has the same name as symbol ~d of page ~d~%"
  id1 ref1 id2 ref2)
 (setq *pivot* pivot)
 (f-error "Use of ambiguous construct"))

#|
=============================================
Parser
=============================================

(f-parse-body grammar) parses the given expression according to the given grammar. f-parse-body complaints in case no or more than one interpretation is found.

Parsing is essentially done by a depth first search for interpretations of the given expression. In a grammar with constructs a*, b, and *c, the string "a b c" may be interpretted as ((a b) c) or (a (b c)). Fully parenthesized, the interprettations read ((a (b)) c) and (a ((b) c)), respectively.

For the sake of explaining the algorithm, we introduce a 'brace-bracket' notation explained by the following example. To write e.g. (((a b) c) c) on brace-bracket form, perform the following steps:

1) Replace left parentheses with left brackets and right parentheses with right brackets in the fully parenthesized form: [[[a [b]] c] c].

2) Replace each left bracket that follows another left bracket by a brace: [{{a [b]] c] c].

3) For each left brace replace the matching right bracket by a right brace: [{{a [b]} c} c].

4) Delete all left braces: [a [b]} c} c].

As other examples, ((a b) c) and (a (b c)) have brace-bracket forms [a [b]} c] and [a [b} c]], respectively.

We shall say that a brace-bracket form is right-associative if right brackets are never followed by right braces. As an example, the brace-bracket form [a [b} c]] of (a (b c)) is right associative whereas the brace-bracket form [a [b]} c] of ((a b) c) is not.

The parsing functions essentially make a brute force search for right-associative brace-bracket forms that conform to the given expression and grammar. In other words, the parsing functions parse the expression under the assumption that all constructs have the same priority and all constructs are right associative. The parsing functions do not construct the brace-bracket forms explicitly. Rather, they try to construct the parse trees right away. Furthermore, the parse functions have some extra arguments that allow to produce intelligible error message when needed.

f-parse-body is implemented by a number of parsing functions that call each other recursively. The prototypical one is f-parse-all.

(f-parse-all subgrammar arg* stack grammar position result) parses, starting from *pivot*, according to the given subgrammar. The arguments are:

subgrammar: A subtree of the full grammar. The given pivot is parsed according to the subgrammar.

arg*: The subexpressions of the current expression that have already been parsed. The last element of arg*, however, is the pivot where the construct started (which is right after the first subexpression for left-open constructs).

stack: A stack of subgrammar/arg* pairs of superexpressions whose parsing has been suspended until further.

grammar: The full grammar. Subexpressions are parsed according to the full grammar.

position: The number of characters parsed so far. The maximum value of this parameter is used when reporting that the given expression has no interpretations.

result: A structure that indicates the result of the parsing so far. Eventually, the value of this parameter is returned as the result of the entire parsing. The result parameter can have the following formats:

NIL indicates that no interpretations have been found so far. The maxposition indicates how many characters were parsed at the most successfull atempt so far and the pivot is the associated pivot.

(tree) indicates that exactly one interpretation has been found so far. The tree is that interpretation expressed as a reversed tree.

(tree1 tree2) indicates that exactly two interpretations have been found so far.

|#

; Global variable that stores grammar during parsing
(defc *grammar* nil)

; Global variable that stores trees found during parsing
(defc *trees* nil)

; Global variable that holds a substitute for the (small) clisp stack
(defc *heap-stack* nil)

; Return T if the given result represents a situation where more than one interpretation has been found.

(deff f-ambiguous ()
 (cdr *trees*))

; Debugging aids

(defmacro debug-parser (x) (declare (ignorable x)) nil)
;(defmacro debug-parser (x) (declare (ignorable x)) x)

(defc *min-pivot* 0)

(defmacro debug-print (format &rest args)
 (declare (ignorable format args))
 (debug-parser `(debug-print1 ,format ,@args)))

(deff debug-print1 (format &rest args)
 (:let depth (- *pivot* *min-pivot*))
 (:when (> depth 80000) (error "Near stack overflow"))
 (:when (unequal 0 (mod depth 100)) nil)
 (indent depth)
 (apply 'format t format args)
 (terpri))

; Error routine for ambiguous parse trees

(deff f-ambiguous-error (tree1 tree2)
 (:when (equalp (car tree1) (car tree2))
  (f-ambiguous-error* (cdr tree1) (cdr tree2)))
 (setq *pivot* (car (last tree1)))
 (f-error "Ambiguous parse tree"))

(deff f-ambiguous-error* (list1 list2)
 (:when (null (cdr list1)) nil)
 (f-ambiguous-error (car list1) (car list2))
 (f-ambiguous-error* (cdr list1) (cdr list2)))

; Action lists
(defc ac-all '(:rec :char :parm :end))
(defc ac-most '(:char :parm :end))
(defc ac-end '(:rec :end))

; Second half of f-parse-chars below

(defmacro f-parse-chars1 (#|code subgrammar arg* stack|#)
'(exec
  (debug-print ".~a." (code-char code))
  (:let (assoc :parm . :node) subgrammar)
  (:let subgrammar (assoc-get code assoc))
  (:when (null subgrammar) nil)
  (setq *max-pivot* (max *max-pivot* *pivot*))
  (f-visit)
  (f-parse-all subgrammar arg* stack ac-all)))

; like f-parse-all, but only parses characters. A call to f-parse-chars corresponds to an atempt to add one letter or space to the right-associative brace-bracket form of the expression.

(defmacro f-parse-chars (#|subgrammar arg* stack|#)
'(exec
  (:when (f-ambiguous) nil)
  (:let code (f-parse-space-or-char))
  (:when (no-item code) nil)
  (f-parse-chars1 #|code subgrammar arg* stack|#)))

; like f-parse-all, but only parses an end-of-expression which is simultaneously the first token of a left-open expression (rec in f-parse-rec stands for 'left-recursive' in left-to-right reading countries, 'right-recursive' in right-to-left reading countries, top-recursive in top-to-botton reading countries etc). A call to f-parse-rec corresponds to an atempt to add one right brace to the right-associative brace-bracket form of the expression.

(defmacro f-parse-rec (#|subgrammar arg* stack|#)
'(exec
  (:let (:assoc :parm . node) subgrammar)
  (:when (null node) nil)
  (:when (f-ambiguous) nil)
  (debug-print " }")
  (:let arg (cons node arg*))
  (:let arg* (list arg *pivot*))
  (:let subgrammar *grammar*)
; (f-visit) We don't care about how often the root is visited
  (:let (nil subgrammar . nil) subgrammar)
; (f-visit) We don't care how often left recursion is attempted
  (f-parse-all subgrammar arg* stack ac-all)))

; like f-parse-end, but takes care of the situation where a superexpression exists (i.e. the stack is non-empty).  A call to f-parse-exp corresponds to an atempt to add one right bracket to the right-associative brace-bracket form of the expression in a situation where there will still be open brackets after the addition.

(defmacro f-parse-exp (#|subgrammar arg* stack|#)
'(exec
  (:let (:assoc :parm . node) subgrammar)
  (:when (null node) nil)
  (debug-print " ]")
  (:let arg (cons node arg*))
  (:let ((subgrammar . arg*) . stack) stack)
; (f-visit) This f-visit would essentially count number of left recursions
  (:let arg* (cons arg arg*))
  (f-parse-all subgrammar arg* stack ac-most)))

; like f-parse-end, but takes care of the situation where no superexpression exists (i.e. the stack is empty). A call to f-parse-eof corresponds to an atempt to add one right bracket to the right-associative brace-bracket form of the expression in a situation where that right bracket will close the last open left bracket.

(defmacro f-parse-eof (#|subgrammar arg*|#)
'(exec
  (:let (:assoc :parm . node) subgrammar)
  (:when (null node) nil)
  (debug-print " ]!")
  (f-skip-space)
  (:let code (f-parse-char))
  (:when (is-item code) nil)
  (:let tree (cons node arg*))
  (push tree *trees*)))

; like f-parse-all, but only parses an end-of-expression which either fits into a superexpression (when the stack is non-empty) or indicates success in finding an interpretation (when the stack is empty and the end of the file is reached). A call to f-parse-end corresponds to an atempt to add one right bracket to the right-associative brace-bracket form of the expression.

(defmacro f-parse-end (#|subgrammar arg* stack|#)
'(exec
  (:when (f-ambiguous) nil)
  (if stack
   (f-parse-exp #|subgrammar arg* stack|#)
   (f-parse-eof #|subgrammar arg*|#))))

; Functions for parsing strings inside the parse tree

(defmacro f-parse-body-string (#|arg* stack|#);()
'(exec
  (:let string (f-parse-string-body f-quote))
  (:let kind 0)
  (:let id string)
  (:let ref nil)
  (:let left nil)
  (:let right nil)
  (:let node (list kind id ref left right))
  (:let subgrammar (list* :assoc :parm node))
  (f-parse-all subgrammar arg* stack ac-end)))

; Second half of f-parse-parm. A call to f-parse-parm1 corresponds to add one left bracket unconditionally to the right-associative brace-bracket form of the expression. Since parsing starts by opening a left bracket, parsing is fired up by a call to f-parse-parm1. The 'space' parameter may be a space character or a soft complaint. A space character indicates that a (possibly understood) space character has been parsed. A soft complaint indicates that no space character has been parsed at the given time instant and, hence, no ordinary construct can start here.

(defmacro f-parse-parm1 (#|space stack|#)
'(exec
  (:let pivot *pivot*)
  (:let code (f-parse-char))
  (:when (no-item code) nil)
  (:when (equalp code f-quote)
   (:let arg* (list pivot))
   (f-parse-body-string #|arg* stack|#))
  (:let subgrammar *grammar*)
; (f-visit) We don't care about how often the root is visited
  (:let arg* (list pivot))
  (f-parse-chars1 #|code subgrammar arg* stack|#)))

; like f-parse-all, but only parses subexpressions. A call to f-parse-parm corresponds to an attempt to add one left bracket to the right-associative brace-bracket form of the expression.

(defmacro f-parse-parm (#|subgrammar arg* stack|#)
'(exec
  (:when (f-ambiguous) nil)
  (:let (:assoc subgrammar . :node) subgrammar)
  (:when (null subgrammar) nil)
  (debug-print " [")
  (:let stack (acons subgrammar arg* stack))
 ;(:let space (f-parse-space))
  (f-parse-parm1 #|space stack|#)))

; modified f-parse-parm which fires up parsing
(defmacro f-parse-top (#|subgrammar arg* stack|#)
'(exec
 ;(:let space f-space)
  (f-parse-parm1 #|space stack|#)))

; Perform one of the four possible kinds of actions

(defmacro f-parse-some (#|subgrammar arg* stack action|#)
'(exec
  (:when (f-ambiguous) nil)
  (:when (equalp action :rec)  (f-parse-rec #|subgrammar arg* stack|#))
  (:when (equalp action :char) (f-parse-chars #|subgrammar arg* stack|#))
  (:when (equalp action :parm) (f-parse-parm #|subgrammar arg* stack|#))
  (:when (equalp action :end)  (f-parse-end #|subgrammar arg* stack|#))
  (:when (equalp action :top)  (f-parse-top #|subgrammar arg* stack|#))
  (error "f-parse-some: internal check failed")))

; The f-parse-all described above. A call to f-parse-all corresponds to an atempt to add one character to the right-associative brace-bracket form of the expression.

(deff f-parse-all (subgrammar arg* stack action*)
 (:let list (list *pivot* subgrammar arg* stack))
 (dolist (action action*)
  (push (cons action list) *heap-stack*)))

(defc *sample* 0)
(defc *sample-delta* 1000000)

(defmacro f-visit ()
 '(when *measure-grammar* (incf (gethash subgrammar *sample-hash* 0))))

(deff f-parse-body1 ()
 (:when (f-ambiguous) nil)
 (:when (null *heap-stack*) nil)
 (:let (action pivot subgrammar arg* stack) (pop *heap-stack*))
 (setq *pivot* pivot)
 (incf *sample*)
 (when (= (mod *sample* *sample-delta*) 0)
  (format t "Currently at file position ~s in lgs file~%" pivot))
 (f-parse-some #|subgrammar arg* stack action|#)
 (f-parse-body1))

; Parse the given pivot according to the given grammar. Complaint if zero or more than one interpretation is found.

(deff f-parse-body (grammar)
 (f-skip-space)
 (:let subgrammar nil)
 (:let arg* nil)
 (:let stack nil)
 (setq *max-pivot* *pivot*)
 (debug-parser (setq *min-pivot* *pivot*))
 (setq *grammar* grammar)
 (setq *trees* nil)
 (setq *heap-stack* nil)
 (setq *sample* 0)
 (setq *measure-grammar* (option "grammar"))
 (when (equalp *measure-grammar* 0) (setq *measure-grammar* nil))
 (when *measure-grammar* (clrhash *sample-hash*))
 (f-parse-all subgrammar arg* stack '(:top))
 (f-parse-body1)
 (when *measure-grammar* (f-print-grammar-visits *grammar* *trees*))
 (:let (tree1 tree2) *trees*)
 (setq *grammar* nil)
 (setq *trees* nil)
 (:when tree2 (f-ambiguous-error tree1 tree2))
 (:when tree1 (f-translate-tree tree1))
 (setq *pivot* *max-pivot*)
 (f-error "No interpretations"))

#|
=============================================
Split flattened tree at principal operator
=============================================
A pruned tree is at tree that has its leftmost (rightmost) branch pruned if its root is left-open (right-open).

A flattened tree is a list of pruned trees. As an example, the flattening of "if x then y else z + u" is ((if x then y else) (z) (+) (u)).

The principal tree of a flattened tree is the pruned tree in the flattened tree that has highest priority index (i.e. lowest priority). A priority index of NIL is lower than all other incices.

If a flattened tree has more than one pruned tree with maximal priority, then the principal tree is the leftmost (rightmost) of these if the maximal priority is associated with right (left) associativity.

(f-max-prio tree*) takes a flattened tree as input and returns a list (prio assoc) where prio is the largest priority of the roots of tree* and assoc is true iff constructs with that priority are right associative.

(f-principal tree*) returns the position of the principal tree in the given flattened tree.

(f-split-tree* tree*) divides the flattened tree tree* into (left tree right) where 'tree' is the principal tree and left and right are the trees to the left and right of the principal tree.

|#

(deff f-max-prio (tree*)
 (f-max-prio1 tree* -1 nil))

(deff f-max-prio1 (tree* prio assoc)
 (:when (null tree*) (cons prio assoc))
 (:let (tree . tree*) tree*)
 (:let (node) tree)
 (:let (nil nil left right) node)
 (:let left (default -1 left))
 (:let right (default -1 right))
 (:let max (max left right))
 (:when (<= max prio) (f-max-prio1 tree* prio assoc))
 (f-max-prio1 tree* max (< left right)))

; return the left priority of the root of the given tree
(deff left-prio (tree)
 (third (car tree)))

; return the right priority of the root of the given tree
(deff right-prio (tree)
 (fourth (car tree)))

; return the position of the principal operator
(deff f-principal (tree*)
 (:let (prio assoc) (f-max-prio tree*))
 (if assoc
  (position prio tree* :key 'right-prio)
  (position prio tree* :key 'left-prio :from-end t)))

; (f-split-tree* tree*) divides the given tree* into (left tree right) where 'tree' is the principal tree and left and right are the trees to the left and right of the principal tree.
(deff f-split-tree* (tree*)
 (:let position (f-principal tree*))
 (:let left (subseq tree* 0 position))
 (:let right (subseq tree* position))
 (:let (tree . right) right)
 (list left tree right))

#|
=============================================
Reorganize tree
=============================================

(f-reorganize tree) returns the given tree with priority and associativity rules enforced.

For each tree, define the 'zone of reorganization' Z as the smallest subset of nodes of the tree that satisfies the following:

The root of the tree belongs to Z.
If a node N in Z is pre-open, then the root of the first subtree of N is in Z.
If a node N in Z is post-open, then the root of the last subtree of N is in Z.

Define the 'reorganization subforrest' of a tree as the tree with its zone of reorganization removed.

(f-reorganize tree) first 'flattens' the tree to destroy the right-associative nature of the output from the parser. The flattened form represents the tree split appart in nodes from the reorganization zone plus the reorganization subforrest. Then f-reorganize reorganizes the reorganization subforrest through recursion, and finally puts the zone of reorganization together into a structure governed by the priorities of the nodes.

As mentioned before, the flattened form is a list of 'pruned trees', one for each node in the reorganization zone. The pruned tree of a node is the subtree rooted at that node, except that branches that contain part of the reorganization zone are removed. Hence, a tree is pruned by removing the first subtree if the root is pre-open and removing the last subtree if the root is post-open.

(f-split-tree tree) prunes the given tree and returns a list (node left middle right) where 'node' is the node of the tree, 'left' is left branch (or nil if the tree isn't pruned to the left), 'right' is the right branch (or nil if the tree isn't pruned to the right), and 'middle' is the list of remaining branches.

(f-flatten tree result) flattens the given tree and appends it in front of the given 'result'. Flatten calls reorganize in order to reorganize all branches. As an example, when flattening "if x then y else z + u" into ((if x then y else) (z) (+) (u)), flatten flattens x, y, z, and u.

|#

(deff f-reorganize (tree)
 (f-organize (f-flatten tree nil)))

(deff f-reorganize* (tree*)
 (map 'list 'f-reorganize tree*))

(deff f-split-right (right tree*) 
 (:when (null right) (cons tree* nil))
 (cons (butlast tree*) (car (last tree*))))

(deff f-split-left (left tree*)
 (:when (null left) (cons nil tree*))
 tree*)

(deff f-split-tree (tree)
 (:let (node . tree*) tree)
 (:let (nil nil left right) node)
 (:let (middle . right) (f-split-right right tree*))
 (:let (left . middle) (f-split-left left middle))
 (list node left middle right))

(deff f-flatten (tree result)
 (:when (null tree) result)
 (:let (node left middle right) (f-split-tree tree))
 (:let middle (f-reorganize* middle))
 (:let tree (cons node middle))
 (:let result (f-flatten right result))
 (:let result (cons tree result))
 (:let result (f-flatten left result))
 result)

(deff f-collect-tree (node left middle right)
 (cons node (f-collect-tree* left middle right)))

(deff f-collect-tree* (left middle right)
 (:let tree* (if (null right) middle (append middle (list right))))
 (:let tree* (if (null left) tree* (cons left tree*)))
 tree*)

; remove priorities from the given node
(deff f-clean-node (node)
 (:let (id ref) node)
 (list id ref))

(deff f-clean-tree (tree)
 (:let (node . tree*) tree)
 (:let node (f-clean-node node))
 (cons node tree*))

(deff f-organize (tree*)
 (:when (null (cdr tree*)) (f-clean-tree (car tree*)))
 (:let (prefix tree suffix) (f-split-tree* tree*))
 (:let (node . tree*) tree)
 (:let (nil nil left right) node)
 (:let node (f-clean-node node))
 (:when (and left right)
  (f-collect-tree node (f-organize prefix) tree* (f-organize suffix)))
 (:when left
  (:let tree (f-collect-tree node (f-organize prefix) tree* nil))
  (f-organize (f-collect-tree* tree suffix nil)))
 (:let tree (f-collect-tree node nil tree* (f-organize suffix)))
 (f-organize (f-collect-tree* nil prefix tree)))

#|
=============================================
Vectorize
=============================================
(f-vectorize bib arity tree) vectorizes the given information into a Logiweb page. The function returns a structure of form (ref vector) where both the ref and the vector are lists of bytes (i.e. cardinals in the range 0..255).

The constructed vector consists of a self reference, a bibliography, an arity array, and a parse tree in that order. The arity array and parse tree are built in reverse order, however, and then reversed.

f-vectorize-tree, f-vectorize-tree*, f-vectorize-root, and f-vectorize-string vectorizes the parse tree.

f-vectorize-arity vectorizes the arity array.

f-encode-card* and f-encode-card reverse a list of cardinals and simultaneously convert the cardinals to bytes.
|#

#|
(deff f-pre (id shift)
 (:let ref (if (= shift 1) 0 1))
 (f-vectorize-root id ref shift))

(deff f-vectorize-smallid (id tree* shift chars assoc result)
 (:when (unequal (length tree*) (aget chars id))
  (format t "Arity mismatch of construct #~d~%" id)
  (raise))
 (f-vectorize-tree* tree* shift chars assoc (cons id result)))
|#

(deff f-vectorize-root (id ref shift)
 (+ 1 ref (* id shift)))

(defc f-page-symbol 1) ; f-vectorize-root of ref=0, id=0

(defc f-empty           nil)
(defc f-preassociative  nil)
(defc f-postassociative nil)
(defc f-assoc-cons      nil)
(defc f-assoc-end       nil)
(defc f-variable        nil)
(defc f-name            nil)
(defc f-def             nil)
(defc f-name-cons       nil)

; vectorize construct
(deff f-vc (grammar shift construct)
 (:let card* (ct2card* construct))
 (:let card* (f-internalize card*))
 (:let (type id ref) (f-grammar-get card* grammar))
 (:when (null ref) nil)
 (:when (unequal type 0) (format t "Warning, ambiguous: ~s" construct))
 (f-vectorize-root id ref shift))

(deff f-vectorize-std-sym (grammar bib)
 (:let g grammar)
 (:let s (+ 1 (length bib)))
 (setq f-empty           (f-vc g s "base empty"                              ))
 (setq f-preassociative  (f-vc g s "base preassociative \" greater than \""  ))
 (setq f-postassociative (f-vc g s "base postassociative \" greater than \"" ))
 (setq f-assoc-cons      (f-vc g s "base priority \" equal \""               ))
 (setq f-assoc-end       (f-vc g s "base priority \" end priority"           ))
 (setq f-variable        (f-vc g s "base asterisk"                           ))
 (setq f-name            (f-vc g s "base name"                               ))
 (setq f-def             (f-vc g s "base Define \" of \" as \" end define"   ))
 (setq f-name-cons       (f-vc g s "\" base linebreak \""                    )))

(deff f-check-sym (sym name)
 (:when sym nil)
 (format t "Missing construct for Name/Prio generation: ~s~%" name)
 (raise))

(deff f-vectorize-priority (shift assoc result)
 (f-check-sym f-empty           "base empty"                              )
 (f-check-sym f-preassociative  "base preassociative \" greater than \""  )
 (f-check-sym f-postassociative "base postassociative \" greater than \"" )
 (f-check-sym f-assoc-cons      "base priority \" equal \""               )
 (f-check-sym f-assoc-end       "base priority \" end priority"           )
 (f-check-sym f-variable        "base asterisk"                           )
 (:let table (list f-empty))
 (:let table (f-vectorize-priority1 shift nil assoc table))
 (revappend table result))

(deff f-vectorize-priority1 (shift prio assoc result)
 (:when (null assoc)
  (if (equalp (car result) f-empty)
   (list*
    f-preassociative
    f-assoc-end
    f-page-symbol
    result)
   (f-add-associativity prio nil result)))
 (:let ((construct :page prio1 id ref) . assoc) assoc)
 (:let result (f-add-associativity prio prio1 result))
 (:when (null prio1) (f-vectorize-priority1 shift prio1 assoc result))
 (:let arity (f-arity construct))
 (:let result (repeat1 arity f-variable result))
 (:let code (f-vectorize-root id ref shift))
 (:let result (cons code result))
 (:let operator (if (equalp prio prio1) f-assoc-cons f-assoc-end))
 (:let result (cons operator result))
 (f-vectorize-priority1 shift prio1 assoc result))

(deff f-add-associativity (prio prio1 result)
 (:when (equalp prio prio1) result)
 (:when (null prio) result)
 (:let result
  (if (> prio 1) result
   (list* f-assoc-cons f-page-symbol result)))
 (:let associativity (if (oddp prio) f-preassociative f-postassociative))
 (cons associativity result))

(deff f-vectorize-name (shift assoc result)
 (f-check-sym f-empty           "base empty"                              )
 (f-check-sym f-variable        "base asterisk"                           )
 (f-check-sym f-name            "base name"                               )
 (f-check-sym f-def             "base Define \" of \" as \" end define"   )
 (f-check-sym f-name-cons       "\" base linebreak \""                    )
 (:let table (f-vectorize-name1 shift assoc (list f-empty)))
 (revappend table result))

(deff f-vectorize-name1 (shift assoc result)
 (:when (null assoc) result)
 (:let ((construct :page :prio id ref) . assoc) assoc)
 (:when (unequal ref 0) (f-vectorize-name1 shift assoc result))
 (:let construct (ct2ct (f-externalize construct)))
 (:let result (list* 0 (length construct) construct result))
 (:let arity (f-arity construct))
 (:let result (repeat1 arity f-variable result))
 (:let result (cons (f-vectorize-root id ref shift) result))
 (:let result (cons f-name result))
 (:let result (cons f-def result))
 (:let result (cons f-name-cons result))
 (f-vectorize-name1 shift assoc result))

(deff f-vectorize-tree (tree shift assoc result)
 (:let (root . tree*) tree)
 (:let (id ref) root)
 (:when (equalp id :prio) (f-vectorize-priority shift assoc result))
 (:when (equalp id :name) (f-vectorize-name shift assoc result))
 (:when (listp id) (f-vectorize-string id result))
;(:when (null ref) (f-vectorize-smallid id tree* shift chars assoc result))
 (:when (null ref) (error "Internal error: use of obsolete feature"))
 (:let code (f-vectorize-root id ref shift))
 (:let result (cons code result))
 (f-vectorize-tree* tree* shift assoc result))

(deff f-vectorize-tree* (tree* shift assoc result)
 (:when (null tree*) result)
 (:let (tree . tree*) tree*)
 (:let result (f-vectorize-tree tree shift assoc result))
 (:let result (f-vectorize-tree* tree* shift assoc result))
 result)

(deff f-vectorize-string (card* result)
 (list* card* (length card*) 0 result))

(deff f-vectorize-arity (arity result)
 (:when (null arity) result)
 (:let ((key . value) . arity) arity)
 (:let result (list* value key result))
 (f-vectorize-arity arity result))

(deff f-encode-card* (card* result)
 (:when (null card*) result)
 (:let (card . card*) card*)
 (:when (listp card) (f-encode-card* card* (append card result)))
 (f-encode-card* card* (f-encode-card card result)))

(deff f-encode-card (card result)
 (:when (< card 128) (cons card result))
 (:mlet (card byte) (floor card 128))
 (:let byte (+ 128 byte))
 (:let result (f-encode-card card result))
 (cons byte result))

(deff f-vectorize-bib (bib vector)
 (:when (null bib) vector)
 (:let (ref . bib) bib)
 (:let ref (f-encode-card (length ref) ref))
 (append ref (f-vectorize-bib bib vector)))

(deff f-vectorize-ripemd (ripemd timestamp vector)
 (:let ref (cons 1 (append ripemd timestamp)))
 (:let length (length ref))
 (:let vector (f-encode-card length (cons 1 (append ripemd vector))))
 (list ref vector))

(deff f-try-given-ripemd (vector)
 (:when (equalp *sourceref* :none) nil)
 (:let timestamp (safe-subseq *sourceref* 21))
 (:let vector1 (append timestamp vector))
 (:let ripemd (ripemd vector1))
 (:let ripemd1 (safe-subseq *sourceref* 1 21))
 (when (verbose '> 0)
  (if (equalp ripemd ripemd1)
   (format t "Frontend: using sourceref as ref~%")
   (format t "Frontend: cannot use sourceref as ref~%")))
 (:when (unequal ripemd ripemd1) nil)
 (f-vectorize-ripemd ripemd timestamp vector1))

(deff f-add-header (ref)
 (when (verbose '> 0) (format t "Frontend: reading ~s again~%" *sourcename*))
 (:let ref (b-byte*2hex* ref))
 (:let vector (path2vector *sourcename*))
 (:when (unequal (safe-subseq vector 0 4) (ct2vector "\"\";;"))
  (progress "Frontend: cannot writing header back to source file~%")
  nil)
 (progress "Frontend: writing header back to source file")
 (:let vector (f-remove-header vector))
 (:let ct (list f-magic-code ref vector))
 (ct2file *sourcename* ct))

(deff f-vectorize (bib arity tree assoc)
 (:let shift (+ 1 (length bib)))
 (:let vector nil)
 (:let vector (f-vectorize-arity (array2assoc (aget arity 0)) vector))
 (:let vector (cons 0 vector)) ; end of arity
 (:let vector (f-vectorize-tree tree shift assoc vector))
 (:let vector (f-encode-card* vector nil))
 (:let vector (cons 0 vector)) ; end of bibliography
 (:let vector (f-vectorize-bib bib vector))
 (:let result (f-try-given-ripemd vector))
 (:when result result)
 (when (verbose '> 0) (progress "Frontend: generate new reference"))
 (:let (mantissa exponent) (lgt))
 (:let timestamp (f-encode-card* (list exponent mantissa) nil))
 (:let vector (append timestamp vector))
 (:let ripemd (ripemd vector))
 (:let result (f-vectorize-ripemd ripemd timestamp vector))
 (:when (equalp *sourceref* :none) result)
 (:let (ref) result)
 (f-add-header ref)
 result)

#|
=============================================
Main: Dump canned version of command
=============================================
|#

(deff purge-conventional-wisdom ()
 (clrhash c-url2ref-hash)
 (clrhash c-ref2url-hash))

(deff clean-canned ()
 (purge-conventional-wisdom)
 (setq *source* (make-array 0)) ; purge current page
 (setq *args* nil) ; Otherwise *args* shows up when invoking without parms
 (gc))

(deff main-canned ()
 (:let filename (option "dump"))
 (:when (equalp filename "") nil)
 (format t "Dumping ~s~%" filename)
 (:let conf (option "dumpconf"))
 (when (unequal conf "") (setq *conf* conf))
 (clean-canned)
 (dump filename 'main)
 (format t "New command ~s dumped, have fun~%" filename)
 :exit)

#|
=============================================
Parse page
=============================================
|#

(deff f-pretty-print-tree (tree grammar)
 (:let depth (option "spydepth"))
 (:let depth (if (= depth -2) 10 depth))
 (:let length (option "spylength"))
 (:let length (if (= length -2) 20 (if (< length 0) nil length)))
 (:let ct (f-tree2ct tree grammar depth length))
 (:let string (ct2string ct))
 (format t "~a~%" string))

(deff f-parse-page (state)
 (when (verbose '> 0) (progress "Frontend: parsing page declaration"))
 (:let name (f-parse-page-decl))
 (when (verbose '> 0) (progress "Frontend: parsing bibliography"))
 (:let (name* bib) (f-parse-bib))
 (:let ref* (map 'list 'ref2card bib))
 (:let name* (cons name name*))
 (:let state (c-load-bib ref* state))
 (:when (main-canned) (raise))
 (when (verbose '>= 0) (progress "Frontend: parsing associativity sections"))
 (:let (grammar arity assoc) (f-parse-assoc name* ref* state))
 (progress "Frontend: parsing body")
 (:let keyword (f-parse-keyword))
 (:when (unequal keyword (ct2ct "BODY")) (f-error "Missing 'BODY' section"))
 (:let tree (f-parse-body grammar))
 (progress "Frontend: invoking priority rules")
 (:let tree (f-reorganize tree))
 (when (level '<= "parse") (f-pretty-print-tree tree grammar))
 (f-vectorize-std-sym grammar bib)
 (list bib arity tree state assoc))

; Construct tree with given id on page 0 in the bibliography (the page itself)
(deff tr (id &rest tree*)
 (cons (list id 0) tree*))

; Construct tree with given id on bed page (page 1 if it exists, else page 0)
(deff trn (id &rest tree*)
 (cons (list id nil) tree*))

; Construct tree that represents string
(deff st (string)
 (trn (ct2ct string)))

(defc *page* nil)
(init-pool nil)

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE \"\" b
BODY b")
(rtst (setq *page* (f-parse-page :test-state)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(ntst (aget (second *page*) 0 2))
(etst (third *page*) (tr 1))
(etst (fourth *page*) :test-state)

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE \"\" b
BODY a")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(ntst (aget (second *page*) 0 2))
(etst (third *page*) (tr 0))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE 7 \"\" b
BODY b")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 7) 0)
(ntst (aget (second *page*) 0 1))
(etst (third *page*) (tr 7))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE 7 \"\" b \"
BODY b a")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 7) 1)
(ntst (aget (second *page*) 0 1))
(etst (third *page*) (tr 7 (tr 0)))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE 7 \"\" b \"
BODY b b a")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 7) 1)
(ntst (aget (second *page*) 0 1))
(etst (third *page*) (tr 7 (tr 7 (tr 0))))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE 7 \"\" b \"
BODY \"abc\"")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 7) 1)
(ntst (aget (second *page*) 0 1))
(etst (third *page*) (st "abc"))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE 7 \"\" b \"
BODY b \"abc\"")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 7) 1)
(ntst (aget (second *page*) 0 1))
(etst (third *page*) (tr 7 (st "abc")))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE 7 \"\" b \"
BODY b \"abc\"")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 7) 1)
(ntst (aget (second *page*) 0 1))
(etst (third *page*) (tr 7 (st "abc")))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE 1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
BODY a plus b")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(ntst (aget (second *page*) 0 5))
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(ntst (aget (second *page*) 0 8))
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 7 (tr 0) (tr 1)))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE
1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
BODY a plus b plus c")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(ntst (aget (second *page*) 0 5))
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(ntst (aget (second *page*) 0 8))
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 7 (tr 7 (tr 0) (tr 1)) (tr 2)))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
POSTASSOCIATIVE
1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
BODY a plus b plus c")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(ntst (aget (second *page*) 0 5))
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(ntst (aget (second *page*) 0 8))
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 7 (tr 0) (tr 7 (tr 1) (tr 2))))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE 8 \"\" \" times \"
PREASSOCIATIVE
1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
BODY a times b plus c times d")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(ntst (aget (second *page*) 0 5))
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(etst (aget (second *page*) 0 8) 2)
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 7 (tr 8 (tr 0) (tr 1)) (tr 8 (tr 2) (tr 3))))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
POSTASSOCIATIVE 8 \"\" \" times \"
PREASSOCIATIVE
1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
BODY a times b plus c times d")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(ntst (aget (second *page*) 0 5))
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(etst (aget (second *page*) 0 8) 2)
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 7 (tr 8 (tr 0) (tr 1)) (tr 8 (tr 2) (tr 3))))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE 8 \"\" \" times \"
POSTASSOCIATIVE 1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"

BODY a times b plus c times d")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(ntst (aget (second *page*) 0 5))
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(etst (aget (second *page*) 0 8) 2)
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 7 (tr 8 (tr 0) (tr 1)) (tr 8 (tr 2) (tr 3))))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
POSTASSOCIATIVE
8 \"\" \" times \"
POSTASSOCIATIVE
1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
BODY a times b plus c times d")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(ntst (aget (second *page*) 0 5))
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(etst (aget (second *page*) 0 8) 2)
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 7 (tr 8 (tr 0) (tr 1)) (tr 8 (tr 2) (tr 3))))
(ntst (fourth *page*))

; Test where 'times' ends in the root even though it has lowest priority
(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE
8 \"\" \" times \"
PREASSOCIATIVE
1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
PREASSOCIATIVE
9 \"\" lambda \" dot \"
BODY a times lambda b dot c plus d")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(ntst (aget (second *page*) 0 5))
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(etst (aget (second *page*) 0 8) 2)
(etst (aget (second *page*) 0 9) 2)
(etst (third *page*) (tr 8 (tr 0) (tr 9 (tr 1) (tr 7 (tr 2) (tr 3)))))
(ntst (fourth *page*))

; Translation of comma to code 4 on bed page
(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE
1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
POSTASSOCIATIVE 4 \"\" \" , \"
BODY a , b")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(etst (aget (second *page*) 0 4) 2)
(ntst (aget (second *page*) 0 5))
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(ntst (aget (second *page*) 0 8))
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 4 (tr 0) (tr 1)))
(ntst (fourth *page*))

; Translation of brackets to code 5 on bed page
(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE
1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
POSTASSOCIATIVE 5 \"\" \" [ \" ] \"
BODY a [ b ] c")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(etst (aget (second *page*) 0 5) 3)
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(ntst (aget (second *page*) 0 8))
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 5 (tr 0) (tr 1) (tr 2)))
(ntst (fourth *page*))

; Normal use of brackets
(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE
1 \"\" b
2 \"\" c
3 \"\" d
7 \"\" \" plus \"
POSTASSOCIATIVE 5 \"\" \"[ \" ]\"
BODY \"uvw\"[ b ]\"xyz\"")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(etst (aget (second *page*) 0 1) 0)
(etst (aget (second *page*) 0 2) 0)
(etst (aget (second *page*) 0 3) 0)
(ntst (aget (second *page*) 0 4))
(etst (aget (second *page*) 0 5) 3)
(ntst (aget (second *page*) 0 6))
(etst (aget (second *page*) 0 7) 2)
(ntst (aget (second *page*) 0 8))
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 5 (st "uvw") (tr 1) (st "xyz")))
(ntst (fourth *page*))

(set-source "PAGE a
BIBLIOGRAPHY
PREASSOCIATIVE
6 \"\" text \" end text
BODY text \"xyz\" end text")
(rtst (setq *page* (f-parse-page nil)))
(ntst (first *page*))
(ntst (aget (second *page*) 0 0))
(ntst (aget (second *page*) 0 1))
(ntst (aget (second *page*) 0 2))
(ntst (aget (second *page*) 0 3))
(ntst (aget (second *page*) 0 4))
(ntst (aget (second *page*) 0 5))
(etst (aget (second *page*) 0 6) 1)
(ntst (aget (second *page*) 0 7))
(ntst (aget (second *page*) 0 8))
(ntst (aget (second *page*) 0 9))
(etst (third *page*) (tr 6 (st "xyz")))
(ntst (fourth *page*))

#|
=============================================
Check name and priority consistency
=============================================
|#

(deff f-check-name (ref assoc state)
 (:when (null assoc) nil)
 (:let ((construct :name :prio id relref) . assoc) assoc)
 (when (equalp relref 0) (f-check-name1 construct id ref state))
 (f-check-name ref assoc state))

(deff f-check-name1 (construct id ref state)
 (:let definition (codex-get state ref id 0 card-name))
 (:when (null definition)
  (:let construct (utf2string (f-externalize construct)))
  (format t "Missing name definition:~%~a~%" construct))
 (:let rhs (fourth definition))
 (:let construct1 (f-internalize (c-tree2card* rhs)))
 (:when (unequal construct construct1)
  (:let construct (utf2string (f-externalize construct)))
  (:let construct1 (utf2string (f-externalize construct1)))
  (format t "Construct mismatch:~%~a~%~a~%" construct construct1)))

#|
=============================================
Codification of translated page
=============================================
|#

(deff add-vector-to-state (ref vector state)
 (:let state (rack-put state vector ref card-vector))
 (setq *max-iter* (option "iterations"))
 (:let state (c-load ref state))
 state)

#|
=============================================
Main function
=============================================
|#

#|
(deff run-frontend (filename)
 (:catch nil nil)
 (:let file (slurp filename))
 (:let (vector) (frontend filename file nil))
 vector)
|#

(deff frontend (state)
 (when (verbose '> 0) (progress "Frontend: process options"))
 (f-process-options)
 (when (verbose '> 0) (progress "Frontend: parsing page"))
 (:let (bib arity tree state assoc) (f-parse-page state))
 (:when (level '<= "parse") nil)
 (progress "Frontend: vectorizing")
 (:let (ref vector) (f-vectorize bib arity tree assoc))
 (:let vector (ct2vector vector))
 (:let ref (ref2card ref))
 (when (verbose '> 0)
  (format t "Reference of generated page:~%~a~%" (ct2string (ref2kana ref))))
 (:let state (add-vector-to-state ref vector state))
 (when (level '= "submit") (dump-rack ref state))
 (when (level '= "submit") (c-memorize ref state))
 (progress "Frontend: checking name consistency")
 (f-check-name ref assoc state)
 (list ref state))










